/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%           IIIII  DDDD   EEEEE  N   N  TTTTT  IIIII  FFFFF  Y   Y            %
%             I    D   D  E      NN  N    T      I    F       Y Y             %
%             I    D   D  EEE    N N N    T      I    FFF      Y              %
%             I    D   D  E      N  NN    T      I    F        Y              %
%           IIIII  DDDD   EEEEE  N   N    T    IIIII  F        Y              %
%                                                                             %
%                                                                             %
%               Identify an Image Format and Characteristics.                 %
%                                                                             %
%                           Software Design                                   %
%                             John Cristy                                     %
%                            September 1994                                   %
%                                                                             %
%                                                                             %
%  Copyright 1999-2004 ImageMagick Studio LLC, a non-profit organization      %
%  dedicated to making software imaging solutions freely available.           %
%                                                                             %
%  You may not use this file except in compliance with the License.  You may  %
%  obtain a copy of the License at                                            %
%                                                                             %
%    http://www.imagemagick.org/www/Copyright.html                            %
%                                                                             %
%  Unless required by applicable law or agreed to in writing, software        %
%  distributed under the License is distributed on an "AS IS" BASIS,          %
%  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.   %
%  See the License for the specific language governing permissions and        %
%  limitations under the License.                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Identify describes the format and characteristics of one or more image
%  files.  It will also report if an image is incomplete or corrupt.
%
%
*/

/*
  Include declarations.
*/
#include "magick/studio.h"
#include "magick/annotate.h"
#include "magick/attribute.h"
#include "magick/client.h"
#include "magick/coder.h"
#include "magick/color.h"
#include "magick/configure.h"
#include "magick/constitute.h"
#include "magick/decorate.h"
#include "magick/delegate.h"
#include "magick/draw.h"
#include "magick/effect.h"
#include "magick/error.h"
#include "magick/gem.h"
#include "magick/geometry.h"
#include "magick/identify.h"
#include "magick/image.h"
#include "magick/list.h"
#include "magick/locale_.h"
#include "magick/log.h"
#include "magick/magic.h"
#include "magick/magick.h"
#include "magick/memory_.h"
#include "magick/module.h"
#include "magick/mogrify.h"
#include "magick/monitor.h"
#include "magick/montage.h"
#include "magick/option.h"
#include "magick/quantize.h"
#include "magick/random.h"
#include "magick/resize.h"
#include "magick/resource_.h"
#include "magick/string_.h"
#include "magick/utility.h"
#include "magick/version.h"

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+   I d e n t i f y I m a g e C o m m a n d                                   %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  IdentifyImageCommand() describes the format and characteristics of one or
%  more image files. It will also report if an image is incomplete or corrupt.
%  The information displayed includes the scene number, the file name, the
%  width and height of the image, whether the image is colormapped or not,
%  the number of colors in the image, the number of bytes in the image, the
%  format of the image (JPEG, PNM, etc.), and finally the number of seconds
%  it took to read and process the image.
%
%  The format of the IdentifyImageCommand method is:
%
%      MagickBooleanType IdentifyImageCommand(ImageInfo *image_info,int argc,
%        char **argv,char **metadata,ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o image_info: The image info.
%
%    o argc: The number of elements in the argument vector.
%
%    o argv: A text array containing the command line arguments.
%
%    o metadata: any metadata is returned here.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/

static void IdentifyUsage(void)
{
  const char
    **p;

  static const char
    *options[]=
    {
      "-authenticate value  decrypt image with this password",
      "-channel type        Red, Green, Blue, Opacity, Index, Cyan, Yellow, ",
      "                     Magenta, Black, or All",
      "-debug events        display copious debugging information",
      "-define format:option",
      "                     define one or more image format options",
      "-density geometry    horizontal and vertical density of the image",
      "-depth value         image depth",
      "-extract geometry    extract area from image",
      "-format \"string\"     output formatted image characteristics",
      "-help                print program options",
      "-interlace type      None, Line, Plane, or Partition",
      "-limit type value    Area, Disk, Map, or Memory resource limit",
      "-list type           Color, Configure, Delegate, Format, Magic, Module,",
      "                     Resource, or Type",
      "-log format          format of debugging information",
      "-sampling-factor geometry",
      "                     horizontal and vertical sampling factor",
      "-size geometry       width and height of image",
      "-strip               strip image of all profiles and comments",
      "-verbose             print detailed information about the image",
      "-version             print version information",
      "-virtual-pixel method",
      "                     Constant, Edge, Mirror, or Tile",
      (char *) NULL
    };

  (void) printf("Version: %s\n",GetMagickVersion((unsigned long *) NULL));
  (void) printf("Copyright: %s\n\n",GetMagickCopyright());
  (void) printf("Usage: %s [options ...] file [ [options ...] "
    "file ... ]\n",GetClientName());
  (void) printf("\nWhere options include:\n");
  for (p=options; *p != (char *) NULL; p++)
    (void) printf("  %s\n",*p);
  Exit(0);
}

MagickExport MagickBooleanType IdentifyImageCommand(ImageInfo *image_info,
  int argc,char **argv,char **metadata,ExceptionInfo *exception)
{
#define DestroyIdentify() \
{ \
  if (format != (char *) NULL) \
    format=(char *) RelinquishMagickMemory(format); \
  DestroyImageList(image); \
  for (i=0; i < argc; i++) \
    argv[i]=(char *) RelinquishMagickMemory(argv[i]); \
  argv=(char **) RelinquishMagickMemory(argv); \
}
#define ThrowIdentifyException(severity,tag,option) \
{ \
  (void) ThrowMagickException(exception,GetMagickModule(),severity,tag,option);\
  DestroyIdentify(); \
  return(MagickFalse); \
}
#define ThrowIdentifyInvalidArgumentException(option,argument) \
{ \
  (void) ThrowMagickException(exception,GetMagickModule(),OptionError, \
    "InvalidArgument",argument,option); \
  DestroyIdentify(); \
  return(MagickFalse); \
}

  char
    *format,
    *option,
    *q;

  Image
    *image;

  ImageInfo
    *identify_info;

  long
    number_images;

  MagickBooleanType
    ping,
    status;

  register Image
    *p;

  register long
    i;

  unsigned long
    count;

  /*
    Set defaults.
  */
  assert(image_info != (ImageInfo *) NULL);
  assert(image_info->signature == MagickSignature);
  if (image_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"");
  assert(exception != (ExceptionInfo *) NULL);
  if (argc < 2)
    IdentifyUsage();
  image=NewImageList();
  count=0;
  format=(char *) NULL;
  number_images=0;
  option=(char *) NULL;
  ping=MagickTrue;
  /*
    Identify an image.
  */
  ReadCommandlLine(argc,&argv);
  status=ExpandFilenames(&argc,&argv);
  if (status == MagickFalse)
    ThrowIdentifyException(ResourceLimitError,"MemoryAllocationFailed",
      strerror(errno));
  for (i=1; i < argc; i++)
  {
    option=argv[i];
    if (LocaleCompare("-format",argv[i]) == 0)
      {
        i++;
        if (i == argc)
          ThrowIdentifyException(OptionError,"MissingArgument",option);
        (void) CloneString(&format,argv[i]);
        break;
      }
  }
  for (i=1; i < argc; i++)
  {
    option=argv[i];
    if ((strlen(option) == 1) || ((*option != '-') && (*option != '+')))
      {
        /*
          Identify image.
        */
        (void) CopyMagickString(image_info->filename,argv[i],MaxTextExtent);
        if (format != (char *) NULL)
          for (q=strchr(format,'%'); q != (char *) NULL; q=strchr(q+1,'%'))
            if ((*(q+1) == 'g') || (*(q+1) == 'k') || (*(q+1) == 'z') ||
                (*(q+1) == '#'))
              {
                ping=MagickFalse;
                break;
              }
        identify_info=CloneImageInfo(image_info);
        identify_info->verbose=MagickFalse;
        if ((image_info->verbose != MagickFalse) || (ping == MagickFalse))
          image=ReadImage(identify_info,exception);
        else
          image=PingImage(identify_info,exception);
        DestroyImageInfo(identify_info);
        if (exception->severity != UndefinedException)
          CatchException(exception);
        status&=image != (Image *) NULL;
        if (image == (Image *) NULL)
          continue;
        for (p=image; p != (Image *) NULL; p=p->next)
        {
          if (p->scene == 0)
            p->scene=count++;
          if (format == (char *) NULL)
            {
              (void) DescribeImage(p,stdout,image_info->verbose);
              continue;
            }
          if (metadata != (char **) NULL)
            {
              char
                *text;

              text=TranslateText(image_info,p,format);
              if (text == (char *) NULL)
                ThrowIdentifyException(ResourceLimitError,
                  "MemoryAllocationFailed",strerror(errno));
              (void) ConcatenateString(&(*metadata),text);
              text=(char *) RelinquishMagickMemory(text);
            }
        }
        DestroyImageList(image);
        image=NewImageList();
        number_images++;
        continue;
      }
    switch (*(option+1))
    {
      case 'c':
      {
        if (LocaleCompare("cache",option+1) == 0)
          {
            if (*option == '-')
              {
                unsigned long
                  limit;

                i++;
                if (i == argc)
                  ThrowIdentifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowIdentifyInvalidArgumentException(option,argv[i]);
                limit=(~0UL);
                if (LocaleCompare("unlimited",argv[i]) != 0)
                  limit=(unsigned long) atol(argv[i]);
                (void) SetMagickResourceLimit(MemoryResource,limit);
                (void) SetMagickResourceLimit(MapResource,2*limit);
              }
            break;
          }
        if (LocaleCompare("channel",option+1) == 0)
          {
            if (*option == '-')
              {
                long
                  channel;

                i++;
                if (i == argc)
                  ThrowIdentifyException(OptionError,"MissingArgument",option);
                channel=ParseChannelOption(argv[i]);
                if (channel < 0)
                  ThrowIdentifyException(OptionError,"UnrecognizedChannelType",
                    argv[i]);
              }
            break;
          }
        ThrowIdentifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'd':
      {
        if (LocaleCompare("debug",option+1) == 0)
          {
            (void) SetLogEventMask("None");
            if (*option == '-')
              {
                LogEventType
                  event_mask;

                i++;
                if (i == argc)
                  ThrowIdentifyException(OptionError,"MissingArgument",option);
                event_mask=SetLogEventMask(argv[i]);
                if (event_mask == UndefinedEvents)
                  ThrowIdentifyException(OptionError,"UnrecognizedEventType",
                    option);
              }
            image_info->debug=IsEventLogging();
            break;
          }
        if (LocaleCompare("define",option+1) == 0)
          {
            i++;
            if (i == argc)
              ThrowIdentifyException(OptionError,"MissingArgument",option);
            if (*option == '+')
              {
                char
                  *define;

                define=RemoveImageOption(image_info,argv[i]);
                if (define == (char *) NULL)
                  ThrowIdentifyException(OptionError,"NoSuchOption",argv[i]);
                define=(char *) RelinquishMagickMemory(define);
                break;
              }
            status=DefineImageOption(image_info,argv[i]);
            if (status == MagickFalse)
              ThrowIdentifyException(OptionError,"UnrecognizedOption",argv[i]);
            break;
          }
        if (LocaleCompare("density",option+1) == 0)
          {
            (void) CloneString(&image_info->density,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == argc)
                  ThrowIdentifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowIdentifyInvalidArgumentException(option,argv[i]);
                (void) CloneString(&image_info->density,argv[i]);
              }
            break;
          }
        if (LocaleCompare("depth",option+1) == 0)
          {
            image_info->depth=QuantumDepth;
            if (*option == '-')
              {
                i++;
                if (i == argc)
                  ThrowIdentifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowIdentifyInvalidArgumentException(option,argv[i]);
                image_info->depth=(unsigned long) atol(argv[i]);
              }
            break;
          }
        ThrowIdentifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'f':
      {
        if (LocaleCompare("format",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == argc)
                  ThrowIdentifyException(OptionError,"MissingArgument",option);
              }
            break;
          }
        ThrowIdentifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'h':
      {
        if (LocaleCompare("help",option+1) == 0)
          IdentifyUsage();
        ThrowIdentifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'i':
      {
        if (LocaleCompare("interlace",option+1) == 0)
          {
            image_info->interlace=UndefinedInterlace;
            if (*option == '-')
              {
                long
                  interlace;

                i++;
                if (i == argc)
                  ThrowIdentifyException(OptionError,"MissingArgument",option);
                interlace=ParseMagickOption(MagickInterlaceOptions,MagickFalse,
                  argv[i]);
                if (interlace < 0)
                  ThrowIdentifyException(OptionError,
                    "UnrecognizedInterlaceType",argv[i]);
                image_info->interlace=(InterlaceType) interlace;
              }
            break;
          }
        ThrowIdentifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'l':
      {
        if (LocaleCompare("limit",option+1) == 0)
          {
            if (*option == '-')
              {
                char
                  *type;

                long
                  resource;

                unsigned long
                  limit;

                i++;
                if (i == argc)
                  ThrowIdentifyException(OptionError,"MissingArgument",option);
                type=argv[i];
                i++;
                if (i == argc)
                  ThrowIdentifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowIdentifyInvalidArgumentException(option,argv[i]);
                limit=(~0UL);
                if (LocaleCompare("unlimited",argv[i]) != 0)
                  limit=(unsigned long) atol(argv[i]);
                resource=ParseMagickOption(MagickResourceOptions,MagickFalse,type);
                if (resource < 0)
                  ThrowIdentifyException(OptionError,
                    "UnrecognizedResourceType",type);
                (void) SetMagickResourceLimit((ResourceType) resource,limit);
                break;
              }
            break;
          }
        if (LocaleCompare("list",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == argc)
                  ThrowIdentifyException(OptionError,"MissingArgument",option);
                switch (*argv[i])
                {
                  case 'C':
                  case 'c':
                  {
                    if (LocaleCompare("Coder",argv[i]) == 0)
                      {
                        (void) ListCoderInfo((FILE *) NULL,exception);
                        break;
                      }
                    if (LocaleCompare("Color",argv[i]) == 0)
                      {
                        (void) ListColorInfo((FILE *) NULL,exception);
                        break;
                      }
                    if (LocaleCompare("Configure",argv[i]) == 0)
                      {
                        (void) ListConfigureInfo((FILE *) NULL,exception);
                        break;
                      }
                    ThrowIdentifyException(OptionError,"UnrecognizedListType",
                      argv[i])
                  }
                  case 'D':
                  case 'd':
                  {
                    if (LocaleCompare("Delegate",argv[i]) == 0)
                      {
                        (void) ListDelegateInfo((FILE *) NULL,exception);
                        break;
                      }
                    ThrowIdentifyException(OptionError,"UnrecognizedListType",
                      argv[i])
                  }
                  case 'F':
                  case 'f':
                  {
                    if (LocaleCompare("Format",argv[i]) == 0)
                      {
                        (void) ListMagickInfo((FILE *) NULL,exception);
                        break;
                      }
                    ThrowIdentifyException(OptionError,"UnrecognizedListType",
                      argv[i])
                  }
                  case 'L':
                  case 'l':
                  {
                    if (LocaleCompare("Locale",argv[i]) == 0)
                      {
                        (void) ListLocaleInfo((FILE *) NULL,exception);
                        break;
                      }
                    if (LocaleCompare("Log",argv[i]) == 0)
                      {
                        (void) ListLogInfo((FILE *) NULL,exception);
                        break;
                      }
                    ThrowIdentifyException(OptionError,"UnrecognizedListType",
                      argv[i])
                  }
                  case 'M':
                  case 'm':
                  {
                    if (LocaleCompare("Magic",argv[i]) == 0)
                      {
                        (void) ListMagicInfo((FILE *) NULL,exception);
                        break;
                      }
#if defined(SupportMagickModules)
                    if (LocaleCompare("Module",argv[i]) == 0)
                      {
                        (void) ListModuleInfo((FILE *) NULL,exception);
                        break;
                      }
#endif
                    ThrowIdentifyException(OptionError,"UnrecognizedListType",
                      argv[i])
                  }
                  case 'R':
                  case 'r':
                  {
                    if (LocaleCompare("Resource",argv[i]) == 0)
                      {
                        (void) ListMagickResourceInfo((FILE *) NULL,exception);
                        break;
                      }
                    ThrowIdentifyException(OptionError,"UnrecognizedListType",
                      argv[i])
                  }
                  case 'T':
                  case 't':
                  {
                    if (LocaleCompare("Type",argv[i]) == 0)
                      {
                        (void) ListTypeInfo((FILE *) NULL,exception);
                        break;
                      }
                    ThrowIdentifyException(OptionError,"UnrecognizedListType",
                      argv[i])
                  }
                  default:
                    ThrowIdentifyException(OptionError,"UnrecognizedListType",
                      argv[i])
                }
                return(MagickTrue);
              }
            break;
          }
        if (LocaleCompare("log",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if ((i == argc) || (strchr(argv[i],'%') == (char *) NULL))
                  ThrowIdentifyException(OptionError,"MissingArgument",option);
                (void) SetLogFormat(argv[i]);
              }
            break;
          }
        ThrowIdentifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'p':
      {
        if (LocaleCompare("ping",option+1) == 0)
          break;
        ThrowIdentifyException(OptionError,"UnrecognizedOption",option)
      }
      case 's':
      {
        if (LocaleCompare("sampling-factor",option+1) == 0)
          {
            (void) CloneString(&image_info->sampling_factor,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == argc)
                  ThrowIdentifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowIdentifyInvalidArgumentException(option,argv[i]);
                (void) CloneString(&image_info->sampling_factor,argv[i]);
              }
            break;
          }
        if (LocaleCompare("size",option+1) == 0)
          {
            (void) CloneString(&image_info->size,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == argc)
                  ThrowIdentifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowIdentifyInvalidArgumentException(option,argv[i]);
                (void) CloneString(&image_info->size,argv[i]);
              }
            break;
          }
        if (LocaleCompare("strip",option+1) == 0)
          break;
        if (LocaleCompare("support",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == argc)
                  ThrowIdentifyException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == MagickFalse)
                  ThrowIdentifyInvalidArgumentException(option,argv[i]);
              }
            break;
          }
        ThrowIdentifyException(OptionError,"UnrecognizedOption",option)
      }
      case 'v':
      {
        if (LocaleCompare("verbose",option+1) == 0)
          {
            image_info->verbose=(MagickBooleanType) (*option == '-');
            break;
          }
        if (LocaleCompare("version",option+1) == 0)
          break;
        if (LocaleCompare("virtual-pixel",option+1) == 0)
          {
            if (*option == '-')
              {
                long
                  method;

                i++;
                if (i == (argc-1))
                  ThrowIdentifyException(OptionError,"MissingArgument",option);
                method=ParseMagickOption(MagickVirtualPixelOptions,MagickFalse,
                  argv[i]);
                if (method < 0)
                  ThrowIdentifyException(OptionError,
                    "UnrecognizedVirtualPixelMethod",argv[i]);
              }
            break;
          }
        ThrowIdentifyException(OptionError,"UnrecognizedOption",option)
      }
      case '?':
        break;
      default:
        ThrowIdentifyException(OptionError,"UnrecognizedOption",option)
    }
  }
  if (i != argc)
    ThrowIdentifyException(OptionError,"MissingAnImageFilename",argv[i]);
  if (number_images == 0)
    ThrowIdentifyException(OptionError,"MissingAnImageFilename",argv[argc-1]);
  if (format != (char *) NULL)
    format=(char *) RelinquishMagickMemory(format);
  DestroyIdentify();
  return(status);
}
