/* knotes-action.cc                      KPilot
**
** Copyright (C) 2001,2002 by Dan Pilone
**
** This file defines the SyncAction for the knotes-conduit plugin.
*/
 
/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program in a file called COPYING; if not, write to
** the Free Software Foundation, Inc., 675 Mass Ave, Cambridge,
** MA 02139, USA.
*/
 
/*
** Bug reports and questions can be sent to kde-pim@kde.org
*/

#include "options.h"

#include <qmap.h>
#include <qtimer.h>

#include <kapplication.h>

#include <kconfig.h>
#include <dcopclient.h>

#include <time.h>  // required by pilot-link includes

#include <pi-memo.h>

#include "pilotMemo.h"
#include "pilotSerialDatabase.h"

#include "KNotesIface_stub.h"

#include "knotes-factory.h"

#include "knotes-action.moc"


class NoteAndMemo
{
public:
	NoteAndMemo() : noteId(-1),memoId(-1) { } ;
	NoteAndMemo(int noteid,int memoid) : noteId(noteid),memoId(memoid) { } ;

	int memo() const { return memoId; } ;
	int note() const { return noteId; } ;
	bool valid() const { return (noteId>0) && (memoId>0); } ;
	QString toString() const { return QString("<%1,%2>").arg(noteId).arg(memoId); } ;

	static NoteAndMemo findNote(const QValueList<NoteAndMemo> &,int note);
	static NoteAndMemo findMemo(const QValueList<NoteAndMemo> &,int memo);

protected:
	int noteId;
	int memoId;
} ;

NoteAndMemo NoteAndMemo::findNote(const QValueList<NoteAndMemo> &l ,int note)
{
	FUNCTIONSETUP;

	for (QValueList<NoteAndMemo>::ConstIterator it =l.begin();
		it != l.end();
		++it)
	{
		if ((*it).note()==note) return *it;
	}

	return NoteAndMemo();
}

NoteAndMemo NoteAndMemo::findMemo(const QValueList<NoteAndMemo> &l ,int memo)
{
	FUNCTIONSETUP;

	for (QValueList<NoteAndMemo>::ConstIterator it =l.begin();
		it != l.end();
		++it)
	{
		if ((*it).memo()==memo) return *it;
	}

	return NoteAndMemo();
}

class KNotesAction::KNotesActionPrivate
{
public:
	KNotesActionPrivate() :
		fDCOP(0L),
		fKNotes(0L),
		fTimer(0L),
		// fDatabase(0L),
		fCounter(0)
	{ } ;

	// These are  the notes that we got from KNotes
	QMap <int,QString> fNotes;
	// This iterates through that list; it's in here because
	// we use slots to process one item at a time and need
	// to keep track of where we are between slot calls.
	QMap <int,QString>::ConstIterator fIndex;
	// The DCOP client for this application, and the KNotes stub.
	DCOPClient *fDCOP;
	KNotesIface_stub *fKNotes;
	// The timer for invoking process() to do some more work.
	QTimer *fTimer;
	// The database we're working with (MemoDB)
	// PilotSerialDatabase *fDatabase;
	// Some counter that needs to be preserved between calls to
	// process(). Typically used to note hom much work is done.
	int fCounter;

	// We need to translate between the ids that KNotes uses and
	// Pilot id's, so we make a list of pairs.
	//
	QValueList<NoteAndMemo> fIdList;
} ;


/* static */ const char * const KNotesAction::noteIdsKey="NoteIds";
/* static */ const char * const KNotesAction::memoIdsKey="MemoIds";


KNotesAction::KNotesAction(KPilotDeviceLink *o,
	const char *n, const QStringList &a) :
	ConduitAction(o,n,a),
	fP(new KNotesActionPrivate)
{
	FUNCTIONSETUP;


	fP->fDCOP = KApplication::kApplication()->dcopClient();

	if (!fP->fDCOP)
	{
		kdWarning() << k_funcinfo
			<< ": Can't get DCOP client."
			<< endl;
	}
}

/* virtual */ KNotesAction::~KNotesAction()
{
	FUNCTIONSETUP;

	KPILOT_DELETE(fP->fTimer);
	KPILOT_DELETE(fP->fKNotes);
	// KPILOT_DELETE(fP->fDatabase);
	KPILOT_DELETE(fP);
}

/* virtual */ bool KNotesAction::exec()
{
	FUNCTIONSETUP;

	QString e;
	if (!fP->fDCOP)
	{
		emit logError(i18n("No DCOP connection could be made. The "
			"conduit cannot function like this."));
		return false;

	}
	if (!PluginUtility::isRunning("knotes"))
	{
		emit logError(i18n("KNotes is not running. The conduit must "
			"be able to make a DCOP connection to KNotes "
			"for synchronization to take place. "
			"Please start KNotes and try again."));
		return false;
	}

	if (!fConfig) return false;

	fP->fKNotes = new KNotesIface_stub("knotes","KNotesIface");

	fP->fNotes = fP->fKNotes->notes();

	openDatabases("MemoDB");

	if (isTest())
	{
		listNotes();
	}
	else
	{
		fP->fTimer = new QTimer(this);
		fStatus = Init;
		resetIndexes();

		connect(fP->fTimer,SIGNAL(timeout()),SLOT(process()));

		fP->fTimer->start(0,false);
	}

	return true;
}

void KNotesAction::resetIndexes()
{
	FUNCTIONSETUP;

	fP->fCounter = 0;
	fP->fIndex = fP->fNotes.begin();
}

void KNotesAction::listNotes()
{
	FUNCTIONSETUP;

	QMap<int,QString>::ConstIterator i = fP->fNotes.begin();
	while (i != fP->fNotes.end())
	{
#ifdef DEBUG
		DEBUGCONDUIT << fname
			<< ": "
			<< i.key()
			<< "->"
			<< i.data()
			<< (fP->fKNotes->isNew("kpilot",i.key()) ?
				" (new)" : "" )
			<< endl;
#endif
		i++;
	}

	emit syncDone(this);
}

/* slot */ void KNotesAction::process()
{
	switch(fStatus)
	{
	case Init:
		getAppInfo();
		getConfigInfo();
		break;
	case ModifiedNotesToPilot :
		if (modifyNoteOnPilot())
		{
			resetIndexes();
			fStatus = NewNotesToPilot;
		}
		break;
	case NewNotesToPilot :
		if (addNewNoteToPilot())
		{
			resetIndexes();
			fStatus = MemosToKNotes;
			fDatabase->resetDBIndex();
		}
		break;
	case MemosToKNotes :
		if (syncMemoToKNotes())
		{
			fStatus=Cleanup;
		}
		break;
	case Cleanup :
		cleanupMemos();
		break;
	default :
		fP->fTimer->stop();
		emit syncDone(this);
	}
}


void KNotesAction::getConfigInfo()
{
	FUNCTIONSETUP;

	if (fConfig)
	{
		KConfigGroupSaver g(fConfig,KNotesConduitFactory::group);

		QValueList<int> notes;
		QValueList<int> memos;


		notes=fConfig->readIntListEntry(noteIdsKey);
		memos=fConfig->readIntListEntry(memoIdsKey);

		if (notes.count() != memos.count())
		{
			kdWarning() << k_funcinfo
				<< ": Notes and memo id lists don't match ("
				<< notes.count()
				<< ","
				<< memos.count()
				<< ")"
				<< endl;
		}

		QValueList<int>::ConstIterator iNotes = notes.begin();
		QValueList<int>::ConstIterator iMemos = memos.begin();

		while((iNotes != notes.end()) && (iMemos != memos.end()))
		{
			fP->fIdList.append(NoteAndMemo(*iNotes,*iMemos));
			++iNotes;
			++iMemos;
		}
	}
}

void KNotesAction::getAppInfo()
{
	FUNCTIONSETUP;


	unsigned char buffer[PilotDatabase::MAX_APPINFO_SIZE];
	int appInfoSize = fDatabase->readAppBlock(buffer,PilotDatabase::MAX_APPINFO_SIZE);
	struct MemoAppInfo memoInfo;

	if (appInfoSize<0)
	{
		fStatus=Error;
		return;
	}

	unpack_MemoAppInfo(&memoInfo,buffer,appInfoSize);
	PilotDatabase::listAppInfo(&memoInfo.category);

	resetIndexes();
	fStatus=ModifiedNotesToPilot;
	
	addSyncLogEntry(i18n("[KNotes conduit: "));
}


bool KNotesAction::modifyNoteOnPilot()
{
	FUNCTIONSETUP;

	if (fP->fIndex == fP->fNotes.end())
	{
		if (fP->fCounter)
		{
			addSyncLogEntry(i18n("Modified one memo.",
				"Modified %n memos.",
				fP->fCounter));
		}
		else
		{
			addSyncLogEntry("No memos were changed.");
		}
		return true;
	}

	if (fP->fKNotes->isModified("kpilot",fP->fIndex.key()))
	{
#ifdef DEBUG
		DEBUGCONDUIT << fname
			<< ": The note #"
			<< fP->fIndex.key()
			<< " with name "
			<< fP->fIndex.data()
			<< " is modified in KNotes."
			<< endl;
#endif

		NoteAndMemo nm = NoteAndMemo::findNote(fP->fIdList,
			fP->fIndex.key());

		if (nm.valid())
		{
			QString text = fP->fIndex.data() + "\n" ;
			text.append(fP->fKNotes->text(fP->fIndex.key()));

			const char *c = text.latin1();
			PilotMemo *a = new PilotMemo((void *)(const_cast<char *>(c)));
			PilotRecord *r = a->pack();
			r->setID(nm.memo());

			int newid = fDatabase->writeRecord(r);

			if (newid != nm.memo())
			{
				kdWarning() << k_funcinfo
					<< ": Memo id changed during write? "
					<< "From "
					<< nm.memo()
					<< " to "
					<< newid
					<< endl;
			}
		}
		else
		{
			kdWarning() << ": Modified note unknown to Pilot" << endl;
		}

		fP->fCounter++;
	}

	++(fP->fIndex);
	return false;
}

bool KNotesAction::addNewNoteToPilot()
{
	FUNCTIONSETUP;

	if (fP->fIndex == fP->fNotes.end())
	{
		if (fP->fCounter)
		{
			addSyncLogEntry(i18n("Added one new memo.",
				"Added %n new memos.",
				fP->fCounter));
		}
		else
		{
			addSyncLogEntry("No memos were added.");
		}
		return true;
	}

	if (fP->fKNotes->isNew("kpilot",fP->fIndex.key()))
	{
#ifdef DEBUG
		DEBUGCONDUIT << fname
			<< ": The note #"
			<< fP->fIndex.key()
			<< " with name "
			<< fP->fIndex.data()
			<< " is new to the Pilot."
			<< endl;
#endif

		QString text = fP->fIndex.data() + "\n" ;
		text.append(fP->fKNotes->text(fP->fIndex.key()));

		const char *c = text.latin1();
		PilotMemo *a = new PilotMemo((void *)(const_cast<char *>(c)));
		PilotRecord *r = a->pack();

		int newid = fDatabase->writeRecord(r);

		fP->fIdList.append(NoteAndMemo(fP->fIndex.key(),newid));

		delete r;
		delete a;

		fP->fCounter++;
	}

	++(fP->fIndex);
	return false;
}

bool KNotesAction::syncMemoToKNotes()
{
	FUNCTIONSETUP;

	PilotRecord *rec = fDatabase->readNextModifiedRec();
	if (!rec)
	{
		if (fP->fCounter)
		{
			addSyncLogEntry(i18n("Added one memo to KNotes.",
				"Added %n memos to KNotes.",fP->fCounter));
		}
		else
		{
			addSyncLogEntry("No memos added to KNotes.");
		}
		return true;
	}

	fP->fCounter++;

	PilotMemo *memo = new PilotMemo(rec);
	NoteAndMemo m = NoteAndMemo::findMemo(fP->fIdList,memo->id());

#ifdef DEBUG
	DEBUGCONDUIT << fname << ": Looking at memo " 
		<< memo->id()
		<< " which was found "
		<< m.toString()
		<< endl;
#endif

	if (m.valid())
	{
		// We knew about the note already, but it
		// has changed on the Pilot.
		//
		//
		if (memo->isDeleted())
		{
#ifdef DEBUG
			DEBUGCONDUIT << fname << ": It's been deleted." << endl;
#endif
			fP->fKNotes->killNote(m.note());
		}
		else
		{
#ifdef DEBUG
			DEBUGCONDUIT << fname << ": It's just modified." << endl;
			DEBUGCONDUIT << fname << ": <"
				<< fP->fNotes[m.note()]
				<< "> <"
				<< memo->shortTitle()
				<< ">"
				<< endl;
#endif
			if (fP->fNotes[m.note()] != memo->shortTitle())
			{
				// Name changed. KNotes might complain though.
				fP->fKNotes->setName(m.note(),memo->shortTitle());
			}
			fP->fKNotes->setText(m.note(),memo->text());
		}
	}
	else
	{
		if (memo->isDeleted())
		{
#ifdef DEBUG
			DEBUGCONDUIT << fname << ": It's new and deleted." << endl;
#endif
			// Do nothing, it's new and deleted at the same time
		}
		else
		{
			int i = fP->fKNotes->newNote(memo->shortTitle(),memo->text());
			fP->fIdList.append(NoteAndMemo(i,memo->id()));
#ifdef DEBUG
			DEBUGCONDUIT << fname << ": It's new with knote id " << i << endl;
#endif
		}
	}

	if (memo) delete memo;
	if (rec) delete rec;

	return false;
}


void KNotesAction::cleanupMemos()
{
	FUNCTIONSETUP;

	// Tell KNotes we're up-to-date
	fP->fKNotes->sync("kpilot");

	if (fConfig)
	{
#ifdef DEBUG
		DEBUGCONDUIT << fname
			<< ": Writing "
			<< fP->fIdList.count()
			<< " pairs to the config file."
			<< endl;
		DEBUGCONDUIT << fname
			<< ": The config file is read-only: "
			<< fConfig->isReadOnly()
			<< endl;
#endif

		KConfigGroupSaver g(fConfig,KNotesConduitFactory::group);

		QValueList<int> notes;
		QValueList<int> memos;

		for (QValueList<NoteAndMemo>::ConstIterator i =
			fP->fIdList.begin();
			i!=fP->fIdList.end();
			++i)
		{
			notes.append((*i).note());
			memos.append((*i).memo());
		}

		fConfig->writeEntry(noteIdsKey,notes);
		fConfig->writeEntry(memoIdsKey,memos);
		fConfig->sync();
	}

	fStatus=Done;
	fDatabase->cleanup();
	fDatabase->resetSyncFlags();

	addSyncLogEntry("]\n");
}


/* virtual */ QString KNotesAction::statusString() const
{
	switch(fStatus)
	{
	case Init : return QString("Init");
	case NewNotesToPilot :
		return QString("NewNotesToPilot key=%1")
			.arg(fP->fIndex.key());
	case Done :
		return QString("Done");
	default :
		return QString("Unknown (%1)").arg(fStatus);
	}
}


// $Log: knotes-action.cc,v $
// Revision 1.14  2002/08/25 13:28:28  mhunter
// CVS_SILENT Corrected typographical errors
//
// When replying, please CC me - I'm not subscribed
//
// Revision 1.13  2002/08/23 22:03:20  adridg
// See ChangeLog - exec() becomes bool, debugging added
//
// Revision 1.12  2002/05/23 17:08:32  adridg
// Some compile fixes for non-debug mode, and KNotes syncing fixes
//
// Revision 1.11  2002/05/23 08:01:29  adridg
// Try to sync KNotes->Pilot
//
// Revision 1.10  2002/05/19 15:01:49  adridg
// Patches for the KNotes conduit
//
// Revision 1.9  2002/05/15 17:15:33  gioele
// kapp.h -> kapplication.h
// I have removed KDE_VERSION checks because all that files included "options.h"
// which #includes <kapplication.h> (which is present also in KDE_2).
// BTW you can't have KDE_VERSION defined if you do not include
// - <kapplication.h>: KDE3 + KDE2 compatible
// - <kdeversion.h>: KDE3 only compatible
//
// Revision 1.8  2002/02/23 20:57:40  adridg
// #ifdef DEBUG stuff
//
// Revision 1.7  2002/01/20 22:42:43  adridg
// CVS_SILENT: Administrative
//
// Revision 1.6  2001/12/31 09:24:25  adridg
// Cleanup, various fixes for runtime loading
//
// Revision 1.5  2001/12/20 22:55:44  adridg
// Making conduits save their configuration and doing syncs
//
// Revision 1.4  2001/12/02 22:08:24  adridg
// CVS_SILENT: I forget
//
// Revision 1.3  2001/10/31 23:46:51  adridg
// CVS_SILENT: Ongoing conduits ports
//
// Revision 1.2  2001/10/29 09:45:19  cschumac
// Make it compile.
//
// Revision 1.1  2001/10/16 21:44:53  adridg
// Split up some files, added behavior
//
// Revision 1.4  2001/10/10 22:39:49  adridg
// Some UI/Credits/About page patches
//
// Revision 1.3  2001/10/10 21:42:09  adridg
// Actually do part of a sync now
//
// Revision 1.2  2001/10/10 13:40:07  cschumac
// Compile fixes.
//
// Revision 1.1  2001/10/08 22:27:42  adridg
// New ui, moved to lib-based conduit
//
//

