
#include "mousepageimpl.h"
#include "kxdata.h"

#include <qcombobox.h>
#include <qcheckbox.h>

#include <klocale.h>
#include <kdebug.h>

#include <assert.h>

MousePage::MousePage( QWidget *parent, const char *name )
    : MousePageBase( parent, name )
{
    connect( m_mouseType, SIGNAL( activated( int ) ),
             this, SLOT( slotMouseTypeChanged( int ) ) );
    connect( m_mouseModel, SIGNAL( activated( int ) ),
             this, SLOT( slotMouseModelChanged( int ) ) );

    connect( m_mouseType, SIGNAL( activated( int ) ),
             this, SIGNAL( changed() ) );
    connect( m_mouseModel, SIGNAL( activated( int ) ),
             this, SIGNAL( changed() ) );
    connect( m_port, SIGNAL( activated( int ) ),
             this, SIGNAL( changed() ) );

    connect( m_emulate3rdButton, SIGNAL( clicked() ),
             this, SIGNAL( changed() ) );
}

MousePage::~MousePage()
{
}

void MousePage::load()
{
    KXPointerData pointer = m_data->currentPointers()[ 0 ];

    switch ( pointer.pointerModel().pointerClass() )
    {
        case KX_USB:
            m_mouseType->setCurrentItem( 0 );
            slotMouseTypeChanged( 0 );
            break;
        case KX_PS2:
            m_mouseType->setCurrentItem( 1 );
            slotMouseTypeChanged( 1 );
            break;
        case KX_Serial:
            m_mouseType->setCurrentItem( 2 );
            slotMouseTypeChanged( 2 );
            break;
        case KX_Bus:
            m_mouseType->setCurrentItem( 3 );
            slotMouseTypeChanged( 3 );
            break;
    }

    for ( int i = 0; i < m_mouseModel->count(); i++ )
        if ( m_mouseModel->text( i ) == pointer.pointerModel().name() )
        {
            m_mouseModel->setCurrentItem( i );
            break;
        }

    m_emulate3rdButton->setChecked( pointer.emulate3rd() );

    m_currentModel = pointer.pointerModel();

    emit enableForward( true );
}

void MousePage::save()
{
    kdDebug() << "Mouse::save()" << endl;

    KXPointerData pointer = m_data->currentPointers()[ 0 ];

    pointer.setEmulate3rd( m_emulate3rdButton->isChecked() );
    pointer.setPointerModel( m_currentModel );
    pointer.setPort( m_port->currentText() );

    m_data->currentPointers()[ 0 ] = pointer;
}

void MousePage::slotMouseTypeChanged( int type )
{
    m_port->clear();

    KXPointerClass klass = typeToClass( type );

    loadModels( klass );

    switch ( klass )
    {
        case KX_USB:
            m_port->insertItem( "/dev/input/mouse0" );
            break;
        case KX_PS2:
            m_port->insertItem( "/dev/psaux" );
            break;
        case KX_Serial:
            m_port->insertItem( "/dev/ttyS0" );
            m_port->insertItem( "/dev/ttyS1" );
            break;
        case KX_Bus:
//            m_port->insertItem( i18n( "N/A" ) );
            break;
    }

    m_port->setEnabled( m_port->count() > 1 );
    slotMouseModelChanged( 0 );
}

void MousePage::slotMouseModelChanged( int model )
{
    KXPointerClass klass = typeToClass( m_mouseType->currentItem() );

    int i = 0;
    KXPointerModelList lst = m_data->pointerModels();
    QValueListConstIterator<KXPointerModel> it = lst.begin();
    QValueListConstIterator<KXPointerModel> end = lst.end();
    for (; it != end; ++it )
        if ( (*it).pointerClass() == klass )
        {
            if ( model == i )
            {
                m_currentModel = *it;

                if ( !(*it).device().isEmpty() )
                {
                    m_port->clear();
                    m_port->insertItem( (*it).device() );
                    m_port->setEnabled( true );
                }

                break;
            }

            i++;
        }

}

void MousePage::loadModels( KXPointerClass klass )
{
    m_mouseModel->clear();

    KXPointerModelList lst = m_data->pointerModels();
    QValueListConstIterator<KXPointerModel> it = lst.begin();
    QValueListConstIterator<KXPointerModel> end = lst.end();
    for (; it != end; ++it )
        if ( (*it).pointerClass() == klass )
            m_mouseModel->insertItem( (*it).name() );

    m_mouseModel->setEnabled( m_mouseModel->count() > 1 );
}

KXPointerClass MousePage::typeToClass( int type )
{
    switch ( type )
    {
        case 0: return KX_USB;
        case 1: return KX_PS2;
        case 2: return KX_Serial;
        case 3: return KX_Bus;
        default: assert( false );
    }

    return KX_PS2; // never reached
}

QString MousePage::quickHelp() const
{
    return i18n("<h1>Mouse Setup</h1><p>Please select the type, model and port of your pointer "
                "device from the dropdown fields in the upper left. "
                "If your mouse does not have a third mouse button you might want to enable the "
                "third button emulation in the lower left. A 3rd button click is emulated by pressing "
                "both the first and second mouse button.</p>");
}

#include "mousepageimpl.moc"

