/*
 *
 * Copyright (C) 2003-2008 Sebastian Trueg <trueg@k3b.org>
 *           (C) 2009      Gustavo Pichorim Boiko <gustavo.boiko@kdemail.net>
 *
 * This file is part of the K3b project.
 * Copyright (C) 1998-2009 Sebastian Trueg <trueg@k3b.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * See the file "COPYING" for the exact licensing terms.
 */


#ifndef K3BAUDIODOC_H
#define K3BAUDIODOC_H

#include <k3bdoc.h>

#include <k3bcdtext.h>
#include <k3btoc.h>

#include <qfile.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qdatetime.h>
#include "k3b_export.h"
#include <kurl.h>

class QDomElement;

namespace K3b {
    class AudioTrack;
    class AudioDataSource;
    class AudioDecoder;
    class AudioFile;

    /**
     * Document class for an audio project.
     * @author Sebastian Trueg
     */
    class LIBK3B_EXPORT AudioDoc : public Doc
    {
        Q_OBJECT

        friend class MixedDoc;
        friend class AudioTrack;
        friend class AudioFile;

    public:
        AudioDoc( QObject* );
        ~AudioDoc();

        QString name() const;

        bool newDocument();

        void clear();

        Device::MediaTypes supportedMediaTypes() const;

        bool hideFirstTrack() const { return m_hideFirstTrack; }
        int numOfTracks() const;

        bool normalize() const { return m_normalize; }

        AudioTrack* firstTrack() const;
        AudioTrack* lastTrack() const;

        /**
         * Slow.
         * \return the AudioTrack with track number trackNum (starting at 1) or 0 if trackNum > numOfTracks()
         */
        AudioTrack* getTrack( int trackNum );

        /**
         * Creates a new audiofile inside this doc which has no track yet.
         */
        AudioFile* createAudioFile( const KUrl& url );

        /** get the current size of the project */
        KIO::filesize_t size() const;
        Msf length() const;

        // CD-Text
        bool cdText() const { return m_cdText; }
        QString title() const { return m_cdTextData.title(); }
        QString artist() const { return m_cdTextData.performer(); }
        QString disc_id() const { return m_cdTextData.discId(); }
        QString arranger() const { return m_cdTextData.arranger(); }
        QString songwriter() const { return m_cdTextData.songwriter(); }
        QString composer() const { return m_cdTextData.composer(); }
        QString upc_ean() const { return m_cdTextData.upcEan(); }
        QString cdTextMessage() const { return m_cdTextData.message(); }

        /**
         * Create complete CD-Text including the tracks' data.
         */
        Device::CdText cdTextData() const;

        int audioRippingParanoiaMode() const { return m_audioRippingParanoiaMode; }
        int audioRippingRetries() const { return m_audioRippingRetries; }
        bool audioRippingIgnoreReadErrors() const { return m_audioRippingIgnoreReadErrors; }

        /**
         * Represent the structure of the doc as CD Table of Contents.
         */
        Device::Toc toToc() const;

        BurnJob* newBurnJob( JobHandler*, QObject* parent = 0 );

        /**
         * Shows dialogs.
         */
        void informAboutNotFoundFiles();

        /**
         * returns the new after track, ie. the the last added track or null if
         * the import failed.
         *
         * This is a blocking method.
         *
         * \param cuefile The Cuefile to be imported
         * \param after   The track after which the new tracks should be inserted
         * \param decoder The decoder to be used for the new tracks. If 0 a new one will be created.
         *
         * BE AWARE THAT THE DECODER HAS TO FIT THE AUDIO FILE IN THE CUE.
         */
        AudioTrack* importCueFile( const QString& cuefile, AudioTrack* after, AudioDecoder* decoder = 0 );

        /**
         * Create a decoder for a specific url. If another AudioFileSource with this
         * url is already part of this project the associated decoder is returned.
         *
         * In the first case the decoder will not be initialized yet (AudioDecoder::analyseFile
         * is not called yet).
         *
         * \param url The url for which a decoder is requested.
         * \param reused If not null this variable is set to true if the decoder is already in
         *               use and AudioDecoder::analyseFile() does not have to be called anymore.
         */
        AudioDecoder* getDecoderForUrl( const KUrl& url, bool* reused = 0 );

        static bool readPlaylistFile( const KUrl& url, KUrl::List& playlist );

    public Q_SLOTS:
        void addUrls( const KUrl::List& );
        void addTrack( const KUrl&, int );
        void addTracks( const KUrl::List&, int );
        /**
         * Adds a track without any testing
         *
         * Slow because it uses getTrack.
         */
        void addTrack( AudioTrack* track, int position = 0 );

        void addSources( AudioTrack* parent, const KUrl::List& urls, AudioDataSource* sourceAfter = 0 );

        void removeTrack( AudioTrack* );
        void moveTrack( AudioTrack* track, AudioTrack* after );

        void setHideFirstTrack( bool b ) { m_hideFirstTrack = b; }
        void setNormalize( bool b ) { m_normalize = b; }

        // CD-Text
        void writeCdText( bool b ) { m_cdText = b; }
        void setTitle( const QString& v );
        void setArtist( const QString& v );
        void setPerformer( const QString& v );
        void setDisc_id( const QString& v );
        void setArranger( const QString& v );
        void setSongwriter( const QString& v );
        void setComposer( const QString& v );
        void setUpc_ean( const QString& v );
        void setCdTextMessage( const QString& v );

        // Audio-CD Ripping
        void setAudioRippingParanoiaMode( int i ) { m_audioRippingParanoiaMode = i; }
        void setAudioRippingRetries( int r ) { m_audioRippingRetries = r; }
        void setAudioRippingIgnoreReadErrors( bool b ) { m_audioRippingIgnoreReadErrors = b; }

        void removeCorruptTracks();

    private Q_SLOTS:
        void slotTrackChanged( K3b::AudioTrack* );
        void slotTrackRemoved( K3b::AudioTrack* );

    Q_SIGNALS:
        void trackAdded( K3b::AudioTrack* );
        void trackChanged( K3b::AudioTrack* );
        void trackRemoved( K3b::AudioTrack* );

        // signals for the model
        void aboutToRemoveTrack( K3b::AudioTrack* );
        void sourceAdded( K3b::AudioTrack*, int position );
        void sourceRemoved( K3b::AudioTrack* );
        void aboutToRemoveSource( K3b::AudioTrack*, int position );

    protected:
        /** reimplemented from Doc */
        bool loadDocumentData( QDomElement* );
        /** reimplemented from Doc */
        bool saveDocumentData( QDomElement* );

        QString typeString() const;

    private:
        // the stuff for adding files
        // ---------------------------------------------------------
        AudioTrack* createTrack( const KUrl& url );

        /**
         * Handle directories and M3u files
         */
        KUrl::List extractUrlList( const KUrl::List& urls );
        // ---------------------------------------------------------

        /**
         * Used by AudioTrack to update the track list
         */
        void setFirstTrack( AudioTrack* track );
        /**
         * Used by AudioTrack to update the track list
         */
        void setLastTrack( AudioTrack* track );

        /**
         * Used by AudioFile to tell the doc that it does not need the decoder anymore.
         */
        void decreaseDecoderUsage( AudioDecoder* );
        void increaseDecoderUsage( AudioDecoder* );

        AudioTrack* m_firstTrack;
        AudioTrack* m_lastTrack;

        bool m_hideFirstTrack;
        bool m_normalize;

        KUrl::List m_notFoundFiles;
        KUrl::List m_unknownFileFormatFiles;

        // CD-Text
        // --------------------------------------------------
        Device::CdText m_cdTextData;
        bool m_cdText;
        // --------------------------------------------------

        // Audio ripping
        int m_audioRippingParanoiaMode;
        int m_audioRippingRetries;
        bool m_audioRippingIgnoreReadErrors;

        //
        // decoder housekeeping
        // --------------------------------------------------
        // used to check if we may delete a decoder
        QMap<AudioDecoder*, int> m_decoderUsageCounterMap;
        // used to check if we already have a decoder for a specific file
        QMap<QString, AudioDecoder*> m_decoderPresenceMap;

        class Private;
        Private* d;
    };
}

#endif
