/**
 * @file libcomprex/io.h Input/output functions
 * 
 * $Id: io.h,v 1.10 2003/03/06 04:36:26 chipx86 Exp $
 *
 * @Copyright (C) 2001-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _LIBCOMPREX_IO_H_
#define _LIBCOMPREX_IO_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <libcomprex/file.h>
#include <libcomprex/fp.h>
#include <libcomprex/types.h>

/**
 * Creates a file structure from a local file.
 *
 * @param filename The file to base the structure off of.
 * 
 * @return The file structure, or @c NULL if the file was not found.
 */
CxFile *cxMakeFile(const char *filename);

/**
 * Opens a file from a local or remote filesystem, or in an archive.
 *
 * @param path The path to the file.
 * @param mode The access mode.
 *
 * @return The file pointer, if the file exists. @c NULL otherwise.
 *
 * @see cxOpenBuffer()
 * @see cxOpenFileHandle()
 * @see cxOpenStream()
 * @see cxClose()
 */
CxFP *cxOpenFile(const char *path, CxAccessMode mode);

/**
 * Opens a file from an existing file handle (CxFile structure).
 *
 * @param file The file handle.
 * @param mode The access mode.
 *
 * @return The file pointer, or @c NULL if it's unable to open.
 *
 * @see cxOpenBuffer()
 * @see cxOpenFile()
 * @see cxOpenStream()
 * @see cxClose()
 */
CxFP *cxOpenFileHandle(CxFile *file, CxAccessMode mode);

/**
 * Opens an existing @c FILE * stream.
 *
 * @param stream The stream to read from.
 *
 * @return The new file pointer, or @c NULL on error.
 *
 * @see cxOpenBuffer()
 * @see cxOpenFile()
 * @see cxOpenFileHandle()
 * @see cxClose()
 */
CxFP *cxOpenStream(FILE *stream);

/**
 * Opens a stream from a buffer.
 *
 * @param buffer The buffer.
 * @param size   The size of the buffer.
 * 
 * @return The file pointer, or @c NULL on error.
 *
 * @see cxOpenFile()
 * @see cxOpenFileHandle()
 * @see cxOpenStream()
 * @see cxClose()
 */
CxFP *cxOpenBuffer(char *buffer, size_t size);

/**
 * Closes a file pointer.
 *
 * The CxFP structure will be freed after this is called.
 *
 * @param fp The file pointer to close.
 *
 * @see cxOpenBuffer()
 * @see cxOpenFile()
 * @see cxOpenFileHandle()
 * @see cxOpenStream()
 */
void cxClose(CxFP *fp);

/**
 * Reads data from a file pointer.
 *
 * @param ptr   The buffer which will store the data.
 * @param size  The size of an element of data.
 * @param nmemb The number of elements of size @a size to receive.
 * @param fp    The file pointer to read from.
 *
 * @return The number of elements read, or 0 on error or end of file.
 *
 * @see cxRewind()
 * @see cxSeek()
 * @see cxTell()
 * @see cxWrite()
 */
size_t cxRead(void *ptr, size_t size, size_t nmemb, CxFP *fp);

/**
 * Writes data to a file pointer.
 *
 * @param ptr   The data to write.
 * @param size  The size of an element of data.
 * @param nmemb The number of elements of size @a size to receive.
 * @param fp    The file pointer to write to.
 *
 * @return The number of elements written, or 0 on error.
 *
 * @see cxRead()
 * @see cxSeek()
 * @see cxTell()
 * @see cxRewind()
 */
size_t cxWrite(const void *ptr, size_t size, size_t nmemb, CxFP *fp);

/**
 * Reads in a line from from a file pointer.
 *
 * No more than @a size - 1 bytes will be read in. If a newline is
 * encountered, it is stored and no further data will be read in. The
 * resulting buffer is NUL-terminated.
 *
 * @param buffer The buffer to store the line in.
 * @param size   The maximum number of bytes to read it. Also the size of
 *               the buffer.
 * @param fp     The file pointer.
 *
 * @return @a buffer on success, or @c NULL on error or end of file.
 */
char *cxGets(char *buffer, size_t size, CxFP *fp);

/**
 * Seeks to a position in the file.
 *
 * @param fp     The file pointer.
 * @param offset The relative or absolute offset to seek to.
 * @param whence @c SEEK_SET, @c SEEK_CUR, or @c SEEK_END.
 *
 * @see cxRewind()
 * @see cxTell()
 */
void cxSeek(CxFP *fp, long offset, int whence);

/**
 * Rewinds to the beginning of the stream.
 *
 * @param fp The file pointer.
 *
 * @see cxSeek()
 * @see cxTell()
 */
void cxRewind(CxFP *fp);

/**
 * Returns the current position in the file.
 *
 * @param fp The file pointer.
 *
 * @return The current position in the file.
 *
 * @see cxRewind()
 * @see cxSeek()
 */
long cxTell(CxFP *fp);

/**
 * Clears the error flag and end-of-file indicator.
 *
 * @param fp The file pointer.
 *
 * @see cxEof()
 * @see cxError()
 * @see cxStrError()
 */
void cxClearErr(CxFP *fp);

/**
 * Tests the end-of-file indicator.
 *
 * @param fp The file pointer.
 *
 * @return 1 if the end-of-file indicator is set; 0 otherwise.
 *
 * @see cxClearErr()
 * @see cxError()
 * @see cxStrError()
 */
int cxEof(CxFP *fp);

/**
 * Tests the error indicator of the file.
 *
 * @param fp The file pointer.
 *
 * @return The error if set, or 0 on no error.
 *
 * @see cxClearErr()
 * @see cxEof()
 * @see cxStrError()
 */
int cxError(CxFP *fp);

/**
 * Returns a string representation of the error.
 *
 * @param fp     The file pointer.
 * @param errnum The error number.
 *
 * @return The string representation of the error.
 *
 * @see cxClearErr()
 * @see cxEof()
 * @see cxError()
 */
const char *cxStrError(CxFP *fp, int errnum);

/**
 * Extracts the specified file.
 *
 * @param file     The file to extract.
 * @param destPath The destination path, or @c NULL to use the
 *                 existing filename.
 *
 * @return The status of the operation.
 */
CxStatus cxExtractFile(CxFile *file, const char *destPath);

#ifdef __cplusplus
}
#endif

#endif /* _LIBCOMPREX_IO_H_ */
