from collections import namedtuple


__all__ = ["devices", "devices_by_ids"]


class MSP430Device(namedtuple("MSP430Device", (
    "name", "jtag_id", "device_id", "ext_id",
    "core", # '430', '430X', '430Xv2'
    "type", # 'flash', 'fram'
    "features", # [] of 'quick.{flash,sram,fram}.{r,w}', 'fast_flash', 'enhanced_verify'
    "pkg_variants", # If ID depends on package, list variant(s)
))):
    __slots__ = ()

    def address_width(self):
        if self.core in ("430X", "430Xv2"):
            return 20
        else:
            return 16

    def is_430Xv2(self):
        return self.core == "430Xv2"

    def is_flash(self):
        return self.type == "flash"

    def is_frame(self):
        return self.type == "fram"

    def has(self, feature):
        return feature in self.features

    def __repr__(self):
        return (f"MSP430Device({self.name!r}, "
            f"{self.jtag_id:#02x}, {self.device_id:#04x}, "
            f"{'None' if self.ext_id is None else self.ext_id.format('#02x')}, "
            f"core={self.core!r}, type={self.type!r}, features={self.features!r}, "
            f"pkg_variants={self.pkg_variants!r})")

MSP430Device.__new__.__defaults__ = ([], None)


devices = [
    MSP430Device('AFE2xx',                   0x89, 0x0253, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash']),
    MSP430Device('F11x1(A)',                 0x89, 0xf112, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('F11x2',                    0x89, 0x1132, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('F12x',                     0x89, 0xf123, None, core='430',    type='flash', features=['quick.flash.r']),
    MSP430Device('F12x2',                    0x89, 0x1232, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('F13x/F14x/F14x1',          0x89, 0xf149, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('F15x/F16x',                0x89, 0xf169, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('F161x',                    0x89, 0xf16c, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
#    MSP430Device('F20x1',                    0x89, 0xf201, 0x01, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash']),
#    MSP430Device('F20x2',                    0x89, 0xf201, 0x02, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash']),
    MSP430Device('F20x1/G2x01/G2x11',        0x89, 0xf201, 0x01, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash']), # Merged
    MSP430Device('F20x2/G2x21/G2x31',        0x89, 0xf201, 0x02, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash']), # Merged
    MSP430Device('F20x3',                    0x89, 0xf201, 0x03, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash']),
    MSP430Device('F21x1',                    0x89, 0xf213, 0x01, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'enhanced_verify']),
    MSP430Device('F21x2',                    0x89, 0xf213, 0x02, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash', 'enhanced_verify']),
    MSP430Device('F22x2/F22x4/G2x44',        0x89, 0xf227, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash', 'enhanced_verify']),
    MSP430Device('F23x/F24x/F24x1/F2410',    0x89, 0xf249, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash', 'enhanced_verify']),
    MSP430Device('F23x0',                    0x89, 0xf237, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash', 'enhanced_verify']),
    MSP430Device('F241x/F261x',              0x89, 0xf26f, None, core='430X',   type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash', 'enhanced_verify']),
    MSP430Device('F412/F413',                0x89, 0xf413, None, core='430',    type='flash', features=['quick.flash.r']),
#    MSP430Device('F415/F417',                0x89, 0xf427, 0x57, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('F415/F417/FW42x',          0x89, 0xf427, 0x57, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']), # Merged
    MSP430Device('F41x2',                    0x89, 0x4152, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash', 'enhanced_verify']),
    MSP430Device('F(E)42x',                  0x89, 0xf427, 0x45, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('F(E)42xA',                 0x89, 0x427a, 0x45, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash']),
    MSP430Device('F(G)42x0',                 0x89, 0xf427, 0x47, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('F43x',                     0x89, 0xf437, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r'], pkg_variants=['80 pins']),
    MSP430Device('F43x/F44x',                0x89, 0xf449, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r'], pkg_variants=['100 pins']),
    MSP430Device('F471xx',                   0x89, 0xf47f, None, core='430X',   type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash', 'enhanced_verify']),
    MSP430Device('F47xx',                    0x89, 0xf449, 0x02, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash', 'enhanced_verify']),
    MSP430Device('FE42x2',                   0x89, 0x4252, 0x45, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('FG43x',                    0x89, 0xf439, 0x47, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('F(G)461x/F461x1',          0x89, 0xf46f, 0x47, core='430X',   type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash', 'enhanced_verify']),
    MSP430Device('F(G)47x',                  0x89, 0xf479, 0x47, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash', 'enhanced_verify']),
#    MSP430Device('FW428',                    0x89, 0xf429, 0x57, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
#    MSP430Device('FW429',                    0x89, 0xf429, 0x57, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('FW429/FW428',              0x89, 0xf429, 0x57, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']), # Merged
#    MSP430Device('FW42x',                    0x89, 0xf427, 0x57, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
#    MSP430Device('G2x01/G2x11',              0x89, 0xf201, 0x01, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash']),
#    MSP430Device('G2x21/G2x31',              0x89, 0xf201, 0x02, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash']),
    MSP430Device('G2xx2',                    0x89, 0x2452, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash']),
    MSP430Device('G2xx3',                    0x89, 0x2553, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash']),
    MSP430Device('G2xx5',                    0x89, 0x2955, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash']),
    MSP430Device('TCH5E',                    0x89, 0x255c, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r', 'fast_flash']),
    MSP430Device('CC430F5123',               0x91, 0x3c81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('CC430F5125',               0x91, 0x3b81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('CC430F5133',               0x91, 0x5133, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('CC430F5135',               0x91, 0x5135, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('CC430F5137',               0x91, 0x5137, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('CC430F5143',               0x91, 0x3a81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('CC430F5145',               0x91, 0x3981, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('CC430F5147',               0x91, 0x3881, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('CC430F6125',               0x91, 0x6125, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('CC430F6126',               0x91, 0x6126, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('CC430F6127',               0x91, 0x6127, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('CC430F6135',               0x91, 0x6135, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('CC430F6137',               0x91, 0x6137, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('CC430F6143',               0x91, 0x3781, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('CC430F6145',               0x91, 0x3681, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('CC430F6147',               0x91, 0x3581, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5131',              0x91, 0x2680, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5132',              0x91, 0x2880, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5151',              0x91, 0x2a80, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5152',              0x91, 0x2c80, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5171',              0x91, 0x2e80, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5172',              0x91, 0x3080, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5212',              0x91, 0x4081, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5213',              0x91, 0x4181, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5214',              0x91, 0x4281, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5217',              0x91, 0x4581, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5218',              0x91, 0x4681, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5219',              0x91, 0x4781, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5222',              0x91, 0x4a81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5223',              0x91, 0x4b81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5224',              0x91, 0x4c81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5227',              0x91, 0x4f81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5228',              0x91, 0x5081, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5229',              0x91, 0x5181, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5232',              0x91, 0xfa81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5234',              0x91, 0xf981, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5237',              0x91, 0xf881, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5239',              0x91, 0xf781, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5242',              0x91, 0xf681, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5244',              0x91, 0xf581, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5247',              0x91, 0xf481, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5249',              0x91, 0xf381, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5252',              0x91, 0x0682, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5253',              0x91, 0x0582, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5254',              0x91, 0x0482, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5255',              0x91, 0x0382, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5256',              0x91, 0x0282, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5257',              0x91, 0x0182, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5258',              0x91, 0x0082, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5259',              0x91, 0xff81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5304',              0x91, 0x1281, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5308',              0x91, 0x1381, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5309',              0x91, 0x1481, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5310',              0x91, 0x1581, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5324',              0x91, 0x1681, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5325',              0x91, 0x1781, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5326',              0x91, 0x1881, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5327',              0x91, 0x1981, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5328',              0x91, 0x1a81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5329',              0x91, 0x1b81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5333',              0x91, 0x2581, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5335',              0x91, 0x2781, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5336',              0x91, 0x2881, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5338',              0x91, 0x2a81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5340',              0x91, 0x1c81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5341',              0x91, 0x1d81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5342',              0x91, 0x1e81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5357',              0x91, 0x3481, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5358',              0x91, 0x3381, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5359',              0x91, 0x3281, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5418',              0x91, 0x5418, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5418A',             0x91, 0x0080, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5419',              0x91, 0x5419, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5419A',             0x91, 0x0180, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5435',              0x91, 0x5435, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5435A',             0x91, 0x0280, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5436',              0x91, 0x5436, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5436A',             0x91, 0x0380, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5437',              0x91, 0x5437, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5437A',             0x91, 0x0480, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5438',              0x91, 0x5438, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5438A',             0x91, 0x0580, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5500',              0x91, 0x3b80, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5501',              0x91, 0x3280, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5502',              0x91, 0x3380, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5503',              0x91, 0x3480, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5504',              0x91, 0x3580, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5505',              0x91, 0x3680, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5506',              0x91, 0x3780, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5507',              0x91, 0x3880, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5508',              0x91, 0x3980, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5509',              0x91, 0x3a80, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5510',              0x91, 0x3180, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5513',              0x91, 0x5513, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5514',              0x91, 0x5514, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5515',              0x91, 0x5515, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5517',              0x91, 0x5517, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5519',              0x91, 0x5519, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5521',              0x91, 0x5521, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5522',              0x91, 0x5522, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5524',              0x91, 0x5524, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5525',              0x91, 0x5525, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5526',              0x91, 0x5526, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5527',              0x91, 0x5527, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5528',              0x91, 0x5528, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5529',              0x91, 0x5529, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5630',              0x91, 0x3c80, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5631',              0x91, 0x3e80, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5632',              0x91, 0x4080, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5633',              0x91, 0x4280, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5634',              0x91, 0x4480, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5635',              0x91, 0x0e80, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5636',              0x91, 0x1080, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5637',              0x91, 0x1280, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5638',              0x91, 0x1480, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5658',              0x91, 0x3181, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F5659',              0x91, 0x3081, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6433',              0x91, 0x1f81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6435',              0x91, 0x2181, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6436',              0x91, 0x2281, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6438',              0x91, 0x2481, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6458',              0x91, 0x2e81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6459',              0x91, 0x2d81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6630',              0x91, 0x4680, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6631',              0x91, 0x4880, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6632',              0x91, 0x4a80, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6633',              0x91, 0x4c80, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6634',              0x91, 0x4e80, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6635',              0x91, 0x1680, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6636',              0x91, 0x1880, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6637',              0x91, 0x1a80, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6638',              0x91, 0x1c80, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6658',              0x91, 0x2c81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6659',              0x91, 0x2b81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6700',              0x91, 0x5480, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6701',              0x91, 0x5580, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6702',              0x91, 0x5680, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6703',              0x91, 0x5780, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6720',              0x91, 0x5880, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6720A',             0x91, 0x7682, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6721',              0x91, 0x5980, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6721A',             0x91, 0x7782, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6722',              0x91, 0x6080, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6723',              0x91, 0x6180, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6723A',             0x91, 0x7982, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6724',              0x91, 0x6d81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6724A',             0x91, 0x7a82, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6725',              0x91, 0x6e81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6725A',             0x91, 0x7b82, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6726',              0x91, 0x6f81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6726A',             0x91, 0x7c82, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6730',              0x91, 0x6280, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6730A',             0x91, 0x8082, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6731',              0x91, 0x6380, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6731A',             0x91, 0x8182, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6732',              0x91, 0x6480, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6733',              0x91, 0x6580, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6733A',             0x91, 0x8382, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6734',              0x91, 0x6a81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6734A',             0x91, 0x8482, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6735',              0x91, 0x6b81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6735A',             0x91, 0x8582, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6736',              0x91, 0x6c81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6736A',             0x91, 0x8682, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6745',              0x91, 0x8881, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67451',             0x91, 0x9781, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67451A',            0x91, 0x2582, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6745A',             0x91, 0x1682, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6746',              0x91, 0x8981, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67461',             0x91, 0x9881, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67461A',            0x91, 0x2682, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6746A',             0x91, 0x1782, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6747',              0x91, 0x8a81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67471',             0x91, 0x9981, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67471A',            0x91, 0x2782, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6747A',             0x91, 0x1882, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6748',              0x91, 0x8b81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67481',             0x91, 0x9a81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67481A',            0x91, 0x2882, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6748A',             0x91, 0x1982, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6749',              0x91, 0x8c81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67491',             0x91, 0x9b81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67491A',            0x91, 0x2982, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6749A',             0x91, 0x1a82, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67621',             0x91, 0x3882, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67621A',            0x91, 0x8782, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67641',             0x91, 0x3982, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67641A',            0x91, 0x8882, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6765',              0x91, 0x8d81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67651',             0x91, 0x9c81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67651A',            0x91, 0x2a82, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6765A',             0x91, 0x1b82, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6766',              0x91, 0x8e81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67661',             0x91, 0x9d81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67661A',            0x91, 0x2b82, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6766A',             0x91, 0x1c82, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6767',              0x91, 0x8f81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67671',             0x91, 0x9e81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67671A',            0x91, 0x2c82, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6767A',             0x91, 0x1d82, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6768',              0x91, 0x9081, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67681',             0x91, 0x9f81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67681A',            0x91, 0x2d82, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6768A',             0x91, 0x1e82, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6769',              0x91, 0x9181, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67691',             0x91, 0xa081, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67691A',            0x91, 0x2e82, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6769A',             0x91, 0x1f82, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6775',              0x91, 0x9281, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67751',             0x91, 0xa181, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67751A',            0x91, 0x2f82, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6775A',             0x91, 0x2082, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6776',              0x91, 0x9381, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67761',             0x91, 0xa281, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67761A',            0x91, 0x3082, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6776A',             0x91, 0x2182, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6777',              0x91, 0x9481, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67771',             0x91, 0xa381, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67771A',            0x91, 0x3182, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6777A',             0x91, 0x2282, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6778',              0x91, 0x9581, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67781',             0x91, 0xa481, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67781A',            0x91, 0x3282, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6778A',             0x91, 0x2382, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6779',              0x91, 0x9681, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67791',             0x91, 0xa581, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F67791A',            0x91, 0x3382, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430F6779A',             0x91, 0x2482, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430FG6425',             0x91, 0x3782, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430FG6426',             0x91, 0x3682, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430FG6625',             0x91, 0x3582, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430FG6626',             0x91, 0x3482, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430SL5438A',            0x91, 0xee81, None, core='430Xv2', type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430FR2000',             0x98, 0x2083, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2032',             0x98, 0x7882, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2033',             0x98, 0x7582, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2100',             0x98, 0x2183, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2110',             0x98, 0xfb82, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2111',             0x98, 0xfa82, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2153',             0x98, 0x1d83, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2155',             0x98, 0x1e83, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2310',             0x98, 0xf182, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2311',             0x98, 0xf082, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2353',             0x98, 0x0d83, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2355',             0x98, 0x0c83, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2422',             0x98, 0x1183, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2433',             0x98, 0x4082, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2475',             0x98, 0x2b83, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2476',             0x98, 0x2a83, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2512',             0x98, 0x1c83, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2522',             0x98, 0x1083, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2532',             0x98, 0x3f82, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2533',             0x98, 0x3d82, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2632',             0x98, 0x3e82, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2633',             0x98, 0x3c82, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2675',             0x98, 0x2983, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR2676',             0x98, 0x2883, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR4131',             0x98, 0xf281, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR4132',             0x98, 0xf181, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR4133',             0x98, 0xf081, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5041',             0x99, 0x0f83, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5043',             0x99, 0x1783, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR50431',            0x99, 0x1883, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5720',             0x91, 0x7081, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5721',             0x91, 0x7780, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5722',             0x91, 0x7181, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5723',             0x91, 0x7281, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5724',             0x91, 0x7381, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5725',             0x91, 0x7880, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5726',             0x91, 0x7481, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5727',             0x91, 0x7980, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5728',             0x91, 0x7a80, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5729',             0x91, 0x7b80, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5730',             0x91, 0x7c80, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5731',             0x91, 0x7e80, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5732',             0x91, 0x7581, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5733',             0x91, 0x7f80, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5734',             0x91, 0x0081, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5735',             0x91, 0x7681, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5736',             0x91, 0x7781, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5737',             0x91, 0x0181, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5738',             0x91, 0x0281, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5739',             0x91, 0x0381, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5847',             0x99, 0x5381, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5848',             0x99, 0x5481, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5849',             0x99, 0x5581, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5857',             0x99, 0x5781, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5858',             0x99, 0x5881, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5859',             0x99, 0x5981, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5867',             0x99, 0x5b81, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5868',             0x99, 0x5c81, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5869',             0x99, 0x5d81, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5870',             0x99, 0x5e82, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5872',             0x99, 0x6082, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5887',             0x99, 0xc181, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5888',             0x99, 0xc281, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5889',             0x99, 0xc381, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5922',             0x99, 0x6182, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w'], pkg_variants=['DGG']),
    MSP430Device('MSP430FR5922',             0x99, 0x6282, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w'], pkg_variants=['PM', 'RGC']),
    MSP430Device('MSP430FR5947',             0x99, 0x5f81, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5948',             0x99, 0x6081, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5949',             0x99, 0x6181, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5957',             0x99, 0x6381, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5958',             0x99, 0x6481, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5959',             0x99, 0x6581, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5962',             0x99, 0xa682, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5964',             0x99, 0xa482, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5967',             0x99, 0x6781, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5968',             0x99, 0x6881, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5969',             0x99, 0x6981, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5970',             0x99, 0x5b82, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5972',             0x99, 0x5d82, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5986',             0x99, 0xdf81, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5987',             0x99, 0xa981, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5988',             0x99, 0xaa81, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5989',             0x99, 0xab81, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5992',             0x99, 0xa382, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR5994',             0x99, 0xa182, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR59941',            0x99, 0xa282, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6035',             0x99, 0xed82, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR60371',            0x99, 0xef82, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6037',             0x99, 0xec82, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6041',             0x99, 0x1483, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6043',             0x99, 0x1283, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR60431',            0x99, 0x1a83, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6045',             0x99, 0xeb82, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR60471',            0x99, 0xee82, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6047',             0x99, 0xea82, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6820',             0x99, 0x5582, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w'], pkg_variants=['DGG']),
    MSP430Device('MSP430FR6820',             0x99, 0x5682, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w'], pkg_variants=['PM', 'RGC']),
    MSP430Device('MSP430FR6822',             0x99, 0x5982, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w'], pkg_variants=['DGG']),
    MSP430Device('MSP430FR6822',             0x99, 0x5a82, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w'], pkg_variants=['PM', 'RGC']),
    MSP430Device('MSP430FR6870',             0x99, 0x4c82, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6872',             0x99, 0x4e82, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6877',             0x99, 0xc481, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6879',             0x99, 0xc681, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6887',             0x99, 0xbe81, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6888',             0x99, 0xbf81, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6889',             0x99, 0xc081, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6920',             0x99, 0x4f82, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w'], pkg_variants=['DGG']),
    MSP430Device('MSP430FR6920',             0x99, 0x5082, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w'], pkg_variants=['PM', 'RGC']),
    MSP430Device('MSP430FR6922',             0x99, 0x5382, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w'], pkg_variants=['DGG']),
    MSP430Device('MSP430FR6922',             0x99, 0x5482, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w'], pkg_variants=['PM', 'RGC']),
    MSP430Device('MSP430FR6927',             0x99, 0xb281, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6928',             0x99, 0xb381, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6970',             0x99, 0x4982, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6972',             0x99, 0x4b82, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6977',             0x99, 0xac81, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6979',             0x99, 0xae81, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6987',             0x99, 0xa681, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6988',             0x99, 0xa781, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
    MSP430Device('MSP430FR6989',             0x99, 0xa881, None, core='430Xv2', type='fram',  features=['quick.sram.r', 'quick.sram.w', 'quick.fram.r', 'quick.fram.w']),
#    MSP430Device('MSP430i2020',              0x89, 0x4020, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
#    MSP430Device('MSP430i2021',              0x89, 0x4020, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
#    MSP430Device('MSP430i2030',              0x89, 0x4020, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
#    MSP430Device('MSP430i2031',              0x89, 0x4020, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
#    MSP430Device('MSP430i2040',              0x89, 0x4020, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
#    MSP430Device('MSP430i2041',              0x89, 0x4020, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']),
    MSP430Device('MSP430i20xx',              0x89, 0x4020, None, core='430',    type='flash', features=['quick.sram.r', 'quick.sram.w', 'quick.flash.r']), # Merged
]


devices_by_ids = {}
for dev in devices:
    devices_by_ids.setdefault(dev.jtag_id, {}).setdefault(dev.device_id, {})[dev.ext_id] = dev
