package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.42.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.42.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@(!`0````````````,`MP`!````T"````````!``````````$BU````
M`````````$``.``(`$``&@`9``8````$````0`````````!``````````$``
M````````P`$```````#``0````````@``````````P````0``````@``````
M```"``````````(````````7`````````!<``````````0`````````!````
M!0```````````````````````````````````*R=````````K)T`````````
M``$```````$````&````6*T```````!8K0$``````%BM`0``````*`4`````
M``"P!0```````````0```````@````8```!PK0```````'"M`0``````<*T!
M``````#@`0```````.`!````````"``````````$````!````!@"````````
M&`(````````8`@```````"P`````````+``````````$`````````%#E=&0$
M````F)0```````"8E````````)B4````````!`$````````$`0````````0`
M````````4N5T9`0```!8K0```````%BM`0``````6*T!``````"H`@``````
M`*@"`````````0`````````O=7-R+VQI8F5X96,O;&0N96QF7W-O```'````
M!`````$```!.971"4T0```#RX34$````!`````,```!085@```````````!A
M````9P``````````````,`````T``````````````#<````.````/P``````
M```O``````````,```!+````8``````````>````2@````8````<````%```
M``````!!````50```%P`````````%@```#L```!7``````````````!C````
M90```"X````2````5@```$,````5````6P```$0`````````1P```"$````T
M````6@```"4````I``````````<```!-``````````L````K````9@```%D`
M``!)````'P```$X```!D`````````%X`````````%P```"T```!,````````
M````````````$0```#(``````````````%\````J``````````D`````````
M(P```#@`````````3P````4```!%````&@```#H`````````````````````
M````2``````````Y````4@```&$`````````8@``````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````$P``````````````````````````````````````````
M`````````"(````@`````````"@`````````"@```"8````G````+```````
M````````#``````````;````&0```!@``````````````!``````````,P``
M`#$````V````'0`````````````````````````$``````````````!`````
M)````````````````````````````````````#X```!"````"```````````
M`````````#T```!1``````````````!4```````````````\````````````
M``!0`````````$8````/`````````%,```!8````-0```%T`````````````
M``````````````````````````````````,`"@#0(```````````````````
M``````,`$`!HK0$`````````````````C````!(`````````````````````
M````#````!(`````````````````````````%````!(`````````````````
M````````:@,``!(`"@"(*P````````@`````````'````"(`````````````
M````````````*P```!(`````````````````````````$0,``!(`````````
M````````````````0P,``!``%0"`L@$`````````````````,0```!(`````
M````````````````````>P,``!(`"@"(,@```````"@`````````.````!(`
M````````````````````````'@,``!(`````````````````````````0```
M`!(`````````````````````````E@,``!(`"@#(.P```````&@`````````
M?0```!(`````````````````````````4@,``!``%0`(LP$`````````````
M````2````!(`````````````````````````H@,``!``%`"`L@$`````````
M````````6````!(`````````````````````````9P```!(`````````````
M````````````;@```!(`````````````````````````>````!(`````````
M````````````````@@```!(`````````````````````````40,``!``%0`(
MLP$`````````````````BP```!(`````````````````````````#0(``!(`
M````````````````````````D0```!(`````````````````````````F```
M`!(`````````````````````````I@```!(`````````````````````````
MMP```!$`````````````````````````FP(``"(`````````````````````
M````Q````!(`````````````````````````J0,``!(`"@#8+0```````"P`
M````````M@,``!(`"@`(+@```````'@`````````P@,``!(`"@"@,P``````
M`-0%````````T@```!8`````````````````````````Y0```!(`````````
M````````````````,`,``!(`````````````````````````\````!(`````
M````````````````````^0```!$`%0``LP$```````@``````````0$``!(`
M````````````````````````"@$``!(`````````````````````````!P,`
M`!(`````````````````````````%@$``!(`````````````````````````
M'0$``!(`````````````````````````SP,``!(`"@!H30```````!P!````
M````,`$``!$`````````````````````````0@$``!(`````````````````
M````````4@$``!(`````````````````````````8`$``!(`````````````
M````````````90$``!(`````````````````````````V@,``!$`%`!XL@$`
M``````@`````````=`$``!(`````````````````````````@`$``!(`````
M````````````````````E0$``!(`````````````````````````H`$``!(`
M````````````````````````90,``!``%0`(LP$`````````````````P0(`
M`!(`````````````````````````LP$``!(`````````````````````````
M%00``!(`"@#0(`````````@`````````P0$``!(`````````````````````
M````R0$``!(`````````````````````````U0$``!(`````````````````
M````````70,``!``%0`(LP$`````````````````Y`,``!(`"@"`+@``````
M`'@!````````VP$``!(`````````````````````````Y@$``!(`````````
M````````````````\0,``!(`"@"(3@```````+@!````````]P$``!(`````
M````````````````````^P,``!``%0"`L@$`````````````````!P0``!(`
M"@#H*@```````$``````````#P0``!(`"@#X4````````"@"````````_@$`
M`!(`````````````````````````@P(``"(`````````````````````````
M%`0``!(`"@#0(`````````@`````````!0(``!(`````````````````````
M````'`0``!(`"@"0*P```````$0"````````*00``!(`"@"P,@```````.P`
M````````#`(``!(`````````````````````````.P0``!(`"@#X+P``````
M`'``````````%0(``!(`````````````````````````T`(``!(`````````
M````````````````)@(``!(`````````````````````````200``!(`"@!H
M,````````)P`````````-0(``!(`````````````````````````/@(``!(`
M````````````````````````1`(``!(`````````````````````````^@(`
M`!(`````````````````````````20(``!(`````````````````````````
M[0(``!$`%0"(L@$```````@`````````40(``!(`````````````````````
M````)0,``!(`````````````````````````6`(``!(`````````````````
M````````400``!(`"@`030```````%0`````````7P(``!$`````````````
M````````````X@(``!$`%`#XL0$```````@`````````6@0``!(`"@!X.0``
M`````$P"````````*0$``!(`````````````````````````9`(``!(`````
M````````````````````-P,``!(``````````````````````````&QI8G!E
M<FPN<V\`<F5A;&QO8P!M96UM;W9E`%]?8WAA7V9I;F%L:7IE`')M9&ER`'-T
M<F-H<@!S=')N8VUP`'-T<G)C:'(`8F]O=%]$>6YA3&]A9&5R`'!E<FQ?8V]N
M<W1R=6-T`'5N;&EN:P!?7VQS=&%T-3``<&5R;%]F<F5E`%]?<W1A=#4P`%]E
M>&ET`&UE;6-P>0!P97)L7V1E<W1R=6-T`%]?<W1A8VM?8VAK7V9A:6P`4$Q?
M9&]?=6YD=6UP`%!E<FQ?<W9?<V5T<'8`4$Q?8W5R<F5N=%]C;VYT97AT`'!E
M<FQ?86QL;V,`8VQO<V5D:7(`96YV:7)O;@!R96%D;&EN:P!?7V]P96YD:7(S
M,`!M96US970`4&5R;%]S869E<WES;6%L;&]C`%]?<W1A8VM?8VAK7V=U87)D
M`%!E<FQ?9W9?9F5T8VAP=@!?7W-P<FEN=&9?8VAK`')E860`4&5R;%]S>7-?
M:6YI=#,`4&5R;%]G971?<W8`4&5R;%]C<F]A:U]N;V-O;G1E>'0`4&5R;%]N
M97=84P!097)L7V%T9F]R:U]U;FQO8VL`4&5R;%]S>7-?=&5R;0!?7V5R<FYO
M`%!E<FQ?;6=?<V5T`&UK9&ER`'!E<FQ?<&%R<V4`4&5R;%]N97=84U]F;&%G
M<P!A8V-E<W,`9V5T<&ED`&UE;6-H<@!V9G!R:6YT9@!097)L7V%T9F]R:U]L
M;V-K`%!E<FQ?<W9?<V5T<'9N`'!E<FQ?<G5N`&QS965K`&]P96X`<W!R:6YT
M9@!G971U:60`<W1R;&5N`%]?<T8`<W1R8VUP`&QI8FTN<V\N,`!L:6)C<GEP
M="YS;RXQ`%]?9&5R96=I<W1E<E]F<F%M95]I;F9O`%]?<F5G:7-T97)?9G)A
M;65?:6YF;P!L:6)P=&AR96%D+G-O+C$`<'1H<F5A9%]A=&9O<FL`<W1R9'5P
M`&QI8F,N<V\N,3(`7U]P<F]G;F%M90!?7W!S7W-T<FEN9W,`7U]G971P=W5I
M9#4P`%]?<WES8V%L;`!?7W-T<&-P>5]C:&L`871E>&ET`%]L:6)C7VEN:70`
M<W1R=&]K`%]?<F5A9&1I<C,P`%]?8G-S7W-T87)T7U\`7U]B<W-?96YD7U\`
M7U]E;F1?7P!?96YD`'!A<E]C=7)R96YT7V5X96,`9V5T7W5S97)N86UE7V9R
M;VU?9V5T<'=U:60`<&%R7V-L96%N=7``7V5D871A`'!A<E]B87-E;F%M90!P
M87)?9&ER;F%M90!P87)?;6MT;7!D:7(`<VAA7W5P9&%T90!M>5]P87)?<&P`
M<&%R7VEN:71?96YV`'-H85]F:6YA;`!?7V)S<U]S=&%R=`!X<U]I;FET`&UA
M:6X`7U]S=&%R=`!P87)?9FEN9'!R;V<`<&%R7W-E='5P7VQI8G!A=&@`<&%R
M7V5N=E]C;&5A;@!P87)?9&EE`'-H85]I;FET`%A37TEN=&5R;F%L<U]005)?
M0D]/5``O=7-R+W!K9R]L:6(O<&5R;#4O-2XT,BXP+V%A<F-H-C0M;F5T8G-D
M+71H<F5A9"UM=6QT:2]#3U)%.B]U<W(O<&MG+VQI8@!8K0$```````,$````
M````N%````````!@K0$```````,$````````0%````````!8KP$```````,$
M````````:*T!``````!PKP$```````,$````````6*T!``````"0KP$`````
M``,$`````````+,!``````"@KP$```````,$````````8*T!``````"HKP$`
M``````,$````````>+(!``````#(KP$```````,$````````B+(!``````#8
MKP$```````,$````````^+$!``````#@KP$```````,$````````>#D`````
M``#XL0$```````,$````````@+(!````````L@$```````,$`````````+(!
M```````8L@$```````,$````````(%<````````@L@$```````,$````````
M4%,````````PL@$```````,$````````H%0````````XL@$```````,$````
M`````%<```````!`L@$```````,$````````"%<```````!(L@$```````,$
M````````$%<```````!0L@$```````,$````````&%<```````!@L@$`````
M``,$````````Z%8```````!HL@$```````,$````````\%8```````!XL@$`
M``````,$````````*%<```````!@KP$```````$$```'``````````````!H
MKP$```````$$```3``````````````!XKP$```````$$```@````````````
M``"`KP$```````$$```A``````````````"(KP$```````8$```F````````
M``````"8KP$```````$$```Q``````````````"PKP$```````$$```Z````
M``````````"XKP$```````$$``!,``````````````#`KP$```````$$``!3
M``````````````#0KP$```````$$``!A````````````````L`$```````($
M```#```````````````(L`$```````($```$```````````````0L`$`````
M``($```%```````````````8L`$```````($```'```````````````@L`$`
M``````($```(```````````````HL`$```````($```)```````````````P
ML`$```````($```+```````````````XL`$```````($```-````````````
M``!`L`$```````($```.``````````````!(L`$```````($```/````````
M``````!0L`$```````($```1``````````````!8L`$```````($```5````
M``````````!@L`$```````($```6``````````````!HL`$```````($```7
M``````````````!PL`$```````($```8``````````````!XL`$```````($
M```9``````````````"`L`$```````($```;``````````````"(L`$`````
M``($```<``````````````"0L`$```````($```=``````````````"8L`$`
M``````($```>``````````````"@L`$```````($```?``````````````"H
ML`$```````($```A``````````````"PL`$```````($```B````````````
M``"XL`$```````($```G``````````````#`L`$```````($```H````````
M``````#(L`$```````($```I``````````````#0L`$```````($```K````
M``````````#8L`$```````($```L``````````````#@L`$```````($```M
M``````````````#HL`$```````($```N``````````````#PL`$```````($
M```O``````````````#XL`$```````($```R````````````````L0$`````
M``($```S```````````````(L0$```````($```T```````````````0L0$`
M``````($```U```````````````8L0$```````($```W```````````````@
ML0$```````($```X```````````````HL0$```````($```Y````````````
M```PL0$```````($```\```````````````XL0$```````($```]````````
M``````!`L0$```````($```_``````````````!(L0$```````($``!`````
M``````````!0L0$```````($``!!``````````````!8L0$```````($``!$
M``````````````!@L0$```````($``!%``````````````!HL0$```````($
M``!'``````````````!PL0$```````($``!+``````````````!XL0$`````
M``($``!,``````````````"`L0$```````($``!.``````````````"(L0$`
M``````($``!1``````````````"0L0$```````($``!4``````````````"8
ML0$```````($``!5``````````````"@L0$```````($``!7````````````
M``"HL0$```````($``!8``````````````"PL0$```````($``!9````````
M``````"XL0$```````($``!:``````````````#`L0$```````($``!;````
M``````````#(L0$```````($``!=``````````````#0L0$```````($``!>
M``````````````#8L0$```````($``!?``````````````#@L0$```````($
M``!D``````````````#HL0$```````($``!E``````````````#PL0$`````
M``($``!F`````````````````````````/![OZG0``"P$?Y'^1#B/Y$@`A_6
M'R`#U1\@`]4?(`/5T```T!$"0/D0`@"1(`(?UM```-`1!D#Y$"(`D2`"'];0
M``#0$0I`^1!"`)$@`A_6T```T!$.0/D08@"1(`(?UM```-`1$D#Y$((`D2`"
M'];0``#0$19`^1"B`)$@`A_6T```T!$:0/D0P@"1(`(?UM```-`1'D#Y$.(`
MD2`"'];0``#0$2)`^1`"`9$@`A_6T```T!$F0/D0(@&1(`(?UM```-`1*D#Y
M$$(!D2`"'];0``#0$2Y`^1!B`9$@`A_6T```T!$R0/D0@@&1(`(?UM```-`1
M-D#Y$*(!D2`"'];0``#0$3I`^1#"`9$@`A_6T```T!$^0/D0X@&1(`(?UM``
M`-`10D#Y$`("D2`"'];0``#0$49`^1`B`I$@`A_6T```T!%*0/D00@*1(`(?
MUM```-`13D#Y$&("D2`"'];0``#0$5)`^1""`I$@`A_6T```T!%60/D0H@*1
M(`(?UM```-`16D#Y$,("D2`"'];0``#0$5Y`^1#B`I$@`A_6T```T!%B0/D0
M`@.1(`(?UM```-`19D#Y$"(#D2`"'];0``#0$6I`^1!"`Y$@`A_6T```T!%N
M0/D08@.1(`(?UM```-`1<D#Y$((#D2`"'];0``#0$79`^1"B`Y$@`A_6T```
MT!%Z0/D0P@.1(`(?UM```-`1?D#Y$.(#D2`"'];0``#0$8)`^1`"!)$@`A_6
MT```T!&&0/D0(@21(`(?UM```-`1BD#Y$$($D2`"'];0``#0$8Y`^1!B!)$@
M`A_6T```T!&20/D0@@21(`(?UM```-`1ED#Y$*($D2`"'];0``#0$9I`^1#"
M!)$@`A_6T```T!&>0/D0X@21(`(?UM```-`1HD#Y$`(%D2`"'];0``#0$:9`
M^1`B!9$@`A_6T```T!&J0/D00@61(`(?UM```-`1KD#Y$&(%D2`"'];0``#0
M$;)`^1""!9$@`A_6T```T!&V0/D0H@61(`(?UM```-`1ND#Y$,(%D2`"'];0
M``#0$;Y`^1#B!9$@`A_6T```T!'"0/D0`@:1(`(?UM```-`1QD#Y$"(&D2`"
M'];0``"P$<I`^1!"!I$@`A_6T```L!'.0/D08@:1(`(?UM```+`1TD#Y$((&
MD2`"'];0``"P$=9`^1"B!I$@`A_6T```L!':0/D0P@:1(`(?UM```+`1WD#Y
M$.(&D2`"'];0``"P$>)`^1`"!Y$@`A_6T```L!'F0/D0(@>1(`(?UM```+`1
MZD#Y$$('D2`"'];0``"P$>Y`^1!B!Y$@`A_6T```L!'R0/D0@@>1(`(?UM``
M`+`1]D#Y$*('D2`"'];0``"P$?I`^1#"!Y$@`A_6X0,"JAL``!3```"0P0``
MD"&`-9$`K$?Y/P``ZV("`%3]>[ZI(4``D?T#`)'S4P&ITP``D!0<`)%SHC61
ME`(3RY3R?9*4`@&+8()?^',B`)$``#_6?P(4ZX'__U3S4T&I_7O"J,`#7];`
M`U_6'R`#U?U[O:G]`P"1LU,!J;5;`JGA"0"T]0,!JM8``)#!``"0P\I'^:("
M0/DAY$?YI0I`^40`0/DU``#Y90``^:0'`+3#``"08^Q'^60``/E!`$#Y(@!`
M.>(``#0A!`"17[P`<8'__U1A``#Y(@!`.6+__S5```"TZ_[_E[+__Y?```"0
MP0``D"%@-9$`N$?Y/P``ZZ(!`%33``"0%!P`D7."-9$A0`"1E`(3RY3R?9*4
M`@&+8()?^',B`)$``#_6?P(4ZX'__U0```"0`&`#D=7^_Y?```"0P0``D"%@
M-9$`T$?Y/P``ZZ(!`%33``"0%!P`D7."-9$A0`"1E`(3RY3R?9*4`@&+8()?
M^',B`)$``#_6?P(4ZX'__U36RD?YH0)`^:`*0+G"`D#YI`L`E)W^_Y?!``"0
MP@``L$(`"I$A[$?Y(@``^8#Y_[7,__\7`@``\$*`#)%C`H#200"`4H``@-(`
M__^7(`"`4L[^_Y<`````_7N\J?T#`)'S"P#Y@`<`M,(``)"X'P#Y0LA'^5@`
M0/DX!P"TM1,`^14`@%("`$`Y7_0`<4080'I!`P!4$P-`^9,%`+2VWP*IM7Y`
MD[0/`/GW`P&J]@,`JO0#&*K@`Q.JX@,5JN$#%JJ._O^7@```-6!J=3@?]`!Q
M0`(`5)..0/CS_O^UM-=!J;;?0JFX'T#YX`,3JO,+0/G]>\2HP`-?UO4#`*JB
M'D`X7_0`<4080'JA__]4M0(`2^'__Q<?(`/5E`(8R[4&`)&4_D.3]`(`N7,"
M%8O@`Q.JN!]`^;370:FVWT*I\PM`^?U[Q*C``U_6M1-`^;@?0/GF__\7$P"`
MTN3__Q<3`(#2N!]`^>'__Q?]>[RI_0,`D?5;`JG6``"0\U,!J=,``)#U`P"J
MP,Y'^7/*1_FTTP"1`0!`^:$?`/D!`(#2X0,4JN`#%:JM__^7X`$`M*,W@+EB
M`D#Y0`P#BP$$0/E!>"/XX?[_M``@`)$!!$#Y`0``^:'__[7A`Q2JX`,5JI__
M_Y=@_O^UULY'^:$?0/G``D#Y(```RJ```+7S4T&I]5M"J?U[Q*C``U_6</[_
MEQ\@`]7]>[&I_0,`D?,+`/G3``"08<Y'^2(`0/FB=P#Y`@"`TL```+2T#P#Y
M]`,`J@$`0#E!`0`UM`]`^7/.1_FA=T#Y8`)`^2```,J`"0"U\PM`^?U[SZC`
M`U_6MA<`^7/^_Y?V`P"JH`8`M+=C`ZD7``#P&```\+HG`/GW0@V1&&,-D;I#
M`9&U$P#YX`,6JO/^_Y?`!0"T%30`D>$#%ZK@`Q6JZO[_ER#__S3A`QBJX`,5
MJN;^_Y>@_O\TN2,`^>`#%*K:_O^7^0,`JN`#%:K7_O^7```9"P`(`!$`?$"3
MU_[_E^,#%:KB`Q2J^0,`J@$``/`A@`V1P?[_E^$#&JK@`QFJ&O[_EQ\$`#&@
M``!4H%M`N0`,%!(?$$!QH`(`5.`#&:H&_O^7X`,9JOS]_Y>Y(T#YUO__%Q\@
M`]6T#T#YMA=`^;[__Q<?(`/5X`,6JB_^_Y?@`Q2JV?W_E[370:FVWT*IN!]`
M^;HG0/FT__\7X`,9JJ3__Y?L__\7M-<!J;;?`JFXYP.INB<`^0S^_Y<?(`/5
M_7NYJ?T#`)'S4P&I\P,!JO5;`JG5``"0]V,#J?0#`*JASD?Y8`)`.2(`0/FB
M-P#Y`@"`TA_T`'%S%I.:X`,3JI;^_Y?V`P"JH9,!D>`#%*H:__^7X`<`M/<#
M`*J/_O^7'P`6ZZ(&`%2X9X"YUP``D!CS?=.``D`Y'_0`<0080'J`#`!4X0,4
MJB`<0#@?]`!Q!!A`>J'__U0A`!3+(7Q`D_?*1_G`"@"1```!B_8"0/E^_O^7
MP&HX^*`+`+2C9X"Y@`8`D>("0/F!`D`Y/_0`<4)X8_@D&$!Z00``.<```%0!
M%$`X01P`.#_T`'$D&$!Z@?__5$$$`)&@!X!20```.6`60#@@%``XP/__-0``
M@%*USD?YHC=`^:$"0/E!``'*H0@`M?-30:GU6T*I]V-#J?U[QZC``U_6``"`
MTF%J8#CA:B`X``0`D:'__S7P__\7UP``D+EK!*F[*P#YX,I'^1@`0/D``T#Y
M@`4`M.$#&*H``(!2`@``%.`#&2HBC$#X&00`$:+__[4`#``1`'Q]D\$``+`B
MX$*Y@@(`-2(`@%(BX`*Y0?[_E_L#`*K``P"T.G]`D^$#&*I8\WW3X@,8JE+]
M_Y?@RD?Y&P``^5H'`)%_>SKXN6<`N;LK0/FY:T2II?__%Q\@`]7A`P"JX`,8
MJD+]_Y?[`P"JX,I'^1L``/D[`0"T.G]`DUCS?=/P__\7`0"`TJ/__Q<``H#2
M&0"`4MS__Q>Y:T2INRM`^0``@!*X__\7N6L$J;LK`/E[_?^7_7NTJ?T#`)'S
M4P&ITP``D/0#`*JA@P"18LY'^4,`0/FC7P#Y`P"`TE3]_Y?@`0`UH2M`N0(`
ME%(A#!02/Q!`<2000GI@`0!4<\Y'^:)?0/EA`D#Y00`!RH$!`+7S4T&I_7O,
MJ,`#7]8``(!2]___%T$`@%+@`Q2JP/W_EQ\``''@%Y\:\?__%UC]_Y<?(`/5
M_T,1T>)_@-+]>P"I_0,`D?,+`/G3``"0]5L"J;4C`9'W8P.I]@,`JF/.1_GW
M`P&J````\.$#%:ID`$#YI"<"^00`@-(`H`V17/W_EQ\``'&-!0!4`P0`4;0/
M`/GA`P"JP```D`#$1_D?(`/5'R`#U4+0.]6DRF,X6&A@^"0#`#3C`P$J='Q`
MD^$%@%+B`Q2JX`,5JJ']_Y=@`@"TXP,4JN(#%:KA`Q:JX`,8JJ?]_Y>T#T#Y
M<\Y'^:$G0OE@`D#Y(```RL`"`+7U6T*I]V-#J?U[0*GS"T#Y_T,1D<`#7]8C
M_?\UM`]`^>(#%ZKA`Q:JX`,8JB']_Y?O__\7'R`#U4'0.]7```"0`,1'^1\@
M`]4?(`/5.&A@^/3__Q>T#P#Y#OW_EQ\@`]7_PQ#1X0,`J@*`@-+]>P"I_0,`
MD?-3`:G4``"0LZ,`D8/.1_G@`Q.J9`!`^:07`OD$`(#2P_S_EP``$\L`!`#1
M80(`BS\`$^N)`0!48&I@.!^\`''```!4"```%!\@`]4@`$`Y'[P`<8$``%0_
M]!\X/P`3ZV'__U3A!8!2X`,3JL#\_Y<@`@"T'P`3ZZ@!`%3@`Q.J7_W_EY3.
M1_FB%T+Y@0)`^4$``<IA`@"U\U-!J?U[0*G_PQ"1P`-?UA\@`]4?```Y\___
M%Y3.1_FA%T+Y@`)`^2```,K@``"U\U-!J0```/#]>T"I`$`-D?_#$)%(_?\7
MS_S_E_U[OZE$T#O5P@``D,```)``Q$?Y'R`#U1\@`]7]`P"10K1'^0,``/"`
M:&#X`0``\/U[P:ACX`R1(0`.D0/]_Q?]>[ZI0=`[U<```)``Q$?Y'R`#U1\@
M`]7]`P"1\PL`^3,``(L@:&#XYO__E\(``)!@`D#Y\PM`^00``/!"\$?YA$`;
MD?U[PJ@!``#P!0"`4N,#!*HA@`Z1"/W_%Q\@`]4``(#2P`-?UO]#%-']>P"I
M_0,`D?<;`/G7``"0\U,!J?0#`*KU$P#Y\P,!JN+.1_FALP&1````\`#@#I%#
M`$#YHX<"^0,`@-*X_?^7]0,`JN$%@%+@`Q2J6/S_ER`"`+0```#PX0,4J@`@
M#Y&!_O^7X`,4JO?.1_FBAT+YX0)`^4$``<K!#0"U\U-!J?U[0*GU$T#Y]QM`
M^?]#%)'``U_6X`,3JK@?`/GT_/^7&```\`%C#Y&)_/^7\P,`JL`+`+2Y:P2I
M&@``\!D``/"V%P#Y6D,-D3F##Y&V(P21NRL`^6`"0#E@!P`TM0``M.$#$ZK@
M`Q6J#?W_EX`%`#3@`Q.J`OW_E_L#`*H`!`#180(`BW\"`>O"`0!48&I@.!^\
M`'&@``!4"@``%"(`0#E?O`!Q@0``5#_T'SA_`@'K8?__5.`#$ZKQ_/^7^P,`
MJN`#%*KN_/^7```;BP`$`)$?_`_QZ`(`5`,``/#F`Q2JY0,9JN0#$ZICH`^1
M`H"`T@$`@%+@`Q:J=?S_EZ'#`9'@`Q:J+OS_EZ```#6@>T"Y``P4$A\@0'%@
M`@!4`6,/D0``@-)*_/^7\P,`JD#Y_[6XYT.INN]$J;870/FF__\7'R`#U=4"
M`+3A`QJJX`,5JM/\_Y=`_O\T.P"`TO,#&JK8__\7(0"`4N`#%JJ,_/^78/W_
M->$#%JH```#P`"`/D1G^_Y?@`Q:JF?S_E[870/FXYT.INN]$J93__Q?S`QJJ
M.P"`TL?__Q>X'T#YBO__%[CG`ZFZ[P2IMA<`^13\_Y<?(`/5_7N^J>$%@%+]
M`P"1\PL`^?,#`*KA^_^7'P``\6`&@)KS"T#Y_7O"J(#\_Q<?(`/5P`(`M/U[
MOJG]`P"1\PL`^?,#`*H"`$`Y0@$`-)O\_Y<`!`"1'P`0\0@"`%3@`Q.J\PM`
M^?U[PJCL_O\7'R`#U?,+0/D```#P_7O"J`!`#9%J_/\7'R`#U0```/``0`V1
M9OS_%Q\@`]4``(#2\PM`^?U[PJC``U_6_7N[J0```/``P`^1_0,`D?-3`:G3
M``"0]5L"J18``/!ASD?Y%```\/=C`ZD7``#P&```\)1"$9$B`$#YHB<`^0(`
M@-*U$P&1/OW_EP```/```!"1._W_EP```/``(!"1./W_EP```/``8!"1-?W_
ME\""$)$S_?^7````\`#`$)$P_?^7X.(.D2[]_Y<``Q&1+/W_E^`#%*HJ_?^7
M````\`"`$9$G_?^7````\``@#Y$D_?^7````\.$#%:H`P!&1W/S_EX```+3A
M`P"JX`,4JJK]_Y<```#PX0,5J@`@$I'4_/^7H```M`(``/#A`P"J0(`2D:']
M_Y<```#PX0,5J@#`$I'+_/^70`(`M.$#`*K@X@Z1F?W_EP$``/#`@A"1(6`3
MD97]_Y=SSD?YH2=`^6`"0/D@``#*X`$`M?-30:GU6T*I]V-#J?U[Q:C``U_6
M````\.$#%:H``!.1M?S_E\#]_[3A`P"J``,1D8/]_Y?J__\7B_O_E_U[O:D`
M``#P```1D?T#`)'S"P#YLP``\*&3`)%BSD?Y0P!`^:,7`/D#`(#2H_S_E^$#
M`*H``(!2H0``M"``0#D?P`!Q!!A`>N`'GQISSD?YHA=`^6$"0/E!``'*@0``
MM?,+0/G]>\.HP`-?UF_[_Y?]>ZZIJ0``\*@``/#]`P"1JH,$D:N#`Y$IS4?Y
MJJL#J>H&@!*K)P#YJE,`N>H/@!*J5P"Y".E'^:RW0ZFABPZIJJ]$J:.3#ZFE
MFQ"IX0,`JJ>/`/FB0P"1H!N`/:$?@#VB(X`]HR>`/:0K@#VE+X`]IC.`/:<W
M@#T@`4#YH"\`^0``@-(`P021K#<!J:HO`JF^^_^7X!^`4OCZ_Y<?(`/5_X,!
MT0$`@-+_0T#10@"`4OU[`*G]`P"1I$-`D?87`/FV``#P\U,!J?0#`"K#SD?Y
M90!`^84L`/D%`(#2N_O_EP`$`-'A`P#K$SQ`DB$\0))A1H':$P`!ZP0(`%2W
M8P.I&```T!<``-"U$P#YN2,`^14!@-(8@Q.1]\(3D;E#`9$U`*#R`@"`4N$#
M$ZK@`Q0JIOO_EQ\$`+%`!@!4X0,9JN(#%:K@`Q0J4/O_EP%\0),?!``Q(`0`
M5#\<`/$I!`!4("``T2`#`(L?`!GKHP,`5,,``)`$```4``0`T1\`&>L#`P!4
M801!^0(`0/E?``'K0?__5',"&<NU$T#YMV-#J0``$XNY(T#YULY'^:%#0)$B
M+$#YP0)`^4$``<J!`@"U\U-!J?U[0*GV%T#Y_X,!D?]#0)'``U_6X`,7JHG_
M_Y=S0D#1?T)`L>'Y_U2W8T.IM1-`^;DC0/D``("2Z___%^`#&*I___^7S?__
M%[=C`ZFU$P#YN2,`^>CZ_Y<?(`/5_7N_J?T#`)%X^_^7;_O_EX```+0``$#Y
M_7O!J,`#7]8``(#2_?__%_U[NZG]`P"1\PL`^;,``/#U6P*I%0``T/<;`/FV
M`A218LY'^?<#`*JA$P&1X`,6JD,`0/FC)P#Y`P"`TO'[_Y?```"TM`\`^?0#
M`*H``$`Y``(`-;0/0/GA`Q>JH`(4D;K\_Y=SSD?YH2=`^6`"0/D@``#*@`,`
MM?5;0JGS"T#Y]QM`^?U[Q:C``U_6'R`#U>`#%ZI3^_^7]0,`JN`#%*I0^_^7
MH`(`BP`(`)%1^_^7Y`,4JN(#%ZH#``#08V`/D?4#`*H!``#0(:`/D3G[_Y?A
M`Q6JX`,6JIS\_Y>T#T#YX?__%[0/`/FB^O^7'R`#U?U[IJG*``"02B$(D?T#
M`)'S4P&IM```\/5;`JGU`P"J^2,`^;.3`9'W&P#Y&0``T(S.1_E`A4.IC0%`
M^:W/`/D-`(#21!5!J:"'"JE"C4*II)<&J4:=1*FBCPFI2*5%J::?"ZE+$4#Y
MX0,3JDHU0/D@XPZ1J"<(J:L_`/FJ2P#YI/O_EX```+0!`$`Y]@,`JH$1`#6Z
M[P2IN!\`^9/__Y?W`P"JH!$`M.`#%ZH1^_^7`/A_TP`$`)$2^_^7X@)`.?@#
M`*I"`0`T&@``T/8#&*I:8Q21X`,6JN$#&JKY^O^7U@H`D>(>0#AB__\UH$]`
M^<`)`+2Z8P*1-@"`T@4``!36!@"10`\6BP"`7_C@"`"TX0,3JG_[_Y?W`P"J
M(/__M`$`0#GA_O\TW/S_EZ#^_S3@`Q>JR_K_E_8#`*I@!P"TX`,6JNOZ_Y?W
M`P"JX`,8JNCZ_Y<``!<+%Q`0$1H``-#W?D"3X`,7JN;Z_Y?E`QBJ0X,/D>(#
M%JH$``#0A(`4D?L#`*H!``#0(:`4D<WZ_Y<!.(!2X`,;JI+Z_Y<?!``Q@`8`
M5*$C`Y'@`QNJ&?K_E_8#`"J@```UH--`N0`,%!(?$$!Q8`L`5*```/"B`D#Y
MXP,;J@$``-``Z$?Y(>`5D1<`@-(`P`21&OK_E[KO1*FX'T#YE,Y'^>`#%ZJB
MST#Y@0)`^4$``<H!'@"U\U-!J?5;0JGW&T#Y^2-`^?U[VJC``U_6'R`#U;JC
M`9$7`(!25MMW^+8!`+3``D`Y8`$`-.`#%JKW!@`1E/S_ER#__S3@`Q:J@_K_
ME_8#`*H@]_^U5MMW^+;^_[46`(#2M?__%U3Z_Y<``$"Y'T0`<4#Y_U2S`D#Y
M3_K_EZ4``/`$`$"YXP,;J@$``-"@Z$?YX@,3JB'@%)$7`(#2`,`$D>GY_Y>Z
M[T2IN!]`^<___Q<?(`/5%/__E^`#%JIF^O^7]P,`JLG__Q<?(`/5N`,"D18`
M@%(`VW;X@`$`M.$#$ZK6!@`1"/O_EV#__[0!`$`Y(?__-%CZ_Y?W`P"J`.W_
MM0#;=OC`_O^UH`&`TGKZ_Y<7``#0^`,`JF(*@%+W0A219___%[CG0+EG^O^7
M'P,`:V'T_U2@TT"Y`"``$A\`!W'A\_]4X`,7JFOZ_Y?A`Q.J]P,`J@```-``
M0!>1ZOK_E^$#`*I`"@"TH`)`^1[]_Y?S`P"JP`D`M.`#$ZI:^O^7^`,`JA\,
M`''M`0!4`'Q`DP$``-``$`#1(6`7D6`"`(M9^O^7``$`-1\3`''`#@!4`!,`
M46#"((L`\%\X'[P`<2`.`%0B_O^7P```->`#$ZH!`(!2,/K_E_,#`"K`!@`U
M`0``T"%@$Y$```#0```1D9?[_Y=3@P^1!_K_EQ4``-`%``#0I4`;D>0#`"H!
M``#0XP,3JN(#&ZHA(!B1X`,7JO@#!:JU8AB1(_K_EQ```!0?(`/5X/G_EP``
M0+D?1`!Q`0(`5/3Y_Y?6!@`1Y`,`*N8#&*KE`Q8JXP,3JN(#&ZKA`Q6JX`,7
MJA/Z_Y<!.(!2X`,7JMCY_Y<?!``QP/W_5.`#&ZI4^?^7X0,7JB#C#I%O^_^7
MX`,7JIO^_Y>X'T#YNN]$J5'__Q<?(`/5LP)`^;+__Q<J_O^7]0,`JB#Y_[<!
M*`#1`@"`4N`#$RKN^?^7P@"`TJ&#!9'@`Q,JFOG_EQ\8`/&```!4````T`"@
M%Y'S_?^7````T`#@#)$"0`"1H6-!N0`00+D_``!KH?;_5*'+0GE`"$!Y/P``
M:R'V_U2AR@#1`@"`4N`#$RJUHP61U?G_EP(%@-+A`Q6JX`,3*H'Y_Y<?H`#Q
M@```5````-``H!>1VOW_EP4``-`!``#0OT,&.:5`&Y'D`Q6J0X,/D>(#&ZHA
MX!>1X`,7JL[Y_Y?`__\7M09`^?7Q_[3@`Q6JU?G_EQ\,`/%I\?]4`!``T0$`
M`-"@`@"+(8`7D=;Y_Y<?``!Q<Q*5FH/__Q>Z[P2IN!\`^2SY_Y<?(`/5_7N[
MJ:```/``Q$?Y'R`#U1\@`]4$``#0_0,`D?-3`:E3T#O5]5L"J;0``/#W8P.I
MM1,!D8/.1_GA`Q6J8FI@^("`$)%D`$#YI"<`^00`@-)#P$(Y0\0".3CZ_Y>`
M``"T`0!`.3_$`'%`"P!4)_W_EZ```/``Q$?Y'R`#U1\@`]5W`@"+8&I@^`$`
M`-!C`(!2(@"`4B'`&)$P^?^7]@,`JF`!`+0!"$#Y`@``T.`"0/E"X!B1(0!`
M^03Y_Y?`"D#Y`0!`^2`,0+G`"K`WH```\`#$1_D?(`/5'R`#U7@"`(M@:F#X
M`0``T&,`@%(B`(!2(0`9D=<``)`8^?^7]@,`JF`!`+0```#0X0,5J@`@#Y$*
M^O^7X@,`JD`'`+3!"D#Y``-`^2$`0/GH^/^7H```\`#$1_D?(`/5'R`#U79J
M8/AS`@"+(@"`4@$``-#@`Q:J(2`9D1'Y_Y<"``#0X0,`JD*`&9'@`Q:JV/C_
MEV("0/GS@@N1````T.$#%:H`X`Z17\0".>WY_Y=@"@#YH`(`M`$X@%(=^?^7
M`00`$3\$`'$$B%%Z800`5)3.1_FA)T#Y@`)`^2```,J@`P"U\U-!J?5;0JGW
M8T.I_7O%J,`#7]8`!$`YX/3_-*7__Q<?(`/58`9`^1'^_Y=@"@#Y(/W_M0``
M`-``H!F1[/C_EQ\@`]7@@@N1P@I`^0$$0/E``$#Y(0!`^4_[_Y?%__\7'R`#
MU>`"0/GU^/^7J?__%Y_X_Y?W@@N1`0``T.,#`"H@0!J1X8I`J2$`0/G8^/^7
M'R`#U?U[OJG]`P"1\PL`^?,#`*H(_?^7'P``<60:0/JA``!4\PM`^?U[PJC`
M`U_6'R`#U6`"0#E@__\TX`,3JH'\_Y=T_/^7`0``T((`@-(A@!213OC_EV#^
M_S7@`Q.J\PM`^?U[PJ@/^O\7_X,*T:@``/`$X`"1">`!D?U[`*G]`P"1HF,`
MD0'-1_E'(`"1(P!`^:-/`?D#`(#2@0!`N8.$0/@A",!:X8`?^#\!!.MA_&#3
M9?QHTR;<>-.E''B2(1P(4\8<<)(A``:JH^!#JB$``ZKA!`'X(?[_5$8`")%$
M`$#Y0B``D4,$0/E%'$#Y03!`^6,`!,I?``;K(0`%RB$``\HC_%_3800!JB%\
M0))!/`#Y8?[_5`400*F``&>>!AQ!J>$`9Y["`&>>@?Q"TR%X!*JC_%O30!QA
M+BQ\0)*!`6>>8Q0%JJN_0:DJ,X_22E"K\@D00/D-`&:>H`!GGFL!"HNB_$+3
M(!QB+D)X!:KO`0J+(0`"RJ-!(XNP%T#Y8P`+BTU\0))C``D+$`(*BPX`9IYK
M_%O3:Q4#JB$``XHA``S*L6-`^2$`!HO.`0>+(0`0B\X!#XMO_$+3:P$."^-Y
M`ZIN_5O30@`#RD(`"XK.%0NJS@$!"T$`#<JO"T.I</U"TQ!Z"ZHA``R+[P$*
MBVP`$,HA``^+8WQ`DLO]6].,`0Z*S_U"TVL5#JKO>0ZJC`$#RFL!`0M.``J+
MC`$-BPT"#\H!?D"2C`$.BV+]6].M`0N*;OU"TT(4"ZJM`0'*RWD+JD(`#`NM
M`0.+[`$+RN-]0)*P/T2I3OQ;T\X5`JJ,`0**$`(*BXP!`\JM`1"+[P$*B\X!
M#0N!`0&+4/Q"T\W]6],">@*J(0`/BZT5#JK/_4+3K0$!"VP!`LKA>0ZJ:WU`
MDK`_1:F,`0Z*C`$+RJ[]6],0`@J+@P$#BTP``<IC`!"+0GQ`DLX5#:J,`0V*
MS@$#"XP!`LJP_4+3`WH-JN\!"HN+`0N+S?U;TVL!#XNM%0ZJS_U"TZT!"PLL
M``/*ZWD.JK`_1JDA?$"2C`$.BJ[]6].,`0'*$`(*BX(!`HML``O*0@`0BV-\
M0)+.%0VJC`$-BLX!`@N,`0/*L/U"T^\!"HN!`0&+S/U;TP)Z#:HA``^+C!4.
MJL_]0M.,`0$+;0$"RN%Y#JIK?4"2L#]'J:T!#HJM`0O*COU;TQ`""HNC`0.+
M30`!RF,`$(M"?$"2SA4,JJT!#(J0_4+3S@$#"PQZ#*JM`0+*[P$*BR,`#,JK
M`0N+S?U;TR%\0))K`0^+L$-`^:T5#JIC``Z*S_U"TZT!"POO>0ZJ8P`!R@X"
M"HN+`0_*8P`"BZ+]6].,?4"28P`.B[!'0/E"%`VJ:P$-BJ[]0M-"``,+SGD-
MJFL!#,H-`@J+XP$.RFL!`8M!_%O3[WU`DFL!#8NP2T#Y(10"JF,``HI-_$+3
M(0`+"ZUY`JIC``_*RP$-RF,`#(LB_%O3#`(*B\Y]0))C``R+0A0!JFL!`8JP
M3T#Y0@`#"VL!#LHL_$+3C'D!JA`""HMK`0^+0?Q;TVL!$(LA%`*J4/Q"TR$`
M"PNC`0S*"WH"JF,``HJ"`0O*KU-`^:U]0)*,?4"20@`!BF,`#<I"``S*8P`.
MBT(`#8ON`0J++_Q;T["W2JEC``Z+[Q4!JB[\0M/O`0,+SGD!JN/]6],!`@J+
M8Q0/JD(``8MA`0[*0D`CBVM]0)(A``^*0WQ`DB$`"\JJ`0J+(0`,B^W]0M-L
M_%O3(0`*BXP5`ZJJ>0^J+$`LBZU?0/G!`0K*D'U`DB$``DHB=)W2*P`+BR+;
MK?)O_$+3H0$"BPW^6]/C>0.J80$!BZT5$*I+`0/*+4`MBVL!#$JO?4"2:T$N
MB^S]6],N`@*+`?Y"TVL!#HLA>!"JC!4/JFQ!+(NP9T#Y:P`!RHY]0))M`0U*
M$`("BZI!*HOK_4+3S?U;TVMY#ZI*`1"+K14.JDU!+8NP:T#Y*@`+RJ]]0))*
M`0Q*$`("BTI!(XOL_5O3P_U"TTH!$(MC>`ZJC!4/JDQ!+(NP;T#Y:@$#RHY]
M0))*`0U*$`("BT%!(8O-_5O3ZOU"TR$`$(M*>0^JK14.JBU`+8NP<T#Y80`*
MRJ]]0)(L``Q*$`("BXM!*XO!_4+3[/U;TR%X#JIK`1"+C!4/JFQ!+(NP=T#Y
M2P$!RHY]0))M`0U*$`("BZ-!(XOK_4+3S?U;TVMY#ZIC`!"+K14.JK![0/EM
M0"V+(P`+RJ]]0))L``Q*$`("BXI!*HO#_4+3[/U;TV-X#JI*`1"+C!4/JDQ!
M+(NP?T#Y:@$#RHY]0))-`0U*$`("BZ%!(8OJ_4+3S?U;TTIY#ZHA`!"+K14.
MJBU`+8NP@T#Y80`*RJ]]0)(L``Q*$`("BXM!*XO!_4+3[/U;TR%X#JIK`1"+
MC!4/JFQ!+(NPAT#Y2P$!RHY]0))M`0U*$`("BZ-!(XOK_4+3S?U;TVMY#ZIC
M`!"+K14.JFU`+8NPBT#Y(P`+RJ]]0))L``Q*$`("BXI!*HO#_4+3[/U;TV-X
M#JI*`1"+C!4/JDQ!+(NPCT#Y:@$#RHY]0))-`0U*$`("BZ%!(8OJ_4+3S?U;
MTTIY#ZHA`!"+K14.JBU`+8MA``K*+``,2L']0M,A>`ZJKWU`DK`[4JF+02N+
M[/U;TQ`"`HN,%0^J:P$0BVM!+(M,`0'*<'U`DHT!#4JC02.+[/U"T\T!`HL.
M_EO3C'D/JF,`#8O.%1"J;D`NBR,`#,IK``M*`_Y"TV-X$*K/?4"2K4-3J6I!
M*HNK`0*+[?U;TTH!"XNM%0^J34$MBXH!`\I.`0Y*ZOU"TTIY#ZH0`@*+P4$A
MBZM]0)(A`!"+<``*R@T"#4IN_5O3</U"T\X5"ZH+>@NJ+D`NBZ]#5*G!?4"2
MK$$LBRW\6]/O`0*+K14!JHP!#XM/`0O*C$$MB^X!#DJ-?4"2$`("B\-!(XNN
M_5O3SA4-JK']0M,O_$+38P`0B\,!`POA>0&J+GH-JG`!`<JO1U6I$`(,2LU]
M0)(L?$"2"D(JBR$`#JKB`0*+;_Q;TR$``XIP_$+30@$"B^H5`ZJ/`0V*0@$"
M"P-Z`ZHA``^JBIN7TF]\0))JX['R*T`KBS$""HO!`0.J:P$1BR$``HJQ`0^*
M4/Q;TR$`$:H0%@*JKD=6J5+\0M,+`@L+0GH"JE!\0)+.`0J+(0`,BV,``JIL
M_5O3<OU"TR$`#HN,%0NJ[@$0BF,`"XJ,`0$+8P`.JD%Z"ZIC``V++GQ`DBT"
M"HN+_5O30@`!JF,`#8N2_4+3#0(.BK&[0/EK%0RJ0@`,BFL!`PM"``VJ0WH,
MJD(`#XMM?$"2+P(*BVS]6],A``.J<OU"TT(`#XNQOT#YSP$-BHP5"ZHA``N*
MC`$""R$`#ZI">@NJ(0`0BT]\0)(P`@J+B_U;TV,``JJ2_4+3(0`0B['#0/FP
M`0^*:Q4,JF,`#(IK`0$+8P`0JD%Z#*IC``Z+,'Q`DBX""HML_5O30@`!JG+]
M0M-C``Z+L<=`^>X!$(J,%0NJ0@`+BHP!`PM"``ZJ0WH+JD(`#8MN?$"2+0(*
MBXO]6],A``.JDOU"TT(`#8NQRT#Y#0(.BFL5#*HA``R*:P$""R$`#:I">@RJ
M(0`/BTU\0)(O`@J+;/U;TV,``JIR_4+3(0`/B['/0/G/`0V*C!4+JF,`"XJ,
M`0$+8P`/JD%Z"ZIC`!"++WQ`DC`""HN+_5O30@`!JI+]0M-C`!"+L=-`^;`!
M#XIK%0RJ0@`,BFL!`PM"`!"J0WH,JD(`#HMP?$"2+@(*BVS]6],A``.J<OU"
MTT(`#HNQUT#Y[@$0BHP5"ZHA``N*C`$""R$`#JI">@NJ(0`-BTY\0)(M`@J+
MB_U;TV,``JJ2_4+3(0`-B[';0/D-`@Z*:Q4,JF,`#(IK`0$+8P`-JD%Z#*IC
M``^++7Q`DB\""HML_5O30@`!JG+]0M-C``^+L=]`^<\!#8J,%0NJ0@`+BHP!
M`PM"``^J0WH+JD(`$(MO?$"2,`(*BXO]6],A``.JDOU"TT(`$(NQXT#YL`$/
MBFL5#*HA``R*:P$""R$`$*I">@RJ(0`.BU!\0)(N`@J+;/U;TV,``JIR_4+3
M(0`.B['G0/GN`1"*C!4+JF,`"XJ,`0$+8P`.JD%Z"ZIC``V++GQ`DBT""HN+
M_5O30@`!JI+]0M-C``V+L>M`^0T"#HIK%0RJ0@`,BFL!`PM"``VJ0WH,JD(`
M#XMM?$"2+P(*BVS]6],A``.J0@`/B['O0/G/`0V*C!4+JB$`"XJ,`0(+<OU"
MTR$`#ZI">@NJ(0`0BXO]6],P`@J+(0`0BVL5#*IK`0$+3WQ`DK$'7JEC``*J
ML`$/BF,`#(J2_4+38P`0JDQZ#*HQ`@J+D'U`DD(`#*IC``Z+;OU;T\X5"ZI"
M``N*8P`1B_$!$(K#`0,+0@`1JF[]0M,A``J+RWD+JD(`#8MM_%O3;GU`DD(`
M`8NM%0.J@0$+JE)`+8NL^T#Y`@(.BB$``XI-?D"2(0`"JF+\0M-#>`.JB@$*
MBR$`#XNL_5O38@$#RB$`"HN,%0VJK_U"TRQ`+(M!`!)*(0`0BY%]0)*JPU^I
MPCJ8TD),N?(K_EO32@$"BVL5$:HA``J+ZGD-JBM`*XMA``K*;WU`DB$`#$KM
M_5O3#`("BR$`#HLN_D+3SGD1JB$`#(NM%0^J3`$.RBU`+8NA!T'YL'U`DHL!
M"THA``*+#/Y;TVM!(XOC_4+3:P$!BXP5$*IA>`^J;$$LBZ,+0?G+`0'*CWU`
MDFL!#4IC``*+[?U;TVI!*HL+_D+3:WD0JDH!`XNM%0^J(P`+RDU!+8NJ#T'Y
ML'U`DF,`#$I*`0*+#/Y;TV-`+HON_4+38P`*BXP5$*K*>0^J;$`LBVX!"LJC
M$T'YCWU`DLX!#4IC``*+[?U;T\%!(8L._D+3(0`#BZT5#ZK#>1"J+4`MBU`!
M`\JA%T'YKGU`DA`"#$HA``*+S/U;TPM"*XOP_4+3:P$!BXP5#JH!>@^J;$$L
MBW```<JK&T'YCWU`DA`"#4IK`0*+[?U;TPI"*HO0_4+32@$+BZT5#ZH+>@ZJ
M34$MBS``"\JJ'T'YKGU`DA`"#$I*`0*+S/U;TP-"(XOP_4+38P`*BXP5#JH*
M>@^J;$`LBW`!"LJC(T'YCWU`DA`"#4IC``*+[?U;TP%"(8O0_4+3(0`#BZT5
M#ZH#>@ZJ+4`MBU`!`\JA)T'YKGU`DA`"#$HA``*+S/U;TPM"*XOP_4+3:P$!
MBXP5#JH!>@^J;$$LBW```<JK*T'YCWU`DA`"#4IK`0*+[?U;TPI"*HO0_4+3
M2@$+BZT5#ZH+>@ZJ34$MBS``"\JJ+T'YKGU`DA`"#$I*`0*+S/U;TP-"(XOP
M_4+38P`*BXP5#JH*>@^J;$`LBW`!"LJC,T'YCWU`DA`"#4IC``*+[?U;TP%"
M(8O0_4+3(0`#BZT5#ZH#>@ZJ+4`MBU`!`\JA-T'YKGU`DA`"#$HA``*+S/U;
MTPM"*XOP_4+3:P$!BXP5#JH!>@^J;$$LBW```<JK.T'YCWU`DA`"#4IK`0*+
M[?U;TPI"*HO0_4+32@$+BZT5#ZH+>@ZJ34$MBS``"\JJ/T'YKGU`DA`"#$I*
M`0*+S/U;TP-"(XOP_4+38P`*BXP5#JH*>@^J;$`LBZ]#0?EC`0K*D'U`DFT`
M#4JA02&+`_Y;T^T!`HO/_4+38Q00JB$`#8OM>0ZJ(4`CBZ]'0?E#`0W*+GQ`
MDF,`#$H,_D+3[P$"BXQY$*IK0"N+P_U;TVL!#XMC%`ZJL`$,RJ]+0?EC``L+
M`0(!2LO]0M,J0"J+X0$"BV+\6]-*`0&+0A0#JF%Y#JI"``H+)@`&"PC-1_F'
M`0<+J0$)"T4`!0N#``,+!AP!J04,`*D)$`#YH4]!^0`!0/D@``#*@```M?U[
M0*G_@PJ1P`-?UD;T_Y<?(`/5_7N_J0`0@-+]`P"1X?3_EQ\8`/DF8(32)7&5
MTL2?F]+#CHK2`CZ<TJ;HK/*E^;WR1!>S\D,&HO)">KCR!A0`J00,`:D"?`*I
M'W@`N?U[P:C``U_6'R`#U?U[O*E"?$"3_0,`D?-3`:D4X`"1]5L"J?4#`*KW
M8P.I]@,"JO,#`:H%C$*I`'A`N:0,`@N_``3K8Y2#FF)T0HNDB@*IX`(`-!<(
M@%+W`@!+WP(7:X#"((O7TI<:^'Y`D^(#&*H-]/^7H'I`N>`"``N@>@"Y'P`!
M<<```%3S4T&I]5M"J?=C0ZG]>\2HP`-?UM8"%TMS`AB+X`,5JHO[_Y??_@!Q
MC0(`5-8"`5'8?@93%V<:4_<"`9%W`A>+8@Y`J7,"`9&"#@"IX`,5JF(.?:F"
M#@&I8@Y^J8(.`JEB#G^I@@X#J7G[_Y=_`A?K@?[_5-8:&$O"?D"3X0,3JN`#
M%*KF\_^7MGH`N?-30:GU6T*I]V-#J?U[Q*C``U_6'R`#U?U[O*GC#X`2_0,`
MD?-3`:GS`P&J]QL`^3?@`)'U6P*I]`,`JC780JFA(@-3(@0`$>#"(HM?X`!Q
MX\HA."T+`%0!"(!2(@`"2T)\0),!`(!2]O/_E^`#$ZI4^_^7?_X#J7_^!*E_
M_@6I?S8`^<#^4-/!_EC38,8!.<#^2--AP@$YH?Y8TV#*`3F@_E#3=LX!.6'2
M`3E@U@$YH/Y(TW7>`3E@V@$YX`,3JD#[_Y=A`D#YX`,3JB'\6-.!`@`Y80)`
M^2'\4-.!!@`Y80)`^2'\2-.!"@`Y80)`^8$.`#EA!D#Y(?Q8TX$2`#EA!D#Y
M(?Q0TX$6`#EA!D#Y(?Q(TX$:`#EA!D#Y@1X`.6$*0/DA_%C3@2(`.6$*0/DA
M_%#3@28`.6$*0/DA_$C3@2H`.6$*0/F!+@`Y80Y`^2'\6-.!,@`Y80Y`^2'\
M4-.!-@`Y80Y`^2'\2-.!.@`Y80Y`^8$^`#EA$D#Y(?Q8TX%"`#EA$D#Y(?Q0
MTX%&`#EA$D#Y(?Q(TX%*`#EA$D#Y@4X`.?-30:GU6T*I]QM`^?U[Q*A7\_\7
M`0>`4B(``DM"?$"3`0"`4I[S_Y>N__\7H```T`!`"I$!($$Y00,`-:$``+`B
M`(!2`B`!.2"P1_F@`0"T_7N_J:```-#]`P"1``!!^2?S_Y>@``"P`-Q'^6`!
M`+3]>\&H(```D``@&)'0\_\7H```L`#<1_G```"T(```D``@&)'*\_\7_7O!
MJ,`#7];``U_6H```T`%`"I$"0$HY8@$`-:(``+`C`(!2`T`*.4#`1_G```"T
M(```D"$@`)$`(!B14O/_%Q\@`]7``U_6`````/U[N:G]`P"1\U,!J;0``+"#
MSD?YHH<$J:(C`9%D`$#YI#<`^00`@-*@7P"YH4,!D:!S`9%U\_^7H```L*(`
M`+``X$?Y0MA'^>$#`JI_\_^7H```L`"\1_D``$`YX`D`-$'0.]6U6P*ILP``
MT+<;`/F@``"P`,1'^1\@`]4?(`/5(VA@^*)?0+E`$``150P`$6$(4CD`?'V3
M(00`,F$($CE*\_^7I@``L*0K0/EG@@N1`P``D,;41_EC8!N1`0``D*5?0+F"
M`$#Y(8`;D0(,`*F_!`!QP@!`^0$,`/D""`#YX`0`^8T)`%2$(`"1I0P`$8$`
M@%*#A$#X`]@A^"$$`!$_``5K@?__5"1\?9-V@@N1'V@D^.,#`*KA__^P!`"`
MTN(#%2K`#D#Y(:`LD5OS_Y?`!0`T=8(+D;:3`9&@#D#Y]O+_E_<#`"KA`Q:J
M````D`#`$)$9]/^7``0`M'.""Y%@#D#YV?+_ESSS_Y>4SD?YX`,7*J(W0/F!
M`D#Y00`!RN$$`+6U6T*I\U-!J;<;0/G]>\>HP`-?UO#R_Y>S``#08H(+D4`,
M`/FU6P*IMQL`^8`#`+2Y\O^70=`[U:```+``Q$?Y'R`#U1\@`]4@:&#X'Q@D
M.:?__Q>@"D#YP```M#SZ_Y?>__\7P`Y`^4_S_Y?1__\7````D.$#%JH`X`Z1
M[O/_EZ`*`/F@^O^T,?K_E]/__Q<$!(#2O?__%\/R_Y<@`(!2<?+_EW!S7W-T
M<FEN9W,@;6ES<VEN9PH``````'!E<FQX<VDN8P``````````0T%#2$4``"X`
M````````+BX````````E<R\E<P```"]P<F]C+V-U<G!R;V,O97AE````````
M`$1Y;F%,;V%D97(Z.F)O;W1?1'EN84QO861E<@``````26YT97)N86QS.CI0
M05(Z.D)/3U0`````4$%27U1%35```````````%!!4E]04D]'3D%-10`````Z
M`````````"\`````````)7,E<R5S``!015),-4Q)0@``````````4$523$Q)
M0@!015),-4]05```````````4$523$E/``!005)?24Y)5$E!3$E:140`4$%2
M7U-005=.140``````%!!4E]#3$5!3@````````!005)?1$5"54<`````````
M4$%27T-!0TA%`````````%!!4E]'3$]"04Q?1$5"54<``````````%!!4E]'
M3$]"04Q?5$U01$E2`````````%!!4E]435!$25(```````!005)?1TQ/0D%,
M7U1%35``4$%27T=,3T)!3%]#3$5!3@``````````,0````````!L<V5E:R!F
M86EL960`````<F5A9"!F86EL960``````$Q$7TQ)0E)!4EE?4$%42`!365-4
M14T``"4P,G@`````<&%R+0`````E<R5S)7,E<P``````````)7,Z(&-R96%T
M:6]N(&]F('!R:79A=&4@<W5B9&ER96-T;W)Y("5S(&9A:6QE9"`H97)R;F\]
M)6DI"@```````"5S.B!P<FEV871E('-U8F1I<F5C=&]R>2`E<R!I<R!U;G-A
M9F4@*'!L96%S92!R96UO=F4@:70@86YD(')E=')Y('EO=7(@;W!E<F%T:6]N
M*0H```````!0051(`````'!A<FP`````+G!A<@````!S:&]R="!R96%D````
M````)7,E<V-A8VAE+25S)7,``"5S)7-T96UP+25U)7,````E<R5S=&5M<"TE
M=2TE=25S```````````8`````````'!E<FP`````,`````````!005(Z.E!A
M8VME<CHZ5D524TE/3@`````Q+C`V,P```%5N86)L92!T;R!C<F5A=&4@8V%C
M:&4@9&ER96-T;W)Y```````````E<SH@8W)E871I;VX@;V8@<')I=F%T92!C
M86-H92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*````````
M```M90```````"TM````````55-%4@````!54T523D%-10``````````5$U0
M1$E2``!414U01$E2`%1%35``````5$U0```````O=&UP+P```&5V86P@)V5X
M96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII9B`P.R`C(&YO
M="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*<&%C:V%G92!?7W!A<E]P;#L*
M;7D@*"1005)?34%'24,L("1&24Q%7V]F9G-E=%]S:7IE+"`D8V%C:&5?;F%M
M95]S:7IE*3L*0D5'24X@>PHD4$%27TU!1TE#(#T@(EQN4$%2+G!M7&XB.PHD
M1DE,15]O9F9S971?<VEZ92`](#0[("`@(R!P86-K*").(BD*)&-A8VAE7VYA
M;65?<VEZ92`](#0P.PHD4$M:25!?34%'24,@/2`B4$M<,#`S7#`P-"(["GT*
M<W5B(&9I;F1?<&%R7VUA9VEC"GL*;7D@*"1F:"D@/2!`7SL*;7D@)&-H=6YK
M7W-I>F4@/2`V-"`J(#$P,C0["FUY("1B=68["FUY("1S:7IE(#T@+7,@)&9H
M.PIM>2`D<&]S(#T@*"1S:7IE+3$I("T@*"1S:7IE+3$I("4@)&-H=6YK7W-I
M>F4[("`@("`@(R!.3U1%.B`D<&]S(&ES(&$@;75L=&EP;&4@;V8@)&-H=6YK
M7W-I>F4*=VAI;&4@*"1P;W,@/CT@,"D@>PIS965K("1F:"P@)'!O<RP@,#L*
M<F5A9"`D9F@L("1B=68L("1C:'5N:U]S:7IE("L@;&5N9W1H*"1005)?34%'
M24,I.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B=68L("1005)?34%'24,I*2`^
M/2`P*2!["G)E='5R;B`D<&]S("L@)&D["GT*)'!O<R`M/2`D8VAU;FM?<VEZ
M93L*?0IR971U<FX@+3$["GT*;7D@*"1P87)?=&5M<"P@)'!R;V=N86UE+"!`
M=&UP9FEL92P@)4UO9'5L94-A8VAE*3L*14Y$('L@:68@*"1%3E9[4$%27T-,
M14%.?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL93HZ0F%S
M96YA;64["G)E<75I<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I<B`]($9I;&4Z
M.D)A<V5N86UE.CID:7)N86UE*"1P87)?=&5M<"D["F]U=',H<7%;4F5M;W9I
M;F<@9FEL97,@:6X@(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN9#HZ9FEN9&1E
M<'1H*'-U8B![("@@+60@*2`_(')M9&ER(#H@=6YL:6YK('TL("1P87)?=&5M
M<"D["G)M9&ER("1P87)?=&5M<#L*:68@*"UD("1P87)?=&5M<"`F)B`D7D\@
M;F4@)TU35VEN,S(G*2!["FUY("1T;7`@/2!N97<@1FEL93HZ5&5M<"@*5$5-
M4$Q!5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A<V5N86UE.CID
M:7)N86UE*"1T;W!D:7(I+`I3549&25@@/3X@)RYC;60G+`I53DQ)3DL@/3X@
M,"P**3L*;7D@)&9I;&5N86UE(#T@)'1M<"T^9FEL96YA;64["G!R:6YT("1T
M;7`@/#PB+BXN(CL*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@73L@9&\*<FT@
M+7)F("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M<"<@73L@=&AE
M;@IB<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*9&]N90IR;2`G
M)&9I;&5N86UE)PHN+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P,"P@)&9I;&5N
M86UE.PIM>2`D8VUD(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL(#(^)C$@)B([
M"G-Y<W1E;2@D8VUD*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG<F]U;F0@<')O
M8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI.PI]"GT@?0I"
M14=)3B!["DEN=&5R;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F:6YE9"`F26YT
M97)N86QS.CI005(Z.D)/3U0["F5V86P@>PI?<&%R7VEN:71?96YV*"D["FUY
M("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT["FUY("5S>7,@/2`H"G!A=&A?
M<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G.B<I+`I?97AE
M("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y9W=I;BDO(#\@
M)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^("]>35-7:6Y\
M3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE*"D["F]U=',H
M<7%;7"1P<F]G;F%M92`]("(D<')O9VYA;64B72D["E]S971?<&%R7W1E;7`H
M*3L*;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@/2`B)$5.5GM005)?5$5-4'TB
M72D["FUY("@D<W1A<G1?<&]S+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*3L*
M34%'24,Z('L*;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@>WT["G5N;&5S
M<R`H;W!E;B!?1D@L("<\.G)A=R<L("1P<F]G;F%M92D@>PIO=71S*'%Q6T-A
M;B=T(')E860@9G)O;2!F:6QE("(D<')O9VYA;64B72D[("`C(&1O;B=T('5S
M92`D(2!H97)E(&%S(&ET(')E<75I<F5S($5R<FYO+G!M"FQA<W0@34%'24,[
M"GT*;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R7VUA9VEC*"I?1D@I.PII9B`H
M)&UA9VEC7W!O<R`\(#`I('L*;W5T<RAQ<5M#86XG="!F:6YD(&UA9VEC('-T
M<FEN9R`B)%!!4E]-04=)0R(@:6X@9FEL92`B)'!R;V=N86UE(ETI.PIL87-T
M($U!1TE#.PI]"F]U=',H(D9O=6YD(%!!4B!M86=I8R!A="!P;W-I=&EO;B`D
M;6%G:6-?<&]S(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F
M<V5T7W-I>F4@+2!L96YG=&@H(EPP0T%#2$4B*2P@,#L*<F5A9"!?1D@L("1B
M=68L(&QE;F=T:"@B7#!#04-(12(I.PII9B`H)&)U9B!N92`B7#!#04-(12(@
M)B8@)&)U9B!N92`B7#!#3$5!3B(I('L*;W5T<R@B3F\@8V%C:&4@;6%R:V5R
M(&9O=6YD(BD["FQA<W0@34%'24,["GT*;W5T<RAQ<5M#86-H92!M87)K97(@
M(B1B=68B(&9O=6YD72D["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?
M;V9F<V5T7W-I>F4L(#`["G)E860@7T9(+"`D8G5F+"`D1DE,15]O9F9S971?
M<VEZ93L*;7D@)&]F9G-E="`]('5N<&%C:R@B3B(L("1B=68I.PIO=71S*")/
M9F9S970@9G)O;2!S=&%R="!O9B!&24Q%<R!I<R`D;V9F<V5T(BD["G-E96L@
M7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@+2`D;V9F<V5T
M+"`P.PHD<W1A<G1?;V9?1DE,15]S96-T:6]N(#T@=&5L;"!?1D@["FUY("5R
M97%U:7)E7VQI<W0["G)E860@7T9(+"`D8G5F+"`T.R`@("`@("`@("`@("`@
M("`@("`@("`@("`@(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B"G=H:6QE("@D
M8G5F(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A9"!?1D@L
M("1B=68L('5N<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA;64@/2`D8G5F
M.PIO=71S*'%Q6U5N<&%C:VEN9R!&24Q%("(D9G5L;&YA;64B+BXN72D["FUY
M("1C<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX?2DO?'P@*2`_
M("0Q(#H@=6YD968["FUY("@D8F%S96YA;64L("1E>'0I(#T@*"1B=68@/7X@
M;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B=68L(#0["G)E
M860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@*&1E9FEN960H
M)&5X="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI)"]I*2!["FUY
M("1F:6QE;F%M92`](%]S879E7V%S*"(D8W)C)&5X="(L("1B=68L(#`W-34I
M.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@/2`D9FEL96YA
M;64["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE?2`]("1F=6QL
M;F%M93L*?0IE;'-E('L*)$UO9'5L94-A8VAE>R1F=6QL;F%M97T@/2!["F)U
M9B`]/B`D8G5F+`IC<F,@/3X@)&-R8RP*;F%M92`]/B`D9G5L;&YA;64L"GT[
M"B1R97%U:7)E7VQI<W1[)&9U;&QN86UE?2LK.PI]"G)E860@7T9(+"`D8G5F
M+"`T.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*;7D@*"1S96QF+"`D;6]D=6QE
M*2`]($!?.PIR971U<FX@:68@<F5F("1M;V1U;&4@;W(@(21M;V1U;&4["FUY
M("1I;F9O(#T@)$UO9'5L94-A8VAE>R1M;V1U;&5].PIR971U<FX@=6YL97-S
M("1I;F9O.PID96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE?3L*)$E.0WLD
M;6]D=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M;V1U;&4B.PII9B`H)$5.5GM0
M05)?0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ1FEL93HZ;F5W*2D@>PIM>2`D
M9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@;W(@9&EE(")#86XG="!C<F5A
M=&4@=&5M<"!F:6QE.B`D(2(["B1F:"T^8FEN;6]D92@I.PHD9F@M/G!R:6YT
M*"1I;F9O+3Y[8G5F?2D["B1F:"T^<V5E:R@P+"`P*3L*<F5T=7)N("1F:#L*
M?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1I;F9O+3Y[8W)C
M?2YP;2(L("1I;F9O+3Y[8G5F?2D["B1I;F9O+3Y[9FEL97T@/2`D9FEL96YA
M;64["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D9FEL96YA;64@;W(@9&EE('%Q
M6T-A;B=T(')E860@(B1F:6QE;F%M92(Z("0A73L*<F5T=7)N("1F:#L*?0ID
M:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C86XG="!F:6YD(&UO9'5L92`D
M;6]D=6QE(2(["GTL($!)3D,I.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E
M(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I<F4@17AP
M;W)T97(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D[
M"G)E<75I<F4@24\Z.D9I;&4["G=H:6QE("AM>2`D9FEL96YA;64@/2`H<V]R
M="!K97ES("5R97%U:7)E7VQI<W0I6S!=*2!["G5N;&5S<R`H)$E.0WLD9FEL
M96YA;65](&]R("1F:6QE;F%M92`]?B`O0E-$4$%.+RD@>PII9B`H)&9I;&5N
M86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER92`D9FEL96YA;64["GT*96QS
M92!["F1O("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE(#U^("]S:71E8W5S
M=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R97%U:7)E7VQI<W1[)&9I;&5N
M86UE?3L*?0II9B`H)&)U9B!N92`D4$M:25!?34%'24,I('L*;W5T<RAQ<5M.
M;R!Z:7`@9F]U;F0@869T97(@1DE,12!S96-T:6]N(&EN(&9I;&4@(B1P<F]G
M;F%M92)=*3L*;&%S="!-04=)0R`["GT*)'-T87)T7W!O<R`]("AT96QL(%]&
M2"D@+2`T.R`@("`@("`@("`@("`@("`C('-T87)T(&]F('II<`I]"FUY($!P
M87)?87)G<SL*;7D@*"1O=70L("1B=6YD;&4L("1L;V=F:"P@)&-A8VAE7VYA
M;64I.PID96QE=&4@)$5.5GM005)?05!07U)%55-%?3L@(R!S86YI=&EZ92`H
M4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*:68@*"$D<W1A<G1?
M<&]S(&]R("@D05)'5ELP72!E<2`G+2UP87(M;W!T:6]N<R<@)B8@<VAI9G0I
M*2!["FUY("5D:7-T7V-M9"`]('%W*`IP("`@8FQI8E]T;U]P87(*:2`@(&EN
M<W1A;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R"G,@("!S:6=N7W!A<@IV("`@
M=F5R:69Y7W!A<@HI.PII9B`H0$%21U8@86YD("1!4D=66S!=(&5Q("<M+7)E
M=7-E)RD@>PIS:&EF="!`05)'5CL*)$5.5GM005)?05!07U)%55-%?2`]('-H
M:69T($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA;"!P87)L(&)E:&%V:6]U<@IM
M>2!`861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6*2!["B1!4D=66S!=(#U^("]>
M+2A;04E-3T),8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0["FEF("@D,2!E<2`G
M22<I('L*<'5S:"!`861D7W1O7VEN8RP@)#(["GT*96QS:68@*"0Q(&5Q("=-
M)RD@>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF("@D,2!E<2`G02<I('L*=6YS
M:&EF="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF("@D,2!E<2`G3R<I('L*)&]U
M="`]("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I('L*)&)U;F1L92`]("=S:71E
M)SL*?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B=6YD;&4@/2`G86QL)SL*?0IE
M;'-I9B`H)#$@97$@)W$G*2!["B1Q=6EE="`](#$["GT*96QS:68@*"0Q(&5Q
M("=,)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@9&EE('%Q6T-A;B=T
M(&]P96X@;&]G(&9I;&4@(B0R(CH@)"%=.PI]"F5L<VEF("@D,2!E<2`G5"<I
M('L*)&-A8VAE7VYA;64@/2`D,CL*?0IS:&EF="A`05)'5BD["FEF("AM>2`D
M8VUD(#T@)&1I<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.5GLG4$%27U1%35`G
M?3L*:6YI=%]I;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0["B9[(E!!4CHZ1&ES
M=#HZ)&-M9")]*"D@=6YL97-S($!!4D=6.PHF>R)005(Z.D1I<W0Z.B1C;60B
M?2@D7RD@9F]R($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF="!`24Y#+"!`861D
M7W1O7VEN8SL*?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER92!)3SHZ1FEL93L*
M<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T.CI32$$["GT*
M;7D@)'!A<B`]('-H:69T*$!!4D=6*3L*;7D@)'II<#L*:68@*&1E9FEN960@
M)'!A<BD@>PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)'!A<B!O<B!D:64@<7%;
M0V%N)W0@9FEN9"!P87(@9FEL92`B)'!A<B(Z("0A73L*8FQE<W,H)&9H+"`G
M24\Z.D9I;&4G*3L*)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH("1Z:7`M
M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P87(I(#T]($%R8VAI=F4Z.EII
M<#HZ05I?3TLH*2`I"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H
M:79E("(D<&%R(ET["GT*;7D@)6UE=&%?<&%R(#T@9&\@>PII9B`H)'II<"!A
M;F0@;7D@)&UE=&$@/2`D>FEP+3YC;VYT96YT<R@G345402YY;6PG*2D@>PHD
M;65T82`]?B!S+RXJ7G!A<CHD+R]M<SL*)&UE=&$@/7X@<R]>7%,N*B\O;7,[
M"B1M971A(#U^("]>("`H6UXZ72LI.B`H+BLI)"]M9SL*?0I].PII9B`H9&5F
M:6YE9"`D<&%R*2!["F]P96X@;7D@)'!H+"`G/#IR87<G+"`D<&%R(&]R(&1I
M92!Q<5M#86XG="!R96%D('!A<B!F:6QE("(D<&%R(CH@)"%=.PIM>2`D8G5F
M.PIR96%D("1P:"P@)&)U9BP@-#L*9&EE('%Q6R(D<&%R(B!I<R!N;W0@82!P
M87(@9FEL95T@=6YL97-S("1B=68@97$@)%!+6DE07TU!1TE#.PIC;&]S92`D
M<&@["GT*0W)E871E4&%T:"@D;W5T*2`["FUY("1F:"`]($E/.CI&:6QE+3YN
M97<H"B1O=70L"DE/.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ1FEL93HZ3U]2
M1%=2*"D@?"!)3SHZ1FEL93HZ3U]44E5.0R@I+`HP-S<W+`HI(&]R(&1I92!Q
M<5M#86XG="!C<F5A=&4@9FEL92`B)&]U="(Z("0A73L*)&9H+3YB:6YM;V1E
M*"D["G-E96L@7T9(+"`P+"`P.PIM>2`D;&]A9&5R.PII9B`H9&5F:6YE9"`D
M<W1A<G1?;V9?1DE,15]S96-T:6]N*2!["G)E860@7T9(+"`D;&]A9&5R+"`D
M<W1A<G1?;V9?1DE,15]S96-T:6]N.PI](&5L<V4@>PIL;V-A;"`D+R`]('5N
M9&5F.PHD;&]A9&5R(#T@/%]&2#X["GT*:68@*"$D14Y6>U!!4E]615)"051)
M37T@86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<0')E;2DO*2!["G)E<75I<F4@
M4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ4&]D4W1R:7`M
M/F%P<&QY*%PD;&]A9&5R+"`D,"D["GT*)&9H+3YP<FEN="@D;&]A9&5R*0IO
M<B!D:64@<7%;17)R;W(@=W)I=&EN9R!L;V%D97(@=&\@(B1O=70B.B`D(5T[
M"FEF("@D8G5N9&QE*2!["G)E<75I<F4@4$%2.CI(96%V>3L*4$%2.CI(96%V
M>3HZ7VEN:71?9'EN86QO861E<B@I.PII;FET7VEN8R@I.PIR97%U:7)E7VUO
M9'5L97,H*3L*;7D@0&EN8R`](&=R97`@>R`A+T)31%!!3B\@?0IG<F5P('L*
M*"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@;F4@)$-O;F9I9SHZ0V]N9FEG
M>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N9FEG.CI#;VYF:6=[<')I=FQI
M8F5X<'TI.PI]($!)3D,["G-\+RM\+WQG+"!S?"\D?'P@9F]R96%C:"!`:6YC
M.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,;V%D97(Z.F1L
M7W-H87)E9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI8E]E>'0@/2`D
M0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T[("`@("`@("`@(R!86%@@;&EB7V5X
M="!V<R!D;&5X="`_"FUY("5W<FET=&5N.PIF;W)E86-H(&UY("1K97D@*'-O
M<G0@:V5Y<R`E9FEL97,I('L*;7D@*"1F:6QE+"`D;F%M92D["FEF("AD969I
M;F5D*&UY("1F8R`]("1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&ME>7TI*2![
M"B@D9FEL92P@)&YA;64I(#T@*"1K97DL("1F8RD["GT*96QS92!["F9O<F5A
M8V@@;7D@)&1I<B`H0&EN8RD@>PII9B`H)&ME>2`]?B!M?%Y<421D:7)<12\H
M+BHI)'QI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("0Q*3L*;&%S=#L*
M?0II9B`H)&ME>2`]?B!M?%XO;&]A9&5R+UM>+UTK+R@N*BDD?"D@>PII9B`H
M;7D@)')E9B`]("1-;V1U;&5#86-H97LD,7TI('L**"1F:6QE+"`D;F%M92D@
M/2`H)')E9BP@)#$I.PIL87-T.PI]"FEF("@M9B`B)&1I<B\D,2(I('L**"1F
M:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D,2D["FQA<W0["GT*?0I]"GT*
M;F5X="!U;FQE<W,@9&5F:6YE9"`D;F%M93L*;F5X="!I9B`D=W)I='1E;GLD
M;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE*2!A;F0@)&9I;&4@/7X@+UPN
M7%$D;&EB7V5X=%Q%)"]I.PIO=71S*'-P<FEN=&8H<7%;4&%C:VEN9R!&24Q%
M("(E<R(N+BY=+"!R968@)&9I;&4@/R`D9FEL92T^>VYA;65](#H@)&9I;&4I
M*3L*;7D@)&-O;G1E;G0["FEF("AR968H)&9I;&4I*2!["B1C;VYT96YT(#T@
M)&9I;&4M/GMB=69].PI]"F5L<V4@>PIL;V-A;"`D+R`]('5N9&5F.PIO<&5N
M(&UY("1T:"P@)SPZ<F%W)RP@)&9I;&4@;W(@9&EE('%Q6T-A;B=T(')E860@
M(B1F:6QE(CH@)"%=.PHD8V]N=&5N="`](#PD=&@^.PIC;&]S92`D=&@["E!!
M4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1C;VYT96YT+"`B/&5M8F5D
M9&5D/B\D;F%M92(I"FEF("$D14Y6>U!!4E]615)"051)37T@86YD("1N86UE
M(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2.CI&:6QT97(Z.E!A=&-H0V]N
M=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F:6QE+"`D;F%M92D["GT*
M)&9H+3YP<FEN="@B1DE,12(L"G!A8VLH)TXG+"!L96YG=&@H)&YA;64I("L@
M.2DL"G-P<FEN=&8H(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z.F-O;7!U=&5#
M4D,S,B@D8V]N=&5N="DL("1N86UE*2P*<&%C:R@G3B<L(&QE;F=T:"@D8V]N
M=&5N="DI+`HD8V]N=&5N="D*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@96UB
M961D960@1DE,12!T;R`B)&]U="(Z("0A73L*;W5T<RAQ<5M7<FET=&5N(&%S
M("(D;F%M92)=*3L*?0I]"FEF("@D>FEP*2!["B1Z:7`M/G=R:71E5&]&:6QE
M2&%N9&QE*"1F:"D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2PIO<B!D:64@<7%;
M17)R;W(@=W)I=&EN9R!Z:7`@<&%R="!O9B`B)&]U=")=.PI]"FEF("@D;65T
M85]P87)[8VQE86Y]*2!["B1F:"T^<')I;G0H(EPP0TQ%04XB*3L*?0IE;'-E
M('L*:68@*"%D969I;F5D("1C86-H95]N86UE*2!["FUY("1C='@@/2!$:6=E
M<W0Z.E-(02T^;F5W*#$I.PIS965K("1F:"P@,"P@,#L*)&-T>"T^861D9FEL
M92@D9F@I.PIS965K("1F:"P@,"P@,CL*)&-A8VAE7VYA;64@/2`D8W1X+3YH
M97AD:6=E<W0["GT*)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A8VAE7VYA;64L
M(#`L("1C86-H95]N86UE7W-I>F4["B1C86-H95]N86UE("X](")<,"(@>"`H
M)&-A8VAE7VYA;65?<VEZ92`M(&QE;F=T:"`D8V%C:&5?;F%M92D["B1F:"T^
M<')I;G0H)&-A8VAE7VYA;64L"B)<,$-!0TA%(BD["GT*;7D@)&]F9G-E="`]
M("1F:"T^=&5L;"`M(&QE;F=T:"@D;&]A9&5R*3L*)&9H+3YP<FEN="AP86-K
M*"=.)RP@)&]F9G-E="DL"B1005)?34%'24,I.PHD9F@M/F-L;W-E(&]R(&1I
M92!Q<5M%<G)O<B!W<FET:6YG('1R86EL97(@;V8@(B1O=70B.B`D(5T["F-H
M;6]D(#`W-34L("1O=70["F5X:70["GT*>PIL87-T('5N;&5S<R!D969I;F5D
M("1S=&%R=%]P;W,["E]F:7A?<')O9VYA;64H*3L*<F5Q=6ER92!005(["E!!
M4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*>PIR97%U:7)E($9I;&4Z
M.D9I;F0["G)E<75I<F4@07)C:&EV93HZ6FEP.PI]"FUY("1F:"`]($E/.CI&
M:6QE+3YN97<[("`@("`@("`@("`@("`@("`@("`@("`@("`@("`C($%R8VAI
M=F4Z.EII<"!O<&5R871E<R!O;B!A;B!)3SHZ2&%N9&QE"B1F:"T^9F1O<&5N
M*&9I;&5N;RA?1D@I+"`G<B<I(&]R(&1I92!Q<5MF9&]P96XH*2!F86EL960Z
M("0A73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H+7,@7T9(*3L*;7D@
M)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH)'II<"T^<F5A9$9R;VU&:6QE
M2&%N9&QE*"1F:"P@)'!R;V=N86UE*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+
M*"DI"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H:79E("(D<')O
M9VYA;64B73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H-C0@*B`Q,#(T
M*3L*<'5S:"!`4$%2.CI,:6)#86-H92P@)'II<#L*)%!!4CHZ3&EB0V%C:&5[
M)'!R;V=N86UE?2`]("1Z:7`["F]U=',H(D5X=')A8W1I;F<@>FEP+BXN(BD[
M"FEF("AD969I;F5D("1%3E9[4$%27U1%35!]*2![(",@<VAO=6QD(&)E('-E
M="!A="!T:&ES('!O:6YT(0IF;W)E86-H(&UY("1M96UB97(@*"`D>FEP+3YM
M96UB97)S("D@>PIN97AT(&EF("1M96UB97(M/FES1&ER96-T;W)Y.PIM>2`D
M;65M8F5R7VYA;64@/2`D;65M8F5R+3YF:6QE3F%M93L*;F5X="!U;FQE<W,@
M)&UE;6)E<E]N86UE(#U^(&U["EX*+S]S:&QI8B\**#\Z)$-O;F9I9SHZ0V]N
M9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E>'1R86-T7VYA
M;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M/F-A=&9I;&4H
M)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H+68@)&1E<W1?
M;F%M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E9%-I>F4H*2D@
M>PIO=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I;F-E(&ET(&%L
M<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(ETI.PI](&5L<V4@>PIO=71S
M*'%Q6T5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D97-T7VYA;64B
M+BXN72D["B@D;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H)&1E<W1?;F%M
M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@<7%;17)R;W(@
M97AT<F%C=&EN9R!Z:7`@;65M8F5R('1O("(D9&5S=%]N86UE(ET["F-H;6]D
M*#`U-34L("1D97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"(["GT*?0I]"GT*
M=6YL97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2!["F1I92`\/"`B
M+B(@=6YL97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP87(@72!;("U)
M9&ER(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O9W)A;2YP;"!=
M"B0P(%L@+4)\+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*)$5.5GM005)?
M4%)/1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`05)'5BD["GT*
M<W5B($-R96%T95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E<75I<F4@1FEL
M93HZ0F%S96YA;64["FUY("@D8F%S96YA;64L("1P871H+"`D97AT*2`]($9I
M;&4Z.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN*B<I*3L*<F5Q
M=6ER92!&:6QE.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H)'!A=&@I('5N
M;&5S<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E<G)O<@I]"G-U
M8B!R97%U:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q=6ER92!$>6YA
M3&]A9&5R.PIR97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R:6-T.PIR97%U
M:7)E('=A<FYI;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@0V%R<#L*<F5Q
M=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q=6ER92!%>'!O
M<G1E<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I<F4@1F-N=&P[
M"G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3<&5C.PIR97%U
M:7)E(%A33&]A9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER92!)3SHZ2&%N
M9&QE.PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R97-S.CI:;&EB
M.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z.E-(03L*
M<F5Q=6ER92!005(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER92!005(Z
M.D1I<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PIR97%U:7)E
M(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@871T<FEB=71E
M<SL*979A;"![(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U:7)E(%=I;C,R
M('T["F5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE=F%L('L@<F5Q
M=6ER92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![(')E<75I<F4@
M5&EE.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E<75I<F4@4&5R
M;$E/.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@>R!R97%U:7)E
M('5T9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD969I;F5D("1%
M3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK*2\I('L*
M)'!A<E]T96UP(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY("1P871H("@*
M*&UA<"`D14Y6>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2(%1%35!$25(@
M5$5-4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI('L*;F5X="!U
M;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD("UW("1P871H
M.PIM>2`D=7-E<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P=W5I9"D@/2!G
M971P=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E9FEN960H)E=I
M;C,R.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN,S(Z.DQO9VEN
M3F%M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S97)N86UE(#T@
M)'!W=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U5315).04U%?2!\
M?"`D14Y6>U5315)]('Q\("=365-414TG.PI]"FUY("1S=&UP9&ER(#T@(B1P
M871H)'-Y<WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U<V5R;F%M92D[
M"FUK9&ER("1S=&UP9&ER+"`P-S4U.PIM>2`D8V%C:&5?9&ER.PII9B`H)$5.
M5GM005)?0TQ%04Y]*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["GT*96QS
M92!["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA;64@;W(@9&EE('%Q
M6T-A;B=T(')E860@(B1P<F]G;F%M92(Z("0A73L*:68@*"AM>2`D;6%G:6-?
M<&]S(#T@9FEN9%]P87)?;6%G:6,H)&9H*2D@/CT@,"D@>PIS965K("1F:"P@
M)&UA9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S:7IE"BT@;&5N9W1H*")<,$-!
M0TA%(BDL(#`["FUY("1B=68["G)E860@)&9H+"`D8G5F+"!L96YG=&@H(EPP
M0T%#2$4B*3L*:68@*"1B=68@97$@(EPP0T%#2$4B*2!["G-E96L@)&9H+"`D
M;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG=&@H(EPP0T%#
M2$4B*0HM("1C86-H95]N86UE7W-I>F4L(#`["G)E860@)&9H+"`D8G5F+"`D
M8V%C:&5?;F%M95]S:7IE.PHD8G5F(#U^(',O7#`O+V<["B1C86-H95]D:7(@
M/2`B8V%C:&4M)&)U9B(["GT*?0IC;&]S92`D9F@["GT*:68@*"$D8V%C:&5?
M9&ER*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["B1%3E9[4$%27T-,14%.
M?2`](#$["GT*)'-T;7!D:7(@+CT@(B1S>7-[7V1E;&EM?21C86-H95]D:7(B
M.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*)$5.5GM005)?5$5-4'T@/2`D<W1M
M<&1I<CL*;&%S=#L*?0HD<&%R7W1E;7`@/2`D,2!I9B`D14Y6>U!!4E]414U0
M?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO.PI]"G-U8B!?<V%V95]A
M<R!["FUY("@D;F%M92P@)&-O;G1E;G1S+"`D;6]D92D@/2!`7SL*;7D@)&9U
M;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M92(["G5N;&5S<R`H+64@)&9U;&QN
M86UE*2!["FUY("1T96UP;F%M92`]("(D9G5L;&YA;64N)"0B.PIM>2`D9F@[
M"BAO<&5N("1F:"P@)SXZ<F%W)RP@)'1E;7!N86UE*0HF)B`H<')I;G0@)&9H
M("1C;VYT96YT<RD*)B8@*&-L;W-E("1F:"D*;W(@9&EE('%Q6T5R<F]R('=R
M:71I;F<@(B1T96UP;F%M92(Z("0A73L*8VAM;V0@)&UO9&4L("1T96UP;F%M
M92!I9B!D969I;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE+"`D9G5L;&YA
M;64I(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D9G5L;&YA;64[
M"GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D14Y6>U!!4E]0
M4D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O*"XK*2\I('L*
M)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*:68@*"1%3E9[
M4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[4$%27U1%35!]
M*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA;64L(')I;F1E
M>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@*R`Q*3L*?0II9B`H(21%3E9[
M4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM
M?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\)RP@)'!R;V=N86UE*2![
M"G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D<')O9VYA;64D<WES>U]E
M>&5](BD@>PHD<')O9VYA;64@+CT@)'-Y<WM?97AE?3L*<F5T=7)N.PI]"GT*
M9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D<WES>W!A=&A?<V5P?5Q%+RP@
M)$5.5GM0051(?2D@>PIN97AT(&EF(&5X:7-T<R`D14Y6>U!!4E]414U0?2!A
M;F0@)&1I<B!E<2`D14Y6>U!!4E]414U0?3L*)&1I<B`]?B!S+UQ1)'-Y<WM?
M9&5L:6U]7$4D+R\["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D96QI;7TD
M<')O9VYA;64D<WES>U]E>&5](BDL(&QA<W0I"FEF("US("(D9&ER)'-Y<WM?
M9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2(["B@H)'!R;V=N86UE(#T@(B1D
M:7(D<WES>U]D96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@+7,@(B1D:7(D
M<WES>U]D96QI;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?<')O9VYA;64@
M>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!345].PII9B`H
M:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI(#P@,"D@>PHD<')O9VYA
M;64@/2`B+B1S>7-[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@)'!W9"`]("AD
M969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I"CH@*"AD969I
M;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H*2`Z(&!P=V1@
M*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<+EPN/UQ1)'-Y
M<WM?9&5L:6U]7$4I+R1P=V0D<WES>U]D96QI;7TO.PHD14Y6>U!!4E]04D]'
M3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?:6YI=%]E;G8@>PII9B`H
M("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2!["G)E='5R;CL*?2!E
M;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@,CL*?0IF;W(@*'%W*"!3
M4$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!344@*2`I('L*
M9&5L971E("1%3E9[(E!!4E\D7R)].PI]"F9O<B`H<7<O(%1-4$1)4B!414U0
M($-,14%.($1%0E5'("\I('L*)$5.5GLB4$%27R1?(GT@/2`D14Y6>R)005)?
M1TQ/0D%,7R1?(GT@:68@97AI<W1S("1%3E9[(E!!4E]'3$]"04Q?)%\B?3L*
M?0II9B`H)$5.5GM005)?5$5-4'TI('L*9&5L971E("1%3E9[4$%27T-,14%.
M?3L*?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*:68@*"1L;V=F
M:"D@>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP<FEN="`B0%]<
M;B(["GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I9SL*<'5S:"!`
M24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF:6=[)%]]+"!Q
M=R@*87)C:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP('-I=&5L:6)E
M>'`*=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP86-K86=E(&UA
M:6X["G)E<75I<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2.CIF:6YD7W!A
M<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q6W!A<BYP;#H@0V%N
M)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@<W5C:"!F:6QE
M(&]R(&1I<F5C=&]R>5QN70IU;FQE<W,@+64@)'!R;V=N86UE.PID;R`D<')O
M9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+7T58251?7"@H
M7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/4B`]("1`(&EF
M("1`.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2(#U^+UY?5$M?
M15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@)#HZ7U]%4E)/
M4CL*,3L*7U]%3D1?7PH``1L#.P0!```?````0(S__QP!``"HC/__1`$``!B.
M__^X`0``*(___S`"``#0C___9`(``&"1___X`@``H)/__UP#```PE/__B`,`
M`%B5___P`P``4);__RP$``"0EO__4`0``/"6__]X!```^);__XP$``!`F?__
M#`4``'"9__\T!0``Z)G__W0%``!@F___L`4``-";___8!0``<)S___0%``#P
MG?__E`8``!B>__^X!@``")___P0'``#@I/__B`<``#"G___$!P``F*?__T`(
M``!XN/__:`@``-"X__^,"```\+G__]0(``"HN___A`$``""\__]P`0``8+S_
M__0'````````$``````````!>E(`!'@>`1L,'P`D````&````!R+__]D````
M`$<.()T$G@-"#1U!DP*4`4W>W=/4#!\`*````$````!<B___;`$```!!#C"=
M!IX%00T=090#DP1!E@&5`@`````````0````;````*BZ__\\`````````#``
M``"`````'+K__W@`````2@X0G0*>`4(-'4;>W0P?`$D,'1"=`IX!0=[=#!\`
M``````!T````M````%B,__\0`0```$$.0)T(G@=!#1U!DP9#F`%$E01(EP*6
M`T*4!4[5U$'7UD'80][=TPP?`$$,'4"3!I4$F`&=")X'2)0%E@.7`D?80=74
M0=?60M[=TPP?`$$,'4"3!I4$F`&=")X'0=5!V$.8`4+8```P````+`$``/",
M__^D`````$$.0)T(G@=!#1U!E026`T*3!I0%8@K>W=76T]0,'P!!"P``D```
M`&`!``!DC?__C`$```!!#O`!G1Z>'4$-'4&3'$>4&T341][=TPP?`$$,'?`!
MDQR4&YT>GAU!EAE$F!>7&$.:%425&DV9%EW90M77V-I!U$'60I0;E1J6&9<8
MF!>:%475U$'7UD'80=I!E!N5&I89EQB8%YD6FA5#U-76U]C9VD&5&I0;09<8
MEAE!F1:8%T&:%6````#T`0``8([__T`"````00YPG0Z>#4$-'4&3#)0+0I4*
ME@E+EPB8!WT*WMW7V-76T]0,'P!!"TB:!9D&09L$8`K;0=K90@M*V=K;0ID&
MF@6;!$3:V4';0YH%F09!FP0````H````6`(``#R0__^,`````$$.P`&=&)X7
M00T=09,6E!56"M[=T]0,'P!!"V0```"$`@``H)#__R0!````00[0"$*=B@&>
MB0%!#1U!DX@!0I6&`9:%`427A`&8@P%,E(<!5M1*W=[7V-76TPP?`$$,'=`(
MDX@!E(<!E88!EH4!EX0!F(,!G8H!GHD!0M1.E(<!````.````.P"``!@D?__
M^`````!!#K`(0YV&`9Z%`4$-'4&3A`&4@P%H"MW>T]0,'P!""TP*W=[3U`P?
M`$$+(````"@#```<DO__0`````!!#A"=`IX!1PT=1=[=#!\`````)````$P#
M```XDO__7`````!!#B"=!)X#1@T=09,"2M[=TPP?`````!````!T`P``<)+_
M_P@`````````?````(@#``!DDO__1`(```!!#I`*09VB`9ZA`4$-'4&7G`%"
MDZ`!E)\!2I6>`54*W=[7U=/4#!\`00M"F)L!1YJ9`9F:`4.6G0%$FY@!>`K9
MV$';VD'60@M3"M9!V=A!V]I!"T/6V=K;0=A"F9H!F)L!09N8`9J9`4&6G0$`
M```D````"`0``"R4__\L`````$$.()T$G@-"#1U!DP)&WMW3#!\`````/```
M`#`$```TE/__>`````!"#B"=!)X#00T=09,"2@K>W=,,'P!""T/>W=,,'P!'
M#!T@DP*=!)X#0][=TPP?`#@```!P!```;)3__W@!````00Y0G0J>"4,-'4&3
M")0'0I4&E@5$EP28`P)("M[=U]C5UM/4#!\`00L``"0```"L!```J)7__W``
M````00XPG0:>!4,-'4&3!%4*WMW3#!\`00L8````U`0``/"5__^<`````$$.
MH`*=))XC0PT=G````/`$``!TEO__?`$```!!#F!"#N"`!$*=C$">BT!!#1U"
MEH=`0I.*0)2)0$Z8A4"7AD!#E8A`09F$0&/50=C70ME)W=[6T]0-'T$.@(`$
M00X`00P=X(`$DXI`E(E`E8A`EH=`EX9`F(5`F81`G8Q`GHM`1MC70=5!V4*5
MB$"7AD"8A4"9A$!#U=?8V4&8A4"7AD!!E8A`09F$0````"````"0!0``5)?_
M_R@`````00X0G0*>`4$-'44*WMT,'P!!"T@```"T!0``6)?__^P`````00Y0
MG0J>"4$-'4&3"$*5!I8%1)<$290'1-1,WMW7U=;3#!\`0@P=4),(E`>5!I8%
MEP2="IX)5-1"E`>```````8``/R7___4!0```$$.H`.=-)XS0PT=09,RE#%"
ME3"6+T29+)<N6YLJFBM!F"T"5=O:0=A+WMW9U]76T]0,'P!"#!V@`Y,RE#&5
M,)8OERZ8+9DLFBN;*ITTGC-CV]I!V$B8+9HKFRH"=0K80=O:0@L"1-C:VT&;
M*IHK09@M```X````A`8``%"=__],`@```$$.4)T*G@E&#1U!DPB4!T25!I8%
MEP28`P)G"M[=U]C5UM/4#!\`00L````L````P`8``&2?__]H`````$$.()T$
MG@-!#1U!DP)'"M[=TPP?`$(+3=[=TPP?``!(````\`8``&2T__\H`@```$$.
M<)T.G@U!#1U!DPR4"U>6"94*0I<(`D;6U4+70=[=T]0,'P!!#!UPDPR4"YT.
MG@U%E@F5"D&7"```)````#P'``!0G___W!````!!#J`%1)U4GE-!#1T#+P0*
MW=X,'P!!"R````!D!P``"+#__U0`````00X0G0*>`4(-'5'>W0P?`````$0`
M``"(!P``/+#__QP!````00Y`G0B>!T(-'4&3!I0%0I4$E@-"EP*8`5L*WMW7
MV-76T]0,'P!!"V+>W=?8U=;3U`P?`````#0```#0!P``%+'__[@!````00Y`
MG0B>!T(-'4&3!I0%0I<"0I4$E@,"7PK>W=?5UM/4#!\`00L`````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````N%````````!`4````````````````````0`````````!````````
M``$`````````:P(````````!`````````'4"`````````0````````"Q`@``
M``````$`````````UP(````````/`````````'`$````````&0````````!8
MK0$``````!L`````````"``````````:`````````&"M`0``````'```````
M```(``````````0`````````2`(````````%`````````!@/````````!@``
M``````!P!0````````H`````````N`0````````+`````````!@`````````
M%0````````````````````,`````````Z*\!```````"`````````.@%````
M````%``````````'`````````!<`````````T!8````````'`````````-`3
M````````"````````````P````````D`````````&`````````#[__]O````
M``````@`````^?__;P`````6````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````<*T!``````!HK0$`````````````````````````
M``!8K0$```````````````````````````````````````"S`0``````````
M``````!@K0$``````'BR`0``````````````````````````````````````
MB+(!`````````````````/BQ`0``````>#D`````````````````````````
M``````````````#`'````````,`<````````P!P```````#`'````````,`<
M````````P!P```````#`'````````,`<````````P!P```````#`'```````
M`,`<````````P!P```````#`'````````,`<````````P!P```````#`'```
M`````,`<````````P!P```````#`'````````,`<````````P!P```````#`
M'````````,`<````````P!P```````#`'````````,`<````````P!P`````
M``#`'````````,`<````````P!P```````#`'````````,`<````````P!P`
M``````#`'````````,`<````````P!P```````#`'````````,`<````````
MP!P```````#`'````````,`<````````P!P```````#`'````````,`<````
M````P!P```````#`'````````,`<````````P!P```````#`'````````,`<
M````````P!P```````#`'````````,`<````````P!P```````#`'```````
M`,`<````````P!P```````#`'````````,`<````````P!P```````#`'```
M`````,`<````````P!P```````"`L@$```````"R`0``````"E!!4BYP;0H`
M`````````"!7````````4%,``````````````````*!4`````````%<`````
M```(5P```````!!7````````&%<``````````````````.A6````````\%8`
M`````````````````"A7````````1T-#.B`H3F5T0E-$(&YB-"`R,#(P,#@Q
M,"D@-RXU+C``1T-#.B`H;F(T(#(P,C`P.#$P*2`W+C4N,``D3F5T0E-$.B!C
M<G0P+E,L=B`Q+C(@,C`Q."\Q,2\R-B`Q-SHS-SHT-"!J;V5R9R!%>'`@)``D
M3F5T0E-$.B!C<G0P+6-O;6UO;BYC+'8@,2XR,R`R,#$X+S$R+S(X(#(P.C$R
M.C,U(&-H<FES=&]S($5X<"`D`"1.971"4T0Z(&-R=&DN4RQV(#$N,2`R,#$T
M+S`X+S$P(#`U.C0W.C,V(&UA='0@17AP("0`)$YE=$)31#H@8W)T8F5G:6XN
M8RQV(#$N,3<@,C`Q."\Q,B\R."`Q.#HQ-SHQ,2!C:')I<W1O<R!%>'`@)``D
M3F5T0E-$.B!C<G1E;F0N4RQV(#$N,2`R,#$T+S`X+S$P(#`U.C0W.C,V(&UA
M='0@17AP("0`\<\"``````````````````@````(````"`````@````;````
M`0``````````*BHJ($YO($QA8F5L(%!R;W9I9&5D("HJ*@``\<\"````````
M``````````@````(````"`````@````;`````0``````````*BHJ($YO($QA
M8F5L(%!R;W9I9&5D("HJ*@``+G-H<W1R=&%B`"YI;G1E<G``+FYO=&4N;F5T
M8G-D+FED96YT`"YN;W1E+FYE=&)S9"YP87@`+FAA<V@`+F1Y;G-Y;0`N9'EN
M<W1R`"YR96QA+F1Y;@`N<F5L82YP;'0`+G1E>'0`+G)O9&%T80`N96A?9G)A
M;65?:&1R`"YE:%]F<F%M90`N:6YI=%]A<G)A>0`N9FEN:5]A<G)A>0`N:F-R
M`"YD>6YA;6EC`"YG;W0`+F=O="YP;'0`+F1A=&$`+F)S<P`N8V]M;65N=``N
M4U5.5U]C=&8`````````````````````````````````````````````````
M```````````````````````````````````````````+`````0````(`````
M``````(``````````@```````!<````````````````````!````````````
M````````$P````<````"`````````!@"````````&`(````````8````````
M````````````!````````````````````"8````'`````@`````````P`@``
M`````#`"````````%`````````````````````0````````````````````W
M````!0````(`````````2`(```````!(`@```````"@#````````!0``````
M```(``````````0`````````/0````L````"`````````'`%````````<`4`
M``````"H"0````````8````#````"``````````8`````````$4````#````
M`@`````````8#P```````!@/````````N`0```````````````````$`````
M``````````````!-````!`````(`````````T!,```````#0$P`````````#
M````````!0`````````(`````````!@`````````5P````0```!"````````
M`-`6````````T!8```````#H!0````````4````3````"``````````8````
M`````%P````!````!@````````#`'````````,`<````````$`0`````````
M`````````!``````````$`````````!A`````0````8`````````T"``````
M``#0(````````%`R```````````````````(````````````````````9P``
M``$````"`````````"!3````````(%,```````!X00``````````````````
M"````````````````````&\````!`````@````````"8E````````)B4````
M````!`$```````````````````0```````````````````!]`````0````(`
M````````H)4```````"@E0````````P(```````````````````(````````
M````````````AP````X````#`````````%BM`0``````6*T````````(````
M````````````````"``````````(`````````),````/`````P````````!@
MK0$``````&"M````````"`````````````````````@`````````"```````
M``"?`````0````,`````````:*T!``````!HK0````````@`````````````
M```````(````````````````````I`````8````#`````````'"M`0``````
M<*T```````#@`0````````8`````````"``````````0`````````*T````!
M`````P````````!0KP$``````%"O````````F`````````````````````@`
M````````"`````````"R`````0````,`````````Z*\!``````#HKP``````
M`!`"```````````````````(``````````@`````````NP````$````#````
M`````/BQ`0``````^+$```````"(````````````````````"```````````
M`````````,$````(`````P````````"`L@$``````("R````````B```````
M`````````````$````````````````````#&`````0```#``````````````
M``````"`L@```````#L````````````````````!``````````$`````````
M'P````$`````````````````````````N[(````````A`0``````````````
M`````0```````````````````,\````!`````````````````````````-RS
M````````CP````````````````````0````````````````````!`````P``
M``````````````````````!KM````````-D````````````````````!````
,````````````````
