/**
 * @file libcomprex/utils.h Utility functions
 * 
 * $Id: utils.h,v 1.19 2003/03/06 04:36:26 chipx86 Exp $
 *
 * @Copyright (C) 2001-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _LIBCOMPREX_UTILS_H_
#define _LIBCOMPREX_UTILS_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <libcomprex/comprex.h>

/* #define DEBUG_REF */

/**
 * Updates the reference count on a structure.
 *
 * @param ptr The structure that is in use.
 */
#ifdef DEBUG_REF
# define CX_LINK(ptr) \
	printf("[%p] inc refcount: %d -> %d\n", (ptr), (ptr)->refCount, \
		   (ptr)->refCount + 1); \
	++(ptr)->refCount
#else
# define CX_LINK(ptr) ++(ptr)->refCount
#endif

/**
 * Decrements the reference count on a structure.
 *
 * @note This should not be called manually. Please use
 *       the appropriate cxDestroy*() function, as it will handle
 *       this automatically.
 *
 * @param ptr The structure that is no longer to be in use.
 */
#ifdef DEBUG_REF
# define CX_UNLINK(ptr) \
	printf("[%p] dec refcount: %d -> %d\n", (ptr), (ptr)->refCount,\
		   (ptr)->refCount - 1); \
	--(ptr)->refCount
#else
# define CX_UNLINK(ptr) --(ptr)->refCount
#endif

/**
 * Splits the specified path into a directory part and a filename part.
 *
 * @param path     The path to split.
 * @param dir      The returned directory part.
 * @param basename The returned filename part.
 *
 * @see cxGetBaseName()
 * @see cxGetBasePath()
 */
void cxSplitPath(const char *path, char **dir, char **basename);

/**
 * Returns the base filename for the specified path.
 *
 * The base filename is the part not containing path information.
 * 
 * @param path The path.
 *
 * @return The base filename.
 *
 * @see cxGetBasePath()
 * @see cxSplitPath()
 */
char *cxGetBaseName(const char *path);

/**
 * Returns the base path for the specified path.
 *
 * The base path is the part not containing the actual filename.
 *
 * @param path The path.
 *
 * @return The base path.
 *
 * @see cxGetBaseName()
 * @see cxSplitPath()
 */
char *cxGetBasePath(const char *path);

/**
 * Fixes up the specified path by processing all ".." and "." that
 * may be in the path.
 *
 * @param path The path
 *
 * @return The path, possibly shortened and processed.
 *
 * @see cxGetFullFilePath()
 */
char *cxFixPath(const char *path);

/**
 * Returns the full, absolute path for the specified filename.
 *
 * This prepends the current working directory, if the specified
 * path is not an absolute path.
 *
 * The path will be processed by using the cxFixPath() function.
 *
 * @param filename The filename
 * 
 * @return The processed, absolute path name to the file.
 *
 * @see cxFixPath()
 */
char *cxGetFullFilePath(const char *filename);

/**
 * Makes a directory and all its parent directories, if missing, on the
 * filesystem.
 *
 * If an archive is specified (not @c NULL), the modes and permissions in the
 * path will reflect those in the archive.
 *
 * @param path    The path to create.
 * @param archive The archive to use as a base for permissions and modes.
 */
void cxMakePhysDirs(const char *path, CxArchive *archive);

/**
 * Applies the information from an FsNode to a physical file or directory.
 *
 * This applies the mode, UID, GID, and timestamp, if available.
 *
 * @param node The FsNode.
 * @param path The path to apply the information to.
 */
void cxApplyFsNodeInfo(CxFsNode *node, const char *path);

#ifdef __cplusplus
}
#endif

#endif /* _LIBCOMPREX_UTILS_H_ */

