package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.42.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.42.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@(!`0````````````,`MP`!````,"$```````!``````````%`%`0``
M`````````$``.``(`$``&@`9``8````$````0`````````!``````````$``
M````````P`$```````#``0````````@``````````P````0``````@``````
M```"``````````(````````7`````````!<``````````0`````````!````
M!0```````````````````````````````````"2?````````))\`````````
M``$```````$````&````0/T```````!`_0$``````$#]`0``````2`4`````
M``!(!@```````````0```````@````8```!8_0```````%C]`0``````6/T!
M``````#@`0```````.`!````````"``````````$````!````!@"````````
M&`(````````8`@```````"P`````````+``````````$`````````%#E=&0$
M````H)<```````"@EP```````*"7````````!`$````````$`0````````0`
M````````4N5T9`0```!`_0```````$#]`0``````0/T!``````#``@``````
M`,`"`````````0`````````O=7-R+VQI8F5X96,O;&0N96QF7W-O```'````
M!`````$```!.971"4T0```"KD$$$````!`````,```!085@```````````!A
M````:```````````````,0````T``````````````#@````.````0```````
M```P``````````,```!,````80`````````?````2P````8````=````%```
M``````!"````5@```%T`````````%P```#P```!8``````````````!D````
M9@```"\````2````5P```$0````5````7````$4`````````2````"(````U
M````6P```"8````J``````````<```!.``````````L````L````9P```%H`
M``!*````(````$\```!E`````````%\`````````&````"X```!-````````
M````````````$0```#,``````````````&`````K``````````D`````````
M)````#D`````````4`````4```!&````&P```#L`````````````````````
M````20`````````Z````4P```&(`````````8P``````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````!,`````````````````````````````````````
M```````````````C````(0`````````I``````````H````G````*````"T`
M``````````````P`````````'````!H````9```````````````0````````
M`#0````R````-P```!X`````````````````````````!```````````````
M00```"4````````````````````6```````````````_````0P````@`````
M```````````````^````4@``````````````50``````````````/0``````
M````````40````````!'````#P````````!4````60```#8```!>````````
M`````````````````````````````````````````````P`*`#`A````````
M`````````````````P`0`%#]`0````````````````"'````$@``````````
M```````````````!````$@`````````````````````````)````$@``````
M``````````````````!P`P``$@`*`#`K````````"``````````1````(@``
M```````````````````````@````$@````````````````````````#9`@``
M$@````````````````````````!)`P``$``5`,`"`@`````````````````F
M````$@````````````````````````"!`P``$@`*`'`U````````(```````
M```M````$@````````````````````````#F`@``$@``````````````````
M```````U````$@````````````````````````"<`P``$@`*`,`^````````
M9`````````!X````$@````````````````````````!8`P``$``5`(@#`@``
M```````````````]````$@````````````````````````"H`P``$``4`(@"
M`@````````````````!-````$@````````````````````````!<````$@``
M``````````````````````!B````$@````````````````````````!I````
M$@````````````````````````!S````$@````````````````````````!]
M````$@````````````````````````!7`P``$``5`(@#`@``````````````
M``"&````$@`````````````````````````(`@``$@``````````````````
M``````",````$@````````````````````````"3````$@``````````````
M``````````"A````$@````````````````````````"R````$0``````````
M``````````````"%`@``(@````````````````````````"_````$@``````
M``````````````````"O`P``$@`*`(0M````````+`````````"\`P``$@`*
M`+`M````````;`````````#(`P``$@`*`)0V````````P`4```````#-````
M%@````````````````````````#@````$@````````````````````````#X
M`@``$@````````````````````````#K````$@``````````````````````
M``#T````$0`5`(`#`@``````"`````````#\````$@``````````````````
M```````%`0``$@````````````````````````#/`@``$@``````````````
M```````````1`0``$@`````````````````````````8`0``$@``````````
M``````````````#5`P``$@`*`#!0````````&`$````````K`0``$0``````
M```````````````````]`0``$@````````````````````````!-`0``$@``
M``````````````````````!;`0``$@````````````````````````!@`0``
M$@````````````````````````#@`P``$0`4`(`"`@``````"`````````!O
M`0``$@````````````````````````![`0``$@``````````````````````
M``"0`0``$@````````````````````````";`0``$@``````````````````
M``````!K`P``$``5`(@#`@````````````````";`@``$@``````````````
M``````````"N`0``$@`````````````````````````;!```$@`*`#`A````
M````"`````````"\`0``$@````````````````````````#$`0``$@``````
M``````````````````#0`0``$@````````````````````````!C`P``$``5
M`(@#`@````````````````#J`P``$@`*`"`N````````T`0```````#6`0``
M$@````````````````````````#A`0``$@````````````````````````#W
M`P``$@`*`%!1````````P`$```````#R`0``$@``````````````````````
M```!!```$``5`,`"`@`````````````````-!```$@`*`)0J````````0```
M```````5!```$@`*`-!3````````)`(```````#Y`0``$@``````````````
M``````````!F`@``(@`````````````````````````:!```$@`*`#`A````
M````"````````````@``$@`````````````````````````B!```$@`*`$`K
M````````1`(````````O!```$@`*`)`U````````!`$````````'`@``$@``
M``````````````````````!!!```$@`*`/`R````````>``````````0`@``
M$@````````````````````````!^`@``$@`````````````````````````A
M`@``$@````````````````````````!/!```$@`*`'`S````````G```````
M```P`@``$@`````````````````````````Y`@``$@``````````````````
M```````_`@``$@````````````````````````#"`@``$@``````````````
M``````````!$`@``$@````````````````````````"U`@``$0`5`,@"`@``
M````"`````````!,`@``$@````````````````````````#M`@``$@``````
M``````````````````!3`@``$@````````````````````````!7!```$@`*
M`/!/````````0`````````!:`@``$0````````````````````````"J`@``
M$0`4```"`@``````"`````````!@!```$@`*`%0\````````9`(````````D
M`0``$@````````````````````````!?`@``$@``````````````````````
M``#_`@``$@``````````````````````````<F5A;&QO8P!M96UM;W9E`%]?
M8WAA7V9I;F%L:7IE`')M9&ER`'-T<F-H<@!S=')N8VUP`'-T<G)C:'(`8F]O
M=%]$>6YA3&]A9&5R`'!E<FQ?8V]N<W1R=6-T`&%B;W)T`'5N;&EN:P!?7VQS
M=&%T-3``<&5R;%]F<F5E`%]?<W1A=#4P`%]E>&ET`&UE;6-P>0!P97)L7V1E
M<W1R=6-T`%]?<W1A8VM?8VAK7V9A:6P`4$Q?9&]?=6YD=6UP`%!E<FQ?<W9?
M<V5T<'8`4$Q?8W5R<F5N=%]C;VYT97AT`'!E<FQ?86QL;V,`8VQO<V5D:7(`
M96YV:7)O;@!R96%D;&EN:P!?7V]P96YD:7(S,`!M96US970`4&5R;%]S869E
M<WES;6%L;&]C`%]?<W1A8VM?8VAK7V=U87)D`%!E<FQ?9W9?9F5T8VAP=@!?
M7W-P<FEN=&9?8VAK`')E860`4&5R;%]S>7-?:6YI=#,`4&5R;%]G971?<W8`
M4&5R;%]C<F]A:U]N;V-O;G1E>'0`4&5R;%]N97=84P!097)L7V%T9F]R:U]U
M;FQO8VL`4&5R;%]S>7-?=&5R;0!?7V5R<FYO`%!E<FQ?;6=?<V5T`&UK9&ER
M`'!E<FQ?<&%R<V4`4&5R;%]N97=84U]F;&%G<P!A8V-E<W,`9V5T<&ED`&UE
M;6-H<@!V9G!R:6YT9@!097)L7V%T9F]R:U]L;V-K`%!E<FQ?<W9?<V5T<'9N
M`'!E<FQ?<G5N`&QS965K`&]P96X`<W!R:6YT9@!G971U:60`<W1R;&5N`%]?
M<T8`<W1R8VUP`%]?9&5R96=I<W1E<E]F<F%M95]I;F9O`'-T<F1U<`!?7W)E
M9VES=&5R7V9R86UE7VEN9F\`<'1H<F5A9%]A=&9O<FL`7U]P<F]G;F%M90!?
M7W!S7W-T<FEN9W,`7U]G971P=W5I9#4P`%]?<WES8V%L;`!?7W-T<&-P>5]C
M:&L`871E>&ET`%]L:6)C7VEN:70`<W1R=&]K`%]?<F5A9&1I<C,P`&QI8G!E
M<FPN<V\`;&EB;2YS;RXP`&QI8F-R>7!T+G-O+C$`;&EB<'1H<F5A9"YS;RXQ
M`&QI8F,N<V\N,3(`7U]B<W-?<W1A<G1?7P!?7V)S<U]E;F1?7P!?7V5N9%]?
M`%]E;F0`<&%R7V-U<G)E;G1?97AE8P!G971?=7-E<FYA;65?9G)O;5]G971P
M=W5I9`!P87)?8VQE86YU<`!?961A=&$`<&%R7V)A<V5N86UE`'!A<E]D:7)N
M86UE`'!A<E]M:W1M<&1I<@!S:&%?=7!D871E`&UY7W!A<E]P;`!P87)?:6YI
M=%]E;G8`<VAA7V9I;F%L`%]?8G-S7W-T87)T`'AS7VEN:70`;6%I;@!?7W-T
M87)T`'!A<E]F:6YD<')O9P!P87)?<V5T=7!?;&EB<&%T:`!P87)?96YV7V-L
M96%N`'!A<E]D:64`<VAA7VEN:70`6%-?26YT97)N86QS7U!!4E]"3T]4`"]U
M<W(O<&MG+VQI8B]P97)L-2\U+C0R+C`O86%R8V@V-"UN971B<V0M=&AR96%D
M+6UU;'1I+T-/4D4Z+W5S<B]P:V<O;&EB````0/T!```````#!````````)!3
M````````2/T!```````#!````````!!3````````0/\!```````#!```````
M`%#]`0``````6/\!```````#!````````$#]`0``````>/\!```````#!```
M`````(`#`@``````B/\!```````#!````````$C]`0``````D/\!```````#
M!````````(`"`@``````L/\!```````#!````````%C]`0``````R/\!````
M```#!````````,@"`@``````V/\!```````#!``````````"`@``````X/\!
M```````#!````````%0\``````````("```````#!````````,`"`@``````
M"`("```````#!`````````@"`@``````(`("```````#!````````.!9````
M````*`("```````#!````````"!6````````.`("```````#!````````&!7
M````````0`("```````#!````````,!9````````2`("```````#!```````
M`,A9````````4`("```````#!````````-!9````````6`("```````#!```
M`````-A9````````:`("```````#!````````*A9````````<`("```````#
M!````````+!9````````@`("```````#!````````.A9````````2/\!````
M```!!```!P``````````````4/\!```````!!```$P``````````````8/\!
M```````!!```(0``````````````:/\!```````!!```(@``````````````
M</\!```````&!```)P``````````````@/\!```````!!```,@``````````
M````F/\!```````!!```.P``````````````J/\!```````!!```30``````
M````````P/\!```````!!```5```````````````T/\!```````!!```8@``
M```````````````"```````"!````P``````````````"``"```````"!```
M!```````````````$``"```````"!```!0``````````````&``"```````"
M!```!P``````````````(``"```````"!```"```````````````*``"````
M```"!```"0``````````````,``"```````"!```"P``````````````.``"
M```````"!```#0``````````````0``"```````"!```#@``````````````
M2``"```````"!```#P``````````````4``"```````"!```$0``````````
M````6``"```````"!```%0``````````````8``"```````"!```%@``````
M````````:``"```````"!```%P``````````````<``"```````"!```&```
M````````````>``"```````"!```&0``````````````@``"```````"!```
M&@``````````````B``"```````"!```'```````````````D``"```````"
M!```'0``````````````F``"```````"!```'@``````````````H``"````
M```"!```'P``````````````J``"```````"!```(```````````````L``"
M```````"!```(@``````````````N``"```````"!```(P``````````````
MP``"```````"!```*```````````````R``"```````"!```*0``````````
M````T``"```````"!```*@``````````````V``"```````"!```+```````
M````````X``"```````"!```+0``````````````Z``"```````"!```+@``
M````````````\``"```````"!```+P``````````````^``"```````"!```
M,`````````````````$"```````"!```,P``````````````"`$"```````"
M!```-```````````````$`$"```````"!```-0``````````````&`$"````
M```"!```-@``````````````(`$"```````"!```.```````````````*`$"
M```````"!```.0``````````````,`$"```````"!```.@``````````````
M.`$"```````"!```/0``````````````0`$"```````"!```/@``````````
M````2`$"```````"!```0```````````````4`$"```````"!```00``````
M````````6`$"```````"!```0@``````````````8`$"```````"!```10``
M````````````:`$"```````"!```1@``````````````<`$"```````"!```
M2```````````````>`$"```````"!```3```````````````@`$"```````"
M!```30``````````````B`$"```````"!```3P``````````````D`$"````
M```"!```4@``````````````F`$"```````"!```50``````````````H`$"
M```````"!```5@``````````````J`$"```````"!```6```````````````
ML`$"```````"!```60``````````````N`$"```````"!```6@``````````
M````P`$"```````"!```6P``````````````R`$"```````"!```7```````
M````````T`$"```````"!```7@``````````````V`$"```````"!```7P``
M````````````X`$"```````"!```8```````````````Z`$"```````"!```
M90``````````````\`$"```````"!```9@``````````````^`$"```````"
M!```9P``````````````\'N_J?```-`1_D?Y$.(_D2`"']8?(`/5'R`#U1\@
M`]7P``#P$0)`^1`"`)$@`A_6\```\!$&0/D0(@"1(`(?UO```/`1"D#Y$$(`
MD2`"'];P``#P$0Y`^1!B`)$@`A_6\```\!$20/D0@@"1(`(?UO```/`1%D#Y
M$*(`D2`"'];P``#P$1I`^1#"`)$@`A_6\```\!$>0/D0X@"1(`(?UO```/`1
M(D#Y$`(!D2`"'];P``#P$29`^1`B`9$@`A_6\```\!$J0/D00@&1(`(?UO``
M`/`1+D#Y$&(!D2`"'];P``#P$3)`^1""`9$@`A_6\```\!$V0/D0H@&1(`(?
MUO```/`1.D#Y$,(!D2`"'];P``#P$3Y`^1#B`9$@`A_6\```\!%"0/D0`@*1
M(`(?UO```/`11D#Y$"("D2`"'];P``#P$4I`^1!"`I$@`A_6\```\!%.0/D0
M8@*1(`(?UO```/`14D#Y$(("D2`"'];P``#P$59`^1"B`I$@`A_6\```\!%:
M0/D0P@*1(`(?UO```/`17D#Y$.("D2`"'];P``#P$6)`^1`"`Y$@`A_6\```
M\!%F0/D0(@.1(`(?UO```/`1:D#Y$$(#D2`"'];P``#P$6Y`^1!B`Y$@`A_6
M\```\!%R0/D0@@.1(`(?UO```/`1=D#Y$*(#D2`"'];P``#P$7I`^1#"`Y$@
M`A_6\```\!%^0/D0X@.1(`(?UO```/`1@D#Y$`($D2`"'];P``#P$89`^1`B
M!)$@`A_6\```\!&*0/D00@21(`(?UO```/`1CD#Y$&($D2`"'];P``#P$9)`
M^1""!)$@`A_6\```\!&60/D0H@21(`(?UO```/`1FD#Y$,($D2`"'];P``#P
M$9Y`^1#B!)$@`A_6\```\!&B0/D0`@61(`(?UO```/`1ID#Y$"(%D2`"'];P
M``#P$:I`^1!"!9$@`A_6\```\!&N0/D08@61(`(?UO```/`1LD#Y$((%D2`"
M'];P``#0$;9`^1"B!9$@`A_6\```T!&Z0/D0P@61(`(?UO```-`1OD#Y$.(%
MD2`"'];P``#0$<)`^1`"!I$@`A_6\```T!'&0/D0(@:1(`(?UO```-`1RD#Y
M$$(&D2`"'];P``#0$<Y`^1!B!I$@`A_6\```T!'20/D0@@:1(`(?UO```-`1
MUD#Y$*(&D2`"'];P``#0$=I`^1#"!I$@`A_6\```T!'>0/D0X@:1(`(?UO``
M`-`1XD#Y$`('D2`"'];P``#0$>9`^1`B!Y$@`A_6\```T!'J0/D00@>1(`(?
MUO```-`1[D#Y$&('D2`"'];P``#0$?)`^1""!Y$@`A_6\```T!'V0/D0H@>1
M(`(?UO```-`1^D#Y$,('D2`"'];P``#0$?Y`^1#B!Y$@`A_6X0,"JC0``!0`
M`````````/U[O:G@``"P`*!'^?T#`)'U$P#Y]0``L+4B-9&_`@#K@@$`5/-3
M`:D4!`#1E`(5RQ,`@-*4_D/3H'IS^```/]9_`A3K<P8`D8'__U3S4T&I]1-`
M^?U[PZC``U_6'R`#U?U[O:G]`P"1\U,!J?,``+!STD?Y]1,`^?4``+"UWD?Y
M?P(5ZV(!`%1@"D"Y'R`0\8$!`%1T`D#Y<V(`D6""7_@``#_6@`(`^7\"%>OC
M_O]4\U-!J?430/G]>\.HP`-?UOS^_Y?]>[RI_0,`D?-3`:GU6P*I]V,#J4$)
M`+3T`P&J]P,`JB$`0/G@``"P`.1'^?4``+"UOD?Y(@!`^10``/F`"D#YH`(`
M^>```+``[$?Y`@<`M`(``/E#`$`YPP``-$($`)%_O`!Q8/__5$,`0#F#__\U
MF___E^```+``V$?Y``<`M.```+``K$?Y^```L!@#-9$?`P#K8@$`5!8$`-$3
M`(#2U@(8R];^0],?(`/5`7MS^"``/]9_`A;K<P8`D8'__U1W``"TX`,7JKG^
M_Y<```"0```%D;;^_Y?W``"PX```L`#$1_GW`C61_P(`ZV(!`%06!`#1$P"`
MTM8"%\O6_D/3'R`#U>!Z<_@``#_6WP(3ZW,&`)&!__]4@0)`^:("0/F`"D"Y
M*@P`E('^_Y?A``#0(0`+D0$``/G.__\7`@``\$(`&)%C`H#200"`4H``@-+K
M_O^7(`"`4KG^_Y>0__^7R/__%P``````````_7N\J?T#`)'S4P&IH`8`M/=C
M`ZGW`P"JX```L`"\1_D8`$#Y.`8`M/5;`JGV`P&J%0"`4N`"0#D?]`!Q!!A`
M>L$"`%03`T#YTP$`M+5^0)/T`QBJ'R`#U>(#%:KA`Q>JX`,3JG/^_Y>````U
M8&IU.!_T`'$``@!4DXY`^//^_[7U6T*I]V-#J>`#$ZKS4T&I_7O$J,`#7];U
M`Q>JH!Y`.!_T`'$$&$!ZH?__5+4"%TOE__\7@`(8R[4&`)%S`A6+`/Q#D\`"
M`+G@`Q.J\U-!J?5;0JGW8T.I_7O$J,`#7]83`(#2Z?__%_=C0ZD3`(#2YO__
M%_\#`]'B``"P0L!'^>$#`)']>PJI_8,"D?-;`/GS`P"J0P!`^>-/`/D#`(#2
M8/[_ET`"`#7A"T"Y`@"44B$,%!(_$$!Q)!!">L`!`%3A``"P(<!'^>-/0/DB
M`$#Y8P`"ZP(`@-*A`0!4_7M*J?-;0/G_`P.1P`-?U@``@%+T__\7X`,3JD$`
M@%+)_O^7'P``<>`7GQKN__\78?[_E_]#`]'A``"P(<!'^?U["JG]@P*1(@!`
M^>)/`/D"`(#2P```M/-3"ZGS`P"J`0!`.8$!`#7S4TNIX```L`#`1_GB3T#Y
M`0!`^4(``>L!`(#2@0@`5/U[2JG_0P.1P`-?UO5;#*EC_O^7]`,`JB`&`+3@
M`Q2JZ_[_ET`%`+0!-$`Y%30`D3^X`'&!``!4H09`.0'__S0?(`/5`#1`.1^X
M`''@!`!4X`,3JM+^_Y?V`P"JX`,5JL_^_Y?``@`+``@`$0!\0)//_O^7XP,5
MJN(#$ZKU`P"J`0``\"%@&)&Y_O^7X`,5JN$#`)$2_O^7'P0`,:```%3@"T"Y
M``P4$A\00'&@`@!4X`,5JO[]_Y?@`Q6J\/W_E^`#%*K"_O^7`/O_M>`#%*HK
M_O^7X`,3JM']_Y?S4TNI]5M,J<#__Q>@!D`Y'[@`<0'[_U2@"D`Y(/G_--7_
M_Q?@`Q6JJO__E^S__Q?S4PNI]5L,J0;^_Y<?(`/5_7N^J>$#`*K]`P"1X@``
ML$+`1_GS"P#Y_T,0T?,C`)%``$#YX`<"^0``@-("@(#2X`,3JK?]_Y<`!`#1
M'P`3ZZ@``%0'```4'_0?.!\`$^N```!4`0!`.3^\`'%@__]4X`,3JN$%@%*Z
M_?^70`(`M!\`$^O(`0!4X`,3JEW^_Y?A``"P(<!'^>,'0ODB`$#Y8P`"ZP(`
M@-*!`@!4_T,0D?,+0/G]>\*HP`-?UA\``#GR__\7X```L`#`1_GB!T+Y`0!`
M^4(``>L!`(#2X0``5/]#$)$```#P`(`8D?,+0/G]>\*H0_[_%\K]_Y<?(`/5
M_7N]J>)_@-+]`P"1XP``L&/`1_GS4P&I\P,`JO5;`JG_0Q#1]2,`D6``0/G@
M!P+Y``"`TO0#`:H```#PX0,5J@"@&)'/_?^7'P``<<T``%0$!`!1X@,`*J#*
M9#@@`P`UQ`(`-4/0.]7@``"P`+A'^1\@`]4?(`/5X@,4JN$#$ZI@:&#XK_W_
ME^```+``P$?YX@="^0$`0/E"``'K`0"`TB$#`%3_0Q"1\U-!J?5;0JG]>\.H
MP`-?UN(#!"H?(`/55GQ`D^`#%:KB`Q:JX06`4@3^_Y>`_/^T1-`[U>```+``
MN$?Y'R`#U1\@`]7C`Q:JX@,5JN$#$ZJ`:&#X!?[_E^+__Q>'_?^7'R`#U1\@
M`]7_@P'1_7L!J?U#`)'S4P*I\P,`JO0#`:KU6P.I]V,$J>```+``P$?Y(0!`
M.0(`0/GB!P#Y`@"`TC_T`'&4%I2:X`,4J@_^_Y?V`P"JX1,`D>`#$ZJK_O^7
M(`@`M/4#`*H(_O^7WP(`Z^D&`%3W!X"Y^```L!B_1_GW\GW38`)`.1_T`'$$
M&$!ZP`L`5.0#$ZJ#'$`X?_0`<6080'JA__]4A``3RX1\0),5`T#YP`H`D0``
M!(OW_?^7H&HW^*`"`+0!`T#YX`,3JB-H=_@!%$`X80``.3_T`'$D&$!ZP```
M5`(40#AB'``X7_0`<4080'J!__]4H0>`4@``@-)A%``X@FI@.&)H(#@`!`"1
MHO__->```+``P$?YX@=`^0$`0/E"``'K`0"`TF$'`%3]>T&I\U-"J?5;0ZGW
M8T2I_X,!D<`#7]8!`(#2@FIA.*)J(3@A!`"1HO__->[__Q?X``"P&+]'^?EK
M!:D:`T#Y0`-`^<`$`+3@`QJJ%0"`4A\@`]4!C$#XX@,5*K4&`!&A__^U0`P`
M$0!\?9/A``#0(BA#N0("`#4B`(!2(B@#N;G]_Y?Y`P"J0`,`M+=^?9/A`QJJ
MX@,7JL?\_Y<9`P#YH`8`$?4'`+D_VR#X^6M%J:C__Q?A`P"JX`,:JKK\_Y<`
M`P#Y^0,`JD`!`+2W?GV3]/__%P0`@-*I__\7``*`TA4`@%+B__\7^6L%J?[\
M_Y?Y:T6IO/__%_U[OZE!T#O5(P``\/T#`)'@``"P`+A'^1\@`]4?(`/5_7O!
MJ&-@'9$@:&#X`0``\.(``+!"J$?Y(0`9D3#]_Q?]>[ZI0=`[U?T#`)'S"P#Y
MX```L`"X1_D?(`/5'R`#U3,``(L@:&#XYO__EV`"0/D$``#P\PM`^81`)I']
M>\*HXP,$JN(``+!"\$?Y`0``\`4`@%(A@!F1-?W_%P``@-+``U_6'R`#U1\@
M`]7]>[NIX@``L$+`1_G]`P"1\U,!J?,#`:KU6P*I_\,2T?4#`*I``$#YX%<"
M^0``@-+A,P"1````\`#@&9']_?^7X06`4O0#`*K@`Q6J@?S_ET`"`+3A`Q6J
M````\``@&I$T__^7X`,5JN$``+`AP$?YXU="^2(`0/EC``+K`@"`TH$-`%3_
MPQ*1\U-!J?5;0JG]>\6HP`-?UN`#$ZHA_?^7%@``\,%B&I&V_/^7\P,`JB#]
M_[3W<P+Y]Z,"D?AW`OD8``#P&*,:D?E[`OD9``#P.8,:D?I_`OE@`D`YX`8`
M-+0``+3A`Q.JX`,4JCG]_Y<`!0`TX`,3JB[]_Y<"!`#1^@,`JF("`HM_`@+K
MHP``5`H``!1?]!\X?P("ZX```%1``$`Y'[P`<6#__U3@`Q.J(/W_E_H#`*K@
M`Q6J'?W_EP``&HL`!`"1'_P/\<@"`%3D`Q.JY@,5JN4#&:KC`QBJ`H"`T@$`
M@%+@`Q>JI?S_E^%#`)'@`Q>J7OS_EZ```#7@&T"Y``P4$A\@0''``@!4P6(:
MD0``@-)Z_/^7\P,`JL#Y_[7W<T+Y^'="^?E[0OGZ?T+YJ/__%U0#`+2``D`Y
M'[@`<:```%03``#P.@"`TG."&)'9__\7@`9`.:#]_S3Z__\7X`,7JB$`@%*Y
M_/^7`/W_->$#%ZH```#P`"`:D<S^_Y?@`Q>JQOS_E_=S0OGX=T+Y^7M"^?I_
M0OF3__\7$P``\#H`@-)S@AB1P___%_=S`OGX=P+Y^7L"^?I_`OE`_/^7_7N^
MJ>$%@%+]`P"1\PL`^?,#`*H*_/^7'P``\6`&@)KS"T#Y_7O"J*W\_Q<``P"T
M_7N^J?T#`)'S"P#Y\P,`J@$`0#DA`0`TR?S_EP`$`)$?`!#Q2`$`5.`#$ZKS
M"T#Y_7O"J"+^_Q?S"T#Y````\/U[PJ@`@!B1F?S_%_,+0/D``(#2_7O"J,`#
M7]8```#P`(`8D9+\_Q<?(`/5_X,!T>```+``P$?Y_7L!J?U#`)'S4P*I%```
M\/,3`)'U6P.IE,(:D?=C!*GY*P#Y]0``L+6^1_D!`$#YX0<`^0$`@-+A`Q.J
MX`,4JD']_Y<``@"TH@)`^>,'@+E`#`.+`01`^4%X(_CA_O^T'R`#U0$(0/D!
M!`#Y`"``D:'__[7A`Q.JX`,4JC+]_Y=`_O^U]0``L+6^1_D4``#PE`(;D>$#
M$ZK@`Q2J*OW_E^`!`+2B`D#YXP>`N4`,`XL!!$#Y07@C^.'^_[0!"$#Y`00`
M^0`@`)&A__^UX0,3JN`#%*H<_?^78/[_M?4``+"UOD?Y%```\)0B&Y'A`Q.J
MX`,4JA3]_Y?@`0"TH@)`^>,'@+E`#`.+`01`^4%X(_CA_O^T`0A`^0$$`/D`
M(`"1H?__M>$#$ZK@`Q2J!OW_EV#^_[7U``"PM;Y'^10``/"48AN1X0,3JN`#
M%*K^_/^7X`$`M*("0/GC!X"Y0`P#BP$$0/E!>"/XX?[_M`$(0/D!!`#Y`"``
MD:'__[7A`Q.JX`,4JO#\_Y=@_O^U]@``L-:^1_D4``#PE8(;D>$#$ZK@`Q6J
MZ/S_E^`!`+3"`D#YXP>`N4`,`XL!!$#Y07@C^.'^_[0!"$#Y`00`^0`@`)&A
M__^UX0,3JN`#%:K:_/^78/[_M?8``)#6OD?Y%0``T+7"&Y'A`Q.JX`,5JM+\
M_Y?@`0"TP@)`^>,'@+E`#`.+`01`^4%X(_CA_O^T`0A`^0$$`/D`(`"1H?__
MM>$#$ZK@`Q6JQ/S_EV#^_[7W``"0][Y'^14``-"VXAF1X0,3JN`#%JJ\_/^7
MX`$`M.("0/GC!X"Y0`P#BP$$0/E!>"/XX?[_M`$(0/D!!`#Y`"``D:'__[7A
M`Q.JX`,6JJ[\_Y=@_O^U^```D!B_1_D6``#0UP(<D>$#$ZK@`Q>JIOS_E^`!
M`+0"`T#YXP>`N4`,`XL!!$#Y07@C^.'^_[0!"$#Y`00`^0`@`)&A__^UX0,3
MJN`#%ZJ8_/^78/[_M?D``)`YOT?Y%P``T/A"')'A`Q.JX`,8JI#\_Y?@`0"T
M(@-`^>,'@+E`#`.+`01`^4%X(_CA_O^T`0A`^0$$`/D`(`"1H?__M>$#$ZK@
M`QBJ@OS_EV#^_[7Y``"0.;]'^1@``-`8@QR1X0,3JN`#&*IZ_/^7X`$`M"(#
M0/GC!X"Y0`P#BP$$0/E!>"/XX?[_M`$(0/D!!`#Y`"``D:'__[7A`Q.JX`,8
MJFS\_Y=@_O^U^0``D#F_1_D8``#0&",:D>$#$ZK@`QBJ9/S_E^`!`+0B`T#Y
MXP>`N4`,`XL!!$#Y07@C^.'^_[0!"$#Y`00`^0`@`)&A__^UX0,3JN`#&*I6
M_/^78/[_M>$#$ZH```#0`,`<D5'\_Y?A`P"J8```M.!"')&-_?^7X0,3J@``
M`-``(!V12?S_E^$#`*J```"T````T`"`'9&$_?^7X0,3J@```-``P!V10/S_
ME^$#`*J``@"TH.(9D7S]_Y?@``"0`,!'^>('0/D!`$#Y0@`!ZP$`@-*!`@!4
M_7M!J8""&Y'S4T*I`0``T/5;0ZDA8!Z1]V-$J?DK0/G_@P&1:_W_%^$#$ZH`
M``#0```>D2?\_Y?A`P"J0/W_M,`"')%C_?^7Y___%^7Z_Y?_@P#1````T.(`
M`)!"P$?YX1,`D?U[`:G]0P"1```<D4,`0/GC!P#Y`P"`TA7\_Y?@`0"T``!`
M.1_``'$$&$!ZX`>?&N$``)`AP$?YXP=`^2(`0/EC``+K`@"`TL$``%3]>T&I
M_X,`D<`#7]8``(!2]?__%\?Z_Y<?(`/5'R`#U?^#!-'K!H`2Z0``D"G!1_GL
M@P.1_7L%J?U#`9'J#X`2Z`,`D>`;@#WA'X`]XB.`/>,G@#WD*X`]Y2^`/>8S
M@#WG-X`]X8L.J>$#`*KB`PBJXY,/J>6;$*GGCP#YX```D`#H1_DC`4#YXR<`
M^0,`@-+C@P21XX\"J>.C`)'L'P#Y`,`$D>LK""E@!$"M``4`K13[_Y?@'X!2
M2OK_EQ\@`]7]>[RI0@"`4OT#`)'A``"0(<!'^?-3`:GT`P`J]5L"J?=C`ZG_
M0P#1_T-`T>-#0)$D`$#Y9`0`^00`@-(!`(#2$/O_EQ,$`-'@`Q/K83Y`D@`\
M0)(@1(#:<P(`ZT0'`%07``#0%@``T!4!@-+X`P"1]X(>D=;"'I$U`*#R'R`#
MU>$#$ZK@`Q0J`@"`4OWZ_Y<?!`"QP`4`5.(#%:KA`QBJX`,4*J?Z_Y<?!``Q
M(`0`5`!\0),?'`#Q"00`5``@`-$!`P"KH@,`5.,``+`$```4(00`T3\`&.L#
M`P!4(@!`^6`(0?E?``#K0?__5"$`&,L@`!.+X0``D"'`1_GC0T"19`1`^2(`
M0/F$``+K`@"`TD$"`%3_0P"1_T-`D?-30:GU6T*I]V-#J?U[Q*C``U_6X`,6
MJHO__Y=S0D#1?T)`L0'Z_U0``("2Z___%^`#%ZJ$__^7T?__%T;Z_Y<?(`/5
M_7N_J?T#`)'6^O^7S?K_ET```+0``$#Y_7O!J,`#7];_`P'1X0``D"'`1_G]
M>P&I_4,`D?-3`JGT`P"J]5L#J14``-"V`A^1(@!`^>('`/D"`(#2X`,6JN$3
M`)%I^_^7@```M/,#`*H``$`YX`$`->```)``P$?YX@=`^0$`0/E"``'K`0"`
MTL$$`%3]>T&IX0,4JO-30JF@`A^1]5M#J?\#`9&7_/\7X`,4JK7Z_Y?U`P"J
MX`,3JK+Z_Y>@`@"+``@`D;/Z_Y?D`Q.JX@,4JO,#`*H#``#0`0``T&-@&I$A
MH!J1F_K_E^```)``P$?YX@=`^0$`0/E"``'K`0"`T@$!`%3]>T&IX0,3JO-3
M0JG@`Q:J]5M#J?\#`9%Y_/\7_/G_E_]#!M'A``"P(4`(D>+C`)$DH`"1_7L4
MJ?T#!9'C``"08\!'^?-3%:GS@P"1*"1!J?5;%JGU`P"J)IQ%J?=C%ZGX(P"1
M@0A`K6``0/G@GP#Y``"`TH`(P#WT$P"1)!!`^18``-`C-$#YP.(9D>$#%*I!
M"`"MZ*<`J6,*`/D$"P#YYA\"J4`(@#T5^_^7@```M`$`0#GW`P"JH1T`-?EK
M&*F/__^7^0,`JF`-`+1L^O^7`/A_TP`$`)%M^O^7(@-`.?,#`*KW`P"J(@$`
M-!H``-!:8Q^1X`,3JN$#&JI4^O^7<PH`D2(?0#AB__\U````T/D#`9$`@!V1
M`P``%""'0/B@`0"TX0,4JO7Z_Y?S`P"J8/__M`$`0#DA__\T+?O_E^#^_S3@
M`Q.J*?K_E_,#`*I@`0"U$P-`^3,!`+1@`D`Y@`P`-.`#$ZHB^_^78`D`-1,'
M0/D8(P"1,___M>`#$ZH_^O^7^`,`JN`#%ZH\^O^7&`,`"Q@3$!$9``#0&']`
MD^`#&*HZ^O^7Y0,7JN(#$ZHC@QJ1!```T(2`'Y'W`P"J`0``T"&@'Y$A^O^7
MX`,7J@$X@%+F^?^7'P0`,:`&`%3AHP&1X`,7JFWY_Y?S`P`JH```->!S0+D`
M#!02'Q!`<<`'`%2B`D#Y`0``T.```)``Z$?YXP,7JB'@()$`P`21;_G_EQ0`
M@-+@``"0`,!'^>*?0/D!`$#Y0@`!ZP$`@-*A'`!4_7M4J>`#%*KS4U6I]5M6
MJ?=C5ZGY:UBI_T,&D<`#7];B^?^7^0,`J@#S_[5S(@"18`)`^8`1`+3A`Q2J
MG_K_EV#__[0!`$`YP?[_-7,B`)'X__\7X`,3JA@C`)'3^?^7\P,`JF#U_[2T
M__\7I_G_EP``0+D?1`!Q(/G_5+,"0/FB^?^7Y0``D*7H1_GC`Q>J!`!`N>(#
M$ZJ@P`21`0``T"'@'Y$]^?^7SO__%Q,`@-*B__\7^H=`N=CY_Y=?`P!K`?C_
M5.!S0+D`(``2'P`'<8'W_U3@`QBJW/G_E^$#%*KT`P"J````T`!`(I%S^O^7
MX0,`JH`,`+2@`D#Y8_S_E_H#`*H`#`"TX`,:JLOY_Y?X`P"J'PP`<<T!`%0`
M?$"3`0``T``0`-$A8"*10`,`B\KY_Y?@```U'Q,`<>`0`%08%P!10,MX.!^\
M`'%@$`!4.O[_E\```#7@`QJJ`0"`4J+Y_Y?U`P`J(`D`-0$``-````#0(6`>
MD0``')&/^_^7-8,:D1@``-`80R:1=_G_EQD``-#D`P`JY0,8JN,#%:KB`Q>J
MX`,4J@$``-`Y8R.1(2`CD97Y_Y</```44_G_EP``0+D?1`!Q`0(`5&?Y_Y=S
M!@`1Y`,`*N4#$RKF`QBJXP,5JN(#%ZKA`QFJX`,4JH;Y_Y?@`Q2J`3B`4DOY
M_Y<?!``QP/W_5.`#%ZK#^/^7X0,4JL#B&9%H^_^7X`,4JJ[^_Y=P__\7K/[_
ME^```)``P$?YXI]`^0$`0/E"``'K`0"`TH$*`%3]>U2IX`,7JO-35:GU6U:I
M]V-7J?]#!I%2^?\7H`&`TGCY_Y<9``#0\P,`JCE#'Y'W`P"J8@J`4@K__Q>Z
M`D#YH/__%S/^_Y?X`P"JP/;_MP$H`-$"`(!2X`,5*OH#!)%,^?^7X0,:JN`#
M%2K"`(#2^/C_EQ\8`/&```!4````T`"@(I'[_?^7X0-!N0!@B%(@:*AR/P``
M:V'T_U1!"T!Y`*F(4C\``&OA\_]4`<L`T0(`@%+@`Q4J\R,$D37Y_Y?@`Q4J
MX0,3J@(%@-+A^/^7'Z``\8```%0```#0`*`BD>3]_Y?D`Q.J(X,:D>(#%ZK@
M`Q2J!0``T`$``-"E0":1(>`BD?_#!#DN^?^7K?__%[4&0/FU[_^TX`,5JC7Y
M_Y<?#`#Q*>__5``0`-$!``#0H`(`BR&`(I$V^?^7'P``<5H3E9IQ__\7COC_
ME_EK&*F,^/^7_T,!T?U[`:G]0P"1X```D`"X1_D?(`/5'R`#U>$``)`AP$?Y
M\U,"J5/0.]7U6P.I]1,`D?<C`/ET:F#X(`!`^>`'`/D``(#2````T.$#%:H`
M@!N1@L)".8+&`CFP^?^7@```M`$`0#D_Q`!Q(`T`5%?\_Y?@``"0`+A'^1\@
M`]4?(`/5=&I@^.`#%*H!``#08P"`4B'`(Y$B`(!2D/C_E_0#`*H``@"T@0I`
M^>```)``N$?Y'R`#U1\@`]5V`@"+`@``T&!J8/A"X".1(0!`^5_X_Y>`"D#Y
M`0!`^2`,0+G`"K`WX```D`"X1_D?(`/5'R`#U7<"`(M@:F#X`0``T&,`@%(A
M`"21(@"`4O8``+!S^/^7]`,`JF`!`+3A`Q6J````T``@&I%]^?^7X@,`JF`'
M`+2!"D#YX`)`^2$`0/E#^/^7X```D`"X1_D?(`/5'R`#U70"`(MS:F#X(@"`
M4@$``-`A("21X`,3JFSX_Y<"``#0X0,`JD*`))'@`Q.J,_C_E].B#)&"`D#Y
MX0,5J@```-``X!F17\0".6#Y_Y=@"@#Y@`(`M`$X@%)X^/^7`00`$3\$`'$$
MB%%Z800`5.```)``P$?YX@=`^0$`0/E"``'K`0"`TF$#`%3]>T&I\U-"J?5;
M0ZGW(T#Y_T,!D<`#7]9@!D#Y$O[_EV`*`/E`_?^U````T`"@))%(^/^7``1`
M.:#S_S26__\7P*(,D8(*0/D!!$#Y0`!`^2$`0/DW^O^7Q/__%\`"0/E0^/^7
MJ?__%_KW_Y?6H@R1XP,`*@```-``0"61P8I`J2$`0/DS^/^7'R`#U1\@`]7]
M>[ZI_0,`D?,+`/GS`P"J"/W_EW\"`/$$&$!Z@0``5/,+0/G]>\*HP`-?UF`"
M0#F`__\TX`,3JJ[[_Y>B^_^7`0``T((`@-(A@!^1I??_EX#^_S7@`Q.J\PM`
M^?U[PJAX^?\7````````````````_7N_J0'@`)$$X`&1XP``D&/`1_G]`P"1
M4>0`;_]#"M'PY0!OX@,`D8?D`&]E`$#YY4<!^04`@-+C`P*J(P!`;2($P3QC
M.(`.001@;T`$6&]C""`N0@1(;R946$\`'#%.9*0@+R%42$_#'#!.`!RB3B$<
M)TX`'*-.!1RA3F2,GTR?``'KP?W_5/"?0*U"H`F1YH]!K>&C`9'EDT*MX1_`
M/?'E`&\?(`/5(H#9/"%``)$@`-\\`!PB;@`<(VX`'#!N\!RG3L<<IDYF'*-.
M`E1!3P`$86^C'*5.A1RD3B0<H4Y`'*!.`1PQ3B&`@#P_``+KP?W_5`<H0*DD
M,X_2""1!J410J_+E/T"I(W2=TN+\6]/L_$+30A0'JHQY!ZHA`0C*(]NM\B$`
M"HHA``G*I0`$BR)`(HM!_4+3"Q!`^2%X"JHM?$"20@`%BZ4!",I"``L+YD-!
MJ:4`!XJE``C*[P$$BT[\6].E``F+SA4"JB$`#,JE``^+(0`"BJ4`#@M/_$+3
M(0`-RN)Y`JK&``2+KOQ;TR$`"(O.%06J(0`&BX8!`LHA``X+C'U`DL8`!8JN
M_$+3Q@`,RA`"!(O&``V++_Q;T\UY!:KO%0&J[A-`^44`#<K&`!"+0GQ`DL8`
M#PNE``&*I0`"RC#\0M/.`02+I0`,B\_\6],,>@&J[Q4&JJ4`#HNE``\+H0$,
MRL_\0M,A``:*YGD&JJU]0)+NOT*I(0`-RB$``HNP_$+3P@$$BZ[\6],A``*+
M@@$&RD(`!8J,?4"2SA4%JD(`#,HN``X+!7H%JN$!!(M"``V+0@`!B\$`!<K0
M?$"2(0`.BB$`$,K-_5O3(0`,BZT5#JKOLT.I30`-"\;]0M/.>`ZJIGQ`DN(!
M!(NO_5O3(0`"BZ(`#LKO%0VJ0@`-BB\`#PNE_4+30@`&RH$!!(NE>`VJ0@`0
MBT(``8OL_5O3P0$%RO#]0M.,%0^J(0`/B@]Z#ZI,``P+[<-$J<Y]0)(A``[*
M(0`&BX;]6]/&%`RJH@$$BQ`"!(LA``*+H@`/RBT`!@NE?$"20@`,BH;]0M-"
M``7*QG@,JJ']6]-"``Z+0@`0BRP4#:JP_4+3X0$&RB$`#8H->@VJ[L-%J4P`
M#`OO?4"2(0`/RB$`!8N%_5O3I10,JL(!!(L0`@2+(0`"B\(`#<HN``4+QGQ`
MDD(`#(J%_4+30@`&RJ5X#*K!_5O30@`/BT(`$(LL%`ZJT/U"TZ$!!<HA``Z*
M#GH.JN_#1JE,``P+K7U`DB$`#<HA``:+AOU;T\84#*KB`02+$`($BR$``HNB
M``[*+P`&"Z5\0))"``R*AOU"TT(`!<K&>`RJX?U;TT(`#8M"`!"++!0/JO#]
M0M/!`0;*(0`/B@]Z#ZKMPT>I3``,"\Y]0)(A``[*(0`%BX7]6].E%`RJH@$$
MBQ`"!(LA``*+P@`/RBT`!0N%_4+3I7@,JD(`#(JA_5O3QGQ`DBP4#:KA`07*
M(0`-BN]]0))"``;*(0`/RD(`#HLA``:+[IM(J4(`$(M,``P+L/U"TPUZ#:J0
M_4+3P@$$BX[]6],A``*+SA4,JJ(`#<HN``X+I7Q`DD(`#(I"``7*#'H,JL$`
M!(M"``^+QOU;TT(``8OO3T#YH0$,RL84#JJM?4"2T@`""R$`#HHA``W*Y`$$
MBR$`!8O0_4+31?Y;TPYZ#JKO4T#Y(0`$BZ44$JI&0":+L``!"X0!#LJ$``9*
MX@$#BX0`#8L&_EO33?Y"TX0``HOB5T#YLGD2JL84$*HE0"6+SP`$"\$!$LHE
M``5*00`#BZ)`+(L,_D+3Y?U;TXUY$*I"``&+I10/JN%;0/FQ``(+AD`FBT0"
M#<J$``9*[/U"TR8``XN!0"Z+)/Y;TX]Y#ZHA``:+A!01JN9?0/F0``$+14`E
MBZ(!#\I%``5*+OY"T]%Y$:JL0#*+P@`#BP7^6]/F8T#Y@@$"BZ44$*HD0"2+
MX0$1RJX``@LD``1*#/Y"T\$``XN&0"V+D'D0JL$``8OF9T#YQ/U;TT5`)8N$
M%`ZJ(@(0RHT``0M"``5*Q@`#BT)`+XO,_4+3CGD.JD(`!HOF:T#YI?U;TZ44
M#:HD0"2+`0(.RJ\``@LA``1*K/U"TXUY#:K&``.+(4`QB^3]6],A``:+14`E
MB^9O0/G"`0W*A!0/JD4`!4J1``$+K$`PB_#]0M/"``.+YG-`^0]Z#ZHE_EO3
M)$`DBZ$!#\J"`0*+I101JB0`!$JP``(+P0`#BRS^0M.&0"Z+C'D1JL$``8OF
M=T#Y14`EBP3^6]/B`0S*A!00JD(`!4J.``$+Q@`#BT)`+8L-_D+3\7M`^4(`
M!HNF>1"J)$`DB\7]6].!`0;*I10.JB$`!$JM``(+,`(#BR%`+XO/_4+3(0`0
MB^YY#JKP?T#YI/U;TX04#:I%0"6+L?U"T\(`#LJ/``$++7H-JD4`!4H"`@.+
M\(-`^21`)(O!`0W*K$`LBR0`!$KE_5O3I10/JH9`)HOQ_4+3`0(#B\$``8N"
M`0*+)GH/JJP``@M%0"6+H@$&RD(`!4J$_5O30D`NBX04#*KPNU"ID?U"TX\`
M`0LL>@RJ)$`DB\$`#,KE_5O3(0`$2O']0M.E%`^J$`(#BR]Z#ZI"`!"+(4`M
MB^W'4:FP``(+S@$#BT5`)8L$_EO3(0`.BX04$*J"`0_*C@`!"T(`!4I"0":+
MI0$#BPW^0M-%``6+QOU;TZ)Y$*K&%`ZJ)$`DB^$!`LK-``4+)``$2B$"`XN$
M0"R+T?U"T_"70/F$``&+K/U;TR%Z#JJ,%0VJID`FBT4``<J.`00+I0`&2A`"
M`XNE0"^+K_U"T^]Y#:JE`!"+\)M`^<;]6]/&%`ZJC$`LBR0`#\K-``4+C``,
M2M']0M,$`@.+@D$BB[#]6]-"``2+)'H.JJ9`)HOLGT#Y$!8-JN4!!,I"`!`+
MI0`&2J[]0M.A0"&+Q7D-JH8!`XM._$+3C``%JD/\6]-C%`*JC`$"BL)Y`JHA
M``:+[3M4J2$``PNC``0*AIN7THP!`ZIFX['R@T$OBR_\0M.L`0:++?Q;TZT5
M`:IC``R+K``"JF,`#0N,`0&*30`%"HP!#:KA>0&JS@$&BX1!)(MM_%O33``!
MJJT5`ZJ$``Z+A``-"XP!`XHM``(*;OQ"TXP!#:J-_%O3A4$EB\QY`ZKO.U6I
MHQ4$JBT`#*KO`0:+S@$&BZ4`#XN/_$+3HP`#"Z4!!(J-`0$*I0`-JFW\6].E
M0"*+XGD$JJT5`ZJ$`0*JI0`.BX0``XJE``T+30`,"H0`#:IM_$+3A$`ABZ%Y
M`ZKO-U:IKOQ;T\X5!:I#``&J8P`%BN\!!HNM`0:+A``/BZ_\0M.$``X++@`"
M"F,`#JKE>06J8T`LBX[\6]/.%02J+``%JF,`#8N,`02*[;M`^6,`#@NN``$*
MC_Q"TXP!#JKD>02J@D$BBV[\6].L`0:+SA4#JNV_0/E"``R+K``$JD(`#@N,
M`0.*C@`%"HP!#JIO_$+3XWD#JJT!!HN!02&+3OQ;T\X5`JJ,``.J(0`-BXP!
M`HHA``X+;@`$"HP!#JI._$+3PGD"JH5!)8OM.UBI+_Q"TZP!!HLM_%O3K14!
MJJ4`#(ML``*JI0`-"XP!`8I-``,*C`$-JN%Y`:K.`0:+A$$DBZW\6]-,``&J
MK14%JH0`#HN$``T+C`$%BBT``@JN_$+3C`$-JHW\6].#02.+S'D%JN\[6:FM
M%02J)0`,JJ4`!(KO`0:+S@$&BV,`#XN/_$+38P`-"XT!`0JE``VJY'D$JJ)`
M(HMM_%O3K14#JH4!!*I"``Z+I0`#BD(`#0N-``P*I0`-JFW\0M.A0"&+I7D#
MJN\W6JE._%O3SA4"JH,`!:IC``**[P$&BZT!!HLA``^+KP`$"B$`#@MC``^J
M3OQ"TVQ`+(O">0*J(_Q;TXP!#8MC%`&JK0`"JB_\0M.#`0,+K`$!BN%Y`:I-
M``4*[C];J8P!#:J$022+;?Q;TZT5`ZK,`0:++@`""H0`#(M,``&JA``-"XP!
M`XJ,`0ZJ;?Q"TZUY`ZKO`0:+A4$EBX[\6]/LXT#YHP`/B\X5!*HE``VJ8P`.
M"Z4`!(JN`0$*C_Q"TZ4`#JIN_%O3HD`BBX4!!HOD>02JSA4#JD(`!8NE`02J
M0@`."V[\0M.E``.*PWD#JNR[7*F/``T*I0`/JD_\0M.E0"&+0?Q;TR$4`JJ,
M`0:+I0`,BXP``ZJE``$+C`$"BF$`!`KB>0*JC`$!JE```PJ!02V+S0$&BZS\
M6],A``V+[^]`^6T``JJ,%06JK0$%BHX!`0NM`1"JI$$DB[#\0M/M`0:+QOU;
MT^_S0/F$``V+QA0.J@UZ!:K1``0++$`LBT$`#<K%.IC2(0`,2D5,N?(A0".+
M[`$%B]#]0M,C_EO3[_=`^0YZ#JIC%!&J(0`,BVP``0N&0":+I`$.RB-`(XN$
M``9*Y@$%BX)`(HLO_D+3A/U;T_%Y$:I"``:+A!0,JN;[0/F2``(+P0$1RD1`
M)(LC``-*P0`%BV9`+8N-_4+30_Y;T[!Y#*K&``&+8Q02JN'_0/EO``8+(@(0
MRDS^0M-"``1*)``%BT)`+HOA_5O3C7D2JD0`!(OB`T'Y(10/JBX`!`O#0".+
M!@(-RNS]0M/#``-*1@`%BV-`,8O"_5O3CWD/JF,`!HOF!T'Y0A0.JE$``PN!
M0"&+I`$/RF)`(HN!``%*Q``%BRQ`,(O0_4+3(?Y;TPYZ#JKF"T'YA`$$BR$4
M$:KC`0[*,``$"V(``DHL_D+3PP`%BT9`+8L"_EO3C'D1JL,``XOF#T'Y0A00
MJDT``PN!0"&+Q`$,RA'^0M.!``%*Q@`%BR1`+XNA_5O3[Q-!^80`!HLA%`VJ
M)GH0JF)`(HLP``0+@P$&RN\!!8MB``)*L?U"TT-`+HL"_EO3[A=!^2UZ#:IC
M``^+0A00JD\``PN!0"&+Q``-RF)`(HN!``%*Q`$%BRQ`+(L._D+3X?U;T]!Y
M$*J$`02+(10/JNP;0?DQ``0+HP$0RH%`(8MB``)*@P$%BT9`)HOL_4+3(OY;
MTX]Y#ZK#``.+0A01JN8?0?E.``,+!`(/RF)`(HN!``%*Q@`%BR1`+8LM_D+3
MP?U;TX0`!HOL(T'YIGD1JB$4#JKC`0;*+0`$"V(``DJ,`06+0T`PB]#]0M.B
M_5O3#GH.JF,`#(OL)T'Y0A0-JE$``PN!0"&+Q``.RF)`(HN!``%*A`$%BRQ`
M+XNO_4+3(?Y;T^UY#:J$`02+(101JNPK0?DP``0+PP$-RB_^0M-B``)*C`$%
MBT)`)HL#_EO31@`,BV,4$*KB+T'Y['D1JH%`(8MO``8+I`$,RL-`(XN$``%*
M00`%BX1`+HOB_5O3#OY"TX0``8OQ,T'YP7D0JD(4#ZJ&`0'*3@`$"\,``THP
M`@6+9D`MB_']0M/#_5O3[3=!^6,4#JHO>@^JQ@`0BW``!@N"0"*+)``/RL9`
M(XN$``)*H@$%BXQ`+(O-_4+3!/Y;TXP!`HOB.T'YA!00JJYY#JJ#``P+[0$.
MRN=`+XNM`09*1@`%BZ%!(8MB_%O30A0#JB$`!HOM/T'Y(0`""P+^0M.&022+
M0G@0JB3\6]/,`0+*A!0!JHP!!DJF`06+9?Q"T\8`!XNC>`.JA``,B\<``/#G
MP$?Y(0`*"T(`"0O%`0L+A``&"V,`"`L$!`"I`P@!J040`/GB1T'YX0!`^4(`
M`>L!`(#2@0``5/]#"I']>\&HP`-?UJ?S_Y<?(`/5'R`#U?U[OZD`$(#2_0,`
MD4'T_Y<?&`#Y(@``D"$``)`?>`"Y0>#!/2(``)`BW,$]0.3!/0($`*T`"(`]
M_7O!J,`#7];]>[RI0GQ`D_T#`)'S4P&I%.``D?,#`:KU6P*I]0,`JO8#`JKW
M8P.I!8Q"J0!X0+FD#`(+OP`$ZV.4@YIB=$*+I(H"J>`"`#07"(!2]P(`2_\"
M%FN`PB"+]]*6&OA^0)/B`QBJ<_/_EZ!Z0+G@`@`+H'H`N1\``7'```!4\U-!
MJ?5;0JGW8T.I_7O$J,`#7];6`A=+<P(8B^`#%:J9^_^7W_X`<6T#`%36`@%1
MV'X&4Q<'`!'W:GK3=P(7BQ\@`]5@!D"MX`,5JH`&`*U@!D&M<P(!D8`&`:V*
M^_^7?P(7ZP'__U36&AA+X0,7JL)^0)/@`Q2J3_/_E[9Z`+GS4T&I]5M"J?=C
M0ZG]>\2HP`-?UO<#$ZKU__\7'R`#U1\@`]7]>[RIX@^`$OT#`)'S4P&I]`,`
MJO,#`:KU6P*I-N``D>@;`/TU%$#Y*!A`_:`B`U,!!``1PLH@.,#"(8L_X`!Q
M30L`5`((@%)"``%+`0"`4D)\0)-;\_^7X`,3JF'[_Y=_-@#Y``0`3\`"`*W`
M"H`]!05P?P`%:'\$!7A_H0)GGN`#$ZHC!&A_(@1P?Z`$`VXA!'A_@`0%;@`%
M!VY@!`EN0`0+;B`$#6Z@'@].8#H`_4S[_Y=A`D#YX`,3JB'\6-.!`@`Y80)`
M^2'\4-.!!@`Y80)`^2'\2-.!"@`Y80)`^8$.`#EA!D#Y(?Q8TX$2`#EA!D#Y
M(?Q0TX$6`#EA!D#Y(?Q(TX$:`#EA!D#Y@1X`.6$*0/DA_%C3@2(`.6$*0/DA
M_%#3@28`.6$*0/DA_$C3@2H`.6$*0/F!+@`Y80Y`^2'\6-.!,@`Y80Y`^2'\
M4-.!-@`Y80Y`^2'\2-.!.@`Y80Y`^8$^`#EA$D#Y(?Q8TX%"`#EA$D#Y(?Q0
MTX%&`#EA$D#Y(?Q(TX%*`#EA$D#Y@4X`.?-30:GU6T*IZ!M`_?U[Q*BW\O\7
M`@>`4D(``4L!`(!20GQ`DP+S_Y>M__\7P```\`!`"Y$!0$$Y00,`-<$``-`A
MI$?Y(@"`4@)``3FA`0"T_7N_J<```/#]`P"1``1!^8?R_Y?```#0`-1'^6`!
M`+3]>\&H(```D```)9$T\_\7P```T`#41_G```"T(```D```)9$N\_\7_7O!
MJ,`#7];``U_6``````````#```#P`4`+D0)`2SE"`0`UP@``T$*T1_DC`(!2
M`T`+.:(``+0A0`"1(```D```)9&T\O\7P`-?U@``````````_X,!T<,``-!C
MP$?Y_7L#J?W#`)'BAP"IXB,`D>%#`)'@'P"YX',`D?-3!*GU6P6I9`!`^>07
M`/D$`(#2U?+_E\(``-!"S$?YP```T`#@1_GA`P*JW_+_E\```-``L$?Y``!`
M.<`)`#1!T#O5TP``\,```-``N$?Y'R`#U1\@`]7T'T"Y(FA@^(`2`!&4#@`1
M`'Q]DT$(4CDA!``R00@2.:SR_Y?C`P"JY@M`^6*B#)'```#0`,A'^0$``)#'
M`$#Y(6`FD00`0/D```"0Y1]`N0"`)I%G!`"I0P0`^60``:F_!`!Q;0D`5"$`
M@-(?(`/5(`P`$<1X8?@A!`"19-@@^+\``6ML__]4I0P`$76B#)%_V"7XX?__
ML.(#%"HA4"N1!`"`TJ`.0/F^\O^70`4`-'6B#)'VDP"1H`Y`^5GR_Y?T`P`J
MX0,6J@```)``P!N1E//_EV`$`+1SH@R18`Y`^3SR_Y>?\O^7P```T`#`1_GB
M%T#Y`0!`^4(``>L!`(#2`04`5/U[0ZG@`Q0J\U-$J?5;1:G_@P&1P`-?UE'R
M_Y?3``#P8J(,D4`,`/E@`P"T&/+_ET'0.]7```#0`+A'^1\@`]4?(`/5(&A@
M^!\8)#FH__\7H`Y`^;;R_Y?5__\7H`I`^6```+1"^O^7V___%^$#%JH```"0
M`.`9D6GS_Y>@"@#YH/K_M#KZ_Y?3__\7A0"`4K[__Q<@`(!2T?'_ER3R_Y<`
M``````````````!P<U]S=')I;F=S(&UI<W-I;F<*```````E<R\E<P```"X`
M````````+W!R;V,O8W5R<')O8R]E>&4`````````1'EN84QO861E<CHZ8F]O
M=%]$>6YA3&]A9&5R``````!);G1E<FYA;',Z.E!!4CHZ0D]/5`````!005)?
M5$5-4```````````4$%27U!23T=.04U%`````#H`````````+P`````````E
M<R5S)7,``%!%4DPU3$E"``````````!015),3$E"`%!%4DPU3U!4````````
M``!015),24\``%!!4E])3DE424%,25I%1`!005)?4U!!5TY%1```````4$%2
M7T-,14%.`````````%!!4E]$14)51P````````!005)?0T%#2$4`````````
M4$%27T=,3T)!3%]$14)51P``````````4$%27T=,3T)!3%]435!$25(`````
M````4$%27U1-4$1)4@```````%!!4E]'3$]"04Q?5$5-4`!005)?1TQ/0D%,
M7T-,14%.```````````Q`````````&QS965K(&9A:6QE9`````!R96%D(&9A
M:6QE9```````3$1?3$E"4D%265]0051(`%-94U1%30``)3`R>`````!P87(M
M`````"5S)7,E<R5S```````````E<SH@8W)E871I;VX@;V8@<')I=F%T92!S
M=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*````````)7,Z('!R
M:79A=&4@<W5B9&ER96-T;W)Y("5S(&ES('5N<V%F92`H<&QE87-E(')E;6]V
M92!I="!A;F0@<F5T<GD@>6]U<B!O<&5R871I;VXI"@```````%!!5$@`````
M<&%R;``````N<&%R`````'-H;W)T(')E860````````E<R5S8V%C:&4M)7,E
M<P``)7,E<W1E;7`M)74E<P```"5S)7-T96UP+25U+25U)7,``````````!@`
M````````<&5R;``````P`````````%!!4CHZ4&%C:V5R.CI615)324].````
M`#$N,#8S````56YA8FQE('1O(&-R96%T92!C86-H92!D:7)E8W1O<GD`````
M`````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E(&-A8VAE('-U8F1I<F5C=&]R
M>2`E<R!F86EL960@*&5R<FYO/25I*0H``````````"UE````````+2T`````
M``!54T52`````%5315).04U%``````````!435!$25(``%1%35!$25(`5$5-
M4`````!435```````"]T;7`O````979A;"`G97AE8R`O=7-R+V)I;B]P97)L
M("`M4R`D,"`D>S$K(B1`(GTG"FEF(#`[(",@;F]T(')U;FYI;F<@=6YD97(@
M<V]M92!S:&5L;`IP86-K86=E(%]?<&%R7W!L.PIM>2`H)%!!4E]-04=)0RP@
M)$9)3$5?;V9F<V5T7W-I>F4L("1C86-H95]N86UE7W-I>F4I.PI"14=)3B![
M"B1005)?34%'24,@/2`B7&Y005(N<&U<;B(["B1&24Q%7V]F9G-E=%]S:7IE
M(#T@-#L@("`C('!A8VLH(DXB*0HD8V%C:&5?;F%M95]S:7IE(#T@-#`["B10
M2UI)4%]-04=)0R`](")02UPP,#-<,#`T(CL*?0IS=6(@9FEN9%]P87)?;6%G
M:6,*>PIM>2`H)&9H*2`]($!?.PIM>2`D8VAU;FM?<VEZ92`](#8T("H@,3`R
M-#L*;7D@)&)U9CL*;7D@)'-I>F4@/2`M<R`D9F@["FUY("1P;W,@/2`H)'-I
M>F4M,2D@+2`H)'-I>F4M,2D@)2`D8VAU;FM?<VEZ93L@("`@("`C($Y/5$4Z
M("1P;W,@:7,@82!M=6QT:7!L92!O9B`D8VAU;FM?<VEZ90IW:&EL92`H)'!O
M<R`^/2`P*2!["G-E96L@)&9H+"`D<&]S+"`P.PIR96%D("1F:"P@)&)U9BP@
M)&-H=6YK7W-I>F4@*R!L96YG=&@H)%!!4E]-04=)0RD["FEF("@H;7D@)&D@
M/2!R:6YD97@H)&)U9BP@)%!!4E]-04=)0RDI(#X](#`I('L*<F5T=7)N("1P
M;W,@*R`D:3L*?0HD<&]S("T]("1C:'5N:U]S:7IE.PI]"G)E='5R;B`M,3L*
M?0IM>2`H)'!A<E]T96UP+"`D<')O9VYA;64L($!T;7!F:6QE+"`E36]D=6QE
M0V%C:&4I.PI%3D0@>R!I9B`H)$5.5GM005)?0TQ%04Y]*2!["G)E<75I<F4@
M1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI"87-E;F%M93L*<F5Q=6ER92!&
M:6QE.CI3<&5C.PIM>2`D=&]P9&ER(#T@1FEL93HZ0F%S96YA;64Z.F1I<FYA
M;64H)'!A<E]T96UP*3L*;W5T<RAQ<5M296UO=FEN9R!F:6QE<R!I;B`B)'!A
M<E]T96UP(ETI.PI&:6QE.CI&:6YD.CIF:6YD9&5P=&@H<W5B('L@*"`M9"`I
M(#\@<FUD:7(@.B!U;FQI;FL@?2P@)'!A<E]T96UP*3L*<FUD:7(@)'!A<E]T
M96UP.PII9B`H+60@)'!A<E]T96UP("8F("1>3R!N92`G35-7:6XS,B<I('L*
M;7D@)'1M<"`](&YE=R!&:6QE.CI496UP*`I414U03$%412`]/B`G=&UP6%A8
M6%@G+`I$25(@/3X@1FEL93HZ0F%S96YA;64Z.F1I<FYA;64H)'1O<&1I<BDL
M"E-51D9)6"`]/B`G+F-M9"<L"E5.3$E.2R`]/B`P+`HI.PIM>2`D9FEL96YA
M;64@/2`D=&UP+3YF:6QE;F%M93L*<')I;G0@)'1M<"`\/"(N+BXB.PIX/3$[
M('=H:6QE(%L@7"1X("UL="`Q,"!=.R!D;PIR;2`M<F8@)R1P87)?=&5M<"<*
M:68@6R!<(2`M9"`G)'!A<E]T96UP)R!=.R!T:&5N"F)R96%K"F9I"G-L965P
M(#$*>#U@97AP<B!<)'@@*R`Q8`ID;VYE"G)M("<D9FEL96YA;64G"BXN+@IC
M;&]S92`D=&UP.PIC:&UO9"`P-S`P+"`D9FEL96YA;64["FUY("1C;60@/2`B
M)&9I;&5N86UE(#XO9&5V+VYU;&P@,CXF,2`F(CL*<WES=&5M*"1C;60I.PIO
M=71S*'%Q6U-P87=N960@8F%C:V=R;W5N9"!P<F]C97-S('1O('!E<F9O<FT@
M8VQE86YU<#H@)&9I;&5N86UE72D["GT*?2!]"D)%1TE.('L*26YT97)N86QS
M.CI005(Z.D)/3U0H*2!I9B!D969I;F5D("9);G1E<FYA;',Z.E!!4CHZ0D]/
M5#L*979A;"!["E]P87)?:6YI=%]E;G8H*3L*;7D@)'%U:65T(#T@(21%3E9[
M4$%27T1%0E5'?3L*;7D@)7-Y<R`]("@*<&%T:%]S97`@("`@/3X@*"1>3R`]
M?B`O7DU35VEN+R`_("<[)R`Z("<Z)RDL"E]E>&4@("`@("`@(#T^("@D7D\@
M/7X@+UXH/SI-4U=I;GQ/4S)\8WEG=VEN*2\@/R`G+F5X92<@.B`G)RDL"E]D
M96QI;2`@("`@(#T^("@D7D\@/7X@+UY-4U=I;GQ/4S(O(#\@)UQ<)R`Z("<O
M)RDL"BD["E]S971?<')O9VYA;64H*3L*;W5T<RAQ<5M<)'!R;V=N86UE(#T@
M(B1P<F]G;F%M92)=*3L*7W-E=%]P87)?=&5M<"@I.PIO=71S*'%Q6UPD14Y6
M>U!!4E]414U0?2`]("(D14Y6>U!!4E]414U0?2)=*3L*;7D@*"1S=&%R=%]P
M;W,L("1S=&%R=%]O9E]&24Q%7W-E8W1I;VXI.PI-04=)0SH@>PIL;V-A;"`D
M4TE'>U]?5T%23E]??2`]('-U8B![?3L*=6YL97-S("AO<&5N(%]&2"P@)SPZ
M<F%W)RP@)'!R;V=N86UE*2!["F]U=',H<7%;0V%N)W0@<F5A9"!F<F]M(&9I
M;&4@(B1P<F]G;F%M92)=*3L@(",@9&]N)W0@=7-E("0A(&AE<F4@87,@:70@
M<F5Q=6ER97,@17)R;F\N<&T*;&%S="!-04=)0SL*?0IM>2`D;6%G:6-?<&]S
M(#T@9FEN9%]P87)?;6%G:6,H*E]&2"D["FEF("@D;6%G:6-?<&]S(#P@,"D@
M>PIO=71S*'%Q6T-A;B=T(&9I;F0@;6%G:6,@<W1R:6YG("(D4$%27TU!1TE#
M(B!I;B!F:6QE("(D<')O9VYA;64B72D["FQA<W0@34%'24,["GT*;W5T<R@B
M1F]U;F0@4$%2(&UA9VEC(&%T('!O<VET:6]N("1M86=I8U]P;W,B*3L*<V5E
M:R!?1D@L("1M86=I8U]P;W,@+2`D1DE,15]O9F9S971?<VEZ92`M(&QE;F=T
M:"@B7#!#04-(12(I+"`P.PIR96%D(%]&2"P@)&)U9BP@;&5N9W1H*")<,$-!
M0TA%(BD["FEF("@D8G5F(&YE(")<,$-!0TA%(B`F)B`D8G5F(&YE(")<,$-,
M14%.(BD@>PIO=71S*").;R!C86-H92!M87)K97(@9F]U;F0B*3L*;&%S="!-
M04=)0SL*?0IO=71S*'%Q6T-A8VAE(&UA<FME<B`B)&)U9B(@9F]U;F1=*3L*
M<V5E:R!?1D@L("1M86=I8U]P;W,@+2`D1DE,15]O9F9S971?<VEZ92P@,#L*
M<F5A9"!?1D@L("1B=68L("1&24Q%7V]F9G-E=%]S:7IE.PIM>2`D;V9F<V5T
M(#T@=6YP86-K*").(BP@)&)U9BD["F]U=',H(D]F9G-E="!F<F]M('-T87)T
M(&]F($9)3$5S(&ES("1O9F9S970B*3L*<V5E:R!?1D@L("1M86=I8U]P;W,@
M+2`D1DE,15]O9F9S971?<VEZ92`M("1O9F9S970L(#`["B1S=&%R=%]O9E]&
M24Q%7W-E8W1I;VX@/2!T96QL(%]&2#L*;7D@)7)E<75I<F5?;&ES=#L*<F5A
M9"!?1D@L("1B=68L(#0[("`@("`@("`@("`@("`@("`@("`@("`@("`@(R!R
M96%D('1H92!F:7)S="`B1DE,12(*=VAI;&4@*"1B=68@97$@(D9)3$4B*2![
M"G)E860@7T9(+"`D8G5F+"`T.PIR96%D(%]&2"P@)&)U9BP@=6YP86-K*").
M(BP@)&)U9BD["FUY("1F=6QL;F%M92`]("1B=68["F]U=',H<7%;56YP86-K
M:6YG($9)3$4@(B1F=6QL;F%M92(N+BY=*3L*;7D@)&-R8R`]("@@)&9U;&QN
M86UE(#U^('-\7BA;82UF7&1=>SA]*2]\?"`I(#\@)#$@.B!U;F1E9CL*;7D@
M*"1B87-E;F%M92P@)&5X="D@/2`H)&)U9B`]?B!M?"@_.BXJ+RD_*"XJ*2A<
M+BXJ*7PI.PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A9"!?1D@L("1B=68L('5N
M<&%C:R@B3B(L("1B=68I.PII9B`H9&5F:6YE9"@D97AT*2!A;F0@)&5X="`A
M?B`O7"XH/SIP;7QP;'QI>'QA;"DD+VDI('L*;7D@)&9I;&5N86UE(#T@7W-A
M=F5?87,H(B1C<F,D97AT(BP@)&)U9BP@,#<U-2D["B1005(Z.DAE879Y.CI&
M=6QL0V%C:&5[)&9U;&QN86UE?2`]("1F:6QE;F%M93L*)%!!4CHZ2&5A=GDZ
M.D9U;&Q#86-H97LD9FEL96YA;65](#T@)&9U;&QN86UE.PI]"F5L<V4@>PHD
M36]D=6QE0V%C:&5[)&9U;&QN86UE?2`]('L*8G5F(#T^("1B=68L"F-R8R`]
M/B`D8W)C+`IN86UE(#T^("1F=6QL;F%M92P*?3L*)')E<75I<F5?;&ES='LD
M9G5L;&YA;65]*RL["GT*<F5A9"!?1D@L("1B=68L(#0["GT*;&]C86P@0$E.
M0R`]("AS=6(@>PIM>2`H)'-E;&8L("1M;V1U;&4I(#T@0%\["G)E='5R;B!I
M9B!R968@)&UO9'5L92!O<B`A)&UO9'5L93L*;7D@)&EN9F\@/2`D36]D=6QE
M0V%C:&5[)&UO9'5L97T["G)E='5R;B!U;FQE<W,@)&EN9F\["F1E;&5T92`D
M<F5Q=6ER95]L:7-T>R1M;V1U;&5].PHD24Y#>R1M;V1U;&5](#T@(B]L;V%D
M97(O)&EN9F\O)&UO9'5L92(["FEF("@D14Y6>U!!4E]#3$5!3GT@86YD(&1E
M9FEN960H)DE/.CI&:6QE.CIN97<I*2!["FUY("1F:"`]($E/.CI&:6QE+3YN
M97=?=&UP9FEL92!O<B!D:64@(D-A;B=T(&-R96%T92!T96UP(&9I;&4Z("0A
M(CL*)&9H+3YB:6YM;V1E*"D["B1F:"T^<')I;G0H)&EN9F\M/GMB=69]*3L*
M)&9H+3YS965K*#`L(#`I.PIR971U<FX@)&9H.PI]"F5L<V4@>PIM>2`D9FEL
M96YA;64@/2!?<V%V95]A<R@B)&EN9F\M/GMC<F-]+G!M(BP@)&EN9F\M/GMB
M=69]*3L*)&EN9F\M/GMF:6QE?2`]("1F:6QE;F%M93L*;W!E;B!M>2`D9F@L
M("<\.G)A=R<L("1F:6QE;F%M92!O<B!D:64@<7%;0V%N)W0@<F5A9"`B)&9I
M;&5N86UE(CH@)"%=.PIR971U<FX@)&9H.PI]"F1I92`B0F]O='-T<F%P<&EN
M9R!F86EL960Z(&-A;B=T(&9I;F0@;6]D=6QE("1M;V1U;&4A(CL*?2P@0$E.
M0RD["G)E<75I<F4@6%-,;V%D97(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q
M=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%>'!O<G1E<CHZ2&5A=GD["E!!
M4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*<F5Q=6ER92!)3SHZ1FEL
M93L*=VAI;&4@*&UY("1F:6QE;F%M92`]("AS;W)T(&ME>7,@)7)E<75I<F5?
M;&ES="E;,%TI('L*=6YL97-S("@D24Y#>R1F:6QE;F%M97T@;W(@)&9I;&5N
M86UE(#U^("]"4T1004XO*2!["FEF("@D9FEL96YA;64@/7X@+UPN<&UC/R0O
M:2D@>PIR97%U:7)E("1F:6QE;F%M93L*?0IE;'-E('L*9&\@)&9I;&5N86UE
M('5N;&5S<R`D9FEL96YA;64@/7X@+W-I=&5C=7-T;VUI>F5<+G!L)"\["GT*
M?0ID96QE=&4@)')E<75I<F5?;&ES='LD9FEL96YA;65].PI]"FEF("@D8G5F
M(&YE("102UI)4%]-04=)0RD@>PIO=71S*'%Q6TYO('II<"!F;W5N9"!A9G1E
M<B!&24Q%('-E8W1I;VX@:6X@9FEL92`B)'!R;V=N86UE(ETI.PIL87-T($U!
M1TE#(#L*?0HD<W1A<G1?<&]S(#T@*'1E;&P@7T9(*2`M(#0[("`@("`@("`@
M("`@("`@(",@<W1A<G0@;V8@>FEP"GT*;7D@0'!A<E]A<F=S.PIM>2`H)&]U
M="P@)&)U;F1L92P@)&QO9V9H+"`D8V%C:&5?;F%M92D["F1E;&5T92`D14Y6
M>U!!4E]!4%!?4D554T5].R`C('-A;FET:7IE("A2155312!M87D@8F4@82!S
M96-U<FET>2!P<F]B;&5M*0II9B`H(21S=&%R=%]P;W,@;W(@*"1!4D=66S!=
M(&5Q("<M+7!A<BUO<'1I;VYS)R`F)B!S:&EF="DI('L*;7D@)61I<W1?8VUD
M(#T@<7<H"G`@("!B;&EB7W1O7W!A<@II("`@:6YS=&%L;%]P87(*=2`@('5N
M:6YS=&%L;%]P87(*<R`@('-I9VY?<&%R"G8@("!V97)I9GE?<&%R"BD["FEF
M("A`05)'5B!A;F0@)$%21U9;,%T@97$@)RTM<F5U<V4G*2!["G-H:69T($!!
M4D=6.PHD14Y6>U!!4E]!4%!?4D554T5](#T@<VAI9G0@0$%21U8["GT*96QS
M92![(",@;F]R;6%L('!A<FP@8F5H879I;W5R"FUY($!A9&1?=&]?:6YC.PIW
M:&EL92`H0$%21U8I('L*)$%21U9;,%T@/7X@+UXM*%M!24U/0DQB<7!I=7-4
M=ETI*"XJ*2\@;W(@;&%S=#L*:68@*"0Q(&5Q("=))RD@>PIP=7-H($!A9&1?
M=&]?:6YC+"`D,CL*?0IE;'-I9B`H)#$@97$@)TTG*2!["F5V86P@(G5S92`D
M,B(["GT*96QS:68@*"0Q(&5Q("=!)RD@>PIU;G-H:69T($!P87)?87)G<RP@
M)#(["GT*96QS:68@*"0Q(&5Q("=/)RD@>PHD;W5T(#T@)#(["GT*96QS:68@
M*"0Q(&5Q("=B)RD@>PHD8G5N9&QE(#T@)W-I=&4G.PI]"F5L<VEF("@D,2!E
M<2`G0B<I('L*)&)U;F1L92`]("=A;&PG.PI]"F5L<VEF("@D,2!E<2`G<2<I
M('L*)'%U:65T(#T@,3L*?0IE;'-I9B`H)#$@97$@)TPG*2!["F]P96X@)&QO
M9V9H+"`B/CXB+"`D,B!O<B!D:64@<7%;0V%N)W0@;W!E;B!L;V<@9FEL92`B
M)#(B.B`D(5T["GT*96QS:68@*"0Q(&5Q("=4)RD@>PHD8V%C:&5?;F%M92`]
M("0R.PI]"G-H:69T*$!!4D=6*3L*:68@*&UY("1C;60@/2`D9&ES=%]C;61[
M)#%]*2!["F1E;&5T92`D14Y6>R=005)?5$5-4"=].PII;FET7VEN8R@I.PIR
M97%U:7)E(%!!4CHZ1&ES=#L*)GLB4$%2.CI$:7-T.CHD8VUD(GTH*2!U;FQE
M<W,@0$%21U8["B9[(E!!4CHZ1&ES=#HZ)&-M9")]*"1?*2!F;W(@0$%21U8[
M"F5X:70["GT*?0IU;G-H:69T($!)3D,L($!A9&1?=&]?:6YC.PI]"GT*:68@
M*"1O=70I('L*>PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($%R8VAI=F4Z
M.EII<#L*<F5Q=6ER92!$:6=E<W0Z.E-(03L*?0IM>2`D<&%R(#T@<VAI9G0H
M0$%21U8I.PIM>2`D>FEP.PII9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@
M)&9H+"`G/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#86XG="!F:6YD('!A<B!F
M:6QE("(D<&%R(CH@)"%=.PIB;&5S<R@D9F@L("=)3SHZ1FEL92<I.PHD>FEP
M(#T@07)C:&EV93HZ6FEP+3YN97<["B@@)'II<"T^<F5A9$9R;VU&:6QE2&%N
M9&QE*"1F:"P@)'!A<BD@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I("D*;W(@
M9&EE('%Q6T5R<F]R(')E861I;F<@>FEP(&%R8VAI=F4@(B1P87(B73L*?0IM
M>2`E;65T85]P87(@/2!D;R!["FEF("@D>FEP(&%N9"!M>2`D;65T82`]("1Z
M:7`M/F-O;G1E;G1S*"=-151!+GEM;"<I*2!["B1M971A(#U^(',O+BI><&%R
M.B0O+VUS.PHD;65T82`]?B!S+UY<4RXJ+R]M<SL*)&UE=&$@/7X@+UX@("A;
M7CI=*RDZ("@N*RDD+VUG.PI]"GT["FEF("AD969I;F5D("1P87(I('L*;W!E
M;B!M>2`D<&@L("<\.G)A=R<L("1P87(@;W(@9&EE('%Q6T-A;B=T(')E860@
M<&%R(&9I;&4@(B1P87(B.B`D(5T["FUY("1B=68["G)E860@)'!H+"`D8G5F
M+"`T.PID:64@<7%;(B1P87(B(&ES(&YO="!A('!A<B!F:6QE72!U;FQE<W,@
M)&)U9B!E<2`D4$M:25!?34%'24,["F-L;W-E("1P:#L*?0I#<F5A=&50871H
M*"1O=70I(#L*;7D@)&9H(#T@24\Z.D9I;&4M/FYE=R@*)&]U="P*24\Z.D9I
M;&4Z.D]?0U)%050H*2!\($E/.CI&:6QE.CI/7U)$5U(H*2!\($E/.CI&:6QE
M.CI/7U1254Y#*"DL"C`W-S<L"BD@;W(@9&EE('%Q6T-A;B=T(&-R96%T92!F
M:6QE("(D;W5T(CH@)"%=.PHD9F@M/F)I;FUO9&4H*3L*<V5E:R!?1D@L(#`L
M(#`["FUY("1L;V%D97(["FEF("AD969I;F5D("1S=&%R=%]O9E]&24Q%7W-E
M8W1I;VXI('L*<F5A9"!?1D@L("1L;V%D97(L("1S=&%R=%]O9E]&24Q%7W-E
M8W1I;VX["GT@96QS92!["FQO8V%L("0O(#T@=6YD968["B1L;V%D97(@/2`\
M7T9(/CL*?0II9B`H(21%3E9[4$%27U9%4D)!5$E-?2!A;F0@)&QO861E<B`]
M?B`O7B@_.B,A?%Q`<F5M*2\I('L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&]D
M4W1R:7`["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1L;V%D97(L
M("0P*3L*?0HD9F@M/G!R:6YT*"1L;V%D97(I"F]R(&1I92!Q<5M%<G)O<B!W
M<FET:6YG(&QO861E<B!T;R`B)&]U="(Z("0A73L*:68@*"1B=6YD;&4I('L*
M<F5Q=6ER92!005(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A
M9&5R*"D["FEN:71?:6YC*"D["G)E<75I<F5?;6]D=6QE<R@I.PIM>2!`:6YC
M(#T@9W)E<"![("$O0E-$4$%.+R!]"F=R97`@>PHH)&)U;F1L92!N92`G<VET
M92<I(&]R"B@D7R!N92`D0V]N9FEG.CI#;VYF:6=[87)C:&QI8F5X<'T@86YD
M"B1?(&YE("1#;VYF:6<Z.D-O;F9I9WMP<FEV;&EB97AP?2D["GT@0$E.0SL*
M<WPO*WPO?&<L('-\+R1\?"!F;W)E86-H($!I;F,["FUY("5F:6QE<SL*)&9I
M;&5S>R1??2LK(&9O<B!`1'EN84QO861E<CHZ9&Q?<VAA<F5D7V]B:F5C=',L
M('9A;'5E<R`E24Y#.PIM>2`D;&EB7V5X="`]("1#;VYF:6<Z.D-O;F9I9WML
M:6)?97AT?3L@("`@("`@("`C(%A86"!L:6)?97AT('9S(&1L97AT(#\*;7D@
M)7=R:71T96X["F9O<F5A8V@@;7D@)&ME>2`H<V]R="!K97ES("5F:6QE<RD@
M>PIM>2`H)&9I;&4L("1N86UE*3L*:68@*&1E9FEN960H;7D@)&9C(#T@)%!!
M4CHZ2&5A=GDZ.D9U;&Q#86-H97LD:V5Y?2DI('L**"1F:6QE+"`D;F%M92D@
M/2`H)&ME>2P@)&9C*3L*?0IE;'-E('L*9F]R96%C:"!M>2`D9&ER("A`:6YC
M*2!["FEF("@D:V5Y(#U^(&U\7EQ1)&1I<EQ%+R@N*BDD?&DI('L**"1F:6QE
M+"`D;F%M92D@/2`H)&ME>2P@)#$I.PIL87-T.PI]"FEF("@D:V5Y(#U^(&U\
M7B]L;V%D97(O6UXO72LO*"XJ*21\*2!["FEF("AM>2`D<F5F(#T@)$UO9'5L
M94-A8VAE>R0Q?2D@>PHH)&9I;&4L("1N86UE*2`]("@D<F5F+"`D,2D["FQA
M<W0["GT*:68@*"UF("(D9&ER+R0Q(BD@>PHH)&9I;&4L("1N86UE*2`]("@B
M)&1I<B\D,2(L("0Q*3L*;&%S=#L*?0I]"GT*?0IN97AT('5N;&5S<R!D969I
M;F5D("1N86UE.PIN97AT(&EF("1W<FET=&5N>R1N86UE?2LK.PIN97AT(&EF
M("%R968H)&9I;&4I(&%N9"`D9FEL92`]?B`O7"Y<421L:6)?97AT7$4D+VD[
M"F]U=',H<W!R:6YT9BAQ<5M086-K:6YG($9)3$4@(B5S(BXN+ETL(')E9B`D
M9FEL92`_("1F:6QE+3Y[;F%M97T@.B`D9FEL92DI.PIM>2`D8V]N=&5N=#L*
M:68@*')E9B@D9FEL92DI('L*)&-O;G1E;G0@/2`D9FEL92T^>V)U9GT["GT*
M96QS92!["FQO8V%L("0O(#T@=6YD968["F]P96X@;7D@)'1H+"`G/#IR87<G
M+"`D9FEL92!O<B!D:64@<7%;0V%N)W0@<F5A9"`B)&9I;&4B.B`D(5T["B1C
M;VYT96YT(#T@/"1T:#X["F-L;W-E("1T:#L*4$%2.CI&:6QT97(Z.E!O9%-T
M<FEP+3YA<'!L>2A<)&-O;G1E;G0L("(\96UB961D960^+R1N86UE(BD*:68@
M(21%3E9[4$%27U9%4D)!5$E-?2!A;F0@)&YA;64@/7X@+UPN*#\Z<&U\:7A\
M86PI)"]I.PI005(Z.D9I;'1E<CHZ4&%T8VA#;VYT96YT+3YN97<M/F%P<&QY
M*%PD8V]N=&5N="P@)&9I;&4L("1N86UE*3L*?0HD9F@M/G!R:6YT*")&24Q%
M(BP*<&%C:R@G3B<L(&QE;F=T:"@D;F%M92D@*R`Y*2P*<W!R:6YT9B@B)3`X
M>"\E<R(L($%R8VAI=F4Z.EII<#HZ8V]M<'5T94-20S,R*"1C;VYT96YT*2P@
M)&YA;64I+`IP86-K*"=.)RP@;&5N9W1H*"1C;VYT96YT*2DL"B1C;VYT96YT
M*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!E;6)E9&1E9"!&24Q%('1O("(D
M;W5T(CH@)"%=.PIO=71S*'%Q6U=R:71T96X@87,@(B1N86UE(ETI.PI]"GT*
M:68@*"1Z:7`I('L*)'II<"T^=W)I=&54;T9I;&5(86YD;&4H)&9H*2`]/2!!
M<F-H:79E.CI::7`Z.D%:7T]+"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG('II
M<"!P87)T(&]F("(D;W5T(ET["GT*:68@*"1M971A7W!A<GMC;&5A;GTI('L*
M)&9H+3YP<FEN="@B7#!#3$5!3B(I.PI]"F5L<V4@>PII9B`H(61E9FEN960@
M)&-A8VAE7VYA;64I('L*;7D@)&-T>"`]($1I9V5S=#HZ4TA!+3YN97<H,2D[
M"G-E96L@)&9H+"`P+"`P.PHD8W1X+3YA9&1F:6QE*"1F:"D["G-E96L@)&9H
M+"`P+"`R.PHD8V%C:&5?;F%M92`]("1C='@M/FAE>&1I9V5S=#L*?0HD8V%C
M:&5?;F%M92`]('-U8G-T<B`D8V%C:&5?;F%M92P@,"P@)&-A8VAE7VYA;65?
M<VEZ93L*)&-A8VAE7VYA;64@+CT@(EPP(B!X("@D8V%C:&5?;F%M95]S:7IE
M("T@;&5N9W1H("1C86-H95]N86UE*3L*)&9H+3YP<FEN="@D8V%C:&5?;F%M
M92P*(EPP0T%#2$4B*3L*?0IM>2`D;V9F<V5T(#T@)&9H+3YT96QL("T@;&5N
M9W1H*"1L;V%D97(I.PHD9F@M/G!R:6YT*'!A8VLH)TXG+"`D;V9F<V5T*2P*
M)%!!4E]-04=)0RD["B1F:"T^8VQO<V4@;W(@9&EE('%Q6T5R<F]R('=R:71I
M;F<@=')A:6QE<B!O9B`B)&]U="(Z("0A73L*8VAM;V0@,#<U-2P@)&]U=#L*
M97AI=#L*?0I["FQA<W0@=6YL97-S(&1E9FEN960@)'-T87)T7W!O<SL*7V9I
M>%]P<F]G;F%M92@I.PIR97%U:7)E(%!!4CL*4$%2.CI(96%V>3HZ7VEN:71?
M9'EN86QO861E<B@I.PI["G)E<75I<F4@1FEL93HZ1FEN9#L*<F5Q=6ER92!!
M<F-H:79E.CI::7`["GT*;7D@)&9H(#T@24\Z.D9I;&4M/FYE=SL@("`@("`@
M("`@("`@("`@("`@("`@("`@("`@(",@07)C:&EV93HZ6FEP(&]P97)A=&5S
M(&]N(&%N($E/.CI(86YD;&4*)&9H+3YF9&]P96XH9FEL96YO*%]&2"DL("=R
M)RD@;W(@9&EE('%Q6V9D;W!E;B@I(&9A:6QE9#H@)"%=.PI!<F-H:79E.CI:
M:7`Z.G-E=$-H=6YK4VEZ92@M<R!?1D@I.PIM>2`D>FEP(#T@07)C:&EV93HZ
M6FEP+3YN97<["B@D>FEP+3YR96%D1G)O;49I;&5(86YD;&4H)&9H+"`D<')O
M9VYA;64I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH*2D*;W(@9&EE('%Q6T5R
M<F]R(')E861I;F<@>FEP(&%R8VAI=F4@(B1P<F]G;F%M92)=.PI!<F-H:79E
M.CI::7`Z.G-E=$-H=6YK4VEZ92@V-"`J(#$P,C0I.PIP=7-H($!005(Z.DQI
M8D-A8VAE+"`D>FEP.PHD4$%2.CI,:6)#86-H97LD<')O9VYA;65](#T@)'II
M<#L*;W5T<R@B17AT<F%C=&EN9R!Z:7`N+BXB*3L*:68@*&1E9FEN960@)$5.
M5GM005)?5$5-4'TI('L@(R!S:&]U;&0@8F4@<V5T(&%T('1H:7,@<&]I;G0A
M"F9O<F5A8V@@;7D@)&UE;6)E<B`H("1Z:7`M/FUE;6)E<G,@*2!["FYE>'0@
M:68@)&UE;6)E<BT^:7-$:7)E8W1O<GD["FUY("1M96UB97)?;F%M92`]("1M
M96UB97(M/F9I;&5.86UE.PIN97AT('5N;&5S<R`D;65M8F5R7VYA;64@/7X@
M;7L*7@HO/W-H;&EB+PHH/SHD0V]N9FEG.CI#;VYF:6=[87)C:&YA;65]+RD_
M"BA;7B]=*RD*)`I]>#L*;7D@)&5X=')A8W1?;F%M92`]("0Q.PIM>2`D9&5S
M=%]N86UE(#T@1FEL93HZ4W!E8RT^8V%T9FEL92@D14Y6>U!!4E]414U0?2P@
M)&5X=')A8W1?;F%M92D["FEF("@M9B`D9&5S=%]N86UE("8F("US(%\@/3T@
M)&UE;6)E<BT^=6YC;VUP<F5S<V5D4VEZ92@I*2!["F]U=',H<7%;4VMI<'!I
M;F<@(B1M96UB97)?;F%M92(@<VEN8V4@:70@86QR96%D>2!E>&ES=',@870@
M(B1D97-T7VYA;64B72D["GT@96QS92!["F]U=',H<7%;17AT<F%C=&EN9R`B
M)&UE;6)E<E]N86UE(B!T;R`B)&1E<W1?;F%M92(N+BY=*3L**"1M96UB97(M
M/F5X=')A8W14;T9I;&5.86UE9"@D9&5S=%]N86UE*2`]/2!!<F-H:79E.CI:
M:7`Z.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!E>'1R86-T:6YG('II<"!M
M96UB97(@=&\@(B1D97-T7VYA;64B73L*8VAM;V0H,#4U-2P@)&1E<W1?;F%M
M92D@:68@)%Y/(&5Q(")H<'5X(CL*?0I]"GT*?0IU;FQE<W,@*"1005(Z.DQI
M8D-A8VAE>R1P<F]G;F%M97TI('L*9&EE(#P\("(N(B!U;FQE<W,@0$%21U8[
M"E5S86=E.B`D,"!;("U!;&EB+G!A<B!=(%L@+4ED:7(@72!;("U-;6]D=6QE
M(%T@6R!S<F,N<&%R(%T@6R!P<F]G<F%M+G!L(%T*)#`@6R`M0GPM8B!=(%LM
M3V]U=&9I;&5=('-R8RYP87(*+@HD14Y6>U!!4E]04D]'3D%-17T@/2`D<')O
M9VYA;64@/2`D,"`]('-H:69T*$!!4D=6*3L*?0IS=6(@0W)E871E4&%T:"![
M"FUY("@D;F%M92D@/2!`7SL*<F5Q=6ER92!&:6QE.CI"87-E;F%M93L*;7D@
M*"1B87-E;F%M92P@)'!A=&@L("1E>'0I(#T@1FEL93HZ0F%S96YA;64Z.F9I
M;&5P87)S92@D;F%M92P@*"=<+BXJ)RDI.PIR97%U:7)E($9I;&4Z.E!A=&@[
M"D9I;&4Z.E!A=&@Z.FUK<&%T:"@D<&%T:"D@=6YL97-S*"UE("1P871H*3L@
M(R!M:W!A=&@@9&EE<R!W:71H(&5R<F]R"GT*<W5B(')E<75I<F5?;6]D=6QE
M<R!["G)E<75I<F4@;&EB.PIR97%U:7)E($1Y;F%,;V%D97(["G)E<75I<F4@
M:6YT96=E<CL*<F5Q=6ER92!S=')I8W0["G)E<75I<F4@=V%R;FEN9W,["G)E
M<75I<F4@=F%R<SL*<F5Q=6ER92!#87)P.PIR97%U:7)E($-A<G`Z.DAE879Y
M.PIR97%U:7)E($5R<FYO.PIR97%U:7)E($5X<&]R=&5R.CI(96%V>3L*<F5Q
M=6ER92!%>'!O<G1E<CL*<F5Q=6ER92!&8VYT;#L*<F5Q=6ER92!&:6QE.CI4
M96UP.PIR97%U:7)E($9I;&4Z.E-P96,["G)E<75I<F4@6%-,;V%D97(["G)E
M<75I<F4@0V]N9FEG.PIR97%U:7)E($E/.CI(86YD;&4["G)E<75I<F4@24\Z
M.D9I;&4["G)E<75I<F4@0V]M<')E<W,Z.EIL:6(["G)E<75I<F4@07)C:&EV
M93HZ6FEP.PIR97%U:7)E($1I9V5S=#HZ4TA!.PIR97%U:7)E(%!!4CL*<F5Q
M=6ER92!005(Z.DAE879Y.PIR97%U:7)E(%!!4CHZ1&ES=#L*<F5Q=6ER92!0
M05(Z.D9I;'1E<CHZ4&]D4W1R:7`["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!A
M=&-H0V]N=&5N=#L*<F5Q=6ER92!A='1R:6)U=&5S.PIE=F%L('L@<F5Q=6ER
M92!#=V0@?3L*979A;"![(')E<75I<F4@5VEN,S(@?3L*979A;"![(')E<75I
M<F4@4V-A;&%R.CI5=&EL('T["F5V86P@>R!R97%U:7)E($%R8VAI=F4Z.E5N
M>FEP.CI"=7)S="!].PIE=F%L('L@<F5Q=6ER92!4:64Z.DAA<V@Z.DYA;65D
M0V%P='5R92!].PIE=F%L('L@<F5Q=6ER92!097)L24\[(')E<75I<F4@4&5R
M;$E/.CIS8V%L87(@?3L*979A;"![(')E<75I<F4@=71F."!].PI]"G-U8B!?
M<V5T7W!A<E]T96UP('L*:68@*&1E9FEN960@)$5.5GM005)?5$5-4'T@86YD
M("1%3E9[4$%27U1%35!](#U^("\H+BLI+RD@>PHD<&%R7W1E;7`@/2`D,3L*
M<F5T=7)N.PI]"F9O<F5A8V@@;7D@)'!A=&@@*`HH;6%P("1%3E9[)%]]+"!Q
M=R@@4$%27U1-4$1)4B!435!$25(@5$5-4$1)4B!414U0(%1-4"`I*2P*<7<H
M($,Z7%Q414U0("]T;7`@+B`I"BD@>PIN97AT('5N;&5S<R!D969I;F5D("1P
M871H(&%N9"`M9"`D<&%T:"!A;F0@+7<@)'!A=&@["FUY("1U<V5R;F%M93L*
M;7D@)'!W=6ED.PIE=F%L('LH)'!W=6ED*2`](&=E='!W=6ED*"0^*2!I9B!D
M969I;F5D("0^.WT["FEF("@@9&5F:6YE9"@F5VEN,S(Z.DQO9VEN3F%M92D@
M*2!["B1U<V5R;F%M92`]("97:6XS,CHZ3&]G:6Y.86UE.PI]"F5L<VEF("AD
M969I;F5D("1P=W5I9"D@>PHD=7-E<FYA;64@/2`D<'=U:60["GT*96QS92![
M"B1U<V5R;F%M92`]("1%3E9[55-%4DY!345]('Q\("1%3E9[55-%4GT@?'P@
M)U-94U1%32<["GT*;7D@)'-T;7!D:7(@/2`B)'!A=&@D<WES>U]D96QI;7UP
M87(M(BYU;G!A8VLH(D@J(BP@)'5S97)N86UE*3L*;6MD:7(@)'-T;7!D:7(L
M(#`W-34["FUY("1C86-H95]D:7(["FEF("@D14Y6>U!!4E]#3$5!3GTI('L*
M)&-A8VAE7V1I<B`](")T96UP+20D(CL*?0IE;'-E('L*;W!E;B!M>2`D9F@L
M("(\.G)A=R(L("1P<F]G;F%M92!O<B!D:64@<7%;0V%N)W0@<F5A9"`B)'!R
M;V=N86UE(CH@)"%=.PII9B`H*&UY("1M86=I8U]P;W,@/2!F:6YD7W!A<E]M
M86=I8R@D9F@I*2`^/2`P*2!["G-E96L@)&9H+"`D;6%G:6-?<&]S"BT@)$9)
M3$5?;V9F<V5T7W-I>F4*+2!L96YG=&@H(EPP0T%#2$4B*2P@,#L*;7D@)&)U
M9CL*<F5A9"`D9F@L("1B=68L(&QE;F=T:"@B7#!#04-(12(I.PII9B`H)&)U
M9B!E<2`B7#!#04-(12(I('L*<V5E:R`D9F@L("1M86=I8U]P;W,*+2`D1DE,
M15]O9F9S971?<VEZ90HM(&QE;F=T:"@B7#!#04-(12(I"BT@)&-A8VAE7VYA
M;65?<VEZ92P@,#L*<F5A9"`D9F@L("1B=68L("1C86-H95]N86UE7W-I>F4[
M"B1B=68@/7X@<R]<,"\O9SL*)&-A8VAE7V1I<B`](")C86-H92TD8G5F(CL*
M?0I]"F-L;W-E("1F:#L*?0II9B`H(21C86-H95]D:7(I('L*)&-A8VAE7V1I
M<B`](")T96UP+20D(CL*)$5.5GM005)?0TQ%04Y](#T@,3L*?0HD<W1M<&1I
M<B`N/2`B)'-Y<WM?9&5L:6U])&-A8VAE7V1I<B(["FUK9&ER("1S=&UP9&ER
M+"`P-S4U.PHD14Y6>U!!4E]414U0?2`]("1S=&UP9&ER.PIL87-T.PI]"B1P
M87)?=&5M<"`]("0Q(&EF("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]4
M14U0?2`]?B`O*"XK*2\["GT*<W5B(%]S879E7V%S('L*;7D@*"1N86UE+"`D
M8V]N=&5N=',L("1M;V1E*2`]($!?.PIM>2`D9G5L;&YA;64@/2`B)'!A<E]T
M96UP+R1N86UE(CL*=6YL97-S("@M92`D9G5L;&YA;64I('L*;7D@)'1E;7!N
M86UE(#T@(B1F=6QL;F%M92XD)"(["FUY("1F:#L**&]P96X@)&9H+"`G/CIR
M87<G+"`D=&5M<&YA;64I"B8F("AP<FEN="`D9F@@)&-O;G1E;G1S*0HF)B`H
M8VQO<V4@)&9H*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R`B)'1E;7!N86UE
M(CH@)"%=.PIC:&UO9"`D;6]D92P@)'1E;7!N86UE(&EF(&1E9FEN960@)&UO
M9&4["G)E;F%M92@D=&5M<&YA;64L("1F=6QL;F%M92D@;W(@=6YL:6YK*"1T
M96UP;F%M92D["GT*<F5T=7)N("1F=6QL;F%M93L*?0IS=6(@7W-E=%]P<F]G
M;F%M92!["FEF("AD969I;F5D("1%3E9[4$%27U!23T=.04U%?2!A;F0@)$5.
M5GM005)?4%)/1TY!345](#U^("\H+BLI+RD@>PHD<')O9VYA;64@/2`D,3L*
M?0HD<')O9VYA;64@?'P]("0P.PII9B`H)$5.5GM005)?5$5-4'T@86YD(&EN
M9&5X*"1P<F]G;F%M92P@)$5.5GM005)?5$5-4'TI(#X](#`I('L*)'!R;V=N
M86UE(#T@<W5B<W1R*"1P<F]G;F%M92P@<FEN9&5X*"1P<F]G;F%M92P@)'-Y
M<WM?9&5L:6U]*2`K(#$I.PI]"FEF("@A)$5.5GM005)?4%)/1TY!345](&]R
M(&EN9&5X*"1P<F]G;F%M92P@)'-Y<WM?9&5L:6U]*2`^/2`P*2!["FEF("AO
M<&5N(&UY("1F:"P@)SPG+"`D<')O9VYA;64I('L*<F5T=7)N(&EF("US("1F
M:#L*?0II9B`H+7,@(B1P<F]G;F%M921S>7-[7V5X97TB*2!["B1P<F]G;F%M
M92`N/2`D<WES>U]E>&5].PIR971U<FX["GT*?0IF;W)E86-H(&UY("1D:7(@
M*'-P;&ET("]<421S>7-[<&%T:%]S97!]7$4O+"`D14Y6>U!!5$A]*2!["FYE
M>'0@:68@97AI<W1S("1%3E9[4$%27U1%35!](&%N9"`D9&ER(&5Q("1%3E9[
M4$%27U1%35!].PHD9&ER(#U^(',O7%$D<WES>U]D96QI;7U<120O+SL**"@D
M<')O9VYA;64@/2`B)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M921S>7-[7V5X
M97TB*2P@;&%S="D*:68@+7,@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64D
M<WES>U]E>&5](CL**"@D<')O9VYA;64@/2`B)&1I<B1S>7-[7V1E;&EM?21P
M<F]G;F%M92(I+"!L87-T*0II9B`M<R`B)&1I<B1S>7-[7V1E;&EM?21P<F]G
M;F%M92(["GT*?0IS=6(@7V9I>%]P<F]G;F%M92!["B0P(#T@)'!R;V=N86UE
M('Q\/2`D14Y6>U!!4E]04D]'3D%-17T["FEF("AI;F1E>"@D<')O9VYA;64L
M("1S>7-[7V1E;&EM?2D@/"`P*2!["B1P<F]G;F%M92`]("(N)'-Y<WM?9&5L
M:6U])'!R;V=N86UE(CL*?0IM>2`D<'=D(#T@*&1E9FEN960@)D-W9#HZ9V5T
M8W=D*2`_($-W9#HZ9V5T8W=D*"D*.B`H*&1E9FEN960@)E=I;C,R.CI'971#
M=V0I(#\@5VEN,S(Z.D=E=$-W9"@I(#H@8'!W9&`I.PIC:&]M<"@D<'=D*3L*
M)'!R;V=N86UE(#U^(',O7B@_/5PN7"X_7%$D<WES>U]D96QI;7U<12DO)'!W
M9"1S>7-[7V1E;&EM?2\["B1%3E9[4$%27U!23T=.04U%?2`]("1P<F]G;F%M
M93L*?0IS=6(@7W!A<E]I;FET7V5N=B!["FEF("@@)$5.5GM005)?24Y)5$E!
M3$E:141]*RL@/3T@,2`I('L*<F5T=7)N.PI](&5L<V4@>PHD14Y6>U!!4E])
M3DE424%,25I%1'T@/2`R.PI]"F9O<B`H<7<H(%-005=.140@5$5-4"!#3$5!
M3B!$14)51R!#04-(12!04D]'3D%-12`I("D@>PID96QE=&4@)$5.5GLB4$%2
M7R1?(GT["GT*9F]R("AQ=R\@5$U01$E2(%1%35`@0TQ%04X@1$5"54<@+RD@
M>PHD14Y6>R)005)?)%\B?2`]("1%3E9[(E!!4E]'3$]"04Q?)%\B?2!I9B!E
M>&ES=',@)$5.5GLB4$%27T=,3T)!3%\D7R)].PI]"FEF("@D14Y6>U!!4E]4
M14U0?2D@>PID96QE=&4@)$5.5GM005)?0TQ%04Y].PI]"GT*<W5B(&]U=',@
M>PIR971U<FX@:68@)'%U:65T.PII9B`H)&QO9V9H*2!["G!R:6YT("1L;V=F
M:"`B0%]<;B(["GT*96QS92!["G!R:6YT(")`7UQN(CL*?0I]"G-U8B!I;FET
M7VEN8R!["G)E<75I<F4@0V]N9FEG.PIP=7-H($!)3D,L(&=R97`@9&5F:6YE
M9"P@;6%P("1#;VYF:6<Z.D-O;F9I9WLD7WTL('%W*`IA<F-H;&EB97AP('!R
M:79L:6)E>'`@<VET96%R8VAE>'`@<VET96QI8F5X<`IV96YD;W)A<F-H97AP
M('9E;F1O<FQI8F5X<`HI.PI]"G!A8VMA9V4@;6%I;CL*<F5Q=6ER92!005([
M"G5N<VAI9G0@0$E.0RP@7"9005(Z.F9I;F1?<&%R.PI005(M/FEM<&]R="A`
M<&%R7V%R9W,I.PID:64@<7%;<&%R+G!L.B!#86XG="!O<&5N('!E<FP@<V-R
M:7!T("(D<')O9VYA;64B.B!.;R!S=6-H(&9I;&4@;W(@9&ER96-T;W)Y7&Y=
M"G5N;&5S<R`M92`D<')O9VYA;64["F1O("1P<F]G;F%M93L*0T]213HZ97AI
M="@D,2D@:68@*"1`(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("1`
M(&EF("1`.PI].PHD.CI?7T524D]2(#T@)$`@:68@)$`["GT*0T]213HZ97AI
M="@D,2D@:68@*"0Z.E]?15)23U(@/7XO7E]42U]%6$E47UPH*%QD*RE<*2\I
M.PID:64@)#HZ7U]%4E)/4B!I9B`D.CI?7T524D]2.PHQ.PI?7T5.1%]?"@!P
M97)L>'-I+F,``````````$-!0TA%```!(T5G`````(FKS>\`````_MRZF```
M``!V5#(0`````/#ATL,````````````````!&P,[!`$``!\```"@B?__'`$`
M``"*__](`0``9(K__W0!``#0B___X`$``,2,__](`@``8(W__W`"``#0CO__
MP`(``,"/___X`@``T)#__RP#``#TDO__@`,``#23__^@`P``D)/__\0#``"@
MD___V`,``.25__]`!```$);__V0$``"`EO__F`0``%";___4!```T)O___@$
M``!PG/__%`4``-"=__]8!0``\)W__W@%``#TGO__M`4``+2D__\D!@``(*?_
M_U@&``"0I___N`8``%"X___D!@``D+C__P0'``"PN?__2`<``'"[__^T`0``
M\+O__Z`!```PO/__A`8````````0``````````%Z4@`$>!X!&PP?`"@````8
M````?(C__UP`````00XPG0:>!425`D64`Y,$2M330M[=U0X`````*````$0`
M``"PB/__9`````!!#C"=!IX%0I,$E`-#E0)1"M[=U=/4#@!!"P`H````<```
M`.B(__]D`0```$$.0)T(G@="E`63!D&6`Y4$09@!EP(``````!````"<````
M2+K__S@`````````*````+````"TN?__>`````!*#A"=`IX!2-[=#@!)#A"=
M`IX!0=[=#@````!D````W````.B)___T`````$$.0)T(G@="DP:4!4*8`9<"
M1I8#E016UM5!V-=#WMW3U`X`00Y`DP:4!94$E@.7`I@!G0B>!T_6U4'8UT'>
MW=/4#@!!#D"3!I0%G0B>!T*7`I@!0=C7`"0```!$`0``=(K__YP`````00[`
M`42=!)X#0I,"5@K=WM,.`$$+``!,````;`$``.B*__]L`0```$$.T`%#G0:>
M!4:4`Y,$1-332=W>#@!!#M`!DP24`YT&G@5!E@&5`G0*U--!UM5!"TG3U-76
M090#DP1!E@&5`C0```"\`0``"(S__^P`````00X@G02>`T8.L`B3`F$*#B!"
MWMW3#@!!"TH*#B!$WMW3#@!!"P``,````/0!``#`C/__"`$```!!#C"=!IX%
M19,$E`-##L`(E0*6`6$*#C!#WMW5UM/4#@!!"U`````H`@``G(W__R0"````
M00Y@09T*G@E"DPB4!TV5!I8%EP28`W\*W=[7V-76T]0.`$$+29H!F0)="MK9
M00M(V=I"F0*:`4/9VD&:`9D"0MK9`!P```!\`@``;(___T``````00X0G0*>
M`4C>W0X`````(````)P"``",C___7`````!!#B"=!)X#0Y,"3-[=TPX`````
M$````,`"``#$C___"`````````!D````U`(``,"/__]$`@```$$.4)T*G@E$
MDPB4!T,.@`J5!I8%6@H.4$/>W=76T]0.`$$+2)<$0I@#0YD"0YH!=`K70=A!
MV4':00M6"M=!V$'90=I!"T37V-G:09<$09@#09D"09H!`"`````\`P``G)'_
M_RP`````00X@G02>`T.3`D;>W=,.`````#````!@`P``I)'__VP`````0@X@
MG02>`T*3`DH*WMW3#@!!"T,*WMW3#@!""T/>W=,.```X````E`,``."1___0
M!````$$.8$.="IX)0I,(E`=&E0:6!9<$F`.9`@,=`0K=WMG7V-76T]0.`$$+
M```@````T`,``'26__]X`````$$.($6=`IX!5`K=W@X`00L````8````]`,`
M`-"6__^<`````$$.H`)%G1J>&0``0````!`$``!4E___7`$```!!#D"=")X'
M19,&E`5$#E"5!)8#EP*8`4$.T(`$>PH.P(`$00Y`1-[=U]C5UM/4#@!!"P`<
M````5`0``'"8__\@`````$$.$)T"G@%&WMT.`````#@```!T!```<)C__P0!
M````00Y`0YT&G@5"DP24`T*5`I8!60K=WM76T]0.`$$+70K=WM76T]0.`$$+
M`&P```"P!```.)G__\`%````00Z0`T6="IX)1),(E`=#E0:6!4.7!)@#6)H!
MF0("9PK:V4'=WM?8U=;3U`X`00L"@MG:3MW>U]C5UM/4#@!!#I`#DPB4!Y4&
ME@67!)@#F0*:`9T*G@D"3-G:09H!F0(P````(`4``(B>__]D`@```$$.4$&=
M")X'2),&E`5"E026`T*7`@)N"MW>U]76T]0.`$$+*````%0%``#`H/__9```
M``!!#B"=!)X#0I,"1PK>W=,.`$$+3=[=TPX````P````@`4``*2U__\D`@``
M`$$.8$.=!IX%2),$E`.5`I8!`EH*W=[5UM/4#@!!"P``````*````+0%``#0
MH/__N!````!!#A"=`IX!1PZ@!0,C!`H.$$'>W0X`00L````<````X`4``&2Q
M__]``````$$.$)T"G@%.WMT.`````$``````!@``A+'__Q@!````00Y`G0B>
M!T.3!I0%0Y4$E@-#EP*8`5D*WMW7V-76T]0.`$$+7PK>W=?8U=;3U`X`00L`
M-````$0&``!@LO__P`$```!!#D"=")X'0Y,&E`5#E026`T(%2`("8`K>W=76
MT]0&2`X`00L`````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````"04P```````!!3````
M```````````````!``````````L#`````````0`````````6`P````````$`
M````````(`,````````!`````````"X#`````````0`````````^`P``````
M``\`````````=@0````````9`````````$#]`0``````&P`````````(````
M`````!H`````````2/T!```````<``````````@`````````!`````````!(
M`@````````4`````````.`\````````&`````````'@%````````"@``````
M``"^!`````````L`````````&``````````5`````````````````````P``
M``````#H_P$```````(```````````8````````4``````````<`````````
M%P`````````0%P````````<`````````^!,````````(`````````!@#````
M````"0`````````8`````````/O__V\`````````"`````#Y__]O`````!<`
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````````!8_0$`
M`````%#]`0```````````````````````````$#]`0``````````````````
M````````````````````@`,"`````````````````$C]`0``````@`("````
M``````````````````````````````````!8_0$`````````````````````
M``````#(`@(```````````````````("``````!4/```````````````````
M`````````````````````!`=````````$!T````````0'0```````!`=````
M````$!T````````0'0```````!`=````````$!T````````0'0```````!`=
M````````$!T````````0'0```````!`=````````$!T````````0'0``````
M`!`=````````$!T````````0'0```````!`=````````$!T````````0'0``
M`````!`=````````$!T````````0'0```````!`=````````$!T````````0
M'0```````!`=````````$!T````````0'0```````!`=````````$!T`````
M```0'0```````!`=````````$!T````````0'0```````!`=````````$!T`
M```````0'0```````!`=````````$!T````````0'0```````!`=````````
M$!T````````0'0```````!`=````````$!T````````0'0```````!`=````
M````$!T````````0'0```````!`=````````$!T````````0'0```````!`=
M````````$!T````````0'0```````!`=````````$!T````````0'0``````
M`!`=````````$!T````````0'0```````!`=````````P`("```````(`@(`
M``````I005(N<&T*``````````#@60```````"!6``````````````````!@
M5P```````,!9````````R%D```````#060```````-A9````````````````
M``"H60```````+!9``````````````````#H60```````$=#0SH@*$YE=$)3
M1"!N8C$@,C`R-3`W,C$I(#$R+C4N,`!'0T,Z("AN8C$@,C`R-3`W,C$I(#$R
M+C4N,``D3F5T0E-$.B!C<G0P+E,L=B`Q+C(@,C`Q."\Q,2\R-B`Q-SHS-SHT
M-"!J;V5R9R!%>'`@)``D3F5T0E-$.B!C<G0P+6-O;6UO;BYC+'8@,2XS,"`R
M,#(U+S`U+S`R(#(S.C`T.C`V(')I87-T<F%D:"!%>'`@)``D3F5T0E-$.B!C
M<G1I+E,L=B`Q+C$@,C`Q-"\P."\Q,"`P-3HT-SHS-B!M871T($5X<"`D`"1.
M971"4T0Z(&-R=&)E9VEN+F,L=B`Q+C$W(#(P,3@O,3(O,C@@,3@Z,3<Z,3$@
M8VAR:7-T;W,@17AP("0`)$YE=$)31#H@8W)T96YD+E,L=B`Q+C$@,C`Q-"\P
M."\Q,"`P-3HT-SHS-B!M871T($5X<"`D``#QSP,`````````````````"```
M``@````(````"````!L````!```````````J*BH@3F\@3&%B96P@4')O=FED
M960@*BHJ``#QSP,`````````````````"`````@````(````"````!L````!
M```````````J*BH@3F\@3&%B96P@4')O=FED960@*BHJ```N<VAS=')T86(`
M+FEN=&5R<``N;F]T92YN971B<V0N:61E;G0`+FYO=&4N;F5T8G-D+G!A>``N
M:&%S:``N9'EN<WEM`"YD>6YS='(`+G)E;&$N9'EN`"YR96QA+G!L=``N=&5X
M=``N<F]D871A`"YE:%]F<F%M95]H9'(`+F5H7V9R86UE`"YI;FET7V%R<F%Y
M`"YF:6YI7V%R<F%Y`"YJ8W(`+F1Y;F%M:6,`+F=O=``N9V]T+G!L=``N9&%T
M80`N8G-S`"YC;VUM96YT`"Y354Y77V-T9@``````````````````````````
M````````````````````````````````````````````````````````````
M"P````$````"```````````"``````````(````````7````````````````
M`````0```````````````````!,````'`````@`````````8`@```````!@"
M````````&`````````````````````0````````````````````F````!P``
M``(`````````,`(````````P`@```````!0````````````````````$````
M````````````````-P````4````"`````````$@"````````2`(````````L
M`P````````4`````````"``````````$`````````#T````+`````@``````
M``!X!0```````'@%````````P`D````````&`````P````@`````````&```
M``````!%`````P````(`````````.`\````````X#P```````+X$````````
M```````````!````````````````````30````0````"`````````/@3````
M````^!,````````8`P````````4`````````"``````````8`````````%<`
M```$````0@`````````0%P```````!`7``````````8````````%````$P``
M``@`````````&`````````!<`````0````8`````````$!T````````0'0``
M`````"`$```````````````````0````````````````````80````$````&
M`````````#`A````````,"$```````#$-```````````````````$```````
M`````````````&<````!`````@``````````5@````````!6````````H$$`
M`````````````````!````````````````````!O`````0````(`````````
MH)<```````"@EP````````0!```````````````````$````````````````
M````?0````$````"`````````*B8````````J)@```````!\!@``````````
M````````"````````````````````(<````.`````P````````!`_0$`````
M`$#]````````"`````````````````````@`````````"`````````"3````
M#P````,`````````2/T!``````!(_0````````@````````````````````(
M``````````@`````````GP````$````#`````````%#]`0``````4/T`````
M```(````````````````````"````````````````````*0````&`````P``
M``````!8_0$``````%C]````````X`$````````&``````````@`````````
M$`````````"M`````0````,`````````./\!```````X_P```````+``````
M```````````````(``````````@`````````L@````$````#`````````.C_
M`0``````Z/\````````8`@``````````````````"``````````(````````
M`+L````!`````P```````````@(````````"`0``````B```````````````
M``````@```````````````````#!````"`````,`````````P`("``````"(
M`@$``````,@```````````````````!`````````````````````Q@````$`
M```P````````````````````B`(!```````]`````````````````````0``
M```````!`````````!\````!````,````````````````````,4"`0``````
M(@$```````````````````$``````````0````````#/`````0``````````
M``````````````#H`P$``````(\````````````````````$````````````
M`````````0````,`````````````````````````=P0!``````#9````````
9`````````````0``````````````````````
