//
// $Id: Arithmetic.h,v 1.18 2007/03/06 20:42:19 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#if !defined(ARITHMETIC_OL_GUARD)
#define ARITHMETIC_OL_GUARD

#include <ObjectiveLib/Config.h>
#if defined(OL_NO_OPENSTEP)
#include <objc/objc.h>
#else
#if defined(OL_GENERATING_DOCUMENTATION)
#include <Foundation/NSValue.h>
#else
#import <Foundation/NSValue.h>
#endif
#endif

/**
 * @protocol OLArithmetic Arithmetic.h ObjectiveLib/Arithmetic.h
 *
 * A protocol for performing simple arithmetic with arbitrary objects.
 * The primary use of this protocol is with @ref Functors "function objects"
 * that perform arithmetic, like OLPlus. All the messages in this protocol
 * perform an arithmetic operation on a passed-in argument, allocate a
 * new object for the result of the operation and return it.
 *
 * @note The objects returned by this protocol should be owned by the
 * sender of the message. That is, no attempt in implementation should
 * be made to autorelease the objects. The sender of the arithmetic
 * message will handle memory management of the returned object. The
 * reason is that this protocol is intended for use in @ref Functors
 * "function objects", which operate their own system of memory
 * management.
 *
 * @sa @ref Functors
 */
@protocol OLArithmetic

/**
 * Add the value of a given object to this object's value. The two values
 * are added together and a new object containing the result is returned.
 * The result should be the same as using operator @c + in Standard C.
 *
 * @param value the value to add
 * @return the result of the addition
 */
- (id) arithmeticAdd: (id)value;

/**
 * Divide this object's value by the value of another object. The result
 * of the division is returned in a newly created object. The result should
 * be the same as using operator @c / in Standard C.
 *
 * @param value the value by which to divide
 * @return the result of the division
 */
- (id) arithmeticDivideBy: (id)value;

/**
 * Compute the remainder of division by the value of a given object. The
 * result is returned in a newly created object. The result should be the
 * same as using operator @c \% in Standard C.
 *
 * @param value the value by which to divide
 * @return the remainder of the division
 */
- (id) arithmeticModulus: (id)value;

/**
 * Multiply this object's value by the value of a given one. The result of
 * the multiplication is returned in a newly created object. The result
 * should be the same as using operator @c * in Standard C.
 *
 * @param value the value by which to multiply
 * @return the result of the multiplication
 */
- (id) arithmeticMultiply: (id)value;

/**
 * Negate the value of this object. The negated value is returned in a
 * newly created object. The result should be the same as using the unary
 * operator @c - in Standard C.
 *
 * @return the result of the negation
 */
- (id) arithmeticNegate;

/**
 * Subtract the value of a given object from this object's value. The given
 * value is subtracted and a new object containing the result is returned.
 * The result should be the same as using binary operator @c - in Standard C.
 *
 * @param value the value to subtract
 * @return the result of the subtraction
 */
- (id) arithmeticSubtract: (id)value;

@end

#if !defined(OL_NO_OPENSTEP)
/**
 * @category NSNumber(OLArithmetic) Arithmetic.h ObjectiveLib/Arithmetic.h
 *
 * A category for implementing the OLArithmetic protocol using the NSNumber
 * class from the Cocoa Foundation library. All objects used by these methods
 * should be forms of NSNumber, or the operations will not work as expected.
 */
@interface NSNumber (OLArithmetic) <OLArithmetic>

/**
 * Add the value of a given object to this object's value. The two values
 * are added together and a new object containing the result is returned.
 * The result is the same as using operator @c + in Standard C.
 *
 * @param value the value to add
 * @return the result of the addition
 */
- (id) arithmeticAdd: (id)value;

/**
 * Divide this object's value by the value of another object. The result
 * of the division is returned in a newly created object. The result is
 * the same as using operator @c / in Standard C.
 *
 * @param value the value by which to divide
 * @return the result of the division
 */
- (id) arithmeticDivideBy: (id)value;

/**
 * Compute the remainder of division by the value of a given object. The
 * result is returned in a newly created object. The result is the
 * same as using operator @c \% in Standard C.
 *
 * @param value the value by which to divide
 * @return the remainder of the division
 */
- (id) arithmeticModulus: (id)value;

/**
 * Multiply this object's value by the value of a given one. The result of
 * the multiplication is returned in a newly created object. The result
 * is the same as using operator @c * in Standard C.
 *
 * @param value the value by which to multiply
 * @return the result of the multiplication
 */
- (id) arithmeticMultiply: (id)value;

/**
 * Negate the value of this object. The negated value is returned in a
 * newly created object. The result is the same as using the unary
 * operator @c - in Standard C.
 *
 * @return the result of the negation
 */
- (id) arithmeticNegate;

/**
 * Subtract the value of a given object from this object's value. The given
 * value is subtracted and a new object containing the result is returned.
 * The result is the same as using binary operator @c - in Standard C.
 *
 * @param value the value to subtract
 * @return the result of the subtraction
 */
- (id) arithmeticSubtract: (id)value;

@end
#endif

#endif
