/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.base.Predicate;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.util.concurrent.RateLimiter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Memtable;
import org.apache.cassandra.db.compaction.AbstractCompactionTask;
import org.apache.cassandra.db.compaction.CompactionController;
import org.apache.cassandra.db.compaction.CompactionManager;
import org.apache.cassandra.db.compaction.CompactionTask;
import org.apache.cassandra.db.lifecycle.LifecycleTransaction;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.ISSTableScanner;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractCompactionStrategy {
    private static final Logger logger = LoggerFactory.getLogger(AbstractCompactionStrategy.class);
    protected static final float DEFAULT_TOMBSTONE_THRESHOLD = 0.2f;
    protected static final long DEFAULT_TOMBSTONE_COMPACTION_INTERVAL = 86400L;
    protected static final boolean DEFAULT_UNCHECKED_TOMBSTONE_COMPACTION_OPTION = false;
    protected static final String TOMBSTONE_THRESHOLD_OPTION = "tombstone_threshold";
    protected static final String TOMBSTONE_COMPACTION_INTERVAL_OPTION = "tombstone_compaction_interval";
    protected static final String UNCHECKED_TOMBSTONE_COMPACTION_OPTION = "unchecked_tombstone_compaction";
    protected static final String COMPACTION_ENABLED = "enabled";
    public Map<String, String> options;
    protected final ColumnFamilyStore cfs;
    protected float tombstoneThreshold;
    protected long tombstoneCompactionInterval;
    protected boolean uncheckedTombstoneCompaction;
    protected boolean disableTombstoneCompactions = false;
    protected boolean isActive = false;
    protected volatile boolean enabled = true;

    protected AbstractCompactionStrategy(ColumnFamilyStore cfs, Map<String, String> options) {
        assert (cfs != null);
        this.cfs = cfs;
        this.options = ImmutableMap.copyOf(options);
        try {
            AbstractCompactionStrategy.validateOptions(options);
            String optionValue = options.get(TOMBSTONE_THRESHOLD_OPTION);
            this.tombstoneThreshold = optionValue == null ? 0.2f : Float.parseFloat(optionValue);
            optionValue = options.get(TOMBSTONE_COMPACTION_INTERVAL_OPTION);
            this.tombstoneCompactionInterval = optionValue == null ? 86400L : Long.parseLong(optionValue);
            optionValue = options.get(UNCHECKED_TOMBSTONE_COMPACTION_OPTION);
            boolean bl = this.uncheckedTombstoneCompaction = optionValue == null ? false : Boolean.parseBoolean(optionValue);
            if (!this.shouldBeEnabled()) {
                this.disable();
            }
        }
        catch (ConfigurationException e) {
            logger.warn("Error setting compaction strategy options ({}), defaults will be used", (Object)e.getMessage());
            this.tombstoneThreshold = 0.2f;
            this.tombstoneCompactionInterval = 86400L;
            this.uncheckedTombstoneCompaction = false;
        }
    }

    public synchronized void pause() {
        this.isActive = false;
    }

    public synchronized void resume() {
        this.isActive = true;
    }

    public void startup() {
        this.isActive = true;
    }

    public void shutdown() {
        this.isActive = false;
    }

    public abstract AbstractCompactionTask getNextBackgroundTask(int var1);

    public abstract Collection<AbstractCompactionTask> getMaximalTask(int var1, boolean var2);

    public abstract AbstractCompactionTask getUserDefinedTask(Collection<SSTableReader> var1, int var2);

    public AbstractCompactionTask getCompactionTask(LifecycleTransaction txn, int gcBefore, long maxSSTableBytes) {
        return new CompactionTask(this.cfs, txn, gcBefore, false);
    }

    public abstract int getEstimatedRemainingTasks();

    public abstract long getMaxSSTableBytes();

    public boolean isEnabled() {
        return this.enabled && this.isActive;
    }

    public void enable() {
        this.enabled = true;
    }

    public void disable() {
        this.enabled = false;
    }

    public boolean isAffectedByMeteredFlusher() {
        return true;
    }

    public long getMemtableReservedSize() {
        return 0L;
    }

    public void replaceFlushed(Memtable memtable, SSTableReader sstable) {
    }

    public List<SSTableReader> filterSSTablesForReads(List<SSTableReader> sstables) {
        return sstables;
    }

    public static Iterable<SSTableReader> filterSuspectSSTables(Iterable<SSTableReader> originalCandidates) {
        return Iterables.filter(originalCandidates, (Predicate)new Predicate<SSTableReader>(){

            public boolean apply(SSTableReader sstable) {
                return !sstable.isMarkedSuspect();
            }
        });
    }

    public ScannerList getScanners(Collection<SSTableReader> sstables, Range<Token> range) {
        RateLimiter limiter = CompactionManager.instance.getRateLimiter();
        ArrayList<ISSTableScanner> scanners = new ArrayList<ISSTableScanner>();
        try {
            for (SSTableReader sstable : sstables) {
                scanners.add(sstable.getScanner(range, limiter));
            }
        }
        catch (Throwable t) {
            try {
                new ScannerList(scanners).close();
            }
            catch (Throwable t2) {
                t.addSuppressed(t2);
            }
            throw t;
        }
        return new ScannerList(scanners);
    }

    public boolean shouldDefragment() {
        return false;
    }

    public String getName() {
        return this.getClass().getSimpleName();
    }

    public synchronized void replaceSSTables(Collection<SSTableReader> removed, Collection<SSTableReader> added) {
        for (SSTableReader remove : removed) {
            this.removeSSTable(remove);
        }
        for (SSTableReader add : added) {
            this.addSSTable(add);
        }
    }

    public abstract void addSSTable(SSTableReader var1);

    public abstract void removeSSTable(SSTableReader var1);

    public ScannerList getScanners(Collection<SSTableReader> toCompact) {
        return this.getScanners(toCompact, null);
    }

    protected boolean worthDroppingTombstones(SSTableReader sstable, int gcBefore) {
        if (this.disableTombstoneCompactions || CompactionController.NEVER_PURGE_TOMBSTONES) {
            return false;
        }
        if (System.currentTimeMillis() < sstable.getCreationTimeFor(Component.DATA) + this.tombstoneCompactionInterval * 1000L) {
            return false;
        }
        double droppableRatio = sstable.getEstimatedDroppableTombstoneRatio(gcBefore);
        if (droppableRatio <= (double)this.tombstoneThreshold) {
            return false;
        }
        if (this.uncheckedTombstoneCompaction) {
            return true;
        }
        Collection<SSTableReader> overlaps = this.cfs.getOverlappingSSTables(Collections.singleton(sstable));
        if (overlaps.isEmpty()) {
            return true;
        }
        if (CompactionController.getFullyExpiredSSTables(this.cfs, Collections.singleton(sstable), overlaps, gcBefore).size() > 0) {
            return true;
        }
        if (sstable.getIndexSummarySize() < 2) {
            return false;
        }
        long keys = sstable.estimatedKeys();
        HashSet<Range<Token>> ranges = new HashSet<Range<Token>>(overlaps.size());
        for (SSTableReader overlap : overlaps) {
            ranges.add(new Range<Token>(overlap.first.getToken(), overlap.last.getToken()));
        }
        long remainingKeys = keys - sstable.estimatedKeysForRanges(ranges);
        long columns = sstable.getEstimatedColumnCount().mean() * remainingKeys;
        double remainingColumnsRatio = (double)columns / (double)(sstable.getEstimatedColumnCount().count() * sstable.getEstimatedColumnCount().mean());
        return remainingColumnsRatio * droppableRatio > (double)this.tombstoneThreshold;
    }

    public static Map<String, String> validateOptions(Map<String, String> options) throws ConfigurationException {
        String unchecked;
        String interval;
        String threshold = options.get(TOMBSTONE_THRESHOLD_OPTION);
        if (threshold != null) {
            try {
                float thresholdValue = Float.parseFloat(threshold);
                if (thresholdValue < 0.0f) {
                    throw new ConfigurationException(String.format("%s must be greater than 0, but was %f", TOMBSTONE_THRESHOLD_OPTION, Float.valueOf(thresholdValue)));
                }
            }
            catch (NumberFormatException e) {
                throw new ConfigurationException(String.format("%s is not a parsable int (base10) for %s", threshold, TOMBSTONE_THRESHOLD_OPTION), e);
            }
        }
        if ((interval = options.get(TOMBSTONE_COMPACTION_INTERVAL_OPTION)) != null) {
            try {
                long tombstoneCompactionInterval = Long.parseLong(interval);
                if (tombstoneCompactionInterval < 0L) {
                    throw new ConfigurationException(String.format("%s must be greater than 0, but was %d", TOMBSTONE_COMPACTION_INTERVAL_OPTION, tombstoneCompactionInterval));
                }
            }
            catch (NumberFormatException e) {
                throw new ConfigurationException(String.format("%s is not a parsable int (base10) for %s", interval, TOMBSTONE_COMPACTION_INTERVAL_OPTION), e);
            }
        }
        if ((unchecked = options.get(UNCHECKED_TOMBSTONE_COMPACTION_OPTION)) != null && !unchecked.equalsIgnoreCase("true") && !unchecked.equalsIgnoreCase("false")) {
            throw new ConfigurationException(String.format("'%s' should be either 'true' or 'false', not '%s'", UNCHECKED_TOMBSTONE_COMPACTION_OPTION, unchecked));
        }
        String compactionEnabled = options.get(COMPACTION_ENABLED);
        if (compactionEnabled != null && !compactionEnabled.equalsIgnoreCase("true") && !compactionEnabled.equalsIgnoreCase("false")) {
            throw new ConfigurationException(String.format("enabled should either be 'true' or 'false', not %s", compactionEnabled));
        }
        HashMap<String, String> uncheckedOptions = new HashMap<String, String>(options);
        uncheckedOptions.remove(TOMBSTONE_THRESHOLD_OPTION);
        uncheckedOptions.remove(TOMBSTONE_COMPACTION_INTERVAL_OPTION);
        uncheckedOptions.remove(UNCHECKED_TOMBSTONE_COMPACTION_OPTION);
        uncheckedOptions.remove(COMPACTION_ENABLED);
        return uncheckedOptions;
    }

    public boolean shouldBeEnabled() {
        String optionValue = this.options.get(COMPACTION_ENABLED);
        return optionValue == null || Boolean.parseBoolean(optionValue);
    }

    public Collection<Collection<SSTableReader>> groupSSTablesForAntiCompaction(Collection<SSTableReader> sstablesToGroup) {
        int groupSize = 2;
        ArrayList<SSTableReader> sortedSSTablesToGroup = new ArrayList<SSTableReader>(sstablesToGroup);
        Collections.sort(sortedSSTablesToGroup, SSTableReader.sstableComparator);
        ArrayList<Collection<SSTableReader>> groupedSSTables = new ArrayList<Collection<SSTableReader>>();
        ArrayList<SSTableReader> currGroup = new ArrayList<SSTableReader>();
        for (SSTableReader sstable : sortedSSTablesToGroup) {
            currGroup.add(sstable);
            if (currGroup.size() != groupSize) continue;
            groupedSSTables.add(currGroup);
            currGroup = new ArrayList();
        }
        if (currGroup.size() != 0) {
            groupedSSTables.add(currGroup);
        }
        return groupedSSTables;
    }

    public static class ScannerList
    implements AutoCloseable {
        public final List<ISSTableScanner> scanners;

        public ScannerList(List<ISSTableScanner> scanners) {
            this.scanners = scanners;
        }

        @Override
        public void close() {
            Throwable t = null;
            for (ISSTableScanner scanner : this.scanners) {
                try {
                    scanner.close();
                }
                catch (Throwable t2) {
                    JVMStabilityInspector.inspectThrowable(t2);
                    if (t == null) {
                        t = t2;
                        continue;
                    }
                    t.addSuppressed(t2);
                }
            }
            if (t != null) {
                throw Throwables.propagate(t);
            }
        }
    }
}

