/*
 *  SPDX-FileCopyrightText: 2016 Michael Abrahams <miabraha@gmail.com>
 *
 *  SPDX-License-Identifier: GPL-2.0-or-later
 */

#include "kis_cursor_cache.h"

#include <QScreen>
#include <QWindow>
#include <QBitmap>
#include <QImage>
#include <qmath.h>
#include <QDebug>
#include <QPainter>
#include <QApplication>

#include "KoResourcePaths.h"

Q_GLOBAL_STATIC(KisCursorCache, s_instance)

// Very old cursors were saved in these byte arrays.
namespace {
    static const unsigned char select_bits[] = {
        0x00, 0x08, 0x00, 0x00, 0x08, 0x00, 0x00, 0x08, 0x00, 0x00, 0x08, 0x00,
        0x00, 0x08, 0x00, 0x00, 0x08, 0x00, 0x00, 0x08, 0x00, 0x00, 0x08, 0x00,
        0x00, 0x08, 0x00, 0x00, 0x08, 0x00, 0x00, 0x08, 0x00, 0xff, 0xff, 0x7f,
        0x00, 0x08, 0x00, 0x00, 0x08, 0x00, 0x00, 0x08, 0x00, 0x00, 0x08, 0x00,
        0x00, 0x08, 0x00, 0x00, 0x08, 0x00, 0x00, 0x08, 0x00, 0x00, 0x08, 0x00,
        0x00, 0x08, 0x00, 0x00, 0x08, 0x00, 0x00, 0x08, 0x00
    };

    static const unsigned char samplerplus_bits[] = {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x80, 0x1e,
        0x00, 0x00, 0x40, 0x1f, 0x00, 0x00, 0xb0, 0x1f, 0x00, 0x00, 0xe8, 0x0f,
        0x00, 0x00, 0xd0, 0x07, 0x00, 0x00, 0xa8, 0x03, 0x00, 0x00, 0x64, 0x03,
        0x00, 0x00, 0x72, 0x01, 0x00, 0x00, 0xb9, 0x00, 0x00, 0x80, 0x1c, 0x00,
        0x00, 0x40, 0x0e, 0x00, 0x00, 0x20, 0x07, 0x00, 0x00, 0x90, 0x03, 0x00,
        0x00, 0xc8, 0x01, 0x01, 0x40, 0xe4, 0x00, 0x01, 0x40, 0x74, 0xc0, 0x07,
        0x40, 0x3c, 0x00, 0x01, 0x40, 0x00, 0x00, 0x01, 0x40, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x1f, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
        0x40, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00
    };

    static const unsigned char samplerminus_bits[] = {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x80, 0x1e,
        0x00, 0x00, 0x40, 0x1f, 0x00, 0x00, 0xb0, 0x1f, 0x00, 0x00, 0xe8, 0x0f,
        0x00, 0x00, 0xd0, 0x07, 0x00, 0x00, 0xa8, 0x03, 0x00, 0x00, 0x64, 0x03,
        0x00, 0x00, 0x72, 0x01, 0x00, 0x00, 0xb9, 0x00, 0x00, 0x80, 0x1c, 0x00,
        0x00, 0x40, 0x0e, 0x00, 0x00, 0x20, 0x07, 0x00, 0x00, 0x90, 0x03, 0x00,
        0x00, 0xc8, 0x01, 0x00, 0x40, 0xe4, 0x00, 0x00, 0x40, 0x74, 0xc0, 0x07,
        0x40, 0x3c, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x1f, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
        0x40, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00
    };

    static const unsigned char pen_bits[] = {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x3a, 0x00, 0x00, 0x7d,
        0x00, 0x80, 0x7e, 0x00, 0x40, 0x7f, 0x00, 0xa0, 0x3f, 0x00, 0xd0, 0x1f,
        0x00, 0xe8, 0x0f, 0x00, 0xf4, 0x07, 0x00, 0xfa, 0x03, 0x00, 0xfd, 0x01,
        0x80, 0xfe, 0x00, 0x40, 0x7f, 0x00, 0xa0, 0x3f, 0x00, 0xf0, 0x1f, 0x00,
        0xd0, 0x0f, 0x00, 0x88, 0x07, 0x00, 0x88, 0x03, 0x00, 0xe4, 0x01, 0x00,
        0x7c, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00
    };

    static const unsigned char brush_bits[] = {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x68, 0x00,
        0x00, 0x00, 0xf4, 0x00, 0x00, 0x00, 0xfa, 0x00, 0x00, 0x00, 0xfd, 0x00,
        0x00, 0x80, 0x7e, 0x00, 0x00, 0x40, 0x3f, 0x00, 0x00, 0xa0, 0x1f, 0x00,
        0x00, 0xd0, 0x0f, 0x00, 0x00, 0xe8, 0x07, 0x00, 0x00, 0xf4, 0x03, 0x00,
        0x00, 0xe4, 0x01, 0x00, 0x00, 0xc2, 0x00, 0x00, 0x80, 0x41, 0x00, 0x00,
        0x40, 0x32, 0x00, 0x00, 0xa0, 0x0f, 0x00, 0x00, 0xd0, 0x0f, 0x00, 0x00,
        0xd0, 0x0f, 0x00, 0x00, 0xe8, 0x07, 0x00, 0x00, 0xf4, 0x01, 0x00, 0x00,
        0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
    };

    static const unsigned char airbrush_bits[] = {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x68, 0x00, 0x00, 0x74,
        0x00, 0x00, 0x7a, 0xf0, 0x00, 0x3d, 0x08, 0x81, 0x1e, 0xe8, 0x41, 0x0f,
        0xe8, 0xa1, 0x07, 0xe8, 0xd1, 0x03, 0xe8, 0xe9, 0x01, 0xe8, 0xf5, 0x00,
        0xe8, 0x7b, 0x00, 0xf0, 0x33, 0x00, 0xf0, 0x23, 0x1f, 0xa0, 0x9f, 0x3f,
        0xd0, 0xff, 0x31, 0xe8, 0xf7, 0x30, 0xf4, 0x03, 0x18, 0xfc, 0x01, 0x0c,
        0xf8, 0x00, 0x06, 0x76, 0x00, 0x03, 0x36, 0x00, 0x03, 0x00, 0x00, 0x00
    };

    static const unsigned char eraser_bits[] = {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x1d, 0x00,
        0x00, 0x80, 0x3e, 0x00, 0x00, 0x40, 0x7f, 0x00, 0x00, 0xa0, 0xff, 0x00,
        0x00, 0xd0, 0xff, 0x00, 0x00, 0xe8, 0x7f, 0x00, 0x00, 0xf4, 0x3f, 0x00,
        0x00, 0xfe, 0x1f, 0x00, 0x00, 0xf9, 0x0f, 0x00, 0x80, 0xf2, 0x07, 0x00,
        0x40, 0xe7, 0x03, 0x00, 0xa0, 0xcf, 0x01, 0x00, 0xd0, 0x9f, 0x00, 0x00,
        0xe8, 0x7f, 0x00, 0x00, 0xfc, 0x3f, 0x00, 0x00, 0xf2, 0x1f, 0x00, 0x00,
        0xe2, 0x0f, 0x00, 0x00, 0xc4, 0x07, 0x00, 0x00, 0x88, 0x03, 0x00, 0x00,
        0x10, 0x01, 0x00, 0x00, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
    };

    static const unsigned char filler_bits[] = {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x28, 0x00,
        0x00, 0x54, 0x00, 0x00, 0x4e, 0x00, 0x00, 0x85, 0x00, 0x80, 0x0a, 0x01,
        0x40, 0x11, 0x01, 0xe0, 0x00, 0x02, 0x58, 0x01, 0x04, 0x2c, 0x02, 0x04,
        0x44, 0x04, 0x08, 0x0c, 0x08, 0x18, 0x3c, 0x00, 0x14, 0x5c, 0x00, 0x0a,
        0x9c, 0x01, 0x05, 0x1c, 0x82, 0x02, 0x18, 0x4c, 0x01, 0x18, 0xb0, 0x00,
        0x08, 0x60, 0x00, 0x00, 0x00, 0x00
    };

    static const unsigned char colorChanger_bits[] = {
        0x00, 0x00, 0x00, 0xe0, 0x00, 0x00, 0x10, 0x01, 0x0e, 0x08, 0x02, 0x11,
        0x04, 0x82, 0x20, 0x64, 0x84, 0x20, 0x92, 0x44, 0x46, 0x12, 0x49, 0x5f,
        0x12, 0x31, 0x5f, 0x22, 0x01, 0x5f, 0xc2, 0x00, 0x4e, 0x02, 0x00, 0x40,
        0xc2, 0x00, 0x46, 0xe2, 0x01, 0x4f, 0xe4, 0x19, 0x2f, 0xe4, 0x3d, 0x2f,
        0xe8, 0x3d, 0x17, 0xd0, 0x3c, 0x10, 0x20, 0x38, 0x08, 0x40, 0x00, 0x06,
        0x80, 0x81, 0x01, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00
    };

    inline QBitmap bitmapFromData(const QSize& size, const unsigned char* data)
    {
        QBitmap result(32, 32);
        result.fill(Qt::color0);
        QPainter painter(&result);
        painter.drawPixmap(0, 0, QBitmap::fromData(size, data));
        return result;
    }

    inline QCursor fromBitmap(QSize size, const unsigned char* bits, int hotspotX, int hotspotY)
    {
        QBitmap bitmap = bitmapFromData(size, bits);
        QBitmap mask = bitmap.createHeuristicMask(false);
        return QCursor(bitmap, mask, hotspotX, hotspotY);
    }

    QCursor loadImpl(const QString &cursorName, int hotspotX, int hotspotY) {
        QPixmap cursorImage = QPixmap(":/" + cursorName);
        if (cursorImage.isNull()) {
            qWarning() << "Could not load cursor from qrc, trying filesystem" << cursorName;

            cursorImage = QPixmap(KoResourcePaths::findResource("kis_pics", cursorName));
            if (cursorImage.isNull()) {
                qWarning() << "Could not load cursor from filesystem" << cursorName;
                return Qt::ArrowCursor;
            }
        }

        return QCursor(cursorImage, hotspotX, hotspotY);
    }

}

KisCursorCache::KisCursorCache() {}

KisCursorCache* KisCursorCache::instance()
{
    if (!s_instance.exists()) {
        s_instance->init();
        // TODO: listen for DPI change signals
    }
    return s_instance;
}

QCursor KisCursorCache::load(const QString & cursorName, int hotspotX, int hotspotY)
{
    if (cursorHash.contains(cursorName)) {
        return cursorHash[ cursorName ].second;
    }

    // Otherwise, construct the cursor
    QCursor newCursor = loadImpl(cursorName, hotspotX, hotspotY);
    cursorHash.insert(cursorName, QPair<QPoint, QCursor>(QPoint(hotspotX, hotspotY), newCursor));
    return newCursor;
}

void KisCursorCache::init()
{
    selectCursor       = fromBitmap(QSize(23, 23), select_bits, 11, 11);
    samplerPlusCursor   = fromBitmap(QSize(32, 32), samplerplus_bits, 6, 25);
    samplerMinusCursor  = fromBitmap(QSize(32, 32), samplerminus_bits, 6, 25);
    penCursor          = fromBitmap(QSize(24, 24), pen_bits, 1, 22);
    brushCursor        = fromBitmap(QSize(25, 23), brush_bits, 1, 21);
    airbrushCursor     = fromBitmap(QSize(24, 24), airbrush_bits, 1, 22);
    eraserCursor       = fromBitmap(QSize(25, 24), eraser_bits, 7, 22);
    colorChangerCursor = fromBitmap(QSize(24, 23), colorChanger_bits, 12, 10);
    fillerCursor       = fromBitmap(QSize(22, 22), filler_bits, 3, 20);
}

