/*
 * i-scream central monitoring system
 * http://www.i-scream.org.uk
 * Copyright (C) 2000-2002 i-scream
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <stdlib.h>
#include <ukcprog.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/termios.h>
#include <signal.h>
#include <errno.h>

#include <libxml/xmlmemory.h>
#include <libxml/parser.h>
#include "genmergesort.h"

#ifdef HAVE_NCURSES_H
#include <ncurses.h>
#else
#include <curses.h>
#endif

struct host_line_t{
	char *hostname;
	int line;

	struct host_line_t *next;
};
typedef struct host_line_t host_line_list_t;

struct diskio_data_t{
	char *name;

	long long read;
	long long write;
	
	struct diskio_data_t *next;
};
typedef struct diskio_data_t diskio_data_list_t;

struct network_data_t{
	char *name;

	long long rx;
	long long tx;
	
	struct network_data_t *next;
};
typedef struct network_data_t network_data_list_t;

/*
struct disk_data_t{
	char *name;
	char *mount_pnt;

	long long total_space;
	long long total_used;
	long long total_avail;
	// Other data we are less intrested in are not stored
	
	struct disk_data_t *next;
};
typedef struct disk_data_t disk_data_list_t;
*/
#define MAXHOSTNAMESIZE 10
struct machine_data_t{

	char *name;

	char sysname[MAXHOSTNAMESIZE+1];

	double cpu_user;
	double cpu_idle;
	double cpu_iowait;
	double cpu_kernel;
	double cpu_swap; 
	double cpu_used; /* 100 - idle */

	long long memory_total;
	long long memory_free;
	long long memory_used;
	double memory_used_pecent;
	
	long long swap_total;
	long long swap_free;
	long long swap_used;
	double swap_used_pecent;

	int pages_in;
	int pages_out;

	double load_1;
	double load_5;
	double load_15;

	int processes_total;
	int processes_sleeping;
	int processes_cpu;
	int processes_zombie;
	int processes_stopped;

	network_data_list_t *network_data_list;
	long long network_io_total_tx;
	long long network_io_total_rx;
	long long network_io_total;

	diskio_data_list_t *diskio_data_list;
	long long disk_io_total_write;
        long long disk_io_total_read;
	long long disk_io_total;

	/* Maybe in the future */
	/*
	disk_data_list_t disk_data_list;
	double disk_total_used;
	*/

	struct machine_data_t *next;
};

typedef struct machine_data_t machine_data_list_t;

#define SORTBYMAXNAME 128
typedef struct{
	int maxx;
	int maxy;

	char units;

	int cpu_user;
	int cpu_idle;
	int cpu_iowait;
	int cpu_kernel;
	int cpu_swap;
	int cpu_used; 

	int memory_total;
	int memory_free;
	int memory_used;
	int memory_used_pecent;

	int swap_total;
	int swap_free;
	int swap_used;
	int swap_used_pecent;

	int load_1;
	int load_5;
	int load_15;

	int pages_in;
	int pages_out;

	int processes;

	int network_io_total_tx;
	int network_io_total_rx;
	int network_all_stats;

	int disk_io_total_write;
	int disk_io_total_read;
	int disk_io_all_stats;

	int disk_total_used;
	int disk_all_stats; 

	char sortby[SORTBYMAXNAME];
}display_config_t;

GENERIC_MERGE_SORT(static, sort_machine_stats, machine_data_list_t, next)

#define MKCMP(x) int cmp_##x(machine_data_list_t *a, machine_data_list_t *b){return ((a->x) == (b->x)? 0 : (((a->x) > (b->x))? -1 : 1));}


int (*sortby_ptr)(machine_data_list_t *a, machine_data_list_t *b);

MKCMP(cpu_used)
MKCMP(load_1)
MKCMP(network_io_total)
MKCMP(network_io_total_tx)
MKCMP(network_io_total_rx)
MKCMP(disk_io_total)
MKCMP(disk_io_total_write)
MKCMP(disk_io_total_read)
MKCMP(memory_used_pecent)
MKCMP(swap_used_pecent)

#define CPU_USED "CPU used"
#define LOAD "Load (1)"
#define NETIORX "total Network RX for all interfaces"
#define NETIOTX "total Network TX for all interfaces"
#define NETIO "total Network IO for all interfaces (rx+tx)"
#define MEM "Memory usage"
#define SWAP "Swap usage"
#define DISKIOR "DiskIO reads"
#define DISKIOW "DiskIO writes"
#define DISKIO "Total DiskIO (reads+writes)"

int sig_winch=0;

#ifndef HAVE_ATOLL
long long int atoll (const char *nptr){
  return strtoll (nptr, (char **) NULL, 10);
}
#endif

#ifndef HAVE_STRLCPY
/*
 * Copyright (c) 1998 Todd C. Miller <Todd.Miller@courtesan.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Copy src to string dst of size siz.  At most siz-1 characters
 * will be copied.  Always NUL terminates (unless siz == 0).
 * Returns strlen(src); if retval >= siz, truncation occurred.
 */
size_t
strlcpy(dst, src, siz)
        char *dst;
        const char *src;
        size_t siz;
{
        register char *d = dst;
        register const char *s = src;
        register size_t n = siz;

        /* Copy as many bytes as will fit */
        if (n != 0 && --n != 0) {
                do {
                        if ((*d++ = *s++) == 0)
                                break;
                } while (--n != 0);
        }

        /* Not enough room in dst, add NUL and traverse rest of src */
        if (n == 0) {
                if (siz != 0)
                        *d = '\0';              /* NUL-terminate dst */
                while (*s++)
                        ;
        }

        return(s - src - 1);    /* count does not include NUL */
}

#endif

	
FILE *create_tcp_connection(char *hostname, int port){
	int sock;
	struct sockaddr_in addr;
	struct in_addr haddr;
	FILE *f;

	if((sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP))<0){
		return NULL;
	}

	if((get_host_addr(hostname, &haddr))!=0){
		close(sock);
		return NULL;
	}

	memset(&addr, 0, sizeof(addr));
	addr.sin_family = AF_INET;
	memcpy(&addr.sin_addr, &haddr, sizeof(haddr));
	addr.sin_port = htons(port);

	if(connect(sock, (struct sockaddr *)&addr, sizeof(addr)) !=0){
		close(sock);
		return NULL;
	}

	if((f=fdopen(sock, "r+"))==NULL){
		close(sock);
		return NULL;
	}

	return f;
}

int tcp_comm(FILE *f, char *send, char **response, char *expected){

	if(send!=NULL){
		fprintf(f, "%s\n", send);
	}
	fflush(f);
	*response=fpgetline(f);
	fseek(f, 0, SEEK_CUR);

	if( (*response==NULL) || (strcmp(*response, "ERROR")==0) ) return -1;

	if(expected==NULL) return 0;

	if((strcmp(expected, *response))==0) return 0;

	return -1;
}

/* Takes a xml char * and a machine_data_list_t. This will parse
 * the xml and put it into the correct entry of the machine_data
 * linked list. This will return the number of entries in the linked
 * list, or -1 on a error.
 */
int parse_xml(char *xml, machine_data_list_t **md){
	xmlDocPtr doc;
	xmlNodePtr cur;
	xmlNodePtr ele;
	xmlAttr *list;

	static int num_hosts=0;

	machine_data_list_t *machine_data_list=*md;

	int found_host=0;

	char *hostname=NULL;
	network_data_list_t *network_data_list=NULL;
	network_data_list_t *ndl_ptr=NULL;
	diskio_data_list_t *diskio_data_list=NULL;
	diskio_data_list_t *didl_ptr=NULL;
	char *tmp;

	doc=xmlParseDoc(xml);
	if(doc==NULL) return -1;
	
	cur = xmlDocGetRootElement(doc);
	if(cur==NULL){
		xmlFreeDoc(doc);
		return -1;
	}

		
	/* Get the hostname */
	list=cur->properties;
	while(list!=NULL){
		if((xmlStrcmp(list->name, (const xmlChar *) "machine_name"))==0){
			hostname = xmlNodeGetContent(list->children);
			if(hostname==NULL){
				return -1;
			}	
		}
		list=list->next;
	}
	if(hostname==NULL){
		return -1;
	}

	/* Get machine_data for host.. Or create if it doesn't have one */
	while(machine_data_list!=NULL){
		if((strncmp(machine_data_list->name, hostname, strlen(hostname)))==0){
			found_host=1;
			break;
		}
		machine_data_list=machine_data_list->next;
	}

	if(!found_host){
		/* We didn't find this host, but we should be at the end of the list */
		machine_data_list=malloc(sizeof(machine_data_list_t));
		if(machine_data_list==NULL) return -1;
		machine_data_list->next=(*md);
		machine_data_list->name=hostname;
		machine_data_list->network_data_list=NULL;
		machine_data_list->diskio_data_list=NULL;
		*md=machine_data_list;
		num_hosts++;
	}
		
	/* Now we want to pull out the data */

	cur = cur->xmlChildrenNode;
	while(cur != NULL) {
		ele=cur->xmlChildrenNode;
		while(ele != NULL){

			/* CPU Stats */
			if(!xmlStrcmp(cur->name, (const xmlChar *) "cpu")){
				tmp=xmlNodeGetContent(ele);
				if(!xmlStrcmp(ele->name, (const xmlChar *) "user")){
					machine_data_list->cpu_user=atof(tmp);
				}
				if(!xmlStrcmp(ele->name, (const xmlChar *) "kernel")){
					machine_data_list->cpu_kernel=atof(tmp);
				}
				if(!xmlStrcmp(ele->name, (const xmlChar *) "idle")){
					machine_data_list->cpu_idle=atof(tmp);
				}
				if(!xmlStrcmp(ele->name, (const xmlChar *) "iowait")){
					machine_data_list->cpu_iowait=atof(tmp);
				}
				if(!xmlStrcmp(ele->name, (const xmlChar *) "swap")){
					machine_data_list->cpu_iowait=atof(tmp);
				}
				
				if(tmp!=NULL) xmlFree(tmp);
			}

			/* Memory Stats */
			if(!xmlStrcmp(cur->name, (const xmlChar *) "memory")){
				tmp=xmlNodeGetContent(ele);
				if(!xmlStrcmp(ele->name, (const xmlChar *) "total")){
					machine_data_list->memory_total=atoll(tmp);
				}
                                if(!xmlStrcmp(ele->name, (const xmlChar *) "free")){
                                        machine_data_list->memory_free=atoll(tmp);
                                }
                                if(!xmlStrcmp(ele->name, (const xmlChar *) "used")){
                                        machine_data_list->memory_used=atoll(tmp);
                                }

				if(tmp!=NULL) xmlFree(tmp);
			}

			/* Load Stats */
			if(!xmlStrcmp(cur->name, (const xmlChar *) "load")){
				tmp=xmlNodeGetContent(ele);
				if(!xmlStrcmp(ele->name, (const xmlChar *) "load1")){
					machine_data_list->load_1=atof(tmp);
				}
                                if(!xmlStrcmp(ele->name, (const xmlChar *) "load5")){
                                        machine_data_list->load_5=atof(tmp);
                                }
                                if(!xmlStrcmp(ele->name, (const xmlChar *) "load15")){
                                        machine_data_list->load_15=atof(tmp);
                                }
			
				if(tmp!=NULL) xmlFree(tmp);
			}

			/* swap stats */
                        if(!xmlStrcmp(cur->name, (const xmlChar *) "swap")){
                                tmp=xmlNodeGetContent(ele);
                                if(!xmlStrcmp(ele->name, (const xmlChar *) "total")){
                                        machine_data_list->swap_total=atoll(tmp);
                                }
                                if(!xmlStrcmp(ele->name, (const xmlChar *) "free")){
                                        machine_data_list->swap_free=atoll(tmp);
                                }
                                if(!xmlStrcmp(ele->name, (const xmlChar *) "used")){
                                        machine_data_list->swap_used=atoll(tmp);
                                }

                                if(tmp!=NULL) xmlFree(tmp);
                        }

			/* Process stat */
			if(!xmlStrcmp(cur->name, (const xmlChar *) "processes")){
				tmp=xmlNodeGetContent(ele);
				if(!xmlStrcmp(ele->name, (const xmlChar *) "sleeping")){
					machine_data_list->processes_sleeping=atoi(tmp);
				}
                                if(!xmlStrcmp(ele->name, (const xmlChar *) "cpu")){
                                        machine_data_list->processes_cpu=atoi(tmp);
                                }
                                if(!xmlStrcmp(ele->name, (const xmlChar *) "zombie")){
                                        machine_data_list->processes_zombie=atoi(tmp);
                                }
                                if(!xmlStrcmp(ele->name, (const xmlChar *) "total")){
                                        machine_data_list->processes_total=atoi(tmp);
                                }

				if(tmp!=NULL) xmlFree(tmp);
			}

			/* paging stats */
                        if(!xmlStrcmp(cur->name, (const xmlChar *) "pages")){
                                tmp=xmlNodeGetContent(ele);
                                if(!xmlStrcmp(ele->name, (const xmlChar *) "pageins")){
                                        machine_data_list->pages_in=atoi(tmp);
                                }
                                if(!xmlStrcmp(ele->name, (const xmlChar *) "pageouts")){
                                        machine_data_list->pages_out=atoi(tmp);
                                }
			
				if(tmp!=NULL) xmlFree(tmp);
                        }
			
			/* OS stats */
			if(!xmlStrcmp(cur->name, (const xmlChar *) "os")){
				tmp=xmlNodeGetContent(ele);
				if(!xmlStrcmp(ele->name, (const xmlChar *) "sysname")){
					strlcpy(machine_data_list->sysname, tmp, MAXHOSTNAMESIZE+1);
				}
				if(tmp!=NULL) xmlFree(tmp);
			}

			/* Network stats */
			/* Needs to connect current stat to a previous one. Or create it if new */
			/* Get name.. Walk list. If match, copy rx/tx values. Else malloc, add to list */
			if(!xmlStrcmp(cur->name, (const xmlChar *) "net")){

				list=ele->properties;
				if(list==NULL) continue;
				network_data_list=malloc(sizeof(network_data_list_t));
				if(network_data_list==NULL) return -1;
				while(list!=NULL){
					tmp=xmlNodeGetContent(list->children);
					if(!xmlStrcmp(list->name, (const xmlChar *) "name")){
						network_data_list->name=strdup(tmp);	
					}
					if(!xmlStrcmp(list->name, (const xmlChar *) "rx")){
						network_data_list->rx=atoll(tmp);
					}
					if(!xmlStrcmp(list->name, (const xmlChar *) "tx")){
						network_data_list->tx=atoll(tmp);
					}
	
					xmlFree(tmp);
					list=list->next;
				}
				if(network_data_list->name==NULL) continue;
				found_host=0;
				ndl_ptr=machine_data_list->network_data_list;
				while(ndl_ptr!=NULL){
					if(!strcmp(ndl_ptr->name, network_data_list->name)){
						found_host=1;
						break;
					}
					ndl_ptr=ndl_ptr->next;	
				}	
				if(found_host){
					ndl_ptr->rx=network_data_list->rx;
					ndl_ptr->tx=network_data_list->tx;
					free(network_data_list->name);
					free(network_data_list);
				}else{
					network_data_list->next=machine_data_list->network_data_list;
					machine_data_list->network_data_list=network_data_list;
				}
			}	

			/* Disk IO stats */
			if(!xmlStrcmp(cur->name, (const xmlChar *) "diskio")){
			
				list=ele->properties;
                                if(list==NULL) continue;
				diskio_data_list=malloc(sizeof(diskio_data_list_t));
				if(diskio_data_list==NULL) return -1;
				while(list!=NULL){
                                        tmp=xmlNodeGetContent(list->children);
					if(!xmlStrcmp(list->name, (const xmlChar *) "name")){
                                                diskio_data_list->name=strdup(tmp);
                                        }
					if(!xmlStrcmp(list->name, (const xmlChar *) "rbytes")){
                                                diskio_data_list->read=atoll(tmp);
                                        }
                                        if(!xmlStrcmp(list->name, (const xmlChar *) "wbytes")){
                                                diskio_data_list->write=atoll(tmp);
                                        }

                                        xmlFree(tmp);
                                        list=list->next;
                                }
				if(diskio_data_list->name==NULL) continue;
                                found_host=0;
				didl_ptr=machine_data_list->diskio_data_list;
				while(didl_ptr!=NULL){
					if(!strcmp(didl_ptr->name, diskio_data_list->name)){
						found_host=1;
						break;
					}
					didl_ptr=didl_ptr->next;
				}
				if(found_host){
					didl_ptr->read=diskio_data_list->read;
					didl_ptr->write=diskio_data_list->write;
					free(diskio_data_list->name);
					free(diskio_data_list);
				}else{
					diskio_data_list->next=machine_data_list->diskio_data_list;
					machine_data_list->diskio_data_list=diskio_data_list;
				}
			}

		

			ele=ele->next;
		}
		cur=cur->next;
	}

	/* Append data we want thats not stored in the server */
	machine_data_list->cpu_used=100.00-machine_data_list->cpu_idle;
	machine_data_list->memory_used_pecent=((double)machine_data_list->memory_used / (double)machine_data_list->memory_total) * 100.00;
	machine_data_list->swap_used_pecent=((double)machine_data_list->swap_used / (double)machine_data_list->swap_total) * 100.00;

	ndl_ptr=machine_data_list->network_data_list;
	machine_data_list->network_io_total_tx=0;
	machine_data_list->network_io_total_rx=0;
	while(ndl_ptr!=NULL){
		machine_data_list->network_io_total_tx+=ndl_ptr->tx;
		machine_data_list->network_io_total_rx+=ndl_ptr->rx;
		ndl_ptr=ndl_ptr->next;
	}
	machine_data_list->network_io_total=machine_data_list->network_io_total_rx+machine_data_list->network_io_total_tx;

	didl_ptr=machine_data_list->diskio_data_list;
	machine_data_list->disk_io_total_read=0;
	machine_data_list->disk_io_total_write=0;
	while(didl_ptr!=NULL){
		machine_data_list->disk_io_total_read+=didl_ptr->read;
		machine_data_list->disk_io_total_write+=didl_ptr->write;
		didl_ptr=didl_ptr->next;
	}
	machine_data_list->disk_io_total=machine_data_list->disk_io_total_read+machine_data_list->disk_io_total_write;

	xmlFreeDoc(doc);
	
	return num_hosts;
	
	
}

void display(machine_data_list_t *machine_data_list, display_config_t *display_config, int *title){
	int line_num=4;
	int counter;
	int x=1;

	if(*title){
		clear();
		move (display_config->maxy-3, 1);
		printw("Sorting by %-64s", display_config->sortby);
		move (display_config->maxy-2, 1);
		if(display_config->units == 'b'){
			printw("Units are measured in bytes/sec");
		}
		if(display_config->units == 'k'){
                        printw("Units are measured in kilobytes/sec");
                }
		if(display_config->units == 'm'){
                        printw("Units are measured in megabytes/sec");
		}

		move(1,1);
		printw("%-11s", "Hostname");
		x=x+11+1;
		if(display_config->cpu_used && (display_config->maxx > x+6)){
			move(1,x);
			printw("%5s", "CPU");
			move(2,x);
			printw("%5s", "used%");
			x+=6;
		}
		if(display_config->load_1 && (display_config->maxx > x+6)){
			move(1,x);
			printw("%5s", "Load");
			move(2,x);
			printw("%5s", "(1m)");
			x+=6;
		}

		if(display_config->pages_in && (display_config->maxx > x+6)){
			move(1,x);
			printw("%5s", "Page");
			move(2,x);
			printw("%5s", "ins");
			x+=6;
		}

		if(display_config->pages_out && (display_config->maxx > x+6)){
			move(1,x);
			printw("%5s", "Page");
			move(2,x);
                        printw("%5s", "outs");
                        x+=6;
		}

		if(display_config->memory_used_pecent && (display_config->maxx > x+6)){
			move(1,x);
			printw("%5s", "Mem");
			move(2,x);
                        printw("%5s", "used");
                        x+=6;
		}

                if(display_config->swap_used_pecent  && (display_config->maxx > x+6)){
			move(1,x);
                        printw("%5s", "Swap");
			move(2,x);
                        printw("%5s", "used");
                        x+=6;
                }

		if(display_config->network_io_total_rx){ 
			if(display_config->units=='b' && (display_config->maxx > x+9)){
				move(1,x);
				printw("%8s", "Net");
				move(2,x);
				printw("%8s", "rx");
				x+=9;
			}

			if(display_config->units=='k' && (display_config->maxx > x+6)){
				move(1,x);
                                printw("%5s", "Net");
                                move(2,x);
                                printw("%5s", "rx");
                                x+=6;
                        }

			if(display_config->units=='m' && (display_config->maxx > x+6)){
                                move(1,x);
                                printw("%5s", "Net");
                                move(2,x);
                                printw("%5s", "rx");
                                x+=6;
                        }

		}

		if(display_config->network_io_total_tx){
                        if(display_config->units=='b' && (display_config->maxx > x+9)){
                                move(1,x);
                                printw("%8s", "Net");
                                move(2,x);
                                printw("%8s", "tx");
                                x+=9;
                        }

                        if(display_config->units=='k' && (display_config->maxx > x+6)){
                                move(1,x);
                                printw("%5s", "Net");
                                move(2,x);
                                printw("%5s", "tx");
                                x+=6;
                        }

                        if(display_config->units=='m' && (display_config->maxx > x+6)){
                                move(1,x);
                                printw("%5s", "Net");
                                move(2,x);
                                printw("%5s", "tx");
                                x+=6;
                        }

                }

		if(display_config->disk_io_total_read){
			if(display_config->units=='b' && (display_config->maxx > x+10)){
				move(1,x);
				printw("%9s", "Disk");
				move(2,x);
				printw("%9s", "read");
				x+=10;
			}

			if(display_config->units=='k' && (display_config->maxx > x+7)){
				move(1,x);
                                printw("%6s", "Disk");
                                move(2,x);
                                printw("%6s", "read");
                                x+=7;
                        }

			if(display_config->units=='m' && (display_config->maxx > x+7)){
                                move(1,x);
                                printw("%6s", "Disk");
                                move(2,x);
                                printw("%6s", "read");
                                x+=7;
                        }

		}

		if(display_config->disk_io_total_read){
                        if(display_config->units=='b' && (display_config->maxx > x+10)){
                                move(1,x);
                                printw("%9s", "Disk");
                                move(2,x);
                                printw("%9s", "write");
                                x+=10;
                        }

                        if(display_config->units=='k' && (display_config->maxx > x+7)){
                                move(1,x);
                                printw("%6s", "Disk");
                                move(2,x);
                                printw("%6s", "write");
                                x+=7;
                        }

                        if(display_config->units=='m' && (display_config->maxx > x+7)){
                                move(1,x);
                                printw("%6s", "Disk");
                                move(2,x);
                                printw("%6s", "write");
                                x+=7;
                        }

                }

		if(display_config->processes && (display_config->maxx > x+25)){
			move(1,x);
			printw("%-24s", "   Number of Process");
			move(2,x);
			printw("%-24s", " Run Slep Zomb Stop  Tot");
			x+=25;
		}

		*title=0;
	}

	/* Counter starts at 8, for padding (eg, headers, borders etc) */
	for(counter=8;counter<display_config->maxy;counter++){	
		if(machine_data_list==NULL) break;
		move(line_num++, 1);
	   	printw("%-11s", machine_data_list->sysname);
		x=13;

		if(display_config->cpu_used && (display_config->maxx > x+6)){
			printw(" %5.1f", machine_data_list->cpu_used);
			x+=6;
		}
		if(display_config->load_1 && (display_config->maxx > x+6)){
			printw(" %5.1f", machine_data_list->load_1);
			x+=6;
		}
		if(display_config->pages_in && (display_config->maxx > x+6)){
			printw(" %5d", machine_data_list->pages_in);
			x+=6;
		}
		if(display_config->pages_out && (display_config->maxx > x+6)){
			printw(" %5d", machine_data_list->pages_out);
			x+=6;
		}
		if(display_config->memory_used_pecent && (display_config->maxx > x+6)){
			printw(" %5.1f", machine_data_list->memory_used_pecent);
			x+=6;
		}
		if(display_config->swap_used_pecent && (display_config->maxx > x+6)){
			printw(" %5.1f", machine_data_list->swap_used_pecent);
			x+=6;
                }

		if(display_config->network_io_total_rx){
			if(display_config->units=='b' && (display_config->maxx > x+9)){
				printw(" %8lld", machine_data_list->network_io_total_rx);
				x+=9;
			}
			if(display_config->units=='k' && (display_config->maxx > x+6)){
                                printw(" %5lld", machine_data_list->network_io_total_rx/1024);
                                x+=6;
                        }
                        if(display_config->units=='m' && (display_config->maxx > x+6)){
                                printw(" %5.2f", (double)(machine_data_list->network_io_total_rx/(1024.00*1024.00)));
                                x+=6;
                        }
		}

		if(display_config->network_io_total_tx){
                        if(display_config->units=='b' && (display_config->maxx > x+9)){
                                printw(" %8lld", machine_data_list->network_io_total_tx);
                                x+=9;
                        }
                        if(display_config->units=='k' && (display_config->maxx > x+6)){
                                printw(" %5lld", machine_data_list->network_io_total_tx/1024);
                                x+=6;
                        }
                        if(display_config->units=='m' && (display_config->maxx > x+6)){
                                printw(" %5.2f", (double)(machine_data_list->network_io_total_tx/(1024.00*1024.00)));
                                x+=6;
                        }
		}

		if(display_config->disk_io_total_read){
			if(display_config->units=='b' && (display_config->maxx > x+10)){
				printw(" %9lld", machine_data_list->disk_io_total_read);
				x+=10;
			}
			if(display_config->units=='k' && (display_config->maxx > x+7)){
                                printw(" %6lld", machine_data_list->disk_io_total_read/1024);
                                x+=7;
			}
			if(display_config->units=='m' && (display_config->maxx > x+7)){
                                printw(" %6.2f", (double)(machine_data_list->disk_io_total_read/(1024.00*1024.00)));
                                x+=7;
                        }
		}

		if(display_config->disk_io_total_write){
                        if(display_config->units=='b' && (display_config->maxx > x+10)){
                                printw(" %9lld", machine_data_list->disk_io_total_write);
                                x+=10;
                        }
                        if(display_config->units=='k' && (display_config->maxx > x+7)){
                                printw(" %6lld", machine_data_list->disk_io_total_write/1024);
                                x+=7;
                        }
                        if(display_config->units=='m' && (display_config->maxx > x+7)){
                                printw(" %6.2f", (double)(machine_data_list->disk_io_total_write/(1024.00*1024.00)));
                                x+=7;
                       	}
		}
		if(display_config->processes && display_config->maxx > x+25){
			printw(" %4d %4d %4d %4d %4d", machine_data_list->processes_cpu, \
				machine_data_list->processes_sleeping, \
				machine_data_list->processes_zombie, \
				machine_data_list->processes_stopped, \
				machine_data_list->processes_total);
			x+=25;
		}

		machine_data_list=machine_data_list->next;
	}


	refresh();
	
}

void sig_winch_handler(int sig){

	sig_winch=1;	
	signal(SIGWINCH, sig_winch_handler);
}

int main(int argc, char **argv){
	WINDOW *window;
	fd_set infds;
	struct winsize size;

	FILE *control;
	FILE *data;

	char *machine_list=NULL;
	char *response=NULL;

	char *servername;
	int server_control_port;
	int server_data_port;

	machine_data_list_t *machine_data_list=NULL;

	int num_hosts;
	int title=1;

	int cmdopt;
        extern int optind;
	extern char *optarg;

	extern int errno;
	
	display_config_t display_config;
	char ch;

	int data_fileno, stdin_fileno, biggest_fileno;

	sortby_ptr=NULL;

	/* What to display defaults */
	display_config.units='b';
	
	display_config.cpu_user=0;
	display_config.cpu_idle=0;
	display_config.cpu_iowait=0;
	display_config.cpu_kernel=0;
	display_config.cpu_swap=0;
	display_config.cpu_used=1; 

	display_config.memory_total=0;
	display_config.memory_free=0;
	display_config.memory_used=0;
	display_config.memory_used_pecent=1;

	display_config.swap_total=0;
	display_config.swap_free=0;
	display_config.swap_used=0;
	display_config.swap_used_pecent=1;

	display_config.load_1=1;
	display_config.load_5=0;
	display_config.load_15=0;

	display_config.pages_in=1;
	display_config.pages_out=1;

	display_config.processes=1;

	display_config.network_io_total_tx=1;
	display_config.network_io_total_rx=1;
	display_config.network_all_stats=0;

	display_config.disk_io_total_write=1;
	display_config.disk_io_total_read=1;
	display_config.disk_io_all_stats=0;

	display_config.disk_total_used=0;
	display_config.disk_all_stats=0;

	signal(SIGWINCH, sig_winch_handler); 

	while((cmdopt=getopt(argc, argv, "s:"))  != -1){
                switch(cmdopt){
			case 's':
				if(!strcmp(optarg, "cpu")){
					sortby_ptr=cmp_cpu_used;
					strlcpy(display_config.sortby, CPU_USED, SORTBYMAXNAME);
				}
				if(!strcmp(optarg, "load")){
					sortby_ptr=cmp_load_1;
					strlcpy(display_config.sortby, LOAD, SORTBYMAXNAME);
				}
				if(!strcmp(optarg, "mem")){
					sortby_ptr=cmp_memory_used_pecent;
					strlcpy(display_config.sortby, MEM, SORTBYMAXNAME);
				}
				if(!strcmp(optarg, "swap")){
                                        sortby_ptr=cmp_swap_used_pecent;
                                        strlcpy(display_config.sortby, SWAP, SORTBYMAXNAME);
                                }
				if(sortby_ptr==NULL){
					errf("Invalid sort type");
					exit(1);
				}
			break;
		}
	}	

	if(sortby_ptr==NULL){
		sortby_ptr=cmp_cpu_used;
		strlcpy(display_config.sortby, "CPU Used", SORTBYMAXNAME);
	}

	if(argc<(optind+2)){
		printf("Usage is %s <-d lines> hostname port <machine list>\n", argv[0]);
		exit(1);
	}

	servername=argv[optind];
	server_control_port=atoi(argv[optind+1]);

	control=create_tcp_connection(servername, server_control_port);
	if(control==NULL){
		errf("Failed to connect (%m)");
		exit(1);
	}

	if(argc==4){
		/* We've been passed a machine list */
		/* list currently needs to be ; seperated */
		machine_list=strdup(argv[3]);
	}	
	
	if((tcp_comm(control, NULL, &response, "PROTOCOL 1.1"))!=0){
		errf("Incorrect version number (%s)", response);
		exit(1);
	}

	if((tcp_comm(control, "stattop", &response, "OK"))!=0){
		errf("Unexpected response %s", response);
		exit(1);
	}

	if(machine_list!=NULL){
		if((tcp_comm(control, "SETHOSTLIST",  &response, "OK"))!=0){
			errf("Unexpected response %s", response);
			exit(1);
		}
		if((tcp_comm(control, machine_list, &response, "OK"))!=0){
                        errf("Unexpected response %s", response);
                        exit(1);
                }
	}

	if((tcp_comm(control, "STARTDATA", &response, NULL))!=0){
		errf("Unexpected response %s", response);
		exit(1);
	}
	
	server_data_port=atoi(response);
	if(server_data_port==0){
		errf("Unexpected response %s", response);
		exit(1);
	}

	data=create_tcp_connection(servername, server_data_port);
	if(data==NULL){
		errf("Failed to connect to host %s on port %d (%m)",servername, server_data_port);
		exit(1);
	}

	initscr();
	nonl();
	cbreak();
	noecho();
	window=newwin(0, 0, 0, 0);
	getmaxyx(window, display_config.maxy, display_config.maxx);

	stdin_fileno=fileno(stdin);
	data_fileno=fileno(data);
	biggest_fileno=(data_fileno>stdin_fileno) ? (data_fileno+1) : (stdin_fileno+1);
		
	for(;;){
		FD_ZERO(&infds);
		FD_SET(stdin_fileno, &infds);
		FD_SET(data_fileno, &infds);
		if((select(biggest_fileno, &infds, NULL, NULL, NULL))==-1){
			if(errno!=EINTR){
				errf("select failed with (%m)");
				exit(1);
			}
		}

		if(sig_winch){
			if (ioctl(fileno(stdout), TIOCGWINSZ, &size) == 0) {
	                	resizeterm(size.ws_row, size.ws_col);
                        	wrefresh(curscr);
			}
			getmaxyx(window, display_config.maxy, display_config.maxx);
			title=1;
			display(machine_data_list, &display_config, &title);
			refresh();
			sig_winch=0;
			continue;
		}
		
		if(FD_ISSET(stdin_fileno, &infds)){
		
			ch=getch();
			switch(ch){
				case KEY_RESIZE:
					sig_winch=1;
					break;						

				/* Quit */
				case 'Q':
				case 'q':
					endwin();
					exit(0);
					break;
				/* Units */
				case 'U':
				case 'u':
					if(display_config.units == 'b'){
						display_config.units = 'k';
					}else if(display_config.units == 'k'){
						display_config.units = 'm';
					}else{
						display_config.units = 'b';
					}
					break;

				/* Sort by */
				case 'C':
					sortby_ptr=cmp_cpu_used;
					strlcpy(display_config.sortby, CPU_USED, SORTBYMAXNAME);
					break;
			
				case 'M':
					sortby_ptr=cmp_memory_used_pecent;
					strlcpy(display_config.sortby, MEM, SORTBYMAXNAME);	
					break;

				case 'L':
					sortby_ptr=cmp_load_1;
					strlcpy(display_config.sortby, LOAD, SORTBYMAXNAME);
					break;			

				case 'S':
					sortby_ptr=cmp_swap_used_pecent;
					strlcpy(display_config.sortby, SWAP, SORTBYMAXNAME);
                                        break;
				
				case 'N':
					if(sortby_ptr==cmp_network_io_total){
						strlcpy(display_config.sortby, NETIORX, SORTBYMAXNAME);
						sortby_ptr=cmp_network_io_total_rx;
					}else if(sortby_ptr==cmp_network_io_total_rx){
						strlcpy(display_config.sortby, NETIOTX, SORTBYMAXNAME);
						sortby_ptr=cmp_network_io_total_tx;
					}else{
						strlcpy(display_config.sortby, NETIO, SORTBYMAXNAME);
						sortby_ptr=cmp_network_io_total;
					}
					break;
				case 'D':
					if(sortby_ptr==cmp_disk_io_total){
						strlcpy(display_config.sortby, DISKIOR, SORTBYMAXNAME);
						sortby_ptr=cmp_disk_io_total_read;
					}else if(sortby_ptr==cmp_disk_io_total_read){
						strlcpy(display_config.sortby, DISKIOW, SORTBYMAXNAME);
						sortby_ptr=cmp_disk_io_total_write;
					}else{
						strlcpy(display_config.sortby, DISKIO, SORTBYMAXNAME);
                                                sortby_ptr=cmp_disk_io_total;
					}
					break;

				/* Display */
			
				case 'd':
					if(display_config.disk_io_total_read){
						display_config.disk_io_total_read=0;
						display_config.disk_io_total_write=0;
					}else{
						display_config.disk_io_total_read=1;
						display_config.disk_io_total_write=1;
					}
					break;	
				case 'n':
					if(display_config.network_io_total_rx){
						display_config.network_io_total_rx=0;
						display_config.network_io_total_tx=0;
					}else{
						display_config.network_io_total_rx=1;
						display_config.network_io_total_tx=1;
					}
					break;
				case 'm':
					if(display_config.memory_used_pecent){
						display_config.memory_used_pecent=0;
					}else{
						display_config.memory_used_pecent=1;
					}
					break;
				
				case 's':
					if(display_config.swap_used_pecent){
                                                display_config.swap_used_pecent=0;
                                        }else{
                                                display_config.swap_used_pecent=1;
                                        }
                                        break;
				case 'l':
					if(display_config.load_1){
						display_config.load_1=0;
					}else{
						display_config.load_1=1;
					}
					break;
				case 'p':
					if(display_config.pages_in){
						display_config.pages_in=0;
						display_config.pages_out=0;
					}else{
						display_config.pages_in=1;
						display_config.pages_out=1;
					}
					break;
				case 'c':
					if(display_config.cpu_used){
						display_config.cpu_used=0;
					}else{
						display_config.cpu_used=1;
					}
					break;
				case 'r':
					if(display_config.processes){
						display_config.processes=0;
					}else{
						display_config.processes=1;
					}
					break;

				default:
					continue;
			}

			/* Increment title so it becomes true (and making the screen update */	
			title++;
			
		}
		if(FD_ISSET(data_fileno, &infds)){
			response=fpgetline(data);
			if (response==NULL){
				errf("Failed to read data (%m)");
				exit(1);
			}
		}


		num_hosts=parse_xml(response, &machine_data_list);
		if(num_hosts==-1) continue;
		machine_data_list=sort_machine_stats(machine_data_list, num_hosts, sortby_ptr);
		display(machine_data_list, &display_config, &title);

	}
	exit(0);
}
