/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.azure.eventprocessorhost;

import com.microsoft.azure.eventprocessorhost.BaseLease;
import com.microsoft.azure.eventprocessorhost.CompleteLease;
import com.microsoft.azure.eventprocessorhost.HostContext;
import com.microsoft.azure.eventprocessorhost.ILeaseManager;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ConcurrentHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InMemoryLeaseManager
implements ILeaseManager {
    private static final Logger TRACE_LOGGER = LoggerFactory.getLogger(InMemoryLeaseManager.class);
    private HostContext hostContext;
    private long millisecondsLatency = 0L;

    public void initialize(HostContext hostContext) {
        this.hostContext = hostContext;
    }

    public void setLatency(long milliseconds) {
        this.millisecondsLatency = milliseconds;
    }

    private void latency(String caller) {
        if (this.millisecondsLatency > 0L) {
            try {
                Thread.sleep(this.millisecondsLatency);
            }
            catch (InterruptedException e) {
                TRACE_LOGGER.info("sleepFAIL " + caller);
            }
        }
    }

    @Override
    public int getLeaseDurationInMilliseconds() {
        return this.hostContext.getPartitionManagerOptions().getLeaseDurationInSeconds() * 1000;
    }

    @Override
    public CompletableFuture<Boolean> leaseStoreExists() {
        boolean exists = InMemoryLeaseStore.singleton.existsMap();
        this.latency("leaseStoreExists");
        TRACE_LOGGER.debug(this.hostContext.withHost("leaseStoreExists() " + exists));
        return CompletableFuture.completedFuture(exists);
    }

    @Override
    public CompletableFuture<Void> createLeaseStoreIfNotExists() {
        TRACE_LOGGER.debug(this.hostContext.withHost("createLeaseStoreIfNotExists()"));
        InMemoryLeaseStore.singleton.initializeMap(this.getLeaseDurationInMilliseconds());
        this.latency("createLeaseStoreIfNotExists");
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Void> deleteLeaseStore() {
        TRACE_LOGGER.debug(this.hostContext.withHost("deleteLeaseStore()"));
        InMemoryLeaseStore.singleton.deleteMap();
        this.latency("deleteLeaseStore");
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<CompleteLease> getLease(String partitionId) {
        TRACE_LOGGER.debug(this.hostContext.withHost("getLease()"));
        this.latency("getLease");
        InMemoryLease leaseInStore = InMemoryLeaseStore.singleton.getLease(partitionId);
        return CompletableFuture.completedFuture(new InMemoryLease(leaseInStore));
    }

    @Override
    public CompletableFuture<List<BaseLease>> getAllLeases() {
        ArrayList<BaseLease> infos = new ArrayList<BaseLease>();
        for (String id : InMemoryLeaseStore.singleton.getPartitionIds()) {
            InMemoryLease leaseInStore = InMemoryLeaseStore.singleton.getLease(id);
            infos.add(new BaseLease(id, leaseInStore.getOwner(), !leaseInStore.isExpiredSync()));
        }
        this.latency("getAllLeasesStateInfo");
        return CompletableFuture.completedFuture(infos);
    }

    @Override
    public CompletableFuture<Void> createAllLeasesIfNotExists(List<String> partitionIds) {
        ArrayList<CompletableFuture<BaseLease>> createFutures = new ArrayList<CompletableFuture<BaseLease>>();
        Iterator<String> iterator = partitionIds.iterator();
        while (iterator.hasNext()) {
            String id;
            String workingId = id = iterator.next();
            CompletableFuture<BaseLease> oneCreate = CompletableFuture.supplyAsync(() -> {
                InMemoryLease leaseInStore = InMemoryLeaseStore.singleton.getLease(workingId);
                InMemoryLease returnLease = null;
                if (leaseInStore != null) {
                    TRACE_LOGGER.debug(this.hostContext.withHostAndPartition(workingId, "createLeaseIfNotExists() found existing lease, OK"));
                    returnLease = new InMemoryLease(leaseInStore);
                } else {
                    TRACE_LOGGER.debug(this.hostContext.withHostAndPartition(workingId, "createLeaseIfNotExists() creating new lease"));
                    InMemoryLease newStoreLease = new InMemoryLease(workingId);
                    InMemoryLeaseStore.singleton.setOrReplaceLease(newStoreLease);
                    returnLease = new InMemoryLease(newStoreLease);
                }
                this.latency("createLeaseIfNotExists " + workingId);
                return returnLease;
            }, this.hostContext.getExecutor());
            createFutures.add(oneCreate);
        }
        CompletableFuture[] dummy = new CompletableFuture[createFutures.size()];
        return CompletableFuture.allOf(createFutures.toArray(dummy));
    }

    @Override
    public CompletableFuture<Void> deleteLease(CompleteLease lease) {
        TRACE_LOGGER.debug(this.hostContext.withHostAndPartition(lease, "deleteLease()"));
        InMemoryLeaseStore.singleton.removeLease((InMemoryLease)lease);
        this.latency("deleteLease " + lease.getPartitionId());
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Boolean> acquireLease(CompleteLease lease) {
        InMemoryLease leaseToAcquire = (InMemoryLease)lease;
        TRACE_LOGGER.debug(this.hostContext.withHostAndPartition(leaseToAcquire, "acquireLease()"));
        boolean retval = true;
        InMemoryLease leaseInStore = InMemoryLeaseStore.singleton.getLease(leaseToAcquire.getPartitionId());
        if (leaseInStore != null) {
            InMemoryLease wasUnowned = InMemoryLeaseStore.singleton.atomicAquireUnowned(leaseToAcquire.getPartitionId(), this.hostContext.getHostName());
            if (wasUnowned != null) {
                leaseToAcquire.setOwner(this.hostContext.getHostName());
                TRACE_LOGGER.debug(this.hostContext.withHostAndPartition(leaseToAcquire, "acquireLease() acquired lease"));
                leaseInStore = wasUnowned;
                leaseToAcquire.setExpirationTime(leaseInStore.getExpirationTime());
            } else {
                if (leaseInStore.isOwnedBy(this.hostContext.getHostName())) {
                    TRACE_LOGGER.debug(this.hostContext.withHostAndPartition(leaseToAcquire, "acquireLease() already hold lease"));
                } else {
                    String oldOwner = leaseInStore.getOwner();
                    InMemoryLeaseStore.singleton.stealLease(leaseInStore, this.hostContext.getHostName());
                    leaseToAcquire.setOwner(this.hostContext.getHostName());
                    TRACE_LOGGER.debug(this.hostContext.withHostAndPartition(leaseToAcquire, "acquireLease() stole lease from " + oldOwner));
                }
                long newExpiration = System.currentTimeMillis() + (long)this.getLeaseDurationInMilliseconds();
                leaseInStore.setExpirationTime(newExpiration);
                leaseToAcquire.setExpirationTime(newExpiration);
            }
        } else {
            TRACE_LOGGER.warn(this.hostContext.withHostAndPartition(leaseToAcquire, "acquireLease() can't find lease"));
            retval = false;
        }
        this.latency("acquireLease " + lease.getPartitionId());
        return CompletableFuture.completedFuture(retval);
    }

    public void notifyOnSteal(String expectedOwner, String partitionId, Callable<?> notifier) {
        InMemoryLeaseStore.singleton.notifyOnSteal(expectedOwner, partitionId, notifier);
    }

    @Override
    public CompletableFuture<Boolean> renewLease(CompleteLease lease) {
        InMemoryLease leaseToRenew = (InMemoryLease)lease;
        TRACE_LOGGER.debug(this.hostContext.withHostAndPartition(leaseToRenew, "renewLease()"));
        boolean retval = true;
        InMemoryLease leaseInStore = InMemoryLeaseStore.singleton.getLease(leaseToRenew.getPartitionId());
        if (leaseInStore != null) {
            if (leaseInStore.isOwnedBy(this.hostContext.getHostName())) {
                long newExpiration = System.currentTimeMillis() + (long)this.getLeaseDurationInMilliseconds();
                leaseInStore.setExpirationTime(newExpiration);
                leaseToRenew.setExpirationTime(newExpiration);
            } else {
                TRACE_LOGGER.debug(this.hostContext.withHostAndPartition(leaseToRenew, "renewLease() not renewed because we don't own lease"));
                retval = false;
            }
        } else {
            TRACE_LOGGER.warn(this.hostContext.withHostAndPartition(leaseToRenew, "renewLease() can't find lease"));
            retval = false;
        }
        this.latency("renewLease " + lease.getPartitionId());
        return CompletableFuture.completedFuture(retval);
    }

    @Override
    public CompletableFuture<Void> releaseLease(CompleteLease lease) {
        InMemoryLease leaseToRelease = (InMemoryLease)lease;
        CompletableFuture<Object> retval = CompletableFuture.completedFuture(null);
        TRACE_LOGGER.debug(this.hostContext.withHostAndPartition(leaseToRelease, "releaseLease()"));
        InMemoryLease leaseInStore = InMemoryLeaseStore.singleton.getLease(leaseToRelease.getPartitionId());
        if (leaseInStore != null) {
            if (!leaseInStore.isExpiredSync() && leaseInStore.isOwnedBy(this.hostContext.getHostName())) {
                TRACE_LOGGER.debug(this.hostContext.withHostAndPartition(leaseToRelease, "releaseLease() released OK"));
                leaseInStore.setOwner("");
                leaseToRelease.setOwner("");
                leaseInStore.setExpirationTime(0L);
                leaseToRelease.setExpirationTime(0L);
            }
        } else {
            TRACE_LOGGER.warn(this.hostContext.withHostAndPartition(leaseToRelease, "releaseLease() can't find lease in store"));
            retval = new CompletableFuture();
            retval.completeExceptionally(new CompletionException(new RuntimeException("releaseLease can't find lease in store for " + leaseToRelease.getPartitionId())));
        }
        this.latency("releaseLease " + lease.getPartitionId());
        return retval;
    }

    @Override
    public CompletableFuture<Boolean> updateLease(CompleteLease lease) {
        InMemoryLease leaseToUpdate = (InMemoryLease)lease;
        TRACE_LOGGER.debug(this.hostContext.withHostAndPartition(leaseToUpdate, "updateLease()"));
        return this.renewLease(leaseToUpdate).thenApply(retval -> {
            if (retval.booleanValue()) {
                InMemoryLease leaseInStore = InMemoryLeaseStore.singleton.getLease(leaseToUpdate.getPartitionId());
                if (leaseInStore != null) {
                    if (!leaseInStore.isExpiredSync() && leaseInStore.isOwnedBy(this.hostContext.getHostName())) {
                        leaseInStore.setEpoch(leaseToUpdate.getEpoch());
                    } else {
                        TRACE_LOGGER.debug(this.hostContext.withHostAndPartition(leaseToUpdate, "updateLease() not updated because we don't own lease"));
                        retval = false;
                    }
                } else {
                    TRACE_LOGGER.warn(this.hostContext.withHostAndPartition(leaseToUpdate, "updateLease() can't find lease"));
                    retval = false;
                }
            }
            this.latency("updateLease " + lease.getPartitionId());
            return retval;
        });
    }

    private static class InMemoryLease
    extends CompleteLease {
        private static final Logger TRACE_LOGGER = LoggerFactory.getLogger(InMemoryLease.class);
        private long expirationTimeMillis = 0L;

        InMemoryLease(String partitionId) {
            super(partitionId);
            this.epoch = 0L;
        }

        InMemoryLease(InMemoryLease source) {
            super(source);
            this.expirationTimeMillis = source.expirationTimeMillis;
            this.epoch = source.epoch;
        }

        long getExpirationTime() {
            return this.expirationTimeMillis;
        }

        void setExpirationTime(long expireAtMillis) {
            this.expirationTimeMillis = expireAtMillis;
        }

        public boolean isExpiredSync() {
            boolean hasExpired;
            boolean bl = hasExpired = System.currentTimeMillis() >= this.expirationTimeMillis;
            if (hasExpired) {
                // empty if block
            }
            TRACE_LOGGER.debug("isExpired(" + this.getPartitionId() + (hasExpired ? ") expired " : ") leased ") + (this.expirationTimeMillis - System.currentTimeMillis()));
            return hasExpired;
        }
    }

    private static class InMemoryLeaseStore {
        static final InMemoryLeaseStore singleton = new InMemoryLeaseStore();
        private static int leaseDurationInMilliseconds;
        private ConcurrentHashMap<String, InMemoryLease> inMemoryLeasesPrivate = null;
        private ConcurrentHashMap<String, Callable<?>> notifiers = new ConcurrentHashMap();

        private InMemoryLeaseStore() {
        }

        synchronized boolean existsMap() {
            return this.inMemoryLeasesPrivate != null;
        }

        synchronized void initializeMap(int leaseDurationInMilliseconds) {
            if (this.inMemoryLeasesPrivate == null) {
                this.inMemoryLeasesPrivate = new ConcurrentHashMap();
            }
            InMemoryLeaseStore.leaseDurationInMilliseconds = leaseDurationInMilliseconds;
        }

        synchronized void deleteMap() {
            this.inMemoryLeasesPrivate = null;
        }

        synchronized InMemoryLease getLease(String partitionId) {
            return this.inMemoryLeasesPrivate.get(partitionId);
        }

        synchronized List<String> getPartitionIds() {
            ArrayList<String> ids = new ArrayList<String>();
            ((ConcurrentHashMap.KeySetView)this.inMemoryLeasesPrivate.keySet()).forEach(key -> ids.add((String)key));
            return ids;
        }

        synchronized InMemoryLease atomicAquireUnowned(String partitionId, String newOwner) {
            InMemoryLease leaseInStore = this.getLease(partitionId);
            if (leaseInStore.isExpiredSync() || leaseInStore.getOwner() == null || leaseInStore.getOwner().isEmpty()) {
                leaseInStore.setOwner(newOwner);
                leaseInStore.setExpirationTime(System.currentTimeMillis() + (long)leaseDurationInMilliseconds);
            } else {
                leaseInStore = null;
            }
            return leaseInStore;
        }

        synchronized void notifyOnSteal(String expectedOwner, String partitionId, Callable<?> notifier) {
            InMemoryLease leaseInStore = this.getLease(partitionId);
            if (!leaseInStore.isOwnedBy(expectedOwner)) {
                try {
                    notifier.call();
                }
                catch (Exception exception) {}
            } else {
                this.notifiers.put(partitionId, notifier);
            }
        }

        synchronized void stealLease(InMemoryLease stealee, String newOwner) {
            stealee.setOwner(newOwner);
            Callable<?> notifier = this.notifiers.get(stealee.getPartitionId());
            if (notifier != null) {
                try {
                    notifier.call();
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
        }

        synchronized void setOrReplaceLease(InMemoryLease newLease) {
            this.inMemoryLeasesPrivate.put(newLease.getPartitionId(), newLease);
        }

        synchronized void removeLease(InMemoryLease goneLease) {
            this.inMemoryLeasesPrivate.remove(goneLease.getPartitionId());
        }
    }
}

