/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.index.translog.transfer;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.OutputStreamIndexOutput;
import org.opensearch.action.LatchedActionListener;
import org.opensearch.common.SetOnce;
import org.opensearch.common.blobstore.BlobMetadata;
import org.opensearch.common.blobstore.BlobPath;
import org.opensearch.common.blobstore.stream.write.WritePriority;
import org.opensearch.common.io.VersionedCodecStreamWrapper;
import org.opensearch.common.io.stream.BytesStreamOutput;
import org.opensearch.common.logging.Loggers;
import org.opensearch.common.lucene.store.ByteArrayIndexInput;
import org.opensearch.core.action.ActionListener;
import org.opensearch.core.common.bytes.BytesReference;
import org.opensearch.core.index.shard.ShardId;
import org.opensearch.index.remote.RemoteStoreUtils;
import org.opensearch.index.remote.RemoteTranslogTransferTracker;
import org.opensearch.index.translog.Translog;
import org.opensearch.index.translog.transfer.FileSnapshot;
import org.opensearch.index.translog.transfer.FileTransferException;
import org.opensearch.index.translog.transfer.FileTransferTracker;
import org.opensearch.index.translog.transfer.TransferService;
import org.opensearch.index.translog.transfer.TransferSnapshot;
import org.opensearch.index.translog.transfer.TranslogTransferMetadata;
import org.opensearch.index.translog.transfer.TranslogTransferMetadataHandler;
import org.opensearch.index.translog.transfer.listener.TranslogTransferListener;

public class TranslogTransferManager {
    private final ShardId shardId;
    private final TransferService transferService;
    private final BlobPath remoteDataTransferPath;
    private final BlobPath remoteMetadataTransferPath;
    private final BlobPath remoteBaseTransferPath;
    private final FileTransferTracker fileTransferTracker;
    private final RemoteTranslogTransferTracker remoteTranslogTransferTracker;
    private static final long TRANSFER_TIMEOUT_IN_MILLIS = 30000L;
    private static final int METADATA_FILES_TO_FETCH = 10;
    private final Logger logger;
    private static final String METADATA_DIR = "metadata";
    private static final String DATA_DIR = "data";
    private static final VersionedCodecStreamWrapper<TranslogTransferMetadata> metadataStreamWrapper = new VersionedCodecStreamWrapper<TranslogTransferMetadata>(new TranslogTransferMetadataHandler(), 1, "md");

    public TranslogTransferManager(ShardId shardId, TransferService transferService, BlobPath remoteBaseTransferPath, FileTransferTracker fileTransferTracker, RemoteTranslogTransferTracker remoteTranslogTransferTracker) {
        this.shardId = shardId;
        this.transferService = transferService;
        this.remoteBaseTransferPath = remoteBaseTransferPath;
        this.remoteDataTransferPath = remoteBaseTransferPath.add(DATA_DIR);
        this.remoteMetadataTransferPath = remoteBaseTransferPath.add(METADATA_DIR);
        this.fileTransferTracker = fileTransferTracker;
        this.logger = Loggers.getLogger(this.getClass(), shardId, new String[0]);
        this.remoteTranslogTransferTracker = remoteTranslogTransferTracker;
    }

    public RemoteTranslogTransferTracker getRemoteTranslogTransferTracker() {
        return this.remoteTranslogTransferTracker;
    }

    public ShardId getShardId() {
        return this.shardId;
    }

    public boolean transferSnapshot(TransferSnapshot transferSnapshot, TranslogTransferListener translogTransferListener) throws IOException {
        ArrayList exceptionList = new ArrayList(transferSnapshot.getTranslogTransferMetadata().getCount());
        HashSet<FileSnapshot.TransferFileSnapshot> toUpload = new HashSet<FileSnapshot.TransferFileSnapshot>(transferSnapshot.getTranslogTransferMetadata().getCount());
        long prevUploadBytesSucceeded = this.remoteTranslogTransferTracker.getUploadBytesSucceeded();
        long prevUploadTimeInMillis = this.remoteTranslogTransferTracker.getTotalUploadTimeInMillis();
        try {
            toUpload.addAll(this.fileTransferTracker.exclusionFilter(transferSnapshot.getTranslogFileSnapshots()));
            toUpload.addAll(this.fileTransferTracker.exclusionFilter(transferSnapshot.getCheckpointFileSnapshots()));
            if (toUpload.isEmpty()) {
                this.logger.trace("Nothing to upload for transfer");
                return true;
            }
            this.fileTransferTracker.recordBytesForFiles(toUpload);
            this.captureStatsBeforeUpload();
            CountDownLatch latch = new CountDownLatch(toUpload.size());
            LatchedActionListener<FileSnapshot.TransferFileSnapshot> latchedActionListener = new LatchedActionListener<FileSnapshot.TransferFileSnapshot>(ActionListener.wrap(this.fileTransferTracker::onSuccess, ex -> {
                assert (ex instanceof FileTransferException);
                this.logger.error(() -> new ParameterizedMessage("Exception during transfer for file {}", (Object)((FileTransferException)ex).getFileSnapshot().getName()), (Throwable)ex);
                FileTransferException e = (FileTransferException)ex;
                FileSnapshot.TransferFileSnapshot file = e.getFileSnapshot();
                this.fileTransferTracker.onFailure(file, (Exception)ex);
                exceptionList.add(ex);
            }), latch);
            HashMap<Long, BlobPath> blobPathMap = new HashMap<Long, BlobPath>();
            toUpload.forEach(fileSnapshot -> blobPathMap.put(fileSnapshot.getPrimaryTerm(), this.remoteDataTransferPath.add(String.valueOf(fileSnapshot.getPrimaryTerm()))));
            long uploadStartTime = System.nanoTime();
            this.fileTransferTracker.recordFileTransferStartTime(uploadStartTime);
            this.transferService.uploadBlobs(toUpload, blobPathMap, latchedActionListener, WritePriority.HIGH);
            try {
                if (!latch.await(30000L, TimeUnit.MILLISECONDS)) {
                    TimeoutException ex2 = new TimeoutException("Timed out waiting for transfer of snapshot " + transferSnapshot + " to complete");
                    exceptionList.forEach(ex2::addSuppressed);
                    throw ex2;
                }
            }
            catch (InterruptedException ex3) {
                exceptionList.forEach(ex3::addSuppressed);
                Thread.currentThread().interrupt();
                throw ex3;
            }
            if (exceptionList.isEmpty()) {
                FileSnapshot.TransferFileSnapshot tlogMetadata = this.prepareMetadata(transferSnapshot);
                long metadataBytesToUpload = tlogMetadata.getContentLength();
                this.remoteTranslogTransferTracker.addUploadBytesStarted(metadataBytesToUpload);
                long metadataUploadStartTime = System.nanoTime();
                try {
                    this.transferService.uploadBlob(tlogMetadata, this.remoteMetadataTransferPath, WritePriority.HIGH);
                }
                catch (Exception exception) {
                    this.remoteTranslogTransferTracker.addUploadTimeInMillis((System.nanoTime() - metadataUploadStartTime) / 1000000L);
                    this.remoteTranslogTransferTracker.addUploadBytesFailed(metadataBytesToUpload);
                    throw exception;
                }
                this.remoteTranslogTransferTracker.addUploadTimeInMillis((System.nanoTime() - metadataUploadStartTime) / 1000000L);
                this.remoteTranslogTransferTracker.addUploadBytesSucceeded(metadataBytesToUpload);
                this.captureStatsOnUploadSuccess(prevUploadBytesSucceeded, prevUploadTimeInMillis);
                translogTransferListener.onUploadComplete(transferSnapshot);
                return true;
            }
            IOException ex4 = new IOException("Failed to upload " + exceptionList.size() + " files during transfer");
            exceptionList.forEach(ex4::addSuppressed);
            throw ex4;
        }
        catch (Exception ex5) {
            this.logger.error(() -> new ParameterizedMessage("Transfer failed for snapshot {}", (Object)transferSnapshot), (Throwable)ex5);
            this.captureStatsOnUploadFailure();
            translogTransferListener.onUploadFailed(transferSnapshot, ex5);
            return false;
        }
    }

    private void captureStatsBeforeUpload() {
        this.remoteTranslogTransferTracker.incrementTotalUploadsStarted();
        this.remoteTranslogTransferTracker.addUploadBytesStarted(this.fileTransferTracker.getTotalBytesToUpload());
    }

    private void captureStatsOnUploadSuccess(long prevUploadBytesSucceeded, long prevUploadTimeInMillis) {
        this.remoteTranslogTransferTracker.setLastSuccessfulUploadTimestamp(System.currentTimeMillis());
        this.remoteTranslogTransferTracker.incrementTotalUploadsSucceeded();
        long totalUploadedBytes = this.remoteTranslogTransferTracker.getUploadBytesSucceeded() - prevUploadBytesSucceeded;
        this.remoteTranslogTransferTracker.updateUploadBytesMovingAverage(totalUploadedBytes);
        long uploadDurationInMillis = this.remoteTranslogTransferTracker.getTotalUploadTimeInMillis() - prevUploadTimeInMillis;
        this.remoteTranslogTransferTracker.updateUploadTimeMovingAverage(uploadDurationInMillis);
        if (uploadDurationInMillis > 0L) {
            this.remoteTranslogTransferTracker.updateUploadBytesPerSecMovingAverage(totalUploadedBytes * 1000L / uploadDurationInMillis);
        }
    }

    private void captureStatsOnUploadFailure() {
        this.remoteTranslogTransferTracker.incrementTotalUploadsFailed();
    }

    public boolean downloadTranslog(String primaryTerm, String generation, Path location) throws IOException {
        this.logger.trace("Downloading translog files with: Primary Term = {}, Generation = {}, Location = {}", (Object)primaryTerm, (Object)generation, (Object)location);
        String ckpFileName = Translog.getCommitCheckpointFileName(Long.parseLong(generation));
        this.downloadToFS(ckpFileName, location, primaryTerm);
        String translogFilename = Translog.getFilename(Long.parseLong(generation));
        this.downloadToFS(translogFilename, location, primaryTerm);
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void downloadToFS(String fileName, Path location, String primaryTerm) throws IOException {
        Path filePath = location.resolve(fileName);
        if (Files.exists(filePath, new LinkOption[0])) {
            Files.delete(filePath);
        }
        boolean downloadStatus = false;
        long bytesToRead = 0L;
        long downloadStartTime = System.nanoTime();
        try (InputStream inputStream = this.transferService.downloadBlob(this.remoteDataTransferPath.add(primaryTerm), fileName);){
            bytesToRead = inputStream.available();
            Files.copy(inputStream, filePath, new CopyOption[0]);
            downloadStatus = true;
        }
        finally {
            this.remoteTranslogTransferTracker.addDownloadTimeInMillis((System.nanoTime() - downloadStartTime) / 1000000L);
            if (downloadStatus) {
                this.remoteTranslogTransferTracker.addDownloadBytesSucceeded(bytesToRead);
            }
        }
        this.fileTransferTracker.add(fileName, true);
    }

    public TranslogTransferMetadata readMetadata() throws IOException {
        SetOnce metadataSetOnce = new SetOnce();
        SetOnce exceptionSetOnce = new SetOnce();
        CountDownLatch latch = new CountDownLatch(1);
        LatchedActionListener<List<BlobMetadata>> latchedActionListener = new LatchedActionListener<List<BlobMetadata>>(ActionListener.wrap(blobMetadataList -> {
            if (blobMetadataList.isEmpty()) {
                return;
            }
            RemoteStoreUtils.verifyNoMultipleWriters(blobMetadataList.stream().map(BlobMetadata::name).collect(Collectors.toList()), TranslogTransferMetadata::getNodeIdByPrimaryTermAndGen);
            String filename = ((BlobMetadata)blobMetadataList.get(0)).name();
            boolean downloadStatus = false;
            long downloadStartTime = System.nanoTime();
            long bytesToRead = 0L;
            try (InputStream inputStream = this.transferService.downloadBlob(this.remoteMetadataTransferPath, filename);){
                bytesToRead = inputStream.available();
                ByteArrayIndexInput indexInput = new ByteArrayIndexInput("metadata file", inputStream.readAllBytes());
                metadataSetOnce.set((Object)metadataStreamWrapper.readStream(indexInput));
                downloadStatus = true;
            }
            catch (IOException e) {
                this.logger.error(() -> new ParameterizedMessage("Exception while reading metadata file: {}", (Object)filename), (Throwable)e);
                exceptionSetOnce.set((Object)e);
            }
            finally {
                this.remoteTranslogTransferTracker.addDownloadTimeInMillis((System.nanoTime() - downloadStartTime) / 1000000L);
                this.logger.debug("translogMetadataDownloadStatus={}", (Object)downloadStatus);
                if (downloadStatus) {
                    this.remoteTranslogTransferTracker.addDownloadBytesSucceeded(bytesToRead);
                }
            }
        }, e -> {
            if (e instanceof RuntimeException) {
                throw (RuntimeException)e;
            }
            this.logger.error(() -> new ParameterizedMessage("Exception while listing metadata files", new Object[0]), (Throwable)e);
            exceptionSetOnce.set((Object)((IOException)e));
        }), latch);
        try {
            this.transferService.listAllInSortedOrder(this.remoteMetadataTransferPath, METADATA_DIR, 10, latchedActionListener);
            latch.await();
        }
        catch (InterruptedException e2) {
            throw new IOException("Exception while reading/downloading metadafile", e2);
        }
        if (exceptionSetOnce.get() != null) {
            throw (IOException)exceptionSetOnce.get();
        }
        return (TranslogTransferMetadata)metadataSetOnce.get();
    }

    private FileSnapshot.TransferFileSnapshot prepareMetadata(TransferSnapshot transferSnapshot) throws IOException {
        Map<String, String> generationPrimaryTermMap = transferSnapshot.getTranslogFileSnapshots().stream().map(s -> {
            assert (s instanceof FileSnapshot.TranslogFileSnapshot);
            return (FileSnapshot.TranslogFileSnapshot)s;
        }).collect(Collectors.toMap(snapshot -> String.valueOf(snapshot.getGeneration()), snapshot -> String.valueOf(snapshot.getPrimaryTerm())));
        TranslogTransferMetadata translogTransferMetadata = transferSnapshot.getTranslogTransferMetadata();
        translogTransferMetadata.setGenerationToPrimaryTermMapper(new HashMap<String, String>(generationPrimaryTermMap));
        return new FileSnapshot.TransferFileSnapshot(translogTransferMetadata.getFileName(), this.getMetadataBytes(translogTransferMetadata), translogTransferMetadata.getPrimaryTerm());
    }

    public byte[] getMetadataBytes(TranslogTransferMetadata metadata) throws IOException {
        byte[] metadataBytes;
        try (BytesStreamOutput output = new BytesStreamOutput();){
            try (OutputStreamIndexOutput indexOutput = new OutputStreamIndexOutput("translog transfer metadata " + metadata.getPrimaryTerm(), metadata.getFileName(), (OutputStream)((Object)output), 4096);){
                metadataStreamWrapper.writeStream((IndexOutput)indexOutput, metadata);
            }
            metadataBytes = BytesReference.toBytes((BytesReference)output.bytes());
        }
        return metadataBytes;
    }

    public void deleteGenerationAsync(long primaryTerm, Set<Long> generations, Runnable onCompletion) {
        ArrayList<String> translogFiles = new ArrayList<String>();
        generations.forEach(generation -> {
            String ckpFileName = Translog.getCommitCheckpointFileName(generation);
            String translogFileName = Translog.getFilename(generation);
            translogFiles.addAll(List.of(ckpFileName, translogFileName));
        });
        this.deleteTranslogFilesAsync(primaryTerm, translogFiles, onCompletion);
    }

    public void deletePrimaryTermsAsync(final long minPrimaryTermToKeep) {
        this.logger.info("Deleting primary terms from remote store lesser than {}", (Object)minPrimaryTermToKeep);
        this.transferService.listFoldersAsync("remote_purge", this.remoteDataTransferPath, new ActionListener<Set<String>>(){

            public void onResponse(Set<String> folders) {
                Set primaryTermsInRemote = folders.stream().filter(folderName -> {
                    try {
                        Long.parseLong(folderName);
                        return true;
                    }
                    catch (Exception exception) {
                        return false;
                    }
                }).map(Long::parseLong).collect(Collectors.toSet());
                Set<Long> primaryTermsToDelete = primaryTermsInRemote.stream().filter(term -> term < minPrimaryTermToKeep).collect(Collectors.toSet());
                primaryTermsToDelete.forEach(term -> TranslogTransferManager.this.deletePrimaryTermAsync((long)term));
            }

            public void onFailure(Exception e) {
                TranslogTransferManager.this.logger.error("Exception occurred while getting primary terms from remote store", (Throwable)e);
            }
        });
    }

    private void deletePrimaryTermAsync(final long primaryTerm) {
        this.transferService.deleteAsync("remote_purge", this.remoteDataTransferPath.add(String.valueOf(primaryTerm)), new ActionListener<Void>(){

            public void onResponse(Void unused) {
                TranslogTransferManager.this.logger.info("Deleted primary term {}", (Object)primaryTerm);
            }

            public void onFailure(Exception e) {
                TranslogTransferManager.this.logger.error((Message)new ParameterizedMessage("Exception occurred while deleting primary term {}", (Object)primaryTerm), (Throwable)e);
            }
        });
    }

    public void delete() {
        this.transferService.deleteAsync("remote_purge", this.remoteBaseTransferPath, new ActionListener<Void>(){

            public void onResponse(Void unused) {
                TranslogTransferManager.this.logger.info("Deleted all remote translog data");
            }

            public void onFailure(Exception e) {
                TranslogTransferManager.this.logger.error("Exception occurred while cleaning translog", (Throwable)e);
            }
        });
    }

    public void deleteStaleTranslogMetadataFilesAsync(final Runnable onCompletion) {
        try {
            this.transferService.listAllInSortedOrderAsync("remote_purge", this.remoteMetadataTransferPath, METADATA_DIR, Integer.MAX_VALUE, new ActionListener<List<BlobMetadata>>(){

                public void onResponse(List<BlobMetadata> blobMetadata) {
                    List sortedMetadataFiles = blobMetadata.stream().map(BlobMetadata::name).collect(Collectors.toList());
                    if (sortedMetadataFiles.size() <= 1) {
                        TranslogTransferManager.this.logger.trace("Remote Metadata file count is {}, so skipping deletion", (Object)sortedMetadataFiles.size());
                        onCompletion.run();
                        return;
                    }
                    List<String> metadataFilesToDelete = sortedMetadataFiles.subList(1, sortedMetadataFiles.size());
                    TranslogTransferManager.this.logger.trace("Deleting remote translog metadata files {}", metadataFilesToDelete);
                    TranslogTransferManager.this.deleteMetadataFilesAsync(metadataFilesToDelete, onCompletion);
                }

                public void onFailure(Exception e) {
                    TranslogTransferManager.this.logger.error("Exception occurred while listing translog metadata files from remote store", (Throwable)e);
                    onCompletion.run();
                }
            });
        }
        catch (Exception e) {
            this.logger.error("Exception occurred while listing translog metadata files from remote store", (Throwable)e);
            onCompletion.run();
        }
    }

    public void deleteTranslogFiles() throws IOException {
        this.transferService.delete(this.remoteMetadataTransferPath);
        this.transferService.delete(this.remoteDataTransferPath);
    }

    private void deleteTranslogFilesAsync(final long primaryTerm, final List<String> files, final Runnable onCompletion) {
        try {
            this.transferService.deleteBlobsAsync("remote_purge", this.remoteDataTransferPath.add(String.valueOf(primaryTerm)), files, new ActionListener<Void>(){

                public void onResponse(Void unused) {
                    TranslogTransferManager.this.fileTransferTracker.delete(files);
                    TranslogTransferManager.this.logger.trace("Deleted translogs for primaryTerm={} files={}", (Object)primaryTerm, (Object)files);
                    onCompletion.run();
                }

                public void onFailure(Exception e) {
                    onCompletion.run();
                    TranslogTransferManager.this.logger.error(() -> new ParameterizedMessage("Exception occurred while deleting translog for primaryTerm={} files={}", (Object)primaryTerm, (Object)files), (Throwable)e);
                }
            });
        }
        catch (Exception e) {
            onCompletion.run();
            throw e;
        }
    }

    private void deleteMetadataFilesAsync(final List<String> files, final Runnable onCompletion) {
        try {
            this.transferService.deleteBlobsAsync("remote_purge", this.remoteMetadataTransferPath, files, new ActionListener<Void>(){

                public void onResponse(Void unused) {
                    onCompletion.run();
                    TranslogTransferManager.this.logger.trace("Deleted remote translog metadata files {}", (Object)files);
                }

                public void onFailure(Exception e) {
                    onCompletion.run();
                    TranslogTransferManager.this.logger.error((Message)new ParameterizedMessage("Exception occurred while deleting remote translog metadata files {}", (Object)files), (Throwable)e);
                }
            });
        }
        catch (Exception e) {
            onCompletion.run();
            throw e;
        }
    }
}

