"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BasicAuthRoutes = void 0;

var _configSchema = require("@osd/config-schema");

var _security_cookie = require("../../../session/security_cookie");

var _common = require("../../../../common");

var _tenant_resolver = require("../../../multitenancy/tenant_resolver");

var _encode_uri_query = require("../../../../../../src/plugins/opensearch_dashboards_utils/common/url/encode_uri_query");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class BasicAuthRoutes {
  constructor(router, config, sessionStorageFactory, securityClient, coreSetup) {
    this.router = router;
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.securityClient = securityClient;
    this.coreSetup = coreSetup;
  }

  setupRoutes() {
    // bootstrap an empty page so that browser app can render the login page
    // using client side routing.
    this.coreSetup.http.resources.register({
      path: _common.LOGIN_PAGE_URI,
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      const clearOldVersionCookie = (0, _security_cookie.clearOldVersionCookieValue)(this.config);
      return response.renderAnonymousCoreApp({
        headers: {
          'set-cookie': clearOldVersionCookie
        }
      });
    }); // login using username and password

    this.router.post({
      path: _common.API_AUTH_LOGIN,
      validate: {
        body: _configSchema.schema.object({
          username: _configSchema.schema.string(),
          password: _configSchema.schema.string()
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      var _this$config$multiten;

      const forbiddenUsernames = this.config.auth.forbidden_usernames;

      if (forbiddenUsernames.indexOf(request.body.username) > -1) {
        context.security_plugin.logger.error(`Denied login for forbidden username ${request.body.username}`);
        return response.badRequest({
          // Cannot login using forbidden user name.
          body: 'Invalid username or password'
        });
      }

      let user;

      try {
        user = await this.securityClient.authenticate(request, {
          username: request.body.username,
          password: request.body.password
        });
      } catch (error) {
        context.security_plugin.logger.error(`Failed authentication: ${error}`);
        return response.unauthorized({
          headers: {
            'www-authenticate': error.message
          }
        });
      }

      this.sessionStorageFactory.asScoped(request).clear();
      const encodedCredentials = Buffer.from(`${request.body.username}:${request.body.password}`).toString('base64');
      const sessionStorage = {
        username: user.username,
        credentials: {
          authHeaderValue: `Basic ${encodedCredentials}`
        },
        authType: 'basicauth',
        isAnonymousAuth: false,
        expiryTime: Date.now() + this.config.session.ttl
      };

      if (user.multitenancy_enabled) {
        const selectTenant = (0, _tenant_resolver.resolveTenant)({
          request,
          username: user.username,
          roles: user.roles,
          availabeTenants: user.tenants,
          config: this.config,
          cookie: sessionStorage,
          multitenancyEnabled: user.multitenancy_enabled,
          privateTenantEnabled: user.private_tenant_enabled,
          defaultTenant: user.default_tenant
        }); // const selectTenant = user.default_tenant;

        sessionStorage.tenant = selectTenant;
      }

      this.sessionStorageFactory.asScoped(request).set(sessionStorage);
      return response.ok({
        body: {
          username: user.username,
          tenants: user.tenants,
          roles: user.roles,
          backendroles: user.backendRoles,
          selectedTenants: (_this$config$multiten = this.config.multitenancy) !== null && _this$config$multiten !== void 0 && _this$config$multiten.enabled ? sessionStorage.tenant : undefined
        }
      });
    }); // logout

    this.router.post({
      path: _common.API_AUTH_LOGOUT,
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      return response.ok({
        body: {}
      });
    }); // anonymous auth

    this.router.get({
      path: _common.ANONYMOUS_AUTH_LOGIN,
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      if (this.config.auth.anonymous_auth_enabled) {
        let user; // If the request contains no redirect path, simply redirect to basepath.

        let redirectUrl = this.coreSetup.http.basePath.serverBasePath ? this.coreSetup.http.basePath.serverBasePath : '/';
        const requestQuery = request.url.searchParams;
        const nextUrl = requestQuery === null || requestQuery === void 0 ? void 0 : requestQuery.get('nextUrl');

        if (nextUrl) {
          redirectUrl = nextUrl;
        }

        context.security_plugin.logger.info('The Redirect Path is ' + redirectUrl);

        try {
          user = await this.securityClient.authenticateWithHeaders(request, {});
        } catch (error) {
          context.security_plugin.logger.error(`Failed authentication: ${error}. Redirecting to Login Page`);
          return response.redirected({
            headers: {
              location: `${this.coreSetup.http.basePath.serverBasePath}${_common.LOGIN_PAGE_URI}${nextUrl ? '?nextUrl=' + (0, _encode_uri_query.encodeUriQuery)(redirectUrl) : ''}`
            }
          });
        }

        this.sessionStorageFactory.asScoped(request).clear();
        const sessionStorage = {
          username: user.username,
          authType: 'basicauth',
          isAnonymousAuth: true,
          expiryTime: Date.now() + this.config.session.ttl
        };

        if (user.multitenancy_enabled) {
          const selectTenant = (0, _tenant_resolver.resolveTenant)({
            request,
            username: user.username,
            roles: user.roles,
            availabeTenants: user.tenants,
            config: this.config,
            cookie: sessionStorage,
            multitenancyEnabled: user.multitenancy_enabled,
            privateTenantEnabled: user.private_tenant_enabled,
            defaultTenant: user.default_tenant
          });
          sessionStorage.tenant = selectTenant;
        }

        this.sessionStorageFactory.asScoped(request).set(sessionStorage);
        return response.redirected({
          headers: {
            location: `${redirectUrl}`
          }
        });
      } else {
        context.security_plugin.logger.error('Anonymous auth is disabled. Redirecting to Login Page');
        return response.redirected({
          headers: {
            location: `${this.coreSetup.http.basePath.serverBasePath}${_common.LOGIN_PAGE_URI}`
          }
        });
      }
    });
  }

}

exports.BasicAuthRoutes = BasicAuthRoutes;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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