"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.configRoutes = configRoutes;

var _configSchema = require("@osd/config-schema");

var _common = require("../../common");

var _helper = require("../utils/helper");

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
function configRoutes(router) {
  router.get({
    path: _common.NODE_API.GET_CONFIGS,
    validate: {
      query: _configSchema.schema.object({
        from_index: _configSchema.schema.number(),
        max_items: _configSchema.schema.number(),
        query: _configSchema.schema.maybe(_configSchema.schema.string()),
        config_type: _configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.string()]),
        is_enabled: _configSchema.schema.maybe(_configSchema.schema.boolean()),
        sort_field: _configSchema.schema.string(),
        sort_order: _configSchema.schema.string(),
        config_id_list: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.string()])),
        'smtp_account.method': _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.string()]))
      })
    }
  }, async (context, request, response) => {
    const config_type = (0, _helper.joinRequestParams)(request.query.config_type);
    const config_id_list = (0, _helper.joinRequestParams)(request.query.config_id_list);
    const encryption_method = (0, _helper.joinRequestParams)(request.query['smtp_account.method']);
    const query = request.query.query; // @ts-ignore

    const client = context.notificationsContext.notificationsClient.asScoped(request);

    try {
      const resp = await client.callAsCurrentUser('notifications.getConfigs', {
        from_index: request.query.from_index,
        max_items: request.query.max_items,
        is_enabled: request.query.is_enabled,
        sort_field: request.query.sort_field,
        sort_order: request.query.sort_order,
        config_type,
        ...(query && {
          text_query: query
        }),
        // text_query will exclude keyword fields
        ...(config_id_list && {
          config_id_list
        }),
        ...(encryption_method && {
          'smtp_account.method': encryption_method
        })
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get({
    path: `${_common.NODE_API.GET_CONFIG}/{configId}`,
    validate: {
      params: _configSchema.schema.object({
        configId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    // @ts-ignore
    const client = context.notificationsContext.notificationsClient.asScoped(request);

    try {
      const resp = await client.callAsCurrentUser('notifications.getConfigById', {
        configId: request.params.configId
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: _common.NODE_API.CREATE_CONFIG,
    validate: {
      body: _configSchema.schema.any()
    }
  }, async (context, request, response) => {
    // @ts-ignore
    const client = context.notificationsContext.notificationsClient.asScoped(request);

    try {
      const resp = await client.callAsCurrentUser('notifications.createConfig', {
        body: request.body
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put({
    path: `${_common.NODE_API.UPDATE_CONFIG}/{configId}`,
    validate: {
      body: _configSchema.schema.any(),
      params: _configSchema.schema.object({
        configId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    // @ts-ignore
    const client = context.notificationsContext.notificationsClient.asScoped(request);

    try {
      const resp = await client.callAsCurrentUser('notifications.updateConfigById', {
        configId: request.params.configId,
        body: request.body
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.delete({
    path: _common.NODE_API.DELETE_CONFIGS,
    validate: {
      query: _configSchema.schema.object({
        config_id_list: _configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.string()])
      })
    }
  }, async (context, request, response) => {
    // @ts-ignore
    const client = context.notificationsContext.notificationsClient.asScoped(request);
    const config_id_list = (0, _helper.joinRequestParams)(request.query.config_id_list);

    try {
      const resp = await client.callAsCurrentUser('notifications.deleteConfigs', {
        config_id_list
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get({
    path: _common.NODE_API.GET_AVAILABLE_FEATURES,
    validate: false
  }, async (context, request, response) => {
    // @ts-ignore
    const client = context.notificationsContext.notificationsClient.asScoped(request);

    try {
      const resp = await client.callAsCurrentUser('notifications.getServerFeatures');
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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