"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PRIVATE_TENANTS = exports.GLOBAL_TENANTS = void 0;
exports.addTenantParameterToResolvedShortLink = addTenantParameterToResolvedShortLink;
exports.isValidTenant = isValidTenant;
exports.resolve = resolve;
exports.resolveTenant = resolveTenant;

var _lodash = require("lodash");

var _std = require("@osd/std");

var _common = require("../../common");

var _router = require("../../../../src/core/server/http/router");

var _short_url_routes = require("../../../../src/plugins/share/common/short_url_routes");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
const PRIVATE_TENANTS = [_common.PRIVATE_TENANT_SYMBOL, 'private'];
exports.PRIVATE_TENANTS = PRIVATE_TENANTS;
const GLOBAL_TENANTS = ['global', _common.GLOBAL_TENANT_SYMBOL, 'Global'];
/**
 * Resovles the tenant the user is using.
 *
 * @param request OpenSearchDashboards request.
 * @param username
 * @param roles
 * @param availabeTenants
 * @param config security plugin config.
 * @param cookie cookie extracted from the request. The cookie should have been parsed by AuthenticationHandler.
 * pass it as parameter instead of extracting again.
 * @param multitenancyEnabled
 * @param privateTenantEnabled
 * @param defaultTenant
 *
 * @returns user preferred tenant of the request.
 */

exports.GLOBAL_TENANTS = GLOBAL_TENANTS;

function resolveTenant({
  request,
  username,
  roles,
  availabeTenants,
  config,
  cookie,
  multitenancyEnabled,
  privateTenantEnabled,
  defaultTenant
}) {
  var _request$url, _request$url$searchPa, _request$url2, _request$url2$searchP, _request$url3, _request$url3$searchP, _config$multitenancy, _config$multitenancy2;

  const DEFAULT_READONLY_ROLES = ['kibana_read_only'];
  let selectedTenant;
  const securityTenant_ = request === null || request === void 0 ? void 0 : (_request$url = request.url) === null || _request$url === void 0 ? void 0 : (_request$url$searchPa = _request$url.searchParams) === null || _request$url$searchPa === void 0 ? void 0 : _request$url$searchPa.get('securityTenant_');
  const securitytenant = request === null || request === void 0 ? void 0 : (_request$url2 = request.url) === null || _request$url2 === void 0 ? void 0 : (_request$url2$searchP = _request$url2.searchParams) === null || _request$url2$searchP === void 0 ? void 0 : _request$url2$searchP.get('securitytenant'); // eslint-disable-next-line @typescript-eslint/naming-convention

  const security_tenant = request === null || request === void 0 ? void 0 : (_request$url3 = request.url) === null || _request$url3 === void 0 ? void 0 : (_request$url3$searchP = _request$url3.searchParams) === null || _request$url3$searchP === void 0 ? void 0 : _request$url3$searchP.get('security_tenant');

  if (securityTenant_) {
    selectedTenant = securityTenant_;
  } else if (securitytenant) {
    selectedTenant = securitytenant;
  } else if (security_tenant) {
    selectedTenant = security_tenant;
  } else if (request.headers.securitytenant || request.headers.securityTenant_) {
    selectedTenant = request.headers.securitytenant ? request.headers.securitytenant : request.headers.securityTenant_;
  } else if (isValidTenant(cookie.tenant)) {
    selectedTenant = cookie.tenant;
  } else if (defaultTenant && multitenancyEnabled) {
    selectedTenant = defaultTenant;
  } else {
    selectedTenant = undefined;
  }

  const isReadonly = roles === null || roles === void 0 ? void 0 : roles.some(role => {
    var _config$readonly_mode;

    return ((_config$readonly_mode = config.readonly_mode) === null || _config$readonly_mode === void 0 ? void 0 : _config$readonly_mode.roles.includes(role)) || DEFAULT_READONLY_ROLES.includes(role);
  });
  const preferredTenants = (_config$multitenancy = config.multitenancy) === null || _config$multitenancy === void 0 ? void 0 : _config$multitenancy.tenants.preferred;
  const globalTenantEnabled = (_config$multitenancy2 = config.multitenancy) === null || _config$multitenancy2 === void 0 ? void 0 : _config$multitenancy2.tenants.enable_global;
  return resolve(username, selectedTenant, preferredTenants, availabeTenants, globalTenantEnabled, multitenancyEnabled, privateTenantEnabled);
}

function resolve(username, requestedTenant, preferredTenants, availableTenants, // is an object like { tenant_name_1: true, tenant_name_2: false, ... }
globalTenantEnabled, multitenancyEnabled, privateTenantEnabled) {
  const availableTenantsClone = (0, _lodash.cloneDeep)(availableTenants);
  delete availableTenantsClone[username];

  if (!globalTenantEnabled && !privateTenantEnabled && (0, _lodash.isEmpty)(availableTenantsClone)) {
    return undefined;
  }

  if (!multitenancyEnabled) {
    if (!globalTenantEnabled) {
      return undefined;
    }

    return _common.GLOBAL_TENANT_SYMBOL;
  }

  if (isValidTenant(requestedTenant)) {
    requestedTenant = requestedTenant;

    if (requestedTenant in availableTenants) {
      return requestedTenant;
    }

    if (privateTenantEnabled && username in availableTenants && PRIVATE_TENANTS.indexOf(requestedTenant) > -1) {
      return _common.PRIVATE_TENANT_SYMBOL;
    }

    if (globalTenantEnabled && GLOBAL_TENANTS.indexOf(requestedTenant) > -1) {
      return _common.GLOBAL_TENANT_SYMBOL;
    }
  }

  if (preferredTenants && !(0, _lodash.isEmpty)(preferredTenants)) {
    for (const element of preferredTenants) {
      const tenant = element.toLowerCase();

      if (globalTenantEnabled && GLOBAL_TENANTS.indexOf(tenant) > -1) {
        return _common.GLOBAL_TENANT_SYMBOL;
      }

      if (privateTenantEnabled && PRIVATE_TENANTS.indexOf(tenant) > -1 && username in availableTenants) {
        return _common.PRIVATE_TENANT_SYMBOL;
      }

      if (tenant in availableTenants) {
        return tenant;
      }
    }
  }

  if (globalTenantEnabled) {
    return _common.GLOBAL_TENANT_SYMBOL;
  }

  if (privateTenantEnabled) {
    return _common.PRIVATE_TENANT_SYMBOL;
  }
  /**
   * Fall back to the first tenant in the available tenants
   * Under the condition of enabling multitenancy, if the user has disabled both 'Global' and 'Private' tenants:
   * it will remove the default global tenant key for custom tenant.
   */


  if (Object.keys(availableTenantsClone).length > 1 && availableTenantsClone.hasOwnProperty(_common.globalTenantName)) {
    delete availableTenantsClone[_common.globalTenantName];
  }

  return (0, _lodash.findKey)(availableTenantsClone, () => true);
}
/**
 * Return true if tenant parameter is a valid tenent.
 *
 * Note: empty string '' is valid, which means global tenant.
 *
 * @param tenant
 */


function isValidTenant(tenant) {
  return tenant !== undefined && tenant !== null;
}
/**
 * If multitenancy is enabled & the URL entered starts with /goto,
 * We will modify the rawResponse to add a new parameter to the URL, the security_tenant (or value for tenant when in multitenancy)
 * With the security_tenant added, the resolved short URL now contains the security_tenant information (so the short URL retains the tenant information).
 **/


function addTenantParameterToResolvedShortLink(request) {
  if (request.url.pathname.startsWith(`${_short_url_routes.GOTO_PREFIX}/`)) {
    const rawRequest = (0, _router.ensureRawRequest)(request);
    const rawResponse = rawRequest.response; // Make sure the request really should redirect

    if (rawResponse.headers.location) {
      const modifiedUrl = (0, _std.modifyUrl)(rawResponse.headers.location, parts => {
        if (parts.query.security_tenant === undefined) {
          parts.query.security_tenant = request.headers.securitytenant;
        } // Mutating the headers toolkit.next({headers: ...}) logs a warning about headers being overwritten

      });
      rawResponse.headers.location = modifiedUrl;
    }
  }

  return request;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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