/*
 * Decompiled with CFR 0.152.
 */
package net.sf.freecol.server.generator;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Random;
import java.util.logging.Logger;
import net.sf.freecol.common.model.Direction;
import net.sf.freecol.common.model.Game;
import net.sf.freecol.common.model.LandMap;
import net.sf.freecol.common.model.Map;
import net.sf.freecol.common.model.Region;
import net.sf.freecol.common.model.Resource;
import net.sf.freecol.common.model.ResourceType;
import net.sf.freecol.common.model.Specification;
import net.sf.freecol.common.model.Tile;
import net.sf.freecol.common.model.TileImprovement;
import net.sf.freecol.common.model.TileImprovementType;
import net.sf.freecol.common.model.TileType;
import net.sf.freecol.common.option.OptionGroup;
import net.sf.freecol.common.util.CollectionUtils;
import net.sf.freecol.common.util.LogBuilder;
import net.sf.freecol.common.util.RandomChoice;
import net.sf.freecol.common.util.RandomUtils;
import net.sf.freecol.server.generator.River;
import net.sf.freecol.server.generator.RiverSection;
import net.sf.freecol.server.model.ServerRegion;

public class TerrainGenerator {
    private static final Logger logger = Logger.getLogger(TerrainGenerator.class.getName());
    public static final int LAND_REGIONS_SCORE_VALUE = 1000;
    public static final int LAND_REGION_MIN_SCORE = 5;
    public static final int LAND_REGION_MAX_SIZE = 75;
    private final Random random;
    private final RandomUtils.RandomIntCache cache;
    private List<TileType> landTileTypes = null;
    private List<TileType> oceanTileTypes = null;

    public TerrainGenerator(Random random) {
        this.random = random;
        this.cache = new RandomUtils.RandomIntCache(logger, "terrain", random, 65536, 512);
    }

    private int limitToRange(int value, int lower, int upper) {
        return Math.max(lower, Math.min(value, upper));
    }

    private int getApproximateLandCount(Game game) {
        OptionGroup mapOptions = game.getMapGeneratorOptions();
        return mapOptions.getInteger("model.option.mapWidth") * mapOptions.getInteger("model.option.mapHeight") * mapOptions.getInteger("model.option.landMass") / 100;
    }

    private TileType getRandomLandTileType(Game game, int latitude) {
        Specification spec = game.getSpecification();
        if (this.landTileTypes == null) {
            this.landTileTypes = CollectionUtils.transform(spec.getTileTypeList(), t -> !t.isElevation() && !t.isWater());
        }
        return this.getRandomTileType(game, this.landTileTypes, latitude);
    }

    private TileType getRandomOceanTileType(Game game, int latitude) {
        Specification spec = game.getSpecification();
        if (this.oceanTileTypes == null) {
            this.oceanTileTypes = CollectionUtils.transform(spec.getTileTypeList(), t -> t.isWater() && t.isHighSeasConnected() && !t.isDirectlyHighSeasConnected());
        }
        return this.getRandomTileType(game, this.oceanTileTypes, latitude);
    }

    private TileType getRandomTileType(Game game, List<TileType> candidates, int latitude) {
        TileType type;
        OptionGroup mapOptions = game.getMapGeneratorOptions();
        Specification spec = game.getSpecification();
        int forestChance = mapOptions.getRange("model.option.forestNumber");
        int temperaturePreference = mapOptions.getRange("model.option.temperature");
        int poleTemperature = -20;
        int equatorTemperature = 40;
        switch (temperaturePreference) {
            case 0: {
                poleTemperature = -20;
                equatorTemperature = 25;
                break;
            }
            case 1: {
                poleTemperature = -20;
                equatorTemperature = 30;
                break;
            }
            case 2: {
                poleTemperature = -10;
                equatorTemperature = 35;
                break;
            }
            case 3: {
                poleTemperature = -5;
                equatorTemperature = 40;
                break;
            }
            case 4: {
                poleTemperature = 0;
                equatorTemperature = 40;
                break;
            }
        }
        int temperatureRange = equatorTemperature - poleTemperature;
        int localeTemperature = poleTemperature + (90 - Math.abs(latitude)) * temperatureRange / 90;
        int temperatureDeviation = 7;
        localeTemperature += this.cache.nextInt(temperatureDeviation * 2) - temperatureDeviation;
        localeTemperature = this.limitToRange(localeTemperature, -20, 40);
        int localeHumidity = spec.getRange("model.option.humidity");
        int humidityDeviation = 20;
        localeHumidity += this.cache.nextInt(humidityDeviation * 2) - humidityDeviation;
        localeHumidity = this.limitToRange(localeHumidity, 0, 100);
        ArrayList<TileType> candidateTileTypes = new ArrayList<TileType>(candidates);
        int i = 0;
        while (i < candidateTileTypes.size()) {
            type = (TileType)candidateTileTypes.get(i);
            if (!type.withinRange(TileType.RangeType.TEMPERATURE, localeTemperature)) {
                candidateTileTypes.remove(i);
                continue;
            }
            ++i;
        }
        switch (candidateTileTypes.size()) {
            case 0: {
                throw new RuntimeException("No TileType for temperature==" + localeTemperature);
            }
            case 1: {
                return CollectionUtils.first(candidateTileTypes);
            }
        }
        i = 0;
        while (i < candidateTileTypes.size()) {
            type = (TileType)candidateTileTypes.get(i);
            if (!type.withinRange(TileType.RangeType.HUMIDITY, localeHumidity)) {
                candidateTileTypes.remove(i);
                continue;
            }
            ++i;
        }
        switch (candidateTileTypes.size()) {
            case 0: {
                throw new RuntimeException("No TileType for humidity==" + localeHumidity);
            }
            case 1: {
                return CollectionUtils.first(candidateTileTypes);
            }
        }
        boolean forested = this.cache.nextInt(100) < forestChance && CollectionUtils.any(candidateTileTypes, TileType::isForested);
        i = 0;
        while (i < candidateTileTypes.size()) {
            TileType type2 = (TileType)candidateTileTypes.get(i);
            if (type2.isForested() != forested) {
                candidateTileTypes.remove(i);
                continue;
            }
            ++i;
        }
        i = candidateTileTypes.size();
        switch (i) {
            case 0: {
                throw new RuntimeException("No TileType for forested==" + forested);
            }
            case 1: {
                return CollectionUtils.first(candidateTileTypes);
            }
        }
        return (TileType)candidateTileTypes.get(this.cache.nextInt(i));
    }

    private List<ServerRegion> createLandRegions(Map map, LogBuilder lb) {
        int c;
        int y;
        Game game = map.getGame();
        int continents = 0;
        boolean[][] landmap = new boolean[map.getWidth()][map.getHeight()];
        int[][] continentmap = new int[map.getWidth()][map.getHeight()];
        int landsize = 0;
        for (int x = 0; x < map.getWidth(); ++x) {
            for (y = 0; y < map.getHeight(); ++y) {
                continentmap[x][y] = 0;
                landmap[x][y] = false;
                if (!map.isValid(x, y)) continue;
                Tile tile = map.getTile(x, y);
                boolean bl = landmap[x][y] = tile.isLand() && tile.getRegion() == null;
                if (!tile.isLand()) continue;
                ++landsize;
            }
        }
        for (int y2 = 0; y2 < map.getHeight(); ++y2) {
            for (int x = 0; x < map.getWidth(); ++x) {
                if (!landmap[x][y2]) continue;
                ++continents;
                boolean[][] continent = Map.floodFillBool(landmap, x, y2);
                for (int yy = 0; yy < map.getHeight(); ++yy) {
                    for (int xx = 0; xx < map.getWidth(); ++xx) {
                        if (!continent[xx][yy]) continue;
                        continentmap[xx][yy] = continents;
                        landmap[xx][yy] = false;
                    }
                }
            }
        }
        lb.add("Number of individual landmasses is ", continents, "\n");
        int[] continentsize = new int[continents + 1];
        for (y = 0; y < map.getHeight(); ++y) {
            for (int x = 0; x < map.getWidth(); ++x) {
                int n = continentmap[x][y];
                continentsize[n] = continentsize[n] + 1;
            }
        }
        int oldcontinents = continents;
        for (int c2 = 1; c2 <= oldcontinents; ++c2) {
            if (continentsize[c2] <= 75) continue;
            boolean[][] splitcontinent = new boolean[map.getWidth()][map.getHeight()];
            int splitX = 0;
            int splitY = 0;
            for (int x = 0; x < map.getWidth(); ++x) {
                for (int y3 = 0; y3 < map.getHeight(); ++y3) {
                    if (continentmap[x][y3] == c2) {
                        splitcontinent[x][y3] = true;
                        splitX = x;
                        splitY = y3;
                        continue;
                    }
                    splitcontinent[x][y3] = false;
                }
            }
            while (continentsize[c2] > 75) {
                int targetsize = 75;
                if (continentsize[c2] < 150) {
                    targetsize = continentsize[c2] / 2;
                }
                ++continents;
                boolean[][] newregion = Map.floodFillBool(splitcontinent, splitX, splitY, targetsize);
                for (int x = 0; x < map.getWidth(); ++x) {
                    for (int y4 = 0; y4 < map.getHeight(); ++y4) {
                        if (newregion[x][y4]) {
                            continentmap[x][y4] = continents;
                            splitcontinent[x][y4] = false;
                            int n = c2;
                            continentsize[n] = continentsize[n] - 1;
                        }
                        if (!splitcontinent[x][y4]) continue;
                        splitX = x;
                        splitY = y4;
                    }
                }
            }
        }
        lb.add("Number of land regions being created: ", continents, "\n");
        ServerRegion[] landregions = new ServerRegion[continents + 1];
        boolean landIndex = true;
        for (c = 1; c <= continents; ++c) {
            landregions[c] = new ServerRegion(game, Region.RegionType.LAND);
        }
        for (int y5 = 0; y5 < map.getHeight(); ++y5) {
            for (int x = 0; x < map.getWidth(); ++x) {
                if (continentmap[x][y5] <= 0) continue;
                Tile tile = map.getTile(x, y5);
                landregions[continentmap[x][y5]].addTile(tile);
            }
        }
        for (c = 1; c <= continents; ++c) {
            ServerRegion sr = landregions[c];
            int score = Math.max((int)((float)sr.getSize() / (float)landsize * 1000.0f), 5);
            sr.setScoreValue(score);
            lb.add("Created land region ", sr.toString(), " (size ", sr.getSize(), ", score ", sr.getScoreValue(), ", parent ", sr.getParent() == null ? "(null)" : sr.getParent().toString(), ")\n");
        }
        return Arrays.asList(Arrays.copyOfRange(landregions, 1, continents + 1));
    }

    private List<ServerRegion> createMountains(Map map, LogBuilder lb) {
        Game game = map.getGame();
        Specification spec = game.getSpecification();
        OptionGroup mapOptions = game.getMapGeneratorOptions();
        float randomHillsRatio = 0.5f;
        int maximumLength = Math.max(mapOptions.getInteger("model.option.mapWidth"), mapOptions.getInteger("model.option.mapHeight")) / 10;
        int number = Math.round((1.0f - randomHillsRatio) * ((float)this.getApproximateLandCount(game) / (float)mapOptions.getRange("model.option.mountainNumber")));
        lb.add("Number of mountain tiles is ", number, "\n", "Maximum length of mountain ranges is ", maximumLength, "\n");
        ArrayList<ServerRegion> result = new ArrayList<ServerRegion>(number);
        TileType hills = spec.getTileType("model.tile.hills");
        TileType mountains = spec.getTileType("model.tile.mountains");
        if (hills == null || mountains == null) {
            throw new RuntimeException("Both Hills and Mountains TileTypes must be defined: " + spec);
        }
        ArrayList tiles = new ArrayList();
        map.forEachTile(t -> t.isGoodMountainTile(mountains), t -> tiles.add(t));
        RandomUtils.randomShuffle(logger, "Randomize mountain tiles", tiles, this.random);
        int counter = 0;
        for (Tile startTile : tiles) {
            if (!startTile.isGoodMountainTile(mountains)) continue;
            ServerRegion mountainRegion = new ServerRegion(game, Region.RegionType.MOUNTAIN);
            Direction direction = Direction.getRandomDirection("getLand", logger, this.random);
            int length = maximumLength - this.cache.nextInt(maximumLength / 2);
            Tile tile = startTile;
            for (int index = 0; index < length; ++index) {
                if (tile.getType() != mountains) {
                    tile.setType(mountains);
                    mountainRegion.addTile(tile);
                    ++counter;
                }
                for (Tile t2 : tile.getSurroundingTiles(1)) {
                    if (!t2.isGoodHillTile() || t2.getType() == mountains) continue;
                    int r = this.cache.nextInt(8);
                    if (r < 2) {
                        t2.setType(mountains);
                        mountainRegion.addTile(t2);
                        ++counter;
                        continue;
                    }
                    if (r >= 7) continue;
                    t2.setType(hills);
                    mountainRegion.addTile(t2);
                }
                if ((tile = tile.getNeighbourOrNull(direction)) == null || !tile.isLand()) break;
            }
            int scoreValue = 2 * mountainRegion.getSize();
            mountainRegion.setScoreValue(scoreValue);
            result.add(mountainRegion);
            lb.add("Created mountain region (direction ", direction, ", length ", length, ", size ", mountainRegion.getSize(), ", score value ", scoreValue, ").\n");
            if (counter < number) continue;
            break;
        }
        lb.add("Added ", counter, " mountain range tiles.\n");
        tiles.clear();
        map.forEachTile(Tile::isGoodHillTile, t -> tiles.add(t));
        RandomUtils.randomShuffle(logger, "Randomize hill tiles", tiles, this.random);
        number = (int)((float)this.getApproximateLandCount(game) * randomHillsRatio) / mapOptions.getRange("model.option.mountainNumber");
        counter = 0;
        for (Tile t3 : tiles) {
            boolean m = this.cache.nextInt(4) == 0;
            t3.setType(m ? mountains : hills);
            if (++counter < number) continue;
            break;
        }
        lb.add("Added ", counter, " random hilly tiles.\n");
        return result;
    }

    private List<ServerRegion> createRivers(Map map, LogBuilder lb) {
        Game game = map.getGame();
        Specification spec = game.getSpecification();
        OptionGroup mapOptions = game.getMapGeneratorOptions();
        ArrayList<ServerRegion> result = new ArrayList<ServerRegion>();
        TileImprovementType riverType = spec.getTileImprovementType("model.improvement.river");
        int number = this.getApproximateLandCount(game) / mapOptions.getRange("model.option.riverNumber");
        HashMap<Tile, River> riverMap = new HashMap<Tile, River>();
        ArrayList<River> rivers = new ArrayList<River>();
        ArrayList tiles = new ArrayList();
        map.forEachTile(t -> t.isGoodRiverTile(riverType), t -> tiles.add(t));
        RandomUtils.randomShuffle(logger, "Randomize river tiles", tiles, this.random);
        int counter = 0;
        for (Tile tile : tiles) {
            if (riverMap.get(tile) != null) continue;
            ServerRegion riverRegion = new ServerRegion(game, Region.RegionType.RIVER);
            River river = new River(map, riverMap, riverRegion, this.random);
            if (river.flowFromSource(tile)) {
                lb.add("Created new river with length ", river.getLength(), "\n");
                result.add(riverRegion);
                rivers.add(river);
                if (++counter < number) continue;
                break;
            }
            lb.add("Failed to generate river at " + tile + ".\n");
        }
        lb.add("Created ", counter, " rivers of maximum ", number, "\n");
        for (River river : rivers) {
            ServerRegion region = river.getRegion();
            int score = 2 * CollectionUtils.sum(river.getSections(), RiverSection::getSize);
            region.setScoreValue(score);
            lb.add("Created river region (length ", river.getLength(), ", score value ", score, ").\n");
        }
        return result;
    }

    private List<ServerRegion> createLakeRegions(Map map, LogBuilder lb) {
        ArrayList<Tile> lakes = new ArrayList<Tile>();
        lb.add("Lakes at:");
        for (int y = 0; y < map.getHeight(); ++y) {
            for (int x = 0; x < map.getWidth(); ++x) {
                Tile tile;
                if (!map.isValid(x, y) || (tile = map.getTile(x, y)).isLand() || map.getTile(x, y).getRegion() != null) continue;
                lakes.add(tile);
                lb.add(" ", x, ",", y);
            }
        }
        lb.add("\n");
        return this.makeLakes(map, lakes);
    }

    private List<ServerRegion> makeLakes(Map map, List<Tile> lakes) {
        Game game = map.getGame();
        TileType lakeType = map.getSpecification().getTileType("model.tile.lake");
        ArrayList<Tile> todo = new ArrayList<Tile>(lakes.size() / 10);
        ArrayList<ServerRegion> result = new ArrayList<ServerRegion>(lakes.size() / 10);
        boolean lakeCount = false;
        while (!lakes.isEmpty()) {
            Tile tile = CollectionUtils.first(lakes);
            if (tile.getRegion() != null) continue;
            ServerRegion lakeRegion = new ServerRegion(game, Region.RegionType.LAKE);
            todo.clear();
            todo.add(tile);
            while (!todo.isEmpty()) {
                Tile t = (Tile)todo.remove(0);
                if (!lakes.contains(t)) continue;
                t.setRegion(lakeRegion);
                t.setType(lakeType);
                lakes.remove(t);
                for (Direction d : Direction.allDirections) {
                    Tile t0 = map.getAdjacentTile(t, d);
                    if (t0 == null) continue;
                    todo.add(t0);
                }
            }
            result.add(lakeRegion);
        }
        return result;
    }

    private void perhapsAddBonus(Tile t, boolean generateBonus) {
        Game game = t.getGame();
        OptionGroup mapOptions = game.getMapGeneratorOptions();
        Specification spec = game.getSpecification();
        TileImprovementType fishBonusLandType = spec.getTileImprovementType("model.improvement.fishBonusLand");
        TileImprovementType fishBonusRiverType = spec.getTileImprovementType("model.improvement.fishBonusRiver");
        int bonusNumber = mapOptions.getRange("model.option.bonusNumber");
        if (t.isLand()) {
            if (generateBonus && this.cache.nextInt(100) < bonusNumber) {
                t.addResource(this.createResource(t));
            }
        } else {
            int adjacentLand = 0;
            boolean adjacentRiver = false;
            for (Direction direction : Direction.values()) {
                Tile otherTile = t.getNeighbourOrNull(direction);
                if (otherTile == null || !otherTile.isLand()) continue;
                ++adjacentLand;
                if (!otherTile.hasRiver()) continue;
                adjacentRiver = true;
            }
            if (adjacentLand > 2) {
                t.add(new TileImprovement(game, t, fishBonusLandType, null));
            }
            if (adjacentRiver && !t.hasRiver()) {
                t.add(new TileImprovement(game, t, fishBonusRiverType, null));
            }
            if (t.getType().isHighSeasConnected()) {
                if (generateBonus && adjacentLand > 1 && this.cache.nextInt(10 - adjacentLand) == 0) {
                    t.addResource(this.createResource(t));
                }
            } else if (this.cache.nextInt(100) < bonusNumber) {
                t.addResource(this.createResource(t));
            }
        }
    }

    private Resource createResource(Tile tile) {
        int maxValue;
        if (tile == null) {
            return null;
        }
        ResourceType resourceType = (ResourceType)RandomChoice.getWeightedRandom(null, null, tile.getType().getResourceTypes(), this.random);
        if (resourceType == null) {
            return null;
        }
        int minValue = resourceType.getMinValue();
        int quantity = minValue == (maxValue = resourceType.getMaxValue()) ? maxValue : minValue + this.cache.nextInt(maxValue - minValue + 1);
        return new Resource(tile.getGame(), tile, resourceType, quantity);
    }

    public static void encodeStyle(Tile tile) {
        Tile t;
        EnumMap<Direction, Boolean> connections = new EnumMap<Direction, Boolean>(Direction.class);
        Iterator<Direction> iterator = Direction.corners.iterator();
        while (iterator.hasNext()) {
            Direction d;
            t = tile.getNeighbourOrNull(d = iterator.next());
            connections.put(d, t != null && t.isLand());
        }
        for (Direction d : Direction.longSides) {
            t = tile.getNeighbourOrNull(d);
            if (t != null && t.isLand()) {
                connections.put(d, Boolean.TRUE);
                connections.put(d.getNextDirection(), Boolean.FALSE);
                connections.put(d.getPreviousDirection(), Boolean.FALSE);
                continue;
            }
            connections.put(d, Boolean.FALSE);
        }
        int result = 0;
        int index = 0;
        for (Direction d : Direction.corners) {
            if (((Boolean)connections.get(d)).booleanValue()) {
                result += (int)Math.pow(2.0, index);
            }
            ++index;
        }
        for (Direction d : Direction.longSides) {
            if (((Boolean)connections.get(d)).booleanValue()) {
                result += (int)Math.pow(2.0, index);
            }
            ++index;
        }
        tile.setStyle(result);
    }

    public Map generateMap(Game game, Map importMap, LandMap landMap, LogBuilder lb) {
        OptionGroup mapOptions = game.getMapGeneratorOptions();
        int width = landMap.getWidth();
        int height = landMap.getHeight();
        boolean importBonuses = importMap != null && mapOptions.getBoolean("model.option.importBonuses");
        boolean importRumours = importMap != null && mapOptions.getBoolean("model.option.importRumours");
        boolean importTerrain = importMap != null && mapOptions.getBoolean("model.option.importTerrain");
        Map map = new Map(game, width, height);
        game.changeMap(map);
        int minimumLatitude = mapOptions.getInteger("model.option.minimumLatitude");
        int maximumLatitude = mapOptions.getInteger("model.option.maximumLatitude");
        minimumLatitude = this.limitToRange(minimumLatitude, -90, 90);
        maximumLatitude = this.limitToRange(maximumLatitude, -90, 90);
        map.setMinimumLatitude(Math.min(minimumLatitude, maximumLatitude));
        map.setMaximumLatitude(Math.max(minimumLatitude, maximumLatitude));
        HashMap<String, ServerRegion> regionMap = new HashMap<String, ServerRegion>();
        if (importTerrain) {
            ServerRegion region;
            lb.add("Imported regions: ");
            for (Region r : importMap.getRegions()) {
                region = new ServerRegion(game, r);
                map.addRegion(region);
                regionMap.put(r.getId(), region);
                lb.add(" ", region.toString());
            }
            for (Region r : importMap.getRegions()) {
                region = (ServerRegion)regionMap.get(r.getId());
                Region x2 = r.getParent();
                if (x2 != null) {
                    x2 = (Region)regionMap.get(x2.getId());
                }
                region.setParent(x2);
                for (Region c : r.getChildren()) {
                    x2 = (Region)regionMap.get(c.getId());
                    if (x2 == null) continue;
                    region.addChild(x2);
                }
            }
            lb.add("\n");
        }
        Map.Layer layer = importRumours ? Map.Layer.RUMOURS : (importBonuses ? Map.Layer.RESOURCES : Map.Layer.RIVERS);
        ArrayList fixRegions = new ArrayList();
        map.populateTiles((x, y) -> {
            Tile t;
            Tile otherTile = null;
            if (importTerrain && importMap.isValid((int)x, (int)y) && (otherTile = importMap.getTile((int)x, (int)y)) != null && otherTile.isLand() == landMap.isLand((int)x, (int)y)) {
                t = map.importTile(otherTile, (int)x, (int)y, layer);
                Region r = otherTile.getRegion();
                if (r == null) {
                    fixRegions.add(t);
                } else {
                    ServerRegion ours = (ServerRegion)regionMap.get(r.getId());
                    if (ours == null) {
                        lb.add("Could not set tile region ", r.getId(), " for tile: ", t, "\n");
                        fixRegions.add(t);
                    } else {
                        ours.addTile(t);
                    }
                }
            } else {
                int latitude = map.getLatitude((int)y);
                TileType tt = landMap.isLand((int)x, (int)y) ? this.getRandomLandTileType(game, latitude) : this.getRandomOceanTileType(game, latitude);
                t = new Tile(game, tt, (int)x, (int)y);
            }
            return t;
        });
        List<ServerRegion> fixed = ServerRegion.requireFixedRegions(map, lb);
        ArrayList<ServerRegion> newRegions = new ArrayList<ServerRegion>();
        if (importTerrain) {
            if (!fixRegions.isEmpty()) {
                newRegions.addAll(this.createLakeRegions(map, lb));
                newRegions.addAll(this.createLandRegions(map, lb));
            }
        } else {
            map.resetHighSeas(mapOptions.getInteger("model.option.distanceToHighSea"), mapOptions.getInteger("model.option.maximumDistanceToEdge"));
            if (landMap.hasLand()) {
                newRegions.addAll(this.createMountains(map, lb));
                newRegions.addAll(this.createRivers(map, lb));
                newRegions.addAll(this.createLakeRegions(map, lb));
                newRegions.addAll(this.createLandRegions(map, lb));
            }
        }
        lb.shrink("\n");
        List<ServerRegion> geographic = CollectionUtils.transform(fixed, ServerRegion::isGeographic);
        for (ServerRegion sr : newRegions) {
            ServerRegion gr = CollectionUtils.find(geographic, g -> g.containsCenter(sr));
            if (gr != null) {
                sr.setParent(gr);
                gr.addChild(sr);
                gr.setSize(gr.getSize() + sr.getSize());
            }
            map.addRegion(sr);
        }
        map.fixupRegions();
        map.forEachTile(t -> {
            this.perhapsAddBonus((Tile)t, !importBonuses);
            if (!t.isLand()) {
                TerrainGenerator.encodeStyle(t);
            }
        });
        map.resetContiguity();
        map.resetHighSeasCount();
        return map;
    }
}

