"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isPropertyWithKey = isPropertyWithKey;
exports.getProperty = getProperty;
exports.getModuleSpecifier = getModuleSpecifier;
exports.getIdentifierDeclarationFromSource = getIdentifierDeclarationFromSource;
exports.getIdentifierDeclaration = getIdentifierDeclaration;
exports.getVariableValue = getVariableValue;
exports.serializeObject = serializeObject;
exports.getResolvedModuleSourceFile = getResolvedModuleSourceFile;
exports.getPropertyValue = getPropertyValue;
exports.pickDeep = pickDeep;
exports.difference = difference;
exports.normalizePath = normalizePath;
exports.flattenKeys = exports.globAsync = exports.writeFileAsync = exports.readFileAsync = void 0;

var ts = _interopRequireWildcard(require("typescript"));

var _lodash = require("lodash");

var path = _interopRequireWildcard(require("path"));

var _glob = _interopRequireDefault(require("glob"));

var _fs = require("fs");

var _util = require("util");

var _normalizePath = _interopRequireDefault(require("normalize-path"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _getRequireWildcardCache() { if (typeof WeakMap !== "function") return null; var cache = new WeakMap(); _getRequireWildcardCache = function () { return cache; }; return cache; }

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
const readFileAsync = (0, _util.promisify)(_fs.readFile);
exports.readFileAsync = readFileAsync;
const writeFileAsync = (0, _util.promisify)(_fs.writeFile);
exports.writeFileAsync = writeFileAsync;
const globAsync = (0, _util.promisify)(_glob.default);
exports.globAsync = globAsync;

function isPropertyWithKey(property, identifierName) {
  if (ts.isPropertyAssignment(property) || ts.isMethodDeclaration(property)) {
    if (ts.isIdentifier(property.name)) {
      return property.name.text === identifierName;
    }
  }

  return false;
}

function getProperty(objectNode, propertyName) {
  let foundProperty = null;
  ts.visitNodes((objectNode === null || objectNode === void 0 ? void 0 : objectNode.properties) || [], node => {
    if (isPropertyWithKey(node, propertyName)) {
      foundProperty = node;
      return node;
    }
  });
  return foundProperty;
}

function getModuleSpecifier(node) {
  if (node.moduleSpecifier) {
    return node.moduleSpecifier.text;
  }

  return getModuleSpecifier(node.parent);
}

function getIdentifierDeclarationFromSource(node, source) {
  if (!ts.isIdentifier(node)) {
    throw new Error(`node is not an identifier ${node.getText()}`);
  }

  const identifierName = node.getText();
  const identifierDefinition = source.locals.get(identifierName);

  if (!identifierDefinition) {
    throw new Error(`Unable to fine identifier in source ${identifierName}`);
  }

  const declarations = identifierDefinition.declarations;
  const latestDeclaration = Array.isArray(declarations) && declarations[declarations.length - 1];

  if (!latestDeclaration) {
    throw new Error(`Unable to fine declaration for identifier ${identifierName}`);
  }

  return latestDeclaration;
}

function getIdentifierDeclaration(node) {
  const source = node.getSourceFile();

  if (!source) {
    throw new Error('Unable to get source from node; check program configs.');
  }

  return getIdentifierDeclarationFromSource(node, source);
}

function getVariableValue(node) {
  if (ts.isStringLiteral(node) || ts.isNumericLiteral(node)) {
    return node.text;
  }

  if (ts.isObjectLiteralExpression(node)) {
    return serializeObject(node);
  }

  throw Error(`Unsuppored Node: cannot get value of node (${node.getText()}) of kind ${node.kind}`);
}

function serializeObject(node) {
  if (!ts.isObjectLiteralExpression(node)) {
    throw new Error(`Expecting Object literal Expression got ${node.getText()}`);
  }

  const value = {};

  for (const property of node.properties) {
    var _property$name;

    const propertyName = (_property$name = property.name) === null || _property$name === void 0 ? void 0 : _property$name.getText();

    if (typeof propertyName === 'undefined') {
      throw new Error(`Unable to get property name ${property.getText()}`);
    }

    if (ts.isPropertyAssignment(property)) {
      value[propertyName] = getVariableValue(property.initializer);
    } else {
      value[propertyName] = getVariableValue(property);
    }
  }

  return value;
}

function getResolvedModuleSourceFile(originalSource, program, importedModuleName) {
  const resolvedModule = originalSource.resolvedModules.get(importedModuleName);
  const resolvedModuleSourceFile = program.getSourceFile(resolvedModule.resolvedFileName);

  if (!resolvedModuleSourceFile) {
    throw new Error(`Unable to find resolved module ${importedModuleName}`);
  }

  return resolvedModuleSourceFile;
}

function getPropertyValue(node, program, config = {}) {
  const {
    chaseImport = false
  } = config;

  if (ts.isPropertyAssignment(node)) {
    const {
      initializer
    } = node;

    if (ts.isIdentifier(initializer)) {
      const identifierName = initializer.getText();
      const declaration = getIdentifierDeclaration(initializer);

      if (ts.isImportSpecifier(declaration)) {
        if (!chaseImport) {
          throw new Error(`Value of node ${identifierName} is imported from another file. Chasing imports is not allowed.`);
        }

        const importedModuleName = getModuleSpecifier(declaration);
        const source = node.getSourceFile();
        const declarationSource = getResolvedModuleSourceFile(source, program, importedModuleName);
        const declarationNode = getIdentifierDeclarationFromSource(initializer, declarationSource);

        if (!ts.isVariableDeclaration(declarationNode)) {
          throw new Error(`Expected ${identifierName} to be variable declaration.`);
        }

        if (!declarationNode.initializer) {
          throw new Error(`Expected ${identifierName} to be initialized.`);
        }

        const serializedObject = serializeObject(declarationNode.initializer);
        return serializedObject;
      }

      return getVariableValue(declaration);
    }

    return getVariableValue(initializer);
  }
}

function pickDeep(collection, identity, thisArg) {
  const picked = (0, _lodash.pick)(collection, identity, thisArg);
  const collections = (0, _lodash.pick)(collection, _lodash.isObject, thisArg);
  (0, _lodash.each)(collections, function (item, key) {
    let object;

    if ((0, _lodash.isArray)(item)) {
      object = (0, _lodash.reduce)(item, function (result, value) {
        const pickedDeep = pickDeep(value, identity, thisArg);

        if (!(0, _lodash.isEmpty)(pickedDeep)) {
          result.push(pickedDeep);
        }

        return result;
      }, []);
    } else {
      object = pickDeep(item, identity, thisArg);
    }

    if (!(0, _lodash.isEmpty)(object)) {
      picked[key || ''] = object;
    }
  });
  return picked;
}

const flattenKeys = (obj, keyPath = []) => {
  if ((0, _lodash.isObject)(obj)) {
    return (0, _lodash.reduce)(obj, (cum, next, key) => {
      const keys = [...keyPath, key];
      return (0, _lodash.merge)(cum, flattenKeys(next, keys));
    }, {});
  }

  return {
    [keyPath.join('.')]: obj
  };
};

exports.flattenKeys = flattenKeys;

function difference(actual, expected) {
  function changes(obj, base) {
    return (0, _lodash.transform)(obj, function (result, value, key) {
      if (key && !(0, _lodash.isEqual)(value, base[key])) {
        result[key] = (0, _lodash.isObject)(value) && (0, _lodash.isObject)(base[key]) ? changes(value, base[key]) : value;
      }
    });
  }

  return changes(actual, expected);
}

function normalizePath(inputPath) {
  return (0, _normalizePath.default)(path.relative('.', inputPath));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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