/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import { i18n } from '@kbn/i18n';
import { EuiBadge, EuiIcon, EuiToolTip, EuiScreenReaderOnly, EuiNotificationBadge } from '@elastic/eui';
import classNames from 'classnames';
import React from 'react';
import { PanelOptionsMenu } from './panel_options_menu';

function renderBadges(badges, embeddable) {
  return badges.map(function (badge) {
    return /*#__PURE__*/React.createElement(EuiBadge, {
      key: badge.id,
      className: "embPanel__headerBadge",
      iconType: badge.getIconType({
        embeddable: embeddable
      }),
      onClick: function onClick() {
        return badge.execute({
          embeddable: embeddable
        });
      },
      onClickAriaLabel: badge.getDisplayName({
        embeddable: embeddable
      })
    }, badge.getDisplayName({
      embeddable: embeddable
    }));
  });
}

function renderNotifications(notifications, embeddable) {
  return notifications.map(function (notification) {
    var context = {
      embeddable: embeddable
    };
    var badge = /*#__PURE__*/React.createElement(EuiNotificationBadge, {
      "data-test-subj": "embeddablePanelNotification-".concat(notification.id),
      key: notification.id,
      style: {
        marginTop: '4px',
        marginRight: '4px'
      },
      onClick: function onClick() {
        return notification.execute(context);
      }
    }, notification.getDisplayName(context));

    if (notification.getDisplayNameTooltip) {
      var tooltip = notification.getDisplayNameTooltip(context);

      if (tooltip) {
        badge = /*#__PURE__*/React.createElement(EuiToolTip, {
          position: "top",
          delay: "regular",
          content: tooltip,
          key: notification.id
        }, badge);
      }
    }

    return badge;
  });
}

function renderTooltip(description) {
  return description !== '' && /*#__PURE__*/React.createElement(EuiToolTip, {
    content: description,
    delay: "regular",
    position: "right"
  }, /*#__PURE__*/React.createElement(EuiIcon, {
    type: "iInCircle"
  }));
}

var VISUALIZE_EMBEDDABLE_TYPE = 'visualization';

function getViewDescription(embeddable) {
  if (embeddable.type === VISUALIZE_EMBEDDABLE_TYPE) {
    var description = embeddable.getVisualizationDescription();

    if (description) {
      return description;
    }
  }

  return '';
}

export function PanelHeader(_ref) {
  var title = _ref.title,
      isViewMode = _ref.isViewMode,
      hidePanelTitles = _ref.hidePanelTitles,
      getActionContextMenuPanel = _ref.getActionContextMenuPanel,
      closeContextMenu = _ref.closeContextMenu,
      badges = _ref.badges,
      notifications = _ref.notifications,
      embeddable = _ref.embeddable,
      headerId = _ref.headerId;
  var viewDescription = getViewDescription(embeddable);
  var showTitle = !isViewMode || title && !hidePanelTitles || viewDescription !== '';
  var showPanelBar = badges.length > 0 || showTitle;
  var classes = classNames('embPanel__header', {
    'embPanel__header--floater': !showPanelBar
  });

  if (!showPanelBar) {
    return /*#__PURE__*/React.createElement("div", {
      className: classes
    }, /*#__PURE__*/React.createElement(PanelOptionsMenu, {
      getActionContextMenuPanel: getActionContextMenuPanel,
      isViewMode: isViewMode,
      closeContextMenu: closeContextMenu,
      title: title
    }));
  }

  return /*#__PURE__*/React.createElement("figcaption", {
    className: classes,
    "data-test-subj": "embeddablePanelHeading-".concat((title || '').replace(/\s/g, ''))
  }, /*#__PURE__*/React.createElement("h2", {
    id: headerId,
    "data-test-subj": "dashboardPanelTitle",
    className: "embPanel__title embPanel__dragger"
  }, showTitle ? /*#__PURE__*/React.createElement("span", {
    className: "embPanel__titleInner"
  }, /*#__PURE__*/React.createElement("span", {
    className: "embPanel__titleText",
    "aria-hidden": "true"
  }, title), /*#__PURE__*/React.createElement(EuiScreenReaderOnly, null, /*#__PURE__*/React.createElement("span", null, i18n.translate('embeddableApi.panel.enhancedDashboardPanelAriaLabel', {
    defaultMessage: 'Dashboard panel: {title}',
    values: {
      title: title
    }
  }))), renderTooltip(viewDescription)) : /*#__PURE__*/React.createElement(EuiScreenReaderOnly, null, /*#__PURE__*/React.createElement("span", null, i18n.translate('embeddableApi.panel.dashboardPanelAriaLabel', {
    defaultMessage: 'Dashboard panel'
  }))), renderBadges(badges, embeddable)), renderNotifications(notifications, embeddable), /*#__PURE__*/React.createElement(PanelOptionsMenu, {
    isViewMode: isViewMode,
    getActionContextMenuPanel: getActionContextMenuPanel,
    closeContextMenu: closeContextMenu,
    title: title
  }));
}