/*	$OpenBSD: if_rgereg.h,v 1.2 2020/07/22 00:48:02 kevlo Exp $	*/

/*
 * Copyright (c) 2019, 2020 Kevin Lo <kevlo@openbsd.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#define RGE_PCI_BAR0		PCI_MAPREG_START
#define RGE_PCI_BAR1		(PCI_MAPREG_START + 4)
#define RGE_PCI_BAR2		(PCI_MAPREG_START + 8)

#define RGE_MAC0		0x0000
#define RGE_MAC4		0x0004
#define RGE_MAR0		0x0008
#define RGE_MAR4		0x000c
#define RGE_TXDESC_ADDR_LO	0x0020
#define RGE_TXDESC_ADDR_HI	0x0024
#define RGE_CMD			0x0037
#define RGE_IMR			0x0038
#define RGE_ISR			0x003c
#define RGE_TXCFG		0x0040
#define RGE_RXCFG		0x0044
#define RGE_TIMERCNT		0x0048
#define RGE_EECMD		0x0050
#define RGE_CFG0		0x0051
#define RGE_CFG1		0x0052
#define RGE_CFG2		0x0053
#define RGE_CFG3		0x0054
#define RGE_CFG4		0x0055
#define RGE_CFG5		0x0056
#define RGE_TDFNR		0x0057
#define RGE_TIMERINT0		0x0058
#define RGE_TIMERINT1		0x005c
#define RGE_CSIDR		0x0064
#define RGE_CSIAR		0x0068
#define RGE_PHYSTAT		0x006c
#define RGE_PMCH		0x006f
#define RGE_EPHYAR		0x0080
#define RGE_TIMERINT2		0x008c
#define RGE_TXSTART		0x0090
#define RGE_MACOCP		0x00b0
#define RGE_PHYOCP		0x00b8
#define RGE_DLLPR		0x00d0
#define RGE_TWICMD		0x00d2
#define RGE_MCUCMD		0x00d3
#define RGE_RXMAXSIZE		0x00da
#define RGE_CPLUSCMD		0x00e0
#define RGE_IM			0x00e2
#define RGE_RXDESC_ADDR_LO	0x00e4
#define RGE_RXDESC_ADDR_HI	0x00e8
#define RGE_PPSW		0x00f2
#define RGE_TIMERINT3		0x00f4
#define RGE_INTMITI(i)		(0x0a00 + (i) * 4)
#define RGE_PHYBASE		0x0a40
#define RGE_ADDR0		0x19e0
#define RGE_ADDR1		0x19e4
#define RGE_EEE_TXIDLE_TIMER	0x6048

/* Flags for register RGE_CMD */
#define RGE_CMD_RXBUF_EMPTY	0x01
#define RGE_CMD_TXENB		0x04
#define RGE_CMD_RXENB		0x08
#define RGE_CMD_RESET		0x10

/* Flags for register RGE_ISR */
#define RGE_ISR_RX_OK		0x00000001
#define RGE_ISR_RX_ERR		0x00000002
#define RGE_ISR_TX_OK		0x00000004
#define RGE_ISR_TX_ERR		0x00000008
#define RGE_ISR_RX_DESC_UNAVAIL	0x00000010
#define RGE_ISR_LINKCHG		0x00000020 
#define RGE_ISR_RX_FIFO_OFLOW	0x00000040
#define RGE_ISR_TX_DESC_UNAVAIL	0x00000080
#define RGE_ISR_SWI		0x00000100
#define RGE_ISR_PCS_TIMEOUT	0x00004000
#define RGE_ISR_SYSTEM_ERR	0x00008000

#define RGE_INTRS		\
	(RGE_ISR_RX_OK | RGE_ISR_RX_ERR | RGE_ISR_TX_OK |		\
	RGE_ISR_TX_ERR | RGE_ISR_RX_DESC_UNAVAIL | RGE_ISR_LINKCHG |	\
	RGE_ISR_TX_DESC_UNAVAIL | RGE_ISR_PCS_TIMEOUT | RGE_ISR_SYSTEM_ERR)

#define RGE_INTRS_TIMER		\
	(RGE_ISR_RX_ERR | RGE_ISR_TX_ERR | RGE_ISR_PCS_TIMEOUT |	\
	RGE_ISR_SYSTEM_ERR)

/* Flags for register RGE_TXCFG */
#define RGE_TXCFG_HWREV		0x7cf00000

/* Flags for register RGE_RXCFG */
#define RGE_RXCFG_ALLPHYS	0x00000001
#define RGE_RXCFG_INDIV		0x00000002
#define RGE_RXCFG_MULTI		0x00000004
#define RGE_RXCFG_BROAD		0x00000008
#define RGE_RXCFG_RUNT		0x00000010
#define RGE_RXCFG_ERRPKT	0x00000020
#define RGE_RXCFG_VLANSTRIP	0x00c00000

/* Flags for register RGE_EECMD */
#define RGE_EECMD_WRITECFG	0xc0

/* Flags for register RGE_CFG1 */
#define RGE_CFG1_SPEED_DOWN	0x10

/* Flags for register RGE_CFG2 */
#define RGE_CFG2_CLKREQ_EN	0x80

/* Flags for register RGE_CFG3 */
#define RGE_CFG3_RDY_TO_L23	0x02

/* Flags for register RGE_CFG5 */
#define RGE_CFG5_PME_STS	0x01

/* Flags for register RGE_CSIAR */
#define RGE_CSIAR_BYTE_EN	0x0000000f
#define RGE_CSIAR_BYTE_EN_SHIFT	12
#define RGE_CSIAR_ADDR_MASK	0x00000fff
#define RGE_CSIAR_BUSY		0x80000000

/* Flags for register RGE_PHYSTAT */
#define RGE_PHYSTAT_FDX		0x0001
#define RGE_PHYSTAT_LINK	0x0002
#define RGE_PHYSTAT_10MBPS	0x0004
#define RGE_PHYSTAT_100MBPS	0x0008
#define RGE_PHYSTAT_1000MBPS	0x0010
#define RGE_PHYSTAT_RXFLOW	0x0020
#define RGE_PHYSTAT_TXFLOW	0x0040
#define RGE_PHYSTAT_2500MBPS	0x0400

/* Flags for register RGE_EPHYAR */
#define RGE_EPHYAR_DATA_MASK	0x0000ffff
#define RGE_EPHYAR_BUSY		0x80000000
#define RGE_EPHYAR_ADDR_MASK	0x0000007f
#define RGE_EPHYAR_ADDR_SHIFT	16

/* Flags for register RGE_TXSTART */
#define RGE_TXSTART_START	0x0001

/* Flags for register RGE_MACOCP */
#define RGE_MACOCP_DATA_MASK	0x0000ffff
#define RGE_MACOCP_BUSY		0x80000000
#define RGE_MACOCP_ADDR_SHIFT	16

/* Flags for register RGE_PHYOCP */
#define RGE_PHYOCP_DATA_MASK	0x0000ffff
#define RGE_PHYOCP_BUSY		0x80000000
#define RGE_PHYOCP_ADDR_SHIFT	16

/* Flags for register RGE_DLLPR. */
#define RGE_DLLPR_PFM_EN	0x40
#define RGE_DLLPR_TX_10M_PS_EN	0x80

/* Flags for register RGE_MCUCMD */
#define RGE_MCUCMD_RXFIFO_EMPTY	0x10
#define RGE_MCUCMD_TXFIFO_EMPTY	0x20
#define RGE_MCUCMD_IS_OOB	0x80

/* Flags for register RGE_CPLUSCMD */
#define RGE_CPLUSCMD_RXCSUM	0x0020

#define RGE_TX_NSEGS		32
#define	RGE_TX_LIST_CNT		1024
#define	RGE_RX_LIST_CNT		1024
#define RGE_ALIGN		256
#define RGE_TX_LIST_SZ		(sizeof(struct rge_tx_desc) * RGE_TX_LIST_CNT)
#define RGE_RX_LIST_SZ		(sizeof(struct rge_rx_desc) * RGE_RX_LIST_CNT)
#define RGE_NEXT_TX_DESC(x)	(((x) + 1) % RGE_TX_LIST_CNT)
#define RGE_NEXT_RX_DESC(x)	(((x) + 1) % RGE_RX_LIST_CNT)
#define RGE_ADDR_LO(y)		((uint64_t) (y) & 0xffffffff)
#define RGE_ADDR_HI(y)		((uint64_t) (y) >> 32)
#define RGE_OWN(x)		(letoh32((x)->rge_cmdsts) & RGE_RDCMDSTS_OWN)
#define RGE_RXBYTES(x)          (letoh32((x)->rge_cmdsts) & \
				RGE_RDCMDSTS_FRAGLEN)

#define RGE_ADV_2500TFDX	0x0080

/* Tx descriptor */
struct rge_tx_desc {
	uint32_t		rge_cmdsts;
	uint32_t		rge_extsts;
	uint32_t		rge_addrlo;
	uint32_t		rge_addrhi;
	uint32_t		reserved[4];
};

#define RGE_TDCMDSTS_COLL	0x000f0000
#define RGE_TDCMDSTS_EXCESSCOLL	0x00100000
#define RGE_TDCMDSTS_TXERR	0x00800000
#define RGE_TDCMDSTS_EOF	0x10000000
#define RGE_TDCMDSTS_SOF	0x20000000
#define RGE_TDCMDSTS_EOR	0x40000000
#define RGE_TDCMDSTS_OWN	0x80000000

#define RGE_TDEXTSTS_VTAG	0x00020000
#define RGE_TDEXTSTS_IPCSUM	0x20000000
#define RGE_TDEXTSTS_TCPCSUM	0x40000000
#define RGE_TDEXTSTS_UDPCSUM	0x80000000

/* Rx descriptor */
struct rge_rx_desc {
	uint32_t		rge_cmdsts;
	uint32_t		rge_extsts;
	uint32_t		rge_addrlo;
	uint32_t		rge_addrhi;
};

#define RGE_RDCMDSTS_TCPCSUMERR	0x00004000
#define RGE_RDCMDSTS_UDPCSUMERR	0x00008000
#define RGE_RDCMDSTS_IPCSUMERR	0x00010000
#define RGE_RDCMDSTS_TCPPKT	0x00020000
#define RGE_RDCMDSTS_UDPPKT	0x00040000
#define RGE_RDCMDSTS_RXERRSUM	0x00200000
#define RGE_RDCMDSTS_EOF	0x10000000
#define RGE_RDCMDSTS_SOF	0x20000000
#define RGE_RDCMDSTS_EOR	0x40000000
#define RGE_RDCMDSTS_OWN	0x80000000
#define RGE_RDCMDSTS_FRAGLEN	0x00003fff

#define RGE_RDEXTSTS_VTAG	0x00010000
#define RGE_RDEXTSTS_VLAN_MASK	0x0000ffff
#define RGE_RDEXTSTS_IPV4	0x40000000
#define RGE_RDEXTSTS_IPV6	0x80000000

struct rge_txq {
	struct mbuf		*txq_mbuf;
	bus_dmamap_t		txq_dmamap;
	int			txq_descidx;
};

struct rge_rxq {
	struct mbuf		*rxq_mbuf;
	bus_dmamap_t		rxq_dmamap;
};

struct rge_list_data {
	struct rge_txq		rge_txq[RGE_TX_LIST_CNT];
	int			rge_txq_prodidx;
	int			rge_txq_considx;
	struct rge_rxq		rge_rxq[RGE_RX_LIST_CNT];
	int			rge_rxq_prodidx;
	int			rge_rxq_considx;

	bus_dma_segment_t	rge_tx_listseg;
	int			rge_tx_listnseg;
	bus_dmamap_t		rge_tx_list_map;
	struct rge_tx_desc	*rge_tx_list;
	struct if_rxring	rge_rx_ring;
	bus_dma_segment_t	rge_rx_listseg;
	int			rge_rx_listnseg;
	bus_dmamap_t		rge_rx_list_map;
	struct rge_rx_desc	*rge_rx_list;
};

/* Microcode version */
#define RGE_MAC_CFG2_MCODE_VER	0x0b11
#define RGE_MAC_CFG3_MCODE_VER	0x0b33
#define RGE_MAC_CFG4_MCODE_VER	0x0b17
#define RGE_MAC_CFG5_MCODE_VER	0x0b36

enum rge_mac_type {
	MAC_CFG_UNKNOWN = 1,
	MAC_CFG2,
	MAC_CFG3,
	MAC_CFG4,
	MAC_CFG5
};

#define RGE_TIMEOUT		100

#define RGE_JUMBO_FRAMELEN	9216
#define RGE_JUMBO_MTU							\
	(RGE_JUMBO_FRAMELEN - ETHER_HDR_LEN - ETHER_CRC_LEN - 		\
	ETHER_VLAN_ENCAP_LEN)

#define RGE_TXCFG_CONFIG	0x03000700
#define RGE_RXCFG_CONFIG	0x40c00700

struct rge_softc {
	struct device		sc_dev;
	struct arpcom		sc_arpcom;	/* Ethernet common data */
	void			*sc_ih;		/* interrupt vectoring */
	bus_space_handle_t	rge_bhandle;	/* bus space handle */
	bus_space_tag_t		rge_btag;	/* bus space tag */
	bus_size_t		rge_bsize;
	bus_dma_tag_t		sc_dmat;
	pci_chipset_tag_t	sc_pc;
	pcitag_t		sc_tag;
	bus_dma_segment_t 	sc_rx_seg;
	bus_dmamap_t		sc_rx_dmamap;
	struct ifmedia		sc_media;	/* media info */
	enum rge_mac_type	rge_type;
	struct mbuf		*rge_head;
	struct mbuf		*rge_tail;

	struct rge_list_data	rge_ldata;

	struct task		sc_task;

	struct timeout		sc_timeout;	/* tick timeout */

	uint16_t		rge_mcodever;
	uint32_t		rge_flags;
#define RGE_FLAG_MSI		0x00000001

	uint32_t		rge_intrs;
	uint32_t		rge_tx_ack;
	uint32_t		rge_rx_ack;
	int			rge_rxbufsz;
	int			rge_timerintr;
#define RGE_IMTYPE_NONE		0
#define RGE_IMTYPE_SIM		1
};

/*
 * Register space access macros.
 */
#define RGE_WRITE_4(sc, reg, val)	\
	bus_space_write_4(sc->rge_btag, sc->rge_bhandle, reg, val)
#define RGE_WRITE_2(sc, reg, val)	\
	bus_space_write_2(sc->rge_btag, sc->rge_bhandle, reg, val)
#define RGE_WRITE_1(sc, reg, val)	\
	bus_space_write_1(sc->rge_btag, sc->rge_bhandle, reg, val)

#define RGE_READ_4(sc, reg)		\
	bus_space_read_4(sc->rge_btag, sc->rge_bhandle, reg)
#define RGE_READ_2(sc, reg)		\
	bus_space_read_2(sc->rge_btag, sc->rge_bhandle, reg)
#define RGE_READ_1(sc, reg)		\
	bus_space_read_1(sc->rge_btag, sc->rge_bhandle, reg)

#define RGE_SETBIT_4(sc, reg, val)	\
	RGE_WRITE_4(sc, reg, RGE_READ_4(sc, reg) | (val))
#define RGE_SETBIT_2(sc, reg, val)	\
	RGE_WRITE_2(sc, reg, RGE_READ_2(sc, reg) | (val))
#define RGE_SETBIT_1(sc, reg, val)	\
	RGE_WRITE_1(sc, reg, RGE_READ_1(sc, reg) | (val))

#define RGE_CLRBIT_4(sc, reg, val)	\
	RGE_WRITE_4(sc, reg, RGE_READ_4(sc, reg) & ~(val))
#define RGE_CLRBIT_2(sc, reg, val)	\
	RGE_WRITE_2(sc, reg, RGE_READ_2(sc, reg) & ~(val))
#define RGE_CLRBIT_1(sc, reg, val)	\
	RGE_WRITE_1(sc, reg, RGE_READ_1(sc, reg) & ~(val))

#define RGE_EPHY_SETBIT(sc, reg, val)	\
	rge_write_ephy(sc, reg, rge_read_ephy(sc, reg) | (val))

#define RGE_EPHY_CLRBIT(sc, reg, val)	\
	rge_write_ephy(sc, reg, rge_read_ephy(sc, reg) & ~(val))

#define RGE_PHY_SETBIT(sc, reg, val)	\
	rge_write_phy_ocp(sc, reg, rge_read_phy_ocp(sc, reg) | (val))

#define RGE_PHY_CLRBIT(sc, reg, val)	\
	rge_write_phy_ocp(sc, reg, rge_read_phy_ocp(sc, reg) & ~(val))

#define RGE_MAC_SETBIT(sc, reg, val)	\
	rge_write_mac_ocp(sc, reg, rge_read_mac_ocp(sc, reg) | (val))

#define RGE_MAC_CLRBIT(sc, reg, val)	\
	rge_write_mac_ocp(sc, reg, rge_read_mac_ocp(sc, reg) & ~(val))

static const struct {
	uint16_t	reg;
	uint16_t	val;
} rtl8125_mac_bps[] = {
	{ 0xf800, 0xe008 }, { 0xf802, 0xe01e }, { 0xf804, 0xe02e }, 
	{ 0xf806, 0xe054 }, { 0xf808, 0xe057 }, { 0xf80a, 0xe059 }, 
	{ 0xf80c, 0xe0c2 }, { 0xf80e, 0xe0cb }, { 0xf810, 0x9996 }, 
	{ 0xf812, 0x49d1 }, { 0xf814, 0xf005 }, { 0xf816, 0x49d4 }, 
	{ 0xf818, 0xf10a }, { 0xf81a, 0x49d8 }, { 0xf81c, 0xf108 }, 
	{ 0xf81e, 0xc00f }, { 0xf820, 0x7100 }, { 0xf822, 0x209c }, 
	{ 0xf824, 0x249c }, { 0xf826, 0xc009 }, { 0xf828, 0x9900 }, 
	{ 0xf82a, 0xe004 }, { 0xf82c, 0xc006 }, { 0xf82e, 0x1900 }, 
	{ 0xf830, 0x9900 }, { 0xf832, 0xc602 }, { 0xf834, 0xbe00 }, 
	{ 0xf836, 0x5a48 }, { 0xf838, 0xe0c2 }, { 0xf83a, 0x0004 }, 
	{ 0xf83c, 0xe10a }, { 0xf83e, 0xc60f }, { 0xf840, 0x73c4 }, 
	{ 0xf842, 0x49b3 }, { 0xf844, 0xf106 }, { 0xf846, 0x73c2 }, 
	{ 0xf848, 0xc608 }, { 0xf84a, 0xb406 }, { 0xf84c, 0xc609 }, 
	{ 0xf84e, 0xff80 }, { 0xf850, 0xc605 }, { 0xf852, 0xb406 }, 
	{ 0xf854, 0xc605 }, { 0xf856, 0xff80 }, { 0xf858, 0x0544 }, 
	{ 0xf85a, 0x0568 }, { 0xf85c, 0xe906 }, { 0xf85e, 0xcde8 }, 
	{ 0xf860, 0xc724 }, { 0xf862, 0xc624 }, { 0xf864, 0x9ee2 }, 
	{ 0xf866, 0x1e01 }, { 0xf868, 0x9ee0 }, { 0xf86a, 0x76e0 }, 
	{ 0xf86c, 0x49e0 }, { 0xf86e, 0xf1fe }, { 0xf870, 0x76e6 }, 
	{ 0xf872, 0x486d }, { 0xf874, 0x4868 }, { 0xf876, 0x9ee4 }, 
	{ 0xf878, 0x1e03 }, { 0xf87a, 0x9ee0 }, { 0xf87c, 0x76e0 }, 
	{ 0xf87e, 0x49e0 }, { 0xf880, 0xf1fe }, { 0xf882, 0xc615 }, 
	{ 0xf884, 0x9ee2 }, { 0xf886, 0x1e01 }, { 0xf888, 0x9ee0 }, 
	{ 0xf88a, 0x76e0 }, { 0xf88c, 0x49e0 }, { 0xf88e, 0xf1fe }, 
	{ 0xf890, 0x76e6 }, { 0xf892, 0x486f }, { 0xf894, 0x9ee4 }, 
	{ 0xf896, 0x1e03 }, { 0xf898, 0x9ee0 }, { 0xf89a, 0x76e0 }, 
	{ 0xf89c, 0x49e0 }, { 0xf89e, 0xf1fe }, { 0xf8a0, 0x7196 }, 
	{ 0xf8a2, 0xc702 }, { 0xf8a4, 0xbf00 }, { 0xf8a6, 0x5a44 }, 
	{ 0xf8a8, 0xeb0e }, { 0xf8aa, 0x0070 }, { 0xf8ac, 0x00c3 }, 
	{ 0xf8ae, 0x1bc0 }, { 0xf8b0, 0xc602 }, { 0xf8b2, 0xbe00 }, 
	{ 0xf8b4, 0x0e26 }, { 0xf8b6, 0xc602 }, { 0xf8b8, 0xbe00 }, 
	{ 0xf8ba, 0x0eba }, { 0xf8bc, 0x1501 }, { 0xf8be, 0xf02a }, 
	{ 0xf8c0, 0x1500 }, { 0xf8c2, 0xf15d }, { 0xf8c4, 0xc661 }, 
	{ 0xf8c6, 0x75c8 }, { 0xf8c8, 0x49d5 }, { 0xf8ca, 0xf00a }, 
	{ 0xf8cc, 0x49d6 }, { 0xf8ce, 0xf008 }, { 0xf8d0, 0x49d7 }, 
	{ 0xf8d2, 0xf006 }, { 0xf8d4, 0x49d8 }, { 0xf8d6, 0xf004 }, 
	{ 0xf8d8, 0x75d2 }, { 0xf8da, 0x49d9 }, { 0xf8dc, 0xf150 }, 
	{ 0xf8de, 0xc553 }, { 0xf8e0, 0x77a0 }, { 0xf8e2, 0x75c8 }, 
	{ 0xf8e4, 0x4855 }, { 0xf8e6, 0x4856 }, { 0xf8e8, 0x4857 }, 
	{ 0xf8ea, 0x4858 }, { 0xf8ec, 0x48da }, { 0xf8ee, 0x48db }, 
	{ 0xf8f0, 0x49fe }, { 0xf8f2, 0xf002 }, { 0xf8f4, 0x485a }, 
	{ 0xf8f6, 0x49ff }, { 0xf8f8, 0xf002 }, { 0xf8fa, 0x485b }, 
	{ 0xf8fc, 0x9dc8 }, { 0xf8fe, 0x75d2 }, { 0xf900, 0x4859 }, 
	{ 0xf902, 0x9dd2 }, { 0xf904, 0xc643 }, { 0xf906, 0x75c0 }, 
	{ 0xf908, 0x49d4 }, { 0xf90a, 0xf033 }, { 0xf90c, 0x49d0 }, 
	{ 0xf90e, 0xf137 }, { 0xf910, 0xe030 }, { 0xf912, 0xc63a }, 
	{ 0xf914, 0x75c8 }, { 0xf916, 0x49d5 }, { 0xf918, 0xf00e }, 
	{ 0xf91a, 0x49d6 }, { 0xf91c, 0xf00c }, { 0xf91e, 0x49d7 }, 
	{ 0xf920, 0xf00a }, { 0xf922, 0x49d8 }, { 0xf924, 0xf008 }, 
	{ 0xf926, 0x75d2 }, { 0xf928, 0x49d9 }, { 0xf92a, 0xf005 }, 
	{ 0xf92c, 0xc62e }, { 0xf92e, 0x75c0 }, { 0xf930, 0x49d7 }, 
	{ 0xf932, 0xf125 }, { 0xf934, 0xc528 }, { 0xf936, 0x77a0 }, 
	{ 0xf938, 0xc627 }, { 0xf93a, 0x75c8 }, { 0xf93c, 0x4855 }, 
	{ 0xf93e, 0x4856 }, { 0xf940, 0x4857 }, { 0xf942, 0x4858 }, 
	{ 0xf944, 0x48da }, { 0xf946, 0x48db }, { 0xf948, 0x49fe }, 
	{ 0xf94a, 0xf002 }, { 0xf94c, 0x485a }, { 0xf94e, 0x49ff }, 
	{ 0xf950, 0xf002 }, { 0xf952, 0x485b }, { 0xf954, 0x9dc8 }, 
	{ 0xf956, 0x75d2 }, { 0xf958, 0x4859 }, { 0xf95a, 0x9dd2 }, 
	{ 0xf95c, 0xc616 }, { 0xf95e, 0x75c0 }, { 0xf960, 0x4857 }, 
	{ 0xf962, 0x9dc0 }, { 0xf964, 0xc613 }, { 0xf966, 0x75c0 }, 
	{ 0xf968, 0x49da }, { 0xf96a, 0xf003 }, { 0xf96c, 0x49d0 }, 
	{ 0xf96e, 0xf107 }, { 0xf970, 0xc60b }, { 0xf972, 0xc50e }, 
	{ 0xf974, 0x48d9 }, { 0xf976, 0x9dc0 }, { 0xf978, 0x4859 }, 
	{ 0xf97a, 0x9dc0 }, { 0xf97c, 0xc608 }, { 0xf97e, 0xc702 }, 
	{ 0xf980, 0xbf00 }, { 0xf982, 0x3ae0 }, { 0xf984, 0xe860 }, 
	{ 0xf986, 0xb400 }, { 0xf988, 0xb5d4 }, { 0xf98a, 0xe908 }, 
	{ 0xf98c, 0xe86c }, { 0xf98e, 0x1200 }, { 0xf990, 0xc409 }, 
	{ 0xf992, 0x6780 }, { 0xf994, 0x48f1 }, { 0xf996, 0x8f80 }, 
	{ 0xf998, 0xc404 }, { 0xf99a, 0xc602 }, { 0xf99c, 0xbe00 }, 
	{ 0xf99e, 0x10aa }, { 0xf9a0, 0xc010 }, { 0xf9a2, 0xea7c }, 
	{ 0xf9a4, 0xc602 }, { 0xf9a6, 0xbe00 }, { 0xf9a8, 0x0000 }, 
	{ 0xfc26, 0x8000 }, { 0xfc2a, 0x0540 }, { 0xfc2e, 0x0e24 },
	{ 0xfc30, 0x0eb8 }, { 0xfc32, 0x3a5c }, { 0xfc34, 0x10a8 },
	{ 0xfc48, 0x007a }
}, rtl8125b_mac_bps[] = {
	{ 0xf800, 0xe008 }, { 0xf802, 0xe013 }, { 0xf804, 0xe01e }, 
	{ 0xf806, 0xe02f }, { 0xf808, 0xe035 }, { 0xf80a, 0xe04f }, 
	{ 0xf80c, 0xe053 }, { 0xf80e, 0xe055 }, { 0xf810, 0x740a }, 
	{ 0xf812, 0x4846 }, { 0xf814, 0x4847 }, { 0xf816, 0x9c0a }, 
	{ 0xf818, 0xc607 }, { 0xf81a, 0x74c0 }, { 0xf81c, 0x48c6 }, 
	{ 0xf81e, 0x9cc0 }, { 0xf820, 0xc602 }, { 0xf822, 0xbe00 }, 
	{ 0xf824, 0x13f0 }, { 0xf826, 0xe054 }, { 0xf828, 0x72ca }, 
	{ 0xf82a, 0x4826 }, { 0xf82c, 0x4827 }, { 0xf82e, 0x9aca }, 
	{ 0xf830, 0xc607 }, { 0xf832, 0x72c0 }, { 0xf834, 0x48a6 }, 
	{ 0xf836, 0x9ac0 }, { 0xf838, 0xc602 }, { 0xf83a, 0xbe00 }, 
	{ 0xf83c, 0x081c }, { 0xf83e, 0xe054 }, { 0xf840, 0xc60f }, 
	{ 0xf842, 0x74c4 }, { 0xf844, 0x49cc }, { 0xf846, 0xf109 }, 
	{ 0xf848, 0xc60c }, { 0xf84a, 0x74ca }, { 0xf84c, 0x48c7 }, 
	{ 0xf84e, 0x9cca }, { 0xf850, 0xc609 }, { 0xf852, 0x74c0 }, 
	{ 0xf854, 0x4846 }, { 0xf856, 0x9cc0 }, { 0xf858, 0xc602 }, 
	{ 0xf85a, 0xbe00 }, { 0xf85c, 0x2494 }, { 0xf85e, 0xe092 }, 
	{ 0xf860, 0xe0c0 }, { 0xf862, 0xe054 }, { 0xf864, 0x7420 }, 
	{ 0xf866, 0x48c0 }, { 0xf868, 0x9c20 }, { 0xf86a, 0x7444 }, 
	{ 0xf86c, 0xc602 }, { 0xf86e, 0xbe00 }, { 0xf870, 0x12dc }, 
	{ 0xf872, 0x733a }, { 0xf874, 0x21b5 }, { 0xf876, 0x25bc }, 
	{ 0xf878, 0x1304 }, { 0xf87a, 0xf111 }, { 0xf87c, 0x1b12 }, 
	{ 0xf87e, 0x1d2a }, { 0xf880, 0x3168 }, { 0xf882, 0x3ada }, 
	{ 0xf884, 0x31ab }, { 0xf886, 0x1a00 }, { 0xf888, 0x9ac0 }, 
	{ 0xf88a, 0x1300 }, { 0xf88c, 0xf1fb }, { 0xf88e, 0x7620 }, 
	{ 0xf890, 0x236e }, { 0xf892, 0x276f }, { 0xf894, 0x1a3c }, 
	{ 0xf896, 0x22a1 }, { 0xf898, 0x41b5 }, { 0xf89a, 0x9ee2 }, 
	{ 0xf89c, 0x76e4 }, { 0xf89e, 0x486f }, { 0xf8a0, 0x9ee4 }, 
	{ 0xf8a2, 0xc602 }, { 0xf8a4, 0xbe00 }, { 0xf8a6, 0x4a26 }, 
	{ 0xf8a8, 0x733a }, { 0xf8aa, 0x49bb }, { 0xf8ac, 0xc602 }, 
	{ 0xf8ae, 0xbe00 }, { 0xf8b0, 0x47a2 }, { 0xf8b2, 0xc602 }, 
	{ 0xf8b4, 0xbe00 }, { 0xf8b6, 0x0000 }, { 0xf8b8, 0xc602 }, 
	{ 0xf8ba, 0xbe00 }, { 0xf8bc, 0x0000 }, { 0xfc26, 0x8000 },
	{ 0xfc28, 0x13e6 }, { 0xfc2a, 0x0812 }, { 0xfc2c, 0x248c },
	{ 0xfc2e, 0x12da }, { 0xfc30, 0x4a20 }, { 0xfc32, 0x47a0 },
	{ 0xfc48, 0x003f }
};

static const struct {
	uint16_t	reg;
	uint16_t	val;
} rtl8125_mac_cfg2_ephy[] = {
	{ 0x0001, 0xa812 }, { 0x0009, 0x520c }, { 0x0004, 0xd000 },
	{ 0x000d, 0xf702 }, { 0x000a, 0x8653 }, { 0x0006, 0x001e },
	{ 0x0008, 0x3595 }, { 0x0020, 0x9455 }, { 0x0021, 0x99ff },
	{ 0x0002, 0x6046 }, { 0x0029, 0xfe00 }, { 0x0023, 0xab62 },
	{ 0x0041, 0xa80c }, { 0x0049, 0x520c }, { 0x0044, 0xd000 },
	{ 0x004d, 0xf702 }, { 0x004a, 0x8653 }, { 0x0046, 0x001e },
	{ 0x0048, 0x3595 }, { 0x0060, 0x9455 }, { 0x0061, 0x99ff },
	{ 0x0042, 0x6046 }, { 0x0069, 0xfe00 }, { 0x0063, 0xab62 }
}, rtl8125_mac_cfg3_ephy[] = {
	{ 0x0004, 0xd000 }, { 0x000a, 0x8653 }, { 0x0023, 0xab66 },
	{ 0x0020, 0x9455 }, { 0x0021, 0x99ff }, { 0x0029, 0xfe04 },
	{ 0x0044, 0xd000 }, { 0x004a, 0x8653 }, { 0x0063, 0xab66 },
	{ 0x0060, 0x9455 }, { 0x0061, 0x99ff }, { 0x0069, 0xfe04 }
}, rtl8125_mac_cfg4_ephy[] = {
	{ 0x0006, 0x001f }, { 0x000a, 0xb66b }, { 0x0001, 0xa852 },
	{ 0x0024, 0x0008 }, { 0x002f, 0x6052 }, { 0x000d, 0xf716 },
	{ 0x0020, 0xd477 }, { 0x0021, 0x4477 }, { 0x0022, 0x0013 },
	{ 0x0023, 0xbb66 }, { 0x000b, 0xa909 }, { 0x0029, 0xff04 },
	{ 0x001b, 0x1ea0 }, { 0x0046, 0x001f }, { 0x004a, 0xb66b },
	{ 0x0041, 0xa84a }, { 0x0064, 0x000c }, { 0x006f, 0x604a },
	{ 0x004d, 0xf716 }, { 0x0060, 0xd477 }, { 0x0061, 0x4477 },
	{ 0x0062, 0x0013 }, { 0x0063, 0xbb66 }, { 0x004b, 0xa909 },
	{ 0x0069, 0xff04 }, { 0x005b, 0x1ea0 }
}, rtl8125_mac_cfg5_ephy[] = {
	{ 0x000b, 0xa908 }, { 0x001e, 0x20eb }, { 0x004b, 0xa908 },
	{ 0x005e, 0x20eb }
};

#define RTL8125_MAC_CFG2_MCU	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8013 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8021 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x803d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8042 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8051 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8051 },	\
	{ 0xa438, 0xa088 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0a50 },	\
	{ 0xa438, 0x8008 },	\
	{ 0xa438, 0xd014 },	\
	{ 0xa438, 0xd1a3 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x401a },	\
	{ 0xa438, 0xd707 },	\
	{ 0xa438, 0x40c2 },	\
	{ 0xa438, 0x60a6 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f8b },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0a86 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0a6c },	\
	{ 0xa438, 0x8080 },	\
	{ 0xa438, 0xd019 },	\
	{ 0xa438, 0xd1a2 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x401a },	\
	{ 0xa438, 0xd707 },	\
	{ 0xa438, 0x40c4 },	\
	{ 0xa438, 0x60a6 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f8b },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0a86 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0a84 },	\
	{ 0xa438, 0xd503 },	\
	{ 0xa438, 0x8970 },	\
	{ 0xa438, 0x0c07 },	\
	{ 0xa438, 0x0901 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xcf09 },	\
	{ 0xa438, 0xd705 },	\
	{ 0xa438, 0x4000 },	\
	{ 0xa438, 0xceff },	\
	{ 0xa438, 0xaf0a },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1213 },	\
	{ 0xa438, 0x8401 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x8580 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1253 },	\
	{ 0xa438, 0xd064 },	\
	{ 0xa438, 0xd181 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4018 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xc50f },	\
	{ 0xa438, 0xd706 },	\
	{ 0xa438, 0x2c59 },	\
	{ 0xa438, 0x804d },	\
	{ 0xa438, 0xc60f },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xc605 },	\
	{ 0xa438, 0xae02 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x10fd },	\
	{ 0xa436, 0xa026 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa024 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa022 },	\
	{ 0xa438, 0x10f4 },	\
	{ 0xa436, 0xa020 },	\
	{ 0xa438, 0x1252 },	\
	{ 0xa436, 0xa006 },	\
	{ 0xa438, 0x1206 },	\
	{ 0xa436, 0xa004 },	\
	{ 0xa438, 0x0a78 },	\
	{ 0xa436, 0xa002 },	\
	{ 0xa438, 0x0a60 },	\
	{ 0xa436, 0xa000 },	\
	{ 0xa438, 0x0a4f },	\
	{ 0xa436, 0xa008 },	\
	{ 0xa438, 0x3f00 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0010 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8066 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x807c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8089 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x808e },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80a0 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80b2 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80c2 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x62db },	\
	{ 0xa438, 0x655c },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0x60e9 },	\
	{ 0xa438, 0x614a },	\
	{ 0xa438, 0x61ab },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0501 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0503 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0505 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0509 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x653c },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0x60e9 },	\
	{ 0xa438, 0x614a },	\
	{ 0xa438, 0x61ab },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0503 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0502 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0506 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x050a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0x60e9 },	\
	{ 0xa438, 0x614a },	\
	{ 0xa438, 0x61ab },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0505 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0506 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0504 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x050c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0x60e9 },	\
	{ 0xa438, 0x614a },	\
	{ 0xa438, 0x61ab },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0509 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x050a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x050c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0508 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0304 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xd73e },	\
	{ 0xa438, 0x60e9 },	\
	{ 0xa438, 0x614a },	\
	{ 0xa438, 0x61ab },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0501 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0321 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0502 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0321 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0504 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0321 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0508 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0321 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0346 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0x8208 },	\
	{ 0xa438, 0x609d },	\
	{ 0xa438, 0xa50f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x001a },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0503 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x001a },	\
	{ 0xa438, 0x607d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x00ab },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x00ab },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x60fd },	\
	{ 0xa438, 0xa50f },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xaa0f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x017b },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0503 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0a05 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x017b },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x60fd },	\
	{ 0xa438, 0xa50f },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xaa0f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x01e0 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0503 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0a05 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x01e0 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x60fd },	\
	{ 0xa438, 0xa50f },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xaa0f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0231 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0503 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0a05 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0231 },	\
	{ 0xa436, 0xa08e },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa08c },	\
	{ 0xa438, 0x0221 },	\
	{ 0xa436, 0xa08a },	\
	{ 0xa438, 0x01ce },	\
	{ 0xa436, 0xa088 },	\
	{ 0xa438, 0x0169 },	\
	{ 0xa436, 0xa086 },	\
	{ 0xa438, 0x00a6 },	\
	{ 0xa436, 0xa084 },	\
	{ 0xa438, 0x000d },	\
	{ 0xa436, 0xa082 },	\
	{ 0xa438, 0x0308 },	\
	{ 0xa436, 0xa080 },	\
	{ 0xa438, 0x029f },	\
	{ 0xa436, 0xa090 },	\
	{ 0xa438, 0x007f },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0020 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8017 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801b },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8029 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8054 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x805a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8064 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80a7 },	\
	{ 0xa438, 0x9430 },	\
	{ 0xa438, 0x9480 },	\
	{ 0xa438, 0xb408 },	\
	{ 0xa438, 0xd120 },	\
	{ 0xa438, 0xd057 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x064b },	\
	{ 0xa438, 0xcb80 },	\
	{ 0xa438, 0x9906 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0567 },	\
	{ 0xa438, 0xcb94 },	\
	{ 0xa438, 0x8190 },	\
	{ 0xa438, 0x82a0 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x8406 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0x8dff },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e4 },	\
	{ 0xa438, 0xa840 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0773 },	\
	{ 0xa438, 0xcb91 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x4063 },	\
	{ 0xa438, 0xd139 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xd140 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0xb404 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0d00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07dc },	\
	{ 0xa438, 0xa610 },	\
	{ 0xa438, 0xa110 },	\
	{ 0xa438, 0xa2a0 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4045 },	\
	{ 0xa438, 0xa180 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x405d },	\
	{ 0xa438, 0xa720 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0742 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07ec },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f74 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0742 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x7fb6 },	\
	{ 0xa438, 0x8190 },	\
	{ 0xa438, 0x82a0 },	\
	{ 0xa438, 0x8404 },	\
	{ 0xa438, 0x8610 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0d01 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07dc },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x064b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07c0 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fa7 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0481 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x94bc },	\
	{ 0xa438, 0x870c },	\
	{ 0xa438, 0xa190 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xa280 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0x8220 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x078e },	\
	{ 0xa438, 0xcb92 },	\
	{ 0xa438, 0xa840 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x4063 },	\
	{ 0xa438, 0xd140 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xd150 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x60a0 },	\
	{ 0xa438, 0x6121 },	\
	{ 0xa438, 0x61a2 },	\
	{ 0xa438, 0x6223 },	\
	{ 0xa438, 0xf02f },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d10 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0xf00f },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d20 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0xf00a },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d30 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0xf005 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d40 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e4 },	\
	{ 0xa438, 0xa610 },	\
	{ 0xa438, 0xa008 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4046 },	\
	{ 0xa438, 0xa002 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x405d },	\
	{ 0xa438, 0xa720 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0742 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07f7 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f74 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0742 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x7fb5 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e4 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x3ad4 },	\
	{ 0xa438, 0x0537 },	\
	{ 0xa438, 0x8610 },	\
	{ 0xa438, 0x8840 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x064b },	\
	{ 0xa438, 0x8301 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x8190 },	\
	{ 0xa438, 0x82a0 },	\
	{ 0xa438, 0x8404 },	\
	{ 0xa438, 0xa70c },	\
	{ 0xa438, 0x9402 },	\
	{ 0xa438, 0x890c },	\
	{ 0xa438, 0x8840 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x064b },	\
	{ 0xa436, 0xa10e },	\
	{ 0xa438, 0x0642 },	\
	{ 0xa436, 0xa10c },	\
	{ 0xa438, 0x0686 },	\
	{ 0xa436, 0xa10a },	\
	{ 0xa438, 0x0788 },	\
	{ 0xa436, 0xa108 },	\
	{ 0xa438, 0x047b },	\
	{ 0xa436, 0xa106 },	\
	{ 0xa438, 0x065c },	\
	{ 0xa436, 0xa104 },	\
	{ 0xa438, 0x0769 },	\
	{ 0xa436, 0xa102 },	\
	{ 0xa438, 0x0565 },	\
	{ 0xa436, 0xa100 },	\
	{ 0xa438, 0x06f9 },	\
	{ 0xa436, 0xa110 },	\
	{ 0xa438, 0x00ff },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8530 },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0x3caf },	\
	{ 0xa438, 0x8593 },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0x9caf },	\
	{ 0xa438, 0x85a5 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x5afb },	\
	{ 0xa438, 0xe083 },	\
	{ 0xa438, 0xfb0c },	\
	{ 0xa438, 0x020d },	\
	{ 0xa438, 0x021b },	\
	{ 0xa438, 0x10bf },	\
	{ 0xa438, 0x86d7 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x86da },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xfbe0 },	\
	{ 0xa438, 0x83fc },	\
	{ 0xa438, 0x0c02 },	\
	{ 0xa438, 0x0d02 },	\
	{ 0xa438, 0x1b10 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0xda02 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0xdd02 },	\
	{ 0xa438, 0x5afb },	\
	{ 0xa438, 0xe083 },	\
	{ 0xa438, 0xfd0c },	\
	{ 0xa438, 0x020d },	\
	{ 0xa438, 0x021b },	\
	{ 0xa438, 0x10bf },	\
	{ 0xa438, 0x86dd },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x86e0 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xfbe0 },	\
	{ 0xa438, 0x83fe },	\
	{ 0xa438, 0x0c02 },	\
	{ 0xa438, 0x0d02 },	\
	{ 0xa438, 0x1b10 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0xe002 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xaf2f },	\
	{ 0xa438, 0xbd02 },	\
	{ 0xa438, 0x2cac },	\
	{ 0xa438, 0x0286 },	\
	{ 0xa438, 0x65af },	\
	{ 0xa438, 0x212b },	\
	{ 0xa438, 0x022c },	\
	{ 0xa438, 0x6002 },	\
	{ 0xa438, 0x86b6 },	\
	{ 0xa438, 0xaf21 },	\
	{ 0xa438, 0x0cd1 },	\
	{ 0xa438, 0x03bf },	\
	{ 0xa438, 0x8710 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x870d },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x8719 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x8716 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x871f },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x871c },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x8728 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x8725 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x8707 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xfbad },	\
	{ 0xa438, 0x281c },	\
	{ 0xa438, 0xd100 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x0a02 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x1302 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x2202 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x2b02 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xae1a },	\
	{ 0xa438, 0xd101 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x0a02 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x1302 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x2202 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x2b02 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xd101 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x3402 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x3102 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x3d02 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x3a02 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x4302 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x4002 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x4c02 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x4902 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xd100 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x2e02 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x3702 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x4602 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xbf87 },	\
	{ 0xa438, 0x4f02 },	\
	{ 0xa438, 0x5ab7 },	\
	{ 0xa438, 0xaf35 },	\
	{ 0xa438, 0x7ff8 },	\
	{ 0xa438, 0xfaef },	\
	{ 0xa438, 0x69bf },	\
	{ 0xa438, 0x86e3 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xfbbf },	\
	{ 0xa438, 0x86fb },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x86e6 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xfbbf },	\
	{ 0xa438, 0x86fe },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x86e9 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xfbbf },	\
	{ 0xa438, 0x8701 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x86ec },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xfbbf },	\
	{ 0xa438, 0x8704 },	\
	{ 0xa438, 0x025a },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x86ef },	\
	{ 0xa438, 0x0262 },	\
	{ 0xa438, 0x7cbf },	\
	{ 0xa438, 0x86f2 },	\
	{ 0xa438, 0x0262 },	\
	{ 0xa438, 0x7cbf },	\
	{ 0xa438, 0x86f5 },	\
	{ 0xa438, 0x0262 },	\
	{ 0xa438, 0x7cbf },	\
	{ 0xa438, 0x86f8 },	\
	{ 0xa438, 0x0262 },	\
	{ 0xa438, 0x7cef },	\
	{ 0xa438, 0x96fe },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0xf8fa },	\
	{ 0xa438, 0xef69 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0xef02 },	\
	{ 0xa438, 0x6273 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0xf202 },	\
	{ 0xa438, 0x6273 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0xf502 },	\
	{ 0xa438, 0x6273 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0xf802 },	\
	{ 0xa438, 0x6273 },	\
	{ 0xa438, 0xef96 },	\
	{ 0xa438, 0xfefc },	\
	{ 0xa438, 0x0420 },	\
	{ 0xa438, 0xb540 },	\
	{ 0xa438, 0x53b5 },	\
	{ 0xa438, 0x4086 },	\
	{ 0xa438, 0xb540 },	\
	{ 0xa438, 0xb9b5 },	\
	{ 0xa438, 0x40c8 },	\
	{ 0xa438, 0xb03a },	\
	{ 0xa438, 0xc8b0 },	\
	{ 0xa438, 0xbac8 },	\
	{ 0xa438, 0xb13a },	\
	{ 0xa438, 0xc8b1 },	\
	{ 0xa438, 0xba77 },	\
	{ 0xa438, 0xbd26 },	\
	{ 0xa438, 0xffbd },	\
	{ 0xa438, 0x2677 },	\
	{ 0xa438, 0xbd28 },	\
	{ 0xa438, 0xffbd },	\
	{ 0xa438, 0x2840 },	\
	{ 0xa438, 0xbd26 },	\
	{ 0xa438, 0xc8bd },	\
	{ 0xa438, 0x2640 },	\
	{ 0xa438, 0xbd28 },	\
	{ 0xa438, 0xc8bd },	\
	{ 0xa438, 0x28bb },	\
	{ 0xa438, 0xa430 },	\
	{ 0xa438, 0x98b0 },	\
	{ 0xa438, 0x1eba },	\
	{ 0xa438, 0xb01e },	\
	{ 0xa438, 0xdcb0 },	\
	{ 0xa438, 0x1e98 },	\
	{ 0xa438, 0xb09e },	\
	{ 0xa438, 0xbab0 },	\
	{ 0xa438, 0x9edc },	\
	{ 0xa438, 0xb09e },	\
	{ 0xa438, 0x98b1 },	\
	{ 0xa438, 0x1eba },	\
	{ 0xa438, 0xb11e },	\
	{ 0xa438, 0xdcb1 },	\
	{ 0xa438, 0x1e98 },	\
	{ 0xa438, 0xb19e },	\
	{ 0xa438, 0xbab1 },	\
	{ 0xa438, 0x9edc },	\
	{ 0xa438, 0xb19e },	\
	{ 0xa438, 0x11b0 },	\
	{ 0xa438, 0x1e22 },	\
	{ 0xa438, 0xb01e },	\
	{ 0xa438, 0x33b0 },	\
	{ 0xa438, 0x1e11 },	\
	{ 0xa438, 0xb09e },	\
	{ 0xa438, 0x22b0 },	\
	{ 0xa438, 0x9e33 },	\
	{ 0xa438, 0xb09e },	\
	{ 0xa438, 0x11b1 },	\
	{ 0xa438, 0x1e22 },	\
	{ 0xa438, 0xb11e },	\
	{ 0xa438, 0x33b1 },	\
	{ 0xa438, 0x1e11 },	\
	{ 0xa438, 0xb19e },	\
	{ 0xa438, 0x22b1 },	\
	{ 0xa438, 0x9e33 },	\
	{ 0xa438, 0xb19e },	\
	{ 0xa436, 0xb85e },	\
	{ 0xa438, 0x2f71 },	\
	{ 0xa436, 0xb860 },	\
	{ 0xa438, 0x20d9 },	\
	{ 0xa436, 0xb862 },	\
	{ 0xa438, 0x2109 },	\
	{ 0xa436, 0xb864 },	\
	{ 0xa438, 0x34e7 },	\
	{ 0xa436, 0xb878 },	\
	{ 0xa438, 0x000f }

#define RTL8125_MAC_CFG3_MCU	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x808b },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x808f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8093 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8097 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x809d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80a1 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80aa },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x607b },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0xf00e },	\
	{ 0xa438, 0x42da },	\
	{ 0xa438, 0xf01e },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x615b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1456 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14a4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14bc },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5f2e },	\
	{ 0xa438, 0xf01c },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1456 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14a4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14bc },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5f2e },	\
	{ 0xa438, 0xf024 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1456 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14a4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14bc },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5f2e },	\
	{ 0xa438, 0xf02c },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1456 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14a4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14bc },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5f2e },	\
	{ 0xa438, 0xf034 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4118 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac11 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa410 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x4779 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac0f },	\
	{ 0xa438, 0xae01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1444 },	\
	{ 0xa438, 0xf034 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4118 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac22 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa420 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x4559 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac0f },	\
	{ 0xa438, 0xae01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1444 },	\
	{ 0xa438, 0xf023 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4118 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac44 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa440 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x4339 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac0f },	\
	{ 0xa438, 0xae01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1444 },	\
	{ 0xa438, 0xf012 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4118 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac88 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa480 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x4119 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac0f },	\
	{ 0xa438, 0xae01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1444 },	\
	{ 0xa438, 0xf001 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1456 },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5fac },	\
	{ 0xa438, 0xc48f },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x141b },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x121a },	\
	{ 0xa438, 0xd0b4 },	\
	{ 0xa438, 0xd1bb },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0898 },	\
	{ 0xa438, 0xd0b4 },	\
	{ 0xa438, 0xd1bb },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0a0e },	\
	{ 0xa438, 0xd064 },	\
	{ 0xa438, 0xd18a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0b7e },	\
	{ 0xa438, 0x401c },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xa804 },	\
	{ 0xa438, 0x8804 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x053b },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xa301 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0648 },	\
	{ 0xa438, 0xc520 },	\
	{ 0xa438, 0xa201 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x252d },	\
	{ 0xa438, 0x1646 },	\
	{ 0xa438, 0xd708 },	\
	{ 0xa438, 0x4006 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1646 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0308 },	\
	{ 0xa436, 0xa026 },	\
	{ 0xa438, 0x0307 },	\
	{ 0xa436, 0xa024 },	\
	{ 0xa438, 0x1645 },	\
	{ 0xa436, 0xa022 },	\
	{ 0xa438, 0x0647 },	\
	{ 0xa436, 0xa020 },	\
	{ 0xa438, 0x053a },	\
	{ 0xa436, 0xa006 },	\
	{ 0xa438, 0x0b7c },	\
	{ 0xa436, 0xa004 },	\
	{ 0xa438, 0x0a0c },	\
	{ 0xa436, 0xa002 },	\
	{ 0xa438, 0x0896 },	\
	{ 0xa436, 0xa000 },	\
	{ 0xa438, 0x11a1 },	\
	{ 0xa436, 0xa008 },	\
	{ 0xa438, 0xff00 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0010 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8015 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0xad02 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x02d7 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x00ed },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0509 },	\
	{ 0xa438, 0xc100 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x008f },	\
	{ 0xa436, 0xa08e },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa08c },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa08a },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa088 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa086 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa084 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa082 },	\
	{ 0xa438, 0x008d },	\
	{ 0xa436, 0xa080 },	\
	{ 0xa438, 0x00eb },	\
	{ 0xa436, 0xa090 },	\
	{ 0xa438, 0x0103 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0020 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8018 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8024 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8051 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8055 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8072 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80dc },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0xfffd },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0xfffd },	\
	{ 0xa438, 0x8301 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x8190 },	\
	{ 0xa438, 0x82a0 },	\
	{ 0xa438, 0x8404 },	\
	{ 0xa438, 0xa70c },	\
	{ 0xa438, 0x9402 },	\
	{ 0xa438, 0x890c },	\
	{ 0xa438, 0x8840 },	\
	{ 0xa438, 0xa380 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x066e },	\
	{ 0xa438, 0xcb91 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x4063 },	\
	{ 0xa438, 0xd139 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xd140 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0xb404 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0d00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e0 },	\
	{ 0xa438, 0xa610 },	\
	{ 0xa438, 0xa110 },	\
	{ 0xa438, 0xa2a0 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4085 },	\
	{ 0xa438, 0xa180 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0x8280 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x405d },	\
	{ 0xa438, 0xa720 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0743 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07f0 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f74 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0743 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x7fb6 },	\
	{ 0xa438, 0x8190 },	\
	{ 0xa438, 0x82a0 },	\
	{ 0xa438, 0x8404 },	\
	{ 0xa438, 0x8610 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0d01 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e0 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x066e },	\
	{ 0xa438, 0xd158 },	\
	{ 0xa438, 0xd04d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x03d4 },	\
	{ 0xa438, 0x94bc },	\
	{ 0xa438, 0x870c },	\
	{ 0xa438, 0x8380 },	\
	{ 0xa438, 0xd10d },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07c4 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0xa190 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xa280 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0xa220 },	\
	{ 0xa438, 0xd130 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07c4 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0xbb80 },	\
	{ 0xa438, 0xd1c4 },	\
	{ 0xa438, 0xd074 },	\
	{ 0xa438, 0xa301 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x604b },	\
	{ 0xa438, 0xa90c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0556 },	\
	{ 0xa438, 0xcb92 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x4063 },	\
	{ 0xa438, 0xd116 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xd119 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x60a0 },	\
	{ 0xa438, 0x6241 },	\
	{ 0xa438, 0x63e2 },	\
	{ 0xa438, 0x6583 },	\
	{ 0xa438, 0xf054 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x611e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d10 },	\
	{ 0xa438, 0xa010 },	\
	{ 0xa438, 0x8740 },	\
	{ 0xa438, 0xf02f },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d50 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0xf02a },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x611e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d20 },	\
	{ 0xa438, 0xa010 },	\
	{ 0xa438, 0x8740 },	\
	{ 0xa438, 0xf021 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d60 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0xf01c },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x611e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d30 },	\
	{ 0xa438, 0xa010 },	\
	{ 0xa438, 0x8740 },	\
	{ 0xa438, 0xf013 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d70 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0xf00e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x611e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d40 },	\
	{ 0xa438, 0xa010 },	\
	{ 0xa438, 0x8740 },	\
	{ 0xa438, 0xf005 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d80 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e8 },	\
	{ 0xa438, 0xa610 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x405d },	\
	{ 0xa438, 0xa720 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5ff4 },	\
	{ 0xa438, 0xa008 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4046 },	\
	{ 0xa438, 0xa002 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0743 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07fb },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x7f6f },	\
	{ 0xa438, 0x7f4e },	\
	{ 0xa438, 0x7f2d },	\
	{ 0xa438, 0x7f0c },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e8 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0743 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x7fb5 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x3ad4 },	\
	{ 0xa438, 0x0556 },	\
	{ 0xa438, 0x8610 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x066e },	\
	{ 0xa438, 0xd1f5 },	\
	{ 0xa438, 0xd049 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x01ec },	\
	{ 0xa436, 0xa10e },	\
	{ 0xa438, 0x01ea },	\
	{ 0xa436, 0xa10c },	\
	{ 0xa438, 0x06a9 },	\
	{ 0xa436, 0xa10a },	\
	{ 0xa438, 0x078a },	\
	{ 0xa436, 0xa108 },	\
	{ 0xa438, 0x03d2 },	\
	{ 0xa436, 0xa106 },	\
	{ 0xa438, 0x067f },	\
	{ 0xa436, 0xa104 },	\
	{ 0xa438, 0x0665 },	\
	{ 0xa436, 0xa102 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa100 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa110 },	\
	{ 0xa438, 0x00fc },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8530 },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0x3caf },	\
	{ 0xa438, 0x8545 },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0x45af },	\
	{ 0xa438, 0x8545 },	\
	{ 0xa438, 0xee82 },	\
	{ 0xa438, 0xf900 },	\
	{ 0xa438, 0x0103 },	\
	{ 0xa438, 0xaf03 },	\
	{ 0xa438, 0xb7f8 },	\
	{ 0xa438, 0xe0a6 },	\
	{ 0xa438, 0x00e1 },	\
	{ 0xa438, 0xa601 },	\
	{ 0xa438, 0xef01 },	\
	{ 0xa438, 0x58f0 },	\
	{ 0xa438, 0xa080 },	\
	{ 0xa438, 0x37a1 },	\
	{ 0xa438, 0x8402 },	\
	{ 0xa438, 0xae16 },	\
	{ 0xa438, 0xa185 },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x11a1 },	\
	{ 0xa438, 0x8702 },	\
	{ 0xa438, 0xae0c },	\
	{ 0xa438, 0xa188 },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x07a1 },	\
	{ 0xa438, 0x8902 },	\
	{ 0xa438, 0xae02 },	\
	{ 0xa438, 0xae1c },	\
	{ 0xa438, 0xe0b4 },	\
	{ 0xa438, 0x62e1 },	\
	{ 0xa438, 0xb463 },	\
	{ 0xa438, 0x6901 },	\
	{ 0xa438, 0xe4b4 },	\
	{ 0xa438, 0x62e5 },	\
	{ 0xa438, 0xb463 },	\
	{ 0xa438, 0xe0b4 },	\
	{ 0xa438, 0x62e1 },	\
	{ 0xa438, 0xb463 },	\
	{ 0xa438, 0x6901 },	\
	{ 0xa438, 0xe4b4 },	\
	{ 0xa438, 0x62e5 },	\
	{ 0xa438, 0xb463 },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa436, 0xb85e },	\
	{ 0xa438, 0x03b3 },	\
	{ 0xa436, 0xb860 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb862 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb864 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb878 },	\
	{ 0xa438, 0x0001 }

#define RTL8125_MAC_CFG4_MCU	\
	{ 0xa436, 0x8024 },	\
	{ 0xa438, 0x3700 },	\
	{ 0xa436, 0xb82e },	\
	{ 0xa438, 0x0001 },	\
	{ 0xb820, 0x0090 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8025 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x803a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8044 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8083 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x808d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x808d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x808d },	\
	{ 0xa438, 0xd712 },	\
	{ 0xa438, 0x4077 },	\
	{ 0xa438, 0xd71e },	\
	{ 0xa438, 0x4159 },	\
	{ 0xa438, 0xd71e },	\
	{ 0xa438, 0x6099 },	\
	{ 0xa438, 0x7f44 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1a14 },	\
	{ 0xa438, 0x9040 },	\
	{ 0xa438, 0x9201 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1b1a },	\
	{ 0xa438, 0xd71e },	\
	{ 0xa438, 0x2425 },	\
	{ 0xa438, 0x1a14 },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x3ce5 },	\
	{ 0xa438, 0x1afb },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1b00 },	\
	{ 0xa438, 0xd712 },	\
	{ 0xa438, 0x4077 },	\
	{ 0xa438, 0xd71e },	\
	{ 0xa438, 0x4159 },	\
	{ 0xa438, 0xd71e },	\
	{ 0xa438, 0x60b9 },	\
	{ 0xa438, 0x2421 },	\
	{ 0xa438, 0x1c17 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1a14 },	\
	{ 0xa438, 0x9040 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1c2c },	\
	{ 0xa438, 0xd71e },	\
	{ 0xa438, 0x2425 },	\
	{ 0xa438, 0x1a14 },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x3ce5 },	\
	{ 0xa438, 0x1c0f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1c13 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0x6072 },	\
	{ 0xa438, 0x8401 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xa401 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x146e },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0b77 },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x665d },	\
	{ 0xa438, 0x653e },	\
	{ 0xa438, 0x641f },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x62c4 },	\
	{ 0xa438, 0x6185 },	\
	{ 0xa438, 0x6066 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x165a },	\
	{ 0xa438, 0xc101 },	\
	{ 0xa438, 0xcb00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1945 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x7fa6 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x807d },	\
	{ 0xa438, 0xc102 },	\
	{ 0xa438, 0xcb00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1945 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x2569 },	\
	{ 0xa438, 0x8058 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x807d },	\
	{ 0xa438, 0xc104 },	\
	{ 0xa438, 0xcb00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1945 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x7fa4 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x807d },	\
	{ 0xa438, 0xc120 },	\
	{ 0xa438, 0xcb00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1945 },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x7fbf },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x807d },	\
	{ 0xa438, 0xc140 },	\
	{ 0xa438, 0xcb00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1945 },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x7fbe },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x807d },	\
	{ 0xa438, 0xc180 },	\
	{ 0xa438, 0xcb00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1945 },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x7fbd },	\
	{ 0xa438, 0xc100 },	\
	{ 0xa438, 0xcb00 },	\
	{ 0xa438, 0xd708 },	\
	{ 0xa438, 0x6018 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x165a },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14f6 },	\
	{ 0xa438, 0xd014 },	\
	{ 0xa438, 0xd1e3 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1356 },	\
	{ 0xa438, 0xd705 },	\
	{ 0xa438, 0x5fbe },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1559 },	\
	{ 0xa436, 0xa026 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa024 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa022 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa020 },	\
	{ 0xa438, 0x1557 },	\
	{ 0xa436, 0xa006 },	\
	{ 0xa438, 0x1677 },	\
	{ 0xa436, 0xa004 },	\
	{ 0xa438, 0x0b75 },	\
	{ 0xa436, 0xa002 },	\
	{ 0xa438, 0x1c17 },	\
	{ 0xa436, 0xa000 },	\
	{ 0xa438, 0x1b04 },	\
	{ 0xa436, 0xa008 },	\
	{ 0xa438, 0x1f00 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0020 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x817f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x82ab },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x83f8 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8444 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8454 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8459 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8465 },	\
	{ 0xa438, 0xcb11 },	\
	{ 0xa438, 0xa50c },	\
	{ 0xa438, 0x8310 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x4076 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x0903 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6083 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d00 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a7d },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a4d },	\
	{ 0xa438, 0xcb12 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5f84 },	\
	{ 0xa438, 0xd102 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x60f3 },	\
	{ 0xa438, 0xd413 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a37 },	\
	{ 0xa438, 0xd410 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a37 },	\
	{ 0xa438, 0xcb13 },	\
	{ 0xa438, 0xa108 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a42 },	\
	{ 0xa438, 0x8108 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xa910 },	\
	{ 0xa438, 0xa780 },	\
	{ 0xa438, 0xd14a },	\
	{ 0xa438, 0xd048 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x6255 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f74 },	\
	{ 0xa438, 0x6326 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x5f07 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0xa004 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a42 },	\
	{ 0xa438, 0x8004 },	\
	{ 0xa438, 0xa001 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a42 },	\
	{ 0xa438, 0x8001 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x0902 },	\
	{ 0xa438, 0xffe2 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5fab },	\
	{ 0xa438, 0xba08 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7f8b },	\
	{ 0xa438, 0x9a08 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x6535 },	\
	{ 0xa438, 0xd40d },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a37 },	\
	{ 0xa438, 0xcb14 },	\
	{ 0xa438, 0xa004 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a42 },	\
	{ 0xa438, 0x8004 },	\
	{ 0xa438, 0xa001 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a42 },	\
	{ 0xa438, 0x8001 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xa780 },	\
	{ 0xa438, 0xd14a },	\
	{ 0xa438, 0xd048 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0x6206 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x5f47 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0xa004 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a42 },	\
	{ 0xa438, 0x8004 },	\
	{ 0xa438, 0xa001 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a42 },	\
	{ 0xa438, 0x8001 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x0902 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8064 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0xd40e },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a37 },	\
	{ 0xa438, 0xb920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5fac },	\
	{ 0xa438, 0x9920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7f8c },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x6073 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x4216 },	\
	{ 0xa438, 0xa004 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a42 },	\
	{ 0xa438, 0x8004 },	\
	{ 0xa438, 0xa001 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a42 },	\
	{ 0xa438, 0x8001 },	\
	{ 0xa438, 0xd120 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0x8504 },	\
	{ 0xa438, 0xcb21 },	\
	{ 0xa438, 0xa301 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f9f },	\
	{ 0xa438, 0x8301 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x40e0 },	\
	{ 0xa438, 0xd196 },	\
	{ 0xa438, 0xd04d },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0xcb22 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a6d },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xa640 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x8910 },	\
	{ 0xa438, 0x8720 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6083 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d01 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d01 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a7d },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0f14 },	\
	{ 0xa438, 0xcb23 },	\
	{ 0xa438, 0x8fc0 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a25 },	\
	{ 0xa438, 0xaf40 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a25 },	\
	{ 0xa438, 0x0cc0 },	\
	{ 0xa438, 0x0f80 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a25 },	\
	{ 0xa438, 0xafc0 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a25 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x5dee },	\
	{ 0xa438, 0xcb24 },	\
	{ 0xa438, 0x8f1f },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x7f6e },	\
	{ 0xa438, 0xa111 },	\
	{ 0xa438, 0xa215 },	\
	{ 0xa438, 0xa401 },	\
	{ 0xa438, 0x8404 },	\
	{ 0xa438, 0xa720 },	\
	{ 0xa438, 0xcb25 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8640 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0b43 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0b86 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xb920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5fac },	\
	{ 0xa438, 0x9920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7f8c },	\
	{ 0xa438, 0xcb26 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5f82 },	\
	{ 0xa438, 0x8111 },	\
	{ 0xa438, 0x8205 },	\
	{ 0xa438, 0x8404 },	\
	{ 0xa438, 0xcb27 },	\
	{ 0xa438, 0xd404 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a37 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6083 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d02 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d02 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a7d },	\
	{ 0xa438, 0xa710 },	\
	{ 0xa438, 0xa104 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a42 },	\
	{ 0xa438, 0x8104 },	\
	{ 0xa438, 0xa001 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a42 },	\
	{ 0xa438, 0x8001 },	\
	{ 0xa438, 0xa120 },	\
	{ 0xa438, 0xaa0f },	\
	{ 0xa438, 0x8110 },	\
	{ 0xa438, 0xa284 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xd193 },	\
	{ 0xa438, 0xd046 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0xcb28 },	\
	{ 0xa438, 0xa110 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fa8 },	\
	{ 0xa438, 0x8110 },	\
	{ 0xa438, 0x8284 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x8710 },	\
	{ 0xa438, 0xb804 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7f82 },	\
	{ 0xa438, 0x9804 },	\
	{ 0xa438, 0xcb29 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5f85 },	\
	{ 0xa438, 0xa710 },	\
	{ 0xa438, 0xb820 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7f65 },	\
	{ 0xa438, 0x9820 },	\
	{ 0xa438, 0xcb2a },	\
	{ 0xa438, 0xa190 },	\
	{ 0xa438, 0xa284 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xd13d },	\
	{ 0xa438, 0xd04a },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x3444 },	\
	{ 0xa438, 0x8149 },	\
	{ 0xa438, 0xa220 },	\
	{ 0xa438, 0xd1a0 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x3444 },	\
	{ 0xa438, 0x8151 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x5f51 },	\
	{ 0xa438, 0xcb2f },	\
	{ 0xa438, 0xa302 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd708 },	\
	{ 0xa438, 0x5f63 },	\
	{ 0xa438, 0xd411 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a37 },	\
	{ 0xa438, 0x8302 },	\
	{ 0xa438, 0xd409 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a37 },	\
	{ 0xa438, 0xb920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5fac },	\
	{ 0xa438, 0x9920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7f8c },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5fa3 },	\
	{ 0xa438, 0x8190 },	\
	{ 0xa438, 0x82a4 },	\
	{ 0xa438, 0x8404 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0xb808 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7fa3 },	\
	{ 0xa438, 0x9808 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0433 },	\
	{ 0xa438, 0xcb15 },	\
	{ 0xa438, 0xa508 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6083 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d01 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d01 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a7d },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a4d },	\
	{ 0xa438, 0xa301 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f9f },	\
	{ 0xa438, 0x8301 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x40e0 },	\
	{ 0xa438, 0xd115 },	\
	{ 0xa438, 0xd04f },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0xd413 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a37 },	\
	{ 0xa438, 0xcb16 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a6d },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0xa640 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x8720 },	\
	{ 0xa438, 0xd17a },	\
	{ 0xa438, 0xd04c },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0f14 },	\
	{ 0xa438, 0xcb17 },	\
	{ 0xa438, 0x8fc0 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a25 },	\
	{ 0xa438, 0xaf40 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a25 },	\
	{ 0xa438, 0x0cc0 },	\
	{ 0xa438, 0x0f80 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a25 },	\
	{ 0xa438, 0xafc0 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a25 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x61ce },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5db4 },	\
	{ 0xa438, 0xcb18 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8640 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0xa720 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0b43 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xffd6 },	\
	{ 0xa438, 0x8f1f },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x7f8e },	\
	{ 0xa438, 0xa131 },	\
	{ 0xa438, 0xaa0f },	\
	{ 0xa438, 0xa2d5 },	\
	{ 0xa438, 0xa407 },	\
	{ 0xa438, 0xa720 },	\
	{ 0xa438, 0x8310 },	\
	{ 0xa438, 0xa308 },	\
	{ 0xa438, 0x8308 },	\
	{ 0xa438, 0xcb19 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x1502 },	\
	{ 0xa438, 0x8640 },	\
	{ 0xa438, 0x9503 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0b43 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0b86 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xb920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5fac },	\
	{ 0xa438, 0x9920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7f8c },	\
	{ 0xa438, 0xcb1a },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5f82 },	\
	{ 0xa438, 0x8111 },	\
	{ 0xa438, 0x82c5 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0x8402 },	\
	{ 0xa438, 0xb804 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7f82 },	\
	{ 0xa438, 0x9804 },	\
	{ 0xa438, 0xcb1b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5f85 },	\
	{ 0xa438, 0xa710 },	\
	{ 0xa438, 0xb820 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7f65 },	\
	{ 0xa438, 0x9820 },	\
	{ 0xa438, 0xcb1c },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6083 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d02 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d02 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a7d },	\
	{ 0xa438, 0xa110 },	\
	{ 0xa438, 0xa284 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0x8402 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fa8 },	\
	{ 0xa438, 0xcb1d },	\
	{ 0xa438, 0xa180 },	\
	{ 0xa438, 0xa402 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fa8 },	\
	{ 0xa438, 0xa220 },	\
	{ 0xa438, 0xd1f5 },	\
	{ 0xa438, 0xd049 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x3444 },	\
	{ 0xa438, 0x8221 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x5f51 },	\
	{ 0xa438, 0xb920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5fac },	\
	{ 0xa438, 0x9920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7f8c },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5fa3 },	\
	{ 0xa438, 0xa504 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6083 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d00 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a7d },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0x8190 },	\
	{ 0xa438, 0x82a4 },	\
	{ 0xa438, 0x8402 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0xb808 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7fa3 },	\
	{ 0xa438, 0x9808 },	\
	{ 0xa438, 0xcb2b },	\
	{ 0xa438, 0xcb2c },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5f84 },	\
	{ 0xa438, 0xd14a },	\
	{ 0xa438, 0xd048 },	\
	{ 0xa438, 0xa780 },	\
	{ 0xa438, 0xcb2d },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f94 },	\
	{ 0xa438, 0x6208 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x5f27 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0xa004 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a42 },	\
	{ 0xa438, 0x8004 },	\
	{ 0xa438, 0xa001 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a42 },	\
	{ 0xa438, 0x8001 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x0902 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xffe9 },	\
	{ 0xa438, 0xcb2e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6083 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d02 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d02 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a7d },	\
	{ 0xa438, 0xa190 },	\
	{ 0xa438, 0xa284 },	\
	{ 0xa438, 0xa406 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fa8 },	\
	{ 0xa438, 0xa220 },	\
	{ 0xa438, 0xd1a0 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x3444 },	\
	{ 0xa438, 0x827d },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x5f51 },	\
	{ 0xa438, 0xcb2f },	\
	{ 0xa438, 0xa302 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd708 },	\
	{ 0xa438, 0x5f63 },	\
	{ 0xa438, 0xd411 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a37 },	\
	{ 0xa438, 0x8302 },	\
	{ 0xa438, 0xd409 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a37 },	\
	{ 0xa438, 0xb920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5fac },	\
	{ 0xa438, 0x9920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7f8c },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5fa3 },	\
	{ 0xa438, 0x8190 },	\
	{ 0xa438, 0x82a4 },	\
	{ 0xa438, 0x8406 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0xb808 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7fa3 },	\
	{ 0xa438, 0x9808 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0433 },	\
	{ 0xa438, 0xcb30 },	\
	{ 0xa438, 0x8380 },	\
	{ 0xa438, 0xcb31 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5f86 },	\
	{ 0xa438, 0x9308 },	\
	{ 0xa438, 0xb204 },	\
	{ 0xa438, 0xb301 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x5fa2 },	\
	{ 0xa438, 0xb302 },	\
	{ 0xa438, 0x9204 },	\
	{ 0xa438, 0xcb32 },	\
	{ 0xa438, 0xd408 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a37 },	\
	{ 0xa438, 0xd141 },	\
	{ 0xa438, 0xd043 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4ccc },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x4c81 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x609e },	\
	{ 0xa438, 0xd1e5 },	\
	{ 0xa438, 0xd04d },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0xd1e5 },	\
	{ 0xa438, 0xd04d },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6083 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d01 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d01 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a7d },	\
	{ 0xa438, 0x8710 },	\
	{ 0xa438, 0xa108 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a42 },	\
	{ 0xa438, 0x8108 },	\
	{ 0xa438, 0xa203 },	\
	{ 0xa438, 0x8120 },	\
	{ 0xa438, 0x8a0f },	\
	{ 0xa438, 0xa111 },	\
	{ 0xa438, 0x8204 },	\
	{ 0xa438, 0xa140 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a42 },	\
	{ 0xa438, 0x8140 },	\
	{ 0xa438, 0xd17a },	\
	{ 0xa438, 0xd04b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0xa204 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fa7 },	\
	{ 0xa438, 0xb920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5fac },	\
	{ 0xa438, 0x9920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7f8c },	\
	{ 0xa438, 0xd404 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a37 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6083 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d02 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d02 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a7d },	\
	{ 0xa438, 0xa710 },	\
	{ 0xa438, 0x8101 },	\
	{ 0xa438, 0x8201 },	\
	{ 0xa438, 0xa104 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a42 },	\
	{ 0xa438, 0x8104 },	\
	{ 0xa438, 0xa120 },	\
	{ 0xa438, 0xaa0f },	\
	{ 0xa438, 0x8110 },	\
	{ 0xa438, 0xa284 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xd193 },	\
	{ 0xa438, 0xd047 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0xa110 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fa8 },	\
	{ 0xa438, 0xa180 },	\
	{ 0xa438, 0xd13d },	\
	{ 0xa438, 0xd04a },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0xf024 },	\
	{ 0xa438, 0xa710 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0x8190 },	\
	{ 0xa438, 0x8204 },	\
	{ 0xa438, 0xa280 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fa7 },	\
	{ 0xa438, 0x8710 },	\
	{ 0xa438, 0xb920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5fac },	\
	{ 0xa438, 0x9920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7f8c },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x8190 },	\
	{ 0xa438, 0x8284 },	\
	{ 0xa438, 0x8406 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x4121 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x60f3 },	\
	{ 0xa438, 0xd1e5 },	\
	{ 0xa438, 0xd04d },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0x8710 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0x8190 },	\
	{ 0xa438, 0x8204 },	\
	{ 0xa438, 0xa280 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0xb920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5fac },	\
	{ 0xa438, 0x9920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7f8c },	\
	{ 0xa438, 0xcb33 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x5f85 },	\
	{ 0xa438, 0xa710 },	\
	{ 0xa438, 0xb820 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd71f },	\
	{ 0xa438, 0x7f65 },	\
	{ 0xa438, 0x9820 },	\
	{ 0xa438, 0xcb34 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xa190 },	\
	{ 0xa438, 0xa284 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fa9 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x6853 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6083 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d00 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a7d },	\
	{ 0xa438, 0x8190 },	\
	{ 0xa438, 0x8284 },	\
	{ 0xa438, 0xcb35 },	\
	{ 0xa438, 0xd407 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a37 },	\
	{ 0xa438, 0x8110 },	\
	{ 0xa438, 0x8204 },	\
	{ 0xa438, 0xa280 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4215 },	\
	{ 0xa438, 0xa304 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb8 },	\
	{ 0xa438, 0xd1c3 },	\
	{ 0xa438, 0xd043 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0x8304 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x4109 },	\
	{ 0xa438, 0xf01e },	\
	{ 0xa438, 0xcb36 },	\
	{ 0xa438, 0xd412 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a37 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6309 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x42c7 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x8180 },	\
	{ 0xa438, 0x8280 },	\
	{ 0xa438, 0x8404 },	\
	{ 0xa438, 0xa004 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a42 },	\
	{ 0xa438, 0x8004 },	\
	{ 0xa438, 0xa001 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a42 },	\
	{ 0xa438, 0x8001 },	\
	{ 0xa438, 0x0c03 },	\
	{ 0xa438, 0x0902 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xd14a },	\
	{ 0xa438, 0xd048 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6083 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d02 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x0d02 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a7d },	\
	{ 0xa438, 0xcc55 },	\
	{ 0xa438, 0xcb37 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xa190 },	\
	{ 0xa438, 0xa2a4 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6041 },	\
	{ 0xa438, 0xa402 },	\
	{ 0xa438, 0xd13d },	\
	{ 0xa438, 0xd04a },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fa9 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x5f71 },	\
	{ 0xa438, 0xcb38 },	\
	{ 0xa438, 0x8224 },	\
	{ 0xa438, 0xa288 },	\
	{ 0xa438, 0x8180 },	\
	{ 0xa438, 0xa110 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6041 },	\
	{ 0xa438, 0x8402 },	\
	{ 0xa438, 0xd415 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a37 },	\
	{ 0xa438, 0xd13d },	\
	{ 0xa438, 0xd04a },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0xcb39 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xa190 },	\
	{ 0xa438, 0xa2a0 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6041 },	\
	{ 0xa438, 0xa402 },	\
	{ 0xa438, 0xd17a },	\
	{ 0xa438, 0xd047 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0560 },	\
	{ 0xa438, 0xa111 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0xd3f5 },	\
	{ 0xa438, 0xd219 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0c31 },	\
	{ 0xa438, 0xd708 },	\
	{ 0xa438, 0x5fa5 },	\
	{ 0xa438, 0xa215 },	\
	{ 0xa438, 0xd30e },	\
	{ 0xa438, 0xd21a },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0c31 },	\
	{ 0xa438, 0xd708 },	\
	{ 0xa438, 0x63e9 },	\
	{ 0xa438, 0xd708 },	\
	{ 0xa438, 0x5f65 },	\
	{ 0xa438, 0xd708 },	\
	{ 0xa438, 0x7f36 },	\
	{ 0xa438, 0xa004 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0c35 },	\
	{ 0xa438, 0x8004 },	\
	{ 0xa438, 0xa001 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0c35 },	\
	{ 0xa438, 0x8001 },	\
	{ 0xa438, 0xd708 },	\
	{ 0xa438, 0x4098 },	\
	{ 0xa438, 0xd102 },	\
	{ 0xa438, 0x9401 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0xd103 },	\
	{ 0xa438, 0xb401 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0c27 },	\
	{ 0xa438, 0xa108 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0c35 },	\
	{ 0xa438, 0x8108 },	\
	{ 0xa438, 0x8110 },	\
	{ 0xa438, 0x8294 },	\
	{ 0xa438, 0xa202 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0bdb },	\
	{ 0xa438, 0xd39c },	\
	{ 0xa438, 0xd210 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0c31 },	\
	{ 0xa438, 0xd708 },	\
	{ 0xa438, 0x5fa5 },	\
	{ 0xa438, 0xd39c },	\
	{ 0xa438, 0xd210 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0c31 },	\
	{ 0xa438, 0xd708 },	\
	{ 0xa438, 0x5fa5 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0c31 },	\
	{ 0xa438, 0xd708 },	\
	{ 0xa438, 0x29b5 },	\
	{ 0xa438, 0x840e },	\
	{ 0xa438, 0xd708 },	\
	{ 0xa438, 0x5f4a },	\
	{ 0xa438, 0x0c1f },	\
	{ 0xa438, 0x1014 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0c31 },	\
	{ 0xa438, 0xd709 },	\
	{ 0xa438, 0x7fa4 },	\
	{ 0xa438, 0x901f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0c23 },	\
	{ 0xa438, 0xcb43 },	\
	{ 0xa438, 0xa508 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x3699 },	\
	{ 0xa438, 0x844a },	\
	{ 0xa438, 0xa504 },	\
	{ 0xa438, 0xa190 },	\
	{ 0xa438, 0xa2a0 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x2109 },	\
	{ 0xa438, 0x05ea },	\
	{ 0xa438, 0xa402 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x05ea },	\
	{ 0xa438, 0xcb90 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0ca0 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x06db },	\
	{ 0xa438, 0xd1ff },	\
	{ 0xa438, 0xd052 },	\
	{ 0xa438, 0xa508 },	\
	{ 0xa438, 0x8718 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xa190 },	\
	{ 0xa438, 0xa2a0 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0c50 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x09ef },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a5e },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x2e70 },	\
	{ 0xa438, 0x06da },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f55 },	\
	{ 0xa438, 0xa90c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0645 },	\
	{ 0xa436, 0xa10e },	\
	{ 0xa438, 0x0644 },	\
	{ 0xa436, 0xa10c },	\
	{ 0xa438, 0x09e9 },	\
	{ 0xa436, 0xa10a },	\
	{ 0xa438, 0x06da },	\
	{ 0xa436, 0xa108 },	\
	{ 0xa438, 0x05e1 },	\
	{ 0xa436, 0xa106 },	\
	{ 0xa438, 0x0be4 },	\
	{ 0xa436, 0xa104 },	\
	{ 0xa438, 0x0435 },	\
	{ 0xa436, 0xa102 },	\
	{ 0xa438, 0x0141 },	\
	{ 0xa436, 0xa100 },	\
	{ 0xa438, 0x026d },	\
	{ 0xa436, 0xa110 },	\
	{ 0xa438, 0x00ff },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x85fe },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0xaf86 },	\
	{ 0xa438, 0x16af },	\
	{ 0xa438, 0x8699 },	\
	{ 0xa438, 0xaf86 },	\
	{ 0xa438, 0xe5af },	\
	{ 0xa438, 0x86f9 },	\
	{ 0xa438, 0xaf87 },	\
	{ 0xa438, 0x7aaf },	\
	{ 0xa438, 0x883a },	\
	{ 0xa438, 0xaf88 },	\
	{ 0xa438, 0x58af },	\
	{ 0xa438, 0x8b6c },	\
	{ 0xa438, 0xd48b },	\
	{ 0xa438, 0x7c02 },	\
	{ 0xa438, 0x8644 },	\
	{ 0xa438, 0x2c00 },	\
	{ 0xa438, 0x503c },	\
	{ 0xa438, 0xffd6 },	\
	{ 0xa438, 0xac27 },	\
	{ 0xa438, 0x18e1 },	\
	{ 0xa438, 0x82fe },	\
	{ 0xa438, 0xad28 },	\
	{ 0xa438, 0x0cd4 },	\
	{ 0xa438, 0x8b84 },	\
	{ 0xa438, 0x0286 },	\
	{ 0xa438, 0x442c },	\
	{ 0xa438, 0x003c },	\
	{ 0xa438, 0xac27 },	\
	{ 0xa438, 0x06ee },	\
	{ 0xa438, 0x8299 },	\
	{ 0xa438, 0x01ae },	\
	{ 0xa438, 0x04ee },	\
	{ 0xa438, 0x8299 },	\
	{ 0xa438, 0x00af },	\
	{ 0xa438, 0x23dc },	\
	{ 0xa438, 0xf9fa },	\
	{ 0xa438, 0xcefa },	\
	{ 0xa438, 0xfbef },	\
	{ 0xa438, 0x79fb },	\
	{ 0xa438, 0xc4bf },	\
	{ 0xa438, 0x8b76 },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x6dac },	\
	{ 0xa438, 0x2804 },	\
	{ 0xa438, 0xd203 },	\
	{ 0xa438, 0xae02 },	\
	{ 0xa438, 0xd201 },	\
	{ 0xa438, 0xbdd8 },	\
	{ 0xa438, 0x19d9 },	\
	{ 0xa438, 0xef94 },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x6d78 },	\
	{ 0xa438, 0x03ef },	\
	{ 0xa438, 0x648a },	\
	{ 0xa438, 0x0002 },	\
	{ 0xa438, 0xbdd8 },	\
	{ 0xa438, 0x19d9 },	\
	{ 0xa438, 0xef94 },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x6d78 },	\
	{ 0xa438, 0x03ef },	\
	{ 0xa438, 0x7402 },	\
	{ 0xa438, 0x72cd },	\
	{ 0xa438, 0xac50 },	\
	{ 0xa438, 0x02ef },	\
	{ 0xa438, 0x643a },	\
	{ 0xa438, 0x019f },	\
	{ 0xa438, 0xe4ef },	\
	{ 0xa438, 0x4678 },	\
	{ 0xa438, 0x03ac },	\
	{ 0xa438, 0x2002 },	\
	{ 0xa438, 0xae02 },	\
	{ 0xa438, 0xd0ff },	\
	{ 0xa438, 0xffef },	\
	{ 0xa438, 0x97ff },	\
	{ 0xa438, 0xfec6 },	\
	{ 0xa438, 0xfefd },	\
	{ 0xa438, 0x041f },	\
	{ 0xa438, 0x771f },	\
	{ 0xa438, 0x221c },	\
	{ 0xa438, 0x450d },	\
	{ 0xa438, 0x481f },	\
	{ 0xa438, 0x00ac },	\
	{ 0xa438, 0x7f04 },	\
	{ 0xa438, 0x1a94 },	\
	{ 0xa438, 0xae08 },	\
	{ 0xa438, 0x1a94 },	\
	{ 0xa438, 0xac7f },	\
	{ 0xa438, 0x03d7 },	\
	{ 0xa438, 0x0100 },	\
	{ 0xa438, 0xef46 },	\
	{ 0xa438, 0x0d48 },	\
	{ 0xa438, 0x1f00 },	\
	{ 0xa438, 0x1c45 },	\
	{ 0xa438, 0xef69 },	\
	{ 0xa438, 0xef57 },	\
	{ 0xa438, 0xef74 },	\
	{ 0xa438, 0x0272 },	\
	{ 0xa438, 0xe8a7 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa438, 0x0d1a },	\
	{ 0xa438, 0x941b },	\
	{ 0xa438, 0x979e },	\
	{ 0xa438, 0x072d },	\
	{ 0xa438, 0x0100 },	\
	{ 0xa438, 0x1a64 },	\
	{ 0xa438, 0xef76 },	\
	{ 0xa438, 0xef97 },	\
	{ 0xa438, 0x0d98 },	\
	{ 0xa438, 0xd400 },	\
	{ 0xa438, 0xff1d },	\
	{ 0xa438, 0x941a },	\
	{ 0xa438, 0x89cf },	\
	{ 0xa438, 0x1a75 },	\
	{ 0xa438, 0xaf74 },	\
	{ 0xa438, 0xf9bf },	\
	{ 0xa438, 0x8b79 },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x6da1 },	\
	{ 0xa438, 0x0005 },	\
	{ 0xa438, 0xe180 },	\
	{ 0xa438, 0xa0ae },	\
	{ 0xa438, 0x03e1 },	\
	{ 0xa438, 0x80a1 },	\
	{ 0xa438, 0xaf26 },	\
	{ 0xa438, 0x9aac },	\
	{ 0xa438, 0x284d },	\
	{ 0xa438, 0xe08f },	\
	{ 0xa438, 0xffef },	\
	{ 0xa438, 0x10c0 },	\
	{ 0xa438, 0xe08f },	\
	{ 0xa438, 0xfe10 },	\
	{ 0xa438, 0x1b08 },	\
	{ 0xa438, 0xa000 },	\
	{ 0xa438, 0x04c8 },	\
	{ 0xa438, 0xaf40 },	\
	{ 0xa438, 0x67c8 },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x8c02 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0xc4bf },	\
	{ 0xa438, 0x8b8f },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x6def },	\
	{ 0xa438, 0x74e0 },	\
	{ 0xa438, 0x830c },	\
	{ 0xa438, 0xad20 },	\
	{ 0xa438, 0x0302 },	\
	{ 0xa438, 0x74ac },	\
	{ 0xa438, 0xccef },	\
	{ 0xa438, 0x971b },	\
	{ 0xa438, 0x76ad },	\
	{ 0xa438, 0x5f02 },	\
	{ 0xa438, 0xae13 },	\
	{ 0xa438, 0xef69 },	\
	{ 0xa438, 0xef30 },	\
	{ 0xa438, 0x1b32 },	\
	{ 0xa438, 0xc4ef },	\
	{ 0xa438, 0x46e4 },	\
	{ 0xa438, 0x8ffb },	\
	{ 0xa438, 0xe58f },	\
	{ 0xa438, 0xfce7 },	\
	{ 0xa438, 0x8ffd },	\
	{ 0xa438, 0xcc10 },	\
	{ 0xa438, 0x11ae },	\
	{ 0xa438, 0xb8d1 },	\
	{ 0xa438, 0x00a1 },	\
	{ 0xa438, 0x1f03 },	\
	{ 0xa438, 0xaf40 },	\
	{ 0xa438, 0x4fbf },	\
	{ 0xa438, 0x8b8c },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x4ec4 },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x8f02 },	\
	{ 0xa438, 0x6c6d },	\
	{ 0xa438, 0xef74 },	\
	{ 0xa438, 0xe083 },	\
	{ 0xa438, 0x0cad },	\
	{ 0xa438, 0x2003 },	\
	{ 0xa438, 0x0274 },	\
	{ 0xa438, 0xaccc },	\
	{ 0xa438, 0xef97 },	\
	{ 0xa438, 0x1b76 },	\
	{ 0xa438, 0xad5f },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x04ef },	\
	{ 0xa438, 0x69ef },	\
	{ 0xa438, 0x3111 },	\
	{ 0xa438, 0xaed1 },	\
	{ 0xa438, 0x0287 },	\
	{ 0xa438, 0x80af },	\
	{ 0xa438, 0x2293 },	\
	{ 0xa438, 0xf8f9 },	\
	{ 0xa438, 0xfafb },	\
	{ 0xa438, 0xef59 },	\
	{ 0xa438, 0xe080 },	\
	{ 0xa438, 0x13ad },	\
	{ 0xa438, 0x252f },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x2802 },	\
	{ 0xa438, 0x6c6d },	\
	{ 0xa438, 0xef64 },	\
	{ 0xa438, 0x1f44 },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xb91b },	\
	{ 0xa438, 0x64ad },	\
	{ 0xa438, 0x4f1d },	\
	{ 0xa438, 0xd688 },	\
	{ 0xa438, 0x2bd7 },	\
	{ 0xa438, 0x882e },	\
	{ 0xa438, 0x0274 },	\
	{ 0xa438, 0x73ad },	\
	{ 0xa438, 0x5008 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x3102 },	\
	{ 0xa438, 0x737c },	\
	{ 0xa438, 0xae03 },	\
	{ 0xa438, 0x0287 },	\
	{ 0xa438, 0xd0bf },	\
	{ 0xa438, 0x882b },	\
	{ 0xa438, 0x0273 },	\
	{ 0xa438, 0x73e0 },	\
	{ 0xa438, 0x824c },	\
	{ 0xa438, 0xf621 },	\
	{ 0xa438, 0xe482 },	\
	{ 0xa438, 0x4cbf },	\
	{ 0xa438, 0x8834 },	\
	{ 0xa438, 0x0273 },	\
	{ 0xa438, 0x7cef },	\
	{ 0xa438, 0x95ff },	\
	{ 0xa438, 0xfefd },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0xf8f9 },	\
	{ 0xa438, 0xfafb },	\
	{ 0xa438, 0xef79 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x1f02 },	\
	{ 0xa438, 0x737c },	\
	{ 0xa438, 0x1f22 },	\
	{ 0xa438, 0xac32 },	\
	{ 0xa438, 0x31ef },	\
	{ 0xa438, 0x12bf },	\
	{ 0xa438, 0x8822 },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x4ed6 },	\
	{ 0xa438, 0x8fba },	\
	{ 0xa438, 0x1f33 },	\
	{ 0xa438, 0xac3c },	\
	{ 0xa438, 0x1eef },	\
	{ 0xa438, 0x13bf },	\
	{ 0xa438, 0x8837 },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x4eef },	\
	{ 0xa438, 0x96d8 },	\
	{ 0xa438, 0x19d9 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x2502 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x2502 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0x1616 },	\
	{ 0xa438, 0x13ae },	\
	{ 0xa438, 0xdf12 },	\
	{ 0xa438, 0xaecc },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x1f02 },	\
	{ 0xa438, 0x7373 },	\
	{ 0xa438, 0xef97 },	\
	{ 0xa438, 0xfffe },	\
	{ 0xa438, 0xfdfc },	\
	{ 0xa438, 0x0466 },	\
	{ 0xa438, 0xac88 },	\
	{ 0xa438, 0x54ac },	\
	{ 0xa438, 0x88f0 },	\
	{ 0xa438, 0xac8a },	\
	{ 0xa438, 0x92ac },	\
	{ 0xa438, 0xbadd },	\
	{ 0xa438, 0xac6c },	\
	{ 0xa438, 0xeeac },	\
	{ 0xa438, 0x6cff },	\
	{ 0xa438, 0xad02 },	\
	{ 0xa438, 0x99ac },	\
	{ 0xa438, 0x0030 },	\
	{ 0xa438, 0xac88 },	\
	{ 0xa438, 0xd4c3 },	\
	{ 0xa438, 0x5000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x00b4 },	\
	{ 0xa438, 0xecee },	\
	{ 0xa438, 0x8298 },	\
	{ 0xa438, 0x00af },	\
	{ 0xa438, 0x1412 },	\
	{ 0xa438, 0xf8bf },	\
	{ 0xa438, 0x8b5d },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x6d58 },	\
	{ 0xa438, 0x03e1 },	\
	{ 0xa438, 0x8fb8 },	\
	{ 0xa438, 0x2901 },	\
	{ 0xa438, 0xe58f },	\
	{ 0xa438, 0xb8a0 },	\
	{ 0xa438, 0x0049 },	\
	{ 0xa438, 0xef47 },	\
	{ 0xa438, 0xe483 },	\
	{ 0xa438, 0x02e5 },	\
	{ 0xa438, 0x8303 },	\
	{ 0xa438, 0xbfc2 },	\
	{ 0xa438, 0x5f1a },	\
	{ 0xa438, 0x95f7 },	\
	{ 0xa438, 0x05ee },	\
	{ 0xa438, 0xffd2 },	\
	{ 0xa438, 0x00d8 },	\
	{ 0xa438, 0xf605 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0xef60 },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x3002 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x3302 },	\
	{ 0xa438, 0x6c6d },	\
	{ 0xa438, 0xf728 },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x3302 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0xf628 },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x3302 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0x0c64 },	\
	{ 0xa438, 0xef46 },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x6002 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0x0289 },	\
	{ 0xa438, 0x9902 },	\
	{ 0xa438, 0x3920 },	\
	{ 0xa438, 0xaf89 },	\
	{ 0xa438, 0x96a0 },	\
	{ 0xa438, 0x0149 },	\
	{ 0xa438, 0xef47 },	\
	{ 0xa438, 0xe483 },	\
	{ 0xa438, 0x04e5 },	\
	{ 0xa438, 0x8305 },	\
	{ 0xa438, 0xbfc2 },	\
	{ 0xa438, 0x5f1a },	\
	{ 0xa438, 0x95f7 },	\
	{ 0xa438, 0x05ee },	\
	{ 0xa438, 0xffd2 },	\
	{ 0xa438, 0x00d8 },	\
	{ 0xa438, 0xf605 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0xef60 },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x3002 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x3302 },	\
	{ 0xa438, 0x6c6d },	\
	{ 0xa438, 0xf729 },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x3302 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0xf629 },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x3302 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0x0c64 },	\
	{ 0xa438, 0xef46 },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x6302 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0x0289 },	\
	{ 0xa438, 0x9902 },	\
	{ 0xa438, 0x3920 },	\
	{ 0xa438, 0xaf89 },	\
	{ 0xa438, 0x96a0 },	\
	{ 0xa438, 0x0249 },	\
	{ 0xa438, 0xef47 },	\
	{ 0xa438, 0xe483 },	\
	{ 0xa438, 0x06e5 },	\
	{ 0xa438, 0x8307 },	\
	{ 0xa438, 0xbfc2 },	\
	{ 0xa438, 0x5f1a },	\
	{ 0xa438, 0x95f7 },	\
	{ 0xa438, 0x05ee },	\
	{ 0xa438, 0xffd2 },	\
	{ 0xa438, 0x00d8 },	\
	{ 0xa438, 0xf605 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0xef60 },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x3002 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x3302 },	\
	{ 0xa438, 0x6c6d },	\
	{ 0xa438, 0xf72a },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x3302 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0xf62a },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x3302 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0x0c64 },	\
	{ 0xa438, 0xef46 },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x6602 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0x0289 },	\
	{ 0xa438, 0x9902 },	\
	{ 0xa438, 0x3920 },	\
	{ 0xa438, 0xaf89 },	\
	{ 0xa438, 0x96ef },	\
	{ 0xa438, 0x47e4 },	\
	{ 0xa438, 0x8308 },	\
	{ 0xa438, 0xe583 },	\
	{ 0xa438, 0x09bf },	\
	{ 0xa438, 0xc25f },	\
	{ 0xa438, 0x1a95 },	\
	{ 0xa438, 0xf705 },	\
	{ 0xa438, 0xeeff },	\
	{ 0xa438, 0xd200 },	\
	{ 0xa438, 0xd8f6 },	\
	{ 0xa438, 0x051f },	\
	{ 0xa438, 0x11ef },	\
	{ 0xa438, 0x60bf },	\
	{ 0xa438, 0x8b30 },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x4ebf },	\
	{ 0xa438, 0x8b33 },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x6df7 },	\
	{ 0xa438, 0x2bbf },	\
	{ 0xa438, 0x8b33 },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x4ef6 },	\
	{ 0xa438, 0x2bbf },	\
	{ 0xa438, 0x8b33 },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x4e0c },	\
	{ 0xa438, 0x64ef },	\
	{ 0xa438, 0x46bf },	\
	{ 0xa438, 0x8b69 },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x4e02 },	\
	{ 0xa438, 0x8999 },	\
	{ 0xa438, 0x0239 },	\
	{ 0xa438, 0x20af },	\
	{ 0xa438, 0x8996 },	\
	{ 0xa438, 0xaf39 },	\
	{ 0xa438, 0x1ef8 },	\
	{ 0xa438, 0xf9fa },	\
	{ 0xa438, 0xe08f },	\
	{ 0xa438, 0xb838 },	\
	{ 0xa438, 0x02ad },	\
	{ 0xa438, 0x2702 },	\
	{ 0xa438, 0xae03 },	\
	{ 0xa438, 0xaf8b },	\
	{ 0xa438, 0x201f },	\
	{ 0xa438, 0x66ef },	\
	{ 0xa438, 0x65bf },	\
	{ 0xa438, 0xc21f },	\
	{ 0xa438, 0x1a96 },	\
	{ 0xa438, 0xf705 },	\
	{ 0xa438, 0xeeff },	\
	{ 0xa438, 0xd200 },	\
	{ 0xa438, 0xdaf6 },	\
	{ 0xa438, 0x05bf },	\
	{ 0xa438, 0xc22f },	\
	{ 0xa438, 0x1a96 },	\
	{ 0xa438, 0xf705 },	\
	{ 0xa438, 0xeeff },	\
	{ 0xa438, 0xd200 },	\
	{ 0xa438, 0xdbf6 },	\
	{ 0xa438, 0x05ef },	\
	{ 0xa438, 0x021f },	\
	{ 0xa438, 0x110d },	\
	{ 0xa438, 0x42bf },	\
	{ 0xa438, 0x8b3c },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x4eef },	\
	{ 0xa438, 0x021b },	\
	{ 0xa438, 0x031f },	\
	{ 0xa438, 0x110d },	\
	{ 0xa438, 0x42bf },	\
	{ 0xa438, 0x8b36 },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x4eef },	\
	{ 0xa438, 0x021a },	\
	{ 0xa438, 0x031f },	\
	{ 0xa438, 0x110d },	\
	{ 0xa438, 0x42bf },	\
	{ 0xa438, 0x8b39 },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x4ebf },	\
	{ 0xa438, 0xc23f },	\
	{ 0xa438, 0x1a96 },	\
	{ 0xa438, 0xf705 },	\
	{ 0xa438, 0xeeff },	\
	{ 0xa438, 0xd200 },	\
	{ 0xa438, 0xdaf6 },	\
	{ 0xa438, 0x05bf },	\
	{ 0xa438, 0xc24f },	\
	{ 0xa438, 0x1a96 },	\
	{ 0xa438, 0xf705 },	\
	{ 0xa438, 0xeeff },	\
	{ 0xa438, 0xd200 },	\
	{ 0xa438, 0xdbf6 },	\
	{ 0xa438, 0x05ef },	\
	{ 0xa438, 0x021f },	\
	{ 0xa438, 0x110d },	\
	{ 0xa438, 0x42bf },	\
	{ 0xa438, 0x8b45 },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x4eef },	\
	{ 0xa438, 0x021b },	\
	{ 0xa438, 0x031f },	\
	{ 0xa438, 0x110d },	\
	{ 0xa438, 0x42bf },	\
	{ 0xa438, 0x8b3f },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x4eef },	\
	{ 0xa438, 0x021a },	\
	{ 0xa438, 0x031f },	\
	{ 0xa438, 0x110d },	\
	{ 0xa438, 0x42bf },	\
	{ 0xa438, 0x8b42 },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x4eef },	\
	{ 0xa438, 0x56d0 },	\
	{ 0xa438, 0x201f },	\
	{ 0xa438, 0x11bf },	\
	{ 0xa438, 0x8b4e },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x4ebf },	\
	{ 0xa438, 0x8b48 },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x4ebf },	\
	{ 0xa438, 0x8b4b },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x4ee1 },	\
	{ 0xa438, 0x8578 },	\
	{ 0xa438, 0xef03 },	\
	{ 0xa438, 0x480a },	\
	{ 0xa438, 0x2805 },	\
	{ 0xa438, 0xef20 },	\
	{ 0xa438, 0x1b01 },	\
	{ 0xa438, 0xad27 },	\
	{ 0xa438, 0x3f1f },	\
	{ 0xa438, 0x44e0 },	\
	{ 0xa438, 0x8560 },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x61bf },	\
	{ 0xa438, 0x8b51 },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x4ee0 },	\
	{ 0xa438, 0x8566 },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x67bf },	\
	{ 0xa438, 0x8b54 },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x4ee0 },	\
	{ 0xa438, 0x856c },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x6dbf },	\
	{ 0xa438, 0x8b57 },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x4ee0 },	\
	{ 0xa438, 0x8572 },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x73bf },	\
	{ 0xa438, 0x8b5a },	\
	{ 0xa438, 0x026c },	\
	{ 0xa438, 0x4ee1 },	\
	{ 0xa438, 0x8fb8 },	\
	{ 0xa438, 0x5900 },	\
	{ 0xa438, 0xf728 },	\
	{ 0xa438, 0xe58f },	\
	{ 0xa438, 0xb8af },	\
	{ 0xa438, 0x8b2c },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x791b },	\
	{ 0xa438, 0x21ad },	\
	{ 0xa438, 0x373e },	\
	{ 0xa438, 0x1f44 },	\
	{ 0xa438, 0xe085 },	\
	{ 0xa438, 0x62e1 },	\
	{ 0xa438, 0x8563 },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x5102 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0xe085 },	\
	{ 0xa438, 0x68e1 },	\
	{ 0xa438, 0x8569 },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x5402 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0xe085 },	\
	{ 0xa438, 0x6ee1 },	\
	{ 0xa438, 0x856f },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x5702 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0xe085 },	\
	{ 0xa438, 0x74e1 },	\
	{ 0xa438, 0x8575 },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x5a02 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xb859 },	\
	{ 0xa438, 0x00f7 },	\
	{ 0xa438, 0x28e5 },	\
	{ 0xa438, 0x8fb8 },	\
	{ 0xa438, 0xae4a },	\
	{ 0xa438, 0x1f44 },	\
	{ 0xa438, 0xe085 },	\
	{ 0xa438, 0x64e1 },	\
	{ 0xa438, 0x8565 },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x5102 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0xe085 },	\
	{ 0xa438, 0x6ae1 },	\
	{ 0xa438, 0x856b },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x5402 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0xe085 },	\
	{ 0xa438, 0x70e1 },	\
	{ 0xa438, 0x8571 },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x5702 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0xe085 },	\
	{ 0xa438, 0x76e1 },	\
	{ 0xa438, 0x8577 },	\
	{ 0xa438, 0xbf8b },	\
	{ 0xa438, 0x5a02 },	\
	{ 0xa438, 0x6c4e },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xb859 },	\
	{ 0xa438, 0x00f7 },	\
	{ 0xa438, 0x28e5 },	\
	{ 0xa438, 0x8fb8 },	\
	{ 0xa438, 0xae0c },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xb839 },	\
	{ 0xa438, 0x04ac },	\
	{ 0xa438, 0x2f04 },	\
	{ 0xa438, 0xee8f },	\
	{ 0xa438, 0xb800 },	\
	{ 0xa438, 0xfefd },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0x8efc },	\
	{ 0xa438, 0xac8c },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xfaf0 },	\
	{ 0xa438, 0xacf8 },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xf6f0 },	\
	{ 0xa438, 0xad00 },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xfef0 },	\
	{ 0xa438, 0xacfc },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xf4f0 },	\
	{ 0xa438, 0xacf2 },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xf0f0 },	\
	{ 0xa438, 0xacb0 },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xaef0 },	\
	{ 0xa438, 0xacac },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xaaf0 },	\
	{ 0xa438, 0xacee },	\
	{ 0xa438, 0xf0b0 },	\
	{ 0xa438, 0x24f0 },	\
	{ 0xa438, 0xb0a4 },	\
	{ 0xa438, 0xf0b1 },	\
	{ 0xa438, 0x24f0 },	\
	{ 0xa438, 0xb1a4 },	\
	{ 0xa438, 0xee8f },	\
	{ 0xa438, 0xb800 },	\
	{ 0xa438, 0xd400 },	\
	{ 0xa438, 0x00af },	\
	{ 0xa438, 0x3976 },	\
	{ 0xa438, 0x66ac },	\
	{ 0xa438, 0xeabb },	\
	{ 0xa438, 0xa430 },	\
	{ 0xa438, 0x6e50 },	\
	{ 0xa438, 0x6e53 },	\
	{ 0xa438, 0x6e56 },	\
	{ 0xa438, 0x6e59 },	\
	{ 0xa438, 0x6e5c },	\
	{ 0xa438, 0x6e5f },	\
	{ 0xa438, 0x6e62 },	\
	{ 0xa438, 0x6e65 },	\
	{ 0xa438, 0xd9ac },	\
	{ 0xa438, 0x70f0 },	\
	{ 0xa438, 0xac6a },	\
	{ 0xa436, 0xb85e },	\
	{ 0xa438, 0x23b7 },	\
	{ 0xa436, 0xb860 },	\
	{ 0xa438, 0x74db },	\
	{ 0xa436, 0xb862 },	\
	{ 0xa438, 0x268c },	\
	{ 0xa436, 0xb864 },	\
	{ 0xa438, 0x3fe5 },	\
	{ 0xa436, 0xb886 },	\
	{ 0xa438, 0x2250 },	\
	{ 0xa436, 0xb888 },	\
	{ 0xa438, 0x140e },	\
	{ 0xa436, 0xb88a },	\
	{ 0xa438, 0x3696 },	\
	{ 0xa436, 0xb88c },	\
	{ 0xa438, 0x3973 },	\
	{ 0xa436, 0xb838 },	\
	{ 0xa438, 0x00ff },	\
	{ 0xb820, 0x0010 },	\
	{ 0xa436, 0x8464 },	\
	{ 0xa438, 0xaf84 },	\
	{ 0xa438, 0x7caf },	\
	{ 0xa438, 0x8485 },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0x13af },	\
	{ 0xa438, 0x851e },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0xb9af },	\
	{ 0xa438, 0x8684 },	\
	{ 0xa438, 0xaf87 },	\
	{ 0xa438, 0x01af },	\
	{ 0xa438, 0x8701 },	\
	{ 0xa438, 0xac38 },	\
	{ 0xa438, 0x03af },	\
	{ 0xa438, 0x38bb },	\
	{ 0xa438, 0xaf38 },	\
	{ 0xa438, 0xc302 },	\
	{ 0xa438, 0x4618 },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0x0a02 },	\
	{ 0xa438, 0x54b7 },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0x1002 },	\
	{ 0xa438, 0x54c0 },	\
	{ 0xa438, 0xd400 },	\
	{ 0xa438, 0x0fbf },	\
	{ 0xa438, 0x8507 },	\
	{ 0xa438, 0x024f },	\
	{ 0xa438, 0x48bf },	\
	{ 0xa438, 0x8504 },	\
	{ 0xa438, 0x024f },	\
	{ 0xa438, 0x6759 },	\
	{ 0xa438, 0xf0a1 },	\
	{ 0xa438, 0x3008 },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0x0d02 },	\
	{ 0xa438, 0x54c0 },	\
	{ 0xa438, 0xae06 },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0x0d02 },	\
	{ 0xa438, 0x54b7 },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0x0402 },	\
	{ 0xa438, 0x4f67 },	\
	{ 0xa438, 0xa183 },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x15a1 },	\
	{ 0xa438, 0x8502 },	\
	{ 0xa438, 0xae10 },	\
	{ 0xa438, 0x59f0 },	\
	{ 0xa438, 0xa180 },	\
	{ 0xa438, 0x16bf },	\
	{ 0xa438, 0x8501 },	\
	{ 0xa438, 0x024f },	\
	{ 0xa438, 0x67a1 },	\
	{ 0xa438, 0x381b },	\
	{ 0xa438, 0xae0b },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xffbf },	\
	{ 0xa438, 0x84fe },	\
	{ 0xa438, 0x024f },	\
	{ 0xa438, 0x48ae },	\
	{ 0xa438, 0x17bf },	\
	{ 0xa438, 0x84fe },	\
	{ 0xa438, 0x0254 },	\
	{ 0xa438, 0xb7bf },	\
	{ 0xa438, 0x84fb },	\
	{ 0xa438, 0x0254 },	\
	{ 0xa438, 0xb7ae },	\
	{ 0xa438, 0x09a1 },	\
	{ 0xa438, 0x5006 },	\
	{ 0xa438, 0xbf84 },	\
	{ 0xa438, 0xfb02 },	\
	{ 0xa438, 0x54c0 },	\
	{ 0xa438, 0xaf04 },	\
	{ 0xa438, 0x4700 },	\
	{ 0xa438, 0xad34 },	\
	{ 0xa438, 0xfdad },	\
	{ 0xa438, 0x0670 },	\
	{ 0xa438, 0xae14 },	\
	{ 0xa438, 0xf0a6 },	\
	{ 0xa438, 0x00b8 },	\
	{ 0xa438, 0xbd32 },	\
	{ 0xa438, 0x30bd },	\
	{ 0xa438, 0x30aa },	\
	{ 0xa438, 0xbd2c },	\
	{ 0xa438, 0xccbd },	\
	{ 0xa438, 0x2ca1 },	\
	{ 0xa438, 0x0705 },	\
	{ 0xa438, 0xec80 },	\
	{ 0xa438, 0xaf40 },	\
	{ 0xa438, 0xf7af },	\
	{ 0xa438, 0x40f5 },	\
	{ 0xa438, 0xd101 },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xa402 },	\
	{ 0xa438, 0x4f48 },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xa702 },	\
	{ 0xa438, 0x54c0 },	\
	{ 0xa438, 0xd10f },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xaa02 },	\
	{ 0xa438, 0x4f48 },	\
	{ 0xa438, 0x024d },	\
	{ 0xa438, 0x6abf },	\
	{ 0xa438, 0x85ad },	\
	{ 0xa438, 0x024f },	\
	{ 0xa438, 0x67bf },	\
	{ 0xa438, 0x8ff7 },	\
	{ 0xa438, 0xddbf },	\
	{ 0xa438, 0x85b0 },	\
	{ 0xa438, 0x024f },	\
	{ 0xa438, 0x67bf },	\
	{ 0xa438, 0x8ff8 },	\
	{ 0xa438, 0xddbf },	\
	{ 0xa438, 0x85b3 },	\
	{ 0xa438, 0x024f },	\
	{ 0xa438, 0x67bf },	\
	{ 0xa438, 0x8ff9 },	\
	{ 0xa438, 0xddbf },	\
	{ 0xa438, 0x85b6 },	\
	{ 0xa438, 0x024f },	\
	{ 0xa438, 0x67bf },	\
	{ 0xa438, 0x8ffa },	\
	{ 0xa438, 0xddd1 },	\
	{ 0xa438, 0x00bf },	\
	{ 0xa438, 0x85aa },	\
	{ 0xa438, 0x024f },	\
	{ 0xa438, 0x4802 },	\
	{ 0xa438, 0x4d6a },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xad02 },	\
	{ 0xa438, 0x4f67 },	\
	{ 0xa438, 0xbf8f },	\
	{ 0xa438, 0xfbdd },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xb002 },	\
	{ 0xa438, 0x4f67 },	\
	{ 0xa438, 0xbf8f },	\
	{ 0xa438, 0xfcdd },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xb302 },	\
	{ 0xa438, 0x4f67 },	\
	{ 0xa438, 0xbf8f },	\
	{ 0xa438, 0xfddd },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xb602 },	\
	{ 0xa438, 0x4f67 },	\
	{ 0xa438, 0xbf8f },	\
	{ 0xa438, 0xfedd },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xa702 },	\
	{ 0xa438, 0x54b7 },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xa102 },	\
	{ 0xa438, 0x54b7 },	\
	{ 0xa438, 0xaf3c },	\
	{ 0xa438, 0x2066 },	\
	{ 0xa438, 0xb800 },	\
	{ 0xa438, 0xb8bd },	\
	{ 0xa438, 0x30ee },	\
	{ 0xa438, 0xbd2c },	\
	{ 0xa438, 0xb8bd },	\
	{ 0xa438, 0x7040 },	\
	{ 0xa438, 0xbd86 },	\
	{ 0xa438, 0xc8bd },	\
	{ 0xa438, 0x8640 },	\
	{ 0xa438, 0xbd88 },	\
	{ 0xa438, 0xc8bd },	\
	{ 0xa438, 0x8802 },	\
	{ 0xa438, 0x1929 },	\
	{ 0xa438, 0xa202 },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x03a2 },	\
	{ 0xa438, 0x032e },	\
	{ 0xa438, 0xd10f },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xaa02 },	\
	{ 0xa438, 0x4f48 },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xf7bf },	\
	{ 0xa438, 0x85ad },	\
	{ 0xa438, 0x024f },	\
	{ 0xa438, 0x48e1 },	\
	{ 0xa438, 0x8ff8 },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xb002 },	\
	{ 0xa438, 0x4f48 },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xf9bf },	\
	{ 0xa438, 0x85b3 },	\
	{ 0xa438, 0x024f },	\
	{ 0xa438, 0x48e1 },	\
	{ 0xa438, 0x8ffa },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xb602 },	\
	{ 0xa438, 0x4f48 },	\
	{ 0xa438, 0xae2c },	\
	{ 0xa438, 0xd100 },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xaa02 },	\
	{ 0xa438, 0x4f48 },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xfbbf },	\
	{ 0xa438, 0x85ad },	\
	{ 0xa438, 0x024f },	\
	{ 0xa438, 0x48e1 },	\
	{ 0xa438, 0x8ffc },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xb002 },	\
	{ 0xa438, 0x4f48 },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xfdbf },	\
	{ 0xa438, 0x85b3 },	\
	{ 0xa438, 0x024f },	\
	{ 0xa438, 0x48e1 },	\
	{ 0xa438, 0x8ffe },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xb602 },	\
	{ 0xa438, 0x4f48 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0x7e02 },	\
	{ 0xa438, 0x4f67 },	\
	{ 0xa438, 0xa100 },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x25a1 },	\
	{ 0xa438, 0x041d },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xf1bf },	\
	{ 0xa438, 0x8675 },	\
	{ 0xa438, 0x024f },	\
	{ 0xa438, 0x48e1 },	\
	{ 0xa438, 0x8ff2 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0x7802 },	\
	{ 0xa438, 0x4f48 },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xf3bf },	\
	{ 0xa438, 0x867b },	\
	{ 0xa438, 0x024f },	\
	{ 0xa438, 0x48ae },	\
	{ 0xa438, 0x29a1 },	\
	{ 0xa438, 0x070b },	\
	{ 0xa438, 0xae24 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0x8102 },	\
	{ 0xa438, 0x4f67 },	\
	{ 0xa438, 0xad28 },	\
	{ 0xa438, 0x1be1 },	\
	{ 0xa438, 0x8ff4 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0x7502 },	\
	{ 0xa438, 0x4f48 },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xf5bf },	\
	{ 0xa438, 0x8678 },	\
	{ 0xa438, 0x024f },	\
	{ 0xa438, 0x48e1 },	\
	{ 0xa438, 0x8ff6 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0x7b02 },	\
	{ 0xa438, 0x4f48 },	\
	{ 0xa438, 0xaf09 },	\
	{ 0xa438, 0x8420 },	\
	{ 0xa438, 0xbc32 },	\
	{ 0xa438, 0x20bc },	\
	{ 0xa438, 0x3e76 },	\
	{ 0xa438, 0xbc08 },	\
	{ 0xa438, 0xfda6 },	\
	{ 0xa438, 0x1a00 },	\
	{ 0xa438, 0xb64e },	\
	{ 0xa438, 0xd101 },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xa402 },	\
	{ 0xa438, 0x4f48 },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xa702 },	\
	{ 0xa438, 0x54c0 },	\
	{ 0xa438, 0xd10f },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xaa02 },	\
	{ 0xa438, 0x4f48 },	\
	{ 0xa438, 0x024d },	\
	{ 0xa438, 0x6abf },	\
	{ 0xa438, 0x85ad },	\
	{ 0xa438, 0x024f },	\
	{ 0xa438, 0x67bf },	\
	{ 0xa438, 0x8ff7 },	\
	{ 0xa438, 0xddbf },	\
	{ 0xa438, 0x85b0 },	\
	{ 0xa438, 0x024f },	\
	{ 0xa438, 0x67bf },	\
	{ 0xa438, 0x8ff8 },	\
	{ 0xa438, 0xddbf },	\
	{ 0xa438, 0x85b3 },	\
	{ 0xa438, 0x024f },	\
	{ 0xa438, 0x67bf },	\
	{ 0xa438, 0x8ff9 },	\
	{ 0xa438, 0xddbf },	\
	{ 0xa438, 0x85b6 },	\
	{ 0xa438, 0x024f },	\
	{ 0xa438, 0x67bf },	\
	{ 0xa438, 0x8ffa },	\
	{ 0xa438, 0xddd1 },	\
	{ 0xa438, 0x00bf },	\
	{ 0xa438, 0x85aa },	\
	{ 0xa438, 0x024f },	\
	{ 0xa438, 0x4802 },	\
	{ 0xa438, 0x4d6a },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xad02 },	\
	{ 0xa438, 0x4f67 },	\
	{ 0xa438, 0xbf8f },	\
	{ 0xa438, 0xfbdd },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xb002 },	\
	{ 0xa438, 0x4f67 },	\
	{ 0xa438, 0xbf8f },	\
	{ 0xa438, 0xfcdd },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xb302 },	\
	{ 0xa438, 0x4f67 },	\
	{ 0xa438, 0xbf8f },	\
	{ 0xa438, 0xfddd },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xb602 },	\
	{ 0xa438, 0x4f67 },	\
	{ 0xa438, 0xbf8f },	\
	{ 0xa438, 0xfedd },	\
	{ 0xa438, 0xbf85 },	\
	{ 0xa438, 0xa702 },	\
	{ 0xa438, 0x54b7 },	\
	{ 0xa438, 0xaf00 },	\
	{ 0xa438, 0x8800 },	\
	{ 0xa436, 0xb818 },	\
	{ 0xa438, 0x38b8 },	\
	{ 0xa436, 0xb81a },	\
	{ 0xa438, 0x0444 },	\
	{ 0xa436, 0xb81c },	\
	{ 0xa438, 0x40ee },	\
	{ 0xa436, 0xb81e },	\
	{ 0xa438, 0x3c1a },	\
	{ 0xa436, 0xb850 },	\
	{ 0xa438, 0x0981 },	\
	{ 0xa436, 0xb852 },	\
	{ 0xa438, 0x0085 },	\
	{ 0xa436, 0xb878 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb884 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb832 },	\
	{ 0xa438, 0x003f },	\
	{ 0xa436, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xb82e },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0x8024 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xb820, 0x0000 },	\
	{ 0xa436, 0x801e },	\
	{ 0xa438, 0x0021 }

#define RTL8125_MAC_CFG5_MCU	\
	{ 0xa436, 0x8024 },	\
	{ 0xa438, 0x3701 },	\
	{ 0xa436, 0xb82e },	\
	{ 0xa438, 0x0001 },	\
	{ 0xb820, 0x0090 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8024 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8051 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8057 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8063 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8068 },	\
	{ 0xa438, 0xd093 },	\
	{ 0xa438, 0xd1c4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x135c },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x5fbc },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xc9f1 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0fc9 },	\
	{ 0xa438, 0xbb50 },	\
	{ 0xa438, 0xd505 },	\
	{ 0xa438, 0xa202 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0x8c0f },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1519 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1548 },	\
	{ 0xa438, 0x2f70 },	\
	{ 0xa438, 0x802a },	\
	{ 0xa438, 0x2f73 },	\
	{ 0xa438, 0x156a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x155c },	\
	{ 0xa438, 0xd505 },	\
	{ 0xa438, 0xa202 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1551 },	\
	{ 0xa438, 0xc0c1 },	\
	{ 0xa438, 0xc0c0 },	\
	{ 0xa438, 0xd05a },	\
	{ 0xa438, 0xd1ba },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x2529 },	\
	{ 0xa438, 0x022a },	\
	{ 0xa438, 0xd0a7 },	\
	{ 0xa438, 0xd1b9 },	\
	{ 0xa438, 0xa208 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x080e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x408b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a65 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a6b },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0915 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0909 },	\
	{ 0xa438, 0x228f },	\
	{ 0xa438, 0x8038 },	\
	{ 0xa438, 0x9801 },	\
	{ 0xa438, 0xd71e },	\
	{ 0xa438, 0x5d61 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x022a },	\
	{ 0xa438, 0x2005 },	\
	{ 0xa438, 0x091a },	\
	{ 0xa438, 0x3bd9 },	\
	{ 0xa438, 0x0919 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0916 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14c5 },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x3181 },	\
	{ 0xa438, 0x8061 },	\
	{ 0xa438, 0x60ad },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x135c },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x5fba },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0cc7 },	\
	{ 0xa438, 0xd096 },	\
	{ 0xa438, 0xd1a9 },	\
	{ 0xa438, 0xd503 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0c94 },	\
	{ 0xa436, 0xa026 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa024 },	\
	{ 0xa438, 0x0c93 },	\
	{ 0xa436, 0xa022 },	\
	{ 0xa438, 0x0cc5 },	\
	{ 0xa436, 0xa020 },	\
	{ 0xa438, 0x0915 },	\
	{ 0xa436, 0xa006 },	\
	{ 0xa438, 0x020a },	\
	{ 0xa436, 0xa004 },	\
	{ 0xa438, 0x155b },	\
	{ 0xa436, 0xa002 },	\
	{ 0xa438, 0x1542 },	\
	{ 0xa436, 0xa000 },	\
	{ 0xa438, 0x0fc7 },	\
	{ 0xa436, 0xa008 },	\
	{ 0xa438, 0x7f00 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0010 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802c },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6090 },	\
	{ 0xa438, 0x60d1 },	\
	{ 0xa438, 0xc95c },	\
	{ 0xa438, 0xf007 },	\
	{ 0xa438, 0x60b1 },	\
	{ 0xa438, 0xc95a },	\
	{ 0xa438, 0xf004 },	\
	{ 0xa438, 0xc956 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xc94e },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x00cd },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6090 },	\
	{ 0xa438, 0x60d1 },	\
	{ 0xa438, 0xc95c },	\
	{ 0xa438, 0xf007 },	\
	{ 0xa438, 0x60b1 },	\
	{ 0xa438, 0xc95a },	\
	{ 0xa438, 0xf004 },	\
	{ 0xa438, 0xc956 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xc94e },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x022a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0132 },	\
	{ 0xa436, 0xa08e },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa08c },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa08a },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa088 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa086 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa084 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa082 },	\
	{ 0xa438, 0x012f },	\
	{ 0xa436, 0xa080 },	\
	{ 0xa438, 0x00cc },	\
	{ 0xa436, 0xa090 },	\
	{ 0xa438, 0x0103 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0020 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801e },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8026 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8036 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8036 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8036 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8036 },	\
	{ 0xa438, 0xd107 },	\
	{ 0xa438, 0xd042 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5ff4 },	\
	{ 0xa438, 0x8280 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6065 },	\
	{ 0xa438, 0xd125 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xd12b },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x077f },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0c50 },	\
	{ 0xa438, 0xd104 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5ff4 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0a2e },	\
	{ 0xa438, 0xcb9b },	\
	{ 0xa438, 0xd110 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0b7b },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5ff4 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x081b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x09df },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x7fb8 },	\
	{ 0xa438, 0xa718 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x074e },	\
	{ 0xa436, 0xa10e },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa10c },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa10a },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa108 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa106 },	\
	{ 0xa438, 0x074d },	\
	{ 0xa436, 0xa104 },	\
	{ 0xa438, 0x0818 },	\
	{ 0xa436, 0xa102 },	\
	{ 0xa438, 0x0a2c },	\
	{ 0xa436, 0xa100 },	\
	{ 0xa438, 0x077e },	\
	{ 0xa436, 0xa110 },	\
	{ 0xa438, 0x000f },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8625 },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0xaf86 },	\
	{ 0xa438, 0x3daf },	\
	{ 0xa438, 0x8689 },	\
	{ 0xa438, 0xaf88 },	\
	{ 0xa438, 0x69af },	\
	{ 0xa438, 0x8887 },	\
	{ 0xa438, 0xaf88 },	\
	{ 0xa438, 0x9caf },	\
	{ 0xa438, 0x889c },	\
	{ 0xa438, 0xaf88 },	\
	{ 0xa438, 0x9caf },	\
	{ 0xa438, 0x889c },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0x49d7 },	\
	{ 0xa438, 0x0040 },	\
	{ 0xa438, 0x0277 },	\
	{ 0xa438, 0x7daf },	\
	{ 0xa438, 0x2727 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7205 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7208 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x71f3 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x71f6 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7229 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x722c },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7217 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x721a },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x721d },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7211 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7220 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7214 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x722f },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7223 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7232 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7226 },	\
	{ 0xa438, 0xf8f9 },	\
	{ 0xa438, 0xfae0 },	\
	{ 0xa438, 0x85b3 },	\
	{ 0xa438, 0x3802 },	\
	{ 0xa438, 0xad27 },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x03af },	\
	{ 0xa438, 0x8830 },	\
	{ 0xa438, 0x1f66 },	\
	{ 0xa438, 0xef65 },	\
	{ 0xa438, 0xbfc2 },	\
	{ 0xa438, 0x1f1a },	\
	{ 0xa438, 0x96f7 },	\
	{ 0xa438, 0x05ee },	\
	{ 0xa438, 0xffd2 },	\
	{ 0xa438, 0x00da },	\
	{ 0xa438, 0xf605 },	\
	{ 0xa438, 0xbfc2 },	\
	{ 0xa438, 0x2f1a },	\
	{ 0xa438, 0x96f7 },	\
	{ 0xa438, 0x05ee },	\
	{ 0xa438, 0xffd2 },	\
	{ 0xa438, 0x00db },	\
	{ 0xa438, 0xf605 },	\
	{ 0xa438, 0xef02 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0x0d42 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x4202 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xef02 },	\
	{ 0xa438, 0x1b03 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0x0d42 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x4502 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xef02 },	\
	{ 0xa438, 0x1a03 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0x0d42 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x4802 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xbfc2 },	\
	{ 0xa438, 0x3f1a },	\
	{ 0xa438, 0x96f7 },	\
	{ 0xa438, 0x05ee },	\
	{ 0xa438, 0xffd2 },	\
	{ 0xa438, 0x00da },	\
	{ 0xa438, 0xf605 },	\
	{ 0xa438, 0xbfc2 },	\
	{ 0xa438, 0x4f1a },	\
	{ 0xa438, 0x96f7 },	\
	{ 0xa438, 0x05ee },	\
	{ 0xa438, 0xffd2 },	\
	{ 0xa438, 0x00db },	\
	{ 0xa438, 0xf605 },	\
	{ 0xa438, 0xef02 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0x0d42 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x4b02 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xef02 },	\
	{ 0xa438, 0x1b03 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0x0d42 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x4e02 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xef02 },	\
	{ 0xa438, 0x1a03 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0x0d42 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x5102 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xef56 },	\
	{ 0xa438, 0xd020 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x5402 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x5702 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x5a02 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0xa0ef },	\
	{ 0xa438, 0x0348 },	\
	{ 0xa438, 0x0a28 },	\
	{ 0xa438, 0x05ef },	\
	{ 0xa438, 0x201b },	\
	{ 0xa438, 0x01ad },	\
	{ 0xa438, 0x2735 },	\
	{ 0xa438, 0x1f44 },	\
	{ 0xa438, 0xe085 },	\
	{ 0xa438, 0x88e1 },	\
	{ 0xa438, 0x8589 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x5d02 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xe085 },	\
	{ 0xa438, 0x8ee1 },	\
	{ 0xa438, 0x858f },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6002 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xe085 },	\
	{ 0xa438, 0x94e1 },	\
	{ 0xa438, 0x8595 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6302 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xe085 },	\
	{ 0xa438, 0x9ae1 },	\
	{ 0xa438, 0x859b },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6602 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xaf88 },	\
	{ 0xa438, 0x3cbf },	\
	{ 0xa438, 0x883f },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x9cad },	\
	{ 0xa438, 0x2835 },	\
	{ 0xa438, 0x1f44 },	\
	{ 0xa438, 0xe08f },	\
	{ 0xa438, 0xf8e1 },	\
	{ 0xa438, 0x8ff9 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x5d02 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xe08f },	\
	{ 0xa438, 0xfae1 },	\
	{ 0xa438, 0x8ffb },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6002 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xe08f },	\
	{ 0xa438, 0xfce1 },	\
	{ 0xa438, 0x8ffd },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6302 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xe08f },	\
	{ 0xa438, 0xfee1 },	\
	{ 0xa438, 0x8fff },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6602 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xaf88 },	\
	{ 0xa438, 0x3ce1 },	\
	{ 0xa438, 0x85a1 },	\
	{ 0xa438, 0x1b21 },	\
	{ 0xa438, 0xad37 },	\
	{ 0xa438, 0x341f },	\
	{ 0xa438, 0x44e0 },	\
	{ 0xa438, 0x858a },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x8bbf },	\
	{ 0xa438, 0x885d },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7de0 },	\
	{ 0xa438, 0x8590 },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x91bf },	\
	{ 0xa438, 0x8860 },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7de0 },	\
	{ 0xa438, 0x8596 },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x97bf },	\
	{ 0xa438, 0x8863 },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7de0 },	\
	{ 0xa438, 0x859c },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x9dbf },	\
	{ 0xa438, 0x8866 },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7dae },	\
	{ 0xa438, 0x401f },	\
	{ 0xa438, 0x44e0 },	\
	{ 0xa438, 0x858c },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x8dbf },	\
	{ 0xa438, 0x885d },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7de0 },	\
	{ 0xa438, 0x8592 },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x93bf },	\
	{ 0xa438, 0x8860 },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7de0 },	\
	{ 0xa438, 0x8598 },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x99bf },	\
	{ 0xa438, 0x8863 },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7de0 },	\
	{ 0xa438, 0x859e },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x9fbf },	\
	{ 0xa438, 0x8866 },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7dae },	\
	{ 0xa438, 0x0ce1 },	\
	{ 0xa438, 0x85b3 },	\
	{ 0xa438, 0x3904 },	\
	{ 0xa438, 0xac2f },	\
	{ 0xa438, 0x04ee },	\
	{ 0xa438, 0x85b3 },	\
	{ 0xa438, 0x00af },	\
	{ 0xa438, 0x39d9 },	\
	{ 0xa438, 0x22ac },	\
	{ 0xa438, 0xeaf0 },	\
	{ 0xa438, 0xacf6 },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xfaf0 },	\
	{ 0xa438, 0xacf8 },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xfcf0 },	\
	{ 0xa438, 0xad00 },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xfef0 },	\
	{ 0xa438, 0xacf0 },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xf4f0 },	\
	{ 0xa438, 0xacf2 },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xb0f0 },	\
	{ 0xa438, 0xacae },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xacf0 },	\
	{ 0xa438, 0xacaa },	\
	{ 0xa438, 0xa100 },	\
	{ 0xa438, 0x0ce1 },	\
	{ 0xa438, 0x8ff7 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x8402 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xaf26 },	\
	{ 0xa438, 0xe9e1 },	\
	{ 0xa438, 0x8ff6 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x8402 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xaf26 },	\
	{ 0xa438, 0xf520 },	\
	{ 0xa438, 0xac86 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x3f02 },	\
	{ 0xa438, 0x6e9c },	\
	{ 0xa438, 0xad28 },	\
	{ 0xa438, 0x03af },	\
	{ 0xa438, 0x3324 },	\
	{ 0xa438, 0xad38 },	\
	{ 0xa438, 0x03af },	\
	{ 0xa438, 0x32e6 },	\
	{ 0xa438, 0xaf32 },	\
	{ 0xa438, 0xfb00 },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8ff6 },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0x0705 },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8ff8 },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0x19cc },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8ffa },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0x28e3 },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8ffc },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0x1047 },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8ffe },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0x0a45 },	\
	{ 0xa436, 0xb85e },	\
	{ 0xa438, 0x271e },	\
	{ 0xa436, 0xb860 },	\
	{ 0xa438, 0x3846 },	\
	{ 0xa436, 0xb862 },	\
	{ 0xa438, 0x26e6 },	\
	{ 0xa436, 0xb864 },	\
	{ 0xa438, 0x32e3 },	\
	{ 0xa436, 0xb886 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb888 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb88a },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb88c },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb838 },	\
	{ 0xa438, 0x000f },	\
	{ 0xb820, 0x0010 },	\
	{ 0xa436, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xb82e },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0x8024 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xb820, 0x0000 },	\
	{ 0xa436, 0x801e },	\
	{ 0xa438, 0x0015 }
