# bash completion for rclone                               -*- shell-script -*-

__rclone_debug()
{
    if [[ -n ${BASH_COMP_DEBUG_FILE} ]]; then
        echo "$*" >> "${BASH_COMP_DEBUG_FILE}"
    fi
}

# Homebrew on Macs have version 1.3 of bash-completion which doesn't include
# _init_completion. This is a very minimal version of that function.
__rclone_init_completion()
{
    COMPREPLY=()
    _get_comp_words_by_ref "$@" cur prev words cword
}

__rclone_index_of_word()
{
    local w word=$1
    shift
    index=0
    for w in "$@"; do
        [[ $w = "$word" ]] && return
        index=$((index+1))
    done
    index=-1
}

__rclone_contains_word()
{
    local w word=$1; shift
    for w in "$@"; do
        [[ $w = "$word" ]] && return
    done
    return 1
}

__rclone_handle_reply()
{
    __rclone_debug "${FUNCNAME[0]}"
    case $cur in
        -*)
            if [[ $(type -t compopt) = "builtin" ]]; then
                compopt -o nospace
            fi
            local allflags
            if [ ${#must_have_one_flag[@]} -ne 0 ]; then
                allflags=("${must_have_one_flag[@]}")
            else
                allflags=("${flags[*]} ${two_word_flags[*]}")
            fi
            COMPREPLY=( $(compgen -W "${allflags[*]}" -- "$cur") )
            if [[ $(type -t compopt) = "builtin" ]]; then
                [[ "${COMPREPLY[0]}" == *= ]] || compopt +o nospace
            fi

            # complete after --flag=abc
            if [[ $cur == *=* ]]; then
                if [[ $(type -t compopt) = "builtin" ]]; then
                    compopt +o nospace
                fi

                local index flag
                flag="${cur%=*}"
                __rclone_index_of_word "${flag}" "${flags_with_completion[@]}"
                COMPREPLY=()
                if [[ ${index} -ge 0 ]]; then
                    PREFIX=""
                    cur="${cur#*=}"
                    ${flags_completion[${index}]}
                    if [ -n "${ZSH_VERSION}" ]; then
                        # zsh completion needs --flag= prefix
                        eval "COMPREPLY=( \"\${COMPREPLY[@]/#/${flag}=}\" )"
                    fi
                fi
            fi
            return 0;
            ;;
    esac

    # check if we are handling a flag with special work handling
    local index
    __rclone_index_of_word "${prev}" "${flags_with_completion[@]}"
    if [[ ${index} -ge 0 ]]; then
        ${flags_completion[${index}]}
        return
    fi

    # we are parsing a flag and don't have a special handler, no completion
    if [[ ${cur} != "${words[cword]}" ]]; then
        return
    fi

    local completions
    completions=("${commands[@]}")
    if [[ ${#must_have_one_noun[@]} -ne 0 ]]; then
        completions=("${must_have_one_noun[@]}")
    fi
    if [[ ${#must_have_one_flag[@]} -ne 0 ]]; then
        completions+=("${must_have_one_flag[@]}")
    fi
    COMPREPLY=( $(compgen -W "${completions[*]}" -- "$cur") )

    if [[ ${#COMPREPLY[@]} -eq 0 && ${#noun_aliases[@]} -gt 0 && ${#must_have_one_noun[@]} -ne 0 ]]; then
        COMPREPLY=( $(compgen -W "${noun_aliases[*]}" -- "$cur") )
    fi

    if [[ ${#COMPREPLY[@]} -eq 0 ]]; then
		if declare -F __rclone_custom_func >/dev/null; then
			# try command name qualified custom func
			__rclone_custom_func
		else
			# otherwise fall back to unqualified for compatibility
			declare -F __custom_func >/dev/null && __custom_func
		fi
    fi

    # available in bash-completion >= 2, not always present on macOS
    if declare -F __ltrim_colon_completions >/dev/null; then
        __ltrim_colon_completions "$cur"
    fi

    # If there is only 1 completion and it is a flag with an = it will be completed
    # but we don't want a space after the =
    if [[ "${#COMPREPLY[@]}" -eq "1" ]] && [[ $(type -t compopt) = "builtin" ]] && [[ "${COMPREPLY[0]}" == --*= ]]; then
       compopt -o nospace
    fi
}

# The arguments should be in the form "ext1|ext2|extn"
__rclone_handle_filename_extension_flag()
{
    local ext="$1"
    _filedir "@(${ext})"
}

__rclone_handle_subdirs_in_dir_flag()
{
    local dir="$1"
    pushd "${dir}" >/dev/null 2>&1 && _filedir -d && popd >/dev/null 2>&1
}

__rclone_handle_flag()
{
    __rclone_debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"

    # if a command required a flag, and we found it, unset must_have_one_flag()
    local flagname=${words[c]}
    local flagvalue
    # if the word contained an =
    if [[ ${words[c]} == *"="* ]]; then
        flagvalue=${flagname#*=} # take in as flagvalue after the =
        flagname=${flagname%=*} # strip everything after the =
        flagname="${flagname}=" # but put the = back
    fi
    __rclone_debug "${FUNCNAME[0]}: looking for ${flagname}"
    if __rclone_contains_word "${flagname}" "${must_have_one_flag[@]}"; then
        must_have_one_flag=()
    fi

    # if you set a flag which only applies to this command, don't show subcommands
    if __rclone_contains_word "${flagname}" "${local_nonpersistent_flags[@]}"; then
      commands=()
    fi

    # keep flag value with flagname as flaghash
    # flaghash variable is an associative array which is only supported in bash > 3.
    if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
        if [ -n "${flagvalue}" ] ; then
            flaghash[${flagname}]=${flagvalue}
        elif [ -n "${words[ $((c+1)) ]}" ] ; then
            flaghash[${flagname}]=${words[ $((c+1)) ]}
        else
            flaghash[${flagname}]="true" # pad "true" for bool flag
        fi
    fi

    # skip the argument to a two word flag
    if [[ ${words[c]} != *"="* ]] && __rclone_contains_word "${words[c]}" "${two_word_flags[@]}"; then
			  __rclone_debug "${FUNCNAME[0]}: found a flag ${words[c]}, skip the next argument"
        c=$((c+1))
        # if we are looking for a flags value, don't show commands
        if [[ $c -eq $cword ]]; then
            commands=()
        fi
    fi

    c=$((c+1))

}

__rclone_handle_noun()
{
    __rclone_debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"

    if __rclone_contains_word "${words[c]}" "${must_have_one_noun[@]}"; then
        must_have_one_noun=()
    elif __rclone_contains_word "${words[c]}" "${noun_aliases[@]}"; then
        must_have_one_noun=()
    fi

    nouns+=("${words[c]}")
    c=$((c+1))
}

__rclone_handle_command()
{
    __rclone_debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"

    local next_command
    if [[ -n ${last_command} ]]; then
        next_command="_${last_command}_${words[c]//:/__}"
    else
        if [[ $c -eq 0 ]]; then
            next_command="_rclone_root_command"
        else
            next_command="_${words[c]//:/__}"
        fi
    fi
    c=$((c+1))
    __rclone_debug "${FUNCNAME[0]}: looking for ${next_command}"
    declare -F "$next_command" >/dev/null && $next_command
}

__rclone_handle_word()
{
    if [[ $c -ge $cword ]]; then
        __rclone_handle_reply
        return
    fi
    __rclone_debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"
    if [[ "${words[c]}" == -* ]]; then
        __rclone_handle_flag
    elif __rclone_contains_word "${words[c]}" "${commands[@]}"; then
        __rclone_handle_command
    elif [[ $c -eq 0 ]]; then
        __rclone_handle_command
    elif __rclone_contains_word "${words[c]}" "${command_aliases[@]}"; then
        # aliashash variable is an associative array which is only supported in bash > 3.
        if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
            words[c]=${aliashash[${words[c]}]}
            __rclone_handle_command
        else
            __rclone_handle_noun
        fi
    else
        __rclone_handle_noun
    fi
    __rclone_handle_word
}


__rclone_custom_func() {
    if [[ ${#COMPREPLY[@]} -eq 0 ]]; then
        local cur cword prev words
        if declare -F _init_completion > /dev/null; then
            _init_completion -n : || return
        else
            __rclone_init_completion -n : || return
        fi
        if [[ $cur != *:* ]]; then
            local remote
            while IFS= read -r remote; do
                [[ $remote != $cur* ]] || COMPREPLY+=("$remote")
            done < <(command rclone listremotes)
            if [[ ${COMPREPLY[@]} ]]; then
                local paths=("$cur"*)
                [[ ! -f ${paths[0]} ]] || COMPREPLY+=("${paths[@]}")
            fi
        else
            local path=${cur#*:}
            if [[ $path == */* ]]; then
                local prefix=$(eval printf '%s' "${path%/*}")
            else
                local prefix=
            fi
            local line
            while IFS= read -r line; do
                local reply=${prefix:+$prefix/}$line
                [[ $reply != $path* ]] || COMPREPLY+=("$reply")
            done < <(rclone lsf "${cur%%:*}:$prefix" 2>/dev/null)
	    [[ ! ${COMPREPLY[@]} ]] || compopt -o filenames
        fi
        [[ ! ${COMPREPLY[@]} ]] || compopt -o nospace
    fi
}

_rclone_about()
{
    last_command="rclone_about"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--full")
    local_nonpersistent_flags+=("--full")
    flags+=("--json")
    local_nonpersistent_flags+=("--json")
    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_authorize()
{
    last_command="rclone_authorize"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_cachestats()
{
    last_command="rclone_cachestats"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_cat()
{
    last_command="rclone_cat"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--count=")
    two_word_flags+=("--count")
    local_nonpersistent_flags+=("--count=")
    flags+=("--discard")
    local_nonpersistent_flags+=("--discard")
    flags+=("--head=")
    two_word_flags+=("--head")
    local_nonpersistent_flags+=("--head=")
    flags+=("--offset=")
    two_word_flags+=("--offset")
    local_nonpersistent_flags+=("--offset=")
    flags+=("--tail=")
    two_word_flags+=("--tail")
    local_nonpersistent_flags+=("--tail=")
    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_check()
{
    last_command="rclone_check"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--download")
    local_nonpersistent_flags+=("--download")
    flags+=("--one-way")
    local_nonpersistent_flags+=("--one-way")
    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_cleanup()
{
    last_command="rclone_cleanup"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_config_create()
{
    last_command="rclone_config_create"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_config_delete()
{
    last_command="rclone_config_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_config_dump()
{
    last_command="rclone_config_dump"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_config_edit()
{
    last_command="rclone_config_edit"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_config_file()
{
    last_command="rclone_config_file"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_config_password()
{
    last_command="rclone_config_password"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_config_providers()
{
    last_command="rclone_config_providers"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_config_show()
{
    last_command="rclone_config_show"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_config_update()
{
    last_command="rclone_config_update"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_config()
{
    last_command="rclone_config"

    command_aliases=()

    commands=()
    commands+=("create")
    commands+=("delete")
    commands+=("dump")
    commands+=("edit")
    commands+=("file")
    commands+=("password")
    commands+=("providers")
    commands+=("show")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_copy()
{
    last_command="rclone_copy"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--create-empty-src-dirs")
    local_nonpersistent_flags+=("--create-empty-src-dirs")
    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_copyto()
{
    last_command="rclone_copyto"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_copyurl()
{
    last_command="rclone_copyurl"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_cryptcheck()
{
    last_command="rclone_cryptcheck"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--one-way")
    local_nonpersistent_flags+=("--one-way")
    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_cryptdecode()
{
    last_command="rclone_cryptdecode"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--reverse")
    local_nonpersistent_flags+=("--reverse")
    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_dbhashsum()
{
    last_command="rclone_dbhashsum"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_dedupe()
{
    last_command="rclone_dedupe"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--dedupe-mode=")
    two_word_flags+=("--dedupe-mode")
    local_nonpersistent_flags+=("--dedupe-mode=")
    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_delete()
{
    last_command="rclone_delete"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_deletefile()
{
    last_command="rclone_deletefile"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_genautocomplete_bash()
{
    last_command="rclone_genautocomplete_bash"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--help")
    flags+=("-h")
    local_nonpersistent_flags+=("--help")
    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_genautocomplete_zsh()
{
    last_command="rclone_genautocomplete_zsh"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_genautocomplete()
{
    last_command="rclone_genautocomplete"

    command_aliases=()

    commands=()
    commands+=("bash")
    commands+=("zsh")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_gendocs()
{
    last_command="rclone_gendocs"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_hashsum()
{
    last_command="rclone_hashsum"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_link()
{
    last_command="rclone_link"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_listremotes()
{
    last_command="rclone_listremotes"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--long")
    local_nonpersistent_flags+=("--long")
    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_ls()
{
    last_command="rclone_ls"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_lsd()
{
    last_command="rclone_lsd"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_lsf()
{
    last_command="rclone_lsf"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--absolute")
    local_nonpersistent_flags+=("--absolute")
    flags+=("--csv")
    local_nonpersistent_flags+=("--csv")
    flags+=("--dir-slash")
    flags+=("-d")
    local_nonpersistent_flags+=("--dir-slash")
    flags+=("--dirs-only")
    local_nonpersistent_flags+=("--dirs-only")
    flags+=("--files-only")
    local_nonpersistent_flags+=("--files-only")
    flags+=("--format=")
    two_word_flags+=("--format")
    two_word_flags+=("-F")
    local_nonpersistent_flags+=("--format=")
    flags+=("--hash=")
    two_word_flags+=("--hash")
    local_nonpersistent_flags+=("--hash=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--separator=")
    two_word_flags+=("--separator")
    two_word_flags+=("-s")
    local_nonpersistent_flags+=("--separator=")
    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_lsjson()
{
    last_command="rclone_lsjson"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--dirs-only")
    local_nonpersistent_flags+=("--dirs-only")
    flags+=("--encrypted")
    flags+=("-M")
    local_nonpersistent_flags+=("--encrypted")
    flags+=("--files-only")
    local_nonpersistent_flags+=("--files-only")
    flags+=("--hash")
    local_nonpersistent_flags+=("--hash")
    flags+=("--no-modtime")
    local_nonpersistent_flags+=("--no-modtime")
    flags+=("--original")
    local_nonpersistent_flags+=("--original")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_lsl()
{
    last_command="rclone_lsl"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_md5sum()
{
    last_command="rclone_md5sum"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_mkdir()
{
    last_command="rclone_mkdir"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_move()
{
    last_command="rclone_move"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--create-empty-src-dirs")
    local_nonpersistent_flags+=("--create-empty-src-dirs")
    flags+=("--delete-empty-src-dirs")
    local_nonpersistent_flags+=("--delete-empty-src-dirs")
    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_moveto()
{
    last_command="rclone_moveto"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_ncdu()
{
    last_command="rclone_ncdu"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_obscure()
{
    last_command="rclone_obscure"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_purge()
{
    last_command="rclone_purge"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_rc()
{
    last_command="rclone_rc"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--json=")
    two_word_flags+=("--json")
    local_nonpersistent_flags+=("--json=")
    flags+=("--loopback")
    local_nonpersistent_flags+=("--loopback")
    flags+=("--no-output")
    local_nonpersistent_flags+=("--no-output")
    flags+=("--pass=")
    two_word_flags+=("--pass")
    local_nonpersistent_flags+=("--pass=")
    flags+=("--url=")
    two_word_flags+=("--url")
    local_nonpersistent_flags+=("--url=")
    flags+=("--user=")
    two_word_flags+=("--user")
    local_nonpersistent_flags+=("--user=")
    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_rcat()
{
    last_command="rclone_rcat"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_rcd()
{
    last_command="rclone_rcd"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    flags+=("--qingstor-env-auth")
    flags+=("--qingstor-secret-access-key=")
    two_word_flags+=("--qingstor-secret-access-key")
    flags+=("--qingstor-upload-concurrency=")
    two_word_flags+=("--qingstor-upload-concurrency")
    flags+=("--qingstor-upload-cutoff=")
    two_word_flags+=("--qingstor-upload-cutoff")
    flags+=("--qingstor-zone=")
    two_word_flags+=("--qingstor-zone")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    two_word_flags+=("--rc-addr")
    flags+=("--rc-cert=")
    two_word_flags+=("--rc-cert")
    flags+=("--rc-client-ca=")
    two_word_flags+=("--rc-client-ca")
    flags+=("--rc-files=")
    two_word_flags+=("--rc-files")
    flags+=("--rc-htpasswd=")
    two_word_flags+=("--rc-htpasswd")
    flags+=("--rc-job-expire-duration=")
    two_word_flags+=("--rc-job-expire-duration")
    flags+=("--rc-job-expire-interval=")
    two_word_flags+=("--rc-job-expire-interval")
    flags+=("--rc-key=")
    two_word_flags+=("--rc-key")
    flags+=("--rc-max-header-bytes=")
    two_word_flags+=("--rc-max-header-bytes")
    flags+=("--rc-no-auth")
    flags+=("--rc-pass=")
    two_word_flags+=("--rc-pass")
    flags+=("--rc-realm=")
    two_word_flags+=("--rc-realm")
    flags+=("--rc-serve")
    flags+=("--rc-server-read-timeout=")
    two_word_flags+=("--rc-server-read-timeout")
    flags+=("--rc-server-write-timeout=")
    two_word_flags+=("--rc-server-write-timeout")
    flags+=("--rc-user=")
    two_word_flags+=("--rc-user")
    flags+=("--retries=")
    two_word_flags+=("--retries")
    flags+=("--retries-sleep=")
    two_word_flags+=("--retries-sleep")
    flags+=("--s3-access-key-id=")
    two_word_flags+=("--s3-access-key-id")
    flags+=("--s3-acl=")
    two_word_flags+=("--s3-acl")
    flags+=("--s3-bucket-acl=")
    two_word_flags+=("--s3-bucket-acl")
    flags+=("--s3-chunk-size=")
    two_word_flags+=("--s3-chunk-size")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-endpoint=")
    two_word_flags+=("--s3-endpoint")
    flags+=("--s3-env-auth")
    flags+=("--s3-force-path-style")
    flags+=("--s3-location-constraint=")
    two_word_flags+=("--s3-location-constraint")
    flags+=("--s3-provider=")
    two_word_flags+=("--s3-provider")
    flags+=("--s3-region=")
    two_word_flags+=("--s3-region")
    flags+=("--s3-secret-access-key=")
    two_word_flags+=("--s3-secret-access-key")
    flags+=("--s3-server-side-encryption=")
    two_word_flags+=("--s3-server-side-encryption")
    flags+=("--s3-session-token=")
    two_word_flags+=("--s3-session-token")
    flags+=("--s3-sse-kms-key-id=")
    two_word_flags+=("--s3-sse-kms-key-id")
    flags+=("--s3-storage-class=")
    two_word_flags+=("--s3-storage-class")
    flags+=("--s3-upload-concurrency=")
    two_word_flags+=("--s3-upload-concurrency")
    flags+=("--s3-upload-cutoff=")
    two_word_flags+=("--s3-upload-cutoff")
    flags+=("--s3-use-accelerate-endpoint")
    flags+=("--s3-v2-auth")
    flags+=("--sftp-ask-password")
    flags+=("--sftp-disable-hashcheck")
    flags+=("--sftp-host=")
    two_word_flags+=("--sftp-host")
    flags+=("--sftp-key-file=")
    two_word_flags+=("--sftp-key-file")
    flags+=("--sftp-key-file-pass=")
    two_word_flags+=("--sftp-key-file-pass")
    flags+=("--sftp-key-use-agent")
    flags+=("--sftp-pass=")
    two_word_flags+=("--sftp-pass")
    flags+=("--sftp-path-override=")
    two_word_flags+=("--sftp-path-override")
    flags+=("--sftp-port=")
    two_word_flags+=("--sftp-port")
    flags+=("--sftp-set-modtime")
    flags+=("--sftp-use-insecure-cipher")
    flags+=("--sftp-user=")
    two_word_flags+=("--sftp-user")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--stats=")
    two_word_flags+=("--stats")
    flags+=("--stats-file-name-length=")
    two_word_flags+=("--stats-file-name-length")
    flags+=("--stats-log-level=")
    two_word_flags+=("--stats-log-level")
    flags+=("--stats-one-line")
    flags+=("--stats-one-line-date")
    flags+=("--stats-one-line-date-format=")
    two_word_flags+=("--stats-one-line-date-format")
    flags+=("--stats-unit=")
    two_word_flags+=("--stats-unit")
    flags+=("--streaming-upload-cutoff=")
    two_word_flags+=("--streaming-upload-cutoff")
    flags+=("--suffix=")
    two_word_flags+=("--suffix")
    flags+=("--suffix-keep-extension")
    flags+=("--swift-application-credential-id=")
    two_word_flags+=("--swift-application-credential-id")
    flags+=("--swift-application-credential-name=")
    two_word_flags+=("--swift-application-credential-name")
    flags+=("--swift-application-credential-secret=")
    two_word_flags+=("--swift-application-credential-secret")
    flags+=("--swift-auth=")
    two_word_flags+=("--swift-auth")
    flags+=("--swift-auth-token=")
    two_word_flags+=("--swift-auth-token")
    flags+=("--swift-auth-version=")
    two_word_flags+=("--swift-auth-version")
    flags+=("--swift-chunk-size=")
    two_word_flags+=("--swift-chunk-size")
    flags+=("--swift-domain=")
    two_word_flags+=("--swift-domain")
    flags+=("--swift-endpoint-type=")
    two_word_flags+=("--swift-endpoint-type")
    flags+=("--swift-env-auth")
    flags+=("--swift-key=")
    two_word_flags+=("--swift-key")
    flags+=("--swift-no-chunk")
    flags+=("--swift-region=")
    two_word_flags+=("--swift-region")
    flags+=("--swift-storage-policy=")
    two_word_flags+=("--swift-storage-policy")
    flags+=("--swift-storage-url=")
    two_word_flags+=("--swift-storage-url")
    flags+=("--swift-tenant=")
    two_word_flags+=("--swift-tenant")
    flags+=("--swift-tenant-domain=")
    two_word_flags+=("--swift-tenant-domain")
    flags+=("--swift-tenant-id=")
    two_word_flags+=("--swift-tenant-id")
    flags+=("--swift-user=")
    two_word_flags+=("--swift-user")
    flags+=("--swift-user-id=")
    two_word_flags+=("--swift-user-id")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    two_word_flags+=("--syslog-facility")
    flags+=("--timeout=")
    two_word_flags+=("--timeout")
    flags+=("--tpslimit=")
    two_word_flags+=("--tpslimit")
    flags+=("--tpslimit-burst=")
    two_word_flags+=("--tpslimit-burst")
    flags+=("--track-renames")
    flags+=("--transfers=")
    two_word_flags+=("--transfers")
    flags+=("--union-remotes=")
    two_word_flags+=("--union-remotes")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-cookies")
    flags+=("--use-mmap")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    two_word_flags+=("--user-agent")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--webdav-bearer-token=")
    two_word_flags+=("--webdav-bearer-token")
    flags+=("--webdav-pass=")
    two_word_flags+=("--webdav-pass")
    flags+=("--webdav-url=")
    two_word_flags+=("--webdav-url")
    flags+=("--webdav-user=")
    two_word_flags+=("--webdav-user")
    flags+=("--webdav-vendor=")
    two_word_flags+=("--webdav-vendor")
    flags+=("--yandex-client-id=")
    two_word_flags+=("--yandex-client-id")
    flags+=("--yandex-client-secret=")
    two_word_flags+=("--yandex-client-secret")
    flags+=("--yandex-unlink")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_rmdir()
{
    last_command="rclone_rmdir"

    command_aliases=()

    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-auth-url=")
    two_word_flags+=("--acd-auth-url")
    flags+=("--acd-client-id=")
    two_word_flags+=("--acd-client-id")
    flags+=("--acd-client-secret=")
    two_word_flags+=("--acd-client-secret")
    flags+=("--acd-templink-threshold=")
    two_word_flags+=("--acd-templink-threshold")
    flags+=("--acd-token-url=")
    two_word_flags+=("--acd-token-url")
    flags+=("--acd-upload-wait-per-gb=")
    two_word_flags+=("--acd-upload-wait-per-gb")
    flags+=("--alias-remote=")
    two_word_flags+=("--alias-remote")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-access-tier=")
    two_word_flags+=("--azureblob-access-tier")
    flags+=("--azureblob-account=")
    two_word_flags+=("--azureblob-account")
    flags+=("--azureblob-chunk-size=")
    two_word_flags+=("--azureblob-chunk-size")
    flags+=("--azureblob-endpoint=")
    two_word_flags+=("--azureblob-endpoint")
    flags+=("--azureblob-key=")
    two_word_flags+=("--azureblob-key")
    flags+=("--azureblob-list-chunk=")
    two_word_flags+=("--azureblob-list-chunk")
    flags+=("--azureblob-sas-url=")
    two_word_flags+=("--azureblob-sas-url")
    flags+=("--azureblob-upload-cutoff=")
    two_word_flags+=("--azureblob-upload-cutoff")
    flags+=("--b2-account=")
    two_word_flags+=("--b2-account")
    flags+=("--b2-chunk-size=")
    two_word_flags+=("--b2-chunk-size")
    flags+=("--b2-disable-checksum")
    flags+=("--b2-download-url=")
    two_word_flags+=("--b2-download-url")
    flags+=("--b2-endpoint=")
    two_word_flags+=("--b2-endpoint")
    flags+=("--b2-hard-delete")
    flags+=("--b2-key=")
    two_word_flags+=("--b2-key")
    flags+=("--b2-test-mode=")
    two_word_flags+=("--b2-test-mode")
    flags+=("--b2-upload-cutoff=")
    two_word_flags+=("--b2-upload-cutoff")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    two_word_flags+=("--backup-dir")
    flags+=("--bind=")
    two_word_flags+=("--bind")
    flags+=("--box-client-id=")
    two_word_flags+=("--box-client-id")
    flags+=("--box-client-secret=")
    two_word_flags+=("--box-client-secret")
    flags+=("--box-commit-retries=")
    two_word_flags+=("--box-commit-retries")
    flags+=("--box-upload-cutoff=")
    two_word_flags+=("--box-upload-cutoff")
    flags+=("--buffer-size=")
    two_word_flags+=("--buffer-size")
    flags+=("--bwlimit=")
    two_word_flags+=("--bwlimit")
    flags+=("--ca-cert=")
    two_word_flags+=("--ca-cert")
    flags+=("--cache-chunk-clean-interval=")
    two_word_flags+=("--cache-chunk-clean-interval")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    two_word_flags+=("--cache-chunk-path")
    flags+=("--cache-chunk-size=")
    two_word_flags+=("--cache-chunk-size")
    flags+=("--cache-chunk-total-size=")
    two_word_flags+=("--cache-chunk-total-size")
    flags+=("--cache-db-path=")
    two_word_flags+=("--cache-db-path")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    two_word_flags+=("--cache-db-wait-time")
    flags+=("--cache-dir=")
    two_word_flags+=("--cache-dir")
    flags+=("--cache-info-age=")
    two_word_flags+=("--cache-info-age")
    flags+=("--cache-plex-insecure=")
    two_word_flags+=("--cache-plex-insecure")
    flags+=("--cache-plex-password=")
    two_word_flags+=("--cache-plex-password")
    flags+=("--cache-plex-url=")
    two_word_flags+=("--cache-plex-url")
    flags+=("--cache-plex-username=")
    two_word_flags+=("--cache-plex-username")
    flags+=("--cache-read-retries=")
    two_word_flags+=("--cache-read-retries")
    flags+=("--cache-remote=")
    two_word_flags+=("--cache-remote")
    flags+=("--cache-rps=")
    two_word_flags+=("--cache-rps")
    flags+=("--cache-tmp-upload-path=")
    two_word_flags+=("--cache-tmp-upload-path")
    flags+=("--cache-tmp-wait-time=")
    two_word_flags+=("--cache-tmp-wait-time")
    flags+=("--cache-workers=")
    two_word_flags+=("--cache-workers")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    two_word_flags+=("--checkers")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--client-cert=")
    two_word_flags+=("--client-cert")
    flags+=("--client-key=")
    two_word_flags+=("--client-key")
    flags+=("--config=")
    two_word_flags+=("--config")
    flags+=("--contimeout=")
    two_word_flags+=("--contimeout")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    two_word_flags+=("--cpuprofile")
    flags+=("--crypt-directory-name-encryption")
    flags+=("--crypt-filename-encryption=")
    two_word_flags+=("--crypt-filename-encryption")
    flags+=("--crypt-password=")
    two_word_flags+=("--crypt-password")
    flags+=("--crypt-password2=")
    two_word_flags+=("--crypt-password2")
    flags+=("--crypt-remote=")
    two_word_flags+=("--crypt-remote")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    two_word_flags+=("--disable")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-allow-import-name-change")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    two_word_flags+=("--drive-chunk-size")
    flags+=("--drive-client-id=")
    two_word_flags+=("--drive-client-id")
    flags+=("--drive-client-secret=")
    two_word_flags+=("--drive-client-secret")
    flags+=("--drive-export-formats=")
    two_word_flags+=("--drive-export-formats")
    flags+=("--drive-formats=")
    two_word_flags+=("--drive-formats")
    flags+=("--drive-impersonate=")
    two_word_flags+=("--drive-impersonate")
    flags+=("--drive-import-formats=")
    two_word_flags+=("--drive-import-formats")
    flags+=("--drive-keep-revision-forever")
    flags+=("--drive-list-chunk=")
    two_word_flags+=("--drive-list-chunk")
    flags+=("--drive-pacer-burst=")
    two_word_flags+=("--drive-pacer-burst")
    flags+=("--drive-pacer-min-sleep=")
    two_word_flags+=("--drive-pacer-min-sleep")
    flags+=("--drive-root-folder-id=")
    two_word_flags+=("--drive-root-folder-id")
    flags+=("--drive-scope=")
    two_word_flags+=("--drive-scope")
    flags+=("--drive-server-side-across-configs")
    flags+=("--drive-service-account-credentials=")
    two_word_flags+=("--drive-service-account-credentials")
    flags+=("--drive-service-account-file=")
    two_word_flags+=("--drive-service-account-file")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-size-as-quota")
    flags+=("--drive-skip-checksum-gphotos")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-team-drive=")
    two_word_flags+=("--drive-team-drive")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    two_word_flags+=("--drive-upload-cutoff")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--drive-v2-download-min-size=")
    two_word_flags+=("--drive-v2-download-min-size")
    flags+=("--dropbox-chunk-size=")
    two_word_flags+=("--dropbox-chunk-size")
    flags+=("--dropbox-client-id=")
    two_word_flags+=("--dropbox-client-id")
    flags+=("--dropbox-client-secret=")
    two_word_flags+=("--dropbox-client-secret")
    flags+=("--dropbox-impersonate=")
    two_word_flags+=("--dropbox-impersonate")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    two_word_flags+=("--dump")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    two_word_flags+=("--exclude")
    flags+=("--exclude-from=")
    two_word_flags+=("--exclude-from")
    flags+=("--exclude-if-present=")
    two_word_flags+=("--exclude-if-present")
    flags+=("--fast-list")
    flags+=("--files-from=")
    two_word_flags+=("--files-from")
    flags+=("--filter=")
    two_word_flags+=("--filter")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    two_word_flags+=("--filter-from")
    flags+=("--ftp-concurrency=")
    two_word_flags+=("--ftp-concurrency")
    flags+=("--ftp-host=")
    two_word_flags+=("--ftp-host")
    flags+=("--ftp-no-check-certificate")
    flags+=("--ftp-pass=")
    two_word_flags+=("--ftp-pass")
    flags+=("--ftp-port=")
    two_word_flags+=("--ftp-port")
    flags+=("--ftp-tls")
    flags+=("--ftp-user=")
    two_word_flags+=("--ftp-user")
    flags+=("--gcs-bucket-acl=")
    two_word_flags+=("--gcs-bucket-acl")
    flags+=("--gcs-bucket-policy-only")
    flags+=("--gcs-client-id=")
    two_word_flags+=("--gcs-client-id")
    flags+=("--gcs-client-secret=")
    two_word_flags+=("--gcs-client-secret")
    flags+=("--gcs-location=")
    two_word_flags+=("--gcs-location")
    flags+=("--gcs-object-acl=")
    two_word_flags+=("--gcs-object-acl")
    flags+=("--gcs-project-number=")
    two_word_flags+=("--gcs-project-number")
    flags+=("--gcs-service-account-file=")
    two_word_flags+=("--gcs-service-account-file")
    flags+=("--gcs-storage-class=")
    two_word_flags+=("--gcs-storage-class")
    flags+=("--http-no-slash")
    flags+=("--http-url=")
    two_word_flags+=("--http-url")
    flags+=("--hubic-chunk-size=")
    two_word_flags+=("--hubic-chunk-size")
    flags+=("--hubic-client-id=")
    two_word_flags+=("--hubic-client-id")
    flags+=("--hubic-client-secret=")
    two_word_flags+=("--hubic-client-secret")
    flags+=("--hubic-no-chunk")
    flags+=("--ignore-case")
    flags+=("--ignore-case-sync")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    two_word_flags+=("--include")
    flags+=("--include-from=")
    two_word_flags+=("--include-from")
    flags+=("--jottacloud-hard-delete")
    flags+=("--jottacloud-md5-memory-limit=")
    two_word_flags+=("--jottacloud-md5-memory-limit")
    flags+=("--jottacloud-unlink")
    flags+=("--jottacloud-upload-resume-limit=")
    two_word_flags+=("--jottacloud-upload-resume-limit")
    flags+=("--jottacloud-user=")
    two_word_flags+=("--jottacloud-user")
    flags+=("--koofr-endpoint=")
    two_word_flags+=("--koofr-endpoint")
    flags+=("--koofr-mountid=")
    two_word_flags+=("--koofr-mountid")
    flags+=("--koofr-password=")
    two_word_flags+=("--koofr-password")
    flags+=("--koofr-user=")
    two_word_flags+=("--koofr-user")
    flags+=("--links")
    flags+=("-l")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--local-nounc=")
    two_word_flags+=("--local-nounc")
    flags+=("--log-file=")
    two_word_flags+=("--log-file")
    flags+=("--log-format=")
    two_word_flags+=("--log-format")
    flags+=("--log-level=")
    two_word_flags+=("--log-level")
    flags+=("--low-level-retries=")
    two_word_flags+=("--low-level-retries")
    flags+=("--max-age=")
    two_word_flags+=("--max-age")
    flags+=("--max-backlog=")
    two_word_flags+=("--max-backlog")
    flags+=("--max-delete=")
    two_word_flags+=("--max-delete")
    flags+=("--max-depth=")
    two_word_flags+=("--max-depth")
    flags+=("--max-size=")
    two_word_flags+=("--max-size")
    flags+=("--max-transfer=")
    two_word_flags+=("--max-transfer")
    flags+=("--mega-debug")
    flags+=("--mega-hard-delete")
    flags+=("--mega-pass=")
    two_word_flags+=("--mega-pass")
    flags+=("--mega-user=")
    two_word_flags+=("--mega-user")
    flags+=("--memprofile=")
    two_word_flags+=("--memprofile")
    flags+=("--min-age=")
    two_word_flags+=("--min-age")
    flags+=("--min-size=")
    two_word_flags+=("--min-size")
    flags+=("--modify-window=")
    two_word_flags+=("--modify-window")
    flags+=("--multi-thread-cutoff=")
    two_word_flags+=("--multi-thread-cutoff")
    flags+=("--multi-thread-streams=")
    two_word_flags+=("--multi-thread-streams")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    two_word_flags+=("--onedrive-chunk-size")
    flags+=("--onedrive-client-id=")
    two_word_flags+=("--onedrive-client-id")
    flags+=("--onedrive-client-secret=")
    two_word_flags+=("--onedrive-client-secret")
    flags+=("--onedrive-drive-id=")
    two_word_flags+=("--onedrive-drive-id")
    flags+=("--onedrive-drive-type=")
    two_word_flags+=("--onedrive-drive-type")
    flags+=("--onedrive-expose-onenote-files")
    flags+=("--opendrive-password=")
    two_word_flags+=("--opendrive-password")
    flags+=("--opendrive-username=")
    two_word_flags+=("--opendrive-username")
    flags+=("--pcloud-client-id=")
    two_word_flags+=("--pcloud-client-id")
    flags+=("--pcloud-client-secret=")
    two_word_flags+=("--pcloud-client-secret")
    flags+=("--progress")
    flags+=("-P")
    flags+=("--qingstor-access-key-id=")
    two_word_flags+=("--qingstor-access-key-id")
    flags+=("--qingstor-chunk-size=")
    two_word_flags+=("--qingstor-chunk-size")
    flags+=("--qingstor-connection-retries=")
    two_word_flags+=("--qingstor-connection-retries")
    flags+=("--qingstor-endpoint=")
    two_word_flags+=("--qingstor-endpoint")
    f