'use strict';

exports.__esModule = true;

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

var DEFAULT_RAW = {
  colon: ': ',
  indent: '    ',
  beforeDecl: '\n',
  beforeRule: '\n',
  beforeOpen: ' ',
  beforeClose: '\n',
  beforeComment: '\n',
  after: '\n',
  emptyBody: '',
  commentLeft: ' ',
  commentRight: ' '
};

function capitalize(str) {
  return str[0].toUpperCase() + str.slice(1);
}

var Stringifier = function () {
  function Stringifier(builder) {
    _classCallCheck(this, Stringifier);

    this.builder = builder;
  }

  Stringifier.prototype.stringify = function stringify(node, semicolon) {
    this[node.type](node, semicolon);
  };

  Stringifier.prototype.root = function root(node) {
    this.body(node);
    if (node.raws.after) this.builder(node.raws.after);
  };

  Stringifier.prototype.comment = function comment(node) {
    var left = this.raw(node, 'left', 'commentLeft');
    var right = this.raw(node, 'right', 'commentRight');
    this.builder('/*' + left + node.text + right + '*/', node);
  };

  Stringifier.prototype.decl = function decl(node, semicolon) {
    var between = this.raw(node, 'between', 'colon');
    var string = node.prop + between + this.rawValue(node, 'value');

    if (node.important) {
      string += node.raws.important || ' !important';
    }

    if (semicolon) string += ';';
    this.builder(string, node);
  };

  Stringifier.prototype.rule = function rule(node) {
    this.block(node, this.rawValue(node, 'selector'));
    if (node.raws.ownSemicolon) {
      this.builder(node.raws.ownSemicolon, node, 'end');
    }
  };

  Stringifier.prototype.atrule = function atrule(node, semicolon) {
    var name = '@' + node.name;
    var params = node.params ? this.rawValue(node, 'params') : '';

    if (typeof node.raws.afterName !== 'undefined') {
      name += node.raws.afterName;
    } else if (params) {
      name += ' ';
    }

    if (node.nodes) {
      this.block(node, name + params);
    } else {
      var end = (node.raws.between || '') + (semicolon ? ';' : '');
      this.builder(name + params + end, node);
    }
  };

  Stringifier.prototype.body = function body(node) {
    var last = node.nodes.length - 1;
    while (last > 0) {
      if (node.nodes[last].type !== 'comment') break;
      last -= 1;
    }

    var semicolon = this.raw(node, 'semicolon');
    for (var i = 0; i < node.nodes.length; i++) {
      var child = node.nodes[i];
      var before = this.raw(child, 'before');
      if (before) this.builder(before);
      this.stringify(child, last !== i || semicolon);
    }
  };

  Stringifier.prototype.block = function block(node, start) {
    var between = this.raw(node, 'between', 'beforeOpen');
    this.builder(start + between + '{', node, 'start');

    var after = void 0;
    if (node.nodes && node.nodes.length) {
      this.body(node);
      after = this.raw(node, 'after');
    } else {
      after = this.raw(node, 'after', 'emptyBody');
    }

    if (after) this.builder(after);
    this.builder('}', node, 'end');
  };

  Stringifier.prototype.raw = function raw(node, own, detect) {
    var value = void 0;
    if (!detect) detect = own;

    // Already had
    if (own) {
      value = node.raws[own];
      if (typeof value !== 'undefined') return value;
    }

    var parent = node.parent;

    // Hack for first rule in CSS
    if (detect === 'before') {
      if (!parent || parent.type === 'root' && parent.first === node) {
        return '';
      }
    }

    // Floating child without parent
    if (!parent) return DEFAULT_RAW[detect];

    // Detect style by other nodes
    var root = node.root();
    if (!root.rawCache) root.rawCache = {};
    if (typeof root.rawCache[detect] !== 'undefined') {
      return root.rawCache[detect];
    }

    if (detect === 'before' || detect === 'after') {
      return this.beforeAfter(node, detect);
    } else {
      var method = 'raw' + capitalize(detect);
      if (this[method]) {
        value = this[method](root, node);
      } else {
        root.walk(function (i) {
          value = i.raws[own];
          if (typeof value !== 'undefined') return false;
        });
      }
    }

    if (typeof value === 'undefined') value = DEFAULT_RAW[detect];

    root.rawCache[detect] = value;
    return value;
  };

  Stringifier.prototype.rawSemicolon = function rawSemicolon(root) {
    var value = void 0;
    root.walk(function (i) {
      if (i.nodes && i.nodes.length && i.last.type === 'decl') {
        value = i.raws.semicolon;
        if (typeof value !== 'undefined') return false;
      }
    });
    return value;
  };

  Stringifier.prototype.rawEmptyBody = function rawEmptyBody(root) {
    var value = void 0;
    root.walk(function (i) {
      if (i.nodes && i.nodes.length === 0) {
        value = i.raws.after;
        if (typeof value !== 'undefined') return false;
      }
    });
    return value;
  };

  Stringifier.prototype.rawIndent = function rawIndent(root) {
    if (root.raws.indent) return root.raws.indent;
    var value = void 0;
    root.walk(function (i) {
      var p = i.parent;
      if (p && p !== root && p.parent && p.parent === root) {
        if (typeof i.raws.before !== 'undefined') {
          var parts = i.raws.before.split('\n');
          value = parts[parts.length - 1];
          value = value.replace(/[^\s]/g, '');
          return false;
        }
      }
    });
    return value;
  };

  Stringifier.prototype.rawBeforeComment = function rawBeforeComment(root, node) {
    var value = void 0;
    root.walkComments(function (i) {
      if (typeof i.raws.before !== 'undefined') {
        value = i.raws.before;
        if (value.indexOf('\n') !== -1) {
          value = value.replace(/[^\n]+$/, '');
        }
        return false;
      }
    });
    if (typeof value === 'undefined') {
      value = this.raw(node, null, 'beforeDecl');
    } else if (value) {
      value = value.replace(/[^\s]/g, '');
    }
    return value;
  };

  Stringifier.prototype.rawBeforeDecl = function rawBeforeDecl(root, node) {
    var value = void 0;
    root.walkDecls(function (i) {
      if (typeof i.raws.before !== 'undefined') {
        value = i.raws.before;
        if (value.indexOf('\n') !== -1) {
          value = value.replace(/[^\n]+$/, '');
        }
        return false;
      }
    });
    if (typeof value === 'undefined') {
      value = this.raw(node, null, 'beforeRule');
    } else if (value) {
      value = value.replace(/[^\s]/g, '');
    }
    return value;
  };

  Stringifier.prototype.rawBeforeRule = function rawBeforeRule(root) {
    var value = void 0;
    root.walk(function (i) {
      if (i.nodes && (i.parent !== root || root.first !== i)) {
        if (typeof i.raws.before !== 'undefined') {
          value = i.raws.before;
          if (value.indexOf('\n') !== -1) {
            value = value.replace(/[^\n]+$/, '');
          }
          return false;
        }
      }
    });
    if (value) value = value.replace(/[^\s]/g, '');
    return value;
  };

  Stringifier.prototype.rawBeforeClose = function rawBeforeClose(root) {
    var value = void 0;
    root.walk(function (i) {
      if (i.nodes && i.nodes.length > 0) {
        if (typeof i.raws.after !== 'undefined') {
          value = i.raws.after;
          if (value.indexOf('\n') !== -1) {
            value = value.replace(/[^\n]+$/, '');
          }
          return false;
        }
      }
    });
    if (value) value = value.replace(/[^\s]/g, '');
    return value;
  };

  Stringifier.prototype.rawBeforeOpen = function rawBeforeOpen(root) {
    var value = void 0;
    root.walk(function (i) {
      if (i.type !== 'decl') {
        value = i.raws.between;
        if (typeof value !== 'undefined') return false;
      }
    });
    return value;
  };

  Stringifier.prototype.rawColon = function rawColon(root) {
    var value = void 0;
    root.walkDecls(function (i) {
      if (typeof i.raws.between !== 'undefined') {
        value = i.raws.between.replace(/[^\s:]/g, '');
        return false;
      }
    });
    return value;
  };

  Stringifier.prototype.beforeAfter = function beforeAfter(node, detect) {
    var value = void 0;
    if (node.type === 'decl') {
      value = this.raw(node, null, 'beforeDecl');
    } else if (node.type === 'comment') {
      value = this.raw(node, null, 'beforeComment');
    } else if (detect === 'before') {
      value = this.raw(node, null, 'beforeRule');
    } else {
      value = this.raw(node, null, 'beforeClose');
    }

    var buf = node.parent;
    var depth = 0;
    while (buf && buf.type !== 'root') {
      depth += 1;
      buf = buf.parent;
    }

    if (value.indexOf('\n') !== -1) {
      var indent = this.raw(node, null, 'indent');
      if (indent.length) {
        for (var step = 0; step < depth; step++) {
          value += indent;
        }
      }
    }

    return value;
  };

  Stringifier.prototype.rawValue = function rawValue(node, prop) {
    var value = node[prop];
    var raw = node.raws[prop];
    if (raw && raw.value === value) {
      return raw.raw;
    } else {
      return value;
    }
  };

  return Stringifier;
}();

exports.default = Stringifier;
module.exports = exports['default'];
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
