# bash completion for rclone                               -*- shell-script -*-

__rclone_debug()
{
    if [[ -n ${BASH_COMP_DEBUG_FILE} ]]; then
        echo "$*" >> "${BASH_COMP_DEBUG_FILE}"
    fi
}

# Homebrew on Macs have version 1.3 of bash-completion which doesn't include
# _init_completion. This is a very minimal version of that function.
__rclone_init_completion()
{
    COMPREPLY=()
    _get_comp_words_by_ref "$@" cur prev words cword
}

__rclone_index_of_word()
{
    local w word=$1
    shift
    index=0
    for w in "$@"; do
        [[ $w = "$word" ]] && return
        index=$((index+1))
    done
    index=-1
}

__rclone_contains_word()
{
    local w word=$1; shift
    for w in "$@"; do
        [[ $w = "$word" ]] && return
    done
    return 1
}

__rclone_handle_reply()
{
    __rclone_debug "${FUNCNAME[0]}"
    case $cur in
        -*)
            if [[ $(type -t compopt) = "builtin" ]]; then
                compopt -o nospace
            fi
            local allflags
            if [ ${#must_have_one_flag[@]} -ne 0 ]; then
                allflags=("${must_have_one_flag[@]}")
            else
                allflags=("${flags[*]} ${two_word_flags[*]}")
            fi
            COMPREPLY=( $(compgen -W "${allflags[*]}" -- "$cur") )
            if [[ $(type -t compopt) = "builtin" ]]; then
                [[ "${COMPREPLY[0]}" == *= ]] || compopt +o nospace
            fi

            # complete after --flag=abc
            if [[ $cur == *=* ]]; then
                if [[ $(type -t compopt) = "builtin" ]]; then
                    compopt +o nospace
                fi

                local index flag
                flag="${cur%=*}"
                __rclone_index_of_word "${flag}" "${flags_with_completion[@]}"
                COMPREPLY=()
                if [[ ${index} -ge 0 ]]; then
                    PREFIX=""
                    cur="${cur#*=}"
                    ${flags_completion[${index}]}
                    if [ -n "${ZSH_VERSION}" ]; then
                        # zsh completion needs --flag= prefix
                        eval "COMPREPLY=( \"\${COMPREPLY[@]/#/${flag}=}\" )"
                    fi
                fi
            fi
            return 0;
            ;;
    esac

    # check if we are handling a flag with special work handling
    local index
    __rclone_index_of_word "${prev}" "${flags_with_completion[@]}"
    if [[ ${index} -ge 0 ]]; then
        ${flags_completion[${index}]}
        return
    fi

    # we are parsing a flag and don't have a special handler, no completion
    if [[ ${cur} != "${words[cword]}" ]]; then
        return
    fi

    local completions
    completions=("${commands[@]}")
    if [[ ${#must_have_one_noun[@]} -ne 0 ]]; then
        completions=("${must_have_one_noun[@]}")
    fi
    if [[ ${#must_have_one_flag[@]} -ne 0 ]]; then
        completions+=("${must_have_one_flag[@]}")
    fi
    COMPREPLY=( $(compgen -W "${completions[*]}" -- "$cur") )

    if [[ ${#COMPREPLY[@]} -eq 0 && ${#noun_aliases[@]} -gt 0 && ${#must_have_one_noun[@]} -ne 0 ]]; then
        COMPREPLY=( $(compgen -W "${noun_aliases[*]}" -- "$cur") )
    fi

    if [[ ${#COMPREPLY[@]} -eq 0 ]]; then
        declare -F __custom_func >/dev/null && __custom_func
    fi

    # available in bash-completion >= 2, not always present on macOS
    if declare -F __ltrim_colon_completions >/dev/null; then
        __ltrim_colon_completions "$cur"
    fi

    # If there is only 1 completion and it is a flag with an = it will be completed
    # but we don't want a space after the =
    if [[ "${#COMPREPLY[@]}" -eq "1" ]] && [[ $(type -t compopt) = "builtin" ]] && [[ "${COMPREPLY[0]}" == --*= ]]; then
       compopt -o nospace
    fi
}

# The arguments should be in the form "ext1|ext2|extn"
__rclone_handle_filename_extension_flag()
{
    local ext="$1"
    _filedir "@(${ext})"
}

__rclone_handle_subdirs_in_dir_flag()
{
    local dir="$1"
    pushd "${dir}" >/dev/null 2>&1 && _filedir -d && popd >/dev/null 2>&1
}

__rclone_handle_flag()
{
    __rclone_debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"

    # if a command required a flag, and we found it, unset must_have_one_flag()
    local flagname=${words[c]}
    local flagvalue
    # if the word contained an =
    if [[ ${words[c]} == *"="* ]]; then
        flagvalue=${flagname#*=} # take in as flagvalue after the =
        flagname=${flagname%=*} # strip everything after the =
        flagname="${flagname}=" # but put the = back
    fi
    __rclone_debug "${FUNCNAME[0]}: looking for ${flagname}"
    if __rclone_contains_word "${flagname}" "${must_have_one_flag[@]}"; then
        must_have_one_flag=()
    fi

    # if you set a flag which only applies to this command, don't show subcommands
    if __rclone_contains_word "${flagname}" "${local_nonpersistent_flags[@]}"; then
      commands=()
    fi

    # keep flag value with flagname as flaghash
    # flaghash variable is an associative array which is only supported in bash > 3.
    if [[ -z "${BASH_VERSION}" || "${BASH_VERSINFO[0]}" -gt 3 ]]; then
        if [ -n "${flagvalue}" ] ; then
            flaghash[${flagname}]=${flagvalue}
        elif [ -n "${words[ $((c+1)) ]}" ] ; then
            flaghash[${flagname}]=${words[ $((c+1)) ]}
        else
            flaghash[${flagname}]="true" # pad "true" for bool flag
        fi
    fi

    # skip the argument to a two word flag
    if __rclone_contains_word "${words[c]}" "${two_word_flags[@]}"; then
        c=$((c+1))
        # if we are looking for a flags value, don't show commands
        if [[ $c -eq $cword ]]; then
            commands=()
        fi
    fi

    c=$((c+1))

}

__rclone_handle_noun()
{
    __rclone_debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"

    if __rclone_contains_word "${words[c]}" "${must_have_one_noun[@]}"; then
        must_have_one_noun=()
    elif __rclone_contains_word "${words[c]}" "${noun_aliases[@]}"; then
        must_have_one_noun=()
    fi

    nouns+=("${words[c]}")
    c=$((c+1))
}

__rclone_handle_command()
{
    __rclone_debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"

    local next_command
    if [[ -n ${last_command} ]]; then
        next_command="_${last_command}_${words[c]//:/__}"
    else
        if [[ $c -eq 0 ]]; then
            next_command="_rclone_root_command"
        else
            next_command="_${words[c]//:/__}"
        fi
    fi
    c=$((c+1))
    __rclone_debug "${FUNCNAME[0]}: looking for ${next_command}"
    declare -F "$next_command" >/dev/null && $next_command
}

__rclone_handle_word()
{
    if [[ $c -ge $cword ]]; then
        __rclone_handle_reply
        return
    fi
    __rclone_debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"
    if [[ "${words[c]}" == -* ]]; then
        __rclone_handle_flag
    elif __rclone_contains_word "${words[c]}" "${commands[@]}"; then
        __rclone_handle_command
    elif [[ $c -eq 0 ]]; then
        __rclone_handle_command
    else
        __rclone_handle_noun
    fi
    __rclone_handle_word
}

_rclone_about()
{
    last_command="rclone_about"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--full")
    local_nonpersistent_flags+=("--full")
    flags+=("--json")
    local_nonpersistent_flags+=("--json")
    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_authorize()
{
    last_command="rclone_authorize"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_cachestats()
{
    last_command="rclone_cachestats"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_cat()
{
    last_command="rclone_cat"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--count=")
    local_nonpersistent_flags+=("--count=")
    flags+=("--discard")
    local_nonpersistent_flags+=("--discard")
    flags+=("--head=")
    local_nonpersistent_flags+=("--head=")
    flags+=("--offset=")
    local_nonpersistent_flags+=("--offset=")
    flags+=("--tail=")
    local_nonpersistent_flags+=("--tail=")
    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_check()
{
    last_command="rclone_check"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--download")
    local_nonpersistent_flags+=("--download")
    flags+=("--one-way")
    local_nonpersistent_flags+=("--one-way")
    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_cleanup()
{
    last_command="rclone_cleanup"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_config_create()
{
    last_command="rclone_config_create"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_config_delete()
{
    last_command="rclone_config_delete"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_config_dump()
{
    last_command="rclone_config_dump"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_config_edit()
{
    last_command="rclone_config_edit"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_config_file()
{
    last_command="rclone_config_file"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_config_password()
{
    last_command="rclone_config_password"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_config_providers()
{
    last_command="rclone_config_providers"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_config_show()
{
    last_command="rclone_config_show"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_config_update()
{
    last_command="rclone_config_update"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_config()
{
    last_command="rclone_config"
    commands=()
    commands+=("create")
    commands+=("delete")
    commands+=("dump")
    commands+=("edit")
    commands+=("file")
    commands+=("password")
    commands+=("providers")
    commands+=("show")
    commands+=("update")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_copy()
{
    last_command="rclone_copy"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_copyto()
{
    last_command="rclone_copyto"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_cryptcheck()
{
    last_command="rclone_cryptcheck"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--one-way")
    local_nonpersistent_flags+=("--one-way")
    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_cryptdecode()
{
    last_command="rclone_cryptdecode"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--reverse")
    local_nonpersistent_flags+=("--reverse")
    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_dbhashsum()
{
    last_command="rclone_dbhashsum"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_dedupe()
{
    last_command="rclone_dedupe"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--dedupe-mode=")
    local_nonpersistent_flags+=("--dedupe-mode=")
    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_delete()
{
    last_command="rclone_delete"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_deletefile()
{
    last_command="rclone_deletefile"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_genautocomplete_bash()
{
    last_command="rclone_genautocomplete_bash"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--help")
    flags+=("-h")
    local_nonpersistent_flags+=("--help")
    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_genautocomplete_zsh()
{
    last_command="rclone_genautocomplete_zsh"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_genautocomplete()
{
    last_command="rclone_genautocomplete"
    commands=()
    commands+=("bash")
    commands+=("zsh")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_gendocs()
{
    last_command="rclone_gendocs"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_hashsum()
{
    last_command="rclone_hashsum"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_link()
{
    last_command="rclone_link"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_listremotes()
{
    last_command="rclone_listremotes"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--long")
    flags+=("-l")
    local_nonpersistent_flags+=("--long")
    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_ls()
{
    last_command="rclone_ls"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_lsd()
{
    last_command="rclone_lsd"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_lsf()
{
    last_command="rclone_lsf"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--absolute")
    local_nonpersistent_flags+=("--absolute")
    flags+=("--csv")
    local_nonpersistent_flags+=("--csv")
    flags+=("--dir-slash")
    flags+=("-d")
    local_nonpersistent_flags+=("--dir-slash")
    flags+=("--dirs-only")
    local_nonpersistent_flags+=("--dirs-only")
    flags+=("--files-only")
    local_nonpersistent_flags+=("--files-only")
    flags+=("--format=")
    two_word_flags+=("-F")
    local_nonpersistent_flags+=("--format=")
    flags+=("--hash=")
    local_nonpersistent_flags+=("--hash=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--separator=")
    two_word_flags+=("-s")
    local_nonpersistent_flags+=("--separator=")
    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_lsjson()
{
    last_command="rclone_lsjson"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--encrypted")
    flags+=("-M")
    local_nonpersistent_flags+=("--encrypted")
    flags+=("--hash")
    local_nonpersistent_flags+=("--hash")
    flags+=("--no-modtime")
    local_nonpersistent_flags+=("--no-modtime")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_lsl()
{
    last_command="rclone_lsl"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_md5sum()
{
    last_command="rclone_md5sum"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_mkdir()
{
    last_command="rclone_mkdir"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_move()
{
    last_command="rclone_move"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--delete-empty-src-dirs")
    local_nonpersistent_flags+=("--delete-empty-src-dirs")
    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_moveto()
{
    last_command="rclone_moveto"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_ncdu()
{
    last_command="rclone_ncdu"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_obscure()
{
    last_command="rclone_obscure"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_purge()
{
    last_command="rclone_purge"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_rc()
{
    last_command="rclone_rc"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--no-output")
    local_nonpersistent_flags+=("--no-output")
    flags+=("--url=")
    local_nonpersistent_flags+=("--url=")
    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_rcat()
{
    last_command="rclone_rcat"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_rmdir()
{
    last_command="rclone_rmdir"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_rmdirs()
{
    last_command="rclone_rmdirs"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--leave-root")
    local_nonpersistent_flags+=("--leave-root")
    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_serve_http()
{
    last_command="rclone_serve_http"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--addr=")
    local_nonpersistent_flags+=("--addr=")
    flags+=("--cert=")
    local_nonpersistent_flags+=("--cert=")
    flags+=("--client-ca=")
    local_nonpersistent_flags+=("--client-ca=")
    flags+=("--dir-cache-time=")
    local_nonpersistent_flags+=("--dir-cache-time=")
    flags+=("--htpasswd=")
    local_nonpersistent_flags+=("--htpasswd=")
    flags+=("--key=")
    local_nonpersistent_flags+=("--key=")
    flags+=("--max-header-bytes=")
    local_nonpersistent_flags+=("--max-header-bytes=")
    flags+=("--no-checksum")
    local_nonpersistent_flags+=("--no-checksum")
    flags+=("--no-modtime")
    local_nonpersistent_flags+=("--no-modtime")
    flags+=("--no-seek")
    local_nonpersistent_flags+=("--no-seek")
    flags+=("--pass=")
    local_nonpersistent_flags+=("--pass=")
    flags+=("--poll-interval=")
    local_nonpersistent_flags+=("--poll-interval=")
    flags+=("--read-only")
    local_nonpersistent_flags+=("--read-only")
    flags+=("--realm=")
    local_nonpersistent_flags+=("--realm=")
    flags+=("--server-read-timeout=")
    local_nonpersistent_flags+=("--server-read-timeout=")
    flags+=("--server-write-timeout=")
    local_nonpersistent_flags+=("--server-write-timeout=")
    flags+=("--user=")
    local_nonpersistent_flags+=("--user=")
    flags+=("--vfs-cache-max-age=")
    local_nonpersistent_flags+=("--vfs-cache-max-age=")
    flags+=("--vfs-cache-mode=")
    local_nonpersistent_flags+=("--vfs-cache-mode=")
    flags+=("--vfs-cache-poll-interval=")
    local_nonpersistent_flags+=("--vfs-cache-poll-interval=")
    flags+=("--vfs-read-chunk-size=")
    local_nonpersistent_flags+=("--vfs-read-chunk-size=")
    flags+=("--vfs-read-chunk-size-limit=")
    local_nonpersistent_flags+=("--vfs-read-chunk-size-limit=")
    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_serve_restic()
{
    last_command="rclone_serve_restic"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--addr=")
    local_nonpersistent_flags+=("--addr=")
    flags+=("--append-only")
    local_nonpersistent_flags+=("--append-only")
    flags+=("--cert=")
    local_nonpersistent_flags+=("--cert=")
    flags+=("--client-ca=")
    local_nonpersistent_flags+=("--client-ca=")
    flags+=("--htpasswd=")
    local_nonpersistent_flags+=("--htpasswd=")
    flags+=("--key=")
    local_nonpersistent_flags+=("--key=")
    flags+=("--max-header-bytes=")
    local_nonpersistent_flags+=("--max-header-bytes=")
    flags+=("--pass=")
    local_nonpersistent_flags+=("--pass=")
    flags+=("--realm=")
    local_nonpersistent_flags+=("--realm=")
    flags+=("--server-read-timeout=")
    local_nonpersistent_flags+=("--server-read-timeout=")
    flags+=("--server-write-timeout=")
    local_nonpersistent_flags+=("--server-write-timeout=")
    flags+=("--stdio")
    local_nonpersistent_flags+=("--stdio")
    flags+=("--user=")
    local_nonpersistent_flags+=("--user=")
    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_serve_webdav()
{
    last_command="rclone_serve_webdav"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--addr=")
    local_nonpersistent_flags+=("--addr=")
    flags+=("--cert=")
    local_nonpersistent_flags+=("--cert=")
    flags+=("--client-ca=")
    local_nonpersistent_flags+=("--client-ca=")
    flags+=("--dir-cache-time=")
    local_nonpersistent_flags+=("--dir-cache-time=")
    flags+=("--htpasswd=")
    local_nonpersistent_flags+=("--htpasswd=")
    flags+=("--key=")
    local_nonpersistent_flags+=("--key=")
    flags+=("--max-header-bytes=")
    local_nonpersistent_flags+=("--max-header-bytes=")
    flags+=("--no-checksum")
    local_nonpersistent_flags+=("--no-checksum")
    flags+=("--no-modtime")
    local_nonpersistent_flags+=("--no-modtime")
    flags+=("--no-seek")
    local_nonpersistent_flags+=("--no-seek")
    flags+=("--pass=")
    local_nonpersistent_flags+=("--pass=")
    flags+=("--poll-interval=")
    local_nonpersistent_flags+=("--poll-interval=")
    flags+=("--read-only")
    local_nonpersistent_flags+=("--read-only")
    flags+=("--realm=")
    local_nonpersistent_flags+=("--realm=")
    flags+=("--server-read-timeout=")
    local_nonpersistent_flags+=("--server-read-timeout=")
    flags+=("--server-write-timeout=")
    local_nonpersistent_flags+=("--server-write-timeout=")
    flags+=("--user=")
    local_nonpersistent_flags+=("--user=")
    flags+=("--vfs-cache-max-age=")
    local_nonpersistent_flags+=("--vfs-cache-max-age=")
    flags+=("--vfs-cache-mode=")
    local_nonpersistent_flags+=("--vfs-cache-mode=")
    flags+=("--vfs-cache-poll-interval=")
    local_nonpersistent_flags+=("--vfs-cache-poll-interval=")
    flags+=("--vfs-read-chunk-size=")
    local_nonpersistent_flags+=("--vfs-read-chunk-size=")
    flags+=("--vfs-read-chunk-size-limit=")
    local_nonpersistent_flags+=("--vfs-read-chunk-size-limit=")
    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_serve()
{
    last_command="rclone_serve"
    commands=()
    commands+=("http")
    commands+=("restic")
    commands+=("webdav")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_sha1sum()
{
    last_command="rclone_sha1sum"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_size()
{
    last_command="rclone_size"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--json")
    local_nonpersistent_flags+=("--json")
    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_sync()
{
    last_command="rclone_sync"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_touch()
{
    last_command="rclone_touch"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--no-create")
    flags+=("-C")
    local_nonpersistent_flags+=("--no-create")
    flags+=("--timestamp=")
    two_word_flags+=("-t")
    local_nonpersistent_flags+=("--timestamp=")
    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_tree()
{
    last_command="rclone_tree"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    flags+=("-a")
    local_nonpersistent_flags+=("--all")
    flags+=("--color")
    flags+=("-C")
    local_nonpersistent_flags+=("--color")
    flags+=("--dirs-only")
    flags+=("-d")
    local_nonpersistent_flags+=("--dirs-only")
    flags+=("--dirsfirst")
    local_nonpersistent_flags+=("--dirsfirst")
    flags+=("--full-path")
    local_nonpersistent_flags+=("--full-path")
    flags+=("--human")
    local_nonpersistent_flags+=("--human")
    flags+=("--level=")
    local_nonpersistent_flags+=("--level=")
    flags+=("--modtime")
    flags+=("-D")
    local_nonpersistent_flags+=("--modtime")
    flags+=("--noindent")
    flags+=("-i")
    local_nonpersistent_flags+=("--noindent")
    flags+=("--noreport")
    local_nonpersistent_flags+=("--noreport")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--protections")
    flags+=("-p")
    local_nonpersistent_flags+=("--protections")
    flags+=("--quote")
    flags+=("-Q")
    local_nonpersistent_flags+=("--quote")
    flags+=("--size")
    flags+=("-s")
    local_nonpersistent_flags+=("--size")
    flags+=("--sort=")
    local_nonpersistent_flags+=("--sort=")
    flags+=("--sort-ctime")
    local_nonpersistent_flags+=("--sort-ctime")
    flags+=("--sort-modtime")
    flags+=("-t")
    local_nonpersistent_flags+=("--sort-modtime")
    flags+=("--sort-reverse")
    flags+=("-r")
    local_nonpersistent_flags+=("--sort-reverse")
    flags+=("--unsorted")
    flags+=("-U")
    local_nonpersistent_flags+=("--unsorted")
    flags+=("--version")
    local_nonpersistent_flags+=("--version")
    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_version()
{
    last_command="rclone_version"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_rclone_root_command()
{
    last_command="rclone"
    commands=()
    commands+=("about")
    commands+=("authorize")
    commands+=("cachestats")
    commands+=("cat")
    commands+=("check")
    commands+=("cleanup")
    commands+=("config")
    commands+=("copy")
    commands+=("copyto")
    commands+=("cryptcheck")
    commands+=("cryptdecode")
    commands+=("dbhashsum")
    commands+=("dedupe")
    commands+=("delete")
    commands+=("deletefile")
    commands+=("genautocomplete")
    commands+=("gendocs")
    commands+=("hashsum")
    commands+=("link")
    commands+=("listremotes")
    commands+=("ls")
    commands+=("lsd")
    commands+=("lsf")
    commands+=("lsjson")
    commands+=("lsl")
    commands+=("md5sum")
    commands+=("mkdir")
    commands+=("move")
    commands+=("moveto")
    commands+=("ncdu")
    commands+=("obscure")
    commands+=("purge")
    commands+=("rc")
    commands+=("rcat")
    commands+=("rmdir")
    commands+=("rmdirs")
    commands+=("serve")
    commands+=("sha1sum")
    commands+=("size")
    commands+=("sync")
    commands+=("touch")
    commands+=("tree")
    commands+=("version")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--acd-templink-threshold=")
    flags+=("--acd-upload-wait-per-gb=")
    flags+=("--ask-password")
    flags+=("--auto-confirm")
    flags+=("--azureblob-chunk-size=")
    flags+=("--azureblob-upload-cutoff=")
    flags+=("--b2-chunk-size=")
    flags+=("--b2-hard-delete")
    flags+=("--b2-test-mode=")
    flags+=("--b2-upload-cutoff=")
    flags+=("--b2-versions")
    flags+=("--backup-dir=")
    flags+=("--bind=")
    flags+=("--box-upload-cutoff=")
    flags+=("--buffer-size=")
    flags+=("--bwlimit=")
    flags+=("--cache-chunk-clean-interval=")
    flags+=("--cache-chunk-no-memory")
    flags+=("--cache-chunk-path=")
    flags+=("--cache-chunk-size=")
    flags+=("--cache-db-path=")
    flags+=("--cache-db-purge")
    flags+=("--cache-db-wait-time=")
    flags+=("--cache-dir=")
    flags+=("--cache-info-age=")
    flags+=("--cache-read-retries=")
    flags+=("--cache-rps=")
    flags+=("--cache-tmp-upload-path=")
    flags+=("--cache-tmp-wait-time=")
    flags+=("--cache-total-chunk-size=")
    flags+=("--cache-workers=")
    flags+=("--cache-writes")
    flags+=("--checkers=")
    flags+=("--checksum")
    flags+=("-c")
    flags+=("--config=")
    flags+=("--contimeout=")
    flags+=("--copy-links")
    flags+=("-L")
    flags+=("--cpuprofile=")
    flags+=("--crypt-show-mapping")
    flags+=("--delete-after")
    flags+=("--delete-before")
    flags+=("--delete-during")
    flags+=("--delete-excluded")
    flags+=("--disable=")
    flags+=("--drive-acknowledge-abuse")
    flags+=("--drive-alternate-export")
    flags+=("--drive-auth-owner-only")
    flags+=("--drive-chunk-size=")
    flags+=("--drive-formats=")
    flags+=("--drive-impersonate=")
    flags+=("--drive-list-chunk=")
    flags+=("--drive-shared-with-me")
    flags+=("--drive-skip-gdocs")
    flags+=("--drive-trashed-only")
    flags+=("--drive-upload-cutoff=")
    flags+=("--drive-use-created-date")
    flags+=("--drive-use-trash")
    flags+=("--dropbox-chunk-size=")
    flags+=("--dry-run")
    flags+=("-n")
    flags+=("--dump=")
    flags+=("--dump-bodies")
    flags+=("--dump-headers")
    flags+=("--exclude=")
    flags+=("--exclude-from=")
    flags+=("--exclude-if-present=")
    flags+=("--fast-list")
    flags+=("--files-from=")
    flags+=("--filter=")
    two_word_flags+=("-f")
    flags+=("--filter-from=")
    flags+=("--gcs-location=")
    flags+=("--gcs-storage-class=")
    flags+=("--ignore-checksum")
    flags+=("--ignore-errors")
    flags+=("--ignore-existing")
    flags+=("--ignore-size")
    flags+=("--ignore-times")
    flags+=("-I")
    flags+=("--immutable")
    flags+=("--include=")
    flags+=("--include-from=")
    flags+=("--local-no-check-updated")
    flags+=("--local-no-unicode-normalization")
    flags+=("--log-file=")
    flags+=("--log-level=")
    flags+=("--low-level-retries=")
    flags+=("--max-age=")
    flags+=("--max-delete=")
    flags+=("--max-depth=")
    flags+=("--max-size=")
    flags+=("--max-transfer=")
    flags+=("--mega-debug")
    flags+=("--memprofile=")
    flags+=("--min-age=")
    flags+=("--min-size=")
    flags+=("--modify-window=")
    flags+=("--no-check-certificate")
    flags+=("--no-gzip-encoding")
    flags+=("--no-traverse")
    flags+=("--no-update-modtime")
    flags+=("--one-file-system")
    flags+=("-x")
    flags+=("--onedrive-chunk-size=")
    flags+=("--quiet")
    flags+=("-q")
    flags+=("--rc")
    flags+=("--rc-addr=")
    flags+=("--rc-cert=")
    flags+=("--rc-client-ca=")
    flags+=("--rc-htpasswd=")
    flags+=("--rc-key=")
    flags+=("--rc-max-header-bytes=")
    flags+=("--rc-pass=")
    flags+=("--rc-realm=")
    flags+=("--rc-server-read-timeout=")
    flags+=("--rc-server-write-timeout=")
    flags+=("--rc-user=")
    flags+=("--retries=")
    flags+=("--retries-sleep=")
    flags+=("--s3-acl=")
    flags+=("--s3-chunk-size=")
    flags+=("--s3-disable-checksum")
    flags+=("--s3-storage-class=")
    flags+=("--s3-upload-concurrency=")
    flags+=("--sftp-ask-password")
    flags+=("--size-only")
    flags+=("--skip-links")
    flags+=("--ssh-path-override=")
    flags+=("--stats=")
    flags+=("--stats-file-name-length=")
    flags+=("--stats-log-level=")
    flags+=("--stats-unit=")
    flags+=("--streaming-upload-cutoff=")
    flags+=("--suffix=")
    flags+=("--swift-chunk-size=")
    flags+=("--syslog")
    flags+=("--syslog-facility=")
    flags+=("--timeout=")
    flags+=("--tpslimit=")
    flags+=("--tpslimit-burst=")
    flags+=("--track-renames")
    flags+=("--transfers=")
    flags+=("--update")
    flags+=("-u")
    flags+=("--use-server-modtime")
    flags+=("--user-agent=")
    flags+=("--verbose")
    flags+=("-v")
    flags+=("--version")
    flags+=("-V")
    local_nonpersistent_flags+=("--version")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

__start_rclone()
{
    local cur prev words cword
    declare -A flaghash 2>/dev/null || :
    if declare -F _init_completion >/dev/null 2>&1; then
        _init_completion -s || return
    else
        __rclone_init_completion -n "=" || return
    fi

    local c=0
    local flags=()
    local two_word_flags=()
    local local_nonpersistent_flags=()
    local flags_with_completion=()
    local flags_completion=()
    local commands=("rclone")
    local must_have_one_flag=()
    local must_have_one_noun=()
    local last_command
    local nouns=()

    __rclone_handle_word
}

if [[ $(type -t compopt) = "builtin" ]]; then
    complete -o default -F __start_rclone rclone
else
    complete -o default -o nospace -F __start_rclone rclone
fi

# ex: ts=4 sw=4 et filetype=sh
