package Color::Library::Dictionary::NBS_ISCC::H;

use strict;
use warnings;

use base qw/Color::Library::Dictionary/;

__PACKAGE__->_register_dictionary;

package Color::Library::Dictionary::NBS_ISCC::H;

=pod

=head1 NAME

Color::Library::Dictionary::NBS_ISCC::H - (NBS/ISCC H) Horticultural Colour Charts

=head1 DESCRIPTION

    Kenneth L. Kelly and Deanne B. Judd.
    "Color: Universal Language and Dictionary of Names",
    National Bureau of Standards,
    Spec. Publ. 440, Dec. 1976, 189 pages.


    Color Name Dictionary: (H) Horticultural Colour Charts

    Horticultural Colour Charts,
    R. F. Wilson 

Most names are of plants or flowers. 199 names map to 261 colors, using only 95 of 267 centroids.

L<http://swissnet.ai.mit.edu/~jaffer/Color/Dictionaries#H>

=head1 COLORS

	aconite violet        aconiteviolet        #2f2140

	agathia green         agathiagreen         #7e9f2e

	agathia green         agathiagreen         #44944a

	amaranth rose         amaranthrose         #e68fac

	amber yellow          amberyellow          #fada5e

	amber yellow          amberyellow          #f8de7e

	amethyst violet       amethystviolet       #9a4eae

	apricot               apricot              #eaa221

	aster violet          asterviolet          #9065ca

	aureolin              aureolin             #f3c300

	azalea pink           azaleapink           #ffb7a5

	azalea pink           azaleapink           #e66761

	barium yellow         bariumyellow         #fada5e

	beetroot purple       beetrootpurple       #78184a

	beetroot purple       beetrootpurple       #54133b

	begonia               begonia              #e66761

	bishop's violet       bishopsviolet        #9a4eae

	blood red             bloodred             #be0032

	bluebird blue         bluebirdblue         #545aa7

	bluebird blue         bluebirdblue         #4e5180

	bluebird blue         bluebirdblue         #604e97

	bluebird blue         bluebirdblue         #604e81

	brick red             brickred             #d9603b

	burnt orange          burntorange          #d9603b

	buttercup yellow      buttercupyellow      #f3c300

	butterfly blue        butterflyblue        #4997d0

	butterfly blue        butterflyblue        #0067a5

	cadmium orange        cadmiumorange        #f38400

	cadmium orange        cadmiumorange        #fd943f

	cadmium orange        cadmiumorange        #ed872d

	camelia rose          cameliarose          #e4717a

	campanula violet      campanulaviolet      #9065ca

	canary yellow         canaryyellow         #f3c300

	capri blue            capriblue            #007a74

	capri blue            capriblue            #007791

	capsicum red          capsicumred          #e25822

	cardinal red          cardinalred          #be0032

	carmine               carmine              #be0032

	carmine rose          carminerose          #ffb5ba

	carmine rose          carminerose          #e4717a

	carnation green       carnationgreen       #7d8984

	carrot red            carrotred            #ed872d

	carrot red            carrotred            #d99058

	cerulein Blue         ceruleinblue         #0067a5

	chartreuse green      chartreusegreen      #bdda57

	cherry                cherry               #be0032

	china rose            chinarose            #b3446c

	chinese coral         chinesecoral         #f88379

	chinese coral         chinesecoral         #cb6d51

	chinese yellow        chineseyellow        #fada5e

	chrysanthemum crimson chrysanthemumcrimson #841b2d

	chrysocolla green     chrysocollagreen     #3eb489

	chrysocolla green     chrysocollagreen     #007959

	citron green          citrongreen          #beb72e

	citron green          citrongreen          #b9b459

	claret rose           claretrose           #bc3f4a

	cobalt blue           cobaltblue           #00a1c2

	cobalt violet         cobaltviolet         #875692

	coral pink            coralpink            #e66761

	cornflower blue       cornflowerblue       #30267a

	crimson               crimson              #be0032

	crimson               crimson              #ce4676

	currant red           currantred           #be0032

	cyclamen purple       cyclamenpurple       #870074

	cypress green         cypressgreen         #83d37d

	dahlia purple         dahliapurple         #673147

	dahlia purple         dahliapurple         #38152c

	dauphin's violet      dauphinsviolet       #604e81

	dawn pink             dawnpink             #ffb5ba

	delft rose            delftrose            #bc3f4a

	dianthus purple       dianthuspurple       #ce4676

	doge purple           dogepurple           #702963

	dresden yellow        dresdenyellow        #e9e450

	dutch vermilion       dutchvermilion       #e25822

	egyptian buff         egyptianbuff         #fbc97f

	egyptian buff         egyptianbuff         #fad6a5

	emerald green         emeraldgreen         #44944a

	empire rose           empirerose           #e4717a

	empire yellow         empireyellow         #f3c300

	enamel blue           enamelblue           #00a1c2

	erythrite red         erythritered         #a8516e

	ethyl blue            ethylblue            #66aabc

	faence blue          faenceblue           #239eba

	fern green            ferngreen            #7e9f2e

	fire red              firered              #e25822

	flax blue             flaxblue             #6c79b8

	flax blue             flaxblue             #545aa7

	flax blue             flaxblue             #7e73b8

	french blue           frenchblue           #30267a

	french rose           frenchrose           #f88379

	fuchsia purple        fuchsiapurple        #ce4676

	fuchsine pink         fuchsinepink         #de6fa1

	garnet brown          garnetbrown          #722f37

	garnet lake           garnetlake           #78184a

	gentian blue          gentianblue          #30267a

	geranium lake         geraniumlake         #be0032

	heliotrope            heliotrope           #7e73b8

	hyacinth blue         hyacinthblue         #9065ca

	imperial purple       imperialpurple       #875692

	indian blue           indianblue           #007791

	indian lake           indianlake           #841b2d

	indian orange         indianorange         #e25822

	indian yellow         indianyellow         #ffc14f

	indian yellow         indianyellow         #fbc97f

	ivy green             ivygreen             #31362b

	jade green            jadegreen            #007a74

	jasper red            jasperred            #bc3f4a

	kingfisher blue       kingfisherblue       #007791

	langite green         langitegreen         #007a74

	lavender green        lavendergreen        #8f9779

	leek green            leekgreen            #4a5d23

	leek green            leekgreen            #515744

	lemon yellow          lemonyellow          #f3c300

	lettuce green         lettucegreen         #7e9f2e

	light chrome yellow   lightchromeyellow    #f3c300

	lilac purple          lilacpurple          #a8516e

	lobelia blue          lobeliablue          #9065ca

	magenta               magenta              #ce4676

	magenta rose          magentarose          #b3446c

	magnolia purple       magnoliapurple       #a8516e

	maize yellow          maizeyellow          #ffc14f

	maize yellow          maizeyellow          #eaa221

	maize yellow          maizeyellow          #fbc97f

	maize yellow          maizeyellow          #e3a857

	maize yellow          maizeyellow          #fada5e

	maize yellow          maizeyellow          #d4af37

	majolica yellow       majolicayellow       #ed872d

	majolica yellow       majolicayellow       #d99058

	majolica yellow       majolicayellow       #eaa221

	majolica yellow       majolicayellow       #e3a857

	mallow purple         mallowpurple         #870074

	mandarin red          mandarinred          #e25822

	marigold orange       marigoldorange       #e25822

	marigold orange       marigoldorange       #f38400

	marigold orange       marigoldorange       #ed872d

	maroon                maroon               #3f1728

	mars orange           marsorange           #cb6d51

	mauve                 mauve                #d399e6

	mauve                 mauve                #875692

	mauvette              mauvette             #8c82b6

	mauvette              mauvette             #9690ab

	methyl violet         methylviolet         #9065ca

	mimeosa yellow        mimeosayellow        #e9e450

	mineral violet        mineralviolet        #875692

	moorish blue          moorishblue          #9065ca

	naples yellow         naplesyellow         #fada5e

	naples yellow         naplesyellow         #f8de7e

	nasturtium orange     nasturtiumorange     #ed872d

	nasturtium red        nasturtiumred        #e25822

	neyron rose           neyronrose           #de6fa1

	nickel green          nickelgreen          #3eb489

	orange                orange               #e25822

	orange buff           orangebuff           #fbc97f

	orange buff           orangebuff           #e3a857

	orange buff           orangebuff           #fada5e

	orange buff           orangebuff           #d4af37

	orange buff           orangebuff           #f8de7e

	orange buff           orangebuff           #c9ae5d

	orchid purple         orchidpurple         #870074

	oriental blue         orientalblue         #00a1c2

	orient pink           orientpink           #f4c2c2

	orient pink           orientpink           #d9a6a9

	orient red            orientred            #be0032

	orpiment orange       orpimentorange       #f38400

	orpiment orange       orpimentorange       #ed872d

	oxblood red           oxbloodred           #722f37

	pansy purple          pansypurple          #841b2d

	pansy purple          pansypurple          #78184a

	pansy violet          pansyviolet          #915c83

	paris green           parisgreen           #44944a

	parsley green         parsleygreen         #4a5d23

	pastel lavender       pastellavender       #d6cadd

	pastel lilac          pastellilac          #d6cadd

	pastel mauve          pastelmauve          #e8ccd7

	peach                 peach                #f4c2c2

	peach                 peach                #d9a6a9

	peach                 peach                #fab57f

	pea green             peagreen             #bdda57

	peony purple          peonypurple          #ce4676

	persian blue          persianblue          #0067a5

	persian rose          persianrose          #870074

	persian rose          persianrose          #ce4676

	persimmon orange      persimmonorange      #f38400

	persimmon orange      persimmonorange      #ed872d

	petunia purple        petuniapurple        #9a4eae

	petunia purple        petuniapurple        #870074

	phlox pink            phloxpink            #de6fa1

	phlox purple          phloxpurple          #870074

	plum purple           plumpurple           #301934

	pod green             podgreen             #8a9a5b

	poppy red             poppyred             #e25822

	porcelain blue        porcelainblue        #007791

	porcelain rose        porcelainrose        #e4717a

	porcelain rose        porcelainrose        #e66761

	primrose yellow       primroseyellow       #dcd300

	princess blue         princessblue         #00416a

	purple madder         purplemadder         #722f37

	rhodamine pink        rhodaminepink        #e68fac

	rhodamine purple      rhodaminepurple      #ce4676

	rhodonite red         rhodonitered         #ab4e52

	rose bengal           rosebengal           #ce4676

	roseine purple        roseinepurple        #870074

	rose madder           rosemadder           #ce4676

	rose opal             roseopal             #bc3f4a

	rose pink             rosepink             #ea9399

	rose pink             rosepink             #efbbcc

	rose purple           rosepurple           #b695c0

	rose red              rosered              #be0032

	royal purple          royalpurple          #602f6b

	ruby red              rubyred              #841b2d

	saffron yellow        saffronyellow        #eaa221

	sage green            sagegreen            #4a5d23

	sage green            sagegreen            #515744

	salmon                salmon               #f4c2c2

	salmon                salmon               #fab57f

	sap green             sapgreen             #8db600

	sap green             sapgreen             #bdda57

	saturn red            saturnred            #e25822

	scarlet               scarlet              #be0032

	scheeles green        scheelesgreen        #7e9f2e

	sea blue              seablue              #545aa7

	sea lavender violet   sealavenderviolet    #7e73b8

	shell pink            shellpink            #f88379

	shrimp red            shrimpred            #f88379

	signal red            signalred            #be0032

	solferino purple      solferinopurple      #ce4676

	spanish orange        spanishorange        #ed872d

	spectrum blue         spectrumblue         #00a1c2

	spectrum violet       spectrumviolet       #401a4c

	spinach green         spinachgreen         #4a5d23

	spinach green         spinachgreen         #355e3b

	spinel pink           spinelpink           #e4717a

	spinel red            spinelred            #b3446c

	spiraea red           spiraeared           #ce4676

	straw yellow          strawyellow          #fada5e

	sulphur yellow        sulphuryellow        #f3c300

	sulphur yellow        sulphuryellow        #dcd300

	sung green            sunggreen            #b2beb5

	tangerine orange      tangerineorange      #f38400

	tangerine orange      tangerineorange      #ed872d

	turkey red            turkeyred            #be0032

	tyrian purple         tyrianpurple         #ce4676

	tyrian rose           tyrianrose           #ce4676

	uranium green         uraniumgreen         #bdda57

	venetian pink         venetianpink         #f9ccca

	verdigris             verdigris            #6aab8e

	vermilion             vermilion            #be0032

	vermilion             vermilion            #e25822

	veronese green        veronesegreen        #bdda57

	veronese green        veronesegreen        #83d37d

	veronica violet       veronicaviolet       #7e73b8

	victoria violet       victoriaviolet       #32174d

	violet                violet               #9065ca

	violet purple         violetpurple         #702963

	viridian green        viridiangreen        #3eb489

	viridian green        viridiangreen        #007959

	willow green          willowgreen          #8a9a5b

	willow green          willowgreen          #8f9779

	wisteria blue         wisteriablue         #7e73b8

	yellow ochre          yellowochre          #eaa221

	yellow ochre          yellowochre          #e3a857


=cut

sub _load_color_list() {
    return [
['nbs-iscc-h:aconiteviolet.212','aconiteviolet','aconite violet',[47,33,64],'2f2140',3088704],
['nbs-iscc-h:agathiagreen.117','agathiagreen','agathia green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-h:agathiagreen.131','agathiagreen','agathia green',[68,148,74],'44944a',4494410],
['nbs-iscc-h:amaranthrose.247','amaranthrose','amaranth rose',[230,143,172],'e68fac',15110060],
['nbs-iscc-h:amberyellow.83','amberyellow','amber yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-h:amberyellow.86','amberyellow','amber yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-h:amethystviolet.216','amethystviolet','amethyst violet',[154,78,174],'9a4eae',10112686],
['nbs-iscc-h:apricot.68','apricot','apricot',[234,162,33],'eaa221',15376929],
['nbs-iscc-h:asterviolet.205','asterviolet','aster violet',[144,101,202],'9065ca',9463242],
['nbs-iscc-h:aureolin.82','aureolin','aureolin',[243,195,0],'f3c300',15975168],
['nbs-iscc-h:azaleapink.25','azaleapink','azalea pink',[255,183,165],'ffb7a5',16758693],
['nbs-iscc-h:azaleapink.27','azaleapink','azalea pink',[230,103,97],'e66761',15099745],
['nbs-iscc-h:bariumyellow.83','bariumyellow','barium yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-h:beetrootpurple.256','beetrootpurple','beetroot purple',[120,24,74],'78184a',7870538],
['nbs-iscc-h:beetrootpurple.257','beetrootpurple','beetroot purple',[84,19,59],'54133b',5509947],
['nbs-iscc-h:begonia.27','begonia','begonia',[230,103,97],'e66761',15099745],
['nbs-iscc-h:bishopsviolet.216','bishopsviolet','bishop\'s violet',[154,78,174],'9a4eae',10112686],
['nbs-iscc-h:bloodred.11','bloodred','blood red',[190,0,50],'be0032',12451890],
['nbs-iscc-h:bluebirdblue.196','bluebirdblue','bluebird blue',[84,90,167],'545aa7',5528231],
['nbs-iscc-h:bluebirdblue.200','bluebirdblue','bluebird blue',[78,81,128],'4e5180',5132672],
['nbs-iscc-h:bluebirdblue.207','bluebirdblue','bluebird blue',[96,78,151],'604e97',6311575],
['nbs-iscc-h:bluebirdblue.211','bluebirdblue','bluebird blue',[96,78,129],'604e81',6311553],
['nbs-iscc-h:brickred.35','brickred','brick red',[217,96,59],'d9603b',14245947],
['nbs-iscc-h:burntorange.35','burntorange','burnt orange',[217,96,59],'d9603b',14245947],
['nbs-iscc-h:buttercupyellow.82','buttercupyellow','buttercup yellow',[243,195,0],'f3c300',15975168],
['nbs-iscc-h:butterflyblue.177','butterflyblue','butterfly blue',[73,151,208],'4997d0',4822992],
['nbs-iscc-h:butterflyblue.178','butterflyblue','butterfly blue',[0,103,165],'0067a5',26533],
['nbs-iscc-h:cadmiumorange.48','cadmiumorange','cadmium orange',[243,132,0],'f38400',15959040],
['nbs-iscc-h:cadmiumorange.49','cadmiumorange','cadmium orange',[253,148,63],'fd943f',16618559],
['nbs-iscc-h:cadmiumorange.50','cadmiumorange','cadmium orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-h:cameliarose.3','cameliarose','camelia rose',[228,113,122],'e4717a',14971258],
['nbs-iscc-h:campanulaviolet.205','campanulaviolet','campanula violet',[144,101,202],'9065ca',9463242],
['nbs-iscc-h:canaryyellow.82','canaryyellow','canary yellow',[243,195,0],'f3c300',15975168],
['nbs-iscc-h:capriblue.160','capriblue','capri blue ',[0,122,116],'007a74',31348],
['nbs-iscc-h:capriblue.169','capriblue','capri blue ',[0,119,145],'007791',30609],
['nbs-iscc-h:capsicumred.34','capsicumred','capsicum red',[226,88,34],'e25822',14833698],
['nbs-iscc-h:cardinalred.11','cardinalred','cardinal red',[190,0,50],'be0032',12451890],
['nbs-iscc-h:carmine.11','carmine','carmine',[190,0,50],'be0032',12451890],
['nbs-iscc-h:carminerose.1','carminerose','carmine rose',[255,181,186],'ffb5ba',16758202],
['nbs-iscc-h:carminerose.3','carminerose','carmine rose',[228,113,122],'e4717a',14971258],
['nbs-iscc-h:carnationgreen.155','carnationgreen','carnation green',[125,137,132],'7d8984',8227204],
['nbs-iscc-h:carrotred.50','carrotred','carrot red',[237,135,45],'ed872d',15566637],
['nbs-iscc-h:carrotred.53','carrotred','carrot red',[217,144,88],'d99058',14258264],
['nbs-iscc-h:ceruleinblue.178','ceruleinblue','cerulein Blue ',[0,103,165],'0067a5',26533],
['nbs-iscc-h:chartreusegreen.116','chartreusegreen','chartreuse green',[189,218,87],'bdda57',12442199],
['nbs-iscc-h:cherry.11','cherry','cherry',[190,0,50],'be0032',12451890],
['nbs-iscc-h:chinarose.255','chinarose','china rose',[179,68,108],'b3446c',11748460],
['nbs-iscc-h:chinesecoral.26','chinesecoral','chinese coral',[248,131,121],'f88379',16286585],
['nbs-iscc-h:chinesecoral.37','chinesecoral','chinese coral',[203,109,81],'cb6d51',13331793],
['nbs-iscc-h:chineseyellow.83','chineseyellow','chinese yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-h:chrysanthemumcrimson.13','chrysanthemumcrimson','chrysanthemum crimson',[132,27,45],'841b2d',8657709],
['nbs-iscc-h:chrysocollagreen.140','chrysocollagreen','chrysocolla green',[62,180,137],'3eb489',4109449],
['nbs-iscc-h:chrysocollagreen.141','chrysocollagreen','chrysocolla green',[0,121,89],'007959',31065],
['nbs-iscc-h:citrongreen.99','citrongreen','citron green',[190,183,46],'beb72e',12498734],
['nbs-iscc-h:citrongreen.102','citrongreen','citron green',[185,180,89],'b9b459',12170329],
['nbs-iscc-h:claretrose.12','claretrose','claret rose',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-h:cobaltblue.176','cobaltblue','cobalt blue',[0,161,194],'00a1c2',41410],
['nbs-iscc-h:cobaltviolet.218','cobaltviolet','cobalt violet',[135,86,146],'875692',8869522],
['nbs-iscc-h:coralpink.27','coralpink','coral pink',[230,103,97],'e66761',15099745],
['nbs-iscc-h:cornflowerblue.194','cornflowerblue','cornflower blue',[48,38,122],'30267a',3155578],
['nbs-iscc-h:crimson.11','crimson','crimson',[190,0,50],'be0032',12451890],
['nbs-iscc-h:crimson.254','crimson','crimson',[206,70,118],'ce4676',13518454],
['nbs-iscc-h:currantred.11','currantred','currant red',[190,0,50],'be0032',12451890],
['nbs-iscc-h:cyclamenpurple.236','cyclamenpurple','cyclamen purple',[135,0,116],'870074',8847476],
['nbs-iscc-h:cypressgreen.130','cypressgreen','cypress green',[131,211,125],'83d37d',8639357],
['nbs-iscc-h:dahliapurple.259','dahliapurple','dahlia purple',[103,49,71],'673147',6762823],
['nbs-iscc-h:dahliapurple.260','dahliapurple','dahlia purple',[56,21,44],'38152c',3675436],
['nbs-iscc-h:dauphinsviolet.211','dauphinsviolet','dauphin\'s violet',[96,78,129],'604e81',6311553],
['nbs-iscc-h:dawnpink.1','dawnpink','dawn pink',[255,181,186],'ffb5ba',16758202],
['nbs-iscc-h:delftrose.12','delftrose','delft rose',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-h:dianthuspurple.254','dianthuspurple','dianthus purple',[206,70,118],'ce4676',13518454],
['nbs-iscc-h:dogepurple.238','dogepurple','doge purple',[112,41,99],'702963',7350627],
['nbs-iscc-h:dresdenyellow.98','dresdenyellow','dresden yellow',[233,228,80],'e9e450',15328336],
['nbs-iscc-h:dutchvermilion.34','dutchvermilion','dutch vermilion',[226,88,34],'e25822',14833698],
['nbs-iscc-h:egyptianbuff.70','egyptianbuff','egyptian buff',[251,201,127],'fbc97f',16501119],
['nbs-iscc-h:egyptianbuff.73','egyptianbuff','egyptian buff',[250,214,165],'fad6a5',16438949],
['nbs-iscc-h:emeraldgreen.131','emeraldgreen','emerald green',[68,148,74],'44944a',4494410],
['nbs-iscc-h:empirerose.3','empirerose','empire rose',[228,113,122],'e4717a',14971258],
['nbs-iscc-h:empireyellow.82','empireyellow','empire yellow',[243,195,0],'f3c300',15975168],
['nbs-iscc-h:enamelblue.176','enamelblue','enamel blue',[0,161,194],'00a1c2',41410],
['nbs-iscc-h:erythritered.258','erythritered','erythrite red',[168,81,110],'a8516e',11030894],
['nbs-iscc-h:ethylblue.172','ethylblue','ethyl blue',[102,170,188],'66aabc',6728380],
['nbs-iscc-h:faenceblue.168','faenceblue','faence blue ',[35,158,186],'239eba',2334394],
['nbs-iscc-h:ferngreen.117','ferngreen','fern green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-h:firered.34','firered','fire red',[226,88,34],'e25822',14833698],
['nbs-iscc-h:flaxblue.195','flaxblue','flax blue',[108,121,184],'6c79b8',7109048],
['nbs-iscc-h:flaxblue.196','flaxblue','flax blue',[84,90,167],'545aa7',5528231],
['nbs-iscc-h:flaxblue.206','flaxblue','flax blue',[126,115,184],'7e73b8',8287160],
['nbs-iscc-h:frenchblue.194','frenchblue','french blue',[48,38,122],'30267a',3155578],
['nbs-iscc-h:frenchrose.26','frenchrose','french rose ',[248,131,121],'f88379',16286585],
['nbs-iscc-h:fuchsiapurple.254','fuchsiapurple','fuchsia purple',[206,70,118],'ce4676',13518454],
['nbs-iscc-h:fuchsinepink.248','fuchsinepink','fuchsine pink',[222,111,161],'de6fa1',14577569],
['nbs-iscc-h:garnetbrown.16','garnetbrown','garnet brown',[114,47,55],'722f37',7483191],
['nbs-iscc-h:garnetlake.256','garnetlake','garnet lake',[120,24,74],'78184a',7870538],
['nbs-iscc-h:gentianblue.194','gentianblue','gentian blue',[48,38,122],'30267a',3155578],
['nbs-iscc-h:geraniumlake.11','geraniumlake','geranium lake',[190,0,50],'be0032',12451890],
['nbs-iscc-h:heliotrope.206','heliotrope','heliotrope',[126,115,184],'7e73b8',8287160],
['nbs-iscc-h:hyacinthblue.205','hyacinthblue','hyacinth blue',[144,101,202],'9065ca',9463242],
['nbs-iscc-h:imperialpurple.218','imperialpurple','imperial purple',[135,86,146],'875692',8869522],
['nbs-iscc-h:indianblue.169','indianblue','indian blue',[0,119,145],'007791',30609],
['nbs-iscc-h:indianlake.13','indianlake','indian lake',[132,27,45],'841b2d',8657709],
['nbs-iscc-h:indianorange.34','indianorange','indian orange',[226,88,34],'e25822',14833698],
['nbs-iscc-h:indianyellow.67','indianyellow','indian yellow',[255,193,79],'ffc14f',16761167],
['nbs-iscc-h:indianyellow.70','indianyellow','indian yellow',[251,201,127],'fbc97f',16501119],
['nbs-iscc-h:ivygreen.128','ivygreen','ivy green',[49,54,43],'31362b',3225131],
['nbs-iscc-h:jadegreen.160','jadegreen','jade green',[0,122,116],'007a74',31348],
['nbs-iscc-h:jasperred.12','jasperred','jasper red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-h:kingfisherblue.169','kingfisherblue','kingfisher blue',[0,119,145],'007791',30609],
['nbs-iscc-h:langitegreen.160','langitegreen','langite green',[0,122,116],'007a74',31348],
['nbs-iscc-h:lavendergreen.122','lavendergreen','lavender green',[143,151,121],'8f9779',9410425],
['nbs-iscc-h:leekgreen.125','leekgreen','leek green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-h:leekgreen.127','leekgreen','leek green',[81,87,68],'515744',5330756],
['nbs-iscc-h:lemonyellow.82','lemonyellow','lemon yellow',[243,195,0],'f3c300',15975168],
['nbs-iscc-h:lettucegreen.117','lettucegreen','lettuce green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-h:lightchromeyellow.82','lightchromeyellow','light chrome yellow',[243,195,0],'f3c300',15975168],
['nbs-iscc-h:lilacpurple.258','lilacpurple','lilac purple',[168,81,110],'a8516e',11030894],
['nbs-iscc-h:lobeliablue.205','lobeliablue','lobelia blue',[144,101,202],'9065ca',9463242],
['nbs-iscc-h:magenta.254','magenta','magenta',[206,70,118],'ce4676',13518454],
['nbs-iscc-h:magentarose.255','magentarose','magenta rose ',[179,68,108],'b3446c',11748460],
['nbs-iscc-h:magnoliapurple.258','magnoliapurple','magnolia purple',[168,81,110],'a8516e',11030894],
['nbs-iscc-h:maizeyellow.67','maizeyellow','maize yellow',[255,193,79],'ffc14f',16761167],
['nbs-iscc-h:maizeyellow.68','maizeyellow','maize yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc-h:maizeyellow.70','maizeyellow','maize yellow',[251,201,127],'fbc97f',16501119],
['nbs-iscc-h:maizeyellow.71','maizeyellow','maize yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-h:maizeyellow.83','maizeyellow','maize yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-h:maizeyellow.84','maizeyellow','maize yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-h:majolicayellow.50','majolicayellow','majolica yellow',[237,135,45],'ed872d',15566637],
['nbs-iscc-h:majolicayellow.53','majolicayellow','majolica yellow',[217,144,88],'d99058',14258264],
['nbs-iscc-h:majolicayellow.68','majolicayellow','majolica yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc-h:majolicayellow.71','majolicayellow','majolica yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-h:mallowpurple.236','mallowpurple','mallow purple',[135,0,116],'870074',8847476],
['nbs-iscc-h:mandarinred.34','mandarinred','mandarin red',[226,88,34],'e25822',14833698],
['nbs-iscc-h:marigoldorange.34','marigoldorange','marigold orange',[226,88,34],'e25822',14833698],
['nbs-iscc-h:marigoldorange.48','marigoldorange','marigold orange',[243,132,0],'f38400',15959040],
['nbs-iscc-h:marigoldorange.50','marigoldorange','marigold orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-h:maroon.17','maroon','maroon',[63,23,40],'3f1728',4134696],
['nbs-iscc-h:marsorange.37','marsorange','mars orange',[203,109,81],'cb6d51',13331793],
['nbs-iscc-h:mauve.217','mauve','mauve',[211,153,230],'d399e6',13867494],
['nbs-iscc-h:mauve.218','mauve','mauve',[135,86,146],'875692',8869522],
['nbs-iscc-h:mauvette.210','mauvette','mauvette',[140,130,182],'8c82b6',9208502],
['nbs-iscc-h:mauvette.214','mauvette','mauvette',[150,144,171],'9690ab',9867435],
['nbs-iscc-h:methylviolet.205','methylviolet','methyl violet ',[144,101,202],'9065ca',9463242],
['nbs-iscc-h:mimeosayellow.98','mimeosayellow','mimeosa yellow ',[233,228,80],'e9e450',15328336],
['nbs-iscc-h:mineralviolet.218','mineralviolet','mineral violet ',[135,86,146],'875692',8869522],
['nbs-iscc-h:moorishblue.205','moorishblue','moorish blue ',[144,101,202],'9065ca',9463242],
['nbs-iscc-h:naplesyellow.83','naplesyellow','naples yellow ',[250,218,94],'fada5e',16439902],
['nbs-iscc-h:naplesyellow.86','naplesyellow','naples yellow ',[248,222,126],'f8de7e',16309886],
['nbs-iscc-h:nasturtiumorange.50','nasturtiumorange','nasturtium orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-h:nasturtiumred.34','nasturtiumred','nasturtium red',[226,88,34],'e25822',14833698],
['nbs-iscc-h:neyronrose.248','neyronrose','neyron rose ',[222,111,161],'de6fa1',14577569],
['nbs-iscc-h:nickelgreen.140','nickelgreen','nickel green',[62,180,137],'3eb489',4109449],
['nbs-iscc-h:orange.34','orange','orange',[226,88,34],'e25822',14833698],
['nbs-iscc-h:orangebuff.70','orangebuff','orange buff',[251,201,127],'fbc97f',16501119],
['nbs-iscc-h:orangebuff.71','orangebuff','orange buff',[227,168,87],'e3a857',14919767],
['nbs-iscc-h:orangebuff.83','orangebuff','orange buff',[250,218,94],'fada5e',16439902],
['nbs-iscc-h:orangebuff.84','orangebuff','orange buff',[212,175,55],'d4af37',13938487],
['nbs-iscc-h:orangebuff.86','orangebuff','orange buff',[248,222,126],'f8de7e',16309886],
['nbs-iscc-h:orangebuff.87','orangebuff','orange buff',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-h:orchidpurple.236','orchidpurple','orchid purple',[135,0,116],'870074',8847476],
['nbs-iscc-h:orientalblue.176','orientalblue','oriental blue',[0,161,194],'00a1c2',41410],
['nbs-iscc-h:orientpink.28','orientpink','orient pink',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-h:orientpink.29','orientpink','orient pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-h:orientred.11','orientred','orient red',[190,0,50],'be0032',12451890],
['nbs-iscc-h:orpimentorange.48','orpimentorange','orpiment orange',[243,132,0],'f38400',15959040],
['nbs-iscc-h:orpimentorange.50','orpimentorange','orpiment orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-h:oxbloodred.16','oxbloodred','oxblood red',[114,47,55],'722f37',7483191],
['nbs-iscc-h:pansypurple.13','pansypurple','pansy purple',[132,27,45],'841b2d',8657709],
['nbs-iscc-h:pansypurple.256','pansypurple','pansy purple',[120,24,74],'78184a',7870538],
['nbs-iscc-h:pansyviolet.241','pansyviolet','pansy violet',[145,92,131],'915c83',9526403],
['nbs-iscc-h:parisgreen.131','parisgreen','paris green',[68,148,74],'44944a',4494410],
['nbs-iscc-h:parsleygreen.125','parsleygreen','parsley green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-h:pastellavender.226','pastellavender','pastel lavender',[214,202,221],'d6cadd',14076637],
['nbs-iscc-h:pastellilac.226','pastellilac','pastel lilac',[214,202,221],'d6cadd',14076637],
['nbs-iscc-h:pastelmauve.252','pastelmauve','pastel mauve',[232,204,215],'e8ccd7',15256791],
['nbs-iscc-h:peach.28','peach','peach',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-h:peach.29','peach','peach',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-h:peach.52','peach','peach',[250,181,127],'fab57f',16430463],
['nbs-iscc-h:peagreen.116','peagreen','pea green',[189,218,87],'bdda57',12442199],
['nbs-iscc-h:peonypurple.254','peonypurple','peony purple',[206,70,118],'ce4676',13518454],
['nbs-iscc-h:persianblue.178','persianblue','persian blue',[0,103,165],'0067a5',26533],
['nbs-iscc-h:persianrose.236','persianrose','persian rose',[135,0,116],'870074',8847476],
['nbs-iscc-h:persianrose.254','persianrose','persian rose',[206,70,118],'ce4676',13518454],
['nbs-iscc-h:persimmonorange.48','persimmonorange','persimmon orange',[243,132,0],'f38400',15959040],
['nbs-iscc-h:persimmonorange.50','persimmonorange','persimmon orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-h:petuniapurple.216','petuniapurple','petunia purple',[154,78,174],'9a4eae',10112686],
['nbs-iscc-h:petuniapurple.236','petuniapurple','petunia purple',[135,0,116],'870074',8847476],
['nbs-iscc-h:phloxpink.248','phloxpink','phlox pink',[222,111,161],'de6fa1',14577569],
['nbs-iscc-h:phloxpurple.236','phloxpurple','phlox purple',[135,0,116],'870074',8847476],
['nbs-iscc-h:plumpurple.225','plumpurple','plum purple',[48,25,52],'301934',3152180],
['nbs-iscc-h:podgreen.120','podgreen','pod green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-h:poppyred.34','poppyred','poppy red',[226,88,34],'e25822',14833698],
['nbs-iscc-h:porcelainblue.169','porcelainblue','porcelain blue',[0,119,145],'007791',30609],
['nbs-iscc-h:porcelainrose.3','porcelainrose','porcelain rose',[228,113,122],'e4717a',14971258],
['nbs-iscc-h:porcelainrose.27','porcelainrose','porcelain rose',[230,103,97],'e66761',15099745],
['nbs-iscc-h:primroseyellow.97','primroseyellow','primrose yellow',[220,211,0],'dcd300',14471936],
['nbs-iscc-h:princessblue.179','princessblue','princess blue',[0,65,106],'00416a',16746],
['nbs-iscc-h:purplemadder.16','purplemadder','purple madder',[114,47,55],'722f37',7483191],
['nbs-iscc-h:rhodaminepink.247','rhodaminepink','rhodamine pink',[230,143,172],'e68fac',15110060],
['nbs-iscc-h:rhodaminepurple.254','rhodaminepurple','rhodamine purple',[206,70,118],'ce4676',13518454],
['nbs-iscc-h:rhodonitered.15','rhodonitered','rhodonite red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-h:rosebengal.254','rosebengal','rose bengal',[206,70,118],'ce4676',13518454],
['nbs-iscc-h:roseinepurple.236','roseinepurple','roseine purple',[135,0,116],'870074',8847476],
['nbs-iscc-h:rosemadder.254','rosemadder','rose madder',[206,70,118],'ce4676',13518454],
['nbs-iscc-h:roseopal.12','roseopal','rose opal',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-h:rosepink.2','rosepink','rose pink',[234,147,153],'ea9399',15373209],
['nbs-iscc-h:rosepink.249','rosepink','rose pink',[239,187,204],'efbbcc',15711180],
['nbs-iscc-h:rosepurple.222','rosepurple','rose purple',[182,149,192],'b695c0',11965888],
['nbs-iscc-h:rosered.11','rosered','rose red',[190,0,50],'be0032',12451890],
['nbs-iscc-h:royalpurple.219','royalpurple','royal purple',[96,47,107],'602f6b',6303595],
['nbs-iscc-h:rubyred.13','rubyred','ruby red',[132,27,45],'841b2d',8657709],
['nbs-iscc-h:saffronyellow.68','saffronyellow','saffron yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc-h:sagegreen.125','sagegreen','sage green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-h:sagegreen.127','sagegreen','sage green',[81,87,68],'515744',5330756],
['nbs-iscc-h:salmon.28','salmon','salmon',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-h:salmon.52','salmon','salmon',[250,181,127],'fab57f',16430463],
['nbs-iscc-h:sapgreen.115','sapgreen','sap green',[141,182,0],'8db600',9287168],
['nbs-iscc-h:sapgreen.116','sapgreen','sap green',[189,218,87],'bdda57',12442199],
['nbs-iscc-h:saturnred.34','saturnred','saturn red',[226,88,34],'e25822',14833698],
['nbs-iscc-h:scarlet.11','scarlet','scarlet',[190,0,50],'be0032',12451890],
['nbs-iscc-h:scheelesgreen.117','scheelesgreen','scheeles green ',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-h:seablue.196','seablue','sea blue',[84,90,167],'545aa7',5528231],
['nbs-iscc-h:sealavenderviolet.206','sealavenderviolet','sea lavender violet',[126,115,184],'7e73b8',8287160],
['nbs-iscc-h:shellpink.26','shellpink','shell pink',[248,131,121],'f88379',16286585],
['nbs-iscc-h:shrimpred.26','shrimpred','shrimp red',[248,131,121],'f88379',16286585],
['nbs-iscc-h:signalred.11','signalred','signal red',[190,0,50],'be0032',12451890],
['nbs-iscc-h:solferinopurple.254','solferinopurple','solferino purple',[206,70,118],'ce4676',13518454],
['nbs-iscc-h:spanishorange.50','spanishorange','spanish orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-h:spectrumblue.176','spectrumblue','spectrum blue',[0,161,194],'00a1c2',41410],
['nbs-iscc-h:spectrumviolet.220','spectrumviolet','spectrum violet',[64,26,76],'401a4c',4201036],
['nbs-iscc-h:spinachgreen.125','spinachgreen','spinach green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-h:spinachgreen.137','spinachgreen','spinach green',[53,94,59],'355e3b',3497531],
['nbs-iscc-h:spinelpink.3','spinelpink','spinel pink',[228,113,122],'e4717a',14971258],
['nbs-iscc-h:spinelred.255','spinelred','spinel red',[179,68,108],'b3446c',11748460],
['nbs-iscc-h:spiraeared.254','spiraeared','spiraea red',[206,70,118],'ce4676',13518454],
['nbs-iscc-h:strawyellow.83','strawyellow','straw yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-h:sulphuryellow.82','sulphuryellow','sulphur yellow',[243,195,0],'f3c300',15975168],
['nbs-iscc-h:sulphuryellow.97','sulphuryellow','sulphur yellow',[220,211,0],'dcd300',14471936],
['nbs-iscc-h:sunggreen.154','sunggreen','sung green',[178,190,181],'b2beb5',11714229],
['nbs-iscc-h:tangerineorange.48','tangerineorange','tangerine orange',[243,132,0],'f38400',15959040],
['nbs-iscc-h:tangerineorange.50','tangerineorange','tangerine orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-h:turkeyred.11','turkeyred','turkey red',[190,0,50],'be0032',12451890],
['nbs-iscc-h:tyrianpurple.254','tyrianpurple','tyrian purple',[206,70,118],'ce4676',13518454],
['nbs-iscc-h:tyrianrose.254','tyrianrose','tyrian rose',[206,70,118],'ce4676',13518454],
['nbs-iscc-h:uraniumgreen.116','uraniumgreen','uranium green ',[189,218,87],'bdda57',12442199],
['nbs-iscc-h:venetianpink.4','venetianpink','venetian pink',[249,204,202],'f9ccca',16370890],
['nbs-iscc-h:verdigris.144','verdigris','verdigris',[106,171,142],'6aab8e',6990734],
['nbs-iscc-h:vermilion.11','vermilion','vermilion',[190,0,50],'be0032',12451890],
['nbs-iscc-h:vermilion.34','vermilion','vermilion',[226,88,34],'e25822',14833698],
['nbs-iscc-h:veronesegreen.116','veronesegreen','veronese green',[189,218,87],'bdda57',12442199],
['nbs-iscc-h:veronesegreen.130','veronesegreen','veronese green',[131,211,125],'83d37d',8639357],
['nbs-iscc-h:veronicaviolet.206','veronicaviolet','veronica violet',[126,115,184],'7e73b8',8287160],
['nbs-iscc-h:victoriaviolet.208','victoriaviolet','victoria violet',[50,23,77],'32174d',3282765],
['nbs-iscc-h:violet.205','violet','violet',[144,101,202],'9065ca',9463242],
['nbs-iscc-h:violetpurple.238','violetpurple','violet purple',[112,41,99],'702963',7350627],
['nbs-iscc-h:viridiangreen.140','viridiangreen','viridian green',[62,180,137],'3eb489',4109449],
['nbs-iscc-h:viridiangreen.141','viridiangreen','viridian green',[0,121,89],'007959',31065],
['nbs-iscc-h:willowgreen.120','willowgreen','willow green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-h:willowgreen.122','willowgreen','willow green',[143,151,121],'8f9779',9410425],
['nbs-iscc-h:wisteriablue.206','wisteriablue','wisteria blue',[126,115,184],'7e73b8',8287160],
['nbs-iscc-h:yellowochre.68','yellowochre','yellow ochre',[234,162,33],'eaa221',15376929],
['nbs-iscc-h:yellowochre.71','yellowochre','yellow ochre',[227,168,87],'e3a857',14919767]
    ];
}

sub _description {
    return {
          'subtitle' => 'Horticultural Colour Charts',
          'title' => 'NBS/ISCC H',
          'description' => '    Kenneth L. Kelly and Deanne B. Judd.
    "Color: Universal Language and Dictionary of Names",
    National Bureau of Standards,
    Spec. Publ. 440, Dec. 1976, 189 pages.


    Color Name Dictionary: (H) Horticultural Colour Charts

    Horticultural Colour Charts,
    R. F. Wilson 

Most names are of plants or flowers. 199 names map to 261 colors, using only 95 of 267 centroids.

[http://swissnet.ai.mit.edu/~jaffer/Color/Dictionaries#H]
'
        }

}

1;
