/* $OpenBSD: x509_req.c,v 1.18 2015/09/30 17:30:16 jsing Exp $ */
/* Copyright (C) 1995-1998 Eric Young (eay@cryptsoft.com)
 * All rights reserved.
 *
 * This package is an SSL implementation written
 * by Eric Young (eay@cryptsoft.com).
 * The implementation was written so as to conform with Netscapes SSL.
 *
 * This library is free for commercial and non-commercial use as long as
 * the following conditions are aheared to.  The following conditions
 * apply to all code found in this distribution, be it the RC4, RSA,
 * lhash, DES, etc., code; not just the SSL code.  The SSL documentation
 * included with this distribution is covered by the same copyright terms
 * except that the holder is Tim Hudson (tjh@cryptsoft.com).
 *
 * Copyright remains Eric Young's, and as such any Copyright notices in
 * the code are not to be removed.
 * If this package is used in a product, Eric Young should be given attribution
 * as the author of the parts of the library used.
 * This can be in the form of a textual message at program startup or
 * in documentation (online or textual) provided with the package.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    "This product includes cryptographic software written by
 *     Eric Young (eay@cryptsoft.com)"
 *    The word 'cryptographic' can be left out if the rouines from the library
 *    being used are not cryptographic related :-).
 * 4. If you include any Windows specific code (or a derivative thereof) from
 *    the apps directory (application code) you must include an acknowledgement:
 *    "This product includes software written by Tim Hudson (tjh@cryptsoft.com)"
 *
 * THIS SOFTWARE IS PROVIDED BY ERIC YOUNG ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * The licence and distribution terms for any publically available version or
 * derivative of this code cannot be changed.  i.e. this code cannot simply be
 * copied and put under another distribution licence
 * [including the GNU Public Licence.]
 */

#include <stdio.h>

#include <openssl/opensslconf.h>

#include <openssl/asn1.h>
#include <openssl/asn1t.h>
#include <openssl/bn.h>
#include <openssl/buffer.h>
#include <openssl/err.h>
#include <openssl/evp.h>
#include <openssl/objects.h>
#include <openssl/pem.h>
#include <openssl/x509.h>

X509_REQ *
X509_to_X509_REQ(X509 *x, EVP_PKEY *pkey, const EVP_MD *md)
{
	X509_REQ *ret;
	X509_REQ_INFO *ri;
	int i;
	EVP_PKEY *pktmp;

	ret = X509_REQ_new();
	if (ret == NULL) {
		X509err(X509_F_X509_TO_X509_REQ, ERR_R_MALLOC_FAILURE);
		goto err;
	}

	ri = ret->req_info;

	if ((ri->version = ASN1_INTEGER_new()) == NULL)
		goto err;
	if (ASN1_INTEGER_set(ri->version, 0) == 0)
		goto err;

	if (!X509_REQ_set_subject_name(ret, X509_get_subject_name(x)))
		goto err;

	if ((pktmp = X509_get_pubkey(x)) == NULL)
		goto err;

	i = X509_REQ_set_pubkey(ret, pktmp);
	EVP_PKEY_free(pktmp);
	if (!i)
		goto err;

	if (pkey != NULL) {
		if (!X509_REQ_sign(ret, pkey, md))
			goto err;
	}
	return (ret);

err:
	X509_REQ_free(ret);
	return (NULL);
}

EVP_PKEY *
X509_REQ_get_pubkey(X509_REQ *req)
{
	if ((req == NULL) || (req->req_info == NULL))
		return (NULL);
	return (X509_PUBKEY_get(req->req_info->pubkey));
}

int
X509_REQ_check_private_key(X509_REQ *x, EVP_PKEY *k)
{
	EVP_PKEY *xk = NULL;
	int ok = 0;

	xk = X509_REQ_get_pubkey(x);
	switch (EVP_PKEY_cmp(xk, k)) {
	case 1:
		ok = 1;
		break;
	case 0:
		X509err(X509_F_X509_REQ_CHECK_PRIVATE_KEY,
		    X509_R_KEY_VALUES_MISMATCH);
		break;
	case -1:
		X509err(X509_F_X509_REQ_CHECK_PRIVATE_KEY,
		    X509_R_KEY_TYPE_MISMATCH);
		break;
	case -2:
#ifndef OPENSSL_NO_EC
		if (k->type == EVP_PKEY_EC) {
			X509err(X509_F_X509_REQ_CHECK_PRIVATE_KEY,
			    ERR_R_EC_LIB);
			break;
		}
#endif
#ifndef OPENSSL_NO_DH
		if (k->type == EVP_PKEY_DH) {
			/* No idea */
			X509err(X509_F_X509_REQ_CHECK_PRIVATE_KEY,
			    X509_R_CANT_CHECK_DH_KEY);
			break;
		}
#endif
		X509err(X509_F_X509_REQ_CHECK_PRIVATE_KEY,
		    X509_R_UNKNOWN_KEY_TYPE);
	}

	EVP_PKEY_free(xk);
	return (ok);
}

/* It seems several organisations had the same idea of including a list of
 * extensions in a certificate request. There are at least two OIDs that are
 * used and there may be more: so the list is configurable.
 */

static int ext_nid_list[] = {NID_ext_req, NID_ms_ext_req, NID_undef};

static int *ext_nids = ext_nid_list;

int
X509_REQ_extension_nid(int req_nid)
{
	int i, nid;

	for (i = 0; ; i++) {
		nid = ext_nids[i];
		if (nid == NID_undef)
			return 0;
		else if (req_nid == nid)
			return 1;
	}
}

int *
X509_REQ_get_extension_nids(void)
{
	return ext_nids;
}

void
X509_REQ_set_extension_nids(int *nids)
{
	ext_nids = nids;
}

STACK_OF(X509_EXTENSION) *
X509_REQ_get_extensions(X509_REQ *req)
{
	X509_ATTRIBUTE *attr;
	ASN1_TYPE *ext = NULL;
	int idx, *pnid;
	const unsigned char *p;

	if ((req == NULL) || (req->req_info == NULL) || !ext_nids)
		return (NULL);
	for (pnid = ext_nids; *pnid != NID_undef; pnid++) {
		idx = X509_REQ_get_attr_by_NID(req, *pnid, -1);
		if (idx == -1)
			continue;
		attr = X509_REQ_get_attr(req, idx);
		if (attr->single)
			ext = attr->value.single;
		else if (sk_ASN1_TYPE_num(attr->value.set))
			ext = sk_ASN1_TYPE_value(attr->value.set, 0);
		break;
	}
	if (!ext || (ext->type != V_ASN1_SEQUENCE))
		return NULL;
	p = ext->value.sequence->data;
	return (STACK_OF(X509_EXTENSION) *)ASN1_item_d2i(NULL, &p,
	    ext->value.sequence->length, ASN1_ITEM_rptr(X509_EXTENSIONS));
}

/* Add a STACK_OF extensions to a certificate request: allow alternative OIDs
 * in case we want to create a non standard one.
 */

int
X509_REQ_add_extensions_nid(X509_REQ *req, STACK_OF(X509_EXTENSION) *exts,
    int nid)
{
	ASN1_TYPE *at = NULL;
	X509_ATTRIBUTE *attr = NULL;

	if (!(at = ASN1_TYPE_new()) ||
	    !(at->value.sequence = ASN1_STRING_new()))
		goto err;

	at->type = V_ASN1_SEQUENCE;
	/* Generate encoding of extensions */
	at->value.sequence->length = ASN1_item_i2d((ASN1_VALUE *)exts,
	    &at->value.sequence->data, ASN1_ITEM_rptr(X509_EXTENSIONS));
	if (!(attr = X509_ATTRIBUTE_new()))
		goto err;
	if (!(attr->value.set = sk_ASN1_TYPE_new_null()))
		goto err;
	if (!sk_ASN1_TYPE_push(attr->value.set, at))
		goto err;
	at = NULL;
	attr->single = 0;
	attr->object = OBJ_nid2obj(nid);
	if (!req->req_info->attributes) {
		if (!(req->req_info->attributes = sk_X509_ATTRIBUTE_new_null()))
			goto err;
	}
	if (!sk_X509_ATTRIBUTE_push(req->req_info->attributes, attr))
		goto err;
	return 1;

err:
	X509_ATTRIBUTE_free(attr);
	ASN1_TYPE_free(at);
	return 0;
}

/* This is the normal usage: use the "official" OID */
int
X509_REQ_add_extensions(X509_REQ *req, STACK_OF(X509_EXTENSION) *exts)
{
	return X509_REQ_add_extensions_nid(req, exts, NID_ext_req);
}

/* Request attribute functions */

int
X509_REQ_get_attr_count(const X509_REQ *req)
{
	return X509at_get_attr_count(req->req_info->attributes);
}

int
X509_REQ_get_attr_by_NID(const X509_REQ *req, int nid, int lastpos)
{
	return X509at_get_attr_by_NID(req->req_info->attributes, nid, lastpos);
}

int
X509_REQ_get_attr_by_OBJ(const X509_REQ *req, ASN1_OBJECT *obj, int lastpos)
{
	return X509at_get_attr_by_OBJ(req->req_info->attributes, obj, lastpos);
}

X509_ATTRIBUTE *
X509_REQ_get_attr(const X509_REQ *req, int loc)
{
	return X509at_get_attr(req->req_info->attributes, loc);
}

X509_ATTRIBUTE *
X509_REQ_delete_attr(X509_REQ *req, int loc)
{
	return X509at_delete_attr(req->req_info->attributes, loc);
}

int
X509_REQ_add1_attr(X509_REQ *req, X509_ATTRIBUTE *attr)
{
	if (X509at_add1_attr(&req->req_info->attributes, attr))
		return 1;
	return 0;
}

int
X509_REQ_add1_attr_by_OBJ(X509_REQ *req, const ASN1_OBJECT *obj, int type,
    const unsigned char *bytes, int len)
{
	if (X509at_add1_attr_by_OBJ(&req->req_info->attributes, obj,
	    type, bytes, len))
		return 1;
	return 0;
}

int
X509_REQ_add1_attr_by_NID(X509_REQ *req, int nid, int type,
    const unsigned char *bytes, int len)
{
	if (X509at_add1_attr_by_NID(&req->req_info->attributes, nid,
	    type, bytes, len))
		return 1;
	return 0;
}

int
X509_REQ_add1_attr_by_txt(X509_REQ *req, const char *attrname, int type,
    const unsigned char *bytes, int len)
{
	if (X509at_add1_attr_by_txt(&req->req_info->attributes, attrname,
	    type, bytes, len))
		return 1;
	return 0;
}
