
use strict;
use warnings;

# If run outside the normal test suite on an ASCII platform, you can
# just create a latin1_to_native() function that just returns its
# inputs, because that's the only function used from test.pl
require "test.pl";

# Test qr/\X/ and the \p{} regular expression constructs.  This file is
# constructed by mktables from the tables it generates, so if mktables is
# buggy, this won't necessarily catch those bugs.  Tests are generated for all
# feasible properties; a few aren't currently feasible; see
# is_code_point_usable() in mktables for details.

# Standard test packages are not used because this manipulates SIG_WARN.  It
# exits 0 if every non-skipped test succeeded; -1 if any failed.

my $Tests = 0;
my $Fails = 0;

sub Expect($$$$) {
    my $expected = shift;
    my $ord = shift;
    my $regex  = shift;
    my $warning_type = shift;   # Type of warning message, like 'deprecated'
                                # or empty if none
    my $line   = (caller)[2];

    # Convert the code point to hex form
    my $string = sprintf "\"\\x{%04X}\"", $ord;

    my @tests = "";

    # The first time through, use all warnings.  If the input should generate
    # a warning, add another time through with them turned off
    push @tests, "no warnings '$warning_type';" if $warning_type;

    foreach my $no_warnings (@tests) {

        # Store any warning messages instead of outputting them
        local $SIG{__WARN__} = $SIG{__WARN__};
        my $warning_message;
        $SIG{__WARN__} = sub { $warning_message = $_[0] };

        $Tests++;

        # A string eval is needed because of the 'no warnings'.
        # Assumes no parens in the regular expression
        my $result = eval "$no_warnings
                            my \$RegObj = qr($regex);
                            $string =~ \$RegObj ? 1 : 0";
        if (not defined $result) {
            print "not ok $Tests - couldn't compile /$regex/; line $line: $@\n";
            $Fails++;
        }
        elsif ($result ^ $expected) {
            print "not ok $Tests - expected $expected but got $result for $string =~ qr/$regex/; line $line\n";
            $Fails++;
        }
        elsif ($warning_message) {
            if (! $warning_type || ($warning_type && $no_warnings)) {
                print "not ok $Tests - for qr/$regex/ did not expect warning message '$warning_message'; line $line\n";
                $Fails++;
            }
            else {
                print "ok $Tests - expected and got a warning message for qr/$regex/; line $line\n";
            }
        }
        elsif ($warning_type && ! $no_warnings) {
            print "not ok $Tests - for qr/$regex/ expected a $warning_type warning message, but got none; line $line\n";
            $Fails++;
        }
        else {
            print "ok $Tests - got $result for $string =~ qr/$regex/; line $line\n";
        }
    }
    return;
}

sub Error($) {
    my $regex  = shift;
    $Tests++;
    if (eval { 'x' =~ qr/$regex/; 1 }) {
        $Fails++;
        my $line = (caller)[2];
        print "not ok $Tests - re compiled ok, but expected error for qr/$regex/; line $line: $@\n";
    }
    else {
        my $line = (caller)[2];
        print "ok $Tests - got and expected error for qr/$regex/; line $line\n";
    }
    return;
}

# GCBTest.txt character that separates grapheme clusters
my $breakable_utf8 = my $breakable = chr(utf8::unicode_to_native(0xF7));
utf8::upgrade($breakable_utf8);

# GCBTest.txt character that indicates that the adjoining code points are part
# of the same grapheme cluster
my $nobreak_utf8 = my $nobreak = chr(utf8::unicode_to_native(0xD7));
utf8::upgrade($nobreak_utf8);

sub Test_X($) {
    # Test qr/\X/ matches.  The input is a line from auxiliary/GCBTest.txt
    # Each such line is a sequence of code points given by their hex numbers,
    # separated by the two characters defined just before this subroutine that
    # indicate that either there can or cannot be a break between the adjacent
    # code points.  If there isn't a break, that means the sequence forms an
    # extended grapheme cluster, which means that \X should match the whole
    # thing.  If there is a break, \X should stop there.  This is all
    # converted by this routine into a match:
    #   $string =~ /(\X)/,
    # Each \X should match the next cluster; and that is what is checked.

    my $template = shift;

    my $line   = (caller)[2];

    # The line contains characters above the ASCII range, but in Latin1.  It
    # may or may not be in utf8, and if it is, it may or may not know it.  So,
    # convert these characters to 8 bits.  If knows is in utf8, simply
    # downgrade.
    if (utf8::is_utf8($template)) {
        utf8::downgrade($template);
    } else {

        # Otherwise, if it is in utf8, but doesn't know it, the next lines
        # convert the two problematic characters to their 8-bit equivalents.
        # If it isn't in utf8, they don't harm anything.
        use bytes;
        $template =~ s/$nobreak_utf8/$nobreak/g;
        $template =~ s/$breakable_utf8/$breakable/g;
    }

    # Get rid of the leading and trailing breakables
    $template =~ s/^ \s* $breakable \s* //x;
    $template =~ s/ \s* $breakable \s* $ //x;

    # And no-breaks become just a space.
    $template =~ s/ \s* $nobreak \s* / /xg;

    # Split the input into segments that are breakable between them.
    my @segments = split /\s*$breakable\s*/, $template;

    my $string = "";
    my $display_string = "";
    my @should_match;
    my @should_display;

    # Convert the code point sequence in each segment into a Perl string of
    # characters
    foreach my $segment (@segments) {
        my @code_points = split /\s+/, $segment;
        my $this_string = "";
        my $this_display = "";
        foreach my $code_point (@code_points) {
            $this_string .= latin1_to_native(chr(hex $code_point));
            $this_display .= "\\x{$code_point}";
        }

        # The next cluster should match the string in this segment.
        push @should_match, $this_string;
        push @should_display, $this_display;
        $string .= $this_string;
        $display_string .= $this_display;
    }

    # If a string can be represented in both non-ut8 and utf8, test both cases
    UPGRADE:
    for my $to_upgrade (0 .. 1) {

        if ($to_upgrade) {

            # If already in utf8, would just be a repeat
            next UPGRADE if utf8::is_utf8($string);

            utf8::upgrade($string);
        }

        # Finally, do the \X match.
        my @matches = $string =~ /(\X)/g;

        # Look through each matched cluster to verify that it matches what we
        # expect.
        my $min = (@matches < @should_match) ? @matches : @should_match;
        for my $i (0 .. $min - 1) {
            $Tests++;
            if ($matches[$i] eq $should_match[$i]) {
                print "ok $Tests - ";
                if ($i == 0) {
                    print "In \"$display_string\" =~ /(\\X)/g, \\X #1";
                } else {
                    print "And \\X #", $i + 1,
                }
                print " correctly matched $should_display[$i]; line $line\n";
            } else {
                $matches[$i] = join("", map { sprintf "\\x{%04X}", $_ }
                                                    unpack("U*", $matches[$i]));
                print "not ok $Tests - In \"$display_string\" =~ /(\\X)/g, \\X #",
                    $i + 1,
                    " should have matched $should_display[$i]",
                    " but instead matched $matches[$i]",
                    ".  Abandoning rest of line $line\n";
                next UPGRADE;
            }
        }

        # And the number of matches should equal the number of expected matches.
        $Tests++;
        if (@matches == @should_match) {
            print "ok $Tests - Nothing was left over; line $line\n";
        } else {
            print "not ok $Tests - There were ", scalar @should_match, " \\X matches expected, but got ", scalar @matches, " instead; line $line\n";
        }
    }

    return;
}

sub Finished() {
    print "1..$Tests\n";
    exit($Fails ? -1 : 0);
}

Error('\p{Script=InGreek}');    # Bug #69018
Test_X("1100 $nobreak 1161");  # Bug #70940
Expect(0, 0x2028, '\p{Print}', ""); # Bug # 71722
Expect(0, 0x2029, '\p{Print}', ""); # Bug # 71722
Expect(1, 0xFF10, '\p{XDigit}', ""); # Bug # 71726
Error('\p{ASCII_Hex_Digit= 	NO:=}');
Error('\P{ASCII_Hex_Digit= 	NO:=}');
Expect(1, 103, '\p{ASCII_Hex_Digit=no}', "");
Expect(0, 103, '\p{^ASCII_Hex_Digit=no}', "");
Expect(0, 103, '\P{ASCII_Hex_Digit=no}', "");
Expect(1, 103, '\P{^ASCII_Hex_Digit=no}', "");
Expect(0, 102, '\p{ASCII_Hex_Digit=no}', "");
Expect(1, 102, '\p{^ASCII_Hex_Digit=no}', "");
Expect(1, 102, '\P{ASCII_Hex_Digit=no}', "");
Expect(0, 102, '\P{^ASCII_Hex_Digit=no}', "");
Expect(1, 103, '\p{ASCII_Hex_Digit=-No}', "");
Expect(0, 103, '\p{^ASCII_Hex_Digit=-No}', "");
Expect(0, 103, '\P{ASCII_Hex_Digit=-No}', "");
Expect(1, 103, '\P{^ASCII_Hex_Digit=-No}', "");
Expect(0, 102, '\p{ASCII_Hex_Digit=-No}', "");
Expect(1, 102, '\p{^ASCII_Hex_Digit=-No}', "");
Expect(1, 102, '\P{ASCII_Hex_Digit=-No}', "");
Expect(0, 102, '\P{^ASCII_Hex_Digit=-No}', "");
Error('\p{AHex=-:=n}');
Error('\P{AHex=-:=n}');
Expect(1, 103, '\p{AHex=n}', "");
Expect(0, 103, '\p{^AHex=n}', "");
Expect(0, 103, '\P{AHex=n}', "");
Expect(1, 103, '\P{^AHex=n}', "");
Expect(0, 102, '\p{AHex=n}', "");
Expect(1, 102, '\p{^AHex=n}', "");
Expect(1, 102, '\P{AHex=n}', "");
Expect(0, 102, '\P{^AHex=n}', "");
Expect(1, 103, '\p{AHex=	-N}', "");
Expect(0, 103, '\p{^AHex=	-N}', "");
Expect(0, 103, '\P{AHex=	-N}', "");
Expect(1, 103, '\P{^AHex=	-N}', "");
Expect(0, 102, '\p{AHex=	-N}', "");
Expect(1, 102, '\p{^AHex=	-N}', "");
Expect(1, 102, '\P{AHex=	-N}', "");
Expect(0, 102, '\P{^AHex=	-N}', "");
Error('\p{Is_ASCII_Hex_Digit: _:=f}');
Error('\P{Is_ASCII_Hex_Digit: _:=f}');
Expect(1, 103, '\p{Is_ASCII_Hex_Digit=f}', "");
Expect(0, 103, '\p{^Is_ASCII_Hex_Digit=f}', "");
Expect(0, 103, '\P{Is_ASCII_Hex_Digit=f}', "");
Expect(1, 103, '\P{^Is_ASCII_Hex_Digit=f}', "");
Expect(0, 102, '\p{Is_ASCII_Hex_Digit=f}', "");
Expect(1, 102, '\p{^Is_ASCII_Hex_Digit=f}', "");
Expect(1, 102, '\P{Is_ASCII_Hex_Digit=f}', "");
Expect(0, 102, '\P{^Is_ASCII_Hex_Digit=f}', "");
Expect(1, 103, '\p{Is_ASCII_Hex_Digit=- F}', "");
Expect(0, 103, '\p{^Is_ASCII_Hex_Digit=- F}', "");
Expect(0, 103, '\P{Is_ASCII_Hex_Digit=- F}', "");
Expect(1, 103, '\P{^Is_ASCII_Hex_Digit=- F}', "");
Expect(0, 102, '\p{Is_ASCII_Hex_Digit=- F}', "");
Expect(1, 102, '\p{^Is_ASCII_Hex_Digit=- F}', "");
Expect(1, 102, '\P{Is_ASCII_Hex_Digit=- F}', "");
Expect(0, 102, '\P{^Is_ASCII_Hex_Digit=- F}', "");
Error('\p{Is_AHex=	 FALSE/a/}');
Error('\P{Is_AHex=	 FALSE/a/}');
Expect(1, 103, '\p{Is_AHex=false}', "");
Expect(0, 103, '\p{^Is_AHex=false}', "");
Expect(0, 103, '\P{Is_AHex=false}', "");
Expect(1, 103, '\P{^Is_AHex=false}', "");
Expect(0, 102, '\p{Is_AHex=false}', "");
Expect(1, 102, '\p{^Is_AHex=false}', "");
Expect(1, 102, '\P{Is_AHex=false}', "");
Expect(0, 102, '\P{^Is_AHex=false}', "");
Expect(1, 103, '\p{Is_AHex=	FALSE}', "");
Expect(0, 103, '\p{^Is_AHex=	FALSE}', "");
Expect(0, 103, '\P{Is_AHex=	FALSE}', "");
Expect(1, 103, '\P{^Is_AHex=	FALSE}', "");
Expect(0, 102, '\p{Is_AHex=	FALSE}', "");
Expect(1, 102, '\p{^Is_AHex=	FALSE}', "");
Expect(1, 102, '\P{Is_AHex=	FALSE}', "");
Expect(0, 102, '\P{^Is_AHex=	FALSE}', "");
Error('\p{ASCII_Hex_Digit=:=_-yes}');
Error('\P{ASCII_Hex_Digit=:=_-yes}');
Expect(1, 102, '\p{ASCII_Hex_Digit=yes}', "");
Expect(0, 102, '\p{^ASCII_Hex_Digit=yes}', "");
Expect(0, 102, '\P{ASCII_Hex_Digit=yes}', "");
Expect(1, 102, '\P{^ASCII_Hex_Digit=yes}', "");
Expect(0, 103, '\p{ASCII_Hex_Digit=yes}', "");
Expect(1, 103, '\p{^ASCII_Hex_Digit=yes}', "");
Expect(1, 103, '\P{ASCII_Hex_Digit=yes}', "");
Expect(0, 103, '\P{^ASCII_Hex_Digit=yes}', "");
Expect(1, 102, '\p{ASCII_Hex_Digit=_yes}', "");
Expect(0, 102, '\p{^ASCII_Hex_Digit=_yes}', "");
Expect(0, 102, '\P{ASCII_Hex_Digit=_yes}', "");
Expect(1, 102, '\P{^ASCII_Hex_Digit=_yes}', "");
Expect(0, 103, '\p{ASCII_Hex_Digit=_yes}', "");
Expect(1, 103, '\p{^ASCII_Hex_Digit=_yes}', "");
Expect(1, 103, '\P{ASCII_Hex_Digit=_yes}', "");
Expect(0, 103, '\P{^ASCII_Hex_Digit=_yes}', "");
Error('\p{AHex=_/a/Y}');
Error('\P{AHex=_/a/Y}');
Expect(1, 102, '\p{AHex=y}', "");
Expect(0, 102, '\p{^AHex=y}', "");
Expect(0, 102, '\P{AHex=y}', "");
Expect(1, 102, '\P{^AHex=y}', "");
Expect(0, 103, '\p{AHex=y}', "");
Expect(1, 103, '\p{^AHex=y}', "");
Expect(1, 103, '\P{AHex=y}', "");
Expect(0, 103, '\P{^AHex=y}', "");
Expect(1, 102, '\p{AHex=  Y}', "");
Expect(0, 102, '\p{^AHex=  Y}', "");
Expect(0, 102, '\P{AHex=  Y}', "");
Expect(1, 102, '\P{^AHex=  Y}', "");
Expect(0, 103, '\p{AHex=  Y}', "");
Expect(1, 103, '\p{^AHex=  Y}', "");
Expect(1, 103, '\P{AHex=  Y}', "");
Expect(0, 103, '\P{^AHex=  Y}', "");
Error('\p{Is_ASCII_Hex_Digit= T:=}');
Error('\P{Is_ASCII_Hex_Digit= T:=}');
Expect(1, 102, '\p{Is_ASCII_Hex_Digit=t}', "");
Expect(0, 102, '\p{^Is_ASCII_Hex_Digit=t}', "");
Expect(0, 102, '\P{Is_ASCII_Hex_Digit=t}', "");
Expect(1, 102, '\P{^Is_ASCII_Hex_Digit=t}', "");
Expect(0, 103, '\p{Is_ASCII_Hex_Digit=t}', "");
Expect(1, 103, '\p{^Is_ASCII_Hex_Digit=t}', "");
Expect(1, 103, '\P{Is_ASCII_Hex_Digit=t}', "");
Expect(0, 103, '\P{^Is_ASCII_Hex_Digit=t}', "");
Expect(1, 102, '\p{Is_ASCII_Hex_Digit:		T}', "");
Expect(0, 102, '\p{^Is_ASCII_Hex_Digit:		T}', "");
Expect(0, 102, '\P{Is_ASCII_Hex_Digit:		T}', "");
Expect(1, 102, '\P{^Is_ASCII_Hex_Digit:		T}', "");
Expect(0, 103, '\p{Is_ASCII_Hex_Digit:		T}', "");
Expect(1, 103, '\p{^Is_ASCII_Hex_Digit:		T}', "");
Expect(1, 103, '\P{Is_ASCII_Hex_Digit:		T}', "");
Expect(0, 103, '\P{^Is_ASCII_Hex_Digit:		T}', "");
Error('\p{Is_AHex:	/a/ true}');
Error('\P{Is_AHex:	/a/ true}');
Expect(1, 102, '\p{Is_AHex=true}', "");
Expect(0, 102, '\p{^Is_AHex=true}', "");
Expect(0, 102, '\P{Is_AHex=true}', "");
Expect(1, 102, '\P{^Is_AHex=true}', "");
Expect(0, 103, '\p{Is_AHex=true}', "");
Expect(1, 103, '\p{^Is_AHex=true}', "");
Expect(1, 103, '\P{Is_AHex=true}', "");
Expect(0, 103, '\P{^Is_AHex=true}', "");
Expect(1, 102, '\p{Is_AHex=	-True}', "");
Expect(0, 102, '\p{^Is_AHex=	-True}', "");
Expect(0, 102, '\P{Is_AHex=	-True}', "");
Expect(1, 102, '\P{^Is_AHex=	-True}', "");
Expect(0, 103, '\p{Is_AHex=	-True}', "");
Expect(1, 103, '\p{^Is_AHex=	-True}', "");
Expect(1, 103, '\P{Is_AHex=	-True}', "");
Expect(0, 103, '\P{^Is_AHex=	-True}', "");
Error('\p{Alphabetic=/a/_ NO}');
Error('\P{Alphabetic=/a/_ NO}');
Expect(1, 195102, '\p{Alphabetic=no}', "");
Expect(0, 195102, '\p{^Alphabetic=no}', "");
Expect(0, 195102, '\P{Alphabetic=no}', "");
Expect(1, 195102, '\P{^Alphabetic=no}', "");
Expect(0, 195101, '\p{Alphabetic=no}', "");
Expect(1, 195101, '\p{^Alphabetic=no}', "");
Expect(1, 195101, '\P{Alphabetic=no}', "");
Expect(0, 195101, '\P{^Alphabetic=no}', "");
Expect(1, 195102, '\p{Alphabetic=  No}', "");
Expect(0, 195102, '\p{^Alphabetic=  No}', "");
Expect(0, 195102, '\P{Alphabetic=  No}', "");
Expect(1, 195102, '\P{^Alphabetic=  No}', "");
Expect(0, 195101, '\p{Alphabetic=  No}', "");
Expect(1, 195101, '\p{^Alphabetic=  No}', "");
Expect(1, 195101, '\P{Alphabetic=  No}', "");
Expect(0, 195101, '\P{^Alphabetic=  No}', "");
Error('\p{Alpha=-	n:=}');
Error('\P{Alpha=-	n:=}');
Expect(1, 195102, '\p{Alpha=n}', "");
Expect(0, 195102, '\p{^Alpha=n}', "");
Expect(0, 195102, '\P{Alpha=n}', "");
Expect(1, 195102, '\P{^Alpha=n}', "");
Expect(0, 195101, '\p{Alpha=n}', "");
Expect(1, 195101, '\p{^Alpha=n}', "");
Expect(1, 195101, '\P{Alpha=n}', "");
Expect(0, 195101, '\P{^Alpha=n}', "");
Expect(1, 195102, '\p{Alpha=		n}', "");
Expect(0, 195102, '\p{^Alpha=		n}', "");
Expect(0, 195102, '\P{Alpha=		n}', "");
Expect(1, 195102, '\P{^Alpha=		n}', "");
Expect(0, 195101, '\p{Alpha=		n}', "");
Expect(1, 195101, '\p{^Alpha=		n}', "");
Expect(1, 195101, '\P{Alpha=		n}', "");
Expect(0, 195101, '\P{^Alpha=		n}', "");
Error('\p{Is_Alphabetic: 	 F:=}');
Error('\P{Is_Alphabetic: 	 F:=}');
Expect(1, 195102, '\p{Is_Alphabetic=f}', "");
Expect(0, 195102, '\p{^Is_Alphabetic=f}', "");
Expect(0, 195102, '\P{Is_Alphabetic=f}', "");
Expect(1, 195102, '\P{^Is_Alphabetic=f}', "");
Expect(0, 195101, '\p{Is_Alphabetic=f}', "");
Expect(1, 195101, '\p{^Is_Alphabetic=f}', "");
Expect(1, 195101, '\P{Is_Alphabetic=f}', "");
Expect(0, 195101, '\P{^Is_Alphabetic=f}', "");
Expect(1, 195102, '\p{Is_Alphabetic: _ F}', "");
Expect(0, 195102, '\p{^Is_Alphabetic: _ F}', "");
Expect(0, 195102, '\P{Is_Alphabetic: _ F}', "");
Expect(1, 195102, '\P{^Is_Alphabetic: _ F}', "");
Expect(0, 195101, '\p{Is_Alphabetic: _ F}', "");
Expect(1, 195101, '\p{^Is_Alphabetic: _ F}', "");
Expect(1, 195101, '\P{Is_Alphabetic: _ F}', "");
Expect(0, 195101, '\P{^Is_Alphabetic: _ F}', "");
Error('\p{Is_Alpha= /a/FALSE}');
Error('\P{Is_Alpha= /a/FALSE}');
Expect(1, 195102, '\p{Is_Alpha=false}', "");
Expect(0, 195102, '\p{^Is_Alpha=false}', "");
Expect(0, 195102, '\P{Is_Alpha=false}', "");
Expect(1, 195102, '\P{^Is_Alpha=false}', "");
Expect(0, 195101, '\p{Is_Alpha=false}', "");
Expect(1, 195101, '\p{^Is_Alpha=false}', "");
Expect(1, 195101, '\P{Is_Alpha=false}', "");
Expect(0, 195101, '\P{^Is_Alpha=false}', "");
Expect(1, 195102, '\p{Is_Alpha=-	FALSE}', "");
Expect(0, 195102, '\p{^Is_Alpha=-	FALSE}', "");
Expect(0, 195102, '\P{Is_Alpha=-	FALSE}', "");
Expect(1, 195102, '\P{^Is_Alpha=-	FALSE}', "");
Expect(0, 195101, '\p{Is_Alpha=-	FALSE}', "");
Expect(1, 195101, '\p{^Is_Alpha=-	FALSE}', "");
Expect(1, 195101, '\P{Is_Alpha=-	FALSE}', "");
Expect(0, 195101, '\P{^Is_Alpha=-	FALSE}', "");
Error('\p{Alphabetic=:=- Yes}');
Error('\P{Alphabetic=:=- Yes}');
Expect(1, 195101, '\p{Alphabetic=yes}', "");
Expect(0, 195101, '\p{^Alphabetic=yes}', "");
Expect(0, 195101, '\P{Alphabetic=yes}', "");
Expect(1, 195101, '\P{^Alphabetic=yes}', "");
Expect(0, 195102, '\p{Alphabetic=yes}', "");
Expect(1, 195102, '\p{^Alphabetic=yes}', "");
Expect(1, 195102, '\P{Alphabetic=yes}', "");
Expect(0, 195102, '\P{^Alphabetic=yes}', "");
Expect(1, 195101, '\p{Alphabetic=_	yes}', "");
Expect(0, 195101, '\p{^Alphabetic=_	yes}', "");
Expect(0, 195101, '\P{Alphabetic=_	yes}', "");
Expect(1, 195101, '\P{^Alphabetic=_	yes}', "");
Expect(0, 195102, '\p{Alphabetic=_	yes}', "");
Expect(1, 195102, '\p{^Alphabetic=_	yes}', "");
Expect(1, 195102, '\P{Alphabetic=_	yes}', "");
Expect(0, 195102, '\P{^Alphabetic=_	yes}', "");
Error('\p{Alpha=/a/Y}');
Error('\P{Alpha=/a/Y}');
Expect(1, 195101, '\p{Alpha=y}', "");
Expect(0, 195101, '\p{^Alpha=y}', "");
Expect(0, 195101, '\P{Alpha=y}', "");
Expect(1, 195101, '\P{^Alpha=y}', "");
Expect(0, 195102, '\p{Alpha=y}', "");
Expect(1, 195102, '\p{^Alpha=y}', "");
Expect(1, 195102, '\P{Alpha=y}', "");
Expect(0, 195102, '\P{^Alpha=y}', "");
Expect(1, 195101, '\p{Alpha=_-Y}', "");
Expect(0, 195101, '\p{^Alpha=_-Y}', "");
Expect(0, 195101, '\P{Alpha=_-Y}', "");
Expect(1, 195101, '\P{^Alpha=_-Y}', "");
Expect(0, 195102, '\p{Alpha=_-Y}', "");
Expect(1, 195102, '\p{^Alpha=_-Y}', "");
Expect(1, 195102, '\P{Alpha=_-Y}', "");
Expect(0, 195102, '\P{^Alpha=_-Y}', "");
Error('\p{Is_Alphabetic=/a/_T}');
Error('\P{Is_Alphabetic=/a/_T}');
Expect(1, 195101, '\p{Is_Alphabetic=t}', "");
Expect(0, 195101, '\p{^Is_Alphabetic=t}', "");
Expect(0, 195101, '\P{Is_Alphabetic=t}', "");
Expect(1, 195101, '\P{^Is_Alphabetic=t}', "");
Expect(0, 195102, '\p{Is_Alphabetic=t}', "");
Expect(1, 195102, '\p{^Is_Alphabetic=t}', "");
Expect(1, 195102, '\P{Is_Alphabetic=t}', "");
Expect(0, 195102, '\P{^Is_Alphabetic=t}', "");
Expect(1, 195101, '\p{Is_Alphabetic= T}', "");
Expect(0, 195101, '\p{^Is_Alphabetic= T}', "");
Expect(0, 195101, '\P{Is_Alphabetic= T}', "");
Expect(1, 195101, '\P{^Is_Alphabetic= T}', "");
Expect(0, 195102, '\p{Is_Alphabetic= T}', "");
Expect(1, 195102, '\p{^Is_Alphabetic= T}', "");
Expect(1, 195102, '\P{Is_Alphabetic= T}', "");
Expect(0, 195102, '\P{^Is_Alphabetic= T}', "");
Error('\p{Is_Alpha=:=-true}');
Error('\P{Is_Alpha=:=-true}');
Expect(1, 195101, '\p{Is_Alpha=true}', "");
Expect(0, 195101, '\p{^Is_Alpha=true}', "");
Expect(0, 195101, '\P{Is_Alpha=true}', "");
Expect(1, 195101, '\P{^Is_Alpha=true}', "");
Expect(0, 195102, '\p{Is_Alpha=true}', "");
Expect(1, 195102, '\p{^Is_Alpha=true}', "");
Expect(1, 195102, '\P{Is_Alpha=true}', "");
Expect(0, 195102, '\P{^Is_Alpha=true}', "");
Expect(1, 195101, '\p{Is_Alpha= TRUE}', "");
Expect(0, 195101, '\p{^Is_Alpha= TRUE}', "");
Expect(0, 195101, '\P{Is_Alpha= TRUE}', "");
Expect(1, 195101, '\P{^Is_Alpha= TRUE}', "");
Expect(0, 195102, '\p{Is_Alpha= TRUE}', "");
Expect(1, 195102, '\p{^Is_Alpha= TRUE}', "");
Expect(1, 195102, '\P{Is_Alpha= TRUE}', "");
Expect(0, 195102, '\P{^Is_Alpha= TRUE}', "");
Error('\p{Bidi_Control=		No/a/}');
Error('\P{Bidi_Control=		No/a/}');
Expect(1, 8298, '\p{Bidi_Control=no}', "");
Expect(0, 8298, '\p{^Bidi_Control=no}', "");
Expect(0, 8298, '\P{Bidi_Control=no}', "");
Expect(1, 8298, '\P{^Bidi_Control=no}', "");
Expect(0, 8297, '\p{Bidi_Control=no}', "");
Expect(1, 8297, '\p{^Bidi_Control=no}', "");
Expect(1, 8297, '\P{Bidi_Control=no}', "");
Expect(0, 8297, '\P{^Bidi_Control=no}', "");
Expect(1, 8298, '\p{Bidi_Control=	 no}', "");
Expect(0, 8298, '\p{^Bidi_Control=	 no}', "");
Expect(0, 8298, '\P{Bidi_Control=	 no}', "");
Expect(1, 8298, '\P{^Bidi_Control=	 no}', "");
Expect(0, 8297, '\p{Bidi_Control=	 no}', "");
Expect(1, 8297, '\p{^Bidi_Control=	 no}', "");
Expect(1, 8297, '\P{Bidi_Control=	 no}', "");
Expect(0, 8297, '\P{^Bidi_Control=	 no}', "");
Error('\p{Bidi_C=:=- N}');
Error('\P{Bidi_C=:=- N}');
Expect(1, 8298, '\p{Bidi_C=n}', "");
Expect(0, 8298, '\p{^Bidi_C=n}', "");
Expect(0, 8298, '\P{Bidi_C=n}', "");
Expect(1, 8298, '\P{^Bidi_C=n}', "");
Expect(0, 8297, '\p{Bidi_C=n}', "");
Expect(1, 8297, '\p{^Bidi_C=n}', "");
Expect(1, 8297, '\P{Bidi_C=n}', "");
Expect(0, 8297, '\P{^Bidi_C=n}', "");
Expect(1, 8298, '\p{Bidi_C=N}', "");
Expect(0, 8298, '\p{^Bidi_C=N}', "");
Expect(0, 8298, '\P{Bidi_C=N}', "");
Expect(1, 8298, '\P{^Bidi_C=N}', "");
Expect(0, 8297, '\p{Bidi_C=N}', "");
Expect(1, 8297, '\p{^Bidi_C=N}', "");
Expect(1, 8297, '\P{Bidi_C=N}', "");
Expect(0, 8297, '\P{^Bidi_C=N}', "");
Error('\p{Is_Bidi_Control=__F:=}');
Error('\P{Is_Bidi_Control=__F:=}');
Expect(1, 8298, '\p{Is_Bidi_Control=f}', "");
Expect(0, 8298, '\p{^Is_Bidi_Control=f}', "");
Expect(0, 8298, '\P{Is_Bidi_Control=f}', "");
Expect(1, 8298, '\P{^Is_Bidi_Control=f}', "");
Expect(0, 8297, '\p{Is_Bidi_Control=f}', "");
Expect(1, 8297, '\p{^Is_Bidi_Control=f}', "");
Expect(1, 8297, '\P{Is_Bidi_Control=f}', "");
Expect(0, 8297, '\P{^Is_Bidi_Control=f}', "");
Expect(1, 8298, '\p{Is_Bidi_Control: 	 F}', "");
Expect(0, 8298, '\p{^Is_Bidi_Control: 	 F}', "");
Expect(0, 8298, '\P{Is_Bidi_Control: 	 F}', "");
Expect(1, 8298, '\P{^Is_Bidi_Control: 	 F}', "");
Expect(0, 8297, '\p{Is_Bidi_Control: 	 F}', "");
Expect(1, 8297, '\p{^Is_Bidi_Control: 	 F}', "");
Expect(1, 8297, '\P{Is_Bidi_Control: 	 F}', "");
Expect(0, 8297, '\P{^Is_Bidi_Control: 	 F}', "");
Error('\p{Is_Bidi_C=:= False}');
Error('\P{Is_Bidi_C=:= False}');
Expect(1, 8298, '\p{Is_Bidi_C=false}', "");
Expect(0, 8298, '\p{^Is_Bidi_C=false}', "");
Expect(0, 8298, '\P{Is_Bidi_C=false}', "");
Expect(1, 8298, '\P{^Is_Bidi_C=false}', "");
Expect(0, 8297, '\p{Is_Bidi_C=false}', "");
Expect(1, 8297, '\p{^Is_Bidi_C=false}', "");
Expect(1, 8297, '\P{Is_Bidi_C=false}', "");
Expect(0, 8297, '\P{^Is_Bidi_C=false}', "");
Expect(1, 8298, '\p{Is_Bidi_C= -false}', "");
Expect(0, 8298, '\p{^Is_Bidi_C= -false}', "");
Expect(0, 8298, '\P{Is_Bidi_C= -false}', "");
Expect(1, 8298, '\P{^Is_Bidi_C= -false}', "");
Expect(0, 8297, '\p{Is_Bidi_C= -false}', "");
Expect(1, 8297, '\p{^Is_Bidi_C= -false}', "");
Expect(1, 8297, '\P{Is_Bidi_C= -false}', "");
Expect(0, 8297, '\P{^Is_Bidi_C= -false}', "");
Error('\p{Bidi_Control=:=--Yes}');
Error('\P{Bidi_Control=:=--Yes}');
Expect(1, 8297, '\p{Bidi_Control=yes}', "");
Expect(0, 8297, '\p{^Bidi_Control=yes}', "");
Expect(0, 8297, '\P{Bidi_Control=yes}', "");
Expect(1, 8297, '\P{^Bidi_Control=yes}', "");
Expect(0, 8298, '\p{Bidi_Control=yes}', "");
Expect(1, 8298, '\p{^Bidi_Control=yes}', "");
Expect(1, 8298, '\P{Bidi_Control=yes}', "");
Expect(0, 8298, '\P{^Bidi_Control=yes}', "");
Expect(1, 8297, '\p{Bidi_Control=-	Yes}', "");
Expect(0, 8297, '\p{^Bidi_Control=-	Yes}', "");
Expect(0, 8297, '\P{Bidi_Control=-	Yes}', "");
Expect(1, 8297, '\P{^Bidi_Control=-	Yes}', "");
Expect(0, 8298, '\p{Bidi_Control=-	Yes}', "");
Expect(1, 8298, '\p{^Bidi_Control=-	Yes}', "");
Expect(1, 8298, '\P{Bidi_Control=-	Yes}', "");
Expect(0, 8298, '\P{^Bidi_Control=-	Yes}', "");
Error('\p{Bidi_C=:=Y}');
Error('\P{Bidi_C=:=Y}');
Expect(1, 8297, '\p{Bidi_C=y}', "");
Expect(0, 8297, '\p{^Bidi_C=y}', "");
Expect(0, 8297, '\P{Bidi_C=y}', "");
Expect(1, 8297, '\P{^Bidi_C=y}', "");
Expect(0, 8298, '\p{Bidi_C=y}', "");
Expect(1, 8298, '\p{^Bidi_C=y}', "");
Expect(1, 8298, '\P{Bidi_C=y}', "");
Expect(0, 8298, '\P{^Bidi_C=y}', "");
Expect(1, 8297, '\p{Bidi_C=	Y}', "");
Expect(0, 8297, '\p{^Bidi_C=	Y}', "");
Expect(0, 8297, '\P{Bidi_C=	Y}', "");
Expect(1, 8297, '\P{^Bidi_C=	Y}', "");
Expect(0, 8298, '\p{Bidi_C=	Y}', "");
Expect(1, 8298, '\p{^Bidi_C=	Y}', "");
Expect(1, 8298, '\P{Bidi_C=	Y}', "");
Expect(0, 8298, '\P{^Bidi_C=	Y}', "");
Error('\p{Is_Bidi_Control=/a/-t}');
Error('\P{Is_Bidi_Control=/a/-t}');
Expect(1, 8297, '\p{Is_Bidi_Control=t}', "");
Expect(0, 8297, '\p{^Is_Bidi_Control=t}', "");
Expect(0, 8297, '\P{Is_Bidi_Control=t}', "");
Expect(1, 8297, '\P{^Is_Bidi_Control=t}', "");
Expect(0, 8298, '\p{Is_Bidi_Control=t}', "");
Expect(1, 8298, '\p{^Is_Bidi_Control=t}', "");
Expect(1, 8298, '\P{Is_Bidi_Control=t}', "");
Expect(0, 8298, '\P{^Is_Bidi_Control=t}', "");
Expect(1, 8297, '\p{Is_Bidi_Control= T}', "");
Expect(0, 8297, '\p{^Is_Bidi_Control= T}', "");
Expect(0, 8297, '\P{Is_Bidi_Control= T}', "");
Expect(1, 8297, '\P{^Is_Bidi_Control= T}', "");
Expect(0, 8298, '\p{Is_Bidi_Control= T}', "");
Expect(1, 8298, '\p{^Is_Bidi_Control= T}', "");
Expect(1, 8298, '\P{Is_Bidi_Control= T}', "");
Expect(0, 8298, '\P{^Is_Bidi_Control= T}', "");
Error('\p{Is_Bidi_C= 	True:=}');
Error('\P{Is_Bidi_C= 	True:=}');
Expect(1, 8297, '\p{Is_Bidi_C=true}', "");
Expect(0, 8297, '\p{^Is_Bidi_C=true}', "");
Expect(0, 8297, '\P{Is_Bidi_C=true}', "");
Expect(1, 8297, '\P{^Is_Bidi_C=true}', "");
Expect(0, 8298, '\p{Is_Bidi_C=true}', "");
Expect(1, 8298, '\p{^Is_Bidi_C=true}', "");
Expect(1, 8298, '\P{Is_Bidi_C=true}', "");
Expect(0, 8298, '\P{^Is_Bidi_C=true}', "");
Expect(1, 8297, '\p{Is_Bidi_C=  True}', "");
Expect(0, 8297, '\p{^Is_Bidi_C=  True}', "");
Expect(0, 8297, '\P{Is_Bidi_C=  True}', "");
Expect(1, 8297, '\P{^Is_Bidi_C=  True}', "");
Expect(0, 8298, '\p{Is_Bidi_C=  True}', "");
Expect(1, 8298, '\p{^Is_Bidi_C=  True}', "");
Expect(1, 8298, '\P{Is_Bidi_C=  True}', "");
Expect(0, 8298, '\P{^Is_Bidi_C=  True}', "");
Error('\p{Bidi_Mirrored=_NO:=}');
Error('\P{Bidi_Mirrored=_NO:=}');
Expect(1, 120772, '\p{Bidi_Mirrored:   no}', "");
Expect(0, 120772, '\p{^Bidi_Mirrored:   no}', "");
Expect(0, 120772, '\P{Bidi_Mirrored:   no}', "");
Expect(1, 120772, '\P{^Bidi_Mirrored:   no}', "");
Expect(0, 120771, '\p{Bidi_Mirrored:   no}', "");
Expect(1, 120771, '\p{^Bidi_Mirrored:   no}', "");
Expect(1, 120771, '\P{Bidi_Mirrored:   no}', "");
Expect(0, 120771, '\P{^Bidi_Mirrored:   no}', "");
Expect(1, 120772, '\p{Bidi_Mirrored=	-No}', "");
Expect(0, 120772, '\p{^Bidi_Mirrored=	-No}', "");
Expect(0, 120772, '\P{Bidi_Mirrored=	-No}', "");
Expect(1, 120772, '\P{^Bidi_Mirrored=	-No}', "");
Expect(0, 120771, '\p{Bidi_Mirrored=	-No}', "");
Expect(1, 120771, '\p{^Bidi_Mirrored=	-No}', "");
Expect(1, 120771, '\P{Bidi_Mirrored=	-No}', "");
Expect(0, 120771, '\P{^Bidi_Mirrored=	-No}', "");
Error('\p{Bidi_M=-N/a/}');
Error('\P{Bidi_M=-N/a/}');
Expect(1, 120772, '\p{Bidi_M=n}', "");
Expect(0, 120772, '\p{^Bidi_M=n}', "");
Expect(0, 120772, '\P{Bidi_M=n}', "");
Expect(1, 120772, '\P{^Bidi_M=n}', "");
Expect(0, 120771, '\p{Bidi_M=n}', "");
Expect(1, 120771, '\p{^Bidi_M=n}', "");
Expect(1, 120771, '\P{Bidi_M=n}', "");
Expect(0, 120771, '\P{^Bidi_M=n}', "");
Expect(1, 120772, '\p{Bidi_M=	_n}', "");
Expect(0, 120772, '\p{^Bidi_M=	_n}', "");
Expect(0, 120772, '\P{Bidi_M=	_n}', "");
Expect(1, 120772, '\P{^Bidi_M=	_n}', "");
Expect(0, 120771, '\p{Bidi_M=	_n}', "");
Expect(1, 120771, '\p{^Bidi_M=	_n}', "");
Expect(1, 120771, '\P{Bidi_M=	_n}', "");
Expect(0, 120771, '\P{^Bidi_M=	_n}', "");
Error('\p{Is_Bidi_Mirrored=/a/_ f}');
Error('\P{Is_Bidi_Mirrored=/a/_ f}');
Expect(1, 120772, '\p{Is_Bidi_Mirrored=f}', "");
Expect(0, 120772, '\p{^Is_Bidi_Mirrored=f}', "");
Expect(0, 120772, '\P{Is_Bidi_Mirrored=f}', "");
Expect(1, 120772, '\P{^Is_Bidi_Mirrored=f}', "");
Expect(0, 120771, '\p{Is_Bidi_Mirrored=f}', "");
Expect(1, 120771, '\p{^Is_Bidi_Mirrored=f}', "");
Expect(1, 120771, '\P{Is_Bidi_Mirrored=f}', "");
Expect(0, 120771, '\P{^Is_Bidi_Mirrored=f}', "");
Expect(1, 120772, '\p{Is_Bidi_Mirrored:	_	F}', "");
Expect(0, 120772, '\p{^Is_Bidi_Mirrored:	_	F}', "");
Expect(0, 120772, '\P{Is_Bidi_Mirrored:	_	F}', "");
Expect(1, 120772, '\P{^Is_Bidi_Mirrored:	_	F}', "");
Expect(0, 120771, '\p{Is_Bidi_Mirrored:	_	F}', "");
Expect(1, 120771, '\p{^Is_Bidi_Mirrored:	_	F}', "");
Expect(1, 120771, '\P{Is_Bidi_Mirrored:	_	F}', "");
Expect(0, 120771, '\P{^Is_Bidi_Mirrored:	_	F}', "");
Error('\p{Is_Bidi_M=-	False/a/}');
Error('\P{Is_Bidi_M=-	False/a/}');
Expect(1, 120772, '\p{Is_Bidi_M=false}', "");
Expect(0, 120772, '\p{^Is_Bidi_M=false}', "");
Expect(0, 120772, '\P{Is_Bidi_M=false}', "");
Expect(1, 120772, '\P{^Is_Bidi_M=false}', "");
Expect(0, 120771, '\p{Is_Bidi_M=false}', "");
Expect(1, 120771, '\p{^Is_Bidi_M=false}', "");
Expect(1, 120771, '\P{Is_Bidi_M=false}', "");
Expect(0, 120771, '\P{^Is_Bidi_M=false}', "");
Error('\p{Bidi_Mirrored=:= 	Yes}');
Error('\P{Bidi_Mirrored=:= 	Yes}');
Expect(1, 120771, '\p{Bidi_Mirrored=yes}', "");
Expect(0, 120771, '\p{^Bidi_Mirrored=yes}', "");
Expect(0, 120771, '\P{Bidi_Mirrored=yes}', "");
Expect(1, 120771, '\P{^Bidi_Mirrored=yes}', "");
Expect(0, 120772, '\p{Bidi_Mirrored=yes}', "");
Expect(1, 120772, '\p{^Bidi_Mirrored=yes}', "");
Expect(1, 120772, '\P{Bidi_Mirrored=yes}', "");
Expect(0, 120772, '\P{^Bidi_Mirrored=yes}', "");
Expect(1, 120771, '\p{Bidi_Mirrored=_ Yes}', "");
Expect(0, 120771, '\p{^Bidi_Mirrored=_ Yes}', "");
Expect(0, 120771, '\P{Bidi_Mirrored=_ Yes}', "");
Expect(1, 120771, '\P{^Bidi_Mirrored=_ Yes}', "");
Expect(0, 120772, '\p{Bidi_Mirrored=_ Yes}', "");
Expect(1, 120772, '\p{^Bidi_Mirrored=_ Yes}', "");
Expect(1, 120772, '\P{Bidi_Mirrored=_ Yes}', "");
Expect(0, 120772, '\P{^Bidi_Mirrored=_ Yes}', "");
Error('\p{Bidi_M=_Y/a/}');
Error('\P{Bidi_M=_Y/a/}');
Expect(1, 120771, '\p{Bidi_M=y}', "");
Expect(0, 120771, '\p{^Bidi_M=y}', "");
Expect(0, 120771, '\P{Bidi_M=y}', "");
Expect(1, 120771, '\P{^Bidi_M=y}', "");
Expect(0, 120772, '\p{Bidi_M=y}', "");
Expect(1, 120772, '\p{^Bidi_M=y}', "");
Expect(1, 120772, '\P{Bidi_M=y}', "");
Expect(0, 120772, '\P{^Bidi_M=y}', "");
Expect(1, 120771, '\p{Bidi_M= _y}', "");
Expect(0, 120771, '\p{^Bidi_M= _y}', "");
Expect(0, 120771, '\P{Bidi_M= _y}', "");
Expect(1, 120771, '\P{^Bidi_M= _y}', "");
Expect(0, 120772, '\p{Bidi_M= _y}', "");
Expect(1, 120772, '\p{^Bidi_M= _y}', "");
Expect(1, 120772, '\P{Bidi_M= _y}', "");
Expect(0, 120772, '\P{^Bidi_M= _y}', "");
Error('\p{Is_Bidi_Mirrored=:=_-T}');
Error('\P{Is_Bidi_Mirrored=:=_-T}');
Expect(1, 120771, '\p{Is_Bidi_Mirrored=t}', "");
Expect(0, 120771, '\p{^Is_Bidi_Mirrored=t}', "");
Expect(0, 120771, '\P{Is_Bidi_Mirrored=t}', "");
Expect(1, 120771, '\P{^Is_Bidi_Mirrored=t}', "");
Expect(0, 120772, '\p{Is_Bidi_Mirrored=t}', "");
Expect(1, 120772, '\p{^Is_Bidi_Mirrored=t}', "");
Expect(1, 120772, '\P{Is_Bidi_Mirrored=t}', "");
Expect(0, 120772, '\P{^Is_Bidi_Mirrored=t}', "");
Expect(1, 120771, '\p{Is_Bidi_Mirrored=_-T}', "");
Expect(0, 120771, '\p{^Is_Bidi_Mirrored=_-T}', "");
Expect(0, 120771, '\P{Is_Bidi_Mirrored=_-T}', "");
Expect(1, 120771, '\P{^Is_Bidi_Mirrored=_-T}', "");
Expect(0, 120772, '\p{Is_Bidi_Mirrored=_-T}', "");
Expect(1, 120772, '\p{^Is_Bidi_Mirrored=_-T}', "");
Expect(1, 120772, '\P{Is_Bidi_Mirrored=_-T}', "");
Expect(0, 120772, '\P{^Is_Bidi_Mirrored=_-T}', "");
Error('\p{Is_Bidi_M=:= _TRUE}');
Error('\P{Is_Bidi_M=:= _TRUE}');
Expect(1, 120771, '\p{Is_Bidi_M=true}', "");
Expect(0, 120771, '\p{^Is_Bidi_M=true}', "");
Expect(0, 120771, '\P{Is_Bidi_M=true}', "");
Expect(1, 120771, '\P{^Is_Bidi_M=true}', "");
Expect(0, 120772, '\p{Is_Bidi_M=true}', "");
Expect(1, 120772, '\p{^Is_Bidi_M=true}', "");
Expect(1, 120772, '\P{Is_Bidi_M=true}', "");
Expect(0, 120772, '\P{^Is_Bidi_M=true}', "");
Expect(1, 120771, '\p{Is_Bidi_M=-	True}', "");
Expect(0, 120771, '\p{^Is_Bidi_M=-	True}', "");
Expect(0, 120771, '\P{Is_Bidi_M=-	True}', "");
Expect(1, 120771, '\P{^Is_Bidi_M=-	True}', "");
Expect(0, 120772, '\p{Is_Bidi_M=-	True}', "");
Expect(1, 120772, '\p{^Is_Bidi_M=-	True}', "");
Expect(1, 120772, '\P{Is_Bidi_M=-	True}', "");
Expect(0, 120772, '\P{^Is_Bidi_M=-	True}', "");
Error('\p{Composition_Exclusion=	-No:=}');
Error('\P{Composition_Exclusion=	-No:=}');
Expect(1, 119233, '\p{Composition_Exclusion=no}', "");
Expect(0, 119233, '\p{^Composition_Exclusion=no}', "");
Expect(0, 119233, '\P{Composition_Exclusion=no}', "");
Expect(1, 119233, '\P{^Composition_Exclusion=no}', "");
Expect(0, 119232, '\p{Composition_Exclusion=no}', "");
Expect(1, 119232, '\p{^Composition_Exclusion=no}', "");
Expect(1, 119232, '\P{Composition_Exclusion=no}', "");
Expect(0, 119232, '\P{^Composition_Exclusion=no}', "");
Expect(1, 119233, '\p{Composition_Exclusion:_NO}', "");
Expect(0, 119233, '\p{^Composition_Exclusion:_NO}', "");
Expect(0, 119233, '\P{Composition_Exclusion:_NO}', "");
Expect(1, 119233, '\P{^Composition_Exclusion:_NO}', "");
Expect(0, 119232, '\p{Composition_Exclusion:_NO}', "");
Expect(1, 119232, '\p{^Composition_Exclusion:_NO}', "");
Expect(1, 119232, '\P{Composition_Exclusion:_NO}', "");
Expect(0, 119232, '\P{^Composition_Exclusion:_NO}', "");
Error('\p{CE=-_N:=}');
Error('\P{CE=-_N:=}');
Expect(1, 119233, '\p{CE=n}', "");
Expect(0, 119233, '\p{^CE=n}', "");
Expect(0, 119233, '\P{CE=n}', "");
Expect(1, 119233, '\P{^CE=n}', "");
Expect(0, 119232, '\p{CE=n}', "");
Expect(1, 119232, '\p{^CE=n}', "");
Expect(1, 119232, '\P{CE=n}', "");
Expect(0, 119232, '\P{^CE=n}', "");
Expect(1, 119233, '\p{CE= N}', "");
Expect(0, 119233, '\p{^CE= N}', "");
Expect(0, 119233, '\P{CE= N}', "");
Expect(1, 119233, '\P{^CE= N}', "");
Expect(0, 119232, '\p{CE= N}', "");
Expect(1, 119232, '\p{^CE= N}', "");
Expect(1, 119232, '\P{CE= N}', "");
Expect(0, 119232, '\P{^CE= N}', "");
Error('\p{Is_Composition_Exclusion=-F:=}');
Error('\P{Is_Composition_Exclusion=-F:=}');
Expect(1, 119233, '\p{Is_Composition_Exclusion=f}', "");
Expect(0, 119233, '\p{^Is_Composition_Exclusion=f}', "");
Expect(0, 119233, '\P{Is_Composition_Exclusion=f}', "");
Expect(1, 119233, '\P{^Is_Composition_Exclusion=f}', "");
Expect(0, 119232, '\p{Is_Composition_Exclusion=f}', "");
Expect(1, 119232, '\p{^Is_Composition_Exclusion=f}', "");
Expect(1, 119232, '\P{Is_Composition_Exclusion=f}', "");
Expect(0, 119232, '\P{^Is_Composition_Exclusion=f}', "");
Expect(1, 119233, '\p{Is_Composition_Exclusion=F}', "");
Expect(0, 119233, '\p{^Is_Composition_Exclusion=F}', "");
Expect(0, 119233, '\P{Is_Composition_Exclusion=F}', "");
Expect(1, 119233, '\P{^Is_Composition_Exclusion=F}', "");
Expect(0, 119232, '\p{Is_Composition_Exclusion=F}', "");
Expect(1, 119232, '\p{^Is_Composition_Exclusion=F}', "");
Expect(1, 119232, '\P{Is_Composition_Exclusion=F}', "");
Expect(0, 119232, '\P{^Is_Composition_Exclusion=F}', "");
Error('\p{Is_CE=:=  false}');
Error('\P{Is_CE=:=  false}');
Expect(1, 119233, '\p{Is_CE=false}', "");
Expect(0, 119233, '\p{^Is_CE=false}', "");
Expect(0, 119233, '\P{Is_CE=false}', "");
Expect(1, 119233, '\P{^Is_CE=false}', "");
Expect(0, 119232, '\p{Is_CE=false}', "");
Expect(1, 119232, '\p{^Is_CE=false}', "");
Expect(1, 119232, '\P{Is_CE=false}', "");
Expect(0, 119232, '\P{^Is_CE=false}', "");
Expect(1, 119233, '\p{Is_CE=False}', "");
Expect(0, 119233, '\p{^Is_CE=False}', "");
Expect(0, 119233, '\P{Is_CE=False}', "");
Expect(1, 119233, '\P{^Is_CE=False}', "");
Expect(0, 119232, '\p{Is_CE=False}', "");
Expect(1, 119232, '\p{^Is_CE=False}', "");
Expect(1, 119232, '\P{Is_CE=False}', "");
Expect(0, 119232, '\P{^Is_CE=False}', "");
Error('\p{Composition_Exclusion= :=yes}');
Error('\P{Composition_Exclusion= :=yes}');
Expect(1, 119232, '\p{Composition_Exclusion=yes}', "");
Expect(0, 119232, '\p{^Composition_Exclusion=yes}', "");
Expect(0, 119232, '\P{Composition_Exclusion=yes}', "");
Expect(1, 119232, '\P{^Composition_Exclusion=yes}', "");
Expect(0, 119233, '\p{Composition_Exclusion=yes}', "");
Expect(1, 119233, '\p{^Composition_Exclusion=yes}', "");
Expect(1, 119233, '\P{Composition_Exclusion=yes}', "");
Expect(0, 119233, '\P{^Composition_Exclusion=yes}', "");
Expect(1, 119232, '\p{Composition_Exclusion=-YES}', "");
Expect(0, 119232, '\p{^Composition_Exclusion=-YES}', "");
Expect(0, 119232, '\P{Composition_Exclusion=-YES}', "");
Expect(1, 119232, '\P{^Composition_Exclusion=-YES}', "");
Expect(0, 119233, '\p{Composition_Exclusion=-YES}', "");
Expect(1, 119233, '\p{^Composition_Exclusion=-YES}', "");
Expect(1, 119233, '\P{Composition_Exclusion=-YES}', "");
Expect(0, 119233, '\P{^Composition_Exclusion=-YES}', "");
Error('\p{CE=	_Y:=}');
Error('\P{CE=	_Y:=}');
Expect(1, 119232, '\p{CE=y}', "");
Expect(0, 119232, '\p{^CE=y}', "");
Expect(0, 119232, '\P{CE=y}', "");
Expect(1, 119232, '\P{^CE=y}', "");
Expect(0, 119233, '\p{CE=y}', "");
Expect(1, 119233, '\p{^CE=y}', "");
Expect(1, 119233, '\P{CE=y}', "");
Expect(0, 119233, '\P{^CE=y}', "");
Expect(1, 119232, '\p{CE=_-Y}', "");
Expect(0, 119232, '\p{^CE=_-Y}', "");
Expect(0, 119232, '\P{CE=_-Y}', "");
Expect(1, 119232, '\P{^CE=_-Y}', "");
Expect(0, 119233, '\p{CE=_-Y}', "");
Expect(1, 119233, '\p{^CE=_-Y}', "");
Expect(1, 119233, '\P{CE=_-Y}', "");
Expect(0, 119233, '\P{^CE=_-Y}', "");
Error('\p{Is_Composition_Exclusion:/a/T}');
Error('\P{Is_Composition_Exclusion:/a/T}');
Expect(1, 119232, '\p{Is_Composition_Exclusion=t}', "");
Expect(0, 119232, '\p{^Is_Composition_Exclusion=t}', "");
Expect(0, 119232, '\P{Is_Composition_Exclusion=t}', "");
Expect(1, 119232, '\P{^Is_Composition_Exclusion=t}', "");
Expect(0, 119233, '\p{Is_Composition_Exclusion=t}', "");
Expect(1, 119233, '\p{^Is_Composition_Exclusion=t}', "");
Expect(1, 119233, '\P{Is_Composition_Exclusion=t}', "");
Expect(0, 119233, '\P{^Is_Composition_Exclusion=t}', "");
Expect(1, 119232, '\p{Is_Composition_Exclusion=	_T}', "");
Expect(0, 119232, '\p{^Is_Composition_Exclusion=	_T}', "");
Expect(0, 119232, '\P{Is_Composition_Exclusion=	_T}', "");
Expect(1, 119232, '\P{^Is_Composition_Exclusion=	_T}', "");
Expect(0, 119233, '\p{Is_Composition_Exclusion=	_T}', "");
Expect(1, 119233, '\p{^Is_Composition_Exclusion=	_T}', "");
Expect(1, 119233, '\P{Is_Composition_Exclusion=	_T}', "");
Expect(0, 119233, '\P{^Is_Composition_Exclusion=	_T}', "");
Error('\p{Is_CE=:=-true}');
Error('\P{Is_CE=:=-true}');
Expect(1, 119232, '\p{Is_CE:true}', "");
Expect(0, 119232, '\p{^Is_CE:true}', "");
Expect(0, 119232, '\P{Is_CE:true}', "");
Expect(1, 119232, '\P{^Is_CE:true}', "");
Expect(0, 119233, '\p{Is_CE:true}', "");
Expect(1, 119233, '\p{^Is_CE:true}', "");
Expect(1, 119233, '\P{Is_CE:true}', "");
Expect(0, 119233, '\P{^Is_CE:true}', "");
Expect(1, 119232, '\p{Is_CE=  TRUE}', "");
Expect(0, 119232, '\p{^Is_CE=  TRUE}', "");
Expect(0, 119232, '\P{Is_CE=  TRUE}', "");
Expect(1, 119232, '\P{^Is_CE=  TRUE}', "");
Expect(0, 119233, '\p{Is_CE=  TRUE}', "");
Expect(1, 119233, '\p{^Is_CE=  TRUE}', "");
Expect(1, 119233, '\P{Is_CE=  TRUE}', "");
Expect(0, 119233, '\P{^Is_CE=  TRUE}', "");
Error('\p{Case_Ignorable=/a/	_No}');
Error('\P{Case_Ignorable=/a/	_No}');
Expect(1, 918000, '\p{Case_Ignorable=no}', "");
Expect(0, 918000, '\p{^Case_Ignorable=no}', "");
Expect(0, 918000, '\P{Case_Ignorable=no}', "");
Expect(1, 918000, '\P{^Case_Ignorable=no}', "");
Expect(0, 917999, '\p{Case_Ignorable=no}', "");
Expect(1, 917999, '\p{^Case_Ignorable=no}', "");
Expect(1, 917999, '\P{Case_Ignorable=no}', "");
Expect(0, 917999, '\P{^Case_Ignorable=no}', "");
Expect(1, 918000, '\p{Case_Ignorable=NO}', "");
Expect(0, 918000, '\p{^Case_Ignorable=NO}', "");
Expect(0, 918000, '\P{Case_Ignorable=NO}', "");
Expect(1, 918000, '\P{^Case_Ignorable=NO}', "");
Expect(0, 917999, '\p{Case_Ignorable=NO}', "");
Expect(1, 917999, '\p{^Case_Ignorable=NO}', "");
Expect(1, 917999, '\P{Case_Ignorable=NO}', "");
Expect(0, 917999, '\P{^Case_Ignorable=NO}', "");
Error('\p{CI=/a/_	N}');
Error('\P{CI=/a/_	N}');
Expect(1, 918000, '\p{CI=n}', "");
Expect(0, 918000, '\p{^CI=n}', "");
Expect(0, 918000, '\P{CI=n}', "");
Expect(1, 918000, '\P{^CI=n}', "");
Expect(0, 917999, '\p{CI=n}', "");
Expect(1, 917999, '\p{^CI=n}', "");
Expect(1, 917999, '\P{CI=n}', "");
Expect(0, 917999, '\P{^CI=n}', "");
Expect(1, 918000, '\p{CI=	N}', "");
Expect(0, 918000, '\p{^CI=	N}', "");
Expect(0, 918000, '\P{CI=	N}', "");
Expect(1, 918000, '\P{^CI=	N}', "");
Expect(0, 917999, '\p{CI=	N}', "");
Expect(1, 917999, '\p{^CI=	N}', "");
Expect(1, 917999, '\P{CI=	N}', "");
Expect(0, 917999, '\P{^CI=	N}', "");
Error('\p{Is_Case_Ignorable= 	F/a/}');
Error('\P{Is_Case_Ignorable= 	F/a/}');
Expect(1, 918000, '\p{Is_Case_Ignorable=f}', "");
Expect(0, 918000, '\p{^Is_Case_Ignorable=f}', "");
Expect(0, 918000, '\P{Is_Case_Ignorable=f}', "");
Expect(1, 918000, '\P{^Is_Case_Ignorable=f}', "");
Expect(0, 917999, '\p{Is_Case_Ignorable=f}', "");
Expect(1, 917999, '\p{^Is_Case_Ignorable=f}', "");
Expect(1, 917999, '\P{Is_Case_Ignorable=f}', "");
Expect(0, 917999, '\P{^Is_Case_Ignorable=f}', "");
Expect(1, 918000, '\p{Is_Case_Ignorable= -f}', "");
Expect(0, 918000, '\p{^Is_Case_Ignorable= -f}', "");
Expect(0, 918000, '\P{Is_Case_Ignorable= -f}', "");
Expect(1, 918000, '\P{^Is_Case_Ignorable= -f}', "");
Expect(0, 917999, '\p{Is_Case_Ignorable= -f}', "");
Expect(1, 917999, '\p{^Is_Case_Ignorable= -f}', "");
Expect(1, 917999, '\P{Is_Case_Ignorable= -f}', "");
Expect(0, 917999, '\P{^Is_Case_Ignorable= -f}', "");
Error('\p{Is_CI:	:=false}');
Error('\P{Is_CI:	:=false}');
Expect(1, 918000, '\p{Is_CI=false}', "");
Expect(0, 918000, '\p{^Is_CI=false}', "");
Expect(0, 918000, '\P{Is_CI=false}', "");
Expect(1, 918000, '\P{^Is_CI=false}', "");
Expect(0, 917999, '\p{Is_CI=false}', "");
Expect(1, 917999, '\p{^Is_CI=false}', "");
Expect(1, 917999, '\P{Is_CI=false}', "");
Expect(0, 917999, '\P{^Is_CI=false}', "");
Expect(1, 918000, '\p{Is_CI:_ FALSE}', "");
Expect(0, 918000, '\p{^Is_CI:_ FALSE}', "");
Expect(0, 918000, '\P{Is_CI:_ FALSE}', "");
Expect(1, 918000, '\P{^Is_CI:_ FALSE}', "");
Expect(0, 917999, '\p{Is_CI:_ FALSE}', "");
Expect(1, 917999, '\p{^Is_CI:_ FALSE}', "");
Expect(1, 917999, '\P{Is_CI:_ FALSE}', "");
Expect(0, 917999, '\P{^Is_CI:_ FALSE}', "");
Error('\p{Case_Ignorable:/a/ -yes}');
Error('\P{Case_Ignorable:/a/ -yes}');
Expect(1, 917999, '\p{Case_Ignorable:   yes}', "");
Expect(0, 917999, '\p{^Case_Ignorable:   yes}', "");
Expect(0, 917999, '\P{Case_Ignorable:   yes}', "");
Expect(1, 917999, '\P{^Case_Ignorable:   yes}', "");
Expect(0, 918000, '\p{Case_Ignorable:   yes}', "");
Expect(1, 918000, '\p{^Case_Ignorable:   yes}', "");
Expect(1, 918000, '\P{Case_Ignorable:   yes}', "");
Expect(0, 918000, '\P{^Case_Ignorable:   yes}', "");
Expect(1, 917999, '\p{Case_Ignorable:	_	Yes}', "");
Expect(0, 917999, '\p{^Case_Ignorable:	_	Yes}', "");
Expect(0, 917999, '\P{Case_Ignorable:	_	Yes}', "");
Expect(1, 917999, '\P{^Case_Ignorable:	_	Yes}', "");
Expect(0, 918000, '\p{Case_Ignorable:	_	Yes}', "");
Expect(1, 918000, '\p{^Case_Ignorable:	_	Yes}', "");
Expect(1, 918000, '\P{Case_Ignorable:	_	Yes}', "");
Expect(0, 918000, '\P{^Case_Ignorable:	_	Yes}', "");
Error('\p{CI=	:=Y}');
Error('\P{CI=	:=Y}');
Expect(1, 917999, '\p{CI=y}', "");
Expect(0, 917999, '\p{^CI=y}', "");
Expect(0, 917999, '\P{CI=y}', "");
Expect(1, 917999, '\P{^CI=y}', "");
Expect(0, 918000, '\p{CI=y}', "");
Expect(1, 918000, '\p{^CI=y}', "");
Expect(1, 918000, '\P{CI=y}', "");
Expect(0, 918000, '\P{^CI=y}', "");
Expect(1, 917999, '\p{CI= _Y}', "");
Expect(0, 917999, '\p{^CI= _Y}', "");
Expect(0, 917999, '\P{CI= _Y}', "");
Expect(1, 917999, '\P{^CI= _Y}', "");
Expect(0, 918000, '\p{CI= _Y}', "");
Expect(1, 918000, '\p{^CI= _Y}', "");
Expect(1, 918000, '\P{CI= _Y}', "");
Expect(0, 918000, '\P{^CI= _Y}', "");
Error('\p{Is_Case_Ignorable=_T:=}');
Error('\P{Is_Case_Ignorable=_T:=}');
Expect(1, 917999, '\p{Is_Case_Ignorable=t}', "");
Expect(0, 917999, '\p{^Is_Case_Ignorable=t}', "");
Expect(0, 917999, '\P{Is_Case_Ignorable=t}', "");
Expect(1, 917999, '\P{^Is_Case_Ignorable=t}', "");
Expect(0, 918000, '\p{Is_Case_Ignorable=t}', "");
Expect(1, 918000, '\p{^Is_Case_Ignorable=t}', "");
Expect(1, 918000, '\P{Is_Case_Ignorable=t}', "");
Expect(0, 918000, '\P{^Is_Case_Ignorable=t}', "");
Expect(1, 917999, '\p{Is_Case_Ignorable=T}', "");
Expect(0, 917999, '\p{^Is_Case_Ignorable=T}', "");
Expect(0, 917999, '\P{Is_Case_Ignorable=T}', "");
Expect(1, 917999, '\P{^Is_Case_Ignorable=T}', "");
Expect(0, 918000, '\p{Is_Case_Ignorable=T}', "");
Expect(1, 918000, '\p{^Is_Case_Ignorable=T}', "");
Expect(1, 918000, '\P{Is_Case_Ignorable=T}', "");
Expect(0, 918000, '\P{^Is_Case_Ignorable=T}', "");
Error('\p{Is_CI=/a/--true}');
Error('\P{Is_CI=/a/--true}');
Expect(1, 917999, '\p{Is_CI=true}', "");
Expect(0, 917999, '\p{^Is_CI=true}', "");
Expect(0, 917999, '\P{Is_CI=true}', "");
Expect(1, 917999, '\P{^Is_CI=true}', "");
Expect(0, 918000, '\p{Is_CI=true}', "");
Expect(1, 918000, '\p{^Is_CI=true}', "");
Expect(1, 918000, '\P{Is_CI=true}', "");
Expect(0, 918000, '\P{^Is_CI=true}', "");
Expect(1, 917999, '\p{Is_CI=	 True}', "");
Expect(0, 917999, '\p{^Is_CI=	 True}', "");
Expect(0, 917999, '\P{Is_CI=	 True}', "");
Expect(1, 917999, '\P{^Is_CI=	 True}', "");
Expect(0, 918000, '\p{Is_CI=	 True}', "");
Expect(1, 918000, '\p{^Is_CI=	 True}', "");
Expect(1, 918000, '\P{Is_CI=	 True}', "");
Expect(0, 918000, '\P{^Is_CI=	 True}', "");
Error('\p{Changes_When_Casefolded=	No:=}');
Error('\P{Changes_When_Casefolded=	No:=}');
Expect(1, 66600, '\p{Changes_When_Casefolded=no}', "");
Expect(0, 66600, '\p{^Changes_When_Casefolded=no}', "");
Expect(0, 66600, '\P{Changes_When_Casefolded=no}', "");
Expect(1, 66600, '\P{^Changes_When_Casefolded=no}', "");
Expect(0, 66599, '\p{Changes_When_Casefolded=no}', "");
Expect(1, 66599, '\p{^Changes_When_Casefolded=no}', "");
Expect(1, 66599, '\P{Changes_When_Casefolded=no}', "");
Expect(0, 66599, '\P{^Changes_When_Casefolded=no}', "");
Expect(1, 66600, '\p{Changes_When_Casefolded= _No}', "");
Expect(0, 66600, '\p{^Changes_When_Casefolded= _No}', "");
Expect(0, 66600, '\P{Changes_When_Casefolded= _No}', "");
Expect(1, 66600, '\P{^Changes_When_Casefolded= _No}', "");
Expect(0, 66599, '\p{Changes_When_Casefolded= _No}', "");
Expect(1, 66599, '\p{^Changes_When_Casefolded= _No}', "");
Expect(1, 66599, '\P{Changes_When_Casefolded= _No}', "");
Expect(0, 66599, '\P{^Changes_When_Casefolded= _No}', "");
Error('\p{CWCF:	n/a/}');
Error('\P{CWCF:	n/a/}');
Expect(1, 66600, '\p{CWCF=n}', "");
Expect(0, 66600, '\p{^CWCF=n}', "");
Expect(0, 66600, '\P{CWCF=n}', "");
Expect(1, 66600, '\P{^CWCF=n}', "");
Expect(0, 66599, '\p{CWCF=n}', "");
Expect(1, 66599, '\p{^CWCF=n}', "");
Expect(1, 66599, '\P{CWCF=n}', "");
Expect(0, 66599, '\P{^CWCF=n}', "");
Expect(1, 66600, '\p{CWCF=_-N}', "");
Expect(0, 66600, '\p{^CWCF=_-N}', "");
Expect(0, 66600, '\P{CWCF=_-N}', "");
Expect(1, 66600, '\P{^CWCF=_-N}', "");
Expect(0, 66599, '\p{CWCF=_-N}', "");
Expect(1, 66599, '\p{^CWCF=_-N}', "");
Expect(1, 66599, '\P{CWCF=_-N}', "");
Expect(0, 66599, '\P{^CWCF=_-N}', "");
Error('\p{Is_Changes_When_Casefolded=/a/_f}');
Error('\P{Is_Changes_When_Casefolded=/a/_f}');
Expect(1, 66600, '\p{Is_Changes_When_Casefolded=f}', "");
Expect(0, 66600, '\p{^Is_Changes_When_Casefolded=f}', "");
Expect(0, 66600, '\P{Is_Changes_When_Casefolded=f}', "");
Expect(1, 66600, '\P{^Is_Changes_When_Casefolded=f}', "");
Expect(0, 66599, '\p{Is_Changes_When_Casefolded=f}', "");
Expect(1, 66599, '\p{^Is_Changes_When_Casefolded=f}', "");
Expect(1, 66599, '\P{Is_Changes_When_Casefolded=f}', "");
Expect(0, 66599, '\P{^Is_Changes_When_Casefolded=f}', "");
Expect(1, 66600, '\p{Is_Changes_When_Casefolded=_-F}', "");
Expect(0, 66600, '\p{^Is_Changes_When_Casefolded=_-F}', "");
Expect(0, 66600, '\P{Is_Changes_When_Casefolded=_-F}', "");
Expect(1, 66600, '\P{^Is_Changes_When_Casefolded=_-F}', "");
Expect(0, 66599, '\p{Is_Changes_When_Casefolded=_-F}', "");
Expect(1, 66599, '\p{^Is_Changes_When_Casefolded=_-F}', "");
Expect(1, 66599, '\P{Is_Changes_When_Casefolded=_-F}', "");
Expect(0, 66599, '\P{^Is_Changes_When_Casefolded=_-F}', "");
Error('\p{Is_CWCF=	/a/False}');
Error('\P{Is_CWCF=	/a/False}');
Expect(1, 66600, '\p{Is_CWCF=false}', "");
Expect(0, 66600, '\p{^Is_CWCF=false}', "");
Expect(0, 66600, '\P{Is_CWCF=false}', "");
Expect(1, 66600, '\P{^Is_CWCF=false}', "");
Expect(0, 66599, '\p{Is_CWCF=false}', "");
Expect(1, 66599, '\p{^Is_CWCF=false}', "");
Expect(1, 66599, '\P{Is_CWCF=false}', "");
Expect(0, 66599, '\P{^Is_CWCF=false}', "");
Expect(1, 66600, '\p{Is_CWCF=--FALSE}', "");
Expect(0, 66600, '\p{^Is_CWCF=--FALSE}', "");
Expect(0, 66600, '\P{Is_CWCF=--FALSE}', "");
Expect(1, 66600, '\P{^Is_CWCF=--FALSE}', "");
Expect(0, 66599, '\p{Is_CWCF=--FALSE}', "");
Expect(1, 66599, '\p{^Is_CWCF=--FALSE}', "");
Expect(1, 66599, '\P{Is_CWCF=--FALSE}', "");
Expect(0, 66599, '\P{^Is_CWCF=--FALSE}', "");
Error('\p{Changes_When_Casefolded=_YES:=}');
Error('\P{Changes_When_Casefolded=_YES:=}');
Expect(1, 66599, '\p{Changes_When_Casefolded=yes}', "");
Expect(0, 66599, '\p{^Changes_When_Casefolded=yes}', "");
Expect(0, 66599, '\P{Changes_When_Casefolded=yes}', "");
Expect(1, 66599, '\P{^Changes_When_Casefolded=yes}', "");
Expect(0, 66600, '\p{Changes_When_Casefolded=yes}', "");
Expect(1, 66600, '\p{^Changes_When_Casefolded=yes}', "");
Expect(1, 66600, '\P{Changes_When_Casefolded=yes}', "");
Expect(0, 66600, '\P{^Changes_When_Casefolded=yes}', "");
Expect(1, 66599, '\p{Changes_When_Casefolded= _Yes}', "");
Expect(0, 66599, '\p{^Changes_When_Casefolded= _Yes}', "");
Expect(0, 66599, '\P{Changes_When_Casefolded= _Yes}', "");
Expect(1, 66599, '\P{^Changes_When_Casefolded= _Yes}', "");
Expect(0, 66600, '\p{Changes_When_Casefolded= _Yes}', "");
Expect(1, 66600, '\p{^Changes_When_Casefolded= _Yes}', "");
Expect(1, 66600, '\P{Changes_When_Casefolded= _Yes}', "");
Expect(0, 66600, '\P{^Changes_When_Casefolded= _Yes}', "");
Error('\p{CWCF= /a/Y}');
Error('\P{CWCF= /a/Y}');
Expect(1, 66599, '\p{CWCF=y}', "");
Expect(0, 66599, '\p{^CWCF=y}', "");
Expect(0, 66599, '\P{CWCF=y}', "");
Expect(1, 66599, '\P{^CWCF=y}', "");
Expect(0, 66600, '\p{CWCF=y}', "");
Expect(1, 66600, '\p{^CWCF=y}', "");
Expect(1, 66600, '\P{CWCF=y}', "");
Expect(0, 66600, '\P{^CWCF=y}', "");
Expect(1, 66599, '\p{CWCF= 	Y}', "");
Expect(0, 66599, '\p{^CWCF= 	Y}', "");
Expect(0, 66599, '\P{CWCF= 	Y}', "");
Expect(1, 66599, '\P{^CWCF= 	Y}', "");
Expect(0, 66600, '\p{CWCF= 	Y}', "");
Expect(1, 66600, '\p{^CWCF= 	Y}', "");
Expect(1, 66600, '\P{CWCF= 	Y}', "");
Expect(0, 66600, '\P{^CWCF= 	Y}', "");
Error('\p{Is_Changes_When_Casefolded=_-T/a/}');
Error('\P{Is_Changes_When_Casefolded=_-T/a/}');
Expect(1, 66599, '\p{Is_Changes_When_Casefolded=t}', "");
Expect(0, 66599, '\p{^Is_Changes_When_Casefolded=t}', "");
Expect(0, 66599, '\P{Is_Changes_When_Casefolded=t}', "");
Expect(1, 66599, '\P{^Is_Changes_When_Casefolded=t}', "");
Expect(0, 66600, '\p{Is_Changes_When_Casefolded=t}', "");
Expect(1, 66600, '\p{^Is_Changes_When_Casefolded=t}', "");
Expect(1, 66600, '\P{Is_Changes_When_Casefolded=t}', "");
Expect(0, 66600, '\P{^Is_Changes_When_Casefolded=t}', "");
Expect(1, 66599, '\p{Is_Changes_When_Casefolded=-T}', "");
Expect(0, 66599, '\p{^Is_Changes_When_Casefolded=-T}', "");
Expect(0, 66599, '\P{Is_Changes_When_Casefolded=-T}', "");
Expect(1, 66599, '\P{^Is_Changes_When_Casefolded=-T}', "");
Expect(0, 66600, '\p{Is_Changes_When_Casefolded=-T}', "");
Expect(1, 66600, '\p{^Is_Changes_When_Casefolded=-T}', "");
Expect(1, 66600, '\P{Is_Changes_When_Casefolded=-T}', "");
Expect(0, 66600, '\P{^Is_Changes_When_Casefolded=-T}', "");
Error('\p{Is_CWCF=	True/a/}');
Error('\P{Is_CWCF=	True/a/}');
Expect(1, 66599, '\p{Is_CWCF=true}', "");
Expect(0, 66599, '\p{^Is_CWCF=true}', "");
Expect(0, 66599, '\P{Is_CWCF=true}', "");
Expect(1, 66599, '\P{^Is_CWCF=true}', "");
Expect(0, 66600, '\p{Is_CWCF=true}', "");
Expect(1, 66600, '\p{^Is_CWCF=true}', "");
Expect(1, 66600, '\P{Is_CWCF=true}', "");
Expect(0, 66600, '\P{^Is_CWCF=true}', "");
Expect(1, 66599, '\p{Is_CWCF= True}', "");
Expect(0, 66599, '\p{^Is_CWCF= True}', "");
Expect(0, 66599, '\P{Is_CWCF= True}', "");
Expect(1, 66599, '\P{^Is_CWCF= True}', "");
Expect(0, 66600, '\p{Is_CWCF= True}', "");
Expect(1, 66600, '\p{^Is_CWCF= True}', "");
Expect(1, 66600, '\P{Is_CWCF= True}', "");
Expect(0, 66600, '\P{^Is_CWCF= True}', "");
Error('\p{Changes_When_Casemapped=	 no/a/}');
Error('\P{Changes_When_Casemapped=	 no/a/}');
Expect(1, 66640, '\p{Changes_When_Casemapped=no}', "");
Expect(0, 66640, '\p{^Changes_When_Casemapped=no}', "");
Expect(0, 66640, '\P{Changes_When_Casemapped=no}', "");
Expect(1, 66640, '\P{^Changes_When_Casemapped=no}', "");
Expect(0, 66639, '\p{Changes_When_Casemapped=no}', "");
Expect(1, 66639, '\p{^Changes_When_Casemapped=no}', "");
Expect(1, 66639, '\P{Changes_When_Casemapped=no}', "");
Expect(0, 66639, '\P{^Changes_When_Casemapped=no}', "");
Expect(1, 66640, '\p{Changes_When_Casemapped:   	_No}', "");
Expect(0, 66640, '\p{^Changes_When_Casemapped:   	_No}', "");
Expect(0, 66640, '\P{Changes_When_Casemapped:   	_No}', "");
Expect(1, 66640, '\P{^Changes_When_Casemapped:   	_No}', "");
Expect(0, 66639, '\p{Changes_When_Casemapped:   	_No}', "");
Expect(1, 66639, '\p{^Changes_When_Casemapped:   	_No}', "");
Expect(1, 66639, '\P{Changes_When_Casemapped:   	_No}', "");
Expect(0, 66639, '\P{^Changes_When_Casemapped:   	_No}', "");
Error('\p{CWCM=n:=}');
Error('\P{CWCM=n:=}');
Expect(1, 66640, '\p{CWCM=n}', "");
Expect(0, 66640, '\p{^CWCM=n}', "");
Expect(0, 66640, '\P{CWCM=n}', "");
Expect(1, 66640, '\P{^CWCM=n}', "");
Expect(0, 66639, '\p{CWCM=n}', "");
Expect(1, 66639, '\p{^CWCM=n}', "");
Expect(1, 66639, '\P{CWCM=n}', "");
Expect(0, 66639, '\P{^CWCM=n}', "");
Expect(1, 66640, '\p{CWCM=_N}', "");
Expect(0, 66640, '\p{^CWCM=_N}', "");
Expect(0, 66640, '\P{CWCM=_N}', "");
Expect(1, 66640, '\P{^CWCM=_N}', "");
Expect(0, 66639, '\p{CWCM=_N}', "");
Expect(1, 66639, '\p{^CWCM=_N}', "");
Expect(1, 66639, '\P{CWCM=_N}', "");
Expect(0, 66639, '\P{^CWCM=_N}', "");
Error('\p{Is_Changes_When_Casemapped:   /a/-_F}');
Error('\P{Is_Changes_When_Casemapped:   /a/-_F}');
Expect(1, 66640, '\p{Is_Changes_When_Casemapped=f}', "");
Expect(0, 66640, '\p{^Is_Changes_When_Casemapped=f}', "");
Expect(0, 66640, '\P{Is_Changes_When_Casemapped=f}', "");
Expect(1, 66640, '\P{^Is_Changes_When_Casemapped=f}', "");
Expect(0, 66639, '\p{Is_Changes_When_Casemapped=f}', "");
Expect(1, 66639, '\p{^Is_Changes_When_Casemapped=f}', "");
Expect(1, 66639, '\P{Is_Changes_When_Casemapped=f}', "");
Expect(0, 66639, '\P{^Is_Changes_When_Casemapped=f}', "");
Expect(1, 66640, '\p{Is_Changes_When_Casemapped=		F}', "");
Expect(0, 66640, '\p{^Is_Changes_When_Casemapped=		F}', "");
Expect(0, 66640, '\P{Is_Changes_When_Casemapped=		F}', "");
Expect(1, 66640, '\P{^Is_Changes_When_Casemapped=		F}', "");
Expect(0, 66639, '\p{Is_Changes_When_Casemapped=		F}', "");
Expect(1, 66639, '\p{^Is_Changes_When_Casemapped=		F}', "");
Expect(1, 66639, '\P{Is_Changes_When_Casemapped=		F}', "");
Expect(0, 66639, '\P{^Is_Changes_When_Casemapped=		F}', "");
Error('\p{Is_CWCM=/a/--false}');
Error('\P{Is_CWCM=/a/--false}');
Expect(1, 66640, '\p{Is_CWCM=false}', "");
Expect(0, 66640, '\p{^Is_CWCM=false}', "");
Expect(0, 66640, '\P{Is_CWCM=false}', "");
Expect(1, 66640, '\P{^Is_CWCM=false}', "");
Expect(0, 66639, '\p{Is_CWCM=false}', "");
Expect(1, 66639, '\p{^Is_CWCM=false}', "");
Expect(1, 66639, '\P{Is_CWCM=false}', "");
Expect(0, 66639, '\P{^Is_CWCM=false}', "");
Expect(1, 66640, '\p{Is_CWCM=_false}', "");
Expect(0, 66640, '\p{^Is_CWCM=_false}', "");
Expect(0, 66640, '\P{Is_CWCM=_false}', "");
Expect(1, 66640, '\P{^Is_CWCM=_false}', "");
Expect(0, 66639, '\p{Is_CWCM=_false}', "");
Expect(1, 66639, '\p{^Is_CWCM=_false}', "");
Expect(1, 66639, '\P{Is_CWCM=_false}', "");
Expect(0, 66639, '\P{^Is_CWCM=_false}', "");
Error('\p{Changes_When_Casemapped=--YES/a/}');
Error('\P{Changes_When_Casemapped=--YES/a/}');
Expect(1, 66639, '\p{Changes_When_Casemapped=yes}', "");
Expect(0, 66639, '\p{^Changes_When_Casemapped=yes}', "");
Expect(0, 66639, '\P{Changes_When_Casemapped=yes}', "");
Expect(1, 66639, '\P{^Changes_When_Casemapped=yes}', "");
Expect(0, 66640, '\p{Changes_When_Casemapped=yes}', "");
Expect(1, 66640, '\p{^Changes_When_Casemapped=yes}', "");
Expect(1, 66640, '\P{Changes_When_Casemapped=yes}', "");
Expect(0, 66640, '\P{^Changes_When_Casemapped=yes}', "");
Expect(1, 66639, '\p{Changes_When_Casemapped=	_YES}', "");
Expect(0, 66639, '\p{^Changes_When_Casemapped=	_YES}', "");
Expect(0, 66639, '\P{Changes_When_Casemapped=	_YES}', "");
Expect(1, 66639, '\P{^Changes_When_Casemapped=	_YES}', "");
Expect(0, 66640, '\p{Changes_When_Casemapped=	_YES}', "");
Expect(1, 66640, '\p{^Changes_When_Casemapped=	_YES}', "");
Expect(1, 66640, '\P{Changes_When_Casemapped=	_YES}', "");
Expect(0, 66640, '\P{^Changes_When_Casemapped=	_YES}', "");
Error('\p{CWCM=:= _Y}');
Error('\P{CWCM=:= _Y}');
Expect(1, 66639, '\p{CWCM=y}', "");
Expect(0, 66639, '\p{^CWCM=y}', "");
Expect(0, 66639, '\P{CWCM=y}', "");
Expect(1, 66639, '\P{^CWCM=y}', "");
Expect(0, 66640, '\p{CWCM=y}', "");
Expect(1, 66640, '\p{^CWCM=y}', "");
Expect(1, 66640, '\P{CWCM=y}', "");
Expect(0, 66640, '\P{^CWCM=y}', "");
Expect(1, 66639, '\p{CWCM=- y}', "");
Expect(0, 66639, '\p{^CWCM=- y}', "");
Expect(0, 66639, '\P{CWCM=- y}', "");
Expect(1, 66639, '\P{^CWCM=- y}', "");
Expect(0, 66640, '\p{CWCM=- y}', "");
Expect(1, 66640, '\p{^CWCM=- y}', "");
Expect(1, 66640, '\P{CWCM=- y}', "");
Expect(0, 66640, '\P{^CWCM=- y}', "");
Error('\p{Is_Changes_When_Casemapped=:=_T}');
Error('\P{Is_Changes_When_Casemapped=:=_T}');
Expect(1, 66639, '\p{Is_Changes_When_Casemapped: t}', "");
Expect(0, 66639, '\p{^Is_Changes_When_Casemapped: t}', "");
Expect(0, 66639, '\P{Is_Changes_When_Casemapped: t}', "");
Expect(1, 66639, '\P{^Is_Changes_When_Casemapped: t}', "");
Expect(0, 66640, '\p{Is_Changes_When_Casemapped: t}', "");
Expect(1, 66640, '\p{^Is_Changes_When_Casemapped: t}', "");
Expect(1, 66640, '\P{Is_Changes_When_Casemapped: t}', "");
Expect(0, 66640, '\P{^Is_Changes_When_Casemapped: t}', "");
Expect(1, 66639, '\p{Is_Changes_When_Casemapped:- T}', "");
Expect(0, 66639, '\p{^Is_Changes_When_Casemapped:- T}', "");
Expect(0, 66639, '\P{Is_Changes_When_Casemapped:- T}', "");
Expect(1, 66639, '\P{^Is_Changes_When_Casemapped:- T}', "");
Expect(0, 66640, '\p{Is_Changes_When_Casemapped:- T}', "");
Expect(1, 66640, '\p{^Is_Changes_When_Casemapped:- T}', "");
Expect(1, 66640, '\P{Is_Changes_When_Casemapped:- T}', "");
Expect(0, 66640, '\P{^Is_Changes_When_Casemapped:- T}', "");
Error('\p{Is_CWCM=	/a/true}');
Error('\P{Is_CWCM=	/a/true}');
Expect(1, 66639, '\p{Is_CWCM=true}', "");
Expect(0, 66639, '\p{^Is_CWCM=true}', "");
Expect(0, 66639, '\P{Is_CWCM=true}', "");
Expect(1, 66639, '\P{^Is_CWCM=true}', "");
Expect(0, 66640, '\p{Is_CWCM=true}', "");
Expect(1, 66640, '\p{^Is_CWCM=true}', "");
Expect(1, 66640, '\P{Is_CWCM=true}', "");
Expect(0, 66640, '\P{^Is_CWCM=true}', "");
Expect(1, 66639, '\p{Is_CWCM=	-TRUE}', "");
Expect(0, 66639, '\p{^Is_CWCM=	-TRUE}', "");
Expect(0, 66639, '\P{Is_CWCM=	-TRUE}', "");
Expect(1, 66639, '\P{^Is_CWCM=	-TRUE}', "");
Expect(0, 66640, '\p{Is_CWCM=	-TRUE}', "");
Expect(1, 66640, '\p{^Is_CWCM=	-TRUE}', "");
Expect(1, 66640, '\P{Is_CWCM=	-TRUE}', "");
Expect(0, 66640, '\P{^Is_CWCM=	-TRUE}', "");
Error('\p{Changes_When_NFKC_Casefolded=:=	NO}');
Error('\P{Changes_When_NFKC_Casefolded=:=	NO}');
Expect(1, 921600, '\p{Changes_When_NFKC_Casefolded:   no}', "");
Expect(0, 921600, '\p{^Changes_When_NFKC_Casefolded:   no}', "");
Expect(0, 921600, '\P{Changes_When_NFKC_Casefolded:   no}', "");
Expect(1, 921600, '\P{^Changes_When_NFKC_Casefolded:   no}', "");
Expect(0, 921599, '\p{Changes_When_NFKC_Casefolded:   no}', "");
Expect(1, 921599, '\p{^Changes_When_NFKC_Casefolded:   no}', "");
Expect(1, 921599, '\P{Changes_When_NFKC_Casefolded:   no}', "");
Expect(0, 921599, '\P{^Changes_When_NFKC_Casefolded:   no}', "");
Expect(1, 921600, '\p{Changes_When_NFKC_Casefolded= 	No}', "");
Expect(0, 921600, '\p{^Changes_When_NFKC_Casefolded= 	No}', "");
Expect(0, 921600, '\P{Changes_When_NFKC_Casefolded= 	No}', "");
Expect(1, 921600, '\P{^Changes_When_NFKC_Casefolded= 	No}', "");
Expect(0, 921599, '\p{Changes_When_NFKC_Casefolded= 	No}', "");
Expect(1, 921599, '\p{^Changes_When_NFKC_Casefolded= 	No}', "");
Expect(1, 921599, '\P{Changes_When_NFKC_Casefolded= 	No}', "");
Expect(0, 921599, '\P{^Changes_When_NFKC_Casefolded= 	No}', "");
Error('\p{CWKCF:	 /a/N}');
Error('\P{CWKCF:	 /a/N}');
Expect(1, 921600, '\p{CWKCF=n}', "");
Expect(0, 921600, '\p{^CWKCF=n}', "");
Expect(0, 921600, '\P{CWKCF=n}', "");
Expect(1, 921600, '\P{^CWKCF=n}', "");
Expect(0, 921599, '\p{CWKCF=n}', "");
Expect(1, 921599, '\p{^CWKCF=n}', "");
Expect(1, 921599, '\P{CWKCF=n}', "");
Expect(0, 921599, '\P{^CWKCF=n}', "");
Expect(1, 921600, '\p{CWKCF=		N}', "");
Expect(0, 921600, '\p{^CWKCF=		N}', "");
Expect(0, 921600, '\P{CWKCF=		N}', "");
Expect(1, 921600, '\P{^CWKCF=		N}', "");
Expect(0, 921599, '\p{CWKCF=		N}', "");
Expect(1, 921599, '\p{^CWKCF=		N}', "");
Expect(1, 921599, '\P{CWKCF=		N}', "");
Expect(0, 921599, '\P{^CWKCF=		N}', "");
Error('\p{Is_Changes_When_NFKC_Casefolded:   _ f:=}');
Error('\P{Is_Changes_When_NFKC_Casefolded:   _ f:=}');
Expect(1, 921600, '\p{Is_Changes_When_NFKC_Casefolded:	f}', "");
Expect(0, 921600, '\p{^Is_Changes_When_NFKC_Casefolded:	f}', "");
Expect(0, 921600, '\P{Is_Changes_When_NFKC_Casefolded:	f}', "");
Expect(1, 921600, '\P{^Is_Changes_When_NFKC_Casefolded:	f}', "");
Expect(0, 921599, '\p{Is_Changes_When_NFKC_Casefolded:	f}', "");
Expect(1, 921599, '\p{^Is_Changes_When_NFKC_Casefolded:	f}', "");
Expect(1, 921599, '\P{Is_Changes_When_NFKC_Casefolded:	f}', "");
Expect(0, 921599, '\P{^Is_Changes_When_NFKC_Casefolded:	f}', "");
Expect(1, 921600, '\p{Is_Changes_When_NFKC_Casefolded=_	f}', "");
Expect(0, 921600, '\p{^Is_Changes_When_NFKC_Casefolded=_	f}', "");
Expect(0, 921600, '\P{Is_Changes_When_NFKC_Casefolded=_	f}', "");
Expect(1, 921600, '\P{^Is_Changes_When_NFKC_Casefolded=_	f}', "");
Expect(0, 921599, '\p{Is_Changes_When_NFKC_Casefolded=_	f}', "");
Expect(1, 921599, '\p{^Is_Changes_When_NFKC_Casefolded=_	f}', "");
Expect(1, 921599, '\P{Is_Changes_When_NFKC_Casefolded=_	f}', "");
Expect(0, 921599, '\P{^Is_Changes_When_NFKC_Casefolded=_	f}', "");
Error('\p{Is_CWKCF=_-False:=}');
Error('\P{Is_CWKCF=_-False:=}');
Expect(1, 921600, '\p{Is_CWKCF=false}', "");
Expect(0, 921600, '\p{^Is_CWKCF=false}', "");
Expect(0, 921600, '\P{Is_CWKCF=false}', "");
Expect(1, 921600, '\P{^Is_CWKCF=false}', "");
Expect(0, 921599, '\p{Is_CWKCF=false}', "");
Expect(1, 921599, '\p{^Is_CWKCF=false}', "");
Expect(1, 921599, '\P{Is_CWKCF=false}', "");
Expect(0, 921599, '\P{^Is_CWKCF=false}', "");
Expect(1, 921600, '\p{Is_CWKCF=  FALSE}', "");
Expect(0, 921600, '\p{^Is_CWKCF=  FALSE}', "");
Expect(0, 921600, '\P{Is_CWKCF=  FALSE}', "");
Expect(1, 921600, '\P{^Is_CWKCF=  FALSE}', "");
Expect(0, 921599, '\p{Is_CWKCF=  FALSE}', "");
Expect(1, 921599, '\p{^Is_CWKCF=  FALSE}', "");
Expect(1, 921599, '\P{Is_CWKCF=  FALSE}', "");
Expect(0, 921599, '\P{^Is_CWKCF=  FALSE}', "");
Error('\p{Changes_When_NFKC_Casefolded=:= YES}');
Error('\P{Changes_When_NFKC_Casefolded=:= YES}');
Expect(1, 921599, '\p{Changes_When_NFKC_Casefolded=yes}', "");
Expect(0, 921599, '\p{^Changes_When_NFKC_Casefolded=yes}', "");
Expect(0, 921599, '\P{Changes_When_NFKC_Casefolded=yes}', "");
Expect(1, 921599, '\P{^Changes_When_NFKC_Casefolded=yes}', "");
Expect(0, 921600, '\p{Changes_When_NFKC_Casefolded=yes}', "");
Expect(1, 921600, '\p{^Changes_When_NFKC_Casefolded=yes}', "");
Expect(1, 921600, '\P{Changes_When_NFKC_Casefolded=yes}', "");
Expect(0, 921600, '\P{^Changes_When_NFKC_Casefolded=yes}', "");
Expect(1, 921599, '\p{Changes_When_NFKC_Casefolded=-Yes}', "");
Expect(0, 921599, '\p{^Changes_When_NFKC_Casefolded=-Yes}', "");
Expect(0, 921599, '\P{Changes_When_NFKC_Casefolded=-Yes}', "");
Expect(1, 921599, '\P{^Changes_When_NFKC_Casefolded=-Yes}', "");
Expect(0, 921600, '\p{Changes_When_NFKC_Casefolded=-Yes}', "");
Expect(1, 921600, '\p{^Changes_When_NFKC_Casefolded=-Yes}', "");
Expect(1, 921600, '\P{Changes_When_NFKC_Casefolded=-Yes}', "");
Expect(0, 921600, '\P{^Changes_When_NFKC_Casefolded=-Yes}', "");
Error('\p{CWKCF=_/a/Y}');
Error('\P{CWKCF=_/a/Y}');
Expect(1, 921599, '\p{CWKCF: y}', "");
Expect(0, 921599, '\p{^CWKCF: y}', "");
Expect(0, 921599, '\P{CWKCF: y}', "");
Expect(1, 921599, '\P{^CWKCF: y}', "");
Expect(0, 921600, '\p{CWKCF: y}', "");
Expect(1, 921600, '\p{^CWKCF: y}', "");
Expect(1, 921600, '\P{CWKCF: y}', "");
Expect(0, 921600, '\P{^CWKCF: y}', "");
Expect(1, 921599, '\p{CWKCF: _Y}', "");
Expect(0, 921599, '\p{^CWKCF: _Y}', "");
Expect(0, 921599, '\P{CWKCF: _Y}', "");
Expect(1, 921599, '\P{^CWKCF: _Y}', "");
Expect(0, 921600, '\p{CWKCF: _Y}', "");
Expect(1, 921600, '\p{^CWKCF: _Y}', "");
Expect(1, 921600, '\P{CWKCF: _Y}', "");
Expect(0, 921600, '\P{^CWKCF: _Y}', "");
Error('\p{Is_Changes_When_NFKC_Casefolded=	:=t}');
Error('\P{Is_Changes_When_NFKC_Casefolded=	:=t}');
Expect(1, 921599, '\p{Is_Changes_When_NFKC_Casefolded=t}', "");
Expect(0, 921599, '\p{^Is_Changes_When_NFKC_Casefolded=t}', "");
Expect(0, 921599, '\P{Is_Changes_When_NFKC_Casefolded=t}', "");
Expect(1, 921599, '\P{^Is_Changes_When_NFKC_Casefolded=t}', "");
Expect(0, 921600, '\p{Is_Changes_When_NFKC_Casefolded=t}', "");
Expect(1, 921600, '\p{^Is_Changes_When_NFKC_Casefolded=t}', "");
Expect(1, 921600, '\P{Is_Changes_When_NFKC_Casefolded=t}', "");
Expect(0, 921600, '\P{^Is_Changes_When_NFKC_Casefolded=t}', "");
Expect(1, 921599, '\p{Is_Changes_When_NFKC_Casefolded:  T}', "");
Expect(0, 921599, '\p{^Is_Changes_When_NFKC_Casefolded:  T}', "");
Expect(0, 921599, '\P{Is_Changes_When_NFKC_Casefolded:  T}', "");
Expect(1, 921599, '\P{^Is_Changes_When_NFKC_Casefolded:  T}', "");
Expect(0, 921600, '\p{Is_Changes_When_NFKC_Casefolded:  T}', "");
Expect(1, 921600, '\p{^Is_Changes_When_NFKC_Casefolded:  T}', "");
Expect(1, 921600, '\P{Is_Changes_When_NFKC_Casefolded:  T}', "");
Expect(0, 921600, '\P{^Is_Changes_When_NFKC_Casefolded:  T}', "");
Error('\p{Is_CWKCF=:=TRUE}');
Error('\P{Is_CWKCF=:=TRUE}');
Expect(1, 921599, '\p{Is_CWKCF:	true}', "");
Expect(0, 921599, '\p{^Is_CWKCF:	true}', "");
Expect(0, 921599, '\P{Is_CWKCF:	true}', "");
Expect(1, 921599, '\P{^Is_CWKCF:	true}', "");
Expect(0, 921600, '\p{Is_CWKCF:	true}', "");
Expect(1, 921600, '\p{^Is_CWKCF:	true}', "");
Expect(1, 921600, '\P{Is_CWKCF:	true}', "");
Expect(0, 921600, '\P{^Is_CWKCF:	true}', "");
Expect(1, 921599, '\p{Is_CWKCF=	True}', "");
Expect(0, 921599, '\p{^Is_CWKCF=	True}', "");
Expect(0, 921599, '\P{Is_CWKCF=	True}', "");
Expect(1, 921599, '\P{^Is_CWKCF=	True}', "");
Expect(0, 921600, '\p{Is_CWKCF=	True}', "");
Expect(1, 921600, '\p{^Is_CWKCF=	True}', "");
Expect(1, 921600, '\P{Is_CWKCF=	True}', "");
Expect(0, 921600, '\P{^Is_CWKCF=	True}', "");
Error('\p{Changes_When_Lowercased=-/a/No}');
Error('\P{Changes_When_Lowercased=-/a/No}');
Expect(1, 66600, '\p{Changes_When_Lowercased=no}', "");
Expect(0, 66600, '\p{^Changes_When_Lowercased=no}', "");
Expect(0, 66600, '\P{Changes_When_Lowercased=no}', "");
Expect(1, 66600, '\P{^Changes_When_Lowercased=no}', "");
Expect(0, 66599, '\p{Changes_When_Lowercased=no}', "");
Expect(1, 66599, '\p{^Changes_When_Lowercased=no}', "");
Expect(1, 66599, '\P{Changes_When_Lowercased=no}', "");
Expect(0, 66599, '\P{^Changes_When_Lowercased=no}', "");
Expect(1, 66600, '\p{Changes_When_Lowercased= 	No}', "");
Expect(0, 66600, '\p{^Changes_When_Lowercased= 	No}', "");
Expect(0, 66600, '\P{Changes_When_Lowercased= 	No}', "");
Expect(1, 66600, '\P{^Changes_When_Lowercased= 	No}', "");
Expect(0, 66599, '\p{Changes_When_Lowercased= 	No}', "");
Expect(1, 66599, '\p{^Changes_When_Lowercased= 	No}', "");
Expect(1, 66599, '\P{Changes_When_Lowercased= 	No}', "");
Expect(0, 66599, '\P{^Changes_When_Lowercased= 	No}', "");
Error('\p{CWL=:=- N}');
Error('\P{CWL=:=- N}');
Expect(1, 66600, '\p{CWL=n}', "");
Expect(0, 66600, '\p{^CWL=n}', "");
Expect(0, 66600, '\P{CWL=n}', "");
Expect(1, 66600, '\P{^CWL=n}', "");
Expect(0, 66599, '\p{CWL=n}', "");
Expect(1, 66599, '\p{^CWL=n}', "");
Expect(1, 66599, '\P{CWL=n}', "");
Expect(0, 66599, '\P{^CWL=n}', "");
Expect(1, 66600, '\p{CWL=--N}', "");
Expect(0, 66600, '\p{^CWL=--N}', "");
Expect(0, 66600, '\P{CWL=--N}', "");
Expect(1, 66600, '\P{^CWL=--N}', "");
Expect(0, 66599, '\p{CWL=--N}', "");
Expect(1, 66599, '\p{^CWL=--N}', "");
Expect(1, 66599, '\P{CWL=--N}', "");
Expect(0, 66599, '\P{^CWL=--N}', "");
Error('\p{Is_Changes_When_Lowercased: _/a/F}');
Error('\P{Is_Changes_When_Lowercased: _/a/F}');
Expect(1, 66600, '\p{Is_Changes_When_Lowercased=f}', "");
Expect(0, 66600, '\p{^Is_Changes_When_Lowercased=f}', "");
Expect(0, 66600, '\P{Is_Changes_When_Lowercased=f}', "");
Expect(1, 66600, '\P{^Is_Changes_When_Lowercased=f}', "");
Expect(0, 66599, '\p{Is_Changes_When_Lowercased=f}', "");
Expect(1, 66599, '\p{^Is_Changes_When_Lowercased=f}', "");
Expect(1, 66599, '\P{Is_Changes_When_Lowercased=f}', "");
Expect(0, 66599, '\P{^Is_Changes_When_Lowercased=f}', "");
Expect(1, 66600, '\p{Is_Changes_When_Lowercased= F}', "");
Expect(0, 66600, '\p{^Is_Changes_When_Lowercased= F}', "");
Expect(0, 66600, '\P{Is_Changes_When_Lowercased= F}', "");
Expect(1, 66600, '\P{^Is_Changes_When_Lowercased= F}', "");
Expect(0, 66599, '\p{Is_Changes_When_Lowercased= F}', "");
Expect(1, 66599, '\p{^Is_Changes_When_Lowercased= F}', "");
Expect(1, 66599, '\P{Is_Changes_When_Lowercased= F}', "");
Expect(0, 66599, '\P{^Is_Changes_When_Lowercased= F}', "");
Error('\p{Is_CWL=:= 	false}');
Error('\P{Is_CWL=:= 	false}');
Expect(1, 66600, '\p{Is_CWL=false}', "");
Expect(0, 66600, '\p{^Is_CWL=false}', "");
Expect(0, 66600, '\P{Is_CWL=false}', "");
Expect(1, 66600, '\P{^Is_CWL=false}', "");
Expect(0, 66599, '\p{Is_CWL=false}', "");
Expect(1, 66599, '\p{^Is_CWL=false}', "");
Expect(1, 66599, '\P{Is_CWL=false}', "");
Expect(0, 66599, '\P{^Is_CWL=false}', "");
Expect(1, 66600, '\p{Is_CWL= _FALSE}', "");
Expect(0, 66600, '\p{^Is_CWL= _FALSE}', "");
Expect(0, 66600, '\P{Is_CWL= _FALSE}', "");
Expect(1, 66600, '\P{^Is_CWL= _FALSE}', "");
Expect(0, 66599, '\p{Is_CWL= _FALSE}', "");
Expect(1, 66599, '\p{^Is_CWL= _FALSE}', "");
Expect(1, 66599, '\P{Is_CWL= _FALSE}', "");
Expect(0, 66599, '\P{^Is_CWL= _FALSE}', "");
Error('\p{Changes_When_Lowercased=-_Yes:=}');
Error('\P{Changes_When_Lowercased=-_Yes:=}');
Expect(1, 66599, '\p{Changes_When_Lowercased=yes}', "");
Expect(0, 66599, '\p{^Changes_When_Lowercased=yes}', "");
Expect(0, 66599, '\P{Changes_When_Lowercased=yes}', "");
Expect(1, 66599, '\P{^Changes_When_Lowercased=yes}', "");
Expect(0, 66600, '\p{Changes_When_Lowercased=yes}', "");
Expect(1, 66600, '\p{^Changes_When_Lowercased=yes}', "");
Expect(1, 66600, '\P{Changes_When_Lowercased=yes}', "");
Expect(0, 66600, '\P{^Changes_When_Lowercased=yes}', "");
Expect(1, 66599, '\p{Changes_When_Lowercased=	-yes}', "");
Expect(0, 66599, '\p{^Changes_When_Lowercased=	-yes}', "");
Expect(0, 66599, '\P{Changes_When_Lowercased=	-yes}', "");
Expect(1, 66599, '\P{^Changes_When_Lowercased=	-yes}', "");
Expect(0, 66600, '\p{Changes_When_Lowercased=	-yes}', "");
Expect(1, 66600, '\p{^Changes_When_Lowercased=	-yes}', "");
Expect(1, 66600, '\P{Changes_When_Lowercased=	-yes}', "");
Expect(0, 66600, '\P{^Changes_When_Lowercased=	-yes}', "");
Error('\p{CWL=_ Y:=}');
Error('\P{CWL=_ Y:=}');
Expect(1, 66599, '\p{CWL=y}', "");
Expect(0, 66599, '\p{^CWL=y}', "");
Expect(0, 66599, '\P{CWL=y}', "");
Expect(1, 66599, '\P{^CWL=y}', "");
Expect(0, 66600, '\p{CWL=y}', "");
Expect(1, 66600, '\p{^CWL=y}', "");
Expect(1, 66600, '\P{CWL=y}', "");
Expect(0, 66600, '\P{^CWL=y}', "");
Expect(1, 66599, '\p{CWL=__Y}', "");
Expect(0, 66599, '\p{^CWL=__Y}', "");
Expect(0, 66599, '\P{CWL=__Y}', "");
Expect(1, 66599, '\P{^CWL=__Y}', "");
Expect(0, 66600, '\p{CWL=__Y}', "");
Expect(1, 66600, '\p{^CWL=__Y}', "");
Expect(1, 66600, '\P{CWL=__Y}', "");
Expect(0, 66600, '\P{^CWL=__Y}', "");
Error('\p{Is_Changes_When_Lowercased=:= _T}');
Error('\P{Is_Changes_When_Lowercased=:= _T}');
Expect(1, 66599, '\p{Is_Changes_When_Lowercased:	t}', "");
Expect(0, 66599, '\p{^Is_Changes_When_Lowercased:	t}', "");
Expect(0, 66599, '\P{Is_Changes_When_Lowercased:	t}', "");
Expect(1, 66599, '\P{^Is_Changes_When_Lowercased:	t}', "");
Expect(0, 66600, '\p{Is_Changes_When_Lowercased:	t}', "");
Expect(1, 66600, '\p{^Is_Changes_When_Lowercased:	t}', "");
Expect(1, 66600, '\P{Is_Changes_When_Lowercased:	t}', "");
Expect(0, 66600, '\P{^Is_Changes_When_Lowercased:	t}', "");
Expect(1, 66599, '\p{Is_Changes_When_Lowercased= -T}', "");
Expect(0, 66599, '\p{^Is_Changes_When_Lowercased= -T}', "");
Expect(0, 66599, '\P{Is_Changes_When_Lowercased= -T}', "");
Expect(1, 66599, '\P{^Is_Changes_When_Lowercased= -T}', "");
Expect(0, 66600, '\p{Is_Changes_When_Lowercased= -T}', "");
Expect(1, 66600, '\p{^Is_Changes_When_Lowercased= -T}', "");
Expect(1, 66600, '\P{Is_Changes_When_Lowercased= -T}', "");
Expect(0, 66600, '\P{^Is_Changes_When_Lowercased= -T}', "");
Error('\p{Is_CWL= /a/True}');
Error('\P{Is_CWL= /a/True}');
Expect(1, 66599, '\p{Is_CWL=true}', "");
Expect(0, 66599, '\p{^Is_CWL=true}', "");
Expect(0, 66599, '\P{Is_CWL=true}', "");
Expect(1, 66599, '\P{^Is_CWL=true}', "");
Expect(0, 66600, '\p{Is_CWL=true}', "");
Expect(1, 66600, '\p{^Is_CWL=true}', "");
Expect(1, 66600, '\P{Is_CWL=true}', "");
Expect(0, 66600, '\P{^Is_CWL=true}', "");
Expect(1, 66599, '\p{Is_CWL=- True}', "");
Expect(0, 66599, '\p{^Is_CWL=- True}', "");
Expect(0, 66599, '\P{Is_CWL=- True}', "");
Expect(1, 66599, '\P{^Is_CWL=- True}', "");
Expect(0, 66600, '\p{Is_CWL=- True}', "");
Expect(1, 66600, '\p{^Is_CWL=- True}', "");
Expect(1, 66600, '\P{Is_CWL=- True}', "");
Expect(0, 66600, '\P{^Is_CWL=- True}', "");
Error('\p{Changes_When_Titlecased=	_No/a/}');
Error('\P{Changes_When_Titlecased=	_No/a/}');
Expect(1, 66640, '\p{Changes_When_Titlecased=no}', "");
Expect(0, 66640, '\p{^Changes_When_Titlecased=no}', "");
Expect(0, 66640, '\P{Changes_When_Titlecased=no}', "");
Expect(1, 66640, '\P{^Changes_When_Titlecased=no}', "");
Expect(0, 66639, '\p{Changes_When_Titlecased=no}', "");
Expect(1, 66639, '\p{^Changes_When_Titlecased=no}', "");
Expect(1, 66639, '\P{Changes_When_Titlecased=no}', "");
Expect(0, 66639, '\P{^Changes_When_Titlecased=no}', "");
Expect(1, 66640, '\p{Changes_When_Titlecased: -No}', "");
Expect(0, 66640, '\p{^Changes_When_Titlecased: -No}', "");
Expect(0, 66640, '\P{Changes_When_Titlecased: -No}', "");
Expect(1, 66640, '\P{^Changes_When_Titlecased: -No}', "");
Expect(0, 66639, '\p{Changes_When_Titlecased: -No}', "");
Expect(1, 66639, '\p{^Changes_When_Titlecased: -No}', "");
Expect(1, 66639, '\P{Changes_When_Titlecased: -No}', "");
Expect(0, 66639, '\P{^Changes_When_Titlecased: -No}', "");
Error('\p{CWT: /a/ N}');
Error('\P{CWT: /a/ N}');
Expect(1, 66640, '\p{CWT:n}', "");
Expect(0, 66640, '\p{^CWT:n}', "");
Expect(0, 66640, '\P{CWT:n}', "");
Expect(1, 66640, '\P{^CWT:n}', "");
Expect(0, 66639, '\p{CWT:n}', "");
Expect(1, 66639, '\p{^CWT:n}', "");
Expect(1, 66639, '\P{CWT:n}', "");
Expect(0, 66639, '\P{^CWT:n}', "");
Expect(1, 66640, '\p{CWT=	_N}', "");
Expect(0, 66640, '\p{^CWT=	_N}', "");
Expect(0, 66640, '\P{CWT=	_N}', "");
Expect(1, 66640, '\P{^CWT=	_N}', "");
Expect(0, 66639, '\p{CWT=	_N}', "");
Expect(1, 66639, '\p{^CWT=	_N}', "");
Expect(1, 66639, '\P{CWT=	_N}', "");
Expect(0, 66639, '\P{^CWT=	_N}', "");
Error('\p{Is_Changes_When_Titlecased=:= 	F}');
Error('\P{Is_Changes_When_Titlecased=:= 	F}');
Expect(1, 66640, '\p{Is_Changes_When_Titlecased=f}', "");
Expect(0, 66640, '\p{^Is_Changes_When_Titlecased=f}', "");
Expect(0, 66640, '\P{Is_Changes_When_Titlecased=f}', "");
Expect(1, 66640, '\P{^Is_Changes_When_Titlecased=f}', "");
Expect(0, 66639, '\p{Is_Changes_When_Titlecased=f}', "");
Expect(1, 66639, '\p{^Is_Changes_When_Titlecased=f}', "");
Expect(1, 66639, '\P{Is_Changes_When_Titlecased=f}', "");
Expect(0, 66639, '\P{^Is_Changes_When_Titlecased=f}', "");
Expect(1, 66640, '\p{Is_Changes_When_Titlecased:		F}', "");
Expect(0, 66640, '\p{^Is_Changes_When_Titlecased:		F}', "");
Expect(0, 66640, '\P{Is_Changes_When_Titlecased:		F}', "");
Expect(1, 66640, '\P{^Is_Changes_When_Titlecased:		F}', "");
Expect(0, 66639, '\p{Is_Changes_When_Titlecased:		F}', "");
Expect(1, 66639, '\p{^Is_Changes_When_Titlecased:		F}', "");
Expect(1, 66639, '\P{Is_Changes_When_Titlecased:		F}', "");
Expect(0, 66639, '\P{^Is_Changes_When_Titlecased:		F}', "");
Error('\p{Is_CWT= FALSE:=}');
Error('\P{Is_CWT= FALSE:=}');
Expect(1, 66640, '\p{Is_CWT=false}', "");
Expect(0, 66640, '\p{^Is_CWT=false}', "");
Expect(0, 66640, '\P{Is_CWT=false}', "");
Expect(1, 66640, '\P{^Is_CWT=false}', "");
Expect(0, 66639, '\p{Is_CWT=false}', "");
Expect(1, 66639, '\p{^Is_CWT=false}', "");
Expect(1, 66639, '\P{Is_CWT=false}', "");
Expect(0, 66639, '\P{^Is_CWT=false}', "");
Expect(1, 66640, '\p{Is_CWT: 	FALSE}', "");
Expect(0, 66640, '\p{^Is_CWT: 	FALSE}', "");
Expect(0, 66640, '\P{Is_CWT: 	FALSE}', "");
Expect(1, 66640, '\P{^Is_CWT: 	FALSE}', "");
Expect(0, 66639, '\p{Is_CWT: 	FALSE}', "");
Expect(1, 66639, '\p{^Is_CWT: 	FALSE}', "");
Expect(1, 66639, '\P{Is_CWT: 	FALSE}', "");
Expect(0, 66639, '\P{^Is_CWT: 	FALSE}', "");
Error('\p{Changes_When_Titlecased= /a/yes}');
Error('\P{Changes_When_Titlecased= /a/yes}');
Expect(1, 66639, '\p{Changes_When_Titlecased=yes}', "");
Expect(0, 66639, '\p{^Changes_When_Titlecased=yes}', "");
Expect(0, 66639, '\P{Changes_When_Titlecased=yes}', "");
Expect(1, 66639, '\P{^Changes_When_Titlecased=yes}', "");
Expect(0, 66640, '\p{Changes_When_Titlecased=yes}', "");
Expect(1, 66640, '\p{^Changes_When_Titlecased=yes}', "");
Expect(1, 66640, '\P{Changes_When_Titlecased=yes}', "");
Expect(0, 66640, '\P{^Changes_When_Titlecased=yes}', "");
Expect(1, 66639, '\p{Changes_When_Titlecased=  YES}', "");
Expect(0, 66639, '\p{^Changes_When_Titlecased=  YES}', "");
Expect(0, 66639, '\P{Changes_When_Titlecased=  YES}', "");
Expect(1, 66639, '\P{^Changes_When_Titlecased=  YES}', "");
Expect(0, 66640, '\p{Changes_When_Titlecased=  YES}', "");
Expect(1, 66640, '\p{^Changes_When_Titlecased=  YES}', "");
Expect(1, 66640, '\P{Changes_When_Titlecased=  YES}', "");
Expect(0, 66640, '\P{^Changes_When_Titlecased=  YES}', "");
Error('\p{CWT=_	Y:=}');
Error('\P{CWT=_	Y:=}');
Expect(1, 66639, '\p{CWT=y}', "");
Expect(0, 66639, '\p{^CWT=y}', "");
Expect(0, 66639, '\P{CWT=y}', "");
Expect(1, 66639, '\P{^CWT=y}', "");
Expect(0, 66640, '\p{CWT=y}', "");
Expect(1, 66640, '\p{^CWT=y}', "");
Expect(1, 66640, '\P{CWT=y}', "");
Expect(0, 66640, '\P{^CWT=y}', "");
Expect(1, 66639, '\p{CWT= -Y}', "");
Expect(0, 66639, '\p{^CWT= -Y}', "");
Expect(0, 66639, '\P{CWT= -Y}', "");
Expect(1, 66639, '\P{^CWT= -Y}', "");
Expect(0, 66640, '\p{CWT= -Y}', "");
Expect(1, 66640, '\p{^CWT= -Y}', "");
Expect(1, 66640, '\P{CWT= -Y}', "");
Expect(0, 66640, '\P{^CWT= -Y}', "");
Error('\p{Is_Changes_When_Titlecased=:=T}');
Error('\P{Is_Changes_When_Titlecased=:=T}');
Expect(1, 66639, '\p{Is_Changes_When_Titlecased:t}', "");
Expect(0, 66639, '\p{^Is_Changes_When_Titlecased:t}', "");
Expect(0, 66639, '\P{Is_Changes_When_Titlecased:t}', "");
Expect(1, 66639, '\P{^Is_Changes_When_Titlecased:t}', "");
Expect(0, 66640, '\p{Is_Changes_When_Titlecased:t}', "");
Expect(1, 66640, '\p{^Is_Changes_When_Titlecased:t}', "");
Expect(1, 66640, '\P{Is_Changes_When_Titlecased:t}', "");
Expect(0, 66640, '\P{^Is_Changes_When_Titlecased:t}', "");
Expect(1, 66639, '\p{Is_Changes_When_Titlecased=-	t}', "");
Expect(0, 66639, '\p{^Is_Changes_When_Titlecased=-	t}', "");
Expect(0, 66639, '\P{Is_Changes_When_Titlecased=-	t}', "");
Expect(1, 66639, '\P{^Is_Changes_When_Titlecased=-	t}', "");
Expect(0, 66640, '\p{Is_Changes_When_Titlecased=-	t}', "");
Expect(1, 66640, '\p{^Is_Changes_When_Titlecased=-	t}', "");
Expect(1, 66640, '\P{Is_Changes_When_Titlecased=-	t}', "");
Expect(0, 66640, '\P{^Is_Changes_When_Titlecased=-	t}', "");
Error('\p{Is_CWT=-/a/true}');
Error('\P{Is_CWT=-/a/true}');
Expect(1, 66639, '\p{Is_CWT=true}', "");
Expect(0, 66639, '\p{^Is_CWT=true}', "");
Expect(0, 66639, '\P{Is_CWT=true}', "");
Expect(1, 66639, '\P{^Is_CWT=true}', "");
Expect(0, 66640, '\p{Is_CWT=true}', "");
Expect(1, 66640, '\p{^Is_CWT=true}', "");
Expect(1, 66640, '\P{Is_CWT=true}', "");
Expect(0, 66640, '\P{^Is_CWT=true}', "");
Expect(1, 66639, '\p{Is_CWT=-True}', "");
Expect(0, 66639, '\p{^Is_CWT=-True}', "");
Expect(0, 66639, '\P{Is_CWT=-True}', "");
Expect(1, 66639, '\P{^Is_CWT=-True}', "");
Expect(0, 66640, '\p{Is_CWT=-True}', "");
Expect(1, 66640, '\p{^Is_CWT=-True}', "");
Expect(1, 66640, '\P{Is_CWT=-True}', "");
Expect(0, 66640, '\P{^Is_CWT=-True}', "");
Error('\p{Changes_When_Uppercased=	/a/NO}');
Error('\P{Changes_When_Uppercased=	/a/NO}');
Expect(1, 66640, '\p{Changes_When_Uppercased=no}', "");
Expect(0, 66640, '\p{^Changes_When_Uppercased=no}', "");
Expect(0, 66640, '\P{Changes_When_Uppercased=no}', "");
Expect(1, 66640, '\P{^Changes_When_Uppercased=no}', "");
Expect(0, 66639, '\p{Changes_When_Uppercased=no}', "");
Expect(1, 66639, '\p{^Changes_When_Uppercased=no}', "");
Expect(1, 66639, '\P{Changes_When_Uppercased=no}', "");
Expect(0, 66639, '\P{^Changes_When_Uppercased=no}', "");
Expect(1, 66640, '\p{Changes_When_Uppercased=--no}', "");
Expect(0, 66640, '\p{^Changes_When_Uppercased=--no}', "");
Expect(0, 66640, '\P{Changes_When_Uppercased=--no}', "");
Expect(1, 66640, '\P{^Changes_When_Uppercased=--no}', "");
Expect(0, 66639, '\p{Changes_When_Uppercased=--no}', "");
Expect(1, 66639, '\p{^Changes_When_Uppercased=--no}', "");
Expect(1, 66639, '\P{Changes_When_Uppercased=--no}', "");
Expect(0, 66639, '\P{^Changes_When_Uppercased=--no}', "");
Error('\p{CWU=N:=}');
Error('\P{CWU=N:=}');
Expect(1, 66640, '\p{CWU=n}', "");
Expect(0, 66640, '\p{^CWU=n}', "");
Expect(0, 66640, '\P{CWU=n}', "");
Expect(1, 66640, '\P{^CWU=n}', "");
Expect(0, 66639, '\p{CWU=n}', "");
Expect(1, 66639, '\p{^CWU=n}', "");
Expect(1, 66639, '\P{CWU=n}', "");
Expect(0, 66639, '\P{^CWU=n}', "");
Expect(1, 66640, '\p{CWU=N}', "");
Expect(0, 66640, '\p{^CWU=N}', "");
Expect(0, 66640, '\P{CWU=N}', "");
Expect(1, 66640, '\P{^CWU=N}', "");
Expect(0, 66639, '\p{CWU=N}', "");
Expect(1, 66639, '\p{^CWU=N}', "");
Expect(1, 66639, '\P{CWU=N}', "");
Expect(0, 66639, '\P{^CWU=N}', "");
Error('\p{Is_Changes_When_Uppercased=-:=F}');
Error('\P{Is_Changes_When_Uppercased=-:=F}');
Expect(1, 66640, '\p{Is_Changes_When_Uppercased=f}', "");
Expect(0, 66640, '\p{^Is_Changes_When_Uppercased=f}', "");
Expect(0, 66640, '\P{Is_Changes_When_Uppercased=f}', "");
Expect(1, 66640, '\P{^Is_Changes_When_Uppercased=f}', "");
Expect(0, 66639, '\p{Is_Changes_When_Uppercased=f}', "");
Expect(1, 66639, '\p{^Is_Changes_When_Uppercased=f}', "");
Expect(1, 66639, '\P{Is_Changes_When_Uppercased=f}', "");
Expect(0, 66639, '\P{^Is_Changes_When_Uppercased=f}', "");
Expect(1, 66640, '\p{Is_Changes_When_Uppercased= 	F}', "");
Expect(0, 66640, '\p{^Is_Changes_When_Uppercased= 	F}', "");
Expect(0, 66640, '\P{Is_Changes_When_Uppercased= 	F}', "");
Expect(1, 66640, '\P{^Is_Changes_When_Uppercased= 	F}', "");
Expect(0, 66639, '\p{Is_Changes_When_Uppercased= 	F}', "");
Expect(1, 66639, '\p{^Is_Changes_When_Uppercased= 	F}', "");
Expect(1, 66639, '\P{Is_Changes_When_Uppercased= 	F}', "");
Expect(0, 66639, '\P{^Is_Changes_When_Uppercased= 	F}', "");
Error('\p{Is_CWU=:=--False}');
Error('\P{Is_CWU=:=--False}');
Expect(1, 66640, '\p{Is_CWU=false}', "");
Expect(0, 66640, '\p{^Is_CWU=false}', "");
Expect(0, 66640, '\P{Is_CWU=false}', "");
Expect(1, 66640, '\P{^Is_CWU=false}', "");
Expect(0, 66639, '\p{Is_CWU=false}', "");
Expect(1, 66639, '\p{^Is_CWU=false}', "");
Expect(1, 66639, '\P{Is_CWU=false}', "");
Expect(0, 66639, '\P{^Is_CWU=false}', "");
Expect(1, 66640, '\p{Is_CWU=-	FALSE}', "");
Expect(0, 66640, '\p{^Is_CWU=-	FALSE}', "");
Expect(0, 66640, '\P{Is_CWU=-	FALSE}', "");
Expect(1, 66640, '\P{^Is_CWU=-	FALSE}', "");
Expect(0, 66639, '\p{Is_CWU=-	FALSE}', "");
Expect(1, 66639, '\p{^Is_CWU=-	FALSE}', "");
Expect(1, 66639, '\P{Is_CWU=-	FALSE}', "");
Expect(0, 66639, '\P{^Is_CWU=-	FALSE}', "");
Error('\p{Changes_When_Uppercased:	_:=YES}');
Error('\P{Changes_When_Uppercased:	_:=YES}');
Expect(1, 66639, '\p{Changes_When_Uppercased=yes}', "");
Expect(0, 66639, '\p{^Changes_When_Uppercased=yes}', "");
Expect(0, 66639, '\P{Changes_When_Uppercased=yes}', "");
Expect(1, 66639, '\P{^Changes_When_Uppercased=yes}', "");
Expect(0, 66640, '\p{Changes_When_Uppercased=yes}', "");
Expect(1, 66640, '\p{^Changes_When_Uppercased=yes}', "");
Expect(1, 66640, '\P{Changes_When_Uppercased=yes}', "");
Expect(0, 66640, '\P{^Changes_When_Uppercased=yes}', "");
Expect(1, 66639, '\p{Changes_When_Uppercased=_-Yes}', "");
Expect(0, 66639, '\p{^Changes_When_Uppercased=_-Yes}', "");
Expect(0, 66639, '\P{Changes_When_Uppercased=_-Yes}', "");
Expect(1, 66639, '\P{^Changes_When_Uppercased=_-Yes}', "");
Expect(0, 66640, '\p{Changes_When_Uppercased=_-Yes}', "");
Expect(1, 66640, '\p{^Changes_When_Uppercased=_-Yes}', "");
Expect(1, 66640, '\P{Changes_When_Uppercased=_-Yes}', "");
Expect(0, 66640, '\P{^Changes_When_Uppercased=_-Yes}', "");
Error('\p{CWU=	Y:=}');
Error('\P{CWU=	Y:=}');
Expect(1, 66639, '\p{CWU=y}', "");
Expect(0, 66639, '\p{^CWU=y}', "");
Expect(0, 66639, '\P{CWU=y}', "");
Expect(1, 66639, '\P{^CWU=y}', "");
Expect(0, 66640, '\p{CWU=y}', "");
Expect(1, 66640, '\p{^CWU=y}', "");
Expect(1, 66640, '\P{CWU=y}', "");
Expect(0, 66640, '\P{^CWU=y}', "");
Expect(1, 66639, '\p{CWU=-y}', "");
Expect(0, 66639, '\p{^CWU=-y}', "");
Expect(0, 66639, '\P{CWU=-y}', "");
Expect(1, 66639, '\P{^CWU=-y}', "");
Expect(0, 66640, '\p{CWU=-y}', "");
Expect(1, 66640, '\p{^CWU=-y}', "");
Expect(1, 66640, '\P{CWU=-y}', "");
Expect(0, 66640, '\P{^CWU=-y}', "");
Error('\p{Is_Changes_When_Uppercased=t/a/}');
Error('\P{Is_Changes_When_Uppercased=t/a/}');
Expect(1, 66639, '\p{Is_Changes_When_Uppercased=t}', "");
Expect(0, 66639, '\p{^Is_Changes_When_Uppercased=t}', "");
Expect(0, 66639, '\P{Is_Changes_When_Uppercased=t}', "");
Expect(1, 66639, '\P{^Is_Changes_When_Uppercased=t}', "");
Expect(0, 66640, '\p{Is_Changes_When_Uppercased=t}', "");
Expect(1, 66640, '\p{^Is_Changes_When_Uppercased=t}', "");
Expect(1, 66640, '\P{Is_Changes_When_Uppercased=t}', "");
Expect(0, 66640, '\P{^Is_Changes_When_Uppercased=t}', "");
Expect(1, 66639, '\p{Is_Changes_When_Uppercased:	  T}', "");
Expect(0, 66639, '\p{^Is_Changes_When_Uppercased:	  T}', "");
Expect(0, 66639, '\P{Is_Changes_When_Uppercased:	  T}', "");
Expect(1, 66639, '\P{^Is_Changes_When_Uppercased:	  T}', "");
Expect(0, 66640, '\p{Is_Changes_When_Uppercased:	  T}', "");
Expect(1, 66640, '\p{^Is_Changes_When_Uppercased:	  T}', "");
Expect(1, 66640, '\P{Is_Changes_When_Uppercased:	  T}', "");
Expect(0, 66640, '\P{^Is_Changes_When_Uppercased:	  T}', "");
Error('\p{Is_CWU=/a/ TRUE}');
Error('\P{Is_CWU=/a/ TRUE}');
Expect(1, 66639, '\p{Is_CWU=true}', "");
Expect(0, 66639, '\p{^Is_CWU=true}', "");
Expect(0, 66639, '\P{Is_CWU=true}', "");
Expect(1, 66639, '\P{^Is_CWU=true}', "");
Expect(0, 66640, '\p{Is_CWU=true}', "");
Expect(1, 66640, '\p{^Is_CWU=true}', "");
Expect(1, 66640, '\P{Is_CWU=true}', "");
Expect(0, 66640, '\P{^Is_CWU=true}', "");
Expect(1, 66639, '\p{Is_CWU= True}', "");
Expect(0, 66639, '\p{^Is_CWU= True}', "");
Expect(0, 66639, '\P{Is_CWU= True}', "");
Expect(1, 66639, '\P{^Is_CWU= True}', "");
Expect(0, 66640, '\p{Is_CWU= True}', "");
Expect(1, 66640, '\p{^Is_CWU= True}', "");
Expect(1, 66640, '\P{Is_CWU= True}', "");
Expect(0, 66640, '\P{^Is_CWU= True}', "");
Error('\p{Cased= _NO:=}');
Error('\P{Cased= _NO:=}');
Expect(1, 120780, '\p{Cased=no}', "");
Expect(0, 120780, '\p{^Cased=no}', "");
Expect(0, 120780, '\P{Cased=no}', "");
Expect(1, 120780, '\P{^Cased=no}', "");
Expect(0, 120779, '\p{Cased=no}', "");
Expect(1, 120779, '\p{^Cased=no}', "");
Expect(1, 120779, '\P{Cased=no}', "");
Expect(0, 120779, '\P{^Cased=no}', "");
Expect(1, 120780, '\p{Cased=-NO}', "");
Expect(0, 120780, '\p{^Cased=-NO}', "");
Expect(0, 120780, '\P{Cased=-NO}', "");
Expect(1, 120780, '\P{^Cased=-NO}', "");
Expect(0, 120779, '\p{Cased=-NO}', "");
Expect(1, 120779, '\p{^Cased=-NO}', "");
Expect(1, 120779, '\P{Cased=-NO}', "");
Expect(0, 120779, '\P{^Cased=-NO}', "");
Error('\p{Is_Cased=-:=N}');
Error('\P{Is_Cased=-:=N}');
Expect(1, 120780, '\p{Is_Cased=n}', "");
Expect(0, 120780, '\p{^Is_Cased=n}', "");
Expect(0, 120780, '\P{Is_Cased=n}', "");
Expect(1, 120780, '\P{^Is_Cased=n}', "");
Expect(0, 120779, '\p{Is_Cased=n}', "");
Expect(1, 120779, '\p{^Is_Cased=n}', "");
Expect(1, 120779, '\P{Is_Cased=n}', "");
Expect(0, 120779, '\P{^Is_Cased=n}', "");
Expect(1, 120780, '\p{Is_Cased=_N}', "");
Expect(0, 120780, '\p{^Is_Cased=_N}', "");
Expect(0, 120780, '\P{Is_Cased=_N}', "");
Expect(1, 120780, '\P{^Is_Cased=_N}', "");
Expect(0, 120779, '\p{Is_Cased=_N}', "");
Expect(1, 120779, '\p{^Is_Cased=_N}', "");
Expect(1, 120779, '\P{Is_Cased=_N}', "");
Expect(0, 120779, '\P{^Is_Cased=_N}', "");
Error('\p{Cased=	:=F}');
Error('\P{Cased=	:=F}');
Expect(1, 120780, '\p{Cased=f}', "");
Expect(0, 120780, '\p{^Cased=f}', "");
Expect(0, 120780, '\P{Cased=f}', "");
Expect(1, 120780, '\P{^Cased=f}', "");
Expect(0, 120779, '\p{Cased=f}', "");
Expect(1, 120779, '\p{^Cased=f}', "");
Expect(1, 120779, '\P{Cased=f}', "");
Expect(0, 120779, '\P{^Cased=f}', "");
Expect(1, 120780, '\p{Cased= F}', "");
Expect(0, 120780, '\p{^Cased= F}', "");
Expect(0, 120780, '\P{Cased= F}', "");
Expect(1, 120780, '\P{^Cased= F}', "");
Expect(0, 120779, '\p{Cased= F}', "");
Expect(1, 120779, '\p{^Cased= F}', "");
Expect(1, 120779, '\P{Cased= F}', "");
Expect(0, 120779, '\P{^Cased= F}', "");
Error('\p{Is_Cased=:=  False}');
Error('\P{Is_Cased=:=  False}');
Expect(1, 120780, '\p{Is_Cased=false}', "");
Expect(0, 120780, '\p{^Is_Cased=false}', "");
Expect(0, 120780, '\P{Is_Cased=false}', "");
Expect(1, 120780, '\P{^Is_Cased=false}', "");
Expect(0, 120779, '\p{Is_Cased=false}', "");
Expect(1, 120779, '\p{^Is_Cased=false}', "");
Expect(1, 120779, '\P{Is_Cased=false}', "");
Expect(0, 120779, '\P{^Is_Cased=false}', "");
Expect(1, 120780, '\p{Is_Cased=_ false}', "");
Expect(0, 120780, '\p{^Is_Cased=_ false}', "");
Expect(0, 120780, '\P{Is_Cased=_ false}', "");
Expect(1, 120780, '\P{^Is_Cased=_ false}', "");
Expect(0, 120779, '\p{Is_Cased=_ false}', "");
Expect(1, 120779, '\p{^Is_Cased=_ false}', "");
Expect(1, 120779, '\P{Is_Cased=_ false}', "");
Expect(0, 120779, '\P{^Is_Cased=_ false}', "");
Error('\p{Cased= yes/a/}');
Error('\P{Cased= yes/a/}');
Expect(1, 120779, '\p{Cased=yes}', "");
Expect(0, 120779, '\p{^Cased=yes}', "");
Expect(0, 120779, '\P{Cased=yes}', "");
Expect(1, 120779, '\P{^Cased=yes}', "");
Expect(0, 120780, '\p{Cased=yes}', "");
Expect(1, 120780, '\p{^Cased=yes}', "");
Expect(1, 120780, '\P{Cased=yes}', "");
Expect(0, 120780, '\P{^Cased=yes}', "");
Expect(1, 120779, '\p{Cased=	Yes}', "");
Expect(0, 120779, '\p{^Cased=	Yes}', "");
Expect(0, 120779, '\P{Cased=	Yes}', "");
Expect(1, 120779, '\P{^Cased=	Yes}', "");
Expect(0, 120780, '\p{Cased=	Yes}', "");
Expect(1, 120780, '\p{^Cased=	Yes}', "");
Expect(1, 120780, '\P{Cased=	Yes}', "");
Expect(0, 120780, '\P{^Cased=	Yes}', "");
Error('\p{Is_Cased=/a/y}');
Error('\P{Is_Cased=/a/y}');
Expect(1, 120779, '\p{Is_Cased:y}', "");
Expect(0, 120779, '\p{^Is_Cased:y}', "");
Expect(0, 120779, '\P{Is_Cased:y}', "");
Expect(1, 120779, '\P{^Is_Cased:y}', "");
Expect(0, 120780, '\p{Is_Cased:y}', "");
Expect(1, 120780, '\p{^Is_Cased:y}', "");
Expect(1, 120780, '\P{Is_Cased:y}', "");
Expect(0, 120780, '\P{^Is_Cased:y}', "");
Expect(1, 120779, '\p{Is_Cased=-	y}', "");
Expect(0, 120779, '\p{^Is_Cased=-	y}', "");
Expect(0, 120779, '\P{Is_Cased=-	y}', "");
Expect(1, 120779, '\P{^Is_Cased=-	y}', "");
Expect(0, 120780, '\p{Is_Cased=-	y}', "");
Expect(1, 120780, '\p{^Is_Cased=-	y}', "");
Expect(1, 120780, '\P{Is_Cased=-	y}', "");
Expect(0, 120780, '\P{^Is_Cased=-	y}', "");
Error('\p{Cased= /a/T}');
Error('\P{Cased= /a/T}');
Expect(1, 120779, '\p{Cased=t}', "");
Expect(0, 120779, '\p{^Cased=t}', "");
Expect(0, 120779, '\P{Cased=t}', "");
Expect(1, 120779, '\P{^Cased=t}', "");
Expect(0, 120780, '\p{Cased=t}', "");
Expect(1, 120780, '\p{^Cased=t}', "");
Expect(1, 120780, '\P{Cased=t}', "");
Expect(0, 120780, '\P{^Cased=t}', "");
Expect(1, 120779, '\p{Cased=  t}', "");
Expect(0, 120779, '\p{^Cased=  t}', "");
Expect(0, 120779, '\P{Cased=  t}', "");
Expect(1, 120779, '\P{^Cased=  t}', "");
Expect(0, 120780, '\p{Cased=  t}', "");
Expect(1, 120780, '\p{^Cased=  t}', "");
Expect(1, 120780, '\P{Cased=  t}', "");
Expect(0, 120780, '\P{^Cased=  t}', "");
Error('\p{Is_Cased= True/a/}');
Error('\P{Is_Cased= True/a/}');
Expect(1, 120779, '\p{Is_Cased=true}', "");
Expect(0, 120779, '\p{^Is_Cased=true}', "");
Expect(0, 120779, '\P{Is_Cased=true}', "");
Expect(1, 120779, '\P{^Is_Cased=true}', "");
Expect(0, 120780, '\p{Is_Cased=true}', "");
Expect(1, 120780, '\p{^Is_Cased=true}', "");
Expect(1, 120780, '\P{Is_Cased=true}', "");
Expect(0, 120780, '\P{^Is_Cased=true}', "");
Expect(1, 120779, '\p{Is_Cased=	_True}', "");
Expect(0, 120779, '\p{^Is_Cased=	_True}', "");
Expect(0, 120779, '\P{Is_Cased=	_True}', "");
Expect(1, 120779, '\P{^Is_Cased=	_True}', "");
Expect(0, 120780, '\p{Is_Cased=	_True}', "");
Expect(1, 120780, '\p{^Is_Cased=	_True}', "");
Expect(1, 120780, '\P{Is_Cased=	_True}', "");
Expect(0, 120780, '\P{^Is_Cased=	_True}', "");
Error('\p{Full_Composition_Exclusion:   :=_ no}');
Error('\P{Full_Composition_Exclusion:   :=_ no}');
Expect(1, 195102, '\p{Full_Composition_Exclusion:no}', "");
Expect(0, 195102, '\p{^Full_Composition_Exclusion:no}', "");
Expect(0, 195102, '\P{Full_Composition_Exclusion:no}', "");
Expect(1, 195102, '\P{^Full_Composition_Exclusion:no}', "");
Expect(0, 195101, '\p{Full_Composition_Exclusion:no}', "");
Expect(1, 195101, '\p{^Full_Composition_Exclusion:no}', "");
Expect(1, 195101, '\P{Full_Composition_Exclusion:no}', "");
Expect(0, 195101, '\P{^Full_Composition_Exclusion:no}', "");
Expect(1, 195102, '\p{Full_Composition_Exclusion= 	No}', "");
Expect(0, 195102, '\p{^Full_Composition_Exclusion= 	No}', "");
Expect(0, 195102, '\P{Full_Composition_Exclusion= 	No}', "");
Expect(1, 195102, '\P{^Full_Composition_Exclusion= 	No}', "");
Expect(0, 195101, '\p{Full_Composition_Exclusion= 	No}', "");
Expect(1, 195101, '\p{^Full_Composition_Exclusion= 	No}', "");
Expect(1, 195101, '\P{Full_Composition_Exclusion= 	No}', "");
Expect(0, 195101, '\P{^Full_Composition_Exclusion= 	No}', "");
Error('\p{Comp_Ex=/a/_N}');
Error('\P{Comp_Ex=/a/_N}');
Expect(1, 195102, '\p{Comp_Ex=n}', "");
Expect(0, 195102, '\p{^Comp_Ex=n}', "");
Expect(0, 195102, '\P{Comp_Ex=n}', "");
Expect(1, 195102, '\P{^Comp_Ex=n}', "");
Expect(0, 195101, '\p{Comp_Ex=n}', "");
Expect(1, 195101, '\p{^Comp_Ex=n}', "");
Expect(1, 195101, '\P{Comp_Ex=n}', "");
Expect(0, 195101, '\P{^Comp_Ex=n}', "");
Expect(1, 195102, '\p{Comp_Ex=--n}', "");
Expect(0, 195102, '\p{^Comp_Ex=--n}', "");
Expect(0, 195102, '\P{Comp_Ex=--n}', "");
Expect(1, 195102, '\P{^Comp_Ex=--n}', "");
Expect(0, 195101, '\p{Comp_Ex=--n}', "");
Expect(1, 195101, '\p{^Comp_Ex=--n}', "");
Expect(1, 195101, '\P{Comp_Ex=--n}', "");
Expect(0, 195101, '\P{^Comp_Ex=--n}', "");
Error('\p{Is_Full_Composition_Exclusion:	/a/f}');
Error('\P{Is_Full_Composition_Exclusion:	/a/f}');
Expect(1, 195102, '\p{Is_Full_Composition_Exclusion=f}', "");
Expect(0, 195102, '\p{^Is_Full_Composition_Exclusion=f}', "");
Expect(0, 195102, '\P{Is_Full_Composition_Exclusion=f}', "");
Expect(1, 195102, '\P{^Is_Full_Composition_Exclusion=f}', "");
Expect(0, 195101, '\p{Is_Full_Composition_Exclusion=f}', "");
Expect(1, 195101, '\p{^Is_Full_Composition_Exclusion=f}', "");
Expect(1, 195101, '\P{Is_Full_Composition_Exclusion=f}', "");
Expect(0, 195101, '\P{^Is_Full_Composition_Exclusion=f}', "");
Expect(1, 195102, '\p{Is_Full_Composition_Exclusion:			F}', "");
Expect(0, 195102, '\p{^Is_Full_Composition_Exclusion:			F}', "");
Expect(0, 195102, '\P{Is_Full_Composition_Exclusion:			F}', "");
Expect(1, 195102, '\P{^Is_Full_Composition_Exclusion:			F}', "");
Expect(0, 195101, '\p{Is_Full_Composition_Exclusion:			F}', "");
Expect(1, 195101, '\p{^Is_Full_Composition_Exclusion:			F}', "");
Expect(1, 195101, '\P{Is_Full_Composition_Exclusion:			F}', "");
Expect(0, 195101, '\P{^Is_Full_Composition_Exclusion:			F}', "");
Error('\p{Is_Comp_Ex= FALSE:=}');
Error('\P{Is_Comp_Ex= FALSE:=}');
Expect(1, 195102, '\p{Is_Comp_Ex=false}', "");
Expect(0, 195102, '\p{^Is_Comp_Ex=false}', "");
Expect(0, 195102, '\P{Is_Comp_Ex=false}', "");
Expect(1, 195102, '\P{^Is_Comp_Ex=false}', "");
Expect(0, 195101, '\p{Is_Comp_Ex=false}', "");
Expect(1, 195101, '\p{^Is_Comp_Ex=false}', "");
Expect(1, 195101, '\P{Is_Comp_Ex=false}', "");
Expect(0, 195101, '\P{^Is_Comp_Ex=false}', "");
Expect(1, 195102, '\p{Is_Comp_Ex=	-FALSE}', "");
Expect(0, 195102, '\p{^Is_Comp_Ex=	-FALSE}', "");
Expect(0, 195102, '\P{Is_Comp_Ex=	-FALSE}', "");
Expect(1, 195102, '\P{^Is_Comp_Ex=	-FALSE}', "");
Expect(0, 195101, '\p{Is_Comp_Ex=	-FALSE}', "");
Expect(1, 195101, '\p{^Is_Comp_Ex=	-FALSE}', "");
Expect(1, 195101, '\P{Is_Comp_Ex=	-FALSE}', "");
Expect(0, 195101, '\P{^Is_Comp_Ex=	-FALSE}', "");
Error('\p{Full_Composition_Exclusion:	-yes:=}');
Error('\P{Full_Composition_Exclusion:	-yes:=}');
Expect(1, 195101, '\p{Full_Composition_Exclusion=yes}', "");
Expect(0, 195101, '\p{^Full_Composition_Exclusion=yes}', "");
Expect(0, 195101, '\P{Full_Composition_Exclusion=yes}', "");
Expect(1, 195101, '\P{^Full_Composition_Exclusion=yes}', "");
Expect(0, 195102, '\p{Full_Composition_Exclusion=yes}', "");
Expect(1, 195102, '\p{^Full_Composition_Exclusion=yes}', "");
Expect(1, 195102, '\P{Full_Composition_Exclusion=yes}', "");
Expect(0, 195102, '\P{^Full_Composition_Exclusion=yes}', "");
Expect(1, 195101, '\p{Full_Composition_Exclusion=- Yes}', "");
Expect(0, 195101, '\p{^Full_Composition_Exclusion=- Yes}', "");
Expect(0, 195101, '\P{Full_Composition_Exclusion=- Yes}', "");
Expect(1, 195101, '\P{^Full_Composition_Exclusion=- Yes}', "");
Expect(0, 195102, '\p{Full_Composition_Exclusion=- Yes}', "");
Expect(1, 195102, '\p{^Full_Composition_Exclusion=- Yes}', "");
Expect(1, 195102, '\P{Full_Composition_Exclusion=- Yes}', "");
Expect(0, 195102, '\P{^Full_Composition_Exclusion=- Yes}', "");
Error('\p{Comp_Ex=:=	 Y}');
Error('\P{Comp_Ex=:=	 Y}');
Expect(1, 195101, '\p{Comp_Ex=y}', "");
Expect(0, 195101, '\p{^Comp_Ex=y}', "");
Expect(0, 195101, '\P{Comp_Ex=y}', "");
Expect(1, 195101, '\P{^Comp_Ex=y}', "");
Expect(0, 195102, '\p{Comp_Ex=y}', "");
Expect(1, 195102, '\p{^Comp_Ex=y}', "");
Expect(1, 195102, '\P{Comp_Ex=y}', "");
Expect(0, 195102, '\P{^Comp_Ex=y}', "");
Expect(1, 195101, '\p{Comp_Ex=-_Y}', "");
Expect(0, 195101, '\p{^Comp_Ex=-_Y}', "");
Expect(0, 195101, '\P{Comp_Ex=-_Y}', "");
Expect(1, 195101, '\P{^Comp_Ex=-_Y}', "");
Expect(0, 195102, '\p{Comp_Ex=-_Y}', "");
Expect(1, 195102, '\p{^Comp_Ex=-_Y}', "");
Expect(1, 195102, '\P{Comp_Ex=-_Y}', "");
Expect(0, 195102, '\P{^Comp_Ex=-_Y}', "");
Error('\p{Is_Full_Composition_Exclusion=	:=t}');
Error('\P{Is_Full_Composition_Exclusion=	:=t}');
Expect(1, 195101, '\p{Is_Full_Composition_Exclusion=t}', "");
Expect(0, 195101, '\p{^Is_Full_Composition_Exclusion=t}', "");
Expect(0, 195101, '\P{Is_Full_Composition_Exclusion=t}', "");
Expect(1, 195101, '\P{^Is_Full_Composition_Exclusion=t}', "");
Expect(0, 195102, '\p{Is_Full_Composition_Exclusion=t}', "");
Expect(1, 195102, '\p{^Is_Full_Composition_Exclusion=t}', "");
Expect(1, 195102, '\P{Is_Full_Composition_Exclusion=t}', "");
Expect(0, 195102, '\P{^Is_Full_Composition_Exclusion=t}', "");
Expect(1, 195101, '\p{Is_Full_Composition_Exclusion=	T}', "");
Expect(0, 195101, '\p{^Is_Full_Composition_Exclusion=	T}', "");
Expect(0, 195101, '\P{Is_Full_Composition_Exclusion=	T}', "");
Expect(1, 195101, '\P{^Is_Full_Composition_Exclusion=	T}', "");
Expect(0, 195102, '\p{Is_Full_Composition_Exclusion=	T}', "");
Expect(1, 195102, '\p{^Is_Full_Composition_Exclusion=	T}', "");
Expect(1, 195102, '\P{Is_Full_Composition_Exclusion=	T}', "");
Expect(0, 195102, '\P{^Is_Full_Composition_Exclusion=	T}', "");
Error('\p{Is_Comp_Ex=- True/a/}');
Error('\P{Is_Comp_Ex=- True/a/}');
Expect(1, 195101, '\p{Is_Comp_Ex=true}', "");
Expect(0, 195101, '\p{^Is_Comp_Ex=true}', "");
Expect(0, 195101, '\P{Is_Comp_Ex=true}', "");
Expect(1, 195101, '\P{^Is_Comp_Ex=true}', "");
Expect(0, 195102, '\p{Is_Comp_Ex=true}', "");
Expect(1, 195102, '\p{^Is_Comp_Ex=true}', "");
Expect(1, 195102, '\P{Is_Comp_Ex=true}', "");
Expect(0, 195102, '\P{^Is_Comp_Ex=true}', "");
Expect(1, 195101, '\p{Is_Comp_Ex:	_TRUE}', "");
Expect(0, 195101, '\p{^Is_Comp_Ex:	_TRUE}', "");
Expect(0, 195101, '\P{Is_Comp_Ex:	_TRUE}', "");
Expect(1, 195101, '\P{^Is_Comp_Ex:	_TRUE}', "");
Expect(0, 195102, '\p{Is_Comp_Ex:	_TRUE}', "");
Expect(1, 195102, '\p{^Is_Comp_Ex:	_TRUE}', "");
Expect(1, 195102, '\P{Is_Comp_Ex:	_TRUE}', "");
Expect(0, 195102, '\P{^Is_Comp_Ex:	_TRUE}', "");
Error('\p{Default_Ignorable_Code_Point=/a/No}');
Error('\P{Default_Ignorable_Code_Point=/a/No}');
Expect(1, 921600, '\p{Default_Ignorable_Code_Point=no}', "");
Expect(0, 921600, '\p{^Default_Ignorable_Code_Point=no}', "");
Expect(0, 921600, '\P{Default_Ignorable_Code_Point=no}', "");
Expect(1, 921600, '\P{^Default_Ignorable_Code_Point=no}', "");
Expect(0, 921599, '\p{Default_Ignorable_Code_Point=no}', "");
Expect(1, 921599, '\p{^Default_Ignorable_Code_Point=no}', "");
Expect(1, 921599, '\P{Default_Ignorable_Code_Point=no}', "");
Expect(0, 921599, '\P{^Default_Ignorable_Code_Point=no}', "");
Expect(1, 921600, '\p{Default_Ignorable_Code_Point=	No}', "");
Expect(0, 921600, '\p{^Default_Ignorable_Code_Point=	No}', "");
Expect(0, 921600, '\P{Default_Ignorable_Code_Point=	No}', "");
Expect(1, 921600, '\P{^Default_Ignorable_Code_Point=	No}', "");
Expect(0, 921599, '\p{Default_Ignorable_Code_Point=	No}', "");
Expect(1, 921599, '\p{^Default_Ignorable_Code_Point=	No}', "");
Expect(1, 921599, '\P{Default_Ignorable_Code_Point=	No}', "");
Expect(0, 921599, '\P{^Default_Ignorable_Code_Point=	No}', "");
Error('\p{DI:-n/a/}');
Error('\P{DI:-n/a/}');
Expect(1, 921600, '\p{DI=n}', "");
Expect(0, 921600, '\p{^DI=n}', "");
Expect(0, 921600, '\P{DI=n}', "");
Expect(1, 921600, '\P{^DI=n}', "");
Expect(0, 921599, '\p{DI=n}', "");
Expect(1, 921599, '\p{^DI=n}', "");
Expect(1, 921599, '\P{DI=n}', "");
Expect(0, 921599, '\P{^DI=n}', "");
Expect(1, 921600, '\p{DI= n}', "");
Expect(0, 921600, '\p{^DI= n}', "");
Expect(0, 921600, '\P{DI= n}', "");
Expect(1, 921600, '\P{^DI= n}', "");
Expect(0, 921599, '\p{DI= n}', "");
Expect(1, 921599, '\p{^DI= n}', "");
Expect(1, 921599, '\P{DI= n}', "");
Expect(0, 921599, '\P{^DI= n}', "");
Error('\p{Is_Default_Ignorable_Code_Point:	-:=F}');
Error('\P{Is_Default_Ignorable_Code_Point:	-:=F}');
Expect(1, 921600, '\p{Is_Default_Ignorable_Code_Point=f}', "");
Expect(0, 921600, '\p{^Is_Default_Ignorable_Code_Point=f}', "");
Expect(0, 921600, '\P{Is_Default_Ignorable_Code_Point=f}', "");
Expect(1, 921600, '\P{^Is_Default_Ignorable_Code_Point=f}', "");
Expect(0, 921599, '\p{Is_Default_Ignorable_Code_Point=f}', "");
Expect(1, 921599, '\p{^Is_Default_Ignorable_Code_Point=f}', "");
Expect(1, 921599, '\P{Is_Default_Ignorable_Code_Point=f}', "");
Expect(0, 921599, '\P{^Is_Default_Ignorable_Code_Point=f}', "");
Expect(1, 921600, '\p{Is_Default_Ignorable_Code_Point= f}', "");
Expect(0, 921600, '\p{^Is_Default_Ignorable_Code_Point= f}', "");
Expect(0, 921600, '\P{Is_Default_Ignorable_Code_Point= f}', "");
Expect(1, 921600, '\P{^Is_Default_Ignorable_Code_Point= f}', "");
Expect(0, 921599, '\p{Is_Default_Ignorable_Code_Point= f}', "");
Expect(1, 921599, '\p{^Is_Default_Ignorable_Code_Point= f}', "");
Expect(1, 921599, '\P{Is_Default_Ignorable_Code_Point= f}', "");
Expect(0, 921599, '\P{^Is_Default_Ignorable_Code_Point= f}', "");
Error('\p{Is_DI=	:=FALSE}');
Error('\P{Is_DI=	:=FALSE}');
Expect(1, 921600, '\p{Is_DI=false}', "");
Expect(0, 921600, '\p{^Is_DI=false}', "");
Expect(0, 921600, '\P{Is_DI=false}', "");
Expect(1, 921600, '\P{^Is_DI=false}', "");
Expect(0, 921599, '\p{Is_DI=false}', "");
Expect(1, 921599, '\p{^Is_DI=false}', "");
Expect(1, 921599, '\P{Is_DI=false}', "");
Expect(0, 921599, '\P{^Is_DI=false}', "");
Expect(1, 921600, '\p{Is_DI=	_False}', "");
Expect(0, 921600, '\p{^Is_DI=	_False}', "");
Expect(0, 921600, '\P{Is_DI=	_False}', "");
Expect(1, 921600, '\P{^Is_DI=	_False}', "");
Expect(0, 921599, '\p{Is_DI=	_False}', "");
Expect(1, 921599, '\p{^Is_DI=	_False}', "");
Expect(1, 921599, '\P{Is_DI=	_False}', "");
Expect(0, 921599, '\P{^Is_DI=	_False}', "");
Error('\p{Default_Ignorable_Code_Point=:= -yes}');
Error('\P{Default_Ignorable_Code_Point=:= -yes}');
Expect(1, 921599, '\p{Default_Ignorable_Code_Point=yes}', "");
Expect(0, 921599, '\p{^Default_Ignorable_Code_Point=yes}', "");
Expect(0, 921599, '\P{Default_Ignorable_Code_Point=yes}', "");
Expect(1, 921599, '\P{^Default_Ignorable_Code_Point=yes}', "");
Expect(0, 921600, '\p{Default_Ignorable_Code_Point=yes}', "");
Expect(1, 921600, '\p{^Default_Ignorable_Code_Point=yes}', "");
Expect(1, 921600, '\P{Default_Ignorable_Code_Point=yes}', "");
Expect(0, 921600, '\P{^Default_Ignorable_Code_Point=yes}', "");
Expect(1, 921599, '\p{Default_Ignorable_Code_Point=_YES}', "");
Expect(0, 921599, '\p{^Default_Ignorable_Code_Point=_YES}', "");
Expect(0, 921599, '\P{Default_Ignorable_Code_Point=_YES}', "");
Expect(1, 921599, '\P{^Default_Ignorable_Code_Point=_YES}', "");
Expect(0, 921600, '\p{Default_Ignorable_Code_Point=_YES}', "");
Expect(1, 921600, '\p{^Default_Ignorable_Code_Point=_YES}', "");
Expect(1, 921600, '\P{Default_Ignorable_Code_Point=_YES}', "");
Expect(0, 921600, '\P{^Default_Ignorable_Code_Point=_YES}', "");
Error('\p{DI=_ Y:=}');
Error('\P{DI=_ Y:=}');
Expect(1, 921599, '\p{DI=y}', "");
Expect(0, 921599, '\p{^DI=y}', "");
Expect(0, 921599, '\P{DI=y}', "");
Expect(1, 921599, '\P{^DI=y}', "");
Expect(0, 921600, '\p{DI=y}', "");
Expect(1, 921600, '\p{^DI=y}', "");
Expect(1, 921600, '\P{DI=y}', "");
Expect(0, 921600, '\P{^DI=y}', "");
Expect(1, 921599, '\p{DI=  Y}', "");
Expect(0, 921599, '\p{^DI=  Y}', "");
Expect(0, 921599, '\P{DI=  Y}', "");
Expect(1, 921599, '\P{^DI=  Y}', "");
Expect(0, 921600, '\p{DI=  Y}', "");
Expect(1, 921600, '\p{^DI=  Y}', "");
Expect(1, 921600, '\P{DI=  Y}', "");
Expect(0, 921600, '\P{^DI=  Y}', "");
Error('\p{Is_Default_Ignorable_Code_Point=:=	-T}');
Error('\P{Is_Default_Ignorable_Code_Point=:=	-T}');
Expect(1, 921599, '\p{Is_Default_Ignorable_Code_Point=t}', "");
Expect(0, 921599, '\p{^Is_Default_Ignorable_Code_Point=t}', "");
Expect(0, 921599, '\P{Is_Default_Ignorable_Code_Point=t}', "");
Expect(1, 921599, '\P{^Is_Default_Ignorable_Code_Point=t}', "");
Expect(0, 921600, '\p{Is_Default_Ignorable_Code_Point=t}', "");
Expect(1, 921600, '\p{^Is_Default_Ignorable_Code_Point=t}', "");
Expect(1, 921600, '\P{Is_Default_Ignorable_Code_Point=t}', "");
Expect(0, 921600, '\P{^Is_Default_Ignorable_Code_Point=t}', "");
Expect(1, 921599, '\p{Is_Default_Ignorable_Code_Point= T}', "");
Expect(0, 921599, '\p{^Is_Default_Ignorable_Code_Point= T}', "");
Expect(0, 921599, '\P{Is_Default_Ignorable_Code_Point= T}', "");
Expect(1, 921599, '\P{^Is_Default_Ignorable_Code_Point= T}', "");
Expect(0, 921600, '\p{Is_Default_Ignorable_Code_Point= T}', "");
Expect(1, 921600, '\p{^Is_Default_Ignorable_Code_Point= T}', "");
Expect(1, 921600, '\P{Is_Default_Ignorable_Code_Point= T}', "");
Expect(0, 921600, '\P{^Is_Default_Ignorable_Code_Point= T}', "");
Error('\p{Is_DI=/a/ 	true}');
Error('\P{Is_DI=/a/ 	true}');
Expect(1, 921599, '\p{Is_DI=true}', "");
Expect(0, 921599, '\p{^Is_DI=true}', "");
Expect(0, 921599, '\P{Is_DI=true}', "");
Expect(1, 921599, '\P{^Is_DI=true}', "");
Expect(0, 921600, '\p{Is_DI=true}', "");
Expect(1, 921600, '\p{^Is_DI=true}', "");
Expect(1, 921600, '\P{Is_DI=true}', "");
Expect(0, 921600, '\P{^Is_DI=true}', "");
Expect(1, 921599, '\p{Is_DI=	True}', "");
Expect(0, 921599, '\p{^Is_DI=	True}', "");
Expect(0, 921599, '\P{Is_DI=	True}', "");
Expect(1, 921599, '\P{^Is_DI=	True}', "");
Expect(0, 921600, '\p{Is_DI=	True}', "");
Expect(1, 921600, '\p{^Is_DI=	True}', "");
Expect(1, 921600, '\P{Is_DI=	True}', "");
Expect(0, 921600, '\P{^Is_DI=	True}', "");
Error('\p{Dash=:=_ No}');
Error('\P{Dash=:=_ No}');
Expect(1, 65294, '\p{Dash:no}', "");
Expect(0, 65294, '\p{^Dash:no}', "");
Expect(0, 65294, '\P{Dash:no}', "");
Expect(1, 65294, '\P{^Dash:no}', "");
Expect(0, 65293, '\p{Dash:no}', "");
Expect(1, 65293, '\p{^Dash:no}', "");
Expect(1, 65293, '\P{Dash:no}', "");
Expect(0, 65293, '\P{^Dash:no}', "");
Expect(1, 65294, '\p{Dash=_-No}', "");
Expect(0, 65294, '\p{^Dash=_-No}', "");
Expect(0, 65294, '\P{Dash=_-No}', "");
Expect(1, 65294, '\P{^Dash=_-No}', "");
Expect(0, 65293, '\p{Dash=_-No}', "");
Expect(1, 65293, '\p{^Dash=_-No}', "");
Expect(1, 65293, '\P{Dash=_-No}', "");
Expect(0, 65293, '\P{^Dash=_-No}', "");
Error('\p{Is_Dash=_N:=}');
Error('\P{Is_Dash=_N:=}');
Expect(1, 65294, '\p{Is_Dash=n}', "");
Expect(0, 65294, '\p{^Is_Dash=n}', "");
Expect(0, 65294, '\P{Is_Dash=n}', "");
Expect(1, 65294, '\P{^Is_Dash=n}', "");
Expect(0, 65293, '\p{Is_Dash=n}', "");
Expect(1, 65293, '\p{^Is_Dash=n}', "");
Expect(1, 65293, '\P{Is_Dash=n}', "");
Expect(0, 65293, '\P{^Is_Dash=n}', "");
Expect(1, 65294, '\p{Is_Dash=-	N}', "");
Expect(0, 65294, '\p{^Is_Dash=-	N}', "");
Expect(0, 65294, '\P{Is_Dash=-	N}', "");
Expect(1, 65294, '\P{^Is_Dash=-	N}', "");
Expect(0, 65293, '\p{Is_Dash=-	N}', "");
Expect(1, 65293, '\p{^Is_Dash=-	N}', "");
Expect(1, 65293, '\P{Is_Dash=-	N}', "");
Expect(0, 65293, '\P{^Is_Dash=-	N}', "");
Error('\p{Dash:   := -f}');
Error('\P{Dash:   := -f}');
Expect(1, 65294, '\p{Dash=f}', "");
Expect(0, 65294, '\p{^Dash=f}', "");
Expect(0, 65294, '\P{Dash=f}', "");
Expect(1, 65294, '\P{^Dash=f}', "");
Expect(0, 65293, '\p{Dash=f}', "");
Expect(1, 65293, '\p{^Dash=f}', "");
Expect(1, 65293, '\P{Dash=f}', "");
Expect(0, 65293, '\P{^Dash=f}', "");
Expect(1, 65294, '\p{Dash=_F}', "");
Expect(0, 65294, '\p{^Dash=_F}', "");
Expect(0, 65294, '\P{Dash=_F}', "");
Expect(1, 65294, '\P{^Dash=_F}', "");
Expect(0, 65293, '\p{Dash=_F}', "");
Expect(1, 65293, '\p{^Dash=_F}', "");
Expect(1, 65293, '\P{Dash=_F}', "");
Expect(0, 65293, '\P{^Dash=_F}', "");
Error('\p{Is_Dash=:=--False}');
Error('\P{Is_Dash=:=--False}');
Expect(1, 65294, '\p{Is_Dash=false}', "");
Expect(0, 65294, '\p{^Is_Dash=false}', "");
Expect(0, 65294, '\P{Is_Dash=false}', "");
Expect(1, 65294, '\P{^Is_Dash=false}', "");
Expect(0, 65293, '\p{Is_Dash=false}', "");
Expect(1, 65293, '\p{^Is_Dash=false}', "");
Expect(1, 65293, '\P{Is_Dash=false}', "");
Expect(0, 65293, '\P{^Is_Dash=false}', "");
Expect(1, 65294, '\p{Is_Dash=_ False}', "");
Expect(0, 65294, '\p{^Is_Dash=_ False}', "");
Expect(0, 65294, '\P{Is_Dash=_ False}', "");
Expect(1, 65294, '\P{^Is_Dash=_ False}', "");
Expect(0, 65293, '\p{Is_Dash=_ False}', "");
Expect(1, 65293, '\p{^Is_Dash=_ False}', "");
Expect(1, 65293, '\P{Is_Dash=_ False}', "");
Expect(0, 65293, '\P{^Is_Dash=_ False}', "");
Error('\p{Dash=/a/ 	Yes}');
Error('\P{Dash=/a/ 	Yes}');
Expect(1, 65293, '\p{Dash=yes}', "");
Expect(0, 65293, '\p{^Dash=yes}', "");
Expect(0, 65293, '\P{Dash=yes}', "");
Expect(1, 65293, '\P{^Dash=yes}', "");
Expect(0, 65294, '\p{Dash=yes}', "");
Expect(1, 65294, '\p{^Dash=yes}', "");
Expect(1, 65294, '\P{Dash=yes}', "");
Expect(0, 65294, '\P{^Dash=yes}', "");
Expect(1, 65293, '\p{Dash: - Yes}', "");
Expect(0, 65293, '\p{^Dash: - Yes}', "");
Expect(0, 65293, '\P{Dash: - Yes}', "");
Expect(1, 65293, '\P{^Dash: - Yes}', "");
Expect(0, 65294, '\p{Dash: - Yes}', "");
Expect(1, 65294, '\p{^Dash: - Yes}', "");
Expect(1, 65294, '\P{Dash: - Yes}', "");
Expect(0, 65294, '\P{^Dash: - Yes}', "");
Error('\p{Is_Dash=:=-y}');
Error('\P{Is_Dash=:=-y}');
Expect(1, 65293, '\p{Is_Dash=y}', "");
Expect(0, 65293, '\p{^Is_Dash=y}', "");
Expect(0, 65293, '\P{Is_Dash=y}', "");
Expect(1, 65293, '\P{^Is_Dash=y}', "");
Expect(0, 65294, '\p{Is_Dash=y}', "");
Expect(1, 65294, '\p{^Is_Dash=y}', "");
Expect(1, 65294, '\P{Is_Dash=y}', "");
Expect(0, 65294, '\P{^Is_Dash=y}', "");
Expect(1, 65293, '\p{Is_Dash=_Y}', "");
Expect(0, 65293, '\p{^Is_Dash=_Y}', "");
Expect(0, 65293, '\P{Is_Dash=_Y}', "");
Expect(1, 65293, '\P{^Is_Dash=_Y}', "");
Expect(0, 65294, '\p{Is_Dash=_Y}', "");
Expect(1, 65294, '\p{^Is_Dash=_Y}', "");
Expect(1, 65294, '\P{Is_Dash=_Y}', "");
Expect(0, 65294, '\P{^Is_Dash=_Y}', "");
Error('\p{Dash=	/a/T}');
Error('\P{Dash=	/a/T}');
Expect(1, 65293, '\p{Dash=t}', "");
Expect(0, 65293, '\p{^Dash=t}', "");
Expect(0, 65293, '\P{Dash=t}', "");
Expect(1, 65293, '\P{^Dash=t}', "");
Expect(0, 65294, '\p{Dash=t}', "");
Expect(1, 65294, '\p{^Dash=t}', "");
Expect(1, 65294, '\P{Dash=t}', "");
Expect(0, 65294, '\P{^Dash=t}', "");
Expect(1, 65293, '\p{Dash=-t}', "");
Expect(0, 65293, '\p{^Dash=-t}', "");
Expect(0, 65293, '\P{Dash=-t}', "");
Expect(1, 65293, '\P{^Dash=-t}', "");
Expect(0, 65294, '\p{Dash=-t}', "");
Expect(1, 65294, '\p{^Dash=-t}', "");
Expect(1, 65294, '\P{Dash=-t}', "");
Expect(0, 65294, '\P{^Dash=-t}', "");
Error('\p{Is_Dash=	/a/True}');
Error('\P{Is_Dash=	/a/True}');
Expect(1, 65293, '\p{Is_Dash=true}', "");
Expect(0, 65293, '\p{^Is_Dash=true}', "");
Expect(0, 65293, '\P{Is_Dash=true}', "");
Expect(1, 65293, '\P{^Is_Dash=true}', "");
Expect(0, 65294, '\p{Is_Dash=true}', "");
Expect(1, 65294, '\p{^Is_Dash=true}', "");
Expect(1, 65294, '\P{Is_Dash=true}', "");
Expect(0, 65294, '\P{^Is_Dash=true}', "");
Expect(1, 65293, '\p{Is_Dash= true}', "");
Expect(0, 65293, '\p{^Is_Dash= true}', "");
Expect(0, 65293, '\P{Is_Dash= true}', "");
Expect(1, 65293, '\P{^Is_Dash= true}', "");
Expect(0, 65294, '\p{Is_Dash= true}', "");
Expect(1, 65294, '\p{^Is_Dash= true}', "");
Expect(1, 65294, '\P{Is_Dash= true}', "");
Expect(0, 65294, '\P{^Is_Dash= true}', "");
Error('\p{Deprecated= /a/NO}');
Error('\P{Deprecated= /a/NO}');
Expect(1, 917632, '\p{Deprecated=no}', "");
Expect(0, 917632, '\p{^Deprecated=no}', "");
Expect(0, 917632, '\P{Deprecated=no}', "");
Expect(1, 917632, '\P{^Deprecated=no}', "");
Expect(0, 917631, '\p{Deprecated=no}', "");
Expect(1, 917631, '\p{^Deprecated=no}', "");
Expect(1, 917631, '\P{Deprecated=no}', "");
Expect(0, 917631, '\P{^Deprecated=no}', "");
Expect(1, 917632, '\p{Deprecated= 	NO}', "");
Expect(0, 917632, '\p{^Deprecated= 	NO}', "");
Expect(0, 917632, '\P{Deprecated= 	NO}', "");
Expect(1, 917632, '\P{^Deprecated= 	NO}', "");
Expect(0, 917631, '\p{Deprecated= 	NO}', "");
Expect(1, 917631, '\p{^Deprecated= 	NO}', "");
Expect(1, 917631, '\P{Deprecated= 	NO}', "");
Expect(0, 917631, '\P{^Deprecated= 	NO}', "");
Error('\p{Dep= 	N:=}');
Error('\P{Dep= 	N:=}');
Expect(1, 917632, '\p{Dep=n}', "");
Expect(0, 917632, '\p{^Dep=n}', "");
Expect(0, 917632, '\P{Dep=n}', "");
Expect(1, 917632, '\P{^Dep=n}', "");
Expect(0, 917631, '\p{Dep=n}', "");
Expect(1, 917631, '\p{^Dep=n}', "");
Expect(1, 917631, '\P{Dep=n}', "");
Expect(0, 917631, '\P{^Dep=n}', "");
Expect(1, 917632, '\p{Dep: _N}', "");
Expect(0, 917632, '\p{^Dep: _N}', "");
Expect(0, 917632, '\P{Dep: _N}', "");
Expect(1, 917632, '\P{^Dep: _N}', "");
Expect(0, 917631, '\p{Dep: _N}', "");
Expect(1, 917631, '\p{^Dep: _N}', "");
Expect(1, 917631, '\P{Dep: _N}', "");
Expect(0, 917631, '\P{^Dep: _N}', "");
Error('\p{Is_Deprecated=_F/a/}');
Error('\P{Is_Deprecated=_F/a/}');
Expect(1, 917632, '\p{Is_Deprecated=f}', "");
Expect(0, 917632, '\p{^Is_Deprecated=f}', "");
Expect(0, 917632, '\P{Is_Deprecated=f}', "");
Expect(1, 917632, '\P{^Is_Deprecated=f}', "");
Expect(0, 917631, '\p{Is_Deprecated=f}', "");
Expect(1, 917631, '\p{^Is_Deprecated=f}', "");
Expect(1, 917631, '\P{Is_Deprecated=f}', "");
Expect(0, 917631, '\P{^Is_Deprecated=f}', "");
Expect(1, 917632, '\p{Is_Deprecated=  F}', "");
Expect(0, 917632, '\p{^Is_Deprecated=  F}', "");
Expect(0, 917632, '\P{Is_Deprecated=  F}', "");
Expect(1, 917632, '\P{^Is_Deprecated=  F}', "");
Expect(0, 917631, '\p{Is_Deprecated=  F}', "");
Expect(1, 917631, '\p{^Is_Deprecated=  F}', "");
Expect(1, 917631, '\P{Is_Deprecated=  F}', "");
Expect(0, 917631, '\P{^Is_Deprecated=  F}', "");
Error('\p{Is_Dep=:=	-FALSE}');
Error('\P{Is_Dep=:=	-FALSE}');
Expect(1, 917632, '\p{Is_Dep=false}', "");
Expect(0, 917632, '\p{^Is_Dep=false}', "");
Expect(0, 917632, '\P{Is_Dep=false}', "");
Expect(1, 917632, '\P{^Is_Dep=false}', "");
Expect(0, 917631, '\p{Is_Dep=false}', "");
Expect(1, 917631, '\p{^Is_Dep=false}', "");
Expect(1, 917631, '\P{Is_Dep=false}', "");
Expect(0, 917631, '\P{^Is_Dep=false}', "");
Expect(1, 917632, '\p{Is_Dep=_-false}', "");
Expect(0, 917632, '\p{^Is_Dep=_-false}', "");
Expect(0, 917632, '\P{Is_Dep=_-false}', "");
Expect(1, 917632, '\P{^Is_Dep=_-false}', "");
Expect(0, 917631, '\p{Is_Dep=_-false}', "");
Expect(1, 917631, '\p{^Is_Dep=_-false}', "");
Expect(1, 917631, '\P{Is_Dep=_-false}', "");
Expect(0, 917631, '\P{^Is_Dep=_-false}', "");
Error('\p{Deprecated=_/a/yes}');
Error('\P{Deprecated=_/a/yes}');
Expect(1, 917631, '\p{Deprecated=yes}', "");
Expect(0, 917631, '\p{^Deprecated=yes}', "");
Expect(0, 917631, '\P{Deprecated=yes}', "");
Expect(1, 917631, '\P{^Deprecated=yes}', "");
Expect(0, 917632, '\p{Deprecated=yes}', "");
Expect(1, 917632, '\p{^Deprecated=yes}', "");
Expect(1, 917632, '\P{Deprecated=yes}', "");
Expect(0, 917632, '\P{^Deprecated=yes}', "");
Expect(1, 917631, '\p{Deprecated=_YES}', "");
Expect(0, 917631, '\p{^Deprecated=_YES}', "");
Expect(0, 917631, '\P{Deprecated=_YES}', "");
Expect(1, 917631, '\P{^Deprecated=_YES}', "");
Expect(0, 917632, '\p{Deprecated=_YES}', "");
Expect(1, 917632, '\p{^Deprecated=_YES}', "");
Expect(1, 917632, '\P{Deprecated=_YES}', "");
Expect(0, 917632, '\P{^Deprecated=_YES}', "");
Error('\p{Dep= Y/a/}');
Error('\P{Dep= Y/a/}');
Expect(1, 917631, '\p{Dep=y}', "");
Expect(0, 917631, '\p{^Dep=y}', "");
Expect(0, 917631, '\P{Dep=y}', "");
Expect(1, 917631, '\P{^Dep=y}', "");
Expect(0, 917632, '\p{Dep=y}', "");
Expect(1, 917632, '\p{^Dep=y}', "");
Expect(1, 917632, '\P{Dep=y}', "");
Expect(0, 917632, '\P{^Dep=y}', "");
Expect(1, 917631, '\p{Dep= -Y}', "");
Expect(0, 917631, '\p{^Dep= -Y}', "");
Expect(0, 917631, '\P{Dep= -Y}', "");
Expect(1, 917631, '\P{^Dep= -Y}', "");
Expect(0, 917632, '\p{Dep= -Y}', "");
Expect(1, 917632, '\p{^Dep= -Y}', "");
Expect(1, 917632, '\P{Dep= -Y}', "");
Expect(0, 917632, '\P{^Dep= -Y}', "");
Error('\p{Is_Deprecated=_t/a/}');
Error('\P{Is_Deprecated=_t/a/}');
Expect(1, 917631, '\p{Is_Deprecated=t}', "");
Expect(0, 917631, '\p{^Is_Deprecated=t}', "");
Expect(0, 917631, '\P{Is_Deprecated=t}', "");
Expect(1, 917631, '\P{^Is_Deprecated=t}', "");
Expect(0, 917632, '\p{Is_Deprecated=t}', "");
Expect(1, 917632, '\p{^Is_Deprecated=t}', "");
Expect(1, 917632, '\P{Is_Deprecated=t}', "");
Expect(0, 917632, '\P{^Is_Deprecated=t}', "");
Expect(1, 917631, '\p{Is_Deprecated=  t}', "");
Expect(0, 917631, '\p{^Is_Deprecated=  t}', "");
Expect(0, 917631, '\P{Is_Deprecated=  t}', "");
Expect(1, 917631, '\P{^Is_Deprecated=  t}', "");
Expect(0, 917632, '\p{Is_Deprecated=  t}', "");
Expect(1, 917632, '\p{^Is_Deprecated=  t}', "");
Expect(1, 917632, '\P{Is_Deprecated=  t}', "");
Expect(0, 917632, '\P{^Is_Deprecated=  t}', "");
Error('\p{Is_Dep=/a/  TRUE}');
Error('\P{Is_Dep=/a/  TRUE}');
Expect(1, 917631, '\p{Is_Dep: true}', "");
Expect(0, 917631, '\p{^Is_Dep: true}', "");
Expect(0, 917631, '\P{Is_Dep: true}', "");
Expect(1, 917631, '\P{^Is_Dep: true}', "");
Expect(0, 917632, '\p{Is_Dep: true}', "");
Expect(1, 917632, '\p{^Is_Dep: true}', "");
Expect(1, 917632, '\P{Is_Dep: true}', "");
Expect(0, 917632, '\P{^Is_Dep: true}', "");
Expect(1, 917631, '\p{Is_Dep=_	TRUE}', "");
Expect(0, 917631, '\p{^Is_Dep=_	TRUE}', "");
Expect(0, 917631, '\P{Is_Dep=_	TRUE}', "");
Expect(1, 917631, '\P{^Is_Dep=_	TRUE}', "");
Expect(0, 917632, '\p{Is_Dep=_	TRUE}', "");
Expect(1, 917632, '\p{^Is_Dep=_	TRUE}', "");
Expect(1, 917632, '\P{Is_Dep=_	TRUE}', "");
Expect(0, 917632, '\P{^Is_Dep=_	TRUE}', "");
Error('\p{Diacritic=:=No}');
Error('\P{Diacritic=:=No}');
Expect(1, 119214, '\p{Diacritic=no}', "");
Expect(0, 119214, '\p{^Diacritic=no}', "");
Expect(0, 119214, '\P{Diacritic=no}', "");
Expect(1, 119214, '\P{^Diacritic=no}', "");
Expect(0, 119213, '\p{Diacritic=no}', "");
Expect(1, 119213, '\p{^Diacritic=no}', "");
Expect(1, 119213, '\P{Diacritic=no}', "");
Expect(0, 119213, '\P{^Diacritic=no}', "");
Expect(1, 119214, '\p{Diacritic=	_no}', "");
Expect(0, 119214, '\p{^Diacritic=	_no}', "");
Expect(0, 119214, '\P{Diacritic=	_no}', "");
Expect(1, 119214, '\P{^Diacritic=	_no}', "");
Expect(0, 119213, '\p{Diacritic=	_no}', "");
Expect(1, 119213, '\p{^Diacritic=	_no}', "");
Expect(1, 119213, '\P{Diacritic=	_no}', "");
Expect(0, 119213, '\P{^Diacritic=	_no}', "");
Error('\p{Dia=__N/a/}');
Error('\P{Dia=__N/a/}');
Expect(1, 119214, '\p{Dia=n}', "");
Expect(0, 119214, '\p{^Dia=n}', "");
Expect(0, 119214, '\P{Dia=n}', "");
Expect(1, 119214, '\P{^Dia=n}', "");
Expect(0, 119213, '\p{Dia=n}', "");
Expect(1, 119213, '\p{^Dia=n}', "");
Expect(1, 119213, '\P{Dia=n}', "");
Expect(0, 119213, '\P{^Dia=n}', "");
Expect(1, 119214, '\p{Dia=_N}', "");
Expect(0, 119214, '\p{^Dia=_N}', "");
Expect(0, 119214, '\P{Dia=_N}', "");
Expect(1, 119214, '\P{^Dia=_N}', "");
Expect(0, 119213, '\p{Dia=_N}', "");
Expect(1, 119213, '\p{^Dia=_N}', "");
Expect(1, 119213, '\P{Dia=_N}', "");
Expect(0, 119213, '\P{^Dia=_N}', "");
Error('\p{Is_Diacritic: F/a/}');
Error('\P{Is_Diacritic: F/a/}');
Expect(1, 119214, '\p{Is_Diacritic=f}', "");
Expect(0, 119214, '\p{^Is_Diacritic=f}', "");
Expect(0, 119214, '\P{Is_Diacritic=f}', "");
Expect(1, 119214, '\P{^Is_Diacritic=f}', "");
Expect(0, 119213, '\p{Is_Diacritic=f}', "");
Expect(1, 119213, '\p{^Is_Diacritic=f}', "");
Expect(1, 119213, '\P{Is_Diacritic=f}', "");
Expect(0, 119213, '\P{^Is_Diacritic=f}', "");
Expect(1, 119214, '\p{Is_Diacritic= f}', "");
Expect(0, 119214, '\p{^Is_Diacritic= f}', "");
Expect(0, 119214, '\P{Is_Diacritic= f}', "");
Expect(1, 119214, '\P{^Is_Diacritic= f}', "");
Expect(0, 119213, '\p{Is_Diacritic= f}', "");
Expect(1, 119213, '\p{^Is_Diacritic= f}', "");
Expect(1, 119213, '\P{Is_Diacritic= f}', "");
Expect(0, 119213, '\P{^Is_Diacritic= f}', "");
Error('\p{Is_Dia=:=-False}');
Error('\P{Is_Dia=:=-False}');
Expect(1, 119214, '\p{Is_Dia=false}', "");
Expect(0, 119214, '\p{^Is_Dia=false}', "");
Expect(0, 119214, '\P{Is_Dia=false}', "");
Expect(1, 119214, '\P{^Is_Dia=false}', "");
Expect(0, 119213, '\p{Is_Dia=false}', "");
Expect(1, 119213, '\p{^Is_Dia=false}', "");
Expect(1, 119213, '\P{Is_Dia=false}', "");
Expect(0, 119213, '\P{^Is_Dia=false}', "");
Expect(1, 119214, '\p{Is_Dia:   _	False}', "");
Expect(0, 119214, '\p{^Is_Dia:   _	False}', "");
Expect(0, 119214, '\P{Is_Dia:   _	False}', "");
Expect(1, 119214, '\P{^Is_Dia:   _	False}', "");
Expect(0, 119213, '\p{Is_Dia:   _	False}', "");
Expect(1, 119213, '\p{^Is_Dia:   _	False}', "");
Expect(1, 119213, '\P{Is_Dia:   _	False}', "");
Expect(0, 119213, '\P{^Is_Dia:   _	False}', "");
Error('\p{Diacritic::=-	Yes}');
Error('\P{Diacritic::=-	Yes}');
Expect(1, 119213, '\p{Diacritic=yes}', "");
Expect(0, 119213, '\p{^Diacritic=yes}', "");
Expect(0, 119213, '\P{Diacritic=yes}', "");
Expect(1, 119213, '\P{^Diacritic=yes}', "");
Expect(0, 119214, '\p{Diacritic=yes}', "");
Expect(1, 119214, '\p{^Diacritic=yes}', "");
Expect(1, 119214, '\P{Diacritic=yes}', "");
Expect(0, 119214, '\P{^Diacritic=yes}', "");
Expect(1, 119213, '\p{Diacritic=-	Yes}', "");
Expect(0, 119213, '\p{^Diacritic=-	Yes}', "");
Expect(0, 119213, '\P{Diacritic=-	Yes}', "");
Expect(1, 119213, '\P{^Diacritic=-	Yes}', "");
Expect(0, 119214, '\p{Diacritic=-	Yes}', "");
Expect(1, 119214, '\p{^Diacritic=-	Yes}', "");
Expect(1, 119214, '\P{Diacritic=-	Yes}', "");
Expect(0, 119214, '\P{^Diacritic=-	Yes}', "");
Error('\p{Dia=:=	Y}');
Error('\P{Dia=:=	Y}');
Expect(1, 119213, '\p{Dia=y}', "");
Expect(0, 119213, '\p{^Dia=y}', "");
Expect(0, 119213, '\P{Dia=y}', "");
Expect(1, 119213, '\P{^Dia=y}', "");
Expect(0, 119214, '\p{Dia=y}', "");
Expect(1, 119214, '\p{^Dia=y}', "");
Expect(1, 119214, '\P{Dia=y}', "");
Expect(0, 119214, '\P{^Dia=y}', "");
Expect(1, 119213, '\p{Dia=-_Y}', "");
Expect(0, 119213, '\p{^Dia=-_Y}', "");
Expect(0, 119213, '\P{Dia=-_Y}', "");
Expect(1, 119213, '\P{^Dia=-_Y}', "");
Expect(0, 119214, '\p{Dia=-_Y}', "");
Expect(1, 119214, '\p{^Dia=-_Y}', "");
Expect(1, 119214, '\P{Dia=-_Y}', "");
Expect(0, 119214, '\P{^Dia=-_Y}', "");
Error('\p{Is_Diacritic=:= T}');
Error('\P{Is_Diacritic=:= T}');
Expect(1, 119213, '\p{Is_Diacritic=t}', "");
Expect(0, 119213, '\p{^Is_Diacritic=t}', "");
Expect(0, 119213, '\P{Is_Diacritic=t}', "");
Expect(1, 119213, '\P{^Is_Diacritic=t}', "");
Expect(0, 119214, '\p{Is_Diacritic=t}', "");
Expect(1, 119214, '\p{^Is_Diacritic=t}', "");
Expect(1, 119214, '\P{Is_Diacritic=t}', "");
Expect(0, 119214, '\P{^Is_Diacritic=t}', "");
Expect(1, 119213, '\p{Is_Diacritic=  T}', "");
Expect(0, 119213, '\p{^Is_Diacritic=  T}', "");
Expect(0, 119213, '\P{Is_Diacritic=  T}', "");
Expect(1, 119213, '\P{^Is_Diacritic=  T}', "");
Expect(0, 119214, '\p{Is_Diacritic=  T}', "");
Expect(1, 119214, '\p{^Is_Diacritic=  T}', "");
Expect(1, 119214, '\P{Is_Diacritic=  T}', "");
Expect(0, 119214, '\P{^Is_Diacritic=  T}', "");
Error('\p{Is_Dia=:=True}');
Error('\P{Is_Dia=:=True}');
Expect(1, 119213, '\p{Is_Dia:   true}', "");
Expect(0, 119213, '\p{^Is_Dia:   true}', "");
Expect(0, 119213, '\P{Is_Dia:   true}', "");
Expect(1, 119213, '\P{^Is_Dia:   true}', "");
Expect(0, 119214, '\p{Is_Dia:   true}', "");
Expect(1, 119214, '\p{^Is_Dia:   true}', "");
Expect(1, 119214, '\P{Is_Dia:   true}', "");
Expect(0, 119214, '\P{^Is_Dia:   true}', "");
Expect(1, 119213, '\p{Is_Dia=	_True}', "");
Expect(0, 119213, '\p{^Is_Dia=	_True}', "");
Expect(0, 119213, '\P{Is_Dia=	_True}', "");
Expect(1, 119213, '\P{^Is_Dia=	_True}', "");
Expect(0, 119214, '\p{Is_Dia=	_True}', "");
Expect(1, 119214, '\p{^Is_Dia=	_True}', "");
Expect(1, 119214, '\P{Is_Dia=	_True}', "");
Expect(0, 119214, '\P{^Is_Dia=	_True}', "");
Error('\p{Extender=/a/no}');
Error('\P{Extender=/a/no}');
Expect(1, 65393, '\p{Extender=no}', "");
Expect(0, 65393, '\p{^Extender=no}', "");
Expect(0, 65393, '\P{Extender=no}', "");
Expect(1, 65393, '\P{^Extender=no}', "");
Expect(0, 65392, '\p{Extender=no}', "");
Expect(1, 65392, '\p{^Extender=no}', "");
Expect(1, 65392, '\P{Extender=no}', "");
Expect(0, 65392, '\P{^Extender=no}', "");
Expect(1, 65393, '\p{Extender:	_no}', "");
Expect(0, 65393, '\p{^Extender:	_no}', "");
Expect(0, 65393, '\P{Extender:	_no}', "");
Expect(1, 65393, '\P{^Extender:	_no}', "");
Expect(0, 65392, '\p{Extender:	_no}', "");
Expect(1, 65392, '\p{^Extender:	_no}', "");
Expect(1, 65392, '\P{Extender:	_no}', "");
Expect(0, 65392, '\P{^Extender:	_no}', "");
Error('\p{Ext=_/a/N}');
Error('\P{Ext=_/a/N}');
Expect(1, 65393, '\p{Ext=n}', "");
Expect(0, 65393, '\p{^Ext=n}', "");
Expect(0, 65393, '\P{Ext=n}', "");
Expect(1, 65393, '\P{^Ext=n}', "");
Expect(0, 65392, '\p{Ext=n}', "");
Expect(1, 65392, '\p{^Ext=n}', "");
Expect(1, 65392, '\P{Ext=n}', "");
Expect(0, 65392, '\P{^Ext=n}', "");
Expect(1, 65393, '\p{Ext=_N}', "");
Expect(0, 65393, '\p{^Ext=_N}', "");
Expect(0, 65393, '\P{Ext=_N}', "");
Expect(1, 65393, '\P{^Ext=_N}', "");
Expect(0, 65392, '\p{Ext=_N}', "");
Expect(1, 65392, '\p{^Ext=_N}', "");
Expect(1, 65392, '\P{Ext=_N}', "");
Expect(0, 65392, '\P{^Ext=_N}', "");
Error('\p{Is_Extender=	 f/a/}');
Error('\P{Is_Extender=	 f/a/}');
Expect(1, 65393, '\p{Is_Extender:   f}', "");
Expect(0, 65393, '\p{^Is_Extender:   f}', "");
Expect(0, 65393, '\P{Is_Extender:   f}', "");
Expect(1, 65393, '\P{^Is_Extender:   f}', "");
Expect(0, 65392, '\p{Is_Extender:   f}', "");
Expect(1, 65392, '\p{^Is_Extender:   f}', "");
Expect(1, 65392, '\P{Is_Extender:   f}', "");
Expect(0, 65392, '\P{^Is_Extender:   f}', "");
Expect(1, 65393, '\p{Is_Extender=-_F}', "");
Expect(0, 65393, '\p{^Is_Extender=-_F}', "");
Expect(0, 65393, '\P{Is_Extender=-_F}', "");
Expect(1, 65393, '\P{^Is_Extender=-_F}', "");
Expect(0, 65392, '\p{Is_Extender=-_F}', "");
Expect(1, 65392, '\p{^Is_Extender=-_F}', "");
Expect(1, 65392, '\P{Is_Extender=-_F}', "");
Expect(0, 65392, '\P{^Is_Extender=-_F}', "");
Error('\p{Is_Ext=--false/a/}');
Error('\P{Is_Ext=--false/a/}');
Expect(1, 65393, '\p{Is_Ext=false}', "");
Expect(0, 65393, '\p{^Is_Ext=false}', "");
Expect(0, 65393, '\P{Is_Ext=false}', "");
Expect(1, 65393, '\P{^Is_Ext=false}', "");
Expect(0, 65392, '\p{Is_Ext=false}', "");
Expect(1, 65392, '\p{^Is_Ext=false}', "");
Expect(1, 65392, '\P{Is_Ext=false}', "");
Expect(0, 65392, '\P{^Is_Ext=false}', "");
Expect(1, 65393, '\p{Is_Ext= -False}', "");
Expect(0, 65393, '\p{^Is_Ext= -False}', "");
Expect(0, 65393, '\P{Is_Ext= -False}', "");
Expect(1, 65393, '\P{^Is_Ext= -False}', "");
Expect(0, 65392, '\p{Is_Ext= -False}', "");
Expect(1, 65392, '\p{^Is_Ext= -False}', "");
Expect(1, 65392, '\P{Is_Ext= -False}', "");
Expect(0, 65392, '\P{^Is_Ext= -False}', "");
Error('\p{Extender= /a/Yes}');
Error('\P{Extender= /a/Yes}');
Expect(1, 65392, '\p{Extender=yes}', "");
Expect(0, 65392, '\p{^Extender=yes}', "");
Expect(0, 65392, '\P{Extender=yes}', "");
Expect(1, 65392, '\P{^Extender=yes}', "");
Expect(0, 65393, '\p{Extender=yes}', "");
Expect(1, 65393, '\p{^Extender=yes}', "");
Expect(1, 65393, '\P{Extender=yes}', "");
Expect(0, 65393, '\P{^Extender=yes}', "");
Expect(1, 65392, '\p{Extender=  Yes}', "");
Expect(0, 65392, '\p{^Extender=  Yes}', "");
Expect(0, 65392, '\P{Extender=  Yes}', "");
Expect(1, 65392, '\P{^Extender=  Yes}', "");
Expect(0, 65393, '\p{Extender=  Yes}', "");
Expect(1, 65393, '\p{^Extender=  Yes}', "");
Expect(1, 65393, '\P{Extender=  Yes}', "");
Expect(0, 65393, '\P{^Extender=  Yes}', "");
Error('\p{Ext=/a/- Y}');
Error('\P{Ext=/a/- Y}');
Expect(1, 65392, '\p{Ext=y}', "");
Expect(0, 65392, '\p{^Ext=y}', "");
Expect(0, 65392, '\P{Ext=y}', "");
Expect(1, 65392, '\P{^Ext=y}', "");
Expect(0, 65393, '\p{Ext=y}', "");
Expect(1, 65393, '\p{^Ext=y}', "");
Expect(1, 65393, '\P{Ext=y}', "");
Expect(0, 65393, '\P{^Ext=y}', "");
Expect(1, 65392, '\p{Ext=  y}', "");
Expect(0, 65392, '\p{^Ext=  y}', "");
Expect(0, 65392, '\P{Ext=  y}', "");
Expect(1, 65392, '\P{^Ext=  y}', "");
Expect(0, 65393, '\p{Ext=  y}', "");
Expect(1, 65393, '\p{^Ext=  y}', "");
Expect(1, 65393, '\P{Ext=  y}', "");
Expect(0, 65393, '\P{^Ext=  y}', "");
Error('\p{Is_Extender=:=	T}');
Error('\P{Is_Extender=:=	T}');
Expect(1, 65392, '\p{Is_Extender=t}', "");
Expect(0, 65392, '\p{^Is_Extender=t}', "");
Expect(0, 65392, '\P{Is_Extender=t}', "");
Expect(1, 65392, '\P{^Is_Extender=t}', "");
Expect(0, 65393, '\p{Is_Extender=t}', "");
Expect(1, 65393, '\p{^Is_Extender=t}', "");
Expect(1, 65393, '\P{Is_Extender=t}', "");
Expect(0, 65393, '\P{^Is_Extender=t}', "");
Expect(1, 65392, '\p{Is_Extender= T}', "");
Expect(0, 65392, '\p{^Is_Extender= T}', "");
Expect(0, 65392, '\P{Is_Extender= T}', "");
Expect(1, 65392, '\P{^Is_Extender= T}', "");
Expect(0, 65393, '\p{Is_Extender= T}', "");
Expect(1, 65393, '\p{^Is_Extender= T}', "");
Expect(1, 65393, '\P{Is_Extender= T}', "");
Expect(0, 65393, '\P{^Is_Extender= T}', "");
Error('\p{Is_Ext=- true/a/}');
Error('\P{Is_Ext=- true/a/}');
Expect(1, 65392, '\p{Is_Ext=true}', "");
Expect(0, 65392, '\p{^Is_Ext=true}', "");
Expect(0, 65392, '\P{Is_Ext=true}', "");
Expect(1, 65392, '\P{^Is_Ext=true}', "");
Expect(0, 65393, '\p{Is_Ext=true}', "");
Expect(1, 65393, '\p{^Is_Ext=true}', "");
Expect(1, 65393, '\P{Is_Ext=true}', "");
Expect(0, 65393, '\P{^Is_Ext=true}', "");
Expect(1, 65392, '\p{Is_Ext=True}', "");
Expect(0, 65392, '\p{^Is_Ext=True}', "");
Expect(0, 65392, '\P{Is_Ext=True}', "");
Expect(1, 65392, '\P{^Is_Ext=True}', "");
Expect(0, 65393, '\p{Is_Ext=True}', "");
Expect(1, 65393, '\p{^Is_Ext=True}', "");
Expect(1, 65393, '\P{Is_Ext=True}', "");
Expect(0, 65393, '\P{^Is_Ext=True}', "");
Error('\p{Grapheme_Cluster_Break=/a/	_CONTROL}');
Error('\P{Grapheme_Cluster_Break=/a/	_CONTROL}');
Expect(1, 921599, '\p{Grapheme_Cluster_Break=control}', "");
Expect(0, 921599, '\p{^Grapheme_Cluster_Break=control}', "");
Expect(0, 921599, '\P{Grapheme_Cluster_Break=control}', "");
Expect(1, 921599, '\P{^Grapheme_Cluster_Break=control}', "");
Expect(0, 921600, '\p{Grapheme_Cluster_Break=control}', "");
Expect(1, 921600, '\p{^Grapheme_Cluster_Break=control}', "");
Expect(1, 921600, '\P{Grapheme_Cluster_Break=control}', "");
Expect(0, 921600, '\P{^Grapheme_Cluster_Break=control}', "");
Expect(1, 921599, '\p{Grapheme_Cluster_Break=Control}', "");
Expect(0, 921599, '\p{^Grapheme_Cluster_Break=Control}', "");
Expect(0, 921599, '\P{Grapheme_Cluster_Break=Control}', "");
Expect(1, 921599, '\P{^Grapheme_Cluster_Break=Control}', "");
Expect(0, 921600, '\p{Grapheme_Cluster_Break=Control}', "");
Expect(1, 921600, '\p{^Grapheme_Cluster_Break=Control}', "");
Expect(1, 921600, '\P{Grapheme_Cluster_Break=Control}', "");
Expect(0, 921600, '\P{^Grapheme_Cluster_Break=Control}', "");
Error('\p{GCB=  CN/a/}');
Error('\P{GCB=  CN/a/}');
Expect(1, 921599, '\p{GCB=cn}', "");
Expect(0, 921599, '\p{^GCB=cn}', "");
Expect(0, 921599, '\P{GCB=cn}', "");
Expect(1, 921599, '\P{^GCB=cn}', "");
Expect(0, 921600, '\p{GCB=cn}', "");
Expect(1, 921600, '\p{^GCB=cn}', "");
Expect(1, 921600, '\P{GCB=cn}', "");
Expect(0, 921600, '\P{^GCB=cn}', "");
Expect(1, 921599, '\p{GCB= CN}', "");
Expect(0, 921599, '\p{^GCB= CN}', "");
Expect(0, 921599, '\P{GCB= CN}', "");
Expect(1, 921599, '\P{^GCB= CN}', "");
Expect(0, 921600, '\p{GCB= CN}', "");
Expect(1, 921600, '\p{^GCB= CN}', "");
Expect(1, 921600, '\P{GCB= CN}', "");
Expect(0, 921600, '\P{^GCB= CN}', "");
Error('\p{Is_Grapheme_Cluster_Break=:=_CONTROL}');
Error('\P{Is_Grapheme_Cluster_Break=:=_CONTROL}');
Expect(1, 921599, '\p{Is_Grapheme_Cluster_Break=control}', "");
Expect(0, 921599, '\p{^Is_Grapheme_Cluster_Break=control}', "");
Expect(0, 921599, '\P{Is_Grapheme_Cluster_Break=control}', "");
Expect(1, 921599, '\P{^Is_Grapheme_Cluster_Break=control}', "");
Expect(0, 921600, '\p{Is_Grapheme_Cluster_Break=control}', "");
Expect(1, 921600, '\p{^Is_Grapheme_Cluster_Break=control}', "");
Expect(1, 921600, '\P{Is_Grapheme_Cluster_Break=control}', "");
Expect(0, 921600, '\P{^Is_Grapheme_Cluster_Break=control}', "");
Expect(1, 921599, '\p{Is_Grapheme_Cluster_Break=_Control}', "");
Expect(0, 921599, '\p{^Is_Grapheme_Cluster_Break=_Control}', "");
Expect(0, 921599, '\P{Is_Grapheme_Cluster_Break=_Control}', "");
Expect(1, 921599, '\P{^Is_Grapheme_Cluster_Break=_Control}', "");
Expect(0, 921600, '\p{Is_Grapheme_Cluster_Break=_Control}', "");
Expect(1, 921600, '\p{^Is_Grapheme_Cluster_Break=_Control}', "");
Expect(1, 921600, '\P{Is_Grapheme_Cluster_Break=_Control}', "");
Expect(0, 921600, '\P{^Is_Grapheme_Cluster_Break=_Control}', "");
Error('\p{Is_GCB=		cn:=}');
Error('\P{Is_GCB=		cn:=}');
Expect(1, 921599, '\p{Is_GCB=cn}', "");
Expect(0, 921599, '\p{^Is_GCB=cn}', "");
Expect(0, 921599, '\P{Is_GCB=cn}', "");
Expect(1, 921599, '\P{^Is_GCB=cn}', "");
Expect(0, 921600, '\p{Is_GCB=cn}', "");
Expect(1, 921600, '\p{^Is_GCB=cn}', "");
Expect(1, 921600, '\P{Is_GCB=cn}', "");
Expect(0, 921600, '\P{^Is_GCB=cn}', "");
Expect(1, 921599, '\p{Is_GCB=	 CN}', "");
Expect(0, 921599, '\p{^Is_GCB=	 CN}', "");
Expect(0, 921599, '\P{Is_GCB=	 CN}', "");
Expect(1, 921599, '\P{^Is_GCB=	 CN}', "");
Expect(0, 921600, '\p{Is_GCB=	 CN}', "");
Expect(1, 921600, '\p{^Is_GCB=	 CN}', "");
Expect(1, 921600, '\P{Is_GCB=	 CN}', "");
Expect(0, 921600, '\P{^Is_GCB=	 CN}', "");
Error('\p{Grapheme_Cluster_Break= :=CR}');
Error('\P{Grapheme_Cluster_Break= :=CR}');
Expect(1, 13, '\p{Grapheme_Cluster_Break=cr}', "");
Expect(0, 13, '\p{^Grapheme_Cluster_Break=cr}', "");
Expect(0, 13, '\P{Grapheme_Cluster_Break=cr}', "");
Expect(1, 13, '\P{^Grapheme_Cluster_Break=cr}', "");
Expect(0, 14, '\p{Grapheme_Cluster_Break=cr}', "");
Expect(1, 14, '\p{^Grapheme_Cluster_Break=cr}', "");
Expect(1, 14, '\P{Grapheme_Cluster_Break=cr}', "");
Expect(0, 14, '\P{^Grapheme_Cluster_Break=cr}', "");
Expect(1, 13, '\p{Grapheme_Cluster_Break= -CR}', "");
Expect(0, 13, '\p{^Grapheme_Cluster_Break= -CR}', "");
Expect(0, 13, '\P{Grapheme_Cluster_Break= -CR}', "");
Expect(1, 13, '\P{^Grapheme_Cluster_Break= -CR}', "");
Expect(0, 14, '\p{Grapheme_Cluster_Break= -CR}', "");
Expect(1, 14, '\p{^Grapheme_Cluster_Break= -CR}', "");
Expect(1, 14, '\P{Grapheme_Cluster_Break= -CR}', "");
Expect(0, 14, '\P{^Grapheme_Cluster_Break= -CR}', "");
Error('\p{GCB=	_CR:=}');
Error('\P{GCB=	_CR:=}');
Expect(1, 13, '\p{GCB=cr}', "");
Expect(0, 13, '\p{^GCB=cr}', "");
Expect(0, 13, '\P{GCB=cr}', "");
Expect(1, 13, '\P{^GCB=cr}', "");
Expect(0, 14, '\p{GCB=cr}', "");
Expect(1, 14, '\p{^GCB=cr}', "");
Expect(1, 14, '\P{GCB=cr}', "");
Expect(0, 14, '\P{^GCB=cr}', "");
Expect(1, 13, '\p{GCB=-CR}', "");
Expect(0, 13, '\p{^GCB=-CR}', "");
Expect(0, 13, '\P{GCB=-CR}', "");
Expect(1, 13, '\P{^GCB=-CR}', "");
Expect(0, 14, '\p{GCB=-CR}', "");
Expect(1, 14, '\p{^GCB=-CR}', "");
Expect(1, 14, '\P{GCB=-CR}', "");
Expect(0, 14, '\P{^GCB=-CR}', "");
Error('\p{Is_Grapheme_Cluster_Break=/a/	-cr}');
Error('\P{Is_Grapheme_Cluster_Break=/a/	-cr}');
Expect(1, 13, '\p{Is_Grapheme_Cluster_Break=cr}', "");
Expect(0, 13, '\p{^Is_Grapheme_Cluster_Break=cr}', "");
Expect(0, 13, '\P{Is_Grapheme_Cluster_Break=cr}', "");
Expect(1, 13, '\P{^Is_Grapheme_Cluster_Break=cr}', "");
Expect(0, 14, '\p{Is_Grapheme_Cluster_Break=cr}', "");
Expect(1, 14, '\p{^Is_Grapheme_Cluster_Break=cr}', "");
Expect(1, 14, '\P{Is_Grapheme_Cluster_Break=cr}', "");
Expect(0, 14, '\P{^Is_Grapheme_Cluster_Break=cr}', "");
Expect(1, 13, '\p{Is_Grapheme_Cluster_Break=_CR}', "");
Expect(0, 13, '\p{^Is_Grapheme_Cluster_Break=_CR}', "");
Expect(0, 13, '\P{Is_Grapheme_Cluster_Break=_CR}', "");
Expect(1, 13, '\P{^Is_Grapheme_Cluster_Break=_CR}', "");
Expect(0, 14, '\p{Is_Grapheme_Cluster_Break=_CR}', "");
Expect(1, 14, '\p{^Is_Grapheme_Cluster_Break=_CR}', "");
Expect(1, 14, '\P{Is_Grapheme_Cluster_Break=_CR}', "");
Expect(0, 14, '\P{^Is_Grapheme_Cluster_Break=_CR}', "");
Error('\p{Is_GCB=-CR:=}');
Error('\P{Is_GCB=-CR:=}');
Expect(1, 13, '\p{Is_GCB=cr}', "");
Expect(0, 13, '\p{^Is_GCB=cr}', "");
Expect(0, 13, '\P{Is_GCB=cr}', "");
Expect(1, 13, '\P{^Is_GCB=cr}', "");
Expect(0, 14, '\p{Is_GCB=cr}', "");
Expect(1, 14, '\p{^Is_GCB=cr}', "");
Expect(1, 14, '\P{Is_GCB=cr}', "");
Expect(0, 14, '\P{^Is_GCB=cr}', "");
Expect(1, 13, '\p{Is_GCB=	 CR}', "");
Expect(0, 13, '\p{^Is_GCB=	 CR}', "");
Expect(0, 13, '\P{Is_GCB=	 CR}', "");
Expect(1, 13, '\P{^Is_GCB=	 CR}', "");
Expect(0, 14, '\p{Is_GCB=	 CR}', "");
Expect(1, 14, '\p{^Is_GCB=	 CR}', "");
Expect(1, 14, '\P{Is_GCB=	 CR}', "");
Expect(0, 14, '\P{^Is_GCB=	 CR}', "");
Error('\p{Grapheme_Cluster_Break: -:=EXTEND}');
Error('\P{Grapheme_Cluster_Break: -:=EXTEND}');
Expect(1, 917999, '\p{Grapheme_Cluster_Break=extend}', "");
Expect(0, 917999, '\p{^Grapheme_Cluster_Break=extend}', "");
Expect(0, 917999, '\P{Grapheme_Cluster_Break=extend}', "");
Expect(1, 917999, '\P{^Grapheme_Cluster_Break=extend}', "");
Expect(0, 918000, '\p{Grapheme_Cluster_Break=extend}', "");
Expect(1, 918000, '\p{^Grapheme_Cluster_Break=extend}', "");
Expect(1, 918000, '\P{Grapheme_Cluster_Break=extend}', "");
Expect(0, 918000, '\P{^Grapheme_Cluster_Break=extend}', "");
Expect(1, 917999, '\p{Grapheme_Cluster_Break=_-Extend}', "");
Expect(0, 917999, '\p{^Grapheme_Cluster_Break=_-Extend}', "");
Expect(0, 917999, '\P{Grapheme_Cluster_Break=_-Extend}', "");
Expect(1, 917999, '\P{^Grapheme_Cluster_Break=_-Extend}', "");
Expect(0, 918000, '\p{Grapheme_Cluster_Break=_-Extend}', "");
Expect(1, 918000, '\p{^Grapheme_Cluster_Break=_-Extend}', "");
Expect(1, 918000, '\P{Grapheme_Cluster_Break=_-Extend}', "");
Expect(0, 918000, '\P{^Grapheme_Cluster_Break=_-Extend}', "");
Error('\p{GCB= :=EX}');
Error('\P{GCB= :=EX}');
Expect(1, 917999, '\p{GCB=ex}', "");
Expect(0, 917999, '\p{^GCB=ex}', "");
Expect(0, 917999, '\P{GCB=ex}', "");
Expect(1, 917999, '\P{^GCB=ex}', "");
Expect(0, 918000, '\p{GCB=ex}', "");
Expect(1, 918000, '\p{^GCB=ex}', "");
Expect(1, 918000, '\P{GCB=ex}', "");
Expect(0, 918000, '\P{^GCB=ex}', "");
Expect(1, 917999, '\p{GCB=EX}', "");
Expect(0, 917999, '\p{^GCB=EX}', "");
Expect(0, 917999, '\P{GCB=EX}', "");
Expect(1, 917999, '\P{^GCB=EX}', "");
Expect(0, 918000, '\p{GCB=EX}', "");
Expect(1, 918000, '\p{^GCB=EX}', "");
Expect(1, 918000, '\P{GCB=EX}', "");
Expect(0, 918000, '\P{^GCB=EX}', "");
Error('\p{Is_Grapheme_Cluster_Break=/a/-Extend}');
Error('\P{Is_Grapheme_Cluster_Break=/a/-Extend}');
Expect(1, 917999, '\p{Is_Grapheme_Cluster_Break=extend}', "");
Expect(0, 917999, '\p{^Is_Grapheme_Cluster_Break=extend}', "");
Expect(0, 917999, '\P{Is_Grapheme_Cluster_Break=extend}', "");
Expect(1, 917999, '\P{^Is_Grapheme_Cluster_Break=extend}', "");
Expect(0, 918000, '\p{Is_Grapheme_Cluster_Break=extend}', "");
Expect(1, 918000, '\p{^Is_Grapheme_Cluster_Break=extend}', "");
Expect(1, 918000, '\P{Is_Grapheme_Cluster_Break=extend}', "");
Expect(0, 918000, '\P{^Is_Grapheme_Cluster_Break=extend}', "");
Expect(1, 917999, '\p{Is_Grapheme_Cluster_Break=__Extend}', "");
Expect(0, 917999, '\p{^Is_Grapheme_Cluster_Break=__Extend}', "");
Expect(0, 917999, '\P{Is_Grapheme_Cluster_Break=__Extend}', "");
Expect(1, 917999, '\P{^Is_Grapheme_Cluster_Break=__Extend}', "");
Expect(0, 918000, '\p{Is_Grapheme_Cluster_Break=__Extend}', "");
Expect(1, 918000, '\p{^Is_Grapheme_Cluster_Break=__Extend}', "");
Expect(1, 918000, '\P{Is_Grapheme_Cluster_Break=__Extend}', "");
Expect(0, 918000, '\P{^Is_Grapheme_Cluster_Break=__Extend}', "");
Error('\p{Is_GCB: _ EX:=}');
Error('\P{Is_GCB: _ EX:=}');
Expect(1, 917999, '\p{Is_GCB=ex}', "");
Expect(0, 917999, '\p{^Is_GCB=ex}', "");
Expect(0, 917999, '\P{Is_GCB=ex}', "");
Expect(1, 917999, '\P{^Is_GCB=ex}', "");
Expect(0, 918000, '\p{Is_GCB=ex}', "");
Expect(1, 918000, '\p{^Is_GCB=ex}', "");
Expect(1, 918000, '\P{Is_GCB=ex}', "");
Expect(0, 918000, '\P{^Is_GCB=ex}', "");
Expect(1, 917999, '\p{Is_GCB= EX}', "");
Expect(0, 917999, '\p{^Is_GCB= EX}', "");
Expect(0, 917999, '\P{Is_GCB= EX}', "");
Expect(1, 917999, '\P{^Is_GCB= EX}', "");
Expect(0, 918000, '\p{Is_GCB= EX}', "");
Expect(1, 918000, '\p{^Is_GCB= EX}', "");
Expect(1, 918000, '\P{Is_GCB= EX}', "");
Expect(0, 918000, '\P{^Is_GCB= EX}', "");
Error('\p{Grapheme_Cluster_Break:	L/a/}');
Error('\P{Grapheme_Cluster_Break:	L/a/}');
Expect(1, 43388, '\p{Grapheme_Cluster_Break=l}', "");
Expect(0, 43388, '\p{^Grapheme_Cluster_Break=l}', "");
Expect(0, 43388, '\P{Grapheme_Cluster_Break=l}', "");
Expect(1, 43388, '\P{^Grapheme_Cluster_Break=l}', "");
Expect(0, 43389, '\p{Grapheme_Cluster_Break=l}', "");
Expect(1, 43389, '\p{^Grapheme_Cluster_Break=l}', "");
Expect(1, 43389, '\P{Grapheme_Cluster_Break=l}', "");
Expect(0, 43389, '\P{^Grapheme_Cluster_Break=l}', "");
Expect(1, 43388, '\p{Grapheme_Cluster_Break=__L}', "");
Expect(0, 43388, '\p{^Grapheme_Cluster_Break=__L}', "");
Expect(0, 43388, '\P{Grapheme_Cluster_Break=__L}', "");
Expect(1, 43388, '\P{^Grapheme_Cluster_Break=__L}', "");
Expect(0, 43389, '\p{Grapheme_Cluster_Break=__L}', "");
Expect(1, 43389, '\p{^Grapheme_Cluster_Break=__L}', "");
Expect(1, 43389, '\P{Grapheme_Cluster_Break=__L}', "");
Expect(0, 43389, '\P{^Grapheme_Cluster_Break=__L}', "");
Error('\p{GCB=__l:=}');
Error('\P{GCB=__l:=}');
Expect(1, 43388, '\p{GCB=l}', "");
Expect(0, 43388, '\p{^GCB=l}', "");
Expect(0, 43388, '\P{GCB=l}', "");
Expect(1, 43388, '\P{^GCB=l}', "");
Expect(0, 43389, '\p{GCB=l}', "");
Expect(1, 43389, '\p{^GCB=l}', "");
Expect(1, 43389, '\P{GCB=l}', "");
Expect(0, 43389, '\P{^GCB=l}', "");
Expect(1, 43388, '\p{GCB= -L}', "");
Expect(0, 43388, '\p{^GCB= -L}', "");
Expect(0, 43388, '\P{GCB= -L}', "");
Expect(1, 43388, '\P{^GCB= -L}', "");
Expect(0, 43389, '\p{GCB= -L}', "");
Expect(1, 43389, '\p{^GCB= -L}', "");
Expect(1, 43389, '\P{GCB= -L}', "");
Expect(0, 43389, '\P{^GCB= -L}', "");
Error('\p{Is_Grapheme_Cluster_Break=/a/	_l}');
Error('\P{Is_Grapheme_Cluster_Break=/a/	_l}');
Expect(1, 43388, '\p{Is_Grapheme_Cluster_Break=l}', "");
Expect(0, 43388, '\p{^Is_Grapheme_Cluster_Break=l}', "");
Expect(0, 43388, '\P{Is_Grapheme_Cluster_Break=l}', "");
Expect(1, 43388, '\P{^Is_Grapheme_Cluster_Break=l}', "");
Expect(0, 43389, '\p{Is_Grapheme_Cluster_Break=l}', "");
Expect(1, 43389, '\p{^Is_Grapheme_Cluster_Break=l}', "");
Expect(1, 43389, '\P{Is_Grapheme_Cluster_Break=l}', "");
Expect(0, 43389, '\P{^Is_Grapheme_Cluster_Break=l}', "");
Expect(1, 43388, '\p{Is_Grapheme_Cluster_Break=	L}', "");
Expect(0, 43388, '\p{^Is_Grapheme_Cluster_Break=	L}', "");
Expect(0, 43388, '\P{Is_Grapheme_Cluster_Break=	L}', "");
Expect(1, 43388, '\P{^Is_Grapheme_Cluster_Break=	L}', "");
Expect(0, 43389, '\p{Is_Grapheme_Cluster_Break=	L}', "");
Expect(1, 43389, '\p{^Is_Grapheme_Cluster_Break=	L}', "");
Expect(1, 43389, '\P{Is_Grapheme_Cluster_Break=	L}', "");
Expect(0, 43389, '\P{^Is_Grapheme_Cluster_Break=	L}', "");
Error('\p{Is_GCB= 	L:=}');
Error('\P{Is_GCB= 	L:=}');
Expect(1, 43388, '\p{Is_GCB=l}', "");
Expect(0, 43388, '\p{^Is_GCB=l}', "");
Expect(0, 43388, '\P{Is_GCB=l}', "");
Expect(1, 43388, '\P{^Is_GCB=l}', "");
Expect(0, 43389, '\p{Is_GCB=l}', "");
Expect(1, 43389, '\p{^Is_GCB=l}', "");
Expect(1, 43389, '\P{Is_GCB=l}', "");
Expect(0, 43389, '\P{^Is_GCB=l}', "");
Expect(1, 43388, '\p{Is_GCB=	-l}', "");
Expect(0, 43388, '\p{^Is_GCB=	-l}', "");
Expect(0, 43388, '\P{Is_GCB=	-l}', "");
Expect(1, 43388, '\P{^Is_GCB=	-l}', "");
Expect(0, 43389, '\p{Is_GCB=	-l}', "");
Expect(1, 43389, '\p{^Is_GCB=	-l}', "");
Expect(1, 43389, '\P{Is_GCB=	-l}', "");
Expect(0, 43389, '\P{^Is_GCB=	-l}', "");
Error('\p{Grapheme_Cluster_Break=:=LF}');
Error('\P{Grapheme_Cluster_Break=:=LF}');
Expect(1, 10, '\p{Grapheme_Cluster_Break=lf}', "");
Expect(0, 10, '\p{^Grapheme_Cluster_Break=lf}', "");
Expect(0, 10, '\P{Grapheme_Cluster_Break=lf}', "");
Expect(1, 10, '\P{^Grapheme_Cluster_Break=lf}', "");
Expect(0, 11, '\p{Grapheme_Cluster_Break=lf}', "");
Expect(1, 11, '\p{^Grapheme_Cluster_Break=lf}', "");
Expect(1, 11, '\P{Grapheme_Cluster_Break=lf}', "");
Expect(0, 11, '\P{^Grapheme_Cluster_Break=lf}', "");
Expect(1, 10, '\p{Grapheme_Cluster_Break= -LF}', "");
Expect(0, 10, '\p{^Grapheme_Cluster_Break= -LF}', "");
Expect(0, 10, '\P{Grapheme_Cluster_Break= -LF}', "");
Expect(1, 10, '\P{^Grapheme_Cluster_Break= -LF}', "");
Expect(0, 11, '\p{Grapheme_Cluster_Break= -LF}', "");
Expect(1, 11, '\p{^Grapheme_Cluster_Break= -LF}', "");
Expect(1, 11, '\P{Grapheme_Cluster_Break= -LF}', "");
Expect(0, 11, '\P{^Grapheme_Cluster_Break= -LF}', "");
Error('\p{GCB=- LF/a/}');
Error('\P{GCB=- LF/a/}');
Expect(1, 10, '\p{GCB=lf}', "");
Expect(0, 10, '\p{^GCB=lf}', "");
Expect(0, 10, '\P{GCB=lf}', "");
Expect(1, 10, '\P{^GCB=lf}', "");
Expect(0, 11, '\p{GCB=lf}', "");
Expect(1, 11, '\p{^GCB=lf}', "");
Expect(1, 11, '\P{GCB=lf}', "");
Expect(0, 11, '\P{^GCB=lf}', "");
Expect(1, 10, '\p{GCB=_LF}', "");
Expect(0, 10, '\p{^GCB=_LF}', "");
Expect(0, 10, '\P{GCB=_LF}', "");
Expect(1, 10, '\P{^GCB=_LF}', "");
Expect(0, 11, '\p{GCB=_LF}', "");
Expect(1, 11, '\p{^GCB=_LF}', "");
Expect(1, 11, '\P{GCB=_LF}', "");
Expect(0, 11, '\P{^GCB=_LF}', "");
Error('\p{Is_Grapheme_Cluster_Break=_	lf/a/}');
Error('\P{Is_Grapheme_Cluster_Break=_	lf/a/}');
Expect(1, 10, '\p{Is_Grapheme_Cluster_Break=lf}', "");
Expect(0, 10, '\p{^Is_Grapheme_Cluster_Break=lf}', "");
Expect(0, 10, '\P{Is_Grapheme_Cluster_Break=lf}', "");
Expect(1, 10, '\P{^Is_Grapheme_Cluster_Break=lf}', "");
Expect(0, 11, '\p{Is_Grapheme_Cluster_Break=lf}', "");
Expect(1, 11, '\p{^Is_Grapheme_Cluster_Break=lf}', "");
Expect(1, 11, '\P{Is_Grapheme_Cluster_Break=lf}', "");
Expect(0, 11, '\P{^Is_Grapheme_Cluster_Break=lf}', "");
Expect(1, 10, '\p{Is_Grapheme_Cluster_Break=-lf}', "");
Expect(0, 10, '\p{^Is_Grapheme_Cluster_Break=-lf}', "");
Expect(0, 10, '\P{Is_Grapheme_Cluster_Break=-lf}', "");
Expect(1, 10, '\P{^Is_Grapheme_Cluster_Break=-lf}', "");
Expect(0, 11, '\p{Is_Grapheme_Cluster_Break=-lf}', "");
Expect(1, 11, '\p{^Is_Grapheme_Cluster_Break=-lf}', "");
Expect(1, 11, '\P{Is_Grapheme_Cluster_Break=-lf}', "");
Expect(0, 11, '\P{^Is_Grapheme_Cluster_Break=-lf}', "");
Error('\p{Is_GCB=-:=LF}');
Error('\P{Is_GCB=-:=LF}');
Expect(1, 10, '\p{Is_GCB=lf}', "");
Expect(0, 10, '\p{^Is_GCB=lf}', "");
Expect(0, 10, '\P{Is_GCB=lf}', "");
Expect(1, 10, '\P{^Is_GCB=lf}', "");
Expect(0, 11, '\p{Is_GCB=lf}', "");
Expect(1, 11, '\p{^Is_GCB=lf}', "");
Expect(1, 11, '\P{Is_GCB=lf}', "");
Expect(0, 11, '\P{^Is_GCB=lf}', "");
Expect(1, 10, '\p{Is_GCB=	LF}', "");
Expect(0, 10, '\p{^Is_GCB=	LF}', "");
Expect(0, 10, '\P{Is_GCB=	LF}', "");
Expect(1, 10, '\P{^Is_GCB=	LF}', "");
Expect(0, 11, '\p{Is_GCB=	LF}', "");
Expect(1, 11, '\p{^Is_GCB=	LF}', "");
Expect(1, 11, '\P{Is_GCB=	LF}', "");
Expect(0, 11, '\P{^Is_GCB=	LF}', "");
Error('\p{Grapheme_Cluster_Break=	:=LV}');
Error('\P{Grapheme_Cluster_Break=	:=LV}');
Expect(1, 55176, '\p{Grapheme_Cluster_Break=lv}', "");
Expect(0, 55176, '\p{^Grapheme_Cluster_Break=lv}', "");
Expect(0, 55176, '\P{Grapheme_Cluster_Break=lv}', "");
Expect(1, 55176, '\P{^Grapheme_Cluster_Break=lv}', "");
Expect(0, 55177, '\p{Grapheme_Cluster_Break=lv}', "");
Expect(1, 55177, '\p{^Grapheme_Cluster_Break=lv}', "");
Expect(1, 55177, '\P{Grapheme_Cluster_Break=lv}', "");
Expect(0, 55177, '\P{^Grapheme_Cluster_Break=lv}', "");
Expect(1, 55176, '\p{Grapheme_Cluster_Break=_lv}', "");
Expect(0, 55176, '\p{^Grapheme_Cluster_Break=_lv}', "");
Expect(0, 55176, '\P{Grapheme_Cluster_Break=_lv}', "");
Expect(1, 55176, '\P{^Grapheme_Cluster_Break=_lv}', "");
Expect(0, 55177, '\p{Grapheme_Cluster_Break=_lv}', "");
Expect(1, 55177, '\p{^Grapheme_Cluster_Break=_lv}', "");
Expect(1, 55177, '\P{Grapheme_Cluster_Break=_lv}', "");
Expect(0, 55177, '\P{^Grapheme_Cluster_Break=_lv}', "");
Error('\p{GCB=/a/ -lv}');
Error('\P{GCB=/a/ -lv}');
Expect(1, 55176, '\p{GCB=lv}', "");
Expect(0, 55176, '\p{^GCB=lv}', "");
Expect(0, 55176, '\P{GCB=lv}', "");
Expect(1, 55176, '\P{^GCB=lv}', "");
Expect(0, 55177, '\p{GCB=lv}', "");
Expect(1, 55177, '\p{^GCB=lv}', "");
Expect(1, 55177, '\P{GCB=lv}', "");
Expect(0, 55177, '\P{^GCB=lv}', "");
Expect(1, 55176, '\p{GCB= 	LV}', "");
Expect(0, 55176, '\p{^GCB= 	LV}', "");
Expect(0, 55176, '\P{GCB= 	LV}', "");
Expect(1, 55176, '\P{^GCB= 	LV}', "");
Expect(0, 55177, '\p{GCB= 	LV}', "");
Expect(1, 55177, '\p{^GCB= 	LV}', "");
Expect(1, 55177, '\P{GCB= 	LV}', "");
Expect(0, 55177, '\P{^GCB= 	LV}', "");
Error('\p{Is_Grapheme_Cluster_Break: _/a/LV}');
Error('\P{Is_Grapheme_Cluster_Break: _/a/LV}');
Expect(1, 55176, '\p{Is_Grapheme_Cluster_Break=lv}', "");
Expect(0, 55176, '\p{^Is_Grapheme_Cluster_Break=lv}', "");
Expect(0, 55176, '\P{Is_Grapheme_Cluster_Break=lv}', "");
Expect(1, 55176, '\P{^Is_Grapheme_Cluster_Break=lv}', "");
Expect(0, 55177, '\p{Is_Grapheme_Cluster_Break=lv}', "");
Expect(1, 55177, '\p{^Is_Grapheme_Cluster_Break=lv}', "");
Expect(1, 55177, '\P{Is_Grapheme_Cluster_Break=lv}', "");
Expect(0, 55177, '\P{^Is_Grapheme_Cluster_Break=lv}', "");
Expect(1, 55176, '\p{Is_Grapheme_Cluster_Break=	LV}', "");
Expect(0, 55176, '\p{^Is_Grapheme_Cluster_Break=	LV}', "");
Expect(0, 55176, '\P{Is_Grapheme_Cluster_Break=	LV}', "");
Expect(1, 55176, '\P{^Is_Grapheme_Cluster_Break=	LV}', "");
Expect(0, 55177, '\p{Is_Grapheme_Cluster_Break=	LV}', "");
Expect(1, 55177, '\p{^Is_Grapheme_Cluster_Break=	LV}', "");
Expect(1, 55177, '\P{Is_Grapheme_Cluster_Break=	LV}', "");
Expect(0, 55177, '\P{^Is_Grapheme_Cluster_Break=	LV}', "");
Error('\p{Is_GCB:	_ lv/a/}');
Error('\P{Is_GCB:	_ lv/a/}');
Expect(1, 55176, '\p{Is_GCB=lv}', "");
Expect(0, 55176, '\p{^Is_GCB=lv}', "");
Expect(0, 55176, '\P{Is_GCB=lv}', "");
Expect(1, 55176, '\P{^Is_GCB=lv}', "");
Expect(0, 55177, '\p{Is_GCB=lv}', "");
Expect(1, 55177, '\p{^Is_GCB=lv}', "");
Expect(1, 55177, '\P{Is_GCB=lv}', "");
Expect(0, 55177, '\P{^Is_GCB=lv}', "");
Expect(1, 55176, '\p{Is_GCB:	 -LV}', "");
Expect(0, 55176, '\p{^Is_GCB:	 -LV}', "");
Expect(0, 55176, '\P{Is_GCB:	 -LV}', "");
Expect(1, 55176, '\P{^Is_GCB:	 -LV}', "");
Expect(0, 55177, '\p{Is_GCB:	 -LV}', "");
Expect(1, 55177, '\p{^Is_GCB:	 -LV}', "");
Expect(1, 55177, '\P{Is_GCB:	 -LV}', "");
Expect(0, 55177, '\P{^Is_GCB:	 -LV}', "");
Error('\p{Grapheme_Cluster_Break=	/a/LVT}');
Error('\P{Grapheme_Cluster_Break=	/a/LVT}');
Expect(1, 55203, '\p{Grapheme_Cluster_Break=lvt}', "");
Expect(0, 55203, '\p{^Grapheme_Cluster_Break=lvt}', "");
Expect(0, 55203, '\P{Grapheme_Cluster_Break=lvt}', "");
Expect(1, 55203, '\P{^Grapheme_Cluster_Break=lvt}', "");
Expect(0, 55204, '\p{Grapheme_Cluster_Break=lvt}', "");
Expect(1, 55204, '\p{^Grapheme_Cluster_Break=lvt}', "");
Expect(1, 55204, '\P{Grapheme_Cluster_Break=lvt}', "");
Expect(0, 55204, '\P{^Grapheme_Cluster_Break=lvt}', "");
Expect(1, 55203, '\p{Grapheme_Cluster_Break: 	LVT}', "");
Expect(0, 55203, '\p{^Grapheme_Cluster_Break: 	LVT}', "");
Expect(0, 55203, '\P{Grapheme_Cluster_Break: 	LVT}', "");
Expect(1, 55203, '\P{^Grapheme_Cluster_Break: 	LVT}', "");
Expect(0, 55204, '\p{Grapheme_Cluster_Break: 	LVT}', "");
Expect(1, 55204, '\p{^Grapheme_Cluster_Break: 	LVT}', "");
Expect(1, 55204, '\P{Grapheme_Cluster_Break: 	LVT}', "");
Expect(0, 55204, '\P{^Grapheme_Cluster_Break: 	LVT}', "");
Error('\p{GCB=:= LVT}');
Error('\P{GCB=:= LVT}');
Expect(1, 55203, '\p{GCB=lvt}', "");
Expect(0, 55203, '\p{^GCB=lvt}', "");
Expect(0, 55203, '\P{GCB=lvt}', "");
Expect(1, 55203, '\P{^GCB=lvt}', "");
Expect(0, 55204, '\p{GCB=lvt}', "");
Expect(1, 55204, '\p{^GCB=lvt}', "");
Expect(1, 55204, '\P{GCB=lvt}', "");
Expect(0, 55204, '\P{^GCB=lvt}', "");
Expect(1, 55203, '\p{GCB:		_LVT}', "");
Expect(0, 55203, '\p{^GCB:		_LVT}', "");
Expect(0, 55203, '\P{GCB:		_LVT}', "");
Expect(1, 55203, '\P{^GCB:		_LVT}', "");
Expect(0, 55204, '\p{GCB:		_LVT}', "");
Expect(1, 55204, '\p{^GCB:		_LVT}', "");
Expect(1, 55204, '\P{GCB:		_LVT}', "");
Expect(0, 55204, '\P{^GCB:		_LVT}', "");
Error('\p{Is_Grapheme_Cluster_Break=:=- LVT}');
Error('\P{Is_Grapheme_Cluster_Break=:=- LVT}');
Expect(1, 55203, '\p{Is_Grapheme_Cluster_Break:   lvt}', "");
Expect(0, 55203, '\p{^Is_Grapheme_Cluster_Break:   lvt}', "");
Expect(0, 55203, '\P{Is_Grapheme_Cluster_Break:   lvt}', "");
Expect(1, 55203, '\P{^Is_Grapheme_Cluster_Break:   lvt}', "");
Expect(0, 55204, '\p{Is_Grapheme_Cluster_Break:   lvt}', "");
Expect(1, 55204, '\p{^Is_Grapheme_Cluster_Break:   lvt}', "");
Expect(1, 55204, '\P{Is_Grapheme_Cluster_Break:   lvt}', "");
Expect(0, 55204, '\P{^Is_Grapheme_Cluster_Break:   lvt}', "");
Expect(1, 55203, '\p{Is_Grapheme_Cluster_Break=-LVT}', "");
Expect(0, 55203, '\p{^Is_Grapheme_Cluster_Break=-LVT}', "");
Expect(0, 55203, '\P{Is_Grapheme_Cluster_Break=-LVT}', "");
Expect(1, 55203, '\P{^Is_Grapheme_Cluster_Break=-LVT}', "");
Expect(0, 55204, '\p{Is_Grapheme_Cluster_Break=-LVT}', "");
Expect(1, 55204, '\p{^Is_Grapheme_Cluster_Break=-LVT}', "");
Expect(1, 55204, '\P{Is_Grapheme_Cluster_Break=-LVT}', "");
Expect(0, 55204, '\P{^Is_Grapheme_Cluster_Break=-LVT}', "");
Error('\p{Is_GCB=/a/	LVT}');
Error('\P{Is_GCB=/a/	LVT}');
Expect(1, 55203, '\p{Is_GCB: lvt}', "");
Expect(0, 55203, '\p{^Is_GCB: lvt}', "");
Expect(0, 55203, '\P{Is_GCB: lvt}', "");
Expect(1, 55203, '\P{^Is_GCB: lvt}', "");
Expect(0, 55204, '\p{Is_GCB: lvt}', "");
Expect(1, 55204, '\p{^Is_GCB: lvt}', "");
Expect(1, 55204, '\P{Is_GCB: lvt}', "");
Expect(0, 55204, '\P{^Is_GCB: lvt}', "");
Expect(1, 55203, '\p{Is_GCB=_ LVT}', "");
Expect(0, 55203, '\p{^Is_GCB=_ LVT}', "");
Expect(0, 55203, '\P{Is_GCB=_ LVT}', "");
Expect(1, 55203, '\P{^Is_GCB=_ LVT}', "");
Expect(0, 55204, '\p{Is_GCB=_ LVT}', "");
Expect(1, 55204, '\p{^Is_GCB=_ LVT}', "");
Expect(1, 55204, '\P{Is_GCB=_ LVT}', "");
Expect(0, 55204, '\P{^Is_GCB=_ LVT}', "");
Error('\p{Grapheme_Cluster_Break=	/a/Prepend}');
Error('\P{Grapheme_Cluster_Break=	/a/Prepend}');
Expect(0, 1, '\p{Grapheme_Cluster_Break=prepend}', "");
Expect(1, 1, '\p{^Grapheme_Cluster_Break=prepend}', "");
Expect(1, 1, '\P{Grapheme_Cluster_Break=prepend}', "");
Expect(0, 1, '\P{^Grapheme_Cluster_Break=prepend}', "");
Expect(0, 1, '\p{Grapheme_Cluster_Break= PREPEND}', "");
Expect(1, 1, '\p{^Grapheme_Cluster_Break= PREPEND}', "");
Expect(1, 1, '\P{Grapheme_Cluster_Break= PREPEND}', "");
Expect(0, 1, '\P{^Grapheme_Cluster_Break= PREPEND}', "");
Error('\p{GCB=:=	-PP}');
Error('\P{GCB=:=	-PP}');
Expect(0, 1, '\p{GCB=pp}', "");
Expect(1, 1, '\p{^GCB=pp}', "");
Expect(1, 1, '\P{GCB=pp}', "");
Expect(0, 1, '\P{^GCB=pp}', "");
Expect(0, 1, '\p{GCB=_PP}', "");
Expect(1, 1, '\p{^GCB=_PP}', "");
Expect(1, 1, '\P{GCB=_PP}', "");
Expect(0, 1, '\P{^GCB=_PP}', "");
Error('\p{Is_Grapheme_Cluster_Break=/a/_ prepend}');
Error('\P{Is_Grapheme_Cluster_Break=/a/_ prepend}');
Expect(0, 1, '\p{Is_Grapheme_Cluster_Break=prepend}', "");
Expect(1, 1, '\p{^Is_Grapheme_Cluster_Break=prepend}', "");
Expect(1, 1, '\P{Is_Grapheme_Cluster_Break=prepend}', "");
Expect(0, 1, '\P{^Is_Grapheme_Cluster_Break=prepend}', "");
Expect(0, 1, '\p{Is_Grapheme_Cluster_Break=_Prepend}', "");
Expect(1, 1, '\p{^Is_Grapheme_Cluster_Break=_Prepend}', "");
Expect(1, 1, '\P{Is_Grapheme_Cluster_Break=_Prepend}', "");
Expect(0, 1, '\P{^Is_Grapheme_Cluster_Break=_Prepend}', "");
Error('\p{Is_GCB=:= _PP}');
Error('\P{Is_GCB=:= _PP}');
Expect(0, 1, '\p{Is_GCB: pp}', "");
Expect(1, 1, '\p{^Is_GCB: pp}', "");
Expect(1, 1, '\P{Is_GCB: pp}', "");
Expect(0, 1, '\P{^Is_GCB: pp}', "");
Expect(0, 1, '\p{Is_GCB=-PP}', "");
Expect(1, 1, '\p{^Is_GCB=-PP}', "");
Expect(1, 1, '\P{Is_GCB=-PP}', "");
Expect(0, 1, '\P{^Is_GCB=-PP}', "");
Error('\p{Grapheme_Cluster_Break=_/a/regional_INDICATOR}');
Error('\P{Grapheme_Cluster_Break=_/a/regional_INDICATOR}');
Expect(1, 127487, '\p{Grapheme_Cluster_Break=regionalindicator}', "");
Expect(0, 127487, '\p{^Grapheme_Cluster_Break=regionalindicator}', "");
Expect(0, 127487, '\P{Grapheme_Cluster_Break=regionalindicator}', "");
Expect(1, 127487, '\P{^Grapheme_Cluster_Break=regionalindicator}', "");
Expect(0, 127488, '\p{Grapheme_Cluster_Break=regionalindicator}', "");
Expect(1, 127488, '\p{^Grapheme_Cluster_Break=regionalindicator}', "");
Expect(1, 127488, '\P{Grapheme_Cluster_Break=regionalindicator}', "");
Expect(0, 127488, '\P{^Grapheme_Cluster_Break=regionalindicator}', "");
Expect(1, 127487, '\p{Grapheme_Cluster_Break=--Regional_Indicator}', "");
Expect(0, 127487, '\p{^Grapheme_Cluster_Break=--Regional_Indicator}', "");
Expect(0, 127487, '\P{Grapheme_Cluster_Break=--Regional_Indicator}', "");
Expect(1, 127487, '\P{^Grapheme_Cluster_Break=--Regional_Indicator}', "");
Expect(0, 127488, '\p{Grapheme_Cluster_Break=--Regional_Indicator}', "");
Expect(1, 127488, '\p{^Grapheme_Cluster_Break=--Regional_Indicator}', "");
Expect(1, 127488, '\P{Grapheme_Cluster_Break=--Regional_Indicator}', "");
Expect(0, 127488, '\P{^Grapheme_Cluster_Break=--Regional_Indicator}', "");
Error('\p{GCB= 	RI/a/}');
Error('\P{GCB= 	RI/a/}');
Expect(1, 127487, '\p{GCB:	ri}', "");
Expect(0, 127487, '\p{^GCB:	ri}', "");
Expect(0, 127487, '\P{GCB:	ri}', "");
Expect(1, 127487, '\P{^GCB:	ri}', "");
Expect(0, 127488, '\p{GCB:	ri}', "");
Expect(1, 127488, '\p{^GCB:	ri}', "");
Expect(1, 127488, '\P{GCB:	ri}', "");
Expect(0, 127488, '\P{^GCB:	ri}', "");
Expect(1, 127487, '\p{GCB=	RI}', "");
Expect(0, 127487, '\p{^GCB=	RI}', "");
Expect(0, 127487, '\P{GCB=	RI}', "");
Expect(1, 127487, '\P{^GCB=	RI}', "");
Expect(0, 127488, '\p{GCB=	RI}', "");
Expect(1, 127488, '\p{^GCB=	RI}', "");
Expect(1, 127488, '\P{GCB=	RI}', "");
Expect(0, 127488, '\P{^GCB=	RI}', "");
Error('\p{Is_Grapheme_Cluster_Break=__REGIONAL_indicator/a/}');
Error('\P{Is_Grapheme_Cluster_Break=__REGIONAL_indicator/a/}');
Expect(1, 127487, '\p{Is_Grapheme_Cluster_Break=regionalindicator}', "");
Expect(0, 127487, '\p{^Is_Grapheme_Cluster_Break=regionalindicator}', "");
Expect(0, 127487, '\P{Is_Grapheme_Cluster_Break=regionalindicator}', "");
Expect(1, 127487, '\P{^Is_Grapheme_Cluster_Break=regionalindicator}', "");
Expect(0, 127488, '\p{Is_Grapheme_Cluster_Break=regionalindicator}', "");
Expect(1, 127488, '\p{^Is_Grapheme_Cluster_Break=regionalindicator}', "");
Expect(1, 127488, '\P{Is_Grapheme_Cluster_Break=regionalindicator}', "");
Expect(0, 127488, '\P{^Is_Grapheme_Cluster_Break=regionalindicator}', "");
Expect(1, 127487, '\p{Is_Grapheme_Cluster_Break=	 REGIONAL_INDICATOR}', "");
Expect(0, 127487, '\p{^Is_Grapheme_Cluster_Break=	 REGIONAL_INDICATOR}', "");
Expect(0, 127487, '\P{Is_Grapheme_Cluster_Break=	 REGIONAL_INDICATOR}', "");
Expect(1, 127487, '\P{^Is_Grapheme_Cluster_Break=	 REGIONAL_INDICATOR}', "");
Expect(0, 127488, '\p{Is_Grapheme_Cluster_Break=	 REGIONAL_INDICATOR}', "");
Expect(1, 127488, '\p{^Is_Grapheme_Cluster_Break=	 REGIONAL_INDICATOR}', "");
Expect(1, 127488, '\P{Is_Grapheme_Cluster_Break=	 REGIONAL_INDICATOR}', "");
Expect(0, 127488, '\P{^Is_Grapheme_Cluster_Break=	 REGIONAL_INDICATOR}', "");
Error('\p{Is_GCB=/a/ RI}');
Error('\P{Is_GCB=/a/ RI}');
Expect(1, 127487, '\p{Is_GCB: ri}', "");
Expect(0, 127487, '\p{^Is_GCB: ri}', "");
Expect(0, 127487, '\P{Is_GCB: ri}', "");
Expect(1, 127487, '\P{^Is_GCB: ri}', "");
Expect(0, 127488, '\p{Is_GCB: ri}', "");
Expect(1, 127488, '\p{^Is_GCB: ri}', "");
Expect(1, 127488, '\P{Is_GCB: ri}', "");
Expect(0, 127488, '\P{^Is_GCB: ri}', "");
Expect(1, 127487, '\p{Is_GCB=	_RI}', "");
Expect(0, 127487, '\p{^Is_GCB=	_RI}', "");
Expect(0, 127487, '\P{Is_GCB=	_RI}', "");
Expect(1, 127487, '\P{^Is_GCB=	_RI}', "");
Expect(0, 127488, '\p{Is_GCB=	_RI}', "");
Expect(1, 127488, '\p{^Is_GCB=	_RI}', "");
Expect(1, 127488, '\P{Is_GCB=	_RI}', "");
Expect(0, 127488, '\P{^Is_GCB=	_RI}', "");
Error('\p{Grapheme_Cluster_Break: - spacingmark/a/}');
Error('\P{Grapheme_Cluster_Break: - spacingmark/a/}');
Expect(1, 119149, '\p{Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 119149, '\p{^Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 119149, '\P{Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 119149, '\P{^Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 119150, '\p{Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 119150, '\p{^Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 119150, '\P{Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 119150, '\P{^Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 119149, '\p{Grapheme_Cluster_Break=-SpacingMark}', "");
Expect(0, 119149, '\p{^Grapheme_Cluster_Break=-SpacingMark}', "");
Expect(0, 119149, '\P{Grapheme_Cluster_Break=-SpacingMark}', "");
Expect(1, 119149, '\P{^Grapheme_Cluster_Break=-SpacingMark}', "");
Expect(0, 119150, '\p{Grapheme_Cluster_Break=-SpacingMark}', "");
Expect(1, 119150, '\p{^Grapheme_Cluster_Break=-SpacingMark}', "");
Expect(1, 119150, '\P{Grapheme_Cluster_Break=-SpacingMark}', "");
Expect(0, 119150, '\P{^Grapheme_Cluster_Break=-SpacingMark}', "");
Error('\p{GCB=:= _SM}');
Error('\P{GCB=:= _SM}');
Expect(1, 119149, '\p{GCB=sm}', "");
Expect(0, 119149, '\p{^GCB=sm}', "");
Expect(0, 119149, '\P{GCB=sm}', "");
Expect(1, 119149, '\P{^GCB=sm}', "");
Expect(0, 119150, '\p{GCB=sm}', "");
Expect(1, 119150, '\p{^GCB=sm}', "");
Expect(1, 119150, '\P{GCB=sm}', "");
Expect(0, 119150, '\P{^GCB=sm}', "");
Expect(1, 119149, '\p{GCB= _SM}', "");
Expect(0, 119149, '\p{^GCB= _SM}', "");
Expect(0, 119149, '\P{GCB= _SM}', "");
Expect(1, 119149, '\P{^GCB= _SM}', "");
Expect(0, 119150, '\p{GCB= _SM}', "");
Expect(1, 119150, '\p{^GCB= _SM}', "");
Expect(1, 119150, '\P{GCB= _SM}', "");
Expect(0, 119150, '\P{^GCB= _SM}', "");
Error('\p{Is_Grapheme_Cluster_Break=/a/  SpacingMark}');
Error('\P{Is_Grapheme_Cluster_Break=/a/  SpacingMark}');
Expect(1, 119149, '\p{Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 119149, '\p{^Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 119149, '\P{Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 119149, '\P{^Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 119150, '\p{Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 119150, '\p{^Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 119150, '\P{Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 119150, '\P{^Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 119149, '\p{Is_Grapheme_Cluster_Break=_spacingmark}', "");
Expect(0, 119149, '\p{^Is_Grapheme_Cluster_Break=_spacingmark}', "");
Expect(0, 119149, '\P{Is_Grapheme_Cluster_Break=_spacingmark}', "");
Expect(1, 119149, '\P{^Is_Grapheme_Cluster_Break=_spacingmark}', "");
Expect(0, 119150, '\p{Is_Grapheme_Cluster_Break=_spacingmark}', "");
Expect(1, 119150, '\p{^Is_Grapheme_Cluster_Break=_spacingmark}', "");
Expect(1, 119150, '\P{Is_Grapheme_Cluster_Break=_spacingmark}', "");
Expect(0, 119150, '\P{^Is_Grapheme_Cluster_Break=_spacingmark}', "");
Error('\p{Is_GCB=/a/_ sm}');
Error('\P{Is_GCB=/a/_ sm}');
Expect(1, 119149, '\p{Is_GCB=sm}', "");
Expect(0, 119149, '\p{^Is_GCB=sm}', "");
Expect(0, 119149, '\P{Is_GCB=sm}', "");
Expect(1, 119149, '\P{^Is_GCB=sm}', "");
Expect(0, 119150, '\p{Is_GCB=sm}', "");
Expect(1, 119150, '\p{^Is_GCB=sm}', "");
Expect(1, 119150, '\P{Is_GCB=sm}', "");
Expect(0, 119150, '\P{^Is_GCB=sm}', "");
Expect(1, 119149, '\p{Is_GCB=_-sm}', "");
Expect(0, 119149, '\p{^Is_GCB=_-sm}', "");
Expect(0, 119149, '\P{Is_GCB=_-sm}', "");
Expect(1, 119149, '\P{^Is_GCB=_-sm}', "");
Expect(0, 119150, '\p{Is_GCB=_-sm}', "");
Expect(1, 119150, '\p{^Is_GCB=_-sm}', "");
Expect(1, 119150, '\P{Is_GCB=_-sm}', "");
Expect(0, 119150, '\P{^Is_GCB=_-sm}', "");
Error('\p{Grapheme_Cluster_Break= /a/t}');
Error('\P{Grapheme_Cluster_Break= /a/t}');
Expect(1, 55291, '\p{Grapheme_Cluster_Break=t}', "");
Expect(0, 55291, '\p{^Grapheme_Cluster_Break=t}', "");
Expect(0, 55291, '\P{Grapheme_Cluster_Break=t}', "");
Expect(1, 55291, '\P{^Grapheme_Cluster_Break=t}', "");
Expect(0, 55292, '\p{Grapheme_Cluster_Break=t}', "");
Expect(1, 55292, '\p{^Grapheme_Cluster_Break=t}', "");
Expect(1, 55292, '\P{Grapheme_Cluster_Break=t}', "");
Expect(0, 55292, '\P{^Grapheme_Cluster_Break=t}', "");
Expect(1, 55291, '\p{Grapheme_Cluster_Break=  T}', "");
Expect(0, 55291, '\p{^Grapheme_Cluster_Break=  T}', "");
Expect(0, 55291, '\P{Grapheme_Cluster_Break=  T}', "");
Expect(1, 55291, '\P{^Grapheme_Cluster_Break=  T}', "");
Expect(0, 55292, '\p{Grapheme_Cluster_Break=  T}', "");
Expect(1, 55292, '\p{^Grapheme_Cluster_Break=  T}', "");
Expect(1, 55292, '\P{Grapheme_Cluster_Break=  T}', "");
Expect(0, 55292, '\P{^Grapheme_Cluster_Break=  T}', "");
Error('\p{GCB=/a/ _T}');
Error('\P{GCB=/a/ _T}');
Expect(1, 55291, '\p{GCB=t}', "");
Expect(0, 55291, '\p{^GCB=t}', "");
Expect(0, 55291, '\P{GCB=t}', "");
Expect(1, 55291, '\P{^GCB=t}', "");
Expect(0, 55292, '\p{GCB=t}', "");
Expect(1, 55292, '\p{^GCB=t}', "");
Expect(1, 55292, '\P{GCB=t}', "");
Expect(0, 55292, '\P{^GCB=t}', "");
Expect(1, 55291, '\p{GCB= 	T}', "");
Expect(0, 55291, '\p{^GCB= 	T}', "");
Expect(0, 55291, '\P{GCB= 	T}', "");
Expect(1, 55291, '\P{^GCB= 	T}', "");
Expect(0, 55292, '\p{GCB= 	T}', "");
Expect(1, 55292, '\p{^GCB= 	T}', "");
Expect(1, 55292, '\P{GCB= 	T}', "");
Expect(0, 55292, '\P{^GCB= 	T}', "");
Error('\p{Is_Grapheme_Cluster_Break=	/a/t}');
Error('\P{Is_Grapheme_Cluster_Break=	/a/t}');
Expect(1, 55291, '\p{Is_Grapheme_Cluster_Break=t}', "");
Expect(0, 55291, '\p{^Is_Grapheme_Cluster_Break=t}', "");
Expect(0, 55291, '\P{Is_Grapheme_Cluster_Break=t}', "");
Expect(1, 55291, '\P{^Is_Grapheme_Cluster_Break=t}', "");
Expect(0, 55292, '\p{Is_Grapheme_Cluster_Break=t}', "");
Expect(1, 55292, '\p{^Is_Grapheme_Cluster_Break=t}', "");
Expect(1, 55292, '\P{Is_Grapheme_Cluster_Break=t}', "");
Expect(0, 55292, '\P{^Is_Grapheme_Cluster_Break=t}', "");
Expect(1, 55291, '\p{Is_Grapheme_Cluster_Break=--t}', "");
Expect(0, 55291, '\p{^Is_Grapheme_Cluster_Break=--t}', "");
Expect(0, 55291, '\P{Is_Grapheme_Cluster_Break=--t}', "");
Expect(1, 55291, '\P{^Is_Grapheme_Cluster_Break=--t}', "");
Expect(0, 55292, '\p{Is_Grapheme_Cluster_Break=--t}', "");
Expect(1, 55292, '\p{^Is_Grapheme_Cluster_Break=--t}', "");
Expect(1, 55292, '\P{Is_Grapheme_Cluster_Break=--t}', "");
Expect(0, 55292, '\P{^Is_Grapheme_Cluster_Break=--t}', "");
Error('\p{Is_GCB=/a/	T}');
Error('\P{Is_GCB=/a/	T}');
Expect(1, 55291, '\p{Is_GCB=t}', "");
Expect(0, 55291, '\p{^Is_GCB=t}', "");
Expect(0, 55291, '\P{Is_GCB=t}', "");
Expect(1, 55291, '\P{^Is_GCB=t}', "");
Expect(0, 55292, '\p{Is_GCB=t}', "");
Expect(1, 55292, '\p{^Is_GCB=t}', "");
Expect(1, 55292, '\P{Is_GCB=t}', "");
Expect(0, 55292, '\P{^Is_GCB=t}', "");
Expect(1, 55291, '\p{Is_GCB=  T}', "");
Expect(0, 55291, '\p{^Is_GCB=  T}', "");
Expect(0, 55291, '\P{Is_GCB=  T}', "");
Expect(1, 55291, '\P{^Is_GCB=  T}', "");
Expect(0, 55292, '\p{Is_GCB=  T}', "");
Expect(1, 55292, '\p{^Is_GCB=  T}', "");
Expect(1, 55292, '\P{Is_GCB=  T}', "");
Expect(0, 55292, '\P{^Is_GCB=  T}', "");
Error('\p{Grapheme_Cluster_Break=:=-V}');
Error('\P{Grapheme_Cluster_Break=:=-V}');
Expect(1, 55238, '\p{Grapheme_Cluster_Break=v}', "");
Expect(0, 55238, '\p{^Grapheme_Cluster_Break=v}', "");
Expect(0, 55238, '\P{Grapheme_Cluster_Break=v}', "");
Expect(1, 55238, '\P{^Grapheme_Cluster_Break=v}', "");
Expect(0, 55239, '\p{Grapheme_Cluster_Break=v}', "");
Expect(1, 55239, '\p{^Grapheme_Cluster_Break=v}', "");
Expect(1, 55239, '\P{Grapheme_Cluster_Break=v}', "");
Expect(0, 55239, '\P{^Grapheme_Cluster_Break=v}', "");
Expect(1, 55238, '\p{Grapheme_Cluster_Break=	-V}', "");
Expect(0, 55238, '\p{^Grapheme_Cluster_Break=	-V}', "");
Expect(0, 55238, '\P{Grapheme_Cluster_Break=	-V}', "");
Expect(1, 55238, '\P{^Grapheme_Cluster_Break=	-V}', "");
Expect(0, 55239, '\p{Grapheme_Cluster_Break=	-V}', "");
Expect(1, 55239, '\p{^Grapheme_Cluster_Break=	-V}', "");
Expect(1, 55239, '\P{Grapheme_Cluster_Break=	-V}', "");
Expect(0, 55239, '\P{^Grapheme_Cluster_Break=	-V}', "");
Error('\p{GCB=	 V:=}');
Error('\P{GCB=	 V:=}');
Expect(1, 55238, '\p{GCB=v}', "");
Expect(0, 55238, '\p{^GCB=v}', "");
Expect(0, 55238, '\P{GCB=v}', "");
Expect(1, 55238, '\P{^GCB=v}', "");
Expect(0, 55239, '\p{GCB=v}', "");
Expect(1, 55239, '\p{^GCB=v}', "");
Expect(1, 55239, '\P{GCB=v}', "");
Expect(0, 55239, '\P{^GCB=v}', "");
Expect(1, 55238, '\p{GCB=_V}', "");
Expect(0, 55238, '\p{^GCB=_V}', "");
Expect(0, 55238, '\P{GCB=_V}', "");
Expect(1, 55238, '\P{^GCB=_V}', "");
Expect(0, 55239, '\p{GCB=_V}', "");
Expect(1, 55239, '\p{^GCB=_V}', "");
Expect(1, 55239, '\P{GCB=_V}', "");
Expect(0, 55239, '\P{^GCB=_V}', "");
Error('\p{Is_Grapheme_Cluster_Break=-/a/v}');
Error('\P{Is_Grapheme_Cluster_Break=-/a/v}');
Expect(1, 55238, '\p{Is_Grapheme_Cluster_Break=v}', "");
Expect(0, 55238, '\p{^Is_Grapheme_Cluster_Break=v}', "");
Expect(0, 55238, '\P{Is_Grapheme_Cluster_Break=v}', "");
Expect(1, 55238, '\P{^Is_Grapheme_Cluster_Break=v}', "");
Expect(0, 55239, '\p{Is_Grapheme_Cluster_Break=v}', "");
Expect(1, 55239, '\p{^Is_Grapheme_Cluster_Break=v}', "");
Expect(1, 55239, '\P{Is_Grapheme_Cluster_Break=v}', "");
Expect(0, 55239, '\P{^Is_Grapheme_Cluster_Break=v}', "");
Expect(1, 55238, '\p{Is_Grapheme_Cluster_Break:  	V}', "");
Expect(0, 55238, '\p{^Is_Grapheme_Cluster_Break:  	V}', "");
Expect(0, 55238, '\P{Is_Grapheme_Cluster_Break:  	V}', "");
Expect(1, 55238, '\P{^Is_Grapheme_Cluster_Break:  	V}', "");
Expect(0, 55239, '\p{Is_Grapheme_Cluster_Break:  	V}', "");
Expect(1, 55239, '\p{^Is_Grapheme_Cluster_Break:  	V}', "");
Expect(1, 55239, '\P{Is_Grapheme_Cluster_Break:  	V}', "");
Expect(0, 55239, '\P{^Is_Grapheme_Cluster_Break:  	V}', "");
Error('\p{Is_GCB=	V:=}');
Error('\P{Is_GCB=	V:=}');
Expect(1, 55238, '\p{Is_GCB=v}', "");
Expect(0, 55238, '\p{^Is_GCB=v}', "");
Expect(0, 55238, '\P{Is_GCB=v}', "");
Expect(1, 55238, '\P{^Is_GCB=v}', "");
Expect(0, 55239, '\p{Is_GCB=v}', "");
Expect(1, 55239, '\p{^Is_GCB=v}', "");
Expect(1, 55239, '\P{Is_GCB=v}', "");
Expect(0, 55239, '\P{^Is_GCB=v}', "");
Expect(1, 55238, '\p{Is_GCB= -V}', "");
Expect(0, 55238, '\p{^Is_GCB= -V}', "");
Expect(0, 55238, '\P{Is_GCB= -V}', "");
Expect(1, 55238, '\P{^Is_GCB= -V}', "");
Expect(0, 55239, '\p{Is_GCB= -V}', "");
Expect(1, 55239, '\p{^Is_GCB= -V}', "");
Expect(1, 55239, '\P{Is_GCB= -V}', "");
Expect(0, 55239, '\P{^Is_GCB= -V}', "");
Error('\p{Grapheme_Cluster_Break=	_OTHER:=}');
Error('\P{Grapheme_Cluster_Break=	_OTHER:=}');
Expect(1, 921600, '\p{Grapheme_Cluster_Break=other}', "");
Expect(0, 921600, '\p{^Grapheme_Cluster_Break=other}', "");
Expect(0, 921600, '\P{Grapheme_Cluster_Break=other}', "");
Expect(1, 921600, '\P{^Grapheme_Cluster_Break=other}', "");
Expect(0, 921599, '\p{Grapheme_Cluster_Break=other}', "");
Expect(1, 921599, '\p{^Grapheme_Cluster_Break=other}', "");
Expect(1, 921599, '\P{Grapheme_Cluster_Break=other}', "");
Expect(0, 921599, '\P{^Grapheme_Cluster_Break=other}', "");
Expect(1, 921600, '\p{Grapheme_Cluster_Break: Other}', "");
Expect(0, 921600, '\p{^Grapheme_Cluster_Break: Other}', "");
Expect(0, 921600, '\P{Grapheme_Cluster_Break: Other}', "");
Expect(1, 921600, '\P{^Grapheme_Cluster_Break: Other}', "");
Expect(0, 921599, '\p{Grapheme_Cluster_Break: Other}', "");
Expect(1, 921599, '\p{^Grapheme_Cluster_Break: Other}', "");
Expect(1, 921599, '\P{Grapheme_Cluster_Break: Other}', "");
Expect(0, 921599, '\P{^Grapheme_Cluster_Break: Other}', "");
Error('\p{GCB=_XX/a/}');
Error('\P{GCB=_XX/a/}');
Expect(1, 921600, '\p{GCB=xx}', "");
Expect(0, 921600, '\p{^GCB=xx}', "");
Expect(0, 921600, '\P{GCB=xx}', "");
Expect(1, 921600, '\P{^GCB=xx}', "");
Expect(0, 921599, '\p{GCB=xx}', "");
Expect(1, 921599, '\p{^GCB=xx}', "");
Expect(1, 921599, '\P{GCB=xx}', "");
Expect(0, 921599, '\P{^GCB=xx}', "");
Expect(1, 921600, '\p{GCB= 	XX}', "");
Expect(0, 921600, '\p{^GCB= 	XX}', "");
Expect(0, 921600, '\P{GCB= 	XX}', "");
Expect(1, 921600, '\P{^GCB= 	XX}', "");
Expect(0, 921599, '\p{GCB= 	XX}', "");
Expect(1, 921599, '\p{^GCB= 	XX}', "");
Expect(1, 921599, '\P{GCB= 	XX}', "");
Expect(0, 921599, '\P{^GCB= 	XX}', "");
Error('\p{Is_Grapheme_Cluster_Break=_/a/Other}');
Error('\P{Is_Grapheme_Cluster_Break=_/a/Other}');
Expect(1, 921600, '\p{Is_Grapheme_Cluster_Break=other}', "");
Expect(0, 921600, '\p{^Is_Grapheme_Cluster_Break=other}', "");
Expect(0, 921600, '\P{Is_Grapheme_Cluster_Break=other}', "");
Expect(1, 921600, '\P{^Is_Grapheme_Cluster_Break=other}', "");
Expect(0, 921599, '\p{Is_Grapheme_Cluster_Break=other}', "");
Expect(1, 921599, '\p{^Is_Grapheme_Cluster_Break=other}', "");
Expect(1, 921599, '\P{Is_Grapheme_Cluster_Break=other}', "");
Expect(0, 921599, '\P{^Is_Grapheme_Cluster_Break=other}', "");
Expect(1, 921600, '\p{Is_Grapheme_Cluster_Break=__Other}', "");
Expect(0, 921600, '\p{^Is_Grapheme_Cluster_Break=__Other}', "");
Expect(0, 921600, '\P{Is_Grapheme_Cluster_Break=__Other}', "");
Expect(1, 921600, '\P{^Is_Grapheme_Cluster_Break=__Other}', "");
Expect(0, 921599, '\p{Is_Grapheme_Cluster_Break=__Other}', "");
Expect(1, 921599, '\p{^Is_Grapheme_Cluster_Break=__Other}', "");
Expect(1, 921599, '\P{Is_Grapheme_Cluster_Break=__Other}', "");
Expect(0, 921599, '\P{^Is_Grapheme_Cluster_Break=__Other}', "");
Error('\p{Is_GCB=/a/-	XX}');
Error('\P{Is_GCB=/a/-	XX}');
Expect(1, 921600, '\p{Is_GCB=xx}', "");
Expect(0, 921600, '\p{^Is_GCB=xx}', "");
Expect(0, 921600, '\P{Is_GCB=xx}', "");
Expect(1, 921600, '\P{^Is_GCB=xx}', "");
Expect(0, 921599, '\p{Is_GCB=xx}', "");
Expect(1, 921599, '\p{^Is_GCB=xx}', "");
Expect(1, 921599, '\P{Is_GCB=xx}', "");
Expect(0, 921599, '\P{^Is_GCB=xx}', "");
Expect(1, 921600, '\p{Is_GCB= 	XX}', "");
Expect(0, 921600, '\p{^Is_GCB= 	XX}', "");
Expect(0, 921600, '\P{Is_GCB= 	XX}', "");
Expect(1, 921600, '\P{^Is_GCB= 	XX}', "");
Expect(0, 921599, '\p{Is_GCB= 	XX}', "");
Expect(1, 921599, '\p{^Is_GCB= 	XX}', "");
Expect(1, 921599, '\P{Is_GCB= 	XX}', "");
Expect(0, 921599, '\P{^Is_GCB= 	XX}', "");
Error('\p{Grapheme_Base= 	NO/a/}');
Error('\P{Grapheme_Base= 	NO/a/}');
Expect(1, 195102, '\p{Grapheme_Base=no}', "");
Expect(0, 195102, '\p{^Grapheme_Base=no}', "");
Expect(0, 195102, '\P{Grapheme_Base=no}', "");
Expect(1, 195102, '\P{^Grapheme_Base=no}', "");
Expect(0, 195101, '\p{Grapheme_Base=no}', "");
Expect(1, 195101, '\p{^Grapheme_Base=no}', "");
Expect(1, 195101, '\P{Grapheme_Base=no}', "");
Expect(0, 195101, '\P{^Grapheme_Base=no}', "");
Expect(1, 195102, '\p{Grapheme_Base:_	NO}', "");
Expect(0, 195102, '\p{^Grapheme_Base:_	NO}', "");
Expect(0, 195102, '\P{Grapheme_Base:_	NO}', "");
Expect(1, 195102, '\P{^Grapheme_Base:_	NO}', "");
Expect(0, 195101, '\p{Grapheme_Base:_	NO}', "");
Expect(1, 195101, '\p{^Grapheme_Base:_	NO}', "");
Expect(1, 195101, '\P{Grapheme_Base:_	NO}', "");
Expect(0, 195101, '\P{^Grapheme_Base:_	NO}', "");
Error('\p{Gr_Base=- N/a/}');
Error('\P{Gr_Base=- N/a/}');
Expect(1, 195102, '\p{Gr_Base=n}', "");
Expect(0, 195102, '\p{^Gr_Base=n}', "");
Expect(0, 195102, '\P{Gr_Base=n}', "");
Expect(1, 195102, '\P{^Gr_Base=n}', "");
Expect(0, 195101, '\p{Gr_Base=n}', "");
Expect(1, 195101, '\p{^Gr_Base=n}', "");
Expect(1, 195101, '\P{Gr_Base=n}', "");
Expect(0, 195101, '\P{^Gr_Base=n}', "");
Expect(1, 195102, '\p{Gr_Base=	 N}', "");
Expect(0, 195102, '\p{^Gr_Base=	 N}', "");
Expect(0, 195102, '\P{Gr_Base=	 N}', "");
Expect(1, 195102, '\P{^Gr_Base=	 N}', "");
Expect(0, 195101, '\p{Gr_Base=	 N}', "");
Expect(1, 195101, '\p{^Gr_Base=	 N}', "");
Expect(1, 195101, '\P{Gr_Base=	 N}', "");
Expect(0, 195101, '\P{^Gr_Base=	 N}', "");
Error('\p{Is_Grapheme_Base=_F/a/}');
Error('\P{Is_Grapheme_Base=_F/a/}');
Expect(1, 195102, '\p{Is_Grapheme_Base: f}', "");
Expect(0, 195102, '\p{^Is_Grapheme_Base: f}', "");
Expect(0, 195102, '\P{Is_Grapheme_Base: f}', "");
Expect(1, 195102, '\P{^Is_Grapheme_Base: f}', "");
Expect(0, 195101, '\p{Is_Grapheme_Base: f}', "");
Expect(1, 195101, '\p{^Is_Grapheme_Base: f}', "");
Expect(1, 195101, '\P{Is_Grapheme_Base: f}', "");
Expect(0, 195101, '\P{^Is_Grapheme_Base: f}', "");
Expect(1, 195102, '\p{Is_Grapheme_Base=-F}', "");
Expect(0, 195102, '\p{^Is_Grapheme_Base=-F}', "");
Expect(0, 195102, '\P{Is_Grapheme_Base=-F}', "");
Expect(1, 195102, '\P{^Is_Grapheme_Base=-F}', "");
Expect(0, 195101, '\p{Is_Grapheme_Base=-F}', "");
Expect(1, 195101, '\p{^Is_Grapheme_Base=-F}', "");
Expect(1, 195101, '\P{Is_Grapheme_Base=-F}', "");
Expect(0, 195101, '\P{^Is_Grapheme_Base=-F}', "");
Error('\p{Is_Gr_Base=	False:=}');
Error('\P{Is_Gr_Base=	False:=}');
Expect(1, 195102, '\p{Is_Gr_Base: false}', "");
Expect(0, 195102, '\p{^Is_Gr_Base: false}', "");
Expect(0, 195102, '\P{Is_Gr_Base: false}', "");
Expect(1, 195102, '\P{^Is_Gr_Base: false}', "");
Expect(0, 195101, '\p{Is_Gr_Base: false}', "");
Expect(1, 195101, '\p{^Is_Gr_Base: false}', "");
Expect(1, 195101, '\P{Is_Gr_Base: false}', "");
Expect(0, 195101, '\P{^Is_Gr_Base: false}', "");
Expect(1, 195102, '\p{Is_Gr_Base:   _ False}', "");
Expect(0, 195102, '\p{^Is_Gr_Base:   _ False}', "");
Expect(0, 195102, '\P{Is_Gr_Base:   _ False}', "");
Expect(1, 195102, '\P{^Is_Gr_Base:   _ False}', "");
Expect(0, 195101, '\p{Is_Gr_Base:   _ False}', "");
Expect(1, 195101, '\p{^Is_Gr_Base:   _ False}', "");
Expect(1, 195101, '\P{Is_Gr_Base:   _ False}', "");
Expect(0, 195101, '\P{^Is_Gr_Base:   _ False}', "");
Error('\p{Grapheme_Base=/a/YES}');
Error('\P{Grapheme_Base=/a/YES}');
Expect(1, 195101, '\p{Grapheme_Base=yes}', "");
Expect(0, 195101, '\p{^Grapheme_Base=yes}', "");
Expect(0, 195101, '\P{Grapheme_Base=yes}', "");
Expect(1, 195101, '\P{^Grapheme_Base=yes}', "");
Expect(0, 195102, '\p{Grapheme_Base=yes}', "");
Expect(1, 195102, '\p{^Grapheme_Base=yes}', "");
Expect(1, 195102, '\P{Grapheme_Base=yes}', "");
Expect(0, 195102, '\P{^Grapheme_Base=yes}', "");
Expect(1, 195101, '\p{Grapheme_Base=		Yes}', "");
Expect(0, 195101, '\p{^Grapheme_Base=		Yes}', "");
Expect(0, 195101, '\P{Grapheme_Base=		Yes}', "");
Expect(1, 195101, '\P{^Grapheme_Base=		Yes}', "");
Expect(0, 195102, '\p{Grapheme_Base=		Yes}', "");
Expect(1, 195102, '\p{^Grapheme_Base=		Yes}', "");
Expect(1, 195102, '\P{Grapheme_Base=		Yes}', "");
Expect(0, 195102, '\P{^Grapheme_Base=		Yes}', "");
Error('\p{Gr_Base=:=_	Y}');
Error('\P{Gr_Base=:=_	Y}');
Expect(1, 195101, '\p{Gr_Base=y}', "");
Expect(0, 195101, '\p{^Gr_Base=y}', "");
Expect(0, 195101, '\P{Gr_Base=y}', "");
Expect(1, 195101, '\P{^Gr_Base=y}', "");
Expect(0, 195102, '\p{Gr_Base=y}', "");
Expect(1, 195102, '\p{^Gr_Base=y}', "");
Expect(1, 195102, '\P{Gr_Base=y}', "");
Expect(0, 195102, '\P{^Gr_Base=y}', "");
Expect(1, 195101, '\p{Gr_Base=	 Y}', "");
Expect(0, 195101, '\p{^Gr_Base=	 Y}', "");
Expect(0, 195101, '\P{Gr_Base=	 Y}', "");
Expect(1, 195101, '\P{^Gr_Base=	 Y}', "");
Expect(0, 195102, '\p{Gr_Base=	 Y}', "");
Expect(1, 195102, '\p{^Gr_Base=	 Y}', "");
Expect(1, 195102, '\P{Gr_Base=	 Y}', "");
Expect(0, 195102, '\P{^Gr_Base=	 Y}', "");
Error('\p{Is_Grapheme_Base:    /a/T}');
Error('\P{Is_Grapheme_Base:    /a/T}');
Expect(1, 195101, '\p{Is_Grapheme_Base=t}', "");
Expect(0, 195101, '\p{^Is_Grapheme_Base=t}', "");
Expect(0, 195101, '\P{Is_Grapheme_Base=t}', "");
Expect(1, 195101, '\P{^Is_Grapheme_Base=t}', "");
Expect(0, 195102, '\p{Is_Grapheme_Base=t}', "");
Expect(1, 195102, '\p{^Is_Grapheme_Base=t}', "");
Expect(1, 195102, '\P{Is_Grapheme_Base=t}', "");
Expect(0, 195102, '\P{^Is_Grapheme_Base=t}', "");
Expect(1, 195101, '\p{Is_Grapheme_Base=		t}', "");
Expect(0, 195101, '\p{^Is_Grapheme_Base=		t}', "");
Expect(0, 195101, '\P{Is_Grapheme_Base=		t}', "");
Expect(1, 195101, '\P{^Is_Grapheme_Base=		t}', "");
Expect(0, 195102, '\p{Is_Grapheme_Base=		t}', "");
Expect(1, 195102, '\p{^Is_Grapheme_Base=		t}', "");
Expect(1, 195102, '\P{Is_Grapheme_Base=		t}', "");
Expect(0, 195102, '\P{^Is_Grapheme_Base=		t}', "");
Error('\p{Is_Gr_Base= TRUE/a/}');
Error('\P{Is_Gr_Base= TRUE/a/}');
Expect(1, 195101, '\p{Is_Gr_Base=true}', "");
Expect(0, 195101, '\p{^Is_Gr_Base=true}', "");
Expect(0, 195101, '\P{Is_Gr_Base=true}', "");
Expect(1, 195101, '\P{^Is_Gr_Base=true}', "");
Expect(0, 195102, '\p{Is_Gr_Base=true}', "");
Expect(1, 195102, '\p{^Is_Gr_Base=true}', "");
Expect(1, 195102, '\P{Is_Gr_Base=true}', "");
Expect(0, 195102, '\P{^Is_Gr_Base=true}', "");
Expect(1, 195101, '\p{Is_Gr_Base= -True}', "");
Expect(0, 195101, '\p{^Is_Gr_Base= -True}', "");
Expect(0, 195101, '\P{Is_Gr_Base= -True}', "");
Expect(1, 195101, '\P{^Is_Gr_Base= -True}', "");
Expect(0, 195102, '\p{Is_Gr_Base= -True}', "");
Expect(1, 195102, '\p{^Is_Gr_Base= -True}', "");
Expect(1, 195102, '\P{Is_Gr_Base= -True}', "");
Expect(0, 195102, '\P{^Is_Gr_Base= -True}', "");
Error('\p{Grapheme_Extend=	/a/No}');
Error('\P{Grapheme_Extend=	/a/No}');
Expect(1, 918000, '\p{Grapheme_Extend=no}', "");
Expect(0, 918000, '\p{^Grapheme_Extend=no}', "");
Expect(0, 918000, '\P{Grapheme_Extend=no}', "");
Expect(1, 918000, '\P{^Grapheme_Extend=no}', "");
Expect(0, 917999, '\p{Grapheme_Extend=no}', "");
Expect(1, 917999, '\p{^Grapheme_Extend=no}', "");
Expect(1, 917999, '\P{Grapheme_Extend=no}', "");
Expect(0, 917999, '\P{^Grapheme_Extend=no}', "");
Expect(1, 918000, '\p{Grapheme_Extend=	-No}', "");
Expect(0, 918000, '\p{^Grapheme_Extend=	-No}', "");
Expect(0, 918000, '\P{Grapheme_Extend=	-No}', "");
Expect(1, 918000, '\P{^Grapheme_Extend=	-No}', "");
Expect(0, 917999, '\p{Grapheme_Extend=	-No}', "");
Expect(1, 917999, '\p{^Grapheme_Extend=	-No}', "");
Expect(1, 917999, '\P{Grapheme_Extend=	-No}', "");
Expect(0, 917999, '\P{^Grapheme_Extend=	-No}', "");
Error('\p{Gr_Ext=	-N:=}');
Error('\P{Gr_Ext=	-N:=}');
Expect(1, 918000, '\p{Gr_Ext=n}', "");
Expect(0, 918000, '\p{^Gr_Ext=n}', "");
Expect(0, 918000, '\P{Gr_Ext=n}', "");
Expect(1, 918000, '\P{^Gr_Ext=n}', "");
Expect(0, 917999, '\p{Gr_Ext=n}', "");
Expect(1, 917999, '\p{^Gr_Ext=n}', "");
Expect(1, 917999, '\P{Gr_Ext=n}', "");
Expect(0, 917999, '\P{^Gr_Ext=n}', "");
Expect(1, 918000, '\p{Gr_Ext=- N}', "");
Expect(0, 918000, '\p{^Gr_Ext=- N}', "");
Expect(0, 918000, '\P{Gr_Ext=- N}', "");
Expect(1, 918000, '\P{^Gr_Ext=- N}', "");
Expect(0, 917999, '\p{Gr_Ext=- N}', "");
Expect(1, 917999, '\p{^Gr_Ext=- N}', "");
Expect(1, 917999, '\P{Gr_Ext=- N}', "");
Expect(0, 917999, '\P{^Gr_Ext=- N}', "");
Error('\p{Is_Grapheme_Extend=_f/a/}');
Error('\P{Is_Grapheme_Extend=_f/a/}');
Expect(1, 918000, '\p{Is_Grapheme_Extend:f}', "");
Expect(0, 918000, '\p{^Is_Grapheme_Extend:f}', "");
Expect(0, 918000, '\P{Is_Grapheme_Extend:f}', "");
Expect(1, 918000, '\P{^Is_Grapheme_Extend:f}', "");
Expect(0, 917999, '\p{Is_Grapheme_Extend:f}', "");
Expect(1, 917999, '\p{^Is_Grapheme_Extend:f}', "");
Expect(1, 917999, '\P{Is_Grapheme_Extend:f}', "");
Expect(0, 917999, '\P{^Is_Grapheme_Extend:f}', "");
Expect(1, 918000, '\p{Is_Grapheme_Extend=-	f}', "");
Expect(0, 918000, '\p{^Is_Grapheme_Extend=-	f}', "");
Expect(0, 918000, '\P{Is_Grapheme_Extend=-	f}', "");
Expect(1, 918000, '\P{^Is_Grapheme_Extend=-	f}', "");
Expect(0, 917999, '\p{Is_Grapheme_Extend=-	f}', "");
Expect(1, 917999, '\p{^Is_Grapheme_Extend=-	f}', "");
Expect(1, 917999, '\P{Is_Grapheme_Extend=-	f}', "");
Expect(0, 917999, '\P{^Is_Grapheme_Extend=-	f}', "");
Error('\p{Is_Gr_Ext: :=False}');
Error('\P{Is_Gr_Ext: :=False}');
Expect(1, 918000, '\p{Is_Gr_Ext=false}', "");
Expect(0, 918000, '\p{^Is_Gr_Ext=false}', "");
Expect(0, 918000, '\P{Is_Gr_Ext=false}', "");
Expect(1, 918000, '\P{^Is_Gr_Ext=false}', "");
Expect(0, 917999, '\p{Is_Gr_Ext=false}', "");
Expect(1, 917999, '\p{^Is_Gr_Ext=false}', "");
Expect(1, 917999, '\P{Is_Gr_Ext=false}', "");
Expect(0, 917999, '\P{^Is_Gr_Ext=false}', "");
Expect(1, 918000, '\p{Is_Gr_Ext=-false}', "");
Expect(0, 918000, '\p{^Is_Gr_Ext=-false}', "");
Expect(0, 918000, '\P{Is_Gr_Ext=-false}', "");
Expect(1, 918000, '\P{^Is_Gr_Ext=-false}', "");
Expect(0, 917999, '\p{Is_Gr_Ext=-false}', "");
Expect(1, 917999, '\p{^Is_Gr_Ext=-false}', "");
Expect(1, 917999, '\P{Is_Gr_Ext=-false}', "");
Expect(0, 917999, '\P{^Is_Gr_Ext=-false}', "");
Error('\p{Grapheme_Extend:   _:=Yes}');
Error('\P{Grapheme_Extend:   _:=Yes}');
Expect(1, 917999, '\p{Grapheme_Extend=yes}', "");
Expect(0, 917999, '\p{^Grapheme_Extend=yes}', "");
Expect(0, 917999, '\P{Grapheme_Extend=yes}', "");
Expect(1, 917999, '\P{^Grapheme_Extend=yes}', "");
Expect(0, 918000, '\p{Grapheme_Extend=yes}', "");
Expect(1, 918000, '\p{^Grapheme_Extend=yes}', "");
Expect(1, 918000, '\P{Grapheme_Extend=yes}', "");
Expect(0, 918000, '\P{^Grapheme_Extend=yes}', "");
Expect(1, 917999, '\p{Grapheme_Extend=__Yes}', "");
Expect(0, 917999, '\p{^Grapheme_Extend=__Yes}', "");
Expect(0, 917999, '\P{Grapheme_Extend=__Yes}', "");
Expect(1, 917999, '\P{^Grapheme_Extend=__Yes}', "");
Expect(0, 918000, '\p{Grapheme_Extend=__Yes}', "");
Expect(1, 918000, '\p{^Grapheme_Extend=__Yes}', "");
Expect(1, 918000, '\P{Grapheme_Extend=__Yes}', "");
Expect(0, 918000, '\P{^Grapheme_Extend=__Yes}', "");
Error('\p{Gr_Ext=:=_Y}');
Error('\P{Gr_Ext=:=_Y}');
Expect(1, 917999, '\p{Gr_Ext=y}', "");
Expect(0, 917999, '\p{^Gr_Ext=y}', "");
Expect(0, 917999, '\P{Gr_Ext=y}', "");
Expect(1, 917999, '\P{^Gr_Ext=y}', "");
Expect(0, 918000, '\p{Gr_Ext=y}', "");
Expect(1, 918000, '\p{^Gr_Ext=y}', "");
Expect(1, 918000, '\P{Gr_Ext=y}', "");
Expect(0, 918000, '\P{^Gr_Ext=y}', "");
Expect(1, 917999, '\p{Gr_Ext=_Y}', "");
Expect(0, 917999, '\p{^Gr_Ext=_Y}', "");
Expect(0, 917999, '\P{Gr_Ext=_Y}', "");
Expect(1, 917999, '\P{^Gr_Ext=_Y}', "");
Expect(0, 918000, '\p{Gr_Ext=_Y}', "");
Expect(1, 918000, '\p{^Gr_Ext=_Y}', "");
Expect(1, 918000, '\P{Gr_Ext=_Y}', "");
Expect(0, 918000, '\P{^Gr_Ext=_Y}', "");
Error('\p{Is_Grapheme_Extend=/a/	T}');
Error('\P{Is_Grapheme_Extend=/a/	T}');
Expect(1, 917999, '\p{Is_Grapheme_Extend=t}', "");
Expect(0, 917999, '\p{^Is_Grapheme_Extend=t}', "");
Expect(0, 917999, '\P{Is_Grapheme_Extend=t}', "");
Expect(1, 917999, '\P{^Is_Grapheme_Extend=t}', "");
Expect(0, 918000, '\p{Is_Grapheme_Extend=t}', "");
Expect(1, 918000, '\p{^Is_Grapheme_Extend=t}', "");
Expect(1, 918000, '\P{Is_Grapheme_Extend=t}', "");
Expect(0, 918000, '\P{^Is_Grapheme_Extend=t}', "");
Expect(1, 917999, '\p{Is_Grapheme_Extend=_T}', "");
Expect(0, 917999, '\p{^Is_Grapheme_Extend=_T}', "");
Expect(0, 917999, '\P{Is_Grapheme_Extend=_T}', "");
Expect(1, 917999, '\P{^Is_Grapheme_Extend=_T}', "");
Expect(0, 918000, '\p{Is_Grapheme_Extend=_T}', "");
Expect(1, 918000, '\p{^Is_Grapheme_Extend=_T}', "");
Expect(1, 918000, '\P{Is_Grapheme_Extend=_T}', "");
Expect(0, 918000, '\P{^Is_Grapheme_Extend=_T}', "");
Error('\p{Is_Gr_Ext=/a/_-TRUE}');
Error('\P{Is_Gr_Ext=/a/_-TRUE}');
Expect(1, 917999, '\p{Is_Gr_Ext=true}', "");
Expect(0, 917999, '\p{^Is_Gr_Ext=true}', "");
Expect(0, 917999, '\P{Is_Gr_Ext=true}', "");
Expect(1, 917999, '\P{^Is_Gr_Ext=true}', "");
Expect(0, 918000, '\p{Is_Gr_Ext=true}', "");
Expect(1, 918000, '\p{^Is_Gr_Ext=true}', "");
Expect(1, 918000, '\P{Is_Gr_Ext=true}', "");
Expect(0, 918000, '\P{^Is_Gr_Ext=true}', "");
Expect(1, 917999, '\p{Is_Gr_Ext=--true}', "");
Expect(0, 917999, '\p{^Is_Gr_Ext=--true}', "");
Expect(0, 917999, '\P{Is_Gr_Ext=--true}', "");
Expect(1, 917999, '\P{^Is_Gr_Ext=--true}', "");
Expect(0, 918000, '\p{Is_Gr_Ext=--true}', "");
Expect(1, 918000, '\p{^Is_Gr_Ext=--true}', "");
Expect(1, 918000, '\P{Is_Gr_Ext=--true}', "");
Expect(0, 918000, '\P{^Is_Gr_Ext=--true}', "");
Error('\p{Grapheme_Link=No}');
Error('\P{Grapheme_Link=No}');
Error('\p{Gr_Link=N}');
Error('\P{Gr_Link=N}');
Error('\p{Is_Grapheme_Link=F}');
Error('\P{Is_Grapheme_Link=F}');
Error('\p{Is_Gr_Link=False}');
Error('\P{Is_Gr_Link=False}');
Error('\p{Grapheme_Link=Yes}');
Error('\P{Grapheme_Link=Yes}');
Error('\p{Gr_Link=Y}');
Error('\P{Gr_Link=Y}');
Error('\p{Is_Grapheme_Link=T}');
Error('\P{Is_Grapheme_Link=T}');
Error('\p{Is_Gr_Link=True}');
Error('\P{Is_Gr_Link=True}');
Error('\p{Hex_Digit= /a/NO}');
Error('\P{Hex_Digit= /a/NO}');
Expect(1, 65351, '\p{Hex_Digit:no}', "");
Expect(0, 65351, '\p{^Hex_Digit:no}', "");
Expect(0, 65351, '\P{Hex_Digit:no}', "");
Expect(1, 65351, '\P{^Hex_Digit:no}', "");
Expect(0, 65350, '\p{Hex_Digit:no}', "");
Expect(1, 65350, '\p{^Hex_Digit:no}', "");
Expect(1, 65350, '\P{Hex_Digit:no}', "");
Expect(0, 65350, '\P{^Hex_Digit:no}', "");
Expect(1, 65351, '\p{Hex_Digit=__No}', "");
Expect(0, 65351, '\p{^Hex_Digit=__No}', "");
Expect(0, 65351, '\P{Hex_Digit=__No}', "");
Expect(1, 65351, '\P{^Hex_Digit=__No}', "");
Expect(0, 65350, '\p{Hex_Digit=__No}', "");
Expect(1, 65350, '\p{^Hex_Digit=__No}', "");
Expect(1, 65350, '\P{Hex_Digit=__No}', "");
Expect(0, 65350, '\P{^Hex_Digit=__No}', "");
Error('\p{Hex= /a/n}');
Error('\P{Hex= /a/n}');
Expect(1, 65351, '\p{Hex=n}', "");
Expect(0, 65351, '\p{^Hex=n}', "");
Expect(0, 65351, '\P{Hex=n}', "");
Expect(1, 65351, '\P{^Hex=n}', "");
Expect(0, 65350, '\p{Hex=n}', "");
Expect(1, 65350, '\p{^Hex=n}', "");
Expect(1, 65350, '\P{Hex=n}', "");
Expect(0, 65350, '\P{^Hex=n}', "");
Expect(1, 65351, '\p{Hex=		N}', "");
Expect(0, 65351, '\p{^Hex=		N}', "");
Expect(0, 65351, '\P{Hex=		N}', "");
Expect(1, 65351, '\P{^Hex=		N}', "");
Expect(0, 65350, '\p{Hex=		N}', "");
Expect(1, 65350, '\p{^Hex=		N}', "");
Expect(1, 65350, '\P{Hex=		N}', "");
Expect(0, 65350, '\P{^Hex=		N}', "");
Error('\p{Is_Hex_Digit=_:=F}');
Error('\P{Is_Hex_Digit=_:=F}');
Expect(1, 65351, '\p{Is_Hex_Digit=f}', "");
Expect(0, 65351, '\p{^Is_Hex_Digit=f}', "");
Expect(0, 65351, '\P{Is_Hex_Digit=f}', "");
Expect(1, 65351, '\P{^Is_Hex_Digit=f}', "");
Expect(0, 65350, '\p{Is_Hex_Digit=f}', "");
Expect(1, 65350, '\p{^Is_Hex_Digit=f}', "");
Expect(1, 65350, '\P{Is_Hex_Digit=f}', "");
Expect(0, 65350, '\P{^Is_Hex_Digit=f}', "");
Expect(1, 65351, '\p{Is_Hex_Digit=-F}', "");
Expect(0, 65351, '\p{^Is_Hex_Digit=-F}', "");
Expect(0, 65351, '\P{Is_Hex_Digit=-F}', "");
Expect(1, 65351, '\P{^Is_Hex_Digit=-F}', "");
Expect(0, 65350, '\p{Is_Hex_Digit=-F}', "");
Expect(1, 65350, '\p{^Is_Hex_Digit=-F}', "");
Expect(1, 65350, '\P{Is_Hex_Digit=-F}', "");
Expect(0, 65350, '\P{^Is_Hex_Digit=-F}', "");
Error('\p{Is_Hex=-:=False}');
Error('\P{Is_Hex=-:=False}');
Expect(1, 65351, '\p{Is_Hex=false}', "");
Expect(0, 65351, '\p{^Is_Hex=false}', "");
Expect(0, 65351, '\P{Is_Hex=false}', "");
Expect(1, 65351, '\P{^Is_Hex=false}', "");
Expect(0, 65350, '\p{Is_Hex=false}', "");
Expect(1, 65350, '\p{^Is_Hex=false}', "");
Expect(1, 65350, '\P{Is_Hex=false}', "");
Expect(0, 65350, '\P{^Is_Hex=false}', "");
Expect(1, 65351, '\p{Is_Hex=  false}', "");
Expect(0, 65351, '\p{^Is_Hex=  false}', "");
Expect(0, 65351, '\P{Is_Hex=  false}', "");
Expect(1, 65351, '\P{^Is_Hex=  false}', "");
Expect(0, 65350, '\p{Is_Hex=  false}', "");
Expect(1, 65350, '\p{^Is_Hex=  false}', "");
Expect(1, 65350, '\P{Is_Hex=  false}', "");
Expect(0, 65350, '\P{^Is_Hex=  false}', "");
Error('\p{Hex_Digit=:=-	Yes}');
Error('\P{Hex_Digit=:=-	Yes}');
Expect(1, 65350, '\p{Hex_Digit=yes}', "");
Expect(0, 65350, '\p{^Hex_Digit=yes}', "");
Expect(0, 65350, '\P{Hex_Digit=yes}', "");
Expect(1, 65350, '\P{^Hex_Digit=yes}', "");
Expect(0, 65351, '\p{Hex_Digit=yes}', "");
Expect(1, 65351, '\p{^Hex_Digit=yes}', "");
Expect(1, 65351, '\P{Hex_Digit=yes}', "");
Expect(0, 65351, '\P{^Hex_Digit=yes}', "");
Expect(1, 65350, '\p{Hex_Digit=		yes}', "");
Expect(0, 65350, '\p{^Hex_Digit=		yes}', "");
Expect(0, 65350, '\P{Hex_Digit=		yes}', "");
Expect(1, 65350, '\P{^Hex_Digit=		yes}', "");
Expect(0, 65351, '\p{Hex_Digit=		yes}', "");
Expect(1, 65351, '\p{^Hex_Digit=		yes}', "");
Expect(1, 65351, '\P{Hex_Digit=		yes}', "");
Expect(0, 65351, '\P{^Hex_Digit=		yes}', "");
Error('\p{Hex=:=	Y}');
Error('\P{Hex=:=	Y}');
Expect(1, 65350, '\p{Hex=y}', "");
Expect(0, 65350, '\p{^Hex=y}', "");
Expect(0, 65350, '\P{Hex=y}', "");
Expect(1, 65350, '\P{^Hex=y}', "");
Expect(0, 65351, '\p{Hex=y}', "");
Expect(1, 65351, '\p{^Hex=y}', "");
Expect(1, 65351, '\P{Hex=y}', "");
Expect(0, 65351, '\P{^Hex=y}', "");
Expect(1, 65350, '\p{Hex=_	y}', "");
Expect(0, 65350, '\p{^Hex=_	y}', "");
Expect(0, 65350, '\P{Hex=_	y}', "");
Expect(1, 65350, '\P{^Hex=_	y}', "");
Expect(0, 65351, '\p{Hex=_	y}', "");
Expect(1, 65351, '\p{^Hex=_	y}', "");
Expect(1, 65351, '\P{Hex=_	y}', "");
Expect(0, 65351, '\P{^Hex=_	y}', "");
Error('\p{Is_Hex_Digit=_:=t}');
Error('\P{Is_Hex_Digit=_:=t}');
Expect(1, 65350, '\p{Is_Hex_Digit=t}', "");
Expect(0, 65350, '\p{^Is_Hex_Digit=t}', "");
Expect(0, 65350, '\P{Is_Hex_Digit=t}', "");
Expect(1, 65350, '\P{^Is_Hex_Digit=t}', "");
Expect(0, 65351, '\p{Is_Hex_Digit=t}', "");
Expect(1, 65351, '\p{^Is_Hex_Digit=t}', "");
Expect(1, 65351, '\P{Is_Hex_Digit=t}', "");
Expect(0, 65351, '\P{^Is_Hex_Digit=t}', "");
Expect(1, 65350, '\p{Is_Hex_Digit= _T}', "");
Expect(0, 65350, '\p{^Is_Hex_Digit= _T}', "");
Expect(0, 65350, '\P{Is_Hex_Digit= _T}', "");
Expect(1, 65350, '\P{^Is_Hex_Digit= _T}', "");
Expect(0, 65351, '\p{Is_Hex_Digit= _T}', "");
Expect(1, 65351, '\p{^Is_Hex_Digit= _T}', "");
Expect(1, 65351, '\P{Is_Hex_Digit= _T}', "");
Expect(0, 65351, '\P{^Is_Hex_Digit= _T}', "");
Error('\p{Is_Hex=-/a/true}');
Error('\P{Is_Hex=-/a/true}');
Expect(1, 65350, '\p{Is_Hex=true}', "");
Expect(0, 65350, '\p{^Is_Hex=true}', "");
Expect(0, 65350, '\P{Is_Hex=true}', "");
Expect(1, 65350, '\P{^Is_Hex=true}', "");
Expect(0, 65351, '\p{Is_Hex=true}', "");
Expect(1, 65351, '\p{^Is_Hex=true}', "");
Expect(1, 65351, '\P{Is_Hex=true}', "");
Expect(0, 65351, '\P{^Is_Hex=true}', "");
Expect(1, 65350, '\p{Is_Hex=- true}', "");
Expect(0, 65350, '\p{^Is_Hex=- true}', "");
Expect(0, 65350, '\P{Is_Hex=- true}', "");
Expect(1, 65350, '\P{^Is_Hex=- true}', "");
Expect(0, 65351, '\p{Is_Hex=- true}', "");
Expect(1, 65351, '\p{^Is_Hex=- true}', "");
Expect(1, 65351, '\P{Is_Hex=- true}', "");
Expect(0, 65351, '\P{^Is_Hex=- true}', "");
Error('\p{Hyphen=/a/-_No}');
Error('\P{Hyphen=/a/-_No}');
Expect(1, 65382, '\p{Hyphen=no}', 'deprecated');
Expect(0, 65382, '\p{^Hyphen=no}', 'deprecated');
Expect(0, 65382, '\P{Hyphen=no}', 'deprecated');
Expect(1, 65382, '\P{^Hyphen=no}', 'deprecated');
Expect(0, 65381, '\p{Hyphen=no}', 'deprecated');
Expect(1, 65381, '\p{^Hyphen=no}', 'deprecated');
Expect(1, 65381, '\P{Hyphen=no}', 'deprecated');
Expect(0, 65381, '\P{^Hyphen=no}', 'deprecated');
Expect(1, 65382, '\p{Hyphen=	no}', 'deprecated');
Expect(0, 65382, '\p{^Hyphen=	no}', 'deprecated');
Expect(0, 65382, '\P{Hyphen=	no}', 'deprecated');
Expect(1, 65382, '\P{^Hyphen=	no}', 'deprecated');
Expect(0, 65381, '\p{Hyphen=	no}', 'deprecated');
Expect(1, 65381, '\p{^Hyphen=	no}', 'deprecated');
Expect(1, 65381, '\P{Hyphen=	no}', 'deprecated');
Expect(0, 65381, '\P{^Hyphen=	no}', 'deprecated');
Error('\p{Is_Hyphen=	:=N}');
Error('\P{Is_Hyphen=	:=N}');
Expect(1, 65382, '\p{Is_Hyphen=n}', 'deprecated');
Expect(0, 65382, '\p{^Is_Hyphen=n}', 'deprecated');
Expect(0, 65382, '\P{Is_Hyphen=n}', 'deprecated');
Expect(1, 65382, '\P{^Is_Hyphen=n}', 'deprecated');
Expect(0, 65381, '\p{Is_Hyphen=n}', 'deprecated');
Expect(1, 65381, '\p{^Is_Hyphen=n}', 'deprecated');
Expect(1, 65381, '\P{Is_Hyphen=n}', 'deprecated');
Expect(0, 65381, '\P{^Is_Hyphen=n}', 'deprecated');
Expect(1, 65382, '\p{Is_Hyphen=_-N}', 'deprecated');
Expect(0, 65382, '\p{^Is_Hyphen=_-N}', 'deprecated');
Expect(0, 65382, '\P{Is_Hyphen=_-N}', 'deprecated');
Expect(1, 65382, '\P{^Is_Hyphen=_-N}', 'deprecated');
Expect(0, 65381, '\p{Is_Hyphen=_-N}', 'deprecated');
Expect(1, 65381, '\p{^Is_Hyphen=_-N}', 'deprecated');
Expect(1, 65381, '\P{Is_Hyphen=_-N}', 'deprecated');
Expect(0, 65381, '\P{^Is_Hyphen=_-N}', 'deprecated');
Error('\p{Hyphen=f:=}');
Error('\P{Hyphen=f:=}');
Expect(1, 65382, '\p{Hyphen=f}', 'deprecated');
Expect(0, 65382, '\p{^Hyphen=f}', 'deprecated');
Expect(0, 65382, '\P{Hyphen=f}', 'deprecated');
Expect(1, 65382, '\P{^Hyphen=f}', 'deprecated');
Expect(0, 65381, '\p{Hyphen=f}', 'deprecated');
Expect(1, 65381, '\p{^Hyphen=f}', 'deprecated');
Expect(1, 65381, '\P{Hyphen=f}', 'deprecated');
Expect(0, 65381, '\P{^Hyphen=f}', 'deprecated');
Expect(1, 65382, '\p{Hyphen=_ F}', 'deprecated');
Expect(0, 65382, '\p{^Hyphen=_ F}', 'deprecated');
Expect(0, 65382, '\P{Hyphen=_ F}', 'deprecated');
Expect(1, 65382, '\P{^Hyphen=_ F}', 'deprecated');
Expect(0, 65381, '\p{Hyphen=_ F}', 'deprecated');
Expect(1, 65381, '\p{^Hyphen=_ F}', 'deprecated');
Expect(1, 65381, '\P{Hyphen=_ F}', 'deprecated');
Expect(0, 65381, '\P{^Hyphen=_ F}', 'deprecated');
Error('\p{Is_Hyphen=/a/- false}');
Error('\P{Is_Hyphen=/a/- false}');
Expect(1, 65382, '\p{Is_Hyphen=false}', 'deprecated');
Expect(0, 65382, '\p{^Is_Hyphen=false}', 'deprecated');
Expect(0, 65382, '\P{Is_Hyphen=false}', 'deprecated');
Expect(1, 65382, '\P{^Is_Hyphen=false}', 'deprecated');
Expect(0, 65381, '\p{Is_Hyphen=false}', 'deprecated');
Expect(1, 65381, '\p{^Is_Hyphen=false}', 'deprecated');
Expect(1, 65381, '\P{Is_Hyphen=false}', 'deprecated');
Expect(0, 65381, '\P{^Is_Hyphen=false}', 'deprecated');
Expect(1, 65382, '\p{Is_Hyphen=_ False}', 'deprecated');
Expect(0, 65382, '\p{^Is_Hyphen=_ False}', 'deprecated');
Expect(0, 65382, '\P{Is_Hyphen=_ False}', 'deprecated');
Expect(1, 65382, '\P{^Is_Hyphen=_ False}', 'deprecated');
Expect(0, 65381, '\p{Is_Hyphen=_ False}', 'deprecated');
Expect(1, 65381, '\p{^Is_Hyphen=_ False}', 'deprecated');
Expect(1, 65381, '\P{Is_Hyphen=_ False}', 'deprecated');
Expect(0, 65381, '\P{^Is_Hyphen=_ False}', 'deprecated');
Error('\p{Hyphen=/a/--Yes}');
Error('\P{Hyphen=/a/--Yes}');
Expect(1, 65381, '\p{Hyphen=yes}', 'deprecated');
Expect(0, 65381, '\p{^Hyphen=yes}', 'deprecated');
Expect(0, 65381, '\P{Hyphen=yes}', 'deprecated');
Expect(1, 65381, '\P{^Hyphen=yes}', 'deprecated');
Expect(0, 65382, '\p{Hyphen=yes}', 'deprecated');
Expect(1, 65382, '\p{^Hyphen=yes}', 'deprecated');
Expect(1, 65382, '\P{Hyphen=yes}', 'deprecated');
Expect(0, 65382, '\P{^Hyphen=yes}', 'deprecated');
Expect(1, 65381, '\p{Hyphen= _yes}', 'deprecated');
Expect(0, 65381, '\p{^Hyphen= _yes}', 'deprecated');
Expect(0, 65381, '\P{Hyphen= _yes}', 'deprecated');
Expect(1, 65381, '\P{^Hyphen= _yes}', 'deprecated');
Expect(0, 65382, '\p{Hyphen= _yes}', 'deprecated');
Expect(1, 65382, '\p{^Hyphen= _yes}', 'deprecated');
Expect(1, 65382, '\P{Hyphen= _yes}', 'deprecated');
Expect(0, 65382, '\P{^Hyphen= _yes}', 'deprecated');
Error('\p{Is_Hyphen=/a/  Y}');
Error('\P{Is_Hyphen=/a/  Y}');
Expect(1, 65381, '\p{Is_Hyphen=y}', 'deprecated');
Expect(0, 65381, '\p{^Is_Hyphen=y}', 'deprecated');
Expect(0, 65381, '\P{Is_Hyphen=y}', 'deprecated');
Expect(1, 65381, '\P{^Is_Hyphen=y}', 'deprecated');
Expect(0, 65382, '\p{Is_Hyphen=y}', 'deprecated');
Expect(1, 65382, '\p{^Is_Hyphen=y}', 'deprecated');
Expect(1, 65382, '\P{Is_Hyphen=y}', 'deprecated');
Expect(0, 65382, '\P{^Is_Hyphen=y}', 'deprecated');
Expect(1, 65381, '\p{Is_Hyphen=	Y}', 'deprecated');
Expect(0, 65381, '\p{^Is_Hyphen=	Y}', 'deprecated');
Expect(0, 65381, '\P{Is_Hyphen=	Y}', 'deprecated');
Expect(1, 65381, '\P{^Is_Hyphen=	Y}', 'deprecated');
Expect(0, 65382, '\p{Is_Hyphen=	Y}', 'deprecated');
Expect(1, 65382, '\p{^Is_Hyphen=	Y}', 'deprecated');
Expect(1, 65382, '\P{Is_Hyphen=	Y}', 'deprecated');
Expect(0, 65382, '\P{^Is_Hyphen=	Y}', 'deprecated');
Error('\p{Hyphen=_:=t}');
Error('\P{Hyphen=_:=t}');
Expect(1, 65381, '\p{Hyphen=t}', 'deprecated');
Expect(0, 65381, '\p{^Hyphen=t}', 'deprecated');
Expect(0, 65381, '\P{Hyphen=t}', 'deprecated');
Expect(1, 65381, '\P{^Hyphen=t}', 'deprecated');
Expect(0, 65382, '\p{Hyphen=t}', 'deprecated');
Expect(1, 65382, '\p{^Hyphen=t}', 'deprecated');
Expect(1, 65382, '\P{Hyphen=t}', 'deprecated');
Expect(0, 65382, '\P{^Hyphen=t}', 'deprecated');
Expect(1, 65381, '\p{Hyphen= 	t}', 'deprecated');
Expect(0, 65381, '\p{^Hyphen= 	t}', 'deprecated');
Expect(0, 65381, '\P{Hyphen= 	t}', 'deprecated');
Expect(1, 65381, '\P{^Hyphen= 	t}', 'deprecated');
Expect(0, 65382, '\p{Hyphen= 	t}', 'deprecated');
Expect(1, 65382, '\p{^Hyphen= 	t}', 'deprecated');
Expect(1, 65382, '\P{Hyphen= 	t}', 'deprecated');
Expect(0, 65382, '\P{^Hyphen= 	t}', 'deprecated');
Error('\p{Is_Hyphen=/a/	True}');
Error('\P{Is_Hyphen=/a/	True}');
Expect(1, 65381, '\p{Is_Hyphen=true}', 'deprecated');
Expect(0, 65381, '\p{^Is_Hyphen=true}', 'deprecated');
Expect(0, 65381, '\P{Is_Hyphen=true}', 'deprecated');
Expect(1, 65381, '\P{^Is_Hyphen=true}', 'deprecated');
Expect(0, 65382, '\p{Is_Hyphen=true}', 'deprecated');
Expect(1, 65382, '\p{^Is_Hyphen=true}', 'deprecated');
Expect(1, 65382, '\P{Is_Hyphen=true}', 'deprecated');
Expect(0, 65382, '\P{^Is_Hyphen=true}', 'deprecated');
Expect(1, 65381, '\p{Is_Hyphen=	_True}', 'deprecated');
Expect(0, 65381, '\p{^Is_Hyphen=	_True}', 'deprecated');
Expect(0, 65381, '\P{Is_Hyphen=	_True}', 'deprecated');
Expect(1, 65381, '\P{^Is_Hyphen=	_True}', 'deprecated');
Expect(0, 65382, '\p{Is_Hyphen=	_True}', 'deprecated');
Expect(1, 65382, '\p{^Is_Hyphen=	_True}', 'deprecated');
Expect(1, 65382, '\P{Is_Hyphen=	_True}', 'deprecated');
Expect(0, 65382, '\P{^Is_Hyphen=	_True}', 'deprecated');
Error('\p{ID_Continue=	No/a/}');
Error('\P{ID_Continue=	No/a/}');
Expect(1, 918000, '\p{ID_Continue=no}', "");
Expect(0, 918000, '\p{^ID_Continue=no}', "");
Expect(0, 918000, '\P{ID_Continue=no}', "");
Expect(1, 918000, '\P{^ID_Continue=no}', "");
Expect(0, 917999, '\p{ID_Continue=no}', "");
Expect(1, 917999, '\p{^ID_Continue=no}', "");
Expect(1, 917999, '\P{ID_Continue=no}', "");
Expect(0, 917999, '\P{^ID_Continue=no}', "");
Expect(1, 918000, '\p{ID_Continue=	_No}', "");
Expect(0, 918000, '\p{^ID_Continue=	_No}', "");
Expect(0, 918000, '\P{ID_Continue=	_No}', "");
Expect(1, 918000, '\P{^ID_Continue=	_No}', "");
Expect(0, 917999, '\p{ID_Continue=	_No}', "");
Expect(1, 917999, '\p{^ID_Continue=	_No}', "");
Expect(1, 917999, '\P{ID_Continue=	_No}', "");
Expect(0, 917999, '\P{^ID_Continue=	_No}', "");
Error('\p{IDC:_N/a/}');
Error('\P{IDC:_N/a/}');
Expect(1, 918000, '\p{IDC=n}', "");
Expect(0, 918000, '\p{^IDC=n}', "");
Expect(0, 918000, '\P{IDC=n}', "");
Expect(1, 918000, '\P{^IDC=n}', "");
Expect(0, 917999, '\p{IDC=n}', "");
Expect(1, 917999, '\p{^IDC=n}', "");
Expect(1, 917999, '\P{IDC=n}', "");
Expect(0, 917999, '\P{^IDC=n}', "");
Expect(1, 918000, '\p{IDC=	-N}', "");
Expect(0, 918000, '\p{^IDC=	-N}', "");
Expect(0, 918000, '\P{IDC=	-N}', "");
Expect(1, 918000, '\P{^IDC=	-N}', "");
Expect(0, 917999, '\p{IDC=	-N}', "");
Expect(1, 917999, '\p{^IDC=	-N}', "");
Expect(1, 917999, '\P{IDC=	-N}', "");
Expect(0, 917999, '\P{^IDC=	-N}', "");
Error('\p{Is_ID_Continue=/a/__F}');
Error('\P{Is_ID_Continue=/a/__F}');
Expect(1, 918000, '\p{Is_ID_Continue=f}', "");
Expect(0, 918000, '\p{^Is_ID_Continue=f}', "");
Expect(0, 918000, '\P{Is_ID_Continue=f}', "");
Expect(1, 918000, '\P{^Is_ID_Continue=f}', "");
Expect(0, 917999, '\p{Is_ID_Continue=f}', "");
Expect(1, 917999, '\p{^Is_ID_Continue=f}', "");
Expect(1, 917999, '\P{Is_ID_Continue=f}', "");
Expect(0, 917999, '\P{^Is_ID_Continue=f}', "");
Expect(1, 918000, '\p{Is_ID_Continue=	F}', "");
Expect(0, 918000, '\p{^Is_ID_Continue=	F}', "");
Expect(0, 918000, '\P{Is_ID_Continue=	F}', "");
Expect(1, 918000, '\P{^Is_ID_Continue=	F}', "");
Expect(0, 917999, '\p{Is_ID_Continue=	F}', "");
Expect(1, 917999, '\p{^Is_ID_Continue=	F}', "");
Expect(1, 917999, '\P{Is_ID_Continue=	F}', "");
Expect(0, 917999, '\P{^Is_ID_Continue=	F}', "");
Error('\p{Is_IDC=/a/--False}');
Error('\P{Is_IDC=/a/--False}');
Expect(1, 918000, '\p{Is_IDC: false}', "");
Expect(0, 918000, '\p{^Is_IDC: false}', "");
Expect(0, 918000, '\P{Is_IDC: false}', "");
Expect(1, 918000, '\P{^Is_IDC: false}', "");
Expect(0, 917999, '\p{Is_IDC: false}', "");
Expect(1, 917999, '\p{^Is_IDC: false}', "");
Expect(1, 917999, '\P{Is_IDC: false}', "");
Expect(0, 917999, '\P{^Is_IDC: false}', "");
Expect(1, 918000, '\p{Is_IDC= _false}', "");
Expect(0, 918000, '\p{^Is_IDC= _false}', "");
Expect(0, 918000, '\P{Is_IDC= _false}', "");
Expect(1, 918000, '\P{^Is_IDC= _false}', "");
Expect(0, 917999, '\p{Is_IDC= _false}', "");
Expect(1, 917999, '\p{^Is_IDC= _false}', "");
Expect(1, 917999, '\P{Is_IDC= _false}', "");
Expect(0, 917999, '\P{^Is_IDC= _false}', "");
Error('\p{ID_Continue=	:=Yes}');
Error('\P{ID_Continue=	:=Yes}');
Expect(1, 917999, '\p{ID_Continue=yes}', "");
Expect(0, 917999, '\p{^ID_Continue=yes}', "");
Expect(0, 917999, '\P{ID_Continue=yes}', "");
Expect(1, 917999, '\P{^ID_Continue=yes}', "");
Expect(0, 918000, '\p{ID_Continue=yes}', "");
Expect(1, 918000, '\p{^ID_Continue=yes}', "");
Expect(1, 918000, '\P{ID_Continue=yes}', "");
Expect(0, 918000, '\P{^ID_Continue=yes}', "");
Expect(1, 917999, '\p{ID_Continue=	-yes}', "");
Expect(0, 917999, '\p{^ID_Continue=	-yes}', "");
Expect(0, 917999, '\P{ID_Continue=	-yes}', "");
Expect(1, 917999, '\P{^ID_Continue=	-yes}', "");
Expect(0, 918000, '\p{ID_Continue=	-yes}', "");
Expect(1, 918000, '\p{^ID_Continue=	-yes}', "");
Expect(1, 918000, '\P{ID_Continue=	-yes}', "");
Expect(0, 918000, '\P{^ID_Continue=	-yes}', "");
Error('\p{IDC=:= Y}');
Error('\P{IDC=:= Y}');
Expect(1, 917999, '\p{IDC:y}', "");
Expect(0, 917999, '\p{^IDC:y}', "");
Expect(0, 917999, '\P{IDC:y}', "");
Expect(1, 917999, '\P{^IDC:y}', "");
Expect(0, 918000, '\p{IDC:y}', "");
Expect(1, 918000, '\p{^IDC:y}', "");
Expect(1, 918000, '\P{IDC:y}', "");
Expect(0, 918000, '\P{^IDC:y}', "");
Expect(1, 917999, '\p{IDC=_	Y}', "");
Expect(0, 917999, '\p{^IDC=_	Y}', "");
Expect(0, 917999, '\P{IDC=_	Y}', "");
Expect(1, 917999, '\P{^IDC=_	Y}', "");
Expect(0, 918000, '\p{IDC=_	Y}', "");
Expect(1, 918000, '\p{^IDC=_	Y}', "");
Expect(1, 918000, '\P{IDC=_	Y}', "");
Expect(0, 918000, '\P{^IDC=_	Y}', "");
Error('\p{Is_ID_Continue=--t/a/}');
Error('\P{Is_ID_Continue=--t/a/}');
Expect(1, 917999, '\p{Is_ID_Continue=t}', "");
Expect(0, 917999, '\p{^Is_ID_Continue=t}', "");
Expect(0, 917999, '\P{Is_ID_Continue=t}', "");
Expect(1, 917999, '\P{^Is_ID_Continue=t}', "");
Expect(0, 918000, '\p{Is_ID_Continue=t}', "");
Expect(1, 918000, '\p{^Is_ID_Continue=t}', "");
Expect(1, 918000, '\P{Is_ID_Continue=t}', "");
Expect(0, 918000, '\P{^Is_ID_Continue=t}', "");
Expect(1, 917999, '\p{Is_ID_Continue= _T}', "");
Expect(0, 917999, '\p{^Is_ID_Continue= _T}', "");
Expect(0, 917999, '\P{Is_ID_Continue= _T}', "");
Expect(1, 917999, '\P{^Is_ID_Continue= _T}', "");
Expect(0, 918000, '\p{Is_ID_Continue= _T}', "");
Expect(1, 918000, '\p{^Is_ID_Continue= _T}', "");
Expect(1, 918000, '\P{Is_ID_Continue= _T}', "");
Expect(0, 918000, '\P{^Is_ID_Continue= _T}', "");
Error('\p{Is_IDC= 	True/a/}');
Error('\P{Is_IDC= 	True/a/}');
Expect(1, 917999, '\p{Is_IDC=true}', "");
Expect(0, 917999, '\p{^Is_IDC=true}', "");
Expect(0, 917999, '\P{Is_IDC=true}', "");
Expect(1, 917999, '\P{^Is_IDC=true}', "");
Expect(0, 918000, '\p{Is_IDC=true}', "");
Expect(1, 918000, '\p{^Is_IDC=true}', "");
Expect(1, 918000, '\P{Is_IDC=true}', "");
Expect(0, 918000, '\P{^Is_IDC=true}', "");
Expect(1, 917999, '\p{Is_IDC=  True}', "");
Expect(0, 917999, '\p{^Is_IDC=  True}', "");
Expect(0, 917999, '\P{Is_IDC=  True}', "");
Expect(1, 917999, '\P{^Is_IDC=  True}', "");
Expect(0, 918000, '\p{Is_IDC=  True}', "");
Expect(1, 918000, '\p{^Is_IDC=  True}', "");
Expect(1, 918000, '\P{Is_IDC=  True}', "");
Expect(0, 918000, '\P{^Is_IDC=  True}', "");
Error('\p{ID_Start=/a/	no}');
Error('\P{ID_Start=/a/	no}');
Expect(1, 195102, '\p{ID_Start=no}', "");
Expect(0, 195102, '\p{^ID_Start=no}', "");
Expect(0, 195102, '\P{ID_Start=no}', "");
Expect(1, 195102, '\P{^ID_Start=no}', "");
Expect(0, 195101, '\p{ID_Start=no}', "");
Expect(1, 195101, '\p{^ID_Start=no}', "");
Expect(1, 195101, '\P{ID_Start=no}', "");
Expect(0, 195101, '\P{^ID_Start=no}', "");
Expect(1, 195102, '\p{ID_Start=- no}', "");
Expect(0, 195102, '\p{^ID_Start=- no}', "");
Expect(0, 195102, '\P{ID_Start=- no}', "");
Expect(1, 195102, '\P{^ID_Start=- no}', "");
Expect(0, 195101, '\p{ID_Start=- no}', "");
Expect(1, 195101, '\p{^ID_Start=- no}', "");
Expect(1, 195101, '\P{ID_Start=- no}', "");
Expect(0, 195101, '\P{^ID_Start=- no}', "");
Error('\p{IDS=-/a/N}');
Error('\P{IDS=-/a/N}');
Expect(1, 195102, '\p{IDS:	n}', "");
Expect(0, 195102, '\p{^IDS:	n}', "");
Expect(0, 195102, '\P{IDS:	n}', "");
Expect(1, 195102, '\P{^IDS:	n}', "");
Expect(0, 195101, '\p{IDS:	n}', "");
Expect(1, 195101, '\p{^IDS:	n}', "");
Expect(1, 195101, '\P{IDS:	n}', "");
Expect(0, 195101, '\P{^IDS:	n}', "");
Expect(1, 195102, '\p{IDS:  	n}', "");
Expect(0, 195102, '\p{^IDS:  	n}', "");
Expect(0, 195102, '\P{IDS:  	n}', "");
Expect(1, 195102, '\P{^IDS:  	n}', "");
Expect(0, 195101, '\p{IDS:  	n}', "");
Expect(1, 195101, '\p{^IDS:  	n}', "");
Expect(1, 195101, '\P{IDS:  	n}', "");
Expect(0, 195101, '\P{^IDS:  	n}', "");
Error('\p{Is_ID_Start=/a/	-F}');
Error('\P{Is_ID_Start=/a/	-F}');
Expect(1, 195102, '\p{Is_ID_Start=f}', "");
Expect(0, 195102, '\p{^Is_ID_Start=f}', "");
Expect(0, 195102, '\P{Is_ID_Start=f}', "");
Expect(1, 195102, '\P{^Is_ID_Start=f}', "");
Expect(0, 195101, '\p{Is_ID_Start=f}', "");
Expect(1, 195101, '\p{^Is_ID_Start=f}', "");
Expect(1, 195101, '\P{Is_ID_Start=f}', "");
Expect(0, 195101, '\P{^Is_ID_Start=f}', "");
Expect(1, 195102, '\p{Is_ID_Start= F}', "");
Expect(0, 195102, '\p{^Is_ID_Start= F}', "");
Expect(0, 195102, '\P{Is_ID_Start= F}', "");
Expect(1, 195102, '\P{^Is_ID_Start= F}', "");
Expect(0, 195101, '\p{Is_ID_Start= F}', "");
Expect(1, 195101, '\p{^Is_ID_Start= F}', "");
Expect(1, 195101, '\P{Is_ID_Start= F}', "");
Expect(0, 195101, '\P{^Is_ID_Start= F}', "");
Error('\p{Is_IDS=	:=False}');
Error('\P{Is_IDS=	:=False}');
Expect(1, 195102, '\p{Is_IDS=false}', "");
Expect(0, 195102, '\p{^Is_IDS=false}', "");
Expect(0, 195102, '\P{Is_IDS=false}', "");
Expect(1, 195102, '\P{^Is_IDS=false}', "");
Expect(0, 195101, '\p{Is_IDS=false}', "");
Expect(1, 195101, '\p{^Is_IDS=false}', "");
Expect(1, 195101, '\P{Is_IDS=false}', "");
Expect(0, 195101, '\P{^Is_IDS=false}', "");
Expect(1, 195102, '\p{Is_IDS= FALSE}', "");
Expect(0, 195102, '\p{^Is_IDS= FALSE}', "");
Expect(0, 195102, '\P{Is_IDS= FALSE}', "");
Expect(1, 195102, '\P{^Is_IDS= FALSE}', "");
Expect(0, 195101, '\p{Is_IDS= FALSE}', "");
Expect(1, 195101, '\p{^Is_IDS= FALSE}', "");
Expect(1, 195101, '\P{Is_IDS= FALSE}', "");
Expect(0, 195101, '\P{^Is_IDS= FALSE}', "");
Error('\p{ID_Start=/a/ -Yes}');
Error('\P{ID_Start=/a/ -Yes}');
Expect(1, 195101, '\p{ID_Start=yes}', "");
Expect(0, 195101, '\p{^ID_Start=yes}', "");
Expect(0, 195101, '\P{ID_Start=yes}', "");
Expect(1, 195101, '\P{^ID_Start=yes}', "");
Expect(0, 195102, '\p{ID_Start=yes}', "");
Expect(1, 195102, '\p{^ID_Start=yes}', "");
Expect(1, 195102, '\P{ID_Start=yes}', "");
Expect(0, 195102, '\P{^ID_Start=yes}', "");
Expect(1, 195101, '\p{ID_Start=_Yes}', "");
Expect(0, 195101, '\p{^ID_Start=_Yes}', "");
Expect(0, 195101, '\P{ID_Start=_Yes}', "");
Expect(1, 195101, '\P{^ID_Start=_Yes}', "");
Expect(0, 195102, '\p{ID_Start=_Yes}', "");
Expect(1, 195102, '\p{^ID_Start=_Yes}', "");
Expect(1, 195102, '\P{ID_Start=_Yes}', "");
Expect(0, 195102, '\P{^ID_Start=_Yes}', "");
Error('\p{IDS=:= Y}');
Error('\P{IDS=:= Y}');
Expect(1, 195101, '\p{IDS:y}', "");
Expect(0, 195101, '\p{^IDS:y}', "");
Expect(0, 195101, '\P{IDS:y}', "");
Expect(1, 195101, '\P{^IDS:y}', "");
Expect(0, 195102, '\p{IDS:y}', "");
Expect(1, 195102, '\p{^IDS:y}', "");
Expect(1, 195102, '\P{IDS:y}', "");
Expect(0, 195102, '\P{^IDS:y}', "");
Expect(1, 195101, '\p{IDS= 	Y}', "");
Expect(0, 195101, '\p{^IDS= 	Y}', "");
Expect(0, 195101, '\P{IDS= 	Y}', "");
Expect(1, 195101, '\P{^IDS= 	Y}', "");
Expect(0, 195102, '\p{IDS= 	Y}', "");
Expect(1, 195102, '\p{^IDS= 	Y}', "");
Expect(1, 195102, '\P{IDS= 	Y}', "");
Expect(0, 195102, '\P{^IDS= 	Y}', "");
Error('\p{Is_ID_Start=_T:=}');
Error('\P{Is_ID_Start=_T:=}');
Expect(1, 195101, '\p{Is_ID_Start=t}', "");
Expect(0, 195101, '\p{^Is_ID_Start=t}', "");
Expect(0, 195101, '\P{Is_ID_Start=t}', "");
Expect(1, 195101, '\P{^Is_ID_Start=t}', "");
Expect(0, 195102, '\p{Is_ID_Start=t}', "");
Expect(1, 195102, '\p{^Is_ID_Start=t}', "");
Expect(1, 195102, '\P{Is_ID_Start=t}', "");
Expect(0, 195102, '\P{^Is_ID_Start=t}', "");
Expect(1, 195101, '\p{Is_ID_Start=--t}', "");
Expect(0, 195101, '\p{^Is_ID_Start=--t}', "");
Expect(0, 195101, '\P{Is_ID_Start=--t}', "");
Expect(1, 195101, '\P{^Is_ID_Start=--t}', "");
Expect(0, 195102, '\p{Is_ID_Start=--t}', "");
Expect(1, 195102, '\p{^Is_ID_Start=--t}', "");
Expect(1, 195102, '\P{Is_ID_Start=--t}', "");
Expect(0, 195102, '\P{^Is_ID_Start=--t}', "");
Error('\p{Is_IDS=-_True/a/}');
Error('\P{Is_IDS=-_True/a/}');
Expect(1, 195101, '\p{Is_IDS=true}', "");
Expect(0, 195101, '\p{^Is_IDS=true}', "");
Expect(0, 195101, '\P{Is_IDS=true}', "");
Expect(1, 195101, '\P{^Is_IDS=true}', "");
Expect(0, 195102, '\p{Is_IDS=true}', "");
Expect(1, 195102, '\p{^Is_IDS=true}', "");
Expect(1, 195102, '\P{Is_IDS=true}', "");
Expect(0, 195102, '\P{^Is_IDS=true}', "");
Expect(1, 195101, '\p{Is_IDS=	 true}', "");
Expect(0, 195101, '\p{^Is_IDS=	 true}', "");
Expect(0, 195101, '\P{Is_IDS=	 true}', "");
Expect(1, 195101, '\P{^Is_IDS=	 true}', "");
Expect(0, 195102, '\p{Is_IDS=	 true}', "");
Expect(1, 195102, '\p{^Is_IDS=	 true}', "");
Expect(1, 195102, '\P{Is_IDS=	 true}', "");
Expect(0, 195102, '\P{^Is_IDS=	 true}', "");
Error('\p{IDS_Binary_Operator=/a/ NO}');
Error('\P{IDS_Binary_Operator=/a/ NO}');
Expect(1, 12284, '\p{IDS_Binary_Operator=no}', "");
Expect(0, 12284, '\p{^IDS_Binary_Operator=no}', "");
Expect(0, 12284, '\P{IDS_Binary_Operator=no}', "");
Expect(1, 12284, '\P{^IDS_Binary_Operator=no}', "");
Expect(0, 12283, '\p{IDS_Binary_Operator=no}', "");
Expect(1, 12283, '\p{^IDS_Binary_Operator=no}', "");
Expect(1, 12283, '\P{IDS_Binary_Operator=no}', "");
Expect(0, 12283, '\P{^IDS_Binary_Operator=no}', "");
Expect(1, 12284, '\p{IDS_Binary_Operator= _no}', "");
Expect(0, 12284, '\p{^IDS_Binary_Operator= _no}', "");
Expect(0, 12284, '\P{IDS_Binary_Operator= _no}', "");
Expect(1, 12284, '\P{^IDS_Binary_Operator= _no}', "");
Expect(0, 12283, '\p{IDS_Binary_Operator= _no}', "");
Expect(1, 12283, '\p{^IDS_Binary_Operator= _no}', "");
Expect(1, 12283, '\P{IDS_Binary_Operator= _no}', "");
Expect(0, 12283, '\P{^IDS_Binary_Operator= _no}', "");
Error('\p{IDSB= :=n}');
Error('\P{IDSB= :=n}');
Expect(1, 12284, '\p{IDSB=n}', "");
Expect(0, 12284, '\p{^IDSB=n}', "");
Expect(0, 12284, '\P{IDSB=n}', "");
Expect(1, 12284, '\P{^IDSB=n}', "");
Expect(0, 12283, '\p{IDSB=n}', "");
Expect(1, 12283, '\p{^IDSB=n}', "");
Expect(1, 12283, '\P{IDSB=n}', "");
Expect(0, 12283, '\P{^IDSB=n}', "");
Expect(1, 12284, '\p{IDSB= N}', "");
Expect(0, 12284, '\p{^IDSB= N}', "");
Expect(0, 12284, '\P{IDSB= N}', "");
Expect(1, 12284, '\P{^IDSB= N}', "");
Expect(0, 12283, '\p{IDSB= N}', "");
Expect(1, 12283, '\p{^IDSB= N}', "");
Expect(1, 12283, '\P{IDSB= N}', "");
Expect(0, 12283, '\P{^IDSB= N}', "");
Error('\p{Is_IDS_Binary_Operator=/a/f}');
Error('\P{Is_IDS_Binary_Operator=/a/f}');
Expect(1, 12284, '\p{Is_IDS_Binary_Operator=f}', "");
Expect(0, 12284, '\p{^Is_IDS_Binary_Operator=f}', "");
Expect(0, 12284, '\P{Is_IDS_Binary_Operator=f}', "");
Expect(1, 12284, '\P{^Is_IDS_Binary_Operator=f}', "");
Expect(0, 12283, '\p{Is_IDS_Binary_Operator=f}', "");
Expect(1, 12283, '\p{^Is_IDS_Binary_Operator=f}', "");
Expect(1, 12283, '\P{Is_IDS_Binary_Operator=f}', "");
Expect(0, 12283, '\P{^Is_IDS_Binary_Operator=f}', "");
Expect(1, 12284, '\p{Is_IDS_Binary_Operator=__F}', "");
Expect(0, 12284, '\p{^Is_IDS_Binary_Operator=__F}', "");
Expect(0, 12284, '\P{Is_IDS_Binary_Operator=__F}', "");
Expect(1, 12284, '\P{^Is_IDS_Binary_Operator=__F}', "");
Expect(0, 12283, '\p{Is_IDS_Binary_Operator=__F}', "");
Expect(1, 12283, '\p{^Is_IDS_Binary_Operator=__F}', "");
Expect(1, 12283, '\P{Is_IDS_Binary_Operator=__F}', "");
Expect(0, 12283, '\P{^Is_IDS_Binary_Operator=__F}', "");
Error('\p{Is_IDSB: /a/_-False}');
Error('\P{Is_IDSB: /a/_-False}');
Expect(1, 12284, '\p{Is_IDSB=false}', "");
Expect(0, 12284, '\p{^Is_IDSB=false}', "");
Expect(0, 12284, '\P{Is_IDSB=false}', "");
Expect(1, 12284, '\P{^Is_IDSB=false}', "");
Expect(0, 12283, '\p{Is_IDSB=false}', "");
Expect(1, 12283, '\p{^Is_IDSB=false}', "");
Expect(1, 12283, '\P{Is_IDSB=false}', "");
Expect(0, 12283, '\P{^Is_IDSB=false}', "");
Expect(1, 12284, '\p{Is_IDSB=FALSE}', "");
Expect(0, 12284, '\p{^Is_IDSB=FALSE}', "");
Expect(0, 12284, '\P{Is_IDSB=FALSE}', "");
Expect(1, 12284, '\P{^Is_IDSB=FALSE}', "");
Expect(0, 12283, '\p{Is_IDSB=FALSE}', "");
Expect(1, 12283, '\p{^Is_IDSB=FALSE}', "");
Expect(1, 12283, '\P{Is_IDSB=FALSE}', "");
Expect(0, 12283, '\P{^Is_IDSB=FALSE}', "");
Error('\p{IDS_Binary_Operator= /a/Yes}');
Error('\P{IDS_Binary_Operator= /a/Yes}');
Expect(1, 12283, '\p{IDS_Binary_Operator=yes}', "");
Expect(0, 12283, '\p{^IDS_Binary_Operator=yes}', "");
Expect(0, 12283, '\P{IDS_Binary_Operator=yes}', "");
Expect(1, 12283, '\P{^IDS_Binary_Operator=yes}', "");
Expect(0, 12284, '\p{IDS_Binary_Operator=yes}', "");
Expect(1, 12284, '\p{^IDS_Binary_Operator=yes}', "");
Expect(1, 12284, '\P{IDS_Binary_Operator=yes}', "");
Expect(0, 12284, '\P{^IDS_Binary_Operator=yes}', "");
Expect(1, 12283, '\p{IDS_Binary_Operator=	 Yes}', "");
Expect(0, 12283, '\p{^IDS_Binary_Operator=	 Yes}', "");
Expect(0, 12283, '\P{IDS_Binary_Operator=	 Yes}', "");
Expect(1, 12283, '\P{^IDS_Binary_Operator=	 Yes}', "");
Expect(0, 12284, '\p{IDS_Binary_Operator=	 Yes}', "");
Expect(1, 12284, '\p{^IDS_Binary_Operator=	 Yes}', "");
Expect(1, 12284, '\P{IDS_Binary_Operator=	 Yes}', "");
Expect(0, 12284, '\P{^IDS_Binary_Operator=	 Yes}', "");
Error('\p{IDSB=	y/a/}');
Error('\P{IDSB=	y/a/}');
Expect(1, 12283, '\p{IDSB=y}', "");
Expect(0, 12283, '\p{^IDSB=y}', "");
Expect(0, 12283, '\P{IDSB=y}', "");
Expect(1, 12283, '\P{^IDSB=y}', "");
Expect(0, 12284, '\p{IDSB=y}', "");
Expect(1, 12284, '\p{^IDSB=y}', "");
Expect(1, 12284, '\P{IDSB=y}', "");
Expect(0, 12284, '\P{^IDSB=y}', "");
Expect(1, 12283, '\p{IDSB=__y}', "");
Expect(0, 12283, '\p{^IDSB=__y}', "");
Expect(0, 12283, '\P{IDSB=__y}', "");
Expect(1, 12283, '\P{^IDSB=__y}', "");
Expect(0, 12284, '\p{IDSB=__y}', "");
Expect(1, 12284, '\p{^IDSB=__y}', "");
Expect(1, 12284, '\P{IDSB=__y}', "");
Expect(0, 12284, '\P{^IDSB=__y}', "");
Error('\p{Is_IDS_Binary_Operator=/a/t}');
Error('\P{Is_IDS_Binary_Operator=/a/t}');
Expect(1, 12283, '\p{Is_IDS_Binary_Operator:   t}', "");
Expect(0, 12283, '\p{^Is_IDS_Binary_Operator:   t}', "");
Expect(0, 12283, '\P{Is_IDS_Binary_Operator:   t}', "");
Expect(1, 12283, '\P{^Is_IDS_Binary_Operator:   t}', "");
Expect(0, 12284, '\p{Is_IDS_Binary_Operator:   t}', "");
Expect(1, 12284, '\p{^Is_IDS_Binary_Operator:   t}', "");
Expect(1, 12284, '\P{Is_IDS_Binary_Operator:   t}', "");
Expect(0, 12284, '\P{^Is_IDS_Binary_Operator:   t}', "");
Expect(1, 12283, '\p{Is_IDS_Binary_Operator= 	t}', "");
Expect(0, 12283, '\p{^Is_IDS_Binary_Operator= 	t}', "");
Expect(0, 12283, '\P{Is_IDS_Binary_Operator= 	t}', "");
Expect(1, 12283, '\P{^Is_IDS_Binary_Operator= 	t}', "");
Expect(0, 12284, '\p{Is_IDS_Binary_Operator= 	t}', "");
Expect(1, 12284, '\p{^Is_IDS_Binary_Operator= 	t}', "");
Expect(1, 12284, '\P{Is_IDS_Binary_Operator= 	t}', "");
Expect(0, 12284, '\P{^Is_IDS_Binary_Operator= 	t}', "");
Error('\p{Is_IDSB= true:=}');
Error('\P{Is_IDSB= true:=}');
Expect(1, 12283, '\p{Is_IDSB=true}', "");
Expect(0, 12283, '\p{^Is_IDSB=true}', "");
Expect(0, 12283, '\P{Is_IDSB=true}', "");
Expect(1, 12283, '\P{^Is_IDSB=true}', "");
Expect(0, 12284, '\p{Is_IDSB=true}', "");
Expect(1, 12284, '\p{^Is_IDSB=true}', "");
Expect(1, 12284, '\P{Is_IDSB=true}', "");
Expect(0, 12284, '\P{^Is_IDSB=true}', "");
Expect(1, 12283, '\p{Is_IDSB=_True}', "");
Expect(0, 12283, '\p{^Is_IDSB=_True}', "");
Expect(0, 12283, '\P{Is_IDSB=_True}', "");
Expect(1, 12283, '\P{^Is_IDSB=_True}', "");
Expect(0, 12284, '\p{Is_IDSB=_True}', "");
Expect(1, 12284, '\p{^Is_IDSB=_True}', "");
Expect(1, 12284, '\P{Is_IDSB=_True}', "");
Expect(0, 12284, '\P{^Is_IDSB=_True}', "");
Error('\p{IDS_Trinary_Operator:/a/- No}');
Error('\P{IDS_Trinary_Operator:/a/- No}');
Expect(1, 12276, '\p{IDS_Trinary_Operator=no}', "");
Expect(0, 12276, '\p{^IDS_Trinary_Operator=no}', "");
Expect(0, 12276, '\P{IDS_Trinary_Operator=no}', "");
Expect(1, 12276, '\P{^IDS_Trinary_Operator=no}', "");
Expect(0, 12275, '\p{IDS_Trinary_Operator=no}', "");
Expect(1, 12275, '\p{^IDS_Trinary_Operator=no}', "");
Expect(1, 12275, '\P{IDS_Trinary_Operator=no}', "");
Expect(0, 12275, '\P{^IDS_Trinary_Operator=no}', "");
Expect(1, 12276, '\p{IDS_Trinary_Operator=-	no}', "");
Expect(0, 12276, '\p{^IDS_Trinary_Operator=-	no}', "");
Expect(0, 12276, '\P{IDS_Trinary_Operator=-	no}', "");
Expect(1, 12276, '\P{^IDS_Trinary_Operator=-	no}', "");
Expect(0, 12275, '\p{IDS_Trinary_Operator=-	no}', "");
Expect(1, 12275, '\p{^IDS_Trinary_Operator=-	no}', "");
Expect(1, 12275, '\P{IDS_Trinary_Operator=-	no}', "");
Expect(0, 12275, '\P{^IDS_Trinary_Operator=-	no}', "");
Error('\p{IDST=/a/	N}');
Error('\P{IDST=/a/	N}');
Expect(1, 12276, '\p{IDST=n}', "");
Expect(0, 12276, '\p{^IDST=n}', "");
Expect(0, 12276, '\P{IDST=n}', "");
Expect(1, 12276, '\P{^IDST=n}', "");
Expect(0, 12275, '\p{IDST=n}', "");
Expect(1, 12275, '\p{^IDST=n}', "");
Expect(1, 12275, '\P{IDST=n}', "");
Expect(0, 12275, '\P{^IDST=n}', "");
Expect(1, 12276, '\p{IDST= n}', "");
Expect(0, 12276, '\p{^IDST= n}', "");
Expect(0, 12276, '\P{IDST= n}', "");
Expect(1, 12276, '\P{^IDST= n}', "");
Expect(0, 12275, '\p{IDST= n}', "");
Expect(1, 12275, '\p{^IDST= n}', "");
Expect(1, 12275, '\P{IDST= n}', "");
Expect(0, 12275, '\P{^IDST= n}', "");
Error('\p{Is_IDS_Trinary_Operator=/a/_-F}');
Error('\P{Is_IDS_Trinary_Operator=/a/_-F}');
Expect(1, 12276, '\p{Is_IDS_Trinary_Operator=f}', "");
Expect(0, 12276, '\p{^Is_IDS_Trinary_Operator=f}', "");
Expect(0, 12276, '\P{Is_IDS_Trinary_Operator=f}', "");
Expect(1, 12276, '\P{^Is_IDS_Trinary_Operator=f}', "");
Expect(0, 12275, '\p{Is_IDS_Trinary_Operator=f}', "");
Expect(1, 12275, '\p{^Is_IDS_Trinary_Operator=f}', "");
Expect(1, 12275, '\P{Is_IDS_Trinary_Operator=f}', "");
Expect(0, 12275, '\P{^Is_IDS_Trinary_Operator=f}', "");
Expect(1, 12276, '\p{Is_IDS_Trinary_Operator=_ F}', "");
Expect(0, 12276, '\p{^Is_IDS_Trinary_Operator=_ F}', "");
Expect(0, 12276, '\P{Is_IDS_Trinary_Operator=_ F}', "");
Expect(1, 12276, '\P{^Is_IDS_Trinary_Operator=_ F}', "");
Expect(0, 12275, '\p{Is_IDS_Trinary_Operator=_ F}', "");
Expect(1, 12275, '\p{^Is_IDS_Trinary_Operator=_ F}', "");
Expect(1, 12275, '\P{Is_IDS_Trinary_Operator=_ F}', "");
Expect(0, 12275, '\P{^Is_IDS_Trinary_Operator=_ F}', "");
Error('\p{Is_IDST=_/a/False}');
Error('\P{Is_IDST=_/a/False}');
Expect(1, 12276, '\p{Is_IDST=false}', "");
Expect(0, 12276, '\p{^Is_IDST=false}', "");
Expect(0, 12276, '\P{Is_IDST=false}', "");
Expect(1, 12276, '\P{^Is_IDST=false}', "");
Expect(0, 12275, '\p{Is_IDST=false}', "");
Expect(1, 12275, '\p{^Is_IDST=false}', "");
Expect(1, 12275, '\P{Is_IDST=false}', "");
Expect(0, 12275, '\P{^Is_IDST=false}', "");
Expect(1, 12276, '\p{Is_IDST= _False}', "");
Expect(0, 12276, '\p{^Is_IDST= _False}', "");
Expect(0, 12276, '\P{Is_IDST= _False}', "");
Expect(1, 12276, '\P{^Is_IDST= _False}', "");
Expect(0, 12275, '\p{Is_IDST= _False}', "");
Expect(1, 12275, '\p{^Is_IDST= _False}', "");
Expect(1, 12275, '\P{Is_IDST= _False}', "");
Expect(0, 12275, '\P{^Is_IDST= _False}', "");
Error('\p{IDS_Trinary_Operator:	 yes/a/}');
Error('\P{IDS_Trinary_Operator:	 yes/a/}');
Expect(1, 12275, '\p{IDS_Trinary_Operator=yes}', "");
Expect(0, 12275, '\p{^IDS_Trinary_Operator=yes}', "");
Expect(0, 12275, '\P{IDS_Trinary_Operator=yes}', "");
Expect(1, 12275, '\P{^IDS_Trinary_Operator=yes}', "");
Expect(0, 12276, '\p{IDS_Trinary_Operator=yes}', "");
Expect(1, 12276, '\p{^IDS_Trinary_Operator=yes}', "");
Expect(1, 12276, '\P{IDS_Trinary_Operator=yes}', "");
Expect(0, 12276, '\P{^IDS_Trinary_Operator=yes}', "");
Expect(1, 12275, '\p{IDS_Trinary_Operator:   --yes}', "");
Expect(0, 12275, '\p{^IDS_Trinary_Operator:   --yes}', "");
Expect(0, 12275, '\P{IDS_Trinary_Operator:   --yes}', "");
Expect(1, 12275, '\P{^IDS_Trinary_Operator:   --yes}', "");
Expect(0, 12276, '\p{IDS_Trinary_Operator:   --yes}', "");
Expect(1, 12276, '\p{^IDS_Trinary_Operator:   --yes}', "");
Expect(1, 12276, '\P{IDS_Trinary_Operator:   --yes}', "");
Expect(0, 12276, '\P{^IDS_Trinary_Operator:   --yes}', "");
Error('\p{IDST=:=-y}');
Error('\P{IDST=:=-y}');
Expect(1, 12275, '\p{IDST=y}', "");
Expect(0, 12275, '\p{^IDST=y}', "");
Expect(0, 12275, '\P{IDST=y}', "");
Expect(1, 12275, '\P{^IDST=y}', "");
Expect(0, 12276, '\p{IDST=y}', "");
Expect(1, 12276, '\p{^IDST=y}', "");
Expect(1, 12276, '\P{IDST=y}', "");
Expect(0, 12276, '\P{^IDST=y}', "");
Expect(1, 12275, '\p{IDST= Y}', "");
Expect(0, 12275, '\p{^IDST= Y}', "");
Expect(0, 12275, '\P{IDST= Y}', "");
Expect(1, 12275, '\P{^IDST= Y}', "");
Expect(0, 12276, '\p{IDST= Y}', "");
Expect(1, 12276, '\p{^IDST= Y}', "");
Expect(1, 12276, '\P{IDST= Y}', "");
Expect(0, 12276, '\P{^IDST= Y}', "");
Error('\p{Is_IDS_Trinary_Operator=/a/T}');
Error('\P{Is_IDS_Trinary_Operator=/a/T}');
Expect(1, 12275, '\p{Is_IDS_Trinary_Operator:   t}', "");
Expect(0, 12275, '\p{^Is_IDS_Trinary_Operator:   t}', "");
Expect(0, 12275, '\P{Is_IDS_Trinary_Operator:   t}', "");
Expect(1, 12275, '\P{^Is_IDS_Trinary_Operator:   t}', "");
Expect(0, 12276, '\p{Is_IDS_Trinary_Operator:   t}', "");
Expect(1, 12276, '\p{^Is_IDS_Trinary_Operator:   t}', "");
Expect(1, 12276, '\P{Is_IDS_Trinary_Operator:   t}', "");
Expect(0, 12276, '\P{^Is_IDS_Trinary_Operator:   t}', "");
Expect(1, 12275, '\p{Is_IDS_Trinary_Operator= T}', "");
Expect(0, 12275, '\p{^Is_IDS_Trinary_Operator= T}', "");
Expect(0, 12275, '\P{Is_IDS_Trinary_Operator= T}', "");
Expect(1, 12275, '\P{^Is_IDS_Trinary_Operator= T}', "");
Expect(0, 12276, '\p{Is_IDS_Trinary_Operator= T}', "");
Expect(1, 12276, '\p{^Is_IDS_Trinary_Operator= T}', "");
Expect(1, 12276, '\P{Is_IDS_Trinary_Operator= T}', "");
Expect(0, 12276, '\P{^Is_IDS_Trinary_Operator= T}', "");
Error('\p{Is_IDST=-true:=}');
Error('\P{Is_IDST=-true:=}');
Expect(1, 12275, '\p{Is_IDST=true}', "");
Expect(0, 12275, '\p{^Is_IDST=true}', "");
Expect(0, 12275, '\P{Is_IDST=true}', "");
Expect(1, 12275, '\P{^Is_IDST=true}', "");
Expect(0, 12276, '\p{Is_IDST=true}', "");
Expect(1, 12276, '\p{^Is_IDST=true}', "");
Expect(1, 12276, '\P{Is_IDST=true}', "");
Expect(0, 12276, '\P{^Is_IDST=true}', "");
Expect(1, 12275, '\p{Is_IDST=_	True}', "");
Expect(0, 12275, '\p{^Is_IDST=_	True}', "");
Expect(0, 12275, '\P{Is_IDST=_	True}', "");
Expect(1, 12275, '\P{^Is_IDST=_	True}', "");
Expect(0, 12276, '\p{Is_IDST=_	True}', "");
Expect(1, 12276, '\p{^Is_IDST=_	True}', "");
Expect(1, 12276, '\P{Is_IDST=_	True}', "");
Expect(0, 12276, '\P{^Is_IDST=_	True}', "");
Error('\p{Ideographic=:=_-no}');
Error('\P{Ideographic=:=_-no}');
Expect(1, 195102, '\p{Ideographic=no}', "");
Expect(0, 195102, '\p{^Ideographic=no}', "");
Expect(0, 195102, '\P{Ideographic=no}', "");
Expect(1, 195102, '\P{^Ideographic=no}', "");
Expect(0, 195101, '\p{Ideographic=no}', "");
Expect(1, 195101, '\p{^Ideographic=no}', "");
Expect(1, 195101, '\P{Ideographic=no}', "");
Expect(0, 195101, '\P{^Ideographic=no}', "");
Expect(1, 195102, '\p{Ideographic=__No}', "");
Expect(0, 195102, '\p{^Ideographic=__No}', "");
Expect(0, 195102, '\P{Ideographic=__No}', "");
Expect(1, 195102, '\P{^Ideographic=__No}', "");
Expect(0, 195101, '\p{Ideographic=__No}', "");
Expect(1, 195101, '\p{^Ideographic=__No}', "");
Expect(1, 195101, '\P{Ideographic=__No}', "");
Expect(0, 195101, '\P{^Ideographic=__No}', "");
Error('\p{Ideo=_N:=}');
Error('\P{Ideo=_N:=}');
Expect(1, 195102, '\p{Ideo=n}', "");
Expect(0, 195102, '\p{^Ideo=n}', "");
Expect(0, 195102, '\P{Ideo=n}', "");
Expect(1, 195102, '\P{^Ideo=n}', "");
Expect(0, 195101, '\p{Ideo=n}', "");
Expect(1, 195101, '\p{^Ideo=n}', "");
Expect(1, 195101, '\P{Ideo=n}', "");
Expect(0, 195101, '\P{^Ideo=n}', "");
Expect(1, 195102, '\p{Ideo=	N}', "");
Expect(0, 195102, '\p{^Ideo=	N}', "");
Expect(0, 195102, '\P{Ideo=	N}', "");
Expect(1, 195102, '\P{^Ideo=	N}', "");
Expect(0, 195101, '\p{Ideo=	N}', "");
Expect(1, 195101, '\p{^Ideo=	N}', "");
Expect(1, 195101, '\P{Ideo=	N}', "");
Expect(0, 195101, '\P{^Ideo=	N}', "");
Error('\p{Is_Ideographic=/a/F}');
Error('\P{Is_Ideographic=/a/F}');
Expect(1, 195102, '\p{Is_Ideographic=f}', "");
Expect(0, 195102, '\p{^Is_Ideographic=f}', "");
Expect(0, 195102, '\P{Is_Ideographic=f}', "");
Expect(1, 195102, '\P{^Is_Ideographic=f}', "");
Expect(0, 195101, '\p{Is_Ideographic=f}', "");
Expect(1, 195101, '\p{^Is_Ideographic=f}', "");
Expect(1, 195101, '\P{Is_Ideographic=f}', "");
Expect(0, 195101, '\P{^Is_Ideographic=f}', "");
Expect(1, 195102, '\p{Is_Ideographic=  F}', "");
Expect(0, 195102, '\p{^Is_Ideographic=  F}', "");
Expect(0, 195102, '\P{Is_Ideographic=  F}', "");
Expect(1, 195102, '\P{^Is_Ideographic=  F}', "");
Expect(0, 195101, '\p{Is_Ideographic=  F}', "");
Expect(1, 195101, '\p{^Is_Ideographic=  F}', "");
Expect(1, 195101, '\P{Is_Ideographic=  F}', "");
Expect(0, 195101, '\P{^Is_Ideographic=  F}', "");
Error('\p{Is_Ideo= /a/False}');
Error('\P{Is_Ideo= /a/False}');
Expect(1, 195102, '\p{Is_Ideo:   false}', "");
Expect(0, 195102, '\p{^Is_Ideo:   false}', "");
Expect(0, 195102, '\P{Is_Ideo:   false}', "");
Expect(1, 195102, '\P{^Is_Ideo:   false}', "");
Expect(0, 195101, '\p{Is_Ideo:   false}', "");
Expect(1, 195101, '\p{^Is_Ideo:   false}', "");
Expect(1, 195101, '\P{Is_Ideo:   false}', "");
Expect(0, 195101, '\P{^Is_Ideo:   false}', "");
Expect(1, 195102, '\p{Is_Ideo=  False}', "");
Expect(0, 195102, '\p{^Is_Ideo=  False}', "");
Expect(0, 195102, '\P{Is_Ideo=  False}', "");
Expect(1, 195102, '\P{^Is_Ideo=  False}', "");
Expect(0, 195101, '\p{Is_Ideo=  False}', "");
Expect(1, 195101, '\p{^Is_Ideo=  False}', "");
Expect(1, 195101, '\P{Is_Ideo=  False}', "");
Expect(0, 195101, '\P{^Is_Ideo=  False}', "");
Error('\p{Ideographic=	_Yes:=}');
Error('\P{Ideographic=	_Yes:=}');
Expect(1, 195101, '\p{Ideographic=yes}', "");
Expect(0, 195101, '\p{^Ideographic=yes}', "");
Expect(0, 195101, '\P{Ideographic=yes}', "");
Expect(1, 195101, '\P{^Ideographic=yes}', "");
Expect(0, 195102, '\p{Ideographic=yes}', "");
Expect(1, 195102, '\p{^Ideographic=yes}', "");
Expect(1, 195102, '\P{Ideographic=yes}', "");
Expect(0, 195102, '\P{^Ideographic=yes}', "");
Expect(1, 195101, '\p{Ideographic= _yes}', "");
Expect(0, 195101, '\p{^Ideographic= _yes}', "");
Expect(0, 195101, '\P{Ideographic= _yes}', "");
Expect(1, 195101, '\P{^Ideographic= _yes}', "");
Expect(0, 195102, '\p{Ideographic= _yes}', "");
Expect(1, 195102, '\p{^Ideographic= _yes}', "");
Expect(1, 195102, '\P{Ideographic= _yes}', "");
Expect(0, 195102, '\P{^Ideographic= _yes}', "");
Error('\p{Ideo=	/a/Y}');
Error('\P{Ideo=	/a/Y}');
Expect(1, 195101, '\p{Ideo=y}', "");
Expect(0, 195101, '\p{^Ideo=y}', "");
Expect(0, 195101, '\P{Ideo=y}', "");
Expect(1, 195101, '\P{^Ideo=y}', "");
Expect(0, 195102, '\p{Ideo=y}', "");
Expect(1, 195102, '\p{^Ideo=y}', "");
Expect(1, 195102, '\P{Ideo=y}', "");
Expect(0, 195102, '\P{^Ideo=y}', "");
Expect(1, 195101, '\p{Ideo=_-y}', "");
Expect(0, 195101, '\p{^Ideo=_-y}', "");
Expect(0, 195101, '\P{Ideo=_-y}', "");
Expect(1, 195101, '\P{^Ideo=_-y}', "");
Expect(0, 195102, '\p{Ideo=_-y}', "");
Expect(1, 195102, '\p{^Ideo=_-y}', "");
Expect(1, 195102, '\P{Ideo=_-y}', "");
Expect(0, 195102, '\P{^Ideo=_-y}', "");
Error('\p{Is_Ideographic= :=T}');
Error('\P{Is_Ideographic= :=T}');
Expect(1, 195101, '\p{Is_Ideographic=t}', "");
Expect(0, 195101, '\p{^Is_Ideographic=t}', "");
Expect(0, 195101, '\P{Is_Ideographic=t}', "");
Expect(1, 195101, '\P{^Is_Ideographic=t}', "");
Expect(0, 195102, '\p{Is_Ideographic=t}', "");
Expect(1, 195102, '\p{^Is_Ideographic=t}', "");
Expect(1, 195102, '\P{Is_Ideographic=t}', "");
Expect(0, 195102, '\P{^Is_Ideographic=t}', "");
Expect(1, 195101, '\p{Is_Ideographic=	 T}', "");
Expect(0, 195101, '\p{^Is_Ideographic=	 T}', "");
Expect(0, 195101, '\P{Is_Ideographic=	 T}', "");
Expect(1, 195101, '\P{^Is_Ideographic=	 T}', "");
Expect(0, 195102, '\p{Is_Ideographic=	 T}', "");
Expect(1, 195102, '\p{^Is_Ideographic=	 T}', "");
Expect(1, 195102, '\P{Is_Ideographic=	 T}', "");
Expect(0, 195102, '\P{^Is_Ideographic=	 T}', "");
Error('\p{Is_Ideo=/a/_True}');
Error('\P{Is_Ideo=/a/_True}');
Expect(1, 195101, '\p{Is_Ideo=true}', "");
Expect(0, 195101, '\p{^Is_Ideo=true}', "");
Expect(0, 195101, '\P{Is_Ideo=true}', "");
Expect(1, 195101, '\P{^Is_Ideo=true}', "");
Expect(0, 195102, '\p{Is_Ideo=true}', "");
Expect(1, 195102, '\p{^Is_Ideo=true}', "");
Expect(1, 195102, '\P{Is_Ideo=true}', "");
Expect(0, 195102, '\P{^Is_Ideo=true}', "");
Expect(1, 195101, '\p{Is_Ideo=  true}', "");
Expect(0, 195101, '\p{^Is_Ideo=  true}', "");
Expect(0, 195101, '\P{Is_Ideo=  true}', "");
Expect(1, 195101, '\P{^Is_Ideo=  true}', "");
Expect(0, 195102, '\p{Is_Ideo=  true}', "");
Expect(1, 195102, '\p{^Is_Ideo=  true}', "");
Expect(1, 195102, '\P{Is_Ideo=  true}', "");
Expect(0, 195102, '\P{^Is_Ideo=  true}', "");
Error('\p{Present_In=- +0_0_0_0_0_0_001.1/a/}');
Error('\P{Present_In=- +0_0_0_0_0_0_001.1/a/}');
Expect(1, 65533, '\p{Present_In=0000000001.1}', "");
Expect(0, 65533, '\p{^Present_In=0000000001.1}', "");
Expect(0, 65533, '\P{Present_In=0000000001.1}', "");
Expect(1, 65533, '\P{^Present_In=0000000001.1}', "");
Expect(0, 65536, '\p{Present_In=0000000001.1}', "");
Expect(1, 65536, '\p{^Present_In=0000000001.1}', "");
Expect(1, 65536, '\P{Present_In=0000000001.1}', "");
Expect(0, 65536, '\P{^Present_In=0000000001.1}', "");
Error('\p{In=:=-+0_0_01.1}');
Error('\P{In=:=-+0_0_01.1}');
Expect(1, 65533, '\p{In=00000001.1}', "");
Expect(0, 65533, '\p{^In=00000001.1}', "");
Expect(0, 65533, '\P{In=00000001.1}', "");
Expect(1, 65533, '\P{^In=00000001.1}', "");
Expect(0, 65536, '\p{In=00000001.1}', "");
Expect(1, 65536, '\p{^In=00000001.1}', "");
Expect(1, 65536, '\P{In=00000001.1}', "");
Expect(0, 65536, '\P{^In=00000001.1}', "");
Error('\p{Is_Present_In=/a/__000000001.1}');
Error('\P{Is_Present_In=/a/__000000001.1}');
Expect(1, 65533, '\p{Is_Present_In=000000001.1}', "");
Expect(0, 65533, '\p{^Is_Present_In=000000001.1}', "");
Expect(0, 65533, '\P{Is_Present_In=000000001.1}', "");
Expect(1, 65533, '\P{^Is_Present_In=000000001.1}', "");
Expect(0, 65536, '\p{Is_Present_In=000000001.1}', "");
Expect(1, 65536, '\p{^Is_Present_In=000000001.1}', "");
Expect(1, 65536, '\P{Is_Present_In=000000001.1}', "");
Expect(0, 65536, '\P{^Is_Present_In=000000001.1}', "");
Error('\p{Is_In= +00000001.1/a/}');
Error('\P{Is_In= +00000001.1/a/}');
Expect(1, 65533, '\p{Is_In=00_00_1.1}', "");
Expect(0, 65533, '\p{^Is_In=00_00_1.1}', "");
Expect(0, 65533, '\P{Is_In=00_00_1.1}', "");
Expect(1, 65533, '\P{^Is_In=00_00_1.1}', "");
Expect(0, 65536, '\p{Is_In=00_00_1.1}', "");
Expect(1, 65536, '\p{^Is_In=00_00_1.1}', "");
Expect(1, 65536, '\P{Is_In=00_00_1.1}', "");
Expect(0, 65536, '\P{^Is_In=00_00_1.1}', "");
Error('\p{Present_In=_/a/+2.0}');
Error('\P{Present_In=_/a/+2.0}');
Expect(1, 983040, '\p{Present_In:000000_2.0}', "");
Expect(0, 983040, '\p{^Present_In:000000_2.0}', "");
Expect(0, 983040, '\P{Present_In:000000_2.0}', "");
Expect(1, 983040, '\P{^Present_In:000000_2.0}', "");
Expect(0, 983037, '\p{Present_In:000000_2.0}', "");
Expect(1, 983037, '\p{^Present_In:000000_2.0}', "");
Expect(1, 983037, '\P{Present_In:000000_2.0}', "");
Expect(0, 983037, '\P{^Present_In:000000_2.0}', "");
Error('\p{In=:=-_2.0}');
Error('\P{In=:=-_2.0}');
Expect(1, 983040, '\p{In=000000002.0}', "");
Expect(0, 983040, '\p{^In=000000002.0}', "");
Expect(0, 983040, '\P{In=000000002.0}', "");
Expect(1, 983040, '\P{^In=000000002.0}', "");
Expect(0, 983037, '\p{In=000000002.0}', "");
Expect(1, 983037, '\p{^In=000000002.0}', "");
Expect(1, 983037, '\P{In=000000002.0}', "");
Expect(0, 983037, '\P{^In=000000002.0}', "");
Error('\p{Is_Present_In=	/a/0000002.0}');
Error('\P{Is_Present_In=	/a/0000002.0}');
Expect(1, 983040, '\p{Is_Present_In=+000000002.0}', "");
Expect(0, 983040, '\p{^Is_Present_In=+000000002.0}', "");
Expect(0, 983040, '\P{Is_Present_In=+000000002.0}', "");
Expect(1, 983040, '\P{^Is_Present_In=+000000002.0}', "");
Expect(0, 983037, '\p{Is_Present_In=+000000002.0}', "");
Expect(1, 983037, '\p{^Is_Present_In=+000000002.0}', "");
Expect(1, 983037, '\P{Is_Present_In=+000000002.0}', "");
Expect(0, 983037, '\P{^Is_Present_In=+000000002.0}', "");
Error('\p{Is_In=/a/-	00_00_02.0}');
Error('\P{Is_In=/a/-	00_00_02.0}');
Expect(1, 983040, '\p{Is_In=2.0}', "");
Expect(0, 983040, '\p{^Is_In=2.0}', "");
Expect(0, 983040, '\P{Is_In=2.0}', "");
Expect(1, 983040, '\P{^Is_In=2.0}', "");
Expect(0, 983037, '\p{Is_In=2.0}', "");
Expect(1, 983037, '\p{^Is_In=2.0}', "");
Expect(1, 983037, '\P{Is_In=2.0}', "");
Expect(0, 983037, '\P{^Is_In=2.0}', "");
Error('\p{Present_In=-0002.1:=}');
Error('\P{Present_In=-0002.1:=}');
Expect(1, 983040, '\p{Present_In=0_2.1}', "");
Expect(0, 983040, '\p{^Present_In=0_2.1}', "");
Expect(0, 983040, '\P{Present_In=0_2.1}', "");
Expect(1, 983040, '\P{^Present_In=0_2.1}', "");
Expect(0, 983037, '\p{Present_In=0_2.1}', "");
Expect(1, 983037, '\p{^Present_In=0_2.1}', "");
Expect(1, 983037, '\P{Present_In=0_2.1}', "");
Expect(0, 983037, '\P{^Present_In=0_2.1}', "");
Error('\p{In: 00000002.1:=}');
Error('\P{In: 00000002.1:=}');
Expect(1, 983040, '\p{In=00000002.1}', "");
Expect(0, 983040, '\p{^In=00000002.1}', "");
Expect(0, 983040, '\P{In=00000002.1}', "");
Expect(1, 983040, '\P{^In=00000002.1}', "");
Expect(0, 983037, '\p{In=00000002.1}', "");
Expect(1, 983037, '\p{^In=00000002.1}', "");
Expect(1, 983037, '\P{In=00000002.1}', "");
Expect(0, 983037, '\P{^In=00000002.1}', "");
Error('\p{Is_Present_In:	:=_-+002.1}');
Error('\P{Is_Present_In:	:=_-+002.1}');
Expect(1, 983040, '\p{Is_Present_In=+002.1}', "");
Expect(0, 983040, '\p{^Is_Present_In=+002.1}', "");
Expect(0, 983040, '\P{Is_Present_In=+002.1}', "");
Expect(1, 983040, '\P{^Is_Present_In=+002.1}', "");
Expect(0, 983037, '\p{Is_Present_In=+002.1}', "");
Expect(1, 983037, '\p{^Is_Present_In=+002.1}', "");
Expect(1, 983037, '\P{Is_Present_In=+002.1}', "");
Expect(0, 983037, '\P{^Is_Present_In=+002.1}', "");
Error('\p{Is_In=__+000_000_000_2.1/a/}');
Error('\P{Is_In=__+000_000_000_2.1/a/}');
Expect(1, 983040, '\p{Is_In=+00002.1}', "");
Expect(0, 983040, '\p{^Is_In=+00002.1}', "");
Expect(0, 983040, '\P{Is_In=+00002.1}', "");
Expect(1, 983040, '\P{^Is_In=+00002.1}', "");
Expect(0, 983037, '\p{Is_In=+00002.1}', "");
Expect(1, 983037, '\p{^Is_In=+00002.1}', "");
Expect(1, 983037, '\P{Is_In=+00002.1}', "");
Expect(0, 983037, '\P{^Is_In=+00002.1}', "");
Error('\p{Present_In=-:=+00_3.0}');
Error('\P{Present_In=-:=+00_3.0}');
Expect(1, 983040, '\p{Present_In=0000003.0}', "");
Expect(0, 983040, '\p{^Present_In=0000003.0}', "");
Expect(0, 983040, '\P{Present_In=0000003.0}', "");
Expect(1, 983040, '\P{^Present_In=0000003.0}', "");
Expect(0, 983037, '\p{Present_In=0000003.0}', "");
Expect(1, 983037, '\p{^Present_In=0000003.0}', "");
Expect(1, 983037, '\P{Present_In=0000003.0}', "");
Expect(0, 983037, '\P{^Present_In=0000003.0}', "");
Error('\p{In= /a/03.0}');
Error('\P{In= /a/03.0}');
Expect(1, 983040, '\p{In=0000003.0}', "");
Expect(0, 983040, '\p{^In=0000003.0}', "");
Expect(0, 983040, '\P{In=0000003.0}', "");
Expect(1, 983040, '\P{^In=0000003.0}', "");
Expect(0, 983037, '\p{In=0000003.0}', "");
Expect(1, 983037, '\p{^In=0000003.0}', "");
Expect(1, 983037, '\P{In=0000003.0}', "");
Expect(0, 983037, '\P{^In=0000003.0}', "");
Error('\p{Is_Present_In=	/a/0_0_03.0}');
Error('\P{Is_Present_In=	/a/0_0_03.0}');
Expect(1, 983040, '\p{Is_Present_In=00003.0}', "");
Expect(0, 983040, '\p{^Is_Present_In=00003.0}', "");
Expect(0, 983040, '\P{Is_Present_In=00003.0}', "");
Expect(1, 983040, '\P{^Is_Present_In=00003.0}', "");
Expect(0, 983037, '\p{Is_Present_In=00003.0}', "");
Expect(1, 983037, '\p{^Is_Present_In=00003.0}', "");
Expect(1, 983037, '\P{Is_Present_In=00003.0}', "");
Expect(0, 983037, '\P{^Is_Present_In=00003.0}', "");
Error('\p{Is_In=	:=003.0}');
Error('\P{Is_In=	:=003.0}');
Expect(1, 983040, '\p{Is_In=00003.0}', "");
Expect(0, 983040, '\p{^Is_In=00003.0}', "");
Expect(0, 983040, '\P{Is_In=00003.0}', "");
Expect(1, 983040, '\P{^Is_In=00003.0}', "");
Expect(0, 983037, '\p{Is_In=00003.0}', "");
Expect(1, 983037, '\p{^Is_In=00003.0}', "");
Expect(1, 983037, '\P{Is_In=00003.0}', "");
Expect(0, 983037, '\P{^Is_In=00003.0}', "");
Error('\p{Present_In:	-	00_00_00_3.1/a/}');
Error('\P{Present_In:	-	00_00_00_3.1/a/}');
Expect(1, 983040, '\p{Present_In=00_3.1}', "");
Expect(0, 983040, '\p{^Present_In=00_3.1}', "");
Expect(0, 983040, '\P{Present_In=00_3.1}', "");
Expect(1, 983040, '\P{^Present_In=00_3.1}', "");
Expect(0, 983037, '\p{Present_In=00_3.1}', "");
Expect(1, 983037, '\p{^Present_In=00_3.1}', "");
Expect(1, 983037, '\P{Present_In=00_3.1}', "");
Expect(0, 983037, '\P{^Present_In=00_3.1}', "");
Error('\p{In=0003.1:=}');
Error('\P{In=0003.1:=}');
Expect(1, 983040, '\p{In=0_0_03.1}', "");
Expect(0, 983040, '\p{^In=0_0_03.1}', "");
Expect(0, 983040, '\P{In=0_0_03.1}', "");
Expect(1, 983040, '\P{^In=0_0_03.1}', "");
Expect(0, 983037, '\p{In=0_0_03.1}', "");
Expect(1, 983037, '\p{^In=0_0_03.1}', "");
Expect(1, 983037, '\P{In=0_0_03.1}', "");
Expect(0, 983037, '\P{^In=0_0_03.1}', "");
Error('\p{Is_Present_In= 	0_3.1:=}');
Error('\P{Is_Present_In= 	0_3.1:=}');
Expect(1, 983040, '\p{Is_Present_In=+00000003.1}', "");
Expect(0, 983040, '\p{^Is_Present_In=+00000003.1}', "");
Expect(0, 983040, '\P{Is_Present_In=+00000003.1}', "");
Expect(1, 983040, '\P{^Is_Present_In=+00000003.1}', "");
Expect(0, 983037, '\p{Is_Present_In=+00000003.1}', "");
Expect(1, 983037, '\p{^Is_Present_In=+00000003.1}', "");
Expect(1, 983037, '\P{Is_Present_In=+00000003.1}', "");
Expect(0, 983037, '\P{^Is_Present_In=+00000003.1}', "");
Error('\p{Is_In=	_0000_3.1/a/}');
Error('\P{Is_In=	_0000_3.1/a/}');
Expect(1, 983040, '\p{Is_In=+00_3.1}', "");
Expect(0, 983040, '\p{^Is_In=+00_3.1}', "");
Expect(0, 983040, '\P{Is_In=+00_3.1}', "");
Expect(1, 983040, '\P{^Is_In=+00_3.1}', "");
Expect(0, 983037, '\p{Is_In=+00_3.1}', "");
Expect(1, 983037, '\p{^Is_In=+00_3.1}', "");
Expect(1, 983037, '\P{Is_In=+00_3.1}', "");
Expect(0, 983037, '\P{^Is_In=+00_3.1}', "");
Error('\p{Present_In= :=+0000003.2}');
Error('\P{Present_In= :=+0000003.2}');
Expect(1, 983040, '\p{Present_In=+0000003.2}', "");
Expect(0, 983040, '\p{^Present_In=+0000003.2}', "");
Expect(0, 983040, '\P{Present_In=+0000003.2}', "");
Expect(1, 983040, '\P{^Present_In=+0000003.2}', "");
Expect(0, 983037, '\p{Present_In=+0000003.2}', "");
Expect(1, 983037, '\p{^Present_In=+0000003.2}', "");
Expect(1, 983037, '\P{Present_In=+0000003.2}', "");
Expect(0, 983037, '\P{^Present_In=+0000003.2}', "");
Error('\p{In=-+0000000003.2/a/}');
Error('\P{In=-+0000000003.2/a/}');
Expect(1, 983040, '\p{In=+00000000_3.2}', "");
Expect(0, 983040, '\p{^In=+00000000_3.2}', "");
Expect(0, 983040, '\P{In=+00000000_3.2}', "");
Expect(1, 983040, '\P{^In=+00000000_3.2}', "");
Expect(0, 983037, '\p{In=+00000000_3.2}', "");
Expect(1, 983037, '\p{^In=+00000000_3.2}', "");
Expect(1, 983037, '\P{In=+00000000_3.2}', "");
Expect(0, 983037, '\P{^In=+00000000_3.2}', "");
Error('\p{Is_Present_In= 	0_0_03.2/a/}');
Error('\P{Is_Present_In= 	0_0_03.2/a/}');
Expect(1, 983040, '\p{Is_Present_In=000000003.2}', "");
Expect(0, 983040, '\p{^Is_Present_In=000000003.2}', "");
Expect(0, 983040, '\P{Is_Present_In=000000003.2}', "");
Expect(1, 983040, '\P{^Is_Present_In=000000003.2}', "");
Expect(0, 983037, '\p{Is_Present_In=000000003.2}', "");
Expect(1, 983037, '\p{^Is_Present_In=000000003.2}', "");
Expect(1, 983037, '\P{Is_Present_In=000000003.2}', "");
Expect(0, 983037, '\P{^Is_Present_In=000000003.2}', "");
Error('\p{Is_In= :=+0_0_0_0_03.2}');
Error('\P{Is_In= :=+0_0_0_0_03.2}');
Expect(1, 983040, '\p{Is_In=+000003.2}', "");
Expect(0, 983040, '\p{^Is_In=+000003.2}', "");
Expect(0, 983040, '\P{Is_In=+000003.2}', "");
Expect(1, 983040, '\P{^Is_In=+000003.2}', "");
Expect(0, 983037, '\p{Is_In=+000003.2}', "");
Expect(1, 983037, '\p{^Is_In=+000003.2}', "");
Expect(1, 983037, '\P{Is_In=+000003.2}', "");
Expect(0, 983037, '\P{^Is_In=+000003.2}', "");
Error('\p{Present_In=	_4.0:=}');
Error('\P{Present_In=	_4.0:=}');
Expect(1, 983040, '\p{Present_In=04.0}', "");
Expect(0, 983040, '\p{^Present_In=04.0}', "");
Expect(0, 983040, '\P{Present_In=04.0}', "");
Expect(1, 983040, '\P{^Present_In=04.0}', "");
Expect(0, 983037, '\p{Present_In=04.0}', "");
Expect(1, 983037, '\p{^Present_In=04.0}', "");
Expect(1, 983037, '\P{Present_In=04.0}', "");
Expect(0, 983037, '\P{^Present_In=04.0}', "");
Error('\p{In= _0000000004.0:=}');
Error('\P{In= _0000000004.0:=}');
Expect(1, 983040, '\p{In: 000_000_4.0}', "");
Expect(0, 983040, '\p{^In: 000_000_4.0}', "");
Expect(0, 983040, '\P{In: 000_000_4.0}', "");
Expect(1, 983040, '\P{^In: 000_000_4.0}', "");
Expect(0, 983037, '\p{In: 000_000_4.0}', "");
Expect(1, 983037, '\p{^In: 000_000_4.0}', "");
Expect(1, 983037, '\P{In: 000_000_4.0}', "");
Expect(0, 983037, '\P{^In: 000_000_4.0}', "");
Error('\p{Is_Present_In=:=-	0_0_0_0_0_0_004.0}');
Error('\P{Is_Present_In=:=-	0_0_0_0_0_0_004.0}');
Expect(1, 983040, '\p{Is_Present_In:0_0_0_0_004.0}', "");
Expect(0, 983040, '\p{^Is_Present_In:0_0_0_0_004.0}', "");
Expect(0, 983040, '\P{Is_Present_In:0_0_0_0_004.0}', "");
Expect(1, 983040, '\P{^Is_Present_In:0_0_0_0_004.0}', "");
Expect(0, 983037, '\p{Is_Present_In:0_0_0_0_004.0}', "");
Expect(1, 983037, '\p{^Is_Present_In:0_0_0_0_004.0}', "");
Expect(1, 983037, '\P{Is_Present_In:0_0_0_0_004.0}', "");
Expect(0, 983037, '\P{^Is_Present_In:0_0_0_0_004.0}', "");
Error('\p{Is_In=-:=4.0}');
Error('\P{Is_In=-:=4.0}');
Expect(1, 983040, '\p{Is_In=000000004.0}', "");
Expect(0, 983040, '\p{^Is_In=000000004.0}', "");
Expect(0, 983040, '\P{Is_In=000000004.0}', "");
Expect(1, 983040, '\P{^Is_In=000000004.0}', "");
Expect(0, 983037, '\p{Is_In=000000004.0}', "");
Expect(1, 983037, '\p{^Is_In=000000004.0}', "");
Expect(1, 983037, '\P{Is_In=000000004.0}', "");
Expect(0, 983037, '\P{^Is_In=000000004.0}', "");
Error('\p{Present_In:  :=04.1}');
Error('\P{Present_In:  :=04.1}');
Expect(1, 983040, '\p{Present_In=4.1}', "");
Expect(0, 983040, '\p{^Present_In=4.1}', "");
Expect(0, 983040, '\P{Present_In=4.1}', "");
Expect(1, 983040, '\P{^Present_In=4.1}', "");
Expect(0, 983037, '\p{Present_In=4.1}', "");
Expect(1, 983037, '\p{^Present_In=4.1}', "");
Expect(1, 983037, '\P{Present_In=4.1}', "");
Expect(0, 983037, '\P{^Present_In=4.1}', "");
Error('\p{In=_-0004.1/a/}');
Error('\P{In=_-0004.1/a/}');
Expect(1, 983040, '\p{In=004.1}', "");
Expect(0, 983040, '\p{^In=004.1}', "");
Expect(0, 983040, '\P{In=004.1}', "");
Expect(1, 983040, '\P{^In=004.1}', "");
Expect(0, 983037, '\p{In=004.1}', "");
Expect(1, 983037, '\p{^In=004.1}', "");
Expect(1, 983037, '\P{In=004.1}', "");
Expect(0, 983037, '\P{^In=004.1}', "");
Error('\p{Is_Present_In=:=_0_0_04.1}');
Error('\P{Is_Present_In=:=_0_0_04.1}');
Expect(1, 983040, '\p{Is_Present_In=+0000004.1}', "");
Expect(0, 983040, '\p{^Is_Present_In=+0000004.1}', "");
Expect(0, 983040, '\P{Is_Present_In=+0000004.1}', "");
Expect(1, 983040, '\P{^Is_Present_In=+0000004.1}', "");
Expect(0, 983037, '\p{Is_Present_In=+0000004.1}', "");
Expect(1, 983037, '\p{^Is_Present_In=+0000004.1}', "");
Expect(1, 983037, '\P{Is_Present_In=+0000004.1}', "");
Expect(0, 983037, '\P{^Is_Present_In=+0000004.1}', "");
Error('\p{Is_In=:=+000000000_4.1}');
Error('\P{Is_In=:=+000000000_4.1}');
Expect(1, 983040, '\p{Is_In=0000004.1}', "");
Expect(0, 983040, '\p{^Is_In=0000004.1}', "");
Expect(0, 983040, '\P{Is_In=0000004.1}', "");
Expect(1, 983040, '\P{^Is_In=0000004.1}', "");
Expect(0, 983037, '\p{Is_In=0000004.1}', "");
Expect(1, 983037, '\p{^Is_In=0000004.1}', "");
Expect(1, 983037, '\P{Is_In=0000004.1}', "");
Expect(0, 983037, '\P{^Is_In=0000004.1}', "");
Error('\p{Present_In= /a/5.0}');
Error('\P{Present_In= /a/5.0}');
Expect(1, 983040, '\p{Present_In=+005.0}', "");
Expect(0, 983040, '\p{^Present_In=+005.0}', "");
Expect(0, 983040, '\P{Present_In=+005.0}', "");
Expect(1, 983040, '\P{^Present_In=+005.0}', "");
Expect(0, 983037, '\p{Present_In=+005.0}', "");
Expect(1, 983037, '\p{^Present_In=+005.0}', "");
Expect(1, 983037, '\P{Present_In=+005.0}', "");
Expect(0, 983037, '\P{^Present_In=+005.0}', "");
Error('\p{In= :=+0005.0}');
Error('\P{In= :=+0005.0}');
Expect(1, 983040, '\p{In=+000_000_5.0}', "");
Expect(0, 983040, '\p{^In=+000_000_5.0}', "");
Expect(0, 983040, '\P{In=+000_000_5.0}', "");
Expect(1, 983040, '\P{^In=+000_000_5.0}', "");
Expect(0, 983037, '\p{In=+000_000_5.0}', "");
Expect(1, 983037, '\p{^In=+000_000_5.0}', "");
Expect(1, 983037, '\P{In=+000_000_5.0}', "");
Expect(0, 983037, '\P{^In=+000_000_5.0}', "");
Error('\p{Is_Present_In=	+005.0/a/}');
Error('\P{Is_Present_In=	+005.0/a/}');
Expect(1, 983040, '\p{Is_Present_In:   00_00_00_5.0}', "");
Expect(0, 983040, '\p{^Is_Present_In:   00_00_00_5.0}', "");
Expect(0, 983040, '\P{Is_Present_In:   00_00_00_5.0}', "");
Expect(1, 983040, '\P{^Is_Present_In:   00_00_00_5.0}', "");
Expect(0, 983037, '\p{Is_Present_In:   00_00_00_5.0}', "");
Expect(1, 983037, '\p{^Is_Present_In:   00_00_00_5.0}', "");
Expect(1, 983037, '\P{Is_Present_In:   00_00_00_5.0}', "");
Expect(0, 983037, '\P{^Is_Present_In:   00_00_00_5.0}', "");
Error('\p{Is_In:	 0005.0:=}');
Error('\P{Is_In:	 0005.0:=}');
Expect(1, 983040, '\p{Is_In=+00005.0}', "");
Expect(0, 983040, '\p{^Is_In=+00005.0}', "");
Expect(0, 983040, '\P{Is_In=+00005.0}', "");
Expect(1, 983040, '\P{^Is_In=+00005.0}', "");
Expect(0, 983037, '\p{Is_In=+00005.0}', "");
Expect(1, 983037, '\p{^Is_In=+00005.0}', "");
Expect(1, 983037, '\P{Is_In=+00005.0}', "");
Expect(0, 983037, '\P{^Is_In=+00005.0}', "");
Error('\p{Present_In:	/a/ +00000005.1}');
Error('\P{Present_In:	/a/ +00000005.1}');
Expect(1, 983040, '\p{Present_In=00_5.1}', "");
Expect(0, 983040, '\p{^Present_In=00_5.1}', "");
Expect(0, 983040, '\P{Present_In=00_5.1}', "");
Expect(1, 983040, '\P{^Present_In=00_5.1}', "");
Expect(0, 983037, '\p{Present_In=00_5.1}', "");
Expect(1, 983037, '\p{^Present_In=00_5.1}', "");
Expect(1, 983037, '\P{Present_In=00_5.1}', "");
Expect(0, 983037, '\P{^Present_In=00_5.1}', "");
Error('\p{In=	:=0000000005.1}');
Error('\P{In=	:=0000000005.1}');
Expect(1, 983040, '\p{In=00_5.1}', "");
Expect(0, 983040, '\p{^In=00_5.1}', "");
Expect(0, 983040, '\P{In=00_5.1}', "");
Expect(1, 983040, '\P{^In=00_5.1}', "");
Expect(0, 983037, '\p{In=00_5.1}', "");
Expect(1, 983037, '\p{^In=00_5.1}', "");
Expect(1, 983037, '\P{In=00_5.1}', "");
Expect(0, 983037, '\P{^In=00_5.1}', "");
Error('\p{Is_Present_In=_-+000_000_000_5.1:=}');
Error('\P{Is_Present_In=_-+000_000_000_5.1:=}');
Expect(1, 983040, '\p{Is_Present_In=00000_5.1}', "");
Expect(0, 983040, '\p{^Is_Present_In=00000_5.1}', "");
Expect(0, 983040, '\P{Is_Present_In=00000_5.1}', "");
Expect(1, 983040, '\P{^Is_Present_In=00000_5.1}', "");
Expect(0, 983037, '\p{Is_Present_In=00000_5.1}', "");
Expect(1, 983037, '\p{^Is_Present_In=00000_5.1}', "");
Expect(1, 983037, '\P{Is_Present_In=00000_5.1}', "");
Expect(0, 983037, '\P{^Is_Present_In=00000_5.1}', "");
Error('\p{Is_In=-/a/000_5.1}');
Error('\P{Is_In=-/a/000_5.1}');
Expect(1, 983040, '\p{Is_In=000005.1}', "");
Expect(0, 983040, '\p{^Is_In=000005.1}', "");
Expect(0, 983040, '\P{Is_In=000005.1}', "");
Expect(1, 983040, '\P{^Is_In=000005.1}', "");
Expect(0, 983037, '\p{Is_In=000005.1}', "");
Expect(1, 983037, '\p{^Is_In=000005.1}', "");
Expect(1, 983037, '\P{Is_In=000005.1}', "");
Expect(0, 983037, '\P{^Is_In=000005.1}', "");
Error('\p{Present_In:/a/0_5.2}');
Error('\P{Present_In:/a/0_5.2}');
Expect(1, 983040, '\p{Present_In=0000005.2}', "");
Expect(0, 983040, '\p{^Present_In=0000005.2}', "");
Expect(0, 983040, '\P{Present_In=0000005.2}', "");
Expect(1, 983040, '\P{^Present_In=0000005.2}', "");
Expect(0, 983037, '\p{Present_In=0000005.2}', "");
Expect(1, 983037, '\p{^Present_In=0000005.2}', "");
Expect(1, 983037, '\P{Present_In=0000005.2}', "");
Expect(0, 983037, '\P{^Present_In=0000005.2}', "");
Error('\p{In=_	5.2:=}');
Error('\P{In=_	5.2:=}');
Expect(1, 983040, '\p{In=000000005.2}', "");
Expect(0, 983040, '\p{^In=000000005.2}', "");
Expect(0, 983040, '\P{In=000000005.2}', "");
Expect(1, 983040, '\P{^In=000000005.2}', "");
Expect(0, 983037, '\p{In=000000005.2}', "");
Expect(1, 983037, '\p{^In=000000005.2}', "");
Expect(1, 983037, '\P{In=000000005.2}', "");
Expect(0, 983037, '\P{^In=000000005.2}', "");
Error('\p{Is_Present_In=:=	000005.2}');
Error('\P{Is_Present_In=:=	000005.2}');
Expect(1, 983040, '\p{Is_Present_In=0_0_05.2}', "");
Expect(0, 983040, '\p{^Is_Present_In=0_0_05.2}', "");
Expect(0, 983040, '\P{Is_Present_In=0_0_05.2}', "");
Expect(1, 983040, '\P{^Is_Present_In=0_0_05.2}', "");
Expect(0, 983037, '\p{Is_Present_In=0_0_05.2}', "");
Expect(1, 983037, '\p{^Is_Present_In=0_0_05.2}', "");
Expect(1, 983037, '\P{Is_Present_In=0_0_05.2}', "");
Expect(0, 983037, '\P{^Is_Present_In=0_0_05.2}', "");
Error('\p{Is_In=:= 	000_000_5.2}');
Error('\P{Is_In=:= 	000_000_5.2}');
Expect(1, 983040, '\p{Is_In=+0_0_0_0_0005.2}', "");
Expect(0, 983040, '\p{^Is_In=+0_0_0_0_0005.2}', "");
Expect(0, 983040, '\P{Is_In=+0_0_0_0_0005.2}', "");
Expect(1, 983040, '\P{^Is_In=+0_0_0_0_0005.2}', "");
Expect(0, 983037, '\p{Is_In=+0_0_0_0_0005.2}', "");
Expect(1, 983037, '\p{^Is_In=+0_0_0_0_0005.2}', "");
Expect(1, 983037, '\P{Is_In=+0_0_0_0_0005.2}', "");
Expect(0, 983037, '\P{^Is_In=+0_0_0_0_0005.2}', "");
Error('\p{Present_In=-006.0:=}');
Error('\P{Present_In=-006.0:=}');
Expect(1, 983040, '\p{Present_In=+000006.0}', "");
Expect(0, 983040, '\p{^Present_In=+000006.0}', "");
Expect(0, 983040, '\P{Present_In=+000006.0}', "");
Expect(1, 983040, '\P{^Present_In=+000006.0}', "");
Expect(0, 983037, '\p{Present_In=+000006.0}', "");
Expect(1, 983037, '\p{^Present_In=+000006.0}', "");
Expect(1, 983037, '\P{Present_In=+000006.0}', "");
Expect(0, 983037, '\P{^Present_In=+000006.0}', "");
Error('\p{In=	:=00_00_06.0}');
Error('\P{In=	:=00_00_06.0}');
Expect(1, 983040, '\p{In=06.0}', "");
Expect(0, 983040, '\p{^In=06.0}', "");
Expect(0, 983040, '\P{In=06.0}', "");
Expect(1, 983040, '\P{^In=06.0}', "");
Expect(0, 983037, '\p{In=06.0}', "");
Expect(1, 983037, '\p{^In=06.0}', "");
Expect(1, 983037, '\P{In=06.0}', "");
Expect(0, 983037, '\P{^In=06.0}', "");
Error('\p{Is_Present_In=/a/ -0_0_0_0_0_0_0006.0}');
Error('\P{Is_Present_In=/a/ -0_0_0_0_0_0_0006.0}');
Expect(1, 983040, '\p{Is_Present_In=+00_00_00_6.0}', "");
Expect(0, 983040, '\p{^Is_Present_In=+00_00_00_6.0}', "");
Expect(0, 983040, '\P{Is_Present_In=+00_00_00_6.0}', "");
Expect(1, 983040, '\P{^Is_Present_In=+00_00_00_6.0}', "");
Expect(0, 983037, '\p{Is_Present_In=+00_00_00_6.0}', "");
Expect(1, 983037, '\p{^Is_Present_In=+00_00_00_6.0}', "");
Expect(1, 983037, '\P{Is_Present_In=+00_00_00_6.0}', "");
Expect(0, 983037, '\P{^Is_Present_In=+00_00_00_6.0}', "");
Error('\p{Is_In=_/a/+0006.0}');
Error('\P{Is_In=_/a/+0006.0}');
Expect(1, 983040, '\p{Is_In=000000006.0}', "");
Expect(0, 983040, '\p{^Is_In=000000006.0}', "");
Expect(0, 983040, '\P{Is_In=000000006.0}', "");
Expect(1, 983040, '\P{^Is_In=000000006.0}', "");
Expect(0, 983037, '\p{Is_In=000000006.0}', "");
Expect(1, 983037, '\p{^Is_In=000000006.0}', "");
Expect(1, 983037, '\P{Is_In=000000006.0}', "");
Expect(0, 983037, '\P{^Is_In=000000006.0}', "");
Error('\p{Present_In= /a/+0_0_0_006.1}');
Error('\P{Present_In= /a/+0_0_0_006.1}');
Expect(1, 983040, '\p{Present_In=+0_6.1}', "");
Expect(0, 983040, '\p{^Present_In=+0_6.1}', "");
Expect(0, 983040, '\P{Present_In=+0_6.1}', "");
Expect(1, 983040, '\P{^Present_In=+0_6.1}', "");
Expect(0, 983037, '\p{Present_In=+0_6.1}', "");
Expect(1, 983037, '\p{^Present_In=+0_6.1}', "");
Expect(1, 983037, '\P{Present_In=+0_6.1}', "");
Expect(0, 983037, '\P{^Present_In=+0_6.1}', "");
Error('\p{In=/a/+00_00_00_006.1}');
Error('\P{In=/a/+00_00_00_006.1}');
Expect(1, 983040, '\p{In=6.1}', "");
Expect(0, 983040, '\p{^In=6.1}', "");
Expect(0, 983040, '\P{In=6.1}', "");
Expect(1, 983040, '\P{^In=6.1}', "");
Expect(0, 983037, '\p{In=6.1}', "");
Expect(1, 983037, '\p{^In=6.1}', "");
Expect(1, 983037, '\P{In=6.1}', "");
Expect(0, 983037, '\P{^In=6.1}', "");
Error('\p{Is_Present_In=:=000000006.1}');
Error('\P{Is_Present_In=:=000000006.1}');
Expect(1, 983040, '\p{Is_Present_In=000000006.1}', "");
Expect(0, 983040, '\p{^Is_Present_In=000000006.1}', "");
Expect(0, 983040, '\P{Is_Present_In=000000006.1}', "");
Expect(1, 983040, '\P{^Is_Present_In=000000006.1}', "");
Expect(0, 983037, '\p{Is_Present_In=000000006.1}', "");
Expect(1, 983037, '\p{^Is_Present_In=000000006.1}', "");
Expect(1, 983037, '\P{Is_Present_In=000000006.1}', "");
Expect(0, 983037, '\P{^Is_Present_In=000000006.1}', "");
Error('\p{Is_In=/a/ 	000000000_6.1}');
Error('\P{Is_In=/a/ 	000000000_6.1}');
Expect(1, 983040, '\p{Is_In:	00_00_00_06.1}', "");
Expect(0, 983040, '\p{^Is_In:	00_00_00_06.1}', "");
Expect(0, 983040, '\P{Is_In:	00_00_00_06.1}', "");
Expect(1, 983040, '\P{^Is_In:	00_00_00_06.1}', "");
Expect(0, 983037, '\p{Is_In:	00_00_00_06.1}', "");
Expect(1, 983037, '\p{^Is_In:	00_00_00_06.1}', "");
Expect(1, 983037, '\P{Is_In:	00_00_00_06.1}', "");
Expect(0, 983037, '\P{^Is_In:	00_00_00_06.1}', "");
Error('\p{Present_In=_:=00000006.2}');
Error('\P{Present_In=_:=00000006.2}');
Expect(1, 983040, '\p{Present_In=6.2}', "");
Expect(0, 983040, '\p{^Present_In=6.2}', "");
Expect(0, 983040, '\P{Present_In=6.2}', "");
Expect(1, 983040, '\P{^Present_In=6.2}', "");
Expect(0, 983037, '\p{Present_In=6.2}', "");
Expect(1, 983037, '\p{^Present_In=6.2}', "");
Expect(1, 983037, '\P{Present_In=6.2}', "");
Expect(0, 983037, '\P{^Present_In=6.2}', "");
Error('\p{In=:=	 000000006.2}');
Error('\P{In=:=	 000000006.2}');
Expect(1, 983040, '\p{In=+06.2}', "");
Expect(0, 983040, '\p{^In=+06.2}', "");
Expect(0, 983040, '\P{In=+06.2}', "");
Expect(1, 983040, '\P{^In=+06.2}', "");
Expect(0, 983037, '\p{In=+06.2}', "");
Expect(1, 983037, '\p{^In=+06.2}', "");
Expect(1, 983037, '\P{In=+06.2}', "");
Expect(0, 983037, '\P{^In=+06.2}', "");
Error('\p{Is_Present_In=__00000006.2:=}');
Error('\P{Is_Present_In=__00000006.2:=}');
Expect(1, 983040, '\p{Is_Present_In=00_00_00_6.2}', "");
Expect(0, 983040, '\p{^Is_Present_In=00_00_00_6.2}', "");
Expect(0, 983040, '\P{Is_Present_In=00_00_00_6.2}', "");
Expect(1, 983040, '\P{^Is_Present_In=00_00_00_6.2}', "");
Expect(0, 983037, '\p{Is_Present_In=00_00_00_6.2}', "");
Expect(1, 983037, '\p{^Is_Present_In=00_00_00_6.2}', "");
Expect(1, 983037, '\P{Is_Present_In=00_00_00_6.2}', "");
Expect(0, 983037, '\P{^Is_Present_In=00_00_00_6.2}', "");
Error('\p{Is_In=-0000006.2:=}');
Error('\P{Is_In=-0000006.2:=}');
Expect(1, 983040, '\p{Is_In=00000_6.2}', "");
Expect(0, 983040, '\p{^Is_In=00000_6.2}', "");
Expect(0, 983040, '\P{Is_In=00000_6.2}', "");
Expect(1, 983040, '\P{^Is_In=00000_6.2}', "");
Expect(0, 983037, '\p{Is_In=00000_6.2}', "");
Expect(1, 983037, '\p{^Is_In=00000_6.2}', "");
Expect(1, 983037, '\P{Is_In=00000_6.2}', "");
Expect(0, 983037, '\P{^Is_In=00000_6.2}', "");
Error('\p{Present_In=	/a/6.3}');
Error('\P{Present_In=	/a/6.3}');
Expect(1, 983040, '\p{Present_In=0000_6.3}', "");
Expect(0, 983040, '\p{^Present_In=0000_6.3}', "");
Expect(0, 983040, '\P{Present_In=0000_6.3}', "");
Expect(1, 983040, '\P{^Present_In=0000_6.3}', "");
Expect(0, 983037, '\p{Present_In=0000_6.3}', "");
Expect(1, 983037, '\p{^Present_In=0000_6.3}', "");
Expect(1, 983037, '\P{Present_In=0000_6.3}', "");
Expect(0, 983037, '\P{^Present_In=0000_6.3}', "");
Error('\p{In:	/a/0000006.3}');
Error('\P{In:	/a/0000006.3}');
Expect(1, 983040, '\p{In=00000006.3}', "");
Expect(0, 983040, '\p{^In=00000006.3}', "");
Expect(0, 983040, '\P{In=00000006.3}', "");
Expect(1, 983040, '\P{^In=00000006.3}', "");
Expect(0, 983037, '\p{In=00000006.3}', "");
Expect(1, 983037, '\p{^In=00000006.3}', "");
Expect(1, 983037, '\P{In=00000006.3}', "");
Expect(0, 983037, '\P{^In=00000006.3}', "");
Error('\p{Is_Present_In=/a/000_6.3}');
Error('\P{Is_Present_In=/a/000_6.3}');
Expect(1, 983040, '\p{Is_Present_In=000_000_000_6.3}', "");
Expect(0, 983040, '\p{^Is_Present_In=000_000_000_6.3}', "");
Expect(0, 983040, '\P{Is_Present_In=000_000_000_6.3}', "");
Expect(1, 983040, '\P{^Is_Present_In=000_000_000_6.3}', "");
Expect(0, 983037, '\p{Is_Present_In=000_000_000_6.3}', "");
Expect(1, 983037, '\p{^Is_Present_In=000_000_000_6.3}', "");
Expect(1, 983037, '\P{Is_Present_In=000_000_000_6.3}', "");
Expect(0, 983037, '\P{^Is_Present_In=000_000_000_6.3}', "");
Error('\p{Is_In=:= 00000006.3}');
Error('\P{Is_In=:= 00000006.3}');
Expect(1, 983040, '\p{Is_In=006.3}', "");
Expect(0, 983040, '\p{^Is_In=006.3}', "");
Expect(0, 983040, '\P{Is_In=006.3}', "");
Expect(1, 983040, '\P{^Is_In=006.3}', "");
Expect(0, 983037, '\p{Is_In=006.3}', "");
Expect(1, 983037, '\p{^Is_In=006.3}', "");
Expect(1, 983037, '\P{Is_In=006.3}', "");
Expect(0, 983037, '\P{^Is_In=006.3}', "");
Error('\p{Present_In=- UNASSIGNED/a/}');
Error('\P{Present_In=- UNASSIGNED/a/}');
Expect(1, 983037, '\p{Present_In=unassigned}', "");
Expect(0, 983037, '\p{^Present_In=unassigned}', "");
Expect(0, 983037, '\P{Present_In=unassigned}', "");
Expect(1, 983037, '\P{^Present_In=unassigned}', "");
Expect(0, 983040, '\p{Present_In=unassigned}', "");
Expect(1, 983040, '\p{^Present_In=unassigned}', "");
Expect(1, 983040, '\P{Present_In=unassigned}', "");
Expect(0, 983040, '\P{^Present_In=unassigned}', "");
Expect(1, 983037, '\p{Present_In=_Unassigned}', "");
Expect(0, 983037, '\p{^Present_In=_Unassigned}', "");
Expect(0, 983037, '\P{Present_In=_Unassigned}', "");
Expect(1, 983037, '\P{^Present_In=_Unassigned}', "");
Expect(0, 983040, '\p{Present_In=_Unassigned}', "");
Expect(1, 983040, '\p{^Present_In=_Unassigned}', "");
Expect(1, 983040, '\P{Present_In=_Unassigned}', "");
Expect(0, 983040, '\P{^Present_In=_Unassigned}', "");
Error('\p{In:   _UNASSIGNED:=}');
Error('\P{In:   _UNASSIGNED:=}');
Expect(1, 983037, '\p{In:   unassigned}', "");
Expect(0, 983037, '\p{^In:   unassigned}', "");
Expect(0, 983037, '\P{In:   unassigned}', "");
Expect(1, 983037, '\P{^In:   unassigned}', "");
Expect(0, 983040, '\p{In:   unassigned}', "");
Expect(1, 983040, '\p{^In:   unassigned}', "");
Expect(1, 983040, '\P{In:   unassigned}', "");
Expect(0, 983040, '\P{^In:   unassigned}', "");
Expect(1, 983037, '\p{In= 	Unassigned}', "");
Expect(0, 983037, '\p{^In= 	Unassigned}', "");
Expect(0, 983037, '\P{In= 	Unassigned}', "");
Expect(1, 983037, '\P{^In= 	Unassigned}', "");
Expect(0, 983040, '\p{In= 	Unassigned}', "");
Expect(1, 983040, '\p{^In= 	Unassigned}', "");
Expect(1, 983040, '\P{In= 	Unassigned}', "");
Expect(0, 983040, '\P{^In= 	Unassigned}', "");
Error('\p{Is_Present_In= _Unassigned:=}');
Error('\P{Is_Present_In= _Unassigned:=}');
Expect(1, 983037, '\p{Is_Present_In=unassigned}', "");
Expect(0, 983037, '\p{^Is_Present_In=unassigned}', "");
Expect(0, 983037, '\P{Is_Present_In=unassigned}', "");
Expect(1, 983037, '\P{^Is_Present_In=unassigned}', "");
Expect(0, 983040, '\p{Is_Present_In=unassigned}', "");
Expect(1, 983040, '\p{^Is_Present_In=unassigned}', "");
Expect(1, 983040, '\P{Is_Present_In=unassigned}', "");
Expect(0, 983040, '\P{^Is_Present_In=unassigned}', "");
Expect(1, 983037, '\p{Is_Present_In=_ Unassigned}', "");
Expect(0, 983037, '\p{^Is_Present_In=_ Unassigned}', "");
Expect(0, 983037, '\P{Is_Present_In=_ Unassigned}', "");
Expect(1, 983037, '\P{^Is_Present_In=_ Unassigned}', "");
Expect(0, 983040, '\p{Is_Present_In=_ Unassigned}', "");
Expect(1, 983040, '\p{^Is_Present_In=_ Unassigned}', "");
Expect(1, 983040, '\P{Is_Present_In=_ Unassigned}', "");
Expect(0, 983040, '\P{^Is_Present_In=_ Unassigned}', "");
Error('\p{Is_In=-/a/Unassigned}');
Error('\P{Is_In=-/a/Unassigned}');
Expect(1, 983037, '\p{Is_In:unassigned}', "");
Expect(0, 983037, '\p{^Is_In:unassigned}', "");
Expect(0, 983037, '\P{Is_In:unassigned}', "");
Expect(1, 983037, '\P{^Is_In:unassigned}', "");
Expect(0, 983040, '\p{Is_In:unassigned}', "");
Expect(1, 983040, '\p{^Is_In:unassigned}', "");
Expect(1, 983040, '\P{Is_In:unassigned}', "");
Expect(0, 983040, '\P{^Is_In:unassigned}', "");
Expect(1, 983037, '\p{Is_In= _UNASSIGNED}', "");
Expect(0, 983037, '\p{^Is_In= _UNASSIGNED}', "");
Expect(0, 983037, '\P{Is_In= _UNASSIGNED}', "");
Expect(1, 983037, '\P{^Is_In= _UNASSIGNED}', "");
Expect(0, 983040, '\p{Is_In= _UNASSIGNED}', "");
Expect(1, 983040, '\p{^Is_In= _UNASSIGNED}', "");
Expect(1, 983040, '\P{Is_In= _UNASSIGNED}', "");
Expect(0, 983040, '\P{^Is_In= _UNASSIGNED}', "");
Error('\p{Indic_Matra_Category=Bottom}');
Error('\P{Indic_Matra_Category=Bottom}');
Error('\p{InMC=Bottom}');
Error('\P{InMC=Bottom}');
Error('\p{Is_Indic_Matra_Category:Bottom}');
Error('\P{Is_Indic_Matra_Category:Bottom}');
Error('\p{Is_InMC=Bottom}');
Error('\P{Is_InMC=Bottom}');
Error('\p{Indic_Matra_Category=Bottom_And_Right}');
Error('\P{Indic_Matra_Category=Bottom_And_Right}');
Error('\p{InMC=Bottom_And_Right}');
Error('\P{InMC=Bottom_And_Right}');
Error('\p{Is_Indic_Matra_Category=Bottom_And_Right}');
Error('\P{Is_Indic_Matra_Category=Bottom_And_Right}');
Error('\p{Is_InMC=Bottom_And_Right}');
Error('\P{Is_InMC=Bottom_And_Right}');
Error('\p{Indic_Matra_Category=Invisible}');
Error('\P{Indic_Matra_Category=Invisible}');
Error('\p{InMC=Invisible}');
Error('\P{InMC=Invisible}');
Error('\p{Is_Indic_Matra_Category=Invisible}');
Error('\P{Is_Indic_Matra_Category=Invisible}');
Error('\p{Is_InMC:   Invisible}');
Error('\P{Is_InMC:   Invisible}');
Error('\p{Indic_Matra_Category=Left}');
Error('\P{Indic_Matra_Category=Left}');
Error('\p{InMC=Left}');
Error('\P{InMC=Left}');
Error('\p{Is_Indic_Matra_Category=Left}');
Error('\P{Is_Indic_Matra_Category=Left}');
Error('\p{Is_InMC=Left}');
Error('\P{Is_InMC=Left}');
Error('\p{Indic_Matra_Category: Left_And_Right}');
Error('\P{Indic_Matra_Category: Left_And_Right}');
Error('\p{InMC=Left_And_Right}');
Error('\P{InMC=Left_And_Right}');
Error('\p{Is_Indic_Matra_Category=Left_And_Right}');
Error('\P{Is_Indic_Matra_Category=Left_And_Right}');
Error('\p{Is_InMC=Left_And_Right}');
Error('\P{Is_InMC=Left_And_Right}');
Error('\p{Indic_Matra_Category=NA}');
Error('\P{Indic_Matra_Category=NA}');
Error('\p{InMC=NA}');
Error('\P{InMC=NA}');
Error('\p{Is_Indic_Matra_Category=NA}');
Error('\P{Is_Indic_Matra_Category=NA}');
Error('\p{Is_InMC=NA}');
Error('\P{Is_InMC=NA}');
Error('\p{Indic_Matra_Category=Overstruck}');
Error('\P{Indic_Matra_Category=Overstruck}');
Error('\p{InMC=Overstruck}');
Error('\P{InMC=Overstruck}');
Error('\p{Is_Indic_Matra_Category=Overstruck}');
Error('\P{Is_Indic_Matra_Category=Overstruck}');
Error('\p{Is_InMC: Overstruck}');
Error('\P{Is_InMC: Overstruck}');
Error('\p{Indic_Matra_Category=Right}');
Error('\P{Indic_Matra_Category=Right}');
Error('\p{InMC=Right}');
Error('\P{InMC=Right}');
Error('\p{Is_Indic_Matra_Category=Right}');
Error('\P{Is_Indic_Matra_Category=Right}');
Error('\p{Is_InMC=Right}');
Error('\P{Is_InMC=Right}');
Error('\p{Indic_Matra_Category=Top}');
Error('\P{Indic_Matra_Category=Top}');
Error('\p{InMC: Top}');
Error('\P{InMC: Top}');
Error('\p{Is_Indic_Matra_Category=Top}');
Error('\P{Is_Indic_Matra_Category=Top}');
Error('\p{Is_InMC=Top}');
Error('\P{Is_InMC=Top}');
Error('\p{Indic_Matra_Category=Top_And_Bottom}');
Error('\P{Indic_Matra_Category=Top_And_Bottom}');
Error('\p{InMC=Top_And_Bottom}');
Error('\P{InMC=Top_And_Bottom}');
Error('\p{Is_Indic_Matra_Category=Top_And_Bottom}');
Error('\P{Is_Indic_Matra_Category=Top_And_Bottom}');
Error('\p{Is_InMC=Top_And_Bottom}');
Error('\P{Is_InMC=Top_And_Bottom}');
Error('\p{Indic_Matra_Category=Top_And_Bottom_And_Right}');
Error('\P{Indic_Matra_Category=Top_And_Bottom_And_Right}');
Error('\p{InMC: Top_And_Bottom_And_Right}');
Error('\P{InMC: Top_And_Bottom_And_Right}');
Error('\p{Is_Indic_Matra_Category=Top_And_Bottom_And_Right}');
Error('\P{Is_Indic_Matra_Category=Top_And_Bottom_And_Right}');
Error('\p{Is_InMC:   Top_And_Bottom_And_Right}');
Error('\P{Is_InMC:   Top_And_Bottom_And_Right}');
Error('\p{Indic_Matra_Category=Top_And_Left}');
Error('\P{Indic_Matra_Category=Top_And_Left}');
Error('\p{InMC=Top_And_Left}');
Error('\P{InMC=Top_And_Left}');
Error('\p{Is_Indic_Matra_Category=Top_And_Left}');
Error('\P{Is_Indic_Matra_Category=Top_And_Left}');
Error('\p{Is_InMC=Top_And_Left}');
Error('\P{Is_InMC=Top_And_Left}');
Error('\p{Indic_Matra_Category=Top_And_Left_And_Right}');
Error('\P{Indic_Matra_Category=Top_And_Left_And_Right}');
Error('\p{InMC=Top_And_Left_And_Right}');
Error('\P{InMC=Top_And_Left_And_Right}');
Error('\p{Is_Indic_Matra_Category:   Top_And_Left_And_Right}');
Error('\P{Is_Indic_Matra_Category:   Top_And_Left_And_Right}');
Error('\p{Is_InMC=Top_And_Left_And_Right}');
Error('\P{Is_InMC=Top_And_Left_And_Right}');
Error('\p{Indic_Matra_Category=Top_And_Right}');
Error('\P{Indic_Matra_Category=Top_And_Right}');
Error('\p{InMC=Top_And_Right}');
Error('\P{InMC=Top_And_Right}');
Error('\p{Is_Indic_Matra_Category=Top_And_Right}');
Error('\P{Is_Indic_Matra_Category=Top_And_Right}');
Error('\p{Is_InMC: Top_And_Right}');
Error('\P{Is_InMC: Top_And_Right}');
Error('\p{Indic_Matra_Category=Visual_Order_Left}');
Error('\P{Indic_Matra_Category=Visual_Order_Left}');
Error('\p{InMC: Visual_Order_Left}');
Error('\P{InMC: Visual_Order_Left}');
Error('\p{Is_Indic_Matra_Category=Visual_Order_Left}');
Error('\P{Is_Indic_Matra_Category=Visual_Order_Left}');
Error('\p{Is_InMC=Visual_Order_Left}');
Error('\P{Is_InMC=Visual_Order_Left}');
Error('\p{Indic_Syllabic_Category=Avagraha}');
Error('\P{Indic_Syllabic_Category=Avagraha}');
Error('\p{InSC=Avagraha}');
Error('\P{InSC=Avagraha}');
Error('\p{Is_Indic_Syllabic_Category=Avagraha}');
Error('\P{Is_Indic_Syllabic_Category=Avagraha}');
Error('\p{Is_InSC:Avagraha}');
Error('\P{Is_InSC:Avagraha}');
Error('\p{Indic_Syllabic_Category=Bindu}');
Error('\P{Indic_Syllabic_Category=Bindu}');
Error('\p{InSC=Bindu}');
Error('\P{InSC=Bindu}');
Error('\p{Is_Indic_Syllabic_Category=Bindu}');
Error('\P{Is_Indic_Syllabic_Category=Bindu}');
Error('\p{Is_InSC=Bindu}');
Error('\P{Is_InSC=Bindu}');
Error('\p{Indic_Syllabic_Category=Consonant}');
Error('\P{Indic_Syllabic_Category=Consonant}');
Error('\p{InSC=Consonant}');
Error('\P{InSC=Consonant}');
Error('\p{Is_Indic_Syllabic_Category=Consonant}');
Error('\P{Is_Indic_Syllabic_Category=Consonant}');
Error('\p{Is_InSC=Consonant}');
Error('\P{Is_InSC=Consonant}');
Error('\p{Indic_Syllabic_Category=Consonant_Dead}');
Error('\P{Indic_Syllabic_Category=Consonant_Dead}');
Error('\p{InSC=Consonant_Dead}');
Error('\P{InSC=Consonant_Dead}');
Error('\p{Is_Indic_Syllabic_Category=Consonant_Dead}');
Error('\P{Is_Indic_Syllabic_Category=Consonant_Dead}');
Error('\p{Is_InSC=Consonant_Dead}');
Error('\P{Is_InSC=Consonant_Dead}');
Error('\p{Indic_Syllabic_Category=Consonant_Final}');
Error('\P{Indic_Syllabic_Category=Consonant_Final}');
Error('\p{InSC=Consonant_Final}');
Error('\P{InSC=Consonant_Final}');
Error('\p{Is_Indic_Syllabic_Category=Consonant_Final}');
Error('\P{Is_Indic_Syllabic_Category=Consonant_Final}');
Error('\p{Is_InSC=Consonant_Final}');
Error('\P{Is_InSC=Consonant_Final}');
Error('\p{Indic_Syllabic_Category=Consonant_Head_Letter}');
Error('\P{Indic_Syllabic_Category=Consonant_Head_Letter}');
Error('\p{InSC=Consonant_Head_Letter}');
Error('\P{InSC=Consonant_Head_Letter}');
Error('\p{Is_Indic_Syllabic_Category=Consonant_Head_Letter}');
Error('\P{Is_Indic_Syllabic_Category=Consonant_Head_Letter}');
Error('\p{Is_InSC:	Consonant_Head_Letter}');
Error('\P{Is_InSC:	Consonant_Head_Letter}');
Error('\p{Indic_Syllabic_Category=Consonant_Medial}');
Error('\P{Indic_Syllabic_Category=Consonant_Medial}');
Error('\p{InSC=Consonant_Medial}');
Error('\P{InSC=Consonant_Medial}');
Error('\p{Is_Indic_Syllabic_Category=Consonant_Medial}');
Error('\P{Is_Indic_Syllabic_Category=Consonant_Medial}');
Error('\p{Is_InSC=Consonant_Medial}');
Error('\P{Is_InSC=Consonant_Medial}');
Error('\p{Indic_Syllabic_Category=Consonant_Placeholder}');
Error('\P{Indic_Syllabic_Category=Consonant_Placeholder}');
Error('\p{InSC=Consonant_Placeholder}');
Error('\P{InSC=Consonant_Placeholder}');
Error('\p{Is_Indic_Syllabic_Category=Consonant_Placeholder}');
Error('\P{Is_Indic_Syllabic_Category=Consonant_Placeholder}');
Error('\p{Is_InSC=Consonant_Placeholder}');
Error('\P{Is_InSC=Consonant_Placeholder}');
Error('\p{Indic_Syllabic_Category=Consonant_Repha}');
Error('\P{Indic_Syllabic_Category=Consonant_Repha}');
Error('\p{InSC=Consonant_Repha}');
Error('\P{InSC=Consonant_Repha}');
Error('\p{Is_Indic_Syllabic_Category=Consonant_Repha}');
Error('\P{Is_Indic_Syllabic_Category=Consonant_Repha}');
Error('\p{Is_InSC:   Consonant_Repha}');
Error('\P{Is_InSC:   Consonant_Repha}');
Error('\p{Indic_Syllabic_Category=Consonant_Subjoined}');
Error('\P{Indic_Syllabic_Category=Consonant_Subjoined}');
Error('\p{InSC=Consonant_Subjoined}');
Error('\P{InSC=Consonant_Subjoined}');
Error('\p{Is_Indic_Syllabic_Category=Consonant_Subjoined}');
Error('\P{Is_Indic_Syllabic_Category=Consonant_Subjoined}');
Error('\p{Is_InSC=Consonant_Subjoined}');
Error('\P{Is_InSC=Consonant_Subjoined}');
Error('\p{Indic_Syllabic_Category=Modifying_Letter}');
Error('\P{Indic_Syllabic_Category=Modifying_Letter}');
Error('\p{InSC=Modifying_Letter}');
Error('\P{InSC=Modifying_Letter}');
Error('\p{Is_Indic_Syllabic_Category=Modifying_Letter}');
Error('\P{Is_Indic_Syllabic_Category=Modifying_Letter}');
Error('\p{Is_InSC=Modifying_Letter}');
Error('\P{Is_InSC=Modifying_Letter}');
Error('\p{Indic_Syllabic_Category=Nukta}');
Error('\P{Indic_Syllabic_Category=Nukta}');
Error('\p{InSC=Nukta}');
Error('\P{InSC=Nukta}');
Error('\p{Is_Indic_Syllabic_Category=Nukta}');
Error('\P{Is_Indic_Syllabic_Category=Nukta}');
Error('\p{Is_InSC=Nukta}');
Error('\P{Is_InSC=Nukta}');
Error('\p{Indic_Syllabic_Category=Other}');
Error('\P{Indic_Syllabic_Category=Other}');
Error('\p{InSC=Other}');
Error('\P{InSC=Other}');
Error('\p{Is_Indic_Syllabic_Category=Other}');
Error('\P{Is_Indic_Syllabic_Category=Other}');
Error('\p{Is_InSC=Other}');
Error('\P{Is_InSC=Other}');
Error('\p{Indic_Syllabic_Category=Register_Shifter}');
Error('\P{Indic_Syllabic_Category=Register_Shifter}');
Error('\p{InSC=Register_Shifter}');
Error('\P{InSC=Register_Shifter}');
Error('\p{Is_Indic_Syllabic_Category=Register_Shifter}');
Error('\P{Is_Indic_Syllabic_Category=Register_Shifter}');
Error('\p{Is_InSC=Register_Shifter}');
Error('\P{Is_InSC=Register_Shifter}');
Error('\p{Indic_Syllabic_Category=Tone_Letter}');
Error('\P{Indic_Syllabic_Category=Tone_Letter}');
Error('\p{InSC=Tone_Letter}');
Error('\P{InSC=Tone_Letter}');
Error('\p{Is_Indic_Syllabic_Category=Tone_Letter}');
Error('\P{Is_Indic_Syllabic_Category=Tone_Letter}');
Error('\p{Is_InSC=Tone_Letter}');
Error('\P{Is_InSC=Tone_Letter}');
Error('\p{Indic_Syllabic_Category=Tone_Mark}');
Error('\P{Indic_Syllabic_Category=Tone_Mark}');
Error('\p{InSC:   Tone_Mark}');
Error('\P{InSC:   Tone_Mark}');
Error('\p{Is_Indic_Syllabic_Category=Tone_Mark}');
Error('\P{Is_Indic_Syllabic_Category=Tone_Mark}');
Error('\p{Is_InSC=Tone_Mark}');
Error('\P{Is_InSC=Tone_Mark}');
Error('\p{Indic_Syllabic_Category: Virama}');
Error('\P{Indic_Syllabic_Category: Virama}');
Error('\p{InSC=Virama}');
Error('\P{InSC=Virama}');
Error('\p{Is_Indic_Syllabic_Category=Virama}');
Error('\P{Is_Indic_Syllabic_Category=Virama}');
Error('\p{Is_InSC:Virama}');
Error('\P{Is_InSC:Virama}');
Error('\p{Indic_Syllabic_Category=Visarga}');
Error('\P{Indic_Syllabic_Category=Visarga}');
Error('\p{InSC:   Visarga}');
Error('\P{InSC:   Visarga}');
Error('\p{Is_Indic_Syllabic_Category=Visarga}');
Error('\P{Is_Indic_Syllabic_Category=Visarga}');
Error('\p{Is_InSC=Visarga}');
Error('\P{Is_InSC=Visarga}');
Error('\p{Indic_Syllabic_Category=Vowel}');
Error('\P{Indic_Syllabic_Category=Vowel}');
Error('\p{InSC=Vowel}');
Error('\P{InSC=Vowel}');
Error('\p{Is_Indic_Syllabic_Category=Vowel}');
Error('\P{Is_Indic_Syllabic_Category=Vowel}');
Error('\p{Is_InSC=Vowel}');
Error('\P{Is_InSC=Vowel}');
Error('\p{Indic_Syllabic_Category=Vowel_Dependent}');
Error('\P{Indic_Syllabic_Category=Vowel_Dependent}');
Error('\p{InSC=Vowel_Dependent}');
Error('\P{InSC=Vowel_Dependent}');
Error('\p{Is_Indic_Syllabic_Category=Vowel_Dependent}');
Error('\P{Is_Indic_Syllabic_Category=Vowel_Dependent}');
Error('\p{Is_InSC=Vowel_Dependent}');
Error('\P{Is_InSC=Vowel_Dependent}');
Error('\p{Indic_Syllabic_Category=Vowel_Independent}');
Error('\P{Indic_Syllabic_Category=Vowel_Independent}');
Error('\p{InSC=Vowel_Independent}');
Error('\P{InSC=Vowel_Independent}');
Error('\p{Is_Indic_Syllabic_Category=Vowel_Independent}');
Error('\P{Is_Indic_Syllabic_Category=Vowel_Independent}');
Error('\p{Is_InSC=Vowel_Independent}');
Error('\P{Is_InSC=Vowel_Independent}');
Error('\p{Jamo_Short_Name=A}');
Error('\P{Jamo_Short_Name=A}');
Error('\p{JSN=A}');
Error('\P{JSN=A}');
Error('\p{Is_Jamo_Short_Name=A}');
Error('\P{Is_Jamo_Short_Name=A}');
Error('\p{Is_JSN: A}');
Error('\P{Is_JSN: A}');
Error('\p{Jamo_Short_Name=AE}');
Error('\P{Jamo_Short_Name=AE}');
Error('\p{JSN=AE}');
Error('\P{JSN=AE}');
Error('\p{Is_Jamo_Short_Name=AE}');
Error('\P{Is_Jamo_Short_Name=AE}');
Error('\p{Is_JSN:AE}');
Error('\P{Is_JSN:AE}');
Error('\p{Jamo_Short_Name=B}');
Error('\P{Jamo_Short_Name=B}');
Error('\p{JSN=B}');
Error('\P{JSN=B}');
Error('\p{Is_Jamo_Short_Name:	B}');
Error('\P{Is_Jamo_Short_Name:	B}');
Error('\p{Is_JSN=B}');
Error('\P{Is_JSN=B}');
Error('\p{Jamo_Short_Name=BB}');
Error('\P{Jamo_Short_Name=BB}');
Error('\p{JSN=BB}');
Error('\P{JSN=BB}');
Error('\p{Is_Jamo_Short_Name=BB}');
Error('\P{Is_Jamo_Short_Name=BB}');
Error('\p{Is_JSN=BB}');
Error('\P{Is_JSN=BB}');
Error('\p{Jamo_Short_Name=BS}');
Error('\P{Jamo_Short_Name=BS}');
Error('\p{JSN=BS}');
Error('\P{JSN=BS}');
Error('\p{Is_Jamo_Short_Name:BS}');
Error('\P{Is_Jamo_Short_Name:BS}');
Error('\p{Is_JSN=BS}');
Error('\P{Is_JSN=BS}');
Error('\p{Jamo_Short_Name=C}');
Error('\P{Jamo_Short_Name=C}');
Error('\p{JSN=C}');
Error('\P{JSN=C}');
Error('\p{Is_Jamo_Short_Name=C}');
Error('\P{Is_Jamo_Short_Name=C}');
Error('\p{Is_JSN=C}');
Error('\P{Is_JSN=C}');
Error('\p{Jamo_Short_Name: D}');
Error('\P{Jamo_Short_Name: D}');
Error('\p{JSN=D}');
Error('\P{JSN=D}');
Error('\p{Is_Jamo_Short_Name=D}');
Error('\P{Is_Jamo_Short_Name=D}');
Error('\p{Is_JSN=D}');
Error('\P{Is_JSN=D}');
Error('\p{Jamo_Short_Name:   DD}');
Error('\P{Jamo_Short_Name:   DD}');
Error('\p{JSN=DD}');
Error('\P{JSN=DD}');
Error('\p{Is_Jamo_Short_Name=DD}');
Error('\P{Is_Jamo_Short_Name=DD}');
Error('\p{Is_JSN=DD}');
Error('\P{Is_JSN=DD}');
Error('\p{Jamo_Short_Name=E}');
Error('\P{Jamo_Short_Name=E}');
Error('\p{JSN:	E}');
Error('\P{JSN:	E}');
Error('\p{Is_Jamo_Short_Name=E}');
Error('\P{Is_Jamo_Short_Name=E}');
Error('\p{Is_JSN=E}');
Error('\P{Is_JSN=E}');
Error('\p{Jamo_Short_Name:EO}');
Error('\P{Jamo_Short_Name:EO}');
Error('\p{JSN:   EO}');
Error('\P{JSN:   EO}');
Error('\p{Is_Jamo_Short_Name=EO}');
Error('\P{Is_Jamo_Short_Name=EO}');
Error('\p{Is_JSN=EO}');
Error('\P{Is_JSN=EO}');
Error('\p{Jamo_Short_Name=EU}');
Error('\P{Jamo_Short_Name=EU}');
Error('\p{JSN=EU}');
Error('\P{JSN=EU}');
Error('\p{Is_Jamo_Short_Name=EU}');
Error('\P{Is_Jamo_Short_Name=EU}');
Error('\p{Is_JSN=EU}');
Error('\P{Is_JSN=EU}');
Error('\p{Jamo_Short_Name=G}');
Error('\P{Jamo_Short_Name=G}');
Error('\p{JSN=G}');
Error('\P{JSN=G}');
Error('\p{Is_Jamo_Short_Name=G}');
Error('\P{Is_Jamo_Short_Name=G}');
Error('\p{Is_JSN=G}');
Error('\P{Is_JSN=G}');
Error('\p{Jamo_Short_Name=GG}');
Error('\P{Jamo_Short_Name=GG}');
Error('\p{JSN=GG}');
Error('\P{JSN=GG}');
Error('\p{Is_Jamo_Short_Name=GG}');
Error('\P{Is_Jamo_Short_Name=GG}');
Error('\p{Is_JSN=GG}');
Error('\P{Is_JSN=GG}');
Error('\p{Jamo_Short_Name: GS}');
Error('\P{Jamo_Short_Name: GS}');
Error('\p{JSN: GS}');
Error('\P{JSN: GS}');
Error('\p{Is_Jamo_Short_Name=GS}');
Error('\P{Is_Jamo_Short_Name=GS}');
Error('\p{Is_JSN=GS}');
Error('\P{Is_JSN=GS}');
Error('\p{Jamo_Short_Name=H}');
Error('\P{Jamo_Short_Name=H}');
Error('\p{JSN:H}');
Error('\P{JSN:H}');
Error('\p{Is_Jamo_Short_Name=H}');
Error('\P{Is_Jamo_Short_Name=H}');
Error('\p{Is_JSN=H}');
Error('\P{Is_JSN=H}');
Error('\p{Jamo_Short_Name=I}');
Error('\P{Jamo_Short_Name=I}');
Error('\p{JSN=I}');
Error('\P{JSN=I}');
Error('\p{Is_Jamo_Short_Name=I}');
Error('\P{Is_Jamo_Short_Name=I}');
Error('\p{Is_JSN:I}');
Error('\P{Is_JSN:I}');
Error('\p{Jamo_Short_Name=J}');
Error('\P{Jamo_Short_Name=J}');
Error('\p{JSN=J}');
Error('\P{JSN=J}');
Error('\p{Is_Jamo_Short_Name:   J}');
Error('\P{Is_Jamo_Short_Name:   J}');
Error('\p{Is_JSN=J}');
Error('\P{Is_JSN=J}');
Error('\p{Jamo_Short_Name=JJ}');
Error('\P{Jamo_Short_Name=JJ}');
Error('\p{JSN=JJ}');
Error('\P{JSN=JJ}');
Error('\p{Is_Jamo_Short_Name=JJ}');
Error('\P{Is_Jamo_Short_Name=JJ}');
Error('\p{Is_JSN=JJ}');
Error('\P{Is_JSN=JJ}');
Error('\p{Jamo_Short_Name:K}');
Error('\P{Jamo_Short_Name:K}');
Error('\p{JSN=K}');
Error('\P{JSN=K}');
Error('\p{Is_Jamo_Short_Name=K}');
Error('\P{Is_Jamo_Short_Name=K}');
Error('\p{Is_JSN=K}');
Error('\P{Is_JSN=K}');
Error('\p{Jamo_Short_Name:	L}');
Error('\P{Jamo_Short_Name:	L}');
Error('\p{JSN=L}');
Error('\P{JSN=L}');
Error('\p{Is_Jamo_Short_Name:	L}');
Error('\P{Is_Jamo_Short_Name:	L}');
Error('\p{Is_JSN=L}');
Error('\P{Is_JSN=L}');
Error('\p{Jamo_Short_Name=LB}');
Error('\P{Jamo_Short_Name=LB}');
Error('\p{JSN=LB}');
Error('\P{JSN=LB}');
Error('\p{Is_Jamo_Short_Name=LB}');
Error('\P{Is_Jamo_Short_Name=LB}');
Error('\p{Is_JSN=LB}');
Error('\P{Is_JSN=LB}');
Error('\p{Jamo_Short_Name=LG}');
Error('\P{Jamo_Short_Name=LG}');
Error('\p{JSN=LG}');
Error('\P{JSN=LG}');
Error('\p{Is_Jamo_Short_Name=LG}');
Error('\P{Is_Jamo_Short_Name=LG}');
Error('\p{Is_JSN=LG}');
Error('\P{Is_JSN=LG}');
Error('\p{Jamo_Short_Name=LH}');
Error('\P{Jamo_Short_Name=LH}');
Error('\p{JSN:LH}');
Error('\P{JSN:LH}');
Error('\p{Is_Jamo_Short_Name=LH}');
Error('\P{Is_Jamo_Short_Name=LH}');
Error('\p{Is_JSN=LH}');
Error('\P{Is_JSN=LH}');
Error('\p{Jamo_Short_Name=LM}');
Error('\P{Jamo_Short_Name=LM}');
Error('\p{JSN=LM}');
Error('\P{JSN=LM}');
Error('\p{Is_Jamo_Short_Name=LM}');
Error('\P{Is_Jamo_Short_Name=LM}');
Error('\p{Is_JSN=LM}');
Error('\P{Is_JSN=LM}');
Error('\p{Jamo_Short_Name=LP}');
Error('\P{Jamo_Short_Name=LP}');
Error('\p{JSN=LP}');
Error('\P{JSN=LP}');
Error('\p{Is_Jamo_Short_Name=LP}');
Error('\P{Is_Jamo_Short_Name=LP}');
Error('\p{Is_JSN=LP}');
Error('\P{Is_JSN=LP}');
Error('\p{Jamo_Short_Name=LS}');
Error('\P{Jamo_Short_Name=LS}');
Error('\p{JSN=LS}');
Error('\P{JSN=LS}');
Error('\p{Is_Jamo_Short_Name=LS}');
Error('\P{Is_Jamo_Short_Name=LS}');
Error('\p{Is_JSN=LS}');
Error('\P{Is_JSN=LS}');
Error('\p{Jamo_Short_Name=LT}');
Error('\P{Jamo_Short_Name=LT}');
Error('\p{JSN=LT}');
Error('\P{JSN=LT}');
Error('\p{Is_Jamo_Short_Name=LT}');
Error('\P{Is_Jamo_Short_Name=LT}');
Error('\p{Is_JSN=LT}');
Error('\P{Is_JSN=LT}');
Error('\p{Jamo_Short_Name=M}');
Error('\P{Jamo_Short_Name=M}');
Error('\p{JSN=M}');
Error('\P{JSN=M}');
Error('\p{Is_Jamo_Short_Name=M}');
Error('\P{Is_Jamo_Short_Name=M}');
Error('\p{Is_JSN: M}');
Error('\P{Is_JSN: M}');
Error('\p{Jamo_Short_Name=N}');
Error('\P{Jamo_Short_Name=N}');
Error('\p{JSN=N}');
Error('\P{JSN=N}');
Error('\p{Is_Jamo_Short_Name=N}');
Error('\P{Is_Jamo_Short_Name=N}');
Error('\p{Is_JSN=N}');
Error('\P{Is_JSN=N}');
Error('\p{Jamo_Short_Name=NG}');
Error('\P{Jamo_Short_Name=NG}');
Error('\p{JSN=NG}');
Error('\P{JSN=NG}');
Error('\p{Is_Jamo_Short_Name=NG}');
Error('\P{Is_Jamo_Short_Name=NG}');
Error('\p{Is_JSN=NG}');
Error('\P{Is_JSN=NG}');
Error('\p{Jamo_Short_Name=NH}');
Error('\P{Jamo_Short_Name=NH}');
Error('\p{JSN=NH}');
Error('\P{JSN=NH}');
Error('\p{Is_Jamo_Short_Name=NH}');
Error('\P{Is_Jamo_Short_Name=NH}');
Error('\p{Is_JSN=NH}');
Error('\P{Is_JSN=NH}');
Error('\p{Jamo_Short_Name:NJ}');
Error('\P{Jamo_Short_Name:NJ}');
Error('\p{JSN=NJ}');
Error('\P{JSN=NJ}');
Error('\p{Is_Jamo_Short_Name=NJ}');
Error('\P{Is_Jamo_Short_Name=NJ}');
Error('\p{Is_JSN=NJ}');
Error('\P{Is_JSN=NJ}');
Error('\p{Jamo_Short_Name: O}');
Error('\P{Jamo_Short_Name: O}');
Error('\p{JSN=O}');
Error('\P{JSN=O}');
Error('\p{Is_Jamo_Short_Name=O}');
Error('\P{Is_Jamo_Short_Name=O}');
Error('\p{Is_JSN:O}');
Error('\P{Is_JSN:O}');
Error('\p{Jamo_Short_Name=OE}');
Error('\P{Jamo_Short_Name=OE}');
Error('\p{JSN:   OE}');
Error('\P{JSN:   OE}');
Error('\p{Is_Jamo_Short_Name=OE}');
Error('\P{Is_Jamo_Short_Name=OE}');
Error('\p{Is_JSN=OE}');
Error('\P{Is_JSN=OE}');
Error('\p{Jamo_Short_Name=P}');
Error('\P{Jamo_Short_Name=P}');
Error('\p{JSN=P}');
Error('\P{JSN=P}');
Error('\p{Is_Jamo_Short_Name:   P}');
Error('\P{Is_Jamo_Short_Name:   P}');
Error('\p{Is_JSN=P}');
Error('\P{Is_JSN=P}');
Error('\p{Jamo_Short_Name=R}');
Error('\P{Jamo_Short_Name=R}');
Error('\p{JSN: R}');
Error('\P{JSN: R}');
Error('\p{Is_Jamo_Short_Name=R}');
Error('\P{Is_Jamo_Short_Name=R}');
Error('\p{Is_JSN=R}');
Error('\P{Is_JSN=R}');
Error('\p{Jamo_Short_Name=S}');
Error('\P{Jamo_Short_Name=S}');
Error('\p{JSN=S}');
Error('\P{JSN=S}');
Error('\p{Is_Jamo_Short_Name:   S}');
Error('\P{Is_Jamo_Short_Name:   S}');
Error('\p{Is_JSN=S}');
Error('\P{Is_JSN=S}');
Error('\p{Jamo_Short_Name=SS}');
Error('\P{Jamo_Short_Name=SS}');
Error('\p{JSN=SS}');
Error('\P{JSN=SS}');
Error('\p{Is_Jamo_Short_Name=SS}');
Error('\P{Is_Jamo_Short_Name=SS}');
Error('\p{Is_JSN=SS}');
Error('\P{Is_JSN=SS}');
Error('\p{Jamo_Short_Name=T}');
Error('\P{Jamo_Short_Name=T}');
Error('\p{JSN=T}');
Error('\P{JSN=T}');
Error('\p{Is_Jamo_Short_Name=T}');
Error('\P{Is_Jamo_Short_Name=T}');
Error('\p{Is_JSN=T}');
Error('\P{Is_JSN=T}');
Error('\p{Jamo_Short_Name=U}');
Error('\P{Jamo_Short_Name=U}');
Error('\p{JSN=U}');
Error('\P{JSN=U}');
Error('\p{Is_Jamo_Short_Name=U}');
Error('\P{Is_Jamo_Short_Name=U}');
Error('\p{Is_JSN=U}');
Error('\P{Is_JSN=U}');
Error('\p{Jamo_Short_Name=WA}');
Error('\P{Jamo_Short_Name=WA}');
Error('\p{JSN=WA}');
Error('\P{JSN=WA}');
Error('\p{Is_Jamo_Short_Name=WA}');
Error('\P{Is_Jamo_Short_Name=WA}');
Error('\p{Is_JSN=WA}');
Error('\P{Is_JSN=WA}');
Error('\p{Jamo_Short_Name=WAE}');
Error('\P{Jamo_Short_Name=WAE}');
Error('\p{JSN:   WAE}');
Error('\P{JSN:   WAE}');
Error('\p{Is_Jamo_Short_Name=WAE}');
Error('\P{Is_Jamo_Short_Name=WAE}');
Error('\p{Is_JSN=WAE}');
Error('\P{Is_JSN=WAE}');
Error('\p{Jamo_Short_Name=WE}');
Error('\P{Jamo_Short_Name=WE}');
Error('\p{JSN=WE}');
Error('\P{JSN=WE}');
Error('\p{Is_Jamo_Short_Name=WE}');
Error('\P{Is_Jamo_Short_Name=WE}');
Error('\p{Is_JSN=WE}');
Error('\P{Is_JSN=WE}');
Error('\p{Jamo_Short_Name:	WEO}');
Error('\P{Jamo_Short_Name:	WEO}');
Error('\p{JSN=WEO}');
Error('\P{JSN=WEO}');
Error('\p{Is_Jamo_Short_Name=WEO}');
Error('\P{Is_Jamo_Short_Name=WEO}');
Error('\p{Is_JSN=WEO}');
Error('\P{Is_JSN=WEO}');
Error('\p{Jamo_Short_Name=WI}');
Error('\P{Jamo_Short_Name=WI}');
Error('\p{JSN=WI}');
Error('\P{JSN=WI}');
Error('\p{Is_Jamo_Short_Name=WI}');
Error('\P{Is_Jamo_Short_Name=WI}');
Error('\p{Is_JSN=WI}');
Error('\P{Is_JSN=WI}');
Error('\p{Jamo_Short_Name=YA}');
Error('\P{Jamo_Short_Name=YA}');
Error('\p{JSN=YA}');
Error('\P{JSN=YA}');
Error('\p{Is_Jamo_Short_Name:   YA}');
Error('\P{Is_Jamo_Short_Name:   YA}');
Error('\p{Is_JSN=YA}');
Error('\P{Is_JSN=YA}');
Error('\p{Jamo_Short_Name=YAE}');
Error('\P{Jamo_Short_Name=YAE}');
Error('\p{JSN=YAE}');
Error('\P{JSN=YAE}');
Error('\p{Is_Jamo_Short_Name: YAE}');
Error('\P{Is_Jamo_Short_Name: YAE}');
Error('\p{Is_JSN=YAE}');
Error('\P{Is_JSN=YAE}');
Error('\p{Jamo_Short_Name=YE}');
Error('\P{Jamo_Short_Name=YE}');
Error('\p{JSN=YE}');
Error('\P{JSN=YE}');
Error('\p{Is_Jamo_Short_Name=YE}');
Error('\P{Is_Jamo_Short_Name=YE}');
Error('\p{Is_JSN=YE}');
Error('\P{Is_JSN=YE}');
Error('\p{Jamo_Short_Name:YEO}');
Error('\P{Jamo_Short_Name:YEO}');
Error('\p{JSN=YEO}');
Error('\P{JSN=YEO}');
Error('\p{Is_Jamo_Short_Name=YEO}');
Error('\P{Is_Jamo_Short_Name=YEO}');
Error('\p{Is_JSN=YEO}');
Error('\P{Is_JSN=YEO}');
Error('\p{Jamo_Short_Name=YI}');
Error('\P{Jamo_Short_Name=YI}');
Error('\p{JSN=YI}');
Error('\P{JSN=YI}');
Error('\p{Is_Jamo_Short_Name=YI}');
Error('\P{Is_Jamo_Short_Name=YI}');
Error('\p{Is_JSN=YI}');
Error('\P{Is_JSN=YI}');
Error('\p{Jamo_Short_Name=YO}');
Error('\P{Jamo_Short_Name=YO}');
Error('\p{JSN=YO}');
Error('\P{JSN=YO}');
Error('\p{Is_Jamo_Short_Name=YO}');
Error('\P{Is_Jamo_Short_Name=YO}');
Error('\p{Is_JSN=YO}');
Error('\P{Is_JSN=YO}');
Error('\p{Jamo_Short_Name=YU}');
Error('\P{Jamo_Short_Name=YU}');
Error('\p{JSN=YU}');
Error('\P{JSN=YU}');
Error('\p{Is_Jamo_Short_Name=YU}');
Error('\P{Is_Jamo_Short_Name=YU}');
Error('\p{Is_JSN=YU}');
Error('\P{Is_JSN=YU}');
Error('\p{Join_Control=:=No}');
Error('\P{Join_Control=:=No}');
Expect(1, 8206, '\p{Join_Control:	no}', "");
Expect(0, 8206, '\p{^Join_Control:	no}', "");
Expect(0, 8206, '\P{Join_Control:	no}', "");
Expect(1, 8206, '\P{^Join_Control:	no}', "");
Expect(0, 8205, '\p{Join_Control:	no}', "");
Expect(1, 8205, '\p{^Join_Control:	no}', "");
Expect(1, 8205, '\P{Join_Control:	no}', "");
Expect(0, 8205, '\P{^Join_Control:	no}', "");
Expect(1, 8206, '\p{Join_Control=		NO}', "");
Expect(0, 8206, '\p{^Join_Control=		NO}', "");
Expect(0, 8206, '\P{Join_Control=		NO}', "");
Expect(1, 8206, '\P{^Join_Control=		NO}', "");
Expect(0, 8205, '\p{Join_Control=		NO}', "");
Expect(1, 8205, '\p{^Join_Control=		NO}', "");
Expect(1, 8205, '\P{Join_Control=		NO}', "");
Expect(0, 8205, '\P{^Join_Control=		NO}', "");
Error('\p{Join_C=	N:=}');
Error('\P{Join_C=	N:=}');
Expect(1, 8206, '\p{Join_C=n}', "");
Expect(0, 8206, '\p{^Join_C=n}', "");
Expect(0, 8206, '\P{Join_C=n}', "");
Expect(1, 8206, '\P{^Join_C=n}', "");
Expect(0, 8205, '\p{Join_C=n}', "");
Expect(1, 8205, '\p{^Join_C=n}', "");
Expect(1, 8205, '\P{Join_C=n}', "");
Expect(0, 8205, '\P{^Join_C=n}', "");
Expect(1, 8206, '\p{Join_C: -	n}', "");
Expect(0, 8206, '\p{^Join_C: -	n}', "");
Expect(0, 8206, '\P{Join_C: -	n}', "");
Expect(1, 8206, '\P{^Join_C: -	n}', "");
Expect(0, 8205, '\p{Join_C: -	n}', "");
Expect(1, 8205, '\p{^Join_C: -	n}', "");
Expect(1, 8205, '\P{Join_C: -	n}', "");
Expect(0, 8205, '\P{^Join_C: -	n}', "");
Error('\p{Is_Join_Control=:= -F}');
Error('\P{Is_Join_Control=:= -F}');
Expect(1, 8206, '\p{Is_Join_Control=f}', "");
Expect(0, 8206, '\p{^Is_Join_Control=f}', "");
Expect(0, 8206, '\P{Is_Join_Control=f}', "");
Expect(1, 8206, '\P{^Is_Join_Control=f}', "");
Expect(0, 8205, '\p{Is_Join_Control=f}', "");
Expect(1, 8205, '\p{^Is_Join_Control=f}', "");
Expect(1, 8205, '\P{Is_Join_Control=f}', "");
Expect(0, 8205, '\P{^Is_Join_Control=f}', "");
Expect(1, 8206, '\p{Is_Join_Control=_ F}', "");
Expect(0, 8206, '\p{^Is_Join_Control=_ F}', "");
Expect(0, 8206, '\P{Is_Join_Control=_ F}', "");
Expect(1, 8206, '\P{^Is_Join_Control=_ F}', "");
Expect(0, 8205, '\p{Is_Join_Control=_ F}', "");
Expect(1, 8205, '\p{^Is_Join_Control=_ F}', "");
Expect(1, 8205, '\P{Is_Join_Control=_ F}', "");
Expect(0, 8205, '\P{^Is_Join_Control=_ F}', "");
Error('\p{Is_Join_C=_:=False}');
Error('\P{Is_Join_C=_:=False}');
Expect(1, 8206, '\p{Is_Join_C=false}', "");
Expect(0, 8206, '\p{^Is_Join_C=false}', "");
Expect(0, 8206, '\P{Is_Join_C=false}', "");
Expect(1, 8206, '\P{^Is_Join_C=false}', "");
Expect(0, 8205, '\p{Is_Join_C=false}', "");
Expect(1, 8205, '\p{^Is_Join_C=false}', "");
Expect(1, 8205, '\P{Is_Join_C=false}', "");
Expect(0, 8205, '\P{^Is_Join_C=false}', "");
Expect(1, 8206, '\p{Is_Join_C=-FALSE}', "");
Expect(0, 8206, '\p{^Is_Join_C=-FALSE}', "");
Expect(0, 8206, '\P{Is_Join_C=-FALSE}', "");
Expect(1, 8206, '\P{^Is_Join_C=-FALSE}', "");
Expect(0, 8205, '\p{Is_Join_C=-FALSE}', "");
Expect(1, 8205, '\p{^Is_Join_C=-FALSE}', "");
Expect(1, 8205, '\P{Is_Join_C=-FALSE}', "");
Expect(0, 8205, '\P{^Is_Join_C=-FALSE}', "");
Error('\p{Join_Control:_:=yes}');
Error('\P{Join_Control:_:=yes}');
Expect(1, 8205, '\p{Join_Control=yes}', "");
Expect(0, 8205, '\p{^Join_Control=yes}', "");
Expect(0, 8205, '\P{Join_Control=yes}', "");
Expect(1, 8205, '\P{^Join_Control=yes}', "");
Expect(0, 8206, '\p{Join_Control=yes}', "");
Expect(1, 8206, '\p{^Join_Control=yes}', "");
Expect(1, 8206, '\P{Join_Control=yes}', "");
Expect(0, 8206, '\P{^Join_Control=yes}', "");
Expect(1, 8205, '\p{Join_Control=_-YES}', "");
Expect(0, 8205, '\p{^Join_Control=_-YES}', "");
Expect(0, 8205, '\P{Join_Control=_-YES}', "");
Expect(1, 8205, '\P{^Join_Control=_-YES}', "");
Expect(0, 8206, '\p{Join_Control=_-YES}', "");
Expect(1, 8206, '\p{^Join_Control=_-YES}', "");
Expect(1, 8206, '\P{Join_Control=_-YES}', "");
Expect(0, 8206, '\P{^Join_Control=_-YES}', "");
Error('\p{Join_C:	:=- y}');
Error('\P{Join_C:	:=- y}');
Expect(1, 8205, '\p{Join_C=y}', "");
Expect(0, 8205, '\p{^Join_C=y}', "");
Expect(0, 8205, '\P{Join_C=y}', "");
Expect(1, 8205, '\P{^Join_C=y}', "");
Expect(0, 8206, '\p{Join_C=y}', "");
Expect(1, 8206, '\p{^Join_C=y}', "");
Expect(1, 8206, '\P{Join_C=y}', "");
Expect(0, 8206, '\P{^Join_C=y}', "");
Expect(1, 8205, '\p{Join_C:	 y}', "");
Expect(0, 8205, '\p{^Join_C:	 y}', "");
Expect(0, 8205, '\P{Join_C:	 y}', "");
Expect(1, 8205, '\P{^Join_C:	 y}', "");
Expect(0, 8206, '\p{Join_C:	 y}', "");
Expect(1, 8206, '\p{^Join_C:	 y}', "");
Expect(1, 8206, '\P{Join_C:	 y}', "");
Expect(0, 8206, '\P{^Join_C:	 y}', "");
Error('\p{Is_Join_Control=/a/ t}');
Error('\P{Is_Join_Control=/a/ t}');
Expect(1, 8205, '\p{Is_Join_Control=t}', "");
Expect(0, 8205, '\p{^Is_Join_Control=t}', "");
Expect(0, 8205, '\P{Is_Join_Control=t}', "");
Expect(1, 8205, '\P{^Is_Join_Control=t}', "");
Expect(0, 8206, '\p{Is_Join_Control=t}', "");
Expect(1, 8206, '\p{^Is_Join_Control=t}', "");
Expect(1, 8206, '\P{Is_Join_Control=t}', "");
Expect(0, 8206, '\P{^Is_Join_Control=t}', "");
Expect(1, 8205, '\p{Is_Join_Control= _T}', "");
Expect(0, 8205, '\p{^Is_Join_Control= _T}', "");
Expect(0, 8205, '\P{Is_Join_Control= _T}', "");
Expect(1, 8205, '\P{^Is_Join_Control= _T}', "");
Expect(0, 8206, '\p{Is_Join_Control= _T}', "");
Expect(1, 8206, '\p{^Is_Join_Control= _T}', "");
Expect(1, 8206, '\P{Is_Join_Control= _T}', "");
Expect(0, 8206, '\P{^Is_Join_Control= _T}', "");
Error('\p{Is_Join_C=	/a/True}');
Error('\P{Is_Join_C=	/a/True}');
Expect(1, 8205, '\p{Is_Join_C=true}', "");
Expect(0, 8205, '\p{^Is_Join_C=true}', "");
Expect(0, 8205, '\P{Is_Join_C=true}', "");
Expect(1, 8205, '\P{^Is_Join_C=true}', "");
Expect(0, 8206, '\p{Is_Join_C=true}', "");
Expect(1, 8206, '\p{^Is_Join_C=true}', "");
Expect(1, 8206, '\P{Is_Join_C=true}', "");
Expect(0, 8206, '\P{^Is_Join_C=true}', "");
Expect(1, 8205, '\p{Is_Join_C= _true}', "");
Expect(0, 8205, '\p{^Is_Join_C= _true}', "");
Expect(0, 8205, '\P{Is_Join_C= _true}', "");
Expect(1, 8205, '\P{^Is_Join_C= _true}', "");
Expect(0, 8206, '\p{Is_Join_C= _true}', "");
Expect(1, 8206, '\p{^Is_Join_C= _true}', "");
Expect(1, 8206, '\P{Is_Join_C= _true}', "");
Expect(0, 8206, '\P{^Is_Join_C= _true}', "");
Error('\p{Logical_Order_Exception=:= _NO}');
Error('\P{Logical_Order_Exception=:= _NO}');
Expect(1, 43709, '\p{Logical_Order_Exception:	no}', "");
Expect(0, 43709, '\p{^Logical_Order_Exception:	no}', "");
Expect(0, 43709, '\P{Logical_Order_Exception:	no}', "");
Expect(1, 43709, '\P{^Logical_Order_Exception:	no}', "");
Expect(0, 43708, '\p{Logical_Order_Exception:	no}', "");
Expect(1, 43708, '\p{^Logical_Order_Exception:	no}', "");
Expect(1, 43708, '\P{Logical_Order_Exception:	no}', "");
Expect(0, 43708, '\P{^Logical_Order_Exception:	no}', "");
Expect(1, 43709, '\p{Logical_Order_Exception=-NO}', "");
Expect(0, 43709, '\p{^Logical_Order_Exception=-NO}', "");
Expect(0, 43709, '\P{Logical_Order_Exception=-NO}', "");
Expect(1, 43709, '\P{^Logical_Order_Exception=-NO}', "");
Expect(0, 43708, '\p{Logical_Order_Exception=-NO}', "");
Expect(1, 43708, '\p{^Logical_Order_Exception=-NO}', "");
Expect(1, 43708, '\P{Logical_Order_Exception=-NO}', "");
Expect(0, 43708, '\P{^Logical_Order_Exception=-NO}', "");
Error('\p{LOE= _n:=}');
Error('\P{LOE= _n:=}');
Expect(1, 43709, '\p{LOE=n}', "");
Expect(0, 43709, '\p{^LOE=n}', "");
Expect(0, 43709, '\P{LOE=n}', "");
Expect(1, 43709, '\P{^LOE=n}', "");
Expect(0, 43708, '\p{LOE=n}', "");
Expect(1, 43708, '\p{^LOE=n}', "");
Expect(1, 43708, '\P{LOE=n}', "");
Expect(0, 43708, '\P{^LOE=n}', "");
Expect(1, 43709, '\p{LOE=-_N}', "");
Expect(0, 43709, '\p{^LOE=-_N}', "");
Expect(0, 43709, '\P{LOE=-_N}', "");
Expect(1, 43709, '\P{^LOE=-_N}', "");
Expect(0, 43708, '\p{LOE=-_N}', "");
Expect(1, 43708, '\p{^LOE=-_N}', "");
Expect(1, 43708, '\P{LOE=-_N}', "");
Expect(0, 43708, '\P{^LOE=-_N}', "");
Error('\p{Is_Logical_Order_Exception:    F:=}');
Error('\P{Is_Logical_Order_Exception:    F:=}');
Expect(1, 43709, '\p{Is_Logical_Order_Exception=f}', "");
Expect(0, 43709, '\p{^Is_Logical_Order_Exception=f}', "");
Expect(0, 43709, '\P{Is_Logical_Order_Exception=f}', "");
Expect(1, 43709, '\P{^Is_Logical_Order_Exception=f}', "");
Expect(0, 43708, '\p{Is_Logical_Order_Exception=f}', "");
Expect(1, 43708, '\p{^Is_Logical_Order_Exception=f}', "");
Expect(1, 43708, '\P{Is_Logical_Order_Exception=f}', "");
Expect(0, 43708, '\P{^Is_Logical_Order_Exception=f}', "");
Expect(1, 43709, '\p{Is_Logical_Order_Exception=--F}', "");
Expect(0, 43709, '\p{^Is_Logical_Order_Exception=--F}', "");
Expect(0, 43709, '\P{Is_Logical_Order_Exception=--F}', "");
Expect(1, 43709, '\P{^Is_Logical_Order_Exception=--F}', "");
Expect(0, 43708, '\p{Is_Logical_Order_Exception=--F}', "");
Expect(1, 43708, '\p{^Is_Logical_Order_Exception=--F}', "");
Expect(1, 43708, '\P{Is_Logical_Order_Exception=--F}', "");
Expect(0, 43708, '\P{^Is_Logical_Order_Exception=--F}', "");
Error('\p{Is_LOE= false/a/}');
Error('\P{Is_LOE= false/a/}');
Expect(1, 43709, '\p{Is_LOE=false}', "");
Expect(0, 43709, '\p{^Is_LOE=false}', "");
Expect(0, 43709, '\P{Is_LOE=false}', "");
Expect(1, 43709, '\P{^Is_LOE=false}', "");
Expect(0, 43708, '\p{Is_LOE=false}', "");
Expect(1, 43708, '\p{^Is_LOE=false}', "");
Expect(1, 43708, '\P{Is_LOE=false}', "");
Expect(0, 43708, '\P{^Is_LOE=false}', "");
Expect(1, 43709, '\p{Is_LOE=--false}', "");
Expect(0, 43709, '\p{^Is_LOE=--false}', "");
Expect(0, 43709, '\P{Is_LOE=--false}', "");
Expect(1, 43709, '\P{^Is_LOE=--false}', "");
Expect(0, 43708, '\p{Is_LOE=--false}', "");
Expect(1, 43708, '\p{^Is_LOE=--false}', "");
Expect(1, 43708, '\P{Is_LOE=--false}', "");
Expect(0, 43708, '\P{^Is_LOE=--false}', "");
Error('\p{Logical_Order_Exception::=_ Yes}');
Error('\P{Logical_Order_Exception::=_ Yes}');
Expect(1, 43708, '\p{Logical_Order_Exception: yes}', "");
Expect(0, 43708, '\p{^Logical_Order_Exception: yes}', "");
Expect(0, 43708, '\P{Logical_Order_Exception: yes}', "");
Expect(1, 43708, '\P{^Logical_Order_Exception: yes}', "");
Expect(0, 43709, '\p{Logical_Order_Exception: yes}', "");
Expect(1, 43709, '\p{^Logical_Order_Exception: yes}', "");
Expect(1, 43709, '\P{Logical_Order_Exception: yes}', "");
Expect(0, 43709, '\P{^Logical_Order_Exception: yes}', "");
Expect(1, 43708, '\p{Logical_Order_Exception: 	yes}', "");
Expect(0, 43708, '\p{^Logical_Order_Exception: 	yes}', "");
Expect(0, 43708, '\P{Logical_Order_Exception: 	yes}', "");
Expect(1, 43708, '\P{^Logical_Order_Exception: 	yes}', "");
Expect(0, 43709, '\p{Logical_Order_Exception: 	yes}', "");
Expect(1, 43709, '\p{^Logical_Order_Exception: 	yes}', "");
Expect(1, 43709, '\P{Logical_Order_Exception: 	yes}', "");
Expect(0, 43709, '\P{^Logical_Order_Exception: 	yes}', "");
Error('\p{LOE=:= _Y}');
Error('\P{LOE=:= _Y}');
Expect(1, 43708, '\p{LOE=y}', "");
Expect(0, 43708, '\p{^LOE=y}', "");
Expect(0, 43708, '\P{LOE=y}', "");
Expect(1, 43708, '\P{^LOE=y}', "");
Expect(0, 43709, '\p{LOE=y}', "");
Expect(1, 43709, '\p{^LOE=y}', "");
Expect(1, 43709, '\P{LOE=y}', "");
Expect(0, 43709, '\P{^LOE=y}', "");
Expect(1, 43708, '\p{LOE=	y}', "");
Expect(0, 43708, '\p{^LOE=	y}', "");
Expect(0, 43708, '\P{LOE=	y}', "");
Expect(1, 43708, '\P{^LOE=	y}', "");
Expect(0, 43709, '\p{LOE=	y}', "");
Expect(1, 43709, '\p{^LOE=	y}', "");
Expect(1, 43709, '\P{LOE=	y}', "");
Expect(0, 43709, '\P{^LOE=	y}', "");
Error('\p{Is_Logical_Order_Exception=__t/a/}');
Error('\P{Is_Logical_Order_Exception=__t/a/}');
Expect(1, 43708, '\p{Is_Logical_Order_Exception=t}', "");
Expect(0, 43708, '\p{^Is_Logical_Order_Exception=t}', "");
Expect(0, 43708, '\P{Is_Logical_Order_Exception=t}', "");
Expect(1, 43708, '\P{^Is_Logical_Order_Exception=t}', "");
Expect(0, 43709, '\p{Is_Logical_Order_Exception=t}', "");
Expect(1, 43709, '\p{^Is_Logical_Order_Exception=t}', "");
Expect(1, 43709, '\P{Is_Logical_Order_Exception=t}', "");
Expect(0, 43709, '\P{^Is_Logical_Order_Exception=t}', "");
Expect(1, 43708, '\p{Is_Logical_Order_Exception=		T}', "");
Expect(0, 43708, '\p{^Is_Logical_Order_Exception=		T}', "");
Expect(0, 43708, '\P{Is_Logical_Order_Exception=		T}', "");
Expect(1, 43708, '\P{^Is_Logical_Order_Exception=		T}', "");
Expect(0, 43709, '\p{Is_Logical_Order_Exception=		T}', "");
Expect(1, 43709, '\p{^Is_Logical_Order_Exception=		T}', "");
Expect(1, 43709, '\P{Is_Logical_Order_Exception=		T}', "");
Expect(0, 43709, '\P{^Is_Logical_Order_Exception=		T}', "");
Error('\p{Is_LOE: _-TRUE/a/}');
Error('\P{Is_LOE: _-TRUE/a/}');
Expect(1, 43708, '\p{Is_LOE=true}', "");
Expect(0, 43708, '\p{^Is_LOE=true}', "");
Expect(0, 43708, '\P{Is_LOE=true}', "");
Expect(1, 43708, '\P{^Is_LOE=true}', "");
Expect(0, 43709, '\p{Is_LOE=true}', "");
Expect(1, 43709, '\p{^Is_LOE=true}', "");
Expect(1, 43709, '\P{Is_LOE=true}', "");
Expect(0, 43709, '\P{^Is_LOE=true}', "");
Expect(1, 43708, '\p{Is_LOE:  TRUE}', "");
Expect(0, 43708, '\p{^Is_LOE:  TRUE}', "");
Expect(0, 43708, '\P{Is_LOE:  TRUE}', "");
Expect(1, 43708, '\P{^Is_LOE:  TRUE}', "");
Expect(0, 43709, '\p{Is_LOE:  TRUE}', "");
Expect(1, 43709, '\p{^Is_LOE:  TRUE}', "");
Expect(1, 43709, '\P{Is_LOE:  TRUE}', "");
Expect(0, 43709, '\P{^Is_LOE:  TRUE}', "");
Error('\p{Lowercase=NO:=}');
Error('\P{Lowercase=NO:=}');
Expect(1, 120780, '\p{Lowercase=no}', "");
Expect(0, 120780, '\p{^Lowercase=no}', "");
Expect(0, 120780, '\P{Lowercase=no}', "");
Expect(1, 120780, '\P{^Lowercase=no}', "");
Expect(0, 120779, '\p{Lowercase=no}', "");
Expect(1, 120779, '\p{^Lowercase=no}', "");
Expect(1, 120779, '\P{Lowercase=no}', "");
Expect(0, 120779, '\P{^Lowercase=no}', "");
Expect(1, 120780, '\p{Lowercase: _	no}', "");
Expect(0, 120780, '\p{^Lowercase: _	no}', "");
Expect(0, 120780, '\P{Lowercase: _	no}', "");
Expect(1, 120780, '\P{^Lowercase: _	no}', "");
Expect(0, 120779, '\p{Lowercase: _	no}', "");
Expect(1, 120779, '\p{^Lowercase: _	no}', "");
Expect(1, 120779, '\P{Lowercase: _	no}', "");
Expect(0, 120779, '\P{^Lowercase: _	no}', "");
Error('\p{Lower=	 n:=}');
Error('\P{Lower=	 n:=}');
Expect(1, 120780, '\p{Lower=n}', "");
Expect(0, 120780, '\p{^Lower=n}', "");
Expect(0, 120780, '\P{Lower=n}', "");
Expect(1, 120780, '\P{^Lower=n}', "");
Expect(0, 120779, '\p{Lower=n}', "");
Expect(1, 120779, '\p{^Lower=n}', "");
Expect(1, 120779, '\P{Lower=n}', "");
Expect(0, 120779, '\P{^Lower=n}', "");
Expect(1, 120780, '\p{Lower= N}', "");
Expect(0, 120780, '\p{^Lower= N}', "");
Expect(0, 120780, '\P{Lower= N}', "");
Expect(1, 120780, '\P{^Lower= N}', "");
Expect(0, 120779, '\p{Lower= N}', "");
Expect(1, 120779, '\p{^Lower= N}', "");
Expect(1, 120779, '\P{Lower= N}', "");
Expect(0, 120779, '\P{^Lower= N}', "");
Error('\p{Is_Lowercase:   _F:=}');
Error('\P{Is_Lowercase:   _F:=}');
Expect(1, 120780, '\p{Is_Lowercase=f}', "");
Expect(0, 120780, '\p{^Is_Lowercase=f}', "");
Expect(0, 120780, '\P{Is_Lowercase=f}', "");
Expect(1, 120780, '\P{^Is_Lowercase=f}', "");
Expect(0, 120779, '\p{Is_Lowercase=f}', "");
Expect(1, 120779, '\p{^Is_Lowercase=f}', "");
Expect(1, 120779, '\P{Is_Lowercase=f}', "");
Expect(0, 120779, '\P{^Is_Lowercase=f}', "");
Expect(1, 120780, '\p{Is_Lowercase=_ F}', "");
Expect(0, 120780, '\p{^Is_Lowercase=_ F}', "");
Expect(0, 120780, '\P{Is_Lowercase=_ F}', "");
Expect(1, 120780, '\P{^Is_Lowercase=_ F}', "");
Expect(0, 120779, '\p{Is_Lowercase=_ F}', "");
Expect(1, 120779, '\p{^Is_Lowercase=_ F}', "");
Expect(1, 120779, '\P{Is_Lowercase=_ F}', "");
Expect(0, 120779, '\P{^Is_Lowercase=_ F}', "");
Error('\p{Is_Lower=_/a/False}');
Error('\P{Is_Lower=_/a/False}');
Expect(1, 120780, '\p{Is_Lower:   false}', "");
Expect(0, 120780, '\p{^Is_Lower:   false}', "");
Expect(0, 120780, '\P{Is_Lower:   false}', "");
Expect(1, 120780, '\P{^Is_Lower:   false}', "");
Expect(0, 120779, '\p{Is_Lower:   false}', "");
Expect(1, 120779, '\p{^Is_Lower:   false}', "");
Expect(1, 120779, '\P{Is_Lower:   false}', "");
Expect(0, 120779, '\P{^Is_Lower:   false}', "");
Expect(1, 120780, '\p{Is_Lower=-False}', "");
Expect(0, 120780, '\p{^Is_Lower=-False}', "");
Expect(0, 120780, '\P{Is_Lower=-False}', "");
Expect(1, 120780, '\P{^Is_Lower=-False}', "");
Expect(0, 120779, '\p{Is_Lower=-False}', "");
Expect(1, 120779, '\p{^Is_Lower=-False}', "");
Expect(1, 120779, '\P{Is_Lower=-False}', "");
Expect(0, 120779, '\P{^Is_Lower=-False}', "");
Error('\p{Lowercase=:= -YES}');
Error('\P{Lowercase=:= -YES}');
Expect(1, 120779, '\p{Lowercase:yes}', "");
Expect(0, 120779, '\p{^Lowercase:yes}', "");
Expect(0, 120779, '\P{Lowercase:yes}', "");
Expect(1, 120779, '\P{^Lowercase:yes}', "");
Expect(0, 120780, '\p{Lowercase:yes}', "");
Expect(1, 120780, '\p{^Lowercase:yes}', "");
Expect(1, 120780, '\P{Lowercase:yes}', "");
Expect(0, 120780, '\P{^Lowercase:yes}', "");
Expect(1, 120779, '\p{Lowercase=		YES}', "");
Expect(0, 120779, '\p{^Lowercase=		YES}', "");
Expect(0, 120779, '\P{Lowercase=		YES}', "");
Expect(1, 120779, '\P{^Lowercase=		YES}', "");
Expect(0, 120780, '\p{Lowercase=		YES}', "");
Expect(1, 120780, '\p{^Lowercase=		YES}', "");
Expect(1, 120780, '\P{Lowercase=		YES}', "");
Expect(0, 120780, '\P{^Lowercase=		YES}', "");
Error('\p{Lower=:=-	Y}');
Error('\P{Lower=:=-	Y}');
Expect(1, 120779, '\p{Lower=y}', "");
Expect(0, 120779, '\p{^Lower=y}', "");
Expect(0, 120779, '\P{Lower=y}', "");
Expect(1, 120779, '\P{^Lower=y}', "");
Expect(0, 120780, '\p{Lower=y}', "");
Expect(1, 120780, '\p{^Lower=y}', "");
Expect(1, 120780, '\P{Lower=y}', "");
Expect(0, 120780, '\P{^Lower=y}', "");
Expect(1, 120779, '\p{Lower= -Y}', "");
Expect(0, 120779, '\p{^Lower= -Y}', "");
Expect(0, 120779, '\P{Lower= -Y}', "");
Expect(1, 120779, '\P{^Lower= -Y}', "");
Expect(0, 120780, '\p{Lower= -Y}', "");
Expect(1, 120780, '\p{^Lower= -Y}', "");
Expect(1, 120780, '\P{Lower= -Y}', "");
Expect(0, 120780, '\P{^Lower= -Y}', "");
Error('\p{Is_Lowercase=_ t/a/}');
Error('\P{Is_Lowercase=_ t/a/}');
Expect(1, 120779, '\p{Is_Lowercase=t}', "");
Expect(0, 120779, '\p{^Is_Lowercase=t}', "");
Expect(0, 120779, '\P{Is_Lowercase=t}', "");
Expect(1, 120779, '\P{^Is_Lowercase=t}', "");
Expect(0, 120780, '\p{Is_Lowercase=t}', "");
Expect(1, 120780, '\p{^Is_Lowercase=t}', "");
Expect(1, 120780, '\P{Is_Lowercase=t}', "");
Expect(0, 120780, '\P{^Is_Lowercase=t}', "");
Expect(1, 120779, '\p{Is_Lowercase= -T}', "");
Expect(0, 120779, '\p{^Is_Lowercase= -T}', "");
Expect(0, 120779, '\P{Is_Lowercase= -T}', "");
Expect(1, 120779, '\P{^Is_Lowercase= -T}', "");
Expect(0, 120780, '\p{Is_Lowercase= -T}', "");
Expect(1, 120780, '\p{^Is_Lowercase= -T}', "");
Expect(1, 120780, '\P{Is_Lowercase= -T}', "");
Expect(0, 120780, '\P{^Is_Lowercase= -T}', "");
Error('\p{Is_Lower=	:=true}');
Error('\P{Is_Lower=	:=true}');
Expect(1, 120779, '\p{Is_Lower=true}', "");
Expect(0, 120779, '\p{^Is_Lower=true}', "");
Expect(0, 120779, '\P{Is_Lower=true}', "");
Expect(1, 120779, '\P{^Is_Lower=true}', "");
Expect(0, 120780, '\p{Is_Lower=true}', "");
Expect(1, 120780, '\p{^Is_Lower=true}', "");
Expect(1, 120780, '\P{Is_Lower=true}', "");
Expect(0, 120780, '\P{^Is_Lower=true}', "");
Expect(1, 120779, '\p{Is_Lower=-TRUE}', "");
Expect(0, 120779, '\p{^Is_Lower=-TRUE}', "");
Expect(0, 120779, '\P{Is_Lower=-TRUE}', "");
Expect(1, 120779, '\P{^Is_Lower=-TRUE}', "");
Expect(0, 120780, '\p{Is_Lower=-TRUE}', "");
Expect(1, 120780, '\p{^Is_Lower=-TRUE}', "");
Expect(1, 120780, '\P{Is_Lower=-TRUE}', "");
Expect(0, 120780, '\P{^Is_Lower=-TRUE}', "");
Error('\p{Math=-/a/No}');
Error('\P{Math=-/a/No}');
Expect(1, 126706, '\p{Math=no}', "");
Expect(0, 126706, '\p{^Math=no}', "");
Expect(0, 126706, '\P{Math=no}', "");
Expect(1, 126706, '\P{^Math=no}', "");
Expect(0, 126705, '\p{Math=no}', "");
Expect(1, 126705, '\p{^Math=no}', "");
Expect(1, 126705, '\P{Math=no}', "");
Expect(0, 126705, '\P{^Math=no}', "");
Expect(1, 126706, '\p{Math=- No}', "");
Expect(0, 126706, '\p{^Math=- No}', "");
Expect(0, 126706, '\P{Math=- No}', "");
Expect(1, 126706, '\P{^Math=- No}', "");
Expect(0, 126705, '\p{Math=- No}', "");
Expect(1, 126705, '\p{^Math=- No}', "");
Expect(1, 126705, '\P{Math=- No}', "");
Expect(0, 126705, '\P{^Math=- No}', "");
Error('\p{Is_Math:   :=N}');
Error('\P{Is_Math:   :=N}');
Expect(1, 126706, '\p{Is_Math=n}', "");
Expect(0, 126706, '\p{^Is_Math=n}', "");
Expect(0, 126706, '\P{Is_Math=n}', "");
Expect(1, 126706, '\P{^Is_Math=n}', "");
Expect(0, 126705, '\p{Is_Math=n}', "");
Expect(1, 126705, '\p{^Is_Math=n}', "");
Expect(1, 126705, '\P{Is_Math=n}', "");
Expect(0, 126705, '\P{^Is_Math=n}', "");
Expect(1, 126706, '\p{Is_Math=__N}', "");
Expect(0, 126706, '\p{^Is_Math=__N}', "");
Expect(0, 126706, '\P{Is_Math=__N}', "");
Expect(1, 126706, '\P{^Is_Math=__N}', "");
Expect(0, 126705, '\p{Is_Math=__N}', "");
Expect(1, 126705, '\p{^Is_Math=__N}', "");
Expect(1, 126705, '\P{Is_Math=__N}', "");
Expect(0, 126705, '\P{^Is_Math=__N}', "");
Error('\p{Math= f:=}');
Error('\P{Math= f:=}');
Expect(1, 126706, '\p{Math=f}', "");
Expect(0, 126706, '\p{^Math=f}', "");
Expect(0, 126706, '\P{Math=f}', "");
Expect(1, 126706, '\P{^Math=f}', "");
Expect(0, 126705, '\p{Math=f}', "");
Expect(1, 126705, '\p{^Math=f}', "");
Expect(1, 126705, '\P{Math=f}', "");
Expect(0, 126705, '\P{^Math=f}', "");
Expect(1, 126706, '\p{Math:  F}', "");
Expect(0, 126706, '\p{^Math:  F}', "");
Expect(0, 126706, '\P{Math:  F}', "");
Expect(1, 126706, '\P{^Math:  F}', "");
Expect(0, 126705, '\p{Math:  F}', "");
Expect(1, 126705, '\p{^Math:  F}', "");
Expect(1, 126705, '\P{Math:  F}', "");
Expect(0, 126705, '\P{^Math:  F}', "");
Error('\p{Is_Math: /a/ false}');
Error('\P{Is_Math: /a/ false}');
Expect(1, 126706, '\p{Is_Math=false}', "");
Expect(0, 126706, '\p{^Is_Math=false}', "");
Expect(0, 126706, '\P{Is_Math=false}', "");
Expect(1, 126706, '\P{^Is_Math=false}', "");
Expect(0, 126705, '\p{Is_Math=false}', "");
Expect(1, 126705, '\p{^Is_Math=false}', "");
Expect(1, 126705, '\P{Is_Math=false}', "");
Expect(0, 126705, '\P{^Is_Math=false}', "");
Expect(1, 126706, '\p{Is_Math=	FALSE}', "");
Expect(0, 126706, '\p{^Is_Math=	FALSE}', "");
Expect(0, 126706, '\P{Is_Math=	FALSE}', "");
Expect(1, 126706, '\P{^Is_Math=	FALSE}', "");
Expect(0, 126705, '\p{Is_Math=	FALSE}', "");
Expect(1, 126705, '\p{^Is_Math=	FALSE}', "");
Expect(1, 126705, '\P{Is_Math=	FALSE}', "");
Expect(0, 126705, '\P{^Is_Math=	FALSE}', "");
Error('\p{Math::=  Yes}');
Error('\P{Math::=  Yes}');
Expect(1, 126705, '\p{Math=yes}', "");
Expect(0, 126705, '\p{^Math=yes}', "");
Expect(0, 126705, '\P{Math=yes}', "");
Expect(1, 126705, '\P{^Math=yes}', "");
Expect(0, 126706, '\p{Math=yes}', "");
Expect(1, 126706, '\p{^Math=yes}', "");
Expect(1, 126706, '\P{Math=yes}', "");
Expect(0, 126706, '\P{^Math=yes}', "");
Expect(1, 126705, '\p{Math= Yes}', "");
Expect(0, 126705, '\p{^Math= Yes}', "");
Expect(0, 126705, '\P{Math= Yes}', "");
Expect(1, 126705, '\P{^Math= Yes}', "");
Expect(0, 126706, '\p{Math= Yes}', "");
Expect(1, 126706, '\p{^Math= Yes}', "");
Expect(1, 126706, '\P{Math= Yes}', "");
Expect(0, 126706, '\P{^Math= Yes}', "");
Error('\p{Is_Math=/a/_Y}');
Error('\P{Is_Math=/a/_Y}');
Expect(1, 126705, '\p{Is_Math=y}', "");
Expect(0, 126705, '\p{^Is_Math=y}', "");
Expect(0, 126705, '\P{Is_Math=y}', "");
Expect(1, 126705, '\P{^Is_Math=y}', "");
Expect(0, 126706, '\p{Is_Math=y}', "");
Expect(1, 126706, '\p{^Is_Math=y}', "");
Expect(1, 126706, '\P{Is_Math=y}', "");
Expect(0, 126706, '\P{^Is_Math=y}', "");
Expect(1, 126705, '\p{Is_Math= 	Y}', "");
Expect(0, 126705, '\p{^Is_Math= 	Y}', "");
Expect(0, 126705, '\P{Is_Math= 	Y}', "");
Expect(1, 126705, '\P{^Is_Math= 	Y}', "");
Expect(0, 126706, '\p{Is_Math= 	Y}', "");
Expect(1, 126706, '\p{^Is_Math= 	Y}', "");
Expect(1, 126706, '\P{Is_Math= 	Y}', "");
Expect(0, 126706, '\P{^Is_Math= 	Y}', "");
Error('\p{Math: /a/T}');
Error('\P{Math: /a/T}');
Expect(1, 126705, '\p{Math=t}', "");
Expect(0, 126705, '\p{^Math=t}', "");
Expect(0, 126705, '\P{Math=t}', "");
Expect(1, 126705, '\P{^Math=t}', "");
Expect(0, 126706, '\p{Math=t}', "");
Expect(1, 126706, '\p{^Math=t}', "");
Expect(1, 126706, '\P{Math=t}', "");
Expect(0, 126706, '\P{^Math=t}', "");
Expect(1, 126705, '\p{Math=		t}', "");
Expect(0, 126705, '\p{^Math=		t}', "");
Expect(0, 126705, '\P{Math=		t}', "");
Expect(1, 126705, '\P{^Math=		t}', "");
Expect(0, 126706, '\p{Math=		t}', "");
Expect(1, 126706, '\p{^Math=		t}', "");
Expect(1, 126706, '\P{Math=		t}', "");
Expect(0, 126706, '\P{^Math=		t}', "");
Error('\p{Is_Math: /a/	_True}');
Error('\P{Is_Math: /a/	_True}');
Expect(1, 126705, '\p{Is_Math=true}', "");
Expect(0, 126705, '\p{^Is_Math=true}', "");
Expect(0, 126705, '\P{Is_Math=true}', "");
Expect(1, 126705, '\P{^Is_Math=true}', "");
Expect(0, 126706, '\p{Is_Math=true}', "");
Expect(1, 126706, '\p{^Is_Math=true}', "");
Expect(1, 126706, '\P{Is_Math=true}', "");
Expect(0, 126706, '\P{^Is_Math=true}', "");
Expect(1, 126705, '\p{Is_Math=_True}', "");
Expect(0, 126705, '\p{^Is_Math=_True}', "");
Expect(0, 126705, '\P{Is_Math=_True}', "");
Expect(1, 126705, '\P{^Is_Math=_True}', "");
Expect(0, 126706, '\p{Is_Math=_True}', "");
Expect(1, 126706, '\p{^Is_Math=_True}', "");
Expect(1, 126706, '\P{Is_Math=_True}', "");
Expect(0, 126706, '\P{^Is_Math=_True}', "");
Error('\p{Noncharacter_Code_Point=	:=No}');
Error('\P{Noncharacter_Code_Point=	:=No}');
Expect(1, 1114109, '\p{Noncharacter_Code_Point=no}', "");
Expect(0, 1114109, '\p{^Noncharacter_Code_Point=no}', "");
Expect(0, 1114109, '\P{Noncharacter_Code_Point=no}', "");
Expect(1, 1114109, '\P{^Noncharacter_Code_Point=no}', "");
Expect(0, 1114111, '\p{Noncharacter_Code_Point=no}', "");
Expect(1, 1114111, '\p{^Noncharacter_Code_Point=no}', "");
Expect(1, 1114111, '\P{Noncharacter_Code_Point=no}', "");
Expect(0, 1114111, '\P{^Noncharacter_Code_Point=no}', "");
Expect(1, 1114109, '\p{Noncharacter_Code_Point= NO}', "");
Expect(0, 1114109, '\p{^Noncharacter_Code_Point= NO}', "");
Expect(0, 1114109, '\P{Noncharacter_Code_Point= NO}', "");
Expect(1, 1114109, '\P{^Noncharacter_Code_Point= NO}', "");
Expect(0, 1114111, '\p{Noncharacter_Code_Point= NO}', "");
Expect(1, 1114111, '\p{^Noncharacter_Code_Point= NO}', "");
Expect(1, 1114111, '\P{Noncharacter_Code_Point= NO}', "");
Expect(0, 1114111, '\P{^Noncharacter_Code_Point= NO}', "");
Error('\p{NChar:	:=n}');
Error('\P{NChar:	:=n}');
Expect(1, 1114109, '\p{NChar:   n}', "");
Expect(0, 1114109, '\p{^NChar:   n}', "");
Expect(0, 1114109, '\P{NChar:   n}', "");
Expect(1, 1114109, '\P{^NChar:   n}', "");
Expect(0, 1114111, '\p{NChar:   n}', "");
Expect(1, 1114111, '\p{^NChar:   n}', "");
Expect(1, 1114111, '\P{NChar:   n}', "");
Expect(0, 1114111, '\P{^NChar:   n}', "");
Expect(1, 1114109, '\p{NChar=-N}', "");
Expect(0, 1114109, '\p{^NChar=-N}', "");
Expect(0, 1114109, '\P{NChar=-N}', "");
Expect(1, 1114109, '\P{^NChar=-N}', "");
Expect(0, 1114111, '\p{NChar=-N}', "");
Expect(1, 1114111, '\p{^NChar=-N}', "");
Expect(1, 1114111, '\P{NChar=-N}', "");
Expect(0, 1114111, '\P{^NChar=-N}', "");
Error('\p{Is_Noncharacter_Code_Point=/a/F}');
Error('\P{Is_Noncharacter_Code_Point=/a/F}');
Expect(1, 1114109, '\p{Is_Noncharacter_Code_Point=f}', "");
Expect(0, 1114109, '\p{^Is_Noncharacter_Code_Point=f}', "");
Expect(0, 1114109, '\P{Is_Noncharacter_Code_Point=f}', "");
Expect(1, 1114109, '\P{^Is_Noncharacter_Code_Point=f}', "");
Expect(0, 1114111, '\p{Is_Noncharacter_Code_Point=f}', "");
Expect(1, 1114111, '\p{^Is_Noncharacter_Code_Point=f}', "");
Expect(1, 1114111, '\P{Is_Noncharacter_Code_Point=f}', "");
Expect(0, 1114111, '\P{^Is_Noncharacter_Code_Point=f}', "");
Expect(1, 1114109, '\p{Is_Noncharacter_Code_Point:   _	F}', "");
Expect(0, 1114109, '\p{^Is_Noncharacter_Code_Point:   _	F}', "");
Expect(0, 1114109, '\P{Is_Noncharacter_Code_Point:   _	F}', "");
Expect(1, 1114109, '\P{^Is_Noncharacter_Code_Point:   _	F}', "");
Expect(0, 1114111, '\p{Is_Noncharacter_Code_Point:   _	F}', "");
Expect(1, 1114111, '\p{^Is_Noncharacter_Code_Point:   _	F}', "");
Expect(1, 1114111, '\P{Is_Noncharacter_Code_Point:   _	F}', "");
Expect(0, 1114111, '\P{^Is_Noncharacter_Code_Point:   _	F}', "");
Error('\p{Is_NChar=:=FALSE}');
Error('\P{Is_NChar=:=FALSE}');
Expect(1, 1114109, '\p{Is_NChar=false}', "");
Expect(0, 1114109, '\p{^Is_NChar=false}', "");
Expect(0, 1114109, '\P{Is_NChar=false}', "");
Expect(1, 1114109, '\P{^Is_NChar=false}', "");
Expect(0, 1114111, '\p{Is_NChar=false}', "");
Expect(1, 1114111, '\p{^Is_NChar=false}', "");
Expect(1, 1114111, '\P{Is_NChar=false}', "");
Expect(0, 1114111, '\P{^Is_NChar=false}', "");
Expect(1, 1114109, '\p{Is_NChar:False}', "");
Expect(0, 1114109, '\p{^Is_NChar:False}', "");
Expect(0, 1114109, '\P{Is_NChar:False}', "");
Expect(1, 1114109, '\P{^Is_NChar:False}', "");
Expect(0, 1114111, '\p{Is_NChar:False}', "");
Expect(1, 1114111, '\p{^Is_NChar:False}', "");
Expect(1, 1114111, '\P{Is_NChar:False}', "");
Expect(0, 1114111, '\P{^Is_NChar:False}', "");
Error('\p{Noncharacter_Code_Point=_/a/Yes}');
Error('\P{Noncharacter_Code_Point=_/a/Yes}');
Expect(1, 1114111, '\p{Noncharacter_Code_Point=yes}', "");
Expect(0, 1114111, '\p{^Noncharacter_Code_Point=yes}', "");
Expect(0, 1114111, '\P{Noncharacter_Code_Point=yes}', "");
Expect(1, 1114111, '\P{^Noncharacter_Code_Point=yes}', "");
Expect(0, 1114109, '\p{Noncharacter_Code_Point=yes}', "");
Expect(1, 1114109, '\p{^Noncharacter_Code_Point=yes}', "");
Expect(1, 1114109, '\P{Noncharacter_Code_Point=yes}', "");
Expect(0, 1114109, '\P{^Noncharacter_Code_Point=yes}', "");
Expect(1, 1114111, '\p{Noncharacter_Code_Point=	 YES}', "");
Expect(0, 1114111, '\p{^Noncharacter_Code_Point=	 YES}', "");
Expect(0, 1114111, '\P{Noncharacter_Code_Point=	 YES}', "");
Expect(1, 1114111, '\P{^Noncharacter_Code_Point=	 YES}', "");
Expect(0, 1114109, '\p{Noncharacter_Code_Point=	 YES}', "");
Expect(1, 1114109, '\p{^Noncharacter_Code_Point=	 YES}', "");
Expect(1, 1114109, '\P{Noncharacter_Code_Point=	 YES}', "");
Expect(0, 1114109, '\P{^Noncharacter_Code_Point=	 YES}', "");
Error('\p{NChar=:=--y}');
Error('\P{NChar=:=--y}');
Expect(1, 1114111, '\p{NChar=y}', "");
Expect(0, 1114111, '\p{^NChar=y}', "");
Expect(0, 1114111, '\P{NChar=y}', "");
Expect(1, 1114111, '\P{^NChar=y}', "");
Expect(0, 1114109, '\p{NChar=y}', "");
Expect(1, 1114109, '\p{^NChar=y}', "");
Expect(1, 1114109, '\P{NChar=y}', "");
Expect(0, 1114109, '\P{^NChar=y}', "");
Expect(1, 1114111, '\p{NChar=_	Y}', "");
Expect(0, 1114111, '\p{^NChar=_	Y}', "");
Expect(0, 1114111, '\P{NChar=_	Y}', "");
Expect(1, 1114111, '\P{^NChar=_	Y}', "");
Expect(0, 1114109, '\p{NChar=_	Y}', "");
Expect(1, 1114109, '\p{^NChar=_	Y}', "");
Expect(1, 1114109, '\P{NChar=_	Y}', "");
Expect(0, 1114109, '\P{^NChar=_	Y}', "");
Error('\p{Is_Noncharacter_Code_Point:	/a/-T}');
Error('\P{Is_Noncharacter_Code_Point:	/a/-T}');
Expect(1, 1114111, '\p{Is_Noncharacter_Code_Point:   t}', "");
Expect(0, 1114111, '\p{^Is_Noncharacter_Code_Point:   t}', "");
Expect(0, 1114111, '\P{Is_Noncharacter_Code_Point:   t}', "");
Expect(1, 1114111, '\P{^Is_Noncharacter_Code_Point:   t}', "");
Expect(0, 1114109, '\p{Is_Noncharacter_Code_Point:   t}', "");
Expect(1, 1114109, '\p{^Is_Noncharacter_Code_Point:   t}', "");
Expect(1, 1114109, '\P{Is_Noncharacter_Code_Point:   t}', "");
Expect(0, 1114109, '\P{^Is_Noncharacter_Code_Point:   t}', "");
Expect(1, 1114111, '\p{Is_Noncharacter_Code_Point=-T}', "");
Expect(0, 1114111, '\p{^Is_Noncharacter_Code_Point=-T}', "");
Expect(0, 1114111, '\P{Is_Noncharacter_Code_Point=-T}', "");
Expect(1, 1114111, '\P{^Is_Noncharacter_Code_Point=-T}', "");
Expect(0, 1114109, '\p{Is_Noncharacter_Code_Point=-T}', "");
Expect(1, 1114109, '\p{^Is_Noncharacter_Code_Point=-T}', "");
Expect(1, 1114109, '\P{Is_Noncharacter_Code_Point=-T}', "");
Expect(0, 1114109, '\P{^Is_Noncharacter_Code_Point=-T}', "");
Error('\p{Is_NChar=-:=True}');
Error('\P{Is_NChar=-:=True}');
Expect(1, 1114111, '\p{Is_NChar: true}', "");
Expect(0, 1114111, '\p{^Is_NChar: true}', "");
Expect(0, 1114111, '\P{Is_NChar: true}', "");
Expect(1, 1114111, '\P{^Is_NChar: true}', "");
Expect(0, 1114109, '\p{Is_NChar: true}', "");
Expect(1, 1114109, '\p{^Is_NChar: true}', "");
Expect(1, 1114109, '\P{Is_NChar: true}', "");
Expect(0, 1114109, '\P{^Is_NChar: true}', "");
Expect(1, 1114111, '\p{Is_NChar=_TRUE}', "");
Expect(0, 1114111, '\p{^Is_NChar=_TRUE}', "");
Expect(0, 1114111, '\P{Is_NChar=_TRUE}', "");
Expect(1, 1114111, '\P{^Is_NChar=_TRUE}', "");
Expect(0, 1114109, '\p{Is_NChar=_TRUE}', "");
Expect(1, 1114109, '\p{^Is_NChar=_TRUE}', "");
Expect(1, 1114109, '\P{Is_NChar=_TRUE}', "");
Expect(0, 1114109, '\P{^Is_NChar=_TRUE}', "");
Error('\p{NFC_Quick_Check=- maybe:=}');
Error('\P{NFC_Quick_Check=- maybe:=}');
Expect(1, 69927, '\p{NFC_Quick_Check=maybe}', "");
Expect(0, 69927, '\p{^NFC_Quick_Check=maybe}', "");
Expect(0, 69927, '\P{NFC_Quick_Check=maybe}', "");
Expect(1, 69927, '\P{^NFC_Quick_Check=maybe}', "");
Expect(0, 69928, '\p{NFC_Quick_Check=maybe}', "");
Expect(1, 69928, '\p{^NFC_Quick_Check=maybe}', "");
Expect(1, 69928, '\P{NFC_Quick_Check=maybe}', "");
Expect(0, 69928, '\P{^NFC_Quick_Check=maybe}', "");
Expect(1, 69927, '\p{NFC_Quick_Check=	maybe}', "");
Expect(0, 69927, '\p{^NFC_Quick_Check=	maybe}', "");
Expect(0, 69927, '\P{NFC_Quick_Check=	maybe}', "");
Expect(1, 69927, '\P{^NFC_Quick_Check=	maybe}', "");
Expect(0, 69928, '\p{NFC_Quick_Check=	maybe}', "");
Expect(1, 69928, '\p{^NFC_Quick_Check=	maybe}', "");
Expect(1, 69928, '\P{NFC_Quick_Check=	maybe}', "");
Expect(0, 69928, '\P{^NFC_Quick_Check=	maybe}', "");
Error('\p{NFC_QC=-M:=}');
Error('\P{NFC_QC=-M:=}');
Expect(1, 69927, '\p{NFC_QC=m}', "");
Expect(0, 69927, '\p{^NFC_QC=m}', "");
Expect(0, 69927, '\P{NFC_QC=m}', "");
Expect(1, 69927, '\P{^NFC_QC=m}', "");
Expect(0, 69928, '\p{NFC_QC=m}', "");
Expect(1, 69928, '\p{^NFC_QC=m}', "");
Expect(1, 69928, '\P{NFC_QC=m}', "");
Expect(0, 69928, '\P{^NFC_QC=m}', "");
Expect(1, 69927, '\p{NFC_QC=-_M}', "");
Expect(0, 69927, '\p{^NFC_QC=-_M}', "");
Expect(0, 69927, '\P{NFC_QC=-_M}', "");
Expect(1, 69927, '\P{^NFC_QC=-_M}', "");
Expect(0, 69928, '\p{NFC_QC=-_M}', "");
Expect(1, 69928, '\p{^NFC_QC=-_M}', "");
Expect(1, 69928, '\P{NFC_QC=-_M}', "");
Expect(0, 69928, '\P{^NFC_QC=-_M}', "");
Error('\p{Is_NFC_Quick_Check=:=_	maybe}');
Error('\P{Is_NFC_Quick_Check=:=_	maybe}');
Expect(1, 69927, '\p{Is_NFC_Quick_Check=maybe}', "");
Expect(0, 69927, '\p{^Is_NFC_Quick_Check=maybe}', "");
Expect(0, 69927, '\P{Is_NFC_Quick_Check=maybe}', "");
Expect(1, 69927, '\P{^Is_NFC_Quick_Check=maybe}', "");
Expect(0, 69928, '\p{Is_NFC_Quick_Check=maybe}', "");
Expect(1, 69928, '\p{^Is_NFC_Quick_Check=maybe}', "");
Expect(1, 69928, '\P{Is_NFC_Quick_Check=maybe}', "");
Expect(0, 69928, '\P{^Is_NFC_Quick_Check=maybe}', "");
Expect(1, 69927, '\p{Is_NFC_Quick_Check=	maybe}', "");
Expect(0, 69927, '\p{^Is_NFC_Quick_Check=	maybe}', "");
Expect(0, 69927, '\P{Is_NFC_Quick_Check=	maybe}', "");
Expect(1, 69927, '\P{^Is_NFC_Quick_Check=	maybe}', "");
Expect(0, 69928, '\p{Is_NFC_Quick_Check=	maybe}', "");
Expect(1, 69928, '\p{^Is_NFC_Quick_Check=	maybe}', "");
Expect(1, 69928, '\P{Is_NFC_Quick_Check=	maybe}', "");
Expect(0, 69928, '\P{^Is_NFC_Quick_Check=	maybe}', "");
Error('\p{Is_NFC_QC=_/a/m}');
Error('\P{Is_NFC_QC=_/a/m}');
Expect(1, 69927, '\p{Is_NFC_QC:   m}', "");
Expect(0, 69927, '\p{^Is_NFC_QC:   m}', "");
Expect(0, 69927, '\P{Is_NFC_QC:   m}', "");
Expect(1, 69927, '\P{^Is_NFC_QC:   m}', "");
Expect(0, 69928, '\p{Is_NFC_QC:   m}', "");
Expect(1, 69928, '\p{^Is_NFC_QC:   m}', "");
Expect(1, 69928, '\P{Is_NFC_QC:   m}', "");
Expect(0, 69928, '\P{^Is_NFC_QC:   m}', "");
Expect(1, 69927, '\p{Is_NFC_QC=_	m}', "");
Expect(0, 69927, '\p{^Is_NFC_QC=_	m}', "");
Expect(0, 69927, '\P{Is_NFC_QC=_	m}', "");
Expect(1, 69927, '\P{^Is_NFC_QC=_	m}', "");
Expect(0, 69928, '\p{Is_NFC_QC=_	m}', "");
Expect(1, 69928, '\p{^Is_NFC_QC=_	m}', "");
Expect(1, 69928, '\P{Is_NFC_QC=_	m}', "");
Expect(0, 69928, '\P{^Is_NFC_QC=_	m}', "");
Error('\p{NFC_Quick_Check=No/a/}');
Error('\P{NFC_Quick_Check=No/a/}');
Expect(1, 195101, '\p{NFC_Quick_Check:   no}', "");
Expect(0, 195101, '\p{^NFC_Quick_Check:   no}', "");
Expect(0, 195101, '\P{NFC_Quick_Check:   no}', "");
Expect(1, 195101, '\P{^NFC_Quick_Check:   no}', "");
Expect(0, 195102, '\p{NFC_Quick_Check:   no}', "");
Expect(1, 195102, '\p{^NFC_Quick_Check:   no}', "");
Expect(1, 195102, '\P{NFC_Quick_Check:   no}', "");
Expect(0, 195102, '\P{^NFC_Quick_Check:   no}', "");
Expect(1, 195101, '\p{NFC_Quick_Check=	 NO}', "");
Expect(0, 195101, '\p{^NFC_Quick_Check=	 NO}', "");
Expect(0, 195101, '\P{NFC_Quick_Check=	 NO}', "");
Expect(1, 195101, '\P{^NFC_Quick_Check=	 NO}', "");
Expect(0, 195102, '\p{NFC_Quick_Check=	 NO}', "");
Expect(1, 195102, '\p{^NFC_Quick_Check=	 NO}', "");
Expect(1, 195102, '\P{NFC_Quick_Check=	 NO}', "");
Expect(0, 195102, '\P{^NFC_Quick_Check=	 NO}', "");
Error('\p{NFC_QC=	-N/a/}');
Error('\P{NFC_QC=	-N/a/}');
Expect(1, 195101, '\p{NFC_QC=n}', "");
Expect(0, 195101, '\p{^NFC_QC=n}', "");
Expect(0, 195101, '\P{NFC_QC=n}', "");
Expect(1, 195101, '\P{^NFC_QC=n}', "");
Expect(0, 195102, '\p{NFC_QC=n}', "");
Expect(1, 195102, '\p{^NFC_QC=n}', "");
Expect(1, 195102, '\P{NFC_QC=n}', "");
Expect(0, 195102, '\P{^NFC_QC=n}', "");
Expect(1, 195101, '\p{NFC_QC=_n}', "");
Expect(0, 195101, '\p{^NFC_QC=_n}', "");
Expect(0, 195101, '\P{NFC_QC=_n}', "");
Expect(1, 195101, '\P{^NFC_QC=_n}', "");
Expect(0, 195102, '\p{NFC_QC=_n}', "");
Expect(1, 195102, '\p{^NFC_QC=_n}', "");
Expect(1, 195102, '\P{NFC_QC=_n}', "");
Expect(0, 195102, '\P{^NFC_QC=_n}', "");
Error('\p{Is_NFC_Quick_Check=:=NO}');
Error('\P{Is_NFC_Quick_Check=:=NO}');
Expect(1, 195101, '\p{Is_NFC_Quick_Check=no}', "");
Expect(0, 195101, '\p{^Is_NFC_Quick_Check=no}', "");
Expect(0, 195101, '\P{Is_NFC_Quick_Check=no}', "");
Expect(1, 195101, '\P{^Is_NFC_Quick_Check=no}', "");
Expect(0, 195102, '\p{Is_NFC_Quick_Check=no}', "");
Expect(1, 195102, '\p{^Is_NFC_Quick_Check=no}', "");
Expect(1, 195102, '\P{Is_NFC_Quick_Check=no}', "");
Expect(0, 195102, '\P{^Is_NFC_Quick_Check=no}', "");
Expect(1, 195101, '\p{Is_NFC_Quick_Check= No}', "");
Expect(0, 195101, '\p{^Is_NFC_Quick_Check= No}', "");
Expect(0, 195101, '\P{Is_NFC_Quick_Check= No}', "");
Expect(1, 195101, '\P{^Is_NFC_Quick_Check= No}', "");
Expect(0, 195102, '\p{Is_NFC_Quick_Check= No}', "");
Expect(1, 195102, '\p{^Is_NFC_Quick_Check= No}', "");
Expect(1, 195102, '\P{Is_NFC_Quick_Check= No}', "");
Expect(0, 195102, '\P{^Is_NFC_Quick_Check= No}', "");
Error('\p{Is_NFC_QC:   N/a/}');
Error('\P{Is_NFC_QC:   N/a/}');
Expect(1, 195101, '\p{Is_NFC_QC=n}', "");
Expect(0, 195101, '\p{^Is_NFC_QC=n}', "");
Expect(0, 195101, '\P{Is_NFC_QC=n}', "");
Expect(1, 195101, '\P{^Is_NFC_QC=n}', "");
Expect(0, 195102, '\p{Is_NFC_QC=n}', "");
Expect(1, 195102, '\p{^Is_NFC_QC=n}', "");
Expect(1, 195102, '\P{Is_NFC_QC=n}', "");
Expect(0, 195102, '\P{^Is_NFC_QC=n}', "");
Expect(1, 195101, '\p{Is_NFC_QC=--N}', "");
Expect(0, 195101, '\p{^Is_NFC_QC=--N}', "");
Expect(0, 195101, '\P{Is_NFC_QC=--N}', "");
Expect(1, 195101, '\P{^Is_NFC_QC=--N}', "");
Expect(0, 195102, '\p{Is_NFC_QC=--N}', "");
Expect(1, 195102, '\p{^Is_NFC_QC=--N}', "");
Expect(1, 195102, '\P{Is_NFC_QC=--N}', "");
Expect(0, 195102, '\P{^Is_NFC_QC=--N}', "");
Error('\p{NFC_Quick_Check=-YES:=}');
Error('\P{NFC_Quick_Check=-YES:=}');
Expect(1, 195102, '\p{NFC_Quick_Check=yes}', "");
Expect(0, 195102, '\p{^NFC_Quick_Check=yes}', "");
Expect(0, 195102, '\P{NFC_Quick_Check=yes}', "");
Expect(1, 195102, '\P{^NFC_Quick_Check=yes}', "");
Expect(0, 195101, '\p{NFC_Quick_Check=yes}', "");
Expect(1, 195101, '\p{^NFC_Quick_Check=yes}', "");
Expect(1, 195101, '\P{NFC_Quick_Check=yes}', "");
Expect(0, 195101, '\P{^NFC_Quick_Check=yes}', "");
Expect(1, 195102, '\p{NFC_Quick_Check=	_YES}', "");
Expect(0, 195102, '\p{^NFC_Quick_Check=	_YES}', "");
Expect(0, 195102, '\P{NFC_Quick_Check=	_YES}', "");
Expect(1, 195102, '\P{^NFC_Quick_Check=	_YES}', "");
Expect(0, 195101, '\p{NFC_Quick_Check=	_YES}', "");
Expect(1, 195101, '\p{^NFC_Quick_Check=	_YES}', "");
Expect(1, 195101, '\P{NFC_Quick_Check=	_YES}', "");
Expect(0, 195101, '\P{^NFC_Quick_Check=	_YES}', "");
Error('\p{NFC_QC:   --Y:=}');
Error('\P{NFC_QC:   --Y:=}');
Expect(1, 195102, '\p{NFC_QC=y}', "");
Expect(0, 195102, '\p{^NFC_QC=y}', "");
Expect(0, 195102, '\P{NFC_QC=y}', "");
Expect(1, 195102, '\P{^NFC_QC=y}', "");
Expect(0, 195101, '\p{NFC_QC=y}', "");
Expect(1, 195101, '\p{^NFC_QC=y}', "");
Expect(1, 195101, '\P{NFC_QC=y}', "");
Expect(0, 195101, '\P{^NFC_QC=y}', "");
Expect(1, 195102, '\p{NFC_QC=	Y}', "");
Expect(0, 195102, '\p{^NFC_QC=	Y}', "");
Expect(0, 195102, '\P{NFC_QC=	Y}', "");
Expect(1, 195102, '\P{^NFC_QC=	Y}', "");
Expect(0, 195101, '\p{NFC_QC=	Y}', "");
Expect(1, 195101, '\p{^NFC_QC=	Y}', "");
Expect(1, 195101, '\P{NFC_QC=	Y}', "");
Expect(0, 195101, '\P{^NFC_QC=	Y}', "");
Error('\p{Is_NFC_Quick_Check=	 Yes/a/}');
Error('\P{Is_NFC_Quick_Check=	 Yes/a/}');
Expect(1, 195102, '\p{Is_NFC_Quick_Check=yes}', "");
Expect(0, 195102, '\p{^Is_NFC_Quick_Check=yes}', "");
Expect(0, 195102, '\P{Is_NFC_Quick_Check=yes}', "");
Expect(1, 195102, '\P{^Is_NFC_Quick_Check=yes}', "");
Expect(0, 195101, '\p{Is_NFC_Quick_Check=yes}', "");
Expect(1, 195101, '\p{^Is_NFC_Quick_Check=yes}', "");
Expect(1, 195101, '\P{Is_NFC_Quick_Check=yes}', "");
Expect(0, 195101, '\P{^Is_NFC_Quick_Check=yes}', "");
Expect(1, 195102, '\p{Is_NFC_Quick_Check=_Yes}', "");
Expect(0, 195102, '\p{^Is_NFC_Quick_Check=_Yes}', "");
Expect(0, 195102, '\P{Is_NFC_Quick_Check=_Yes}', "");
Expect(1, 195102, '\P{^Is_NFC_Quick_Check=_Yes}', "");
Expect(0, 195101, '\p{Is_NFC_Quick_Check=_Yes}', "");
Expect(1, 195101, '\p{^Is_NFC_Quick_Check=_Yes}', "");
Expect(1, 195101, '\P{Is_NFC_Quick_Check=_Yes}', "");
Expect(0, 195101, '\P{^Is_NFC_Quick_Check=_Yes}', "");
Error('\p{Is_NFC_QC=_ Y:=}');
Error('\P{Is_NFC_QC=_ Y:=}');
Expect(1, 195102, '\p{Is_NFC_QC:	y}', "");
Expect(0, 195102, '\p{^Is_NFC_QC:	y}', "");
Expect(0, 195102, '\P{Is_NFC_QC:	y}', "");
Expect(1, 195102, '\P{^Is_NFC_QC:	y}', "");
Expect(0, 195101, '\p{Is_NFC_QC:	y}', "");
Expect(1, 195101, '\p{^Is_NFC_QC:	y}', "");
Expect(1, 195101, '\P{Is_NFC_QC:	y}', "");
Expect(0, 195101, '\P{^Is_NFC_QC:	y}', "");
Expect(1, 195102, '\p{Is_NFC_QC=-	Y}', "");
Expect(0, 195102, '\p{^Is_NFC_QC=-	Y}', "");
Expect(0, 195102, '\P{Is_NFC_QC=-	Y}', "");
Expect(1, 195102, '\P{^Is_NFC_QC=-	Y}', "");
Expect(0, 195101, '\p{Is_NFC_QC=-	Y}', "");
Expect(1, 195101, '\p{^Is_NFC_QC=-	Y}', "");
Expect(1, 195101, '\P{Is_NFC_QC=-	Y}', "");
Expect(0, 195101, '\P{^Is_NFC_QC=-	Y}', "");
Error('\p{NFD_Quick_Check: -no:=}');
Error('\P{NFD_Quick_Check: -no:=}');
Expect(1, 195101, '\p{NFD_Quick_Check=no}', "");
Expect(0, 195101, '\p{^NFD_Quick_Check=no}', "");
Expect(0, 195101, '\P{NFD_Quick_Check=no}', "");
Expect(1, 195101, '\P{^NFD_Quick_Check=no}', "");
Expect(0, 195102, '\p{NFD_Quick_Check=no}', "");
Expect(1, 195102, '\p{^NFD_Quick_Check=no}', "");
Expect(1, 195102, '\P{NFD_Quick_Check=no}', "");
Expect(0, 195102, '\P{^NFD_Quick_Check=no}', "");
Expect(1, 195101, '\p{NFD_Quick_Check=	No}', "");
Expect(0, 195101, '\p{^NFD_Quick_Check=	No}', "");
Expect(0, 195101, '\P{NFD_Quick_Check=	No}', "");
Expect(1, 195101, '\P{^NFD_Quick_Check=	No}', "");
Expect(0, 195102, '\p{NFD_Quick_Check=	No}', "");
Expect(1, 195102, '\p{^NFD_Quick_Check=	No}', "");
Expect(1, 195102, '\P{NFD_Quick_Check=	No}', "");
Expect(0, 195102, '\P{^NFD_Quick_Check=	No}', "");
Error('\p{NFD_QC=:=n}');
Error('\P{NFD_QC=:=n}');
Expect(1, 195101, '\p{NFD_QC:n}', "");
Expect(0, 195101, '\p{^NFD_QC:n}', "");
Expect(0, 195101, '\P{NFD_QC:n}', "");
Expect(1, 195101, '\P{^NFD_QC:n}', "");
Expect(0, 195102, '\p{NFD_QC:n}', "");
Expect(1, 195102, '\p{^NFD_QC:n}', "");
Expect(1, 195102, '\P{NFD_QC:n}', "");
Expect(0, 195102, '\P{^NFD_QC:n}', "");
Expect(1, 195101, '\p{NFD_QC=	N}', "");
Expect(0, 195101, '\p{^NFD_QC=	N}', "");
Expect(0, 195101, '\P{NFD_QC=	N}', "");
Expect(1, 195101, '\P{^NFD_QC=	N}', "");
Expect(0, 195102, '\p{NFD_QC=	N}', "");
Expect(1, 195102, '\p{^NFD_QC=	N}', "");
Expect(1, 195102, '\P{NFD_QC=	N}', "");
Expect(0, 195102, '\P{^NFD_QC=	N}', "");
Error('\p{Is_NFD_Quick_Check: No:=}');
Error('\P{Is_NFD_Quick_Check: No:=}');
Expect(1, 195101, '\p{Is_NFD_Quick_Check=no}', "");
Expect(0, 195101, '\p{^Is_NFD_Quick_Check=no}', "");
Expect(0, 195101, '\P{Is_NFD_Quick_Check=no}', "");
Expect(1, 195101, '\P{^Is_NFD_Quick_Check=no}', "");
Expect(0, 195102, '\p{Is_NFD_Quick_Check=no}', "");
Expect(1, 195102, '\p{^Is_NFD_Quick_Check=no}', "");
Expect(1, 195102, '\P{Is_NFD_Quick_Check=no}', "");
Expect(0, 195102, '\P{^Is_NFD_Quick_Check=no}', "");
Expect(1, 195101, '\p{Is_NFD_Quick_Check=_NO}', "");
Expect(0, 195101, '\p{^Is_NFD_Quick_Check=_NO}', "");
Expect(0, 195101, '\P{Is_NFD_Quick_Check=_NO}', "");
Expect(1, 195101, '\P{^Is_NFD_Quick_Check=_NO}', "");
Expect(0, 195102, '\p{Is_NFD_Quick_Check=_NO}', "");
Expect(1, 195102, '\p{^Is_NFD_Quick_Check=_NO}', "");
Expect(1, 195102, '\P{Is_NFD_Quick_Check=_NO}', "");
Expect(0, 195102, '\P{^Is_NFD_Quick_Check=_NO}', "");
Error('\p{Is_NFD_QC= :=n}');
Error('\P{Is_NFD_QC= :=n}');
Expect(1, 195101, '\p{Is_NFD_QC=n}', "");
Expect(0, 195101, '\p{^Is_NFD_QC=n}', "");
Expect(0, 195101, '\P{Is_NFD_QC=n}', "");
Expect(1, 195101, '\P{^Is_NFD_QC=n}', "");
Expect(0, 195102, '\p{Is_NFD_QC=n}', "");
Expect(1, 195102, '\p{^Is_NFD_QC=n}', "");
Expect(1, 195102, '\P{Is_NFD_QC=n}', "");
Expect(0, 195102, '\P{^Is_NFD_QC=n}', "");
Expect(1, 195101, '\p{Is_NFD_QC= 	n}', "");
Expect(0, 195101, '\p{^Is_NFD_QC= 	n}', "");
Expect(0, 195101, '\P{Is_NFD_QC= 	n}', "");
Expect(1, 195101, '\P{^Is_NFD_QC= 	n}', "");
Expect(0, 195102, '\p{Is_NFD_QC= 	n}', "");
Expect(1, 195102, '\p{^Is_NFD_QC= 	n}', "");
Expect(1, 195102, '\P{Is_NFD_QC= 	n}', "");
Expect(0, 195102, '\P{^Is_NFD_QC= 	n}', "");
Error('\p{NFD_Quick_Check=_-Yes:=}');
Error('\P{NFD_Quick_Check=_-Yes:=}');
Expect(1, 195102, '\p{NFD_Quick_Check=yes}', "");
Expect(0, 195102, '\p{^NFD_Quick_Check=yes}', "");
Expect(0, 195102, '\P{NFD_Quick_Check=yes}', "");
Expect(1, 195102, '\P{^NFD_Quick_Check=yes}', "");
Expect(0, 195101, '\p{NFD_Quick_Check=yes}', "");
Expect(1, 195101, '\p{^NFD_Quick_Check=yes}', "");
Expect(1, 195101, '\P{NFD_Quick_Check=yes}', "");
Expect(0, 195101, '\P{^NFD_Quick_Check=yes}', "");
Expect(1, 195102, '\p{NFD_Quick_Check=YES}', "");
Expect(0, 195102, '\p{^NFD_Quick_Check=YES}', "");
Expect(0, 195102, '\P{NFD_Quick_Check=YES}', "");
Expect(1, 195102, '\P{^NFD_Quick_Check=YES}', "");
Expect(0, 195101, '\p{NFD_Quick_Check=YES}', "");
Expect(1, 195101, '\p{^NFD_Quick_Check=YES}', "");
Expect(1, 195101, '\P{NFD_Quick_Check=YES}', "");
Expect(0, 195101, '\P{^NFD_Quick_Check=YES}', "");
Error('\p{NFD_QC=_ Y:=}');
Error('\P{NFD_QC=_ Y:=}');
Expect(1, 195102, '\p{NFD_QC=y}', "");
Expect(0, 195102, '\p{^NFD_QC=y}', "");
Expect(0, 195102, '\P{NFD_QC=y}', "");
Expect(1, 195102, '\P{^NFD_QC=y}', "");
Expect(0, 195101, '\p{NFD_QC=y}', "");
Expect(1, 195101, '\p{^NFD_QC=y}', "");
Expect(1, 195101, '\P{NFD_QC=y}', "");
Expect(0, 195101, '\P{^NFD_QC=y}', "");
Expect(1, 195102, '\p{NFD_QC= -Y}', "");
Expect(0, 195102, '\p{^NFD_QC= -Y}', "");
Expect(0, 195102, '\P{NFD_QC= -Y}', "");
Expect(1, 195102, '\P{^NFD_QC= -Y}', "");
Expect(0, 195101, '\p{NFD_QC= -Y}', "");
Expect(1, 195101, '\p{^NFD_QC= -Y}', "");
Expect(1, 195101, '\P{NFD_QC= -Y}', "");
Expect(0, 195101, '\P{^NFD_QC= -Y}', "");
Error('\p{Is_NFD_Quick_Check=-/a/YES}');
Error('\P{Is_NFD_Quick_Check=-/a/YES}');
Expect(1, 195102, '\p{Is_NFD_Quick_Check=yes}', "");
Expect(0, 195102, '\p{^Is_NFD_Quick_Check=yes}', "");
Expect(0, 195102, '\P{Is_NFD_Quick_Check=yes}', "");
Expect(1, 195102, '\P{^Is_NFD_Quick_Check=yes}', "");
Expect(0, 195101, '\p{Is_NFD_Quick_Check=yes}', "");
Expect(1, 195101, '\p{^Is_NFD_Quick_Check=yes}', "");
Expect(1, 195101, '\P{Is_NFD_Quick_Check=yes}', "");
Expect(0, 195101, '\P{^Is_NFD_Quick_Check=yes}', "");
Expect(1, 195102, '\p{Is_NFD_Quick_Check=-Yes}', "");
Expect(0, 195102, '\p{^Is_NFD_Quick_Check=-Yes}', "");
Expect(0, 195102, '\P{Is_NFD_Quick_Check=-Yes}', "");
Expect(1, 195102, '\P{^Is_NFD_Quick_Check=-Yes}', "");
Expect(0, 195101, '\p{Is_NFD_Quick_Check=-Yes}', "");
Expect(1, 195101, '\p{^Is_NFD_Quick_Check=-Yes}', "");
Expect(1, 195101, '\P{Is_NFD_Quick_Check=-Yes}', "");
Expect(0, 195101, '\P{^Is_NFD_Quick_Check=-Yes}', "");
Error('\p{Is_NFD_QC=_-Y:=}');
Error('\P{Is_NFD_QC=_-Y:=}');
Expect(1, 195102, '\p{Is_NFD_QC=y}', "");
Expect(0, 195102, '\p{^Is_NFD_QC=y}', "");
Expect(0, 195102, '\P{Is_NFD_QC=y}', "");
Expect(1, 195102, '\P{^Is_NFD_QC=y}', "");
Expect(0, 195101, '\p{Is_NFD_QC=y}', "");
Expect(1, 195101, '\p{^Is_NFD_QC=y}', "");
Expect(1, 195101, '\P{Is_NFD_QC=y}', "");
Expect(0, 195101, '\P{^Is_NFD_QC=y}', "");
Expect(1, 195102, '\p{Is_NFD_QC: -Y}', "");
Expect(0, 195102, '\p{^Is_NFD_QC: -Y}', "");
Expect(0, 195102, '\P{Is_NFD_QC: -Y}', "");
Expect(1, 195102, '\P{^Is_NFD_QC: -Y}', "");
Expect(0, 195101, '\p{Is_NFD_QC: -Y}', "");
Expect(1, 195101, '\p{^Is_NFD_QC: -Y}', "");
Expect(1, 195101, '\P{Is_NFD_QC: -Y}', "");
Expect(0, 195101, '\P{^Is_NFD_QC: -Y}', "");
Error('\p{NFKC_Quick_Check:   /a/_ maybe}');
Error('\P{NFKC_Quick_Check:   /a/_ maybe}');
Expect(1, 69927, '\p{NFKC_Quick_Check=maybe}', "");
Expect(0, 69927, '\p{^NFKC_Quick_Check=maybe}', "");
Expect(0, 69927, '\P{NFKC_Quick_Check=maybe}', "");
Expect(1, 69927, '\P{^NFKC_Quick_Check=maybe}', "");
Expect(0, 69928, '\p{NFKC_Quick_Check=maybe}', "");
Expect(1, 69928, '\p{^NFKC_Quick_Check=maybe}', "");
Expect(1, 69928, '\P{NFKC_Quick_Check=maybe}', "");
Expect(0, 69928, '\P{^NFKC_Quick_Check=maybe}', "");
Expect(1, 69927, '\p{NFKC_Quick_Check=  Maybe}', "");
Expect(0, 69927, '\p{^NFKC_Quick_Check=  Maybe}', "");
Expect(0, 69927, '\P{NFKC_Quick_Check=  Maybe}', "");
Expect(1, 69927, '\P{^NFKC_Quick_Check=  Maybe}', "");
Expect(0, 69928, '\p{NFKC_Quick_Check=  Maybe}', "");
Expect(1, 69928, '\p{^NFKC_Quick_Check=  Maybe}', "");
Expect(1, 69928, '\P{NFKC_Quick_Check=  Maybe}', "");
Expect(0, 69928, '\P{^NFKC_Quick_Check=  Maybe}', "");
Error('\p{NFKC_QC=:=M}');
Error('\P{NFKC_QC=:=M}');
Expect(1, 69927, '\p{NFKC_QC=m}', "");
Expect(0, 69927, '\p{^NFKC_QC=m}', "");
Expect(0, 69927, '\P{NFKC_QC=m}', "");
Expect(1, 69927, '\P{^NFKC_QC=m}', "");
Expect(0, 69928, '\p{NFKC_QC=m}', "");
Expect(1, 69928, '\p{^NFKC_QC=m}', "");
Expect(1, 69928, '\P{NFKC_QC=m}', "");
Expect(0, 69928, '\P{^NFKC_QC=m}', "");
Expect(1, 69927, '\p{NFKC_QC=  m}', "");
Expect(0, 69927, '\p{^NFKC_QC=  m}', "");
Expect(0, 69927, '\P{NFKC_QC=  m}', "");
Expect(1, 69927, '\P{^NFKC_QC=  m}', "");
Expect(0, 69928, '\p{NFKC_QC=  m}', "");
Expect(1, 69928, '\p{^NFKC_QC=  m}', "");
Expect(1, 69928, '\P{NFKC_QC=  m}', "");
Expect(0, 69928, '\P{^NFKC_QC=  m}', "");
Error('\p{Is_NFKC_Quick_Check:	 maybe:=}');
Error('\P{Is_NFKC_Quick_Check:	 maybe:=}');
Expect(1, 69927, '\p{Is_NFKC_Quick_Check=maybe}', "");
Expect(0, 69927, '\p{^Is_NFKC_Quick_Check=maybe}', "");
Expect(0, 69927, '\P{Is_NFKC_Quick_Check=maybe}', "");
Expect(1, 69927, '\P{^Is_NFKC_Quick_Check=maybe}', "");
Expect(0, 69928, '\p{Is_NFKC_Quick_Check=maybe}', "");
Expect(1, 69928, '\p{^Is_NFKC_Quick_Check=maybe}', "");
Expect(1, 69928, '\P{Is_NFKC_Quick_Check=maybe}', "");
Expect(0, 69928, '\P{^Is_NFKC_Quick_Check=maybe}', "");
Expect(1, 69927, '\p{Is_NFKC_Quick_Check=	_Maybe}', "");
Expect(0, 69927, '\p{^Is_NFKC_Quick_Check=	_Maybe}', "");
Expect(0, 69927, '\P{Is_NFKC_Quick_Check=	_Maybe}', "");
Expect(1, 69927, '\P{^Is_NFKC_Quick_Check=	_Maybe}', "");
Expect(0, 69928, '\p{Is_NFKC_Quick_Check=	_Maybe}', "");
Expect(1, 69928, '\p{^Is_NFKC_Quick_Check=	_Maybe}', "");
Expect(1, 69928, '\P{Is_NFKC_Quick_Check=	_Maybe}', "");
Expect(0, 69928, '\P{^Is_NFKC_Quick_Check=	_Maybe}', "");
Error('\p{Is_NFKC_QC=-	m:=}');
Error('\P{Is_NFKC_QC=-	m:=}');
Expect(1, 69927, '\p{Is_NFKC_QC=m}', "");
Expect(0, 69927, '\p{^Is_NFKC_QC=m}', "");
Expect(0, 69927, '\P{Is_NFKC_QC=m}', "");
Expect(1, 69927, '\P{^Is_NFKC_QC=m}', "");
Expect(0, 69928, '\p{Is_NFKC_QC=m}', "");
Expect(1, 69928, '\p{^Is_NFKC_QC=m}', "");
Expect(1, 69928, '\P{Is_NFKC_QC=m}', "");
Expect(0, 69928, '\P{^Is_NFKC_QC=m}', "");
Expect(1, 69927, '\p{Is_NFKC_QC:-M}', "");
Expect(0, 69927, '\p{^Is_NFKC_QC:-M}', "");
Expect(0, 69927, '\P{Is_NFKC_QC:-M}', "");
Expect(1, 69927, '\P{^Is_NFKC_QC:-M}', "");
Expect(0, 69928, '\p{Is_NFKC_QC:-M}', "");
Expect(1, 69928, '\p{^Is_NFKC_QC:-M}', "");
Expect(1, 69928, '\P{Is_NFKC_QC:-M}', "");
Expect(0, 69928, '\P{^Is_NFKC_QC:-M}', "");
Error('\p{NFKC_Quick_Check=:=-	NO}');
Error('\P{NFKC_Quick_Check=:=-	NO}');
Expect(1, 195101, '\p{NFKC_Quick_Check=no}', "");
Expect(0, 195101, '\p{^NFKC_Quick_Check=no}', "");
Expect(0, 195101, '\P{NFKC_Quick_Check=no}', "");
Expect(1, 195101, '\P{^NFKC_Quick_Check=no}', "");
Expect(0, 195102, '\p{NFKC_Quick_Check=no}', "");
Expect(1, 195102, '\p{^NFKC_Quick_Check=no}', "");
Expect(1, 195102, '\P{NFKC_Quick_Check=no}', "");
Expect(0, 195102, '\P{^NFKC_Quick_Check=no}', "");
Expect(1, 195101, '\p{NFKC_Quick_Check=_	NO}', "");
Expect(0, 195101, '\p{^NFKC_Quick_Check=_	NO}', "");
Expect(0, 195101, '\P{NFKC_Quick_Check=_	NO}', "");
Expect(1, 195101, '\P{^NFKC_Quick_Check=_	NO}', "");
Expect(0, 195102, '\p{NFKC_Quick_Check=_	NO}', "");
Expect(1, 195102, '\p{^NFKC_Quick_Check=_	NO}', "");
Expect(1, 195102, '\P{NFKC_Quick_Check=_	NO}', "");
Expect(0, 195102, '\P{^NFKC_Quick_Check=_	NO}', "");
Error('\p{NFKC_QC=_n/a/}');
Error('\P{NFKC_QC=_n/a/}');
Expect(1, 195101, '\p{NFKC_QC=n}', "");
Expect(0, 195101, '\p{^NFKC_QC=n}', "");
Expect(0, 195101, '\P{NFKC_QC=n}', "");
Expect(1, 195101, '\P{^NFKC_QC=n}', "");
Expect(0, 195102, '\p{NFKC_QC=n}', "");
Expect(1, 195102, '\p{^NFKC_QC=n}', "");
Expect(1, 195102, '\P{NFKC_QC=n}', "");
Expect(0, 195102, '\P{^NFKC_QC=n}', "");
Expect(1, 195101, '\p{NFKC_QC= 	N}', "");
Expect(0, 195101, '\p{^NFKC_QC= 	N}', "");
Expect(0, 195101, '\P{NFKC_QC= 	N}', "");
Expect(1, 195101, '\P{^NFKC_QC= 	N}', "");
Expect(0, 195102, '\p{NFKC_QC= 	N}', "");
Expect(1, 195102, '\p{^NFKC_QC= 	N}', "");
Expect(1, 195102, '\P{NFKC_QC= 	N}', "");
Expect(0, 195102, '\P{^NFKC_QC= 	N}', "");
Error('\p{Is_NFKC_Quick_Check=:=No}');
Error('\P{Is_NFKC_Quick_Check=:=No}');
Expect(1, 195101, '\p{Is_NFKC_Quick_Check=no}', "");
Expect(0, 195101, '\p{^Is_NFKC_Quick_Check=no}', "");
Expect(0, 195101, '\P{Is_NFKC_Quick_Check=no}', "");
Expect(1, 195101, '\P{^Is_NFKC_Quick_Check=no}', "");
Expect(0, 195102, '\p{Is_NFKC_Quick_Check=no}', "");
Expect(1, 195102, '\p{^Is_NFKC_Quick_Check=no}', "");
Expect(1, 195102, '\P{Is_NFKC_Quick_Check=no}', "");
Expect(0, 195102, '\P{^Is_NFKC_Quick_Check=no}', "");
Expect(1, 195101, '\p{Is_NFKC_Quick_Check=	No}', "");
Expect(0, 195101, '\p{^Is_NFKC_Quick_Check=	No}', "");
Expect(0, 195101, '\P{Is_NFKC_Quick_Check=	No}', "");
Expect(1, 195101, '\P{^Is_NFKC_Quick_Check=	No}', "");
Expect(0, 195102, '\p{Is_NFKC_Quick_Check=	No}', "");
Expect(1, 195102, '\p{^Is_NFKC_Quick_Check=	No}', "");
Expect(1, 195102, '\P{Is_NFKC_Quick_Check=	No}', "");
Expect(0, 195102, '\P{^Is_NFKC_Quick_Check=	No}', "");
Error('\p{Is_NFKC_QC=_N/a/}');
Error('\P{Is_NFKC_QC=_N/a/}');
Expect(1, 195101, '\p{Is_NFKC_QC=n}', "");
Expect(0, 195101, '\p{^Is_NFKC_QC=n}', "");
Expect(0, 195101, '\P{Is_NFKC_QC=n}', "");
Expect(1, 195101, '\P{^Is_NFKC_QC=n}', "");
Expect(0, 195102, '\p{Is_NFKC_QC=n}', "");
Expect(1, 195102, '\p{^Is_NFKC_QC=n}', "");
Expect(1, 195102, '\P{Is_NFKC_QC=n}', "");
Expect(0, 195102, '\P{^Is_NFKC_QC=n}', "");
Expect(1, 195101, '\p{Is_NFKC_QC=	 N}', "");
Expect(0, 195101, '\p{^Is_NFKC_QC=	 N}', "");
Expect(0, 195101, '\P{Is_NFKC_QC=	 N}', "");
Expect(1, 195101, '\P{^Is_NFKC_QC=	 N}', "");
Expect(0, 195102, '\p{Is_NFKC_QC=	 N}', "");
Expect(1, 195102, '\p{^Is_NFKC_QC=	 N}', "");
Expect(1, 195102, '\P{Is_NFKC_QC=	 N}', "");
Expect(0, 195102, '\P{^Is_NFKC_QC=	 N}', "");
Error('\p{NFKC_Quick_Check:	_/a/Yes}');
Error('\P{NFKC_Quick_Check:	_/a/Yes}');
Expect(1, 195102, '\p{NFKC_Quick_Check:   yes}', "");
Expect(0, 195102, '\p{^NFKC_Quick_Check:   yes}', "");
Expect(0, 195102, '\P{NFKC_Quick_Check:   yes}', "");
Expect(1, 195102, '\P{^NFKC_Quick_Check:   yes}', "");
Expect(0, 195101, '\p{NFKC_Quick_Check:   yes}', "");
Expect(1, 195101, '\p{^NFKC_Quick_Check:   yes}', "");
Expect(1, 195101, '\P{NFKC_Quick_Check:   yes}', "");
Expect(0, 195101, '\P{^NFKC_Quick_Check:   yes}', "");
Expect(1, 195102, '\p{NFKC_Quick_Check=_	Yes}', "");
Expect(0, 195102, '\p{^NFKC_Quick_Check=_	Yes}', "");
Expect(0, 195102, '\P{NFKC_Quick_Check=_	Yes}', "");
Expect(1, 195102, '\P{^NFKC_Quick_Check=_	Yes}', "");
Expect(0, 195101, '\p{NFKC_Quick_Check=_	Yes}', "");
Expect(1, 195101, '\p{^NFKC_Quick_Check=_	Yes}', "");
Expect(1, 195101, '\P{NFKC_Quick_Check=_	Yes}', "");
Expect(0, 195101, '\P{^NFKC_Quick_Check=_	Yes}', "");
Error('\p{NFKC_QC=_ Y:=}');
Error('\P{NFKC_QC=_ Y:=}');
Expect(1, 195102, '\p{NFKC_QC:	y}', "");
Expect(0, 195102, '\p{^NFKC_QC:	y}', "");
Expect(0, 195102, '\P{NFKC_QC:	y}', "");
Expect(1, 195102, '\P{^NFKC_QC:	y}', "");
Expect(0, 195101, '\p{NFKC_QC:	y}', "");
Expect(1, 195101, '\p{^NFKC_QC:	y}', "");
Expect(1, 195101, '\P{NFKC_QC:	y}', "");
Expect(0, 195101, '\P{^NFKC_QC:	y}', "");
Expect(1, 195102, '\p{NFKC_QC=		Y}', "");
Expect(0, 195102, '\p{^NFKC_QC=		Y}', "");
Expect(0, 195102, '\P{NFKC_QC=		Y}', "");
Expect(1, 195102, '\P{^NFKC_QC=		Y}', "");
Expect(0, 195101, '\p{NFKC_QC=		Y}', "");
Expect(1, 195101, '\p{^NFKC_QC=		Y}', "");
Expect(1, 195101, '\P{NFKC_QC=		Y}', "");
Expect(0, 195101, '\P{^NFKC_QC=		Y}', "");
Error('\p{Is_NFKC_Quick_Check=/a/  YES}');
Error('\P{Is_NFKC_Quick_Check=/a/  YES}');
Expect(1, 195102, '\p{Is_NFKC_Quick_Check=yes}', "");
Expect(0, 195102, '\p{^Is_NFKC_Quick_Check=yes}', "");
Expect(0, 195102, '\P{Is_NFKC_Quick_Check=yes}', "");
Expect(1, 195102, '\P{^Is_NFKC_Quick_Check=yes}', "");
Expect(0, 195101, '\p{Is_NFKC_Quick_Check=yes}', "");
Expect(1, 195101, '\p{^Is_NFKC_Quick_Check=yes}', "");
Expect(1, 195101, '\P{Is_NFKC_Quick_Check=yes}', "");
Expect(0, 195101, '\P{^Is_NFKC_Quick_Check=yes}', "");
Expect(1, 195102, '\p{Is_NFKC_Quick_Check=-_yes}', "");
Expect(0, 195102, '\p{^Is_NFKC_Quick_Check=-_yes}', "");
Expect(0, 195102, '\P{Is_NFKC_Quick_Check=-_yes}', "");
Expect(1, 195102, '\P{^Is_NFKC_Quick_Check=-_yes}', "");
Expect(0, 195101, '\p{Is_NFKC_Quick_Check=-_yes}', "");
Expect(1, 195101, '\p{^Is_NFKC_Quick_Check=-_yes}', "");
Expect(1, 195101, '\P{Is_NFKC_Quick_Check=-_yes}', "");
Expect(0, 195101, '\P{^Is_NFKC_Quick_Check=-_yes}', "");
Error('\p{Is_NFKC_QC=/a/Y}');
Error('\P{Is_NFKC_QC=/a/Y}');
Expect(1, 195102, '\p{Is_NFKC_QC=y}', "");
Expect(0, 195102, '\p{^Is_NFKC_QC=y}', "");
Expect(0, 195102, '\P{Is_NFKC_QC=y}', "");
Expect(1, 195102, '\P{^Is_NFKC_QC=y}', "");
Expect(0, 195101, '\p{Is_NFKC_QC=y}', "");
Expect(1, 195101, '\p{^Is_NFKC_QC=y}', "");
Expect(1, 195101, '\P{Is_NFKC_QC=y}', "");
Expect(0, 195101, '\P{^Is_NFKC_QC=y}', "");
Expect(1, 195102, '\p{Is_NFKC_QC=_ Y}', "");
Expect(0, 195102, '\p{^Is_NFKC_QC=_ Y}', "");
Expect(0, 195102, '\P{Is_NFKC_QC=_ Y}', "");
Expect(1, 195102, '\P{^Is_NFKC_QC=_ Y}', "");
Expect(0, 195101, '\p{Is_NFKC_QC=_ Y}', "");
Expect(1, 195101, '\p{^Is_NFKC_QC=_ Y}', "");
Expect(1, 195101, '\P{Is_NFKC_QC=_ Y}', "");
Expect(0, 195101, '\P{^Is_NFKC_QC=_ Y}', "");
Error('\p{NFKD_Quick_Check=/a/	 No}');
Error('\P{NFKD_Quick_Check=/a/	 No}');
Expect(1, 195101, '\p{NFKD_Quick_Check=no}', "");
Expect(0, 195101, '\p{^NFKD_Quick_Check=no}', "");
Expect(0, 195101, '\P{NFKD_Quick_Check=no}', "");
Expect(1, 195101, '\P{^NFKD_Quick_Check=no}', "");
Expect(0, 195102, '\p{NFKD_Quick_Check=no}', "");
Expect(1, 195102, '\p{^NFKD_Quick_Check=no}', "");
Expect(1, 195102, '\P{NFKD_Quick_Check=no}', "");
Expect(0, 195102, '\P{^NFKD_Quick_Check=no}', "");
Expect(1, 195101, '\p{NFKD_Quick_Check= 	No}', "");
Expect(0, 195101, '\p{^NFKD_Quick_Check= 	No}', "");
Expect(0, 195101, '\P{NFKD_Quick_Check= 	No}', "");
Expect(1, 195101, '\P{^NFKD_Quick_Check= 	No}', "");
Expect(0, 195102, '\p{NFKD_Quick_Check= 	No}', "");
Expect(1, 195102, '\p{^NFKD_Quick_Check= 	No}', "");
Expect(1, 195102, '\P{NFKD_Quick_Check= 	No}', "");
Expect(0, 195102, '\P{^NFKD_Quick_Check= 	No}', "");
Error('\p{NFKD_QC=_:=N}');
Error('\P{NFKD_QC=_:=N}');
Expect(1, 195101, '\p{NFKD_QC=n}', "");
Expect(0, 195101, '\p{^NFKD_QC=n}', "");
Expect(0, 195101, '\P{NFKD_QC=n}', "");
Expect(1, 195101, '\P{^NFKD_QC=n}', "");
Expect(0, 195102, '\p{NFKD_QC=n}', "");
Expect(1, 195102, '\p{^NFKD_QC=n}', "");
Expect(1, 195102, '\P{NFKD_QC=n}', "");
Expect(0, 195102, '\P{^NFKD_QC=n}', "");
Expect(1, 195101, '\p{NFKD_QC= 	N}', "");
Expect(0, 195101, '\p{^NFKD_QC= 	N}', "");
Expect(0, 195101, '\P{NFKD_QC= 	N}', "");
Expect(1, 195101, '\P{^NFKD_QC= 	N}', "");
Expect(0, 195102, '\p{NFKD_QC= 	N}', "");
Expect(1, 195102, '\p{^NFKD_QC= 	N}', "");
Expect(1, 195102, '\P{NFKD_QC= 	N}', "");
Expect(0, 195102, '\P{^NFKD_QC= 	N}', "");
Error('\p{Is_NFKD_Quick_Check::= _no}');
Error('\P{Is_NFKD_Quick_Check::= _no}');
Expect(1, 195101, '\p{Is_NFKD_Quick_Check=no}', "");
Expect(0, 195101, '\p{^Is_NFKD_Quick_Check=no}', "");
Expect(0, 195101, '\P{Is_NFKD_Quick_Check=no}', "");
Expect(1, 195101, '\P{^Is_NFKD_Quick_Check=no}', "");
Expect(0, 195102, '\p{Is_NFKD_Quick_Check=no}', "");
Expect(1, 195102, '\p{^Is_NFKD_Quick_Check=no}', "");
Expect(1, 195102, '\P{Is_NFKD_Quick_Check=no}', "");
Expect(0, 195102, '\P{^Is_NFKD_Quick_Check=no}', "");
Expect(1, 195101, '\p{Is_NFKD_Quick_Check= no}', "");
Expect(0, 195101, '\p{^Is_NFKD_Quick_Check= no}', "");
Expect(0, 195101, '\P{Is_NFKD_Quick_Check= no}', "");
Expect(1, 195101, '\P{^Is_NFKD_Quick_Check= no}', "");
Expect(0, 195102, '\p{Is_NFKD_Quick_Check= no}', "");
Expect(1, 195102, '\p{^Is_NFKD_Quick_Check= no}', "");
Expect(1, 195102, '\P{Is_NFKD_Quick_Check= no}', "");
Expect(0, 195102, '\P{^Is_NFKD_Quick_Check= no}', "");
Error('\p{Is_NFKD_QC=-/a/N}');
Error('\P{Is_NFKD_QC=-/a/N}');
Expect(1, 195101, '\p{Is_NFKD_QC=n}', "");
Expect(0, 195101, '\p{^Is_NFKD_QC=n}', "");
Expect(0, 195101, '\P{Is_NFKD_QC=n}', "");
Expect(1, 195101, '\P{^Is_NFKD_QC=n}', "");
Expect(0, 195102, '\p{Is_NFKD_QC=n}', "");
Expect(1, 195102, '\p{^Is_NFKD_QC=n}', "");
Expect(1, 195102, '\P{Is_NFKD_QC=n}', "");
Expect(0, 195102, '\P{^Is_NFKD_QC=n}', "");
Expect(1, 195101, '\p{Is_NFKD_QC=_ N}', "");
Expect(0, 195101, '\p{^Is_NFKD_QC=_ N}', "");
Expect(0, 195101, '\P{Is_NFKD_QC=_ N}', "");
Expect(1, 195101, '\P{^Is_NFKD_QC=_ N}', "");
Expect(0, 195102, '\p{Is_NFKD_QC=_ N}', "");
Expect(1, 195102, '\p{^Is_NFKD_QC=_ N}', "");
Expect(1, 195102, '\P{Is_NFKD_QC=_ N}', "");
Expect(0, 195102, '\P{^Is_NFKD_QC=_ N}', "");
Error('\p{NFKD_Quick_Check=-Yes/a/}');
Error('\P{NFKD_Quick_Check=-Yes/a/}');
Expect(1, 195102, '\p{NFKD_Quick_Check=yes}', "");
Expect(0, 195102, '\p{^NFKD_Quick_Check=yes}', "");
Expect(0, 195102, '\P{NFKD_Quick_Check=yes}', "");
Expect(1, 195102, '\P{^NFKD_Quick_Check=yes}', "");
Expect(0, 195101, '\p{NFKD_Quick_Check=yes}', "");
Expect(1, 195101, '\p{^NFKD_Quick_Check=yes}', "");
Expect(1, 195101, '\P{NFKD_Quick_Check=yes}', "");
Expect(0, 195101, '\P{^NFKD_Quick_Check=yes}', "");
Expect(1, 195102, '\p{NFKD_Quick_Check=	 Yes}', "");
Expect(0, 195102, '\p{^NFKD_Quick_Check=	 Yes}', "");
Expect(0, 195102, '\P{NFKD_Quick_Check=	 Yes}', "");
Expect(1, 195102, '\P{^NFKD_Quick_Check=	 Yes}', "");
Expect(0, 195101, '\p{NFKD_Quick_Check=	 Yes}', "");
Expect(1, 195101, '\p{^NFKD_Quick_Check=	 Yes}', "");
Expect(1, 195101, '\P{NFKD_Quick_Check=	 Yes}', "");
Expect(0, 195101, '\P{^NFKD_Quick_Check=	 Yes}', "");
Error('\p{NFKD_QC=--Y:=}');
Error('\P{NFKD_QC=--Y:=}');
Expect(1, 195102, '\p{NFKD_QC=y}', "");
Expect(0, 195102, '\p{^NFKD_QC=y}', "");
Expect(0, 195102, '\P{NFKD_QC=y}', "");
Expect(1, 195102, '\P{^NFKD_QC=y}', "");
Expect(0, 195101, '\p{NFKD_QC=y}', "");
Expect(1, 195101, '\p{^NFKD_QC=y}', "");
Expect(1, 195101, '\P{NFKD_QC=y}', "");
Expect(0, 195101, '\P{^NFKD_QC=y}', "");
Expect(1, 195102, '\p{NFKD_QC=-y}', "");
Expect(0, 195102, '\p{^NFKD_QC=-y}', "");
Expect(0, 195102, '\P{NFKD_QC=-y}', "");
Expect(1, 195102, '\P{^NFKD_QC=-y}', "");
Expect(0, 195101, '\p{NFKD_QC=-y}', "");
Expect(1, 195101, '\p{^NFKD_QC=-y}', "");
Expect(1, 195101, '\P{NFKD_QC=-y}', "");
Expect(0, 195101, '\P{^NFKD_QC=-y}', "");
Error('\p{Is_NFKD_Quick_Check=	:=Yes}');
Error('\P{Is_NFKD_Quick_Check=	:=Yes}');
Expect(1, 195102, '\p{Is_NFKD_Quick_Check=yes}', "");
Expect(0, 195102, '\p{^Is_NFKD_Quick_Check=yes}', "");
Expect(0, 195102, '\P{Is_NFKD_Quick_Check=yes}', "");
Expect(1, 195102, '\P{^Is_NFKD_Quick_Check=yes}', "");
Expect(0, 195101, '\p{Is_NFKD_Quick_Check=yes}', "");
Expect(1, 195101, '\p{^Is_NFKD_Quick_Check=yes}', "");
Expect(1, 195101, '\P{Is_NFKD_Quick_Check=yes}', "");
Expect(0, 195101, '\P{^Is_NFKD_Quick_Check=yes}', "");
Expect(1, 195102, '\p{Is_NFKD_Quick_Check=_YES}', "");
Expect(0, 195102, '\p{^Is_NFKD_Quick_Check=_YES}', "");
Expect(0, 195102, '\P{Is_NFKD_Quick_Check=_YES}', "");
Expect(1, 195102, '\P{^Is_NFKD_Quick_Check=_YES}', "");
Expect(0, 195101, '\p{Is_NFKD_Quick_Check=_YES}', "");
Expect(1, 195101, '\p{^Is_NFKD_Quick_Check=_YES}', "");
Expect(1, 195101, '\P{Is_NFKD_Quick_Check=_YES}', "");
Expect(0, 195101, '\P{^Is_NFKD_Quick_Check=_YES}', "");
Error('\p{Is_NFKD_QC=	:=Y}');
Error('\P{Is_NFKD_QC=	:=Y}');
Expect(1, 195102, '\p{Is_NFKD_QC=y}', "");
Expect(0, 195102, '\p{^Is_NFKD_QC=y}', "");
Expect(0, 195102, '\P{Is_NFKD_QC=y}', "");
Expect(1, 195102, '\P{^Is_NFKD_QC=y}', "");
Expect(0, 195101, '\p{Is_NFKD_QC=y}', "");
Expect(1, 195101, '\p{^Is_NFKD_QC=y}', "");
Expect(1, 195101, '\P{Is_NFKD_QC=y}', "");
Expect(0, 195101, '\P{^Is_NFKD_QC=y}', "");
Expect(1, 195102, '\p{Is_NFKD_QC=	-Y}', "");
Expect(0, 195102, '\p{^Is_NFKD_QC=	-Y}', "");
Expect(0, 195102, '\P{Is_NFKD_QC=	-Y}', "");
Expect(1, 195102, '\P{^Is_NFKD_QC=	-Y}', "");
Expect(0, 195101, '\p{Is_NFKD_QC=	-Y}', "");
Expect(1, 195101, '\p{^Is_NFKD_QC=	-Y}', "");
Expect(1, 195101, '\P{Is_NFKD_QC=	-Y}', "");
Expect(0, 195101, '\P{^Is_NFKD_QC=	-Y}', "");
Error('\p{Other_Alphabetic=No}');
Error('\P{Other_Alphabetic=No}');
Error('\p{OAlpha: N}');
Error('\P{OAlpha: N}');
Error('\p{Is_Other_Alphabetic=F}');
Error('\P{Is_Other_Alphabetic=F}');
Error('\p{Is_OAlpha=False}');
Error('\P{Is_OAlpha=False}');
Error('\p{Other_Alphabetic:   Yes}');
Error('\P{Other_Alphabetic:   Yes}');
Error('\p{OAlpha=Y}');
Error('\P{OAlpha=Y}');
Error('\p{Is_Other_Alphabetic=T}');
Error('\P{Is_Other_Alphabetic=T}');
Error('\p{Is_OAlpha=True}');
Error('\P{Is_OAlpha=True}');
Error('\p{Other_Default_Ignorable_Code_Point: No}');
Error('\P{Other_Default_Ignorable_Code_Point: No}');
Error('\p{ODI=N}');
Error('\P{ODI=N}');
Error('\p{Is_Other_Default_Ignorable_Code_Point=F}');
Error('\P{Is_Other_Default_Ignorable_Code_Point=F}');
Error('\p{Is_ODI=False}');
Error('\P{Is_ODI=False}');
Error('\p{Other_Default_Ignorable_Code_Point=Yes}');
Error('\P{Other_Default_Ignorable_Code_Point=Yes}');
Error('\p{ODI=Y}');
Error('\P{ODI=Y}');
Error('\p{Is_Other_Default_Ignorable_Code_Point: T}');
Error('\P{Is_Other_Default_Ignorable_Code_Point: T}');
Error('\p{Is_ODI=True}');
Error('\P{Is_ODI=True}');
Error('\p{Other_Grapheme_Extend=No}');
Error('\P{Other_Grapheme_Extend=No}');
Error('\p{OGr_Ext:	N}');
Error('\P{OGr_Ext:	N}');
Error('\p{Is_Other_Grapheme_Extend=F}');
Error('\P{Is_Other_Grapheme_Extend=F}');
Error('\p{Is_OGr_Ext=False}');
Error('\P{Is_OGr_Ext=False}');
Error('\p{Other_Grapheme_Extend:	Yes}');
Error('\P{Other_Grapheme_Extend:	Yes}');
Error('\p{OGr_Ext=Y}');
Error('\P{OGr_Ext=Y}');
Error('\p{Is_Other_Grapheme_Extend=T}');
Error('\P{Is_Other_Grapheme_Extend=T}');
Error('\p{Is_OGr_Ext=True}');
Error('\P{Is_OGr_Ext=True}');
Error('\p{Other_ID_Continue=No}');
Error('\P{Other_ID_Continue=No}');
Error('\p{OIDC=N}');
Error('\P{OIDC=N}');
Error('\p{Is_Other_ID_Continue=F}');
Error('\P{Is_Other_ID_Continue=F}');
Error('\p{Is_OIDC=False}');
Error('\P{Is_OIDC=False}');
Error('\p{Other_ID_Continue=Yes}');
Error('\P{Other_ID_Continue=Yes}');
Error('\p{OIDC=Y}');
Error('\P{OIDC=Y}');
Error('\p{Is_Other_ID_Continue=T}');
Error('\P{Is_Other_ID_Continue=T}');
Error('\p{Is_OIDC=True}');
Error('\P{Is_OIDC=True}');
Error('\p{Other_ID_Start:	No}');
Error('\P{Other_ID_Start:	No}');
Error('\p{OIDS=N}');
Error('\P{OIDS=N}');
Error('\p{Is_Other_ID_Start=F}');
Error('\P{Is_Other_ID_Start=F}');
Error('\p{Is_OIDS=False}');
Error('\P{Is_OIDS=False}');
Error('\p{Other_ID_Start=Yes}');
Error('\P{Other_ID_Start=Yes}');
Error('\p{OIDS=Y}');
Error('\P{OIDS=Y}');
Error('\p{Is_Other_ID_Start:T}');
Error('\P{Is_Other_ID_Start:T}');
Error('\p{Is_OIDS=True}');
Error('\P{Is_OIDS=True}');
Error('\p{Other_Lowercase=No}');
Error('\P{Other_Lowercase=No}');
Error('\p{OLower=N}');
Error('\P{OLower=N}');
Error('\p{Is_Other_Lowercase=F}');
Error('\P{Is_Other_Lowercase=F}');
Error('\p{Is_OLower:False}');
Error('\P{Is_OLower:False}');
Error('\p{Other_Lowercase=Yes}');
Error('\P{Other_Lowercase=Yes}');
Error('\p{OLower=Y}');
Error('\P{OLower=Y}');
Error('\p{Is_Other_Lowercase=T}');
Error('\P{Is_Other_Lowercase=T}');
Error('\p{Is_OLower=True}');
Error('\P{Is_OLower=True}');
Error('\p{Other_Math=No}');
Error('\P{Other_Math=No}');
Error('\p{OMath=N}');
Error('\P{OMath=N}');
Error('\p{Is_Other_Math=F}');
Error('\P{Is_Other_Math=F}');
Error('\p{Is_OMath=False}');
Error('\P{Is_OMath=False}');
Error('\p{Other_Math=Yes}');
Error('\P{Other_Math=Yes}');
Error('\p{OMath=Y}');
Error('\P{OMath=Y}');
Error('\p{Is_Other_Math=T}');
Error('\P{Is_Other_Math=T}');
Error('\p{Is_OMath:   True}');
Error('\P{Is_OMath:   True}');
Error('\p{Other_Uppercase=No}');
Error('\P{Other_Uppercase=No}');
Error('\p{OUpper=N}');
Error('\P{OUpper=N}');
Error('\p{Is_Other_Uppercase=F}');
Error('\P{Is_Other_Uppercase=F}');
Error('\p{Is_OUpper: False}');
Error('\P{Is_OUpper: False}');
Error('\p{Other_Uppercase: Yes}');
Error('\P{Other_Uppercase: Yes}');
Error('\p{OUpper=Y}');
Error('\P{OUpper=Y}');
Error('\p{Is_Other_Uppercase=T}');
Error('\P{Is_Other_Uppercase=T}');
Error('\p{Is_OUpper=True}');
Error('\P{Is_OUpper=True}');
Error('\p{Pattern_Syntax=	No/a/}');
Error('\P{Pattern_Syntax=	No/a/}');
Expect(1, 65095, '\p{Pattern_Syntax=no}', "");
Expect(0, 65095, '\p{^Pattern_Syntax=no}', "");
Expect(0, 65095, '\P{Pattern_Syntax=no}', "");
Expect(1, 65095, '\P{^Pattern_Syntax=no}', "");
Expect(0, 65094, '\p{Pattern_Syntax=no}', "");
Expect(1, 65094, '\p{^Pattern_Syntax=no}', "");
Expect(1, 65094, '\P{Pattern_Syntax=no}', "");
Expect(0, 65094, '\P{^Pattern_Syntax=no}', "");
Expect(1, 65095, '\p{Pattern_Syntax:  _No}', "");
Expect(0, 65095, '\p{^Pattern_Syntax:  _No}', "");
Expect(0, 65095, '\P{Pattern_Syntax:  _No}', "");
Expect(1, 65095, '\P{^Pattern_Syntax:  _No}', "");
Expect(0, 65094, '\p{Pattern_Syntax:  _No}', "");
Expect(1, 65094, '\p{^Pattern_Syntax:  _No}', "");
Expect(1, 65094, '\P{Pattern_Syntax:  _No}', "");
Expect(0, 65094, '\P{^Pattern_Syntax:  _No}', "");
Error('\p{Pat_Syn= /a/N}');
Error('\P{Pat_Syn= /a/N}');
Expect(1, 65095, '\p{Pat_Syn=n}', "");
Expect(0, 65095, '\p{^Pat_Syn=n}', "");
Expect(0, 65095, '\P{Pat_Syn=n}', "");
Expect(1, 65095, '\P{^Pat_Syn=n}', "");
Expect(0, 65094, '\p{Pat_Syn=n}', "");
Expect(1, 65094, '\p{^Pat_Syn=n}', "");
Expect(1, 65094, '\P{Pat_Syn=n}', "");
Expect(0, 65094, '\P{^Pat_Syn=n}', "");
Expect(1, 65095, '\p{Pat_Syn=- N}', "");
Expect(0, 65095, '\p{^Pat_Syn=- N}', "");
Expect(0, 65095, '\P{Pat_Syn=- N}', "");
Expect(1, 65095, '\P{^Pat_Syn=- N}', "");
Expect(0, 65094, '\p{Pat_Syn=- N}', "");
Expect(1, 65094, '\p{^Pat_Syn=- N}', "");
Expect(1, 65094, '\P{Pat_Syn=- N}', "");
Expect(0, 65094, '\P{^Pat_Syn=- N}', "");
Error('\p{Is_Pattern_Syntax=-_F/a/}');
Error('\P{Is_Pattern_Syntax=-_F/a/}');
Expect(1, 65095, '\p{Is_Pattern_Syntax:	f}', "");
Expect(0, 65095, '\p{^Is_Pattern_Syntax:	f}', "");
Expect(0, 65095, '\P{Is_Pattern_Syntax:	f}', "");
Expect(1, 65095, '\P{^Is_Pattern_Syntax:	f}', "");
Expect(0, 65094, '\p{Is_Pattern_Syntax:	f}', "");
Expect(1, 65094, '\p{^Is_Pattern_Syntax:	f}', "");
Expect(1, 65094, '\P{Is_Pattern_Syntax:	f}', "");
Expect(0, 65094, '\P{^Is_Pattern_Syntax:	f}', "");
Expect(1, 65095, '\p{Is_Pattern_Syntax= f}', "");
Expect(0, 65095, '\p{^Is_Pattern_Syntax= f}', "");
Expect(0, 65095, '\P{Is_Pattern_Syntax= f}', "");
Expect(1, 65095, '\P{^Is_Pattern_Syntax= f}', "");
Expect(0, 65094, '\p{Is_Pattern_Syntax= f}', "");
Expect(1, 65094, '\p{^Is_Pattern_Syntax= f}', "");
Expect(1, 65094, '\P{Is_Pattern_Syntax= f}', "");
Expect(0, 65094, '\P{^Is_Pattern_Syntax= f}', "");
Error('\p{Is_Pat_Syn:   := _False}');
Error('\P{Is_Pat_Syn:   := _False}');
Expect(1, 65095, '\p{Is_Pat_Syn=false}', "");
Expect(0, 65095, '\p{^Is_Pat_Syn=false}', "");
Expect(0, 65095, '\P{Is_Pat_Syn=false}', "");
Expect(1, 65095, '\P{^Is_Pat_Syn=false}', "");
Expect(0, 65094, '\p{Is_Pat_Syn=false}', "");
Expect(1, 65094, '\p{^Is_Pat_Syn=false}', "");
Expect(1, 65094, '\P{Is_Pat_Syn=false}', "");
Expect(0, 65094, '\P{^Is_Pat_Syn=false}', "");
Expect(1, 65095, '\p{Is_Pat_Syn= _False}', "");
Expect(0, 65095, '\p{^Is_Pat_Syn= _False}', "");
Expect(0, 65095, '\P{Is_Pat_Syn= _False}', "");
Expect(1, 65095, '\P{^Is_Pat_Syn= _False}', "");
Expect(0, 65094, '\p{Is_Pat_Syn= _False}', "");
Expect(1, 65094, '\p{^Is_Pat_Syn= _False}', "");
Expect(1, 65094, '\P{Is_Pat_Syn= _False}', "");
Expect(0, 65094, '\P{^Is_Pat_Syn= _False}', "");
Error('\p{Pattern_Syntax=_	Yes:=}');
Error('\P{Pattern_Syntax=_	Yes:=}');
Expect(1, 65094, '\p{Pattern_Syntax=yes}', "");
Expect(0, 65094, '\p{^Pattern_Syntax=yes}', "");
Expect(0, 65094, '\P{Pattern_Syntax=yes}', "");
Expect(1, 65094, '\P{^Pattern_Syntax=yes}', "");
Expect(0, 65095, '\p{Pattern_Syntax=yes}', "");
Expect(1, 65095, '\p{^Pattern_Syntax=yes}', "");
Expect(1, 65095, '\P{Pattern_Syntax=yes}', "");
Expect(0, 65095, '\P{^Pattern_Syntax=yes}', "");
Expect(1, 65094, '\p{Pattern_Syntax=		Yes}', "");
Expect(0, 65094, '\p{^Pattern_Syntax=		Yes}', "");
Expect(0, 65094, '\P{Pattern_Syntax=		Yes}', "");
Expect(1, 65094, '\P{^Pattern_Syntax=		Yes}', "");
Expect(0, 65095, '\p{Pattern_Syntax=		Yes}', "");
Expect(1, 65095, '\p{^Pattern_Syntax=		Yes}', "");
Expect(1, 65095, '\P{Pattern_Syntax=		Yes}', "");
Expect(0, 65095, '\P{^Pattern_Syntax=		Yes}', "");
Error('\p{Pat_Syn=:=_	Y}');
Error('\P{Pat_Syn=:=_	Y}');
Expect(1, 65094, '\p{Pat_Syn=y}', "");
Expect(0, 65094, '\p{^Pat_Syn=y}', "");
Expect(0, 65094, '\P{Pat_Syn=y}', "");
Expect(1, 65094, '\P{^Pat_Syn=y}', "");
Expect(0, 65095, '\p{Pat_Syn=y}', "");
Expect(1, 65095, '\p{^Pat_Syn=y}', "");
Expect(1, 65095, '\P{Pat_Syn=y}', "");
Expect(0, 65095, '\P{^Pat_Syn=y}', "");
Expect(1, 65094, '\p{Pat_Syn=__Y}', "");
Expect(0, 65094, '\p{^Pat_Syn=__Y}', "");
Expect(0, 65094, '\P{Pat_Syn=__Y}', "");
Expect(1, 65094, '\P{^Pat_Syn=__Y}', "");
Expect(0, 65095, '\p{Pat_Syn=__Y}', "");
Expect(1, 65095, '\p{^Pat_Syn=__Y}', "");
Expect(1, 65095, '\P{Pat_Syn=__Y}', "");
Expect(0, 65095, '\P{^Pat_Syn=__Y}', "");
Error('\p{Is_Pattern_Syntax=/a/T}');
Error('\P{Is_Pattern_Syntax=/a/T}');
Expect(1, 65094, '\p{Is_Pattern_Syntax=t}', "");
Expect(0, 65094, '\p{^Is_Pattern_Syntax=t}', "");
Expect(0, 65094, '\P{Is_Pattern_Syntax=t}', "");
Expect(1, 65094, '\P{^Is_Pattern_Syntax=t}', "");
Expect(0, 65095, '\p{Is_Pattern_Syntax=t}', "");
Expect(1, 65095, '\p{^Is_Pattern_Syntax=t}', "");
Expect(1, 65095, '\P{Is_Pattern_Syntax=t}', "");
Expect(0, 65095, '\P{^Is_Pattern_Syntax=t}', "");
Expect(1, 65094, '\p{Is_Pattern_Syntax=	 T}', "");
Expect(0, 65094, '\p{^Is_Pattern_Syntax=	 T}', "");
Expect(0, 65094, '\P{Is_Pattern_Syntax=	 T}', "");
Expect(1, 65094, '\P{^Is_Pattern_Syntax=	 T}', "");
Expect(0, 65095, '\p{Is_Pattern_Syntax=	 T}', "");
Expect(1, 65095, '\p{^Is_Pattern_Syntax=	 T}', "");
Expect(1, 65095, '\P{Is_Pattern_Syntax=	 T}', "");
Expect(0, 65095, '\P{^Is_Pattern_Syntax=	 T}', "");
Error('\p{Is_Pat_Syn=_True/a/}');
Error('\P{Is_Pat_Syn=_True/a/}');
Expect(1, 65094, '\p{Is_Pat_Syn=true}', "");
Expect(0, 65094, '\p{^Is_Pat_Syn=true}', "");
Expect(0, 65094, '\P{Is_Pat_Syn=true}', "");
Expect(1, 65094, '\P{^Is_Pat_Syn=true}', "");
Expect(0, 65095, '\p{Is_Pat_Syn=true}', "");
Expect(1, 65095, '\p{^Is_Pat_Syn=true}', "");
Expect(1, 65095, '\P{Is_Pat_Syn=true}', "");
Expect(0, 65095, '\P{^Is_Pat_Syn=true}', "");
Expect(1, 65094, '\p{Is_Pat_Syn=-	true}', "");
Expect(0, 65094, '\p{^Is_Pat_Syn=-	true}', "");
Expect(0, 65094, '\P{Is_Pat_Syn=-	true}', "");
Expect(1, 65094, '\P{^Is_Pat_Syn=-	true}', "");
Expect(0, 65095, '\p{Is_Pat_Syn=-	true}', "");
Expect(1, 65095, '\p{^Is_Pat_Syn=-	true}', "");
Expect(1, 65095, '\P{Is_Pat_Syn=-	true}', "");
Expect(0, 65095, '\P{^Is_Pat_Syn=-	true}', "");
Error('\p{Pattern_White_Space= No:=}');
Error('\P{Pattern_White_Space= No:=}');
Expect(1, 8234, '\p{Pattern_White_Space=no}', "");
Expect(0, 8234, '\p{^Pattern_White_Space=no}', "");
Expect(0, 8234, '\P{Pattern_White_Space=no}', "");
Expect(1, 8234, '\P{^Pattern_White_Space=no}', "");
Expect(0, 8233, '\p{Pattern_White_Space=no}', "");
Expect(1, 8233, '\p{^Pattern_White_Space=no}', "");
Expect(1, 8233, '\P{Pattern_White_Space=no}', "");
Expect(0, 8233, '\P{^Pattern_White_Space=no}', "");
Expect(1, 8234, '\p{Pattern_White_Space=- No}', "");
Expect(0, 8234, '\p{^Pattern_White_Space=- No}', "");
Expect(0, 8234, '\P{Pattern_White_Space=- No}', "");
Expect(1, 8234, '\P{^Pattern_White_Space=- No}', "");
Expect(0, 8233, '\p{Pattern_White_Space=- No}', "");
Expect(1, 8233, '\p{^Pattern_White_Space=- No}', "");
Expect(1, 8233, '\P{Pattern_White_Space=- No}', "");
Expect(0, 8233, '\P{^Pattern_White_Space=- No}', "");
Error('\p{Pat_WS=	N:=}');
Error('\P{Pat_WS=	N:=}');
Expect(1, 8234, '\p{Pat_WS=n}', "");
Expect(0, 8234, '\p{^Pat_WS=n}', "");
Expect(0, 8234, '\P{Pat_WS=n}', "");
Expect(1, 8234, '\P{^Pat_WS=n}', "");
Expect(0, 8233, '\p{Pat_WS=n}', "");
Expect(1, 8233, '\p{^Pat_WS=n}', "");
Expect(1, 8233, '\P{Pat_WS=n}', "");
Expect(0, 8233, '\P{^Pat_WS=n}', "");
Expect(1, 8234, '\p{Pat_WS:_n}', "");
Expect(0, 8234, '\p{^Pat_WS:_n}', "");
Expect(0, 8234, '\P{Pat_WS:_n}', "");
Expect(1, 8234, '\P{^Pat_WS:_n}', "");
Expect(0, 8233, '\p{Pat_WS:_n}', "");
Expect(1, 8233, '\p{^Pat_WS:_n}', "");
Expect(1, 8233, '\P{Pat_WS:_n}', "");
Expect(0, 8233, '\P{^Pat_WS:_n}', "");
Error('\p{Is_Pattern_White_Space=	F:=}');
Error('\P{Is_Pattern_White_Space=	F:=}');
Expect(1, 8234, '\p{Is_Pattern_White_Space=f}', "");
Expect(0, 8234, '\p{^Is_Pattern_White_Space=f}', "");
Expect(0, 8234, '\P{Is_Pattern_White_Space=f}', "");
Expect(1, 8234, '\P{^Is_Pattern_White_Space=f}', "");
Expect(0, 8233, '\p{Is_Pattern_White_Space=f}', "");
Expect(1, 8233, '\p{^Is_Pattern_White_Space=f}', "");
Expect(1, 8233, '\P{Is_Pattern_White_Space=f}', "");
Expect(0, 8233, '\P{^Is_Pattern_White_Space=f}', "");
Expect(1, 8234, '\p{Is_Pattern_White_Space=_ F}', "");
Expect(0, 8234, '\p{^Is_Pattern_White_Space=_ F}', "");
Expect(0, 8234, '\P{Is_Pattern_White_Space=_ F}', "");
Expect(1, 8234, '\P{^Is_Pattern_White_Space=_ F}', "");
Expect(0, 8233, '\p{Is_Pattern_White_Space=_ F}', "");
Expect(1, 8233, '\p{^Is_Pattern_White_Space=_ F}', "");
Expect(1, 8233, '\P{Is_Pattern_White_Space=_ F}', "");
Expect(0, 8233, '\P{^Is_Pattern_White_Space=_ F}', "");
Error('\p{Is_Pat_WS=:=--FALSE}');
Error('\P{Is_Pat_WS=:=--FALSE}');
Expect(1, 8234, '\p{Is_Pat_WS=false}', "");
Expect(0, 8234, '\p{^Is_Pat_WS=false}', "");
Expect(0, 8234, '\P{Is_Pat_WS=false}', "");
Expect(1, 8234, '\P{^Is_Pat_WS=false}', "");
Expect(0, 8233, '\p{Is_Pat_WS=false}', "");
Expect(1, 8233, '\p{^Is_Pat_WS=false}', "");
Expect(1, 8233, '\P{Is_Pat_WS=false}', "");
Expect(0, 8233, '\P{^Is_Pat_WS=false}', "");
Expect(1, 8234, '\p{Is_Pat_WS=  False}', "");
Expect(0, 8234, '\p{^Is_Pat_WS=  False}', "");
Expect(0, 8234, '\P{Is_Pat_WS=  False}', "");
Expect(1, 8234, '\P{^Is_Pat_WS=  False}', "");
Expect(0, 8233, '\p{Is_Pat_WS=  False}', "");
Expect(1, 8233, '\p{^Is_Pat_WS=  False}', "");
Expect(1, 8233, '\P{Is_Pat_WS=  False}', "");
Expect(0, 8233, '\P{^Is_Pat_WS=  False}', "");
Error('\p{Pattern_White_Space=/a/ -Yes}');
Error('\P{Pattern_White_Space=/a/ -Yes}');
Expect(1, 8233, '\p{Pattern_White_Space=yes}', "");
Expect(0, 8233, '\p{^Pattern_White_Space=yes}', "");
Expect(0, 8233, '\P{Pattern_White_Space=yes}', "");
Expect(1, 8233, '\P{^Pattern_White_Space=yes}', "");
Expect(0, 8234, '\p{Pattern_White_Space=yes}', "");
Expect(1, 8234, '\p{^Pattern_White_Space=yes}', "");
Expect(1, 8234, '\P{Pattern_White_Space=yes}', "");
Expect(0, 8234, '\P{^Pattern_White_Space=yes}', "");
Expect(1, 8233, '\p{Pattern_White_Space=-yes}', "");
Expect(0, 8233, '\p{^Pattern_White_Space=-yes}', "");
Expect(0, 8233, '\P{Pattern_White_Space=-yes}', "");
Expect(1, 8233, '\P{^Pattern_White_Space=-yes}', "");
Expect(0, 8234, '\p{Pattern_White_Space=-yes}', "");
Expect(1, 8234, '\p{^Pattern_White_Space=-yes}', "");
Expect(1, 8234, '\P{Pattern_White_Space=-yes}', "");
Expect(0, 8234, '\P{^Pattern_White_Space=-yes}', "");
Error('\p{Pat_WS=-/a/Y}');
Error('\P{Pat_WS=-/a/Y}');
Expect(1, 8233, '\p{Pat_WS=y}', "");
Expect(0, 8233, '\p{^Pat_WS=y}', "");
Expect(0, 8233, '\P{Pat_WS=y}', "");
Expect(1, 8233, '\P{^Pat_WS=y}', "");
Expect(0, 8234, '\p{Pat_WS=y}', "");
Expect(1, 8234, '\p{^Pat_WS=y}', "");
Expect(1, 8234, '\P{Pat_WS=y}', "");
Expect(0, 8234, '\P{^Pat_WS=y}', "");
Expect(1, 8233, '\p{Pat_WS=Y}', "");
Expect(0, 8233, '\p{^Pat_WS=Y}', "");
Expect(0, 8233, '\P{Pat_WS=Y}', "");
Expect(1, 8233, '\P{^Pat_WS=Y}', "");
Expect(0, 8234, '\p{Pat_WS=Y}', "");
Expect(1, 8234, '\p{^Pat_WS=Y}', "");
Expect(1, 8234, '\P{Pat_WS=Y}', "");
Expect(0, 8234, '\P{^Pat_WS=Y}', "");
Error('\p{Is_Pattern_White_Space=:=_-t}');
Error('\P{Is_Pattern_White_Space=:=_-t}');
Expect(1, 8233, '\p{Is_Pattern_White_Space=t}', "");
Expect(0, 8233, '\p{^Is_Pattern_White_Space=t}', "");
Expect(0, 8233, '\P{Is_Pattern_White_Space=t}', "");
Expect(1, 8233, '\P{^Is_Pattern_White_Space=t}', "");
Expect(0, 8234, '\p{Is_Pattern_White_Space=t}', "");
Expect(1, 8234, '\p{^Is_Pattern_White_Space=t}', "");
Expect(1, 8234, '\P{Is_Pattern_White_Space=t}', "");
Expect(0, 8234, '\P{^Is_Pattern_White_Space=t}', "");
Expect(1, 8233, '\p{Is_Pattern_White_Space=- T}', "");
Expect(0, 8233, '\p{^Is_Pattern_White_Space=- T}', "");
Expect(0, 8233, '\P{Is_Pattern_White_Space=- T}', "");
Expect(1, 8233, '\P{^Is_Pattern_White_Space=- T}', "");
Expect(0, 8234, '\p{Is_Pattern_White_Space=- T}', "");
Expect(1, 8234, '\p{^Is_Pattern_White_Space=- T}', "");
Expect(1, 8234, '\P{Is_Pattern_White_Space=- T}', "");
Expect(0, 8234, '\P{^Is_Pattern_White_Space=- T}', "");
Error('\p{Is_Pat_WS=:= True}');
Error('\P{Is_Pat_WS=:= True}');
Expect(1, 8233, '\p{Is_Pat_WS=true}', "");
Expect(0, 8233, '\p{^Is_Pat_WS=true}', "");
Expect(0, 8233, '\P{Is_Pat_WS=true}', "");
Expect(1, 8233, '\P{^Is_Pat_WS=true}', "");
Expect(0, 8234, '\p{Is_Pat_WS=true}', "");
Expect(1, 8234, '\p{^Is_Pat_WS=true}', "");
Expect(1, 8234, '\P{Is_Pat_WS=true}', "");
Expect(0, 8234, '\P{^Is_Pat_WS=true}', "");
Expect(1, 8233, '\p{Is_Pat_WS=	True}', "");
Expect(0, 8233, '\p{^Is_Pat_WS=	True}', "");
Expect(0, 8233, '\P{Is_Pat_WS=	True}', "");
Expect(1, 8233, '\P{^Is_Pat_WS=	True}', "");
Expect(0, 8234, '\p{Is_Pat_WS=	True}', "");
Expect(1, 8234, '\p{^Is_Pat_WS=	True}', "");
Expect(1, 8234, '\P{Is_Pat_WS=	True}', "");
Expect(0, 8234, '\P{^Is_Pat_WS=	True}', "");
Error('\p{Quotation_Mark=-:=NO}');
Error('\P{Quotation_Mark=-:=NO}');
Expect(1, 65380, '\p{Quotation_Mark=no}', "");
Expect(0, 65380, '\p{^Quotation_Mark=no}', "");
Expect(0, 65380, '\P{Quotation_Mark=no}', "");
Expect(1, 65380, '\P{^Quotation_Mark=no}', "");
Expect(0, 65379, '\p{Quotation_Mark=no}', "");
Expect(1, 65379, '\p{^Quotation_Mark=no}', "");
Expect(1, 65379, '\P{Quotation_Mark=no}', "");
Expect(0, 65379, '\P{^Quotation_Mark=no}', "");
Expect(1, 65380, '\p{Quotation_Mark=-	No}', "");
Expect(0, 65380, '\p{^Quotation_Mark=-	No}', "");
Expect(0, 65380, '\P{Quotation_Mark=-	No}', "");
Expect(1, 65380, '\P{^Quotation_Mark=-	No}', "");
Expect(0, 65379, '\p{Quotation_Mark=-	No}', "");
Expect(1, 65379, '\p{^Quotation_Mark=-	No}', "");
Expect(1, 65379, '\P{Quotation_Mark=-	No}', "");
Expect(0, 65379, '\P{^Quotation_Mark=-	No}', "");
Error('\p{QMark= :=N}');
Error('\P{QMark= :=N}');
Expect(1, 65380, '\p{QMark=n}', "");
Expect(0, 65380, '\p{^QMark=n}', "");
Expect(0, 65380, '\P{QMark=n}', "");
Expect(1, 65380, '\P{^QMark=n}', "");
Expect(0, 65379, '\p{QMark=n}', "");
Expect(1, 65379, '\p{^QMark=n}', "");
Expect(1, 65379, '\P{QMark=n}', "");
Expect(0, 65379, '\P{^QMark=n}', "");
Expect(1, 65380, '\p{QMark=  N}', "");
Expect(0, 65380, '\p{^QMark=  N}', "");
Expect(0, 65380, '\P{QMark=  N}', "");
Expect(1, 65380, '\P{^QMark=  N}', "");
Expect(0, 65379, '\p{QMark=  N}', "");
Expect(1, 65379, '\p{^QMark=  N}', "");
Expect(1, 65379, '\P{QMark=  N}', "");
Expect(0, 65379, '\P{^QMark=  N}', "");
Error('\p{Is_Quotation_Mark=:= 	F}');
Error('\P{Is_Quotation_Mark=:= 	F}');
Expect(1, 65380, '\p{Is_Quotation_Mark=f}', "");
Expect(0, 65380, '\p{^Is_Quotation_Mark=f}', "");
Expect(0, 65380, '\P{Is_Quotation_Mark=f}', "");
Expect(1, 65380, '\P{^Is_Quotation_Mark=f}', "");
Expect(0, 65379, '\p{Is_Quotation_Mark=f}', "");
Expect(1, 65379, '\p{^Is_Quotation_Mark=f}', "");
Expect(1, 65379, '\P{Is_Quotation_Mark=f}', "");
Expect(0, 65379, '\P{^Is_Quotation_Mark=f}', "");
Expect(1, 65380, '\p{Is_Quotation_Mark=-F}', "");
Expect(0, 65380, '\p{^Is_Quotation_Mark=-F}', "");
Expect(0, 65380, '\P{Is_Quotation_Mark=-F}', "");
Expect(1, 65380, '\P{^Is_Quotation_Mark=-F}', "");
Expect(0, 65379, '\p{Is_Quotation_Mark=-F}', "");
Expect(1, 65379, '\p{^Is_Quotation_Mark=-F}', "");
Expect(1, 65379, '\P{Is_Quotation_Mark=-F}', "");
Expect(0, 65379, '\P{^Is_Quotation_Mark=-F}', "");
Error('\p{Is_QMark=-/a/false}');
Error('\P{Is_QMark=-/a/false}');
Expect(1, 65380, '\p{Is_QMark=false}', "");
Expect(0, 65380, '\p{^Is_QMark=false}', "");
Expect(0, 65380, '\P{Is_QMark=false}', "");
Expect(1, 65380, '\P{^Is_QMark=false}', "");
Expect(0, 65379, '\p{Is_QMark=false}', "");
Expect(1, 65379, '\p{^Is_QMark=false}', "");
Expect(1, 65379, '\P{Is_QMark=false}', "");
Expect(0, 65379, '\P{^Is_QMark=false}', "");
Expect(1, 65380, '\p{Is_QMark=-False}', "");
Expect(0, 65380, '\p{^Is_QMark=-False}', "");
Expect(0, 65380, '\P{Is_QMark=-False}', "");
Expect(1, 65380, '\P{^Is_QMark=-False}', "");
Expect(0, 65379, '\p{Is_QMark=-False}', "");
Expect(1, 65379, '\p{^Is_QMark=-False}', "");
Expect(1, 65379, '\P{Is_QMark=-False}', "");
Expect(0, 65379, '\P{^Is_QMark=-False}', "");
Error('\p{Quotation_Mark=-/a/Yes}');
Error('\P{Quotation_Mark=-/a/Yes}');
Expect(1, 65379, '\p{Quotation_Mark=yes}', "");
Expect(0, 65379, '\p{^Quotation_Mark=yes}', "");
Expect(0, 65379, '\P{Quotation_Mark=yes}', "");
Expect(1, 65379, '\P{^Quotation_Mark=yes}', "");
Expect(0, 65380, '\p{Quotation_Mark=yes}', "");
Expect(1, 65380, '\p{^Quotation_Mark=yes}', "");
Expect(1, 65380, '\P{Quotation_Mark=yes}', "");
Expect(0, 65380, '\P{^Quotation_Mark=yes}', "");
Expect(1, 65379, '\p{Quotation_Mark=_-YES}', "");
Expect(0, 65379, '\p{^Quotation_Mark=_-YES}', "");
Expect(0, 65379, '\P{Quotation_Mark=_-YES}', "");
Expect(1, 65379, '\P{^Quotation_Mark=_-YES}', "");
Expect(0, 65380, '\p{Quotation_Mark=_-YES}', "");
Expect(1, 65380, '\p{^Quotation_Mark=_-YES}', "");
Expect(1, 65380, '\P{Quotation_Mark=_-YES}', "");
Expect(0, 65380, '\P{^Quotation_Mark=_-YES}', "");
Error('\p{QMark:	/a/ -Y}');
Error('\P{QMark:	/a/ -Y}');
Expect(1, 65379, '\p{QMark:	y}', "");
Expect(0, 65379, '\p{^QMark:	y}', "");
Expect(0, 65379, '\P{QMark:	y}', "");
Expect(1, 65379, '\P{^QMark:	y}', "");
Expect(0, 65380, '\p{QMark:	y}', "");
Expect(1, 65380, '\p{^QMark:	y}', "");
Expect(1, 65380, '\P{QMark:	y}', "");
Expect(0, 65380, '\P{^QMark:	y}', "");
Expect(1, 65379, '\p{QMark=Y}', "");
Expect(0, 65379, '\p{^QMark=Y}', "");
Expect(0, 65379, '\P{QMark=Y}', "");
Expect(1, 65379, '\P{^QMark=Y}', "");
Expect(0, 65380, '\p{QMark=Y}', "");
Expect(1, 65380, '\p{^QMark=Y}', "");
Expect(1, 65380, '\P{QMark=Y}', "");
Expect(0, 65380, '\P{^QMark=Y}', "");
Error('\p{Is_Quotation_Mark=	_T:=}');
Error('\P{Is_Quotation_Mark=	_T:=}');
Expect(1, 65379, '\p{Is_Quotation_Mark=t}', "");
Expect(0, 65379, '\p{^Is_Quotation_Mark=t}', "");
Expect(0, 65379, '\P{Is_Quotation_Mark=t}', "");
Expect(1, 65379, '\P{^Is_Quotation_Mark=t}', "");
Expect(0, 65380, '\p{Is_Quotation_Mark=t}', "");
Expect(1, 65380, '\p{^Is_Quotation_Mark=t}', "");
Expect(1, 65380, '\P{Is_Quotation_Mark=t}', "");
Expect(0, 65380, '\P{^Is_Quotation_Mark=t}', "");
Expect(1, 65379, '\p{Is_Quotation_Mark=-T}', "");
Expect(0, 65379, '\p{^Is_Quotation_Mark=-T}', "");
Expect(0, 65379, '\P{Is_Quotation_Mark=-T}', "");
Expect(1, 65379, '\P{^Is_Quotation_Mark=-T}', "");
Expect(0, 65380, '\p{Is_Quotation_Mark=-T}', "");
Expect(1, 65380, '\p{^Is_Quotation_Mark=-T}', "");
Expect(1, 65380, '\P{Is_Quotation_Mark=-T}', "");
Expect(0, 65380, '\P{^Is_Quotation_Mark=-T}', "");
Error('\p{Is_QMark=	:=TRUE}');
Error('\P{Is_QMark=	:=TRUE}');
Expect(1, 65379, '\p{Is_QMark=true}', "");
Expect(0, 65379, '\p{^Is_QMark=true}', "");
Expect(0, 65379, '\P{Is_QMark=true}', "");
Expect(1, 65379, '\P{^Is_QMark=true}', "");
Expect(0, 65380, '\p{Is_QMark=true}', "");
Expect(1, 65380, '\p{^Is_QMark=true}', "");
Expect(1, 65380, '\P{Is_QMark=true}', "");
Expect(0, 65380, '\P{^Is_QMark=true}', "");
Expect(1, 65379, '\p{Is_QMark=	True}', "");
Expect(0, 65379, '\p{^Is_QMark=	True}', "");
Expect(0, 65379, '\P{Is_QMark=	True}', "");
Expect(1, 65379, '\P{^Is_QMark=	True}', "");
Expect(0, 65380, '\p{Is_QMark=	True}', "");
Expect(1, 65380, '\p{^Is_QMark=	True}', "");
Expect(1, 65380, '\P{Is_QMark=	True}', "");
Expect(0, 65380, '\P{^Is_QMark=	True}', "");
Error('\p{Radical=:= _NO}');
Error('\P{Radical=:= _NO}');
Expect(1, 12246, '\p{Radical=no}', "");
Expect(0, 12246, '\p{^Radical=no}', "");
Expect(0, 12246, '\P{Radical=no}', "");
Expect(1, 12246, '\P{^Radical=no}', "");
Expect(0, 12245, '\p{Radical=no}', "");
Expect(1, 12245, '\p{^Radical=no}', "");
Expect(1, 12245, '\P{Radical=no}', "");
Expect(0, 12245, '\P{^Radical=no}', "");
Expect(1, 12246, '\p{Radical=- No}', "");
Expect(0, 12246, '\p{^Radical=- No}', "");
Expect(0, 12246, '\P{Radical=- No}', "");
Expect(1, 12246, '\P{^Radical=- No}', "");
Expect(0, 12245, '\p{Radical=- No}', "");
Expect(1, 12245, '\p{^Radical=- No}', "");
Expect(1, 12245, '\P{Radical=- No}', "");
Expect(0, 12245, '\P{^Radical=- No}', "");
Error('\p{Is_Radical=-_n/a/}');
Error('\P{Is_Radical=-_n/a/}');
Expect(1, 12246, '\p{Is_Radical=n}', "");
Expect(0, 12246, '\p{^Is_Radical=n}', "");
Expect(0, 12246, '\P{Is_Radical=n}', "");
Expect(1, 12246, '\P{^Is_Radical=n}', "");
Expect(0, 12245, '\p{Is_Radical=n}', "");
Expect(1, 12245, '\p{^Is_Radical=n}', "");
Expect(1, 12245, '\P{Is_Radical=n}', "");
Expect(0, 12245, '\P{^Is_Radical=n}', "");
Expect(1, 12246, '\p{Is_Radical= 	n}', "");
Expect(0, 12246, '\p{^Is_Radical= 	n}', "");
Expect(0, 12246, '\P{Is_Radical= 	n}', "");
Expect(1, 12246, '\P{^Is_Radical= 	n}', "");
Expect(0, 12245, '\p{Is_Radical= 	n}', "");
Expect(1, 12245, '\p{^Is_Radical= 	n}', "");
Expect(1, 12245, '\P{Is_Radical= 	n}', "");
Expect(0, 12245, '\P{^Is_Radical= 	n}', "");
Error('\p{Radical=:=	-f}');
Error('\P{Radical=:=	-f}');
Expect(1, 12246, '\p{Radical=f}', "");
Expect(0, 12246, '\p{^Radical=f}', "");
Expect(0, 12246, '\P{Radical=f}', "");
Expect(1, 12246, '\P{^Radical=f}', "");
Expect(0, 12245, '\p{Radical=f}', "");
Expect(1, 12245, '\p{^Radical=f}', "");
Expect(1, 12245, '\P{Radical=f}', "");
Expect(0, 12245, '\P{^Radical=f}', "");
Expect(1, 12246, '\p{Radical=-_F}', "");
Expect(0, 12246, '\p{^Radical=-_F}', "");
Expect(0, 12246, '\P{Radical=-_F}', "");
Expect(1, 12246, '\P{^Radical=-_F}', "");
Expect(0, 12245, '\p{Radical=-_F}', "");
Expect(1, 12245, '\p{^Radical=-_F}', "");
Expect(1, 12245, '\P{Radical=-_F}', "");
Expect(0, 12245, '\P{^Radical=-_F}', "");
Error('\p{Is_Radical=	-false:=}');
Error('\P{Is_Radical=	-false:=}');
Expect(1, 12246, '\p{Is_Radical=false}', "");
Expect(0, 12246, '\p{^Is_Radical=false}', "");
Expect(0, 12246, '\P{Is_Radical=false}', "");
Expect(1, 12246, '\P{^Is_Radical=false}', "");
Expect(0, 12245, '\p{Is_Radical=false}', "");
Expect(1, 12245, '\p{^Is_Radical=false}', "");
Expect(1, 12245, '\P{Is_Radical=false}', "");
Expect(0, 12245, '\P{^Is_Radical=false}', "");
Expect(1, 12246, '\p{Is_Radical=- False}', "");
Expect(0, 12246, '\p{^Is_Radical=- False}', "");
Expect(0, 12246, '\P{Is_Radical=- False}', "");
Expect(1, 12246, '\P{^Is_Radical=- False}', "");
Expect(0, 12245, '\p{Is_Radical=- False}', "");
Expect(1, 12245, '\p{^Is_Radical=- False}', "");
Expect(1, 12245, '\P{Is_Radical=- False}', "");
Expect(0, 12245, '\P{^Is_Radical=- False}', "");
Error('\p{Radical= /a/yes}');
Error('\P{Radical= /a/yes}');
Expect(1, 12245, '\p{Radical=yes}', "");
Expect(0, 12245, '\p{^Radical=yes}', "");
Expect(0, 12245, '\P{Radical=yes}', "");
Expect(1, 12245, '\P{^Radical=yes}', "");
Expect(0, 12246, '\p{Radical=yes}', "");
Expect(1, 12246, '\p{^Radical=yes}', "");
Expect(1, 12246, '\P{Radical=yes}', "");
Expect(0, 12246, '\P{^Radical=yes}', "");
Expect(1, 12245, '\p{Radical:  yes}', "");
Expect(0, 12245, '\p{^Radical:  yes}', "");
Expect(0, 12245, '\P{Radical:  yes}', "");
Expect(1, 12245, '\P{^Radical:  yes}', "");
Expect(0, 12246, '\p{Radical:  yes}', "");
Expect(1, 12246, '\p{^Radical:  yes}', "");
Expect(1, 12246, '\P{Radical:  yes}', "");
Expect(0, 12246, '\P{^Radical:  yes}', "");
Error('\p{Is_Radical=_	Y/a/}');
Error('\P{Is_Radical=_	Y/a/}');
Expect(1, 12245, '\p{Is_Radical=y}', "");
Expect(0, 12245, '\p{^Is_Radical=y}', "");
Expect(0, 12245, '\P{Is_Radical=y}', "");
Expect(1, 12245, '\P{^Is_Radical=y}', "");
Expect(0, 12246, '\p{Is_Radical=y}', "");
Expect(1, 12246, '\p{^Is_Radical=y}', "");
Expect(1, 12246, '\P{Is_Radical=y}', "");
Expect(0, 12246, '\P{^Is_Radical=y}', "");
Expect(1, 12245, '\p{Is_Radical= y}', "");
Expect(0, 12245, '\p{^Is_Radical= y}', "");
Expect(0, 12245, '\P{Is_Radical= y}', "");
Expect(1, 12245, '\P{^Is_Radical= y}', "");
Expect(0, 12246, '\p{Is_Radical= y}', "");
Expect(1, 12246, '\p{^Is_Radical= y}', "");
Expect(1, 12246, '\P{Is_Radical= y}', "");
Expect(0, 12246, '\P{^Is_Radical= y}', "");
Error('\p{Radical=_/a/T}');
Error('\P{Radical=_/a/T}');
Expect(1, 12245, '\p{Radical=t}', "");
Expect(0, 12245, '\p{^Radical=t}', "");
Expect(0, 12245, '\P{Radical=t}', "");
Expect(1, 12245, '\P{^Radical=t}', "");
Expect(0, 12246, '\p{Radical=t}', "");
Expect(1, 12246, '\p{^Radical=t}', "");
Expect(1, 12246, '\P{Radical=t}', "");
Expect(0, 12246, '\P{^Radical=t}', "");
Expect(1, 12245, '\p{Radical= 	T}', "");
Expect(0, 12245, '\p{^Radical= 	T}', "");
Expect(0, 12245, '\P{Radical= 	T}', "");
Expect(1, 12245, '\P{^Radical= 	T}', "");
Expect(0, 12246, '\p{Radical= 	T}', "");
Expect(1, 12246, '\p{^Radical= 	T}', "");
Expect(1, 12246, '\P{Radical= 	T}', "");
Expect(0, 12246, '\P{^Radical= 	T}', "");
Error('\p{Is_Radical: 	/a/TRUE}');
Error('\P{Is_Radical: 	/a/TRUE}');
Expect(1, 12245, '\p{Is_Radical:	true}', "");
Expect(0, 12245, '\p{^Is_Radical:	true}', "");
Expect(0, 12245, '\P{Is_Radical:	true}', "");
Expect(1, 12245, '\P{^Is_Radical:	true}', "");
Expect(0, 12246, '\p{Is_Radical:	true}', "");
Expect(1, 12246, '\p{^Is_Radical:	true}', "");
Expect(1, 12246, '\P{Is_Radical:	true}', "");
Expect(0, 12246, '\P{^Is_Radical:	true}', "");
Expect(1, 12245, '\p{Is_Radical=--True}', "");
Expect(0, 12245, '\p{^Is_Radical=--True}', "");
Expect(0, 12245, '\P{Is_Radical=--True}', "");
Expect(1, 12245, '\P{^Is_Radical=--True}', "");
Expect(0, 12246, '\p{Is_Radical=--True}', "");
Expect(1, 12246, '\p{^Is_Radical=--True}', "");
Expect(1, 12246, '\P{Is_Radical=--True}', "");
Expect(0, 12246, '\P{^Is_Radical=--True}', "");
Error('\p{Sentence_Break=:=_ATerm}');
Error('\P{Sentence_Break=:=_ATerm}');
Expect(1, 65294, '\p{Sentence_Break:   aterm}', "");
Expect(0, 65294, '\p{^Sentence_Break:   aterm}', "");
Expect(0, 65294, '\P{Sentence_Break:   aterm}', "");
Expect(1, 65294, '\P{^Sentence_Break:   aterm}', "");
Expect(0, 65295, '\p{Sentence_Break:   aterm}', "");
Expect(1, 65295, '\p{^Sentence_Break:   aterm}', "");
Expect(1, 65295, '\P{Sentence_Break:   aterm}', "");
Expect(0, 65295, '\P{^Sentence_Break:   aterm}', "");
Expect(1, 65294, '\p{Sentence_Break= 	aterm}', "");
Expect(0, 65294, '\p{^Sentence_Break= 	aterm}', "");
Expect(0, 65294, '\P{Sentence_Break= 	aterm}', "");
Expect(1, 65294, '\P{^Sentence_Break= 	aterm}', "");
Expect(0, 65295, '\p{Sentence_Break= 	aterm}', "");
Expect(1, 65295, '\p{^Sentence_Break= 	aterm}', "");
Expect(1, 65295, '\P{Sentence_Break= 	aterm}', "");
Expect(0, 65295, '\P{^Sentence_Break= 	aterm}', "");
Error('\p{SB=/a/ _AT}');
Error('\P{SB=/a/ _AT}');
Expect(1, 65294, '\p{SB=at}', "");
Expect(0, 65294, '\p{^SB=at}', "");
Expect(0, 65294, '\P{SB=at}', "");
Expect(1, 65294, '\P{^SB=at}', "");
Expect(0, 65295, '\p{SB=at}', "");
Expect(1, 65295, '\p{^SB=at}', "");
Expect(1, 65295, '\P{SB=at}', "");
Expect(0, 65295, '\P{^SB=at}', "");
Expect(1, 65294, '\p{SB=-AT}', "");
Expect(0, 65294, '\p{^SB=-AT}', "");
Expect(0, 65294, '\P{SB=-AT}', "");
Expect(1, 65294, '\P{^SB=-AT}', "");
Expect(0, 65295, '\p{SB=-AT}', "");
Expect(1, 65295, '\p{^SB=-AT}', "");
Expect(1, 65295, '\P{SB=-AT}', "");
Expect(0, 65295, '\P{^SB=-AT}', "");
Error('\p{Is_Sentence_Break=:=_ ATerm}');
Error('\P{Is_Sentence_Break=:=_ ATerm}');
Expect(1, 65294, '\p{Is_Sentence_Break:aterm}', "");
Expect(0, 65294, '\p{^Is_Sentence_Break:aterm}', "");
Expect(0, 65294, '\P{Is_Sentence_Break:aterm}', "");
Expect(1, 65294, '\P{^Is_Sentence_Break:aterm}', "");
Expect(0, 65295, '\p{Is_Sentence_Break:aterm}', "");
Expect(1, 65295, '\p{^Is_Sentence_Break:aterm}', "");
Expect(1, 65295, '\P{Is_Sentence_Break:aterm}', "");
Expect(0, 65295, '\P{^Is_Sentence_Break:aterm}', "");
Expect(1, 65294, '\p{Is_Sentence_Break=_	ATERM}', "");
Expect(0, 65294, '\p{^Is_Sentence_Break=_	ATERM}', "");
Expect(0, 65294, '\P{Is_Sentence_Break=_	ATERM}', "");
Expect(1, 65294, '\P{^Is_Sentence_Break=_	ATERM}', "");
Expect(0, 65295, '\p{Is_Sentence_Break=_	ATERM}', "");
Expect(1, 65295, '\p{^Is_Sentence_Break=_	ATERM}', "");
Expect(1, 65295, '\P{Is_Sentence_Break=_	ATERM}', "");
Expect(0, 65295, '\P{^Is_Sentence_Break=_	ATERM}', "");
Error('\p{Is_SB: /a/__AT}');
Error('\P{Is_SB: /a/__AT}');
Expect(1, 65294, '\p{Is_SB=at}', "");
Expect(0, 65294, '\p{^Is_SB=at}', "");
Expect(0, 65294, '\P{Is_SB=at}', "");
Expect(1, 65294, '\P{^Is_SB=at}', "");
Expect(0, 65295, '\p{Is_SB=at}', "");
Expect(1, 65295, '\p{^Is_SB=at}', "");
Expect(1, 65295, '\P{Is_SB=at}', "");
Expect(0, 65295, '\P{^Is_SB=at}', "");
Expect(1, 65294, '\p{Is_SB: 	AT}', "");
Expect(0, 65294, '\p{^Is_SB: 	AT}', "");
Expect(0, 65294, '\P{Is_SB: 	AT}', "");
Expect(1, 65294, '\P{^Is_SB: 	AT}', "");
Expect(0, 65295, '\p{Is_SB: 	AT}', "");
Expect(1, 65295, '\p{^Is_SB: 	AT}', "");
Expect(1, 65295, '\P{Is_SB: 	AT}', "");
Expect(0, 65295, '\P{^Is_SB: 	AT}', "");
Error('\p{Sentence_Break=/a/ Close}');
Error('\P{Sentence_Break=/a/ Close}');
Expect(1, 65379, '\p{Sentence_Break=close}', "");
Expect(0, 65379, '\p{^Sentence_Break=close}', "");
Expect(0, 65379, '\P{Sentence_Break=close}', "");
Expect(1, 65379, '\P{^Sentence_Break=close}', "");
Expect(0, 65380, '\p{Sentence_Break=close}', "");
Expect(1, 65380, '\p{^Sentence_Break=close}', "");
Expect(1, 65380, '\P{Sentence_Break=close}', "");
Expect(0, 65380, '\P{^Sentence_Break=close}', "");
Expect(1, 65379, '\p{Sentence_Break=	_CLOSE}', "");
Expect(0, 65379, '\p{^Sentence_Break=	_CLOSE}', "");
Expect(0, 65379, '\P{Sentence_Break=	_CLOSE}', "");
Expect(1, 65379, '\P{^Sentence_Break=	_CLOSE}', "");
Expect(0, 65380, '\p{Sentence_Break=	_CLOSE}', "");
Expect(1, 65380, '\p{^Sentence_Break=	_CLOSE}', "");
Expect(1, 65380, '\P{Sentence_Break=	_CLOSE}', "");
Expect(0, 65380, '\P{^Sentence_Break=	_CLOSE}', "");
Error('\p{SB=_cl/a/}');
Error('\P{SB=_cl/a/}');
Expect(1, 65379, '\p{SB=cl}', "");
Expect(0, 65379, '\p{^SB=cl}', "");
Expect(0, 65379, '\P{SB=cl}', "");
Expect(1, 65379, '\P{^SB=cl}', "");
Expect(0, 65380, '\p{SB=cl}', "");
Expect(1, 65380, '\p{^SB=cl}', "");
Expect(1, 65380, '\P{SB=cl}', "");
Expect(0, 65380, '\P{^SB=cl}', "");
Expect(1, 65379, '\p{SB=	CL}', "");
Expect(0, 65379, '\p{^SB=	CL}', "");
Expect(0, 65379, '\P{SB=	CL}', "");
Expect(1, 65379, '\P{^SB=	CL}', "");
Expect(0, 65380, '\p{SB=	CL}', "");
Expect(1, 65380, '\p{^SB=	CL}', "");
Expect(1, 65380, '\P{SB=	CL}', "");
Expect(0, 65380, '\P{^SB=	CL}', "");
Error('\p{Is_Sentence_Break= 	CLOSE/a/}');
Error('\P{Is_Sentence_Break= 	CLOSE/a/}');
Expect(1, 65379, '\p{Is_Sentence_Break=close}', "");
Expect(0, 65379, '\p{^Is_Sentence_Break=close}', "");
Expect(0, 65379, '\P{Is_Sentence_Break=close}', "");
Expect(1, 65379, '\P{^Is_Sentence_Break=close}', "");
Expect(0, 65380, '\p{Is_Sentence_Break=close}', "");
Expect(1, 65380, '\p{^Is_Sentence_Break=close}', "");
Expect(1, 65380, '\P{Is_Sentence_Break=close}', "");
Expect(0, 65380, '\P{^Is_Sentence_Break=close}', "");
Expect(1, 65379, '\p{Is_Sentence_Break=		Close}', "");
Expect(0, 65379, '\p{^Is_Sentence_Break=		Close}', "");
Expect(0, 65379, '\P{Is_Sentence_Break=		Close}', "");
Expect(1, 65379, '\P{^Is_Sentence_Break=		Close}', "");
Expect(0, 65380, '\p{Is_Sentence_Break=		Close}', "");
Expect(1, 65380, '\p{^Is_Sentence_Break=		Close}', "");
Expect(1, 65380, '\P{Is_Sentence_Break=		Close}', "");
Expect(0, 65380, '\P{^Is_Sentence_Break=		Close}', "");
Error('\p{Is_SB=_/a/CL}');
Error('\P{Is_SB=_/a/CL}');
Expect(1, 65379, '\p{Is_SB=cl}', "");
Expect(0, 65379, '\p{^Is_SB=cl}', "");
Expect(0, 65379, '\P{Is_SB=cl}', "");
Expect(1, 65379, '\P{^Is_SB=cl}', "");
Expect(0, 65380, '\p{Is_SB=cl}', "");
Expect(1, 65380, '\p{^Is_SB=cl}', "");
Expect(1, 65380, '\P{Is_SB=cl}', "");
Expect(0, 65380, '\P{^Is_SB=cl}', "");
Expect(1, 65379, '\p{Is_SB: -CL}', "");
Expect(0, 65379, '\p{^Is_SB: -CL}', "");
Expect(0, 65379, '\P{Is_SB: -CL}', "");
Expect(1, 65379, '\P{^Is_SB: -CL}', "");
Expect(0, 65380, '\p{Is_SB: -CL}', "");
Expect(1, 65380, '\p{^Is_SB: -CL}', "");
Expect(1, 65380, '\P{Is_SB: -CL}', "");
Expect(0, 65380, '\P{^Is_SB: -CL}', "");
Error('\p{Sentence_Break=/a/--CR}');
Error('\P{Sentence_Break=/a/--CR}');
Expect(1, 13, '\p{Sentence_Break=cr}', "");
Expect(0, 13, '\p{^Sentence_Break=cr}', "");
Expect(0, 13, '\P{Sentence_Break=cr}', "");
Expect(1, 13, '\P{^Sentence_Break=cr}', "");
Expect(0, 14, '\p{Sentence_Break=cr}', "");
Expect(1, 14, '\p{^Sentence_Break=cr}', "");
Expect(1, 14, '\P{Sentence_Break=cr}', "");
Expect(0, 14, '\P{^Sentence_Break=cr}', "");
Expect(1, 13, '\p{Sentence_Break=_ CR}', "");
Expect(0, 13, '\p{^Sentence_Break=_ CR}', "");
Expect(0, 13, '\P{Sentence_Break=_ CR}', "");
Expect(1, 13, '\P{^Sentence_Break=_ CR}', "");
Expect(0, 14, '\p{Sentence_Break=_ CR}', "");
Expect(1, 14, '\p{^Sentence_Break=_ CR}', "");
Expect(1, 14, '\P{Sentence_Break=_ CR}', "");
Expect(0, 14, '\P{^Sentence_Break=_ CR}', "");
Error('\p{SB=:=	CR}');
Error('\P{SB=:=	CR}');
Expect(1, 13, '\p{SB=cr}', "");
Expect(0, 13, '\p{^SB=cr}', "");
Expect(0, 13, '\P{SB=cr}', "");
Expect(1, 13, '\P{^SB=cr}', "");
Expect(0, 14, '\p{SB=cr}', "");
Expect(1, 14, '\p{^SB=cr}', "");
Expect(1, 14, '\P{SB=cr}', "");
Expect(0, 14, '\P{^SB=cr}', "");
Expect(1, 13, '\p{SB= -CR}', "");
Expect(0, 13, '\p{^SB= -CR}', "");
Expect(0, 13, '\P{SB= -CR}', "");
Expect(1, 13, '\P{^SB= -CR}', "");
Expect(0, 14, '\p{SB= -CR}', "");
Expect(1, 14, '\p{^SB= -CR}', "");
Expect(1, 14, '\P{SB= -CR}', "");
Expect(0, 14, '\P{^SB= -CR}', "");
Error('\p{Is_Sentence_Break=:=CR}');
Error('\P{Is_Sentence_Break=:=CR}');
Expect(1, 13, '\p{Is_Sentence_Break=cr}', "");
Expect(0, 13, '\p{^Is_Sentence_Break=cr}', "");
Expect(0, 13, '\P{Is_Sentence_Break=cr}', "");
Expect(1, 13, '\P{^Is_Sentence_Break=cr}', "");
Expect(0, 14, '\p{Is_Sentence_Break=cr}', "");
Expect(1, 14, '\p{^Is_Sentence_Break=cr}', "");
Expect(1, 14, '\P{Is_Sentence_Break=cr}', "");
Expect(0, 14, '\P{^Is_Sentence_Break=cr}', "");
Expect(1, 13, '\p{Is_Sentence_Break= CR}', "");
Expect(0, 13, '\p{^Is_Sentence_Break= CR}', "");
Expect(0, 13, '\P{Is_Sentence_Break= CR}', "");
Expect(1, 13, '\P{^Is_Sentence_Break= CR}', "");
Expect(0, 14, '\p{Is_Sentence_Break= CR}', "");
Expect(1, 14, '\p{^Is_Sentence_Break= CR}', "");
Expect(1, 14, '\P{Is_Sentence_Break= CR}', "");
Expect(0, 14, '\P{^Is_Sentence_Break= CR}', "");
Error('\p{Is_SB=_-cr/a/}');
Error('\P{Is_SB=_-cr/a/}');
Expect(1, 13, '\p{Is_SB=cr}', "");
Expect(0, 13, '\p{^Is_SB=cr}', "");
Expect(0, 13, '\P{Is_SB=cr}', "");
Expect(1, 13, '\P{^Is_SB=cr}', "");
Expect(0, 14, '\p{Is_SB=cr}', "");
Expect(1, 14, '\p{^Is_SB=cr}', "");
Expect(1, 14, '\P{Is_SB=cr}', "");
Expect(0, 14, '\P{^Is_SB=cr}', "");
Expect(1, 13, '\p{Is_SB=	CR}', "");
Expect(0, 13, '\p{^Is_SB=	CR}', "");
Expect(0, 13, '\P{Is_SB=	CR}', "");
Expect(1, 13, '\P{^Is_SB=	CR}', "");
Expect(0, 14, '\p{Is_SB=	CR}', "");
Expect(1, 14, '\p{^Is_SB=	CR}', "");
Expect(1, 14, '\P{Is_SB=	CR}', "");
Expect(0, 14, '\P{^Is_SB=	CR}', "");
Error('\p{Sentence_Break=/a/extend}');
Error('\P{Sentence_Break=/a/extend}');
Expect(1, 917999, '\p{Sentence_Break=extend}', "");
Expect(0, 917999, '\p{^Sentence_Break=extend}', "");
Expect(0, 917999, '\P{Sentence_Break=extend}', "");
Expect(1, 917999, '\P{^Sentence_Break=extend}', "");
Expect(0, 918000, '\p{Sentence_Break=extend}', "");
Expect(1, 918000, '\p{^Sentence_Break=extend}', "");
Expect(1, 918000, '\P{Sentence_Break=extend}', "");
Expect(0, 918000, '\P{^Sentence_Break=extend}', "");
Expect(1, 917999, '\p{Sentence_Break= 	EXTEND}', "");
Expect(0, 917999, '\p{^Sentence_Break= 	EXTEND}', "");
Expect(0, 917999, '\P{Sentence_Break= 	EXTEND}', "");
Expect(1, 917999, '\P{^Sentence_Break= 	EXTEND}', "");
Expect(0, 918000, '\p{Sentence_Break= 	EXTEND}', "");
Expect(1, 918000, '\p{^Sentence_Break= 	EXTEND}', "");
Expect(1, 918000, '\P{Sentence_Break= 	EXTEND}', "");
Expect(0, 918000, '\P{^Sentence_Break= 	EXTEND}', "");
Error('\p{SB=-ex/a/}');
Error('\P{SB=-ex/a/}');
Expect(1, 917999, '\p{SB=ex}', "");
Expect(0, 917999, '\p{^SB=ex}', "");
Expect(0, 917999, '\P{SB=ex}', "");
Expect(1, 917999, '\P{^SB=ex}', "");
Expect(0, 918000, '\p{SB=ex}', "");
Expect(1, 918000, '\p{^SB=ex}', "");
Expect(1, 918000, '\P{SB=ex}', "");
Expect(0, 918000, '\P{^SB=ex}', "");
Expect(1, 917999, '\p{SB= ex}', "");
Expect(0, 917999, '\p{^SB= ex}', "");
Expect(0, 917999, '\P{SB= ex}', "");
Expect(1, 917999, '\P{^SB= ex}', "");
Expect(0, 918000, '\p{SB= ex}', "");
Expect(1, 918000, '\p{^SB= ex}', "");
Expect(1, 918000, '\P{SB= ex}', "");
Expect(0, 918000, '\P{^SB= ex}', "");
Error('\p{Is_Sentence_Break=/a/Extend}');
Error('\P{Is_Sentence_Break=/a/Extend}');
Expect(1, 917999, '\p{Is_Sentence_Break=extend}', "");
Expect(0, 917999, '\p{^Is_Sentence_Break=extend}', "");
Expect(0, 917999, '\P{Is_Sentence_Break=extend}', "");
Expect(1, 917999, '\P{^Is_Sentence_Break=extend}', "");
Expect(0, 918000, '\p{Is_Sentence_Break=extend}', "");
Expect(1, 918000, '\p{^Is_Sentence_Break=extend}', "");
Expect(1, 918000, '\P{Is_Sentence_Break=extend}', "");
Expect(0, 918000, '\P{^Is_Sentence_Break=extend}', "");
Expect(1, 917999, '\p{Is_Sentence_Break=  EXTEND}', "");
Expect(0, 917999, '\p{^Is_Sentence_Break=  EXTEND}', "");
Expect(0, 917999, '\P{Is_Sentence_Break=  EXTEND}', "");
Expect(1, 917999, '\P{^Is_Sentence_Break=  EXTEND}', "");
Expect(0, 918000, '\p{Is_Sentence_Break=  EXTEND}', "");
Expect(1, 918000, '\p{^Is_Sentence_Break=  EXTEND}', "");
Expect(1, 918000, '\P{Is_Sentence_Break=  EXTEND}', "");
Expect(0, 918000, '\P{^Is_Sentence_Break=  EXTEND}', "");
Error('\p{Is_SB=_/a/ex}');
Error('\P{Is_SB=_/a/ex}');
Expect(1, 917999, '\p{Is_SB:   ex}', "");
Expect(0, 917999, '\p{^Is_SB:   ex}', "");
Expect(0, 917999, '\P{Is_SB:   ex}', "");
Expect(1, 917999, '\P{^Is_SB:   ex}', "");
Expect(0, 918000, '\p{Is_SB:   ex}', "");
Expect(1, 918000, '\p{^Is_SB:   ex}', "");
Expect(1, 918000, '\P{Is_SB:   ex}', "");
Expect(0, 918000, '\P{^Is_SB:   ex}', "");
Expect(1, 917999, '\p{Is_SB= 	EX}', "");
Expect(0, 917999, '\p{^Is_SB= 	EX}', "");
Expect(0, 917999, '\P{Is_SB= 	EX}', "");
Expect(1, 917999, '\P{^Is_SB= 	EX}', "");
Expect(0, 918000, '\p{Is_SB= 	EX}', "");
Expect(1, 918000, '\p{^Is_SB= 	EX}', "");
Expect(1, 918000, '\P{Is_SB= 	EX}', "");
Expect(0, 918000, '\P{^Is_SB= 	EX}', "");
Error('\p{Sentence_Break=:= Format}');
Error('\P{Sentence_Break=:= Format}');
Expect(1, 917631, '\p{Sentence_Break=format}', "");
Expect(0, 917631, '\p{^Sentence_Break=format}', "");
Expect(0, 917631, '\P{Sentence_Break=format}', "");
Expect(1, 917631, '\P{^Sentence_Break=format}', "");
Expect(0, 917632, '\p{Sentence_Break=format}', "");
Expect(1, 917632, '\p{^Sentence_Break=format}', "");
Expect(1, 917632, '\P{Sentence_Break=format}', "");
Expect(0, 917632, '\P{^Sentence_Break=format}', "");
Expect(1, 917631, '\p{Sentence_Break=	-Format}', "");
Expect(0, 917631, '\p{^Sentence_Break=	-Format}', "");
Expect(0, 917631, '\P{Sentence_Break=	-Format}', "");
Expect(1, 917631, '\P{^Sentence_Break=	-Format}', "");
Expect(0, 917632, '\p{Sentence_Break=	-Format}', "");
Expect(1, 917632, '\p{^Sentence_Break=	-Format}', "");
Expect(1, 917632, '\P{Sentence_Break=	-Format}', "");
Expect(0, 917632, '\P{^Sentence_Break=	-Format}', "");
Error('\p{SB=_	FO/a/}');
Error('\P{SB=_	FO/a/}');
Expect(1, 917631, '\p{SB:	fo}', "");
Expect(0, 917631, '\p{^SB:	fo}', "");
Expect(0, 917631, '\P{SB:	fo}', "");
Expect(1, 917631, '\P{^SB:	fo}', "");
Expect(0, 917632, '\p{SB:	fo}', "");
Expect(1, 917632, '\p{^SB:	fo}', "");
Expect(1, 917632, '\P{SB:	fo}', "");
Expect(0, 917632, '\P{^SB:	fo}', "");
Expect(1, 917631, '\p{SB=- FO}', "");
Expect(0, 917631, '\p{^SB=- FO}', "");
Expect(0, 917631, '\P{SB=- FO}', "");
Expect(1, 917631, '\P{^SB=- FO}', "");
Expect(0, 917632, '\p{SB=- FO}', "");
Expect(1, 917632, '\p{^SB=- FO}', "");
Expect(1, 917632, '\P{SB=- FO}', "");
Expect(0, 917632, '\P{^SB=- FO}', "");
Error('\p{Is_Sentence_Break=/a/ FORMAT}');
Error('\P{Is_Sentence_Break=/a/ FORMAT}');
Expect(1, 917631, '\p{Is_Sentence_Break=format}', "");
Expect(0, 917631, '\p{^Is_Sentence_Break=format}', "");
Expect(0, 917631, '\P{Is_Sentence_Break=format}', "");
Expect(1, 917631, '\P{^Is_Sentence_Break=format}', "");
Expect(0, 917632, '\p{Is_Sentence_Break=format}', "");
Expect(1, 917632, '\p{^Is_Sentence_Break=format}', "");
Expect(1, 917632, '\P{Is_Sentence_Break=format}', "");
Expect(0, 917632, '\P{^Is_Sentence_Break=format}', "");
Expect(1, 917631, '\p{Is_Sentence_Break=-	Format}', "");
Expect(0, 917631, '\p{^Is_Sentence_Break=-	Format}', "");
Expect(0, 917631, '\P{Is_Sentence_Break=-	Format}', "");
Expect(1, 917631, '\P{^Is_Sentence_Break=-	Format}', "");
Expect(0, 917632, '\p{Is_Sentence_Break=-	Format}', "");
Expect(1, 917632, '\p{^Is_Sentence_Break=-	Format}', "");
Expect(1, 917632, '\P{Is_Sentence_Break=-	Format}', "");
Expect(0, 917632, '\P{^Is_Sentence_Break=-	Format}', "");
Error('\p{Is_SB=/a/-_FO}');
Error('\P{Is_SB=/a/-_FO}');
Expect(1, 917631, '\p{Is_SB=fo}', "");
Expect(0, 917631, '\p{^Is_SB=fo}', "");
Expect(0, 917631, '\P{Is_SB=fo}', "");
Expect(1, 917631, '\P{^Is_SB=fo}', "");
Expect(0, 917632, '\p{Is_SB=fo}', "");
Expect(1, 917632, '\p{^Is_SB=fo}', "");
Expect(1, 917632, '\P{Is_SB=fo}', "");
Expect(0, 917632, '\P{^Is_SB=fo}', "");
Expect(1, 917631, '\p{Is_SB=	_FO}', "");
Expect(0, 917631, '\p{^Is_SB=	_FO}', "");
Expect(0, 917631, '\P{Is_SB=	_FO}', "");
Expect(1, 917631, '\P{^Is_SB=	_FO}', "");
Expect(0, 917632, '\p{Is_SB=	_FO}', "");
Expect(1, 917632, '\p{^Is_SB=	_FO}', "");
Expect(1, 917632, '\P{Is_SB=	_FO}', "");
Expect(0, 917632, '\P{^Is_SB=	_FO}', "");
Error('\p{Sentence_Break=/a/ OLetter}');
Error('\P{Sentence_Break=/a/ OLetter}');
Expect(1, 195101, '\p{Sentence_Break=oletter}', "");
Expect(0, 195101, '\p{^Sentence_Break=oletter}', "");
Expect(0, 195101, '\P{Sentence_Break=oletter}', "");
Expect(1, 195101, '\P{^Sentence_Break=oletter}', "");
Expect(0, 195102, '\p{Sentence_Break=oletter}', "");
Expect(1, 195102, '\p{^Sentence_Break=oletter}', "");
Expect(1, 195102, '\P{Sentence_Break=oletter}', "");
Expect(0, 195102, '\P{^Sentence_Break=oletter}', "");
Expect(1, 195101, '\p{Sentence_Break= oletter}', "");
Expect(0, 195101, '\p{^Sentence_Break= oletter}', "");
Expect(0, 195101, '\P{Sentence_Break= oletter}', "");
Expect(1, 195101, '\P{^Sentence_Break= oletter}', "");
Expect(0, 195102, '\p{Sentence_Break= oletter}', "");
Expect(1, 195102, '\p{^Sentence_Break= oletter}', "");
Expect(1, 195102, '\P{Sentence_Break= oletter}', "");
Expect(0, 195102, '\P{^Sentence_Break= oletter}', "");
Error('\p{SB=_LE/a/}');
Error('\P{SB=_LE/a/}');
Expect(1, 195101, '\p{SB=le}', "");
Expect(0, 195101, '\p{^SB=le}', "");
Expect(0, 195101, '\P{SB=le}', "");
Expect(1, 195101, '\P{^SB=le}', "");
Expect(0, 195102, '\p{SB=le}', "");
Expect(1, 195102, '\p{^SB=le}', "");
Expect(1, 195102, '\P{SB=le}', "");
Expect(0, 195102, '\P{^SB=le}', "");
Expect(1, 195101, '\p{SB=	le}', "");
Expect(0, 195101, '\p{^SB=	le}', "");
Expect(0, 195101, '\P{SB=	le}', "");
Expect(1, 195101, '\P{^SB=	le}', "");
Expect(0, 195102, '\p{SB=	le}', "");
Expect(1, 195102, '\p{^SB=	le}', "");
Expect(1, 195102, '\P{SB=	le}', "");
Expect(0, 195102, '\P{^SB=	le}', "");
Error('\p{Is_Sentence_Break=_	oletter:=}');
Error('\P{Is_Sentence_Break=_	oletter:=}');
Expect(1, 195101, '\p{Is_Sentence_Break=oletter}', "");
Expect(0, 195101, '\p{^Is_Sentence_Break=oletter}', "");
Expect(0, 195101, '\P{Is_Sentence_Break=oletter}', "");
Expect(1, 195101, '\P{^Is_Sentence_Break=oletter}', "");
Expect(0, 195102, '\p{Is_Sentence_Break=oletter}', "");
Expect(1, 195102, '\p{^Is_Sentence_Break=oletter}', "");
Expect(1, 195102, '\P{Is_Sentence_Break=oletter}', "");
Expect(0, 195102, '\P{^Is_Sentence_Break=oletter}', "");
Expect(1, 195101, '\p{Is_Sentence_Break= OLetter}', "");
Expect(0, 195101, '\p{^Is_Sentence_Break= OLetter}', "");
Expect(0, 195101, '\P{Is_Sentence_Break= OLetter}', "");
Expect(1, 195101, '\P{^Is_Sentence_Break= OLetter}', "");
Expect(0, 195102, '\p{Is_Sentence_Break= OLetter}', "");
Expect(1, 195102, '\p{^Is_Sentence_Break= OLetter}', "");
Expect(1, 195102, '\P{Is_Sentence_Break= OLetter}', "");
Expect(0, 195102, '\P{^Is_Sentence_Break= OLetter}', "");
Error('\p{Is_SB=:=-_LE}');
Error('\P{Is_SB=:=-_LE}');
Expect(1, 195101, '\p{Is_SB=le}', "");
Expect(0, 195101, '\p{^Is_SB=le}', "");
Expect(0, 195101, '\P{Is_SB=le}', "");
Expect(1, 195101, '\P{^Is_SB=le}', "");
Expect(0, 195102, '\p{Is_SB=le}', "");
Expect(1, 195102, '\p{^Is_SB=le}', "");
Expect(1, 195102, '\P{Is_SB=le}', "");
Expect(0, 195102, '\P{^Is_SB=le}', "");
Expect(1, 195101, '\p{Is_SB=_	LE}', "");
Expect(0, 195101, '\p{^Is_SB=_	LE}', "");
Expect(0, 195101, '\P{Is_SB=_	LE}', "");
Expect(1, 195101, '\P{^Is_SB=_	LE}', "");
Expect(0, 195102, '\p{Is_SB=_	LE}', "");
Expect(1, 195102, '\p{^Is_SB=_	LE}', "");
Expect(1, 195102, '\P{Is_SB=_	LE}', "");
Expect(0, 195102, '\P{^Is_SB=_	LE}', "");
Error('\p{Sentence_Break=	:=LF}');
Error('\P{Sentence_Break=	:=LF}');
Expect(1, 10, '\p{Sentence_Break=lf}', "");
Expect(0, 10, '\p{^Sentence_Break=lf}', "");
Expect(0, 10, '\P{Sentence_Break=lf}', "");
Expect(1, 10, '\P{^Sentence_Break=lf}', "");
Expect(0, 11, '\p{Sentence_Break=lf}', "");
Expect(1, 11, '\p{^Sentence_Break=lf}', "");
Expect(1, 11, '\P{Sentence_Break=lf}', "");
Expect(0, 11, '\P{^Sentence_Break=lf}', "");
Expect(1, 10, '\p{Sentence_Break:   -_lf}', "");
Expect(0, 10, '\p{^Sentence_Break:   -_lf}', "");
Expect(0, 10, '\P{Sentence_Break:   -_lf}', "");
Expect(1, 10, '\P{^Sentence_Break:   -_lf}', "");
Expect(0, 11, '\p{Sentence_Break:   -_lf}', "");
Expect(1, 11, '\p{^Sentence_Break:   -_lf}', "");
Expect(1, 11, '\P{Sentence_Break:   -_lf}', "");
Expect(0, 11, '\P{^Sentence_Break:   -_lf}', "");
Error('\p{SB= lf:=}');
Error('\P{SB= lf:=}');
Expect(1, 10, '\p{SB=lf}', "");
Expect(0, 10, '\p{^SB=lf}', "");
Expect(0, 10, '\P{SB=lf}', "");
Expect(1, 10, '\P{^SB=lf}', "");
Expect(0, 11, '\p{SB=lf}', "");
Expect(1, 11, '\p{^SB=lf}', "");
Expect(1, 11, '\P{SB=lf}', "");
Expect(0, 11, '\P{^SB=lf}', "");
Expect(1, 10, '\p{SB=	 LF}', "");
Expect(0, 10, '\p{^SB=	 LF}', "");
Expect(0, 10, '\P{SB=	 LF}', "");
Expect(1, 10, '\P{^SB=	 LF}', "");
Expect(0, 11, '\p{SB=	 LF}', "");
Expect(1, 11, '\p{^SB=	 LF}', "");
Expect(1, 11, '\P{SB=	 LF}', "");
Expect(0, 11, '\P{^SB=	 LF}', "");
Error('\p{Is_Sentence_Break=:=	-lf}');
Error('\P{Is_Sentence_Break=:=	-lf}');
Expect(1, 10, '\p{Is_Sentence_Break=lf}', "");
Expect(0, 10, '\p{^Is_Sentence_Break=lf}', "");
Expect(0, 10, '\P{Is_Sentence_Break=lf}', "");
Expect(1, 10, '\P{^Is_Sentence_Break=lf}', "");
Expect(0, 11, '\p{Is_Sentence_Break=lf}', "");
Expect(1, 11, '\p{^Is_Sentence_Break=lf}', "");
Expect(1, 11, '\P{Is_Sentence_Break=lf}', "");
Expect(0, 11, '\P{^Is_Sentence_Break=lf}', "");
Expect(1, 10, '\p{Is_Sentence_Break=	 LF}', "");
Expect(0, 10, '\p{^Is_Sentence_Break=	 LF}', "");
Expect(0, 10, '\P{Is_Sentence_Break=	 LF}', "");
Expect(1, 10, '\P{^Is_Sentence_Break=	 LF}', "");
Expect(0, 11, '\p{Is_Sentence_Break=	 LF}', "");
Expect(1, 11, '\p{^Is_Sentence_Break=	 LF}', "");
Expect(1, 11, '\P{Is_Sentence_Break=	 LF}', "");
Expect(0, 11, '\P{^Is_Sentence_Break=	 LF}', "");
Error('\p{Is_SB: /a/_-LF}');
Error('\P{Is_SB: /a/_-LF}');
Expect(1, 10, '\p{Is_SB:	lf}', "");
Expect(0, 10, '\p{^Is_SB:	lf}', "");
Expect(0, 10, '\P{Is_SB:	lf}', "");
Expect(1, 10, '\P{^Is_SB:	lf}', "");
Expect(0, 11, '\p{Is_SB:	lf}', "");
Expect(1, 11, '\p{^Is_SB:	lf}', "");
Expect(1, 11, '\P{Is_SB:	lf}', "");
Expect(0, 11, '\P{^Is_SB:	lf}', "");
Expect(1, 10, '\p{Is_SB: __LF}', "");
Expect(0, 10, '\p{^Is_SB: __LF}', "");
Expect(0, 10, '\P{Is_SB: __LF}', "");
Expect(1, 10, '\P{^Is_SB: __LF}', "");
Expect(0, 11, '\p{Is_SB: __LF}', "");
Expect(1, 11, '\p{^Is_SB: __LF}', "");
Expect(1, 11, '\P{Is_SB: __LF}', "");
Expect(0, 11, '\P{^Is_SB: __LF}', "");
Error('\p{Sentence_Break=-Lower:=}');
Error('\P{Sentence_Break=-Lower:=}');
Expect(1, 120779, '\p{Sentence_Break=lower}', "");
Expect(0, 120779, '\p{^Sentence_Break=lower}', "");
Expect(0, 120779, '\P{Sentence_Break=lower}', "");
Expect(1, 120779, '\P{^Sentence_Break=lower}', "");
Expect(0, 120780, '\p{Sentence_Break=lower}', "");
Expect(1, 120780, '\p{^Sentence_Break=lower}', "");
Expect(1, 120780, '\P{Sentence_Break=lower}', "");
Expect(0, 120780, '\P{^Sentence_Break=lower}', "");
Expect(1, 120779, '\p{Sentence_Break=-_Lower}', "");
Expect(0, 120779, '\p{^Sentence_Break=-_Lower}', "");
Expect(0, 120779, '\P{Sentence_Break=-_Lower}', "");
Expect(1, 120779, '\P{^Sentence_Break=-_Lower}', "");
Expect(0, 120780, '\p{Sentence_Break=-_Lower}', "");
Expect(1, 120780, '\p{^Sentence_Break=-_Lower}', "");
Expect(1, 120780, '\P{Sentence_Break=-_Lower}', "");
Expect(0, 120780, '\P{^Sentence_Break=-_Lower}', "");
Error('\p{SB=_/a/LO}');
Error('\P{SB=_/a/LO}');
Expect(1, 120779, '\p{SB=lo}', "");
Expect(0, 120779, '\p{^SB=lo}', "");
Expect(0, 120779, '\P{SB=lo}', "");
Expect(1, 120779, '\P{^SB=lo}', "");
Expect(0, 120780, '\p{SB=lo}', "");
Expect(1, 120780, '\p{^SB=lo}', "");
Expect(1, 120780, '\P{SB=lo}', "");
Expect(0, 120780, '\P{^SB=lo}', "");
Expect(1, 120779, '\p{SB=- LO}', "");
Expect(0, 120779, '\p{^SB=- LO}', "");
Expect(0, 120779, '\P{SB=- LO}', "");
Expect(1, 120779, '\P{^SB=- LO}', "");
Expect(0, 120780, '\p{SB=- LO}', "");
Expect(1, 120780, '\p{^SB=- LO}', "");
Expect(1, 120780, '\P{SB=- LO}', "");
Expect(0, 120780, '\P{^SB=- LO}', "");
Error('\p{Is_Sentence_Break=/a/	-Lower}');
Error('\P{Is_Sentence_Break=/a/	-Lower}');
Expect(1, 120779, '\p{Is_Sentence_Break=lower}', "");
Expect(0, 120779, '\p{^Is_Sentence_Break=lower}', "");
Expect(0, 120779, '\P{Is_Sentence_Break=lower}', "");
Expect(1, 120779, '\P{^Is_Sentence_Break=lower}', "");
Expect(0, 120780, '\p{Is_Sentence_Break=lower}', "");
Expect(1, 120780, '\p{^Is_Sentence_Break=lower}', "");
Expect(1, 120780, '\P{Is_Sentence_Break=lower}', "");
Expect(0, 120780, '\P{^Is_Sentence_Break=lower}', "");
Expect(1, 120779, '\p{Is_Sentence_Break=		lower}', "");
Expect(0, 120779, '\p{^Is_Sentence_Break=		lower}', "");
Expect(0, 120779, '\P{Is_Sentence_Break=		lower}', "");
Expect(1, 120779, '\P{^Is_Sentence_Break=		lower}', "");
Expect(0, 120780, '\p{Is_Sentence_Break=		lower}', "");
Expect(1, 120780, '\p{^Is_Sentence_Break=		lower}', "");
Expect(1, 120780, '\P{Is_Sentence_Break=		lower}', "");
Expect(0, 120780, '\P{^Is_Sentence_Break=		lower}', "");
Error('\p{Is_SB=/a/--lo}');
Error('\P{Is_SB=/a/--lo}');
Expect(1, 120779, '\p{Is_SB=lo}', "");
Expect(0, 120779, '\p{^Is_SB=lo}', "");
Expect(0, 120779, '\P{Is_SB=lo}', "");
Expect(1, 120779, '\P{^Is_SB=lo}', "");
Expect(0, 120780, '\p{Is_SB=lo}', "");
Expect(1, 120780, '\p{^Is_SB=lo}', "");
Expect(1, 120780, '\P{Is_SB=lo}', "");
Expect(0, 120780, '\P{^Is_SB=lo}', "");
Expect(1, 120779, '\p{Is_SB= LO}', "");
Expect(0, 120779, '\p{^Is_SB= LO}', "");
Expect(0, 120779, '\P{Is_SB= LO}', "");
Expect(1, 120779, '\P{^Is_SB= LO}', "");
Expect(0, 120780, '\p{Is_SB= LO}', "");
Expect(1, 120780, '\p{^Is_SB= LO}', "");
Expect(1, 120780, '\P{Is_SB= LO}', "");
Expect(0, 120780, '\P{^Is_SB= LO}', "");
Error('\p{Sentence_Break=:= _Numeric}');
Error('\P{Sentence_Break=:= _Numeric}');
Expect(1, 120831, '\p{Sentence_Break=numeric}', "");
Expect(0, 120831, '\p{^Sentence_Break=numeric}', "");
Expect(0, 120831, '\P{Sentence_Break=numeric}', "");
Expect(1, 120831, '\P{^Sentence_Break=numeric}', "");
Expect(0, 120832, '\p{Sentence_Break=numeric}', "");
Expect(1, 120832, '\p{^Sentence_Break=numeric}', "");
Expect(1, 120832, '\P{Sentence_Break=numeric}', "");
Expect(0, 120832, '\P{^Sentence_Break=numeric}', "");
Expect(1, 120831, '\p{Sentence_Break=_ Numeric}', "");
Expect(0, 120831, '\p{^Sentence_Break=_ Numeric}', "");
Expect(0, 120831, '\P{Sentence_Break=_ Numeric}', "");
Expect(1, 120831, '\P{^Sentence_Break=_ Numeric}', "");
Expect(0, 120832, '\p{Sentence_Break=_ Numeric}', "");
Expect(1, 120832, '\p{^Sentence_Break=_ Numeric}', "");
Expect(1, 120832, '\P{Sentence_Break=_ Numeric}', "");
Expect(0, 120832, '\P{^Sentence_Break=_ Numeric}', "");
Error('\p{SB=/a/-_NU}');
Error('\P{SB=/a/-_NU}');
Expect(1, 120831, '\p{SB=nu}', "");
Expect(0, 120831, '\p{^SB=nu}', "");
Expect(0, 120831, '\P{SB=nu}', "");
Expect(1, 120831, '\P{^SB=nu}', "");
Expect(0, 120832, '\p{SB=nu}', "");
Expect(1, 120832, '\p{^SB=nu}', "");
Expect(1, 120832, '\P{SB=nu}', "");
Expect(0, 120832, '\P{^SB=nu}', "");
Expect(1, 120831, '\p{SB=__NU}', "");
Expect(0, 120831, '\p{^SB=__NU}', "");
Expect(0, 120831, '\P{SB=__NU}', "");
Expect(1, 120831, '\P{^SB=__NU}', "");
Expect(0, 120832, '\p{SB=__NU}', "");
Expect(1, 120832, '\p{^SB=__NU}', "");
Expect(1, 120832, '\P{SB=__NU}', "");
Expect(0, 120832, '\P{^SB=__NU}', "");
Error('\p{Is_Sentence_Break=_/a/numeric}');
Error('\P{Is_Sentence_Break=_/a/numeric}');
Expect(1, 120831, '\p{Is_Sentence_Break=numeric}', "");
Expect(0, 120831, '\p{^Is_Sentence_Break=numeric}', "");
Expect(0, 120831, '\P{Is_Sentence_Break=numeric}', "");
Expect(1, 120831, '\P{^Is_Sentence_Break=numeric}', "");
Expect(0, 120832, '\p{Is_Sentence_Break=numeric}', "");
Expect(1, 120832, '\p{^Is_Sentence_Break=numeric}', "");
Expect(1, 120832, '\P{Is_Sentence_Break=numeric}', "");
Expect(0, 120832, '\P{^Is_Sentence_Break=numeric}', "");
Expect(1, 120831, '\p{Is_Sentence_Break= NUMERIC}', "");
Expect(0, 120831, '\p{^Is_Sentence_Break= NUMERIC}', "");
Expect(0, 120831, '\P{Is_Sentence_Break= NUMERIC}', "");
Expect(1, 120831, '\P{^Is_Sentence_Break= NUMERIC}', "");
Expect(0, 120832, '\p{Is_Sentence_Break= NUMERIC}', "");
Expect(1, 120832, '\p{^Is_Sentence_Break= NUMERIC}', "");
Expect(1, 120832, '\P{Is_Sentence_Break= NUMERIC}', "");
Expect(0, 120832, '\P{^Is_Sentence_Break= NUMERIC}', "");
Error('\p{Is_SB=-:=NU}');
Error('\P{Is_SB=-:=NU}');
Expect(1, 120831, '\p{Is_SB=nu}', "");
Expect(0, 120831, '\p{^Is_SB=nu}', "");
Expect(0, 120831, '\P{Is_SB=nu}', "");
Expect(1, 120831, '\P{^Is_SB=nu}', "");
Expect(0, 120832, '\p{Is_SB=nu}', "");
Expect(1, 120832, '\p{^Is_SB=nu}', "");
Expect(1, 120832, '\P{Is_SB=nu}', "");
Expect(0, 120832, '\P{^Is_SB=nu}', "");
Expect(1, 120831, '\p{Is_SB=_	nu}', "");
Expect(0, 120831, '\p{^Is_SB=_	nu}', "");
Expect(0, 120831, '\P{Is_SB=_	nu}', "");
Expect(1, 120831, '\P{^Is_SB=_	nu}', "");
Expect(0, 120832, '\p{Is_SB=_	nu}', "");
Expect(1, 120832, '\p{^Is_SB=_	nu}', "");
Expect(1, 120832, '\P{Is_SB=_	nu}', "");
Expect(0, 120832, '\P{^Is_SB=_	nu}', "");
Error('\p{Sentence_Break=	/a/SContinue}');
Error('\P{Sentence_Break=	/a/SContinue}');
Expect(1, 65380, '\p{Sentence_Break=scontinue}', "");
Expect(0, 65380, '\p{^Sentence_Break=scontinue}', "");
Expect(0, 65380, '\P{Sentence_Break=scontinue}', "");
Expect(1, 65380, '\P{^Sentence_Break=scontinue}', "");
Expect(0, 65381, '\p{Sentence_Break=scontinue}', "");
Expect(1, 65381, '\p{^Sentence_Break=scontinue}', "");
Expect(1, 65381, '\P{Sentence_Break=scontinue}', "");
Expect(0, 65381, '\P{^Sentence_Break=scontinue}', "");
Expect(1, 65380, '\p{Sentence_Break:			SContinue}', "");
Expect(0, 65380, '\p{^Sentence_Break:			SContinue}', "");
Expect(0, 65380, '\P{Sentence_Break:			SContinue}', "");
Expect(1, 65380, '\P{^Sentence_Break:			SContinue}', "");
Expect(0, 65381, '\p{Sentence_Break:			SContinue}', "");
Expect(1, 65381, '\p{^Sentence_Break:			SContinue}', "");
Expect(1, 65381, '\P{Sentence_Break:			SContinue}', "");
Expect(0, 65381, '\P{^Sentence_Break:			SContinue}', "");
Error('\p{SB=	sc/a/}');
Error('\P{SB=	sc/a/}');
Expect(1, 65380, '\p{SB:	sc}', "");
Expect(0, 65380, '\p{^SB:	sc}', "");
Expect(0, 65380, '\P{SB:	sc}', "");
Expect(1, 65380, '\P{^SB:	sc}', "");
Expect(0, 65381, '\p{SB:	sc}', "");
Expect(1, 65381, '\p{^SB:	sc}', "");
Expect(1, 65381, '\P{SB:	sc}', "");
Expect(0, 65381, '\P{^SB:	sc}', "");
Expect(1, 65380, '\p{SB=_ SC}', "");
Expect(0, 65380, '\p{^SB=_ SC}', "");
Expect(0, 65380, '\P{SB=_ SC}', "");
Expect(1, 65380, '\P{^SB=_ SC}', "");
Expect(0, 65381, '\p{SB=_ SC}', "");
Expect(1, 65381, '\p{^SB=_ SC}', "");
Expect(1, 65381, '\P{SB=_ SC}', "");
Expect(0, 65381, '\P{^SB=_ SC}', "");
Error('\p{Is_Sentence_Break=:=		scontinue}');
Error('\P{Is_Sentence_Break=:=		scontinue}');
Expect(1, 65380, '\p{Is_Sentence_Break=scontinue}', "");
Expect(0, 65380, '\p{^Is_Sentence_Break=scontinue}', "");
Expect(0, 65380, '\P{Is_Sentence_Break=scontinue}', "");
Expect(1, 65380, '\P{^Is_Sentence_Break=scontinue}', "");
Expect(0, 65381, '\p{Is_Sentence_Break=scontinue}', "");
Expect(1, 65381, '\p{^Is_Sentence_Break=scontinue}', "");
Expect(1, 65381, '\P{Is_Sentence_Break=scontinue}', "");
Expect(0, 65381, '\P{^Is_Sentence_Break=scontinue}', "");
Expect(1, 65380, '\p{Is_Sentence_Break: _-SContinue}', "");
Expect(0, 65380, '\p{^Is_Sentence_Break: _-SContinue}', "");
Expect(0, 65380, '\P{Is_Sentence_Break: _-SContinue}', "");
Expect(1, 65380, '\P{^Is_Sentence_Break: _-SContinue}', "");
Expect(0, 65381, '\p{Is_Sentence_Break: _-SContinue}', "");
Expect(1, 65381, '\p{^Is_Sentence_Break: _-SContinue}', "");
Expect(1, 65381, '\P{Is_Sentence_Break: _-SContinue}', "");
Expect(0, 65381, '\P{^Is_Sentence_Break: _-SContinue}', "");
Error('\p{Is_SB= /a/SC}');
Error('\P{Is_SB= /a/SC}');
Expect(1, 65380, '\p{Is_SB=sc}', "");
Expect(0, 65380, '\p{^Is_SB=sc}', "");
Expect(0, 65380, '\P{Is_SB=sc}', "");
Expect(1, 65380, '\P{^Is_SB=sc}', "");
Expect(0, 65381, '\p{Is_SB=sc}', "");
Expect(1, 65381, '\p{^Is_SB=sc}', "");
Expect(1, 65381, '\P{Is_SB=sc}', "");
Expect(0, 65381, '\P{^Is_SB=sc}', "");
Expect(1, 65380, '\p{Is_SB=		SC}', "");
Expect(0, 65380, '\p{^Is_SB=		SC}', "");
Expect(0, 65380, '\P{Is_SB=		SC}', "");
Expect(1, 65380, '\P{^Is_SB=		SC}', "");
Expect(0, 65381, '\p{Is_SB=		SC}', "");
Expect(1, 65381, '\p{^Is_SB=		SC}', "");
Expect(1, 65381, '\P{Is_SB=		SC}', "");
Expect(0, 65381, '\P{^Is_SB=		SC}', "");
Error('\p{Sentence_Break=_:=SEP}');
Error('\P{Sentence_Break=_:=SEP}');
Expect(1, 8233, '\p{Sentence_Break=sep}', "");
Expect(0, 8233, '\p{^Sentence_Break=sep}', "");
Expect(0, 8233, '\P{Sentence_Break=sep}', "");
Expect(1, 8233, '\P{^Sentence_Break=sep}', "");
Expect(0, 8234, '\p{Sentence_Break=sep}', "");
Expect(1, 8234, '\p{^Sentence_Break=sep}', "");
Expect(1, 8234, '\P{Sentence_Break=sep}', "");
Expect(0, 8234, '\P{^Sentence_Break=sep}', "");
Expect(1, 8233, '\p{Sentence_Break=-	sep}', "");
Expect(0, 8233, '\p{^Sentence_Break=-	sep}', "");
Expect(0, 8233, '\P{Sentence_Break=-	sep}', "");
Expect(1, 8233, '\P{^Sentence_Break=-	sep}', "");
Expect(0, 8234, '\p{Sentence_Break=-	sep}', "");
Expect(1, 8234, '\p{^Sentence_Break=-	sep}', "");
Expect(1, 8234, '\P{Sentence_Break=-	sep}', "");
Expect(0, 8234, '\P{^Sentence_Break=-	sep}', "");
Error('\p{SB= _se/a/}');
Error('\P{SB= _se/a/}');
Expect(1, 8233, '\p{SB=se}', "");
Expect(0, 8233, '\p{^SB=se}', "");
Expect(0, 8233, '\P{SB=se}', "");
Expect(1, 8233, '\P{^SB=se}', "");
Expect(0, 8234, '\p{SB=se}', "");
Expect(1, 8234, '\p{^SB=se}', "");
Expect(1, 8234, '\P{SB=se}', "");
Expect(0, 8234, '\P{^SB=se}', "");
Expect(1, 8233, '\p{SB= 	SE}', "");
Expect(0, 8233, '\p{^SB= 	SE}', "");
Expect(0, 8233, '\P{SB= 	SE}', "");
Expect(1, 8233, '\P{^SB= 	SE}', "");
Expect(0, 8234, '\p{SB= 	SE}', "");
Expect(1, 8234, '\p{^SB= 	SE}', "");
Expect(1, 8234, '\P{SB= 	SE}', "");
Expect(0, 8234, '\P{^SB= 	SE}', "");
Error('\p{Is_Sentence_Break=:=	SEP}');
Error('\P{Is_Sentence_Break=:=	SEP}');
Expect(1, 8233, '\p{Is_Sentence_Break=sep}', "");
Expect(0, 8233, '\p{^Is_Sentence_Break=sep}', "");
Expect(0, 8233, '\P{Is_Sentence_Break=sep}', "");
Expect(1, 8233, '\P{^Is_Sentence_Break=sep}', "");
Expect(0, 8234, '\p{Is_Sentence_Break=sep}', "");
Expect(1, 8234, '\p{^Is_Sentence_Break=sep}', "");
Expect(1, 8234, '\P{Is_Sentence_Break=sep}', "");
Expect(0, 8234, '\P{^Is_Sentence_Break=sep}', "");
Expect(1, 8233, '\p{Is_Sentence_Break=--SEP}', "");
Expect(0, 8233, '\p{^Is_Sentence_Break=--SEP}', "");
Expect(0, 8233, '\P{Is_Sentence_Break=--SEP}', "");
Expect(1, 8233, '\P{^Is_Sentence_Break=--SEP}', "");
Expect(0, 8234, '\p{Is_Sentence_Break=--SEP}', "");
Expect(1, 8234, '\p{^Is_Sentence_Break=--SEP}', "");
Expect(1, 8234, '\P{Is_Sentence_Break=--SEP}', "");
Expect(0, 8234, '\P{^Is_Sentence_Break=--SEP}', "");
Error('\p{Is_SB::=	 SE}');
Error('\P{Is_SB::=	 SE}');
Expect(1, 8233, '\p{Is_SB:se}', "");
Expect(0, 8233, '\p{^Is_SB:se}', "");
Expect(0, 8233, '\P{Is_SB:se}', "");
Expect(1, 8233, '\P{^Is_SB:se}', "");
Expect(0, 8234, '\p{Is_SB:se}', "");
Expect(1, 8234, '\p{^Is_SB:se}', "");
Expect(1, 8234, '\P{Is_SB:se}', "");
Expect(0, 8234, '\P{^Is_SB:se}', "");
Expect(1, 8233, '\p{Is_SB=- SE}', "");
Expect(0, 8233, '\p{^Is_SB=- SE}', "");
Expect(0, 8233, '\P{Is_SB=- SE}', "");
Expect(1, 8233, '\P{^Is_SB=- SE}', "");
Expect(0, 8234, '\p{Is_SB=- SE}', "");
Expect(1, 8234, '\p{^Is_SB=- SE}', "");
Expect(1, 8234, '\P{Is_SB=- SE}', "");
Expect(0, 8234, '\P{^Is_SB=- SE}', "");
Error('\p{Sentence_Break=SP:=}');
Error('\P{Sentence_Break=SP:=}');
Expect(1, 12288, '\p{Sentence_Break:	sp}', "");
Expect(0, 12288, '\p{^Sentence_Break:	sp}', "");
Expect(0, 12288, '\P{Sentence_Break:	sp}', "");
Expect(1, 12288, '\P{^Sentence_Break:	sp}', "");
Expect(0, 12289, '\p{Sentence_Break:	sp}', "");
Expect(1, 12289, '\p{^Sentence_Break:	sp}', "");
Expect(1, 12289, '\P{Sentence_Break:	sp}', "");
Expect(0, 12289, '\P{^Sentence_Break:	sp}', "");
Expect(1, 12288, '\p{Sentence_Break= SP}', "");
Expect(0, 12288, '\p{^Sentence_Break= SP}', "");
Expect(0, 12288, '\P{Sentence_Break= SP}', "");
Expect(1, 12288, '\P{^Sentence_Break= SP}', "");
Expect(0, 12289, '\p{Sentence_Break= SP}', "");
Expect(1, 12289, '\p{^Sentence_Break= SP}', "");
Expect(1, 12289, '\P{Sentence_Break= SP}', "");
Expect(0, 12289, '\P{^Sentence_Break= SP}', "");
Error('\p{SB=/a/ SP}');
Error('\P{SB=/a/ SP}');
Expect(1, 12288, '\p{SB=sp}', "");
Expect(0, 12288, '\p{^SB=sp}', "");
Expect(0, 12288, '\P{SB=sp}', "");
Expect(1, 12288, '\P{^SB=sp}', "");
Expect(0, 12289, '\p{SB=sp}', "");
Expect(1, 12289, '\p{^SB=sp}', "");
Expect(1, 12289, '\P{SB=sp}', "");
Expect(0, 12289, '\P{^SB=sp}', "");
Expect(1, 12288, '\p{SB= _Sp}', "");
Expect(0, 12288, '\p{^SB= _Sp}', "");
Expect(0, 12288, '\P{SB= _Sp}', "");
Expect(1, 12288, '\P{^SB= _Sp}', "");
Expect(0, 12289, '\p{SB= _Sp}', "");
Expect(1, 12289, '\p{^SB= _Sp}', "");
Expect(1, 12289, '\P{SB= _Sp}', "");
Expect(0, 12289, '\P{^SB= _Sp}', "");
Error('\p{Is_Sentence_Break=:=_ sp}');
Error('\P{Is_Sentence_Break=:=_ sp}');
Expect(1, 12288, '\p{Is_Sentence_Break=sp}', "");
Expect(0, 12288, '\p{^Is_Sentence_Break=sp}', "");
Expect(0, 12288, '\P{Is_Sentence_Break=sp}', "");
Expect(1, 12288, '\P{^Is_Sentence_Break=sp}', "");
Expect(0, 12289, '\p{Is_Sentence_Break=sp}', "");
Expect(1, 12289, '\p{^Is_Sentence_Break=sp}', "");
Expect(1, 12289, '\P{Is_Sentence_Break=sp}', "");
Expect(0, 12289, '\P{^Is_Sentence_Break=sp}', "");
Expect(1, 12288, '\p{Is_Sentence_Break= Sp}', "");
Expect(0, 12288, '\p{^Is_Sentence_Break= Sp}', "");
Expect(0, 12288, '\P{Is_Sentence_Break= Sp}', "");
Expect(1, 12288, '\P{^Is_Sentence_Break= Sp}', "");
Expect(0, 12289, '\p{Is_Sentence_Break= Sp}', "");
Expect(1, 12289, '\p{^Is_Sentence_Break= Sp}', "");
Expect(1, 12289, '\P{Is_Sentence_Break= Sp}', "");
Expect(0, 12289, '\P{^Is_Sentence_Break= Sp}', "");
Error('\p{Is_SB=/a/	Sp}');
Error('\P{Is_SB=/a/	Sp}');
Expect(1, 12288, '\p{Is_SB=sp}', "");
Expect(0, 12288, '\p{^Is_SB=sp}', "");
Expect(0, 12288, '\P{Is_SB=sp}', "");
Expect(1, 12288, '\P{^Is_SB=sp}', "");
Expect(0, 12289, '\p{Is_SB=sp}', "");
Expect(1, 12289, '\p{^Is_SB=sp}', "");
Expect(1, 12289, '\P{Is_SB=sp}', "");
Expect(0, 12289, '\P{^Is_SB=sp}', "");
Expect(1, 12288, '\p{Is_SB=_	SP}', "");
Expect(0, 12288, '\p{^Is_SB=_	SP}', "");
Expect(0, 12288, '\P{Is_SB=_	SP}', "");
Expect(1, 12288, '\P{^Is_SB=_	SP}', "");
Expect(0, 12289, '\p{Is_SB=_	SP}', "");
Expect(1, 12289, '\p{^Is_SB=_	SP}', "");
Expect(1, 12289, '\P{Is_SB=_	SP}', "");
Expect(0, 12289, '\P{^Is_SB=_	SP}', "");
Error('\p{Sentence_Break=		sterm/a/}');
Error('\P{Sentence_Break=		sterm/a/}');
Expect(1, 70086, '\p{Sentence_Break=sterm}', "");
Expect(0, 70086, '\p{^Sentence_Break=sterm}', "");
Expect(0, 70086, '\P{Sentence_Break=sterm}', "");
Expect(1, 70086, '\P{^Sentence_Break=sterm}', "");
Expect(0, 70087, '\p{Sentence_Break=sterm}', "");
Expect(1, 70087, '\p{^Sentence_Break=sterm}', "");
Expect(1, 70087, '\P{Sentence_Break=sterm}', "");
Expect(0, 70087, '\P{^Sentence_Break=sterm}', "");
Expect(1, 70086, '\p{Sentence_Break=-STerm}', "");
Expect(0, 70086, '\p{^Sentence_Break=-STerm}', "");
Expect(0, 70086, '\P{Sentence_Break=-STerm}', "");
Expect(1, 70086, '\P{^Sentence_Break=-STerm}', "");
Expect(0, 70087, '\p{Sentence_Break=-STerm}', "");
Expect(1, 70087, '\p{^Sentence_Break=-STerm}', "");
Expect(1, 70087, '\P{Sentence_Break=-STerm}', "");
Expect(0, 70087, '\P{^Sentence_Break=-STerm}', "");
Error('\p{SB=	:=ST}');
Error('\P{SB=	:=ST}');
Expect(1, 70086, '\p{SB=st}', "");
Expect(0, 70086, '\p{^SB=st}', "");
Expect(0, 70086, '\P{SB=st}', "");
Expect(1, 70086, '\P{^SB=st}', "");
Expect(0, 70087, '\p{SB=st}', "");
Expect(1, 70087, '\p{^SB=st}', "");
Expect(1, 70087, '\P{SB=st}', "");
Expect(0, 70087, '\P{^SB=st}', "");
Expect(1, 70086, '\p{SB= ST}', "");
Expect(0, 70086, '\p{^SB= ST}', "");
Expect(0, 70086, '\P{SB= ST}', "");
Expect(1, 70086, '\P{^SB= ST}', "");
Expect(0, 70087, '\p{SB= ST}', "");
Expect(1, 70087, '\p{^SB= ST}', "");
Expect(1, 70087, '\P{SB= ST}', "");
Expect(0, 70087, '\P{^SB= ST}', "");
Error('\p{Is_Sentence_Break=	-STerm:=}');
Error('\P{Is_Sentence_Break=	-STerm:=}');
Expect(1, 70086, '\p{Is_Sentence_Break=sterm}', "");
Expect(0, 70086, '\p{^Is_Sentence_Break=sterm}', "");
Expect(0, 70086, '\P{Is_Sentence_Break=sterm}', "");
Expect(1, 70086, '\P{^Is_Sentence_Break=sterm}', "");
Expect(0, 70087, '\p{Is_Sentence_Break=sterm}', "");
Expect(1, 70087, '\p{^Is_Sentence_Break=sterm}', "");
Expect(1, 70087, '\P{Is_Sentence_Break=sterm}', "");
Expect(0, 70087, '\P{^Is_Sentence_Break=sterm}', "");
Expect(1, 70086, '\p{Is_Sentence_Break=  STerm}', "");
Expect(0, 70086, '\p{^Is_Sentence_Break=  STerm}', "");
Expect(0, 70086, '\P{Is_Sentence_Break=  STerm}', "");
Expect(1, 70086, '\P{^Is_Sentence_Break=  STerm}', "");
Expect(0, 70087, '\p{Is_Sentence_Break=  STerm}', "");
Expect(1, 70087, '\p{^Is_Sentence_Break=  STerm}', "");
Expect(1, 70087, '\P{Is_Sentence_Break=  STerm}', "");
Expect(0, 70087, '\P{^Is_Sentence_Break=  STerm}', "");
Error('\p{Is_SB=/a/ st}');
Error('\P{Is_SB=/a/ st}');
Expect(1, 70086, '\p{Is_SB=st}', "");
Expect(0, 70086, '\p{^Is_SB=st}', "");
Expect(0, 70086, '\P{Is_SB=st}', "");
Expect(1, 70086, '\P{^Is_SB=st}', "");
Expect(0, 70087, '\p{Is_SB=st}', "");
Expect(1, 70087, '\p{^Is_SB=st}', "");
Expect(1, 70087, '\P{Is_SB=st}', "");
Expect(0, 70087, '\P{^Is_SB=st}', "");
Expect(1, 70086, '\p{Is_SB=_	ST}', "");
Expect(0, 70086, '\p{^Is_SB=_	ST}', "");
Expect(0, 70086, '\P{Is_SB=_	ST}', "");
Expect(1, 70086, '\P{^Is_SB=_	ST}', "");
Expect(0, 70087, '\p{Is_SB=_	ST}', "");
Expect(1, 70087, '\p{^Is_SB=_	ST}', "");
Expect(1, 70087, '\P{Is_SB=_	ST}', "");
Expect(0, 70087, '\P{^Is_SB=_	ST}', "");
Error('\p{Sentence_Break:		Upper:=}');
Error('\P{Sentence_Break:		Upper:=}');
Expect(1, 120778, '\p{Sentence_Break=upper}', "");
Expect(0, 120778, '\p{^Sentence_Break=upper}', "");
Expect(0, 120778, '\P{Sentence_Break=upper}', "");
Expect(1, 120778, '\P{^Sentence_Break=upper}', "");
Expect(0, 120779, '\p{Sentence_Break=upper}', "");
Expect(1, 120779, '\p{^Sentence_Break=upper}', "");
Expect(1, 120779, '\P{Sentence_Break=upper}', "");
Expect(0, 120779, '\P{^Sentence_Break=upper}', "");
Expect(1, 120778, '\p{Sentence_Break=	-Upper}', "");
Expect(0, 120778, '\p{^Sentence_Break=	-Upper}', "");
Expect(0, 120778, '\P{Sentence_Break=	-Upper}', "");
Expect(1, 120778, '\P{^Sentence_Break=	-Upper}', "");
Expect(0, 120779, '\p{Sentence_Break=	-Upper}', "");
Expect(1, 120779, '\p{^Sentence_Break=	-Upper}', "");
Expect(1, 120779, '\P{Sentence_Break=	-Upper}', "");
Expect(0, 120779, '\P{^Sentence_Break=	-Upper}', "");
Error('\p{SB=:=	UP}');
Error('\P{SB=:=	UP}');
Expect(1, 120778, '\p{SB=up}', "");
Expect(0, 120778, '\p{^SB=up}', "");
Expect(0, 120778, '\P{SB=up}', "");
Expect(1, 120778, '\P{^SB=up}', "");
Expect(0, 120779, '\p{SB=up}', "");
Expect(1, 120779, '\p{^SB=up}', "");
Expect(1, 120779, '\P{SB=up}', "");
Expect(0, 120779, '\P{^SB=up}', "");
Expect(1, 120778, '\p{SB=-up}', "");
Expect(0, 120778, '\p{^SB=-up}', "");
Expect(0, 120778, '\P{SB=-up}', "");
Expect(1, 120778, '\P{^SB=-up}', "");
Expect(0, 120779, '\p{SB=-up}', "");
Expect(1, 120779, '\p{^SB=-up}', "");
Expect(1, 120779, '\P{SB=-up}', "");
Expect(0, 120779, '\P{^SB=-up}', "");
Error('\p{Is_Sentence_Break=/a/UPPER}');
Error('\P{Is_Sentence_Break=/a/UPPER}');
Expect(1, 120778, '\p{Is_Sentence_Break=upper}', "");
Expect(0, 120778, '\p{^Is_Sentence_Break=upper}', "");
Expect(0, 120778, '\P{Is_Sentence_Break=upper}', "");
Expect(1, 120778, '\P{^Is_Sentence_Break=upper}', "");
Expect(0, 120779, '\p{Is_Sentence_Break=upper}', "");
Expect(1, 120779, '\p{^Is_Sentence_Break=upper}', "");
Expect(1, 120779, '\P{Is_Sentence_Break=upper}', "");
Expect(0, 120779, '\P{^Is_Sentence_Break=upper}', "");
Expect(1, 120778, '\p{Is_Sentence_Break=_-Upper}', "");
Expect(0, 120778, '\p{^Is_Sentence_Break=_-Upper}', "");
Expect(0, 120778, '\P{Is_Sentence_Break=_-Upper}', "");
Expect(1, 120778, '\P{^Is_Sentence_Break=_-Upper}', "");
Expect(0, 120779, '\p{Is_Sentence_Break=_-Upper}', "");
Expect(1, 120779, '\p{^Is_Sentence_Break=_-Upper}', "");
Expect(1, 120779, '\P{Is_Sentence_Break=_-Upper}', "");
Expect(0, 120779, '\P{^Is_Sentence_Break=_-Upper}', "");
Error('\p{Is_SB= UP/a/}');
Error('\P{Is_SB= UP/a/}');
Expect(1, 120778, '\p{Is_SB=up}', "");
Expect(0, 120778, '\p{^Is_SB=up}', "");
Expect(0, 120778, '\P{Is_SB=up}', "");
Expect(1, 120778, '\P{^Is_SB=up}', "");
Expect(0, 120779, '\p{Is_SB=up}', "");
Expect(1, 120779, '\p{^Is_SB=up}', "");
Expect(1, 120779, '\P{Is_SB=up}', "");
Expect(0, 120779, '\P{^Is_SB=up}', "");
Expect(1, 120778, '\p{Is_SB= 	up}', "");
Expect(0, 120778, '\p{^Is_SB= 	up}', "");
Expect(0, 120778, '\P{Is_SB= 	up}', "");
Expect(1, 120778, '\P{^Is_SB= 	up}', "");
Expect(0, 120779, '\p{Is_SB= 	up}', "");
Expect(1, 120779, '\p{^Is_SB= 	up}', "");
Expect(1, 120779, '\P{Is_SB= 	up}', "");
Expect(0, 120779, '\P{^Is_SB= 	up}', "");
Error('\p{Sentence_Break=/a/_Other}');
Error('\P{Sentence_Break=/a/_Other}');
Expect(1, 918000, '\p{Sentence_Break=other}', "");
Expect(0, 918000, '\p{^Sentence_Break=other}', "");
Expect(0, 918000, '\P{Sentence_Break=other}', "");
Expect(1, 918000, '\P{^Sentence_Break=other}', "");
Expect(0, 917999, '\p{Sentence_Break=other}', "");
Expect(1, 917999, '\p{^Sentence_Break=other}', "");
Expect(1, 917999, '\P{Sentence_Break=other}', "");
Expect(0, 917999, '\P{^Sentence_Break=other}', "");
Expect(1, 918000, '\p{Sentence_Break:  	Other}', "");
Expect(0, 918000, '\p{^Sentence_Break:  	Other}', "");
Expect(0, 918000, '\P{Sentence_Break:  	Other}', "");
Expect(1, 918000, '\P{^Sentence_Break:  	Other}', "");
Expect(0, 917999, '\p{Sentence_Break:  	Other}', "");
Expect(1, 917999, '\p{^Sentence_Break:  	Other}', "");
Expect(1, 917999, '\P{Sentence_Break:  	Other}', "");
Expect(0, 917999, '\P{^Sentence_Break:  	Other}', "");
Error('\p{SB:   /a/_ XX}');
Error('\P{SB:   /a/_ XX}');
Expect(1, 918000, '\p{SB=xx}', "");
Expect(0, 918000, '\p{^SB=xx}', "");
Expect(0, 918000, '\P{SB=xx}', "");
Expect(1, 918000, '\P{^SB=xx}', "");
Expect(0, 917999, '\p{SB=xx}', "");
Expect(1, 917999, '\p{^SB=xx}', "");
Expect(1, 917999, '\P{SB=xx}', "");
Expect(0, 917999, '\P{^SB=xx}', "");
Expect(1, 918000, '\p{SB=__XX}', "");
Expect(0, 918000, '\p{^SB=__XX}', "");
Expect(0, 918000, '\P{SB=__XX}', "");
Expect(1, 918000, '\P{^SB=__XX}', "");
Expect(0, 917999, '\p{SB=__XX}', "");
Expect(1, 917999, '\p{^SB=__XX}', "");
Expect(1, 917999, '\P{SB=__XX}', "");
Expect(0, 917999, '\P{^SB=__XX}', "");
Error('\p{Is_Sentence_Break= :=Other}');
Error('\P{Is_Sentence_Break= :=Other}');
Expect(1, 918000, '\p{Is_Sentence_Break=other}', "");
Expect(0, 918000, '\p{^Is_Sentence_Break=other}', "");
Expect(0, 918000, '\P{Is_Sentence_Break=other}', "");
Expect(1, 918000, '\P{^Is_Sentence_Break=other}', "");
Expect(0, 917999, '\p{Is_Sentence_Break=other}', "");
Expect(1, 917999, '\p{^Is_Sentence_Break=other}', "");
Expect(1, 917999, '\P{Is_Sentence_Break=other}', "");
Expect(0, 917999, '\P{^Is_Sentence_Break=other}', "");
Expect(1, 918000, '\p{Is_Sentence_Break=--other}', "");
Expect(0, 918000, '\p{^Is_Sentence_Break=--other}', "");
Expect(0, 918000, '\P{Is_Sentence_Break=--other}', "");
Expect(1, 918000, '\P{^Is_Sentence_Break=--other}', "");
Expect(0, 917999, '\p{Is_Sentence_Break=--other}', "");
Expect(1, 917999, '\p{^Is_Sentence_Break=--other}', "");
Expect(1, 917999, '\P{Is_Sentence_Break=--other}', "");
Expect(0, 917999, '\P{^Is_Sentence_Break=--other}', "");
Error('\p{Is_SB=		XX:=}');
Error('\P{Is_SB=		XX:=}');
Expect(1, 918000, '\p{Is_SB=xx}', "");
Expect(0, 918000, '\p{^Is_SB=xx}', "");
Expect(0, 918000, '\P{Is_SB=xx}', "");
Expect(1, 918000, '\P{^Is_SB=xx}', "");
Expect(0, 917999, '\p{Is_SB=xx}', "");
Expect(1, 917999, '\p{^Is_SB=xx}', "");
Expect(1, 917999, '\P{Is_SB=xx}', "");
Expect(0, 917999, '\P{^Is_SB=xx}', "");
Expect(1, 918000, '\p{Is_SB=-XX}', "");
Expect(0, 918000, '\p{^Is_SB=-XX}', "");
Expect(0, 918000, '\P{Is_SB=-XX}', "");
Expect(1, 918000, '\P{^Is_SB=-XX}', "");
Expect(0, 917999, '\p{Is_SB=-XX}', "");
Expect(1, 917999, '\p{^Is_SB=-XX}', "");
Expect(1, 917999, '\P{Is_SB=-XX}', "");
Expect(0, 917999, '\P{^Is_SB=-XX}', "");
Error('\p{Soft_Dotted=/a/-_NO}');
Error('\P{Soft_Dotted=/a/-_NO}');
Expect(1, 120468, '\p{Soft_Dotted=no}', "");
Expect(0, 120468, '\p{^Soft_Dotted=no}', "");
Expect(0, 120468, '\P{Soft_Dotted=no}', "");
Expect(1, 120468, '\P{^Soft_Dotted=no}', "");
Expect(0, 120467, '\p{Soft_Dotted=no}', "");
Expect(1, 120467, '\p{^Soft_Dotted=no}', "");
Expect(1, 120467, '\P{Soft_Dotted=no}', "");
Expect(0, 120467, '\P{^Soft_Dotted=no}', "");
Expect(1, 120468, '\p{Soft_Dotted=_-no}', "");
Expect(0, 120468, '\p{^Soft_Dotted=_-no}', "");
Expect(0, 120468, '\P{Soft_Dotted=_-no}', "");
Expect(1, 120468, '\P{^Soft_Dotted=_-no}', "");
Expect(0, 120467, '\p{Soft_Dotted=_-no}', "");
Expect(1, 120467, '\p{^Soft_Dotted=_-no}', "");
Expect(1, 120467, '\P{Soft_Dotted=_-no}', "");
Expect(0, 120467, '\P{^Soft_Dotted=_-no}', "");
Error('\p{SD=	:=N}');
Error('\P{SD=	:=N}');
Expect(1, 120468, '\p{SD=n}', "");
Expect(0, 120468, '\p{^SD=n}', "");
Expect(0, 120468, '\P{SD=n}', "");
Expect(1, 120468, '\P{^SD=n}', "");
Expect(0, 120467, '\p{SD=n}', "");
Expect(1, 120467, '\p{^SD=n}', "");
Expect(1, 120467, '\P{SD=n}', "");
Expect(0, 120467, '\P{^SD=n}', "");
Expect(1, 120468, '\p{SD=	-n}', "");
Expect(0, 120468, '\p{^SD=	-n}', "");
Expect(0, 120468, '\P{SD=	-n}', "");
Expect(1, 120468, '\P{^SD=	-n}', "");
Expect(0, 120467, '\p{SD=	-n}', "");
Expect(1, 120467, '\p{^SD=	-n}', "");
Expect(1, 120467, '\P{SD=	-n}', "");
Expect(0, 120467, '\P{^SD=	-n}', "");
Error('\p{Is_Soft_Dotted: :=F}');
Error('\P{Is_Soft_Dotted: :=F}');
Expect(1, 120468, '\p{Is_Soft_Dotted: f}', "");
Expect(0, 120468, '\p{^Is_Soft_Dotted: f}', "");
Expect(0, 120468, '\P{Is_Soft_Dotted: f}', "");
Expect(1, 120468, '\P{^Is_Soft_Dotted: f}', "");
Expect(0, 120467, '\p{Is_Soft_Dotted: f}', "");
Expect(1, 120467, '\p{^Is_Soft_Dotted: f}', "");
Expect(1, 120467, '\P{Is_Soft_Dotted: f}', "");
Expect(0, 120467, '\P{^Is_Soft_Dotted: f}', "");
Expect(1, 120468, '\p{Is_Soft_Dotted=_-F}', "");
Expect(0, 120468, '\p{^Is_Soft_Dotted=_-F}', "");
Expect(0, 120468, '\P{Is_Soft_Dotted=_-F}', "");
Expect(1, 120468, '\P{^Is_Soft_Dotted=_-F}', "");
Expect(0, 120467, '\p{Is_Soft_Dotted=_-F}', "");
Expect(1, 120467, '\p{^Is_Soft_Dotted=_-F}', "");
Expect(1, 120467, '\P{Is_Soft_Dotted=_-F}', "");
Expect(0, 120467, '\P{^Is_Soft_Dotted=_-F}', "");
Error('\p{Is_SD=:=_FALSE}');
Error('\P{Is_SD=:=_FALSE}');
Expect(1, 120468, '\p{Is_SD=false}', "");
Expect(0, 120468, '\p{^Is_SD=false}', "");
Expect(0, 120468, '\P{Is_SD=false}', "");
Expect(1, 120468, '\P{^Is_SD=false}', "");
Expect(0, 120467, '\p{Is_SD=false}', "");
Expect(1, 120467, '\p{^Is_SD=false}', "");
Expect(1, 120467, '\P{Is_SD=false}', "");
Expect(0, 120467, '\P{^Is_SD=false}', "");
Expect(1, 120468, '\p{Is_SD= False}', "");
Expect(0, 120468, '\p{^Is_SD= False}', "");
Expect(0, 120468, '\P{Is_SD= False}', "");
Expect(1, 120468, '\P{^Is_SD= False}', "");
Expect(0, 120467, '\p{Is_SD= False}', "");
Expect(1, 120467, '\p{^Is_SD= False}', "");
Expect(1, 120467, '\P{Is_SD= False}', "");
Expect(0, 120467, '\P{^Is_SD= False}', "");
Error('\p{Soft_Dotted=/a/- yes}');
Error('\P{Soft_Dotted=/a/- yes}');
Expect(1, 120467, '\p{Soft_Dotted=yes}', "");
Expect(0, 120467, '\p{^Soft_Dotted=yes}', "");
Expect(0, 120467, '\P{Soft_Dotted=yes}', "");
Expect(1, 120467, '\P{^Soft_Dotted=yes}', "");
Expect(0, 120468, '\p{Soft_Dotted=yes}', "");
Expect(1, 120468, '\p{^Soft_Dotted=yes}', "");
Expect(1, 120468, '\P{Soft_Dotted=yes}', "");
Expect(0, 120468, '\P{^Soft_Dotted=yes}', "");
Expect(1, 120467, '\p{Soft_Dotted=	Yes}', "");
Expect(0, 120467, '\p{^Soft_Dotted=	Yes}', "");
Expect(0, 120467, '\P{Soft_Dotted=	Yes}', "");
Expect(1, 120467, '\P{^Soft_Dotted=	Yes}', "");
Expect(0, 120468, '\p{Soft_Dotted=	Yes}', "");
Expect(1, 120468, '\p{^Soft_Dotted=	Yes}', "");
Expect(1, 120468, '\P{Soft_Dotted=	Yes}', "");
Expect(0, 120468, '\P{^Soft_Dotted=	Yes}', "");
Error('\p{SD: y/a/}');
Error('\P{SD: y/a/}');
Expect(1, 120467, '\p{SD:y}', "");
Expect(0, 120467, '\p{^SD:y}', "");
Expect(0, 120467, '\P{SD:y}', "");
Expect(1, 120467, '\P{^SD:y}', "");
Expect(0, 120468, '\p{SD:y}', "");
Expect(1, 120468, '\p{^SD:y}', "");
Expect(1, 120468, '\P{SD:y}', "");
Expect(0, 120468, '\P{^SD:y}', "");
Expect(1, 120467, '\p{SD= Y}', "");
Expect(0, 120467, '\p{^SD= Y}', "");
Expect(0, 120467, '\P{SD= Y}', "");
Expect(1, 120467, '\P{^SD= Y}', "");
Expect(0, 120468, '\p{SD= Y}', "");
Expect(1, 120468, '\p{^SD= Y}', "");
Expect(1, 120468, '\P{SD= Y}', "");
Expect(0, 120468, '\P{^SD= Y}', "");
Error('\p{Is_Soft_Dotted=/a/-T}');
Error('\P{Is_Soft_Dotted=/a/-T}');
Expect(1, 120467, '\p{Is_Soft_Dotted=t}', "");
Expect(0, 120467, '\p{^Is_Soft_Dotted=t}', "");
Expect(0, 120467, '\P{Is_Soft_Dotted=t}', "");
Expect(1, 120467, '\P{^Is_Soft_Dotted=t}', "");
Expect(0, 120468, '\p{Is_Soft_Dotted=t}', "");
Expect(1, 120468, '\p{^Is_Soft_Dotted=t}', "");
Expect(1, 120468, '\P{Is_Soft_Dotted=t}', "");
Expect(0, 120468, '\P{^Is_Soft_Dotted=t}', "");
Expect(1, 120467, '\p{Is_Soft_Dotted=  T}', "");
Expect(0, 120467, '\p{^Is_Soft_Dotted=  T}', "");
Expect(0, 120467, '\P{Is_Soft_Dotted=  T}', "");
Expect(1, 120467, '\P{^Is_Soft_Dotted=  T}', "");
Expect(0, 120468, '\p{Is_Soft_Dotted=  T}', "");
Expect(1, 120468, '\p{^Is_Soft_Dotted=  T}', "");
Expect(1, 120468, '\P{Is_Soft_Dotted=  T}', "");
Expect(0, 120468, '\P{^Is_Soft_Dotted=  T}', "");
Error('\p{Is_SD=-	true/a/}');
Error('\P{Is_SD=-	true/a/}');
Expect(1, 120467, '\p{Is_SD=true}', "");
Expect(0, 120467, '\p{^Is_SD=true}', "");
Expect(0, 120467, '\P{Is_SD=true}', "");
Expect(1, 120467, '\P{^Is_SD=true}', "");
Expect(0, 120468, '\p{Is_SD=true}', "");
Expect(1, 120468, '\p{^Is_SD=true}', "");
Expect(1, 120468, '\P{Is_SD=true}', "");
Expect(0, 120468, '\P{^Is_SD=true}', "");
Expect(1, 120467, '\p{Is_SD=_true}', "");
Expect(0, 120467, '\p{^Is_SD=_true}', "");
Expect(0, 120467, '\P{Is_SD=_true}', "");
Expect(1, 120467, '\P{^Is_SD=_true}', "");
Expect(0, 120468, '\p{Is_SD=_true}', "");
Expect(1, 120468, '\p{^Is_SD=_true}', "");
Expect(1, 120468, '\P{Is_SD=_true}', "");
Expect(0, 120468, '\P{^Is_SD=_true}', "");
Error('\p{STerm=	-No/a/}');
Error('\P{STerm=	-No/a/}');
Expect(1, 70087, '\p{STerm=no}', "");
Expect(0, 70087, '\p{^STerm=no}', "");
Expect(0, 70087, '\P{STerm=no}', "");
Expect(1, 70087, '\P{^STerm=no}', "");
Expect(0, 70086, '\p{STerm=no}', "");
Expect(1, 70086, '\p{^STerm=no}', "");
Expect(1, 70086, '\P{STerm=no}', "");
Expect(0, 70086, '\P{^STerm=no}', "");
Expect(1, 70087, '\p{STerm=		NO}', "");
Expect(0, 70087, '\p{^STerm=		NO}', "");
Expect(0, 70087, '\P{STerm=		NO}', "");
Expect(1, 70087, '\P{^STerm=		NO}', "");
Expect(0, 70086, '\p{STerm=		NO}', "");
Expect(1, 70086, '\p{^STerm=		NO}', "");
Expect(1, 70086, '\P{STerm=		NO}', "");
Expect(0, 70086, '\P{^STerm=		NO}', "");
Error('\p{Is_STerm=:= N}');
Error('\P{Is_STerm=:= N}');
Expect(1, 70087, '\p{Is_STerm=n}', "");
Expect(0, 70087, '\p{^Is_STerm=n}', "");
Expect(0, 70087, '\P{Is_STerm=n}', "");
Expect(1, 70087, '\P{^Is_STerm=n}', "");
Expect(0, 70086, '\p{Is_STerm=n}', "");
Expect(1, 70086, '\p{^Is_STerm=n}', "");
Expect(1, 70086, '\P{Is_STerm=n}', "");
Expect(0, 70086, '\P{^Is_STerm=n}', "");
Expect(1, 70087, '\p{Is_STerm=		N}', "");
Expect(0, 70087, '\p{^Is_STerm=		N}', "");
Expect(0, 70087, '\P{Is_STerm=		N}', "");
Expect(1, 70087, '\P{^Is_STerm=		N}', "");
Expect(0, 70086, '\p{Is_STerm=		N}', "");
Expect(1, 70086, '\p{^Is_STerm=		N}', "");
Expect(1, 70086, '\P{Is_STerm=		N}', "");
Expect(0, 70086, '\P{^Is_STerm=		N}', "");
Error('\p{STerm: :=- f}');
Error('\P{STerm: :=- f}');
Expect(1, 70087, '\p{STerm=f}', "");
Expect(0, 70087, '\p{^STerm=f}', "");
Expect(0, 70087, '\P{STerm=f}', "");
Expect(1, 70087, '\P{^STerm=f}', "");
Expect(0, 70086, '\p{STerm=f}', "");
Expect(1, 70086, '\p{^STerm=f}', "");
Expect(1, 70086, '\P{STerm=f}', "");
Expect(0, 70086, '\P{^STerm=f}', "");
Expect(1, 70087, '\p{STerm= 	F}', "");
Expect(0, 70087, '\p{^STerm= 	F}', "");
Expect(0, 70087, '\P{STerm= 	F}', "");
Expect(1, 70087, '\P{^STerm= 	F}', "");
Expect(0, 70086, '\p{STerm= 	F}', "");
Expect(1, 70086, '\p{^STerm= 	F}', "");
Expect(1, 70086, '\P{STerm= 	F}', "");
Expect(0, 70086, '\P{^STerm= 	F}', "");
Error('\p{Is_STerm=	:=false}');
Error('\P{Is_STerm=	:=false}');
Expect(1, 70087, '\p{Is_STerm=false}', "");
Expect(0, 70087, '\p{^Is_STerm=false}', "");
Expect(0, 70087, '\P{Is_STerm=false}', "");
Expect(1, 70087, '\P{^Is_STerm=false}', "");
Expect(0, 70086, '\p{Is_STerm=false}', "");
Expect(1, 70086, '\p{^Is_STerm=false}', "");
Expect(1, 70086, '\P{Is_STerm=false}', "");
Expect(0, 70086, '\P{^Is_STerm=false}', "");
Expect(1, 70087, '\p{Is_STerm=--false}', "");
Expect(0, 70087, '\p{^Is_STerm=--false}', "");
Expect(0, 70087, '\P{Is_STerm=--false}', "");
Expect(1, 70087, '\P{^Is_STerm=--false}', "");
Expect(0, 70086, '\p{Is_STerm=--false}', "");
Expect(1, 70086, '\p{^Is_STerm=--false}', "");
Expect(1, 70086, '\P{Is_STerm=--false}', "");
Expect(0, 70086, '\P{^Is_STerm=--false}', "");
Error('\p{STerm=:= Yes}');
Error('\P{STerm=:= Yes}');
Expect(1, 70086, '\p{STerm=yes}', "");
Expect(0, 70086, '\p{^STerm=yes}', "");
Expect(0, 70086, '\P{STerm=yes}', "");
Expect(1, 70086, '\P{^STerm=yes}', "");
Expect(0, 70087, '\p{STerm=yes}', "");
Expect(1, 70087, '\p{^STerm=yes}', "");
Expect(1, 70087, '\P{STerm=yes}', "");
Expect(0, 70087, '\P{^STerm=yes}', "");
Expect(1, 70086, '\p{STerm= Yes}', "");
Expect(0, 70086, '\p{^STerm= Yes}', "");
Expect(0, 70086, '\P{STerm= Yes}', "");
Expect(1, 70086, '\P{^STerm= Yes}', "");
Expect(0, 70087, '\p{STerm= Yes}', "");
Expect(1, 70087, '\p{^STerm= Yes}', "");
Expect(1, 70087, '\P{STerm= Yes}', "");
Expect(0, 70087, '\P{^STerm= Yes}', "");
Error('\p{Is_STerm=-/a/Y}');
Error('\P{Is_STerm=-/a/Y}');
Expect(1, 70086, '\p{Is_STerm=y}', "");
Expect(0, 70086, '\p{^Is_STerm=y}', "");
Expect(0, 70086, '\P{Is_STerm=y}', "");
Expect(1, 70086, '\P{^Is_STerm=y}', "");
Expect(0, 70087, '\p{Is_STerm=y}', "");
Expect(1, 70087, '\p{^Is_STerm=y}', "");
Expect(1, 70087, '\P{Is_STerm=y}', "");
Expect(0, 70087, '\P{^Is_STerm=y}', "");
Expect(1, 70086, '\p{Is_STerm=_y}', "");
Expect(0, 70086, '\p{^Is_STerm=_y}', "");
Expect(0, 70086, '\P{Is_STerm=_y}', "");
Expect(1, 70086, '\P{^Is_STerm=_y}', "");
Expect(0, 70087, '\p{Is_STerm=_y}', "");
Expect(1, 70087, '\p{^Is_STerm=_y}', "");
Expect(1, 70087, '\P{Is_STerm=_y}', "");
Expect(0, 70087, '\P{^Is_STerm=_y}', "");
Error('\p{STerm=:=T}');
Error('\P{STerm=:=T}');
Expect(1, 70086, '\p{STerm=t}', "");
Expect(0, 70086, '\p{^STerm=t}', "");
Expect(0, 70086, '\P{STerm=t}', "");
Expect(1, 70086, '\P{^STerm=t}', "");
Expect(0, 70087, '\p{STerm=t}', "");
Expect(1, 70087, '\p{^STerm=t}', "");
Expect(1, 70087, '\P{STerm=t}', "");
Expect(0, 70087, '\P{^STerm=t}', "");
Expect(1, 70086, '\p{STerm=	-T}', "");
Expect(0, 70086, '\p{^STerm=	-T}', "");
Expect(0, 70086, '\P{STerm=	-T}', "");
Expect(1, 70086, '\P{^STerm=	-T}', "");
Expect(0, 70087, '\p{STerm=	-T}', "");
Expect(1, 70087, '\p{^STerm=	-T}', "");
Expect(1, 70087, '\P{STerm=	-T}', "");
Expect(0, 70087, '\P{^STerm=	-T}', "");
Error('\p{Is_STerm=/a/_-TRUE}');
Error('\P{Is_STerm=/a/_-TRUE}');
Expect(1, 70086, '\p{Is_STerm=true}', "");
Expect(0, 70086, '\p{^Is_STerm=true}', "");
Expect(0, 70086, '\P{Is_STerm=true}', "");
Expect(1, 70086, '\P{^Is_STerm=true}', "");
Expect(0, 70087, '\p{Is_STerm=true}', "");
Expect(1, 70087, '\p{^Is_STerm=true}', "");
Expect(1, 70087, '\P{Is_STerm=true}', "");
Expect(0, 70087, '\P{^Is_STerm=true}', "");
Expect(1, 70086, '\p{Is_STerm=	True}', "");
Expect(0, 70086, '\p{^Is_STerm=	True}', "");
Expect(0, 70086, '\P{Is_STerm=	True}', "");
Expect(1, 70086, '\P{^Is_STerm=	True}', "");
Expect(0, 70087, '\p{Is_STerm=	True}', "");
Expect(1, 70087, '\p{^Is_STerm=	True}', "");
Expect(1, 70087, '\P{Is_STerm=	True}', "");
Expect(0, 70087, '\P{^Is_STerm=	True}', "");
Error('\p{Terminal_Punctuation=/a/no}');
Error('\P{Terminal_Punctuation=/a/no}');
Expect(1, 74868, '\p{Terminal_Punctuation=no}', "");
Expect(0, 74868, '\p{^Terminal_Punctuation=no}', "");
Expect(0, 74868, '\P{Terminal_Punctuation=no}', "");
Expect(1, 74868, '\P{^Terminal_Punctuation=no}', "");
Expect(0, 74867, '\p{Terminal_Punctuation=no}', "");
Expect(1, 74867, '\p{^Terminal_Punctuation=no}', "");
Expect(1, 74867, '\P{Terminal_Punctuation=no}', "");
Expect(0, 74867, '\P{^Terminal_Punctuation=no}', "");
Expect(1, 74868, '\p{Terminal_Punctuation=  NO}', "");
Expect(0, 74868, '\p{^Terminal_Punctuation=  NO}', "");
Expect(0, 74868, '\P{Terminal_Punctuation=  NO}', "");
Expect(1, 74868, '\P{^Terminal_Punctuation=  NO}', "");
Expect(0, 74867, '\p{Terminal_Punctuation=  NO}', "");
Expect(1, 74867, '\p{^Terminal_Punctuation=  NO}', "");
Expect(1, 74867, '\P{Terminal_Punctuation=  NO}', "");
Expect(0, 74867, '\P{^Terminal_Punctuation=  NO}', "");
Error('\p{Term= /a/N}');
Error('\P{Term= /a/N}');
Expect(1, 74868, '\p{Term=n}', "");
Expect(0, 74868, '\p{^Term=n}', "");
Expect(0, 74868, '\P{Term=n}', "");
Expect(1, 74868, '\P{^Term=n}', "");
Expect(0, 74867, '\p{Term=n}', "");
Expect(1, 74867, '\p{^Term=n}', "");
Expect(1, 74867, '\P{Term=n}', "");
Expect(0, 74867, '\P{^Term=n}', "");
Expect(1, 74868, '\p{Term= 	N}', "");
Expect(0, 74868, '\p{^Term= 	N}', "");
Expect(0, 74868, '\P{Term= 	N}', "");
Expect(1, 74868, '\P{^Term= 	N}', "");
Expect(0, 74867, '\p{Term= 	N}', "");
Expect(1, 74867, '\p{^Term= 	N}', "");
Expect(1, 74867, '\P{Term= 	N}', "");
Expect(0, 74867, '\P{^Term= 	N}', "");
Error('\p{Is_Terminal_Punctuation= _F/a/}');
Error('\P{Is_Terminal_Punctuation= _F/a/}');
Expect(1, 74868, '\p{Is_Terminal_Punctuation=f}', "");
Expect(0, 74868, '\p{^Is_Terminal_Punctuation=f}', "");
Expect(0, 74868, '\P{Is_Terminal_Punctuation=f}', "");
Expect(1, 74868, '\P{^Is_Terminal_Punctuation=f}', "");
Expect(0, 74867, '\p{Is_Terminal_Punctuation=f}', "");
Expect(1, 74867, '\p{^Is_Terminal_Punctuation=f}', "");
Expect(1, 74867, '\P{Is_Terminal_Punctuation=f}', "");
Expect(0, 74867, '\P{^Is_Terminal_Punctuation=f}', "");
Expect(1, 74868, '\p{Is_Terminal_Punctuation:-	F}', "");
Expect(0, 74868, '\p{^Is_Terminal_Punctuation:-	F}', "");
Expect(0, 74868, '\P{Is_Terminal_Punctuation:-	F}', "");
Expect(1, 74868, '\P{^Is_Terminal_Punctuation:-	F}', "");
Expect(0, 74867, '\p{Is_Terminal_Punctuation:-	F}', "");
Expect(1, 74867, '\p{^Is_Terminal_Punctuation:-	F}', "");
Expect(1, 74867, '\P{Is_Terminal_Punctuation:-	F}', "");
Expect(0, 74867, '\P{^Is_Terminal_Punctuation:-	F}', "");
Error('\p{Is_Term=-/a/FALSE}');
Error('\P{Is_Term=-/a/FALSE}');
Expect(1, 74868, '\p{Is_Term: false}', "");
Expect(0, 74868, '\p{^Is_Term: false}', "");
Expect(0, 74868, '\P{Is_Term: false}', "");
Expect(1, 74868, '\P{^Is_Term: false}', "");
Expect(0, 74867, '\p{Is_Term: false}', "");
Expect(1, 74867, '\p{^Is_Term: false}', "");
Expect(1, 74867, '\P{Is_Term: false}', "");
Expect(0, 74867, '\P{^Is_Term: false}', "");
Expect(1, 74868, '\p{Is_Term=-False}', "");
Expect(0, 74868, '\p{^Is_Term=-False}', "");
Expect(0, 74868, '\P{Is_Term=-False}', "");
Expect(1, 74868, '\P{^Is_Term=-False}', "");
Expect(0, 74867, '\p{Is_Term=-False}', "");
Expect(1, 74867, '\p{^Is_Term=-False}', "");
Expect(1, 74867, '\P{Is_Term=-False}', "");
Expect(0, 74867, '\P{^Is_Term=-False}', "");
Error('\p{Terminal_Punctuation=/a/	-yes}');
Error('\P{Terminal_Punctuation=/a/	-yes}');
Expect(1, 74867, '\p{Terminal_Punctuation=yes}', "");
Expect(0, 74867, '\p{^Terminal_Punctuation=yes}', "");
Expect(0, 74867, '\P{Terminal_Punctuation=yes}', "");
Expect(1, 74867, '\P{^Terminal_Punctuation=yes}', "");
Expect(0, 74868, '\p{Terminal_Punctuation=yes}', "");
Expect(1, 74868, '\p{^Terminal_Punctuation=yes}', "");
Expect(1, 74868, '\P{Terminal_Punctuation=yes}', "");
Expect(0, 74868, '\P{^Terminal_Punctuation=yes}', "");
Expect(1, 74867, '\p{Terminal_Punctuation= 	YES}', "");
Expect(0, 74867, '\p{^Terminal_Punctuation= 	YES}', "");
Expect(0, 74867, '\P{Terminal_Punctuation= 	YES}', "");
Expect(1, 74867, '\P{^Terminal_Punctuation= 	YES}', "");
Expect(0, 74868, '\p{Terminal_Punctuation= 	YES}', "");
Expect(1, 74868, '\p{^Terminal_Punctuation= 	YES}', "");
Expect(1, 74868, '\P{Terminal_Punctuation= 	YES}', "");
Expect(0, 74868, '\P{^Terminal_Punctuation= 	YES}', "");
Error('\p{Term=/a/-	y}');
Error('\P{Term=/a/-	y}');
Expect(1, 74867, '\p{Term=y}', "");
Expect(0, 74867, '\p{^Term=y}', "");
Expect(0, 74867, '\P{Term=y}', "");
Expect(1, 74867, '\P{^Term=y}', "");
Expect(0, 74868, '\p{Term=y}', "");
Expect(1, 74868, '\p{^Term=y}', "");
Expect(1, 74868, '\P{Term=y}', "");
Expect(0, 74868, '\P{^Term=y}', "");
Expect(1, 74867, '\p{Term=	 Y}', "");
Expect(0, 74867, '\p{^Term=	 Y}', "");
Expect(0, 74867, '\P{Term=	 Y}', "");
Expect(1, 74867, '\P{^Term=	 Y}', "");
Expect(0, 74868, '\p{Term=	 Y}', "");
Expect(1, 74868, '\p{^Term=	 Y}', "");
Expect(1, 74868, '\P{Term=	 Y}', "");
Expect(0, 74868, '\P{^Term=	 Y}', "");
Error('\p{Is_Terminal_Punctuation=:=		t}');
Error('\P{Is_Terminal_Punctuation=:=		t}');
Expect(1, 74867, '\p{Is_Terminal_Punctuation=t}', "");
Expect(0, 74867, '\p{^Is_Terminal_Punctuation=t}', "");
Expect(0, 74867, '\P{Is_Terminal_Punctuation=t}', "");
Expect(1, 74867, '\P{^Is_Terminal_Punctuation=t}', "");
Expect(0, 74868, '\p{Is_Terminal_Punctuation=t}', "");
Expect(1, 74868, '\p{^Is_Terminal_Punctuation=t}', "");
Expect(1, 74868, '\P{Is_Terminal_Punctuation=t}', "");
Expect(0, 74868, '\P{^Is_Terminal_Punctuation=t}', "");
Expect(1, 74867, '\p{Is_Terminal_Punctuation=  T}', "");
Expect(0, 74867, '\p{^Is_Terminal_Punctuation=  T}', "");
Expect(0, 74867, '\P{Is_Terminal_Punctuation=  T}', "");
Expect(1, 74867, '\P{^Is_Terminal_Punctuation=  T}', "");
Expect(0, 74868, '\p{Is_Terminal_Punctuation=  T}', "");
Expect(1, 74868, '\p{^Is_Terminal_Punctuation=  T}', "");
Expect(1, 74868, '\P{Is_Terminal_Punctuation=  T}', "");
Expect(0, 74868, '\P{^Is_Terminal_Punctuation=  T}', "");
Error('\p{Is_Term=/a/_ True}');
Error('\P{Is_Term=/a/_ True}');
Expect(1, 74867, '\p{Is_Term=true}', "");
Expect(0, 74867, '\p{^Is_Term=true}', "");
Expect(0, 74867, '\P{Is_Term=true}', "");
Expect(1, 74867, '\P{^Is_Term=true}', "");
Expect(0, 74868, '\p{Is_Term=true}', "");
Expect(1, 74868, '\p{^Is_Term=true}', "");
Expect(1, 74868, '\P{Is_Term=true}', "");
Expect(0, 74868, '\P{^Is_Term=true}', "");
Expect(1, 74867, '\p{Is_Term=-True}', "");
Expect(0, 74867, '\p{^Is_Term=-True}', "");
Expect(0, 74867, '\P{Is_Term=-True}', "");
Expect(1, 74867, '\P{^Is_Term=-True}', "");
Expect(0, 74868, '\p{Is_Term=-True}', "");
Expect(1, 74868, '\p{^Is_Term=-True}', "");
Expect(1, 74868, '\P{Is_Term=-True}', "");
Expect(0, 74868, '\P{^Is_Term=-True}', "");
Error('\p{Unified_Ideograph=:=__NO}');
Error('\P{Unified_Ideograph=:=__NO}');
Expect(1, 178206, '\p{Unified_Ideograph=no}', "");
Expect(0, 178206, '\p{^Unified_Ideograph=no}', "");
Expect(0, 178206, '\P{Unified_Ideograph=no}', "");
Expect(1, 178206, '\P{^Unified_Ideograph=no}', "");
Expect(0, 178205, '\p{Unified_Ideograph=no}', "");
Expect(1, 178205, '\p{^Unified_Ideograph=no}', "");
Expect(1, 178205, '\P{Unified_Ideograph=no}', "");
Expect(0, 178205, '\P{^Unified_Ideograph=no}', "");
Expect(1, 178206, '\p{Unified_Ideograph=_no}', "");
Expect(0, 178206, '\p{^Unified_Ideograph=_no}', "");
Expect(0, 178206, '\P{Unified_Ideograph=_no}', "");
Expect(1, 178206, '\P{^Unified_Ideograph=_no}', "");
Expect(0, 178205, '\p{Unified_Ideograph=_no}', "");
Expect(1, 178205, '\p{^Unified_Ideograph=_no}', "");
Expect(1, 178205, '\P{Unified_Ideograph=_no}', "");
Expect(0, 178205, '\P{^Unified_Ideograph=_no}', "");
Error('\p{UIdeo=/a/ _N}');
Error('\P{UIdeo=/a/ _N}');
Expect(1, 178206, '\p{UIdeo=n}', "");
Expect(0, 178206, '\p{^UIdeo=n}', "");
Expect(0, 178206, '\P{UIdeo=n}', "");
Expect(1, 178206, '\P{^UIdeo=n}', "");
Expect(0, 178205, '\p{UIdeo=n}', "");
Expect(1, 178205, '\p{^UIdeo=n}', "");
Expect(1, 178205, '\P{UIdeo=n}', "");
Expect(0, 178205, '\P{^UIdeo=n}', "");
Expect(1, 178206, '\p{UIdeo= N}', "");
Expect(0, 178206, '\p{^UIdeo= N}', "");
Expect(0, 178206, '\P{UIdeo= N}', "");
Expect(1, 178206, '\P{^UIdeo= N}', "");
Expect(0, 178205, '\p{UIdeo= N}', "");
Expect(1, 178205, '\p{^UIdeo= N}', "");
Expect(1, 178205, '\P{UIdeo= N}', "");
Expect(0, 178205, '\P{^UIdeo= N}', "");
Error('\p{Is_Unified_Ideograph=_f:=}');
Error('\P{Is_Unified_Ideograph=_f:=}');
Expect(1, 178206, '\p{Is_Unified_Ideograph=f}', "");
Expect(0, 178206, '\p{^Is_Unified_Ideograph=f}', "");
Expect(0, 178206, '\P{Is_Unified_Ideograph=f}', "");
Expect(1, 178206, '\P{^Is_Unified_Ideograph=f}', "");
Expect(0, 178205, '\p{Is_Unified_Ideograph=f}', "");
Expect(1, 178205, '\p{^Is_Unified_Ideograph=f}', "");
Expect(1, 178205, '\P{Is_Unified_Ideograph=f}', "");
Expect(0, 178205, '\P{^Is_Unified_Ideograph=f}', "");
Expect(1, 178206, '\p{Is_Unified_Ideograph:    F}', "");
Expect(0, 178206, '\p{^Is_Unified_Ideograph:    F}', "");
Expect(0, 178206, '\P{Is_Unified_Ideograph:    F}', "");
Expect(1, 178206, '\P{^Is_Unified_Ideograph:    F}', "");
Expect(0, 178205, '\p{Is_Unified_Ideograph:    F}', "");
Expect(1, 178205, '\p{^Is_Unified_Ideograph:    F}', "");
Expect(1, 178205, '\P{Is_Unified_Ideograph:    F}', "");
Expect(0, 178205, '\P{^Is_Unified_Ideograph:    F}', "");
Error('\p{Is_UIdeo= _False:=}');
Error('\P{Is_UIdeo= _False:=}');
Expect(1, 178206, '\p{Is_UIdeo=false}', "");
Expect(0, 178206, '\p{^Is_UIdeo=false}', "");
Expect(0, 178206, '\P{Is_UIdeo=false}', "");
Expect(1, 178206, '\P{^Is_UIdeo=false}', "");
Expect(0, 178205, '\p{Is_UIdeo=false}', "");
Expect(1, 178205, '\p{^Is_UIdeo=false}', "");
Expect(1, 178205, '\P{Is_UIdeo=false}', "");
Expect(0, 178205, '\P{^Is_UIdeo=false}', "");
Expect(1, 178206, '\p{Is_UIdeo=_	false}', "");
Expect(0, 178206, '\p{^Is_UIdeo=_	false}', "");
Expect(0, 178206, '\P{Is_UIdeo=_	false}', "");
Expect(1, 178206, '\P{^Is_UIdeo=_	false}', "");
Expect(0, 178205, '\p{Is_UIdeo=_	false}', "");
Expect(1, 178205, '\p{^Is_UIdeo=_	false}', "");
Expect(1, 178205, '\P{Is_UIdeo=_	false}', "");
Expect(0, 178205, '\P{^Is_UIdeo=_	false}', "");
Error('\p{Unified_Ideograph= Yes/a/}');
Error('\P{Unified_Ideograph= Yes/a/}');
Expect(1, 178205, '\p{Unified_Ideograph=yes}', "");
Expect(0, 178205, '\p{^Unified_Ideograph=yes}', "");
Expect(0, 178205, '\P{Unified_Ideograph=yes}', "");
Expect(1, 178205, '\P{^Unified_Ideograph=yes}', "");
Expect(0, 178206, '\p{Unified_Ideograph=yes}', "");
Expect(1, 178206, '\p{^Unified_Ideograph=yes}', "");
Expect(1, 178206, '\P{Unified_Ideograph=yes}', "");
Expect(0, 178206, '\P{^Unified_Ideograph=yes}', "");
Expect(1, 178205, '\p{Unified_Ideograph=_ yes}', "");
Expect(0, 178205, '\p{^Unified_Ideograph=_ yes}', "");
Expect(0, 178205, '\P{Unified_Ideograph=_ yes}', "");
Expect(1, 178205, '\P{^Unified_Ideograph=_ yes}', "");
Expect(0, 178206, '\p{Unified_Ideograph=_ yes}', "");
Expect(1, 178206, '\p{^Unified_Ideograph=_ yes}', "");
Expect(1, 178206, '\P{Unified_Ideograph=_ yes}', "");
Expect(0, 178206, '\P{^Unified_Ideograph=_ yes}', "");
Error('\p{UIdeo=:=y}');
Error('\P{UIdeo=:=y}');
Expect(1, 178205, '\p{UIdeo=y}', "");
Expect(0, 178205, '\p{^UIdeo=y}', "");
Expect(0, 178205, '\P{UIdeo=y}', "");
Expect(1, 178205, '\P{^UIdeo=y}', "");
Expect(0, 178206, '\p{UIdeo=y}', "");
Expect(1, 178206, '\p{^UIdeo=y}', "");
Expect(1, 178206, '\P{UIdeo=y}', "");
Expect(0, 178206, '\P{^UIdeo=y}', "");
Expect(1, 178205, '\p{UIdeo=_ Y}', "");
Expect(0, 178205, '\p{^UIdeo=_ Y}', "");
Expect(0, 178205, '\P{UIdeo=_ Y}', "");
Expect(1, 178205, '\P{^UIdeo=_ Y}', "");
Expect(0, 178206, '\p{UIdeo=_ Y}', "");
Expect(1, 178206, '\p{^UIdeo=_ Y}', "");
Expect(1, 178206, '\P{UIdeo=_ Y}', "");
Expect(0, 178206, '\P{^UIdeo=_ Y}', "");
Error('\p{Is_Unified_Ideograph=	T:=}');
Error('\P{Is_Unified_Ideograph=	T:=}');
Expect(1, 178205, '\p{Is_Unified_Ideograph=t}', "");
Expect(0, 178205, '\p{^Is_Unified_Ideograph=t}', "");
Expect(0, 178205, '\P{Is_Unified_Ideograph=t}', "");
Expect(1, 178205, '\P{^Is_Unified_Ideograph=t}', "");
Expect(0, 178206, '\p{Is_Unified_Ideograph=t}', "");
Expect(1, 178206, '\p{^Is_Unified_Ideograph=t}', "");
Expect(1, 178206, '\P{Is_Unified_Ideograph=t}', "");
Expect(0, 178206, '\P{^Is_Unified_Ideograph=t}', "");
Expect(1, 178205, '\p{Is_Unified_Ideograph= t}', "");
Expect(0, 178205, '\p{^Is_Unified_Ideograph= t}', "");
Expect(0, 178205, '\P{Is_Unified_Ideograph= t}', "");
Expect(1, 178205, '\P{^Is_Unified_Ideograph= t}', "");
Expect(0, 178206, '\p{Is_Unified_Ideograph= t}', "");
Expect(1, 178206, '\p{^Is_Unified_Ideograph= t}', "");
Expect(1, 178206, '\P{Is_Unified_Ideograph= t}', "");
Expect(0, 178206, '\P{^Is_Unified_Ideograph= t}', "");
Error('\p{Is_UIdeo=		true:=}');
Error('\P{Is_UIdeo=		true:=}');
Expect(1, 178205, '\p{Is_UIdeo=true}', "");
Expect(0, 178205, '\p{^Is_UIdeo=true}', "");
Expect(0, 178205, '\P{Is_UIdeo=true}', "");
Expect(1, 178205, '\P{^Is_UIdeo=true}', "");
Expect(0, 178206, '\p{Is_UIdeo=true}', "");
Expect(1, 178206, '\p{^Is_UIdeo=true}', "");
Expect(1, 178206, '\P{Is_UIdeo=true}', "");
Expect(0, 178206, '\P{^Is_UIdeo=true}', "");
Expect(1, 178205, '\p{Is_UIdeo=-True}', "");
Expect(0, 178205, '\p{^Is_UIdeo=-True}', "");
Expect(0, 178205, '\P{Is_UIdeo=-True}', "");
Expect(1, 178205, '\P{^Is_UIdeo=-True}', "");
Expect(0, 178206, '\p{Is_UIdeo=-True}', "");
Expect(1, 178206, '\p{^Is_UIdeo=-True}', "");
Expect(1, 178206, '\P{Is_UIdeo=-True}', "");
Expect(0, 178206, '\P{^Is_UIdeo=-True}', "");
Error('\p{Uppercase=-No:=}');
Error('\P{Uppercase=-No:=}');
Expect(1, 120779, '\p{Uppercase=no}', "");
Expect(0, 120779, '\p{^Uppercase=no}', "");
Expect(0, 120779, '\P{Uppercase=no}', "");
Expect(1, 120779, '\P{^Uppercase=no}', "");
Expect(0, 120778, '\p{Uppercase=no}', "");
Expect(1, 120778, '\p{^Uppercase=no}', "");
Expect(1, 120778, '\P{Uppercase=no}', "");
Expect(0, 120778, '\P{^Uppercase=no}', "");
Expect(1, 120779, '\p{Uppercase=-No}', "");
Expect(0, 120779, '\p{^Uppercase=-No}', "");
Expect(0, 120779, '\P{Uppercase=-No}', "");
Expect(1, 120779, '\P{^Uppercase=-No}', "");
Expect(0, 120778, '\p{Uppercase=-No}', "");
Expect(1, 120778, '\p{^Uppercase=-No}', "");
Expect(1, 120778, '\P{Uppercase=-No}', "");
Expect(0, 120778, '\P{^Uppercase=-No}', "");
Error('\p{Upper=:=--N}');
Error('\P{Upper=:=--N}');
Expect(1, 120779, '\p{Upper=n}', "");
Expect(0, 120779, '\p{^Upper=n}', "");
Expect(0, 120779, '\P{Upper=n}', "");
Expect(1, 120779, '\P{^Upper=n}', "");
Expect(0, 120778, '\p{Upper=n}', "");
Expect(1, 120778, '\p{^Upper=n}', "");
Expect(1, 120778, '\P{Upper=n}', "");
Expect(0, 120778, '\P{^Upper=n}', "");
Expect(1, 120779, '\p{Upper=-	N}', "");
Expect(0, 120779, '\p{^Upper=-	N}', "");
Expect(0, 120779, '\P{Upper=-	N}', "");
Expect(1, 120779, '\P{^Upper=-	N}', "");
Expect(0, 120778, '\p{Upper=-	N}', "");
Expect(1, 120778, '\p{^Upper=-	N}', "");
Expect(1, 120778, '\P{Upper=-	N}', "");
Expect(0, 120778, '\P{^Upper=-	N}', "");
Error('\p{Is_Uppercase=/a/  F}');
Error('\P{Is_Uppercase=/a/  F}');
Expect(1, 120779, '\p{Is_Uppercase=f}', "");
Expect(0, 120779, '\p{^Is_Uppercase=f}', "");
Expect(0, 120779, '\P{Is_Uppercase=f}', "");
Expect(1, 120779, '\P{^Is_Uppercase=f}', "");
Expect(0, 120778, '\p{Is_Uppercase=f}', "");
Expect(1, 120778, '\p{^Is_Uppercase=f}', "");
Expect(1, 120778, '\P{Is_Uppercase=f}', "");
Expect(0, 120778, '\P{^Is_Uppercase=f}', "");
Expect(1, 120779, '\p{Is_Uppercase:	- F}', "");
Expect(0, 120779, '\p{^Is_Uppercase:	- F}', "");
Expect(0, 120779, '\P{Is_Uppercase:	- F}', "");
Expect(1, 120779, '\P{^Is_Uppercase:	- F}', "");
Expect(0, 120778, '\p{Is_Uppercase:	- F}', "");
Expect(1, 120778, '\p{^Is_Uppercase:	- F}', "");
Expect(1, 120778, '\P{Is_Uppercase:	- F}', "");
Expect(0, 120778, '\P{^Is_Uppercase:	- F}', "");
Error('\p{Is_Upper=- FALSE/a/}');
Error('\P{Is_Upper=- FALSE/a/}');
Expect(1, 120779, '\p{Is_Upper=false}', "");
Expect(0, 120779, '\p{^Is_Upper=false}', "");
Expect(0, 120779, '\P{Is_Upper=false}', "");
Expect(1, 120779, '\P{^Is_Upper=false}', "");
Expect(0, 120778, '\p{Is_Upper=false}', "");
Expect(1, 120778, '\p{^Is_Upper=false}', "");
Expect(1, 120778, '\P{Is_Upper=false}', "");
Expect(0, 120778, '\P{^Is_Upper=false}', "");
Expect(1, 120779, '\p{Is_Upper=-FALSE}', "");
Expect(0, 120779, '\p{^Is_Upper=-FALSE}', "");
Expect(0, 120779, '\P{Is_Upper=-FALSE}', "");
Expect(1, 120779, '\P{^Is_Upper=-FALSE}', "");
Expect(0, 120778, '\p{Is_Upper=-FALSE}', "");
Expect(1, 120778, '\p{^Is_Upper=-FALSE}', "");
Expect(1, 120778, '\P{Is_Upper=-FALSE}', "");
Expect(0, 120778, '\P{^Is_Upper=-FALSE}', "");
Error('\p{Uppercase=__yes/a/}');
Error('\P{Uppercase=__yes/a/}');
Expect(1, 120778, '\p{Uppercase=yes}', "");
Expect(0, 120778, '\p{^Uppercase=yes}', "");
Expect(0, 120778, '\P{Uppercase=yes}', "");
Expect(1, 120778, '\P{^Uppercase=yes}', "");
Expect(0, 120779, '\p{Uppercase=yes}', "");
Expect(1, 120779, '\p{^Uppercase=yes}', "");
Expect(1, 120779, '\P{Uppercase=yes}', "");
Expect(0, 120779, '\P{^Uppercase=yes}', "");
Expect(1, 120778, '\p{Uppercase=_	yes}', "");
Expect(0, 120778, '\p{^Uppercase=_	yes}', "");
Expect(0, 120778, '\P{Uppercase=_	yes}', "");
Expect(1, 120778, '\P{^Uppercase=_	yes}', "");
Expect(0, 120779, '\p{Uppercase=_	yes}', "");
Expect(1, 120779, '\p{^Uppercase=_	yes}', "");
Expect(1, 120779, '\P{Uppercase=_	yes}', "");
Expect(0, 120779, '\P{^Uppercase=_	yes}', "");
Error('\p{Upper= :=Y}');
Error('\P{Upper= :=Y}');
Expect(1, 120778, '\p{Upper=y}', "");
Expect(0, 120778, '\p{^Upper=y}', "");
Expect(0, 120778, '\P{Upper=y}', "");
Expect(1, 120778, '\P{^Upper=y}', "");
Expect(0, 120779, '\p{Upper=y}', "");
Expect(1, 120779, '\p{^Upper=y}', "");
Expect(1, 120779, '\P{Upper=y}', "");
Expect(0, 120779, '\P{^Upper=y}', "");
Expect(1, 120778, '\p{Upper: _Y}', "");
Expect(0, 120778, '\p{^Upper: _Y}', "");
Expect(0, 120778, '\P{Upper: _Y}', "");
Expect(1, 120778, '\P{^Upper: _Y}', "");
Expect(0, 120779, '\p{Upper: _Y}', "");
Expect(1, 120779, '\p{^Upper: _Y}', "");
Expect(1, 120779, '\P{Upper: _Y}', "");
Expect(0, 120779, '\P{^Upper: _Y}', "");
Error('\p{Is_Uppercase=--T:=}');
Error('\P{Is_Uppercase=--T:=}');
Expect(1, 120778, '\p{Is_Uppercase=t}', "");
Expect(0, 120778, '\p{^Is_Uppercase=t}', "");
Expect(0, 120778, '\P{Is_Uppercase=t}', "");
Expect(1, 120778, '\P{^Is_Uppercase=t}', "");
Expect(0, 120779, '\p{Is_Uppercase=t}', "");
Expect(1, 120779, '\p{^Is_Uppercase=t}', "");
Expect(1, 120779, '\P{Is_Uppercase=t}', "");
Expect(0, 120779, '\P{^Is_Uppercase=t}', "");
Expect(1, 120778, '\p{Is_Uppercase= 	T}', "");
Expect(0, 120778, '\p{^Is_Uppercase= 	T}', "");
Expect(0, 120778, '\P{Is_Uppercase= 	T}', "");
Expect(1, 120778, '\P{^Is_Uppercase= 	T}', "");
Expect(0, 120779, '\p{Is_Uppercase= 	T}', "");
Expect(1, 120779, '\p{^Is_Uppercase= 	T}', "");
Expect(1, 120779, '\P{Is_Uppercase= 	T}', "");
Expect(0, 120779, '\P{^Is_Uppercase= 	T}', "");
Error('\p{Is_Upper:   :=  TRUE}');
Error('\P{Is_Upper:   :=  TRUE}');
Expect(1, 120778, '\p{Is_Upper=true}', "");
Expect(0, 120778, '\p{^Is_Upper=true}', "");
Expect(0, 120778, '\P{Is_Upper=true}', "");
Expect(1, 120778, '\P{^Is_Upper=true}', "");
Expect(0, 120779, '\p{Is_Upper=true}', "");
Expect(1, 120779, '\p{^Is_Upper=true}', "");
Expect(1, 120779, '\P{Is_Upper=true}', "");
Expect(0, 120779, '\P{^Is_Upper=true}', "");
Expect(1, 120778, '\p{Is_Upper=_true}', "");
Expect(0, 120778, '\p{^Is_Upper=_true}', "");
Expect(0, 120778, '\P{Is_Upper=_true}', "");
Expect(1, 120778, '\P{^Is_Upper=_true}', "");
Expect(0, 120779, '\p{Is_Upper=_true}', "");
Expect(1, 120779, '\p{^Is_Upper=_true}', "");
Expect(1, 120779, '\P{Is_Upper=_true}', "");
Expect(0, 120779, '\P{^Is_Upper=_true}', "");
Error('\p{Variation_Selector=	/a/no}');
Error('\P{Variation_Selector=	/a/no}');
Expect(1, 918000, '\p{Variation_Selector:   no}', "");
Expect(0, 918000, '\p{^Variation_Selector:   no}', "");
Expect(0, 918000, '\P{Variation_Selector:   no}', "");
Expect(1, 918000, '\P{^Variation_Selector:   no}', "");
Expect(0, 917999, '\p{Variation_Selector:   no}', "");
Expect(1, 917999, '\p{^Variation_Selector:   no}', "");
Expect(1, 917999, '\P{Variation_Selector:   no}', "");
Expect(0, 917999, '\P{^Variation_Selector:   no}', "");
Expect(1, 918000, '\p{Variation_Selector=	no}', "");
Expect(0, 918000, '\p{^Variation_Selector=	no}', "");
Expect(0, 918000, '\P{Variation_Selector=	no}', "");
Expect(1, 918000, '\P{^Variation_Selector=	no}', "");
Expect(0, 917999, '\p{Variation_Selector=	no}', "");
Expect(1, 917999, '\p{^Variation_Selector=	no}', "");
Expect(1, 917999, '\P{Variation_Selector=	no}', "");
Expect(0, 917999, '\P{^Variation_Selector=	no}', "");
Error('\p{VS=_	N/a/}');
Error('\P{VS=_	N/a/}');
Expect(1, 918000, '\p{VS:   n}', "");
Expect(0, 918000, '\p{^VS:   n}', "");
Expect(0, 918000, '\P{VS:   n}', "");
Expect(1, 918000, '\P{^VS:   n}', "");
Expect(0, 917999, '\p{VS:   n}', "");
Expect(1, 917999, '\p{^VS:   n}', "");
Expect(1, 917999, '\P{VS:   n}', "");
Expect(0, 917999, '\P{^VS:   n}', "");
Expect(1, 918000, '\p{VS= 	n}', "");
Expect(0, 918000, '\p{^VS= 	n}', "");
Expect(0, 918000, '\P{VS= 	n}', "");
Expect(1, 918000, '\P{^VS= 	n}', "");
Expect(0, 917999, '\p{VS= 	n}', "");
Expect(1, 917999, '\p{^VS= 	n}', "");
Expect(1, 917999, '\P{VS= 	n}', "");
Expect(0, 917999, '\P{^VS= 	n}', "");
Error('\p{Is_Variation_Selector:/a/__F}');
Error('\P{Is_Variation_Selector:/a/__F}');
Expect(1, 918000, '\p{Is_Variation_Selector=f}', "");
Expect(0, 918000, '\p{^Is_Variation_Selector=f}', "");
Expect(0, 918000, '\P{Is_Variation_Selector=f}', "");
Expect(1, 918000, '\P{^Is_Variation_Selector=f}', "");
Expect(0, 917999, '\p{Is_Variation_Selector=f}', "");
Expect(1, 917999, '\p{^Is_Variation_Selector=f}', "");
Expect(1, 917999, '\P{Is_Variation_Selector=f}', "");
Expect(0, 917999, '\P{^Is_Variation_Selector=f}', "");
Expect(1, 918000, '\p{Is_Variation_Selector:-_f}', "");
Expect(0, 918000, '\p{^Is_Variation_Selector:-_f}', "");
Expect(0, 918000, '\P{Is_Variation_Selector:-_f}', "");
Expect(1, 918000, '\P{^Is_Variation_Selector:-_f}', "");
Expect(0, 917999, '\p{Is_Variation_Selector:-_f}', "");
Expect(1, 917999, '\p{^Is_Variation_Selector:-_f}', "");
Expect(1, 917999, '\P{Is_Variation_Selector:-_f}', "");
Expect(0, 917999, '\P{^Is_Variation_Selector:-_f}', "");
Error('\p{Is_VS=_/a/False}');
Error('\P{Is_VS=_/a/False}');
Expect(1, 918000, '\p{Is_VS=false}', "");
Expect(0, 918000, '\p{^Is_VS=false}', "");
Expect(0, 918000, '\P{Is_VS=false}', "");
Expect(1, 918000, '\P{^Is_VS=false}', "");
Expect(0, 917999, '\p{Is_VS=false}', "");
Expect(1, 917999, '\p{^Is_VS=false}', "");
Expect(1, 917999, '\P{Is_VS=false}', "");
Expect(0, 917999, '\P{^Is_VS=false}', "");
Expect(1, 918000, '\p{Is_VS=- false}', "");
Expect(0, 918000, '\p{^Is_VS=- false}', "");
Expect(0, 918000, '\P{Is_VS=- false}', "");
Expect(1, 918000, '\P{^Is_VS=- false}', "");
Expect(0, 917999, '\p{Is_VS=- false}', "");
Expect(1, 917999, '\p{^Is_VS=- false}', "");
Expect(1, 917999, '\P{Is_VS=- false}', "");
Expect(0, 917999, '\P{^Is_VS=- false}', "");
Error('\p{Variation_Selector=/a/yes}');
Error('\P{Variation_Selector=/a/yes}');
Expect(1, 917999, '\p{Variation_Selector=yes}', "");
Expect(0, 917999, '\p{^Variation_Selector=yes}', "");
Expect(0, 917999, '\P{Variation_Selector=yes}', "");
Expect(1, 917999, '\P{^Variation_Selector=yes}', "");
Expect(0, 918000, '\p{Variation_Selector=yes}', "");
Expect(1, 918000, '\p{^Variation_Selector=yes}', "");
Expect(1, 918000, '\P{Variation_Selector=yes}', "");
Expect(0, 918000, '\P{^Variation_Selector=yes}', "");
Expect(1, 917999, '\p{Variation_Selector=		YES}', "");
Expect(0, 917999, '\p{^Variation_Selector=		YES}', "");
Expect(0, 917999, '\P{Variation_Selector=		YES}', "");
Expect(1, 917999, '\P{^Variation_Selector=		YES}', "");
Expect(0, 918000, '\p{Variation_Selector=		YES}', "");
Expect(1, 918000, '\p{^Variation_Selector=		YES}', "");
Expect(1, 918000, '\P{Variation_Selector=		YES}', "");
Expect(0, 918000, '\P{^Variation_Selector=		YES}', "");
Error('\p{VS=	_y:=}');
Error('\P{VS=	_y:=}');
Expect(1, 917999, '\p{VS=y}', "");
Expect(0, 917999, '\p{^VS=y}', "");
Expect(0, 917999, '\P{VS=y}', "");
Expect(1, 917999, '\P{^VS=y}', "");
Expect(0, 918000, '\p{VS=y}', "");
Expect(1, 918000, '\p{^VS=y}', "");
Expect(1, 918000, '\P{VS=y}', "");
Expect(0, 918000, '\P{^VS=y}', "");
Expect(1, 917999, '\p{VS=_y}', "");
Expect(0, 917999, '\p{^VS=_y}', "");
Expect(0, 917999, '\P{VS=_y}', "");
Expect(1, 917999, '\P{^VS=_y}', "");
Expect(0, 918000, '\p{VS=_y}', "");
Expect(1, 918000, '\p{^VS=_y}', "");
Expect(1, 918000, '\P{VS=_y}', "");
Expect(0, 918000, '\P{^VS=_y}', "");
Error('\p{Is_Variation_Selector=_/a/T}');
Error('\P{Is_Variation_Selector=_/a/T}');
Expect(1, 917999, '\p{Is_Variation_Selector:   t}', "");
Expect(0, 917999, '\p{^Is_Variation_Selector:   t}', "");
Expect(0, 917999, '\P{Is_Variation_Selector:   t}', "");
Expect(1, 917999, '\P{^Is_Variation_Selector:   t}', "");
Expect(0, 918000, '\p{Is_Variation_Selector:   t}', "");
Expect(1, 918000, '\p{^Is_Variation_Selector:   t}', "");
Expect(1, 918000, '\P{Is_Variation_Selector:   t}', "");
Expect(0, 918000, '\P{^Is_Variation_Selector:   t}', "");
Expect(1, 917999, '\p{Is_Variation_Selector=--T}', "");
Expect(0, 917999, '\p{^Is_Variation_Selector=--T}', "");
Expect(0, 917999, '\P{Is_Variation_Selector=--T}', "");
Expect(1, 917999, '\P{^Is_Variation_Selector=--T}', "");
Expect(0, 918000, '\p{Is_Variation_Selector=--T}', "");
Expect(1, 918000, '\p{^Is_Variation_Selector=--T}', "");
Expect(1, 918000, '\P{Is_Variation_Selector=--T}', "");
Expect(0, 918000, '\P{^Is_Variation_Selector=--T}', "");
Error('\p{Is_VS= /a/TRUE}');
Error('\P{Is_VS= /a/TRUE}');
Expect(1, 917999, '\p{Is_VS=true}', "");
Expect(0, 917999, '\p{^Is_VS=true}', "");
Expect(0, 917999, '\P{Is_VS=true}', "");
Expect(1, 917999, '\P{^Is_VS=true}', "");
Expect(0, 918000, '\p{Is_VS=true}', "");
Expect(1, 918000, '\p{^Is_VS=true}', "");
Expect(1, 918000, '\P{Is_VS=true}', "");
Expect(0, 918000, '\P{^Is_VS=true}', "");
Expect(1, 917999, '\p{Is_VS:			True}', "");
Expect(0, 917999, '\p{^Is_VS:			True}', "");
Expect(0, 917999, '\P{Is_VS:			True}', "");
Expect(1, 917999, '\P{^Is_VS:			True}', "");
Expect(0, 918000, '\p{Is_VS:			True}', "");
Expect(1, 918000, '\p{^Is_VS:			True}', "");
Expect(1, 918000, '\P{Is_VS:			True}', "");
Expect(0, 918000, '\P{^Is_VS:			True}', "");
Error('\p{Word_Break= :=CR}');
Error('\P{Word_Break= :=CR}');
Expect(1, 13, '\p{Word_Break=cr}', "");
Expect(0, 13, '\p{^Word_Break=cr}', "");
Expect(0, 13, '\P{Word_Break=cr}', "");
Expect(1, 13, '\P{^Word_Break=cr}', "");
Expect(0, 14, '\p{Word_Break=cr}', "");
Expect(1, 14, '\p{^Word_Break=cr}', "");
Expect(1, 14, '\P{Word_Break=cr}', "");
Expect(0, 14, '\P{^Word_Break=cr}', "");
Expect(1, 13, '\p{Word_Break=_-CR}', "");
Expect(0, 13, '\p{^Word_Break=_-CR}', "");
Expect(0, 13, '\P{Word_Break=_-CR}', "");
Expect(1, 13, '\P{^Word_Break=_-CR}', "");
Expect(0, 14, '\p{Word_Break=_-CR}', "");
Expect(1, 14, '\p{^Word_Break=_-CR}', "");
Expect(1, 14, '\P{Word_Break=_-CR}', "");
Expect(0, 14, '\P{^Word_Break=_-CR}', "");
Error('\p{WB=--cr:=}');
Error('\P{WB=--cr:=}');
Expect(1, 13, '\p{WB=cr}', "");
Expect(0, 13, '\p{^WB=cr}', "");
Expect(0, 13, '\P{WB=cr}', "");
Expect(1, 13, '\P{^WB=cr}', "");
Expect(0, 14, '\p{WB=cr}', "");
Expect(1, 14, '\p{^WB=cr}', "");
Expect(1, 14, '\P{WB=cr}', "");
Expect(0, 14, '\P{^WB=cr}', "");
Expect(1, 13, '\p{WB= CR}', "");
Expect(0, 13, '\p{^WB= CR}', "");
Expect(0, 13, '\P{WB= CR}', "");
Expect(1, 13, '\P{^WB= CR}', "");
Expect(0, 14, '\p{WB= CR}', "");
Expect(1, 14, '\p{^WB= CR}', "");
Expect(1, 14, '\P{WB= CR}', "");
Expect(0, 14, '\P{^WB= CR}', "");
Error('\p{Is_Word_Break=:=-	CR}');
Error('\P{Is_Word_Break=:=-	CR}');
Expect(1, 13, '\p{Is_Word_Break:	cr}', "");
Expect(0, 13, '\p{^Is_Word_Break:	cr}', "");
Expect(0, 13, '\P{Is_Word_Break:	cr}', "");
Expect(1, 13, '\P{^Is_Word_Break:	cr}', "");
Expect(0, 14, '\p{Is_Word_Break:	cr}', "");
Expect(1, 14, '\p{^Is_Word_Break:	cr}', "");
Expect(1, 14, '\P{Is_Word_Break:	cr}', "");
Expect(0, 14, '\P{^Is_Word_Break:	cr}', "");
Expect(1, 13, '\p{Is_Word_Break=CR}', "");
Expect(0, 13, '\p{^Is_Word_Break=CR}', "");
Expect(0, 13, '\P{Is_Word_Break=CR}', "");
Expect(1, 13, '\P{^Is_Word_Break=CR}', "");
Expect(0, 14, '\p{Is_Word_Break=CR}', "");
Expect(1, 14, '\p{^Is_Word_Break=CR}', "");
Expect(1, 14, '\P{Is_Word_Break=CR}', "");
Expect(0, 14, '\P{^Is_Word_Break=CR}', "");
Error('\p{Is_WB=/a/CR}');
Error('\P{Is_WB=/a/CR}');
Expect(1, 13, '\p{Is_WB=cr}', "");
Expect(0, 13, '\p{^Is_WB=cr}', "");
Expect(0, 13, '\P{Is_WB=cr}', "");
Expect(1, 13, '\P{^Is_WB=cr}', "");
Expect(0, 14, '\p{Is_WB=cr}', "");
Expect(1, 14, '\p{^Is_WB=cr}', "");
Expect(1, 14, '\P{Is_WB=cr}', "");
Expect(0, 14, '\P{^Is_WB=cr}', "");
Expect(1, 13, '\p{Is_WB=		CR}', "");
Expect(0, 13, '\p{^Is_WB=		CR}', "");
Expect(0, 13, '\P{Is_WB=		CR}', "");
Expect(1, 13, '\P{^Is_WB=		CR}', "");
Expect(0, 14, '\p{Is_WB=		CR}', "");
Expect(1, 14, '\p{^Is_WB=		CR}', "");
Expect(1, 14, '\P{Is_WB=		CR}', "");
Expect(0, 14, '\P{^Is_WB=		CR}', "");
Error('\p{Word_Break:/a/-DOUBLE_Quote}');
Error('\P{Word_Break:/a/-DOUBLE_Quote}');
Expect(1, 34, '\p{Word_Break=doublequote}', "");
Expect(0, 34, '\p{^Word_Break=doublequote}', "");
Expect(0, 34, '\P{Word_Break=doublequote}', "");
Expect(1, 34, '\P{^Word_Break=doublequote}', "");
Expect(0, 35, '\p{Word_Break=doublequote}', "");
Expect(1, 35, '\p{^Word_Break=doublequote}', "");
Expect(1, 35, '\P{Word_Break=doublequote}', "");
Expect(0, 35, '\P{^Word_Break=doublequote}', "");
Expect(1, 34, '\p{Word_Break:   - double_Quote}', "");
Expect(0, 34, '\p{^Word_Break:   - double_Quote}', "");
Expect(0, 34, '\P{Word_Break:   - double_Quote}', "");
Expect(1, 34, '\P{^Word_Break:   - double_Quote}', "");
Expect(0, 35, '\p{Word_Break:   - double_Quote}', "");
Expect(1, 35, '\p{^Word_Break:   - double_Quote}', "");
Expect(1, 35, '\P{Word_Break:   - double_Quote}', "");
Expect(0, 35, '\P{^Word_Break:   - double_Quote}', "");
Error('\p{WB=_	DQ/a/}');
Error('\P{WB=_	DQ/a/}');
Expect(1, 34, '\p{WB=dq}', "");
Expect(0, 34, '\p{^WB=dq}', "");
Expect(0, 34, '\P{WB=dq}', "");
Expect(1, 34, '\P{^WB=dq}', "");
Expect(0, 35, '\p{WB=dq}', "");
Expect(1, 35, '\p{^WB=dq}', "");
Expect(1, 35, '\P{WB=dq}', "");
Expect(0, 35, '\P{^WB=dq}', "");
Expect(1, 34, '\p{WB= -DQ}', "");
Expect(0, 34, '\p{^WB= -DQ}', "");
Expect(0, 34, '\P{WB= -DQ}', "");
Expect(1, 34, '\P{^WB= -DQ}', "");
Expect(0, 35, '\p{WB= -DQ}', "");
Expect(1, 35, '\p{^WB= -DQ}', "");
Expect(1, 35, '\P{WB= -DQ}', "");
Expect(0, 35, '\P{^WB= -DQ}', "");
Error('\p{Is_Word_Break=/a/_Double_Quote}');
Error('\P{Is_Word_Break=/a/_Double_Quote}');
Expect(1, 34, '\p{Is_Word_Break=doublequote}', "");
Expect(0, 34, '\p{^Is_Word_Break=doublequote}', "");
Expect(0, 34, '\P{Is_Word_Break=doublequote}', "");
Expect(1, 34, '\P{^Is_Word_Break=doublequote}', "");
Expect(0, 35, '\p{Is_Word_Break=doublequote}', "");
Expect(1, 35, '\p{^Is_Word_Break=doublequote}', "");
Expect(1, 35, '\P{Is_Word_Break=doublequote}', "");
Expect(0, 35, '\P{^Is_Word_Break=doublequote}', "");
Expect(1, 34, '\p{Is_Word_Break=	-Double_quote}', "");
Expect(0, 34, '\p{^Is_Word_Break=	-Double_quote}', "");
Expect(0, 34, '\P{Is_Word_Break=	-Double_quote}', "");
Expect(1, 34, '\P{^Is_Word_Break=	-Double_quote}', "");
Expect(0, 35, '\p{Is_Word_Break=	-Double_quote}', "");
Expect(1, 35, '\p{^Is_Word_Break=	-Double_quote}', "");
Expect(1, 35, '\P{Is_Word_Break=	-Double_quote}', "");
Expect(0, 35, '\P{^Is_Word_Break=	-Double_quote}', "");
Error('\p{Is_WB=	:=dq}');
Error('\P{Is_WB=	:=dq}');
Expect(1, 34, '\p{Is_WB=dq}', "");
Expect(0, 34, '\p{^Is_WB=dq}', "");
Expect(0, 34, '\P{Is_WB=dq}', "");
Expect(1, 34, '\P{^Is_WB=dq}', "");
Expect(0, 35, '\p{Is_WB=dq}', "");
Expect(1, 35, '\p{^Is_WB=dq}', "");
Expect(1, 35, '\P{Is_WB=dq}', "");
Expect(0, 35, '\P{^Is_WB=dq}', "");
Expect(1, 34, '\p{Is_WB=- DQ}', "");
Expect(0, 34, '\p{^Is_WB=- DQ}', "");
Expect(0, 34, '\P{Is_WB=- DQ}', "");
Expect(1, 34, '\P{^Is_WB=- DQ}', "");
Expect(0, 35, '\p{Is_WB=- DQ}', "");
Expect(1, 35, '\p{^Is_WB=- DQ}', "");
Expect(1, 35, '\P{Is_WB=- DQ}', "");
Expect(0, 35, '\P{^Is_WB=- DQ}', "");
Error('\p{Word_Break=-:=EXTENDNUMLET}');
Error('\P{Word_Break=-:=EXTENDNUMLET}');
Expect(1, 65343, '\p{Word_Break=extendnumlet}', "");
Expect(0, 65343, '\p{^Word_Break=extendnumlet}', "");
Expect(0, 65343, '\P{Word_Break=extendnumlet}', "");
Expect(1, 65343, '\P{^Word_Break=extendnumlet}', "");
Expect(0, 65344, '\p{Word_Break=extendnumlet}', "");
Expect(1, 65344, '\p{^Word_Break=extendnumlet}', "");
Expect(1, 65344, '\P{Word_Break=extendnumlet}', "");
Expect(0, 65344, '\P{^Word_Break=extendnumlet}', "");
Expect(1, 65343, '\p{Word_Break= _EXTENDNUMLET}', "");
Expect(0, 65343, '\p{^Word_Break= _EXTENDNUMLET}', "");
Expect(0, 65343, '\P{Word_Break= _EXTENDNUMLET}', "");
Expect(1, 65343, '\P{^Word_Break= _EXTENDNUMLET}', "");
Expect(0, 65344, '\p{Word_Break= _EXTENDNUMLET}', "");
Expect(1, 65344, '\p{^Word_Break= _EXTENDNUMLET}', "");
Expect(1, 65344, '\P{Word_Break= _EXTENDNUMLET}', "");
Expect(0, 65344, '\P{^Word_Break= _EXTENDNUMLET}', "");
Error('\p{WB=:=	_EX}');
Error('\P{WB=:=	_EX}');
Expect(1, 65343, '\p{WB=ex}', "");
Expect(0, 65343, '\p{^WB=ex}', "");
Expect(0, 65343, '\P{WB=ex}', "");
Expect(1, 65343, '\P{^WB=ex}', "");
Expect(0, 65344, '\p{WB=ex}', "");
Expect(1, 65344, '\p{^WB=ex}', "");
Expect(1, 65344, '\P{WB=ex}', "");
Expect(0, 65344, '\P{^WB=ex}', "");
Expect(1, 65343, '\p{WB= -EX}', "");
Expect(0, 65343, '\p{^WB= -EX}', "");
Expect(0, 65343, '\P{WB= -EX}', "");
Expect(1, 65343, '\P{^WB= -EX}', "");
Expect(0, 65344, '\p{WB= -EX}', "");
Expect(1, 65344, '\p{^WB= -EX}', "");
Expect(1, 65344, '\P{WB= -EX}', "");
Expect(0, 65344, '\P{^WB= -EX}', "");
Error('\p{Is_Word_Break= _ExtendNumLet:=}');
Error('\P{Is_Word_Break= _ExtendNumLet:=}');
Expect(1, 65343, '\p{Is_Word_Break=extendnumlet}', "");
Expect(0, 65343, '\p{^Is_Word_Break=extendnumlet}', "");
Expect(0, 65343, '\P{Is_Word_Break=extendnumlet}', "");
Expect(1, 65343, '\P{^Is_Word_Break=extendnumlet}', "");
Expect(0, 65344, '\p{Is_Word_Break=extendnumlet}', "");
Expect(1, 65344, '\p{^Is_Word_Break=extendnumlet}', "");
Expect(1, 65344, '\P{Is_Word_Break=extendnumlet}', "");
Expect(0, 65344, '\P{^Is_Word_Break=extendnumlet}', "");
Expect(1, 65343, '\p{Is_Word_Break=	 EXTENDNUMLET}', "");
Expect(0, 65343, '\p{^Is_Word_Break=	 EXTENDNUMLET}', "");
Expect(0, 65343, '\P{Is_Word_Break=	 EXTENDNUMLET}', "");
Expect(1, 65343, '\P{^Is_Word_Break=	 EXTENDNUMLET}', "");
Expect(0, 65344, '\p{Is_Word_Break=	 EXTENDNUMLET}', "");
Expect(1, 65344, '\p{^Is_Word_Break=	 EXTENDNUMLET}', "");
Expect(1, 65344, '\P{Is_Word_Break=	 EXTENDNUMLET}', "");
Expect(0, 65344, '\P{^Is_Word_Break=	 EXTENDNUMLET}', "");
Error('\p{Is_WB=_/a/EX}');
Error('\P{Is_WB=_/a/EX}');
Expect(1, 65343, '\p{Is_WB=ex}', "");
Expect(0, 65343, '\p{^Is_WB=ex}', "");
Expect(0, 65343, '\P{Is_WB=ex}', "");
Expect(1, 65343, '\P{^Is_WB=ex}', "");
Expect(0, 65344, '\p{Is_WB=ex}', "");
Expect(1, 65344, '\p{^Is_WB=ex}', "");
Expect(1, 65344, '\P{Is_WB=ex}', "");
Expect(0, 65344, '\P{^Is_WB=ex}', "");
Expect(1, 65343, '\p{Is_WB= EX}', "");
Expect(0, 65343, '\p{^Is_WB= EX}', "");
Expect(0, 65343, '\P{Is_WB= EX}', "");
Expect(1, 65343, '\P{^Is_WB= EX}', "");
Expect(0, 65344, '\p{Is_WB= EX}', "");
Expect(1, 65344, '\p{^Is_WB= EX}', "");
Expect(1, 65344, '\P{Is_WB= EX}', "");
Expect(0, 65344, '\P{^Is_WB= EX}', "");
Error('\p{Word_Break= EXTEND/a/}');
Error('\P{Word_Break= EXTEND/a/}');
Expect(1, 917999, '\p{Word_Break=extend}', "");
Expect(0, 917999, '\p{^Word_Break=extend}', "");
Expect(0, 917999, '\P{Word_Break=extend}', "");
Expect(1, 917999, '\P{^Word_Break=extend}', "");
Expect(0, 918000, '\p{Word_Break=extend}', "");
Expect(1, 918000, '\p{^Word_Break=extend}', "");
Expect(1, 918000, '\P{Word_Break=extend}', "");
Expect(0, 918000, '\P{^Word_Break=extend}', "");
Expect(1, 917999, '\p{Word_Break=	_extend}', "");
Expect(0, 917999, '\p{^Word_Break=	_extend}', "");
Expect(0, 917999, '\P{Word_Break=	_extend}', "");
Expect(1, 917999, '\P{^Word_Break=	_extend}', "");
Expect(0, 918000, '\p{Word_Break=	_extend}', "");
Expect(1, 918000, '\p{^Word_Break=	_extend}', "");
Expect(1, 918000, '\P{Word_Break=	_extend}', "");
Expect(0, 918000, '\P{^Word_Break=	_extend}', "");
Error('\p{WB:__Extend/a/}');
Error('\P{WB:__Extend/a/}');
Expect(1, 917999, '\p{WB=extend}', "");
Expect(0, 917999, '\p{^WB=extend}', "");
Expect(0, 917999, '\P{WB=extend}', "");
Expect(1, 917999, '\P{^WB=extend}', "");
Expect(0, 918000, '\p{WB=extend}', "");
Expect(1, 918000, '\p{^WB=extend}', "");
Expect(1, 918000, '\P{WB=extend}', "");
Expect(0, 918000, '\P{^WB=extend}', "");
Expect(1, 917999, '\p{WB:		Extend}', "");
Expect(0, 917999, '\p{^WB:		Extend}', "");
Expect(0, 917999, '\P{WB:		Extend}', "");
Expect(1, 917999, '\P{^WB:		Extend}', "");
Expect(0, 918000, '\p{WB:		Extend}', "");
Expect(1, 918000, '\p{^WB:		Extend}', "");
Expect(1, 918000, '\P{WB:		Extend}', "");
Expect(0, 918000, '\P{^WB:		Extend}', "");
Error('\p{Is_Word_Break=/a/EXTEND}');
Error('\P{Is_Word_Break=/a/EXTEND}');
Expect(1, 917999, '\p{Is_Word_Break=extend}', "");
Expect(0, 917999, '\p{^Is_Word_Break=extend}', "");
Expect(0, 917999, '\P{Is_Word_Break=extend}', "");
Expect(1, 917999, '\P{^Is_Word_Break=extend}', "");
Expect(0, 918000, '\p{Is_Word_Break=extend}', "");
Expect(1, 918000, '\p{^Is_Word_Break=extend}', "");
Expect(1, 918000, '\P{Is_Word_Break=extend}', "");
Expect(0, 918000, '\P{^Is_Word_Break=extend}', "");
Expect(1, 917999, '\p{Is_Word_Break=- extend}', "");
Expect(0, 917999, '\p{^Is_Word_Break=- extend}', "");
Expect(0, 917999, '\P{Is_Word_Break=- extend}', "");
Expect(1, 917999, '\P{^Is_Word_Break=- extend}', "");
Expect(0, 918000, '\p{Is_Word_Break=- extend}', "");
Expect(1, 918000, '\p{^Is_Word_Break=- extend}', "");
Expect(1, 918000, '\P{Is_Word_Break=- extend}', "");
Expect(0, 918000, '\P{^Is_Word_Break=- extend}', "");
Error('\p{Is_WB=:=_	Extend}');
Error('\P{Is_WB=:=_	Extend}');
Expect(1, 917999, '\p{Is_WB=extend}', "");
Expect(0, 917999, '\p{^Is_WB=extend}', "");
Expect(0, 917999, '\P{Is_WB=extend}', "");
Expect(1, 917999, '\P{^Is_WB=extend}', "");
Expect(0, 918000, '\p{Is_WB=extend}', "");
Expect(1, 918000, '\p{^Is_WB=extend}', "");
Expect(1, 918000, '\P{Is_WB=extend}', "");
Expect(0, 918000, '\P{^Is_WB=extend}', "");
Expect(1, 917999, '\p{Is_WB=- Extend}', "");
Expect(0, 917999, '\p{^Is_WB=- Extend}', "");
Expect(0, 917999, '\P{Is_WB=- Extend}', "");
Expect(1, 917999, '\P{^Is_WB=- Extend}', "");
Expect(0, 918000, '\p{Is_WB=- Extend}', "");
Expect(1, 918000, '\p{^Is_WB=- Extend}', "");
Expect(1, 918000, '\P{Is_WB=- Extend}', "");
Expect(0, 918000, '\P{^Is_WB=- Extend}', "");
Error('\p{Word_Break=-:=Format}');
Error('\P{Word_Break=-:=Format}');
Expect(1, 917631, '\p{Word_Break:	format}', "");
Expect(0, 917631, '\p{^Word_Break:	format}', "");
Expect(0, 917631, '\P{Word_Break:	format}', "");
Expect(1, 917631, '\P{^Word_Break:	format}', "");
Expect(0, 917632, '\p{Word_Break:	format}', "");
Expect(1, 917632, '\p{^Word_Break:	format}', "");
Expect(1, 917632, '\P{Word_Break:	format}', "");
Expect(0, 917632, '\P{^Word_Break:	format}', "");
Expect(1, 917631, '\p{Word_Break=_-Format}', "");
Expect(0, 917631, '\p{^Word_Break=_-Format}', "");
Expect(0, 917631, '\P{Word_Break=_-Format}', "");
Expect(1, 917631, '\P{^Word_Break=_-Format}', "");
Expect(0, 917632, '\p{Word_Break=_-Format}', "");
Expect(1, 917632, '\p{^Word_Break=_-Format}', "");
Expect(1, 917632, '\P{Word_Break=_-Format}', "");
Expect(0, 917632, '\P{^Word_Break=_-Format}', "");
Error('\p{WB= /a/fo}');
Error('\P{WB= /a/fo}');
Expect(1, 917631, '\p{WB=fo}', "");
Expect(0, 917631, '\p{^WB=fo}', "");
Expect(0, 917631, '\P{WB=fo}', "");
Expect(1, 917631, '\P{^WB=fo}', "");
Expect(0, 917632, '\p{WB=fo}', "");
Expect(1, 917632, '\p{^WB=fo}', "");
Expect(1, 917632, '\P{WB=fo}', "");
Expect(0, 917632, '\P{^WB=fo}', "");
Expect(1, 917631, '\p{WB=_FO}', "");
Expect(0, 917631, '\p{^WB=_FO}', "");
Expect(0, 917631, '\P{WB=_FO}', "");
Expect(1, 917631, '\P{^WB=_FO}', "");
Expect(0, 917632, '\p{WB=_FO}', "");
Expect(1, 917632, '\p{^WB=_FO}', "");
Expect(1, 917632, '\P{WB=_FO}', "");
Expect(0, 917632, '\P{^WB=_FO}', "");
Error('\p{Is_Word_Break=		Format:=}');
Error('\P{Is_Word_Break=		Format:=}');
Expect(1, 917631, '\p{Is_Word_Break=format}', "");
Expect(0, 917631, '\p{^Is_Word_Break=format}', "");
Expect(0, 917631, '\P{Is_Word_Break=format}', "");
Expect(1, 917631, '\P{^Is_Word_Break=format}', "");
Expect(0, 917632, '\p{Is_Word_Break=format}', "");
Expect(1, 917632, '\p{^Is_Word_Break=format}', "");
Expect(1, 917632, '\P{Is_Word_Break=format}', "");
Expect(0, 917632, '\P{^Is_Word_Break=format}', "");
Expect(1, 917631, '\p{Is_Word_Break=_FORMAT}', "");
Expect(0, 917631, '\p{^Is_Word_Break=_FORMAT}', "");
Expect(0, 917631, '\P{Is_Word_Break=_FORMAT}', "");
Expect(1, 917631, '\P{^Is_Word_Break=_FORMAT}', "");
Expect(0, 917632, '\p{Is_Word_Break=_FORMAT}', "");
Expect(1, 917632, '\p{^Is_Word_Break=_FORMAT}', "");
Expect(1, 917632, '\P{Is_Word_Break=_FORMAT}', "");
Expect(0, 917632, '\P{^Is_Word_Break=_FORMAT}', "");
Error('\p{Is_WB=-	FO/a/}');
Error('\P{Is_WB=-	FO/a/}');
Expect(1, 917631, '\p{Is_WB=fo}', "");
Expect(0, 917631, '\p{^Is_WB=fo}', "");
Expect(0, 917631, '\P{Is_WB=fo}', "");
Expect(1, 917631, '\P{^Is_WB=fo}', "");
Expect(0, 917632, '\p{Is_WB=fo}', "");
Expect(1, 917632, '\p{^Is_WB=fo}', "");
Expect(1, 917632, '\P{Is_WB=fo}', "");
Expect(0, 917632, '\P{^Is_WB=fo}', "");
Expect(1, 917631, '\p{Is_WB=_FO}', "");
Expect(0, 917631, '\p{^Is_WB=_FO}', "");
Expect(0, 917631, '\P{Is_WB=_FO}', "");
Expect(1, 917631, '\P{^Is_WB=_FO}', "");
Expect(0, 917632, '\p{Is_WB=_FO}', "");
Expect(1, 917632, '\p{^Is_WB=_FO}', "");
Expect(1, 917632, '\P{Is_WB=_FO}', "");
Expect(0, 917632, '\P{^Is_WB=_FO}', "");
Error('\p{Word_Break=/a/ -HEBREW_Letter}');
Error('\P{Word_Break=/a/ -HEBREW_Letter}');
Expect(1, 64335, '\p{Word_Break=hebrewletter}', "");
Expect(0, 64335, '\p{^Word_Break=hebrewletter}', "");
Expect(0, 64335, '\P{Word_Break=hebrewletter}', "");
Expect(1, 64335, '\P{^Word_Break=hebrewletter}', "");
Expect(0, 64336, '\p{Word_Break=hebrewletter}', "");
Expect(1, 64336, '\p{^Word_Break=hebrewletter}', "");
Expect(1, 64336, '\P{Word_Break=hebrewletter}', "");
Expect(0, 64336, '\P{^Word_Break=hebrewletter}', "");
Expect(1, 64335, '\p{Word_Break=		HEBREW_LETTER}', "");
Expect(0, 64335, '\p{^Word_Break=		HEBREW_LETTER}', "");
Expect(0, 64335, '\P{Word_Break=		HEBREW_LETTER}', "");
Expect(1, 64335, '\P{^Word_Break=		HEBREW_LETTER}', "");
Expect(0, 64336, '\p{Word_Break=		HEBREW_LETTER}', "");
Expect(1, 64336, '\p{^Word_Break=		HEBREW_LETTER}', "");
Expect(1, 64336, '\P{Word_Break=		HEBREW_LETTER}', "");
Expect(0, 64336, '\P{^Word_Break=		HEBREW_LETTER}', "");
Error('\p{WB=:=HL}');
Error('\P{WB=:=HL}');
Expect(1, 64335, '\p{WB=hl}', "");
Expect(0, 64335, '\p{^WB=hl}', "");
Expect(0, 64335, '\P{WB=hl}', "");
Expect(1, 64335, '\P{^WB=hl}', "");
Expect(0, 64336, '\p{WB=hl}', "");
Expect(1, 64336, '\p{^WB=hl}', "");
Expect(1, 64336, '\P{WB=hl}', "");
Expect(0, 64336, '\P{^WB=hl}', "");
Expect(1, 64335, '\p{WB=	-hl}', "");
Expect(0, 64335, '\p{^WB=	-hl}', "");
Expect(0, 64335, '\P{WB=	-hl}', "");
Expect(1, 64335, '\P{^WB=	-hl}', "");
Expect(0, 64336, '\p{WB=	-hl}', "");
Expect(1, 64336, '\p{^WB=	-hl}', "");
Expect(1, 64336, '\P{WB=	-hl}', "");
Expect(0, 64336, '\P{^WB=	-hl}', "");
Error('\p{Is_Word_Break= /a/Hebrew_Letter}');
Error('\P{Is_Word_Break= /a/Hebrew_Letter}');
Expect(1, 64335, '\p{Is_Word_Break=hebrewletter}', "");
Expect(0, 64335, '\p{^Is_Word_Break=hebrewletter}', "");
Expect(0, 64335, '\P{Is_Word_Break=hebrewletter}', "");
Expect(1, 64335, '\P{^Is_Word_Break=hebrewletter}', "");
Expect(0, 64336, '\p{Is_Word_Break=hebrewletter}', "");
Expect(1, 64336, '\p{^Is_Word_Break=hebrewletter}', "");
Expect(1, 64336, '\P{Is_Word_Break=hebrewletter}', "");
Expect(0, 64336, '\P{^Is_Word_Break=hebrewletter}', "");
Expect(1, 64335, '\p{Is_Word_Break=	HEBREW_Letter}', "");
Expect(0, 64335, '\p{^Is_Word_Break=	HEBREW_Letter}', "");
Expect(0, 64335, '\P{Is_Word_Break=	HEBREW_Letter}', "");
Expect(1, 64335, '\P{^Is_Word_Break=	HEBREW_Letter}', "");
Expect(0, 64336, '\p{Is_Word_Break=	HEBREW_Letter}', "");
Expect(1, 64336, '\p{^Is_Word_Break=	HEBREW_Letter}', "");
Expect(1, 64336, '\P{Is_Word_Break=	HEBREW_Letter}', "");
Expect(0, 64336, '\P{^Is_Word_Break=	HEBREW_Letter}', "");
Error('\p{Is_WB=--HL/a/}');
Error('\P{Is_WB=--HL/a/}');
Expect(1, 64335, '\p{Is_WB:   hl}', "");
Expect(0, 64335, '\p{^Is_WB:   hl}', "");
Expect(0, 64335, '\P{Is_WB:   hl}', "");
Expect(1, 64335, '\P{^Is_WB:   hl}', "");
Expect(0, 64336, '\p{Is_WB:   hl}', "");
Expect(1, 64336, '\p{^Is_WB:   hl}', "");
Expect(1, 64336, '\P{Is_WB:   hl}', "");
Expect(0, 64336, '\P{^Is_WB:   hl}', "");
Expect(1, 64335, '\p{Is_WB=- HL}', "");
Expect(0, 64335, '\p{^Is_WB=- HL}', "");
Expect(0, 64335, '\P{Is_WB=- HL}', "");
Expect(1, 64335, '\P{^Is_WB=- HL}', "");
Expect(0, 64336, '\p{Is_WB=- HL}', "");
Expect(1, 64336, '\p{^Is_WB=- HL}', "");
Expect(1, 64336, '\P{Is_WB=- HL}', "");
Expect(0, 64336, '\P{^Is_WB=- HL}', "");
Error('\p{Word_Break=/a/_-Katakana}');
Error('\P{Word_Break=/a/_-Katakana}');
Expect(1, 110592, '\p{Word_Break=katakana}', "");
Expect(0, 110592, '\p{^Word_Break=katakana}', "");
Expect(0, 110592, '\P{Word_Break=katakana}', "");
Expect(1, 110592, '\P{^Word_Break=katakana}', "");
Expect(0, 110593, '\p{Word_Break=katakana}', "");
Expect(1, 110593, '\p{^Word_Break=katakana}', "");
Expect(1, 110593, '\P{Word_Break=katakana}', "");
Expect(0, 110593, '\P{^Word_Break=katakana}', "");
Expect(1, 110592, '\p{Word_Break=	KATAKANA}', "");
Expect(0, 110592, '\p{^Word_Break=	KATAKANA}', "");
Expect(0, 110592, '\P{Word_Break=	KATAKANA}', "");
Expect(1, 110592, '\P{^Word_Break=	KATAKANA}', "");
Expect(0, 110593, '\p{Word_Break=	KATAKANA}', "");
Expect(1, 110593, '\p{^Word_Break=	KATAKANA}', "");
Expect(1, 110593, '\P{Word_Break=	KATAKANA}', "");
Expect(0, 110593, '\P{^Word_Break=	KATAKANA}', "");
Error('\p{WB=/a/	KA}');
Error('\P{WB=/a/	KA}');
Expect(1, 110592, '\p{WB=ka}', "");
Expect(0, 110592, '\p{^WB=ka}', "");
Expect(0, 110592, '\P{WB=ka}', "");
Expect(1, 110592, '\P{^WB=ka}', "");
Expect(0, 110593, '\p{WB=ka}', "");
Expect(1, 110593, '\p{^WB=ka}', "");
Expect(1, 110593, '\P{WB=ka}', "");
Expect(0, 110593, '\P{^WB=ka}', "");
Expect(1, 110592, '\p{WB= KA}', "");
Expect(0, 110592, '\p{^WB= KA}', "");
Expect(0, 110592, '\P{WB= KA}', "");
Expect(1, 110592, '\P{^WB= KA}', "");
Expect(0, 110593, '\p{WB= KA}', "");
Expect(1, 110593, '\p{^WB= KA}', "");
Expect(1, 110593, '\P{WB= KA}', "");
Expect(0, 110593, '\P{^WB= KA}', "");
Error('\p{Is_Word_Break=:= 	KATAKANA}');
Error('\P{Is_Word_Break=:= 	KATAKANA}');
Expect(1, 110592, '\p{Is_Word_Break=katakana}', "");
Expect(0, 110592, '\p{^Is_Word_Break=katakana}', "");
Expect(0, 110592, '\P{Is_Word_Break=katakana}', "");
Expect(1, 110592, '\P{^Is_Word_Break=katakana}', "");
Expect(0, 110593, '\p{Is_Word_Break=katakana}', "");
Expect(1, 110593, '\p{^Is_Word_Break=katakana}', "");
Expect(1, 110593, '\P{Is_Word_Break=katakana}', "");
Expect(0, 110593, '\P{^Is_Word_Break=katakana}', "");
Expect(1, 110592, '\p{Is_Word_Break=	-Katakana}', "");
Expect(0, 110592, '\p{^Is_Word_Break=	-Katakana}', "");
Expect(0, 110592, '\P{Is_Word_Break=	-Katakana}', "");
Expect(1, 110592, '\P{^Is_Word_Break=	-Katakana}', "");
Expect(0, 110593, '\p{Is_Word_Break=	-Katakana}', "");
Expect(1, 110593, '\p{^Is_Word_Break=	-Katakana}', "");
Expect(1, 110593, '\P{Is_Word_Break=	-Katakana}', "");
Expect(0, 110593, '\P{^Is_Word_Break=	-Katakana}', "");
Error('\p{Is_WB=:= 	ka}');
Error('\P{Is_WB=:= 	ka}');
Expect(1, 110592, '\p{Is_WB=ka}', "");
Expect(0, 110592, '\p{^Is_WB=ka}', "");
Expect(0, 110592, '\P{Is_WB=ka}', "");
Expect(1, 110592, '\P{^Is_WB=ka}', "");
Expect(0, 110593, '\p{Is_WB=ka}', "");
Expect(1, 110593, '\p{^Is_WB=ka}', "");
Expect(1, 110593, '\P{Is_WB=ka}', "");
Expect(0, 110593, '\P{^Is_WB=ka}', "");
Expect(1, 110592, '\p{Is_WB=- KA}', "");
Expect(0, 110592, '\p{^Is_WB=- KA}', "");
Expect(0, 110592, '\P{Is_WB=- KA}', "");
Expect(1, 110592, '\P{^Is_WB=- KA}', "");
Expect(0, 110593, '\p{Is_WB=- KA}', "");
Expect(1, 110593, '\p{^Is_WB=- KA}', "");
Expect(1, 110593, '\P{Is_WB=- KA}', "");
Expect(0, 110593, '\P{^Is_WB=- KA}', "");
Error('\p{Word_Break=ALetter:=}');
Error('\P{Word_Break=ALetter:=}');
Expect(1, 126651, '\p{Word_Break=aletter}', "");
Expect(0, 126651, '\p{^Word_Break=aletter}', "");
Expect(0, 126651, '\P{Word_Break=aletter}', "");
Expect(1, 126651, '\P{^Word_Break=aletter}', "");
Expect(0, 126652, '\p{Word_Break=aletter}', "");
Expect(1, 126652, '\p{^Word_Break=aletter}', "");
Expect(1, 126652, '\P{Word_Break=aletter}', "");
Expect(0, 126652, '\P{^Word_Break=aletter}', "");
Expect(1, 126651, '\p{Word_Break=	_aletter}', "");
Expect(0, 126651, '\p{^Word_Break=	_aletter}', "");
Expect(0, 126651, '\P{Word_Break=	_aletter}', "");
Expect(1, 126651, '\P{^Word_Break=	_aletter}', "");
Expect(0, 126652, '\p{Word_Break=	_aletter}', "");
Expect(1, 126652, '\p{^Word_Break=	_aletter}', "");
Expect(1, 126652, '\P{Word_Break=	_aletter}', "");
Expect(0, 126652, '\P{^Word_Break=	_aletter}', "");
Error('\p{WB= :=le}');
Error('\P{WB= :=le}');
Expect(1, 126651, '\p{WB=le}', "");
Expect(0, 126651, '\p{^WB=le}', "");
Expect(0, 126651, '\P{WB=le}', "");
Expect(1, 126651, '\P{^WB=le}', "");
Expect(0, 126652, '\p{WB=le}', "");
Expect(1, 126652, '\p{^WB=le}', "");
Expect(1, 126652, '\P{WB=le}', "");
Expect(0, 126652, '\P{^WB=le}', "");
Expect(1, 126651, '\p{WB=- LE}', "");
Expect(0, 126651, '\p{^WB=- LE}', "");
Expect(0, 126651, '\P{WB=- LE}', "");
Expect(1, 126651, '\P{^WB=- LE}', "");
Expect(0, 126652, '\p{WB=- LE}', "");
Expect(1, 126652, '\p{^WB=- LE}', "");
Expect(1, 126652, '\P{WB=- LE}', "");
Expect(0, 126652, '\P{^WB=- LE}', "");
Error('\p{Is_Word_Break=:=ALetter}');
Error('\P{Is_Word_Break=:=ALetter}');
Expect(1, 126651, '\p{Is_Word_Break=aletter}', "");
Expect(0, 126651, '\p{^Is_Word_Break=aletter}', "");
Expect(0, 126651, '\P{Is_Word_Break=aletter}', "");
Expect(1, 126651, '\P{^Is_Word_Break=aletter}', "");
Expect(0, 126652, '\p{Is_Word_Break=aletter}', "");
Expect(1, 126652, '\p{^Is_Word_Break=aletter}', "");
Expect(1, 126652, '\P{Is_Word_Break=aletter}', "");
Expect(0, 126652, '\P{^Is_Word_Break=aletter}', "");
Expect(1, 126651, '\p{Is_Word_Break:  -ALetter}', "");
Expect(0, 126651, '\p{^Is_Word_Break:  -ALetter}', "");
Expect(0, 126651, '\P{Is_Word_Break:  -ALetter}', "");
Expect(1, 126651, '\P{^Is_Word_Break:  -ALetter}', "");
Expect(0, 126652, '\p{Is_Word_Break:  -ALetter}', "");
Expect(1, 126652, '\p{^Is_Word_Break:  -ALetter}', "");
Expect(1, 126652, '\P{Is_Word_Break:  -ALetter}', "");
Expect(0, 126652, '\P{^Is_Word_Break:  -ALetter}', "");
Error('\p{Is_WB= LE/a/}');
Error('\P{Is_WB= LE/a/}');
Expect(1, 126651, '\p{Is_WB=le}', "");
Expect(0, 126651, '\p{^Is_WB=le}', "");
Expect(0, 126651, '\P{Is_WB=le}', "");
Expect(1, 126651, '\P{^Is_WB=le}', "");
Expect(0, 126652, '\p{Is_WB=le}', "");
Expect(1, 126652, '\p{^Is_WB=le}', "");
Expect(1, 126652, '\P{Is_WB=le}', "");
Expect(0, 126652, '\P{^Is_WB=le}', "");
Expect(1, 126651, '\p{Is_WB=_le}', "");
Expect(0, 126651, '\p{^Is_WB=_le}', "");
Expect(0, 126651, '\P{Is_WB=_le}', "");
Expect(1, 126651, '\P{^Is_WB=_le}', "");
Expect(0, 126652, '\p{Is_WB=_le}', "");
Expect(1, 126652, '\p{^Is_WB=_le}', "");
Expect(1, 126652, '\P{Is_WB=_le}', "");
Expect(0, 126652, '\P{^Is_WB=_le}', "");
Error('\p{Word_Break=/a/LF}');
Error('\P{Word_Break=/a/LF}');
Expect(1, 10, '\p{Word_Break: lf}', "");
Expect(0, 10, '\p{^Word_Break: lf}', "");
Expect(0, 10, '\P{Word_Break: lf}', "");
Expect(1, 10, '\P{^Word_Break: lf}', "");
Expect(0, 11, '\p{Word_Break: lf}', "");
Expect(1, 11, '\p{^Word_Break: lf}', "");
Expect(1, 11, '\P{Word_Break: lf}', "");
Expect(0, 11, '\P{^Word_Break: lf}', "");
Expect(1, 10, '\p{Word_Break=_LF}', "");
Expect(0, 10, '\p{^Word_Break=_LF}', "");
Expect(0, 10, '\P{Word_Break=_LF}', "");
Expect(1, 10, '\P{^Word_Break=_LF}', "");
Expect(0, 11, '\p{Word_Break=_LF}', "");
Expect(1, 11, '\p{^Word_Break=_LF}', "");
Expect(1, 11, '\P{Word_Break=_LF}', "");
Expect(0, 11, '\P{^Word_Break=_LF}', "");
Error('\p{WB=-LF:=}');
Error('\P{WB=-LF:=}');
Expect(1, 10, '\p{WB=lf}', "");
Expect(0, 10, '\p{^WB=lf}', "");
Expect(0, 10, '\P{WB=lf}', "");
Expect(1, 10, '\P{^WB=lf}', "");
Expect(0, 11, '\p{WB=lf}', "");
Expect(1, 11, '\p{^WB=lf}', "");
Expect(1, 11, '\P{WB=lf}', "");
Expect(0, 11, '\P{^WB=lf}', "");
Expect(1, 10, '\p{WB=-lf}', "");
Expect(0, 10, '\p{^WB=-lf}', "");
Expect(0, 10, '\P{WB=-lf}', "");
Expect(1, 10, '\P{^WB=-lf}', "");
Expect(0, 11, '\p{WB=-lf}', "");
Expect(1, 11, '\p{^WB=-lf}', "");
Expect(1, 11, '\P{WB=-lf}', "");
Expect(0, 11, '\P{^WB=-lf}', "");
Error('\p{Is_Word_Break=/a/--LF}');
Error('\P{Is_Word_Break=/a/--LF}');
Expect(1, 10, '\p{Is_Word_Break=lf}', "");
Expect(0, 10, '\p{^Is_Word_Break=lf}', "");
Expect(0, 10, '\P{Is_Word_Break=lf}', "");
Expect(1, 10, '\P{^Is_Word_Break=lf}', "");
Expect(0, 11, '\p{Is_Word_Break=lf}', "");
Expect(1, 11, '\p{^Is_Word_Break=lf}', "");
Expect(1, 11, '\P{Is_Word_Break=lf}', "");
Expect(0, 11, '\P{^Is_Word_Break=lf}', "");
Expect(1, 10, '\p{Is_Word_Break=_lf}', "");
Expect(0, 10, '\p{^Is_Word_Break=_lf}', "");
Expect(0, 10, '\P{Is_Word_Break=_lf}', "");
Expect(1, 10, '\P{^Is_Word_Break=_lf}', "");
Expect(0, 11, '\p{Is_Word_Break=_lf}', "");
Expect(1, 11, '\p{^Is_Word_Break=_lf}', "");
Expect(1, 11, '\P{Is_Word_Break=_lf}', "");
Expect(0, 11, '\P{^Is_Word_Break=_lf}', "");
Error('\p{Is_WB:   :=LF}');
Error('\P{Is_WB:   :=LF}');
Expect(1, 10, '\p{Is_WB:	lf}', "");
Expect(0, 10, '\p{^Is_WB:	lf}', "");
Expect(0, 10, '\P{Is_WB:	lf}', "");
Expect(1, 10, '\P{^Is_WB:	lf}', "");
Expect(0, 11, '\p{Is_WB:	lf}', "");
Expect(1, 11, '\p{^Is_WB:	lf}', "");
Expect(1, 11, '\P{Is_WB:	lf}', "");
Expect(0, 11, '\P{^Is_WB:	lf}', "");
Expect(1, 10, '\p{Is_WB= LF}', "");
Expect(0, 10, '\p{^Is_WB= LF}', "");
Expect(0, 10, '\P{Is_WB= LF}', "");
Expect(1, 10, '\P{^Is_WB= LF}', "");
Expect(0, 11, '\p{Is_WB= LF}', "");
Expect(1, 11, '\p{^Is_WB= LF}', "");
Expect(1, 11, '\P{Is_WB= LF}', "");
Expect(0, 11, '\P{^Is_WB= LF}', "");
Error('\p{Word_Break=	:=MIDNUMLET}');
Error('\P{Word_Break=	:=MIDNUMLET}');
Expect(1, 65294, '\p{Word_Break=midnumlet}', "");
Expect(0, 65294, '\p{^Word_Break=midnumlet}', "");
Expect(0, 65294, '\P{Word_Break=midnumlet}', "");
Expect(1, 65294, '\P{^Word_Break=midnumlet}', "");
Expect(0, 65295, '\p{Word_Break=midnumlet}', "");
Expect(1, 65295, '\p{^Word_Break=midnumlet}', "");
Expect(1, 65295, '\P{Word_Break=midnumlet}', "");
Expect(0, 65295, '\P{^Word_Break=midnumlet}', "");
Expect(1, 65294, '\p{Word_Break=-_MidNumLet}', "");
Expect(0, 65294, '\p{^Word_Break=-_MidNumLet}', "");
Expect(0, 65294, '\P{Word_Break=-_MidNumLet}', "");
Expect(1, 65294, '\P{^Word_Break=-_MidNumLet}', "");
Expect(0, 65295, '\p{Word_Break=-_MidNumLet}', "");
Expect(1, 65295, '\p{^Word_Break=-_MidNumLet}', "");
Expect(1, 65295, '\P{Word_Break=-_MidNumLet}', "");
Expect(0, 65295, '\P{^Word_Break=-_MidNumLet}', "");
Error('\p{WB=_/a/MB}');
Error('\P{WB=_/a/MB}');
Expect(1, 65294, '\p{WB=mb}', "");
Expect(0, 65294, '\p{^WB=mb}', "");
Expect(0, 65294, '\P{WB=mb}', "");
Expect(1, 65294, '\P{^WB=mb}', "");
Expect(0, 65295, '\p{WB=mb}', "");
Expect(1, 65295, '\p{^WB=mb}', "");
Expect(1, 65295, '\P{WB=mb}', "");
Expect(0, 65295, '\P{^WB=mb}', "");
Expect(1, 65294, '\p{WB=	_MB}', "");
Expect(0, 65294, '\p{^WB=	_MB}', "");
Expect(0, 65294, '\P{WB=	_MB}', "");
Expect(1, 65294, '\P{^WB=	_MB}', "");
Expect(0, 65295, '\p{WB=	_MB}', "");
Expect(1, 65295, '\p{^WB=	_MB}', "");
Expect(1, 65295, '\P{WB=	_MB}', "");
Expect(0, 65295, '\P{^WB=	_MB}', "");
Error('\p{Is_Word_Break=- MIDNUMLET/a/}');
Error('\P{Is_Word_Break=- MIDNUMLET/a/}');
Expect(1, 65294, '\p{Is_Word_Break=midnumlet}', "");
Expect(0, 65294, '\p{^Is_Word_Break=midnumlet}', "");
Expect(0, 65294, '\P{Is_Word_Break=midnumlet}', "");
Expect(1, 65294, '\P{^Is_Word_Break=midnumlet}', "");
Expect(0, 65295, '\p{Is_Word_Break=midnumlet}', "");
Expect(1, 65295, '\p{^Is_Word_Break=midnumlet}', "");
Expect(1, 65295, '\P{Is_Word_Break=midnumlet}', "");
Expect(0, 65295, '\P{^Is_Word_Break=midnumlet}', "");
Expect(1, 65294, '\p{Is_Word_Break=-_MidNumLet}', "");
Expect(0, 65294, '\p{^Is_Word_Break=-_MidNumLet}', "");
Expect(0, 65294, '\P{Is_Word_Break=-_MidNumLet}', "");
Expect(1, 65294, '\P{^Is_Word_Break=-_MidNumLet}', "");
Expect(0, 65295, '\p{Is_Word_Break=-_MidNumLet}', "");
Expect(1, 65295, '\p{^Is_Word_Break=-_MidNumLet}', "");
Expect(1, 65295, '\P{Is_Word_Break=-_MidNumLet}', "");
Expect(0, 65295, '\P{^Is_Word_Break=-_MidNumLet}', "");
Error('\p{Is_WB=MB/a/}');
Error('\P{Is_WB=MB/a/}');
Expect(1, 65294, '\p{Is_WB=mb}', "");
Expect(0, 65294, '\p{^Is_WB=mb}', "");
Expect(0, 65294, '\P{Is_WB=mb}', "");
Expect(1, 65294, '\P{^Is_WB=mb}', "");
Expect(0, 65295, '\p{Is_WB=mb}', "");
Expect(1, 65295, '\p{^Is_WB=mb}', "");
Expect(1, 65295, '\P{Is_WB=mb}', "");
Expect(0, 65295, '\P{^Is_WB=mb}', "");
Expect(1, 65294, '\p{Is_WB=MB}', "");
Expect(0, 65294, '\p{^Is_WB=MB}', "");
Expect(0, 65294, '\P{Is_WB=MB}', "");
Expect(1, 65294, '\P{^Is_WB=MB}', "");
Expect(0, 65295, '\p{Is_WB=MB}', "");
Expect(1, 65295, '\p{^Is_WB=MB}', "");
Expect(1, 65295, '\P{Is_WB=MB}', "");
Expect(0, 65295, '\P{^Is_WB=MB}', "");
Error('\p{Word_Break= 	midletter/a/}');
Error('\P{Word_Break= 	midletter/a/}');
Expect(1, 65306, '\p{Word_Break=midletter}', "");
Expect(0, 65306, '\p{^Word_Break=midletter}', "");
Expect(0, 65306, '\P{Word_Break=midletter}', "");
Expect(1, 65306, '\P{^Word_Break=midletter}', "");
Expect(0, 65307, '\p{Word_Break=midletter}', "");
Expect(1, 65307, '\p{^Word_Break=midletter}', "");
Expect(1, 65307, '\P{Word_Break=midletter}', "");
Expect(0, 65307, '\P{^Word_Break=midletter}', "");
Expect(1, 65306, '\p{Word_Break=-_midletter}', "");
Expect(0, 65306, '\p{^Word_Break=-_midletter}', "");
Expect(0, 65306, '\P{Word_Break=-_midletter}', "");
Expect(1, 65306, '\P{^Word_Break=-_midletter}', "");
Expect(0, 65307, '\p{Word_Break=-_midletter}', "");
Expect(1, 65307, '\p{^Word_Break=-_midletter}', "");
Expect(1, 65307, '\P{Word_Break=-_midletter}', "");
Expect(0, 65307, '\P{^Word_Break=-_midletter}', "");
Error('\p{WB=	/a/ML}');
Error('\P{WB=	/a/ML}');
Expect(1, 65306, '\p{WB=ml}', "");
Expect(0, 65306, '\p{^WB=ml}', "");
Expect(0, 65306, '\P{WB=ml}', "");
Expect(1, 65306, '\P{^WB=ml}', "");
Expect(0, 65307, '\p{WB=ml}', "");
Expect(1, 65307, '\p{^WB=ml}', "");
Expect(1, 65307, '\P{WB=ml}', "");
Expect(0, 65307, '\P{^WB=ml}', "");
Expect(1, 65306, '\p{WB= ml}', "");
Expect(0, 65306, '\p{^WB= ml}', "");
Expect(0, 65306, '\P{WB= ml}', "");
Expect(1, 65306, '\P{^WB= ml}', "");
Expect(0, 65307, '\p{WB= ml}', "");
Expect(1, 65307, '\p{^WB= ml}', "");
Expect(1, 65307, '\P{WB= ml}', "");
Expect(0, 65307, '\P{^WB= ml}', "");
Error('\p{Is_Word_Break=	/a/MidLetter}');
Error('\P{Is_Word_Break=	/a/MidLetter}');
Expect(1, 65306, '\p{Is_Word_Break=midletter}', "");
Expect(0, 65306, '\p{^Is_Word_Break=midletter}', "");
Expect(0, 65306, '\P{Is_Word_Break=midletter}', "");
Expect(1, 65306, '\P{^Is_Word_Break=midletter}', "");
Expect(0, 65307, '\p{Is_Word_Break=midletter}', "");
Expect(1, 65307, '\p{^Is_Word_Break=midletter}', "");
Expect(1, 65307, '\P{Is_Word_Break=midletter}', "");
Expect(0, 65307, '\P{^Is_Word_Break=midletter}', "");
Expect(1, 65306, '\p{Is_Word_Break=-	midletter}', "");
Expect(0, 65306, '\p{^Is_Word_Break=-	midletter}', "");
Expect(0, 65306, '\P{Is_Word_Break=-	midletter}', "");
Expect(1, 65306, '\P{^Is_Word_Break=-	midletter}', "");
Expect(0, 65307, '\p{Is_Word_Break=-	midletter}', "");
Expect(1, 65307, '\p{^Is_Word_Break=-	midletter}', "");
Expect(1, 65307, '\P{Is_Word_Break=-	midletter}', "");
Expect(0, 65307, '\P{^Is_Word_Break=-	midletter}', "");
Error('\p{Is_WB=- ML/a/}');
Error('\P{Is_WB=- ML/a/}');
Expect(1, 65306, '\p{Is_WB=ml}', "");
Expect(0, 65306, '\p{^Is_WB=ml}', "");
Expect(0, 65306, '\P{Is_WB=ml}', "");
Expect(1, 65306, '\P{^Is_WB=ml}', "");
Expect(0, 65307, '\p{Is_WB=ml}', "");
Expect(1, 65307, '\p{^Is_WB=ml}', "");
Expect(1, 65307, '\P{Is_WB=ml}', "");
Expect(0, 65307, '\P{^Is_WB=ml}', "");
Expect(1, 65306, '\p{Is_WB=_-ML}', "");
Expect(0, 65306, '\p{^Is_WB=_-ML}', "");
Expect(0, 65306, '\P{Is_WB=_-ML}', "");
Expect(1, 65306, '\P{^Is_WB=_-ML}', "");
Expect(0, 65307, '\p{Is_WB=_-ML}', "");
Expect(1, 65307, '\p{^Is_WB=_-ML}', "");
Expect(1, 65307, '\P{Is_WB=_-ML}', "");
Expect(0, 65307, '\P{^Is_WB=_-ML}', "");
Error('\p{Word_Break= 	MidNum/a/}');
Error('\P{Word_Break= 	MidNum/a/}');
Expect(1, 65307, '\p{Word_Break=midnum}', "");
Expect(0, 65307, '\p{^Word_Break=midnum}', "");
Expect(0, 65307, '\P{Word_Break=midnum}', "");
Expect(1, 65307, '\P{^Word_Break=midnum}', "");
Expect(0, 65308, '\p{Word_Break=midnum}', "");
Expect(1, 65308, '\p{^Word_Break=midnum}', "");
Expect(1, 65308, '\P{Word_Break=midnum}', "");
Expect(0, 65308, '\P{^Word_Break=midnum}', "");
Expect(1, 65307, '\p{Word_Break=_ MIDNUM}', "");
Expect(0, 65307, '\p{^Word_Break=_ MIDNUM}', "");
Expect(0, 65307, '\P{Word_Break=_ MIDNUM}', "");
Expect(1, 65307, '\P{^Word_Break=_ MIDNUM}', "");
Expect(0, 65308, '\p{Word_Break=_ MIDNUM}', "");
Expect(1, 65308, '\p{^Word_Break=_ MIDNUM}', "");
Expect(1, 65308, '\P{Word_Break=_ MIDNUM}', "");
Expect(0, 65308, '\P{^Word_Break=_ MIDNUM}', "");
Error('\p{WB=_	MN:=}');
Error('\P{WB=_	MN:=}');
Expect(1, 65307, '\p{WB=mn}', "");
Expect(0, 65307, '\p{^WB=mn}', "");
Expect(0, 65307, '\P{WB=mn}', "");
Expect(1, 65307, '\P{^WB=mn}', "");
Expect(0, 65308, '\p{WB=mn}', "");
Expect(1, 65308, '\p{^WB=mn}', "");
Expect(1, 65308, '\P{WB=mn}', "");
Expect(0, 65308, '\P{^WB=mn}', "");
Expect(1, 65307, '\p{WB: _MN}', "");
Expect(0, 65307, '\p{^WB: _MN}', "");
Expect(0, 65307, '\P{WB: _MN}', "");
Expect(1, 65307, '\P{^WB: _MN}', "");
Expect(0, 65308, '\p{WB: _MN}', "");
Expect(1, 65308, '\p{^WB: _MN}', "");
Expect(1, 65308, '\P{WB: _MN}', "");
Expect(0, 65308, '\P{^WB: _MN}', "");
Error('\p{Is_Word_Break=/a/midnum}');
Error('\P{Is_Word_Break=/a/midnum}');
Expect(1, 65307, '\p{Is_Word_Break=midnum}', "");
Expect(0, 65307, '\p{^Is_Word_Break=midnum}', "");
Expect(0, 65307, '\P{Is_Word_Break=midnum}', "");
Expect(1, 65307, '\P{^Is_Word_Break=midnum}', "");
Expect(0, 65308, '\p{Is_Word_Break=midnum}', "");
Expect(1, 65308, '\p{^Is_Word_Break=midnum}', "");
Expect(1, 65308, '\P{Is_Word_Break=midnum}', "");
Expect(0, 65308, '\P{^Is_Word_Break=midnum}', "");
Expect(1, 65307, '\p{Is_Word_Break=-	MidNum}', "");
Expect(0, 65307, '\p{^Is_Word_Break=-	MidNum}', "");
Expect(0, 65307, '\P{Is_Word_Break=-	MidNum}', "");
Expect(1, 65307, '\P{^Is_Word_Break=-	MidNum}', "");
Expect(0, 65308, '\p{Is_Word_Break=-	MidNum}', "");
Expect(1, 65308, '\p{^Is_Word_Break=-	MidNum}', "");
Expect(1, 65308, '\P{Is_Word_Break=-	MidNum}', "");
Expect(0, 65308, '\P{^Is_Word_Break=-	MidNum}', "");
Error('\p{Is_WB=:=-MN}');
Error('\P{Is_WB=:=-MN}');
Expect(1, 65307, '\p{Is_WB: mn}', "");
Expect(0, 65307, '\p{^Is_WB: mn}', "");
Expect(0, 65307, '\P{Is_WB: mn}', "");
Expect(1, 65307, '\P{^Is_WB: mn}', "");
Expect(0, 65308, '\p{Is_WB: mn}', "");
Expect(1, 65308, '\p{^Is_WB: mn}', "");
Expect(1, 65308, '\P{Is_WB: mn}', "");
Expect(0, 65308, '\P{^Is_WB: mn}', "");
Expect(1, 65307, '\p{Is_WB=	-MN}', "");
Expect(0, 65307, '\p{^Is_WB=	-MN}', "");
Expect(0, 65307, '\P{Is_WB=	-MN}', "");
Expect(1, 65307, '\P{^Is_WB=	-MN}', "");
Expect(0, 65308, '\p{Is_WB=	-MN}', "");
Expect(1, 65308, '\p{^Is_WB=	-MN}', "");
Expect(1, 65308, '\P{Is_WB=	-MN}', "");
Expect(0, 65308, '\P{^Is_WB=	-MN}', "");
Error('\p{Word_Break=	:=Newline}');
Error('\P{Word_Break=	:=Newline}');
Expect(1, 8233, '\p{Word_Break=newline}', "");
Expect(0, 8233, '\p{^Word_Break=newline}', "");
Expect(0, 8233, '\P{Word_Break=newline}', "");
Expect(1, 8233, '\P{^Word_Break=newline}', "");
Expect(0, 8234, '\p{Word_Break=newline}', "");
Expect(1, 8234, '\p{^Word_Break=newline}', "");
Expect(1, 8234, '\P{Word_Break=newline}', "");
Expect(0, 8234, '\P{^Word_Break=newline}', "");
Expect(1, 8233, '\p{Word_Break=-Newline}', "");
Expect(0, 8233, '\p{^Word_Break=-Newline}', "");
Expect(0, 8233, '\P{Word_Break=-Newline}', "");
Expect(1, 8233, '\P{^Word_Break=-Newline}', "");
Expect(0, 8234, '\p{Word_Break=-Newline}', "");
Expect(1, 8234, '\p{^Word_Break=-Newline}', "");
Expect(1, 8234, '\P{Word_Break=-Newline}', "");
Expect(0, 8234, '\P{^Word_Break=-Newline}', "");
Error('\p{WB:   /a/-_nl}');
Error('\P{WB:   /a/-_nl}');
Expect(1, 8233, '\p{WB=nl}', "");
Expect(0, 8233, '\p{^WB=nl}', "");
Expect(0, 8233, '\P{WB=nl}', "");
Expect(1, 8233, '\P{^WB=nl}', "");
Expect(0, 8234, '\p{WB=nl}', "");
Expect(1, 8234, '\p{^WB=nl}', "");
Expect(1, 8234, '\P{WB=nl}', "");
Expect(0, 8234, '\P{^WB=nl}', "");
Expect(1, 8233, '\p{WB=  NL}', "");
Expect(0, 8233, '\p{^WB=  NL}', "");
Expect(0, 8233, '\P{WB=  NL}', "");
Expect(1, 8233, '\P{^WB=  NL}', "");
Expect(0, 8234, '\p{WB=  NL}', "");
Expect(1, 8234, '\p{^WB=  NL}', "");
Expect(1, 8234, '\P{WB=  NL}', "");
Expect(0, 8234, '\P{^WB=  NL}', "");
Error('\p{Is_Word_Break=/a/Newline}');
Error('\P{Is_Word_Break=/a/Newline}');
Expect(1, 8233, '\p{Is_Word_Break=newline}', "");
Expect(0, 8233, '\p{^Is_Word_Break=newline}', "");
Expect(0, 8233, '\P{Is_Word_Break=newline}', "");
Expect(1, 8233, '\P{^Is_Word_Break=newline}', "");
Expect(0, 8234, '\p{Is_Word_Break=newline}', "");
Expect(1, 8234, '\p{^Is_Word_Break=newline}', "");
Expect(1, 8234, '\P{Is_Word_Break=newline}', "");
Expect(0, 8234, '\P{^Is_Word_Break=newline}', "");
Expect(1, 8233, '\p{Is_Word_Break= newline}', "");
Expect(0, 8233, '\p{^Is_Word_Break= newline}', "");
Expect(0, 8233, '\P{Is_Word_Break= newline}', "");
Expect(1, 8233, '\P{^Is_Word_Break= newline}', "");
Expect(0, 8234, '\p{Is_Word_Break= newline}', "");
Expect(1, 8234, '\p{^Is_Word_Break= newline}', "");
Expect(1, 8234, '\P{Is_Word_Break= newline}', "");
Expect(0, 8234, '\P{^Is_Word_Break= newline}', "");
Error('\p{Is_WB:	/a/-_NL}');
Error('\P{Is_WB:	/a/-_NL}');
Expect(1, 8233, '\p{Is_WB=nl}', "");
Expect(0, 8233, '\p{^Is_WB=nl}', "");
Expect(0, 8233, '\P{Is_WB=nl}', "");
Expect(1, 8233, '\P{^Is_WB=nl}', "");
Expect(0, 8234, '\p{Is_WB=nl}', "");
Expect(1, 8234, '\p{^Is_WB=nl}', "");
Expect(1, 8234, '\P{Is_WB=nl}', "");
Expect(0, 8234, '\P{^Is_WB=nl}', "");
Expect(1, 8233, '\p{Is_WB=  nl}', "");
Expect(0, 8233, '\p{^Is_WB=  nl}', "");
Expect(0, 8233, '\P{Is_WB=  nl}', "");
Expect(1, 8233, '\P{^Is_WB=  nl}', "");
Expect(0, 8234, '\p{Is_WB=  nl}', "");
Expect(1, 8234, '\p{^Is_WB=  nl}', "");
Expect(1, 8234, '\P{Is_WB=  nl}', "");
Expect(0, 8234, '\P{^Is_WB=  nl}', "");
Error('\p{Word_Break= /a/numeric}');
Error('\P{Word_Break= /a/numeric}');
Expect(1, 120831, '\p{Word_Break=numeric}', "");
Expect(0, 120831, '\p{^Word_Break=numeric}', "");
Expect(0, 120831, '\P{Word_Break=numeric}', "");
Expect(1, 120831, '\P{^Word_Break=numeric}', "");
Expect(0, 120832, '\p{Word_Break=numeric}', "");
Expect(1, 120832, '\p{^Word_Break=numeric}', "");
Expect(1, 120832, '\P{Word_Break=numeric}', "");
Expect(0, 120832, '\P{^Word_Break=numeric}', "");
Expect(1, 120831, '\p{Word_Break=	Numeric}', "");
Expect(0, 120831, '\p{^Word_Break=	Numeric}', "");
Expect(0, 120831, '\P{Word_Break=	Numeric}', "");
Expect(1, 120831, '\P{^Word_Break=	Numeric}', "");
Expect(0, 120832, '\p{Word_Break=	Numeric}', "");
Expect(1, 120832, '\p{^Word_Break=	Numeric}', "");
Expect(1, 120832, '\P{Word_Break=	Numeric}', "");
Expect(0, 120832, '\P{^Word_Break=	Numeric}', "");
Error('\p{WB=_:=NU}');
Error('\P{WB=_:=NU}');
Expect(1, 120831, '\p{WB=nu}', "");
Expect(0, 120831, '\p{^WB=nu}', "");
Expect(0, 120831, '\P{WB=nu}', "");
Expect(1, 120831, '\P{^WB=nu}', "");
Expect(0, 120832, '\p{WB=nu}', "");
Expect(1, 120832, '\p{^WB=nu}', "");
Expect(1, 120832, '\P{WB=nu}', "");
Expect(0, 120832, '\P{^WB=nu}', "");
Expect(1, 120831, '\p{WB=		NU}', "");
Expect(0, 120831, '\p{^WB=		NU}', "");
Expect(0, 120831, '\P{WB=		NU}', "");
Expect(1, 120831, '\P{^WB=		NU}', "");
Expect(0, 120832, '\p{WB=		NU}', "");
Expect(1, 120832, '\p{^WB=		NU}', "");
Expect(1, 120832, '\P{WB=		NU}', "");
Expect(0, 120832, '\P{^WB=		NU}', "");
Error('\p{Is_Word_Break=/a/ NUMERIC}');
Error('\P{Is_Word_Break=/a/ NUMERIC}');
Expect(1, 120831, '\p{Is_Word_Break=numeric}', "");
Expect(0, 120831, '\p{^Is_Word_Break=numeric}', "");
Expect(0, 120831, '\P{Is_Word_Break=numeric}', "");
Expect(1, 120831, '\P{^Is_Word_Break=numeric}', "");
Expect(0, 120832, '\p{Is_Word_Break=numeric}', "");
Expect(1, 120832, '\p{^Is_Word_Break=numeric}', "");
Expect(1, 120832, '\P{Is_Word_Break=numeric}', "");
Expect(0, 120832, '\P{^Is_Word_Break=numeric}', "");
Expect(1, 120831, '\p{Is_Word_Break=NUMERIC}', "");
Expect(0, 120831, '\p{^Is_Word_Break=NUMERIC}', "");
Expect(0, 120831, '\P{Is_Word_Break=NUMERIC}', "");
Expect(1, 120831, '\P{^Is_Word_Break=NUMERIC}', "");
Expect(0, 120832, '\p{Is_Word_Break=NUMERIC}', "");
Expect(1, 120832, '\p{^Is_Word_Break=NUMERIC}', "");
Expect(1, 120832, '\P{Is_Word_Break=NUMERIC}', "");
Expect(0, 120832, '\P{^Is_Word_Break=NUMERIC}', "");
Error('\p{Is_WB=:=- NU}');
Error('\P{Is_WB=:=- NU}');
Expect(1, 120831, '\p{Is_WB=nu}', "");
Expect(0, 120831, '\p{^Is_WB=nu}', "");
Expect(0, 120831, '\P{Is_WB=nu}', "");
Expect(1, 120831, '\P{^Is_WB=nu}', "");
Expect(0, 120832, '\p{Is_WB=nu}', "");
Expect(1, 120832, '\p{^Is_WB=nu}', "");
Expect(1, 120832, '\P{Is_WB=nu}', "");
Expect(0, 120832, '\P{^Is_WB=nu}', "");
Expect(1, 120831, '\p{Is_WB=		NU}', "");
Expect(0, 120831, '\p{^Is_WB=		NU}', "");
Expect(0, 120831, '\P{Is_WB=		NU}', "");
Expect(1, 120831, '\P{^Is_WB=		NU}', "");
Expect(0, 120832, '\p{Is_WB=		NU}', "");
Expect(1, 120832, '\p{^Is_WB=		NU}', "");
Expect(1, 120832, '\P{Is_WB=		NU}', "");
Expect(0, 120832, '\P{^Is_WB=		NU}', "");
Error('\p{Word_Break=:=		REGIONAL_indicator}');
Error('\P{Word_Break=:=		REGIONAL_indicator}');
Expect(1, 127487, '\p{Word_Break=regionalindicator}', "");
Expect(0, 127487, '\p{^Word_Break=regionalindicator}', "");
Expect(0, 127487, '\P{Word_Break=regionalindicator}', "");
Expect(1, 127487, '\P{^Word_Break=regionalindicator}', "");
Expect(0, 127488, '\p{Word_Break=regionalindicator}', "");
Expect(1, 127488, '\p{^Word_Break=regionalindicator}', "");
Expect(1, 127488, '\P{Word_Break=regionalindicator}', "");
Expect(0, 127488, '\P{^Word_Break=regionalindicator}', "");
Expect(1, 127487, '\p{Word_Break= _Regional_Indicator}', "");
Expect(0, 127487, '\p{^Word_Break= _Regional_Indicator}', "");
Expect(0, 127487, '\P{Word_Break= _Regional_Indicator}', "");
Expect(1, 127487, '\P{^Word_Break= _Regional_Indicator}', "");
Expect(0, 127488, '\p{Word_Break= _Regional_Indicator}', "");
Expect(1, 127488, '\p{^Word_Break= _Regional_Indicator}', "");
Expect(1, 127488, '\P{Word_Break= _Regional_Indicator}', "");
Expect(0, 127488, '\P{^Word_Break= _Regional_Indicator}', "");
Error('\p{WB=- RI/a/}');
Error('\P{WB=- RI/a/}');
Expect(1, 127487, '\p{WB=ri}', "");
Expect(0, 127487, '\p{^WB=ri}', "");
Expect(0, 127487, '\P{WB=ri}', "");
Expect(1, 127487, '\P{^WB=ri}', "");
Expect(0, 127488, '\p{WB=ri}', "");
Expect(1, 127488, '\p{^WB=ri}', "");
Expect(1, 127488, '\P{WB=ri}', "");
Expect(0, 127488, '\P{^WB=ri}', "");
Expect(1, 127487, '\p{WB=	 RI}', "");
Expect(0, 127487, '\p{^WB=	 RI}', "");
Expect(0, 127487, '\P{WB=	 RI}', "");
Expect(1, 127487, '\P{^WB=	 RI}', "");
Expect(0, 127488, '\p{WB=	 RI}', "");
Expect(1, 127488, '\p{^WB=	 RI}', "");
Expect(1, 127488, '\P{WB=	 RI}', "");
Expect(0, 127488, '\P{^WB=	 RI}', "");
Error('\p{Is_Word_Break=-Regional_Indicator/a/}');
Error('\P{Is_Word_Break=-Regional_Indicator/a/}');
Expect(1, 127487, '\p{Is_Word_Break=regionalindicator}', "");
Expect(0, 127487, '\p{^Is_Word_Break=regionalindicator}', "");
Expect(0, 127487, '\P{Is_Word_Break=regionalindicator}', "");
Expect(1, 127487, '\P{^Is_Word_Break=regionalindicator}', "");
Expect(0, 127488, '\p{Is_Word_Break=regionalindicator}', "");
Expect(1, 127488, '\p{^Is_Word_Break=regionalindicator}', "");
Expect(1, 127488, '\P{Is_Word_Break=regionalindicator}', "");
Expect(0, 127488, '\P{^Is_Word_Break=regionalindicator}', "");
Expect(1, 127487, '\p{Is_Word_Break=-_regional_indicator}', "");
Expect(0, 127487, '\p{^Is_Word_Break=-_regional_indicator}', "");
Expect(0, 127487, '\P{Is_Word_Break=-_regional_indicator}', "");
Expect(1, 127487, '\P{^Is_Word_Break=-_regional_indicator}', "");
Expect(0, 127488, '\p{Is_Word_Break=-_regional_indicator}', "");
Expect(1, 127488, '\p{^Is_Word_Break=-_regional_indicator}', "");
Expect(1, 127488, '\P{Is_Word_Break=-_regional_indicator}', "");
Expect(0, 127488, '\P{^Is_Word_Break=-_regional_indicator}', "");
Error('\p{Is_WB=RI/a/}');
Error('\P{Is_WB=RI/a/}');
Expect(1, 127487, '\p{Is_WB=ri}', "");
Expect(0, 127487, '\p{^Is_WB=ri}', "");
Expect(0, 127487, '\P{Is_WB=ri}', "");
Expect(1, 127487, '\P{^Is_WB=ri}', "");
Expect(0, 127488, '\p{Is_WB=ri}', "");
Expect(1, 127488, '\p{^Is_WB=ri}', "");
Expect(1, 127488, '\P{Is_WB=ri}', "");
Expect(0, 127488, '\P{^Is_WB=ri}', "");
Expect(1, 127487, '\p{Is_WB=_ RI}', "");
Expect(0, 127487, '\p{^Is_WB=_ RI}', "");
Expect(0, 127487, '\P{Is_WB=_ RI}', "");
Expect(1, 127487, '\P{^Is_WB=_ RI}', "");
Expect(0, 127488, '\p{Is_WB=_ RI}', "");
Expect(1, 127488, '\p{^Is_WB=_ RI}', "");
Expect(1, 127488, '\P{Is_WB=_ RI}', "");
Expect(0, 127488, '\P{^Is_WB=_ RI}', "");
Error('\p{Word_Break=		Single_Quote/a/}');
Error('\P{Word_Break=		Single_Quote/a/}');
Expect(1, 39, '\p{Word_Break=singlequote}', "");
Expect(0, 39, '\p{^Word_Break=singlequote}', "");
Expect(0, 39, '\P{Word_Break=singlequote}', "");
Expect(1, 39, '\P{^Word_Break=singlequote}', "");
Expect(0, 40, '\p{Word_Break=singlequote}', "");
Expect(1, 40, '\p{^Word_Break=singlequote}', "");
Expect(1, 40, '\P{Word_Break=singlequote}', "");
Expect(0, 40, '\P{^Word_Break=singlequote}', "");
Expect(1, 39, '\p{Word_Break=__single_QUOTE}', "");
Expect(0, 39, '\p{^Word_Break=__single_QUOTE}', "");
Expect(0, 39, '\P{Word_Break=__single_QUOTE}', "");
Expect(1, 39, '\P{^Word_Break=__single_QUOTE}', "");
Expect(0, 40, '\p{Word_Break=__single_QUOTE}', "");
Expect(1, 40, '\p{^Word_Break=__single_QUOTE}', "");
Expect(1, 40, '\P{Word_Break=__single_QUOTE}', "");
Expect(0, 40, '\P{^Word_Break=__single_QUOTE}', "");
Error('\p{WB=_SQ:=}');
Error('\P{WB=_SQ:=}');
Expect(1, 39, '\p{WB=sq}', "");
Expect(0, 39, '\p{^WB=sq}', "");
Expect(0, 39, '\P{WB=sq}', "");
Expect(1, 39, '\P{^WB=sq}', "");
Expect(0, 40, '\p{WB=sq}', "");
Expect(1, 40, '\p{^WB=sq}', "");
Expect(1, 40, '\P{WB=sq}', "");
Expect(0, 40, '\P{^WB=sq}', "");
Expect(1, 39, '\p{WB: 	SQ}', "");
Expect(0, 39, '\p{^WB: 	SQ}', "");
Expect(0, 39, '\P{WB: 	SQ}', "");
Expect(1, 39, '\P{^WB: 	SQ}', "");
Expect(0, 40, '\p{WB: 	SQ}', "");
Expect(1, 40, '\p{^WB: 	SQ}', "");
Expect(1, 40, '\P{WB: 	SQ}', "");
Expect(0, 40, '\P{^WB: 	SQ}', "");
Error('\p{Is_Word_Break=-:=Single_Quote}');
Error('\P{Is_Word_Break=-:=Single_Quote}');
Expect(1, 39, '\p{Is_Word_Break=singlequote}', "");
Expect(0, 39, '\p{^Is_Word_Break=singlequote}', "");
Expect(0, 39, '\P{Is_Word_Break=singlequote}', "");
Expect(1, 39, '\P{^Is_Word_Break=singlequote}', "");
Expect(0, 40, '\p{Is_Word_Break=singlequote}', "");
Expect(1, 40, '\p{^Is_Word_Break=singlequote}', "");
Expect(1, 40, '\P{Is_Word_Break=singlequote}', "");
Expect(0, 40, '\P{^Is_Word_Break=singlequote}', "");
Expect(1, 39, '\p{Is_Word_Break=_ Single_Quote}', "");
Expect(0, 39, '\p{^Is_Word_Break=_ Single_Quote}', "");
Expect(0, 39, '\P{Is_Word_Break=_ Single_Quote}', "");
Expect(1, 39, '\P{^Is_Word_Break=_ Single_Quote}', "");
Expect(0, 40, '\p{Is_Word_Break=_ Single_Quote}', "");
Expect(1, 40, '\p{^Is_Word_Break=_ Single_Quote}', "");
Expect(1, 40, '\P{Is_Word_Break=_ Single_Quote}', "");
Expect(0, 40, '\P{^Is_Word_Break=_ Single_Quote}', "");
Error('\p{Is_WB= :=SQ}');
Error('\P{Is_WB= :=SQ}');
Expect(1, 39, '\p{Is_WB:	sq}', "");
Expect(0, 39, '\p{^Is_WB:	sq}', "");
Expect(0, 39, '\P{Is_WB:	sq}', "");
Expect(1, 39, '\P{^Is_WB:	sq}', "");
Expect(0, 40, '\p{Is_WB:	sq}', "");
Expect(1, 40, '\p{^Is_WB:	sq}', "");
Expect(1, 40, '\P{Is_WB:	sq}', "");
Expect(0, 40, '\P{^Is_WB:	sq}', "");
Expect(1, 39, '\p{Is_WB= -SQ}', "");
Expect(0, 39, '\p{^Is_WB= -SQ}', "");
Expect(0, 39, '\P{Is_WB= -SQ}', "");
Expect(1, 39, '\P{^Is_WB= -SQ}', "");
Expect(0, 40, '\p{Is_WB= -SQ}', "");
Expect(1, 40, '\p{^Is_WB= -SQ}', "");
Expect(1, 40, '\P{Is_WB= -SQ}', "");
Expect(0, 40, '\P{^Is_WB= -SQ}', "");
Error('\p{Word_Break=__Other/a/}');
Error('\P{Word_Break=__Other/a/}');
Expect(1, 918000, '\p{Word_Break=other}', "");
Expect(0, 918000, '\p{^Word_Break=other}', "");
Expect(0, 918000, '\P{Word_Break=other}', "");
Expect(1, 918000, '\P{^Word_Break=other}', "");
Expect(0, 917999, '\p{Word_Break=other}', "");
Expect(1, 917999, '\p{^Word_Break=other}', "");
Expect(1, 917999, '\P{Word_Break=other}', "");
Expect(0, 917999, '\P{^Word_Break=other}', "");
Expect(1, 918000, '\p{Word_Break= Other}', "");
Expect(0, 918000, '\p{^Word_Break= Other}', "");
Expect(0, 918000, '\P{Word_Break= Other}', "");
Expect(1, 918000, '\P{^Word_Break= Other}', "");
Expect(0, 917999, '\p{Word_Break= Other}', "");
Expect(1, 917999, '\p{^Word_Break= Other}', "");
Expect(1, 917999, '\P{Word_Break= Other}', "");
Expect(0, 917999, '\P{^Word_Break= Other}', "");
Error('\p{WB=/a/	XX}');
Error('\P{WB=/a/	XX}');
Expect(1, 918000, '\p{WB:	xx}', "");
Expect(0, 918000, '\p{^WB:	xx}', "");
Expect(0, 918000, '\P{WB:	xx}', "");
Expect(1, 918000, '\P{^WB:	xx}', "");
Expect(0, 917999, '\p{WB:	xx}', "");
Expect(1, 917999, '\p{^WB:	xx}', "");
Expect(1, 917999, '\P{WB:	xx}', "");
Expect(0, 917999, '\P{^WB:	xx}', "");
Expect(1, 918000, '\p{WB=	_XX}', "");
Expect(0, 918000, '\p{^WB=	_XX}', "");
Expect(0, 918000, '\P{WB=	_XX}', "");
Expect(1, 918000, '\P{^WB=	_XX}', "");
Expect(0, 917999, '\p{WB=	_XX}', "");
Expect(1, 917999, '\p{^WB=	_XX}', "");
Expect(1, 917999, '\P{WB=	_XX}', "");
Expect(0, 917999, '\P{^WB=	_XX}', "");
Error('\p{Is_Word_Break:    :=Other}');
Error('\P{Is_Word_Break:    :=Other}');
Expect(1, 918000, '\p{Is_Word_Break=other}', "");
Expect(0, 918000, '\p{^Is_Word_Break=other}', "");
Expect(0, 918000, '\P{Is_Word_Break=other}', "");
Expect(1, 918000, '\P{^Is_Word_Break=other}', "");
Expect(0, 917999, '\p{Is_Word_Break=other}', "");
Expect(1, 917999, '\p{^Is_Word_Break=other}', "");
Expect(1, 917999, '\P{Is_Word_Break=other}', "");
Expect(0, 917999, '\P{^Is_Word_Break=other}', "");
Expect(1, 918000, '\p{Is_Word_Break=-_Other}', "");
Expect(0, 918000, '\p{^Is_Word_Break=-_Other}', "");
Expect(0, 918000, '\P{Is_Word_Break=-_Other}', "");
Expect(1, 918000, '\P{^Is_Word_Break=-_Other}', "");
Expect(0, 917999, '\p{Is_Word_Break=-_Other}', "");
Expect(1, 917999, '\p{^Is_Word_Break=-_Other}', "");
Expect(1, 917999, '\P{Is_Word_Break=-_Other}', "");
Expect(0, 917999, '\P{^Is_Word_Break=-_Other}', "");
Error('\p{Is_WB: 	:=xx}');
Error('\P{Is_WB: 	:=xx}');
Expect(1, 918000, '\p{Is_WB:   xx}', "");
Expect(0, 918000, '\p{^Is_WB:   xx}', "");
Expect(0, 918000, '\P{Is_WB:   xx}', "");
Expect(1, 918000, '\P{^Is_WB:   xx}', "");
Expect(0, 917999, '\p{Is_WB:   xx}', "");
Expect(1, 917999, '\p{^Is_WB:   xx}', "");
Expect(1, 917999, '\P{Is_WB:   xx}', "");
Expect(0, 917999, '\P{^Is_WB:   xx}', "");
Expect(1, 918000, '\p{Is_WB=	XX}', "");
Expect(0, 918000, '\p{^Is_WB=	XX}', "");
Expect(0, 918000, '\P{Is_WB=	XX}', "");
Expect(1, 918000, '\P{^Is_WB=	XX}', "");
Expect(0, 917999, '\p{Is_WB=	XX}', "");
Expect(1, 917999, '\p{^Is_WB=	XX}', "");
Expect(1, 917999, '\P{Is_WB=	XX}', "");
Expect(0, 917999, '\P{^Is_WB=	XX}', "");
Error('\p{White_Space=:=no}');
Error('\P{White_Space=:=no}');
Expect(1, 12289, '\p{White_Space=no}', "");
Expect(0, 12289, '\p{^White_Space=no}', "");
Expect(0, 12289, '\P{White_Space=no}', "");
Expect(1, 12289, '\P{^White_Space=no}', "");
Expect(0, 12288, '\p{White_Space=no}', "");
Expect(1, 12288, '\p{^White_Space=no}', "");
Expect(1, 12288, '\P{White_Space=no}', "");
Expect(0, 12288, '\P{^White_Space=no}', "");
Expect(1, 12289, '\p{White_Space= no}', "");
Expect(0, 12289, '\p{^White_Space= no}', "");
Expect(0, 12289, '\P{White_Space= no}', "");
Expect(1, 12289, '\P{^White_Space= no}', "");
Expect(0, 12288, '\p{White_Space= no}', "");
Expect(1, 12288, '\p{^White_Space= no}', "");
Expect(1, 12288, '\P{White_Space= no}', "");
Expect(0, 12288, '\P{^White_Space= no}', "");
Error('\p{WSpace=_/a/N}');
Error('\P{WSpace=_/a/N}');
Expect(1, 12289, '\p{WSpace=n}', "");
Expect(0, 12289, '\p{^WSpace=n}', "");
Expect(0, 12289, '\P{WSpace=n}', "");
Expect(1, 12289, '\P{^WSpace=n}', "");
Expect(0, 12288, '\p{WSpace=n}', "");
Expect(1, 12288, '\p{^WSpace=n}', "");
Expect(1, 12288, '\P{WSpace=n}', "");
Expect(0, 12288, '\P{^WSpace=n}', "");
Expect(1, 12289, '\p{WSpace: 	_N}', "");
Expect(0, 12289, '\p{^WSpace: 	_N}', "");
Expect(0, 12289, '\P{WSpace: 	_N}', "");
Expect(1, 12289, '\P{^WSpace: 	_N}', "");
Expect(0, 12288, '\p{WSpace: 	_N}', "");
Expect(1, 12288, '\p{^WSpace: 	_N}', "");
Expect(1, 12288, '\P{WSpace: 	_N}', "");
Expect(0, 12288, '\P{^WSpace: 	_N}', "");
Error('\p{Space= -F/a/}');
Error('\P{Space= -F/a/}');
Expect(1, 12289, '\p{Space=f}', "");
Expect(0, 12289, '\p{^Space=f}', "");
Expect(0, 12289, '\P{Space=f}', "");
Expect(1, 12289, '\P{^Space=f}', "");
Expect(0, 12288, '\p{Space=f}', "");
Expect(1, 12288, '\p{^Space=f}', "");
Expect(1, 12288, '\P{Space=f}', "");
Expect(0, 12288, '\P{^Space=f}', "");
Expect(1, 12289, '\p{Space=--f}', "");
Expect(0, 12289, '\p{^Space=--f}', "");
Expect(0, 12289, '\P{Space=--f}', "");
Expect(1, 12289, '\P{^Space=--f}', "");
Expect(0, 12288, '\p{Space=--f}', "");
Expect(1, 12288, '\p{^Space=--f}', "");
Expect(1, 12288, '\P{Space=--f}', "");
Expect(0, 12288, '\P{^Space=--f}', "");
Error('\p{Is_White_Space=-:=False}');
Error('\P{Is_White_Space=-:=False}');
Expect(1, 12289, '\p{Is_White_Space=false}', "");
Expect(0, 12289, '\p{^Is_White_Space=false}', "");
Expect(0, 12289, '\P{Is_White_Space=false}', "");
Expect(1, 12289, '\P{^Is_White_Space=false}', "");
Expect(0, 12288, '\p{Is_White_Space=false}', "");
Expect(1, 12288, '\p{^Is_White_Space=false}', "");
Expect(1, 12288, '\P{Is_White_Space=false}', "");
Expect(0, 12288, '\P{^Is_White_Space=false}', "");
Expect(1, 12289, '\p{Is_White_Space=	-false}', "");
Expect(0, 12289, '\p{^Is_White_Space=	-false}', "");
Expect(0, 12289, '\P{Is_White_Space=	-false}', "");
Expect(1, 12289, '\P{^Is_White_Space=	-false}', "");
Expect(0, 12288, '\p{Is_White_Space=	-false}', "");
Expect(1, 12288, '\p{^Is_White_Space=	-false}', "");
Expect(1, 12288, '\P{Is_White_Space=	-false}', "");
Expect(0, 12288, '\P{^Is_White_Space=	-false}', "");
Error('\p{Is_WSpace= /a/NO}');
Error('\P{Is_WSpace= /a/NO}');
Expect(1, 12289, '\p{Is_WSpace:   no}', "");
Expect(0, 12289, '\p{^Is_WSpace:   no}', "");
Expect(0, 12289, '\P{Is_WSpace:   no}', "");
Expect(1, 12289, '\P{^Is_WSpace:   no}', "");
Expect(0, 12288, '\p{Is_WSpace:   no}', "");
Expect(1, 12288, '\p{^Is_WSpace:   no}', "");
Expect(1, 12288, '\P{Is_WSpace:   no}', "");
Expect(0, 12288, '\P{^Is_WSpace:   no}', "");
Expect(1, 12289, '\p{Is_WSpace=	No}', "");
Expect(0, 12289, '\p{^Is_WSpace=	No}', "");
Expect(0, 12289, '\P{Is_WSpace=	No}', "");
Expect(1, 12289, '\P{^Is_WSpace=	No}', "");
Expect(0, 12288, '\p{Is_WSpace=	No}', "");
Expect(1, 12288, '\p{^Is_WSpace=	No}', "");
Expect(1, 12288, '\P{Is_WSpace=	No}', "");
Expect(0, 12288, '\P{^Is_WSpace=	No}', "");
Error('\p{Is_Space=/a/N}');
Error('\P{Is_Space=/a/N}');
Expect(1, 12289, '\p{Is_Space=n}', "");
Expect(0, 12289, '\p{^Is_Space=n}', "");
Expect(0, 12289, '\P{Is_Space=n}', "");
Expect(1, 12289, '\P{^Is_Space=n}', "");
Expect(0, 12288, '\p{Is_Space=n}', "");
Expect(1, 12288, '\p{^Is_Space=n}', "");
Expect(1, 12288, '\P{Is_Space=n}', "");
Expect(0, 12288, '\P{^Is_Space=n}', "");
Expect(1, 12289, '\p{Is_Space=	_n}', "");
Expect(0, 12289, '\p{^Is_Space=	_n}', "");
Expect(0, 12289, '\P{Is_Space=	_n}', "");
Expect(1, 12289, '\P{^Is_Space=	_n}', "");
Expect(0, 12288, '\p{Is_Space=	_n}', "");
Expect(1, 12288, '\p{^Is_Space=	_n}', "");
Expect(1, 12288, '\P{Is_Space=	_n}', "");
Expect(0, 12288, '\P{^Is_Space=	_n}', "");
Error('\p{White_Space= :=Yes}');
Error('\P{White_Space= :=Yes}');
Expect(1, 12288, '\p{White_Space=yes}', "");
Expect(0, 12288, '\p{^White_Space=yes}', "");
Expect(0, 12288, '\P{White_Space=yes}', "");
Expect(1, 12288, '\P{^White_Space=yes}', "");
Expect(0, 12289, '\p{White_Space=yes}', "");
Expect(1, 12289, '\p{^White_Space=yes}', "");
Expect(1, 12289, '\P{White_Space=yes}', "");
Expect(0, 12289, '\P{^White_Space=yes}', "");
Expect(1, 12288, '\p{White_Space:   _yes}', "");
Expect(0, 12288, '\p{^White_Space:   _yes}', "");
Expect(0, 12288, '\P{White_Space:   _yes}', "");
Expect(1, 12288, '\P{^White_Space:   _yes}', "");
Expect(0, 12289, '\p{White_Space:   _yes}', "");
Expect(1, 12289, '\p{^White_Space:   _yes}', "");
Expect(1, 12289, '\P{White_Space:   _yes}', "");
Expect(0, 12289, '\P{^White_Space:   _yes}', "");
Error('\p{WSpace:   _ Y/a/}');
Error('\P{WSpace:   _ Y/a/}');
Expect(1, 12288, '\p{WSpace=y}', "");
Expect(0, 12288, '\p{^WSpace=y}', "");
Expect(0, 12288, '\P{WSpace=y}', "");
Expect(1, 12288, '\P{^WSpace=y}', "");
Expect(0, 12289, '\p{WSpace=y}', "");
Expect(1, 12289, '\p{^WSpace=y}', "");
Expect(1, 12289, '\P{WSpace=y}', "");
Expect(0, 12289, '\P{^WSpace=y}', "");
Expect(1, 12288, '\p{WSpace= -Y}', "");
Expect(0, 12288, '\p{^WSpace= -Y}', "");
Expect(0, 12288, '\P{WSpace= -Y}', "");
Expect(1, 12288, '\P{^WSpace= -Y}', "");
Expect(0, 12289, '\p{WSpace= -Y}', "");
Expect(1, 12289, '\p{^WSpace= -Y}', "");
Expect(1, 12289, '\P{WSpace= -Y}', "");
Expect(0, 12289, '\P{^WSpace= -Y}', "");
Error('\p{Space: 	_T:=}');
Error('\P{Space: 	_T:=}');
Expect(1, 12288, '\p{Space=t}', "");
Expect(0, 12288, '\p{^Space=t}', "");
Expect(0, 12288, '\P{Space=t}', "");
Expect(1, 12288, '\P{^Space=t}', "");
Expect(0, 12289, '\p{Space=t}', "");
Expect(1, 12289, '\p{^Space=t}', "");
Expect(1, 12289, '\P{Space=t}', "");
Expect(0, 12289, '\P{^Space=t}', "");
Expect(1, 12288, '\p{Space=_	t}', "");
Expect(0, 12288, '\p{^Space=_	t}', "");
Expect(0, 12288, '\P{Space=_	t}', "");
Expect(1, 12288, '\P{^Space=_	t}', "");
Expect(0, 12289, '\p{Space=_	t}', "");
Expect(1, 12289, '\p{^Space=_	t}', "");
Expect(1, 12289, '\P{Space=_	t}', "");
Expect(0, 12289, '\P{^Space=_	t}', "");
Error('\p{Is_White_Space=-/a/True}');
Error('\P{Is_White_Space=-/a/True}');
Expect(1, 12288, '\p{Is_White_Space=true}', "");
Expect(0, 12288, '\p{^Is_White_Space=true}', "");
Expect(0, 12288, '\P{Is_White_Space=true}', "");
Expect(1, 12288, '\P{^Is_White_Space=true}', "");
Expect(0, 12289, '\p{Is_White_Space=true}', "");
Expect(1, 12289, '\p{^Is_White_Space=true}', "");
Expect(1, 12289, '\P{Is_White_Space=true}', "");
Expect(0, 12289, '\P{^Is_White_Space=true}', "");
Expect(1, 12288, '\p{Is_White_Space=  True}', "");
Expect(0, 12288, '\p{^Is_White_Space=  True}', "");
Expect(0, 12288, '\P{Is_White_Space=  True}', "");
Expect(1, 12288, '\P{^Is_White_Space=  True}', "");
Expect(0, 12289, '\p{Is_White_Space=  True}', "");
Expect(1, 12289, '\p{^Is_White_Space=  True}', "");
Expect(1, 12289, '\P{Is_White_Space=  True}', "");
Expect(0, 12289, '\P{^Is_White_Space=  True}', "");
Error('\p{Is_WSpace:	:=	 YES}');
Error('\P{Is_WSpace:	:=	 YES}');
Expect(1, 12288, '\p{Is_WSpace=yes}', "");
Expect(0, 12288, '\p{^Is_WSpace=yes}', "");
Expect(0, 12288, '\P{Is_WSpace=yes}', "");
Expect(1, 12288, '\P{^Is_WSpace=yes}', "");
Expect(0, 12289, '\p{Is_WSpace=yes}', "");
Expect(1, 12289, '\p{^Is_WSpace=yes}', "");
Expect(1, 12289, '\P{Is_WSpace=yes}', "");
Expect(0, 12289, '\P{^Is_WSpace=yes}', "");
Expect(1, 12288, '\p{Is_WSpace=	yes}', "");
Expect(0, 12288, '\p{^Is_WSpace=	yes}', "");
Expect(0, 12288, '\P{Is_WSpace=	yes}', "");
Expect(1, 12288, '\P{^Is_WSpace=	yes}', "");
Expect(0, 12289, '\p{Is_WSpace=	yes}', "");
Expect(1, 12289, '\p{^Is_WSpace=	yes}', "");
Expect(1, 12289, '\P{Is_WSpace=	yes}', "");
Expect(0, 12289, '\P{^Is_WSpace=	yes}', "");
Error('\p{Is_Space=	:=Y}');
Error('\P{Is_Space=	:=Y}');
Expect(1, 12288, '\p{Is_Space=y}', "");
Expect(0, 12288, '\p{^Is_Space=y}', "");
Expect(0, 12288, '\P{Is_Space=y}', "");
Expect(1, 12288, '\P{^Is_Space=y}', "");
Expect(0, 12289, '\p{Is_Space=y}', "");
Expect(1, 12289, '\p{^Is_Space=y}', "");
Expect(1, 12289, '\P{Is_Space=y}', "");
Expect(0, 12289, '\P{^Is_Space=y}', "");
Expect(1, 12288, '\p{Is_Space=-_Y}', "");
Expect(0, 12288, '\p{^Is_Space=-_Y}', "");
Expect(0, 12288, '\P{Is_Space=-_Y}', "");
Expect(1, 12288, '\P{^Is_Space=-_Y}', "");
Expect(0, 12289, '\p{Is_Space=-_Y}', "");
Expect(1, 12289, '\p{^Is_Space=-_Y}', "");
Expect(1, 12289, '\P{Is_Space=-_Y}', "");
Expect(0, 12289, '\P{^Is_Space=-_Y}', "");
Error('\p{XID_Continue=-:=NO}');
Error('\P{XID_Continue=-:=NO}');
Expect(1, 918000, '\p{XID_Continue=no}', "");
Expect(0, 918000, '\p{^XID_Continue=no}', "");
Expect(0, 918000, '\P{XID_Continue=no}', "");
Expect(1, 918000, '\P{^XID_Continue=no}', "");
Expect(0, 917999, '\p{XID_Continue=no}', "");
Expect(1, 917999, '\p{^XID_Continue=no}', "");
Expect(1, 917999, '\P{XID_Continue=no}', "");
Expect(0, 917999, '\P{^XID_Continue=no}', "");
Expect(1, 918000, '\p{XID_Continue= _no}', "");
Expect(0, 918000, '\p{^XID_Continue= _no}', "");
Expect(0, 918000, '\P{XID_Continue= _no}', "");
Expect(1, 918000, '\P{^XID_Continue= _no}', "");
Expect(0, 917999, '\p{XID_Continue= _no}', "");
Expect(1, 917999, '\p{^XID_Continue= _no}', "");
Expect(1, 917999, '\P{XID_Continue= _no}', "");
Expect(0, 917999, '\P{^XID_Continue= _no}', "");
Error('\p{XIDC=:=-_n}');
Error('\P{XIDC=:=-_n}');
Expect(1, 918000, '\p{XIDC=n}', "");
Expect(0, 918000, '\p{^XIDC=n}', "");
Expect(0, 918000, '\P{XIDC=n}', "");
Expect(1, 918000, '\P{^XIDC=n}', "");
Expect(0, 917999, '\p{XIDC=n}', "");
Expect(1, 917999, '\p{^XIDC=n}', "");
Expect(1, 917999, '\P{XIDC=n}', "");
Expect(0, 917999, '\P{^XIDC=n}', "");
Expect(1, 918000, '\p{XIDC=		N}', "");
Expect(0, 918000, '\p{^XIDC=		N}', "");
Expect(0, 918000, '\P{XIDC=		N}', "");
Expect(1, 918000, '\P{^XIDC=		N}', "");
Expect(0, 917999, '\p{XIDC=		N}', "");
Expect(1, 917999, '\p{^XIDC=		N}', "");
Expect(1, 917999, '\P{XIDC=		N}', "");
Expect(0, 917999, '\P{^XIDC=		N}', "");
Error('\p{Is_XID_Continue:	/a/_F}');
Error('\P{Is_XID_Continue:	/a/_F}');
Expect(1, 918000, '\p{Is_XID_Continue=f}', "");
Expect(0, 918000, '\p{^Is_XID_Continue=f}', "");
Expect(0, 918000, '\P{Is_XID_Continue=f}', "");
Expect(1, 918000, '\P{^Is_XID_Continue=f}', "");
Expect(0, 917999, '\p{Is_XID_Continue=f}', "");
Expect(1, 917999, '\p{^Is_XID_Continue=f}', "");
Expect(1, 917999, '\P{Is_XID_Continue=f}', "");
Expect(0, 917999, '\P{^Is_XID_Continue=f}', "");
Expect(1, 918000, '\p{Is_XID_Continue=	F}', "");
Expect(0, 918000, '\p{^Is_XID_Continue=	F}', "");
Expect(0, 918000, '\P{Is_XID_Continue=	F}', "");
Expect(1, 918000, '\P{^Is_XID_Continue=	F}', "");
Expect(0, 917999, '\p{Is_XID_Continue=	F}', "");
Expect(1, 917999, '\p{^Is_XID_Continue=	F}', "");
Expect(1, 917999, '\P{Is_XID_Continue=	F}', "");
Expect(0, 917999, '\P{^Is_XID_Continue=	F}', "");
Error('\p{Is_XIDC=:=	false}');
Error('\P{Is_XIDC=:=	false}');
Expect(1, 918000, '\p{Is_XIDC=false}', "");
Expect(0, 918000, '\p{^Is_XIDC=false}', "");
Expect(0, 918000, '\P{Is_XIDC=false}', "");
Expect(1, 918000, '\P{^Is_XIDC=false}', "");
Expect(0, 917999, '\p{Is_XIDC=false}', "");
Expect(1, 917999, '\p{^Is_XIDC=false}', "");
Expect(1, 917999, '\P{Is_XIDC=false}', "");
Expect(0, 917999, '\P{^Is_XIDC=false}', "");
Expect(1, 918000, '\p{Is_XIDC= false}', "");
Expect(0, 918000, '\p{^Is_XIDC= false}', "");
Expect(0, 918000, '\P{Is_XIDC= false}', "");
Expect(1, 918000, '\P{^Is_XIDC= false}', "");
Expect(0, 917999, '\p{Is_XIDC= false}', "");
Expect(1, 917999, '\p{^Is_XIDC= false}', "");
Expect(1, 917999, '\P{Is_XIDC= false}', "");
Expect(0, 917999, '\P{^Is_XIDC= false}', "");
Error('\p{XID_Continue=-	Yes/a/}');
Error('\P{XID_Continue=-	Yes/a/}');
Expect(1, 917999, '\p{XID_Continue=yes}', "");
Expect(0, 917999, '\p{^XID_Continue=yes}', "");
Expect(0, 917999, '\P{XID_Continue=yes}', "");
Expect(1, 917999, '\P{^XID_Continue=yes}', "");
Expect(0, 918000, '\p{XID_Continue=yes}', "");
Expect(1, 918000, '\p{^XID_Continue=yes}', "");
Expect(1, 918000, '\P{XID_Continue=yes}', "");
Expect(0, 918000, '\P{^XID_Continue=yes}', "");
Expect(1, 917999, '\p{XID_Continue=_Yes}', "");
Expect(0, 917999, '\p{^XID_Continue=_Yes}', "");
Expect(0, 917999, '\P{XID_Continue=_Yes}', "");
Expect(1, 917999, '\P{^XID_Continue=_Yes}', "");
Expect(0, 918000, '\p{XID_Continue=_Yes}', "");
Expect(1, 918000, '\p{^XID_Continue=_Yes}', "");
Expect(1, 918000, '\P{XID_Continue=_Yes}', "");
Expect(0, 918000, '\P{^XID_Continue=_Yes}', "");
Error('\p{XIDC=-/a/y}');
Error('\P{XIDC=-/a/y}');
Expect(1, 917999, '\p{XIDC:   y}', "");
Expect(0, 917999, '\p{^XIDC:   y}', "");
Expect(0, 917999, '\P{XIDC:   y}', "");
Expect(1, 917999, '\P{^XIDC:   y}', "");
Expect(0, 918000, '\p{XIDC:   y}', "");
Expect(1, 918000, '\p{^XIDC:   y}', "");
Expect(1, 918000, '\P{XIDC:   y}', "");
Expect(0, 918000, '\P{^XIDC:   y}', "");
Expect(1, 917999, '\p{XIDC: 	_y}', "");
Expect(0, 917999, '\p{^XIDC: 	_y}', "");
Expect(0, 917999, '\P{XIDC: 	_y}', "");
Expect(1, 917999, '\P{^XIDC: 	_y}', "");
Expect(0, 918000, '\p{XIDC: 	_y}', "");
Expect(1, 918000, '\p{^XIDC: 	_y}', "");
Expect(1, 918000, '\P{XIDC: 	_y}', "");
Expect(0, 918000, '\P{^XIDC: 	_y}', "");
Error('\p{Is_XID_Continue=- T:=}');
Error('\P{Is_XID_Continue=- T:=}');
Expect(1, 917999, '\p{Is_XID_Continue=t}', "");
Expect(0, 917999, '\p{^Is_XID_Continue=t}', "");
Expect(0, 917999, '\P{Is_XID_Continue=t}', "");
Expect(1, 917999, '\P{^Is_XID_Continue=t}', "");
Expect(0, 918000, '\p{Is_XID_Continue=t}', "");
Expect(1, 918000, '\p{^Is_XID_Continue=t}', "");
Expect(1, 918000, '\P{Is_XID_Continue=t}', "");
Expect(0, 918000, '\P{^Is_XID_Continue=t}', "");
Expect(1, 917999, '\p{Is_XID_Continue:T}', "");
Expect(0, 917999, '\p{^Is_XID_Continue:T}', "");
Expect(0, 917999, '\P{Is_XID_Continue:T}', "");
Expect(1, 917999, '\P{^Is_XID_Continue:T}', "");
Expect(0, 918000, '\p{Is_XID_Continue:T}', "");
Expect(1, 918000, '\p{^Is_XID_Continue:T}', "");
Expect(1, 918000, '\P{Is_XID_Continue:T}', "");
Expect(0, 918000, '\P{^Is_XID_Continue:T}', "");
Error('\p{Is_XIDC= 	True/a/}');
Error('\P{Is_XIDC= 	True/a/}');
Expect(1, 917999, '\p{Is_XIDC=true}', "");
Expect(0, 917999, '\p{^Is_XIDC=true}', "");
Expect(0, 917999, '\P{Is_XIDC=true}', "");
Expect(1, 917999, '\P{^Is_XIDC=true}', "");
Expect(0, 918000, '\p{Is_XIDC=true}', "");
Expect(1, 918000, '\p{^Is_XIDC=true}', "");
Expect(1, 918000, '\P{Is_XIDC=true}', "");
Expect(0, 918000, '\P{^Is_XIDC=true}', "");
Expect(1, 917999, '\p{Is_XIDC=_True}', "");
Expect(0, 917999, '\p{^Is_XIDC=_True}', "");
Expect(0, 917999, '\P{Is_XIDC=_True}', "");
Expect(1, 917999, '\P{^Is_XIDC=_True}', "");
Expect(0, 918000, '\p{Is_XIDC=_True}', "");
Expect(1, 918000, '\p{^Is_XIDC=_True}', "");
Expect(1, 918000, '\P{Is_XIDC=_True}', "");
Expect(0, 918000, '\P{^Is_XIDC=_True}', "");
Error('\p{XID_Start=:= NO}');
Error('\P{XID_Start=:= NO}');
Expect(1, 195102, '\p{XID_Start=no}', "");
Expect(0, 195102, '\p{^XID_Start=no}', "");
Expect(0, 195102, '\P{XID_Start=no}', "");
Expect(1, 195102, '\P{^XID_Start=no}', "");
Expect(0, 195101, '\p{XID_Start=no}', "");
Expect(1, 195101, '\p{^XID_Start=no}', "");
Expect(1, 195101, '\P{XID_Start=no}', "");
Expect(0, 195101, '\P{^XID_Start=no}', "");
Expect(1, 195102, '\p{XID_Start= No}', "");
Expect(0, 195102, '\p{^XID_Start= No}', "");
Expect(0, 195102, '\P{XID_Start= No}', "");
Expect(1, 195102, '\P{^XID_Start= No}', "");
Expect(0, 195101, '\p{XID_Start= No}', "");
Expect(1, 195101, '\p{^XID_Start= No}', "");
Expect(1, 195101, '\P{XID_Start= No}', "");
Expect(0, 195101, '\P{^XID_Start= No}', "");
Error('\p{XIDS=/a/_-N}');
Error('\P{XIDS=/a/_-N}');
Expect(1, 195102, '\p{XIDS=n}', "");
Expect(0, 195102, '\p{^XIDS=n}', "");
Expect(0, 195102, '\P{XIDS=n}', "");
Expect(1, 195102, '\P{^XIDS=n}', "");
Expect(0, 195101, '\p{XIDS=n}', "");
Expect(1, 195101, '\p{^XIDS=n}', "");
Expect(1, 195101, '\P{XIDS=n}', "");
Expect(0, 195101, '\P{^XIDS=n}', "");
Expect(1, 195102, '\p{XIDS=- N}', "");
Expect(0, 195102, '\p{^XIDS=- N}', "");
Expect(0, 195102, '\P{XIDS=- N}', "");
Expect(1, 195102, '\P{^XIDS=- N}', "");
Expect(0, 195101, '\p{XIDS=- N}', "");
Expect(1, 195101, '\p{^XIDS=- N}', "");
Expect(1, 195101, '\P{XIDS=- N}', "");
Expect(0, 195101, '\P{^XIDS=- N}', "");
Error('\p{Is_XID_Start=:=-_F}');
Error('\P{Is_XID_Start=:=-_F}');
Expect(1, 195102, '\p{Is_XID_Start=f}', "");
Expect(0, 195102, '\p{^Is_XID_Start=f}', "");
Expect(0, 195102, '\P{Is_XID_Start=f}', "");
Expect(1, 195102, '\P{^Is_XID_Start=f}', "");
Expect(0, 195101, '\p{Is_XID_Start=f}', "");
Expect(1, 195101, '\p{^Is_XID_Start=f}', "");
Expect(1, 195101, '\P{Is_XID_Start=f}', "");
Expect(0, 195101, '\P{^Is_XID_Start=f}', "");
Expect(1, 195102, '\p{Is_XID_Start=_F}', "");
Expect(0, 195102, '\p{^Is_XID_Start=_F}', "");
Expect(0, 195102, '\P{Is_XID_Start=_F}', "");
Expect(1, 195102, '\P{^Is_XID_Start=_F}', "");
Expect(0, 195101, '\p{Is_XID_Start=_F}', "");
Expect(1, 195101, '\p{^Is_XID_Start=_F}', "");
Expect(1, 195101, '\P{Is_XID_Start=_F}', "");
Expect(0, 195101, '\P{^Is_XID_Start=_F}', "");
Error('\p{Is_XIDS= :=FALSE}');
Error('\P{Is_XIDS= :=FALSE}');
Expect(1, 195102, '\p{Is_XIDS=false}', "");
Expect(0, 195102, '\p{^Is_XIDS=false}', "");
Expect(0, 195102, '\P{Is_XIDS=false}', "");
Expect(1, 195102, '\P{^Is_XIDS=false}', "");
Expect(0, 195101, '\p{Is_XIDS=false}', "");
Expect(1, 195101, '\p{^Is_XIDS=false}', "");
Expect(1, 195101, '\P{Is_XIDS=false}', "");
Expect(0, 195101, '\P{^Is_XIDS=false}', "");
Expect(1, 195102, '\p{Is_XIDS=	FALSE}', "");
Expect(0, 195102, '\p{^Is_XIDS=	FALSE}', "");
Expect(0, 195102, '\P{Is_XIDS=	FALSE}', "");
Expect(1, 195102, '\P{^Is_XIDS=	FALSE}', "");
Expect(0, 195101, '\p{Is_XIDS=	FALSE}', "");
Expect(1, 195101, '\p{^Is_XIDS=	FALSE}', "");
Expect(1, 195101, '\P{Is_XIDS=	FALSE}', "");
Expect(0, 195101, '\P{^Is_XIDS=	FALSE}', "");
Error('\p{XID_Start:	_/a/Yes}');
Error('\P{XID_Start:	_/a/Yes}');
Expect(1, 195101, '\p{XID_Start=yes}', "");
Expect(0, 195101, '\p{^XID_Start=yes}', "");
Expect(0, 195101, '\P{XID_Start=yes}', "");
Expect(1, 195101, '\P{^XID_Start=yes}', "");
Expect(0, 195102, '\p{XID_Start=yes}', "");
Expect(1, 195102, '\p{^XID_Start=yes}', "");
Expect(1, 195102, '\P{XID_Start=yes}', "");
Expect(0, 195102, '\P{^XID_Start=yes}', "");
Expect(1, 195101, '\p{XID_Start=		YES}', "");
Expect(0, 195101, '\p{^XID_Start=		YES}', "");
Expect(0, 195101, '\P{XID_Start=		YES}', "");
Expect(1, 195101, '\P{^XID_Start=		YES}', "");
Expect(0, 195102, '\p{XID_Start=		YES}', "");
Expect(1, 195102, '\p{^XID_Start=		YES}', "");
Expect(1, 195102, '\P{XID_Start=		YES}', "");
Expect(0, 195102, '\P{^XID_Start=		YES}', "");
Error('\p{XIDS=/a/_Y}');
Error('\P{XIDS=/a/_Y}');
Expect(1, 195101, '\p{XIDS=y}', "");
Expect(0, 195101, '\p{^XIDS=y}', "");
Expect(0, 195101, '\P{XIDS=y}', "");
Expect(1, 195101, '\P{^XIDS=y}', "");
Expect(0, 195102, '\p{XIDS=y}', "");
Expect(1, 195102, '\p{^XIDS=y}', "");
Expect(1, 195102, '\P{XIDS=y}', "");
Expect(0, 195102, '\P{^XIDS=y}', "");
Expect(1, 195101, '\p{XIDS=_	y}', "");
Expect(0, 195101, '\p{^XIDS=_	y}', "");
Expect(0, 195101, '\P{XIDS=_	y}', "");
Expect(1, 195101, '\P{^XIDS=_	y}', "");
Expect(0, 195102, '\p{XIDS=_	y}', "");
Expect(1, 195102, '\p{^XIDS=_	y}', "");
Expect(1, 195102, '\P{XIDS=_	y}', "");
Expect(0, 195102, '\P{^XIDS=_	y}', "");
Error('\p{Is_XID_Start=-t/a/}');
Error('\P{Is_XID_Start=-t/a/}');
Expect(1, 195101, '\p{Is_XID_Start=t}', "");
Expect(0, 195101, '\p{^Is_XID_Start=t}', "");
Expect(0, 195101, '\P{Is_XID_Start=t}', "");
Expect(1, 195101, '\P{^Is_XID_Start=t}', "");
Expect(0, 195102, '\p{Is_XID_Start=t}', "");
Expect(1, 195102, '\p{^Is_XID_Start=t}', "");
Expect(1, 195102, '\P{Is_XID_Start=t}', "");
Expect(0, 195102, '\P{^Is_XID_Start=t}', "");
Expect(1, 195101, '\p{Is_XID_Start=_T}', "");
Expect(0, 195101, '\p{^Is_XID_Start=_T}', "");
Expect(0, 195101, '\P{Is_XID_Start=_T}', "");
Expect(1, 195101, '\P{^Is_XID_Start=_T}', "");
Expect(0, 195102, '\p{Is_XID_Start=_T}', "");
Expect(1, 195102, '\p{^Is_XID_Start=_T}', "");
Expect(1, 195102, '\P{Is_XID_Start=_T}', "");
Expect(0, 195102, '\P{^Is_XID_Start=_T}', "");
Error('\p{Is_XIDS= :=True}');
Error('\P{Is_XIDS= :=True}');
Expect(1, 195101, '\p{Is_XIDS=true}', "");
Expect(0, 195101, '\p{^Is_XIDS=true}', "");
Expect(0, 195101, '\P{Is_XIDS=true}', "");
Expect(1, 195101, '\P{^Is_XIDS=true}', "");
Expect(0, 195102, '\p{Is_XIDS=true}', "");
Expect(1, 195102, '\p{^Is_XIDS=true}', "");
Expect(1, 195102, '\P{Is_XIDS=true}', "");
Expect(0, 195102, '\P{^Is_XIDS=true}', "");
Expect(1, 195101, '\p{Is_XIDS=_	True}', "");
Expect(0, 195101, '\p{^Is_XIDS=_	True}', "");
Expect(0, 195101, '\P{Is_XIDS=_	True}', "");
Expect(1, 195101, '\P{^Is_XIDS=_	True}', "");
Expect(0, 195102, '\p{Is_XIDS=_	True}', "");
Expect(1, 195102, '\p{^Is_XIDS=_	True}', "");
Expect(1, 195102, '\P{Is_XIDS=_	True}', "");
Expect(0, 195102, '\P{^Is_XIDS=_	True}', "");
Error('\p{Expands_On_NFC=No}');
Error('\P{Expands_On_NFC=No}');
Error('\p{XO_NFC=N}');
Error('\P{XO_NFC=N}');
Error('\p{Is_Expands_On_NFC=F}');
Error('\P{Is_Expands_On_NFC=F}');
Error('\p{Is_XO_NFC:	False}');
Error('\P{Is_XO_NFC:	False}');
Error('\p{Expands_On_NFC=Yes}');
Error('\P{Expands_On_NFC=Yes}');
Error('\p{XO_NFC=Y}');
Error('\P{XO_NFC=Y}');
Error('\p{Is_Expands_On_NFC=T}');
Error('\P{Is_Expands_On_NFC=T}');
Error('\p{Is_XO_NFC:	True}');
Error('\P{Is_XO_NFC:	True}');
Error('\p{Expands_On_NFD:	No}');
Error('\P{Expands_On_NFD:	No}');
Error('\p{XO_NFD=N}');
Error('\P{XO_NFD=N}');
Error('\p{Is_Expands_On_NFD=F}');
Error('\P{Is_Expands_On_NFD=F}');
Error('\p{Is_XO_NFD=False}');
Error('\P{Is_XO_NFD=False}');
Error('\p{Expands_On_NFD=Yes}');
Error('\P{Expands_On_NFD=Yes}');
Error('\p{XO_NFD=Y}');
Error('\P{XO_NFD=Y}');
Error('\p{Is_Expands_On_NFD=T}');
Error('\P{Is_Expands_On_NFD=T}');
Error('\p{Is_XO_NFD=True}');
Error('\P{Is_XO_NFD=True}');
Error('\p{Expands_On_NFKC: No}');
Error('\P{Expands_On_NFKC: No}');
Error('\p{XO_NFKC: N}');
Error('\P{XO_NFKC: N}');
Error('\p{Is_Expands_On_NFKC=F}');
Error('\P{Is_Expands_On_NFKC=F}');
Error('\p{Is_XO_NFKC=False}');
Error('\P{Is_XO_NFKC=False}');
Error('\p{Expands_On_NFKC:   Yes}');
Error('\P{Expands_On_NFKC:   Yes}');
Error('\p{XO_NFKC=Y}');
Error('\P{XO_NFKC=Y}');
Error('\p{Is_Expands_On_NFKC=T}');
Error('\P{Is_Expands_On_NFKC=T}');
Error('\p{Is_XO_NFKC=True}');
Error('\P{Is_XO_NFKC=True}');
Error('\p{Expands_On_NFKD=No}');
Error('\P{Expands_On_NFKD=No}');
Error('\p{XO_NFKD=N}');
Error('\P{XO_NFKD=N}');
Error('\p{Is_Expands_On_NFKD=F}');
Error('\P{Is_Expands_On_NFKD=F}');
Error('\p{Is_XO_NFKD=False}');
Error('\P{Is_XO_NFKD=False}');
Error('\p{Expands_On_NFKD:   Yes}');
Error('\P{Expands_On_NFKD:   Yes}');
Error('\p{XO_NFKD=Y}');
Error('\P{XO_NFKD=Y}');
Error('\p{Is_Expands_On_NFKD:   T}');
Error('\P{Is_Expands_On_NFKD:   T}');
Error('\p{Is_XO_NFKD=True}');
Error('\P{Is_XO_NFKD=True}');
Error('\p{Age= :=V1_1}');
Error('\P{Age= :=V1_1}');
Expect(1, 65533, '\p{Age=v11}', "");
Expect(0, 65533, '\p{^Age=v11}', "");
Expect(0, 65533, '\P{Age=v11}', "");
Expect(1, 65533, '\P{^Age=v11}', "");
Expect(0, 65536, '\p{Age=v11}', "");
Expect(1, 65536, '\p{^Age=v11}', "");
Expect(1, 65536, '\P{Age=v11}', "");
Expect(0, 65536, '\P{^Age=v11}', "");
Expect(1, 65533, '\p{Age=_V1_1}', "");
Expect(0, 65533, '\p{^Age=_V1_1}', "");
Expect(0, 65533, '\P{Age=_V1_1}', "");
Expect(1, 65533, '\P{^Age=_V1_1}', "");
Expect(0, 65536, '\p{Age=_V1_1}', "");
Expect(1, 65536, '\p{^Age=_V1_1}', "");
Expect(1, 65536, '\P{Age=_V1_1}', "");
Expect(0, 65536, '\P{^Age=_V1_1}', "");
Error('\p{Is_Age=:=__+0001.1}');
Error('\P{Is_Age=:=__+0001.1}');
Expect(1, 65533, '\p{Is_Age=+1.1}', "");
Expect(0, 65533, '\p{^Is_Age=+1.1}', "");
Expect(0, 65533, '\P{Is_Age=+1.1}', "");
Expect(1, 65533, '\P{^Is_Age=+1.1}', "");
Expect(0, 65536, '\p{Is_Age=+1.1}', "");
Expect(1, 65536, '\p{^Is_Age=+1.1}', "");
Expect(1, 65536, '\P{Is_Age=+1.1}', "");
Expect(0, 65536, '\P{^Is_Age=+1.1}', "");
Error('\p{Age=/a/V2_0}');
Error('\P{Age=/a/V2_0}');
Expect(1, 983040, '\p{Age=v20}', "");
Expect(0, 983040, '\p{^Age=v20}', "");
Expect(0, 983040, '\P{Age=v20}', "");
Expect(1, 983040, '\P{^Age=v20}', "");
Expect(0, 983037, '\p{Age=v20}', "");
Expect(1, 983037, '\p{^Age=v20}', "");
Expect(1, 983037, '\P{Age=v20}', "");
Expect(0, 983037, '\P{^Age=v20}', "");
Expect(1, 983040, '\p{Age=__V2_0}', "");
Expect(0, 983040, '\p{^Age=__V2_0}', "");
Expect(0, 983040, '\P{Age=__V2_0}', "");
Expect(1, 983040, '\P{^Age=__V2_0}', "");
Expect(0, 983037, '\p{Age=__V2_0}', "");
Expect(1, 983037, '\p{^Age=__V2_0}', "");
Expect(1, 983037, '\P{Age=__V2_0}', "");
Expect(0, 983037, '\P{^Age=__V2_0}', "");
Error('\p{Is_Age=-:=+2.0}');
Error('\P{Is_Age=-:=+2.0}');
Expect(1, 983040, '\p{Is_Age=+0002.0}', "");
Expect(0, 983040, '\p{^Is_Age=+0002.0}', "");
Expect(0, 983040, '\P{Is_Age=+0002.0}', "");
Expect(1, 983040, '\P{^Is_Age=+0002.0}', "");
Expect(0, 983037, '\p{Is_Age=+0002.0}', "");
Expect(1, 983037, '\p{^Is_Age=+0002.0}', "");
Expect(1, 983037, '\P{Is_Age=+0002.0}', "");
Expect(0, 983037, '\P{^Is_Age=+0002.0}', "");
Error('\p{Age=_v2_1/a/}');
Error('\P{Age=_v2_1/a/}');
Expect(1, 65532, '\p{Age=v21}', "");
Expect(0, 65532, '\p{^Age=v21}', "");
Expect(0, 65532, '\P{Age=v21}', "");
Expect(1, 65532, '\P{^Age=v21}', "");
Expect(0, 65533, '\p{Age=v21}', "");
Expect(1, 65533, '\p{^Age=v21}', "");
Expect(1, 65533, '\P{Age=v21}', "");
Expect(0, 65533, '\P{^Age=v21}', "");
Expect(1, 65532, '\p{Age=-V2_1}', "");
Expect(0, 65532, '\p{^Age=-V2_1}', "");
Expect(0, 65532, '\P{Age=-V2_1}', "");
Expect(1, 65532, '\P{^Age=-V2_1}', "");
Expect(0, 65533, '\p{Age=-V2_1}', "");
Expect(1, 65533, '\p{^Age=-V2_1}', "");
Expect(1, 65533, '\P{Age=-V2_1}', "");
Expect(0, 65533, '\P{^Age=-V2_1}', "");
Error('\p{Is_Age: __2.1/a/}');
Error('\P{Is_Age: __2.1/a/}');
Expect(1, 65532, '\p{Is_Age=02.1}', "");
Expect(0, 65532, '\p{^Is_Age=02.1}', "");
Expect(0, 65532, '\P{Is_Age=02.1}', "");
Expect(1, 65532, '\P{^Is_Age=02.1}', "");
Expect(0, 65533, '\p{Is_Age=02.1}', "");
Expect(1, 65533, '\p{^Is_Age=02.1}', "");
Expect(1, 65533, '\P{Is_Age=02.1}', "");
Expect(0, 65533, '\P{^Is_Age=02.1}', "");
Error('\p{Age=-:=v3_0}');
Error('\P{Age=-:=v3_0}');
Expect(1, 65531, '\p{Age=v30}', "");
Expect(0, 65531, '\p{^Age=v30}', "");
Expect(0, 65531, '\P{Age=v30}', "");
Expect(1, 65531, '\P{^Age=v30}', "");
Expect(0, 65532, '\p{Age=v30}', "");
Expect(1, 65532, '\p{^Age=v30}', "");
Expect(1, 65532, '\P{Age=v30}', "");
Expect(0, 65532, '\P{^Age=v30}', "");
Expect(1, 65531, '\p{Age: --v3_0}', "");
Expect(0, 65531, '\p{^Age: --v3_0}', "");
Expect(0, 65531, '\P{Age: --v3_0}', "");
Expect(1, 65531, '\P{^Age: --v3_0}', "");
Expect(0, 65532, '\p{Age: --v3_0}', "");
Expect(1, 65532, '\p{^Age: --v3_0}', "");
Expect(1, 65532, '\P{Age: --v3_0}', "");
Expect(0, 65532, '\P{^Age: --v3_0}', "");
Error('\p{Is_Age=_:=+3.0}');
Error('\P{Is_Age=_:=+3.0}');
Expect(1, 65531, '\p{Is_Age=0000003.0}', "");
Expect(0, 65531, '\p{^Is_Age=0000003.0}', "");
Expect(0, 65531, '\P{Is_Age=0000003.0}', "");
Expect(1, 65531, '\P{^Is_Age=0000003.0}', "");
Expect(0, 65532, '\p{Is_Age=0000003.0}', "");
Expect(1, 65532, '\p{^Is_Age=0000003.0}', "");
Expect(1, 65532, '\P{Is_Age=0000003.0}', "");
Expect(0, 65532, '\P{^Is_Age=0000003.0}', "");
Error('\p{Age=:=	 V3_1}');
Error('\P{Age=:=	 V3_1}');
Expect(1, 917631, '\p{Age:v31}', "");
Expect(0, 917631, '\p{^Age:v31}', "");
Expect(0, 917631, '\P{Age:v31}', "");
Expect(1, 917631, '\P{^Age:v31}', "");
Expect(0, 917632, '\p{Age:v31}', "");
Expect(1, 917632, '\p{^Age:v31}', "");
Expect(1, 917632, '\P{Age:v31}', "");
Expect(0, 917632, '\P{^Age:v31}', "");
Expect(1, 917631, '\p{Age=V3_1}', "");
Expect(0, 917631, '\p{^Age=V3_1}', "");
Expect(0, 917631, '\P{Age=V3_1}', "");
Expect(1, 917631, '\P{^Age=V3_1}', "");
Expect(0, 917632, '\p{Age=V3_1}', "");
Expect(1, 917632, '\p{^Age=V3_1}', "");
Expect(1, 917632, '\P{Age=V3_1}', "");
Expect(0, 917632, '\P{^Age=V3_1}', "");
Error('\p{Is_Age=/a/  000_3.1}');
Error('\P{Is_Age=/a/  000_3.1}');
Expect(1, 917631, '\p{Is_Age=0000003.1}', "");
Expect(0, 917631, '\p{^Is_Age=0000003.1}', "");
Expect(0, 917631, '\P{Is_Age=0000003.1}', "");
Expect(1, 917631, '\P{^Is_Age=0000003.1}', "");
Expect(0, 917632, '\p{Is_Age=0000003.1}', "");
Expect(1, 917632, '\p{^Is_Age=0000003.1}', "");
Expect(1, 917632, '\P{Is_Age=0000003.1}', "");
Expect(0, 917632, '\P{^Is_Age=0000003.1}', "");
Error('\p{Age=V3_2:=}');
Error('\P{Age=V3_2:=}');
Expect(1, 65376, '\p{Age: v32}', "");
Expect(0, 65376, '\p{^Age: v32}', "");
Expect(0, 65376, '\P{Age: v32}', "");
Expect(1, 65376, '\P{^Age: v32}', "");
Expect(0, 65377, '\p{Age: v32}', "");
Expect(1, 65377, '\p{^Age: v32}', "");
Expect(1, 65377, '\P{Age: v32}', "");
Expect(0, 65377, '\P{^Age: v32}', "");
Expect(1, 65376, '\p{Age=v3_2}', "");
Expect(0, 65376, '\p{^Age=v3_2}', "");
Expect(0, 65376, '\P{Age=v3_2}', "");
Expect(1, 65376, '\P{^Age=v3_2}', "");
Expect(0, 65377, '\p{Age=v3_2}', "");
Expect(1, 65377, '\p{^Age=v3_2}', "");
Expect(1, 65377, '\P{Age=v3_2}', "");
Expect(0, 65377, '\P{^Age=v3_2}', "");
Error('\p{Is_Age=_-3.2/a/}');
Error('\P{Is_Age=_-3.2/a/}');
Expect(1, 65376, '\p{Is_Age=+0000_3.2}', "");
Expect(0, 65376, '\p{^Is_Age=+0000_3.2}', "");
Expect(0, 65376, '\P{Is_Age=+0000_3.2}', "");
Expect(1, 65376, '\P{^Is_Age=+0000_3.2}', "");
Expect(0, 65377, '\p{Is_Age=+0000_3.2}', "");
Expect(1, 65377, '\p{^Is_Age=+0000_3.2}', "");
Expect(1, 65377, '\P{Is_Age=+0000_3.2}', "");
Expect(0, 65377, '\P{^Is_Age=+0000_3.2}', "");
Error('\p{Age=_	V4_0/a/}');
Error('\P{Age=_	V4_0/a/}');
Expect(1, 917999, '\p{Age:	v40}', "");
Expect(0, 917999, '\p{^Age:	v40}', "");
Expect(0, 917999, '\P{Age:	v40}', "");
Expect(1, 917999, '\P{^Age:	v40}', "");
Expect(0, 918000, '\p{Age:	v40}', "");
Expect(1, 918000, '\p{^Age:	v40}', "");
Expect(1, 918000, '\P{Age:	v40}', "");
Expect(0, 918000, '\P{^Age:	v40}', "");
Expect(1, 917999, '\p{Age=	V4_0}', "");
Expect(0, 917999, '\p{^Age=	V4_0}', "");
Expect(0, 917999, '\P{Age=	V4_0}', "");
Expect(1, 917999, '\P{^Age=	V4_0}', "");
Expect(0, 918000, '\p{Age=	V4_0}', "");
Expect(1, 918000, '\p{^Age=	V4_0}', "");
Expect(1, 918000, '\P{Age=	V4_0}', "");
Expect(0, 918000, '\P{^Age=	V4_0}', "");
Error('\p{Is_Age= 	+00000004.0/a/}');
Error('\P{Is_Age= 	+00000004.0/a/}');
Expect(1, 917999, '\p{Is_Age=00004.0}', "");
Expect(0, 917999, '\p{^Is_Age=00004.0}', "");
Expect(0, 917999, '\P{Is_Age=00004.0}', "");
Expect(1, 917999, '\P{^Is_Age=00004.0}', "");
Expect(0, 918000, '\p{Is_Age=00004.0}', "");
Expect(1, 918000, '\p{^Is_Age=00004.0}', "");
Expect(1, 918000, '\P{Is_Age=00004.0}', "");
Expect(0, 918000, '\P{^Is_Age=00004.0}', "");
Error('\p{Age=  V4_1:=}');
Error('\P{Age=  V4_1:=}');
Expect(1, 120485, '\p{Age=v41}', "");
Expect(0, 120485, '\p{^Age=v41}', "");
Expect(0, 120485, '\P{Age=v41}', "");
Expect(1, 120485, '\P{^Age=v41}', "");
Expect(0, 120486, '\p{Age=v41}', "");
Expect(1, 120486, '\p{^Age=v41}', "");
Expect(1, 120486, '\P{Age=v41}', "");
Expect(0, 120486, '\P{^Age=v41}', "");
Expect(1, 120485, '\p{Age:_	V4_1}', "");
Expect(0, 120485, '\p{^Age:_	V4_1}', "");
Expect(0, 120485, '\P{Age:_	V4_1}', "");
Expect(1, 120485, '\P{^Age:_	V4_1}', "");
Expect(0, 120486, '\p{Age:_	V4_1}', "");
Expect(1, 120486, '\p{^Age:_	V4_1}', "");
Expect(1, 120486, '\P{Age:_	V4_1}', "");
Expect(0, 120486, '\P{^Age:_	V4_1}', "");
Error('\p{Is_Age: :=_000000004.1}');
Error('\P{Is_Age: :=_000000004.1}');
Expect(1, 120485, '\p{Is_Age=+0000000004.1}', "");
Expect(0, 120485, '\p{^Is_Age=+0000000004.1}', "");
Expect(0, 120485, '\P{Is_Age=+0000000004.1}', "");
Expect(1, 120485, '\P{^Is_Age=+0000000004.1}', "");
Expect(0, 120486, '\p{Is_Age=+0000000004.1}', "");
Expect(1, 120486, '\p{^Is_Age=+0000000004.1}', "");
Expect(1, 120486, '\P{Is_Age=+0000000004.1}', "");
Expect(0, 120486, '\P{^Is_Age=+0000000004.1}', "");
Error('\p{Age::= V5_0}');
Error('\P{Age::= V5_0}');
Expect(1, 120779, '\p{Age=v50}', "");
Expect(0, 120779, '\p{^Age=v50}', "");
Expect(0, 120779, '\P{Age=v50}', "");
Expect(1, 120779, '\P{^Age=v50}', "");
Expect(0, 120780, '\p{Age=v50}', "");
Expect(1, 120780, '\p{^Age=v50}', "");
Expect(1, 120780, '\P{Age=v50}', "");
Expect(0, 120780, '\P{^Age=v50}', "");
Expect(1, 120779, '\p{Age=__v5_0}', "");
Expect(0, 120779, '\p{^Age=__v5_0}', "");
Expect(0, 120779, '\P{Age=__v5_0}', "");
Expect(1, 120779, '\P{^Age=__v5_0}', "");
Expect(0, 120780, '\p{Age=__v5_0}', "");
Expect(1, 120780, '\p{^Age=__v5_0}', "");
Expect(1, 120780, '\P{Age=__v5_0}', "");
Expect(0, 120780, '\P{^Age=__v5_0}', "");
Error('\p{Is_Age:	 /a/+00000_5.0}');
Error('\P{Is_Age:	 /a/+00000_5.0}');
Expect(1, 120779, '\p{Is_Age=000000005.0}', "");
Expect(0, 120779, '\p{^Is_Age=000000005.0}', "");
Expect(0, 120779, '\P{Is_Age=000000005.0}', "");
Expect(1, 120779, '\P{^Is_Age=000000005.0}', "");
Expect(0, 120780, '\p{Is_Age=000000005.0}', "");
Expect(1, 120780, '\p{^Is_Age=000000005.0}', "");
Expect(1, 120780, '\P{Is_Age=000000005.0}', "");
Expect(0, 120780, '\P{^Is_Age=000000005.0}', "");
Error('\p{Age: /a/v5_1}');
Error('\P{Age: /a/v5_1}');
Expect(1, 127123, '\p{Age=v51}', "");
Expect(0, 127123, '\p{^Age=v51}', "");
Expect(0, 127123, '\P{Age=v51}', "");
Expect(1, 127123, '\P{^Age=v51}', "");
Expect(0, 127124, '\p{Age=v51}', "");
Expect(1, 127124, '\p{^Age=v51}', "");
Expect(1, 127124, '\P{Age=v51}', "");
Expect(0, 127124, '\P{^Age=v51}', "");
Expect(1, 127123, '\p{Age=-	V5_1}', "");
Expect(0, 127123, '\p{^Age=-	V5_1}', "");
Expect(0, 127123, '\P{Age=-	V5_1}', "");
Expect(1, 127123, '\P{^Age=-	V5_1}', "");
Expect(0, 127124, '\p{Age=-	V5_1}', "");
Expect(1, 127124, '\p{^Age=-	V5_1}', "");
Expect(1, 127124, '\P{Age=-	V5_1}', "");
Expect(0, 127124, '\P{^Age=-	V5_1}', "");
Error('\p{Is_Age=:=0_0_0_05.1}');
Error('\P{Is_Age=:=0_0_0_05.1}');
Expect(1, 127123, '\p{Is_Age=0000000005.1}', "");
Expect(0, 127123, '\p{^Is_Age=0000000005.1}', "");
Expect(0, 127123, '\P{Is_Age=0000000005.1}', "");
Expect(1, 127123, '\P{^Is_Age=0000000005.1}', "");
Expect(0, 127124, '\p{Is_Age=0000000005.1}', "");
Expect(1, 127124, '\p{^Is_Age=0000000005.1}', "");
Expect(1, 127124, '\P{Is_Age=0000000005.1}', "");
Expect(0, 127124, '\P{^Is_Age=0000000005.1}', "");
Error('\p{Age=:=-_v5_2}');
Error('\P{Age=:=-_v5_2}');
Expect(1, 177972, '\p{Age=v52}', "");
Expect(0, 177972, '\p{^Age=v52}', "");
Expect(0, 177972, '\P{Age=v52}', "");
Expect(1, 177972, '\P{^Age=v52}', "");
Expect(0, 177973, '\p{Age=v52}', "");
Expect(1, 177973, '\p{^Age=v52}', "");
Expect(1, 177973, '\P{Age=v52}', "");
Expect(0, 177973, '\P{^Age=v52}', "");
Expect(1, 177972, '\p{Age=		V5_2}', "");
Expect(0, 177972, '\p{^Age=		V5_2}', "");
Expect(0, 177972, '\P{Age=		V5_2}', "");
Expect(1, 177972, '\P{^Age=		V5_2}', "");
Expect(0, 177973, '\p{Age=		V5_2}', "");
Expect(1, 177973, '\p{^Age=		V5_2}', "");
Expect(1, 177973, '\P{Age=		V5_2}', "");
Expect(0, 177973, '\P{^Age=		V5_2}', "");
Error('\p{Is_Age= 0_0_0_0_0_00005.2:=}');
Error('\P{Is_Age= 0_0_0_0_0_00005.2:=}');
Expect(1, 177972, '\p{Is_Age=0000000005.2}', "");
Expect(0, 177972, '\p{^Is_Age=0000000005.2}', "");
Expect(0, 177972, '\P{Is_Age=0000000005.2}', "");
Expect(1, 177972, '\P{^Is_Age=0000000005.2}', "");
Expect(0, 177973, '\p{Is_Age=0000000005.2}', "");
Expect(1, 177973, '\p{^Is_Age=0000000005.2}', "");
Expect(1, 177973, '\P{Is_Age=0000000005.2}', "");
Expect(0, 177973, '\P{^Is_Age=0000000005.2}', "");
Error('\p{Age= -V6_0:=}');
Error('\P{Age= -V6_0:=}');
Expect(1, 178205, '\p{Age: v60}', "");
Expect(0, 178205, '\p{^Age: v60}', "");
Expect(0, 178205, '\P{Age: v60}', "");
Expect(1, 178205, '\P{^Age: v60}', "");
Expect(0, 178206, '\p{Age: v60}', "");
Expect(1, 178206, '\p{^Age: v60}', "");
Expect(1, 178206, '\P{Age: v60}', "");
Expect(0, 178206, '\P{^Age: v60}', "");
Expect(1, 178205, '\p{Age=- V6_0}', "");
Expect(0, 178205, '\p{^Age=- V6_0}', "");
Expect(0, 178205, '\P{Age=- V6_0}', "");
Expect(1, 178205, '\P{^Age=- V6_0}', "");
Expect(0, 178206, '\p{Age=- V6_0}', "");
Expect(1, 178206, '\p{^Age=- V6_0}', "");
Expect(1, 178206, '\P{Age=- V6_0}', "");
Expect(0, 178206, '\P{^Age=- V6_0}', "");
Error('\p{Is_Age=:= _+0_0_0_06.0}');
Error('\P{Is_Age=:= _+0_0_0_06.0}');
Expect(1, 178205, '\p{Is_Age=0_0_0_0_0_006.0}', "");
Expect(0, 178205, '\p{^Is_Age=0_0_0_0_0_006.0}', "");
Expect(0, 178205, '\P{Is_Age=0_0_0_0_0_006.0}', "");
Expect(1, 178205, '\P{^Is_Age=0_0_0_0_0_006.0}', "");
Expect(0, 178206, '\p{Is_Age=0_0_0_0_0_006.0}', "");
Expect(1, 178206, '\p{^Is_Age=0_0_0_0_0_006.0}', "");
Expect(1, 178206, '\P{Is_Age=0_0_0_0_0_006.0}', "");
Expect(0, 178206, '\P{^Is_Age=0_0_0_0_0_006.0}', "");
Error('\p{Age=-V6_1/a/}');
Error('\P{Age=-V6_1/a/}');
Expect(1, 128564, '\p{Age:   v61}', "");
Expect(0, 128564, '\p{^Age:   v61}', "");
Expect(0, 128564, '\P{Age:   v61}', "");
Expect(1, 128564, '\P{^Age:   v61}', "");
Expect(0, 128565, '\p{Age:   v61}', "");
Expect(1, 128565, '\p{^Age:   v61}', "");
Expect(1, 128565, '\P{Age:   v61}', "");
Expect(0, 128565, '\P{^Age:   v61}', "");
Expect(1, 128564, '\p{Age=- v6_1}', "");
Expect(0, 128564, '\p{^Age=- v6_1}', "");
Expect(0, 128564, '\P{Age=- v6_1}', "");
Expect(1, 128564, '\P{^Age=- v6_1}', "");
Expect(0, 128565, '\p{Age=- v6_1}', "");
Expect(1, 128565, '\p{^Age=- v6_1}', "");
Expect(1, 128565, '\P{Age=- v6_1}', "");
Expect(0, 128565, '\P{^Age=- v6_1}', "");
Error('\p{Is_Age=_ +006.1:=}');
Error('\P{Is_Age=_ +006.1:=}');
Expect(1, 128564, '\p{Is_Age=0_0_0_0_06.1}', "");
Expect(0, 128564, '\p{^Is_Age=0_0_0_0_06.1}', "");
Expect(0, 128564, '\P{Is_Age=0_0_0_0_06.1}', "");
Expect(1, 128564, '\P{^Is_Age=0_0_0_0_06.1}', "");
Expect(0, 128565, '\p{Is_Age=0_0_0_0_06.1}', "");
Expect(1, 128565, '\p{^Is_Age=0_0_0_0_06.1}', "");
Expect(1, 128565, '\P{Is_Age=0_0_0_0_06.1}', "");
Expect(0, 128565, '\P{^Is_Age=0_0_0_0_06.1}', "");
Error('\p{Age=/a/	v6_2}');
Error('\P{Age=/a/	v6_2}');
Expect(1, 8378, '\p{Age=v62}', "");
Expect(0, 8378, '\p{^Age=v62}', "");
Expect(0, 8378, '\P{Age=v62}', "");
Expect(1, 8378, '\P{^Age=v62}', "");
Expect(0, 8379, '\p{Age=v62}', "");
Expect(1, 8379, '\p{^Age=v62}', "");
Expect(1, 8379, '\P{Age=v62}', "");
Expect(0, 8379, '\P{^Age=v62}', "");
Expect(1, 8378, '\p{Age:   _	V6_2}', "");
Expect(0, 8378, '\p{^Age:   _	V6_2}', "");
Expect(0, 8378, '\P{Age:   _	V6_2}', "");
Expect(1, 8378, '\P{^Age:   _	V6_2}', "");
Expect(0, 8379, '\p{Age:   _	V6_2}', "");
Expect(1, 8379, '\p{^Age:   _	V6_2}', "");
Expect(1, 8379, '\P{Age:   _	V6_2}', "");
Expect(0, 8379, '\P{^Age:   _	V6_2}', "");
Error('\p{Is_Age=-00000006.2/a/}');
Error('\P{Is_Age=-00000006.2/a/}');
Expect(1, 8378, '\p{Is_Age:   06.2}', "");
Expect(0, 8378, '\p{^Is_Age:   06.2}', "");
Expect(0, 8378, '\P{Is_Age:   06.2}', "");
Expect(1, 8378, '\P{^Is_Age:   06.2}', "");
Expect(0, 8379, '\p{Is_Age:   06.2}', "");
Expect(1, 8379, '\p{^Is_Age:   06.2}', "");
Expect(1, 8379, '\P{Is_Age:   06.2}', "");
Expect(0, 8379, '\P{^Is_Age:   06.2}', "");
Error('\p{Age=--V6_3/a/}');
Error('\P{Age=--V6_3/a/}');
Expect(1, 8297, '\p{Age=v63}', "");
Expect(0, 8297, '\p{^Age=v63}', "");
Expect(0, 8297, '\P{Age=v63}', "");
Expect(1, 8297, '\P{^Age=v63}', "");
Expect(0, 8298, '\p{Age=v63}', "");
Expect(1, 8298, '\p{^Age=v63}', "");
Expect(1, 8298, '\P{Age=v63}', "");
Expect(0, 8298, '\P{^Age=v63}', "");
Expect(1, 8297, '\p{Age=--V6_3}', "");
Expect(0, 8297, '\p{^Age=--V6_3}', "");
Expect(0, 8297, '\P{Age=--V6_3}', "");
Expect(1, 8297, '\P{^Age=--V6_3}', "");
Expect(0, 8298, '\p{Age=--V6_3}', "");
Expect(1, 8298, '\p{^Age=--V6_3}', "");
Expect(1, 8298, '\P{Age=--V6_3}', "");
Expect(0, 8298, '\P{^Age=--V6_3}', "");
Error('\p{Is_Age=:= 06.3}');
Error('\P{Is_Age=:= 06.3}');
Expect(1, 8297, '\p{Is_Age=+000006.3}', "");
Expect(0, 8297, '\p{^Is_Age=+000006.3}', "");
Expect(0, 8297, '\P{Is_Age=+000006.3}', "");
Expect(1, 8297, '\P{^Is_Age=+000006.3}', "");
Expect(0, 8298, '\p{Is_Age=+000006.3}', "");
Expect(1, 8298, '\p{^Is_Age=+000006.3}', "");
Expect(1, 8298, '\P{Is_Age=+000006.3}', "");
Expect(0, 8298, '\P{^Is_Age=+000006.3}', "");
Error('\p{Age=:=Unassigned}');
Error('\P{Age=:=Unassigned}');
Expect(1, 983037, '\p{Age=unassigned}', "");
Expect(0, 983037, '\p{^Age=unassigned}', "");
Expect(0, 983037, '\P{Age=unassigned}', "");
Expect(1, 983037, '\P{^Age=unassigned}', "");
Expect(0, 983040, '\p{Age=unassigned}', "");
Expect(1, 983040, '\p{^Age=unassigned}', "");
Expect(1, 983040, '\P{Age=unassigned}', "");
Expect(0, 983040, '\P{^Age=unassigned}', "");
Expect(1, 983037, '\p{Age=	Unassigned}', "");
Expect(0, 983037, '\p{^Age=	Unassigned}', "");
Expect(0, 983037, '\P{Age=	Unassigned}', "");
Expect(1, 983037, '\P{^Age=	Unassigned}', "");
Expect(0, 983040, '\p{Age=	Unassigned}', "");
Expect(1, 983040, '\p{^Age=	Unassigned}', "");
Expect(1, 983040, '\P{Age=	Unassigned}', "");
Expect(0, 983040, '\P{^Age=	Unassigned}', "");
Error('\p{Is_Age=	:=NA}');
Error('\P{Is_Age=	:=NA}');
Expect(1, 983037, '\p{Is_Age=na}', "");
Expect(0, 983037, '\p{^Is_Age=na}', "");
Expect(0, 983037, '\P{Is_Age=na}', "");
Expect(1, 983037, '\P{^Is_Age=na}', "");
Expect(0, 983040, '\p{Is_Age=na}', "");
Expect(1, 983040, '\p{^Is_Age=na}', "");
Expect(1, 983040, '\P{Is_Age=na}', "");
Expect(0, 983040, '\P{^Is_Age=na}', "");
Expect(1, 983037, '\p{Is_Age:   	NA}', "");
Expect(0, 983037, '\p{^Is_Age:   	NA}', "");
Expect(0, 983037, '\P{Is_Age:   	NA}', "");
Expect(1, 983037, '\P{^Is_Age:   	NA}', "");
Expect(0, 983040, '\p{Is_Age:   	NA}', "");
Expect(1, 983040, '\p{^Is_Age:   	NA}', "");
Expect(1, 983040, '\P{Is_Age:   	NA}', "");
Expect(0, 983040, '\P{^Is_Age:   	NA}', "");
Error('\p{Bidi_Class=:= arabic_Letter}');
Error('\P{Bidi_Class=:= arabic_Letter}');
Expect(1, 126719, '\p{Bidi_Class=arabicletter}', "");
Expect(0, 126719, '\p{^Bidi_Class=arabicletter}', "");
Expect(0, 126719, '\P{Bidi_Class=arabicletter}', "");
Expect(1, 126719, '\P{^Bidi_Class=arabicletter}', "");
Expect(0, 126720, '\p{Bidi_Class=arabicletter}', "");
Expect(1, 126720, '\p{^Bidi_Class=arabicletter}', "");
Expect(1, 126720, '\P{Bidi_Class=arabicletter}', "");
Expect(0, 126720, '\P{^Bidi_Class=arabicletter}', "");
Expect(1, 126719, '\p{Bidi_Class=__Arabic_letter}', "");
Expect(0, 126719, '\p{^Bidi_Class=__Arabic_letter}', "");
Expect(0, 126719, '\P{Bidi_Class=__Arabic_letter}', "");
Expect(1, 126719, '\P{^Bidi_Class=__Arabic_letter}', "");
Expect(0, 126720, '\p{Bidi_Class=__Arabic_letter}', "");
Expect(1, 126720, '\p{^Bidi_Class=__Arabic_letter}', "");
Expect(1, 126720, '\P{Bidi_Class=__Arabic_letter}', "");
Expect(0, 126720, '\P{^Bidi_Class=__Arabic_letter}', "");
Error('\p{Bc=:=	 AL}');
Error('\P{Bc=:=	 AL}');
Expect(1, 126719, '\p{Bc:   al}', "");
Expect(0, 126719, '\p{^Bc:   al}', "");
Expect(0, 126719, '\P{Bc:   al}', "");
Expect(1, 126719, '\P{^Bc:   al}', "");
Expect(0, 126720, '\p{Bc:   al}', "");
Expect(1, 126720, '\p{^Bc:   al}', "");
Expect(1, 126720, '\P{Bc:   al}', "");
Expect(0, 126720, '\P{^Bc:   al}', "");
Expect(1, 126719, '\p{Bc=__AL}', "");
Expect(0, 126719, '\p{^Bc=__AL}', "");
Expect(0, 126719, '\P{Bc=__AL}', "");
Expect(1, 126719, '\P{^Bc=__AL}', "");
Expect(0, 126720, '\p{Bc=__AL}', "");
Expect(1, 126720, '\p{^Bc=__AL}', "");
Expect(1, 126720, '\P{Bc=__AL}', "");
Expect(0, 126720, '\P{^Bc=__AL}', "");
Error('\p{Is_Bidi_Class=_/a/Arabic_Letter}');
Error('\P{Is_Bidi_Class=_/a/Arabic_Letter}');
Expect(1, 126719, '\p{Is_Bidi_Class=arabicletter}', "");
Expect(0, 126719, '\p{^Is_Bidi_Class=arabicletter}', "");
Expect(0, 126719, '\P{Is_Bidi_Class=arabicletter}', "");
Expect(1, 126719, '\P{^Is_Bidi_Class=arabicletter}', "");
Expect(0, 126720, '\p{Is_Bidi_Class=arabicletter}', "");
Expect(1, 126720, '\p{^Is_Bidi_Class=arabicletter}', "");
Expect(1, 126720, '\P{Is_Bidi_Class=arabicletter}', "");
Expect(0, 126720, '\P{^Is_Bidi_Class=arabicletter}', "");
Expect(1, 126719, '\p{Is_Bidi_Class=	_Arabic_Letter}', "");
Expect(0, 126719, '\p{^Is_Bidi_Class=	_Arabic_Letter}', "");
Expect(0, 126719, '\P{Is_Bidi_Class=	_Arabic_Letter}', "");
Expect(1, 126719, '\P{^Is_Bidi_Class=	_Arabic_Letter}', "");
Expect(0, 126720, '\p{Is_Bidi_Class=	_Arabic_Letter}', "");
Expect(1, 126720, '\p{^Is_Bidi_Class=	_Arabic_Letter}', "");
Expect(1, 126720, '\P{Is_Bidi_Class=	_Arabic_Letter}', "");
Expect(0, 126720, '\P{^Is_Bidi_Class=	_Arabic_Letter}', "");
Error('\p{Is_Bc= /a/AL}');
Error('\P{Is_Bc= /a/AL}');
Expect(1, 126719, '\p{Is_Bc=al}', "");
Expect(0, 126719, '\p{^Is_Bc=al}', "");
Expect(0, 126719, '\P{Is_Bc=al}', "");
Expect(1, 126719, '\P{^Is_Bc=al}', "");
Expect(0, 126720, '\p{Is_Bc=al}', "");
Expect(1, 126720, '\p{^Is_Bc=al}', "");
Expect(1, 126720, '\P{Is_Bc=al}', "");
Expect(0, 126720, '\P{^Is_Bc=al}', "");
Expect(1, 126719, '\p{Is_Bc=_-AL}', "");
Expect(0, 126719, '\p{^Is_Bc=_-AL}', "");
Expect(0, 126719, '\P{Is_Bc=_-AL}', "");
Expect(1, 126719, '\P{^Is_Bc=_-AL}', "");
Expect(0, 126720, '\p{Is_Bc=_-AL}', "");
Expect(1, 126720, '\p{^Is_Bc=_-AL}', "");
Expect(1, 126720, '\P{Is_Bc=_-AL}', "");
Expect(0, 126720, '\P{^Is_Bc=_-AL}', "");
Error('\p{Bidi_Class=-:=Arabic_number}');
Error('\P{Bidi_Class=-:=Arabic_number}');
Expect(1, 69246, '\p{Bidi_Class=arabicnumber}', "");
Expect(0, 69246, '\p{^Bidi_Class=arabicnumber}', "");
Expect(0, 69246, '\P{Bidi_Class=arabicnumber}', "");
Expect(1, 69246, '\P{^Bidi_Class=arabicnumber}', "");
Expect(0, 69247, '\p{Bidi_Class=arabicnumber}', "");
Expect(1, 69247, '\p{^Bidi_Class=arabicnumber}', "");
Expect(1, 69247, '\P{Bidi_Class=arabicnumber}', "");
Expect(0, 69247, '\P{^Bidi_Class=arabicnumber}', "");
Expect(1, 69246, '\p{Bidi_Class= _arabic_number}', "");
Expect(0, 69246, '\p{^Bidi_Class= _arabic_number}', "");
Expect(0, 69246, '\P{Bidi_Class= _arabic_number}', "");
Expect(1, 69246, '\P{^Bidi_Class= _arabic_number}', "");
Expect(0, 69247, '\p{Bidi_Class= _arabic_number}', "");
Expect(1, 69247, '\p{^Bidi_Class= _arabic_number}', "");
Expect(1, 69247, '\P{Bidi_Class= _arabic_number}', "");
Expect(0, 69247, '\P{^Bidi_Class= _arabic_number}', "");
Error('\p{Bc=	/a/AN}');
Error('\P{Bc=	/a/AN}');
Expect(1, 69246, '\p{Bc=an}', "");
Expect(0, 69246, '\p{^Bc=an}', "");
Expect(0, 69246, '\P{Bc=an}', "");
Expect(1, 69246, '\P{^Bc=an}', "");
Expect(0, 69247, '\p{Bc=an}', "");
Expect(1, 69247, '\p{^Bc=an}', "");
Expect(1, 69247, '\P{Bc=an}', "");
Expect(0, 69247, '\P{^Bc=an}', "");
Expect(1, 69246, '\p{Bc=--AN}', "");
Expect(0, 69246, '\p{^Bc=--AN}', "");
Expect(0, 69246, '\P{Bc=--AN}', "");
Expect(1, 69246, '\P{^Bc=--AN}', "");
Expect(0, 69247, '\p{Bc=--AN}', "");
Expect(1, 69247, '\p{^Bc=--AN}', "");
Expect(1, 69247, '\P{Bc=--AN}', "");
Expect(0, 69247, '\P{^Bc=--AN}', "");
Error('\p{Is_Bidi_Class=:= ARABIC_NUMBER}');
Error('\P{Is_Bidi_Class=:= ARABIC_NUMBER}');
Expect(1, 69246, '\p{Is_Bidi_Class=arabicnumber}', "");
Expect(0, 69246, '\p{^Is_Bidi_Class=arabicnumber}', "");
Expect(0, 69246, '\P{Is_Bidi_Class=arabicnumber}', "");
Expect(1, 69246, '\P{^Is_Bidi_Class=arabicnumber}', "");
Expect(0, 69247, '\p{Is_Bidi_Class=arabicnumber}', "");
Expect(1, 69247, '\p{^Is_Bidi_Class=arabicnumber}', "");
Expect(1, 69247, '\P{Is_Bidi_Class=arabicnumber}', "");
Expect(0, 69247, '\P{^Is_Bidi_Class=arabicnumber}', "");
Expect(1, 69246, '\p{Is_Bidi_Class= _ARABIC_NUMBER}', "");
Expect(0, 69246, '\p{^Is_Bidi_Class= _ARABIC_NUMBER}', "");
Expect(0, 69246, '\P{Is_Bidi_Class= _ARABIC_NUMBER}', "");
Expect(1, 69246, '\P{^Is_Bidi_Class= _ARABIC_NUMBER}', "");
Expect(0, 69247, '\p{Is_Bidi_Class= _ARABIC_NUMBER}', "");
Expect(1, 69247, '\p{^Is_Bidi_Class= _ARABIC_NUMBER}', "");
Expect(1, 69247, '\P{Is_Bidi_Class= _ARABIC_NUMBER}', "");
Expect(0, 69247, '\P{^Is_Bidi_Class= _ARABIC_NUMBER}', "");
Error('\p{Is_Bc:/a/_an}');
Error('\P{Is_Bc:/a/_an}');
Expect(1, 69246, '\p{Is_Bc=an}', "");
Expect(0, 69246, '\p{^Is_Bc=an}', "");
Expect(0, 69246, '\P{Is_Bc=an}', "");
Expect(1, 69246, '\P{^Is_Bc=an}', "");
Expect(0, 69247, '\p{Is_Bc=an}', "");
Expect(1, 69247, '\p{^Is_Bc=an}', "");
Expect(1, 69247, '\P{Is_Bc=an}', "");
Expect(0, 69247, '\P{^Is_Bc=an}', "");
Expect(1, 69246, '\p{Is_Bc=_	AN}', "");
Expect(0, 69246, '\p{^Is_Bc=_	AN}', "");
Expect(0, 69246, '\P{Is_Bc=_	AN}', "");
Expect(1, 69246, '\P{^Is_Bc=_	AN}', "");
Expect(0, 69247, '\p{Is_Bc=_	AN}', "");
Expect(1, 69247, '\p{^Is_Bc=_	AN}', "");
Expect(1, 69247, '\P{Is_Bc=_	AN}', "");
Expect(0, 69247, '\P{^Is_Bc=_	AN}', "");
Error('\p{Bidi_Class= :=Paragraph_SEPARATOR}');
Error('\P{Bidi_Class= :=Paragraph_SEPARATOR}');
Expect(1, 8233, '\p{Bidi_Class=paragraphseparator}', "");
Expect(0, 8233, '\p{^Bidi_Class=paragraphseparator}', "");
Expect(0, 8233, '\P{Bidi_Class=paragraphseparator}', "");
Expect(1, 8233, '\P{^Bidi_Class=paragraphseparator}', "");
Expect(0, 8234, '\p{Bidi_Class=paragraphseparator}', "");
Expect(1, 8234, '\p{^Bidi_Class=paragraphseparator}', "");
Expect(1, 8234, '\P{Bidi_Class=paragraphseparator}', "");
Expect(0, 8234, '\P{^Bidi_Class=paragraphseparator}', "");
Expect(1, 8233, '\p{Bidi_Class=-	Paragraph_Separator}', "");
Expect(0, 8233, '\p{^Bidi_Class=-	Paragraph_Separator}', "");
Expect(0, 8233, '\P{Bidi_Class=-	Paragraph_Separator}', "");
Expect(1, 8233, '\P{^Bidi_Class=-	Paragraph_Separator}', "");
Expect(0, 8234, '\p{Bidi_Class=-	Paragraph_Separator}', "");
Expect(1, 8234, '\p{^Bidi_Class=-	Paragraph_Separator}', "");
Expect(1, 8234, '\P{Bidi_Class=-	Paragraph_Separator}', "");
Expect(0, 8234, '\P{^Bidi_Class=-	Paragraph_Separator}', "");
Error('\p{Bc=	:=B}');
Error('\P{Bc=	:=B}');
Expect(1, 8233, '\p{Bc:	b}', "");
Expect(0, 8233, '\p{^Bc:	b}', "");
Expect(0, 8233, '\P{Bc:	b}', "");
Expect(1, 8233, '\P{^Bc:	b}', "");
Expect(0, 8234, '\p{Bc:	b}', "");
Expect(1, 8234, '\p{^Bc:	b}', "");
Expect(1, 8234, '\P{Bc:	b}', "");
Expect(0, 8234, '\P{^Bc:	b}', "");
Expect(1, 8233, '\p{Bc=	 B}', "");
Expect(0, 8233, '\p{^Bc=	 B}', "");
Expect(0, 8233, '\P{Bc=	 B}', "");
Expect(1, 8233, '\P{^Bc=	 B}', "");
Expect(0, 8234, '\p{Bc=	 B}', "");
Expect(1, 8234, '\p{^Bc=	 B}', "");
Expect(1, 8234, '\P{Bc=	 B}', "");
Expect(0, 8234, '\P{^Bc=	 B}', "");
Error('\p{Is_Bidi_Class=/a/-	PARAGRAPH_separator}');
Error('\P{Is_Bidi_Class=/a/-	PARAGRAPH_separator}');
Expect(1, 8233, '\p{Is_Bidi_Class=paragraphseparator}', "");
Expect(0, 8233, '\p{^Is_Bidi_Class=paragraphseparator}', "");
Expect(0, 8233, '\P{Is_Bidi_Class=paragraphseparator}', "");
Expect(1, 8233, '\P{^Is_Bidi_Class=paragraphseparator}', "");
Expect(0, 8234, '\p{Is_Bidi_Class=paragraphseparator}', "");
Expect(1, 8234, '\p{^Is_Bidi_Class=paragraphseparator}', "");
Expect(1, 8234, '\P{Is_Bidi_Class=paragraphseparator}', "");
Expect(0, 8234, '\P{^Is_Bidi_Class=paragraphseparator}', "");
Expect(1, 8233, '\p{Is_Bidi_Class:  	Paragraph_Separator}', "");
Expect(0, 8233, '\p{^Is_Bidi_Class:  	Paragraph_Separator}', "");
Expect(0, 8233, '\P{Is_Bidi_Class:  	Paragraph_Separator}', "");
Expect(1, 8233, '\P{^Is_Bidi_Class:  	Paragraph_Separator}', "");
Expect(0, 8234, '\p{Is_Bidi_Class:  	Paragraph_Separator}', "");
Expect(1, 8234, '\p{^Is_Bidi_Class:  	Paragraph_Separator}', "");
Expect(1, 8234, '\P{Is_Bidi_Class:  	Paragraph_Separator}', "");
Expect(0, 8234, '\P{^Is_Bidi_Class:  	Paragraph_Separator}', "");
Error('\p{Is_Bc=_:=b}');
Error('\P{Is_Bc=_:=b}');
Expect(1, 8233, '\p{Is_Bc:b}', "");
Expect(0, 8233, '\p{^Is_Bc:b}', "");
Expect(0, 8233, '\P{Is_Bc:b}', "");
Expect(1, 8233, '\P{^Is_Bc:b}', "");
Expect(0, 8234, '\p{Is_Bc:b}', "");
Expect(1, 8234, '\p{^Is_Bc:b}', "");
Expect(1, 8234, '\P{Is_Bc:b}', "");
Expect(0, 8234, '\P{^Is_Bc:b}', "");
Expect(1, 8233, '\p{Is_Bc:   B}', "");
Expect(0, 8233, '\p{^Is_Bc:   B}', "");
Expect(0, 8233, '\P{Is_Bc:   B}', "");
Expect(1, 8233, '\P{^Is_Bc:   B}', "");
Expect(0, 8234, '\p{Is_Bc:   B}', "");
Expect(1, 8234, '\p{^Is_Bc:   B}', "");
Expect(1, 8234, '\P{Is_Bc:   B}', "");
Expect(0, 8234, '\P{^Is_Bc:   B}', "");
Error('\p{Bidi_Class=:=	boundary_NEUTRAL}');
Error('\P{Bidi_Class=:=	boundary_NEUTRAL}');
Expect(1, 921599, '\p{Bidi_Class=boundaryneutral}', "");
Expect(0, 921599, '\p{^Bidi_Class=boundaryneutral}', "");
Expect(0, 921599, '\P{Bidi_Class=boundaryneutral}', "");
Expect(1, 921599, '\P{^Bidi_Class=boundaryneutral}', "");
Expect(0, 1114109, '\p{Bidi_Class=boundaryneutral}', "");
Expect(1, 1114109, '\p{^Bidi_Class=boundaryneutral}', "");
Expect(1, 1114109, '\P{Bidi_Class=boundaryneutral}', "");
Expect(0, 1114109, '\P{^Bidi_Class=boundaryneutral}', "");
Expect(1, 921599, '\p{Bidi_Class= _boundary_Neutral}', "");
Expect(0, 921599, '\p{^Bidi_Class= _boundary_Neutral}', "");
Expect(0, 921599, '\P{Bidi_Class= _boundary_Neutral}', "");
Expect(1, 921599, '\P{^Bidi_Class= _boundary_Neutral}', "");
Expect(0, 1114109, '\p{Bidi_Class= _boundary_Neutral}', "");
Expect(1, 1114109, '\p{^Bidi_Class= _boundary_Neutral}', "");
Expect(1, 1114109, '\P{Bidi_Class= _boundary_Neutral}', "");
Expect(0, 1114109, '\P{^Bidi_Class= _boundary_Neutral}', "");
Error('\p{Bc::=_BN}');
Error('\P{Bc::=_BN}');
Expect(1, 921599, '\p{Bc=bn}', "");
Expect(0, 921599, '\p{^Bc=bn}', "");
Expect(0, 921599, '\P{Bc=bn}', "");
Expect(1, 921599, '\P{^Bc=bn}', "");
Expect(0, 1114109, '\p{Bc=bn}', "");
Expect(1, 1114109, '\p{^Bc=bn}', "");
Expect(1, 1114109, '\P{Bc=bn}', "");
Expect(0, 1114109, '\P{^Bc=bn}', "");
Expect(1, 921599, '\p{Bc= _BN}', "");
Expect(0, 921599, '\p{^Bc= _BN}', "");
Expect(0, 921599, '\P{Bc= _BN}', "");
Expect(1, 921599, '\P{^Bc= _BN}', "");
Expect(0, 1114109, '\p{Bc= _BN}', "");
Expect(1, 1114109, '\p{^Bc= _BN}', "");
Expect(1, 1114109, '\P{Bc= _BN}', "");
Expect(0, 1114109, '\P{^Bc= _BN}', "");
Error('\p{Is_Bidi_Class=_ Boundary_NEUTRAL/a/}');
Error('\P{Is_Bidi_Class=_ Boundary_NEUTRAL/a/}');
Expect(1, 921599, '\p{Is_Bidi_Class=boundaryneutral}', "");
Expect(0, 921599, '\p{^Is_Bidi_Class=boundaryneutral}', "");
Expect(0, 921599, '\P{Is_Bidi_Class=boundaryneutral}', "");
Expect(1, 921599, '\P{^Is_Bidi_Class=boundaryneutral}', "");
Expect(0, 1114109, '\p{Is_Bidi_Class=boundaryneutral}', "");
Expect(1, 1114109, '\p{^Is_Bidi_Class=boundaryneutral}', "");
Expect(1, 1114109, '\P{Is_Bidi_Class=boundaryneutral}', "");
Expect(0, 1114109, '\P{^Is_Bidi_Class=boundaryneutral}', "");
Expect(1, 921599, '\p{Is_Bidi_Class=	boundary_neutral}', "");
Expect(0, 921599, '\p{^Is_Bidi_Class=	boundary_neutral}', "");
Expect(0, 921599, '\P{Is_Bidi_Class=	boundary_neutral}', "");
Expect(1, 921599, '\P{^Is_Bidi_Class=	boundary_neutral}', "");
Expect(0, 1114109, '\p{Is_Bidi_Class=	boundary_neutral}', "");
Expect(1, 1114109, '\p{^Is_Bidi_Class=	boundary_neutral}', "");
Expect(1, 1114109, '\P{Is_Bidi_Class=	boundary_neutral}', "");
Expect(0, 1114109, '\P{^Is_Bidi_Class=	boundary_neutral}', "");
Error('\p{Is_Bc=-BN/a/}');
Error('\P{Is_Bc=-BN/a/}');
Expect(1, 921599, '\p{Is_Bc=bn}', "");
Expect(0, 921599, '\p{^Is_Bc=bn}', "");
Expect(0, 921599, '\P{Is_Bc=bn}', "");
Expect(1, 921599, '\P{^Is_Bc=bn}', "");
Expect(0, 1114109, '\p{Is_Bc=bn}', "");
Expect(1, 1114109, '\p{^Is_Bc=bn}', "");
Expect(1, 1114109, '\P{Is_Bc=bn}', "");
Expect(0, 1114109, '\P{^Is_Bc=bn}', "");
Expect(1, 921599, '\p{Is_Bc= -BN}', "");
Expect(0, 921599, '\p{^Is_Bc= -BN}', "");
Expect(0, 921599, '\P{Is_Bc= -BN}', "");
Expect(1, 921599, '\P{^Is_Bc= -BN}', "");
Expect(0, 1114109, '\p{Is_Bc= -BN}', "");
Expect(1, 1114109, '\p{^Is_Bc= -BN}', "");
Expect(1, 1114109, '\P{Is_Bc= -BN}', "");
Expect(0, 1114109, '\P{^Is_Bc= -BN}', "");
Error('\p{Bidi_Class=_:=Common_Separator}');
Error('\P{Bidi_Class=_:=Common_Separator}');
Expect(1, 65306, '\p{Bidi_Class=commonseparator}', "");
Expect(0, 65306, '\p{^Bidi_Class=commonseparator}', "");
Expect(0, 65306, '\P{Bidi_Class=commonseparator}', "");
Expect(1, 65306, '\P{^Bidi_Class=commonseparator}', "");
Expect(0, 65307, '\p{Bidi_Class=commonseparator}', "");
Expect(1, 65307, '\p{^Bidi_Class=commonseparator}', "");
Expect(1, 65307, '\P{Bidi_Class=commonseparator}', "");
Expect(0, 65307, '\P{^Bidi_Class=commonseparator}', "");
Expect(1, 65306, '\p{Bidi_Class=_COMMON_Separator}', "");
Expect(0, 65306, '\p{^Bidi_Class=_COMMON_Separator}', "");
Expect(0, 65306, '\P{Bidi_Class=_COMMON_Separator}', "");
Expect(1, 65306, '\P{^Bidi_Class=_COMMON_Separator}', "");
Expect(0, 65307, '\p{Bidi_Class=_COMMON_Separator}', "");
Expect(1, 65307, '\p{^Bidi_Class=_COMMON_Separator}', "");
Expect(1, 65307, '\P{Bidi_Class=_COMMON_Separator}', "");
Expect(0, 65307, '\P{^Bidi_Class=_COMMON_Separator}', "");
Error('\p{Bc= 	cs/a/}');
Error('\P{Bc= 	cs/a/}');
Expect(1, 65306, '\p{Bc=cs}', "");
Expect(0, 65306, '\p{^Bc=cs}', "");
Expect(0, 65306, '\P{Bc=cs}', "");
Expect(1, 65306, '\P{^Bc=cs}', "");
Expect(0, 65307, '\p{Bc=cs}', "");
Expect(1, 65307, '\p{^Bc=cs}', "");
Expect(1, 65307, '\P{Bc=cs}', "");
Expect(0, 65307, '\P{^Bc=cs}', "");
Expect(1, 65306, '\p{Bc=_cs}', "");
Expect(0, 65306, '\p{^Bc=_cs}', "");
Expect(0, 65306, '\P{Bc=_cs}', "");
Expect(1, 65306, '\P{^Bc=_cs}', "");
Expect(0, 65307, '\p{Bc=_cs}', "");
Expect(1, 65307, '\p{^Bc=_cs}', "");
Expect(1, 65307, '\P{Bc=_cs}', "");
Expect(0, 65307, '\P{^Bc=_cs}', "");
Error('\p{Is_Bidi_Class=/a/-	common_Separator}');
Error('\P{Is_Bidi_Class=/a/-	common_Separator}');
Expect(1, 65306, '\p{Is_Bidi_Class=commonseparator}', "");
Expect(0, 65306, '\p{^Is_Bidi_Class=commonseparator}', "");
Expect(0, 65306, '\P{Is_Bidi_Class=commonseparator}', "");
Expect(1, 65306, '\P{^Is_Bidi_Class=commonseparator}', "");
Expect(0, 65307, '\p{Is_Bidi_Class=commonseparator}', "");
Expect(1, 65307, '\p{^Is_Bidi_Class=commonseparator}', "");
Expect(1, 65307, '\P{Is_Bidi_Class=commonseparator}', "");
Expect(0, 65307, '\P{^Is_Bidi_Class=commonseparator}', "");
Expect(1, 65306, '\p{Is_Bidi_Class=_ COMMON_SEPARATOR}', "");
Expect(0, 65306, '\p{^Is_Bidi_Class=_ COMMON_SEPARATOR}', "");
Expect(0, 65306, '\P{Is_Bidi_Class=_ COMMON_SEPARATOR}', "");
Expect(1, 65306, '\P{^Is_Bidi_Class=_ COMMON_SEPARATOR}', "");
Expect(0, 65307, '\p{Is_Bidi_Class=_ COMMON_SEPARATOR}', "");
Expect(1, 65307, '\p{^Is_Bidi_Class=_ COMMON_SEPARATOR}', "");
Expect(1, 65307, '\P{Is_Bidi_Class=_ COMMON_SEPARATOR}', "");
Expect(0, 65307, '\P{^Is_Bidi_Class=_ COMMON_SEPARATOR}', "");
Error('\p{Is_Bc= :=CS}');
Error('\P{Is_Bc= :=CS}');
Expect(1, 65306, '\p{Is_Bc=cs}', "");
Expect(0, 65306, '\p{^Is_Bc=cs}', "");
Expect(0, 65306, '\P{Is_Bc=cs}', "");
Expect(1, 65306, '\P{^Is_Bc=cs}', "");
Expect(0, 65307, '\p{Is_Bc=cs}', "");
Expect(1, 65307, '\p{^Is_Bc=cs}', "");
Expect(1, 65307, '\P{Is_Bc=cs}', "");
Expect(0, 65307, '\P{^Is_Bc=cs}', "");
Expect(1, 65306, '\p{Is_Bc= 	CS}', "");
Expect(0, 65306, '\p{^Is_Bc= 	CS}', "");
Expect(0, 65306, '\P{Is_Bc= 	CS}', "");
Expect(1, 65306, '\P{^Is_Bc= 	CS}', "");
Expect(0, 65307, '\p{Is_Bc= 	CS}', "");
Expect(1, 65307, '\p{^Is_Bc= 	CS}', "");
Expect(1, 65307, '\P{Is_Bc= 	CS}', "");
Expect(0, 65307, '\P{^Is_Bc= 	CS}', "");
Error('\p{Bidi_Class=	 european_number:=}');
Error('\P{Bidi_Class=	 european_number:=}');
Expect(1, 127242, '\p{Bidi_Class=europeannumber}', "");
Expect(0, 127242, '\p{^Bidi_Class=europeannumber}', "");
Expect(0, 127242, '\P{Bidi_Class=europeannumber}', "");
Expect(1, 127242, '\P{^Bidi_Class=europeannumber}', "");
Expect(0, 127243, '\p{Bidi_Class=europeannumber}', "");
Expect(1, 127243, '\p{^Bidi_Class=europeannumber}', "");
Expect(1, 127243, '\P{Bidi_Class=europeannumber}', "");
Expect(0, 127243, '\P{^Bidi_Class=europeannumber}', "");
Expect(1, 127242, '\p{Bidi_Class=European_number}', "");
Expect(0, 127242, '\p{^Bidi_Class=European_number}', "");
Expect(0, 127242, '\P{Bidi_Class=European_number}', "");
Expect(1, 127242, '\P{^Bidi_Class=European_number}', "");
Expect(0, 127243, '\p{Bidi_Class=European_number}', "");
Expect(1, 127243, '\p{^Bidi_Class=European_number}', "");
Expect(1, 127243, '\P{Bidi_Class=European_number}', "");
Expect(0, 127243, '\P{^Bidi_Class=European_number}', "");
Error('\p{Bc=/a/  en}');
Error('\P{Bc=/a/  en}');
Expect(1, 127242, '\p{Bc=en}', "");
Expect(0, 127242, '\p{^Bc=en}', "");
Expect(0, 127242, '\P{Bc=en}', "");
Expect(1, 127242, '\P{^Bc=en}', "");
Expect(0, 127243, '\p{Bc=en}', "");
Expect(1, 127243, '\p{^Bc=en}', "");
Expect(1, 127243, '\P{Bc=en}', "");
Expect(0, 127243, '\P{^Bc=en}', "");
Expect(1, 127242, '\p{Bc=		EN}', "");
Expect(0, 127242, '\p{^Bc=		EN}', "");
Expect(0, 127242, '\P{Bc=		EN}', "");
Expect(1, 127242, '\P{^Bc=		EN}', "");
Expect(0, 127243, '\p{Bc=		EN}', "");
Expect(1, 127243, '\p{^Bc=		EN}', "");
Expect(1, 127243, '\P{Bc=		EN}', "");
Expect(0, 127243, '\P{^Bc=		EN}', "");
Error('\p{Is_Bidi_Class=:=EUROPEAN_NUMBER}');
Error('\P{Is_Bidi_Class=:=EUROPEAN_NUMBER}');
Expect(1, 127242, '\p{Is_Bidi_Class:   europeannumber}', "");
Expect(0, 127242, '\p{^Is_Bidi_Class:   europeannumber}', "");
Expect(0, 127242, '\P{Is_Bidi_Class:   europeannumber}', "");
Expect(1, 127242, '\P{^Is_Bidi_Class:   europeannumber}', "");
Expect(0, 127243, '\p{Is_Bidi_Class:   europeannumber}', "");
Expect(1, 127243, '\p{^Is_Bidi_Class:   europeannumber}', "");
Expect(1, 127243, '\P{Is_Bidi_Class:   europeannumber}', "");
Expect(0, 127243, '\P{^Is_Bidi_Class:   europeannumber}', "");
Expect(1, 127242, '\p{Is_Bidi_Class=_	European_Number}', "");
Expect(0, 127242, '\p{^Is_Bidi_Class=_	European_Number}', "");
Expect(0, 127242, '\P{Is_Bidi_Class=_	European_Number}', "");
Expect(1, 127242, '\P{^Is_Bidi_Class=_	European_Number}', "");
Expect(0, 127243, '\p{Is_Bidi_Class=_	European_Number}', "");
Expect(1, 127243, '\p{^Is_Bidi_Class=_	European_Number}', "");
Expect(1, 127243, '\P{Is_Bidi_Class=_	European_Number}', "");
Expect(0, 127243, '\P{^Is_Bidi_Class=_	European_Number}', "");
Error('\p{Is_Bc=_:=EN}');
Error('\P{Is_Bc=_:=EN}');
Expect(1, 127242, '\p{Is_Bc=en}', "");
Expect(0, 127242, '\p{^Is_Bc=en}', "");
Expect(0, 127242, '\P{Is_Bc=en}', "");
Expect(1, 127242, '\P{^Is_Bc=en}', "");
Expect(0, 127243, '\p{Is_Bc=en}', "");
Expect(1, 127243, '\p{^Is_Bc=en}', "");
Expect(1, 127243, '\P{Is_Bc=en}', "");
Expect(0, 127243, '\P{^Is_Bc=en}', "");
Expect(1, 127242, '\p{Is_Bc=_en}', "");
Expect(0, 127242, '\p{^Is_Bc=_en}', "");
Expect(0, 127242, '\P{Is_Bc=_en}', "");
Expect(1, 127242, '\P{^Is_Bc=_en}', "");
Expect(0, 127243, '\p{Is_Bc=_en}', "");
Expect(1, 127243, '\p{^Is_Bc=_en}', "");
Expect(1, 127243, '\P{Is_Bc=_en}', "");
Expect(0, 127243, '\P{^Is_Bc=_en}', "");
Error('\p{Bidi_Class= :=European_SEPARATOR}');
Error('\P{Bidi_Class= :=European_SEPARATOR}');
Expect(1, 65293, '\p{Bidi_Class:   europeanseparator}', "");
Expect(0, 65293, '\p{^Bidi_Class:   europeanseparator}', "");
Expect(0, 65293, '\P{Bidi_Class:   europeanseparator}', "");
Expect(1, 65293, '\P{^Bidi_Class:   europeanseparator}', "");
Expect(0, 65294, '\p{Bidi_Class:   europeanseparator}', "");
Expect(1, 65294, '\p{^Bidi_Class:   europeanseparator}', "");
Expect(1, 65294, '\P{Bidi_Class:   europeanseparator}', "");
Expect(0, 65294, '\P{^Bidi_Class:   europeanseparator}', "");
Expect(1, 65293, '\p{Bidi_Class= -European_Separator}', "");
Expect(0, 65293, '\p{^Bidi_Class= -European_Separator}', "");
Expect(0, 65293, '\P{Bidi_Class= -European_Separator}', "");
Expect(1, 65293, '\P{^Bidi_Class= -European_Separator}', "");
Expect(0, 65294, '\p{Bidi_Class= -European_Separator}', "");
Expect(1, 65294, '\p{^Bidi_Class= -European_Separator}', "");
Expect(1, 65294, '\P{Bidi_Class= -European_Separator}', "");
Expect(0, 65294, '\P{^Bidi_Class= -European_Separator}', "");
Error('\p{Bc=:= ES}');
Error('\P{Bc=:= ES}');
Expect(1, 65293, '\p{Bc:es}', "");
Expect(0, 65293, '\p{^Bc:es}', "");
Expect(0, 65293, '\P{Bc:es}', "");
Expect(1, 65293, '\P{^Bc:es}', "");
Expect(0, 65294, '\p{Bc:es}', "");
Expect(1, 65294, '\p{^Bc:es}', "");
Expect(1, 65294, '\P{Bc:es}', "");
Expect(0, 65294, '\P{^Bc:es}', "");
Expect(1, 65293, '\p{Bc=_ES}', "");
Expect(0, 65293, '\p{^Bc=_ES}', "");
Expect(0, 65293, '\P{Bc=_ES}', "");
Expect(1, 65293, '\P{^Bc=_ES}', "");
Expect(0, 65294, '\p{Bc=_ES}', "");
Expect(1, 65294, '\p{^Bc=_ES}', "");
Expect(1, 65294, '\P{Bc=_ES}', "");
Expect(0, 65294, '\P{^Bc=_ES}', "");
Error('\p{Is_Bidi_Class= -european_SEPARATOR:=}');
Error('\P{Is_Bidi_Class= -european_SEPARATOR:=}');
Expect(1, 65293, '\p{Is_Bidi_Class=europeanseparator}', "");
Expect(0, 65293, '\p{^Is_Bidi_Class=europeanseparator}', "");
Expect(0, 65293, '\P{Is_Bidi_Class=europeanseparator}', "");
Expect(1, 65293, '\P{^Is_Bidi_Class=europeanseparator}', "");
Expect(0, 65294, '\p{Is_Bidi_Class=europeanseparator}', "");
Expect(1, 65294, '\p{^Is_Bidi_Class=europeanseparator}', "");
Expect(1, 65294, '\P{Is_Bidi_Class=europeanseparator}', "");
Expect(0, 65294, '\P{^Is_Bidi_Class=europeanseparator}', "");
Expect(1, 65293, '\p{Is_Bidi_Class=- European_separator}', "");
Expect(0, 65293, '\p{^Is_Bidi_Class=- European_separator}', "");
Expect(0, 65293, '\P{Is_Bidi_Class=- European_separator}', "");
Expect(1, 65293, '\P{^Is_Bidi_Class=- European_separator}', "");
Expect(0, 65294, '\p{Is_Bidi_Class=- European_separator}', "");
Expect(1, 65294, '\p{^Is_Bidi_Class=- European_separator}', "");
Expect(1, 65294, '\P{Is_Bidi_Class=- European_separator}', "");
Expect(0, 65294, '\P{^Is_Bidi_Class=- European_separator}', "");
Error('\p{Is_Bc=	/a/ES}');
Error('\P{Is_Bc=	/a/ES}');
Expect(1, 65293, '\p{Is_Bc=es}', "");
Expect(0, 65293, '\p{^Is_Bc=es}', "");
Expect(0, 65293, '\P{Is_Bc=es}', "");
Expect(1, 65293, '\P{^Is_Bc=es}', "");
Expect(0, 65294, '\p{Is_Bc=es}', "");
Expect(1, 65294, '\p{^Is_Bc=es}', "");
Expect(1, 65294, '\P{Is_Bc=es}', "");
Expect(0, 65294, '\P{^Is_Bc=es}', "");
Expect(1, 65293, '\p{Is_Bc:-ES}', "");
Expect(0, 65293, '\p{^Is_Bc:-ES}', "");
Expect(0, 65293, '\P{Is_Bc:-ES}', "");
Expect(1, 65293, '\P{^Is_Bc:-ES}', "");
Expect(0, 65294, '\p{Is_Bc:-ES}', "");
Expect(1, 65294, '\p{^Is_Bc:-ES}', "");
Expect(1, 65294, '\P{Is_Bc:-ES}', "");
Expect(0, 65294, '\P{^Is_Bc:-ES}', "");
Error('\p{Bidi_Class:   _:=EUROPEAN_TERMINATOR}');
Error('\P{Bidi_Class:   _:=EUROPEAN_TERMINATOR}');
Expect(1, 65510, '\p{Bidi_Class:europeanterminator}', "");
Expect(0, 65510, '\p{^Bidi_Class:europeanterminator}', "");
Expect(0, 65510, '\P{Bidi_Class:europeanterminator}', "");
Expect(1, 65510, '\P{^Bidi_Class:europeanterminator}', "");
Expect(0, 65511, '\p{Bidi_Class:europeanterminator}', "");
Expect(1, 65511, '\p{^Bidi_Class:europeanterminator}', "");
Expect(1, 65511, '\P{Bidi_Class:europeanterminator}', "");
Expect(0, 65511, '\P{^Bidi_Class:europeanterminator}', "");
Expect(1, 65510, '\p{Bidi_Class=-European_TERMINATOR}', "");
Expect(0, 65510, '\p{^Bidi_Class=-European_TERMINATOR}', "");
Expect(0, 65510, '\P{Bidi_Class=-European_TERMINATOR}', "");
Expect(1, 65510, '\P{^Bidi_Class=-European_TERMINATOR}', "");
Expect(0, 65511, '\p{Bidi_Class=-European_TERMINATOR}', "");
Expect(1, 65511, '\p{^Bidi_Class=-European_TERMINATOR}', "");
Expect(1, 65511, '\P{Bidi_Class=-European_TERMINATOR}', "");
Expect(0, 65511, '\P{^Bidi_Class=-European_TERMINATOR}', "");
Error('\p{Bc=ET/a/}');
Error('\P{Bc=ET/a/}');
Expect(1, 65510, '\p{Bc=et}', "");
Expect(0, 65510, '\p{^Bc=et}', "");
Expect(0, 65510, '\P{Bc=et}', "");
Expect(1, 65510, '\P{^Bc=et}', "");
Expect(0, 65511, '\p{Bc=et}', "");
Expect(1, 65511, '\p{^Bc=et}', "");
Expect(1, 65511, '\P{Bc=et}', "");
Expect(0, 65511, '\P{^Bc=et}', "");
Expect(1, 65510, '\p{Bc= ET}', "");
Expect(0, 65510, '\p{^Bc= ET}', "");
Expect(0, 65510, '\P{Bc= ET}', "");
Expect(1, 65510, '\P{^Bc= ET}', "");
Expect(0, 65511, '\p{Bc= ET}', "");
Expect(1, 65511, '\p{^Bc= ET}', "");
Expect(1, 65511, '\P{Bc= ET}', "");
Expect(0, 65511, '\P{^Bc= ET}', "");
Error('\p{Is_Bidi_Class= /a/european_Terminator}');
Error('\P{Is_Bidi_Class= /a/european_Terminator}');
Expect(1, 65510, '\p{Is_Bidi_Class=europeanterminator}', "");
Expect(0, 65510, '\p{^Is_Bidi_Class=europeanterminator}', "");
Expect(0, 65510, '\P{Is_Bidi_Class=europeanterminator}', "");
Expect(1, 65510, '\P{^Is_Bidi_Class=europeanterminator}', "");
Expect(0, 65511, '\p{Is_Bidi_Class=europeanterminator}', "");
Expect(1, 65511, '\p{^Is_Bidi_Class=europeanterminator}', "");
Expect(1, 65511, '\P{Is_Bidi_Class=europeanterminator}', "");
Expect(0, 65511, '\P{^Is_Bidi_Class=europeanterminator}', "");
Expect(1, 65510, '\p{Is_Bidi_Class=-	european_Terminator}', "");
Expect(0, 65510, '\p{^Is_Bidi_Class=-	european_Terminator}', "");
Expect(0, 65510, '\P{Is_Bidi_Class=-	european_Terminator}', "");
Expect(1, 65510, '\P{^Is_Bidi_Class=-	european_Terminator}', "");
Expect(0, 65511, '\p{Is_Bidi_Class=-	european_Terminator}', "");
Expect(1, 65511, '\p{^Is_Bidi_Class=-	european_Terminator}', "");
Expect(1, 65511, '\P{Is_Bidi_Class=-	european_Terminator}', "");
Expect(0, 65511, '\P{^Is_Bidi_Class=-	european_Terminator}', "");
Error('\p{Is_Bc=:=et}');
Error('\P{Is_Bc=:=et}');
Expect(1, 65510, '\p{Is_Bc=et}', "");
Expect(0, 65510, '\p{^Is_Bc=et}', "");
Expect(0, 65510, '\P{Is_Bc=et}', "");
Expect(1, 65510, '\P{^Is_Bc=et}', "");
Expect(0, 65511, '\p{Is_Bc=et}', "");
Expect(1, 65511, '\p{^Is_Bc=et}', "");
Expect(1, 65511, '\P{Is_Bc=et}', "");
Expect(0, 65511, '\P{^Is_Bc=et}', "");
Expect(1, 65510, '\p{Is_Bc=	_ET}', "");
Expect(0, 65510, '\p{^Is_Bc=	_ET}', "");
Expect(0, 65510, '\P{Is_Bc=	_ET}', "");
Expect(1, 65510, '\P{^Is_Bc=	_ET}', "");
Expect(0, 65511, '\p{Is_Bc=	_ET}', "");
Expect(1, 65511, '\p{^Is_Bc=	_ET}', "");
Expect(1, 65511, '\P{Is_Bc=	_ET}', "");
Expect(0, 65511, '\P{^Is_Bc=	_ET}', "");
Error('\p{Bidi_Class=/a/	_First_Strong_isolate}');
Error('\P{Bidi_Class=/a/	_First_Strong_isolate}');
Expect(1, 8296, '\p{Bidi_Class=firststrongisolate}', "");
Expect(0, 8296, '\p{^Bidi_Class=firststrongisolate}', "");
Expect(0, 8296, '\P{Bidi_Class=firststrongisolate}', "");
Expect(1, 8296, '\P{^Bidi_Class=firststrongisolate}', "");
Expect(0, 8297, '\p{Bidi_Class=firststrongisolate}', "");
Expect(1, 8297, '\p{^Bidi_Class=firststrongisolate}', "");
Expect(1, 8297, '\P{Bidi_Class=firststrongisolate}', "");
Expect(0, 8297, '\P{^Bidi_Class=firststrongisolate}', "");
Expect(1, 8296, '\p{Bidi_Class=	FIRST_Strong_ISOLATE}', "");
Expect(0, 8296, '\p{^Bidi_Class=	FIRST_Strong_ISOLATE}', "");
Expect(0, 8296, '\P{Bidi_Class=	FIRST_Strong_ISOLATE}', "");
Expect(1, 8296, '\P{^Bidi_Class=	FIRST_Strong_ISOLATE}', "");
Expect(0, 8297, '\p{Bidi_Class=	FIRST_Strong_ISOLATE}', "");
Expect(1, 8297, '\p{^Bidi_Class=	FIRST_Strong_ISOLATE}', "");
Expect(1, 8297, '\P{Bidi_Class=	FIRST_Strong_ISOLATE}', "");
Expect(0, 8297, '\P{^Bidi_Class=	FIRST_Strong_ISOLATE}', "");
Error('\p{Bc:_:=FSI}');
Error('\P{Bc:_:=FSI}');
Expect(1, 8296, '\p{Bc=fsi}', "");
Expect(0, 8296, '\p{^Bc=fsi}', "");
Expect(0, 8296, '\P{Bc=fsi}', "");
Expect(1, 8296, '\P{^Bc=fsi}', "");
Expect(0, 8297, '\p{Bc=fsi}', "");
Expect(1, 8297, '\p{^Bc=fsi}', "");
Expect(1, 8297, '\P{Bc=fsi}', "");
Expect(0, 8297, '\P{^Bc=fsi}', "");
Expect(1, 8296, '\p{Bc=  FSI}', "");
Expect(0, 8296, '\p{^Bc=  FSI}', "");
Expect(0, 8296, '\P{Bc=  FSI}', "");
Expect(1, 8296, '\P{^Bc=  FSI}', "");
Expect(0, 8297, '\p{Bc=  FSI}', "");
Expect(1, 8297, '\p{^Bc=  FSI}', "");
Expect(1, 8297, '\P{Bc=  FSI}', "");
Expect(0, 8297, '\P{^Bc=  FSI}', "");
Error('\p{Is_Bidi_Class= /a/first_STRONG_ISOLATE}');
Error('\P{Is_Bidi_Class= /a/first_STRONG_ISOLATE}');
Expect(1, 8296, '\p{Is_Bidi_Class=firststrongisolate}', "");
Expect(0, 8296, '\p{^Is_Bidi_Class=firststrongisolate}', "");
Expect(0, 8296, '\P{Is_Bidi_Class=firststrongisolate}', "");
Expect(1, 8296, '\P{^Is_Bidi_Class=firststrongisolate}', "");
Expect(0, 8297, '\p{Is_Bidi_Class=firststrongisolate}', "");
Expect(1, 8297, '\p{^Is_Bidi_Class=firststrongisolate}', "");
Expect(1, 8297, '\P{Is_Bidi_Class=firststrongisolate}', "");
Expect(0, 8297, '\P{^Is_Bidi_Class=firststrongisolate}', "");
Expect(1, 8296, '\p{Is_Bidi_Class=	_First_STRONG_Isolate}', "");
Expect(0, 8296, '\p{^Is_Bidi_Class=	_First_STRONG_Isolate}', "");
Expect(0, 8296, '\P{Is_Bidi_Class=	_First_STRONG_Isolate}', "");
Expect(1, 8296, '\P{^Is_Bidi_Class=	_First_STRONG_Isolate}', "");
Expect(0, 8297, '\p{Is_Bidi_Class=	_First_STRONG_Isolate}', "");
Expect(1, 8297, '\p{^Is_Bidi_Class=	_First_STRONG_Isolate}', "");
Expect(1, 8297, '\P{Is_Bidi_Class=	_First_STRONG_Isolate}', "");
Expect(0, 8297, '\P{^Is_Bidi_Class=	_First_STRONG_Isolate}', "");
Error('\p{Is_Bc=	:=fsi}');
Error('\P{Is_Bc=	:=fsi}');
Expect(1, 8296, '\p{Is_Bc=fsi}', "");
Expect(0, 8296, '\p{^Is_Bc=fsi}', "");
Expect(0, 8296, '\P{Is_Bc=fsi}', "");
Expect(1, 8296, '\P{^Is_Bc=fsi}', "");
Expect(0, 8297, '\p{Is_Bc=fsi}', "");
Expect(1, 8297, '\p{^Is_Bc=fsi}', "");
Expect(1, 8297, '\P{Is_Bc=fsi}', "");
Expect(0, 8297, '\P{^Is_Bc=fsi}', "");
Expect(1, 8296, '\p{Is_Bc=--fsi}', "");
Expect(0, 8296, '\p{^Is_Bc=--fsi}', "");
Expect(0, 8296, '\P{Is_Bc=--fsi}', "");
Expect(1, 8296, '\P{^Is_Bc=--fsi}', "");
Expect(0, 8297, '\p{Is_Bc=--fsi}', "");
Expect(1, 8297, '\p{^Is_Bc=--fsi}', "");
Expect(1, 8297, '\P{Is_Bc=--fsi}', "");
Expect(0, 8297, '\P{^Is_Bc=--fsi}', "");
Error('\p{Bidi_Class=/a/	-LEFT_To_Right}');
Error('\P{Bidi_Class=/a/	-LEFT_To_Right}');
Expect(1, 1114109, '\p{Bidi_Class=lefttoright}', "");
Expect(0, 1114109, '\p{^Bidi_Class=lefttoright}', "");
Expect(0, 1114109, '\P{Bidi_Class=lefttoright}', "");
Expect(1, 1114109, '\P{^Bidi_Class=lefttoright}', "");
Expect(0, 921599, '\p{Bidi_Class=lefttoright}', "");
Expect(1, 921599, '\p{^Bidi_Class=lefttoright}', "");
Expect(1, 921599, '\P{Bidi_Class=lefttoright}', "");
Expect(0, 921599, '\P{^Bidi_Class=lefttoright}', "");
Expect(1, 1114109, '\p{Bidi_Class=	_left_to_Right}', "");
Expect(0, 1114109, '\p{^Bidi_Class=	_left_to_Right}', "");
Expect(0, 1114109, '\P{Bidi_Class=	_left_to_Right}', "");
Expect(1, 1114109, '\P{^Bidi_Class=	_left_to_Right}', "");
Expect(0, 921599, '\p{Bidi_Class=	_left_to_Right}', "");
Expect(1, 921599, '\p{^Bidi_Class=	_left_to_Right}', "");
Expect(1, 921599, '\P{Bidi_Class=	_left_to_Right}', "");
Expect(0, 921599, '\P{^Bidi_Class=	_left_to_Right}', "");
Error('\p{Bc=_l/a/}');
Error('\P{Bc=_l/a/}');
Expect(1, 1114109, '\p{Bc:l}', "");
Expect(0, 1114109, '\p{^Bc:l}', "");
Expect(0, 1114109, '\P{Bc:l}', "");
Expect(1, 1114109, '\P{^Bc:l}', "");
Expect(0, 921599, '\p{Bc:l}', "");
Expect(1, 921599, '\p{^Bc:l}', "");
Expect(1, 921599, '\P{Bc:l}', "");
Expect(0, 921599, '\P{^Bc:l}', "");
Expect(1, 1114109, '\p{Bc=__L}', "");
Expect(0, 1114109, '\p{^Bc=__L}', "");
Expect(0, 1114109, '\P{Bc=__L}', "");
Expect(1, 1114109, '\P{^Bc=__L}', "");
Expect(0, 921599, '\p{Bc=__L}', "");
Expect(1, 921599, '\p{^Bc=__L}', "");
Expect(1, 921599, '\P{Bc=__L}', "");
Expect(0, 921599, '\P{^Bc=__L}', "");
Error('\p{Is_Bidi_Class=:=-Left_to_right}');
Error('\P{Is_Bidi_Class=:=-Left_to_right}');
Expect(1, 1114109, '\p{Is_Bidi_Class=lefttoright}', "");
Expect(0, 1114109, '\p{^Is_Bidi_Class=lefttoright}', "");
Expect(0, 1114109, '\P{Is_Bidi_Class=lefttoright}', "");
Expect(1, 1114109, '\P{^Is_Bidi_Class=lefttoright}', "");
Expect(0, 921599, '\p{Is_Bidi_Class=lefttoright}', "");
Expect(1, 921599, '\p{^Is_Bidi_Class=lefttoright}', "");
Expect(1, 921599, '\P{Is_Bidi_Class=lefttoright}', "");
Expect(0, 921599, '\P{^Is_Bidi_Class=lefttoright}', "");
Expect(1, 1114109, '\p{Is_Bidi_Class=	_LEFT_To_right}', "");
Expect(0, 1114109, '\p{^Is_Bidi_Class=	_LEFT_To_right}', "");
Expect(0, 1114109, '\P{Is_Bidi_Class=	_LEFT_To_right}', "");
Expect(1, 1114109, '\P{^Is_Bidi_Class=	_LEFT_To_right}', "");
Expect(0, 921599, '\p{Is_Bidi_Class=	_LEFT_To_right}', "");
Expect(1, 921599, '\p{^Is_Bidi_Class=	_LEFT_To_right}', "");
Expect(1, 921599, '\P{Is_Bidi_Class=	_LEFT_To_right}', "");
Expect(0, 921599, '\P{^Is_Bidi_Class=	_LEFT_To_right}', "");
Error('\p{Is_Bc:	:=_L}');
Error('\P{Is_Bc:	:=_L}');
Expect(1, 1114109, '\p{Is_Bc=l}', "");
Expect(0, 1114109, '\p{^Is_Bc=l}', "");
Expect(0, 1114109, '\P{Is_Bc=l}', "");
Expect(1, 1114109, '\P{^Is_Bc=l}', "");
Expect(0, 921599, '\p{Is_Bc=l}', "");
Expect(1, 921599, '\p{^Is_Bc=l}', "");
Expect(1, 921599, '\P{Is_Bc=l}', "");
Expect(0, 921599, '\P{^Is_Bc=l}', "");
Expect(1, 1114109, '\p{Is_Bc=	 l}', "");
Expect(0, 1114109, '\p{^Is_Bc=	 l}', "");
Expect(0, 1114109, '\P{Is_Bc=	 l}', "");
Expect(1, 1114109, '\P{^Is_Bc=	 l}', "");
Expect(0, 921599, '\p{Is_Bc=	 l}', "");
Expect(1, 921599, '\p{^Is_Bc=	 l}', "");
Expect(1, 921599, '\P{Is_Bc=	 l}', "");
Expect(0, 921599, '\P{^Is_Bc=	 l}', "");
Error('\p{Bidi_Class=:= Left_To_Right_embedding}');
Error('\P{Bidi_Class=:= Left_To_Right_embedding}');
Expect(1, 8234, '\p{Bidi_Class=lefttorightembedding}', "");
Expect(0, 8234, '\p{^Bidi_Class=lefttorightembedding}', "");
Expect(0, 8234, '\P{Bidi_Class=lefttorightembedding}', "");
Expect(1, 8234, '\P{^Bidi_Class=lefttorightembedding}', "");
Expect(0, 8235, '\p{Bidi_Class=lefttorightembedding}', "");
Expect(1, 8235, '\p{^Bidi_Class=lefttorightembedding}', "");
Expect(1, 8235, '\P{Bidi_Class=lefttorightembedding}', "");
Expect(0, 8235, '\P{^Bidi_Class=lefttorightembedding}', "");
Expect(1, 8234, '\p{Bidi_Class= Left_To_RIGHT_Embedding}', "");
Expect(0, 8234, '\p{^Bidi_Class= Left_To_RIGHT_Embedding}', "");
Expect(0, 8234, '\P{Bidi_Class= Left_To_RIGHT_Embedding}', "");
Expect(1, 8234, '\P{^Bidi_Class= Left_To_RIGHT_Embedding}', "");
Expect(0, 8235, '\p{Bidi_Class= Left_To_RIGHT_Embedding}', "");
Expect(1, 8235, '\p{^Bidi_Class= Left_To_RIGHT_Embedding}', "");
Expect(1, 8235, '\P{Bidi_Class= Left_To_RIGHT_Embedding}', "");
Expect(0, 8235, '\P{^Bidi_Class= Left_To_RIGHT_Embedding}', "");
Error('\p{Bc=:=_LRE}');
Error('\P{Bc=:=_LRE}');
Expect(1, 8234, '\p{Bc=lre}', "");
Expect(0, 8234, '\p{^Bc=lre}', "");
Expect(0, 8234, '\P{Bc=lre}', "");
Expect(1, 8234, '\P{^Bc=lre}', "");
Expect(0, 8235, '\p{Bc=lre}', "");
Expect(1, 8235, '\p{^Bc=lre}', "");
Expect(1, 8235, '\P{Bc=lre}', "");
Expect(0, 8235, '\P{^Bc=lre}', "");
Expect(1, 8234, '\p{Bc: 	lre}', "");
Expect(0, 8234, '\p{^Bc: 	lre}', "");
Expect(0, 8234, '\P{Bc: 	lre}', "");
Expect(1, 8234, '\P{^Bc: 	lre}', "");
Expect(0, 8235, '\p{Bc: 	lre}', "");
Expect(1, 8235, '\p{^Bc: 	lre}', "");
Expect(1, 8235, '\P{Bc: 	lre}', "");
Expect(0, 8235, '\P{^Bc: 	lre}', "");
Error('\p{Is_Bidi_Class=  Left_To_Right_embedding:=}');
Error('\P{Is_Bidi_Class=  Left_To_Right_embedding:=}');
Expect(1, 8234, '\p{Is_Bidi_Class=lefttorightembedding}', "");
Expect(0, 8234, '\p{^Is_Bidi_Class=lefttorightembedding}', "");
Expect(0, 8234, '\P{Is_Bidi_Class=lefttorightembedding}', "");
Expect(1, 8234, '\P{^Is_Bidi_Class=lefttorightembedding}', "");
Expect(0, 8235, '\p{Is_Bidi_Class=lefttorightembedding}', "");
Expect(1, 8235, '\p{^Is_Bidi_Class=lefttorightembedding}', "");
Expect(1, 8235, '\P{Is_Bidi_Class=lefttorightembedding}', "");
Expect(0, 8235, '\P{^Is_Bidi_Class=lefttorightembedding}', "");
Expect(1, 8234, '\p{Is_Bidi_Class=-left_To_Right_Embedding}', "");
Expect(0, 8234, '\p{^Is_Bidi_Class=-left_To_Right_Embedding}', "");
Expect(0, 8234, '\P{Is_Bidi_Class=-left_To_Right_Embedding}', "");
Expect(1, 8234, '\P{^Is_Bidi_Class=-left_To_Right_Embedding}', "");
Expect(0, 8235, '\p{Is_Bidi_Class=-left_To_Right_Embedding}', "");
Expect(1, 8235, '\p{^Is_Bidi_Class=-left_To_Right_Embedding}', "");
Expect(1, 8235, '\P{Is_Bidi_Class=-left_To_Right_Embedding}', "");
Expect(0, 8235, '\P{^Is_Bidi_Class=-left_To_Right_Embedding}', "");
Error('\p{Is_Bc=_/a/lre}');
Error('\P{Is_Bc=_/a/lre}');
Expect(1, 8234, '\p{Is_Bc=lre}', "");
Expect(0, 8234, '\p{^Is_Bc=lre}', "");
Expect(0, 8234, '\P{Is_Bc=lre}', "");
Expect(1, 8234, '\P{^Is_Bc=lre}', "");
Expect(0, 8235, '\p{Is_Bc=lre}', "");
Expect(1, 8235, '\p{^Is_Bc=lre}', "");
Expect(1, 8235, '\P{Is_Bc=lre}', "");
Expect(0, 8235, '\P{^Is_Bc=lre}', "");
Expect(1, 8234, '\p{Is_Bc= lre}', "");
Expect(0, 8234, '\p{^Is_Bc= lre}', "");
Expect(0, 8234, '\P{Is_Bc= lre}', "");
Expect(1, 8234, '\P{^Is_Bc= lre}', "");
Expect(0, 8235, '\p{Is_Bc= lre}', "");
Expect(1, 8235, '\p{^Is_Bc= lre}', "");
Expect(1, 8235, '\P{Is_Bc= lre}', "");
Expect(0, 8235, '\P{^Is_Bc= lre}', "");
Error('\p{Bidi_Class=-/a/Left_To_RIGHT_ISOLATE}');
Error('\P{Bidi_Class=-/a/Left_To_RIGHT_ISOLATE}');
Expect(1, 8294, '\p{Bidi_Class=lefttorightisolate}', "");
Expect(0, 8294, '\p{^Bidi_Class=lefttorightisolate}', "");
Expect(0, 8294, '\P{Bidi_Class=lefttorightisolate}', "");
Expect(1, 8294, '\P{^Bidi_Class=lefttorightisolate}', "");
Expect(0, 8295, '\p{Bidi_Class=lefttorightisolate}', "");
Expect(1, 8295, '\p{^Bidi_Class=lefttorightisolate}', "");
Expect(1, 8295, '\P{Bidi_Class=lefttorightisolate}', "");
Expect(0, 8295, '\P{^Bidi_Class=lefttorightisolate}', "");
Expect(1, 8294, '\p{Bidi_Class=-Left_To_Right_Isolate}', "");
Expect(0, 8294, '\p{^Bidi_Class=-Left_To_Right_Isolate}', "");
Expect(0, 8294, '\P{Bidi_Class=-Left_To_Right_Isolate}', "");
Expect(1, 8294, '\P{^Bidi_Class=-Left_To_Right_Isolate}', "");
Expect(0, 8295, '\p{Bidi_Class=-Left_To_Right_Isolate}', "");
Expect(1, 8295, '\p{^Bidi_Class=-Left_To_Right_Isolate}', "");
Expect(1, 8295, '\P{Bidi_Class=-Left_To_Right_Isolate}', "");
Expect(0, 8295, '\P{^Bidi_Class=-Left_To_Right_Isolate}', "");
Error('\p{Bc:   __LRI/a/}');
Error('\P{Bc:   __LRI/a/}');
Expect(1, 8294, '\p{Bc=lri}', "");
Expect(0, 8294, '\p{^Bc=lri}', "");
Expect(0, 8294, '\P{Bc=lri}', "");
Expect(1, 8294, '\P{^Bc=lri}', "");
Expect(0, 8295, '\p{Bc=lri}', "");
Expect(1, 8295, '\p{^Bc=lri}', "");
Expect(1, 8295, '\P{Bc=lri}', "");
Expect(0, 8295, '\P{^Bc=lri}', "");
Expect(1, 8294, '\p{Bc=LRI}', "");
Expect(0, 8294, '\p{^Bc=LRI}', "");
Expect(0, 8294, '\P{Bc=LRI}', "");
Expect(1, 8294, '\P{^Bc=LRI}', "");
Expect(0, 8295, '\p{Bc=LRI}', "");
Expect(1, 8295, '\p{^Bc=LRI}', "");
Expect(1, 8295, '\P{Bc=LRI}', "");
Expect(0, 8295, '\P{^Bc=LRI}', "");
Error('\p{Is_Bidi_Class=/a/Left_To_right_isolate}');
Error('\P{Is_Bidi_Class=/a/Left_To_right_isolate}');
Expect(1, 8294, '\p{Is_Bidi_Class=lefttorightisolate}', "");
Expect(0, 8294, '\p{^Is_Bidi_Class=lefttorightisolate}', "");
Expect(0, 8294, '\P{Is_Bidi_Class=lefttorightisolate}', "");
Expect(1, 8294, '\P{^Is_Bidi_Class=lefttorightisolate}', "");
Expect(0, 8295, '\p{Is_Bidi_Class=lefttorightisolate}', "");
Expect(1, 8295, '\p{^Is_Bidi_Class=lefttorightisolate}', "");
Expect(1, 8295, '\P{Is_Bidi_Class=lefttorightisolate}', "");
Expect(0, 8295, '\P{^Is_Bidi_Class=lefttorightisolate}', "");
Expect(1, 8294, '\p{Is_Bidi_Class=-Left_To_right_Isolate}', "");
Expect(0, 8294, '\p{^Is_Bidi_Class=-Left_To_right_Isolate}', "");
Expect(0, 8294, '\P{Is_Bidi_Class=-Left_To_right_Isolate}', "");
Expect(1, 8294, '\P{^Is_Bidi_Class=-Left_To_right_Isolate}', "");
Expect(0, 8295, '\p{Is_Bidi_Class=-Left_To_right_Isolate}', "");
Expect(1, 8295, '\p{^Is_Bidi_Class=-Left_To_right_Isolate}', "");
Expect(1, 8295, '\P{Is_Bidi_Class=-Left_To_right_Isolate}', "");
Expect(0, 8295, '\P{^Is_Bidi_Class=-Left_To_right_Isolate}', "");
Error('\p{Is_Bc=:=-LRI}');
Error('\P{Is_Bc=:=-LRI}');
Expect(1, 8294, '\p{Is_Bc:	lri}', "");
Expect(0, 8294, '\p{^Is_Bc:	lri}', "");
Expect(0, 8294, '\P{Is_Bc:	lri}', "");
Expect(1, 8294, '\P{^Is_Bc:	lri}', "");
Expect(0, 8295, '\p{Is_Bc:	lri}', "");
Expect(1, 8295, '\p{^Is_Bc:	lri}', "");
Expect(1, 8295, '\P{Is_Bc:	lri}', "");
Expect(0, 8295, '\P{^Is_Bc:	lri}', "");
Expect(1, 8294, '\p{Is_Bc=-LRI}', "");
Expect(0, 8294, '\p{^Is_Bc=-LRI}', "");
Expect(0, 8294, '\P{Is_Bc=-LRI}', "");
Expect(1, 8294, '\P{^Is_Bc=-LRI}', "");
Expect(0, 8295, '\p{Is_Bc=-LRI}', "");
Expect(1, 8295, '\p{^Is_Bc=-LRI}', "");
Expect(1, 8295, '\P{Is_Bc=-LRI}', "");
Expect(0, 8295, '\P{^Is_Bc=-LRI}', "");
Error('\p{Bidi_Class=/a/Left_TO_RIGHT_Override}');
Error('\P{Bidi_Class=/a/Left_TO_RIGHT_Override}');
Expect(1, 8237, '\p{Bidi_Class=lefttorightoverride}', "");
Expect(0, 8237, '\p{^Bidi_Class=lefttorightoverride}', "");
Expect(0, 8237, '\P{Bidi_Class=lefttorightoverride}', "");
Expect(1, 8237, '\P{^Bidi_Class=lefttorightoverride}', "");
Expect(0, 8238, '\p{Bidi_Class=lefttorightoverride}', "");
Expect(1, 8238, '\p{^Bidi_Class=lefttorightoverride}', "");
Expect(1, 8238, '\P{Bidi_Class=lefttorightoverride}', "");
Expect(0, 8238, '\P{^Bidi_Class=lefttorightoverride}', "");
Expect(1, 8237, '\p{Bidi_Class=	_left_To_Right_Override}', "");
Expect(0, 8237, '\p{^Bidi_Class=	_left_To_Right_Override}', "");
Expect(0, 8237, '\P{Bidi_Class=	_left_To_Right_Override}', "");
Expect(1, 8237, '\P{^Bidi_Class=	_left_To_Right_Override}', "");
Expect(0, 8238, '\p{Bidi_Class=	_left_To_Right_Override}', "");
Expect(1, 8238, '\p{^Bidi_Class=	_left_To_Right_Override}', "");
Expect(1, 8238, '\P{Bidi_Class=	_left_To_Right_Override}', "");
Expect(0, 8238, '\P{^Bidi_Class=	_left_To_Right_Override}', "");
Error('\p{Bc=:= LRO}');
Error('\P{Bc=:= LRO}');
Expect(1, 8237, '\p{Bc=lro}', "");
Expect(0, 8237, '\p{^Bc=lro}', "");
Expect(0, 8237, '\P{Bc=lro}', "");
Expect(1, 8237, '\P{^Bc=lro}', "");
Expect(0, 8238, '\p{Bc=lro}', "");
Expect(1, 8238, '\p{^Bc=lro}', "");
Expect(1, 8238, '\P{Bc=lro}', "");
Expect(0, 8238, '\P{^Bc=lro}', "");
Expect(1, 8237, '\p{Bc= _lro}', "");
Expect(0, 8237, '\p{^Bc= _lro}', "");
Expect(0, 8237, '\P{Bc= _lro}', "");
Expect(1, 8237, '\P{^Bc= _lro}', "");
Expect(0, 8238, '\p{Bc= _lro}', "");
Expect(1, 8238, '\p{^Bc= _lro}', "");
Expect(1, 8238, '\P{Bc= _lro}', "");
Expect(0, 8238, '\P{^Bc= _lro}', "");
Error('\p{Is_Bidi_Class=  Left_To_Right_OVERRIDE:=}');
Error('\P{Is_Bidi_Class=  Left_To_Right_OVERRIDE:=}');
Expect(1, 8237, '\p{Is_Bidi_Class=lefttorightoverride}', "");
Expect(0, 8237, '\p{^Is_Bidi_Class=lefttorightoverride}', "");
Expect(0, 8237, '\P{Is_Bidi_Class=lefttorightoverride}', "");
Expect(1, 8237, '\P{^Is_Bidi_Class=lefttorightoverride}', "");
Expect(0, 8238, '\p{Is_Bidi_Class=lefttorightoverride}', "");
Expect(1, 8238, '\p{^Is_Bidi_Class=lefttorightoverride}', "");
Expect(1, 8238, '\P{Is_Bidi_Class=lefttorightoverride}', "");
Expect(0, 8238, '\P{^Is_Bidi_Class=lefttorightoverride}', "");
Expect(1, 8237, '\p{Is_Bidi_Class=  Left_TO_RIGHT_Override}', "");
Expect(0, 8237, '\p{^Is_Bidi_Class=  Left_TO_RIGHT_Override}', "");
Expect(0, 8237, '\P{Is_Bidi_Class=  Left_TO_RIGHT_Override}', "");
Expect(1, 8237, '\P{^Is_Bidi_Class=  Left_TO_RIGHT_Override}', "");
Expect(0, 8238, '\p{Is_Bidi_Class=  Left_TO_RIGHT_Override}', "");
Expect(1, 8238, '\p{^Is_Bidi_Class=  Left_TO_RIGHT_Override}', "");
Expect(1, 8238, '\P{Is_Bidi_Class=  Left_TO_RIGHT_Override}', "");
Expect(0, 8238, '\P{^Is_Bidi_Class=  Left_TO_RIGHT_Override}', "");
Error('\p{Is_Bc=:=-lro}');
Error('\P{Is_Bc=:=-lro}');
Expect(1, 8237, '\p{Is_Bc=lro}', "");
Expect(0, 8237, '\p{^Is_Bc=lro}', "");
Expect(0, 8237, '\P{Is_Bc=lro}', "");
Expect(1, 8237, '\P{^Is_Bc=lro}', "");
Expect(0, 8238, '\p{Is_Bc=lro}', "");
Expect(1, 8238, '\p{^Is_Bc=lro}', "");
Expect(1, 8238, '\P{Is_Bc=lro}', "");
Expect(0, 8238, '\P{^Is_Bc=lro}', "");
Expect(1, 8237, '\p{Is_Bc=_LRO}', "");
Expect(0, 8237, '\p{^Is_Bc=_LRO}', "");
Expect(0, 8237, '\P{Is_Bc=_LRO}', "");
Expect(1, 8237, '\P{^Is_Bc=_LRO}', "");
Expect(0, 8238, '\p{Is_Bc=_LRO}', "");
Expect(1, 8238, '\p{^Is_Bc=_LRO}', "");
Expect(1, 8238, '\P{Is_Bc=_LRO}', "");
Expect(0, 8238, '\P{^Is_Bc=_LRO}', "");
Error('\p{Bidi_Class=/a/_	Nonspacing_MARK}');
Error('\P{Bidi_Class=/a/_	Nonspacing_MARK}');
Expect(1, 917999, '\p{Bidi_Class=nonspacingmark}', "");
Expect(0, 917999, '\p{^Bidi_Class=nonspacingmark}', "");
Expect(0, 917999, '\P{Bidi_Class=nonspacingmark}', "");
Expect(1, 917999, '\P{^Bidi_Class=nonspacingmark}', "");
Expect(0, 918000, '\p{Bidi_Class=nonspacingmark}', "");
Expect(1, 918000, '\p{^Bidi_Class=nonspacingmark}', "");
Expect(1, 918000, '\P{Bidi_Class=nonspacingmark}', "");
Expect(0, 918000, '\P{^Bidi_Class=nonspacingmark}', "");
Expect(1, 917999, '\p{Bidi_Class=_Nonspacing_Mark}', "");
Expect(0, 917999, '\p{^Bidi_Class=_Nonspacing_Mark}', "");
Expect(0, 917999, '\P{Bidi_Class=_Nonspacing_Mark}', "");
Expect(1, 917999, '\P{^Bidi_Class=_Nonspacing_Mark}', "");
Expect(0, 918000, '\p{Bidi_Class=_Nonspacing_Mark}', "");
Expect(1, 918000, '\p{^Bidi_Class=_Nonspacing_Mark}', "");
Expect(1, 918000, '\P{Bidi_Class=_Nonspacing_Mark}', "");
Expect(0, 918000, '\P{^Bidi_Class=_Nonspacing_Mark}', "");
Error('\p{Bc=:=-_nsm}');
Error('\P{Bc=:=-_nsm}');
Expect(1, 917999, '\p{Bc=nsm}', "");
Expect(0, 917999, '\p{^Bc=nsm}', "");
Expect(0, 917999, '\P{Bc=nsm}', "");
Expect(1, 917999, '\P{^Bc=nsm}', "");
Expect(0, 918000, '\p{Bc=nsm}', "");
Expect(1, 918000, '\p{^Bc=nsm}', "");
Expect(1, 918000, '\P{Bc=nsm}', "");
Expect(0, 918000, '\P{^Bc=nsm}', "");
Expect(1, 917999, '\p{Bc=- NSM}', "");
Expect(0, 917999, '\p{^Bc=- NSM}', "");
Expect(0, 917999, '\P{Bc=- NSM}', "");
Expect(1, 917999, '\P{^Bc=- NSM}', "");
Expect(0, 918000, '\p{Bc=- NSM}', "");
Expect(1, 918000, '\p{^Bc=- NSM}', "");
Expect(1, 918000, '\P{Bc=- NSM}', "");
Expect(0, 918000, '\P{^Bc=- NSM}', "");
Error('\p{Is_Bidi_Class=	 Nonspacing_Mark/a/}');
Error('\P{Is_Bidi_Class=	 Nonspacing_Mark/a/}');
Expect(1, 917999, '\p{Is_Bidi_Class=nonspacingmark}', "");
Expect(0, 917999, '\p{^Is_Bidi_Class=nonspacingmark}', "");
Expect(0, 917999, '\P{Is_Bidi_Class=nonspacingmark}', "");
Expect(1, 917999, '\P{^Is_Bidi_Class=nonspacingmark}', "");
Expect(0, 918000, '\p{Is_Bidi_Class=nonspacingmark}', "");
Expect(1, 918000, '\p{^Is_Bidi_Class=nonspacingmark}', "");
Expect(1, 918000, '\P{Is_Bidi_Class=nonspacingmark}', "");
Expect(0, 918000, '\P{^Is_Bidi_Class=nonspacingmark}', "");
Expect(1, 917999, '\p{Is_Bidi_Class=-	NONSPACING_mark}', "");
Expect(0, 917999, '\p{^Is_Bidi_Class=-	NONSPACING_mark}', "");
Expect(0, 917999, '\P{Is_Bidi_Class=-	NONSPACING_mark}', "");
Expect(1, 917999, '\P{^Is_Bidi_Class=-	NONSPACING_mark}', "");
Expect(0, 918000, '\p{Is_Bidi_Class=-	NONSPACING_mark}', "");
Expect(1, 918000, '\p{^Is_Bidi_Class=-	NONSPACING_mark}', "");
Expect(1, 918000, '\P{Is_Bidi_Class=-	NONSPACING_mark}', "");
Expect(0, 918000, '\P{^Is_Bidi_Class=-	NONSPACING_mark}', "");
Error('\p{Is_Bc:			NSM/a/}');
Error('\P{Is_Bc:			NSM/a/}');
Expect(1, 917999, '\p{Is_Bc=nsm}', "");
Expect(0, 917999, '\p{^Is_Bc=nsm}', "");
Expect(0, 917999, '\P{Is_Bc=nsm}', "");
Expect(1, 917999, '\P{^Is_Bc=nsm}', "");
Expect(0, 918000, '\p{Is_Bc=nsm}', "");
Expect(1, 918000, '\p{^Is_Bc=nsm}', "");
Expect(1, 918000, '\P{Is_Bc=nsm}', "");
Expect(0, 918000, '\P{^Is_Bc=nsm}', "");
Expect(1, 917999, '\p{Is_Bc= NSM}', "");
Expect(0, 917999, '\p{^Is_Bc= NSM}', "");
Expect(0, 917999, '\P{Is_Bc= NSM}', "");
Expect(1, 917999, '\P{^Is_Bc= NSM}', "");
Expect(0, 918000, '\p{Is_Bc= NSM}', "");
Expect(1, 918000, '\p{^Is_Bc= NSM}', "");
Expect(1, 918000, '\P{Is_Bc= NSM}', "");
Expect(0, 918000, '\P{^Is_Bc= NSM}', "");
Error('\p{Bidi_Class= Other_Neutral:=}');
Error('\P{Bidi_Class= Other_Neutral:=}');
Expect(1, 128883, '\p{Bidi_Class=otherneutral}', "");
Expect(0, 128883, '\p{^Bidi_Class=otherneutral}', "");
Expect(0, 128883, '\P{Bidi_Class=otherneutral}', "");
Expect(1, 128883, '\P{^Bidi_Class=otherneutral}', "");
Expect(0, 128884, '\p{Bidi_Class=otherneutral}', "");
Expect(1, 128884, '\p{^Bidi_Class=otherneutral}', "");
Expect(1, 128884, '\P{Bidi_Class=otherneutral}', "");
Expect(0, 128884, '\P{^Bidi_Class=otherneutral}', "");
Expect(1, 128883, '\p{Bidi_Class=- OTHER_Neutral}', "");
Expect(0, 128883, '\p{^Bidi_Class=- OTHER_Neutral}', "");
Expect(0, 128883, '\P{Bidi_Class=- OTHER_Neutral}', "");
Expect(1, 128883, '\P{^Bidi_Class=- OTHER_Neutral}', "");
Expect(0, 128884, '\p{Bidi_Class=- OTHER_Neutral}', "");
Expect(1, 128884, '\p{^Bidi_Class=- OTHER_Neutral}', "");
Expect(1, 128884, '\P{Bidi_Class=- OTHER_Neutral}', "");
Expect(0, 128884, '\P{^Bidi_Class=- OTHER_Neutral}', "");
Error('\p{Bc:   /a/ -ON}');
Error('\P{Bc:   /a/ -ON}');
Expect(1, 128883, '\p{Bc=on}', "");
Expect(0, 128883, '\p{^Bc=on}', "");
Expect(0, 128883, '\P{Bc=on}', "");
Expect(1, 128883, '\P{^Bc=on}', "");
Expect(0, 128884, '\p{Bc=on}', "");
Expect(1, 128884, '\p{^Bc=on}', "");
Expect(1, 128884, '\P{Bc=on}', "");
Expect(0, 128884, '\P{^Bc=on}', "");
Expect(1, 128883, '\p{Bc=	_ON}', "");
Expect(0, 128883, '\p{^Bc=	_ON}', "");
Expect(0, 128883, '\P{Bc=	_ON}', "");
Expect(1, 128883, '\P{^Bc=	_ON}', "");
Expect(0, 128884, '\p{Bc=	_ON}', "");
Expect(1, 128884, '\p{^Bc=	_ON}', "");
Expect(1, 128884, '\P{Bc=	_ON}', "");
Expect(0, 128884, '\P{^Bc=	_ON}', "");
Error('\p{Is_Bidi_Class=-OTHER_Neutral/a/}');
Error('\P{Is_Bidi_Class=-OTHER_Neutral/a/}');
Expect(1, 128883, '\p{Is_Bidi_Class=otherneutral}', "");
Expect(0, 128883, '\p{^Is_Bidi_Class=otherneutral}', "");
Expect(0, 128883, '\P{Is_Bidi_Class=otherneutral}', "");
Expect(1, 128883, '\P{^Is_Bidi_Class=otherneutral}', "");
Expect(0, 128884, '\p{Is_Bidi_Class=otherneutral}', "");
Expect(1, 128884, '\p{^Is_Bidi_Class=otherneutral}', "");
Expect(1, 128884, '\P{Is_Bidi_Class=otherneutral}', "");
Expect(0, 128884, '\P{^Is_Bidi_Class=otherneutral}', "");
Expect(1, 128883, '\p{Is_Bidi_Class=	Other_Neutral}', "");
Expect(0, 128883, '\p{^Is_Bidi_Class=	Other_Neutral}', "");
Expect(0, 128883, '\P{Is_Bidi_Class=	Other_Neutral}', "");
Expect(1, 128883, '\P{^Is_Bidi_Class=	Other_Neutral}', "");
Expect(0, 128884, '\p{Is_Bidi_Class=	Other_Neutral}', "");
Expect(1, 128884, '\p{^Is_Bidi_Class=	Other_Neutral}', "");
Expect(1, 128884, '\P{Is_Bidi_Class=	Other_Neutral}', "");
Expect(0, 128884, '\P{^Is_Bidi_Class=	Other_Neutral}', "");
Error('\p{Is_Bc=:=__ON}');
Error('\P{Is_Bc=:=__ON}');
Expect(1, 128883, '\p{Is_Bc=on}', "");
Expect(0, 128883, '\p{^Is_Bc=on}', "");
Expect(0, 128883, '\P{Is_Bc=on}', "");
Expect(1, 128883, '\P{^Is_Bc=on}', "");
Expect(0, 128884, '\p{Is_Bc=on}', "");
Expect(1, 128884, '\p{^Is_Bc=on}', "");
Expect(1, 128884, '\P{Is_Bc=on}', "");
Expect(0, 128884, '\P{^Is_Bc=on}', "");
Expect(1, 128883, '\p{Is_Bc:   	_ON}', "");
Expect(0, 128883, '\p{^Is_Bc:   	_ON}', "");
Expect(0, 128883, '\P{Is_Bc:   	_ON}', "");
Expect(1, 128883, '\P{^Is_Bc:   	_ON}', "");
Expect(0, 128884, '\p{Is_Bc:   	_ON}', "");
Expect(1, 128884, '\p{^Is_Bc:   	_ON}', "");
Expect(1, 128884, '\P{Is_Bc:   	_ON}', "");
Expect(0, 128884, '\P{^Is_Bc:   	_ON}', "");
Error('\p{Bidi_Class=-/a/POP_directional_Format}');
Error('\P{Bidi_Class=-/a/POP_directional_Format}');
Expect(1, 8236, '\p{Bidi_Class=popdirectionalformat}', "");
Expect(0, 8236, '\p{^Bidi_Class=popdirectionalformat}', "");
Expect(0, 8236, '\P{Bidi_Class=popdirectionalformat}', "");
Expect(1, 8236, '\P{^Bidi_Class=popdirectionalformat}', "");
Expect(0, 8237, '\p{Bidi_Class=popdirectionalformat}', "");
Expect(1, 8237, '\p{^Bidi_Class=popdirectionalformat}', "");
Expect(1, 8237, '\P{Bidi_Class=popdirectionalformat}', "");
Expect(0, 8237, '\P{^Bidi_Class=popdirectionalformat}', "");
Expect(1, 8236, '\p{Bidi_Class=	POP_DIRECTIONAL_Format}', "");
Expect(0, 8236, '\p{^Bidi_Class=	POP_DIRECTIONAL_Format}', "");
Expect(0, 8236, '\P{Bidi_Class=	POP_DIRECTIONAL_Format}', "");
Expect(1, 8236, '\P{^Bidi_Class=	POP_DIRECTIONAL_Format}', "");
Expect(0, 8237, '\p{Bidi_Class=	POP_DIRECTIONAL_Format}', "");
Expect(1, 8237, '\p{^Bidi_Class=	POP_DIRECTIONAL_Format}', "");
Expect(1, 8237, '\P{Bidi_Class=	POP_DIRECTIONAL_Format}', "");
Expect(0, 8237, '\P{^Bidi_Class=	POP_DIRECTIONAL_Format}', "");
Error('\p{Bc:	 /a/PDF}');
Error('\P{Bc:	 /a/PDF}');
Expect(1, 8236, '\p{Bc=pdf}', "");
Expect(0, 8236, '\p{^Bc=pdf}', "");
Expect(0, 8236, '\P{Bc=pdf}', "");
Expect(1, 8236, '\P{^Bc=pdf}', "");
Expect(0, 8237, '\p{Bc=pdf}', "");
Expect(1, 8237, '\p{^Bc=pdf}', "");
Expect(1, 8237, '\P{Bc=pdf}', "");
Expect(0, 8237, '\P{^Bc=pdf}', "");
Expect(1, 8236, '\p{Bc= PDF}', "");
Expect(0, 8236, '\p{^Bc= PDF}', "");
Expect(0, 8236, '\P{Bc= PDF}', "");
Expect(1, 8236, '\P{^Bc= PDF}', "");
Expect(0, 8237, '\p{Bc= PDF}', "");
Expect(1, 8237, '\p{^Bc= PDF}', "");
Expect(1, 8237, '\P{Bc= PDF}', "");
Expect(0, 8237, '\P{^Bc= PDF}', "");
Error('\p{Is_Bidi_Class=_Pop_Directional_Format/a/}');
Error('\P{Is_Bidi_Class=_Pop_Directional_Format/a/}');
Expect(1, 8236, '\p{Is_Bidi_Class=popdirectionalformat}', "");
Expect(0, 8236, '\p{^Is_Bidi_Class=popdirectionalformat}', "");
Expect(0, 8236, '\P{Is_Bidi_Class=popdirectionalformat}', "");
Expect(1, 8236, '\P{^Is_Bidi_Class=popdirectionalformat}', "");
Expect(0, 8237, '\p{Is_Bidi_Class=popdirectionalformat}', "");
Expect(1, 8237, '\p{^Is_Bidi_Class=popdirectionalformat}', "");
Expect(1, 8237, '\P{Is_Bidi_Class=popdirectionalformat}', "");
Expect(0, 8237, '\P{^Is_Bidi_Class=popdirectionalformat}', "");
Expect(1, 8236, '\p{Is_Bidi_Class=-_pop_DIRECTIONAL_Format}', "");
Expect(0, 8236, '\p{^Is_Bidi_Class=-_pop_DIRECTIONAL_Format}', "");
Expect(0, 8236, '\P{Is_Bidi_Class=-_pop_DIRECTIONAL_Format}', "");
Expect(1, 8236, '\P{^Is_Bidi_Class=-_pop_DIRECTIONAL_Format}', "");
Expect(0, 8237, '\p{Is_Bidi_Class=-_pop_DIRECTIONAL_Format}', "");
Expect(1, 8237, '\p{^Is_Bidi_Class=-_pop_DIRECTIONAL_Format}', "");
Expect(1, 8237, '\P{Is_Bidi_Class=-_pop_DIRECTIONAL_Format}', "");
Expect(0, 8237, '\P{^Is_Bidi_Class=-_pop_DIRECTIONAL_Format}', "");
Error('\p{Is_Bc=	pdf:=}');
Error('\P{Is_Bc=	pdf:=}');
Expect(1, 8236, '\p{Is_Bc=pdf}', "");
Expect(0, 8236, '\p{^Is_Bc=pdf}', "");
Expect(0, 8236, '\P{Is_Bc=pdf}', "");
Expect(1, 8236, '\P{^Is_Bc=pdf}', "");
Expect(0, 8237, '\p{Is_Bc=pdf}', "");
Expect(1, 8237, '\p{^Is_Bc=pdf}', "");
Expect(1, 8237, '\P{Is_Bc=pdf}', "");
Expect(0, 8237, '\P{^Is_Bc=pdf}', "");
Expect(1, 8236, '\p{Is_Bc=__PDF}', "");
Expect(0, 8236, '\p{^Is_Bc=__PDF}', "");
Expect(0, 8236, '\P{Is_Bc=__PDF}', "");
Expect(1, 8236, '\P{^Is_Bc=__PDF}', "");
Expect(0, 8237, '\p{Is_Bc=__PDF}', "");
Expect(1, 8237, '\p{^Is_Bc=__PDF}', "");
Expect(1, 8237, '\P{Is_Bc=__PDF}', "");
Expect(0, 8237, '\P{^Is_Bc=__PDF}', "");
Error('\p{Bidi_Class=	-Pop_Directional_Isolate/a/}');
Error('\P{Bidi_Class=	-Pop_Directional_Isolate/a/}');
Expect(1, 8297, '\p{Bidi_Class=popdirectionalisolate}', "");
Expect(0, 8297, '\p{^Bidi_Class=popdirectionalisolate}', "");
Expect(0, 8297, '\P{Bidi_Class=popdirectionalisolate}', "");
Expect(1, 8297, '\P{^Bidi_Class=popdirectionalisolate}', "");
Expect(0, 8298, '\p{Bidi_Class=popdirectionalisolate}', "");
Expect(1, 8298, '\p{^Bidi_Class=popdirectionalisolate}', "");
Expect(1, 8298, '\P{Bidi_Class=popdirectionalisolate}', "");
Expect(0, 8298, '\P{^Bidi_Class=popdirectionalisolate}', "");
Expect(1, 8297, '\p{Bidi_Class=	POP_Directional_Isolate}', "");
Expect(0, 8297, '\p{^Bidi_Class=	POP_Directional_Isolate}', "");
Expect(0, 8297, '\P{Bidi_Class=	POP_Directional_Isolate}', "");
Expect(1, 8297, '\P{^Bidi_Class=	POP_Directional_Isolate}', "");
Expect(0, 8298, '\p{Bidi_Class=	POP_Directional_Isolate}', "");
Expect(1, 8298, '\p{^Bidi_Class=	POP_Directional_Isolate}', "");
Expect(1, 8298, '\P{Bidi_Class=	POP_Directional_Isolate}', "");
Expect(0, 8298, '\P{^Bidi_Class=	POP_Directional_Isolate}', "");
Error('\p{Bc=-:=PDI}');
Error('\P{Bc=-:=PDI}');
Expect(1, 8297, '\p{Bc=pdi}', "");
Expect(0, 8297, '\p{^Bc=pdi}', "");
Expect(0, 8297, '\P{Bc=pdi}', "");
Expect(1, 8297, '\P{^Bc=pdi}', "");
Expect(0, 8298, '\p{Bc=pdi}', "");
Expect(1, 8298, '\p{^Bc=pdi}', "");
Expect(1, 8298, '\P{Bc=pdi}', "");
Expect(0, 8298, '\P{^Bc=pdi}', "");
Expect(1, 8297, '\p{Bc=- PDI}', "");
Expect(0, 8297, '\p{^Bc=- PDI}', "");
Expect(0, 8297, '\P{Bc=- PDI}', "");
Expect(1, 8297, '\P{^Bc=- PDI}', "");
Expect(0, 8298, '\p{Bc=- PDI}', "");
Expect(1, 8298, '\p{^Bc=- PDI}', "");
Expect(1, 8298, '\P{Bc=- PDI}', "");
Expect(0, 8298, '\P{^Bc=- PDI}', "");
Error('\p{Is_Bidi_Class=:=-Pop_Directional_isolate}');
Error('\P{Is_Bidi_Class=:=-Pop_Directional_isolate}');
Expect(1, 8297, '\p{Is_Bidi_Class:   popdirectionalisolate}', "");
Expect(0, 8297, '\p{^Is_Bidi_Class:   popdirectionalisolate}', "");
Expect(0, 8297, '\P{Is_Bidi_Class:   popdirectionalisolate}', "");
Expect(1, 8297, '\P{^Is_Bidi_Class:   popdirectionalisolate}', "");
Expect(0, 8298, '\p{Is_Bidi_Class:   popdirectionalisolate}', "");
Expect(1, 8298, '\p{^Is_Bidi_Class:   popdirectionalisolate}', "");
Expect(1, 8298, '\P{Is_Bidi_Class:   popdirectionalisolate}', "");
Expect(0, 8298, '\P{^Is_Bidi_Class:   popdirectionalisolate}', "");
Expect(1, 8297, '\p{Is_Bidi_Class=- pop_DIRECTIONAL_Isolate}', "");
Expect(0, 8297, '\p{^Is_Bidi_Class=- pop_DIRECTIONAL_Isolate}', "");
Expect(0, 8297, '\P{Is_Bidi_Class=- pop_DIRECTIONAL_Isolate}', "");
Expect(1, 8297, '\P{^Is_Bidi_Class=- pop_DIRECTIONAL_Isolate}', "");
Expect(0, 8298, '\p{Is_Bidi_Class=- pop_DIRECTIONAL_Isolate}', "");
Expect(1, 8298, '\p{^Is_Bidi_Class=- pop_DIRECTIONAL_Isolate}', "");
Expect(1, 8298, '\P{Is_Bidi_Class=- pop_DIRECTIONAL_Isolate}', "");
Expect(0, 8298, '\P{^Is_Bidi_Class=- pop_DIRECTIONAL_Isolate}', "");
Error('\p{Is_Bc:	/a/ _PDI}');
Error('\P{Is_Bc:	/a/ _PDI}');
Expect(1, 8297, '\p{Is_Bc=pdi}', "");
Expect(0, 8297, '\p{^Is_Bc=pdi}', "");
Expect(0, 8297, '\P{Is_Bc=pdi}', "");
Expect(1, 8297, '\P{^Is_Bc=pdi}', "");
Expect(0, 8298, '\p{Is_Bc=pdi}', "");
Expect(1, 8298, '\p{^Is_Bc=pdi}', "");
Expect(1, 8298, '\P{Is_Bc=pdi}', "");
Expect(0, 8298, '\P{^Is_Bc=pdi}', "");
Expect(1, 8297, '\p{Is_Bc= _PDI}', "");
Expect(0, 8297, '\p{^Is_Bc= _PDI}', "");
Expect(0, 8297, '\P{Is_Bc= _PDI}', "");
Expect(1, 8297, '\P{^Is_Bc= _PDI}', "");
Expect(0, 8298, '\p{Is_Bc= _PDI}', "");
Expect(1, 8298, '\p{^Is_Bc= _PDI}', "");
Expect(1, 8298, '\P{Is_Bc= _PDI}', "");
Expect(0, 8298, '\P{^Is_Bc= _PDI}', "");
Error('\p{Bidi_Class=:= 	right_To_left}');
Error('\P{Bidi_Class=:= 	right_To_left}');
Expect(1, 126975, '\p{Bidi_Class=righttoleft}', "");
Expect(0, 126975, '\p{^Bidi_Class=righttoleft}', "");
Expect(0, 126975, '\P{Bidi_Class=righttoleft}', "");
Expect(1, 126975, '\P{^Bidi_Class=righttoleft}', "");
Expect(0, 126976, '\p{Bidi_Class=righttoleft}', "");
Expect(1, 126976, '\p{^Bidi_Class=righttoleft}', "");
Expect(1, 126976, '\P{Bidi_Class=righttoleft}', "");
Expect(0, 126976, '\P{^Bidi_Class=righttoleft}', "");
Expect(1, 126975, '\p{Bidi_Class=-Right_to_Left}', "");
Expect(0, 126975, '\p{^Bidi_Class=-Right_to_Left}', "");
Expect(0, 126975, '\P{Bidi_Class=-Right_to_Left}', "");
Expect(1, 126975, '\P{^Bidi_Class=-Right_to_Left}', "");
Expect(0, 126976, '\p{Bidi_Class=-Right_to_Left}', "");
Expect(1, 126976, '\p{^Bidi_Class=-Right_to_Left}', "");
Expect(1, 126976, '\P{Bidi_Class=-Right_to_Left}', "");
Expect(0, 126976, '\P{^Bidi_Class=-Right_to_Left}', "");
Error('\p{Bc:  R/a/}');
Error('\P{Bc:  R/a/}');
Expect(1, 126975, '\p{Bc=r}', "");
Expect(0, 126975, '\p{^Bc=r}', "");
Expect(0, 126975, '\P{Bc=r}', "");
Expect(1, 126975, '\P{^Bc=r}', "");
Expect(0, 126976, '\p{Bc=r}', "");
Expect(1, 126976, '\p{^Bc=r}', "");
Expect(1, 126976, '\P{Bc=r}', "");
Expect(0, 126976, '\P{^Bc=r}', "");
Expect(1, 126975, '\p{Bc=_R}', "");
Expect(0, 126975, '\p{^Bc=_R}', "");
Expect(0, 126975, '\P{Bc=_R}', "");
Expect(1, 126975, '\P{^Bc=_R}', "");
Expect(0, 126976, '\p{Bc=_R}', "");
Expect(1, 126976, '\p{^Bc=_R}', "");
Expect(1, 126976, '\P{Bc=_R}', "");
Expect(0, 126976, '\P{^Bc=_R}', "");
Error('\p{Is_Bidi_Class=__RIGHT_To_left/a/}');
Error('\P{Is_Bidi_Class=__RIGHT_To_left/a/}');
Expect(1, 126975, '\p{Is_Bidi_Class=righttoleft}', "");
Expect(0, 126975, '\p{^Is_Bidi_Class=righttoleft}', "");
Expect(0, 126975, '\P{Is_Bidi_Class=righttoleft}', "");
Expect(1, 126975, '\P{^Is_Bidi_Class=righttoleft}', "");
Expect(0, 126976, '\p{Is_Bidi_Class=righttoleft}', "");
Expect(1, 126976, '\p{^Is_Bidi_Class=righttoleft}', "");
Expect(1, 126976, '\P{Is_Bidi_Class=righttoleft}', "");
Expect(0, 126976, '\P{^Is_Bidi_Class=righttoleft}', "");
Expect(1, 126975, '\p{Is_Bidi_Class=		Right_to_left}', "");
Expect(0, 126975, '\p{^Is_Bidi_Class=		Right_to_left}', "");
Expect(0, 126975, '\P{Is_Bidi_Class=		Right_to_left}', "");
Expect(1, 126975, '\P{^Is_Bidi_Class=		Right_to_left}', "");
Expect(0, 126976, '\p{Is_Bidi_Class=		Right_to_left}', "");
Expect(1, 126976, '\p{^Is_Bidi_Class=		Right_to_left}', "");
Expect(1, 126976, '\P{Is_Bidi_Class=		Right_to_left}', "");
Expect(0, 126976, '\P{^Is_Bidi_Class=		Right_to_left}', "");
Error('\p{Is_Bc=	/a/R}');
Error('\P{Is_Bc=	/a/R}');
Expect(1, 126975, '\p{Is_Bc=r}', "");
Expect(0, 126975, '\p{^Is_Bc=r}', "");
Expect(0, 126975, '\P{Is_Bc=r}', "");
Expect(1, 126975, '\P{^Is_Bc=r}', "");
Expect(0, 126976, '\p{Is_Bc=r}', "");
Expect(1, 126976, '\p{^Is_Bc=r}', "");
Expect(1, 126976, '\P{Is_Bc=r}', "");
Expect(0, 126976, '\P{^Is_Bc=r}', "");
Expect(1, 126975, '\p{Is_Bc=-	R}', "");
Expect(0, 126975, '\p{^Is_Bc=-	R}', "");
Expect(0, 126975, '\P{Is_Bc=-	R}', "");
Expect(1, 126975, '\P{^Is_Bc=-	R}', "");
Expect(0, 126976, '\p{Is_Bc=-	R}', "");
Expect(1, 126976, '\p{^Is_Bc=-	R}', "");
Expect(1, 126976, '\P{Is_Bc=-	R}', "");
Expect(0, 126976, '\P{^Is_Bc=-	R}', "");
Error('\p{Bidi_Class=  Right_To_Left_Embedding/a/}');
Error('\P{Bidi_Class=  Right_To_Left_Embedding/a/}');
Expect(1, 8235, '\p{Bidi_Class=righttoleftembedding}', "");
Expect(0, 8235, '\p{^Bidi_Class=righttoleftembedding}', "");
Expect(0, 8235, '\P{Bidi_Class=righttoleftembedding}', "");
Expect(1, 8235, '\P{^Bidi_Class=righttoleftembedding}', "");
Expect(0, 8236, '\p{Bidi_Class=righttoleftembedding}', "");
Expect(1, 8236, '\p{^Bidi_Class=righttoleftembedding}', "");
Expect(1, 8236, '\P{Bidi_Class=righttoleftembedding}', "");
Expect(0, 8236, '\P{^Bidi_Class=righttoleftembedding}', "");
Expect(1, 8235, '\p{Bidi_Class=_ Right_TO_left_Embedding}', "");
Expect(0, 8235, '\p{^Bidi_Class=_ Right_TO_left_Embedding}', "");
Expect(0, 8235, '\P{Bidi_Class=_ Right_TO_left_Embedding}', "");
Expect(1, 8235, '\P{^Bidi_Class=_ Right_TO_left_Embedding}', "");
Expect(0, 8236, '\p{Bidi_Class=_ Right_TO_left_Embedding}', "");
Expect(1, 8236, '\p{^Bidi_Class=_ Right_TO_left_Embedding}', "");
Expect(1, 8236, '\P{Bidi_Class=_ Right_TO_left_Embedding}', "");
Expect(0, 8236, '\P{^Bidi_Class=_ Right_TO_left_Embedding}', "");
Error('\p{Bc=/a/__RLE}');
Error('\P{Bc=/a/__RLE}');
Expect(1, 8235, '\p{Bc=rle}', "");
Expect(0, 8235, '\p{^Bc=rle}', "");
Expect(0, 8235, '\P{Bc=rle}', "");
Expect(1, 8235, '\P{^Bc=rle}', "");
Expect(0, 8236, '\p{Bc=rle}', "");
Expect(1, 8236, '\p{^Bc=rle}', "");
Expect(1, 8236, '\P{Bc=rle}', "");
Expect(0, 8236, '\P{^Bc=rle}', "");
Expect(1, 8235, '\p{Bc=__RLE}', "");
Expect(0, 8235, '\p{^Bc=__RLE}', "");
Expect(0, 8235, '\P{Bc=__RLE}', "");
Expect(1, 8235, '\P{^Bc=__RLE}', "");
Expect(0, 8236, '\p{Bc=__RLE}', "");
Expect(1, 8236, '\p{^Bc=__RLE}', "");
Expect(1, 8236, '\P{Bc=__RLE}', "");
Expect(0, 8236, '\P{^Bc=__RLE}', "");
Error('\p{Is_Bidi_Class=:=-	right_TO_left_embedding}');
Error('\P{Is_Bidi_Class=:=-	right_TO_left_embedding}');
Expect(1, 8235, '\p{Is_Bidi_Class=righttoleftembedding}', "");
Expect(0, 8235, '\p{^Is_Bidi_Class=righttoleftembedding}', "");
Expect(0, 8235, '\P{Is_Bidi_Class=righttoleftembedding}', "");
Expect(1, 8235, '\P{^Is_Bidi_Class=righttoleftembedding}', "");
Expect(0, 8236, '\p{Is_Bidi_Class=righttoleftembedding}', "");
Expect(1, 8236, '\p{^Is_Bidi_Class=righttoleftembedding}', "");
Expect(1, 8236, '\P{Is_Bidi_Class=righttoleftembedding}', "");
Expect(0, 8236, '\P{^Is_Bidi_Class=righttoleftembedding}', "");
Expect(1, 8235, '\p{Is_Bidi_Class= RIGHT_to_Left_Embedding}', "");
Expect(0, 8235, '\p{^Is_Bidi_Class= RIGHT_to_Left_Embedding}', "");
Expect(0, 8235, '\P{Is_Bidi_Class= RIGHT_to_Left_Embedding}', "");
Expect(1, 8235, '\P{^Is_Bidi_Class= RIGHT_to_Left_Embedding}', "");
Expect(0, 8236, '\p{Is_Bidi_Class= RIGHT_to_Left_Embedding}', "");
Expect(1, 8236, '\p{^Is_Bidi_Class= RIGHT_to_Left_Embedding}', "");
Expect(1, 8236, '\P{Is_Bidi_Class= RIGHT_to_Left_Embedding}', "");
Expect(0, 8236, '\P{^Is_Bidi_Class= RIGHT_to_Left_Embedding}', "");
Error('\p{Is_Bc=_:=RLE}');
Error('\P{Is_Bc=_:=RLE}');
Expect(1, 8235, '\p{Is_Bc=rle}', "");
Expect(0, 8235, '\p{^Is_Bc=rle}', "");
Expect(0, 8235, '\P{Is_Bc=rle}', "");
Expect(1, 8235, '\P{^Is_Bc=rle}', "");
Expect(0, 8236, '\p{Is_Bc=rle}', "");
Expect(1, 8236, '\p{^Is_Bc=rle}', "");
Expect(1, 8236, '\P{Is_Bc=rle}', "");
Expect(0, 8236, '\P{^Is_Bc=rle}', "");
Expect(1, 8235, '\p{Is_Bc: -rle}', "");
Expect(0, 8235, '\p{^Is_Bc: -rle}', "");
Expect(0, 8235, '\P{Is_Bc: -rle}', "");
Expect(1, 8235, '\P{^Is_Bc: -rle}', "");
Expect(0, 8236, '\p{Is_Bc: -rle}', "");
Expect(1, 8236, '\p{^Is_Bc: -rle}', "");
Expect(1, 8236, '\P{Is_Bc: -rle}', "");
Expect(0, 8236, '\P{^Is_Bc: -rle}', "");
Error('\p{Bidi_Class=	:=Right_to_LEFT_Isolate}');
Error('\P{Bidi_Class=	:=Right_to_LEFT_Isolate}');
Expect(1, 8295, '\p{Bidi_Class=righttoleftisolate}', "");
Expect(0, 8295, '\p{^Bidi_Class=righttoleftisolate}', "");
Expect(0, 8295, '\P{Bidi_Class=righttoleftisolate}', "");
Expect(1, 8295, '\P{^Bidi_Class=righttoleftisolate}', "");
Expect(0, 8296, '\p{Bidi_Class=righttoleftisolate}', "");
Expect(1, 8296, '\p{^Bidi_Class=righttoleftisolate}', "");
Expect(1, 8296, '\P{Bidi_Class=righttoleftisolate}', "");
Expect(0, 8296, '\P{^Bidi_Class=righttoleftisolate}', "");
Expect(1, 8295, '\p{Bidi_Class= -right_to_left_Isolate}', "");
Expect(0, 8295, '\p{^Bidi_Class= -right_to_left_Isolate}', "");
Expect(0, 8295, '\P{Bidi_Class= -right_to_left_Isolate}', "");
Expect(1, 8295, '\P{^Bidi_Class= -right_to_left_Isolate}', "");
Expect(0, 8296, '\p{Bidi_Class= -right_to_left_Isolate}', "");
Expect(1, 8296, '\p{^Bidi_Class= -right_to_left_Isolate}', "");
Expect(1, 8296, '\P{Bidi_Class= -right_to_left_Isolate}', "");
Expect(0, 8296, '\P{^Bidi_Class= -right_to_left_Isolate}', "");
Error('\p{Bc=/a/ _RLI}');
Error('\P{Bc=/a/ _RLI}');
Expect(1, 8295, '\p{Bc=rli}', "");
Expect(0, 8295, '\p{^Bc=rli}', "");
Expect(0, 8295, '\P{Bc=rli}', "");
Expect(1, 8295, '\P{^Bc=rli}', "");
Expect(0, 8296, '\p{Bc=rli}', "");
Expect(1, 8296, '\p{^Bc=rli}', "");
Expect(1, 8296, '\P{Bc=rli}', "");
Expect(0, 8296, '\P{^Bc=rli}', "");
Expect(1, 8295, '\p{Bc=_ rli}', "");
Expect(0, 8295, '\p{^Bc=_ rli}', "");
Expect(0, 8295, '\P{Bc=_ rli}', "");
Expect(1, 8295, '\P{^Bc=_ rli}', "");
Expect(0, 8296, '\p{Bc=_ rli}', "");
Expect(1, 8296, '\p{^Bc=_ rli}', "");
Expect(1, 8296, '\P{Bc=_ rli}', "");
Expect(0, 8296, '\P{^Bc=_ rli}', "");
Error('\p{Is_Bidi_Class:   :=-Right_TO_left_ISOLATE}');
Error('\P{Is_Bidi_Class:   :=-Right_TO_left_ISOLATE}');
Expect(1, 8295, '\p{Is_Bidi_Class: righttoleftisolate}', "");
Expect(0, 8295, '\p{^Is_Bidi_Class: righttoleftisolate}', "");
Expect(0, 8295, '\P{Is_Bidi_Class: righttoleftisolate}', "");
Expect(1, 8295, '\P{^Is_Bidi_Class: righttoleftisolate}', "");
Expect(0, 8296, '\p{Is_Bidi_Class: righttoleftisolate}', "");
Expect(1, 8296, '\p{^Is_Bidi_Class: righttoleftisolate}', "");
Expect(1, 8296, '\P{Is_Bidi_Class: righttoleftisolate}', "");
Expect(0, 8296, '\P{^Is_Bidi_Class: righttoleftisolate}', "");
Expect(1, 8295, '\p{Is_Bidi_Class:   Right_TO_Left_ISOLATE}', "");
Expect(0, 8295, '\p{^Is_Bidi_Class:   Right_TO_Left_ISOLATE}', "");
Expect(0, 8295, '\P{Is_Bidi_Class:   Right_TO_Left_ISOLATE}', "");
Expect(1, 8295, '\P{^Is_Bidi_Class:   Right_TO_Left_ISOLATE}', "");
Expect(0, 8296, '\p{Is_Bidi_Class:   Right_TO_Left_ISOLATE}', "");
Expect(1, 8296, '\p{^Is_Bidi_Class:   Right_TO_Left_ISOLATE}', "");
Expect(1, 8296, '\P{Is_Bidi_Class:   Right_TO_Left_ISOLATE}', "");
Expect(0, 8296, '\P{^Is_Bidi_Class:   Right_TO_Left_ISOLATE}', "");
Error('\p{Is_Bc=/a/ rli}');
Error('\P{Is_Bc=/a/ rli}');
Expect(1, 8295, '\p{Is_Bc=rli}', "");
Expect(0, 8295, '\p{^Is_Bc=rli}', "");
Expect(0, 8295, '\P{Is_Bc=rli}', "");
Expect(1, 8295, '\P{^Is_Bc=rli}', "");
Expect(0, 8296, '\p{Is_Bc=rli}', "");
Expect(1, 8296, '\p{^Is_Bc=rli}', "");
Expect(1, 8296, '\P{Is_Bc=rli}', "");
Expect(0, 8296, '\P{^Is_Bc=rli}', "");
Expect(1, 8295, '\p{Is_Bc=_-RLI}', "");
Expect(0, 8295, '\p{^Is_Bc=_-RLI}', "");
Expect(0, 8295, '\P{Is_Bc=_-RLI}', "");
Expect(1, 8295, '\P{^Is_Bc=_-RLI}', "");
Expect(0, 8296, '\p{Is_Bc=_-RLI}', "");
Expect(1, 8296, '\p{^Is_Bc=_-RLI}', "");
Expect(1, 8296, '\P{Is_Bc=_-RLI}', "");
Expect(0, 8296, '\P{^Is_Bc=_-RLI}', "");
Error('\p{Bidi_Class=-Right_To_left_override/a/}');
Error('\P{Bidi_Class=-Right_To_left_override/a/}');
Expect(1, 8238, '\p{Bidi_Class=righttoleftoverride}', "");
Expect(0, 8238, '\p{^Bidi_Class=righttoleftoverride}', "");
Expect(0, 8238, '\P{Bidi_Class=righttoleftoverride}', "");
Expect(1, 8238, '\P{^Bidi_Class=righttoleftoverride}', "");
Expect(0, 8239, '\p{Bidi_Class=righttoleftoverride}', "");
Expect(1, 8239, '\p{^Bidi_Class=righttoleftoverride}', "");
Expect(1, 8239, '\P{Bidi_Class=righttoleftoverride}', "");
Expect(0, 8239, '\P{^Bidi_Class=righttoleftoverride}', "");
Expect(1, 8238, '\p{Bidi_Class=_	right_To_Left_Override}', "");
Expect(0, 8238, '\p{^Bidi_Class=_	right_To_Left_Override}', "");
Expect(0, 8238, '\P{Bidi_Class=_	right_To_Left_Override}', "");
Expect(1, 8238, '\P{^Bidi_Class=_	right_To_Left_Override}', "");
Expect(0, 8239, '\p{Bidi_Class=_	right_To_Left_Override}', "");
Expect(1, 8239, '\p{^Bidi_Class=_	right_To_Left_Override}', "");
Expect(1, 8239, '\P{Bidi_Class=_	right_To_Left_Override}', "");
Expect(0, 8239, '\P{^Bidi_Class=_	right_To_Left_Override}', "");
Error('\p{Bc=-	RLO/a/}');
Error('\P{Bc=-	RLO/a/}');
Expect(1, 8238, '\p{Bc=rlo}', "");
Expect(0, 8238, '\p{^Bc=rlo}', "");
Expect(0, 8238, '\P{Bc=rlo}', "");
Expect(1, 8238, '\P{^Bc=rlo}', "");
Expect(0, 8239, '\p{Bc=rlo}', "");
Expect(1, 8239, '\p{^Bc=rlo}', "");
Expect(1, 8239, '\P{Bc=rlo}', "");
Expect(0, 8239, '\P{^Bc=rlo}', "");
Expect(1, 8238, '\p{Bc=__RLO}', "");
Expect(0, 8238, '\p{^Bc=__RLO}', "");
Expect(0, 8238, '\P{Bc=__RLO}', "");
Expect(1, 8238, '\P{^Bc=__RLO}', "");
Expect(0, 8239, '\p{Bc=__RLO}', "");
Expect(1, 8239, '\p{^Bc=__RLO}', "");
Expect(1, 8239, '\P{Bc=__RLO}', "");
Expect(0, 8239, '\P{^Bc=__RLO}', "");
Error('\p{Is_Bidi_Class=/a/--Right_To_Left_override}');
Error('\P{Is_Bidi_Class=/a/--Right_To_Left_override}');
Expect(1, 8238, '\p{Is_Bidi_Class=righttoleftoverride}', "");
Expect(0, 8238, '\p{^Is_Bidi_Class=righttoleftoverride}', "");
Expect(0, 8238, '\P{Is_Bidi_Class=righttoleftoverride}', "");
Expect(1, 8238, '\P{^Is_Bidi_Class=righttoleftoverride}', "");
Expect(0, 8239, '\p{Is_Bidi_Class=righttoleftoverride}', "");
Expect(1, 8239, '\p{^Is_Bidi_Class=righttoleftoverride}', "");
Expect(1, 8239, '\P{Is_Bidi_Class=righttoleftoverride}', "");
Expect(0, 8239, '\P{^Is_Bidi_Class=righttoleftoverride}', "");
Expect(1, 8238, '\p{Is_Bidi_Class= _Right_to_left_override}', "");
Expect(0, 8238, '\p{^Is_Bidi_Class= _Right_to_left_override}', "");
Expect(0, 8238, '\P{Is_Bidi_Class= _Right_to_left_override}', "");
Expect(1, 8238, '\P{^Is_Bidi_Class= _Right_to_left_override}', "");
Expect(0, 8239, '\p{Is_Bidi_Class= _Right_to_left_override}', "");
Expect(1, 8239, '\p{^Is_Bidi_Class= _Right_to_left_override}', "");
Expect(1, 8239, '\P{Is_Bidi_Class= _Right_to_left_override}', "");
Expect(0, 8239, '\P{^Is_Bidi_Class= _Right_to_left_override}', "");
Error('\p{Is_Bc=:=_-RLO}');
Error('\P{Is_Bc=:=_-RLO}');
Expect(1, 8238, '\p{Is_Bc=rlo}', "");
Expect(0, 8238, '\p{^Is_Bc=rlo}', "");
Expect(0, 8238, '\P{Is_Bc=rlo}', "");
Expect(1, 8238, '\P{^Is_Bc=rlo}', "");
Expect(0, 8239, '\p{Is_Bc=rlo}', "");
Expect(1, 8239, '\p{^Is_Bc=rlo}', "");
Expect(1, 8239, '\P{Is_Bc=rlo}', "");
Expect(0, 8239, '\P{^Is_Bc=rlo}', "");
Expect(1, 8238, '\p{Is_Bc= 	rlo}', "");
Expect(0, 8238, '\p{^Is_Bc= 	rlo}', "");
Expect(0, 8238, '\P{Is_Bc= 	rlo}', "");
Expect(1, 8238, '\P{^Is_Bc= 	rlo}', "");
Expect(0, 8239, '\p{Is_Bc= 	rlo}', "");
Expect(1, 8239, '\p{^Is_Bc= 	rlo}', "");
Expect(1, 8239, '\P{Is_Bc= 	rlo}', "");
Expect(0, 8239, '\P{^Is_Bc= 	rlo}', "");
Error('\p{Bidi_Class=	segment_Separator:=}');
Error('\P{Bidi_Class=	segment_Separator:=}');
Expect(1, 31, '\p{Bidi_Class: segmentseparator}', "");
Expect(0, 31, '\p{^Bidi_Class: segmentseparator}', "");
Expect(0, 31, '\P{Bidi_Class: segmentseparator}', "");
Expect(1, 31, '\P{^Bidi_Class: segmentseparator}', "");
Expect(0, 32, '\p{Bidi_Class: segmentseparator}', "");
Expect(1, 32, '\p{^Bidi_Class: segmentseparator}', "");
Expect(1, 32, '\P{Bidi_Class: segmentseparator}', "");
Expect(0, 32, '\P{^Bidi_Class: segmentseparator}', "");
Expect(1, 31, '\p{Bidi_Class=	Segment_Separator}', "");
Expect(0, 31, '\p{^Bidi_Class=	Segment_Separator}', "");
Expect(0, 31, '\P{Bidi_Class=	Segment_Separator}', "");
Expect(1, 31, '\P{^Bidi_Class=	Segment_Separator}', "");
Expect(0, 32, '\p{Bidi_Class=	Segment_Separator}', "");
Expect(1, 32, '\p{^Bidi_Class=	Segment_Separator}', "");
Expect(1, 32, '\P{Bidi_Class=	Segment_Separator}', "");
Expect(0, 32, '\P{^Bidi_Class=	Segment_Separator}', "");
Error('\p{Bc=	-S/a/}');
Error('\P{Bc=	-S/a/}');
Expect(1, 31, '\p{Bc=s}', "");
Expect(0, 31, '\p{^Bc=s}', "");
Expect(0, 31, '\P{Bc=s}', "");
Expect(1, 31, '\P{^Bc=s}', "");
Expect(0, 32, '\p{Bc=s}', "");
Expect(1, 32, '\p{^Bc=s}', "");
Expect(1, 32, '\P{Bc=s}', "");
Expect(0, 32, '\P{^Bc=s}', "");
Expect(1, 31, '\p{Bc:__s}', "");
Expect(0, 31, '\p{^Bc:__s}', "");
Expect(0, 31, '\P{Bc:__s}', "");
Expect(1, 31, '\P{^Bc:__s}', "");
Expect(0, 32, '\p{Bc:__s}', "");
Expect(1, 32, '\p{^Bc:__s}', "");
Expect(1, 32, '\P{Bc:__s}', "");
Expect(0, 32, '\P{^Bc:__s}', "");
Error('\p{Is_Bidi_Class=Segment_Separator/a/}');
Error('\P{Is_Bidi_Class=Segment_Separator/a/}');
Expect(1, 31, '\p{Is_Bidi_Class=segmentseparator}', "");
Expect(0, 31, '\p{^Is_Bidi_Class=segmentseparator}', "");
Expect(0, 31, '\P{Is_Bidi_Class=segmentseparator}', "");
Expect(1, 31, '\P{^Is_Bidi_Class=segmentseparator}', "");
Expect(0, 32, '\p{Is_Bidi_Class=segmentseparator}', "");
Expect(1, 32, '\p{^Is_Bidi_Class=segmentseparator}', "");
Expect(1, 32, '\P{Is_Bidi_Class=segmentseparator}', "");
Expect(0, 32, '\P{^Is_Bidi_Class=segmentseparator}', "");
Expect(1, 31, '\p{Is_Bidi_Class=  Segment_Separator}', "");
Expect(0, 31, '\p{^Is_Bidi_Class=  Segment_Separator}', "");
Expect(0, 31, '\P{Is_Bidi_Class=  Segment_Separator}', "");
Expect(1, 31, '\P{^Is_Bidi_Class=  Segment_Separator}', "");
Expect(0, 32, '\p{Is_Bidi_Class=  Segment_Separator}', "");
Expect(1, 32, '\p{^Is_Bidi_Class=  Segment_Separator}', "");
Expect(1, 32, '\P{Is_Bidi_Class=  Segment_Separator}', "");
Expect(0, 32, '\P{^Is_Bidi_Class=  Segment_Separator}', "");
Error('\p{Is_Bc=-/a/S}');
Error('\P{Is_Bc=-/a/S}');
Expect(1, 31, '\p{Is_Bc=s}', "");
Expect(0, 31, '\p{^Is_Bc=s}', "");
Expect(0, 31, '\P{Is_Bc=s}', "");
Expect(1, 31, '\P{^Is_Bc=s}', "");
Expect(0, 32, '\p{Is_Bc=s}', "");
Expect(1, 32, '\p{^Is_Bc=s}', "");
Expect(1, 32, '\P{Is_Bc=s}', "");
Expect(0, 32, '\P{^Is_Bc=s}', "");
Expect(1, 31, '\p{Is_Bc=-	S}', "");
Expect(0, 31, '\p{^Is_Bc=-	S}', "");
Expect(0, 31, '\P{Is_Bc=-	S}', "");
Expect(1, 31, '\P{^Is_Bc=-	S}', "");
Expect(0, 32, '\p{Is_Bc=-	S}', "");
Expect(1, 32, '\p{^Is_Bc=-	S}', "");
Expect(1, 32, '\P{Is_Bc=-	S}', "");
Expect(0, 32, '\P{^Is_Bc=-	S}', "");
Error('\p{Bidi_Class= white_Space/a/}');
Error('\P{Bidi_Class= white_Space/a/}');
Expect(1, 12288, '\p{Bidi_Class=whitespace}', "");
Expect(0, 12288, '\p{^Bidi_Class=whitespace}', "");
Expect(0, 12288, '\P{Bidi_Class=whitespace}', "");
Expect(1, 12288, '\P{^Bidi_Class=whitespace}', "");
Expect(0, 12289, '\p{Bidi_Class=whitespace}', "");
Expect(1, 12289, '\p{^Bidi_Class=whitespace}', "");
Expect(1, 12289, '\P{Bidi_Class=whitespace}', "");
Expect(0, 12289, '\P{^Bidi_Class=whitespace}', "");
Expect(1, 12288, '\p{Bidi_Class:		_white_SPACE}', "");
Expect(0, 12288, '\p{^Bidi_Class:		_white_SPACE}', "");
Expect(0, 12288, '\P{Bidi_Class:		_white_SPACE}', "");
Expect(1, 12288, '\P{^Bidi_Class:		_white_SPACE}', "");
Expect(0, 12289, '\p{Bidi_Class:		_white_SPACE}', "");
Expect(1, 12289, '\p{^Bidi_Class:		_white_SPACE}', "");
Expect(1, 12289, '\P{Bidi_Class:		_white_SPACE}', "");
Expect(0, 12289, '\P{^Bidi_Class:		_white_SPACE}', "");
Error('\p{Bc=-:=WS}');
Error('\P{Bc=-:=WS}');
Expect(1, 12288, '\p{Bc=ws}', "");
Expect(0, 12288, '\p{^Bc=ws}', "");
Expect(0, 12288, '\P{Bc=ws}', "");
Expect(1, 12288, '\P{^Bc=ws}', "");
Expect(0, 12289, '\p{Bc=ws}', "");
Expect(1, 12289, '\p{^Bc=ws}', "");
Expect(1, 12289, '\P{Bc=ws}', "");
Expect(0, 12289, '\P{^Bc=ws}', "");
Expect(1, 12288, '\p{Bc=	_WS}', "");
Expect(0, 12288, '\p{^Bc=	_WS}', "");
Expect(0, 12288, '\P{Bc=	_WS}', "");
Expect(1, 12288, '\P{^Bc=	_WS}', "");
Expect(0, 12289, '\p{Bc=	_WS}', "");
Expect(1, 12289, '\p{^Bc=	_WS}', "");
Expect(1, 12289, '\P{Bc=	_WS}', "");
Expect(0, 12289, '\P{^Bc=	_WS}', "");
Error('\p{Is_Bidi_Class= 	White_SPACE/a/}');
Error('\P{Is_Bidi_Class= 	White_SPACE/a/}');
Expect(1, 12288, '\p{Is_Bidi_Class:	whitespace}', "");
Expect(0, 12288, '\p{^Is_Bidi_Class:	whitespace}', "");
Expect(0, 12288, '\P{Is_Bidi_Class:	whitespace}', "");
Expect(1, 12288, '\P{^Is_Bidi_Class:	whitespace}', "");
Expect(0, 12289, '\p{Is_Bidi_Class:	whitespace}', "");
Expect(1, 12289, '\p{^Is_Bidi_Class:	whitespace}', "");
Expect(1, 12289, '\P{Is_Bidi_Class:	whitespace}', "");
Expect(0, 12289, '\P{^Is_Bidi_Class:	whitespace}', "");
Expect(1, 12288, '\p{Is_Bidi_Class=- white_space}', "");
Expect(0, 12288, '\p{^Is_Bidi_Class=- white_space}', "");
Expect(0, 12288, '\P{Is_Bidi_Class=- white_space}', "");
Expect(1, 12288, '\P{^Is_Bidi_Class=- white_space}', "");
Expect(0, 12289, '\p{Is_Bidi_Class=- white_space}', "");
Expect(1, 12289, '\p{^Is_Bidi_Class=- white_space}', "");
Expect(1, 12289, '\P{Is_Bidi_Class=- white_space}', "");
Expect(0, 12289, '\P{^Is_Bidi_Class=- white_space}', "");
Error('\p{Is_Bc=  WS/a/}');
Error('\P{Is_Bc=  WS/a/}');
Expect(1, 12288, '\p{Is_Bc=ws}', "");
Expect(0, 12288, '\p{^Is_Bc=ws}', "");
Expect(0, 12288, '\P{Is_Bc=ws}', "");
Expect(1, 12288, '\P{^Is_Bc=ws}', "");
Expect(0, 12289, '\p{Is_Bc=ws}', "");
Expect(1, 12289, '\p{^Is_Bc=ws}', "");
Expect(1, 12289, '\P{Is_Bc=ws}', "");
Expect(0, 12289, '\P{^Is_Bc=ws}', "");
Expect(1, 12288, '\p{Is_Bc: WS}', "");
Expect(0, 12288, '\p{^Is_Bc: WS}', "");
Expect(0, 12288, '\P{Is_Bc: WS}', "");
Expect(1, 12288, '\P{^Is_Bc: WS}', "");
Expect(0, 12289, '\p{Is_Bc: WS}', "");
Expect(1, 12289, '\p{^Is_Bc: WS}', "");
Expect(1, 12289, '\P{Is_Bc: WS}', "");
Expect(0, 12289, '\P{^Is_Bc: WS}', "");
Error('\p{Block= 	Basic_latin:=}');
Error('\P{Block= 	Basic_latin:=}');
Expect(1, 127, '\p{Block=basiclatin}', "");
Expect(0, 127, '\p{^Block=basiclatin}', "");
Expect(0, 127, '\P{Block=basiclatin}', "");
Expect(1, 127, '\P{^Block=basiclatin}', "");
Expect(0, 128, '\p{Block=basiclatin}', "");
Expect(1, 128, '\p{^Block=basiclatin}', "");
Expect(1, 128, '\P{Block=basiclatin}', "");
Expect(0, 128, '\P{^Block=basiclatin}', "");
Expect(1, 127, '\p{Block= BASIC_Latin}', "");
Expect(0, 127, '\p{^Block= BASIC_Latin}', "");
Expect(0, 127, '\P{Block= BASIC_Latin}', "");
Expect(1, 127, '\P{^Block= BASIC_Latin}', "");
Expect(0, 128, '\p{Block= BASIC_Latin}', "");
Expect(1, 128, '\p{^Block= BASIC_Latin}', "");
Expect(1, 128, '\P{Block= BASIC_Latin}', "");
Expect(0, 128, '\P{^Block= BASIC_Latin}', "");
Error('\p{Blk=_/a/ASCII}');
Error('\P{Blk=_/a/ASCII}');
Expect(1, 127, '\p{Blk=ascii}', "");
Expect(0, 127, '\p{^Blk=ascii}', "");
Expect(0, 127, '\P{Blk=ascii}', "");
Expect(1, 127, '\P{^Blk=ascii}', "");
Expect(0, 128, '\p{Blk=ascii}', "");
Expect(1, 128, '\p{^Blk=ascii}', "");
Expect(1, 128, '\P{Blk=ascii}', "");
Expect(0, 128, '\P{^Blk=ascii}', "");
Expect(1, 127, '\p{Blk:	_-ascii}', "");
Expect(0, 127, '\p{^Blk:	_-ascii}', "");
Expect(0, 127, '\P{Blk:	_-ascii}', "");
Expect(1, 127, '\P{^Blk:	_-ascii}', "");
Expect(0, 128, '\p{Blk:	_-ascii}', "");
Expect(1, 128, '\p{^Blk:	_-ascii}', "");
Expect(1, 128, '\P{Blk:	_-ascii}', "");
Expect(0, 128, '\P{^Blk:	_-ascii}', "");
Error('\p{Is_Block=_/a/Basic_Latin}');
Error('\P{Is_Block=_/a/Basic_Latin}');
Expect(1, 127, '\p{Is_Block=basiclatin}', "");
Expect(0, 127, '\p{^Is_Block=basiclatin}', "");
Expect(0, 127, '\P{Is_Block=basiclatin}', "");
Expect(1, 127, '\P{^Is_Block=basiclatin}', "");
Expect(0, 128, '\p{Is_Block=basiclatin}', "");
Expect(1, 128, '\p{^Is_Block=basiclatin}', "");
Expect(1, 128, '\P{Is_Block=basiclatin}', "");
Expect(0, 128, '\P{^Is_Block=basiclatin}', "");
Expect(1, 127, '\p{Is_Block=_basic_latin}', "");
Expect(0, 127, '\p{^Is_Block=_basic_latin}', "");
Expect(0, 127, '\P{Is_Block=_basic_latin}', "");
Expect(1, 127, '\P{^Is_Block=_basic_latin}', "");
Expect(0, 128, '\p{Is_Block=_basic_latin}', "");
Expect(1, 128, '\p{^Is_Block=_basic_latin}', "");
Expect(1, 128, '\P{Is_Block=_basic_latin}', "");
Expect(0, 128, '\P{^Is_Block=_basic_latin}', "");
Error('\p{Is_Blk=-	ASCII:=}');
Error('\P{Is_Blk=-	ASCII:=}');
Expect(1, 127, '\p{Is_Blk:   ascii}', "");
Expect(0, 127, '\p{^Is_Blk:   ascii}', "");
Expect(0, 127, '\P{Is_Blk:   ascii}', "");
Expect(1, 127, '\P{^Is_Blk:   ascii}', "");
Expect(0, 128, '\p{Is_Blk:   ascii}', "");
Expect(1, 128, '\p{^Is_Blk:   ascii}', "");
Expect(1, 128, '\P{Is_Blk:   ascii}', "");
Expect(0, 128, '\P{^Is_Blk:   ascii}', "");
Expect(1, 127, '\p{Is_Blk= _ascii}', "");
Expect(0, 127, '\p{^Is_Blk= _ascii}', "");
Expect(0, 127, '\P{Is_Blk= _ascii}', "");
Expect(1, 127, '\P{^Is_Blk= _ascii}', "");
Expect(0, 128, '\p{Is_Blk= _ascii}', "");
Expect(1, 128, '\p{^Is_Blk= _ascii}', "");
Expect(1, 128, '\P{Is_Blk= _ascii}', "");
Expect(0, 128, '\P{^Is_Blk= _ascii}', "");
Error('\p{Block=	-Aegean_numbers/a/}');
Error('\P{Block=	-Aegean_numbers/a/}');
Expect(1, 65855, '\p{Block=aegeannumbers}', "");
Expect(0, 65855, '\p{^Block=aegeannumbers}', "");
Expect(0, 65855, '\P{Block=aegeannumbers}', "");
Expect(1, 65855, '\P{^Block=aegeannumbers}', "");
Expect(0, 65856, '\p{Block=aegeannumbers}', "");
Expect(1, 65856, '\p{^Block=aegeannumbers}', "");
Expect(1, 65856, '\P{Block=aegeannumbers}', "");
Expect(0, 65856, '\P{^Block=aegeannumbers}', "");
Expect(1, 65855, '\p{Block:	-Aegean_Numbers}', "");
Expect(0, 65855, '\p{^Block:	-Aegean_Numbers}', "");
Expect(0, 65855, '\P{Block:	-Aegean_Numbers}', "");
Expect(1, 65855, '\P{^Block:	-Aegean_Numbers}', "");
Expect(0, 65856, '\p{Block:	-Aegean_Numbers}', "");
Expect(1, 65856, '\p{^Block:	-Aegean_Numbers}', "");
Expect(1, 65856, '\P{Block:	-Aegean_Numbers}', "");
Expect(0, 65856, '\P{^Block:	-Aegean_Numbers}', "");
Error('\p{Blk=_	AEGEAN_NUMBERS/a/}');
Error('\P{Blk=_	AEGEAN_NUMBERS/a/}');
Expect(1, 65855, '\p{Blk:aegeannumbers}', "");
Expect(0, 65855, '\p{^Blk:aegeannumbers}', "");
Expect(0, 65855, '\P{Blk:aegeannumbers}', "");
Expect(1, 65855, '\P{^Blk:aegeannumbers}', "");
Expect(0, 65856, '\p{Blk:aegeannumbers}', "");
Expect(1, 65856, '\p{^Blk:aegeannumbers}', "");
Expect(1, 65856, '\P{Blk:aegeannumbers}', "");
Expect(0, 65856, '\P{^Blk:aegeannumbers}', "");
Expect(1, 65855, '\p{Blk= Aegean_numbers}', "");
Expect(0, 65855, '\p{^Blk= Aegean_numbers}', "");
Expect(0, 65855, '\P{Blk= Aegean_numbers}', "");
Expect(1, 65855, '\P{^Blk= Aegean_numbers}', "");
Expect(0, 65856, '\p{Blk= Aegean_numbers}', "");
Expect(1, 65856, '\p{^Blk= Aegean_numbers}', "");
Expect(1, 65856, '\P{Blk= Aegean_numbers}', "");
Expect(0, 65856, '\P{^Blk= Aegean_numbers}', "");
Error('\p{Is_Block=/a/_AEGEAN_numbers}');
Error('\P{Is_Block=/a/_AEGEAN_numbers}');
Expect(1, 65855, '\p{Is_Block=aegeannumbers}', "");
Expect(0, 65855, '\p{^Is_Block=aegeannumbers}', "");
Expect(0, 65855, '\P{Is_Block=aegeannumbers}', "");
Expect(1, 65855, '\P{^Is_Block=aegeannumbers}', "");
Expect(0, 65856, '\p{Is_Block=aegeannumbers}', "");
Expect(1, 65856, '\p{^Is_Block=aegeannumbers}', "");
Expect(1, 65856, '\P{Is_Block=aegeannumbers}', "");
Expect(0, 65856, '\P{^Is_Block=aegeannumbers}', "");
Expect(1, 65855, '\p{Is_Block=--AEGEAN_NUMBERS}', "");
Expect(0, 65855, '\p{^Is_Block=--AEGEAN_NUMBERS}', "");
Expect(0, 65855, '\P{Is_Block=--AEGEAN_NUMBERS}', "");
Expect(1, 65855, '\P{^Is_Block=--AEGEAN_NUMBERS}', "");
Expect(0, 65856, '\p{Is_Block=--AEGEAN_NUMBERS}', "");
Expect(1, 65856, '\p{^Is_Block=--AEGEAN_NUMBERS}', "");
Expect(1, 65856, '\P{Is_Block=--AEGEAN_NUMBERS}', "");
Expect(0, 65856, '\P{^Is_Block=--AEGEAN_NUMBERS}', "");
Error('\p{Is_Blk= Aegean_numbers/a/}');
Error('\P{Is_Blk= Aegean_numbers/a/}');
Expect(1, 65855, '\p{Is_Blk=aegeannumbers}', "");
Expect(0, 65855, '\p{^Is_Blk=aegeannumbers}', "");
Expect(0, 65855, '\P{Is_Blk=aegeannumbers}', "");
Expect(1, 65855, '\P{^Is_Blk=aegeannumbers}', "");
Expect(0, 65856, '\p{Is_Blk=aegeannumbers}', "");
Expect(1, 65856, '\p{^Is_Blk=aegeannumbers}', "");
Expect(1, 65856, '\P{Is_Blk=aegeannumbers}', "");
Expect(0, 65856, '\P{^Is_Blk=aegeannumbers}', "");
Expect(1, 65855, '\p{Is_Blk=- aegean_NUMBERS}', "");
Expect(0, 65855, '\p{^Is_Blk=- aegean_NUMBERS}', "");
Expect(0, 65855, '\P{Is_Blk=- aegean_NUMBERS}', "");
Expect(1, 65855, '\P{^Is_Blk=- aegean_NUMBERS}', "");
Expect(0, 65856, '\p{Is_Blk=- aegean_NUMBERS}', "");
Expect(1, 65856, '\p{^Is_Blk=- aegean_NUMBERS}', "");
Expect(1, 65856, '\P{Is_Blk=- aegean_NUMBERS}', "");
Expect(0, 65856, '\P{^Is_Blk=- aegean_NUMBERS}', "");
Error('\p{Block:   alchemical_Symbols:=}');
Error('\P{Block:   alchemical_Symbols:=}');
Expect(1, 128895, '\p{Block=alchemicalsymbols}', "");
Expect(0, 128895, '\p{^Block=alchemicalsymbols}', "");
Expect(0, 128895, '\P{Block=alchemicalsymbols}', "");
Expect(1, 128895, '\P{^Block=alchemicalsymbols}', "");
Expect(0, 128896, '\p{Block=alchemicalsymbols}', "");
Expect(1, 128896, '\p{^Block=alchemicalsymbols}', "");
Expect(1, 128896, '\P{Block=alchemicalsymbols}', "");
Expect(0, 128896, '\P{^Block=alchemicalsymbols}', "");
Expect(1, 128895, '\p{Block= ALCHEMICAL_SYMBOLS}', "");
Expect(0, 128895, '\p{^Block= ALCHEMICAL_SYMBOLS}', "");
Expect(0, 128895, '\P{Block= ALCHEMICAL_SYMBOLS}', "");
Expect(1, 128895, '\P{^Block= ALCHEMICAL_SYMBOLS}', "");
Expect(0, 128896, '\p{Block= ALCHEMICAL_SYMBOLS}', "");
Expect(1, 128896, '\p{^Block= ALCHEMICAL_SYMBOLS}', "");
Expect(1, 128896, '\P{Block= ALCHEMICAL_SYMBOLS}', "");
Expect(0, 128896, '\P{^Block= ALCHEMICAL_SYMBOLS}', "");
Error('\p{Blk=/a/-Alchemical}');
Error('\P{Blk=/a/-Alchemical}');
Expect(1, 128895, '\p{Blk=alchemical}', "");
Expect(0, 128895, '\p{^Blk=alchemical}', "");
Expect(0, 128895, '\P{Blk=alchemical}', "");
Expect(1, 128895, '\P{^Blk=alchemical}', "");
Expect(0, 128896, '\p{Blk=alchemical}', "");
Expect(1, 128896, '\p{^Blk=alchemical}', "");
Expect(1, 128896, '\P{Blk=alchemical}', "");
Expect(0, 128896, '\P{^Blk=alchemical}', "");
Expect(1, 128895, '\p{Blk=__Alchemical}', "");
Expect(0, 128895, '\p{^Blk=__Alchemical}', "");
Expect(0, 128895, '\P{Blk=__Alchemical}', "");
Expect(1, 128895, '\P{^Blk=__Alchemical}', "");
Expect(0, 128896, '\p{Blk=__Alchemical}', "");
Expect(1, 128896, '\p{^Blk=__Alchemical}', "");
Expect(1, 128896, '\P{Blk=__Alchemical}', "");
Expect(0, 128896, '\P{^Blk=__Alchemical}', "");
Error('\p{Is_Block:	/a/--alchemical_Symbols}');
Error('\P{Is_Block:	/a/--alchemical_Symbols}');
Expect(1, 128895, '\p{Is_Block=alchemicalsymbols}', "");
Expect(0, 128895, '\p{^Is_Block=alchemicalsymbols}', "");
Expect(0, 128895, '\P{Is_Block=alchemicalsymbols}', "");
Expect(1, 128895, '\P{^Is_Block=alchemicalsymbols}', "");
Expect(0, 128896, '\p{Is_Block=alchemicalsymbols}', "");
Expect(1, 128896, '\p{^Is_Block=alchemicalsymbols}', "");
Expect(1, 128896, '\P{Is_Block=alchemicalsymbols}', "");
Expect(0, 128896, '\P{^Is_Block=alchemicalsymbols}', "");
Expect(1, 128895, '\p{Is_Block=-Alchemical_Symbols}', "");
Expect(0, 128895, '\p{^Is_Block=-Alchemical_Symbols}', "");
Expect(0, 128895, '\P{Is_Block=-Alchemical_Symbols}', "");
Expect(1, 128895, '\P{^Is_Block=-Alchemical_Symbols}', "");
Expect(0, 128896, '\p{Is_Block=-Alchemical_Symbols}', "");
Expect(1, 128896, '\p{^Is_Block=-Alchemical_Symbols}', "");
Expect(1, 128896, '\P{Is_Block=-Alchemical_Symbols}', "");
Expect(0, 128896, '\P{^Is_Block=-Alchemical_Symbols}', "");
Error('\p{Is_Blk=--Alchemical/a/}');
Error('\P{Is_Blk=--Alchemical/a/}');
Expect(1, 128895, '\p{Is_Blk=alchemical}', "");
Expect(0, 128895, '\p{^Is_Blk=alchemical}', "");
Expect(0, 128895, '\P{Is_Blk=alchemical}', "");
Expect(1, 128895, '\P{^Is_Blk=alchemical}', "");
Expect(0, 128896, '\p{Is_Blk=alchemical}', "");
Expect(1, 128896, '\p{^Is_Blk=alchemical}', "");
Expect(1, 128896, '\P{Is_Blk=alchemical}', "");
Expect(0, 128896, '\P{^Is_Blk=alchemical}', "");
Expect(1, 128895, '\p{Is_Blk=	 Alchemical}', "");
Expect(0, 128895, '\p{^Is_Blk=	 Alchemical}', "");
Expect(0, 128895, '\P{Is_Blk=	 Alchemical}', "");
Expect(1, 128895, '\P{^Is_Blk=	 Alchemical}', "");
Expect(0, 128896, '\p{Is_Blk=	 Alchemical}', "");
Expect(1, 128896, '\p{^Is_Blk=	 Alchemical}', "");
Expect(1, 128896, '\P{Is_Blk=	 Alchemical}', "");
Expect(0, 128896, '\P{^Is_Blk=	 Alchemical}', "");
Error('\p{Block=/a/--Alphabetic_Presentation_FORMS}');
Error('\P{Block=/a/--Alphabetic_Presentation_FORMS}');
Expect(1, 64335, '\p{Block=alphabeticpresentationforms}', "");
Expect(0, 64335, '\p{^Block=alphabeticpresentationforms}', "");
Expect(0, 64335, '\P{Block=alphabeticpresentationforms}', "");
Expect(1, 64335, '\P{^Block=alphabeticpresentationforms}', "");
Expect(0, 64336, '\p{Block=alphabeticpresentationforms}', "");
Expect(1, 64336, '\p{^Block=alphabeticpresentationforms}', "");
Expect(1, 64336, '\P{Block=alphabeticpresentationforms}', "");
Expect(0, 64336, '\P{^Block=alphabeticpresentationforms}', "");
Expect(1, 64335, '\p{Block=-ALPHABETIC_presentation_FORMS}', "");
Expect(0, 64335, '\p{^Block=-ALPHABETIC_presentation_FORMS}', "");
Expect(0, 64335, '\P{Block=-ALPHABETIC_presentation_FORMS}', "");
Expect(1, 64335, '\P{^Block=-ALPHABETIC_presentation_FORMS}', "");
Expect(0, 64336, '\p{Block=-ALPHABETIC_presentation_FORMS}', "");
Expect(1, 64336, '\p{^Block=-ALPHABETIC_presentation_FORMS}', "");
Expect(1, 64336, '\P{Block=-ALPHABETIC_presentation_FORMS}', "");
Expect(0, 64336, '\P{^Block=-ALPHABETIC_presentation_FORMS}', "");
Error('\p{Blk=/a/_Alphabetic_PF}');
Error('\P{Blk=/a/_Alphabetic_PF}');
Expect(1, 64335, '\p{Blk=alphabeticpf}', "");
Expect(0, 64335, '\p{^Blk=alphabeticpf}', "");
Expect(0, 64335, '\P{Blk=alphabeticpf}', "");
Expect(1, 64335, '\P{^Blk=alphabeticpf}', "");
Expect(0, 64336, '\p{Blk=alphabeticpf}', "");
Expect(1, 64336, '\p{^Blk=alphabeticpf}', "");
Expect(1, 64336, '\P{Blk=alphabeticpf}', "");
Expect(0, 64336, '\P{^Blk=alphabeticpf}', "");
Expect(1, 64335, '\p{Blk=_alphabetic_PF}', "");
Expect(0, 64335, '\p{^Blk=_alphabetic_PF}', "");
Expect(0, 64335, '\P{Blk=_alphabetic_PF}', "");
Expect(1, 64335, '\P{^Blk=_alphabetic_PF}', "");
Expect(0, 64336, '\p{Blk=_alphabetic_PF}', "");
Expect(1, 64336, '\p{^Blk=_alphabetic_PF}', "");
Expect(1, 64336, '\P{Blk=_alphabetic_PF}', "");
Expect(0, 64336, '\P{^Blk=_alphabetic_PF}', "");
Error('\p{Is_Block=--Alphabetic_PRESENTATION_Forms/a/}');
Error('\P{Is_Block=--Alphabetic_PRESENTATION_Forms/a/}');
Expect(1, 64335, '\p{Is_Block=alphabeticpresentationforms}', "");
Expect(0, 64335, '\p{^Is_Block=alphabeticpresentationforms}', "");
Expect(0, 64335, '\P{Is_Block=alphabeticpresentationforms}', "");
Expect(1, 64335, '\P{^Is_Block=alphabeticpresentationforms}', "");
Expect(0, 64336, '\p{Is_Block=alphabeticpresentationforms}', "");
Expect(1, 64336, '\p{^Is_Block=alphabeticpresentationforms}', "");
Expect(1, 64336, '\P{Is_Block=alphabeticpresentationforms}', "");
Expect(0, 64336, '\P{^Is_Block=alphabeticpresentationforms}', "");
Expect(1, 64335, '\p{Is_Block:   		Alphabetic_presentation_Forms}', "");
Expect(0, 64335, '\p{^Is_Block:   		Alphabetic_presentation_Forms}', "");
Expect(0, 64335, '\P{Is_Block:   		Alphabetic_presentation_Forms}', "");
Expect(1, 64335, '\P{^Is_Block:   		Alphabetic_presentation_Forms}', "");
Expect(0, 64336, '\p{Is_Block:   		Alphabetic_presentation_Forms}', "");
Expect(1, 64336, '\p{^Is_Block:   		Alphabetic_presentation_Forms}', "");
Expect(1, 64336, '\P{Is_Block:   		Alphabetic_presentation_Forms}', "");
Expect(0, 64336, '\P{^Is_Block:   		Alphabetic_presentation_Forms}', "");
Error('\p{Is_Blk=:=	_Alphabetic_pf}');
Error('\P{Is_Blk=:=	_Alphabetic_pf}');
Expect(1, 64335, '\p{Is_Blk=alphabeticpf}', "");
Expect(0, 64335, '\p{^Is_Blk=alphabeticpf}', "");
Expect(0, 64335, '\P{Is_Blk=alphabeticpf}', "");
Expect(1, 64335, '\P{^Is_Blk=alphabeticpf}', "");
Expect(0, 64336, '\p{Is_Blk=alphabeticpf}', "");
Expect(1, 64336, '\p{^Is_Blk=alphabeticpf}', "");
Expect(1, 64336, '\P{Is_Blk=alphabeticpf}', "");
Expect(0, 64336, '\P{^Is_Blk=alphabeticpf}', "");
Expect(1, 64335, '\p{Is_Blk:	__ALPHABETIC_pf}', "");
Expect(0, 64335, '\p{^Is_Blk:	__ALPHABETIC_pf}', "");
Expect(0, 64335, '\P{Is_Blk:	__ALPHABETIC_pf}', "");
Expect(1, 64335, '\P{^Is_Blk:	__ALPHABETIC_pf}', "");
Expect(0, 64336, '\p{Is_Blk:	__ALPHABETIC_pf}', "");
Expect(1, 64336, '\p{^Is_Blk:	__ALPHABETIC_pf}', "");
Expect(1, 64336, '\P{Is_Blk:	__ALPHABETIC_pf}', "");
Expect(0, 64336, '\P{^Is_Blk:	__ALPHABETIC_pf}', "");
Error('\p{Block=:=__Ancient_Greek_Musical_Notation}');
Error('\P{Block=:=__Ancient_Greek_Musical_Notation}');
Expect(1, 119375, '\p{Block=ancientgreekmusicalnotation}', "");
Expect(0, 119375, '\p{^Block=ancientgreekmusicalnotation}', "");
Expect(0, 119375, '\P{Block=ancientgreekmusicalnotation}', "");
Expect(1, 119375, '\P{^Block=ancientgreekmusicalnotation}', "");
Expect(0, 119376, '\p{Block=ancientgreekmusicalnotation}', "");
Expect(1, 119376, '\p{^Block=ancientgreekmusicalnotation}', "");
Expect(1, 119376, '\P{Block=ancientgreekmusicalnotation}', "");
Expect(0, 119376, '\P{^Block=ancientgreekmusicalnotation}', "");
Expect(1, 119375, '\p{Block=		Ancient_Greek_Musical_Notation}', "");
Expect(0, 119375, '\p{^Block=		Ancient_Greek_Musical_Notation}', "");
Expect(0, 119375, '\P{Block=		Ancient_Greek_Musical_Notation}', "");
Expect(1, 119375, '\P{^Block=		Ancient_Greek_Musical_Notation}', "");
Expect(0, 119376, '\p{Block=		Ancient_Greek_Musical_Notation}', "");
Expect(1, 119376, '\p{^Block=		Ancient_Greek_Musical_Notation}', "");
Expect(1, 119376, '\P{Block=		Ancient_Greek_Musical_Notation}', "");
Expect(0, 119376, '\P{^Block=		Ancient_Greek_Musical_Notation}', "");
Error('\p{Blk=	/a/ancient_Greek_MUSIC}');
Error('\P{Blk=	/a/ancient_Greek_MUSIC}');
Expect(1, 119375, '\p{Blk=ancientgreekmusic}', "");
Expect(0, 119375, '\p{^Blk=ancientgreekmusic}', "");
Expect(0, 119375, '\P{Blk=ancientgreekmusic}', "");
Expect(1, 119375, '\P{^Blk=ancientgreekmusic}', "");
Expect(0, 119376, '\p{Blk=ancientgreekmusic}', "");
Expect(1, 119376, '\p{^Blk=ancientgreekmusic}', "");
Expect(1, 119376, '\P{Blk=ancientgreekmusic}', "");
Expect(0, 119376, '\P{^Blk=ancientgreekmusic}', "");
Expect(1, 119375, '\p{Blk=_	Ancient_greek_music}', "");
Expect(0, 119375, '\p{^Blk=_	Ancient_greek_music}', "");
Expect(0, 119375, '\P{Blk=_	Ancient_greek_music}', "");
Expect(1, 119375, '\P{^Blk=_	Ancient_greek_music}', "");
Expect(0, 119376, '\p{Blk=_	Ancient_greek_music}', "");
Expect(1, 119376, '\p{^Blk=_	Ancient_greek_music}', "");
Expect(1, 119376, '\P{Blk=_	Ancient_greek_music}', "");
Expect(0, 119376, '\P{^Blk=_	Ancient_greek_music}', "");
Error('\p{Is_Block: 	ANCIENT_GREEK_MUSICAL_Notation:=}');
Error('\P{Is_Block: 	ANCIENT_GREEK_MUSICAL_Notation:=}');
Expect(1, 119375, '\p{Is_Block=ancientgreekmusicalnotation}', "");
Expect(0, 119375, '\p{^Is_Block=ancientgreekmusicalnotation}', "");
Expect(0, 119375, '\P{Is_Block=ancientgreekmusicalnotation}', "");
Expect(1, 119375, '\P{^Is_Block=ancientgreekmusicalnotation}', "");
Expect(0, 119376, '\p{Is_Block=ancientgreekmusicalnotation}', "");
Expect(1, 119376, '\p{^Is_Block=ancientgreekmusicalnotation}', "");
Expect(1, 119376, '\P{Is_Block=ancientgreekmusicalnotation}', "");
Expect(0, 119376, '\P{^Is_Block=ancientgreekmusicalnotation}', "");
Expect(1, 119375, '\p{Is_Block=		ANCIENT_Greek_MUSICAL_NOTATION}', "");
Expect(0, 119375, '\p{^Is_Block=		ANCIENT_Greek_MUSICAL_NOTATION}', "");
Expect(0, 119375, '\P{Is_Block=		ANCIENT_Greek_MUSICAL_NOTATION}', "");
Expect(1, 119375, '\P{^Is_Block=		ANCIENT_Greek_MUSICAL_NOTATION}', "");
Expect(0, 119376, '\p{Is_Block=		ANCIENT_Greek_MUSICAL_NOTATION}', "");
Expect(1, 119376, '\p{^Is_Block=		ANCIENT_Greek_MUSICAL_NOTATION}', "");
Expect(1, 119376, '\P{Is_Block=		ANCIENT_Greek_MUSICAL_NOTATION}', "");
Expect(0, 119376, '\P{^Is_Block=		ANCIENT_Greek_MUSICAL_NOTATION}', "");
Error('\p{Is_Blk=__Ancient_Greek_Music:=}');
Error('\P{Is_Blk=__Ancient_Greek_Music:=}');
Expect(1, 119375, '\p{Is_Blk:	ancientgreekmusic}', "");
Expect(0, 119375, '\p{^Is_Blk:	ancientgreekmusic}', "");
Expect(0, 119375, '\P{Is_Blk:	ancientgreekmusic}', "");
Expect(1, 119375, '\P{^Is_Blk:	ancientgreekmusic}', "");
Expect(0, 119376, '\p{Is_Blk:	ancientgreekmusic}', "");
Expect(1, 119376, '\p{^Is_Blk:	ancientgreekmusic}', "");
Expect(1, 119376, '\P{Is_Blk:	ancientgreekmusic}', "");
Expect(0, 119376, '\P{^Is_Blk:	ancientgreekmusic}', "");
Expect(1, 119375, '\p{Is_Blk=_	Ancient_greek_Music}', "");
Expect(0, 119375, '\p{^Is_Blk=_	Ancient_greek_Music}', "");
Expect(0, 119375, '\P{Is_Blk=_	Ancient_greek_Music}', "");
Expect(1, 119375, '\P{^Is_Blk=_	Ancient_greek_Music}', "");
Expect(0, 119376, '\p{Is_Blk=_	Ancient_greek_Music}', "");
Expect(1, 119376, '\p{^Is_Blk=_	Ancient_greek_Music}', "");
Expect(1, 119376, '\P{Is_Blk=_	Ancient_greek_Music}', "");
Expect(0, 119376, '\P{^Is_Blk=_	Ancient_greek_Music}', "");
Error('\p{Block= ancient_Greek_NUMBERS/a/}');
Error('\P{Block= ancient_Greek_NUMBERS/a/}');
Expect(1, 65935, '\p{Block=ancientgreeknumbers}', "");
Expect(0, 65935, '\p{^Block=ancientgreeknumbers}', "");
Expect(0, 65935, '\P{Block=ancientgreeknumbers}', "");
Expect(1, 65935, '\P{^Block=ancientgreeknumbers}', "");
Expect(0, 65936, '\p{Block=ancientgreeknumbers}', "");
Expect(1, 65936, '\p{^Block=ancientgreeknumbers}', "");
Expect(1, 65936, '\P{Block=ancientgreeknumbers}', "");
Expect(0, 65936, '\P{^Block=ancientgreeknumbers}', "");
Expect(1, 65935, '\p{Block=		ANCIENT_GREEK_Numbers}', "");
Expect(0, 65935, '\p{^Block=		ANCIENT_GREEK_Numbers}', "");
Expect(0, 65935, '\P{Block=		ANCIENT_GREEK_Numbers}', "");
Expect(1, 65935, '\P{^Block=		ANCIENT_GREEK_Numbers}', "");
Expect(0, 65936, '\p{Block=		ANCIENT_GREEK_Numbers}', "");
Expect(1, 65936, '\p{^Block=		ANCIENT_GREEK_Numbers}', "");
Expect(1, 65936, '\P{Block=		ANCIENT_GREEK_Numbers}', "");
Expect(0, 65936, '\P{^Block=		ANCIENT_GREEK_Numbers}', "");
Error('\p{Blk=/a/_ ancient_Greek_Numbers}');
Error('\P{Blk=/a/_ ancient_Greek_Numbers}');
Expect(1, 65935, '\p{Blk=ancientgreeknumbers}', "");
Expect(0, 65935, '\p{^Blk=ancientgreeknumbers}', "");
Expect(0, 65935, '\P{Blk=ancientgreeknumbers}', "");
Expect(1, 65935, '\P{^Blk=ancientgreeknumbers}', "");
Expect(0, 65936, '\p{Blk=ancientgreeknumbers}', "");
Expect(1, 65936, '\p{^Blk=ancientgreeknumbers}', "");
Expect(1, 65936, '\P{Blk=ancientgreeknumbers}', "");
Expect(0, 65936, '\P{^Blk=ancientgreeknumbers}', "");
Expect(1, 65935, '\p{Blk=-ANCIENT_greek_Numbers}', "");
Expect(0, 65935, '\p{^Blk=-ANCIENT_greek_Numbers}', "");
Expect(0, 65935, '\P{Blk=-ANCIENT_greek_Numbers}', "");
Expect(1, 65935, '\P{^Blk=-ANCIENT_greek_Numbers}', "");
Expect(0, 65936, '\p{Blk=-ANCIENT_greek_Numbers}', "");
Expect(1, 65936, '\p{^Blk=-ANCIENT_greek_Numbers}', "");
Expect(1, 65936, '\P{Blk=-ANCIENT_greek_Numbers}', "");
Expect(0, 65936, '\P{^Blk=-ANCIENT_greek_Numbers}', "");
Error('\p{Is_Block:-:=ancient_GREEK_Numbers}');
Error('\P{Is_Block:-:=ancient_GREEK_Numbers}');
Expect(1, 65935, '\p{Is_Block=ancientgreeknumbers}', "");
Expect(0, 65935, '\p{^Is_Block=ancientgreeknumbers}', "");
Expect(0, 65935, '\P{Is_Block=ancientgreeknumbers}', "");
Expect(1, 65935, '\P{^Is_Block=ancientgreeknumbers}', "");
Expect(0, 65936, '\p{Is_Block=ancientgreeknumbers}', "");
Expect(1, 65936, '\p{^Is_Block=ancientgreeknumbers}', "");
Expect(1, 65936, '\P{Is_Block=ancientgreeknumbers}', "");
Expect(0, 65936, '\P{^Is_Block=ancientgreeknumbers}', "");
Expect(1, 65935, '\p{Is_Block= ANCIENT_greek_NUMBERS}', "");
Expect(0, 65935, '\p{^Is_Block= ANCIENT_greek_NUMBERS}', "");
Expect(0, 65935, '\P{Is_Block= ANCIENT_greek_NUMBERS}', "");
Expect(1, 65935, '\P{^Is_Block= ANCIENT_greek_NUMBERS}', "");
Expect(0, 65936, '\p{Is_Block= ANCIENT_greek_NUMBERS}', "");
Expect(1, 65936, '\p{^Is_Block= ANCIENT_greek_NUMBERS}', "");
Expect(1, 65936, '\P{Is_Block= ANCIENT_greek_NUMBERS}', "");
Expect(0, 65936, '\P{^Is_Block= ANCIENT_greek_NUMBERS}', "");
Error('\p{Is_Blk=-/a/ANCIENT_Greek_numbers}');
Error('\P{Is_Blk=-/a/ANCIENT_Greek_numbers}');
Expect(1, 65935, '\p{Is_Blk=ancientgreeknumbers}', "");
Expect(0, 65935, '\p{^Is_Blk=ancientgreeknumbers}', "");
Expect(0, 65935, '\P{Is_Blk=ancientgreeknumbers}', "");
Expect(1, 65935, '\P{^Is_Blk=ancientgreeknumbers}', "");
Expect(0, 65936, '\p{Is_Blk=ancientgreeknumbers}', "");
Expect(1, 65936, '\p{^Is_Blk=ancientgreeknumbers}', "");
Expect(1, 65936, '\P{Is_Blk=ancientgreeknumbers}', "");
Expect(0, 65936, '\P{^Is_Blk=ancientgreeknumbers}', "");
Expect(1, 65935, '\p{Is_Blk= Ancient_GREEK_NUMBERS}', "");
Expect(0, 65935, '\p{^Is_Blk= Ancient_GREEK_NUMBERS}', "");
Expect(0, 65935, '\P{Is_Blk= Ancient_GREEK_NUMBERS}', "");
Expect(1, 65935, '\P{^Is_Blk= Ancient_GREEK_NUMBERS}', "");
Expect(0, 65936, '\p{Is_Blk= Ancient_GREEK_NUMBERS}', "");
Expect(1, 65936, '\p{^Is_Blk= Ancient_GREEK_NUMBERS}', "");
Expect(1, 65936, '\P{Is_Blk= Ancient_GREEK_NUMBERS}', "");
Expect(0, 65936, '\P{^Is_Blk= Ancient_GREEK_NUMBERS}', "");
Error('\p{Block=:=_	ANCIENT_SYMBOLS}');
Error('\P{Block=:=_	ANCIENT_SYMBOLS}');
Expect(1, 65999, '\p{Block=ancientsymbols}', "");
Expect(0, 65999, '\p{^Block=ancientsymbols}', "");
Expect(0, 65999, '\P{Block=ancientsymbols}', "");
Expect(1, 65999, '\P{^Block=ancientsymbols}', "");
Expect(0, 66000, '\p{Block=ancientsymbols}', "");
Expect(1, 66000, '\p{^Block=ancientsymbols}', "");
Expect(1, 66000, '\P{Block=ancientsymbols}', "");
Expect(0, 66000, '\P{^Block=ancientsymbols}', "");
Expect(1, 65999, '\p{Block= -Ancient_Symbols}', "");
Expect(0, 65999, '\p{^Block= -Ancient_Symbols}', "");
Expect(0, 65999, '\P{Block= -Ancient_Symbols}', "");
Expect(1, 65999, '\P{^Block= -Ancient_Symbols}', "");
Expect(0, 66000, '\p{Block= -Ancient_Symbols}', "");
Expect(1, 66000, '\p{^Block= -Ancient_Symbols}', "");
Expect(1, 66000, '\P{Block= -Ancient_Symbols}', "");
Expect(0, 66000, '\P{^Block= -Ancient_Symbols}', "");
Error('\p{Blk:	_Ancient_symbols:=}');
Error('\P{Blk:	_Ancient_symbols:=}');
Expect(1, 65999, '\p{Blk=ancientsymbols}', "");
Expect(0, 65999, '\p{^Blk=ancientsymbols}', "");
Expect(0, 65999, '\P{Blk=ancientsymbols}', "");
Expect(1, 65999, '\P{^Blk=ancientsymbols}', "");
Expect(0, 66000, '\p{Blk=ancientsymbols}', "");
Expect(1, 66000, '\p{^Blk=ancientsymbols}', "");
Expect(1, 66000, '\P{Blk=ancientsymbols}', "");
Expect(0, 66000, '\P{^Blk=ancientsymbols}', "");
Expect(1, 65999, '\p{Blk= -ancient_symbols}', "");
Expect(0, 65999, '\p{^Blk= -ancient_symbols}', "");
Expect(0, 65999, '\P{Blk= -ancient_symbols}', "");
Expect(1, 65999, '\P{^Blk= -ancient_symbols}', "");
Expect(0, 66000, '\p{Blk= -ancient_symbols}', "");
Expect(1, 66000, '\p{^Blk= -ancient_symbols}', "");
Expect(1, 66000, '\P{Blk= -ancient_symbols}', "");
Expect(0, 66000, '\P{^Blk= -ancient_symbols}', "");
Error('\p{Is_Block= Ancient_Symbols:=}');
Error('\P{Is_Block= Ancient_Symbols:=}');
Expect(1, 65999, '\p{Is_Block=ancientsymbols}', "");
Expect(0, 65999, '\p{^Is_Block=ancientsymbols}', "");
Expect(0, 65999, '\P{Is_Block=ancientsymbols}', "");
Expect(1, 65999, '\P{^Is_Block=ancientsymbols}', "");
Expect(0, 66000, '\p{Is_Block=ancientsymbols}', "");
Expect(1, 66000, '\p{^Is_Block=ancientsymbols}', "");
Expect(1, 66000, '\P{Is_Block=ancientsymbols}', "");
Expect(0, 66000, '\P{^Is_Block=ancientsymbols}', "");
Expect(1, 65999, '\p{Is_Block= _Ancient_Symbols}', "");
Expect(0, 65999, '\p{^Is_Block= _Ancient_Symbols}', "");
Expect(0, 65999, '\P{Is_Block= _Ancient_Symbols}', "");
Expect(1, 65999, '\P{^Is_Block= _Ancient_Symbols}', "");
Expect(0, 66000, '\p{Is_Block= _Ancient_Symbols}', "");
Expect(1, 66000, '\p{^Is_Block= _Ancient_Symbols}', "");
Expect(1, 66000, '\P{Is_Block= _Ancient_Symbols}', "");
Expect(0, 66000, '\P{^Is_Block= _Ancient_Symbols}', "");
Error('\p{Is_Blk=	/a/ancient_Symbols}');
Error('\P{Is_Blk=	/a/ancient_Symbols}');
Expect(1, 65999, '\p{Is_Blk=ancientsymbols}', "");
Expect(0, 65999, '\p{^Is_Blk=ancientsymbols}', "");
Expect(0, 65999, '\P{Is_Blk=ancientsymbols}', "");
Expect(1, 65999, '\P{^Is_Blk=ancientsymbols}', "");
Expect(0, 66000, '\p{Is_Blk=ancientsymbols}', "");
Expect(1, 66000, '\p{^Is_Blk=ancientsymbols}', "");
Expect(1, 66000, '\P{Is_Blk=ancientsymbols}', "");
Expect(0, 66000, '\P{^Is_Blk=ancientsymbols}', "");
Expect(1, 65999, '\p{Is_Blk=	-ancient_SYMBOLS}', "");
Expect(0, 65999, '\p{^Is_Blk=	-ancient_SYMBOLS}', "");
Expect(0, 65999, '\P{Is_Blk=	-ancient_SYMBOLS}', "");
Expect(1, 65999, '\P{^Is_Blk=	-ancient_SYMBOLS}', "");
Expect(0, 66000, '\p{Is_Blk=	-ancient_SYMBOLS}', "");
Expect(1, 66000, '\p{^Is_Blk=	-ancient_SYMBOLS}', "");
Expect(1, 66000, '\P{Is_Blk=	-ancient_SYMBOLS}', "");
Expect(0, 66000, '\P{^Is_Blk=	-ancient_SYMBOLS}', "");
Error('\p{Block= -Arabic:=}');
Error('\P{Block= -Arabic:=}');
Expect(1, 1791, '\p{Block=arabic}', "");
Expect(0, 1791, '\p{^Block=arabic}', "");
Expect(0, 1791, '\P{Block=arabic}', "");
Expect(1, 1791, '\P{^Block=arabic}', "");
Expect(0, 1792, '\p{Block=arabic}', "");
Expect(1, 1792, '\p{^Block=arabic}', "");
Expect(1, 1792, '\P{Block=arabic}', "");
Expect(0, 1792, '\P{^Block=arabic}', "");
Expect(1, 1791, '\p{Block=	ARABIC}', "");
Expect(0, 1791, '\p{^Block=	ARABIC}', "");
Expect(0, 1791, '\P{Block=	ARABIC}', "");
Expect(1, 1791, '\P{^Block=	ARABIC}', "");
Expect(0, 1792, '\p{Block=	ARABIC}', "");
Expect(1, 1792, '\p{^Block=	ARABIC}', "");
Expect(1, 1792, '\P{Block=	ARABIC}', "");
Expect(0, 1792, '\P{^Block=	ARABIC}', "");
Error('\p{Blk= ARABIC/a/}');
Error('\P{Blk= ARABIC/a/}');
Expect(1, 1791, '\p{Blk=arabic}', "");
Expect(0, 1791, '\p{^Blk=arabic}', "");
Expect(0, 1791, '\P{Blk=arabic}', "");
Expect(1, 1791, '\P{^Blk=arabic}', "");
Expect(0, 1792, '\p{Blk=arabic}', "");
Expect(1, 1792, '\p{^Blk=arabic}', "");
Expect(1, 1792, '\P{Blk=arabic}', "");
Expect(0, 1792, '\P{^Blk=arabic}', "");
Expect(1, 1791, '\p{Blk=	arabic}', "");
Expect(0, 1791, '\p{^Blk=	arabic}', "");
Expect(0, 1791, '\P{Blk=	arabic}', "");
Expect(1, 1791, '\P{^Blk=	arabic}', "");
Expect(0, 1792, '\p{Blk=	arabic}', "");
Expect(1, 1792, '\p{^Blk=	arabic}', "");
Expect(1, 1792, '\P{Blk=	arabic}', "");
Expect(0, 1792, '\P{^Blk=	arabic}', "");
Error('\p{Is_Block: /a/--arabic}');
Error('\P{Is_Block: /a/--arabic}');
Expect(1, 1791, '\p{Is_Block=arabic}', "");
Expect(0, 1791, '\p{^Is_Block=arabic}', "");
Expect(0, 1791, '\P{Is_Block=arabic}', "");
Expect(1, 1791, '\P{^Is_Block=arabic}', "");
Expect(0, 1792, '\p{Is_Block=arabic}', "");
Expect(1, 1792, '\p{^Is_Block=arabic}', "");
Expect(1, 1792, '\P{Is_Block=arabic}', "");
Expect(0, 1792, '\P{^Is_Block=arabic}', "");
Expect(1, 1791, '\p{Is_Block=_	Arabic}', "");
Expect(0, 1791, '\p{^Is_Block=_	Arabic}', "");
Expect(0, 1791, '\P{Is_Block=_	Arabic}', "");
Expect(1, 1791, '\P{^Is_Block=_	Arabic}', "");
Expect(0, 1792, '\p{Is_Block=_	Arabic}', "");
Expect(1, 1792, '\p{^Is_Block=_	Arabic}', "");
Expect(1, 1792, '\P{Is_Block=_	Arabic}', "");
Expect(0, 1792, '\P{^Is_Block=_	Arabic}', "");
Error('\p{Is_Blk=-/a/ARABIC}');
Error('\P{Is_Blk=-/a/ARABIC}');
Expect(1, 1791, '\p{Is_Blk=arabic}', "");
Expect(0, 1791, '\p{^Is_Blk=arabic}', "");
Expect(0, 1791, '\P{Is_Blk=arabic}', "");
Expect(1, 1791, '\P{^Is_Blk=arabic}', "");
Expect(0, 1792, '\p{Is_Blk=arabic}', "");
Expect(1, 1792, '\p{^Is_Blk=arabic}', "");
Expect(1, 1792, '\P{Is_Blk=arabic}', "");
Expect(0, 1792, '\P{^Is_Blk=arabic}', "");
Error('\p{Block=:=_-ARABIC_Extended_A}');
Error('\P{Block=:=_-ARABIC_Extended_A}');
Expect(1, 2303, '\p{Block:   arabicextendeda}', "");
Expect(0, 2303, '\p{^Block:   arabicextendeda}', "");
Expect(0, 2303, '\P{Block:   arabicextendeda}', "");
Expect(1, 2303, '\P{^Block:   arabicextendeda}', "");
Expect(0, 2304, '\p{Block:   arabicextendeda}', "");
Expect(1, 2304, '\p{^Block:   arabicextendeda}', "");
Expect(1, 2304, '\P{Block:   arabicextendeda}', "");
Expect(0, 2304, '\P{^Block:   arabicextendeda}', "");
Expect(1, 2303, '\p{Block= -ARABIC_EXTENDED_A}', "");
Expect(0, 2303, '\p{^Block= -ARABIC_EXTENDED_A}', "");
Expect(0, 2303, '\P{Block= -ARABIC_EXTENDED_A}', "");
Expect(1, 2303, '\P{^Block= -ARABIC_EXTENDED_A}', "");
Expect(0, 2304, '\p{Block= -ARABIC_EXTENDED_A}', "");
Expect(1, 2304, '\p{^Block= -ARABIC_EXTENDED_A}', "");
Expect(1, 2304, '\P{Block= -ARABIC_EXTENDED_A}', "");
Expect(0, 2304, '\P{^Block= -ARABIC_EXTENDED_A}', "");
Error('\p{Blk=_/a/Arabic_Ext_A}');
Error('\P{Blk=_/a/Arabic_Ext_A}');
Expect(1, 2303, '\p{Blk=arabicexta}', "");
Expect(0, 2303, '\p{^Blk=arabicexta}', "");
Expect(0, 2303, '\P{Blk=arabicexta}', "");
Expect(1, 2303, '\P{^Blk=arabicexta}', "");
Expect(0, 2304, '\p{Blk=arabicexta}', "");
Expect(1, 2304, '\p{^Blk=arabicexta}', "");
Expect(1, 2304, '\P{Blk=arabicexta}', "");
Expect(0, 2304, '\P{^Blk=arabicexta}', "");
Expect(1, 2303, '\p{Blk=  Arabic_Ext_a}', "");
Expect(0, 2303, '\p{^Blk=  Arabic_Ext_a}', "");
Expect(0, 2303, '\P{Blk=  Arabic_Ext_a}', "");
Expect(1, 2303, '\P{^Blk=  Arabic_Ext_a}', "");
Expect(0, 2304, '\p{Blk=  Arabic_Ext_a}', "");
Expect(1, 2304, '\p{^Blk=  Arabic_Ext_a}', "");
Expect(1, 2304, '\P{Blk=  Arabic_Ext_a}', "");
Expect(0, 2304, '\P{^Blk=  Arabic_Ext_a}', "");
Error('\p{Is_Block=_:=Arabic_EXTENDED_A}');
Error('\P{Is_Block=_:=Arabic_EXTENDED_A}');
Expect(1, 2303, '\p{Is_Block=arabicextendeda}', "");
Expect(0, 2303, '\p{^Is_Block=arabicextendeda}', "");
Expect(0, 2303, '\P{Is_Block=arabicextendeda}', "");
Expect(1, 2303, '\P{^Is_Block=arabicextendeda}', "");
Expect(0, 2304, '\p{Is_Block=arabicextendeda}', "");
Expect(1, 2304, '\p{^Is_Block=arabicextendeda}', "");
Expect(1, 2304, '\P{Is_Block=arabicextendeda}', "");
Expect(0, 2304, '\P{^Is_Block=arabicextendeda}', "");
Expect(1, 2303, '\p{Is_Block:	_ARABIC_EXTENDED_A}', "");
Expect(0, 2303, '\p{^Is_Block:	_ARABIC_EXTENDED_A}', "");
Expect(0, 2303, '\P{Is_Block:	_ARABIC_EXTENDED_A}', "");
Expect(1, 2303, '\P{^Is_Block:	_ARABIC_EXTENDED_A}', "");
Expect(0, 2304, '\p{Is_Block:	_ARABIC_EXTENDED_A}', "");
Expect(1, 2304, '\p{^Is_Block:	_ARABIC_EXTENDED_A}', "");
Expect(1, 2304, '\P{Is_Block:	_ARABIC_EXTENDED_A}', "");
Expect(0, 2304, '\P{^Is_Block:	_ARABIC_EXTENDED_A}', "");
Error('\p{Is_Blk=  Arabic_EXT_A/a/}');
Error('\P{Is_Blk=  Arabic_EXT_A/a/}');
Expect(1, 2303, '\p{Is_Blk=arabicexta}', "");
Expect(0, 2303, '\p{^Is_Blk=arabicexta}', "");
Expect(0, 2303, '\P{Is_Blk=arabicexta}', "");
Expect(1, 2303, '\P{^Is_Blk=arabicexta}', "");
Expect(0, 2304, '\p{Is_Blk=arabicexta}', "");
Expect(1, 2304, '\p{^Is_Blk=arabicexta}', "");
Expect(1, 2304, '\P{Is_Blk=arabicexta}', "");
Expect(0, 2304, '\P{^Is_Blk=arabicexta}', "");
Expect(1, 2303, '\p{Is_Blk= 	arabic_Ext_A}', "");
Expect(0, 2303, '\p{^Is_Blk= 	arabic_Ext_A}', "");
Expect(0, 2303, '\P{Is_Blk= 	arabic_Ext_A}', "");
Expect(1, 2303, '\P{^Is_Blk= 	arabic_Ext_A}', "");
Expect(0, 2304, '\p{Is_Blk= 	arabic_Ext_A}', "");
Expect(1, 2304, '\p{^Is_Blk= 	arabic_Ext_A}', "");
Expect(1, 2304, '\P{Is_Blk= 	arabic_Ext_A}', "");
Expect(0, 2304, '\P{^Is_Blk= 	arabic_Ext_A}', "");
Error('\p{Block=_:=arabic_MATHEMATICAL_Alphabetic_Symbols}');
Error('\P{Block=_:=arabic_MATHEMATICAL_Alphabetic_Symbols}');
Expect(1, 126719, '\p{Block=arabicmathematicalalphabeticsymbols}', "");
Expect(0, 126719, '\p{^Block=arabicmathematicalalphabeticsymbols}', "");
Expect(0, 126719, '\P{Block=arabicmathematicalalphabeticsymbols}', "");
Expect(1, 126719, '\P{^Block=arabicmathematicalalphabeticsymbols}', "");
Expect(0, 126720, '\p{Block=arabicmathematicalalphabeticsymbols}', "");
Expect(1, 126720, '\p{^Block=arabicmathematicalalphabeticsymbols}', "");
Expect(1, 126720, '\P{Block=arabicmathematicalalphabeticsymbols}', "");
Expect(0, 126720, '\P{^Block=arabicmathematicalalphabeticsymbols}', "");
Expect(1, 126719, '\p{Block= 	Arabic_MATHEMATICAL_alphabetic_symbols}', "");
Expect(0, 126719, '\p{^Block= 	Arabic_MATHEMATICAL_alphabetic_symbols}', "");
Expect(0, 126719, '\P{Block= 	Arabic_MATHEMATICAL_alphabetic_symbols}', "");
Expect(1, 126719, '\P{^Block= 	Arabic_MATHEMATICAL_alphabetic_symbols}', "");
Expect(0, 126720, '\p{Block= 	Arabic_MATHEMATICAL_alphabetic_symbols}', "");
Expect(1, 126720, '\p{^Block= 	Arabic_MATHEMATICAL_alphabetic_symbols}', "");
Expect(1, 126720, '\P{Block= 	Arabic_MATHEMATICAL_alphabetic_symbols}', "");
Expect(0, 126720, '\P{^Block= 	Arabic_MATHEMATICAL_alphabetic_symbols}', "");
Error('\p{Blk= 	ARABIC_Math/a/}');
Error('\P{Blk= 	ARABIC_Math/a/}');
Expect(1, 126719, '\p{Blk=arabicmath}', "");
Expect(0, 126719, '\p{^Blk=arabicmath}', "");
Expect(0, 126719, '\P{Blk=arabicmath}', "");
Expect(1, 126719, '\P{^Blk=arabicmath}', "");
Expect(0, 126720, '\p{Blk=arabicmath}', "");
Expect(1, 126720, '\p{^Blk=arabicmath}', "");
Expect(1, 126720, '\P{Blk=arabicmath}', "");
Expect(0, 126720, '\P{^Blk=arabicmath}', "");
Expect(1, 126719, '\p{Blk=		arabic_Math}', "");
Expect(0, 126719, '\p{^Blk=		arabic_Math}', "");
Expect(0, 126719, '\P{Blk=		arabic_Math}', "");
Expect(1, 126719, '\P{^Blk=		arabic_Math}', "");
Expect(0, 126720, '\p{Blk=		arabic_Math}', "");
Expect(1, 126720, '\p{^Blk=		arabic_Math}', "");
Expect(1, 126720, '\P{Blk=		arabic_Math}', "");
Expect(0, 126720, '\P{^Blk=		arabic_Math}', "");
Error('\p{Is_Block=	Arabic_Mathematical_Alphabetic_Symbols:=}');
Error('\P{Is_Block=	Arabic_Mathematical_Alphabetic_Symbols:=}');
Expect(1, 126719, '\p{Is_Block=arabicmathematicalalphabeticsymbols}', "");
Expect(0, 126719, '\p{^Is_Block=arabicmathematicalalphabeticsymbols}', "");
Expect(0, 126719, '\P{Is_Block=arabicmathematicalalphabeticsymbols}', "");
Expect(1, 126719, '\P{^Is_Block=arabicmathematicalalphabeticsymbols}', "");
Expect(0, 126720, '\p{Is_Block=arabicmathematicalalphabeticsymbols}', "");
Expect(1, 126720, '\p{^Is_Block=arabicmathematicalalphabeticsymbols}', "");
Expect(1, 126720, '\P{Is_Block=arabicmathematicalalphabeticsymbols}', "");
Expect(0, 126720, '\P{^Is_Block=arabicmathematicalalphabeticsymbols}', "");
Expect(1, 126719, '\p{Is_Block=_ Arabic_mathematical_ALPHABETIC_symbols}', "");
Expect(0, 126719, '\p{^Is_Block=_ Arabic_mathematical_ALPHABETIC_symbols}', "");
Expect(0, 126719, '\P{Is_Block=_ Arabic_mathematical_ALPHABETIC_symbols}', "");
Expect(1, 126719, '\P{^Is_Block=_ Arabic_mathematical_ALPHABETIC_symbols}', "");
Expect(0, 126720, '\p{Is_Block=_ Arabic_mathematical_ALPHABETIC_symbols}', "");
Expect(1, 126720, '\p{^Is_Block=_ Arabic_mathematical_ALPHABETIC_symbols}', "");
Expect(1, 126720, '\P{Is_Block=_ Arabic_mathematical_ALPHABETIC_symbols}', "");
Expect(0, 126720, '\P{^Is_Block=_ Arabic_mathematical_ALPHABETIC_symbols}', "");
Error('\p{Is_Blk:	/a/-ARABIC_MATH}');
Error('\P{Is_Blk:	/a/-ARABIC_MATH}');
Expect(1, 126719, '\p{Is_Blk=arabicmath}', "");
Expect(0, 126719, '\p{^Is_Blk=arabicmath}', "");
Expect(0, 126719, '\P{Is_Blk=arabicmath}', "");
Expect(1, 126719, '\P{^Is_Blk=arabicmath}', "");
Expect(0, 126720, '\p{Is_Blk=arabicmath}', "");
Expect(1, 126720, '\p{^Is_Blk=arabicmath}', "");
Expect(1, 126720, '\P{Is_Blk=arabicmath}', "");
Expect(0, 126720, '\P{^Is_Blk=arabicmath}', "");
Expect(1, 126719, '\p{Is_Blk=--Arabic_Math}', "");
Expect(0, 126719, '\p{^Is_Blk=--Arabic_Math}', "");
Expect(0, 126719, '\P{Is_Blk=--Arabic_Math}', "");
Expect(1, 126719, '\P{^Is_Blk=--Arabic_Math}', "");
Expect(0, 126720, '\p{Is_Blk=--Arabic_Math}', "");
Expect(1, 126720, '\p{^Is_Blk=--Arabic_Math}', "");
Expect(1, 126720, '\P{Is_Blk=--Arabic_Math}', "");
Expect(0, 126720, '\P{^Is_Blk=--Arabic_Math}', "");
Error('\p{Block=		ARABIC_Presentation_FORMS_A:=}');
Error('\P{Block=		ARABIC_Presentation_FORMS_A:=}');
Expect(1, 65023, '\p{Block=arabicpresentationformsa}', "");
Expect(0, 65023, '\p{^Block=arabicpresentationformsa}', "");
Expect(0, 65023, '\P{Block=arabicpresentationformsa}', "");
Expect(1, 65023, '\P{^Block=arabicpresentationformsa}', "");
Expect(0, 65024, '\p{Block=arabicpresentationformsa}', "");
Expect(1, 65024, '\p{^Block=arabicpresentationformsa}', "");
Expect(1, 65024, '\P{Block=arabicpresentationformsa}', "");
Expect(0, 65024, '\P{^Block=arabicpresentationformsa}', "");
Expect(1, 65023, '\p{Block=-_arabic_presentation_Forms_A}', "");
Expect(0, 65023, '\p{^Block=-_arabic_presentation_Forms_A}', "");
Expect(0, 65023, '\P{Block=-_arabic_presentation_Forms_A}', "");
Expect(1, 65023, '\P{^Block=-_arabic_presentation_Forms_A}', "");
Expect(0, 65024, '\p{Block=-_arabic_presentation_Forms_A}', "");
Expect(1, 65024, '\p{^Block=-_arabic_presentation_Forms_A}', "");
Expect(1, 65024, '\P{Block=-_arabic_presentation_Forms_A}', "");
Expect(0, 65024, '\P{^Block=-_arabic_presentation_Forms_A}', "");
Error('\p{Blk=		ARABIC_pf_A/a/}');
Error('\P{Blk=		ARABIC_pf_A/a/}');
Expect(1, 65023, '\p{Blk=arabicpfa}', "");
Expect(0, 65023, '\p{^Blk=arabicpfa}', "");
Expect(0, 65023, '\P{Blk=arabicpfa}', "");
Expect(1, 65023, '\P{^Blk=arabicpfa}', "");
Expect(0, 65024, '\p{Blk=arabicpfa}', "");
Expect(1, 65024, '\p{^Blk=arabicpfa}', "");
Expect(1, 65024, '\P{Blk=arabicpfa}', "");
Expect(0, 65024, '\P{^Blk=arabicpfa}', "");
Expect(1, 65023, '\p{Blk=_arabic_PF_A}', "");
Expect(0, 65023, '\p{^Blk=_arabic_PF_A}', "");
Expect(0, 65023, '\P{Blk=_arabic_PF_A}', "");
Expect(1, 65023, '\P{^Blk=_arabic_PF_A}', "");
Expect(0, 65024, '\p{Blk=_arabic_PF_A}', "");
Expect(1, 65024, '\p{^Blk=_arabic_PF_A}', "");
Expect(1, 65024, '\P{Blk=_arabic_PF_A}', "");
Expect(0, 65024, '\P{^Blk=_arabic_PF_A}', "");
Error('\p{Is_Block=- arabic_presentation_Forms_A:=}');
Error('\P{Is_Block=- arabic_presentation_Forms_A:=}');
Expect(1, 65023, '\p{Is_Block=arabicpresentationformsa}', "");
Expect(0, 65023, '\p{^Is_Block=arabicpresentationformsa}', "");
Expect(0, 65023, '\P{Is_Block=arabicpresentationformsa}', "");
Expect(1, 65023, '\P{^Is_Block=arabicpresentationformsa}', "");
Expect(0, 65024, '\p{Is_Block=arabicpresentationformsa}', "");
Expect(1, 65024, '\p{^Is_Block=arabicpresentationformsa}', "");
Expect(1, 65024, '\P{Is_Block=arabicpresentationformsa}', "");
Expect(0, 65024, '\P{^Is_Block=arabicpresentationformsa}', "");
Expect(1, 65023, '\p{Is_Block=- Arabic_PRESENTATION_forms_A}', "");
Expect(0, 65023, '\p{^Is_Block=- Arabic_PRESENTATION_forms_A}', "");
Expect(0, 65023, '\P{Is_Block=- Arabic_PRESENTATION_forms_A}', "");
Expect(1, 65023, '\P{^Is_Block=- Arabic_PRESENTATION_forms_A}', "");
Expect(0, 65024, '\p{Is_Block=- Arabic_PRESENTATION_forms_A}', "");
Expect(1, 65024, '\p{^Is_Block=- Arabic_PRESENTATION_forms_A}', "");
Expect(1, 65024, '\P{Is_Block=- Arabic_PRESENTATION_forms_A}', "");
Expect(0, 65024, '\P{^Is_Block=- Arabic_PRESENTATION_forms_A}', "");
Error('\p{Is_Blk= _arabic_pf_A/a/}');
Error('\P{Is_Blk= _arabic_pf_A/a/}');
Expect(1, 65023, '\p{Is_Blk=arabicpfa}', "");
Expect(0, 65023, '\p{^Is_Blk=arabicpfa}', "");
Expect(0, 65023, '\P{Is_Blk=arabicpfa}', "");
Expect(1, 65023, '\P{^Is_Blk=arabicpfa}', "");
Expect(0, 65024, '\p{Is_Blk=arabicpfa}', "");
Expect(1, 65024, '\p{^Is_Blk=arabicpfa}', "");
Expect(1, 65024, '\P{Is_Blk=arabicpfa}', "");
Expect(0, 65024, '\P{^Is_Blk=arabicpfa}', "");
Expect(1, 65023, '\p{Is_Blk=-_ARABIC_PF_A}', "");
Expect(0, 65023, '\p{^Is_Blk=-_ARABIC_PF_A}', "");
Expect(0, 65023, '\P{Is_Blk=-_ARABIC_PF_A}', "");
Expect(1, 65023, '\P{^Is_Blk=-_ARABIC_PF_A}', "");
Expect(0, 65024, '\p{Is_Blk=-_ARABIC_PF_A}', "");
Expect(1, 65024, '\p{^Is_Blk=-_ARABIC_PF_A}', "");
Expect(1, 65024, '\P{Is_Blk=-_ARABIC_PF_A}', "");
Expect(0, 65024, '\P{^Is_Blk=-_ARABIC_PF_A}', "");
Error('\p{Block=-/a/ARABIC_presentation_FORMS_B}');
Error('\P{Block=-/a/ARABIC_presentation_FORMS_B}');
Expect(1, 65279, '\p{Block=arabicpresentationformsb}', "");
Expect(0, 65279, '\p{^Block=arabicpresentationformsb}', "");
Expect(0, 65279, '\P{Block=arabicpresentationformsb}', "");
Expect(1, 65279, '\P{^Block=arabicpresentationformsb}', "");
Expect(0, 65280, '\p{Block=arabicpresentationformsb}', "");
Expect(1, 65280, '\p{^Block=arabicpresentationformsb}', "");
Expect(1, 65280, '\P{Block=arabicpresentationformsb}', "");
Expect(0, 65280, '\P{^Block=arabicpresentationformsb}', "");
Expect(1, 65279, '\p{Block=		ARABIC_Presentation_Forms_B}', "");
Expect(0, 65279, '\p{^Block=		ARABIC_Presentation_Forms_B}', "");
Expect(0, 65279, '\P{Block=		ARABIC_Presentation_Forms_B}', "");
Expect(1, 65279, '\P{^Block=		ARABIC_Presentation_Forms_B}', "");
Expect(0, 65280, '\p{Block=		ARABIC_Presentation_Forms_B}', "");
Expect(1, 65280, '\p{^Block=		ARABIC_Presentation_Forms_B}', "");
Expect(1, 65280, '\P{Block=		ARABIC_Presentation_Forms_B}', "");
Expect(0, 65280, '\P{^Block=		ARABIC_Presentation_Forms_B}', "");
Error('\p{Blk= :=ARABIC_PF_B}');
Error('\P{Blk= :=ARABIC_PF_B}');
Expect(1, 65279, '\p{Blk=arabicpfb}', "");
Expect(0, 65279, '\p{^Blk=arabicpfb}', "");
Expect(0, 65279, '\P{Blk=arabicpfb}', "");
Expect(1, 65279, '\P{^Blk=arabicpfb}', "");
Expect(0, 65280, '\p{Blk=arabicpfb}', "");
Expect(1, 65280, '\p{^Blk=arabicpfb}', "");
Expect(1, 65280, '\P{Blk=arabicpfb}', "");
Expect(0, 65280, '\P{^Blk=arabicpfb}', "");
Expect(1, 65279, '\p{Blk=	-Arabic_PF_b}', "");
Expect(0, 65279, '\p{^Blk=	-Arabic_PF_b}', "");
Expect(0, 65279, '\P{Blk=	-Arabic_PF_b}', "");
Expect(1, 65279, '\P{^Blk=	-Arabic_PF_b}', "");
Expect(0, 65280, '\p{Blk=	-Arabic_PF_b}', "");
Expect(1, 65280, '\p{^Blk=	-Arabic_PF_b}', "");
Expect(1, 65280, '\P{Blk=	-Arabic_PF_b}', "");
Expect(0, 65280, '\P{^Blk=	-Arabic_PF_b}', "");
Error('\p{Is_Block=-	arabic_presentation_forms_B/a/}');
Error('\P{Is_Block=-	arabic_presentation_forms_B/a/}');
Expect(1, 65279, '\p{Is_Block=arabicpresentationformsb}', "");
Expect(0, 65279, '\p{^Is_Block=arabicpresentationformsb}', "");
Expect(0, 65279, '\P{Is_Block=arabicpresentationformsb}', "");
Expect(1, 65279, '\P{^Is_Block=arabicpresentationformsb}', "");
Expect(0, 65280, '\p{Is_Block=arabicpresentationformsb}', "");
Expect(1, 65280, '\p{^Is_Block=arabicpresentationformsb}', "");
Expect(1, 65280, '\P{Is_Block=arabicpresentationformsb}', "");
Expect(0, 65280, '\P{^Is_Block=arabicpresentationformsb}', "");
Expect(1, 65279, '\p{Is_Block=	Arabic_Presentation_Forms_B}', "");
Expect(0, 65279, '\p{^Is_Block=	Arabic_Presentation_Forms_B}', "");
Expect(0, 65279, '\P{Is_Block=	Arabic_Presentation_Forms_B}', "");
Expect(1, 65279, '\P{^Is_Block=	Arabic_Presentation_Forms_B}', "");
Expect(0, 65280, '\p{Is_Block=	Arabic_Presentation_Forms_B}', "");
Expect(1, 65280, '\p{^Is_Block=	Arabic_Presentation_Forms_B}', "");
Expect(1, 65280, '\P{Is_Block=	Arabic_Presentation_Forms_B}', "");
Expect(0, 65280, '\P{^Is_Block=	Arabic_Presentation_Forms_B}', "");
Error('\p{Is_Blk=__arabic_PF_B/a/}');
Error('\P{Is_Blk=__arabic_PF_B/a/}');
Expect(1, 65279, '\p{Is_Blk=arabicpfb}', "");
Expect(0, 65279, '\p{^Is_Blk=arabicpfb}', "");
Expect(0, 65279, '\P{Is_Blk=arabicpfb}', "");
Expect(1, 65279, '\P{^Is_Blk=arabicpfb}', "");
Expect(0, 65280, '\p{Is_Blk=arabicpfb}', "");
Expect(1, 65280, '\p{^Is_Blk=arabicpfb}', "");
Expect(1, 65280, '\P{Is_Blk=arabicpfb}', "");
Expect(0, 65280, '\P{^Is_Blk=arabicpfb}', "");
Expect(1, 65279, '\p{Is_Blk= -Arabic_pf_b}', "");
Expect(0, 65279, '\p{^Is_Blk= -Arabic_pf_b}', "");
Expect(0, 65279, '\P{Is_Blk= -Arabic_pf_b}', "");
Expect(1, 65279, '\P{^Is_Blk= -Arabic_pf_b}', "");
Expect(0, 65280, '\p{Is_Blk= -Arabic_pf_b}', "");
Expect(1, 65280, '\p{^Is_Blk= -Arabic_pf_b}', "");
Expect(1, 65280, '\P{Is_Blk= -Arabic_pf_b}', "");
Expect(0, 65280, '\P{^Is_Blk= -Arabic_pf_b}', "");
Error('\p{Block=-:=Arabic_supplement}');
Error('\P{Block=-:=Arabic_supplement}');
Expect(1, 1919, '\p{Block=arabicsupplement}', "");
Expect(0, 1919, '\p{^Block=arabicsupplement}', "");
Expect(0, 1919, '\P{Block=arabicsupplement}', "");
Expect(1, 1919, '\P{^Block=arabicsupplement}', "");
Expect(0, 1920, '\p{Block=arabicsupplement}', "");
Expect(1, 1920, '\p{^Block=arabicsupplement}', "");
Expect(1, 1920, '\P{Block=arabicsupplement}', "");
Expect(0, 1920, '\P{^Block=arabicsupplement}', "");
Expect(1, 1919, '\p{Block= arabic_SUPPLEMENT}', "");
Expect(0, 1919, '\p{^Block= arabic_SUPPLEMENT}', "");
Expect(0, 1919, '\P{Block= arabic_SUPPLEMENT}', "");
Expect(1, 1919, '\P{^Block= arabic_SUPPLEMENT}', "");
Expect(0, 1920, '\p{Block= arabic_SUPPLEMENT}', "");
Expect(1, 1920, '\p{^Block= arabic_SUPPLEMENT}', "");
Expect(1, 1920, '\P{Block= arabic_SUPPLEMENT}', "");
Expect(0, 1920, '\P{^Block= arabic_SUPPLEMENT}', "");
Error('\p{Blk=:= Arabic_Sup}');
Error('\P{Blk=:= Arabic_Sup}');
Expect(1, 1919, '\p{Blk=arabicsup}', "");
Expect(0, 1919, '\p{^Blk=arabicsup}', "");
Expect(0, 1919, '\P{Blk=arabicsup}', "");
Expect(1, 1919, '\P{^Blk=arabicsup}', "");
Expect(0, 1920, '\p{Blk=arabicsup}', "");
Expect(1, 1920, '\p{^Blk=arabicsup}', "");
Expect(1, 1920, '\P{Blk=arabicsup}', "");
Expect(0, 1920, '\P{^Blk=arabicsup}', "");
Expect(1, 1919, '\p{Blk:	_ Arabic_SUP}', "");
Expect(0, 1919, '\p{^Blk:	_ Arabic_SUP}', "");
Expect(0, 1919, '\P{Blk:	_ Arabic_SUP}', "");
Expect(1, 1919, '\P{^Blk:	_ Arabic_SUP}', "");
Expect(0, 1920, '\p{Blk:	_ Arabic_SUP}', "");
Expect(1, 1920, '\p{^Blk:	_ Arabic_SUP}', "");
Expect(1, 1920, '\P{Blk:	_ Arabic_SUP}', "");
Expect(0, 1920, '\P{^Blk:	_ Arabic_SUP}', "");
Error('\p{Is_Block=:= 	Arabic_Supplement}');
Error('\P{Is_Block=:= 	Arabic_Supplement}');
Expect(1, 1919, '\p{Is_Block=arabicsupplement}', "");
Expect(0, 1919, '\p{^Is_Block=arabicsupplement}', "");
Expect(0, 1919, '\P{Is_Block=arabicsupplement}', "");
Expect(1, 1919, '\P{^Is_Block=arabicsupplement}', "");
Expect(0, 1920, '\p{Is_Block=arabicsupplement}', "");
Expect(1, 1920, '\p{^Is_Block=arabicsupplement}', "");
Expect(1, 1920, '\P{Is_Block=arabicsupplement}', "");
Expect(0, 1920, '\P{^Is_Block=arabicsupplement}', "");
Expect(1, 1919, '\p{Is_Block=_	Arabic_Supplement}', "");
Expect(0, 1919, '\p{^Is_Block=_	Arabic_Supplement}', "");
Expect(0, 1919, '\P{Is_Block=_	Arabic_Supplement}', "");
Expect(1, 1919, '\P{^Is_Block=_	Arabic_Supplement}', "");
Expect(0, 1920, '\p{Is_Block=_	Arabic_Supplement}', "");
Expect(1, 1920, '\p{^Is_Block=_	Arabic_Supplement}', "");
Expect(1, 1920, '\P{Is_Block=_	Arabic_Supplement}', "");
Expect(0, 1920, '\P{^Is_Block=_	Arabic_Supplement}', "");
Error('\p{Is_Blk=/a/arabic_Sup}');
Error('\P{Is_Blk=/a/arabic_Sup}');
Expect(1, 1919, '\p{Is_Blk=arabicsup}', "");
Expect(0, 1919, '\p{^Is_Blk=arabicsup}', "");
Expect(0, 1919, '\P{Is_Blk=arabicsup}', "");
Expect(1, 1919, '\P{^Is_Blk=arabicsup}', "");
Expect(0, 1920, '\p{Is_Blk=arabicsup}', "");
Expect(1, 1920, '\p{^Is_Blk=arabicsup}', "");
Expect(1, 1920, '\P{Is_Blk=arabicsup}', "");
Expect(0, 1920, '\P{^Is_Blk=arabicsup}', "");
Expect(1, 1919, '\p{Is_Blk=		Arabic_Sup}', "");
Expect(0, 1919, '\p{^Is_Blk=		Arabic_Sup}', "");
Expect(0, 1919, '\P{Is_Blk=		Arabic_Sup}', "");
Expect(1, 1919, '\P{^Is_Blk=		Arabic_Sup}', "");
Expect(0, 1920, '\p{Is_Blk=		Arabic_Sup}', "");
Expect(1, 1920, '\p{^Is_Blk=		Arabic_Sup}', "");
Expect(1, 1920, '\P{Is_Blk=		Arabic_Sup}', "");
Expect(0, 1920, '\P{^Is_Blk=		Arabic_Sup}', "");
Error('\p{Block=:=ARMENIAN}');
Error('\P{Block=:=ARMENIAN}');
Expect(1, 1423, '\p{Block=armenian}', "");
Expect(0, 1423, '\p{^Block=armenian}', "");
Expect(0, 1423, '\P{Block=armenian}', "");
Expect(1, 1423, '\P{^Block=armenian}', "");
Expect(0, 1424, '\p{Block=armenian}', "");
Expect(1, 1424, '\p{^Block=armenian}', "");
Expect(1, 1424, '\P{Block=armenian}', "");
Expect(0, 1424, '\P{^Block=armenian}', "");
Expect(1, 1423, '\p{Block=_	Armenian}', "");
Expect(0, 1423, '\p{^Block=_	Armenian}', "");
Expect(0, 1423, '\P{Block=_	Armenian}', "");
Expect(1, 1423, '\P{^Block=_	Armenian}', "");
Expect(0, 1424, '\p{Block=_	Armenian}', "");
Expect(1, 1424, '\p{^Block=_	Armenian}', "");
Expect(1, 1424, '\P{Block=_	Armenian}', "");
Expect(0, 1424, '\P{^Block=_	Armenian}', "");
Error('\p{Blk= 	armenian/a/}');
Error('\P{Blk= 	armenian/a/}');
Expect(1, 1423, '\p{Blk=armenian}', "");
Expect(0, 1423, '\p{^Blk=armenian}', "");
Expect(0, 1423, '\P{Blk=armenian}', "");
Expect(1, 1423, '\P{^Blk=armenian}', "");
Expect(0, 1424, '\p{Blk=armenian}', "");
Expect(1, 1424, '\p{^Blk=armenian}', "");
Expect(1, 1424, '\P{Blk=armenian}', "");
Expect(0, 1424, '\P{^Blk=armenian}', "");
Expect(1, 1423, '\p{Blk=	Armenian}', "");
Expect(0, 1423, '\p{^Blk=	Armenian}', "");
Expect(0, 1423, '\P{Blk=	Armenian}', "");
Expect(1, 1423, '\P{^Blk=	Armenian}', "");
Expect(0, 1424, '\p{Blk=	Armenian}', "");
Expect(1, 1424, '\p{^Blk=	Armenian}', "");
Expect(1, 1424, '\P{Blk=	Armenian}', "");
Expect(0, 1424, '\P{^Blk=	Armenian}', "");
Error('\p{Is_Block=_-armenian:=}');
Error('\P{Is_Block=_-armenian:=}');
Expect(1, 1423, '\p{Is_Block=armenian}', "");
Expect(0, 1423, '\p{^Is_Block=armenian}', "");
Expect(0, 1423, '\P{Is_Block=armenian}', "");
Expect(1, 1423, '\P{^Is_Block=armenian}', "");
Expect(0, 1424, '\p{Is_Block=armenian}', "");
Expect(1, 1424, '\p{^Is_Block=armenian}', "");
Expect(1, 1424, '\P{Is_Block=armenian}', "");
Expect(0, 1424, '\P{^Is_Block=armenian}', "");
Expect(1, 1423, '\p{Is_Block= ARMENIAN}', "");
Expect(0, 1423, '\p{^Is_Block= ARMENIAN}', "");
Expect(0, 1423, '\P{Is_Block= ARMENIAN}', "");
Expect(1, 1423, '\P{^Is_Block= ARMENIAN}', "");
Expect(0, 1424, '\p{Is_Block= ARMENIAN}', "");
Expect(1, 1424, '\p{^Is_Block= ARMENIAN}', "");
Expect(1, 1424, '\P{Is_Block= ARMENIAN}', "");
Expect(0, 1424, '\P{^Is_Block= ARMENIAN}', "");
Error('\p{Is_Blk=:=	-Armenian}');
Error('\P{Is_Blk=:=	-Armenian}');
Expect(1, 1423, '\p{Is_Blk=armenian}', "");
Expect(0, 1423, '\p{^Is_Blk=armenian}', "");
Expect(0, 1423, '\P{Is_Blk=armenian}', "");
Expect(1, 1423, '\P{^Is_Blk=armenian}', "");
Expect(0, 1424, '\p{Is_Blk=armenian}', "");
Expect(1, 1424, '\p{^Is_Blk=armenian}', "");
Expect(1, 1424, '\P{Is_Blk=armenian}', "");
Expect(0, 1424, '\P{^Is_Blk=armenian}', "");
Expect(1, 1423, '\p{Is_Blk=_Armenian}', "");
Expect(0, 1423, '\p{^Is_Blk=_Armenian}', "");
Expect(0, 1423, '\P{Is_Blk=_Armenian}', "");
Expect(1, 1423, '\P{^Is_Blk=_Armenian}', "");
Expect(0, 1424, '\p{Is_Blk=_Armenian}', "");
Expect(1, 1424, '\p{^Is_Blk=_Armenian}', "");
Expect(1, 1424, '\P{Is_Blk=_Armenian}', "");
Expect(0, 1424, '\P{^Is_Blk=_Armenian}', "");
Error('\p{Block=-_ARROWS:=}');
Error('\P{Block=-_ARROWS:=}');
Expect(1, 8703, '\p{Block=arrows}', "");
Expect(0, 8703, '\p{^Block=arrows}', "");
Expect(0, 8703, '\P{Block=arrows}', "");
Expect(1, 8703, '\P{^Block=arrows}', "");
Expect(0, 8704, '\p{Block=arrows}', "");
Expect(1, 8704, '\p{^Block=arrows}', "");
Expect(1, 8704, '\P{Block=arrows}', "");
Expect(0, 8704, '\P{^Block=arrows}', "");
Expect(1, 8703, '\p{Block=-ARROWS}', "");
Expect(0, 8703, '\p{^Block=-ARROWS}', "");
Expect(0, 8703, '\P{Block=-ARROWS}', "");
Expect(1, 8703, '\P{^Block=-ARROWS}', "");
Expect(0, 8704, '\p{Block=-ARROWS}', "");
Expect(1, 8704, '\p{^Block=-ARROWS}', "");
Expect(1, 8704, '\P{Block=-ARROWS}', "");
Expect(0, 8704, '\P{^Block=-ARROWS}', "");
Error('\p{Blk:/a/--ARROWS}');
Error('\P{Blk:/a/--ARROWS}');
Expect(1, 8703, '\p{Blk=arrows}', "");
Expect(0, 8703, '\p{^Blk=arrows}', "");
Expect(0, 8703, '\P{Blk=arrows}', "");
Expect(1, 8703, '\P{^Blk=arrows}', "");
Expect(0, 8704, '\p{Blk=arrows}', "");
Expect(1, 8704, '\p{^Blk=arrows}', "");
Expect(1, 8704, '\P{Blk=arrows}', "");
Expect(0, 8704, '\P{^Blk=arrows}', "");
Expect(1, 8703, '\p{Blk=-	ARROWS}', "");
Expect(0, 8703, '\p{^Blk=-	ARROWS}', "");
Expect(0, 8703, '\P{Blk=-	ARROWS}', "");
Expect(1, 8703, '\P{^Blk=-	ARROWS}', "");
Expect(0, 8704, '\p{Blk=-	ARROWS}', "");
Expect(1, 8704, '\p{^Blk=-	ARROWS}', "");
Expect(1, 8704, '\P{Blk=-	ARROWS}', "");
Expect(0, 8704, '\P{^Blk=-	ARROWS}', "");
Error('\p{Is_Block=	:=Arrows}');
Error('\P{Is_Block=	:=Arrows}');
Expect(1, 8703, '\p{Is_Block: arrows}', "");
Expect(0, 8703, '\p{^Is_Block: arrows}', "");
Expect(0, 8703, '\P{Is_Block: arrows}', "");
Expect(1, 8703, '\P{^Is_Block: arrows}', "");
Expect(0, 8704, '\p{Is_Block: arrows}', "");
Expect(1, 8704, '\p{^Is_Block: arrows}', "");
Expect(1, 8704, '\P{Is_Block: arrows}', "");
Expect(0, 8704, '\P{^Is_Block: arrows}', "");
Expect(1, 8703, '\p{Is_Block= -arrows}', "");
Expect(0, 8703, '\p{^Is_Block= -arrows}', "");
Expect(0, 8703, '\P{Is_Block= -arrows}', "");
Expect(1, 8703, '\P{^Is_Block= -arrows}', "");
Expect(0, 8704, '\p{Is_Block= -arrows}', "");
Expect(1, 8704, '\p{^Is_Block= -arrows}', "");
Expect(1, 8704, '\P{Is_Block= -arrows}', "");
Expect(0, 8704, '\P{^Is_Block= -arrows}', "");
Error('\p{Is_Blk=/a/-_Arrows}');
Error('\P{Is_Blk=/a/-_Arrows}');
Expect(1, 8703, '\p{Is_Blk=arrows}', "");
Expect(0, 8703, '\p{^Is_Blk=arrows}', "");
Expect(0, 8703, '\P{Is_Blk=arrows}', "");
Expect(1, 8703, '\P{^Is_Blk=arrows}', "");
Expect(0, 8704, '\p{Is_Blk=arrows}', "");
Expect(1, 8704, '\p{^Is_Blk=arrows}', "");
Expect(1, 8704, '\P{Is_Blk=arrows}', "");
Expect(0, 8704, '\P{^Is_Blk=arrows}', "");
Expect(1, 8703, '\p{Is_Blk=-_Arrows}', "");
Expect(0, 8703, '\p{^Is_Blk=-_Arrows}', "");
Expect(0, 8703, '\P{Is_Blk=-_Arrows}', "");
Expect(1, 8703, '\P{^Is_Blk=-_Arrows}', "");
Expect(0, 8704, '\p{Is_Blk=-_Arrows}', "");
Expect(1, 8704, '\p{^Is_Blk=-_Arrows}', "");
Expect(1, 8704, '\P{Is_Blk=-_Arrows}', "");
Expect(0, 8704, '\P{^Is_Blk=-_Arrows}', "");
Error('\p{Block:_Avestan:=}');
Error('\P{Block:_Avestan:=}');
Expect(1, 68415, '\p{Block=avestan}', "");
Expect(0, 68415, '\p{^Block=avestan}', "");
Expect(0, 68415, '\P{Block=avestan}', "");
Expect(1, 68415, '\P{^Block=avestan}', "");
Expect(0, 68416, '\p{Block=avestan}', "");
Expect(1, 68416, '\p{^Block=avestan}', "");
Expect(1, 68416, '\P{Block=avestan}', "");
Expect(0, 68416, '\P{^Block=avestan}', "");
Expect(1, 68415, '\p{Block=	 avestan}', "");
Expect(0, 68415, '\p{^Block=	 avestan}', "");
Expect(0, 68415, '\P{Block=	 avestan}', "");
Expect(1, 68415, '\P{^Block=	 avestan}', "");
Expect(0, 68416, '\p{Block=	 avestan}', "");
Expect(1, 68416, '\p{^Block=	 avestan}', "");
Expect(1, 68416, '\P{Block=	 avestan}', "");
Expect(0, 68416, '\P{^Block=	 avestan}', "");
Error('\p{Blk=/a/-	Avestan}');
Error('\P{Blk=/a/-	Avestan}');
Expect(1, 68415, '\p{Blk=avestan}', "");
Expect(0, 68415, '\p{^Blk=avestan}', "");
Expect(0, 68415, '\P{Blk=avestan}', "");
Expect(1, 68415, '\P{^Blk=avestan}', "");
Expect(0, 68416, '\p{Blk=avestan}', "");
Expect(1, 68416, '\p{^Blk=avestan}', "");
Expect(1, 68416, '\P{Blk=avestan}', "");
Expect(0, 68416, '\P{^Blk=avestan}', "");
Expect(1, 68415, '\p{Blk: -AVESTAN}', "");
Expect(0, 68415, '\p{^Blk: -AVESTAN}', "");
Expect(0, 68415, '\P{Blk: -AVESTAN}', "");
Expect(1, 68415, '\P{^Blk: -AVESTAN}', "");
Expect(0, 68416, '\p{Blk: -AVESTAN}', "");
Expect(1, 68416, '\p{^Blk: -AVESTAN}', "");
Expect(1, 68416, '\P{Blk: -AVESTAN}', "");
Expect(0, 68416, '\P{^Blk: -AVESTAN}', "");
Error('\p{Is_Block=-	avestan/a/}');
Error('\P{Is_Block=-	avestan/a/}');
Expect(1, 68415, '\p{Is_Block=avestan}', "");
Expect(0, 68415, '\p{^Is_Block=avestan}', "");
Expect(0, 68415, '\P{Is_Block=avestan}', "");
Expect(1, 68415, '\P{^Is_Block=avestan}', "");
Expect(0, 68416, '\p{Is_Block=avestan}', "");
Expect(1, 68416, '\p{^Is_Block=avestan}', "");
Expect(1, 68416, '\P{Is_Block=avestan}', "");
Expect(0, 68416, '\P{^Is_Block=avestan}', "");
Expect(1, 68415, '\p{Is_Block=--Avestan}', "");
Expect(0, 68415, '\p{^Is_Block=--Avestan}', "");
Expect(0, 68415, '\P{Is_Block=--Avestan}', "");
Expect(1, 68415, '\P{^Is_Block=--Avestan}', "");
Expect(0, 68416, '\p{Is_Block=--Avestan}', "");
Expect(1, 68416, '\p{^Is_Block=--Avestan}', "");
Expect(1, 68416, '\P{Is_Block=--Avestan}', "");
Expect(0, 68416, '\P{^Is_Block=--Avestan}', "");
Error('\p{Is_Blk=-/a/avestan}');
Error('\P{Is_Blk=-/a/avestan}');
Expect(1, 68415, '\p{Is_Blk:avestan}', "");
Expect(0, 68415, '\p{^Is_Blk:avestan}', "");
Expect(0, 68415, '\P{Is_Blk:avestan}', "");
Expect(1, 68415, '\P{^Is_Blk:avestan}', "");
Expect(0, 68416, '\p{Is_Blk:avestan}', "");
Expect(1, 68416, '\p{^Is_Blk:avestan}', "");
Expect(1, 68416, '\P{Is_Blk:avestan}', "");
Expect(0, 68416, '\P{^Is_Blk:avestan}', "");
Expect(1, 68415, '\p{Is_Blk=_avestan}', "");
Expect(0, 68415, '\p{^Is_Blk=_avestan}', "");
Expect(0, 68415, '\P{Is_Blk=_avestan}', "");
Expect(1, 68415, '\P{^Is_Blk=_avestan}', "");
Expect(0, 68416, '\p{Is_Blk=_avestan}', "");
Expect(1, 68416, '\p{^Is_Blk=_avestan}', "");
Expect(1, 68416, '\P{Is_Blk=_avestan}', "");
Expect(0, 68416, '\P{^Is_Blk=_avestan}', "");
Error('\p{Block=:=	_BALINESE}');
Error('\P{Block=:=	_BALINESE}');
Expect(1, 7039, '\p{Block=balinese}', "");
Expect(0, 7039, '\p{^Block=balinese}', "");
Expect(0, 7039, '\P{Block=balinese}', "");
Expect(1, 7039, '\P{^Block=balinese}', "");
Expect(0, 7040, '\p{Block=balinese}', "");
Expect(1, 7040, '\p{^Block=balinese}', "");
Expect(1, 7040, '\P{Block=balinese}', "");
Expect(0, 7040, '\P{^Block=balinese}', "");
Expect(1, 7039, '\p{Block=-	Balinese}', "");
Expect(0, 7039, '\p{^Block=-	Balinese}', "");
Expect(0, 7039, '\P{Block=-	Balinese}', "");
Expect(1, 7039, '\P{^Block=-	Balinese}', "");
Expect(0, 7040, '\p{Block=-	Balinese}', "");
Expect(1, 7040, '\p{^Block=-	Balinese}', "");
Expect(1, 7040, '\P{Block=-	Balinese}', "");
Expect(0, 7040, '\P{^Block=-	Balinese}', "");
Error('\p{Blk=:= 	Balinese}');
Error('\P{Blk=:= 	Balinese}');
Expect(1, 7039, '\p{Blk=balinese}', "");
Expect(0, 7039, '\p{^Blk=balinese}', "");
Expect(0, 7039, '\P{Blk=balinese}', "");
Expect(1, 7039, '\P{^Blk=balinese}', "");
Expect(0, 7040, '\p{Blk=balinese}', "");
Expect(1, 7040, '\p{^Blk=balinese}', "");
Expect(1, 7040, '\P{Blk=balinese}', "");
Expect(0, 7040, '\P{^Blk=balinese}', "");
Expect(1, 7039, '\p{Blk=	Balinese}', "");
Expect(0, 7039, '\p{^Blk=	Balinese}', "");
Expect(0, 7039, '\P{Blk=	Balinese}', "");
Expect(1, 7039, '\P{^Blk=	Balinese}', "");
Expect(0, 7040, '\p{Blk=	Balinese}', "");
Expect(1, 7040, '\p{^Blk=	Balinese}', "");
Expect(1, 7040, '\P{Blk=	Balinese}', "");
Expect(0, 7040, '\P{^Blk=	Balinese}', "");
Error('\p{Is_Block=_/a/BALINESE}');
Error('\P{Is_Block=_/a/BALINESE}');
Expect(1, 7039, '\p{Is_Block=balinese}', "");
Expect(0, 7039, '\p{^Is_Block=balinese}', "");
Expect(0, 7039, '\P{Is_Block=balinese}', "");
Expect(1, 7039, '\P{^Is_Block=balinese}', "");
Expect(0, 7040, '\p{Is_Block=balinese}', "");
Expect(1, 7040, '\p{^Is_Block=balinese}', "");
Expect(1, 7040, '\P{Is_Block=balinese}', "");
Expect(0, 7040, '\P{^Is_Block=balinese}', "");
Expect(1, 7039, '\p{Is_Block= _Balinese}', "");
Expect(0, 7039, '\p{^Is_Block= _Balinese}', "");
Expect(0, 7039, '\P{Is_Block= _Balinese}', "");
Expect(1, 7039, '\P{^Is_Block= _Balinese}', "");
Expect(0, 7040, '\p{Is_Block= _Balinese}', "");
Expect(1, 7040, '\p{^Is_Block= _Balinese}', "");
Expect(1, 7040, '\P{Is_Block= _Balinese}', "");
Expect(0, 7040, '\P{^Is_Block= _Balinese}', "");
Error('\p{Is_Blk=:=_balinese}');
Error('\P{Is_Blk=:=_balinese}');
Expect(1, 7039, '\p{Is_Blk=balinese}', "");
Expect(0, 7039, '\p{^Is_Blk=balinese}', "");
Expect(0, 7039, '\P{Is_Blk=balinese}', "");
Expect(1, 7039, '\P{^Is_Blk=balinese}', "");
Expect(0, 7040, '\p{Is_Blk=balinese}', "");
Expect(1, 7040, '\p{^Is_Blk=balinese}', "");
Expect(1, 7040, '\P{Is_Blk=balinese}', "");
Expect(0, 7040, '\P{^Is_Blk=balinese}', "");
Expect(1, 7039, '\p{Is_Blk=_Balinese}', "");
Expect(0, 7039, '\p{^Is_Blk=_Balinese}', "");
Expect(0, 7039, '\P{Is_Blk=_Balinese}', "");
Expect(1, 7039, '\P{^Is_Blk=_Balinese}', "");
Expect(0, 7040, '\p{Is_Blk=_Balinese}', "");
Expect(1, 7040, '\p{^Is_Blk=_Balinese}', "");
Expect(1, 7040, '\P{Is_Blk=_Balinese}', "");
Expect(0, 7040, '\P{^Is_Blk=_Balinese}', "");
Error('\p{Block= /a/Bamum}');
Error('\P{Block= /a/Bamum}');
Expect(1, 42751, '\p{Block=bamum}', "");
Expect(0, 42751, '\p{^Block=bamum}', "");
Expect(0, 42751, '\P{Block=bamum}', "");
Expect(1, 42751, '\P{^Block=bamum}', "");
Expect(0, 42752, '\p{Block=bamum}', "");
Expect(1, 42752, '\p{^Block=bamum}', "");
Expect(1, 42752, '\P{Block=bamum}', "");
Expect(0, 42752, '\P{^Block=bamum}', "");
Expect(1, 42751, '\p{Block= Bamum}', "");
Expect(0, 42751, '\p{^Block= Bamum}', "");
Expect(0, 42751, '\P{Block= Bamum}', "");
Expect(1, 42751, '\P{^Block= Bamum}', "");
Expect(0, 42752, '\p{Block= Bamum}', "");
Expect(1, 42752, '\p{^Block= Bamum}', "");
Expect(1, 42752, '\P{Block= Bamum}', "");
Expect(0, 42752, '\P{^Block= Bamum}', "");
Error('\p{Blk=_/a/BAMUM}');
Error('\P{Blk=_/a/BAMUM}');
Expect(1, 42751, '\p{Blk=bamum}', "");
Expect(0, 42751, '\p{^Blk=bamum}', "");
Expect(0, 42751, '\P{Blk=bamum}', "");
Expect(1, 42751, '\P{^Blk=bamum}', "");
Expect(0, 42752, '\p{Blk=bamum}', "");
Expect(1, 42752, '\p{^Blk=bamum}', "");
Expect(1, 42752, '\P{Blk=bamum}', "");
Expect(0, 42752, '\P{^Blk=bamum}', "");
Expect(1, 42751, '\p{Blk:   _-Bamum}', "");
Expect(0, 42751, '\p{^Blk:   _-Bamum}', "");
Expect(0, 42751, '\P{Blk:   _-Bamum}', "");
Expect(1, 42751, '\P{^Blk:   _-Bamum}', "");
Expect(0, 42752, '\p{Blk:   _-Bamum}', "");
Expect(1, 42752, '\p{^Blk:   _-Bamum}', "");
Expect(1, 42752, '\P{Blk:   _-Bamum}', "");
Expect(0, 42752, '\P{^Blk:   _-Bamum}', "");
Error('\p{Is_Block:		Bamum:=}');
Error('\P{Is_Block:		Bamum:=}');
Expect(1, 42751, '\p{Is_Block=bamum}', "");
Expect(0, 42751, '\p{^Is_Block=bamum}', "");
Expect(0, 42751, '\P{Is_Block=bamum}', "");
Expect(1, 42751, '\P{^Is_Block=bamum}', "");
Expect(0, 42752, '\p{Is_Block=bamum}', "");
Expect(1, 42752, '\p{^Is_Block=bamum}', "");
Expect(1, 42752, '\P{Is_Block=bamum}', "");
Expect(0, 42752, '\P{^Is_Block=bamum}', "");
Expect(1, 42751, '\p{Is_Block=_Bamum}', "");
Expect(0, 42751, '\p{^Is_Block=_Bamum}', "");
Expect(0, 42751, '\P{Is_Block=_Bamum}', "");
Expect(1, 42751, '\P{^Is_Block=_Bamum}', "");
Expect(0, 42752, '\p{Is_Block=_Bamum}', "");
Expect(1, 42752, '\p{^Is_Block=_Bamum}', "");
Expect(1, 42752, '\P{Is_Block=_Bamum}', "");
Expect(0, 42752, '\P{^Is_Block=_Bamum}', "");
Error('\p{Is_Blk=/a/BAMUM}');
Error('\P{Is_Blk=/a/BAMUM}');
Expect(1, 42751, '\p{Is_Blk=bamum}', "");
Expect(0, 42751, '\p{^Is_Blk=bamum}', "");
Expect(0, 42751, '\P{Is_Blk=bamum}', "");
Expect(1, 42751, '\P{^Is_Blk=bamum}', "");
Expect(0, 42752, '\p{Is_Blk=bamum}', "");
Expect(1, 42752, '\p{^Is_Blk=bamum}', "");
Expect(1, 42752, '\P{Is_Blk=bamum}', "");
Expect(0, 42752, '\P{^Is_Blk=bamum}', "");
Expect(1, 42751, '\p{Is_Blk= Bamum}', "");
Expect(0, 42751, '\p{^Is_Blk= Bamum}', "");
Expect(0, 42751, '\P{Is_Blk= Bamum}', "");
Expect(1, 42751, '\P{^Is_Blk= Bamum}', "");
Expect(0, 42752, '\p{Is_Blk= Bamum}', "");
Expect(1, 42752, '\p{^Is_Blk= Bamum}', "");
Expect(1, 42752, '\P{Is_Blk= Bamum}', "");
Expect(0, 42752, '\P{^Is_Blk= Bamum}', "");
Error('\p{Block=/a/	 Bamum_SUPPLEMENT}');
Error('\P{Block=/a/	 Bamum_SUPPLEMENT}');
Expect(1, 92735, '\p{Block=bamumsupplement}', "");
Expect(0, 92735, '\p{^Block=bamumsupplement}', "");
Expect(0, 92735, '\P{Block=bamumsupplement}', "");
Expect(1, 92735, '\P{^Block=bamumsupplement}', "");
Expect(0, 92736, '\p{Block=bamumsupplement}', "");
Expect(1, 92736, '\p{^Block=bamumsupplement}', "");
Expect(1, 92736, '\P{Block=bamumsupplement}', "");
Expect(0, 92736, '\P{^Block=bamumsupplement}', "");
Expect(1, 92735, '\p{Block=-Bamum_supplement}', "");
Expect(0, 92735, '\p{^Block=-Bamum_supplement}', "");
Expect(0, 92735, '\P{Block=-Bamum_supplement}', "");
Expect(1, 92735, '\P{^Block=-Bamum_supplement}', "");
Expect(0, 92736, '\p{Block=-Bamum_supplement}', "");
Expect(1, 92736, '\p{^Block=-Bamum_supplement}', "");
Expect(1, 92736, '\P{Block=-Bamum_supplement}', "");
Expect(0, 92736, '\P{^Block=-Bamum_supplement}', "");
Error('\p{Blk=_Bamum_sup:=}');
Error('\P{Blk=_Bamum_sup:=}');
Expect(1, 92735, '\p{Blk=bamumsup}', "");
Expect(0, 92735, '\p{^Blk=bamumsup}', "");
Expect(0, 92735, '\P{Blk=bamumsup}', "");
Expect(1, 92735, '\P{^Blk=bamumsup}', "");
Expect(0, 92736, '\p{Blk=bamumsup}', "");
Expect(1, 92736, '\p{^Blk=bamumsup}', "");
Expect(1, 92736, '\P{Blk=bamumsup}', "");
Expect(0, 92736, '\P{^Blk=bamumsup}', "");
Expect(1, 92735, '\p{Blk=_-Bamum_SUP}', "");
Expect(0, 92735, '\p{^Blk=_-Bamum_SUP}', "");
Expect(0, 92735, '\P{Blk=_-Bamum_SUP}', "");
Expect(1, 92735, '\P{^Blk=_-Bamum_SUP}', "");
Expect(0, 92736, '\p{Blk=_-Bamum_SUP}', "");
Expect(1, 92736, '\p{^Blk=_-Bamum_SUP}', "");
Expect(1, 92736, '\P{Blk=_-Bamum_SUP}', "");
Expect(0, 92736, '\P{^Blk=_-Bamum_SUP}', "");
Error('\p{Is_Block=-BAMUM_supplement/a/}');
Error('\P{Is_Block=-BAMUM_supplement/a/}');
Expect(1, 92735, '\p{Is_Block=bamumsupplement}', "");
Expect(0, 92735, '\p{^Is_Block=bamumsupplement}', "");
Expect(0, 92735, '\P{Is_Block=bamumsupplement}', "");
Expect(1, 92735, '\P{^Is_Block=bamumsupplement}', "");
Expect(0, 92736, '\p{Is_Block=bamumsupplement}', "");
Expect(1, 92736, '\p{^Is_Block=bamumsupplement}', "");
Expect(1, 92736, '\P{Is_Block=bamumsupplement}', "");
Expect(0, 92736, '\P{^Is_Block=bamumsupplement}', "");
Expect(1, 92735, '\p{Is_Block=	-bamum_Supplement}', "");
Expect(0, 92735, '\p{^Is_Block=	-bamum_Supplement}', "");
Expect(0, 92735, '\P{Is_Block=	-bamum_Supplement}', "");
Expect(1, 92735, '\P{^Is_Block=	-bamum_Supplement}', "");
Expect(0, 92736, '\p{Is_Block=	-bamum_Supplement}', "");
Expect(1, 92736, '\p{^Is_Block=	-bamum_Supplement}', "");
Expect(1, 92736, '\P{Is_Block=	-bamum_Supplement}', "");
Expect(0, 92736, '\P{^Is_Block=	-bamum_Supplement}', "");
Error('\p{Is_Blk= :=Bamum_SUP}');
Error('\P{Is_Blk= :=Bamum_SUP}');
Expect(1, 92735, '\p{Is_Blk=bamumsup}', "");
Expect(0, 92735, '\p{^Is_Blk=bamumsup}', "");
Expect(0, 92735, '\P{Is_Blk=bamumsup}', "");
Expect(1, 92735, '\P{^Is_Blk=bamumsup}', "");
Expect(0, 92736, '\p{Is_Blk=bamumsup}', "");
Expect(1, 92736, '\p{^Is_Blk=bamumsup}', "");
Expect(1, 92736, '\P{Is_Blk=bamumsup}', "");
Expect(0, 92736, '\P{^Is_Blk=bamumsup}', "");
Expect(1, 92735, '\p{Is_Blk= 	bamum_Sup}', "");
Expect(0, 92735, '\p{^Is_Blk= 	bamum_Sup}', "");
Expect(0, 92735, '\P{Is_Blk= 	bamum_Sup}', "");
Expect(1, 92735, '\P{^Is_Blk= 	bamum_Sup}', "");
Expect(0, 92736, '\p{Is_Blk= 	bamum_Sup}', "");
Expect(1, 92736, '\p{^Is_Blk= 	bamum_Sup}', "");
Expect(1, 92736, '\P{Is_Blk= 	bamum_Sup}', "");
Expect(0, 92736, '\P{^Is_Blk= 	bamum_Sup}', "");
Error('\p{Block=_:=BATAK}');
Error('\P{Block=_:=BATAK}');
Expect(1, 7167, '\p{Block=batak}', "");
Expect(0, 7167, '\p{^Block=batak}', "");
Expect(0, 7167, '\P{Block=batak}', "");
Expect(1, 7167, '\P{^Block=batak}', "");
Expect(0, 7168, '\p{Block=batak}', "");
Expect(1, 7168, '\p{^Block=batak}', "");
Expect(1, 7168, '\P{Block=batak}', "");
Expect(0, 7168, '\P{^Block=batak}', "");
Expect(1, 7167, '\p{Block= 	Batak}', "");
Expect(0, 7167, '\p{^Block= 	Batak}', "");
Expect(0, 7167, '\P{Block= 	Batak}', "");
Expect(1, 7167, '\P{^Block= 	Batak}', "");
Expect(0, 7168, '\p{Block= 	Batak}', "");
Expect(1, 7168, '\p{^Block= 	Batak}', "");
Expect(1, 7168, '\P{Block= 	Batak}', "");
Expect(0, 7168, '\P{^Block= 	Batak}', "");
Error('\p{Blk= 	Batak/a/}');
Error('\P{Blk= 	Batak/a/}');
Expect(1, 7167, '\p{Blk=batak}', "");
Expect(0, 7167, '\p{^Blk=batak}', "");
Expect(0, 7167, '\P{Blk=batak}', "");
Expect(1, 7167, '\P{^Blk=batak}', "");
Expect(0, 7168, '\p{Blk=batak}', "");
Expect(1, 7168, '\p{^Blk=batak}', "");
Expect(1, 7168, '\P{Blk=batak}', "");
Expect(0, 7168, '\P{^Blk=batak}', "");
Expect(1, 7167, '\p{Blk=_-BATAK}', "");
Expect(0, 7167, '\p{^Blk=_-BATAK}', "");
Expect(0, 7167, '\P{Blk=_-BATAK}', "");
Expect(1, 7167, '\P{^Blk=_-BATAK}', "");
Expect(0, 7168, '\p{Blk=_-BATAK}', "");
Expect(1, 7168, '\p{^Blk=_-BATAK}', "");
Expect(1, 7168, '\P{Blk=_-BATAK}', "");
Expect(0, 7168, '\P{^Blk=_-BATAK}', "");
Error('\p{Is_Block=	/a/Batak}');
Error('\P{Is_Block=	/a/Batak}');
Expect(1, 7167, '\p{Is_Block: batak}', "");
Expect(0, 7167, '\p{^Is_Block: batak}', "");
Expect(0, 7167, '\P{Is_Block: batak}', "");
Expect(1, 7167, '\P{^Is_Block: batak}', "");
Expect(0, 7168, '\p{Is_Block: batak}', "");
Expect(1, 7168, '\p{^Is_Block: batak}', "");
Expect(1, 7168, '\P{Is_Block: batak}', "");
Expect(0, 7168, '\P{^Is_Block: batak}', "");
Expect(1, 7167, '\p{Is_Block=_BATAK}', "");
Expect(0, 7167, '\p{^Is_Block=_BATAK}', "");
Expect(0, 7167, '\P{Is_Block=_BATAK}', "");
Expect(1, 7167, '\P{^Is_Block=_BATAK}', "");
Expect(0, 7168, '\p{Is_Block=_BATAK}', "");
Expect(1, 7168, '\p{^Is_Block=_BATAK}', "");
Expect(1, 7168, '\P{Is_Block=_BATAK}', "");
Expect(0, 7168, '\P{^Is_Block=_BATAK}', "");
Error('\p{Is_Blk=_ Batak/a/}');
Error('\P{Is_Blk=_ Batak/a/}');
Expect(1, 7167, '\p{Is_Blk=batak}', "");
Expect(0, 7167, '\p{^Is_Blk=batak}', "");
Expect(0, 7167, '\P{Is_Blk=batak}', "");
Expect(1, 7167, '\P{^Is_Blk=batak}', "");
Expect(0, 7168, '\p{Is_Blk=batak}', "");
Expect(1, 7168, '\p{^Is_Blk=batak}', "");
Expect(1, 7168, '\P{Is_Blk=batak}', "");
Expect(0, 7168, '\P{^Is_Blk=batak}', "");
Expect(1, 7167, '\p{Is_Blk=	-batak}', "");
Expect(0, 7167, '\p{^Is_Blk=	-batak}', "");
Expect(0, 7167, '\P{Is_Blk=	-batak}', "");
Expect(1, 7167, '\P{^Is_Blk=	-batak}', "");
Expect(0, 7168, '\p{Is_Blk=	-batak}', "");
Expect(1, 7168, '\p{^Is_Blk=	-batak}', "");
Expect(1, 7168, '\P{Is_Blk=	-batak}', "");
Expect(0, 7168, '\P{^Is_Blk=	-batak}', "");
Error('\p{Block=_-BENGALI/a/}');
Error('\P{Block=_-BENGALI/a/}');
Expect(1, 2559, '\p{Block=bengali}', "");
Expect(0, 2559, '\p{^Block=bengali}', "");
Expect(0, 2559, '\P{Block=bengali}', "");
Expect(1, 2559, '\P{^Block=bengali}', "");
Expect(0, 2560, '\p{Block=bengali}', "");
Expect(1, 2560, '\p{^Block=bengali}', "");
Expect(1, 2560, '\P{Block=bengali}', "");
Expect(0, 2560, '\P{^Block=bengali}', "");
Expect(1, 2559, '\p{Block=--bengali}', "");
Expect(0, 2559, '\p{^Block=--bengali}', "");
Expect(0, 2559, '\P{Block=--bengali}', "");
Expect(1, 2559, '\P{^Block=--bengali}', "");
Expect(0, 2560, '\p{Block=--bengali}', "");
Expect(1, 2560, '\p{^Block=--bengali}', "");
Expect(1, 2560, '\P{Block=--bengali}', "");
Expect(0, 2560, '\P{^Block=--bengali}', "");
Error('\p{Blk=:=_ Bengali}');
Error('\P{Blk=:=_ Bengali}');
Expect(1, 2559, '\p{Blk:   bengali}', "");
Expect(0, 2559, '\p{^Blk:   bengali}', "");
Expect(0, 2559, '\P{Blk:   bengali}', "");
Expect(1, 2559, '\P{^Blk:   bengali}', "");
Expect(0, 2560, '\p{Blk:   bengali}', "");
Expect(1, 2560, '\p{^Blk:   bengali}', "");
Expect(1, 2560, '\P{Blk:   bengali}', "");
Expect(0, 2560, '\P{^Blk:   bengali}', "");
Expect(1, 2559, '\p{Blk=	-BENGALI}', "");
Expect(0, 2559, '\p{^Blk=	-BENGALI}', "");
Expect(0, 2559, '\P{Blk=	-BENGALI}', "");
Expect(1, 2559, '\P{^Blk=	-BENGALI}', "");
Expect(0, 2560, '\p{Blk=	-BENGALI}', "");
Expect(1, 2560, '\p{^Blk=	-BENGALI}', "");
Expect(1, 2560, '\P{Blk=	-BENGALI}', "");
Expect(0, 2560, '\P{^Blk=	-BENGALI}', "");
Error('\p{Is_Block= _BENGALI/a/}');
Error('\P{Is_Block= _BENGALI/a/}');
Expect(1, 2559, '\p{Is_Block=bengali}', "");
Expect(0, 2559, '\p{^Is_Block=bengali}', "");
Expect(0, 2559, '\P{Is_Block=bengali}', "");
Expect(1, 2559, '\P{^Is_Block=bengali}', "");
Expect(0, 2560, '\p{Is_Block=bengali}', "");
Expect(1, 2560, '\p{^Is_Block=bengali}', "");
Expect(1, 2560, '\P{Is_Block=bengali}', "");
Expect(0, 2560, '\P{^Is_Block=bengali}', "");
Expect(1, 2559, '\p{Is_Block: Bengali}', "");
Expect(0, 2559, '\p{^Is_Block: Bengali}', "");
Expect(0, 2559, '\P{Is_Block: Bengali}', "");
Expect(1, 2559, '\P{^Is_Block: Bengali}', "");
Expect(0, 2560, '\p{Is_Block: Bengali}', "");
Expect(1, 2560, '\p{^Is_Block: Bengali}', "");
Expect(1, 2560, '\P{Is_Block: Bengali}', "");
Expect(0, 2560, '\P{^Is_Block: Bengali}', "");
Error('\p{Is_Blk:	/a/ BENGALI}');
Error('\P{Is_Blk:	/a/ BENGALI}');
Expect(1, 2559, '\p{Is_Blk:	bengali}', "");
Expect(0, 2559, '\p{^Is_Blk:	bengali}', "");
Expect(0, 2559, '\P{Is_Blk:	bengali}', "");
Expect(1, 2559, '\P{^Is_Blk:	bengali}', "");
Expect(0, 2560, '\p{Is_Blk:	bengali}', "");
Expect(1, 2560, '\p{^Is_Blk:	bengali}', "");
Expect(1, 2560, '\P{Is_Blk:	bengali}', "");
Expect(0, 2560, '\P{^Is_Blk:	bengali}', "");
Expect(1, 2559, '\p{Is_Blk=-bengali}', "");
Expect(0, 2559, '\p{^Is_Blk=-bengali}', "");
Expect(0, 2559, '\P{Is_Blk=-bengali}', "");
Expect(1, 2559, '\P{^Is_Blk=-bengali}', "");
Expect(0, 2560, '\p{Is_Blk=-bengali}', "");
Expect(1, 2560, '\p{^Is_Blk=-bengali}', "");
Expect(1, 2560, '\P{Is_Blk=-bengali}', "");
Expect(0, 2560, '\P{^Is_Blk=-bengali}', "");
Error('\p{Block=_:=BLOCK_Elements}');
Error('\P{Block=_:=BLOCK_Elements}');
Expect(1, 9631, '\p{Block=blockelements}', "");
Expect(0, 9631, '\p{^Block=blockelements}', "");
Expect(0, 9631, '\P{Block=blockelements}', "");
Expect(1, 9631, '\P{^Block=blockelements}', "");
Expect(0, 9632, '\p{Block=blockelements}', "");
Expect(1, 9632, '\p{^Block=blockelements}', "");
Expect(1, 9632, '\P{Block=blockelements}', "");
Expect(0, 9632, '\P{^Block=blockelements}', "");
Expect(1, 9631, '\p{Block= _Block_elements}', "");
Expect(0, 9631, '\p{^Block= _Block_elements}', "");
Expect(0, 9631, '\P{Block= _Block_elements}', "");
Expect(1, 9631, '\P{^Block= _Block_elements}', "");
Expect(0, 9632, '\p{Block= _Block_elements}', "");
Expect(1, 9632, '\p{^Block= _Block_elements}', "");
Expect(1, 9632, '\P{Block= _Block_elements}', "");
Expect(0, 9632, '\P{^Block= _Block_elements}', "");
Error('\p{Blk=Block_Elements:=}');
Error('\P{Blk=Block_Elements:=}');
Expect(1, 9631, '\p{Blk=blockelements}', "");
Expect(0, 9631, '\p{^Blk=blockelements}', "");
Expect(0, 9631, '\P{Blk=blockelements}', "");
Expect(1, 9631, '\P{^Blk=blockelements}', "");
Expect(0, 9632, '\p{Blk=blockelements}', "");
Expect(1, 9632, '\p{^Blk=blockelements}', "");
Expect(1, 9632, '\P{Blk=blockelements}', "");
Expect(0, 9632, '\P{^Blk=blockelements}', "");
Expect(1, 9631, '\p{Blk=	BLOCK_ELEMENTS}', "");
Expect(0, 9631, '\p{^Blk=	BLOCK_ELEMENTS}', "");
Expect(0, 9631, '\P{Blk=	BLOCK_ELEMENTS}', "");
Expect(1, 9631, '\P{^Blk=	BLOCK_ELEMENTS}', "");
Expect(0, 9632, '\p{Blk=	BLOCK_ELEMENTS}', "");
Expect(1, 9632, '\p{^Blk=	BLOCK_ELEMENTS}', "");
Expect(1, 9632, '\P{Blk=	BLOCK_ELEMENTS}', "");
Expect(0, 9632, '\P{^Blk=	BLOCK_ELEMENTS}', "");
Error('\p{Is_Block=/a/_-Block_elements}');
Error('\P{Is_Block=/a/_-Block_elements}');
Expect(1, 9631, '\p{Is_Block=blockelements}', "");
Expect(0, 9631, '\p{^Is_Block=blockelements}', "");
Expect(0, 9631, '\P{Is_Block=blockelements}', "");
Expect(1, 9631, '\P{^Is_Block=blockelements}', "");
Expect(0, 9632, '\p{Is_Block=blockelements}', "");
Expect(1, 9632, '\p{^Is_Block=blockelements}', "");
Expect(1, 9632, '\P{Is_Block=blockelements}', "");
Expect(0, 9632, '\P{^Is_Block=blockelements}', "");
Expect(1, 9631, '\p{Is_Block=  Block_Elements}', "");
Expect(0, 9631, '\p{^Is_Block=  Block_Elements}', "");
Expect(0, 9631, '\P{Is_Block=  Block_Elements}', "");
Expect(1, 9631, '\P{^Is_Block=  Block_Elements}', "");
Expect(0, 9632, '\p{Is_Block=  Block_Elements}', "");
Expect(1, 9632, '\p{^Is_Block=  Block_Elements}', "");
Expect(1, 9632, '\P{Is_Block=  Block_Elements}', "");
Expect(0, 9632, '\P{^Is_Block=  Block_Elements}', "");
Error('\p{Is_Blk= block_Elements/a/}');
Error('\P{Is_Blk= block_Elements/a/}');
Expect(1, 9631, '\p{Is_Blk=blockelements}', "");
Expect(0, 9631, '\p{^Is_Blk=blockelements}', "");
Expect(0, 9631, '\P{Is_Blk=blockelements}', "");
Expect(1, 9631, '\P{^Is_Blk=blockelements}', "");
Expect(0, 9632, '\p{Is_Blk=blockelements}', "");
Expect(1, 9632, '\p{^Is_Blk=blockelements}', "");
Expect(1, 9632, '\P{Is_Blk=blockelements}', "");
Expect(0, 9632, '\P{^Is_Blk=blockelements}', "");
Expect(1, 9631, '\p{Is_Blk=_-BLOCK_ELEMENTS}', "");
Expect(0, 9631, '\p{^Is_Blk=_-BLOCK_ELEMENTS}', "");
Expect(0, 9631, '\P{Is_Blk=_-BLOCK_ELEMENTS}', "");
Expect(1, 9631, '\P{^Is_Blk=_-BLOCK_ELEMENTS}', "");
Expect(0, 9632, '\p{Is_Blk=_-BLOCK_ELEMENTS}', "");
Expect(1, 9632, '\p{^Is_Blk=_-BLOCK_ELEMENTS}', "");
Expect(1, 9632, '\P{Is_Blk=_-BLOCK_ELEMENTS}', "");
Expect(0, 9632, '\P{^Is_Blk=_-BLOCK_ELEMENTS}', "");
Error('\p{Block=:=	-Bopomofo}');
Error('\P{Block=:=	-Bopomofo}');
Expect(1, 12591, '\p{Block=bopomofo}', "");
Expect(0, 12591, '\p{^Block=bopomofo}', "");
Expect(0, 12591, '\P{Block=bopomofo}', "");
Expect(1, 12591, '\P{^Block=bopomofo}', "");
Expect(0, 12592, '\p{Block=bopomofo}', "");
Expect(1, 12592, '\p{^Block=bopomofo}', "");
Expect(1, 12592, '\P{Block=bopomofo}', "");
Expect(0, 12592, '\P{^Block=bopomofo}', "");
Expect(1, 12591, '\p{Block=_	BOPOMOFO}', "");
Expect(0, 12591, '\p{^Block=_	BOPOMOFO}', "");
Expect(0, 12591, '\P{Block=_	BOPOMOFO}', "");
Expect(1, 12591, '\P{^Block=_	BOPOMOFO}', "");
Expect(0, 12592, '\p{Block=_	BOPOMOFO}', "");
Expect(1, 12592, '\p{^Block=_	BOPOMOFO}', "");
Expect(1, 12592, '\P{Block=_	BOPOMOFO}', "");
Expect(0, 12592, '\P{^Block=_	BOPOMOFO}', "");
Error('\p{Blk:   :=_Bopomofo}');
Error('\P{Blk:   :=_Bopomofo}');
Expect(1, 12591, '\p{Blk=bopomofo}', "");
Expect(0, 12591, '\p{^Blk=bopomofo}', "");
Expect(0, 12591, '\P{Blk=bopomofo}', "");
Expect(1, 12591, '\P{^Blk=bopomofo}', "");
Expect(0, 12592, '\p{Blk=bopomofo}', "");
Expect(1, 12592, '\p{^Blk=bopomofo}', "");
Expect(1, 12592, '\P{Blk=bopomofo}', "");
Expect(0, 12592, '\P{^Blk=bopomofo}', "");
Expect(1, 12591, '\p{Blk=  Bopomofo}', "");
Expect(0, 12591, '\p{^Blk=  Bopomofo}', "");
Expect(0, 12591, '\P{Blk=  Bopomofo}', "");
Expect(1, 12591, '\P{^Blk=  Bopomofo}', "");
Expect(0, 12592, '\p{Blk=  Bopomofo}', "");
Expect(1, 12592, '\p{^Blk=  Bopomofo}', "");
Expect(1, 12592, '\P{Blk=  Bopomofo}', "");
Expect(0, 12592, '\P{^Blk=  Bopomofo}', "");
Error('\p{Is_Block= /a/bopomofo}');
Error('\P{Is_Block= /a/bopomofo}');
Expect(1, 12591, '\p{Is_Block:	bopomofo}', "");
Expect(0, 12591, '\p{^Is_Block:	bopomofo}', "");
Expect(0, 12591, '\P{Is_Block:	bopomofo}', "");
Expect(1, 12591, '\P{^Is_Block:	bopomofo}', "");
Expect(0, 12592, '\p{Is_Block:	bopomofo}', "");
Expect(1, 12592, '\p{^Is_Block:	bopomofo}', "");
Expect(1, 12592, '\P{Is_Block:	bopomofo}', "");
Expect(0, 12592, '\P{^Is_Block:	bopomofo}', "");
Expect(1, 12591, '\p{Is_Block=-	BOPOMOFO}', "");
Expect(0, 12591, '\p{^Is_Block=-	BOPOMOFO}', "");
Expect(0, 12591, '\P{Is_Block=-	BOPOMOFO}', "");
Expect(1, 12591, '\P{^Is_Block=-	BOPOMOFO}', "");
Expect(0, 12592, '\p{Is_Block=-	BOPOMOFO}', "");
Expect(1, 12592, '\p{^Is_Block=-	BOPOMOFO}', "");
Expect(1, 12592, '\P{Is_Block=-	BOPOMOFO}', "");
Expect(0, 12592, '\P{^Is_Block=-	BOPOMOFO}', "");
Error('\p{Is_Blk= :=BOPOMOFO}');
Error('\P{Is_Blk= :=BOPOMOFO}');
Expect(1, 12591, '\p{Is_Blk=bopomofo}', "");
Expect(0, 12591, '\p{^Is_Blk=bopomofo}', "");
Expect(0, 12591, '\P{Is_Blk=bopomofo}', "");
Expect(1, 12591, '\P{^Is_Blk=bopomofo}', "");
Expect(0, 12592, '\p{Is_Blk=bopomofo}', "");
Expect(1, 12592, '\p{^Is_Blk=bopomofo}', "");
Expect(1, 12592, '\P{Is_Blk=bopomofo}', "");
Expect(0, 12592, '\P{^Is_Blk=bopomofo}', "");
Expect(1, 12591, '\p{Is_Blk=		Bopomofo}', "");
Expect(0, 12591, '\p{^Is_Blk=		Bopomofo}', "");
Expect(0, 12591, '\P{Is_Blk=		Bopomofo}', "");
Expect(1, 12591, '\P{^Is_Blk=		Bopomofo}', "");
Expect(0, 12592, '\p{Is_Blk=		Bopomofo}', "");
Expect(1, 12592, '\p{^Is_Blk=		Bopomofo}', "");
Expect(1, 12592, '\P{Is_Blk=		Bopomofo}', "");
Expect(0, 12592, '\P{^Is_Blk=		Bopomofo}', "");
Error('\p{Block=	:=BOPOMOFO_Extended}');
Error('\P{Block=	:=BOPOMOFO_Extended}');
Expect(1, 12735, '\p{Block=bopomofoextended}', "");
Expect(0, 12735, '\p{^Block=bopomofoextended}', "");
Expect(0, 12735, '\P{Block=bopomofoextended}', "");
Expect(1, 12735, '\P{^Block=bopomofoextended}', "");
Expect(0, 12736, '\p{Block=bopomofoextended}', "");
Expect(1, 12736, '\p{^Block=bopomofoextended}', "");
Expect(1, 12736, '\P{Block=bopomofoextended}', "");
Expect(0, 12736, '\P{^Block=bopomofoextended}', "");
Expect(1, 12735, '\p{Block=	Bopomofo_Extended}', "");
Expect(0, 12735, '\p{^Block=	Bopomofo_Extended}', "");
Expect(0, 12735, '\P{Block=	Bopomofo_Extended}', "");
Expect(1, 12735, '\P{^Block=	Bopomofo_Extended}', "");
Expect(0, 12736, '\p{Block=	Bopomofo_Extended}', "");
Expect(1, 12736, '\p{^Block=	Bopomofo_Extended}', "");
Expect(1, 12736, '\P{Block=	Bopomofo_Extended}', "");
Expect(0, 12736, '\P{^Block=	Bopomofo_Extended}', "");
Error('\p{Blk= /a/Bopomofo_EXT}');
Error('\P{Blk= /a/Bopomofo_EXT}');
Expect(1, 12735, '\p{Blk=bopomofoext}', "");
Expect(0, 12735, '\p{^Blk=bopomofoext}', "");
Expect(0, 12735, '\P{Blk=bopomofoext}', "");
Expect(1, 12735, '\P{^Blk=bopomofoext}', "");
Expect(0, 12736, '\p{Blk=bopomofoext}', "");
Expect(1, 12736, '\p{^Blk=bopomofoext}', "");
Expect(1, 12736, '\P{Blk=bopomofoext}', "");
Expect(0, 12736, '\P{^Blk=bopomofoext}', "");
Expect(1, 12735, '\p{Blk=_Bopomofo_ext}', "");
Expect(0, 12735, '\p{^Blk=_Bopomofo_ext}', "");
Expect(0, 12735, '\P{Blk=_Bopomofo_ext}', "");
Expect(1, 12735, '\P{^Blk=_Bopomofo_ext}', "");
Expect(0, 12736, '\p{Blk=_Bopomofo_ext}', "");
Expect(1, 12736, '\p{^Blk=_Bopomofo_ext}', "");
Expect(1, 12736, '\P{Blk=_Bopomofo_ext}', "");
Expect(0, 12736, '\P{^Blk=_Bopomofo_ext}', "");
Error('\p{Is_Block=		BOPOMOFO_extended:=}');
Error('\P{Is_Block=		BOPOMOFO_extended:=}');
Expect(1, 12735, '\p{Is_Block=bopomofoextended}', "");
Expect(0, 12735, '\p{^Is_Block=bopomofoextended}', "");
Expect(0, 12735, '\P{Is_Block=bopomofoextended}', "");
Expect(1, 12735, '\P{^Is_Block=bopomofoextended}', "");
Expect(0, 12736, '\p{Is_Block=bopomofoextended}', "");
Expect(1, 12736, '\p{^Is_Block=bopomofoextended}', "");
Expect(1, 12736, '\P{Is_Block=bopomofoextended}', "");
Expect(0, 12736, '\P{^Is_Block=bopomofoextended}', "");
Expect(1, 12735, '\p{Is_Block=_Bopomofo_EXTENDED}', "");
Expect(0, 12735, '\p{^Is_Block=_Bopomofo_EXTENDED}', "");
Expect(0, 12735, '\P{Is_Block=_Bopomofo_EXTENDED}', "");
Expect(1, 12735, '\P{^Is_Block=_Bopomofo_EXTENDED}', "");
Expect(0, 12736, '\p{Is_Block=_Bopomofo_EXTENDED}', "");
Expect(1, 12736, '\p{^Is_Block=_Bopomofo_EXTENDED}', "");
Expect(1, 12736, '\P{Is_Block=_Bopomofo_EXTENDED}', "");
Expect(0, 12736, '\P{^Is_Block=_Bopomofo_EXTENDED}', "");
Error('\p{Is_Blk=:= 	BOPOMOFO_ext}');
Error('\P{Is_Blk=:= 	BOPOMOFO_ext}');
Expect(1, 12735, '\p{Is_Blk=bopomofoext}', "");
Expect(0, 12735, '\p{^Is_Blk=bopomofoext}', "");
Expect(0, 12735, '\P{Is_Blk=bopomofoext}', "");
Expect(1, 12735, '\P{^Is_Blk=bopomofoext}', "");
Expect(0, 12736, '\p{Is_Blk=bopomofoext}', "");
Expect(1, 12736, '\p{^Is_Blk=bopomofoext}', "");
Expect(1, 12736, '\P{Is_Blk=bopomofoext}', "");
Expect(0, 12736, '\P{^Is_Blk=bopomofoext}', "");
Expect(1, 12735, '\p{Is_Blk=BOPOMOFO_EXT}', "");
Expect(0, 12735, '\p{^Is_Blk=BOPOMOFO_EXT}', "");
Expect(0, 12735, '\P{Is_Blk=BOPOMOFO_EXT}', "");
Expect(1, 12735, '\P{^Is_Blk=BOPOMOFO_EXT}', "");
Expect(0, 12736, '\p{Is_Blk=BOPOMOFO_EXT}', "");
Expect(1, 12736, '\p{^Is_Blk=BOPOMOFO_EXT}', "");
Expect(1, 12736, '\P{Is_Blk=BOPOMOFO_EXT}', "");
Expect(0, 12736, '\P{^Is_Blk=BOPOMOFO_EXT}', "");
Error('\p{Block=/a/Box_Drawing}');
Error('\P{Block=/a/Box_Drawing}');
Expect(1, 9599, '\p{Block=boxdrawing}', "");
Expect(0, 9599, '\p{^Block=boxdrawing}', "");
Expect(0, 9599, '\P{Block=boxdrawing}', "");
Expect(1, 9599, '\P{^Block=boxdrawing}', "");
Expect(0, 9600, '\p{Block=boxdrawing}', "");
Expect(1, 9600, '\p{^Block=boxdrawing}', "");
Expect(1, 9600, '\P{Block=boxdrawing}', "");
Expect(0, 9600, '\P{^Block=boxdrawing}', "");
Expect(1, 9599, '\p{Block=_ box_Drawing}', "");
Expect(0, 9599, '\p{^Block=_ box_Drawing}', "");
Expect(0, 9599, '\P{Block=_ box_Drawing}', "");
Expect(1, 9599, '\P{^Block=_ box_Drawing}', "");
Expect(0, 9600, '\p{Block=_ box_Drawing}', "");
Expect(1, 9600, '\p{^Block=_ box_Drawing}', "");
Expect(1, 9600, '\P{Block=_ box_Drawing}', "");
Expect(0, 9600, '\P{^Block=_ box_Drawing}', "");
Error('\p{Blk= 	box_Drawing/a/}');
Error('\P{Blk= 	box_Drawing/a/}');
Expect(1, 9599, '\p{Blk=boxdrawing}', "");
Expect(0, 9599, '\p{^Blk=boxdrawing}', "");
Expect(0, 9599, '\P{Blk=boxdrawing}', "");
Expect(1, 9599, '\P{^Blk=boxdrawing}', "");
Expect(0, 9600, '\p{Blk=boxdrawing}', "");
Expect(1, 9600, '\p{^Blk=boxdrawing}', "");
Expect(1, 9600, '\P{Blk=boxdrawing}', "");
Expect(0, 9600, '\P{^Blk=boxdrawing}', "");
Expect(1, 9599, '\p{Blk=--Box_DRAWING}', "");
Expect(0, 9599, '\p{^Blk=--Box_DRAWING}', "");
Expect(0, 9599, '\P{Blk=--Box_DRAWING}', "");
Expect(1, 9599, '\P{^Blk=--Box_DRAWING}', "");
Expect(0, 9600, '\p{Blk=--Box_DRAWING}', "");
Expect(1, 9600, '\p{^Blk=--Box_DRAWING}', "");
Expect(1, 9600, '\P{Blk=--Box_DRAWING}', "");
Expect(0, 9600, '\P{^Blk=--Box_DRAWING}', "");
Error('\p{Is_Block=:= Box_Drawing}');
Error('\P{Is_Block=:= Box_Drawing}');
Expect(1, 9599, '\p{Is_Block=boxdrawing}', "");
Expect(0, 9599, '\p{^Is_Block=boxdrawing}', "");
Expect(0, 9599, '\P{Is_Block=boxdrawing}', "");
Expect(1, 9599, '\P{^Is_Block=boxdrawing}', "");
Expect(0, 9600, '\p{Is_Block=boxdrawing}', "");
Expect(1, 9600, '\p{^Is_Block=boxdrawing}', "");
Expect(1, 9600, '\P{Is_Block=boxdrawing}', "");
Expect(0, 9600, '\P{^Is_Block=boxdrawing}', "");
Expect(1, 9599, '\p{Is_Block=-box_Drawing}', "");
Expect(0, 9599, '\p{^Is_Block=-box_Drawing}', "");
Expect(0, 9599, '\P{Is_Block=-box_Drawing}', "");
Expect(1, 9599, '\P{^Is_Block=-box_Drawing}', "");
Expect(0, 9600, '\p{Is_Block=-box_Drawing}', "");
Expect(1, 9600, '\p{^Is_Block=-box_Drawing}', "");
Expect(1, 9600, '\P{Is_Block=-box_Drawing}', "");
Expect(0, 9600, '\P{^Is_Block=-box_Drawing}', "");
Error('\p{Is_Blk=/a/-Box_Drawing}');
Error('\P{Is_Blk=/a/-Box_Drawing}');
Expect(1, 9599, '\p{Is_Blk: boxdrawing}', "");
Expect(0, 9599, '\p{^Is_Blk: boxdrawing}', "");
Expect(0, 9599, '\P{Is_Blk: boxdrawing}', "");
Expect(1, 9599, '\P{^Is_Blk: boxdrawing}', "");
Expect(0, 9600, '\p{Is_Blk: boxdrawing}', "");
Expect(1, 9600, '\p{^Is_Blk: boxdrawing}', "");
Expect(1, 9600, '\P{Is_Blk: boxdrawing}', "");
Expect(0, 9600, '\P{^Is_Blk: boxdrawing}', "");
Expect(1, 9599, '\p{Is_Blk: 		Box_Drawing}', "");
Expect(0, 9599, '\p{^Is_Blk: 		Box_Drawing}', "");
Expect(0, 9599, '\P{Is_Blk: 		Box_Drawing}', "");
Expect(1, 9599, '\P{^Is_Blk: 		Box_Drawing}', "");
Expect(0, 9600, '\p{Is_Blk: 		Box_Drawing}', "");
Expect(1, 9600, '\p{^Is_Blk: 		Box_Drawing}', "");
Expect(1, 9600, '\P{Is_Blk: 		Box_Drawing}', "");
Expect(0, 9600, '\P{^Is_Blk: 		Box_Drawing}', "");
Error('\p{Block=/a/ -brahmi}');
Error('\P{Block=/a/ -brahmi}');
Expect(1, 69759, '\p{Block=brahmi}', "");
Expect(0, 69759, '\p{^Block=brahmi}', "");
Expect(0, 69759, '\P{Block=brahmi}', "");
Expect(1, 69759, '\P{^Block=brahmi}', "");
Expect(0, 69760, '\p{Block=brahmi}', "");
Expect(1, 69760, '\p{^Block=brahmi}', "");
Expect(1, 69760, '\P{Block=brahmi}', "");
Expect(0, 69760, '\P{^Block=brahmi}', "");
Expect(1, 69759, '\p{Block= Brahmi}', "");
Expect(0, 69759, '\p{^Block= Brahmi}', "");
Expect(0, 69759, '\P{Block= Brahmi}', "");
Expect(1, 69759, '\P{^Block= Brahmi}', "");
Expect(0, 69760, '\p{Block= Brahmi}', "");
Expect(1, 69760, '\p{^Block= Brahmi}', "");
Expect(1, 69760, '\P{Block= Brahmi}', "");
Expect(0, 69760, '\P{^Block= Brahmi}', "");
Error('\p{Blk=/a/Brahmi}');
Error('\P{Blk=/a/Brahmi}');
Expect(1, 69759, '\p{Blk:   brahmi}', "");
Expect(0, 69759, '\p{^Blk:   brahmi}', "");
Expect(0, 69759, '\P{Blk:   brahmi}', "");
Expect(1, 69759, '\P{^Blk:   brahmi}', "");
Expect(0, 69760, '\p{Blk:   brahmi}', "");
Expect(1, 69760, '\p{^Blk:   brahmi}', "");
Expect(1, 69760, '\P{Blk:   brahmi}', "");
Expect(0, 69760, '\P{^Blk:   brahmi}', "");
Expect(1, 69759, '\p{Blk=--brahmi}', "");
Expect(0, 69759, '\p{^Blk=--brahmi}', "");
Expect(0, 69759, '\P{Blk=--brahmi}', "");
Expect(1, 69759, '\P{^Blk=--brahmi}', "");
Expect(0, 69760, '\p{Blk=--brahmi}', "");
Expect(1, 69760, '\p{^Blk=--brahmi}', "");
Expect(1, 69760, '\P{Blk=--brahmi}', "");
Expect(0, 69760, '\P{^Blk=--brahmi}', "");
Error('\p{Is_Block=		Brahmi:=}');
Error('\P{Is_Block=		Brahmi:=}');
Expect(1, 69759, '\p{Is_Block=brahmi}', "");
Expect(0, 69759, '\p{^Is_Block=brahmi}', "");
Expect(0, 69759, '\P{Is_Block=brahmi}', "");
Expect(1, 69759, '\P{^Is_Block=brahmi}', "");
Expect(0, 69760, '\p{Is_Block=brahmi}', "");
Expect(1, 69760, '\p{^Is_Block=brahmi}', "");
Expect(1, 69760, '\P{Is_Block=brahmi}', "");
Expect(0, 69760, '\P{^Is_Block=brahmi}', "");
Expect(1, 69759, '\p{Is_Block=_Brahmi}', "");
Expect(0, 69759, '\p{^Is_Block=_Brahmi}', "");
Expect(0, 69759, '\P{Is_Block=_Brahmi}', "");
Expect(1, 69759, '\P{^Is_Block=_Brahmi}', "");
Expect(0, 69760, '\p{Is_Block=_Brahmi}', "");
Expect(1, 69760, '\p{^Is_Block=_Brahmi}', "");
Expect(1, 69760, '\P{Is_Block=_Brahmi}', "");
Expect(0, 69760, '\P{^Is_Block=_Brahmi}', "");
Error('\p{Is_Blk=:= _Brahmi}');
Error('\P{Is_Blk=:= _Brahmi}');
Expect(1, 69759, '\p{Is_Blk=brahmi}', "");
Expect(0, 69759, '\p{^Is_Blk=brahmi}', "");
Expect(0, 69759, '\P{Is_Blk=brahmi}', "");
Expect(1, 69759, '\P{^Is_Blk=brahmi}', "");
Expect(0, 69760, '\p{Is_Blk=brahmi}', "");
Expect(1, 69760, '\p{^Is_Blk=brahmi}', "");
Expect(1, 69760, '\P{Is_Blk=brahmi}', "");
Expect(0, 69760, '\P{^Is_Blk=brahmi}', "");
Expect(1, 69759, '\p{Is_Blk=--Brahmi}', "");
Expect(0, 69759, '\p{^Is_Blk=--Brahmi}', "");
Expect(0, 69759, '\P{Is_Blk=--Brahmi}', "");
Expect(1, 69759, '\P{^Is_Blk=--Brahmi}', "");
Expect(0, 69760, '\p{Is_Blk=--Brahmi}', "");
Expect(1, 69760, '\p{^Is_Blk=--Brahmi}', "");
Expect(1, 69760, '\P{Is_Blk=--Brahmi}', "");
Expect(0, 69760, '\P{^Is_Blk=--Brahmi}', "");
Error('\p{Block:	:=_-Braille_patterns}');
Error('\P{Block:	:=_-Braille_patterns}');
Expect(1, 10495, '\p{Block:braillepatterns}', "");
Expect(0, 10495, '\p{^Block:braillepatterns}', "");
Expect(0, 10495, '\P{Block:braillepatterns}', "");
Expect(1, 10495, '\P{^Block:braillepatterns}', "");
Expect(0, 10496, '\p{Block:braillepatterns}', "");
Expect(1, 10496, '\p{^Block:braillepatterns}', "");
Expect(1, 10496, '\P{Block:braillepatterns}', "");
Expect(0, 10496, '\P{^Block:braillepatterns}', "");
Expect(1, 10495, '\p{Block=_Braille_Patterns}', "");
Expect(0, 10495, '\p{^Block=_Braille_Patterns}', "");
Expect(0, 10495, '\P{Block=_Braille_Patterns}', "");
Expect(1, 10495, '\P{^Block=_Braille_Patterns}', "");
Expect(0, 10496, '\p{Block=_Braille_Patterns}', "");
Expect(1, 10496, '\p{^Block=_Braille_Patterns}', "");
Expect(1, 10496, '\P{Block=_Braille_Patterns}', "");
Expect(0, 10496, '\P{^Block=_Braille_Patterns}', "");
Error('\p{Blk=_:=Braille}');
Error('\P{Blk=_:=Braille}');
Expect(1, 10495, '\p{Blk:	braille}', "");
Expect(0, 10495, '\p{^Blk:	braille}', "");
Expect(0, 10495, '\P{Blk:	braille}', "");
Expect(1, 10495, '\P{^Blk:	braille}', "");
Expect(0, 10496, '\p{Blk:	braille}', "");
Expect(1, 10496, '\p{^Blk:	braille}', "");
Expect(1, 10496, '\P{Blk:	braille}', "");
Expect(0, 10496, '\P{^Blk:	braille}', "");
Expect(1, 10495, '\p{Blk:   Braille}', "");
Expect(0, 10495, '\p{^Blk:   Braille}', "");
Expect(0, 10495, '\P{Blk:   Braille}', "");
Expect(1, 10495, '\P{^Blk:   Braille}', "");
Expect(0, 10496, '\p{Blk:   Braille}', "");
Expect(1, 10496, '\p{^Blk:   Braille}', "");
Expect(1, 10496, '\P{Blk:   Braille}', "");
Expect(0, 10496, '\P{^Blk:   Braille}', "");
Error('\p{Is_Block=_-Braille_PATTERNS:=}');
Error('\P{Is_Block=_-Braille_PATTERNS:=}');
Expect(1, 10495, '\p{Is_Block=braillepatterns}', "");
Expect(0, 10495, '\p{^Is_Block=braillepatterns}', "");
Expect(0, 10495, '\P{Is_Block=braillepatterns}', "");
Expect(1, 10495, '\P{^Is_Block=braillepatterns}', "");
Expect(0, 10496, '\p{Is_Block=braillepatterns}', "");
Expect(1, 10496, '\p{^Is_Block=braillepatterns}', "");
Expect(1, 10496, '\P{Is_Block=braillepatterns}', "");
Expect(0, 10496, '\P{^Is_Block=braillepatterns}', "");
Expect(1, 10495, '\p{Is_Block=	 Braille_patterns}', "");
Expect(0, 10495, '\p{^Is_Block=	 Braille_patterns}', "");
Expect(0, 10495, '\P{Is_Block=	 Braille_patterns}', "");
Expect(1, 10495, '\P{^Is_Block=	 Braille_patterns}', "");
Expect(0, 10496, '\p{Is_Block=	 Braille_patterns}', "");
Expect(1, 10496, '\p{^Is_Block=	 Braille_patterns}', "");
Expect(1, 10496, '\P{Is_Block=	 Braille_patterns}', "");
Expect(0, 10496, '\P{^Is_Block=	 Braille_patterns}', "");
Error('\p{Is_Blk=-braille:=}');
Error('\P{Is_Blk=-braille:=}');
Expect(1, 10495, '\p{Is_Blk: braille}', "");
Expect(0, 10495, '\p{^Is_Blk: braille}', "");
Expect(0, 10495, '\P{Is_Blk: braille}', "");
Expect(1, 10495, '\P{^Is_Blk: braille}', "");
Expect(0, 10496, '\p{Is_Blk: braille}', "");
Expect(1, 10496, '\p{^Is_Blk: braille}', "");
Expect(1, 10496, '\P{Is_Blk: braille}', "");
Expect(0, 10496, '\P{^Is_Blk: braille}', "");
Expect(1, 10495, '\p{Is_Blk=	-braille}', "");
Expect(0, 10495, '\p{^Is_Blk=	-braille}', "");
Expect(0, 10495, '\P{Is_Blk=	-braille}', "");
Expect(1, 10495, '\P{^Is_Blk=	-braille}', "");
Expect(0, 10496, '\p{Is_Blk=	-braille}', "");
Expect(1, 10496, '\p{^Is_Blk=	-braille}', "");
Expect(1, 10496, '\P{Is_Blk=	-braille}', "");
Expect(0, 10496, '\P{^Is_Blk=	-braille}', "");
Error('\p{Block:	_:=Buginese}');
Error('\P{Block:	_:=Buginese}');
Expect(1, 6687, '\p{Block=buginese}', "");
Expect(0, 6687, '\p{^Block=buginese}', "");
Expect(0, 6687, '\P{Block=buginese}', "");
Expect(1, 6687, '\P{^Block=buginese}', "");
Expect(0, 6688, '\p{Block=buginese}', "");
Expect(1, 6688, '\p{^Block=buginese}', "");
Expect(1, 6688, '\P{Block=buginese}', "");
Expect(0, 6688, '\P{^Block=buginese}', "");
Expect(1, 6687, '\p{Block:    -Buginese}', "");
Expect(0, 6687, '\p{^Block:    -Buginese}', "");
Expect(0, 6687, '\P{Block:    -Buginese}', "");
Expect(1, 6687, '\P{^Block:    -Buginese}', "");
Expect(0, 6688, '\p{Block:    -Buginese}', "");
Expect(1, 6688, '\p{^Block:    -Buginese}', "");
Expect(1, 6688, '\P{Block:    -Buginese}', "");
Expect(0, 6688, '\P{^Block:    -Buginese}', "");
Error('\p{Blk=/a/-Buginese}');
Error('\P{Blk=/a/-Buginese}');
Expect(1, 6687, '\p{Blk:	buginese}', "");
Expect(0, 6687, '\p{^Blk:	buginese}', "");
Expect(0, 6687, '\P{Blk:	buginese}', "");
Expect(1, 6687, '\P{^Blk:	buginese}', "");
Expect(0, 6688, '\p{Blk:	buginese}', "");
Expect(1, 6688, '\p{^Blk:	buginese}', "");
Expect(1, 6688, '\P{Blk:	buginese}', "");
Expect(0, 6688, '\P{^Blk:	buginese}', "");
Expect(1, 6687, '\p{Blk=--Buginese}', "");
Expect(0, 6687, '\p{^Blk=--Buginese}', "");
Expect(0, 6687, '\P{Blk=--Buginese}', "");
Expect(1, 6687, '\P{^Blk=--Buginese}', "");
Expect(0, 6688, '\p{Blk=--Buginese}', "");
Expect(1, 6688, '\p{^Blk=--Buginese}', "");
Expect(1, 6688, '\P{Blk=--Buginese}', "");
Expect(0, 6688, '\P{^Blk=--Buginese}', "");
Error('\p{Is_Block= /a/buginese}');
Error('\P{Is_Block= /a/buginese}');
Expect(1, 6687, '\p{Is_Block=buginese}', "");
Expect(0, 6687, '\p{^Is_Block=buginese}', "");
Expect(0, 6687, '\P{Is_Block=buginese}', "");
Expect(1, 6687, '\P{^Is_Block=buginese}', "");
Expect(0, 6688, '\p{Is_Block=buginese}', "");
Expect(1, 6688, '\p{^Is_Block=buginese}', "");
Expect(1, 6688, '\P{Is_Block=buginese}', "");
Expect(0, 6688, '\P{^Is_Block=buginese}', "");
Expect(1, 6687, '\p{Is_Block=	_Buginese}', "");
Expect(0, 6687, '\p{^Is_Block=	_Buginese}', "");
Expect(0, 6687, '\P{Is_Block=	_Buginese}', "");
Expect(1, 6687, '\P{^Is_Block=	_Buginese}', "");
Expect(0, 6688, '\p{Is_Block=	_Buginese}', "");
Expect(1, 6688, '\p{^Is_Block=	_Buginese}', "");
Expect(1, 6688, '\P{Is_Block=	_Buginese}', "");
Expect(0, 6688, '\P{^Is_Block=	_Buginese}', "");
Error('\p{Is_Blk=	 Buginese/a/}');
Error('\P{Is_Blk=	 Buginese/a/}');
Expect(1, 6687, '\p{Is_Blk=buginese}', "");
Expect(0, 6687, '\p{^Is_Blk=buginese}', "");
Expect(0, 6687, '\P{Is_Blk=buginese}', "");
Expect(1, 6687, '\P{^Is_Blk=buginese}', "");
Expect(0, 6688, '\p{Is_Blk=buginese}', "");
Expect(1, 6688, '\p{^Is_Blk=buginese}', "");
Expect(1, 6688, '\P{Is_Blk=buginese}', "");
Expect(0, 6688, '\P{^Is_Blk=buginese}', "");
Expect(1, 6687, '\p{Is_Blk= Buginese}', "");
Expect(0, 6687, '\p{^Is_Blk= Buginese}', "");
Expect(0, 6687, '\P{Is_Blk= Buginese}', "");
Expect(1, 6687, '\P{^Is_Blk= Buginese}', "");
Expect(0, 6688, '\p{Is_Blk= Buginese}', "");
Expect(1, 6688, '\p{^Is_Blk= Buginese}', "");
Expect(1, 6688, '\P{Is_Blk= Buginese}', "");
Expect(0, 6688, '\P{^Is_Blk= Buginese}', "");
Error('\p{Block= :=BUHID}');
Error('\P{Block= :=BUHID}');
Expect(1, 5983, '\p{Block=buhid}', "");
Expect(0, 5983, '\p{^Block=buhid}', "");
Expect(0, 5983, '\P{Block=buhid}', "");
Expect(1, 5983, '\P{^Block=buhid}', "");
Expect(0, 5984, '\p{Block=buhid}', "");
Expect(1, 5984, '\p{^Block=buhid}', "");
Expect(1, 5984, '\P{Block=buhid}', "");
Expect(0, 5984, '\P{^Block=buhid}', "");
Expect(1, 5983, '\p{Block=-BUHID}', "");
Expect(0, 5983, '\p{^Block=-BUHID}', "");
Expect(0, 5983, '\P{Block=-BUHID}', "");
Expect(1, 5983, '\P{^Block=-BUHID}', "");
Expect(0, 5984, '\p{Block=-BUHID}', "");
Expect(1, 5984, '\p{^Block=-BUHID}', "");
Expect(1, 5984, '\P{Block=-BUHID}', "");
Expect(0, 5984, '\P{^Block=-BUHID}', "");
Error('\p{Blk= :=Buhid}');
Error('\P{Blk= :=Buhid}');
Expect(1, 5983, '\p{Blk=buhid}', "");
Expect(0, 5983, '\p{^Blk=buhid}', "");
Expect(0, 5983, '\P{Blk=buhid}', "");
Expect(1, 5983, '\P{^Blk=buhid}', "");
Expect(0, 5984, '\p{Blk=buhid}', "");
Expect(1, 5984, '\p{^Blk=buhid}', "");
Expect(1, 5984, '\P{Blk=buhid}', "");
Expect(0, 5984, '\P{^Blk=buhid}', "");
Expect(1, 5983, '\p{Blk=	_buhid}', "");
Expect(0, 5983, '\p{^Blk=	_buhid}', "");
Expect(0, 5983, '\P{Blk=	_buhid}', "");
Expect(1, 5983, '\P{^Blk=	_buhid}', "");
Expect(0, 5984, '\p{Blk=	_buhid}', "");
Expect(1, 5984, '\p{^Blk=	_buhid}', "");
Expect(1, 5984, '\P{Blk=	_buhid}', "");
Expect(0, 5984, '\P{^Blk=	_buhid}', "");
Error('\p{Is_Block:	 -Buhid/a/}');
Error('\P{Is_Block:	 -Buhid/a/}');
Expect(1, 5983, '\p{Is_Block=buhid}', "");
Expect(0, 5983, '\p{^Is_Block=buhid}', "");
Expect(0, 5983, '\P{Is_Block=buhid}', "");
Expect(1, 5983, '\P{^Is_Block=buhid}', "");
Expect(0, 5984, '\p{Is_Block=buhid}', "");
Expect(1, 5984, '\p{^Is_Block=buhid}', "");
Expect(1, 5984, '\P{Is_Block=buhid}', "");
Expect(0, 5984, '\P{^Is_Block=buhid}', "");
Expect(1, 5983, '\p{Is_Block=	-BUHID}', "");
Expect(0, 5983, '\p{^Is_Block=	-BUHID}', "");
Expect(0, 5983, '\P{Is_Block=	-BUHID}', "");
Expect(1, 5983, '\P{^Is_Block=	-BUHID}', "");
Expect(0, 5984, '\p{Is_Block=	-BUHID}', "");
Expect(1, 5984, '\p{^Is_Block=	-BUHID}', "");
Expect(1, 5984, '\P{Is_Block=	-BUHID}', "");
Expect(0, 5984, '\P{^Is_Block=	-BUHID}', "");
Error('\p{Is_Blk=-BUHID/a/}');
Error('\P{Is_Blk=-BUHID/a/}');
Expect(1, 5983, '\p{Is_Blk=buhid}', "");
Expect(0, 5983, '\p{^Is_Blk=buhid}', "");
Expect(0, 5983, '\P{Is_Blk=buhid}', "");
Expect(1, 5983, '\P{^Is_Blk=buhid}', "");
Expect(0, 5984, '\p{Is_Blk=buhid}', "");
Expect(1, 5984, '\p{^Is_Blk=buhid}', "");
Expect(1, 5984, '\P{Is_Blk=buhid}', "");
Expect(0, 5984, '\P{^Is_Blk=buhid}', "");
Expect(1, 5983, '\p{Is_Blk= -Buhid}', "");
Expect(0, 5983, '\p{^Is_Blk= -Buhid}', "");
Expect(0, 5983, '\P{Is_Blk= -Buhid}', "");
Expect(1, 5983, '\P{^Is_Blk= -Buhid}', "");
Expect(0, 5984, '\p{Is_Blk= -Buhid}', "");
Expect(1, 5984, '\p{^Is_Blk= -Buhid}', "");
Expect(1, 5984, '\P{Is_Blk= -Buhid}', "");
Expect(0, 5984, '\P{^Is_Blk= -Buhid}', "");
Error('\p{Block=_ BYZANTINE_musical_symbols:=}');
Error('\P{Block=_ BYZANTINE_musical_symbols:=}');
Expect(1, 119039, '\p{Block=byzantinemusicalsymbols}', "");
Expect(0, 119039, '\p{^Block=byzantinemusicalsymbols}', "");
Expect(0, 119039, '\P{Block=byzantinemusicalsymbols}', "");
Expect(1, 119039, '\P{^Block=byzantinemusicalsymbols}', "");
Expect(0, 119040, '\p{Block=byzantinemusicalsymbols}', "");
Expect(1, 119040, '\p{^Block=byzantinemusicalsymbols}', "");
Expect(1, 119040, '\P{Block=byzantinemusicalsymbols}', "");
Expect(0, 119040, '\P{^Block=byzantinemusicalsymbols}', "");
Expect(1, 119039, '\p{Block= _byzantine_musical_SYMBOLS}', "");
Expect(0, 119039, '\p{^Block= _byzantine_musical_SYMBOLS}', "");
Expect(0, 119039, '\P{Block= _byzantine_musical_SYMBOLS}', "");
Expect(1, 119039, '\P{^Block= _byzantine_musical_SYMBOLS}', "");
Expect(0, 119040, '\p{Block= _byzantine_musical_SYMBOLS}', "");
Expect(1, 119040, '\p{^Block= _byzantine_musical_SYMBOLS}', "");
Expect(1, 119040, '\P{Block= _byzantine_musical_SYMBOLS}', "");
Expect(0, 119040, '\P{^Block= _byzantine_musical_SYMBOLS}', "");
Error('\p{Blk: 	byzantine_Music:=}');
Error('\P{Blk: 	byzantine_Music:=}');
Expect(1, 119039, '\p{Blk=byzantinemusic}', "");
Expect(0, 119039, '\p{^Blk=byzantinemusic}', "");
Expect(0, 119039, '\P{Blk=byzantinemusic}', "");
Expect(1, 119039, '\P{^Blk=byzantinemusic}', "");
Expect(0, 119040, '\p{Blk=byzantinemusic}', "");
Expect(1, 119040, '\p{^Blk=byzantinemusic}', "");
Expect(1, 119040, '\P{Blk=byzantinemusic}', "");
Expect(0, 119040, '\P{^Blk=byzantinemusic}', "");
Expect(1, 119039, '\p{Blk=_	Byzantine_Music}', "");
Expect(0, 119039, '\p{^Blk=_	Byzantine_Music}', "");
Expect(0, 119039, '\P{Blk=_	Byzantine_Music}', "");
Expect(1, 119039, '\P{^Blk=_	Byzantine_Music}', "");
Expect(0, 119040, '\p{Blk=_	Byzantine_Music}', "");
Expect(1, 119040, '\p{^Blk=_	Byzantine_Music}', "");
Expect(1, 119040, '\P{Blk=_	Byzantine_Music}', "");
Expect(0, 119040, '\P{^Blk=_	Byzantine_Music}', "");
Error('\p{Is_Block=	_BYZANTINE_Musical_Symbols:=}');
Error('\P{Is_Block=	_BYZANTINE_Musical_Symbols:=}');
Expect(1, 119039, '\p{Is_Block=byzantinemusicalsymbols}', "");
Expect(0, 119039, '\p{^Is_Block=byzantinemusicalsymbols}', "");
Expect(0, 119039, '\P{Is_Block=byzantinemusicalsymbols}', "");
Expect(1, 119039, '\P{^Is_Block=byzantinemusicalsymbols}', "");
Expect(0, 119040, '\p{Is_Block=byzantinemusicalsymbols}', "");
Expect(1, 119040, '\p{^Is_Block=byzantinemusicalsymbols}', "");
Expect(1, 119040, '\P{Is_Block=byzantinemusicalsymbols}', "");
Expect(0, 119040, '\P{^Is_Block=byzantinemusicalsymbols}', "");
Expect(1, 119039, '\p{Is_Block=	 BYZANTINE_Musical_Symbols}', "");
Expect(0, 119039, '\p{^Is_Block=	 BYZANTINE_Musical_Symbols}', "");
Expect(0, 119039, '\P{Is_Block=	 BYZANTINE_Musical_Symbols}', "");
Expect(1, 119039, '\P{^Is_Block=	 BYZANTINE_Musical_Symbols}', "");
Expect(0, 119040, '\p{Is_Block=	 BYZANTINE_Musical_Symbols}', "");
Expect(1, 119040, '\p{^Is_Block=	 BYZANTINE_Musical_Symbols}', "");
Expect(1, 119040, '\P{Is_Block=	 BYZANTINE_Musical_Symbols}', "");
Expect(0, 119040, '\P{^Is_Block=	 BYZANTINE_Musical_Symbols}', "");
Error('\p{Is_Blk=-	byzantine_Music:=}');
Error('\P{Is_Blk=-	byzantine_Music:=}');
Expect(1, 119039, '\p{Is_Blk: byzantinemusic}', "");
Expect(0, 119039, '\p{^Is_Blk: byzantinemusic}', "");
Expect(0, 119039, '\P{Is_Blk: byzantinemusic}', "");
Expect(1, 119039, '\P{^Is_Blk: byzantinemusic}', "");
Expect(0, 119040, '\p{Is_Blk: byzantinemusic}', "");
Expect(1, 119040, '\p{^Is_Blk: byzantinemusic}', "");
Expect(1, 119040, '\P{Is_Blk: byzantinemusic}', "");
Expect(0, 119040, '\P{^Is_Blk: byzantinemusic}', "");
Expect(1, 119039, '\p{Is_Blk= -BYZANTINE_MUSIC}', "");
Expect(0, 119039, '\p{^Is_Blk= -BYZANTINE_MUSIC}', "");
Expect(0, 119039, '\P{Is_Blk= -BYZANTINE_MUSIC}', "");
Expect(1, 119039, '\P{^Is_Blk= -BYZANTINE_MUSIC}', "");
Expect(0, 119040, '\p{Is_Blk= -BYZANTINE_MUSIC}', "");
Expect(1, 119040, '\p{^Is_Blk= -BYZANTINE_MUSIC}', "");
Expect(1, 119040, '\P{Is_Blk= -BYZANTINE_MUSIC}', "");
Expect(0, 119040, '\P{^Is_Blk= -BYZANTINE_MUSIC}', "");
Error('\p{Block=-:=CJK_Unified_Ideographs}');
Error('\P{Block=-:=CJK_Unified_Ideographs}');
Expect(1, 40959, '\p{Block=cjkunifiedideographs}', "");
Expect(0, 40959, '\p{^Block=cjkunifiedideographs}', "");
Expect(0, 40959, '\P{Block=cjkunifiedideographs}', "");
Expect(1, 40959, '\P{^Block=cjkunifiedideographs}', "");
Expect(0, 40960, '\p{Block=cjkunifiedideographs}', "");
Expect(1, 40960, '\p{^Block=cjkunifiedideographs}', "");
Expect(1, 40960, '\P{Block=cjkunifiedideographs}', "");
Expect(0, 40960, '\P{^Block=cjkunifiedideographs}', "");
Expect(1, 40959, '\p{Block=- CJK_Unified_IDEOGRAPHS}', "");
Expect(0, 40959, '\p{^Block=- CJK_Unified_IDEOGRAPHS}', "");
Expect(0, 40959, '\P{Block=- CJK_Unified_IDEOGRAPHS}', "");
Expect(1, 40959, '\P{^Block=- CJK_Unified_IDEOGRAPHS}', "");
Expect(0, 40960, '\p{Block=- CJK_Unified_IDEOGRAPHS}', "");
Expect(1, 40960, '\p{^Block=- CJK_Unified_IDEOGRAPHS}', "");
Expect(1, 40960, '\P{Block=- CJK_Unified_IDEOGRAPHS}', "");
Expect(0, 40960, '\P{^Block=- CJK_Unified_IDEOGRAPHS}', "");
Error('\p{Blk=	CJK:=}');
Error('\P{Blk=	CJK:=}');
Expect(1, 40959, '\p{Blk=cjk}', "");
Expect(0, 40959, '\p{^Blk=cjk}', "");
Expect(0, 40959, '\P{Blk=cjk}', "");
Expect(1, 40959, '\P{^Blk=cjk}', "");
Expect(0, 40960, '\p{Blk=cjk}', "");
Expect(1, 40960, '\p{^Blk=cjk}', "");
Expect(1, 40960, '\P{Blk=cjk}', "");
Expect(0, 40960, '\P{^Blk=cjk}', "");
Expect(1, 40959, '\p{Blk= CJK}', "");
Expect(0, 40959, '\p{^Blk= CJK}', "");
Expect(0, 40959, '\P{Blk= CJK}', "");
Expect(1, 40959, '\P{^Blk= CJK}', "");
Expect(0, 40960, '\p{Blk= CJK}', "");
Expect(1, 40960, '\p{^Blk= CJK}', "");
Expect(1, 40960, '\P{Blk= CJK}', "");
Expect(0, 40960, '\P{^Blk= CJK}', "");
Error('\p{Is_Block=	 cjk_unified_Ideographs/a/}');
Error('\P{Is_Block=	 cjk_unified_Ideographs/a/}');
Expect(1, 40959, '\p{Is_Block:	cjkunifiedideographs}', "");
Expect(0, 40959, '\p{^Is_Block:	cjkunifiedideographs}', "");
Expect(0, 40959, '\P{Is_Block:	cjkunifiedideographs}', "");
Expect(1, 40959, '\P{^Is_Block:	cjkunifiedideographs}', "");
Expect(0, 40960, '\p{Is_Block:	cjkunifiedideographs}', "");
Expect(1, 40960, '\p{^Is_Block:	cjkunifiedideographs}', "");
Expect(1, 40960, '\P{Is_Block:	cjkunifiedideographs}', "");
Expect(0, 40960, '\P{^Is_Block:	cjkunifiedideographs}', "");
Expect(1, 40959, '\p{Is_Block:   CJK_UNIFIED_Ideographs}', "");
Expect(0, 40959, '\p{^Is_Block:   CJK_UNIFIED_Ideographs}', "");
Expect(0, 40959, '\P{Is_Block:   CJK_UNIFIED_Ideographs}', "");
Expect(1, 40959, '\P{^Is_Block:   CJK_UNIFIED_Ideographs}', "");
Expect(0, 40960, '\p{Is_Block:   CJK_UNIFIED_Ideographs}', "");
Expect(1, 40960, '\p{^Is_Block:   CJK_UNIFIED_Ideographs}', "");
Expect(1, 40960, '\P{Is_Block:   CJK_UNIFIED_Ideographs}', "");
Expect(0, 40960, '\P{^Is_Block:   CJK_UNIFIED_Ideographs}', "");
Error('\p{Is_Blk:	/a/__CJK}');
Error('\P{Is_Blk:	/a/__CJK}');
Expect(1, 40959, '\p{Is_Blk=cjk}', "");
Expect(0, 40959, '\p{^Is_Blk=cjk}', "");
Expect(0, 40959, '\P{Is_Blk=cjk}', "");
Expect(1, 40959, '\P{^Is_Blk=cjk}', "");
Expect(0, 40960, '\p{Is_Blk=cjk}', "");
Expect(1, 40960, '\p{^Is_Blk=cjk}', "");
Expect(1, 40960, '\P{Is_Blk=cjk}', "");
Expect(0, 40960, '\P{^Is_Blk=cjk}', "");
Expect(1, 40959, '\p{Is_Blk= cjk}', "");
Expect(0, 40959, '\p{^Is_Blk= cjk}', "");
Expect(0, 40959, '\P{Is_Blk= cjk}', "");
Expect(1, 40959, '\P{^Is_Blk= cjk}', "");
Expect(0, 40960, '\p{Is_Blk= cjk}', "");
Expect(1, 40960, '\p{^Is_Blk= cjk}', "");
Expect(1, 40960, '\P{Is_Blk= cjk}', "");
Expect(0, 40960, '\P{^Is_Blk= cjk}', "");
Error('\p{Block=_CJK_compatibility/a/}');
Error('\P{Block=_CJK_compatibility/a/}');
Expect(1, 13311, '\p{Block=cjkcompatibility}', "");
Expect(0, 13311, '\p{^Block=cjkcompatibility}', "");
Expect(0, 13311, '\P{Block=cjkcompatibility}', "");
Expect(1, 13311, '\P{^Block=cjkcompatibility}', "");
Expect(0, 13312, '\p{Block=cjkcompatibility}', "");
Expect(1, 13312, '\p{^Block=cjkcompatibility}', "");
Expect(1, 13312, '\P{Block=cjkcompatibility}', "");
Expect(0, 13312, '\P{^Block=cjkcompatibility}', "");
Expect(1, 13311, '\p{Block= -CJK_Compatibility}', "");
Expect(0, 13311, '\p{^Block= -CJK_Compatibility}', "");
Expect(0, 13311, '\P{Block= -CJK_Compatibility}', "");
Expect(1, 13311, '\P{^Block= -CJK_Compatibility}', "");
Expect(0, 13312, '\p{Block= -CJK_Compatibility}', "");
Expect(1, 13312, '\p{^Block= -CJK_Compatibility}', "");
Expect(1, 13312, '\P{Block= -CJK_Compatibility}', "");
Expect(0, 13312, '\P{^Block= -CJK_Compatibility}', "");
Error('\p{Blk=	/a/cjk_compat}');
Error('\P{Blk=	/a/cjk_compat}');
Expect(1, 13311, '\p{Blk=cjkcompat}', "");
Expect(0, 13311, '\p{^Blk=cjkcompat}', "");
Expect(0, 13311, '\P{Blk=cjkcompat}', "");
Expect(1, 13311, '\P{^Blk=cjkcompat}', "");
Expect(0, 13312, '\p{Blk=cjkcompat}', "");
Expect(1, 13312, '\p{^Blk=cjkcompat}', "");
Expect(1, 13312, '\P{Blk=cjkcompat}', "");
Expect(0, 13312, '\P{^Blk=cjkcompat}', "");
Expect(1, 13311, '\p{Blk=	-CJK_compat}', "");
Expect(0, 13311, '\p{^Blk=	-CJK_compat}', "");
Expect(0, 13311, '\P{Blk=	-CJK_compat}', "");
Expect(1, 13311, '\P{^Blk=	-CJK_compat}', "");
Expect(0, 13312, '\p{Blk=	-CJK_compat}', "");
Expect(1, 13312, '\p{^Blk=	-CJK_compat}', "");
Expect(1, 13312, '\P{Blk=	-CJK_compat}', "");
Expect(0, 13312, '\P{^Blk=	-CJK_compat}', "");
Error('\p{Is_Block:   /a/-	CJK_COMPATIBILITY}');
Error('\P{Is_Block:   /a/-	CJK_COMPATIBILITY}');
Expect(1, 13311, '\p{Is_Block=cjkcompatibility}', "");
Expect(0, 13311, '\p{^Is_Block=cjkcompatibility}', "");
Expect(0, 13311, '\P{Is_Block=cjkcompatibility}', "");
Expect(1, 13311, '\P{^Is_Block=cjkcompatibility}', "");
Expect(0, 13312, '\p{Is_Block=cjkcompatibility}', "");
Expect(1, 13312, '\p{^Is_Block=cjkcompatibility}', "");
Expect(1, 13312, '\P{Is_Block=cjkcompatibility}', "");
Expect(0, 13312, '\P{^Is_Block=cjkcompatibility}', "");
Expect(1, 13311, '\p{Is_Block=	-CJK_Compatibility}', "");
Expect(0, 13311, '\p{^Is_Block=	-CJK_Compatibility}', "");
Expect(0, 13311, '\P{Is_Block=	-CJK_Compatibility}', "");
Expect(1, 13311, '\P{^Is_Block=	-CJK_Compatibility}', "");
Expect(0, 13312, '\p{Is_Block=	-CJK_Compatibility}', "");
Expect(1, 13312, '\p{^Is_Block=	-CJK_Compatibility}', "");
Expect(1, 13312, '\P{Is_Block=	-CJK_Compatibility}', "");
Expect(0, 13312, '\P{^Is_Block=	-CJK_Compatibility}', "");
Error('\p{Is_Blk= CJK_Compat:=}');
Error('\P{Is_Blk= CJK_Compat:=}');
Expect(1, 13311, '\p{Is_Blk=cjkcompat}', "");
Expect(0, 13311, '\p{^Is_Blk=cjkcompat}', "");
Expect(0, 13311, '\P{Is_Blk=cjkcompat}', "");
Expect(1, 13311, '\P{^Is_Blk=cjkcompat}', "");
Expect(0, 13312, '\p{Is_Blk=cjkcompat}', "");
Expect(1, 13312, '\p{^Is_Blk=cjkcompat}', "");
Expect(1, 13312, '\P{Is_Blk=cjkcompat}', "");
Expect(0, 13312, '\P{^Is_Blk=cjkcompat}', "");
Expect(1, 13311, '\p{Is_Blk=-CJK_compat}', "");
Expect(0, 13311, '\p{^Is_Blk=-CJK_compat}', "");
Expect(0, 13311, '\P{Is_Blk=-CJK_compat}', "");
Expect(1, 13311, '\P{^Is_Blk=-CJK_compat}', "");
Expect(0, 13312, '\p{Is_Blk=-CJK_compat}', "");
Expect(1, 13312, '\p{^Is_Blk=-CJK_compat}', "");
Expect(1, 13312, '\P{Is_Blk=-CJK_compat}', "");
Expect(0, 13312, '\P{^Is_Blk=-CJK_compat}', "");
Error('\p{Block=- CJK_Compatibility_forms:=}');
Error('\P{Block=- CJK_Compatibility_forms:=}');
Expect(1, 65103, '\p{Block=cjkcompatibilityforms}', "");
Expect(0, 65103, '\p{^Block=cjkcompatibilityforms}', "");
Expect(0, 65103, '\P{Block=cjkcompatibilityforms}', "");
Expect(1, 65103, '\P{^Block=cjkcompatibilityforms}', "");
Expect(0, 65104, '\p{Block=cjkcompatibilityforms}', "");
Expect(1, 65104, '\p{^Block=cjkcompatibilityforms}', "");
Expect(1, 65104, '\P{Block=cjkcompatibilityforms}', "");
Expect(0, 65104, '\P{^Block=cjkcompatibilityforms}', "");
Expect(1, 65103, '\p{Block=_CJK_Compatibility_Forms}', "");
Expect(0, 65103, '\p{^Block=_CJK_Compatibility_Forms}', "");
Expect(0, 65103, '\P{Block=_CJK_Compatibility_Forms}', "");
Expect(1, 65103, '\P{^Block=_CJK_Compatibility_Forms}', "");
Expect(0, 65104, '\p{Block=_CJK_Compatibility_Forms}', "");
Expect(1, 65104, '\p{^Block=_CJK_Compatibility_Forms}', "");
Expect(1, 65104, '\P{Block=_CJK_Compatibility_Forms}', "");
Expect(0, 65104, '\P{^Block=_CJK_Compatibility_Forms}', "");
Error('\p{Blk:	/a/CJK_COMPAT_FORMS}');
Error('\P{Blk:	/a/CJK_COMPAT_FORMS}');
Expect(1, 65103, '\p{Blk=cjkcompatforms}', "");
Expect(0, 65103, '\p{^Blk=cjkcompatforms}', "");
Expect(0, 65103, '\P{Blk=cjkcompatforms}', "");
Expect(1, 65103, '\P{^Blk=cjkcompatforms}', "");
Expect(0, 65104, '\p{Blk=cjkcompatforms}', "");
Expect(1, 65104, '\p{^Blk=cjkcompatforms}', "");
Expect(1, 65104, '\P{Blk=cjkcompatforms}', "");
Expect(0, 65104, '\P{^Blk=cjkcompatforms}', "");
Expect(1, 65103, '\p{Blk=--CJK_Compat_Forms}', "");
Expect(0, 65103, '\p{^Blk=--CJK_Compat_Forms}', "");
Expect(0, 65103, '\P{Blk=--CJK_Compat_Forms}', "");
Expect(1, 65103, '\P{^Blk=--CJK_Compat_Forms}', "");
Expect(0, 65104, '\p{Blk=--CJK_Compat_Forms}', "");
Expect(1, 65104, '\p{^Blk=--CJK_Compat_Forms}', "");
Expect(1, 65104, '\P{Blk=--CJK_Compat_Forms}', "");
Expect(0, 65104, '\P{^Blk=--CJK_Compat_Forms}', "");
Error('\p{Is_Block=-:=CJK_compatibility_Forms}');
Error('\P{Is_Block=-:=CJK_compatibility_Forms}');
Expect(1, 65103, '\p{Is_Block=cjkcompatibilityforms}', "");
Expect(0, 65103, '\p{^Is_Block=cjkcompatibilityforms}', "");
Expect(0, 65103, '\P{Is_Block=cjkcompatibilityforms}', "");
Expect(1, 65103, '\P{^Is_Block=cjkcompatibilityforms}', "");
Expect(0, 65104, '\p{Is_Block=cjkcompatibilityforms}', "");
Expect(1, 65104, '\p{^Is_Block=cjkcompatibilityforms}', "");
Expect(1, 65104, '\P{Is_Block=cjkcompatibilityforms}', "");
Expect(0, 65104, '\P{^Is_Block=cjkcompatibilityforms}', "");
Expect(1, 65103, '\p{Is_Block=_cjk_Compatibility_Forms}', "");
Expect(0, 65103, '\p{^Is_Block=_cjk_Compatibility_Forms}', "");
Expect(0, 65103, '\P{Is_Block=_cjk_Compatibility_Forms}', "");
Expect(1, 65103, '\P{^Is_Block=_cjk_Compatibility_Forms}', "");
Expect(0, 65104, '\p{Is_Block=_cjk_Compatibility_Forms}', "");
Expect(1, 65104, '\p{^Is_Block=_cjk_Compatibility_Forms}', "");
Expect(1, 65104, '\P{Is_Block=_cjk_Compatibility_Forms}', "");
Expect(0, 65104, '\P{^Is_Block=_cjk_Compatibility_Forms}', "");
Error('\p{Is_Blk=_/a/cjk_COMPAT_Forms}');
Error('\P{Is_Blk=_/a/cjk_COMPAT_Forms}');
Expect(1, 65103, '\p{Is_Blk=cjkcompatforms}', "");
Expect(0, 65103, '\p{^Is_Blk=cjkcompatforms}', "");
Expect(0, 65103, '\P{Is_Blk=cjkcompatforms}', "");
Expect(1, 65103, '\P{^Is_Blk=cjkcompatforms}', "");
Expect(0, 65104, '\p{Is_Blk=cjkcompatforms}', "");
Expect(1, 65104, '\p{^Is_Blk=cjkcompatforms}', "");
Expect(1, 65104, '\P{Is_Blk=cjkcompatforms}', "");
Expect(0, 65104, '\P{^Is_Blk=cjkcompatforms}', "");
Expect(1, 65103, '\p{Is_Blk= 	CJK_Compat_FORMS}', "");
Expect(0, 65103, '\p{^Is_Blk= 	CJK_Compat_FORMS}', "");
Expect(0, 65103, '\P{Is_Blk= 	CJK_Compat_FORMS}', "");
Expect(1, 65103, '\P{^Is_Blk= 	CJK_Compat_FORMS}', "");
Expect(0, 65104, '\p{Is_Blk= 	CJK_Compat_FORMS}', "");
Expect(1, 65104, '\p{^Is_Blk= 	CJK_Compat_FORMS}', "");
Expect(1, 65104, '\P{Is_Blk= 	CJK_Compat_FORMS}', "");
Expect(0, 65104, '\P{^Is_Blk= 	CJK_Compat_FORMS}', "");
Error('\p{Block=:=_CJK_Compatibility_ideographs}');
Error('\P{Block=:=_CJK_Compatibility_ideographs}');
Expect(1, 64255, '\p{Block=cjkcompatibilityideographs}', "");
Expect(0, 64255, '\p{^Block=cjkcompatibilityideographs}', "");
Expect(0, 64255, '\P{Block=cjkcompatibilityideographs}', "");
Expect(1, 64255, '\P{^Block=cjkcompatibilityideographs}', "");
Expect(0, 64256, '\p{Block=cjkcompatibilityideographs}', "");
Expect(1, 64256, '\p{^Block=cjkcompatibilityideographs}', "");
Expect(1, 64256, '\P{Block=cjkcompatibilityideographs}', "");
Expect(0, 64256, '\P{^Block=cjkcompatibilityideographs}', "");
Expect(1, 64255, '\p{Block= 	cjk_Compatibility_IDEOGRAPHS}', "");
Expect(0, 64255, '\p{^Block= 	cjk_Compatibility_IDEOGRAPHS}', "");
Expect(0, 64255, '\P{Block= 	cjk_Compatibility_IDEOGRAPHS}', "");
Expect(1, 64255, '\P{^Block= 	cjk_Compatibility_IDEOGRAPHS}', "");
Expect(0, 64256, '\p{Block= 	cjk_Compatibility_IDEOGRAPHS}', "");
Expect(1, 64256, '\p{^Block= 	cjk_Compatibility_IDEOGRAPHS}', "");
Expect(1, 64256, '\P{Block= 	cjk_Compatibility_IDEOGRAPHS}', "");
Expect(0, 64256, '\P{^Block= 	cjk_Compatibility_IDEOGRAPHS}', "");
Error('\p{Blk=_:=CJK_compat_ideographs}');
Error('\P{Blk=_:=CJK_compat_ideographs}');
Expect(1, 64255, '\p{Blk=cjkcompatideographs}', "");
Expect(0, 64255, '\p{^Blk=cjkcompatideographs}', "");
Expect(0, 64255, '\P{Blk=cjkcompatideographs}', "");
Expect(1, 64255, '\P{^Blk=cjkcompatideographs}', "");
Expect(0, 64256, '\p{Blk=cjkcompatideographs}', "");
Expect(1, 64256, '\p{^Blk=cjkcompatideographs}', "");
Expect(1, 64256, '\P{Blk=cjkcompatideographs}', "");
Expect(0, 64256, '\P{^Blk=cjkcompatideographs}', "");
Expect(1, 64255, '\p{Blk=--CJK_compat_Ideographs}', "");
Expect(0, 64255, '\p{^Blk=--CJK_compat_Ideographs}', "");
Expect(0, 64255, '\P{Blk=--CJK_compat_Ideographs}', "");
Expect(1, 64255, '\P{^Blk=--CJK_compat_Ideographs}', "");
Expect(0, 64256, '\p{Blk=--CJK_compat_Ideographs}', "");
Expect(1, 64256, '\p{^Blk=--CJK_compat_Ideographs}', "");
Expect(1, 64256, '\P{Blk=--CJK_compat_Ideographs}', "");
Expect(0, 64256, '\P{^Blk=--CJK_compat_Ideographs}', "");
Error('\p{Is_Block= -CJK_Compatibility_ideographs/a/}');
Error('\P{Is_Block= -CJK_Compatibility_ideographs/a/}');
Expect(1, 64255, '\p{Is_Block=cjkcompatibilityideographs}', "");
Expect(0, 64255, '\p{^Is_Block=cjkcompatibilityideographs}', "");
Expect(0, 64255, '\P{Is_Block=cjkcompatibilityideographs}', "");
Expect(1, 64255, '\P{^Is_Block=cjkcompatibilityideographs}', "");
Expect(0, 64256, '\p{Is_Block=cjkcompatibilityideographs}', "");
Expect(1, 64256, '\p{^Is_Block=cjkcompatibilityideographs}', "");
Expect(1, 64256, '\P{Is_Block=cjkcompatibilityideographs}', "");
Expect(0, 64256, '\P{^Is_Block=cjkcompatibilityideographs}', "");
Expect(1, 64255, '\p{Is_Block=--cjk_Compatibility_ideographs}', "");
Expect(0, 64255, '\p{^Is_Block=--cjk_Compatibility_ideographs}', "");
Expect(0, 64255, '\P{Is_Block=--cjk_Compatibility_ideographs}', "");
Expect(1, 64255, '\P{^Is_Block=--cjk_Compatibility_ideographs}', "");
Expect(0, 64256, '\p{Is_Block=--cjk_Compatibility_ideographs}', "");
Expect(1, 64256, '\p{^Is_Block=--cjk_Compatibility_ideographs}', "");
Expect(1, 64256, '\P{Is_Block=--cjk_Compatibility_ideographs}', "");
Expect(0, 64256, '\P{^Is_Block=--cjk_Compatibility_ideographs}', "");
Error('\p{Is_Blk= _CJK_COMPAT_Ideographs:=}');
Error('\P{Is_Blk= _CJK_COMPAT_Ideographs:=}');
Expect(1, 64255, '\p{Is_Blk=cjkcompatideographs}', "");
Expect(0, 64255, '\p{^Is_Blk=cjkcompatideographs}', "");
Expect(0, 64255, '\P{Is_Blk=cjkcompatideographs}', "");
Expect(1, 64255, '\P{^Is_Blk=cjkcompatideographs}', "");
Expect(0, 64256, '\p{Is_Blk=cjkcompatideographs}', "");
Expect(1, 64256, '\p{^Is_Blk=cjkcompatideographs}', "");
Expect(1, 64256, '\P{Is_Blk=cjkcompatideographs}', "");
Expect(0, 64256, '\P{^Is_Blk=cjkcompatideographs}', "");
Expect(1, 64255, '\p{Is_Blk=__CJK_Compat_Ideographs}', "");
Expect(0, 64255, '\p{^Is_Blk=__CJK_Compat_Ideographs}', "");
Expect(0, 64255, '\P{Is_Blk=__CJK_Compat_Ideographs}', "");
Expect(1, 64255, '\P{^Is_Blk=__CJK_Compat_Ideographs}', "");
Expect(0, 64256, '\p{Is_Blk=__CJK_Compat_Ideographs}', "");
Expect(1, 64256, '\p{^Is_Blk=__CJK_Compat_Ideographs}', "");
Expect(1, 64256, '\P{Is_Blk=__CJK_Compat_Ideographs}', "");
Expect(0, 64256, '\P{^Is_Blk=__CJK_Compat_Ideographs}', "");
Error('\p{Block:_:=CJK_Compatibility_Ideographs_Supplement}');
Error('\P{Block:_:=CJK_Compatibility_Ideographs_Supplement}');
Expect(1, 195103, '\p{Block=cjkcompatibilityideographssupplement}', "");
Expect(0, 195103, '\p{^Block=cjkcompatibilityideographssupplement}', "");
Expect(0, 195103, '\P{Block=cjkcompatibilityideographssupplement}', "");
Expect(1, 195103, '\P{^Block=cjkcompatibilityideographssupplement}', "");
Expect(0, 195104, '\p{Block=cjkcompatibilityideographssupplement}', "");
Expect(1, 195104, '\p{^Block=cjkcompatibilityideographssupplement}', "");
Expect(1, 195104, '\P{Block=cjkcompatibilityideographssupplement}', "");
Expect(0, 195104, '\P{^Block=cjkcompatibilityideographssupplement}', "");
Expect(1, 195103, '\p{Block:    CJK_Compatibility_Ideographs_SUPPLEMENT}', "");
Expect(0, 195103, '\p{^Block:    CJK_Compatibility_Ideographs_SUPPLEMENT}', "");
Expect(0, 195103, '\P{Block:    CJK_Compatibility_Ideographs_SUPPLEMENT}', "");
Expect(1, 195103, '\P{^Block:    CJK_Compatibility_Ideographs_SUPPLEMENT}', "");
Expect(0, 195104, '\p{Block:    CJK_Compatibility_Ideographs_SUPPLEMENT}', "");
Expect(1, 195104, '\p{^Block:    CJK_Compatibility_Ideographs_SUPPLEMENT}', "");
Expect(1, 195104, '\P{Block:    CJK_Compatibility_Ideographs_SUPPLEMENT}', "");
Expect(0, 195104, '\P{^Block:    CJK_Compatibility_Ideographs_SUPPLEMENT}', "");
Error('\p{Blk=:= _CJK_Compat_IDEOGRAPHS_sup}');
Error('\P{Blk=:= _CJK_Compat_IDEOGRAPHS_sup}');
Expect(1, 195103, '\p{Blk=cjkcompatideographssup}', "");
Expect(0, 195103, '\p{^Blk=cjkcompatideographssup}', "");
Expect(0, 195103, '\P{Blk=cjkcompatideographssup}', "");
Expect(1, 195103, '\P{^Blk=cjkcompatideographssup}', "");
Expect(0, 195104, '\p{Blk=cjkcompatideographssup}', "");
Expect(1, 195104, '\p{^Blk=cjkcompatideographssup}', "");
Expect(1, 195104, '\P{Blk=cjkcompatideographssup}', "");
Expect(0, 195104, '\P{^Blk=cjkcompatideographssup}', "");
Expect(1, 195103, '\p{Blk=_-CJK_compat_ideographs_Sup}', "");
Expect(0, 195103, '\p{^Blk=_-CJK_compat_ideographs_Sup}', "");
Expect(0, 195103, '\P{Blk=_-CJK_compat_ideographs_Sup}', "");
Expect(1, 195103, '\P{^Blk=_-CJK_compat_ideographs_Sup}', "");
Expect(0, 195104, '\p{Blk=_-CJK_compat_ideographs_Sup}', "");
Expect(1, 195104, '\p{^Blk=_-CJK_compat_ideographs_Sup}', "");
Expect(1, 195104, '\P{Blk=_-CJK_compat_ideographs_Sup}', "");
Expect(0, 195104, '\P{^Blk=_-CJK_compat_ideographs_Sup}', "");
Error('\p{Is_Block=		cjk_compatibility_ideographs_supplement/a/}');
Error('\P{Is_Block=		cjk_compatibility_ideographs_supplement/a/}');
Expect(1, 195103, '\p{Is_Block=cjkcompatibilityideographssupplement}', "");
Expect(0, 195103, '\p{^Is_Block=cjkcompatibilityideographssupplement}', "");
Expect(0, 195103, '\P{Is_Block=cjkcompatibilityideographssupplement}', "");
Expect(1, 195103, '\P{^Is_Block=cjkcompatibilityideographssupplement}', "");
Expect(0, 195104, '\p{Is_Block=cjkcompatibilityideographssupplement}', "");
Expect(1, 195104, '\p{^Is_Block=cjkcompatibilityideographssupplement}', "");
Expect(1, 195104, '\P{Is_Block=cjkcompatibilityideographssupplement}', "");
Expect(0, 195104, '\P{^Is_Block=cjkcompatibilityideographssupplement}', "");
Expect(1, 195103, '\p{Is_Block=		CJK_Compatibility_ideographs_supplement}', "");
Expect(0, 195103, '\p{^Is_Block=		CJK_Compatibility_ideographs_supplement}', "");
Expect(0, 195103, '\P{Is_Block=		CJK_Compatibility_ideographs_supplement}', "");
Expect(1, 195103, '\P{^Is_Block=		CJK_Compatibility_ideographs_supplement}', "");
Expect(0, 195104, '\p{Is_Block=		CJK_Compatibility_ideographs_supplement}', "");
Expect(1, 195104, '\p{^Is_Block=		CJK_Compatibility_ideographs_supplement}', "");
Expect(1, 195104, '\P{Is_Block=		CJK_Compatibility_ideographs_supplement}', "");
Expect(0, 195104, '\P{^Is_Block=		CJK_Compatibility_ideographs_supplement}', "");
Error('\p{Is_Blk:/a/ -CJK_Compat_ideographs_Sup}');
Error('\P{Is_Blk:/a/ -CJK_Compat_ideographs_Sup}');
Expect(1, 195103, '\p{Is_Blk=cjkcompatideographssup}', "");
Expect(0, 195103, '\p{^Is_Blk=cjkcompatideographssup}', "");
Expect(0, 195103, '\P{Is_Blk=cjkcompatideographssup}', "");
Expect(1, 195103, '\P{^Is_Blk=cjkcompatideographssup}', "");
Expect(0, 195104, '\p{Is_Blk=cjkcompatideographssup}', "");
Expect(1, 195104, '\p{^Is_Blk=cjkcompatideographssup}', "");
Expect(1, 195104, '\P{Is_Blk=cjkcompatideographssup}', "");
Expect(0, 195104, '\P{^Is_Blk=cjkcompatideographssup}', "");
Expect(1, 195103, '\p{Is_Blk=	CJK_compat_Ideographs_sup}', "");
Expect(0, 195103, '\p{^Is_Blk=	CJK_compat_Ideographs_sup}', "");
Expect(0, 195103, '\P{Is_Blk=	CJK_compat_Ideographs_sup}', "");
Expect(1, 195103, '\P{^Is_Blk=	CJK_compat_Ideographs_sup}', "");
Expect(0, 195104, '\p{Is_Blk=	CJK_compat_Ideographs_sup}', "");
Expect(1, 195104, '\p{^Is_Blk=	CJK_compat_Ideographs_sup}', "");
Expect(1, 195104, '\P{Is_Blk=	CJK_compat_Ideographs_sup}', "");
Expect(0, 195104, '\P{^Is_Blk=	CJK_compat_Ideographs_sup}', "");
Error('\p{Block= -CJK_unified_ideographs_Extension_A:=}');
Error('\P{Block= -CJK_unified_ideographs_Extension_A:=}');
Expect(1, 19903, '\p{Block=cjkunifiedideographsextensiona}', "");
Expect(0, 19903, '\p{^Block=cjkunifiedideographsextensiona}', "");
Expect(0, 19903, '\P{Block=cjkunifiedideographsextensiona}', "");
Expect(1, 19903, '\P{^Block=cjkunifiedideographsextensiona}', "");
Expect(0, 19904, '\p{Block=cjkunifiedideographsextensiona}', "");
Expect(1, 19904, '\p{^Block=cjkunifiedideographsextensiona}', "");
Expect(1, 19904, '\P{Block=cjkunifiedideographsextensiona}', "");
Expect(0, 19904, '\P{^Block=cjkunifiedideographsextensiona}', "");
Expect(1, 19903, '\p{Block=-_CJK_unified_ideographs_extension_A}', "");
Expect(0, 19903, '\p{^Block=-_CJK_unified_ideographs_extension_A}', "");
Expect(0, 19903, '\P{Block=-_CJK_unified_ideographs_extension_A}', "");
Expect(1, 19903, '\P{^Block=-_CJK_unified_ideographs_extension_A}', "");
Expect(0, 19904, '\p{Block=-_CJK_unified_ideographs_extension_A}', "");
Expect(1, 19904, '\p{^Block=-_CJK_unified_ideographs_extension_A}', "");
Expect(1, 19904, '\P{Block=-_CJK_unified_ideographs_extension_A}', "");
Expect(0, 19904, '\P{^Block=-_CJK_unified_ideographs_extension_A}', "");
Error('\p{Blk= :=CJK_ext_A}');
Error('\P{Blk= :=CJK_ext_A}');
Expect(1, 19903, '\p{Blk=cjkexta}', "");
Expect(0, 19903, '\p{^Blk=cjkexta}', "");
Expect(0, 19903, '\P{Blk=cjkexta}', "");
Expect(1, 19903, '\P{^Blk=cjkexta}', "");
Expect(0, 19904, '\p{Blk=cjkexta}', "");
Expect(1, 19904, '\p{^Blk=cjkexta}', "");
Expect(1, 19904, '\P{Blk=cjkexta}', "");
Expect(0, 19904, '\P{^Blk=cjkexta}', "");
Expect(1, 19903, '\p{Blk=cjk_Ext_a}', "");
Expect(0, 19903, '\p{^Blk=cjk_Ext_a}', "");
Expect(0, 19903, '\P{Blk=cjk_Ext_a}', "");
Expect(1, 19903, '\P{^Blk=cjk_Ext_a}', "");
Expect(0, 19904, '\p{Blk=cjk_Ext_a}', "");
Expect(1, 19904, '\p{^Blk=cjk_Ext_a}', "");
Expect(1, 19904, '\P{Blk=cjk_Ext_a}', "");
Expect(0, 19904, '\P{^Blk=cjk_Ext_a}', "");
Error('\p{Is_Block:	:= CJK_unified_Ideographs_extension_A}');
Error('\P{Is_Block:	:= CJK_unified_Ideographs_extension_A}');
Expect(1, 19903, '\p{Is_Block=cjkunifiedideographsextensiona}', "");
Expect(0, 19903, '\p{^Is_Block=cjkunifiedideographsextensiona}', "");
Expect(0, 19903, '\P{Is_Block=cjkunifiedideographsextensiona}', "");
Expect(1, 19903, '\P{^Is_Block=cjkunifiedideographsextensiona}', "");
Expect(0, 19904, '\p{Is_Block=cjkunifiedideographsextensiona}', "");
Expect(1, 19904, '\p{^Is_Block=cjkunifiedideographsextensiona}', "");
Expect(1, 19904, '\P{Is_Block=cjkunifiedideographsextensiona}', "");
Expect(0, 19904, '\P{^Is_Block=cjkunifiedideographsextensiona}', "");
Expect(1, 19903, '\p{Is_Block= -CJK_unified_Ideographs_extension_A}', "");
Expect(0, 19903, '\p{^Is_Block= -CJK_unified_Ideographs_extension_A}', "");
Expect(0, 19903, '\P{Is_Block= -CJK_unified_Ideographs_extension_A}', "");
Expect(1, 19903, '\P{^Is_Block= -CJK_unified_Ideographs_extension_A}', "");
Expect(0, 19904, '\p{Is_Block= -CJK_unified_Ideographs_extension_A}', "");
Expect(1, 19904, '\p{^Is_Block= -CJK_unified_Ideographs_extension_A}', "");
Expect(1, 19904, '\P{Is_Block= -CJK_unified_Ideographs_extension_A}', "");
Expect(0, 19904, '\P{^Is_Block= -CJK_unified_Ideographs_extension_A}', "");
Error('\p{Is_Blk=	:=CJK_ext_A}');
Error('\P{Is_Blk=	:=CJK_ext_A}');
Expect(1, 19903, '\p{Is_Blk=cjkexta}', "");
Expect(0, 19903, '\p{^Is_Blk=cjkexta}', "");
Expect(0, 19903, '\P{Is_Blk=cjkexta}', "");
Expect(1, 19903, '\P{^Is_Blk=cjkexta}', "");
Expect(0, 19904, '\p{Is_Blk=cjkexta}', "");
Expect(1, 19904, '\p{^Is_Blk=cjkexta}', "");
Expect(1, 19904, '\P{Is_Blk=cjkexta}', "");
Expect(0, 19904, '\P{^Is_Blk=cjkexta}', "");
Expect(1, 19903, '\p{Is_Blk=_CJK_ext_A}', "");
Expect(0, 19903, '\p{^Is_Blk=_CJK_ext_A}', "");
Expect(0, 19903, '\P{Is_Blk=_CJK_ext_A}', "");
Expect(1, 19903, '\P{^Is_Blk=_CJK_ext_A}', "");
Expect(0, 19904, '\p{Is_Blk=_CJK_ext_A}', "");
Expect(1, 19904, '\p{^Is_Blk=_CJK_ext_A}', "");
Expect(1, 19904, '\P{Is_Blk=_CJK_ext_A}', "");
Expect(0, 19904, '\P{^Is_Blk=_CJK_ext_A}', "");
Error('\p{Block:   -CJK_unified_ideographs_Extension_B:=}');
Error('\P{Block:   -CJK_unified_ideographs_Extension_B:=}');
Expect(1, 173791, '\p{Block=cjkunifiedideographsextensionb}', "");
Expect(0, 173791, '\p{^Block=cjkunifiedideographsextensionb}', "");
Expect(0, 173791, '\P{Block=cjkunifiedideographsextensionb}', "");
Expect(1, 173791, '\P{^Block=cjkunifiedideographsextensionb}', "");
Expect(0, 173792, '\p{Block=cjkunifiedideographsextensionb}', "");
Expect(1, 173792, '\p{^Block=cjkunifiedideographsextensionb}', "");
Expect(1, 173792, '\P{Block=cjkunifiedideographsextensionb}', "");
Expect(0, 173792, '\P{^Block=cjkunifiedideographsextensionb}', "");
Expect(1, 173791, '\p{Block=-	CJK_Unified_Ideographs_extension_B}', "");
Expect(0, 173791, '\p{^Block=-	CJK_Unified_Ideographs_extension_B}', "");
Expect(0, 173791, '\P{Block=-	CJK_Unified_Ideographs_extension_B}', "");
Expect(1, 173791, '\P{^Block=-	CJK_Unified_Ideographs_extension_B}', "");
Expect(0, 173792, '\p{Block=-	CJK_Unified_Ideographs_extension_B}', "");
Expect(1, 173792, '\p{^Block=-	CJK_Unified_Ideographs_extension_B}', "");
Expect(1, 173792, '\P{Block=-	CJK_Unified_Ideographs_extension_B}', "");
Expect(0, 173792, '\P{^Block=-	CJK_Unified_Ideographs_extension_B}', "");
Error('\p{Blk=/a/ 	cjk_EXT_b}');
Error('\P{Blk=/a/ 	cjk_EXT_b}');
Expect(1, 173791, '\p{Blk=cjkextb}', "");
Expect(0, 173791, '\p{^Blk=cjkextb}', "");
Expect(0, 173791, '\P{Blk=cjkextb}', "");
Expect(1, 173791, '\P{^Blk=cjkextb}', "");
Expect(0, 173792, '\p{Blk=cjkextb}', "");
Expect(1, 173792, '\p{^Blk=cjkextb}', "");
Expect(1, 173792, '\P{Blk=cjkextb}', "");
Expect(0, 173792, '\P{^Blk=cjkextb}', "");
Expect(1, 173791, '\p{Blk=_ CJK_ext_b}', "");
Expect(0, 173791, '\p{^Blk=_ CJK_ext_b}', "");
Expect(0, 173791, '\P{Blk=_ CJK_ext_b}', "");
Expect(1, 173791, '\P{^Blk=_ CJK_ext_b}', "");
Expect(0, 173792, '\p{Blk=_ CJK_ext_b}', "");
Expect(1, 173792, '\p{^Blk=_ CJK_ext_b}', "");
Expect(1, 173792, '\P{Blk=_ CJK_ext_b}', "");
Expect(0, 173792, '\P{^Blk=_ CJK_ext_b}', "");
Error('\p{Is_Block=		CJK_Unified_Ideographs_Extension_b:=}');
Error('\P{Is_Block=		CJK_Unified_Ideographs_Extension_b:=}');
Expect(1, 173791, '\p{Is_Block=cjkunifiedideographsextensionb}', "");
Expect(0, 173791, '\p{^Is_Block=cjkunifiedideographsextensionb}', "");
Expect(0, 173791, '\P{Is_Block=cjkunifiedideographsextensionb}', "");
Expect(1, 173791, '\P{^Is_Block=cjkunifiedideographsextensionb}', "");
Expect(0, 173792, '\p{Is_Block=cjkunifiedideographsextensionb}', "");
Expect(1, 173792, '\p{^Is_Block=cjkunifiedideographsextensionb}', "");
Expect(1, 173792, '\P{Is_Block=cjkunifiedideographsextensionb}', "");
Expect(0, 173792, '\P{^Is_Block=cjkunifiedideographsextensionb}', "");
Expect(1, 173791, '\p{Is_Block=  cjk_Unified_Ideographs_Extension_B}', "");
Expect(0, 173791, '\p{^Is_Block=  cjk_Unified_Ideographs_Extension_B}', "");
Expect(0, 173791, '\P{Is_Block=  cjk_Unified_Ideographs_Extension_B}', "");
Expect(1, 173791, '\P{^Is_Block=  cjk_Unified_Ideographs_Extension_B}', "");
Expect(0, 173792, '\p{Is_Block=  cjk_Unified_Ideographs_Extension_B}', "");
Expect(1, 173792, '\p{^Is_Block=  cjk_Unified_Ideographs_Extension_B}', "");
Expect(1, 173792, '\P{Is_Block=  cjk_Unified_Ideographs_Extension_B}', "");
Expect(0, 173792, '\P{^Is_Block=  cjk_Unified_Ideographs_Extension_B}', "");
Error('\p{Is_Blk=/a/	CJK_Ext_B}');
Error('\P{Is_Blk=/a/	CJK_Ext_B}');
Expect(1, 173791, '\p{Is_Blk=cjkextb}', "");
Expect(0, 173791, '\p{^Is_Blk=cjkextb}', "");
Expect(0, 173791, '\P{Is_Blk=cjkextb}', "");
Expect(1, 173791, '\P{^Is_Blk=cjkextb}', "");
Expect(0, 173792, '\p{Is_Blk=cjkextb}', "");
Expect(1, 173792, '\p{^Is_Blk=cjkextb}', "");
Expect(1, 173792, '\P{Is_Blk=cjkextb}', "");
Expect(0, 173792, '\P{^Is_Blk=cjkextb}', "");
Expect(1, 173791, '\p{Is_Blk= 	cjk_Ext_b}', "");
Expect(0, 173791, '\p{^Is_Blk= 	cjk_Ext_b}', "");
Expect(0, 173791, '\P{Is_Blk= 	cjk_Ext_b}', "");
Expect(1, 173791, '\P{^Is_Blk= 	cjk_Ext_b}', "");
Expect(0, 173792, '\p{Is_Blk= 	cjk_Ext_b}', "");
Expect(1, 173792, '\p{^Is_Blk= 	cjk_Ext_b}', "");
Expect(1, 173792, '\P{Is_Blk= 	cjk_Ext_b}', "");
Expect(0, 173792, '\P{^Is_Blk= 	cjk_Ext_b}', "");
Error('\p{Block:   :=	_cjk_UNIFIED_IDEOGRAPHS_Extension_C}');
Error('\P{Block:   :=	_cjk_UNIFIED_IDEOGRAPHS_Extension_C}');
Expect(1, 177983, '\p{Block=cjkunifiedideographsextensionc}', "");
Expect(0, 177983, '\p{^Block=cjkunifiedideographsextensionc}', "");
Expect(0, 177983, '\P{Block=cjkunifiedideographsextensionc}', "");
Expect(1, 177983, '\P{^Block=cjkunifiedideographsextensionc}', "");
Expect(0, 177984, '\p{Block=cjkunifiedideographsextensionc}', "");
Expect(1, 177984, '\p{^Block=cjkunifiedideographsextensionc}', "");
Expect(1, 177984, '\P{Block=cjkunifiedideographsextensionc}', "");
Expect(0, 177984, '\P{^Block=cjkunifiedideographsextensionc}', "");
Expect(1, 177983, '\p{Block=--CJK_Unified_Ideographs_Extension_C}', "");
Expect(0, 177983, '\p{^Block=--CJK_Unified_Ideographs_Extension_C}', "");
Expect(0, 177983, '\P{Block=--CJK_Unified_Ideographs_Extension_C}', "");
Expect(1, 177983, '\P{^Block=--CJK_Unified_Ideographs_Extension_C}', "");
Expect(0, 177984, '\p{Block=--CJK_Unified_Ideographs_Extension_C}', "");
Expect(1, 177984, '\p{^Block=--CJK_Unified_Ideographs_Extension_C}', "");
Expect(1, 177984, '\P{Block=--CJK_Unified_Ideographs_Extension_C}', "");
Expect(0, 177984, '\P{^Block=--CJK_Unified_Ideographs_Extension_C}', "");
Error('\p{Blk=-CJK_EXT_c:=}');
Error('\P{Blk=-CJK_EXT_c:=}');
Expect(1, 177983, '\p{Blk=cjkextc}', "");
Expect(0, 177983, '\p{^Blk=cjkextc}', "");
Expect(0, 177983, '\P{Blk=cjkextc}', "");
Expect(1, 177983, '\P{^Blk=cjkextc}', "");
Expect(0, 177984, '\p{Blk=cjkextc}', "");
Expect(1, 177984, '\p{^Blk=cjkextc}', "");
Expect(1, 177984, '\P{Blk=cjkextc}', "");
Expect(0, 177984, '\P{^Blk=cjkextc}', "");
Expect(1, 177983, '\p{Blk:		CJK_Ext_C}', "");
Expect(0, 177983, '\p{^Blk:		CJK_Ext_C}', "");
Expect(0, 177983, '\P{Blk:		CJK_Ext_C}', "");
Expect(1, 177983, '\P{^Blk:		CJK_Ext_C}', "");
Expect(0, 177984, '\p{Blk:		CJK_Ext_C}', "");
Expect(1, 177984, '\p{^Blk:		CJK_Ext_C}', "");
Expect(1, 177984, '\P{Blk:		CJK_Ext_C}', "");
Expect(0, 177984, '\P{^Blk:		CJK_Ext_C}', "");
Error('\p{Is_Block=- CJK_Unified_ideographs_Extension_C/a/}');
Error('\P{Is_Block=- CJK_Unified_ideographs_Extension_C/a/}');
Expect(1, 177983, '\p{Is_Block=cjkunifiedideographsextensionc}', "");
Expect(0, 177983, '\p{^Is_Block=cjkunifiedideographsextensionc}', "");
Expect(0, 177983, '\P{Is_Block=cjkunifiedideographsextensionc}', "");
Expect(1, 177983, '\P{^Is_Block=cjkunifiedideographsextensionc}', "");
Expect(0, 177984, '\p{Is_Block=cjkunifiedideographsextensionc}', "");
Expect(1, 177984, '\p{^Is_Block=cjkunifiedideographsextensionc}', "");
Expect(1, 177984, '\P{Is_Block=cjkunifiedideographsextensionc}', "");
Expect(0, 177984, '\P{^Is_Block=cjkunifiedideographsextensionc}', "");
Expect(1, 177983, '\p{Is_Block:    CJK_UNIFIED_IDEOGRAPHS_EXTENSION_C}', "");
Expect(0, 177983, '\p{^Is_Block:    CJK_UNIFIED_IDEOGRAPHS_EXTENSION_C}', "");
Expect(0, 177983, '\P{Is_Block:    CJK_UNIFIED_IDEOGRAPHS_EXTENSION_C}', "");
Expect(1, 177983, '\P{^Is_Block:    CJK_UNIFIED_IDEOGRAPHS_EXTENSION_C}', "");
Expect(0, 177984, '\p{Is_Block:    CJK_UNIFIED_IDEOGRAPHS_EXTENSION_C}', "");
Expect(1, 177984, '\p{^Is_Block:    CJK_UNIFIED_IDEOGRAPHS_EXTENSION_C}', "");
Expect(1, 177984, '\P{Is_Block:    CJK_UNIFIED_IDEOGRAPHS_EXTENSION_C}', "");
Expect(0, 177984, '\P{^Is_Block:    CJK_UNIFIED_IDEOGRAPHS_EXTENSION_C}', "");
Error('\p{Is_Blk=		CJK_Ext_c:=}');
Error('\P{Is_Blk=		CJK_Ext_c:=}');
Expect(1, 177983, '\p{Is_Blk=cjkextc}', "");
Expect(0, 177983, '\p{^Is_Blk=cjkextc}', "");
Expect(0, 177983, '\P{Is_Blk=cjkextc}', "");
Expect(1, 177983, '\P{^Is_Blk=cjkextc}', "");
Expect(0, 177984, '\p{Is_Blk=cjkextc}', "");
Expect(1, 177984, '\p{^Is_Blk=cjkextc}', "");
Expect(1, 177984, '\P{Is_Blk=cjkextc}', "");
Expect(0, 177984, '\P{^Is_Blk=cjkextc}', "");
Expect(1, 177983, '\p{Is_Blk=	cjk_ext_c}', "");
Expect(0, 177983, '\p{^Is_Blk=	cjk_ext_c}', "");
Expect(0, 177983, '\P{Is_Blk=	cjk_ext_c}', "");
Expect(1, 177983, '\P{^Is_Blk=	cjk_ext_c}', "");
Expect(0, 177984, '\p{Is_Blk=	cjk_ext_c}', "");
Expect(1, 177984, '\p{^Is_Blk=	cjk_ext_c}', "");
Expect(1, 177984, '\P{Is_Blk=	cjk_ext_c}', "");
Expect(0, 177984, '\P{^Is_Blk=	cjk_ext_c}', "");
Error('\p{Block=-/a/CJK_Unified_IDEOGRAPHS_Extension_D}');
Error('\P{Block=-/a/CJK_Unified_IDEOGRAPHS_Extension_D}');
Expect(1, 178207, '\p{Block=cjkunifiedideographsextensiond}', "");
Expect(0, 178207, '\p{^Block=cjkunifiedideographsextensiond}', "");
Expect(0, 178207, '\P{Block=cjkunifiedideographsextensiond}', "");
Expect(1, 178207, '\P{^Block=cjkunifiedideographsextensiond}', "");
Expect(0, 178208, '\p{Block=cjkunifiedideographsextensiond}', "");
Expect(1, 178208, '\p{^Block=cjkunifiedideographsextensiond}', "");
Expect(1, 178208, '\P{Block=cjkunifiedideographsextensiond}', "");
Expect(0, 178208, '\P{^Block=cjkunifiedideographsextensiond}', "");
Expect(1, 178207, '\p{Block=	_CJK_UNIFIED_Ideographs_extension_D}', "");
Expect(0, 178207, '\p{^Block=	_CJK_UNIFIED_Ideographs_extension_D}', "");
Expect(0, 178207, '\P{Block=	_CJK_UNIFIED_Ideographs_extension_D}', "");
Expect(1, 178207, '\P{^Block=	_CJK_UNIFIED_Ideographs_extension_D}', "");
Expect(0, 178208, '\p{Block=	_CJK_UNIFIED_Ideographs_extension_D}', "");
Expect(1, 178208, '\p{^Block=	_CJK_UNIFIED_Ideographs_extension_D}', "");
Expect(1, 178208, '\P{Block=	_CJK_UNIFIED_Ideographs_extension_D}', "");
Expect(0, 178208, '\P{^Block=	_CJK_UNIFIED_Ideographs_extension_D}', "");
Error('\p{Blk=__CJK_EXT_d:=}');
Error('\P{Blk=__CJK_EXT_d:=}');
Expect(1, 178207, '\p{Blk=cjkextd}', "");
Expect(0, 178207, '\p{^Blk=cjkextd}', "");
Expect(0, 178207, '\P{Blk=cjkextd}', "");
Expect(1, 178207, '\P{^Blk=cjkextd}', "");
Expect(0, 178208, '\p{Blk=cjkextd}', "");
Expect(1, 178208, '\p{^Blk=cjkextd}', "");
Expect(1, 178208, '\P{Blk=cjkextd}', "");
Expect(0, 178208, '\P{^Blk=cjkextd}', "");
Expect(1, 178207, '\p{Blk=--CJK_Ext_D}', "");
Expect(0, 178207, '\p{^Blk=--CJK_Ext_D}', "");
Expect(0, 178207, '\P{Blk=--CJK_Ext_D}', "");
Expect(1, 178207, '\P{^Blk=--CJK_Ext_D}', "");
Expect(0, 178208, '\p{Blk=--CJK_Ext_D}', "");
Expect(1, 178208, '\p{^Blk=--CJK_Ext_D}', "");
Expect(1, 178208, '\P{Blk=--CJK_Ext_D}', "");
Expect(0, 178208, '\P{^Blk=--CJK_Ext_D}', "");
Error('\p{Is_Block:	:=CJK_unified_Ideographs_Extension_D}');
Error('\P{Is_Block:	:=CJK_unified_Ideographs_Extension_D}');
Expect(1, 178207, '\p{Is_Block=cjkunifiedideographsextensiond}', "");
Expect(0, 178207, '\p{^Is_Block=cjkunifiedideographsextensiond}', "");
Expect(0, 178207, '\P{Is_Block=cjkunifiedideographsextensiond}', "");
Expect(1, 178207, '\P{^Is_Block=cjkunifiedideographsextensiond}', "");
Expect(0, 178208, '\p{Is_Block=cjkunifiedideographsextensiond}', "");
Expect(1, 178208, '\p{^Is_Block=cjkunifiedideographsextensiond}', "");
Expect(1, 178208, '\P{Is_Block=cjkunifiedideographsextensiond}', "");
Expect(0, 178208, '\P{^Is_Block=cjkunifiedideographsextensiond}', "");
Expect(1, 178207, '\p{Is_Block=CJK_Unified_Ideographs_Extension_D}', "");
Expect(0, 178207, '\p{^Is_Block=CJK_Unified_Ideographs_Extension_D}', "");
Expect(0, 178207, '\P{Is_Block=CJK_Unified_Ideographs_Extension_D}', "");
Expect(1, 178207, '\P{^Is_Block=CJK_Unified_Ideographs_Extension_D}', "");
Expect(0, 178208, '\p{Is_Block=CJK_Unified_Ideographs_Extension_D}', "");
Expect(1, 178208, '\p{^Is_Block=CJK_Unified_Ideographs_Extension_D}', "");
Expect(1, 178208, '\P{Is_Block=CJK_Unified_Ideographs_Extension_D}', "");
Expect(0, 178208, '\P{^Is_Block=CJK_Unified_Ideographs_Extension_D}', "");
Error('\p{Is_Blk=:= -CJK_Ext_D}');
Error('\P{Is_Blk=:= -CJK_Ext_D}');
Expect(1, 178207, '\p{Is_Blk=cjkextd}', "");
Expect(0, 178207, '\p{^Is_Blk=cjkextd}', "");
Expect(0, 178207, '\P{Is_Blk=cjkextd}', "");
Expect(1, 178207, '\P{^Is_Blk=cjkextd}', "");
Expect(0, 178208, '\p{Is_Blk=cjkextd}', "");
Expect(1, 178208, '\p{^Is_Blk=cjkextd}', "");
Expect(1, 178208, '\P{Is_Blk=cjkextd}', "");
Expect(0, 178208, '\P{^Is_Blk=cjkextd}', "");
Expect(1, 178207, '\p{Is_Blk=  CJK_Ext_D}', "");
Expect(0, 178207, '\p{^Is_Blk=  CJK_Ext_D}', "");
Expect(0, 178207, '\P{Is_Blk=  CJK_Ext_D}', "");
Expect(1, 178207, '\P{^Is_Blk=  CJK_Ext_D}', "");
Expect(0, 178208, '\p{Is_Blk=  CJK_Ext_D}', "");
Expect(1, 178208, '\p{^Is_Blk=  CJK_Ext_D}', "");
Expect(1, 178208, '\P{Is_Blk=  CJK_Ext_D}', "");
Expect(0, 178208, '\P{^Is_Blk=  CJK_Ext_D}', "");
Error('\p{Block=cjk_RADICALS_Supplement/a/}');
Error('\P{Block=cjk_RADICALS_Supplement/a/}');
Expect(1, 12031, '\p{Block=cjkradicalssupplement}', "");
Expect(0, 12031, '\p{^Block=cjkradicalssupplement}', "");
Expect(0, 12031, '\P{Block=cjkradicalssupplement}', "");
Expect(1, 12031, '\P{^Block=cjkradicalssupplement}', "");
Expect(0, 12032, '\p{Block=cjkradicalssupplement}', "");
Expect(1, 12032, '\p{^Block=cjkradicalssupplement}', "");
Expect(1, 12032, '\P{Block=cjkradicalssupplement}', "");
Expect(0, 12032, '\P{^Block=cjkradicalssupplement}', "");
Expect(1, 12031, '\p{Block=--cjk_Radicals_Supplement}', "");
Expect(0, 12031, '\p{^Block=--cjk_Radicals_Supplement}', "");
Expect(0, 12031, '\P{Block=--cjk_Radicals_Supplement}', "");
Expect(1, 12031, '\P{^Block=--cjk_Radicals_Supplement}', "");
Expect(0, 12032, '\p{Block=--cjk_Radicals_Supplement}', "");
Expect(1, 12032, '\p{^Block=--cjk_Radicals_Supplement}', "");
Expect(1, 12032, '\P{Block=--cjk_Radicals_Supplement}', "");
Expect(0, 12032, '\P{^Block=--cjk_Radicals_Supplement}', "");
Error('\p{Blk=	:=CJK_RADICALS_Sup}');
Error('\P{Blk=	:=CJK_RADICALS_Sup}');
Expect(1, 12031, '\p{Blk=cjkradicalssup}', "");
Expect(0, 12031, '\p{^Blk=cjkradicalssup}', "");
Expect(0, 12031, '\P{Blk=cjkradicalssup}', "");
Expect(1, 12031, '\P{^Blk=cjkradicalssup}', "");
Expect(0, 12032, '\p{Blk=cjkradicalssup}', "");
Expect(1, 12032, '\p{^Blk=cjkradicalssup}', "");
Expect(1, 12032, '\P{Blk=cjkradicalssup}', "");
Expect(0, 12032, '\P{^Blk=cjkradicalssup}', "");
Expect(1, 12031, '\p{Blk=  CJK_radicals_sup}', "");
Expect(0, 12031, '\p{^Blk=  CJK_radicals_sup}', "");
Expect(0, 12031, '\P{Blk=  CJK_radicals_sup}', "");
Expect(1, 12031, '\P{^Blk=  CJK_radicals_sup}', "");
Expect(0, 12032, '\p{Blk=  CJK_radicals_sup}', "");
Expect(1, 12032, '\p{^Blk=  CJK_radicals_sup}', "");
Expect(1, 12032, '\P{Blk=  CJK_radicals_sup}', "");
Expect(0, 12032, '\P{^Blk=  CJK_radicals_sup}', "");
Error('\p{Is_Block=:=	CJK_radicals_SUPPLEMENT}');
Error('\P{Is_Block=:=	CJK_radicals_SUPPLEMENT}');
Expect(1, 12031, '\p{Is_Block=cjkradicalssupplement}', "");
Expect(0, 12031, '\p{^Is_Block=cjkradicalssupplement}', "");
Expect(0, 12031, '\P{Is_Block=cjkradicalssupplement}', "");
Expect(1, 12031, '\P{^Is_Block=cjkradicalssupplement}', "");
Expect(0, 12032, '\p{Is_Block=cjkradicalssupplement}', "");
Expect(1, 12032, '\p{^Is_Block=cjkradicalssupplement}', "");
Expect(1, 12032, '\P{Is_Block=cjkradicalssupplement}', "");
Expect(0, 12032, '\P{^Is_Block=cjkradicalssupplement}', "");
Expect(1, 12031, '\p{Is_Block=	CJK_Radicals_Supplement}', "");
Expect(0, 12031, '\p{^Is_Block=	CJK_Radicals_Supplement}', "");
Expect(0, 12031, '\P{Is_Block=	CJK_Radicals_Supplement}', "");
Expect(1, 12031, '\P{^Is_Block=	CJK_Radicals_Supplement}', "");
Expect(0, 12032, '\p{Is_Block=	CJK_Radicals_Supplement}', "");
Expect(1, 12032, '\p{^Is_Block=	CJK_Radicals_Supplement}', "");
Expect(1, 12032, '\P{Is_Block=	CJK_Radicals_Supplement}', "");
Expect(0, 12032, '\P{^Is_Block=	CJK_Radicals_Supplement}', "");
Error('\p{Is_Blk=:=--CJK_Radicals_sup}');
Error('\P{Is_Blk=:=--CJK_Radicals_sup}');
Expect(1, 12031, '\p{Is_Blk=cjkradicalssup}', "");
Expect(0, 12031, '\p{^Is_Blk=cjkradicalssup}', "");
Expect(0, 12031, '\P{Is_Blk=cjkradicalssup}', "");
Expect(1, 12031, '\P{^Is_Blk=cjkradicalssup}', "");
Expect(0, 12032, '\p{Is_Blk=cjkradicalssup}', "");
Expect(1, 12032, '\p{^Is_Blk=cjkradicalssup}', "");
Expect(1, 12032, '\P{Is_Blk=cjkradicalssup}', "");
Expect(0, 12032, '\P{^Is_Blk=cjkradicalssup}', "");
Expect(1, 12031, '\p{Is_Blk=_ CJK_radicals_sup}', "");
Expect(0, 12031, '\p{^Is_Blk=_ CJK_radicals_sup}', "");
Expect(0, 12031, '\P{Is_Blk=_ CJK_radicals_sup}', "");
Expect(1, 12031, '\P{^Is_Blk=_ CJK_radicals_sup}', "");
Expect(0, 12032, '\p{Is_Blk=_ CJK_radicals_sup}', "");
Expect(1, 12032, '\p{^Is_Blk=_ CJK_radicals_sup}', "");
Expect(1, 12032, '\P{Is_Blk=_ CJK_radicals_sup}', "");
Expect(0, 12032, '\P{^Is_Blk=_ CJK_radicals_sup}', "");
Error('\p{Block=-	cjk_Strokes/a/}');
Error('\P{Block=-	cjk_Strokes/a/}');
Expect(1, 12783, '\p{Block:   cjkstrokes}', "");
Expect(0, 12783, '\p{^Block:   cjkstrokes}', "");
Expect(0, 12783, '\P{Block:   cjkstrokes}', "");
Expect(1, 12783, '\P{^Block:   cjkstrokes}', "");
Expect(0, 12784, '\p{Block:   cjkstrokes}', "");
Expect(1, 12784, '\p{^Block:   cjkstrokes}', "");
Expect(1, 12784, '\P{Block:   cjkstrokes}', "");
Expect(0, 12784, '\P{^Block:   cjkstrokes}', "");
Expect(1, 12783, '\p{Block= -CJK_Strokes}', "");
Expect(0, 12783, '\p{^Block= -CJK_Strokes}', "");
Expect(0, 12783, '\P{Block= -CJK_Strokes}', "");
Expect(1, 12783, '\P{^Block= -CJK_Strokes}', "");
Expect(0, 12784, '\p{Block= -CJK_Strokes}', "");
Expect(1, 12784, '\p{^Block= -CJK_Strokes}', "");
Expect(1, 12784, '\P{Block= -CJK_Strokes}', "");
Expect(0, 12784, '\P{^Block= -CJK_Strokes}', "");
Error('\p{Blk=- CJK_strokes:=}');
Error('\P{Blk=- CJK_strokes:=}');
Expect(1, 12783, '\p{Blk=cjkstrokes}', "");
Expect(0, 12783, '\p{^Blk=cjkstrokes}', "");
Expect(0, 12783, '\P{Blk=cjkstrokes}', "");
Expect(1, 12783, '\P{^Blk=cjkstrokes}', "");
Expect(0, 12784, '\p{Blk=cjkstrokes}', "");
Expect(1, 12784, '\p{^Blk=cjkstrokes}', "");
Expect(1, 12784, '\P{Blk=cjkstrokes}', "");
Expect(0, 12784, '\P{^Blk=cjkstrokes}', "");
Expect(1, 12783, '\p{Blk=_CJK_Strokes}', "");
Expect(0, 12783, '\p{^Blk=_CJK_Strokes}', "");
Expect(0, 12783, '\P{Blk=_CJK_Strokes}', "");
Expect(1, 12783, '\P{^Blk=_CJK_Strokes}', "");
Expect(0, 12784, '\p{Blk=_CJK_Strokes}', "");
Expect(1, 12784, '\p{^Blk=_CJK_Strokes}', "");
Expect(1, 12784, '\P{Blk=_CJK_Strokes}', "");
Expect(0, 12784, '\P{^Blk=_CJK_Strokes}', "");
Error('\p{Is_Block=_-CJK_STROKES:=}');
Error('\P{Is_Block=_-CJK_STROKES:=}');
Expect(1, 12783, '\p{Is_Block=cjkstrokes}', "");
Expect(0, 12783, '\p{^Is_Block=cjkstrokes}', "");
Expect(0, 12783, '\P{Is_Block=cjkstrokes}', "");
Expect(1, 12783, '\P{^Is_Block=cjkstrokes}', "");
Expect(0, 12784, '\p{Is_Block=cjkstrokes}', "");
Expect(1, 12784, '\p{^Is_Block=cjkstrokes}', "");
Expect(1, 12784, '\P{Is_Block=cjkstrokes}', "");
Expect(0, 12784, '\P{^Is_Block=cjkstrokes}', "");
Expect(1, 12783, '\p{Is_Block= _CJK_Strokes}', "");
Expect(0, 12783, '\p{^Is_Block= _CJK_Strokes}', "");
Expect(0, 12783, '\P{Is_Block= _CJK_Strokes}', "");
Expect(1, 12783, '\P{^Is_Block= _CJK_Strokes}', "");
Expect(0, 12784, '\p{Is_Block= _CJK_Strokes}', "");
Expect(1, 12784, '\p{^Is_Block= _CJK_Strokes}', "");
Expect(1, 12784, '\P{Is_Block= _CJK_Strokes}', "");
Expect(0, 12784, '\P{^Is_Block= _CJK_Strokes}', "");
Error('\p{Is_Blk=/a/CJK_strokes}');
Error('\P{Is_Blk=/a/CJK_strokes}');
Expect(1, 12783, '\p{Is_Blk=cjkstrokes}', "");
Expect(0, 12783, '\p{^Is_Blk=cjkstrokes}', "");
Expect(0, 12783, '\P{Is_Blk=cjkstrokes}', "");
Expect(1, 12783, '\P{^Is_Blk=cjkstrokes}', "");
Expect(0, 12784, '\p{Is_Blk=cjkstrokes}', "");
Expect(1, 12784, '\p{^Is_Blk=cjkstrokes}', "");
Expect(1, 12784, '\P{Is_Blk=cjkstrokes}', "");
Expect(0, 12784, '\P{^Is_Blk=cjkstrokes}', "");
Expect(1, 12783, '\p{Is_Blk:	-	CJK_STROKES}', "");
Expect(0, 12783, '\p{^Is_Blk:	-	CJK_STROKES}', "");
Expect(0, 12783, '\P{Is_Blk:	-	CJK_STROKES}', "");
Expect(1, 12783, '\P{^Is_Blk:	-	CJK_STROKES}', "");
Expect(0, 12784, '\p{Is_Blk:	-	CJK_STROKES}', "");
Expect(1, 12784, '\p{^Is_Blk:	-	CJK_STROKES}', "");
Expect(1, 12784, '\P{Is_Blk:	-	CJK_STROKES}', "");
Expect(0, 12784, '\P{^Is_Blk:	-	CJK_STROKES}', "");
Error('\p{Block=/a/__CJK_symbols_and_punctuation}');
Error('\P{Block=/a/__CJK_symbols_and_punctuation}');
Expect(1, 12351, '\p{Block=cjksymbolsandpunctuation}', "");
Expect(0, 12351, '\p{^Block=cjksymbolsandpunctuation}', "");
Expect(0, 12351, '\P{Block=cjksymbolsandpunctuation}', "");
Expect(1, 12351, '\P{^Block=cjksymbolsandpunctuation}', "");
Expect(0, 12352, '\p{Block=cjksymbolsandpunctuation}', "");
Expect(1, 12352, '\p{^Block=cjksymbolsandpunctuation}', "");
Expect(1, 12352, '\P{Block=cjksymbolsandpunctuation}', "");
Expect(0, 12352, '\P{^Block=cjksymbolsandpunctuation}', "");
Expect(1, 12351, '\p{Block:	 CJK_Symbols_AND_punctuation}', "");
Expect(0, 12351, '\p{^Block:	 CJK_Symbols_AND_punctuation}', "");
Expect(0, 12351, '\P{Block:	 CJK_Symbols_AND_punctuation}', "");
Expect(1, 12351, '\P{^Block:	 CJK_Symbols_AND_punctuation}', "");
Expect(0, 12352, '\p{Block:	 CJK_Symbols_AND_punctuation}', "");
Expect(1, 12352, '\p{^Block:	 CJK_Symbols_AND_punctuation}', "");
Expect(1, 12352, '\P{Block:	 CJK_Symbols_AND_punctuation}', "");
Expect(0, 12352, '\P{^Block:	 CJK_Symbols_AND_punctuation}', "");
Error('\p{Blk=	CJK_SYMBOLS:=}');
Error('\P{Blk=	CJK_SYMBOLS:=}');
Expect(1, 12351, '\p{Blk=cjksymbols}', "");
Expect(0, 12351, '\p{^Blk=cjksymbols}', "");
Expect(0, 12351, '\P{Blk=cjksymbols}', "");
Expect(1, 12351, '\P{^Blk=cjksymbols}', "");
Expect(0, 12352, '\p{Blk=cjksymbols}', "");
Expect(1, 12352, '\p{^Blk=cjksymbols}', "");
Expect(1, 12352, '\P{Blk=cjksymbols}', "");
Expect(0, 12352, '\P{^Blk=cjksymbols}', "");
Expect(1, 12351, '\p{Blk=	 cjk_symbols}', "");
Expect(0, 12351, '\p{^Blk=	 cjk_symbols}', "");
Expect(0, 12351, '\P{Blk=	 cjk_symbols}', "");
Expect(1, 12351, '\P{^Blk=	 cjk_symbols}', "");
Expect(0, 12352, '\p{Blk=	 cjk_symbols}', "");
Expect(1, 12352, '\p{^Blk=	 cjk_symbols}', "");
Expect(1, 12352, '\P{Blk=	 cjk_symbols}', "");
Expect(0, 12352, '\P{^Blk=	 cjk_symbols}', "");
Error('\p{Is_Block=/a/	 CJK_Symbols_And_PUNCTUATION}');
Error('\P{Is_Block=/a/	 CJK_Symbols_And_PUNCTUATION}');
Expect(1, 12351, '\p{Is_Block=cjksymbolsandpunctuation}', "");
Expect(0, 12351, '\p{^Is_Block=cjksymbolsandpunctuation}', "");
Expect(0, 12351, '\P{Is_Block=cjksymbolsandpunctuation}', "");
Expect(1, 12351, '\P{^Is_Block=cjksymbolsandpunctuation}', "");
Expect(0, 12352, '\p{Is_Block=cjksymbolsandpunctuation}', "");
Expect(1, 12352, '\p{^Is_Block=cjksymbolsandpunctuation}', "");
Expect(1, 12352, '\P{Is_Block=cjksymbolsandpunctuation}', "");
Expect(0, 12352, '\P{^Is_Block=cjksymbolsandpunctuation}', "");
Expect(1, 12351, '\p{Is_Block=_CJK_Symbols_And_punctuation}', "");
Expect(0, 12351, '\p{^Is_Block=_CJK_Symbols_And_punctuation}', "");
Expect(0, 12351, '\P{Is_Block=_CJK_Symbols_And_punctuation}', "");
Expect(1, 12351, '\P{^Is_Block=_CJK_Symbols_And_punctuation}', "");
Expect(0, 12352, '\p{Is_Block=_CJK_Symbols_And_punctuation}', "");
Expect(1, 12352, '\p{^Is_Block=_CJK_Symbols_And_punctuation}', "");
Expect(1, 12352, '\P{Is_Block=_CJK_Symbols_And_punctuation}', "");
Expect(0, 12352, '\P{^Is_Block=_CJK_Symbols_And_punctuation}', "");
Error('\p{Is_Blk=/a/  cjk_symbols}');
Error('\P{Is_Blk=/a/  cjk_symbols}');
Expect(1, 12351, '\p{Is_Blk=cjksymbols}', "");
Expect(0, 12351, '\p{^Is_Blk=cjksymbols}', "");
Expect(0, 12351, '\P{Is_Blk=cjksymbols}', "");
Expect(1, 12351, '\P{^Is_Blk=cjksymbols}', "");
Expect(0, 12352, '\p{Is_Blk=cjksymbols}', "");
Expect(1, 12352, '\p{^Is_Blk=cjksymbols}', "");
Expect(1, 12352, '\P{Is_Blk=cjksymbols}', "");
Expect(0, 12352, '\P{^Is_Blk=cjksymbols}', "");
Expect(1, 12351, '\p{Is_Blk=__cjk_symbols}', "");
Expect(0, 12351, '\p{^Is_Blk=__cjk_symbols}', "");
Expect(0, 12351, '\P{Is_Blk=__cjk_symbols}', "");
Expect(1, 12351, '\P{^Is_Blk=__cjk_symbols}', "");
Expect(0, 12352, '\p{Is_Blk=__cjk_symbols}', "");
Expect(1, 12352, '\p{^Is_Blk=__cjk_symbols}', "");
Expect(1, 12352, '\P{Is_Blk=__cjk_symbols}', "");
Expect(0, 12352, '\P{^Is_Blk=__cjk_symbols}', "");
Error('\p{Block=:=--Carian}');
Error('\P{Block=:=--Carian}');
Expect(1, 66271, '\p{Block=carian}', "");
Expect(0, 66271, '\p{^Block=carian}', "");
Expect(0, 66271, '\P{Block=carian}', "");
Expect(1, 66271, '\P{^Block=carian}', "");
Expect(0, 66272, '\p{Block=carian}', "");
Expect(1, 66272, '\p{^Block=carian}', "");
Expect(1, 66272, '\P{Block=carian}', "");
Expect(0, 66272, '\P{^Block=carian}', "");
Expect(1, 66271, '\p{Block=	_Carian}', "");
Expect(0, 66271, '\p{^Block=	_Carian}', "");
Expect(0, 66271, '\P{Block=	_Carian}', "");
Expect(1, 66271, '\P{^Block=	_Carian}', "");
Expect(0, 66272, '\p{Block=	_Carian}', "");
Expect(1, 66272, '\p{^Block=	_Carian}', "");
Expect(1, 66272, '\P{Block=	_Carian}', "");
Expect(0, 66272, '\P{^Block=	_Carian}', "");
Error('\p{Blk=/a/_	Carian}');
Error('\P{Blk=/a/_	Carian}');
Expect(1, 66271, '\p{Blk=carian}', "");
Expect(0, 66271, '\p{^Blk=carian}', "");
Expect(0, 66271, '\P{Blk=carian}', "");
Expect(1, 66271, '\P{^Blk=carian}', "");
Expect(0, 66272, '\p{Blk=carian}', "");
Expect(1, 66272, '\p{^Blk=carian}', "");
Expect(1, 66272, '\P{Blk=carian}', "");
Expect(0, 66272, '\P{^Blk=carian}', "");
Expect(1, 66271, '\p{Blk: _	carian}', "");
Expect(0, 66271, '\p{^Blk: _	carian}', "");
Expect(0, 66271, '\P{Blk: _	carian}', "");
Expect(1, 66271, '\P{^Blk: _	carian}', "");
Expect(0, 66272, '\p{Blk: _	carian}', "");
Expect(1, 66272, '\p{^Blk: _	carian}', "");
Expect(1, 66272, '\P{Blk: _	carian}', "");
Expect(0, 66272, '\P{^Blk: _	carian}', "");
Error('\p{Is_Block= CARIAN/a/}');
Error('\P{Is_Block= CARIAN/a/}');
Expect(1, 66271, '\p{Is_Block=carian}', "");
Expect(0, 66271, '\p{^Is_Block=carian}', "");
Expect(0, 66271, '\P{Is_Block=carian}', "");
Expect(1, 66271, '\P{^Is_Block=carian}', "");
Expect(0, 66272, '\p{Is_Block=carian}', "");
Expect(1, 66272, '\p{^Is_Block=carian}', "");
Expect(1, 66272, '\P{Is_Block=carian}', "");
Expect(0, 66272, '\P{^Is_Block=carian}', "");
Expect(1, 66271, '\p{Is_Block=-_carian}', "");
Expect(0, 66271, '\p{^Is_Block=-_carian}', "");
Expect(0, 66271, '\P{Is_Block=-_carian}', "");
Expect(1, 66271, '\P{^Is_Block=-_carian}', "");
Expect(0, 66272, '\p{Is_Block=-_carian}', "");
Expect(1, 66272, '\p{^Is_Block=-_carian}', "");
Expect(1, 66272, '\P{Is_Block=-_carian}', "");
Expect(0, 66272, '\P{^Is_Block=-_carian}', "");
Error('\p{Is_Blk=:=	-Carian}');
Error('\P{Is_Blk=:=	-Carian}');
Expect(1, 66271, '\p{Is_Blk=carian}', "");
Expect(0, 66271, '\p{^Is_Blk=carian}', "");
Expect(0, 66271, '\P{Is_Blk=carian}', "");
Expect(1, 66271, '\P{^Is_Blk=carian}', "");
Expect(0, 66272, '\p{Is_Blk=carian}', "");
Expect(1, 66272, '\p{^Is_Blk=carian}', "");
Expect(1, 66272, '\P{Is_Blk=carian}', "");
Expect(0, 66272, '\P{^Is_Blk=carian}', "");
Expect(1, 66271, '\p{Is_Blk=		Carian}', "");
Expect(0, 66271, '\p{^Is_Blk=		Carian}', "");
Expect(0, 66271, '\P{Is_Blk=		Carian}', "");
Expect(1, 66271, '\P{^Is_Blk=		Carian}', "");
Expect(0, 66272, '\p{Is_Blk=		Carian}', "");
Expect(1, 66272, '\p{^Is_Blk=		Carian}', "");
Expect(1, 66272, '\P{Is_Blk=		Carian}', "");
Expect(0, 66272, '\P{^Is_Blk=		Carian}', "");
Error('\p{Block=-/a/CHAKMA}');
Error('\P{Block=-/a/CHAKMA}');
Expect(1, 69967, '\p{Block=chakma}', "");
Expect(0, 69967, '\p{^Block=chakma}', "");
Expect(0, 69967, '\P{Block=chakma}', "");
Expect(1, 69967, '\P{^Block=chakma}', "");
Expect(0, 69968, '\p{Block=chakma}', "");
Expect(1, 69968, '\p{^Block=chakma}', "");
Expect(1, 69968, '\P{Block=chakma}', "");
Expect(0, 69968, '\P{^Block=chakma}', "");
Expect(1, 69967, '\p{Block=--chakma}', "");
Expect(0, 69967, '\p{^Block=--chakma}', "");
Expect(0, 69967, '\P{Block=--chakma}', "");
Expect(1, 69967, '\P{^Block=--chakma}', "");
Expect(0, 69968, '\p{Block=--chakma}', "");
Expect(1, 69968, '\p{^Block=--chakma}', "");
Expect(1, 69968, '\P{Block=--chakma}', "");
Expect(0, 69968, '\P{^Block=--chakma}', "");
Error('\p{Blk:	/a/chakma}');
Error('\P{Blk:	/a/chakma}');
Expect(1, 69967, '\p{Blk=chakma}', "");
Expect(0, 69967, '\p{^Blk=chakma}', "");
Expect(0, 69967, '\P{Blk=chakma}', "");
Expect(1, 69967, '\P{^Blk=chakma}', "");
Expect(0, 69968, '\p{Blk=chakma}', "");
Expect(1, 69968, '\p{^Blk=chakma}', "");
Expect(1, 69968, '\P{Blk=chakma}', "");
Expect(0, 69968, '\P{^Blk=chakma}', "");
Expect(1, 69967, '\p{Blk=--chakma}', "");
Expect(0, 69967, '\p{^Blk=--chakma}', "");
Expect(0, 69967, '\P{Blk=--chakma}', "");
Expect(1, 69967, '\P{^Blk=--chakma}', "");
Expect(0, 69968, '\p{Blk=--chakma}', "");
Expect(1, 69968, '\p{^Blk=--chakma}', "");
Expect(1, 69968, '\P{Blk=--chakma}', "");
Expect(0, 69968, '\P{^Blk=--chakma}', "");
Error('\p{Is_Block:	:= -chakma}');
Error('\P{Is_Block:	:= -chakma}');
Expect(1, 69967, '\p{Is_Block=chakma}', "");
Expect(0, 69967, '\p{^Is_Block=chakma}', "");
Expect(0, 69967, '\P{Is_Block=chakma}', "");
Expect(1, 69967, '\P{^Is_Block=chakma}', "");
Expect(0, 69968, '\p{Is_Block=chakma}', "");
Expect(1, 69968, '\p{^Is_Block=chakma}', "");
Expect(1, 69968, '\P{Is_Block=chakma}', "");
Expect(0, 69968, '\P{^Is_Block=chakma}', "");
Expect(1, 69967, '\p{Is_Block= -Chakma}', "");
Expect(0, 69967, '\p{^Is_Block= -Chakma}', "");
Expect(0, 69967, '\P{Is_Block= -Chakma}', "");
Expect(1, 69967, '\P{^Is_Block= -Chakma}', "");
Expect(0, 69968, '\p{Is_Block= -Chakma}', "");
Expect(1, 69968, '\p{^Is_Block= -Chakma}', "");
Expect(1, 69968, '\P{Is_Block= -Chakma}', "");
Expect(0, 69968, '\P{^Is_Block= -Chakma}', "");
Error('\p{Is_Blk=:=-Chakma}');
Error('\P{Is_Blk=:=-Chakma}');
Expect(1, 69967, '\p{Is_Blk=chakma}', "");
Expect(0, 69967, '\p{^Is_Blk=chakma}', "");
Expect(0, 69967, '\P{Is_Blk=chakma}', "");
Expect(1, 69967, '\P{^Is_Blk=chakma}', "");
Expect(0, 69968, '\p{Is_Blk=chakma}', "");
Expect(1, 69968, '\p{^Is_Blk=chakma}', "");
Expect(1, 69968, '\P{Is_Blk=chakma}', "");
Expect(0, 69968, '\P{^Is_Blk=chakma}', "");
Expect(1, 69967, '\p{Is_Blk=		chakma}', "");
Expect(0, 69967, '\p{^Is_Blk=		chakma}', "");
Expect(0, 69967, '\P{Is_Blk=		chakma}', "");
Expect(1, 69967, '\P{^Is_Blk=		chakma}', "");
Expect(0, 69968, '\p{Is_Blk=		chakma}', "");
Expect(1, 69968, '\p{^Is_Blk=		chakma}', "");
Expect(1, 69968, '\P{Is_Blk=		chakma}', "");
Expect(0, 69968, '\P{^Is_Blk=		chakma}', "");
Error('\p{Block= _Cham:=}');
Error('\P{Block= _Cham:=}');
Expect(1, 43615, '\p{Block=cham}', "");
Expect(0, 43615, '\p{^Block=cham}', "");
Expect(0, 43615, '\P{Block=cham}', "");
Expect(1, 43615, '\P{^Block=cham}', "");
Expect(0, 43616, '\p{Block=cham}', "");
Expect(1, 43616, '\p{^Block=cham}', "");
Expect(1, 43616, '\P{Block=cham}', "");
Expect(0, 43616, '\P{^Block=cham}', "");
Expect(1, 43615, '\p{Block:-cham}', "");
Expect(0, 43615, '\p{^Block:-cham}', "");
Expect(0, 43615, '\P{Block:-cham}', "");
Expect(1, 43615, '\P{^Block:-cham}', "");
Expect(0, 43616, '\p{Block:-cham}', "");
Expect(1, 43616, '\p{^Block:-cham}', "");
Expect(1, 43616, '\P{Block:-cham}', "");
Expect(0, 43616, '\P{^Block:-cham}', "");
Error('\p{Blk=	/a/cham}');
Error('\P{Blk=	/a/cham}');
Expect(1, 43615, '\p{Blk=cham}', "");
Expect(0, 43615, '\p{^Blk=cham}', "");
Expect(0, 43615, '\P{Blk=cham}', "");
Expect(1, 43615, '\P{^Blk=cham}', "");
Expect(0, 43616, '\p{Blk=cham}', "");
Expect(1, 43616, '\p{^Blk=cham}', "");
Expect(1, 43616, '\P{Blk=cham}', "");
Expect(0, 43616, '\P{^Blk=cham}', "");
Expect(1, 43615, '\p{Blk=_CHAM}', "");
Expect(0, 43615, '\p{^Blk=_CHAM}', "");
Expect(0, 43615, '\P{Blk=_CHAM}', "");
Expect(1, 43615, '\P{^Blk=_CHAM}', "");
Expect(0, 43616, '\p{Blk=_CHAM}', "");
Expect(1, 43616, '\p{^Blk=_CHAM}', "");
Expect(1, 43616, '\P{Blk=_CHAM}', "");
Expect(0, 43616, '\P{^Blk=_CHAM}', "");
Error('\p{Is_Block: :=Cham}');
Error('\P{Is_Block: :=Cham}');
Expect(1, 43615, '\p{Is_Block=cham}', "");
Expect(0, 43615, '\p{^Is_Block=cham}', "");
Expect(0, 43615, '\P{Is_Block=cham}', "");
Expect(1, 43615, '\P{^Is_Block=cham}', "");
Expect(0, 43616, '\p{Is_Block=cham}', "");
Expect(1, 43616, '\p{^Is_Block=cham}', "");
Expect(1, 43616, '\P{Is_Block=cham}', "");
Expect(0, 43616, '\P{^Is_Block=cham}', "");
Expect(1, 43615, '\p{Is_Block=-cham}', "");
Expect(0, 43615, '\p{^Is_Block=-cham}', "");
Expect(0, 43615, '\P{Is_Block=-cham}', "");
Expect(1, 43615, '\P{^Is_Block=-cham}', "");
Expect(0, 43616, '\p{Is_Block=-cham}', "");
Expect(1, 43616, '\p{^Is_Block=-cham}', "");
Expect(1, 43616, '\P{Is_Block=-cham}', "");
Expect(0, 43616, '\P{^Is_Block=-cham}', "");
Error('\p{Is_Blk=--cham:=}');
Error('\P{Is_Blk=--cham:=}');
Expect(1, 43615, '\p{Is_Blk=cham}', "");
Expect(0, 43615, '\p{^Is_Blk=cham}', "");
Expect(0, 43615, '\P{Is_Blk=cham}', "");
Expect(1, 43615, '\P{^Is_Blk=cham}', "");
Expect(0, 43616, '\p{Is_Blk=cham}', "");
Expect(1, 43616, '\p{^Is_Blk=cham}', "");
Expect(1, 43616, '\P{Is_Blk=cham}', "");
Expect(0, 43616, '\P{^Is_Blk=cham}', "");
Expect(1, 43615, '\p{Is_Blk:	 CHAM}', "");
Expect(0, 43615, '\p{^Is_Blk:	 CHAM}', "");
Expect(0, 43615, '\P{Is_Blk:	 CHAM}', "");
Expect(1, 43615, '\P{^Is_Blk:	 CHAM}', "");
Expect(0, 43616, '\p{Is_Blk:	 CHAM}', "");
Expect(1, 43616, '\p{^Is_Blk:	 CHAM}', "");
Expect(1, 43616, '\P{Is_Blk:	 CHAM}', "");
Expect(0, 43616, '\P{^Is_Blk:	 CHAM}', "");
Error('\p{Block=--Cherokee:=}');
Error('\P{Block=--Cherokee:=}');
Expect(1, 5119, '\p{Block=cherokee}', "");
Expect(0, 5119, '\p{^Block=cherokee}', "");
Expect(0, 5119, '\P{Block=cherokee}', "");
Expect(1, 5119, '\P{^Block=cherokee}', "");
Expect(0, 5120, '\p{Block=cherokee}', "");
Expect(1, 5120, '\p{^Block=cherokee}', "");
Expect(1, 5120, '\P{Block=cherokee}', "");
Expect(0, 5120, '\P{^Block=cherokee}', "");
Expect(1, 5119, '\p{Block=_CHEROKEE}', "");
Expect(0, 5119, '\p{^Block=_CHEROKEE}', "");
Expect(0, 5119, '\P{Block=_CHEROKEE}', "");
Expect(1, 5119, '\P{^Block=_CHEROKEE}', "");
Expect(0, 5120, '\p{Block=_CHEROKEE}', "");
Expect(1, 5120, '\p{^Block=_CHEROKEE}', "");
Expect(1, 5120, '\P{Block=_CHEROKEE}', "");
Expect(0, 5120, '\P{^Block=_CHEROKEE}', "");
Error('\p{Blk:	:=	 Cherokee}');
Error('\P{Blk:	:=	 Cherokee}');
Expect(1, 5119, '\p{Blk=cherokee}', "");
Expect(0, 5119, '\p{^Blk=cherokee}', "");
Expect(0, 5119, '\P{Blk=cherokee}', "");
Expect(1, 5119, '\P{^Blk=cherokee}', "");
Expect(0, 5120, '\p{Blk=cherokee}', "");
Expect(1, 5120, '\p{^Blk=cherokee}', "");
Expect(1, 5120, '\P{Blk=cherokee}', "");
Expect(0, 5120, '\P{^Blk=cherokee}', "");
Expect(1, 5119, '\p{Blk=		cherokee}', "");
Expect(0, 5119, '\p{^Blk=		cherokee}', "");
Expect(0, 5119, '\P{Blk=		cherokee}', "");
Expect(1, 5119, '\P{^Blk=		cherokee}', "");
Expect(0, 5120, '\p{Blk=		cherokee}', "");
Expect(1, 5120, '\p{^Blk=		cherokee}', "");
Expect(1, 5120, '\P{Blk=		cherokee}', "");
Expect(0, 5120, '\P{^Blk=		cherokee}', "");
Error('\p{Is_Block=:=		Cherokee}');
Error('\P{Is_Block=:=		Cherokee}');
Expect(1, 5119, '\p{Is_Block=cherokee}', "");
Expect(0, 5119, '\p{^Is_Block=cherokee}', "");
Expect(0, 5119, '\P{Is_Block=cherokee}', "");
Expect(1, 5119, '\P{^Is_Block=cherokee}', "");
Expect(0, 5120, '\p{Is_Block=cherokee}', "");
Expect(1, 5120, '\p{^Is_Block=cherokee}', "");
Expect(1, 5120, '\P{Is_Block=cherokee}', "");
Expect(0, 5120, '\P{^Is_Block=cherokee}', "");
Expect(1, 5119, '\p{Is_Block=_	Cherokee}', "");
Expect(0, 5119, '\p{^Is_Block=_	Cherokee}', "");
Expect(0, 5119, '\P{Is_Block=_	Cherokee}', "");
Expect(1, 5119, '\P{^Is_Block=_	Cherokee}', "");
Expect(0, 5120, '\p{Is_Block=_	Cherokee}', "");
Expect(1, 5120, '\p{^Is_Block=_	Cherokee}', "");
Expect(1, 5120, '\P{Is_Block=_	Cherokee}', "");
Expect(0, 5120, '\P{^Is_Block=_	Cherokee}', "");
Error('\p{Is_Blk=-Cherokee/a/}');
Error('\P{Is_Blk=-Cherokee/a/}');
Expect(1, 5119, '\p{Is_Blk=cherokee}', "");
Expect(0, 5119, '\p{^Is_Blk=cherokee}', "");
Expect(0, 5119, '\P{Is_Blk=cherokee}', "");
Expect(1, 5119, '\P{^Is_Blk=cherokee}', "");
Expect(0, 5120, '\p{Is_Blk=cherokee}', "");
Expect(1, 5120, '\p{^Is_Blk=cherokee}', "");
Expect(1, 5120, '\P{Is_Blk=cherokee}', "");
Expect(0, 5120, '\P{^Is_Blk=cherokee}', "");
Expect(1, 5119, '\p{Is_Blk=	 Cherokee}', "");
Expect(0, 5119, '\p{^Is_Blk=	 Cherokee}', "");
Expect(0, 5119, '\P{Is_Blk=	 Cherokee}', "");
Expect(1, 5119, '\P{^Is_Blk=	 Cherokee}', "");
Expect(0, 5120, '\p{Is_Blk=	 Cherokee}', "");
Expect(1, 5120, '\p{^Is_Blk=	 Cherokee}', "");
Expect(1, 5120, '\P{Is_Blk=	 Cherokee}', "");
Expect(0, 5120, '\P{^Is_Blk=	 Cherokee}', "");
Error('\p{Block=	/a/Hangul_compatibility_Jamo}');
Error('\P{Block=	/a/Hangul_compatibility_Jamo}');
Expect(1, 12687, '\p{Block=hangulcompatibilityjamo}', "");
Expect(0, 12687, '\p{^Block=hangulcompatibilityjamo}', "");
Expect(0, 12687, '\P{Block=hangulcompatibilityjamo}', "");
Expect(1, 12687, '\P{^Block=hangulcompatibilityjamo}', "");
Expect(0, 12688, '\p{Block=hangulcompatibilityjamo}', "");
Expect(1, 12688, '\p{^Block=hangulcompatibilityjamo}', "");
Expect(1, 12688, '\P{Block=hangulcompatibilityjamo}', "");
Expect(0, 12688, '\P{^Block=hangulcompatibilityjamo}', "");
Expect(1, 12687, '\p{Block:	 	HANGUL_COMPATIBILITY_jamo}', "");
Expect(0, 12687, '\p{^Block:	 	HANGUL_COMPATIBILITY_jamo}', "");
Expect(0, 12687, '\P{Block:	 	HANGUL_COMPATIBILITY_jamo}', "");
Expect(1, 12687, '\P{^Block:	 	HANGUL_COMPATIBILITY_jamo}', "");
Expect(0, 12688, '\p{Block:	 	HANGUL_COMPATIBILITY_jamo}', "");
Expect(1, 12688, '\p{^Block:	 	HANGUL_COMPATIBILITY_jamo}', "");
Expect(1, 12688, '\P{Block:	 	HANGUL_COMPATIBILITY_jamo}', "");
Expect(0, 12688, '\P{^Block:	 	HANGUL_COMPATIBILITY_jamo}', "");
Error('\p{Blk=_-COMPAT_Jamo:=}');
Error('\P{Blk=_-COMPAT_Jamo:=}');
Expect(1, 12687, '\p{Blk=compatjamo}', "");
Expect(0, 12687, '\p{^Blk=compatjamo}', "");
Expect(0, 12687, '\P{Blk=compatjamo}', "");
Expect(1, 12687, '\P{^Blk=compatjamo}', "");
Expect(0, 12688, '\p{Blk=compatjamo}', "");
Expect(1, 12688, '\p{^Blk=compatjamo}', "");
Expect(1, 12688, '\P{Blk=compatjamo}', "");
Expect(0, 12688, '\P{^Blk=compatjamo}', "");
Expect(1, 12687, '\p{Blk= _Compat_jamo}', "");
Expect(0, 12687, '\p{^Blk= _Compat_jamo}', "");
Expect(0, 12687, '\P{Blk= _Compat_jamo}', "");
Expect(1, 12687, '\P{^Blk= _Compat_jamo}', "");
Expect(0, 12688, '\p{Blk= _Compat_jamo}', "");
Expect(1, 12688, '\p{^Blk= _Compat_jamo}', "");
Expect(1, 12688, '\P{Blk= _Compat_jamo}', "");
Expect(0, 12688, '\P{^Blk= _Compat_jamo}', "");
Error('\p{Is_Block= Hangul_compatibility_Jamo:=}');
Error('\P{Is_Block= Hangul_compatibility_Jamo:=}');
Expect(1, 12687, '\p{Is_Block=hangulcompatibilityjamo}', "");
Expect(0, 12687, '\p{^Is_Block=hangulcompatibilityjamo}', "");
Expect(0, 12687, '\P{Is_Block=hangulcompatibilityjamo}', "");
Expect(1, 12687, '\P{^Is_Block=hangulcompatibilityjamo}', "");
Expect(0, 12688, '\p{Is_Block=hangulcompatibilityjamo}', "");
Expect(1, 12688, '\p{^Is_Block=hangulcompatibilityjamo}', "");
Expect(1, 12688, '\P{Is_Block=hangulcompatibilityjamo}', "");
Expect(0, 12688, '\P{^Is_Block=hangulcompatibilityjamo}', "");
Expect(1, 12687, '\p{Is_Block=	-Hangul_compatibility_Jamo}', "");
Expect(0, 12687, '\p{^Is_Block=	-Hangul_compatibility_Jamo}', "");
Expect(0, 12687, '\P{Is_Block=	-Hangul_compatibility_Jamo}', "");
Expect(1, 12687, '\P{^Is_Block=	-Hangul_compatibility_Jamo}', "");
Expect(0, 12688, '\p{Is_Block=	-Hangul_compatibility_Jamo}', "");
Expect(1, 12688, '\p{^Is_Block=	-Hangul_compatibility_Jamo}', "");
Expect(1, 12688, '\P{Is_Block=	-Hangul_compatibility_Jamo}', "");
Expect(0, 12688, '\P{^Is_Block=	-Hangul_compatibility_Jamo}', "");
Error('\p{Is_Blk=:=  Compat_jamo}');
Error('\P{Is_Blk=:=  Compat_jamo}');
Expect(1, 12687, '\p{Is_Blk=compatjamo}', "");
Expect(0, 12687, '\p{^Is_Blk=compatjamo}', "");
Expect(0, 12687, '\P{Is_Blk=compatjamo}', "");
Expect(1, 12687, '\P{^Is_Blk=compatjamo}', "");
Expect(0, 12688, '\p{Is_Blk=compatjamo}', "");
Expect(1, 12688, '\p{^Is_Blk=compatjamo}', "");
Expect(1, 12688, '\P{Is_Blk=compatjamo}', "");
Expect(0, 12688, '\P{^Is_Blk=compatjamo}', "");
Expect(1, 12687, '\p{Is_Blk=- Compat_jamo}', "");
Expect(0, 12687, '\p{^Is_Blk=- Compat_jamo}', "");
Expect(0, 12687, '\P{Is_Blk=- Compat_jamo}', "");
Expect(1, 12687, '\P{^Is_Blk=- Compat_jamo}', "");
Expect(0, 12688, '\p{Is_Blk=- Compat_jamo}', "");
Expect(1, 12688, '\p{^Is_Blk=- Compat_jamo}', "");
Expect(1, 12688, '\P{Is_Blk=- Compat_jamo}', "");
Expect(0, 12688, '\P{^Is_Blk=- Compat_jamo}', "");
Error('\p{Block:   := control_PICTURES}');
Error('\P{Block:   := control_PICTURES}');
Expect(1, 9279, '\p{Block:   controlpictures}', "");
Expect(0, 9279, '\p{^Block:   controlpictures}', "");
Expect(0, 9279, '\P{Block:   controlpictures}', "");
Expect(1, 9279, '\P{^Block:   controlpictures}', "");
Expect(0, 9280, '\p{Block:   controlpictures}', "");
Expect(1, 9280, '\p{^Block:   controlpictures}', "");
Expect(1, 9280, '\P{Block:   controlpictures}', "");
Expect(0, 9280, '\P{^Block:   controlpictures}', "");
Expect(1, 9279, '\p{Block=- control_PICTURES}', "");
Expect(0, 9279, '\p{^Block=- control_PICTURES}', "");
Expect(0, 9279, '\P{Block=- control_PICTURES}', "");
Expect(1, 9279, '\P{^Block=- control_PICTURES}', "");
Expect(0, 9280, '\p{Block=- control_PICTURES}', "");
Expect(1, 9280, '\p{^Block=- control_PICTURES}', "");
Expect(1, 9280, '\P{Block=- control_PICTURES}', "");
Expect(0, 9280, '\P{^Block=- control_PICTURES}', "");
Error('\p{Blk= 	CONTROL_PICTURES:=}');
Error('\P{Blk= 	CONTROL_PICTURES:=}');
Expect(1, 9279, '\p{Blk: controlpictures}', "");
Expect(0, 9279, '\p{^Blk: controlpictures}', "");
Expect(0, 9279, '\P{Blk: controlpictures}', "");
Expect(1, 9279, '\P{^Blk: controlpictures}', "");
Expect(0, 9280, '\p{Blk: controlpictures}', "");
Expect(1, 9280, '\p{^Blk: controlpictures}', "");
Expect(1, 9280, '\P{Blk: controlpictures}', "");
Expect(0, 9280, '\P{^Blk: controlpictures}', "");
Expect(1, 9279, '\p{Blk=_Control_pictures}', "");
Expect(0, 9279, '\p{^Blk=_Control_pictures}', "");
Expect(0, 9279, '\P{Blk=_Control_pictures}', "");
Expect(1, 9279, '\P{^Blk=_Control_pictures}', "");
Expect(0, 9280, '\p{Blk=_Control_pictures}', "");
Expect(1, 9280, '\p{^Blk=_Control_pictures}', "");
Expect(1, 9280, '\P{Blk=_Control_pictures}', "");
Expect(0, 9280, '\P{^Blk=_Control_pictures}', "");
Error('\p{Is_Block=-:=CONTROL_Pictures}');
Error('\P{Is_Block=-:=CONTROL_Pictures}');
Expect(1, 9279, '\p{Is_Block=controlpictures}', "");
Expect(0, 9279, '\p{^Is_Block=controlpictures}', "");
Expect(0, 9279, '\P{Is_Block=controlpictures}', "");
Expect(1, 9279, '\P{^Is_Block=controlpictures}', "");
Expect(0, 9280, '\p{Is_Block=controlpictures}', "");
Expect(1, 9280, '\p{^Is_Block=controlpictures}', "");
Expect(1, 9280, '\P{Is_Block=controlpictures}', "");
Expect(0, 9280, '\P{^Is_Block=controlpictures}', "");
Expect(1, 9279, '\p{Is_Block:_control_PICTURES}', "");
Expect(0, 9279, '\p{^Is_Block:_control_PICTURES}', "");
Expect(0, 9279, '\P{Is_Block:_control_PICTURES}', "");
Expect(1, 9279, '\P{^Is_Block:_control_PICTURES}', "");
Expect(0, 9280, '\p{Is_Block:_control_PICTURES}', "");
Expect(1, 9280, '\p{^Is_Block:_control_PICTURES}', "");
Expect(1, 9280, '\P{Is_Block:_control_PICTURES}', "");
Expect(0, 9280, '\P{^Is_Block:_control_PICTURES}', "");
Error('\p{Is_Blk=	/a/CONTROL_PICTURES}');
Error('\P{Is_Blk=	/a/CONTROL_PICTURES}');
Expect(1, 9279, '\p{Is_Blk=controlpictures}', "");
Expect(0, 9279, '\p{^Is_Blk=controlpictures}', "");
Expect(0, 9279, '\P{Is_Blk=controlpictures}', "");
Expect(1, 9279, '\P{^Is_Blk=controlpictures}', "");
Expect(0, 9280, '\p{Is_Blk=controlpictures}', "");
Expect(1, 9280, '\p{^Is_Blk=controlpictures}', "");
Expect(1, 9280, '\P{Is_Blk=controlpictures}', "");
Expect(0, 9280, '\P{^Is_Blk=controlpictures}', "");
Expect(1, 9279, '\p{Is_Blk=_	Control_PICTURES}', "");
Expect(0, 9279, '\p{^Is_Blk=_	Control_PICTURES}', "");
Expect(0, 9279, '\P{Is_Blk=_	Control_PICTURES}', "");
Expect(1, 9279, '\P{^Is_Blk=_	Control_PICTURES}', "");
Expect(0, 9280, '\p{Is_Blk=_	Control_PICTURES}', "");
Expect(1, 9280, '\p{^Is_Blk=_	Control_PICTURES}', "");
Expect(1, 9280, '\P{Is_Blk=_	Control_PICTURES}', "");
Expect(0, 9280, '\P{^Is_Blk=_	Control_PICTURES}', "");
Error('\p{Block= :=Coptic}');
Error('\P{Block= :=Coptic}');
Expect(1, 11519, '\p{Block:	coptic}', "");
Expect(0, 11519, '\p{^Block:	coptic}', "");
Expect(0, 11519, '\P{Block:	coptic}', "");
Expect(1, 11519, '\P{^Block:	coptic}', "");
Expect(0, 11520, '\p{Block:	coptic}', "");
Expect(1, 11520, '\p{^Block:	coptic}', "");
Expect(1, 11520, '\P{Block:	coptic}', "");
Expect(0, 11520, '\P{^Block:	coptic}', "");
Expect(1, 11519, '\p{Block=	 Coptic}', "");
Expect(0, 11519, '\p{^Block=	 Coptic}', "");
Expect(0, 11519, '\P{Block=	 Coptic}', "");
Expect(1, 11519, '\P{^Block=	 Coptic}', "");
Expect(0, 11520, '\p{Block=	 Coptic}', "");
Expect(1, 11520, '\p{^Block=	 Coptic}', "");
Expect(1, 11520, '\P{Block=	 Coptic}', "");
Expect(0, 11520, '\P{^Block=	 Coptic}', "");
Error('\p{Blk=_:=Coptic}');
Error('\P{Blk=_:=Coptic}');
Expect(1, 11519, '\p{Blk=coptic}', "");
Expect(0, 11519, '\p{^Blk=coptic}', "");
Expect(0, 11519, '\P{Blk=coptic}', "");
Expect(1, 11519, '\P{^Blk=coptic}', "");
Expect(0, 11520, '\p{Blk=coptic}', "");
Expect(1, 11520, '\p{^Blk=coptic}', "");
Expect(1, 11520, '\P{Blk=coptic}', "");
Expect(0, 11520, '\P{^Blk=coptic}', "");
Expect(1, 11519, '\p{Blk:	-_Coptic}', "");
Expect(0, 11519, '\p{^Blk:	-_Coptic}', "");
Expect(0, 11519, '\P{Blk:	-_Coptic}', "");
Expect(1, 11519, '\P{^Blk:	-_Coptic}', "");
Expect(0, 11520, '\p{Blk:	-_Coptic}', "");
Expect(1, 11520, '\p{^Blk:	-_Coptic}', "");
Expect(1, 11520, '\P{Blk:	-_Coptic}', "");
Expect(0, 11520, '\P{^Blk:	-_Coptic}', "");
Error('\p{Is_Block=_/a/Coptic}');
Error('\P{Is_Block=_/a/Coptic}');
Expect(1, 11519, '\p{Is_Block=coptic}', "");
Expect(0, 11519, '\p{^Is_Block=coptic}', "");
Expect(0, 11519, '\P{Is_Block=coptic}', "");
Expect(1, 11519, '\P{^Is_Block=coptic}', "");
Expect(0, 11520, '\p{Is_Block=coptic}', "");
Expect(1, 11520, '\p{^Is_Block=coptic}', "");
Expect(1, 11520, '\P{Is_Block=coptic}', "");
Expect(0, 11520, '\P{^Is_Block=coptic}', "");
Expect(1, 11519, '\p{Is_Block=	-Coptic}', "");
Expect(0, 11519, '\p{^Is_Block=	-Coptic}', "");
Expect(0, 11519, '\P{Is_Block=	-Coptic}', "");
Expect(1, 11519, '\P{^Is_Block=	-Coptic}', "");
Expect(0, 11520, '\p{Is_Block=	-Coptic}', "");
Expect(1, 11520, '\p{^Is_Block=	-Coptic}', "");
Expect(1, 11520, '\P{Is_Block=	-Coptic}', "");
Expect(0, 11520, '\P{^Is_Block=	-Coptic}', "");
Error('\p{Is_Blk= /a/COPTIC}');
Error('\P{Is_Blk= /a/COPTIC}');
Expect(1, 11519, '\p{Is_Blk=coptic}', "");
Expect(0, 11519, '\p{^Is_Blk=coptic}', "");
Expect(0, 11519, '\P{Is_Blk=coptic}', "");
Expect(1, 11519, '\P{^Is_Blk=coptic}', "");
Expect(0, 11520, '\p{Is_Blk=coptic}', "");
Expect(1, 11520, '\p{^Is_Blk=coptic}', "");
Expect(1, 11520, '\P{Is_Blk=coptic}', "");
Expect(0, 11520, '\P{^Is_Blk=coptic}', "");
Expect(1, 11519, '\p{Is_Blk=COPTIC}', "");
Expect(0, 11519, '\p{^Is_Blk=COPTIC}', "");
Expect(0, 11519, '\P{Is_Blk=COPTIC}', "");
Expect(1, 11519, '\P{^Is_Blk=COPTIC}', "");
Expect(0, 11520, '\p{Is_Blk=COPTIC}', "");
Expect(1, 11520, '\p{^Is_Blk=COPTIC}', "");
Expect(1, 11520, '\P{Is_Blk=COPTIC}', "");
Expect(0, 11520, '\P{^Is_Blk=COPTIC}', "");
Error('\p{Block= _COUNTING_ROD_Numerals:=}');
Error('\P{Block= _COUNTING_ROD_Numerals:=}');
Expect(1, 119679, '\p{Block=countingrodnumerals}', "");
Expect(0, 119679, '\p{^Block=countingrodnumerals}', "");
Expect(0, 119679, '\P{Block=countingrodnumerals}', "");
Expect(1, 119679, '\P{^Block=countingrodnumerals}', "");
Expect(0, 119680, '\p{Block=countingrodnumerals}', "");
Expect(1, 119680, '\p{^Block=countingrodnumerals}', "");
Expect(1, 119680, '\P{Block=countingrodnumerals}', "");
Expect(0, 119680, '\P{^Block=countingrodnumerals}', "");
Expect(1, 119679, '\p{Block=	-counting_Rod_numerals}', "");
Expect(0, 119679, '\p{^Block=	-counting_Rod_numerals}', "");
Expect(0, 119679, '\P{Block=	-counting_Rod_numerals}', "");
Expect(1, 119679, '\P{^Block=	-counting_Rod_numerals}', "");
Expect(0, 119680, '\p{Block=	-counting_Rod_numerals}', "");
Expect(1, 119680, '\p{^Block=	-counting_Rod_numerals}', "");
Expect(1, 119680, '\P{Block=	-counting_Rod_numerals}', "");
Expect(0, 119680, '\P{^Block=	-counting_Rod_numerals}', "");
Error('\p{Blk= Counting_rod:=}');
Error('\P{Blk= Counting_rod:=}');
Expect(1, 119679, '\p{Blk=countingrod}', "");
Expect(0, 119679, '\p{^Blk=countingrod}', "");
Expect(0, 119679, '\P{Blk=countingrod}', "");
Expect(1, 119679, '\P{^Blk=countingrod}', "");
Expect(0, 119680, '\p{Blk=countingrod}', "");
Expect(1, 119680, '\p{^Blk=countingrod}', "");
Expect(1, 119680, '\P{Blk=countingrod}', "");
Expect(0, 119680, '\P{^Blk=countingrod}', "");
Expect(1, 119679, '\p{Blk=_	counting_Rod}', "");
Expect(0, 119679, '\p{^Blk=_	counting_Rod}', "");
Expect(0, 119679, '\P{Blk=_	counting_Rod}', "");
Expect(1, 119679, '\P{^Blk=_	counting_Rod}', "");
Expect(0, 119680, '\p{Blk=_	counting_Rod}', "");
Expect(1, 119680, '\p{^Blk=_	counting_Rod}', "");
Expect(1, 119680, '\P{Blk=_	counting_Rod}', "");
Expect(0, 119680, '\P{^Blk=_	counting_Rod}', "");
Error('\p{Is_Block=:=	-counting_Rod_numerals}');
Error('\P{Is_Block=:=	-counting_Rod_numerals}');
Expect(1, 119679, '\p{Is_Block=countingrodnumerals}', "");
Expect(0, 119679, '\p{^Is_Block=countingrodnumerals}', "");
Expect(0, 119679, '\P{Is_Block=countingrodnumerals}', "");
Expect(1, 119679, '\P{^Is_Block=countingrodnumerals}', "");
Expect(0, 119680, '\p{Is_Block=countingrodnumerals}', "");
Expect(1, 119680, '\p{^Is_Block=countingrodnumerals}', "");
Expect(1, 119680, '\P{Is_Block=countingrodnumerals}', "");
Expect(0, 119680, '\P{^Is_Block=countingrodnumerals}', "");
Expect(1, 119679, '\p{Is_Block=	_Counting_Rod_NUMERALS}', "");
Expect(0, 119679, '\p{^Is_Block=	_Counting_Rod_NUMERALS}', "");
Expect(0, 119679, '\P{Is_Block=	_Counting_Rod_NUMERALS}', "");
Expect(1, 119679, '\P{^Is_Block=	_Counting_Rod_NUMERALS}', "");
Expect(0, 119680, '\p{Is_Block=	_Counting_Rod_NUMERALS}', "");
Expect(1, 119680, '\p{^Is_Block=	_Counting_Rod_NUMERALS}', "");
Expect(1, 119680, '\P{Is_Block=	_Counting_Rod_NUMERALS}', "");
Expect(0, 119680, '\P{^Is_Block=	_Counting_Rod_NUMERALS}', "");
Error('\p{Is_Blk=/a/-_Counting_Rod}');
Error('\P{Is_Blk=/a/-_Counting_Rod}');
Expect(1, 119679, '\p{Is_Blk=countingrod}', "");
Expect(0, 119679, '\p{^Is_Blk=countingrod}', "");
Expect(0, 119679, '\P{Is_Blk=countingrod}', "");
Expect(1, 119679, '\P{^Is_Blk=countingrod}', "");
Expect(0, 119680, '\p{Is_Blk=countingrod}', "");
Expect(1, 119680, '\p{^Is_Blk=countingrod}', "");
Expect(1, 119680, '\P{Is_Blk=countingrod}', "");
Expect(0, 119680, '\P{^Is_Blk=countingrod}', "");
Expect(1, 119679, '\p{Is_Blk=	Counting_Rod}', "");
Expect(0, 119679, '\p{^Is_Blk=	Counting_Rod}', "");
Expect(0, 119679, '\P{Is_Blk=	Counting_Rod}', "");
Expect(1, 119679, '\P{^Is_Blk=	Counting_Rod}', "");
Expect(0, 119680, '\p{Is_Blk=	Counting_Rod}', "");
Expect(1, 119680, '\p{^Is_Blk=	Counting_Rod}', "");
Expect(1, 119680, '\P{Is_Blk=	Counting_Rod}', "");
Expect(0, 119680, '\P{^Is_Blk=	Counting_Rod}', "");
Error('\p{Block=Cuneiform:=}');
Error('\P{Block=Cuneiform:=}');
Expect(1, 74751, '\p{Block=cuneiform}', "");
Expect(0, 74751, '\p{^Block=cuneiform}', "");
Expect(0, 74751, '\P{Block=cuneiform}', "");
Expect(1, 74751, '\P{^Block=cuneiform}', "");
Expect(0, 74752, '\p{Block=cuneiform}', "");
Expect(1, 74752, '\p{^Block=cuneiform}', "");
Expect(1, 74752, '\P{Block=cuneiform}', "");
Expect(0, 74752, '\P{^Block=cuneiform}', "");
Expect(1, 74751, '\p{Block= CUNEIFORM}', "");
Expect(0, 74751, '\p{^Block= CUNEIFORM}', "");
Expect(0, 74751, '\P{Block= CUNEIFORM}', "");
Expect(1, 74751, '\P{^Block= CUNEIFORM}', "");
Expect(0, 74752, '\p{Block= CUNEIFORM}', "");
Expect(1, 74752, '\p{^Block= CUNEIFORM}', "");
Expect(1, 74752, '\P{Block= CUNEIFORM}', "");
Expect(0, 74752, '\P{^Block= CUNEIFORM}', "");
Error('\p{Blk=:=	Cuneiform}');
Error('\P{Blk=:=	Cuneiform}');
Expect(1, 74751, '\p{Blk=cuneiform}', "");
Expect(0, 74751, '\p{^Blk=cuneiform}', "");
Expect(0, 74751, '\P{Blk=cuneiform}', "");
Expect(1, 74751, '\P{^Blk=cuneiform}', "");
Expect(0, 74752, '\p{Blk=cuneiform}', "");
Expect(1, 74752, '\p{^Blk=cuneiform}', "");
Expect(1, 74752, '\P{Blk=cuneiform}', "");
Expect(0, 74752, '\P{^Blk=cuneiform}', "");
Expect(1, 74751, '\p{Blk= CUNEIFORM}', "");
Expect(0, 74751, '\p{^Blk= CUNEIFORM}', "");
Expect(0, 74751, '\P{Blk= CUNEIFORM}', "");
Expect(1, 74751, '\P{^Blk= CUNEIFORM}', "");
Expect(0, 74752, '\p{Blk= CUNEIFORM}', "");
Expect(1, 74752, '\p{^Blk= CUNEIFORM}', "");
Expect(1, 74752, '\P{Blk= CUNEIFORM}', "");
Expect(0, 74752, '\P{^Blk= CUNEIFORM}', "");
Error('\p{Is_Block=/a/	Cuneiform}');
Error('\P{Is_Block=/a/	Cuneiform}');
Expect(1, 74751, '\p{Is_Block=cuneiform}', "");
Expect(0, 74751, '\p{^Is_Block=cuneiform}', "");
Expect(0, 74751, '\P{Is_Block=cuneiform}', "");
Expect(1, 74751, '\P{^Is_Block=cuneiform}', "");
Expect(0, 74752, '\p{Is_Block=cuneiform}', "");
Expect(1, 74752, '\p{^Is_Block=cuneiform}', "");
Expect(1, 74752, '\P{Is_Block=cuneiform}', "");
Expect(0, 74752, '\P{^Is_Block=cuneiform}', "");
Expect(1, 74751, '\p{Is_Block=_CUNEIFORM}', "");
Expect(0, 74751, '\p{^Is_Block=_CUNEIFORM}', "");
Expect(0, 74751, '\P{Is_Block=_CUNEIFORM}', "");
Expect(1, 74751, '\P{^Is_Block=_CUNEIFORM}', "");
Expect(0, 74752, '\p{Is_Block=_CUNEIFORM}', "");
Expect(1, 74752, '\p{^Is_Block=_CUNEIFORM}', "");
Expect(1, 74752, '\P{Is_Block=_CUNEIFORM}', "");
Expect(0, 74752, '\P{^Is_Block=_CUNEIFORM}', "");
Error('\p{Is_Blk:    _CUNEIFORM/a/}');
Error('\P{Is_Blk:    _CUNEIFORM/a/}');
Expect(1, 74751, '\p{Is_Blk=cuneiform}', "");
Expect(0, 74751, '\p{^Is_Blk=cuneiform}', "");
Expect(0, 74751, '\P{Is_Blk=cuneiform}', "");
Expect(1, 74751, '\P{^Is_Blk=cuneiform}', "");
Expect(0, 74752, '\p{Is_Blk=cuneiform}', "");
Expect(1, 74752, '\p{^Is_Blk=cuneiform}', "");
Expect(1, 74752, '\P{Is_Blk=cuneiform}', "");
Expect(0, 74752, '\P{^Is_Blk=cuneiform}', "");
Expect(1, 74751, '\p{Is_Blk=_ CUNEIFORM}', "");
Expect(0, 74751, '\p{^Is_Blk=_ CUNEIFORM}', "");
Expect(0, 74751, '\P{Is_Blk=_ CUNEIFORM}', "");
Expect(1, 74751, '\P{^Is_Blk=_ CUNEIFORM}', "");
Expect(0, 74752, '\p{Is_Blk=_ CUNEIFORM}', "");
Expect(1, 74752, '\p{^Is_Blk=_ CUNEIFORM}', "");
Expect(1, 74752, '\P{Is_Blk=_ CUNEIFORM}', "");
Expect(0, 74752, '\P{^Is_Blk=_ CUNEIFORM}', "");
Error('\p{Block=	:=Cuneiform_Numbers_And_Punctuation}');
Error('\P{Block=	:=Cuneiform_Numbers_And_Punctuation}');
Expect(1, 74879, '\p{Block:	cuneiformnumbersandpunctuation}', "");
Expect(0, 74879, '\p{^Block:	cuneiformnumbersandpunctuation}', "");
Expect(0, 74879, '\P{Block:	cuneiformnumbersandpunctuation}', "");
Expect(1, 74879, '\P{^Block:	cuneiformnumbersandpunctuation}', "");
Expect(0, 74880, '\p{Block:	cuneiformnumbersandpunctuation}', "");
Expect(1, 74880, '\p{^Block:	cuneiformnumbersandpunctuation}', "");
Expect(1, 74880, '\P{Block:	cuneiformnumbersandpunctuation}', "");
Expect(0, 74880, '\P{^Block:	cuneiformnumbersandpunctuation}', "");
Expect(1, 74879, '\p{Block=--Cuneiform_NUMBERS_AND_Punctuation}', "");
Expect(0, 74879, '\p{^Block=--Cuneiform_NUMBERS_AND_Punctuation}', "");
Expect(0, 74879, '\P{Block=--Cuneiform_NUMBERS_AND_Punctuation}', "");
Expect(1, 74879, '\P{^Block=--Cuneiform_NUMBERS_AND_Punctuation}', "");
Expect(0, 74880, '\p{Block=--Cuneiform_NUMBERS_AND_Punctuation}', "");
Expect(1, 74880, '\p{^Block=--Cuneiform_NUMBERS_AND_Punctuation}', "");
Expect(1, 74880, '\P{Block=--Cuneiform_NUMBERS_AND_Punctuation}', "");
Expect(0, 74880, '\P{^Block=--Cuneiform_NUMBERS_AND_Punctuation}', "");
Error('\p{Blk=:= _Cuneiform_numbers}');
Error('\P{Blk=:= _Cuneiform_numbers}');
Expect(1, 74879, '\p{Blk=cuneiformnumbers}', "");
Expect(0, 74879, '\p{^Blk=cuneiformnumbers}', "");
Expect(0, 74879, '\P{Blk=cuneiformnumbers}', "");
Expect(1, 74879, '\P{^Blk=cuneiformnumbers}', "");
Expect(0, 74880, '\p{Blk=cuneiformnumbers}', "");
Expect(1, 74880, '\p{^Blk=cuneiformnumbers}', "");
Expect(1, 74880, '\P{Blk=cuneiformnumbers}', "");
Expect(0, 74880, '\P{^Blk=cuneiformnumbers}', "");
Expect(1, 74879, '\p{Blk=-Cuneiform_numbers}', "");
Expect(0, 74879, '\p{^Blk=-Cuneiform_numbers}', "");
Expect(0, 74879, '\P{Blk=-Cuneiform_numbers}', "");
Expect(1, 74879, '\P{^Blk=-Cuneiform_numbers}', "");
Expect(0, 74880, '\p{Blk=-Cuneiform_numbers}', "");
Expect(1, 74880, '\p{^Blk=-Cuneiform_numbers}', "");
Expect(1, 74880, '\P{Blk=-Cuneiform_numbers}', "");
Expect(0, 74880, '\P{^Blk=-Cuneiform_numbers}', "");
Error('\p{Is_Block=:=- Cuneiform_NUMBERS_and_Punctuation}');
Error('\P{Is_Block=:=- Cuneiform_NUMBERS_and_Punctuation}');
Expect(1, 74879, '\p{Is_Block=cuneiformnumbersandpunctuation}', "");
Expect(0, 74879, '\p{^Is_Block=cuneiformnumbersandpunctuation}', "");
Expect(0, 74879, '\P{Is_Block=cuneiformnumbersandpunctuation}', "");
Expect(1, 74879, '\P{^Is_Block=cuneiformnumbersandpunctuation}', "");
Expect(0, 74880, '\p{Is_Block=cuneiformnumbersandpunctuation}', "");
Expect(1, 74880, '\p{^Is_Block=cuneiformnumbersandpunctuation}', "");
Expect(1, 74880, '\P{Is_Block=cuneiformnumbersandpunctuation}', "");
Expect(0, 74880, '\P{^Is_Block=cuneiformnumbersandpunctuation}', "");
Expect(1, 74879, '\p{Is_Block=- Cuneiform_Numbers_AND_PUNCTUATION}', "");
Expect(0, 74879, '\p{^Is_Block=- Cuneiform_Numbers_AND_PUNCTUATION}', "");
Expect(0, 74879, '\P{Is_Block=- Cuneiform_Numbers_AND_PUNCTUATION}', "");
Expect(1, 74879, '\P{^Is_Block=- Cuneiform_Numbers_AND_PUNCTUATION}', "");
Expect(0, 74880, '\p{Is_Block=- Cuneiform_Numbers_AND_PUNCTUATION}', "");
Expect(1, 74880, '\p{^Is_Block=- Cuneiform_Numbers_AND_PUNCTUATION}', "");
Expect(1, 74880, '\P{Is_Block=- Cuneiform_Numbers_AND_PUNCTUATION}', "");
Expect(0, 74880, '\P{^Is_Block=- Cuneiform_Numbers_AND_PUNCTUATION}', "");
Error('\p{Is_Blk=	 cuneiform_Numbers:=}');
Error('\P{Is_Blk=	 cuneiform_Numbers:=}');
Expect(1, 74879, '\p{Is_Blk=cuneiformnumbers}', "");
Expect(0, 74879, '\p{^Is_Blk=cuneiformnumbers}', "");
Expect(0, 74879, '\P{Is_Blk=cuneiformnumbers}', "");
Expect(1, 74879, '\P{^Is_Blk=cuneiformnumbers}', "");
Expect(0, 74880, '\p{Is_Blk=cuneiformnumbers}', "");
Expect(1, 74880, '\p{^Is_Blk=cuneiformnumbers}', "");
Expect(1, 74880, '\P{Is_Blk=cuneiformnumbers}', "");
Expect(0, 74880, '\P{^Is_Blk=cuneiformnumbers}', "");
Expect(1, 74879, '\p{Is_Blk= _cuneiform_Numbers}', "");
Expect(0, 74879, '\p{^Is_Blk= _cuneiform_Numbers}', "");
Expect(0, 74879, '\P{Is_Blk= _cuneiform_Numbers}', "");
Expect(1, 74879, '\P{^Is_Blk= _cuneiform_Numbers}', "");
Expect(0, 74880, '\p{Is_Blk= _cuneiform_Numbers}', "");
Expect(1, 74880, '\p{^Is_Blk= _cuneiform_Numbers}', "");
Expect(1, 74880, '\P{Is_Blk= _cuneiform_Numbers}', "");
Expect(0, 74880, '\P{^Is_Blk= _cuneiform_Numbers}', "");
Error('\p{Block= Currency_Symbols:=}');
Error('\P{Block= Currency_Symbols:=}');
Expect(1, 8399, '\p{Block=currencysymbols}', "");
Expect(0, 8399, '\p{^Block=currencysymbols}', "");
Expect(0, 8399, '\P{Block=currencysymbols}', "");
Expect(1, 8399, '\P{^Block=currencysymbols}', "");
Expect(0, 8400, '\p{Block=currencysymbols}', "");
Expect(1, 8400, '\p{^Block=currencysymbols}', "");
Expect(1, 8400, '\P{Block=currencysymbols}', "");
Expect(0, 8400, '\P{^Block=currencysymbols}', "");
Expect(1, 8399, '\p{Block:__Currency_Symbols}', "");
Expect(0, 8399, '\p{^Block:__Currency_Symbols}', "");
Expect(0, 8399, '\P{Block:__Currency_Symbols}', "");
Expect(1, 8399, '\P{^Block:__Currency_Symbols}', "");
Expect(0, 8400, '\p{Block:__Currency_Symbols}', "");
Expect(1, 8400, '\p{^Block:__Currency_Symbols}', "");
Expect(1, 8400, '\P{Block:__Currency_Symbols}', "");
Expect(0, 8400, '\P{^Block:__Currency_Symbols}', "");
Error('\p{Blk=_currency_Symbols:=}');
Error('\P{Blk=_currency_Symbols:=}');
Expect(1, 8399, '\p{Blk=currencysymbols}', "");
Expect(0, 8399, '\p{^Blk=currencysymbols}', "");
Expect(0, 8399, '\P{Blk=currencysymbols}', "");
Expect(1, 8399, '\P{^Blk=currencysymbols}', "");
Expect(0, 8400, '\p{Blk=currencysymbols}', "");
Expect(1, 8400, '\p{^Blk=currencysymbols}', "");
Expect(1, 8400, '\P{Blk=currencysymbols}', "");
Expect(0, 8400, '\P{^Blk=currencysymbols}', "");
Expect(1, 8399, '\p{Blk=-Currency_symbols}', "");
Expect(0, 8399, '\p{^Blk=-Currency_symbols}', "");
Expect(0, 8399, '\P{Blk=-Currency_symbols}', "");
Expect(1, 8399, '\P{^Blk=-Currency_symbols}', "");
Expect(0, 8400, '\p{Blk=-Currency_symbols}', "");
Expect(1, 8400, '\p{^Blk=-Currency_symbols}', "");
Expect(1, 8400, '\P{Blk=-Currency_symbols}', "");
Expect(0, 8400, '\P{^Blk=-Currency_symbols}', "");
Error('\p{Is_Block=_/a/CURRENCY_symbols}');
Error('\P{Is_Block=_/a/CURRENCY_symbols}');
Expect(1, 8399, '\p{Is_Block=currencysymbols}', "");
Expect(0, 8399, '\p{^Is_Block=currencysymbols}', "");
Expect(0, 8399, '\P{Is_Block=currencysymbols}', "");
Expect(1, 8399, '\P{^Is_Block=currencysymbols}', "");
Expect(0, 8400, '\p{Is_Block=currencysymbols}', "");
Expect(1, 8400, '\p{^Is_Block=currencysymbols}', "");
Expect(1, 8400, '\P{Is_Block=currencysymbols}', "");
Expect(0, 8400, '\P{^Is_Block=currencysymbols}', "");
Expect(1, 8399, '\p{Is_Block: - currency_symbols}', "");
Expect(0, 8399, '\p{^Is_Block: - currency_symbols}', "");
Expect(0, 8399, '\P{Is_Block: - currency_symbols}', "");
Expect(1, 8399, '\P{^Is_Block: - currency_symbols}', "");
Expect(0, 8400, '\p{Is_Block: - currency_symbols}', "");
Expect(1, 8400, '\p{^Is_Block: - currency_symbols}', "");
Expect(1, 8400, '\P{Is_Block: - currency_symbols}', "");
Expect(0, 8400, '\P{^Is_Block: - currency_symbols}', "");
Error('\p{Is_Blk=/a/-	Currency_Symbols}');
Error('\P{Is_Blk=/a/-	Currency_Symbols}');
Expect(1, 8399, '\p{Is_Blk=currencysymbols}', "");
Expect(0, 8399, '\p{^Is_Blk=currencysymbols}', "");
Expect(0, 8399, '\P{Is_Blk=currencysymbols}', "");
Expect(1, 8399, '\P{^Is_Blk=currencysymbols}', "");
Expect(0, 8400, '\p{Is_Blk=currencysymbols}', "");
Expect(1, 8400, '\p{^Is_Blk=currencysymbols}', "");
Expect(1, 8400, '\P{Is_Blk=currencysymbols}', "");
Expect(0, 8400, '\P{^Is_Blk=currencysymbols}', "");
Expect(1, 8399, '\p{Is_Blk=_	currency_symbols}', "");
Expect(0, 8399, '\p{^Is_Blk=_	currency_symbols}', "");
Expect(0, 8399, '\P{Is_Blk=_	currency_symbols}', "");
Expect(1, 8399, '\P{^Is_Blk=_	currency_symbols}', "");
Expect(0, 8400, '\p{Is_Blk=_	currency_symbols}', "");
Expect(1, 8400, '\p{^Is_Blk=_	currency_symbols}', "");
Expect(1, 8400, '\P{Is_Blk=_	currency_symbols}', "");
Expect(0, 8400, '\P{^Is_Blk=_	currency_symbols}', "");
Error('\p{Block=:=-	CYPRIOT_Syllabary}');
Error('\P{Block=:=-	CYPRIOT_Syllabary}');
Expect(1, 67647, '\p{Block=cypriotsyllabary}', "");
Expect(0, 67647, '\p{^Block=cypriotsyllabary}', "");
Expect(0, 67647, '\P{Block=cypriotsyllabary}', "");
Expect(1, 67647, '\P{^Block=cypriotsyllabary}', "");
Expect(0, 67648, '\p{Block=cypriotsyllabary}', "");
Expect(1, 67648, '\p{^Block=cypriotsyllabary}', "");
Expect(1, 67648, '\P{Block=cypriotsyllabary}', "");
Expect(0, 67648, '\P{^Block=cypriotsyllabary}', "");
Expect(1, 67647, '\p{Block=_Cypriot_syllabary}', "");
Expect(0, 67647, '\p{^Block=_Cypriot_syllabary}', "");
Expect(0, 67647, '\P{Block=_Cypriot_syllabary}', "");
Expect(1, 67647, '\P{^Block=_Cypriot_syllabary}', "");
Expect(0, 67648, '\p{Block=_Cypriot_syllabary}', "");
Expect(1, 67648, '\p{^Block=_Cypriot_syllabary}', "");
Expect(1, 67648, '\P{Block=_Cypriot_syllabary}', "");
Expect(0, 67648, '\P{^Block=_Cypriot_syllabary}', "");
Error('\p{Blk=/a/cypriot_syllabary}');
Error('\P{Blk=/a/cypriot_syllabary}');
Expect(1, 67647, '\p{Blk=cypriotsyllabary}', "");
Expect(0, 67647, '\p{^Blk=cypriotsyllabary}', "");
Expect(0, 67647, '\P{Blk=cypriotsyllabary}', "");
Expect(1, 67647, '\P{^Blk=cypriotsyllabary}', "");
Expect(0, 67648, '\p{Blk=cypriotsyllabary}', "");
Expect(1, 67648, '\p{^Blk=cypriotsyllabary}', "");
Expect(1, 67648, '\P{Blk=cypriotsyllabary}', "");
Expect(0, 67648, '\P{^Blk=cypriotsyllabary}', "");
Expect(1, 67647, '\p{Blk=_	Cypriot_Syllabary}', "");
Expect(0, 67647, '\p{^Blk=_	Cypriot_Syllabary}', "");
Expect(0, 67647, '\P{Blk=_	Cypriot_Syllabary}', "");
Expect(1, 67647, '\P{^Blk=_	Cypriot_Syllabary}', "");
Expect(0, 67648, '\p{Blk=_	Cypriot_Syllabary}', "");
Expect(1, 67648, '\p{^Blk=_	Cypriot_Syllabary}', "");
Expect(1, 67648, '\P{Blk=_	Cypriot_Syllabary}', "");
Expect(0, 67648, '\P{^Blk=_	Cypriot_Syllabary}', "");
Error('\p{Is_Block= -Cypriot_SYLLABARY/a/}');
Error('\P{Is_Block= -Cypriot_SYLLABARY/a/}');
Expect(1, 67647, '\p{Is_Block=cypriotsyllabary}', "");
Expect(0, 67647, '\p{^Is_Block=cypriotsyllabary}', "");
Expect(0, 67647, '\P{Is_Block=cypriotsyllabary}', "");
Expect(1, 67647, '\P{^Is_Block=cypriotsyllabary}', "");
Expect(0, 67648, '\p{Is_Block=cypriotsyllabary}', "");
Expect(1, 67648, '\p{^Is_Block=cypriotsyllabary}', "");
Expect(1, 67648, '\P{Is_Block=cypriotsyllabary}', "");
Expect(0, 67648, '\P{^Is_Block=cypriotsyllabary}', "");
Expect(1, 67647, '\p{Is_Block= Cypriot_SYLLABARY}', "");
Expect(0, 67647, '\p{^Is_Block= Cypriot_SYLLABARY}', "");
Expect(0, 67647, '\P{Is_Block= Cypriot_SYLLABARY}', "");
Expect(1, 67647, '\P{^Is_Block= Cypriot_SYLLABARY}', "");
Expect(0, 67648, '\p{Is_Block= Cypriot_SYLLABARY}', "");
Expect(1, 67648, '\p{^Is_Block= Cypriot_SYLLABARY}', "");
Expect(1, 67648, '\P{Is_Block= Cypriot_SYLLABARY}', "");
Expect(0, 67648, '\P{^Is_Block= Cypriot_SYLLABARY}', "");
Error('\p{Is_Blk: :=	-CYPRIOT_syllabary}');
Error('\P{Is_Blk: :=	-CYPRIOT_syllabary}');
Expect(1, 67647, '\p{Is_Blk:   cypriotsyllabary}', "");
Expect(0, 67647, '\p{^Is_Blk:   cypriotsyllabary}', "");
Expect(0, 67647, '\P{Is_Blk:   cypriotsyllabary}', "");
Expect(1, 67647, '\P{^Is_Blk:   cypriotsyllabary}', "");
Expect(0, 67648, '\p{Is_Blk:   cypriotsyllabary}', "");
Expect(1, 67648, '\p{^Is_Blk:   cypriotsyllabary}', "");
Expect(1, 67648, '\P{Is_Blk:   cypriotsyllabary}', "");
Expect(0, 67648, '\P{^Is_Blk:   cypriotsyllabary}', "");
Expect(1, 67647, '\p{Is_Blk=_cypriot_Syllabary}', "");
Expect(0, 67647, '\p{^Is_Blk=_cypriot_Syllabary}', "");
Expect(0, 67647, '\P{Is_Blk=_cypriot_Syllabary}', "");
Expect(1, 67647, '\P{^Is_Blk=_cypriot_Syllabary}', "");
Expect(0, 67648, '\p{Is_Blk=_cypriot_Syllabary}', "");
Expect(1, 67648, '\p{^Is_Blk=_cypriot_Syllabary}', "");
Expect(1, 67648, '\P{Is_Blk=_cypriot_Syllabary}', "");
Expect(0, 67648, '\P{^Is_Blk=_cypriot_Syllabary}', "");
Error('\p{Block= :=Cyrillic}');
Error('\P{Block= :=Cyrillic}');
Expect(1, 1279, '\p{Block=cyrillic}', "");
Expect(0, 1279, '\p{^Block=cyrillic}', "");
Expect(0, 1279, '\P{Block=cyrillic}', "");
Expect(1, 1279, '\P{^Block=cyrillic}', "");
Expect(0, 1280, '\p{Block=cyrillic}', "");
Expect(1, 1280, '\p{^Block=cyrillic}', "");
Expect(1, 1280, '\P{Block=cyrillic}', "");
Expect(0, 1280, '\P{^Block=cyrillic}', "");
Expect(1, 1279, '\p{Block=	Cyrillic}', "");
Expect(0, 1279, '\p{^Block=	Cyrillic}', "");
Expect(0, 1279, '\P{Block=	Cyrillic}', "");
Expect(1, 1279, '\P{^Block=	Cyrillic}', "");
Expect(0, 1280, '\p{Block=	Cyrillic}', "");
Expect(1, 1280, '\p{^Block=	Cyrillic}', "");
Expect(1, 1280, '\P{Block=	Cyrillic}', "");
Expect(0, 1280, '\P{^Block=	Cyrillic}', "");
Error('\p{Blk=- Cyrillic/a/}');
Error('\P{Blk=- Cyrillic/a/}');
Expect(1, 1279, '\p{Blk:	cyrillic}', "");
Expect(0, 1279, '\p{^Blk:	cyrillic}', "");
Expect(0, 1279, '\P{Blk:	cyrillic}', "");
Expect(1, 1279, '\P{^Blk:	cyrillic}', "");
Expect(0, 1280, '\p{Blk:	cyrillic}', "");
Expect(1, 1280, '\p{^Blk:	cyrillic}', "");
Expect(1, 1280, '\P{Blk:	cyrillic}', "");
Expect(0, 1280, '\P{^Blk:	cyrillic}', "");
Expect(1, 1279, '\p{Blk=	_Cyrillic}', "");
Expect(0, 1279, '\p{^Blk=	_Cyrillic}', "");
Expect(0, 1279, '\P{Blk=	_Cyrillic}', "");
Expect(1, 1279, '\P{^Blk=	_Cyrillic}', "");
Expect(0, 1280, '\p{Blk=	_Cyrillic}', "");
Expect(1, 1280, '\p{^Blk=	_Cyrillic}', "");
Expect(1, 1280, '\P{Blk=	_Cyrillic}', "");
Expect(0, 1280, '\P{^Blk=	_Cyrillic}', "");
Error('\p{Is_Block=:=	_CYRILLIC}');
Error('\P{Is_Block=:=	_CYRILLIC}');
Expect(1, 1279, '\p{Is_Block=cyrillic}', "");
Expect(0, 1279, '\p{^Is_Block=cyrillic}', "");
Expect(0, 1279, '\P{Is_Block=cyrillic}', "");
Expect(1, 1279, '\P{^Is_Block=cyrillic}', "");
Expect(0, 1280, '\p{Is_Block=cyrillic}', "");
Expect(1, 1280, '\p{^Is_Block=cyrillic}', "");
Expect(1, 1280, '\P{Is_Block=cyrillic}', "");
Expect(0, 1280, '\P{^Is_Block=cyrillic}', "");
Expect(1, 1279, '\p{Is_Block:	Cyrillic}', "");
Expect(0, 1279, '\p{^Is_Block:	Cyrillic}', "");
Expect(0, 1279, '\P{Is_Block:	Cyrillic}', "");
Expect(1, 1279, '\P{^Is_Block:	Cyrillic}', "");
Expect(0, 1280, '\p{Is_Block:	Cyrillic}', "");
Expect(1, 1280, '\p{^Is_Block:	Cyrillic}', "");
Expect(1, 1280, '\P{Is_Block:	Cyrillic}', "");
Expect(0, 1280, '\P{^Is_Block:	Cyrillic}', "");
Error('\p{Is_Blk:	/a/cyrillic}');
Error('\P{Is_Blk:	/a/cyrillic}');
Expect(1, 1279, '\p{Is_Blk=cyrillic}', "");
Expect(0, 1279, '\p{^Is_Blk=cyrillic}', "");
Expect(0, 1279, '\P{Is_Blk=cyrillic}', "");
Expect(1, 1279, '\P{^Is_Blk=cyrillic}', "");
Expect(0, 1280, '\p{Is_Blk=cyrillic}', "");
Expect(1, 1280, '\p{^Is_Blk=cyrillic}', "");
Expect(1, 1280, '\P{Is_Blk=cyrillic}', "");
Expect(0, 1280, '\P{^Is_Blk=cyrillic}', "");
Expect(1, 1279, '\p{Is_Blk= CYRILLIC}', "");
Expect(0, 1279, '\p{^Is_Blk= CYRILLIC}', "");
Expect(0, 1279, '\P{Is_Blk= CYRILLIC}', "");
Expect(1, 1279, '\P{^Is_Blk= CYRILLIC}', "");
Expect(0, 1280, '\p{Is_Blk= CYRILLIC}', "");
Expect(1, 1280, '\p{^Is_Blk= CYRILLIC}', "");
Expect(1, 1280, '\P{Is_Blk= CYRILLIC}', "");
Expect(0, 1280, '\P{^Is_Blk= CYRILLIC}', "");
Error('\p{Block:		Cyrillic_EXTENDED_A:=}');
Error('\P{Block:		Cyrillic_EXTENDED_A:=}');
Expect(1, 11775, '\p{Block:cyrillicextendeda}', "");
Expect(0, 11775, '\p{^Block:cyrillicextendeda}', "");
Expect(0, 11775, '\P{Block:cyrillicextendeda}', "");
Expect(1, 11775, '\P{^Block:cyrillicextendeda}', "");
Expect(0, 11776, '\p{Block:cyrillicextendeda}', "");
Expect(1, 11776, '\p{^Block:cyrillicextendeda}', "");
Expect(1, 11776, '\P{Block:cyrillicextendeda}', "");
Expect(0, 11776, '\P{^Block:cyrillicextendeda}', "");
Expect(1, 11775, '\p{Block=_	cyrillic_Extended_A}', "");
Expect(0, 11775, '\p{^Block=_	cyrillic_Extended_A}', "");
Expect(0, 11775, '\P{Block=_	cyrillic_Extended_A}', "");
Expect(1, 11775, '\P{^Block=_	cyrillic_Extended_A}', "");
Expect(0, 11776, '\p{Block=_	cyrillic_Extended_A}', "");
Expect(1, 11776, '\p{^Block=_	cyrillic_Extended_A}', "");
Expect(1, 11776, '\P{Block=_	cyrillic_Extended_A}', "");
Expect(0, 11776, '\P{^Block=_	cyrillic_Extended_A}', "");
Error('\p{Blk=-_Cyrillic_ext_A/a/}');
Error('\P{Blk=-_Cyrillic_ext_A/a/}');
Expect(1, 11775, '\p{Blk:	cyrillicexta}', "");
Expect(0, 11775, '\p{^Blk:	cyrillicexta}', "");
Expect(0, 11775, '\P{Blk:	cyrillicexta}', "");
Expect(1, 11775, '\P{^Blk:	cyrillicexta}', "");
Expect(0, 11776, '\p{Blk:	cyrillicexta}', "");
Expect(1, 11776, '\p{^Blk:	cyrillicexta}', "");
Expect(1, 11776, '\P{Blk:	cyrillicexta}', "");
Expect(0, 11776, '\P{^Blk:	cyrillicexta}', "");
Expect(1, 11775, '\p{Blk:  -Cyrillic_Ext_a}', "");
Expect(0, 11775, '\p{^Blk:  -Cyrillic_Ext_a}', "");
Expect(0, 11775, '\P{Blk:  -Cyrillic_Ext_a}', "");
Expect(1, 11775, '\P{^Blk:  -Cyrillic_Ext_a}', "");
Expect(0, 11776, '\p{Blk:  -Cyrillic_Ext_a}', "");
Expect(1, 11776, '\p{^Blk:  -Cyrillic_Ext_a}', "");
Expect(1, 11776, '\P{Blk:  -Cyrillic_Ext_a}', "");
Expect(0, 11776, '\P{^Blk:  -Cyrillic_Ext_a}', "");
Error('\p{Is_Block=_/a/Cyrillic_extended_A}');
Error('\P{Is_Block=_/a/Cyrillic_extended_A}');
Expect(1, 11775, '\p{Is_Block:   cyrillicextendeda}', "");
Expect(0, 11775, '\p{^Is_Block:   cyrillicextendeda}', "");
Expect(0, 11775, '\P{Is_Block:   cyrillicextendeda}', "");
Expect(1, 11775, '\P{^Is_Block:   cyrillicextendeda}', "");
Expect(0, 11776, '\p{Is_Block:   cyrillicextendeda}', "");
Expect(1, 11776, '\p{^Is_Block:   cyrillicextendeda}', "");
Expect(1, 11776, '\P{Is_Block:   cyrillicextendeda}', "");
Expect(0, 11776, '\P{^Is_Block:   cyrillicextendeda}', "");
Expect(1, 11775, '\p{Is_Block=_-Cyrillic_Extended_A}', "");
Expect(0, 11775, '\p{^Is_Block=_-Cyrillic_Extended_A}', "");
Expect(0, 11775, '\P{Is_Block=_-Cyrillic_Extended_A}', "");
Expect(1, 11775, '\P{^Is_Block=_-Cyrillic_Extended_A}', "");
Expect(0, 11776, '\p{Is_Block=_-Cyrillic_Extended_A}', "");
Expect(1, 11776, '\p{^Is_Block=_-Cyrillic_Extended_A}', "");
Expect(1, 11776, '\P{Is_Block=_-Cyrillic_Extended_A}', "");
Expect(0, 11776, '\P{^Is_Block=_-Cyrillic_Extended_A}', "");
Error('\p{Is_Blk: :=  Cyrillic_ext_a}');
Error('\P{Is_Blk: :=  Cyrillic_ext_a}');
Expect(1, 11775, '\p{Is_Blk=cyrillicexta}', "");
Expect(0, 11775, '\p{^Is_Blk=cyrillicexta}', "");
Expect(0, 11775, '\P{Is_Blk=cyrillicexta}', "");
Expect(1, 11775, '\P{^Is_Blk=cyrillicexta}', "");
Expect(0, 11776, '\p{Is_Blk=cyrillicexta}', "");
Expect(1, 11776, '\p{^Is_Blk=cyrillicexta}', "");
Expect(1, 11776, '\P{Is_Blk=cyrillicexta}', "");
Expect(0, 11776, '\P{^Is_Blk=cyrillicexta}', "");
Expect(1, 11775, '\p{Is_Blk=CYRILLIC_Ext_A}', "");
Expect(0, 11775, '\p{^Is_Blk=CYRILLIC_Ext_A}', "");
Expect(0, 11775, '\P{Is_Blk=CYRILLIC_Ext_A}', "");
Expect(1, 11775, '\P{^Is_Blk=CYRILLIC_Ext_A}', "");
Expect(0, 11776, '\p{Is_Blk=CYRILLIC_Ext_A}', "");
Expect(1, 11776, '\p{^Is_Blk=CYRILLIC_Ext_A}', "");
Expect(1, 11776, '\P{Is_Blk=CYRILLIC_Ext_A}', "");
Expect(0, 11776, '\P{^Is_Blk=CYRILLIC_Ext_A}', "");
Error('\p{Block= /a/Cyrillic_Extended_B}');
Error('\P{Block= /a/Cyrillic_Extended_B}');
Expect(1, 42655, '\p{Block=cyrillicextendedb}', "");
Expect(0, 42655, '\p{^Block=cyrillicextendedb}', "");
Expect(0, 42655, '\P{Block=cyrillicextendedb}', "");
Expect(1, 42655, '\P{^Block=cyrillicextendedb}', "");
Expect(0, 42656, '\p{Block=cyrillicextendedb}', "");
Expect(1, 42656, '\p{^Block=cyrillicextendedb}', "");
Expect(1, 42656, '\P{Block=cyrillicextendedb}', "");
Expect(0, 42656, '\P{^Block=cyrillicextendedb}', "");
Expect(1, 42655, '\p{Block= Cyrillic_Extended_b}', "");
Expect(0, 42655, '\p{^Block= Cyrillic_Extended_b}', "");
Expect(0, 42655, '\P{Block= Cyrillic_Extended_b}', "");
Expect(1, 42655, '\P{^Block= Cyrillic_Extended_b}', "");
Expect(0, 42656, '\p{Block= Cyrillic_Extended_b}', "");
Expect(1, 42656, '\p{^Block= Cyrillic_Extended_b}', "");
Expect(1, 42656, '\P{Block= Cyrillic_Extended_b}', "");
Expect(0, 42656, '\P{^Block= Cyrillic_Extended_b}', "");
Error('\p{Blk=:=Cyrillic_Ext_B}');
Error('\P{Blk=:=Cyrillic_Ext_B}');
Expect(1, 42655, '\p{Blk=cyrillicextb}', "");
Expect(0, 42655, '\p{^Blk=cyrillicextb}', "");
Expect(0, 42655, '\P{Blk=cyrillicextb}', "");
Expect(1, 42655, '\P{^Blk=cyrillicextb}', "");
Expect(0, 42656, '\p{Blk=cyrillicextb}', "");
Expect(1, 42656, '\p{^Blk=cyrillicextb}', "");
Expect(1, 42656, '\P{Blk=cyrillicextb}', "");
Expect(0, 42656, '\P{^Blk=cyrillicextb}', "");
Expect(1, 42655, '\p{Blk=_-Cyrillic_ext_b}', "");
Expect(0, 42655, '\p{^Blk=_-Cyrillic_ext_b}', "");
Expect(0, 42655, '\P{Blk=_-Cyrillic_ext_b}', "");
Expect(1, 42655, '\P{^Blk=_-Cyrillic_ext_b}', "");
Expect(0, 42656, '\p{Blk=_-Cyrillic_ext_b}', "");
Expect(1, 42656, '\p{^Blk=_-Cyrillic_ext_b}', "");
Expect(1, 42656, '\P{Blk=_-Cyrillic_ext_b}', "");
Expect(0, 42656, '\P{^Blk=_-Cyrillic_ext_b}', "");
Error('\p{Is_Block=_CYRILLIC_EXTENDED_B/a/}');
Error('\P{Is_Block=_CYRILLIC_EXTENDED_B/a/}');
Expect(1, 42655, '\p{Is_Block=cyrillicextendedb}', "");
Expect(0, 42655, '\p{^Is_Block=cyrillicextendedb}', "");
Expect(0, 42655, '\P{Is_Block=cyrillicextendedb}', "");
Expect(1, 42655, '\P{^Is_Block=cyrillicextendedb}', "");
Expect(0, 42656, '\p{Is_Block=cyrillicextendedb}', "");
Expect(1, 42656, '\p{^Is_Block=cyrillicextendedb}', "");
Expect(1, 42656, '\P{Is_Block=cyrillicextendedb}', "");
Expect(0, 42656, '\P{^Is_Block=cyrillicextendedb}', "");
Expect(1, 42655, '\p{Is_Block=-CYRILLIC_extended_B}', "");
Expect(0, 42655, '\p{^Is_Block=-CYRILLIC_extended_B}', "");
Expect(0, 42655, '\P{Is_Block=-CYRILLIC_extended_B}', "");
Expect(1, 42655, '\P{^Is_Block=-CYRILLIC_extended_B}', "");
Expect(0, 42656, '\p{Is_Block=-CYRILLIC_extended_B}', "");
Expect(1, 42656, '\p{^Is_Block=-CYRILLIC_extended_B}', "");
Expect(1, 42656, '\P{Is_Block=-CYRILLIC_extended_B}', "");
Expect(0, 42656, '\P{^Is_Block=-CYRILLIC_extended_B}', "");
Error('\p{Is_Blk=	:=cyrillic_ext_B}');
Error('\P{Is_Blk=	:=cyrillic_ext_B}');
Expect(1, 42655, '\p{Is_Blk=cyrillicextb}', "");
Expect(0, 42655, '\p{^Is_Blk=cyrillicextb}', "");
Expect(0, 42655, '\P{Is_Blk=cyrillicextb}', "");
Expect(1, 42655, '\P{^Is_Blk=cyrillicextb}', "");
Expect(0, 42656, '\p{Is_Blk=cyrillicextb}', "");
Expect(1, 42656, '\p{^Is_Blk=cyrillicextb}', "");
Expect(1, 42656, '\P{Is_Blk=cyrillicextb}', "");
Expect(0, 42656, '\P{^Is_Blk=cyrillicextb}', "");
Expect(1, 42655, '\p{Is_Blk=__Cyrillic_EXT_B}', "");
Expect(0, 42655, '\p{^Is_Blk=__Cyrillic_EXT_B}', "");
Expect(0, 42655, '\P{Is_Blk=__Cyrillic_EXT_B}', "");
Expect(1, 42655, '\P{^Is_Blk=__Cyrillic_EXT_B}', "");
Expect(0, 42656, '\p{Is_Blk=__Cyrillic_EXT_B}', "");
Expect(1, 42656, '\p{^Is_Blk=__Cyrillic_EXT_B}', "");
Expect(1, 42656, '\P{Is_Blk=__Cyrillic_EXT_B}', "");
Expect(0, 42656, '\P{^Is_Blk=__Cyrillic_EXT_B}', "");
Error('\p{Block=:=	-cyrillic_supplement}');
Error('\P{Block=:=	-cyrillic_supplement}');
Expect(1, 1327, '\p{Block: cyrillicsupplement}', "");
Expect(0, 1327, '\p{^Block: cyrillicsupplement}', "");
Expect(0, 1327, '\P{Block: cyrillicsupplement}', "");
Expect(1, 1327, '\P{^Block: cyrillicsupplement}', "");
Expect(0, 1328, '\p{Block: cyrillicsupplement}', "");
Expect(1, 1328, '\p{^Block: cyrillicsupplement}', "");
Expect(1, 1328, '\P{Block: cyrillicsupplement}', "");
Expect(0, 1328, '\P{^Block: cyrillicsupplement}', "");
Expect(1, 1327, '\p{Block=	CYRILLIC_Supplement}', "");
Expect(0, 1327, '\p{^Block=	CYRILLIC_Supplement}', "");
Expect(0, 1327, '\P{Block=	CYRILLIC_Supplement}', "");
Expect(1, 1327, '\P{^Block=	CYRILLIC_Supplement}', "");
Expect(0, 1328, '\p{Block=	CYRILLIC_Supplement}', "");
Expect(1, 1328, '\p{^Block=	CYRILLIC_Supplement}', "");
Expect(1, 1328, '\P{Block=	CYRILLIC_Supplement}', "");
Expect(0, 1328, '\P{^Block=	CYRILLIC_Supplement}', "");
Error('\p{Blk:	 Cyrillic_SUP:=}');
Error('\P{Blk:	 Cyrillic_SUP:=}');
Expect(1, 1327, '\p{Blk:cyrillicsup}', "");
Expect(0, 1327, '\p{^Blk:cyrillicsup}', "");
Expect(0, 1327, '\P{Blk:cyrillicsup}', "");
Expect(1, 1327, '\P{^Blk:cyrillicsup}', "");
Expect(0, 1328, '\p{Blk:cyrillicsup}', "");
Expect(1, 1328, '\p{^Blk:cyrillicsup}', "");
Expect(1, 1328, '\P{Blk:cyrillicsup}', "");
Expect(0, 1328, '\P{^Blk:cyrillicsup}', "");
Expect(1, 1327, '\p{Blk= _Cyrillic_Sup}', "");
Expect(0, 1327, '\p{^Blk= _Cyrillic_Sup}', "");
Expect(0, 1327, '\P{Blk= _Cyrillic_Sup}', "");
Expect(1, 1327, '\P{^Blk= _Cyrillic_Sup}', "");
Expect(0, 1328, '\p{Blk= _Cyrillic_Sup}', "");
Expect(1, 1328, '\p{^Blk= _Cyrillic_Sup}', "");
Expect(1, 1328, '\P{Blk= _Cyrillic_Sup}', "");
Expect(0, 1328, '\P{^Blk= _Cyrillic_Sup}', "");
Error('\p{Is_Block=:=-	Cyrillic_Supplementary}');
Error('\P{Is_Block=:=-	Cyrillic_Supplementary}');
Expect(1, 1327, '\p{Is_Block=cyrillicsupplementary}', "");
Expect(0, 1327, '\p{^Is_Block=cyrillicsupplementary}', "");
Expect(0, 1327, '\P{Is_Block=cyrillicsupplementary}', "");
Expect(1, 1327, '\P{^Is_Block=cyrillicsupplementary}', "");
Expect(0, 1328, '\p{Is_Block=cyrillicsupplementary}', "");
Expect(1, 1328, '\p{^Is_Block=cyrillicsupplementary}', "");
Expect(1, 1328, '\P{Is_Block=cyrillicsupplementary}', "");
Expect(0, 1328, '\P{^Is_Block=cyrillicsupplementary}', "");
Expect(1, 1327, '\p{Is_Block=_Cyrillic_supplementary}', "");
Expect(0, 1327, '\p{^Is_Block=_Cyrillic_supplementary}', "");
Expect(0, 1327, '\P{Is_Block=_Cyrillic_supplementary}', "");
Expect(1, 1327, '\P{^Is_Block=_Cyrillic_supplementary}', "");
Expect(0, 1328, '\p{Is_Block=_Cyrillic_supplementary}', "");
Expect(1, 1328, '\p{^Is_Block=_Cyrillic_supplementary}', "");
Expect(1, 1328, '\P{Is_Block=_Cyrillic_supplementary}', "");
Expect(0, 1328, '\P{^Is_Block=_Cyrillic_supplementary}', "");
Error('\p{Is_Blk=_/a/Cyrillic_supplement}');
Error('\P{Is_Blk=_/a/Cyrillic_supplement}');
Expect(1, 1327, '\p{Is_Blk=cyrillicsupplement}', "");
Expect(0, 1327, '\p{^Is_Blk=cyrillicsupplement}', "");
Expect(0, 1327, '\P{Is_Blk=cyrillicsupplement}', "");
Expect(1, 1327, '\P{^Is_Blk=cyrillicsupplement}', "");
Expect(0, 1328, '\p{Is_Blk=cyrillicsupplement}', "");
Expect(1, 1328, '\p{^Is_Blk=cyrillicsupplement}', "");
Expect(1, 1328, '\P{Is_Blk=cyrillicsupplement}', "");
Expect(0, 1328, '\P{^Is_Blk=cyrillicsupplement}', "");
Expect(1, 1327, '\p{Is_Blk=_Cyrillic_supplement}', "");
Expect(0, 1327, '\p{^Is_Blk=_Cyrillic_supplement}', "");
Expect(0, 1327, '\P{Is_Blk=_Cyrillic_supplement}', "");
Expect(1, 1327, '\P{^Is_Blk=_Cyrillic_supplement}', "");
Expect(0, 1328, '\p{Is_Blk=_Cyrillic_supplement}', "");
Expect(1, 1328, '\p{^Is_Blk=_Cyrillic_supplement}', "");
Expect(1, 1328, '\P{Is_Blk=_Cyrillic_supplement}', "");
Expect(0, 1328, '\P{^Is_Blk=_Cyrillic_supplement}', "");
Error('\p{Block:   :=--Deseret}');
Error('\P{Block:   :=--Deseret}');
Expect(1, 66639, '\p{Block=deseret}', "");
Expect(0, 66639, '\p{^Block=deseret}', "");
Expect(0, 66639, '\P{Block=deseret}', "");
Expect(1, 66639, '\P{^Block=deseret}', "");
Expect(0, 66640, '\p{Block=deseret}', "");
Expect(1, 66640, '\p{^Block=deseret}', "");
Expect(1, 66640, '\P{Block=deseret}', "");
Expect(0, 66640, '\P{^Block=deseret}', "");
Expect(1, 66639, '\p{Block= DESERET}', "");
Expect(0, 66639, '\p{^Block= DESERET}', "");
Expect(0, 66639, '\P{Block= DESERET}', "");
Expect(1, 66639, '\P{^Block= DESERET}', "");
Expect(0, 66640, '\p{Block= DESERET}', "");
Expect(1, 66640, '\p{^Block= DESERET}', "");
Expect(1, 66640, '\P{Block= DESERET}', "");
Expect(0, 66640, '\P{^Block= DESERET}', "");
Error('\p{Blk= _Deseret:=}');
Error('\P{Blk= _Deseret:=}');
Expect(1, 66639, '\p{Blk=deseret}', "");
Expect(0, 66639, '\p{^Blk=deseret}', "");
Expect(0, 66639, '\P{Blk=deseret}', "");
Expect(1, 66639, '\P{^Blk=deseret}', "");
Expect(0, 66640, '\p{Blk=deseret}', "");
Expect(1, 66640, '\p{^Blk=deseret}', "");
Expect(1, 66640, '\P{Blk=deseret}', "");
Expect(0, 66640, '\P{^Blk=deseret}', "");
Expect(1, 66639, '\p{Blk=	 Deseret}', "");
Expect(0, 66639, '\p{^Blk=	 Deseret}', "");
Expect(0, 66639, '\P{Blk=	 Deseret}', "");
Expect(1, 66639, '\P{^Blk=	 Deseret}', "");
Expect(0, 66640, '\p{Blk=	 Deseret}', "");
Expect(1, 66640, '\p{^Blk=	 Deseret}', "");
Expect(1, 66640, '\P{Blk=	 Deseret}', "");
Expect(0, 66640, '\P{^Blk=	 Deseret}', "");
Error('\p{Is_Block=/a/Deseret}');
Error('\P{Is_Block=/a/Deseret}');
Expect(1, 66639, '\p{Is_Block=deseret}', "");
Expect(0, 66639, '\p{^Is_Block=deseret}', "");
Expect(0, 66639, '\P{Is_Block=deseret}', "");
Expect(1, 66639, '\P{^Is_Block=deseret}', "");
Expect(0, 66640, '\p{Is_Block=deseret}', "");
Expect(1, 66640, '\p{^Is_Block=deseret}', "");
Expect(1, 66640, '\P{Is_Block=deseret}', "");
Expect(0, 66640, '\P{^Is_Block=deseret}', "");
Expect(1, 66639, '\p{Is_Block=_-Deseret}', "");
Expect(0, 66639, '\p{^Is_Block=_-Deseret}', "");
Expect(0, 66639, '\P{Is_Block=_-Deseret}', "");
Expect(1, 66639, '\P{^Is_Block=_-Deseret}', "");
Expect(0, 66640, '\p{Is_Block=_-Deseret}', "");
Expect(1, 66640, '\p{^Is_Block=_-Deseret}', "");
Expect(1, 66640, '\P{Is_Block=_-Deseret}', "");
Expect(0, 66640, '\P{^Is_Block=_-Deseret}', "");
Error('\p{Is_Blk= :=Deseret}');
Error('\P{Is_Blk= :=Deseret}');
Expect(1, 66639, '\p{Is_Blk=deseret}', "");
Expect(0, 66639, '\p{^Is_Blk=deseret}', "");
Expect(0, 66639, '\P{Is_Blk=deseret}', "");
Expect(1, 66639, '\P{^Is_Blk=deseret}', "");
Expect(0, 66640, '\p{Is_Blk=deseret}', "");
Expect(1, 66640, '\p{^Is_Blk=deseret}', "");
Expect(1, 66640, '\P{Is_Blk=deseret}', "");
Expect(0, 66640, '\P{^Is_Blk=deseret}', "");
Expect(1, 66639, '\p{Is_Blk=  deseret}', "");
Expect(0, 66639, '\p{^Is_Blk=  deseret}', "");
Expect(0, 66639, '\P{Is_Blk=  deseret}', "");
Expect(1, 66639, '\P{^Is_Blk=  deseret}', "");
Expect(0, 66640, '\p{Is_Blk=  deseret}', "");
Expect(1, 66640, '\p{^Is_Blk=  deseret}', "");
Expect(1, 66640, '\P{Is_Blk=  deseret}', "");
Expect(0, 66640, '\P{^Is_Blk=  deseret}', "");
Error('\p{Block=_Devanagari/a/}');
Error('\P{Block=_Devanagari/a/}');
Expect(1, 2431, '\p{Block=devanagari}', "");
Expect(0, 2431, '\p{^Block=devanagari}', "");
Expect(0, 2431, '\P{Block=devanagari}', "");
Expect(1, 2431, '\P{^Block=devanagari}', "");
Expect(0, 2432, '\p{Block=devanagari}', "");
Expect(1, 2432, '\p{^Block=devanagari}', "");
Expect(1, 2432, '\P{Block=devanagari}', "");
Expect(0, 2432, '\P{^Block=devanagari}', "");
Expect(1, 2431, '\p{Block= -Devanagari}', "");
Expect(0, 2431, '\p{^Block= -Devanagari}', "");
Expect(0, 2431, '\P{Block= -Devanagari}', "");
Expect(1, 2431, '\P{^Block= -Devanagari}', "");
Expect(0, 2432, '\p{Block= -Devanagari}', "");
Expect(1, 2432, '\p{^Block= -Devanagari}', "");
Expect(1, 2432, '\P{Block= -Devanagari}', "");
Expect(0, 2432, '\P{^Block= -Devanagari}', "");
Error('\p{Blk=_Devanagari:=}');
Error('\P{Blk=_Devanagari:=}');
Expect(1, 2431, '\p{Blk=devanagari}', "");
Expect(0, 2431, '\p{^Blk=devanagari}', "");
Expect(0, 2431, '\P{Blk=devanagari}', "");
Expect(1, 2431, '\P{^Blk=devanagari}', "");
Expect(0, 2432, '\p{Blk=devanagari}', "");
Expect(1, 2432, '\p{^Blk=devanagari}', "");
Expect(1, 2432, '\P{Blk=devanagari}', "");
Expect(0, 2432, '\P{^Blk=devanagari}', "");
Expect(1, 2431, '\p{Blk= DEVANAGARI}', "");
Expect(0, 2431, '\p{^Blk= DEVANAGARI}', "");
Expect(0, 2431, '\P{Blk= DEVANAGARI}', "");
Expect(1, 2431, '\P{^Blk= DEVANAGARI}', "");
Expect(0, 2432, '\p{Blk= DEVANAGARI}', "");
Expect(1, 2432, '\p{^Blk= DEVANAGARI}', "");
Expect(1, 2432, '\P{Blk= DEVANAGARI}', "");
Expect(0, 2432, '\P{^Blk= DEVANAGARI}', "");
Error('\p{Is_Block= :=devanagari}');
Error('\P{Is_Block= :=devanagari}');
Expect(1, 2431, '\p{Is_Block=devanagari}', "");
Expect(0, 2431, '\p{^Is_Block=devanagari}', "");
Expect(0, 2431, '\P{Is_Block=devanagari}', "");
Expect(1, 2431, '\P{^Is_Block=devanagari}', "");
Expect(0, 2432, '\p{Is_Block=devanagari}', "");
Expect(1, 2432, '\p{^Is_Block=devanagari}', "");
Expect(1, 2432, '\P{Is_Block=devanagari}', "");
Expect(0, 2432, '\P{^Is_Block=devanagari}', "");
Expect(1, 2431, '\p{Is_Block=_	DEVANAGARI}', "");
Expect(0, 2431, '\p{^Is_Block=_	DEVANAGARI}', "");
Expect(0, 2431, '\P{Is_Block=_	DEVANAGARI}', "");
Expect(1, 2431, '\P{^Is_Block=_	DEVANAGARI}', "");
Expect(0, 2432, '\p{Is_Block=_	DEVANAGARI}', "");
Expect(1, 2432, '\p{^Is_Block=_	DEVANAGARI}', "");
Expect(1, 2432, '\P{Is_Block=_	DEVANAGARI}', "");
Expect(0, 2432, '\P{^Is_Block=_	DEVANAGARI}', "");
Error('\p{Is_Blk=/a/-DEVANAGARI}');
Error('\P{Is_Blk=/a/-DEVANAGARI}');
Expect(1, 2431, '\p{Is_Blk=devanagari}', "");
Expect(0, 2431, '\p{^Is_Blk=devanagari}', "");
Expect(0, 2431, '\P{Is_Blk=devanagari}', "");
Expect(1, 2431, '\P{^Is_Blk=devanagari}', "");
Expect(0, 2432, '\p{Is_Blk=devanagari}', "");
Expect(1, 2432, '\p{^Is_Blk=devanagari}', "");
Expect(1, 2432, '\P{Is_Blk=devanagari}', "");
Expect(0, 2432, '\P{^Is_Blk=devanagari}', "");
Expect(1, 2431, '\p{Is_Blk=		DEVANAGARI}', "");
Expect(0, 2431, '\p{^Is_Blk=		DEVANAGARI}', "");
Expect(0, 2431, '\P{Is_Blk=		DEVANAGARI}', "");
Expect(1, 2431, '\P{^Is_Blk=		DEVANAGARI}', "");
Expect(0, 2432, '\p{Is_Blk=		DEVANAGARI}', "");
Expect(1, 2432, '\p{^Is_Blk=		DEVANAGARI}', "");
Expect(1, 2432, '\P{Is_Blk=		DEVANAGARI}', "");
Expect(0, 2432, '\P{^Is_Blk=		DEVANAGARI}', "");
Error('\p{Block=:= _DEVANAGARI_extended}');
Error('\P{Block=:= _DEVANAGARI_extended}');
Expect(1, 43263, '\p{Block=devanagariextended}', "");
Expect(0, 43263, '\p{^Block=devanagariextended}', "");
Expect(0, 43263, '\P{Block=devanagariextended}', "");
Expect(1, 43263, '\P{^Block=devanagariextended}', "");
Expect(0, 43264, '\p{Block=devanagariextended}', "");
Expect(1, 43264, '\p{^Block=devanagariextended}', "");
Expect(1, 43264, '\P{Block=devanagariextended}', "");
Expect(0, 43264, '\P{^Block=devanagariextended}', "");
Expect(1, 43263, '\p{Block=_Devanagari_Extended}', "");
Expect(0, 43263, '\p{^Block=_Devanagari_Extended}', "");
Expect(0, 43263, '\P{Block=_Devanagari_Extended}', "");
Expect(1, 43263, '\P{^Block=_Devanagari_Extended}', "");
Expect(0, 43264, '\p{Block=_Devanagari_Extended}', "");
Expect(1, 43264, '\p{^Block=_Devanagari_Extended}', "");
Expect(1, 43264, '\P{Block=_Devanagari_Extended}', "");
Expect(0, 43264, '\P{^Block=_Devanagari_Extended}', "");
Error('\p{Blk=- Devanagari_ext:=}');
Error('\P{Blk=- Devanagari_ext:=}');
Expect(1, 43263, '\p{Blk=devanagariext}', "");
Expect(0, 43263, '\p{^Blk=devanagariext}', "");
Expect(0, 43263, '\P{Blk=devanagariext}', "");
Expect(1, 43263, '\P{^Blk=devanagariext}', "");
Expect(0, 43264, '\p{Blk=devanagariext}', "");
Expect(1, 43264, '\p{^Blk=devanagariext}', "");
Expect(1, 43264, '\P{Blk=devanagariext}', "");
Expect(0, 43264, '\P{^Blk=devanagariext}', "");
Expect(1, 43263, '\p{Blk:		_devanagari_ext}', "");
Expect(0, 43263, '\p{^Blk:		_devanagari_ext}', "");
Expect(0, 43263, '\P{Blk:		_devanagari_ext}', "");
Expect(1, 43263, '\P{^Blk:		_devanagari_ext}', "");
Expect(0, 43264, '\p{Blk:		_devanagari_ext}', "");
Expect(1, 43264, '\p{^Blk:		_devanagari_ext}', "");
Expect(1, 43264, '\P{Blk:		_devanagari_ext}', "");
Expect(0, 43264, '\P{^Blk:		_devanagari_ext}', "");
Error('\p{Is_Block=:= devanagari_Extended}');
Error('\P{Is_Block=:= devanagari_Extended}');
Expect(1, 43263, '\p{Is_Block=devanagariextended}', "");
Expect(0, 43263, '\p{^Is_Block=devanagariextended}', "");
Expect(0, 43263, '\P{Is_Block=devanagariextended}', "");
Expect(1, 43263, '\P{^Is_Block=devanagariextended}', "");
Expect(0, 43264, '\p{Is_Block=devanagariextended}', "");
Expect(1, 43264, '\p{^Is_Block=devanagariextended}', "");
Expect(1, 43264, '\P{Is_Block=devanagariextended}', "");
Expect(0, 43264, '\P{^Is_Block=devanagariextended}', "");
Expect(1, 43263, '\p{Is_Block:	-DEVANAGARI_Extended}', "");
Expect(0, 43263, '\p{^Is_Block:	-DEVANAGARI_Extended}', "");
Expect(0, 43263, '\P{Is_Block:	-DEVANAGARI_Extended}', "");
Expect(1, 43263, '\P{^Is_Block:	-DEVANAGARI_Extended}', "");
Expect(0, 43264, '\p{Is_Block:	-DEVANAGARI_Extended}', "");
Expect(1, 43264, '\p{^Is_Block:	-DEVANAGARI_Extended}', "");
Expect(1, 43264, '\P{Is_Block:	-DEVANAGARI_Extended}', "");
Expect(0, 43264, '\P{^Is_Block:	-DEVANAGARI_Extended}', "");
Error('\p{Is_Blk=/a/  devanagari_ext}');
Error('\P{Is_Blk=/a/  devanagari_ext}');
Expect(1, 43263, '\p{Is_Blk=devanagariext}', "");
Expect(0, 43263, '\p{^Is_Blk=devanagariext}', "");
Expect(0, 43263, '\P{Is_Blk=devanagariext}', "");
Expect(1, 43263, '\P{^Is_Blk=devanagariext}', "");
Expect(0, 43264, '\p{Is_Blk=devanagariext}', "");
Expect(1, 43264, '\p{^Is_Blk=devanagariext}', "");
Expect(1, 43264, '\P{Is_Blk=devanagariext}', "");
Expect(0, 43264, '\P{^Is_Blk=devanagariext}', "");
Expect(1, 43263, '\p{Is_Blk=	_devanagari_EXT}', "");
Expect(0, 43263, '\p{^Is_Blk=	_devanagari_EXT}', "");
Expect(0, 43263, '\P{Is_Blk=	_devanagari_EXT}', "");
Expect(1, 43263, '\P{^Is_Blk=	_devanagari_EXT}', "");
Expect(0, 43264, '\p{Is_Blk=	_devanagari_EXT}', "");
Expect(1, 43264, '\p{^Is_Blk=	_devanagari_EXT}', "");
Expect(1, 43264, '\P{Is_Blk=	_devanagari_EXT}', "");
Expect(0, 43264, '\P{^Is_Blk=	_devanagari_EXT}', "");
Error('\p{Block= /a/combining_diacritical_MARKS}');
Error('\P{Block= /a/combining_diacritical_MARKS}');
Expect(1, 879, '\p{Block=combiningdiacriticalmarks}', "");
Expect(0, 879, '\p{^Block=combiningdiacriticalmarks}', "");
Expect(0, 879, '\P{Block=combiningdiacriticalmarks}', "");
Expect(1, 879, '\P{^Block=combiningdiacriticalmarks}', "");
Expect(0, 880, '\p{Block=combiningdiacriticalmarks}', "");
Expect(1, 880, '\p{^Block=combiningdiacriticalmarks}', "");
Expect(1, 880, '\P{Block=combiningdiacriticalmarks}', "");
Expect(0, 880, '\P{^Block=combiningdiacriticalmarks}', "");
Expect(1, 879, '\p{Block=-	Combining_Diacritical_Marks}', "");
Expect(0, 879, '\p{^Block=-	Combining_Diacritical_Marks}', "");
Expect(0, 879, '\P{Block=-	Combining_Diacritical_Marks}', "");
Expect(1, 879, '\P{^Block=-	Combining_Diacritical_Marks}', "");
Expect(0, 880, '\p{Block=-	Combining_Diacritical_Marks}', "");
Expect(1, 880, '\p{^Block=-	Combining_Diacritical_Marks}', "");
Expect(1, 880, '\P{Block=-	Combining_Diacritical_Marks}', "");
Expect(0, 880, '\P{^Block=-	Combining_Diacritical_Marks}', "");
Error('\p{Blk=:=	_diacriticals}');
Error('\P{Blk=:=	_diacriticals}');
Expect(1, 879, '\p{Blk=diacriticals}', "");
Expect(0, 879, '\p{^Blk=diacriticals}', "");
Expect(0, 879, '\P{Blk=diacriticals}', "");
Expect(1, 879, '\P{^Blk=diacriticals}', "");
Expect(0, 880, '\p{Blk=diacriticals}', "");
Expect(1, 880, '\p{^Blk=diacriticals}', "");
Expect(1, 880, '\P{Blk=diacriticals}', "");
Expect(0, 880, '\P{^Blk=diacriticals}', "");
Expect(1, 879, '\p{Blk=--DIACRITICALS}', "");
Expect(0, 879, '\p{^Blk=--DIACRITICALS}', "");
Expect(0, 879, '\P{Blk=--DIACRITICALS}', "");
Expect(1, 879, '\P{^Blk=--DIACRITICALS}', "");
Expect(0, 880, '\p{Blk=--DIACRITICALS}', "");
Expect(1, 880, '\p{^Blk=--DIACRITICALS}', "");
Expect(1, 880, '\P{Blk=--DIACRITICALS}', "");
Expect(0, 880, '\P{^Blk=--DIACRITICALS}', "");
Error('\p{Is_Block=-/a/combining_DIACRITICAL_MARKS}');
Error('\P{Is_Block=-/a/combining_DIACRITICAL_MARKS}');
Expect(1, 879, '\p{Is_Block=combiningdiacriticalmarks}', "");
Expect(0, 879, '\p{^Is_Block=combiningdiacriticalmarks}', "");
Expect(0, 879, '\P{Is_Block=combiningdiacriticalmarks}', "");
Expect(1, 879, '\P{^Is_Block=combiningdiacriticalmarks}', "");
Expect(0, 880, '\p{Is_Block=combiningdiacriticalmarks}', "");
Expect(1, 880, '\p{^Is_Block=combiningdiacriticalmarks}', "");
Expect(1, 880, '\P{Is_Block=combiningdiacriticalmarks}', "");
Expect(0, 880, '\P{^Is_Block=combiningdiacriticalmarks}', "");
Expect(1, 879, '\p{Is_Block=_	combining_diacritical_MARKS}', "");
Expect(0, 879, '\p{^Is_Block=_	combining_diacritical_MARKS}', "");
Expect(0, 879, '\P{Is_Block=_	combining_diacritical_MARKS}', "");
Expect(1, 879, '\P{^Is_Block=_	combining_diacritical_MARKS}', "");
Expect(0, 880, '\p{Is_Block=_	combining_diacritical_MARKS}', "");
Expect(1, 880, '\p{^Is_Block=_	combining_diacritical_MARKS}', "");
Expect(1, 880, '\P{Is_Block=_	combining_diacritical_MARKS}', "");
Expect(0, 880, '\P{^Is_Block=_	combining_diacritical_MARKS}', "");
Error('\p{Is_Blk=:=_	diacriticals}');
Error('\P{Is_Blk=:=_	diacriticals}');
Expect(1, 879, '\p{Is_Blk=diacriticals}', "");
Expect(0, 879, '\p{^Is_Blk=diacriticals}', "");
Expect(0, 879, '\P{Is_Blk=diacriticals}', "");
Expect(1, 879, '\P{^Is_Blk=diacriticals}', "");
Expect(0, 880, '\p{Is_Blk=diacriticals}', "");
Expect(1, 880, '\p{^Is_Blk=diacriticals}', "");
Expect(1, 880, '\P{Is_Blk=diacriticals}', "");
Expect(0, 880, '\P{^Is_Blk=diacriticals}', "");
Expect(1, 879, '\p{Is_Blk:   -	Diacriticals}', "");
Expect(0, 879, '\p{^Is_Blk:   -	Diacriticals}', "");
Expect(0, 879, '\P{Is_Blk:   -	Diacriticals}', "");
Expect(1, 879, '\P{^Is_Blk:   -	Diacriticals}', "");
Expect(0, 880, '\p{Is_Blk:   -	Diacriticals}', "");
Expect(1, 880, '\p{^Is_Blk:   -	Diacriticals}', "");
Expect(1, 880, '\P{Is_Blk:   -	Diacriticals}', "");
Expect(0, 880, '\P{^Is_Blk:   -	Diacriticals}', "");
Error('\p{Block=	_Combining_Diacritical_Marks_for_SYMBOLS/a/}');
Error('\P{Block=	_Combining_Diacritical_Marks_for_SYMBOLS/a/}');
Expect(1, 8447, '\p{Block:combiningdiacriticalmarksforsymbols}', "");
Expect(0, 8447, '\p{^Block:combiningdiacriticalmarksforsymbols}', "");
Expect(0, 8447, '\P{Block:combiningdiacriticalmarksforsymbols}', "");
Expect(1, 8447, '\P{^Block:combiningdiacriticalmarksforsymbols}', "");
Expect(0, 8448, '\p{Block:combiningdiacriticalmarksforsymbols}', "");
Expect(1, 8448, '\p{^Block:combiningdiacriticalmarksforsymbols}', "");
Expect(1, 8448, '\P{Block:combiningdiacriticalmarksforsymbols}', "");
Expect(0, 8448, '\P{^Block:combiningdiacriticalmarksforsymbols}', "");
Expect(1, 8447, '\p{Block:   	_Combining_Diacritical_Marks_For_Symbols}', "");
Expect(0, 8447, '\p{^Block:   	_Combining_Diacritical_Marks_For_Symbols}', "");
Expect(0, 8447, '\P{Block:   	_Combining_Diacritical_Marks_For_Symbols}', "");
Expect(1, 8447, '\P{^Block:   	_Combining_Diacritical_Marks_For_Symbols}', "");
Expect(0, 8448, '\p{Block:   	_Combining_Diacritical_Marks_For_Symbols}', "");
Expect(1, 8448, '\p{^Block:   	_Combining_Diacritical_Marks_For_Symbols}', "");
Expect(1, 8448, '\P{Block:   	_Combining_Diacritical_Marks_For_Symbols}', "");
Expect(0, 8448, '\P{^Block:   	_Combining_Diacritical_Marks_For_Symbols}', "");
Error('\p{Blk=/a/_DIACRITICALS_for_Symbols}');
Error('\P{Blk=/a/_DIACRITICALS_for_Symbols}');
Expect(1, 8447, '\p{Blk=diacriticalsforsymbols}', "");
Expect(0, 8447, '\p{^Blk=diacriticalsforsymbols}', "");
Expect(0, 8447, '\P{Blk=diacriticalsforsymbols}', "");
Expect(1, 8447, '\P{^Blk=diacriticalsforsymbols}', "");
Expect(0, 8448, '\p{Blk=diacriticalsforsymbols}', "");
Expect(1, 8448, '\p{^Blk=diacriticalsforsymbols}', "");
Expect(1, 8448, '\P{Blk=diacriticalsforsymbols}', "");
Expect(0, 8448, '\P{^Blk=diacriticalsforsymbols}', "");
Expect(1, 8447, '\p{Blk=- DIACRITICALS_For_SYMBOLS}', "");
Expect(0, 8447, '\p{^Blk=- DIACRITICALS_For_SYMBOLS}', "");
Expect(0, 8447, '\P{Blk=- DIACRITICALS_For_SYMBOLS}', "");
Expect(1, 8447, '\P{^Blk=- DIACRITICALS_For_SYMBOLS}', "");
Expect(0, 8448, '\p{Blk=- DIACRITICALS_For_SYMBOLS}', "");
Expect(1, 8448, '\p{^Blk=- DIACRITICALS_For_SYMBOLS}', "");
Expect(1, 8448, '\P{Blk=- DIACRITICALS_For_SYMBOLS}', "");
Expect(0, 8448, '\P{^Blk=- DIACRITICALS_For_SYMBOLS}', "");
Error('\p{Is_Block=/a/_Combining_marks_FOR_Symbols}');
Error('\P{Is_Block=/a/_Combining_marks_FOR_Symbols}');
Expect(1, 8447, '\p{Is_Block=combiningmarksforsymbols}', "");
Expect(0, 8447, '\p{^Is_Block=combiningmarksforsymbols}', "");
Expect(0, 8447, '\P{Is_Block=combiningmarksforsymbols}', "");
Expect(1, 8447, '\P{^Is_Block=combiningmarksforsymbols}', "");
Expect(0, 8448, '\p{Is_Block=combiningmarksforsymbols}', "");
Expect(1, 8448, '\p{^Is_Block=combiningmarksforsymbols}', "");
Expect(1, 8448, '\P{Is_Block=combiningmarksforsymbols}', "");
Expect(0, 8448, '\P{^Is_Block=combiningmarksforsymbols}', "");
Expect(1, 8447, '\p{Is_Block=	combining_marks_For_Symbols}', "");
Expect(0, 8447, '\p{^Is_Block=	combining_marks_For_Symbols}', "");
Expect(0, 8447, '\P{Is_Block=	combining_marks_For_Symbols}', "");
Expect(1, 8447, '\P{^Is_Block=	combining_marks_For_Symbols}', "");
Expect(0, 8448, '\p{Is_Block=	combining_marks_For_Symbols}', "");
Expect(1, 8448, '\p{^Is_Block=	combining_marks_For_Symbols}', "");
Expect(1, 8448, '\P{Is_Block=	combining_marks_For_Symbols}', "");
Expect(0, 8448, '\P{^Is_Block=	combining_marks_For_Symbols}', "");
Error('\p{Is_Blk=-COMBINING_Diacritical_Marks_For_Symbols/a/}');
Error('\P{Is_Blk=-COMBINING_Diacritical_Marks_For_Symbols/a/}');
Expect(1, 8447, '\p{Is_Blk=combiningdiacriticalmarksforsymbols}', "");
Expect(0, 8447, '\p{^Is_Blk=combiningdiacriticalmarksforsymbols}', "");
Expect(0, 8447, '\P{Is_Blk=combiningdiacriticalmarksforsymbols}', "");
Expect(1, 8447, '\P{^Is_Blk=combiningdiacriticalmarksforsymbols}', "");
Expect(0, 8448, '\p{Is_Blk=combiningdiacriticalmarksforsymbols}', "");
Expect(1, 8448, '\p{^Is_Blk=combiningdiacriticalmarksforsymbols}', "");
Expect(1, 8448, '\P{Is_Blk=combiningdiacriticalmarksforsymbols}', "");
Expect(0, 8448, '\P{^Is_Blk=combiningdiacriticalmarksforsymbols}', "");
Expect(1, 8447, '\p{Is_Blk=-COMBINING_Diacritical_MARKS_FOR_SYMBOLS}', "");
Expect(0, 8447, '\p{^Is_Blk=-COMBINING_Diacritical_MARKS_FOR_SYMBOLS}', "");
Expect(0, 8447, '\P{Is_Blk=-COMBINING_Diacritical_MARKS_FOR_SYMBOLS}', "");
Expect(1, 8447, '\P{^Is_Blk=-COMBINING_Diacritical_MARKS_FOR_SYMBOLS}', "");
Expect(0, 8448, '\p{Is_Blk=-COMBINING_Diacritical_MARKS_FOR_SYMBOLS}', "");
Expect(1, 8448, '\p{^Is_Blk=-COMBINING_Diacritical_MARKS_FOR_SYMBOLS}', "");
Expect(1, 8448, '\P{Is_Blk=-COMBINING_Diacritical_MARKS_FOR_SYMBOLS}', "");
Expect(0, 8448, '\P{^Is_Blk=-COMBINING_Diacritical_MARKS_FOR_SYMBOLS}', "");
Error('\p{Block=:= -COMBINING_diacritical_Marks_Supplement}');
Error('\P{Block=:= -COMBINING_diacritical_Marks_Supplement}');
Expect(1, 7679, '\p{Block=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7679, '\p{^Block=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7679, '\P{Block=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7679, '\P{^Block=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7680, '\p{Block=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7680, '\p{^Block=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7680, '\P{Block=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7680, '\P{^Block=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7679, '\p{Block=	_Combining_DIACRITICAL_Marks_SUPPLEMENT}', "");
Expect(0, 7679, '\p{^Block=	_Combining_DIACRITICAL_Marks_SUPPLEMENT}', "");
Expect(0, 7679, '\P{Block=	_Combining_DIACRITICAL_Marks_SUPPLEMENT}', "");
Expect(1, 7679, '\P{^Block=	_Combining_DIACRITICAL_Marks_SUPPLEMENT}', "");
Expect(0, 7680, '\p{Block=	_Combining_DIACRITICAL_Marks_SUPPLEMENT}', "");
Expect(1, 7680, '\p{^Block=	_Combining_DIACRITICAL_Marks_SUPPLEMENT}', "");
Expect(1, 7680, '\P{Block=	_Combining_DIACRITICAL_Marks_SUPPLEMENT}', "");
Expect(0, 7680, '\P{^Block=	_Combining_DIACRITICAL_Marks_SUPPLEMENT}', "");
Error('\p{Blk=:= Diacriticals_SUP}');
Error('\P{Blk=:= Diacriticals_SUP}');
Expect(1, 7679, '\p{Blk=diacriticalssup}', "");
Expect(0, 7679, '\p{^Blk=diacriticalssup}', "");
Expect(0, 7679, '\P{Blk=diacriticalssup}', "");
Expect(1, 7679, '\P{^Blk=diacriticalssup}', "");
Expect(0, 7680, '\p{Blk=diacriticalssup}', "");
Expect(1, 7680, '\p{^Blk=diacriticalssup}', "");
Expect(1, 7680, '\P{Blk=diacriticalssup}', "");
Expect(0, 7680, '\P{^Blk=diacriticalssup}', "");
Expect(1, 7679, '\p{Blk:- diacriticals_Sup}', "");
Expect(0, 7679, '\p{^Blk:- diacriticals_Sup}', "");
Expect(0, 7679, '\P{Blk:- diacriticals_Sup}', "");
Expect(1, 7679, '\P{^Blk:- diacriticals_Sup}', "");
Expect(0, 7680, '\p{Blk:- diacriticals_Sup}', "");
Expect(1, 7680, '\p{^Blk:- diacriticals_Sup}', "");
Expect(1, 7680, '\P{Blk:- diacriticals_Sup}', "");
Expect(0, 7680, '\P{^Blk:- diacriticals_Sup}', "");
Error('\p{Is_Block=:=-_combining_Diacritical_MARKS_Supplement}');
Error('\P{Is_Block=:=-_combining_Diacritical_MARKS_Supplement}');
Expect(1, 7679, '\p{Is_Block=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7679, '\p{^Is_Block=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7679, '\P{Is_Block=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7679, '\P{^Is_Block=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7680, '\p{Is_Block=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7680, '\p{^Is_Block=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7680, '\P{Is_Block=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7680, '\P{^Is_Block=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7679, '\p{Is_Block=-combining_Diacritical_Marks_Supplement}', "");
Expect(0, 7679, '\p{^Is_Block=-combining_Diacritical_Marks_Supplement}', "");
Expect(0, 7679, '\P{Is_Block=-combining_Diacritical_Marks_Supplement}', "");
Expect(1, 7679, '\P{^Is_Block=-combining_Diacritical_Marks_Supplement}', "");
Expect(0, 7680, '\p{Is_Block=-combining_Diacritical_Marks_Supplement}', "");
Expect(1, 7680, '\p{^Is_Block=-combining_Diacritical_Marks_Supplement}', "");
Expect(1, 7680, '\P{Is_Block=-combining_Diacritical_Marks_Supplement}', "");
Expect(0, 7680, '\P{^Is_Block=-combining_Diacritical_Marks_Supplement}', "");
Error('\p{Is_Blk::=_Diacriticals_sup}');
Error('\P{Is_Blk::=_Diacriticals_sup}');
Expect(1, 7679, '\p{Is_Blk=diacriticalssup}', "");
Expect(0, 7679, '\p{^Is_Blk=diacriticalssup}', "");
Expect(0, 7679, '\P{Is_Blk=diacriticalssup}', "");
Expect(1, 7679, '\P{^Is_Blk=diacriticalssup}', "");
Expect(0, 7680, '\p{Is_Blk=diacriticalssup}', "");
Expect(1, 7680, '\p{^Is_Blk=diacriticalssup}', "");
Expect(1, 7680, '\P{Is_Blk=diacriticalssup}', "");
Expect(0, 7680, '\P{^Is_Blk=diacriticalssup}', "");
Expect(1, 7679, '\p{Is_Blk= Diacriticals_SUP}', "");
Expect(0, 7679, '\p{^Is_Blk= Diacriticals_SUP}', "");
Expect(0, 7679, '\P{Is_Blk= Diacriticals_SUP}', "");
Expect(1, 7679, '\P{^Is_Blk= Diacriticals_SUP}', "");
Expect(0, 7680, '\p{Is_Blk= Diacriticals_SUP}', "");
Expect(1, 7680, '\p{^Is_Blk= Diacriticals_SUP}', "");
Expect(1, 7680, '\P{Is_Blk= Diacriticals_SUP}', "");
Expect(0, 7680, '\P{^Is_Blk= Diacriticals_SUP}', "");
Error('\p{Block=-:=dingbats}');
Error('\P{Block=-:=dingbats}');
Expect(1, 10175, '\p{Block=dingbats}', "");
Expect(0, 10175, '\p{^Block=dingbats}', "");
Expect(0, 10175, '\P{Block=dingbats}', "");
Expect(1, 10175, '\P{^Block=dingbats}', "");
Expect(0, 10176, '\p{Block=dingbats}', "");
Expect(1, 10176, '\p{^Block=dingbats}', "");
Expect(1, 10176, '\P{Block=dingbats}', "");
Expect(0, 10176, '\P{^Block=dingbats}', "");
Expect(1, 10175, '\p{Block=	-Dingbats}', "");
Expect(0, 10175, '\p{^Block=	-Dingbats}', "");
Expect(0, 10175, '\P{Block=	-Dingbats}', "");
Expect(1, 10175, '\P{^Block=	-Dingbats}', "");
Expect(0, 10176, '\p{Block=	-Dingbats}', "");
Expect(1, 10176, '\p{^Block=	-Dingbats}', "");
Expect(1, 10176, '\P{Block=	-Dingbats}', "");
Expect(0, 10176, '\P{^Block=	-Dingbats}', "");
Error('\p{Blk=_ DINGBATS/a/}');
Error('\P{Blk=_ DINGBATS/a/}');
Expect(1, 10175, '\p{Blk=dingbats}', "");
Expect(0, 10175, '\p{^Blk=dingbats}', "");
Expect(0, 10175, '\P{Blk=dingbats}', "");
Expect(1, 10175, '\P{^Blk=dingbats}', "");
Expect(0, 10176, '\p{Blk=dingbats}', "");
Expect(1, 10176, '\p{^Blk=dingbats}', "");
Expect(1, 10176, '\P{Blk=dingbats}', "");
Expect(0, 10176, '\P{^Blk=dingbats}', "");
Expect(1, 10175, '\p{Blk=_dingbats}', "");
Expect(0, 10175, '\p{^Blk=_dingbats}', "");
Expect(0, 10175, '\P{Blk=_dingbats}', "");
Expect(1, 10175, '\P{^Blk=_dingbats}', "");
Expect(0, 10176, '\p{Blk=_dingbats}', "");
Expect(1, 10176, '\p{^Blk=_dingbats}', "");
Expect(1, 10176, '\P{Blk=_dingbats}', "");
Expect(0, 10176, '\P{^Blk=_dingbats}', "");
Error('\p{Is_Block=-:=Dingbats}');
Error('\P{Is_Block=-:=Dingbats}');
Expect(1, 10175, '\p{Is_Block: dingbats}', "");
Expect(0, 10175, '\p{^Is_Block: dingbats}', "");
Expect(0, 10175, '\P{Is_Block: dingbats}', "");
Expect(1, 10175, '\P{^Is_Block: dingbats}', "");
Expect(0, 10176, '\p{Is_Block: dingbats}', "");
Expect(1, 10176, '\p{^Is_Block: dingbats}', "");
Expect(1, 10176, '\P{Is_Block: dingbats}', "");
Expect(0, 10176, '\P{^Is_Block: dingbats}', "");
Expect(1, 10175, '\p{Is_Block=		Dingbats}', "");
Expect(0, 10175, '\p{^Is_Block=		Dingbats}', "");
Expect(0, 10175, '\P{Is_Block=		Dingbats}', "");
Expect(1, 10175, '\P{^Is_Block=		Dingbats}', "");
Expect(0, 10176, '\p{Is_Block=		Dingbats}', "");
Expect(1, 10176, '\p{^Is_Block=		Dingbats}', "");
Expect(1, 10176, '\P{Is_Block=		Dingbats}', "");
Expect(0, 10176, '\P{^Is_Block=		Dingbats}', "");
Error('\p{Is_Blk=-:=Dingbats}');
Error('\P{Is_Blk=-:=Dingbats}');
Expect(1, 10175, '\p{Is_Blk=dingbats}', "");
Expect(0, 10175, '\p{^Is_Blk=dingbats}', "");
Expect(0, 10175, '\P{Is_Blk=dingbats}', "");
Expect(1, 10175, '\P{^Is_Blk=dingbats}', "");
Expect(0, 10176, '\p{Is_Blk=dingbats}', "");
Expect(1, 10176, '\p{^Is_Blk=dingbats}', "");
Expect(1, 10176, '\P{Is_Blk=dingbats}', "");
Expect(0, 10176, '\P{^Is_Blk=dingbats}', "");
Expect(1, 10175, '\p{Is_Blk=-_dingbats}', "");
Expect(0, 10175, '\p{^Is_Blk=-_dingbats}', "");
Expect(0, 10175, '\P{Is_Blk=-_dingbats}', "");
Expect(1, 10175, '\P{^Is_Blk=-_dingbats}', "");
Expect(0, 10176, '\p{Is_Blk=-_dingbats}', "");
Expect(1, 10176, '\p{^Is_Blk=-_dingbats}', "");
Expect(1, 10176, '\P{Is_Blk=-_dingbats}', "");
Expect(0, 10176, '\P{^Is_Blk=-_dingbats}', "");
Error('\p{Block= -DOMINO_Tiles/a/}');
Error('\P{Block= -DOMINO_Tiles/a/}');
Expect(1, 127135, '\p{Block=dominotiles}', "");
Expect(0, 127135, '\p{^Block=dominotiles}', "");
Expect(0, 127135, '\P{Block=dominotiles}', "");
Expect(1, 127135, '\P{^Block=dominotiles}', "");
Expect(0, 127136, '\p{Block=dominotiles}', "");
Expect(1, 127136, '\p{^Block=dominotiles}', "");
Expect(1, 127136, '\P{Block=dominotiles}', "");
Expect(0, 127136, '\P{^Block=dominotiles}', "");
Expect(1, 127135, '\p{Block=		domino_Tiles}', "");
Expect(0, 127135, '\p{^Block=		domino_Tiles}', "");
Expect(0, 127135, '\P{Block=		domino_Tiles}', "");
Expect(1, 127135, '\P{^Block=		domino_Tiles}', "");
Expect(0, 127136, '\p{Block=		domino_Tiles}', "");
Expect(1, 127136, '\p{^Block=		domino_Tiles}', "");
Expect(1, 127136, '\P{Block=		domino_Tiles}', "");
Expect(0, 127136, '\P{^Block=		domino_Tiles}', "");
Error('\p{Blk=	-Domino:=}');
Error('\P{Blk=	-Domino:=}');
Expect(1, 127135, '\p{Blk=domino}', "");
Expect(0, 127135, '\p{^Blk=domino}', "");
Expect(0, 127135, '\P{Blk=domino}', "");
Expect(1, 127135, '\P{^Blk=domino}', "");
Expect(0, 127136, '\p{Blk=domino}', "");
Expect(1, 127136, '\p{^Blk=domino}', "");
Expect(1, 127136, '\P{Blk=domino}', "");
Expect(0, 127136, '\P{^Blk=domino}', "");
Expect(1, 127135, '\p{Blk= 	domino}', "");
Expect(0, 127135, '\p{^Blk= 	domino}', "");
Expect(0, 127135, '\P{Blk= 	domino}', "");
Expect(1, 127135, '\P{^Blk= 	domino}', "");
Expect(0, 127136, '\p{Blk= 	domino}', "");
Expect(1, 127136, '\p{^Blk= 	domino}', "");
Expect(1, 127136, '\P{Blk= 	domino}', "");
Expect(0, 127136, '\P{^Blk= 	domino}', "");
Error('\p{Is_Block=/a/	Domino_Tiles}');
Error('\P{Is_Block=/a/	Domino_Tiles}');
Expect(1, 127135, '\p{Is_Block:dominotiles}', "");
Expect(0, 127135, '\p{^Is_Block:dominotiles}', "");
Expect(0, 127135, '\P{Is_Block:dominotiles}', "");
Expect(1, 127135, '\P{^Is_Block:dominotiles}', "");
Expect(0, 127136, '\p{Is_Block:dominotiles}', "");
Expect(1, 127136, '\p{^Is_Block:dominotiles}', "");
Expect(1, 127136, '\P{Is_Block:dominotiles}', "");
Expect(0, 127136, '\P{^Is_Block:dominotiles}', "");
Expect(1, 127135, '\p{Is_Block= _Domino_Tiles}', "");
Expect(0, 127135, '\p{^Is_Block= _Domino_Tiles}', "");
Expect(0, 127135, '\P{Is_Block= _Domino_Tiles}', "");
Expect(1, 127135, '\P{^Is_Block= _Domino_Tiles}', "");
Expect(0, 127136, '\p{Is_Block= _Domino_Tiles}', "");
Expect(1, 127136, '\p{^Is_Block= _Domino_Tiles}', "");
Expect(1, 127136, '\P{Is_Block= _Domino_Tiles}', "");
Expect(0, 127136, '\P{^Is_Block= _Domino_Tiles}', "");
Error('\p{Is_Blk=:=-Domino}');
Error('\P{Is_Blk=:=-Domino}');
Expect(1, 127135, '\p{Is_Blk=domino}', "");
Expect(0, 127135, '\p{^Is_Blk=domino}', "");
Expect(0, 127135, '\P{Is_Blk=domino}', "");
Expect(1, 127135, '\P{^Is_Blk=domino}', "");
Expect(0, 127136, '\p{Is_Blk=domino}', "");
Expect(1, 127136, '\p{^Is_Blk=domino}', "");
Expect(1, 127136, '\P{Is_Blk=domino}', "");
Expect(0, 127136, '\P{^Is_Blk=domino}', "");
Expect(1, 127135, '\p{Is_Blk=-	DOMINO}', "");
Expect(0, 127135, '\p{^Is_Blk=-	DOMINO}', "");
Expect(0, 127135, '\P{Is_Blk=-	DOMINO}', "");
Expect(1, 127135, '\P{^Is_Blk=-	DOMINO}', "");
Expect(0, 127136, '\p{Is_Blk=-	DOMINO}', "");
Expect(1, 127136, '\p{^Is_Blk=-	DOMINO}', "");
Expect(1, 127136, '\P{Is_Blk=-	DOMINO}', "");
Expect(0, 127136, '\P{^Is_Blk=-	DOMINO}', "");
Error('\p{Block= 	egyptian_hieroglyphs:=}');
Error('\P{Block= 	egyptian_hieroglyphs:=}');
Expect(1, 78895, '\p{Block=egyptianhieroglyphs}', "");
Expect(0, 78895, '\p{^Block=egyptianhieroglyphs}', "");
Expect(0, 78895, '\P{Block=egyptianhieroglyphs}', "");
Expect(1, 78895, '\P{^Block=egyptianhieroglyphs}', "");
Expect(0, 78896, '\p{Block=egyptianhieroglyphs}', "");
Expect(1, 78896, '\p{^Block=egyptianhieroglyphs}', "");
Expect(1, 78896, '\P{Block=egyptianhieroglyphs}', "");
Expect(0, 78896, '\P{^Block=egyptianhieroglyphs}', "");
Expect(1, 78895, '\p{Block:  Egyptian_HIEROGLYPHS}', "");
Expect(0, 78895, '\p{^Block:  Egyptian_HIEROGLYPHS}', "");
Expect(0, 78895, '\P{Block:  Egyptian_HIEROGLYPHS}', "");
Expect(1, 78895, '\P{^Block:  Egyptian_HIEROGLYPHS}', "");
Expect(0, 78896, '\p{Block:  Egyptian_HIEROGLYPHS}', "");
Expect(1, 78896, '\p{^Block:  Egyptian_HIEROGLYPHS}', "");
Expect(1, 78896, '\P{Block:  Egyptian_HIEROGLYPHS}', "");
Expect(0, 78896, '\P{^Block:  Egyptian_HIEROGLYPHS}', "");
Error('\p{Blk:-/a/egyptian_Hieroglyphs}');
Error('\P{Blk:-/a/egyptian_Hieroglyphs}');
Expect(1, 78895, '\p{Blk:   egyptianhieroglyphs}', "");
Expect(0, 78895, '\p{^Blk:   egyptianhieroglyphs}', "");
Expect(0, 78895, '\P{Blk:   egyptianhieroglyphs}', "");
Expect(1, 78895, '\P{^Blk:   egyptianhieroglyphs}', "");
Expect(0, 78896, '\p{Blk:   egyptianhieroglyphs}', "");
Expect(1, 78896, '\p{^Blk:   egyptianhieroglyphs}', "");
Expect(1, 78896, '\P{Blk:   egyptianhieroglyphs}', "");
Expect(0, 78896, '\P{^Blk:   egyptianhieroglyphs}', "");
Expect(1, 78895, '\p{Blk=EGYPTIAN_HIEROGLYPHS}', "");
Expect(0, 78895, '\p{^Blk=EGYPTIAN_HIEROGLYPHS}', "");
Expect(0, 78895, '\P{Blk=EGYPTIAN_HIEROGLYPHS}', "");
Expect(1, 78895, '\P{^Blk=EGYPTIAN_HIEROGLYPHS}', "");
Expect(0, 78896, '\p{Blk=EGYPTIAN_HIEROGLYPHS}', "");
Expect(1, 78896, '\p{^Blk=EGYPTIAN_HIEROGLYPHS}', "");
Expect(1, 78896, '\P{Blk=EGYPTIAN_HIEROGLYPHS}', "");
Expect(0, 78896, '\P{^Blk=EGYPTIAN_HIEROGLYPHS}', "");
Error('\p{Is_Block=/a/ Egyptian_hieroglyphs}');
Error('\P{Is_Block=/a/ Egyptian_hieroglyphs}');
Expect(1, 78895, '\p{Is_Block=egyptianhieroglyphs}', "");
Expect(0, 78895, '\p{^Is_Block=egyptianhieroglyphs}', "");
Expect(0, 78895, '\P{Is_Block=egyptianhieroglyphs}', "");
Expect(1, 78895, '\P{^Is_Block=egyptianhieroglyphs}', "");
Expect(0, 78896, '\p{Is_Block=egyptianhieroglyphs}', "");
Expect(1, 78896, '\p{^Is_Block=egyptianhieroglyphs}', "");
Expect(1, 78896, '\P{Is_Block=egyptianhieroglyphs}', "");
Expect(0, 78896, '\P{^Is_Block=egyptianhieroglyphs}', "");
Expect(1, 78895, '\p{Is_Block=- EGYPTIAN_hieroglyphs}', "");
Expect(0, 78895, '\p{^Is_Block=- EGYPTIAN_hieroglyphs}', "");
Expect(0, 78895, '\P{Is_Block=- EGYPTIAN_hieroglyphs}', "");
Expect(1, 78895, '\P{^Is_Block=- EGYPTIAN_hieroglyphs}', "");
Expect(0, 78896, '\p{Is_Block=- EGYPTIAN_hieroglyphs}', "");
Expect(1, 78896, '\p{^Is_Block=- EGYPTIAN_hieroglyphs}', "");
Expect(1, 78896, '\P{Is_Block=- EGYPTIAN_hieroglyphs}', "");
Expect(0, 78896, '\P{^Is_Block=- EGYPTIAN_hieroglyphs}', "");
Error('\p{Is_Blk=_egyptian_Hieroglyphs/a/}');
Error('\P{Is_Blk=_egyptian_Hieroglyphs/a/}');
Expect(1, 78895, '\p{Is_Blk=egyptianhieroglyphs}', "");
Expect(0, 78895, '\p{^Is_Blk=egyptianhieroglyphs}', "");
Expect(0, 78895, '\P{Is_Blk=egyptianhieroglyphs}', "");
Expect(1, 78895, '\P{^Is_Blk=egyptianhieroglyphs}', "");
Expect(0, 78896, '\p{Is_Blk=egyptianhieroglyphs}', "");
Expect(1, 78896, '\p{^Is_Blk=egyptianhieroglyphs}', "");
Expect(1, 78896, '\P{Is_Blk=egyptianhieroglyphs}', "");
Expect(0, 78896, '\P{^Is_Blk=egyptianhieroglyphs}', "");
Expect(1, 78895, '\p{Is_Blk=- Egyptian_hieroglyphs}', "");
Expect(0, 78895, '\p{^Is_Blk=- Egyptian_hieroglyphs}', "");
Expect(0, 78895, '\P{Is_Blk=- Egyptian_hieroglyphs}', "");
Expect(1, 78895, '\P{^Is_Blk=- Egyptian_hieroglyphs}', "");
Expect(0, 78896, '\p{Is_Blk=- Egyptian_hieroglyphs}', "");
Expect(1, 78896, '\p{^Is_Blk=- Egyptian_hieroglyphs}', "");
Expect(1, 78896, '\P{Is_Blk=- Egyptian_hieroglyphs}', "");
Expect(0, 78896, '\P{^Is_Blk=- Egyptian_hieroglyphs}', "");
Error('\p{Block= :=Emoticons}');
Error('\P{Block= :=Emoticons}');
Expect(1, 128591, '\p{Block=emoticons}', "");
Expect(0, 128591, '\p{^Block=emoticons}', "");
Expect(0, 128591, '\P{Block=emoticons}', "");
Expect(1, 128591, '\P{^Block=emoticons}', "");
Expect(0, 128592, '\p{Block=emoticons}', "");
Expect(1, 128592, '\p{^Block=emoticons}', "");
Expect(1, 128592, '\P{Block=emoticons}', "");
Expect(0, 128592, '\P{^Block=emoticons}', "");
Expect(1, 128591, '\p{Block=	_EMOTICONS}', "");
Expect(0, 128591, '\p{^Block=	_EMOTICONS}', "");
Expect(0, 128591, '\P{Block=	_EMOTICONS}', "");
Expect(1, 128591, '\P{^Block=	_EMOTICONS}', "");
Expect(0, 128592, '\p{Block=	_EMOTICONS}', "");
Expect(1, 128592, '\p{^Block=	_EMOTICONS}', "");
Expect(1, 128592, '\P{Block=	_EMOTICONS}', "");
Expect(0, 128592, '\P{^Block=	_EMOTICONS}', "");
Error('\p{Blk=:=  EMOTICONS}');
Error('\P{Blk=:=  EMOTICONS}');
Expect(1, 128591, '\p{Blk=emoticons}', "");
Expect(0, 128591, '\p{^Blk=emoticons}', "");
Expect(0, 128591, '\P{Blk=emoticons}', "");
Expect(1, 128591, '\P{^Blk=emoticons}', "");
Expect(0, 128592, '\p{Blk=emoticons}', "");
Expect(1, 128592, '\p{^Blk=emoticons}', "");
Expect(1, 128592, '\P{Blk=emoticons}', "");
Expect(0, 128592, '\P{^Blk=emoticons}', "");
Expect(1, 128591, '\p{Blk=-_EMOTICONS}', "");
Expect(0, 128591, '\p{^Blk=-_EMOTICONS}', "");
Expect(0, 128591, '\P{Blk=-_EMOTICONS}', "");
Expect(1, 128591, '\P{^Blk=-_EMOTICONS}', "");
Expect(0, 128592, '\p{Blk=-_EMOTICONS}', "");
Expect(1, 128592, '\p{^Blk=-_EMOTICONS}', "");
Expect(1, 128592, '\P{Blk=-_EMOTICONS}', "");
Expect(0, 128592, '\P{^Blk=-_EMOTICONS}', "");
Error('\p{Is_Block=-:=Emoticons}');
Error('\P{Is_Block=-:=Emoticons}');
Expect(1, 128591, '\p{Is_Block=emoticons}', "");
Expect(0, 128591, '\p{^Is_Block=emoticons}', "");
Expect(0, 128591, '\P{Is_Block=emoticons}', "");
Expect(1, 128591, '\P{^Is_Block=emoticons}', "");
Expect(0, 128592, '\p{Is_Block=emoticons}', "");
Expect(1, 128592, '\p{^Is_Block=emoticons}', "");
Expect(1, 128592, '\P{Is_Block=emoticons}', "");
Expect(0, 128592, '\P{^Is_Block=emoticons}', "");
Expect(1, 128591, '\p{Is_Block=_	emoticons}', "");
Expect(0, 128591, '\p{^Is_Block=_	emoticons}', "");
Expect(0, 128591, '\P{Is_Block=_	emoticons}', "");
Expect(1, 128591, '\P{^Is_Block=_	emoticons}', "");
Expect(0, 128592, '\p{Is_Block=_	emoticons}', "");
Expect(1, 128592, '\p{^Is_Block=_	emoticons}', "");
Expect(1, 128592, '\P{Is_Block=_	emoticons}', "");
Expect(0, 128592, '\P{^Is_Block=_	emoticons}', "");
Error('\p{Is_Blk=/a/ 	EMOTICONS}');
Error('\P{Is_Blk=/a/ 	EMOTICONS}');
Expect(1, 128591, '\p{Is_Blk=emoticons}', "");
Expect(0, 128591, '\p{^Is_Blk=emoticons}', "");
Expect(0, 128591, '\P{Is_Blk=emoticons}', "");
Expect(1, 128591, '\P{^Is_Blk=emoticons}', "");
Expect(0, 128592, '\p{Is_Blk=emoticons}', "");
Expect(1, 128592, '\p{^Is_Blk=emoticons}', "");
Expect(1, 128592, '\P{Is_Blk=emoticons}', "");
Expect(0, 128592, '\P{^Is_Blk=emoticons}', "");
Expect(1, 128591, '\p{Is_Blk=	_Emoticons}', "");
Expect(0, 128591, '\p{^Is_Blk=	_Emoticons}', "");
Expect(0, 128591, '\P{Is_Blk=	_Emoticons}', "");
Expect(1, 128591, '\P{^Is_Blk=	_Emoticons}', "");
Expect(0, 128592, '\p{Is_Blk=	_Emoticons}', "");
Expect(1, 128592, '\p{^Is_Blk=	_Emoticons}', "");
Expect(1, 128592, '\P{Is_Blk=	_Emoticons}', "");
Expect(0, 128592, '\P{^Is_Blk=	_Emoticons}', "");
Error('\p{Block=-	enclosed_Alphanumerics:=}');
Error('\P{Block=-	enclosed_Alphanumerics:=}');
Expect(1, 9471, '\p{Block=enclosedalphanumerics}', "");
Expect(0, 9471, '\p{^Block=enclosedalphanumerics}', "");
Expect(0, 9471, '\P{Block=enclosedalphanumerics}', "");
Expect(1, 9471, '\P{^Block=enclosedalphanumerics}', "");
Expect(0, 9472, '\p{Block=enclosedalphanumerics}', "");
Expect(1, 9472, '\p{^Block=enclosedalphanumerics}', "");
Expect(1, 9472, '\P{Block=enclosedalphanumerics}', "");
Expect(0, 9472, '\P{^Block=enclosedalphanumerics}', "");
Expect(1, 9471, '\p{Block=		ENCLOSED_alphanumerics}', "");
Expect(0, 9471, '\p{^Block=		ENCLOSED_alphanumerics}', "");
Expect(0, 9471, '\P{Block=		ENCLOSED_alphanumerics}', "");
Expect(1, 9471, '\P{^Block=		ENCLOSED_alphanumerics}', "");
Expect(0, 9472, '\p{Block=		ENCLOSED_alphanumerics}', "");
Expect(1, 9472, '\p{^Block=		ENCLOSED_alphanumerics}', "");
Expect(1, 9472, '\P{Block=		ENCLOSED_alphanumerics}', "");
Expect(0, 9472, '\P{^Block=		ENCLOSED_alphanumerics}', "");
Error('\p{Blk:	:=__Enclosed_alphanum}');
Error('\P{Blk:	:=__Enclosed_alphanum}');
Expect(1, 9471, '\p{Blk=enclosedalphanum}', "");
Expect(0, 9471, '\p{^Blk=enclosedalphanum}', "");
Expect(0, 9471, '\P{Blk=enclosedalphanum}', "");
Expect(1, 9471, '\P{^Blk=enclosedalphanum}', "");
Expect(0, 9472, '\p{Blk=enclosedalphanum}', "");
Expect(1, 9472, '\p{^Blk=enclosedalphanum}', "");
Expect(1, 9472, '\P{Blk=enclosedalphanum}', "");
Expect(0, 9472, '\P{^Blk=enclosedalphanum}', "");
Expect(1, 9471, '\p{Blk=_enclosed_ALPHANUM}', "");
Expect(0, 9471, '\p{^Blk=_enclosed_ALPHANUM}', "");
Expect(0, 9471, '\P{Blk=_enclosed_ALPHANUM}', "");
Expect(1, 9471, '\P{^Blk=_enclosed_ALPHANUM}', "");
Expect(0, 9472, '\p{Blk=_enclosed_ALPHANUM}', "");
Expect(1, 9472, '\p{^Blk=_enclosed_ALPHANUM}', "");
Expect(1, 9472, '\P{Blk=_enclosed_ALPHANUM}', "");
Expect(0, 9472, '\P{^Blk=_enclosed_ALPHANUM}', "");
Error('\p{Is_Block=/a/ 	Enclosed_ALPHANUMERICS}');
Error('\P{Is_Block=/a/ 	Enclosed_ALPHANUMERICS}');
Expect(1, 9471, '\p{Is_Block=enclosedalphanumerics}', "");
Expect(0, 9471, '\p{^Is_Block=enclosedalphanumerics}', "");
Expect(0, 9471, '\P{Is_Block=enclosedalphanumerics}', "");
Expect(1, 9471, '\P{^Is_Block=enclosedalphanumerics}', "");
Expect(0, 9472, '\p{Is_Block=enclosedalphanumerics}', "");
Expect(1, 9472, '\p{^Is_Block=enclosedalphanumerics}', "");
Expect(1, 9472, '\P{Is_Block=enclosedalphanumerics}', "");
Expect(0, 9472, '\P{^Is_Block=enclosedalphanumerics}', "");
Expect(1, 9471, '\p{Is_Block= 	enclosed_ALPHANUMERICS}', "");
Expect(0, 9471, '\p{^Is_Block= 	enclosed_ALPHANUMERICS}', "");
Expect(0, 9471, '\P{Is_Block= 	enclosed_ALPHANUMERICS}', "");
Expect(1, 9471, '\P{^Is_Block= 	enclosed_ALPHANUMERICS}', "");
Expect(0, 9472, '\p{Is_Block= 	enclosed_ALPHANUMERICS}', "");
Expect(1, 9472, '\p{^Is_Block= 	enclosed_ALPHANUMERICS}', "");
Expect(1, 9472, '\P{Is_Block= 	enclosed_ALPHANUMERICS}', "");
Expect(0, 9472, '\P{^Is_Block= 	enclosed_ALPHANUMERICS}', "");
Error('\p{Is_Blk= /a/Enclosed_alphanum}');
Error('\P{Is_Blk= /a/Enclosed_alphanum}');
Expect(1, 9471, '\p{Is_Blk=enclosedalphanum}', "");
Expect(0, 9471, '\p{^Is_Blk=enclosedalphanum}', "");
Expect(0, 9471, '\P{Is_Blk=enclosedalphanum}', "");
Expect(1, 9471, '\P{^Is_Blk=enclosedalphanum}', "");
Expect(0, 9472, '\p{Is_Blk=enclosedalphanum}', "");
Expect(1, 9472, '\p{^Is_Blk=enclosedalphanum}', "");
Expect(1, 9472, '\P{Is_Blk=enclosedalphanum}', "");
Expect(0, 9472, '\P{^Is_Blk=enclosedalphanum}', "");
Expect(1, 9471, '\p{Is_Blk= -Enclosed_alphanum}', "");
Expect(0, 9471, '\p{^Is_Blk= -Enclosed_alphanum}', "");
Expect(0, 9471, '\P{Is_Blk= -Enclosed_alphanum}', "");
Expect(1, 9471, '\P{^Is_Blk= -Enclosed_alphanum}', "");
Expect(0, 9472, '\p{Is_Blk= -Enclosed_alphanum}', "");
Expect(1, 9472, '\p{^Is_Blk= -Enclosed_alphanum}', "");
Expect(1, 9472, '\P{Is_Blk= -Enclosed_alphanum}', "");
Expect(0, 9472, '\P{^Is_Blk= -Enclosed_alphanum}', "");
Error('\p{Block=/a/--Enclosed_Alphanumeric_Supplement}');
Error('\P{Block=/a/--Enclosed_Alphanumeric_Supplement}');
Expect(1, 127487, '\p{Block=enclosedalphanumericsupplement}', "");
Expect(0, 127487, '\p{^Block=enclosedalphanumericsupplement}', "");
Expect(0, 127487, '\P{Block=enclosedalphanumericsupplement}', "");
Expect(1, 127487, '\P{^Block=enclosedalphanumericsupplement}', "");
Expect(0, 127488, '\p{Block=enclosedalphanumericsupplement}', "");
Expect(1, 127488, '\p{^Block=enclosedalphanumericsupplement}', "");
Expect(1, 127488, '\P{Block=enclosedalphanumericsupplement}', "");
Expect(0, 127488, '\P{^Block=enclosedalphanumericsupplement}', "");
Expect(1, 127487, '\p{Block=-	Enclosed_alphanumeric_SUPPLEMENT}', "");
Expect(0, 127487, '\p{^Block=-	Enclosed_alphanumeric_SUPPLEMENT}', "");
Expect(0, 127487, '\P{Block=-	Enclosed_alphanumeric_SUPPLEMENT}', "");
Expect(1, 127487, '\P{^Block=-	Enclosed_alphanumeric_SUPPLEMENT}', "");
Expect(0, 127488, '\p{Block=-	Enclosed_alphanumeric_SUPPLEMENT}', "");
Expect(1, 127488, '\p{^Block=-	Enclosed_alphanumeric_SUPPLEMENT}', "");
Expect(1, 127488, '\P{Block=-	Enclosed_alphanumeric_SUPPLEMENT}', "");
Expect(0, 127488, '\P{^Block=-	Enclosed_alphanumeric_SUPPLEMENT}', "");
Error('\p{Blk=/a/	enclosed_ALPHANUM_Sup}');
Error('\P{Blk=/a/	enclosed_ALPHANUM_Sup}');
Expect(1, 127487, '\p{Blk=enclosedalphanumsup}', "");
Expect(0, 127487, '\p{^Blk=enclosedalphanumsup}', "");
Expect(0, 127487, '\P{Blk=enclosedalphanumsup}', "");
Expect(1, 127487, '\P{^Blk=enclosedalphanumsup}', "");
Expect(0, 127488, '\p{Blk=enclosedalphanumsup}', "");
Expect(1, 127488, '\p{^Blk=enclosedalphanumsup}', "");
Expect(1, 127488, '\P{Blk=enclosedalphanumsup}', "");
Expect(0, 127488, '\P{^Blk=enclosedalphanumsup}', "");
Expect(1, 127487, '\p{Blk=-_ENCLOSED_alphanum_Sup}', "");
Expect(0, 127487, '\p{^Blk=-_ENCLOSED_alphanum_Sup}', "");
Expect(0, 127487, '\P{Blk=-_ENCLOSED_alphanum_Sup}', "");
Expect(1, 127487, '\P{^Blk=-_ENCLOSED_alphanum_Sup}', "");
Expect(0, 127488, '\p{Blk=-_ENCLOSED_alphanum_Sup}', "");
Expect(1, 127488, '\p{^Blk=-_ENCLOSED_alphanum_Sup}', "");
Expect(1, 127488, '\P{Blk=-_ENCLOSED_alphanum_Sup}', "");
Expect(0, 127488, '\P{^Blk=-_ENCLOSED_alphanum_Sup}', "");
Error('\p{Is_Block=:=_	Enclosed_Alphanumeric_Supplement}');
Error('\P{Is_Block=:=_	Enclosed_Alphanumeric_Supplement}');
Expect(1, 127487, '\p{Is_Block=enclosedalphanumericsupplement}', "");
Expect(0, 127487, '\p{^Is_Block=enclosedalphanumericsupplement}', "");
Expect(0, 127487, '\P{Is_Block=enclosedalphanumericsupplement}', "");
Expect(1, 127487, '\P{^Is_Block=enclosedalphanumericsupplement}', "");
Expect(0, 127488, '\p{Is_Block=enclosedalphanumericsupplement}', "");
Expect(1, 127488, '\p{^Is_Block=enclosedalphanumericsupplement}', "");
Expect(1, 127488, '\P{Is_Block=enclosedalphanumericsupplement}', "");
Expect(0, 127488, '\P{^Is_Block=enclosedalphanumericsupplement}', "");
Expect(1, 127487, '\p{Is_Block=		Enclosed_Alphanumeric_supplement}', "");
Expect(0, 127487, '\p{^Is_Block=		Enclosed_Alphanumeric_supplement}', "");
Expect(0, 127487, '\P{Is_Block=		Enclosed_Alphanumeric_supplement}', "");
Expect(1, 127487, '\P{^Is_Block=		Enclosed_Alphanumeric_supplement}', "");
Expect(0, 127488, '\p{Is_Block=		Enclosed_Alphanumeric_supplement}', "");
Expect(1, 127488, '\p{^Is_Block=		Enclosed_Alphanumeric_supplement}', "");
Expect(1, 127488, '\P{Is_Block=		Enclosed_Alphanumeric_supplement}', "");
Expect(0, 127488, '\P{^Is_Block=		Enclosed_Alphanumeric_supplement}', "");
Error('\p{Is_Blk=	/a/Enclosed_ALPHANUM_sup}');
Error('\P{Is_Blk=	/a/Enclosed_ALPHANUM_sup}');
Expect(1, 127487, '\p{Is_Blk=enclosedalphanumsup}', "");
Expect(0, 127487, '\p{^Is_Blk=enclosedalphanumsup}', "");
Expect(0, 127487, '\P{Is_Blk=enclosedalphanumsup}', "");
Expect(1, 127487, '\P{^Is_Blk=enclosedalphanumsup}', "");
Expect(0, 127488, '\p{Is_Blk=enclosedalphanumsup}', "");
Expect(1, 127488, '\p{^Is_Blk=enclosedalphanumsup}', "");
Expect(1, 127488, '\P{Is_Blk=enclosedalphanumsup}', "");
Expect(0, 127488, '\P{^Is_Blk=enclosedalphanumsup}', "");
Expect(1, 127487, '\p{Is_Blk=-Enclosed_Alphanum_Sup}', "");
Expect(0, 127487, '\p{^Is_Blk=-Enclosed_Alphanum_Sup}', "");
Expect(0, 127487, '\P{Is_Blk=-Enclosed_Alphanum_Sup}', "");
Expect(1, 127487, '\P{^Is_Blk=-Enclosed_Alphanum_Sup}', "");
Expect(0, 127488, '\p{Is_Blk=-Enclosed_Alphanum_Sup}', "");
Expect(1, 127488, '\p{^Is_Blk=-Enclosed_Alphanum_Sup}', "");
Expect(1, 127488, '\P{Is_Blk=-Enclosed_Alphanum_Sup}', "");
Expect(0, 127488, '\P{^Is_Blk=-Enclosed_Alphanum_Sup}', "");
Error('\p{Block= /a/enclosed_CJK_Letters_AND_months}');
Error('\P{Block= /a/enclosed_CJK_Letters_AND_months}');
Expect(1, 13055, '\p{Block=enclosedcjklettersandmonths}', "");
Expect(0, 13055, '\p{^Block=enclosedcjklettersandmonths}', "");
Expect(0, 13055, '\P{Block=enclosedcjklettersandmonths}', "");
Expect(1, 13055, '\P{^Block=enclosedcjklettersandmonths}', "");
Expect(0, 13056, '\p{Block=enclosedcjklettersandmonths}', "");
Expect(1, 13056, '\p{^Block=enclosedcjklettersandmonths}', "");
Expect(1, 13056, '\P{Block=enclosedcjklettersandmonths}', "");
Expect(0, 13056, '\P{^Block=enclosedcjklettersandmonths}', "");
Expect(1, 13055, '\p{Block= Enclosed_cjk_Letters_AND_Months}', "");
Expect(0, 13055, '\p{^Block= Enclosed_cjk_Letters_AND_Months}', "");
Expect(0, 13055, '\P{Block= Enclosed_cjk_Letters_AND_Months}', "");
Expect(1, 13055, '\P{^Block= Enclosed_cjk_Letters_AND_Months}', "");
Expect(0, 13056, '\p{Block= Enclosed_cjk_Letters_AND_Months}', "");
Expect(1, 13056, '\p{^Block= Enclosed_cjk_Letters_AND_Months}', "");
Expect(1, 13056, '\P{Block= Enclosed_cjk_Letters_AND_Months}', "");
Expect(0, 13056, '\P{^Block= Enclosed_cjk_Letters_AND_Months}', "");
Error('\p{Blk=:= 	Enclosed_cjk}');
Error('\P{Blk=:= 	Enclosed_cjk}');
Expect(1, 13055, '\p{Blk=enclosedcjk}', "");
Expect(0, 13055, '\p{^Blk=enclosedcjk}', "");
Expect(0, 13055, '\P{Blk=enclosedcjk}', "");
Expect(1, 13055, '\P{^Blk=enclosedcjk}', "");
Expect(0, 13056, '\p{Blk=enclosedcjk}', "");
Expect(1, 13056, '\p{^Blk=enclosedcjk}', "");
Expect(1, 13056, '\P{Blk=enclosedcjk}', "");
Expect(0, 13056, '\P{^Blk=enclosedcjk}', "");
Expect(1, 13055, '\p{Blk=_-enclosed_CJK}', "");
Expect(0, 13055, '\p{^Blk=_-enclosed_CJK}', "");
Expect(0, 13055, '\P{Blk=_-enclosed_CJK}', "");
Expect(1, 13055, '\P{^Blk=_-enclosed_CJK}', "");
Expect(0, 13056, '\p{Blk=_-enclosed_CJK}', "");
Expect(1, 13056, '\p{^Blk=_-enclosed_CJK}', "");
Expect(1, 13056, '\P{Blk=_-enclosed_CJK}', "");
Expect(0, 13056, '\P{^Blk=_-enclosed_CJK}', "");
Error('\p{Is_Block=-:=Enclosed_CJK_letters_AND_Months}');
Error('\P{Is_Block=-:=Enclosed_CJK_letters_AND_Months}');
Expect(1, 13055, '\p{Is_Block=enclosedcjklettersandmonths}', "");
Expect(0, 13055, '\p{^Is_Block=enclosedcjklettersandmonths}', "");
Expect(0, 13055, '\P{Is_Block=enclosedcjklettersandmonths}', "");
Expect(1, 13055, '\P{^Is_Block=enclosedcjklettersandmonths}', "");
Expect(0, 13056, '\p{Is_Block=enclosedcjklettersandmonths}', "");
Expect(1, 13056, '\p{^Is_Block=enclosedcjklettersandmonths}', "");
Expect(1, 13056, '\P{Is_Block=enclosedcjklettersandmonths}', "");
Expect(0, 13056, '\P{^Is_Block=enclosedcjklettersandmonths}', "");
Expect(1, 13055, '\p{Is_Block= _ENCLOSED_CJK_letters_and_MONTHS}', "");
Expect(0, 13055, '\p{^Is_Block= _ENCLOSED_CJK_letters_and_MONTHS}', "");
Expect(0, 13055, '\P{Is_Block= _ENCLOSED_CJK_letters_and_MONTHS}', "");
Expect(1, 13055, '\P{^Is_Block= _ENCLOSED_CJK_letters_and_MONTHS}', "");
Expect(0, 13056, '\p{Is_Block= _ENCLOSED_CJK_letters_and_MONTHS}', "");
Expect(1, 13056, '\p{^Is_Block= _ENCLOSED_CJK_letters_and_MONTHS}', "");
Expect(1, 13056, '\P{Is_Block= _ENCLOSED_CJK_letters_and_MONTHS}', "");
Expect(0, 13056, '\P{^Is_Block= _ENCLOSED_CJK_letters_and_MONTHS}', "");
Error('\p{Is_Blk=_Enclosed_CJK:=}');
Error('\P{Is_Blk=_Enclosed_CJK:=}');
Expect(1, 13055, '\p{Is_Blk=enclosedcjk}', "");
Expect(0, 13055, '\p{^Is_Blk=enclosedcjk}', "");
Expect(0, 13055, '\P{Is_Blk=enclosedcjk}', "");
Expect(1, 13055, '\P{^Is_Blk=enclosedcjk}', "");
Expect(0, 13056, '\p{Is_Blk=enclosedcjk}', "");
Expect(1, 13056, '\p{^Is_Blk=enclosedcjk}', "");
Expect(1, 13056, '\P{Is_Blk=enclosedcjk}', "");
Expect(0, 13056, '\P{^Is_Blk=enclosedcjk}', "");
Expect(1, 13055, '\p{Is_Blk=	 Enclosed_CJK}', "");
Expect(0, 13055, '\p{^Is_Blk=	 Enclosed_CJK}', "");
Expect(0, 13055, '\P{Is_Blk=	 Enclosed_CJK}', "");
Expect(1, 13055, '\P{^Is_Blk=	 Enclosed_CJK}', "");
Expect(0, 13056, '\p{Is_Blk=	 Enclosed_CJK}', "");
Expect(1, 13056, '\p{^Is_Blk=	 Enclosed_CJK}', "");
Expect(1, 13056, '\P{Is_Blk=	 Enclosed_CJK}', "");
Expect(0, 13056, '\P{^Is_Blk=	 Enclosed_CJK}', "");
Error('\p{Block= ENCLOSED_Ideographic_Supplement:=}');
Error('\P{Block= ENCLOSED_Ideographic_Supplement:=}');
Expect(1, 127743, '\p{Block=enclosedideographicsupplement}', "");
Expect(0, 127743, '\p{^Block=enclosedideographicsupplement}', "");
Expect(0, 127743, '\P{Block=enclosedideographicsupplement}', "");
Expect(1, 127743, '\P{^Block=enclosedideographicsupplement}', "");
Expect(0, 127744, '\p{Block=enclosedideographicsupplement}', "");
Expect(1, 127744, '\p{^Block=enclosedideographicsupplement}', "");
Expect(1, 127744, '\P{Block=enclosedideographicsupplement}', "");
Expect(0, 127744, '\P{^Block=enclosedideographicsupplement}', "");
Expect(1, 127743, '\p{Block=_-ENCLOSED_IDEOGRAPHIC_Supplement}', "");
Expect(0, 127743, '\p{^Block=_-ENCLOSED_IDEOGRAPHIC_Supplement}', "");
Expect(0, 127743, '\P{Block=_-ENCLOSED_IDEOGRAPHIC_Supplement}', "");
Expect(1, 127743, '\P{^Block=_-ENCLOSED_IDEOGRAPHIC_Supplement}', "");
Expect(0, 127744, '\p{Block=_-ENCLOSED_IDEOGRAPHIC_Supplement}', "");
Expect(1, 127744, '\p{^Block=_-ENCLOSED_IDEOGRAPHIC_Supplement}', "");
Expect(1, 127744, '\P{Block=_-ENCLOSED_IDEOGRAPHIC_Supplement}', "");
Expect(0, 127744, '\P{^Block=_-ENCLOSED_IDEOGRAPHIC_Supplement}', "");
Error('\p{Blk: /a/	 Enclosed_ideographic_sup}');
Error('\P{Blk: /a/	 Enclosed_ideographic_sup}');
Expect(1, 127743, '\p{Blk=enclosedideographicsup}', "");
Expect(0, 127743, '\p{^Blk=enclosedideographicsup}', "");
Expect(0, 127743, '\P{Blk=enclosedideographicsup}', "");
Expect(1, 127743, '\P{^Blk=enclosedideographicsup}', "");
Expect(0, 127744, '\p{Blk=enclosedideographicsup}', "");
Expect(1, 127744, '\p{^Blk=enclosedideographicsup}', "");
Expect(1, 127744, '\P{Blk=enclosedideographicsup}', "");
Expect(0, 127744, '\P{^Blk=enclosedideographicsup}', "");
Expect(1, 127743, '\p{Blk= ENCLOSED_IDEOGRAPHIC_SUP}', "");
Expect(0, 127743, '\p{^Blk= ENCLOSED_IDEOGRAPHIC_SUP}', "");
Expect(0, 127743, '\P{Blk= ENCLOSED_IDEOGRAPHIC_SUP}', "");
Expect(1, 127743, '\P{^Blk= ENCLOSED_IDEOGRAPHIC_SUP}', "");
Expect(0, 127744, '\p{Blk= ENCLOSED_IDEOGRAPHIC_SUP}', "");
Expect(1, 127744, '\p{^Blk= ENCLOSED_IDEOGRAPHIC_SUP}', "");
Expect(1, 127744, '\P{Blk= ENCLOSED_IDEOGRAPHIC_SUP}', "");
Expect(0, 127744, '\P{^Blk= ENCLOSED_IDEOGRAPHIC_SUP}', "");
Error('\p{Is_Block=_-Enclosed_Ideographic_SUPPLEMENT:=}');
Error('\P{Is_Block=_-Enclosed_Ideographic_SUPPLEMENT:=}');
Expect(1, 127743, '\p{Is_Block=enclosedideographicsupplement}', "");
Expect(0, 127743, '\p{^Is_Block=enclosedideographicsupplement}', "");
Expect(0, 127743, '\P{Is_Block=enclosedideographicsupplement}', "");
Expect(1, 127743, '\P{^Is_Block=enclosedideographicsupplement}', "");
Expect(0, 127744, '\p{Is_Block=enclosedideographicsupplement}', "");
Expect(1, 127744, '\p{^Is_Block=enclosedideographicsupplement}', "");
Expect(1, 127744, '\P{Is_Block=enclosedideographicsupplement}', "");
Expect(0, 127744, '\P{^Is_Block=enclosedideographicsupplement}', "");
Expect(1, 127743, '\p{Is_Block=_Enclosed_IDEOGRAPHIC_supplement}', "");
Expect(0, 127743, '\p{^Is_Block=_Enclosed_IDEOGRAPHIC_supplement}', "");
Expect(0, 127743, '\P{Is_Block=_Enclosed_IDEOGRAPHIC_supplement}', "");
Expect(1, 127743, '\P{^Is_Block=_Enclosed_IDEOGRAPHIC_supplement}', "");
Expect(0, 127744, '\p{Is_Block=_Enclosed_IDEOGRAPHIC_supplement}', "");
Expect(1, 127744, '\p{^Is_Block=_Enclosed_IDEOGRAPHIC_supplement}', "");
Expect(1, 127744, '\P{Is_Block=_Enclosed_IDEOGRAPHIC_supplement}', "");
Expect(0, 127744, '\P{^Is_Block=_Enclosed_IDEOGRAPHIC_supplement}', "");
Error('\p{Is_Blk=	:=Enclosed_Ideographic_SUP}');
Error('\P{Is_Blk=	:=Enclosed_Ideographic_SUP}');
Expect(1, 127743, '\p{Is_Blk=enclosedideographicsup}', "");
Expect(0, 127743, '\p{^Is_Blk=enclosedideographicsup}', "");
Expect(0, 127743, '\P{Is_Blk=enclosedideographicsup}', "");
Expect(1, 127743, '\P{^Is_Blk=enclosedideographicsup}', "");
Expect(0, 127744, '\p{Is_Blk=enclosedideographicsup}', "");
Expect(1, 127744, '\p{^Is_Blk=enclosedideographicsup}', "");
Expect(1, 127744, '\P{Is_Blk=enclosedideographicsup}', "");
Expect(0, 127744, '\P{^Is_Blk=enclosedideographicsup}', "");
Expect(1, 127743, '\p{Is_Blk=-enclosed_Ideographic_Sup}', "");
Expect(0, 127743, '\p{^Is_Blk=-enclosed_Ideographic_Sup}', "");
Expect(0, 127743, '\P{Is_Blk=-enclosed_Ideographic_Sup}', "");
Expect(1, 127743, '\P{^Is_Blk=-enclosed_Ideographic_Sup}', "");
Expect(0, 127744, '\p{Is_Blk=-enclosed_Ideographic_Sup}', "");
Expect(1, 127744, '\p{^Is_Blk=-enclosed_Ideographic_Sup}', "");
Expect(1, 127744, '\P{Is_Blk=-enclosed_Ideographic_Sup}', "");
Expect(0, 127744, '\P{^Is_Blk=-enclosed_Ideographic_Sup}', "");
Error('\p{Block=/a/ethiopic}');
Error('\P{Block=/a/ethiopic}');
Expect(1, 4991, '\p{Block=ethiopic}', "");
Expect(0, 4991, '\p{^Block=ethiopic}', "");
Expect(0, 4991, '\P{Block=ethiopic}', "");
Expect(1, 4991, '\P{^Block=ethiopic}', "");
Expect(0, 4992, '\p{Block=ethiopic}', "");
Expect(1, 4992, '\p{^Block=ethiopic}', "");
Expect(1, 4992, '\P{Block=ethiopic}', "");
Expect(0, 4992, '\P{^Block=ethiopic}', "");
Expect(1, 4991, '\p{Block=_	ethiopic}', "");
Expect(0, 4991, '\p{^Block=_	ethiopic}', "");
Expect(0, 4991, '\P{Block=_	ethiopic}', "");
Expect(1, 4991, '\P{^Block=_	ethiopic}', "");
Expect(0, 4992, '\p{Block=_	ethiopic}', "");
Expect(1, 4992, '\p{^Block=_	ethiopic}', "");
Expect(1, 4992, '\P{Block=_	ethiopic}', "");
Expect(0, 4992, '\P{^Block=_	ethiopic}', "");
Error('\p{Blk=	 Ethiopic/a/}');
Error('\P{Blk=	 Ethiopic/a/}');
Expect(1, 4991, '\p{Blk=ethiopic}', "");
Expect(0, 4991, '\p{^Blk=ethiopic}', "");
Expect(0, 4991, '\P{Blk=ethiopic}', "");
Expect(1, 4991, '\P{^Blk=ethiopic}', "");
Expect(0, 4992, '\p{Blk=ethiopic}', "");
Expect(1, 4992, '\p{^Blk=ethiopic}', "");
Expect(1, 4992, '\P{Blk=ethiopic}', "");
Expect(0, 4992, '\P{^Blk=ethiopic}', "");
Expect(1, 4991, '\p{Blk=_ Ethiopic}', "");
Expect(0, 4991, '\p{^Blk=_ Ethiopic}', "");
Expect(0, 4991, '\P{Blk=_ Ethiopic}', "");
Expect(1, 4991, '\P{^Blk=_ Ethiopic}', "");
Expect(0, 4992, '\p{Blk=_ Ethiopic}', "");
Expect(1, 4992, '\p{^Blk=_ Ethiopic}', "");
Expect(1, 4992, '\P{Blk=_ Ethiopic}', "");
Expect(0, 4992, '\P{^Blk=_ Ethiopic}', "");
Error('\p{Is_Block=/a/-_ETHIOPIC}');
Error('\P{Is_Block=/a/-_ETHIOPIC}');
Expect(1, 4991, '\p{Is_Block=ethiopic}', "");
Expect(0, 4991, '\p{^Is_Block=ethiopic}', "");
Expect(0, 4991, '\P{Is_Block=ethiopic}', "");
Expect(1, 4991, '\P{^Is_Block=ethiopic}', "");
Expect(0, 4992, '\p{Is_Block=ethiopic}', "");
Expect(1, 4992, '\p{^Is_Block=ethiopic}', "");
Expect(1, 4992, '\P{Is_Block=ethiopic}', "");
Expect(0, 4992, '\P{^Is_Block=ethiopic}', "");
Expect(1, 4991, '\p{Is_Block=_Ethiopic}', "");
Expect(0, 4991, '\p{^Is_Block=_Ethiopic}', "");
Expect(0, 4991, '\P{Is_Block=_Ethiopic}', "");
Expect(1, 4991, '\P{^Is_Block=_Ethiopic}', "");
Expect(0, 4992, '\p{Is_Block=_Ethiopic}', "");
Expect(1, 4992, '\p{^Is_Block=_Ethiopic}', "");
Expect(1, 4992, '\P{Is_Block=_Ethiopic}', "");
Expect(0, 4992, '\P{^Is_Block=_Ethiopic}', "");
Error('\p{Is_Blk=-/a/ETHIOPIC}');
Error('\P{Is_Blk=-/a/ETHIOPIC}');
Expect(1, 4991, '\p{Is_Blk=ethiopic}', "");
Expect(0, 4991, '\p{^Is_Blk=ethiopic}', "");
Expect(0, 4991, '\P{Is_Blk=ethiopic}', "");
Expect(1, 4991, '\P{^Is_Blk=ethiopic}', "");
Expect(0, 4992, '\p{Is_Blk=ethiopic}', "");
Expect(1, 4992, '\p{^Is_Blk=ethiopic}', "");
Expect(1, 4992, '\P{Is_Blk=ethiopic}', "");
Expect(0, 4992, '\P{^Is_Blk=ethiopic}', "");
Expect(1, 4991, '\p{Is_Blk: 	Ethiopic}', "");
Expect(0, 4991, '\p{^Is_Blk: 	Ethiopic}', "");
Expect(0, 4991, '\P{Is_Blk: 	Ethiopic}', "");
Expect(1, 4991, '\P{^Is_Blk: 	Ethiopic}', "");
Expect(0, 4992, '\p{Is_Blk: 	Ethiopic}', "");
Expect(1, 4992, '\p{^Is_Blk: 	Ethiopic}', "");
Expect(1, 4992, '\P{Is_Blk: 	Ethiopic}', "");
Expect(0, 4992, '\P{^Is_Blk: 	Ethiopic}', "");
Error('\p{Block=/a/-_Ethiopic_Extended}');
Error('\P{Block=/a/-_Ethiopic_Extended}');
Expect(1, 11743, '\p{Block=ethiopicextended}', "");
Expect(0, 11743, '\p{^Block=ethiopicextended}', "");
Expect(0, 11743, '\P{Block=ethiopicextended}', "");
Expect(1, 11743, '\P{^Block=ethiopicextended}', "");
Expect(0, 11744, '\p{Block=ethiopicextended}', "");
Expect(1, 11744, '\p{^Block=ethiopicextended}', "");
Expect(1, 11744, '\P{Block=ethiopicextended}', "");
Expect(0, 11744, '\P{^Block=ethiopicextended}', "");
Expect(1, 11743, '\p{Block=ethiopic_Extended}', "");
Expect(0, 11743, '\p{^Block=ethiopic_Extended}', "");
Expect(0, 11743, '\P{Block=ethiopic_Extended}', "");
Expect(1, 11743, '\P{^Block=ethiopic_Extended}', "");
Expect(0, 11744, '\p{Block=ethiopic_Extended}', "");
Expect(1, 11744, '\p{^Block=ethiopic_Extended}', "");
Expect(1, 11744, '\P{Block=ethiopic_Extended}', "");
Expect(0, 11744, '\P{^Block=ethiopic_Extended}', "");
Error('\p{Blk=__ethiopic_Ext:=}');
Error('\P{Blk=__ethiopic_Ext:=}');
Expect(1, 11743, '\p{Blk=ethiopicext}', "");
Expect(0, 11743, '\p{^Blk=ethiopicext}', "");
Expect(0, 11743, '\P{Blk=ethiopicext}', "");
Expect(1, 11743, '\P{^Blk=ethiopicext}', "");
Expect(0, 11744, '\p{Blk=ethiopicext}', "");
Expect(1, 11744, '\p{^Blk=ethiopicext}', "");
Expect(1, 11744, '\P{Blk=ethiopicext}', "");
Expect(0, 11744, '\P{^Blk=ethiopicext}', "");
Expect(1, 11743, '\p{Blk=  ethiopic_EXT}', "");
Expect(0, 11743, '\p{^Blk=  ethiopic_EXT}', "");
Expect(0, 11743, '\P{Blk=  ethiopic_EXT}', "");
Expect(1, 11743, '\P{^Blk=  ethiopic_EXT}', "");
Expect(0, 11744, '\p{Blk=  ethiopic_EXT}', "");
Expect(1, 11744, '\p{^Blk=  ethiopic_EXT}', "");
Expect(1, 11744, '\P{Blk=  ethiopic_EXT}', "");
Expect(0, 11744, '\P{^Blk=  ethiopic_EXT}', "");
Error('\p{Is_Block=:=-	Ethiopic_EXTENDED}');
Error('\P{Is_Block=:=-	Ethiopic_EXTENDED}');
Expect(1, 11743, '\p{Is_Block=ethiopicextended}', "");
Expect(0, 11743, '\p{^Is_Block=ethiopicextended}', "");
Expect(0, 11743, '\P{Is_Block=ethiopicextended}', "");
Expect(1, 11743, '\P{^Is_Block=ethiopicextended}', "");
Expect(0, 11744, '\p{Is_Block=ethiopicextended}', "");
Expect(1, 11744, '\p{^Is_Block=ethiopicextended}', "");
Expect(1, 11744, '\P{Is_Block=ethiopicextended}', "");
Expect(0, 11744, '\P{^Is_Block=ethiopicextended}', "");
Expect(1, 11743, '\p{Is_Block=_Ethiopic_extended}', "");
Expect(0, 11743, '\p{^Is_Block=_Ethiopic_extended}', "");
Expect(0, 11743, '\P{Is_Block=_Ethiopic_extended}', "");
Expect(1, 11743, '\P{^Is_Block=_Ethiopic_extended}', "");
Expect(0, 11744, '\p{Is_Block=_Ethiopic_extended}', "");
Expect(1, 11744, '\p{^Is_Block=_Ethiopic_extended}', "");
Expect(1, 11744, '\P{Is_Block=_Ethiopic_extended}', "");
Expect(0, 11744, '\P{^Is_Block=_Ethiopic_extended}', "");
Error('\p{Is_Blk=	 ethiopic_Ext/a/}');
Error('\P{Is_Blk=	 ethiopic_Ext/a/}');
Expect(1, 11743, '\p{Is_Blk=ethiopicext}', "");
Expect(0, 11743, '\p{^Is_Blk=ethiopicext}', "");
Expect(0, 11743, '\P{Is_Blk=ethiopicext}', "");
Expect(1, 11743, '\P{^Is_Blk=ethiopicext}', "");
Expect(0, 11744, '\p{Is_Blk=ethiopicext}', "");
Expect(1, 11744, '\p{^Is_Blk=ethiopicext}', "");
Expect(1, 11744, '\P{Is_Blk=ethiopicext}', "");
Expect(0, 11744, '\P{^Is_Blk=ethiopicext}', "");
Expect(1, 11743, '\p{Is_Blk=	ethiopic_ext}', "");
Expect(0, 11743, '\p{^Is_Blk=	ethiopic_ext}', "");
Expect(0, 11743, '\P{Is_Blk=	ethiopic_ext}', "");
Expect(1, 11743, '\P{^Is_Blk=	ethiopic_ext}', "");
Expect(0, 11744, '\p{Is_Blk=	ethiopic_ext}', "");
Expect(1, 11744, '\p{^Is_Blk=	ethiopic_ext}', "");
Expect(1, 11744, '\P{Is_Blk=	ethiopic_ext}', "");
Expect(0, 11744, '\P{^Is_Blk=	ethiopic_ext}', "");
Error('\p{Block:- ETHIOPIC_EXTENDED_a/a/}');
Error('\P{Block:- ETHIOPIC_EXTENDED_a/a/}');
Expect(1, 43823, '\p{Block=ethiopicextendeda}', "");
Expect(0, 43823, '\p{^Block=ethiopicextendeda}', "");
Expect(0, 43823, '\P{Block=ethiopicextendeda}', "");
Expect(1, 43823, '\P{^Block=ethiopicextendeda}', "");
Expect(0, 43824, '\p{Block=ethiopicextendeda}', "");
Expect(1, 43824, '\p{^Block=ethiopicextendeda}', "");
Expect(1, 43824, '\P{Block=ethiopicextendeda}', "");
Expect(0, 43824, '\P{^Block=ethiopicextendeda}', "");
Expect(1, 43823, '\p{Block=-Ethiopic_extended_A}', "");
Expect(0, 43823, '\p{^Block=-Ethiopic_extended_A}', "");
Expect(0, 43823, '\P{Block=-Ethiopic_extended_A}', "");
Expect(1, 43823, '\P{^Block=-Ethiopic_extended_A}', "");
Expect(0, 43824, '\p{Block=-Ethiopic_extended_A}', "");
Expect(1, 43824, '\p{^Block=-Ethiopic_extended_A}', "");
Expect(1, 43824, '\P{Block=-Ethiopic_extended_A}', "");
Expect(0, 43824, '\P{^Block=-Ethiopic_extended_A}', "");
Error('\p{Blk=:=-ETHIOPIC_Ext_a}');
Error('\P{Blk=:=-ETHIOPIC_Ext_a}');
Expect(1, 43823, '\p{Blk=ethiopicexta}', "");
Expect(0, 43823, '\p{^Blk=ethiopicexta}', "");
Expect(0, 43823, '\P{Blk=ethiopicexta}', "");
Expect(1, 43823, '\P{^Blk=ethiopicexta}', "");
Expect(0, 43824, '\p{Blk=ethiopicexta}', "");
Expect(1, 43824, '\p{^Blk=ethiopicexta}', "");
Expect(1, 43824, '\P{Blk=ethiopicexta}', "");
Expect(0, 43824, '\P{^Blk=ethiopicexta}', "");
Expect(1, 43823, '\p{Blk= -ethiopic_Ext_a}', "");
Expect(0, 43823, '\p{^Blk= -ethiopic_Ext_a}', "");
Expect(0, 43823, '\P{Blk= -ethiopic_Ext_a}', "");
Expect(1, 43823, '\P{^Blk= -ethiopic_Ext_a}', "");
Expect(0, 43824, '\p{Blk= -ethiopic_Ext_a}', "");
Expect(1, 43824, '\p{^Blk= -ethiopic_Ext_a}', "");
Expect(1, 43824, '\P{Blk= -ethiopic_Ext_a}', "");
Expect(0, 43824, '\P{^Blk= -ethiopic_Ext_a}', "");
Error('\p{Is_Block=:=		Ethiopic_Extended_A}');
Error('\P{Is_Block=:=		Ethiopic_Extended_A}');
Expect(1, 43823, '\p{Is_Block=ethiopicextendeda}', "");
Expect(0, 43823, '\p{^Is_Block=ethiopicextendeda}', "");
Expect(0, 43823, '\P{Is_Block=ethiopicextendeda}', "");
Expect(1, 43823, '\P{^Is_Block=ethiopicextendeda}', "");
Expect(0, 43824, '\p{Is_Block=ethiopicextendeda}', "");
Expect(1, 43824, '\p{^Is_Block=ethiopicextendeda}', "");
Expect(1, 43824, '\P{Is_Block=ethiopicextendeda}', "");
Expect(0, 43824, '\P{^Is_Block=ethiopicextendeda}', "");
Expect(1, 43823, '\p{Is_Block: 	_Ethiopic_extended_a}', "");
Expect(0, 43823, '\p{^Is_Block: 	_Ethiopic_extended_a}', "");
Expect(0, 43823, '\P{Is_Block: 	_Ethiopic_extended_a}', "");
Expect(1, 43823, '\P{^Is_Block: 	_Ethiopic_extended_a}', "");
Expect(0, 43824, '\p{Is_Block: 	_Ethiopic_extended_a}', "");
Expect(1, 43824, '\p{^Is_Block: 	_Ethiopic_extended_a}', "");
Expect(1, 43824, '\P{Is_Block: 	_Ethiopic_extended_a}', "");
Expect(0, 43824, '\P{^Is_Block: 	_Ethiopic_extended_a}', "");
Error('\p{Is_Blk:  Ethiopic_Ext_A:=}');
Error('\P{Is_Blk:  Ethiopic_Ext_A:=}');
Expect(1, 43823, '\p{Is_Blk=ethiopicexta}', "");
Expect(0, 43823, '\p{^Is_Blk=ethiopicexta}', "");
Expect(0, 43823, '\P{Is_Blk=ethiopicexta}', "");
Expect(1, 43823, '\P{^Is_Blk=ethiopicexta}', "");
Expect(0, 43824, '\p{Is_Blk=ethiopicexta}', "");
Expect(1, 43824, '\p{^Is_Blk=ethiopicexta}', "");
Expect(1, 43824, '\P{Is_Blk=ethiopicexta}', "");
Expect(0, 43824, '\P{^Is_Blk=ethiopicexta}', "");
Expect(1, 43823, '\p{Is_Blk=_ Ethiopic_Ext_A}', "");
Expect(0, 43823, '\p{^Is_Blk=_ Ethiopic_Ext_A}', "");
Expect(0, 43823, '\P{Is_Blk=_ Ethiopic_Ext_A}', "");
Expect(1, 43823, '\P{^Is_Blk=_ Ethiopic_Ext_A}', "");
Expect(0, 43824, '\p{Is_Blk=_ Ethiopic_Ext_A}', "");
Expect(1, 43824, '\p{^Is_Blk=_ Ethiopic_Ext_A}', "");
Expect(1, 43824, '\P{Is_Blk=_ Ethiopic_Ext_A}', "");
Expect(0, 43824, '\P{^Is_Blk=_ Ethiopic_Ext_A}', "");
Error('\p{Block=_:=Ethiopic_SUPPLEMENT}');
Error('\P{Block=_:=Ethiopic_SUPPLEMENT}');
Expect(1, 5023, '\p{Block=ethiopicsupplement}', "");
Expect(0, 5023, '\p{^Block=ethiopicsupplement}', "");
Expect(0, 5023, '\P{Block=ethiopicsupplement}', "");
Expect(1, 5023, '\P{^Block=ethiopicsupplement}', "");
Expect(0, 5024, '\p{Block=ethiopicsupplement}', "");
Expect(1, 5024, '\p{^Block=ethiopicsupplement}', "");
Expect(1, 5024, '\P{Block=ethiopicsupplement}', "");
Expect(0, 5024, '\P{^Block=ethiopicsupplement}', "");
Expect(1, 5023, '\p{Block=_	Ethiopic_SUPPLEMENT}', "");
Expect(0, 5023, '\p{^Block=_	Ethiopic_SUPPLEMENT}', "");
Expect(0, 5023, '\P{Block=_	Ethiopic_SUPPLEMENT}', "");
Expect(1, 5023, '\P{^Block=_	Ethiopic_SUPPLEMENT}', "");
Expect(0, 5024, '\p{Block=_	Ethiopic_SUPPLEMENT}', "");
Expect(1, 5024, '\p{^Block=_	Ethiopic_SUPPLEMENT}', "");
Expect(1, 5024, '\P{Block=_	Ethiopic_SUPPLEMENT}', "");
Expect(0, 5024, '\P{^Block=_	Ethiopic_SUPPLEMENT}', "");
Error('\p{Blk=/a/ Ethiopic_sup}');
Error('\P{Blk=/a/ Ethiopic_sup}');
Expect(1, 5023, '\p{Blk=ethiopicsup}', "");
Expect(0, 5023, '\p{^Blk=ethiopicsup}', "");
Expect(0, 5023, '\P{Blk=ethiopicsup}', "");
Expect(1, 5023, '\P{^Blk=ethiopicsup}', "");
Expect(0, 5024, '\p{Blk=ethiopicsup}', "");
Expect(1, 5024, '\p{^Blk=ethiopicsup}', "");
Expect(1, 5024, '\P{Blk=ethiopicsup}', "");
Expect(0, 5024, '\P{^Blk=ethiopicsup}', "");
Expect(1, 5023, '\p{Blk= ETHIOPIC_Sup}', "");
Expect(0, 5023, '\p{^Blk= ETHIOPIC_Sup}', "");
Expect(0, 5023, '\P{Blk= ETHIOPIC_Sup}', "");
Expect(1, 5023, '\P{^Blk= ETHIOPIC_Sup}', "");
Expect(0, 5024, '\p{Blk= ETHIOPIC_Sup}', "");
Expect(1, 5024, '\p{^Blk= ETHIOPIC_Sup}', "");
Expect(1, 5024, '\P{Blk= ETHIOPIC_Sup}', "");
Expect(0, 5024, '\P{^Blk= ETHIOPIC_Sup}', "");
Error('\p{Is_Block=	Ethiopic_Supplement:=}');
Error('\P{Is_Block=	Ethiopic_Supplement:=}');
Expect(1, 5023, '\p{Is_Block=ethiopicsupplement}', "");
Expect(0, 5023, '\p{^Is_Block=ethiopicsupplement}', "");
Expect(0, 5023, '\P{Is_Block=ethiopicsupplement}', "");
Expect(1, 5023, '\P{^Is_Block=ethiopicsupplement}', "");
Expect(0, 5024, '\p{Is_Block=ethiopicsupplement}', "");
Expect(1, 5024, '\p{^Is_Block=ethiopicsupplement}', "");
Expect(1, 5024, '\P{Is_Block=ethiopicsupplement}', "");
Expect(0, 5024, '\P{^Is_Block=ethiopicsupplement}', "");
Expect(1, 5023, '\p{Is_Block=_	Ethiopic_Supplement}', "");
Expect(0, 5023, '\p{^Is_Block=_	Ethiopic_Supplement}', "");
Expect(0, 5023, '\P{Is_Block=_	Ethiopic_Supplement}', "");
Expect(1, 5023, '\P{^Is_Block=_	Ethiopic_Supplement}', "");
Expect(0, 5024, '\p{Is_Block=_	Ethiopic_Supplement}', "");
Expect(1, 5024, '\p{^Is_Block=_	Ethiopic_Supplement}', "");
Expect(1, 5024, '\P{Is_Block=_	Ethiopic_Supplement}', "");
Expect(0, 5024, '\P{^Is_Block=_	Ethiopic_Supplement}', "");
Error('\p{Is_Blk=/a/_	Ethiopic_Sup}');
Error('\P{Is_Blk=/a/_	Ethiopic_Sup}');
Expect(1, 5023, '\p{Is_Blk=ethiopicsup}', "");
Expect(0, 5023, '\p{^Is_Blk=ethiopicsup}', "");
Expect(0, 5023, '\P{Is_Blk=ethiopicsup}', "");
Expect(1, 5023, '\P{^Is_Blk=ethiopicsup}', "");
Expect(0, 5024, '\p{Is_Blk=ethiopicsup}', "");
Expect(1, 5024, '\p{^Is_Blk=ethiopicsup}', "");
Expect(1, 5024, '\P{Is_Blk=ethiopicsup}', "");
Expect(0, 5024, '\P{^Is_Blk=ethiopicsup}', "");
Expect(1, 5023, '\p{Is_Blk=-ethiopic_Sup}', "");
Expect(0, 5023, '\p{^Is_Blk=-ethiopic_Sup}', "");
Expect(0, 5023, '\P{Is_Blk=-ethiopic_Sup}', "");
Expect(1, 5023, '\P{^Is_Blk=-ethiopic_Sup}', "");
Expect(0, 5024, '\p{Is_Blk=-ethiopic_Sup}', "");
Expect(1, 5024, '\p{^Is_Blk=-ethiopic_Sup}', "");
Expect(1, 5024, '\P{Is_Blk=-ethiopic_Sup}', "");
Expect(0, 5024, '\P{^Is_Blk=-ethiopic_Sup}', "");
Error('\p{Block=/a/--Geometric_shapes}');
Error('\P{Block=/a/--Geometric_shapes}');
Expect(1, 9727, '\p{Block=geometricshapes}', "");
Expect(0, 9727, '\p{^Block=geometricshapes}', "");
Expect(0, 9727, '\P{Block=geometricshapes}', "");
Expect(1, 9727, '\P{^Block=geometricshapes}', "");
Expect(0, 9728, '\p{Block=geometricshapes}', "");
Expect(1, 9728, '\p{^Block=geometricshapes}', "");
Expect(1, 9728, '\P{Block=geometricshapes}', "");
Expect(0, 9728, '\P{^Block=geometricshapes}', "");
Expect(1, 9727, '\p{Block=-geometric_SHAPES}', "");
Expect(0, 9727, '\p{^Block=-geometric_SHAPES}', "");
Expect(0, 9727, '\P{Block=-geometric_SHAPES}', "");
Expect(1, 9727, '\P{^Block=-geometric_SHAPES}', "");
Expect(0, 9728, '\p{Block=-geometric_SHAPES}', "");
Expect(1, 9728, '\p{^Block=-geometric_SHAPES}', "");
Expect(1, 9728, '\P{Block=-geometric_SHAPES}', "");
Expect(0, 9728, '\P{^Block=-geometric_SHAPES}', "");
Error('\p{Blk:    _GEOMETRIC_shapes:=}');
Error('\P{Blk:    _GEOMETRIC_shapes:=}');
Expect(1, 9727, '\p{Blk=geometricshapes}', "");
Expect(0, 9727, '\p{^Blk=geometricshapes}', "");
Expect(0, 9727, '\P{Blk=geometricshapes}', "");
Expect(1, 9727, '\P{^Blk=geometricshapes}', "");
Expect(0, 9728, '\p{Blk=geometricshapes}', "");
Expect(1, 9728, '\p{^Blk=geometricshapes}', "");
Expect(1, 9728, '\P{Blk=geometricshapes}', "");
Expect(0, 9728, '\P{^Blk=geometricshapes}', "");
Expect(1, 9727, '\p{Blk=--Geometric_SHAPES}', "");
Expect(0, 9727, '\p{^Blk=--Geometric_SHAPES}', "");
Expect(0, 9727, '\P{Blk=--Geometric_SHAPES}', "");
Expect(1, 9727, '\P{^Blk=--Geometric_SHAPES}', "");
Expect(0, 9728, '\p{Blk=--Geometric_SHAPES}', "");
Expect(1, 9728, '\p{^Blk=--Geometric_SHAPES}', "");
Expect(1, 9728, '\P{Blk=--Geometric_SHAPES}', "");
Expect(0, 9728, '\P{^Blk=--Geometric_SHAPES}', "");
Error('\p{Is_Block=-/a/Geometric_Shapes}');
Error('\P{Is_Block=-/a/Geometric_Shapes}');
Expect(1, 9727, '\p{Is_Block=geometricshapes}', "");
Expect(0, 9727, '\p{^Is_Block=geometricshapes}', "");
Expect(0, 9727, '\P{Is_Block=geometricshapes}', "");
Expect(1, 9727, '\P{^Is_Block=geometricshapes}', "");
Expect(0, 9728, '\p{Is_Block=geometricshapes}', "");
Expect(1, 9728, '\p{^Is_Block=geometricshapes}', "");
Expect(1, 9728, '\P{Is_Block=geometricshapes}', "");
Expect(0, 9728, '\P{^Is_Block=geometricshapes}', "");
Expect(1, 9727, '\p{Is_Block= _Geometric_shapes}', "");
Expect(0, 9727, '\p{^Is_Block= _Geometric_shapes}', "");
Expect(0, 9727, '\P{Is_Block= _Geometric_shapes}', "");
Expect(1, 9727, '\P{^Is_Block= _Geometric_shapes}', "");
Expect(0, 9728, '\p{Is_Block= _Geometric_shapes}', "");
Expect(1, 9728, '\p{^Is_Block= _Geometric_shapes}', "");
Expect(1, 9728, '\P{Is_Block= _Geometric_shapes}', "");
Expect(0, 9728, '\P{^Is_Block= _Geometric_shapes}', "");
Error('\p{Is_Blk=_/a/Geometric_shapes}');
Error('\P{Is_Blk=_/a/Geometric_shapes}');
Expect(1, 9727, '\p{Is_Blk=geometricshapes}', "");
Expect(0, 9727, '\p{^Is_Blk=geometricshapes}', "");
Expect(0, 9727, '\P{Is_Blk=geometricshapes}', "");
Expect(1, 9727, '\P{^Is_Blk=geometricshapes}', "");
Expect(0, 9728, '\p{Is_Blk=geometricshapes}', "");
Expect(1, 9728, '\p{^Is_Blk=geometricshapes}', "");
Expect(1, 9728, '\P{Is_Blk=geometricshapes}', "");
Expect(0, 9728, '\P{^Is_Blk=geometricshapes}', "");
Expect(1, 9727, '\p{Is_Blk=	_GEOMETRIC_shapes}', "");
Expect(0, 9727, '\p{^Is_Blk=	_GEOMETRIC_shapes}', "");
Expect(0, 9727, '\P{Is_Blk=	_GEOMETRIC_shapes}', "");
Expect(1, 9727, '\P{^Is_Blk=	_GEOMETRIC_shapes}', "");
Expect(0, 9728, '\p{Is_Blk=	_GEOMETRIC_shapes}', "");
Expect(1, 9728, '\p{^Is_Blk=	_GEOMETRIC_shapes}', "");
Expect(1, 9728, '\P{Is_Blk=	_GEOMETRIC_shapes}', "");
Expect(0, 9728, '\P{^Is_Blk=	_GEOMETRIC_shapes}', "");
Error('\p{Block=:=  Georgian}');
Error('\P{Block=:=  Georgian}');
Expect(1, 4351, '\p{Block=georgian}', "");
Expect(0, 4351, '\p{^Block=georgian}', "");
Expect(0, 4351, '\P{Block=georgian}', "");
Expect(1, 4351, '\P{^Block=georgian}', "");
Expect(0, 4352, '\p{Block=georgian}', "");
Expect(1, 4352, '\p{^Block=georgian}', "");
Expect(1, 4352, '\P{Block=georgian}', "");
Expect(0, 4352, '\P{^Block=georgian}', "");
Expect(1, 4351, '\p{Block=	_georgian}', "");
Expect(0, 4351, '\p{^Block=	_georgian}', "");
Expect(0, 4351, '\P{Block=	_georgian}', "");
Expect(1, 4351, '\P{^Block=	_georgian}', "");
Expect(0, 4352, '\p{Block=	_georgian}', "");
Expect(1, 4352, '\p{^Block=	_georgian}', "");
Expect(1, 4352, '\P{Block=	_georgian}', "");
Expect(0, 4352, '\P{^Block=	_georgian}', "");
Error('\p{Blk=:=_Georgian}');
Error('\P{Blk=:=_Georgian}');
Expect(1, 4351, '\p{Blk=georgian}', "");
Expect(0, 4351, '\p{^Blk=georgian}', "");
Expect(0, 4351, '\P{Blk=georgian}', "");
Expect(1, 4351, '\P{^Blk=georgian}', "");
Expect(0, 4352, '\p{Blk=georgian}', "");
Expect(1, 4352, '\p{^Blk=georgian}', "");
Expect(1, 4352, '\P{Blk=georgian}', "");
Expect(0, 4352, '\P{^Blk=georgian}', "");
Expect(1, 4351, '\p{Blk=_	Georgian}', "");
Expect(0, 4351, '\p{^Blk=_	Georgian}', "");
Expect(0, 4351, '\P{Blk=_	Georgian}', "");
Expect(1, 4351, '\P{^Blk=_	Georgian}', "");
Expect(0, 4352, '\p{Blk=_	Georgian}', "");
Expect(1, 4352, '\p{^Blk=_	Georgian}', "");
Expect(1, 4352, '\P{Blk=_	Georgian}', "");
Expect(0, 4352, '\P{^Blk=_	Georgian}', "");
Error('\p{Is_Block=/a/Georgian}');
Error('\P{Is_Block=/a/Georgian}');
Expect(1, 4351, '\p{Is_Block=georgian}', "");
Expect(0, 4351, '\p{^Is_Block=georgian}', "");
Expect(0, 4351, '\P{Is_Block=georgian}', "");
Expect(1, 4351, '\P{^Is_Block=georgian}', "");
Expect(0, 4352, '\p{Is_Block=georgian}', "");
Expect(1, 4352, '\p{^Is_Block=georgian}', "");
Expect(1, 4352, '\P{Is_Block=georgian}', "");
Expect(0, 4352, '\P{^Is_Block=georgian}', "");
Expect(1, 4351, '\p{Is_Block=-Georgian}', "");
Expect(0, 4351, '\p{^Is_Block=-Georgian}', "");
Expect(0, 4351, '\P{Is_Block=-Georgian}', "");
Expect(1, 4351, '\P{^Is_Block=-Georgian}', "");
Expect(0, 4352, '\p{Is_Block=-Georgian}', "");
Expect(1, 4352, '\p{^Is_Block=-Georgian}', "");
Expect(1, 4352, '\P{Is_Block=-Georgian}', "");
Expect(0, 4352, '\P{^Is_Block=-Georgian}', "");
Error('\p{Is_Blk=/a/		Georgian}');
Error('\P{Is_Blk=/a/		Georgian}');
Expect(1, 4351, '\p{Is_Blk=georgian}', "");
Expect(0, 4351, '\p{^Is_Blk=georgian}', "");
Expect(0, 4351, '\P{Is_Blk=georgian}', "");
Expect(1, 4351, '\P{^Is_Blk=georgian}', "");
Expect(0, 4352, '\p{Is_Blk=georgian}', "");
Expect(1, 4352, '\p{^Is_Blk=georgian}', "");
Expect(1, 4352, '\P{Is_Blk=georgian}', "");
Expect(0, 4352, '\P{^Is_Blk=georgian}', "");
Expect(1, 4351, '\p{Is_Blk=_-Georgian}', "");
Expect(0, 4351, '\p{^Is_Blk=_-Georgian}', "");
Expect(0, 4351, '\P{Is_Blk=_-Georgian}', "");
Expect(1, 4351, '\P{^Is_Blk=_-Georgian}', "");
Expect(0, 4352, '\p{Is_Blk=_-Georgian}', "");
Expect(1, 4352, '\p{^Is_Blk=_-Georgian}', "");
Expect(1, 4352, '\P{Is_Blk=_-Georgian}', "");
Expect(0, 4352, '\P{^Is_Blk=_-Georgian}', "");
Error('\p{Block=:= -georgian_Supplement}');
Error('\P{Block=:= -georgian_Supplement}');
Expect(1, 11567, '\p{Block: georgiansupplement}', "");
Expect(0, 11567, '\p{^Block: georgiansupplement}', "");
Expect(0, 11567, '\P{Block: georgiansupplement}', "");
Expect(1, 11567, '\P{^Block: georgiansupplement}', "");
Expect(0, 11568, '\p{Block: georgiansupplement}', "");
Expect(1, 11568, '\p{^Block: georgiansupplement}', "");
Expect(1, 11568, '\P{Block: georgiansupplement}', "");
Expect(0, 11568, '\P{^Block: georgiansupplement}', "");
Expect(1, 11567, '\p{Block=	-Georgian_Supplement}', "");
Expect(0, 11567, '\p{^Block=	-Georgian_Supplement}', "");
Expect(0, 11567, '\P{Block=	-Georgian_Supplement}', "");
Expect(1, 11567, '\P{^Block=	-Georgian_Supplement}', "");
Expect(0, 11568, '\p{Block=	-Georgian_Supplement}', "");
Expect(1, 11568, '\p{^Block=	-Georgian_Supplement}', "");
Expect(1, 11568, '\P{Block=	-Georgian_Supplement}', "");
Expect(0, 11568, '\P{^Block=	-Georgian_Supplement}', "");
Error('\p{Blk=  GEORGIAN_Sup:=}');
Error('\P{Blk=  GEORGIAN_Sup:=}');
Expect(1, 11567, '\p{Blk=georgiansup}', "");
Expect(0, 11567, '\p{^Blk=georgiansup}', "");
Expect(0, 11567, '\P{Blk=georgiansup}', "");
Expect(1, 11567, '\P{^Blk=georgiansup}', "");
Expect(0, 11568, '\p{Blk=georgiansup}', "");
Expect(1, 11568, '\p{^Blk=georgiansup}', "");
Expect(1, 11568, '\P{Blk=georgiansup}', "");
Expect(0, 11568, '\P{^Blk=georgiansup}', "");
Expect(1, 11567, '\p{Blk=--Georgian_Sup}', "");
Expect(0, 11567, '\p{^Blk=--Georgian_Sup}', "");
Expect(0, 11567, '\P{Blk=--Georgian_Sup}', "");
Expect(1, 11567, '\P{^Blk=--Georgian_Sup}', "");
Expect(0, 11568, '\p{Blk=--Georgian_Sup}', "");
Expect(1, 11568, '\p{^Blk=--Georgian_Sup}', "");
Expect(1, 11568, '\P{Blk=--Georgian_Sup}', "");
Expect(0, 11568, '\P{^Blk=--Georgian_Sup}', "");
Error('\p{Is_Block= /a/Georgian_Supplement}');
Error('\P{Is_Block= /a/Georgian_Supplement}');
Expect(1, 11567, '\p{Is_Block=georgiansupplement}', "");
Expect(0, 11567, '\p{^Is_Block=georgiansupplement}', "");
Expect(0, 11567, '\P{Is_Block=georgiansupplement}', "");
Expect(1, 11567, '\P{^Is_Block=georgiansupplement}', "");
Expect(0, 11568, '\p{Is_Block=georgiansupplement}', "");
Expect(1, 11568, '\p{^Is_Block=georgiansupplement}', "");
Expect(1, 11568, '\P{Is_Block=georgiansupplement}', "");
Expect(0, 11568, '\P{^Is_Block=georgiansupplement}', "");
Expect(1, 11567, '\p{Is_Block=Georgian_Supplement}', "");
Expect(0, 11567, '\p{^Is_Block=Georgian_Supplement}', "");
Expect(0, 11567, '\P{Is_Block=Georgian_Supplement}', "");
Expect(1, 11567, '\P{^Is_Block=Georgian_Supplement}', "");
Expect(0, 11568, '\p{Is_Block=Georgian_Supplement}', "");
Expect(1, 11568, '\p{^Is_Block=Georgian_Supplement}', "");
Expect(1, 11568, '\P{Is_Block=Georgian_Supplement}', "");
Expect(0, 11568, '\P{^Is_Block=Georgian_Supplement}', "");
Error('\p{Is_Blk=:=_GEORGIAN_sup}');
Error('\P{Is_Blk=:=_GEORGIAN_sup}');
Expect(1, 11567, '\p{Is_Blk=georgiansup}', "");
Expect(0, 11567, '\p{^Is_Blk=georgiansup}', "");
Expect(0, 11567, '\P{Is_Blk=georgiansup}', "");
Expect(1, 11567, '\P{^Is_Blk=georgiansup}', "");
Expect(0, 11568, '\p{Is_Blk=georgiansup}', "");
Expect(1, 11568, '\p{^Is_Blk=georgiansup}', "");
Expect(1, 11568, '\P{Is_Blk=georgiansup}', "");
Expect(0, 11568, '\P{^Is_Blk=georgiansup}', "");
Expect(1, 11567, '\p{Is_Blk= 	Georgian_SUP}', "");
Expect(0, 11567, '\p{^Is_Blk= 	Georgian_SUP}', "");
Expect(0, 11567, '\P{Is_Blk= 	Georgian_SUP}', "");
Expect(1, 11567, '\P{^Is_Blk= 	Georgian_SUP}', "");
Expect(0, 11568, '\p{Is_Blk= 	Georgian_SUP}', "");
Expect(1, 11568, '\p{^Is_Blk= 	Georgian_SUP}', "");
Expect(1, 11568, '\P{Is_Blk= 	Georgian_SUP}', "");
Expect(0, 11568, '\P{^Is_Blk= 	Georgian_SUP}', "");
Error('\p{Block=glagolitic/a/}');
Error('\P{Block=glagolitic/a/}');
Expect(1, 11359, '\p{Block=glagolitic}', "");
Expect(0, 11359, '\p{^Block=glagolitic}', "");
Expect(0, 11359, '\P{Block=glagolitic}', "");
Expect(1, 11359, '\P{^Block=glagolitic}', "");
Expect(0, 11360, '\p{Block=glagolitic}', "");
Expect(1, 11360, '\p{^Block=glagolitic}', "");
Expect(1, 11360, '\P{Block=glagolitic}', "");
Expect(0, 11360, '\P{^Block=glagolitic}', "");
Expect(1, 11359, '\p{Block= 	GLAGOLITIC}', "");
Expect(0, 11359, '\p{^Block= 	GLAGOLITIC}', "");
Expect(0, 11359, '\P{Block= 	GLAGOLITIC}', "");
Expect(1, 11359, '\P{^Block= 	GLAGOLITIC}', "");
Expect(0, 11360, '\p{Block= 	GLAGOLITIC}', "");
Expect(1, 11360, '\p{^Block= 	GLAGOLITIC}', "");
Expect(1, 11360, '\P{Block= 	GLAGOLITIC}', "");
Expect(0, 11360, '\P{^Block= 	GLAGOLITIC}', "");
Error('\p{Blk=/a/glagolitic}');
Error('\P{Blk=/a/glagolitic}');
Expect(1, 11359, '\p{Blk:glagolitic}', "");
Expect(0, 11359, '\p{^Blk:glagolitic}', "");
Expect(0, 11359, '\P{Blk:glagolitic}', "");
Expect(1, 11359, '\P{^Blk:glagolitic}', "");
Expect(0, 11360, '\p{Blk:glagolitic}', "");
Expect(1, 11360, '\p{^Blk:glagolitic}', "");
Expect(1, 11360, '\P{Blk:glagolitic}', "");
Expect(0, 11360, '\P{^Blk:glagolitic}', "");
Expect(1, 11359, '\p{Blk=-GLAGOLITIC}', "");
Expect(0, 11359, '\p{^Blk=-GLAGOLITIC}', "");
Expect(0, 11359, '\P{Blk=-GLAGOLITIC}', "");
Expect(1, 11359, '\P{^Blk=-GLAGOLITIC}', "");
Expect(0, 11360, '\p{Blk=-GLAGOLITIC}', "");
Expect(1, 11360, '\p{^Blk=-GLAGOLITIC}', "");
Expect(1, 11360, '\P{Blk=-GLAGOLITIC}', "");
Expect(0, 11360, '\P{^Blk=-GLAGOLITIC}', "");
Error('\p{Is_Block:			Glagolitic:=}');
Error('\P{Is_Block:			Glagolitic:=}');
Expect(1, 11359, '\p{Is_Block=glagolitic}', "");
Expect(0, 11359, '\p{^Is_Block=glagolitic}', "");
Expect(0, 11359, '\P{Is_Block=glagolitic}', "");
Expect(1, 11359, '\P{^Is_Block=glagolitic}', "");
Expect(0, 11360, '\p{Is_Block=glagolitic}', "");
Expect(1, 11360, '\p{^Is_Block=glagolitic}', "");
Expect(1, 11360, '\P{Is_Block=glagolitic}', "");
Expect(0, 11360, '\P{^Is_Block=glagolitic}', "");
Expect(1, 11359, '\p{Is_Block=-Glagolitic}', "");
Expect(0, 11359, '\p{^Is_Block=-Glagolitic}', "");
Expect(0, 11359, '\P{Is_Block=-Glagolitic}', "");
Expect(1, 11359, '\P{^Is_Block=-Glagolitic}', "");
Expect(0, 11360, '\p{Is_Block=-Glagolitic}', "");
Expect(1, 11360, '\p{^Is_Block=-Glagolitic}', "");
Expect(1, 11360, '\P{Is_Block=-Glagolitic}', "");
Expect(0, 11360, '\P{^Is_Block=-Glagolitic}', "");
Error('\p{Is_Blk=_:=GLAGOLITIC}');
Error('\P{Is_Blk=_:=GLAGOLITIC}');
Expect(1, 11359, '\p{Is_Blk=glagolitic}', "");
Expect(0, 11359, '\p{^Is_Blk=glagolitic}', "");
Expect(0, 11359, '\P{Is_Blk=glagolitic}', "");
Expect(1, 11359, '\P{^Is_Blk=glagolitic}', "");
Expect(0, 11360, '\p{Is_Blk=glagolitic}', "");
Expect(1, 11360, '\p{^Is_Blk=glagolitic}', "");
Expect(1, 11360, '\P{Is_Blk=glagolitic}', "");
Expect(0, 11360, '\P{^Is_Blk=glagolitic}', "");
Expect(1, 11359, '\p{Is_Blk=__GLAGOLITIC}', "");
Expect(0, 11359, '\p{^Is_Blk=__GLAGOLITIC}', "");
Expect(0, 11359, '\P{Is_Blk=__GLAGOLITIC}', "");
Expect(1, 11359, '\P{^Is_Blk=__GLAGOLITIC}', "");
Expect(0, 11360, '\p{Is_Blk=__GLAGOLITIC}', "");
Expect(1, 11360, '\p{^Is_Blk=__GLAGOLITIC}', "");
Expect(1, 11360, '\P{Is_Blk=__GLAGOLITIC}', "");
Expect(0, 11360, '\P{^Is_Blk=__GLAGOLITIC}', "");
Error('\p{Block=-_Gothic/a/}');
Error('\P{Block=-_Gothic/a/}');
Expect(1, 66383, '\p{Block=gothic}', "");
Expect(0, 66383, '\p{^Block=gothic}', "");
Expect(0, 66383, '\P{Block=gothic}', "");
Expect(1, 66383, '\P{^Block=gothic}', "");
Expect(0, 66384, '\p{Block=gothic}', "");
Expect(1, 66384, '\p{^Block=gothic}', "");
Expect(1, 66384, '\P{Block=gothic}', "");
Expect(0, 66384, '\P{^Block=gothic}', "");
Expect(1, 66383, '\p{Block= GOTHIC}', "");
Expect(0, 66383, '\p{^Block= GOTHIC}', "");
Expect(0, 66383, '\P{Block= GOTHIC}', "");
Expect(1, 66383, '\P{^Block= GOTHIC}', "");
Expect(0, 66384, '\p{Block= GOTHIC}', "");
Expect(1, 66384, '\p{^Block= GOTHIC}', "");
Expect(1, 66384, '\P{Block= GOTHIC}', "");
Expect(0, 66384, '\P{^Block= GOTHIC}', "");
Error('\p{Blk:  GOTHIC/a/}');
Error('\P{Blk:  GOTHIC/a/}');
Expect(1, 66383, '\p{Blk=gothic}', "");
Expect(0, 66383, '\p{^Blk=gothic}', "");
Expect(0, 66383, '\P{Blk=gothic}', "");
Expect(1, 66383, '\P{^Blk=gothic}', "");
Expect(0, 66384, '\p{Blk=gothic}', "");
Expect(1, 66384, '\p{^Blk=gothic}', "");
Expect(1, 66384, '\P{Blk=gothic}', "");
Expect(0, 66384, '\P{^Blk=gothic}', "");
Expect(1, 66383, '\p{Blk=  Gothic}', "");
Expect(0, 66383, '\p{^Blk=  Gothic}', "");
Expect(0, 66383, '\P{Blk=  Gothic}', "");
Expect(1, 66383, '\P{^Blk=  Gothic}', "");
Expect(0, 66384, '\p{Blk=  Gothic}', "");
Expect(1, 66384, '\p{^Blk=  Gothic}', "");
Expect(1, 66384, '\P{Blk=  Gothic}', "");
Expect(0, 66384, '\P{^Blk=  Gothic}', "");
Error('\p{Is_Block=__gothic:=}');
Error('\P{Is_Block=__gothic:=}');
Expect(1, 66383, '\p{Is_Block=gothic}', "");
Expect(0, 66383, '\p{^Is_Block=gothic}', "");
Expect(0, 66383, '\P{Is_Block=gothic}', "");
Expect(1, 66383, '\P{^Is_Block=gothic}', "");
Expect(0, 66384, '\p{Is_Block=gothic}', "");
Expect(1, 66384, '\p{^Is_Block=gothic}', "");
Expect(1, 66384, '\P{Is_Block=gothic}', "");
Expect(0, 66384, '\P{^Is_Block=gothic}', "");
Expect(1, 66383, '\p{Is_Block=-_gothic}', "");
Expect(0, 66383, '\p{^Is_Block=-_gothic}', "");
Expect(0, 66383, '\P{Is_Block=-_gothic}', "");
Expect(1, 66383, '\P{^Is_Block=-_gothic}', "");
Expect(0, 66384, '\p{Is_Block=-_gothic}', "");
Expect(1, 66384, '\p{^Is_Block=-_gothic}', "");
Expect(1, 66384, '\P{Is_Block=-_gothic}', "");
Expect(0, 66384, '\P{^Is_Block=-_gothic}', "");
Error('\p{Is_Blk=_:=Gothic}');
Error('\P{Is_Blk=_:=Gothic}');
Expect(1, 66383, '\p{Is_Blk=gothic}', "");
Expect(0, 66383, '\p{^Is_Blk=gothic}', "");
Expect(0, 66383, '\P{Is_Blk=gothic}', "");
Expect(1, 66383, '\P{^Is_Blk=gothic}', "");
Expect(0, 66384, '\p{Is_Blk=gothic}', "");
Expect(1, 66384, '\p{^Is_Blk=gothic}', "");
Expect(1, 66384, '\P{Is_Blk=gothic}', "");
Expect(0, 66384, '\P{^Is_Blk=gothic}', "");
Expect(1, 66383, '\p{Is_Blk=	Gothic}', "");
Expect(0, 66383, '\p{^Is_Blk=	Gothic}', "");
Expect(0, 66383, '\P{Is_Blk=	Gothic}', "");
Expect(1, 66383, '\P{^Is_Blk=	Gothic}', "");
Expect(0, 66384, '\p{Is_Blk=	Gothic}', "");
Expect(1, 66384, '\p{^Is_Blk=	Gothic}', "");
Expect(1, 66384, '\P{Is_Blk=	Gothic}', "");
Expect(0, 66384, '\P{^Is_Blk=	Gothic}', "");
Error('\p{Block= _greek_AND_coptic:=}');
Error('\P{Block= _greek_AND_coptic:=}');
Expect(1, 1023, '\p{Block=greekandcoptic}', "");
Expect(0, 1023, '\p{^Block=greekandcoptic}', "");
Expect(0, 1023, '\P{Block=greekandcoptic}', "");
Expect(1, 1023, '\P{^Block=greekandcoptic}', "");
Expect(0, 1024, '\p{Block=greekandcoptic}', "");
Expect(1, 1024, '\p{^Block=greekandcoptic}', "");
Expect(1, 1024, '\P{Block=greekandcoptic}', "");
Expect(0, 1024, '\P{^Block=greekandcoptic}', "");
Expect(1, 1023, '\p{Block=		GREEK_And_COPTIC}', "");
Expect(0, 1023, '\p{^Block=		GREEK_And_COPTIC}', "");
Expect(0, 1023, '\P{Block=		GREEK_And_COPTIC}', "");
Expect(1, 1023, '\P{^Block=		GREEK_And_COPTIC}', "");
Expect(0, 1024, '\p{Block=		GREEK_And_COPTIC}', "");
Expect(1, 1024, '\p{^Block=		GREEK_And_COPTIC}', "");
Expect(1, 1024, '\P{Block=		GREEK_And_COPTIC}', "");
Expect(0, 1024, '\P{^Block=		GREEK_And_COPTIC}', "");
Error('\p{Blk=	:=Greek}');
Error('\P{Blk=	:=Greek}');
Expect(1, 1023, '\p{Blk=greek}', "");
Expect(0, 1023, '\p{^Blk=greek}', "");
Expect(0, 1023, '\P{Blk=greek}', "");
Expect(1, 1023, '\P{^Blk=greek}', "");
Expect(0, 1024, '\p{Blk=greek}', "");
Expect(1, 1024, '\p{^Blk=greek}', "");
Expect(1, 1024, '\P{Blk=greek}', "");
Expect(0, 1024, '\P{^Blk=greek}', "");
Expect(1, 1023, '\p{Blk=--greek}', "");
Expect(0, 1023, '\p{^Blk=--greek}', "");
Expect(0, 1023, '\P{Blk=--greek}', "");
Expect(1, 1023, '\P{^Blk=--greek}', "");
Expect(0, 1024, '\p{Blk=--greek}', "");
Expect(1, 1024, '\p{^Blk=--greek}', "");
Expect(1, 1024, '\P{Blk=--greek}', "");
Expect(0, 1024, '\P{^Blk=--greek}', "");
Error('\p{Is_Block:		:=Greek_and_coptic}');
Error('\P{Is_Block:		:=Greek_and_coptic}');
Expect(1, 1023, '\p{Is_Block=greekandcoptic}', "");
Expect(0, 1023, '\p{^Is_Block=greekandcoptic}', "");
Expect(0, 1023, '\P{Is_Block=greekandcoptic}', "");
Expect(1, 1023, '\P{^Is_Block=greekandcoptic}', "");
Expect(0, 1024, '\p{Is_Block=greekandcoptic}', "");
Expect(1, 1024, '\p{^Is_Block=greekandcoptic}', "");
Expect(1, 1024, '\P{Is_Block=greekandcoptic}', "");
Expect(0, 1024, '\P{^Is_Block=greekandcoptic}', "");
Expect(1, 1023, '\p{Is_Block=__Greek_And_Coptic}', "");
Expect(0, 1023, '\p{^Is_Block=__Greek_And_Coptic}', "");
Expect(0, 1023, '\P{Is_Block=__Greek_And_Coptic}', "");
Expect(1, 1023, '\P{^Is_Block=__Greek_And_Coptic}', "");
Expect(0, 1024, '\p{Is_Block=__Greek_And_Coptic}', "");
Expect(1, 1024, '\p{^Is_Block=__Greek_And_Coptic}', "");
Expect(1, 1024, '\P{Is_Block=__Greek_And_Coptic}', "");
Expect(0, 1024, '\P{^Is_Block=__Greek_And_Coptic}', "");
Error('\p{Is_Blk= -Greek:=}');
Error('\P{Is_Blk= -Greek:=}');
Expect(1, 1023, '\p{Is_Blk:   greek}', "");
Expect(0, 1023, '\p{^Is_Blk:   greek}', "");
Expect(0, 1023, '\P{Is_Blk:   greek}', "");
Expect(1, 1023, '\P{^Is_Blk:   greek}', "");
Expect(0, 1024, '\p{Is_Blk:   greek}', "");
Expect(1, 1024, '\p{^Is_Blk:   greek}', "");
Expect(1, 1024, '\P{Is_Blk:   greek}', "");
Expect(0, 1024, '\P{^Is_Blk:   greek}', "");
Expect(1, 1023, '\p{Is_Blk=-greek}', "");
Expect(0, 1023, '\p{^Is_Blk=-greek}', "");
Expect(0, 1023, '\P{Is_Blk=-greek}', "");
Expect(1, 1023, '\P{^Is_Blk=-greek}', "");
Expect(0, 1024, '\p{Is_Blk=-greek}', "");
Expect(1, 1024, '\p{^Is_Blk=-greek}', "");
Expect(1, 1024, '\P{Is_Blk=-greek}', "");
Expect(0, 1024, '\P{^Is_Blk=-greek}', "");
Error('\p{Block=-:=Greek_EXTENDED}');
Error('\P{Block=-:=Greek_EXTENDED}');
Expect(1, 8191, '\p{Block=greekextended}', "");
Expect(0, 8191, '\p{^Block=greekextended}', "");
Expect(0, 8191, '\P{Block=greekextended}', "");
Expect(1, 8191, '\P{^Block=greekextended}', "");
Expect(0, 8192, '\p{Block=greekextended}', "");
Expect(1, 8192, '\p{^Block=greekextended}', "");
Expect(1, 8192, '\P{Block=greekextended}', "");
Expect(0, 8192, '\P{^Block=greekextended}', "");
Expect(1, 8191, '\p{Block=  Greek_EXTENDED}', "");
Expect(0, 8191, '\p{^Block=  Greek_EXTENDED}', "");
Expect(0, 8191, '\P{Block=  Greek_EXTENDED}', "");
Expect(1, 8191, '\P{^Block=  Greek_EXTENDED}', "");
Expect(0, 8192, '\p{Block=  Greek_EXTENDED}', "");
Expect(1, 8192, '\p{^Block=  Greek_EXTENDED}', "");
Expect(1, 8192, '\P{Block=  Greek_EXTENDED}', "");
Expect(0, 8192, '\P{^Block=  Greek_EXTENDED}', "");
Error('\p{Blk=:=--Greek_EXT}');
Error('\P{Blk=:=--Greek_EXT}');
Expect(1, 8191, '\p{Blk=greekext}', "");
Expect(0, 8191, '\p{^Blk=greekext}', "");
Expect(0, 8191, '\P{Blk=greekext}', "");
Expect(1, 8191, '\P{^Blk=greekext}', "");
Expect(0, 8192, '\p{Blk=greekext}', "");
Expect(1, 8192, '\p{^Blk=greekext}', "");
Expect(1, 8192, '\P{Blk=greekext}', "");
Expect(0, 8192, '\P{^Blk=greekext}', "");
Expect(1, 8191, '\p{Blk=_ GREEK_EXT}', "");
Expect(0, 8191, '\p{^Blk=_ GREEK_EXT}', "");
Expect(0, 8191, '\P{Blk=_ GREEK_EXT}', "");
Expect(1, 8191, '\P{^Blk=_ GREEK_EXT}', "");
Expect(0, 8192, '\p{Blk=_ GREEK_EXT}', "");
Expect(1, 8192, '\p{^Blk=_ GREEK_EXT}', "");
Expect(1, 8192, '\P{Blk=_ GREEK_EXT}', "");
Expect(0, 8192, '\P{^Blk=_ GREEK_EXT}', "");
Error('\p{Is_Block:	:= greek_Extended}');
Error('\P{Is_Block:	:= greek_Extended}');
Expect(1, 8191, '\p{Is_Block=greekextended}', "");
Expect(0, 8191, '\p{^Is_Block=greekextended}', "");
Expect(0, 8191, '\P{Is_Block=greekextended}', "");
Expect(1, 8191, '\P{^Is_Block=greekextended}', "");
Expect(0, 8192, '\p{Is_Block=greekextended}', "");
Expect(1, 8192, '\p{^Is_Block=greekextended}', "");
Expect(1, 8192, '\P{Is_Block=greekextended}', "");
Expect(0, 8192, '\P{^Is_Block=greekextended}', "");
Expect(1, 8191, '\p{Is_Block:		greek_Extended}', "");
Expect(0, 8191, '\p{^Is_Block:		greek_Extended}', "");
Expect(0, 8191, '\P{Is_Block:		greek_Extended}', "");
Expect(1, 8191, '\P{^Is_Block:		greek_Extended}', "");
Expect(0, 8192, '\p{Is_Block:		greek_Extended}', "");
Expect(1, 8192, '\p{^Is_Block:		greek_Extended}', "");
Expect(1, 8192, '\P{Is_Block:		greek_Extended}', "");
Expect(0, 8192, '\P{^Is_Block:		greek_Extended}', "");
Error('\p{Is_Blk=	/a/Greek_Ext}');
Error('\P{Is_Blk=	/a/Greek_Ext}');
Expect(1, 8191, '\p{Is_Blk=greekext}', "");
Expect(0, 8191, '\p{^Is_Blk=greekext}', "");
Expect(0, 8191, '\P{Is_Blk=greekext}', "");
Expect(1, 8191, '\P{^Is_Blk=greekext}', "");
Expect(0, 8192, '\p{Is_Blk=greekext}', "");
Expect(1, 8192, '\p{^Is_Blk=greekext}', "");
Expect(1, 8192, '\P{Is_Blk=greekext}', "");
Expect(0, 8192, '\P{^Is_Blk=greekext}', "");
Expect(1, 8191, '\p{Is_Blk=	greek_EXT}', "");
Expect(0, 8191, '\p{^Is_Blk=	greek_EXT}', "");
Expect(0, 8191, '\P{Is_Blk=	greek_EXT}', "");
Expect(1, 8191, '\P{^Is_Blk=	greek_EXT}', "");
Expect(0, 8192, '\p{Is_Blk=	greek_EXT}', "");
Expect(1, 8192, '\p{^Is_Blk=	greek_EXT}', "");
Expect(1, 8192, '\P{Is_Blk=	greek_EXT}', "");
Expect(0, 8192, '\P{^Is_Blk=	greek_EXT}', "");
Error('\p{Block=- Gujarati/a/}');
Error('\P{Block=- Gujarati/a/}');
Expect(1, 2815, '\p{Block=gujarati}', "");
Expect(0, 2815, '\p{^Block=gujarati}', "");
Expect(0, 2815, '\P{Block=gujarati}', "");
Expect(1, 2815, '\P{^Block=gujarati}', "");
Expect(0, 2816, '\p{Block=gujarati}', "");
Expect(1, 2816, '\p{^Block=gujarati}', "");
Expect(1, 2816, '\P{Block=gujarati}', "");
Expect(0, 2816, '\P{^Block=gujarati}', "");
Expect(1, 2815, '\p{Block:	_-GUJARATI}', "");
Expect(0, 2815, '\p{^Block:	_-GUJARATI}', "");
Expect(0, 2815, '\P{Block:	_-GUJARATI}', "");
Expect(1, 2815, '\P{^Block:	_-GUJARATI}', "");
Expect(0, 2816, '\p{Block:	_-GUJARATI}', "");
Expect(1, 2816, '\p{^Block:	_-GUJARATI}', "");
Expect(1, 2816, '\P{Block:	_-GUJARATI}', "");
Expect(0, 2816, '\P{^Block:	_-GUJARATI}', "");
Error('\p{Blk=/a/Gujarati}');
Error('\P{Blk=/a/Gujarati}');
Expect(1, 2815, '\p{Blk=gujarati}', "");
Expect(0, 2815, '\p{^Blk=gujarati}', "");
Expect(0, 2815, '\P{Blk=gujarati}', "");
Expect(1, 2815, '\P{^Blk=gujarati}', "");
Expect(0, 2816, '\p{Blk=gujarati}', "");
Expect(1, 2816, '\p{^Blk=gujarati}', "");
Expect(1, 2816, '\P{Blk=gujarati}', "");
Expect(0, 2816, '\P{^Blk=gujarati}', "");
Expect(1, 2815, '\p{Blk=	-gujarati}', "");
Expect(0, 2815, '\p{^Blk=	-gujarati}', "");
Expect(0, 2815, '\P{Blk=	-gujarati}', "");
Expect(1, 2815, '\P{^Blk=	-gujarati}', "");
Expect(0, 2816, '\p{Blk=	-gujarati}', "");
Expect(1, 2816, '\p{^Blk=	-gujarati}', "");
Expect(1, 2816, '\P{Blk=	-gujarati}', "");
Expect(0, 2816, '\P{^Blk=	-gujarati}', "");
Error('\p{Is_Block=_GUJARATI:=}');
Error('\P{Is_Block=_GUJARATI:=}');
Expect(1, 2815, '\p{Is_Block=gujarati}', "");
Expect(0, 2815, '\p{^Is_Block=gujarati}', "");
Expect(0, 2815, '\P{Is_Block=gujarati}', "");
Expect(1, 2815, '\P{^Is_Block=gujarati}', "");
Expect(0, 2816, '\p{Is_Block=gujarati}', "");
Expect(1, 2816, '\p{^Is_Block=gujarati}', "");
Expect(1, 2816, '\P{Is_Block=gujarati}', "");
Expect(0, 2816, '\P{^Is_Block=gujarati}', "");
Expect(1, 2815, '\p{Is_Block=_gujarati}', "");
Expect(0, 2815, '\p{^Is_Block=_gujarati}', "");
Expect(0, 2815, '\P{Is_Block=_gujarati}', "");
Expect(1, 2815, '\P{^Is_Block=_gujarati}', "");
Expect(0, 2816, '\p{Is_Block=_gujarati}', "");
Expect(1, 2816, '\p{^Is_Block=_gujarati}', "");
Expect(1, 2816, '\P{Is_Block=_gujarati}', "");
Expect(0, 2816, '\P{^Is_Block=_gujarati}', "");
Error('\p{Is_Blk=/a/_ Gujarati}');
Error('\P{Is_Blk=/a/_ Gujarati}');
Expect(1, 2815, '\p{Is_Blk=gujarati}', "");
Expect(0, 2815, '\p{^Is_Blk=gujarati}', "");
Expect(0, 2815, '\P{Is_Blk=gujarati}', "");
Expect(1, 2815, '\P{^Is_Blk=gujarati}', "");
Expect(0, 2816, '\p{Is_Blk=gujarati}', "");
Expect(1, 2816, '\p{^Is_Blk=gujarati}', "");
Expect(1, 2816, '\P{Is_Blk=gujarati}', "");
Expect(0, 2816, '\P{^Is_Blk=gujarati}', "");
Expect(1, 2815, '\p{Is_Blk=	-gujarati}', "");
Expect(0, 2815, '\p{^Is_Blk=	-gujarati}', "");
Expect(0, 2815, '\P{Is_Blk=	-gujarati}', "");
Expect(1, 2815, '\P{^Is_Blk=	-gujarati}', "");
Expect(0, 2816, '\p{Is_Blk=	-gujarati}', "");
Expect(1, 2816, '\p{^Is_Blk=	-gujarati}', "");
Expect(1, 2816, '\P{Is_Blk=	-gujarati}', "");
Expect(0, 2816, '\P{^Is_Blk=	-gujarati}', "");
Error('\p{Block=:=GURMUKHI}');
Error('\P{Block=:=GURMUKHI}');
Expect(1, 2687, '\p{Block=gurmukhi}', "");
Expect(0, 2687, '\p{^Block=gurmukhi}', "");
Expect(0, 2687, '\P{Block=gurmukhi}', "");
Expect(1, 2687, '\P{^Block=gurmukhi}', "");
Expect(0, 2688, '\p{Block=gurmukhi}', "");
Expect(1, 2688, '\p{^Block=gurmukhi}', "");
Expect(1, 2688, '\P{Block=gurmukhi}', "");
Expect(0, 2688, '\P{^Block=gurmukhi}', "");
Expect(1, 2687, '\p{Block=- Gurmukhi}', "");
Expect(0, 2687, '\p{^Block=- Gurmukhi}', "");
Expect(0, 2687, '\P{Block=- Gurmukhi}', "");
Expect(1, 2687, '\P{^Block=- Gurmukhi}', "");
Expect(0, 2688, '\p{Block=- Gurmukhi}', "");
Expect(1, 2688, '\p{^Block=- Gurmukhi}', "");
Expect(1, 2688, '\P{Block=- Gurmukhi}', "");
Expect(0, 2688, '\P{^Block=- Gurmukhi}', "");
Error('\p{Blk=:=-_Gurmukhi}');
Error('\P{Blk=:=-_Gurmukhi}');
Expect(1, 2687, '\p{Blk=gurmukhi}', "");
Expect(0, 2687, '\p{^Blk=gurmukhi}', "");
Expect(0, 2687, '\P{Blk=gurmukhi}', "");
Expect(1, 2687, '\P{^Blk=gurmukhi}', "");
Expect(0, 2688, '\p{Blk=gurmukhi}', "");
Expect(1, 2688, '\p{^Blk=gurmukhi}', "");
Expect(1, 2688, '\P{Blk=gurmukhi}', "");
Expect(0, 2688, '\P{^Blk=gurmukhi}', "");
Expect(1, 2687, '\p{Blk= Gurmukhi}', "");
Expect(0, 2687, '\p{^Blk= Gurmukhi}', "");
Expect(0, 2687, '\P{Blk= Gurmukhi}', "");
Expect(1, 2687, '\P{^Blk= Gurmukhi}', "");
Expect(0, 2688, '\p{Blk= Gurmukhi}', "");
Expect(1, 2688, '\p{^Blk= Gurmukhi}', "");
Expect(1, 2688, '\P{Blk= Gurmukhi}', "");
Expect(0, 2688, '\P{^Blk= Gurmukhi}', "");
Error('\p{Is_Block=_-gurmukhi:=}');
Error('\P{Is_Block=_-gurmukhi:=}');
Expect(1, 2687, '\p{Is_Block=gurmukhi}', "");
Expect(0, 2687, '\p{^Is_Block=gurmukhi}', "");
Expect(0, 2687, '\P{Is_Block=gurmukhi}', "");
Expect(1, 2687, '\P{^Is_Block=gurmukhi}', "");
Expect(0, 2688, '\p{Is_Block=gurmukhi}', "");
Expect(1, 2688, '\p{^Is_Block=gurmukhi}', "");
Expect(1, 2688, '\P{Is_Block=gurmukhi}', "");
Expect(0, 2688, '\P{^Is_Block=gurmukhi}', "");
Expect(1, 2687, '\p{Is_Block= gurmukhi}', "");
Expect(0, 2687, '\p{^Is_Block= gurmukhi}', "");
Expect(0, 2687, '\P{Is_Block= gurmukhi}', "");
Expect(1, 2687, '\P{^Is_Block= gurmukhi}', "");
Expect(0, 2688, '\p{Is_Block= gurmukhi}', "");
Expect(1, 2688, '\p{^Is_Block= gurmukhi}', "");
Expect(1, 2688, '\P{Is_Block= gurmukhi}', "");
Expect(0, 2688, '\P{^Is_Block= gurmukhi}', "");
Error('\p{Is_Blk:   :=--gurmukhi}');
Error('\P{Is_Blk:   :=--gurmukhi}');
Expect(1, 2687, '\p{Is_Blk=gurmukhi}', "");
Expect(0, 2687, '\p{^Is_Blk=gurmukhi}', "");
Expect(0, 2687, '\P{Is_Blk=gurmukhi}', "");
Expect(1, 2687, '\P{^Is_Blk=gurmukhi}', "");
Expect(0, 2688, '\p{Is_Blk=gurmukhi}', "");
Expect(1, 2688, '\p{^Is_Blk=gurmukhi}', "");
Expect(1, 2688, '\P{Is_Blk=gurmukhi}', "");
Expect(0, 2688, '\P{^Is_Blk=gurmukhi}', "");
Expect(1, 2687, '\p{Is_Blk=	_Gurmukhi}', "");
Expect(0, 2687, '\p{^Is_Blk=	_Gurmukhi}', "");
Expect(0, 2687, '\P{Is_Blk=	_Gurmukhi}', "");
Expect(1, 2687, '\P{^Is_Blk=	_Gurmukhi}', "");
Expect(0, 2688, '\p{Is_Blk=	_Gurmukhi}', "");
Expect(1, 2688, '\p{^Is_Blk=	_Gurmukhi}', "");
Expect(1, 2688, '\P{Is_Blk=	_Gurmukhi}', "");
Expect(0, 2688, '\P{^Is_Blk=	_Gurmukhi}', "");
Error('\p{Block:	/a/-_Halfwidth_And_fullwidth_forms}');
Error('\P{Block:	/a/-_Halfwidth_And_fullwidth_forms}');
Expect(1, 65519, '\p{Block=halfwidthandfullwidthforms}', "");
Expect(0, 65519, '\p{^Block=halfwidthandfullwidthforms}', "");
Expect(0, 65519, '\P{Block=halfwidthandfullwidthforms}', "");
Expect(1, 65519, '\P{^Block=halfwidthandfullwidthforms}', "");
Expect(0, 65520, '\p{Block=halfwidthandfullwidthforms}', "");
Expect(1, 65520, '\p{^Block=halfwidthandfullwidthforms}', "");
Expect(1, 65520, '\P{Block=halfwidthandfullwidthforms}', "");
Expect(0, 65520, '\P{^Block=halfwidthandfullwidthforms}', "");
Expect(1, 65519, '\p{Block=-	halfwidth_And_Fullwidth_forms}', "");
Expect(0, 65519, '\p{^Block=-	halfwidth_And_Fullwidth_forms}', "");
Expect(0, 65519, '\P{Block=-	halfwidth_And_Fullwidth_forms}', "");
Expect(1, 65519, '\P{^Block=-	halfwidth_And_Fullwidth_forms}', "");
Expect(0, 65520, '\p{Block=-	halfwidth_And_Fullwidth_forms}', "");
Expect(1, 65520, '\p{^Block=-	halfwidth_And_Fullwidth_forms}', "");
Expect(1, 65520, '\P{Block=-	halfwidth_And_Fullwidth_forms}', "");
Expect(0, 65520, '\P{^Block=-	halfwidth_And_Fullwidth_forms}', "");
Error('\p{Blk=:=		Half_and_Full_Forms}');
Error('\P{Blk=:=		Half_and_Full_Forms}');
Expect(1, 65519, '\p{Blk=halfandfullforms}', "");
Expect(0, 65519, '\p{^Blk=halfandfullforms}', "");
Expect(0, 65519, '\P{Blk=halfandfullforms}', "");
Expect(1, 65519, '\P{^Blk=halfandfullforms}', "");
Expect(0, 65520, '\p{Blk=halfandfullforms}', "");
Expect(1, 65520, '\p{^Blk=halfandfullforms}', "");
Expect(1, 65520, '\P{Blk=halfandfullforms}', "");
Expect(0, 65520, '\P{^Blk=halfandfullforms}', "");
Expect(1, 65519, '\p{Blk=	 HALF_and_Full_forms}', "");
Expect(0, 65519, '\p{^Blk=	 HALF_and_Full_forms}', "");
Expect(0, 65519, '\P{Blk=	 HALF_and_Full_forms}', "");
Expect(1, 65519, '\P{^Blk=	 HALF_and_Full_forms}', "");
Expect(0, 65520, '\p{Blk=	 HALF_and_Full_forms}', "");
Expect(1, 65520, '\p{^Blk=	 HALF_and_Full_forms}', "");
Expect(1, 65520, '\P{Blk=	 HALF_and_Full_forms}', "");
Expect(0, 65520, '\P{^Blk=	 HALF_and_Full_forms}', "");
Error('\p{Is_Block=:=	-Halfwidth_And_Fullwidth_Forms}');
Error('\P{Is_Block=:=	-Halfwidth_And_Fullwidth_Forms}');
Expect(1, 65519, '\p{Is_Block=halfwidthandfullwidthforms}', "");
Expect(0, 65519, '\p{^Is_Block=halfwidthandfullwidthforms}', "");
Expect(0, 65519, '\P{Is_Block=halfwidthandfullwidthforms}', "");
Expect(1, 65519, '\P{^Is_Block=halfwidthandfullwidthforms}', "");
Expect(0, 65520, '\p{Is_Block=halfwidthandfullwidthforms}', "");
Expect(1, 65520, '\p{^Is_Block=halfwidthandfullwidthforms}', "");
Expect(1, 65520, '\P{Is_Block=halfwidthandfullwidthforms}', "");
Expect(0, 65520, '\P{^Is_Block=halfwidthandfullwidthforms}', "");
Expect(1, 65519, '\p{Is_Block:  Halfwidth_and_fullwidth_Forms}', "");
Expect(0, 65519, '\p{^Is_Block:  Halfwidth_and_fullwidth_Forms}', "");
Expect(0, 65519, '\P{Is_Block:  Halfwidth_and_fullwidth_Forms}', "");
Expect(1, 65519, '\P{^Is_Block:  Halfwidth_and_fullwidth_Forms}', "");
Expect(0, 65520, '\p{Is_Block:  Halfwidth_and_fullwidth_Forms}', "");
Expect(1, 65520, '\p{^Is_Block:  Halfwidth_and_fullwidth_Forms}', "");
Expect(1, 65520, '\P{Is_Block:  Halfwidth_and_fullwidth_Forms}', "");
Expect(0, 65520, '\P{^Is_Block:  Halfwidth_and_fullwidth_Forms}', "");
Error('\p{Is_Blk=:=-Half_and_full_FORMS}');
Error('\P{Is_Blk=:=-Half_and_full_FORMS}');
Expect(1, 65519, '\p{Is_Blk=halfandfullforms}', "");
Expect(0, 65519, '\p{^Is_Blk=halfandfullforms}', "");
Expect(0, 65519, '\P{Is_Blk=halfandfullforms}', "");
Expect(1, 65519, '\P{^Is_Blk=halfandfullforms}', "");
Expect(0, 65520, '\p{Is_Blk=halfandfullforms}', "");
Expect(1, 65520, '\p{^Is_Blk=halfandfullforms}', "");
Expect(1, 65520, '\P{Is_Blk=halfandfullforms}', "");
Expect(0, 65520, '\P{^Is_Blk=halfandfullforms}', "");
Expect(1, 65519, '\p{Is_Blk=	 HALF_And_full_Forms}', "");
Expect(0, 65519, '\p{^Is_Blk=	 HALF_And_full_Forms}', "");
Expect(0, 65519, '\P{Is_Blk=	 HALF_And_full_Forms}', "");
Expect(1, 65519, '\P{^Is_Blk=	 HALF_And_full_Forms}', "");
Expect(0, 65520, '\p{Is_Blk=	 HALF_And_full_Forms}', "");
Expect(1, 65520, '\p{^Is_Blk=	 HALF_And_full_Forms}', "");
Expect(1, 65520, '\P{Is_Blk=	 HALF_And_full_Forms}', "");
Expect(0, 65520, '\P{^Is_Blk=	 HALF_And_full_Forms}', "");
Error('\p{Block: 		COMBINING_Half_Marks:=}');
Error('\P{Block: 		COMBINING_Half_Marks:=}');
Expect(1, 65071, '\p{Block:   combininghalfmarks}', "");
Expect(0, 65071, '\p{^Block:   combininghalfmarks}', "");
Expect(0, 65071, '\P{Block:   combininghalfmarks}', "");
Expect(1, 65071, '\P{^Block:   combininghalfmarks}', "");
Expect(0, 65072, '\p{Block:   combininghalfmarks}', "");
Expect(1, 65072, '\p{^Block:   combininghalfmarks}', "");
Expect(1, 65072, '\P{Block:   combininghalfmarks}', "");
Expect(0, 65072, '\P{^Block:   combininghalfmarks}', "");
Expect(1, 65071, '\p{Block=_Combining_HALF_MARKS}', "");
Expect(0, 65071, '\p{^Block=_Combining_HALF_MARKS}', "");
Expect(0, 65071, '\P{Block=_Combining_HALF_MARKS}', "");
Expect(1, 65071, '\P{^Block=_Combining_HALF_MARKS}', "");
Expect(0, 65072, '\p{Block=_Combining_HALF_MARKS}', "");
Expect(1, 65072, '\p{^Block=_Combining_HALF_MARKS}', "");
Expect(1, 65072, '\P{Block=_Combining_HALF_MARKS}', "");
Expect(0, 65072, '\P{^Block=_Combining_HALF_MARKS}', "");
Error('\p{Blk=/a/	-HALF_Marks}');
Error('\P{Blk=/a/	-HALF_Marks}');
Expect(1, 65071, '\p{Blk=halfmarks}', "");
Expect(0, 65071, '\p{^Blk=halfmarks}', "");
Expect(0, 65071, '\P{Blk=halfmarks}', "");
Expect(1, 65071, '\P{^Blk=halfmarks}', "");
Expect(0, 65072, '\p{Blk=halfmarks}', "");
Expect(1, 65072, '\p{^Blk=halfmarks}', "");
Expect(1, 65072, '\P{Blk=halfmarks}', "");
Expect(0, 65072, '\P{^Blk=halfmarks}', "");
Expect(1, 65071, '\p{Blk=- half_marks}', "");
Expect(0, 65071, '\p{^Blk=- half_marks}', "");
Expect(0, 65071, '\P{Blk=- half_marks}', "");
Expect(1, 65071, '\P{^Blk=- half_marks}', "");
Expect(0, 65072, '\p{Blk=- half_marks}', "");
Expect(1, 65072, '\p{^Blk=- half_marks}', "");
Expect(1, 65072, '\P{Blk=- half_marks}', "");
Expect(0, 65072, '\P{^Blk=- half_marks}', "");
Error('\p{Is_Block=-/a/COMBINING_Half_Marks}');
Error('\P{Is_Block=-/a/COMBINING_Half_Marks}');
Expect(1, 65071, '\p{Is_Block=combininghalfmarks}', "");
Expect(0, 65071, '\p{^Is_Block=combininghalfmarks}', "");
Expect(0, 65071, '\P{Is_Block=combininghalfmarks}', "");
Expect(1, 65071, '\P{^Is_Block=combininghalfmarks}', "");
Expect(0, 65072, '\p{Is_Block=combininghalfmarks}', "");
Expect(1, 65072, '\p{^Is_Block=combininghalfmarks}', "");
Expect(1, 65072, '\P{Is_Block=combininghalfmarks}', "");
Expect(0, 65072, '\P{^Is_Block=combininghalfmarks}', "");
Expect(1, 65071, '\p{Is_Block=--Combining_HALF_Marks}', "");
Expect(0, 65071, '\p{^Is_Block=--Combining_HALF_Marks}', "");
Expect(0, 65071, '\P{Is_Block=--Combining_HALF_Marks}', "");
Expect(1, 65071, '\P{^Is_Block=--Combining_HALF_Marks}', "");
Expect(0, 65072, '\p{Is_Block=--Combining_HALF_Marks}', "");
Expect(1, 65072, '\p{^Is_Block=--Combining_HALF_Marks}', "");
Expect(1, 65072, '\P{Is_Block=--Combining_HALF_Marks}', "");
Expect(0, 65072, '\P{^Is_Block=--Combining_HALF_Marks}', "");
Error('\p{Is_Blk=_ half_marks:=}');
Error('\P{Is_Blk=_ half_marks:=}');
Expect(1, 65071, '\p{Is_Blk=halfmarks}', "");
Expect(0, 65071, '\p{^Is_Blk=halfmarks}', "");
Expect(0, 65071, '\P{Is_Blk=halfmarks}', "");
Expect(1, 65071, '\P{^Is_Blk=halfmarks}', "");
Expect(0, 65072, '\p{Is_Blk=halfmarks}', "");
Expect(1, 65072, '\p{^Is_Blk=halfmarks}', "");
Expect(1, 65072, '\P{Is_Blk=halfmarks}', "");
Expect(0, 65072, '\P{^Is_Blk=halfmarks}', "");
Expect(1, 65071, '\p{Is_Blk=--Half_marks}', "");
Expect(0, 65071, '\p{^Is_Blk=--Half_marks}', "");
Expect(0, 65071, '\P{Is_Blk=--Half_marks}', "");
Expect(1, 65071, '\P{^Is_Blk=--Half_marks}', "");
Expect(0, 65072, '\p{Is_Blk=--Half_marks}', "");
Expect(1, 65072, '\p{^Is_Blk=--Half_marks}', "");
Expect(1, 65072, '\P{Is_Blk=--Half_marks}', "");
Expect(0, 65072, '\P{^Is_Blk=--Half_marks}', "");
Error('\p{Block=/a/_-Hangul_Syllables}');
Error('\P{Block=/a/_-Hangul_Syllables}');
Expect(1, 55215, '\p{Block=hangulsyllables}', "");
Expect(0, 55215, '\p{^Block=hangulsyllables}', "");
Expect(0, 55215, '\P{Block=hangulsyllables}', "");
Expect(1, 55215, '\P{^Block=hangulsyllables}', "");
Expect(0, 55216, '\p{Block=hangulsyllables}', "");
Expect(1, 55216, '\p{^Block=hangulsyllables}', "");
Expect(1, 55216, '\P{Block=hangulsyllables}', "");
Expect(0, 55216, '\P{^Block=hangulsyllables}', "");
Expect(1, 55215, '\p{Block= hangul_syllables}', "");
Expect(0, 55215, '\p{^Block= hangul_syllables}', "");
Expect(0, 55215, '\P{Block= hangul_syllables}', "");
Expect(1, 55215, '\P{^Block= hangul_syllables}', "");
Expect(0, 55216, '\p{Block= hangul_syllables}', "");
Expect(1, 55216, '\p{^Block= hangul_syllables}', "");
Expect(1, 55216, '\P{Block= hangul_syllables}', "");
Expect(0, 55216, '\P{^Block= hangul_syllables}', "");
Error('\p{Blk=	_hangul/a/}');
Error('\P{Blk=	_hangul/a/}');
Expect(1, 55215, '\p{Blk=hangul}', "");
Expect(0, 55215, '\p{^Blk=hangul}', "");
Expect(0, 55215, '\P{Blk=hangul}', "");
Expect(1, 55215, '\P{^Blk=hangul}', "");
Expect(0, 55216, '\p{Blk=hangul}', "");
Expect(1, 55216, '\p{^Blk=hangul}', "");
Expect(1, 55216, '\P{Blk=hangul}', "");
Expect(0, 55216, '\P{^Blk=hangul}', "");
Expect(1, 55215, '\p{Blk=		hangul}', "");
Expect(0, 55215, '\p{^Blk=		hangul}', "");
Expect(0, 55215, '\P{Blk=		hangul}', "");
Expect(1, 55215, '\P{^Blk=		hangul}', "");
Expect(0, 55216, '\p{Blk=		hangul}', "");
Expect(1, 55216, '\p{^Blk=		hangul}', "");
Expect(1, 55216, '\P{Blk=		hangul}', "");
Expect(0, 55216, '\P{^Blk=		hangul}', "");
Error('\p{Is_Block=_:=hangul_syllables}');
Error('\P{Is_Block=_:=hangul_syllables}');
Expect(1, 55215, '\p{Is_Block=hangulsyllables}', "");
Expect(0, 55215, '\p{^Is_Block=hangulsyllables}', "");
Expect(0, 55215, '\P{Is_Block=hangulsyllables}', "");
Expect(1, 55215, '\P{^Is_Block=hangulsyllables}', "");
Expect(0, 55216, '\p{Is_Block=hangulsyllables}', "");
Expect(1, 55216, '\p{^Is_Block=hangulsyllables}', "");
Expect(1, 55216, '\P{Is_Block=hangulsyllables}', "");
Expect(0, 55216, '\P{^Is_Block=hangulsyllables}', "");
Expect(1, 55215, '\p{Is_Block:	_	HANGUL_Syllables}', "");
Expect(0, 55215, '\p{^Is_Block:	_	HANGUL_Syllables}', "");
Expect(0, 55215, '\P{Is_Block:	_	HANGUL_Syllables}', "");
Expect(1, 55215, '\P{^Is_Block:	_	HANGUL_Syllables}', "");
Expect(0, 55216, '\p{Is_Block:	_	HANGUL_Syllables}', "");
Expect(1, 55216, '\p{^Is_Block:	_	HANGUL_Syllables}', "");
Expect(1, 55216, '\P{Is_Block:	_	HANGUL_Syllables}', "");
Expect(0, 55216, '\P{^Is_Block:	_	HANGUL_Syllables}', "");
Error('\p{Is_Blk=-	Hangul/a/}');
Error('\P{Is_Blk=-	Hangul/a/}');
Expect(1, 55215, '\p{Is_Blk=hangul}', "");
Expect(0, 55215, '\p{^Is_Blk=hangul}', "");
Expect(0, 55215, '\P{Is_Blk=hangul}', "");
Expect(1, 55215, '\P{^Is_Blk=hangul}', "");
Expect(0, 55216, '\p{Is_Blk=hangul}', "");
Expect(1, 55216, '\p{^Is_Blk=hangul}', "");
Expect(1, 55216, '\P{Is_Blk=hangul}', "");
Expect(0, 55216, '\P{^Is_Blk=hangul}', "");
Expect(1, 55215, '\p{Is_Blk=-HANGUL}', "");
Expect(0, 55215, '\p{^Is_Blk=-HANGUL}', "");
Expect(0, 55215, '\P{Is_Blk=-HANGUL}', "");
Expect(1, 55215, '\P{^Is_Blk=-HANGUL}', "");
Expect(0, 55216, '\p{Is_Blk=-HANGUL}', "");
Expect(1, 55216, '\p{^Is_Blk=-HANGUL}', "");
Expect(1, 55216, '\P{Is_Blk=-HANGUL}', "");
Expect(0, 55216, '\P{^Is_Blk=-HANGUL}', "");
Error('\p{Block=-:=Hanunoo}');
Error('\P{Block=-:=Hanunoo}');
Expect(1, 5951, '\p{Block=hanunoo}', "");
Expect(0, 5951, '\p{^Block=hanunoo}', "");
Expect(0, 5951, '\P{Block=hanunoo}', "");
Expect(1, 5951, '\P{^Block=hanunoo}', "");
Expect(0, 5952, '\p{Block=hanunoo}', "");
Expect(1, 5952, '\p{^Block=hanunoo}', "");
Expect(1, 5952, '\P{Block=hanunoo}', "");
Expect(0, 5952, '\P{^Block=hanunoo}', "");
Expect(1, 5951, '\p{Block=_Hanunoo}', "");
Expect(0, 5951, '\p{^Block=_Hanunoo}', "");
Expect(0, 5951, '\P{Block=_Hanunoo}', "");
Expect(1, 5951, '\P{^Block=_Hanunoo}', "");
Expect(0, 5952, '\p{Block=_Hanunoo}', "");
Expect(1, 5952, '\p{^Block=_Hanunoo}', "");
Expect(1, 5952, '\P{Block=_Hanunoo}', "");
Expect(0, 5952, '\P{^Block=_Hanunoo}', "");
Error('\p{Blk=	:=HANUNOO}');
Error('\P{Blk=	:=HANUNOO}');
Expect(1, 5951, '\p{Blk=hanunoo}', "");
Expect(0, 5951, '\p{^Blk=hanunoo}', "");
Expect(0, 5951, '\P{Blk=hanunoo}', "");
Expect(1, 5951, '\P{^Blk=hanunoo}', "");
Expect(0, 5952, '\p{Blk=hanunoo}', "");
Expect(1, 5952, '\p{^Blk=hanunoo}', "");
Expect(1, 5952, '\P{Blk=hanunoo}', "");
Expect(0, 5952, '\P{^Blk=hanunoo}', "");
Expect(1, 5951, '\p{Blk=-Hanunoo}', "");
Expect(0, 5951, '\p{^Blk=-Hanunoo}', "");
Expect(0, 5951, '\P{Blk=-Hanunoo}', "");
Expect(1, 5951, '\P{^Blk=-Hanunoo}', "");
Expect(0, 5952, '\p{Blk=-Hanunoo}', "");
Expect(1, 5952, '\p{^Blk=-Hanunoo}', "");
Expect(1, 5952, '\P{Blk=-Hanunoo}', "");
Expect(0, 5952, '\P{^Blk=-Hanunoo}', "");
Error('\p{Is_Block=-hanunoo/a/}');
Error('\P{Is_Block=-hanunoo/a/}');
Expect(1, 5951, '\p{Is_Block=hanunoo}', "");
Expect(0, 5951, '\p{^Is_Block=hanunoo}', "");
Expect(0, 5951, '\P{Is_Block=hanunoo}', "");
Expect(1, 5951, '\P{^Is_Block=hanunoo}', "");
Expect(0, 5952, '\p{Is_Block=hanunoo}', "");
Expect(1, 5952, '\p{^Is_Block=hanunoo}', "");
Expect(1, 5952, '\P{Is_Block=hanunoo}', "");
Expect(0, 5952, '\P{^Is_Block=hanunoo}', "");
Expect(1, 5951, '\p{Is_Block=	 HANUNOO}', "");
Expect(0, 5951, '\p{^Is_Block=	 HANUNOO}', "");
Expect(0, 5951, '\P{Is_Block=	 HANUNOO}', "");
Expect(1, 5951, '\P{^Is_Block=	 HANUNOO}', "");
Expect(0, 5952, '\p{Is_Block=	 HANUNOO}', "");
Expect(1, 5952, '\p{^Is_Block=	 HANUNOO}', "");
Expect(1, 5952, '\P{Is_Block=	 HANUNOO}', "");
Expect(0, 5952, '\P{^Is_Block=	 HANUNOO}', "");
Error('\p{Is_Blk=:=_	HANUNOO}');
Error('\P{Is_Blk=:=_	HANUNOO}');
Expect(1, 5951, '\p{Is_Blk=hanunoo}', "");
Expect(0, 5951, '\p{^Is_Blk=hanunoo}', "");
Expect(0, 5951, '\P{Is_Blk=hanunoo}', "");
Expect(1, 5951, '\P{^Is_Blk=hanunoo}', "");
Expect(0, 5952, '\p{Is_Blk=hanunoo}', "");
Expect(1, 5952, '\p{^Is_Blk=hanunoo}', "");
Expect(1, 5952, '\P{Is_Blk=hanunoo}', "");
Expect(0, 5952, '\P{^Is_Blk=hanunoo}', "");
Expect(1, 5951, '\p{Is_Blk=- Hanunoo}', "");
Expect(0, 5951, '\p{^Is_Blk=- Hanunoo}', "");
Expect(0, 5951, '\P{Is_Blk=- Hanunoo}', "");
Expect(1, 5951, '\P{^Is_Blk=- Hanunoo}', "");
Expect(0, 5952, '\p{Is_Blk=- Hanunoo}', "");
Expect(1, 5952, '\p{^Is_Blk=- Hanunoo}', "");
Expect(1, 5952, '\P{Is_Blk=- Hanunoo}', "");
Expect(0, 5952, '\P{^Is_Blk=- Hanunoo}', "");
Error('\p{Block=_HEBREW/a/}');
Error('\P{Block=_HEBREW/a/}');
Expect(1, 1535, '\p{Block=hebrew}', "");
Expect(0, 1535, '\p{^Block=hebrew}', "");
Expect(0, 1535, '\P{Block=hebrew}', "");
Expect(1, 1535, '\P{^Block=hebrew}', "");
Expect(0, 1536, '\p{Block=hebrew}', "");
Expect(1, 1536, '\p{^Block=hebrew}', "");
Expect(1, 1536, '\P{Block=hebrew}', "");
Expect(0, 1536, '\P{^Block=hebrew}', "");
Expect(1, 1535, '\p{Block=_-HEBREW}', "");
Expect(0, 1535, '\p{^Block=_-HEBREW}', "");
Expect(0, 1535, '\P{Block=_-HEBREW}', "");
Expect(1, 1535, '\P{^Block=_-HEBREW}', "");
Expect(0, 1536, '\p{Block=_-HEBREW}', "");
Expect(1, 1536, '\p{^Block=_-HEBREW}', "");
Expect(1, 1536, '\P{Block=_-HEBREW}', "");
Expect(0, 1536, '\P{^Block=_-HEBREW}', "");
Error('\p{Blk: _-Hebrew/a/}');
Error('\P{Blk: _-Hebrew/a/}');
Expect(1, 1535, '\p{Blk=hebrew}', "");
Expect(0, 1535, '\p{^Blk=hebrew}', "");
Expect(0, 1535, '\P{Blk=hebrew}', "");
Expect(1, 1535, '\P{^Blk=hebrew}', "");
Expect(0, 1536, '\p{Blk=hebrew}', "");
Expect(1, 1536, '\p{^Blk=hebrew}', "");
Expect(1, 1536, '\P{Blk=hebrew}', "");
Expect(0, 1536, '\P{^Blk=hebrew}', "");
Expect(1, 1535, '\p{Blk=_ hebrew}', "");
Expect(0, 1535, '\p{^Blk=_ hebrew}', "");
Expect(0, 1535, '\P{Blk=_ hebrew}', "");
Expect(1, 1535, '\P{^Blk=_ hebrew}', "");
Expect(0, 1536, '\p{Blk=_ hebrew}', "");
Expect(1, 1536, '\p{^Blk=_ hebrew}', "");
Expect(1, 1536, '\P{Blk=_ hebrew}', "");
Expect(0, 1536, '\P{^Blk=_ hebrew}', "");
Error('\p{Is_Block=/a/_	Hebrew}');
Error('\P{Is_Block=/a/_	Hebrew}');
Expect(1, 1535, '\p{Is_Block: hebrew}', "");
Expect(0, 1535, '\p{^Is_Block: hebrew}', "");
Expect(0, 1535, '\P{Is_Block: hebrew}', "");
Expect(1, 1535, '\P{^Is_Block: hebrew}', "");
Expect(0, 1536, '\p{Is_Block: hebrew}', "");
Expect(1, 1536, '\p{^Is_Block: hebrew}', "");
Expect(1, 1536, '\P{Is_Block: hebrew}', "");
Expect(0, 1536, '\P{^Is_Block: hebrew}', "");
Expect(1, 1535, '\p{Is_Block=_Hebrew}', "");
Expect(0, 1535, '\p{^Is_Block=_Hebrew}', "");
Expect(0, 1535, '\P{Is_Block=_Hebrew}', "");
Expect(1, 1535, '\P{^Is_Block=_Hebrew}', "");
Expect(0, 1536, '\p{Is_Block=_Hebrew}', "");
Expect(1, 1536, '\p{^Is_Block=_Hebrew}', "");
Expect(1, 1536, '\P{Is_Block=_Hebrew}', "");
Expect(0, 1536, '\P{^Is_Block=_Hebrew}', "");
Error('\p{Is_Blk=/a/ -hebrew}');
Error('\P{Is_Blk=/a/ -hebrew}');
Expect(1, 1535, '\p{Is_Blk=hebrew}', "");
Expect(0, 1535, '\p{^Is_Blk=hebrew}', "");
Expect(0, 1535, '\P{Is_Blk=hebrew}', "");
Expect(1, 1535, '\P{^Is_Blk=hebrew}', "");
Expect(0, 1536, '\p{Is_Blk=hebrew}', "");
Expect(1, 1536, '\p{^Is_Blk=hebrew}', "");
Expect(1, 1536, '\P{Is_Blk=hebrew}', "");
Expect(0, 1536, '\P{^Is_Blk=hebrew}', "");
Expect(1, 1535, '\p{Is_Blk=	 hebrew}', "");
Expect(0, 1535, '\p{^Is_Blk=	 hebrew}', "");
Expect(0, 1535, '\P{Is_Blk=	 hebrew}', "");
Expect(1, 1535, '\P{^Is_Blk=	 hebrew}', "");
Expect(0, 1536, '\p{Is_Blk=	 hebrew}', "");
Expect(1, 1536, '\p{^Is_Blk=	 hebrew}', "");
Expect(1, 1536, '\P{Is_Blk=	 hebrew}', "");
Expect(0, 1536, '\P{^Is_Blk=	 hebrew}', "");
Error('\p{Block=_ HIGH_Private_use_Surrogates:=}');
Error('\P{Block=_ HIGH_Private_use_Surrogates:=}');
Expect(1, 56319, '\p{Block=highprivateusesurrogates}', "");
Expect(0, 56319, '\p{^Block=highprivateusesurrogates}', "");
Expect(0, 56319, '\P{Block=highprivateusesurrogates}', "");
Expect(1, 56319, '\P{^Block=highprivateusesurrogates}', "");
Expect(0, 57344, '\p{Block=highprivateusesurrogates}', "");
Expect(1, 57344, '\p{^Block=highprivateusesurrogates}', "");
Expect(1, 57344, '\P{Block=highprivateusesurrogates}', "");
Expect(0, 57344, '\P{^Block=highprivateusesurrogates}', "");
Expect(1, 56319, '\p{Block= 	High_PRIVATE_USE_surrogates}', "");
Expect(0, 56319, '\p{^Block= 	High_PRIVATE_USE_surrogates}', "");
Expect(0, 56319, '\P{Block= 	High_PRIVATE_USE_surrogates}', "");
Expect(1, 56319, '\P{^Block= 	High_PRIVATE_USE_surrogates}', "");
Expect(0, 57344, '\p{Block= 	High_PRIVATE_USE_surrogates}', "");
Expect(1, 57344, '\p{^Block= 	High_PRIVATE_USE_surrogates}', "");
Expect(1, 57344, '\P{Block= 	High_PRIVATE_USE_surrogates}', "");
Expect(0, 57344, '\P{^Block= 	High_PRIVATE_USE_surrogates}', "");
Error('\p{Blk=/a/-_High_PU_Surrogates}');
Error('\P{Blk=/a/-_High_PU_Surrogates}');
Expect(1, 56319, '\p{Blk=highpusurrogates}', "");
Expect(0, 56319, '\p{^Blk=highpusurrogates}', "");
Expect(0, 56319, '\P{Blk=highpusurrogates}', "");
Expect(1, 56319, '\P{^Blk=highpusurrogates}', "");
Expect(0, 57344, '\p{Blk=highpusurrogates}', "");
Expect(1, 57344, '\p{^Blk=highpusurrogates}', "");
Expect(1, 57344, '\P{Blk=highpusurrogates}', "");
Expect(0, 57344, '\P{^Blk=highpusurrogates}', "");
Expect(1, 56319, '\p{Blk=- High_PU_surrogates}', "");
Expect(0, 56319, '\p{^Blk=- High_PU_surrogates}', "");
Expect(0, 56319, '\P{Blk=- High_PU_surrogates}', "");
Expect(1, 56319, '\P{^Blk=- High_PU_surrogates}', "");
Expect(0, 57344, '\p{Blk=- High_PU_surrogates}', "");
Expect(1, 57344, '\p{^Blk=- High_PU_surrogates}', "");
Expect(1, 57344, '\P{Blk=- High_PU_surrogates}', "");
Expect(0, 57344, '\P{^Blk=- High_PU_surrogates}', "");
Error('\p{Is_Block=/a/high_Private_USE_SURROGATES}');
Error('\P{Is_Block=/a/high_Private_USE_SURROGATES}');
Expect(1, 56319, '\p{Is_Block=highprivateusesurrogates}', "");
Expect(0, 56319, '\p{^Is_Block=highprivateusesurrogates}', "");
Expect(0, 56319, '\P{Is_Block=highprivateusesurrogates}', "");
Expect(1, 56319, '\P{^Is_Block=highprivateusesurrogates}', "");
Expect(0, 57344, '\p{Is_Block=highprivateusesurrogates}', "");
Expect(1, 57344, '\p{^Is_Block=highprivateusesurrogates}', "");
Expect(1, 57344, '\P{Is_Block=highprivateusesurrogates}', "");
Expect(0, 57344, '\P{^Is_Block=highprivateusesurrogates}', "");
Expect(1, 56319, '\p{Is_Block=	_High_private_USE_SURROGATES}', "");
Expect(0, 56319, '\p{^Is_Block=	_High_private_USE_SURROGATES}', "");
Expect(0, 56319, '\P{Is_Block=	_High_private_USE_SURROGATES}', "");
Expect(1, 56319, '\P{^Is_Block=	_High_private_USE_SURROGATES}', "");
Expect(0, 57344, '\p{Is_Block=	_High_private_USE_SURROGATES}', "");
Expect(1, 57344, '\p{^Is_Block=	_High_private_USE_SURROGATES}', "");
Expect(1, 57344, '\P{Is_Block=	_High_private_USE_SURROGATES}', "");
Expect(0, 57344, '\P{^Is_Block=	_High_private_USE_SURROGATES}', "");
Error('\p{Is_Blk=_/a/High_PU_Surrogates}');
Error('\P{Is_Blk=_/a/High_PU_Surrogates}');
Expect(1, 56319, '\p{Is_Blk=highpusurrogates}', "");
Expect(0, 56319, '\p{^Is_Blk=highpusurrogates}', "");
Expect(0, 56319, '\P{Is_Blk=highpusurrogates}', "");
Expect(1, 56319, '\P{^Is_Blk=highpusurrogates}', "");
Expect(0, 57344, '\p{Is_Blk=highpusurrogates}', "");
Expect(1, 57344, '\p{^Is_Blk=highpusurrogates}', "");
Expect(1, 57344, '\P{Is_Blk=highpusurrogates}', "");
Expect(0, 57344, '\P{^Is_Blk=highpusurrogates}', "");
Expect(1, 56319, '\p{Is_Blk:	 High_PU_Surrogates}', "");
Expect(0, 56319, '\p{^Is_Blk:	 High_PU_Surrogates}', "");
Expect(0, 56319, '\P{Is_Blk:	 High_PU_Surrogates}', "");
Expect(1, 56319, '\P{^Is_Blk:	 High_PU_Surrogates}', "");
Expect(0, 57344, '\p{Is_Blk:	 High_PU_Surrogates}', "");
Expect(1, 57344, '\p{^Is_Blk:	 High_PU_Surrogates}', "");
Expect(1, 57344, '\P{Is_Blk:	 High_PU_Surrogates}', "");
Expect(0, 57344, '\P{^Is_Blk:	 High_PU_Surrogates}', "");
Error('\p{Block=-/a/HIGH_Surrogates}');
Error('\P{Block=-/a/HIGH_Surrogates}');
Expect(1, 56191, '\p{Block=highsurrogates}', "");
Expect(0, 56191, '\p{^Block=highsurrogates}', "");
Expect(0, 56191, '\P{Block=highsurrogates}', "");
Expect(1, 56191, '\P{^Block=highsurrogates}', "");
Expect(0, 57344, '\p{Block=highsurrogates}', "");
Expect(1, 57344, '\p{^Block=highsurrogates}', "");
Expect(1, 57344, '\P{Block=highsurrogates}', "");
Expect(0, 57344, '\P{^Block=highsurrogates}', "");
Expect(1, 56191, '\p{Block=HIGH_Surrogates}', "");
Expect(0, 56191, '\p{^Block=HIGH_Surrogates}', "");
Expect(0, 56191, '\P{Block=HIGH_Surrogates}', "");
Expect(1, 56191, '\P{^Block=HIGH_Surrogates}', "");
Expect(0, 57344, '\p{Block=HIGH_Surrogates}', "");
Expect(1, 57344, '\p{^Block=HIGH_Surrogates}', "");
Expect(1, 57344, '\P{Block=HIGH_Surrogates}', "");
Expect(0, 57344, '\P{^Block=HIGH_Surrogates}', "");
Error('\p{Blk=_ HIGH_Surrogates:=}');
Error('\P{Blk=_ HIGH_Surrogates:=}');
Expect(1, 56191, '\p{Blk=highsurrogates}', "");
Expect(0, 56191, '\p{^Blk=highsurrogates}', "");
Expect(0, 56191, '\P{Blk=highsurrogates}', "");
Expect(1, 56191, '\P{^Blk=highsurrogates}', "");
Expect(0, 57344, '\p{Blk=highsurrogates}', "");
Expect(1, 57344, '\p{^Blk=highsurrogates}', "");
Expect(1, 57344, '\P{Blk=highsurrogates}', "");
Expect(0, 57344, '\P{^Blk=highsurrogates}', "");
Expect(1, 56191, '\p{Blk=_ HIGH_Surrogates}', "");
Expect(0, 56191, '\p{^Blk=_ HIGH_Surrogates}', "");
Expect(0, 56191, '\P{Blk=_ HIGH_Surrogates}', "");
Expect(1, 56191, '\P{^Blk=_ HIGH_Surrogates}', "");
Expect(0, 57344, '\p{Blk=_ HIGH_Surrogates}', "");
Expect(1, 57344, '\p{^Blk=_ HIGH_Surrogates}', "");
Expect(1, 57344, '\P{Blk=_ HIGH_Surrogates}', "");
Expect(0, 57344, '\P{^Blk=_ HIGH_Surrogates}', "");
Error('\p{Is_Block= /a/High_SURROGATES}');
Error('\P{Is_Block= /a/High_SURROGATES}');
Expect(1, 56191, '\p{Is_Block=highsurrogates}', "");
Expect(0, 56191, '\p{^Is_Block=highsurrogates}', "");
Expect(0, 56191, '\P{Is_Block=highsurrogates}', "");
Expect(1, 56191, '\P{^Is_Block=highsurrogates}', "");
Expect(0, 57344, '\p{Is_Block=highsurrogates}', "");
Expect(1, 57344, '\p{^Is_Block=highsurrogates}', "");
Expect(1, 57344, '\P{Is_Block=highsurrogates}', "");
Expect(0, 57344, '\P{^Is_Block=highsurrogates}', "");
Expect(1, 56191, '\p{Is_Block=  High_SURROGATES}', "");
Expect(0, 56191, '\p{^Is_Block=  High_SURROGATES}', "");
Expect(0, 56191, '\P{Is_Block=  High_SURROGATES}', "");
Expect(1, 56191, '\P{^Is_Block=  High_SURROGATES}', "");
Expect(0, 57344, '\p{Is_Block=  High_SURROGATES}', "");
Expect(1, 57344, '\p{^Is_Block=  High_SURROGATES}', "");
Expect(1, 57344, '\P{Is_Block=  High_SURROGATES}', "");
Expect(0, 57344, '\P{^Is_Block=  High_SURROGATES}', "");
Error('\p{Is_Blk: 	High_SURROGATES:=}');
Error('\P{Is_Blk: 	High_SURROGATES:=}');
Expect(1, 56191, '\p{Is_Blk:   highsurrogates}', "");
Expect(0, 56191, '\p{^Is_Blk:   highsurrogates}', "");
Expect(0, 56191, '\P{Is_Blk:   highsurrogates}', "");
Expect(1, 56191, '\P{^Is_Blk:   highsurrogates}', "");
Expect(0, 57344, '\p{Is_Blk:   highsurrogates}', "");
Expect(1, 57344, '\p{^Is_Blk:   highsurrogates}', "");
Expect(1, 57344, '\P{Is_Blk:   highsurrogates}', "");
Expect(0, 57344, '\P{^Is_Blk:   highsurrogates}', "");
Expect(1, 56191, '\p{Is_Blk=High_SURROGATES}', "");
Expect(0, 56191, '\p{^Is_Blk=High_SURROGATES}', "");
Expect(0, 56191, '\P{Is_Blk=High_SURROGATES}', "");
Expect(1, 56191, '\P{^Is_Blk=High_SURROGATES}', "");
Expect(0, 57344, '\p{Is_Blk=High_SURROGATES}', "");
Expect(1, 57344, '\p{^Is_Blk=High_SURROGATES}', "");
Expect(1, 57344, '\P{Is_Blk=High_SURROGATES}', "");
Expect(0, 57344, '\P{^Is_Blk=High_SURROGATES}', "");
Error('\p{Block:   /a/-Hiragana}');
Error('\P{Block:   /a/-Hiragana}');
Expect(1, 12447, '\p{Block:	hiragana}', "");
Expect(0, 12447, '\p{^Block:	hiragana}', "");
Expect(0, 12447, '\P{Block:	hiragana}', "");
Expect(1, 12447, '\P{^Block:	hiragana}', "");
Expect(0, 12448, '\p{Block:	hiragana}', "");
Expect(1, 12448, '\p{^Block:	hiragana}', "");
Expect(1, 12448, '\P{Block:	hiragana}', "");
Expect(0, 12448, '\P{^Block:	hiragana}', "");
Expect(1, 12447, '\p{Block: HIRAGANA}', "");
Expect(0, 12447, '\p{^Block: HIRAGANA}', "");
Expect(0, 12447, '\P{Block: HIRAGANA}', "");
Expect(1, 12447, '\P{^Block: HIRAGANA}', "");
Expect(0, 12448, '\p{Block: HIRAGANA}', "");
Expect(1, 12448, '\p{^Block: HIRAGANA}', "");
Expect(1, 12448, '\P{Block: HIRAGANA}', "");
Expect(0, 12448, '\P{^Block: HIRAGANA}', "");
Error('\p{Blk=/a/	_Hiragana}');
Error('\P{Blk=/a/	_Hiragana}');
Expect(1, 12447, '\p{Blk: hiragana}', "");
Expect(0, 12447, '\p{^Blk: hiragana}', "");
Expect(0, 12447, '\P{Blk: hiragana}', "");
Expect(1, 12447, '\P{^Blk: hiragana}', "");
Expect(0, 12448, '\p{Blk: hiragana}', "");
Expect(1, 12448, '\p{^Blk: hiragana}', "");
Expect(1, 12448, '\P{Blk: hiragana}', "");
Expect(0, 12448, '\P{^Blk: hiragana}', "");
Expect(1, 12447, '\p{Blk= -Hiragana}', "");
Expect(0, 12447, '\p{^Blk= -Hiragana}', "");
Expect(0, 12447, '\P{Blk= -Hiragana}', "");
Expect(1, 12447, '\P{^Blk= -Hiragana}', "");
Expect(0, 12448, '\p{Blk= -Hiragana}', "");
Expect(1, 12448, '\p{^Blk= -Hiragana}', "");
Expect(1, 12448, '\P{Blk= -Hiragana}', "");
Expect(0, 12448, '\P{^Blk= -Hiragana}', "");
Error('\p{Is_Block::=_ Hiragana}');
Error('\P{Is_Block::=_ Hiragana}');
Expect(1, 12447, '\p{Is_Block: hiragana}', "");
Expect(0, 12447, '\p{^Is_Block: hiragana}', "");
Expect(0, 12447, '\P{Is_Block: hiragana}', "");
Expect(1, 12447, '\P{^Is_Block: hiragana}', "");
Expect(0, 12448, '\p{Is_Block: hiragana}', "");
Expect(1, 12448, '\p{^Is_Block: hiragana}', "");
Expect(1, 12448, '\P{Is_Block: hiragana}', "");
Expect(0, 12448, '\P{^Is_Block: hiragana}', "");
Expect(1, 12447, '\p{Is_Block=-Hiragana}', "");
Expect(0, 12447, '\p{^Is_Block=-Hiragana}', "");
Expect(0, 12447, '\P{Is_Block=-Hiragana}', "");
Expect(1, 12447, '\P{^Is_Block=-Hiragana}', "");
Expect(0, 12448, '\p{Is_Block=-Hiragana}', "");
Expect(1, 12448, '\p{^Is_Block=-Hiragana}', "");
Expect(1, 12448, '\P{Is_Block=-Hiragana}', "");
Expect(0, 12448, '\P{^Is_Block=-Hiragana}', "");
Error('\p{Is_Blk=:=	_Hiragana}');
Error('\P{Is_Blk=:=	_Hiragana}');
Expect(1, 12447, '\p{Is_Blk=hiragana}', "");
Expect(0, 12447, '\p{^Is_Blk=hiragana}', "");
Expect(0, 12447, '\P{Is_Blk=hiragana}', "");
Expect(1, 12447, '\P{^Is_Blk=hiragana}', "");
Expect(0, 12448, '\p{Is_Blk=hiragana}', "");
Expect(1, 12448, '\p{^Is_Blk=hiragana}', "");
Expect(1, 12448, '\P{Is_Blk=hiragana}', "");
Expect(0, 12448, '\P{^Is_Blk=hiragana}', "");
Expect(1, 12447, '\p{Is_Blk: hiragana}', "");
Expect(0, 12447, '\p{^Is_Blk: hiragana}', "");
Expect(0, 12447, '\P{Is_Blk: hiragana}', "");
Expect(1, 12447, '\P{^Is_Blk: hiragana}', "");
Expect(0, 12448, '\p{Is_Blk: hiragana}', "");
Expect(1, 12448, '\p{^Is_Blk: hiragana}', "");
Expect(1, 12448, '\P{Is_Blk: hiragana}', "");
Expect(0, 12448, '\P{^Is_Blk: hiragana}', "");
Error('\p{Block=:= Ideographic_Description_characters}');
Error('\P{Block=:= Ideographic_Description_characters}');
Expect(1, 12287, '\p{Block=ideographicdescriptioncharacters}', "");
Expect(0, 12287, '\p{^Block=ideographicdescriptioncharacters}', "");
Expect(0, 12287, '\P{Block=ideographicdescriptioncharacters}', "");
Expect(1, 12287, '\P{^Block=ideographicdescriptioncharacters}', "");
Expect(0, 12288, '\p{Block=ideographicdescriptioncharacters}', "");
Expect(1, 12288, '\p{^Block=ideographicdescriptioncharacters}', "");
Expect(1, 12288, '\P{Block=ideographicdescriptioncharacters}', "");
Expect(0, 12288, '\P{^Block=ideographicdescriptioncharacters}', "");
Expect(1, 12287, '\p{Block=	 ideographic_description_Characters}', "");
Expect(0, 12287, '\p{^Block=	 ideographic_description_Characters}', "");
Expect(0, 12287, '\P{Block=	 ideographic_description_Characters}', "");
Expect(1, 12287, '\P{^Block=	 ideographic_description_Characters}', "");
Expect(0, 12288, '\p{Block=	 ideographic_description_Characters}', "");
Expect(1, 12288, '\p{^Block=	 ideographic_description_Characters}', "");
Expect(1, 12288, '\P{Block=	 ideographic_description_Characters}', "");
Expect(0, 12288, '\P{^Block=	 ideographic_description_Characters}', "");
Error('\p{Blk=-_IDC/a/}');
Error('\P{Blk=-_IDC/a/}');
Expect(1, 12287, '\p{Blk=idc}', "");
Expect(0, 12287, '\p{^Blk=idc}', "");
Expect(0, 12287, '\P{Blk=idc}', "");
Expect(1, 12287, '\P{^Blk=idc}', "");
Expect(0, 12288, '\p{Blk=idc}', "");
Expect(1, 12288, '\p{^Blk=idc}', "");
Expect(1, 12288, '\P{Blk=idc}', "");
Expect(0, 12288, '\P{^Blk=idc}', "");
Expect(1, 12287, '\p{Blk=-IDC}', "");
Expect(0, 12287, '\p{^Blk=-IDC}', "");
Expect(0, 12287, '\P{Blk=-IDC}', "");
Expect(1, 12287, '\P{^Blk=-IDC}', "");
Expect(0, 12288, '\p{Blk=-IDC}', "");
Expect(1, 12288, '\p{^Blk=-IDC}', "");
Expect(1, 12288, '\P{Blk=-IDC}', "");
Expect(0, 12288, '\P{^Blk=-IDC}', "");
Error('\p{Is_Block=-:=Ideographic_Description_characters}');
Error('\P{Is_Block=-:=Ideographic_Description_characters}');
Expect(1, 12287, '\p{Is_Block=ideographicdescriptioncharacters}', "");
Expect(0, 12287, '\p{^Is_Block=ideographicdescriptioncharacters}', "");
Expect(0, 12287, '\P{Is_Block=ideographicdescriptioncharacters}', "");
Expect(1, 12287, '\P{^Is_Block=ideographicdescriptioncharacters}', "");
Expect(0, 12288, '\p{Is_Block=ideographicdescriptioncharacters}', "");
Expect(1, 12288, '\p{^Is_Block=ideographicdescriptioncharacters}', "");
Expect(1, 12288, '\P{Is_Block=ideographicdescriptioncharacters}', "");
Expect(0, 12288, '\P{^Is_Block=ideographicdescriptioncharacters}', "");
Expect(1, 12287, '\p{Is_Block= 	ideographic_Description_Characters}', "");
Expect(0, 12287, '\p{^Is_Block= 	ideographic_Description_Characters}', "");
Expect(0, 12287, '\P{Is_Block= 	ideographic_Description_Characters}', "");
Expect(1, 12287, '\P{^Is_Block= 	ideographic_Description_Characters}', "");
Expect(0, 12288, '\p{Is_Block= 	ideographic_Description_Characters}', "");
Expect(1, 12288, '\p{^Is_Block= 	ideographic_Description_Characters}', "");
Expect(1, 12288, '\P{Is_Block= 	ideographic_Description_Characters}', "");
Expect(0, 12288, '\P{^Is_Block= 	ideographic_Description_Characters}', "");
Error('\p{Is_Blk= :=idc}');
Error('\P{Is_Blk= :=idc}');
Expect(1, 12287, '\p{Is_Blk=idc}', "");
Expect(0, 12287, '\p{^Is_Blk=idc}', "");
Expect(0, 12287, '\P{Is_Blk=idc}', "");
Expect(1, 12287, '\P{^Is_Blk=idc}', "");
Expect(0, 12288, '\p{Is_Blk=idc}', "");
Expect(1, 12288, '\p{^Is_Blk=idc}', "");
Expect(1, 12288, '\P{Is_Blk=idc}', "");
Expect(0, 12288, '\P{^Is_Blk=idc}', "");
Expect(1, 12287, '\p{Is_Blk=-_IDC}', "");
Expect(0, 12287, '\p{^Is_Blk=-_IDC}', "");
Expect(0, 12287, '\P{Is_Blk=-_IDC}', "");
Expect(1, 12287, '\P{^Is_Blk=-_IDC}', "");
Expect(0, 12288, '\p{Is_Blk=-_IDC}', "");
Expect(1, 12288, '\p{^Is_Blk=-_IDC}', "");
Expect(1, 12288, '\P{Is_Blk=-_IDC}', "");
Expect(0, 12288, '\P{^Is_Blk=-_IDC}', "");
Error('\p{Block=	_ipa_extensions:=}');
Error('\P{Block=	_ipa_extensions:=}');
Expect(1, 687, '\p{Block=ipaextensions}', "");
Expect(0, 687, '\p{^Block=ipaextensions}', "");
Expect(0, 687, '\P{Block=ipaextensions}', "");
Expect(1, 687, '\P{^Block=ipaextensions}', "");
Expect(0, 688, '\p{Block=ipaextensions}', "");
Expect(1, 688, '\p{^Block=ipaextensions}', "");
Expect(1, 688, '\P{Block=ipaextensions}', "");
Expect(0, 688, '\P{^Block=ipaextensions}', "");
Expect(1, 687, '\p{Block:   IPA_EXTENSIONS}', "");
Expect(0, 687, '\p{^Block:   IPA_EXTENSIONS}', "");
Expect(0, 687, '\P{Block:   IPA_EXTENSIONS}', "");
Expect(1, 687, '\P{^Block:   IPA_EXTENSIONS}', "");
Expect(0, 688, '\p{Block:   IPA_EXTENSIONS}', "");
Expect(1, 688, '\p{^Block:   IPA_EXTENSIONS}', "");
Expect(1, 688, '\P{Block:   IPA_EXTENSIONS}', "");
Expect(0, 688, '\P{^Block:   IPA_EXTENSIONS}', "");
Error('\p{Blk=-	IPA_EXT:=}');
Error('\P{Blk=-	IPA_EXT:=}');
Expect(1, 687, '\p{Blk=ipaext}', "");
Expect(0, 687, '\p{^Blk=ipaext}', "");
Expect(0, 687, '\P{Blk=ipaext}', "");
Expect(1, 687, '\P{^Blk=ipaext}', "");
Expect(0, 688, '\p{Blk=ipaext}', "");
Expect(1, 688, '\p{^Blk=ipaext}', "");
Expect(1, 688, '\P{Blk=ipaext}', "");
Expect(0, 688, '\P{^Blk=ipaext}', "");
Expect(1, 687, '\p{Blk= _IPA_Ext}', "");
Expect(0, 687, '\p{^Blk= _IPA_Ext}', "");
Expect(0, 687, '\P{Blk= _IPA_Ext}', "");
Expect(1, 687, '\P{^Blk= _IPA_Ext}', "");
Expect(0, 688, '\p{Blk= _IPA_Ext}', "");
Expect(1, 688, '\p{^Blk= _IPA_Ext}', "");
Expect(1, 688, '\P{Blk= _IPA_Ext}', "");
Expect(0, 688, '\P{^Blk= _IPA_Ext}', "");
Error('\p{Is_Block=-:=IPA_extensions}');
Error('\P{Is_Block=-:=IPA_extensions}');
Expect(1, 687, '\p{Is_Block=ipaextensions}', "");
Expect(0, 687, '\p{^Is_Block=ipaextensions}', "");
Expect(0, 687, '\P{Is_Block=ipaextensions}', "");
Expect(1, 687, '\P{^Is_Block=ipaextensions}', "");
Expect(0, 688, '\p{Is_Block=ipaextensions}', "");
Expect(1, 688, '\p{^Is_Block=ipaextensions}', "");
Expect(1, 688, '\P{Is_Block=ipaextensions}', "");
Expect(0, 688, '\P{^Is_Block=ipaextensions}', "");
Expect(1, 687, '\p{Is_Block=	IPA_EXTENSIONS}', "");
Expect(0, 687, '\p{^Is_Block=	IPA_EXTENSIONS}', "");
Expect(0, 687, '\P{Is_Block=	IPA_EXTENSIONS}', "");
Expect(1, 687, '\P{^Is_Block=	IPA_EXTENSIONS}', "");
Expect(0, 688, '\p{Is_Block=	IPA_EXTENSIONS}', "");
Expect(1, 688, '\p{^Is_Block=	IPA_EXTENSIONS}', "");
Expect(1, 688, '\P{Is_Block=	IPA_EXTENSIONS}', "");
Expect(0, 688, '\P{^Is_Block=	IPA_EXTENSIONS}', "");
Error('\p{Is_Blk=	/a/IPA_Ext}');
Error('\P{Is_Blk=	/a/IPA_Ext}');
Expect(1, 687, '\p{Is_Blk=ipaext}', "");
Expect(0, 687, '\p{^Is_Blk=ipaext}', "");
Expect(0, 687, '\P{Is_Blk=ipaext}', "");
Expect(1, 687, '\P{^Is_Blk=ipaext}', "");
Expect(0, 688, '\p{Is_Blk=ipaext}', "");
Expect(1, 688, '\p{^Is_Blk=ipaext}', "");
Expect(1, 688, '\P{Is_Blk=ipaext}', "");
Expect(0, 688, '\P{^Is_Blk=ipaext}', "");
Expect(1, 687, '\p{Is_Blk= _IPA_EXT}', "");
Expect(0, 687, '\p{^Is_Blk= _IPA_EXT}', "");
Expect(0, 687, '\P{Is_Blk= _IPA_EXT}', "");
Expect(1, 687, '\P{^Is_Blk= _IPA_EXT}', "");
Expect(0, 688, '\p{Is_Blk= _IPA_EXT}', "");
Expect(1, 688, '\p{^Is_Blk= _IPA_EXT}', "");
Expect(1, 688, '\P{Is_Blk= _IPA_EXT}', "");
Expect(0, 688, '\P{^Is_Blk= _IPA_EXT}', "");
Error('\p{Block=	/a/Imperial_Aramaic}');
Error('\P{Block=	/a/Imperial_Aramaic}');
Expect(1, 67679, '\p{Block=imperialaramaic}', "");
Expect(0, 67679, '\p{^Block=imperialaramaic}', "");
Expect(0, 67679, '\P{Block=imperialaramaic}', "");
Expect(1, 67679, '\P{^Block=imperialaramaic}', "");
Expect(0, 67680, '\p{Block=imperialaramaic}', "");
Expect(1, 67680, '\p{^Block=imperialaramaic}', "");
Expect(1, 67680, '\P{Block=imperialaramaic}', "");
Expect(0, 67680, '\P{^Block=imperialaramaic}', "");
Expect(1, 67679, '\p{Block=_IMPERIAL_Aramaic}', "");
Expect(0, 67679, '\p{^Block=_IMPERIAL_Aramaic}', "");
Expect(0, 67679, '\P{Block=_IMPERIAL_Aramaic}', "");
Expect(1, 67679, '\P{^Block=_IMPERIAL_Aramaic}', "");
Expect(0, 67680, '\p{Block=_IMPERIAL_Aramaic}', "");
Expect(1, 67680, '\p{^Block=_IMPERIAL_Aramaic}', "");
Expect(1, 67680, '\P{Block=_IMPERIAL_Aramaic}', "");
Expect(0, 67680, '\P{^Block=_IMPERIAL_Aramaic}', "");
Error('\p{Blk=:=_	imperial_Aramaic}');
Error('\P{Blk=:=_	imperial_Aramaic}');
Expect(1, 67679, '\p{Blk=imperialaramaic}', "");
Expect(0, 67679, '\p{^Blk=imperialaramaic}', "");
Expect(0, 67679, '\P{Blk=imperialaramaic}', "");
Expect(1, 67679, '\P{^Blk=imperialaramaic}', "");
Expect(0, 67680, '\p{Blk=imperialaramaic}', "");
Expect(1, 67680, '\p{^Blk=imperialaramaic}', "");
Expect(1, 67680, '\P{Blk=imperialaramaic}', "");
Expect(0, 67680, '\P{^Blk=imperialaramaic}', "");
Expect(1, 67679, '\p{Blk= Imperial_aramaic}', "");
Expect(0, 67679, '\p{^Blk= Imperial_aramaic}', "");
Expect(0, 67679, '\P{Blk= Imperial_aramaic}', "");
Expect(1, 67679, '\P{^Blk= Imperial_aramaic}', "");
Expect(0, 67680, '\p{Blk= Imperial_aramaic}', "");
Expect(1, 67680, '\p{^Blk= Imperial_aramaic}', "");
Expect(1, 67680, '\P{Blk= Imperial_aramaic}', "");
Expect(0, 67680, '\P{^Blk= Imperial_aramaic}', "");
Error('\p{Is_Block=:= -Imperial_ARAMAIC}');
Error('\P{Is_Block=:= -Imperial_ARAMAIC}');
Expect(1, 67679, '\p{Is_Block=imperialaramaic}', "");
Expect(0, 67679, '\p{^Is_Block=imperialaramaic}', "");
Expect(0, 67679, '\P{Is_Block=imperialaramaic}', "");
Expect(1, 67679, '\P{^Is_Block=imperialaramaic}', "");
Expect(0, 67680, '\p{Is_Block=imperialaramaic}', "");
Expect(1, 67680, '\p{^Is_Block=imperialaramaic}', "");
Expect(1, 67680, '\P{Is_Block=imperialaramaic}', "");
Expect(0, 67680, '\P{^Is_Block=imperialaramaic}', "");
Expect(1, 67679, '\p{Is_Block:   --imperial_Aramaic}', "");
Expect(0, 67679, '\p{^Is_Block:   --imperial_Aramaic}', "");
Expect(0, 67679, '\P{Is_Block:   --imperial_Aramaic}', "");
Expect(1, 67679, '\P{^Is_Block:   --imperial_Aramaic}', "");
Expect(0, 67680, '\p{Is_Block:   --imperial_Aramaic}', "");
Expect(1, 67680, '\p{^Is_Block:   --imperial_Aramaic}', "");
Expect(1, 67680, '\P{Is_Block:   --imperial_Aramaic}', "");
Expect(0, 67680, '\P{^Is_Block:   --imperial_Aramaic}', "");
Error('\p{Is_Blk=:=-imperial_Aramaic}');
Error('\P{Is_Blk=:=-imperial_Aramaic}');
Expect(1, 67679, '\p{Is_Blk=imperialaramaic}', "");
Expect(0, 67679, '\p{^Is_Blk=imperialaramaic}', "");
Expect(0, 67679, '\P{Is_Blk=imperialaramaic}', "");
Expect(1, 67679, '\P{^Is_Blk=imperialaramaic}', "");
Expect(0, 67680, '\p{Is_Blk=imperialaramaic}', "");
Expect(1, 67680, '\p{^Is_Blk=imperialaramaic}', "");
Expect(1, 67680, '\P{Is_Blk=imperialaramaic}', "");
Expect(0, 67680, '\P{^Is_Blk=imperialaramaic}', "");
Expect(1, 67679, '\p{Is_Blk=--IMPERIAL_ARAMAIC}', "");
Expect(0, 67679, '\p{^Is_Blk=--IMPERIAL_ARAMAIC}', "");
Expect(0, 67679, '\P{Is_Blk=--IMPERIAL_ARAMAIC}', "");
Expect(1, 67679, '\P{^Is_Blk=--IMPERIAL_ARAMAIC}', "");
Expect(0, 67680, '\p{Is_Blk=--IMPERIAL_ARAMAIC}', "");
Expect(1, 67680, '\p{^Is_Blk=--IMPERIAL_ARAMAIC}', "");
Expect(1, 67680, '\P{Is_Blk=--IMPERIAL_ARAMAIC}', "");
Expect(0, 67680, '\P{^Is_Blk=--IMPERIAL_ARAMAIC}', "");
Error('\p{Block=	_Common_Indic_Number_Forms/a/}');
Error('\P{Block=	_Common_Indic_Number_Forms/a/}');
Expect(1, 43071, '\p{Block=commonindicnumberforms}', "");
Expect(0, 43071, '\p{^Block=commonindicnumberforms}', "");
Expect(0, 43071, '\P{Block=commonindicnumberforms}', "");
Expect(1, 43071, '\P{^Block=commonindicnumberforms}', "");
Expect(0, 43072, '\p{Block=commonindicnumberforms}', "");
Expect(1, 43072, '\p{^Block=commonindicnumberforms}', "");
Expect(1, 43072, '\P{Block=commonindicnumberforms}', "");
Expect(0, 43072, '\P{^Block=commonindicnumberforms}', "");
Expect(1, 43071, '\p{Block=		COMMON_INDIC_number_forms}', "");
Expect(0, 43071, '\p{^Block=		COMMON_INDIC_number_forms}', "");
Expect(0, 43071, '\P{Block=		COMMON_INDIC_number_forms}', "");
Expect(1, 43071, '\P{^Block=		COMMON_INDIC_number_forms}', "");
Expect(0, 43072, '\p{Block=		COMMON_INDIC_number_forms}', "");
Expect(1, 43072, '\p{^Block=		COMMON_INDIC_number_forms}', "");
Expect(1, 43072, '\P{Block=		COMMON_INDIC_number_forms}', "");
Expect(0, 43072, '\P{^Block=		COMMON_INDIC_number_forms}', "");
Error('\p{Blk=/a/indic_Number_Forms}');
Error('\P{Blk=/a/indic_Number_Forms}');
Expect(1, 43071, '\p{Blk=indicnumberforms}', "");
Expect(0, 43071, '\p{^Blk=indicnumberforms}', "");
Expect(0, 43071, '\P{Blk=indicnumberforms}', "");
Expect(1, 43071, '\P{^Blk=indicnumberforms}', "");
Expect(0, 43072, '\p{Blk=indicnumberforms}', "");
Expect(1, 43072, '\p{^Blk=indicnumberforms}', "");
Expect(1, 43072, '\P{Blk=indicnumberforms}', "");
Expect(0, 43072, '\P{^Blk=indicnumberforms}', "");
Expect(1, 43071, '\p{Blk=--Indic_NUMBER_Forms}', "");
Expect(0, 43071, '\p{^Blk=--Indic_NUMBER_Forms}', "");
Expect(0, 43071, '\P{Blk=--Indic_NUMBER_Forms}', "");
Expect(1, 43071, '\P{^Blk=--Indic_NUMBER_Forms}', "");
Expect(0, 43072, '\p{Blk=--Indic_NUMBER_Forms}', "");
Expect(1, 43072, '\p{^Blk=--Indic_NUMBER_Forms}', "");
Expect(1, 43072, '\P{Blk=--Indic_NUMBER_Forms}', "");
Expect(0, 43072, '\P{^Blk=--Indic_NUMBER_Forms}', "");
Error('\p{Is_Block=:=common_Indic_Number_forms}');
Error('\P{Is_Block=:=common_Indic_Number_forms}');
Expect(1, 43071, '\p{Is_Block=commonindicnumberforms}', "");
Expect(0, 43071, '\p{^Is_Block=commonindicnumberforms}', "");
Expect(0, 43071, '\P{Is_Block=commonindicnumberforms}', "");
Expect(1, 43071, '\P{^Is_Block=commonindicnumberforms}', "");
Expect(0, 43072, '\p{Is_Block=commonindicnumberforms}', "");
Expect(1, 43072, '\p{^Is_Block=commonindicnumberforms}', "");
Expect(1, 43072, '\P{Is_Block=commonindicnumberforms}', "");
Expect(0, 43072, '\P{^Is_Block=commonindicnumberforms}', "");
Expect(1, 43071, '\p{Is_Block=-_Common_Indic_NUMBER_FORMS}', "");
Expect(0, 43071, '\p{^Is_Block=-_Common_Indic_NUMBER_FORMS}', "");
Expect(0, 43071, '\P{Is_Block=-_Common_Indic_NUMBER_FORMS}', "");
Expect(1, 43071, '\P{^Is_Block=-_Common_Indic_NUMBER_FORMS}', "");
Expect(0, 43072, '\p{Is_Block=-_Common_Indic_NUMBER_FORMS}', "");
Expect(1, 43072, '\p{^Is_Block=-_Common_Indic_NUMBER_FORMS}', "");
Expect(1, 43072, '\P{Is_Block=-_Common_Indic_NUMBER_FORMS}', "");
Expect(0, 43072, '\P{^Is_Block=-_Common_Indic_NUMBER_FORMS}', "");
Error('\p{Is_Blk=:=--indic_number_FORMS}');
Error('\P{Is_Blk=:=--indic_number_FORMS}');
Expect(1, 43071, '\p{Is_Blk:	indicnumberforms}', "");
Expect(0, 43071, '\p{^Is_Blk:	indicnumberforms}', "");
Expect(0, 43071, '\P{Is_Blk:	indicnumberforms}', "");
Expect(1, 43071, '\P{^Is_Blk:	indicnumberforms}', "");
Expect(0, 43072, '\p{Is_Blk:	indicnumberforms}', "");
Expect(1, 43072, '\p{^Is_Blk:	indicnumberforms}', "");
Expect(1, 43072, '\P{Is_Blk:	indicnumberforms}', "");
Expect(0, 43072, '\P{^Is_Blk:	indicnumberforms}', "");
Expect(1, 43071, '\p{Is_Blk:   -	INDIC_NUMBER_Forms}', "");
Expect(0, 43071, '\p{^Is_Blk:   -	INDIC_NUMBER_Forms}', "");
Expect(0, 43071, '\P{Is_Blk:   -	INDIC_NUMBER_Forms}', "");
Expect(1, 43071, '\P{^Is_Blk:   -	INDIC_NUMBER_Forms}', "");
Expect(0, 43072, '\p{Is_Blk:   -	INDIC_NUMBER_Forms}', "");
Expect(1, 43072, '\p{^Is_Blk:   -	INDIC_NUMBER_Forms}', "");
Expect(1, 43072, '\P{Is_Blk:   -	INDIC_NUMBER_Forms}', "");
Expect(0, 43072, '\P{^Is_Blk:   -	INDIC_NUMBER_Forms}', "");
Error('\p{Block=-_Inscriptional_PAHLAVI/a/}');
Error('\P{Block=-_Inscriptional_PAHLAVI/a/}');
Expect(1, 68479, '\p{Block=inscriptionalpahlavi}', "");
Expect(0, 68479, '\p{^Block=inscriptionalpahlavi}', "");
Expect(0, 68479, '\P{Block=inscriptionalpahlavi}', "");
Expect(1, 68479, '\P{^Block=inscriptionalpahlavi}', "");
Expect(0, 68480, '\p{Block=inscriptionalpahlavi}', "");
Expect(1, 68480, '\p{^Block=inscriptionalpahlavi}', "");
Expect(1, 68480, '\P{Block=inscriptionalpahlavi}', "");
Expect(0, 68480, '\P{^Block=inscriptionalpahlavi}', "");
Expect(1, 68479, '\p{Block= Inscriptional_Pahlavi}', "");
Expect(0, 68479, '\p{^Block= Inscriptional_Pahlavi}', "");
Expect(0, 68479, '\P{Block= Inscriptional_Pahlavi}', "");
Expect(1, 68479, '\P{^Block= Inscriptional_Pahlavi}', "");
Expect(0, 68480, '\p{Block= Inscriptional_Pahlavi}', "");
Expect(1, 68480, '\p{^Block= Inscriptional_Pahlavi}', "");
Expect(1, 68480, '\P{Block= Inscriptional_Pahlavi}', "");
Expect(0, 68480, '\P{^Block= Inscriptional_Pahlavi}', "");
Error('\p{Blk=_INSCRIPTIONAL_Pahlavi/a/}');
Error('\P{Blk=_INSCRIPTIONAL_Pahlavi/a/}');
Expect(1, 68479, '\p{Blk=inscriptionalpahlavi}', "");
Expect(0, 68479, '\p{^Blk=inscriptionalpahlavi}', "");
Expect(0, 68479, '\P{Blk=inscriptionalpahlavi}', "");
Expect(1, 68479, '\P{^Blk=inscriptionalpahlavi}', "");
Expect(0, 68480, '\p{Blk=inscriptionalpahlavi}', "");
Expect(1, 68480, '\p{^Blk=inscriptionalpahlavi}', "");
Expect(1, 68480, '\P{Blk=inscriptionalpahlavi}', "");
Expect(0, 68480, '\P{^Blk=inscriptionalpahlavi}', "");
Expect(1, 68479, '\p{Blk=__Inscriptional_PAHLAVI}', "");
Expect(0, 68479, '\p{^Blk=__Inscriptional_PAHLAVI}', "");
Expect(0, 68479, '\P{Blk=__Inscriptional_PAHLAVI}', "");
Expect(1, 68479, '\P{^Blk=__Inscriptional_PAHLAVI}', "");
Expect(0, 68480, '\p{Blk=__Inscriptional_PAHLAVI}', "");
Expect(1, 68480, '\p{^Blk=__Inscriptional_PAHLAVI}', "");
Expect(1, 68480, '\P{Blk=__Inscriptional_PAHLAVI}', "");
Expect(0, 68480, '\P{^Blk=__Inscriptional_PAHLAVI}', "");
Error('\p{Is_Block=_/a/inscriptional_Pahlavi}');
Error('\P{Is_Block=_/a/inscriptional_Pahlavi}');
Expect(1, 68479, '\p{Is_Block=inscriptionalpahlavi}', "");
Expect(0, 68479, '\p{^Is_Block=inscriptionalpahlavi}', "");
Expect(0, 68479, '\P{Is_Block=inscriptionalpahlavi}', "");
Expect(1, 68479, '\P{^Is_Block=inscriptionalpahlavi}', "");
Expect(0, 68480, '\p{Is_Block=inscriptionalpahlavi}', "");
Expect(1, 68480, '\p{^Is_Block=inscriptionalpahlavi}', "");
Expect(1, 68480, '\P{Is_Block=inscriptionalpahlavi}', "");
Expect(0, 68480, '\P{^Is_Block=inscriptionalpahlavi}', "");
Expect(1, 68479, '\p{Is_Block:   	_Inscriptional_PAHLAVI}', "");
Expect(0, 68479, '\p{^Is_Block:   	_Inscriptional_PAHLAVI}', "");
Expect(0, 68479, '\P{Is_Block:   	_Inscriptional_PAHLAVI}', "");
Expect(1, 68479, '\P{^Is_Block:   	_Inscriptional_PAHLAVI}', "");
Expect(0, 68480, '\p{Is_Block:   	_Inscriptional_PAHLAVI}', "");
Expect(1, 68480, '\p{^Is_Block:   	_Inscriptional_PAHLAVI}', "");
Expect(1, 68480, '\P{Is_Block:   	_Inscriptional_PAHLAVI}', "");
Expect(0, 68480, '\P{^Is_Block:   	_Inscriptional_PAHLAVI}', "");
Error('\p{Is_Blk=_ inscriptional_Pahlavi/a/}');
Error('\P{Is_Blk=_ inscriptional_Pahlavi/a/}');
Expect(1, 68479, '\p{Is_Blk=inscriptionalpahlavi}', "");
Expect(0, 68479, '\p{^Is_Blk=inscriptionalpahlavi}', "");
Expect(0, 68479, '\P{Is_Blk=inscriptionalpahlavi}', "");
Expect(1, 68479, '\P{^Is_Blk=inscriptionalpahlavi}', "");
Expect(0, 68480, '\p{Is_Blk=inscriptionalpahlavi}', "");
Expect(1, 68480, '\p{^Is_Blk=inscriptionalpahlavi}', "");
Expect(1, 68480, '\P{Is_Blk=inscriptionalpahlavi}', "");
Expect(0, 68480, '\P{^Is_Blk=inscriptionalpahlavi}', "");
Expect(1, 68479, '\p{Is_Blk=-inscriptional_Pahlavi}', "");
Expect(0, 68479, '\p{^Is_Blk=-inscriptional_Pahlavi}', "");
Expect(0, 68479, '\P{Is_Blk=-inscriptional_Pahlavi}', "");
Expect(1, 68479, '\P{^Is_Blk=-inscriptional_Pahlavi}', "");
Expect(0, 68480, '\p{Is_Blk=-inscriptional_Pahlavi}', "");
Expect(1, 68480, '\p{^Is_Blk=-inscriptional_Pahlavi}', "");
Expect(1, 68480, '\P{Is_Blk=-inscriptional_Pahlavi}', "");
Expect(0, 68480, '\P{^Is_Blk=-inscriptional_Pahlavi}', "");
Error('\p{Block=-:=inscriptional_parthian}');
Error('\P{Block=-:=inscriptional_parthian}');
Expect(1, 68447, '\p{Block=inscriptionalparthian}', "");
Expect(0, 68447, '\p{^Block=inscriptionalparthian}', "");
Expect(0, 68447, '\P{Block=inscriptionalparthian}', "");
Ex