
use strict;
use warnings;

# If run outside the normal test suite on an ASCII platform, you can
# just create a latin1_to_native() function that just returns its
# inputs, because that's the only function used from test.pl
require "test.pl";

# Test qr/\X/ and the \p{} regular expression constructs.  This file is
# constructed by mktables from the tables it generates, so if mktables is
# buggy, this won't necessarily catch those bugs.  Tests are generated for all
# feasible properties; a few aren't currently feasible; see
# is_code_point_usable() in mktables for details.

# Standard test packages are not used because this manipulates SIG_WARN.  It
# exits 0 if every non-skipped test succeeded; -1 if any failed.

my $Tests = 0;
my $Fails = 0;

sub Expect($$$$) {
    my $expected = shift;
    my $ord = shift;
    my $regex  = shift;
    my $warning_type = shift;   # Type of warning message, like 'deprecated'
                                # or empty if none
    my $line   = (caller)[2];
    $ord = ord(latin1_to_native(chr($ord)));

    # Convert the code point to hex form
    my $string = sprintf "\"\\x{%04X}\"", $ord;

    my @tests = "";

    # The first time through, use all warnings.  If the input should generate
    # a warning, add another time through with them turned off
    push @tests, "no warnings '$warning_type';" if $warning_type;

    foreach my $no_warnings (@tests) {

        # Store any warning messages instead of outputting them
        local $SIG{__WARN__} = $SIG{__WARN__};
        my $warning_message;
        $SIG{__WARN__} = sub { $warning_message = $_[0] };

        $Tests++;

        # A string eval is needed because of the 'no warnings'.
        # Assumes no parens in the regular expression
        my $result = eval "$no_warnings
                            my \$RegObj = qr($regex);
                            $string =~ \$RegObj ? 1 : 0";
        if (not defined $result) {
            print "not ok $Tests - couldn't compile /$regex/; line $line: $@\n";
            $Fails++;
        }
        elsif ($result ^ $expected) {
            print "not ok $Tests - expected $expected but got $result for $string =~ qr/$regex/; line $line\n";
            $Fails++;
        }
        elsif ($warning_message) {
            if (! $warning_type || ($warning_type && $no_warnings)) {
                print "not ok $Tests - for qr/$regex/ did not expect warning message '$warning_message'; line $line\n";
                $Fails++;
            }
            else {
                print "ok $Tests - expected and got a warning message for qr/$regex/; line $line\n";
            }
        }
        elsif ($warning_type && ! $no_warnings) {
            print "not ok $Tests - for qr/$regex/ expected a $warning_type warning message, but got none; line $line\n";
            $Fails++;
        }
        else {
            print "ok $Tests - got $result for $string =~ qr/$regex/; line $line\n";
        }
    }
    return;
}

sub Error($) {
    my $regex  = shift;
    $Tests++;
    if (eval { 'x' =~ qr/$regex/; 1 }) {
        $Fails++;
        my $line = (caller)[2];
        print "not ok $Tests - re compiled ok, but expected error for qr/$regex/; line $line: $@\n";
    }
    else {
        my $line = (caller)[2];
        print "ok $Tests - got and expected error for qr/$regex/; line $line\n";
    }
    return;
}

# GCBTest.txt character that separates grapheme clusters
my $breakable_utf8 = my $breakable = chr(0xF7);
utf8::upgrade($breakable_utf8);

# GCBTest.txt character that indicates that the adjoining code points are part
# of the same grapheme cluster
my $nobreak_utf8 = my $nobreak = chr(0xD7);
utf8::upgrade($nobreak_utf8);

sub Test_X($) {
    # Test qr/\X/ matches.  The input is a line from auxiliary/GCBTest.txt
    # Each such line is a sequence of code points given by their hex numbers,
    # separated by the two characters defined just before this subroutine that
    # indicate that either there can or cannot be a break between the adjacent
    # code points.  If there isn't a break, that means the sequence forms an
    # extended grapheme cluster, which means that \X should match the whole
    # thing.  If there is a break, \X should stop there.  This is all
    # converted by this routine into a match:
    #   $string =~ /(\X)/,
    # Each \X should match the next cluster; and that is what is checked.

    my $template = shift;

    my $line   = (caller)[2];

    # The line contains characters above the ASCII range, but in Latin1.  It
    # may or may not be in utf8, and if it is, it may or may not know it.  So,
    # convert these characters to 8 bits.  If knows is in utf8, simply
    # downgrade.
    if (utf8::is_utf8($template)) {
        utf8::downgrade($template);
    } else {

        # Otherwise, if it is in utf8, but doesn't know it, the next lines
        # convert the two problematic characters to their 8-bit equivalents.
        # If it isn't in utf8, they don't harm anything.
        use bytes;
        $template =~ s/$nobreak_utf8/$nobreak/g;
        $template =~ s/$breakable_utf8/$breakable/g;
    }

    # Get rid of the leading and trailing breakables
    $template =~ s/^ \s* $breakable \s* //x;
    $template =~ s/ \s* $breakable \s* $ //x;

    # And no-breaks become just a space.
    $template =~ s/ \s* $nobreak \s* / /xg;

    # Split the input into segments that are breakable between them.
    my @segments = split /\s*$breakable\s*/, $template;

    my $string = "";
    my $display_string = "";
    my @should_match;
    my @should_display;

    # Convert the code point sequence in each segment into a Perl string of
    # characters
    foreach my $segment (@segments) {
        my @code_points = split /\s+/, $segment;
        my $this_string = "";
        my $this_display = "";
        foreach my $code_point (@code_points) {
            $this_string .= latin1_to_native(chr(hex $code_point));
            $this_display .= "\\x{$code_point}";
        }

        # The next cluster should match the string in this segment.
        push @should_match, $this_string;
        push @should_display, $this_display;
        $string .= $this_string;
        $display_string .= $this_display;
    }

    # If a string can be represented in both non-ut8 and utf8, test both cases
    UPGRADE:
    for my $to_upgrade (0 .. 1) {

        if ($to_upgrade) {

            # If already in utf8, would just be a repeat
            next UPGRADE if utf8::is_utf8($string);

            utf8::upgrade($string);
        }

        # Finally, do the \X match.
        my @matches = $string =~ /(\X)/g;

        # Look through each matched cluster to verify that it matches what we
        # expect.
        my $min = (@matches < @should_match) ? @matches : @should_match;
        for my $i (0 .. $min - 1) {
            $Tests++;
            if ($matches[$i] eq $should_match[$i]) {
                print "ok $Tests - ";
                if ($i == 0) {
                    print "In \"$display_string\" =~ /(\\X)/g, \\X #1";
                } else {
                    print "And \\X #", $i + 1,
                }
                print " correctly matched $should_display[$i]; line $line\n";
            } else {
                $matches[$i] = join("", map { sprintf "\\x{%04X}", $_ }
                                                    unpack("U*", $matches[$i]));
                print "not ok $Tests - In \"$display_string\" =~ /(\\X)/g, \\X #",
                    $i + 1,
                    " should have matched $should_display[$i]",
                    " but instead matched $matches[$i]",
                    ".  Abandoning rest of line $line\n";
                next UPGRADE;
            }
        }

        # And the number of matches should equal the number of expected matches.
        $Tests++;
        if (@matches == @should_match) {
            print "ok $Tests - Nothing was left over; line $line\n";
        } else {
            print "not ok $Tests - There were ", scalar @should_match, " \\X matches expected, but got ", scalar @matches, " instead; line $line\n";
        }
    }

    return;
}

sub Finished() {
    print "1..$Tests\n";
    exit($Fails ? -1 : 0);
}

Error('\p{Script=InGreek}');    # Bug #69018
Test_X("1100 $nobreak 1161");  # Bug #70940
Expect(0, 0x2028, '\p{Print}', ""); # Bug # 71722
Expect(0, 0x2029, '\p{Print}', ""); # Bug # 71722
Expect(1, 0xFF10, '\p{XDigit}', ""); # Bug # 71726
Error('\p{Line_Break=:=	Space}');
Error('\P{Line_Break=:=	Space}');
Expect(1, 32, '\p{Line_Break=space}', "");
Expect(0, 32, '\p{^Line_Break=space}', "");
Expect(0, 32, '\P{Line_Break=space}', "");
Expect(1, 32, '\P{^Line_Break=space}', "");
Expect(0, 33, '\p{Line_Break=space}', "");
Expect(1, 33, '\p{^Line_Break=space}', "");
Expect(1, 33, '\P{Line_Break=space}', "");
Expect(0, 33, '\P{^Line_Break=space}', "");
Expect(1, 32, '\p{Line_Break=--Space}', "");
Expect(0, 32, '\p{^Line_Break=--Space}', "");
Expect(0, 32, '\P{Line_Break=--Space}', "");
Expect(1, 32, '\P{^Line_Break=--Space}', "");
Expect(0, 33, '\p{Line_Break=--Space}', "");
Expect(1, 33, '\p{^Line_Break=--Space}', "");
Expect(1, 33, '\P{Line_Break=--Space}', "");
Expect(0, 33, '\P{^Line_Break=--Space}', "");
Error('\p{Lb=		SP:=}');
Error('\P{Lb=		SP:=}');
Expect(1, 32, '\p{Lb=sp}', "");
Expect(0, 32, '\p{^Lb=sp}', "");
Expect(0, 32, '\P{Lb=sp}', "");
Expect(1, 32, '\P{^Lb=sp}', "");
Expect(0, 33, '\p{Lb=sp}', "");
Expect(1, 33, '\p{^Lb=sp}', "");
Expect(1, 33, '\P{Lb=sp}', "");
Expect(0, 33, '\P{^Lb=sp}', "");
Expect(1, 32, '\p{Lb:	SP}', "");
Expect(0, 32, '\p{^Lb:	SP}', "");
Expect(0, 32, '\P{Lb:	SP}', "");
Expect(1, 32, '\P{^Lb:	SP}', "");
Expect(0, 33, '\p{Lb:	SP}', "");
Expect(1, 33, '\p{^Lb:	SP}', "");
Expect(1, 33, '\P{Lb:	SP}', "");
Expect(0, 33, '\P{^Lb:	SP}', "");
Error('\p{Is_Line_Break=/a/		SPACE}');
Error('\P{Is_Line_Break=/a/		SPACE}');
Expect(1, 32, '\p{Is_Line_Break=space}', "");
Expect(0, 32, '\p{^Is_Line_Break=space}', "");
Expect(0, 32, '\P{Is_Line_Break=space}', "");
Expect(1, 32, '\P{^Is_Line_Break=space}', "");
Expect(0, 33, '\p{Is_Line_Break=space}', "");
Expect(1, 33, '\p{^Is_Line_Break=space}', "");
Expect(1, 33, '\P{Is_Line_Break=space}', "");
Expect(0, 33, '\P{^Is_Line_Break=space}', "");
Expect(1, 32, '\p{Is_Line_Break=-Space}', "");
Expect(0, 32, '\p{^Is_Line_Break=-Space}', "");
Expect(0, 32, '\P{Is_Line_Break=-Space}', "");
Expect(1, 32, '\P{^Is_Line_Break=-Space}', "");
Expect(0, 33, '\p{Is_Line_Break=-Space}', "");
Expect(1, 33, '\p{^Is_Line_Break=-Space}', "");
Expect(1, 33, '\P{Is_Line_Break=-Space}', "");
Expect(0, 33, '\P{^Is_Line_Break=-Space}', "");
Error('\p{Is_Lb=		SP/a/}');
Error('\P{Is_Lb=		SP/a/}');
Expect(1, 32, '\p{Is_Lb=sp}', "");
Expect(0, 32, '\p{^Is_Lb=sp}', "");
Expect(0, 32, '\P{Is_Lb=sp}', "");
Expect(1, 32, '\P{^Is_Lb=sp}', "");
Expect(0, 33, '\p{Is_Lb=sp}', "");
Expect(1, 33, '\p{^Is_Lb=sp}', "");
Expect(1, 33, '\P{Is_Lb=sp}', "");
Expect(0, 33, '\P{^Is_Lb=sp}', "");
Expect(1, 32, '\p{Is_Lb=_-sp}', "");
Expect(0, 32, '\p{^Is_Lb=_-sp}', "");
Expect(0, 32, '\P{Is_Lb=_-sp}', "");
Expect(1, 32, '\P{^Is_Lb=_-sp}', "");
Expect(0, 33, '\p{Is_Lb=_-sp}', "");
Expect(1, 33, '\p{^Is_Lb=_-sp}', "");
Expect(1, 33, '\P{Is_Lb=_-sp}', "");
Expect(0, 33, '\P{^Is_Lb=_-sp}', "");
Error('\p{Line_Break=_ GLUE:=}');
Error('\P{Line_Break=_ GLUE:=}');
Expect(1, 8239, '\p{Line_Break=glue}', "");
Expect(0, 8239, '\p{^Line_Break=glue}', "");
Expect(0, 8239, '\P{Line_Break=glue}', "");
Expect(1, 8239, '\P{^Line_Break=glue}', "");
Expect(0, 8240, '\p{Line_Break=glue}', "");
Expect(1, 8240, '\p{^Line_Break=glue}', "");
Expect(1, 8240, '\P{Line_Break=glue}', "");
Expect(0, 8240, '\P{^Line_Break=glue}', "");
Expect(1, 8239, '\p{Line_Break=_	glue}', "");
Expect(0, 8239, '\p{^Line_Break=_	glue}', "");
Expect(0, 8239, '\P{Line_Break=_	glue}', "");
Expect(1, 8239, '\P{^Line_Break=_	glue}', "");
Expect(0, 8240, '\p{Line_Break=_	glue}', "");
Expect(1, 8240, '\p{^Line_Break=_	glue}', "");
Expect(1, 8240, '\P{Line_Break=_	glue}', "");
Expect(0, 8240, '\P{^Line_Break=_	glue}', "");
Error('\p{Lb:   /a/	GL}');
Error('\P{Lb:   /a/	GL}');
Expect(1, 8239, '\p{Lb=gl}', "");
Expect(0, 8239, '\p{^Lb=gl}', "");
Expect(0, 8239, '\P{Lb=gl}', "");
Expect(1, 8239, '\P{^Lb=gl}', "");
Expect(0, 8240, '\p{Lb=gl}', "");
Expect(1, 8240, '\p{^Lb=gl}', "");
Expect(1, 8240, '\P{Lb=gl}', "");
Expect(0, 8240, '\P{^Lb=gl}', "");
Expect(1, 8239, '\p{Lb= gl}', "");
Expect(0, 8239, '\p{^Lb= gl}', "");
Expect(0, 8239, '\P{Lb= gl}', "");
Expect(1, 8239, '\P{^Lb= gl}', "");
Expect(0, 8240, '\p{Lb= gl}', "");
Expect(1, 8240, '\p{^Lb= gl}', "");
Expect(1, 8240, '\P{Lb= gl}', "");
Expect(0, 8240, '\P{^Lb= gl}', "");
Error('\p{Is_Line_Break::=-	Glue}');
Error('\P{Is_Line_Break::=-	Glue}');
Expect(1, 8239, '\p{Is_Line_Break=glue}', "");
Expect(0, 8239, '\p{^Is_Line_Break=glue}', "");
Expect(0, 8239, '\P{Is_Line_Break=glue}', "");
Expect(1, 8239, '\P{^Is_Line_Break=glue}', "");
Expect(0, 8240, '\p{Is_Line_Break=glue}', "");
Expect(1, 8240, '\p{^Is_Line_Break=glue}', "");
Expect(1, 8240, '\P{Is_Line_Break=glue}', "");
Expect(0, 8240, '\P{^Is_Line_Break=glue}', "");
Expect(1, 8239, '\p{Is_Line_Break:  Glue}', "");
Expect(0, 8239, '\p{^Is_Line_Break:  Glue}', "");
Expect(0, 8239, '\P{Is_Line_Break:  Glue}', "");
Expect(1, 8239, '\P{^Is_Line_Break:  Glue}', "");
Expect(0, 8240, '\p{Is_Line_Break:  Glue}', "");
Expect(1, 8240, '\p{^Is_Line_Break:  Glue}', "");
Expect(1, 8240, '\P{Is_Line_Break:  Glue}', "");
Expect(0, 8240, '\P{^Is_Line_Break:  Glue}', "");
Error('\p{Is_Lb=	:=GL}');
Error('\P{Is_Lb=	:=GL}');
Expect(1, 8239, '\p{Is_Lb=gl}', "");
Expect(0, 8239, '\p{^Is_Lb=gl}', "");
Expect(0, 8239, '\P{Is_Lb=gl}', "");
Expect(1, 8239, '\P{^Is_Lb=gl}', "");
Expect(0, 8240, '\p{Is_Lb=gl}', "");
Expect(1, 8240, '\p{^Is_Lb=gl}', "");
Expect(1, 8240, '\P{Is_Lb=gl}', "");
Expect(0, 8240, '\P{^Is_Lb=gl}', "");
Expect(1, 8239, '\p{Is_Lb=_	GL}', "");
Expect(0, 8239, '\p{^Is_Lb=_	GL}', "");
Expect(0, 8239, '\P{Is_Lb=_	GL}', "");
Expect(1, 8239, '\P{^Is_Lb=_	GL}', "");
Expect(0, 8240, '\p{Is_Lb=_	GL}', "");
Expect(1, 8240, '\p{^Is_Lb=_	GL}', "");
Expect(1, 8240, '\P{Is_Lb=_	GL}', "");
Expect(0, 8240, '\P{^Is_Lb=_	GL}', "");
Error('\p{Line_Break= Surrogate:=}');
Error('\P{Line_Break= Surrogate:=}');
Expect(1, 57343, '\p{Line_Break=surrogate}', 'deprecated');
Expect(0, 57343, '\p{^Line_Break=surrogate}', 'deprecated');
Expect(0, 57343, '\P{Line_Break=surrogate}', 'deprecated');
Expect(1, 57343, '\P{^Line_Break=surrogate}', 'deprecated');
Expect(0, 57344, '\p{Line_Break=surrogate}', 'deprecated');
Expect(1, 57344, '\p{^Line_Break=surrogate}', 'deprecated');
Expect(1, 57344, '\P{Line_Break=surrogate}', 'deprecated');
Expect(0, 57344, '\P{^Line_Break=surrogate}', 'deprecated');
Expect(1, 57343, '\p{Line_Break:   -_Surrogate}', 'deprecated');
Expect(0, 57343, '\p{^Line_Break:   -_Surrogate}', 'deprecated');
Expect(0, 57343, '\P{Line_Break:   -_Surrogate}', 'deprecated');
Expect(1, 57343, '\P{^Line_Break:   -_Surrogate}', 'deprecated');
Expect(0, 57344, '\p{Line_Break:   -_Surrogate}', 'deprecated');
Expect(1, 57344, '\p{^Line_Break:   -_Surrogate}', 'deprecated');
Expect(1, 57344, '\P{Line_Break:   -_Surrogate}', 'deprecated');
Expect(0, 57344, '\P{^Line_Break:   -_Surrogate}', 'deprecated');
Error('\p{Lb=-_sg/a/}');
Error('\P{Lb=-_sg/a/}');
Expect(1, 57343, '\p{Lb=sg}', 'deprecated');
Expect(0, 57343, '\p{^Lb=sg}', 'deprecated');
Expect(0, 57343, '\P{Lb=sg}', 'deprecated');
Expect(1, 57343, '\P{^Lb=sg}', 'deprecated');
Expect(0, 57344, '\p{Lb=sg}', 'deprecated');
Expect(1, 57344, '\p{^Lb=sg}', 'deprecated');
Expect(1, 57344, '\P{Lb=sg}', 'deprecated');
Expect(0, 57344, '\P{^Lb=sg}', 'deprecated');
Expect(1, 57343, '\p{Lb=	 SG}', 'deprecated');
Expect(0, 57343, '\p{^Lb=	 SG}', 'deprecated');
Expect(0, 57343, '\P{Lb=	 SG}', 'deprecated');
Expect(1, 57343, '\P{^Lb=	 SG}', 'deprecated');
Expect(0, 57344, '\p{Lb=	 SG}', 'deprecated');
Expect(1, 57344, '\p{^Lb=	 SG}', 'deprecated');
Expect(1, 57344, '\P{Lb=	 SG}', 'deprecated');
Expect(0, 57344, '\P{^Lb=	 SG}', 'deprecated');
Error('\p{Is_Line_Break:	:=_SURROGATE}');
Error('\P{Is_Line_Break:	:=_SURROGATE}');
Expect(1, 57343, '\p{Is_Line_Break=surrogate}', 'deprecated');
Expect(0, 57343, '\p{^Is_Line_Break=surrogate}', 'deprecated');
Expect(0, 57343, '\P{Is_Line_Break=surrogate}', 'deprecated');
Expect(1, 57343, '\P{^Is_Line_Break=surrogate}', 'deprecated');
Expect(0, 57344, '\p{Is_Line_Break=surrogate}', 'deprecated');
Expect(1, 57344, '\p{^Is_Line_Break=surrogate}', 'deprecated');
Expect(1, 57344, '\P{Is_Line_Break=surrogate}', 'deprecated');
Expect(0, 57344, '\P{^Is_Line_Break=surrogate}', 'deprecated');
Expect(1, 57343, '\p{Is_Line_Break= SURROGATE}', 'deprecated');
Expect(0, 57343, '\p{^Is_Line_Break= SURROGATE}', 'deprecated');
Expect(0, 57343, '\P{Is_Line_Break= SURROGATE}', 'deprecated');
Expect(1, 57343, '\P{^Is_Line_Break= SURROGATE}', 'deprecated');
Expect(0, 57344, '\p{Is_Line_Break= SURROGATE}', 'deprecated');
Expect(1, 57344, '\p{^Is_Line_Break= SURROGATE}', 'deprecated');
Expect(1, 57344, '\P{Is_Line_Break= SURROGATE}', 'deprecated');
Expect(0, 57344, '\P{^Is_Line_Break= SURROGATE}', 'deprecated');
Error('\p{Is_Lb=/a/--sg}');
Error('\P{Is_Lb=/a/--sg}');
Expect(1, 57343, '\p{Is_Lb=sg}', 'deprecated');
Expect(0, 57343, '\p{^Is_Lb=sg}', 'deprecated');
Expect(0, 57343, '\P{Is_Lb=sg}', 'deprecated');
Expect(1, 57343, '\P{^Is_Lb=sg}', 'deprecated');
Expect(0, 57344, '\p{Is_Lb=sg}', 'deprecated');
Expect(1, 57344, '\p{^Is_Lb=sg}', 'deprecated');
Expect(1, 57344, '\P{Is_Lb=sg}', 'deprecated');
Expect(0, 57344, '\P{^Is_Lb=sg}', 'deprecated');
Expect(1, 57343, '\p{Is_Lb: sg}', 'deprecated');
Expect(0, 57343, '\p{^Is_Lb: sg}', 'deprecated');
Expect(0, 57343, '\P{Is_Lb: sg}', 'deprecated');
Expect(1, 57343, '\P{^Is_Lb: sg}', 'deprecated');
Expect(0, 57344, '\p{Is_Lb: sg}', 'deprecated');
Expect(1, 57344, '\p{^Is_Lb: sg}', 'deprecated');
Expect(1, 57344, '\P{Is_Lb: sg}', 'deprecated');
Expect(0, 57344, '\P{^Is_Lb: sg}', 'deprecated');
Error('\p{Line_Break= Unknown/a/}');
Error('\P{Line_Break= Unknown/a/}');
Expect(1, 918000, '\p{Line_Break=unknown}', "");
Expect(0, 918000, '\p{^Line_Break=unknown}', "");
Expect(0, 918000, '\P{Line_Break=unknown}', "");
Expect(1, 918000, '\P{^Line_Break=unknown}', "");
Expect(0, 917999, '\p{Line_Break=unknown}', "");
Expect(1, 917999, '\p{^Line_Break=unknown}', "");
Expect(1, 917999, '\P{Line_Break=unknown}', "");
Expect(0, 917999, '\P{^Line_Break=unknown}', "");
Expect(1, 918000, '\p{Line_Break=-	unknown}', "");
Expect(0, 918000, '\p{^Line_Break=-	unknown}', "");
Expect(0, 918000, '\P{Line_Break=-	unknown}', "");
Expect(1, 918000, '\P{^Line_Break=-	unknown}', "");
Expect(0, 917999, '\p{Line_Break=-	unknown}', "");
Expect(1, 917999, '\p{^Line_Break=-	unknown}', "");
Expect(1, 917999, '\P{Line_Break=-	unknown}', "");
Expect(0, 917999, '\P{^Line_Break=-	unknown}', "");
Error('\p{Lb=:=XX}');
Error('\P{Lb=:=XX}');
Expect(1, 918000, '\p{Lb=xx}', "");
Expect(0, 918000, '\p{^Lb=xx}', "");
Expect(0, 918000, '\P{Lb=xx}', "");
Expect(1, 918000, '\P{^Lb=xx}', "");
Expect(0, 917999, '\p{Lb=xx}', "");
Expect(1, 917999, '\p{^Lb=xx}', "");
Expect(1, 917999, '\P{Lb=xx}', "");
Expect(0, 917999, '\P{^Lb=xx}', "");
Expect(1, 918000, '\p{Lb=__XX}', "");
Expect(0, 918000, '\p{^Lb=__XX}', "");
Expect(0, 918000, '\P{Lb=__XX}', "");
Expect(1, 918000, '\P{^Lb=__XX}', "");
Expect(0, 917999, '\p{Lb=__XX}', "");
Expect(1, 917999, '\p{^Lb=__XX}', "");
Expect(1, 917999, '\P{Lb=__XX}', "");
Expect(0, 917999, '\P{^Lb=__XX}', "");
Error('\p{Is_Line_Break=	UNKNOWN/a/}');
Error('\P{Is_Line_Break=	UNKNOWN/a/}');
Expect(1, 918000, '\p{Is_Line_Break:unknown}', "");
Expect(0, 918000, '\p{^Is_Line_Break:unknown}', "");
Expect(0, 918000, '\P{Is_Line_Break:unknown}', "");
Expect(1, 918000, '\P{^Is_Line_Break:unknown}', "");
Expect(0, 917999, '\p{Is_Line_Break:unknown}', "");
Expect(1, 917999, '\p{^Is_Line_Break:unknown}', "");
Expect(1, 917999, '\P{Is_Line_Break:unknown}', "");
Expect(0, 917999, '\P{^Is_Line_Break:unknown}', "");
Expect(1, 918000, '\p{Is_Line_Break=_-Unknown}', "");
Expect(0, 918000, '\p{^Is_Line_Break=_-Unknown}', "");
Expect(0, 918000, '\P{Is_Line_Break=_-Unknown}', "");
Expect(1, 918000, '\P{^Is_Line_Break=_-Unknown}', "");
Expect(0, 917999, '\p{Is_Line_Break=_-Unknown}', "");
Expect(1, 917999, '\p{^Is_Line_Break=_-Unknown}', "");
Expect(1, 917999, '\P{Is_Line_Break=_-Unknown}', "");
Expect(0, 917999, '\P{^Is_Line_Break=_-Unknown}', "");
Error('\p{Is_Lb=/a/		XX}');
Error('\P{Is_Lb=/a/		XX}');
Expect(1, 918000, '\p{Is_Lb=xx}', "");
Expect(0, 918000, '\p{^Is_Lb=xx}', "");
Expect(0, 918000, '\P{Is_Lb=xx}', "");
Expect(1, 918000, '\P{^Is_Lb=xx}', "");
Expect(0, 917999, '\p{Is_Lb=xx}', "");
Expect(1, 917999, '\p{^Is_Lb=xx}', "");
Expect(1, 917999, '\P{Is_Lb=xx}', "");
Expect(0, 917999, '\P{^Is_Lb=xx}', "");
Expect(1, 918000, '\p{Is_Lb=	-XX}', "");
Expect(0, 918000, '\p{^Is_Lb=	-XX}', "");
Expect(0, 918000, '\P{Is_Lb=	-XX}', "");
Expect(1, 918000, '\P{^Is_Lb=	-XX}', "");
Expect(0, 917999, '\p{Is_Lb=	-XX}', "");
Expect(1, 917999, '\p{^Is_Lb=	-XX}', "");
Expect(1, 917999, '\P{Is_Lb=	-XX}', "");
Expect(0, 917999, '\P{^Is_Lb=	-XX}', "");
Error('\p{Line_Break=-numeric:=}');
Error('\P{Line_Break=-numeric:=}');
Expect(1, 120831, '\p{Line_Break=numeric}', "");
Expect(0, 120831, '\p{^Line_Break=numeric}', "");
Expect(0, 120831, '\P{Line_Break=numeric}', "");
Expect(1, 120831, '\P{^Line_Break=numeric}', "");
Expect(0, 120832, '\p{Line_Break=numeric}', "");
Expect(1, 120832, '\p{^Line_Break=numeric}', "");
Expect(1, 120832, '\P{Line_Break=numeric}', "");
Expect(0, 120832, '\P{^Line_Break=numeric}', "");
Expect(1, 120831, '\p{Line_Break= NUMERIC}', "");
Expect(0, 120831, '\p{^Line_Break= NUMERIC}', "");
Expect(0, 120831, '\P{Line_Break= NUMERIC}', "");
Expect(1, 120831, '\P{^Line_Break= NUMERIC}', "");
Expect(0, 120832, '\p{Line_Break= NUMERIC}', "");
Expect(1, 120832, '\p{^Line_Break= NUMERIC}', "");
Expect(1, 120832, '\P{Line_Break= NUMERIC}', "");
Expect(0, 120832, '\P{^Line_Break= NUMERIC}', "");
Error('\p{Lb=/a/	-NU}');
Error('\P{Lb=/a/	-NU}');
Expect(1, 120831, '\p{Lb=nu}', "");
Expect(0, 120831, '\p{^Lb=nu}', "");
Expect(0, 120831, '\P{Lb=nu}', "");
Expect(1, 120831, '\P{^Lb=nu}', "");
Expect(0, 120832, '\p{Lb=nu}', "");
Expect(1, 120832, '\p{^Lb=nu}', "");
Expect(1, 120832, '\P{Lb=nu}', "");
Expect(0, 120832, '\P{^Lb=nu}', "");
Expect(1, 120831, '\p{Lb=-nu}', "");
Expect(0, 120831, '\p{^Lb=-nu}', "");
Expect(0, 120831, '\P{Lb=-nu}', "");
Expect(1, 120831, '\P{^Lb=-nu}', "");
Expect(0, 120832, '\p{Lb=-nu}', "");
Expect(1, 120832, '\p{^Lb=-nu}', "");
Expect(1, 120832, '\P{Lb=-nu}', "");
Expect(0, 120832, '\P{^Lb=-nu}', "");
Error('\p{Is_Line_Break=	 NUMERIC:=}');
Error('\P{Is_Line_Break=	 NUMERIC:=}');
Expect(1, 120831, '\p{Is_Line_Break=numeric}', "");
Expect(0, 120831, '\p{^Is_Line_Break=numeric}', "");
Expect(0, 120831, '\P{Is_Line_Break=numeric}', "");
Expect(1, 120831, '\P{^Is_Line_Break=numeric}', "");
Expect(0, 120832, '\p{Is_Line_Break=numeric}', "");
Expect(1, 120832, '\p{^Is_Line_Break=numeric}', "");
Expect(1, 120832, '\P{Is_Line_Break=numeric}', "");
Expect(0, 120832, '\P{^Is_Line_Break=numeric}', "");
Expect(1, 120831, '\p{Is_Line_Break=  Numeric}', "");
Expect(0, 120831, '\p{^Is_Line_Break=  Numeric}', "");
Expect(0, 120831, '\P{Is_Line_Break=  Numeric}', "");
Expect(1, 120831, '\P{^Is_Line_Break=  Numeric}', "");
Expect(0, 120832, '\p{Is_Line_Break=  Numeric}', "");
Expect(1, 120832, '\p{^Is_Line_Break=  Numeric}', "");
Expect(1, 120832, '\P{Is_Line_Break=  Numeric}', "");
Expect(0, 120832, '\P{^Is_Line_Break=  Numeric}', "");
Error('\p{Is_Lb=-NU/a/}');
Error('\P{Is_Lb=-NU/a/}');
Expect(1, 120831, '\p{Is_Lb=nu}', "");
Expect(0, 120831, '\p{^Is_Lb=nu}', "");
Expect(0, 120831, '\P{Is_Lb=nu}', "");
Expect(1, 120831, '\P{^Is_Lb=nu}', "");
Expect(0, 120832, '\p{Is_Lb=nu}', "");
Expect(1, 120832, '\p{^Is_Lb=nu}', "");
Expect(1, 120832, '\P{Is_Lb=nu}', "");
Expect(0, 120832, '\P{^Is_Lb=nu}', "");
Expect(1, 120831, '\p{Is_Lb=NU}', "");
Expect(0, 120831, '\p{^Is_Lb=NU}', "");
Expect(0, 120831, '\P{Is_Lb=NU}', "");
Expect(1, 120831, '\P{^Is_Lb=NU}', "");
Expect(0, 120832, '\p{Is_Lb=NU}', "");
Expect(1, 120832, '\p{^Is_Lb=NU}', "");
Expect(1, 120832, '\P{Is_Lb=NU}', "");
Expect(0, 120832, '\P{^Is_Lb=NU}', "");
Error('\p{Line_Break=:=- Break_before}');
Error('\P{Line_Break=:=- Break_before}');
Expect(1, 43125, '\p{Line_Break=breakbefore}', "");
Expect(0, 43125, '\p{^Line_Break=breakbefore}', "");
Expect(0, 43125, '\P{Line_Break=breakbefore}', "");
Expect(1, 43125, '\P{^Line_Break=breakbefore}', "");
Expect(0, 43126, '\p{Line_Break=breakbefore}', "");
Expect(1, 43126, '\p{^Line_Break=breakbefore}', "");
Expect(1, 43126, '\P{Line_Break=breakbefore}', "");
Expect(0, 43126, '\P{^Line_Break=breakbefore}', "");
Expect(1, 43125, '\p{Line_Break=-_Break_BEFORE}', "");
Expect(0, 43125, '\p{^Line_Break=-_Break_BEFORE}', "");
Expect(0, 43125, '\P{Line_Break=-_Break_BEFORE}', "");
Expect(1, 43125, '\P{^Line_Break=-_Break_BEFORE}', "");
Expect(0, 43126, '\p{Line_Break=-_Break_BEFORE}', "");
Expect(1, 43126, '\p{^Line_Break=-_Break_BEFORE}', "");
Expect(1, 43126, '\P{Line_Break=-_Break_BEFORE}', "");
Expect(0, 43126, '\P{^Line_Break=-_Break_BEFORE}', "");
Error('\p{Lb: /a/ bb}');
Error('\P{Lb: /a/ bb}');
Expect(1, 43125, '\p{Lb=bb}', "");
Expect(0, 43125, '\p{^Lb=bb}', "");
Expect(0, 43125, '\P{Lb=bb}', "");
Expect(1, 43125, '\P{^Lb=bb}', "");
Expect(0, 43126, '\p{Lb=bb}', "");
Expect(1, 43126, '\p{^Lb=bb}', "");
Expect(1, 43126, '\P{Lb=bb}', "");
Expect(0, 43126, '\P{^Lb=bb}', "");
Expect(1, 43125, '\p{Lb=		BB}', "");
Expect(0, 43125, '\p{^Lb=		BB}', "");
Expect(0, 43125, '\P{Lb=		BB}', "");
Expect(1, 43125, '\P{^Lb=		BB}', "");
Expect(0, 43126, '\p{Lb=		BB}', "");
Expect(1, 43126, '\p{^Lb=		BB}', "");
Expect(1, 43126, '\P{Lb=		BB}', "");
Expect(0, 43126, '\P{^Lb=		BB}', "");
Error('\p{Is_Line_Break=	:=Break_BEFORE}');
Error('\P{Is_Line_Break=	:=Break_BEFORE}');
Expect(1, 43125, '\p{Is_Line_Break:breakbefore}', "");
Expect(0, 43125, '\p{^Is_Line_Break:breakbefore}', "");
Expect(0, 43125, '\P{Is_Line_Break:breakbefore}', "");
Expect(1, 43125, '\P{^Is_Line_Break:breakbefore}', "");
Expect(0, 43126, '\p{Is_Line_Break:breakbefore}', "");
Expect(1, 43126, '\p{^Is_Line_Break:breakbefore}', "");
Expect(1, 43126, '\P{Is_Line_Break:breakbefore}', "");
Expect(0, 43126, '\P{^Is_Line_Break:breakbefore}', "");
Expect(1, 43125, '\p{Is_Line_Break:   BREAK_BEFORE}', "");
Expect(0, 43125, '\p{^Is_Line_Break:   BREAK_BEFORE}', "");
Expect(0, 43125, '\P{Is_Line_Break:   BREAK_BEFORE}', "");
Expect(1, 43125, '\P{^Is_Line_Break:   BREAK_BEFORE}', "");
Expect(0, 43126, '\p{Is_Line_Break:   BREAK_BEFORE}', "");
Expect(1, 43126, '\p{^Is_Line_Break:   BREAK_BEFORE}', "");
Expect(1, 43126, '\P{Is_Line_Break:   BREAK_BEFORE}', "");
Expect(0, 43126, '\P{^Is_Line_Break:   BREAK_BEFORE}', "");
Error('\p{Is_Lb=-/a/BB}');
Error('\P{Is_Lb=-/a/BB}');
Expect(1, 43125, '\p{Is_Lb=bb}', "");
Expect(0, 43125, '\p{^Is_Lb=bb}', "");
Expect(0, 43125, '\P{Is_Lb=bb}', "");
Expect(1, 43125, '\P{^Is_Lb=bb}', "");
Expect(0, 43126, '\p{Is_Lb=bb}', "");
Expect(1, 43126, '\p{^Is_Lb=bb}', "");
Expect(1, 43126, '\P{Is_Lb=bb}', "");
Expect(0, 43126, '\P{^Is_Lb=bb}', "");
Expect(1, 43125, '\p{Is_Lb= _BB}', "");
Expect(0, 43125, '\p{^Is_Lb= _BB}', "");
Expect(0, 43125, '\P{Is_Lb= _BB}', "");
Expect(1, 43125, '\P{^Is_Lb= _BB}', "");
Expect(0, 43126, '\p{Is_Lb= _BB}', "");
Expect(1, 43126, '\p{^Is_Lb= _BB}', "");
Expect(1, 43126, '\P{Is_Lb= _BB}', "");
Expect(0, 43126, '\P{^Is_Lb= _BB}', "");
Error('\p{Line_Break=_ZWSpace:=}');
Error('\P{Line_Break=_ZWSpace:=}');
Expect(1, 8203, '\p{Line_Break=zwspace}', "");
Expect(0, 8203, '\p{^Line_Break=zwspace}', "");
Expect(0, 8203, '\P{Line_Break=zwspace}', "");
Expect(1, 8203, '\P{^Line_Break=zwspace}', "");
Expect(0, 8204, '\p{Line_Break=zwspace}', "");
Expect(1, 8204, '\p{^Line_Break=zwspace}', "");
Expect(1, 8204, '\P{Line_Break=zwspace}', "");
Expect(0, 8204, '\P{^Line_Break=zwspace}', "");
Expect(1, 8203, '\p{Line_Break=-	ZWSpace}', "");
Expect(0, 8203, '\p{^Line_Break=-	ZWSpace}', "");
Expect(0, 8203, '\P{Line_Break=-	ZWSpace}', "");
Expect(1, 8203, '\P{^Line_Break=-	ZWSpace}', "");
Expect(0, 8204, '\p{Line_Break=-	ZWSpace}', "");
Expect(1, 8204, '\p{^Line_Break=-	ZWSpace}', "");
Expect(1, 8204, '\P{Line_Break=-	ZWSpace}', "");
Expect(0, 8204, '\P{^Line_Break=-	ZWSpace}', "");
Error('\p{Lb:/a/ _ZW}');
Error('\P{Lb:/a/ _ZW}');
Expect(1, 8203, '\p{Lb=zw}', "");
Expect(0, 8203, '\p{^Lb=zw}', "");
Expect(0, 8203, '\P{Lb=zw}', "");
Expect(1, 8203, '\P{^Lb=zw}', "");
Expect(0, 8204, '\p{Lb=zw}', "");
Expect(1, 8204, '\p{^Lb=zw}', "");
Expect(1, 8204, '\P{Lb=zw}', "");
Expect(0, 8204, '\P{^Lb=zw}', "");
Expect(1, 8203, '\p{Lb=_ZW}', "");
Expect(0, 8203, '\p{^Lb=_ZW}', "");
Expect(0, 8203, '\P{Lb=_ZW}', "");
Expect(1, 8203, '\P{^Lb=_ZW}', "");
Expect(0, 8204, '\p{Lb=_ZW}', "");
Expect(1, 8204, '\p{^Lb=_ZW}', "");
Expect(1, 8204, '\P{Lb=_ZW}', "");
Expect(0, 8204, '\P{^Lb=_ZW}', "");
Error('\p{Is_Line_Break=/a/  ZWSpace}');
Error('\P{Is_Line_Break=/a/  ZWSpace}');
Expect(1, 8203, '\p{Is_Line_Break=zwspace}', "");
Expect(0, 8203, '\p{^Is_Line_Break=zwspace}', "");
Expect(0, 8203, '\P{Is_Line_Break=zwspace}', "");
Expect(1, 8203, '\P{^Is_Line_Break=zwspace}', "");
Expect(0, 8204, '\p{Is_Line_Break=zwspace}', "");
Expect(1, 8204, '\p{^Is_Line_Break=zwspace}', "");
Expect(1, 8204, '\P{Is_Line_Break=zwspace}', "");
Expect(0, 8204, '\P{^Is_Line_Break=zwspace}', "");
Expect(1, 8203, '\p{Is_Line_Break=		ZWSPACE}', "");
Expect(0, 8203, '\p{^Is_Line_Break=		ZWSPACE}', "");
Expect(0, 8203, '\P{Is_Line_Break=		ZWSPACE}', "");
Expect(1, 8203, '\P{^Is_Line_Break=		ZWSPACE}', "");
Expect(0, 8204, '\p{Is_Line_Break=		ZWSPACE}', "");
Expect(1, 8204, '\p{^Is_Line_Break=		ZWSPACE}', "");
Expect(1, 8204, '\P{Is_Line_Break=		ZWSPACE}', "");
Expect(0, 8204, '\P{^Is_Line_Break=		ZWSPACE}', "");
Error('\p{Is_Lb=/a/_ZW}');
Error('\P{Is_Lb=/a/_ZW}');
Expect(1, 8203, '\p{Is_Lb=zw}', "");
Expect(0, 8203, '\p{^Is_Lb=zw}', "");
Expect(0, 8203, '\P{Is_Lb=zw}', "");
Expect(1, 8203, '\P{^Is_Lb=zw}', "");
Expect(0, 8204, '\p{Is_Lb=zw}', "");
Expect(1, 8204, '\p{^Is_Lb=zw}', "");
Expect(1, 8204, '\P{Is_Lb=zw}', "");
Expect(0, 8204, '\P{^Is_Lb=zw}', "");
Expect(1, 8203, '\p{Is_Lb: ZW}', "");
Expect(0, 8203, '\p{^Is_Lb: ZW}', "");
Expect(0, 8203, '\P{Is_Lb: ZW}', "");
Expect(1, 8203, '\P{^Is_Lb: ZW}', "");
Expect(0, 8204, '\p{Is_Lb: ZW}', "");
Expect(1, 8204, '\p{^Is_Lb: ZW}', "");
Expect(1, 8204, '\P{Is_Lb: ZW}', "");
Expect(0, 8204, '\P{^Is_Lb: ZW}', "");
Error('\p{Line_Break=/a/__jl}');
Error('\P{Line_Break=/a/__jl}');
Expect(1, 43388, '\p{Line_Break=jl}', "");
Expect(0, 43388, '\p{^Line_Break=jl}', "");
Expect(0, 43388, '\P{Line_Break=jl}', "");
Expect(1, 43388, '\P{^Line_Break=jl}', "");
Expect(0, 43389, '\p{Line_Break=jl}', "");
Expect(1, 43389, '\p{^Line_Break=jl}', "");
Expect(1, 43389, '\P{Line_Break=jl}', "");
Expect(0, 43389, '\P{^Line_Break=jl}', "");
Expect(1, 43388, '\p{Line_Break= 	jl}', "");
Expect(0, 43388, '\p{^Line_Break= 	jl}', "");
Expect(0, 43388, '\P{Line_Break= 	jl}', "");
Expect(1, 43388, '\P{^Line_Break= 	jl}', "");
Expect(0, 43389, '\p{Line_Break= 	jl}', "");
Expect(1, 43389, '\p{^Line_Break= 	jl}', "");
Expect(1, 43389, '\P{Line_Break= 	jl}', "");
Expect(0, 43389, '\P{^Line_Break= 	jl}', "");
Error('\p{Lb=:=	-jl}');
Error('\P{Lb=:=	-jl}');
Expect(1, 43388, '\p{Lb=jl}', "");
Expect(0, 43388, '\p{^Lb=jl}', "");
Expect(0, 43388, '\P{Lb=jl}', "");
Expect(1, 43388, '\P{^Lb=jl}', "");
Expect(0, 43389, '\p{Lb=jl}', "");
Expect(1, 43389, '\p{^Lb=jl}', "");
Expect(1, 43389, '\P{Lb=jl}', "");
Expect(0, 43389, '\P{^Lb=jl}', "");
Expect(1, 43388, '\p{Lb=_JL}', "");
Expect(0, 43388, '\p{^Lb=_JL}', "");
Expect(0, 43388, '\P{Lb=_JL}', "");
Expect(1, 43388, '\P{^Lb=_JL}', "");
Expect(0, 43389, '\p{Lb=_JL}', "");
Expect(1, 43389, '\p{^Lb=_JL}', "");
Expect(1, 43389, '\P{Lb=_JL}', "");
Expect(0, 43389, '\P{^Lb=_JL}', "");
Error('\p{Is_Line_Break= -JL/a/}');
Error('\P{Is_Line_Break= -JL/a/}');
Expect(1, 43388, '\p{Is_Line_Break=jl}', "");
Expect(0, 43388, '\p{^Is_Line_Break=jl}', "");
Expect(0, 43388, '\P{Is_Line_Break=jl}', "");
Expect(1, 43388, '\P{^Is_Line_Break=jl}', "");
Expect(0, 43389, '\p{Is_Line_Break=jl}', "");
Expect(1, 43389, '\p{^Is_Line_Break=jl}', "");
Expect(1, 43389, '\P{Is_Line_Break=jl}', "");
Expect(0, 43389, '\P{^Is_Line_Break=jl}', "");
Expect(1, 43388, '\p{Is_Line_Break= _JL}', "");
Expect(0, 43388, '\p{^Is_Line_Break= _JL}', "");
Expect(0, 43388, '\P{Is_Line_Break= _JL}', "");
Expect(1, 43388, '\P{^Is_Line_Break= _JL}', "");
Expect(0, 43389, '\p{Is_Line_Break= _JL}', "");
Expect(1, 43389, '\p{^Is_Line_Break= _JL}', "");
Expect(1, 43389, '\P{Is_Line_Break= _JL}', "");
Expect(0, 43389, '\P{^Is_Line_Break= _JL}', "");
Error('\p{Is_Lb=:=jl}');
Error('\P{Is_Lb=:=jl}');
Expect(1, 43388, '\p{Is_Lb:	jl}', "");
Expect(0, 43388, '\p{^Is_Lb:	jl}', "");
Expect(0, 43388, '\P{Is_Lb:	jl}', "");
Expect(1, 43388, '\P{^Is_Lb:	jl}', "");
Expect(0, 43389, '\p{Is_Lb:	jl}', "");
Expect(1, 43389, '\p{^Is_Lb:	jl}', "");
Expect(1, 43389, '\P{Is_Lb:	jl}', "");
Expect(0, 43389, '\P{^Is_Lb:	jl}', "");
Expect(1, 43388, '\p{Is_Lb: jl}', "");
Expect(0, 43388, '\p{^Is_Lb: jl}', "");
Expect(0, 43388, '\P{Is_Lb: jl}', "");
Expect(1, 43388, '\P{^Is_Lb: jl}', "");
Expect(0, 43389, '\p{Is_Lb: jl}', "");
Expect(1, 43389, '\p{^Is_Lb: jl}', "");
Expect(1, 43389, '\P{Is_Lb: jl}', "");
Expect(0, 43389, '\P{^Is_Lb: jl}', "");
Error('\p{Line_Break=/a/ 	BREAK_Symbols}');
Error('\P{Line_Break=/a/ 	BREAK_Symbols}');
Expect(1, 47, '\p{Line_Break=breaksymbols}', "");
Expect(0, 47, '\p{^Line_Break=breaksymbols}', "");
Expect(0, 47, '\P{Line_Break=breaksymbols}', "");
Expect(1, 47, '\P{^Line_Break=breaksymbols}', "");
Expect(0, 48, '\p{Line_Break=breaksymbols}', "");
Expect(1, 48, '\p{^Line_Break=breaksymbols}', "");
Expect(1, 48, '\P{Line_Break=breaksymbols}', "");
Expect(0, 48, '\P{^Line_Break=breaksymbols}', "");
Expect(1, 47, '\p{Line_Break= 	Break_Symbols}', "");
Expect(0, 47, '\p{^Line_Break= 	Break_Symbols}', "");
Expect(0, 47, '\P{Line_Break= 	Break_Symbols}', "");
Expect(1, 47, '\P{^Line_Break= 	Break_Symbols}', "");
Expect(0, 48, '\p{Line_Break= 	Break_Symbols}', "");
Expect(1, 48, '\p{^Line_Break= 	Break_Symbols}', "");
Expect(1, 48, '\P{Line_Break= 	Break_Symbols}', "");
Expect(0, 48, '\P{^Line_Break= 	Break_Symbols}', "");
Error('\p{Lb:	/a/	-sy}');
Error('\P{Lb:	/a/	-sy}');
Expect(1, 47, '\p{Lb:   sy}', "");
Expect(0, 47, '\p{^Lb:   sy}', "");
Expect(0, 47, '\P{Lb:   sy}', "");
Expect(1, 47, '\P{^Lb:   sy}', "");
Expect(0, 48, '\p{Lb:   sy}', "");
Expect(1, 48, '\p{^Lb:   sy}', "");
Expect(1, 48, '\P{Lb:   sy}', "");
Expect(0, 48, '\P{^Lb:   sy}', "");
Expect(1, 47, '\p{Lb=SY}', "");
Expect(0, 47, '\p{^Lb=SY}', "");
Expect(0, 47, '\P{Lb=SY}', "");
Expect(1, 47, '\P{^Lb=SY}', "");
Expect(0, 48, '\p{Lb=SY}', "");
Expect(1, 48, '\p{^Lb=SY}', "");
Expect(1, 48, '\P{Lb=SY}', "");
Expect(0, 48, '\P{^Lb=SY}', "");
Error('\p{Is_Line_Break=	/a/break_Symbols}');
Error('\P{Is_Line_Break=	/a/break_Symbols}');
Expect(1, 47, '\p{Is_Line_Break=breaksymbols}', "");
Expect(0, 47, '\p{^Is_Line_Break=breaksymbols}', "");
Expect(0, 47, '\P{Is_Line_Break=breaksymbols}', "");
Expect(1, 47, '\P{^Is_Line_Break=breaksymbols}', "");
Expect(0, 48, '\p{Is_Line_Break=breaksymbols}', "");
Expect(1, 48, '\p{^Is_Line_Break=breaksymbols}', "");
Expect(1, 48, '\P{Is_Line_Break=breaksymbols}', "");
Expect(0, 48, '\P{^Is_Line_Break=breaksymbols}', "");
Expect(1, 47, '\p{Is_Line_Break=_	BREAK_symbols}', "");
Expect(0, 47, '\p{^Is_Line_Break=_	BREAK_symbols}', "");
Expect(0, 47, '\P{Is_Line_Break=_	BREAK_symbols}', "");
Expect(1, 47, '\P{^Is_Line_Break=_	BREAK_symbols}', "");
Expect(0, 48, '\p{Is_Line_Break=_	BREAK_symbols}', "");
Expect(1, 48, '\p{^Is_Line_Break=_	BREAK_symbols}', "");
Expect(1, 48, '\P{Is_Line_Break=_	BREAK_symbols}', "");
Expect(0, 48, '\P{^Is_Line_Break=_	BREAK_symbols}', "");
Error('\p{Is_Lb=	:=SY}');
Error('\P{Is_Lb=	:=SY}');
Expect(1, 47, '\p{Is_Lb=sy}', "");
Expect(0, 47, '\p{^Is_Lb=sy}', "");
Expect(0, 47, '\P{Is_Lb=sy}', "");
Expect(1, 47, '\P{^Is_Lb=sy}', "");
Expect(0, 48, '\p{Is_Lb=sy}', "");
Expect(1, 48, '\p{^Is_Lb=sy}', "");
Expect(1, 48, '\P{Is_Lb=sy}', "");
Expect(0, 48, '\P{^Is_Lb=sy}', "");
Expect(1, 47, '\p{Is_Lb:	_SY}', "");
Expect(0, 47, '\p{^Is_Lb:	_SY}', "");
Expect(0, 47, '\P{Is_Lb:	_SY}', "");
Expect(1, 47, '\P{^Is_Lb:	_SY}', "");
Expect(0, 48, '\p{Is_Lb:	_SY}', "");
Expect(1, 48, '\p{^Is_Lb:	_SY}', "");
Expect(1, 48, '\P{Is_Lb:	_SY}', "");
Expect(0, 48, '\P{^Is_Lb:	_SY}', "");
Error('\p{Line_Break=/a/- Prefix_Numeric}');
Error('\P{Line_Break=/a/- Prefix_Numeric}');
Expect(1, 65510, '\p{Line_Break=prefixnumeric}', "");
Expect(0, 65510, '\p{^Line_Break=prefixnumeric}', "");
Expect(0, 65510, '\P{Line_Break=prefixnumeric}', "");
Expect(1, 65510, '\P{^Line_Break=prefixnumeric}', "");
Expect(0, 65511, '\p{Line_Break=prefixnumeric}', "");
Expect(1, 65511, '\p{^Line_Break=prefixnumeric}', "");
Expect(1, 65511, '\P{Line_Break=prefixnumeric}', "");
Expect(0, 65511, '\P{^Line_Break=prefixnumeric}', "");
Expect(1, 65510, '\p{Line_Break=_-Prefix_Numeric}', "");
Expect(0, 65510, '\p{^Line_Break=_-Prefix_Numeric}', "");
Expect(0, 65510, '\P{Line_Break=_-Prefix_Numeric}', "");
Expect(1, 65510, '\P{^Line_Break=_-Prefix_Numeric}', "");
Expect(0, 65511, '\p{Line_Break=_-Prefix_Numeric}', "");
Expect(1, 65511, '\p{^Line_Break=_-Prefix_Numeric}', "");
Expect(1, 65511, '\P{Line_Break=_-Prefix_Numeric}', "");
Expect(0, 65511, '\P{^Line_Break=_-Prefix_Numeric}', "");
Error('\p{Lb:/a/_	pr}');
Error('\P{Lb:/a/_	pr}');
Expect(1, 65510, '\p{Lb:   pr}', "");
Expect(0, 65510, '\p{^Lb:   pr}', "");
Expect(0, 65510, '\P{Lb:   pr}', "");
Expect(1, 65510, '\P{^Lb:   pr}', "");
Expect(0, 65511, '\p{Lb:   pr}', "");
Expect(1, 65511, '\p{^Lb:   pr}', "");
Expect(1, 65511, '\P{Lb:   pr}', "");
Expect(0, 65511, '\P{^Lb:   pr}', "");
Expect(1, 65510, '\p{Lb= -PR}', "");
Expect(0, 65510, '\p{^Lb= -PR}', "");
Expect(0, 65510, '\P{Lb= -PR}', "");
Expect(1, 65510, '\P{^Lb= -PR}', "");
Expect(0, 65511, '\p{Lb= -PR}', "");
Expect(1, 65511, '\p{^Lb= -PR}', "");
Expect(1, 65511, '\P{Lb= -PR}', "");
Expect(0, 65511, '\P{^Lb= -PR}', "");
Error('\p{Is_Line_Break=_:=Prefix_numeric}');
Error('\P{Is_Line_Break=_:=Prefix_numeric}');
Expect(1, 65510, '\p{Is_Line_Break=prefixnumeric}', "");
Expect(0, 65510, '\p{^Is_Line_Break=prefixnumeric}', "");
Expect(0, 65510, '\P{Is_Line_Break=prefixnumeric}', "");
Expect(1, 65510, '\P{^Is_Line_Break=prefixnumeric}', "");
Expect(0, 65511, '\p{Is_Line_Break=prefixnumeric}', "");
Expect(1, 65511, '\p{^Is_Line_Break=prefixnumeric}', "");
Expect(1, 65511, '\P{Is_Line_Break=prefixnumeric}', "");
Expect(0, 65511, '\P{^Is_Line_Break=prefixnumeric}', "");
Expect(1, 65510, '\p{Is_Line_Break=	prefix_Numeric}', "");
Expect(0, 65510, '\p{^Is_Line_Break=	prefix_Numeric}', "");
Expect(0, 65510, '\P{Is_Line_Break=	prefix_Numeric}', "");
Expect(1, 65510, '\P{^Is_Line_Break=	prefix_Numeric}', "");
Expect(0, 65511, '\p{Is_Line_Break=	prefix_Numeric}', "");
Expect(1, 65511, '\p{^Is_Line_Break=	prefix_Numeric}', "");
Expect(1, 65511, '\P{Is_Line_Break=	prefix_Numeric}', "");
Expect(0, 65511, '\P{^Is_Line_Break=	prefix_Numeric}', "");
Error('\p{Is_Lb=-	PR:=}');
Error('\P{Is_Lb=-	PR:=}');
Expect(1, 65510, '\p{Is_Lb=pr}', "");
Expect(0, 65510, '\p{^Is_Lb=pr}', "");
Expect(0, 65510, '\P{Is_Lb=pr}', "");
Expect(1, 65510, '\P{^Is_Lb=pr}', "");
Expect(0, 65511, '\p{Is_Lb=pr}', "");
Expect(1, 65511, '\p{^Is_Lb=pr}', "");
Expect(1, 65511, '\P{Is_Lb=pr}', "");
Expect(0, 65511, '\P{^Is_Lb=pr}', "");
Expect(1, 65510, '\p{Is_Lb=_ pr}', "");
Expect(0, 65510, '\p{^Is_Lb=_ pr}', "");
Expect(0, 65510, '\P{Is_Lb=_ pr}', "");
Expect(1, 65510, '\P{^Is_Lb=_ pr}', "");
Expect(0, 65511, '\p{Is_Lb=_ pr}', "");
Expect(1, 65511, '\p{^Is_Lb=_ pr}', "");
Expect(1, 65511, '\P{Is_Lb=_ pr}', "");
Expect(0, 65511, '\P{^Is_Lb=_ pr}', "");
Error('\p{Line_Break=	close_PARENTHESIS:=}');
Error('\P{Line_Break=	close_PARENTHESIS:=}');
Expect(1, 93, '\p{Line_Break:   closeparenthesis}', "");
Expect(0, 93, '\p{^Line_Break:   closeparenthesis}', "");
Expect(0, 93, '\P{Line_Break:   closeparenthesis}', "");
Expect(1, 93, '\P{^Line_Break:   closeparenthesis}', "");
Expect(0, 94, '\p{Line_Break:   closeparenthesis}', "");
Expect(1, 94, '\p{^Line_Break:   closeparenthesis}', "");
Expect(1, 94, '\P{Line_Break:   closeparenthesis}', "");
Expect(0, 94, '\P{^Line_Break:   closeparenthesis}', "");
Expect(1, 93, '\p{Line_Break:  -close_PARENTHESIS}', "");
Expect(0, 93, '\p{^Line_Break:  -close_PARENTHESIS}', "");
Expect(0, 93, '\P{Line_Break:  -close_PARENTHESIS}', "");
Expect(1, 93, '\P{^Line_Break:  -close_PARENTHESIS}', "");
Expect(0, 94, '\p{Line_Break:  -close_PARENTHESIS}', "");
Expect(1, 94, '\p{^Line_Break:  -close_PARENTHESIS}', "");
Expect(1, 94, '\P{Line_Break:  -close_PARENTHESIS}', "");
Expect(0, 94, '\P{^Line_Break:  -close_PARENTHESIS}', "");
Error('\p{Lb=:=- CP}');
Error('\P{Lb=:=- CP}');
Expect(1, 93, '\p{Lb:   cp}', "");
Expect(0, 93, '\p{^Lb:   cp}', "");
Expect(0, 93, '\P{Lb:   cp}', "");
Expect(1, 93, '\P{^Lb:   cp}', "");
Expect(0, 94, '\p{Lb:   cp}', "");
Expect(1, 94, '\p{^Lb:   cp}', "");
Expect(1, 94, '\P{Lb:   cp}', "");
Expect(0, 94, '\P{^Lb:   cp}', "");
Expect(1, 93, '\p{Lb=- CP}', "");
Expect(0, 93, '\p{^Lb=- CP}', "");
Expect(0, 93, '\P{Lb=- CP}', "");
Expect(1, 93, '\P{^Lb=- CP}', "");
Expect(0, 94, '\p{Lb=- CP}', "");
Expect(1, 94, '\p{^Lb=- CP}', "");
Expect(1, 94, '\P{Lb=- CP}', "");
Expect(0, 94, '\P{^Lb=- CP}', "");
Error('\p{Is_Line_Break=	-close_Parenthesis:=}');
Error('\P{Is_Line_Break=	-close_Parenthesis:=}');
Expect(1, 93, '\p{Is_Line_Break=closeparenthesis}', "");
Expect(0, 93, '\p{^Is_Line_Break=closeparenthesis}', "");
Expect(0, 93, '\P{Is_Line_Break=closeparenthesis}', "");
Expect(1, 93, '\P{^Is_Line_Break=closeparenthesis}', "");
Expect(0, 94, '\p{Is_Line_Break=closeparenthesis}', "");
Expect(1, 94, '\p{^Is_Line_Break=closeparenthesis}', "");
Expect(1, 94, '\P{Is_Line_Break=closeparenthesis}', "");
Expect(0, 94, '\P{^Is_Line_Break=closeparenthesis}', "");
Expect(1, 93, '\p{Is_Line_Break=  Close_Parenthesis}', "");
Expect(0, 93, '\p{^Is_Line_Break=  Close_Parenthesis}', "");
Expect(0, 93, '\P{Is_Line_Break=  Close_Parenthesis}', "");
Expect(1, 93, '\P{^Is_Line_Break=  Close_Parenthesis}', "");
Expect(0, 94, '\p{Is_Line_Break=  Close_Parenthesis}', "");
Expect(1, 94, '\p{^Is_Line_Break=  Close_Parenthesis}', "");
Expect(1, 94, '\P{Is_Line_Break=  Close_Parenthesis}', "");
Expect(0, 94, '\P{^Is_Line_Break=  Close_Parenthesis}', "");
Error('\p{Is_Lb=/a/ CP}');
Error('\P{Is_Lb=/a/ CP}');
Expect(1, 93, '\p{Is_Lb:   cp}', "");
Expect(0, 93, '\p{^Is_Lb:   cp}', "");
Expect(0, 93, '\P{Is_Lb:   cp}', "");
Expect(1, 93, '\P{^Is_Lb:   cp}', "");
Expect(0, 94, '\p{Is_Lb:   cp}', "");
Expect(1, 94, '\p{^Is_Lb:   cp}', "");
Expect(1, 94, '\P{Is_Lb:   cp}', "");
Expect(0, 94, '\P{^Is_Lb:   cp}', "");
Expect(1, 93, '\p{Is_Lb=- cp}', "");
Expect(0, 93, '\p{^Is_Lb=- cp}', "");
Expect(0, 93, '\P{Is_Lb=- cp}', "");
Expect(1, 93, '\P{^Is_Lb=- cp}', "");
Expect(0, 94, '\p{Is_Lb=- cp}', "");
Expect(1, 94, '\p{^Is_Lb=- cp}', "");
Expect(1, 94, '\P{Is_Lb=- cp}', "");
Expect(0, 94, '\P{^Is_Lb=- cp}', "");
Error('\p{Line_Break=:=-H3}');
Error('\P{Line_Break=:=-H3}');
Expect(1, 55203, '\p{Line_Break=h3}', "");
Expect(0, 55203, '\p{^Line_Break=h3}', "");
Expect(0, 55203, '\P{Line_Break=h3}', "");
Expect(1, 55203, '\P{^Line_Break=h3}', "");
Expect(0, 55204, '\p{Line_Break=h3}', "");
Expect(1, 55204, '\p{^Line_Break=h3}', "");
Expect(1, 55204, '\P{Line_Break=h3}', "");
Expect(0, 55204, '\P{^Line_Break=h3}', "");
Expect(1, 55203, '\p{Line_Break= H3}', "");
Expect(0, 55203, '\p{^Line_Break= H3}', "");
Expect(0, 55203, '\P{Line_Break= H3}', "");
Expect(1, 55203, '\P{^Line_Break= H3}', "");
Expect(0, 55204, '\p{Line_Break= H3}', "");
Expect(1, 55204, '\p{^Line_Break= H3}', "");
Expect(1, 55204, '\P{Line_Break= H3}', "");
Expect(0, 55204, '\P{^Line_Break= H3}', "");
Error('\p{Lb=/a/H3}');
Error('\P{Lb=/a/H3}');
Expect(1, 55203, '\p{Lb:	h3}', "");
Expect(0, 55203, '\p{^Lb:	h3}', "");
Expect(0, 55203, '\P{Lb:	h3}', "");
Expect(1, 55203, '\P{^Lb:	h3}', "");
Expect(0, 55204, '\p{Lb:	h3}', "");
Expect(1, 55204, '\p{^Lb:	h3}', "");
Expect(1, 55204, '\P{Lb:	h3}', "");
Expect(0, 55204, '\P{^Lb:	h3}', "");
Expect(1, 55203, '\p{Lb= -H3}', "");
Expect(0, 55203, '\p{^Lb= -H3}', "");
Expect(0, 55203, '\P{Lb= -H3}', "");
Expect(1, 55203, '\P{^Lb= -H3}', "");
Expect(0, 55204, '\p{Lb= -H3}', "");
Expect(1, 55204, '\p{^Lb= -H3}', "");
Expect(1, 55204, '\P{Lb= -H3}', "");
Expect(0, 55204, '\P{^Lb= -H3}', "");
Error('\p{Is_Line_Break=:=_	h3}');
Error('\P{Is_Line_Break=:=_	h3}');
Expect(1, 55203, '\p{Is_Line_Break=h3}', "");
Expect(0, 55203, '\p{^Is_Line_Break=h3}', "");
Expect(0, 55203, '\P{Is_Line_Break=h3}', "");
Expect(1, 55203, '\P{^Is_Line_Break=h3}', "");
Expect(0, 55204, '\p{Is_Line_Break=h3}', "");
Expect(1, 55204, '\p{^Is_Line_Break=h3}', "");
Expect(1, 55204, '\P{Is_Line_Break=h3}', "");
Expect(0, 55204, '\P{^Is_Line_Break=h3}', "");
Expect(1, 55203, '\p{Is_Line_Break:--H3}', "");
Expect(0, 55203, '\p{^Is_Line_Break:--H3}', "");
Expect(0, 55203, '\P{Is_Line_Break:--H3}', "");
Expect(1, 55203, '\P{^Is_Line_Break:--H3}', "");
Expect(0, 55204, '\p{Is_Line_Break:--H3}', "");
Expect(1, 55204, '\p{^Is_Line_Break:--H3}', "");
Expect(1, 55204, '\P{Is_Line_Break:--H3}', "");
Expect(0, 55204, '\P{^Is_Line_Break:--H3}', "");
Error('\p{Is_Lb=_/a/H3}');
Error('\P{Is_Lb=_/a/H3}');
Expect(1, 55203, '\p{Is_Lb=h3}', "");
Expect(0, 55203, '\p{^Is_Lb=h3}', "");
Expect(0, 55203, '\P{Is_Lb=h3}', "");
Expect(1, 55203, '\P{^Is_Lb=h3}', "");
Expect(0, 55204, '\p{Is_Lb=h3}', "");
Expect(1, 55204, '\p{^Is_Lb=h3}', "");
Expect(1, 55204, '\P{Is_Lb=h3}', "");
Expect(0, 55204, '\P{^Is_Lb=h3}', "");
Expect(1, 55203, '\p{Is_Lb:   H3}', "");
Expect(0, 55203, '\p{^Is_Lb:   H3}', "");
Expect(0, 55203, '\P{Is_Lb:   H3}', "");
Expect(1, 55203, '\P{^Is_Lb:   H3}', "");
Expect(0, 55204, '\p{Is_Lb:   H3}', "");
Expect(1, 55204, '\p{^Is_Lb:   H3}', "");
Expect(1, 55204, '\P{Is_Lb:   H3}', "");
Expect(0, 55204, '\P{^Is_Lb:   H3}', "");
Error('\p{Line_Break=/a/ Hebrew_LETTER}');
Error('\P{Line_Break=/a/ Hebrew_LETTER}');
Expect(1, 64335, '\p{Line_Break: hebrewletter}', "");
Expect(0, 64335, '\p{^Line_Break: hebrewletter}', "");
Expect(0, 64335, '\P{Line_Break: hebrewletter}', "");
Expect(1, 64335, '\P{^Line_Break: hebrewletter}', "");
Expect(0, 64336, '\p{Line_Break: hebrewletter}', "");
Expect(1, 64336, '\p{^Line_Break: hebrewletter}', "");
Expect(1, 64336, '\P{Line_Break: hebrewletter}', "");
Expect(0, 64336, '\P{^Line_Break: hebrewletter}', "");
Expect(1, 64335, '\p{Line_Break= hebrew_letter}', "");
Expect(0, 64335, '\p{^Line_Break= hebrew_letter}', "");
Expect(0, 64335, '\P{Line_Break= hebrew_letter}', "");
Expect(1, 64335, '\P{^Line_Break= hebrew_letter}', "");
Expect(0, 64336, '\p{Line_Break= hebrew_letter}', "");
Expect(1, 64336, '\p{^Line_Break= hebrew_letter}', "");
Expect(1, 64336, '\P{Line_Break= hebrew_letter}', "");
Expect(0, 64336, '\P{^Line_Break= hebrew_letter}', "");
Error('\p{Lb=	_hl/a/}');
Error('\P{Lb=	_hl/a/}');
Expect(1, 64335, '\p{Lb=hl}', "");
Expect(0, 64335, '\p{^Lb=hl}', "");
Expect(0, 64335, '\P{Lb=hl}', "");
Expect(1, 64335, '\P{^Lb=hl}', "");
Expect(0, 64336, '\p{Lb=hl}', "");
Expect(1, 64336, '\p{^Lb=hl}', "");
Expect(1, 64336, '\P{Lb=hl}', "");
Expect(0, 64336, '\P{^Lb=hl}', "");
Expect(1, 64335, '\p{Lb= HL}', "");
Expect(0, 64335, '\p{^Lb= HL}', "");
Expect(0, 64335, '\P{Lb= HL}', "");
Expect(1, 64335, '\P{^Lb= HL}', "");
Expect(0, 64336, '\p{Lb= HL}', "");
Expect(1, 64336, '\p{^Lb= HL}', "");
Expect(1, 64336, '\P{Lb= HL}', "");
Expect(0, 64336, '\P{^Lb= HL}', "");
Error('\p{Is_Line_Break=/a/	hebrew_Letter}');
Error('\P{Is_Line_Break=/a/	hebrew_Letter}');
Expect(1, 64335, '\p{Is_Line_Break=hebrewletter}', "");
Expect(0, 64335, '\p{^Is_Line_Break=hebrewletter}', "");
Expect(0, 64335, '\P{Is_Line_Break=hebrewletter}', "");
Expect(1, 64335, '\P{^Is_Line_Break=hebrewletter}', "");
Expect(0, 64336, '\p{Is_Line_Break=hebrewletter}', "");
Expect(1, 64336, '\p{^Is_Line_Break=hebrewletter}', "");
Expect(1, 64336, '\P{Is_Line_Break=hebrewletter}', "");
Expect(0, 64336, '\P{^Is_Line_Break=hebrewletter}', "");
Expect(1, 64335, '\p{Is_Line_Break:- HEBREW_letter}', "");
Expect(0, 64335, '\p{^Is_Line_Break:- HEBREW_letter}', "");
Expect(0, 64335, '\P{Is_Line_Break:- HEBREW_letter}', "");
Expect(1, 64335, '\P{^Is_Line_Break:- HEBREW_letter}', "");
Expect(0, 64336, '\p{Is_Line_Break:- HEBREW_letter}', "");
Expect(1, 64336, '\p{^Is_Line_Break:- HEBREW_letter}', "");
Expect(1, 64336, '\P{Is_Line_Break:- HEBREW_letter}', "");
Expect(0, 64336, '\P{^Is_Line_Break:- HEBREW_letter}', "");
Error('\p{Is_Lb=/a/_HL}');
Error('\P{Is_Lb=/a/_HL}');
Expect(1, 64335, '\p{Is_Lb=hl}', "");
Expect(0, 64335, '\p{^Is_Lb=hl}', "");
Expect(0, 64335, '\P{Is_Lb=hl}', "");
Expect(1, 64335, '\P{^Is_Lb=hl}', "");
Expect(0, 64336, '\p{Is_Lb=hl}', "");
Expect(1, 64336, '\p{^Is_Lb=hl}', "");
Expect(1, 64336, '\P{Is_Lb=hl}', "");
Expect(0, 64336, '\P{^Is_Lb=hl}', "");
Expect(1, 64335, '\p{Is_Lb= _HL}', "");
Expect(0, 64335, '\p{^Is_Lb= _HL}', "");
Expect(0, 64335, '\P{Is_Lb= _HL}', "");
Expect(1, 64335, '\P{^Is_Lb= _HL}', "");
Expect(0, 64336, '\p{Is_Lb= _HL}', "");
Expect(1, 64336, '\p{^Is_Lb= _HL}', "");
Expect(1, 64336, '\P{Is_Lb= _HL}', "");
Expect(0, 64336, '\P{^Is_Lb= _HL}', "");
Error('\p{Line_Break= :=CARRIAGE_Return}');
Error('\P{Line_Break= :=CARRIAGE_Return}');
Expect(1, 13, '\p{Line_Break=carriagereturn}', "");
Expect(0, 13, '\p{^Line_Break=carriagereturn}', "");
Expect(0, 13, '\P{Line_Break=carriagereturn}', "");
Expect(1, 13, '\P{^Line_Break=carriagereturn}', "");
Expect(0, 14, '\p{Line_Break=carriagereturn}', "");
Expect(1, 14, '\p{^Line_Break=carriagereturn}', "");
Expect(1, 14, '\P{Line_Break=carriagereturn}', "");
Expect(0, 14, '\P{^Line_Break=carriagereturn}', "");
Expect(1, 13, '\p{Line_Break=_CARRIAGE_return}', "");
Expect(0, 13, '\p{^Line_Break=_CARRIAGE_return}', "");
Expect(0, 13, '\P{Line_Break=_CARRIAGE_return}', "");
Expect(1, 13, '\P{^Line_Break=_CARRIAGE_return}', "");
Expect(0, 14, '\p{Line_Break=_CARRIAGE_return}', "");
Expect(1, 14, '\p{^Line_Break=_CARRIAGE_return}', "");
Expect(1, 14, '\P{Line_Break=_CARRIAGE_return}', "");
Expect(0, 14, '\P{^Line_Break=_CARRIAGE_return}', "");
Error('\p{Lb=/a/- CR}');
Error('\P{Lb=/a/- CR}');
Expect(1, 13, '\p{Lb=cr}', "");
Expect(0, 13, '\p{^Lb=cr}', "");
Expect(0, 13, '\P{Lb=cr}', "");
Expect(1, 13, '\P{^Lb=cr}', "");
Expect(0, 14, '\p{Lb=cr}', "");
Expect(1, 14, '\p{^Lb=cr}', "");
Expect(1, 14, '\P{Lb=cr}', "");
Expect(0, 14, '\P{^Lb=cr}', "");
Expect(1, 13, '\p{Lb=	-cr}', "");
Expect(0, 13, '\p{^Lb=	-cr}', "");
Expect(0, 13, '\P{Lb=	-cr}', "");
Expect(1, 13, '\P{^Lb=	-cr}', "");
Expect(0, 14, '\p{Lb=	-cr}', "");
Expect(1, 14, '\p{^Lb=	-cr}', "");
Expect(1, 14, '\P{Lb=	-cr}', "");
Expect(0, 14, '\P{^Lb=	-cr}', "");
Error('\p{Is_Line_Break:-CARRIAGE_return:=}');
Error('\P{Is_Line_Break:-CARRIAGE_return:=}');
Expect(1, 13, '\p{Is_Line_Break: carriagereturn}', "");
Expect(0, 13, '\p{^Is_Line_Break: carriagereturn}', "");
Expect(0, 13, '\P{Is_Line_Break: carriagereturn}', "");
Expect(1, 13, '\P{^Is_Line_Break: carriagereturn}', "");
Expect(0, 14, '\p{Is_Line_Break: carriagereturn}', "");
Expect(1, 14, '\p{^Is_Line_Break: carriagereturn}', "");
Expect(1, 14, '\P{Is_Line_Break: carriagereturn}', "");
Expect(0, 14, '\P{^Is_Line_Break: carriagereturn}', "");
Expect(1, 13, '\p{Is_Line_Break:_ carriage_RETURN}', "");
Expect(0, 13, '\p{^Is_Line_Break:_ carriage_RETURN}', "");
Expect(0, 13, '\P{Is_Line_Break:_ carriage_RETURN}', "");
Expect(1, 13, '\P{^Is_Line_Break:_ carriage_RETURN}', "");
Expect(0, 14, '\p{Is_Line_Break:_ carriage_RETURN}', "");
Expect(1, 14, '\p{^Is_Line_Break:_ carriage_RETURN}', "");
Expect(1, 14, '\P{Is_Line_Break:_ carriage_RETURN}', "");
Expect(0, 14, '\P{^Is_Line_Break:_ carriage_RETURN}', "");
Error('\p{Is_Lb:	 :=cr}');
Error('\P{Is_Lb:	 :=cr}');
Expect(1, 13, '\p{Is_Lb=cr}', "");
Expect(0, 13, '\p{^Is_Lb=cr}', "");
Expect(0, 13, '\P{Is_Lb=cr}', "");
Expect(1, 13, '\P{^Is_Lb=cr}', "");
Expect(0, 14, '\p{Is_Lb=cr}', "");
Expect(1, 14, '\p{^Is_Lb=cr}', "");
Expect(1, 14, '\P{Is_Lb=cr}', "");
Expect(0, 14, '\P{^Is_Lb=cr}', "");
Expect(1, 13, '\p{Is_Lb: - cr}', "");
Expect(0, 13, '\p{^Is_Lb: - cr}', "");
Expect(0, 13, '\P{Is_Lb: - cr}', "");
Expect(1, 13, '\P{^Is_Lb: - cr}', "");
Expect(0, 14, '\p{Is_Lb: - cr}', "");
Expect(1, 14, '\p{^Is_Lb: - cr}', "");
Expect(1, 14, '\P{Is_Lb: - cr}', "");
Expect(0, 14, '\P{^Is_Lb: - cr}', "");
Error('\p{Line_Break= /a/CLOSE_punctuation}');
Error('\P{Line_Break= /a/CLOSE_punctuation}');
Expect(1, 78715, '\p{Line_Break=closepunctuation}', "");
Expect(0, 78715, '\p{^Line_Break=closepunctuation}', "");
Expect(0, 78715, '\P{Line_Break=closepunctuation}', "");
Expect(1, 78715, '\P{^Line_Break=closepunctuation}', "");
Expect(0, 78716, '\p{Line_Break=closepunctuation}', "");
Expect(1, 78716, '\p{^Line_Break=closepunctuation}', "");
Expect(1, 78716, '\P{Line_Break=closepunctuation}', "");
Expect(0, 78716, '\P{^Line_Break=closepunctuation}', "");
Expect(1, 78715, '\p{Line_Break=- Close_punctuation}', "");
Expect(0, 78715, '\p{^Line_Break=- Close_punctuation}', "");
Expect(0, 78715, '\P{Line_Break=- Close_punctuation}', "");
Expect(1, 78715, '\P{^Line_Break=- Close_punctuation}', "");
Expect(0, 78716, '\p{Line_Break=- Close_punctuation}', "");
Expect(1, 78716, '\p{^Line_Break=- Close_punctuation}', "");
Expect(1, 78716, '\P{Line_Break=- Close_punctuation}', "");
Expect(0, 78716, '\P{^Line_Break=- Close_punctuation}', "");
Error('\p{Lb:	/a/-	CL}');
Error('\P{Lb:	/a/-	CL}');
Expect(1, 78715, '\p{Lb=cl}', "");
Expect(0, 78715, '\p{^Lb=cl}', "");
Expect(0, 78715, '\P{Lb=cl}', "");
Expect(1, 78715, '\P{^Lb=cl}', "");
Expect(0, 78716, '\p{Lb=cl}', "");
Expect(1, 78716, '\p{^Lb=cl}', "");
Expect(1, 78716, '\P{Lb=cl}', "");
Expect(0, 78716, '\P{^Lb=cl}', "");
Expect(1, 78715, '\p{Lb:	 -CL}', "");
Expect(0, 78715, '\p{^Lb:	 -CL}', "");
Expect(0, 78715, '\P{Lb:	 -CL}', "");
Expect(1, 78715, '\P{^Lb:	 -CL}', "");
Expect(0, 78716, '\p{Lb:	 -CL}', "");
Expect(1, 78716, '\p{^Lb:	 -CL}', "");
Expect(1, 78716, '\P{Lb:	 -CL}', "");
Expect(0, 78716, '\P{^Lb:	 -CL}', "");
Error('\p{Is_Line_Break: 	Close_punctuation:=}');
Error('\P{Is_Line_Break: 	Close_punctuation:=}');
Expect(1, 78715, '\p{Is_Line_Break:	closepunctuation}', "");
Expect(0, 78715, '\p{^Is_Line_Break:	closepunctuation}', "");
Expect(0, 78715, '\P{Is_Line_Break:	closepunctuation}', "");
Expect(1, 78715, '\P{^Is_Line_Break:	closepunctuation}', "");
Expect(0, 78716, '\p{Is_Line_Break:	closepunctuation}', "");
Expect(1, 78716, '\p{^Is_Line_Break:	closepunctuation}', "");
Expect(1, 78716, '\P{Is_Line_Break:	closepunctuation}', "");
Expect(0, 78716, '\P{^Is_Line_Break:	closepunctuation}', "");
Expect(1, 78715, '\p{Is_Line_Break=-_CLOSE_PUNCTUATION}', "");
Expect(0, 78715, '\p{^Is_Line_Break=-_CLOSE_PUNCTUATION}', "");
Expect(0, 78715, '\P{Is_Line_Break=-_CLOSE_PUNCTUATION}', "");
Expect(1, 78715, '\P{^Is_Line_Break=-_CLOSE_PUNCTUATION}', "");
Expect(0, 78716, '\p{Is_Line_Break=-_CLOSE_PUNCTUATION}', "");
Expect(1, 78716, '\p{^Is_Line_Break=-_CLOSE_PUNCTUATION}', "");
Expect(1, 78716, '\P{Is_Line_Break=-_CLOSE_PUNCTUATION}', "");
Expect(0, 78716, '\P{^Is_Line_Break=-_CLOSE_PUNCTUATION}', "");
Error('\p{Is_Lb=_	CL:=}');
Error('\P{Is_Lb=_	CL:=}');
Expect(1, 78715, '\p{Is_Lb=cl}', "");
Expect(0, 78715, '\p{^Is_Lb=cl}', "");
Expect(0, 78715, '\P{Is_Lb=cl}', "");
Expect(1, 78715, '\P{^Is_Lb=cl}', "");
Expect(0, 78716, '\p{Is_Lb=cl}', "");
Expect(1, 78716, '\p{^Is_Lb=cl}', "");
Expect(1, 78716, '\P{Is_Lb=cl}', "");
Expect(0, 78716, '\P{^Is_Lb=cl}', "");
Expect(1, 78715, '\p{Is_Lb=-CL}', "");
Expect(0, 78715, '\p{^Is_Lb=-CL}', "");
Expect(0, 78715, '\P{Is_Lb=-CL}', "");
Expect(1, 78715, '\P{^Is_Lb=-CL}', "");
Expect(0, 78716, '\p{Is_Lb=-CL}', "");
Expect(1, 78716, '\p{^Is_Lb=-CL}', "");
Expect(1, 78716, '\P{Is_Lb=-CL}', "");
Expect(0, 78716, '\P{^Is_Lb=-CL}', "");
Error('\p{Line_Break: -_Combining_MARK/a/}');
Error('\P{Line_Break: -_Combining_MARK/a/}');
Expect(1, 917999, '\p{Line_Break:   combiningmark}', "");
Expect(0, 917999, '\p{^Line_Break:   combiningmark}', "");
Expect(0, 917999, '\P{Line_Break:   combiningmark}', "");
Expect(1, 917999, '\P{^Line_Break:   combiningmark}', "");
Expect(0, 918000, '\p{Line_Break:   combiningmark}', "");
Expect(1, 918000, '\p{^Line_Break:   combiningmark}', "");
Expect(1, 918000, '\P{Line_Break:   combiningmark}', "");
Expect(0, 918000, '\P{^Line_Break:   combiningmark}', "");
Expect(1, 917999, '\p{Line_Break=-combining_mark}', "");
Expect(0, 917999, '\p{^Line_Break=-combining_mark}', "");
Expect(0, 917999, '\P{Line_Break=-combining_mark}', "");
Expect(1, 917999, '\P{^Line_Break=-combining_mark}', "");
Expect(0, 918000, '\p{Line_Break=-combining_mark}', "");
Expect(1, 918000, '\p{^Line_Break=-combining_mark}', "");
Expect(1, 918000, '\P{Line_Break=-combining_mark}', "");
Expect(0, 918000, '\P{^Line_Break=-combining_mark}', "");
Error('\p{Lb=/a/CM}');
Error('\P{Lb=/a/CM}');
Expect(1, 917999, '\p{Lb=cm}', "");
Expect(0, 917999, '\p{^Lb=cm}', "");
Expect(0, 917999, '\P{Lb=cm}', "");
Expect(1, 917999, '\P{^Lb=cm}', "");
Expect(0, 918000, '\p{Lb=cm}', "");
Expect(1, 918000, '\p{^Lb=cm}', "");
Expect(1, 918000, '\P{Lb=cm}', "");
Expect(0, 918000, '\P{^Lb=cm}', "");
Expect(1, 917999, '\p{Lb:   -CM}', "");
Expect(0, 917999, '\p{^Lb:   -CM}', "");
Expect(0, 917999, '\P{Lb:   -CM}', "");
Expect(1, 917999, '\P{^Lb:   -CM}', "");
Expect(0, 918000, '\p{Lb:   -CM}', "");
Expect(1, 918000, '\p{^Lb:   -CM}', "");
Expect(1, 918000, '\P{Lb:   -CM}', "");
Expect(0, 918000, '\P{^Lb:   -CM}', "");
Error('\p{Is_Line_Break= /a/Combining_mark}');
Error('\P{Is_Line_Break= /a/Combining_mark}');
Expect(1, 917999, '\p{Is_Line_Break=combiningmark}', "");
Expect(0, 917999, '\p{^Is_Line_Break=combiningmark}', "");
Expect(0, 917999, '\P{Is_Line_Break=combiningmark}', "");
Expect(1, 917999, '\P{^Is_Line_Break=combiningmark}', "");
Expect(0, 918000, '\p{Is_Line_Break=combiningmark}', "");
Expect(1, 918000, '\p{^Is_Line_Break=combiningmark}', "");
Expect(1, 918000, '\P{Is_Line_Break=combiningmark}', "");
Expect(0, 918000, '\P{^Is_Line_Break=combiningmark}', "");
Expect(1, 917999, '\p{Is_Line_Break= combining_Mark}', "");
Expect(0, 917999, '\p{^Is_Line_Break= combining_Mark}', "");
Expect(0, 917999, '\P{Is_Line_Break= combining_Mark}', "");
Expect(1, 917999, '\P{^Is_Line_Break= combining_Mark}', "");
Expect(0, 918000, '\p{Is_Line_Break= combining_Mark}', "");
Expect(1, 918000, '\p{^Is_Line_Break= combining_Mark}', "");
Expect(1, 918000, '\P{Is_Line_Break= combining_Mark}', "");
Expect(0, 918000, '\P{^Is_Line_Break= combining_Mark}', "");
Error('\p{Is_Lb=/a/ CM}');
Error('\P{Is_Lb=/a/ CM}');
Expect(1, 917999, '\p{Is_Lb=cm}', "");
Expect(0, 917999, '\p{^Is_Lb=cm}', "");
Expect(0, 917999, '\P{Is_Lb=cm}', "");
Expect(1, 917999, '\P{^Is_Lb=cm}', "");
Expect(0, 918000, '\p{Is_Lb=cm}', "");
Expect(1, 918000, '\p{^Is_Lb=cm}', "");
Expect(1, 918000, '\P{Is_Lb=cm}', "");
Expect(0, 918000, '\P{^Is_Lb=cm}', "");
Error('\p{Line_Break=/a/	-Mandatory_Break}');
Error('\P{Line_Break=/a/	-Mandatory_Break}');
Expect(1, 8233, '\p{Line_Break=mandatorybreak}', "");
Expect(0, 8233, '\p{^Line_Break=mandatorybreak}', "");
Expect(0, 8233, '\P{Line_Break=mandatorybreak}', "");
Expect(1, 8233, '\P{^Line_Break=mandatorybreak}', "");
Expect(0, 8234, '\p{Line_Break=mandatorybreak}', "");
Expect(1, 8234, '\p{^Line_Break=mandatorybreak}', "");
Expect(1, 8234, '\P{Line_Break=mandatorybreak}', "");
Expect(0, 8234, '\P{^Line_Break=mandatorybreak}', "");
Expect(1, 8233, '\p{Line_Break=		Mandatory_break}', "");
Expect(0, 8233, '\p{^Line_Break=		Mandatory_break}', "");
Expect(0, 8233, '\P{Line_Break=		Mandatory_break}', "");
Expect(1, 8233, '\P{^Line_Break=		Mandatory_break}', "");
Expect(0, 8234, '\p{Line_Break=		Mandatory_break}', "");
Expect(1, 8234, '\p{^Line_Break=		Mandatory_break}', "");
Expect(1, 8234, '\P{Line_Break=		Mandatory_break}', "");
Expect(0, 8234, '\P{^Line_Break=		Mandatory_break}', "");
Error('\p{Lb=:=_BK}');
Error('\P{Lb=:=_BK}');
Expect(1, 8233, '\p{Lb=bk}', "");
Expect(0, 8233, '\p{^Lb=bk}', "");
Expect(0, 8233, '\P{Lb=bk}', "");
Expect(1, 8233, '\P{^Lb=bk}', "");
Expect(0, 8234, '\p{Lb=bk}', "");
Expect(1, 8234, '\p{^Lb=bk}', "");
Expect(1, 8234, '\P{Lb=bk}', "");
Expect(0, 8234, '\P{^Lb=bk}', "");
Expect(1, 8233, '\p{Lb=- BK}', "");
Expect(0, 8233, '\p{^Lb=- BK}', "");
Expect(0, 8233, '\P{Lb=- BK}', "");
Expect(1, 8233, '\P{^Lb=- BK}', "");
Expect(0, 8234, '\p{Lb=- BK}', "");
Expect(1, 8234, '\p{^Lb=- BK}', "");
Expect(1, 8234, '\P{Lb=- BK}', "");
Expect(0, 8234, '\P{^Lb=- BK}', "");
Error('\p{Is_Line_Break=-:=Mandatory_Break}');
Error('\P{Is_Line_Break=-:=Mandatory_Break}');
Expect(1, 8233, '\p{Is_Line_Break=mandatorybreak}', "");
Expect(0, 8233, '\p{^Is_Line_Break=mandatorybreak}', "");
Expect(0, 8233, '\P{Is_Line_Break=mandatorybreak}', "");
Expect(1, 8233, '\P{^Is_Line_Break=mandatorybreak}', "");
Expect(0, 8234, '\p{Is_Line_Break=mandatorybreak}', "");
Expect(1, 8234, '\p{^Is_Line_Break=mandatorybreak}', "");
Expect(1, 8234, '\P{Is_Line_Break=mandatorybreak}', "");
Expect(0, 8234, '\P{^Is_Line_Break=mandatorybreak}', "");
Expect(1, 8233, '\p{Is_Line_Break=_MANDATORY_BREAK}', "");
Expect(0, 8233, '\p{^Is_Line_Break=_MANDATORY_BREAK}', "");
Expect(0, 8233, '\P{Is_Line_Break=_MANDATORY_BREAK}', "");
Expect(1, 8233, '\P{^Is_Line_Break=_MANDATORY_BREAK}', "");
Expect(0, 8234, '\p{Is_Line_Break=_MANDATORY_BREAK}', "");
Expect(1, 8234, '\p{^Is_Line_Break=_MANDATORY_BREAK}', "");
Expect(1, 8234, '\P{Is_Line_Break=_MANDATORY_BREAK}', "");
Expect(0, 8234, '\P{^Is_Line_Break=_MANDATORY_BREAK}', "");
Error('\p{Is_Lb=	 BK/a/}');
Error('\P{Is_Lb=	 BK/a/}');
Expect(1, 8233, '\p{Is_Lb: bk}', "");
Expect(0, 8233, '\p{^Is_Lb: bk}', "");
Expect(0, 8233, '\P{Is_Lb: bk}', "");
Expect(1, 8233, '\P{^Is_Lb: bk}', "");
Expect(0, 8234, '\p{Is_Lb: bk}', "");
Expect(1, 8234, '\p{^Is_Lb: bk}', "");
Expect(1, 8234, '\P{Is_Lb: bk}', "");
Expect(0, 8234, '\P{^Is_Lb: bk}', "");
Expect(1, 8233, '\p{Is_Lb=		BK}', "");
Expect(0, 8233, '\p{^Is_Lb=		BK}', "");
Expect(0, 8233, '\P{Is_Lb=		BK}', "");
Expect(1, 8233, '\P{^Is_Lb=		BK}', "");
Expect(0, 8234, '\p{Is_Lb=		BK}', "");
Expect(1, 8234, '\p{^Is_Lb=		BK}', "");
Expect(1, 8234, '\P{Is_Lb=		BK}', "");
Expect(0, 8234, '\P{^Is_Lb=		BK}', "");
Error('\p{Line_Break=-nonstarter/a/}');
Error('\P{Line_Break=-nonstarter/a/}');
Expect(1, 65439, '\p{Line_Break=nonstarter}', "");
Expect(0, 65439, '\p{^Line_Break=nonstarter}', "");
Expect(0, 65439, '\P{Line_Break=nonstarter}', "");
Expect(1, 65439, '\P{^Line_Break=nonstarter}', "");
Expect(0, 65440, '\p{Line_Break=nonstarter}', "");
Expect(1, 65440, '\p{^Line_Break=nonstarter}', "");
Expect(1, 65440, '\P{Line_Break=nonstarter}', "");
Expect(0, 65440, '\P{^Line_Break=nonstarter}', "");
Expect(1, 65439, '\p{Line_Break=		nonstarter}', "");
Expect(0, 65439, '\p{^Line_Break=		nonstarter}', "");
Expect(0, 65439, '\P{Line_Break=		nonstarter}', "");
Expect(1, 65439, '\P{^Line_Break=		nonstarter}', "");
Expect(0, 65440, '\p{Line_Break=		nonstarter}', "");
Expect(1, 65440, '\p{^Line_Break=		nonstarter}', "");
Expect(1, 65440, '\P{Line_Break=		nonstarter}', "");
Expect(0, 65440, '\P{^Line_Break=		nonstarter}', "");
Error('\p{Lb=	NS/a/}');
Error('\P{Lb=	NS/a/}');
Expect(1, 65439, '\p{Lb=ns}', "");
Expect(0, 65439, '\p{^Lb=ns}', "");
Expect(0, 65439, '\P{Lb=ns}', "");
Expect(1, 65439, '\P{^Lb=ns}', "");
Expect(0, 65440, '\p{Lb=ns}', "");
Expect(1, 65440, '\p{^Lb=ns}', "");
Expect(1, 65440, '\P{Lb=ns}', "");
Expect(0, 65440, '\P{^Lb=ns}', "");
Expect(1, 65439, '\p{Lb=--NS}', "");
Expect(0, 65439, '\p{^Lb=--NS}', "");
Expect(0, 65439, '\P{Lb=--NS}', "");
Expect(1, 65439, '\P{^Lb=--NS}', "");
Expect(0, 65440, '\p{Lb=--NS}', "");
Expect(1, 65440, '\p{^Lb=--NS}', "");
Expect(1, 65440, '\P{Lb=--NS}', "");
Expect(0, 65440, '\P{^Lb=--NS}', "");
Error('\p{Is_Line_Break:	_:=NONSTARTER}');
Error('\P{Is_Line_Break:	_:=NONSTARTER}');
Expect(1, 65439, '\p{Is_Line_Break=nonstarter}', "");
Expect(0, 65439, '\p{^Is_Line_Break=nonstarter}', "");
Expect(0, 65439, '\P{Is_Line_Break=nonstarter}', "");
Expect(1, 65439, '\P{^Is_Line_Break=nonstarter}', "");
Expect(0, 65440, '\p{Is_Line_Break=nonstarter}', "");
Expect(1, 65440, '\p{^Is_Line_Break=nonstarter}', "");
Expect(1, 65440, '\P{Is_Line_Break=nonstarter}', "");
Expect(0, 65440, '\P{^Is_Line_Break=nonstarter}', "");
Expect(1, 65439, '\p{Is_Line_Break=Nonstarter}', "");
Expect(0, 65439, '\p{^Is_Line_Break=Nonstarter}', "");
Expect(0, 65439, '\P{Is_Line_Break=Nonstarter}', "");
Expect(1, 65439, '\P{^Is_Line_Break=Nonstarter}', "");
Expect(0, 65440, '\p{Is_Line_Break=Nonstarter}', "");
Expect(1, 65440, '\p{^Is_Line_Break=Nonstarter}', "");
Expect(1, 65440, '\P{Is_Line_Break=Nonstarter}', "");
Expect(0, 65440, '\P{^Is_Line_Break=Nonstarter}', "");
Error('\p{Is_Lb=:= ns}');
Error('\P{Is_Lb=:= ns}');
Expect(1, 65439, '\p{Is_Lb:ns}', "");
Expect(0, 65439, '\p{^Is_Lb:ns}', "");
Expect(0, 65439, '\P{Is_Lb:ns}', "");
Expect(1, 65439, '\P{^Is_Lb:ns}', "");
Expect(0, 65440, '\p{Is_Lb:ns}', "");
Expect(1, 65440, '\p{^Is_Lb:ns}', "");
Expect(1, 65440, '\P{Is_Lb:ns}', "");
Expect(0, 65440, '\P{^Is_Lb:ns}', "");
Expect(1, 65439, '\p{Is_Lb=__NS}', "");
Expect(0, 65439, '\p{^Is_Lb=__NS}', "");
Expect(0, 65439, '\P{Is_Lb=__NS}', "");
Expect(1, 65439, '\P{^Is_Lb=__NS}', "");
Expect(0, 65440, '\p{Is_Lb=__NS}', "");
Expect(1, 65440, '\p{^Is_Lb=__NS}', "");
Expect(1, 65440, '\P{Is_Lb=__NS}', "");
Expect(0, 65440, '\P{^Is_Lb=__NS}', "");
Error('\p{Line_Break=:=JV}');
Error('\P{Line_Break=:=JV}');
Expect(1, 55238, '\p{Line_Break=jv}', "");
Expect(0, 55238, '\p{^Line_Break=jv}', "");
Expect(0, 55238, '\P{Line_Break=jv}', "");
Expect(1, 55238, '\P{^Line_Break=jv}', "");
Expect(0, 55239, '\p{Line_Break=jv}', "");
Expect(1, 55239, '\p{^Line_Break=jv}', "");
Expect(1, 55239, '\P{Line_Break=jv}', "");
Expect(0, 55239, '\P{^Line_Break=jv}', "");
Expect(1, 55238, '\p{Line_Break= JV}', "");
Expect(0, 55238, '\p{^Line_Break= JV}', "");
Expect(0, 55238, '\P{Line_Break= JV}', "");
Expect(1, 55238, '\P{^Line_Break= JV}', "");
Expect(0, 55239, '\p{Line_Break= JV}', "");
Expect(1, 55239, '\p{^Line_Break= JV}', "");
Expect(1, 55239, '\P{Line_Break= JV}', "");
Expect(0, 55239, '\P{^Line_Break= JV}', "");
Error('\p{Lb=:=_jv}');
Error('\P{Lb=:=_jv}');
Expect(1, 55238, '\p{Lb=jv}', "");
Expect(0, 55238, '\p{^Lb=jv}', "");
Expect(0, 55238, '\P{Lb=jv}', "");
Expect(1, 55238, '\P{^Lb=jv}', "");
Expect(0, 55239, '\p{Lb=jv}', "");
Expect(1, 55239, '\p{^Lb=jv}', "");
Expect(1, 55239, '\P{Lb=jv}', "");
Expect(0, 55239, '\P{^Lb=jv}', "");
Expect(1, 55238, '\p{Lb=		JV}', "");
Expect(0, 55238, '\p{^Lb=		JV}', "");
Expect(0, 55238, '\P{Lb=		JV}', "");
Expect(1, 55238, '\P{^Lb=		JV}', "");
Expect(0, 55239, '\p{Lb=		JV}', "");
Expect(1, 55239, '\p{^Lb=		JV}', "");
Expect(1, 55239, '\P{Lb=		JV}', "");
Expect(0, 55239, '\P{^Lb=		JV}', "");
Error('\p{Is_Line_Break=-:=JV}');
Error('\P{Is_Line_Break=-:=JV}');
Expect(1, 55238, '\p{Is_Line_Break=jv}', "");
Expect(0, 55238, '\p{^Is_Line_Break=jv}', "");
Expect(0, 55238, '\P{Is_Line_Break=jv}', "");
Expect(1, 55238, '\P{^Is_Line_Break=jv}', "");
Expect(0, 55239, '\p{Is_Line_Break=jv}', "");
Expect(1, 55239, '\p{^Is_Line_Break=jv}', "");
Expect(1, 55239, '\P{Is_Line_Break=jv}', "");
Expect(0, 55239, '\P{^Is_Line_Break=jv}', "");
Expect(1, 55238, '\p{Is_Line_Break=_JV}', "");
Expect(0, 55238, '\p{^Is_Line_Break=_JV}', "");
Expect(0, 55238, '\P{Is_Line_Break=_JV}', "");
Expect(1, 55238, '\P{^Is_Line_Break=_JV}', "");
Expect(0, 55239, '\p{Is_Line_Break=_JV}', "");
Expect(1, 55239, '\p{^Is_Line_Break=_JV}', "");
Expect(1, 55239, '\P{Is_Line_Break=_JV}', "");
Expect(0, 55239, '\P{^Is_Line_Break=_JV}', "");
Error('\p{Is_Lb=_-JV/a/}');
Error('\P{Is_Lb=_-JV/a/}');
Expect(1, 55238, '\p{Is_Lb=jv}', "");
Expect(0, 55238, '\p{^Is_Lb=jv}', "");
Expect(0, 55238, '\P{Is_Lb=jv}', "");
Expect(1, 55238, '\P{^Is_Lb=jv}', "");
Expect(0, 55239, '\p{Is_Lb=jv}', "");
Expect(1, 55239, '\p{^Is_Lb=jv}', "");
Expect(1, 55239, '\P{Is_Lb=jv}', "");
Expect(0, 55239, '\P{^Is_Lb=jv}', "");
Error('\p{Line_Break=:=_quotation}');
Error('\P{Line_Break=:=_quotation}');
Expect(1, 11809, '\p{Line_Break=quotation}', "");
Expect(0, 11809, '\p{^Line_Break=quotation}', "");
Expect(0, 11809, '\P{Line_Break=quotation}', "");
Expect(1, 11809, '\P{^Line_Break=quotation}', "");
Expect(0, 11810, '\p{Line_Break=quotation}', "");
Expect(1, 11810, '\p{^Line_Break=quotation}', "");
Expect(1, 11810, '\P{Line_Break=quotation}', "");
Expect(0, 11810, '\P{^Line_Break=quotation}', "");
Expect(1, 11809, '\p{Line_Break: --Quotation}', "");
Expect(0, 11809, '\p{^Line_Break: --Quotation}', "");
Expect(0, 11809, '\P{Line_Break: --Quotation}', "");
Expect(1, 11809, '\P{^Line_Break: --Quotation}', "");
Expect(0, 11810, '\p{Line_Break: --Quotation}', "");
Expect(1, 11810, '\p{^Line_Break: --Quotation}', "");
Expect(1, 11810, '\P{Line_Break: --Quotation}', "");
Expect(0, 11810, '\P{^Line_Break: --Quotation}', "");
Error('\p{Lb=--QU:=}');
Error('\P{Lb=--QU:=}');
Expect(1, 11809, '\p{Lb:qu}', "");
Expect(0, 11809, '\p{^Lb:qu}', "");
Expect(0, 11809, '\P{Lb:qu}', "");
Expect(1, 11809, '\P{^Lb:qu}', "");
Expect(0, 11810, '\p{Lb:qu}', "");
Expect(1, 11810, '\p{^Lb:qu}', "");
Expect(1, 11810, '\P{Lb:qu}', "");
Expect(0, 11810, '\P{^Lb:qu}', "");
Expect(1, 11809, '\p{Lb:	_QU}', "");
Expect(0, 11809, '\p{^Lb:	_QU}', "");
Expect(0, 11809, '\P{Lb:	_QU}', "");
Expect(1, 11809, '\P{^Lb:	_QU}', "");
Expect(0, 11810, '\p{Lb:	_QU}', "");
Expect(1, 11810, '\p{^Lb:	_QU}', "");
Expect(1, 11810, '\P{Lb:	_QU}', "");
Expect(0, 11810, '\P{^Lb:	_QU}', "");
Error('\p{Is_Line_Break=-Quotation:=}');
Error('\P{Is_Line_Break=-Quotation:=}');
Expect(1, 11809, '\p{Is_Line_Break=quotation}', "");
Expect(0, 11809, '\p{^Is_Line_Break=quotation}', "");
Expect(0, 11809, '\P{Is_Line_Break=quotation}', "");
Expect(1, 11809, '\P{^Is_Line_Break=quotation}', "");
Expect(0, 11810, '\p{Is_Line_Break=quotation}', "");
Expect(1, 11810, '\p{^Is_Line_Break=quotation}', "");
Expect(1, 11810, '\P{Is_Line_Break=quotation}', "");
Expect(0, 11810, '\P{^Is_Line_Break=quotation}', "");
Expect(1, 11809, '\p{Is_Line_Break=_ QUOTATION}', "");
Expect(0, 11809, '\p{^Is_Line_Break=_ QUOTATION}', "");
Expect(0, 11809, '\P{Is_Line_Break=_ QUOTATION}', "");
Expect(1, 11809, '\P{^Is_Line_Break=_ QUOTATION}', "");
Expect(0, 11810, '\p{Is_Line_Break=_ QUOTATION}', "");
Expect(1, 11810, '\p{^Is_Line_Break=_ QUOTATION}', "");
Expect(1, 11810, '\P{Is_Line_Break=_ QUOTATION}', "");
Expect(0, 11810, '\P{^Is_Line_Break=_ QUOTATION}', "");
Error('\p{Is_Lb=_/a/qu}');
Error('\P{Is_Lb=_/a/qu}');
Expect(1, 11809, '\p{Is_Lb=qu}', "");
Expect(0, 11809, '\p{^Is_Lb=qu}', "");
Expect(0, 11809, '\P{Is_Lb=qu}', "");
Expect(1, 11809, '\P{^Is_Lb=qu}', "");
Expect(0, 11810, '\p{Is_Lb=qu}', "");
Expect(1, 11810, '\p{^Is_Lb=qu}', "");
Expect(1, 11810, '\P{Is_Lb=qu}', "");
Expect(0, 11810, '\P{^Is_Lb=qu}', "");
Expect(1, 11809, '\p{Is_Lb=_-QU}', "");
Expect(0, 11809, '\p{^Is_Lb=_-QU}', "");
Expect(0, 11809, '\P{Is_Lb=_-QU}', "");
Expect(1, 11809, '\P{^Is_Lb=_-QU}', "");
Expect(0, 11810, '\p{Is_Lb=_-QU}', "");
Expect(1, 11810, '\p{^Is_Lb=_-QU}', "");
Expect(1, 11810, '\P{Is_Lb=_-QU}', "");
Expect(0, 11810, '\P{^Is_Lb=_-QU}', "");
Error('\p{Line_Break=:=-infix_Numeric}');
Error('\P{Line_Break=:=-infix_Numeric}');
Expect(1, 65044, '\p{Line_Break=infixnumeric}', "");
Expect(0, 65044, '\p{^Line_Break=infixnumeric}', "");
Expect(0, 65044, '\P{Line_Break=infixnumeric}', "");
Expect(1, 65044, '\P{^Line_Break=infixnumeric}', "");
Expect(0, 65045, '\p{Line_Break=infixnumeric}', "");
Expect(1, 65045, '\p{^Line_Break=infixnumeric}', "");
Expect(1, 65045, '\P{Line_Break=infixnumeric}', "");
Expect(0, 65045, '\P{^Line_Break=infixnumeric}', "");
Expect(1, 65044, '\p{Line_Break=	Infix_Numeric}', "");
Expect(0, 65044, '\p{^Line_Break=	Infix_Numeric}', "");
Expect(0, 65044, '\P{Line_Break=	Infix_Numeric}', "");
Expect(1, 65044, '\P{^Line_Break=	Infix_Numeric}', "");
Expect(0, 65045, '\p{Line_Break=	Infix_Numeric}', "");
Expect(1, 65045, '\p{^Line_Break=	Infix_Numeric}', "");
Expect(1, 65045, '\P{Line_Break=	Infix_Numeric}', "");
Expect(0, 65045, '\P{^Line_Break=	Infix_Numeric}', "");
Error('\p{Lb: := 	IS}');
Error('\P{Lb: := 	IS}');
Expect(1, 65044, '\p{Lb=is}', "");
Expect(0, 65044, '\p{^Lb=is}', "");
Expect(0, 65044, '\P{Lb=is}', "");
Expect(1, 65044, '\P{^Lb=is}', "");
Expect(0, 65045, '\p{Lb=is}', "");
Expect(1, 65045, '\p{^Lb=is}', "");
Expect(1, 65045, '\P{Lb=is}', "");
Expect(0, 65045, '\P{^Lb=is}', "");
Expect(1, 65044, '\p{Lb=_IS}', "");
Expect(0, 65044, '\p{^Lb=_IS}', "");
Expect(0, 65044, '\P{Lb=_IS}', "");
Expect(1, 65044, '\P{^Lb=_IS}', "");
Expect(0, 65045, '\p{Lb=_IS}', "");
Expect(1, 65045, '\p{^Lb=_IS}', "");
Expect(1, 65045, '\P{Lb=_IS}', "");
Expect(0, 65045, '\P{^Lb=_IS}', "");
Error('\p{Is_Line_Break=- infix_Numeric:=}');
Error('\P{Is_Line_Break=- infix_Numeric:=}');
Expect(1, 65044, '\p{Is_Line_Break=infixnumeric}', "");
Expect(0, 65044, '\p{^Is_Line_Break=infixnumeric}', "");
Expect(0, 65044, '\P{Is_Line_Break=infixnumeric}', "");
Expect(1, 65044, '\P{^Is_Line_Break=infixnumeric}', "");
Expect(0, 65045, '\p{Is_Line_Break=infixnumeric}', "");
Expect(1, 65045, '\p{^Is_Line_Break=infixnumeric}', "");
Expect(1, 65045, '\P{Is_Line_Break=infixnumeric}', "");
Expect(0, 65045, '\P{^Is_Line_Break=infixnumeric}', "");
Expect(1, 65044, '\p{Is_Line_Break=-_infix_NUMERIC}', "");
Expect(0, 65044, '\p{^Is_Line_Break=-_infix_NUMERIC}', "");
Expect(0, 65044, '\P{Is_Line_Break=-_infix_NUMERIC}', "");
Expect(1, 65044, '\P{^Is_Line_Break=-_infix_NUMERIC}', "");
Expect(0, 65045, '\p{Is_Line_Break=-_infix_NUMERIC}', "");
Expect(1, 65045, '\p{^Is_Line_Break=-_infix_NUMERIC}', "");
Expect(1, 65045, '\P{Is_Line_Break=-_infix_NUMERIC}', "");
Expect(0, 65045, '\P{^Is_Line_Break=-_infix_NUMERIC}', "");
Error('\p{Is_Lb=:=__is}');
Error('\P{Is_Lb=:=__is}');
Expect(1, 65044, '\p{Is_Lb=is}', "");
Expect(0, 65044, '\p{^Is_Lb=is}', "");
Expect(0, 65044, '\P{Is_Lb=is}', "");
Expect(1, 65044, '\P{^Is_Lb=is}', "");
Expect(0, 65045, '\p{Is_Lb=is}', "");
Expect(1, 65045, '\p{^Is_Lb=is}', "");
Expect(1, 65045, '\P{Is_Lb=is}', "");
Expect(0, 65045, '\P{^Is_Lb=is}', "");
Expect(1, 65044, '\p{Is_Lb=  is}', "");
Expect(0, 65044, '\p{^Is_Lb=  is}', "");
Expect(0, 65044, '\P{Is_Lb=  is}', "");
Expect(1, 65044, '\P{^Is_Lb=  is}', "");
Expect(0, 65045, '\p{Is_Lb=  is}', "");
Expect(1, 65045, '\p{^Is_Lb=  is}', "");
Expect(1, 65045, '\P{Is_Lb=  is}', "");
Expect(0, 65045, '\P{^Is_Lb=  is}', "");
Error('\p{Line_Break=	 inseparable:=}');
Error('\P{Line_Break=	 inseparable:=}');
Expect(1, 65049, '\p{Line_Break=inseparable}', "");
Expect(0, 65049, '\p{^Line_Break=inseparable}', "");
Expect(0, 65049, '\P{Line_Break=inseparable}', "");
Expect(1, 65049, '\P{^Line_Break=inseparable}', "");
Expect(0, 65050, '\p{Line_Break=inseparable}', "");
Expect(1, 65050, '\p{^Line_Break=inseparable}', "");
Expect(1, 65050, '\P{Line_Break=inseparable}', "");
Expect(0, 65050, '\P{^Line_Break=inseparable}', "");
Expect(1, 65049, '\p{Line_Break:  INSEPARABLE}', "");
Expect(0, 65049, '\p{^Line_Break:  INSEPARABLE}', "");
Expect(0, 65049, '\P{Line_Break:  INSEPARABLE}', "");
Expect(1, 65049, '\P{^Line_Break:  INSEPARABLE}', "");
Expect(0, 65050, '\p{Line_Break:  INSEPARABLE}', "");
Expect(1, 65050, '\p{^Line_Break:  INSEPARABLE}', "");
Expect(1, 65050, '\P{Line_Break:  INSEPARABLE}', "");
Expect(0, 65050, '\P{^Line_Break:  INSEPARABLE}', "");
Error('\p{Lb=	_in/a/}');
Error('\P{Lb=	_in/a/}');
Expect(1, 65049, '\p{Lb=in}', "");
Expect(0, 65049, '\p{^Lb=in}', "");
Expect(0, 65049, '\P{Lb=in}', "");
Expect(1, 65049, '\P{^Lb=in}', "");
Expect(0, 65050, '\p{Lb=in}', "");
Expect(1, 65050, '\p{^Lb=in}', "");
Expect(1, 65050, '\P{Lb=in}', "");
Expect(0, 65050, '\P{^Lb=in}', "");
Expect(1, 65049, '\p{Lb=_-IN}', "");
Expect(0, 65049, '\p{^Lb=_-IN}', "");
Expect(0, 65049, '\P{Lb=_-IN}', "");
Expect(1, 65049, '\P{^Lb=_-IN}', "");
Expect(0, 65050, '\p{Lb=_-IN}', "");
Expect(1, 65050, '\p{^Lb=_-IN}', "");
Expect(1, 65050, '\P{Lb=_-IN}', "");
Expect(0, 65050, '\P{^Lb=_-IN}', "");
Error('\p{Is_Line_Break: _INSEPERABLE/a/}');
Error('\P{Is_Line_Break: _INSEPERABLE/a/}');
Expect(1, 65049, '\p{Is_Line_Break=inseperable}', "");
Expect(0, 65049, '\p{^Is_Line_Break=inseperable}', "");
Expect(0, 65049, '\P{Is_Line_Break=inseperable}', "");
Expect(1, 65049, '\P{^Is_Line_Break=inseperable}', "");
Expect(0, 65050, '\p{Is_Line_Break=inseperable}', "");
Expect(1, 65050, '\p{^Is_Line_Break=inseperable}', "");
Expect(1, 65050, '\P{Is_Line_Break=inseperable}', "");
Expect(0, 65050, '\P{^Is_Line_Break=inseperable}', "");
Expect(1, 65049, '\p{Is_Line_Break=	Inseperable}', "");
Expect(0, 65049, '\p{^Is_Line_Break=	Inseperable}', "");
Expect(0, 65049, '\P{Is_Line_Break=	Inseperable}', "");
Expect(1, 65049, '\P{^Is_Line_Break=	Inseperable}', "");
Expect(0, 65050, '\p{Is_Line_Break=	Inseperable}', "");
Expect(1, 65050, '\p{^Is_Line_Break=	Inseperable}', "");
Expect(1, 65050, '\P{Is_Line_Break=	Inseperable}', "");
Expect(0, 65050, '\P{^Is_Line_Break=	Inseperable}', "");
Error('\p{Is_Lb:   _INSEPARABLE:=}');
Error('\P{Is_Lb:   _INSEPARABLE:=}');
Expect(1, 65049, '\p{Is_Lb=inseparable}', "");
Expect(0, 65049, '\p{^Is_Lb=inseparable}', "");
Expect(0, 65049, '\P{Is_Lb=inseparable}', "");
Expect(1, 65049, '\P{^Is_Lb=inseparable}', "");
Expect(0, 65050, '\p{Is_Lb=inseparable}', "");
Expect(1, 65050, '\p{^Is_Lb=inseparable}', "");
Expect(1, 65050, '\P{Is_Lb=inseparable}', "");
Expect(0, 65050, '\P{^Is_Lb=inseparable}', "");
Expect(1, 65049, '\p{Is_Lb:-	Inseparable}', "");
Expect(0, 65049, '\p{^Is_Lb:-	Inseparable}', "");
Expect(0, 65049, '\P{Is_Lb:-	Inseparable}', "");
Expect(1, 65049, '\P{^Is_Lb:-	Inseparable}', "");
Expect(0, 65050, '\p{Is_Lb:-	Inseparable}', "");
Expect(1, 65050, '\p{^Is_Lb:-	Inseparable}', "");
Expect(1, 65050, '\P{Is_Lb:-	Inseparable}', "");
Expect(0, 65050, '\P{^Is_Lb:-	Inseparable}', "");
Error('\p{Line_Break=/a/Complex_Context}');
Error('\P{Line_Break=/a/Complex_Context}');
Expect(1, 43743, '\p{Line_Break=complexcontext}', "");
Expect(0, 43743, '\p{^Line_Break=complexcontext}', "");
Expect(0, 43743, '\P{Line_Break=complexcontext}', "");
Expect(1, 43743, '\P{^Line_Break=complexcontext}', "");
Expect(0, 43744, '\p{Line_Break=complexcontext}', "");
Expect(1, 43744, '\p{^Line_Break=complexcontext}', "");
Expect(1, 43744, '\P{Line_Break=complexcontext}', "");
Expect(0, 43744, '\P{^Line_Break=complexcontext}', "");
Expect(1, 43743, '\p{Line_Break=_Complex_Context}', "");
Expect(0, 43743, '\p{^Line_Break=_Complex_Context}', "");
Expect(0, 43743, '\P{Line_Break=_Complex_Context}', "");
Expect(1, 43743, '\P{^Line_Break=_Complex_Context}', "");
Expect(0, 43744, '\p{Line_Break=_Complex_Context}', "");
Expect(1, 43744, '\p{^Line_Break=_Complex_Context}', "");
Expect(1, 43744, '\P{Line_Break=_Complex_Context}', "");
Expect(0, 43744, '\P{^Line_Break=_Complex_Context}', "");
Error('\p{Lb=-:=SA}');
Error('\P{Lb=-:=SA}');
Expect(1, 43743, '\p{Lb=sa}', "");
Expect(0, 43743, '\p{^Lb=sa}', "");
Expect(0, 43743, '\P{Lb=sa}', "");
Expect(1, 43743, '\P{^Lb=sa}', "");
Expect(0, 43744, '\p{Lb=sa}', "");
Expect(1, 43744, '\p{^Lb=sa}', "");
Expect(1, 43744, '\P{Lb=sa}', "");
Expect(0, 43744, '\P{^Lb=sa}', "");
Expect(1, 43743, '\p{Lb=-	sa}', "");
Expect(0, 43743, '\p{^Lb=-	sa}', "");
Expect(0, 43743, '\P{Lb=-	sa}', "");
Expect(1, 43743, '\P{^Lb=-	sa}', "");
Expect(0, 43744, '\p{Lb=-	sa}', "");
Expect(1, 43744, '\p{^Lb=-	sa}', "");
Expect(1, 43744, '\P{Lb=-	sa}', "");
Expect(0, 43744, '\P{^Lb=-	sa}', "");
Error('\p{Is_Line_Break=	:=Complex_Context}');
Error('\P{Is_Line_Break=	:=Complex_Context}');
Expect(1, 43743, '\p{Is_Line_Break=complexcontext}', "");
Expect(0, 43743, '\p{^Is_Line_Break=complexcontext}', "");
Expect(0, 43743, '\P{Is_Line_Break=complexcontext}', "");
Expect(1, 43743, '\P{^Is_Line_Break=complexcontext}', "");
Expect(0, 43744, '\p{Is_Line_Break=complexcontext}', "");
Expect(1, 43744, '\p{^Is_Line_Break=complexcontext}', "");
Expect(1, 43744, '\P{Is_Line_Break=complexcontext}', "");
Expect(0, 43744, '\P{^Is_Line_Break=complexcontext}', "");
Expect(1, 43743, '\p{Is_Line_Break=_ Complex_Context}', "");
Expect(0, 43743, '\p{^Is_Line_Break=_ Complex_Context}', "");
Expect(0, 43743, '\P{Is_Line_Break=_ Complex_Context}', "");
Expect(1, 43743, '\P{^Is_Line_Break=_ Complex_Context}', "");
Expect(0, 43744, '\p{Is_Line_Break=_ Complex_Context}', "");
Expect(1, 43744, '\p{^Is_Line_Break=_ Complex_Context}', "");
Expect(1, 43744, '\P{Is_Line_Break=_ Complex_Context}', "");
Expect(0, 43744, '\P{^Is_Line_Break=_ Complex_Context}', "");
Error('\p{Is_Lb=_:=SA}');
Error('\P{Is_Lb=_:=SA}');
Expect(1, 43743, '\p{Is_Lb=sa}', "");
Expect(0, 43743, '\p{^Is_Lb=sa}', "");
Expect(0, 43743, '\P{Is_Lb=sa}', "");
Expect(1, 43743, '\P{^Is_Lb=sa}', "");
Expect(0, 43744, '\p{Is_Lb=sa}', "");
Expect(1, 43744, '\p{^Is_Lb=sa}', "");
Expect(1, 43744, '\P{Is_Lb=sa}', "");
Expect(0, 43744, '\P{^Is_Lb=sa}', "");
Expect(1, 43743, '\p{Is_Lb=SA}', "");
Expect(0, 43743, '\p{^Is_Lb=SA}', "");
Expect(0, 43743, '\P{Is_Lb=SA}', "");
Expect(1, 43743, '\P{^Is_Lb=SA}', "");
Expect(0, 43744, '\p{Is_Lb=SA}', "");
Expect(1, 43744, '\p{^Is_Lb=SA}', "");
Expect(1, 43744, '\P{Is_Lb=SA}', "");
Expect(0, 43744, '\P{^Is_Lb=SA}', "");
Error('\p{Line_Break=:= -hyphen}');
Error('\P{Line_Break=:= -hyphen}');
Expect(1, 45, '\p{Line_Break=hyphen}', "");
Expect(0, 45, '\p{^Line_Break=hyphen}', "");
Expect(0, 45, '\P{Line_Break=hyphen}', "");
Expect(1, 45, '\P{^Line_Break=hyphen}', "");
Expect(0, 46, '\p{Line_Break=hyphen}', "");
Expect(1, 46, '\p{^Line_Break=hyphen}', "");
Expect(1, 46, '\P{Line_Break=hyphen}', "");
Expect(0, 46, '\P{^Line_Break=hyphen}', "");
Expect(1, 45, '\p{Line_Break=- hyphen}', "");
Expect(0, 45, '\p{^Line_Break=- hyphen}', "");
Expect(0, 45, '\P{Line_Break=- hyphen}', "");
Expect(1, 45, '\P{^Line_Break=- hyphen}', "");
Expect(0, 46, '\p{Line_Break=- hyphen}', "");
Expect(1, 46, '\p{^Line_Break=- hyphen}', "");
Expect(1, 46, '\P{Line_Break=- hyphen}', "");
Expect(0, 46, '\P{^Line_Break=- hyphen}', "");
Error('\p{Lb=__hy/a/}');
Error('\P{Lb=__hy/a/}');
Expect(1, 45, '\p{Lb=hy}', "");
Expect(0, 45, '\p{^Lb=hy}', "");
Expect(0, 45, '\P{Lb=hy}', "");
Expect(1, 45, '\P{^Lb=hy}', "");
Expect(0, 46, '\p{Lb=hy}', "");
Expect(1, 46, '\p{^Lb=hy}', "");
Expect(1, 46, '\P{Lb=hy}', "");
Expect(0, 46, '\P{^Lb=hy}', "");
Expect(1, 45, '\p{Lb=HY}', "");
Expect(0, 45, '\p{^Lb=HY}', "");
Expect(0, 45, '\P{Lb=HY}', "");
Expect(1, 45, '\P{^Lb=HY}', "");
Expect(0, 46, '\p{Lb=HY}', "");
Expect(1, 46, '\p{^Lb=HY}', "");
Expect(1, 46, '\P{Lb=HY}', "");
Expect(0, 46, '\P{^Lb=HY}', "");
Error('\p{Is_Line_Break=_ Hyphen:=}');
Error('\P{Is_Line_Break=_ Hyphen:=}');
Expect(1, 45, '\p{Is_Line_Break=hyphen}', "");
Expect(0, 45, '\p{^Is_Line_Break=hyphen}', "");
Expect(0, 45, '\P{Is_Line_Break=hyphen}', "");
Expect(1, 45, '\P{^Is_Line_Break=hyphen}', "");
Expect(0, 46, '\p{Is_Line_Break=hyphen}', "");
Expect(1, 46, '\p{^Is_Line_Break=hyphen}', "");
Expect(1, 46, '\P{Is_Line_Break=hyphen}', "");
Expect(0, 46, '\P{^Is_Line_Break=hyphen}', "");
Expect(1, 45, '\p{Is_Line_Break=- hyphen}', "");
Expect(0, 45, '\p{^Is_Line_Break=- hyphen}', "");
Expect(0, 45, '\P{Is_Line_Break=- hyphen}', "");
Expect(1, 45, '\P{^Is_Line_Break=- hyphen}', "");
Expect(0, 46, '\p{Is_Line_Break=- hyphen}', "");
Expect(1, 46, '\p{^Is_Line_Break=- hyphen}', "");
Expect(1, 46, '\P{Is_Line_Break=- hyphen}', "");
Expect(0, 46, '\P{^Is_Line_Break=- hyphen}', "");
Error('\p{Is_Lb=_HY:=}');
Error('\P{Is_Lb=_HY:=}');
Expect(1, 45, '\p{Is_Lb=hy}', "");
Expect(0, 45, '\p{^Is_Lb=hy}', "");
Expect(0, 45, '\P{Is_Lb=hy}', "");
Expect(1, 45, '\P{^Is_Lb=hy}', "");
Expect(0, 46, '\p{Is_Lb=hy}', "");
Expect(1, 46, '\p{^Is_Lb=hy}', "");
Expect(1, 46, '\P{Is_Lb=hy}', "");
Expect(0, 46, '\P{^Is_Lb=hy}', "");
Expect(1, 45, '\p{Is_Lb=_hy}', "");
Expect(0, 45, '\p{^Is_Lb=_hy}', "");
Expect(0, 45, '\P{Is_Lb=_hy}', "");
Expect(1, 45, '\P{^Is_Lb=_hy}', "");
Expect(0, 46, '\p{Is_Lb=_hy}', "");
Expect(1, 46, '\p{^Is_Lb=_hy}', "");
Expect(1, 46, '\P{Is_Lb=_hy}', "");
Expect(0, 46, '\P{^Is_Lb=_hy}', "");
Error('\p{Line_Break=-/a/Break_AFTER}');
Error('\P{Line_Break=-/a/Break_AFTER}');
Expect(1, 74867, '\p{Line_Break=breakafter}', "");
Expect(0, 74867, '\p{^Line_Break=breakafter}', "");
Expect(0, 74867, '\P{Line_Break=breakafter}', "");
Expect(1, 74867, '\P{^Line_Break=breakafter}', "");
Expect(0, 74868, '\p{Line_Break=breakafter}', "");
Expect(1, 74868, '\p{^Line_Break=breakafter}', "");
Expect(1, 74868, '\P{Line_Break=breakafter}', "");
Expect(0, 74868, '\P{^Line_Break=breakafter}', "");
Expect(1, 74867, '\p{Line_Break=_-BREAK_After}', "");
Expect(0, 74867, '\p{^Line_Break=_-BREAK_After}', "");
Expect(0, 74867, '\P{Line_Break=_-BREAK_After}', "");
Expect(1, 74867, '\P{^Line_Break=_-BREAK_After}', "");
Expect(0, 74868, '\p{Line_Break=_-BREAK_After}', "");
Expect(1, 74868, '\p{^Line_Break=_-BREAK_After}', "");
Expect(1, 74868, '\P{Line_Break=_-BREAK_After}', "");
Expect(0, 74868, '\P{^Line_Break=_-BREAK_After}', "");
Error('\p{Lb=--ba/a/}');
Error('\P{Lb=--ba/a/}');
Expect(1, 74867, '\p{Lb=ba}', "");
Expect(0, 74867, '\p{^Lb=ba}', "");
Expect(0, 74867, '\P{Lb=ba}', "");
Expect(1, 74867, '\P{^Lb=ba}', "");
Expect(0, 74868, '\p{Lb=ba}', "");
Expect(1, 74868, '\p{^Lb=ba}', "");
Expect(1, 74868, '\P{Lb=ba}', "");
Expect(0, 74868, '\P{^Lb=ba}', "");
Expect(1, 74867, '\p{Lb=	BA}', "");
Expect(0, 74867, '\p{^Lb=	BA}', "");
Expect(0, 74867, '\P{Lb=	BA}', "");
Expect(1, 74867, '\P{^Lb=	BA}', "");
Expect(0, 74868, '\p{Lb=	BA}', "");
Expect(1, 74868, '\p{^Lb=	BA}', "");
Expect(1, 74868, '\P{Lb=	BA}', "");
Expect(0, 74868, '\P{^Lb=	BA}', "");
Error('\p{Is_Line_Break=	-Break_after:=}');
Error('\P{Is_Line_Break=	-Break_after:=}');
Expect(1, 74867, '\p{Is_Line_Break:breakafter}', "");
Expect(0, 74867, '\p{^Is_Line_Break:breakafter}', "");
Expect(0, 74867, '\P{Is_Line_Break:breakafter}', "");
Expect(1, 74867, '\P{^Is_Line_Break:breakafter}', "");
Expect(0, 74868, '\p{Is_Line_Break:breakafter}', "");
Expect(1, 74868, '\p{^Is_Line_Break:breakafter}', "");
Expect(1, 74868, '\P{Is_Line_Break:breakafter}', "");
Expect(0, 74868, '\P{^Is_Line_Break:breakafter}', "");
Expect(1, 74867, '\p{Is_Line_Break=- Break_After}', "");
Expect(0, 74867, '\p{^Is_Line_Break=- Break_After}', "");
Expect(0, 74867, '\P{Is_Line_Break=- Break_After}', "");
Expect(1, 74867, '\P{^Is_Line_Break=- Break_After}', "");
Expect(0, 74868, '\p{Is_Line_Break=- Break_After}', "");
Expect(1, 74868, '\p{^Is_Line_Break=- Break_After}', "");
Expect(1, 74868, '\P{Is_Line_Break=- Break_After}', "");
Expect(0, 74868, '\P{^Is_Line_Break=- Break_After}', "");
Error('\p{Is_Lb=	/a/BA}');
Error('\P{Is_Lb=	/a/BA}');
Expect(1, 74867, '\p{Is_Lb=ba}', "");
Expect(0, 74867, '\p{^Is_Lb=ba}', "");
Expect(0, 74867, '\P{Is_Lb=ba}', "");
Expect(1, 74867, '\P{^Is_Lb=ba}', "");
Expect(0, 74868, '\p{Is_Lb=ba}', "");
Expect(1, 74868, '\p{^Is_Lb=ba}', "");
Expect(1, 74868, '\P{Is_Lb=ba}', "");
Expect(0, 74868, '\P{^Is_Lb=ba}', "");
Expect(1, 74867, '\p{Is_Lb=_-BA}', "");
Expect(0, 74867, '\p{^Is_Lb=_-BA}', "");
Expect(0, 74867, '\P{Is_Lb=_-BA}', "");
Expect(1, 74867, '\P{^Is_Lb=_-BA}', "");
Expect(0, 74868, '\p{Is_Lb=_-BA}', "");
Expect(1, 74868, '\p{^Is_Lb=_-BA}', "");
Expect(1, 74868, '\P{Is_Lb=_-BA}', "");
Expect(0, 74868, '\P{^Is_Lb=_-BA}', "");
Error('\p{Line_Break=/a/_-Line_Feed}');
Error('\P{Line_Break=/a/_-Line_Feed}');
Expect(1, 10, '\p{Line_Break=linefeed}', "");
Expect(0, 10, '\p{^Line_Break=linefeed}', "");
Expect(0, 10, '\P{Line_Break=linefeed}', "");
Expect(1, 10, '\P{^Line_Break=linefeed}', "");
Expect(0, 11, '\p{Line_Break=linefeed}', "");
Expect(1, 11, '\p{^Line_Break=linefeed}', "");
Expect(1, 11, '\P{Line_Break=linefeed}', "");
Expect(0, 11, '\P{^Line_Break=linefeed}', "");
Expect(1, 10, '\p{Line_Break=	-line_feed}', "");
Expect(0, 10, '\p{^Line_Break=	-line_feed}', "");
Expect(0, 10, '\P{Line_Break=	-line_feed}', "");
Expect(1, 10, '\P{^Line_Break=	-line_feed}', "");
Expect(0, 11, '\p{Line_Break=	-line_feed}', "");
Expect(1, 11, '\p{^Line_Break=	-line_feed}', "");
Expect(1, 11, '\P{Line_Break=	-line_feed}', "");
Expect(0, 11, '\P{^Line_Break=	-line_feed}', "");
Error('\p{Lb:-:=LF}');
Error('\P{Lb:-:=LF}');
Expect(1, 10, '\p{Lb=lf}', "");
Expect(0, 10, '\p{^Lb=lf}', "");
Expect(0, 10, '\P{Lb=lf}', "");
Expect(1, 10, '\P{^Lb=lf}', "");
Expect(0, 11, '\p{Lb=lf}', "");
Expect(1, 11, '\p{^Lb=lf}', "");
Expect(1, 11, '\P{Lb=lf}', "");
Expect(0, 11, '\P{^Lb=lf}', "");
Expect(1, 10, '\p{Lb=	_LF}', "");
Expect(0, 10, '\p{^Lb=	_LF}', "");
Expect(0, 10, '\P{Lb=	_LF}', "");
Expect(1, 10, '\P{^Lb=	_LF}', "");
Expect(0, 11, '\p{Lb=	_LF}', "");
Expect(1, 11, '\p{^Lb=	_LF}', "");
Expect(1, 11, '\P{Lb=	_LF}', "");
Expect(0, 11, '\P{^Lb=	_LF}', "");
Error('\p{Is_Line_Break=/a/__LINE_feed}');
Error('\P{Is_Line_Break=/a/__LINE_feed}');
Expect(1, 10, '\p{Is_Line_Break=linefeed}', "");
Expect(0, 10, '\p{^Is_Line_Break=linefeed}', "");
Expect(0, 10, '\P{Is_Line_Break=linefeed}', "");
Expect(1, 10, '\P{^Is_Line_Break=linefeed}', "");
Expect(0, 11, '\p{Is_Line_Break=linefeed}', "");
Expect(1, 11, '\p{^Is_Line_Break=linefeed}', "");
Expect(1, 11, '\P{Is_Line_Break=linefeed}', "");
Expect(0, 11, '\P{^Is_Line_Break=linefeed}', "");
Expect(1, 10, '\p{Is_Line_Break=	 line_feed}', "");
Expect(0, 10, '\p{^Is_Line_Break=	 line_feed}', "");
Expect(0, 10, '\P{Is_Line_Break=	 line_feed}', "");
Expect(1, 10, '\P{^Is_Line_Break=	 line_feed}', "");
Expect(0, 11, '\p{Is_Line_Break=	 line_feed}', "");
Expect(1, 11, '\p{^Is_Line_Break=	 line_feed}', "");
Expect(1, 11, '\P{Is_Line_Break=	 line_feed}', "");
Expect(0, 11, '\P{^Is_Line_Break=	 line_feed}', "");
Error('\p{Is_Lb=_:=LF}');
Error('\P{Is_Lb=_:=LF}');
Expect(1, 10, '\p{Is_Lb:   lf}', "");
Expect(0, 10, '\p{^Is_Lb:   lf}', "");
Expect(0, 10, '\P{Is_Lb:   lf}', "");
Expect(1, 10, '\P{^Is_Lb:   lf}', "");
Expect(0, 11, '\p{Is_Lb:   lf}', "");
Expect(1, 11, '\p{^Is_Lb:   lf}', "");
Expect(1, 11, '\P{Is_Lb:   lf}', "");
Expect(0, 11, '\P{^Is_Lb:   lf}', "");
Expect(1, 10, '\p{Is_Lb= _LF}', "");
Expect(0, 10, '\p{^Is_Lb= _LF}', "");
Expect(0, 10, '\P{Is_Lb= _LF}', "");
Expect(1, 10, '\P{^Is_Lb= _LF}', "");
Expect(0, 11, '\p{Is_Lb= _LF}', "");
Expect(1, 11, '\p{^Is_Lb= _LF}', "");
Expect(1, 11, '\P{Is_Lb= _LF}', "");
Expect(0, 11, '\P{^Is_Lb= _LF}', "");
Error('\p{Line_Break=:= Ambiguous}');
Error('\P{Line_Break=:= Ambiguous}');
Expect(1, 127386, '\p{Line_Break=ambiguous}', "");
Expect(0, 127386, '\p{^Line_Break=ambiguous}', "");
Expect(0, 127386, '\P{Line_Break=ambiguous}', "");
Expect(1, 127386, '\P{^Line_Break=ambiguous}', "");
Expect(0, 127387, '\p{Line_Break=ambiguous}', "");
Expect(1, 127387, '\p{^Line_Break=ambiguous}', "");
Expect(1, 127387, '\P{Line_Break=ambiguous}', "");
Expect(0, 127387, '\P{^Line_Break=ambiguous}', "");
Expect(1, 127386, '\p{Line_Break= _Ambiguous}', "");
Expect(0, 127386, '\p{^Line_Break= _Ambiguous}', "");
Expect(0, 127386, '\P{Line_Break= _Ambiguous}', "");
Expect(1, 127386, '\P{^Line_Break= _Ambiguous}', "");
Expect(0, 127387, '\p{Line_Break= _Ambiguous}', "");
Expect(1, 127387, '\p{^Line_Break= _Ambiguous}', "");
Expect(1, 127387, '\P{Line_Break= _Ambiguous}', "");
Expect(0, 127387, '\P{^Line_Break= _Ambiguous}', "");
Error('\p{Lb:	AI:=}');
Error('\P{Lb:	AI:=}');
Expect(1, 127386, '\p{Lb=ai}', "");
Expect(0, 127386, '\p{^Lb=ai}', "");
Expect(0, 127386, '\P{Lb=ai}', "");
Expect(1, 127386, '\P{^Lb=ai}', "");
Expect(0, 127387, '\p{Lb=ai}', "");
Expect(1, 127387, '\p{^Lb=ai}', "");
Expect(1, 127387, '\P{Lb=ai}', "");
Expect(0, 127387, '\P{^Lb=ai}', "");
Expect(1, 127386, '\p{Lb:	AI}', "");
Expect(0, 127386, '\p{^Lb:	AI}', "");
Expect(0, 127386, '\P{Lb:	AI}', "");
Expect(1, 127386, '\P{^Lb:	AI}', "");
Expect(0, 127387, '\p{Lb:	AI}', "");
Expect(1, 127387, '\p{^Lb:	AI}', "");
Expect(1, 127387, '\P{Lb:	AI}', "");
Expect(0, 127387, '\P{^Lb:	AI}', "");
Error('\p{Is_Line_Break=	-AMBIGUOUS:=}');
Error('\P{Is_Line_Break=	-AMBIGUOUS:=}');
Expect(1, 127386, '\p{Is_Line_Break=ambiguous}', "");
Expect(0, 127386, '\p{^Is_Line_Break=ambiguous}', "");
Expect(0, 127386, '\P{Is_Line_Break=ambiguous}', "");
Expect(1, 127386, '\P{^Is_Line_Break=ambiguous}', "");
Expect(0, 127387, '\p{Is_Line_Break=ambiguous}', "");
Expect(1, 127387, '\p{^Is_Line_Break=ambiguous}', "");
Expect(1, 127387, '\P{Is_Line_Break=ambiguous}', "");
Expect(0, 127387, '\P{^Is_Line_Break=ambiguous}', "");
Expect(1, 127386, '\p{Is_Line_Break=_Ambiguous}', "");
Expect(0, 127386, '\p{^Is_Line_Break=_Ambiguous}', "");
Expect(0, 127386, '\P{Is_Line_Break=_Ambiguous}', "");
Expect(1, 127386, '\P{^Is_Line_Break=_Ambiguous}', "");
Expect(0, 127387, '\p{Is_Line_Break=_Ambiguous}', "");
Expect(1, 127387, '\p{^Is_Line_Break=_Ambiguous}', "");
Expect(1, 127387, '\P{Is_Line_Break=_Ambiguous}', "");
Expect(0, 127387, '\P{^Is_Line_Break=_Ambiguous}', "");
Error('\p{Is_Lb=	:=AI}');
Error('\P{Is_Lb=	:=AI}');
Expect(1, 127386, '\p{Is_Lb=ai}', "");
Expect(0, 127386, '\p{^Is_Lb=ai}', "");
Expect(0, 127386, '\P{Is_Lb=ai}', "");
Expect(1, 127386, '\P{^Is_Lb=ai}', "");
Expect(0, 127387, '\p{Is_Lb=ai}', "");
Expect(1, 127387, '\p{^Is_Lb=ai}', "");
Expect(1, 127387, '\P{Is_Lb=ai}', "");
Expect(0, 127387, '\P{^Is_Lb=ai}', "");
Expect(1, 127386, '\p{Is_Lb:_	AI}', "");
Expect(0, 127386, '\p{^Is_Lb:_	AI}', "");
Expect(0, 127386, '\P{Is_Lb:_	AI}', "");
Expect(1, 127386, '\P{^Is_Lb:_	AI}', "");
Expect(0, 127387, '\p{Is_Lb:_	AI}', "");
Expect(1, 127387, '\p{^Is_Lb:_	AI}', "");
Expect(1, 127387, '\P{Is_Lb:_	AI}', "");
Expect(0, 127387, '\P{^Is_Lb:_	AI}', "");
Error('\p{Line_Break=__conditional_japanese_STARTER/a/}');
Error('\P{Line_Break=__conditional_japanese_STARTER/a/}');
Expect(1, 65392, '\p{Line_Break:	conditionaljapanesestarter}', "");
Expect(0, 65392, '\p{^Line_Break:	conditionaljapanesestarter}', "");
Expect(0, 65392, '\P{Line_Break:	conditionaljapanesestarter}', "");
Expect(1, 65392, '\P{^Line_Break:	conditionaljapanesestarter}', "");
Expect(0, 65393, '\p{Line_Break:	conditionaljapanesestarter}', "");
Expect(1, 65393, '\p{^Line_Break:	conditionaljapanesestarter}', "");
Expect(1, 65393, '\P{Line_Break:	conditionaljapanesestarter}', "");
Expect(0, 65393, '\P{^Line_Break:	conditionaljapanesestarter}', "");
Expect(1, 65392, '\p{Line_Break=_ conditional_Japanese_Starter}', "");
Expect(0, 65392, '\p{^Line_Break=_ conditional_Japanese_Starter}', "");
Expect(0, 65392, '\P{Line_Break=_ conditional_Japanese_Starter}', "");
Expect(1, 65392, '\P{^Line_Break=_ conditional_Japanese_Starter}', "");
Expect(0, 65393, '\p{Line_Break=_ conditional_Japanese_Starter}', "");
Expect(1, 65393, '\p{^Line_Break=_ conditional_Japanese_Starter}', "");
Expect(1, 65393, '\P{Line_Break=_ conditional_Japanese_Starter}', "");
Expect(0, 65393, '\P{^Line_Break=_ conditional_Japanese_Starter}', "");
Error('\p{Lb=-CJ:=}');
Error('\P{Lb=-CJ:=}');
Expect(1, 65392, '\p{Lb=cj}', "");
Expect(0, 65392, '\p{^Lb=cj}', "");
Expect(0, 65392, '\P{Lb=cj}', "");
Expect(1, 65392, '\P{^Lb=cj}', "");
Expect(0, 65393, '\p{Lb=cj}', "");
Expect(1, 65393, '\p{^Lb=cj}', "");
Expect(1, 65393, '\P{Lb=cj}', "");
Expect(0, 65393, '\P{^Lb=cj}', "");
Expect(1, 65392, '\p{Lb=-CJ}', "");
Expect(0, 65392, '\p{^Lb=-CJ}', "");
Expect(0, 65392, '\P{Lb=-CJ}', "");
Expect(1, 65392, '\P{^Lb=-CJ}', "");
Expect(0, 65393, '\p{Lb=-CJ}', "");
Expect(1, 65393, '\p{^Lb=-CJ}', "");
Expect(1, 65393, '\P{Lb=-CJ}', "");
Expect(0, 65393, '\P{^Lb=-CJ}', "");
Error('\p{Is_Line_Break=:=Conditional_Japanese_STARTER}');
Error('\P{Is_Line_Break=:=Conditional_Japanese_STARTER}');
Expect(1, 65392, '\p{Is_Line_Break=conditionaljapanesestarter}', "");
Expect(0, 65392, '\p{^Is_Line_Break=conditionaljapanesestarter}', "");
Expect(0, 65392, '\P{Is_Line_Break=conditionaljapanesestarter}', "");
Expect(1, 65392, '\P{^Is_Line_Break=conditionaljapanesestarter}', "");
Expect(0, 65393, '\p{Is_Line_Break=conditionaljapanesestarter}', "");
Expect(1, 65393, '\p{^Is_Line_Break=conditionaljapanesestarter}', "");
Expect(1, 65393, '\P{Is_Line_Break=conditionaljapanesestarter}', "");
Expect(0, 65393, '\P{^Is_Line_Break=conditionaljapanesestarter}', "");
Expect(1, 65392, '\p{Is_Line_Break=_-CONDITIONAL_Japanese_starter}', "");
Expect(0, 65392, '\p{^Is_Line_Break=_-CONDITIONAL_Japanese_starter}', "");
Expect(0, 65392, '\P{Is_Line_Break=_-CONDITIONAL_Japanese_starter}', "");
Expect(1, 65392, '\P{^Is_Line_Break=_-CONDITIONAL_Japanese_starter}', "");
Expect(0, 65393, '\p{Is_Line_Break=_-CONDITIONAL_Japanese_starter}', "");
Expect(1, 65393, '\p{^Is_Line_Break=_-CONDITIONAL_Japanese_starter}', "");
Expect(1, 65393, '\P{Is_Line_Break=_-CONDITIONAL_Japanese_starter}', "");
Expect(0, 65393, '\P{^Is_Line_Break=_-CONDITIONAL_Japanese_starter}', "");
Error('\p{Is_Lb= -cj:=}');
Error('\P{Is_Lb= -cj:=}');
Expect(1, 65392, '\p{Is_Lb=cj}', "");
Expect(0, 65392, '\p{^Is_Lb=cj}', "");
Expect(0, 65392, '\P{Is_Lb=cj}', "");
Expect(1, 65392, '\P{^Is_Lb=cj}', "");
Expect(0, 65393, '\p{Is_Lb=cj}', "");
Expect(1, 65393, '\p{^Is_Lb=cj}', "");
Expect(1, 65393, '\P{Is_Lb=cj}', "");
Expect(0, 65393, '\P{^Is_Lb=cj}', "");
Expect(1, 65392, '\p{Is_Lb= CJ}', "");
Expect(0, 65392, '\p{^Is_Lb= CJ}', "");
Expect(0, 65392, '\P{Is_Lb= CJ}', "");
Expect(1, 65392, '\P{^Is_Lb= CJ}', "");
Expect(0, 65393, '\p{Is_Lb= CJ}', "");
Expect(1, 65393, '\p{^Is_Lb= CJ}', "");
Expect(1, 65393, '\P{Is_Lb= CJ}', "");
Expect(0, 65393, '\P{^Is_Lb= CJ}', "");
Error('\p{Line_Break=_/a/Contingent_break}');
Error('\P{Line_Break=_/a/Contingent_break}');
Expect(1, 65532, '\p{Line_Break=contingentbreak}', "");
Expect(0, 65532, '\p{^Line_Break=contingentbreak}', "");
Expect(0, 65532, '\P{Line_Break=contingentbreak}', "");
Expect(1, 65532, '\P{^Line_Break=contingentbreak}', "");
Expect(0, 65533, '\p{Line_Break=contingentbreak}', "");
Expect(1, 65533, '\p{^Line_Break=contingentbreak}', "");
Expect(1, 65533, '\P{Line_Break=contingentbreak}', "");
Expect(0, 65533, '\P{^Line_Break=contingentbreak}', "");
Expect(1, 65532, '\p{Line_Break=_-Contingent_Break}', "");
Expect(0, 65532, '\p{^Line_Break=_-Contingent_Break}', "");
Expect(0, 65532, '\P{Line_Break=_-Contingent_Break}', "");
Expect(1, 65532, '\P{^Line_Break=_-Contingent_Break}', "");
Expect(0, 65533, '\p{Line_Break=_-Contingent_Break}', "");
Expect(1, 65533, '\p{^Line_Break=_-Contingent_Break}', "");
Expect(1, 65533, '\P{Line_Break=_-Contingent_Break}', "");
Expect(0, 65533, '\P{^Line_Break=_-Contingent_Break}', "");
Error('\p{Lb= _cb/a/}');
Error('\P{Lb= _cb/a/}');
Expect(1, 65532, '\p{Lb=cb}', "");
Expect(0, 65532, '\p{^Lb=cb}', "");
Expect(0, 65532, '\P{Lb=cb}', "");
Expect(1, 65532, '\P{^Lb=cb}', "");
Expect(0, 65533, '\p{Lb=cb}', "");
Expect(1, 65533, '\p{^Lb=cb}', "");
Expect(1, 65533, '\P{Lb=cb}', "");
Expect(0, 65533, '\P{^Lb=cb}', "");
Expect(1, 65532, '\p{Lb=	 cb}', "");
Expect(0, 65532, '\p{^Lb=	 cb}', "");
Expect(0, 65532, '\P{Lb=	 cb}', "");
Expect(1, 65532, '\P{^Lb=	 cb}', "");
Expect(0, 65533, '\p{Lb=	 cb}', "");
Expect(1, 65533, '\p{^Lb=	 cb}', "");
Expect(1, 65533, '\P{Lb=	 cb}', "");
Expect(0, 65533, '\P{^Lb=	 cb}', "");
Error('\p{Is_Line_Break=	:=Contingent_Break}');
Error('\P{Is_Line_Break=	:=Contingent_Break}');
Expect(1, 65532, '\p{Is_Line_Break=contingentbreak}', "");
Expect(0, 65532, '\p{^Is_Line_Break=contingentbreak}', "");
Expect(0, 65532, '\P{Is_Line_Break=contingentbreak}', "");
Expect(1, 65532, '\P{^Is_Line_Break=contingentbreak}', "");
Expect(0, 65533, '\p{Is_Line_Break=contingentbreak}', "");
Expect(1, 65533, '\p{^Is_Line_Break=contingentbreak}', "");
Expect(1, 65533, '\P{Is_Line_Break=contingentbreak}', "");
Expect(0, 65533, '\P{^Is_Line_Break=contingentbreak}', "");
Expect(1, 65532, '\p{Is_Line_Break=		Contingent_break}', "");
Expect(0, 65532, '\p{^Is_Line_Break=		Contingent_break}', "");
Expect(0, 65532, '\P{Is_Line_Break=		Contingent_break}', "");
Expect(1, 65532, '\P{^Is_Line_Break=		Contingent_break}', "");
Expect(0, 65533, '\p{Is_Line_Break=		Contingent_break}', "");
Expect(1, 65533, '\p{^Is_Line_Break=		Contingent_break}', "");
Expect(1, 65533, '\P{Is_Line_Break=		Contingent_break}', "");
Expect(0, 65533, '\P{^Is_Line_Break=		Contingent_break}', "");
Error('\p{Is_Lb=/a/- CB}');
Error('\P{Is_Lb=/a/- CB}');
Expect(1, 65532, '\p{Is_Lb=cb}', "");
Expect(0, 65532, '\p{^Is_Lb=cb}', "");
Expect(0, 65532, '\P{Is_Lb=cb}', "");
Expect(1, 65532, '\P{^Is_Lb=cb}', "");
Expect(0, 65533, '\p{Is_Lb=cb}', "");
Expect(1, 65533, '\p{^Is_Lb=cb}', "");
Expect(1, 65533, '\P{Is_Lb=cb}', "");
Expect(0, 65533, '\P{^Is_Lb=cb}', "");
Expect(1, 65532, '\p{Is_Lb=--CB}', "");
Expect(0, 65532, '\p{^Is_Lb=--CB}', "");
Expect(0, 65532, '\P{Is_Lb=--CB}', "");
Expect(1, 65532, '\P{^Is_Lb=--CB}', "");
Expect(0, 65533, '\p{Is_Lb=--CB}', "");
Expect(1, 65533, '\p{^Is_Lb=--CB}', "");
Expect(1, 65533, '\P{Is_Lb=--CB}', "");
Expect(0, 65533, '\P{^Is_Lb=--CB}', "");
Error('\p{Line_Break=	/a/IDEOGRAPHIC}');
Error('\P{Line_Break=	/a/IDEOGRAPHIC}');
Expect(1, 262141, '\p{Line_Break=ideographic}', "");
Expect(0, 262141, '\p{^Line_Break=ideographic}', "");
Expect(0, 262141, '\P{Line_Break=ideographic}', "");
Expect(1, 262141, '\P{^Line_Break=ideographic}', "");
Expect(0, 262144, '\p{Line_Break=ideographic}', "");
Expect(1, 262144, '\p{^Line_Break=ideographic}', "");
Expect(1, 262144, '\P{Line_Break=ideographic}', "");
Expect(0, 262144, '\P{^Line_Break=ideographic}', "");
Expect(1, 262141, '\p{Line_Break=	_ideographic}', "");
Expect(0, 262141, '\p{^Line_Break=	_ideographic}', "");
Expect(0, 262141, '\P{Line_Break=	_ideographic}', "");
Expect(1, 262141, '\P{^Line_Break=	_ideographic}', "");
Expect(0, 262144, '\p{Line_Break=	_ideographic}', "");
Expect(1, 262144, '\p{^Line_Break=	_ideographic}', "");
Expect(1, 262144, '\P{Line_Break=	_ideographic}', "");
Expect(0, 262144, '\P{^Line_Break=	_ideographic}', "");
Error('\p{Lb=:=_ ID}');
Error('\P{Lb=:=_ ID}');
Expect(1, 262141, '\p{Lb=id}', "");
Expect(0, 262141, '\p{^Lb=id}', "");
Expect(0, 262141, '\P{Lb=id}', "");
Expect(1, 262141, '\P{^Lb=id}', "");
Expect(0, 262144, '\p{Lb=id}', "");
Expect(1, 262144, '\p{^Lb=id}', "");
Expect(1, 262144, '\P{Lb=id}', "");
Expect(0, 262144, '\P{^Lb=id}', "");
Expect(1, 262141, '\p{Lb= 	ID}', "");
Expect(0, 262141, '\p{^Lb= 	ID}', "");
Expect(0, 262141, '\P{Lb= 	ID}', "");
Expect(1, 262141, '\P{^Lb= 	ID}', "");
Expect(0, 262144, '\p{Lb= 	ID}', "");
Expect(1, 262144, '\p{^Lb= 	ID}', "");
Expect(1, 262144, '\P{Lb= 	ID}', "");
Expect(0, 262144, '\P{^Lb= 	ID}', "");
Error('\p{Is_Line_Break=-/a/IDEOGRAPHIC}');
Error('\P{Is_Line_Break=-/a/IDEOGRAPHIC}');
Expect(1, 262141, '\p{Is_Line_Break=ideographic}', "");
Expect(0, 262141, '\p{^Is_Line_Break=ideographic}', "");
Expect(0, 262141, '\P{Is_Line_Break=ideographic}', "");
Expect(1, 262141, '\P{^Is_Line_Break=ideographic}', "");
Expect(0, 262144, '\p{Is_Line_Break=ideographic}', "");
Expect(1, 262144, '\p{^Is_Line_Break=ideographic}', "");
Expect(1, 262144, '\P{Is_Line_Break=ideographic}', "");
Expect(0, 262144, '\P{^Is_Line_Break=ideographic}', "");
Expect(1, 262141, '\p{Is_Line_Break=- ideographic}', "");
Expect(0, 262141, '\p{^Is_Line_Break=- ideographic}', "");
Expect(0, 262141, '\P{Is_Line_Break=- ideographic}', "");
Expect(1, 262141, '\P{^Is_Line_Break=- ideographic}', "");
Expect(0, 262144, '\p{Is_Line_Break=- ideographic}', "");
Expect(1, 262144, '\p{^Is_Line_Break=- ideographic}', "");
Expect(1, 262144, '\P{Is_Line_Break=- ideographic}', "");
Expect(0, 262144, '\P{^Is_Line_Break=- ideographic}', "");
Error('\p{Is_Lb:   		ID:=}');
Error('\P{Is_Lb:   		ID:=}');
Expect(1, 262141, '\p{Is_Lb=id}', "");
Expect(0, 262141, '\p{^Is_Lb=id}', "");
Expect(0, 262141, '\P{Is_Lb=id}', "");
Expect(1, 262141, '\P{^Is_Lb=id}', "");
Expect(0, 262144, '\p{Is_Lb=id}', "");
Expect(1, 262144, '\p{^Is_Lb=id}', "");
Expect(1, 262144, '\P{Is_Lb=id}', "");
Expect(0, 262144, '\P{^Is_Lb=id}', "");
Expect(1, 262141, '\p{Is_Lb=-id}', "");
Expect(0, 262141, '\p{^Is_Lb=-id}', "");
Expect(0, 262141, '\P{Is_Lb=-id}', "");
Expect(1, 262141, '\P{^Is_Lb=-id}', "");
Expect(0, 262144, '\p{Is_Lb=-id}', "");
Expect(1, 262144, '\p{^Is_Lb=-id}', "");
Expect(1, 262144, '\P{Is_Lb=-id}', "");
Expect(0, 262144, '\P{^Is_Lb=-id}', "");
Error('\p{Line_Break=:=_Exclamation}');
Error('\P{Line_Break=:=_Exclamation}');
Expect(1, 65311, '\p{Line_Break=exclamation}', "");
Expect(0, 65311, '\p{^Line_Break=exclamation}', "");
Expect(0, 65311, '\P{Line_Break=exclamation}', "");
Expect(1, 65311, '\P{^Line_Break=exclamation}', "");
Expect(0, 65312, '\p{Line_Break=exclamation}', "");
Expect(1, 65312, '\p{^Line_Break=exclamation}', "");
Expect(1, 65312, '\P{Line_Break=exclamation}', "");
Expect(0, 65312, '\P{^Line_Break=exclamation}', "");
Expect(1, 65311, '\p{Line_Break=_ Exclamation}', "");
Expect(0, 65311, '\p{^Line_Break=_ Exclamation}', "");
Expect(0, 65311, '\P{Line_Break=_ Exclamation}', "");
Expect(1, 65311, '\P{^Line_Break=_ Exclamation}', "");
Expect(0, 65312, '\p{Line_Break=_ Exclamation}', "");
Expect(1, 65312, '\p{^Line_Break=_ Exclamation}', "");
Expect(1, 65312, '\P{Line_Break=_ Exclamation}', "");
Expect(0, 65312, '\P{^Line_Break=_ Exclamation}', "");
Error('\p{Lb=/a/EX}');
Error('\P{Lb=/a/EX}');
Expect(1, 65311, '\p{Lb:   ex}', "");
Expect(0, 65311, '\p{^Lb:   ex}', "");
Expect(0, 65311, '\P{Lb:   ex}', "");
Expect(1, 65311, '\P{^Lb:   ex}', "");
Expect(0, 65312, '\p{Lb:   ex}', "");
Expect(1, 65312, '\p{^Lb:   ex}', "");
Expect(1, 65312, '\P{Lb:   ex}', "");
Expect(0, 65312, '\P{^Lb:   ex}', "");
Expect(1, 65311, '\p{Lb=-ex}', "");
Expect(0, 65311, '\p{^Lb=-ex}', "");
Expect(0, 65311, '\P{Lb=-ex}', "");
Expect(1, 65311, '\P{^Lb=-ex}', "");
Expect(0, 65312, '\p{Lb=-ex}', "");
Expect(1, 65312, '\p{^Lb=-ex}', "");
Expect(1, 65312, '\P{Lb=-ex}', "");
Expect(0, 65312, '\P{^Lb=-ex}', "");
Error('\p{Is_Line_Break= :=Exclamation}');
Error('\P{Is_Line_Break= :=Exclamation}');
Expect(1, 65311, '\p{Is_Line_Break=exclamation}', "");
Expect(0, 65311, '\p{^Is_Line_Break=exclamation}', "");
Expect(0, 65311, '\P{Is_Line_Break=exclamation}', "");
Expect(1, 65311, '\P{^Is_Line_Break=exclamation}', "");
Expect(0, 65312, '\p{Is_Line_Break=exclamation}', "");
Expect(1, 65312, '\p{^Is_Line_Break=exclamation}', "");
Expect(1, 65312, '\P{Is_Line_Break=exclamation}', "");
Expect(0, 65312, '\P{^Is_Line_Break=exclamation}', "");
Expect(1, 65311, '\p{Is_Line_Break=_Exclamation}', "");
Expect(0, 65311, '\p{^Is_Line_Break=_Exclamation}', "");
Expect(0, 65311, '\P{Is_Line_Break=_Exclamation}', "");
Expect(1, 65311, '\P{^Is_Line_Break=_Exclamation}', "");
Expect(0, 65312, '\p{Is_Line_Break=_Exclamation}', "");
Expect(1, 65312, '\p{^Is_Line_Break=_Exclamation}', "");
Expect(1, 65312, '\P{Is_Line_Break=_Exclamation}', "");
Expect(0, 65312, '\P{^Is_Line_Break=_Exclamation}', "");
Error('\p{Is_Lb= :=EX}');
Error('\P{Is_Lb= :=EX}');
Expect(1, 65311, '\p{Is_Lb=ex}', "");
Expect(0, 65311, '\p{^Is_Lb=ex}', "");
Expect(0, 65311, '\P{Is_Lb=ex}', "");
Expect(1, 65311, '\P{^Is_Lb=ex}', "");
Expect(0, 65312, '\p{Is_Lb=ex}', "");
Expect(1, 65312, '\p{^Is_Lb=ex}', "");
Expect(1, 65312, '\P{Is_Lb=ex}', "");
Expect(0, 65312, '\P{^Is_Lb=ex}', "");
Expect(1, 65311, '\p{Is_Lb=	EX}', "");
Expect(0, 65311, '\p{^Is_Lb=	EX}', "");
Expect(0, 65311, '\P{Is_Lb=	EX}', "");
Expect(1, 65311, '\P{^Is_Lb=	EX}', "");
Expect(0, 65312, '\p{Is_Lb=	EX}', "");
Expect(1, 65312, '\p{^Is_Lb=	EX}', "");
Expect(1, 65312, '\P{Is_Lb=	EX}', "");
Expect(0, 65312, '\P{^Is_Lb=	EX}', "");
Error('\p{Line_Break=:=	 Word_joiner}');
Error('\P{Line_Break=:=	 Word_joiner}');
Expect(1, 65279, '\p{Line_Break=wordjoiner}', "");
Expect(0, 65279, '\p{^Line_Break=wordjoiner}', "");
Expect(0, 65279, '\P{Line_Break=wordjoiner}', "");
Expect(1, 65279, '\P{^Line_Break=wordjoiner}', "");
Expect(0, 65280, '\p{Line_Break=wordjoiner}', "");
Expect(1, 65280, '\p{^Line_Break=wordjoiner}', "");
Expect(1, 65280, '\P{Line_Break=wordjoiner}', "");
Expect(0, 65280, '\P{^Line_Break=wordjoiner}', "");
Expect(1, 65279, '\p{Line_Break=  word_JOINER}', "");
Expect(0, 65279, '\p{^Line_Break=  word_JOINER}', "");
Expect(0, 65279, '\P{Line_Break=  word_JOINER}', "");
Expect(1, 65279, '\P{^Line_Break=  word_JOINER}', "");
Expect(0, 65280, '\p{Line_Break=  word_JOINER}', "");
Expect(1, 65280, '\p{^Line_Break=  word_JOINER}', "");
Expect(1, 65280, '\P{Line_Break=  word_JOINER}', "");
Expect(0, 65280, '\P{^Line_Break=  word_JOINER}', "");
Error('\p{Lb=-WJ:=}');
Error('\P{Lb=-WJ:=}');
Expect(1, 65279, '\p{Lb=wj}', "");
Expect(0, 65279, '\p{^Lb=wj}', "");
Expect(0, 65279, '\P{Lb=wj}', "");
Expect(1, 65279, '\P{^Lb=wj}', "");
Expect(0, 65280, '\p{Lb=wj}', "");
Expect(1, 65280, '\p{^Lb=wj}', "");
Expect(1, 65280, '\P{Lb=wj}', "");
Expect(0, 65280, '\P{^Lb=wj}', "");
Error('\p{Is_Line_Break=-:=WORD_joiner}');
Error('\P{Is_Line_Break=-:=WORD_joiner}');
Expect(1, 65279, '\p{Is_Line_Break=wordjoiner}', "");
Expect(0, 65279, '\p{^Is_Line_Break=wordjoiner}', "");
Expect(0, 65279, '\P{Is_Line_Break=wordjoiner}', "");
Expect(1, 65279, '\P{^Is_Line_Break=wordjoiner}', "");
Expect(0, 65280, '\p{Is_Line_Break=wordjoiner}', "");
Expect(1, 65280, '\p{^Is_Line_Break=wordjoiner}', "");
Expect(1, 65280, '\P{Is_Line_Break=wordjoiner}', "");
Expect(0, 65280, '\P{^Is_Line_Break=wordjoiner}', "");
Expect(1, 65279, '\p{Is_Line_Break=--WORD_JOINER}', "");
Expect(0, 65279, '\p{^Is_Line_Break=--WORD_JOINER}', "");
Expect(0, 65279, '\P{Is_Line_Break=--WORD_JOINER}', "");
Expect(1, 65279, '\P{^Is_Line_Break=--WORD_JOINER}', "");
Expect(0, 65280, '\p{Is_Line_Break=--WORD_JOINER}', "");
Expect(1, 65280, '\p{^Is_Line_Break=--WORD_JOINER}', "");
Expect(1, 65280, '\P{Is_Line_Break=--WORD_JOINER}', "");
Expect(0, 65280, '\P{^Is_Line_Break=--WORD_JOINER}', "");
Error('\p{Is_Lb=	wj:=}');
Error('\P{Is_Lb=	wj:=}');
Expect(1, 65279, '\p{Is_Lb=wj}', "");
Expect(0, 65279, '\p{^Is_Lb=wj}', "");
Expect(0, 65279, '\P{Is_Lb=wj}', "");
Expect(1, 65279, '\P{^Is_Lb=wj}', "");
Expect(0, 65280, '\p{Is_Lb=wj}', "");
Expect(1, 65280, '\p{^Is_Lb=wj}', "");
Expect(1, 65280, '\P{Is_Lb=wj}', "");
Expect(0, 65280, '\P{^Is_Lb=wj}', "");
Expect(1, 65279, '\p{Is_Lb=- WJ}', "");
Expect(0, 65279, '\p{^Is_Lb=- WJ}', "");
Expect(0, 65279, '\P{Is_Lb=- WJ}', "");
Expect(1, 65279, '\P{^Is_Lb=- WJ}', "");
Expect(0, 65280, '\p{Is_Lb=- WJ}', "");
Expect(1, 65280, '\p{^Is_Lb=- WJ}', "");
Expect(1, 65280, '\P{Is_Lb=- WJ}', "");
Expect(0, 65280, '\P{^Is_Lb=- WJ}', "");
Error('\p{Line_Break= H2/a/}');
Error('\P{Line_Break= H2/a/}');
Expect(1, 55176, '\p{Line_Break=h2}', "");
Expect(0, 55176, '\p{^Line_Break=h2}', "");
Expect(0, 55176, '\P{Line_Break=h2}', "");
Expect(1, 55176, '\P{^Line_Break=h2}', "");
Expect(0, 55177, '\p{Line_Break=h2}', "");
Expect(1, 55177, '\p{^Line_Break=h2}', "");
Expect(1, 55177, '\P{Line_Break=h2}', "");
Expect(0, 55177, '\P{^Line_Break=h2}', "");
Expect(1, 55176, '\p{Line_Break:    -H2}', "");
Expect(0, 55176, '\p{^Line_Break:    -H2}', "");
Expect(0, 55176, '\P{Line_Break:    -H2}', "");
Expect(1, 55176, '\P{^Line_Break:    -H2}', "");
Expect(0, 55177, '\p{Line_Break:    -H2}', "");
Expect(1, 55177, '\p{^Line_Break:    -H2}', "");
Expect(1, 55177, '\P{Line_Break:    -H2}', "");
Expect(0, 55177, '\P{^Line_Break:    -H2}', "");
Error('\p{Lb=:=H2}');
Error('\P{Lb=:=H2}');
Expect(1, 55176, '\p{Lb=h2}', "");
Expect(0, 55176, '\p{^Lb=h2}', "");
Expect(0, 55176, '\P{Lb=h2}', "");
Expect(1, 55176, '\P{^Lb=h2}', "");
Expect(0, 55177, '\p{Lb=h2}', "");
Expect(1, 55177, '\p{^Lb=h2}', "");
Expect(1, 55177, '\P{Lb=h2}', "");
Expect(0, 55177, '\P{^Lb=h2}', "");
Expect(1, 55176, '\p{Lb:  -H2}', "");
Expect(0, 55176, '\p{^Lb:  -H2}', "");
Expect(0, 55176, '\P{Lb:  -H2}', "");
Expect(1, 55176, '\P{^Lb:  -H2}', "");
Expect(0, 55177, '\p{Lb:  -H2}', "");
Expect(1, 55177, '\p{^Lb:  -H2}', "");
Expect(1, 55177, '\P{Lb:  -H2}', "");
Expect(0, 55177, '\P{^Lb:  -H2}', "");
Error('\p{Is_Line_Break=-:=H2}');
Error('\P{Is_Line_Break=-:=H2}');
Expect(1, 55176, '\p{Is_Line_Break=h2}', "");
Expect(0, 55176, '\p{^Is_Line_Break=h2}', "");
Expect(0, 55176, '\P{Is_Line_Break=h2}', "");
Expect(1, 55176, '\P{^Is_Line_Break=h2}', "");
Expect(0, 55177, '\p{Is_Line_Break=h2}', "");
Expect(1, 55177, '\p{^Is_Line_Break=h2}', "");
Expect(1, 55177, '\P{Is_Line_Break=h2}', "");
Expect(0, 55177, '\P{^Is_Line_Break=h2}', "");
Expect(1, 55176, '\p{Is_Line_Break:	-H2}', "");
Expect(0, 55176, '\p{^Is_Line_Break:	-H2}', "");
Expect(0, 55176, '\P{Is_Line_Break:	-H2}', "");
Expect(1, 55176, '\P{^Is_Line_Break:	-H2}', "");
Expect(0, 55177, '\p{Is_Line_Break:	-H2}', "");
Expect(1, 55177, '\p{^Is_Line_Break:	-H2}', "");
Expect(1, 55177, '\P{Is_Line_Break:	-H2}', "");
Expect(0, 55177, '\P{^Is_Line_Break:	-H2}', "");
Error('\p{Is_Lb: __H2:=}');
Error('\P{Is_Lb: __H2:=}');
Expect(1, 55176, '\p{Is_Lb=h2}', "");
Expect(0, 55176, '\p{^Is_Lb=h2}', "");
Expect(0, 55176, '\P{Is_Lb=h2}', "");
Expect(1, 55176, '\P{^Is_Lb=h2}', "");
Expect(0, 55177, '\p{Is_Lb=h2}', "");
Expect(1, 55177, '\p{^Is_Lb=h2}', "");
Expect(1, 55177, '\P{Is_Lb=h2}', "");
Expect(0, 55177, '\P{^Is_Lb=h2}', "");
Expect(1, 55176, '\p{Is_Lb=	-H2}', "");
Expect(0, 55176, '\p{^Is_Lb=	-H2}', "");
Expect(0, 55176, '\P{Is_Lb=	-H2}', "");
Expect(1, 55176, '\P{^Is_Lb=	-H2}', "");
Expect(0, 55177, '\p{Is_Lb=	-H2}', "");
Expect(1, 55177, '\p{^Is_Lb=	-H2}', "");
Expect(1, 55177, '\P{Is_Lb=	-H2}', "");
Expect(0, 55177, '\P{^Is_Lb=	-H2}', "");
Error('\p{Line_Break=/a/__BREAK_Both}');
Error('\P{Line_Break=/a/__BREAK_Both}');
Expect(1, 11835, '\p{Line_Break=breakboth}', "");
Expect(0, 11835, '\p{^Line_Break=breakboth}', "");
Expect(0, 11835, '\P{Line_Break=breakboth}', "");
Expect(1, 11835, '\P{^Line_Break=breakboth}', "");
Expect(0, 11836, '\p{Line_Break=breakboth}', "");
Expect(1, 11836, '\p{^Line_Break=breakboth}', "");
Expect(1, 11836, '\P{Line_Break=breakboth}', "");
Expect(0, 11836, '\P{^Line_Break=breakboth}', "");
Expect(1, 11835, '\p{Line_Break=	Break_BOTH}', "");
Expect(0, 11835, '\p{^Line_Break=	Break_BOTH}', "");
Expect(0, 11835, '\P{Line_Break=	Break_BOTH}', "");
Expect(1, 11835, '\P{^Line_Break=	Break_BOTH}', "");
Expect(0, 11836, '\p{Line_Break=	Break_BOTH}', "");
Expect(1, 11836, '\p{^Line_Break=	Break_BOTH}', "");
Expect(1, 11836, '\P{Line_Break=	Break_BOTH}', "");
Expect(0, 11836, '\P{^Line_Break=	Break_BOTH}', "");
Error('\p{Lb= 	B2/a/}');
Error('\P{Lb= 	B2/a/}');
Expect(1, 11835, '\p{Lb: b2}', "");
Expect(0, 11835, '\p{^Lb: b2}', "");
Expect(0, 11835, '\P{Lb: b2}', "");
Expect(1, 11835, '\P{^Lb: b2}', "");
Expect(0, 11836, '\p{Lb: b2}', "");
Expect(1, 11836, '\p{^Lb: b2}', "");
Expect(1, 11836, '\P{Lb: b2}', "");
Expect(0, 11836, '\P{^Lb: b2}', "");
Expect(1, 11835, '\p{Lb: -B2}', "");
Expect(0, 11835, '\p{^Lb: -B2}', "");
Expect(0, 11835, '\P{Lb: -B2}', "");
Expect(1, 11835, '\P{^Lb: -B2}', "");
Expect(0, 11836, '\p{Lb: -B2}', "");
Expect(1, 11836, '\p{^Lb: -B2}', "");
Expect(1, 11836, '\P{Lb: -B2}', "");
Expect(0, 11836, '\P{^Lb: -B2}', "");
Error('\p{Is_Line_Break:   _Break_BOTH:=}');
Error('\P{Is_Line_Break:   _Break_BOTH:=}');
Expect(1, 11835, '\p{Is_Line_Break=breakboth}', "");
Expect(0, 11835, '\p{^Is_Line_Break=breakboth}', "");
Expect(0, 11835, '\P{Is_Line_Break=breakboth}', "");
Expect(1, 11835, '\P{^Is_Line_Break=breakboth}', "");
Expect(0, 11836, '\p{Is_Line_Break=breakboth}', "");
Expect(1, 11836, '\p{^Is_Line_Break=breakboth}', "");
Expect(1, 11836, '\P{Is_Line_Break=breakboth}', "");
Expect(0, 11836, '\P{^Is_Line_Break=breakboth}', "");
Expect(1, 11835, '\p{Is_Line_Break=_ break_both}', "");
Expect(0, 11835, '\p{^Is_Line_Break=_ break_both}', "");
Expect(0, 11835, '\P{Is_Line_Break=_ break_both}', "");
Expect(1, 11835, '\P{^Is_Line_Break=_ break_both}', "");
Expect(0, 11836, '\p{Is_Line_Break=_ break_both}', "");
Expect(1, 11836, '\p{^Is_Line_Break=_ break_both}', "");
Expect(1, 11836, '\P{Is_Line_Break=_ break_both}', "");
Expect(0, 11836, '\P{^Is_Line_Break=_ break_both}', "");
Error('\p{Is_Lb=/a/B2}');
Error('\P{Is_Lb=/a/B2}');
Expect(1, 11835, '\p{Is_Lb=b2}', "");
Expect(0, 11835, '\p{^Is_Lb=b2}', "");
Expect(0, 11835, '\P{Is_Lb=b2}', "");
Expect(1, 11835, '\P{^Is_Lb=b2}', "");
Expect(0, 11836, '\p{Is_Lb=b2}', "");
Expect(1, 11836, '\p{^Is_Lb=b2}', "");
Expect(1, 11836, '\P{Is_Lb=b2}', "");
Expect(0, 11836, '\P{^Is_Lb=b2}', "");
Expect(1, 11835, '\p{Is_Lb=-	B2}', "");
Expect(0, 11835, '\p{^Is_Lb=-	B2}', "");
Expect(0, 11835, '\P{Is_Lb=-	B2}', "");
Expect(1, 11835, '\P{^Is_Lb=-	B2}', "");
Expect(0, 11836, '\p{Is_Lb=-	B2}', "");
Expect(1, 11836, '\p{^Is_Lb=-	B2}', "");
Expect(1, 11836, '\P{Is_Lb=-	B2}', "");
Expect(0, 11836, '\P{^Is_Lb=-	B2}', "");
Error('\p{Line_Break=:=Alphabetic}');
Error('\P{Line_Break=:=Alphabetic}');
Expect(1, 128883, '\p{Line_Break=alphabetic}', "");
Expect(0, 128883, '\p{^Line_Break=alphabetic}', "");
Expect(0, 128883, '\P{Line_Break=alphabetic}', "");
Expect(1, 128883, '\P{^Line_Break=alphabetic}', "");
Expect(0, 128884, '\p{Line_Break=alphabetic}', "");
Expect(1, 128884, '\p{^Line_Break=alphabetic}', "");
Expect(1, 128884, '\P{Line_Break=alphabetic}', "");
Expect(0, 128884, '\P{^Line_Break=alphabetic}', "");
Expect(1, 128883, '\p{Line_Break=--Alphabetic}', "");
Expect(0, 128883, '\p{^Line_Break=--Alphabetic}', "");
Expect(0, 128883, '\P{Line_Break=--Alphabetic}', "");
Expect(1, 128883, '\P{^Line_Break=--Alphabetic}', "");
Expect(0, 128884, '\p{Line_Break=--Alphabetic}', "");
Expect(1, 128884, '\p{^Line_Break=--Alphabetic}', "");
Expect(1, 128884, '\P{Line_Break=--Alphabetic}', "");
Expect(0, 128884, '\P{^Line_Break=--Alphabetic}', "");
Error('\p{Lb=:=_	al}');
Error('\P{Lb=:=_	al}');
Expect(1, 128883, '\p{Lb: al}', "");
Expect(0, 128883, '\p{^Lb: al}', "");
Expect(0, 128883, '\P{Lb: al}', "");
Expect(1, 128883, '\P{^Lb: al}', "");
Expect(0, 128884, '\p{Lb: al}', "");
Expect(1, 128884, '\p{^Lb: al}', "");
Expect(1, 128884, '\P{Lb: al}', "");
Expect(0, 128884, '\P{^Lb: al}', "");
Expect(1, 128883, '\p{Lb=-AL}', "");
Expect(0, 128883, '\p{^Lb=-AL}', "");
Expect(0, 128883, '\P{Lb=-AL}', "");
Expect(1, 128883, '\P{^Lb=-AL}', "");
Expect(0, 128884, '\p{Lb=-AL}', "");
Expect(1, 128884, '\p{^Lb=-AL}', "");
Expect(1, 128884, '\P{Lb=-AL}', "");
Expect(0, 128884, '\P{^Lb=-AL}', "");
Error('\p{Is_Line_Break=	:=Alphabetic}');
Error('\P{Is_Line_Break=	:=Alphabetic}');
Expect(1, 128883, '\p{Is_Line_Break=alphabetic}', "");
Expect(0, 128883, '\p{^Is_Line_Break=alphabetic}', "");
Expect(0, 128883, '\P{Is_Line_Break=alphabetic}', "");
Expect(1, 128883, '\P{^Is_Line_Break=alphabetic}', "");
Expect(0, 128884, '\p{Is_Line_Break=alphabetic}', "");
Expect(1, 128884, '\p{^Is_Line_Break=alphabetic}', "");
Expect(1, 128884, '\P{Is_Line_Break=alphabetic}', "");
Expect(0, 128884, '\P{^Is_Line_Break=alphabetic}', "");
Expect(1, 128883, '\p{Is_Line_Break= 	alphabetic}', "");
Expect(0, 128883, '\p{^Is_Line_Break= 	alphabetic}', "");
Expect(0, 128883, '\P{Is_Line_Break= 	alphabetic}', "");
Expect(1, 128883, '\P{^Is_Line_Break= 	alphabetic}', "");
Expect(0, 128884, '\p{Is_Line_Break= 	alphabetic}', "");
Expect(1, 128884, '\p{^Is_Line_Break= 	alphabetic}', "");
Expect(1, 128884, '\P{Is_Line_Break= 	alphabetic}', "");
Expect(0, 128884, '\P{^Is_Line_Break= 	alphabetic}', "");
Error('\p{Is_Lb=/a/-	al}');
Error('\P{Is_Lb=/a/-	al}');
Expect(1, 128883, '\p{Is_Lb=al}', "");
Expect(0, 128883, '\p{^Is_Lb=al}', "");
Expect(0, 128883, '\P{Is_Lb=al}', "");
Expect(1, 128883, '\P{^Is_Lb=al}', "");
Expect(0, 128884, '\p{Is_Lb=al}', "");
Expect(1, 128884, '\p{^Is_Lb=al}', "");
Expect(1, 128884, '\P{Is_Lb=al}', "");
Expect(0, 128884, '\P{^Is_Lb=al}', "");
Expect(1, 128883, '\p{Is_Lb=-AL}', "");
Expect(0, 128883, '\p{^Is_Lb=-AL}', "");
Expect(0, 128883, '\P{Is_Lb=-AL}', "");
Expect(1, 128883, '\P{^Is_Lb=-AL}', "");
Expect(0, 128884, '\p{Is_Lb=-AL}', "");
Expect(1, 128884, '\p{^Is_Lb=-AL}', "");
Expect(1, 128884, '\P{Is_Lb=-AL}', "");
Expect(0, 128884, '\P{^Is_Lb=-AL}', "");
Error('\p{Line_Break=/a/_POSTFIX_Numeric}');
Error('\P{Line_Break=/a/_POSTFIX_Numeric}');
Expect(1, 65504, '\p{Line_Break=postfixnumeric}', "");
Expect(0, 65504, '\p{^Line_Break=postfixnumeric}', "");
Expect(0, 65504, '\P{Line_Break=postfixnumeric}', "");
Expect(1, 65504, '\P{^Line_Break=postfixnumeric}', "");
Expect(0, 65505, '\p{Line_Break=postfixnumeric}', "");
Expect(1, 65505, '\p{^Line_Break=postfixnumeric}', "");
Expect(1, 65505, '\P{Line_Break=postfixnumeric}', "");
Expect(0, 65505, '\P{^Line_Break=postfixnumeric}', "");
Expect(1, 65504, '\p{Line_Break:   _ Postfix_Numeric}', "");
Expect(0, 65504, '\p{^Line_Break:   _ Postfix_Numeric}', "");
Expect(0, 65504, '\P{Line_Break:   _ Postfix_Numeric}', "");
Expect(1, 65504, '\P{^Line_Break:   _ Postfix_Numeric}', "");
Expect(0, 65505, '\p{Line_Break:   _ Postfix_Numeric}', "");
Expect(1, 65505, '\p{^Line_Break:   _ Postfix_Numeric}', "");
Expect(1, 65505, '\P{Line_Break:   _ Postfix_Numeric}', "");
Expect(0, 65505, '\P{^Line_Break:   _ Postfix_Numeric}', "");
Error('\p{Lb=	PO:=}');
Error('\P{Lb=	PO:=}');
Expect(1, 65504, '\p{Lb=po}', "");
Expect(0, 65504, '\p{^Lb=po}', "");
Expect(0, 65504, '\P{Lb=po}', "");
Expect(1, 65504, '\P{^Lb=po}', "");
Expect(0, 65505, '\p{Lb=po}', "");
Expect(1, 65505, '\p{^Lb=po}', "");
Expect(1, 65505, '\P{Lb=po}', "");
Expect(0, 65505, '\P{^Lb=po}', "");
Expect(1, 65504, '\p{Lb=_-PO}', "");
Expect(0, 65504, '\p{^Lb=_-PO}', "");
Expect(0, 65504, '\P{Lb=_-PO}', "");
Expect(1, 65504, '\P{^Lb=_-PO}', "");
Expect(0, 65505, '\p{Lb=_-PO}', "");
Expect(1, 65505, '\p{^Lb=_-PO}', "");
Expect(1, 65505, '\P{Lb=_-PO}', "");
Expect(0, 65505, '\P{^Lb=_-PO}', "");
Error('\p{Is_Line_Break=:=  Postfix_NUMERIC}');
Error('\P{Is_Line_Break=:=  Postfix_NUMERIC}');
Expect(1, 65504, '\p{Is_Line_Break=postfixnumeric}', "");
Expect(0, 65504, '\p{^Is_Line_Break=postfixnumeric}', "");
Expect(0, 65504, '\P{Is_Line_Break=postfixnumeric}', "");
Expect(1, 65504, '\P{^Is_Line_Break=postfixnumeric}', "");
Expect(0, 65505, '\p{Is_Line_Break=postfixnumeric}', "");
Expect(1, 65505, '\p{^Is_Line_Break=postfixnumeric}', "");
Expect(1, 65505, '\P{Is_Line_Break=postfixnumeric}', "");
Expect(0, 65505, '\P{^Is_Line_Break=postfixnumeric}', "");
Expect(1, 65504, '\p{Is_Line_Break=- Postfix_NUMERIC}', "");
Expect(0, 65504, '\p{^Is_Line_Break=- Postfix_NUMERIC}', "");
Expect(0, 65504, '\P{Is_Line_Break=- Postfix_NUMERIC}', "");
Expect(1, 65504, '\P{^Is_Line_Break=- Postfix_NUMERIC}', "");
Expect(0, 65505, '\p{Is_Line_Break=- Postfix_NUMERIC}', "");
Expect(1, 65505, '\p{^Is_Line_Break=- Postfix_NUMERIC}', "");
Expect(1, 65505, '\P{Is_Line_Break=- Postfix_NUMERIC}', "");
Expect(0, 65505, '\P{^Is_Line_Break=- Postfix_NUMERIC}', "");
Error('\p{Is_Lb=--PO:=}');
Error('\P{Is_Lb=--PO:=}');
Expect(1, 65504, '\p{Is_Lb=po}', "");
Expect(0, 65504, '\p{^Is_Lb=po}', "");
Expect(0, 65504, '\P{Is_Lb=po}', "");
Expect(1, 65504, '\P{^Is_Lb=po}', "");
Expect(0, 65505, '\p{Is_Lb=po}', "");
Expect(1, 65505, '\p{^Is_Lb=po}', "");
Expect(1, 65505, '\P{Is_Lb=po}', "");
Expect(0, 65505, '\P{^Is_Lb=po}', "");
Expect(1, 65504, '\p{Is_Lb=--po}', "");
Expect(0, 65504, '\p{^Is_Lb=--po}', "");
Expect(0, 65504, '\P{Is_Lb=--po}', "");
Expect(1, 65504, '\P{^Is_Lb=--po}', "");
Expect(0, 65505, '\p{Is_Lb=--po}', "");
Expect(1, 65505, '\p{^Is_Lb=--po}', "");
Expect(1, 65505, '\P{Is_Lb=--po}', "");
Expect(0, 65505, '\P{^Is_Lb=--po}', "");
Error('\p{Line_Break=-:=OPEN_Punctuation}');
Error('\P{Line_Break=-:=OPEN_Punctuation}');
Expect(1, 78713, '\p{Line_Break=openpunctuation}', "");
Expect(0, 78713, '\p{^Line_Break=openpunctuation}', "");
Expect(0, 78713, '\P{Line_Break=openpunctuation}', "");
Expect(1, 78713, '\P{^Line_Break=openpunctuation}', "");
Expect(0, 78714, '\p{Line_Break=openpunctuation}', "");
Expect(1, 78714, '\p{^Line_Break=openpunctuation}', "");
Expect(1, 78714, '\P{Line_Break=openpunctuation}', "");
Expect(0, 78714, '\P{^Line_Break=openpunctuation}', "");
Expect(1, 78713, '\p{Line_Break=	OPEN_Punctuation}', "");
Expect(0, 78713, '\p{^Line_Break=	OPEN_Punctuation}', "");
Expect(0, 78713, '\P{Line_Break=	OPEN_Punctuation}', "");
Expect(1, 78713, '\P{^Line_Break=	OPEN_Punctuation}', "");
Expect(0, 78714, '\p{Line_Break=	OPEN_Punctuation}', "");
Expect(1, 78714, '\p{^Line_Break=	OPEN_Punctuation}', "");
Expect(1, 78714, '\P{Line_Break=	OPEN_Punctuation}', "");
Expect(0, 78714, '\P{^Line_Break=	OPEN_Punctuation}', "");
Error('\p{Lb=/a/ op}');
Error('\P{Lb=/a/ op}');
Expect(1, 78713, '\p{Lb=op}', "");
Expect(0, 78713, '\p{^Lb=op}', "");
Expect(0, 78713, '\P{Lb=op}', "");
Expect(1, 78713, '\P{^Lb=op}', "");
Expect(0, 78714, '\p{Lb=op}', "");
Expect(1, 78714, '\p{^Lb=op}', "");
Expect(1, 78714, '\P{Lb=op}', "");
Expect(0, 78714, '\P{^Lb=op}', "");
Expect(1, 78713, '\p{Lb=_OP}', "");
Expect(0, 78713, '\p{^Lb=_OP}', "");
Expect(0, 78713, '\P{Lb=_OP}', "");
Expect(1, 78713, '\P{^Lb=_OP}', "");
Expect(0, 78714, '\p{Lb=_OP}', "");
Expect(1, 78714, '\p{^Lb=_OP}', "");
Expect(1, 78714, '\P{Lb=_OP}', "");
Expect(0, 78714, '\P{^Lb=_OP}', "");
Error('\p{Is_Line_Break=open_PUNCTUATION:=}');
Error('\P{Is_Line_Break=open_PUNCTUATION:=}');
Expect(1, 78713, '\p{Is_Line_Break=openpunctuation}', "");
Expect(0, 78713, '\p{^Is_Line_Break=openpunctuation}', "");
Expect(0, 78713, '\P{Is_Line_Break=openpunctuation}', "");
Expect(1, 78713, '\P{^Is_Line_Break=openpunctuation}', "");
Expect(0, 78714, '\p{Is_Line_Break=openpunctuation}', "");
Expect(1, 78714, '\p{^Is_Line_Break=openpunctuation}', "");
Expect(1, 78714, '\P{Is_Line_Break=openpunctuation}', "");
Expect(0, 78714, '\P{^Is_Line_Break=openpunctuation}', "");
Expect(1, 78713, '\p{Is_Line_Break=	 Open_punctuation}', "");
Expect(0, 78713, '\p{^Is_Line_Break=	 Open_punctuation}', "");
Expect(0, 78713, '\P{Is_Line_Break=	 Open_punctuation}', "");
Expect(1, 78713, '\P{^Is_Line_Break=	 Open_punctuation}', "");
Expect(0, 78714, '\p{Is_Line_Break=	 Open_punctuation}', "");
Expect(1, 78714, '\p{^Is_Line_Break=	 Open_punctuation}', "");
Expect(1, 78714, '\P{Is_Line_Break=	 Open_punctuation}', "");
Expect(0, 78714, '\P{^Is_Line_Break=	 Open_punctuation}', "");
Error('\p{Is_Lb=/a/op}');
Error('\P{Is_Lb=/a/op}');
Expect(1, 78713, '\p{Is_Lb=op}', "");
Expect(0, 78713, '\p{^Is_Lb=op}', "");
Expect(0, 78713, '\P{Is_Lb=op}', "");
Expect(1, 78713, '\P{^Is_Lb=op}', "");
Expect(0, 78714, '\p{Is_Lb=op}', "");
Expect(1, 78714, '\p{^Is_Lb=op}', "");
Expect(1, 78714, '\P{Is_Lb=op}', "");
Expect(0, 78714, '\P{^Is_Lb=op}', "");
Expect(1, 78713, '\p{Is_Lb=_op}', "");
Expect(0, 78713, '\p{^Is_Lb=_op}', "");
Expect(0, 78713, '\P{Is_Lb=_op}', "");
Expect(1, 78713, '\P{^Is_Lb=_op}', "");
Expect(0, 78714, '\p{Is_Lb=_op}', "");
Expect(1, 78714, '\p{^Is_Lb=_op}', "");
Expect(1, 78714, '\P{Is_Lb=_op}', "");
Expect(0, 78714, '\P{^Is_Lb=_op}', "");
Error('\p{Line_Break= NEXT_Line/a/}');
Error('\P{Line_Break= NEXT_Line/a/}');
Expect(1, 133, '\p{Line_Break=nextline}', "");
Expect(0, 133, '\p{^Line_Break=nextline}', "");
Expect(0, 133, '\P{Line_Break=nextline}', "");
Expect(1, 133, '\P{^Line_Break=nextline}', "");
Expect(0, 134, '\p{Line_Break=nextline}', "");
Expect(1, 134, '\p{^Line_Break=nextline}', "");
Expect(1, 134, '\P{Line_Break=nextline}', "");
Expect(0, 134, '\P{^Line_Break=nextline}', "");
Expect(1, 133, '\p{Line_Break=	NEXT_Line}', "");
Expect(0, 133, '\p{^Line_Break=	NEXT_Line}', "");
Expect(0, 133, '\P{Line_Break=	NEXT_Line}', "");
Expect(1, 133, '\P{^Line_Break=	NEXT_Line}', "");
Expect(0, 134, '\p{Line_Break=	NEXT_Line}', "");
Expect(1, 134, '\p{^Line_Break=	NEXT_Line}', "");
Expect(1, 134, '\P{Line_Break=	NEXT_Line}', "");
Expect(0, 134, '\P{^Line_Break=	NEXT_Line}', "");
Error('\p{Lb=-NL:=}');
Error('\P{Lb=-NL:=}');
Expect(1, 133, '\p{Lb=nl}', "");
Expect(0, 133, '\p{^Lb=nl}', "");
Expect(0, 133, '\P{Lb=nl}', "");
Expect(1, 133, '\P{^Lb=nl}', "");
Expect(0, 134, '\p{Lb=nl}', "");
Expect(1, 134, '\p{^Lb=nl}', "");
Expect(1, 134, '\P{Lb=nl}', "");
Expect(0, 134, '\P{^Lb=nl}', "");
Expect(1, 133, '\p{Lb=-NL}', "");
Expect(0, 133, '\p{^Lb=-NL}', "");
Expect(0, 133, '\P{Lb=-NL}', "");
Expect(1, 133, '\P{^Lb=-NL}', "");
Expect(0, 134, '\p{Lb=-NL}', "");
Expect(1, 134, '\p{^Lb=-NL}', "");
Expect(1, 134, '\P{Lb=-NL}', "");
Expect(0, 134, '\P{^Lb=-NL}', "");
Error('\p{Is_Line_Break= -Next_line/a/}');
Error('\P{Is_Line_Break= -Next_line/a/}');
Expect(1, 133, '\p{Is_Line_Break=nextline}', "");
Expect(0, 133, '\p{^Is_Line_Break=nextline}', "");
Expect(0, 133, '\P{Is_Line_Break=nextline}', "");
Expect(1, 133, '\P{^Is_Line_Break=nextline}', "");
Expect(0, 134, '\p{Is_Line_Break=nextline}', "");
Expect(1, 134, '\p{^Is_Line_Break=nextline}', "");
Expect(1, 134, '\P{Is_Line_Break=nextline}', "");
Expect(0, 134, '\P{^Is_Line_Break=nextline}', "");
Expect(1, 133, '\p{Is_Line_Break=_	next_line}', "");
Expect(0, 133, '\p{^Is_Line_Break=_	next_line}', "");
Expect(0, 133, '\P{Is_Line_Break=_	next_line}', "");
Expect(1, 133, '\P{^Is_Line_Break=_	next_line}', "");
Expect(0, 134, '\p{Is_Line_Break=_	next_line}', "");
Expect(1, 134, '\p{^Is_Line_Break=_	next_line}', "");
Expect(1, 134, '\P{Is_Line_Break=_	next_line}', "");
Expect(0, 134, '\P{^Is_Line_Break=_	next_line}', "");
Error('\p{Is_Lb=_ NL/a/}');
Error('\P{Is_Lb=_ NL/a/}');
Expect(1, 133, '\p{Is_Lb=nl}', "");
Expect(0, 133, '\p{^Is_Lb=nl}', "");
Expect(0, 133, '\P{Is_Lb=nl}', "");
Expect(1, 133, '\P{^Is_Lb=nl}', "");
Expect(0, 134, '\p{Is_Lb=nl}', "");
Expect(1, 134, '\p{^Is_Lb=nl}', "");
Expect(1, 134, '\P{Is_Lb=nl}', "");
Expect(0, 134, '\P{^Is_Lb=nl}', "");
Expect(1, 133, '\p{Is_Lb= 	NL}', "");
Expect(0, 133, '\p{^Is_Lb= 	NL}', "");
Expect(0, 133, '\P{Is_Lb= 	NL}', "");
Expect(1, 133, '\P{^Is_Lb= 	NL}', "");
Expect(0, 134, '\p{Is_Lb= 	NL}', "");
Expect(1, 134, '\p{^Is_Lb= 	NL}', "");
Expect(1, 134, '\P{Is_Lb= 	NL}', "");
Expect(0, 134, '\P{^Is_Lb= 	NL}', "");
Error('\p{Line_Break= :=JT}');
Error('\P{Line_Break= :=JT}');
Expect(1, 55291, '\p{Line_Break=jt}', "");
Expect(0, 55291, '\p{^Line_Break=jt}', "");
Expect(0, 55291, '\P{Line_Break=jt}', "");
Expect(1, 55291, '\P{^Line_Break=jt}', "");
Expect(0, 55292, '\p{Line_Break=jt}', "");
Expect(1, 55292, '\p{^Line_Break=jt}', "");
Expect(1, 55292, '\P{Line_Break=jt}', "");
Expect(0, 55292, '\P{^Line_Break=jt}', "");
Expect(1, 55291, '\p{Line_Break=		jt}', "");
Expect(0, 55291, '\p{^Line_Break=		jt}', "");
Expect(0, 55291, '\P{Line_Break=		jt}', "");
Expect(1, 55291, '\P{^Line_Break=		jt}', "");
Expect(0, 55292, '\p{Line_Break=		jt}', "");
Expect(1, 55292, '\p{^Line_Break=		jt}', "");
Expect(1, 55292, '\P{Line_Break=		jt}', "");
Expect(0, 55292, '\P{^Line_Break=		jt}', "");
Error('\p{Lb:		/a/JT}');
Error('\P{Lb:		/a/JT}');
Expect(1, 55291, '\p{Lb=jt}', "");
Expect(0, 55291, '\p{^Lb=jt}', "");
Expect(0, 55291, '\P{Lb=jt}', "");
Expect(1, 55291, '\P{^Lb=jt}', "");
Expect(0, 55292, '\p{Lb=jt}', "");
Expect(1, 55292, '\p{^Lb=jt}', "");
Expect(1, 55292, '\P{Lb=jt}', "");
Expect(0, 55292, '\P{^Lb=jt}', "");
Expect(1, 55291, '\p{Lb=	 JT}', "");
Expect(0, 55291, '\p{^Lb=	 JT}', "");
Expect(0, 55291, '\P{Lb=	 JT}', "");
Expect(1, 55291, '\P{^Lb=	 JT}', "");
Expect(0, 55292, '\p{Lb=	 JT}', "");
Expect(1, 55292, '\p{^Lb=	 JT}', "");
Expect(1, 55292, '\P{Lb=	 JT}', "");
Expect(0, 55292, '\P{^Lb=	 JT}', "");
Error('\p{Is_Line_Break:   := -JT}');
Error('\P{Is_Line_Break:   := -JT}');
Expect(1, 55291, '\p{Is_Line_Break:   jt}', "");
Expect(0, 55291, '\p{^Is_Line_Break:   jt}', "");
Expect(0, 55291, '\P{Is_Line_Break:   jt}', "");
Expect(1, 55291, '\P{^Is_Line_Break:   jt}', "");
Expect(0, 55292, '\p{Is_Line_Break:   jt}', "");
Expect(1, 55292, '\p{^Is_Line_Break:   jt}', "");
Expect(1, 55292, '\P{Is_Line_Break:   jt}', "");
Expect(0, 55292, '\P{^Is_Line_Break:   jt}', "");
Expect(1, 55291, '\p{Is_Line_Break= -JT}', "");
Expect(0, 55291, '\p{^Is_Line_Break= -JT}', "");
Expect(0, 55291, '\P{Is_Line_Break= -JT}', "");
Expect(1, 55291, '\P{^Is_Line_Break= -JT}', "");
Expect(0, 55292, '\p{Is_Line_Break= -JT}', "");
Expect(1, 55292, '\p{^Is_Line_Break= -JT}', "");
Expect(1, 55292, '\P{Is_Line_Break= -JT}', "");
Expect(0, 55292, '\P{^Is_Line_Break= -JT}', "");
Error('\p{Is_Lb= _JT:=}');
Error('\P{Is_Lb= _JT:=}');
Expect(1, 55291, '\p{Is_Lb=jt}', "");
Expect(0, 55291, '\p{^Is_Lb=jt}', "");
Expect(0, 55291, '\P{Is_Lb=jt}', "");
Expect(1, 55291, '\P{^Is_Lb=jt}', "");
Expect(0, 55292, '\p{Is_Lb=jt}', "");
Expect(1, 55292, '\p{^Is_Lb=jt}', "");
Expect(1, 55292, '\P{Is_Lb=jt}', "");
Expect(0, 55292, '\P{^Is_Lb=jt}', "");
Expect(1, 55291, '\p{Is_Lb=-jt}', "");
Expect(0, 55291, '\p{^Is_Lb=-jt}', "");
Expect(0, 55291, '\P{Is_Lb=-jt}', "");
Expect(1, 55291, '\P{^Is_Lb=-jt}', "");
Expect(0, 55292, '\p{Is_Lb=-jt}', "");
Expect(1, 55292, '\p{^Is_Lb=-jt}', "");
Expect(1, 55292, '\P{Is_Lb=-jt}', "");
Expect(0, 55292, '\P{^Is_Lb=-jt}', "");
Error('\p{Terminal_Punctuation=/a/- Yes}');
Error('\P{Terminal_Punctuation=/a/- Yes}');
Expect(1, 74867, '\p{Terminal_Punctuation=yes}', "");
Expect(0, 74867, '\p{^Terminal_Punctuation=yes}', "");
Expect(0, 74867, '\P{Terminal_Punctuation=yes}', "");
Expect(1, 74867, '\P{^Terminal_Punctuation=yes}', "");
Expect(0, 74868, '\p{Terminal_Punctuation=yes}', "");
Expect(1, 74868, '\p{^Terminal_Punctuation=yes}', "");
Expect(1, 74868, '\P{Terminal_Punctuation=yes}', "");
Expect(0, 74868, '\P{^Terminal_Punctuation=yes}', "");
Expect(1, 74867, '\p{Terminal_Punctuation=	-Yes}', "");
Expect(0, 74867, '\p{^Terminal_Punctuation=	-Yes}', "");
Expect(0, 74867, '\P{Terminal_Punctuation=	-Yes}', "");
Expect(1, 74867, '\P{^Terminal_Punctuation=	-Yes}', "");
Expect(0, 74868, '\p{Terminal_Punctuation=	-Yes}', "");
Expect(1, 74868, '\p{^Terminal_Punctuation=	-Yes}', "");
Expect(1, 74868, '\P{Terminal_Punctuation=	-Yes}', "");
Expect(0, 74868, '\P{^Terminal_Punctuation=	-Yes}', "");
Error('\p{Term=	Y:=}');
Error('\P{Term=	Y:=}');
Expect(1, 74867, '\p{Term=y}', "");
Expect(0, 74867, '\p{^Term=y}', "");
Expect(0, 74867, '\P{Term=y}', "");
Expect(1, 74867, '\P{^Term=y}', "");
Expect(0, 74868, '\p{Term=y}', "");
Expect(1, 74868, '\p{^Term=y}', "");
Expect(1, 74868, '\P{Term=y}', "");
Expect(0, 74868, '\P{^Term=y}', "");
Expect(1, 74867, '\p{Term=__Y}', "");
Expect(0, 74867, '\p{^Term=__Y}', "");
Expect(0, 74867, '\P{Term=__Y}', "");
Expect(1, 74867, '\P{^Term=__Y}', "");
Expect(0, 74868, '\p{Term=__Y}', "");
Expect(1, 74868, '\p{^Term=__Y}', "");
Expect(1, 74868, '\P{Term=__Y}', "");
Expect(0, 74868, '\P{^Term=__Y}', "");
Error('\p{Is_Terminal_Punctuation=	-t/a/}');
Error('\P{Is_Terminal_Punctuation=	-t/a/}');
Expect(1, 74867, '\p{Is_Terminal_Punctuation=t}', "");
Expect(0, 74867, '\p{^Is_Terminal_Punctuation=t}', "");
Expect(0, 74867, '\P{Is_Terminal_Punctuation=t}', "");
Expect(1, 74867, '\P{^Is_Terminal_Punctuation=t}', "");
Expect(0, 74868, '\p{Is_Terminal_Punctuation=t}', "");
Expect(1, 74868, '\p{^Is_Terminal_Punctuation=t}', "");
Expect(1, 74868, '\P{Is_Terminal_Punctuation=t}', "");
Expect(0, 74868, '\P{^Is_Terminal_Punctuation=t}', "");
Expect(1, 74867, '\p{Is_Terminal_Punctuation=_t}', "");
Expect(0, 74867, '\p{^Is_Terminal_Punctuation=_t}', "");
Expect(0, 74867, '\P{Is_Terminal_Punctuation=_t}', "");
Expect(1, 74867, '\P{^Is_Terminal_Punctuation=_t}', "");
Expect(0, 74868, '\p{Is_Terminal_Punctuation=_t}', "");
Expect(1, 74868, '\p{^Is_Terminal_Punctuation=_t}', "");
Expect(1, 74868, '\P{Is_Terminal_Punctuation=_t}', "");
Expect(0, 74868, '\P{^Is_Terminal_Punctuation=_t}', "");
Error('\p{Is_Term=	_true:=}');
Error('\P{Is_Term=	_true:=}');
Expect(1, 74867, '\p{Is_Term=true}', "");
Expect(0, 74867, '\p{^Is_Term=true}', "");
Expect(0, 74867, '\P{Is_Term=true}', "");
Expect(1, 74867, '\P{^Is_Term=true}', "");
Expect(0, 74868, '\p{Is_Term=true}', "");
Expect(1, 74868, '\p{^Is_Term=true}', "");
Expect(1, 74868, '\P{Is_Term=true}', "");
Expect(0, 74868, '\P{^Is_Term=true}', "");
Expect(1, 74867, '\p{Is_Term:_	true}', "");
Expect(0, 74867, '\p{^Is_Term:_	true}', "");
Expect(0, 74867, '\P{Is_Term:_	true}', "");
Expect(1, 74867, '\P{^Is_Term:_	true}', "");
Expect(0, 74868, '\p{Is_Term:_	true}', "");
Expect(1, 74868, '\p{^Is_Term:_	true}', "");
Expect(1, 74868, '\P{Is_Term:_	true}', "");
Expect(0, 74868, '\P{^Is_Term:_	true}', "");
Error('\p{Terminal_Punctuation:   :=	No}');
Error('\P{Terminal_Punctuation:   :=	No}');
Expect(1, 74868, '\p{Terminal_Punctuation=no}', "");
Expect(0, 74868, '\p{^Terminal_Punctuation=no}', "");
Expect(0, 74868, '\P{Terminal_Punctuation=no}', "");
Expect(1, 74868, '\P{^Terminal_Punctuation=no}', "");
Expect(0, 74867, '\p{Terminal_Punctuation=no}', "");
Expect(1, 74867, '\p{^Terminal_Punctuation=no}', "");
Expect(1, 74867, '\P{Terminal_Punctuation=no}', "");
Expect(0, 74867, '\P{^Terminal_Punctuation=no}', "");
Expect(1, 74868, '\p{Terminal_Punctuation=	No}', "");
Expect(0, 74868, '\p{^Terminal_Punctuation=	No}', "");
Expect(0, 74868, '\P{Terminal_Punctuation=	No}', "");
Expect(1, 74868, '\P{^Terminal_Punctuation=	No}', "");
Expect(0, 74867, '\p{Terminal_Punctuation=	No}', "");
Expect(1, 74867, '\p{^Terminal_Punctuation=	No}', "");
Expect(1, 74867, '\P{Terminal_Punctuation=	No}', "");
Expect(0, 74867, '\P{^Terminal_Punctuation=	No}', "");
Error('\p{Term=		N:=}');
Error('\P{Term=		N:=}');
Expect(1, 74868, '\p{Term:n}', "");
Expect(0, 74868, '\p{^Term:n}', "");
Expect(0, 74868, '\P{Term:n}', "");
Expect(1, 74868, '\P{^Term:n}', "");
Expect(0, 74867, '\p{Term:n}', "");
Expect(1, 74867, '\p{^Term:n}', "");
Expect(1, 74867, '\P{Term:n}', "");
Expect(0, 74867, '\P{^Term:n}', "");
Expect(1, 74868, '\p{Term=	n}', "");
Expect(0, 74868, '\p{^Term=	n}', "");
Expect(0, 74868, '\P{Term=	n}', "");
Expect(1, 74868, '\P{^Term=	n}', "");
Expect(0, 74867, '\p{Term=	n}', "");
Expect(1, 74867, '\p{^Term=	n}', "");
Expect(1, 74867, '\P{Term=	n}', "");
Expect(0, 74867, '\P{^Term=	n}', "");
Error('\p{Is_Terminal_Punctuation: /a/F}');
Error('\P{Is_Terminal_Punctuation: /a/F}');
Expect(1, 74868, '\p{Is_Terminal_Punctuation=f}', "");
Expect(0, 74868, '\p{^Is_Terminal_Punctuation=f}', "");
Expect(0, 74868, '\P{Is_Terminal_Punctuation=f}', "");
Expect(1, 74868, '\P{^Is_Terminal_Punctuation=f}', "");
Expect(0, 74867, '\p{Is_Terminal_Punctuation=f}', "");
Expect(1, 74867, '\p{^Is_Terminal_Punctuation=f}', "");
Expect(1, 74867, '\P{Is_Terminal_Punctuation=f}', "");
Expect(0, 74867, '\P{^Is_Terminal_Punctuation=f}', "");
Expect(1, 74868, '\p{Is_Terminal_Punctuation=	f}', "");
Expect(0, 74868, '\p{^Is_Terminal_Punctuation=	f}', "");
Expect(0, 74868, '\P{Is_Terminal_Punctuation=	f}', "");
Expect(1, 74868, '\P{^Is_Terminal_Punctuation=	f}', "");
Expect(0, 74867, '\p{Is_Terminal_Punctuation=	f}', "");
Expect(1, 74867, '\p{^Is_Terminal_Punctuation=	f}', "");
Expect(1, 74867, '\P{Is_Terminal_Punctuation=	f}', "");
Expect(0, 74867, '\P{^Is_Terminal_Punctuation=	f}', "");
Error('\p{Is_Term=/a/ false}');
Error('\P{Is_Term=/a/ false}');
Expect(1, 74868, '\p{Is_Term=false}', "");
Expect(0, 74868, '\p{^Is_Term=false}', "");
Expect(0, 74868, '\P{Is_Term=false}', "");
Expect(1, 74868, '\P{^Is_Term=false}', "");
Expect(0, 74867, '\p{Is_Term=false}', "");
Expect(1, 74867, '\p{^Is_Term=false}', "");
Expect(1, 74867, '\P{Is_Term=false}', "");
Expect(0, 74867, '\P{^Is_Term=false}', "");
Expect(1, 74868, '\p{Is_Term=_FALSE}', "");
Expect(0, 74868, '\p{^Is_Term=_FALSE}', "");
Expect(0, 74868, '\P{Is_Term=_FALSE}', "");
Expect(1, 74868, '\P{^Is_Term=_FALSE}', "");
Expect(0, 74867, '\p{Is_Term=_FALSE}', "");
Expect(1, 74867, '\p{^Is_Term=_FALSE}', "");
Expect(1, 74867, '\P{Is_Term=_FALSE}', "");
Expect(0, 74867, '\P{^Is_Term=_FALSE}', "");
Error('\p{Expands_On_NFKD=Yes}');
Error('\P{Expands_On_NFKD=Yes}');
Error('\p{XO_NFKD=Y}');
Error('\P{XO_NFKD=Y}');
Error('\p{Is_Expands_On_NFKD=T}');
Error('\P{Is_Expands_On_NFKD=T}');
Error('\p{Is_XO_NFKD=True}');
Error('\P{Is_XO_NFKD=True}');
Error('\p{Expands_On_NFKD=No}');
Error('\P{Expands_On_NFKD=No}');
Error('\p{XO_NFKD=N}');
Error('\P{XO_NFKD=N}');
Error('\p{Is_Expands_On_NFKD=F}');
Error('\P{Is_Expands_On_NFKD=F}');
Error('\p{Is_XO_NFKD: False}');
Error('\P{Is_XO_NFKD: False}');
Error('\p{Pattern_Syntax=:=_ YES}');
Error('\P{Pattern_Syntax=:=_ YES}');
Expect(1, 65094, '\p{Pattern_Syntax=yes}', "");
Expect(0, 65094, '\p{^Pattern_Syntax=yes}', "");
Expect(0, 65094, '\P{Pattern_Syntax=yes}', "");
Expect(1, 65094, '\P{^Pattern_Syntax=yes}', "");
Expect(0, 65095, '\p{Pattern_Syntax=yes}', "");
Expect(1, 65095, '\p{^Pattern_Syntax=yes}', "");
Expect(1, 65095, '\P{Pattern_Syntax=yes}', "");
Expect(0, 65095, '\P{^Pattern_Syntax=yes}', "");
Expect(1, 65094, '\p{Pattern_Syntax=_-Yes}', "");
Expect(0, 65094, '\p{^Pattern_Syntax=_-Yes}', "");
Expect(0, 65094, '\P{Pattern_Syntax=_-Yes}', "");
Expect(1, 65094, '\P{^Pattern_Syntax=_-Yes}', "");
Expect(0, 65095, '\p{Pattern_Syntax=_-Yes}', "");
Expect(1, 65095, '\p{^Pattern_Syntax=_-Yes}', "");
Expect(1, 65095, '\P{Pattern_Syntax=_-Yes}', "");
Expect(0, 65095, '\P{^Pattern_Syntax=_-Yes}', "");
Error('\p{Pat_Syn=_	Y/a/}');
Error('\P{Pat_Syn=_	Y/a/}');
Expect(1, 65094, '\p{Pat_Syn=y}', "");
Expect(0, 65094, '\p{^Pat_Syn=y}', "");
Expect(0, 65094, '\P{Pat_Syn=y}', "");
Expect(1, 65094, '\P{^Pat_Syn=y}', "");
Expect(0, 65095, '\p{Pat_Syn=y}', "");
Expect(1, 65095, '\p{^Pat_Syn=y}', "");
Expect(1, 65095, '\P{Pat_Syn=y}', "");
Expect(0, 65095, '\P{^Pat_Syn=y}', "");
Expect(1, 65094, '\p{Pat_Syn=-_Y}', "");
Expect(0, 65094, '\p{^Pat_Syn=-_Y}', "");
Expect(0, 65094, '\P{Pat_Syn=-_Y}', "");
Expect(1, 65094, '\P{^Pat_Syn=-_Y}', "");
Expect(0, 65095, '\p{Pat_Syn=-_Y}', "");
Expect(1, 65095, '\p{^Pat_Syn=-_Y}', "");
Expect(1, 65095, '\P{Pat_Syn=-_Y}', "");
Expect(0, 65095, '\P{^Pat_Syn=-_Y}', "");
Error('\p{Is_Pattern_Syntax=	_t:=}');
Error('\P{Is_Pattern_Syntax=	_t:=}');
Expect(1, 65094, '\p{Is_Pattern_Syntax:	t}', "");
Expect(0, 65094, '\p{^Is_Pattern_Syntax:	t}', "");
Expect(0, 65094, '\P{Is_Pattern_Syntax:	t}', "");
Expect(1, 65094, '\P{^Is_Pattern_Syntax:	t}', "");
Expect(0, 65095, '\p{Is_Pattern_Syntax:	t}', "");
Expect(1, 65095, '\p{^Is_Pattern_Syntax:	t}', "");
Expect(1, 65095, '\P{Is_Pattern_Syntax:	t}', "");
Expect(0, 65095, '\P{^Is_Pattern_Syntax:	t}', "");
Expect(1, 65094, '\p{Is_Pattern_Syntax=  T}', "");
Expect(0, 65094, '\p{^Is_Pattern_Syntax=  T}', "");
Expect(0, 65094, '\P{Is_Pattern_Syntax=  T}', "");
Expect(1, 65094, '\P{^Is_Pattern_Syntax=  T}', "");
Expect(0, 65095, '\p{Is_Pattern_Syntax=  T}', "");
Expect(1, 65095, '\p{^Is_Pattern_Syntax=  T}', "");
Expect(1, 65095, '\P{Is_Pattern_Syntax=  T}', "");
Expect(0, 65095, '\P{^Is_Pattern_Syntax=  T}', "");
Error('\p{Is_Pat_Syn=/a/	True}');
Error('\P{Is_Pat_Syn=/a/	True}');
Expect(1, 65094, '\p{Is_Pat_Syn=true}', "");
Expect(0, 65094, '\p{^Is_Pat_Syn=true}', "");
Expect(0, 65094, '\P{Is_Pat_Syn=true}', "");
Expect(1, 65094, '\P{^Is_Pat_Syn=true}', "");
Expect(0, 65095, '\p{Is_Pat_Syn=true}', "");
Expect(1, 65095, '\p{^Is_Pat_Syn=true}', "");
Expect(1, 65095, '\P{Is_Pat_Syn=true}', "");
Expect(0, 65095, '\P{^Is_Pat_Syn=true}', "");
Expect(1, 65094, '\p{Is_Pat_Syn=- True}', "");
Expect(0, 65094, '\p{^Is_Pat_Syn=- True}', "");
Expect(0, 65094, '\P{Is_Pat_Syn=- True}', "");
Expect(1, 65094, '\P{^Is_Pat_Syn=- True}', "");
Expect(0, 65095, '\p{Is_Pat_Syn=- True}', "");
Expect(1, 65095, '\p{^Is_Pat_Syn=- True}', "");
Expect(1, 65095, '\P{Is_Pat_Syn=- True}', "");
Expect(0, 65095, '\P{^Is_Pat_Syn=- True}', "");
Error('\p{Pattern_Syntax=		No/a/}');
Error('\P{Pattern_Syntax=		No/a/}');
Expect(1, 65095, '\p{Pattern_Syntax:no}', "");
Expect(0, 65095, '\p{^Pattern_Syntax:no}', "");
Expect(0, 65095, '\P{Pattern_Syntax:no}', "");
Expect(1, 65095, '\P{^Pattern_Syntax:no}', "");
Expect(0, 65094, '\p{Pattern_Syntax:no}', "");
Expect(1, 65094, '\p{^Pattern_Syntax:no}', "");
Expect(1, 65094, '\P{Pattern_Syntax:no}', "");
Expect(0, 65094, '\P{^Pattern_Syntax:no}', "");
Expect(1, 65095, '\p{Pattern_Syntax=	no}', "");
Expect(0, 65095, '\p{^Pattern_Syntax=	no}', "");
Expect(0, 65095, '\P{Pattern_Syntax=	no}', "");
Expect(1, 65095, '\P{^Pattern_Syntax=	no}', "");
Expect(0, 65094, '\p{Pattern_Syntax=	no}', "");
Expect(1, 65094, '\p{^Pattern_Syntax=	no}', "");
Expect(1, 65094, '\P{Pattern_Syntax=	no}', "");
Expect(0, 65094, '\P{^Pattern_Syntax=	no}', "");
Error('\p{Pat_Syn=_	n:=}');
Error('\P{Pat_Syn=_	n:=}');
Expect(1, 65095, '\p{Pat_Syn:n}', "");
Expect(0, 65095, '\p{^Pat_Syn:n}', "");
Expect(0, 65095, '\P{Pat_Syn:n}', "");
Expect(1, 65095, '\P{^Pat_Syn:n}', "");
Expect(0, 65094, '\p{Pat_Syn:n}', "");
Expect(1, 65094, '\p{^Pat_Syn:n}', "");
Expect(1, 65094, '\P{Pat_Syn:n}', "");
Expect(0, 65094, '\P{^Pat_Syn:n}', "");
Expect(1, 65095, '\p{Pat_Syn= -N}', "");
Expect(0, 65095, '\p{^Pat_Syn= -N}', "");
Expect(0, 65095, '\P{Pat_Syn= -N}', "");
Expect(1, 65095, '\P{^Pat_Syn= -N}', "");
Expect(0, 65094, '\p{Pat_Syn= -N}', "");
Expect(1, 65094, '\p{^Pat_Syn= -N}', "");
Expect(1, 65094, '\P{Pat_Syn= -N}', "");
Expect(0, 65094, '\P{^Pat_Syn= -N}', "");
Error('\p{Is_Pattern_Syntax=/a/  F}');
Error('\P{Is_Pattern_Syntax=/a/  F}');
Expect(1, 65095, '\p{Is_Pattern_Syntax=f}', "");
Expect(0, 65095, '\p{^Is_Pattern_Syntax=f}', "");
Expect(0, 65095, '\P{Is_Pattern_Syntax=f}', "");
Expect(1, 65095, '\P{^Is_Pattern_Syntax=f}', "");
Expect(0, 65094, '\p{Is_Pattern_Syntax=f}', "");
Expect(1, 65094, '\p{^Is_Pattern_Syntax=f}', "");
Expect(1, 65094, '\P{Is_Pattern_Syntax=f}', "");
Expect(0, 65094, '\P{^Is_Pattern_Syntax=f}', "");
Expect(1, 65095, '\p{Is_Pattern_Syntax=  F}', "");
Expect(0, 65095, '\p{^Is_Pattern_Syntax=  F}', "");
Expect(0, 65095, '\P{Is_Pattern_Syntax=  F}', "");
Expect(1, 65095, '\P{^Is_Pattern_Syntax=  F}', "");
Expect(0, 65094, '\p{Is_Pattern_Syntax=  F}', "");
Expect(1, 65094, '\p{^Is_Pattern_Syntax=  F}', "");
Expect(1, 65094, '\P{Is_Pattern_Syntax=  F}', "");
Expect(0, 65094, '\P{^Is_Pattern_Syntax=  F}', "");
Error('\p{Is_Pat_Syn=:=	-false}');
Error('\P{Is_Pat_Syn=:=	-false}');
Expect(1, 65095, '\p{Is_Pat_Syn=false}', "");
Expect(0, 65095, '\p{^Is_Pat_Syn=false}', "");
Expect(0, 65095, '\P{Is_Pat_Syn=false}', "");
Expect(1, 65095, '\P{^Is_Pat_Syn=false}', "");
Expect(0, 65094, '\p{Is_Pat_Syn=false}', "");
Expect(1, 65094, '\p{^Is_Pat_Syn=false}', "");
Expect(1, 65094, '\P{Is_Pat_Syn=false}', "");
Expect(0, 65094, '\P{^Is_Pat_Syn=false}', "");
Expect(1, 65095, '\p{Is_Pat_Syn=-	False}', "");
Expect(0, 65095, '\p{^Is_Pat_Syn=-	False}', "");
Expect(0, 65095, '\P{Is_Pat_Syn=-	False}', "");
Expect(1, 65095, '\P{^Is_Pat_Syn=-	False}', "");
Expect(0, 65094, '\p{Is_Pat_Syn=-	False}', "");
Expect(1, 65094, '\p{^Is_Pat_Syn=-	False}', "");
Expect(1, 65094, '\P{Is_Pat_Syn=-	False}', "");
Expect(0, 65094, '\P{^Is_Pat_Syn=-	False}', "");
Error('\p{ID_Start=_/a/Yes}');
Error('\P{ID_Start=_/a/Yes}');
Expect(1, 195101, '\p{ID_Start=yes}', "");
Expect(0, 195101, '\p{^ID_Start=yes}', "");
Expect(0, 195101, '\P{ID_Start=yes}', "");
Expect(1, 195101, '\P{^ID_Start=yes}', "");
Expect(0, 195102, '\p{ID_Start=yes}', "");
Expect(1, 195102, '\p{^ID_Start=yes}', "");
Expect(1, 195102, '\P{ID_Start=yes}', "");
Expect(0, 195102, '\P{^ID_Start=yes}', "");
Expect(1, 195101, '\p{ID_Start:-_YES}', "");
Expect(0, 195101, '\p{^ID_Start:-_YES}', "");
Expect(0, 195101, '\P{ID_Start:-_YES}', "");
Expect(1, 195101, '\P{^ID_Start:-_YES}', "");
Expect(0, 195102, '\p{ID_Start:-_YES}', "");
Expect(1, 195102, '\p{^ID_Start:-_YES}', "");
Expect(1, 195102, '\P{ID_Start:-_YES}', "");
Expect(0, 195102, '\P{^ID_Start:-_YES}', "");
Error('\p{IDS=_-Y/a/}');
Error('\P{IDS=_-Y/a/}');
Expect(1, 195101, '\p{IDS: y}', "");
Expect(0, 195101, '\p{^IDS: y}', "");
Expect(0, 195101, '\P{IDS: y}', "");
Expect(1, 195101, '\P{^IDS: y}', "");
Expect(0, 195102, '\p{IDS: y}', "");
Expect(1, 195102, '\p{^IDS: y}', "");
Expect(1, 195102, '\P{IDS: y}', "");
Expect(0, 195102, '\P{^IDS: y}', "");
Expect(1, 195101, '\p{IDS= -y}', "");
Expect(0, 195101, '\p{^IDS= -y}', "");
Expect(0, 195101, '\P{IDS= -y}', "");
Expect(1, 195101, '\P{^IDS= -y}', "");
Expect(0, 195102, '\p{IDS= -y}', "");
Expect(1, 195102, '\p{^IDS= -y}', "");
Expect(1, 195102, '\P{IDS= -y}', "");
Expect(0, 195102, '\P{^IDS= -y}', "");
Error('\p{Is_ID_Start= T/a/}');
Error('\P{Is_ID_Start= T/a/}');
Expect(1, 195101, '\p{Is_ID_Start=t}', "");
Expect(0, 195101, '\p{^Is_ID_Start=t}', "");
Expect(0, 195101, '\P{Is_ID_Start=t}', "");
Expect(1, 195101, '\P{^Is_ID_Start=t}', "");
Expect(0, 195102, '\p{Is_ID_Start=t}', "");
Expect(1, 195102, '\p{^Is_ID_Start=t}', "");
Expect(1, 195102, '\P{Is_ID_Start=t}', "");
Expect(0, 195102, '\P{^Is_ID_Start=t}', "");
Expect(1, 195101, '\p{Is_ID_Start=__T}', "");
Expect(0, 195101, '\p{^Is_ID_Start=__T}', "");
Expect(0, 195101, '\P{Is_ID_Start=__T}', "");
Expect(1, 195101, '\P{^Is_ID_Start=__T}', "");
Expect(0, 195102, '\p{Is_ID_Start=__T}', "");
Expect(1, 195102, '\p{^Is_ID_Start=__T}', "");
Expect(1, 195102, '\P{Is_ID_Start=__T}', "");
Expect(0, 195102, '\P{^Is_ID_Start=__T}', "");
Error('\p{Is_IDS=/a/true}');
Error('\P{Is_IDS=/a/true}');
Expect(1, 195101, '\p{Is_IDS=true}', "");
Expect(0, 195101, '\p{^Is_IDS=true}', "");
Expect(0, 195101, '\P{Is_IDS=true}', "");
Expect(1, 195101, '\P{^Is_IDS=true}', "");
Expect(0, 195102, '\p{Is_IDS=true}', "");
Expect(1, 195102, '\p{^Is_IDS=true}', "");
Expect(1, 195102, '\P{Is_IDS=true}', "");
Expect(0, 195102, '\P{^Is_IDS=true}', "");
Expect(1, 195101, '\p{Is_IDS=		TRUE}', "");
Expect(0, 195101, '\p{^Is_IDS=		TRUE}', "");
Expect(0, 195101, '\P{Is_IDS=		TRUE}', "");
Expect(1, 195101, '\P{^Is_IDS=		TRUE}', "");
Expect(0, 195102, '\p{Is_IDS=		TRUE}', "");
Expect(1, 195102, '\p{^Is_IDS=		TRUE}', "");
Expect(1, 195102, '\P{Is_IDS=		TRUE}', "");
Expect(0, 195102, '\P{^Is_IDS=		TRUE}', "");
Error('\p{ID_Start=_no/a/}');
Error('\P{ID_Start=_no/a/}');
Expect(1, 195102, '\p{ID_Start=no}', "");
Expect(0, 195102, '\p{^ID_Start=no}', "");
Expect(0, 195102, '\P{ID_Start=no}', "");
Expect(1, 195102, '\P{^ID_Start=no}', "");
Expect(0, 195101, '\p{ID_Start=no}', "");
Expect(1, 195101, '\p{^ID_Start=no}', "");
Expect(1, 195101, '\P{ID_Start=no}', "");
Expect(0, 195101, '\P{^ID_Start=no}', "");
Expect(1, 195102, '\p{ID_Start=_-NO}', "");
Expect(0, 195102, '\p{^ID_Start=_-NO}', "");
Expect(0, 195102, '\P{ID_Start=_-NO}', "");
Expect(1, 195102, '\P{^ID_Start=_-NO}', "");
Expect(0, 195101, '\p{ID_Start=_-NO}', "");
Expect(1, 195101, '\p{^ID_Start=_-NO}', "");
Expect(1, 195101, '\P{ID_Start=_-NO}', "");
Expect(0, 195101, '\P{^ID_Start=_-NO}', "");
Error('\p{IDS=:=_N}');
Error('\P{IDS=:=_N}');
Expect(1, 195102, '\p{IDS=n}', "");
Expect(0, 195102, '\p{^IDS=n}', "");
Expect(0, 195102, '\P{IDS=n}', "");
Expect(1, 195102, '\P{^IDS=n}', "");
Expect(0, 195101, '\p{IDS=n}', "");
Expect(1, 195101, '\p{^IDS=n}', "");
Expect(1, 195101, '\P{IDS=n}', "");
Expect(0, 195101, '\P{^IDS=n}', "");
Expect(1, 195102, '\p{IDS=-_N}', "");
Expect(0, 195102, '\p{^IDS=-_N}', "");
Expect(0, 195102, '\P{IDS=-_N}', "");
Expect(1, 195102, '\P{^IDS=-_N}', "");
Expect(0, 195101, '\p{IDS=-_N}', "");
Expect(1, 195101, '\p{^IDS=-_N}', "");
Expect(1, 195101, '\P{IDS=-_N}', "");
Expect(0, 195101, '\P{^IDS=-_N}', "");
Error('\p{Is_ID_Start=_:=F}');
Error('\P{Is_ID_Start=_:=F}');
Expect(1, 195102, '\p{Is_ID_Start:f}', "");
Expect(0, 195102, '\p{^Is_ID_Start:f}', "");
Expect(0, 195102, '\P{Is_ID_Start:f}', "");
Expect(1, 195102, '\P{^Is_ID_Start:f}', "");
Expect(0, 195101, '\p{Is_ID_Start:f}', "");
Expect(1, 195101, '\p{^Is_ID_Start:f}', "");
Expect(1, 195101, '\P{Is_ID_Start:f}', "");
Expect(0, 195101, '\P{^Is_ID_Start:f}', "");
Expect(1, 195102, '\p{Is_ID_Start=F}', "");
Expect(0, 195102, '\p{^Is_ID_Start=F}', "");
Expect(0, 195102, '\P{Is_ID_Start=F}', "");
Expect(1, 195102, '\P{^Is_ID_Start=F}', "");
Expect(0, 195101, '\p{Is_ID_Start=F}', "");
Expect(1, 195101, '\p{^Is_ID_Start=F}', "");
Expect(1, 195101, '\P{Is_ID_Start=F}', "");
Expect(0, 195101, '\P{^Is_ID_Start=F}', "");
Error('\p{Is_IDS=:= False}');
Error('\P{Is_IDS=:= False}');
Expect(1, 195102, '\p{Is_IDS=false}', "");
Expect(0, 195102, '\p{^Is_IDS=false}', "");
Expect(0, 195102, '\P{Is_IDS=false}', "");
Expect(1, 195102, '\P{^Is_IDS=false}', "");
Expect(0, 195101, '\p{Is_IDS=false}', "");
Expect(1, 195101, '\p{^Is_IDS=false}', "");
Expect(1, 195101, '\P{Is_IDS=false}', "");
Expect(0, 195101, '\P{^Is_IDS=false}', "");
Expect(1, 195102, '\p{Is_IDS=	false}', "");
Expect(0, 195102, '\p{^Is_IDS=	false}', "");
Expect(0, 195102, '\P{Is_IDS=	false}', "");
Expect(1, 195102, '\P{^Is_IDS=	false}', "");
Expect(0, 195101, '\p{Is_IDS=	false}', "");
Expect(1, 195101, '\p{^Is_IDS=	false}', "");
Expect(1, 195101, '\P{Is_IDS=	false}', "");
Expect(0, 195101, '\P{^Is_IDS=	false}', "");
Error('\p{Sentence_Break=/a/ sp}');
Error('\P{Sentence_Break=/a/ sp}');
Expect(1, 12288, '\p{Sentence_Break=sp}', "");
Expect(0, 12288, '\p{^Sentence_Break=sp}', "");
Expect(0, 12288, '\P{Sentence_Break=sp}', "");
Expect(1, 12288, '\P{^Sentence_Break=sp}', "");
Expect(0, 12289, '\p{Sentence_Break=sp}', "");
Expect(1, 12289, '\p{^Sentence_Break=sp}', "");
Expect(1, 12289, '\P{Sentence_Break=sp}', "");
Expect(0, 12289, '\P{^Sentence_Break=sp}', "");
Expect(1, 12288, '\p{Sentence_Break=	_Sp}', "");
Expect(0, 12288, '\p{^Sentence_Break=	_Sp}', "");
Expect(0, 12288, '\P{Sentence_Break=	_Sp}', "");
Expect(1, 12288, '\P{^Sentence_Break=	_Sp}', "");
Expect(0, 12289, '\p{Sentence_Break=	_Sp}', "");
Expect(1, 12289, '\p{^Sentence_Break=	_Sp}', "");
Expect(1, 12289, '\P{Sentence_Break=	_Sp}', "");
Expect(0, 12289, '\P{^Sentence_Break=	_Sp}', "");
Error('\p{SB=	/a/SP}');
Error('\P{SB=	/a/SP}');
Expect(1, 12288, '\p{SB=sp}', "");
Expect(0, 12288, '\p{^SB=sp}', "");
Expect(0, 12288, '\P{SB=sp}', "");
Expect(1, 12288, '\P{^SB=sp}', "");
Expect(0, 12289, '\p{SB=sp}', "");
Expect(1, 12289, '\p{^SB=sp}', "");
Expect(1, 12289, '\P{SB=sp}', "");
Expect(0, 12289, '\P{^SB=sp}', "");
Expect(1, 12288, '\p{SB:--SP}', "");
Expect(0, 12288, '\p{^SB:--SP}', "");
Expect(0, 12288, '\P{SB:--SP}', "");
Expect(1, 12288, '\P{^SB:--SP}', "");
Expect(0, 12289, '\p{SB:--SP}', "");
Expect(1, 12289, '\p{^SB:--SP}', "");
Expect(1, 12289, '\P{SB:--SP}', "");
Expect(0, 12289, '\P{^SB:--SP}', "");
Error('\p{Is_Sentence_Break=_/a/Sp}');
Error('\P{Is_Sentence_Break=_/a/Sp}');
Expect(1, 12288, '\p{Is_Sentence_Break: sp}', "");
Expect(0, 12288, '\p{^Is_Sentence_Break: sp}', "");
Expect(0, 12288, '\P{Is_Sentence_Break: sp}', "");
Expect(1, 12288, '\P{^Is_Sentence_Break: sp}', "");
Expect(0, 12289, '\p{Is_Sentence_Break: sp}', "");
Expect(1, 12289, '\p{^Is_Sentence_Break: sp}', "");
Expect(1, 12289, '\P{Is_Sentence_Break: sp}', "");
Expect(0, 12289, '\P{^Is_Sentence_Break: sp}', "");
Expect(1, 12288, '\p{Is_Sentence_Break=_ sp}', "");
Expect(0, 12288, '\p{^Is_Sentence_Break=_ sp}', "");
Expect(0, 12288, '\P{Is_Sentence_Break=_ sp}', "");
Expect(1, 12288, '\P{^Is_Sentence_Break=_ sp}', "");
Expect(0, 12289, '\p{Is_Sentence_Break=_ sp}', "");
Expect(1, 12289, '\p{^Is_Sentence_Break=_ sp}', "");
Expect(1, 12289, '\P{Is_Sentence_Break=_ sp}', "");
Expect(0, 12289, '\P{^Is_Sentence_Break=_ sp}', "");
Error('\p{Is_SB=_Sp/a/}');
Error('\P{Is_SB=_Sp/a/}');
Expect(1, 12288, '\p{Is_SB=sp}', "");
Expect(0, 12288, '\p{^Is_SB=sp}', "");
Expect(0, 12288, '\P{Is_SB=sp}', "");
Expect(1, 12288, '\P{^Is_SB=sp}', "");
Expect(0, 12289, '\p{Is_SB=sp}', "");
Expect(1, 12289, '\p{^Is_SB=sp}', "");
Expect(1, 12289, '\P{Is_SB=sp}', "");
Expect(0, 12289, '\P{^Is_SB=sp}', "");
Expect(1, 12288, '\p{Is_SB=-_Sp}', "");
Expect(0, 12288, '\p{^Is_SB=-_Sp}', "");
Expect(0, 12288, '\P{Is_SB=-_Sp}', "");
Expect(1, 12288, '\P{^Is_SB=-_Sp}', "");
Expect(0, 12289, '\p{Is_SB=-_Sp}', "");
Expect(1, 12289, '\p{^Is_SB=-_Sp}', "");
Expect(1, 12289, '\P{Is_SB=-_Sp}', "");
Expect(0, 12289, '\P{^Is_SB=-_Sp}', "");
Error('\p{Sentence_Break=:=  scontinue}');
Error('\P{Sentence_Break=:=  scontinue}');
Expect(1, 65380, '\p{Sentence_Break=scontinue}', "");
Expect(0, 65380, '\p{^Sentence_Break=scontinue}', "");
Expect(0, 65380, '\P{Sentence_Break=scontinue}', "");
Expect(1, 65380, '\P{^Sentence_Break=scontinue}', "");
Expect(0, 65381, '\p{Sentence_Break=scontinue}', "");
Expect(1, 65381, '\p{^Sentence_Break=scontinue}', "");
Expect(1, 65381, '\P{Sentence_Break=scontinue}', "");
Expect(0, 65381, '\P{^Sentence_Break=scontinue}', "");
Expect(1, 65380, '\p{Sentence_Break=_	SContinue}', "");
Expect(0, 65380, '\p{^Sentence_Break=_	SContinue}', "");
Expect(0, 65380, '\P{Sentence_Break=_	SContinue}', "");
Expect(1, 65380, '\P{^Sentence_Break=_	SContinue}', "");
Expect(0, 65381, '\p{Sentence_Break=_	SContinue}', "");
Expect(1, 65381, '\p{^Sentence_Break=_	SContinue}', "");
Expect(1, 65381, '\P{Sentence_Break=_	SContinue}', "");
Expect(0, 65381, '\P{^Sentence_Break=_	SContinue}', "");
Error('\p{SB=/a/-	SC}');
Error('\P{SB=/a/-	SC}');
Expect(1, 65380, '\p{SB=sc}', "");
Expect(0, 65380, '\p{^SB=sc}', "");
Expect(0, 65380, '\P{SB=sc}', "");
Expect(1, 65380, '\P{^SB=sc}', "");
Expect(0, 65381, '\p{SB=sc}', "");
Expect(1, 65381, '\p{^SB=sc}', "");
Expect(1, 65381, '\P{SB=sc}', "");
Expect(0, 65381, '\P{^SB=sc}', "");
Expect(1, 65380, '\p{SB=SC}', "");
Expect(0, 65380, '\p{^SB=SC}', "");
Expect(0, 65380, '\P{SB=SC}', "");
Expect(1, 65380, '\P{^SB=SC}', "");
Expect(0, 65381, '\p{SB=SC}', "");
Expect(1, 65381, '\p{^SB=SC}', "");
Expect(1, 65381, '\P{SB=SC}', "");
Expect(0, 65381, '\P{^SB=SC}', "");
Error('\p{Is_Sentence_Break=:=scontinue}');
Error('\P{Is_Sentence_Break=:=scontinue}');
Expect(1, 65380, '\p{Is_Sentence_Break=scontinue}', "");
Expect(0, 65380, '\p{^Is_Sentence_Break=scontinue}', "");
Expect(0, 65380, '\P{Is_Sentence_Break=scontinue}', "");
Expect(1, 65380, '\P{^Is_Sentence_Break=scontinue}', "");
Expect(0, 65381, '\p{Is_Sentence_Break=scontinue}', "");
Expect(1, 65381, '\p{^Is_Sentence_Break=scontinue}', "");
Expect(1, 65381, '\P{Is_Sentence_Break=scontinue}', "");
Expect(0, 65381, '\P{^Is_Sentence_Break=scontinue}', "");
Expect(1, 65380, '\p{Is_Sentence_Break:	 _SContinue}', "");
Expect(0, 65380, '\p{^Is_Sentence_Break:	 _SContinue}', "");
Expect(0, 65380, '\P{Is_Sentence_Break:	 _SContinue}', "");
Expect(1, 65380, '\P{^Is_Sentence_Break:	 _SContinue}', "");
Expect(0, 65381, '\p{Is_Sentence_Break:	 _SContinue}', "");
Expect(1, 65381, '\p{^Is_Sentence_Break:	 _SContinue}', "");
Expect(1, 65381, '\P{Is_Sentence_Break:	 _SContinue}', "");
Expect(0, 65381, '\P{^Is_Sentence_Break:	 _SContinue}', "");
Error('\p{Is_SB=:=SC}');
Error('\P{Is_SB=:=SC}');
Expect(1, 65380, '\p{Is_SB=sc}', "");
Expect(0, 65380, '\p{^Is_SB=sc}', "");
Expect(0, 65380, '\P{Is_SB=sc}', "");
Expect(1, 65380, '\P{^Is_SB=sc}', "");
Expect(0, 65381, '\p{Is_SB=sc}', "");
Expect(1, 65381, '\p{^Is_SB=sc}', "");
Expect(1, 65381, '\P{Is_SB=sc}', "");
Expect(0, 65381, '\P{^Is_SB=sc}', "");
Expect(1, 65380, '\p{Is_SB=	 SC}', "");
Expect(0, 65380, '\p{^Is_SB=	 SC}', "");
Expect(0, 65380, '\P{Is_SB=	 SC}', "");
Expect(1, 65380, '\P{^Is_SB=	 SC}', "");
Expect(0, 65381, '\p{Is_SB=	 SC}', "");
Expect(1, 65381, '\p{^Is_SB=	 SC}', "");
Expect(1, 65381, '\P{Is_SB=	 SC}', "");
Expect(0, 65381, '\P{^Is_SB=	 SC}', "");
Error('\p{Sentence_Break=ATerm/a/}');
Error('\P{Sentence_Break=ATerm/a/}');
Expect(1, 65294, '\p{Sentence_Break=aterm}', "");
Expect(0, 65294, '\p{^Sentence_Break=aterm}', "");
Expect(0, 65294, '\P{Sentence_Break=aterm}', "");
Expect(1, 65294, '\P{^Sentence_Break=aterm}', "");
Expect(0, 65295, '\p{Sentence_Break=aterm}', "");
Expect(1, 65295, '\p{^Sentence_Break=aterm}', "");
Expect(1, 65295, '\P{Sentence_Break=aterm}', "");
Expect(0, 65295, '\P{^Sentence_Break=aterm}', "");
Expect(1, 65294, '\p{Sentence_Break=__ATERM}', "");
Expect(0, 65294, '\p{^Sentence_Break=__ATERM}', "");
Expect(0, 65294, '\P{Sentence_Break=__ATERM}', "");
Expect(1, 65294, '\P{^Sentence_Break=__ATERM}', "");
Expect(0, 65295, '\p{Sentence_Break=__ATERM}', "");
Expect(1, 65295, '\p{^Sentence_Break=__ATERM}', "");
Expect(1, 65295, '\P{Sentence_Break=__ATERM}', "");
Expect(0, 65295, '\P{^Sentence_Break=__ATERM}', "");
Error('\p{SB=/a/_at}');
Error('\P{SB=/a/_at}');
Expect(1, 65294, '\p{SB=at}', "");
Expect(0, 65294, '\p{^SB=at}', "");
Expect(0, 65294, '\P{SB=at}', "");
Expect(1, 65294, '\P{^SB=at}', "");
Expect(0, 65295, '\p{SB=at}', "");
Expect(1, 65295, '\p{^SB=at}', "");
Expect(1, 65295, '\P{SB=at}', "");
Expect(0, 65295, '\P{^SB=at}', "");
Expect(1, 65294, '\p{SB=-_AT}', "");
Expect(0, 65294, '\p{^SB=-_AT}', "");
Expect(0, 65294, '\P{SB=-_AT}', "");
Expect(1, 65294, '\P{^SB=-_AT}', "");
Expect(0, 65295, '\p{SB=-_AT}', "");
Expect(1, 65295, '\p{^SB=-_AT}', "");
Expect(1, 65295, '\P{SB=-_AT}', "");
Expect(0, 65295, '\P{^SB=-_AT}', "");
Error('\p{Is_Sentence_Break=:=ATERM}');
Error('\P{Is_Sentence_Break=:=ATERM}');
Expect(1, 65294, '\p{Is_Sentence_Break=aterm}', "");
Expect(0, 65294, '\p{^Is_Sentence_Break=aterm}', "");
Expect(0, 65294, '\P{Is_Sentence_Break=aterm}', "");
Expect(1, 65294, '\P{^Is_Sentence_Break=aterm}', "");
Expect(0, 65295, '\p{Is_Sentence_Break=aterm}', "");
Expect(1, 65295, '\p{^Is_Sentence_Break=aterm}', "");
Expect(1, 65295, '\P{Is_Sentence_Break=aterm}', "");
Expect(0, 65295, '\P{^Is_Sentence_Break=aterm}', "");
Expect(1, 65294, '\p{Is_Sentence_Break=-_ATerm}', "");
Expect(0, 65294, '\p{^Is_Sentence_Break=-_ATerm}', "");
Expect(0, 65294, '\P{Is_Sentence_Break=-_ATerm}', "");
Expect(1, 65294, '\P{^Is_Sentence_Break=-_ATerm}', "");
Expect(0, 65295, '\p{Is_Sentence_Break=-_ATerm}', "");
Expect(1, 65295, '\p{^Is_Sentence_Break=-_ATerm}', "");
Expect(1, 65295, '\P{Is_Sentence_Break=-_ATerm}', "");
Expect(0, 65295, '\P{^Is_Sentence_Break=-_ATerm}', "");
Error('\p{Is_SB=/a/	AT}');
Error('\P{Is_SB=/a/	AT}');
Expect(1, 65294, '\p{Is_SB:	at}', "");
Expect(0, 65294, '\p{^Is_SB:	at}', "");
Expect(0, 65294, '\P{Is_SB:	at}', "");
Expect(1, 65294, '\P{^Is_SB:	at}', "");
Expect(0, 65295, '\p{Is_SB:	at}', "");
Expect(1, 65295, '\p{^Is_SB:	at}', "");
Expect(1, 65295, '\P{Is_SB:	at}', "");
Expect(0, 65295, '\P{^Is_SB:	at}', "");
Expect(1, 65294, '\p{Is_SB= at}', "");
Expect(0, 65294, '\p{^Is_SB= at}', "");
Expect(0, 65294, '\P{Is_SB= at}', "");
Expect(1, 65294, '\P{^Is_SB= at}', "");
Expect(0, 65295, '\p{Is_SB= at}', "");
Expect(1, 65295, '\p{^Is_SB= at}', "");
Expect(1, 65295, '\P{Is_SB= at}', "");
Expect(0, 65295, '\P{^Is_SB= at}', "");
Error('\p{Sentence_Break=_OLetter:=}');
Error('\P{Sentence_Break=_OLetter:=}');
Expect(1, 195101, '\p{Sentence_Break=oletter}', "");
Expect(0, 195101, '\p{^Sentence_Break=oletter}', "");
Expect(0, 195101, '\P{Sentence_Break=oletter}', "");
Expect(1, 195101, '\P{^Sentence_Break=oletter}', "");
Expect(0, 195102, '\p{Sentence_Break=oletter}', "");
Expect(1, 195102, '\p{^Sentence_Break=oletter}', "");
Expect(1, 195102, '\P{Sentence_Break=oletter}', "");
Expect(0, 195102, '\P{^Sentence_Break=oletter}', "");
Expect(1, 195101, '\p{Sentence_Break=_OLetter}', "");
Expect(0, 195101, '\p{^Sentence_Break=_OLetter}', "");
Expect(0, 195101, '\P{Sentence_Break=_OLetter}', "");
Expect(1, 195101, '\P{^Sentence_Break=_OLetter}', "");
Expect(0, 195102, '\p{Sentence_Break=_OLetter}', "");
Expect(1, 195102, '\p{^Sentence_Break=_OLetter}', "");
Expect(1, 195102, '\P{Sentence_Break=_OLetter}', "");
Expect(0, 195102, '\P{^Sentence_Break=_OLetter}', "");
Error('\p{SB=/a/	LE}');
Error('\P{SB=/a/	LE}');
Expect(1, 195101, '\p{SB=le}', "");
Expect(0, 195101, '\p{^SB=le}', "");
Expect(0, 195101, '\P{SB=le}', "");
Expect(1, 195101, '\P{^SB=le}', "");
Expect(0, 195102, '\p{SB=le}', "");
Expect(1, 195102, '\p{^SB=le}', "");
Expect(1, 195102, '\P{SB=le}', "");
Expect(0, 195102, '\P{^SB=le}', "");
Expect(1, 195101, '\p{SB=_LE}', "");
Expect(0, 195101, '\p{^SB=_LE}', "");
Expect(0, 195101, '\P{SB=_LE}', "");
Expect(1, 195101, '\P{^SB=_LE}', "");
Expect(0, 195102, '\p{SB=_LE}', "");
Expect(1, 195102, '\p{^SB=_LE}', "");
Expect(1, 195102, '\P{SB=_LE}', "");
Expect(0, 195102, '\P{^SB=_LE}', "");
Error('\p{Is_Sentence_Break=-	OLetter/a/}');
Error('\P{Is_Sentence_Break=-	OLetter/a/}');
Expect(1, 195101, '\p{Is_Sentence_Break=oletter}', "");
Expect(0, 195101, '\p{^Is_Sentence_Break=oletter}', "");
Expect(0, 195101, '\P{Is_Sentence_Break=oletter}', "");
Expect(1, 195101, '\P{^Is_Sentence_Break=oletter}', "");
Expect(0, 195102, '\p{Is_Sentence_Break=oletter}', "");
Expect(1, 195102, '\p{^Is_Sentence_Break=oletter}', "");
Expect(1, 195102, '\P{Is_Sentence_Break=oletter}', "");
Expect(0, 195102, '\P{^Is_Sentence_Break=oletter}', "");
Expect(1, 195101, '\p{Is_Sentence_Break=_OLetter}', "");
Expect(0, 195101, '\p{^Is_Sentence_Break=_OLetter}', "");
Expect(0, 195101, '\P{Is_Sentence_Break=_OLetter}', "");
Expect(1, 195101, '\P{^Is_Sentence_Break=_OLetter}', "");
Expect(0, 195102, '\p{Is_Sentence_Break=_OLetter}', "");
Expect(1, 195102, '\p{^Is_Sentence_Break=_OLetter}', "");
Expect(1, 195102, '\P{Is_Sentence_Break=_OLetter}', "");
Expect(0, 195102, '\P{^Is_Sentence_Break=_OLetter}', "");
Error('\p{Is_SB:	/a/--LE}');
Error('\P{Is_SB:	/a/--LE}');
Expect(1, 195101, '\p{Is_SB=le}', "");
Expect(0, 195101, '\p{^Is_SB=le}', "");
Expect(0, 195101, '\P{Is_SB=le}', "");
Expect(1, 195101, '\P{^Is_SB=le}', "");
Expect(0, 195102, '\p{Is_SB=le}', "");
Expect(1, 195102, '\p{^Is_SB=le}', "");
Expect(1, 195102, '\P{Is_SB=le}', "");
Expect(0, 195102, '\P{^Is_SB=le}', "");
Expect(1, 195101, '\p{Is_SB= 	LE}', "");
Expect(0, 195101, '\p{^Is_SB= 	LE}', "");
Expect(0, 195101, '\P{Is_SB= 	LE}', "");
Expect(1, 195101, '\P{^Is_SB= 	LE}', "");
Expect(0, 195102, '\p{Is_SB= 	LE}', "");
Expect(1, 195102, '\p{^Is_SB= 	LE}', "");
Expect(1, 195102, '\P{Is_SB= 	LE}', "");
Expect(0, 195102, '\P{^Is_SB= 	LE}', "");
Error('\p{Sentence_Break=	:=CR}');
Error('\P{Sentence_Break=	:=CR}');
Expect(1, 13, '\p{Sentence_Break=cr}', "");
Expect(0, 13, '\p{^Sentence_Break=cr}', "");
Expect(0, 13, '\P{Sentence_Break=cr}', "");
Expect(1, 13, '\P{^Sentence_Break=cr}', "");
Expect(0, 14, '\p{Sentence_Break=cr}', "");
Expect(1, 14, '\p{^Sentence_Break=cr}', "");
Expect(1, 14, '\P{Sentence_Break=cr}', "");
Expect(0, 14, '\P{^Sentence_Break=cr}', "");
Expect(1, 13, '\p{Sentence_Break=-CR}', "");
Expect(0, 13, '\p{^Sentence_Break=-CR}', "");
Expect(0, 13, '\P{Sentence_Break=-CR}', "");
Expect(1, 13, '\P{^Sentence_Break=-CR}', "");
Expect(0, 14, '\p{Sentence_Break=-CR}', "");
Expect(1, 14, '\p{^Sentence_Break=-CR}', "");
Expect(1, 14, '\P{Sentence_Break=-CR}', "");
Expect(0, 14, '\P{^Sentence_Break=-CR}', "");
Error('\p{SB=:=-_CR}');
Error('\P{SB=:=-_CR}');
Expect(1, 13, '\p{SB=cr}', "");
Expect(0, 13, '\p{^SB=cr}', "");
Expect(0, 13, '\P{SB=cr}', "");
Expect(1, 13, '\P{^SB=cr}', "");
Expect(0, 14, '\p{SB=cr}', "");
Expect(1, 14, '\p{^SB=cr}', "");
Expect(1, 14, '\P{SB=cr}', "");
Expect(0, 14, '\P{^SB=cr}', "");
Expect(1, 13, '\p{SB=	-CR}', "");
Expect(0, 13, '\p{^SB=	-CR}', "");
Expect(0, 13, '\P{SB=	-CR}', "");
Expect(1, 13, '\P{^SB=	-CR}', "");
Expect(0, 14, '\p{SB=	-CR}', "");
Expect(1, 14, '\p{^SB=	-CR}', "");
Expect(1, 14, '\P{SB=	-CR}', "");
Expect(0, 14, '\P{^SB=	-CR}', "");
Error('\p{Is_Sentence_Break=_/a/CR}');
Error('\P{Is_Sentence_Break=_/a/CR}');
Expect(1, 13, '\p{Is_Sentence_Break=cr}', "");
Expect(0, 13, '\p{^Is_Sentence_Break=cr}', "");
Expect(0, 13, '\P{Is_Sentence_Break=cr}', "");
Expect(1, 13, '\P{^Is_Sentence_Break=cr}', "");
Expect(0, 14, '\p{Is_Sentence_Break=cr}', "");
Expect(1, 14, '\p{^Is_Sentence_Break=cr}', "");
Expect(1, 14, '\P{Is_Sentence_Break=cr}', "");
Expect(0, 14, '\P{^Is_Sentence_Break=cr}', "");
Expect(1, 13, '\p{Is_Sentence_Break:   --CR}', "");
Expect(0, 13, '\p{^Is_Sentence_Break:   --CR}', "");
Expect(0, 13, '\P{Is_Sentence_Break:   --CR}', "");
Expect(1, 13, '\P{^Is_Sentence_Break:   --CR}', "");
Expect(0, 14, '\p{Is_Sentence_Break:   --CR}', "");
Expect(1, 14, '\p{^Is_Sentence_Break:   --CR}', "");
Expect(1, 14, '\P{Is_Sentence_Break:   --CR}', "");
Expect(0, 14, '\P{^Is_Sentence_Break:   --CR}', "");
Error('\p{Is_SB=/a/	-CR}');
Error('\P{Is_SB=/a/	-CR}');
Expect(1, 13, '\p{Is_SB=cr}', "");
Expect(0, 13, '\p{^Is_SB=cr}', "");
Expect(0, 13, '\P{Is_SB=cr}', "");
Expect(1, 13, '\P{^Is_SB=cr}', "");
Expect(0, 14, '\p{Is_SB=cr}', "");
Expect(1, 14, '\p{^Is_SB=cr}', "");
Expect(1, 14, '\P{Is_SB=cr}', "");
Expect(0, 14, '\P{^Is_SB=cr}', "");
Expect(1, 13, '\p{Is_SB=_	cr}', "");
Expect(0, 13, '\p{^Is_SB=_	cr}', "");
Expect(0, 13, '\P{Is_SB=_	cr}', "");
Expect(1, 13, '\P{^Is_SB=_	cr}', "");
Expect(0, 14, '\p{Is_SB=_	cr}', "");
Expect(1, 14, '\p{^Is_SB=_	cr}', "");
Expect(1, 14, '\P{Is_SB=_	cr}', "");
Expect(0, 14, '\P{^Is_SB=_	cr}', "");
Error('\p{Sentence_Break=/a/Other}');
Error('\P{Sentence_Break=/a/Other}');
Expect(1, 918000, '\p{Sentence_Break=other}', "");
Expect(0, 918000, '\p{^Sentence_Break=other}', "");
Expect(0, 918000, '\P{Sentence_Break=other}', "");
Expect(1, 918000, '\P{^Sentence_Break=other}', "");
Expect(0, 917999, '\p{Sentence_Break=other}', "");
Expect(1, 917999, '\p{^Sentence_Break=other}', "");
Expect(1, 917999, '\P{Sentence_Break=other}', "");
Expect(0, 917999, '\P{^Sentence_Break=other}', "");
Expect(1, 918000, '\p{Sentence_Break:	 -Other}', "");
Expect(0, 918000, '\p{^Sentence_Break:	 -Other}', "");
Expect(0, 918000, '\P{Sentence_Break:	 -Other}', "");
Expect(1, 918000, '\P{^Sentence_Break:	 -Other}', "");
Expect(0, 917999, '\p{Sentence_Break:	 -Other}', "");
Expect(1, 917999, '\p{^Sentence_Break:	 -Other}', "");
Expect(1, 917999, '\P{Sentence_Break:	 -Other}', "");
Expect(0, 917999, '\P{^Sentence_Break:	 -Other}', "");
Error('\p{SB:   /a/-xx}');
Error('\P{SB:   /a/-xx}');
Expect(1, 918000, '\p{SB=xx}', "");
Expect(0, 918000, '\p{^SB=xx}', "");
Expect(0, 918000, '\P{SB=xx}', "");
Expect(1, 918000, '\P{^SB=xx}', "");
Expect(0, 917999, '\p{SB=xx}', "");
Expect(1, 917999, '\p{^SB=xx}', "");
Expect(1, 917999, '\P{SB=xx}', "");
Expect(0, 917999, '\P{^SB=xx}', "");
Expect(1, 918000, '\p{SB= XX}', "");
Expect(0, 918000, '\p{^SB= XX}', "");
Expect(0, 918000, '\P{SB= XX}', "");
Expect(1, 918000, '\P{^SB= XX}', "");
Expect(0, 917999, '\p{SB= XX}', "");
Expect(1, 917999, '\p{^SB= XX}', "");
Expect(1, 917999, '\P{SB= XX}', "");
Expect(0, 917999, '\P{^SB= XX}', "");
Error('\p{Is_Sentence_Break=OTHER:=}');
Error('\P{Is_Sentence_Break=OTHER:=}');
Expect(1, 918000, '\p{Is_Sentence_Break: other}', "");
Expect(0, 918000, '\p{^Is_Sentence_Break: other}', "");
Expect(0, 918000, '\P{Is_Sentence_Break: other}', "");
Expect(1, 918000, '\P{^Is_Sentence_Break: other}', "");
Expect(0, 917999, '\p{Is_Sentence_Break: other}', "");
Expect(1, 917999, '\p{^Is_Sentence_Break: other}', "");
Expect(1, 917999, '\P{Is_Sentence_Break: other}', "");
Expect(0, 917999, '\P{^Is_Sentence_Break: other}', "");
Expect(1, 918000, '\p{Is_Sentence_Break=	Other}', "");
Expect(0, 918000, '\p{^Is_Sentence_Break=	Other}', "");
Expect(0, 918000, '\P{Is_Sentence_Break=	Other}', "");
Expect(1, 918000, '\P{^Is_Sentence_Break=	Other}', "");
Expect(0, 917999, '\p{Is_Sentence_Break=	Other}', "");
Expect(1, 917999, '\p{^Is_Sentence_Break=	Other}', "");
Expect(1, 917999, '\P{Is_Sentence_Break=	Other}', "");
Expect(0, 917999, '\P{^Is_Sentence_Break=	Other}', "");
Error('\p{Is_SB=/a/  XX}');
Error('\P{Is_SB=/a/  XX}');
Expect(1, 918000, '\p{Is_SB=xx}', "");
Expect(0, 918000, '\p{^Is_SB=xx}', "");
Expect(0, 918000, '\P{Is_SB=xx}', "");
Expect(1, 918000, '\P{^Is_SB=xx}', "");
Expect(0, 917999, '\p{Is_SB=xx}', "");
Expect(1, 917999, '\p{^Is_SB=xx}', "");
Expect(1, 917999, '\P{Is_SB=xx}', "");
Expect(0, 917999, '\P{^Is_SB=xx}', "");
Expect(1, 918000, '\p{Is_SB=--XX}', "");
Expect(0, 918000, '\p{^Is_SB=--XX}', "");
Expect(0, 918000, '\P{Is_SB=--XX}', "");
Expect(1, 918000, '\P{^Is_SB=--XX}', "");
Expect(0, 917999, '\p{Is_SB=--XX}', "");
Expect(1, 917999, '\p{^Is_SB=--XX}', "");
Expect(1, 917999, '\P{Is_SB=--XX}', "");
Expect(0, 917999, '\P{^Is_SB=--XX}', "");
Error('\p{Sentence_Break=-	NUMERIC/a/}');
Error('\P{Sentence_Break=-	NUMERIC/a/}');
Expect(1, 120831, '\p{Sentence_Break=numeric}', "");
Expect(0, 120831, '\p{^Sentence_Break=numeric}', "");
Expect(0, 120831, '\P{Sentence_Break=numeric}', "");
Expect(1, 120831, '\P{^Sentence_Break=numeric}', "");
Expect(0, 120832, '\p{Sentence_Break=numeric}', "");
Expect(1, 120832, '\p{^Sentence_Break=numeric}', "");
Expect(1, 120832, '\P{Sentence_Break=numeric}', "");
Expect(0, 120832, '\P{^Sentence_Break=numeric}', "");
Expect(1, 120831, '\p{Sentence_Break= 	Numeric}', "");
Expect(0, 120831, '\p{^Sentence_Break= 	Numeric}', "");
Expect(0, 120831, '\P{Sentence_Break= 	Numeric}', "");
Expect(1, 120831, '\P{^Sentence_Break= 	Numeric}', "");
Expect(0, 120832, '\p{Sentence_Break= 	Numeric}', "");
Expect(1, 120832, '\p{^Sentence_Break= 	Numeric}', "");
Expect(1, 120832, '\P{Sentence_Break= 	Numeric}', "");
Expect(0, 120832, '\P{^Sentence_Break= 	Numeric}', "");
Error('\p{SB=_	NU/a/}');
Error('\P{SB=_	NU/a/}');
Expect(1, 120831, '\p{SB=nu}', "");
Expect(0, 120831, '\p{^SB=nu}', "");
Expect(0, 120831, '\P{SB=nu}', "");
Expect(1, 120831, '\P{^SB=nu}', "");
Expect(0, 120832, '\p{SB=nu}', "");
Expect(1, 120832, '\p{^SB=nu}', "");
Expect(1, 120832, '\P{SB=nu}', "");
Expect(0, 120832, '\P{^SB=nu}', "");
Expect(1, 120831, '\p{SB= -nu}', "");
Expect(0, 120831, '\p{^SB= -nu}', "");
Expect(0, 120831, '\P{SB= -nu}', "");
Expect(1, 120831, '\P{^SB= -nu}', "");
Expect(0, 120832, '\p{SB= -nu}', "");
Expect(1, 120832, '\p{^SB= -nu}', "");
Expect(1, 120832, '\P{SB= -nu}', "");
Expect(0, 120832, '\P{^SB= -nu}', "");
Error('\p{Is_Sentence_Break=	/a/NUMERIC}');
Error('\P{Is_Sentence_Break=	/a/NUMERIC}');
Expect(1, 120831, '\p{Is_Sentence_Break=numeric}', "");
Expect(0, 120831, '\p{^Is_Sentence_Break=numeric}', "");
Expect(0, 120831, '\P{Is_Sentence_Break=numeric}', "");
Expect(1, 120831, '\P{^Is_Sentence_Break=numeric}', "");
Expect(0, 120832, '\p{Is_Sentence_Break=numeric}', "");
Expect(1, 120832, '\p{^Is_Sentence_Break=numeric}', "");
Expect(1, 120832, '\P{Is_Sentence_Break=numeric}', "");
Expect(0, 120832, '\P{^Is_Sentence_Break=numeric}', "");
Expect(1, 120831, '\p{Is_Sentence_Break=-	numeric}', "");
Expect(0, 120831, '\p{^Is_Sentence_Break=-	numeric}', "");
Expect(0, 120831, '\P{Is_Sentence_Break=-	numeric}', "");
Expect(1, 120831, '\P{^Is_Sentence_Break=-	numeric}', "");
Expect(0, 120832, '\p{Is_Sentence_Break=-	numeric}', "");
Expect(1, 120832, '\p{^Is_Sentence_Break=-	numeric}', "");
Expect(1, 120832, '\P{Is_Sentence_Break=-	numeric}', "");
Expect(0, 120832, '\P{^Is_Sentence_Break=-	numeric}', "");
Error('\p{Is_SB=/a/ 	nu}');
Error('\P{Is_SB=/a/ 	nu}');
Expect(1, 120831, '\p{Is_SB=nu}', "");
Expect(0, 120831, '\p{^Is_SB=nu}', "");
Expect(0, 120831, '\P{Is_SB=nu}', "");
Expect(1, 120831, '\P{^Is_SB=nu}', "");
Expect(0, 120832, '\p{Is_SB=nu}', "");
Expect(1, 120832, '\p{^Is_SB=nu}', "");
Expect(1, 120832, '\P{Is_SB=nu}', "");
Expect(0, 120832, '\P{^Is_SB=nu}', "");
Expect(1, 120831, '\p{Is_SB:  NU}', "");
Expect(0, 120831, '\p{^Is_SB:  NU}', "");
Expect(0, 120831, '\P{Is_SB:  NU}', "");
Expect(1, 120831, '\P{^Is_SB:  NU}', "");
Expect(0, 120832, '\p{Is_SB:  NU}', "");
Expect(1, 120832, '\p{^Is_SB:  NU}', "");
Expect(1, 120832, '\P{Is_SB:  NU}', "");
Expect(0, 120832, '\P{^Is_SB:  NU}', "");
Error('\p{Sentence_Break: FORMAT:=}');
Error('\P{Sentence_Break: FORMAT:=}');
Expect(1, 917631, '\p{Sentence_Break=format}', "");
Expect(0, 917631, '\p{^Sentence_Break=format}', "");
Expect(0, 917631, '\P{Sentence_Break=format}', "");
Expect(1, 917631, '\P{^Sentence_Break=format}', "");
Expect(0, 917632, '\p{Sentence_Break=format}', "");
Expect(1, 917632, '\p{^Sentence_Break=format}', "");
Expect(1, 917632, '\P{Sentence_Break=format}', "");
Expect(0, 917632, '\P{^Sentence_Break=format}', "");
Expect(1, 917631, '\p{Sentence_Break=_-Format}', "");
Expect(0, 917631, '\p{^Sentence_Break=_-Format}', "");
Expect(0, 917631, '\P{Sentence_Break=_-Format}', "");
Expect(1, 917631, '\P{^Sentence_Break=_-Format}', "");
Expect(0, 917632, '\p{Sentence_Break=_-Format}', "");
Expect(1, 917632, '\p{^Sentence_Break=_-Format}', "");
Expect(1, 917632, '\P{Sentence_Break=_-Format}', "");
Expect(0, 917632, '\P{^Sentence_Break=_-Format}', "");
Error('\p{SB= :=fo}');
Error('\P{SB= :=fo}');
Expect(1, 917631, '\p{SB=fo}', "");
Expect(0, 917631, '\p{^SB=fo}', "");
Expect(0, 917631, '\P{SB=fo}', "");
Expect(1, 917631, '\P{^SB=fo}', "");
Expect(0, 917632, '\p{SB=fo}', "");
Expect(1, 917632, '\p{^SB=fo}', "");
Expect(1, 917632, '\P{SB=fo}', "");
Expect(0, 917632, '\P{^SB=fo}', "");
Expect(1, 917631, '\p{SB=-FO}', "");
Expect(0, 917631, '\p{^SB=-FO}', "");
Expect(0, 917631, '\P{SB=-FO}', "");
Expect(1, 917631, '\P{^SB=-FO}', "");
Expect(0, 917632, '\p{SB=-FO}', "");
Expect(1, 917632, '\p{^SB=-FO}', "");
Expect(1, 917632, '\P{SB=-FO}', "");
Expect(0, 917632, '\P{^SB=-FO}', "");
Error('\p{Is_Sentence_Break: format/a/}');
Error('\P{Is_Sentence_Break: format/a/}');
Expect(1, 917631, '\p{Is_Sentence_Break=format}', "");
Expect(0, 917631, '\p{^Is_Sentence_Break=format}', "");
Expect(0, 917631, '\P{Is_Sentence_Break=format}', "");
Expect(1, 917631, '\P{^Is_Sentence_Break=format}', "");
Expect(0, 917632, '\p{Is_Sentence_Break=format}', "");
Expect(1, 917632, '\p{^Is_Sentence_Break=format}', "");
Expect(1, 917632, '\P{Is_Sentence_Break=format}', "");
Expect(0, 917632, '\P{^Is_Sentence_Break=format}', "");
Expect(1, 917631, '\p{Is_Sentence_Break=_ Format}', "");
Expect(0, 917631, '\p{^Is_Sentence_Break=_ Format}', "");
Expect(0, 917631, '\P{Is_Sentence_Break=_ Format}', "");
Expect(1, 917631, '\P{^Is_Sentence_Break=_ Format}', "");
Expect(0, 917632, '\p{Is_Sentence_Break=_ Format}', "");
Expect(1, 917632, '\p{^Is_Sentence_Break=_ Format}', "");
Expect(1, 917632, '\P{Is_Sentence_Break=_ Format}', "");
Expect(0, 917632, '\P{^Is_Sentence_Break=_ Format}', "");
Error('\p{Is_SB::=fo}');
Error('\P{Is_SB::=fo}');
Expect(1, 917631, '\p{Is_SB=fo}', "");
Expect(0, 917631, '\p{^Is_SB=fo}', "");
Expect(0, 917631, '\P{Is_SB=fo}', "");
Expect(1, 917631, '\P{^Is_SB=fo}', "");
Expect(0, 917632, '\p{Is_SB=fo}', "");
Expect(1, 917632, '\p{^Is_SB=fo}', "");
Expect(1, 917632, '\P{Is_SB=fo}', "");
Expect(0, 917632, '\P{^Is_SB=fo}', "");
Expect(1, 917631, '\p{Is_SB=-FO}', "");
Expect(0, 917631, '\p{^Is_SB=-FO}', "");
Expect(0, 917631, '\P{Is_SB=-FO}', "");
Expect(1, 917631, '\P{^Is_SB=-FO}', "");
Expect(0, 917632, '\p{Is_SB=-FO}', "");
Expect(1, 917632, '\p{^Is_SB=-FO}', "");
Expect(1, 917632, '\P{Is_SB=-FO}', "");
Expect(0, 917632, '\P{^Is_SB=-FO}', "");
Error('\p{Sentence_Break= /a/lower}');
Error('\P{Sentence_Break= /a/lower}');
Expect(1, 120779, '\p{Sentence_Break=lower}', "");
Expect(0, 120779, '\p{^Sentence_Break=lower}', "");
Expect(0, 120779, '\P{Sentence_Break=lower}', "");
Expect(1, 120779, '\P{^Sentence_Break=lower}', "");
Expect(0, 120780, '\p{Sentence_Break=lower}', "");
Expect(1, 120780, '\p{^Sentence_Break=lower}', "");
Expect(1, 120780, '\P{Sentence_Break=lower}', "");
Expect(0, 120780, '\P{^Sentence_Break=lower}', "");
Expect(1, 120779, '\p{Sentence_Break=-	Lower}', "");
Expect(0, 120779, '\p{^Sentence_Break=-	Lower}', "");
Expect(0, 120779, '\P{Sentence_Break=-	Lower}', "");
Expect(1, 120779, '\P{^Sentence_Break=-	Lower}', "");
Expect(0, 120780, '\p{Sentence_Break=-	Lower}', "");
Expect(1, 120780, '\p{^Sentence_Break=-	Lower}', "");
Expect(1, 120780, '\P{Sentence_Break=-	Lower}', "");
Expect(0, 120780, '\P{^Sentence_Break=-	Lower}', "");
Error('\p{SB= 	LO/a/}');
Error('\P{SB= 	LO/a/}');
Expect(1, 120779, '\p{SB=lo}', "");
Expect(0, 120779, '\p{^SB=lo}', "");
Expect(0, 120779, '\P{SB=lo}', "");
Expect(1, 120779, '\P{^SB=lo}', "");
Expect(0, 120780, '\p{SB=lo}', "");
Expect(1, 120780, '\p{^SB=lo}', "");
Expect(1, 120780, '\P{SB=lo}', "");
Expect(0, 120780, '\P{^SB=lo}', "");
Expect(1, 120779, '\p{SB=	LO}', "");
Expect(0, 120779, '\p{^SB=	LO}', "");
Expect(0, 120779, '\P{SB=	LO}', "");
Expect(1, 120779, '\P{^SB=	LO}', "");
Expect(0, 120780, '\p{SB=	LO}', "");
Expect(1, 120780, '\p{^SB=	LO}', "");
Expect(1, 120780, '\P{SB=	LO}', "");
Expect(0, 120780, '\P{^SB=	LO}', "");
Error('\p{Is_Sentence_Break=	:=lower}');
Error('\P{Is_Sentence_Break=	:=lower}');
Expect(1, 120779, '\p{Is_Sentence_Break:   lower}', "");
Expect(0, 120779, '\p{^Is_Sentence_Break:   lower}', "");
Expect(0, 120779, '\P{Is_Sentence_Break:   lower}', "");
Expect(1, 120779, '\P{^Is_Sentence_Break:   lower}', "");
Expect(0, 120780, '\p{Is_Sentence_Break:   lower}', "");
Expect(1, 120780, '\p{^Is_Sentence_Break:   lower}', "");
Expect(1, 120780, '\P{Is_Sentence_Break:   lower}', "");
Expect(0, 120780, '\P{^Is_Sentence_Break:   lower}', "");
Expect(1, 120779, '\p{Is_Sentence_Break=__lower}', "");
Expect(0, 120779, '\p{^Is_Sentence_Break=__lower}', "");
Expect(0, 120779, '\P{Is_Sentence_Break=__lower}', "");
Expect(1, 120779, '\P{^Is_Sentence_Break=__lower}', "");
Expect(0, 120780, '\p{Is_Sentence_Break=__lower}', "");
Expect(1, 120780, '\p{^Is_Sentence_Break=__lower}', "");
Expect(1, 120780, '\P{Is_Sentence_Break=__lower}', "");
Expect(0, 120780, '\P{^Is_Sentence_Break=__lower}', "");
Error('\p{Is_SB=:=LO}');
Error('\P{Is_SB=:=LO}');
Expect(1, 120779, '\p{Is_SB=lo}', "");
Expect(0, 120779, '\p{^Is_SB=lo}', "");
Expect(0, 120779, '\P{Is_SB=lo}', "");
Expect(1, 120779, '\P{^Is_SB=lo}', "");
Expect(0, 120780, '\p{Is_SB=lo}', "");
Expect(1, 120780, '\p{^Is_SB=lo}', "");
Expect(1, 120780, '\P{Is_SB=lo}', "");
Expect(0, 120780, '\P{^Is_SB=lo}', "");
Expect(1, 120779, '\p{Is_SB= LO}', "");
Expect(0, 120779, '\p{^Is_SB= LO}', "");
Expect(0, 120779, '\P{Is_SB= LO}', "");
Expect(1, 120779, '\P{^Is_SB= LO}', "");
Expect(0, 120780, '\p{Is_SB= LO}', "");
Expect(1, 120780, '\p{^Is_SB= LO}', "");
Expect(1, 120780, '\P{Is_SB= LO}', "");
Expect(0, 120780, '\P{^Is_SB= LO}', "");
Error('\p{Sentence_Break=-Upper/a/}');
Error('\P{Sentence_Break=-Upper/a/}');
Expect(1, 120778, '\p{Sentence_Break=upper}', "");
Expect(0, 120778, '\p{^Sentence_Break=upper}', "");
Expect(0, 120778, '\P{Sentence_Break=upper}', "");
Expect(1, 120778, '\P{^Sentence_Break=upper}', "");
Expect(0, 120779, '\p{Sentence_Break=upper}', "");
Expect(1, 120779, '\p{^Sentence_Break=upper}', "");
Expect(1, 120779, '\P{Sentence_Break=upper}', "");
Expect(0, 120779, '\P{^Sentence_Break=upper}', "");
Expect(1, 120778, '\p{Sentence_Break=		Upper}', "");
Expect(0, 120778, '\p{^Sentence_Break=		Upper}', "");
Expect(0, 120778, '\P{Sentence_Break=		Upper}', "");
Expect(1, 120778, '\P{^Sentence_Break=		Upper}', "");
Expect(0, 120779, '\p{Sentence_Break=		Upper}', "");
Expect(1, 120779, '\p{^Sentence_Break=		Upper}', "");
Expect(1, 120779, '\P{Sentence_Break=		Upper}', "");
Expect(0, 120779, '\P{^Sentence_Break=		Upper}', "");
Error('\p{SB=-UP/a/}');
Error('\P{SB=-UP/a/}');
Expect(1, 120778, '\p{SB=up}', "");
Expect(0, 120778, '\p{^SB=up}', "");
Expect(0, 120778, '\P{SB=up}', "");
Expect(1, 120778, '\P{^SB=up}', "");
Expect(0, 120779, '\p{SB=up}', "");
Expect(1, 120779, '\p{^SB=up}', "");
Expect(1, 120779, '\P{SB=up}', "");
Expect(0, 120779, '\P{^SB=up}', "");
Expect(1, 120778, '\p{SB=-	UP}', "");
Expect(0, 120778, '\p{^SB=-	UP}', "");
Expect(0, 120778, '\P{SB=-	UP}', "");
Expect(1, 120778, '\P{^SB=-	UP}', "");
Expect(0, 120779, '\p{SB=-	UP}', "");
Expect(1, 120779, '\p{^SB=-	UP}', "");
Expect(1, 120779, '\P{SB=-	UP}', "");
Expect(0, 120779, '\P{^SB=-	UP}', "");
Error('\p{Is_Sentence_Break=-_UPPER:=}');
Error('\P{Is_Sentence_Break=-_UPPER:=}');
Expect(1, 120778, '\p{Is_Sentence_Break=upper}', "");
Expect(0, 120778, '\p{^Is_Sentence_Break=upper}', "");
Expect(0, 120778, '\P{Is_Sentence_Break=upper}', "");
Expect(1, 120778, '\P{^Is_Sentence_Break=upper}', "");
Expect(0, 120779, '\p{Is_Sentence_Break=upper}', "");
Expect(1, 120779, '\p{^Is_Sentence_Break=upper}', "");
Expect(1, 120779, '\P{Is_Sentence_Break=upper}', "");
Expect(0, 120779, '\P{^Is_Sentence_Break=upper}', "");
Expect(1, 120778, '\p{Is_Sentence_Break=  upper}', "");
Expect(0, 120778, '\p{^Is_Sentence_Break=  upper}', "");
Expect(0, 120778, '\P{Is_Sentence_Break=  upper}', "");
Expect(1, 120778, '\P{^Is_Sentence_Break=  upper}', "");
Expect(0, 120779, '\p{Is_Sentence_Break=  upper}', "");
Expect(1, 120779, '\p{^Is_Sentence_Break=  upper}', "");
Expect(1, 120779, '\P{Is_Sentence_Break=  upper}', "");
Expect(0, 120779, '\P{^Is_Sentence_Break=  upper}', "");
Error('\p{Is_SB=_ UP:=}');
Error('\P{Is_SB=_ UP:=}');
Expect(1, 120778, '\p{Is_SB=up}', "");
Expect(0, 120778, '\p{^Is_SB=up}', "");
Expect(0, 120778, '\P{Is_SB=up}', "");
Expect(1, 120778, '\P{^Is_SB=up}', "");
Expect(0, 120779, '\p{Is_SB=up}', "");
Expect(1, 120779, '\p{^Is_SB=up}', "");
Expect(1, 120779, '\P{Is_SB=up}', "");
Expect(0, 120779, '\P{^Is_SB=up}', "");
Expect(1, 120778, '\p{Is_SB=-UP}', "");
Expect(0, 120778, '\p{^Is_SB=-UP}', "");
Expect(0, 120778, '\P{Is_SB=-UP}', "");
Expect(1, 120778, '\P{^Is_SB=-UP}', "");
Expect(0, 120779, '\p{Is_SB=-UP}', "");
Expect(1, 120779, '\p{^Is_SB=-UP}', "");
Expect(1, 120779, '\P{Is_SB=-UP}', "");
Expect(0, 120779, '\P{^Is_SB=-UP}', "");
Error('\p{Sentence_Break=  SEP:=}');
Error('\P{Sentence_Break=  SEP:=}');
Expect(1, 8233, '\p{Sentence_Break=sep}', "");
Expect(0, 8233, '\p{^Sentence_Break=sep}', "");
Expect(0, 8233, '\P{Sentence_Break=sep}', "");
Expect(1, 8233, '\P{^Sentence_Break=sep}', "");
Expect(0, 8234, '\p{Sentence_Break=sep}', "");
Expect(1, 8234, '\p{^Sentence_Break=sep}', "");
Expect(1, 8234, '\P{Sentence_Break=sep}', "");
Expect(0, 8234, '\P{^Sentence_Break=sep}', "");
Expect(1, 8233, '\p{Sentence_Break=	_Sep}', "");
Expect(0, 8233, '\p{^Sentence_Break=	_Sep}', "");
Expect(0, 8233, '\P{Sentence_Break=	_Sep}', "");
Expect(1, 8233, '\P{^Sentence_Break=	_Sep}', "");
Expect(0, 8234, '\p{Sentence_Break=	_Sep}', "");
Expect(1, 8234, '\p{^Sentence_Break=	_Sep}', "");
Expect(1, 8234, '\P{Sentence_Break=	_Sep}', "");
Expect(0, 8234, '\P{^Sentence_Break=	_Sep}', "");
Error('\p{SB=	/a/SE}');
Error('\P{SB=	/a/SE}');
Expect(1, 8233, '\p{SB=se}', "");
Expect(0, 8233, '\p{^SB=se}', "");
Expect(0, 8233, '\P{SB=se}', "");
Expect(1, 8233, '\P{^SB=se}', "");
Expect(0, 8234, '\p{SB=se}', "");
Expect(1, 8234, '\p{^SB=se}', "");
Expect(1, 8234, '\P{SB=se}', "");
Expect(0, 8234, '\P{^SB=se}', "");
Expect(1, 8233, '\p{SB=_SE}', "");
Expect(0, 8233, '\p{^SB=_SE}', "");
Expect(0, 8233, '\P{SB=_SE}', "");
Expect(1, 8233, '\P{^SB=_SE}', "");
Expect(0, 8234, '\p{SB=_SE}', "");
Expect(1, 8234, '\p{^SB=_SE}', "");
Expect(1, 8234, '\P{SB=_SE}', "");
Expect(0, 8234, '\P{^SB=_SE}', "");
Error('\p{Is_Sentence_Break=_	Sep:=}');
Error('\P{Is_Sentence_Break=_	Sep:=}');
Expect(1, 8233, '\p{Is_Sentence_Break:	sep}', "");
Expect(0, 8233, '\p{^Is_Sentence_Break:	sep}', "");
Expect(0, 8233, '\P{Is_Sentence_Break:	sep}', "");
Expect(1, 8233, '\P{^Is_Sentence_Break:	sep}', "");
Expect(0, 8234, '\p{Is_Sentence_Break:	sep}', "");
Expect(1, 8234, '\p{^Is_Sentence_Break:	sep}', "");
Expect(1, 8234, '\P{Is_Sentence_Break:	sep}', "");
Expect(0, 8234, '\P{^Is_Sentence_Break:	sep}', "");
Expect(1, 8233, '\p{Is_Sentence_Break=	 Sep}', "");
Expect(0, 8233, '\p{^Is_Sentence_Break=	 Sep}', "");
Expect(0, 8233, '\P{Is_Sentence_Break=	 Sep}', "");
Expect(1, 8233, '\P{^Is_Sentence_Break=	 Sep}', "");
Expect(0, 8234, '\p{Is_Sentence_Break=	 Sep}', "");
Expect(1, 8234, '\p{^Is_Sentence_Break=	 Sep}', "");
Expect(1, 8234, '\P{Is_Sentence_Break=	 Sep}', "");
Expect(0, 8234, '\P{^Is_Sentence_Break=	 Sep}', "");
Error('\p{Is_SB=/a/-SE}');
Error('\P{Is_SB=/a/-SE}');
Expect(1, 8233, '\p{Is_SB=se}', "");
Expect(0, 8233, '\p{^Is_SB=se}', "");
Expect(0, 8233, '\P{Is_SB=se}', "");
Expect(1, 8233, '\P{^Is_SB=se}', "");
Expect(0, 8234, '\p{Is_SB=se}', "");
Expect(1, 8234, '\p{^Is_SB=se}', "");
Expect(1, 8234, '\P{Is_SB=se}', "");
Expect(0, 8234, '\P{^Is_SB=se}', "");
Expect(1, 8233, '\p{Is_SB= se}', "");
Expect(0, 8233, '\p{^Is_SB= se}', "");
Expect(0, 8233, '\P{Is_SB= se}', "");
Expect(1, 8233, '\P{^Is_SB= se}', "");
Expect(0, 8234, '\p{Is_SB= se}', "");
Expect(1, 8234, '\p{^Is_SB= se}', "");
Expect(1, 8234, '\P{Is_SB= se}', "");
Expect(0, 8234, '\P{^Is_SB= se}', "");
Error('\p{Sentence_Break=/a/	extend}');
Error('\P{Sentence_Break=/a/	extend}');
Expect(1, 917999, '\p{Sentence_Break=extend}', "");
Expect(0, 917999, '\p{^Sentence_Break=extend}', "");
Expect(0, 917999, '\P{Sentence_Break=extend}', "");
Expect(1, 917999, '\P{^Sentence_Break=extend}', "");
Expect(0, 918000, '\p{Sentence_Break=extend}', "");
Expect(1, 918000, '\p{^Sentence_Break=extend}', "");
Expect(1, 918000, '\P{Sentence_Break=extend}', "");
Expect(0, 918000, '\P{^Sentence_Break=extend}', "");
Expect(1, 917999, '\p{Sentence_Break=	_EXTEND}', "");
Expect(0, 917999, '\p{^Sentence_Break=	_EXTEND}', "");
Expect(0, 917999, '\P{Sentence_Break=	_EXTEND}', "");
Expect(1, 917999, '\P{^Sentence_Break=	_EXTEND}', "");
Expect(0, 918000, '\p{Sentence_Break=	_EXTEND}', "");
Expect(1, 918000, '\p{^Sentence_Break=	_EXTEND}', "");
Expect(1, 918000, '\P{Sentence_Break=	_EXTEND}', "");
Expect(0, 918000, '\P{^Sentence_Break=	_EXTEND}', "");
Error('\p{SB=/a/ex}');
Error('\P{SB=/a/ex}');
Expect(1, 917999, '\p{SB=ex}', "");
Expect(0, 917999, '\p{^SB=ex}', "");
Expect(0, 917999, '\P{SB=ex}', "");
Expect(1, 917999, '\P{^SB=ex}', "");
Expect(0, 918000, '\p{SB=ex}', "");
Expect(1, 918000, '\p{^SB=ex}', "");
Expect(1, 918000, '\P{SB=ex}', "");
Expect(0, 918000, '\P{^SB=ex}', "");
Expect(1, 917999, '\p{SB= _ex}', "");
Expect(0, 917999, '\p{^SB= _ex}', "");
Expect(0, 917999, '\P{SB= _ex}', "");
Expect(1, 917999, '\P{^SB= _ex}', "");
Expect(0, 918000, '\p{SB= _ex}', "");
Expect(1, 918000, '\p{^SB= _ex}', "");
Expect(1, 918000, '\P{SB= _ex}', "");
Expect(0, 918000, '\P{^SB= _ex}', "");
Error('\p{Is_Sentence_Break= _EXTEND:=}');
Error('\P{Is_Sentence_Break= _EXTEND:=}');
Expect(1, 917999, '\p{Is_Sentence_Break:   extend}', "");
Expect(0, 917999, '\p{^Is_Sentence_Break:   extend}', "");
Expect(0, 917999, '\P{Is_Sentence_Break:   extend}', "");
Expect(1, 917999, '\P{^Is_Sentence_Break:   extend}', "");
Expect(0, 918000, '\p{Is_Sentence_Break:   extend}', "");
Expect(1, 918000, '\p{^Is_Sentence_Break:   extend}', "");
Expect(1, 918000, '\P{Is_Sentence_Break:   extend}', "");
Expect(0, 918000, '\P{^Is_Sentence_Break:   extend}', "");
Expect(1, 917999, '\p{Is_Sentence_Break= -EXTEND}', "");
Expect(0, 917999, '\p{^Is_Sentence_Break= -EXTEND}', "");
Expect(0, 917999, '\P{Is_Sentence_Break= -EXTEND}', "");
Expect(1, 917999, '\P{^Is_Sentence_Break= -EXTEND}', "");
Expect(0, 918000, '\p{Is_Sentence_Break= -EXTEND}', "");
Expect(1, 918000, '\p{^Is_Sentence_Break= -EXTEND}', "");
Expect(1, 918000, '\P{Is_Sentence_Break= -EXTEND}', "");
Expect(0, 918000, '\P{^Is_Sentence_Break= -EXTEND}', "");
Error('\p{Is_SB=:=_EX}');
Error('\P{Is_SB=:=_EX}');
Expect(1, 917999, '\p{Is_SB=ex}', "");
Expect(0, 917999, '\p{^Is_SB=ex}', "");
Expect(0, 917999, '\P{Is_SB=ex}', "");
Expect(1, 917999, '\P{^Is_SB=ex}', "");
Expect(0, 918000, '\p{Is_SB=ex}', "");
Expect(1, 918000, '\p{^Is_SB=ex}', "");
Expect(1, 918000, '\P{Is_SB=ex}', "");
Expect(0, 918000, '\P{^Is_SB=ex}', "");
Expect(1, 917999, '\p{Is_SB=  ex}', "");
Expect(0, 917999, '\p{^Is_SB=  ex}', "");
Expect(0, 917999, '\P{Is_SB=  ex}', "");
Expect(1, 917999, '\P{^Is_SB=  ex}', "");
Expect(0, 918000, '\p{Is_SB=  ex}', "");
Expect(1, 918000, '\p{^Is_SB=  ex}', "");
Expect(1, 918000, '\P{Is_SB=  ex}', "");
Expect(0, 918000, '\P{^Is_SB=  ex}', "");
Error('\p{Sentence_Break=/a/-sterm}');
Error('\P{Sentence_Break=/a/-sterm}');
Expect(1, 70086, '\p{Sentence_Break=sterm}', "");
Expect(0, 70086, '\p{^Sentence_Break=sterm}', "");
Expect(0, 70086, '\P{Sentence_Break=sterm}', "");
Expect(1, 70086, '\P{^Sentence_Break=sterm}', "");
Expect(0, 70087, '\p{Sentence_Break=sterm}', "");
Expect(1, 70087, '\p{^Sentence_Break=sterm}', "");
Expect(1, 70087, '\P{Sentence_Break=sterm}', "");
Expect(0, 70087, '\P{^Sentence_Break=sterm}', "");
Expect(1, 70086, '\p{Sentence_Break=STerm}', "");
Expect(0, 70086, '\p{^Sentence_Break=STerm}', "");
Expect(0, 70086, '\P{Sentence_Break=STerm}', "");
Expect(1, 70086, '\P{^Sentence_Break=STerm}', "");
Expect(0, 70087, '\p{Sentence_Break=STerm}', "");
Expect(1, 70087, '\p{^Sentence_Break=STerm}', "");
Expect(1, 70087, '\P{Sentence_Break=STerm}', "");
Expect(0, 70087, '\P{^Sentence_Break=STerm}', "");
Error('\p{SB=-	ST/a/}');
Error('\P{SB=-	ST/a/}');
Expect(1, 70086, '\p{SB=st}', "");
Expect(0, 70086, '\p{^SB=st}', "");
Expect(0, 70086, '\P{SB=st}', "");
Expect(1, 70086, '\P{^SB=st}', "");
Expect(0, 70087, '\p{SB=st}', "");
Expect(1, 70087, '\p{^SB=st}', "");
Expect(1, 70087, '\P{SB=st}', "");
Expect(0, 70087, '\P{^SB=st}', "");
Expect(1, 70086, '\p{SB=_ST}', "");
Expect(0, 70086, '\p{^SB=_ST}', "");
Expect(0, 70086, '\P{SB=_ST}', "");
Expect(1, 70086, '\P{^SB=_ST}', "");
Expect(0, 70087, '\p{SB=_ST}', "");
Expect(1, 70087, '\p{^SB=_ST}', "");
Expect(1, 70087, '\P{SB=_ST}', "");
Expect(0, 70087, '\P{^SB=_ST}', "");
Error('\p{Is_Sentence_Break=		sterm/a/}');
Error('\P{Is_Sentence_Break=		sterm/a/}');
Expect(1, 70086, '\p{Is_Sentence_Break=sterm}', "");
Expect(0, 70086, '\p{^Is_Sentence_Break=sterm}', "");
Expect(0, 70086, '\P{Is_Sentence_Break=sterm}', "");
Expect(1, 70086, '\P{^Is_Sentence_Break=sterm}', "");
Expect(0, 70087, '\p{Is_Sentence_Break=sterm}', "");
Expect(1, 70087, '\p{^Is_Sentence_Break=sterm}', "");
Expect(1, 70087, '\P{Is_Sentence_Break=sterm}', "");
Expect(0, 70087, '\P{^Is_Sentence_Break=sterm}', "");
Expect(1, 70086, '\p{Is_Sentence_Break=_STERM}', "");
Expect(0, 70086, '\p{^Is_Sentence_Break=_STERM}', "");
Expect(0, 70086, '\P{Is_Sentence_Break=_STERM}', "");
Expect(1, 70086, '\P{^Is_Sentence_Break=_STERM}', "");
Expect(0, 70087, '\p{Is_Sentence_Break=_STERM}', "");
Expect(1, 70087, '\p{^Is_Sentence_Break=_STERM}', "");
Expect(1, 70087, '\P{Is_Sentence_Break=_STERM}', "");
Expect(0, 70087, '\P{^Is_Sentence_Break=_STERM}', "");
Error('\p{Is_SB:   _ST:=}');
Error('\P{Is_SB:   _ST:=}');
Expect(1, 70086, '\p{Is_SB=st}', "");
Expect(0, 70086, '\p{^Is_SB=st}', "");
Expect(0, 70086, '\P{Is_SB=st}', "");
Expect(1, 70086, '\P{^Is_SB=st}', "");
Expect(0, 70087, '\p{Is_SB=st}', "");
Expect(1, 70087, '\p{^Is_SB=st}', "");
Expect(1, 70087, '\P{Is_SB=st}', "");
Expect(0, 70087, '\P{^Is_SB=st}', "");
Expect(1, 70086, '\p{Is_SB=	st}', "");
Expect(0, 70086, '\p{^Is_SB=	st}', "");
Expect(0, 70086, '\P{Is_SB=	st}', "");
Expect(1, 70086, '\P{^Is_SB=	st}', "");
Expect(0, 70087, '\p{Is_SB=	st}', "");
Expect(1, 70087, '\p{^Is_SB=	st}', "");
Expect(1, 70087, '\P{Is_SB=	st}', "");
Expect(0, 70087, '\P{^Is_SB=	st}', "");
Error('\p{Sentence_Break=-_LF:=}');
Error('\P{Sentence_Break=-_LF:=}');
Expect(1, 10, '\p{Sentence_Break=lf}', "");
Expect(0, 10, '\p{^Sentence_Break=lf}', "");
Expect(0, 10, '\P{Sentence_Break=lf}', "");
Expect(1, 10, '\P{^Sentence_Break=lf}', "");
Expect(0, 11, '\p{Sentence_Break=lf}', "");
Expect(1, 11, '\p{^Sentence_Break=lf}', "");
Expect(1, 11, '\P{Sentence_Break=lf}', "");
Expect(0, 11, '\P{^Sentence_Break=lf}', "");
Expect(1, 10, '\p{Sentence_Break: - lf}', "");
Expect(0, 10, '\p{^Sentence_Break: - lf}', "");
Expect(0, 10, '\P{Sentence_Break: - lf}', "");
Expect(1, 10, '\P{^Sentence_Break: - lf}', "");
Expect(0, 11, '\p{Sentence_Break: - lf}', "");
Expect(1, 11, '\p{^Sentence_Break: - lf}', "");
Expect(1, 11, '\P{Sentence_Break: - lf}', "");
Expect(0, 11, '\P{^Sentence_Break: - lf}', "");
Error('\p{SB=	:=LF}');
Error('\P{SB=	:=LF}');
Expect(1, 10, '\p{SB=lf}', "");
Expect(0, 10, '\p{^SB=lf}', "");
Expect(0, 10, '\P{SB=lf}', "");
Expect(1, 10, '\P{^SB=lf}', "");
Expect(0, 11, '\p{SB=lf}', "");
Expect(1, 11, '\p{^SB=lf}', "");
Expect(1, 11, '\P{SB=lf}', "");
Expect(0, 11, '\P{^SB=lf}', "");
Expect(1, 10, '\p{SB= LF}', "");
Expect(0, 10, '\p{^SB= LF}', "");
Expect(0, 10, '\P{SB= LF}', "");
Expect(1, 10, '\P{^SB= LF}', "");
Expect(0, 11, '\p{SB= LF}', "");
Expect(1, 11, '\p{^SB= LF}', "");
Expect(1, 11, '\P{SB= LF}', "");
Expect(0, 11, '\P{^SB= LF}', "");
Error('\p{Is_Sentence_Break=	 LF/a/}');
Error('\P{Is_Sentence_Break=	 LF/a/}');
Expect(1, 10, '\p{Is_Sentence_Break=lf}', "");
Expect(0, 10, '\p{^Is_Sentence_Break=lf}', "");
Expect(0, 10, '\P{Is_Sentence_Break=lf}', "");
Expect(1, 10, '\P{^Is_Sentence_Break=lf}', "");
Expect(0, 11, '\p{Is_Sentence_Break=lf}', "");
Expect(1, 11, '\p{^Is_Sentence_Break=lf}', "");
Expect(1, 11, '\P{Is_Sentence_Break=lf}', "");
Expect(0, 11, '\P{^Is_Sentence_Break=lf}', "");
Expect(1, 10, '\p{Is_Sentence_Break=-_lf}', "");
Expect(0, 10, '\p{^Is_Sentence_Break=-_lf}', "");
Expect(0, 10, '\P{Is_Sentence_Break=-_lf}', "");
Expect(1, 10, '\P{^Is_Sentence_Break=-_lf}', "");
Expect(0, 11, '\p{Is_Sentence_Break=-_lf}', "");
Expect(1, 11, '\p{^Is_Sentence_Break=-_lf}', "");
Expect(1, 11, '\P{Is_Sentence_Break=-_lf}', "");
Expect(0, 11, '\P{^Is_Sentence_Break=-_lf}', "");
Error('\p{Is_SB=/a/_	LF}');
Error('\P{Is_SB=/a/_	LF}');
Expect(1, 10, '\p{Is_SB=lf}', "");
Expect(0, 10, '\p{^Is_SB=lf}', "");
Expect(0, 10, '\P{Is_SB=lf}', "");
Expect(1, 10, '\P{^Is_SB=lf}', "");
Expect(0, 11, '\p{Is_SB=lf}', "");
Expect(1, 11, '\p{^Is_SB=lf}', "");
Expect(1, 11, '\P{Is_SB=lf}', "");
Expect(0, 11, '\P{^Is_SB=lf}', "");
Expect(1, 10, '\p{Is_SB=-LF}', "");
Expect(0, 10, '\p{^Is_SB=-LF}', "");
Expect(0, 10, '\P{Is_SB=-LF}', "");
Expect(1, 10, '\P{^Is_SB=-LF}', "");
Expect(0, 11, '\p{Is_SB=-LF}', "");
Expect(1, 11, '\p{^Is_SB=-LF}', "");
Expect(1, 11, '\P{Is_SB=-LF}', "");
Expect(0, 11, '\P{^Is_SB=-LF}', "");
Error('\p{Sentence_Break=:=Close}');
Error('\P{Sentence_Break=:=Close}');
Expect(1, 65379, '\p{Sentence_Break=close}', "");
Expect(0, 65379, '\p{^Sentence_Break=close}', "");
Expect(0, 65379, '\P{Sentence_Break=close}', "");
Expect(1, 65379, '\P{^Sentence_Break=close}', "");
Expect(0, 65380, '\p{Sentence_Break=close}', "");
Expect(1, 65380, '\p{^Sentence_Break=close}', "");
Expect(1, 65380, '\P{Sentence_Break=close}', "");
Expect(0, 65380, '\P{^Sentence_Break=close}', "");
Expect(1, 65379, '\p{Sentence_Break=	-Close}', "");
Expect(0, 65379, '\p{^Sentence_Break=	-Close}', "");
Expect(0, 65379, '\P{Sentence_Break=	-Close}', "");
Expect(1, 65379, '\P{^Sentence_Break=	-Close}', "");
Expect(0, 65380, '\p{Sentence_Break=	-Close}', "");
Expect(1, 65380, '\p{^Sentence_Break=	-Close}', "");
Expect(1, 65380, '\P{Sentence_Break=	-Close}', "");
Expect(0, 65380, '\P{^Sentence_Break=	-Close}', "");
Error('\p{SB=:=CL}');
Error('\P{SB=:=CL}');
Expect(1, 65379, '\p{SB=cl}', "");
Expect(0, 65379, '\p{^SB=cl}', "");
Expect(0, 65379, '\P{SB=cl}', "");
Expect(1, 65379, '\P{^SB=cl}', "");
Expect(0, 65380, '\p{SB=cl}', "");
Expect(1, 65380, '\p{^SB=cl}', "");
Expect(1, 65380, '\P{SB=cl}', "");
Expect(0, 65380, '\P{^SB=cl}', "");
Expect(1, 65379, '\p{SB: 	-CL}', "");
Expect(0, 65379, '\p{^SB: 	-CL}', "");
Expect(0, 65379, '\P{SB: 	-CL}', "");
Expect(1, 65379, '\P{^SB: 	-CL}', "");
Expect(0, 65380, '\p{SB: 	-CL}', "");
Expect(1, 65380, '\p{^SB: 	-CL}', "");
Expect(1, 65380, '\P{SB: 	-CL}', "");
Expect(0, 65380, '\P{^SB: 	-CL}', "");
Error('\p{Is_Sentence_Break=:=-close}');
Error('\P{Is_Sentence_Break=:=-close}');
Expect(1, 65379, '\p{Is_Sentence_Break=close}', "");
Expect(0, 65379, '\p{^Is_Sentence_Break=close}', "");
Expect(0, 65379, '\P{Is_Sentence_Break=close}', "");
Expect(1, 65379, '\P{^Is_Sentence_Break=close}', "");
Expect(0, 65380, '\p{Is_Sentence_Break=close}', "");
Expect(1, 65380, '\p{^Is_Sentence_Break=close}', "");
Expect(1, 65380, '\P{Is_Sentence_Break=close}', "");
Expect(0, 65380, '\P{^Is_Sentence_Break=close}', "");
Expect(1, 65379, '\p{Is_Sentence_Break:_ close}', "");
Expect(0, 65379, '\p{^Is_Sentence_Break:_ close}', "");
Expect(0, 65379, '\P{Is_Sentence_Break:_ close}', "");
Expect(1, 65379, '\P{^Is_Sentence_Break:_ close}', "");
Expect(0, 65380, '\p{Is_Sentence_Break:_ close}', "");
Expect(1, 65380, '\p{^Is_Sentence_Break:_ close}', "");
Expect(1, 65380, '\P{Is_Sentence_Break:_ close}', "");
Expect(0, 65380, '\P{^Is_Sentence_Break:_ close}', "");
Error('\p{Is_SB=  cl/a/}');
Error('\P{Is_SB=  cl/a/}');
Expect(1, 65379, '\p{Is_SB=cl}', "");
Expect(0, 65379, '\p{^Is_SB=cl}', "");
Expect(0, 65379, '\P{Is_SB=cl}', "");
Expect(1, 65379, '\P{^Is_SB=cl}', "");
Expect(0, 65380, '\p{Is_SB=cl}', "");
Expect(1, 65380, '\p{^Is_SB=cl}', "");
Expect(1, 65380, '\P{Is_SB=cl}', "");
Expect(0, 65380, '\P{^Is_SB=cl}', "");
Expect(1, 65379, '\p{Is_SB:_CL}', "");
Expect(0, 65379, '\p{^Is_SB:_CL}', "");
Expect(0, 65379, '\P{Is_SB:_CL}', "");
Expect(1, 65379, '\P{^Is_SB:_CL}', "");
Expect(0, 65380, '\p{Is_SB:_CL}', "");
Expect(1, 65380, '\p{^Is_SB:_CL}', "");
Expect(1, 65380, '\P{Is_SB:_CL}', "");
Expect(0, 65380, '\P{^Is_SB:_CL}', "");
Error('\p{Script_Extensions=/a/	Gothic}');
Error('\P{Script_Extensions=/a/	Gothic}');
Expect(1, 66378, '\p{Script_Extensions=gothic}', "");
Expect(0, 66378, '\p{^Script_Extensions=gothic}', "");
Expect(0, 66378, '\P{Script_Extensions=gothic}', "");
Expect(1, 66378, '\P{^Script_Extensions=gothic}', "");
Expect(0, 66379, '\p{Script_Extensions=gothic}', "");
Expect(1, 66379, '\p{^Script_Extensions=gothic}', "");
Expect(1, 66379, '\P{Script_Extensions=gothic}', "");
Expect(0, 66379, '\P{^Script_Extensions=gothic}', "");
Expect(1, 66378, '\p{Script_Extensions=-Gothic}', "");
Expect(0, 66378, '\p{^Script_Extensions=-Gothic}', "");
Expect(0, 66378, '\P{Script_Extensions=-Gothic}', "");
Expect(1, 66378, '\P{^Script_Extensions=-Gothic}', "");
Expect(0, 66379, '\p{Script_Extensions=-Gothic}', "");
Expect(1, 66379, '\p{^Script_Extensions=-Gothic}', "");
Expect(1, 66379, '\P{Script_Extensions=-Gothic}', "");
Expect(0, 66379, '\P{^Script_Extensions=-Gothic}', "");
Error('\p{Scx=/a/-	Goth}');
Error('\P{Scx=/a/-	Goth}');
Expect(1, 66378, '\p{Scx:goth}', "");
Expect(0, 66378, '\p{^Scx:goth}', "");
Expect(0, 66378, '\P{Scx:goth}', "");
Expect(1, 66378, '\P{^Scx:goth}', "");
Expect(0, 66379, '\p{Scx:goth}', "");
Expect(1, 66379, '\p{^Scx:goth}', "");
Expect(1, 66379, '\P{Scx:goth}', "");
Expect(0, 66379, '\P{^Scx:goth}', "");
Expect(1, 66378, '\p{Scx=_Goth}', "");
Expect(0, 66378, '\p{^Scx=_Goth}', "");
Expect(0, 66378, '\P{Scx=_Goth}', "");
Expect(1, 66378, '\P{^Scx=_Goth}', "");
Expect(0, 66379, '\p{Scx=_Goth}', "");
Expect(1, 66379, '\p{^Scx=_Goth}', "");
Expect(1, 66379, '\P{Scx=_Goth}', "");
Expect(0, 66379, '\P{^Scx=_Goth}', "");
Error('\p{Is_Script_Extensions=_Gothic:=}');
Error('\P{Is_Script_Extensions=_Gothic:=}');
Expect(1, 66378, '\p{Is_Script_Extensions=gothic}', "");
Expect(0, 66378, '\p{^Is_Script_Extensions=gothic}', "");
Expect(0, 66378, '\P{Is_Script_Extensions=gothic}', "");
Expect(1, 66378, '\P{^Is_Script_Extensions=gothic}', "");
Expect(0, 66379, '\p{Is_Script_Extensions=gothic}', "");
Expect(1, 66379, '\p{^Is_Script_Extensions=gothic}', "");
Expect(1, 66379, '\P{Is_Script_Extensions=gothic}', "");
Expect(0, 66379, '\P{^Is_Script_Extensions=gothic}', "");
Expect(1, 66378, '\p{Is_Script_Extensions=Gothic}', "");
Expect(0, 66378, '\p{^Is_Script_Extensions=Gothic}', "");
Expect(0, 66378, '\P{Is_Script_Extensions=Gothic}', "");
Expect(1, 66378, '\P{^Is_Script_Extensions=Gothic}', "");
Expect(0, 66379, '\p{Is_Script_Extensions=Gothic}', "");
Expect(1, 66379, '\p{^Is_Script_Extensions=Gothic}', "");
Expect(1, 66379, '\P{Is_Script_Extensions=Gothic}', "");
Expect(0, 66379, '\P{^Is_Script_Extensions=Gothic}', "");
Error('\p{Is_Scx=--Goth:=}');
Error('\P{Is_Scx=--Goth:=}');
Expect(1, 66378, '\p{Is_Scx=goth}', "");
Expect(0, 66378, '\p{^Is_Scx=goth}', "");
Expect(0, 66378, '\P{Is_Scx=goth}', "");
Expect(1, 66378, '\P{^Is_Scx=goth}', "");
Expect(0, 66379, '\p{Is_Scx=goth}', "");
Expect(1, 66379, '\p{^Is_Scx=goth}', "");
Expect(1, 66379, '\P{Is_Scx=goth}', "");
Expect(0, 66379, '\P{^Is_Scx=goth}', "");
Expect(1, 66378, '\p{Is_Scx=--GOTH}', "");
Expect(0, 66378, '\p{^Is_Scx=--GOTH}', "");
Expect(0, 66378, '\P{Is_Scx=--GOTH}', "");
Expect(1, 66378, '\P{^Is_Scx=--GOTH}', "");
Expect(0, 66379, '\p{Is_Scx=--GOTH}', "");
Expect(1, 66379, '\p{^Is_Scx=--GOTH}', "");
Expect(1, 66379, '\P{Is_Scx=--GOTH}', "");
Expect(0, 66379, '\P{^Is_Scx=--GOTH}', "");
Error('\p{Script_Extensions=:=_-New_tai_lue}');
Error('\P{Script_Extensions=:=_-New_tai_lue}');
Expect(1, 6623, '\p{Script_Extensions=newtailue}', "");
Expect(0, 6623, '\p{^Script_Extensions=newtailue}', "");
Expect(0, 6623, '\P{Script_Extensions=newtailue}', "");
Expect(1, 6623, '\P{^Script_Extensions=newtailue}', "");
Expect(0, 6624, '\p{Script_Extensions=newtailue}', "");
Expect(1, 6624, '\p{^Script_Extensions=newtailue}', "");
Expect(1, 6624, '\P{Script_Extensions=newtailue}', "");
Expect(0, 6624, '\P{^Script_Extensions=newtailue}', "");
Expect(1, 6623, '\p{Script_Extensions:   - New_Tai_Lue}', "");
Expect(0, 6623, '\p{^Script_Extensions:   - New_Tai_Lue}', "");
Expect(0, 6623, '\P{Script_Extensions:   - New_Tai_Lue}', "");
Expect(1, 6623, '\P{^Script_Extensions:   - New_Tai_Lue}', "");
Expect(0, 6624, '\p{Script_Extensions:   - New_Tai_Lue}', "");
Expect(1, 6624, '\p{^Script_Extensions:   - New_Tai_Lue}', "");
Expect(1, 6624, '\P{Script_Extensions:   - New_Tai_Lue}', "");
Expect(0, 6624, '\P{^Script_Extensions:   - New_Tai_Lue}', "");
Error('\p{Scx=	/a/Talu}');
Error('\P{Scx=	/a/Talu}');
Expect(1, 6623, '\p{Scx=talu}', "");
Expect(0, 6623, '\p{^Scx=talu}', "");
Expect(0, 6623, '\P{Scx=talu}', "");
Expect(1, 6623, '\P{^Scx=talu}', "");
Expect(0, 6624, '\p{Scx=talu}', "");
Expect(1, 6624, '\p{^Scx=talu}', "");
Expect(1, 6624, '\P{Scx=talu}', "");
Expect(0, 6624, '\P{^Scx=talu}', "");
Expect(1, 6623, '\p{Scx=-Talu}', "");
Expect(0, 6623, '\p{^Scx=-Talu}', "");
Expect(0, 6623, '\P{Scx=-Talu}', "");
Expect(1, 6623, '\P{^Scx=-Talu}', "");
Expect(0, 6624, '\p{Scx=-Talu}', "");
Expect(1, 6624, '\p{^Scx=-Talu}', "");
Expect(1, 6624, '\P{Scx=-Talu}', "");
Expect(0, 6624, '\P{^Scx=-Talu}', "");
Error('\p{Is_Script_Extensions:--New_TAI_Lue/a/}');
Error('\P{Is_Script_Extensions:--New_TAI_Lue/a/}');
Expect(1, 6623, '\p{Is_Script_Extensions=newtailue}', "");
Expect(0, 6623, '\p{^Is_Script_Extensions=newtailue}', "");
Expect(0, 6623, '\P{Is_Script_Extensions=newtailue}', "");
Expect(1, 6623, '\P{^Is_Script_Extensions=newtailue}', "");
Expect(0, 6624, '\p{Is_Script_Extensions=newtailue}', "");
Expect(1, 6624, '\p{^Is_Script_Extensions=newtailue}', "");
Expect(1, 6624, '\P{Is_Script_Extensions=newtailue}', "");
Expect(0, 6624, '\P{^Is_Script_Extensions=newtailue}', "");
Expect(1, 6623, '\p{Is_Script_Extensions=--new_Tai_Lue}', "");
Expect(0, 6623, '\p{^Is_Script_Extensions=--new_Tai_Lue}', "");
Expect(0, 6623, '\P{Is_Script_Extensions=--new_Tai_Lue}', "");
Expect(1, 6623, '\P{^Is_Script_Extensions=--new_Tai_Lue}', "");
Expect(0, 6624, '\p{Is_Script_Extensions=--new_Tai_Lue}', "");
Expect(1, 6624, '\p{^Is_Script_Extensions=--new_Tai_Lue}', "");
Expect(1, 6624, '\P{Is_Script_Extensions=--new_Tai_Lue}', "");
Expect(0, 6624, '\P{^Is_Script_Extensions=--new_Tai_Lue}', "");
Error('\p{Is_Scx=:=-Talu}');
Error('\P{Is_Scx=:=-Talu}');
Expect(1, 6623, '\p{Is_Scx:   talu}', "");
Expect(0, 6623, '\p{^Is_Scx:   talu}', "");
Expect(0, 6623, '\P{Is_Scx:   talu}', "");
Expect(1, 6623, '\P{^Is_Scx:   talu}', "");
Expect(0, 6624, '\p{Is_Scx:   talu}', "");
Expect(1, 6624, '\p{^Is_Scx:   talu}', "");
Expect(1, 6624, '\P{Is_Scx:   talu}', "");
Expect(0, 6624, '\P{^Is_Scx:   talu}', "");
Expect(1, 6623, '\p{Is_Scx=-TALU}', "");
Expect(0, 6623, '\p{^Is_Scx=-TALU}', "");
Expect(0, 6623, '\P{Is_Scx=-TALU}', "");
Expect(1, 6623, '\P{^Is_Scx=-TALU}', "");
Expect(0, 6624, '\p{Is_Scx=-TALU}', "");
Expect(1, 6624, '\p{^Is_Scx=-TALU}', "");
Expect(1, 6624, '\P{Is_Scx=-TALU}', "");
Expect(0, 6624, '\P{^Is_Scx=-TALU}', "");
Error('\p{Script_Extensions=_/a/old_Italic}');
Error('\P{Script_Extensions=_/a/old_Italic}');
Expect(1, 66339, '\p{Script_Extensions=olditalic}', "");
Expect(0, 66339, '\p{^Script_Extensions=olditalic}', "");
Expect(0, 66339, '\P{Script_Extensions=olditalic}', "");
Expect(1, 66339, '\P{^Script_Extensions=olditalic}', "");
Expect(0, 66340, '\p{Script_Extensions=olditalic}', "");
Expect(1, 66340, '\p{^Script_Extensions=olditalic}', "");
Expect(1, 66340, '\P{Script_Extensions=olditalic}', "");
Expect(0, 66340, '\P{^Script_Extensions=olditalic}', "");
Expect(1, 66339, '\p{Script_Extensions=_-Old_Italic}', "");
Expect(0, 66339, '\p{^Script_Extensions=_-Old_Italic}', "");
Expect(0, 66339, '\P{Script_Extensions=_-Old_Italic}', "");
Expect(1, 66339, '\P{^Script_Extensions=_-Old_Italic}', "");
Expect(0, 66340, '\p{Script_Extensions=_-Old_Italic}', "");
Expect(1, 66340, '\p{^Script_Extensions=_-Old_Italic}', "");
Expect(1, 66340, '\P{Script_Extensions=_-Old_Italic}', "");
Expect(0, 66340, '\P{^Script_Extensions=_-Old_Italic}', "");
Error('\p{Scx=:=ITAL}');
Error('\P{Scx=:=ITAL}');
Expect(1, 66339, '\p{Scx=ital}', "");
Expect(0, 66339, '\p{^Scx=ital}', "");
Expect(0, 66339, '\P{Scx=ital}', "");
Expect(1, 66339, '\P{^Scx=ital}', "");
Expect(0, 66340, '\p{Scx=ital}', "");
Expect(1, 66340, '\p{^Scx=ital}', "");
Expect(1, 66340, '\P{Scx=ital}', "");
Expect(0, 66340, '\P{^Scx=ital}', "");
Expect(1, 66339, '\p{Scx=  Ital}', "");
Expect(0, 66339, '\p{^Scx=  Ital}', "");
Expect(0, 66339, '\P{Scx=  Ital}', "");
Expect(1, 66339, '\P{^Scx=  Ital}', "");
Expect(0, 66340, '\p{Scx=  Ital}', "");
Expect(1, 66340, '\p{^Scx=  Ital}', "");
Expect(1, 66340, '\P{Scx=  Ital}', "");
Expect(0, 66340, '\P{^Scx=  Ital}', "");
Error('\p{Is_Script_Extensions=/a/-_old_Italic}');
Error('\P{Is_Script_Extensions=/a/-_old_Italic}');
Expect(1, 66339, '\p{Is_Script_Extensions=olditalic}', "");
Expect(0, 66339, '\p{^Is_Script_Extensions=olditalic}', "");
Expect(0, 66339, '\P{Is_Script_Extensions=olditalic}', "");
Expect(1, 66339, '\P{^Is_Script_Extensions=olditalic}', "");
Expect(0, 66340, '\p{Is_Script_Extensions=olditalic}', "");
Expect(1, 66340, '\p{^Is_Script_Extensions=olditalic}', "");
Expect(1, 66340, '\P{Is_Script_Extensions=olditalic}', "");
Expect(0, 66340, '\P{^Is_Script_Extensions=olditalic}', "");
Expect(1, 66339, '\p{Is_Script_Extensions=	_Old_ITALIC}', "");
Expect(0, 66339, '\p{^Is_Script_Extensions=	_Old_ITALIC}', "");
Expect(0, 66339, '\P{Is_Script_Extensions=	_Old_ITALIC}', "");
Expect(1, 66339, '\P{^Is_Script_Extensions=	_Old_ITALIC}', "");
Expect(0, 66340, '\p{Is_Script_Extensions=	_Old_ITALIC}', "");
Expect(1, 66340, '\p{^Is_Script_Extensions=	_Old_ITALIC}', "");
Expect(1, 66340, '\P{Is_Script_Extensions=	_Old_ITALIC}', "");
Expect(0, 66340, '\P{^Is_Script_Extensions=	_Old_ITALIC}', "");
Error('\p{Is_Scx=-:=ITAL}');
Error('\P{Is_Scx=-:=ITAL}');
Expect(1, 66339, '\p{Is_Scx:   ital}', "");
Expect(0, 66339, '\p{^Is_Scx:   ital}', "");
Expect(0, 66339, '\P{Is_Scx:   ital}', "");
Expect(1, 66339, '\P{^Is_Scx:   ital}', "");
Expect(0, 66340, '\p{Is_Scx:   ital}', "");
Expect(1, 66340, '\p{^Is_Scx:   ital}', "");
Expect(1, 66340, '\P{Is_Scx:   ital}', "");
Expect(0, 66340, '\P{^Is_Scx:   ital}', "");
Expect(1, 66339, '\p{Is_Scx=	 Ital}', "");
Expect(0, 66339, '\p{^Is_Scx=	 Ital}', "");
Expect(0, 66339, '\P{Is_Scx=	 Ital}', "");
Expect(1, 66339, '\P{^Is_Scx=	 Ital}', "");
Expect(0, 66340, '\p{Is_Scx=	 Ital}', "");
Expect(1, 66340, '\p{^Is_Scx=	 Ital}', "");
Expect(1, 66340, '\P{Is_Scx=	 Ital}', "");
Expect(0, 66340, '\P{^Is_Scx=	 Ital}', "");
Error('\p{Script_Extensions=:=	 Bamum}');
Error('\P{Script_Extensions=:=	 Bamum}');
Expect(1, 92728, '\p{Script_Extensions=bamum}', "");
Expect(0, 92728, '\p{^Script_Extensions=bamum}', "");
Expect(0, 92728, '\P{Script_Extensions=bamum}', "");
Expect(1, 92728, '\P{^Script_Extensions=bamum}', "");
Expect(0, 92729, '\p{Script_Extensions=bamum}', "");
Expect(1, 92729, '\p{^Script_Extensions=bamum}', "");
Expect(1, 92729, '\P{Script_Extensions=bamum}', "");
Expect(0, 92729, '\P{^Script_Extensions=bamum}', "");
Expect(1, 92728, '\p{Script_Extensions=		Bamum}', "");
Expect(0, 92728, '\p{^Script_Extensions=		Bamum}', "");
Expect(0, 92728, '\P{Script_Extensions=		Bamum}', "");
Expect(1, 92728, '\P{^Script_Extensions=		Bamum}', "");
Expect(0, 92729, '\p{Script_Extensions=		Bamum}', "");
Expect(1, 92729, '\p{^Script_Extensions=		Bamum}', "");
Expect(1, 92729, '\P{Script_Extensions=		Bamum}', "");
Expect(0, 92729, '\P{^Script_Extensions=		Bamum}', "");
Error('\p{Scx=	/a/Bamu}');
Error('\P{Scx=	/a/Bamu}');
Expect(1, 92728, '\p{Scx=bamu}', "");
Expect(0, 92728, '\p{^Scx=bamu}', "");
Expect(0, 92728, '\P{Scx=bamu}', "");
Expect(1, 92728, '\P{^Scx=bamu}', "");
Expect(0, 92729, '\p{Scx=bamu}', "");
Expect(1, 92729, '\p{^Scx=bamu}', "");
Expect(1, 92729, '\P{Scx=bamu}', "");
Expect(0, 92729, '\P{^Scx=bamu}', "");
Expect(1, 92728, '\p{Scx=_Bamu}', "");
Expect(0, 92728, '\p{^Scx=_Bamu}', "");
Expect(0, 92728, '\P{Scx=_Bamu}', "");
Expect(1, 92728, '\P{^Scx=_Bamu}', "");
Expect(0, 92729, '\p{Scx=_Bamu}', "");
Expect(1, 92729, '\p{^Scx=_Bamu}', "");
Expect(1, 92729, '\P{Scx=_Bamu}', "");
Expect(0, 92729, '\P{^Scx=_Bamu}', "");
Error('\p{Is_Script_Extensions=- bamum:=}');
Error('\P{Is_Script_Extensions=- bamum:=}');
Expect(1, 92728, '\p{Is_Script_Extensions=bamum}', "");
Expect(0, 92728, '\p{^Is_Script_Extensions=bamum}', "");
Expect(0, 92728, '\P{Is_Script_Extensions=bamum}', "");
Expect(1, 92728, '\P{^Is_Script_Extensions=bamum}', "");
Expect(0, 92729, '\p{Is_Script_Extensions=bamum}', "");
Expect(1, 92729, '\p{^Is_Script_Extensions=bamum}', "");
Expect(1, 92729, '\P{Is_Script_Extensions=bamum}', "");
Expect(0, 92729, '\P{^Is_Script_Extensions=bamum}', "");
Expect(1, 92728, '\p{Is_Script_Extensions=_ bamum}', "");
Expect(0, 92728, '\p{^Is_Script_Extensions=_ bamum}', "");
Expect(0, 92728, '\P{Is_Script_Extensions=_ bamum}', "");
Expect(1, 92728, '\P{^Is_Script_Extensions=_ bamum}', "");
Expect(0, 92729, '\p{Is_Script_Extensions=_ bamum}', "");
Expect(1, 92729, '\p{^Is_Script_Extensions=_ bamum}', "");
Expect(1, 92729, '\P{Is_Script_Extensions=_ bamum}', "");
Expect(0, 92729, '\P{^Is_Script_Extensions=_ bamum}', "");
Error('\p{Is_Scx=-:=bamu}');
Error('\P{Is_Scx=-:=bamu}');
Expect(1, 92728, '\p{Is_Scx=bamu}', "");
Expect(0, 92728, '\p{^Is_Scx=bamu}', "");
Expect(0, 92728, '\P{Is_Scx=bamu}', "");
Expect(1, 92728, '\P{^Is_Scx=bamu}', "");
Expect(0, 92729, '\p{Is_Scx=bamu}', "");
Expect(1, 92729, '\p{^Is_Scx=bamu}', "");
Expect(1, 92729, '\P{Is_Scx=bamu}', "");
Expect(0, 92729, '\P{^Is_Scx=bamu}', "");
Expect(1, 92728, '\p{Is_Scx: --Bamu}', "");
Expect(0, 92728, '\p{^Is_Scx: --Bamu}', "");
Expect(0, 92728, '\P{Is_Scx: --Bamu}', "");
Expect(1, 92728, '\P{^Is_Scx: --Bamu}', "");
Expect(0, 92729, '\p{Is_Scx: --Bamu}', "");
Expect(1, 92729, '\p{^Is_Scx: --Bamu}', "");
Expect(1, 92729, '\P{Is_Scx: --Bamu}', "");
Expect(0, 92729, '\P{^Is_Scx: --Bamu}', "");
Error('\p{Script_Extensions=-_BRAHMI/a/}');
Error('\P{Script_Extensions=-_BRAHMI/a/}');
Expect(1, 69743, '\p{Script_Extensions:	brahmi}', "");
Expect(0, 69743, '\p{^Script_Extensions:	brahmi}', "");
Expect(0, 69743, '\P{Script_Extensions:	brahmi}', "");
Expect(1, 69743, '\P{^Script_Extensions:	brahmi}', "");
Expect(0, 69744, '\p{Script_Extensions:	brahmi}', "");
Expect(1, 69744, '\p{^Script_Extensions:	brahmi}', "");
Expect(1, 69744, '\P{Script_Extensions:	brahmi}', "");
Expect(0, 69744, '\P{^Script_Extensions:	brahmi}', "");
Expect(1, 69743, '\p{Script_Extensions= -BRAHMI}', "");
Expect(0, 69743, '\p{^Script_Extensions= -BRAHMI}', "");
Expect(0, 69743, '\P{Script_Extensions= -BRAHMI}', "");
Expect(1, 69743, '\P{^Script_Extensions= -BRAHMI}', "");
Expect(0, 69744, '\p{Script_Extensions= -BRAHMI}', "");
Expect(1, 69744, '\p{^Script_Extensions= -BRAHMI}', "");
Expect(1, 69744, '\P{Script_Extensions= -BRAHMI}', "");
Expect(0, 69744, '\P{^Script_Extensions= -BRAHMI}', "");
Error('\p{Scx=/a/ 	Brah}');
Error('\P{Scx=/a/ 	Brah}');
Expect(1, 69743, '\p{Scx=brah}', "");
Expect(0, 69743, '\p{^Scx=brah}', "");
Expect(0, 69743, '\P{Scx=brah}', "");
Expect(1, 69743, '\P{^Scx=brah}', "");
Expect(0, 69744, '\p{Scx=brah}', "");
Expect(1, 69744, '\p{^Scx=brah}', "");
Expect(1, 69744, '\P{Scx=brah}', "");
Expect(0, 69744, '\P{^Scx=brah}', "");
Error('\p{Is_Script_Extensions=/a/_-Brahmi}');
Error('\P{Is_Script_Extensions=/a/_-Brahmi}');
Expect(1, 69743, '\p{Is_Script_Extensions=brahmi}', "");
Expect(0, 69743, '\p{^Is_Script_Extensions=brahmi}', "");
Expect(0, 69743, '\P{Is_Script_Extensions=brahmi}', "");
Expect(1, 69743, '\P{^Is_Script_Extensions=brahmi}', "");
Expect(0, 69744, '\p{Is_Script_Extensions=brahmi}', "");
Expect(1, 69744, '\p{^Is_Script_Extensions=brahmi}', "");
Expect(1, 69744, '\P{Is_Script_Extensions=brahmi}', "");
Expect(0, 69744, '\P{^Is_Script_Extensions=brahmi}', "");
Expect(1, 69743, '\p{Is_Script_Extensions=	Brahmi}', "");
Expect(0, 69743, '\p{^Is_Script_Extensions=	Brahmi}', "");
Expect(0, 69743, '\P{Is_Script_Extensions=	Brahmi}', "");
Expect(1, 69743, '\P{^Is_Script_Extensions=	Brahmi}', "");
Expect(0, 69744, '\p{Is_Script_Extensions=	Brahmi}', "");
Expect(1, 69744, '\p{^Is_Script_Extensions=	Brahmi}', "");
Expect(1, 69744, '\P{Is_Script_Extensions=	Brahmi}', "");
Expect(0, 69744, '\P{^Is_Script_Extensions=	Brahmi}', "");
Error('\p{Is_Scx=/a/ -Brah}');
Error('\P{Is_Scx=/a/ -Brah}');
Expect(1, 69743, '\p{Is_Scx=brah}', "");
Expect(0, 69743, '\p{^Is_Scx=brah}', "");
Expect(0, 69743, '\P{Is_Scx=brah}', "");
Expect(1, 69743, '\P{^Is_Scx=brah}', "");
Expect(0, 69744, '\p{Is_Scx=brah}', "");
Expect(1, 69744, '\p{^Is_Scx=brah}', "");
Expect(1, 69744, '\P{Is_Scx=brah}', "");
Expect(0, 69744, '\P{^Is_Scx=brah}', "");
Expect(1, 69743, '\p{Is_Scx=_Brah}', "");
Expect(0, 69743, '\p{^Is_Scx=_Brah}', "");
Expect(0, 69743, '\P{Is_Scx=_Brah}', "");
Expect(1, 69743, '\P{^Is_Scx=_Brah}', "");
Expect(0, 69744, '\p{Is_Scx=_Brah}', "");
Expect(1, 69744, '\p{^Is_Scx=_Brah}', "");
Expect(1, 69744, '\P{Is_Scx=_Brah}', "");
Expect(0, 69744, '\P{^Is_Scx=_Brah}', "");
Error('\p{Script_Extensions= _Inscriptional_parthian:=}');
Error('\P{Script_Extensions= _Inscriptional_parthian:=}');
Expect(1, 68447, '\p{Script_Extensions=inscriptionalparthian}', "");
Expect(0, 68447, '\p{^Script_Extensions=inscriptionalparthian}', "");
Expect(0, 68447, '\P{Script_Extensions=inscriptionalparthian}', "");
Expect(1, 68447, '\P{^Script_Extensions=inscriptionalparthian}', "");
Expect(0, 68448, '\p{Script_Extensions=inscriptionalparthian}', "");
Expect(1, 68448, '\p{^Script_Extensions=inscriptionalparthian}', "");
Expect(1, 68448, '\P{Script_Extensions=inscriptionalparthian}', "");
Expect(0, 68448, '\P{^Script_Extensions=inscriptionalparthian}', "");
Expect(1, 68447, '\p{Script_Extensions=	 inscriptional_parthian}', "");
Expect(0, 68447, '\p{^Script_Extensions=	 inscriptional_parthian}', "");
Expect(0, 68447, '\P{Script_Extensions=	 inscriptional_parthian}', "");
Expect(1, 68447, '\P{^Script_Extensions=	 inscriptional_parthian}', "");
Expect(0, 68448, '\p{Script_Extensions=	 inscriptional_parthian}', "");
Expect(1, 68448, '\p{^Script_Extensions=	 inscriptional_parthian}', "");
Expect(1, 68448, '\P{Script_Extensions=	 inscriptional_parthian}', "");
Expect(0, 68448, '\P{^Script_Extensions=	 inscriptional_parthian}', "");
Error('\p{Scx=__Prti:=}');
Error('\P{Scx=__Prti:=}');
Expect(1, 68447, '\p{Scx=prti}', "");
Expect(0, 68447, '\p{^Scx=prti}', "");
Expect(0, 68447, '\P{Scx=prti}', "");
Expect(1, 68447, '\P{^Scx=prti}', "");
Expect(0, 68448, '\p{Scx=prti}', "");
Expect(1, 68448, '\p{^Scx=prti}', "");
Expect(1, 68448, '\P{Scx=prti}', "");
Expect(0, 68448, '\P{^Scx=prti}', "");
Expect(1, 68447, '\p{Scx:	-PRTI}', "");
Expect(0, 68447, '\p{^Scx:	-PRTI}', "");
Expect(0, 68447, '\P{Scx:	-PRTI}', "");
Expect(1, 68447, '\P{^Scx:	-PRTI}', "");
Expect(0, 68448, '\p{Scx:	-PRTI}', "");
Expect(1, 68448, '\p{^Scx:	-PRTI}', "");
Expect(1, 68448, '\P{Scx:	-PRTI}', "");
Expect(0, 68448, '\P{^Scx:	-PRTI}', "");
Error('\p{Is_Script_Extensions=-/a/inscriptional_Parthian}');
Error('\P{Is_Script_Extensions=-/a/inscriptional_Parthian}');
Expect(1, 68447, '\p{Is_Script_Extensions=inscriptionalparthian}', "");
Expect(0, 68447, '\p{^Is_Script_Extensions=inscriptionalparthian}', "");
Expect(0, 68447, '\P{Is_Script_Extensions=inscriptionalparthian}', "");
Expect(1, 68447, '\P{^Is_Script_Extensions=inscriptionalparthian}', "");
Expect(0, 68448, '\p{Is_Script_Extensions=inscriptionalparthian}', "");
Expect(1, 68448, '\p{^Is_Script_Extensions=inscriptionalparthian}', "");
Expect(1, 68448, '\P{Is_Script_Extensions=inscriptionalparthian}', "");
Expect(0, 68448, '\P{^Is_Script_Extensions=inscriptionalparthian}', "");
Expect(1, 68447, '\p{Is_Script_Extensions=__Inscriptional_Parthian}', "");
Expect(0, 68447, '\p{^Is_Script_Extensions=__Inscriptional_Parthian}', "");
Expect(0, 68447, '\P{Is_Script_Extensions=__Inscriptional_Parthian}', "");
Expect(1, 68447, '\P{^Is_Script_Extensions=__Inscriptional_Parthian}', "");
Expect(0, 68448, '\p{Is_Script_Extensions=__Inscriptional_Parthian}', "");
Expect(1, 68448, '\p{^Is_Script_Extensions=__Inscriptional_Parthian}', "");
Expect(1, 68448, '\P{Is_Script_Extensions=__Inscriptional_Parthian}', "");
Expect(0, 68448, '\P{^Is_Script_Extensions=__Inscriptional_Parthian}', "");
Error('\p{Is_Scx::=_Prti}');
Error('\P{Is_Scx::=_Prti}');
Expect(1, 68447, '\p{Is_Scx=prti}', "");
Expect(0, 68447, '\p{^Is_Scx=prti}', "");
Expect(0, 68447, '\P{Is_Scx=prti}', "");
Expect(1, 68447, '\P{^Is_Scx=prti}', "");
Expect(0, 68448, '\p{Is_Scx=prti}', "");
Expect(1, 68448, '\p{^Is_Scx=prti}', "");
Expect(1, 68448, '\P{Is_Scx=prti}', "");
Expect(0, 68448, '\P{^Is_Scx=prti}', "");
Expect(1, 68447, '\p{Is_Scx= _PRTI}', "");
Expect(0, 68447, '\p{^Is_Scx= _PRTI}', "");
Expect(0, 68447, '\P{Is_Scx= _PRTI}', "");
Expect(1, 68447, '\P{^Is_Scx= _PRTI}', "");
Expect(0, 68448, '\p{Is_Scx= _PRTI}', "");
Expect(1, 68448, '\p{^Is_Scx= _PRTI}', "");
Expect(1, 68448, '\P{Is_Scx= _PRTI}', "");
Expect(0, 68448, '\P{^Is_Scx= _PRTI}', "");
Error('\p{Script_Extensions=/a/ -vai}');
Error('\P{Script_Extensions=/a/ -vai}');
Expect(1, 42539, '\p{Script_Extensions=vai}', "");
Expect(0, 42539, '\p{^Script_Extensions=vai}', "");
Expect(0, 42539, '\P{Script_Extensions=vai}', "");
Expect(1, 42539, '\P{^Script_Extensions=vai}', "");
Expect(0, 42540, '\p{Script_Extensions=vai}', "");
Expect(1, 42540, '\p{^Script_Extensions=vai}', "");
Expect(1, 42540, '\P{Script_Extensions=vai}', "");
Expect(0, 42540, '\P{^Script_Extensions=vai}', "");
Expect(1, 42539, '\p{Script_Extensions:   - Vai}', "");
Expect(0, 42539, '\p{^Script_Extensions:   - Vai}', "");
Expect(0, 42539, '\P{Script_Extensions:   - Vai}', "");
Expect(1, 42539, '\P{^Script_Extensions:   - Vai}', "");
Expect(0, 42540, '\p{Script_Extensions:   - Vai}', "");
Expect(1, 42540, '\p{^Script_Extensions:   - Vai}', "");
Expect(1, 42540, '\P{Script_Extensions:   - Vai}', "");
Expect(0, 42540, '\P{^Script_Extensions:   - Vai}', "");
Error('\p{Scx=/a/VAII}');
Error('\P{Scx=/a/VAII}');
Expect(1, 42539, '\p{Scx=vaii}', "");
Expect(0, 42539, '\p{^Scx=vaii}', "");
Expect(0, 42539, '\P{Scx=vaii}', "");
Expect(1, 42539, '\P{^Scx=vaii}', "");
Expect(0, 42540, '\p{Scx=vaii}', "");
Expect(1, 42540, '\p{^Scx=vaii}', "");
Expect(1, 42540, '\P{Scx=vaii}', "");
Expect(0, 42540, '\P{^Scx=vaii}', "");
Expect(1, 42539, '\p{Scx=		Vaii}', "");
Expect(0, 42539, '\p{^Scx=		Vaii}', "");
Expect(0, 42539, '\P{Scx=		Vaii}', "");
Expect(1, 42539, '\P{^Scx=		Vaii}', "");
Expect(0, 42540, '\p{Scx=		Vaii}', "");
Expect(1, 42540, '\p{^Scx=		Vaii}', "");
Expect(1, 42540, '\P{Scx=		Vaii}', "");
Expect(0, 42540, '\P{^Scx=		Vaii}', "");
Error('\p{Is_Script_Extensions=:=  Vai}');
Error('\P{Is_Script_Extensions=:=  Vai}');
Expect(1, 42539, '\p{Is_Script_Extensions=vai}', "");
Expect(0, 42539, '\p{^Is_Script_Extensions=vai}', "");
Expect(0, 42539, '\P{Is_Script_Extensions=vai}', "");
Expect(1, 42539, '\P{^Is_Script_Extensions=vai}', "");
Expect(0, 42540, '\p{Is_Script_Extensions=vai}', "");
Expect(1, 42540, '\p{^Is_Script_Extensions=vai}', "");
Expect(1, 42540, '\P{Is_Script_Extensions=vai}', "");
Expect(0, 42540, '\P{^Is_Script_Extensions=vai}', "");
Expect(1, 42539, '\p{Is_Script_Extensions= _Vai}', "");
Expect(0, 42539, '\p{^Is_Script_Extensions= _Vai}', "");
Expect(0, 42539, '\P{Is_Script_Extensions= _Vai}', "");
Expect(1, 42539, '\P{^Is_Script_Extensions= _Vai}', "");
Expect(0, 42540, '\p{Is_Script_Extensions= _Vai}', "");
Expect(1, 42540, '\p{^Is_Script_Extensions= _Vai}', "");
Expect(1, 42540, '\P{Is_Script_Extensions= _Vai}', "");
Expect(0, 42540, '\P{^Is_Script_Extensions= _Vai}', "");
Error('\p{Is_Scx=_ Vaii:=}');
Error('\P{Is_Scx=_ Vaii:=}');
Expect(1, 42539, '\p{Is_Scx=vaii}', "");
Expect(0, 42539, '\p{^Is_Scx=vaii}', "");
Expect(0, 42539, '\P{Is_Scx=vaii}', "");
Expect(1, 42539, '\P{^Is_Scx=vaii}', "");
Expect(0, 42540, '\p{Is_Scx=vaii}', "");
Expect(1, 42540, '\p{^Is_Scx=vaii}', "");
Expect(1, 42540, '\P{Is_Scx=vaii}', "");
Expect(0, 42540, '\P{^Is_Scx=vaii}', "");
Expect(1, 42539, '\p{Is_Scx:   -	Vaii}', "");
Expect(0, 42539, '\p{^Is_Scx:   -	Vaii}', "");
Expect(0, 42539, '\P{Is_Scx:   -	Vaii}', "");
Expect(1, 42539, '\P{^Is_Scx:   -	Vaii}', "");
Expect(0, 42540, '\p{Is_Scx:   -	Vaii}', "");
Expect(1, 42540, '\p{^Is_Scx:   -	Vaii}', "");
Expect(1, 42540, '\P{Is_Scx:   -	Vaii}', "");
Expect(0, 42540, '\P{^Is_Scx:   -	Vaii}', "");
Error('\p{Script_Extensions=/a/ Cherokee}');
Error('\P{Script_Extensions=/a/ Cherokee}');
Expect(1, 5108, '\p{Script_Extensions=cherokee}', "");
Expect(0, 5108, '\p{^Script_Extensions=cherokee}', "");
Expect(0, 5108, '\P{Script_Extensions=cherokee}', "");
Expect(1, 5108, '\P{^Script_Extensions=cherokee}', "");
Expect(0, 5109, '\p{Script_Extensions=cherokee}', "");
Expect(1, 5109, '\p{^Script_Extensions=cherokee}', "");
Expect(1, 5109, '\P{Script_Extensions=cherokee}', "");
Expect(0, 5109, '\P{^Script_Extensions=cherokee}', "");
Expect(1, 5108, '\p{Script_Extensions=  CHEROKEE}', "");
Expect(0, 5108, '\p{^Script_Extensions=  CHEROKEE}', "");
Expect(0, 5108, '\P{Script_Extensions=  CHEROKEE}', "");
Expect(1, 5108, '\P{^Script_Extensions=  CHEROKEE}', "");
Expect(0, 5109, '\p{Script_Extensions=  CHEROKEE}', "");
Expect(1, 5109, '\p{^Script_Extensions=  CHEROKEE}', "");
Expect(1, 5109, '\P{Script_Extensions=  CHEROKEE}', "");
Expect(0, 5109, '\P{^Script_Extensions=  CHEROKEE}', "");
Error('\p{Scx=	/a/Cher}');
Error('\P{Scx=	/a/Cher}');
Expect(1, 5108, '\p{Scx:cher}', "");
Expect(0, 5108, '\p{^Scx:cher}', "");
Expect(0, 5108, '\P{Scx:cher}', "");
Expect(1, 5108, '\P{^Scx:cher}', "");
Expect(0, 5109, '\p{Scx:cher}', "");
Expect(1, 5109, '\p{^Scx:cher}', "");
Expect(1, 5109, '\P{Scx:cher}', "");
Expect(0, 5109, '\P{^Scx:cher}', "");
Expect(1, 5108, '\p{Scx= -Cher}', "");
Expect(0, 5108, '\p{^Scx= -Cher}', "");
Expect(0, 5108, '\P{Scx= -Cher}', "");
Expect(1, 5108, '\P{^Scx= -Cher}', "");
Expect(0, 5109, '\p{Scx= -Cher}', "");
Expect(1, 5109, '\p{^Scx= -Cher}', "");
Expect(1, 5109, '\P{Scx= -Cher}', "");
Expect(0, 5109, '\P{^Scx= -Cher}', "");
Error('\p{Is_Script_Extensions:		_cherokee/a/}');
Error('\P{Is_Script_Extensions:		_cherokee/a/}');
Expect(1, 5108, '\p{Is_Script_Extensions=cherokee}', "");
Expect(0, 5108, '\p{^Is_Script_Extensions=cherokee}', "");
Expect(0, 5108, '\P{Is_Script_Extensions=cherokee}', "");
Expect(1, 5108, '\P{^Is_Script_Extensions=cherokee}', "");
Expect(0, 5109, '\p{Is_Script_Extensions=cherokee}', "");
Expect(1, 5109, '\p{^Is_Script_Extensions=cherokee}', "");
Expect(1, 5109, '\P{Is_Script_Extensions=cherokee}', "");
Expect(0, 5109, '\P{^Is_Script_Extensions=cherokee}', "");
Expect(1, 5108, '\p{Is_Script_Extensions:		CHEROKEE}', "");
Expect(0, 5108, '\p{^Is_Script_Extensions:		CHEROKEE}', "");
Expect(0, 5108, '\P{Is_Script_Extensions:		CHEROKEE}', "");
Expect(1, 5108, '\P{^Is_Script_Extensions:		CHEROKEE}', "");
Expect(0, 5109, '\p{Is_Script_Extensions:		CHEROKEE}', "");
Expect(1, 5109, '\p{^Is_Script_Extensions:		CHEROKEE}', "");
Expect(1, 5109, '\P{Is_Script_Extensions:		CHEROKEE}', "");
Expect(0, 5109, '\P{^Is_Script_Extensions:		CHEROKEE}', "");
Error('\p{Is_Scx=:=		Cher}');
Error('\P{Is_Scx=:=		Cher}');
Expect(1, 5108, '\p{Is_Scx=cher}', "");
Expect(0, 5108, '\p{^Is_Scx=cher}', "");
Expect(0, 5108, '\P{Is_Scx=cher}', "");
Expect(1, 5108, '\P{^Is_Scx=cher}', "");
Expect(0, 5109, '\p{Is_Scx=cher}', "");
Expect(1, 5109, '\p{^Is_Scx=cher}', "");
Expect(1, 5109, '\P{Is_Scx=cher}', "");
Expect(0, 5109, '\P{^Is_Scx=cher}', "");
Expect(1, 5108, '\p{Is_Scx=-CHER}', "");
Expect(0, 5108, '\p{^Is_Scx=-CHER}', "");
Expect(0, 5108, '\P{Is_Scx=-CHER}', "");
Expect(1, 5108, '\P{^Is_Scx=-CHER}', "");
Expect(0, 5109, '\p{Is_Scx=-CHER}', "");
Expect(1, 5109, '\p{^Is_Scx=-CHER}', "");
Expect(1, 5109, '\P{Is_Scx=-CHER}', "");
Expect(0, 5109, '\P{^Is_Scx=-CHER}', "");
Error('\p{Script_Extensions=-:=Runic}');
Error('\P{Script_Extensions=-:=Runic}');
Expect(1, 5872, '\p{Script_Extensions:runic}', "");
Expect(0, 5872, '\p{^Script_Extensions:runic}', "");
Expect(0, 5872, '\P{Script_Extensions:runic}', "");
Expect(1, 5872, '\P{^Script_Extensions:runic}', "");
Expect(0, 5873, '\p{Script_Extensions:runic}', "");
Expect(1, 5873, '\p{^Script_Extensions:runic}', "");
Expect(1, 5873, '\P{Script_Extensions:runic}', "");
Expect(0, 5873, '\P{^Script_Extensions:runic}', "");
Expect(1, 5872, '\p{Script_Extensions=	_RUNIC}', "");
Expect(0, 5872, '\p{^Script_Extensions=	_RUNIC}', "");
Expect(0, 5872, '\P{Script_Extensions=	_RUNIC}', "");
Expect(1, 5872, '\P{^Script_Extensions=	_RUNIC}', "");
Expect(0, 5873, '\p{Script_Extensions=	_RUNIC}', "");
Expect(1, 5873, '\p{^Script_Extensions=	_RUNIC}', "");
Expect(1, 5873, '\P{Script_Extensions=	_RUNIC}', "");
Expect(0, 5873, '\P{^Script_Extensions=	_RUNIC}', "");
Error('\p{Scx:_:=Runr}');
Error('\P{Scx:_:=Runr}');
Expect(1, 5872, '\p{Scx=runr}', "");
Expect(0, 5872, '\p{^Scx=runr}', "");
Expect(0, 5872, '\P{Scx=runr}', "");
Expect(1, 5872, '\P{^Scx=runr}', "");
Expect(0, 5873, '\p{Scx=runr}', "");
Expect(1, 5873, '\p{^Scx=runr}', "");
Expect(1, 5873, '\P{Scx=runr}', "");
Expect(0, 5873, '\P{^Scx=runr}', "");
Expect(1, 5872, '\p{Scx:    runr}', "");
Expect(0, 5872, '\p{^Scx:    runr}', "");
Expect(0, 5872, '\P{Scx:    runr}', "");
Expect(1, 5872, '\P{^Scx:    runr}', "");
Expect(0, 5873, '\p{Scx:    runr}', "");
Expect(1, 5873, '\p{^Scx:    runr}', "");
Expect(1, 5873, '\P{Scx:    runr}', "");
Expect(0, 5873, '\P{^Scx:    runr}', "");
Error('\p{Is_Script_Extensions=	_Runic/a/}');
Error('\P{Is_Script_Extensions=	_Runic/a/}');
Expect(1, 5872, '\p{Is_Script_Extensions=runic}', "");
Expect(0, 5872, '\p{^Is_Script_Extensions=runic}', "");
Expect(0, 5872, '\P{Is_Script_Extensions=runic}', "");
Expect(1, 5872, '\P{^Is_Script_Extensions=runic}', "");
Expect(0, 5873, '\p{Is_Script_Extensions=runic}', "");
Expect(1, 5873, '\p{^Is_Script_Extensions=runic}', "");
Expect(1, 5873, '\P{Is_Script_Extensions=runic}', "");
Expect(0, 5873, '\P{^Is_Script_Extensions=runic}', "");
Expect(1, 5872, '\p{Is_Script_Extensions=--RUNIC}', "");
Expect(0, 5872, '\p{^Is_Script_Extensions=--RUNIC}', "");
Expect(0, 5872, '\P{Is_Script_Extensions=--RUNIC}', "");
Expect(1, 5872, '\P{^Is_Script_Extensions=--RUNIC}', "");
Expect(0, 5873, '\p{Is_Script_Extensions=--RUNIC}', "");
Expect(1, 5873, '\p{^Is_Script_Extensions=--RUNIC}', "");
Expect(1, 5873, '\P{Is_Script_Extensions=--RUNIC}', "");
Expect(0, 5873, '\P{^Is_Script_Extensions=--RUNIC}', "");
Error('\p{Is_Scx= Runr:=}');
Error('\P{Is_Scx= Runr:=}');
Expect(1, 5872, '\p{Is_Scx=runr}', "");
Expect(0, 5872, '\p{^Is_Scx=runr}', "");
Expect(0, 5872, '\P{Is_Scx=runr}', "");
Expect(1, 5872, '\P{^Is_Scx=runr}', "");
Expect(0, 5873, '\p{Is_Scx=runr}', "");
Expect(1, 5873, '\p{^Is_Scx=runr}', "");
Expect(1, 5873, '\P{Is_Scx=runr}', "");
Expect(0, 5873, '\P{^Is_Scx=runr}', "");
Expect(1, 5872, '\p{Is_Scx:	-_runr}', "");
Expect(0, 5872, '\p{^Is_Scx:	-_runr}', "");
Expect(0, 5872, '\P{Is_Scx:	-_runr}', "");
Expect(1, 5872, '\P{^Is_Scx:	-_runr}', "");
Expect(0, 5873, '\p{Is_Scx:	-_runr}', "");
Expect(1, 5873, '\p{^Is_Scx:	-_runr}', "");
Expect(1, 5873, '\P{Is_Scx:	-_runr}', "");
Expect(0, 5873, '\P{^Is_Scx:	-_runr}', "");
Error('\p{Script_Extensions= 	MEROITIC_Hieroglyphs:=}');
Error('\P{Script_Extensions= 	MEROITIC_Hieroglyphs:=}');
Expect(1, 67999, '\p{Script_Extensions: meroitichieroglyphs}', "");
Expect(0, 67999, '\p{^Script_Extensions: meroitichieroglyphs}', "");
Expect(0, 67999, '\P{Script_Extensions: meroitichieroglyphs}', "");
Expect(1, 67999, '\P{^Script_Extensions: meroitichieroglyphs}', "");
Expect(0, 68000, '\p{Script_Extensions: meroitichieroglyphs}', "");
Expect(1, 68000, '\p{^Script_Extensions: meroitichieroglyphs}', "");
Expect(1, 68000, '\P{Script_Extensions: meroitichieroglyphs}', "");
Expect(0, 68000, '\P{^Script_Extensions: meroitichieroglyphs}', "");
Expect(1, 67999, '\p{Script_Extensions=	meroitic_HIEROGLYPHS}', "");
Expect(0, 67999, '\p{^Script_Extensions=	meroitic_HIEROGLYPHS}', "");
Expect(0, 67999, '\P{Script_Extensions=	meroitic_HIEROGLYPHS}', "");
Expect(1, 67999, '\P{^Script_Extensions=	meroitic_HIEROGLYPHS}', "");
Expect(0, 68000, '\p{Script_Extensions=	meroitic_HIEROGLYPHS}', "");
Expect(1, 68000, '\p{^Script_Extensions=	meroitic_HIEROGLYPHS}', "");
Expect(1, 68000, '\P{Script_Extensions=	meroitic_HIEROGLYPHS}', "");
Expect(0, 68000, '\P{^Script_Extensions=	meroitic_HIEROGLYPHS}', "");
Error('\p{Scx=:= -Mero}');
Error('\P{Scx=:= -Mero}');
Expect(1, 67999, '\p{Scx=mero}', "");
Expect(0, 67999, '\p{^Scx=mero}', "");
Expect(0, 67999, '\P{Scx=mero}', "");
Expect(1, 67999, '\P{^Scx=mero}', "");
Expect(0, 68000, '\p{Scx=mero}', "");
Expect(1, 68000, '\p{^Scx=mero}', "");
Expect(1, 68000, '\P{Scx=mero}', "");
Expect(0, 68000, '\P{^Scx=mero}', "");
Expect(1, 67999, '\p{Scx: _	Mero}', "");
Expect(0, 67999, '\p{^Scx: _	Mero}', "");
Expect(0, 67999, '\P{Scx: _	Mero}', "");
Expect(1, 67999, '\P{^Scx: _	Mero}', "");
Expect(0, 68000, '\p{Scx: _	Mero}', "");
Expect(1, 68000, '\p{^Scx: _	Mero}', "");
Expect(1, 68000, '\P{Scx: _	Mero}', "");
Expect(0, 68000, '\P{^Scx: _	Mero}', "");
Error('\p{Is_Script_Extensions=:=		Meroitic_hieroglyphs}');
Error('\P{Is_Script_Extensions=:=		Meroitic_hieroglyphs}');
Expect(1, 67999, '\p{Is_Script_Extensions=meroitichieroglyphs}', "");
Expect(0, 67999, '\p{^Is_Script_Extensions=meroitichieroglyphs}', "");
Expect(0, 67999, '\P{Is_Script_Extensions=meroitichieroglyphs}', "");
Expect(1, 67999, '\P{^Is_Script_Extensions=meroitichieroglyphs}', "");
Expect(0, 68000, '\p{Is_Script_Extensions=meroitichieroglyphs}', "");
Expect(1, 68000, '\p{^Is_Script_Extensions=meroitichieroglyphs}', "");
Expect(1, 68000, '\P{Is_Script_Extensions=meroitichieroglyphs}', "");
Expect(0, 68000, '\P{^Is_Script_Extensions=meroitichieroglyphs}', "");
Expect(1, 67999, '\p{Is_Script_Extensions=_	MEROITIC_Hieroglyphs}', "");
Expect(0, 67999, '\p{^Is_Script_Extensions=_	MEROITIC_Hieroglyphs}', "");
Expect(0, 67999, '\P{Is_Script_Extensions=_	MEROITIC_Hieroglyphs}', "");
Expect(1, 67999, '\P{^Is_Script_Extensions=_	MEROITIC_Hieroglyphs}', "");
Expect(0, 68000, '\p{Is_Script_Extensions=_	MEROITIC_Hieroglyphs}', "");
Expect(1, 68000, '\p{^Is_Script_Extensions=_	MEROITIC_Hieroglyphs}', "");
Expect(1, 68000, '\P{Is_Script_Extensions=_	MEROITIC_Hieroglyphs}', "");
Expect(0, 68000, '\P{^Is_Script_Extensions=_	MEROITIC_Hieroglyphs}', "");
Error('\p{Is_Scx=_ Mero:=}');
Error('\P{Is_Scx=_ Mero:=}');
Expect(1, 67999, '\p{Is_Scx=mero}', "");
Expect(0, 67999, '\p{^Is_Scx=mero}', "");
Expect(0, 67999, '\P{Is_Scx=mero}', "");
Expect(1, 67999, '\P{^Is_Scx=mero}', "");
Expect(0, 68000, '\p{Is_Scx=mero}', "");
Expect(1, 68000, '\p{^Is_Scx=mero}', "");
Expect(1, 68000, '\P{Is_Scx=mero}', "");
Expect(0, 68000, '\P{^Is_Scx=mero}', "");
Expect(1, 67999, '\p{Is_Scx= -mero}', "");
Expect(0, 67999, '\p{^Is_Scx= -mero}', "");
Expect(0, 67999, '\P{Is_Scx= -mero}', "");
Expect(1, 67999, '\P{^Is_Scx= -mero}', "");
Expect(0, 68000, '\p{Is_Scx= -mero}', "");
Expect(1, 68000, '\p{^Is_Scx= -mero}', "");
Expect(1, 68000, '\P{Is_Scx= -mero}', "");
Expect(0, 68000, '\P{^Is_Scx= -mero}', "");
Error('\p{Script_Extensions=	OL_Chiki:=}');
Error('\P{Script_Extensions=	OL_Chiki:=}');
Expect(1, 7295, '\p{Script_Extensions=olchiki}', "");
Expect(0, 7295, '\p{^Script_Extensions=olchiki}', "");
Expect(0, 7295, '\P{Script_Extensions=olchiki}', "");
Expect(1, 7295, '\P{^Script_Extensions=olchiki}', "");
Expect(0, 7296, '\p{Script_Extensions=olchiki}', "");
Expect(1, 7296, '\p{^Script_Extensions=olchiki}', "");
Expect(1, 7296, '\P{Script_Extensions=olchiki}', "");
Expect(0, 7296, '\P{^Script_Extensions=olchiki}', "");
Expect(1, 7295, '\p{Script_Extensions=	_Ol_CHIKI}', "");
Expect(0, 7295, '\p{^Script_Extensions=	_Ol_CHIKI}', "");
Expect(0, 7295, '\P{Script_Extensions=	_Ol_CHIKI}', "");
Expect(1, 7295, '\P{^Script_Extensions=	_Ol_CHIKI}', "");
Expect(0, 7296, '\p{Script_Extensions=	_Ol_CHIKI}', "");
Expect(1, 7296, '\p{^Script_Extensions=	_Ol_CHIKI}', "");
Expect(1, 7296, '\P{Script_Extensions=	_Ol_CHIKI}', "");
Expect(0, 7296, '\P{^Script_Extensions=	_Ol_CHIKI}', "");
Error('\p{Scx=:=Olck}');
Error('\P{Scx=:=Olck}');
Expect(1, 7295, '\p{Scx=olck}', "");
Expect(0, 7295, '\p{^Scx=olck}', "");
Expect(0, 7295, '\P{Scx=olck}', "");
Expect(1, 7295, '\P{^Scx=olck}', "");
Expect(0, 7296, '\p{Scx=olck}', "");
Expect(1, 7296, '\p{^Scx=olck}', "");
Expect(1, 7296, '\P{Scx=olck}', "");
Expect(0, 7296, '\P{^Scx=olck}', "");
Expect(1, 7295, '\p{Scx=_-Olck}', "");
Expect(0, 7295, '\p{^Scx=_-Olck}', "");
Expect(0, 7295, '\P{Scx=_-Olck}', "");
Expect(1, 7295, '\P{^Scx=_-Olck}', "");
Expect(0, 7296, '\p{Scx=_-Olck}', "");
Expect(1, 7296, '\p{^Scx=_-Olck}', "");
Expect(1, 7296, '\P{Scx=_-Olck}', "");
Expect(0, 7296, '\P{^Scx=_-Olck}', "");
Error('\p{Is_Script_Extensions=-/a/Ol_CHIKI}');
Error('\P{Is_Script_Extensions=-/a/Ol_CHIKI}');
Expect(1, 7295, '\p{Is_Script_Extensions=olchiki}', "");
Expect(0, 7295, '\p{^Is_Script_Extensions=olchiki}', "");
Expect(0, 7295, '\P{Is_Script_Extensions=olchiki}', "");
Expect(1, 7295, '\P{^Is_Script_Extensions=olchiki}', "");
Expect(0, 7296, '\p{Is_Script_Extensions=olchiki}', "");
Expect(1, 7296, '\p{^Is_Script_Extensions=olchiki}', "");
Expect(1, 7296, '\P{Is_Script_Extensions=olchiki}', "");
Expect(0, 7296, '\P{^Is_Script_Extensions=olchiki}', "");
Expect(1, 7295, '\p{Is_Script_Extensions=_-Ol_Chiki}', "");
Expect(0, 7295, '\p{^Is_Script_Extensions=_-Ol_Chiki}', "");
Expect(0, 7295, '\P{Is_Script_Extensions=_-Ol_Chiki}', "");
Expect(1, 7295, '\P{^Is_Script_Extensions=_-Ol_Chiki}', "");
Expect(0, 7296, '\p{Is_Script_Extensions=_-Ol_Chiki}', "");
Expect(1, 7296, '\p{^Is_Script_Extensions=_-Ol_Chiki}', "");
Expect(1, 7296, '\P{Is_Script_Extensions=_-Ol_Chiki}', "");
Expect(0, 7296, '\P{^Is_Script_Extensions=_-Ol_Chiki}', "");
Error('\p{Is_Scx=_:=Olck}');
Error('\P{Is_Scx=_:=Olck}');
Expect(1, 7295, '\p{Is_Scx=olck}', "");
Expect(0, 7295, '\p{^Is_Scx=olck}', "");
Expect(0, 7295, '\P{Is_Scx=olck}', "");
Expect(1, 7295, '\P{^Is_Scx=olck}', "");
Expect(0, 7296, '\p{Is_Scx=olck}', "");
Expect(1, 7296, '\p{^Is_Scx=olck}', "");
Expect(1, 7296, '\P{Is_Scx=olck}', "");
Expect(0, 7296, '\P{^Is_Scx=olck}', "");
Expect(1, 7295, '\p{Is_Scx=		Olck}', "");
Expect(0, 7295, '\p{^Is_Scx=		Olck}', "");
Expect(0, 7295, '\P{Is_Scx=		Olck}', "");
Expect(1, 7295, '\P{^Is_Scx=		Olck}', "");
Expect(0, 7296, '\p{Is_Scx=		Olck}', "");
Expect(1, 7296, '\p{^Is_Scx=		Olck}', "");
Expect(1, 7296, '\P{Is_Scx=		Olck}', "");
Expect(0, 7296, '\P{^Is_Scx=		Olck}', "");
Error('\p{Script_Extensions=_Phags_PA/a/}');
Error('\P{Script_Extensions=_Phags_PA/a/}');
Expect(1, 43127, '\p{Script_Extensions=phagspa}', "");
Expect(0, 43127, '\p{^Script_Extensions=phagspa}', "");
Expect(0, 43127, '\P{Script_Extensions=phagspa}', "");
Expect(1, 43127, '\P{^Script_Extensions=phagspa}', "");
Expect(0, 43128, '\p{Script_Extensions=phagspa}', "");
Expect(1, 43128, '\p{^Script_Extensions=phagspa}', "");
Expect(1, 43128, '\P{Script_Extensions=phagspa}', "");
Expect(0, 43128, '\P{^Script_Extensions=phagspa}', "");
Expect(1, 43127, '\p{Script_Extensions=-_PHAGS_Pa}', "");
Expect(0, 43127, '\p{^Script_Extensions=-_PHAGS_Pa}', "");
Expect(0, 43127, '\P{Script_Extensions=-_PHAGS_Pa}', "");
Expect(1, 43127, '\P{^Script_Extensions=-_PHAGS_Pa}', "");
Expect(0, 43128, '\p{Script_Extensions=-_PHAGS_Pa}', "");
Expect(1, 43128, '\p{^Script_Extensions=-_PHAGS_Pa}', "");
Expect(1, 43128, '\P{Script_Extensions=-_PHAGS_Pa}', "");
Expect(0, 43128, '\P{^Script_Extensions=-_PHAGS_Pa}', "");
Error('\p{Scx=/a/-_PHAG}');
Error('\P{Scx=/a/-_PHAG}');
Expect(1, 43127, '\p{Scx=phag}', "");
Expect(0, 43127, '\p{^Scx=phag}', "");
Expect(0, 43127, '\P{Scx=phag}', "");
Expect(1, 43127, '\P{^Scx=phag}', "");
Expect(0, 43128, '\p{Scx=phag}', "");
Expect(1, 43128, '\p{^Scx=phag}', "");
Expect(1, 43128, '\P{Scx=phag}', "");
Expect(0, 43128, '\P{^Scx=phag}', "");
Expect(1, 43127, '\p{Scx=_PHAG}', "");
Expect(0, 43127, '\p{^Scx=_PHAG}', "");
Expect(0, 43127, '\P{Scx=_PHAG}', "");
Expect(1, 43127, '\P{^Scx=_PHAG}', "");
Expect(0, 43128, '\p{Scx=_PHAG}', "");
Expect(1, 43128, '\p{^Scx=_PHAG}', "");
Expect(1, 43128, '\P{Scx=_PHAG}', "");
Expect(0, 43128, '\P{^Scx=_PHAG}', "");
Error('\p{Is_Script_Extensions=_ Phags_Pa/a/}');
Error('\P{Is_Script_Extensions=_ Phags_Pa/a/}');
Expect(1, 43127, '\p{Is_Script_Extensions:	phagspa}', "");
Expect(0, 43127, '\p{^Is_Script_Extensions:	phagspa}', "");
Expect(0, 43127, '\P{Is_Script_Extensions:	phagspa}', "");
Expect(1, 43127, '\P{^Is_Script_Extensions:	phagspa}', "");
Expect(0, 43128, '\p{Is_Script_Extensions:	phagspa}', "");
Expect(1, 43128, '\p{^Is_Script_Extensions:	phagspa}', "");
Expect(1, 43128, '\P{Is_Script_Extensions:	phagspa}', "");
Expect(0, 43128, '\P{^Is_Script_Extensions:	phagspa}', "");
Expect(1, 43127, '\p{Is_Script_Extensions= _phags_pa}', "");
Expect(0, 43127, '\p{^Is_Script_Extensions= _phags_pa}', "");
Expect(0, 43127, '\P{Is_Script_Extensions= _phags_pa}', "");
Expect(1, 43127, '\P{^Is_Script_Extensions= _phags_pa}', "");
Expect(0, 43128, '\p{Is_Script_Extensions= _phags_pa}', "");
Expect(1, 43128, '\p{^Is_Script_Extensions= _phags_pa}', "");
Expect(1, 43128, '\P{Is_Script_Extensions= _phags_pa}', "");
Expect(0, 43128, '\P{^Is_Script_Extensions= _phags_pa}', "");
Error('\p{Is_Scx=_/a/Phag}');
Error('\P{Is_Scx=_/a/Phag}');
Expect(1, 43127, '\p{Is_Scx: phag}', "");
Expect(0, 43127, '\p{^Is_Scx: phag}', "");
Expect(0, 43127, '\P{Is_Scx: phag}', "");
Expect(1, 43127, '\P{^Is_Scx: phag}', "");
Expect(0, 43128, '\p{Is_Scx: phag}', "");
Expect(1, 43128, '\p{^Is_Scx: phag}', "");
Expect(1, 43128, '\P{Is_Scx: phag}', "");
Expect(0, 43128, '\P{^Is_Scx: phag}', "");
Expect(1, 43127, '\p{Is_Scx: 	_Phag}', "");
Expect(0, 43127, '\p{^Is_Scx: 	_Phag}', "");
Expect(0, 43127, '\P{Is_Scx: 	_Phag}', "");
Expect(1, 43127, '\P{^Is_Scx: 	_Phag}', "");
Expect(0, 43128, '\p{Is_Scx: 	_Phag}', "");
Expect(1, 43128, '\p{^Is_Scx: 	_Phag}', "");
Expect(1, 43128, '\P{Is_Scx: 	_Phag}', "");
Expect(0, 43128, '\P{^Is_Scx: 	_Phag}', "");
Error('\p{Script_Extensions=_:=Ugaritic}');
Error('\P{Script_Extensions=_:=Ugaritic}');
Expect(1, 66463, '\p{Script_Extensions:   ugaritic}', "");
Expect(0, 66463, '\p{^Script_Extensions:   ugaritic}', "");
Expect(0, 66463, '\P{Script_Extensions:   ugaritic}', "");
Expect(1, 66463, '\P{^Script_Extensions:   ugaritic}', "");
Expect(0, 66464, '\p{Script_Extensions:   ugaritic}', "");
Expect(1, 66464, '\p{^Script_Extensions:   ugaritic}', "");
Expect(1, 66464, '\P{Script_Extensions:   ugaritic}', "");
Expect(0, 66464, '\P{^Script_Extensions:   ugaritic}', "");
Expect(1, 66463, '\p{Script_Extensions=-	UGARITIC}', "");
Expect(0, 66463, '\p{^Script_Extensions=-	UGARITIC}', "");
Expect(0, 66463, '\P{Script_Extensions=-	UGARITIC}', "");
Expect(1, 66463, '\P{^Script_Extensions=-	UGARITIC}', "");
Expect(0, 66464, '\p{Script_Extensions=-	UGARITIC}', "");
Expect(1, 66464, '\p{^Script_Extensions=-	UGARITIC}', "");
Expect(1, 66464, '\P{Script_Extensions=-	UGARITIC}', "");
Expect(0, 66464, '\P{^Script_Extensions=-	UGARITIC}', "");
Error('\p{Scx=_ Ugar/a/}');
Error('\P{Scx=_ Ugar/a/}');
Expect(1, 66463, '\p{Scx:   ugar}', "");
Expect(0, 66463, '\p{^Scx:   ugar}', "");
Expect(0, 66463, '\P{Scx:   ugar}', "");
Expect(1, 66463, '\P{^Scx:   ugar}', "");
Expect(0, 66464, '\p{Scx:   ugar}', "");
Expect(1, 66464, '\p{^Scx:   ugar}', "");
Expect(1, 66464, '\P{Scx:   ugar}', "");
Expect(0, 66464, '\P{^Scx:   ugar}', "");
Expect(1, 66463, '\p{Scx=	UGAR}', "");
Expect(0, 66463, '\p{^Scx=	UGAR}', "");
Expect(0, 66463, '\P{Scx=	UGAR}', "");
Expect(1, 66463, '\P{^Scx=	UGAR}', "");
Expect(0, 66464, '\p{Scx=	UGAR}', "");
Expect(1, 66464, '\p{^Scx=	UGAR}', "");
Expect(1, 66464, '\P{Scx=	UGAR}', "");
Expect(0, 66464, '\P{^Scx=	UGAR}', "");
Error('\p{Is_Script_Extensions: _Ugaritic/a/}');
Error('\P{Is_Script_Extensions: _Ugaritic/a/}');
Expect(1, 66463, '\p{Is_Script_Extensions=ugaritic}', "");
Expect(0, 66463, '\p{^Is_Script_Extensions=ugaritic}', "");
Expect(0, 66463, '\P{Is_Script_Extensions=ugaritic}', "");
Expect(1, 66463, '\P{^Is_Script_Extensions=ugaritic}', "");
Expect(0, 66464, '\p{Is_Script_Extensions=ugaritic}', "");
Expect(1, 66464, '\p{^Is_Script_Extensions=ugaritic}', "");
Expect(1, 66464, '\P{Is_Script_Extensions=ugaritic}', "");
Expect(0, 66464, '\P{^Is_Script_Extensions=ugaritic}', "");
Expect(1, 66463, '\p{Is_Script_Extensions=_ugaritic}', "");
Expect(0, 66463, '\p{^Is_Script_Extensions=_ugaritic}', "");
Expect(0, 66463, '\P{Is_Script_Extensions=_ugaritic}', "");
Expect(1, 66463, '\P{^Is_Script_Extensions=_ugaritic}', "");
Expect(0, 66464, '\p{Is_Script_Extensions=_ugaritic}', "");
Expect(1, 66464, '\p{^Is_Script_Extensions=_ugaritic}', "");
Expect(1, 66464, '\P{Is_Script_Extensions=_ugaritic}', "");
Expect(0, 66464, '\P{^Is_Script_Extensions=_ugaritic}', "");
Error('\p{Is_Scx=-	Ugar/a/}');
Error('\P{Is_Scx=-	Ugar/a/}');
Expect(1, 66463, '\p{Is_Scx: ugar}', "");
Expect(0, 66463, '\p{^Is_Scx: ugar}', "");
Expect(0, 66463, '\P{Is_Scx: ugar}', "");
Expect(1, 66463, '\P{^Is_Scx: ugar}', "");
Expect(0, 66464, '\p{Is_Scx: ugar}', "");
Expect(1, 66464, '\p{^Is_Scx: ugar}', "");
Expect(1, 66464, '\P{Is_Scx: ugar}', "");
Expect(0, 66464, '\P{^Is_Scx: ugar}', "");
Expect(1, 66463, '\p{Is_Scx=	_UGAR}', "");
Expect(0, 66463, '\p{^Is_Scx=	_UGAR}', "");
Expect(0, 66463, '\P{Is_Scx=	_UGAR}', "");
Expect(1, 66463, '\P{^Is_Scx=	_UGAR}', "");
Expect(0, 66464, '\p{Is_Scx=	_UGAR}', "");
Expect(1, 66464, '\p{^Is_Scx=	_UGAR}', "");
Expect(1, 66464, '\P{Is_Scx=	_UGAR}', "");
Expect(0, 66464, '\P{^Is_Scx=	_UGAR}', "");
Error('\p{Script_Extensions=-_Armenian:=}');
Error('\P{Script_Extensions=-_Armenian:=}');
Expect(1, 64279, '\p{Script_Extensions=armenian}', "");
Expect(0, 64279, '\p{^Script_Extensions=armenian}', "");
Expect(0, 64279, '\P{Script_Extensions=armenian}', "");
Expect(1, 64279, '\P{^Script_Extensions=armenian}', "");
Expect(0, 64280, '\p{Script_Extensions=armenian}', "");
Expect(1, 64280, '\p{^Script_Extensions=armenian}', "");
Expect(1, 64280, '\P{Script_Extensions=armenian}', "");
Expect(0, 64280, '\P{^Script_Extensions=armenian}', "");
Expect(1, 64279, '\p{Script_Extensions=-_ARMENIAN}', "");
Expect(0, 64279, '\p{^Script_Extensions=-_ARMENIAN}', "");
Expect(0, 64279, '\P{Script_Extensions=-_ARMENIAN}', "");
Expect(1, 64279, '\P{^Script_Extensions=-_ARMENIAN}', "");
Expect(0, 64280, '\p{Script_Extensions=-_ARMENIAN}', "");
Expect(1, 64280, '\p{^Script_Extensions=-_ARMENIAN}', "");
Expect(1, 64280, '\P{Script_Extensions=-_ARMENIAN}', "");
Expect(0, 64280, '\P{^Script_Extensions=-_ARMENIAN}', "");
Error('\p{Scx::=Armn}');
Error('\P{Scx::=Armn}');
Expect(1, 64279, '\p{Scx=armn}', "");
Expect(0, 64279, '\p{^Scx=armn}', "");
Expect(0, 64279, '\P{Scx=armn}', "");
Expect(1, 64279, '\P{^Scx=armn}', "");
Expect(0, 64280, '\p{Scx=armn}', "");
Expect(1, 64280, '\p{^Scx=armn}', "");
Expect(1, 64280, '\P{Scx=armn}', "");
Expect(0, 64280, '\P{^Scx=armn}', "");
Expect(1, 64279, '\p{Scx=-Armn}', "");
Expect(0, 64279, '\p{^Scx=-Armn}', "");
Expect(0, 64279, '\P{Scx=-Armn}', "");
Expect(1, 64279, '\P{^Scx=-Armn}', "");
Expect(0, 64280, '\p{Scx=-Armn}', "");
Expect(1, 64280, '\p{^Scx=-Armn}', "");
Expect(1, 64280, '\P{Scx=-Armn}', "");
Expect(0, 64280, '\P{^Scx=-Armn}', "");
Error('\p{Is_Script_Extensions=:=_ARMENIAN}');
Error('\P{Is_Script_Extensions=:=_ARMENIAN}');
Expect(1, 64279, '\p{Is_Script_Extensions=armenian}', "");
Expect(0, 64279, '\p{^Is_Script_Extensions=armenian}', "");
Expect(0, 64279, '\P{Is_Script_Extensions=armenian}', "");
Expect(1, 64279, '\P{^Is_Script_Extensions=armenian}', "");
Expect(0, 64280, '\p{Is_Script_Extensions=armenian}', "");
Expect(1, 64280, '\p{^Is_Script_Extensions=armenian}', "");
Expect(1, 64280, '\P{Is_Script_Extensions=armenian}', "");
Expect(0, 64280, '\P{^Is_Script_Extensions=armenian}', "");
Expect(1, 64279, '\p{Is_Script_Extensions=	_Armenian}', "");
Expect(0, 64279, '\p{^Is_Script_Extensions=	_Armenian}', "");
Expect(0, 64279, '\P{Is_Script_Extensions=	_Armenian}', "");
Expect(1, 64279, '\P{^Is_Script_Extensions=	_Armenian}', "");
Expect(0, 64280, '\p{Is_Script_Extensions=	_Armenian}', "");
Expect(1, 64280, '\p{^Is_Script_Extensions=	_Armenian}', "");
Expect(1, 64280, '\P{Is_Script_Extensions=	_Armenian}', "");
Expect(0, 64280, '\P{^Is_Script_Extensions=	_Armenian}', "");
Error('\p{Is_Scx=	 armn/a/}');
Error('\P{Is_Scx=	 armn/a/}');
Expect(1, 64279, '\p{Is_Scx=armn}', "");
Expect(0, 64279, '\p{^Is_Scx=armn}', "");
Expect(0, 64279, '\P{Is_Scx=armn}', "");
Expect(1, 64279, '\P{^Is_Scx=armn}', "");
Expect(0, 64280, '\p{Is_Scx=armn}', "");
Expect(1, 64280, '\p{^Is_Scx=armn}', "");
Expect(1, 64280, '\P{Is_Scx=armn}', "");
Expect(0, 64280, '\P{^Is_Scx=armn}', "");
Expect(1, 64279, '\p{Is_Scx=  ARMN}', "");
Expect(0, 64279, '\p{^Is_Scx=  ARMN}', "");
Expect(0, 64279, '\P{Is_Scx=  ARMN}', "");
Expect(1, 64279, '\P{^Is_Scx=  ARMN}', "");
Expect(0, 64280, '\p{Is_Scx=  ARMN}', "");
Expect(1, 64280, '\p{^Is_Scx=  ARMN}', "");
Expect(1, 64280, '\P{Is_Scx=  ARMN}', "");
Expect(0, 64280, '\P{^Is_Scx=  ARMN}', "");
Error('\p{Script_Extensions= -SYRIAC/a/}');
Error('\P{Script_Extensions= -SYRIAC/a/}');
Expect(1, 1871, '\p{Script_Extensions=syriac}', "");
Expect(0, 1871, '\p{^Script_Extensions=syriac}', "");
Expect(0, 1871, '\P{Script_Extensions=syriac}', "");
Expect(1, 1871, '\P{^Script_Extensions=syriac}', "");
Expect(0, 1872, '\p{Script_Extensions=syriac}', "");
Expect(1, 1872, '\p{^Script_Extensions=syriac}', "");
Expect(1, 1872, '\P{Script_Extensions=syriac}', "");
Expect(0, 1872, '\P{^Script_Extensions=syriac}', "");
Expect(1, 1871, '\p{Script_Extensions:     syriac}', "");
Expect(0, 1871, '\p{^Script_Extensions:     syriac}', "");
Expect(0, 1871, '\P{Script_Extensions:     syriac}', "");
Expect(1, 1871, '\P{^Script_Extensions:     syriac}', "");
Expect(0, 1872, '\p{Script_Extensions:     syriac}', "");
Expect(1, 1872, '\p{^Script_Extensions:     syriac}', "");
Expect(1, 1872, '\P{Script_Extensions:     syriac}', "");
Expect(0, 1872, '\P{^Script_Extensions:     syriac}', "");
Error('\p{Scx:_/a/SYRC}');
Error('\P{Scx:_/a/SYRC}');
Expect(1, 1871, '\p{Scx=syrc}', "");
Expect(0, 1871, '\p{^Scx=syrc}', "");
Expect(0, 1871, '\P{Scx=syrc}', "");
Expect(1, 1871, '\P{^Scx=syrc}', "");
Expect(0, 1872, '\p{Scx=syrc}', "");
Expect(1, 1872, '\p{^Scx=syrc}', "");
Expect(1, 1872, '\P{Scx=syrc}', "");
Expect(0, 1872, '\P{^Scx=syrc}', "");
Expect(1, 1871, '\p{Scx=		syrc}', "");
Expect(0, 1871, '\p{^Scx=		syrc}', "");
Expect(0, 1871, '\P{Scx=		syrc}', "");
Expect(1, 1871, '\P{^Scx=		syrc}', "");
Expect(0, 1872, '\p{Scx=		syrc}', "");
Expect(1, 1872, '\p{^Scx=		syrc}', "");
Expect(1, 1872, '\P{Scx=		syrc}', "");
Expect(0, 1872, '\P{^Scx=		syrc}', "");
Error('\p{Is_Script_Extensions:    -Syriac:=}');
Error('\P{Is_Script_Extensions:    -Syriac:=}');
Expect(1, 1871, '\p{Is_Script_Extensions:	syriac}', "");
Expect(0, 1871, '\p{^Is_Script_Extensions:	syriac}', "");
Expect(0, 1871, '\P{Is_Script_Extensions:	syriac}', "");
Expect(1, 1871, '\P{^Is_Script_Extensions:	syriac}', "");
Expect(0, 1872, '\p{Is_Script_Extensions:	syriac}', "");
Expect(1, 1872, '\p{^Is_Script_Extensions:	syriac}', "");
Expect(1, 1872, '\P{Is_Script_Extensions:	syriac}', "");
Expect(0, 1872, '\P{^Is_Script_Extensions:	syriac}', "");
Expect(1, 1871, '\p{Is_Script_Extensions=- Syriac}', "");
Expect(0, 1871, '\p{^Is_Script_Extensions=- Syriac}', "");
Expect(0, 1871, '\P{Is_Script_Extensions=- Syriac}', "");
Expect(1, 1871, '\P{^Is_Script_Extensions=- Syriac}', "");
Expect(0, 1872, '\p{Is_Script_Extensions=- Syriac}', "");
Expect(1, 1872, '\p{^Is_Script_Extensions=- Syriac}', "");
Expect(1, 1872, '\P{Is_Script_Extensions=- Syriac}', "");
Expect(0, 1872, '\P{^Is_Script_Extensions=- Syriac}', "");
Error('\p{Is_Scx=	/a/Syrc}');
Error('\P{Is_Scx=	/a/Syrc}');
Expect(1, 1871, '\p{Is_Scx=syrc}', "");
Expect(0, 1871, '\p{^Is_Scx=syrc}', "");
Expect(0, 1871, '\P{Is_Scx=syrc}', "");
Expect(1, 1871, '\P{^Is_Scx=syrc}', "");
Expect(0, 1872, '\p{Is_Scx=syrc}', "");
Expect(1, 1872, '\p{^Is_Scx=syrc}', "");
Expect(1, 1872, '\P{Is_Scx=syrc}', "");
Expect(0, 1872, '\P{^Is_Scx=syrc}', "");
Expect(1, 1871, '\p{Is_Scx=_-Syrc}', "");
Expect(0, 1871, '\p{^Is_Scx=_-Syrc}', "");
Expect(0, 1871, '\P{Is_Scx=_-Syrc}', "");
Expect(1, 1871, '\P{^Is_Scx=_-Syrc}', "");
Expect(0, 1872, '\p{Is_Scx=_-Syrc}', "");
Expect(1, 1872, '\p{^Is_Scx=_-Syrc}', "");
Expect(1, 1872, '\P{Is_Scx=_-Syrc}', "");
Expect(0, 1872, '\P{^Is_Scx=_-Syrc}', "");
Error('\p{Script_Extensions=:= -katakana}');
Error('\P{Script_Extensions=:= -katakana}');
Expect(1, 110592, '\p{Script_Extensions=katakana}', "");
Expect(0, 110592, '\p{^Script_Extensions=katakana}', "");
Expect(0, 110592, '\P{Script_Extensions=katakana}', "");
Expect(1, 110592, '\P{^Script_Extensions=katakana}', "");
Expect(0, 110593, '\p{Script_Extensions=katakana}', "");
Expect(1, 110593, '\p{^Script_Extensions=katakana}', "");
Expect(1, 110593, '\P{Script_Extensions=katakana}', "");
Expect(0, 110593, '\P{^Script_Extensions=katakana}', "");
Expect(1, 110592, '\p{Script_Extensions=-_KATAKANA}', "");
Expect(0, 110592, '\p{^Script_Extensions=-_KATAKANA}', "");
Expect(0, 110592, '\P{Script_Extensions=-_KATAKANA}', "");
Expect(1, 110592, '\P{^Script_Extensions=-_KATAKANA}', "");
Expect(0, 110593, '\p{Script_Extensions=-_KATAKANA}', "");
Expect(1, 110593, '\p{^Script_Extensions=-_KATAKANA}', "");
Expect(1, 110593, '\P{Script_Extensions=-_KATAKANA}', "");
Expect(0, 110593, '\P{^Script_Extensions=-_KATAKANA}', "");
Error('\p{Scx=KANA/a/}');
Error('\P{Scx=KANA/a/}');
Expect(1, 110592, '\p{Scx=kana}', "");
Expect(0, 110592, '\p{^Scx=kana}', "");
Expect(0, 110592, '\P{Scx=kana}', "");
Expect(1, 110592, '\P{^Scx=kana}', "");
Expect(0, 110593, '\p{Scx=kana}', "");
Expect(1, 110593, '\p{^Scx=kana}', "");
Expect(1, 110593, '\P{Scx=kana}', "");
Expect(0, 110593, '\P{^Scx=kana}', "");
Expect(1, 110592, '\p{Scx= -Kana}', "");
Expect(0, 110592, '\p{^Scx= -Kana}', "");
Expect(0, 110592, '\P{Scx= -Kana}', "");
Expect(1, 110592, '\P{^Scx= -Kana}', "");
Expect(0, 110593, '\p{Scx= -Kana}', "");
Expect(1, 110593, '\p{^Scx= -Kana}', "");
Expect(1, 110593, '\P{Scx= -Kana}', "");
Expect(0, 110593, '\P{^Scx= -Kana}', "");
Error('\p{Is_Script_Extensions= 	Katakana/a/}');
Error('\P{Is_Script_Extensions= 	Katakana/a/}');
Expect(1, 110592, '\p{Is_Script_Extensions=katakana}', "");
Expect(0, 110592, '\p{^Is_Script_Extensions=katakana}', "");
Expect(0, 110592, '\P{Is_Script_Extensions=katakana}', "");
Expect(1, 110592, '\P{^Is_Script_Extensions=katakana}', "");
Expect(0, 110593, '\p{Is_Script_Extensions=katakana}', "");
Expect(1, 110593, '\p{^Is_Script_Extensions=katakana}', "");
Expect(1, 110593, '\P{Is_Script_Extensions=katakana}', "");
Expect(0, 110593, '\P{^Is_Script_Extensions=katakana}', "");
Expect(1, 110592, '\p{Is_Script_Extensions=	Katakana}', "");
Expect(0, 110592, '\p{^Is_Script_Extensions=	Katakana}', "");
Expect(0, 110592, '\P{Is_Script_Extensions=	Katakana}', "");
Expect(1, 110592, '\P{^Is_Script_Extensions=	Katakana}', "");
Expect(0, 110593, '\p{Is_Script_Extensions=	Katakana}', "");
Expect(1, 110593, '\p{^Is_Script_Extensions=	Katakana}', "");
Expect(1, 110593, '\P{Is_Script_Extensions=	Katakana}', "");
Expect(0, 110593, '\P{^Is_Script_Extensions=	Katakana}', "");
Error('\p{Is_Scx=:=	-kana}');
Error('\P{Is_Scx=:=	-kana}');
Expect(1, 110592, '\p{Is_Scx=kana}', "");
Expect(0, 110592, '\p{^Is_Scx=kana}', "");
Expect(0, 110592, '\P{Is_Scx=kana}', "");
Expect(1, 110592, '\P{^Is_Scx=kana}', "");
Expect(0, 110593, '\p{Is_Scx=kana}', "");
Expect(1, 110593, '\p{^Is_Scx=kana}', "");
Expect(1, 110593, '\P{Is_Scx=kana}', "");
Expect(0, 110593, '\P{^Is_Scx=kana}', "");
Expect(1, 110592, '\p{Is_Scx=	-Kana}', "");
Expect(0, 110592, '\p{^Is_Scx=	-Kana}', "");
Expect(0, 110592, '\P{Is_Scx=	-Kana}', "");
Expect(1, 110592, '\P{^Is_Scx=	-Kana}', "");
Expect(0, 110593, '\p{Is_Scx=	-Kana}', "");
Expect(1, 110593, '\p{^Is_Scx=	-Kana}', "");
Expect(1, 110593, '\P{Is_Scx=	-Kana}', "");
Expect(0, 110593, '\P{^Is_Scx=	-Kana}', "");
Error('\p{Script_Extensions=	/a/Yi}');
Error('\P{Script_Extensions=	/a/Yi}');
Expect(1, 65381, '\p{Script_Extensions=yi}', "");
Expect(0, 65381, '\p{^Script_Extensions=yi}', "");
Expect(0, 65381, '\P{Script_Extensions=yi}', "");
Expect(1, 65381, '\P{^Script_Extensions=yi}', "");
Expect(0, 65382, '\p{Script_Extensions=yi}', "");
Expect(1, 65382, '\p{^Script_Extensions=yi}', "");
Expect(1, 65382, '\P{Script_Extensions=yi}', "");
Expect(0, 65382, '\P{^Script_Extensions=yi}', "");
Expect(1, 65381, '\p{Script_Extensions=	 Yi}', "");
Expect(0, 65381, '\p{^Script_Extensions=	 Yi}', "");
Expect(0, 65381, '\P{Script_Extensions=	 Yi}', "");
Expect(1, 65381, '\P{^Script_Extensions=	 Yi}', "");
Expect(0, 65382, '\p{Script_Extensions=	 Yi}', "");
Expect(1, 65382, '\p{^Script_Extensions=	 Yi}', "");
Expect(1, 65382, '\P{Script_Extensions=	 Yi}', "");
Expect(0, 65382, '\P{^Script_Extensions=	 Yi}', "");
Error('\p{Scx=:= YIII}');
Error('\P{Scx=:= YIII}');
Expect(1, 65381, '\p{Scx:   yiii}', "");
Expect(0, 65381, '\p{^Scx:   yiii}', "");
Expect(0, 65381, '\P{Scx:   yiii}', "");
Expect(1, 65381, '\P{^Scx:   yiii}', "");
Expect(0, 65382, '\p{Scx:   yiii}', "");
Expect(1, 65382, '\p{^Scx:   yiii}', "");
Expect(1, 65382, '\P{Scx:   yiii}', "");
Expect(0, 65382, '\P{^Scx:   yiii}', "");
Expect(1, 65381, '\p{Scx=-yiii}', "");
Expect(0, 65381, '\p{^Scx=-yiii}', "");
Expect(0, 65381, '\P{Scx=-yiii}', "");
Expect(1, 65381, '\P{^Scx=-yiii}', "");
Expect(0, 65382, '\p{Scx=-yiii}', "");
Expect(1, 65382, '\p{^Scx=-yiii}', "");
Expect(1, 65382, '\P{Scx=-yiii}', "");
Expect(0, 65382, '\P{^Scx=-yiii}', "");
Error('\p{Is_Script_Extensions=:=Yi}');
Error('\P{Is_Script_Extensions=:=Yi}');
Expect(1, 65381, '\p{Is_Script_Extensions=yi}', "");
Expect(0, 65381, '\p{^Is_Script_Extensions=yi}', "");
Expect(0, 65381, '\P{Is_Script_Extensions=yi}', "");
Expect(1, 65381, '\P{^Is_Script_Extensions=yi}', "");
Expect(0, 65382, '\p{Is_Script_Extensions=yi}', "");
Expect(1, 65382, '\p{^Is_Script_Extensions=yi}', "");
Expect(1, 65382, '\P{Is_Script_Extensions=yi}', "");
Expect(0, 65382, '\P{^Is_Script_Extensions=yi}', "");
Expect(1, 65381, '\p{Is_Script_Extensions= 	YI}', "");
Expect(0, 65381, '\p{^Is_Script_Extensions= 	YI}', "");
Expect(0, 65381, '\P{Is_Script_Extensions= 	YI}', "");
Expect(1, 65381, '\P{^Is_Script_Extensions= 	YI}', "");
Expect(0, 65382, '\p{Is_Script_Extensions= 	YI}', "");
Expect(1, 65382, '\p{^Is_Script_Extensions= 	YI}', "");
Expect(1, 65382, '\P{Is_Script_Extensions= 	YI}', "");
Expect(0, 65382, '\P{^Is_Script_Extensions= 	YI}', "");
Error('\p{Is_Scx=	:=yiii}');
Error('\P{Is_Scx=	:=yiii}');
Expect(1, 65381, '\p{Is_Scx=yiii}', "");
Expect(0, 65381, '\p{^Is_Scx=yiii}', "");
Expect(0, 65381, '\P{Is_Scx=yiii}', "");
Expect(1, 65381, '\P{^Is_Scx=yiii}', "");
Expect(0, 65382, '\p{Is_Scx=yiii}', "");
Expect(1, 65382, '\p{^Is_Scx=yiii}', "");
Expect(1, 65382, '\P{Is_Scx=yiii}', "");
Expect(0, 65382, '\P{^Is_Scx=yiii}', "");
Expect(1, 65381, '\p{Is_Scx=-yiii}', "");
Expect(0, 65381, '\p{^Is_Scx=-yiii}', "");
Expect(0, 65381, '\P{Is_Scx=-yiii}', "");
Expect(1, 65381, '\P{^Is_Scx=-yiii}', "");
Expect(0, 65382, '\p{Is_Scx=-yiii}', "");
Expect(1, 65382, '\p{^Is_Scx=-yiii}', "");
Expect(1, 65382, '\P{Is_Scx=-yiii}', "");
Expect(0, 65382, '\P{^Is_Scx=-yiii}', "");
Error('\p{Script_Extensions=/a/_GLAGOLITIC}');
Error('\P{Script_Extensions=/a/_GLAGOLITIC}');
Expect(1, 11358, '\p{Script_Extensions=glagolitic}', "");
Expect(0, 11358, '\p{^Script_Extensions=glagolitic}', "");
Expect(0, 11358, '\P{Script_Extensions=glagolitic}', "");
Expect(1, 11358, '\P{^Script_Extensions=glagolitic}', "");
Expect(0, 11359, '\p{Script_Extensions=glagolitic}', "");
Expect(1, 11359, '\p{^Script_Extensions=glagolitic}', "");
Expect(1, 11359, '\P{Script_Extensions=glagolitic}', "");
Expect(0, 11359, '\P{^Script_Extensions=glagolitic}', "");
Expect(1, 11358, '\p{Script_Extensions=- GLAGOLITIC}', "");
Expect(0, 11358, '\p{^Script_Extensions=- GLAGOLITIC}', "");
Expect(0, 11358, '\P{Script_Extensions=- GLAGOLITIC}', "");
Expect(1, 11358, '\P{^Script_Extensions=- GLAGOLITIC}', "");
Expect(0, 11359, '\p{Script_Extensions=- GLAGOLITIC}', "");
Expect(1, 11359, '\p{^Script_Extensions=- GLAGOLITIC}', "");
Expect(1, 11359, '\P{Script_Extensions=- GLAGOLITIC}', "");
Expect(0, 11359, '\P{^Script_Extensions=- GLAGOLITIC}', "");
Error('\p{Scx=  Glag/a/}');
Error('\P{Scx=  Glag/a/}');
Expect(1, 11358, '\p{Scx=glag}', "");
Expect(0, 11358, '\p{^Scx=glag}', "");
Expect(0, 11358, '\P{Scx=glag}', "");
Expect(1, 11358, '\P{^Scx=glag}', "");
Expect(0, 11359, '\p{Scx=glag}', "");
Expect(1, 11359, '\p{^Scx=glag}', "");
Expect(1, 11359, '\P{Scx=glag}', "");
Expect(0, 11359, '\P{^Scx=glag}', "");
Expect(1, 11358, '\p{Scx=-	Glag}', "");
Expect(0, 11358, '\p{^Scx=-	Glag}', "");
Expect(0, 11358, '\P{Scx=-	Glag}', "");
Expect(1, 11358, '\P{^Scx=-	Glag}', "");
Expect(0, 11359, '\p{Scx=-	Glag}', "");
Expect(1, 11359, '\p{^Scx=-	Glag}', "");
Expect(1, 11359, '\P{Scx=-	Glag}', "");
Expect(0, 11359, '\P{^Scx=-	Glag}', "");
Error('\p{Is_Script_Extensions= _GLAGOLITIC:=}');
Error('\P{Is_Script_Extensions= _GLAGOLITIC:=}');
Expect(1, 11358, '\p{Is_Script_Extensions=glagolitic}', "");
Expect(0, 11358, '\p{^Is_Script_Extensions=glagolitic}', "");
Expect(0, 11358, '\P{Is_Script_Extensions=glagolitic}', "");
Expect(1, 11358, '\P{^Is_Script_Extensions=glagolitic}', "");
Expect(0, 11359, '\p{Is_Script_Extensions=glagolitic}', "");
Expect(1, 11359, '\p{^Is_Script_Extensions=glagolitic}', "");
Expect(1, 11359, '\P{Is_Script_Extensions=glagolitic}', "");
Expect(0, 11359, '\P{^Is_Script_Extensions=glagolitic}', "");
Expect(1, 11358, '\p{Is_Script_Extensions=-	Glagolitic}', "");
Expect(0, 11358, '\p{^Is_Script_Extensions=-	Glagolitic}', "");
Expect(0, 11358, '\P{Is_Script_Extensions=-	Glagolitic}', "");
Expect(1, 11358, '\P{^Is_Script_Extensions=-	Glagolitic}', "");
Expect(0, 11359, '\p{Is_Script_Extensions=-	Glagolitic}', "");
Expect(1, 11359, '\p{^Is_Script_Extensions=-	Glagolitic}', "");
Expect(1, 11359, '\P{Is_Script_Extensions=-	Glagolitic}', "");
Expect(0, 11359, '\P{^Is_Script_Extensions=-	Glagolitic}', "");
Error('\p{Is_Scx= 	GLAG/a/}');
Error('\P{Is_Scx= 	GLAG/a/}');
Expect(1, 11358, '\p{Is_Scx=glag}', "");
Expect(0, 11358, '\p{^Is_Scx=glag}', "");
Expect(0, 11358, '\P{Is_Scx=glag}', "");
Expect(1, 11358, '\P{^Is_Scx=glag}', "");
Expect(0, 11359, '\p{Is_Scx=glag}', "");
Expect(1, 11359, '\p{^Is_Scx=glag}', "");
Expect(1, 11359, '\P{Is_Scx=glag}', "");
Expect(0, 11359, '\P{^Is_Scx=glag}', "");
Expect(1, 11358, '\p{Is_Scx=	 Glag}', "");
Expect(0, 11358, '\p{^Is_Scx=	 Glag}', "");
Expect(0, 11358, '\P{Is_Scx=	 Glag}', "");
Expect(1, 11358, '\P{^Is_Scx=	 Glag}', "");
Expect(0, 11359, '\p{Is_Scx=	 Glag}', "");
Expect(1, 11359, '\p{^Is_Scx=	 Glag}', "");
Expect(1, 11359, '\P{Is_Scx=	 Glag}', "");
Expect(0, 11359, '\P{^Is_Scx=	 Glag}', "");
Error('\p{Script_Extensions=	-Lepcha/a/}');
Error('\P{Script_Extensions=	-Lepcha/a/}');
Expect(1, 7247, '\p{Script_Extensions:	lepcha}', "");
Expect(0, 7247, '\p{^Script_Extensions:	lepcha}', "");
Expect(0, 7247, '\P{Script_Extensions:	lepcha}', "");
Expect(1, 7247, '\P{^Script_Extensions:	lepcha}', "");
Expect(0, 7248, '\p{Script_Extensions:	lepcha}', "");
Expect(1, 7248, '\p{^Script_Extensions:	lepcha}', "");
Expect(1, 7248, '\P{Script_Extensions:	lepcha}', "");
Expect(0, 7248, '\P{^Script_Extensions:	lepcha}', "");
Expect(1, 7247, '\p{Script_Extensions=-lepcha}', "");
Expect(0, 7247, '\p{^Script_Extensions=-lepcha}', "");
Expect(0, 7247, '\P{Script_Extensions=-lepcha}', "");
Expect(1, 7247, '\P{^Script_Extensions=-lepcha}', "");
Expect(0, 7248, '\p{Script_Extensions=-lepcha}', "");
Expect(1, 7248, '\p{^Script_Extensions=-lepcha}', "");
Expect(1, 7248, '\P{Script_Extensions=-lepcha}', "");
Expect(0, 7248, '\P{^Script_Extensions=-lepcha}', "");
Error('\p{Scx=/a/-LEPC}');
Error('\P{Scx=/a/-LEPC}');
Expect(1, 7247, '\p{Scx=lepc}', "");
Expect(0, 7247, '\p{^Scx=lepc}', "");
Expect(0, 7247, '\P{Scx=lepc}', "");
Expect(1, 7247, '\P{^Scx=lepc}', "");
Expect(0, 7248, '\p{Scx=lepc}', "");
Expect(1, 7248, '\p{^Scx=lepc}', "");
Expect(1, 7248, '\P{Scx=lepc}', "");
Expect(0, 7248, '\P{^Scx=lepc}', "");
Expect(1, 7247, '\p{Scx=_lepc}', "");
Expect(0, 7247, '\p{^Scx=_lepc}', "");
Expect(0, 7247, '\P{Scx=_lepc}', "");
Expect(1, 7247, '\P{^Scx=_lepc}', "");
Expect(0, 7248, '\p{Scx=_lepc}', "");
Expect(1, 7248, '\p{^Scx=_lepc}', "");
Expect(1, 7248, '\P{Scx=_lepc}', "");
Expect(0, 7248, '\P{^Scx=_lepc}', "");
Error('\p{Is_Script_Extensions=-:=Lepcha}');
Error('\P{Is_Script_Extensions=-:=Lepcha}');
Expect(1, 7247, '\p{Is_Script_Extensions=lepcha}', "");
Expect(0, 7247, '\p{^Is_Script_Extensions=lepcha}', "");
Expect(0, 7247, '\P{Is_Script_Extensions=lepcha}', "");
Expect(1, 7247, '\P{^Is_Script_Extensions=lepcha}', "");
Expect(0, 7248, '\p{Is_Script_Extensions=lepcha}', "");
Expect(1, 7248, '\p{^Is_Script_Extensions=lepcha}', "");
Expect(1, 7248, '\P{Is_Script_Extensions=lepcha}', "");
Expect(0, 7248, '\P{^Is_Script_Extensions=lepcha}', "");
Expect(1, 7247, '\p{Is_Script_Extensions=		LEPCHA}', "");
Expect(0, 7247, '\p{^Is_Script_Extensions=		LEPCHA}', "");
Expect(0, 7247, '\P{Is_Script_Extensions=		LEPCHA}', "");
Expect(1, 7247, '\P{^Is_Script_Extensions=		LEPCHA}', "");
Expect(0, 7248, '\p{Is_Script_Extensions=		LEPCHA}', "");
Expect(1, 7248, '\p{^Is_Script_Extensions=		LEPCHA}', "");
Expect(1, 7248, '\P{Is_Script_Extensions=		LEPCHA}', "");
Expect(0, 7248, '\P{^Is_Script_Extensions=		LEPCHA}', "");
Error('\p{Is_Scx: :=  LEPC}');
Error('\P{Is_Scx: :=  LEPC}');
Expect(1, 7247, '\p{Is_Scx=lepc}', "");
Expect(0, 7247, '\p{^Is_Scx=lepc}', "");
Expect(0, 7247, '\P{Is_Scx=lepc}', "");
Expect(1, 7247, '\P{^Is_Scx=lepc}', "");
Expect(0, 7248, '\p{Is_Scx=lepc}', "");
Expect(1, 7248, '\p{^Is_Scx=lepc}', "");
Expect(1, 7248, '\P{Is_Scx=lepc}', "");
Expect(0, 7248, '\P{^Is_Scx=lepc}', "");
Expect(1, 7247, '\p{Is_Scx= _LEPC}', "");
Expect(0, 7247, '\p{^Is_Scx= _LEPC}', "");
Expect(0, 7247, '\P{Is_Scx= _LEPC}', "");
Expect(1, 7247, '\P{^Is_Scx= _LEPC}', "");
Expect(0, 7248, '\p{Is_Scx= _LEPC}', "");
Expect(1, 7248, '\p{^Is_Scx= _LEPC}', "");
Expect(1, 7248, '\P{Is_Scx= _LEPC}', "");
Expect(0, 7248, '\P{^Is_Scx= _LEPC}', "");
Error('\p{Script_Extensions=_thaana:=}');
Error('\P{Script_Extensions=_thaana:=}');
Expect(1, 65021, '\p{Script_Extensions=thaana}', "");
Expect(0, 65021, '\p{^Script_Extensions=thaana}', "");
Expect(0, 65021, '\P{Script_Extensions=thaana}', "");
Expect(1, 65021, '\P{^Script_Extensions=thaana}', "");
Expect(0, 65022, '\p{Script_Extensions=thaana}', "");
Expect(1, 65022, '\p{^Script_Extensions=thaana}', "");
Expect(1, 65022, '\P{Script_Extensions=thaana}', "");
Expect(0, 65022, '\P{^Script_Extensions=thaana}', "");
Expect(1, 65021, '\p{Script_Extensions=-THAANA}', "");
Expect(0, 65021, '\p{^Script_Extensions=-THAANA}', "");
Expect(0, 65021, '\P{Script_Extensions=-THAANA}', "");
Expect(1, 65021, '\P{^Script_Extensions=-THAANA}', "");
Expect(0, 65022, '\p{Script_Extensions=-THAANA}', "");
Expect(1, 65022, '\p{^Script_Extensions=-THAANA}', "");
Expect(1, 65022, '\P{Script_Extensions=-THAANA}', "");
Expect(0, 65022, '\P{^Script_Extensions=-THAANA}', "");
Error('\p{Scx=--Thaa:=}');
Error('\P{Scx=--Thaa:=}');
Expect(1, 65021, '\p{Scx=thaa}', "");
Expect(0, 65021, '\p{^Scx=thaa}', "");
Expect(0, 65021, '\P{Scx=thaa}', "");
Expect(1, 65021, '\P{^Scx=thaa}', "");
Expect(0, 65022, '\p{Scx=thaa}', "");
Expect(1, 65022, '\p{^Scx=thaa}', "");
Expect(1, 65022, '\P{Scx=thaa}', "");
Expect(0, 65022, '\P{^Scx=thaa}', "");
Expect(1, 65021, '\p{Scx=	 Thaa}', "");
Expect(0, 65021, '\p{^Scx=	 Thaa}', "");
Expect(0, 65021, '\P{Scx=	 Thaa}', "");
Expect(1, 65021, '\P{^Scx=	 Thaa}', "");
Expect(0, 65022, '\p{Scx=	 Thaa}', "");
Expect(1, 65022, '\p{^Scx=	 Thaa}', "");
Expect(1, 65022, '\P{Scx=	 Thaa}', "");
Expect(0, 65022, '\P{^Scx=	 Thaa}', "");
Error('\p{Is_Script_Extensions=:= 	Thaana}');
Error('\P{Is_Script_Extensions=:= 	Thaana}');
Expect(1, 65021, '\p{Is_Script_Extensions=thaana}', "");
Expect(0, 65021, '\p{^Is_Script_Extensions=thaana}', "");
Expect(0, 65021, '\P{Is_Script_Extensions=thaana}', "");
Expect(1, 65021, '\P{^Is_Script_Extensions=thaana}', "");
Expect(0, 65022, '\p{Is_Script_Extensions=thaana}', "");
Expect(1, 65022, '\p{^Is_Script_Extensions=thaana}', "");
Expect(1, 65022, '\P{Is_Script_Extensions=thaana}', "");
Expect(0, 65022, '\P{^Is_Script_Extensions=thaana}', "");
Expect(1, 65021, '\p{Is_Script_Extensions=-thaana}', "");
Expect(0, 65021, '\p{^Is_Script_Extensions=-thaana}', "");
Expect(0, 65021, '\P{Is_Script_Extensions=-thaana}', "");
Expect(1, 65021, '\P{^Is_Script_Extensions=-thaana}', "");
Expect(0, 65022, '\p{Is_Script_Extensions=-thaana}', "");
Expect(1, 65022, '\p{^Is_Script_Extensions=-thaana}', "");
Expect(1, 65022, '\P{Is_Script_Extensions=-thaana}', "");
Expect(0, 65022, '\P{^Is_Script_Extensions=-thaana}', "");
Error('\p{Is_Scx=/a/--thaa}');
Error('\P{Is_Scx=/a/--thaa}');
Expect(1, 65021, '\p{Is_Scx=thaa}', "");
Expect(0, 65021, '\p{^Is_Scx=thaa}', "");
Expect(0, 65021, '\P{Is_Scx=thaa}', "");
Expect(1, 65021, '\P{^Is_Scx=thaa}', "");
Expect(0, 65022, '\p{Is_Scx=thaa}', "");
Expect(1, 65022, '\p{^Is_Scx=thaa}', "");
Expect(1, 65022, '\P{Is_Scx=thaa}', "");
Expect(0, 65022, '\P{^Is_Scx=thaa}', "");
Expect(1, 65021, '\p{Is_Scx=THAA}', "");
Expect(0, 65021, '\p{^Is_Scx=THAA}', "");
Expect(0, 65021, '\P{Is_Scx=THAA}', "");
Expect(1, 65021, '\P{^Is_Scx=THAA}', "");
Expect(0, 65022, '\p{Is_Scx=THAA}', "");
Expect(1, 65022, '\p{^Is_Scx=THAA}', "");
Expect(1, 65022, '\P{Is_Scx=THAA}', "");
Expect(0, 65022, '\P{^Is_Scx=THAA}', "");
Error('\p{Script_Extensions=__Coptic:=}');
Error('\P{Script_Extensions=__Coptic:=}');
Expect(1, 11519, '\p{Script_Extensions=coptic}', "");
Expect(0, 11519, '\p{^Script_Extensions=coptic}', "");
Expect(0, 11519, '\P{Script_Extensions=coptic}', "");
Expect(1, 11519, '\P{^Script_Extensions=coptic}', "");
Expect(0, 11520, '\p{Script_Extensions=coptic}', "");
Expect(1, 11520, '\p{^Script_Extensions=coptic}', "");
Expect(1, 11520, '\P{Script_Extensions=coptic}', "");
Expect(0, 11520, '\P{^Script_Extensions=coptic}', "");
Expect(1, 11519, '\p{Script_Extensions=	-Coptic}', "");
Expect(0, 11519, '\p{^Script_Extensions=	-Coptic}', "");
Expect(0, 11519, '\P{Script_Extensions=	-Coptic}', "");
Expect(1, 11519, '\P{^Script_Extensions=	-Coptic}', "");
Expect(0, 11520, '\p{Script_Extensions=	-Coptic}', "");
Expect(1, 11520, '\p{^Script_Extensions=	-Coptic}', "");
Expect(1, 11520, '\P{Script_Extensions=	-Coptic}', "");
Expect(0, 11520, '\P{^Script_Extensions=	-Coptic}', "");
Error('\p{Scx=/a/ 	Copt}');
Error('\P{Scx=/a/ 	Copt}');
Expect(1, 11519, '\p{Scx=copt}', "");
Expect(0, 11519, '\p{^Scx=copt}', "");
Expect(0, 11519, '\P{Scx=copt}', "");
Expect(1, 11519, '\P{^Scx=copt}', "");
Expect(0, 11520, '\p{Scx=copt}', "");
Expect(1, 11520, '\p{^Scx=copt}', "");
Expect(1, 11520, '\P{Scx=copt}', "");
Expect(0, 11520, '\P{^Scx=copt}', "");
Expect(1, 11519, '\p{Scx=	Copt}', "");
Expect(0, 11519, '\p{^Scx=	Copt}', "");
Expect(0, 11519, '\P{Scx=	Copt}', "");
Expect(1, 11519, '\P{^Scx=	Copt}', "");
Expect(0, 11520, '\p{Scx=	Copt}', "");
Expect(1, 11520, '\p{^Scx=	Copt}', "");
Expect(1, 11520, '\P{Scx=	Copt}', "");
Expect(0, 11520, '\P{^Scx=	Copt}', "");
Error('\p{Is_Script_Extensions= :=Qaac}');
Error('\P{Is_Script_Extensions= :=Qaac}');
Expect(1, 11519, '\p{Is_Script_Extensions=qaac}', "");
Expect(0, 11519, '\p{^Is_Script_Extensions=qaac}', "");
Expect(0, 11519, '\P{Is_Script_Extensions=qaac}', "");
Expect(1, 11519, '\P{^Is_Script_Extensions=qaac}', "");
Expect(0, 11520, '\p{Is_Script_Extensions=qaac}', "");
Expect(1, 11520, '\p{^Is_Script_Extensions=qaac}', "");
Expect(1, 11520, '\P{Is_Script_Extensions=qaac}', "");
Expect(0, 11520, '\P{^Is_Script_Extensions=qaac}', "");
Expect(1, 11519, '\p{Is_Script_Extensions:	_-qaac}', "");
Expect(0, 11519, '\p{^Is_Script_Extensions:	_-qaac}', "");
Expect(0, 11519, '\P{Is_Script_Extensions:	_-qaac}', "");
Expect(1, 11519, '\P{^Is_Script_Extensions:	_-qaac}', "");
Expect(0, 11520, '\p{Is_Script_Extensions:	_-qaac}', "");
Expect(1, 11520, '\p{^Is_Script_Extensions:	_-qaac}', "");
Expect(1, 11520, '\P{Is_Script_Extensions:	_-qaac}', "");
Expect(0, 11520, '\P{^Is_Script_Extensions:	_-qaac}', "");
Error('\p{Is_Scx= :=Coptic}');
Error('\P{Is_Scx= :=Coptic}');
Expect(1, 11519, '\p{Is_Scx=coptic}', "");
Expect(0, 11519, '\p{^Is_Scx=coptic}', "");
Expect(0, 11519, '\P{Is_Scx=coptic}', "");
Expect(1, 11519, '\P{^Is_Scx=coptic}', "");
Expect(0, 11520, '\p{Is_Scx=coptic}', "");
Expect(1, 11520, '\p{^Is_Scx=coptic}', "");
Expect(1, 11520, '\P{Is_Scx=coptic}', "");
Expect(0, 11520, '\P{^Is_Scx=coptic}', "");
Expect(1, 11519, '\p{Is_Scx= _COPTIC}', "");
Expect(0, 11519, '\p{^Is_Scx= _COPTIC}', "");
Expect(0, 11519, '\P{Is_Scx= _COPTIC}', "");
Expect(1, 11519, '\P{^Is_Scx= _COPTIC}', "");
Expect(0, 11520, '\p{Is_Scx= _COPTIC}', "");
Expect(1, 11520, '\p{^Is_Scx= _COPTIC}', "");
Expect(1, 11520, '\P{Is_Scx= _COPTIC}', "");
Expect(0, 11520, '\P{^Is_Scx= _COPTIC}', "");
Error('\p{Script_Extensions=	:=BALINESE}');
Error('\P{Script_Extensions=	:=BALINESE}');
Expect(1, 7036, '\p{Script_Extensions=balinese}', "");
Expect(0, 7036, '\p{^Script_Extensions=balinese}', "");
Expect(0, 7036, '\P{Script_Extensions=balinese}', "");
Expect(1, 7036, '\P{^Script_Extensions=balinese}', "");
Expect(0, 7037, '\p{Script_Extensions=balinese}', "");
Expect(1, 7037, '\p{^Script_Extensions=balinese}', "");
Expect(1, 7037, '\P{Script_Extensions=balinese}', "");
Expect(0, 7037, '\P{^Script_Extensions=balinese}', "");
Expect(1, 7036, '\p{Script_Extensions=--Balinese}', "");
Expect(0, 7036, '\p{^Script_Extensions=--Balinese}', "");
Expect(0, 7036, '\P{Script_Extensions=--Balinese}', "");
Expect(1, 7036, '\P{^Script_Extensions=--Balinese}', "");
Expect(0, 7037, '\p{Script_Extensions=--Balinese}', "");
Expect(1, 7037, '\p{^Script_Extensions=--Balinese}', "");
Expect(1, 7037, '\P{Script_Extensions=--Balinese}', "");
Expect(0, 7037, '\P{^Script_Extensions=--Balinese}', "");
Error('\p{Scx=	:=bali}');
Error('\P{Scx=	:=bali}');
Expect(1, 7036, '\p{Scx=bali}', "");
Expect(0, 7036, '\p{^Scx=bali}', "");
Expect(0, 7036, '\P{Scx=bali}', "");
Expect(1, 7036, '\P{^Scx=bali}', "");
Expect(0, 7037, '\p{Scx=bali}', "");
Expect(1, 7037, '\p{^Scx=bali}', "");
Expect(1, 7037, '\P{Scx=bali}', "");
Expect(0, 7037, '\P{^Scx=bali}', "");
Expect(1, 7036, '\p{Scx=-_bali}', "");
Expect(0, 7036, '\p{^Scx=-_bali}', "");
Expect(0, 7036, '\P{Scx=-_bali}', "");
Expect(1, 7036, '\P{^Scx=-_bali}', "");
Expect(0, 7037, '\p{Scx=-_bali}', "");
Expect(1, 7037, '\p{^Scx=-_bali}', "");
Expect(1, 7037, '\P{Scx=-_bali}', "");
Expect(0, 7037, '\P{^Scx=-_bali}', "");
Error('\p{Is_Script_Extensions=:=_	BALINESE}');
Error('\P{Is_Script_Extensions=:=_	BALINESE}');
Expect(1, 7036, '\p{Is_Script_Extensions=balinese}', "");
Expect(0, 7036, '\p{^Is_Script_Extensions=balinese}', "");
Expect(0, 7036, '\P{Is_Script_Extensions=balinese}', "");
Expect(1, 7036, '\P{^Is_Script_Extensions=balinese}', "");
Expect(0, 7037, '\p{Is_Script_Extensions=balinese}', "");
Expect(1, 7037, '\p{^Is_Script_Extensions=balinese}', "");
Expect(1, 7037, '\P{Is_Script_Extensions=balinese}', "");
Expect(0, 7037, '\P{^Is_Script_Extensions=balinese}', "");
Expect(1, 7036, '\p{Is_Script_Extensions:	 _BALINESE}', "");
Expect(0, 7036, '\p{^Is_Script_Extensions:	 _BALINESE}', "");
Expect(0, 7036, '\P{Is_Script_Extensions:	 _BALINESE}', "");
Expect(1, 7036, '\P{^Is_Script_Extensions:	 _BALINESE}', "");
Expect(0, 7037, '\p{Is_Script_Extensions:	 _BALINESE}', "");
Expect(1, 7037, '\p{^Is_Script_Extensions:	 _BALINESE}', "");
Expect(1, 7037, '\P{Is_Script_Extensions:	 _BALINESE}', "");
Expect(0, 7037, '\P{^Is_Script_Extensions:	 _BALINESE}', "");
Error('\p{Is_Scx=/a/_bali}');
Error('\P{Is_Scx=/a/_bali}');
Expect(1, 7036, '\p{Is_Scx:	bali}', "");
Expect(0, 7036, '\p{^Is_Scx:	bali}', "");
Expect(0, 7036, '\P{Is_Scx:	bali}', "");
Expect(1, 7036, '\P{^Is_Scx:	bali}', "");
Expect(0, 7037, '\p{Is_Scx:	bali}', "");
Expect(1, 7037, '\p{^Is_Scx:	bali}', "");
Expect(1, 7037, '\P{Is_Scx:	bali}', "");
Expect(0, 7037, '\P{^Is_Scx:	bali}', "");
Expect(1, 7036, '\p{Is_Scx= -BALI}', "");
Expect(0, 7036, '\p{^Is_Scx= -BALI}', "");
Expect(0, 7036, '\P{Is_Scx= -BALI}', "");
Expect(1, 7036, '\P{^Is_Scx= -BALI}', "");
Expect(0, 7037, '\p{Is_Scx= -BALI}', "");
Expect(1, 7037, '\p{^Is_Scx= -BALI}', "");
Expect(1, 7037, '\P{Is_Scx= -BALI}', "");
Expect(0, 7037, '\P{^Is_Scx= -BALI}', "");
Error('\p{Script_Extensions=Katakana_Or_Hiragana}');
Error('\P{Script_Extensions=Katakana_Or_Hiragana}');
Error('\p{Scx:	Hrkt}');
Error('\P{Scx:	Hrkt}');
Error('\p{Is_Script_Extensions=Katakana_Or_Hiragana}');
Error('\P{Is_Script_Extensions=Katakana_Or_Hiragana}');
Error('\p{Is_Scx=Hrkt}');
Error('\P{Is_Scx=Hrkt}');
Error('\p{Script_Extensions=/a/_-Meetei_mayek}');
Error('\P{Script_Extensions=/a/_-Meetei_mayek}');
Expect(1, 44025, '\p{Script_Extensions=meeteimayek}', "");
Expect(0, 44025, '\p{^Script_Extensions=meeteimayek}', "");
Expect(0, 44025, '\P{Script_Extensions=meeteimayek}', "");
Expect(1, 44025, '\P{^Script_Extensions=meeteimayek}', "");
Expect(0, 44026, '\p{Script_Extensions=meeteimayek}', "");
Expect(1, 44026, '\p{^Script_Extensions=meeteimayek}', "");
Expect(1, 44026, '\P{Script_Extensions=meeteimayek}', "");
Expect(0, 44026, '\P{^Script_Extensions=meeteimayek}', "");
Expect(1, 44025, '\p{Script_Extensions= _meetei_Mayek}', "");
Expect(0, 44025, '\p{^Script_Extensions= _meetei_Mayek}', "");
Expect(0, 44025, '\P{Script_Extensions= _meetei_Mayek}', "");
Expect(1, 44025, '\P{^Script_Extensions= _meetei_Mayek}', "");
Expect(0, 44026, '\p{Script_Extensions= _meetei_Mayek}', "");
Expect(1, 44026, '\p{^Script_Extensions= _meetei_Mayek}', "");
Expect(1, 44026, '\P{Script_Extensions= _meetei_Mayek}', "");
Expect(0, 44026, '\P{^Script_Extensions= _meetei_Mayek}', "");
Error('\p{Scx=/a/Mtei}');
Error('\P{Scx=/a/Mtei}');
Expect(1, 44025, '\p{Scx:   mtei}', "");
Expect(0, 44025, '\p{^Scx:   mtei}', "");
Expect(0, 44025, '\P{Scx:   mtei}', "");
Expect(1, 44025, '\P{^Scx:   mtei}', "");
Expect(0, 44026, '\p{Scx:   mtei}', "");
Expect(1, 44026, '\p{^Scx:   mtei}', "");
Expect(1, 44026, '\P{Scx:   mtei}', "");
Expect(0, 44026, '\P{^Scx:   mtei}', "");
Expect(1, 44025, '\p{Scx=_ MTEI}', "");
Expect(0, 44025, '\p{^Scx=_ MTEI}', "");
Expect(0, 44025, '\P{Scx=_ MTEI}', "");
Expect(1, 44025, '\P{^Scx=_ MTEI}', "");
Expect(0, 44026, '\p{Scx=_ MTEI}', "");
Expect(1, 44026, '\p{^Scx=_ MTEI}', "");
Expect(1, 44026, '\P{Scx=_ MTEI}', "");
Expect(0, 44026, '\P{^Scx=_ MTEI}', "");
Error('\p{Is_Script_Extensions=-/a/meetei_Mayek}');
Error('\P{Is_Script_Extensions=-/a/meetei_Mayek}');
Expect(1, 44025, '\p{Is_Script_Extensions:	meeteimayek}', "");
Expect(0, 44025, '\p{^Is_Script_Extensions:	meeteimayek}', "");
Expect(0, 44025, '\P{Is_Script_Extensions:	meeteimayek}', "");
Expect(1, 44025, '\P{^Is_Script_Extensions:	meeteimayek}', "");
Expect(0, 44026, '\p{Is_Script_Extensions:	meeteimayek}', "");
Expect(1, 44026, '\p{^Is_Script_Extensions:	meeteimayek}', "");
Expect(1, 44026, '\P{Is_Script_Extensions:	meeteimayek}', "");
Expect(0, 44026, '\P{^Is_Script_Extensions:	meeteimayek}', "");
Expect(1, 44025, '\p{Is_Script_Extensions=		Meetei_Mayek}', "");
Expect(0, 44025, '\p{^Is_Script_Extensions=		Meetei_Mayek}', "");
Expect(0, 44025, '\P{Is_Script_Extensions=		Meetei_Mayek}', "");
Expect(1, 44025, '\P{^Is_Script_Extensions=		Meetei_Mayek}', "");
Expect(0, 44026, '\p{Is_Script_Extensions=		Meetei_Mayek}', "");
Expect(1, 44026, '\p{^Is_Script_Extensions=		Meetei_Mayek}', "");
Expect(1, 44026, '\P{Is_Script_Extensions=		Meetei_Mayek}', "");
Expect(0, 44026, '\P{^Is_Script_Extensions=		Meetei_Mayek}', "");
Error('\p{Is_Scx=/a/- Mtei}');
Error('\P{Is_Scx=/a/- Mtei}');
Expect(1, 44025, '\p{Is_Scx=mtei}', "");
Expect(0, 44025, '\p{^Is_Scx=mtei}', "");
Expect(0, 44025, '\P{Is_Scx=mtei}', "");
Expect(1, 44025, '\P{^Is_Scx=mtei}', "");
Expect(0, 44026, '\p{Is_Scx=mtei}', "");
Expect(1, 44026, '\p{^Is_Scx=mtei}', "");
Expect(1, 44026, '\P{Is_Scx=mtei}', "");
Expect(0, 44026, '\P{^Is_Scx=mtei}', "");
Expect(1, 44025, '\p{Is_Scx=-Mtei}', "");
Expect(0, 44025, '\p{^Is_Scx=-Mtei}', "");
Expect(0, 44025, '\P{Is_Scx=-Mtei}', "");
Expect(1, 44025, '\P{^Is_Scx=-Mtei}', "");
Expect(0, 44026, '\p{Is_Scx=-Mtei}', "");
Expect(1, 44026, '\p{^Is_Scx=-Mtei}', "");
Expect(1, 44026, '\P{Is_Scx=-Mtei}', "");
Expect(0, 44026, '\P{^Is_Scx=-Mtei}', "");
Error('\p{Script_Extensions=/a/	_Syloti_nagri}');
Error('\P{Script_Extensions=/a/	_Syloti_nagri}');
Expect(1, 43051, '\p{Script_Extensions=sylotinagri}', "");
Expect(0, 43051, '\p{^Script_Extensions=sylotinagri}', "");
Expect(0, 43051, '\P{Script_Extensions=sylotinagri}', "");
Expect(1, 43051, '\P{^Script_Extensions=sylotinagri}', "");
Expect(0, 43052, '\p{Script_Extensions=sylotinagri}', "");
Expect(1, 43052, '\p{^Script_Extensions=sylotinagri}', "");
Expect(1, 43052, '\P{Script_Extensions=sylotinagri}', "");
Expect(0, 43052, '\P{^Script_Extensions=sylotinagri}', "");
Expect(1, 43051, '\p{Script_Extensions=	SYLOTI_Nagri}', "");
Expect(0, 43051, '\p{^Script_Extensions=	SYLOTI_Nagri}', "");
Expect(0, 43051, '\P{Script_Extensions=	SYLOTI_Nagri}', "");
Expect(1, 43051, '\P{^Script_Extensions=	SYLOTI_Nagri}', "");
Expect(0, 43052, '\p{Script_Extensions=	SYLOTI_Nagri}', "");
Expect(1, 43052, '\p{^Script_Extensions=	SYLOTI_Nagri}', "");
Expect(1, 43052, '\P{Script_Extensions=	SYLOTI_Nagri}', "");
Expect(0, 43052, '\P{^Script_Extensions=	SYLOTI_Nagri}', "");
Error('\p{Scx=-	sylo:=}');
Error('\P{Scx=-	sylo:=}');
Expect(1, 43051, '\p{Scx=sylo}', "");
Expect(0, 43051, '\p{^Scx=sylo}', "");
Expect(0, 43051, '\P{Scx=sylo}', "");
Expect(1, 43051, '\P{^Scx=sylo}', "");
Expect(0, 43052, '\p{Scx=sylo}', "");
Expect(1, 43052, '\p{^Scx=sylo}', "");
Expect(1, 43052, '\P{Scx=sylo}', "");
Expect(0, 43052, '\P{^Scx=sylo}', "");
Expect(1, 43051, '\p{Scx= 	Sylo}', "");
Expect(0, 43051, '\p{^Scx= 	Sylo}', "");
Expect(0, 43051, '\P{Scx= 	Sylo}', "");
Expect(1, 43051, '\P{^Scx= 	Sylo}', "");
Expect(0, 43052, '\p{Scx= 	Sylo}', "");
Expect(1, 43052, '\p{^Scx= 	Sylo}', "");
Expect(1, 43052, '\P{Scx= 	Sylo}', "");
Expect(0, 43052, '\P{^Scx= 	Sylo}', "");
Error('\p{Is_Script_Extensions=/a/	Syloti_nagri}');
Error('\P{Is_Script_Extensions=/a/	Syloti_nagri}');
Expect(1, 43051, '\p{Is_Script_Extensions=sylotinagri}', "");
Expect(0, 43051, '\p{^Is_Script_Extensions=sylotinagri}', "");
Expect(0, 43051, '\P{Is_Script_Extensions=sylotinagri}', "");
Expect(1, 43051, '\P{^Is_Script_Extensions=sylotinagri}', "");
Expect(0, 43052, '\p{Is_Script_Extensions=sylotinagri}', "");
Expect(1, 43052, '\p{^Is_Script_Extensions=sylotinagri}', "");
Expect(1, 43052, '\P{Is_Script_Extensions=sylotinagri}', "");
Expect(0, 43052, '\P{^Is_Script_Extensions=sylotinagri}', "");
Expect(1, 43051, '\p{Is_Script_Extensions= Syloti_Nagri}', "");
Expect(0, 43051, '\p{^Is_Script_Extensions= Syloti_Nagri}', "");
Expect(0, 43051, '\P{Is_Script_Extensions= Syloti_Nagri}', "");
Expect(1, 43051, '\P{^Is_Script_Extensions= Syloti_Nagri}', "");
Expect(0, 43052, '\p{Is_Script_Extensions= Syloti_Nagri}', "");
Expect(1, 43052, '\p{^Is_Script_Extensions= Syloti_Nagri}', "");
Expect(1, 43052, '\P{Is_Script_Extensions= Syloti_Nagri}', "");
Expect(0, 43052, '\P{^Is_Script_Extensions= Syloti_Nagri}', "");
Error('\p{Is_Scx= 	Sylo/a/}');
Error('\P{Is_Scx= 	Sylo/a/}');
Expect(1, 43051, '\p{Is_Scx=sylo}', "");
Expect(0, 43051, '\p{^Is_Scx=sylo}', "");
Expect(0, 43051, '\P{Is_Scx=sylo}', "");
Expect(1, 43051, '\P{^Is_Scx=sylo}', "");
Expect(0, 43052, '\p{Is_Scx=sylo}', "");
Expect(1, 43052, '\p{^Is_Scx=sylo}', "");
Expect(1, 43052, '\P{Is_Scx=sylo}', "");
Expect(0, 43052, '\P{^Is_Scx=sylo}', "");
Expect(1, 43051, '\p{Is_Scx=-	SYLO}', "");
Expect(0, 43051, '\p{^Is_Scx=-	SYLO}', "");
Expect(0, 43051, '\P{Is_Scx=-	SYLO}', "");
Expect(1, 43051, '\P{^Is_Scx=-	SYLO}', "");
Expect(0, 43052, '\p{Is_Scx=-	SYLO}', "");
Expect(1, 43052, '\p{^Is_Scx=-	SYLO}', "");
Expect(1, 43052, '\P{Is_Scx=-	SYLO}', "");
Expect(0, 43052, '\P{^Is_Scx=-	SYLO}', "");
Error('\p{Script_Extensions=-	tagbanwa/a/}');
Error('\P{Script_Extensions=-	tagbanwa/a/}');
Expect(1, 6003, '\p{Script_Extensions=tagbanwa}', "");
Expect(0, 6003, '\p{^Script_Extensions=tagbanwa}', "");
Expect(0, 6003, '\P{Script_Extensions=tagbanwa}', "");
Expect(1, 6003, '\P{^Script_Extensions=tagbanwa}', "");
Expect(0, 6004, '\p{Script_Extensions=tagbanwa}', "");
Expect(1, 6004, '\p{^Script_Extensions=tagbanwa}', "");
Expect(1, 6004, '\P{Script_Extensions=tagbanwa}', "");
Expect(0, 6004, '\P{^Script_Extensions=tagbanwa}', "");
Expect(1, 6003, '\p{Script_Extensions: 		Tagbanwa}', "");
Expect(0, 6003, '\p{^Script_Extensions: 		Tagbanwa}', "");
Expect(0, 6003, '\P{Script_Extensions: 		Tagbanwa}', "");
Expect(1, 6003, '\P{^Script_Extensions: 		Tagbanwa}', "");
Expect(0, 6004, '\p{Script_Extensions: 		Tagbanwa}', "");
Expect(1, 6004, '\p{^Script_Extensions: 		Tagbanwa}', "");
Expect(1, 6004, '\P{Script_Extensions: 		Tagbanwa}', "");
Expect(0, 6004, '\P{^Script_Extensions: 		Tagbanwa}', "");
Error('\p{Scx=_:=TAGB}');
Error('\P{Scx=_:=TAGB}');
Expect(1, 6003, '\p{Scx=tagb}', "");
Expect(0, 6003, '\p{^Scx=tagb}', "");
Expect(0, 6003, '\P{Scx=tagb}', "");
Expect(1, 6003, '\P{^Scx=tagb}', "");
Expect(0, 6004, '\p{Scx=tagb}', "");
Expect(1, 6004, '\p{^Scx=tagb}', "");
Expect(1, 6004, '\P{Scx=tagb}', "");
Expect(0, 6004, '\P{^Scx=tagb}', "");
Expect(1, 6003, '\p{Scx=	_tagb}', "");
Expect(0, 6003, '\p{^Scx=	_tagb}', "");
Expect(0, 6003, '\P{Scx=	_tagb}', "");
Expect(1, 6003, '\P{^Scx=	_tagb}', "");
Expect(0, 6004, '\p{Scx=	_tagb}', "");
Expect(1, 6004, '\p{^Scx=	_tagb}', "");
Expect(1, 6004, '\P{Scx=	_tagb}', "");
Expect(0, 6004, '\P{^Scx=	_tagb}', "");
Error('\p{Is_Script_Extensions=:=	_tagbanwa}');
Error('\P{Is_Script_Extensions=:=	_tagbanwa}');
Expect(1, 6003, '\p{Is_Script_Extensions=tagbanwa}', "");
Expect(0, 6003, '\p{^Is_Script_Extensions=tagbanwa}', "");
Expect(0, 6003, '\P{Is_Script_Extensions=tagbanwa}', "");
Expect(1, 6003, '\P{^Is_Script_Extensions=tagbanwa}', "");
Expect(0, 6004, '\p{Is_Script_Extensions=tagbanwa}', "");
Expect(1, 6004, '\p{^Is_Script_Extensions=tagbanwa}', "");
Expect(1, 6004, '\P{Is_Script_Extensions=tagbanwa}', "");
Expect(0, 6004, '\P{^Is_Script_Extensions=tagbanwa}', "");
Expect(1, 6003, '\p{Is_Script_Extensions=	tagbanwa}', "");
Expect(0, 6003, '\p{^Is_Script_Extensions=	tagbanwa}', "");
Expect(0, 6003, '\P{Is_Script_Extensions=	tagbanwa}', "");
Expect(1, 6003, '\P{^Is_Script_Extensions=	tagbanwa}', "");
Expect(0, 6004, '\p{Is_Script_Extensions=	tagbanwa}', "");
Expect(1, 6004, '\p{^Is_Script_Extensions=	tagbanwa}', "");
Expect(1, 6004, '\P{Is_Script_Extensions=	tagbanwa}', "");
Expect(0, 6004, '\P{^Is_Script_Extensions=	tagbanwa}', "");
Error('\p{Is_Scx=_Tagb/a/}');
Error('\P{Is_Scx=_Tagb/a/}');
Expect(1, 6003, '\p{Is_Scx=tagb}', "");
Expect(0, 6003, '\p{^Is_Scx=tagb}', "");
Expect(0, 6003, '\P{Is_Scx=tagb}', "");
Expect(1, 6003, '\P{^Is_Scx=tagb}', "");
Expect(0, 6004, '\p{Is_Scx=tagb}', "");
Expect(1, 6004, '\p{^Is_Scx=tagb}', "");
Expect(1, 6004, '\P{Is_Scx=tagb}', "");
Expect(0, 6004, '\P{^Is_Scx=tagb}', "");
Expect(1, 6003, '\p{Is_Scx=TAGB}', "");
Expect(0, 6003, '\p{^Is_Scx=TAGB}', "");
Expect(0, 6003, '\P{Is_Scx=TAGB}', "");
Expect(1, 6003, '\P{^Is_Scx=TAGB}', "");
Expect(0, 6004, '\p{Is_Scx=TAGB}', "");
Expect(1, 6004, '\p{^Is_Scx=TAGB}', "");
Expect(1, 6004, '\P{Is_Scx=TAGB}', "");
Expect(0, 6004, '\P{^Is_Scx=TAGB}', "");
Error('\p{Script_Extensions=:=_Phoenician}');
Error('\P{Script_Extensions=:=_Phoenician}');
Expect(1, 67871, '\p{Script_Extensions=phoenician}', "");
Expect(0, 67871, '\p{^Script_Extensions=phoenician}', "");
Expect(0, 67871, '\P{Script_Extensions=phoenician}', "");
Expect(1, 67871, '\P{^Script_Extensions=phoenician}', "");
Expect(0, 67872, '\p{Script_Extensions=phoenician}', "");
Expect(1, 67872, '\p{^Script_Extensions=phoenician}', "");
Expect(1, 67872, '\P{Script_Extensions=phoenician}', "");
Expect(0, 67872, '\P{^Script_Extensions=phoenician}', "");
Expect(1, 67871, '\p{Script_Extensions=- Phoenician}', "");
Expect(0, 67871, '\p{^Script_Extensions=- Phoenician}', "");
Expect(0, 67871, '\P{Script_Extensions=- Phoenician}', "");
Expect(1, 67871, '\P{^Script_Extensions=- Phoenician}', "");
Expect(0, 67872, '\p{Script_Extensions=- Phoenician}', "");
Expect(1, 67872, '\p{^Script_Extensions=- Phoenician}', "");
Expect(1, 67872, '\P{Script_Extensions=- Phoenician}', "");
Expect(0, 67872, '\P{^Script_Extensions=- Phoenician}', "");
Error('\p{Scx=/a/_ PHNX}');
Error('\P{Scx=/a/_ PHNX}');
Expect(1, 67871, '\p{Scx=phnx}', "");
Expect(0, 67871, '\p{^Scx=phnx}', "");
Expect(0, 67871, '\P{Scx=phnx}', "");
Expect(1, 67871, '\P{^Scx=phnx}', "");
Expect(0, 67872, '\p{Scx=phnx}', "");
Expect(1, 67872, '\p{^Scx=phnx}', "");
Expect(1, 67872, '\P{Scx=phnx}', "");
Expect(0, 67872, '\P{^Scx=phnx}', "");
Expect(1, 67871, '\p{Scx= Phnx}', "");
Expect(0, 67871, '\p{^Scx= Phnx}', "");
Expect(0, 67871, '\P{Scx= Phnx}', "");
Expect(1, 67871, '\P{^Scx= Phnx}', "");
Expect(0, 67872, '\p{Scx= Phnx}', "");
Expect(1, 67872, '\p{^Scx= Phnx}', "");
Expect(1, 67872, '\P{Scx= Phnx}', "");
Expect(0, 67872, '\P{^Scx= Phnx}', "");
Error('\p{Is_Script_Extensions=/a/-PHOENICIAN}');
Error('\P{Is_Script_Extensions=/a/-PHOENICIAN}');
Expect(1, 67871, '\p{Is_Script_Extensions=phoenician}', "");
Expect(0, 67871, '\p{^Is_Script_Extensions=phoenician}', "");
Expect(0, 67871, '\P{Is_Script_Extensions=phoenician}', "");
Expect(1, 67871, '\P{^Is_Script_Extensions=phoenician}', "");
Expect(0, 67872, '\p{Is_Script_Extensions=phoenician}', "");
Expect(1, 67872, '\p{^Is_Script_Extensions=phoenician}', "");
Expect(1, 67872, '\P{Is_Script_Extensions=phoenician}', "");
Expect(0, 67872, '\P{^Is_Script_Extensions=phoenician}', "");
Expect(1, 67871, '\p{Is_Script_Extensions=_Phoenician}', "");
Expect(0, 67871, '\p{^Is_Script_Extensions=_Phoenician}', "");
Expect(0, 67871, '\P{Is_Script_Extensions=_Phoenician}', "");
Expect(1, 67871, '\P{^Is_Script_Extensions=_Phoenician}', "");
Expect(0, 67872, '\p{Is_Script_Extensions=_Phoenician}', "");
Expect(1, 67872, '\p{^Is_Script_Extensions=_Phoenician}', "");
Expect(1, 67872, '\P{Is_Script_Extensions=_Phoenician}', "");
Expect(0, 67872, '\P{^Is_Script_Extensions=_Phoenician}', "");
Error('\p{Is_Scx=/a/	-phnx}');
Error('\P{Is_Scx=/a/	-phnx}');
Expect(1, 67871, '\p{Is_Scx=phnx}', "");
Expect(0, 67871, '\p{^Is_Scx=phnx}', "");
Expect(0, 67871, '\P{Is_Scx=phnx}', "");
Expect(1, 67871, '\P{^Is_Scx=phnx}', "");
Expect(0, 67872, '\p{Is_Scx=phnx}', "");
Expect(1, 67872, '\p{^Is_Scx=phnx}', "");
Expect(1, 67872, '\P{Is_Scx=phnx}', "");
Expect(0, 67872, '\P{^Is_Scx=phnx}', "");
Expect(1, 67871, '\p{Is_Scx=		Phnx}', "");
Expect(0, 67871, '\p{^Is_Scx=		Phnx}', "");
Expect(0, 67871, '\P{Is_Scx=		Phnx}', "");
Expect(1, 67871, '\P{^Is_Scx=		Phnx}', "");
Expect(0, 67872, '\p{Is_Scx=		Phnx}', "");
Expect(1, 67872, '\p{^Is_Scx=		Phnx}', "");
Expect(1, 67872, '\P{Is_Scx=		Phnx}', "");
Expect(0, 67872, '\P{^Is_Scx=		Phnx}', "");
Error('\p{Script_Extensions=--Devanagari:=}');
Error('\P{Script_Extensions=--Devanagari:=}');
Expect(1, 43259, '\p{Script_Extensions=devanagari}', "");
Expect(0, 43259, '\p{^Script_Extensions=devanagari}', "");
Expect(0, 43259, '\P{Script_Extensions=devanagari}', "");
Expect(1, 43259, '\P{^Script_Extensions=devanagari}', "");
Expect(0, 43260, '\p{Script_Extensions=devanagari}', "");
Expect(1, 43260, '\p{^Script_Extensions=devanagari}', "");
Expect(1, 43260, '\P{Script_Extensions=devanagari}', "");
Expect(0, 43260, '\P{^Script_Extensions=devanagari}', "");
Expect(1, 43259, '\p{Script_Extensions=_	Devanagari}', "");
Expect(0, 43259, '\p{^Script_Extensions=_	Devanagari}', "");
Expect(0, 43259, '\P{Script_Extensions=_	Devanagari}', "");
Expect(1, 43259, '\P{^Script_Extensions=_	Devanagari}', "");
Expect(0, 43260, '\p{Script_Extensions=_	Devanagari}', "");
Expect(1, 43260, '\p{^Script_Extensions=_	Devanagari}', "");
Expect(1, 43260, '\P{Script_Extensions=_	Devanagari}', "");
Expect(0, 43260, '\P{^Script_Extensions=_	Devanagari}', "");
Error('\p{Scx= :=Deva}');
Error('\P{Scx= :=Deva}');
Expect(1, 43259, '\p{Scx=deva}', "");
Expect(0, 43259, '\p{^Scx=deva}', "");
Expect(0, 43259, '\P{Scx=deva}', "");
Expect(1, 43259, '\P{^Scx=deva}', "");
Expect(0, 43260, '\p{Scx=deva}', "");
Expect(1, 43260, '\p{^Scx=deva}', "");
Expect(1, 43260, '\P{Scx=deva}', "");
Expect(0, 43260, '\P{^Scx=deva}', "");
Expect(1, 43259, '\p{Scx= 	Deva}', "");
Expect(0, 43259, '\p{^Scx= 	Deva}', "");
Expect(0, 43259, '\P{Scx= 	Deva}', "");
Expect(1, 43259, '\P{^Scx= 	Deva}', "");
Expect(0, 43260, '\p{Scx= 	Deva}', "");
Expect(1, 43260, '\p{^Scx= 	Deva}', "");
Expect(1, 43260, '\P{Scx= 	Deva}', "");
Expect(0, 43260, '\P{^Scx= 	Deva}', "");
Error('\p{Is_Script_Extensions=:=	DEVANAGARI}');
Error('\P{Is_Script_Extensions=:=	DEVANAGARI}');
Expect(1, 43259, '\p{Is_Script_Extensions=devanagari}', "");
Expect(0, 43259, '\p{^Is_Script_Extensions=devanagari}', "");
Expect(0, 43259, '\P{Is_Script_Extensions=devanagari}', "");
Expect(1, 43259, '\P{^Is_Script_Extensions=devanagari}', "");
Expect(0, 43260, '\p{Is_Script_Extensions=devanagari}', "");
Expect(1, 43260, '\p{^Is_Script_Extensions=devanagari}', "");
Expect(1, 43260, '\P{Is_Script_Extensions=devanagari}', "");
Expect(0, 43260, '\P{^Is_Script_Extensions=devanagari}', "");
Expect(1, 43259, '\p{Is_Script_Extensions=	_devanagari}', "");
Expect(0, 43259, '\p{^Is_Script_Extensions=	_devanagari}', "");
Expect(0, 43259, '\P{Is_Script_Extensions=	_devanagari}', "");
Expect(1, 43259, '\P{^Is_Script_Extensions=	_devanagari}', "");
Expect(0, 43260, '\p{Is_Script_Extensions=	_devanagari}', "");
Expect(1, 43260, '\p{^Is_Script_Extensions=	_devanagari}', "");
Expect(1, 43260, '\P{Is_Script_Extensions=	_devanagari}', "");
Expect(0, 43260, '\P{^Is_Script_Extensions=	_devanagari}', "");
Error('\p{Is_Scx= _Deva:=}');
Error('\P{Is_Scx= _Deva:=}');
Expect(1, 43259, '\p{Is_Scx=deva}', "");
Expect(0, 43259, '\p{^Is_Scx=deva}', "");
Expect(0, 43259, '\P{Is_Scx=deva}', "");
Expect(1, 43259, '\P{^Is_Scx=deva}', "");
Expect(0, 43260, '\p{Is_Scx=deva}', "");
Expect(1, 43260, '\p{^Is_Scx=deva}', "");
Expect(1, 43260, '\P{Is_Scx=deva}', "");
Expect(0, 43260, '\P{^Is_Scx=deva}', "");
Expect(1, 43259, '\p{Is_Scx=-_DEVA}', "");
Expect(0, 43259, '\p{^Is_Scx=-_DEVA}', "");
Expect(0, 43259, '\P{Is_Scx=-_DEVA}', "");
Expect(1, 43259, '\P{^Is_Scx=-_DEVA}', "");
Expect(0, 43260, '\p{Is_Scx=-_DEVA}', "");
Expect(1, 43260, '\p{^Is_Scx=-_DEVA}', "");
Expect(1, 43260, '\P{Is_Scx=-_DEVA}', "");
Expect(0, 43260, '\P{^Is_Scx=-_DEVA}', "");
Error('\p{Script_Extensions=_:=CARIAN}');
Error('\P{Script_Extensions=_:=CARIAN}');
Expect(1, 66256, '\p{Script_Extensions=carian}', "");
Expect(0, 66256, '\p{^Script_Extensions=carian}', "");
Expect(0, 66256, '\P{Script_Extensions=carian}', "");
Expect(1, 66256, '\P{^Script_Extensions=carian}', "");
Expect(0, 66257, '\p{Script_Extensions=carian}', "");
Expect(1, 66257, '\p{^Script_Extensions=carian}', "");
Expect(1, 66257, '\P{Script_Extensions=carian}', "");
Expect(0, 66257, '\P{^Script_Extensions=carian}', "");
Expect(1, 66256, '\p{Script_Extensions=_	carian}', "");
Expect(0, 66256, '\p{^Script_Extensions=_	carian}', "");
Expect(0, 66256, '\P{Script_Extensions=_	carian}', "");
Expect(1, 66256, '\P{^Script_Extensions=_	carian}', "");
Expect(0, 66257, '\p{Script_Extensions=_	carian}', "");
Expect(1, 66257, '\p{^Script_Extensions=_	carian}', "");
Expect(1, 66257, '\P{Script_Extensions=_	carian}', "");
Expect(0, 66257, '\P{^Script_Extensions=_	carian}', "");
Error('\p{Scx=-:=Cari}');
Error('\P{Scx=-:=Cari}');
Expect(1, 66256, '\p{Scx:cari}', "");
Expect(0, 66256, '\p{^Scx:cari}', "");
Expect(0, 66256, '\P{Scx:cari}', "");
Expect(1, 66256, '\P{^Scx:cari}', "");
Expect(0, 66257, '\p{Scx:cari}', "");
Expect(1, 66257, '\p{^Scx:cari}', "");
Expect(1, 66257, '\P{Scx:cari}', "");
Expect(0, 66257, '\P{^Scx:cari}', "");
Expect(1, 66256, '\p{Scx=--Cari}', "");
Expect(0, 66256, '\p{^Scx=--Cari}', "");
Expect(0, 66256, '\P{Scx=--Cari}', "");
Expect(1, 66256, '\P{^Scx=--Cari}', "");
Expect(0, 66257, '\p{Scx=--Cari}', "");
Expect(1, 66257, '\p{^Scx=--Cari}', "");
Expect(1, 66257, '\P{Scx=--Cari}', "");
Expect(0, 66257, '\P{^Scx=--Cari}', "");
Error('\p{Is_Script_Extensions=_:=CARIAN}');
Error('\P{Is_Script_Extensions=_:=CARIAN}');
Expect(1, 66256, '\p{Is_Script_Extensions=carian}', "");
Expect(0, 66256, '\p{^Is_Script_Extensions=carian}', "");
Expect(0, 66256, '\P{Is_Script_Extensions=carian}', "");
Expect(1, 66256, '\P{^Is_Script_Extensions=carian}', "");
Expect(0, 66257, '\p{Is_Script_Extensions=carian}', "");
Expect(1, 66257, '\p{^Is_Script_Extensions=carian}', "");
Expect(1, 66257, '\P{Is_Script_Extensions=carian}', "");
Expect(0, 66257, '\P{^Is_Script_Extensions=carian}', "");
Expect(1, 66256, '\p{Is_Script_Extensions=_Carian}', "");
Expect(0, 66256, '\p{^Is_Script_Extensions=_Carian}', "");
Expect(0, 66256, '\P{Is_Script_Extensions=_Carian}', "");
Expect(1, 66256, '\P{^Is_Script_Extensions=_Carian}', "");
Expect(0, 66257, '\p{Is_Script_Extensions=_Carian}', "");
Expect(1, 66257, '\p{^Is_Script_Extensions=_Carian}', "");
Expect(1, 66257, '\P{Is_Script_Extensions=_Carian}', "");
Expect(0, 66257, '\P{^Is_Script_Extensions=_Carian}', "");
Error('\p{Is_Scx=:=Cari}');
Error('\P{Is_Scx=:=Cari}');
Expect(1, 66256, '\p{Is_Scx=cari}', "");
Expect(0, 66256, '\p{^Is_Scx=cari}', "");
Expect(0, 66256, '\P{Is_Scx=cari}', "");
Expect(1, 66256, '\P{^Is_Scx=cari}', "");
Expect(0, 66257, '\p{Is_Scx=cari}', "");
Expect(1, 66257, '\p{^Is_Scx=cari}', "");
Expect(1, 66257, '\P{Is_Scx=cari}', "");
Expect(0, 66257, '\P{^Is_Scx=cari}', "");
Error('\p{Script_Extensions=:=--sharada}');
Error('\P{Script_Extensions=:=--sharada}');
Expect(1, 70105, '\p{Script_Extensions=sharada}', "");
Expect(0, 70105, '\p{^Script_Extensions=sharada}', "");
Expect(0, 70105, '\P{Script_Extensions=sharada}', "");
Expect(1, 70105, '\P{^Script_Extensions=sharada}', "");
Expect(0, 70106, '\p{Script_Extensions=sharada}', "");
Expect(1, 70106, '\p{^Script_Extensions=sharada}', "");
Expect(1, 70106, '\P{Script_Extensions=sharada}', "");
Expect(0, 70106, '\P{^Script_Extensions=sharada}', "");
Expect(1, 70105, '\p{Script_Extensions=	 Sharada}', "");
Expect(0, 70105, '\p{^Script_Extensions=	 Sharada}', "");
Expect(0, 70105, '\P{Script_Extensions=	 Sharada}', "");
Expect(1, 70105, '\P{^Script_Extensions=	 Sharada}', "");
Expect(0, 70106, '\p{Script_Extensions=	 Sharada}', "");
Expect(1, 70106, '\p{^Script_Extensions=	 Sharada}', "");
Expect(1, 70106, '\P{Script_Extensions=	 Sharada}', "");
Expect(0, 70106, '\P{^Script_Extensions=	 Sharada}', "");
Error('\p{Scx=-	shrd:=}');
Error('\P{Scx=-	shrd:=}');
Expect(1, 70105, '\p{Scx:   shrd}', "");
Expect(0, 70105, '\p{^Scx:   shrd}', "");
Expect(0, 70105, '\P{Scx:   shrd}', "");
Expect(1, 70105, '\P{^Scx:   shrd}', "");
Expect(0, 70106, '\p{Scx:   shrd}', "");
Expect(1, 70106, '\p{^Scx:   shrd}', "");
Expect(1, 70106, '\P{Scx:   shrd}', "");
Expect(0, 70106, '\P{^Scx:   shrd}', "");
Expect(1, 70105, '\p{Scx=	 Shrd}', "");
Expect(0, 70105, '\p{^Scx=	 Shrd}', "");
Expect(0, 70105, '\P{Scx=	 Shrd}', "");
Expect(1, 70105, '\P{^Scx=	 Shrd}', "");
Expect(0, 70106, '\p{Scx=	 Shrd}', "");
Expect(1, 70106, '\p{^Scx=	 Shrd}', "");
Expect(1, 70106, '\P{Scx=	 Shrd}', "");
Expect(0, 70106, '\P{^Scx=	 Shrd}', "");
Error('\p{Is_Script_Extensions=-:=Sharada}');
Error('\P{Is_Script_Extensions=-:=Sharada}');
Expect(1, 70105, '\p{Is_Script_Extensions=sharada}', "");
Expect(0, 70105, '\p{^Is_Script_Extensions=sharada}', "");
Expect(0, 70105, '\P{Is_Script_Extensions=sharada}', "");
Expect(1, 70105, '\P{^Is_Script_Extensions=sharada}', "");
Expect(0, 70106, '\p{Is_Script_Extensions=sharada}', "");
Expect(1, 70106, '\p{^Is_Script_Extensions=sharada}', "");
Expect(1, 70106, '\P{Is_Script_Extensions=sharada}', "");
Expect(0, 70106, '\P{^Is_Script_Extensions=sharada}', "");
Expect(1, 70105, '\p{Is_Script_Extensions=	sharada}', "");
Expect(0, 70105, '\p{^Is_Script_Extensions=	sharada}', "");
Expect(0, 70105, '\P{Is_Script_Extensions=	sharada}', "");
Expect(1, 70105, '\P{^Is_Script_Extensions=	sharada}', "");
Expect(0, 70106, '\p{Is_Script_Extensions=	sharada}', "");
Expect(1, 70106, '\p{^Is_Script_Extensions=	sharada}', "");
Expect(1, 70106, '\P{Is_Script_Extensions=	sharada}', "");
Expect(0, 70106, '\P{^Is_Script_Extensions=	sharada}', "");
Error('\p{Is_Scx=/a/-Shrd}');
Error('\P{Is_Scx=/a/-Shrd}');
Expect(1, 70105, '\p{Is_Scx=shrd}', "");
Expect(0, 70105, '\p{^Is_Scx=shrd}', "");
Expect(0, 70105, '\P{Is_Scx=shrd}', "");
Expect(1, 70105, '\P{^Is_Scx=shrd}', "");
Expect(0, 70106, '\p{Is_Scx=shrd}', "");
Expect(1, 70106, '\p{^Is_Scx=shrd}', "");
Expect(1, 70106, '\P{Is_Scx=shrd}', "");
Expect(0, 70106, '\P{^Is_Scx=shrd}', "");
Expect(1, 70105, '\p{Is_Scx= 	Shrd}', "");
Expect(0, 70105, '\p{^Is_Scx= 	Shrd}', "");
Expect(0, 70105, '\P{Is_Scx= 	Shrd}', "");
Expect(1, 70105, '\P{^Is_Scx= 	Shrd}', "");
Expect(0, 70106, '\p{Is_Scx= 	Shrd}', "");
Expect(1, 70106, '\p{^Is_Scx= 	Shrd}', "");
Expect(1, 70106, '\P{Is_Scx= 	Shrd}', "");
Expect(0, 70106, '\P{^Is_Scx= 	Shrd}', "");
Error('\p{Script_Extensions= :=Buhid}');
Error('\P{Script_Extensions= :=Buhid}');
Expect(1, 5971, '\p{Script_Extensions=buhid}', "");
Expect(0, 5971, '\p{^Script_Extensions=buhid}', "");
Expect(0, 5971, '\P{Script_Extensions=buhid}', "");
Expect(1, 5971, '\P{^Script_Extensions=buhid}', "");
Expect(0, 5972, '\p{Script_Extensions=buhid}', "");
Expect(1, 5972, '\p{^Script_Extensions=buhid}', "");
Expect(1, 5972, '\P{Script_Extensions=buhid}', "");
Expect(0, 5972, '\P{^Script_Extensions=buhid}', "");
Expect(1, 5971, '\p{Script_Extensions=-Buhid}', "");
Expect(0, 5971, '\p{^Script_Extensions=-Buhid}', "");
Expect(0, 5971, '\P{Script_Extensions=-Buhid}', "");
Expect(1, 5971, '\P{^Script_Extensions=-Buhid}', "");
Expect(0, 5972, '\p{Script_Extensions=-Buhid}', "");
Expect(1, 5972, '\p{^Script_Extensions=-Buhid}', "");
Expect(1, 5972, '\P{Script_Extensions=-Buhid}', "");
Expect(0, 5972, '\P{^Script_Extensions=-Buhid}', "");
Error('\p{Scx=-	BUHD/a/}');
Error('\P{Scx=-	BUHD/a/}');
Expect(1, 5971, '\p{Scx=buhd}', "");
Expect(0, 5971, '\p{^Scx=buhd}', "");
Expect(0, 5971, '\P{Scx=buhd}', "");
Expect(1, 5971, '\P{^Scx=buhd}', "");
Expect(0, 5972, '\p{Scx=buhd}', "");
Expect(1, 5972, '\p{^Scx=buhd}', "");
Expect(1, 5972, '\P{Scx=buhd}', "");
Expect(0, 5972, '\P{^Scx=buhd}', "");
Expect(1, 5971, '\p{Scx=	-buhd}', "");
Expect(0, 5971, '\p{^Scx=	-buhd}', "");
Expect(0, 5971, '\P{Scx=	-buhd}', "");
Expect(1, 5971, '\P{^Scx=	-buhd}', "");
Expect(0, 5972, '\p{Scx=	-buhd}', "");
Expect(1, 5972, '\p{^Scx=	-buhd}', "");
Expect(1, 5972, '\P{Scx=	-buhd}', "");
Expect(0, 5972, '\P{^Scx=	-buhd}', "");
Error('\p{Is_Script_Extensions=-:=Buhid}');
Error('\P{Is_Script_Extensions=-:=Buhid}');
Expect(1, 5971, '\p{Is_Script_Extensions=buhid}', "");
Expect(0, 5971, '\p{^Is_Script_Extensions=buhid}', "");
Expect(0, 5971, '\P{Is_Script_Extensions=buhid}', "");
Expect(1, 5971, '\P{^Is_Script_Extensions=buhid}', "");
Expect(0, 5972, '\p{Is_Script_Extensions=buhid}', "");
Expect(1, 5972, '\p{^Is_Script_Extensions=buhid}', "");
Expect(1, 5972, '\P{Is_Script_Extensions=buhid}', "");
Expect(0, 5972, '\P{^Is_Script_Extensions=buhid}', "");
Expect(1, 5971, '\p{Is_Script_Extensions=		BUHID}', "");
Expect(0, 5971, '\p{^Is_Script_Extensions=		BUHID}', "");
Expect(0, 5971, '\P{Is_Script_Extensions=		BUHID}', "");
Expect(1, 5971, '\P{^Is_Script_Extensions=		BUHID}', "");
Expect(0, 5972, '\p{Is_Script_Extensions=		BUHID}', "");
Expect(1, 5972, '\p{^Is_Script_Extensions=		BUHID}', "");
Expect(1, 5972, '\P{Is_Script_Extensions=		BUHID}', "");
Expect(0, 5972, '\P{^Is_Script_Extensions=		BUHID}', "");
Error('\p{Is_Scx=:= BUHD}');
Error('\P{Is_Scx=:= BUHD}');
Expect(1, 5971, '\p{Is_Scx=buhd}', "");
Expect(0, 5971, '\p{^Is_Scx=buhd}', "");
Expect(0, 5971, '\P{Is_Scx=buhd}', "");
Expect(1, 5971, '\P{^Is_Scx=buhd}', "");
Expect(0, 5972, '\p{Is_Scx=buhd}', "");
Expect(1, 5972, '\p{^Is_Scx=buhd}', "");
Expect(1, 5972, '\P{Is_Scx=buhd}', "");
Expect(0, 5972, '\P{^Is_Scx=buhd}', "");
Expect(1, 5971, '\p{Is_Scx=- buhd}', "");
Expect(0, 5971, '\p{^Is_Scx=- buhd}', "");
Expect(0, 5971, '\P{Is_Scx=- buhd}', "");
Expect(1, 5971, '\P{^Is_Scx=- buhd}', "");
Expect(0, 5972, '\p{Is_Scx=- buhd}', "");
Expect(1, 5972, '\p{^Is_Scx=- buhd}', "");
Expect(1, 5972, '\P{Is_Scx=- buhd}', "");
Expect(0, 5972, '\P{^Is_Scx=- buhd}', "");
Error('\p{Script_Extensions=/a/-Limbu}');
Error('\P{Script_Extensions=/a/-Limbu}');
Expect(1, 6479, '\p{Script_Extensions=limbu}', "");
Expect(0, 6479, '\p{^Script_Extensions=limbu}', "");
Expect(0, 6479, '\P{Script_Extensions=limbu}', "");
Expect(1, 6479, '\P{^Script_Extensions=limbu}', "");
Expect(0, 6480, '\p{Script_Extensions=limbu}', "");
Expect(1, 6480, '\p{^Script_Extensions=limbu}', "");
Expect(1, 6480, '\P{Script_Extensions=limbu}', "");
Expect(0, 6480, '\P{^Script_Extensions=limbu}', "");
Expect(1, 6479, '\p{Script_Extensions=	_Limbu}', "");
Expect(0, 6479, '\p{^Script_Extensions=	_Limbu}', "");
Expect(0, 6479, '\P{Script_Extensions=	_Limbu}', "");
Expect(1, 6479, '\P{^Script_Extensions=	_Limbu}', "");
Expect(0, 6480, '\p{Script_Extensions=	_Limbu}', "");
Expect(1, 6480, '\p{^Script_Extensions=	_Limbu}', "");
Expect(1, 6480, '\P{Script_Extensions=	_Limbu}', "");
Expect(0, 6480, '\P{^Script_Extensions=	_Limbu}', "");
Error('\p{Scx:	/a/_	Limb}');
Error('\P{Scx:	/a/_	Limb}');
Expect(1, 6479, '\p{Scx=limb}', "");
Expect(0, 6479, '\p{^Scx=limb}', "");
Expect(0, 6479, '\P{Scx=limb}', "");
Expect(1, 6479, '\P{^Scx=limb}', "");
Expect(0, 6480, '\p{Scx=limb}', "");
Expect(1, 6480, '\p{^Scx=limb}', "");
Expect(1, 6480, '\P{Scx=limb}', "");
Expect(0, 6480, '\P{^Scx=limb}', "");
Expect(1, 6479, '\p{Scx=__Limb}', "");
Expect(0, 6479, '\p{^Scx=__Limb}', "");
Expect(0, 6479, '\P{Scx=__Limb}', "");
Expect(1, 6479, '\P{^Scx=__Limb}', "");
Expect(0, 6480, '\p{Scx=__Limb}', "");
Expect(1, 6480, '\p{^Scx=__Limb}', "");
Expect(1, 6480, '\P{Scx=__Limb}', "");
Expect(0, 6480, '\P{^Scx=__Limb}', "");
Error('\p{Is_Script_Extensions=:=-Limbu}');
Error('\P{Is_Script_Extensions=:=-Limbu}');
Expect(1, 6479, '\p{Is_Script_Extensions=limbu}', "");
Expect(0, 6479, '\p{^Is_Script_Extensions=limbu}', "");
Expect(0, 6479, '\P{Is_Script_Extensions=limbu}', "");
Expect(1, 6479, '\P{^Is_Script_Extensions=limbu}', "");
Expect(0, 6480, '\p{Is_Script_Extensions=limbu}', "");
Expect(1, 6480, '\p{^Is_Script_Extensions=limbu}', "");
Expect(1, 6480, '\P{Is_Script_Extensions=limbu}', "");
Expect(0, 6480, '\P{^Is_Script_Extensions=limbu}', "");
Expect(1, 6479, '\p{Is_Script_Extensions:    Limbu}', "");
Expect(0, 6479, '\p{^Is_Script_Extensions:    Limbu}', "");
Expect(0, 6479, '\P{Is_Script_Extensions:    Limbu}', "");
Expect(1, 6479, '\P{^Is_Script_Extensions:    Limbu}', "");
Expect(0, 6480, '\p{Is_Script_Extensions:    Limbu}', "");
Expect(1, 6480, '\p{^Is_Script_Extensions:    Limbu}', "");
Expect(1, 6480, '\P{Is_Script_Extensions:    Limbu}', "");
Expect(0, 6480, '\P{^Is_Script_Extensions:    Limbu}', "");
Error('\p{Is_Scx=:=- limb}');
Error('\P{Is_Scx=:=- limb}');
Expect(1, 6479, '\p{Is_Scx=limb}', "");
Expect(0, 6479, '\p{^Is_Scx=limb}', "");
Expect(0, 6479, '\P{Is_Scx=limb}', "");
Expect(1, 6479, '\P{^Is_Scx=limb}', "");
Expect(0, 6480, '\p{Is_Scx=limb}', "");
Expect(1, 6480, '\p{^Is_Scx=limb}', "");
Expect(1, 6480, '\P{Is_Scx=limb}', "");
Expect(0, 6480, '\P{^Is_Scx=limb}', "");
Expect(1, 6479, '\p{Is_Scx:-_LIMB}', "");
Expect(0, 6479, '\p{^Is_Scx:-_LIMB}', "");
Expect(0, 6479, '\P{Is_Scx:-_LIMB}', "");
Expect(1, 6479, '\P{^Is_Scx:-_LIMB}', "");
Expect(0, 6480, '\p{Is_Scx:-_LIMB}', "");
Expect(1, 6480, '\p{^Is_Scx:-_LIMB}', "");
Expect(1, 6480, '\P{Is_Scx:-_LIMB}', "");
Expect(0, 6480, '\P{^Is_Scx:-_LIMB}', "");
Error('\p{Script_Extensions=_/a/egyptian_hieroglyphs}');
Error('\P{Script_Extensions=_/a/egyptian_hieroglyphs}');
Expect(1, 78894, '\p{Script_Extensions=egyptianhieroglyphs}', "");
Expect(0, 78894, '\p{^Script_Extensions=egyptianhieroglyphs}', "");
Expect(0, 78894, '\P{Script_Extensions=egyptianhieroglyphs}', "");
Expect(1, 78894, '\P{^Script_Extensions=egyptianhieroglyphs}', "");
Expect(0, 78895, '\p{Script_Extensions=egyptianhieroglyphs}', "");
Expect(1, 78895, '\p{^Script_Extensions=egyptianhieroglyphs}', "");
Expect(1, 78895, '\P{Script_Extensions=egyptianhieroglyphs}', "");
Expect(0, 78895, '\P{^Script_Extensions=egyptianhieroglyphs}', "");
Expect(1, 78894, '\p{Script_Extensions=--Egyptian_HIEROGLYPHS}', "");
Expect(0, 78894, '\p{^Script_Extensions=--Egyptian_HIEROGLYPHS}', "");
Expect(0, 78894, '\P{Script_Extensions=--Egyptian_HIEROGLYPHS}', "");
Expect(1, 78894, '\P{^Script_Extensions=--Egyptian_HIEROGLYPHS}', "");
Expect(0, 78895, '\p{Script_Extensions=--Egyptian_HIEROGLYPHS}', "");
Expect(1, 78895, '\p{^Script_Extensions=--Egyptian_HIEROGLYPHS}', "");
Expect(1, 78895, '\P{Script_Extensions=--Egyptian_HIEROGLYPHS}', "");
Expect(0, 78895, '\P{^Script_Extensions=--Egyptian_HIEROGLYPHS}', "");
Error('\p{Scx= Egyp:=}');
Error('\P{Scx= Egyp:=}');
Expect(1, 78894, '\p{Scx=egyp}', "");
Expect(0, 78894, '\p{^Scx=egyp}', "");
Expect(0, 78894, '\P{Scx=egyp}', "");
Expect(1, 78894, '\P{^Scx=egyp}', "");
Expect(0, 78895, '\p{Scx=egyp}', "");
Expect(1, 78895, '\p{^Scx=egyp}', "");
Expect(1, 78895, '\P{Scx=egyp}', "");
Expect(0, 78895, '\P{^Scx=egyp}', "");
Expect(1, 78894, '\p{Scx:   --EGYP}', "");
Expect(0, 78894, '\p{^Scx:   --EGYP}', "");
Expect(0, 78894, '\P{Scx:   --EGYP}', "");
Expect(1, 78894, '\P{^Scx:   --EGYP}', "");
Expect(0, 78895, '\p{Scx:   --EGYP}', "");
Expect(1, 78895, '\p{^Scx:   --EGYP}', "");
Expect(1, 78895, '\P{Scx:   --EGYP}', "");
Expect(0, 78895, '\P{^Scx:   --EGYP}', "");
Error('\p{Is_Script_Extensions=  Egyptian_Hieroglyphs/a/}');
Error('\P{Is_Script_Extensions=  Egyptian_Hieroglyphs/a/}');
Expect(1, 78894, '\p{Is_Script_Extensions=egyptianhieroglyphs}', "");
Expect(0, 78894, '\p{^Is_Script_Extensions=egyptianhieroglyphs}', "");
Expect(0, 78894, '\P{Is_Script_Extensions=egyptianhieroglyphs}', "");
Expect(1, 78894, '\P{^Is_Script_Extensions=egyptianhieroglyphs}', "");
Expect(0, 78895, '\p{Is_Script_Extensions=egyptianhieroglyphs}', "");
Expect(1, 78895, '\p{^Is_Script_Extensions=egyptianhieroglyphs}', "");
Expect(1, 78895, '\P{Is_Script_Extensions=egyptianhieroglyphs}', "");
Expect(0, 78895, '\P{^Is_Script_Extensions=egyptianhieroglyphs}', "");
Expect(1, 78894, '\p{Is_Script_Extensions= egyptian_HIEROGLYPHS}', "");
Expect(0, 78894, '\p{^Is_Script_Extensions= egyptian_HIEROGLYPHS}', "");
Expect(0, 78894, '\P{Is_Script_Extensions= egyptian_HIEROGLYPHS}', "");
Expect(1, 78894, '\P{^Is_Script_Extensions= egyptian_HIEROGLYPHS}', "");
Expect(0, 78895, '\p{Is_Script_Extensions= egyptian_HIEROGLYPHS}', "");
Expect(1, 78895, '\p{^Is_Script_Extensions= egyptian_HIEROGLYPHS}', "");
Expect(1, 78895, '\P{Is_Script_Extensions= egyptian_HIEROGLYPHS}', "");
Expect(0, 78895, '\P{^Is_Script_Extensions= egyptian_HIEROGLYPHS}', "");
Error('\p{Is_Scx=:=	 Egyp}');
Error('\P{Is_Scx=:=	 Egyp}');
Expect(1, 78894, '\p{Is_Scx=egyp}', "");
Expect(0, 78894, '\p{^Is_Scx=egyp}', "");
Expect(0, 78894, '\P{Is_Scx=egyp}', "");
Expect(1, 78894, '\P{^Is_Scx=egyp}', "");
Expect(0, 78895, '\p{Is_Scx=egyp}', "");
Expect(1, 78895, '\p{^Is_Scx=egyp}', "");
Expect(1, 78895, '\P{Is_Scx=egyp}', "");
Expect(0, 78895, '\P{^Is_Scx=egyp}', "");
Expect(1, 78894, '\p{Is_Scx=	-egyp}', "");
Expect(0, 78894, '\p{^Is_Scx=	-egyp}', "");
Expect(0, 78894, '\P{Is_Scx=	-egyp}', "");
Expect(1, 78894, '\P{^Is_Scx=	-egyp}', "");
Expect(0, 78895, '\p{Is_Scx=	-egyp}', "");
Expect(1, 78895, '\p{^Is_Scx=	-egyp}', "");
Expect(1, 78895, '\P{Is_Scx=	-egyp}', "");
Expect(0, 78895, '\P{^Is_Scx=	-egyp}', "");
Error('\p{Script_Extensions=:=	-javanese}');
Error('\P{Script_Extensions=:=	-javanese}');
Expect(1, 43487, '\p{Script_Extensions=javanese}', "");
Expect(0, 43487, '\p{^Script_Extensions=javanese}', "");
Expect(0, 43487, '\P{Script_Extensions=javanese}', "");
Expect(1, 43487, '\P{^Script_Extensions=javanese}', "");
Expect(0, 43488, '\p{Script_Extensions=javanese}', "");
Expect(1, 43488, '\p{^Script_Extensions=javanese}', "");
Expect(1, 43488, '\P{Script_Extensions=javanese}', "");
Expect(0, 43488, '\P{^Script_Extensions=javanese}', "");
Expect(1, 43487, '\p{Script_Extensions=JAVANESE}', "");
Expect(0, 43487, '\p{^Script_Extensions=JAVANESE}', "");
Expect(0, 43487, '\P{Script_Extensions=JAVANESE}', "");
Expect(1, 43487, '\P{^Script_Extensions=JAVANESE}', "");
Expect(0, 43488, '\p{Script_Extensions=JAVANESE}', "");
Expect(1, 43488, '\p{^Script_Extensions=JAVANESE}', "");
Expect(1, 43488, '\P{Script_Extensions=JAVANESE}', "");
Expect(0, 43488, '\P{^Script_Extensions=JAVANESE}', "");
Error('\p{Scx=:= JAVA}');
Error('\P{Scx=:= JAVA}');
Expect(1, 43487, '\p{Scx=java}', "");
Expect(0, 43487, '\p{^Scx=java}', "");
Expect(0, 43487, '\P{Scx=java}', "");
Expect(1, 43487, '\P{^Scx=java}', "");
Expect(0, 43488, '\p{Scx=java}', "");
Expect(1, 43488, '\p{^Scx=java}', "");
Expect(1, 43488, '\P{Scx=java}', "");
Expect(0, 43488, '\P{^Scx=java}', "");
Expect(1, 43487, '\p{Scx:	 	JAVA}', "");
Expect(0, 43487, '\p{^Scx:	 	JAVA}', "");
Expect(0, 43487, '\P{Scx:	 	JAVA}', "");
Expect(1, 43487, '\P{^Scx:	 	JAVA}', "");
Expect(0, 43488, '\p{Scx:	 	JAVA}', "");
Expect(1, 43488, '\p{^Scx:	 	JAVA}', "");
Expect(1, 43488, '\P{Scx:	 	JAVA}', "");
Expect(0, 43488, '\P{^Scx:	 	JAVA}', "");
Error('\p{Is_Script_Extensions=	:=Javanese}');
Error('\P{Is_Script_Extensions=	:=Javanese}');
Expect(1, 43487, '\p{Is_Script_Extensions=javanese}', "");
Expect(0, 43487, '\p{^Is_Script_Extensions=javanese}', "");
Expect(0, 43487, '\P{Is_Script_Extensions=javanese}', "");
Expect(1, 43487, '\P{^Is_Script_Extensions=javanese}', "");
Expect(0, 43488, '\p{Is_Script_Extensions=javanese}', "");
Expect(1, 43488, '\p{^Is_Script_Extensions=javanese}', "");
Expect(1, 43488, '\P{Is_Script_Extensions=javanese}', "");
Expect(0, 43488, '\P{^Is_Script_Extensions=javanese}', "");
Expect(1, 43487, '\p{Is_Script_Extensions=--Javanese}', "");
Expect(0, 43487, '\p{^Is_Script_Extensions=--Javanese}', "");
Expect(0, 43487, '\P{Is_Script_Extensions=--Javanese}', "");
Expect(1, 43487, '\P{^Is_Script_Extensions=--Javanese}', "");
Expect(0, 43488, '\p{Is_Script_Extensions=--Javanese}', "");
Expect(1, 43488, '\p{^Is_Script_Extensions=--Javanese}', "");
Expect(1, 43488, '\P{Is_Script_Extensions=--Javanese}', "");
Expect(0, 43488, '\P{^Is_Script_Extensions=--Javanese}', "");
Error('\p{Is_Scx=-_Java:=}');
Error('\P{Is_Scx=-_Java:=}');
Expect(1, 43487, '\p{Is_Scx=java}', "");
Expect(0, 43487, '\p{^Is_Scx=java}', "");
Expect(0, 43487, '\P{Is_Scx=java}', "");
Expect(1, 43487, '\P{^Is_Scx=java}', "");
Expect(0, 43488, '\p{Is_Scx=java}', "");
Expect(1, 43488, '\p{^Is_Scx=java}', "");
Expect(1, 43488, '\P{Is_Scx=java}', "");
Expect(0, 43488, '\P{^Is_Scx=java}', "");
Expect(1, 43487, '\p{Is_Scx:  JAVA}', "");
Expect(0, 43487, '\p{^Is_Scx:  JAVA}', "");
Expect(0, 43487, '\P{Is_Scx:  JAVA}', "");
Expect(1, 43487, '\P{^Is_Scx:  JAVA}', "");
Expect(0, 43488, '\p{Is_Scx:  JAVA}', "");
Expect(1, 43488, '\p{^Is_Scx:  JAVA}', "");
Expect(1, 43488, '\P{Is_Scx:  JAVA}', "");
Expect(0, 43488, '\P{^Is_Scx:  JAVA}', "");
Error('\p{Script_Extensions=:=_-Chakma}');
Error('\P{Script_Extensions=:=_-Chakma}');
Expect(1, 69955, '\p{Script_Extensions=chakma}', "");
Expect(0, 69955, '\p{^Script_Extensions=chakma}', "");
Expect(0, 69955, '\P{Script_Extensions=chakma}', "");
Expect(1, 69955, '\P{^Script_Extensions=chakma}', "");
Expect(0, 69956, '\p{Script_Extensions=chakma}', "");
Expect(1, 69956, '\p{^Script_Extensions=chakma}', "");
Expect(1, 69956, '\P{Script_Extensions=chakma}', "");
Expect(0, 69956, '\P{^Script_Extensions=chakma}', "");
Expect(1, 69955, '\p{Script_Extensions=-	CHAKMA}', "");
Expect(0, 69955, '\p{^Script_Extensions=-	CHAKMA}', "");
Expect(0, 69955, '\P{Script_Extensions=-	CHAKMA}', "");
Expect(1, 69955, '\P{^Script_Extensions=-	CHAKMA}', "");
Expect(0, 69956, '\p{Script_Extensions=-	CHAKMA}', "");
Expect(1, 69956, '\p{^Script_Extensions=-	CHAKMA}', "");
Expect(1, 69956, '\P{Script_Extensions=-	CHAKMA}', "");
Expect(0, 69956, '\P{^Script_Extensions=-	CHAKMA}', "");
Error('\p{Scx=Cakm:=}');
Error('\P{Scx=Cakm:=}');
Expect(1, 69955, '\p{Scx=cakm}', "");
Expect(0, 69955, '\p{^Scx=cakm}', "");
Expect(0, 69955, '\P{Scx=cakm}', "");
Expect(1, 69955, '\P{^Scx=cakm}', "");
Expect(0, 69956, '\p{Scx=cakm}', "");
Expect(1, 69956, '\p{^Scx=cakm}', "");
Expect(1, 69956, '\P{Scx=cakm}', "");
Expect(0, 69956, '\P{^Scx=cakm}', "");
Expect(1, 69955, '\p{Scx=	cakm}', "");
Expect(0, 69955, '\p{^Scx=	cakm}', "");
Expect(0, 69955, '\P{Scx=	cakm}', "");
Expect(1, 69955, '\P{^Scx=	cakm}', "");
Expect(0, 69956, '\p{Scx=	cakm}', "");
Expect(1, 69956, '\p{^Scx=	cakm}', "");
Expect(1, 69956, '\P{Scx=	cakm}', "");
Expect(0, 69956, '\P{^Scx=	cakm}', "");
Error('\p{Is_Script_Extensions=:= _Chakma}');
Error('\P{Is_Script_Extensions=:= _Chakma}');
Expect(1, 69955, '\p{Is_Script_Extensions=chakma}', "");
Expect(0, 69955, '\p{^Is_Script_Extensions=chakma}', "");
Expect(0, 69955, '\P{Is_Script_Extensions=chakma}', "");
Expect(1, 69955, '\P{^Is_Script_Extensions=chakma}', "");
Expect(0, 69956, '\p{Is_Script_Extensions=chakma}', "");
Expect(1, 69956, '\p{^Is_Script_Extensions=chakma}', "");
Expect(1, 69956, '\P{Is_Script_Extensions=chakma}', "");
Expect(0, 69956, '\P{^Is_Script_Extensions=chakma}', "");
Expect(1, 69955, '\p{Is_Script_Extensions=	_CHAKMA}', "");
Expect(0, 69955, '\p{^Is_Script_Extensions=	_CHAKMA}', "");
Expect(0, 69955, '\P{Is_Script_Extensions=	_CHAKMA}', "");
Expect(1, 69955, '\P{^Is_Script_Extensions=	_CHAKMA}', "");
Expect(0, 69956, '\p{Is_Script_Extensions=	_CHAKMA}', "");
Expect(1, 69956, '\p{^Is_Script_Extensions=	_CHAKMA}', "");
Expect(1, 69956, '\P{Is_Script_Extensions=	_CHAKMA}', "");
Expect(0, 69956, '\P{^Is_Script_Extensions=	_CHAKMA}', "");
Error('\p{Is_Scx=-:=CAKM}');
Error('\P{Is_Scx=-:=CAKM}');
Expect(1, 69955, '\p{Is_Scx=cakm}', "");
Expect(0, 69955, '\p{^Is_Scx=cakm}', "");
Expect(0, 69955, '\P{Is_Scx=cakm}', "");
Expect(1, 69955, '\P{^Is_Scx=cakm}', "");
Expect(0, 69956, '\p{Is_Scx=cakm}', "");
Expect(1, 69956, '\p{^Is_Scx=cakm}', "");
Expect(1, 69956, '\P{Is_Scx=cakm}', "");
Expect(0, 69956, '\P{^Is_Scx=cakm}', "");
Expect(1, 69955, '\p{Is_Scx=	-cakm}', "");
Expect(0, 69955, '\p{^Is_Scx=	-cakm}', "");
Expect(0, 69955, '\P{Is_Scx=	-cakm}', "");
Expect(1, 69955, '\P{^Is_Scx=	-cakm}', "");
Expect(0, 69956, '\p{Is_Scx=	-cakm}', "");
Expect(1, 69956, '\p{^Is_Scx=	-cakm}', "");
Expect(1, 69956, '\P{Is_Scx=	-cakm}', "");
Expect(0, 69956, '\P{^Is_Scx=	-cakm}', "");
Error('\p{Script_Extensions=/a/_ Common}');
Error('\P{Script_Extensions=/a/_ Common}');
Expect(1, 917631, '\p{Script_Extensions=common}', "");
Expect(0, 917631, '\p{^Script_Extensions=common}', "");
Expect(0, 917631, '\P{Script_Extensions=common}', "");
Expect(1, 917631, '\P{^Script_Extensions=common}', "");
Expect(0, 917632, '\p{Script_Extensions=common}', "");
Expect(1, 917632, '\p{^Script_Extensions=common}', "");
Expect(1, 917632, '\P{Script_Extensions=common}', "");
Expect(0, 917632, '\P{^Script_Extensions=common}', "");
Error('\p{Scx=/a/_ zyyy}');
Error('\P{Scx=/a/_ zyyy}');
Expect(1, 917631, '\p{Scx=zyyy}', "");
Expect(0, 917631, '\p{^Scx=zyyy}', "");
Expect(0, 917631, '\P{Scx=zyyy}', "");
Expect(1, 917631, '\P{^Scx=zyyy}', "");
Expect(0, 917632, '\p{Scx=zyyy}', "");
Expect(1, 917632, '\p{^Scx=zyyy}', "");
Expect(1, 917632, '\P{Scx=zyyy}', "");
Expect(0, 917632, '\P{^Scx=zyyy}', "");
Expect(1, 917631, '\p{Scx= zyyy}', "");
Expect(0, 917631, '\p{^Scx= zyyy}', "");
Expect(0, 917631, '\P{Scx= zyyy}', "");
Expect(1, 917631, '\P{^Scx= zyyy}', "");
Expect(0, 917632, '\p{Scx= zyyy}', "");
Expect(1, 917632, '\p{^Scx= zyyy}', "");
Expect(1, 917632, '\P{Scx= zyyy}', "");
Expect(0, 917632, '\P{^Scx= zyyy}', "");
Error('\p{Is_Script_Extensions=/a/_ Common}');
Error('\P{Is_Script_Extensions=/a/_ Common}');
Expect(1, 917631, '\p{Is_Script_Extensions=common}', "");
Expect(0, 917631, '\p{^Is_Script_Extensions=common}', "");
Expect(0, 917631, '\P{Is_Script_Extensions=common}', "");
Expect(1, 917631, '\P{^Is_Script_Extensions=common}', "");
Expect(0, 917632, '\p{Is_Script_Extensions=common}', "");
Expect(1, 917632, '\p{^Is_Script_Extensions=common}', "");
Expect(1, 917632, '\P{Is_Script_Extensions=common}', "");
Expect(0, 917632, '\P{^Is_Script_Extensions=common}', "");
Expect(1, 917631, '\p{Is_Script_Extensions= _common}', "");
Expect(0, 917631, '\p{^Is_Script_Extensions= _common}', "");
Expect(0, 917631, '\P{Is_Script_Extensions= _common}', "");
Expect(1, 917631, '\P{^Is_Script_Extensions= _common}', "");
Expect(0, 917632, '\p{Is_Script_Extensions= _common}', "");
Expect(1, 917632, '\p{^Is_Script_Extensions= _common}', "");
Expect(1, 917632, '\P{Is_Script_Extensions= _common}', "");
Expect(0, 917632, '\P{^Is_Script_Extensions= _common}', "");
Error('\p{Is_Scx=:=Zyyy}');
Error('\P{Is_Scx=:=Zyyy}');
Expect(1, 917631, '\p{Is_Scx=zyyy}', "");
Expect(0, 917631, '\p{^Is_Scx=zyyy}', "");
Expect(0, 917631, '\P{Is_Scx=zyyy}', "");
Expect(1, 917631, '\P{^Is_Scx=zyyy}', "");
Expect(0, 917632, '\p{Is_Scx=zyyy}', "");
Expect(1, 917632, '\p{^Is_Scx=zyyy}', "");
Expect(1, 917632, '\P{Is_Scx=zyyy}', "");
Expect(0, 917632, '\P{^Is_Scx=zyyy}', "");
Expect(1, 917631, '\p{Is_Scx=_Zyyy}', "");
Expect(0, 917631, '\p{^Is_Scx=_Zyyy}', "");
Expect(0, 917631, '\P{Is_Scx=_Zyyy}', "");
Expect(1, 917631, '\P{^Is_Scx=_Zyyy}', "");
Expect(0, 917632, '\p{Is_Scx=_Zyyy}', "");
Expect(1, 917632, '\p{^Is_Scx=_Zyyy}', "");
Expect(1, 917632, '\P{Is_Scx=_Zyyy}', "");
Expect(0, 917632, '\P{^Is_Scx=_Zyyy}', "");
Error('\p{Script_Extensions:	/a/Latin}');
Error('\P{Script_Extensions:	/a/Latin}');
Expect(1, 65370, '\p{Script_Extensions=latin}', "");
Expect(0, 65370, '\p{^Script_Extensions=latin}', "");
Expect(0, 65370, '\P{Script_Extensions=latin}', "");
Expect(1, 65370, '\P{^Script_Extensions=latin}', "");
Expect(0, 65371, '\p{Script_Extensions=latin}', "");
Expect(1, 65371, '\p{^Script_Extensions=latin}', "");
Expect(1, 65371, '\P{Script_Extensions=latin}', "");
Expect(0, 65371, '\P{^Script_Extensions=latin}', "");
Expect(1, 65370, '\p{Script_Extensions=- Latin}', "");
Expect(0, 65370, '\p{^Script_Extensions=- Latin}', "");
Expect(0, 65370, '\P{Script_Extensions=- Latin}', "");
Expect(1, 65370, '\P{^Script_Extensions=- Latin}', "");
Expect(0, 65371, '\p{Script_Extensions=- Latin}', "");
Expect(1, 65371, '\p{^Script_Extensions=- Latin}', "");
Expect(1, 65371, '\P{Script_Extensions=- Latin}', "");
Expect(0, 65371, '\P{^Script_Extensions=- Latin}', "");
Error('\p{Scx=_:=Latn}');
Error('\P{Scx=_:=Latn}');
Expect(1, 65370, '\p{Scx:latn}', "");
Expect(0, 65370, '\p{^Scx:latn}', "");
Expect(0, 65370, '\P{Scx:latn}', "");
Expect(1, 65370, '\P{^Scx:latn}', "");
Expect(0, 65371, '\p{Scx:latn}', "");
Expect(1, 65371, '\p{^Scx:latn}', "");
Expect(1, 65371, '\P{Scx:latn}', "");
Expect(0, 65371, '\P{^Scx:latn}', "");
Expect(1, 65370, '\p{Scx=_-LATN}', "");
Expect(0, 65370, '\p{^Scx=_-LATN}', "");
Expect(0, 65370, '\P{Scx=_-LATN}', "");
Expect(1, 65370, '\P{^Scx=_-LATN}', "");
Expect(0, 65371, '\p{Scx=_-LATN}', "");
Expect(1, 65371, '\p{^Scx=_-LATN}', "");
Expect(1, 65371, '\P{Scx=_-LATN}', "");
Expect(0, 65371, '\P{^Scx=_-LATN}', "");
Error('\p{Is_Script_Extensions=_ Latin:=}');
Error('\P{Is_Script_Extensions=_ Latin:=}');
Expect(1, 65370, '\p{Is_Script_Extensions=latin}', "");
Expect(0, 65370, '\p{^Is_Script_Extensions=latin}', "");
Expect(0, 65370, '\P{Is_Script_Extensions=latin}', "");
Expect(1, 65370, '\P{^Is_Script_Extensions=latin}', "");
Expect(0, 65371, '\p{Is_Script_Extensions=latin}', "");
Expect(1, 65371, '\p{^Is_Script_Extensions=latin}', "");
Expect(1, 65371, '\P{Is_Script_Extensions=latin}', "");
Expect(0, 65371, '\P{^Is_Script_Extensions=latin}', "");
Expect(1, 65370, '\p{Is_Script_Extensions=__Latin}', "");
Expect(0, 65370, '\p{^Is_Script_Extensions=__Latin}', "");
Expect(0, 65370, '\P{Is_Script_Extensions=__Latin}', "");
Expect(1, 65370, '\P{^Is_Script_Extensions=__Latin}', "");
Expect(0, 65371, '\p{Is_Script_Extensions=__Latin}', "");
Expect(1, 65371, '\p{^Is_Script_Extensions=__Latin}', "");
Expect(1, 65371, '\P{Is_Script_Extensions=__Latin}', "");
Expect(0, 65371, '\P{^Is_Script_Extensions=__Latin}', "");
Error('\p{Is_Scx=_-Latn/a/}');
Error('\P{Is_Scx=_-Latn/a/}');
Expect(1, 65370, '\p{Is_Scx=latn}', "");
Expect(0, 65370, '\p{^Is_Scx=latn}', "");
Expect(0, 65370, '\P{Is_Scx=latn}', "");
Expect(1, 65370, '\P{^Is_Scx=latn}', "");
Expect(0, 65371, '\p{Is_Scx=latn}', "");
Expect(1, 65371, '\p{^Is_Scx=latn}', "");
Expect(1, 65371, '\P{Is_Scx=latn}', "");
Expect(0, 65371, '\P{^Is_Scx=latn}', "");
Expect(1, 65370, '\p{Is_Scx=Latn}', "");
Expect(0, 65370, '\p{^Is_Scx=Latn}', "");
Expect(0, 65370, '\P{Is_Scx=Latn}', "");
Expect(1, 65370, '\P{^Is_Scx=Latn}', "");
Expect(0, 65371, '\p{Is_Scx=Latn}', "");
Expect(1, 65371, '\p{^Is_Scx=Latn}', "");
Expect(1, 65371, '\P{Is_Scx=Latn}', "");
Expect(0, 65371, '\P{^Is_Scx=Latn}', "");
Error('\p{Script_Extensions=/a/Ethiopic}');
Error('\P{Script_Extensions=/a/Ethiopic}');
Expect(1, 43822, '\p{Script_Extensions=ethiopic}', "");
Expect(0, 43822, '\p{^Script_Extensions=ethiopic}', "");
Expect(0, 43822, '\P{Script_Extensions=ethiopic}', "");
Expect(1, 43822, '\P{^Script_Extensions=ethiopic}', "");
Expect(0, 43823, '\p{Script_Extensions=ethiopic}', "");
Expect(1, 43823, '\p{^Script_Extensions=ethiopic}', "");
Expect(1, 43823, '\P{Script_Extensions=ethiopic}', "");
Expect(0, 43823, '\P{^Script_Extensions=ethiopic}', "");
Expect(1, 43822, '\p{Script_Extensions=	 ethiopic}', "");
Expect(0, 43822, '\p{^Script_Extensions=	 ethiopic}', "");
Expect(0, 43822, '\P{Script_Extensions=	 ethiopic}', "");
Expect(1, 43822, '\P{^Script_Extensions=	 ethiopic}', "");
Expect(0, 43823, '\p{Script_Extensions=	 ethiopic}', "");
Expect(1, 43823, '\p{^Script_Extensions=	 ethiopic}', "");
Expect(1, 43823, '\P{Script_Extensions=	 ethiopic}', "");
Expect(0, 43823, '\P{^Script_Extensions=	 ethiopic}', "");
Error('\p{Scx:   __ethi/a/}');
Error('\P{Scx:   __ethi/a/}');
Expect(1, 43822, '\p{Scx=ethi}', "");
Expect(0, 43822, '\p{^Scx=ethi}', "");
Expect(0, 43822, '\P{Scx=ethi}', "");
Expect(1, 43822, '\P{^Scx=ethi}', "");
Expect(0, 43823, '\p{Scx=ethi}', "");
Expect(1, 43823, '\p{^Scx=ethi}', "");
Expect(1, 43823, '\P{Scx=ethi}', "");
Expect(0, 43823, '\P{^Scx=ethi}', "");
Expect(1, 43822, '\p{Scx=-	Ethi}', "");
Expect(0, 43822, '\p{^Scx=-	Ethi}', "");
Expect(0, 43822, '\P{Scx=-	Ethi}', "");
Expect(1, 43822, '\P{^Scx=-	Ethi}', "");
Expect(0, 43823, '\p{Scx=-	Ethi}', "");
Expect(1, 43823, '\p{^Scx=-	Ethi}', "");
Expect(1, 43823, '\P{Scx=-	Ethi}', "");
Expect(0, 43823, '\P{^Scx=-	Ethi}', "");
Error('\p{Is_Script_Extensions=:=_ Ethiopic}');
Error('\P{Is_Script_Extensions=:=_ Ethiopic}');
Expect(1, 43822, '\p{Is_Script_Extensions=ethiopic}', "");
Expect(0, 43822, '\p{^Is_Script_Extensions=ethiopic}', "");
Expect(0, 43822, '\P{Is_Script_Extensions=ethiopic}', "");
Expect(1, 43822, '\P{^Is_Script_Extensions=ethiopic}', "");
Expect(0, 43823, '\p{Is_Script_Extensions=ethiopic}', "");
Expect(1, 43823, '\p{^Is_Script_Extensions=ethiopic}', "");
Expect(1, 43823, '\P{Is_Script_Extensions=ethiopic}', "");
Expect(0, 43823, '\P{^Is_Script_Extensions=ethiopic}', "");
Expect(1, 43822, '\p{Is_Script_Extensions=		Ethiopic}', "");
Expect(0, 43822, '\p{^Is_Script_Extensions=		Ethiopic}', "");
Expect(0, 43822, '\P{Is_Script_Extensions=		Ethiopic}', "");
Expect(1, 43822, '\P{^Is_Script_Extensions=		Ethiopic}', "");
Expect(0, 43823, '\p{Is_Script_Extensions=		Ethiopic}', "");
Expect(1, 43823, '\p{^Is_Script_Extensions=		Ethiopic}', "");
Expect(1, 43823, '\P{Is_Script_Extensions=		Ethiopic}', "");
Expect(0, 43823, '\P{^Is_Script_Extensions=		Ethiopic}', "");
Error('\p{Is_Scx=_	ethi:=}');
Error('\P{Is_Scx=_	ethi:=}');
Expect(1, 43822, '\p{Is_Scx=ethi}', "");
Expect(0, 43822, '\p{^Is_Scx=ethi}', "");
Expect(0, 43822, '\P{Is_Scx=ethi}', "");
Expect(1, 43822, '\P{^Is_Scx=ethi}', "");
Expect(0, 43823, '\p{Is_Scx=ethi}', "");
Expect(1, 43823, '\p{^Is_Scx=ethi}', "");
Expect(1, 43823, '\P{Is_Scx=ethi}', "");
Expect(0, 43823, '\P{^Is_Scx=ethi}', "");
Expect(1, 43822, '\p{Is_Scx=_-ethi}', "");
Expect(0, 43822, '\p{^Is_Scx=_-ethi}', "");
Expect(0, 43822, '\P{Is_Scx=_-ethi}', "");
Expect(1, 43822, '\P{^Is_Scx=_-ethi}', "");
Expect(0, 43823, '\p{Is_Scx=_-ethi}', "");
Expect(1, 43823, '\p{^Is_Scx=_-ethi}', "");
Expect(1, 43823, '\P{Is_Scx=_-ethi}', "");
Expect(0, 43823, '\P{^Is_Scx=_-ethi}', "");
Error('\p{Script_Extensions=Buginese/a/}');
Error('\P{Script_Extensions=Buginese/a/}');
Expect(1, 6687, '\p{Script_Extensions:buginese}', "");
Expect(0, 6687, '\p{^Script_Extensions:buginese}', "");
Expect(0, 6687, '\P{Script_Extensions:buginese}', "");
Expect(1, 6687, '\P{^Script_Extensions:buginese}', "");
Expect(0, 6688, '\p{Script_Extensions:buginese}', "");
Expect(1, 6688, '\p{^Script_Extensions:buginese}', "");
Expect(1, 6688, '\P{Script_Extensions:buginese}', "");
Expect(0, 6688, '\P{^Script_Extensions:buginese}', "");
Expect(1, 6687, '\p{Script_Extensions= -BUGINESE}', "");
Expect(0, 6687, '\p{^Script_Extensions= -BUGINESE}', "");
Expect(0, 6687, '\P{Script_Extensions= -BUGINESE}', "");
Expect(1, 6687, '\P{^Script_Extensions= -BUGINESE}', "");
Expect(0, 6688, '\p{Script_Extensions= -BUGINESE}', "");
Expect(1, 6688, '\p{^Script_Extensions= -BUGINESE}', "");
Expect(1, 6688, '\P{Script_Extensions= -BUGINESE}', "");
Expect(0, 6688, '\P{^Script_Extensions= -BUGINESE}', "");
Error('\p{Scx=/a/	Bugi}');
Error('\P{Scx=/a/	Bugi}');
Expect(1, 6687, '\p{Scx=bugi}', "");
Expect(0, 6687, '\p{^Scx=bugi}', "");
Expect(0, 6687, '\P{Scx=bugi}', "");
Expect(1, 6687, '\P{^Scx=bugi}', "");
Expect(0, 6688, '\p{Scx=bugi}', "");
Expect(1, 6688, '\p{^Scx=bugi}', "");
Expect(1, 6688, '\P{Scx=bugi}', "");
Expect(0, 6688, '\P{^Scx=bugi}', "");
Expect(1, 6687, '\p{Scx=-	Bugi}', "");
Expect(0, 6687, '\p{^Scx=-	Bugi}', "");
Expect(0, 6687, '\P{Scx=-	Bugi}', "");
Expect(1, 6687, '\P{^Scx=-	Bugi}', "");
Expect(0, 6688, '\p{Scx=-	Bugi}', "");
Expect(1, 6688, '\p{^Scx=-	Bugi}', "");
Expect(1, 6688, '\P{Scx=-	Bugi}', "");
Expect(0, 6688, '\P{^Scx=-	Bugi}', "");
Error('\p{Is_Script_Extensions= Buginese/a/}');
Error('\P{Is_Script_Extensions= Buginese/a/}');
Expect(1, 6687, '\p{Is_Script_Extensions=buginese}', "");
Expect(0, 6687, '\p{^Is_Script_Extensions=buginese}', "");
Expect(0, 6687, '\P{Is_Script_Extensions=buginese}', "");
Expect(1, 6687, '\P{^Is_Script_Extensions=buginese}', "");
Expect(0, 6688, '\p{Is_Script_Extensions=buginese}', "");
Expect(1, 6688, '\p{^Is_Script_Extensions=buginese}', "");
Expect(1, 6688, '\P{Is_Script_Extensions=buginese}', "");
Expect(0, 6688, '\P{^Is_Script_Extensions=buginese}', "");
Expect(1, 6687, '\p{Is_Script_Extensions:Buginese}', "");
Expect(0, 6687, '\p{^Is_Script_Extensions:Buginese}', "");
Expect(0, 6687, '\P{Is_Script_Extensions:Buginese}', "");
Expect(1, 6687, '\P{^Is_Script_Extensions:Buginese}', "");
Expect(0, 6688, '\p{Is_Script_Extensions:Buginese}', "");
Expect(1, 6688, '\p{^Is_Script_Extensions:Buginese}', "");
Expect(1, 6688, '\P{Is_Script_Extensions:Buginese}', "");
Expect(0, 6688, '\P{^Is_Script_Extensions:Buginese}', "");
Error('\p{Is_Scx:   :=-_BUGI}');
Error('\P{Is_Scx:   :=-_BUGI}');
Expect(1, 6687, '\p{Is_Scx:	bugi}', "");
Expect(0, 6687, '\p{^Is_Scx:	bugi}', "");
Expect(0, 6687, '\P{Is_Scx:	bugi}', "");
Expect(1, 6687, '\P{^Is_Scx:	bugi}', "");
Expect(0, 6688, '\p{Is_Scx:	bugi}', "");
Expect(1, 6688, '\p{^Is_Scx:	bugi}', "");
Expect(1, 6688, '\P{Is_Scx:	bugi}', "");
Expect(0, 6688, '\P{^Is_Scx:	bugi}', "");
Expect(1, 6687, '\p{Is_Scx=	BUGI}', "");
Expect(0, 6687, '\p{^Is_Scx=	BUGI}', "");
Expect(0, 6687, '\P{Is_Scx=	BUGI}', "");
Expect(1, 6687, '\P{^Is_Scx=	BUGI}', "");
Expect(0, 6688, '\p{Is_Scx=	BUGI}', "");
Expect(1, 6688, '\p{^Is_Scx=	BUGI}', "");
Expect(1, 6688, '\P{Is_Scx=	BUGI}', "");
Expect(0, 6688, '\P{^Is_Scx=	BUGI}', "");
Error('\p{Script_Extensions: -:=Hebrew}');
Error('\P{Script_Extensions: -:=Hebrew}');
Expect(1, 64335, '\p{Script_Extensions=hebrew}', "");
Expect(0, 64335, '\p{^Script_Extensions=hebrew}', "");
Expect(0, 64335, '\P{Script_Extensions=hebrew}', "");
Expect(1, 64335, '\P{^Script_Extensions=hebrew}', "");
Expect(0, 64336, '\p{Script_Extensions=hebrew}', "");
Expect(1, 64336, '\p{^Script_Extensions=hebrew}', "");
Expect(1, 64336, '\P{Script_Extensions=hebrew}', "");
Expect(0, 64336, '\P{^Script_Extensions=hebrew}', "");
Expect(1, 64335, '\p{Script_Extensions= -Hebrew}', "");
Expect(0, 64335, '\p{^Script_Extensions= -Hebrew}', "");
Expect(0, 64335, '\P{Script_Extensions= -Hebrew}', "");
Expect(1, 64335, '\P{^Script_Extensions= -Hebrew}', "");
Expect(0, 64336, '\p{Script_Extensions= -Hebrew}', "");
Expect(1, 64336, '\p{^Script_Extensions= -Hebrew}', "");
Expect(1, 64336, '\P{Script_Extensions= -Hebrew}', "");
Expect(0, 64336, '\P{^Script_Extensions= -Hebrew}', "");
Error('\p{Scx= _Hebr/a/}');
Error('\P{Scx= _Hebr/a/}');
Expect(1, 64335, '\p{Scx=hebr}', "");
Expect(0, 64335, '\p{^Scx=hebr}', "");
Expect(0, 64335, '\P{Scx=hebr}', "");
Expect(1, 64335, '\P{^Scx=hebr}', "");
Expect(0, 64336, '\p{Scx=hebr}', "");
Expect(1, 64336, '\p{^Scx=hebr}', "");
Expect(1, 64336, '\P{Scx=hebr}', "");
Expect(0, 64336, '\P{^Scx=hebr}', "");
Expect(1, 64335, '\p{Scx=  hebr}', "");
Expect(0, 64335, '\p{^Scx=  hebr}', "");
Expect(0, 64335, '\P{Scx=  hebr}', "");
Expect(1, 64335, '\P{^Scx=  hebr}', "");
Expect(0, 64336, '\p{Scx=  hebr}', "");
Expect(1, 64336, '\p{^Scx=  hebr}', "");
Expect(1, 64336, '\P{Scx=  hebr}', "");
Expect(0, 64336, '\P{^Scx=  hebr}', "");
Error('\p{Is_Script_Extensions=_:=HEBREW}');
Error('\P{Is_Script_Extensions=_:=HEBREW}');
Expect(1, 64335, '\p{Is_Script_Extensions=hebrew}', "");
Expect(0, 64335, '\p{^Is_Script_Extensions=hebrew}', "");
Expect(0, 64335, '\P{Is_Script_Extensions=hebrew}', "");
Expect(1, 64335, '\P{^Is_Script_Extensions=hebrew}', "");
Expect(0, 64336, '\p{Is_Script_Extensions=hebrew}', "");
Expect(1, 64336, '\p{^Is_Script_Extensions=hebrew}', "");
Expect(1, 64336, '\P{Is_Script_Extensions=hebrew}', "");
Expect(0, 64336, '\P{^Is_Script_Extensions=hebrew}', "");
Expect(1, 64335, '\p{Is_Script_Extensions: - Hebrew}', "");
Expect(0, 64335, '\p{^Is_Script_Extensions: - Hebrew}', "");
Expect(0, 64335, '\P{Is_Script_Extensions: - Hebrew}', "");
Expect(1, 64335, '\P{^Is_Script_Extensions: - Hebrew}', "");
Expect(0, 64336, '\p{Is_Script_Extensions: - Hebrew}', "");
Expect(1, 64336, '\p{^Is_Script_Extensions: - Hebrew}', "");
Expect(1, 64336, '\P{Is_Script_Extensions: - Hebrew}', "");
Expect(0, 64336, '\P{^Is_Script_Extensions: - Hebrew}', "");
Error('\p{Is_Scx:   Hebr:=}');
Error('\P{Is_Scx:   Hebr:=}');
Expect(1, 64335, '\p{Is_Scx:hebr}', "");
Expect(0, 64335, '\p{^Is_Scx:hebr}', "");
Expect(0, 64335, '\P{Is_Scx:hebr}', "");
Expect(1, 64335, '\P{^Is_Scx:hebr}', "");
Expect(0, 64336, '\p{Is_Scx:hebr}', "");
Expect(1, 64336, '\p{^Is_Scx:hebr}', "");
Expect(1, 64336, '\P{Is_Scx:hebr}', "");
Expect(0, 64336, '\P{^Is_Scx:hebr}', "");
Expect(1, 64335, '\p{Is_Scx=--HEBR}', "");
Expect(0, 64335, '\p{^Is_Scx=--HEBR}', "");
Expect(0, 64335, '\P{Is_Scx=--HEBR}', "");
Expect(1, 64335, '\P{^Is_Scx=--HEBR}', "");
Expect(0, 64336, '\p{Is_Scx=--HEBR}', "");
Expect(1, 64336, '\p{^Is_Scx=--HEBR}', "");
Expect(1, 64336, '\P{Is_Scx=--HEBR}', "");
Expect(0, 64336, '\P{^Is_Scx=--HEBR}', "");
Error('\p{Script_Extensions=:= _SORA_Sompeng}');
Error('\P{Script_Extensions=:= _SORA_Sompeng}');
Expect(1, 69881, '\p{Script_Extensions=sorasompeng}', "");
Expect(0, 69881, '\p{^Script_Extensions=sorasompeng}', "");
Expect(0, 69881, '\P{Script_Extensions=sorasompeng}', "");
Expect(1, 69881, '\P{^Script_Extensions=sorasompeng}', "");
Expect(0, 69882, '\p{Script_Extensions=sorasompeng}', "");
Expect(1, 69882, '\p{^Script_Extensions=sorasompeng}', "");
Expect(1, 69882, '\P{Script_Extensions=sorasompeng}', "");
Expect(0, 69882, '\P{^Script_Extensions=sorasompeng}', "");
Expect(1, 69881, '\p{Script_Extensions= 	sora_Sompeng}', "");
Expect(0, 69881, '\p{^Script_Extensions= 	sora_Sompeng}', "");
Expect(0, 69881, '\P{Script_Extensions= 	sora_Sompeng}', "");
Expect(1, 69881, '\P{^Script_Extensions= 	sora_Sompeng}', "");
Expect(0, 69882, '\p{Script_Extensions= 	sora_Sompeng}', "");
Expect(1, 69882, '\p{^Script_Extensions= 	sora_Sompeng}', "");
Expect(1, 69882, '\P{Script_Extensions= 	sora_Sompeng}', "");
Expect(0, 69882, '\P{^Script_Extensions= 	sora_Sompeng}', "");
Error('\p{Scx=:=		Sora}');
Error('\P{Scx=:=		Sora}');
Expect(1, 69881, '\p{Scx=sora}', "");
Expect(0, 69881, '\p{^Scx=sora}', "");
Expect(0, 69881, '\P{Scx=sora}', "");
Expect(1, 69881, '\P{^Scx=sora}', "");
Expect(0, 69882, '\p{Scx=sora}', "");
Expect(1, 69882, '\p{^Scx=sora}', "");
Expect(1, 69882, '\P{Scx=sora}', "");
Expect(0, 69882, '\P{^Scx=sora}', "");
Expect(1, 69881, '\p{Scx=_ Sora}', "");
Expect(0, 69881, '\p{^Scx=_ Sora}', "");
Expect(0, 69881, '\P{Scx=_ Sora}', "");
Expect(1, 69881, '\P{^Scx=_ Sora}', "");
Expect(0, 69882, '\p{Scx=_ Sora}', "");
Expect(1, 69882, '\p{^Scx=_ Sora}', "");
Expect(1, 69882, '\P{Scx=_ Sora}', "");
Expect(0, 69882, '\P{^Scx=_ Sora}', "");
Error('\p{Is_Script_Extensions=:= -SORA_Sompeng}');
Error('\P{Is_Script_Extensions=:= -SORA_Sompeng}');
Expect(1, 69881, '\p{Is_Script_Extensions=sorasompeng}', "");
Expect(0, 69881, '\p{^Is_Script_Extensions=sorasompeng}', "");
Expect(0, 69881, '\P{Is_Script_Extensions=sorasompeng}', "");
Expect(1, 69881, '\P{^Is_Script_Extensions=sorasompeng}', "");
Expect(0, 69882, '\p{Is_Script_Extensions=sorasompeng}', "");
Expect(1, 69882, '\p{^Is_Script_Extensions=sorasompeng}', "");
Expect(1, 69882, '\P{Is_Script_Extensions=sorasompeng}', "");
Expect(0, 69882, '\P{^Is_Script_Extensions=sorasompeng}', "");
Expect(1, 69881, '\p{Is_Script_Extensions=		sora_SOMPENG}', "");
Expect(0, 69881, '\p{^Is_Script_Extensions=		sora_SOMPENG}', "");
Expect(0, 69881, '\P{Is_Script_Extensions=		sora_SOMPENG}', "");
Expect(1, 69881, '\P{^Is_Script_Extensions=		sora_SOMPENG}', "");
Expect(0, 69882, '\p{Is_Script_Extensions=		sora_SOMPENG}', "");
Expect(1, 69882, '\p{^Is_Script_Extensions=		sora_SOMPENG}', "");
Expect(1, 69882, '\P{Is_Script_Extensions=		sora_SOMPENG}', "");
Expect(0, 69882, '\P{^Is_Script_Extensions=		sora_SOMPENG}', "");
Error('\p{Is_Scx=_:=Sora}');
Error('\P{Is_Scx=_:=Sora}');
Expect(1, 69881, '\p{Is_Scx=sora}', "");
Expect(0, 69881, '\p{^Is_Scx=sora}', "");
Expect(0, 69881, '\P{Is_Scx=sora}', "");
Expect(1, 69881, '\P{^Is_Scx=sora}', "");
Expect(0, 69882, '\p{Is_Scx=sora}', "");
Expect(1, 69882, '\p{^Is_Scx=sora}', "");
Expect(1, 69882, '\P{Is_Scx=sora}', "");
Expect(0, 69882, '\P{^Is_Scx=sora}', "");
Expect(1, 69881, '\p{Is_Scx=_	SORA}', "");
Expect(0, 69881, '\p{^Is_Scx=_	SORA}', "");
Expect(0, 69881, '\P{Is_Scx=_	SORA}', "");
Expect(1, 69881, '\P{^Is_Scx=_	SORA}', "");
Expect(0, 69882, '\p{Is_Scx=_	SORA}', "");
Expect(1, 69882, '\p{^Is_Scx=_	SORA}', "");
Expect(1, 69882, '\P{Is_Scx=_	SORA}', "");
Expect(0, 69882, '\P{^Is_Scx=_	SORA}', "");
Error('\p{Script_Extensions=	_Inscriptional_PAHLAVI:=}');
Error('\P{Script_Extensions=	_Inscriptional_PAHLAVI:=}');
Expect(1, 68479, '\p{Script_Extensions:   inscriptionalpahlavi}', "");
Expect(0, 68479, '\p{^Script_Extensions:   inscriptionalpahlavi}', "");
Expect(0, 68479, '\P{Script_Extensions:   inscriptionalpahlavi}', "");
Expect(1, 68479, '\P{^Script_Extensions:   inscriptionalpahlavi}', "");
Expect(0, 68480, '\p{Script_Extensions:   inscriptionalpahlavi}', "");
Expect(1, 68480, '\p{^Script_Extensions:   inscriptionalpahlavi}', "");
Expect(1, 68480, '\P{Script_Extensions:   inscriptionalpahlavi}', "");
Expect(0, 68480, '\P{^Script_Extensions:   inscriptionalpahlavi}', "");
Expect(1, 68479, '\p{Script_Extensions=--inscriptional_Pahlavi}', "");
Expect(0, 68479, '\p{^Script_Extensions=--inscriptional_Pahlavi}', "");
Expect(0, 68479, '\P{Script_Extensions=--inscriptional_Pahlavi}', "");
Expect(1, 68479, '\P{^Script_Extensions=--inscriptional_Pahlavi}', "");
Expect(0, 68480, '\p{Script_Extensions=--inscriptional_Pahlavi}', "");
Expect(1, 68480, '\p{^Script_Extensions=--inscriptional_Pahlavi}', "");
Expect(1, 68480, '\P{Script_Extensions=--inscriptional_Pahlavi}', "");
Expect(0, 68480, '\P{^Script_Extensions=--inscriptional_Pahlavi}', "");
Error('\p{Scx:-	PHLI:=}');
Error('\P{Scx:-	PHLI:=}');
Expect(1, 68479, '\p{Scx=phli}', "");
Expect(0, 68479, '\p{^Scx=phli}', "");
Expect(0, 68479, '\P{Scx=phli}', "");
Expect(1, 68479, '\P{^Scx=phli}', "");
Expect(0, 68480, '\p{Scx=phli}', "");
Expect(1, 68480, '\p{^Scx=phli}', "");
Expect(1, 68480, '\P{Scx=phli}', "");
Expect(0, 68480, '\P{^Scx=phli}', "");
Expect(1, 68479, '\p{Scx= Phli}', "");
Expect(0, 68479, '\p{^Scx= Phli}', "");
Expect(0, 68479, '\P{Scx= Phli}', "");
Expect(1, 68479, '\P{^Scx= Phli}', "");
Expect(0, 68480, '\p{Scx= Phli}', "");
Expect(1, 68480, '\p{^Scx= Phli}', "");
Expect(1, 68480, '\P{Scx= Phli}', "");
Expect(0, 68480, '\P{^Scx= Phli}', "");
Error('\p{Is_Script_Extensions=/a/ 	inscriptional_PAHLAVI}');
Error('\P{Is_Script_Extensions=/a/ 	inscriptional_PAHLAVI}');
Expect(1, 68479, '\p{Is_Script_Extensions=inscriptionalpahlavi}', "");
Expect(0, 68479, '\p{^Is_Script_Extensions=inscriptionalpahlavi}', "");
Expect(0, 68479, '\P{Is_Script_Extensions=inscriptionalpahlavi}', "");
Expect(1, 68479, '\P{^Is_Script_Extensions=inscriptionalpahlavi}', "");
Expect(0, 68480, '\p{Is_Script_Extensions=inscriptionalpahlavi}', "");
Expect(1, 68480, '\p{^Is_Script_Extensions=inscriptionalpahlavi}', "");
Expect(1, 68480, '\P{Is_Script_Extensions=inscriptionalpahlavi}', "");
Expect(0, 68480, '\P{^Is_Script_Extensions=inscriptionalpahlavi}', "");
Expect(1, 68479, '\p{Is_Script_Extensions= 	INSCRIPTIONAL_Pahlavi}', "");
Expect(0, 68479, '\p{^Is_Script_Extensions= 	INSCRIPTIONAL_Pahlavi}', "");
Expect(0, 68479, '\P{Is_Script_Extensions= 	INSCRIPTIONAL_Pahlavi}', "");
Expect(1, 68479, '\P{^Is_Script_Extensions= 	INSCRIPTIONAL_Pahlavi}', "");
Expect(0, 68480, '\p{Is_Script_Extensions= 	INSCRIPTIONAL_Pahlavi}', "");
Expect(1, 68480, '\p{^Is_Script_Extensions= 	INSCRIPTIONAL_Pahlavi}', "");
Expect(1, 68480, '\P{Is_Script_Extensions= 	INSCRIPTIONAL_Pahlavi}', "");
Expect(0, 68480, '\P{^Is_Script_Extensions= 	INSCRIPTIONAL_Pahlavi}', "");
Error('\p{Is_Scx=	/a/phli}');
Error('\P{Is_Scx=	/a/phli}');
Expect(1, 68479, '\p{Is_Scx=phli}', "");
Expect(0, 68479, '\p{^Is_Scx=phli}', "");
Expect(0, 68479, '\P{Is_Scx=phli}', "");
Expect(1, 68479, '\P{^Is_Scx=phli}', "");
Expect(0, 68480, '\p{Is_Scx=phli}', "");
Expect(1, 68480, '\p{^Is_Scx=phli}', "");
Expect(1, 68480, '\P{Is_Scx=phli}', "");
Expect(0, 68480, '\P{^Is_Scx=phli}', "");
Expect(1, 68479, '\p{Is_Scx=		Phli}', "");
Expect(0, 68479, '\p{^Is_Scx=		Phli}', "");
Expect(0, 68479, '\P{Is_Scx=		Phli}', "");
Expect(1, 68479, '\P{^Is_Scx=		Phli}', "");
Expect(0, 68480, '\p{Is_Scx=		Phli}', "");
Expect(1, 68480, '\p{^Is_Scx=		Phli}', "");
Expect(1, 68480, '\P{Is_Scx=		Phli}', "");
Expect(0, 68480, '\P{^Is_Scx=		Phli}', "");
Error('\p{Script_Extensions=/a/ tai_Tham}');
Error('\P{Script_Extensions=/a/ tai_Tham}');
Expect(1, 6829, '\p{Script_Extensions=taitham}', "");
Expect(0, 6829, '\p{^Script_Extensions=taitham}', "");
Expect(0, 6829, '\P{Script_Extensions=taitham}', "");
Expect(1, 6829, '\P{^Script_Extensions=taitham}', "");
Expect(0, 6830, '\p{Script_Extensions=taitham}', "");
Expect(1, 6830, '\p{^Script_Extensions=taitham}', "");
Expect(1, 6830, '\P{Script_Extensions=taitham}', "");
Expect(0, 6830, '\P{^Script_Extensions=taitham}', "");
Expect(1, 6829, '\p{Script_Extensions=_ TAI_tham}', "");
Expect(0, 6829, '\p{^Script_Extensions=_ TAI_tham}', "");
Expect(0, 6829, '\P{Script_Extensions=_ TAI_tham}', "");
Expect(1, 6829, '\P{^Script_Extensions=_ TAI_tham}', "");
Expect(0, 6830, '\p{Script_Extensions=_ TAI_tham}', "");
Expect(1, 6830, '\p{^Script_Extensions=_ TAI_tham}', "");
Expect(1, 6830, '\P{Script_Extensions=_ TAI_tham}', "");
Expect(0, 6830, '\P{^Script_Extensions=_ TAI_tham}', "");
Error('\p{Scx=_/a/LANA}');
Error('\P{Scx=_/a/LANA}');
Expect(1, 6829, '\p{Scx: lana}', "");
Expect(0, 6829, '\p{^Scx: lana}', "");
Expect(0, 6829, '\P{Scx: lana}', "");
Expect(1, 6829, '\P{^Scx: lana}', "");
Expect(0, 6830, '\p{Scx: lana}', "");
Expect(1, 6830, '\p{^Scx: lana}', "");
Expect(1, 6830, '\P{Scx: lana}', "");
Expect(0, 6830, '\P{^Scx: lana}', "");
Expect(1, 6829, '\p{Scx=_lana}', "");
Expect(0, 6829, '\p{^Scx=_lana}', "");
Expect(0, 6829, '\P{Scx=_lana}', "");
Expect(1, 6829, '\P{^Scx=_lana}', "");
Expect(0, 6830, '\p{Scx=_lana}', "");
Expect(1, 6830, '\p{^Scx=_lana}', "");
Expect(1, 6830, '\P{Scx=_lana}', "");
Expect(0, 6830, '\P{^Scx=_lana}', "");
Error('\p{Is_Script_Extensions=_/a/Tai_THAM}');
Error('\P{Is_Script_Extensions=_/a/Tai_THAM}');
Expect(1, 6829, '\p{Is_Script_Extensions=taitham}', "");
Expect(0, 6829, '\p{^Is_Script_Extensions=taitham}', "");
Expect(0, 6829, '\P{Is_Script_Extensions=taitham}', "");
Expect(1, 6829, '\P{^Is_Script_Extensions=taitham}', "");
Expect(0, 6830, '\p{Is_Script_Extensions=taitham}', "");
Expect(1, 6830, '\p{^Is_Script_Extensions=taitham}', "");
Expect(1, 6830, '\P{Is_Script_Extensions=taitham}', "");
Expect(0, 6830, '\P{^Is_Script_Extensions=taitham}', "");
Expect(1, 6829, '\p{Is_Script_Extensions=-tai_tham}', "");
Expect(0, 6829, '\p{^Is_Script_Extensions=-tai_tham}', "");
Expect(0, 6829, '\P{Is_Script_Extensions=-tai_tham}', "");
Expect(1, 6829, '\P{^Is_Script_Extensions=-tai_tham}', "");
Expect(0, 6830, '\p{Is_Script_Extensions=-tai_tham}', "");
Expect(1, 6830, '\p{^Is_Script_Extensions=-tai_tham}', "");
Expect(1, 6830, '\P{Is_Script_Extensions=-tai_tham}', "");
Expect(0, 6830, '\P{^Is_Script_Extensions=-tai_tham}', "");
Error('\p{Is_Scx=:=	Lana}');
Error('\P{Is_Scx=:=	Lana}');
Expect(1, 6829, '\p{Is_Scx=lana}', "");
Expect(0, 6829, '\p{^Is_Scx=lana}', "");
Expect(0, 6829, '\P{Is_Scx=lana}', "");
Expect(1, 6829, '\P{^Is_Scx=lana}', "");
Expect(0, 6830, '\p{Is_Scx=lana}', "");
Expect(1, 6830, '\p{^Is_Scx=lana}', "");
Expect(1, 6830, '\P{Is_Scx=lana}', "");
Expect(0, 6830, '\P{^Is_Scx=lana}', "");
Expect(1, 6829, '\p{Is_Scx=	_Lana}', "");
Expect(0, 6829, '\p{^Is_Scx=	_Lana}', "");
Expect(0, 6829, '\P{Is_Scx=	_Lana}', "");
Expect(1, 6829, '\P{^Is_Scx=	_Lana}', "");
Expect(0, 6830, '\p{Is_Scx=	_Lana}', "");
Expect(1, 6830, '\p{^Is_Scx=	_Lana}', "");
Expect(1, 6830, '\P{Is_Scx=	_Lana}', "");
Expect(0, 6830, '\P{^Is_Scx=	_Lana}', "");
Error('\p{Script_Extensions:	/a/Mandaic}');
Error('\P{Script_Extensions:	/a/Mandaic}');
Expect(1, 2142, '\p{Script_Extensions=mandaic}', "");
Expect(0, 2142, '\p{^Script_Extensions=mandaic}', "");
Expect(0, 2142, '\P{Script_Extensions=mandaic}', "");
Expect(1, 2142, '\P{^Script_Extensions=mandaic}', "");
Expect(0, 2143, '\p{Script_Extensions=mandaic}', "");
Expect(1, 2143, '\p{^Script_Extensions=mandaic}', "");
Expect(1, 2143, '\P{Script_Extensions=mandaic}', "");
Expect(0, 2143, '\P{^Script_Extensions=mandaic}', "");
Expect(1, 2142, '\p{Script_Extensions=	-Mandaic}', "");
Expect(0, 2142, '\p{^Script_Extensions=	-Mandaic}', "");
Expect(0, 2142, '\P{Script_Extensions=	-Mandaic}', "");
Expect(1, 2142, '\P{^Script_Extensions=	-Mandaic}', "");
Expect(0, 2143, '\p{Script_Extensions=	-Mandaic}', "");
Expect(1, 2143, '\p{^Script_Extensions=	-Mandaic}', "");
Expect(1, 2143, '\P{Script_Extensions=	-Mandaic}', "");
Expect(0, 2143, '\P{^Script_Extensions=	-Mandaic}', "");
Error('\p{Scx=  Mand:=}');
Error('\P{Scx=  Mand:=}');
Expect(1, 2142, '\p{Scx=mand}', "");
Expect(0, 2142, '\p{^Scx=mand}', "");
Expect(0, 2142, '\P{Scx=mand}', "");
Expect(1, 2142, '\P{^Scx=mand}', "");
Expect(0, 2143, '\p{Scx=mand}', "");
Expect(1, 2143, '\p{^Scx=mand}', "");
Expect(1, 2143, '\P{Scx=mand}', "");
Expect(0, 2143, '\P{^Scx=mand}', "");
Expect(1, 2142, '\p{Scx=  mand}', "");
Expect(0, 2142, '\p{^Scx=  mand}', "");
Expect(0, 2142, '\P{Scx=  mand}', "");
Expect(1, 2142, '\P{^Scx=  mand}', "");
Expect(0, 2143, '\p{Scx=  mand}', "");
Expect(1, 2143, '\p{^Scx=  mand}', "");
Expect(1, 2143, '\P{Scx=  mand}', "");
Expect(0, 2143, '\P{^Scx=  mand}', "");
Error('\p{Is_Script_Extensions=:= -Mandaic}');
Error('\P{Is_Script_Extensions=:= -Mandaic}');
Expect(1, 2142, '\p{Is_Script_Extensions=mandaic}', "");
Expect(0, 2142, '\p{^Is_Script_Extensions=mandaic}', "");
Expect(0, 2142, '\P{Is_Script_Extensions=mandaic}', "");
Expect(1, 2142, '\P{^Is_Script_Extensions=mandaic}', "");
Expect(0, 2143, '\p{Is_Script_Extensions=mandaic}', "");
Expect(1, 2143, '\p{^Is_Script_Extensions=mandaic}', "");
Expect(1, 2143, '\P{Is_Script_Extensions=mandaic}', "");
Expect(0, 2143, '\P{^Is_Script_Extensions=mandaic}', "");
Expect(1, 2142, '\p{Is_Script_Extensions=  Mandaic}', "");
Expect(0, 2142, '\p{^Is_Script_Extensions=  Mandaic}', "");
Expect(0, 2142, '\P{Is_Script_Extensions=  Mandaic}', "");
Expect(1, 2142, '\P{^Is_Script_Extensions=  Mandaic}', "");
Expect(0, 2143, '\p{Is_Script_Extensions=  Mandaic}', "");
Expect(1, 2143, '\p{^Is_Script_Extensions=  Mandaic}', "");
Expect(1, 2143, '\P{Is_Script_Extensions=  Mandaic}', "");
Expect(0, 2143, '\P{^Is_Script_Extensions=  Mandaic}', "");
Error('\p{Is_Scx=/a/Mand}');
Error('\P{Is_Scx=/a/Mand}');
Expect(1, 2142, '\p{Is_Scx=mand}', "");
Expect(0, 2142, '\p{^Is_Scx=mand}', "");
Expect(0, 2142, '\P{Is_Scx=mand}', "");
Expect(1, 2142, '\P{^Is_Scx=mand}', "");
Expect(0, 2143, '\p{Is_Scx=mand}', "");
Expect(1, 2143, '\p{^Is_Scx=mand}', "");
Expect(1, 2143, '\P{Is_Scx=mand}', "");
Expect(0, 2143, '\P{^Is_Scx=mand}', "");
Expect(1, 2142, '\p{Is_Scx=_	Mand}', "");
Expect(0, 2142, '\p{^Is_Scx=_	Mand}', "");
Expect(0, 2142, '\P{Is_Scx=_	Mand}', "");
Expect(1, 2142, '\P{^Is_Scx=_	Mand}', "");
Expect(0, 2143, '\p{Is_Scx=_	Mand}', "");
Expect(1, 2143, '\p{^Is_Scx=_	Mand}', "");
Expect(1, 2143, '\P{Is_Scx=_	Mand}', "");
Expect(0, 2143, '\P{^Is_Scx=_	Mand}', "");
Error('\p{Script_Extensions=_/a/Gurmukhi}');
Error('\P{Script_Extensions=_/a/Gurmukhi}');
Expect(1, 43065, '\p{Script_Extensions=gurmukhi}', "");
Expect(0, 43065, '\p{^Script_Extensions=gurmukhi}', "");
Expect(0, 43065, '\P{Script_Extensions=gurmukhi}', "");
Expect(1, 43065, '\P{^Script_Extensions=gurmukhi}', "");
Expect(0, 43066, '\p{Script_Extensions=gurmukhi}', "");
Expect(1, 43066, '\p{^Script_Extensions=gurmukhi}', "");
Expect(1, 43066, '\P{Script_Extensions=gurmukhi}', "");
Expect(0, 43066, '\P{^Script_Extensions=gurmukhi}', "");
Expect(1, 43065, '\p{Script_Extensions=-_GURMUKHI}', "");
Expect(0, 43065, '\p{^Script_Extensions=-_GURMUKHI}', "");
Expect(0, 43065, '\P{Script_Extensions=-_GURMUKHI}', "");
Expect(1, 43065, '\P{^Script_Extensions=-_GURMUKHI}', "");
Expect(0, 43066, '\p{Script_Extensions=-_GURMUKHI}', "");
Expect(1, 43066, '\p{^Script_Extensions=-_GURMUKHI}', "");
Expect(1, 43066, '\P{Script_Extensions=-_GURMUKHI}', "");
Expect(0, 43066, '\P{^Script_Extensions=-_GURMUKHI}', "");
Error('\p{Scx=		Guru/a/}');
Error('\P{Scx=		Guru/a/}');
Expect(1, 43065, '\p{Scx=guru}', "");
Expect(0, 43065, '\p{^Scx=guru}', "");
Expect(0, 43065, '\P{Scx=guru}', "");
Expect(1, 43065, '\P{^Scx=guru}', "");
Expect(0, 43066, '\p{Scx=guru}', "");
Expect(1, 43066, '\p{^Scx=guru}', "");
Expect(1, 43066, '\P{Scx=guru}', "");
Expect(0, 43066, '\P{^Scx=guru}', "");
Expect(1, 43065, '\p{Scx=--GURU}', "");
Expect(0, 43065, '\p{^Scx=--GURU}', "");
Expect(0, 43065, '\P{Scx=--GURU}', "");
Expect(1, 43065, '\P{^Scx=--GURU}', "");
Expect(0, 43066, '\p{Scx=--GURU}', "");
Expect(1, 43066, '\p{^Scx=--GURU}', "");
Expect(1, 43066, '\P{Scx=--GURU}', "");
Expect(0, 43066, '\P{^Scx=--GURU}', "");
Error('\p{Is_Script_Extensions=:=		Gurmukhi}');
Error('\P{Is_Script_Extensions=:=		Gurmukhi}');
Expect(1, 43065, '\p{Is_Script_Extensions=gurmukhi}', "");
Expect(0, 43065, '\p{^Is_Script_Extensions=gurmukhi}', "");
Expect(0, 43065, '\P{Is_Script_Extensions=gurmukhi}', "");
Expect(1, 43065, '\P{^Is_Script_Extensions=gurmukhi}', "");
Expect(0, 43066, '\p{Is_Script_Extensions=gurmukhi}', "");
Expect(1, 43066, '\p{^Is_Script_Extensions=gurmukhi}', "");
Expect(1, 43066, '\P{Is_Script_Extensions=gurmukhi}', "");
Expect(0, 43066, '\P{^Is_Script_Extensions=gurmukhi}', "");
Expect(1, 43065, '\p{Is_Script_Extensions=		GURMUKHI}', "");
Expect(0, 43065, '\p{^Is_Script_Extensions=		GURMUKHI}', "");
Expect(0, 43065, '\P{Is_Script_Extensions=		GURMUKHI}', "");
Expect(1, 43065, '\P{^Is_Script_Extensions=		GURMUKHI}', "");
Expect(0, 43066, '\p{Is_Script_Extensions=		GURMUKHI}', "");
Expect(1, 43066, '\p{^Is_Script_Extensions=		GURMUKHI}', "");
Expect(1, 43066, '\P{Is_Script_Extensions=		GURMUKHI}', "");
Expect(0, 43066, '\P{^Is_Script_Extensions=		GURMUKHI}', "");
Error('\p{Is_Scx=:=-	GURU}');
Error('\P{Is_Scx=:=-	GURU}');
Expect(1, 43065, '\p{Is_Scx=guru}', "");
Expect(0, 43065, '\p{^Is_Scx=guru}', "");
Expect(0, 43065, '\P{Is_Scx=guru}', "");
Expect(1, 43065, '\P{^Is_Scx=guru}', "");
Expect(0, 43066, '\p{Is_Scx=guru}', "");
Expect(1, 43066, '\p{^Is_Scx=guru}', "");
Expect(1, 43066, '\P{Is_Scx=guru}', "");
Expect(0, 43066, '\P{^Is_Scx=guru}', "");
Expect(1, 43065, '\p{Is_Scx=-Guru}', "");
Expect(0, 43065, '\p{^Is_Scx=-Guru}', "");
Expect(0, 43065, '\P{Is_Scx=-Guru}', "");
Expect(1, 43065, '\P{^Is_Scx=-Guru}', "");
Expect(0, 43066, '\p{Is_Scx=-Guru}', "");
Expect(1, 43066, '\p{^Is_Scx=-Guru}', "");
Expect(1, 43066, '\P{Is_Scx=-Guru}', "");
Expect(0, 43066, '\P{^Is_Scx=-Guru}', "");
Error('\p{Script_Extensions=/a/Batak}');
Error('\P{Script_Extensions=/a/Batak}');
Expect(1, 7167, '\p{Script_Extensions=batak}', "");
Expect(0, 7167, '\p{^Script_Extensions=batak}', "");
Expect(0, 7167, '\P{Script_Extensions=batak}', "");
Expect(1, 7167, '\P{^Script_Extensions=batak}', "");
Expect(0, 7168, '\p{Script_Extensions=batak}', "");
Expect(1, 7168, '\p{^Script_Extensions=batak}', "");
Expect(1, 7168, '\P{Script_Extensions=batak}', "");
Expect(0, 7168, '\P{^Script_Extensions=batak}', "");
Expect(1, 7167, '\p{Script_Extensions=_-Batak}', "");
Expect(0, 7167, '\p{^Script_Extensions=_-Batak}', "");
Expect(0, 7167, '\P{Script_Extensions=_-Batak}', "");
Expect(1, 7167, '\P{^Script_Extensions=_-Batak}', "");
Expect(0, 7168, '\p{Script_Extensions=_-Batak}', "");
Expect(1, 7168, '\p{^Script_Extensions=_-Batak}', "");
Expect(1, 7168, '\P{Script_Extensions=_-Batak}', "");
Expect(0, 7168, '\P{^Script_Extensions=_-Batak}', "");
Error('\p{Scx=/a/  Batk}');
Error('\P{Scx=/a/  Batk}');
Expect(1, 7167, '\p{Scx=batk}', "");
Expect(0, 7167, '\p{^Scx=batk}', "");
Expect(0, 7167, '\P{Scx=batk}', "");
Expect(1, 7167, '\P{^Scx=batk}', "");
Expect(0, 7168, '\p{Scx=batk}', "");
Expect(1, 7168, '\p{^Scx=batk}', "");
Expect(1, 7168, '\P{Scx=batk}', "");
Expect(0, 7168, '\P{^Scx=batk}', "");
Expect(1, 7167, '\p{Scx=- batk}', "");
Expect(0, 7167, '\p{^Scx=- batk}', "");
Expect(0, 7167, '\P{Scx=- batk}', "");
Expect(1, 7167, '\P{^Scx=- batk}', "");
Expect(0, 7168, '\p{Scx=- batk}', "");
Expect(1, 7168, '\p{^Scx=- batk}', "");
Expect(1, 7168, '\P{Scx=- batk}', "");
Expect(0, 7168, '\P{^Scx=- batk}', "");
Error('\p{Is_Script_Extensions=/a/_ batak}');
Error('\P{Is_Script_Extensions=/a/_ batak}');
Expect(1, 7167, '\p{Is_Script_Extensions=batak}', "");
Expect(0, 7167, '\p{^Is_Script_Extensions=batak}', "");
Expect(0, 7167, '\P{Is_Script_Extensions=batak}', "");
Expect(1, 7167, '\P{^Is_Script_Extensions=batak}', "");
Expect(0, 7168, '\p{Is_Script_Extensions=batak}', "");
Expect(1, 7168, '\p{^Is_Script_Extensions=batak}', "");
Expect(1, 7168, '\P{Is_Script_Extensions=batak}', "");
Expect(0, 7168, '\P{^Is_Script_Extensions=batak}', "");
Expect(1, 7167, '\p{Is_Script_Extensions=-_batak}', "");
Expect(0, 7167, '\p{^Is_Script_Extensions=-_batak}', "");
Expect(0, 7167, '\P{Is_Script_Extensions=-_batak}', "");
Expect(1, 7167, '\P{^Is_Script_Extensions=-_batak}', "");
Expect(0, 7168, '\p{Is_Script_Extensions=-_batak}', "");
Expect(1, 7168, '\p{^Is_Script_Extensions=-_batak}', "");
Expect(1, 7168, '\P{Is_Script_Extensions=-_batak}', "");
Expect(0, 7168, '\P{^Is_Script_Extensions=-_batak}', "");
Error('\p{Is_Scx:	-BATK/a/}');
Error('\P{Is_Scx:	-BATK/a/}');
Expect(1, 7167, '\p{Is_Scx=batk}', "");
Expect(0, 7167, '\p{^Is_Scx=batk}', "");
Expect(0, 7167, '\P{Is_Scx=batk}', "");
Expect(1, 7167, '\P{^Is_Scx=batk}', "");
Expect(0, 7168, '\p{Is_Scx=batk}', "");
Expect(1, 7168, '\p{^Is_Scx=batk}', "");
Expect(1, 7168, '\P{Is_Scx=batk}', "");
Expect(0, 7168, '\P{^Is_Scx=batk}', "");
Expect(1, 7167, '\p{Is_Scx=	-BATK}', "");
Expect(0, 7167, '\p{^Is_Scx=	-BATK}', "");
Expect(0, 7167, '\P{Is_Scx=	-BATK}', "");
Expect(1, 7167, '\P{^Is_Scx=	-BATK}', "");
Expect(0, 7168, '\p{Is_Scx=	-BATK}', "");
Expect(1, 7168, '\p{^Is_Scx=	-BATK}', "");
Expect(1, 7168, '\P{Is_Scx=	-BATK}', "");
Expect(0, 7168, '\P{^Is_Scx=	-BATK}', "");
Error('\p{Script_Extensions:/a/cypriot}');
Error('\P{Script_Extensions:/a/cypriot}');
Expect(1, 67647, '\p{Script_Extensions:cypriot}', "");
Expect(0, 67647, '\p{^Script_Extensions:cypriot}', "");
Expect(0, 67647, '\P{Script_Extensions:cypriot}', "");
Expect(1, 67647, '\P{^Script_Extensions:cypriot}', "");
Expect(0, 67648, '\p{Script_Extensions:cypriot}', "");
Expect(1, 67648, '\p{^Script_Extensions:cypriot}', "");
Expect(1, 67648, '\P{Script_Extensions:cypriot}', "");
Expect(0, 67648, '\P{^Script_Extensions:cypriot}', "");
Expect(1, 67647, '\p{Script_Extensions=_-cypriot}', "");
Expect(0, 67647, '\p{^Script_Extensions=_-cypriot}', "");
Expect(0, 67647, '\P{Script_Extensions=_-cypriot}', "");
Expect(1, 67647, '\P{^Script_Extensions=_-cypriot}', "");
Expect(0, 67648, '\p{Script_Extensions=_-cypriot}', "");
Expect(1, 67648, '\p{^Script_Extensions=_-cypriot}', "");
Expect(1, 67648, '\P{Script_Extensions=_-cypriot}', "");
Expect(0, 67648, '\P{^Script_Extensions=_-cypriot}', "");
Error('\p{Scx=:= Cprt}');
Error('\P{Scx=:= Cprt}');
Expect(1, 67647, '\p{Scx=cprt}', "");
Expect(0, 67647, '\p{^Scx=cprt}', "");
Expect(0, 67647, '\P{Scx=cprt}', "");
Expect(1, 67647, '\P{^Scx=cprt}', "");
Expect(0, 67648, '\p{Scx=cprt}', "");
Expect(1, 67648, '\p{^Scx=cprt}', "");
Expect(1, 67648, '\P{Scx=cprt}', "");
Expect(0, 67648, '\P{^Scx=cprt}', "");
Expect(1, 67647, '\p{Scx:-cprt}', "");
Expect(0, 67647, '\p{^Scx:-cprt}', "");
Expect(0, 67647, '\P{Scx:-cprt}', "");
Expect(1, 67647, '\P{^Scx:-cprt}', "");
Expect(0, 67648, '\p{Scx:-cprt}', "");
Expect(1, 67648, '\p{^Scx:-cprt}', "");
Expect(1, 67648, '\P{Scx:-cprt}', "");
Expect(0, 67648, '\P{^Scx:-cprt}', "");
Error('\p{Is_Script_Extensions=/a/ -Cypriot}');
Error('\P{Is_Script_Extensions=/a/ -Cypriot}');
Expect(1, 67647, '\p{Is_Script_Extensions=cypriot}', "");
Expect(0, 67647, '\p{^Is_Script_Extensions=cypriot}', "");
Expect(0, 67647, '\P{Is_Script_Extensions=cypriot}', "");
Expect(1, 67647, '\P{^Is_Script_Extensions=cypriot}', "");
Expect(0, 67648, '\p{Is_Script_Extensions=cypriot}', "");
Expect(1, 67648, '\p{^Is_Script_Extensions=cypriot}', "");
Expect(1, 67648, '\P{Is_Script_Extensions=cypriot}', "");
Expect(0, 67648, '\P{^Is_Script_Extensions=cypriot}', "");
Expect(1, 67647, '\p{Is_Script_Extensions=--Cypriot}', "");
Expect(0, 67647, '\p{^Is_Script_Extensions=--Cypriot}', "");
Expect(0, 67647, '\P{Is_Script_Extensions=--Cypriot}', "");
Expect(1, 67647, '\P{^Is_Script_Extensions=--Cypriot}', "");
Expect(0, 67648, '\p{Is_Script_Extensions=--Cypriot}', "");
Expect(1, 67648, '\p{^Is_Script_Extensions=--Cypriot}', "");
Expect(1, 67648, '\P{Is_Script_Extensions=--Cypriot}', "");
Expect(0, 67648, '\P{^Is_Script_Extensions=--Cypriot}', "");
Error('\p{Is_Scx=/a/--CPRT}');
Error('\P{Is_Scx=/a/--CPRT}');
Expect(1, 67647, '\p{Is_Scx=cprt}', "");
Expect(0, 67647, '\p{^Is_Scx=cprt}', "");
Expect(0, 67647, '\P{Is_Scx=cprt}', "");
Expect(1, 67647, '\P{^Is_Scx=cprt}', "");
Expect(0, 67648, '\p{Is_Scx=cprt}', "");
Expect(1, 67648, '\p{^Is_Scx=cprt}', "");
Expect(1, 67648, '\P{Is_Scx=cprt}', "");
Expect(0, 67648, '\P{^Is_Scx=cprt}', "");
Expect(1, 67647, '\p{Is_Scx=_	Cprt}', "");
Expect(0, 67647, '\p{^Is_Scx=_	Cprt}', "");
Expect(0, 67647, '\P{Is_Scx=_	Cprt}', "");
Expect(1, 67647, '\P{^Is_Scx=_	Cprt}', "");
Expect(0, 67648, '\p{Is_Scx=_	Cprt}', "");
Expect(1, 67648, '\p{^Is_Scx=_	Cprt}', "");
Expect(1, 67648, '\P{Is_Scx=_	Cprt}', "");
Expect(0, 67648, '\P{^Is_Scx=_	Cprt}', "");
Error('\p{Script_Extensions= HANGUL:=}');
Error('\P{Script_Extensions= HANGUL:=}');
Expect(1, 65500, '\p{Script_Extensions=hangul}', "");
Expect(0, 65500, '\p{^Script_Extensions=hangul}', "");
Expect(0, 65500, '\P{Script_Extensions=hangul}', "");
Expect(1, 65500, '\P{^Script_Extensions=hangul}', "");
Expect(0, 65501, '\p{Script_Extensions=hangul}', "");
Expect(1, 65501, '\p{^Script_Extensions=hangul}', "");
Expect(1, 65501, '\P{Script_Extensions=hangul}', "");
Expect(0, 65501, '\P{^Script_Extensions=hangul}', "");
Expect(1, 65500, '\p{Script_Extensions=- hangul}', "");
Expect(0, 65500, '\p{^Script_Extensions=- hangul}', "");
Expect(0, 65500, '\P{Script_Extensions=- hangul}', "");
Expect(1, 65500, '\P{^Script_Extensions=- hangul}', "");
Expect(0, 65501, '\p{Script_Extensions=- hangul}', "");
Expect(1, 65501, '\p{^Script_Extensions=- hangul}', "");
Expect(1, 65501, '\P{Script_Extensions=- hangul}', "");
Expect(0, 65501, '\P{^Script_Extensions=- hangul}', "");
Error('\p{Scx=	/a/HANG}');
Error('\P{Scx=	/a/HANG}');
Expect(1, 65500, '\p{Scx=hang}', "");
Expect(0, 65500, '\p{^Scx=hang}', "");
Expect(0, 65500, '\P{Scx=hang}', "");
Expect(1, 65500, '\P{^Scx=hang}', "");
Expect(0, 65501, '\p{Scx=hang}', "");
Expect(1, 65501, '\p{^Scx=hang}', "");
Expect(1, 65501, '\P{Scx=hang}', "");
Expect(0, 65501, '\P{^Scx=hang}', "");
Expect(1, 65500, '\p{Scx:   - Hang}', "");
Expect(0, 65500, '\p{^Scx:   - Hang}', "");
Expect(0, 65500, '\P{Scx:   - Hang}', "");
Expect(1, 65500, '\P{^Scx:   - Hang}', "");
Expect(0, 65501, '\p{Scx:   - Hang}', "");
Expect(1, 65501, '\p{^Scx:   - Hang}', "");
Expect(1, 65501, '\P{Scx:   - Hang}', "");
Expect(0, 65501, '\P{^Scx:   - Hang}', "");
Error('\p{Is_Script_Extensions=:=	 Hangul}');
Error('\P{Is_Script_Extensions=:=	 Hangul}');
Expect(1, 65500, '\p{Is_Script_Extensions: hangul}', "");
Expect(0, 65500, '\p{^Is_Script_Extensions: hangul}', "");
Expect(0, 65500, '\P{Is_Script_Extensions: hangul}', "");
Expect(1, 65500, '\P{^Is_Script_Extensions: hangul}', "");
Expect(0, 65501, '\p{Is_Script_Extensions: hangul}', "");
Expect(1, 65501, '\p{^Is_Script_Extensions: hangul}', "");
Expect(1, 65501, '\P{Is_Script_Extensions: hangul}', "");
Expect(0, 65501, '\P{^Is_Script_Extensions: hangul}', "");
Expect(1, 65500, '\p{Is_Script_Extensions:		hangul}', "");
Expect(0, 65500, '\p{^Is_Script_Extensions:		hangul}', "");
Expect(0, 65500, '\P{Is_Script_Extensions:		hangul}', "");
Expect(1, 65500, '\P{^Is_Script_Extensions:		hangul}', "");
Expect(0, 65501, '\p{Is_Script_Extensions:		hangul}', "");
Expect(1, 65501, '\p{^Is_Script_Extensions:		hangul}', "");
Expect(1, 65501, '\P{Is_Script_Extensions:		hangul}', "");
Expect(0, 65501, '\P{^Is_Script_Extensions:		hangul}', "");
Error('\p{Is_Scx=__Hang/a/}');
Error('\P{Is_Scx=__Hang/a/}');
Expect(1, 65500, '\p{Is_Scx=hang}', "");
Expect(0, 65500, '\p{^Is_Scx=hang}', "");
Expect(0, 65500, '\P{Is_Scx=hang}', "");
Expect(1, 65500, '\P{^Is_Scx=hang}', "");
Expect(0, 65501, '\p{Is_Scx=hang}', "");
Expect(1, 65501, '\p{^Is_Scx=hang}', "");
Expect(1, 65501, '\P{Is_Scx=hang}', "");
Expect(0, 65501, '\P{^Is_Scx=hang}', "");
Expect(1, 65500, '\p{Is_Scx= 	HANG}', "");
Expect(0, 65500, '\p{^Is_Scx= 	HANG}', "");
Expect(0, 65500, '\P{Is_Scx= 	HANG}', "");
Expect(1, 65500, '\P{^Is_Scx= 	HANG}', "");
Expect(0, 65501, '\p{Is_Scx= 	HANG}', "");
Expect(1, 65501, '\p{^Is_Scx= 	HANG}', "");
Expect(1, 65501, '\P{Is_Scx= 	HANG}', "");
Expect(0, 65501, '\P{^Is_Scx= 	HANG}', "");
Error('\p{Script_Extensions=-/a/Deseret}');
Error('\P{Script_Extensions=-/a/Deseret}');
Expect(1, 66639, '\p{Script_Extensions=deseret}', "");
Expect(0, 66639, '\p{^Script_Extensions=deseret}', "");
Expect(0, 66639, '\P{Script_Extensions=deseret}', "");
Expect(1, 66639, '\P{^Script_Extensions=deseret}', "");
Expect(0, 66640, '\p{Script_Extensions=deseret}', "");
Expect(1, 66640, '\p{^Script_Extensions=deseret}', "");
Expect(1, 66640, '\P{Script_Extensions=deseret}', "");
Expect(0, 66640, '\P{^Script_Extensions=deseret}', "");
Expect(1, 66639, '\p{Script_Extensions= DESERET}', "");
Expect(0, 66639, '\p{^Script_Extensions= DESERET}', "");
Expect(0, 66639, '\P{Script_Extensions= DESERET}', "");
Expect(1, 66639, '\P{^Script_Extensions= DESERET}', "");
Expect(0, 66640, '\p{Script_Extensions= DESERET}', "");
Expect(1, 66640, '\p{^Script_Extensions= DESERET}', "");
Expect(1, 66640, '\P{Script_Extensions= DESERET}', "");
Expect(0, 66640, '\P{^Script_Extensions= DESERET}', "");
Error('\p{Scx= 	Dsrt/a/}');
Error('\P{Scx= 	Dsrt/a/}');
Expect(1, 66639, '\p{Scx=dsrt}', "");
Expect(0, 66639, '\p{^Scx=dsrt}', "");
Expect(0, 66639, '\P{Scx=dsrt}', "");
Expect(1, 66639, '\P{^Scx=dsrt}', "");
Expect(0, 66640, '\p{Scx=dsrt}', "");
Expect(1, 66640, '\p{^Scx=dsrt}', "");
Expect(1, 66640, '\P{Scx=dsrt}', "");
Expect(0, 66640, '\P{^Scx=dsrt}', "");
Expect(1, 66639, '\p{Scx=_DSRT}', "");
Expect(0, 66639, '\p{^Scx=_DSRT}', "");
Expect(0, 66639, '\P{Scx=_DSRT}', "");
Expect(1, 66639, '\P{^Scx=_DSRT}', "");
Expect(0, 66640, '\p{Scx=_DSRT}', "");
Expect(1, 66640, '\p{^Scx=_DSRT}', "");
Expect(1, 66640, '\P{Scx=_DSRT}', "");
Expect(0, 66640, '\P{^Scx=_DSRT}', "");
Error('\p{Is_Script_Extensions=-/a/Deseret}');
Error('\P{Is_Script_Extensions=-/a/Deseret}');
Expect(1, 66639, '\p{Is_Script_Extensions:deseret}', "");
Expect(0, 66639, '\p{^Is_Script_Extensions:deseret}', "");
Expect(0, 66639, '\P{Is_Script_Extensions:deseret}', "");
Expect(1, 66639, '\P{^Is_Script_Extensions:deseret}', "");
Expect(0, 66640, '\p{Is_Script_Extensions:deseret}', "");
Expect(1, 66640, '\p{^Is_Script_Extensions:deseret}', "");
Expect(1, 66640, '\P{Is_Script_Extensions:deseret}', "");
Expect(0, 66640, '\P{^Is_Script_Extensions:deseret}', "");
Expect(1, 66639, '\p{Is_Script_Extensions=	_deseret}', "");
Expect(0, 66639, '\p{^Is_Script_Extensions=	_deseret}', "");
Expect(0, 66639, '\P{Is_Script_Extensions=	_deseret}', "");
Expect(1, 66639, '\P{^Is_Script_Extensions=	_deseret}', "");
Expect(0, 66640, '\p{Is_Script_Extensions=	_deseret}', "");
Expect(1, 66640, '\p{^Is_Script_Extensions=	_deseret}', "");
Expect(1, 66640, '\P{Is_Script_Extensions=	_deseret}', "");
Expect(0, 66640, '\P{^Is_Script_Extensions=	_deseret}', "");
Error('\p{Is_Scx=_DSRT:=}');
Error('\P{Is_Scx=_DSRT:=}');
Expect(1, 66639, '\p{Is_Scx=dsrt}', "");
Expect(0, 66639, '\p{^Is_Scx=dsrt}', "");
Expect(0, 66639, '\P{Is_Scx=dsrt}', "");
Expect(1, 66639, '\P{^Is_Scx=dsrt}', "");
Expect(0, 66640, '\p{Is_Scx=dsrt}', "");
Expect(1, 66640, '\p{^Is_Scx=dsrt}', "");
Expect(1, 66640, '\P{Is_Scx=dsrt}', "");
Expect(0, 66640, '\P{^Is_Scx=dsrt}', "");
Expect(1, 66639, '\p{Is_Scx=-_Dsrt}', "");
Expect(0, 66639, '\p{^Is_Scx=-_Dsrt}', "");
Expect(0, 66639, '\P{Is_Scx=-_Dsrt}', "");
Expect(1, 66639, '\P{^Is_Scx=-_Dsrt}', "");
Expect(0, 66640, '\p{Is_Scx=-_Dsrt}', "");
Expect(1, 66640, '\p{^Is_Scx=-_Dsrt}', "");
Expect(1, 66640, '\P{Is_Scx=-_Dsrt}', "");
Expect(0, 66640, '\P{^Is_Scx=-_Dsrt}', "");
Error('\p{Script_Extensions=	UNKNOWN/a/}');
Error('\P{Script_Extensions=	UNKNOWN/a/}');
Expect(1, 918000, '\p{Script_Extensions=unknown}', "");
Expect(0, 918000, '\p{^Script_Extensions=unknown}', "");
Expect(0, 918000, '\P{Script_Extensions=unknown}', "");
Expect(1, 918000, '\P{^Script_Extensions=unknown}', "");
Expect(0, 917999, '\p{Script_Extensions=unknown}', "");
Expect(1, 917999, '\p{^Script_Extensions=unknown}', "");
Expect(1, 917999, '\P{Script_Extensions=unknown}', "");
Expect(0, 917999, '\P{^Script_Extensions=unknown}', "");
Expect(1, 918000, '\p{Script_Extensions=-	Unknown}', "");
Expect(0, 918000, '\p{^Script_Extensions=-	Unknown}', "");
Expect(0, 918000, '\P{Script_Extensions=-	Unknown}', "");
Expect(1, 918000, '\P{^Script_Extensions=-	Unknown}', "");
Expect(0, 917999, '\p{Script_Extensions=-	Unknown}', "");
Expect(1, 917999, '\p{^Script_Extensions=-	Unknown}', "");
Expect(1, 917999, '\P{Script_Extensions=-	Unknown}', "");
Expect(0, 917999, '\P{^Script_Extensions=-	Unknown}', "");
Error('\p{Scx=:=--ZZZZ}');
Error('\P{Scx=:=--ZZZZ}');
Expect(1, 918000, '\p{Scx=zzzz}', "");
Expect(0, 918000, '\p{^Scx=zzzz}', "");
Expect(0, 918000, '\P{Scx=zzzz}', "");
Expect(1, 918000, '\P{^Scx=zzzz}', "");
Expect(0, 917999, '\p{Scx=zzzz}', "");
Expect(1, 917999, '\p{^Scx=zzzz}', "");
Expect(1, 917999, '\P{Scx=zzzz}', "");
Expect(0, 917999, '\P{^Scx=zzzz}', "");
Expect(1, 918000, '\p{Scx=_	Zzzz}', "");
Expect(0, 918000, '\p{^Scx=_	Zzzz}', "");
Expect(0, 918000, '\P{Scx=_	Zzzz}', "");
Expect(1, 918000, '\P{^Scx=_	Zzzz}', "");
Expect(0, 917999, '\p{Scx=_	Zzzz}', "");
Expect(1, 917999, '\p{^Scx=_	Zzzz}', "");
Expect(1, 917999, '\P{Scx=_	Zzzz}', "");
Expect(0, 917999, '\P{^Scx=_	Zzzz}', "");
Error('\p{Is_Script_Extensions=:=	unknown}');
Error('\P{Is_Script_Extensions=:=	unknown}');
Expect(1, 918000, '\p{Is_Script_Extensions=unknown}', "");
Expect(0, 918000, '\p{^Is_Script_Extensions=unknown}', "");
Expect(0, 918000, '\P{Is_Script_Extensions=unknown}', "");
Expect(1, 918000, '\P{^Is_Script_Extensions=unknown}', "");
Expect(0, 917999, '\p{Is_Script_Extensions=unknown}', "");
Expect(1, 917999, '\p{^Is_Script_Extensions=unknown}', "");
Expect(1, 917999, '\P{Is_Script_Extensions=unknown}', "");
Expect(0, 917999, '\P{^Is_Script_Extensions=unknown}', "");
Expect(1, 918000, '\p{Is_Script_Extensions=_	Unknown}', "");
Expect(0, 918000, '\p{^Is_Script_Extensions=_	Unknown}', "");
Expect(0, 918000, '\P{Is_Script_Extensions=_	Unknown}', "");
Expect(1, 918000, '\P{^Is_Script_Extensions=_	Unknown}', "");
Expect(0, 917999, '\p{Is_Script_Extensions=_	Unknown}', "");
Expect(1, 917999, '\p{^Is_Script_Extensions=_	Unknown}', "");
Expect(1, 917999, '\P{Is_Script_Extensions=_	Unknown}', "");
Expect(0, 917999, '\P{^Is_Script_Extensions=_	Unknown}', "");
Error('\p{Is_Scx:   :=  Zzzz}');
Error('\P{Is_Scx:   :=  Zzzz}');
Expect(1, 918000, '\p{Is_Scx=zzzz}', "");
Expect(0, 918000, '\p{^Is_Scx=zzzz}', "");
Expect(0, 918000, '\P{Is_Scx=zzzz}', "");
Expect(1, 918000, '\P{^Is_Scx=zzzz}', "");
Expect(0, 917999, '\p{Is_Scx=zzzz}', "");
Expect(1, 917999, '\p{^Is_Scx=zzzz}', "");
Expect(1, 917999, '\P{Is_Scx=zzzz}', "");
Expect(0, 917999, '\P{^Is_Scx=zzzz}', "");
Expect(1, 918000, '\p{Is_Scx=	zzzz}', "");
Expect(0, 918000, '\p{^Is_Scx=	zzzz}', "");
Expect(0, 918000, '\P{Is_Scx=	zzzz}', "");
Expect(1, 918000, '\P{^Is_Scx=	zzzz}', "");
Expect(0, 917999, '\p{Is_Scx=	zzzz}', "");
Expect(1, 917999, '\p{^Is_Scx=	zzzz}', "");
Expect(1, 917999, '\P{Is_Scx=	zzzz}', "");
Expect(0, 917999, '\P{^Is_Scx=	zzzz}', "");
Error('\p{Script_Extensions:   := cuneiform}');
Error('\P{Script_Extensions:   := cuneiform}');
Expect(1, 74867, '\p{Script_Extensions=cuneiform}', "");
Expect(0, 74867, '\p{^Script_Extensions=cuneiform}', "");
Expect(0, 74867, '\P{Script_Extensions=cuneiform}', "");
Expect(1, 74867, '\P{^Script_Extensions=cuneiform}', "");
Expect(0, 74868, '\p{Script_Extensions=cuneiform}', "");
Expect(1, 74868, '\p{^Script_Extensions=cuneiform}', "");
Expect(1, 74868, '\P{Script_Extensions=cuneiform}', "");
Expect(0, 74868, '\P{^Script_Extensions=cuneiform}', "");
Expect(1, 74867, '\p{Script_Extensions= -Cuneiform}', "");
Expect(0, 74867, '\p{^Script_Extensions= -Cuneiform}', "");
Expect(0, 74867, '\P{Script_Extensions= -Cuneiform}', "");
Expect(1, 74867, '\P{^Script_Extensions= -Cuneiform}', "");
Expect(0, 74868, '\p{Script_Extensions= -Cuneiform}', "");
Expect(1, 74868, '\p{^Script_Extensions= -Cuneiform}', "");
Expect(1, 74868, '\P{Script_Extensions= -Cuneiform}', "");
Expect(0, 74868, '\P{^Script_Extensions= -Cuneiform}', "");
Error('\p{Scx=	/a/xsux}');
Error('\P{Scx=	/a/xsux}');
Expect(1, 74867, '\p{Scx=xsux}', "");
Expect(0, 74867, '\p{^Scx=xsux}', "");
Expect(0, 74867, '\P{Scx=xsux}', "");
Expect(1, 74867, '\P{^Scx=xsux}', "");
Expect(0, 74868, '\p{Scx=xsux}', "");
Expect(1, 74868, '\p{^Scx=xsux}', "");
Expect(1, 74868, '\P{Scx=xsux}', "");
Expect(0, 74868, '\P{^Scx=xsux}', "");
Expect(1, 74867, '\p{Scx=	 Xsux}', "");
Expect(0, 74867, '\p{^Scx=	 Xsux}', "");
Expect(0, 74867, '\P{Scx=	 Xsux}', "");
Expect(1, 74867, '\P{^Scx=	 Xsux}', "");
Expect(0, 74868, '\p{Scx=	 Xsux}', "");
Expect(1, 74868, '\p{^Scx=	 Xsux}', "");
Expect(1, 74868, '\P{Scx=	 Xsux}', "");
Expect(0, 74868, '\P{^Scx=	 Xsux}', "");
Error('\p{Is_Script_Extensions: _cuneiform:=}');
Error('\P{Is_Script_Extensions: _cuneiform:=}');
Expect(1, 74867, '\p{Is_Script_Extensions=cuneiform}', "");
Expect(0, 74867, '\p{^Is_Script_Extensions=cuneiform}', "");
Expect(0, 74867, '\P{Is_Script_Extensions=cuneiform}', "");
Expect(1, 74867, '\P{^Is_Script_Extensions=cuneiform}', "");
Expect(0, 74868, '\p{Is_Script_Extensions=cuneiform}', "");
Expect(1, 74868, '\p{^Is_Script_Extensions=cuneiform}', "");
Expect(1, 74868, '\P{Is_Script_Extensions=cuneiform}', "");
Expect(0, 74868, '\P{^Is_Script_Extensions=cuneiform}', "");
Expect(1, 74867, '\p{Is_Script_Extensions=	_CUNEIFORM}', "");
Expect(0, 74867, '\p{^Is_Script_Extensions=	_CUNEIFORM}', "");
Expect(0, 74867, '\P{Is_Script_Extensions=	_CUNEIFORM}', "");
Expect(1, 74867, '\P{^Is_Script_Extensions=	_CUNEIFORM}', "");
Expect(0, 74868, '\p{Is_Script_Extensions=	_CUNEIFORM}', "");
Expect(1, 74868, '\p{^Is_Script_Extensions=	_CUNEIFORM}', "");
Expect(1, 74868, '\P{Is_Script_Extensions=	_CUNEIFORM}', "");
Expect(0, 74868, '\P{^Is_Script_Extensions=	_CUNEIFORM}', "");
Error('\p{Is_Scx=Xsux:=}');
Error('\P{Is_Scx=Xsux:=}');
Expect(1, 74867, '\p{Is_Scx:xsux}', "");
Expect(0, 74867, '\p{^Is_Scx:xsux}', "");
Expect(0, 74867, '\P{Is_Scx:xsux}', "");
Expect(1, 74867, '\P{^Is_Scx:xsux}', "");
Expect(0, 74868, '\p{Is_Scx:xsux}', "");
Expect(1, 74868, '\p{^Is_Scx:xsux}', "");
Expect(1, 74868, '\P{Is_Scx:xsux}', "");
Expect(0, 74868, '\P{^Is_Scx:xsux}', "");
Expect(1, 74867, '\p{Is_Scx=-xsux}', "");
Expect(0, 74867, '\p{^Is_Scx=-xsux}', "");
Expect(0, 74867, '\P{Is_Scx=-xsux}', "");
Expect(1, 74867, '\P{^Is_Scx=-xsux}', "");
Expect(0, 74868, '\p{Is_Scx=-xsux}', "");
Expect(1, 74868, '\p{^Is_Scx=-xsux}', "");
Expect(1, 74868, '\P{Is_Scx=-xsux}', "");
Expect(0, 74868, '\P{^Is_Scx=-xsux}', "");
Error('\p{Script_Extensions=/a/ -inherited}');
Error('\P{Script_Extensions=/a/ -inherited}');
Expect(1, 917999, '\p{Script_Extensions=inherited}', "");
Expect(0, 917999, '\p{^Script_Extensions=inherited}', "");
Expect(0, 917999, '\P{Script_Extensions=inherited}', "");
Expect(1, 917999, '\P{^Script_Extensions=inherited}', "");
Expect(0, 918000, '\p{Script_Extensions=inherited}', "");
Expect(1, 918000, '\p{^Script_Extensions=inherited}', "");
Expect(1, 918000, '\P{Script_Extensions=inherited}', "");
Expect(0, 918000, '\P{^Script_Extensions=inherited}', "");
Expect(1, 917999, '\p{Script_Extensions=		INHERITED}', "");
Expect(0, 917999, '\p{^Script_Extensions=		INHERITED}', "");
Expect(0, 917999, '\P{Script_Extensions=		INHERITED}', "");
Expect(1, 917999, '\P{^Script_Extensions=		INHERITED}', "");
Expect(0, 918000, '\p{Script_Extensions=		INHERITED}', "");
Expect(1, 918000, '\p{^Script_Extensions=		INHERITED}', "");
Expect(1, 918000, '\P{Script_Extensions=		INHERITED}', "");
Expect(0, 918000, '\P{^Script_Extensions=		INHERITED}', "");
Error('\p{Scx=:=  ZINH}');
Error('\P{Scx=:=  ZINH}');
Expect(1, 917999, '\p{Scx: zinh}', "");
Expect(0, 917999, '\p{^Scx: zinh}', "");
Expect(0, 917999, '\P{Scx: zinh}', "");
Expect(1, 917999, '\P{^Scx: zinh}', "");
Expect(0, 918000, '\p{Scx: zinh}', "");
Expect(1, 918000, '\p{^Scx: zinh}', "");
Expect(1, 918000, '\P{Scx: zinh}', "");
Expect(0, 918000, '\P{^Scx: zinh}', "");
Expect(1, 917999, '\p{Scx=	_Zinh}', "");
Expect(0, 917999, '\p{^Scx=	_Zinh}', "");
Expect(0, 917999, '\P{Scx=	_Zinh}', "");
Expect(1, 917999, '\P{^Scx=	_Zinh}', "");
Expect(0, 918000, '\p{Scx=	_Zinh}', "");
Expect(1, 918000, '\p{^Scx=	_Zinh}', "");
Expect(1, 918000, '\P{Scx=	_Zinh}', "");
Expect(0, 918000, '\P{^Scx=	_Zinh}', "");
Error('\p{Is_Script_Extensions= :=QAAI}');
Error('\P{Is_Script_Extensions= :=QAAI}');
Expect(1, 917999, '\p{Is_Script_Extensions=qaai}', "");
Expect(0, 917999, '\p{^Is_Script_Extensions=qaai}', "");
Expect(0, 917999, '\P{Is_Script_Extensions=qaai}', "");
Expect(1, 917999, '\P{^Is_Script_Extensions=qaai}', "");
Expect(0, 918000, '\p{Is_Script_Extensions=qaai}', "");
Expect(1, 918000, '\p{^Is_Script_Extensions=qaai}', "");
Expect(1, 918000, '\P{Is_Script_Extensions=qaai}', "");
Expect(0, 918000, '\P{^Is_Script_Extensions=qaai}', "");
Expect(1, 917999, '\p{Is_Script_Extensions=_-QAAI}', "");
Expect(0, 917999, '\p{^Is_Script_Extensions=_-QAAI}', "");
Expect(0, 917999, '\P{Is_Script_Extensions=_-QAAI}', "");
Expect(1, 917999, '\P{^Is_Script_Extensions=_-QAAI}', "");
Expect(0, 918000, '\p{Is_Script_Extensions=_-QAAI}', "");
Expect(1, 918000, '\p{^Is_Script_Extensions=_-QAAI}', "");
Expect(1, 918000, '\P{Is_Script_Extensions=_-QAAI}', "");
Expect(0, 918000, '\P{^Is_Script_Extensions=_-QAAI}', "");
Error('\p{Is_Scx=:=	Inherited}');
Error('\P{Is_Scx=:=	Inherited}');
Expect(1, 917999, '\p{Is_Scx:   inherited}', "");
Expect(0, 917999, '\p{^Is_Scx:   inherited}', "");
Expect(0, 917999, '\P{Is_Scx:   inherited}', "");
Expect(1, 917999, '\P{^Is_Scx:   inherited}', "");
Expect(0, 918000, '\p{Is_Scx:   inherited}', "");
Expect(1, 918000, '\p{^Is_Scx:   inherited}', "");
Expect(1, 918000, '\P{Is_Scx:   inherited}', "");
Expect(0, 918000, '\P{^Is_Scx:   inherited}', "");
Expect(1, 917999, '\p{Is_Scx:	__Inherited}', "");
Expect(0, 917999, '\p{^Is_Scx:	__Inherited}', "");
Expect(0, 917999, '\P{Is_Scx:	__Inherited}', "");
Expect(1, 917999, '\P{^Is_Scx:	__Inherited}', "");
Expect(0, 918000, '\p{Is_Scx:	__Inherited}', "");
Expect(1, 918000, '\p{^Is_Scx:	__Inherited}', "");
Expect(1, 918000, '\P{Is_Scx:	__Inherited}', "");
Expect(0, 918000, '\P{^Is_Scx:	__Inherited}', "");
Error('\p{Script_Extensions= :=Kaithi}');
Error('\P{Script_Extensions= :=Kaithi}');
Expect(1, 69825, '\p{Script_Extensions=kaithi}', "");
Expect(0, 69825, '\p{^Script_Extensions=kaithi}', "");
Expect(0, 69825, '\P{Script_Extensions=kaithi}', "");
Expect(1, 69825, '\P{^Script_Extensions=kaithi}', "");
Expect(0, 69826, '\p{Script_Extensions=kaithi}', "");
Expect(1, 69826, '\p{^Script_Extensions=kaithi}', "");
Expect(1, 69826, '\P{Script_Extensions=kaithi}', "");
Expect(0, 69826, '\P{^Script_Extensions=kaithi}', "");
Expect(1, 69825, '\p{Script_Extensions=  Kaithi}', "");
Expect(0, 69825, '\p{^Script_Extensions=  Kaithi}', "");
Expect(0, 69825, '\P{Script_Extensions=  Kaithi}', "");
Expect(1, 69825, '\P{^Script_Extensions=  Kaithi}', "");
Expect(0, 69826, '\p{Script_Extensions=  Kaithi}', "");
Expect(1, 69826, '\p{^Script_Extensions=  Kaithi}', "");
Expect(1, 69826, '\P{Script_Extensions=  Kaithi}', "");
Expect(0, 69826, '\P{^Script_Extensions=  Kaithi}', "");
Error('\p{Scx=__KTHI:=}');
Error('\P{Scx=__KTHI:=}');
Expect(1, 69825, '\p{Scx:	kthi}', "");
Expect(0, 69825, '\p{^Scx:	kthi}', "");
Expect(0, 69825, '\P{Scx:	kthi}', "");
Expect(1, 69825, '\P{^Scx:	kthi}', "");
Expect(0, 69826, '\p{Scx:	kthi}', "");
Expect(1, 69826, '\p{^Scx:	kthi}', "");
Expect(1, 69826, '\P{Scx:	kthi}', "");
Expect(0, 69826, '\P{^Scx:	kthi}', "");
Expect(1, 69825, '\p{Scx: -	kthi}', "");
Expect(0, 69825, '\p{^Scx: -	kthi}', "");
Expect(0, 69825, '\P{Scx: -	kthi}', "");
Expect(1, 69825, '\P{^Scx: -	kthi}', "");
Expect(0, 69826, '\p{Scx: -	kthi}', "");
Expect(1, 69826, '\p{^Scx: -	kthi}', "");
Expect(1, 69826, '\P{Scx: -	kthi}', "");
Expect(0, 69826, '\P{^Scx: -	kthi}', "");
Error('\p{Is_Script_Extensions=/a/_-Kaithi}');
Error('\P{Is_Script_Extensions=/a/_-Kaithi}');
Expect(1, 69825, '\p{Is_Script_Extensions=kaithi}', "");
Expect(0, 69825, '\p{^Is_Script_Extensions=kaithi}', "");
Expect(0, 69825, '\P{Is_Script_Extensions=kaithi}', "");
Expect(1, 69825, '\P{^Is_Script_Extensions=kaithi}', "");
Expect(0, 69826, '\p{Is_Script_Extensions=kaithi}', "");
Expect(1, 69826, '\p{^Is_Script_Extensions=kaithi}', "");
Expect(1, 69826, '\P{Is_Script_Extensions=kaithi}', "");
Expect(0, 69826, '\P{^Is_Script_Extensions=kaithi}', "");
Expect(1, 69825, '\p{Is_Script_Extensions=__Kaithi}', "");
Expect(0, 69825, '\p{^Is_Script_Extensions=__Kaithi}', "");
Expect(0, 69825, '\P{Is_Script_Extensions=__Kaithi}', "");
Expect(1, 69825, '\P{^Is_Script_Extensions=__Kaithi}', "");
Expect(0, 69826, '\p{Is_Script_Extensions=__Kaithi}', "");
Expect(1, 69826, '\p{^Is_Script_Extensions=__Kaithi}', "");
Expect(1, 69826, '\P{Is_Script_Extensions=__Kaithi}', "");
Expect(0, 69826, '\P{^Is_Script_Extensions=__Kaithi}', "");
Error('\p{Is_Scx=  KTHI:=}');
Error('\P{Is_Scx=  KTHI:=}');
Expect(1, 69825, '\p{Is_Scx=kthi}', "");
Expect(0, 69825, '\p{^Is_Scx=kthi}', "");
Expect(0, 69825, '\P{Is_Scx=kthi}', "");
Expect(1, 69825, '\P{^Is_Scx=kthi}', "");
Expect(0, 69826, '\p{Is_Scx=kthi}', "");
Expect(1, 69826, '\p{^Is_Scx=kthi}', "");
Expect(1, 69826, '\P{Is_Scx=kthi}', "");
Expect(0, 69826, '\P{^Is_Scx=kthi}', "");
Expect(1, 69825, '\p{Is_Scx=	kthi}', "");
Expect(0, 69825, '\p{^Is_Scx=	kthi}', "");
Expect(0, 69825, '\P{Is_Scx=	kthi}', "");
Expect(1, 69825, '\P{^Is_Scx=	kthi}', "");
Expect(0, 69826, '\p{Is_Scx=	kthi}', "");
Expect(1, 69826, '\p{^Is_Scx=	kthi}', "");
Expect(1, 69826, '\P{Is_Scx=	kthi}', "");
Expect(0, 69826, '\P{^Is_Scx=	kthi}', "");
Error('\p{Script_Extensions= /a/ARABIC}');
Error('\P{Script_Extensions= /a/ARABIC}');
Expect(1, 126705, '\p{Script_Extensions=arabic}', "");
Expect(0, 126705, '\p{^Script_Extensions=arabic}', "");
Expect(0, 126705, '\P{Script_Extensions=arabic}', "");
Expect(1, 126705, '\P{^Script_Extensions=arabic}', "");
Expect(0, 126706, '\p{Script_Extensions=arabic}', "");
Expect(1, 126706, '\p{^Script_Extensions=arabic}', "");
Expect(1, 126706, '\P{Script_Extensions=arabic}', "");
Expect(0, 126706, '\P{^Script_Extensions=arabic}', "");
Expect(1, 126705, '\p{Script_Extensions=-_Arabic}', "");
Expect(0, 126705, '\p{^Script_Extensions=-_Arabic}', "");
Expect(0, 126705, '\P{Script_Extensions=-_Arabic}', "");
Expect(1, 126705, '\P{^Script_Extensions=-_Arabic}', "");
Expect(0, 126706, '\p{Script_Extensions=-_Arabic}', "");
Expect(1, 126706, '\p{^Script_Extensions=-_Arabic}', "");
Expect(1, 126706, '\P{Script_Extensions=-_Arabic}', "");
Expect(0, 126706, '\P{^Script_Extensions=-_Arabic}', "");
Error('\p{Scx=:= 	ARAB}');
Error('\P{Scx=:= 	ARAB}');
Expect(1, 126705, '\p{Scx=arab}', "");
Expect(0, 126705, '\p{^Scx=arab}', "");
Expect(0, 126705, '\P{Scx=arab}', "");
Expect(1, 126705, '\P{^Scx=arab}', "");
Expect(0, 126706, '\p{Scx=arab}', "");
Expect(1, 126706, '\p{^Scx=arab}', "");
Expect(1, 126706, '\P{Scx=arab}', "");
Expect(0, 126706, '\P{^Scx=arab}', "");
Expect(1, 126705, '\p{Scx=--Arab}', "");
Expect(0, 126705, '\p{^Scx=--Arab}', "");
Expect(0, 126705, '\P{Scx=--Arab}', "");
Expect(1, 126705, '\P{^Scx=--Arab}', "");
Expect(0, 126706, '\p{Scx=--Arab}', "");
Expect(1, 126706, '\p{^Scx=--Arab}', "");
Expect(1, 126706, '\P{Scx=--Arab}', "");
Expect(0, 126706, '\P{^Scx=--Arab}', "");
Error('\p{Is_Script_Extensions=_ arabic:=}');
Error('\P{Is_Script_Extensions=_ arabic:=}');
Expect(1, 126705, '\p{Is_Script_Extensions=arabic}', "");
Expect(0, 126705, '\p{^Is_Script_Extensions=arabic}', "");
Expect(0, 126705, '\P{Is_Script_Extensions=arabic}', "");
Expect(1, 126705, '\P{^Is_Script_Extensions=arabic}', "");
Expect(0, 126706, '\p{Is_Script_Extensions=arabic}', "");
Expect(1, 126706, '\p{^Is_Script_Extensions=arabic}', "");
Expect(1, 126706, '\P{Is_Script_Extensions=arabic}', "");
Expect(0, 126706, '\P{^Is_Script_Extensions=arabic}', "");
Expect(1, 126705, '\p{Is_Script_Extensions: 	 Arabic}', "");
Expect(0, 126705, '\p{^Is_Script_Extensions: 	 Arabic}', "");
Expect(0, 126705, '\P{Is_Script_Extensions: 	 Arabic}', "");
Expect(1, 126705, '\P{^Is_Script_Extensions: 	 Arabic}', "");
Expect(0, 126706, '\p{Is_Script_Extensions: 	 Arabic}', "");
Expect(1, 126706, '\p{^Is_Script_Extensions: 	 Arabic}', "");
Expect(1, 126706, '\P{Is_Script_Extensions: 	 Arabic}', "");
Expect(0, 126706, '\P{^Is_Script_Extensions: 	 Arabic}', "");
Error('\p{Is_Scx= 	ARAB:=}');
Error('\P{Is_Scx= 	ARAB:=}');
Expect(1, 126705, '\p{Is_Scx: arab}', "");
Expect(0, 126705, '\p{^Is_Scx: arab}', "");
Expect(0, 126705, '\P{Is_Scx: arab}', "");
Expect(1, 126705, '\P{^Is_Scx: arab}', "");
Expect(0, 126706, '\p{Is_Scx: arab}', "");
Expect(1, 126706, '\p{^Is_Scx: arab}', "");
Expect(1, 126706, '\P{Is_Scx: arab}', "");
Expect(0, 126706, '\P{^Is_Scx: arab}', "");
Expect(1, 126705, '\p{Is_Scx: _ARAB}', "");
Expect(0, 126705, '\p{^Is_Scx: _ARAB}', "");
Expect(0, 126705, '\P{Is_Scx: _ARAB}', "");
Expect(1, 126705, '\P{^Is_Scx: _ARAB}', "");
Expect(0, 126706, '\p{Is_Scx: _ARAB}', "");
Expect(1, 126706, '\p{^Is_Scx: _ARAB}', "");
Expect(1, 126706, '\P{Is_Scx: _ARAB}', "");
Expect(0, 126706, '\P{^Is_Scx: _ARAB}', "");
Error('\p{Script_Extensions=/a/__Canadian_Aboriginal}');
Error('\P{Script_Extensions=/a/__Canadian_Aboriginal}');
Expect(1, 6389, '\p{Script_Extensions=canadianaboriginal}', "");
Expect(0, 6389, '\p{^Script_Extensions=canadianaboriginal}', "");
Expect(0, 6389, '\P{Script_Extensions=canadianaboriginal}', "");
Expect(1, 6389, '\P{^Script_Extensions=canadianaboriginal}', "");
Expect(0, 6390, '\p{Script_Extensions=canadianaboriginal}', "");
Expect(1, 6390, '\p{^Script_Extensions=canadianaboriginal}', "");
Expect(1, 6390, '\P{Script_Extensions=canadianaboriginal}', "");
Expect(0, 6390, '\P{^Script_Extensions=canadianaboriginal}', "");
Expect(1, 6389, '\p{Script_Extensions=	Canadian_ABORIGINAL}', "");
Expect(0, 6389, '\p{^Script_Extensions=	Canadian_ABORIGINAL}', "");
Expect(0, 6389, '\P{Script_Extensions=	Canadian_ABORIGINAL}', "");
Expect(1, 6389, '\P{^Script_Extensions=	Canadian_ABORIGINAL}', "");
Expect(0, 6390, '\p{Script_Extensions=	Canadian_ABORIGINAL}', "");
Expect(1, 6390, '\p{^Script_Extensions=	Canadian_ABORIGINAL}', "");
Expect(1, 6390, '\P{Script_Extensions=	Canadian_ABORIGINAL}', "");
Expect(0, 6390, '\P{^Script_Extensions=	Canadian_ABORIGINAL}', "");
Error('\p{Scx=	/a/Cans}');
Error('\P{Scx=	/a/Cans}');
Expect(1, 6389, '\p{Scx=cans}', "");
Expect(0, 6389, '\p{^Scx=cans}', "");
Expect(0, 6389, '\P{Scx=cans}', "");
Expect(1, 6389, '\P{^Scx=cans}', "");
Expect(0, 6390, '\p{Scx=cans}', "");
Expect(1, 6390, '\p{^Scx=cans}', "");
Expect(1, 6390, '\P{Scx=cans}', "");
Expect(0, 6390, '\P{^Scx=cans}', "");
Expect(1, 6389, '\p{Scx=	-cans}', "");
Expect(0, 6389, '\p{^Scx=	-cans}', "");
Expect(0, 6389, '\P{Scx=	-cans}', "");
Expect(1, 6389, '\P{^Scx=	-cans}', "");
Expect(0, 6390, '\p{Scx=	-cans}', "");
Expect(1, 6390, '\p{^Scx=	-cans}', "");
Expect(1, 6390, '\P{Scx=	-cans}', "");
Expect(0, 6390, '\P{^Scx=	-cans}', "");
Error('\p{Is_Script_Extensions=_Canadian_ABORIGINAL:=}');
Error('\P{Is_Script_Extensions=_Canadian_ABORIGINAL:=}');
Expect(1, 6389, '\p{Is_Script_Extensions=canadianaboriginal}', "");
Expect(0, 6389, '\p{^Is_Script_Extensions=canadianaboriginal}', "");
Expect(0, 6389, '\P{Is_Script_Extensions=canadianaboriginal}', "");
Expect(1, 6389, '\P{^Is_Script_Extensions=canadianaboriginal}', "");
Expect(0, 6390, '\p{Is_Script_Extensions=canadianaboriginal}', "");
Expect(1, 6390, '\p{^Is_Script_Extensions=canadianaboriginal}', "");
Expect(1, 6390, '\P{Is_Script_Extensions=canadianaboriginal}', "");
Expect(0, 6390, '\P{^Is_Script_Extensions=canadianaboriginal}', "");
Expect(1, 6389, '\p{Is_Script_Extensions=	_Canadian_ABORIGINAL}', "");
Expect(0, 6389, '\p{^Is_Script_Extensions=	_Canadian_ABORIGINAL}', "");
Expect(0, 6389, '\P{Is_Script_Extensions=	_Canadian_ABORIGINAL}', "");
Expect(1, 6389, '\P{^Is_Script_Extensions=	_Canadian_ABORIGINAL}', "");
Expect(0, 6390, '\p{Is_Script_Extensions=	_Canadian_ABORIGINAL}', "");
Expect(1, 6390, '\p{^Is_Script_Extensions=	_Canadian_ABORIGINAL}', "");
Expect(1, 6390, '\P{Is_Script_Extensions=	_Canadian_ABORIGINAL}', "");
Expect(0, 6390, '\P{^Is_Script_Extensions=	_Canadian_ABORIGINAL}', "");
Error('\p{Is_Scx=/a/  Cans}');
Error('\P{Is_Scx=/a/  Cans}');
Expect(1, 6389, '\p{Is_Scx:cans}', "");
Expect(0, 6389, '\p{^Is_Scx:cans}', "");
Expect(0, 6389, '\P{Is_Scx:cans}', "");
Expect(1, 6389, '\P{^Is_Scx:cans}', "");
Expect(0, 6390, '\p{Is_Scx:cans}', "");
Expect(1, 6390, '\p{^Is_Scx:cans}', "");
Expect(1, 6390, '\P{Is_Scx:cans}', "");
Expect(0, 6390, '\P{^Is_Scx:cans}', "");
Expect(1, 6389, '\p{Is_Scx=_Cans}', "");
Expect(0, 6389, '\p{^Is_Scx=_Cans}', "");
Expect(0, 6389, '\P{Is_Scx=_Cans}', "");
Expect(1, 6389, '\P{^Is_Scx=_Cans}', "");
Expect(0, 6390, '\p{Is_Scx=_Cans}', "");
Expect(1, 6390, '\p{^Is_Scx=_Cans}', "");
Expect(1, 6390, '\P{Is_Scx=_Cans}', "");
Expect(0, 6390, '\P{^Is_Scx=_Cans}', "");
Error('\p{Script_Extensions= :=Shavian}');
Error('\P{Script_Extensions= :=Shavian}');
Expect(1, 66687, '\p{Script_Extensions=shavian}', "");
Expect(0, 66687, '\p{^Script_Extensions=shavian}', "");
Expect(0, 66687, '\P{Script_Extensions=shavian}', "");
Expect(1, 66687, '\P{^Script_Extensions=shavian}', "");
Expect(0, 66688, '\p{Script_Extensions=shavian}', "");
Expect(1, 66688, '\p{^Script_Extensions=shavian}', "");
Expect(1, 66688, '\P{Script_Extensions=shavian}', "");
Expect(0, 66688, '\P{^Script_Extensions=shavian}', "");
Expect(1, 66687, '\p{Script_Extensions: _ SHAVIAN}', "");
Expect(0, 66687, '\p{^Script_Extensions: _ SHAVIAN}', "");
Expect(0, 66687, '\P{Script_Extensions: _ SHAVIAN}', "");
Expect(1, 66687, '\P{^Script_Extensions: _ SHAVIAN}', "");
Expect(0, 66688, '\p{Script_Extensions: _ SHAVIAN}', "");
Expect(1, 66688, '\p{^Script_Extensions: _ SHAVIAN}', "");
Expect(1, 66688, '\P{Script_Extensions: _ SHAVIAN}', "");
Expect(0, 66688, '\P{^Script_Extensions: _ SHAVIAN}', "");
Error('\p{Scx= Shaw:=}');
Error('\P{Scx= Shaw:=}');
Expect(1, 66687, '\p{Scx=shaw}', "");
Expect(0, 66687, '\p{^Scx=shaw}', "");
Expect(0, 66687, '\P{Scx=shaw}', "");
Expect(1, 66687, '\P{^Scx=shaw}', "");
Expect(0, 66688, '\p{Scx=shaw}', "");
Expect(1, 66688, '\p{^Scx=shaw}', "");
Expect(1, 66688, '\P{Scx=shaw}', "");
Expect(0, 66688, '\P{^Scx=shaw}', "");
Expect(1, 66687, '\p{Scx=-Shaw}', "");
Expect(0, 66687, '\p{^Scx=-Shaw}', "");
Expect(0, 66687, '\P{Scx=-Shaw}', "");
Expect(1, 66687, '\P{^Scx=-Shaw}', "");
Expect(0, 66688, '\p{Scx=-Shaw}', "");
Expect(1, 66688, '\p{^Scx=-Shaw}', "");
Expect(1, 66688, '\P{Scx=-Shaw}', "");
Expect(0, 66688, '\P{^Scx=-Shaw}', "");
Error('\p{Is_Script_Extensions=/a/-	shavian}');
Error('\P{Is_Script_Extensions=/a/-	shavian}');
Expect(1, 66687, '\p{Is_Script_Extensions=shavian}', "");
Expect(0, 66687, '\p{^Is_Script_Extensions=shavian}', "");
Expect(0, 66687, '\P{Is_Script_Extensions=shavian}', "");
Expect(1, 66687, '\P{^Is_Script_Extensions=shavian}', "");
Expect(0, 66688, '\p{Is_Script_Extensions=shavian}', "");
Expect(1, 66688, '\p{^Is_Script_Extensions=shavian}', "");
Expect(1, 66688, '\P{Is_Script_Extensions=shavian}', "");
Expect(0, 66688, '\P{^Is_Script_Extensions=shavian}', "");
Expect(1, 66687, '\p{Is_Script_Extensions=--shavian}', "");
Expect(0, 66687, '\p{^Is_Script_Extensions=--shavian}', "");
Expect(0, 66687, '\P{Is_Script_Extensions=--shavian}', "");
Expect(1, 66687, '\P{^Is_Script_Extensions=--shavian}', "");
Expect(0, 66688, '\p{Is_Script_Extensions=--shavian}', "");
Expect(1, 66688, '\p{^Is_Script_Extensions=--shavian}', "");
Expect(1, 66688, '\P{Is_Script_Extensions=--shavian}', "");
Expect(0, 66688, '\P{^Is_Script_Extensions=--shavian}', "");
Error('\p{Is_Scx=- shaw/a/}');
Error('\P{Is_Scx=- shaw/a/}');
Expect(1, 66687, '\p{Is_Scx: shaw}', "");
Expect(0, 66687, '\p{^Is_Scx: shaw}', "");
Expect(0, 66687, '\P{Is_Scx: shaw}', "");
Expect(1, 66687, '\P{^Is_Scx: shaw}', "");
Expect(0, 66688, '\p{Is_Scx: shaw}', "");
Expect(1, 66688, '\p{^Is_Scx: shaw}', "");
Expect(1, 66688, '\P{Is_Scx: shaw}', "");
Expect(0, 66688, '\P{^Is_Scx: shaw}', "");
Expect(1, 66687, '\p{Is_Scx=_Shaw}', "");
Expect(0, 66687, '\p{^Is_Scx=_Shaw}', "");
Expect(0, 66687, '\P{Is_Scx=_Shaw}', "");
Expect(1, 66687, '\P{^Is_Scx=_Shaw}', "");
Expect(0, 66688, '\p{Is_Scx=_Shaw}', "");
Expect(1, 66688, '\p{^Is_Scx=_Shaw}', "");
Expect(1, 66688, '\P{Is_Scx=_Shaw}', "");
Expect(0, 66688, '\P{^Is_Scx=_Shaw}', "");
Error('\p{Script_Extensions=:= Old_South_Arabian}');
Error('\P{Script_Extensions=:= Old_South_Arabian}');
Expect(1, 68223, '\p{Script_Extensions: oldsoutharabian}', "");
Expect(0, 68223, '\p{^Script_Extensions: oldsoutharabian}', "");
Expect(0, 68223, '\P{Script_Extensions: oldsoutharabian}', "");
Expect(1, 68223, '\P{^Script_Extensions: oldsoutharabian}', "");
Expect(0, 68224, '\p{Script_Extensions: oldsoutharabian}', "");
Expect(1, 68224, '\p{^Script_Extensions: oldsoutharabian}', "");
Expect(1, 68224, '\P{Script_Extensions: oldsoutharabian}', "");
Expect(0, 68224, '\P{^Script_Extensions: oldsoutharabian}', "");
Expect(1, 68223, '\p{Script_Extensions=	old_South_Arabian}', "");
Expect(0, 68223, '\p{^Script_Extensions=	old_South_Arabian}', "");
Expect(0, 68223, '\P{Script_Extensions=	old_South_Arabian}', "");
Expect(1, 68223, '\P{^Script_Extensions=	old_South_Arabian}', "");
Expect(0, 68224, '\p{Script_Extensions=	old_South_Arabian}', "");
Expect(1, 68224, '\p{^Script_Extensions=	old_South_Arabian}', "");
Expect(1, 68224, '\P{Script_Extensions=	old_South_Arabian}', "");
Expect(0, 68224, '\P{^Script_Extensions=	old_South_Arabian}', "");
Error('\p{Scx=:=sarb}');
Error('\P{Scx=:=sarb}');
Expect(1, 68223, '\p{Scx=sarb}', "");
Expect(0, 68223, '\p{^Scx=sarb}', "");
Expect(0, 68223, '\P{Scx=sarb}', "");
Expect(1, 68223, '\P{^Scx=sarb}', "");
Expect(0, 68224, '\p{Scx=sarb}', "");
Expect(1, 68224, '\p{^Scx=sarb}', "");
Expect(1, 68224, '\P{Scx=sarb}', "");
Expect(0, 68224, '\P{^Scx=sarb}', "");
Expect(1, 68223, '\p{Scx=-_Sarb}', "");
Expect(0, 68223, '\p{^Scx=-_Sarb}', "");
Expect(0, 68223, '\P{Scx=-_Sarb}', "");
Expect(1, 68223, '\P{^Scx=-_Sarb}', "");
Expect(0, 68224, '\p{Scx=-_Sarb}', "");
Expect(1, 68224, '\p{^Scx=-_Sarb}', "");
Expect(1, 68224, '\P{Scx=-_Sarb}', "");
Expect(0, 68224, '\P{^Scx=-_Sarb}', "");
Error('\p{Is_Script_Extensions=_ OLD_South_arabian:=}');
Error('\P{Is_Script_Extensions=_ OLD_South_arabian:=}');
Expect(1, 68223, '\p{Is_Script_Extensions=oldsoutharabian}', "");
Expect(0, 68223, '\p{^Is_Script_Extensions=oldsoutharabian}', "");
Expect(0, 68223, '\P{Is_Script_Extensions=oldsoutharabian}', "");
Expect(1, 68223, '\P{^Is_Script_Extensions=oldsoutharabian}', "");
Expect(0, 68224, '\p{Is_Script_Extensions=oldsoutharabian}', "");
Expect(1, 68224, '\p{^Is_Script_Extensions=oldsoutharabian}', "");
Expect(1, 68224, '\P{Is_Script_Extensions=oldsoutharabian}', "");
Expect(0, 68224, '\P{^Is_Script_Extensions=oldsoutharabian}', "");
Expect(1, 68223, '\p{Is_Script_Extensions=-old_south_arabian}', "");
Expect(0, 68223, '\p{^Is_Script_Extensions=-old_south_arabian}', "");
Expect(0, 68223, '\P{Is_Script_Extensions=-old_south_arabian}', "");
Expect(1, 68223, '\P{^Is_Script_Extensions=-old_south_arabian}', "");
Expect(0, 68224, '\p{Is_Script_Extensions=-old_south_arabian}', "");
Expect(1, 68224, '\p{^Is_Script_Extensions=-old_south_arabian}', "");
Expect(1, 68224, '\P{Is_Script_Extensions=-old_south_arabian}', "");
Expect(0, 68224, '\P{^Is_Script_Extensions=-old_south_arabian}', "");
Error('\p{Is_Scx=_-Sarb/a/}');
Error('\P{Is_Scx=_-Sarb/a/}');
Expect(1, 68223, '\p{Is_Scx=sarb}', "");
Expect(0, 68223, '\p{^Is_Scx=sarb}', "");
Expect(0, 68223, '\P{Is_Scx=sarb}', "");
Expect(1, 68223, '\P{^Is_Scx=sarb}', "");
Expect(0, 68224, '\p{Is_Scx=sarb}', "");
Expect(1, 68224, '\p{^Is_Scx=sarb}', "");
Expect(1, 68224, '\P{Is_Scx=sarb}', "");
Expect(0, 68224, '\P{^Is_Scx=sarb}', "");
Expect(1, 68223, '\p{Is_Scx=-SARB}', "");
Expect(0, 68223, '\p{^Is_Scx=-SARB}', "");
Expect(0, 68223, '\P{Is_Scx=-SARB}', "");
Expect(1, 68223, '\P{^Is_Scx=-SARB}', "");
Expect(0, 68224, '\p{Is_Scx=-SARB}', "");
Expect(1, 68224, '\p{^Is_Scx=-SARB}', "");
Expect(1, 68224, '\P{Is_Scx=-SARB}', "");
Expect(0, 68224, '\P{^Is_Scx=-SARB}', "");
Error('\p{Script_Extensions=	/a/Lydian}');
Error('\P{Script_Extensions=	/a/Lydian}');
Expect(1, 67903, '\p{Script_Extensions=lydian}', "");
Expect(0, 67903, '\p{^Script_Extensions=lydian}', "");
Expect(0, 67903, '\P{Script_Extensions=lydian}', "");
Expect(1, 67903, '\P{^Script_Extensions=lydian}', "");
Expect(0, 67904, '\p{Script_Extensions=lydian}', "");
Expect(1, 67904, '\p{^Script_Extensions=lydian}', "");
Expect(1, 67904, '\P{Script_Extensions=lydian}', "");
Expect(0, 67904, '\P{^Script_Extensions=lydian}', "");
Expect(1, 67903, '\p{Script_Extensions=-	Lydian}', "");
Expect(0, 67903, '\p{^Script_Extensions=-	Lydian}', "");
Expect(0, 67903, '\P{Script_Extensions=-	Lydian}', "");
Expect(1, 67903, '\P{^Script_Extensions=-	Lydian}', "");
Expect(0, 67904, '\p{Script_Extensions=-	Lydian}', "");
Expect(1, 67904, '\p{^Script_Extensions=-	Lydian}', "");
Expect(1, 67904, '\P{Script_Extensions=-	Lydian}', "");
Expect(0, 67904, '\P{^Script_Extensions=-	Lydian}', "");
Error('\p{Scx: := -Lydi}');
Error('\P{Scx: := -Lydi}');
Expect(1, 67903, '\p{Scx=lydi}', "");
Expect(0, 67903, '\p{^Scx=lydi}', "");
Expect(0, 67903, '\P{Scx=lydi}', "");
Expect(1, 67903, '\P{^Scx=lydi}', "");
Expect(0, 67904, '\p{Scx=lydi}', "");
Expect(1, 67904, '\p{^Scx=lydi}', "");
Expect(1, 67904, '\P{Scx=lydi}', "");
Expect(0, 67904, '\P{^Scx=lydi}', "");
Expect(1, 67903, '\p{Scx= _Lydi}', "");
Expect(0, 67903, '\p{^Scx= _Lydi}', "");
Expect(0, 67903, '\P{Scx= _Lydi}', "");
Expect(1, 67903, '\P{^Scx= _Lydi}', "");
Expect(0, 67904, '\p{Scx= _Lydi}', "");
Expect(1, 67904, '\p{^Scx= _Lydi}', "");
Expect(1, 67904, '\P{Scx= _Lydi}', "");
Expect(0, 67904, '\P{^Scx= _Lydi}', "");
Error('\p{Is_Script_Extensions= /a/Lydian}');
Error('\P{Is_Script_Extensions= /a/Lydian}');
Expect(1, 67903, '\p{Is_Script_Extensions=lydian}', "");
Expect(0, 67903, '\p{^Is_Script_Extensions=lydian}', "");
Expect(0, 67903, '\P{Is_Script_Extensions=lydian}', "");
Expect(1, 67903, '\P{^Is_Script_Extensions=lydian}', "");
Expect(0, 67904, '\p{Is_Script_Extensions=lydian}', "");
Expect(1, 67904, '\p{^Is_Script_Extensions=lydian}', "");
Expect(1, 67904, '\P{Is_Script_Extensions=lydian}', "");
Expect(0, 67904, '\P{^Is_Script_Extensions=lydian}', "");
Expect(1, 67903, '\p{Is_Script_Extensions=		Lydian}', "");
Expect(0, 67903, '\p{^Is_Script_Extensions=		Lydian}', "");
Expect(0, 67903, '\P{Is_Script_Extensions=		Lydian}', "");
Expect(1, 67903, '\P{^Is_Script_Extensions=		Lydian}', "");
Expect(0, 67904, '\p{Is_Script_Extensions=		Lydian}', "");
Expect(1, 67904, '\p{^Is_Script_Extensions=		Lydian}', "");
Expect(1, 67904, '\P{Is_Script_Extensions=		Lydian}', "");
Expect(0, 67904, '\P{^Is_Script_Extensions=		Lydian}', "");
Error('\p{Is_Scx=_-lydi/a/}');
Error('\P{Is_Scx=_-lydi/a/}');
Expect(1, 67903, '\p{Is_Scx=lydi}', "");
Expect(0, 67903, '\p{^Is_Scx=lydi}', "");
Expect(0, 67903, '\P{Is_Scx=lydi}', "");
Expect(1, 67903, '\P{^Is_Scx=lydi}', "");
Expect(0, 67904, '\p{Is_Scx=lydi}', "");
Expect(1, 67904, '\p{^Is_Scx=lydi}', "");
Expect(1, 67904, '\P{Is_Scx=lydi}', "");
Expect(0, 67904, '\P{^Is_Scx=lydi}', "");
Expect(1, 67903, '\p{Is_Scx=_Lydi}', "");
Expect(0, 67903, '\p{^Is_Scx=_Lydi}', "");
Expect(0, 67903, '\P{Is_Scx=_Lydi}', "");
Expect(1, 67903, '\P{^Is_Scx=_Lydi}', "");
Expect(0, 67904, '\p{Is_Scx=_Lydi}', "");
Expect(1, 67904, '\p{^Is_Scx=_Lydi}', "");
Expect(1, 67904, '\P{Is_Scx=_Lydi}', "");
Expect(0, 67904, '\P{^Is_Scx=_Lydi}', "");
Error('\p{Script_Extensions=-Lao:=}');
Error('\P{Script_Extensions=-Lao:=}');
Expect(1, 3807, '\p{Script_Extensions=lao}', "");
Expect(0, 3807, '\p{^Script_Extensions=lao}', "");
Expect(0, 3807, '\P{Script_Extensions=lao}', "");
Expect(1, 3807, '\P{^Script_Extensions=lao}', "");
Expect(0, 3808, '\p{Script_Extensions=lao}', "");
Expect(1, 3808, '\p{^Script_Extensions=lao}', "");
Expect(1, 3808, '\P{Script_Extensions=lao}', "");
Expect(0, 3808, '\P{^Script_Extensions=lao}', "");
Expect(1, 3807, '\p{Script_Extensions:Lao}', "");
Expect(0, 3807, '\p{^Script_Extensions:Lao}', "");
Expect(0, 3807, '\P{Script_Extensions:Lao}', "");
Expect(1, 3807, '\P{^Script_Extensions:Lao}', "");
Expect(0, 3808, '\p{Script_Extensions:Lao}', "");
Expect(1, 3808, '\p{^Script_Extensions:Lao}', "");
Expect(1, 3808, '\P{Script_Extensions:Lao}', "");
Expect(0, 3808, '\P{^Script_Extensions:Lao}', "");
Error('\p{Scx=:=- Laoo}');
Error('\P{Scx=:=- Laoo}');
Expect(1, 3807, '\p{Scx=laoo}', "");
Expect(0, 3807, '\p{^Scx=laoo}', "");
Expect(0, 3807, '\P{Scx=laoo}', "");
Expect(1, 3807, '\P{^Scx=laoo}', "");
Expect(0, 3808, '\p{Scx=laoo}', "");
Expect(1, 3808, '\p{^Scx=laoo}', "");
Expect(1, 3808, '\P{Scx=laoo}', "");
Expect(0, 3808, '\P{^Scx=laoo}', "");
Expect(1, 3807, '\p{Scx=_ LAOO}', "");
Expect(0, 3807, '\p{^Scx=_ LAOO}', "");
Expect(0, 3807, '\P{Scx=_ LAOO}', "");
Expect(1, 3807, '\P{^Scx=_ LAOO}', "");
Expect(0, 3808, '\p{Scx=_ LAOO}', "");
Expect(1, 3808, '\p{^Scx=_ LAOO}', "");
Expect(1, 3808, '\P{Scx=_ LAOO}', "");
Expect(0, 3808, '\P{^Scx=_ LAOO}', "");
Error('\p{Is_Script_Extensions= 	Lao:=}');
Error('\P{Is_Script_Extensions= 	Lao:=}');
Expect(1, 3807, '\p{Is_Script_Extensions=lao}', "");
Expect(0, 3807, '\p{^Is_Script_Extensions=lao}', "");
Expect(0, 3807, '\P{Is_Script_Extensions=lao}', "");
Expect(1, 3807, '\P{^Is_Script_Extensions=lao}', "");
Expect(0, 3808, '\p{Is_Script_Extensions=lao}', "");
Expect(1, 3808, '\p{^Is_Script_Extensions=lao}', "");
Expect(1, 3808, '\P{Is_Script_Extensions=lao}', "");
Expect(0, 3808, '\P{^Is_Script_Extensions=lao}', "");
Expect(1, 3807, '\p{Is_Script_Extensions=	-lao}', "");
Expect(0, 3807, '\p{^Is_Script_Extensions=	-lao}', "");
Expect(0, 3807, '\P{Is_Script_Extensions=	-lao}', "");
Expect(1, 3807, '\P{^Is_Script_Extensions=	-lao}', "");
Expect(0, 3808, '\p{Is_Script_Extensions=	-lao}', "");
Expect(1, 3808, '\p{^Is_Script_Extensions=	-lao}', "");
Expect(1, 3808, '\P{Is_Script_Extensions=	-lao}', "");
Expect(0, 3808, '\P{^Is_Script_Extensions=	-lao}', "");
Error('\p{Is_Scx=/a/Laoo}');
Error('\P{Is_Scx=/a/Laoo}');
Expect(1, 3807, '\p{Is_Scx=laoo}', "");
Expect(0, 3807, '\p{^Is_Scx=laoo}', "");
Expect(0, 3807, '\P{Is_Scx=laoo}', "");
Expect(1, 3807, '\P{^Is_Scx=laoo}', "");
Expect(0, 3808, '\p{Is_Scx=laoo}', "");
Expect(1, 3808, '\p{^Is_Scx=laoo}', "");
Expect(1, 3808, '\P{Is_Scx=laoo}', "");
Expect(0, 3808, '\P{^Is_Scx=laoo}', "");
Expect(1, 3807, '\p{Is_Scx=- LAOO}', "");
Expect(0, 3807, '\p{^Is_Scx=- LAOO}', "");
Expect(0, 3807, '\P{Is_Scx=- LAOO}', "");
Expect(1, 3807, '\P{^Is_Scx=- LAOO}', "");
Expect(0, 3808, '\p{Is_Scx=- LAOO}', "");
Expect(1, 3808, '\p{^Is_Scx=- LAOO}', "");
Expect(1, 3808, '\P{Is_Scx=- LAOO}', "");
Expect(0, 3808, '\P{^Is_Scx=- LAOO}', "");
Error('\p{Script_Extensions=-:=MONGOLIAN}');
Error('\P{Script_Extensions=-:=MONGOLIAN}');
Expect(1, 6314, '\p{Script_Extensions=mongolian}', "");
Expect(0, 6314, '\p{^Script_Extensions=mongolian}', "");
Expect(0, 6314, '\P{Script_Extensions=mongolian}', "");
Expect(1, 6314, '\P{^Script_Extensions=mongolian}', "");
Expect(0, 6315, '\p{Script_Extensions=mongolian}', "");
Expect(1, 6315, '\p{^Script_Extensions=mongolian}', "");
Expect(1, 6315, '\P{Script_Extensions=mongolian}', "");
Expect(0, 6315, '\P{^Script_Extensions=mongolian}', "");
Expect(1, 6314, '\p{Script_Extensions=-	MONGOLIAN}', "");
Expect(0, 6314, '\p{^Script_Extensions=-	MONGOLIAN}', "");
Expect(0, 6314, '\P{Script_Extensions=-	MONGOLIAN}', "");
Expect(1, 6314, '\P{^Script_Extensions=-	MONGOLIAN}', "");
Expect(0, 6315, '\p{Script_Extensions=-	MONGOLIAN}', "");
Expect(1, 6315, '\p{^Script_Extensions=-	MONGOLIAN}', "");
Expect(1, 6315, '\P{Script_Extensions=-	MONGOLIAN}', "");
Expect(0, 6315, '\P{^Script_Extensions=-	MONGOLIAN}', "");
Error('\p{Scx=--mong:=}');
Error('\P{Scx=--mong:=}');
Expect(1, 6314, '\p{Scx=mong}', "");
Expect(0, 6314, '\p{^Scx=mong}', "");
Expect(0, 6314, '\P{Scx=mong}', "");
Expect(1, 6314, '\P{^Scx=mong}', "");
Expect(0, 6315, '\p{Scx=mong}', "");
Expect(1, 6315, '\p{^Scx=mong}', "");
Expect(1, 6315, '\P{Scx=mong}', "");
Expect(0, 6315, '\P{^Scx=mong}', "");
Expect(1, 6314, '\p{Scx=	MONG}', "");
Expect(0, 6314, '\p{^Scx=	MONG}', "");
Expect(0, 6314, '\P{Scx=	MONG}', "");
Expect(1, 6314, '\P{^Scx=	MONG}', "");
Expect(0, 6315, '\p{Scx=	MONG}', "");
Expect(1, 6315, '\p{^Scx=	MONG}', "");
Expect(1, 6315, '\P{Scx=	MONG}', "");
Expect(0, 6315, '\P{^Scx=	MONG}', "");
Error('\p{Is_Script_Extensions=_ MONGOLIAN:=}');
Error('\P{Is_Script_Extensions=_ MONGOLIAN:=}');
Expect(1, 6314, '\p{Is_Script_Extensions=mongolian}', "");
Expect(0, 6314, '\p{^Is_Script_Extensions=mongolian}', "");
Expect(0, 6314, '\P{Is_Script_Extensions=mongolian}', "");
Expect(1, 6314, '\P{^Is_Script_Extensions=mongolian}', "");
Expect(0, 6315, '\p{Is_Script_Extensions=mongolian}', "");
Expect(1, 6315, '\p{^Is_Script_Extensions=mongolian}', "");
Expect(1, 6315, '\P{Is_Script_Extensions=mongolian}', "");
Expect(0, 6315, '\P{^Is_Script_Extensions=mongolian}', "");
Expect(1, 6314, '\p{Is_Script_Extensions=_Mongolian}', "");
Expect(0, 6314, '\p{^Is_Script_Extensions=_Mongolian}', "");
Expect(0, 6314, '\P{Is_Script_Extensions=_Mongolian}', "");
Expect(1, 6314, '\P{^Is_Script_Extensions=_Mongolian}', "");
Expect(0, 6315, '\p{Is_Script_Extensions=_Mongolian}', "");
Expect(1, 6315, '\p{^Is_Script_Extensions=_Mongolian}', "");
Expect(1, 6315, '\P{Is_Script_Extensions=_Mongolian}', "");
Expect(0, 6315, '\P{^Is_Script_Extensions=_Mongolian}', "");
Error('\p{Is_Scx=/a/	Mong}');
Error('\P{Is_Scx=/a/	Mong}');
Expect(1, 6314, '\p{Is_Scx=mong}', "");
Expect(0, 6314, '\p{^Is_Scx=mong}', "");
Expect(0, 6314, '\P{Is_Scx=mong}', "");
Expect(1, 6314, '\P{^Is_Scx=mong}', "");
Expect(0, 6315, '\p{Is_Scx=mong}', "");
Expect(1, 6315, '\p{^Is_Scx=mong}', "");
Expect(1, 6315, '\P{Is_Scx=mong}', "");
Expect(0, 6315, '\P{^Is_Scx=mong}', "");
Expect(1, 6314, '\p{Is_Scx:	Mong}', "");
Expect(0, 6314, '\p{^Is_Scx:	Mong}', "");
Expect(0, 6314, '\P{Is_Scx:	Mong}', "");
Expect(1, 6314, '\P{^Is_Scx:	Mong}', "");
Expect(0, 6315, '\p{Is_Scx:	Mong}', "");
Expect(1, 6315, '\p{^Is_Scx:	Mong}', "");
Expect(1, 6315, '\P{Is_Scx:	Mong}', "");
Expect(0, 6315, '\P{^Is_Scx:	Mong}', "");
Error('\p{Script_Extensions=_cyrillic/a/}');
Error('\P{Script_Extensions=_cyrillic/a/}');
Expect(1, 42655, '\p{Script_Extensions=cyrillic}', "");
Expect(0, 42655, '\p{^Script_Extensions=cyrillic}', "");
Expect(0, 42655, '\P{Script_Extensions=cyrillic}', "");
Expect(1, 42655, '\P{^Script_Extensions=cyrillic}', "");
Expect(0, 42656, '\p{Script_Extensions=cyrillic}', "");
Expect(1, 42656, '\p{^Script_Extensions=cyrillic}', "");
Expect(1, 42656, '\P{Script_Extensions=cyrillic}', "");
Expect(0, 42656, '\P{^Script_Extensions=cyrillic}', "");
Expect(1, 42655, '\p{Script_Extensions:--CYRILLIC}', "");
Expect(0, 42655, '\p{^Script_Extensions:--CYRILLIC}', "");
Expect(0, 42655, '\P{Script_Extensions:--CYRILLIC}', "");
Expect(1, 42655, '\P{^Script_Extensions:--CYRILLIC}', "");
Expect(0, 42656, '\p{Script_Extensions:--CYRILLIC}', "");
Expect(1, 42656, '\p{^Script_Extensions:--CYRILLIC}', "");
Expect(1, 42656, '\P{Script_Extensions:--CYRILLIC}', "");
Expect(0, 42656, '\P{^Script_Extensions:--CYRILLIC}', "");
Error('\p{Scx=_cyrl:=}');
Error('\P{Scx=_cyrl:=}');
Expect(1, 42655, '\p{Scx=cyrl}', "");
Expect(0, 42655, '\p{^Scx=cyrl}', "");
Expect(0, 42655, '\P{Scx=cyrl}', "");
Expect(1, 42655, '\P{^Scx=cyrl}', "");
Expect(0, 42656, '\p{Scx=cyrl}', "");
Expect(1, 42656, '\p{^Scx=cyrl}', "");
Expect(1, 42656, '\P{Scx=cyrl}', "");
Expect(0, 42656, '\P{^Scx=cyrl}', "");
Expect(1, 42655, '\p{Scx=		Cyrl}', "");
Expect(0, 42655, '\p{^Scx=		Cyrl}', "");
Expect(0, 42655, '\P{Scx=		Cyrl}', "");
Expect(1, 42655, '\P{^Scx=		Cyrl}', "");
Expect(0, 42656, '\p{Scx=		Cyrl}', "");
Expect(1, 42656, '\p{^Scx=		Cyrl}', "");
Expect(1, 42656, '\P{Scx=		Cyrl}', "");
Expect(0, 42656, '\P{^Scx=		Cyrl}', "");
Error('\p{Is_Script_Extensions=	:=Cyrillic}');
Error('\P{Is_Script_Extensions=	:=Cyrillic}');
Expect(1, 42655, '\p{Is_Script_Extensions=cyrillic}', "");
Expect(0, 42655, '\p{^Is_Script_Extensions=cyrillic}', "");
Expect(0, 42655, '\P{Is_Script_Extensions=cyrillic}', "");
Expect(1, 42655, '\P{^Is_Script_Extensions=cyrillic}', "");
Expect(0, 42656, '\p{Is_Script_Extensions=cyrillic}', "");
Expect(1, 42656, '\p{^Is_Script_Extensions=cyrillic}', "");
Expect(1, 42656, '\P{Is_Script_Extensions=cyrillic}', "");
Expect(0, 42656, '\P{^Is_Script_Extensions=cyrillic}', "");
Expect(1, 42655, '\p{Is_Script_Extensions:   __cyrillic}', "");
Expect(0, 42655, '\p{^Is_Script_Extensions:   __cyrillic}', "");
Expect(0, 42655, '\P{Is_Script_Extensions:   __cyrillic}', "");
Expect(1, 42655, '\P{^Is_Script_Extensions:   __cyrillic}', "");
Expect(0, 42656, '\p{Is_Script_Extensions:   __cyrillic}', "");
Expect(1, 42656, '\p{^Is_Script_Extensions:   __cyrillic}', "");
Expect(1, 42656, '\P{Is_Script_Extensions:   __cyrillic}', "");
Expect(0, 42656, '\P{^Is_Script_Extensions:   __cyrillic}', "");
Error('\p{Is_Scx=__Cyrl:=}');
Error('\P{Is_Scx=__Cyrl:=}');
Expect(1, 42655, '\p{Is_Scx=cyrl}', "");
Expect(0, 42655, '\p{^Is_Scx=cyrl}', "");
Expect(0, 42655, '\P{Is_Scx=cyrl}', "");
Expect(1, 42655, '\P{^Is_Scx=cyrl}', "");
Expect(0, 42656, '\p{Is_Scx=cyrl}', "");
Expect(1, 42656, '\p{^Is_Scx=cyrl}', "");
Expect(1, 42656, '\P{Is_Scx=cyrl}', "");
Expect(0, 42656, '\P{^Is_Scx=cyrl}', "");
Expect(1, 42655, '\p{Is_Scx=--CYRL}', "");
Expect(0, 42655, '\p{^Is_Scx=--CYRL}', "");
Expect(0, 42655, '\P{Is_Scx=--CYRL}', "");
Expect(1, 42655, '\P{^Is_Scx=--CYRL}', "");
Expect(0, 42656, '\p{Is_Scx=--CYRL}', "");
Expect(1, 42656, '\p{^Is_Scx=--CYRL}', "");
Expect(1, 42656, '\P{Is_Scx=--CYRL}', "");
Expect(0, 42656, '\P{^Is_Scx=--CYRL}', "");
Error('\p{Script_Extensions=	:=Hiragana}');
Error('\P{Script_Extensions=	:=Hiragana}');
Expect(1, 127488, '\p{Script_Extensions=hiragana}', "");
Expect(0, 127488, '\p{^Script_Extensions=hiragana}', "");
Expect(0, 127488, '\P{Script_Extensions=hiragana}', "");
Expect(1, 127488, '\P{^Script_Extensions=hiragana}', "");
Expect(0, 127489, '\p{Script_Extensions=hiragana}', "");
Expect(1, 127489, '\p{^Script_Extensions=hiragana}', "");
Expect(1, 127489, '\P{Script_Extensions=hiragana}', "");
Expect(0, 127489, '\P{^Script_Extensions=hiragana}', "");
Expect(1, 127488, '\p{Script_Extensions=- HIRAGANA}', "");
Expect(0, 127488, '\p{^Script_Extensions=- HIRAGANA}', "");
Expect(0, 127488, '\P{Script_Extensions=- HIRAGANA}', "");
Expect(1, 127488, '\P{^Script_Extensions=- HIRAGANA}', "");
Expect(0, 127489, '\p{Script_Extensions=- HIRAGANA}', "");
Expect(1, 127489, '\p{^Script_Extensions=- HIRAGANA}', "");
Expect(1, 127489, '\P{Script_Extensions=- HIRAGANA}', "");
Expect(0, 127489, '\P{^Script_Extensions=- HIRAGANA}', "");
Error('\p{Scx= /a/HIRA}');
Error('\P{Scx= /a/HIRA}');
Expect(1, 127488, '\p{Scx=hira}', "");
Expect(0, 127488, '\p{^Scx=hira}', "");
Expect(0, 127488, '\P{Scx=hira}', "");
Expect(1, 127488, '\P{^Scx=hira}', "");
Expect(0, 127489, '\p{Scx=hira}', "");
Expect(1, 127489, '\p{^Scx=hira}', "");
Expect(1, 127489, '\P{Scx=hira}', "");
Expect(0, 127489, '\P{^Scx=hira}', "");
Expect(1, 127488, '\p{Scx=	-Hira}', "");
Expect(0, 127488, '\p{^Scx=	-Hira}', "");
Expect(0, 127488, '\P{Scx=	-Hira}', "");
Expect(1, 127488, '\P{^Scx=	-Hira}', "");
Expect(0, 127489, '\p{Scx=	-Hira}', "");
Expect(1, 127489, '\p{^Scx=	-Hira}', "");
Expect(1, 127489, '\P{Scx=	-Hira}', "");
Expect(0, 127489, '\P{^Scx=	-Hira}', "");
Error('\p{Is_Script_Extensions=hiragana/a/}');
Error('\P{Is_Script_Extensions=hiragana/a/}');
Expect(1, 127488, '\p{Is_Script_Extensions=hiragana}', "");
Expect(0, 127488, '\p{^Is_Script_Extensions=hiragana}', "");
Expect(0, 127488, '\P{Is_Script_Extensions=hiragana}', "");
Expect(1, 127488, '\P{^Is_Script_Extensions=hiragana}', "");
Expect(0, 127489, '\p{Is_Script_Extensions=hiragana}', "");
Expect(1, 127489, '\p{^Is_Script_Extensions=hiragana}', "");
Expect(1, 127489, '\P{Is_Script_Extensions=hiragana}', "");
Expect(0, 127489, '\P{^Is_Script_Extensions=hiragana}', "");
Expect(1, 127488, '\p{Is_Script_Extensions=-_hiragana}', "");
Expect(0, 127488, '\p{^Is_Script_Extensions=-_hiragana}', "");
Expect(0, 127488, '\P{Is_Script_Extensions=-_hiragana}', "");
Expect(1, 127488, '\P{^Is_Script_Extensions=-_hiragana}', "");
Expect(0, 127489, '\p{Is_Script_Extensions=-_hiragana}', "");
Expect(1, 127489, '\p{^Is_Script_Extensions=-_hiragana}', "");
Expect(1, 127489, '\P{Is_Script_Extensions=-_hiragana}', "");
Expect(0, 127489, '\P{^Is_Script_Extensions=-_hiragana}', "");
Error('\p{Is_Scx:__hira/a/}');
Error('\P{Is_Scx:__hira/a/}');
Expect(1, 127488, '\p{Is_Scx=hira}', "");
Expect(0, 127488, '\p{^Is_Scx=hira}', "");
Expect(0, 127488, '\P{Is_Scx=hira}', "");
Expect(1, 127488, '\P{^Is_Scx=hira}', "");
Expect(0, 127489, '\p{Is_Scx=hira}', "");
Expect(1, 127489, '\p{^Is_Scx=hira}', "");
Expect(1, 127489, '\P{Is_Scx=hira}', "");
Expect(0, 127489, '\P{^Is_Scx=hira}', "");
Expect(1, 127488, '\p{Is_Scx=--HIRA}', "");
Expect(0, 127488, '\p{^Is_Scx=--HIRA}', "");
Expect(0, 127488, '\P{Is_Scx=--HIRA}', "");
Expect(1, 127488, '\P{^Is_Scx=--HIRA}', "");
Expect(0, 127489, '\p{Is_Scx=--HIRA}', "");
Expect(1, 127489, '\p{^Is_Scx=--HIRA}', "");
Expect(1, 127489, '\P{Is_Scx=--HIRA}', "");
Expect(0, 127489, '\P{^Is_Scx=--HIRA}', "");
Error('\p{Script_Extensions=_:=Braille}');
Error('\P{Script_Extensions=_:=Braille}');
Expect(1, 10495, '\p{Script_Extensions=braille}', "");
Expect(0, 10495, '\p{^Script_Extensions=braille}', "");
Expect(0, 10495, '\P{Script_Extensions=braille}', "");
Expect(1, 10495, '\P{^Script_Extensions=braille}', "");
Expect(0, 10496, '\p{Script_Extensions=braille}', "");
Expect(1, 10496, '\p{^Script_Extensions=braille}', "");
Expect(1, 10496, '\P{Script_Extensions=braille}', "");
Expect(0, 10496, '\P{^Script_Extensions=braille}', "");
Expect(1, 10495, '\p{Script_Extensions=_-Braille}', "");
Expect(0, 10495, '\p{^Script_Extensions=_-Braille}', "");
Expect(0, 10495, '\P{Script_Extensions=_-Braille}', "");
Expect(1, 10495, '\P{^Script_Extensions=_-Braille}', "");
Expect(0, 10496, '\p{Script_Extensions=_-Braille}', "");
Expect(1, 10496, '\p{^Script_Extensions=_-Braille}', "");
Expect(1, 10496, '\P{Script_Extensions=_-Braille}', "");
Expect(0, 10496, '\P{^Script_Extensions=_-Braille}', "");
Error('\p{Scx=-_brai:=}');
Error('\P{Scx=-_brai:=}');
Expect(1, 10495, '\p{Scx=brai}', "");
Expect(0, 10495, '\p{^Scx=brai}', "");
Expect(0, 10495, '\P{Scx=brai}', "");
Expect(1, 10495, '\P{^Scx=brai}', "");
Expect(0, 10496, '\p{Scx=brai}', "");
Expect(1, 10496, '\p{^Scx=brai}', "");
Expect(1, 10496, '\P{Scx=brai}', "");
Expect(0, 10496, '\P{^Scx=brai}', "");
Expect(1, 10495, '\p{Scx=__Brai}', "");
Expect(0, 10495, '\p{^Scx=__Brai}', "");
Expect(0, 10495, '\P{Scx=__Brai}', "");
Expect(1, 10495, '\P{^Scx=__Brai}', "");
Expect(0, 10496, '\p{Scx=__Brai}', "");
Expect(1, 10496, '\p{^Scx=__Brai}', "");
Expect(1, 10496, '\P{Scx=__Brai}', "");
Expect(0, 10496, '\P{^Scx=__Brai}', "");
Error('\p{Is_Script_Extensions:   	/a/BRAILLE}');
Error('\P{Is_Script_Extensions:   	/a/BRAILLE}');
Expect(1, 10495, '\p{Is_Script_Extensions:	braille}', "");
Expect(0, 10495, '\p{^Is_Script_Extensions:	braille}', "");
Expect(0, 10495, '\P{Is_Script_Extensions:	braille}', "");
Expect(1, 10495, '\P{^Is_Script_Extensions:	braille}', "");
Expect(0, 10496, '\p{Is_Script_Extensions:	braille}', "");
Expect(1, 10496, '\p{^Is_Script_Extensions:	braille}', "");
Expect(1, 10496, '\P{Is_Script_Extensions:	braille}', "");
Expect(0, 10496, '\P{^Is_Script_Extensions:	braille}', "");
Expect(1, 10495, '\p{Is_Script_Extensions=_BRAILLE}', "");
Expect(0, 10495, '\p{^Is_Script_Extensions=_BRAILLE}', "");
Expect(0, 10495, '\P{Is_Script_Extensions=_BRAILLE}', "");
Expect(1, 10495, '\P{^Is_Script_Extensions=_BRAILLE}', "");
Expect(0, 10496, '\p{Is_Script_Extensions=_BRAILLE}', "");
Expect(1, 10496, '\p{^Is_Script_Extensions=_BRAILLE}', "");
Expect(1, 10496, '\P{Is_Script_Extensions=_BRAILLE}', "");
Expect(0, 10496, '\P{^Is_Script_Extensions=_BRAILLE}', "");
Error('\p{Is_Scx=/a/		brai}');
Error('\P{Is_Scx=/a/		brai}');
Expect(1, 10495, '\p{Is_Scx=brai}', "");
Expect(0, 10495, '\p{^Is_Scx=brai}', "");
Expect(0, 10495, '\P{Is_Scx=brai}', "");
Expect(1, 10495, '\P{^Is_Scx=brai}', "");
Expect(0, 10496, '\p{Is_Scx=brai}', "");
Expect(1, 10496, '\p{^Is_Scx=brai}', "");
Expect(1, 10496, '\P{Is_Scx=brai}', "");
Expect(0, 10496, '\P{^Is_Scx=brai}', "");
Expect(1, 10495, '\p{Is_Scx=_	Brai}', "");
Expect(0, 10495, '\p{^Is_Scx=_	Brai}', "");
Expect(0, 10495, '\P{Is_Scx=_	Brai}', "");
Expect(1, 10495, '\P{^Is_Scx=_	Brai}', "");
Expect(0, 10496, '\p{Is_Scx=_	Brai}', "");
Expect(1, 10496, '\p{^Is_Scx=_	Brai}', "");
Expect(1, 10496, '\P{Is_Scx=_	Brai}', "");
Expect(0, 10496, '\P{^Is_Scx=_	Brai}', "");
Error('\p{Script_Extensions=_ Bengali/a/}');
Error('\P{Script_Extensions=_ Bengali/a/}');
Expect(1, 2555, '\p{Script_Extensions=bengali}', "");
Expect(0, 2555, '\p{^Script_Extensions=bengali}', "");
Expect(0, 2555, '\P{Script_Extensions=bengali}', "");
Expect(1, 2555, '\P{^Script_Extensions=bengali}', "");
Expect(0, 2556, '\p{Script_Extensions=bengali}', "");
Expect(1, 2556, '\p{^Script_Extensions=bengali}', "");
Expect(1, 2556, '\P{Script_Extensions=bengali}', "");
Expect(0, 2556, '\P{^Script_Extensions=bengali}', "");
Expect(1, 2555, '\p{Script_Extensions=-	bengali}', "");
Expect(0, 2555, '\p{^Script_Extensions=-	bengali}', "");
Expect(0, 2555, '\P{Script_Extensions=-	bengali}', "");
Expect(1, 2555, '\P{^Script_Extensions=-	bengali}', "");
Expect(0, 2556, '\p{Script_Extensions=-	bengali}', "");
Expect(1, 2556, '\p{^Script_Extensions=-	bengali}', "");
Expect(1, 2556, '\P{Script_Extensions=-	bengali}', "");
Expect(0, 2556, '\P{^Script_Extensions=-	bengali}', "");
Error('\p{Scx=:=Beng}');
Error('\P{Scx=:=Beng}');
Expect(1, 2555, '\p{Scx=beng}', "");
Expect(0, 2555, '\p{^Scx=beng}', "");
Expect(0, 2555, '\P{Scx=beng}', "");
Expect(1, 2555, '\P{^Scx=beng}', "");
Expect(0, 2556, '\p{Scx=beng}', "");
Expect(1, 2556, '\p{^Scx=beng}', "");
Expect(1, 2556, '\P{Scx=beng}', "");
Expect(0, 2556, '\P{^Scx=beng}', "");
Expect(1, 2555, '\p{Scx=	beng}', "");
Expect(0, 2555, '\p{^Scx=	beng}', "");
Expect(0, 2555, '\P{Scx=	beng}', "");
Expect(1, 2555, '\P{^Scx=	beng}', "");
Expect(0, 2556, '\p{Scx=	beng}', "");
Expect(1, 2556, '\p{^Scx=	beng}', "");
Expect(1, 2556, '\P{Scx=	beng}', "");
Expect(0, 2556, '\P{^Scx=	beng}', "");
Error('\p{Is_Script_Extensions=	Bengali/a/}');
Error('\P{Is_Script_Extensions=	Bengali/a/}');
Expect(1, 2555, '\p{Is_Script_Extensions=bengali}', "");
Expect(0, 2555, '\p{^Is_Script_Extensions=bengali}', "");
Expect(0, 2555, '\P{Is_Script_Extensions=bengali}', "");
Expect(1, 2555, '\P{^Is_Script_Extensions=bengali}', "");
Expect(0, 2556, '\p{Is_Script_Extensions=bengali}', "");
Expect(1, 2556, '\p{^Is_Script_Extensions=bengali}', "");
Expect(1, 2556, '\P{Is_Script_Extensions=bengali}', "");
Expect(0, 2556, '\P{^Is_Script_Extensions=bengali}', "");
Expect(1, 2555, '\p{Is_Script_Extensions=	Bengali}', "");
Expect(0, 2555, '\p{^Is_Script_Extensions=	Bengali}', "");
Expect(0, 2555, '\P{Is_Script_Extensions=	Bengali}', "");
Expect(1, 2555, '\P{^Is_Script_Extensions=	Bengali}', "");
Expect(0, 2556, '\p{Is_Script_Extensions=	Bengali}', "");
Expect(1, 2556, '\p{^Is_Script_Extensions=	Bengali}', "");
Expect(1, 2556, '\P{Is_Script_Extensions=	Bengali}', "");
Expect(0, 2556, '\P{^Is_Script_Extensions=	Bengali}', "");
Error('\p{Is_Scx=	/a/Beng}');
Error('\P{Is_Scx=	/a/Beng}');
Expect(1, 2555, '\p{Is_Scx=beng}', "");
Expect(0, 2555, '\p{^Is_Scx=beng}', "");
Expect(0, 2555, '\P{Is_Scx=beng}', "");
Expect(1, 2555, '\P{^Is_Scx=beng}', "");
Expect(0, 2556, '\p{Is_Scx=beng}', "");
Expect(1, 2556, '\p{^Is_Scx=beng}', "");
Expect(1, 2556, '\P{Is_Scx=beng}', "");
Expect(0, 2556, '\P{^Is_Scx=beng}', "");
Expect(1, 2555, '\p{Is_Scx=-_BENG}', "");
Expect(0, 2555, '\p{^Is_Scx=-_BENG}', "");
Expect(0, 2555, '\P{Is_Scx=-_BENG}', "");
Expect(1, 2555, '\P{^Is_Scx=-_BENG}', "");
Expect(0, 2556, '\p{Is_Scx=-_BENG}', "");
Expect(1, 2556, '\p{^Is_Scx=-_BENG}', "");
Expect(1, 2556, '\P{Is_Scx=-_BENG}', "");
Expect(0, 2556, '\P{^Is_Scx=-_BENG}', "");
Error('\p{Script_Extensions=  Kharoshthi:=}');
Error('\P{Script_Extensions=  Kharoshthi:=}');
Expect(1, 68184, '\p{Script_Extensions=kharoshthi}', "");
Expect(0, 68184, '\p{^Script_Extensions=kharoshthi}', "");
Expect(0, 68184, '\P{Script_Extensions=kharoshthi}', "");
Expect(1, 68184, '\P{^Script_Extensions=kharoshthi}', "");
Expect(0, 68185, '\p{Script_Extensions=kharoshthi}', "");
Expect(1, 68185, '\p{^Script_Extensions=kharoshthi}', "");
Expect(1, 68185, '\P{Script_Extensions=kharoshthi}', "");
Expect(0, 68185, '\P{^Script_Extensions=kharoshthi}', "");
Expect(1, 68184, '\p{Script_Extensions=_	Kharoshthi}', "");
Expect(0, 68184, '\p{^Script_Extensions=_	Kharoshthi}', "");
Expect(0, 68184, '\P{Script_Extensions=_	Kharoshthi}', "");
Expect(1, 68184, '\P{^Script_Extensions=_	Kharoshthi}', "");
Expect(0, 68185, '\p{Script_Extensions=_	Kharoshthi}', "");
Expect(1, 68185, '\p{^Script_Extensions=_	Kharoshthi}', "");
Expect(1, 68185, '\P{Script_Extensions=_	Kharoshthi}', "");
Expect(0, 68185, '\P{^Script_Extensions=_	Kharoshthi}', "");
Error('\p{Scx=/a/Khar}');
Error('\P{Scx=/a/Khar}');
Expect(1, 68184, '\p{Scx=khar}', "");
Expect(0, 68184, '\p{^Scx=khar}', "");
Expect(0, 68184, '\P{Scx=khar}', "");
Expect(1, 68184, '\P{^Scx=khar}', "");
Expect(0, 68185, '\p{Scx=khar}', "");
Expect(1, 68185, '\p{^Scx=khar}', "");
Expect(1, 68185, '\P{Scx=khar}', "");
Expect(0, 68185, '\P{^Scx=khar}', "");
Expect(1, 68184, '\p{Scx=__KHAR}', "");
Expect(0, 68184, '\p{^Scx=__KHAR}', "");
Expect(0, 68184, '\P{Scx=__KHAR}', "");
Expect(1, 68184, '\P{^Scx=__KHAR}', "");
Expect(0, 68185, '\p{Scx=__KHAR}', "");
Expect(1, 68185, '\p{^Scx=__KHAR}', "");
Expect(1, 68185, '\P{Scx=__KHAR}', "");
Expect(0, 68185, '\P{^Scx=__KHAR}', "");
Error('\p{Is_Script_Extensions=_	Kharoshthi:=}');
Error('\P{Is_Script_Extensions=_	Kharoshthi:=}');
Expect(1, 68184, '\p{Is_Script_Extensions=kharoshthi}', "");
Expect(0, 68184, '\p{^Is_Script_Extensions=kharoshthi}', "");
Expect(0, 68184, '\P{Is_Script_Extensions=kharoshthi}', "");
Expect(1, 68184, '\P{^Is_Script_Extensions=kharoshthi}', "");
Expect(0, 68185, '\p{Is_Script_Extensions=kharoshthi}', "");
Expect(1, 68185, '\p{^Is_Script_Extensions=kharoshthi}', "");
Expect(1, 68185, '\P{Is_Script_Extensions=kharoshthi}', "");
Expect(0, 68185, '\P{^Is_Script_Extensions=kharoshthi}', "");
Expect(1, 68184, '\p{Is_Script_Extensions=_-Kharoshthi}', "");
Expect(0, 68184, '\p{^Is_Script_Extensions=_-Kharoshthi}', "");
Expect(0, 68184, '\P{Is_Script_Extensions=_-Kharoshthi}', "");
Expect(1, 68184, '\P{^Is_Script_Extensions=_-Kharoshthi}', "");
Expect(0, 68185, '\p{Is_Script_Extensions=_-Kharoshthi}', "");
Expect(1, 68185, '\p{^Is_Script_Extensions=_-Kharoshthi}', "");
Expect(1, 68185, '\P{Is_Script_Extensions=_-Kharoshthi}', "");
Expect(0, 68185, '\P{^Is_Script_Extensions=_-Kharoshthi}', "");
Error('\p{Is_Scx=/a/ KHAR}');
Error('\P{Is_Scx=/a/ KHAR}');
Expect(1, 68184, '\p{Is_Scx=khar}', "");
Expect(0, 68184, '\p{^Is_Scx=khar}', "");
Expect(0, 68184, '\P{Is_Scx=khar}', "");
Expect(1, 68184, '\P{^Is_Scx=khar}', "");
Expect(0, 68185, '\p{Is_Scx=khar}', "");
Expect(1, 68185, '\p{^Is_Scx=khar}', "");
Expect(1, 68185, '\P{Is_Scx=khar}', "");
Expect(0, 68185, '\P{^Is_Scx=khar}', "");
Expect(1, 68184, '\p{Is_Scx=--Khar}', "");
Expect(0, 68184, '\p{^Is_Scx=--Khar}', "");
Expect(0, 68184, '\P{Is_Scx=--Khar}', "");
Expect(1, 68184, '\P{^Is_Scx=--Khar}', "");
Expect(0, 68185, '\p{Is_Scx=--Khar}', "");
Expect(1, 68185, '\p{^Is_Scx=--Khar}', "");
Expect(1, 68185, '\P{Is_Scx=--Khar}', "");
Expect(0, 68185, '\P{^Is_Scx=--Khar}', "");
Error('\p{Script_Extensions:  lycian/a/}');
Error('\P{Script_Extensions:  lycian/a/}');
Expect(1, 66204, '\p{Script_Extensions=lycian}', "");
Expect(0, 66204, '\p{^Script_Extensions=lycian}', "");
Expect(0, 66204, '\P{Script_Extensions=lycian}', "");
Expect(1, 66204, '\P{^Script_Extensions=lycian}', "");
Expect(0, 66205, '\p{Script_Extensions=lycian}', "");
Expect(1, 66205, '\p{^Script_Extensions=lycian}', "");
Expect(1, 66205, '\P{Script_Extensions=lycian}', "");
Expect(0, 66205, '\P{^Script_Extensions=lycian}', "");
Expect(1, 66204, '\p{Script_Extensions:   _	Lycian}', "");
Expect(0, 66204, '\p{^Script_Extensions:   _	Lycian}', "");
Expect(0, 66204, '\P{Script_Extensions:   _	Lycian}', "");
Expect(1, 66204, '\P{^Script_Extensions:   _	Lycian}', "");
Expect(0, 66205, '\p{Script_Extensions:   _	Lycian}', "");
Expect(1, 66205, '\p{^Script_Extensions:   _	Lycian}', "");
Expect(1, 66205, '\P{Script_Extensions:   _	Lycian}', "");
Expect(0, 66205, '\P{^Script_Extensions:   _	Lycian}', "");
Error('\p{Scx=/a/	Lyci}');
Error('\P{Scx=/a/	Lyci}');
Expect(1, 66204, '\p{Scx=lyci}', "");
Expect(0, 66204, '\p{^Scx=lyci}', "");
Expect(0, 66204, '\P{Scx=lyci}', "");
Expect(1, 66204, '\P{^Scx=lyci}', "");
Expect(0, 66205, '\p{Scx=lyci}', "");
Expect(1, 66205, '\p{^Scx=lyci}', "");
Expect(1, 66205, '\P{Scx=lyci}', "");
Expect(0, 66205, '\P{^Scx=lyci}', "");
Expect(1, 66204, '\p{Scx=  LYCI}', "");
Expect(0, 66204, '\p{^Scx=  LYCI}', "");
Expect(0, 66204, '\P{Scx=  LYCI}', "");
Expect(1, 66204, '\P{^Scx=  LYCI}', "");
Expect(0, 66205, '\p{Scx=  LYCI}', "");
Expect(1, 66205, '\p{^Scx=  LYCI}', "");
Expect(1, 66205, '\P{Scx=  LYCI}', "");
Expect(0, 66205, '\P{^Scx=  LYCI}', "");
Error('\p{Is_Script_Extensions=/a/--lycian}');
Error('\P{Is_Script_Extensions=/a/--lycian}');
Expect(1, 66204, '\p{Is_Script_Extensions=lycian}', "");
Expect(0, 66204, '\p{^Is_Script_Extensions=lycian}', "");
Expect(0, 66204, '\P{Is_Script_Extensions=lycian}', "");
Expect(1, 66204, '\P{^Is_Script_Extensions=lycian}', "");
Expect(0, 66205, '\p{Is_Script_Extensions=lycian}', "");
Expect(1, 66205, '\p{^Is_Script_Extensions=lycian}', "");
Expect(1, 66205, '\P{Is_Script_Extensions=lycian}', "");
Expect(0, 66205, '\P{^Is_Script_Extensions=lycian}', "");
Expect(1, 66204, '\p{Is_Script_Extensions=- Lycian}', "");
Expect(0, 66204, '\p{^Is_Script_Extensions=- Lycian}', "");
Expect(0, 66204, '\P{Is_Script_Extensions=- Lycian}', "");
Expect(1, 66204, '\P{^Is_Script_Extensions=- Lycian}', "");
Expect(0, 66205, '\p{Is_Script_Extensions=- Lycian}', "");
Expect(1, 66205, '\p{^Is_Script_Extensions=- Lycian}', "");
Expect(1, 66205, '\P{Is_Script_Extensions=- Lycian}', "");
Expect(0, 66205, '\P{^Is_Script_Extensions=- Lycian}', "");
Error('\p{Is_Scx=:=_	Lyci}');
Error('\P{Is_Scx=:=_	Lyci}');
Expect(1, 66204, '\p{Is_Scx=lyci}', "");
Expect(0, 66204, '\p{^Is_Scx=lyci}', "");
Expect(0, 66204, '\P{Is_Scx=lyci}', "");
Expect(1, 66204, '\P{^Is_Scx=lyci}', "");
Expect(0, 66205, '\p{Is_Scx=lyci}', "");
Expect(1, 66205, '\p{^Is_Scx=lyci}', "");
Expect(1, 66205, '\P{Is_Scx=lyci}', "");
Expect(0, 66205, '\P{^Is_Scx=lyci}', "");
Expect(1, 66204, '\p{Is_Scx=_LYCI}', "");
Expect(0, 66204, '\p{^Is_Scx=_LYCI}', "");
Expect(0, 66204, '\P{Is_Scx=_LYCI}', "");
Expect(1, 66204, '\P{^Is_Scx=_LYCI}', "");
Expect(0, 66205, '\p{Is_Scx=_LYCI}', "");
Expect(1, 66205, '\p{^Is_Scx=_LYCI}', "");
Expect(1, 66205, '\P{Is_Scx=_LYCI}', "");
Expect(0, 66205, '\P{^Is_Scx=_LYCI}', "");
Error('\p{Script_Extensions=-/a/bopomofo}');
Error('\P{Script_Extensions=-/a/bopomofo}');
Expect(1, 65381, '\p{Script_Extensions:	bopomofo}', "");
Expect(0, 65381, '\p{^Script_Extensions:	bopomofo}', "");
Expect(0, 65381, '\P{Script_Extensions:	bopomofo}', "");
Expect(1, 65381, '\P{^Script_Extensions:	bopomofo}', "");
Expect(0, 65382, '\p{Script_Extensions:	bopomofo}', "");
Expect(1, 65382, '\p{^Script_Extensions:	bopomofo}', "");
Expect(1, 65382, '\P{Script_Extensions:	bopomofo}', "");
Expect(0, 65382, '\P{^Script_Extensions:	bopomofo}', "");
Expect(1, 65381, '\p{Script_Extensions= -Bopomofo}', "");
Expect(0, 65381, '\p{^Script_Extensions= -Bopomofo}', "");
Expect(0, 65381, '\P{Script_Extensions= -Bopomofo}', "");
Expect(1, 65381, '\P{^Script_Extensions= -Bopomofo}', "");
Expect(0, 65382, '\p{Script_Extensions= -Bopomofo}', "");
Expect(1, 65382, '\p{^Script_Extensions= -Bopomofo}', "");
Expect(1, 65382, '\P{Script_Extensions= -Bopomofo}', "");
Expect(0, 65382, '\P{^Script_Extensions= -Bopomofo}', "");
Error('\p{Scx= 	Bopo/a/}');
Error('\P{Scx= 	Bopo/a/}');
Expect(1, 65381, '\p{Scx=bopo}', "");
Expect(0, 65381, '\p{^Scx=bopo}', "");
Expect(0, 65381, '\P{Scx=bopo}', "");
Expect(1, 65381, '\P{^Scx=bopo}', "");
Expect(0, 65382, '\p{Scx=bopo}', "");
Expect(1, 65382, '\p{^Scx=bopo}', "");
Expect(1, 65382, '\P{Scx=bopo}', "");
Expect(0, 65382, '\P{^Scx=bopo}', "");
Expect(1, 65381, '\p{Scx=--bopo}', "");
Expect(0, 65381, '\p{^Scx=--bopo}', "");
Expect(0, 65381, '\P{Scx=--bopo}', "");
Expect(1, 65381, '\P{^Scx=--bopo}', "");
Expect(0, 65382, '\p{Scx=--bopo}', "");
Expect(1, 65382, '\p{^Scx=--bopo}', "");
Expect(1, 65382, '\P{Scx=--bopo}', "");
Expect(0, 65382, '\P{^Scx=--bopo}', "");
Error('\p{Is_Script_Extensions:   	:=bopomofo}');
Error('\P{Is_Script_Extensions:   	:=bopomofo}');
Expect(1, 65381, '\p{Is_Script_Extensions=bopomofo}', "");
Expect(0, 65381, '\p{^Is_Script_Extensions=bopomofo}', "");
Expect(0, 65381, '\P{Is_Script_Extensions=bopomofo}', "");
Expect(1, 65381, '\P{^Is_Script_Extensions=bopomofo}', "");
Expect(0, 65382, '\p{Is_Script_Extensions=bopomofo}', "");
Expect(1, 65382, '\p{^Is_Script_Extensions=bopomofo}', "");
Expect(1, 65382, '\P{Is_Script_Extensions=bopomofo}', "");
Expect(0, 65382, '\P{^Is_Script_Extensions=bopomofo}', "");
Expect(1, 65381, '\p{Is_Script_Extensions= _bopomofo}', "");
Expect(0, 65381, '\p{^Is_Script_Extensions= _bopomofo}', "");
Expect(0, 65381, '\P{Is_Script_Extensions= _bopomofo}', "");
Expect(1, 65381, '\P{^Is_Script_Extensions= _bopomofo}', "");
Expect(0, 65382, '\p{Is_Script_Extensions= _bopomofo}', "");
Expect(1, 65382, '\p{^Is_Script_Extensions= _bopomofo}', "");
Expect(1, 65382, '\P{Is_Script_Extensions= _bopomofo}', "");
Expect(0, 65382, '\P{^Is_Script_Extensions= _bopomofo}', "");
Error('\p{Is_Scx=  Bopo/a/}');
Error('\P{Is_Scx=  Bopo/a/}');
Expect(1, 65381, '\p{Is_Scx=bopo}', "");
Expect(0, 65381, '\p{^Is_Scx=bopo}', "");
Expect(0, 65381, '\P{Is_Scx=bopo}', "");
Expect(1, 65381, '\P{^Is_Scx=bopo}', "");
Expect(0, 65382, '\p{Is_Scx=bopo}', "");
Expect(1, 65382, '\p{^Is_Scx=bopo}', "");
Expect(1, 65382, '\P{Is_Scx=bopo}', "");
Expect(0, 65382, '\P{^Is_Scx=bopo}', "");
Expect(1, 65381, '\p{Is_Scx= _BOPO}', "");
Expect(0, 65381, '\p{^Is_Scx= _BOPO}', "");
Expect(0, 65381, '\P{Is_Scx= _BOPO}', "");
Expect(1, 65381, '\P{^Is_Scx= _BOPO}', "");
Expect(0, 65382, '\p{Is_Scx= _BOPO}', "");
Expect(1, 65382, '\p{^Is_Scx= _BOPO}', "");
Expect(1, 65382, '\P{Is_Scx= _BOPO}', "");
Expect(0, 65382, '\P{^Is_Scx= _BOPO}', "");
Error('\p{Script_Extensions= /a/Meroitic_Cursive}');
Error('\P{Script_Extensions= /a/Meroitic_Cursive}');
Expect(1, 68031, '\p{Script_Extensions=meroiticcursive}', "");
Expect(0, 68031, '\p{^Script_Extensions=meroiticcursive}', "");
Expect(0, 68031, '\P{Script_Extensions=meroiticcursive}', "");
Expect(1, 68031, '\P{^Script_Extensions=meroiticcursive}', "");
Expect(0, 68032, '\p{Script_Extensions=meroiticcursive}', "");
Expect(1, 68032, '\p{^Script_Extensions=meroiticcursive}', "");
Expect(1, 68032, '\P{Script_Extensions=meroiticcursive}', "");
Expect(0, 68032, '\P{^Script_Extensions=meroiticcursive}', "");
Expect(1, 68031, '\p{Script_Extensions:	meroitic_CURSIVE}', "");
Expect(0, 68031, '\p{^Script_Extensions:	meroitic_CURSIVE}', "");
Expect(0, 68031, '\P{Script_Extensions:	meroitic_CURSIVE}', "");
Expect(1, 68031, '\P{^Script_Extensions:	meroitic_CURSIVE}', "");
Expect(0, 68032, '\p{Script_Extensions:	meroitic_CURSIVE}', "");
Expect(1, 68032, '\p{^Script_Extensions:	meroitic_CURSIVE}', "");
Expect(1, 68032, '\P{Script_Extensions:	meroitic_CURSIVE}', "");
Expect(0, 68032, '\P{^Script_Extensions:	meroitic_CURSIVE}', "");
Error('\p{Scx=:=Merc}');
Error('\P{Scx=:=Merc}');
Expect(1, 68031, '\p{Scx=merc}', "");
Expect(0, 68031, '\p{^Scx=merc}', "");
Expect(0, 68031, '\P{Scx=merc}', "");
Expect(1, 68031, '\P{^Scx=merc}', "");
Expect(0, 68032, '\p{Scx=merc}', "");
Expect(1, 68032, '\p{^Scx=merc}', "");
Expect(1, 68032, '\P{Scx=merc}', "");
Expect(0, 68032, '\P{^Scx=merc}', "");
Expect(1, 68031, '\p{Scx=	Merc}', "");
Expect(0, 68031, '\p{^Scx=	Merc}', "");
Expect(0, 68031, '\P{Scx=	Merc}', "");
Expect(1, 68031, '\P{^Scx=	Merc}', "");
Expect(0, 68032, '\p{Scx=	Merc}', "");
Expect(1, 68032, '\p{^Scx=	Merc}', "");
Expect(1, 68032, '\P{Scx=	Merc}', "");
Expect(0, 68032, '\P{^Scx=	Merc}', "");
Error('\p{Is_Script_Extensions=/a/MEROITIC_Cursive}');
Error('\P{Is_Script_Extensions=/a/MEROITIC_Cursive}');
Expect(1, 68031, '\p{Is_Script_Extensions=meroiticcursive}', "");
Expect(0, 68031, '\p{^Is_Script_Extensions=meroiticcursive}', "");
Expect(0, 68031, '\P{Is_Script_Extensions=meroiticcursive}', "");
Expect(1, 68031, '\P{^Is_Script_Extensions=meroiticcursive}', "");
Expect(0, 68032, '\p{Is_Script_Extensions=meroiticcursive}', "");
Expect(1, 68032, '\p{^Is_Script_Extensions=meroiticcursive}', "");
Expect(1, 68032, '\P{Is_Script_Extensions=meroiticcursive}', "");
Expect(0, 68032, '\P{^Is_Script_Extensions=meroiticcursive}', "");
Expect(1, 68031, '\p{Is_Script_Extensions:		Meroitic_CURSIVE}', "");
Expect(0, 68031, '\p{^Is_Script_Extensions:		Meroitic_CURSIVE}', "");
Expect(0, 68031, '\P{Is_Script_Extensions:		Meroitic_CURSIVE}', "");
Expect(1, 68031, '\P{^Is_Script_Extensions:		Meroitic_CURSIVE}', "");
Expect(0, 68032, '\p{Is_Script_Extensions:		Meroitic_CURSIVE}', "");
Expect(1, 68032, '\p{^Is_Script_Extensions:		Meroitic_CURSIVE}', "");
Expect(1, 68032, '\P{Is_Script_Extensions:		Meroitic_CURSIVE}', "");
Expect(0, 68032, '\P{^Is_Script_Extensions:		Meroitic_CURSIVE}', "");
Error('\p{Is_Scx=	/a/merc}');
Error('\P{Is_Scx=	/a/merc}');
Expect(1, 68031, '\p{Is_Scx=merc}', "");
Expect(0, 68031, '\p{^Is_Scx=merc}', "");
Expect(0, 68031, '\P{Is_Scx=merc}', "");
Expect(1, 68031, '\P{^Is_Scx=merc}', "");
Expect(0, 68032, '\p{Is_Scx=merc}', "");
Expect(1, 68032, '\p{^Is_Scx=merc}', "");
Expect(1, 68032, '\P{Is_Scx=merc}', "");
Expect(0, 68032, '\P{^Is_Scx=merc}', "");
Expect(1, 68031, '\p{Is_Scx:		-Merc}', "");
Expect(0, 68031, '\p{^Is_Scx:		-Merc}', "");
Expect(0, 68031, '\P{Is_Scx:		-Merc}', "");
Expect(1, 68031, '\P{^Is_Scx:		-Merc}', "");
Expect(0, 68032, '\p{Is_Scx:		-Merc}', "");
Expect(1, 68032, '\p{^Is_Scx:		-Merc}', "");
Expect(1, 68032, '\P{Is_Scx:		-Merc}', "");
Expect(0, 68032, '\P{^Is_Scx:		-Merc}', "");
Error('\p{Script_Extensions=-_Rejang/a/}');
Error('\P{Script_Extensions=-_Rejang/a/}');
Expect(1, 43359, '\p{Script_Extensions=rejang}', "");
Expect(0, 43359, '\p{^Script_Extensions=rejang}', "");
Expect(0, 43359, '\P{Script_Extensions=rejang}', "");
Expect(1, 43359, '\P{^Script_Extensions=rejang}', "");
Expect(0, 43360, '\p{Script_Extensions=rejang}', "");
Expect(1, 43360, '\p{^Script_Extensions=rejang}', "");
Expect(1, 43360, '\P{Script_Extensions=rejang}', "");
Expect(0, 43360, '\P{^Script_Extensions=rejang}', "");
Expect(1, 43359, '\p{Script_Extensions=_Rejang}', "");
Expect(0, 43359, '\p{^Script_Extensions=_Rejang}', "");
Expect(0, 43359, '\P{Script_Extensions=_Rejang}', "");
Expect(1, 43359, '\P{^Script_Extensions=_Rejang}', "");
Expect(0, 43360, '\p{Script_Extensions=_Rejang}', "");
Expect(1, 43360, '\p{^Script_Extensions=_Rejang}', "");
Expect(1, 43360, '\P{Script_Extensions=_Rejang}', "");
Expect(0, 43360, '\P{^Script_Extensions=_Rejang}', "");
Error('\p{Scx=:=Rjng}');
Error('\P{Scx=:=Rjng}');
Expect(1, 43359, '\p{Scx:rjng}', "");
Expect(0, 43359, '\p{^Scx:rjng}', "");
Expect(0, 43359, '\P{Scx:rjng}', "");
Expect(1, 43359, '\P{^Scx:rjng}', "");
Expect(0, 43360, '\p{Scx:rjng}', "");
Expect(1, 43360, '\p{^Scx:rjng}', "");
Expect(1, 43360, '\P{Scx:rjng}', "");
Expect(0, 43360, '\P{^Scx:rjng}', "");
Expect(1, 43359, '\p{Scx=		RJNG}', "");
Expect(0, 43359, '\p{^Scx=		RJNG}', "");
Expect(0, 43359, '\P{Scx=		RJNG}', "");
Expect(1, 43359, '\P{^Scx=		RJNG}', "");
Expect(0, 43360, '\p{Scx=		RJNG}', "");
Expect(1, 43360, '\p{^Scx=		RJNG}', "");
Expect(1, 43360, '\P{Scx=		RJNG}', "");
Expect(0, 43360, '\P{^Scx=		RJNG}', "");
Error('\p{Is_Script_Extensions=-rejang/a/}');
Error('\P{Is_Script_Extensions=-rejang/a/}');
Expect(1, 43359, '\p{Is_Script_Extensions=rejang}', "");
Expect(0, 43359, '\p{^Is_Script_Extensions=rejang}', "");
Expect(0, 43359, '\P{Is_Script_Extensions=rejang}', "");
Expect(1, 43359, '\P{^Is_Script_Extensions=rejang}', "");
Expect(0, 43360, '\p{Is_Script_Extensions=rejang}', "");
Expect(1, 43360, '\p{^Is_Script_Extensions=rejang}', "");
Expect(1, 43360, '\P{Is_Script_Extensions=rejang}', "");
Expect(0, 43360, '\P{^Is_Script_Extensions=rejang}', "");
Expect(1, 43359, '\p{Is_Script_Extensions:		-REJANG}', "");
Expect(0, 43359, '\p{^Is_Script_Extensions:		-REJANG}', "");
Expect(0, 43359, '\P{Is_Script_Extensions:		-REJANG}', "");
Expect(1, 43359, '\P{^Is_Script_Extensions:		-REJANG}', "");
Expect(0, 43360, '\p{Is_Script_Extensions:		-REJANG}', "");
Expect(1, 43360, '\p{^Is_Script_Extensions:		-REJANG}', "");
Expect(1, 43360, '\P{Is_Script_Extensions:		-REJANG}', "");
Expect(0, 43360, '\P{^Is_Script_Extensions:		-REJANG}', "");
Error('\p{Is_Scx=	:=RJNG}');
Error('\P{Is_Scx=	:=RJNG}');
Expect(1, 43359, '\p{Is_Scx=rjng}', "");
Expect(0, 43359, '\p{^Is_Scx=rjng}', "");
Expect(0, 43359, '\P{Is_Scx=rjng}', "");
Expect(1, 43359, '\P{^Is_Scx=rjng}', "");
Expect(0, 43360, '\p{Is_Scx=rjng}', "");
Expect(1, 43360, '\p{^Is_Scx=rjng}', "");
Expect(1, 43360, '\P{Is_Scx=rjng}', "");
Expect(0, 43360, '\P{^Is_Scx=rjng}', "");
Expect(1, 43359, '\p{Is_Scx=-Rjng}', "");
Expect(0, 43359, '\p{^Is_Scx=-Rjng}', "");
Expect(0, 43359, '\P{Is_Scx=-Rjng}', "");
Expect(1, 43359, '\P{^Is_Scx=-Rjng}', "");
Expect(0, 43360, '\p{Is_Scx=-Rjng}', "");
Expect(1, 43360, '\p{^Is_Scx=-Rjng}', "");
Expect(1, 43360, '\P{Is_Scx=-Rjng}', "");
Expect(0, 43360, '\P{^Is_Scx=-Rjng}', "");
Error('\p{Script_Extensions=-	hanunoo/a/}');
Error('\P{Script_Extensions=-	hanunoo/a/}');
Expect(1, 5942, '\p{Script_Extensions=hanunoo}', "");
Expect(0, 5942, '\p{^Script_Extensions=hanunoo}', "");
Expect(0, 5942, '\P{Script_Extensions=hanunoo}', "");
Expect(1, 5942, '\P{^Script_Extensions=hanunoo}', "");
Expect(0, 5943, '\p{Script_Extensions=hanunoo}', "");
Expect(1, 5943, '\p{^Script_Extensions=hanunoo}', "");
Expect(1, 5943, '\P{Script_Extensions=hanunoo}', "");
Expect(0, 5943, '\P{^Script_Extensions=hanunoo}', "");
Expect(1, 5942, '\p{Script_Extensions=-_Hanunoo}', "");
Expect(0, 5942, '\p{^Script_Extensions=-_Hanunoo}', "");
Expect(0, 5942, '\P{Script_Extensions=-_Hanunoo}', "");
Expect(1, 5942, '\P{^Script_Extensions=-_Hanunoo}', "");
Expect(0, 5943, '\p{Script_Extensions=-_Hanunoo}', "");
Expect(1, 5943, '\p{^Script_Extensions=-_Hanunoo}', "");
Expect(1, 5943, '\P{Script_Extensions=-_Hanunoo}', "");
Expect(0, 5943, '\P{^Script_Extensions=-_Hanunoo}', "");
Error('\p{Scx=:=- Hano}');
Error('\P{Scx=:=- Hano}');
Expect(1, 5942, '\p{Scx=hano}', "");
Expect(0, 5942, '\p{^Scx=hano}', "");
Expect(0, 5942, '\P{Scx=hano}', "");
Expect(1, 5942, '\P{^Scx=hano}', "");
Expect(0, 5943, '\p{Scx=hano}', "");
Expect(1, 5943, '\p{^Scx=hano}', "");
Expect(1, 5943, '\P{Scx=hano}', "");
Expect(0, 5943, '\P{^Scx=hano}', "");
Expect(1, 5942, '\p{Scx: 	_Hano}', "");
Expect(0, 5942, '\p{^Scx: 	_Hano}', "");
Expect(0, 5942, '\P{Scx: 	_Hano}', "");
Expect(1, 5942, '\P{^Scx: 	_Hano}', "");
Expect(0, 5943, '\p{Scx: 	_Hano}', "");
Expect(1, 5943, '\p{^Scx: 	_Hano}', "");
Expect(1, 5943, '\P{Scx: 	_Hano}', "");
Expect(0, 5943, '\P{^Scx: 	_Hano}', "");
Error('\p{Is_Script_Extensions=-Hanunoo/a/}');
Error('\P{Is_Script_Extensions=-Hanunoo/a/}');
Expect(1, 5942, '\p{Is_Script_Extensions=hanunoo}', "");
Expect(0, 5942, '\p{^Is_Script_Extensions=hanunoo}', "");
Expect(0, 5942, '\P{Is_Script_Extensions=hanunoo}', "");
Expect(1, 5942, '\P{^Is_Script_Extensions=hanunoo}', "");
Expect(0, 5943, '\p{Is_Script_Extensions=hanunoo}', "");
Expect(1, 5943, '\p{^Is_Script_Extensions=hanunoo}', "");
Expect(1, 5943, '\P{Is_Script_Extensions=hanunoo}', "");
Expect(0, 5943, '\P{^Is_Script_Extensions=hanunoo}', "");
Expect(1, 5942, '\p{Is_Script_Extensions=--hanunoo}', "");
Expect(0, 5942, '\p{^Is_Script_Extensions=--hanunoo}', "");
Expect(0, 5942, '\P{Is_Script_Extensions=--hanunoo}', "");
Expect(1, 5942, '\P{^Is_Script_Extensions=--hanunoo}', "");
Expect(0, 5943, '\p{Is_Script_Extensions=--hanunoo}', "");
Expect(1, 5943, '\p{^Is_Script_Extensions=--hanunoo}', "");
Expect(1, 5943, '\P{Is_Script_Extensions=--hanunoo}', "");
Expect(0, 5943, '\P{^Is_Script_Extensions=--hanunoo}', "");
Error('\p{Is_Scx=-:=HANO}');
Error('\P{Is_Scx=-:=HANO}');
Expect(1, 5942, '\p{Is_Scx=hano}', "");
Expect(0, 5942, '\p{^Is_Scx=hano}', "");
Expect(0, 5942, '\P{Is_Scx=hano}', "");
Expect(1, 5942, '\P{^Is_Scx=hano}', "");
Expect(0, 5943, '\p{Is_Scx=hano}', "");
Expect(1, 5943, '\p{^Is_Scx=hano}', "");
Expect(1, 5943, '\P{Is_Scx=hano}', "");
Expect(0, 5943, '\P{^Is_Scx=hano}', "");
Expect(1, 5942, '\p{Is_Scx= Hano}', "");
Expect(0, 5942, '\p{^Is_Scx= Hano}', "");
Expect(0, 5942, '\P{Is_Scx= Hano}', "");
Expect(1, 5942, '\P{^Is_Scx= Hano}', "");
Expect(0, 5943, '\p{Is_Scx= Hano}', "");
Expect(1, 5943, '\p{^Is_Scx= Hano}', "");
Expect(1, 5943, '\P{Is_Scx= Hano}', "");
Expect(0, 5943, '\P{^Is_Scx= Hano}', "");
Error('\p{Script_Extensions=:=-	IMPERIAL_ARAMAIC}');
Error('\P{Script_Extensions=:=-	IMPERIAL_ARAMAIC}');
Expect(1, 67679, '\p{Script_Extensions=imperialaramaic}', "");
Expect(0, 67679, '\p{^Script_Extensions=imperialaramaic}', "");
Expect(0, 67679, '\P{Script_Extensions=imperialaramaic}', "");
Expect(1, 67679, '\P{^Script_Extensions=imperialaramaic}', "");
Expect(0, 67680, '\p{Script_Extensions=imperialaramaic}', "");
Expect(1, 67680, '\p{^Script_Extensions=imperialaramaic}', "");
Expect(1, 67680, '\P{Script_Extensions=imperialaramaic}', "");
Expect(0, 67680, '\P{^Script_Extensions=imperialaramaic}', "");
Expect(1, 67679, '\p{Script_Extensions=_imperial_ARAMAIC}', "");
Expect(0, 67679, '\p{^Script_Extensions=_imperial_ARAMAIC}', "");
Expect(0, 67679, '\P{Script_Extensions=_imperial_ARAMAIC}', "");
Expect(1, 67679, '\P{^Script_Extensions=_imperial_ARAMAIC}', "");
Expect(0, 67680, '\p{Script_Extensions=_imperial_ARAMAIC}', "");
Expect(1, 67680, '\p{^Script_Extensions=_imperial_ARAMAIC}', "");
Expect(1, 67680, '\P{Script_Extensions=_imperial_ARAMAIC}', "");
Expect(0, 67680, '\P{^Script_Extensions=_imperial_ARAMAIC}', "");
Error('\p{Scx= /a/Armi}');
Error('\P{Scx= /a/Armi}');
Expect(1, 67679, '\p{Scx=armi}', "");
Expect(0, 67679, '\p{^Scx=armi}', "");
Expect(0, 67679, '\P{Scx=armi}', "");
Expect(1, 67679, '\P{^Scx=armi}', "");
Expect(0, 67680, '\p{Scx=armi}', "");
Expect(1, 67680, '\p{^Scx=armi}', "");
Expect(1, 67680, '\P{Scx=armi}', "");
Expect(0, 67680, '\P{^Scx=armi}', "");
Expect(1, 67679, '\p{Scx= Armi}', "");
Expect(0, 67679, '\p{^Scx= Armi}', "");
Expect(0, 67679, '\P{Scx= Armi}', "");
Expect(1, 67679, '\P{^Scx= Armi}', "");
Expect(0, 67680, '\p{Scx= Armi}', "");
Expect(1, 67680, '\p{^Scx= Armi}', "");
Expect(1, 67680, '\P{Scx= Armi}', "");
Expect(0, 67680, '\P{^Scx= Armi}', "");
Error('\p{Is_Script_Extensions= _IMPERIAL_aramaic/a/}');
Error('\P{Is_Script_Extensions= _IMPERIAL_aramaic/a/}');
Expect(1, 67679, '\p{Is_Script_Extensions=imperialaramaic}', "");
Expect(0, 67679, '\p{^Is_Script_Extensions=imperialaramaic}', "");
Expect(0, 67679, '\P{Is_Script_Extensions=imperialaramaic}', "");
Expect(1, 67679, '\P{^Is_Script_Extensions=imperialaramaic}', "");
Expect(0, 67680, '\p{Is_Script_Extensions=imperialaramaic}', "");
Expect(1, 67680, '\p{^Is_Script_Extensions=imperialaramaic}', "");
Expect(1, 67680, '\P{Is_Script_Extensions=imperialaramaic}', "");
Expect(0, 67680, '\P{^Is_Script_Extensions=imperialaramaic}', "");
Expect(1, 67679, '\p{Is_Script_Extensions=- Imperial_Aramaic}', "");
Expect(0, 67679, '\p{^Is_Script_Extensions=- Imperial_Aramaic}', "");
Expect(0, 67679, '\P{Is_Script_Extensions=- Imperial_Aramaic}', "");
Expect(1, 67679, '\P{^Is_Script_Extensions=- Imperial_Aramaic}', "");
Expect(0, 67680, '\p{Is_Script_Extensions=- Imperial_Aramaic}', "");
Expect(1, 67680, '\p{^Is_Script_Extensions=- Imperial_Aramaic}', "");
Expect(1, 67680, '\P{Is_Script_Extensions=- Imperial_Aramaic}', "");
Expect(0, 67680, '\P{^Is_Script_Extensions=- Imperial_Aramaic}', "");
Error('\p{Is_Scx=/a/_Armi}');
Error('\P{Is_Scx=/a/_Armi}');
Expect(1, 67679, '\p{Is_Scx=armi}', "");
Expect(0, 67679, '\p{^Is_Scx=armi}', "");
Expect(0, 67679, '\P{Is_Scx=armi}', "");
Expect(1, 67679, '\P{^Is_Scx=armi}', "");
Expect(0, 67680, '\p{Is_Scx=armi}', "");
Expect(1, 67680, '\p{^Is_Scx=armi}', "");
Expect(1, 67680, '\P{Is_Scx=armi}', "");
Expect(0, 67680, '\P{^Is_Scx=armi}', "");
Expect(1, 67679, '\p{Is_Scx=--armi}', "");
Expect(0, 67679, '\p{^Is_Scx=--armi}', "");
Expect(0, 67679, '\P{Is_Scx=--armi}', "");
Expect(1, 67679, '\P{^Is_Scx=--armi}', "");
Expect(0, 67680, '\p{Is_Scx=--armi}', "");
Expect(1, 67680, '\p{^Is_Scx=--armi}', "");
Expect(1, 67680, '\P{Is_Scx=--armi}', "");
Expect(0, 67680, '\P{^Is_Scx=--armi}', "");
Error('\p{Script_Extensions=:=	 kannada}');
Error('\P{Script_Extensions=:=	 kannada}');
Expect(1, 3314, '\p{Script_Extensions=kannada}', "");
Expect(0, 3314, '\p{^Script_Extensions=kannada}', "");
Expect(0, 3314, '\P{Script_Extensions=kannada}', "");
Expect(1, 3314, '\P{^Script_Extensions=kannada}', "");
Expect(0, 3315, '\p{Script_Extensions=kannada}', "");
Expect(1, 3315, '\p{^Script_Extensions=kannada}', "");
Expect(1, 3315, '\P{Script_Extensions=kannada}', "");
Expect(0, 3315, '\P{^Script_Extensions=kannada}', "");
Expect(1, 3314, '\p{Script_Extensions:-	Kannada}', "");
Expect(0, 3314, '\p{^Script_Extensions:-	Kannada}', "");
Expect(0, 3314, '\P{Script_Extensions:-	Kannada}', "");
Expect(1, 3314, '\P{^Script_Extensions:-	Kannada}', "");
Expect(0, 3315, '\p{Script_Extensions:-	Kannada}', "");
Expect(1, 3315, '\p{^Script_Extensions:-	Kannada}', "");
Expect(1, 3315, '\P{Script_Extensions:-	Kannada}', "");
Expect(0, 3315, '\P{^Script_Extensions:-	Kannada}', "");
Error('\p{Scx=/a/__Knda}');
Error('\P{Scx=/a/__Knda}');
Expect(1, 3314, '\p{Scx=knda}', "");
Expect(0, 3314, '\p{^Scx=knda}', "");
Expect(0, 3314, '\P{Scx=knda}', "");
Expect(1, 3314, '\P{^Scx=knda}', "");
Expect(0, 3315, '\p{Scx=knda}', "");
Expect(1, 3315, '\p{^Scx=knda}', "");
Expect(1, 3315, '\P{Scx=knda}', "");
Expect(0, 3315, '\P{^Scx=knda}', "");
Expect(1, 3314, '\p{Scx=	knda}', "");
Expect(0, 3314, '\p{^Scx=	knda}', "");
Expect(0, 3314, '\P{Scx=	knda}', "");
Expect(1, 3314, '\P{^Scx=	knda}', "");
Expect(0, 3315, '\p{Scx=	knda}', "");
Expect(1, 3315, '\p{^Scx=	knda}', "");
Expect(1, 3315, '\P{Scx=	knda}', "");
Expect(0, 3315, '\P{^Scx=	knda}', "");
Error('\p{Is_Script_Extensions=/a/_-Kannada}');
Error('\P{Is_Script_Extensions=/a/_-Kannada}');
Expect(1, 3314, '\p{Is_Script_Extensions=kannada}', "");
Expect(0, 3314, '\p{^Is_Script_Extensions=kannada}', "");
Expect(0, 3314, '\P{Is_Script_Extensions=kannada}', "");
Expect(1, 3314, '\P{^Is_Script_Extensions=kannada}', "");
Expect(0, 3315, '\p{Is_Script_Extensions=kannada}', "");
Expect(1, 3315, '\p{^Is_Script_Extensions=kannada}', "");
Expect(1, 3315, '\P{Is_Script_Extensions=kannada}', "");
Expect(0, 3315, '\P{^Is_Script_Extensions=kannada}', "");
Expect(1, 3314, '\p{Is_Script_Extensions=_ kannada}', "");
Expect(0, 3314, '\p{^Is_Script_Extensions=_ kannada}', "");
Expect(0, 3314, '\P{Is_Script_Extensions=_ kannada}', "");
Expect(1, 3314, '\P{^Is_Script_Extensions=_ kannada}', "");
Expect(0, 3315, '\p{Is_Script_Extensions=_ kannada}', "");
Expect(1, 3315, '\p{^Is_Script_Extensions=_ kannada}', "");
Expect(1, 3315, '\P{Is_Script_Extensions=_ kannada}', "");
Expect(0, 3315, '\P{^Is_Script_Extensions=_ kannada}', "");
Error('\p{Is_Scx:	:=KNDA}');
Error('\P{Is_Scx:	:=KNDA}');
Expect(1, 3314, '\p{Is_Scx=knda}', "");
Expect(0, 3314, '\p{^Is_Scx=knda}', "");
Expect(0, 3314, '\P{Is_Scx=knda}', "");
Expect(1, 3314, '\P{^Is_Scx=knda}', "");
Expect(0, 3315, '\p{Is_Scx=knda}', "");
Expect(1, 3315, '\p{^Is_Scx=knda}', "");
Expect(1, 3315, '\P{Is_Scx=knda}', "");
Expect(0, 3315, '\P{^Is_Scx=knda}', "");
Expect(1, 3314, '\p{Is_Scx=-KNDA}', "");
Expect(0, 3314, '\p{^Is_Scx=-KNDA}', "");
Expect(0, 3314, '\P{Is_Scx=-KNDA}', "");
Expect(1, 3314, '\P{^Is_Scx=-KNDA}', "");
Expect(0, 3315, '\p{Is_Scx=-KNDA}', "");
Expect(1, 3315, '\p{^Is_Scx=-KNDA}', "");
Expect(1, 3315, '\P{Is_Scx=-KNDA}', "");
Expect(0, 3315, '\P{^Is_Scx=-KNDA}', "");
Error('\p{Script_Extensions=-:=Khmer}');
Error('\P{Script_Extensions=-:=Khmer}');
Expect(1, 6655, '\p{Script_Extensions=khmer}', "");
Expect(0, 6655, '\p{^Script_Extensions=khmer}', "");
Expect(0, 6655, '\P{Script_Extensions=khmer}', "");
Expect(1, 6655, '\P{^Script_Extensions=khmer}', "");
Expect(0, 6656, '\p{Script_Extensions=khmer}', "");
Expect(1, 6656, '\p{^Script_Extensions=khmer}', "");
Expect(1, 6656, '\P{Script_Extensions=khmer}', "");
Expect(0, 6656, '\P{^Script_Extensions=khmer}', "");
Expect(1, 6655, '\p{Script_Extensions=_-Khmer}', "");
Expect(0, 6655, '\p{^Script_Extensions=_-Khmer}', "");
Expect(0, 6655, '\P{Script_Extensions=_-Khmer}', "");
Expect(1, 6655, '\P{^Script_Extensions=_-Khmer}', "");
Expect(0, 6656, '\p{Script_Extensions=_-Khmer}', "");
Expect(1, 6656, '\p{^Script_Extensions=_-Khmer}', "");
Expect(1, 6656, '\P{Script_Extensions=_-Khmer}', "");
Expect(0, 6656, '\P{^Script_Extensions=_-Khmer}', "");
Error('\p{Scx=/a/ Khmr}');
Error('\P{Scx=/a/ Khmr}');
Expect(1, 6655, '\p{Scx=khmr}', "");
Expect(0, 6655, '\p{^Scx=khmr}', "");
Expect(0, 6655, '\P{Scx=khmr}', "");
Expect(1, 6655, '\P{^Scx=khmr}', "");
Expect(0, 6656, '\p{Scx=khmr}', "");
Expect(1, 6656, '\p{^Scx=khmr}', "");
Expect(1, 6656, '\P{Scx=khmr}', "");
Expect(0, 6656, '\P{^Scx=khmr}', "");
Expect(1, 6655, '\p{Scx=__KHMR}', "");
Expect(0, 6655, '\p{^Scx=__KHMR}', "");
Expect(0, 6655, '\P{Scx=__KHMR}', "");
Expect(1, 6655, '\P{^Scx=__KHMR}', "");
Expect(0, 6656, '\p{Scx=__KHMR}', "");
Expect(1, 6656, '\p{^Scx=__KHMR}', "");
Expect(1, 6656, '\P{Scx=__KHMR}', "");
Expect(0, 6656, '\P{^Scx=__KHMR}', "");
Error('\p{Is_Script_Extensions=_-khmer/a/}');
Error('\P{Is_Script_Extensions=_-khmer/a/}');
Expect(1, 6655, '\p{Is_Script_Extensions=khmer}', "");
Expect(0, 6655, '\p{^Is_Script_Extensions=khmer}', "");
Expect(0, 6655, '\P{Is_Script_Extensions=khmer}', "");
Expect(1, 6655, '\P{^Is_Script_Extensions=khmer}', "");
Expect(0, 6656, '\p{Is_Script_Extensions=khmer}', "");
Expect(1, 6656, '\p{^Is_Script_Extensions=khmer}', "");
Expect(1, 6656, '\P{Is_Script_Extensions=khmer}', "");
Expect(0, 6656, '\P{^Is_Script_Extensions=khmer}', "");
Expect(1, 6655, '\p{Is_Script_Extensions=-	khmer}', "");
Expect(0, 6655, '\p{^Is_Script_Extensions=-	khmer}', "");
Expect(0, 6655, '\P{Is_Script_Extensions=-	khmer}', "");
Expect(1, 6655, '\P{^Is_Script_Extensions=-	khmer}', "");
Expect(0, 6656, '\p{Is_Script_Extensions=-	khmer}', "");
Expect(1, 6656, '\p{^Is_Script_Extensions=-	khmer}', "");
Expect(1, 6656, '\P{Is_Script_Extensions=-	khmer}', "");
Expect(0, 6656, '\P{^Is_Script_Extensions=-	khmer}', "");
Error('\p{Is_Scx=-:=Khmr}');
Error('\P{Is_Scx=-:=Khmr}');
Expect(1, 6655, '\p{Is_Scx=khmr}', "");
Expect(0, 6655, '\p{^Is_Scx=khmr}', "");
Expect(0, 6655, '\P{Is_Scx=khmr}', "");
Expect(1, 6655, '\P{^Is_Scx=khmr}', "");
Expect(0, 6656, '\p{Is_Scx=khmr}', "");
Expect(1, 6656, '\p{^Is_Scx=khmr}', "");
Expect(1, 6656, '\P{Is_Scx=khmr}', "");
Expect(0, 6656, '\P{^Is_Scx=khmr}', "");
Expect(1, 6655, '\p{Is_Scx:   KHMR}', "");
Expect(0, 6655, '\p{^Is_Scx:   KHMR}', "");
Expect(0, 6655, '\P{Is_Scx:   KHMR}', "");
Expect(1, 6655, '\P{^Is_Scx:   KHMR}', "");
Expect(0, 6656, '\p{Is_Scx:   KHMR}', "");
Expect(1, 6656, '\p{^Is_Scx:   KHMR}', "");
Expect(1, 6656, '\P{Is_Scx:   KHMR}', "");
Expect(0, 6656, '\P{^Is_Scx:   KHMR}', "");
Error('\p{Script_Extensions=/a/_	CHAM}');
Error('\P{Script_Extensions=/a/_	CHAM}');
Expect(1, 43615, '\p{Script_Extensions=cham}', "");
Expect(0, 43615, '\p{^Script_Extensions=cham}', "");
Expect(0, 43615, '\P{Script_Extensions=cham}', "");
Expect(1, 43615, '\P{^Script_Extensions=cham}', "");
Expect(0, 43616, '\p{Script_Extensions=cham}', "");
Expect(1, 43616, '\p{^Script_Extensions=cham}', "");
Expect(1, 43616, '\P{Script_Extensions=cham}', "");
Expect(0, 43616, '\P{^Script_Extensions=cham}', "");
Expect(1, 43615, '\p{Script_Extensions:   - Cham}', "");
Expect(0, 43615, '\p{^Script_Extensions:   - Cham}', "");
Expect(0, 43615, '\P{Script_Extensions:   - Cham}', "");
Expect(1, 43615, '\P{^Script_Extensions:   - Cham}', "");
Expect(0, 43616, '\p{Script_Extensions:   - Cham}', "");
Expect(1, 43616, '\p{^Script_Extensions:   - Cham}', "");
Expect(1, 43616, '\P{Script_Extensions:   - Cham}', "");
Expect(0, 43616, '\P{^Script_Extensions:   - Cham}', "");
Error('\p{Scx=--CHAM:=}');
Error('\P{Scx=--CHAM:=}');
Expect(1, 43615, '\p{Scx=cham}', "");
Expect(0, 43615, '\p{^Scx=cham}', "");
Expect(0, 43615, '\P{Scx=cham}', "");
Expect(1, 43615, '\P{^Scx=cham}', "");
Expect(0, 43616, '\p{Scx=cham}', "");
Expect(1, 43616, '\p{^Scx=cham}', "");
Expect(1, 43616, '\P{Scx=cham}', "");
Expect(0, 43616, '\P{^Scx=cham}', "");
Expect(1, 43615, '\p{Scx=_	Cham}', "");
Expect(0, 43615, '\p{^Scx=_	Cham}', "");
Expect(0, 43615, '\P{Scx=_	Cham}', "");
Expect(1, 43615, '\P{^Scx=_	Cham}', "");
Expect(0, 43616, '\p{Scx=_	Cham}', "");
Expect(1, 43616, '\p{^Scx=_	Cham}', "");
Expect(1, 43616, '\P{Scx=_	Cham}', "");
Expect(0, 43616, '\P{^Scx=_	Cham}', "");
Error('\p{Is_Script_Extensions=-:=cham}');
Error('\P{Is_Script_Extensions=-:=cham}');
Expect(1, 43615, '\p{Is_Script_Extensions=cham}', "");
Expect(0, 43615, '\p{^Is_Script_Extensions=cham}', "");
Expect(0, 43615, '\P{Is_Script_Extensions=cham}', "");
Expect(1, 43615, '\P{^Is_Script_Extensions=cham}', "");
Expect(0, 43616, '\p{Is_Script_Extensions=cham}', "");
Expect(1, 43616, '\p{^Is_Script_Extensions=cham}', "");
Expect(1, 43616, '\P{Is_Script_Extensions=cham}', "");
Expect(0, 43616, '\P{^Is_Script_Extensions=cham}', "");
Expect(1, 43615, '\p{Is_Script_Extensions=_Cham}', "");
Expect(0, 43615, '\p{^Is_Script_Extensions=_Cham}', "");
Expect(0, 43615, '\P{Is_Script_Extensions=_Cham}', "");
Expect(1, 43615, '\P{^Is_Script_Extensions=_Cham}', "");
Expect(0, 43616, '\p{Is_Script_Extensions=_Cham}', "");
Expect(1, 43616, '\p{^Is_Script_Extensions=_Cham}', "");
Expect(1, 43616, '\P{Is_Script_Extensions=_Cham}', "");
Expect(0, 43616, '\P{^Is_Script_Extensions=_Cham}', "");
Error('\p{Is_Scx=/a/	-CHAM}');
Error('\P{Is_Scx=/a/	-CHAM}');
Expect(1, 43615, '\p{Is_Scx=cham}', "");
Expect(0, 43615, '\p{^Is_Scx=cham}', "");
Expect(0, 43615, '\P{Is_Scx=cham}', "");
Expect(1, 43615, '\P{^Is_Scx=cham}', "");
Expect(0, 43616, '\p{Is_Scx=cham}', "");
Expect(1, 43616, '\p{^Is_Scx=cham}', "");
Expect(1, 43616, '\P{Is_Scx=cham}', "");
Expect(0, 43616, '\P{^Is_Scx=cham}', "");
Expect(1, 43615, '\p{Is_Scx=-_Cham}', "");
Expect(0, 43615, '\p{^Is_Scx=-_Cham}', "");
Expect(0, 43615, '\P{Is_Scx=-_Cham}', "");
Expect(1, 43615, '\P{^Is_Scx=-_Cham}', "");
Expect(0, 43616, '\p{Is_Scx=-_Cham}', "");
Expect(1, 43616, '\p{^Is_Scx=-_Cham}', "");
Expect(1, 43616, '\P{Is_Scx=-_Cham}', "");
Expect(0, 43616, '\P{^Is_Scx=-_Cham}', "");
Error('\p{Script_Extensions=tibetan/a/}');
Error('\P{Script_Extensions=tibetan/a/}');
Expect(1, 4058, '\p{Script_Extensions=tibetan}', "");
Expect(0, 4058, '\p{^Script_Extensions=tibetan}', "");
Expect(0, 4058, '\P{Script_Extensions=tibetan}', "");
Expect(1, 4058, '\P{^Script_Extensions=tibetan}', "");
Expect(0, 4059, '\p{Script_Extensions=tibetan}', "");
Expect(1, 4059, '\p{^Script_Extensions=tibetan}', "");
Expect(1, 4059, '\P{Script_Extensions=tibetan}', "");
Expect(0, 4059, '\P{^Script_Extensions=tibetan}', "");
Expect(1, 4058, '\p{Script_Extensions:	TIBETAN}', "");
Expect(0, 4058, '\p{^Script_Extensions:	TIBETAN}', "");
Expect(0, 4058, '\P{Script_Extensions:	TIBETAN}', "");
Expect(1, 4058, '\P{^Script_Extensions:	TIBETAN}', "");
Expect(0, 4059, '\p{Script_Extensions:	TIBETAN}', "");
Expect(1, 4059, '\p{^Script_Extensions:	TIBETAN}', "");
Expect(1, 4059, '\P{Script_Extensions:	TIBETAN}', "");
Expect(0, 4059, '\P{^Script_Extensions:	TIBETAN}', "");
Error('\p{Scx=:=	_Tibt}');
Error('\P{Scx=:=	_Tibt}');
Expect(1, 4058, '\p{Scx=tibt}', "");
Expect(0, 4058, '\p{^Scx=tibt}', "");
Expect(0, 4058, '\P{Scx=tibt}', "");
Expect(1, 4058, '\P{^Scx=tibt}', "");
Expect(0, 4059, '\p{Scx=tibt}', "");
Expect(1, 4059, '\p{^Scx=tibt}', "");
Expect(1, 4059, '\P{Scx=tibt}', "");
Expect(0, 4059, '\P{^Scx=tibt}', "");
Expect(1, 4058, '\p{Scx= Tibt}', "");
Expect(0, 4058, '\p{^Scx= Tibt}', "");
Expect(0, 4058, '\P{Scx= Tibt}', "");
Expect(1, 4058, '\P{^Scx= Tibt}', "");
Expect(0, 4059, '\p{Scx= Tibt}', "");
Expect(1, 4059, '\p{^Scx= Tibt}', "");
Expect(1, 4059, '\P{Scx= Tibt}', "");
Expect(0, 4059, '\P{^Scx= Tibt}', "");
Error('\p{Is_Script_Extensions=_/a/Tibetan}');
Error('\P{Is_Script_Extensions=_/a/Tibetan}');
Expect(1, 4058, '\p{Is_Script_Extensions=tibetan}', "");
Expect(0, 4058, '\p{^Is_Script_Extensions=tibetan}', "");
Expect(0, 4058, '\P{Is_Script_Extensions=tibetan}', "");
Expect(1, 4058, '\P{^Is_Script_Extensions=tibetan}', "");
Expect(0, 4059, '\p{Is_Script_Extensions=tibetan}', "");
Expect(1, 4059, '\p{^Is_Script_Extensions=tibetan}', "");
Expect(1, 4059, '\P{Is_Script_Extensions=tibetan}', "");
Expect(0, 4059, '\P{^Is_Script_Extensions=tibetan}', "");
Expect(1, 4058, '\p{Is_Script_Extensions=	 Tibetan}', "");
Expect(0, 4058, '\p{^Is_Script_Extensions=	 Tibetan}', "");
Expect(0, 4058, '\P{Is_Script_Extensions=	 Tibetan}', "");
Expect(1, 4058, '\P{^Is_Script_Extensions=	 Tibetan}', "");
Expect(0, 4059, '\p{Is_Script_Extensions=	 Tibetan}', "");
Expect(1, 4059, '\p{^Is_Script_Extensions=	 Tibetan}', "");
Expect(1, 4059, '\P{Is_Script_Extensions=	 Tibetan}', "");
Expect(0, 4059, '\P{^Is_Script_Extensions=	 Tibetan}', "");
Error('\p{Is_Scx:   :=Tibt}');
Error('\P{Is_Scx:   :=Tibt}');
Expect(1, 4058, '\p{Is_Scx=tibt}', "");
Expect(0, 4058, '\p{^Is_Scx=tibt}', "");
Expect(0, 4058, '\P{Is_Scx=tibt}', "");
Expect(1, 4058, '\P{^Is_Scx=tibt}', "");
Expect(0, 4059, '\p{Is_Scx=tibt}', "");
Expect(1, 4059, '\p{^Is_Scx=tibt}', "");
Expect(1, 4059, '\P{Is_Scx=tibt}', "");
Expect(0, 4059, '\P{^Is_Scx=tibt}', "");
Expect(1, 4058, '\p{Is_Scx= TIBT}', "");
Expect(0, 4058, '\p{^Is_Scx= TIBT}', "");
Expect(0, 4058, '\P{Is_Scx= TIBT}', "");
Expect(1, 4058, '\P{^Is_Scx= TIBT}', "");
Expect(0, 4059, '\p{Is_Scx= TIBT}', "");
Expect(1, 4059, '\p{^Is_Scx= TIBT}', "");
Expect(1, 4059, '\P{Is_Scx= TIBT}', "");
Expect(0, 4059, '\P{^Is_Scx= TIBT}', "");
Error('\p{Script_Extensions=:=_	ogham}');
Error('\P{Script_Extensions=:=_	ogham}');
Expect(1, 5788, '\p{Script_Extensions=ogham}', "");
Expect(0, 5788, '\p{^Script_Extensions=ogham}', "");
Expect(0, 5788, '\P{Script_Extensions=ogham}', "");
Expect(1, 5788, '\P{^Script_Extensions=ogham}', "");
Expect(0, 5789, '\p{Script_Extensions=ogham}', "");
Expect(1, 5789, '\p{^Script_Extensions=ogham}', "");
Expect(1, 5789, '\P{Script_Extensions=ogham}', "");
Expect(0, 5789, '\P{^Script_Extensions=ogham}', "");
Expect(1, 5788, '\p{Script_Extensions=	-Ogham}', "");
Expect(0, 5788, '\p{^Script_Extensions=	-Ogham}', "");
Expect(0, 5788, '\P{Script_Extensions=	-Ogham}', "");
Expect(1, 5788, '\P{^Script_Extensions=	-Ogham}', "");
Expect(0, 5789, '\p{Script_Extensions=	-Ogham}', "");
Expect(1, 5789, '\p{^Script_Extensions=	-Ogham}', "");
Expect(1, 5789, '\P{Script_Extensions=	-Ogham}', "");
Expect(0, 5789, '\P{^Script_Extensions=	-Ogham}', "");
Error('\p{Scx= _OGAM/a/}');
Error('\P{Scx= _OGAM/a/}');
Expect(1, 5788, '\p{Scx=ogam}', "");
Expect(0, 5788, '\p{^Scx=ogam}', "");
Expect(0, 5788, '\P{Scx=ogam}', "");
Expect(1, 5788, '\P{^Scx=ogam}', "");
Expect(0, 5789, '\p{Scx=ogam}', "");
Expect(1, 5789, '\p{^Scx=ogam}', "");
Expect(1, 5789, '\P{Scx=ogam}', "");
Expect(0, 5789, '\P{^Scx=ogam}', "");
Expect(1, 5788, '\p{Scx=		ogam}', "");
Expect(0, 5788, '\p{^Scx=		ogam}', "");
Expect(0, 5788, '\P{Scx=		ogam}', "");
Expect(1, 5788, '\P{^Scx=		ogam}', "");
Expect(0, 5789, '\p{Scx=		ogam}', "");
Expect(1, 5789, '\p{^Scx=		ogam}', "");
Expect(1, 5789, '\P{Scx=		ogam}', "");
Expect(0, 5789, '\P{^Scx=		ogam}', "");
Error('\p{Is_Script_Extensions= /a/Ogham}');
Error('\P{Is_Script_Extensions= /a/Ogham}');
Expect(1, 5788, '\p{Is_Script_Extensions=ogham}', "");
Expect(0, 5788, '\p{^Is_Script_Extensions=ogham}', "");
Expect(0, 5788, '\P{Is_Script_Extensions=ogham}', "");
Expect(1, 5788, '\P{^Is_Script_Extensions=ogham}', "");
Expect(0, 5789, '\p{Is_Script_Extensions=ogham}', "");
Expect(1, 5789, '\p{^Is_Script_Extensions=ogham}', "");
Expect(1, 5789, '\P{Is_Script_Extensions=ogham}', "");
Expect(0, 5789, '\P{^Is_Script_Extensions=ogham}', "");
Expect(1, 5788, '\p{Is_Script_Extensions=_	Ogham}', "");
Expect(0, 5788, '\p{^Is_Script_Extensions=_	Ogham}', "");
Expect(0, 5788, '\P{Is_Script_Extensions=_	Ogham}', "");
Expect(1, 5788, '\P{^Is_Script_Extensions=_	Ogham}', "");
Expect(0, 5789, '\p{Is_Script_Extensions=_	Ogham}', "");
Expect(1, 5789, '\p{^Is_Script_Extensions=_	Ogham}', "");
Expect(1, 5789, '\P{Is_Script_Extensions=_	Ogham}', "");
Expect(0, 5789, '\P{^Is_Script_Extensions=_	Ogham}', "");
Error('\p{Is_Scx=:=OGAM}');
Error('\P{Is_Scx=:=OGAM}');
Expect(1, 5788, '\p{Is_Scx=ogam}', "");
Expect(0, 5788, '\p{^Is_Scx=ogam}', "");
Expect(0, 5788, '\P{Is_Scx=ogam}', "");
Expect(1, 5788, '\P{^Is_Scx=ogam}', "");
Expect(0, 5789, '\p{Is_Scx=ogam}', "");
Expect(1, 5789, '\p{^Is_Scx=ogam}', "");
Expect(1, 5789, '\P{Is_Scx=ogam}', "");
Expect(0, 5789, '\P{^Is_Scx=ogam}', "");
Expect(1, 5788, '\p{Is_Scx=_	Ogam}', "");
Expect(0, 5788, '\p{^Is_Scx=_	Ogam}', "");
Expect(0, 5788, '\P{Is_Scx=_	Ogam}', "");
Expect(1, 5788, '\P{^Is_Scx=_	Ogam}', "");
Expect(0, 5789, '\p{Is_Scx=_	Ogam}', "");
Expect(1, 5789, '\p{^Is_Scx=_	Ogam}', "");
Expect(1, 5789, '\P{Is_Scx=_	Ogam}', "");
Expect(0, 5789, '\P{^Is_Scx=_	Ogam}', "");
Error('\p{Script_Extensions= /a/Tai_Viet}');
Error('\P{Script_Extensions= /a/Tai_Viet}');
Expect(1, 43743, '\p{Script_Extensions=taiviet}', "");
Expect(0, 43743, '\p{^Script_Extensions=taiviet}', "");
Expect(0, 43743, '\P{Script_Extensions=taiviet}', "");
Expect(1, 43743, '\P{^Script_Extensions=taiviet}', "");
Expect(0, 43744, '\p{Script_Extensions=taiviet}', "");
Expect(1, 43744, '\p{^Script_Extensions=taiviet}', "");
Expect(1, 43744, '\P{Script_Extensions=taiviet}', "");
Expect(0, 43744, '\P{^Script_Extensions=taiviet}', "");
Expect(1, 43743, '\p{Script_Extensions=_Tai_Viet}', "");
Expect(0, 43743, '\p{^Script_Extensions=_Tai_Viet}', "");
Expect(0, 43743, '\P{Script_Extensions=_Tai_Viet}', "");
Expect(1, 43743, '\P{^Script_Extensions=_Tai_Viet}', "");
Expect(0, 43744, '\p{Script_Extensions=_Tai_Viet}', "");
Expect(1, 43744, '\p{^Script_Extensions=_Tai_Viet}', "");
Expect(1, 43744, '\P{Script_Extensions=_Tai_Viet}', "");
Expect(0, 43744, '\P{^Script_Extensions=_Tai_Viet}', "");
Error('\p{Scx=/a/Tavt}');
Error('\P{Scx=/a/Tavt}');
Expect(1, 43743, '\p{Scx:tavt}', "");
Expect(0, 43743, '\p{^Scx:tavt}', "");
Expect(0, 43743, '\P{Scx:tavt}', "");
Expect(1, 43743, '\P{^Scx:tavt}', "");
Expect(0, 43744, '\p{Scx:tavt}', "");
Expect(1, 43744, '\p{^Scx:tavt}', "");
Expect(1, 43744, '\P{Scx:tavt}', "");
Expect(0, 43744, '\P{^Scx:tavt}', "");
Expect(1, 43743, '\p{Scx=_TAVT}', "");
Expect(0, 43743, '\p{^Scx=_TAVT}', "");
Expect(0, 43743, '\P{Scx=_TAVT}', "");
Expect(1, 43743, '\P{^Scx=_TAVT}', "");
Expect(0, 43744, '\p{Scx=_TAVT}', "");
Expect(1, 43744, '\p{^Scx=_TAVT}', "");
Expect(1, 43744, '\P{Scx=_TAVT}', "");
Expect(0, 43744, '\P{^Scx=_TAVT}', "");
Error('\p{Is_Script_Extensions=:=tai_VIET}');
Error('\P{Is_Script_Extensions=:=tai_VIET}');
Expect(1, 43743, '\p{Is_Script_Extensions=taiviet}', "");
Expect(0, 43743, '\p{^Is_Script_Extensions=taiviet}', "");
Expect(0, 43743, '\P{Is_Script_Extensions=taiviet}', "");
Expect(1, 43743, '\P{^Is_Script_Extensions=taiviet}', "");
Expect(0, 43744, '\p{Is_Script_Extensions=taiviet}', "");
Expect(1, 43744, '\p{^Is_Script_Extensions=taiviet}', "");
Expect(1, 43744, '\P{Is_Script_Extensions=taiviet}', "");
Expect(0, 43744, '\P{^Is_Script_Extensions=taiviet}', "");
Expect(1, 43743, '\p{Is_Script_Extensions=- Tai_Viet}', "");
Expect(0, 43743, '\p{^Is_Script_Extensions=- Tai_Viet}', "");
Expect(0, 43743, '\P{Is_Script_Extensions=- Tai_Viet}', "");
Expect(1, 43743, '\P{^Is_Script_Extensions=- Tai_Viet}', "");
Expect(0, 43744, '\p{Is_Script_Extensions=- Tai_Viet}', "");
Expect(1, 43744, '\p{^Is_Script_Extensions=- Tai_Viet}', "");
Expect(1, 43744, '\P{Is_Script_Extensions=- Tai_Viet}', "");
Expect(0, 43744, '\P{^Is_Script_Extensions=- Tai_Viet}', "");
Error('\p{Is_Scx=/a/-Tavt}');
Error('\P{Is_Scx=/a/-Tavt}');
Expect(1, 43743, '\p{Is_Scx=tavt}', "");
Expect(0, 43743, '\p{^Is_Scx=tavt}', "");
Expect(0, 43743, '\P{Is_Scx=tavt}', "");
Expect(1, 43743, '\P{^Is_Scx=tavt}', "");
Expect(0, 43744, '\p{Is_Scx=tavt}', "");
Expect(1, 43744, '\p{^Is_Scx=tavt}', "");
Expect(1, 43744, '\P{Is_Scx=tavt}', "");
Expect(0, 43744, '\P{^Is_Scx=tavt}', "");
Expect(1, 43743, '\p{Is_Scx:	_ tavt}', "");
Expect(0, 43743, '\p{^Is_Scx:	_ tavt}', "");
Expect(0, 43743, '\P{Is_Scx:	_ tavt}', "");
Expect(1, 43743, '\P{^Is_Scx:	_ tavt}', "");
Expect(0, 43744, '\p{Is_Scx:	_ tavt}', "");
Expect(1, 43744, '\p{^Is_Scx:	_ tavt}', "");
Expect(1, 43744, '\P{Is_Scx:	_ tavt}', "");
Expect(0, 43744, '\P{^Is_Scx:	_ tavt}', "");
Error('\p{Script_Extensions: :=MALAYALAM}');
Error('\P{Script_Extensions: :=MALAYALAM}');
Expect(1, 3455, '\p{Script_Extensions=malayalam}', "");
Expect(0, 3455, '\p{^Script_Extensions=malayalam}', "");
Expect(0, 3455, '\P{Script_Extensions=malayalam}', "");
Expect(1, 3455, '\P{^Script_Extensions=malayalam}', "");
Expect(0, 3456, '\p{Script_Extensions=malayalam}', "");
Expect(1, 3456, '\p{^Script_Extensions=malayalam}', "");
Expect(1, 3456, '\P{Script_Extensions=malayalam}', "");
Expect(0, 3456, '\P{^Script_Extensions=malayalam}', "");
Expect(1, 3455, '\p{Script_Extensions=_	MALAYALAM}', "");
Expect(0, 3455, '\p{^Script_Extensions=_	MALAYALAM}', "");
Expect(0, 3455, '\P{Script_Extensions=_	MALAYALAM}', "");
Expect(1, 3455, '\P{^Script_Extensions=_	MALAYALAM}', "");
Expect(0, 3456, '\p{Script_Extensions=_	MALAYALAM}', "");
Expect(1, 3456, '\p{^Script_Extensions=_	MALAYALAM}', "");
Expect(1, 3456, '\P{Script_Extensions=_	MALAYALAM}', "");
Expect(0, 3456, '\P{^Script_Extensions=_	MALAYALAM}', "");
Error('\p{Scx=_:=MLYM}');
Error('\P{Scx=_:=MLYM}');
Expect(1, 3455, '\p{Scx=mlym}', "");
Expect(0, 3455, '\p{^Scx=mlym}', "");
Expect(0, 3455, '\P{Scx=mlym}', "");
Expect(1, 3455, '\P{^Scx=mlym}', "");
Expect(0, 3456, '\p{Scx=mlym}', "");
Expect(1, 3456, '\p{^Scx=mlym}', "");
Expect(1, 3456, '\P{Scx=mlym}', "");
Expect(0, 3456, '\P{^Scx=mlym}', "");
Expect(1, 3455, '\p{Scx=-Mlym}', "");
Expect(0, 3455, '\p{^Scx=-Mlym}', "");
Expect(0, 3455, '\P{Scx=-Mlym}', "");
Expect(1, 3455, '\P{^Scx=-Mlym}', "");
Expect(0, 3456, '\p{Scx=-Mlym}', "");
Expect(1, 3456, '\p{^Scx=-Mlym}', "");
Expect(1, 3456, '\P{Scx=-Mlym}', "");
Expect(0, 3456, '\P{^Scx=-Mlym}', "");
Error('\p{Is_Script_Extensions=-/a/Malayalam}');
Error('\P{Is_Script_Extensions=-/a/Malayalam}');
Expect(1, 3455, '\p{Is_Script_Extensions=malayalam}', "");
Expect(0, 3455, '\p{^Is_Script_Extensions=malayalam}', "");
Expect(0, 3455, '\P{Is_Script_Extensions=malayalam}', "");
Expect(1, 3455, '\P{^Is_Script_Extensions=malayalam}', "");
Expect(0, 3456, '\p{Is_Script_Extensions=malayalam}', "");
Expect(1, 3456, '\p{^Is_Script_Extensions=malayalam}', "");
Expect(1, 3456, '\P{Is_Script_Extensions=malayalam}', "");
Expect(0, 3456, '\P{^Is_Script_Extensions=malayalam}', "");
Expect(1, 3455, '\p{Is_Script_Extensions=-	Malayalam}', "");
Expect(0, 3455, '\p{^Is_Script_Extensions=-	Malayalam}', "");
Expect(0, 3455, '\P{Is_Script_Extensions=-	Malayalam}', "");
Expect(1, 3455, '\P{^Is_Script_Extensions=-	Malayalam}', "");
Expect(0, 3456, '\p{Is_Script_Extensions=-	Malayalam}', "");
Expect(1, 3456, '\p{^Is_Script_Extensions=-	Malayalam}', "");
Expect(1, 3456, '\P{Is_Script_Extensions=-	Malayalam}', "");
Expect(0, 3456, '\P{^Is_Script_Extensions=-	Malayalam}', "");
Error('\p{Is_Scx=-:=MLYM}');
Error('\P{Is_Scx=-:=MLYM}');
Expect(1, 3455, '\p{Is_Scx=mlym}', "");
Expect(0, 3455, '\p{^Is_Scx=mlym}', "");
Expect(0, 3455, '\P{Is_Scx=mlym}', "");
Expect(1, 3455, '\P{^Is_Scx=mlym}', "");
Expect(0, 3456, '\p{Is_Scx=mlym}', "");
Expect(1, 3456, '\p{^Is_Scx=mlym}', "");
Expect(1, 3456, '\P{Is_Scx=mlym}', "");
Expect(0, 3456, '\P{^Is_Scx=mlym}', "");
Expect(1, 3455, '\p{Is_Scx=_-MLYM}', "");
Expect(0, 3455, '\p{^Is_Scx=_-MLYM}', "");
Expect(0, 3455, '\P{Is_Scx=_-MLYM}', "");
Expect(1, 3455, '\P{^Is_Scx=_-MLYM}', "");
Expect(0, 3456, '\p{Is_Scx=_-MLYM}', "");
Expect(1, 3456, '\p{^Is_Scx=_-MLYM}', "");
Expect(1, 3456, '\P{Is_Scx=_-MLYM}', "");
Expect(0, 3456, '\P{^Is_Scx=_-MLYM}', "");
Error('\p{Script_Extensions= :=OLD_persian}');
Error('\P{Script_Extensions= :=OLD_persian}');
Expect(1, 66517, '\p{Script_Extensions=oldpersian}', "");
Expect(0, 66517, '\p{^Script_Extensions=oldpersian}', "");
Expect(0, 66517, '\P{Script_Extensions=oldpersian}', "");
Expect(1, 66517, '\P{^Script_Extensions=oldpersian}', "");
Expect(0, 66518, '\p{Script_Extensions=oldpersian}', "");
Expect(1, 66518, '\p{^Script_Extensions=oldpersian}', "");
Expect(1, 66518, '\P{Script_Extensions=oldpersian}', "");
Expect(0, 66518, '\P{^Script_Extensions=oldpersian}', "");
Expect(1, 66517, '\p{Script_Extensions=__old_Persian}', "");
Expect(0, 66517, '\p{^Script_Extensions=__old_Persian}', "");
Expect(0, 66517, '\P{Script_Extensions=__old_Persian}', "");
Expect(1, 66517, '\P{^Script_Extensions=__old_Persian}', "");
Expect(0, 66518, '\p{Script_Extensions=__old_Persian}', "");
Expect(1, 66518, '\p{^Script_Extensions=__old_Persian}', "");
Expect(1, 66518, '\P{Script_Extensions=__old_Persian}', "");
Expect(0, 66518, '\P{^Script_Extensions=__old_Persian}', "");
Error('\p{Scx= 	xpeo/a/}');
Error('\P{Scx= 	xpeo/a/}');
Expect(1, 66517, '\p{Scx=xpeo}', "");
Expect(0, 66517, '\p{^Scx=xpeo}', "");
Expect(0, 66517, '\P{Scx=xpeo}', "");
Expect(1, 66517, '\P{^Scx=xpeo}', "");
Expect(0, 66518, '\p{Scx=xpeo}', "");
Expect(1, 66518, '\p{^Scx=xpeo}', "");
Expect(1, 66518, '\P{Scx=xpeo}', "");
Expect(0, 66518, '\P{^Scx=xpeo}', "");
Expect(1, 66517, '\p{Scx=_	XPEO}', "");
Expect(0, 66517, '\p{^Scx=_	XPEO}', "");
Expect(0, 66517, '\P{Scx=_	XPEO}', "");
Expect(1, 66517, '\P{^Scx=_	XPEO}', "");
Expect(0, 66518, '\p{Scx=_	XPEO}', "");
Expect(1, 66518, '\p{^Scx=_	XPEO}', "");
Expect(1, 66518, '\P{Scx=_	XPEO}', "");
Expect(0, 66518, '\P{^Scx=_	XPEO}', "");
Error('\p{Is_Script_Extensions=:=Old_Persian}');
Error('\P{Is_Script_Extensions=:=Old_Persian}');
Expect(1, 66517, '\p{Is_Script_Extensions=oldpersian}', "");
Expect(0, 66517, '\p{^Is_Script_Extensions=oldpersian}', "");
Expect(0, 66517, '\P{Is_Script_Extensions=oldpersian}', "");
Expect(1, 66517, '\P{^Is_Script_Extensions=oldpersian}', "");
Expect(0, 66518, '\p{Is_Script_Extensions=oldpersian}', "");
Expect(1, 66518, '\p{^Is_Script_Extensions=oldpersian}', "");
Expect(1, 66518, '\P{Is_Script_Extensions=oldpersian}', "");
Expect(0, 66518, '\P{^Is_Script_Extensions=oldpersian}', "");
Expect(1, 66517, '\p{Is_Script_Extensions:   -	old_Persian}', "");
Expect(0, 66517, '\p{^Is_Script_Extensions:   -	old_Persian}', "");
Expect(0, 66517, '\P{Is_Script_Extensions:   -	old_Persian}', "");
Expect(1, 66517, '\P{^Is_Script_Extensions:   -	old_Persian}', "");
Expect(0, 66518, '\p{Is_Script_Extensions:   -	old_Persian}', "");
Expect(1, 66518, '\p{^Is_Script_Extensions:   -	old_Persian}', "");
Expect(1, 66518, '\P{Is_Script_Extensions:   -	old_Persian}', "");
Expect(0, 66518, '\P{^Is_Script_Extensions:   -	old_Persian}', "");
Error('\p{Is_Scx=/a/-_Xpeo}');
Error('\P{Is_Scx=/a/-_Xpeo}');
Expect(1, 66517, '\p{Is_Scx=xpeo}', "");
Expect(0, 66517, '\p{^Is_Scx=xpeo}', "");
Expect(0, 66517, '\P{Is_Scx=xpeo}', "");
Expect(1, 66517, '\P{^Is_Scx=xpeo}', "");
Expect(0, 66518, '\p{Is_Scx=xpeo}', "");
Expect(1, 66518, '\p{^Is_Scx=xpeo}', "");
Expect(1, 66518, '\P{Is_Scx=xpeo}', "");
Expect(0, 66518, '\P{^Is_Scx=xpeo}', "");
Expect(1, 66517, '\p{Is_Scx= XPEO}', "");
Expect(0, 66517, '\p{^Is_Scx= XPEO}', "");
Expect(0, 66517, '\P{Is_Scx= XPEO}', "");
Expect(1, 66517, '\P{^Is_Scx= XPEO}', "");
Expect(0, 66518, '\p{Is_Scx= XPEO}', "");
Expect(1, 66518, '\p{^Is_Scx= XPEO}', "");
Expect(1, 66518, '\P{Is_Scx= XPEO}', "");
Expect(0, 66518, '\P{^Is_Scx= XPEO}', "");
Error('\p{Script_Extensions=:=	 myanmar}');
Error('\P{Script_Extensions=:=	 myanmar}');
Expect(1, 43643, '\p{Script_Extensions=myanmar}', "");
Expect(0, 43643, '\p{^Script_Extensions=myanmar}', "");
Expect(0, 43643, '\P{Script_Extensions=myanmar}', "");
Expect(1, 43643, '\P{^Script_Extensions=myanmar}', "");
Expect(0, 43644, '\p{Script_Extensions=myanmar}', "");
Expect(1, 43644, '\p{^Script_Extensions=myanmar}', "");
Expect(1, 43644, '\P{Script_Extensions=myanmar}', "");
Expect(0, 43644, '\P{^Script_Extensions=myanmar}', "");
Expect(1, 43643, '\p{Script_Extensions=  Myanmar}', "");
Expect(0, 43643, '\p{^Script_Extensions=  Myanmar}', "");
Expect(0, 43643, '\P{Script_Extensions=  Myanmar}', "");
Expect(1, 43643, '\P{^Script_Extensions=  Myanmar}', "");
Expect(0, 43644, '\p{Script_Extensions=  Myanmar}', "");
Expect(1, 43644, '\p{^Script_Extensions=  Myanmar}', "");
Expect(1, 43644, '\P{Script_Extensions=  Myanmar}', "");
Expect(0, 43644, '\P{^Script_Extensions=  Myanmar}', "");
Error('\p{Scx:	- mymr/a/}');
Error('\P{Scx:	- mymr/a/}');
Expect(1, 43643, '\p{Scx=mymr}', "");
Expect(0, 43643, '\p{^Scx=mymr}', "");
Expect(0, 43643, '\P{Scx=mymr}', "");
Expect(1, 43643, '\P{^Scx=mymr}', "");
Expect(0, 43644, '\p{Scx=mymr}', "");
Expect(1, 43644, '\p{^Scx=mymr}', "");
Expect(1, 43644, '\P{Scx=mymr}', "");
Expect(0, 43644, '\P{^Scx=mymr}', "");
Expect(1, 43643, '\p{Scx:   	Mymr}', "");
Expect(0, 43643, '\p{^Scx:   	Mymr}', "");
Expect(0, 43643, '\P{Scx:   	Mymr}', "");
Expect(1, 43643, '\P{^Scx:   	Mymr}', "");
Expect(0, 43644, '\p{Scx:   	Mymr}', "");
Expect(1, 43644, '\p{^Scx:   	Mymr}', "");
Expect(1, 43644, '\P{Scx:   	Mymr}', "");
Expect(0, 43644, '\P{^Scx:   	Mymr}', "");
Error('\p{Is_Script_Extensions=  MYANMAR:=}');
Error('\P{Is_Script_Extensions=  MYANMAR:=}');
Expect(1, 43643, '\p{Is_Script_Extensions=myanmar}', "");
Expect(0, 43643, '\p{^Is_Script_Extensions=myanmar}', "");
Expect(0, 43643, '\P{Is_Script_Extensions=myanmar}', "");
Expect(1, 43643, '\P{^Is_Script_Extensions=myanmar}', "");
Expect(0, 43644, '\p{Is_Script_Extensions=myanmar}', "");
Expect(1, 43644, '\p{^Is_Script_Extensions=myanmar}', "");
Expect(1, 43644, '\P{Is_Script_Extensions=myanmar}', "");
Expect(0, 43644, '\P{^Is_Script_Extensions=myanmar}', "");
Expect(1, 43643, '\p{Is_Script_Extensions=-_Myanmar}', "");
Expect(0, 43643, '\p{^Is_Script_Extensions=-_Myanmar}', "");
Expect(0, 43643, '\P{Is_Script_Extensions=-_Myanmar}', "");
Expect(1, 43643, '\P{^Is_Script_Extensions=-_Myanmar}', "");
Expect(0, 43644, '\p{Is_Script_Extensions=-_Myanmar}', "");
Expect(1, 43644, '\p{^Is_Script_Extensions=-_Myanmar}', "");
Expect(1, 43644, '\P{Is_Script_Extensions=-_Myanmar}', "");
Expect(0, 43644, '\P{^Is_Script_Extensions=-_Myanmar}', "");
Error('\p{Is_Scx=-_Mymr/a/}');
Error('\P{Is_Scx=-_Mymr/a/}');
Expect(1, 43643, '\p{Is_Scx=mymr}', "");
Expect(0, 43643, '\p{^Is_Scx=mymr}', "");
Expect(0, 43643, '\P{Is_Scx=mymr}', "");
Expect(1, 43643, '\P{^Is_Scx=mymr}', "");
Expect(0, 43644, '\p{Is_Scx=mymr}', "");
Expect(1, 43644, '\p{^Is_Scx=mymr}', "");
Expect(1, 43644, '\P{Is_Scx=mymr}', "");
Expect(0, 43644, '\P{^Is_Scx=mymr}', "");
Expect(1, 43643, '\p{Is_Scx=  Mymr}', "");
Expect(0, 43643, '\p{^Is_Scx=  Mymr}', "");
Expect(0, 43643, '\P{Is_Scx=  Mymr}', "");
Expect(1, 43643, '\P{^Is_Scx=  Mymr}', "");
Expect(0, 43644, '\p{Is_Scx=  Mymr}', "");
Expect(1, 43644, '\p{^Is_Scx=  Mymr}', "");
Expect(1, 43644, '\P{Is_Scx=  Mymr}', "");
Expect(0, 43644, '\P{^Is_Scx=  Mymr}', "");
Error('\p{Script_Extensions=--OLD_Turkic:=}');
Error('\P{Script_Extensions=--OLD_Turkic:=}');
Expect(1, 68680, '\p{Script_Extensions=oldturkic}', "");
Expect(0, 68680, '\p{^Script_Extensions=oldturkic}', "");
Expect(0, 68680, '\P{Script_Extensions=oldturkic}', "");
Expect(1, 68680, '\P{^Script_Extensions=oldturkic}', "");
Expect(0, 68681, '\p{Script_Extensions=oldturkic}', "");
Expect(1, 68681, '\p{^Script_Extensions=oldturkic}', "");
Expect(1, 68681, '\P{Script_Extensions=oldturkic}', "");
Expect(0, 68681, '\P{^Script_Extensions=oldturkic}', "");
Expect(1, 68680, '\p{Script_Extensions=_Old_Turkic}', "");
Expect(0, 68680, '\p{^Script_Extensions=_Old_Turkic}', "");
Expect(0, 68680, '\P{Script_Extensions=_Old_Turkic}', "");
Expect(1, 68680, '\P{^Script_Extensions=_Old_Turkic}', "");
Expect(0, 68681, '\p{Script_Extensions=_Old_Turkic}', "");
Expect(1, 68681, '\p{^Script_Extensions=_Old_Turkic}', "");
Expect(1, 68681, '\P{Script_Extensions=_Old_Turkic}', "");
Expect(0, 68681, '\P{^Script_Extensions=_Old_Turkic}', "");
Error('\p{Scx= :=Orkh}');
Error('\P{Scx= :=Orkh}');
Expect(1, 68680, '\p{Scx=orkh}', "");
Expect(0, 68680, '\p{^Scx=orkh}', "");
Expect(0, 68680, '\P{Scx=orkh}', "");
Expect(1, 68680, '\P{^Scx=orkh}', "");
Expect(0, 68681, '\p{Scx=orkh}', "");
Expect(1, 68681, '\p{^Scx=orkh}', "");
Expect(1, 68681, '\P{Scx=orkh}', "");
Expect(0, 68681, '\P{^Scx=orkh}', "");
Expect(1, 68680, '\p{Scx=--Orkh}', "");
Expect(0, 68680, '\p{^Scx=--Orkh}', "");
Expect(0, 68680, '\P{Scx=--Orkh}', "");
Expect(1, 68680, '\P{^Scx=--Orkh}', "");
Expect(0, 68681, '\p{Scx=--Orkh}', "");
Expect(1, 68681, '\p{^Scx=--Orkh}', "");
Expect(1, 68681, '\P{Scx=--Orkh}', "");
Expect(0, 68681, '\P{^Scx=--Orkh}', "");
Error('\p{Is_Script_Extensions=/a/ 	Old_TURKIC}');
Error('\P{Is_Script_Extensions=/a/ 	Old_TURKIC}');
Expect(1, 68680, '\p{Is_Script_Extensions=oldturkic}', "");
Expect(0, 68680, '\p{^Is_Script_Extensions=oldturkic}', "");
Expect(0, 68680, '\P{Is_Script_Extensions=oldturkic}', "");
Expect(1, 68680, '\P{^Is_Script_Extensions=oldturkic}', "");
Expect(0, 68681, '\p{Is_Script_Extensions=oldturkic}', "");
Expect(1, 68681, '\p{^Is_Script_Extensions=oldturkic}', "");
Expect(1, 68681, '\P{Is_Script_Extensions=oldturkic}', "");
Expect(0, 68681, '\P{^Is_Script_Extensions=oldturkic}', "");
Expect(1, 68680, '\p{Is_Script_Extensions= Old_turkic}', "");
Expect(0, 68680, '\p{^Is_Script_Extensions= Old_turkic}', "");
Expect(0, 68680, '\P{Is_Script_Extensions= Old_turkic}', "");
Expect(1, 68680, '\P{^Is_Script_Extensions= Old_turkic}', "");
Expect(0, 68681, '\p{Is_Script_Extensions= Old_turkic}', "");
Expect(1, 68681, '\p{^Is_Script_Extensions= Old_turkic}', "");
Expect(1, 68681, '\P{Is_Script_Extensions= Old_turkic}', "");
Expect(0, 68681, '\P{^Is_Script_Extensions= Old_turkic}', "");
Error('\p{Is_Scx=_ orkh/a/}');
Error('\P{Is_Scx=_ orkh/a/}');
Expect(1, 68680, '\p{Is_Scx=orkh}', "");
Expect(0, 68680, '\p{^Is_Scx=orkh}', "");
Expect(0, 68680, '\P{Is_Scx=orkh}', "");
Expect(1, 68680, '\P{^Is_Scx=orkh}', "");
Expect(0, 68681, '\p{Is_Scx=orkh}', "");
Expect(1, 68681, '\p{^Is_Scx=orkh}', "");
Expect(1, 68681, '\P{Is_Scx=orkh}', "");
Expect(0, 68681, '\P{^Is_Scx=orkh}', "");
Expect(1, 68680, '\p{Is_Scx=- Orkh}', "");
Expect(0, 68680, '\p{^Is_Scx=- Orkh}', "");
Expect(0, 68680, '\P{Is_Scx=- Orkh}', "");
Expect(1, 68680, '\P{^Is_Scx=- Orkh}', "");
Expect(0, 68681, '\p{Is_Scx=- Orkh}', "");
Expect(1, 68681, '\p{^Is_Scx=- Orkh}', "");
Expect(1, 68681, '\P{Is_Scx=- Orkh}', "");
Expect(0, 68681, '\P{^Is_Scx=- Orkh}', "");
Error('\p{Script_Extensions=	/a/osmanya}');
Error('\P{Script_Extensions=	/a/osmanya}');
Expect(1, 66729, '\p{Script_Extensions=osmanya}', "");
Expect(0, 66729, '\p{^Script_Extensions=osmanya}', "");
Expect(0, 66729, '\P{Script_Extensions=osmanya}', "");
Expect(1, 66729, '\P{^Script_Extensions=osmanya}', "");
Expect(0, 66730, '\p{Script_Extensions=osmanya}', "");
Expect(1, 66730, '\p{^Script_Extensions=osmanya}', "");
Expect(1, 66730, '\P{Script_Extensions=osmanya}', "");
Expect(0, 66730, '\P{^Script_Extensions=osmanya}', "");
Expect(1, 66729, '\p{Script_Extensions= _Osmanya}', "");
Expect(0, 66729, '\p{^Script_Extensions= _Osmanya}', "");
Expect(0, 66729, '\P{Script_Extensions= _Osmanya}', "");
Expect(1, 66729, '\P{^Script_Extensions= _Osmanya}', "");
Expect(0, 66730, '\p{Script_Extensions= _Osmanya}', "");
Expect(1, 66730, '\p{^Script_Extensions= _Osmanya}', "");
Expect(1, 66730, '\P{Script_Extensions= _Osmanya}', "");
Expect(0, 66730, '\P{^Script_Extensions= _Osmanya}', "");
Error('\p{Scx=/a/Osma}');
Error('\P{Scx=/a/Osma}');
Expect(1, 66729, '\p{Scx=osma}', "");
Expect(0, 66729, '\p{^Scx=osma}', "");
Expect(0, 66729, '\P{Scx=osma}', "");
Expect(1, 66729, '\P{^Scx=osma}', "");
Expect(0, 66730, '\p{Scx=osma}', "");
Expect(1, 66730, '\p{^Scx=osma}', "");
Expect(1, 66730, '\P{Scx=osma}', "");
Expect(0, 66730, '\P{^Scx=osma}', "");
Expect(1, 66729, '\p{Scx=	 OSMA}', "");
Expect(0, 66729, '\p{^Scx=	 OSMA}', "");
Expect(0, 66729, '\P{Scx=	 OSMA}', "");
Expect(1, 66729, '\P{^Scx=	 OSMA}', "");
Expect(0, 66730, '\p{Scx=	 OSMA}', "");
Expect(1, 66730, '\p{^Scx=	 OSMA}', "");
Expect(1, 66730, '\P{Scx=	 OSMA}', "");
Expect(0, 66730, '\P{^Scx=	 OSMA}', "");
Error('\p{Is_Script_Extensions=--Osmanya/a/}');
Error('\P{Is_Script_Extensions=--Osmanya/a/}');
Expect(1, 66729, '\p{Is_Script_Extensions=osmanya}', "");
Expect(0, 66729, '\p{^Is_Script_Extensions=osmanya}', "");
Expect(0, 66729, '\P{Is_Script_Extensions=osmanya}', "");
Expect(1, 66729, '\P{^Is_Script_Extensions=osmanya}', "");
Expect(0, 66730, '\p{Is_Script_Extensions=osmanya}', "");
Expect(1, 66730, '\p{^Is_Script_Extensions=osmanya}', "");
Expect(1, 66730, '\P{Is_Script_Extensions=osmanya}', "");
Expect(0, 66730, '\P{^Is_Script_Extensions=osmanya}', "");
Expect(1, 66729, '\p{Is_Script_Extensions=_Osmanya}', "");
Expect(0, 66729, '\p{^Is_Script_Extensions=_Osmanya}', "");
Expect(0, 66729, '\P{Is_Script_Extensions=_Osmanya}', "");
Expect(1, 66729, '\P{^Is_Script_Extensions=_Osmanya}', "");
Expect(0, 66730, '\p{Is_Script_Extensions=_Osmanya}', "");
Expect(1, 66730, '\p{^Is_Script_Extensions=_Osmanya}', "");
Expect(1, 66730, '\P{Is_Script_Extensions=_Osmanya}', "");
Expect(0, 66730, '\P{^Is_Script_Extensions=_Osmanya}', "");
Error('\p{Is_Scx=:=osma}');
Error('\P{Is_Scx=:=osma}');
Expect(1, 66729, '\p{Is_Scx=osma}', "");
Expect(0, 66729, '\p{^Is_Scx=osma}', "");
Expect(0, 66729, '\P{Is_Scx=osma}', "");
Expect(1, 66729, '\P{^Is_Scx=osma}', "");
Expect(0, 66730, '\p{Is_Scx=osma}', "");
Expect(1, 66730, '\p{^Is_Scx=osma}', "");
Expect(1, 66730, '\P{Is_Scx=osma}', "");
Expect(0, 66730, '\P{^Is_Scx=osma}', "");
Expect(1, 66729, '\p{Is_Scx=- osma}', "");
Expect(0, 66729, '\p{^Is_Scx=- osma}', "");
Expect(0, 66729, '\P{Is_Scx=- osma}', "");
Expect(1, 66729, '\P{^Is_Scx=- osma}', "");
Expect(0, 66730, '\p{Is_Scx=- osma}', "");
Expect(1, 66730, '\p{^Is_Scx=- osma}', "");
Expect(1, 66730, '\P{Is_Scx=- osma}', "");
Expect(0, 66730, '\P{^Is_Scx=- osma}', "");
Error('\p{Script_Extensions=/a/ 	TAGALOG}');
Error('\P{Script_Extensions=/a/ 	TAGALOG}');
Expect(1, 5942, '\p{Script_Extensions=tagalog}', "");
Expect(0, 5942, '\p{^Script_Extensions=tagalog}', "");
Expect(0, 5942, '\P{Script_Extensions=tagalog}', "");
Expect(1, 5942, '\P{^Script_Extensions=tagalog}', "");
Expect(0, 5943, '\p{Script_Extensions=tagalog}', "");
Expect(1, 5943, '\p{^Script_Extensions=tagalog}', "");
Expect(1, 5943, '\P{Script_Extensions=tagalog}', "");
Expect(0, 5943, '\P{^Script_Extensions=tagalog}', "");
Expect(1, 5942, '\p{Script_Extensions=-tagalog}', "");
Expect(0, 5942, '\p{^Script_Extensions=-tagalog}', "");
Expect(0, 5942, '\P{Script_Extensions=-tagalog}', "");
Expect(1, 5942, '\P{^Script_Extensions=-tagalog}', "");
Expect(0, 5943, '\p{Script_Extensions=-tagalog}', "");
Expect(1, 5943, '\p{^Script_Extensions=-tagalog}', "");
Expect(1, 5943, '\P{Script_Extensions=-tagalog}', "");
Expect(0, 5943, '\P{^Script_Extensions=-tagalog}', "");
Error('\p{Scx=:=tglg}');
Error('\P{Scx=:=tglg}');
Expect(1, 5942, '\p{Scx=tglg}', "");
Expect(0, 5942, '\p{^Scx=tglg}', "");
Expect(0, 5942, '\P{Scx=tglg}', "");
Expect(1, 5942, '\P{^Scx=tglg}', "");
Expect(0, 5943, '\p{Scx=tglg}', "");
Expect(1, 5943, '\p{^Scx=tglg}', "");
Expect(1, 5943, '\P{Scx=tglg}', "");
Expect(0, 5943, '\P{^Scx=tglg}', "");
Expect(1, 5942, '\p{Scx=Tglg}', "");
Expect(0, 5942, '\p{^Scx=Tglg}', "");
Expect(0, 5942, '\P{Scx=Tglg}', "");
Expect(1, 5942, '\P{^Scx=Tglg}', "");
Expect(0, 5943, '\p{Scx=Tglg}', "");
Expect(1, 5943, '\p{^Scx=Tglg}', "");
Expect(1, 5943, '\P{Scx=Tglg}', "");
Expect(0, 5943, '\P{^Scx=Tglg}', "");
Error('\p{Is_Script_Extensions=-	tagalog/a/}');
Error('\P{Is_Script_Extensions=-	tagalog/a/}');
Expect(1, 5942, '\p{Is_Script_Extensions=tagalog}', "");
Expect(0, 5942, '\p{^Is_Script_Extensions=tagalog}', "");
Expect(0, 5942, '\P{Is_Script_Extensions=tagalog}', "");
Expect(1, 5942, '\P{^Is_Script_Extensions=tagalog}', "");
Expect(0, 5943, '\p{Is_Script_Extensions=tagalog}', "");
Expect(1, 5943, '\p{^Is_Script_Extensions=tagalog}', "");
Expect(1, 5943, '\P{Is_Script_Extensions=tagalog}', "");
Expect(0, 5943, '\P{^Is_Script_Extensions=tagalog}', "");
Expect(1, 5942, '\p{Is_Script_Extensions:			Tagalog}', "");
Expect(0, 5942, '\p{^Is_Script_Extensions:			Tagalog}', "");
Expect(0, 5942, '\P{Is_Script_Extensions:			Tagalog}', "");
Expect(1, 5942, '\P{^Is_Script_Extensions:			Tagalog}', "");
Expect(0, 5943, '\p{Is_Script_Extensions:			Tagalog}', "");
Expect(1, 5943, '\p{^Is_Script_Extensions:			Tagalog}', "");
Expect(1, 5943, '\P{Is_Script_Extensions:			Tagalog}', "");
Expect(0, 5943, '\P{^Is_Script_Extensions:			Tagalog}', "");
Error('\p{Is_Scx= :=Tglg}');
Error('\P{Is_Scx= :=Tglg}');
Expect(1, 5942, '\p{Is_Scx:   tglg}', "");
Expect(0, 5942, '\p{^Is_Scx:   tglg}', "");
Expect(0, 5942, '\P{Is_Scx:   tglg}', "");
Expect(1, 5942, '\P{^Is_Scx:   tglg}', "");
Expect(0, 5943, '\p{Is_Scx:   tglg}', "");
Expect(1, 5943, '\p{^Is_Scx:   tglg}', "");
Expect(1, 5943, '\P{Is_Scx:   tglg}', "");
Expect(0, 5943, '\P{^Is_Scx:   tglg}', "");
Expect(1, 5942, '\p{Is_Scx=	tglg}', "");
Expect(0, 5942, '\p{^Is_Scx=	tglg}', "");
Expect(0, 5942, '\P{Is_Scx=	tglg}', "");
Expect(1, 5942, '\P{^Is_Scx=	tglg}', "");
Expect(0, 5943, '\p{Is_Scx=	tglg}', "");
Expect(1, 5943, '\p{^Is_Scx=	tglg}', "");
Expect(1, 5943, '\P{Is_Scx=	tglg}', "");
Expect(0, 5943, '\P{^Is_Scx=	tglg}', "");
Error('\p{Script_Extensions=:=_-Gujarati}');
Error('\P{Script_Extensions=:=_-Gujarati}');
Expect(1, 43065, '\p{Script_Extensions=gujarati}', "");
Expect(0, 43065, '\p{^Script_Extensions=gujarati}', "");
Expect(0, 43065, '\P{Script_Extensions=gujarati}', "");
Expect(1, 43065, '\P{^Script_Extensions=gujarati}', "");
Expect(0, 43066, '\p{Script_Extensions=gujarati}', "");
Expect(1, 43066, '\p{^Script_Extensions=gujarati}', "");
Expect(1, 43066, '\P{Script_Extensions=gujarati}', "");
Expect(0, 43066, '\P{^Script_Extensions=gujarati}', "");
Expect(1, 43065, '\p{Script_Extensions=	 Gujarati}', "");
Expect(0, 43065, '\p{^Script_Extensions=	 Gujarati}', "");
Expect(0, 43065, '\P{Script_Extensions=	 Gujarati}', "");
Expect(1, 43065, '\P{^Script_Extensions=	 Gujarati}', "");
Expect(0, 43066, '\p{Script_Extensions=	 Gujarati}', "");
Expect(1, 43066, '\p{^Script_Extensions=	 Gujarati}', "");
Expect(1, 43066, '\P{Script_Extensions=	 Gujarati}', "");
Expect(0, 43066, '\P{^Script_Extensions=	 Gujarati}', "");
Error('\p{Scx= /a/Gujr}');
Error('\P{Scx= /a/Gujr}');
Expect(1, 43065, '\p{Scx=gujr}', "");
Expect(0, 43065, '\p{^Scx=gujr}', "");
Expect(0, 43065, '\P{Scx=gujr}', "");
Expect(1, 43065, '\P{^Scx=gujr}', "");
Expect(0, 43066, '\p{Scx=gujr}', "");
Expect(1, 43066, '\p{^Scx=gujr}', "");
Expect(1, 43066, '\P{Scx=gujr}', "");
Expect(0, 43066, '\P{^Scx=gujr}', "");
Expect(1, 43065, '\p{Scx= GUJR}', "");
Expect(0, 43065, '\p{^Scx= GUJR}', "");
Expect(0, 43065, '\P{Scx= GUJR}', "");
Expect(1, 43065, '\P{^Scx= GUJR}', "");
Expect(0, 43066, '\p{Scx= GUJR}', "");
Expect(1, 43066, '\p{^Scx= GUJR}', "");
Expect(1, 43066, '\P{Scx= GUJR}', "");
Expect(0, 43066, '\P{^Scx= GUJR}', "");
Error('\p{Is_Script_Extensions=:=_ GUJARATI}');
Error('\P{Is_Script_Extensions=:=_ GUJARATI}');
Expect(1, 43065, '\p{Is_Script_Extensions:gujarati}', "");
Expect(0, 43065, '\p{^Is_Script_Extensions:gujarati}', "");
Expect(0, 43065, '\P{Is_Script_Extensions:gujarati}', "");
Expect(1, 43065, '\P{^Is_Script_Extensions:gujarati}', "");
Expect(0, 43066, '\p{Is_Script_Extensions:gujarati}', "");
Expect(1, 43066, '\p{^Is_Script_Extensions:gujarati}', "");
Expect(1, 43066, '\P{Is_Script_Extensions:gujarati}', "");
Expect(0, 43066, '\P{^Is_Script_Extensions:gujarati}', "");
Expect(1, 43065, '\p{Is_Script_Extensions:-	GUJARATI}', "");
Expect(0, 43065, '\p{^Is_Script_Extensions:-	GUJARATI}', "");
Expect(0, 43065, '\P{Is_Script_Extensions:-	GUJARATI}', "");
Expect(1, 43065, '\P{^Is_Script_Extensions:-	GUJARATI}', "");
Expect(0, 43066, '\p{Is_Script_Extensions:-	GUJARATI}', "");
Expect(1, 43066, '\p{^Is_Script_Extensions:-	GUJARATI}', "");
Expect(1, 43066, '\P{Is_Script_Extensions:-	GUJARATI}', "");
Expect(0, 43066, '\P{^Is_Script_Extensions:-	GUJARATI}', "");
Error('\p{Is_Scx=/a/--Gujr}');
Error('\P{Is_Scx=/a/--Gujr}');
Expect(1, 43065, '\p{Is_Scx=gujr}', "");
Expect(0, 43065, '\p{^Is_Scx=gujr}', "");
Expect(0, 43065, '\P{Is_Scx=gujr}', "");
Expect(1, 43065, '\P{^Is_Scx=gujr}', "");
Expect(0, 43066, '\p{Is_Scx=gujr}', "");
Expect(1, 43066, '\p{^Is_Scx=gujr}', "");
Expect(1, 43066, '\P{Is_Scx=gujr}', "");
Expect(0, 43066, '\P{^Is_Scx=gujr}', "");
Expect(1, 43065, '\p{Is_Scx=GUJR}', "");
Expect(0, 43065, '\p{^Is_Scx=GUJR}', "");
Expect(0, 43065, '\P{Is_Scx=GUJR}', "");
Expect(1, 43065, '\P{^Is_Scx=GUJR}', "");
Expect(0, 43066, '\p{Is_Scx=GUJR}', "");
Expect(1, 43066, '\p{^Is_Scx=GUJR}', "");
Expect(1, 43066, '\P{Is_Scx=GUJR}', "");
Expect(0, 43066, '\P{^Is_Scx=GUJR}', "");
Error('\p{Script_Extensions=:= -oriya}');
Error('\P{Script_Extensions=:= -oriya}');
Expect(1, 2935, '\p{Script_Extensions=oriya}', "");
Expect(0, 2935, '\p{^Script_Extensions=oriya}', "");
Expect(0, 2935, '\P{Script_Extensions=oriya}', "");
Expect(1, 2935, '\P{^Script_Extensions=oriya}', "");
Expect(0, 2936, '\p{Script_Extensions=oriya}', "");
Expect(1, 2936, '\p{^Script_Extensions=oriya}', "");
Expect(1, 2936, '\P{Script_Extensions=oriya}', "");
Expect(0, 2936, '\P{^Script_Extensions=oriya}', "");
Expect(1, 2935, '\p{Script_Extensions=-	Oriya}', "");
Expect(0, 2935, '\p{^Script_Extensions=-	Oriya}', "");
Expect(0, 2935, '\P{Script_Extensions=-	Oriya}', "");
Expect(1, 2935, '\P{^Script_Extensions=-	Oriya}', "");
Expect(0, 2936, '\p{Script_Extensions=-	Oriya}', "");
Expect(1, 2936, '\p{^Script_Extensions=-	Oriya}', "");
Expect(1, 2936, '\P{Script_Extensions=-	Oriya}', "");
Expect(0, 2936, '\P{^Script_Extensions=-	Oriya}', "");
Error('\p{Scx=	ORYA/a/}');
Error('\P{Scx=	ORYA/a/}');
Expect(1, 2935, '\p{Scx=orya}', "");
Expect(0, 2935, '\p{^Scx=orya}', "");
Expect(0, 2935, '\P{Scx=orya}', "");
Expect(1, 2935, '\P{^Scx=orya}', "");
Expect(0, 2936, '\p{Scx=orya}', "");
Expect(1, 2936, '\p{^Scx=orya}', "");
Expect(1, 2936, '\P{Scx=orya}', "");
Expect(0, 2936, '\P{^Scx=orya}', "");
Expect(1, 2935, '\p{Scx=__Orya}', "");
Expect(0, 2935, '\p{^Scx=__Orya}', "");
Expect(0, 2935, '\P{Scx=__Orya}', "");
Expect(1, 2935, '\P{^Scx=__Orya}', "");
Expect(0, 2936, '\p{Scx=__Orya}', "");
Expect(1, 2936, '\p{^Scx=__Orya}', "");
Expect(1, 2936, '\P{Scx=__Orya}', "");
Expect(0, 2936, '\P{^Scx=__Orya}', "");
Error('\p{Is_Script_Extensions=-:=oriya}');
Error('\P{Is_Script_Extensions=-:=oriya}');
Expect(1, 2935, '\p{Is_Script_Extensions=oriya}', "");
Expect(0, 2935, '\p{^Is_Script_Extensions=oriya}', "");
Expect(0, 2935, '\P{Is_Script_Extensions=oriya}', "");
Expect(1, 2935, '\P{^Is_Script_Extensions=oriya}', "");
Expect(0, 2936, '\p{Is_Script_Extensions=oriya}', "");
Expect(1, 2936, '\p{^Is_Script_Extensions=oriya}', "");
Expect(1, 2936, '\P{Is_Script_Extensions=oriya}', "");
Expect(0, 2936, '\P{^Is_Script_Extensions=oriya}', "");
Expect(1, 2935, '\p{Is_Script_Extensions= 	ORIYA}', "");
Expect(0, 2935, '\p{^Is_Script_Extensions= 	ORIYA}', "");
Expect(0, 2935, '\P{Is_Script_Extensions= 	ORIYA}', "");
Expect(1, 2935, '\P{^Is_Script_Extensions= 	ORIYA}', "");
Expect(0, 2936, '\p{Is_Script_Extensions= 	ORIYA}', "");
Expect(1, 2936, '\p{^Is_Script_Extensions= 	ORIYA}', "");
Expect(1, 2936, '\P{Is_Script_Extensions= 	ORIYA}', "");
Expect(0, 2936, '\P{^Is_Script_Extensions= 	ORIYA}', "");
Error('\p{Is_Scx:-/a/orya}');
Error('\P{Is_Scx:-/a/orya}');
Expect(1, 2935, '\p{Is_Scx=orya}', "");
Expect(0, 2935, '\p{^Is_Scx=orya}', "");
Expect(0, 2935, '\P{Is_Scx=orya}', "");
Expect(1, 2935, '\P{^Is_Scx=orya}', "");
Expect(0, 2936, '\p{Is_Scx=orya}', "");
Expect(1, 2936, '\p{^Is_Scx=orya}', "");
Expect(1, 2936, '\P{Is_Scx=orya}', "");
Expect(0, 2936, '\P{^Is_Scx=orya}', "");
Expect(1, 2935, '\p{Is_Scx=	ORYA}', "");
Expect(0, 2935, '\p{^Is_Scx=	ORYA}', "");
Expect(0, 2935, '\P{Is_Scx=	ORYA}', "");
Expect(1, 2935, '\P{^Is_Scx=	ORYA}', "");
Expect(0, 2936, '\p{Is_Scx=	ORYA}', "");
Expect(1, 2936, '\p{^Is_Scx=	ORYA}', "");
Expect(1, 2936, '\P{Is_Scx=	ORYA}', "");
Expect(0, 2936, '\P{^Is_Scx=	ORYA}', "");
Error('\p{Script_Extensions=/a/__lisu}');
Error('\P{Script_Extensions=/a/__lisu}');
Expect(1, 42239, '\p{Script_Extensions=lisu}', "");
Expect(0, 42239, '\p{^Script_Extensions=lisu}', "");
Expect(0, 42239, '\P{Script_Extensions=lisu}', "");
Expect(1, 42239, '\P{^Script_Extensions=lisu}', "");
Expect(0, 42240, '\p{Script_Extensions=lisu}', "");
Expect(1, 42240, '\p{^Script_Extensions=lisu}', "");
Expect(1, 42240, '\P{Script_Extensions=lisu}', "");
Expect(0, 42240, '\P{^Script_Extensions=lisu}', "");
Expect(1, 42239, '\p{Script_Extensions= 	Lisu}', "");
Expect(0, 42239, '\p{^Script_Extensions= 	Lisu}', "");
Expect(0, 42239, '\P{Script_Extensions= 	Lisu}', "");
Expect(1, 42239, '\P{^Script_Extensions= 	Lisu}', "");
Expect(0, 42240, '\p{Script_Extensions= 	Lisu}', "");
Expect(1, 42240, '\p{^Script_Extensions= 	Lisu}', "");
Expect(1, 42240, '\P{Script_Extensions= 	Lisu}', "");
Expect(0, 42240, '\P{^Script_Extensions= 	Lisu}', "");
Error('\p{Scx=/a/Lisu}');
Error('\P{Scx=/a/Lisu}');
Expect(1, 42239, '\p{Scx=lisu}', "");
Expect(0, 42239, '\p{^Scx=lisu}', "");
Expect(0, 42239, '\P{Scx=lisu}', "");
Expect(1, 42239, '\P{^Scx=lisu}', "");
Expect(0, 42240, '\p{Scx=lisu}', "");
Expect(1, 42240, '\p{^Scx=lisu}', "");
Expect(1, 42240, '\P{Scx=lisu}', "");
Expect(0, 42240, '\P{^Scx=lisu}', "");
Expect(1, 42239, '\p{Scx=_-LISU}', "");
Expect(0, 42239, '\p{^Scx=_-LISU}', "");
Expect(0, 42239, '\P{Scx=_-LISU}', "");
Expect(1, 42239, '\P{^Scx=_-LISU}', "");
Expect(0, 42240, '\p{Scx=_-LISU}', "");
Expect(1, 42240, '\p{^Scx=_-LISU}', "");
Expect(1, 42240, '\P{Scx=_-LISU}', "");
Expect(0, 42240, '\P{^Scx=_-LISU}', "");
Error('\p{Is_Script_Extensions=:=	-Lisu}');
Error('\P{Is_Script_Extensions=:=	-Lisu}');
Expect(1, 42239, '\p{Is_Script_Extensions=lisu}', "");
Expect(0, 42239, '\p{^Is_Script_Extensions=lisu}', "");
Expect(0, 42239, '\P{Is_Script_Extensions=lisu}', "");
Expect(1, 42239, '\P{^Is_Script_Extensions=lisu}', "");
Expect(0, 42240, '\p{Is_Script_Extensions=lisu}', "");
Expect(1, 42240, '\p{^Is_Script_Extensions=lisu}', "");
Expect(1, 42240, '\P{Is_Script_Extensions=lisu}', "");
Expect(0, 42240, '\P{^Is_Script_Extensions=lisu}', "");
Expect(1, 42239, '\p{Is_Script_Extensions:	-_Lisu}', "");
Expect(0, 42239, '\p{^Is_Script_Extensions:	-_Lisu}', "");
Expect(0, 42239, '\P{Is_Script_Extensions:	-_Lisu}', "");
Expect(1, 42239, '\P{^Is_Script_Extensions:	-_Lisu}', "");
Expect(0, 42240, '\p{Is_Script_Extensions:	-_Lisu}', "");
Expect(1, 42240, '\p{^Is_Script_Extensions:	-_Lisu}', "");
Expect(1, 42240, '\P{Is_Script_Extensions:	-_Lisu}', "");
Expect(0, 42240, '\P{^Is_Script_Extensions:	-_Lisu}', "");
Error('\p{Is_Scx=-:=lisu}');
Error('\P{Is_Scx=-:=lisu}');
Expect(1, 42239, '\p{Is_Scx=lisu}', "");
Expect(0, 42239, '\p{^Is_Scx=lisu}', "");
Expect(0, 42239, '\P{Is_Scx=lisu}', "");
Expect(1, 42239, '\P{^Is_Scx=lisu}', "");
Expect(0, 42240, '\p{Is_Scx=lisu}', "");
Expect(1, 42240, '\p{^Is_Scx=lisu}', "");
Expect(1, 42240, '\P{Is_Scx=lisu}', "");
Expect(0, 42240, '\P{^Is_Scx=lisu}', "");
Expect(1, 42239, '\p{Is_Scx=	_Lisu}', "");
Expect(0, 42239, '\p{^Is_Scx=	_Lisu}', "");
Expect(0, 42239, '\P{Is_Scx=	_Lisu}', "");
Expect(1, 42239, '\P{^Is_Scx=	_Lisu}', "");
Expect(0, 42240, '\p{Is_Scx=	_Lisu}', "");
Expect(1, 42240, '\p{^Is_Scx=	_Lisu}', "");
Expect(1, 42240, '\P{Is_Scx=	_Lisu}', "");
Expect(0, 42240, '\P{^Is_Scx=	_Lisu}', "");
Error('\p{Script_Extensions:   /a/-	avestan}');
Error('\P{Script_Extensions:   /a/-	avestan}');
Expect(1, 68415, '\p{Script_Extensions=avestan}', "");
Expect(0, 68415, '\p{^Script_Extensions=avestan}', "");
Expect(0, 68415, '\P{Script_Extensions=avestan}', "");
Expect(1, 68415, '\P{^Script_Extensions=avestan}', "");
Expect(0, 68416, '\p{Script_Extensions=avestan}', "");
Expect(1, 68416, '\p{^Script_Extensions=avestan}', "");
Expect(1, 68416, '\P{Script_Extensions=avestan}', "");
Expect(0, 68416, '\P{^Script_Extensions=avestan}', "");
Expect(1, 68415, '\p{Script_Extensions=	 Avestan}', "");
Expect(0, 68415, '\p{^Script_Extensions=	 Avestan}', "");
Expect(0, 68415, '\P{Script_Extensions=	 Avestan}', "");
Expect(1, 68415, '\P{^Script_Extensions=	 Avestan}', "");
Expect(0, 68416, '\p{Script_Extensions=	 Avestan}', "");
Expect(1, 68416, '\p{^Script_Extensions=	 Avestan}', "");
Expect(1, 68416, '\P{Script_Extensions=	 Avestan}', "");
Expect(0, 68416, '\P{^Script_Extensions=	 Avestan}', "");
Error('\p{Scx=/a/-avst}');
Error('\P{Scx=/a/-avst}');
Expect(1, 68415, '\p{Scx:   avst}', "");
Expect(0, 68415, '\p{^Scx:   avst}', "");
Expect(0, 68415, '\P{Scx:   avst}', "");
Expect(1, 68415, '\P{^Scx:   avst}', "");
Expect(0, 68416, '\p{Scx:   avst}', "");
Expect(1, 68416, '\p{^Scx:   avst}', "");
Expect(1, 68416, '\P{Scx:   avst}', "");
Expect(0, 68416, '\P{^Scx:   avst}', "");
Expect(1, 68415, '\p{Scx=_ Avst}', "");
Expect(0, 68415, '\p{^Scx=_ Avst}', "");
Expect(0, 68415, '\P{Scx=_ Avst}', "");
Expect(1, 68415, '\P{^Scx=_ Avst}', "");
Expect(0, 68416, '\p{Scx=_ Avst}', "");
Expect(1, 68416, '\p{^Scx=_ Avst}', "");
Expect(1, 68416, '\P{Scx=_ Avst}', "");
Expect(0, 68416, '\P{^Scx=_ Avst}', "");
Error('\p{Is_Script_Extensions: := avestan}');
Error('\P{Is_Script_Extensions: := avestan}');
Expect(1, 68415, '\p{Is_Script_Extensions=avestan}', "");
Expect(0, 68415, '\p{^Is_Script_Extensions=avestan}', "");
Expect(0, 68415, '\P{Is_Script_Extensions=avestan}', "");
Expect(1, 68415, '\P{^Is_Script_Extensions=avestan}', "");
Expect(0, 68416, '\p{Is_Script_Extensions=avestan}', "");
Expect(1, 68416, '\p{^Is_Script_Extensions=avestan}', "");
Expect(1, 68416, '\P{Is_Script_Extensions=avestan}', "");
Expect(0, 68416, '\P{^Is_Script_Extensions=avestan}', "");
Expect(1, 68415, '\p{Is_Script_Extensions:	Avestan}', "");
Expect(0, 68415, '\p{^Is_Script_Extensions:	Avestan}', "");
Expect(0, 68415, '\P{Is_Script_Extensions:	Avestan}', "");
Expect(1, 68415, '\P{^Is_Script_Extensions:	Avestan}', "");
Expect(0, 68416, '\p{Is_Script_Extensions:	Avestan}', "");
Expect(1, 68416, '\p{^Is_Script_Extensions:	Avestan}', "");
Expect(1, 68416, '\P{Is_Script_Extensions:	Avestan}', "");
Expect(0, 68416, '\P{^Is_Script_Extensions:	Avestan}', "");
Error('\p{Is_Scx=/a/- avst}');
Error('\P{Is_Scx=/a/- avst}');
Expect(1, 68415, '\p{Is_Scx=avst}', "");
Expect(0, 68415, '\p{^Is_Scx=avst}', "");
Expect(0, 68415, '\P{Is_Scx=avst}', "");
Expect(1, 68415, '\P{^Is_Scx=avst}', "");
Expect(0, 68416, '\p{Is_Scx=avst}', "");
Expect(1, 68416, '\p{^Is_Scx=avst}', "");
Expect(1, 68416, '\P{Is_Scx=avst}', "");
Expect(0, 68416, '\P{^Is_Scx=avst}', "");
Expect(1, 68415, '\p{Is_Scx= 	avst}', "");
Expect(0, 68415, '\p{^Is_Scx= 	avst}', "");
Expect(0, 68415, '\P{Is_Scx= 	avst}', "");
Expect(1, 68415, '\P{^Is_Scx= 	avst}', "");
Expect(0, 68416, '\p{Is_Scx= 	avst}', "");
Expect(1, 68416, '\p{^Is_Scx= 	avst}', "");
Expect(1, 68416, '\P{Is_Scx= 	avst}', "");
Expect(0, 68416, '\P{^Is_Scx= 	avst}', "");
Error('\p{Script_Extensions=:=	Tai_le}');
Error('\P{Script_Extensions=:=	Tai_le}');
Expect(1, 6516, '\p{Script_Extensions=taile}', "");
Expect(0, 6516, '\p{^Script_Extensions=taile}', "");
Expect(0, 6516, '\P{Script_Extensions=taile}', "");
Expect(1, 6516, '\P{^Script_Extensions=taile}', "");
Expect(0, 6517, '\p{Script_Extensions=taile}', "");
Expect(1, 6517, '\p{^Script_Extensions=taile}', "");
Expect(1, 6517, '\P{Script_Extensions=taile}', "");
Expect(0, 6517, '\P{^Script_Extensions=taile}', "");
Expect(1, 6516, '\p{Script_Extensions:   	-Tai_LE}', "");
Expect(0, 6516, '\p{^Script_Extensions:   	-Tai_LE}', "");
Expect(0, 6516, '\P{Script_Extensions:   	-Tai_LE}', "");
Expect(1, 6516, '\P{^Script_Extensions:   	-Tai_LE}', "");
Expect(0, 6517, '\p{Script_Extensions:   	-Tai_LE}', "");
Expect(1, 6517, '\p{^Script_Extensions:   	-Tai_LE}', "");
Expect(1, 6517, '\P{Script_Extensions:   	-Tai_LE}', "");
Expect(0, 6517, '\P{^Script_Extensions:   	-Tai_LE}', "");
Error('\p{Scx= 	Tale/a/}');
Error('\P{Scx= 	Tale/a/}');
Expect(1, 6516, '\p{Scx=tale}', "");
Expect(0, 6516, '\p{^Scx=tale}', "");
Expect(0, 6516, '\P{Scx=tale}', "");
Expect(1, 6516, '\P{^Scx=tale}', "");
Expect(0, 6517, '\p{Scx=tale}', "");
Expect(1, 6517, '\p{^Scx=tale}', "");
Expect(1, 6517, '\P{Scx=tale}', "");
Expect(0, 6517, '\P{^Scx=tale}', "");
Expect(1, 6516, '\p{Scx= Tale}', "");
Expect(0, 6516, '\p{^Scx= Tale}', "");
Expect(0, 6516, '\P{Scx= Tale}', "");
Expect(1, 6516, '\P{^Scx= Tale}', "");
Expect(0, 6517, '\p{Scx= Tale}', "");
Expect(1, 6517, '\p{^Scx= Tale}', "");
Expect(1, 6517, '\P{Scx= Tale}', "");
Expect(0, 6517, '\P{^Scx= Tale}', "");
Error('\p{Is_Script_Extensions=	tai_le:=}');
Error('\P{Is_Script_Extensions=	tai_le:=}');
Expect(1, 6516, '\p{Is_Script_Extensions=taile}', "");
Expect(0, 6516, '\p{^Is_Script_Extensions=taile}', "");
Expect(0, 6516, '\P{Is_Script_Extensions=taile}', "");
Expect(1, 6516, '\P{^Is_Script_Extensions=taile}', "");
Expect(0, 6517, '\p{Is_Script_Extensions=taile}', "");
Expect(1, 6517, '\p{^Is_Script_Extensions=taile}', "");
Expect(1, 6517, '\P{Is_Script_Extensions=taile}', "");
Expect(0, 6517, '\P{^Is_Script_Extensions=taile}', "");
Expect(1, 6516, '\p{Is_Script_Extensions=_ tai_Le}', "");
Expect(0, 6516, '\p{^Is_Script_Extensions=_ tai_Le}', "");
Expect(0, 6516, '\P{Is_Script_Extensions=_ tai_Le}', "");
Expect(1, 6516, '\P{^Is_Script_Extensions=_ tai_Le}', "");
Expect(0, 6517, '\p{Is_Script_Extensions=_ tai_Le}', "");
Expect(1, 6517, '\p{^Is_Script_Extensions=_ tai_Le}', "");
Expect(1, 6517, '\P{Is_Script_Extensions=_ tai_Le}', "");
Expect(0, 6517, '\P{^Is_Script_Extensions=_ tai_Le}', "");
Error('\p{Is_Scx:   _/a/TALE}');
Error('\P{Is_Scx:   _/a/TALE}');
Expect(1, 6516, '\p{Is_Scx=tale}', "");
Expect(0, 6516, '\p{^Is_Scx=tale}', "");
Expect(0, 6516, '\P{Is_Scx=tale}', "");
Expect(1, 6516, '\P{^Is_Scx=tale}', "");
Expect(0, 6517, '\p{Is_Scx=tale}', "");
Expect(1, 6517, '\p{^Is_Scx=tale}', "");
Expect(1, 6517, '\P{Is_Scx=tale}', "");
Expect(0, 6517, '\P{^Is_Scx=tale}', "");
Expect(1, 6516, '\p{Is_Scx:   __Tale}', "");
Expect(0, 6516, '\p{^Is_Scx:   __Tale}', "");
Expect(0, 6516, '\P{Is_Scx:   __Tale}', "");
Expect(1, 6516, '\P{^Is_Scx:   __Tale}', "");
Expect(0, 6517, '\p{Is_Scx:   __Tale}', "");
Expect(1, 6517, '\p{^Is_Scx:   __Tale}', "");
Expect(1, 6517, '\P{Is_Scx:   __Tale}', "");
Expect(0, 6517, '\P{^Is_Scx:   __Tale}', "");
Error('\p{Script_Extensions=	:=Georgian}');
Error('\P{Script_Extensions=	:=Georgian}');
Expect(1, 11565, '\p{Script_Extensions=georgian}', "");
Expect(0, 11565, '\p{^Script_Extensions=georgian}', "");
Expect(0, 11565, '\P{Script_Extensions=georgian}', "");
Expect(1, 11565, '\P{^Script_Extensions=georgian}', "");
Expect(0, 11566, '\p{Script_Extensions=georgian}', "");
Expect(1, 11566, '\p{^Script_Extensions=georgian}', "");
Expect(1, 11566, '\P{Script_Extensions=georgian}', "");
Expect(0, 11566, '\P{^Script_Extensions=georgian}', "");
Expect(1, 11565, '\p{Script_Extensions=	 Georgian}', "");
Expect(0, 11565, '\p{^Script_Extensions=	 Georgian}', "");
Expect(0, 11565, '\P{Script_Extensions=	 Georgian}', "");
Expect(1, 11565, '\P{^Script_Extensions=	 Georgian}', "");
Expect(0, 11566, '\p{Script_Extensions=	 Georgian}', "");
Expect(1, 11566, '\p{^Script_Extensions=	 Georgian}', "");
Expect(1, 11566, '\P{Script_Extensions=	 Georgian}', "");
Expect(0, 11566, '\P{^Script_Extensions=	 Georgian}', "");
Error('\p{Scx=:=	 Geor}');
Error('\P{Scx=:=	 Geor}');
Expect(1, 11565, '\p{Scx=geor}', "");
Expect(0, 11565, '\p{^Scx=geor}', "");
Expect(0, 11565, '\P{Scx=geor}', "");
Expect(1, 11565, '\P{^Scx=geor}', "");
Expect(0, 11566, '\p{Scx=geor}', "");
Expect(1, 11566, '\p{^Scx=geor}', "");
Expect(1, 11566, '\P{Scx=geor}', "");
Expect(0, 11566, '\P{^Scx=geor}', "");
Expect(1, 11565, '\p{Scx=	Geor}', "");
Expect(0, 11565, '\p{^Scx=	Geor}', "");
Expect(0, 11565, '\P{Scx=	Geor}', "");
Expect(1, 11565, '\P{^Scx=	Geor}', "");
Expect(0, 11566, '\p{Scx=	Geor}', "");
Expect(1, 11566, '\p{^Scx=	Geor}', "");
Expect(1, 11566, '\P{Scx=	Geor}', "");
Expect(0, 11566, '\P{^Scx=	Geor}', "");
Error('\p{Is_Script_Extensions=	-GEORGIAN/a/}');
Error('\P{Is_Script_Extensions=	-GEORGIAN/a/}');
Expect(1, 11565, '\p{Is_Script_Extensions=georgian}', "");
Expect(0, 11565, '\p{^Is_Script_Extensions=georgian}', "");
Expect(0, 11565, '\P{Is_Script_Extensions=georgian}', "");
Expect(1, 11565, '\P{^Is_Script_Extensions=georgian}', "");
Expect(0, 11566, '\p{Is_Script_Extensions=georgian}', "");
Expect(1, 11566, '\p{^Is_Script_Extensions=georgian}', "");
Expect(1, 11566, '\P{Is_Script_Extensions=georgian}', "");
Expect(0, 11566, '\P{^Is_Script_Extensions=georgian}', "");
Expect(1, 11565, '\p{Is_Script_Extensions=_-Georgian}', "");
Expect(0, 11565, '\p{^Is_Script_Extensions=_-Georgian}', "");
Expect(0, 11565, '\P{Is_Script_Extensions=_-Georgian}', "");
Expect(1, 11565, '\P{^Is_Script_Extensions=_-Georgian}', "");
Expect(0, 11566, '\p{Is_Script_Extensions=_-Georgian}', "");
Expect(1, 11566, '\p{^Is_Script_Extensions=_-Georgian}', "");
Expect(1, 11566, '\P{Is_Script_Extensions=_-Georgian}', "");
Expect(0, 11566, '\P{^Is_Script_Extensions=_-Georgian}', "");
Error('\p{Is_Scx=:=_GEOR}');
Error('\P{Is_Scx=:=_GEOR}');
Expect(1, 11565, '\p{Is_Scx=geor}', "");
Expect(0, 11565, '\p{^Is_Scx=geor}', "");
Expect(0, 11565, '\P{Is_Scx=geor}', "");
Expect(1, 11565, '\P{^Is_Scx=geor}', "");
Expect(0, 11566, '\p{Is_Scx=geor}', "");
Expect(1, 11566, '\p{^Is_Scx=geor}', "");
Expect(1, 11566, '\P{Is_Scx=geor}', "");
Expect(0, 11566, '\P{^Is_Scx=geor}', "");
Expect(1, 11565, '\p{Is_Scx=	-Geor}', "");
Expect(0, 11565, '\p{^Is_Scx=	-Geor}', "");
Expect(0, 11565, '\P{Is_Scx=	-Geor}', "");
Expect(1, 11565, '\P{^Is_Scx=	-Geor}', "");
Expect(0, 11566, '\p{Is_Scx=	-Geor}', "");
Expect(1, 11566, '\p{^Is_Scx=	-Geor}', "");
Expect(1, 11566, '\P{Is_Scx=	-Geor}', "");
Expect(0, 11566, '\P{^Is_Scx=	-Geor}', "");
Error('\p{Script_Extensions= /a/Telugu}');
Error('\P{Script_Extensions= /a/Telugu}');
Expect(1, 3199, '\p{Script_Extensions=telugu}', "");
Expect(0, 3199, '\p{^Script_Extensions=telugu}', "");
Expect(0, 3199, '\P{Script_Extensions=telugu}', "");
Expect(1, 3199, '\P{^Script_Extensions=telugu}', "");
Expect(0, 3200, '\p{Script_Extensions=telugu}', "");
Expect(1, 3200, '\p{^Script_Extensions=telugu}', "");
Expect(1, 3200, '\P{Script_Extensions=telugu}', "");
Expect(0, 3200, '\P{^Script_Extensions=telugu}', "");
Expect(1, 3199, '\p{Script_Extensions=--Telugu}', "");
Expect(0, 3199, '\p{^Script_Extensions=--Telugu}', "");
Expect(0, 3199, '\P{Script_Extensions=--Telugu}', "");
Expect(1, 3199, '\P{^Script_Extensions=--Telugu}', "");
Expect(0, 3200, '\p{Script_Extensions=--Telugu}', "");
Expect(1, 3200, '\p{^Script_Extensions=--Telugu}', "");
Expect(1, 3200, '\P{Script_Extensions=--Telugu}', "");
Expect(0, 3200, '\P{^Script_Extensions=--Telugu}', "");
Error('\p{Scx=	 Telu:=}');
Error('\P{Scx=	 Telu:=}');
Expect(1, 3199, '\p{Scx=telu}', "");
Expect(0, 3199, '\p{^Scx=telu}', "");
Expect(0, 3199, '\P{Scx=telu}', "");
Expect(1, 3199, '\P{^Scx=telu}', "");
Expect(0, 3200, '\p{Scx=telu}', "");
Expect(1, 3200, '\p{^Scx=telu}', "");
Expect(1, 3200, '\P{Scx=telu}', "");
Expect(0, 3200, '\P{^Scx=telu}', "");
Expect(1, 3199, '\p{Scx=	TELU}', "");
Expect(0, 3199, '\p{^Scx=	TELU}', "");
Expect(0, 3199, '\P{Scx=	TELU}', "");
Expect(1, 3199, '\P{^Scx=	TELU}', "");
Expect(0, 3200, '\p{Scx=	TELU}', "");
Expect(1, 3200, '\p{^Scx=	TELU}', "");
Expect(1, 3200, '\P{Scx=	TELU}', "");
Expect(0, 3200, '\P{^Scx=	TELU}', "");
Error('\p{Is_Script_Extensions=	/a/Telugu}');
Error('\P{Is_Script_Extensions=	/a/Telugu}');
Expect(1, 3199, '\p{Is_Script_Extensions=telugu}', "");
Expect(0, 3199, '\p{^Is_Script_Extensions=telugu}', "");
Expect(0, 3199, '\P{Is_Script_Extensions=telugu}', "");
Expect(1, 3199, '\P{^Is_Script_Extensions=telugu}', "");
Expect(0, 3200, '\p{Is_Script_Extensions=telugu}', "");
Expect(1, 3200, '\p{^Is_Script_Extensions=telugu}', "");
Expect(1, 3200, '\P{Is_Script_Extensions=telugu}', "");
Expect(0, 3200, '\P{^Is_Script_Extensions=telugu}', "");
Expect(1, 3199, '\p{Is_Script_Extensions=_TELUGU}', "");
Expect(0, 3199, '\p{^Is_Script_Extensions=_TELUGU}', "");
Expect(0, 3199, '\P{Is_Script_Extensions=_TELUGU}', "");
Expect(1, 3199, '\P{^Is_Script_Extensions=_TELUGU}', "");
Expect(0, 3200, '\p{Is_Script_Extensions=_TELUGU}', "");
Expect(1, 3200, '\p{^Is_Script_Extensions=_TELUGU}', "");
Expect(1, 3200, '\P{Is_Script_Extensions=_TELUGU}', "");
Expect(0, 3200, '\P{^Is_Script_Extensions=_TELUGU}', "");
Error('\p{Is_Scx=/a/Telu}');
Error('\P{Is_Scx=/a/Telu}');
Expect(1, 3199, '\p{Is_Scx=telu}', "");
Expect(0, 3199, '\p{^Is_Scx=telu}', "");
Expect(0, 3199, '\P{Is_Scx=telu}', "");
Expect(1, 3199, '\P{^Is_Scx=telu}', "");
Expect(0, 3200, '\p{Is_Scx=telu}', "");
Expect(1, 3200, '\p{^Is_Scx=telu}', "");
Expect(1, 3200, '\P{Is_Scx=telu}', "");
Expect(0, 3200, '\P{^Is_Scx=telu}', "");
Expect(1, 3199, '\p{Is_Scx=	-TELU}', "");
Expect(0, 3199, '\p{^Is_Scx=	-TELU}', "");
Expect(0, 3199, '\P{Is_Scx=	-TELU}', "");
Expect(1, 3199, '\P{^Is_Scx=	-TELU}', "");
Expect(0, 3200, '\p{Is_Scx=	-TELU}', "");
Expect(1, 3200, '\p{^Is_Scx=	-TELU}', "");
Expect(1, 3200, '\P{Is_Scx=	-TELU}', "");
Expect(0, 3200, '\P{^Is_Scx=	-TELU}', "");
Error('\p{Script_Extensions=_/a/TIFINAGH}');
Error('\P{Script_Extensions=_/a/TIFINAGH}');
Expect(1, 11647, '\p{Script_Extensions=tifinagh}', "");
Expect(0, 11647, '\p{^Script_Extensions=tifinagh}', "");
Expect(0, 11647, '\P{Script_Extensions=tifinagh}', "");
Expect(1, 11647, '\P{^Script_Extensions=tifinagh}', "");
Expect(0, 11648, '\p{Script_Extensions=tifinagh}', "");
Expect(1, 11648, '\p{^Script_Extensions=tifinagh}', "");
Expect(1, 11648, '\P{Script_Extensions=tifinagh}', "");
Expect(0, 11648, '\P{^Script_Extensions=tifinagh}', "");
Expect(1, 11647, '\p{Script_Extensions=__Tifinagh}', "");
Expect(0, 11647, '\p{^Script_Extensions=__Tifinagh}', "");
Expect(0, 11647, '\P{Script_Extensions=__Tifinagh}', "");
Expect(1, 11647, '\P{^Script_Extensions=__Tifinagh}', "");
Expect(0, 11648, '\p{Script_Extensions=__Tifinagh}', "");
Expect(1, 11648, '\p{^Script_Extensions=__Tifinagh}', "");
Expect(1, 11648, '\P{Script_Extensions=__Tifinagh}', "");
Expect(0, 11648, '\P{^Script_Extensions=__Tifinagh}', "");
Error('\p{Scx:/a/--tfng}');
Error('\P{Scx:/a/--tfng}');
Expect(1, 11647, '\p{Scx=tfng}', "");
Expect(0, 11647, '\p{^Scx=tfng}', "");
Expect(0, 11647, '\P{Scx=tfng}', "");
Expect(1, 11647, '\P{^Scx=tfng}', "");
Expect(0, 11648, '\p{Scx=tfng}', "");
Expect(1, 11648, '\p{^Scx=tfng}', "");
Expect(1, 11648, '\P{Scx=tfng}', "");
Expect(0, 11648, '\P{^Scx=tfng}', "");
Expect(1, 11647, '\p{Scx=	-Tfng}', "");
Expect(0, 11647, '\p{^Scx=	-Tfng}', "");
Expect(0, 11647, '\P{Scx=	-Tfng}', "");
Expect(1, 11647, '\P{^Scx=	-Tfng}', "");
Expect(0, 11648, '\p{Scx=	-Tfng}', "");
Expect(1, 11648, '\p{^Scx=	-Tfng}', "");
Expect(1, 11648, '\P{Scx=	-Tfng}', "");
Expect(0, 11648, '\P{^Scx=	-Tfng}', "");
Error('\p{Is_Script_Extensions=/a/_-tifinagh}');
Error('\P{Is_Script_Extensions=/a/_-tifinagh}');
Expect(1, 11647, '\p{Is_Script_Extensions=tifinagh}', "");
Expect(0, 11647, '\p{^Is_Script_Extensions=tifinagh}', "");
Expect(0, 11647, '\P{Is_Script_Extensions=tifinagh}', "");
Expect(1, 11647, '\P{^Is_Script_Extensions=tifinagh}', "");
Expect(0, 11648, '\p{Is_Script_Extensions=tifinagh}', "");
Expect(1, 11648, '\p{^Is_Script_Extensions=tifinagh}', "");
Expect(1, 11648, '\P{Is_Script_Extensions=tifinagh}', "");
Expect(0, 11648, '\P{^Is_Script_Extensions=tifinagh}', "");
Expect(1, 11647, '\p{Is_Script_Extensions=_Tifinagh}', "");
Expect(0, 11647, '\p{^Is_Script_Extensions=_Tifinagh}', "");
Expect(0, 11647, '\P{Is_Script_Extensions=_Tifinagh}', "");
Expect(1, 11647, '\P{^Is_Script_Extensions=_Tifinagh}', "");
Expect(0, 11648, '\p{Is_Script_Extensions=_Tifinagh}', "");
Expect(1, 11648, '\p{^Is_Script_Extensions=_Tifinagh}', "");
Expect(1, 11648, '\P{Is_Script_Extensions=_Tifinagh}', "");
Expect(0, 11648, '\P{^Is_Script_Extensions=_Tifinagh}', "");
Error('\p{Is_Scx=	/a/TFNG}');
Error('\P{Is_Scx=	/a/TFNG}');
Expect(1, 11647, '\p{Is_Scx=tfng}', "");
Expect(0, 11647, '\p{^Is_Scx=tfng}', "");
Expect(0, 11647, '\P{Is_Scx=tfng}', "");
Expect(1, 11647, '\P{^Is_Scx=tfng}', "");
Expect(0, 11648, '\p{Is_Scx=tfng}', "");
Expect(1, 11648, '\p{^Is_Scx=tfng}', "");
Expect(1, 11648, '\P{Is_Scx=tfng}', "");
Expect(0, 11648, '\P{^Is_Scx=tfng}', "");
Expect(1, 11647, '\p{Is_Scx=	-Tfng}', "");
Expect(0, 11647, '\p{^Is_Scx=	-Tfng}', "");
Expect(0, 11647, '\P{Is_Scx=	-Tfng}', "");
Expect(1, 11647, '\P{^Is_Scx=	-Tfng}', "");
Expect(0, 11648, '\p{Is_Scx=	-Tfng}', "");
Expect(1, 11648, '\p{^Is_Scx=	-Tfng}', "");
Expect(1, 11648, '\P{Is_Scx=	-Tfng}', "");
Expect(0, 11648, '\P{^Is_Scx=	-Tfng}', "");
Error('\p{Script_Extensions= :=Sinhala}');
Error('\P{Script_Extensions= :=Sinhala}');
Expect(1, 3572, '\p{Script_Extensions=sinhala}', "");
Expect(0, 3572, '\p{^Script_Extensions=sinhala}', "");
Expect(0, 3572, '\P{Script_Extensions=sinhala}', "");
Expect(1, 3572, '\P{^Script_Extensions=sinhala}', "");
Expect(0, 3573, '\p{Script_Extensions=sinhala}', "");
Expect(1, 3573, '\p{^Script_Extensions=sinhala}', "");
Expect(1, 3573, '\P{Script_Extensions=sinhala}', "");
Expect(0, 3573, '\P{^Script_Extensions=sinhala}', "");
Expect(1, 3572, '\p{Script_Extensions=_SINHALA}', "");
Expect(0, 3572, '\p{^Script_Extensions=_SINHALA}', "");
Expect(0, 3572, '\P{Script_Extensions=_SINHALA}', "");
Expect(1, 3572, '\P{^Script_Extensions=_SINHALA}', "");
Expect(0, 3573, '\p{Script_Extensions=_SINHALA}', "");
Expect(1, 3573, '\p{^Script_Extensions=_SINHALA}', "");
Expect(1, 3573, '\P{Script_Extensions=_SINHALA}', "");
Expect(0, 3573, '\P{^Script_Extensions=_SINHALA}', "");
Error('\p{Scx=/a/_-SINH}');
Error('\P{Scx=/a/_-SINH}');
Expect(1, 3572, '\p{Scx=sinh}', "");
Expect(0, 3572, '\p{^Scx=sinh}', "");
Expect(0, 3572, '\P{Scx=sinh}', "");
Expect(1, 3572, '\P{^Scx=sinh}', "");
Expect(0, 3573, '\p{Scx=sinh}', "");
Expect(1, 3573, '\p{^Scx=sinh}', "");
Expect(1, 3573, '\P{Scx=sinh}', "");
Expect(0, 3573, '\P{^Scx=sinh}', "");
Expect(1, 3572, '\p{Scx= 	Sinh}', "");
Expect(0, 3572, '\p{^Scx= 	Sinh}', "");
Expect(0, 3572, '\P{Scx= 	Sinh}', "");
Expect(1, 3572, '\P{^Scx= 	Sinh}', "");
Expect(0, 3573, '\p{Scx= 	Sinh}', "");
Expect(1, 3573, '\p{^Scx= 	Sinh}', "");
Expect(1, 3573, '\P{Scx= 	Sinh}', "");
Expect(0, 3573, '\P{^Scx= 	Sinh}', "");
Error('\p{Is_Script_Extensions=-_SINHALA/a/}');
Error('\P{Is_Script_Extensions=-_SINHALA/a/}');
Expect(1, 3572, '\p{Is_Script_Extensions=sinhala}', "");
Expect(0, 3572, '\p{^Is_Script_Extensions=sinhala}', "");
Expect(0, 3572, '\P{Is_Script_Extensions=sinhala}', "");
Expect(1, 3572, '\P{^Is_Script_Extensions=sinhala}', "");
Expect(0, 3573, '\p{Is_Script_Extensions=sinhala}', "");
Expect(1, 3573, '\p{^Is_Script_Extensions=sinhala}', "");
Expect(1, 3573, '\P{Is_Script_Extensions=sinhala}', "");
Expect(0, 3573, '\P{^Is_Script_Extensions=sinhala}', "");
Expect(1, 3572, '\p{Is_Script_Extensions=-sinhala}', "");
Expect(0, 3572, '\p{^Is_Script_Extensions=-sinhala}', "");
Expect(0, 3572, '\P{Is_Script_Extensions=-sinhala}', "");
Expect(1, 3572, '\P{^Is_Script_Extensions=-sinhala}', "");
Expect(0, 3573, '\p{Is_Script_Extensions=-sinhala}', "");
Expect(1, 3573, '\p{^Is_Script_Extensions=-sinhala}', "");
Expect(1, 3573, '\P{Is_Script_Extensions=-sinhala}', "");
Expect(0, 3573, '\P{^Is_Script_Extensions=-sinhala}', "");
Error('\p{Is_Scx=_/a/sinh}');
Error('\P{Is_Scx=_/a/sinh}');
Expect(1, 3572, '\p{Is_Scx=sinh}', "");
Expect(0, 3572, '\p{^Is_Scx=sinh}', "");
Expect(0, 3572, '\P{Is_Scx=sinh}', "");
Expect(1, 3572, '\P{^Is_Scx=sinh}', "");
Expect(0, 3573, '\p{Is_Scx=sinh}', "");
Expect(1, 3573, '\p{^Is_Scx=sinh}', "");
Expect(1, 3573, '\P{Is_Scx=sinh}', "");
Expect(0, 3573, '\P{^Is_Scx=sinh}', "");
Expect(1, 3572, '\p{Is_Scx:	_Sinh}', "");
Expect(0, 3572, '\p{^Is_Scx:	_Sinh}', "");
Expect(0, 3572, '\P{Is_Scx:	_Sinh}', "");
Expect(1, 3572, '\P{^Is_Scx:	_Sinh}', "");
Expect(0, 3573, '\p{Is_Scx:	_Sinh}', "");
Expect(1, 3573, '\p{^Is_Scx:	_Sinh}', "");
Expect(1, 3573, '\P{Is_Scx:	_Sinh}', "");
Expect(0, 3573, '\P{^Is_Scx:	_Sinh}', "");
Error('\p{Script_Extensions=:=Miao}');
Error('\P{Script_Extensions=:=Miao}');
Expect(1, 94111, '\p{Script_Extensions=miao}', "");
Expect(0, 94111, '\p{^Script_Extensions=miao}', "");
Expect(0, 94111, '\P{Script_Extensions=miao}', "");
Expect(1, 94111, '\P{^Script_Extensions=miao}', "");
Expect(0, 94112, '\p{Script_Extensions=miao}', "");
Expect(1, 94112, '\p{^Script_Extensions=miao}', "");
Expect(1, 94112, '\P{Script_Extensions=miao}', "");
Expect(0, 94112, '\P{^Script_Extensions=miao}', "");
Expect(1, 94111, '\p{Script_Extensions=_	miao}', "");
Expect(0, 94111, '\p{^Script_Extensions=_	miao}', "");
Expect(0, 94111, '\P{Script_Extensions=_	miao}', "");
Expect(1, 94111, '\P{^Script_Extensions=_	miao}', "");
Expect(0, 94112, '\p{Script_Extensions=_	miao}', "");
Expect(1, 94112, '\p{^Script_Extensions=_	miao}', "");
Expect(1, 94112, '\P{Script_Extensions=_	miao}', "");
Expect(0, 94112, '\P{^Script_Extensions=_	miao}', "");
Error('\p{Scx= /a/PLRD}');
Error('\P{Scx= /a/PLRD}');
Expect(1, 94111, '\p{Scx=plrd}', "");
Expect(0, 94111, '\p{^Scx=plrd}', "");
Expect(0, 94111, '\P{Scx=plrd}', "");
Expect(1, 94111, '\P{^Scx=plrd}', "");
Expect(0, 94112, '\p{Scx=plrd}', "");
Expect(1, 94112, '\p{^Scx=plrd}', "");
Expect(1, 94112, '\P{Scx=plrd}', "");
Expect(0, 94112, '\P{^Scx=plrd}', "");
Expect(1, 94111, '\p{Scx=	_PLRD}', "");
Expect(0, 94111, '\p{^Scx=	_PLRD}', "");
Expect(0, 94111, '\P{Scx=	_PLRD}', "");
Expect(1, 94111, '\P{^Scx=	_PLRD}', "");
Expect(0, 94112, '\p{Scx=	_PLRD}', "");
Expect(1, 94112, '\p{^Scx=	_PLRD}', "");
Expect(1, 94112, '\P{Scx=	_PLRD}', "");
Expect(0, 94112, '\P{^Scx=	_PLRD}', "");
Error('\p{Is_Script_Extensions=:= miao}');
Error('\P{Is_Script_Extensions=:= miao}');
Expect(1, 94111, '\p{Is_Script_Extensions:miao}', "");
Expect(0, 94111, '\p{^Is_Script_Extensions:miao}', "");
Expect(0, 94111, '\P{Is_Script_Extensions:miao}', "");
Expect(1, 94111, '\P{^Is_Script_Extensions:miao}', "");
Expect(0, 94112, '\p{Is_Script_Extensions:miao}', "");
Expect(1, 94112, '\p{^Is_Script_Extensions:miao}', "");
Expect(1, 94112, '\P{Is_Script_Extensions:miao}', "");
Expect(0, 94112, '\P{^Is_Script_Extensions:miao}', "");
Expect(1, 94111, '\p{Is_Script_Extensions=		miao}', "");
Expect(0, 94111, '\p{^Is_Script_Extensions=		miao}', "");
Expect(0, 94111, '\P{Is_Script_Extensions=		miao}', "");
Expect(1, 94111, '\P{^Is_Script_Extensions=		miao}', "");
Expect(0, 94112, '\p{Is_Script_Extensions=		miao}', "");
Expect(1, 94112, '\p{^Is_Script_Extensions=		miao}', "");
Expect(1, 94112, '\P{Is_Script_Extensions=		miao}', "");
Expect(0, 94112, '\P{^Is_Script_Extensions=		miao}', "");
Error('\p{Is_Scx:_ plrd/a/}');
Error('\P{Is_Scx:_ plrd/a/}');
Expect(1, 94111, '\p{Is_Scx: plrd}', "");
Expect(0, 94111, '\p{^Is_Scx: plrd}', "");
Expect(0, 94111, '\P{Is_Scx: plrd}', "");
Expect(1, 94111, '\P{^Is_Scx: plrd}', "");
Expect(0, 94112, '\p{Is_Scx: plrd}', "");
Expect(1, 94112, '\p{^Is_Scx: plrd}', "");
Expect(1, 94112, '\P{Is_Scx: plrd}', "");
Expect(0, 94112, '\P{^Is_Scx: plrd}', "");
Expect(1, 94111, '\p{Is_Scx=-PLRD}', "");
Expect(0, 94111, '\p{^Is_Scx=-PLRD}', "");
Expect(0, 94111, '\P{Is_Scx=-PLRD}', "");
Expect(1, 94111, '\P{^Is_Scx=-PLRD}', "");
Expect(0, 94112, '\p{Is_Scx=-PLRD}', "");
Expect(1, 94112, '\p{^Is_Scx=-PLRD}', "");
Expect(1, 94112, '\P{Is_Scx=-PLRD}', "");
Expect(0, 94112, '\P{^Is_Scx=-PLRD}', "");
Error('\p{Script_Extensions=	:=linear_b}');
Error('\P{Script_Extensions=	:=linear_b}');
Expect(1, 65855, '\p{Script_Extensions=linearb}', "");
Expect(0, 65855, '\p{^Script_Extensions=linearb}', "");
Expect(0, 65855, '\P{Script_Extensions=linearb}', "");
Expect(1, 65855, '\P{^Script_Extensions=linearb}', "");
Expect(0, 65856, '\p{Script_Extensions=linearb}', "");
Expect(1, 65856, '\p{^Script_Extensions=linearb}', "");
Expect(1, 65856, '\P{Script_Extensions=linearb}', "");
Expect(0, 65856, '\P{^Script_Extensions=linearb}', "");
Expect(1, 65855, '\p{Script_Extensions= -LINEAR_B}', "");
Expect(0, 65855, '\p{^Script_Extensions= -LINEAR_B}', "");
Expect(0, 65855, '\P{Script_Extensions= -LINEAR_B}', "");
Expect(1, 65855, '\P{^Script_Extensions= -LINEAR_B}', "");
Expect(0, 65856, '\p{Script_Extensions= -LINEAR_B}', "");
Expect(1, 65856, '\p{^Script_Extensions= -LINEAR_B}', "");
Expect(1, 65856, '\P{Script_Extensions= -LINEAR_B}', "");
Expect(0, 65856, '\P{^Script_Extensions= -LINEAR_B}', "");
Error('\p{Scx=:=	_Linb}');
Error('\P{Scx=:=	_Linb}');
Expect(1, 65855, '\p{Scx=linb}', "");
Expect(0, 65855, '\p{^Scx=linb}', "");
Expect(0, 65855, '\P{Scx=linb}', "");
Expect(1, 65855, '\P{^Scx=linb}', "");
Expect(0, 65856, '\p{Scx=linb}', "");
Expect(1, 65856, '\p{^Scx=linb}', "");
Expect(1, 65856, '\P{Scx=linb}', "");
Expect(0, 65856, '\P{^Scx=linb}', "");
Expect(1, 65855, '\p{Scx=__linb}', "");
Expect(0, 65855, '\p{^Scx=__linb}', "");
Expect(0, 65855, '\P{Scx=__linb}', "");
Expect(1, 65855, '\P{^Scx=__linb}', "");
Expect(0, 65856, '\p{Scx=__linb}', "");
Expect(1, 65856, '\p{^Scx=__linb}', "");
Expect(1, 65856, '\P{Scx=__linb}', "");
Expect(0, 65856, '\P{^Scx=__linb}', "");
Error('\p{Is_Script_Extensions=_:=LINEAR_B}');
Error('\P{Is_Script_Extensions=_:=LINEAR_B}');
Expect(1, 65855, '\p{Is_Script_Extensions=linearb}', "");
Expect(0, 65855, '\p{^Is_Script_Extensions=linearb}', "");
Expect(0, 65855, '\P{Is_Script_Extensions=linearb}', "");
Expect(1, 65855, '\P{^Is_Script_Extensions=linearb}', "");
Expect(0, 65856, '\p{Is_Script_Extensions=linearb}', "");
Expect(1, 65856, '\p{^Is_Script_Extensions=linearb}', "");
Expect(1, 65856, '\P{Is_Script_Extensions=linearb}', "");
Expect(0, 65856, '\P{^Is_Script_Extensions=linearb}', "");
Expect(1, 65855, '\p{Is_Script_Extensions=		Linear_B}', "");
Expect(0, 65855, '\p{^Is_Script_Extensions=		Linear_B}', "");
Expect(0, 65855, '\P{Is_Script_Extensions=		Linear_B}', "");
Expect(1, 65855, '\P{^Is_Script_Extensions=		Linear_B}', "");
Expect(0, 65856, '\p{Is_Script_Extensions=		Linear_B}', "");
Expect(1, 65856, '\p{^Is_Script_Extensions=		Linear_B}', "");
Expect(1, 65856, '\P{Is_Script_Extensions=		Linear_B}', "");
Expect(0, 65856, '\P{^Is_Script_Extensions=		Linear_B}', "");
Error('\p{Is_Scx=	:=linb}');
Error('\P{Is_Scx=	:=linb}');
Expect(1, 65855, '\p{Is_Scx=linb}', "");
Expect(0, 65855, '\p{^Is_Scx=linb}', "");
Expect(0, 65855, '\P{Is_Scx=linb}', "");
Expect(1, 65855, '\P{^Is_Scx=linb}', "");
Expect(0, 65856, '\p{Is_Scx=linb}', "");
Expect(1, 65856, '\p{^Is_Scx=linb}', "");
Expect(1, 65856, '\P{Is_Scx=linb}', "");
Expect(0, 65856, '\P{^Is_Scx=linb}', "");
Expect(1, 65855, '\p{Is_Scx=_Linb}', "");
Expect(0, 65855, '\p{^Is_Scx=_Linb}', "");
Expect(0, 65855, '\P{Is_Scx=_Linb}', "");
Expect(1, 65855, '\P{^Is_Scx=_Linb}', "");
Expect(0, 65856, '\p{Is_Scx=_Linb}', "");
Expect(1, 65856, '\p{^Is_Scx=_Linb}', "");
Expect(1, 65856, '\P{Is_Scx=_Linb}', "");
Expect(0, 65856, '\P{^Is_Scx=_Linb}', "");
Error('\p{Script_Extensions=/a/_SAMARITAN}');
Error('\P{Script_Extensions=/a/_SAMARITAN}');
Expect(1, 2110, '\p{Script_Extensions=samaritan}', "");
Expect(0, 2110, '\p{^Script_Extensions=samaritan}', "");
Expect(0, 2110, '\P{Script_Extensions=samaritan}', "");
Expect(1, 2110, '\P{^Script_Extensions=samaritan}', "");
Expect(0, 2111, '\p{Script_Extensions=samaritan}', "");
Expect(1, 2111, '\p{^Script_Extensions=samaritan}', "");
Expect(1, 2111, '\P{Script_Extensions=samaritan}', "");
Expect(0, 2111, '\P{^Script_Extensions=samaritan}', "");
Expect(1, 2110, '\p{Script_Extensions=_ Samaritan}', "");
Expect(0, 2110, '\p{^Script_Extensions=_ Samaritan}', "");
Expect(0, 2110, '\P{Script_Extensions=_ Samaritan}', "");
Expect(1, 2110, '\P{^Script_Extensions=_ Samaritan}', "");
Expect(0, 2111, '\p{Script_Extensions=_ Samaritan}', "");
Expect(1, 2111, '\p{^Script_Extensions=_ Samaritan}', "");
Expect(1, 2111, '\P{Script_Extensions=_ Samaritan}', "");
Expect(0, 2111, '\P{^Script_Extensions=_ Samaritan}', "");
Error('\p{Scx=:=_	Samr}');
Error('\P{Scx=:=_	Samr}');
Expect(1, 2110, '\p{Scx=samr}', "");
Expect(0, 2110, '\p{^Scx=samr}', "");
Expect(0, 2110, '\P{Scx=samr}', "");
Expect(1, 2110, '\P{^Scx=samr}', "");
Expect(0, 2111, '\p{Scx=samr}', "");
Expect(1, 2111, '\p{^Scx=samr}', "");
Expect(1, 2111, '\P{Scx=samr}', "");
Expect(0, 2111, '\P{^Scx=samr}', "");
Expect(1, 2110, '\p{Scx:  Samr}', "");
Expect(0, 2110, '\p{^Scx:  Samr}', "");
Expect(0, 2110, '\P{Scx:  Samr}', "");
Expect(1, 2110, '\P{^Scx:  Samr}', "");
Expect(0, 2111, '\p{Scx:  Samr}', "");
Expect(1, 2111, '\p{^Scx:  Samr}', "");
Expect(1, 2111, '\P{Scx:  Samr}', "");
Expect(0, 2111, '\P{^Scx:  Samr}', "");
Error('\p{Is_Script_Extensions=:= samaritan}');
Error('\P{Is_Script_Extensions=:= samaritan}');
Expect(1, 2110, '\p{Is_Script_Extensions=samaritan}', "");
Expect(0, 2110, '\p{^Is_Script_Extensions=samaritan}', "");
Expect(0, 2110, '\P{Is_Script_Extensions=samaritan}', "");
Expect(1, 2110, '\P{^Is_Script_Extensions=samaritan}', "");
Expect(0, 2111, '\p{Is_Script_Extensions=samaritan}', "");
Expect(1, 2111, '\p{^Is_Script_Extensions=samaritan}', "");
Expect(1, 2111, '\P{Is_Script_Extensions=samaritan}', "");
Expect(0, 2111, '\P{^Is_Script_Extensions=samaritan}', "");
Expect(1, 2110, '\p{Is_Script_Extensions=--Samaritan}', "");
Expect(0, 2110, '\p{^Is_Script_Extensions=--Samaritan}', "");
Expect(0, 2110, '\P{Is_Script_Extensions=--Samaritan}', "");
Expect(1, 2110, '\P{^Is_Script_Extensions=--Samaritan}', "");
Expect(0, 2111, '\p{Is_Script_Extensions=--Samaritan}', "");
Expect(1, 2111, '\p{^Is_Script_Extensions=--Samaritan}', "");
Expect(1, 2111, '\P{Is_Script_Extensions=--Samaritan}', "");
Expect(0, 2111, '\P{^Is_Script_Extensions=--Samaritan}', "");
Error('\p{Is_Scx=/a/ -samr}');
Error('\P{Is_Scx=/a/ -samr}');
Expect(1, 2110, '\p{Is_Scx=samr}', "");
Expect(0, 2110, '\p{^Is_Scx=samr}', "");
Expect(0, 2110, '\P{Is_Scx=samr}', "");
Expect(1, 2110, '\P{^Is_Scx=samr}', "");
Expect(0, 2111, '\p{Is_Scx=samr}', "");
Expect(1, 2111, '\p{^Is_Scx=samr}', "");
Expect(1, 2111, '\P{Is_Scx=samr}', "");
Expect(0, 2111, '\P{^Is_Scx=samr}', "");
Expect(1, 2110, '\p{Is_Scx:-	samr}', "");
Expect(0, 2110, '\p{^Is_Scx:-	samr}', "");
Expect(0, 2110, '\P{Is_Scx:-	samr}', "");
Expect(1, 2110, '\P{^Is_Scx:-	samr}', "");
Expect(0, 2111, '\p{Is_Scx:-	samr}', "");
Expect(1, 2111, '\p{^Is_Scx:-	samr}', "");
Expect(1, 2111, '\P{Is_Scx:-	samr}', "");
Expect(0, 2111, '\P{^Is_Scx:-	samr}', "");
Error('\p{Script_Extensions=saurashtra:=}');
Error('\P{Script_Extensions=saurashtra:=}');
Expect(1, 43225, '\p{Script_Extensions=saurashtra}', "");
Expect(0, 43225, '\p{^Script_Extensions=saurashtra}', "");
Expect(0, 43225, '\P{Script_Extensions=saurashtra}', "");
Expect(1, 43225, '\P{^Script_Extensions=saurashtra}', "");
Expect(0, 43226, '\p{Script_Extensions=saurashtra}', "");
Expect(1, 43226, '\p{^Script_Extensions=saurashtra}', "");
Expect(1, 43226, '\P{Script_Extensions=saurashtra}', "");
Expect(0, 43226, '\P{^Script_Extensions=saurashtra}', "");
Expect(1, 43225, '\p{Script_Extensions=-	Saurashtra}', "");
Expect(0, 43225, '\p{^Script_Extensions=-	Saurashtra}', "");
Expect(0, 43225, '\P{Script_Extensions=-	Saurashtra}', "");
Expect(1, 43225, '\P{^Script_Extensions=-	Saurashtra}', "");
Expect(0, 43226, '\p{Script_Extensions=-	Saurashtra}', "");
Expect(1, 43226, '\p{^Script_Extensions=-	Saurashtra}', "");
Expect(1, 43226, '\P{Script_Extensions=-	Saurashtra}', "");
Expect(0, 43226, '\P{^Script_Extensions=-	Saurashtra}', "");
Error('\p{Scx=/a/Saur}');
Error('\P{Scx=/a/Saur}');
Expect(1, 43225, '\p{Scx=saur}', "");
Expect(0, 43225, '\p{^Scx=saur}', "");
Expect(0, 43225, '\P{Scx=saur}', "");
Expect(1, 43225, '\P{^Scx=saur}', "");
Expect(0, 43226, '\p{Scx=saur}', "");
Expect(1, 43226, '\p{^Scx=saur}', "");
Expect(1, 43226, '\P{Scx=saur}', "");
Expect(0, 43226, '\P{^Scx=saur}', "");
Expect(1, 43225, '\p{Scx=--SAUR}', "");
Expect(0, 43225, '\p{^Scx=--SAUR}', "");
Expect(0, 43225, '\P{Scx=--SAUR}', "");
Expect(1, 43225, '\P{^Scx=--SAUR}', "");
Expect(0, 43226, '\p{Scx=--SAUR}', "");
Expect(1, 43226, '\p{^Scx=--SAUR}', "");
Expect(1, 43226, '\P{Scx=--SAUR}', "");
Expect(0, 43226, '\P{^Scx=--SAUR}', "");
Error('\p{Is_Script_Extensions=:=_-SAURASHTRA}');
Error('\P{Is_Script_Extensions=:=_-SAURASHTRA}');
Expect(1, 43225, '\p{Is_Script_Extensions: saurashtra}', "");
Expect(0, 43225, '\p{^Is_Script_Extensions: saurashtra}', "");
Expect(0, 43225, '\P{Is_Script_Extensions: saurashtra}', "");
Expect(1, 43225, '\P{^Is_Script_Extensions: saurashtra}', "");
Expect(0, 43226, '\p{Is_Script_Extensions: saurashtra}', "");
Expect(1, 43226, '\p{^Is_Script_Extensions: saurashtra}', "");
Expect(1, 43226, '\P{Is_Script_Extensions: saurashtra}', "");
Expect(0, 43226, '\P{^Is_Script_Extensions: saurashtra}', "");
Expect(1, 43225, '\p{Is_Script_Extensions=-saurashtra}', "");
Expect(0, 43225, '\p{^Is_Script_Extensions=-saurashtra}', "");
Expect(0, 43225, '\P{Is_Script_Extensions=-saurashtra}', "");
Expect(1, 43225, '\P{^Is_Script_Extensions=-saurashtra}', "");
Expect(0, 43226, '\p{Is_Script_Extensions=-saurashtra}', "");
Expect(1, 43226, '\p{^Is_Script_Extensions=-saurashtra}', "");
Expect(1, 43226, '\P{Is_Script_Extensions=-saurashtra}', "");
Expect(0, 43226, '\P{^Is_Script_Extensions=-saurashtra}', "");
Error('\p{Is_Scx:	:=	 Saur}');
Error('\P{Is_Scx:	:=	 Saur}');
Expect(1, 43225, '\p{Is_Scx=saur}', "");
Expect(0, 43225, '\p{^Is_Scx=saur}', "");
Expect(0, 43225, '\P{Is_Scx=saur}', "");
Expect(1, 43225, '\P{^Is_Scx=saur}', "");
Expect(0, 43226, '\p{Is_Scx=saur}', "");
Expect(1, 43226, '\p{^Is_Scx=saur}', "");
Expect(1, 43226, '\P{Is_Scx=saur}', "");
Expect(0, 43226, '\P{^Is_Scx=saur}', "");
Expect(1, 43225, '\p{Is_Scx:  Saur}', "");
Expect(0, 43225, '\p{^Is_Scx:  Saur}', "");
Expect(0, 43225, '\P{Is_Scx:  Saur}', "");
Expect(1, 43225, '\P{^Is_Scx:  Saur}', "");
Expect(0, 43226, '\p{Is_Scx:  Saur}', "");
Expect(1, 43226, '\p{^Is_Scx:  Saur}', "");
Expect(1, 43226, '\P{Is_Scx:  Saur}', "");
Expect(0, 43226, '\P{^Is_Scx:  Saur}', "");
Error('\p{Script_Extensions=	sundanese:=}');
Error('\P{Script_Extensions=	sundanese:=}');
Expect(1, 7367, '\p{Script_Extensions:	sundanese}', "");
Expect(0, 7367, '\p{^Script_Extensions:	sundanese}', "");
Expect(0, 7367, '\P{Script_Extensions:	sundanese}', "");
Expect(1, 7367, '\P{^Script_Extensions:	sundanese}', "");
Expect(0, 7368, '\p{Script_Extensions:	sundanese}', "");
Expect(1, 7368, '\p{^Script_Extensions:	sundanese}', "");
Expect(1, 7368, '\P{Script_Extensions:	sundanese}', "");
Expect(0, 7368, '\P{^Script_Extensions:	sundanese}', "");
Expect(1, 7367, '\p{Script_Extensions=_	sundanese}', "");
Expect(0, 7367, '\p{^Script_Extensions=_	sundanese}', "");
Expect(0, 7367, '\P{Script_Extensions=_	sundanese}', "");
Expect(1, 7367, '\P{^Script_Extensions=_	sundanese}', "");
Expect(0, 7368, '\p{Script_Extensions=_	sundanese}', "");
Expect(1, 7368, '\p{^Script_Extensions=_	sundanese}', "");
Expect(1, 7368, '\P{Script_Extensions=_	sundanese}', "");
Expect(0, 7368, '\P{^Script_Extensions=_	sundanese}', "");
Error('\p{Scx=:=-_Sund}');
Error('\P{Scx=:=-_Sund}');
Expect(1, 7367, '\p{Scx=sund}', "");
Expect(0, 7367, '\p{^Scx=sund}', "");
Expect(0, 7367, '\P{Scx=sund}', "");
Expect(1, 7367, '\P{^Scx=sund}', "");
Expect(0, 7368, '\p{Scx=sund}', "");
Expect(1, 7368, '\p{^Scx=sund}', "");
Expect(1, 7368, '\P{Scx=sund}', "");
Expect(0, 7368, '\P{^Scx=sund}', "");
Expect(1, 7367, '\p{Scx=__Sund}', "");
Expect(0, 7367, '\p{^Scx=__Sund}', "");
Expect(0, 7367, '\P{Scx=__Sund}', "");
Expect(1, 7367, '\P{^Scx=__Sund}', "");
Expect(0, 7368, '\p{Scx=__Sund}', "");
Expect(1, 7368, '\p{^Scx=__Sund}', "");
Expect(1, 7368, '\P{Scx=__Sund}', "");
Expect(0, 7368, '\P{^Scx=__Sund}', "");
Error('\p{Is_Script_Extensions=--sundanese:=}');
Error('\P{Is_Script_Extensions=--sundanese:=}');
Expect(1, 7367, '\p{Is_Script_Extensions=sundanese}', "");
Expect(0, 7367, '\p{^Is_Script_Extensions=sundanese}', "");
Expect(0, 7367, '\P{Is_Script_Extensions=sundanese}', "");
Expect(1, 7367, '\P{^Is_Script_Extensions=sundanese}', "");
Expect(0, 7368, '\p{Is_Script_Extensions=sundanese}', "");
Expect(1, 7368, '\p{^Is_Script_Extensions=sundanese}', "");
Expect(1, 7368, '\P{Is_Script_Extensions=sundanese}', "");
Expect(0, 7368, '\P{^Is_Script_Extensions=sundanese}', "");
Expect(1, 7367, '\p{Is_Script_Extensions=	 Sundanese}', "");
Expect(0, 7367, '\p{^Is_Script_Extensions=	 Sundanese}', "");
Expect(0, 7367, '\P{Is_Script_Extensions=	 Sundanese}', "");
Expect(1, 7367, '\P{^Is_Script_Extensions=	 Sundanese}', "");
Expect(0, 7368, '\p{Is_Script_Extensions=	 Sundanese}', "");
Expect(1, 7368, '\p{^Is_Script_Extensions=	 Sundanese}', "");
Expect(1, 7368, '\P{Is_Script_Extensions=	 Sundanese}', "");
Expect(0, 7368, '\P{^Is_Script_Extensions=	 Sundanese}', "");
Error('\p{Is_Scx=/a/--Sund}');
Error('\P{Is_Scx=/a/--Sund}');
Expect(1, 7367, '\p{Is_Scx:   sund}', "");
Expect(0, 7367, '\p{^Is_Scx:   sund}', "");
Expect(0, 7367, '\P{Is_Scx:   sund}', "");
Expect(1, 7367, '\P{^Is_Scx:   sund}', "");
Expect(0, 7368, '\p{Is_Scx:   sund}', "");
Expect(1, 7368, '\p{^Is_Scx:   sund}', "");
Expect(1, 7368, '\P{Is_Scx:   sund}', "");
Expect(0, 7368, '\P{^Is_Scx:   sund}', "");
Expect(1, 7367, '\p{Is_Scx= Sund}', "");
Expect(0, 7367, '\p{^Is_Scx= Sund}', "");
Expect(0, 7367, '\P{Is_Scx= Sund}', "");
Expect(1, 7367, '\P{^Is_Scx= Sund}', "");
Expect(0, 7368, '\p{Is_Scx= Sund}', "");
Expect(1, 7368, '\p{^Is_Scx= Sund}', "");
Expect(1, 7368, '\P{Is_Scx= Sund}', "");
Expect(0, 7368, '\P{^Is_Scx= Sund}', "");
Error('\p{Script_Extensions=:=- nko}');
Error('\P{Script_Extensions=:=- nko}');
Expect(1, 2042, '\p{Script_Extensions=nko}', "");
Expect(0, 2042, '\p{^Script_Extensions=nko}', "");
Expect(0, 2042, '\P{Script_Extensions=nko}', "");
Expect(1, 2042, '\P{^Script_Extensions=nko}', "");
Expect(0, 2043, '\p{Script_Extensions=nko}', "");
Expect(1, 2043, '\p{^Script_Extensions=nko}', "");
Expect(1, 2043, '\P{Script_Extensions=nko}', "");
Expect(0, 2043, '\P{^Script_Extensions=nko}', "");
Expect(1, 2042, '\p{Script_Extensions=- NKO}', "");
Expect(0, 2042, '\p{^Script_Extensions=- NKO}', "");
Expect(0, 2042, '\P{Script_Extensions=- NKO}', "");
Expect(1, 2042, '\P{^Script_Extensions=- NKO}', "");
Expect(0, 2043, '\p{Script_Extensions=- NKO}', "");
Expect(1, 2043, '\p{^Script_Extensions=- NKO}', "");
Expect(1, 2043, '\P{Script_Extensions=- NKO}', "");
Expect(0, 2043, '\P{^Script_Extensions=- NKO}', "");
Error('\p{Scx=_	NKOO:=}');
Error('\P{Scx=_	NKOO:=}');
Expect(1, 2042, '\p{Scx=nkoo}', "");
Expect(0, 2042, '\p{^Scx=nkoo}', "");
Expect(0, 2042, '\P{Scx=nkoo}', "");
Expect(1, 2042, '\P{^Scx=nkoo}', "");
Expect(0, 2043, '\p{Scx=nkoo}', "");
Expect(1, 2043, '\p{^Scx=nkoo}', "");
Expect(1, 2043, '\P{Scx=nkoo}', "");
Expect(0, 2043, '\P{^Scx=nkoo}', "");
Expect(1, 2042, '\p{Scx= -Nkoo}', "");
Expect(0, 2042, '\p{^Scx= -Nkoo}', "");
Expect(0, 2042, '\P{Scx= -Nkoo}', "");
Expect(1, 2042, '\P{^Scx= -Nkoo}', "");
Expect(0, 2043, '\p{Scx= -Nkoo}', "");
Expect(1, 2043, '\p{^Scx= -Nkoo}', "");
Expect(1, 2043, '\P{Scx= -Nkoo}', "");
Expect(0, 2043, '\P{^Scx= -Nkoo}', "");
Error('\p{Is_Script_Extensions=_	nko/a/}');
Error('\P{Is_Script_Extensions=_	nko/a/}');
Expect(1, 2042, '\p{Is_Script_Extensions=nko}', "");
Expect(0, 2042, '\p{^Is_Script_Extensions=nko}', "");
Expect(0, 2042, '\P{Is_Script_Extensions=nko}', "");
Expect(1, 2042, '\P{^Is_Script_Extensions=nko}', "");
Expect(0, 2043, '\p{Is_Script_Extensions=nko}', "");
Expect(1, 2043, '\p{^Is_Script_Extensions=nko}', "");
Expect(1, 2043, '\P{Is_Script_Extensions=nko}', "");
Expect(0, 2043, '\P{^Is_Script_Extensions=nko}', "");
Expect(1, 2042, '\p{Is_Script_Extensions=  nko}', "");
Expect(0, 2042, '\p{^Is_Script_Extensions=  nko}', "");
Expect(0, 2042, '\P{Is_Script_Extensions=  nko}', "");
Expect(1, 2042, '\P{^Is_Script_Extensions=  nko}', "");
Expect(0, 2043, '\p{Is_Script_Extensions=  nko}', "");
Expect(1, 2043, '\p{^Is_Script_Extensions=  nko}', "");
Expect(1, 2043, '\P{Is_Script_Extensions=  nko}', "");
Expect(0, 2043, '\P{^Is_Script_Extensions=  nko}', "");
Error('\p{Is_Scx:   NKOO/a/}');
Error('\P{Is_Scx:   NKOO/a/}');
Expect(1, 2042, '\p{Is_Scx=nkoo}', "");
Expect(0, 2042, '\p{^Is_Scx=nkoo}', "");
Expect(0, 2042, '\P{Is_Scx=nkoo}', "");
Expect(1, 2042, '\P{^Is_Scx=nkoo}', "");
Expect(0, 2043, '\p{Is_Scx=nkoo}', "");
Expect(1, 2043, '\p{^Is_Scx=nkoo}', "");
Expect(1, 2043, '\P{Is_Scx=nkoo}', "");
Expect(0, 2043, '\P{^Is_Scx=nkoo}', "");
Expect(1, 2042, '\p{Is_Scx=--Nkoo}', "");
Expect(0, 2042, '\p{^Is_Scx=--Nkoo}', "");
Expect(0, 2042, '\P{Is_Scx=--Nkoo}', "");
Expect(1, 2042, '\P{^Is_Scx=--Nkoo}', "");
Expect(0, 2043, '\p{Is_Scx=--Nkoo}', "");
Expect(1, 2043, '\p{^Is_Scx=--Nkoo}', "");
Expect(1, 2043, '\P{Is_Scx=--Nkoo}', "");
Expect(0, 2043, '\P{^Is_Scx=--Nkoo}', "");
Error('\p{Script_Extensions: -	han/a/}');
Error('\P{Script_Extensions: -	han/a/}');
Expect(1, 195101, '\p{Script_Extensions:han}', "");
Expect(0, 195101, '\p{^Script_Extensions:han}', "");
Expect(0, 195101, '\P{Script_Extensions:han}', "");
Expect(1, 195101, '\P{^Script_Extensions:han}', "");
Expect(0, 195102, '\p{Script_Extensions:han}', "");
Expect(1, 195102, '\p{^Script_Extensions:han}', "");
Expect(1, 195102, '\P{Script_Extensions:han}', "");
Expect(0, 195102, '\P{^Script_Extensions:han}', "");
Expect(1, 195101, '\p{Script_Extensions=Han}', "");
Expect(0, 195101, '\p{^Script_Extensions=Han}', "");
Expect(0, 195101, '\P{Script_Extensions=Han}', "");
Expect(1, 195101, '\P{^Script_Extensions=Han}', "");
Expect(0, 195102, '\p{Script_Extensions=Han}', "");
Expect(1, 195102, '\p{^Script_Extensions=Han}', "");
Expect(1, 195102, '\P{Script_Extensions=Han}', "");
Expect(0, 195102, '\P{^Script_Extensions=Han}', "");
Error('\p{Scx=:=hani}');
Error('\P{Scx=:=hani}');
Expect(1, 195101, '\p{Scx=hani}', "");
Expect(0, 195101, '\p{^Scx=hani}', "");
Expect(0, 195101, '\P{Scx=hani}', "");
Expect(1, 195101, '\P{^Scx=hani}', "");
Expect(0, 195102, '\p{Scx=hani}', "");
Expect(1, 195102, '\p{^Scx=hani}', "");
Expect(1, 195102, '\P{Scx=hani}', "");
Expect(0, 195102, '\P{^Scx=hani}', "");
Expect(1, 195101, '\p{Scx=__HANI}', "");
Expect(0, 195101, '\p{^Scx=__HANI}', "");
Expect(0, 195101, '\P{Scx=__HANI}', "");
Expect(1, 195101, '\P{^Scx=__HANI}', "");
Expect(0, 195102, '\p{Scx=__HANI}', "");
Expect(1, 195102, '\p{^Scx=__HANI}', "");
Expect(1, 195102, '\P{Scx=__HANI}', "");
Expect(0, 195102, '\P{^Scx=__HANI}', "");
Error('\p{Is_Script_Extensions=_-HAN/a/}');
Error('\P{Is_Script_Extensions=_-HAN/a/}');
Expect(1, 195101, '\p{Is_Script_Extensions=han}', "");
Expect(0, 195101, '\p{^Is_Script_Extensions=han}', "");
Expect(0, 195101, '\P{Is_Script_Extensions=han}', "");
Expect(1, 195101, '\P{^Is_Script_Extensions=han}', "");
Expect(0, 195102, '\p{Is_Script_Extensions=han}', "");
Expect(1, 195102, '\p{^Is_Script_Extensions=han}', "");
Expect(1, 195102, '\P{Is_Script_Extensions=han}', "");
Expect(0, 195102, '\P{^Is_Script_Extensions=han}', "");
Expect(1, 195101, '\p{Is_Script_Extensions=__Han}', "");
Expect(0, 195101, '\p{^Is_Script_Extensions=__Han}', "");
Expect(0, 195101, '\P{Is_Script_Extensions=__Han}', "");
Expect(1, 195101, '\P{^Is_Script_Extensions=__Han}', "");
Expect(0, 195102, '\p{Is_Script_Extensions=__Han}', "");
Expect(1, 195102, '\p{^Is_Script_Extensions=__Han}', "");
Expect(1, 195102, '\P{Is_Script_Extensions=__Han}', "");
Expect(0, 195102, '\P{^Is_Script_Extensions=__Han}', "");
Error('\p{Is_Scx:   	:=Hani}');
Error('\P{Is_Scx:   	:=Hani}');
Expect(1, 195101, '\p{Is_Scx=hani}', "");
Expect(0, 195101, '\p{^Is_Scx=hani}', "");
Expect(0, 195101, '\P{Is_Scx=hani}', "");
Expect(1, 195101, '\P{^Is_Scx=hani}', "");
Expect(0, 195102, '\p{Is_Scx=hani}', "");
Expect(1, 195102, '\p{^Is_Scx=hani}', "");
Expect(1, 195102, '\P{Is_Scx=hani}', "");
Expect(0, 195102, '\P{^Is_Scx=hani}', "");
Expect(1, 195101, '\p{Is_Scx=-Hani}', "");
Expect(0, 195101, '\p{^Is_Scx=-Hani}', "");
Expect(0, 195101, '\P{Is_Scx=-Hani}', "");
Expect(1, 195101, '\P{^Is_Scx=-Hani}', "");
Expect(0, 195102, '\p{Is_Scx=-Hani}', "");
Expect(1, 195102, '\p{^Is_Scx=-Hani}', "");
Expect(1, 195102, '\P{Is_Scx=-Hani}', "");
Expect(0, 195102, '\P{^Is_Scx=-Hani}', "");
Error('\p{Script_Extensions:	:=Takri}');
Error('\P{Script_Extensions:	:=Takri}');
Expect(1, 71369, '\p{Script_Extensions=takri}', "");
Expect(0, 71369, '\p{^Script_Extensions=takri}', "");
Expect(0, 71369, '\P{Script_Extensions=takri}', "");
Expect(1, 71369, '\P{^Script_Extensions=takri}', "");
Expect(0, 71370, '\p{Script_Extensions=takri}', "");
Expect(1, 71370, '\p{^Script_Extensions=takri}', "");
Expect(1, 71370, '\P{Script_Extensions=takri}', "");
Expect(0, 71370, '\P{^Script_Extensions=takri}', "");
Expect(1, 71369, '\p{Script_Extensions=	Takri}', "");
Expect(0, 71369, '\p{^Script_Extensions=	Takri}', "");
Expect(0, 71369, '\P{Script_Extensions=	Takri}', "");
Expect(1, 71369, '\P{^Script_Extensions=	Takri}', "");
Expect(0, 71370, '\p{Script_Extensions=	Takri}', "");
Expect(1, 71370, '\p{^Script_Extensions=	Takri}', "");
Expect(1, 71370, '\P{Script_Extensions=	Takri}', "");
Expect(0, 71370, '\P{^Script_Extensions=	Takri}', "");
Error('\p{Scx=	_Takr:=}');
Error('\P{Scx=	_Takr:=}');
Expect(1, 71369, '\p{Scx=takr}', "");
Expect(0, 71369, '\p{^Scx=takr}', "");
Expect(0, 71369, '\P{Scx=takr}', "");
Expect(1, 71369, '\P{^Scx=takr}', "");
Expect(0, 71370, '\p{Scx=takr}', "");
Expect(1, 71370, '\p{^Scx=takr}', "");
Expect(1, 71370, '\P{Scx=takr}', "");
Expect(0, 71370, '\P{^Scx=takr}', "");
Expect(1, 71369, '\p{Scx=	_TAKR}', "");
Expect(0, 71369, '\p{^Scx=	_TAKR}', "");
Expect(0, 71369, '\P{Scx=	_TAKR}', "");
Expect(1, 71369, '\P{^Scx=	_TAKR}', "");
Expect(0, 71370, '\p{Scx=	_TAKR}', "");
Expect(1, 71370, '\p{^Scx=	_TAKR}', "");
Expect(1, 71370, '\P{Scx=	_TAKR}', "");
Expect(0, 71370, '\P{^Scx=	_TAKR}', "");
Error('\p{Is_Script_Extensions=-/a/takri}');
Error('\P{Is_Script_Extensions=-/a/takri}');
Expect(1, 71369, '\p{Is_Script_Extensions=takri}', "");
Expect(0, 71369, '\p{^Is_Script_Extensions=takri}', "");
Expect(0, 71369, '\P{Is_Script_Extensions=takri}', "");
Expect(1, 71369, '\P{^Is_Script_Extensions=takri}', "");
Expect(0, 71370, '\p{Is_Script_Extensions=takri}', "");
Expect(1, 71370, '\p{^Is_Script_Extensions=takri}', "");
Expect(1, 71370, '\P{Is_Script_Extensions=takri}', "");
Expect(0, 71370, '\P{^Is_Script_Extensions=takri}', "");
Expect(1, 71369, '\p{Is_Script_Extensions=		TAKRI}', "");
Expect(0, 71369, '\p{^Is_Script_Extensions=		TAKRI}', "");
Expect(0, 71369, '\P{Is_Script_Extensions=		TAKRI}', "");
Expect(1, 71369, '\P{^Is_Script_Extensions=		TAKRI}', "");
Expect(0, 71370, '\p{Is_Script_Extensions=		TAKRI}', "");
Expect(1, 71370, '\p{^Is_Script_Extensions=		TAKRI}', "");
Expect(1, 71370, '\P{Is_Script_Extensions=		TAKRI}', "");
Expect(0, 71370, '\P{^Is_Script_Extensions=		TAKRI}', "");
Error('\p{Is_Scx=:= -TAKR}');
Error('\P{Is_Scx=:= -TAKR}');
Expect(1, 71369, '\p{Is_Scx:   takr}', "");
Expect(0, 71369, '\p{^Is_Scx:   takr}', "");
Expect(0, 71369, '\P{Is_Scx:   takr}', "");
Expect(1, 71369, '\P{^Is_Scx:   takr}', "");
Expect(0, 71370, '\p{Is_Scx:   takr}', "");
Expect(1, 71370, '\p{^Is_Scx:   takr}', "");
Expect(1, 71370, '\P{Is_Scx:   takr}', "");
Expect(0, 71370, '\P{^Is_Scx:   takr}', "");
Expect(1, 71369, '\p{Is_Scx:	_TAKR}', "");
Expect(0, 71369, '\p{^Is_Scx:	_TAKR}', "");
Expect(0, 71369, '\P{Is_Scx:	_TAKR}', "");
Expect(1, 71369, '\P{^Is_Scx:	_TAKR}', "");
Expect(0, 71370, '\p{Is_Scx:	_TAKR}', "");
Expect(1, 71370, '\p{^Is_Scx:	_TAKR}', "");
Expect(1, 71370, '\P{Is_Scx:	_TAKR}', "");
Expect(0, 71370, '\P{^Is_Scx:	_TAKR}', "");
Error('\p{Script_Extensions=	_greek/a/}');
Error('\P{Script_Extensions=	_greek/a/}');
Expect(1, 119365, '\p{Script_Extensions=greek}', "");
Expect(0, 119365, '\p{^Script_Extensions=greek}', "");
Expect(0, 119365, '\P{Script_Extensions=greek}', "");
Expect(1, 119365, '\P{^Script_Extensions=greek}', "");
Expect(0, 119366, '\p{Script_Extensions=greek}', "");
Expect(1, 119366, '\p{^Script_Extensions=greek}', "");
Expect(1, 119366, '\P{Script_Extensions=greek}', "");
Expect(0, 119366, '\P{^Script_Extensions=greek}', "");
Expect(1, 119365, '\p{Script_Extensions:   Greek}', "");
Expect(0, 119365, '\p{^Script_Extensions:   Greek}', "");
Expect(0, 119365, '\P{Script_Extensions:   Greek}', "");
Expect(1, 119365, '\P{^Script_Extensions:   Greek}', "");
Expect(0, 119366, '\p{Script_Extensions:   Greek}', "");
Expect(1, 119366, '\p{^Script_Extensions:   Greek}', "");
Expect(1, 119366, '\P{Script_Extensions:   Greek}', "");
Expect(0, 119366, '\P{^Script_Extensions:   Greek}', "");
Error('\p{Scx=:=_grek}');
Error('\P{Scx=:=_grek}');
Expect(1, 119365, '\p{Scx=grek}', "");
Expect(0, 119365, '\p{^Scx=grek}', "");
Expect(0, 119365, '\P{Scx=grek}', "");
Expect(1, 119365, '\P{^Scx=grek}', "");
Expect(0, 119366, '\p{Scx=grek}', "");
Expect(1, 119366, '\p{^Scx=grek}', "");
Expect(1, 119366, '\P{Scx=grek}', "");
Expect(0, 119366, '\P{^Scx=grek}', "");
Expect(1, 119365, '\p{Scx=__grek}', "");
Expect(0, 119365, '\p{^Scx=__grek}', "");
Expect(0, 119365, '\P{Scx=__grek}', "");
Expect(1, 119365, '\P{^Scx=__grek}', "");
Expect(0, 119366, '\p{Scx=__grek}', "");
Expect(1, 119366, '\p{^Scx=__grek}', "");
Expect(1, 119366, '\P{Scx=__grek}', "");
Expect(0, 119366, '\P{^Scx=__grek}', "");
Error('\p{Is_Script_Extensions=  Greek/a/}');
Error('\P{Is_Script_Extensions=  Greek/a/}');
Expect(1, 119365, '\p{Is_Script_Extensions=greek}', "");
Expect(0, 119365, '\p{^Is_Script_Extensions=greek}', "");
Expect(0, 119365, '\P{Is_Script_Extensions=greek}', "");
Expect(1, 119365, '\P{^Is_Script_Extensions=greek}', "");
Expect(0, 119366, '\p{Is_Script_Extensions=greek}', "");
Expect(1, 119366, '\p{^Is_Script_Extensions=greek}', "");
Expect(1, 119366, '\P{Is_Script_Extensions=greek}', "");
Expect(0, 119366, '\P{^Is_Script_Extensions=greek}', "");
Expect(1, 119365, '\p{Is_Script_Extensions= 	GREEK}', "");
Expect(0, 119365, '\p{^Is_Script_Extensions= 	GREEK}', "");
Expect(0, 119365, '\P{Is_Script_Extensions= 	GREEK}', "");
Expect(1, 119365, '\P{^Is_Script_Extensions= 	GREEK}', "");
Expect(0, 119366, '\p{Is_Script_Extensions= 	GREEK}', "");
Expect(1, 119366, '\p{^Is_Script_Extensions= 	GREEK}', "");
Expect(1, 119366, '\P{Is_Script_Extensions= 	GREEK}', "");
Expect(0, 119366, '\P{^Is_Script_Extensions= 	GREEK}', "");
Error('\p{Is_Scx: -/a/Grek}');
Error('\P{Is_Scx: -/a/Grek}');
Expect(1, 119365, '\p{Is_Scx=grek}', "");
Expect(0, 119365, '\p{^Is_Scx=grek}', "");
Expect(0, 119365, '\P{Is_Scx=grek}', "");
Expect(1, 119365, '\P{^Is_Scx=grek}', "");
Expect(0, 119366, '\p{Is_Scx=grek}', "");
Expect(1, 119366, '\p{^Is_Scx=grek}', "");
Expect(1, 119366, '\P{Is_Scx=grek}', "");
Expect(0, 119366, '\P{^Is_Scx=grek}', "");
Expect(1, 119365, '\p{Is_Scx=	-Grek}', "");
Expect(0, 119365, '\p{^Is_Scx=	-Grek}', "");
Expect(0, 119365, '\P{Is_Scx=	-Grek}', "");
Expect(1, 119365, '\P{^Is_Scx=	-Grek}', "");
Expect(0, 119366, '\p{Is_Scx=	-Grek}', "");
Expect(1, 119366, '\p{^Is_Scx=	-Grek}', "");
Expect(1, 119366, '\P{Is_Scx=	-Grek}', "");
Expect(0, 119366, '\P{^Is_Scx=	-Grek}', "");
Error('\p{Script_Extensions= :=Tamil}');
Error('\P{Script_Extensions= :=Tamil}');
Expect(1, 3066, '\p{Script_Extensions=tamil}', "");
Expect(0, 3066, '\p{^Script_Extensions=tamil}', "");
Expect(0, 3066, '\P{Script_Extensions=tamil}', "");
Expect(1, 3066, '\P{^Script_Extensions=tamil}', "");
Expect(0, 3067, '\p{Script_Extensions=tamil}', "");
Expect(1, 3067, '\p{^Script_Extensions=tamil}', "");
Expect(1, 3067, '\P{Script_Extensions=tamil}', "");
Expect(0, 3067, '\P{^Script_Extensions=tamil}', "");
Expect(1, 3066, '\p{Script_Extensions= Tamil}', "");
Expect(0, 3066, '\p{^Script_Extensions= Tamil}', "");
Expect(0, 3066, '\P{Script_Extensions= Tamil}', "");
Expect(1, 3066, '\P{^Script_Extensions= Tamil}', "");
Expect(0, 3067, '\p{Script_Extensions= Tamil}', "");
Expect(1, 3067, '\p{^Script_Extensions= Tamil}', "");
Expect(1, 3067, '\P{Script_Extensions= Tamil}', "");
Expect(0, 3067, '\P{^Script_Extensions= Tamil}', "");
Error('\p{Scx:		Taml:=}');
Error('\P{Scx:		Taml:=}');
Expect(1, 3066, '\p{Scx=taml}', "");
Expect(0, 3066, '\p{^Scx=taml}', "");
Expect(0, 3066, '\P{Scx=taml}', "");
Expect(1, 3066, '\P{^Scx=taml}', "");
Expect(0, 3067, '\p{Scx=taml}', "");
Expect(1, 3067, '\p{^Scx=taml}', "");
Expect(1, 3067, '\P{Scx=taml}', "");
Expect(0, 3067, '\P{^Scx=taml}', "");
Expect(1, 3066, '\p{Scx=		TAML}', "");
Expect(0, 3066, '\p{^Scx=		TAML}', "");
Expect(0, 3066, '\P{Scx=		TAML}', "");
Expect(1, 3066, '\P{^Scx=		TAML}', "");
Expect(0, 3067, '\p{Scx=		TAML}', "");
Expect(1, 3067, '\p{^Scx=		TAML}', "");
Expect(1, 3067, '\P{Scx=		TAML}', "");
Expect(0, 3067, '\P{^Scx=		TAML}', "");
Error('\p{Is_Script_Extensions=__Tamil:=}');
Error('\P{Is_Script_Extensions=__Tamil:=}');
Expect(1, 3066, '\p{Is_Script_Extensions=tamil}', "");
Expect(0, 3066, '\p{^Is_Script_Extensions=tamil}', "");
Expect(0, 3066, '\P{Is_Script_Extensions=tamil}', "");
Expect(1, 3066, '\P{^Is_Script_Extensions=tamil}', "");
Expect(0, 3067, '\p{Is_Script_Extensions=tamil}', "");
Expect(1, 3067, '\p{^Is_Script_Extensions=tamil}', "");
Expect(1, 3067, '\P{Is_Script_Extensions=tamil}', "");
Expect(0, 3067, '\P{^Is_Script_Extensions=tamil}', "");
Expect(1, 3066, '\p{Is_Script_Extensions= tamil}', "");
Expect(0, 3066, '\p{^Is_Script_Extensions= tamil}', "");
Expect(0, 3066, '\P{Is_Script_Extensions= tamil}', "");
Expect(1, 3066, '\P{^Is_Script_Extensions= tamil}', "");
Expect(0, 3067, '\p{Is_Script_Extensions= tamil}', "");
Expect(1, 3067, '\p{^Is_Script_Extensions= tamil}', "");
Expect(1, 3067, '\P{Is_Script_Extensions= tamil}', "");
Expect(0, 3067, '\P{^Is_Script_Extensions= tamil}', "");
Error('\p{Is_Scx=Taml:=}');
Error('\P{Is_Scx=Taml:=}');
Expect(1, 3066, '\p{Is_Scx=taml}', "");
Expect(0, 3066, '\p{^Is_Scx=taml}', "");
Expect(0, 3066, '\P{Is_Scx=taml}', "");
Expect(1, 3066, '\P{^Is_Scx=taml}', "");
Expect(0, 3067, '\p{Is_Scx=taml}', "");
Expect(1, 3067, '\p{^Is_Scx=taml}', "");
Expect(1, 3067, '\P{Is_Scx=taml}', "");
Expect(0, 3067, '\P{^Is_Scx=taml}', "");
Expect(1, 3066, '\p{Is_Scx=__TAML}', "");
Expect(0, 3066, '\p{^Is_Scx=__TAML}', "");
Expect(0, 3066, '\P{Is_Scx=__TAML}', "");
Expect(1, 3066, '\P{^Is_Scx=__TAML}', "");
Expect(0, 3067, '\p{Is_Scx=__TAML}', "");
Expect(1, 3067, '\p{^Is_Scx=__TAML}', "");
Expect(1, 3067, '\P{Is_Scx=__TAML}', "");
Expect(0, 3067, '\P{^Is_Scx=__TAML}', "");
Error('\p{Script_Extensions= _kayah_li/a/}');
Error('\P{Script_Extensions= _kayah_li/a/}');
Expect(1, 43311, '\p{Script_Extensions=kayahli}', "");
Expect(0, 43311, '\p{^Script_Extensions=kayahli}', "");
Expect(0, 43311, '\P{Script_Extensions=kayahli}', "");
Expect(1, 43311, '\P{^Script_Extensions=kayahli}', "");
Expect(0, 43312, '\p{Script_Extensions=kayahli}', "");
Expect(1, 43312, '\p{^Script_Extensions=kayahli}', "");
Expect(1, 43312, '\P{Script_Extensions=kayahli}', "");
Expect(0, 43312, '\P{^Script_Extensions=kayahli}', "");
Expect(1, 43311, '\p{Script_Extensions= -KAYAH_LI}', "");
Expect(0, 43311, '\p{^Script_Extensions= -KAYAH_LI}', "");
Expect(0, 43311, '\P{Script_Extensions= -KAYAH_LI}', "");
Expect(1, 43311, '\P{^Script_Extensions= -KAYAH_LI}', "");
Expect(0, 43312, '\p{Script_Extensions= -KAYAH_LI}', "");
Expect(1, 43312, '\p{^Script_Extensions= -KAYAH_LI}', "");
Expect(1, 43312, '\P{Script_Extensions= -KAYAH_LI}', "");
Expect(0, 43312, '\P{^Script_Extensions= -KAYAH_LI}', "");
Error('\p{Scx= _KALI/a/}');
Error('\P{Scx= _KALI/a/}');
Expect(1, 43311, '\p{Scx=kali}', "");
Expect(0, 43311, '\p{^Scx=kali}', "");
Expect(0, 43311, '\P{Scx=kali}', "");
Expect(1, 43311, '\P{^Scx=kali}', "");
Expect(0, 43312, '\p{Scx=kali}', "");
Expect(1, 43312, '\p{^Scx=kali}', "");
Expect(1, 43312, '\P{Scx=kali}', "");
Expect(0, 43312, '\P{^Scx=kali}', "");
Expect(1, 43311, '\p{Scx=	_Kali}', "");
Expect(0, 43311, '\p{^Scx=	_Kali}', "");
Expect(0, 43311, '\P{Scx=	_Kali}', "");
Expect(1, 43311, '\P{^Scx=	_Kali}', "");
Expect(0, 43312, '\p{Scx=	_Kali}', "");
Expect(1, 43312, '\p{^Scx=	_Kali}', "");
Expect(1, 43312, '\P{Scx=	_Kali}', "");
Expect(0, 43312, '\P{^Scx=	_Kali}', "");
Error('\p{Is_Script_Extensions=_ kayah_li:=}');
Error('\P{Is_Script_Extensions=_ kayah_li:=}');
Expect(1, 43311, '\p{Is_Script_Extensions=kayahli}', "");
Expect(0, 43311, '\p{^Is_Script_Extensions=kayahli}', "");
Expect(0, 43311, '\P{Is_Script_Extensions=kayahli}', "");
Expect(1, 43311, '\P{^Is_Script_Extensions=kayahli}', "");
Expect(0, 43312, '\p{Is_Script_Extensions=kayahli}', "");
Expect(1, 43312, '\p{^Is_Script_Extensions=kayahli}', "");
Expect(1, 43312, '\P{Is_Script_Extensions=kayahli}', "");
Expect(0, 43312, '\P{^Is_Script_Extensions=kayahli}', "");
Expect(1, 43311, '\p{Is_Script_Extensions=-Kayah_li}', "");
Expect(0, 43311, '\p{^Is_Script_Extensions=-Kayah_li}', "");
Expect(0, 43311, '\P{Is_Script_Extensions=-Kayah_li}', "");
Expect(1, 43311, '\P{^Is_Script_Extensions=-Kayah_li}', "");
Expect(0, 43312, '\p{Is_Script_Extensions=-Kayah_li}', "");
Expect(1, 43312, '\p{^Is_Script_Extensions=-Kayah_li}', "");
Expect(1, 43312, '\P{Is_Script_Extensions=-Kayah_li}', "");
Expect(0, 43312, '\P{^Is_Script_Extensions=-Kayah_li}', "");
Error('\p{Is_Scx= /a/KALI}');
Error('\P{Is_Scx= /a/KALI}');
Expect(1, 43311, '\p{Is_Scx=kali}', "");
Expect(0, 43311, '\p{^Is_Scx=kali}', "");
Expect(0, 43311, '\P{Is_Scx=kali}', "");
Expect(1, 43311, '\P{^Is_Scx=kali}', "");
Expect(0, 43312, '\p{Is_Scx=kali}', "");
Expect(1, 43312, '\p{^Is_Scx=kali}', "");
Expect(1, 43312, '\P{Is_Scx=kali}', "");
Expect(0, 43312, '\P{^Is_Scx=kali}', "");
Expect(1, 43311, '\p{Is_Scx=	Kali}', "");
Expect(0, 43311, '\p{^Is_Scx=	Kali}', "");
Expect(0, 43311, '\P{Is_Scx=	Kali}', "");
Expect(1, 43311, '\P{^Is_Scx=	Kali}', "");
Expect(0, 43312, '\p{Is_Scx=	Kali}', "");
Expect(1, 43312, '\p{^Is_Scx=	Kali}', "");
Expect(1, 43312, '\P{Is_Scx=	Kali}', "");
Expect(0, 43312, '\P{^Is_Scx=	Kali}', "");
Error('\p{Script_Extensions=	/a/Thai}');
Error('\P{Script_Extensions=	/a/Thai}');
Expect(1, 3675, '\p{Script_Extensions=thai}', "");
Expect(0, 3675, '\p{^Script_Extensions=thai}', "");
Expect(0, 3675, '\P{Script_Extensions=thai}', "");
Expect(1, 3675, '\P{^Script_Extensions=thai}', "");
Expect(0, 3676, '\p{Script_Extensions=thai}', "");
Expect(1, 3676, '\p{^Script_Extensions=thai}', "");
Expect(1, 3676, '\P{Script_Extensions=thai}', "");
Expect(0, 3676, '\P{^Script_Extensions=thai}', "");
Expect(1, 3675, '\p{Script_Extensions= 	Thai}', "");
Expect(0, 3675, '\p{^Script_Extensions= 	Thai}', "");
Expect(0, 3675, '\P{Script_Extensions= 	Thai}', "");
Expect(1, 3675, '\P{^Script_Extensions= 	Thai}', "");
Expect(0, 3676, '\p{Script_Extensions= 	Thai}', "");
Expect(1, 3676, '\p{^Script_Extensions= 	Thai}', "");
Expect(1, 3676, '\P{Script_Extensions= 	Thai}', "");
Expect(0, 3676, '\P{^Script_Extensions= 	Thai}', "");
Error('\p{Scx=-/a/thai}');
Error('\P{Scx=-/a/thai}');
Expect(1, 3675, '\p{Scx=thai}', "");
Expect(0, 3675, '\p{^Scx=thai}', "");
Expect(0, 3675, '\P{Scx=thai}', "");
Expect(1, 3675, '\P{^Scx=thai}', "");
Expect(0, 3676, '\p{Scx=thai}', "");
Expect(1, 3676, '\p{^Scx=thai}', "");
Expect(1, 3676, '\P{Scx=thai}', "");
Expect(0, 3676, '\P{^Scx=thai}', "");
Expect(1, 3675, '\p{Scx=_Thai}', "");
Expect(0, 3675, '\p{^Scx=_Thai}', "");
Expect(0, 3675, '\P{Scx=_Thai}', "");
Expect(1, 3675, '\P{^Scx=_Thai}', "");
Expect(0, 3676, '\p{Scx=_Thai}', "");
Expect(1, 3676, '\p{^Scx=_Thai}', "");
Expect(1, 3676, '\P{Scx=_Thai}', "");
Expect(0, 3676, '\P{^Scx=_Thai}', "");
Error('\p{Is_Script_Extensions=:=-	Thai}');
Error('\P{Is_Script_Extensions=:=-	Thai}');
Expect(1, 3675, '\p{Is_Script_Extensions=thai}', "");
Expect(0, 3675, '\p{^Is_Script_Extensions=thai}', "");
Expect(0, 3675, '\P{Is_Script_Extensions=thai}', "");
Expect(1, 3675, '\P{^Is_Script_Extensions=thai}', "");
Expect(0, 3676, '\p{Is_Script_Extensions=thai}', "");
Expect(1, 3676, '\p{^Is_Script_Extensions=thai}', "");
Expect(1, 3676, '\P{Is_Script_Extensions=thai}', "");
Expect(0, 3676, '\P{^Is_Script_Extensions=thai}', "");
Expect(1, 3675, '\p{Is_Script_Extensions= 	THAI}', "");
Expect(0, 3675, '\p{^Is_Script_Extensions= 	THAI}', "");
Expect(0, 3675, '\P{Is_Script_Extensions= 	THAI}', "");
Expect(1, 3675, '\P{^Is_Script_Extensions= 	THAI}', "");
Expect(0, 3676, '\p{Is_Script_Extensions= 	THAI}', "");
Expect(1, 3676, '\p{^Is_Script_Extensions= 	THAI}', "");
Expect(1, 3676, '\P{Is_Script_Extensions= 	THAI}', "");
Expect(0, 3676, '\P{^Is_Script_Extensions= 	THAI}', "");
Error('\p{Is_Scx=_:=thai}');
Error('\P{Is_Scx=_:=thai}');
Expect(1, 3675, '\p{Is_Scx=thai}', "");
Expect(0, 3675, '\p{^Is_Scx=thai}', "");
Expect(0, 3675, '\P{Is_Scx=thai}', "");
Expect(1, 3675, '\P{^Is_Scx=thai}', "");
Expect(0, 3676, '\p{Is_Scx=thai}', "");
Expect(1, 3676, '\p{^Is_Scx=thai}', "");
Expect(1, 3676, '\P{Is_Scx=thai}', "");
Expect(0, 3676, '\P{^Is_Scx=thai}', "");
Expect(1, 3675, '\p{Is_Scx:_	THAI}', "");
Expect(0, 3675, '\p{^Is_Scx:_	THAI}', "");
Expect(0, 3675, '\P{Is_Scx:_	THAI}', "");
Expect(1, 3675, '\P{^Is_Scx:_	THAI}', "");
Expect(0, 3676, '\p{Is_Scx:_	THAI}', "");
Expect(1, 3676, '\p{^Is_Scx:_	THAI}', "");
Expect(1, 3676, '\P{Is_Scx:_	THAI}', "");
Expect(0, 3676, '\P{^Is_Scx:_	THAI}', "");
Error('\p{NFKC_Quick_Check:		:=Yes}');
Error('\P{NFKC_Quick_Check:		:=Yes}');
Expect(1, 195102, '\p{NFKC_Quick_Check:   yes}', "");
Expect(0, 195102, '\p{^NFKC_Quick_Check:   yes}', "");
Expect(0, 195102, '\P{NFKC_Quick_Check:   yes}', "");
Expect(1, 195102, '\P{^NFKC_Quick_Check:   yes}', "");
Expect(0, 195101, '\p{NFKC_Quick_Check:   yes}', "");
Expect(1, 195101, '\p{^NFKC_Quick_Check:   yes}', "");
Expect(1, 195101, '\P{NFKC_Quick_Check:   yes}', "");
Expect(0, 195101, '\P{^NFKC_Quick_Check:   yes}', "");
Expect(1, 195102, '\p{NFKC_Quick_Check:    -Yes}', "");
Expect(0, 195102, '\p{^NFKC_Quick_Check:    -Yes}', "");
Expect(0, 195102, '\P{NFKC_Quick_Check:    -Yes}', "");
Expect(1, 195102, '\P{^NFKC_Quick_Check:    -Yes}', "");
Expect(0, 195101, '\p{NFKC_Quick_Check:    -Yes}', "");
Expect(1, 195101, '\p{^NFKC_Quick_Check:    -Yes}', "");
Expect(1, 195101, '\P{NFKC_Quick_Check:    -Yes}', "");
Expect(0, 195101, '\P{^NFKC_Quick_Check:    -Yes}', "");
Error('\p{NFKC_QC: :=	 Y}');
Error('\P{NFKC_QC: :=	 Y}');
Expect(1, 195102, '\p{NFKC_QC: y}', "");
Expect(0, 195102, '\p{^NFKC_QC: y}', "");
Expect(0, 195102, '\P{NFKC_QC: y}', "");
Expect(1, 195102, '\P{^NFKC_QC: y}', "");
Expect(0, 195101, '\p{NFKC_QC: y}', "");
Expect(1, 195101, '\p{^NFKC_QC: y}', "");
Expect(1, 195101, '\P{NFKC_QC: y}', "");
Expect(0, 195101, '\P{^NFKC_QC: y}', "");
Expect(1, 195102, '\p{NFKC_QC=_	Y}', "");
Expect(0, 195102, '\p{^NFKC_QC=_	Y}', "");
Expect(0, 195102, '\P{NFKC_QC=_	Y}', "");
Expect(1, 195102, '\P{^NFKC_QC=_	Y}', "");
Expect(0, 195101, '\p{NFKC_QC=_	Y}', "");
Expect(1, 195101, '\p{^NFKC_QC=_	Y}', "");
Expect(1, 195101, '\P{NFKC_QC=_	Y}', "");
Expect(0, 195101, '\P{^NFKC_QC=_	Y}', "");
Error('\p{Is_NFKC_Quick_Check=:=--yes}');
Error('\P{Is_NFKC_Quick_Check=:=--yes}');
Expect(1, 195102, '\p{Is_NFKC_Quick_Check=yes}', "");
Expect(0, 195102, '\p{^Is_NFKC_Quick_Check=yes}', "");
Expect(0, 195102, '\P{Is_NFKC_Quick_Check=yes}', "");
Expect(1, 195102, '\P{^Is_NFKC_Quick_Check=yes}', "");
Expect(0, 195101, '\p{Is_NFKC_Quick_Check=yes}', "");
Expect(1, 195101, '\p{^Is_NFKC_Quick_Check=yes}', "");
Expect(1, 195101, '\P{Is_NFKC_Quick_Check=yes}', "");
Expect(0, 195101, '\P{^Is_NFKC_Quick_Check=yes}', "");
Expect(1, 195102, '\p{Is_NFKC_Quick_Check:   	Yes}', "");
Expect(0, 195102, '\p{^Is_NFKC_Quick_Check:   	Yes}', "");
Expect(0, 195102, '\P{Is_NFKC_Quick_Check:   	Yes}', "");
Expect(1, 195102, '\P{^Is_NFKC_Quick_Check:   	Yes}', "");
Expect(0, 195101, '\p{Is_NFKC_Quick_Check:   	Yes}', "");
Expect(1, 195101, '\p{^Is_NFKC_Quick_Check:   	Yes}', "");
Expect(1, 195101, '\P{Is_NFKC_Quick_Check:   	Yes}', "");
Expect(0, 195101, '\P{^Is_NFKC_Quick_Check:   	Yes}', "");
Error('\p{Is_NFKC_QC=	_Y:=}');
Error('\P{Is_NFKC_QC=	_Y:=}');
Expect(1, 195102, '\p{Is_NFKC_QC=y}', "");
Expect(0, 195102, '\p{^Is_NFKC_QC=y}', "");
Expect(0, 195102, '\P{Is_NFKC_QC=y}', "");
Expect(1, 195102, '\P{^Is_NFKC_QC=y}', "");
Expect(0, 195101, '\p{Is_NFKC_QC=y}', "");
Expect(1, 195101, '\p{^Is_NFKC_QC=y}', "");
Expect(1, 195101, '\P{Is_NFKC_QC=y}', "");
Expect(0, 195101, '\P{^Is_NFKC_QC=y}', "");
Expect(1, 195102, '\p{Is_NFKC_QC=		y}', "");
Expect(0, 195102, '\p{^Is_NFKC_QC=		y}', "");
Expect(0, 195102, '\P{Is_NFKC_QC=		y}', "");
Expect(1, 195102, '\P{^Is_NFKC_QC=		y}', "");
Expect(0, 195101, '\p{Is_NFKC_QC=		y}', "");
Expect(1, 195101, '\p{^Is_NFKC_QC=		y}', "");
Expect(1, 195101, '\P{Is_NFKC_QC=		y}', "");
Expect(0, 195101, '\P{^Is_NFKC_QC=		y}', "");
Error('\p{NFKC_Quick_Check=	:=No}');
Error('\P{NFKC_Quick_Check=	:=No}');
Expect(1, 195101, '\p{NFKC_Quick_Check=no}', "");
Expect(0, 195101, '\p{^NFKC_Quick_Check=no}', "");
Expect(0, 195101, '\P{NFKC_Quick_Check=no}', "");
Expect(1, 195101, '\P{^NFKC_Quick_Check=no}', "");
Expect(0, 195102, '\p{NFKC_Quick_Check=no}', "");
Expect(1, 195102, '\p{^NFKC_Quick_Check=no}', "");
Expect(1, 195102, '\P{NFKC_Quick_Check=no}', "");
Expect(0, 195102, '\P{^NFKC_Quick_Check=no}', "");
Expect(1, 195101, '\p{NFKC_Quick_Check=-no}', "");
Expect(0, 195101, '\p{^NFKC_Quick_Check=-no}', "");
Expect(0, 195101, '\P{NFKC_Quick_Check=-no}', "");
Expect(1, 195101, '\P{^NFKC_Quick_Check=-no}', "");
Expect(0, 195102, '\p{NFKC_Quick_Check=-no}', "");
Expect(1, 195102, '\p{^NFKC_Quick_Check=-no}', "");
Expect(1, 195102, '\P{NFKC_Quick_Check=-no}', "");
Expect(0, 195102, '\P{^NFKC_Quick_Check=-no}', "");
Error('\p{NFKC_QC=	:=N}');
Error('\P{NFKC_QC=	:=N}');
Expect(1, 195101, '\p{NFKC_QC:n}', "");
Expect(0, 195101, '\p{^NFKC_QC:n}', "");
Expect(0, 195101, '\P{NFKC_QC:n}', "");
Expect(1, 195101, '\P{^NFKC_QC:n}', "");
Expect(0, 195102, '\p{NFKC_QC:n}', "");
Expect(1, 195102, '\p{^NFKC_QC:n}', "");
Expect(1, 195102, '\P{NFKC_QC:n}', "");
Expect(0, 195102, '\P{^NFKC_QC:n}', "");
Expect(1, 195101, '\p{NFKC_QC= _N}', "");
Expect(0, 195101, '\p{^NFKC_QC= _N}', "");
Expect(0, 195101, '\P{NFKC_QC= _N}', "");
Expect(1, 195101, '\P{^NFKC_QC= _N}', "");
Expect(0, 195102, '\p{NFKC_QC= _N}', "");
Expect(1, 195102, '\p{^NFKC_QC= _N}', "");
Expect(1, 195102, '\P{NFKC_QC= _N}', "");
Expect(0, 195102, '\P{^NFKC_QC= _N}', "");
Error('\p{Is_NFKC_Quick_Check:  /a/no}');
Error('\P{Is_NFKC_Quick_Check:  /a/no}');
Expect(1, 195101, '\p{Is_NFKC_Quick_Check=no}', "");
Expect(0, 195101, '\p{^Is_NFKC_Quick_Check=no}', "");
Expect(0, 195101, '\P{Is_NFKC_Quick_Check=no}', "");
Expect(1, 195101, '\P{^Is_NFKC_Quick_Check=no}', "");
Expect(0, 195102, '\p{Is_NFKC_Quick_Check=no}', "");
Expect(1, 195102, '\p{^Is_NFKC_Quick_Check=no}', "");
Expect(1, 195102, '\P{Is_NFKC_Quick_Check=no}', "");
Expect(0, 195102, '\P{^Is_NFKC_Quick_Check=no}', "");
Expect(1, 195101, '\p{Is_NFKC_Quick_Check=-	no}', "");
Expect(0, 195101, '\p{^Is_NFKC_Quick_Check=-	no}', "");
Expect(0, 195101, '\P{Is_NFKC_Quick_Check=-	no}', "");
Expect(1, 195101, '\P{^Is_NFKC_Quick_Check=-	no}', "");
Expect(0, 195102, '\p{Is_NFKC_Quick_Check=-	no}', "");
Expect(1, 195102, '\p{^Is_NFKC_Quick_Check=-	no}', "");
Expect(1, 195102, '\P{Is_NFKC_Quick_Check=-	no}', "");
Expect(0, 195102, '\P{^Is_NFKC_Quick_Check=-	no}', "");
Error('\p{Is_NFKC_QC=:=__N}');
Error('\P{Is_NFKC_QC=:=__N}');
Expect(1, 195101, '\p{Is_NFKC_QC=n}', "");
Expect(0, 195101, '\p{^Is_NFKC_QC=n}', "");
Expect(0, 195101, '\P{Is_NFKC_QC=n}', "");
Expect(1, 195101, '\P{^Is_NFKC_QC=n}', "");
Expect(0, 195102, '\p{Is_NFKC_QC=n}', "");
Expect(1, 195102, '\p{^Is_NFKC_QC=n}', "");
Expect(1, 195102, '\P{Is_NFKC_QC=n}', "");
Expect(0, 195102, '\P{^Is_NFKC_QC=n}', "");
Expect(1, 195101, '\p{Is_NFKC_QC=-	N}', "");
Expect(0, 195101, '\p{^Is_NFKC_QC=-	N}', "");
Expect(0, 195101, '\P{Is_NFKC_QC=-	N}', "");
Expect(1, 195101, '\P{^Is_NFKC_QC=-	N}', "");
Expect(0, 195102, '\p{Is_NFKC_QC=-	N}', "");
Expect(1, 195102, '\p{^Is_NFKC_QC=-	N}', "");
Expect(1, 195102, '\P{Is_NFKC_QC=-	N}', "");
Expect(0, 195102, '\P{^Is_NFKC_QC=-	N}', "");
Error('\p{NFKC_Quick_Check= _MAYBE/a/}');
Error('\P{NFKC_Quick_Check= _MAYBE/a/}');
Expect(1, 69927, '\p{NFKC_Quick_Check:	maybe}', "");
Expect(0, 69927, '\p{^NFKC_Quick_Check:	maybe}', "");
Expect(0, 69927, '\P{NFKC_Quick_Check:	maybe}', "");
Expect(1, 69927, '\P{^NFKC_Quick_Check:	maybe}', "");
Expect(0, 69928, '\p{NFKC_Quick_Check:	maybe}', "");
Expect(1, 69928, '\p{^NFKC_Quick_Check:	maybe}', "");
Expect(1, 69928, '\P{NFKC_Quick_Check:	maybe}', "");
Expect(0, 69928, '\P{^NFKC_Quick_Check:	maybe}', "");
Expect(1, 69927, '\p{NFKC_Quick_Check=	-maybe}', "");
Expect(0, 69927, '\p{^NFKC_Quick_Check=	-maybe}', "");
Expect(0, 69927, '\P{NFKC_Quick_Check=	-maybe}', "");
Expect(1, 69927, '\P{^NFKC_Quick_Check=	-maybe}', "");
Expect(0, 69928, '\p{NFKC_Quick_Check=	-maybe}', "");
Expect(1, 69928, '\p{^NFKC_Quick_Check=	-maybe}', "");
Expect(1, 69928, '\P{NFKC_Quick_Check=	-maybe}', "");
Expect(0, 69928, '\P{^NFKC_Quick_Check=	-maybe}', "");
Error('\p{NFKC_QC:  _M/a/}');
Error('\P{NFKC_QC:  _M/a/}');
Expect(1, 69927, '\p{NFKC_QC=m}', "");
Expect(0, 69927, '\p{^NFKC_QC=m}', "");
Expect(0, 69927, '\P{NFKC_QC=m}', "");
Expect(1, 69927, '\P{^NFKC_QC=m}', "");
Expect(0, 69928, '\p{NFKC_QC=m}', "");
Expect(1, 69928, '\p{^NFKC_QC=m}', "");
Expect(1, 69928, '\P{NFKC_QC=m}', "");
Expect(0, 69928, '\P{^NFKC_QC=m}', "");
Expect(1, 69927, '\p{NFKC_QC=- M}', "");
Expect(0, 69927, '\p{^NFKC_QC=- M}', "");
Expect(0, 69927, '\P{NFKC_QC=- M}', "");
Expect(1, 69927, '\P{^NFKC_QC=- M}', "");
Expect(0, 69928, '\p{NFKC_QC=- M}', "");
Expect(1, 69928, '\p{^NFKC_QC=- M}', "");
Expect(1, 69928, '\P{NFKC_QC=- M}', "");
Expect(0, 69928, '\P{^NFKC_QC=- M}', "");
Error('\p{Is_NFKC_Quick_Check=--Maybe/a/}');
Error('\P{Is_NFKC_Quick_Check=--Maybe/a/}');
Expect(1, 69927, '\p{Is_NFKC_Quick_Check=maybe}', "");
Expect(0, 69927, '\p{^Is_NFKC_Quick_Check=maybe}', "");
Expect(0, 69927, '\P{Is_NFKC_Quick_Check=maybe}', "");
Expect(1, 69927, '\P{^Is_NFKC_Quick_Check=maybe}', "");
Expect(0, 69928, '\p{Is_NFKC_Quick_Check=maybe}', "");
Expect(1, 69928, '\p{^Is_NFKC_Quick_Check=maybe}', "");
Expect(1, 69928, '\P{Is_NFKC_Quick_Check=maybe}', "");
Expect(0, 69928, '\P{^Is_NFKC_Quick_Check=maybe}', "");
Expect(1, 69927, '\p{Is_NFKC_Quick_Check:   	MAYBE}', "");
Expect(0, 69927, '\p{^Is_NFKC_Quick_Check:   	MAYBE}', "");
Expect(0, 69927, '\P{Is_NFKC_Quick_Check:   	MAYBE}', "");
Expect(1, 69927, '\P{^Is_NFKC_Quick_Check:   	MAYBE}', "");
Expect(0, 69928, '\p{Is_NFKC_Quick_Check:   	MAYBE}', "");
Expect(1, 69928, '\p{^Is_NFKC_Quick_Check:   	MAYBE}', "");
Expect(1, 69928, '\P{Is_NFKC_Quick_Check:   	MAYBE}', "");
Expect(0, 69928, '\P{^Is_NFKC_Quick_Check:   	MAYBE}', "");
Error('\p{Is_NFKC_QC=-:=m}');
Error('\P{Is_NFKC_QC=-:=m}');
Expect(1, 69927, '\p{Is_NFKC_QC=m}', "");
Expect(0, 69927, '\p{^Is_NFKC_QC=m}', "");
Expect(0, 69927, '\P{Is_NFKC_QC=m}', "");
Expect(1, 69927, '\P{^Is_NFKC_QC=m}', "");
Expect(0, 69928, '\p{Is_NFKC_QC=m}', "");
Expect(1, 69928, '\p{^Is_NFKC_QC=m}', "");
Expect(1, 69928, '\P{Is_NFKC_QC=m}', "");
Expect(0, 69928, '\P{^Is_NFKC_QC=m}', "");
Expect(1, 69927, '\p{Is_NFKC_QC=- m}', "");
Expect(0, 69927, '\p{^Is_NFKC_QC=- m}', "");
Expect(0, 69927, '\P{Is_NFKC_QC=- m}', "");
Expect(1, 69927, '\P{^Is_NFKC_QC=- m}', "");
Expect(0, 69928, '\p{Is_NFKC_QC=- m}', "");
Expect(1, 69928, '\p{^Is_NFKC_QC=- m}', "");
Expect(1, 69928, '\P{Is_NFKC_QC=- m}', "");
Expect(0, 69928, '\P{^Is_NFKC_QC=- m}', "");
Error('\p{Other_Alphabetic=Yes}');
Error('\P{Other_Alphabetic=Yes}');
Error('\p{OAlpha=Y}');
Error('\P{OAlpha=Y}');
Error('\p{Is_Other_Alphabetic=T}');
Error('\P{Is_Other_Alphabetic=T}');
Error('\p{Is_OAlpha=True}');
Error('\P{Is_OAlpha=True}');
Error('\p{Other_Alphabetic:   No}');
Error('\P{Other_Alphabetic:   No}');
Error('\p{OAlpha=N}');
Error('\P{OAlpha=N}');
Error('\p{Is_Other_Alphabetic=F}');
Error('\P{Is_Other_Alphabetic=F}');
Error('\p{Is_OAlpha:False}');
Error('\P{Is_OAlpha:False}');
Error('\p{Indic_Matra_Category=Left}');
Error('\P{Indic_Matra_Category=Left}');
Error('\p{InMC=Left}');
Error('\P{InMC=Left}');
Error('\p{Is_Indic_Matra_Category:   Left}');
Error('\P{Is_Indic_Matra_Category:   Left}');
Error('\p{Is_InMC=Left}');
Error('\P{Is_InMC=Left}');
Error('\p{Indic_Matra_Category:Right}');
Error('\P{Indic_Matra_Category:Right}');
Error('\p{InMC=Right}');
Error('\P{InMC=Right}');
Error('\p{Is_Indic_Matra_Category=Right}');
Error('\P{Is_Indic_Matra_Category=Right}');
Error('\p{Is_InMC=Right}');
Error('\P{Is_InMC=Right}');
Error('\p{Indic_Matra_Category: Top_And_Left}');
Error('\P{Indic_Matra_Category: Top_And_Left}');
Error('\p{InMC=Top_And_Left}');
Error('\P{InMC=Top_And_Left}');
Error('\p{Is_Indic_Matra_Category=Top_And_Left}');
Error('\P{Is_Indic_Matra_Category=Top_And_Left}');
Error('\p{Is_InMC=Top_And_Left}');
Error('\P{Is_InMC=Top_And_Left}');
Error('\p{Indic_Matra_Category=Top_And_Bottom_And_Right}');
Error('\P{Indic_Matra_Category=Top_And_Bottom_And_Right}');
Error('\p{InMC=Top_And_Bottom_And_Right}');
Error('\P{InMC=Top_And_Bottom_And_Right}');
Error('\p{Is_Indic_Matra_Category:Top_And_Bottom_And_Right}');
Error('\P{Is_Indic_Matra_Category:Top_And_Bottom_And_Right}');
Error('\p{Is_InMC:   Top_And_Bottom_And_Right}');
Error('\P{Is_InMC:   Top_And_Bottom_And_Right}');
Error('\p{Indic_Matra_Category=Overstruck}');
Error('\P{Indic_Matra_Category=Overstruck}');
Error('\p{InMC=Overstruck}');
Error('\P{InMC=Overstruck}');
Error('\p{Is_Indic_Matra_Category=Overstruck}');
Error('\P{Is_Indic_Matra_Category=Overstruck}');
Error('\p{Is_InMC=Overstruck}');
Error('\P{Is_InMC=Overstruck}');
Error('\p{Indic_Matra_Category=Top_And_Bottom}');
Error('\P{Indic_Matra_Category=Top_And_Bottom}');
Error('\p{InMC=Top_And_Bottom}');
Error('\P{InMC=Top_And_Bottom}');
Error('\p{Is_Indic_Matra_Category=Top_And_Bottom}');
Error('\P{Is_Indic_Matra_Category=Top_And_Bottom}');
Error('\p{Is_InMC=Top_And_Bottom}');
Error('\P{Is_InMC=Top_And_Bottom}');
Error('\p{Indic_Matra_Category=Bottom_And_Right}');
Error('\P{Indic_Matra_Category=Bottom_And_Right}');
Error('\p{InMC=Bottom_And_Right}');
Error('\P{InMC=Bottom_And_Right}');
Error('\p{Is_Indic_Matra_Category=Bottom_And_Right}');
Error('\P{Is_Indic_Matra_Category=Bottom_And_Right}');
Error('\p{Is_InMC=Bottom_And_Right}');
Error('\P{Is_InMC=Bottom_And_Right}');
Error('\p{Indic_Matra_Category:   Visual_Order_Left}');
Error('\P{Indic_Matra_Category:   Visual_Order_Left}');
Error('\p{InMC=Visual_Order_Left}');
Error('\P{InMC=Visual_Order_Left}');
Error('\p{Is_Indic_Matra_Category=Visual_Order_Left}');
Error('\P{Is_Indic_Matra_Category=Visual_Order_Left}');
Error('\p{Is_InMC:	Visual_Order_Left}');
Error('\P{Is_InMC:	Visual_Order_Left}');
Error('\p{Indic_Matra_Category=Top_And_Right}');
Error('\P{Indic_Matra_Category=Top_And_Right}');
Error('\p{InMC=Top_And_Right}');
Error('\P{InMC=Top_And_Right}');
Error('\p{Is_Indic_Matra_Category=Top_And_Right}');
Error('\P{Is_Indic_Matra_Category=Top_And_Right}');
Error('\p{Is_InMC=Top_And_Right}');
Error('\P{Is_InMC=Top_And_Right}');
Error('\p{Indic_Matra_Category=Top}');
Error('\P{Indic_Matra_Category=Top}');
Error('\p{InMC=Top}');
Error('\P{InMC=Top}');
Error('\p{Is_Indic_Matra_Category=Top}');
Error('\P{Is_Indic_Matra_Category=Top}');
Error('\p{Is_InMC=Top}');
Error('\P{Is_InMC=Top}');
Error('\p{Indic_Matra_Category=NA}');
Error('\P{Indic_Matra_Category=NA}');
Error('\p{InMC=NA}');
Error('\P{InMC=NA}');
Error('\p{Is_Indic_Matra_Category=NA}');
Error('\P{Is_Indic_Matra_Category=NA}');
Error('\p{Is_InMC=NA}');
Error('\P{Is_InMC=NA}');
Error('\p{Indic_Matra_Category=Bottom}');
Error('\P{Indic_Matra_Category=Bottom}');
Error('\p{InMC=Bottom}');
Error('\P{InMC=Bottom}');
Error('\p{Is_Indic_Matra_Category=Bottom}');
Error('\P{Is_Indic_Matra_Category=Bottom}');
Error('\p{Is_InMC=Bottom}');
Error('\P{Is_InMC=Bottom}');
Error('\p{Indic_Matra_Category=Invisible}');
Error('\P{Indic_Matra_Category=Invisible}');
Error('\p{InMC=Invisible}');
Error('\P{InMC=Invisible}');
Error('\p{Is_Indic_Matra_Category=Invisible}');
Error('\P{Is_Indic_Matra_Category=Invisible}');
Error('\p{Is_InMC=Invisible}');
Error('\P{Is_InMC=Invisible}');
Error('\p{Indic_Matra_Category=Left_And_Right}');
Error('\P{Indic_Matra_Category=Left_And_Right}');
Error('\p{InMC=Left_And_Right}');
Error('\P{InMC=Left_And_Right}');
Error('\p{Is_Indic_Matra_Category=Left_And_Right}');
Error('\P{Is_Indic_Matra_Category=Left_And_Right}');
Error('\p{Is_InMC=Left_And_Right}');
Error('\P{Is_InMC=Left_And_Right}');
Error('\p{Indic_Matra_Category=Top_And_Left_And_Right}');
Error('\P{Indic_Matra_Category=Top_And_Left_And_Right}');
Error('\p{InMC:Top_And_Left_And_Right}');
Error('\P{InMC:Top_And_Left_And_Right}');
Error('\p{Is_Indic_Matra_Category=Top_And_Left_And_Right}');
Error('\P{Is_Indic_Matra_Category=Top_And_Left_And_Right}');
Error('\p{Is_InMC=Top_And_Left_And_Right}');
Error('\P{Is_InMC=Top_And_Left_And_Right}');
Error('\p{General_Category=/a/_-Symbol}');
Error('\P{General_Category=/a/_-Symbol}');
Expect(1, 128883, '\p{General_Category=symbol}', "");
Expect(0, 128883, '\p{^General_Category=symbol}', "");
Expect(0, 128883, '\P{General_Category=symbol}', "");
Expect(1, 128883, '\P{^General_Category=symbol}', "");
Expect(0, 128884, '\p{General_Category=symbol}', "");
Expect(1, 128884, '\p{^General_Category=symbol}', "");
Expect(1, 128884, '\P{General_Category=symbol}', "");
Expect(0, 128884, '\P{^General_Category=symbol}', "");
Expect(1, 128883, '\p{General_Category= Symbol}', "");
Expect(0, 128883, '\p{^General_Category= Symbol}', "");
Expect(0, 128883, '\P{General_Category= Symbol}', "");
Expect(1, 128883, '\P{^General_Category= Symbol}', "");
Expect(0, 128884, '\p{General_Category= Symbol}', "");
Expect(1, 128884, '\p{^General_Category= Symbol}', "");
Expect(1, 128884, '\P{General_Category= Symbol}', "");
Expect(0, 128884, '\P{^General_Category= Symbol}', "");
Error('\p{Gc:    S:=}');
Error('\P{Gc:    S:=}');
Expect(1, 128883, '\p{Gc=s}', "");
Expect(0, 128883, '\p{^Gc=s}', "");
Expect(0, 128883, '\P{Gc=s}', "");
Expect(1, 128883, '\P{^Gc=s}', "");
Expect(0, 128884, '\p{Gc=s}', "");
Expect(1, 128884, '\p{^Gc=s}', "");
Expect(1, 128884, '\P{Gc=s}', "");
Expect(0, 128884, '\P{^Gc=s}', "");
Expect(1, 128883, '\p{Gc= -s}', "");
Expect(0, 128883, '\p{^Gc= -s}', "");
Expect(0, 128883, '\P{Gc= -s}', "");
Expect(1, 128883, '\P{^Gc= -s}', "");
Expect(0, 128884, '\p{Gc= -s}', "");
Expect(1, 128884, '\p{^Gc= -s}', "");
Expect(1, 128884, '\P{Gc= -s}', "");
Expect(0, 128884, '\P{^Gc= -s}', "");
Error('\p{Category=_	symbol:=}');
Error('\P{Category=_	symbol:=}');
Expect(1, 128883, '\p{Category:symbol}', "");
Expect(0, 128883, '\p{^Category:symbol}', "");
Expect(0, 128883, '\P{Category:symbol}', "");
Expect(1, 128883, '\P{^Category:symbol}', "");
Expect(0, 128884, '\p{Category:symbol}', "");
Expect(1, 128884, '\p{^Category:symbol}', "");
Expect(1, 128884, '\P{Category:symbol}', "");
Expect(0, 128884, '\P{^Category:symbol}', "");
Expect(1, 128883, '\p{Category=_-symbol}', "");
Expect(0, 128883, '\p{^Category=_-symbol}', "");
Expect(0, 128883, '\P{Category=_-symbol}', "");
Expect(1, 128883, '\P{^Category=_-symbol}', "");
Expect(0, 128884, '\p{Category=_-symbol}', "");
Expect(1, 128884, '\p{^Category=_-symbol}', "");
Expect(1, 128884, '\P{Category=_-symbol}', "");
Expect(0, 128884, '\P{^Category=_-symbol}', "");
Error('\p{Is_General_Category=_	S/a/}');
Error('\P{Is_General_Category=_	S/a/}');
Expect(1, 128883, '\p{Is_General_Category=s}', "");
Expect(0, 128883, '\p{^Is_General_Category=s}', "");
Expect(0, 128883, '\P{Is_General_Category=s}', "");
Expect(1, 128883, '\P{^Is_General_Category=s}', "");
Expect(0, 128884, '\p{Is_General_Category=s}', "");
Expect(1, 128884, '\p{^Is_General_Category=s}', "");
Expect(1, 128884, '\P{Is_General_Category=s}', "");
Expect(0, 128884, '\P{^Is_General_Category=s}', "");
Expect(1, 128883, '\p{Is_General_Category= S}', "");
Expect(0, 128883, '\p{^Is_General_Category= S}', "");
Expect(0, 128883, '\P{Is_General_Category= S}', "");
Expect(1, 128883, '\P{^Is_General_Category= S}', "");
Expect(0, 128884, '\p{Is_General_Category= S}', "");
Expect(1, 128884, '\p{^Is_General_Category= S}', "");
Expect(1, 128884, '\P{Is_General_Category= S}', "");
Expect(0, 128884, '\P{^Is_General_Category= S}', "");
Error('\p{Is_Gc=/a/Symbol}');
Error('\P{Is_Gc=/a/Symbol}');
Expect(1, 128883, '\p{Is_Gc=symbol}', "");
Expect(0, 128883, '\p{^Is_Gc=symbol}', "");
Expect(0, 128883, '\P{Is_Gc=symbol}', "");
Expect(1, 128883, '\P{^Is_Gc=symbol}', "");
Expect(0, 128884, '\p{Is_Gc=symbol}', "");
Expect(1, 128884, '\p{^Is_Gc=symbol}', "");
Expect(1, 128884, '\P{Is_Gc=symbol}', "");
Expect(0, 128884, '\P{^Is_Gc=symbol}', "");
Expect(1, 128883, '\p{Is_Gc=_-Symbol}', "");
Expect(0, 128883, '\p{^Is_Gc=_-Symbol}', "");
Expect(0, 128883, '\P{Is_Gc=_-Symbol}', "");
Expect(1, 128883, '\P{^Is_Gc=_-Symbol}', "");
Expect(0, 128884, '\p{Is_Gc=_-Symbol}', "");
Expect(1, 128884, '\p{^Is_Gc=_-Symbol}', "");
Expect(1, 128884, '\P{Is_Gc=_-Symbol}', "");
Expect(0, 128884, '\P{^Is_Gc=_-Symbol}', "");
Error('\p{Is_Category= -s/a/}');
Error('\P{Is_Category= -s/a/}');
Expect(1, 128883, '\p{Is_Category=s}', "");
Expect(0, 128883, '\p{^Is_Category=s}', "");
Expect(0, 128883, '\P{Is_Category=s}', "");
Expect(1, 128883, '\P{^Is_Category=s}', "");
Expect(0, 128884, '\p{Is_Category=s}', "");
Expect(1, 128884, '\p{^Is_Category=s}', "");
Expect(1, 128884, '\P{Is_Category=s}', "");
Expect(0, 128884, '\P{^Is_Category=s}', "");
Expect(1, 128883, '\p{Is_Category= S}', "");
Expect(0, 128883, '\p{^Is_Category= S}', "");
Expect(0, 128883, '\P{Is_Category= S}', "");
Expect(1, 128883, '\P{^Is_Category= S}', "");
Expect(0, 128884, '\p{Is_Category= S}', "");
Expect(1, 128884, '\p{^Is_Category= S}', "");
Expect(1, 128884, '\P{Is_Category= S}', "");
Expect(0, 128884, '\P{^Is_Category= S}', "");
Error('\p{General_Category=-/a/spacing_Mark}');
Error('\P{General_Category=-/a/spacing_Mark}');
Expect(1, 119154, '\p{General_Category:	spacingmark}', "");
Expect(0, 119154, '\p{^General_Category:	spacingmark}', "");
Expect(0, 119154, '\P{General_Category:	spacingmark}', "");
Expect(1, 119154, '\P{^General_Category:	spacingmark}', "");
Expect(0, 119155, '\p{General_Category:	spacingmark}', "");
Expect(1, 119155, '\p{^General_Category:	spacingmark}', "");
Expect(1, 119155, '\P{General_Category:	spacingmark}', "");
Expect(0, 119155, '\P{^General_Category:	spacingmark}', "");
Expect(1, 119154, '\p{General_Category= Spacing_Mark}', "");
Expect(0, 119154, '\p{^General_Category= Spacing_Mark}', "");
Expect(0, 119154, '\P{General_Category= Spacing_Mark}', "");
Expect(1, 119154, '\P{^General_Category= Spacing_Mark}', "");
Expect(0, 119155, '\p{General_Category= Spacing_Mark}', "");
Expect(1, 119155, '\p{^General_Category= Spacing_Mark}', "");
Expect(1, 119155, '\P{General_Category= Spacing_Mark}', "");
Expect(0, 119155, '\P{^General_Category= Spacing_Mark}', "");
Error('\p{Gc=/a/ _Mc}');
Error('\P{Gc=/a/ _Mc}');
Expect(1, 119154, '\p{Gc=mc}', "");
Expect(0, 119154, '\p{^Gc=mc}', "");
Expect(0, 119154, '\P{Gc=mc}', "");
Expect(1, 119154, '\P{^Gc=mc}', "");
Expect(0, 119155, '\p{Gc=mc}', "");
Expect(1, 119155, '\p{^Gc=mc}', "");
Expect(1, 119155, '\P{Gc=mc}', "");
Expect(0, 119155, '\P{^Gc=mc}', "");
Expect(1, 119154, '\p{Gc=		MC}', "");
Expect(0, 119154, '\p{^Gc=		MC}', "");
Expect(0, 119154, '\P{Gc=		MC}', "");
Expect(1, 119154, '\P{^Gc=		MC}', "");
Expect(0, 119155, '\p{Gc=		MC}', "");
Expect(1, 119155, '\p{^Gc=		MC}', "");
Expect(1, 119155, '\P{Gc=		MC}', "");
Expect(0, 119155, '\P{^Gc=		MC}', "");
Error('\p{Category=_:=Spacing_MARK}');
Error('\P{Category=_:=Spacing_MARK}');
Expect(1, 119154, '\p{Category=spacingmark}', "");
Expect(0, 119154, '\p{^Category=spacingmark}', "");
Expect(0, 119154, '\P{Category=spacingmark}', "");
Expect(1, 119154, '\P{^Category=spacingmark}', "");
Expect(0, 119155, '\p{Category=spacingmark}', "");
Expect(1, 119155, '\p{^Category=spacingmark}', "");
Expect(1, 119155, '\P{Category=spacingmark}', "");
Expect(0, 119155, '\P{^Category=spacingmark}', "");
Expect(1, 119154, '\p{Category=_	Spacing_Mark}', "");
Expect(0, 119154, '\p{^Category=_	Spacing_Mark}', "");
Expect(0, 119154, '\P{Category=_	Spacing_Mark}', "");
Expect(1, 119154, '\P{^Category=_	Spacing_Mark}', "");
Expect(0, 119155, '\p{Category=_	Spacing_Mark}', "");
Expect(1, 119155, '\p{^Category=_	Spacing_Mark}', "");
Expect(1, 119155, '\P{Category=_	Spacing_Mark}', "");
Expect(0, 119155, '\P{^Category=_	Spacing_Mark}', "");
Error('\p{Is_General_Category=:=_	Mc}');
Error('\P{Is_General_Category=:=_	Mc}');
Expect(1, 119154, '\p{Is_General_Category=mc}', "");
Expect(0, 119154, '\p{^Is_General_Category=mc}', "");
Expect(0, 119154, '\P{Is_General_Category=mc}', "");
Expect(1, 119154, '\P{^Is_General_Category=mc}', "");
Expect(0, 119155, '\p{Is_General_Category=mc}', "");
Expect(1, 119155, '\p{^Is_General_Category=mc}', "");
Expect(1, 119155, '\P{Is_General_Category=mc}', "");
Expect(0, 119155, '\P{^Is_General_Category=mc}', "");
Expect(1, 119154, '\p{Is_General_Category= Mc}', "");
Expect(0, 119154, '\p{^Is_General_Category= Mc}', "");
Expect(0, 119154, '\P{Is_General_Category= Mc}', "");
Expect(1, 119154, '\P{^Is_General_Category= Mc}', "");
Expect(0, 119155, '\p{Is_General_Category= Mc}', "");
Expect(1, 119155, '\p{^Is_General_Category= Mc}', "");
Expect(1, 119155, '\P{Is_General_Category= Mc}', "");
Expect(0, 119155, '\P{^Is_General_Category= Mc}', "");
Error('\p{Is_Gc= 	spacing_Mark:=}');
Error('\P{Is_Gc= 	spacing_Mark:=}');
Expect(1, 119154, '\p{Is_Gc=spacingmark}', "");
Expect(0, 119154, '\p{^Is_Gc=spacingmark}', "");
Expect(0, 119154, '\P{Is_Gc=spacingmark}', "");
Expect(1, 119154, '\P{^Is_Gc=spacingmark}', "");
Expect(0, 119155, '\p{Is_Gc=spacingmark}', "");
Expect(1, 119155, '\p{^Is_Gc=spacingmark}', "");
Expect(1, 119155, '\P{Is_Gc=spacingmark}', "");
Expect(0, 119155, '\P{^Is_Gc=spacingmark}', "");
Expect(1, 119154, '\p{Is_Gc=- Spacing_mark}', "");
Expect(0, 119154, '\p{^Is_Gc=- Spacing_mark}', "");
Expect(0, 119154, '\P{Is_Gc=- Spacing_mark}', "");
Expect(1, 119154, '\P{^Is_Gc=- Spacing_mark}', "");
Expect(0, 119155, '\p{Is_Gc=- Spacing_mark}', "");
Expect(1, 119155, '\p{^Is_Gc=- Spacing_mark}', "");
Expect(1, 119155, '\P{Is_Gc=- Spacing_mark}', "");
Expect(0, 119155, '\P{^Is_Gc=- Spacing_mark}', "");
Error('\p{Is_Category::= Mc}');
Error('\P{Is_Category::= Mc}');
Expect(1, 119154, '\p{Is_Category: mc}', "");
Expect(0, 119154, '\p{^Is_Category: mc}', "");
Expect(0, 119154, '\P{Is_Category: mc}', "");
Expect(1, 119154, '\P{^Is_Category: mc}', "");
Expect(0, 119155, '\p{Is_Category: mc}', "");
Expect(1, 119155, '\p{^Is_Category: mc}', "");
Expect(1, 119155, '\P{Is_Category: mc}', "");
Expect(0, 119155, '\P{^Is_Category: mc}', "");
Expect(1, 119154, '\p{Is_Category= mc}', "");
Expect(0, 119154, '\p{^Is_Category= mc}', "");
Expect(0, 119154, '\P{Is_Category= mc}', "");
Expect(1, 119154, '\P{^Is_Category= mc}', "");
Expect(0, 119155, '\p{Is_Category= mc}', "");
Expect(1, 119155, '\p{^Is_Category= mc}', "");
Expect(1, 119155, '\P{Is_Category= mc}', "");
Expect(0, 119155, '\P{^Is_Category= mc}', "");
Error('\p{General_Category=:= 	Cased_Letter}');
Error('\P{General_Category=:= 	Cased_Letter}');
Expect(1, 120779, '\p{General_Category:	casedletter}', "");
Expect(0, 120779, '\p{^General_Category:	casedletter}', "");
Expect(0, 120779, '\P{General_Category:	casedletter}', "");
Expect(1, 120779, '\P{^General_Category:	casedletter}', "");
Expect(0, 120780, '\p{General_Category:	casedletter}', "");
Expect(1, 120780, '\p{^General_Category:	casedletter}', "");
Expect(1, 120780, '\P{General_Category:	casedletter}', "");
Expect(0, 120780, '\P{^General_Category:	casedletter}', "");
Expect(1, 120779, '\p{General_Category=-Cased_Letter}', "");
Expect(0, 120779, '\p{^General_Category=-Cased_Letter}', "");
Expect(0, 120779, '\P{General_Category=-Cased_Letter}', "");
Expect(1, 120779, '\P{^General_Category=-Cased_Letter}', "");
Expect(0, 120780, '\p{General_Category=-Cased_Letter}', "");
Expect(1, 120780, '\p{^General_Category=-Cased_Letter}', "");
Expect(1, 120780, '\P{General_Category=-Cased_Letter}', "");
Expect(0, 120780, '\P{^General_Category=-Cased_Letter}', "");
Error('\p{Gc=__LC:=}');
Error('\P{Gc=__LC:=}');
Expect(1, 120779, '\p{Gc: lc}', "");
Expect(0, 120779, '\p{^Gc: lc}', "");
Expect(0, 120779, '\P{Gc: lc}', "");
Expect(1, 120779, '\P{^Gc: lc}', "");
Expect(0, 120780, '\p{Gc: lc}', "");
Expect(1, 120780, '\p{^Gc: lc}', "");
Expect(1, 120780, '\P{Gc: lc}', "");
Expect(0, 120780, '\P{^Gc: lc}', "");
Expect(1, 120779, '\p{Gc=-LC}', "");
Expect(0, 120779, '\p{^Gc=-LC}', "");
Expect(0, 120779, '\P{Gc=-LC}', "");
Expect(1, 120779, '\P{^Gc=-LC}', "");
Expect(0, 120780, '\p{Gc=-LC}', "");
Expect(1, 120780, '\p{^Gc=-LC}', "");
Expect(1, 120780, '\P{Gc=-LC}', "");
Expect(0, 120780, '\P{^Gc=-LC}', "");
Error('\p{Category=L_:=}');
Error('\P{Category=L_:=}');
Expect(1, 120779, '\p{Category=l_}', "");
Expect(0, 120779, '\p{^Category=l_}', "");
Expect(0, 120779, '\P{Category=l_}', "");
Expect(1, 120779, '\P{^Category=l_}', "");
Expect(0, 120780, '\p{Category=l_}', "");
Expect(1, 120780, '\p{^Category=l_}', "");
Expect(1, 120780, '\P{Category=l_}', "");
Expect(0, 120780, '\P{^Category=l_}', "");
Expect(1, 120779, '\p{Category=-l_}', "");
Expect(0, 120779, '\p{^Category=-l_}', "");
Expect(0, 120779, '\P{Category=-l_}', "");
Expect(1, 120779, '\P{^Category=-l_}', "");
Expect(0, 120780, '\p{Category=-l_}', "");
Expect(1, 120780, '\p{^Category=-l_}', "");
Expect(1, 120780, '\P{Category=-l_}', "");
Expect(0, 120780, '\P{^Category=-l_}', "");
Error('\p{Is_General_Category=/a/_	L&}');
Error('\P{Is_General_Category=/a/_	L&}');
Expect(1, 120779, '\p{Is_General_Category=l&}', "");
Expect(0, 120779, '\p{^Is_General_Category=l&}', "");
Expect(0, 120779, '\P{Is_General_Category=l&}', "");
Expect(1, 120779, '\P{^Is_General_Category=l&}', "");
Expect(0, 120780, '\p{Is_General_Category=l&}', "");
Expect(1, 120780, '\p{^Is_General_Category=l&}', "");
Expect(1, 120780, '\P{Is_General_Category=l&}', "");
Expect(0, 120780, '\P{^Is_General_Category=l&}', "");
Expect(1, 120779, '\p{Is_General_Category= l&}', "");
Expect(0, 120779, '\p{^Is_General_Category= l&}', "");
Expect(0, 120779, '\P{Is_General_Category= l&}', "");
Expect(1, 120779, '\P{^Is_General_Category= l&}', "");
Expect(0, 120780, '\p{Is_General_Category= l&}', "");
Expect(1, 120780, '\p{^Is_General_Category= l&}', "");
Expect(1, 120780, '\P{Is_General_Category= l&}', "");
Expect(0, 120780, '\P{^Is_General_Category= l&}', "");
Error('\p{Is_Gc=/a/ CASED_Letter}');
Error('\P{Is_Gc=/a/ CASED_Letter}');
Expect(1, 120779, '\p{Is_Gc=casedletter}', "");
Expect(0, 120779, '\p{^Is_Gc=casedletter}', "");
Expect(0, 120779, '\P{Is_Gc=casedletter}', "");
Expect(1, 120779, '\P{^Is_Gc=casedletter}', "");
Expect(0, 120780, '\p{Is_Gc=casedletter}', "");
Expect(1, 120780, '\p{^Is_Gc=casedletter}', "");
Expect(1, 120780, '\P{Is_Gc=casedletter}', "");
Expect(0, 120780, '\P{^Is_Gc=casedletter}', "");
Expect(1, 120779, '\p{Is_Gc=_cased_letter}', "");
Expect(0, 120779, '\p{^Is_Gc=_cased_letter}', "");
Expect(0, 120779, '\P{Is_Gc=_cased_letter}', "");
Expect(1, 120779, '\P{^Is_Gc=_cased_letter}', "");
Expect(0, 120780, '\p{Is_Gc=_cased_letter}', "");
Expect(1, 120780, '\p{^Is_Gc=_cased_letter}', "");
Expect(1, 120780, '\P{Is_Gc=_cased_letter}', "");
Expect(0, 120780, '\P{^Is_Gc=_cased_letter}', "");
Error('\p{Is_Category=	LC/a/}');
Error('\P{Is_Category=	LC/a/}');
Expect(1, 120779, '\p{Is_Category:   lc}', "");
Expect(0, 120779, '\p{^Is_Category:   lc}', "");
Expect(0, 120779, '\P{Is_Category:   lc}', "");
Expect(1, 120779, '\P{^Is_Category:   lc}', "");
Expect(0, 120780, '\p{Is_Category:   lc}', "");
Expect(1, 120780, '\p{^Is_Category:   lc}', "");
Expect(1, 120780, '\P{Is_Category:   lc}', "");
Expect(0, 120780, '\P{^Is_Category:   lc}', "");
Expect(1, 120779, '\p{Is_Category=-LC}', "");
Expect(0, 120779, '\p{^Is_Category=-LC}', "");
Expect(0, 120779, '\P{Is_Category=-LC}', "");
Expect(1, 120779, '\P{^Is_Category=-LC}', "");
Expect(0, 120780, '\p{Is_Category=-LC}', "");
Expect(1, 120780, '\p{^Is_Category=-LC}', "");
Expect(1, 120780, '\P{Is_Category=-LC}', "");
Expect(0, 120780, '\P{^Is_Category=-LC}', "");
Error('\p{General_Category=:=_	Connector_PUNCTUATION}');
Error('\P{General_Category=:=_	Connector_PUNCTUATION}');
Expect(1, 65343, '\p{General_Category=connectorpunctuation}', "");
Expect(0, 65343, '\p{^General_Category=connectorpunctuation}', "");
Expect(0, 65343, '\P{General_Category=connectorpunctuation}', "");
Expect(1, 65343, '\P{^General_Category=connectorpunctuation}', "");
Expect(0, 65344, '\p{General_Category=connectorpunctuation}', "");
Expect(1, 65344, '\p{^General_Category=connectorpunctuation}', "");
Expect(1, 65344, '\P{General_Category=connectorpunctuation}', "");
Expect(0, 65344, '\P{^General_Category=connectorpunctuation}', "");
Expect(1, 65343, '\p{General_Category=_ Connector_punctuation}', "");
Expect(0, 65343, '\p{^General_Category=_ Connector_punctuation}', "");
Expect(0, 65343, '\P{General_Category=_ Connector_punctuation}', "");
Expect(1, 65343, '\P{^General_Category=_ Connector_punctuation}', "");
Expect(0, 65344, '\p{General_Category=_ Connector_punctuation}', "");
Expect(1, 65344, '\p{^General_Category=_ Connector_punctuation}', "");
Expect(1, 65344, '\P{General_Category=_ Connector_punctuation}', "");
Expect(0, 65344, '\P{^General_Category=_ Connector_punctuation}', "");
Error('\p{Gc=-	PC/a/}');
Error('\P{Gc=-	PC/a/}');
Expect(1, 65343, '\p{Gc=pc}', "");
Expect(0, 65343, '\p{^Gc=pc}', "");
Expect(0, 65343, '\P{Gc=pc}', "");
Expect(1, 65343, '\P{^Gc=pc}', "");
Expect(0, 65344, '\p{Gc=pc}', "");
Expect(1, 65344, '\p{^Gc=pc}', "");
Expect(1, 65344, '\P{Gc=pc}', "");
Expect(0, 65344, '\P{^Gc=pc}', "");
Expect(1, 65343, '\p{Gc=-_Pc}', "");
Expect(0, 65343, '\p{^Gc=-_Pc}', "");
Expect(0, 65343, '\P{Gc=-_Pc}', "");
Expect(1, 65343, '\P{^Gc=-_Pc}', "");
Expect(0, 65344, '\p{Gc=-_Pc}', "");
Expect(1, 65344, '\p{^Gc=-_Pc}', "");
Expect(1, 65344, '\P{Gc=-_Pc}', "");
Expect(0, 65344, '\P{^Gc=-_Pc}', "");
Error('\p{Category=:=_	CONNECTOR_PUNCTUATION}');
Error('\P{Category=:=_	CONNECTOR_PUNCTUATION}');
Expect(1, 65343, '\p{Category=connectorpunctuation}', "");
Expect(0, 65343, '\p{^Category=connectorpunctuation}', "");
Expect(0, 65343, '\P{Category=connectorpunctuation}', "");
Expect(1, 65343, '\P{^Category=connectorpunctuation}', "");
Expect(0, 65344, '\p{Category=connectorpunctuation}', "");
Expect(1, 65344, '\p{^Category=connectorpunctuation}', "");
Expect(1, 65344, '\P{Category=connectorpunctuation}', "");
Expect(0, 65344, '\P{^Category=connectorpunctuation}', "");
Expect(1, 65343, '\p{Category: _Connector_Punctuation}', "");
Expect(0, 65343, '\p{^Category: _Connector_Punctuation}', "");
Expect(0, 65343, '\P{Category: _Connector_Punctuation}', "");
Expect(1, 65343, '\P{^Category: _Connector_Punctuation}', "");
Expect(0, 65344, '\p{Category: _Connector_Punctuation}', "");
Expect(1, 65344, '\p{^Category: _Connector_Punctuation}', "");
Expect(1, 65344, '\P{Category: _Connector_Punctuation}', "");
Expect(0, 65344, '\P{^Category: _Connector_Punctuation}', "");
Error('\p{Is_General_Category=	_Pc:=}');
Error('\P{Is_General_Category=	_Pc:=}');
Expect(1, 65343, '\p{Is_General_Category:   pc}', "");
Expect(0, 65343, '\p{^Is_General_Category:   pc}', "");
Expect(0, 65343, '\P{Is_General_Category:   pc}', "");
Expect(1, 65343, '\P{^Is_General_Category:   pc}', "");
Expect(0, 65344, '\p{Is_General_Category:   pc}', "");
Expect(1, 65344, '\p{^Is_General_Category:   pc}', "");
Expect(1, 65344, '\P{Is_General_Category:   pc}', "");
Expect(0, 65344, '\P{^Is_General_Category:   pc}', "");
Expect(1, 65343, '\p{Is_General_Category=-pc}', "");
Expect(0, 65343, '\p{^Is_General_Category=-pc}', "");
Expect(0, 65343, '\P{Is_General_Category=-pc}', "");
Expect(1, 65343, '\P{^Is_General_Category=-pc}', "");
Expect(0, 65344, '\p{Is_General_Category=-pc}', "");
Expect(1, 65344, '\p{^Is_General_Category=-pc}', "");
Expect(1, 65344, '\P{Is_General_Category=-pc}', "");
Expect(0, 65344, '\P{^Is_General_Category=-pc}', "");
Error('\p{Is_Gc=-/a/CONNECTOR_punctuation}');
Error('\P{Is_Gc=-/a/CONNECTOR_punctuation}');
Expect(1, 65343, '\p{Is_Gc=connectorpunctuation}', "");
Expect(0, 65343, '\p{^Is_Gc=connectorpunctuation}', "");
Expect(0, 65343, '\P{Is_Gc=connectorpunctuation}', "");
Expect(1, 65343, '\P{^Is_Gc=connectorpunctuation}', "");
Expect(0, 65344, '\p{Is_Gc=connectorpunctuation}', "");
Expect(1, 65344, '\p{^Is_Gc=connectorpunctuation}', "");
Expect(1, 65344, '\P{Is_Gc=connectorpunctuation}', "");
Expect(0, 65344, '\P{^Is_Gc=connectorpunctuation}', "");
Expect(1, 65343, '\p{Is_Gc=-	CONNECTOR_Punctuation}', "");
Expect(0, 65343, '\p{^Is_Gc=-	CONNECTOR_Punctuation}', "");
Expect(0, 65343, '\P{Is_Gc=-	CONNECTOR_Punctuation}', "");
Expect(1, 65343, '\P{^Is_Gc=-	CONNECTOR_Punctuation}', "");
Expect(0, 65344, '\p{Is_Gc=-	CONNECTOR_Punctuation}', "");
Expect(1, 65344, '\p{^Is_Gc=-	CONNECTOR_Punctuation}', "");
Expect(1, 65344, '\P{Is_Gc=-	CONNECTOR_Punctuation}', "");
Expect(0, 65344, '\P{^Is_Gc=-	CONNECTOR_Punctuation}', "");
Error('\p{Is_Category=:=PC}');
Error('\P{Is_Category=:=PC}');
Expect(1, 65343, '\p{Is_Category=pc}', "");
Expect(0, 65343, '\p{^Is_Category=pc}', "");
Expect(0, 65343, '\P{Is_Category=pc}', "");
Expect(1, 65343, '\P{^Is_Category=pc}', "");
Expect(0, 65344, '\p{Is_Category=pc}', "");
Expect(1, 65344, '\p{^Is_Category=pc}', "");
Expect(1, 65344, '\P{Is_Category=pc}', "");
Expect(0, 65344, '\P{^Is_Category=pc}', "");
Expect(1, 65343, '\p{Is_Category=- PC}', "");
Expect(0, 65343, '\p{^Is_Category=- PC}', "");
Expect(0, 65343, '\P{Is_Category=- PC}', "");
Expect(1, 65343, '\P{^Is_Category=- PC}', "");
Expect(0, 65344, '\p{Is_Category=- PC}', "");
Expect(1, 65344, '\p{^Is_Category=- PC}', "");
Expect(1, 65344, '\P{Is_Category=- PC}', "");
Expect(0, 65344, '\P{^Is_Category=- PC}', "");
Error('\p{General_Category= -Surrogate:=}');
Error('\P{General_Category= -Surrogate:=}');
Expect(1, 57343, '\p{General_Category=surrogate}', "");
Expect(0, 57343, '\p{^General_Category=surrogate}', "");
Expect(0, 57343, '\P{General_Category=surrogate}', "");
Expect(1, 57343, '\P{^General_Category=surrogate}', "");
Expect(0, 57344, '\p{General_Category=surrogate}', "");
Expect(1, 57344, '\p{^General_Category=surrogate}', "");
Expect(1, 57344, '\P{General_Category=surrogate}', "");
Expect(0, 57344, '\P{^General_Category=surrogate}', "");
Expect(1, 57343, '\p{General_Category=_Surrogate}', "");
Expect(0, 57343, '\p{^General_Category=_Surrogate}', "");
Expect(0, 57343, '\P{General_Category=_Surrogate}', "");
Expect(1, 57343, '\P{^General_Category=_Surrogate}', "");
Expect(0, 57344, '\p{General_Category=_Surrogate}', "");
Expect(1, 57344, '\p{^General_Category=_Surrogate}', "");
Expect(1, 57344, '\P{General_Category=_Surrogate}', "");
Expect(0, 57344, '\P{^General_Category=_Surrogate}', "");
Error('\p{Gc=:=Cs}');
Error('\P{Gc=:=Cs}');
Expect(1, 57343, '\p{Gc=cs}', "");
Expect(0, 57343, '\p{^Gc=cs}', "");
Expect(0, 57343, '\P{Gc=cs}', "");
Expect(1, 57343, '\P{^Gc=cs}', "");
Expect(0, 57344, '\p{Gc=cs}', "");
Expect(1, 57344, '\p{^Gc=cs}', "");
Expect(1, 57344, '\P{Gc=cs}', "");
Expect(0, 57344, '\P{^Gc=cs}', "");
Expect(1, 57343, '\p{Gc= cs}', "");
Expect(0, 57343, '\p{^Gc= cs}', "");
Expect(0, 57343, '\P{Gc= cs}', "");
Expect(1, 57343, '\P{^Gc= cs}', "");
Expect(0, 57344, '\p{Gc= cs}', "");
Expect(1, 57344, '\p{^Gc= cs}', "");
Expect(1, 57344, '\P{Gc= cs}', "");
Expect(0, 57344, '\P{^Gc= cs}', "");
Error('\p{Category=:=- Surrogate}');
Error('\P{Category=:=- Surrogate}');
Expect(1, 57343, '\p{Category=surrogate}', "");
Expect(0, 57343, '\p{^Category=surrogate}', "");
Expect(0, 57343, '\P{Category=surrogate}', "");
Expect(1, 57343, '\P{^Category=surrogate}', "");
Expect(0, 57344, '\p{Category=surrogate}', "");
Expect(1, 57344, '\p{^Category=surrogate}', "");
Expect(1, 57344, '\P{Category=surrogate}', "");
Expect(0, 57344, '\P{^Category=surrogate}', "");
Expect(1, 57343, '\p{Category= _SURROGATE}', "");
Expect(0, 57343, '\p{^Category= _SURROGATE}', "");
Expect(0, 57343, '\P{Category= _SURROGATE}', "");
Expect(1, 57343, '\P{^Category= _SURROGATE}', "");
Expect(0, 57344, '\p{Category= _SURROGATE}', "");
Expect(1, 57344, '\p{^Category= _SURROGATE}', "");
Expect(1, 57344, '\P{Category= _SURROGATE}', "");
Expect(0, 57344, '\P{^Category= _SURROGATE}', "");
Error('\p{Is_General_Category:/a/_-cs}');
Error('\P{Is_General_Category:/a/_-cs}');
Expect(1, 57343, '\p{Is_General_Category=cs}', "");
Expect(0, 57343, '\p{^Is_General_Category=cs}', "");
Expect(0, 57343, '\P{Is_General_Category=cs}', "");
Expect(1, 57343, '\P{^Is_General_Category=cs}', "");
Expect(0, 57344, '\p{Is_General_Category=cs}', "");
Expect(1, 57344, '\p{^Is_General_Category=cs}', "");
Expect(1, 57344, '\P{Is_General_Category=cs}', "");
Expect(0, 57344, '\P{^Is_General_Category=cs}', "");
Expect(1, 57343, '\p{Is_General_Category= -Cs}', "");
Expect(0, 57343, '\p{^Is_General_Category= -Cs}', "");
Expect(0, 57343, '\P{Is_General_Category= -Cs}', "");
Expect(1, 57343, '\P{^Is_General_Category= -Cs}', "");
Expect(0, 57344, '\p{Is_General_Category= -Cs}', "");
Expect(1, 57344, '\p{^Is_General_Category= -Cs}', "");
Expect(1, 57344, '\P{Is_General_Category= -Cs}', "");
Expect(0, 57344, '\P{^Is_General_Category= -Cs}', "");
Error('\p{Is_Gc=--Surrogate:=}');
Error('\P{Is_Gc=--Surrogate:=}');
Expect(1, 57343, '\p{Is_Gc=surrogate}', "");
Expect(0, 57343, '\p{^Is_Gc=surrogate}', "");
Expect(0, 57343, '\P{Is_Gc=surrogate}', "");
Expect(1, 57343, '\P{^Is_Gc=surrogate}', "");
Expect(0, 57344, '\p{Is_Gc=surrogate}', "");
Expect(1, 57344, '\p{^Is_Gc=surrogate}', "");
Expect(1, 57344, '\P{Is_Gc=surrogate}', "");
Expect(0, 57344, '\P{^Is_Gc=surrogate}', "");
Expect(1, 57343, '\p{Is_Gc: _	surrogate}', "");
Expect(0, 57343, '\p{^Is_Gc: _	surrogate}', "");
Expect(0, 57343, '\P{Is_Gc: _	surrogate}', "");
Expect(1, 57343, '\P{^Is_Gc: _	surrogate}', "");
Expect(0, 57344, '\p{Is_Gc: _	surrogate}', "");
Expect(1, 57344, '\p{^Is_Gc: _	surrogate}', "");
Expect(1, 57344, '\P{Is_Gc: _	surrogate}', "");
Expect(0, 57344, '\P{^Is_Gc: _	surrogate}', "");
Error('\p{Is_Category= /a/Cs}');
Error('\P{Is_Category= /a/Cs}');
Expect(1, 57343, '\p{Is_Category=cs}', "");
Expect(0, 57343, '\p{^Is_Category=cs}', "");
Expect(0, 57343, '\P{Is_Category=cs}', "");
Expect(1, 57343, '\P{^Is_Category=cs}', "");
Expect(0, 57344, '\p{Is_Category=cs}', "");
Expect(1, 57344, '\p{^Is_Category=cs}', "");
Expect(1, 57344, '\P{Is_Category=cs}', "");
Expect(0, 57344, '\P{^Is_Category=cs}', "");
Expect(1, 57343, '\p{Is_Category:	  Cs}', "");
Expect(0, 57343, '\p{^Is_Category:	  Cs}', "");
Expect(0, 57343, '\P{Is_Category:	  Cs}', "");
Expect(1, 57343, '\P{^Is_Category:	  Cs}', "");
Expect(0, 57344, '\p{Is_Category:	  Cs}', "");
Expect(1, 57344, '\p{^Is_Category:	  Cs}', "");
Expect(1, 57344, '\P{Is_Category:	  Cs}', "");
Expect(0, 57344, '\P{^Is_Category:	  Cs}', "");
Error('\p{General_Category=  PRIVATE_use:=}');
Error('\P{General_Category=  PRIVATE_use:=}');
Expect(1, 1114109, '\p{General_Category:privateuse}', "");
Expect(0, 1114109, '\p{^General_Category:privateuse}', "");
Expect(0, 1114109, '\P{General_Category:privateuse}', "");
Expect(1, 1114109, '\P{^General_Category:privateuse}', "");
Expect(0, 63744, '\p{General_Category:privateuse}', "");
Expect(1, 63744, '\p{^General_Category:privateuse}', "");
Expect(1, 63744, '\P{General_Category:privateuse}', "");
Expect(0, 63744, '\P{^General_Category:privateuse}', "");
Expect(1, 1114109, '\p{General_Category=-PRIVATE_USE}', "");
Expect(0, 1114109, '\p{^General_Category=-PRIVATE_USE}', "");
Expect(0, 1114109, '\P{General_Category=-PRIVATE_USE}', "");
Expect(1, 1114109, '\P{^General_Category=-PRIVATE_USE}', "");
Expect(0, 63744, '\p{General_Category=-PRIVATE_USE}', "");
Expect(1, 63744, '\p{^General_Category=-PRIVATE_USE}', "");
Expect(1, 63744, '\P{General_Category=-PRIVATE_USE}', "");
Expect(0, 63744, '\P{^General_Category=-PRIVATE_USE}', "");
Error('\p{Gc=-:=co}');
Error('\P{Gc=-:=co}');
Expect(1, 1114109, '\p{Gc=co}', "");
Expect(0, 1114109, '\p{^Gc=co}', "");
Expect(0, 1114109, '\P{Gc=co}', "");
Expect(1, 1114109, '\P{^Gc=co}', "");
Expect(0, 63744, '\p{Gc=co}', "");
Expect(1, 63744, '\p{^Gc=co}', "");
Expect(1, 63744, '\P{Gc=co}', "");
Expect(0, 63744, '\P{^Gc=co}', "");
Expect(1, 1114109, '\p{Gc=_ Co}', "");
Expect(0, 1114109, '\p{^Gc=_ Co}', "");
Expect(0, 1114109, '\P{Gc=_ Co}', "");
Expect(1, 1114109, '\P{^Gc=_ Co}', "");
Expect(0, 63744, '\p{Gc=_ Co}', "");
Expect(1, 63744, '\p{^Gc=_ Co}', "");
Expect(1, 63744, '\P{Gc=_ Co}', "");
Expect(0, 63744, '\P{^Gc=_ Co}', "");
Error('\p{Category: /a/	PRIVATE_use}');
Error('\P{Category: /a/	PRIVATE_use}');
Expect(1, 1114109, '\p{Category=privateuse}', "");
Expect(0, 1114109, '\p{^Category=privateuse}', "");
Expect(0, 1114109, '\P{Category=privateuse}', "");
Expect(1, 1114109, '\P{^Category=privateuse}', "");
Expect(0, 63744, '\p{Category=privateuse}', "");
Expect(1, 63744, '\p{^Category=privateuse}', "");
Expect(1, 63744, '\P{Category=privateuse}', "");
Expect(0, 63744, '\P{^Category=privateuse}', "");
Expect(1, 1114109, '\p{Category:  Private_Use}', "");
Expect(0, 1114109, '\p{^Category:  Private_Use}', "");
Expect(0, 1114109, '\P{Category:  Private_Use}', "");
Expect(1, 1114109, '\P{^Category:  Private_Use}', "");
Expect(0, 63744, '\p{Category:  Private_Use}', "");
Expect(1, 63744, '\p{^Category:  Private_Use}', "");
Expect(1, 63744, '\P{Category:  Private_Use}', "");
Expect(0, 63744, '\P{^Category:  Private_Use}', "");
Error('\p{Is_General_Category= co/a/}');
Error('\P{Is_General_Category= co/a/}');
Expect(1, 1114109, '\p{Is_General_Category=co}', "");
Expect(0, 1114109, '\p{^Is_General_Category=co}', "");
Expect(0, 1114109, '\P{Is_General_Category=co}', "");
Expect(1, 1114109, '\P{^Is_General_Category=co}', "");
Expect(0, 63744, '\p{Is_General_Category=co}', "");
Expect(1, 63744, '\p{^Is_General_Category=co}', "");
Expect(1, 63744, '\P{Is_General_Category=co}', "");
Expect(0, 63744, '\P{^Is_General_Category=co}', "");
Expect(1, 1114109, '\p{Is_General_Category=	_Co}', "");
Expect(0, 1114109, '\p{^Is_General_Category=	_Co}', "");
Expect(0, 1114109, '\P{Is_General_Category=	_Co}', "");
Expect(1, 1114109, '\P{^Is_General_Category=	_Co}', "");
Expect(0, 63744, '\p{Is_General_Category=	_Co}', "");
Expect(1, 63744, '\p{^Is_General_Category=	_Co}', "");
Expect(1, 63744, '\P{Is_General_Category=	_Co}', "");
Expect(0, 63744, '\P{^Is_General_Category=	_Co}', "");
Error('\p{Is_Gc= /a/private_Use}');
Error('\P{Is_Gc= /a/private_Use}');
Expect(1, 1114109, '\p{Is_Gc=privateuse}', "");
Expect(0, 1114109, '\p{^Is_Gc=privateuse}', "");
Expect(0, 1114109, '\P{Is_Gc=privateuse}', "");
Expect(1, 1114109, '\P{^Is_Gc=privateuse}', "");
Expect(0, 63744, '\p{Is_Gc=privateuse}', "");
Expect(1, 63744, '\p{^Is_Gc=privateuse}', "");
Expect(1, 63744, '\P{Is_Gc=privateuse}', "");
Expect(0, 63744, '\P{^Is_Gc=privateuse}', "");
Expect(1, 1114109, '\p{Is_Gc=-_Private_Use}', "");
Expect(0, 1114109, '\p{^Is_Gc=-_Private_Use}', "");
Expect(0, 1114109, '\P{Is_Gc=-_Private_Use}', "");
Expect(1, 1114109, '\P{^Is_Gc=-_Private_Use}', "");
Expect(0, 63744, '\p{Is_Gc=-_Private_Use}', "");
Expect(1, 63744, '\p{^Is_Gc=-_Private_Use}', "");
Expect(1, 63744, '\P{Is_Gc=-_Private_Use}', "");
Expect(0, 63744, '\P{^Is_Gc=-_Private_Use}', "");
Error('\p{Is_Category=/a/_ co}');
Error('\P{Is_Category=/a/_ co}');
Expect(1, 1114109, '\p{Is_Category=co}', "");
Expect(0, 1114109, '\p{^Is_Category=co}', "");
Expect(0, 1114109, '\P{Is_Category=co}', "");
Expect(1, 1114109, '\P{^Is_Category=co}', "");
Expect(0, 63744, '\p{Is_Category=co}', "");
Expect(1, 63744, '\p{^Is_Category=co}', "");
Expect(1, 63744, '\P{Is_Category=co}', "");
Expect(0, 63744, '\P{^Is_Category=co}', "");
Expect(1, 1114109, '\p{Is_Category=-	co}', "");
Expect(0, 1114109, '\p{^Is_Category=-	co}', "");
Expect(0, 1114109, '\P{Is_Category=-	co}', "");
Expect(1, 1114109, '\P{^Is_Category=-	co}', "");
Expect(0, 63744, '\p{Is_Category=-	co}', "");
Expect(1, 63744, '\p{^Is_Category=-	co}', "");
Expect(1, 63744, '\P{Is_Category=-	co}', "");
Expect(0, 63744, '\P{^Is_Category=-	co}', "");
Error('\p{General_Category=:=	lowercase_letter}');
Error('\P{General_Category=:=	lowercase_letter}');
Expect(1, 120779, '\p{General_Category:lowercaseletter}', "");
Expect(0, 120779, '\p{^General_Category:lowercaseletter}', "");
Expect(0, 120779, '\P{General_Category:lowercaseletter}', "");
Expect(1, 120779, '\P{^General_Category:lowercaseletter}', "");
Expect(0, 120780, '\p{General_Category:lowercaseletter}', "");
Expect(1, 120780, '\p{^General_Category:lowercaseletter}', "");
Expect(1, 120780, '\P{General_Category:lowercaseletter}', "");
Expect(0, 120780, '\P{^General_Category:lowercaseletter}', "");
Expect(1, 120779, '\p{General_Category=	Lowercase_letter}', "");
Expect(0, 120779, '\p{^General_Category=	Lowercase_letter}', "");
Expect(0, 120779, '\P{General_Category=	Lowercase_letter}', "");
Expect(1, 120779, '\P{^General_Category=	Lowercase_letter}', "");
Expect(0, 120780, '\p{General_Category=	Lowercase_letter}', "");
Expect(1, 120780, '\p{^General_Category=	Lowercase_letter}', "");
Expect(1, 120780, '\P{General_Category=	Lowercase_letter}', "");
Expect(0, 120780, '\P{^General_Category=	Lowercase_letter}', "");
Error('\p{Gc=:=	Ll}');
Error('\P{Gc=:=	Ll}');
Expect(1, 120779, '\p{Gc=ll}', "");
Expect(0, 120779, '\p{^Gc=ll}', "");
Expect(0, 120779, '\P{Gc=ll}', "");
Expect(1, 120779, '\P{^Gc=ll}', "");
Expect(0, 120780, '\p{Gc=ll}', "");
Expect(1, 120780, '\p{^Gc=ll}', "");
Expect(1, 120780, '\P{Gc=ll}', "");
Expect(0, 120780, '\P{^Gc=ll}', "");
Expect(1, 120779, '\p{Gc= -ll}', "");
Expect(0, 120779, '\p{^Gc= -ll}', "");
Expect(0, 120779, '\P{Gc= -ll}', "");
Expect(1, 120779, '\P{^Gc= -ll}', "");
Expect(0, 120780, '\p{Gc= -ll}', "");
Expect(1, 120780, '\p{^Gc= -ll}', "");
Expect(1, 120780, '\P{Gc= -ll}', "");
Expect(0, 120780, '\P{^Gc= -ll}', "");
Error('\p{Category=/a/ _Lowercase_LETTER}');
Error('\P{Category=/a/ _Lowercase_LETTER}');
Expect(1, 120779, '\p{Category=lowercaseletter}', "");
Expect(0, 120779, '\p{^Category=lowercaseletter}', "");
Expect(0, 120779, '\P{Category=lowercaseletter}', "");
Expect(1, 120779, '\P{^Category=lowercaseletter}', "");
Expect(0, 120780, '\p{Category=lowercaseletter}', "");
Expect(1, 120780, '\p{^Category=lowercaseletter}', "");
Expect(1, 120780, '\P{Category=lowercaseletter}', "");
Expect(0, 120780, '\P{^Category=lowercaseletter}', "");
Expect(1, 120779, '\p{Category=	_LOWERCASE_letter}', "");
Expect(0, 120779, '\p{^Category=	_LOWERCASE_letter}', "");
Expect(0, 120779, '\P{Category=	_LOWERCASE_letter}', "");
Expect(1, 120779, '\P{^Category=	_LOWERCASE_letter}', "");
Expect(0, 120780, '\p{Category=	_LOWERCASE_letter}', "");
Expect(1, 120780, '\p{^Category=	_LOWERCASE_letter}', "");
Expect(1, 120780, '\P{Category=	_LOWERCASE_letter}', "");
Expect(0, 120780, '\P{^Category=	_LOWERCASE_letter}', "");
Error('\p{Is_General_Category=:=_-Ll}');
Error('\P{Is_General_Category=:=_-Ll}');
Expect(1, 120779, '\p{Is_General_Category=ll}', "");
Expect(0, 120779, '\p{^Is_General_Category=ll}', "");
Expect(0, 120779, '\P{Is_General_Category=ll}', "");
Expect(1, 120779, '\P{^Is_General_Category=ll}', "");
Expect(0, 120780, '\p{Is_General_Category=ll}', "");
Expect(1, 120780, '\p{^Is_General_Category=ll}', "");
Expect(1, 120780, '\P{Is_General_Category=ll}', "");
Expect(0, 120780, '\P{^Is_General_Category=ll}', "");
Expect(1, 120779, '\p{Is_General_Category=_ll}', "");
Expect(0, 120779, '\p{^Is_General_Category=_ll}', "");
Expect(0, 120779, '\P{Is_General_Category=_ll}', "");
Expect(1, 120779, '\P{^Is_General_Category=_ll}', "");
Expect(0, 120780, '\p{Is_General_Category=_ll}', "");
Expect(1, 120780, '\p{^Is_General_Category=_ll}', "");
Expect(1, 120780, '\P{Is_General_Category=_ll}', "");
Expect(0, 120780, '\P{^Is_General_Category=_ll}', "");
Error('\p{Is_Gc=/a/Lowercase_letter}');
Error('\P{Is_Gc=/a/Lowercase_letter}');
Expect(1, 120779, '\p{Is_Gc=lowercaseletter}', "");
Expect(0, 120779, '\p{^Is_Gc=lowercaseletter}', "");
Expect(0, 120779, '\P{Is_Gc=lowercaseletter}', "");
Expect(1, 120779, '\P{^Is_Gc=lowercaseletter}', "");
Expect(0, 120780, '\p{Is_Gc=lowercaseletter}', "");
Expect(1, 120780, '\p{^Is_Gc=lowercaseletter}', "");
Expect(1, 120780, '\P{Is_Gc=lowercaseletter}', "");
Expect(0, 120780, '\P{^Is_Gc=lowercaseletter}', "");
Expect(1, 120779, '\p{Is_Gc=__LOWERCASE_Letter}', "");
Expect(0, 120779, '\p{^Is_Gc=__LOWERCASE_Letter}', "");
Expect(0, 120779, '\P{Is_Gc=__LOWERCASE_Letter}', "");
Expect(1, 120779, '\P{^Is_Gc=__LOWERCASE_Letter}', "");
Expect(0, 120780, '\p{Is_Gc=__LOWERCASE_Letter}', "");
Expect(1, 120780, '\p{^Is_Gc=__LOWERCASE_Letter}', "");
Expect(1, 120780, '\P{Is_Gc=__LOWERCASE_Letter}', "");
Expect(0, 120780, '\P{^Is_Gc=__LOWERCASE_Letter}', "");
Error('\p{Is_Category=-/a/ll}');
Error('\P{Is_Category=-/a/ll}');
Expect(1, 120779, '\p{Is_Category=ll}', "");
Expect(0, 120779, '\p{^Is_Category=ll}', "");
Expect(0, 120779, '\P{Is_Category=ll}', "");
Expect(1, 120779, '\P{^Is_Category=ll}', "");
Expect(0, 120780, '\p{Is_Category=ll}', "");
Expect(1, 120780, '\p{^Is_Category=ll}', "");
Expect(1, 120780, '\P{Is_Category=ll}', "");
Expect(0, 120780, '\P{^Is_Category=ll}', "");
Expect(1, 120779, '\p{Is_Category= _LL}', "");
Expect(0, 120779, '\p{^Is_Category= _LL}', "");
Expect(0, 120779, '\P{Is_Category= _LL}', "");
Expect(1, 120779, '\P{^Is_Category= _LL}', "");
Expect(0, 120780, '\p{Is_Category= _LL}', "");
Expect(1, 120780, '\p{^Is_Category= _LL}', "");
Expect(1, 120780, '\P{Is_Category= _LL}', "");
Expect(0, 120780, '\P{^Is_Category= _LL}', "");
Error('\p{General_Category=:=-	Close_PUNCTUATION}');
Error('\P{General_Category=:=-	Close_PUNCTUATION}');
Expect(1, 65379, '\p{General_Category=closepunctuation}', "");
Expect(0, 65379, '\p{^General_Category=closepunctuation}', "");
Expect(0, 65379, '\P{General_Category=closepunctuation}', "");
Expect(1, 65379, '\P{^General_Category=closepunctuation}', "");
Expect(0, 65380, '\p{General_Category=closepunctuation}', "");
Expect(1, 65380, '\p{^General_Category=closepunctuation}', "");
Expect(1, 65380, '\P{General_Category=closepunctuation}', "");
Expect(0, 65380, '\P{^General_Category=closepunctuation}', "");
Expect(1, 65379, '\p{General_Category= close_Punctuation}', "");
Expect(0, 65379, '\p{^General_Category= close_Punctuation}', "");
Expect(0, 65379, '\P{General_Category= close_Punctuation}', "");
Expect(1, 65379, '\P{^General_Category= close_Punctuation}', "");
Expect(0, 65380, '\p{General_Category= close_Punctuation}', "");
Expect(1, 65380, '\p{^General_Category= close_Punctuation}', "");
Expect(1, 65380, '\P{General_Category= close_Punctuation}', "");
Expect(0, 65380, '\P{^General_Category= close_Punctuation}', "");
Error('\p{Gc=Pe/a/}');
Error('\P{Gc=Pe/a/}');
Expect(1, 65379, '\p{Gc=pe}', "");
Expect(0, 65379, '\p{^Gc=pe}', "");
Expect(0, 65379, '\P{Gc=pe}', "");
Expect(1, 65379, '\P{^Gc=pe}', "");
Expect(0, 65380, '\p{Gc=pe}', "");
Expect(1, 65380, '\p{^Gc=pe}', "");
Expect(1, 65380, '\P{Gc=pe}', "");
Expect(0, 65380, '\P{^Gc=pe}', "");
Expect(1, 65379, '\p{Gc: PE}', "");
Expect(0, 65379, '\p{^Gc: PE}', "");
Expect(0, 65379, '\P{Gc: PE}', "");
Expect(1, 65379, '\P{^Gc: PE}', "");
Expect(0, 65380, '\p{Gc: PE}', "");
Expect(1, 65380, '\p{^Gc: PE}', "");
Expect(1, 65380, '\P{Gc: PE}', "");
Expect(0, 65380, '\P{^Gc: PE}', "");
Error('\p{Category=:=_-Close_Punctuation}');
Error('\P{Category=:=_-Close_Punctuation}');
Expect(1, 65379, '\p{Category=closepunctuation}', "");
Expect(0, 65379, '\p{^Category=closepunctuation}', "");
Expect(0, 65379, '\P{Category=closepunctuation}', "");
Expect(1, 65379, '\P{^Category=closepunctuation}', "");
Expect(0, 65380, '\p{Category=closepunctuation}', "");
Expect(1, 65380, '\p{^Category=closepunctuation}', "");
Expect(1, 65380, '\P{Category=closepunctuation}', "");
Expect(0, 65380, '\P{^Category=closepunctuation}', "");
Expect(1, 65379, '\p{Category=_ Close_punctuation}', "");
Expect(0, 65379, '\p{^Category=_ Close_punctuation}', "");
Expect(0, 65379, '\P{Category=_ Close_punctuation}', "");
Expect(1, 65379, '\P{^Category=_ Close_punctuation}', "");
Expect(0, 65380, '\p{Category=_ Close_punctuation}', "");
Expect(1, 65380, '\p{^Category=_ Close_punctuation}', "");
Expect(1, 65380, '\P{Category=_ Close_punctuation}', "");
Expect(0, 65380, '\P{^Category=_ Close_punctuation}', "");
Error('\p{Is_General_Category=:=Pe}');
Error('\P{Is_General_Category=:=Pe}');
Expect(1, 65379, '\p{Is_General_Category:	pe}', "");
Expect(0, 65379, '\p{^Is_General_Category:	pe}', "");
Expect(0, 65379, '\P{Is_General_Category:	pe}', "");
Expect(1, 65379, '\P{^Is_General_Category:	pe}', "");
Expect(0, 65380, '\p{Is_General_Category:	pe}', "");
Expect(1, 65380, '\p{^Is_General_Category:	pe}', "");
Expect(1, 65380, '\P{Is_General_Category:	pe}', "");
Expect(0, 65380, '\P{^Is_General_Category:	pe}', "");
Expect(1, 65379, '\p{Is_General_Category= PE}', "");
Expect(0, 65379, '\p{^Is_General_Category= PE}', "");
Expect(0, 65379, '\P{Is_General_Category= PE}', "");
Expect(1, 65379, '\P{^Is_General_Category= PE}', "");
Expect(0, 65380, '\p{Is_General_Category= PE}', "");
Expect(1, 65380, '\p{^Is_General_Category= PE}', "");
Expect(1, 65380, '\P{Is_General_Category= PE}', "");
Expect(0, 65380, '\P{^Is_General_Category= PE}', "");
Error('\p{Is_Gc=	close_punctuation/a/}');
Error('\P{Is_Gc=	close_punctuation/a/}');
Expect(1, 65379, '\p{Is_Gc=closepunctuation}', "");
Expect(0, 65379, '\p{^Is_Gc=closepunctuation}', "");
Expect(0, 65379, '\P{Is_Gc=closepunctuation}', "");
Expect(1, 65379, '\P{^Is_Gc=closepunctuation}', "");
Expect(0, 65380, '\p{Is_Gc=closepunctuation}', "");
Expect(1, 65380, '\p{^Is_Gc=closepunctuation}', "");
Expect(1, 65380, '\P{Is_Gc=closepunctuation}', "");
Expect(0, 65380, '\P{^Is_Gc=closepunctuation}', "");
Expect(1, 65379, '\p{Is_Gc=	Close_Punctuation}', "");
Expect(0, 65379, '\p{^Is_Gc=	Close_Punctuation}', "");
Expect(0, 65379, '\P{Is_Gc=	Close_Punctuation}', "");
Expect(1, 65379, '\P{^Is_Gc=	Close_Punctuation}', "");
Expect(0, 65380, '\p{Is_Gc=	Close_Punctuation}', "");
Expect(1, 65380, '\p{^Is_Gc=	Close_Punctuation}', "");
Expect(1, 65380, '\P{Is_Gc=	Close_Punctuation}', "");
Expect(0, 65380, '\P{^Is_Gc=	Close_Punctuation}', "");
Error('\p{Is_Category=_/a/pe}');
Error('\P{Is_Category=_/a/pe}');
Expect(1, 65379, '\p{Is_Category=pe}', "");
Expect(0, 65379, '\p{^Is_Category=pe}', "");
Expect(0, 65379, '\P{Is_Category=pe}', "");
Expect(1, 65379, '\P{^Is_Category=pe}', "");
Expect(0, 65380, '\p{Is_Category=pe}', "");
Expect(1, 65380, '\p{^Is_Category=pe}', "");
Expect(1, 65380, '\P{Is_Category=pe}', "");
Expect(0, 65380, '\P{^Is_Category=pe}', "");
Expect(1, 65379, '\p{Is_Category=__Pe}', "");
Expect(0, 65379, '\p{^Is_Category=__Pe}', "");
Expect(0, 65379, '\P{Is_Category=__Pe}', "");
Expect(1, 65379, '\P{^Is_Category=__Pe}', "");
Expect(0, 65380, '\p{Is_Category=__Pe}', "");
Expect(1, 65380, '\p{^Is_Category=__Pe}', "");
Expect(1, 65380, '\P{Is_Category=__Pe}', "");
Expect(0, 65380, '\P{^Is_Category=__Pe}', "");
Error('\p{General_Category= Letter:=}');
Error('\P{General_Category= Letter:=}');
Expect(1, 195101, '\p{General_Category:letter}', "");
Expect(0, 195101, '\p{^General_Category:letter}', "");
Expect(0, 195101, '\P{General_Category:letter}', "");
Expect(1, 195101, '\P{^General_Category:letter}', "");
Expect(0, 195102, '\p{General_Category:letter}', "");
Expect(1, 195102, '\p{^General_Category:letter}', "");
Expect(1, 195102, '\P{General_Category:letter}', "");
Expect(0, 195102, '\P{^General_Category:letter}', "");
Expect(1, 195101, '\p{General_Category=	LETTER}', "");
Expect(0, 195101, '\p{^General_Category=	LETTER}', "");
Expect(0, 195101, '\P{General_Category=	LETTER}', "");
Expect(1, 195101, '\P{^General_Category=	LETTER}', "");
Expect(0, 195102, '\p{General_Category=	LETTER}', "");
Expect(1, 195102, '\p{^General_Category=	LETTER}', "");
Expect(1, 195102, '\P{General_Category=	LETTER}', "");
Expect(0, 195102, '\P{^General_Category=	LETTER}', "");
Error('\p{Gc=:=	-l}');
Error('\P{Gc=:=	-l}');
Expect(1, 195101, '\p{Gc=l}', "");
Expect(0, 195101, '\p{^Gc=l}', "");
Expect(0, 195101, '\P{Gc=l}', "");
Expect(1, 195101, '\P{^Gc=l}', "");
Expect(0, 195102, '\p{Gc=l}', "");
Expect(1, 195102, '\p{^Gc=l}', "");
Expect(1, 195102, '\P{Gc=l}', "");
Expect(0, 195102, '\P{^Gc=l}', "");
Expect(1, 195101, '\p{Gc=	-L}', "");
Expect(0, 195101, '\p{^Gc=	-L}', "");
Expect(0, 195101, '\P{Gc=	-L}', "");
Expect(1, 195101, '\P{^Gc=	-L}', "");
Expect(0, 195102, '\p{Gc=	-L}', "");
Expect(1, 195102, '\p{^Gc=	-L}', "");
Expect(1, 195102, '\P{Gc=	-L}', "");
Expect(0, 195102, '\P{^Gc=	-L}', "");
Error('\p{Category=--LETTER:=}');
Error('\P{Category=--LETTER:=}');
Expect(1, 195101, '\p{Category=letter}', "");
Expect(0, 195101, '\p{^Category=letter}', "");
Expect(0, 195101, '\P{Category=letter}', "");
Expect(1, 195101, '\P{^Category=letter}', "");
Expect(0, 195102, '\p{Category=letter}', "");
Expect(1, 195102, '\p{^Category=letter}', "");
Expect(1, 195102, '\P{Category=letter}', "");
Expect(0, 195102, '\P{^Category=letter}', "");
Expect(1, 195101, '\p{Category= letter}', "");
Expect(0, 195101, '\p{^Category= letter}', "");
Expect(0, 195101, '\P{Category= letter}', "");
Expect(1, 195101, '\P{^Category= letter}', "");
Expect(0, 195102, '\p{Category= letter}', "");
Expect(1, 195102, '\p{^Category= letter}', "");
Expect(1, 195102, '\P{Category= letter}', "");
Expect(0, 195102, '\P{^Category= letter}', "");
Error('\p{Is_General_Category=:=	-L}');
Error('\P{Is_General_Category=:=	-L}');
Expect(1, 195101, '\p{Is_General_Category=l}', "");
Expect(0, 195101, '\p{^Is_General_Category=l}', "");
Expect(0, 195101, '\P{Is_General_Category=l}', "");
Expect(1, 195101, '\P{^Is_General_Category=l}', "");
Expect(0, 195102, '\p{Is_General_Category=l}', "");
Expect(1, 195102, '\p{^Is_General_Category=l}', "");
Expect(1, 195102, '\P{Is_General_Category=l}', "");
Expect(0, 195102, '\P{^Is_General_Category=l}', "");
Expect(1, 195101, '\p{Is_General_Category=--l}', "");
Expect(0, 195101, '\p{^Is_General_Category=--l}', "");
Expect(0, 195101, '\P{Is_General_Category=--l}', "");
Expect(1, 195101, '\P{^Is_General_Category=--l}', "");
Expect(0, 195102, '\p{Is_General_Category=--l}', "");
Expect(1, 195102, '\p{^Is_General_Category=--l}', "");
Expect(1, 195102, '\P{Is_General_Category=--l}', "");
Expect(0, 195102, '\P{^Is_General_Category=--l}', "");
Error('\p{Is_Gc= _Letter/a/}');
Error('\P{Is_Gc= _Letter/a/}');
Expect(1, 195101, '\p{Is_Gc=letter}', "");
Expect(0, 195101, '\p{^Is_Gc=letter}', "");
Expect(0, 195101, '\P{Is_Gc=letter}', "");
Expect(1, 195101, '\P{^Is_Gc=letter}', "");
Expect(0, 195102, '\p{Is_Gc=letter}', "");
Expect(1, 195102, '\p{^Is_Gc=letter}', "");
Expect(1, 195102, '\P{Is_Gc=letter}', "");
Expect(0, 195102, '\P{^Is_Gc=letter}', "");
Expect(1, 195101, '\p{Is_Gc:		LETTER}', "");
Expect(0, 195101, '\p{^Is_Gc:		LETTER}', "");
Expect(0, 195101, '\P{Is_Gc:		LETTER}', "");
Expect(1, 195101, '\P{^Is_Gc:		LETTER}', "");
Expect(0, 195102, '\p{Is_Gc:		LETTER}', "");
Expect(1, 195102, '\p{^Is_Gc:		LETTER}', "");
Expect(1, 195102, '\P{Is_Gc:		LETTER}', "");
Expect(0, 195102, '\P{^Is_Gc:		LETTER}', "");
Error('\p{Is_Category=/a/_L}');
Error('\P{Is_Category=/a/_L}');
Expect(1, 195101, '\p{Is_Category=l}', "");
Expect(0, 195101, '\p{^Is_Category=l}', "");
Expect(0, 195101, '\P{Is_Category=l}', "");
Expect(1, 195101, '\P{^Is_Category=l}', "");
Expect(0, 195102, '\p{Is_Category=l}', "");
Expect(1, 195102, '\p{^Is_Category=l}', "");
Expect(1, 195102, '\P{Is_Category=l}', "");
Expect(0, 195102, '\P{^Is_Category=l}', "");
Expect(1, 195101, '\p{Is_Category=	L}', "");
Expect(0, 195101, '\p{^Is_Category=	L}', "");
Expect(0, 195101, '\P{Is_Category=	L}', "");
Expect(1, 195101, '\P{^Is_Category=	L}', "");
Expect(0, 195102, '\p{Is_Category=	L}', "");
Expect(1, 195102, '\p{^Is_Category=	L}', "");
Expect(1, 195102, '\P{Is_Category=	L}', "");
Expect(0, 195102, '\P{^Is_Category=	L}', "");
Error('\p{General_Category=/a/-	MODIFIER_Symbol}');
Error('\P{General_Category=/a/-	MODIFIER_Symbol}');
Expect(1, 65507, '\p{General_Category=modifiersymbol}', "");
Expect(0, 65507, '\p{^General_Category=modifiersymbol}', "");
Expect(0, 65507, '\P{General_Category=modifiersymbol}', "");
Expect(1, 65507, '\P{^General_Category=modifiersymbol}', "");
Expect(0, 65508, '\p{General_Category=modifiersymbol}', "");
Expect(1, 65508, '\p{^General_Category=modifiersymbol}', "");
Expect(1, 65508, '\P{General_Category=modifiersymbol}', "");
Expect(0, 65508, '\P{^General_Category=modifiersymbol}', "");
Expect(1, 65507, '\p{General_Category:     Modifier_SYMBOL}', "");
Expect(0, 65507, '\p{^General_Category:     Modifier_SYMBOL}', "");
Expect(0, 65507, '\P{General_Category:     Modifier_SYMBOL}', "");
Expect(1, 65507, '\P{^General_Category:     Modifier_SYMBOL}', "");
Expect(0, 65508, '\p{General_Category:     Modifier_SYMBOL}', "");
Expect(1, 65508, '\p{^General_Category:     Modifier_SYMBOL}', "");
Expect(1, 65508, '\P{General_Category:     Modifier_SYMBOL}', "");
Expect(0, 65508, '\P{^General_Category:     Modifier_SYMBOL}', "");
Error('\p{Gc=:=Sk}');
Error('\P{Gc=:=Sk}');
Expect(1, 65507, '\p{Gc=sk}', "");
Expect(0, 65507, '\p{^Gc=sk}', "");
Expect(0, 65507, '\P{Gc=sk}', "");
Expect(1, 65507, '\P{^Gc=sk}', "");
Expect(0, 65508, '\p{Gc=sk}', "");
Expect(1, 65508, '\p{^Gc=sk}', "");
Expect(1, 65508, '\P{Gc=sk}', "");
Expect(0, 65508, '\P{^Gc=sk}', "");
Expect(1, 65507, '\p{Gc=-_Sk}', "");
Expect(0, 65507, '\p{^Gc=-_Sk}', "");
Expect(0, 65507, '\P{Gc=-_Sk}', "");
Expect(1, 65507, '\P{^Gc=-_Sk}', "");
Expect(0, 65508, '\p{Gc=-_Sk}', "");
Expect(1, 65508, '\p{^Gc=-_Sk}', "");
Expect(1, 65508, '\P{Gc=-_Sk}', "");
Expect(0, 65508, '\P{^Gc=-_Sk}', "");
Error('\p{Category=-/a/modifier_Symbol}');
Error('\P{Category=-/a/modifier_Symbol}');
Expect(1, 65507, '\p{Category=modifiersymbol}', "");
Expect(0, 65507, '\p{^Category=modifiersymbol}', "");
Expect(0, 65507, '\P{Category=modifiersymbol}', "");
Expect(1, 65507, '\P{^Category=modifiersymbol}', "");
Expect(0, 65508, '\p{Category=modifiersymbol}', "");
Expect(1, 65508, '\p{^Category=modifiersymbol}', "");
Expect(1, 65508, '\P{Category=modifiersymbol}', "");
Expect(0, 65508, '\P{^Category=modifiersymbol}', "");
Expect(1, 65507, '\p{Category=-_Modifier_SYMBOL}', "");
Expect(0, 65507, '\p{^Category=-_Modifier_SYMBOL}', "");
Expect(0, 65507, '\P{Category=-_Modifier_SYMBOL}', "");
Expect(1, 65507, '\P{^Category=-_Modifier_SYMBOL}', "");
Expect(0, 65508, '\p{Category=-_Modifier_SYMBOL}', "");
Expect(1, 65508, '\p{^Category=-_Modifier_SYMBOL}', "");
Expect(1, 65508, '\P{Category=-_Modifier_SYMBOL}', "");
Expect(0, 65508, '\P{^Category=-_Modifier_SYMBOL}', "");
Error('\p{Is_General_Category=/a/ Sk}');
Error('\P{Is_General_Category=/a/ Sk}');
Expect(1, 65507, '\p{Is_General_Category:sk}', "");
Expect(0, 65507, '\p{^Is_General_Category:sk}', "");
Expect(0, 65507, '\P{Is_General_Category:sk}', "");
Expect(1, 65507, '\P{^Is_General_Category:sk}', "");
Expect(0, 65508, '\p{Is_General_Category:sk}', "");
Expect(1, 65508, '\p{^Is_General_Category:sk}', "");
Expect(1, 65508, '\P{Is_General_Category:sk}', "");
Expect(0, 65508, '\P{^Is_General_Category:sk}', "");
Expect(1, 65507, '\p{Is_General_Category=__SK}', "");
Expect(0, 65507, '\p{^Is_General_Category=__SK}', "");
Expect(0, 65507, '\P{Is_General_Category=__SK}', "");
Expect(1, 65507, '\P{^Is_General_Category=__SK}', "");
Expect(0, 65508, '\p{Is_General_Category=__SK}', "");
Expect(1, 65508, '\p{^Is_General_Category=__SK}', "");
Expect(1, 65508, '\P{Is_General_Category=__SK}', "");
Expect(0, 65508, '\P{^Is_General_Category=__SK}', "");
Error('\p{Is_Gc=_-Modifier_Symbol:=}');
Error('\P{Is_Gc=_-Modifier_Symbol:=}');
Expect(1, 65507, '\p{Is_Gc=modifiersymbol}', "");
Expect(0, 65507, '\p{^Is_Gc=modifiersymbol}', "");
Expect(0, 65507, '\P{Is_Gc=modifiersymbol}', "");
Expect(1, 65507, '\P{^Is_Gc=modifiersymbol}', "");
Expect(0, 65508, '\p{Is_Gc=modifiersymbol}', "");
Expect(1, 65508, '\p{^Is_Gc=modifiersymbol}', "");
Expect(1, 65508, '\P{Is_Gc=modifiersymbol}', "");
Expect(0, 65508, '\P{^Is_Gc=modifiersymbol}', "");
Expect(1, 65507, '\p{Is_Gc=__Modifier_Symbol}', "");
Expect(0, 65507, '\p{^Is_Gc=__Modifier_Symbol}', "");
Expect(0, 65507, '\P{Is_Gc=__Modifier_Symbol}', "");
Expect(1, 65507, '\P{^Is_Gc=__Modifier_Symbol}', "");
Expect(0, 65508, '\p{Is_Gc=__Modifier_Symbol}', "");
Expect(1, 65508, '\p{^Is_Gc=__Modifier_Symbol}', "");
Expect(1, 65508, '\P{Is_Gc=__Modifier_Symbol}', "");
Expect(0, 65508, '\P{^Is_Gc=__Modifier_Symbol}', "");
Error('\p{Is_Category=-_SK:=}');
Error('\P{Is_Category=-_SK:=}');
Expect(1, 65507, '\p{Is_Category=sk}', "");
Expect(0, 65507, '\p{^Is_Category=sk}', "");
Expect(0, 65507, '\P{Is_Category=sk}', "");
Expect(1, 65507, '\P{^Is_Category=sk}', "");
Expect(0, 65508, '\p{Is_Category=sk}', "");
Expect(1, 65508, '\p{^Is_Category=sk}', "");
Expect(1, 65508, '\P{Is_Category=sk}', "");
Expect(0, 65508, '\P{^Is_Category=sk}', "");
Expect(1, 65507, '\p{Is_Category=-	Sk}', "");
Expect(0, 65507, '\p{^Is_Category=-	Sk}', "");
Expect(0, 65507, '\P{Is_Category=-	Sk}', "");
Expect(1, 65507, '\P{^Is_Category=-	Sk}', "");
Expect(0, 65508, '\p{Is_Category=-	Sk}', "");
Expect(1, 65508, '\p{^Is_Category=-	Sk}', "");
Expect(1, 65508, '\P{Is_Category=-	Sk}', "");
Expect(0, 65508, '\P{^Is_Category=-	Sk}', "");
Error('\p{General_Category=:=-Mark}');
Error('\P{General_Category=:=-Mark}');
Expect(1, 917999, '\p{General_Category=mark}', "");
Expect(0, 917999, '\p{^General_Category=mark}', "");
Expect(0, 917999, '\P{General_Category=mark}', "");
Expect(1, 917999, '\P{^General_Category=mark}', "");
Expect(0, 918000, '\p{General_Category=mark}', "");
Expect(1, 918000, '\p{^General_Category=mark}', "");
Expect(1, 918000, '\P{General_Category=mark}', "");
Expect(0, 918000, '\P{^General_Category=mark}', "");
Expect(1, 917999, '\p{General_Category:  _MARK}', "");
Expect(0, 917999, '\p{^General_Category:  _MARK}', "");
Expect(0, 917999, '\P{General_Category:  _MARK}', "");
Expect(1, 917999, '\P{^General_Category:  _MARK}', "");
Expect(0, 918000, '\p{General_Category:  _MARK}', "");
Expect(1, 918000, '\p{^General_Category:  _MARK}', "");
Expect(1, 918000, '\P{General_Category:  _MARK}', "");
Expect(0, 918000, '\P{^General_Category:  _MARK}', "");
Error('\p{Gc=:=	-M}');
Error('\P{Gc=:=	-M}');
Expect(1, 917999, '\p{Gc=m}', "");
Expect(0, 917999, '\p{^Gc=m}', "");
Expect(0, 917999, '\P{Gc=m}', "");
Expect(1, 917999, '\P{^Gc=m}', "");
Expect(0, 918000, '\p{Gc=m}', "");
Expect(1, 918000, '\p{^Gc=m}', "");
Expect(1, 918000, '\P{Gc=m}', "");
Expect(0, 918000, '\P{^Gc=m}', "");
Expect(1, 917999, '\p{Gc=--m}', "");
Expect(0, 917999, '\p{^Gc=--m}', "");
Expect(0, 917999, '\P{Gc=--m}', "");
Expect(1, 917999, '\P{^Gc=--m}', "");
Expect(0, 918000, '\p{Gc=--m}', "");
Expect(1, 918000, '\p{^Gc=--m}', "");
Expect(1, 918000, '\P{Gc=--m}', "");
Expect(0, 918000, '\P{^Gc=--m}', "");
Error('\p{Category=_/a/Combining_mark}');
Error('\P{Category=_/a/Combining_mark}');
Expect(1, 917999, '\p{Category=combiningmark}', "");
Expect(0, 917999, '\p{^Category=combiningmark}', "");
Expect(0, 917999, '\P{Category=combiningmark}', "");
Expect(1, 917999, '\P{^Category=combiningmark}', "");
Expect(0, 918000, '\p{Category=combiningmark}', "");
Expect(1, 918000, '\p{^Category=combiningmark}', "");
Expect(1, 918000, '\P{Category=combiningmark}', "");
Expect(0, 918000, '\P{^Category=combiningmark}', "");
Expect(1, 917999, '\p{Category=- COMBINING_MARK}', "");
Expect(0, 917999, '\p{^Category=- COMBINING_MARK}', "");
Expect(0, 917999, '\P{Category=- COMBINING_MARK}', "");
Expect(1, 917999, '\P{^Category=- COMBINING_MARK}', "");
Expect(0, 918000, '\p{Category=- COMBINING_MARK}', "");
Expect(1, 918000, '\p{^Category=- COMBINING_MARK}', "");
Expect(1, 918000, '\P{Category=- COMBINING_MARK}', "");
Expect(0, 918000, '\P{^Category=- COMBINING_MARK}', "");
Error('\p{Is_General_Category=	mark:=}');
Error('\P{Is_General_Category=	mark:=}');
Expect(1, 917999, '\p{Is_General_Category=mark}', "");
Expect(0, 917999, '\p{^Is_General_Category=mark}', "");
Expect(0, 917999, '\P{Is_General_Category=mark}', "");
Expect(1, 917999, '\P{^Is_General_Category=mark}', "");
Expect(0, 918000, '\p{Is_General_Category=mark}', "");
Expect(1, 918000, '\p{^Is_General_Category=mark}', "");
Expect(1, 918000, '\P{Is_General_Category=mark}', "");
Expect(0, 918000, '\P{^Is_General_Category=mark}', "");
Expect(1, 917999, '\p{Is_General_Category=	mark}', "");
Expect(0, 917999, '\p{^Is_General_Category=	mark}', "");
Expect(0, 917999, '\P{Is_General_Category=	mark}', "");
Expect(1, 917999, '\P{^Is_General_Category=	mark}', "");
Expect(0, 918000, '\p{Is_General_Category=	mark}', "");
Expect(1, 918000, '\p{^Is_General_Category=	mark}', "");
Expect(1, 918000, '\P{Is_General_Category=	mark}', "");
Expect(0, 918000, '\P{^Is_General_Category=	mark}', "");
Error('\p{Is_Gc=	 M/a/}');
Error('\P{Is_Gc=	 M/a/}');
Expect(1, 917999, '\p{Is_Gc=m}', "");
Expect(0, 917999, '\p{^Is_Gc=m}', "");
Expect(0, 917999, '\P{Is_Gc=m}', "");
Expect(1, 917999, '\P{^Is_Gc=m}', "");
Expect(0, 918000, '\p{Is_Gc=m}', "");
Expect(1, 918000, '\p{^Is_Gc=m}', "");
Expect(1, 918000, '\P{Is_Gc=m}', "");
Expect(0, 918000, '\P{^Is_Gc=m}', "");
Expect(1, 917999, '\p{Is_Gc=M}', "");
Expect(0, 917999, '\p{^Is_Gc=M}', "");
Expect(0, 917999, '\P{Is_Gc=M}', "");
Expect(1, 917999, '\P{^Is_Gc=M}', "");
Expect(0, 918000, '\p{Is_Gc=M}', "");
Expect(1, 918000, '\p{^Is_Gc=M}', "");
Expect(1, 918000, '\P{Is_Gc=M}', "");
Expect(0, 918000, '\P{^Is_Gc=M}', "");
Error('\p{Is_Category:    COMBINING_MARK:=}');
Error('\P{Is_Category:    COMBINING_MARK:=}');
Expect(1, 917999, '\p{Is_Category=combiningmark}', "");
Expect(0, 917999, '\p{^Is_Category=combiningmark}', "");
Expect(0, 917999, '\P{Is_Category=combiningmark}', "");
Expect(1, 917999, '\P{^Is_Category=combiningmark}', "");
Expect(0, 918000, '\p{Is_Category=combiningmark}', "");
Expect(1, 918000, '\p{^Is_Category=combiningmark}', "");
Expect(1, 918000, '\P{Is_Category=combiningmark}', "");
Expect(0, 918000, '\P{^Is_Category=combiningmark}', "");
Expect(1, 917999, '\p{Is_Category=__combining_Mark}', "");
Expect(0, 917999, '\p{^Is_Category=__combining_Mark}', "");
Expect(0, 917999, '\P{Is_Category=__combining_Mark}', "");
Expect(1, 917999, '\P{^Is_Category=__combining_Mark}', "");
Expect(0, 918000, '\p{Is_Category=__combining_Mark}', "");
Expect(1, 918000, '\p{^Is_Category=__combining_Mark}', "");
Expect(1, 918000, '\P{Is_Category=__combining_Mark}', "");
Expect(0, 918000, '\P{^Is_Category=__combining_Mark}', "");
Error('\p{General_Category=	/a/control}');
Error('\P{General_Category=	/a/control}');
Expect(1, 159, '\p{General_Category=control}', "");
Expect(0, 159, '\p{^General_Category=control}', "");
Expect(0, 159, '\P{General_Category=control}', "");
Expect(1, 159, '\P{^General_Category=control}', "");
Expect(0, 160, '\p{General_Category=control}', "");
Expect(1, 160, '\p{^General_Category=control}', "");
Expect(1, 160, '\P{General_Category=control}', "");
Expect(0, 160, '\P{^General_Category=control}', "");
Expect(1, 159, '\p{General_Category=-Control}', "");
Expect(0, 159, '\p{^General_Category=-Control}', "");
Expect(0, 159, '\P{General_Category=-Control}', "");
Expect(1, 159, '\P{^General_Category=-Control}', "");
Expect(0, 160, '\p{General_Category=-Control}', "");
Expect(1, 160, '\p{^General_Category=-Control}', "");
Expect(1, 160, '\P{General_Category=-Control}', "");
Expect(0, 160, '\P{^General_Category=-Control}', "");
Error('\p{Gc= 	Cc:=}');
Error('\P{Gc= 	Cc:=}');
Expect(1, 159, '\p{Gc=cc}', "");
Expect(0, 159, '\p{^Gc=cc}', "");
Expect(0, 159, '\P{Gc=cc}', "");
Expect(1, 159, '\P{^Gc=cc}', "");
Expect(0, 160, '\p{Gc=cc}', "");
Expect(1, 160, '\p{^Gc=cc}', "");
Expect(1, 160, '\P{Gc=cc}', "");
Expect(0, 160, '\P{^Gc=cc}', "");
Expect(1, 159, '\p{Gc=- Cc}', "");
Expect(0, 159, '\p{^Gc=- Cc}', "");
Expect(0, 159, '\P{Gc=- Cc}', "");
Expect(1, 159, '\P{^Gc=- Cc}', "");
Expect(0, 160, '\p{Gc=- Cc}', "");
Expect(1, 160, '\p{^Gc=- Cc}', "");
Expect(1, 160, '\P{Gc=- Cc}', "");
Expect(0, 160, '\P{^Gc=- Cc}', "");
Error('\p{Category=_/a/Cntrl}');
Error('\P{Category=_/a/Cntrl}');
Expect(1, 159, '\p{Category=cntrl}', "");
Expect(0, 159, '\p{^Category=cntrl}', "");
Expect(0, 159, '\P{Category=cntrl}', "");
Expect(1, 159, '\P{^Category=cntrl}', "");
Expect(0, 160, '\p{Category=cntrl}', "");
Expect(1, 160, '\p{^Category=cntrl}', "");
Expect(1, 160, '\P{Category=cntrl}', "");
Expect(0, 160, '\P{^Category=cntrl}', "");
Expect(1, 159, '\p{Category=_CNTRL}', "");
Expect(0, 159, '\p{^Category=_CNTRL}', "");
Expect(0, 159, '\P{Category=_CNTRL}', "");
Expect(1, 159, '\P{^Category=_CNTRL}', "");
Expect(0, 160, '\p{Category=_CNTRL}', "");
Expect(1, 160, '\p{^Category=_CNTRL}', "");
Expect(1, 160, '\P{Category=_CNTRL}', "");
Expect(0, 160, '\P{^Category=_CNTRL}', "");
Error('\p{Is_General_Category= /a/Control}');
Error('\P{Is_General_Category= /a/Control}');
Expect(1, 159, '\p{Is_General_Category: control}', "");
Expect(0, 159, '\p{^Is_General_Category: control}', "");
Expect(0, 159, '\P{Is_General_Category: control}', "");
Expect(1, 159, '\P{^Is_General_Category: control}', "");
Expect(0, 160, '\p{Is_General_Category: control}', "");
Expect(1, 160, '\p{^Is_General_Category: control}', "");
Expect(1, 160, '\P{Is_General_Category: control}', "");
Expect(0, 160, '\P{^Is_General_Category: control}', "");
Expect(1, 159, '\p{Is_General_Category:     Control}', "");
Expect(0, 159, '\p{^Is_General_Category:     Control}', "");
Expect(0, 159, '\P{Is_General_Category:     Control}', "");
Expect(1, 159, '\P{^Is_General_Category:     Control}', "");
Expect(0, 160, '\p{Is_General_Category:     Control}', "");
Expect(1, 160, '\p{^Is_General_Category:     Control}', "");
Expect(1, 160, '\P{Is_General_Category:     Control}', "");
Expect(0, 160, '\P{^Is_General_Category:     Control}', "");
Error('\p{Is_Gc= Cc/a/}');
Error('\P{Is_Gc= Cc/a/}');
Expect(1, 159, '\p{Is_Gc=cc}', "");
Expect(0, 159, '\p{^Is_Gc=cc}', "");
Expect(0, 159, '\P{Is_Gc=cc}', "");
Expect(1, 159, '\P{^Is_Gc=cc}', "");
Expect(0, 160, '\p{Is_Gc=cc}', "");
Expect(1, 160, '\p{^Is_Gc=cc}', "");
Expect(1, 160, '\P{Is_Gc=cc}', "");
Expect(0, 160, '\P{^Is_Gc=cc}', "");
Expect(1, 159, '\p{Is_Gc=-CC}', "");
Expect(0, 159, '\p{^Is_Gc=-CC}', "");
Expect(0, 159, '\P{Is_Gc=-CC}', "");
Expect(1, 159, '\P{^Is_Gc=-CC}', "");
Expect(0, 160, '\p{Is_Gc=-CC}', "");
Expect(1, 160, '\p{^Is_Gc=-CC}', "");
Expect(1, 160, '\P{Is_Gc=-CC}', "");
Expect(0, 160, '\P{^Is_Gc=-CC}', "");
Error('\p{Is_Category=/a/ 	Cntrl}');
Error('\P{Is_Category=/a/ 	Cntrl}');
Expect(1, 159, '\p{Is_Category=cntrl}', "");
Expect(0, 159, '\p{^Is_Category=cntrl}', "");
Expect(0, 159, '\P{Is_Category=cntrl}', "");
Expect(1, 159, '\P{^Is_Category=cntrl}', "");
Expect(0, 160, '\p{Is_Category=cntrl}', "");
Expect(1, 160, '\p{^Is_Category=cntrl}', "");
Expect(1, 160, '\P{Is_Category=cntrl}', "");
Expect(0, 160, '\P{^Is_Category=cntrl}', "");
Expect(1, 159, '\p{Is_Category=_Cntrl}', "");
Expect(0, 159, '\p{^Is_Category=_Cntrl}', "");
Expect(0, 159, '\P{Is_Category=_Cntrl}', "");
Expect(1, 159, '\P{^Is_Category=_Cntrl}', "");
Expect(0, 160, '\p{Is_Category=_Cntrl}', "");
Expect(1, 160, '\p{^Is_Category=_Cntrl}', "");
Expect(1, 160, '\P{Is_Category=_Cntrl}', "");
Expect(0, 160, '\P{^Is_Category=_Cntrl}', "");
Error('\p{General_Category=		LINE_Separator:=}');
Error('\P{General_Category=		LINE_Separator:=}');
Expect(1, 8232, '\p{General_Category=lineseparator}', "");
Expect(0, 8232, '\p{^General_Category=lineseparator}', "");
Expect(0, 8232, '\P{General_Category=lineseparator}', "");
Expect(1, 8232, '\P{^General_Category=lineseparator}', "");
Expect(0, 8233, '\p{General_Category=lineseparator}', "");
Expect(1, 8233, '\p{^General_Category=lineseparator}', "");
Expect(1, 8233, '\P{General_Category=lineseparator}', "");
Expect(0, 8233, '\P{^General_Category=lineseparator}', "");
Expect(1, 8232, '\p{General_Category=_	Line_Separator}', "");
Expect(0, 8232, '\p{^General_Category=_	Line_Separator}', "");
Expect(0, 8232, '\P{General_Category=_	Line_Separator}', "");
Expect(1, 8232, '\P{^General_Category=_	Line_Separator}', "");
Expect(0, 8233, '\p{General_Category=_	Line_Separator}', "");
Expect(1, 8233, '\p{^General_Category=_	Line_Separator}', "");
Expect(1, 8233, '\P{General_Category=_	Line_Separator}', "");
Expect(0, 8233, '\P{^General_Category=_	Line_Separator}', "");
Error('\p{Gc=/a/		zl}');
Error('\P{Gc=/a/		zl}');
Expect(1, 8232, '\p{Gc=zl}', "");
Expect(0, 8232, '\p{^Gc=zl}', "");
Expect(0, 8232, '\P{Gc=zl}', "");
Expect(1, 8232, '\P{^Gc=zl}', "");
Expect(0, 8233, '\p{Gc=zl}', "");
Expect(1, 8233, '\p{^Gc=zl}', "");
Expect(1, 8233, '\P{Gc=zl}', "");
Expect(0, 8233, '\P{^Gc=zl}', "");
Expect(1, 8232, '\p{Gc: -ZL}', "");
Expect(0, 8232, '\p{^Gc: -ZL}', "");
Expect(0, 8232, '\P{Gc: -ZL}', "");
Expect(1, 8232, '\P{^Gc: -ZL}', "");
Expect(0, 8233, '\p{Gc: -ZL}', "");
Expect(1, 8233, '\p{^Gc: -ZL}', "");
Expect(1, 8233, '\P{Gc: -ZL}', "");
Expect(0, 8233, '\P{^Gc: -ZL}', "");
Error('\p{Category=/a/__LINE_separator}');
Error('\P{Category=/a/__LINE_separator}');
Expect(1, 8232, '\p{Category=lineseparator}', "");
Expect(0, 8232, '\p{^Category=lineseparator}', "");
Expect(0, 8232, '\P{Category=lineseparator}', "");
Expect(1, 8232, '\P{^Category=lineseparator}', "");
Expect(0, 8233, '\p{Category=lineseparator}', "");
Expect(1, 8233, '\p{^Category=lineseparator}', "");
Expect(1, 8233, '\P{Category=lineseparator}', "");
Expect(0, 8233, '\P{^Category=lineseparator}', "");
Expect(1, 8232, '\p{Category=- Line_Separator}', "");
Expect(0, 8232, '\p{^Category=- Line_Separator}', "");
Expect(0, 8232, '\P{Category=- Line_Separator}', "");
Expect(1, 8232, '\P{^Category=- Line_Separator}', "");
Expect(0, 8233, '\p{Category=- Line_Separator}', "");
Expect(1, 8233, '\p{^Category=- Line_Separator}', "");
Expect(1, 8233, '\P{Category=- Line_Separator}', "");
Expect(0, 8233, '\P{^Category=- Line_Separator}', "");
Error('\p{Is_General_Category=:=_-Zl}');
Error('\P{Is_General_Category=:=_-Zl}');
Expect(1, 8232, '\p{Is_General_Category=zl}', "");
Expect(0, 8232, '\p{^Is_General_Category=zl}', "");
Expect(0, 8232, '\P{Is_General_Category=zl}', "");
Expect(1, 8232, '\P{^Is_General_Category=zl}', "");
Expect(0, 8233, '\p{Is_General_Category=zl}', "");
Expect(1, 8233, '\p{^Is_General_Category=zl}', "");
Expect(1, 8233, '\P{Is_General_Category=zl}', "");
Expect(0, 8233, '\P{^Is_General_Category=zl}', "");
Expect(1, 8232, '\p{Is_General_Category=Zl}', "");
Expect(0, 8232, '\p{^Is_General_Category=Zl}', "");
Expect(0, 8232, '\P{Is_General_Category=Zl}', "");
Expect(1, 8232, '\P{^Is_General_Category=Zl}', "");
Expect(0, 8233, '\p{Is_General_Category=Zl}', "");
Expect(1, 8233, '\p{^Is_General_Category=Zl}', "");
Expect(1, 8233, '\P{Is_General_Category=Zl}', "");
Expect(0, 8233, '\P{^Is_General_Category=Zl}', "");
Error('\p{Is_Gc=/a/-line_Separator}');
Error('\P{Is_Gc=/a/-line_Separator}');
Expect(1, 8232, '\p{Is_Gc=lineseparator}', "");
Expect(0, 8232, '\p{^Is_Gc=lineseparator}', "");
Expect(0, 8232, '\P{Is_Gc=lineseparator}', "");
Expect(1, 8232, '\P{^Is_Gc=lineseparator}', "");
Expect(0, 8233, '\p{Is_Gc=lineseparator}', "");
Expect(1, 8233, '\p{^Is_Gc=lineseparator}', "");
Expect(1, 8233, '\P{Is_Gc=lineseparator}', "");
Expect(0, 8233, '\P{^Is_Gc=lineseparator}', "");
Expect(1, 8232, '\p{Is_Gc:	-Line_SEPARATOR}', "");
Expect(0, 8232, '\p{^Is_Gc:	-Line_SEPARATOR}', "");
Expect(0, 8232, '\P{Is_Gc:	-Line_SEPARATOR}', "");
Expect(1, 8232, '\P{^Is_Gc:	-Line_SEPARATOR}', "");
Expect(0, 8233, '\p{Is_Gc:	-Line_SEPARATOR}', "");
Expect(1, 8233, '\p{^Is_Gc:	-Line_SEPARATOR}', "");
Expect(1, 8233, '\P{Is_Gc:	-Line_SEPARATOR}', "");
Expect(0, 8233, '\P{^Is_Gc:	-Line_SEPARATOR}', "");
Error('\p{Is_Category=:=_zl}');
Error('\P{Is_Category=:=_zl}');
Expect(1, 8232, '\p{Is_Category: zl}', "");
Expect(0, 8232, '\p{^Is_Category: zl}', "");
Expect(0, 8232, '\P{Is_Category: zl}', "");
Expect(1, 8232, '\P{^Is_Category: zl}', "");
Expect(0, 8233, '\p{Is_Category: zl}', "");
Expect(1, 8233, '\p{^Is_Category: zl}', "");
Expect(1, 8233, '\P{Is_Category: zl}', "");
Expect(0, 8233, '\P{^Is_Category: zl}', "");
Expect(1, 8232, '\p{Is_Category=-Zl}', "");
Expect(0, 8232, '\p{^Is_Category=-Zl}', "");
Expect(0, 8232, '\P{Is_Category=-Zl}', "");
Expect(1, 8232, '\P{^Is_Category=-Zl}', "");
Expect(0, 8233, '\p{Is_Category=-Zl}', "");
Expect(1, 8233, '\p{^Is_Category=-Zl}', "");
Expect(1, 8233, '\P{Is_Category=-Zl}', "");
Expect(0, 8233, '\P{^Is_Category=-Zl}', "");
Error('\p{General_Category=:=- Open_Punctuation}');
Error('\P{General_Category=:=- Open_Punctuation}');
Expect(1, 65378, '\p{General_Category=openpunctuation}', "");
Expect(0, 65378, '\p{^General_Category=openpunctuation}', "");
Expect(0, 65378, '\P{General_Category=openpunctuation}', "");
Expect(1, 65378, '\P{^General_Category=openpunctuation}', "");
Expect(0, 65379, '\p{General_Category=openpunctuation}', "");
Expect(1, 65379, '\p{^General_Category=openpunctuation}', "");
Expect(1, 65379, '\P{General_Category=openpunctuation}', "");
Expect(0, 65379, '\P{^General_Category=openpunctuation}', "");
Expect(1, 65378, '\p{General_Category:  -Open_Punctuation}', "");
Expect(0, 65378, '\p{^General_Category:  -Open_Punctuation}', "");
Expect(0, 65378, '\P{General_Category:  -Open_Punctuation}', "");
Expect(1, 65378, '\P{^General_Category:  -Open_Punctuation}', "");
Expect(0, 65379, '\p{General_Category:  -Open_Punctuation}', "");
Expect(1, 65379, '\p{^General_Category:  -Open_Punctuation}', "");
Expect(1, 65379, '\P{General_Category:  -Open_Punctuation}', "");
Expect(0, 65379, '\P{^General_Category:  -Open_Punctuation}', "");
Error('\p{Gc=/a/PS}');
Error('\P{Gc=/a/PS}');
Expect(1, 65378, '\p{Gc=ps}', "");
Expect(0, 65378, '\p{^Gc=ps}', "");
Expect(0, 65378, '\P{Gc=ps}', "");
Expect(1, 65378, '\P{^Gc=ps}', "");
Expect(0, 65379, '\p{Gc=ps}', "");
Expect(1, 65379, '\p{^Gc=ps}', "");
Expect(1, 65379, '\P{Gc=ps}', "");
Expect(0, 65379, '\P{^Gc=ps}', "");
Expect(1, 65378, '\p{Gc=--Ps}', "");
Expect(0, 65378, '\p{^Gc=--Ps}', "");
Expect(0, 65378, '\P{Gc=--Ps}', "");
Expect(1, 65378, '\P{^Gc=--Ps}', "");
Expect(0, 65379, '\p{Gc=--Ps}', "");
Expect(1, 65379, '\p{^Gc=--Ps}', "");
Expect(1, 65379, '\P{Gc=--Ps}', "");
Expect(0, 65379, '\P{^Gc=--Ps}', "");
Error('\p{Category= _open_Punctuation:=}');
Error('\P{Category= _open_Punctuation:=}');
Expect(1, 65378, '\p{Category=openpunctuation}', "");
Expect(0, 65378, '\p{^Category=openpunctuation}', "");
Expect(0, 65378, '\P{Category=openpunctuation}', "");
Expect(1, 65378, '\P{^Category=openpunctuation}', "");
Expect(0, 65379, '\p{Category=openpunctuation}', "");
Expect(1, 65379, '\p{^Category=openpunctuation}', "");
Expect(1, 65379, '\P{Category=openpunctuation}', "");
Expect(0, 65379, '\P{^Category=openpunctuation}', "");
Expect(1, 65378, '\p{Category= 	Open_Punctuation}', "");
Expect(0, 65378, '\p{^Category= 	Open_Punctuation}', "");
Expect(0, 65378, '\P{Category= 	Open_Punctuation}', "");
Expect(1, 65378, '\P{^Category= 	Open_Punctuation}', "");
Expect(0, 65379, '\p{Category= 	Open_Punctuation}', "");
Expect(1, 65379, '\p{^Category= 	Open_Punctuation}', "");
Expect(1, 65379, '\P{Category= 	Open_Punctuation}', "");
Expect(0, 65379, '\P{^Category= 	Open_Punctuation}', "");
Error('\p{Is_General_Category=	ps:=}');
Error('\P{Is_General_Category=	ps:=}');
Expect(1, 65378, '\p{Is_General_Category=ps}', "");
Expect(0, 65378, '\p{^Is_General_Category=ps}', "");
Expect(0, 65378, '\P{Is_General_Category=ps}', "");
Expect(1, 65378, '\P{^Is_General_Category=ps}', "");
Expect(0, 65379, '\p{Is_General_Category=ps}', "");
Expect(1, 65379, '\p{^Is_General_Category=ps}', "");
Expect(1, 65379, '\P{Is_General_Category=ps}', "");
Expect(0, 65379, '\P{^Is_General_Category=ps}', "");
Expect(1, 65378, '\p{Is_General_Category= _PS}', "");
Expect(0, 65378, '\p{^Is_General_Category= _PS}', "");
Expect(0, 65378, '\P{Is_General_Category= _PS}', "");
Expect(1, 65378, '\P{^Is_General_Category= _PS}', "");
Expect(0, 65379, '\p{Is_General_Category= _PS}', "");
Expect(1, 65379, '\p{^Is_General_Category= _PS}', "");
Expect(1, 65379, '\P{Is_General_Category= _PS}', "");
Expect(0, 65379, '\P{^Is_General_Category= _PS}', "");
Error('\p{Is_Gc:		:=Open_PUNCTUATION}');
Error('\P{Is_Gc:		:=Open_PUNCTUATION}');
Expect(1, 65378, '\p{Is_Gc=openpunctuation}', "");
Expect(0, 65378, '\p{^Is_Gc=openpunctuation}', "");
Expect(0, 65378, '\P{Is_Gc=openpunctuation}', "");
Expect(1, 65378, '\P{^Is_Gc=openpunctuation}', "");
Expect(0, 65379, '\p{Is_Gc=openpunctuation}', "");
Expect(1, 65379, '\p{^Is_Gc=openpunctuation}', "");
Expect(1, 65379, '\P{Is_Gc=openpunctuation}', "");
Expect(0, 65379, '\P{^Is_Gc=openpunctuation}', "");
Expect(1, 65378, '\p{Is_Gc: -Open_PUNCTUATION}', "");
Expect(0, 65378, '\p{^Is_Gc: -Open_PUNCTUATION}', "");
Expect(0, 65378, '\P{Is_Gc: -Open_PUNCTUATION}', "");
Expect(1, 65378, '\P{^Is_Gc: -Open_PUNCTUATION}', "");
Expect(0, 65379, '\p{Is_Gc: -Open_PUNCTUATION}', "");
Expect(1, 65379, '\p{^Is_Gc: -Open_PUNCTUATION}', "");
Expect(1, 65379, '\P{Is_Gc: -Open_PUNCTUATION}', "");
Expect(0, 65379, '\P{^Is_Gc: -Open_PUNCTUATION}', "");
Error('\p{Is_Category= :=PS}');
Error('\P{Is_Category= :=PS}');
Expect(1, 65378, '\p{Is_Category=ps}', "");
Expect(0, 65378, '\p{^Is_Category=ps}', "");
Expect(0, 65378, '\P{Is_Category=ps}', "");
Expect(1, 65378, '\P{^Is_Category=ps}', "");
Expect(0, 65379, '\p{Is_Category=ps}', "");
Expect(1, 65379, '\p{^Is_Category=ps}', "");
Expect(1, 65379, '\P{Is_Category=ps}', "");
Expect(0, 65379, '\P{^Is_Category=ps}', "");
Expect(1, 65378, '\p{Is_Category=	 ps}', "");
Expect(0, 65378, '\p{^Is_Category=	 ps}', "");
Expect(0, 65378, '\P{Is_Category=	 ps}', "");
Expect(1, 65378, '\P{^Is_Category=	 ps}', "");
Expect(0, 65379, '\p{Is_Category=	 ps}', "");
Expect(1, 65379, '\p{^Is_Category=	 ps}', "");
Expect(1, 65379, '\P{Is_Category=	 ps}', "");
Expect(0, 65379, '\P{^Is_Category=	 ps}', "");
Error('\p{General_Category=_-MATH_SYMBOL/a/}');
Error('\P{General_Category=_-MATH_SYMBOL/a/}');
Expect(1, 126705, '\p{General_Category=mathsymbol}', "");
Expect(0, 126705, '\p{^General_Category=mathsymbol}', "");
Expect(0, 126705, '\P{General_Category=mathsymbol}', "");
Expect(1, 126705, '\P{^General_Category=mathsymbol}', "");
Expect(0, 126706, '\p{General_Category=mathsymbol}', "");
Expect(1, 126706, '\p{^General_Category=mathsymbol}', "");
Expect(1, 126706, '\P{General_Category=mathsymbol}', "");
Expect(0, 126706, '\P{^General_Category=mathsymbol}', "");
Expect(1, 126705, '\p{General_Category= _Math_Symbol}', "");
Expect(0, 126705, '\p{^General_Category= _Math_Symbol}', "");
Expect(0, 126705, '\P{General_Category= _Math_Symbol}', "");
Expect(1, 126705, '\P{^General_Category= _Math_Symbol}', "");
Expect(0, 126706, '\p{General_Category= _Math_Symbol}', "");
Expect(1, 126706, '\p{^General_Category= _Math_Symbol}', "");
Expect(1, 126706, '\P{General_Category= _Math_Symbol}', "");
Expect(0, 126706, '\P{^General_Category= _Math_Symbol}', "");
Error('\p{Gc=_	sm:=}');
Error('\P{Gc=_	sm:=}');
Expect(1, 126705, '\p{Gc=sm}', "");
Expect(0, 126705, '\p{^Gc=sm}', "");
Expect(0, 126705, '\P{Gc=sm}', "");
Expect(1, 126705, '\P{^Gc=sm}', "");
Expect(0, 126706, '\p{Gc=sm}', "");
Expect(1, 126706, '\p{^Gc=sm}', "");
Expect(1, 126706, '\P{Gc=sm}', "");
Expect(0, 126706, '\P{^Gc=sm}', "");
Expect(1, 126705, '\p{Gc=	-SM}', "");
Expect(0, 126705, '\p{^Gc=	-SM}', "");
Expect(0, 126705, '\P{Gc=	-SM}', "");
Expect(1, 126705, '\P{^Gc=	-SM}', "");
Expect(0, 126706, '\p{Gc=	-SM}', "");
Expect(1, 126706, '\p{^Gc=	-SM}', "");
Expect(1, 126706, '\P{Gc=	-SM}', "");
Expect(0, 126706, '\P{^Gc=	-SM}', "");
Error('\p{Category= Math_Symbol:=}');
Error('\P{Category= Math_Symbol:=}');
Expect(1, 126705, '\p{Category=mathsymbol}', "");
Expect(0, 126705, '\p{^Category=mathsymbol}', "");
Expect(0, 126705, '\P{Category=mathsymbol}', "");
Expect(1, 126705, '\P{^Category=mathsymbol}', "");
Expect(0, 126706, '\p{Category=mathsymbol}', "");
Expect(1, 126706, '\p{^Category=mathsymbol}', "");
Expect(1, 126706, '\P{Category=mathsymbol}', "");
Expect(0, 126706, '\P{^Category=mathsymbol}', "");
Expect(1, 126705, '\p{Category:	_Math_symbol}', "");
Expect(0, 126705, '\p{^Category:	_Math_symbol}', "");
Expect(0, 126705, '\P{Category:	_Math_symbol}', "");
Expect(1, 126705, '\P{^Category:	_Math_symbol}', "");
Expect(0, 126706, '\p{Category:	_Math_symbol}', "");
Expect(1, 126706, '\p{^Category:	_Math_symbol}', "");
Expect(1, 126706, '\P{Category:	_Math_symbol}', "");
Expect(0, 126706, '\P{^Category:	_Math_symbol}', "");
Error('\p{Is_General_Category=/a/--SM}');
Error('\P{Is_General_Category=/a/--SM}');
Expect(1, 126705, '\p{Is_General_Category=sm}', "");
Expect(0, 126705, '\p{^Is_General_Category=sm}', "");
Expect(0, 126705, '\P{Is_General_Category=sm}', "");
Expect(1, 126705, '\P{^Is_General_Category=sm}', "");
Expect(0, 126706, '\p{Is_General_Category=sm}', "");
Expect(1, 126706, '\p{^Is_General_Category=sm}', "");
Expect(1, 126706, '\P{Is_General_Category=sm}', "");
Expect(0, 126706, '\P{^Is_General_Category=sm}', "");
Expect(1, 126705, '\p{Is_General_Category=	sm}', "");
Expect(0, 126705, '\p{^Is_General_Category=	sm}', "");
Expect(0, 126705, '\P{Is_General_Category=	sm}', "");
Expect(1, 126705, '\P{^Is_General_Category=	sm}', "");
Expect(0, 126706, '\p{Is_General_Category=	sm}', "");
Expect(1, 126706, '\p{^Is_General_Category=	sm}', "");
Expect(1, 126706, '\P{Is_General_Category=	sm}', "");
Expect(0, 126706, '\P{^Is_General_Category=	sm}', "");
Error('\p{Is_Gc=--Math_SYMBOL:=}');
Error('\P{Is_Gc=--Math_SYMBOL:=}');
Expect(1, 126705, '\p{Is_Gc=mathsymbol}', "");
Expect(0, 126705, '\p{^Is_Gc=mathsymbol}', "");
Expect(0, 126705, '\P{Is_Gc=mathsymbol}', "");
Expect(1, 126705, '\P{^Is_Gc=mathsymbol}', "");
Expect(0, 126706, '\p{Is_Gc=mathsymbol}', "");
Expect(1, 126706, '\p{^Is_Gc=mathsymbol}', "");
Expect(1, 126706, '\P{Is_Gc=mathsymbol}', "");
Expect(0, 126706, '\P{^Is_Gc=mathsymbol}', "");
Expect(1, 126705, '\p{Is_Gc:	-	Math_Symbol}', "");
Expect(0, 126705, '\p{^Is_Gc:	-	Math_Symbol}', "");
Expect(0, 126705, '\P{Is_Gc:	-	Math_Symbol}', "");
Expect(1, 126705, '\P{^Is_Gc:	-	Math_Symbol}', "");
Expect(0, 126706, '\p{Is_Gc:	-	Math_Symbol}', "");
Expect(1, 126706, '\p{^Is_Gc:	-	Math_Symbol}', "");
Expect(1, 126706, '\P{Is_Gc:	-	Math_Symbol}', "");
Expect(0, 126706, '\P{^Is_Gc:	-	Math_Symbol}', "");
Error('\p{Is_Category=--Sm:=}');
Error('\P{Is_Category=--Sm:=}');
Expect(1, 126705, '\p{Is_Category=sm}', "");
Expect(0, 126705, '\p{^Is_Category=sm}', "");
Expect(0, 126705, '\P{Is_Category=sm}', "");
Expect(1, 126705, '\P{^Is_Category=sm}', "");
Expect(0, 126706, '\p{Is_Category=sm}', "");
Expect(1, 126706, '\p{^Is_Category=sm}', "");
Expect(1, 126706, '\P{Is_Category=sm}', "");
Expect(0, 126706, '\P{^Is_Category=sm}', "");
Expect(1, 126705, '\p{Is_Category:   - sm}', "");
Expect(0, 126705, '\p{^Is_Category:   - sm}', "");
Expect(0, 126705, '\P{Is_Category:   - sm}', "");
Expect(1, 126705, '\P{^Is_Category:   - sm}', "");
Expect(0, 126706, '\p{Is_Category:   - sm}', "");
Expect(1, 126706, '\p{^Is_Category:   - sm}', "");
Expect(1, 126706, '\P{Is_Category:   - sm}', "");
Expect(0, 126706, '\P{^Is_Category:   - sm}', "");
Error('\p{General_Category=/a/ 	Other_letter}');
Error('\P{General_Category=/a/ 	Other_letter}');
Expect(1, 195101, '\p{General_Category=otherletter}', "");
Expect(0, 195101, '\p{^General_Category=otherletter}', "");
Expect(0, 195101, '\P{General_Category=otherletter}', "");
Expect(1, 195101, '\P{^General_Category=otherletter}', "");
Expect(0, 195102, '\p{General_Category=otherletter}', "");
Expect(1, 195102, '\p{^General_Category=otherletter}', "");
Expect(1, 195102, '\P{General_Category=otherletter}', "");
Expect(0, 195102, '\P{^General_Category=otherletter}', "");
Expect(1, 195101, '\p{General_Category=- other_Letter}', "");
Expect(0, 195101, '\p{^General_Category=- other_Letter}', "");
Expect(0, 195101, '\P{General_Category=- other_Letter}', "");
Expect(1, 195101, '\P{^General_Category=- other_Letter}', "");
Expect(0, 195102, '\p{General_Category=- other_Letter}', "");
Expect(1, 195102, '\p{^General_Category=- other_Letter}', "");
Expect(1, 195102, '\P{General_Category=- other_Letter}', "");
Expect(0, 195102, '\P{^General_Category=- other_Letter}', "");
Error('\p{Gc=/a/_Lo}');
Error('\P{Gc=/a/_Lo}');
Expect(1, 195101, '\p{Gc=lo}', "");
Expect(0, 195101, '\p{^Gc=lo}', "");
Expect(0, 195101, '\P{Gc=lo}', "");
Expect(1, 195101, '\P{^Gc=lo}', "");
Expect(0, 195102, '\p{Gc=lo}', "");
Expect(1, 195102, '\p{^Gc=lo}', "");
Expect(1, 195102, '\P{Gc=lo}', "");
Expect(0, 195102, '\P{^Gc=lo}', "");
Expect(1, 195101, '\p{Gc: -LO}', "");
Expect(0, 195101, '\p{^Gc: -LO}', "");
Expect(0, 195101, '\P{Gc: -LO}', "");
Expect(1, 195101, '\P{^Gc: -LO}', "");
Expect(0, 195102, '\p{Gc: -LO}', "");
Expect(1, 195102, '\p{^Gc: -LO}', "");
Expect(1, 195102, '\P{Gc: -LO}', "");
Expect(0, 195102, '\P{^Gc: -LO}', "");
Error('\p{Category=-OTHER_LETTER:=}');
Error('\P{Category=-OTHER_LETTER:=}');
Expect(1, 195101, '\p{Category=otherletter}', "");
Expect(0, 195101, '\p{^Category=otherletter}', "");
Expect(0, 195101, '\P{Category=otherletter}', "");
Expect(1, 195101, '\P{^Category=otherletter}', "");
Expect(0, 195102, '\p{Category=otherletter}', "");
Expect(1, 195102, '\p{^Category=otherletter}', "");
Expect(1, 195102, '\P{Category=otherletter}', "");
Expect(0, 195102, '\P{^Category=otherletter}', "");
Expect(1, 195101, '\p{Category= _other_LETTER}', "");
Expect(0, 195101, '\p{^Category= _other_LETTER}', "");
Expect(0, 195101, '\P{Category= _other_LETTER}', "");
Expect(1, 195101, '\P{^Category= _other_LETTER}', "");
Expect(0, 195102, '\p{Category= _other_LETTER}', "");
Expect(1, 195102, '\p{^Category= _other_LETTER}', "");
Expect(1, 195102, '\P{Category= _other_LETTER}', "");
Expect(0, 195102, '\P{^Category= _other_LETTER}', "");
Error('\p{Is_General_Category=:=	_lo}');
Error('\P{Is_General_Category=:=	_lo}');
Expect(1, 195101, '\p{Is_General_Category=lo}', "");
Expect(0, 195101, '\p{^Is_General_Category=lo}', "");
Expect(0, 195101, '\P{Is_General_Category=lo}', "");
Expect(1, 195101, '\P{^Is_General_Category=lo}', "");
Expect(0, 195102, '\p{Is_General_Category=lo}', "");
Expect(1, 195102, '\p{^Is_General_Category=lo}', "");
Expect(1, 195102, '\P{Is_General_Category=lo}', "");
Expect(0, 195102, '\P{^Is_General_Category=lo}', "");
Expect(1, 195101, '\p{Is_General_Category= _lo}', "");
Expect(0, 195101, '\p{^Is_General_Category= _lo}', "");
Expect(0, 195101, '\P{Is_General_Category= _lo}', "");
Expect(1, 195101, '\P{^Is_General_Category= _lo}', "");
Expect(0, 195102, '\p{Is_General_Category= _lo}', "");
Expect(1, 195102, '\p{^Is_General_Category= _lo}', "");
Expect(1, 195102, '\P{Is_General_Category= _lo}', "");
Expect(0, 195102, '\P{^Is_General_Category= _lo}', "");
Error('\p{Is_Gc= :=Other_Letter}');
Error('\P{Is_Gc= :=Other_Letter}');
Expect(1, 195101, '\p{Is_Gc=otherletter}', "");
Expect(0, 195101, '\p{^Is_Gc=otherletter}', "");
Expect(0, 195101, '\P{Is_Gc=otherletter}', "");
Expect(1, 195101, '\P{^Is_Gc=otherletter}', "");
Expect(0, 195102, '\p{Is_Gc=otherletter}', "");
Expect(1, 195102, '\p{^Is_Gc=otherletter}', "");
Expect(1, 195102, '\P{Is_Gc=otherletter}', "");
Expect(0, 195102, '\P{^Is_Gc=otherletter}', "");
Expect(1, 195101, '\p{Is_Gc=	-other_Letter}', "");
Expect(0, 195101, '\p{^Is_Gc=	-other_Letter}', "");
Expect(0, 195101, '\P{Is_Gc=	-other_Letter}', "");
Expect(1, 195101, '\P{^Is_Gc=	-other_Letter}', "");
Expect(0, 195102, '\p{Is_Gc=	-other_Letter}', "");
Expect(1, 195102, '\p{^Is_Gc=	-other_Letter}', "");
Expect(1, 195102, '\P{Is_Gc=	-other_Letter}', "");
Expect(0, 195102, '\P{^Is_Gc=	-other_Letter}', "");
Error('\p{Is_Category:   _	lo:=}');
Error('\P{Is_Category:   _	lo:=}');
Expect(1, 195101, '\p{Is_Category=lo}', "");
Expect(0, 195101, '\p{^Is_Category=lo}', "");
Expect(0, 195101, '\P{Is_Category=lo}', "");
Expect(1, 195101, '\P{^Is_Category=lo}', "");
Expect(0, 195102, '\p{Is_Category=lo}', "");
Expect(1, 195102, '\p{^Is_Category=lo}', "");
Expect(1, 195102, '\P{Is_Category=lo}', "");
Expect(0, 195102, '\P{^Is_Category=lo}', "");
Expect(1, 195101, '\p{Is_Category=-Lo}', "");
Expect(0, 195101, '\p{^Is_Category=-Lo}', "");
Expect(0, 195101, '\P{Is_Category=-Lo}', "");
Expect(1, 195101, '\P{^Is_Category=-Lo}', "");
Expect(0, 195102, '\p{Is_Category=-Lo}', "");
Expect(1, 195102, '\p{^Is_Category=-Lo}', "");
Expect(1, 195102, '\P{Is_Category=-Lo}', "");
Expect(0, 195102, '\P{^Is_Category=-Lo}', "");
Error('\p{General_Category=	:=SEPARATOR}');
Error('\P{General_Category=	:=SEPARATOR}');
Expect(1, 12288, '\p{General_Category=separator}', "");
Expect(0, 12288, '\p{^General_Category=separator}', "");
Expect(0, 12288, '\P{General_Category=separator}', "");
Expect(1, 12288, '\P{^General_Category=separator}', "");
Expect(0, 12289, '\p{General_Category=separator}', "");
Expect(1, 12289, '\p{^General_Category=separator}', "");
Expect(1, 12289, '\P{General_Category=separator}', "");
Expect(0, 12289, '\P{^General_Category=separator}', "");
Expect(1, 12288, '\p{General_Category=	-separator}', "");
Expect(0, 12288, '\p{^General_Category=	-separator}', "");
Expect(0, 12288, '\P{General_Category=	-separator}', "");
Expect(1, 12288, '\P{^General_Category=	-separator}', "");
Expect(0, 12289, '\p{General_Category=	-separator}', "");
Expect(1, 12289, '\p{^General_Category=	-separator}', "");
Expect(1, 12289, '\P{General_Category=	-separator}', "");
Expect(0, 12289, '\P{^General_Category=	-separator}', "");
Error('\p{Gc=/a/_ Z}');
Error('\P{Gc=/a/_ Z}');
Expect(1, 12288, '\p{Gc=z}', "");
Expect(0, 12288, '\p{^Gc=z}', "");
Expect(0, 12288, '\P{Gc=z}', "");
Expect(1, 12288, '\P{^Gc=z}', "");
Expect(0, 12289, '\p{Gc=z}', "");
Expect(1, 12289, '\p{^Gc=z}', "");
Expect(1, 12289, '\P{Gc=z}', "");
Expect(0, 12289, '\P{^Gc=z}', "");
Expect(1, 12288, '\p{Gc=-z}', "");
Expect(0, 12288, '\p{^Gc=-z}', "");
Expect(0, 12288, '\P{Gc=-z}', "");
Expect(1, 12288, '\P{^Gc=-z}', "");
Expect(0, 12289, '\p{Gc=-z}', "");
Expect(1, 12289, '\p{^Gc=-z}', "");
Expect(1, 12289, '\P{Gc=-z}', "");
Expect(0, 12289, '\P{^Gc=-z}', "");
Error('\p{Category:   separator:=}');
Error('\P{Category:   separator:=}');
Expect(1, 12288, '\p{Category: separator}', "");
Expect(0, 12288, '\p{^Category: separator}', "");
Expect(0, 12288, '\P{Category: separator}', "");
Expect(1, 12288, '\P{^Category: separator}', "");
Expect(0, 12289, '\p{Category: separator}', "");
Expect(1, 12289, '\p{^Category: separator}', "");
Expect(1, 12289, '\P{Category: separator}', "");
Expect(0, 12289, '\P{^Category: separator}', "");
Expect(1, 12288, '\p{Category=_ Separator}', "");
Expect(0, 12288, '\p{^Category=_ Separator}', "");
Expect(0, 12288, '\P{Category=_ Separator}', "");
Expect(1, 12288, '\P{^Category=_ Separator}', "");
Expect(0, 12289, '\p{Category=_ Separator}', "");
Expect(1, 12289, '\p{^Category=_ Separator}', "");
Expect(1, 12289, '\P{Category=_ Separator}', "");
Expect(0, 12289, '\P{^Category=_ Separator}', "");
Error('\p{Is_General_Category: --z:=}');
Error('\P{Is_General_Category: --z:=}');
Expect(1, 12288, '\p{Is_General_Category:	z}', "");
Expect(0, 12288, '\p{^Is_General_Category:	z}', "");
Expect(0, 12288, '\P{Is_General_Category:	z}', "");
Expect(1, 12288, '\P{^Is_General_Category:	z}', "");
Expect(0, 12289, '\p{Is_General_Category:	z}', "");
Expect(1, 12289, '\p{^Is_General_Category:	z}', "");
Expect(1, 12289, '\P{Is_General_Category:	z}', "");
Expect(0, 12289, '\P{^Is_General_Category:	z}', "");
Expect(1, 12288, '\p{Is_General_Category=-Z}', "");
Expect(0, 12288, '\p{^Is_General_Category=-Z}', "");
Expect(0, 12288, '\P{Is_General_Category=-Z}', "");
Expect(1, 12288, '\P{^Is_General_Category=-Z}', "");
Expect(0, 12289, '\p{Is_General_Category=-Z}', "");
Expect(1, 12289, '\p{^Is_General_Category=-Z}', "");
Expect(1, 12289, '\P{Is_General_Category=-Z}', "");
Expect(0, 12289, '\P{^Is_General_Category=-Z}', "");
Error('\p{Is_Gc=:=_	Separator}');
Error('\P{Is_Gc=:=_	Separator}');
Expect(1, 12288, '\p{Is_Gc=separator}', "");
Expect(0, 12288, '\p{^Is_Gc=separator}', "");
Expect(0, 12288, '\P{Is_Gc=separator}', "");
Expect(1, 12288, '\P{^Is_Gc=separator}', "");
Expect(0, 12289, '\p{Is_Gc=separator}', "");
Expect(1, 12289, '\p{^Is_Gc=separator}', "");
Expect(1, 12289, '\P{Is_Gc=separator}', "");
Expect(0, 12289, '\P{^Is_Gc=separator}', "");
Expect(1, 12288, '\p{Is_Gc:_-SEPARATOR}', "");
Expect(0, 12288, '\p{^Is_Gc:_-SEPARATOR}', "");
Expect(0, 12288, '\P{Is_Gc:_-SEPARATOR}', "");
Expect(1, 12288, '\P{^Is_Gc:_-SEPARATOR}', "");
Expect(0, 12289, '\p{Is_Gc:_-SEPARATOR}', "");
Expect(1, 12289, '\p{^Is_Gc:_-SEPARATOR}', "");
Expect(1, 12289, '\P{Is_Gc:_-SEPARATOR}', "");
Expect(0, 12289, '\P{^Is_Gc:_-SEPARATOR}', "");
Error('\p{Is_Category=_	Z/a/}');
Error('\P{Is_Category=_	Z/a/}');
Expect(1, 12288, '\p{Is_Category=z}', "");
Expect(0, 12288, '\p{^Is_Category=z}', "");
Expect(0, 12288, '\P{Is_Category=z}', "");
Expect(1, 12288, '\P{^Is_Category=z}', "");
Expect(0, 12289, '\p{Is_Category=z}', "");
Expect(1, 12289, '\p{^Is_Category=z}', "");
Expect(1, 12289, '\P{Is_Category=z}', "");
Expect(0, 12289, '\P{^Is_Category=z}', "");
Expect(1, 12288, '\p{Is_Category:	__Z}', "");
Expect(0, 12288, '\p{^Is_Category:	__Z}', "");
Expect(0, 12288, '\P{Is_Category:	__Z}', "");
Expect(1, 12288, '\P{^Is_Category:	__Z}', "");
Expect(0, 12289, '\p{Is_Category:	__Z}', "");
Expect(1, 12289, '\p{^Is_Category:	__Z}', "");
Expect(1, 12289, '\P{Is_Category:	__Z}', "");
Expect(0, 12289, '\P{^Is_Category:	__Z}', "");
Error('\p{General_Category=_/a/other_PUNCTUATION}');
Error('\P{General_Category=_/a/other_PUNCTUATION}');
Expect(1, 74867, '\p{General_Category=otherpunctuation}', "");
Expect(0, 74867, '\p{^General_Category=otherpunctuation}', "");
Expect(0, 74867, '\P{General_Category=otherpunctuation}', "");
Expect(1, 74867, '\P{^General_Category=otherpunctuation}', "");
Expect(0, 74868, '\p{General_Category=otherpunctuation}', "");
Expect(1, 74868, '\p{^General_Category=otherpunctuation}', "");
Expect(1, 74868, '\P{General_Category=otherpunctuation}', "");
Expect(0, 74868, '\P{^General_Category=otherpunctuation}', "");
Expect(1, 74867, '\p{General_Category=		Other_PUNCTUATION}', "");
Expect(0, 74867, '\p{^General_Category=		Other_PUNCTUATION}', "");
Expect(0, 74867, '\P{General_Category=		Other_PUNCTUATION}', "");
Expect(1, 74867, '\P{^General_Category=		Other_PUNCTUATION}', "");
Expect(0, 74868, '\p{General_Category=		Other_PUNCTUATION}', "");
Expect(1, 74868, '\p{^General_Category=		Other_PUNCTUATION}', "");
Expect(1, 74868, '\P{General_Category=		Other_PUNCTUATION}', "");
Expect(0, 74868, '\P{^General_Category=		Other_PUNCTUATION}', "");
Error('\p{Gc=-:=PO}');
Error('\P{Gc=-:=PO}');
Expect(1, 74867, '\p{Gc=po}', "");
Expect(0, 74867, '\p{^Gc=po}', "");
Expect(0, 74867, '\P{Gc=po}', "");
Expect(1, 74867, '\P{^Gc=po}', "");
Expect(0, 74868, '\p{Gc=po}', "");
Expect(1, 74868, '\p{^Gc=po}', "");
Expect(1, 74868, '\P{Gc=po}', "");
Expect(0, 74868, '\P{^Gc=po}', "");
Expect(1, 74867, '\p{Gc:	-Po}', "");
Expect(0, 74867, '\p{^Gc:	-Po}', "");
Expect(0, 74867, '\P{Gc:	-Po}', "");
Expect(1, 74867, '\P{^Gc:	-Po}', "");
Expect(0, 74868, '\p{Gc:	-Po}', "");
Expect(1, 74868, '\p{^Gc:	-Po}', "");
Expect(1, 74868, '\P{Gc:	-Po}', "");
Expect(0, 74868, '\P{^Gc:	-Po}', "");
Error('\p{Category=/a/ Other_Punctuation}');
Error('\P{Category=/a/ Other_Punctuation}');
Expect(1, 74867, '\p{Category: otherpunctuation}', "");
Expect(0, 74867, '\p{^Category: otherpunctuation}', "");
Expect(0, 74867, '\P{Category: otherpunctuation}', "");
Expect(1, 74867, '\P{^Category: otherpunctuation}', "");
Expect(0, 74868, '\p{Category: otherpunctuation}', "");
Expect(1, 74868, '\p{^Category: otherpunctuation}', "");
Expect(1, 74868, '\P{Category: otherpunctuation}', "");
Expect(0, 74868, '\P{^Category: otherpunctuation}', "");
Expect(1, 74867, '\p{Category=		Other_Punctuation}', "");
Expect(0, 74867, '\p{^Category=		Other_Punctuation}', "");
Expect(0, 74867, '\P{Category=		Other_Punctuation}', "");
Expect(1, 74867, '\P{^Category=		Other_Punctuation}', "");
Expect(0, 74868, '\p{Category=		Other_Punctuation}', "");
Expect(1, 74868, '\p{^Category=		Other_Punctuation}', "");
Expect(1, 74868, '\P{Category=		Other_Punctuation}', "");
Expect(0, 74868, '\P{^Category=		Other_Punctuation}', "");
Error('\p{Is_General_Category= -PO:=}');
Error('\P{Is_General_Category= -PO:=}');
Expect(1, 74867, '\p{Is_General_Category:po}', "");
Expect(0, 74867, '\p{^Is_General_Category:po}', "");
Expect(0, 74867, '\P{Is_General_Category:po}', "");
Expect(1, 74867, '\P{^Is_General_Category:po}', "");
Expect(0, 74868, '\p{Is_General_Category:po}', "");
Expect(1, 74868, '\p{^Is_General_Category:po}', "");
Expect(1, 74868, '\P{Is_General_Category:po}', "");
Expect(0, 74868, '\P{^Is_General_Category:po}', "");
Expect(1, 74867, '\p{Is_General_Category= 	po}', "");
Expect(0, 74867, '\p{^Is_General_Category= 	po}', "");
Expect(0, 74867, '\P{Is_General_Category= 	po}', "");
Expect(1, 74867, '\P{^Is_General_Category= 	po}', "");
Expect(0, 74868, '\p{Is_General_Category= 	po}', "");
Expect(1, 74868, '\p{^Is_General_Category= 	po}', "");
Expect(1, 74868, '\P{Is_General_Category= 	po}', "");
Expect(0, 74868, '\P{^Is_General_Category= 	po}', "");
Error('\p{Is_Gc=/a/_	OTHER_Punctuation}');
Error('\P{Is_Gc=/a/_	OTHER_Punctuation}');
Expect(1, 74867, '\p{Is_Gc=otherpunctuation}', "");
Expect(0, 74867, '\p{^Is_Gc=otherpunctuation}', "");
Expect(0, 74867, '\P{Is_Gc=otherpunctuation}', "");
Expect(1, 74867, '\P{^Is_Gc=otherpunctuation}', "");
Expect(0, 74868, '\p{Is_Gc=otherpunctuation}', "");
Expect(1, 74868, '\p{^Is_Gc=otherpunctuation}', "");
Expect(1, 74868, '\P{Is_Gc=otherpunctuation}', "");
Expect(0, 74868, '\P{^Is_Gc=otherpunctuation}', "");
Expect(1, 74867, '\p{Is_Gc=_OTHER_PUNCTUATION}', "");
Expect(0, 74867, '\p{^Is_Gc=_OTHER_PUNCTUATION}', "");
Expect(0, 74867, '\P{Is_Gc=_OTHER_PUNCTUATION}', "");
Expect(1, 74867, '\P{^Is_Gc=_OTHER_PUNCTUATION}', "");
Expect(0, 74868, '\p{Is_Gc=_OTHER_PUNCTUATION}', "");
Expect(1, 74868, '\p{^Is_Gc=_OTHER_PUNCTUATION}', "");
Expect(1, 74868, '\P{Is_Gc=_OTHER_PUNCTUATION}', "");
Expect(0, 74868, '\P{^Is_Gc=_OTHER_PUNCTUATION}', "");
Error('\p{Is_Category=:=_	PO}');
Error('\P{Is_Category=:=_	PO}');
Expect(1, 74867, '\p{Is_Category=po}', "");
Expect(0, 74867, '\p{^Is_Category=po}', "");
Expect(0, 74867, '\P{Is_Category=po}', "");
Expect(1, 74867, '\P{^Is_Category=po}', "");
Expect(0, 74868, '\p{Is_Category=po}', "");
Expect(1, 74868, '\p{^Is_Category=po}', "");
Expect(1, 74868, '\P{Is_Category=po}', "");
Expect(0, 74868, '\P{^Is_Category=po}', "");
Expect(1, 74867, '\p{Is_Category=_Po}', "");
Expect(0, 74867, '\p{^Is_Category=_Po}', "");
Expect(0, 74867, '\P{Is_Category=_Po}', "");
Expect(1, 74867, '\P{^Is_Category=_Po}', "");
Expect(0, 74868, '\p{Is_Category=_Po}', "");
Expect(1, 74868, '\p{^Is_Category=_Po}', "");
Expect(1, 74868, '\P{Is_Category=_Po}', "");
Expect(0, 74868, '\P{^Is_Category=_Po}', "");
Error('\p{General_Category=:=		Space_Separator}');
Error('\P{General_Category=:=		Space_Separator}');
Expect(1, 12288, '\p{General_Category=spaceseparator}', "");
Expect(0, 12288, '\p{^General_Category=spaceseparator}', "");
Expect(0, 12288, '\P{General_Category=spaceseparator}', "");
Expect(1, 12288, '\P{^General_Category=spaceseparator}', "");
Expect(0, 12289, '\p{General_Category=spaceseparator}', "");
Expect(1, 12289, '\p{^General_Category=spaceseparator}', "");
Expect(1, 12289, '\P{General_Category=spaceseparator}', "");
Expect(0, 12289, '\P{^General_Category=spaceseparator}', "");
Expect(1, 12288, '\p{General_Category=	 SPACE_separator}', "");
Expect(0, 12288, '\p{^General_Category=	 SPACE_separator}', "");
Expect(0, 12288, '\P{General_Category=	 SPACE_separator}', "");
Expect(1, 12288, '\P{^General_Category=	 SPACE_separator}', "");
Expect(0, 12289, '\p{General_Category=	 SPACE_separator}', "");
Expect(1, 12289, '\p{^General_Category=	 SPACE_separator}', "");
Expect(1, 12289, '\P{General_Category=	 SPACE_separator}', "");
Expect(0, 12289, '\P{^General_Category=	 SPACE_separator}', "");
Error('\p{Gc=/a/-Zs}');
Error('\P{Gc=/a/-Zs}');
Expect(1, 12288, '\p{Gc=zs}', "");
Expect(0, 12288, '\p{^Gc=zs}', "");
Expect(0, 12288, '\P{Gc=zs}', "");
Expect(1, 12288, '\P{^Gc=zs}', "");
Expect(0, 12289, '\p{Gc=zs}', "");
Expect(1, 12289, '\p{^Gc=zs}', "");
Expect(1, 12289, '\P{Gc=zs}', "");
Expect(0, 12289, '\P{^Gc=zs}', "");
Expect(1, 12288, '\p{Gc:		Zs}', "");
Expect(0, 12288, '\p{^Gc:		Zs}', "");
Expect(0, 12288, '\P{Gc:		Zs}', "");
Expect(1, 12288, '\P{^Gc:		Zs}', "");
Expect(0, 12289, '\p{Gc:		Zs}', "");
Expect(1, 12289, '\p{^Gc:		Zs}', "");
Expect(1, 12289, '\P{Gc:		Zs}', "");
Expect(0, 12289, '\P{^Gc:		Zs}', "");
Error('\p{Category= /a/Space_separator}');
Error('\P{Category= /a/Space_separator}');
Expect(1, 12288, '\p{Category=spaceseparator}', "");
Expect(0, 12288, '\p{^Category=spaceseparator}', "");
Expect(0, 12288, '\P{Category=spaceseparator}', "");
Expect(1, 12288, '\P{^Category=spaceseparator}', "");
Expect(0, 12289, '\p{Category=spaceseparator}', "");
Expect(1, 12289, '\p{^Category=spaceseparator}', "");
Expect(1, 12289, '\P{Category=spaceseparator}', "");
Expect(0, 12289, '\P{^Category=spaceseparator}', "");
Expect(1, 12288, '\p{Category=	_Space_Separator}', "");
Expect(0, 12288, '\p{^Category=	_Space_Separator}', "");
Expect(0, 12288, '\P{Category=	_Space_Separator}', "");
Expect(1, 12288, '\P{^Category=	_Space_Separator}', "");
Expect(0, 12289, '\p{Category=	_Space_Separator}', "");
Expect(1, 12289, '\p{^Category=	_Space_Separator}', "");
Expect(1, 12289, '\P{Category=	_Space_Separator}', "");
Expect(0, 12289, '\P{^Category=	_Space_Separator}', "");
Error('\p{Is_General_Category=:=	 zs}');
Error('\P{Is_General_Category=:=	 zs}');
Expect(1, 12288, '\p{Is_General_Category=zs}', "");
Expect(0, 12288, '\p{^Is_General_Category=zs}', "");
Expect(0, 12288, '\P{Is_General_Category=zs}', "");
Expect(1, 12288, '\P{^Is_General_Category=zs}', "");
Expect(0, 12289, '\p{Is_General_Category=zs}', "");
Expect(1, 12289, '\p{^Is_General_Category=zs}', "");
Expect(1, 12289, '\P{Is_General_Category=zs}', "");
Expect(0, 12289, '\P{^Is_General_Category=zs}', "");
Expect(1, 12288, '\p{Is_General_Category=_ Zs}', "");
Expect(0, 12288, '\p{^Is_General_Category=_ Zs}', "");
Expect(0, 12288, '\P{Is_General_Category=_ Zs}', "");
Expect(1, 12288, '\P{^Is_General_Category=_ Zs}', "");
Expect(0, 12289, '\p{Is_General_Category=_ Zs}', "");
Expect(1, 12289, '\p{^Is_General_Category=_ Zs}', "");
Expect(1, 12289, '\P{Is_General_Category=_ Zs}', "");
Expect(0, 12289, '\P{^Is_General_Category=_ Zs}', "");
Error('\p{Is_Gc=:=	Space_SEPARATOR}');
Error('\P{Is_Gc=:=	Space_SEPARATOR}');
Expect(1, 12288, '\p{Is_Gc=spaceseparator}', "");
Expect(0, 12288, '\p{^Is_Gc=spaceseparator}', "");
Expect(0, 12288, '\P{Is_Gc=spaceseparator}', "");
Expect(1, 12288, '\P{^Is_Gc=spaceseparator}', "");
Expect(0, 12289, '\p{Is_Gc=spaceseparator}', "");
Expect(1, 12289, '\p{^Is_Gc=spaceseparator}', "");
Expect(1, 12289, '\P{Is_Gc=spaceseparator}', "");
Expect(0, 12289, '\P{^Is_Gc=spaceseparator}', "");
Expect(1, 12288, '\p{Is_Gc= SPACE_Separator}', "");
Expect(0, 12288, '\p{^Is_Gc= SPACE_Separator}', "");
Expect(0, 12288, '\P{Is_Gc= SPACE_Separator}', "");
Expect(1, 12288, '\P{^Is_Gc= SPACE_Separator}', "");
Expect(0, 12289, '\p{Is_Gc= SPACE_Separator}', "");
Expect(1, 12289, '\p{^Is_Gc= SPACE_Separator}', "");
Expect(1, 12289, '\P{Is_Gc= SPACE_Separator}', "");
Expect(0, 12289, '\P{^Is_Gc= SPACE_Separator}', "");
Error('\p{Is_Category: _/a/Zs}');
Error('\P{Is_Category: _/a/Zs}');
Expect(1, 12288, '\p{Is_Category=zs}', "");
Expect(0, 12288, '\p{^Is_Category=zs}', "");
Expect(0, 12288, '\P{Is_Category=zs}', "");
Expect(1, 12288, '\P{^Is_Category=zs}', "");
Expect(0, 12289, '\p{Is_Category=zs}', "");
Expect(1, 12289, '\p{^Is_Category=zs}', "");
Expect(1, 12289, '\P{Is_Category=zs}', "");
Expect(0, 12289, '\P{^Is_Category=zs}', "");
Expect(1, 12288, '\p{Is_Category=	 zs}', "");
Expect(0, 12288, '\p{^Is_Category=	 zs}', "");
Expect(0, 12288, '\P{Is_Category=	 zs}', "");
Expect(1, 12288, '\P{^Is_Category=	 zs}', "");
Expect(0, 12289, '\p{Is_Category=	 zs}', "");
Expect(1, 12289, '\p{^Is_Category=	 zs}', "");
Expect(1, 12289, '\P{Is_Category=	 zs}', "");
Expect(0, 12289, '\P{^Is_Category=	 zs}', "");
Error('\p{General_Category:   	 OTHER_NUMBER/a/}');
Error('\P{General_Category:   	 OTHER_NUMBER/a/}');
Expect(1, 127242, '\p{General_Category=othernumber}', "");
Expect(0, 127242, '\p{^General_Category=othernumber}', "");
Expect(0, 127242, '\P{General_Category=othernumber}', "");
Expect(1, 127242, '\P{^General_Category=othernumber}', "");
Expect(0, 127243, '\p{General_Category=othernumber}', "");
Expect(1, 127243, '\p{^General_Category=othernumber}', "");
Expect(1, 127243, '\P{General_Category=othernumber}', "");
Expect(0, 127243, '\P{^General_Category=othernumber}', "");
Expect(1, 127242, '\p{General_Category=_Other_Number}', "");
Expect(0, 127242, '\p{^General_Category=_Other_Number}', "");
Expect(0, 127242, '\P{General_Category=_Other_Number}', "");
Expect(1, 127242, '\P{^General_Category=_Other_Number}', "");
Expect(0, 127243, '\p{General_Category=_Other_Number}', "");
Expect(1, 127243, '\p{^General_Category=_Other_Number}', "");
Expect(1, 127243, '\P{General_Category=_Other_Number}', "");
Expect(0, 127243, '\P{^General_Category=_Other_Number}', "");
Error('\p{Gc=:=-NO}');
Error('\P{Gc=:=-NO}');
Expect(1, 127242, '\p{Gc=no}', "");
Expect(0, 127242, '\p{^Gc=no}', "");
Expect(0, 127242, '\P{Gc=no}', "");
Expect(1, 127242, '\P{^Gc=no}', "");
Expect(0, 127243, '\p{Gc=no}', "");
Expect(1, 127243, '\p{^Gc=no}', "");
Expect(1, 127243, '\P{Gc=no}', "");
Expect(0, 127243, '\P{^Gc=no}', "");
Expect(1, 127242, '\p{Gc:  No}', "");
Expect(0, 127242, '\p{^Gc:  No}', "");
Expect(0, 127242, '\P{Gc:  No}', "");
Expect(1, 127242, '\P{^Gc:  No}', "");
Expect(0, 127243, '\p{Gc:  No}', "");
Expect(1, 127243, '\p{^Gc:  No}', "");
Expect(1, 127243, '\P{Gc:  No}', "");
Expect(0, 127243, '\P{^Gc:  No}', "");
Error('\p{Category=-/a/other_Number}');
Error('\P{Category=-/a/other_Number}');
Expect(1, 127242, '\p{Category: othernumber}', "");
Expect(0, 127242, '\p{^Category: othernumber}', "");
Expect(0, 127242, '\P{Category: othernumber}', "");
Expect(1, 127242, '\P{^Category: othernumber}', "");
Expect(0, 127243, '\p{Category: othernumber}', "");
Expect(1, 127243, '\p{^Category: othernumber}', "");
Expect(1, 127243, '\P{Category: othernumber}', "");
Expect(0, 127243, '\P{^Category: othernumber}', "");
Expect(1, 127242, '\p{Category= _Other_number}', "");
Expect(0, 127242, '\p{^Category= _Other_number}', "");
Expect(0, 127242, '\P{Category= _Other_number}', "");
Expect(1, 127242, '\P{^Category= _Other_number}', "");
Expect(0, 127243, '\p{Category= _Other_number}', "");
Expect(1, 127243, '\p{^Category= _Other_number}', "");
Expect(1, 127243, '\P{Category= _Other_number}', "");
Expect(0, 127243, '\P{^Category= _Other_number}', "");
Error('\p{Is_General_Category=_/a/NO}');
Error('\P{Is_General_Category=_/a/NO}');
Expect(1, 127242, '\p{Is_General_Category=no}', "");
Expect(0, 127242, '\p{^Is_General_Category=no}', "");
Expect(0, 127242, '\P{Is_General_Category=no}', "");
Expect(1, 127242, '\P{^Is_General_Category=no}', "");
Expect(0, 127243, '\p{Is_General_Category=no}', "");
Expect(1, 127243, '\p{^Is_General_Category=no}', "");
Expect(1, 127243, '\P{Is_General_Category=no}', "");
Expect(0, 127243, '\P{^Is_General_Category=no}', "");
Expect(1, 127242, '\p{Is_General_Category=- No}', "");
Expect(0, 127242, '\p{^Is_General_Category=- No}', "");
Expect(0, 127242, '\P{Is_General_Category=- No}', "");
Expect(1, 127242, '\P{^Is_General_Category=- No}', "");
Expect(0, 127243, '\p{Is_General_Category=- No}', "");
Expect(1, 127243, '\p{^Is_General_Category=- No}', "");
Expect(1, 127243, '\P{Is_General_Category=- No}', "");
Expect(0, 127243, '\P{^Is_General_Category=- No}', "");
Error('\p{Is_Gc=__Other_number/a/}');
Error('\P{Is_Gc=__Other_number/a/}');
Expect(1, 127242, '\p{Is_Gc=othernumber}', "");
Expect(0, 127242, '\p{^Is_Gc=othernumber}', "");
Expect(0, 127242, '\P{Is_Gc=othernumber}', "");
Expect(1, 127242, '\P{^Is_Gc=othernumber}', "");
Expect(0, 127243, '\p{Is_Gc=othernumber}', "");
Expect(1, 127243, '\p{^Is_Gc=othernumber}', "");
Expect(1, 127243, '\P{Is_Gc=othernumber}', "");
Expect(0, 127243, '\P{^Is_Gc=othernumber}', "");
Expect(1, 127242, '\p{Is_Gc= -OTHER_Number}', "");
Expect(0, 127242, '\p{^Is_Gc= -OTHER_Number}', "");
Expect(0, 127242, '\P{Is_Gc= -OTHER_Number}', "");
Expect(1, 127242, '\P{^Is_Gc= -OTHER_Number}', "");
Expect(0, 127243, '\p{Is_Gc= -OTHER_Number}', "");
Expect(1, 127243, '\p{^Is_Gc= -OTHER_Number}', "");
Expect(1, 127243, '\P{Is_Gc= -OTHER_Number}', "");
Expect(0, 127243, '\P{^Is_Gc= -OTHER_Number}', "");
Error('\p{Is_Category=-NO/a/}');
Error('\P{Is_Category=-NO/a/}');
Expect(1, 127242, '\p{Is_Category=no}', "");
Expect(0, 127242, '\p{^Is_Category=no}', "");
Expect(0, 127242, '\P{Is_Category=no}', "");
Expect(1, 127242, '\P{^Is_Category=no}', "");
Expect(0, 127243, '\p{Is_Category=no}', "");
Expect(1, 127243, '\p{^Is_Category=no}', "");
Expect(1, 127243, '\P{Is_Category=no}', "");
Expect(0, 127243, '\P{^Is_Category=no}', "");
Expect(1, 127242, '\p{Is_Category=_	no}', "");
Expect(0, 127242, '\p{^Is_Category=_	no}', "");
Expect(0, 127242, '\P{Is_Category=_	no}', "");
Expect(1, 127242, '\P{^Is_Category=_	no}', "");
Expect(0, 127243, '\p{Is_Category=_	no}', "");
Expect(1, 127243, '\p{^Is_Category=_	no}', "");
Expect(1, 127243, '\P{Is_Category=_	no}', "");
Expect(0, 127243, '\P{^Is_Category=_	no}', "");
Error('\p{General_Category=:=- other}');
Error('\P{General_Category=:=- other}');
Expect(1, 918000, '\p{General_Category:   other}', "");
Expect(0, 918000, '\p{^General_Category:   other}', "");
Expect(0, 918000, '\P{General_Category:   other}', "");
Expect(1, 918000, '\P{^General_Category:   other}', "");
Expect(0, 917999, '\p{General_Category:   other}', "");
Expect(1, 917999, '\p{^General_Category:   other}', "");
Expect(1, 917999, '\P{General_Category:   other}', "");
Expect(0, 917999, '\P{^General_Category:   other}', "");
Expect(1, 918000, '\p{General_Category=	 Other}', "");
Expect(0, 918000, '\p{^General_Category=	 Other}', "");
Expect(0, 918000, '\P{General_Category=	 Other}', "");
Expect(1, 918000, '\P{^General_Category=	 Other}', "");
Expect(0, 917999, '\p{General_Category=	 Other}', "");
Expect(1, 917999, '\p{^General_Category=	 Other}', "");
Expect(1, 917999, '\P{General_Category=	 Other}', "");
Expect(0, 917999, '\P{^General_Category=	 Other}', "");
Error('\p{Gc:		-C/a/}');
Error('\P{Gc:		-C/a/}');
Expect(1, 918000, '\p{Gc=c}', "");
Expect(0, 918000, '\p{^Gc=c}', "");
Expect(0, 918000, '\P{Gc=c}', "");
Expect(1, 918000, '\P{^Gc=c}', "");
Expect(0, 917999, '\p{Gc=c}', "");
Expect(1, 917999, '\p{^Gc=c}', "");
Expect(1, 917999, '\P{Gc=c}', "");
Expect(0, 917999, '\P{^Gc=c}', "");
Expect(1, 918000, '\p{Gc:   _	C}', "");
Expect(0, 918000, '\p{^Gc:   _	C}', "");
Expect(0, 918000, '\P{Gc:   _	C}', "");
Expect(1, 918000, '\P{^Gc:   _	C}', "");
Expect(0, 917999, '\p{Gc:   _	C}', "");
Expect(1, 917999, '\p{^Gc:   _	C}', "");
Expect(1, 917999, '\P{Gc:   _	C}', "");
Expect(0, 917999, '\P{^Gc:   _	C}', "");
Error('\p{Category=/a/_Other}');
Error('\P{Category=/a/_Other}');
Expect(1, 918000, '\p{Category=other}', "");
Expect(0, 918000, '\p{^Category=other}', "");
Expect(0, 918000, '\P{Category=other}', "");
Expect(1, 918000, '\P{^Category=other}', "");
Expect(0, 917999, '\p{Category=other}', "");
Expect(1, 917999, '\p{^Category=other}', "");
Expect(1, 917999, '\P{Category=other}', "");
Expect(0, 917999, '\P{^Category=other}', "");
Expect(1, 918000, '\p{Category=_-Other}', "");
Expect(0, 918000, '\p{^Category=_-Other}', "");
Expect(0, 918000, '\P{Category=_-Other}', "");
Expect(1, 918000, '\P{^Category=_-Other}', "");
Expect(0, 917999, '\p{Category=_-Other}', "");
Expect(1, 917999, '\p{^Category=_-Other}', "");
Expect(1, 917999, '\P{Category=_-Other}', "");
Expect(0, 917999, '\P{^Category=_-Other}', "");
Error('\p{Is_General_Category=:=-	C}');
Error('\P{Is_General_Category=:=-	C}');
Expect(1, 918000, '\p{Is_General_Category=c}', "");
Expect(0, 918000, '\p{^Is_General_Category=c}', "");
Expect(0, 918000, '\P{Is_General_Category=c}', "");
Expect(1, 918000, '\P{^Is_General_Category=c}', "");
Expect(0, 917999, '\p{Is_General_Category=c}', "");
Expect(1, 917999, '\p{^Is_General_Category=c}', "");
Expect(1, 917999, '\P{Is_General_Category=c}', "");
Expect(0, 917999, '\P{^Is_General_Category=c}', "");
Expect(1, 918000, '\p{Is_General_Category=-_C}', "");
Expect(0, 918000, '\p{^Is_General_Category=-_C}', "");
Expect(0, 918000, '\P{Is_General_Category=-_C}', "");
Expect(1, 918000, '\P{^Is_General_Category=-_C}', "");
Expect(0, 917999, '\p{Is_General_Category=-_C}', "");
Expect(1, 917999, '\p{^Is_General_Category=-_C}', "");
Expect(1, 917999, '\P{Is_General_Category=-_C}', "");
Expect(0, 917999, '\P{^Is_General_Category=-_C}', "");
Error('\p{Is_Gc= :=OTHER}');
Error('\P{Is_Gc= :=OTHER}');
Expect(1, 918000, '\p{Is_Gc=other}', "");
Expect(0, 918000, '\p{^Is_Gc=other}', "");
Expect(0, 918000, '\P{Is_Gc=other}', "");
Expect(1, 918000, '\P{^Is_Gc=other}', "");
Expect(0, 917999, '\p{Is_Gc=other}', "");
Expect(1, 917999, '\p{^Is_Gc=other}', "");
Expect(1, 917999, '\P{Is_Gc=other}', "");
Expect(0, 917999, '\P{^Is_Gc=other}', "");
Expect(1, 918000, '\p{Is_Gc=	-Other}', "");
Expect(0, 918000, '\p{^Is_Gc=	-Other}', "");
Expect(0, 918000, '\P{Is_Gc=	-Other}', "");
Expect(1, 918000, '\P{^Is_Gc=	-Other}', "");
Expect(0, 917999, '\p{Is_Gc=	-Other}', "");
Expect(1, 917999, '\p{^Is_Gc=	-Other}', "");
Expect(1, 917999, '\P{Is_Gc=	-Other}', "");
Expect(0, 917999, '\P{^Is_Gc=	-Other}', "");
Error('\p{Is_Category=_:=C}');
Error('\P{Is_Category=_:=C}');
Expect(1, 918000, '\p{Is_Category=c}', "");
Expect(0, 918000, '\p{^Is_Category=c}', "");
Expect(0, 918000, '\P{Is_Category=c}', "");
Expect(1, 918000, '\P{^Is_Category=c}', "");
Expect(0, 917999, '\p{Is_Category=c}', "");
Expect(1, 917999, '\p{^Is_Category=c}', "");
Expect(1, 917999, '\P{Is_Category=c}', "");
Expect(0, 917999, '\P{^Is_Category=c}', "");
Expect(1, 918000, '\p{Is_Category= _C}', "");
Expect(0, 918000, '\p{^Is_Category= _C}', "");
Expect(0, 918000, '\P{Is_Category= _C}', "");
Expect(1, 918000, '\P{^Is_Category= _C}', "");
Expect(0, 917999, '\p{Is_Category= _C}', "");
Expect(1, 917999, '\p{^Is_Category= _C}', "");
Expect(1, 917999, '\P{Is_Category= _C}', "");
Expect(0, 917999, '\P{^Is_Category= _C}', "");
Error('\p{General_Category:	/a/ -Enclosing_Mark}');
Error('\P{General_Category:	/a/ -Enclosing_Mark}');
Expect(1, 42610, '\p{General_Category=enclosingmark}', "");
Expect(0, 42610, '\p{^General_Category=enclosingmark}', "");
Expect(0, 42610, '\P{General_Category=enclosingmark}', "");
Expect(1, 42610, '\P{^General_Category=enclosingmark}', "");
Expect(0, 42611, '\p{General_Category=enclosingmark}', "");
Expect(1, 42611, '\p{^General_Category=enclosingmark}', "");
Expect(1, 42611, '\P{General_Category=enclosingmark}', "");
Expect(0, 42611, '\P{^General_Category=enclosingmark}', "");
Expect(1, 42610, '\p{General_Category= Enclosing_MARK}', "");
Expect(0, 42610, '\p{^General_Category= Enclosing_MARK}', "");
Expect(0, 42610, '\P{General_Category= Enclosing_MARK}', "");
Expect(1, 42610, '\P{^General_Category= Enclosing_MARK}', "");
Expect(0, 42611, '\p{General_Category= Enclosing_MARK}', "");
Expect(1, 42611, '\p{^General_Category= Enclosing_MARK}', "");
Expect(1, 42611, '\P{General_Category= Enclosing_MARK}', "");
Expect(0, 42611, '\P{^General_Category= Enclosing_MARK}', "");
Error('\p{Gc=/a/-_me}');
Error('\P{Gc=/a/-_me}');
Expect(1, 42610, '\p{Gc=me}', "");
Expect(0, 42610, '\p{^Gc=me}', "");
Expect(0, 42610, '\P{Gc=me}', "");
Expect(1, 42610, '\P{^Gc=me}', "");
Expect(0, 42611, '\p{Gc=me}', "");
Expect(1, 42611, '\p{^Gc=me}', "");
Expect(1, 42611, '\P{Gc=me}', "");
Expect(0, 42611, '\P{^Gc=me}', "");
Expect(1, 42610, '\p{Gc=_ Me}', "");
Expect(0, 42610, '\p{^Gc=_ Me}', "");
Expect(0, 42610, '\P{Gc=_ Me}', "");
Expect(1, 42610, '\P{^Gc=_ Me}', "");
Expect(0, 42611, '\p{Gc=_ Me}', "");
Expect(1, 42611, '\p{^Gc=_ Me}', "");
Expect(1, 42611, '\P{Gc=_ Me}', "");
Expect(0, 42611, '\P{^Gc=_ Me}', "");
Error('\p{Category=:=Enclosing_Mark}');
Error('\P{Category=:=Enclosing_Mark}');
Expect(1, 42610, '\p{Category: enclosingmark}', "");
Expect(0, 42610, '\p{^Category: enclosingmark}', "");
Expect(0, 42610, '\P{Category: enclosingmark}', "");
Expect(1, 42610, '\P{^Category: enclosingmark}', "");
Expect(0, 42611, '\p{Category: enclosingmark}', "");
Expect(1, 42611, '\p{^Category: enclosingmark}', "");
Expect(1, 42611, '\P{Category: enclosingmark}', "");
Expect(0, 42611, '\P{^Category: enclosingmark}', "");
Expect(1, 42610, '\p{Category= _Enclosing_MARK}', "");
Expect(0, 42610, '\p{^Category= _Enclosing_MARK}', "");
Expect(0, 42610, '\P{Category= _Enclosing_MARK}', "");
Expect(1, 42610, '\P{^Category= _Enclosing_MARK}', "");
Expect(0, 42611, '\p{Category= _Enclosing_MARK}', "");
Expect(1, 42611, '\p{^Category= _Enclosing_MARK}', "");
Expect(1, 42611, '\P{Category= _Enclosing_MARK}', "");
Expect(0, 42611, '\P{^Category= _Enclosing_MARK}', "");
Error('\p{Is_General_Category= _me/a/}');
Error('\P{Is_General_Category= _me/a/}');
Expect(1, 42610, '\p{Is_General_Category=me}', "");
Expect(0, 42610, '\p{^Is_General_Category=me}', "");
Expect(0, 42610, '\P{Is_General_Category=me}', "");
Expect(1, 42610, '\P{^Is_General_Category=me}', "");
Expect(0, 42611, '\p{Is_General_Category=me}', "");
Expect(1, 42611, '\p{^Is_General_Category=me}', "");
Expect(1, 42611, '\P{Is_General_Category=me}', "");
Expect(0, 42611, '\P{^Is_General_Category=me}', "");
Expect(1, 42610, '\p{Is_General_Category= Me}', "");
Expect(0, 42610, '\p{^Is_General_Category= Me}', "");
Expect(0, 42610, '\P{Is_General_Category= Me}', "");
Expect(1, 42610, '\P{^Is_General_Category= Me}', "");
Expect(0, 42611, '\p{Is_General_Category= Me}', "");
Expect(1, 42611, '\p{^Is_General_Category= Me}', "");
Expect(1, 42611, '\P{Is_General_Category= Me}', "");
Expect(0, 42611, '\P{^Is_General_Category= Me}', "");
Error('\p{Is_Gc=:=ENCLOSING_MARK}');
Error('\P{Is_Gc=:=ENCLOSING_MARK}');
Expect(1, 42610, '\p{Is_Gc=enclosingmark}', "");
Expect(0, 42610, '\p{^Is_Gc=enclosingmark}', "");
Expect(0, 42610, '\P{Is_Gc=enclosingmark}', "");
Expect(1, 42610, '\P{^Is_Gc=enclosingmark}', "");
Expect(0, 42611, '\p{Is_Gc=enclosingmark}', "");
Expect(1, 42611, '\p{^Is_Gc=enclosingmark}', "");
Expect(1, 42611, '\P{Is_Gc=enclosingmark}', "");
Expect(0, 42611, '\P{^Is_Gc=enclosingmark}', "");
Expect(1, 42610, '\p{Is_Gc:_ENCLOSING_Mark}', "");
Expect(0, 42610, '\p{^Is_Gc:_ENCLOSING_Mark}', "");
Expect(0, 42610, '\P{Is_Gc:_ENCLOSING_Mark}', "");
Expect(1, 42610, '\P{^Is_Gc:_ENCLOSING_Mark}', "");
Expect(0, 42611, '\p{Is_Gc:_ENCLOSING_Mark}', "");
Expect(1, 42611, '\p{^Is_Gc:_ENCLOSING_Mark}', "");
Expect(1, 42611, '\P{Is_Gc:_ENCLOSING_Mark}', "");
Expect(0, 42611, '\P{^Is_Gc:_ENCLOSING_Mark}', "");
Error('\p{Is_Category=-Me:=}');
Error('\P{Is_Category=-Me:=}');
Expect(1, 42610, '\p{Is_Category: me}', "");
Expect(0, 42610, '\p{^Is_Category: me}', "");
Expect(0, 42610, '\P{Is_Category: me}', "");
Expect(1, 42610, '\P{^Is_Category: me}', "");
Expect(0, 42611, '\p{Is_Category: me}', "");
Expect(1, 42611, '\p{^Is_Category: me}', "");
Expect(1, 42611, '\P{Is_Category: me}', "");
Expect(0, 42611, '\P{^Is_Category: me}', "");
Expect(1, 42610, '\p{Is_Category= _me}', "");
Expect(0, 42610, '\p{^Is_Category= _me}', "");
Expect(0, 42610, '\P{Is_Category= _me}', "");
Expect(1, 42610, '\P{^Is_Category= _me}', "");
Expect(0, 42611, '\p{Is_Category= _me}', "");
Expect(1, 42611, '\p{^Is_Category= _me}', "");
Expect(1, 42611, '\P{Is_Category= _me}', "");
Expect(0, 42611, '\P{^Is_Category= _me}', "");
Error('\p{General_Category=:=	 Titlecase_Letter}');
Error('\P{General_Category=:=	 Titlecase_Letter}');
Expect(1, 8188, '\p{General_Category=titlecaseletter}', "");
Expect(0, 8188, '\p{^General_Category=titlecaseletter}', "");
Expect(0, 8188, '\P{General_Category=titlecaseletter}', "");
Expect(1, 8188, '\P{^General_Category=titlecaseletter}', "");
Expect(0, 8189, '\p{General_Category=titlecaseletter}', "");
Expect(1, 8189, '\p{^General_Category=titlecaseletter}', "");
Expect(1, 8189, '\P{General_Category=titlecaseletter}', "");
Expect(0, 8189, '\P{^General_Category=titlecaseletter}', "");
Expect(1, 8188, '\p{General_Category= 	titlecase_Letter}', "");
Expect(0, 8188, '\p{^General_Category= 	titlecase_Letter}', "");
Expect(0, 8188, '\P{General_Category= 	titlecase_Letter}', "");
Expect(1, 8188, '\P{^General_Category= 	titlecase_Letter}', "");
Expect(0, 8189, '\p{General_Category= 	titlecase_Letter}', "");
Expect(1, 8189, '\p{^General_Category= 	titlecase_Letter}', "");
Expect(1, 8189, '\P{General_Category= 	titlecase_Letter}', "");
Expect(0, 8189, '\P{^General_Category= 	titlecase_Letter}', "");
Error('\p{Gc=:=Lt}');
Error('\P{Gc=:=Lt}');
Expect(1, 8188, '\p{Gc=lt}', "");
Expect(0, 8188, '\p{^Gc=lt}', "");
Expect(0, 8188, '\P{Gc=lt}', "");
Expect(1, 8188, '\P{^Gc=lt}', "");
Expect(0, 8189, '\p{Gc=lt}', "");
Expect(1, 8189, '\p{^Gc=lt}', "");
Expect(1, 8189, '\P{Gc=lt}', "");
Expect(0, 8189, '\P{^Gc=lt}', "");
Expect(1, 8188, '\p{Gc= LT}', "");
Expect(0, 8188, '\p{^Gc= LT}', "");
Expect(0, 8188, '\P{Gc= LT}', "");
Expect(1, 8188, '\P{^Gc= LT}', "");
Expect(0, 8189, '\p{Gc= LT}', "");
Expect(1, 8189, '\p{^Gc= LT}', "");
Expect(1, 8189, '\P{Gc= LT}', "");
Expect(0, 8189, '\P{^Gc= LT}', "");
Error('\p{Category= _titlecase_letter/a/}');
Error('\P{Category= _titlecase_letter/a/}');
Expect(1, 8188, '\p{Category=titlecaseletter}', "");
Expect(0, 8188, '\p{^Category=titlecaseletter}', "");
Expect(0, 8188, '\P{Category=titlecaseletter}', "");
Expect(1, 8188, '\P{^Category=titlecaseletter}', "");
Expect(0, 8189, '\p{Category=titlecaseletter}', "");
Expect(1, 8189, '\p{^Category=titlecaseletter}', "");
Expect(1, 8189, '\P{Category=titlecaseletter}', "");
Expect(0, 8189, '\P{^Category=titlecaseletter}', "");
Expect(1, 8188, '\p{Category= _TITLECASE_letter}', "");
Expect(0, 8188, '\p{^Category= _TITLECASE_letter}', "");
Expect(0, 8188, '\P{Category= _TITLECASE_letter}', "");
Expect(1, 8188, '\P{^Category= _TITLECASE_letter}', "");
Expect(0, 8189, '\p{Category= _TITLECASE_letter}', "");
Expect(1, 8189, '\p{^Category= _TITLECASE_letter}', "");
Expect(1, 8189, '\P{Category= _TITLECASE_letter}', "");
Expect(0, 8189, '\P{^Category= _TITLECASE_letter}', "");
Error('\p{Is_General_Category=-:=lt}');
Error('\P{Is_General_Category=-:=lt}');
Expect(1, 8188, '\p{Is_General_Category=lt}', "");
Expect(0, 8188, '\p{^Is_General_Category=lt}', "");
Expect(0, 8188, '\P{Is_General_Category=lt}', "");
Expect(1, 8188, '\P{^Is_General_Category=lt}', "");
Expect(0, 8189, '\p{Is_General_Category=lt}', "");
Expect(1, 8189, '\p{^Is_General_Category=lt}', "");
Expect(1, 8189, '\P{Is_General_Category=lt}', "");
Expect(0, 8189, '\P{^Is_General_Category=lt}', "");
Expect(1, 8188, '\p{Is_General_Category=Lt}', "");
Expect(0, 8188, '\p{^Is_General_Category=Lt}', "");
Expect(0, 8188, '\P{Is_General_Category=Lt}', "");
Expect(1, 8188, '\P{^Is_General_Category=Lt}', "");
Expect(0, 8189, '\p{Is_General_Category=Lt}', "");
Expect(1, 8189, '\p{^Is_General_Category=Lt}', "");
Expect(1, 8189, '\P{Is_General_Category=Lt}', "");
Expect(0, 8189, '\P{^Is_General_Category=Lt}', "");
Error('\p{Is_Gc: /a/_ Titlecase_letter}');
Error('\P{Is_Gc: /a/_ Titlecase_letter}');
Expect(1, 8188, '\p{Is_Gc=titlecaseletter}', "");
Expect(0, 8188, '\p{^Is_Gc=titlecaseletter}', "");
Expect(0, 8188, '\P{Is_Gc=titlecaseletter}', "");
Expect(1, 8188, '\P{^Is_Gc=titlecaseletter}', "");
Expect(0, 8189, '\p{Is_Gc=titlecaseletter}', "");
Expect(1, 8189, '\p{^Is_Gc=titlecaseletter}', "");
Expect(1, 8189, '\P{Is_Gc=titlecaseletter}', "");
Expect(0, 8189, '\P{^Is_Gc=titlecaseletter}', "");
Expect(1, 8188, '\p{Is_Gc=_titlecase_LETTER}', "");
Expect(0, 8188, '\p{^Is_Gc=_titlecase_LETTER}', "");
Expect(0, 8188, '\P{Is_Gc=_titlecase_LETTER}', "");
Expect(1, 8188, '\P{^Is_Gc=_titlecase_LETTER}', "");
Expect(0, 8189, '\p{Is_Gc=_titlecase_LETTER}', "");
Expect(1, 8189, '\p{^Is_Gc=_titlecase_LETTER}', "");
Expect(1, 8189, '\P{Is_Gc=_titlecase_LETTER}', "");
Expect(0, 8189, '\P{^Is_Gc=_titlecase_LETTER}', "");
Error('\p{Is_Category:		/a/LT}');
Error('\P{Is_Category:		/a/LT}');
Expect(1, 8188, '\p{Is_Category=lt}', "");
Expect(0, 8188, '\p{^Is_Category=lt}', "");
Expect(0, 8188, '\P{Is_Category=lt}', "");
Expect(1, 8188, '\P{^Is_Category=lt}', "");
Expect(0, 8189, '\p{Is_Category=lt}', "");
Expect(1, 8189, '\p{^Is_Category=lt}', "");
Expect(1, 8189, '\P{Is_Category=lt}', "");
Expect(0, 8189, '\P{^Is_Category=lt}', "");
Expect(1, 8188, '\p{Is_Category=  Lt}', "");
Expect(0, 8188, '\p{^Is_Category=  Lt}', "");
Expect(0, 8188, '\P{Is_Category=  Lt}', "");
Expect(1, 8188, '\P{^Is_Category=  Lt}', "");
Expect(0, 8189, '\p{Is_Category=  Lt}', "");
Expect(1, 8189, '\p{^Is_Category=  Lt}', "");
Expect(1, 8189, '\P{Is_Category=  Lt}', "");
Expect(0, 8189, '\P{^Is_Category=  Lt}', "");
Error('\p{General_Category=-/a/Punctuation}');
Error('\P{General_Category=-/a/Punctuation}');
Expect(1, 74867, '\p{General_Category=punctuation}', "");
Expect(0, 74867, '\p{^General_Category=punctuation}', "");
Expect(0, 74867, '\P{General_Category=punctuation}', "");
Expect(1, 74867, '\P{^General_Category=punctuation}', "");
Expect(0, 74868, '\p{General_Category=punctuation}', "");
Expect(1, 74868, '\p{^General_Category=punctuation}', "");
Expect(1, 74868, '\P{General_Category=punctuation}', "");
Expect(0, 74868, '\P{^General_Category=punctuation}', "");
Expect(1, 74867, '\p{General_Category=- Punctuation}', "");
Expect(0, 74867, '\p{^General_Category=- Punctuation}', "");
Expect(0, 74867, '\P{General_Category=- Punctuation}', "");
Expect(1, 74867, '\P{^General_Category=- Punctuation}', "");
Expect(0, 74868, '\p{General_Category=- Punctuation}', "");
Expect(1, 74868, '\p{^General_Category=- Punctuation}', "");
Expect(1, 74868, '\P{General_Category=- Punctuation}', "");
Expect(0, 74868, '\P{^General_Category=- Punctuation}', "");
Error('\p{Gc=-:=p}');
Error('\P{Gc=-:=p}');
Expect(1, 74867, '\p{Gc=p}', "");
Expect(0, 74867, '\p{^Gc=p}', "");
Expect(0, 74867, '\P{Gc=p}', "");
Expect(1, 74867, '\P{^Gc=p}', "");
Expect(0, 74868, '\p{Gc=p}', "");
Expect(1, 74868, '\p{^Gc=p}', "");
Expect(1, 74868, '\P{Gc=p}', "");
Expect(0, 74868, '\P{^Gc=p}', "");
Expect(1, 74867, '\p{Gc= _P}', "");
Expect(0, 74867, '\p{^Gc= _P}', "");
Expect(0, 74867, '\P{Gc= _P}', "");
Expect(1, 74867, '\P{^Gc= _P}', "");
Expect(0, 74868, '\p{Gc= _P}', "");
Expect(1, 74868, '\p{^Gc= _P}', "");
Expect(1, 74868, '\P{Gc= _P}', "");
Expect(0, 74868, '\P{^Gc= _P}', "");
Error('\p{Category=/a/		punct}');
Error('\P{Category=/a/		punct}');
Expect(1, 74867, '\p{Category:   punct}', "");
Expect(0, 74867, '\p{^Category:   punct}', "");
Expect(0, 74867, '\P{Category:   punct}', "");
Expect(1, 74867, '\P{^Category:   punct}', "");
Expect(0, 74868, '\p{Category:   punct}', "");
Expect(1, 74868, '\p{^Category:   punct}', "");
Expect(1, 74868, '\P{Category:   punct}', "");
Expect(0, 74868, '\P{^Category:   punct}', "");
Expect(1, 74867, '\p{Category= _Punct}', "");
Expect(0, 74867, '\p{^Category= _Punct}', "");
Expect(0, 74867, '\P{Category= _Punct}', "");
Expect(1, 74867, '\P{^Category= _Punct}', "");
Expect(0, 74868, '\p{Category= _Punct}', "");
Expect(1, 74868, '\p{^Category= _Punct}', "");
Expect(1, 74868, '\P{Category= _Punct}', "");
Expect(0, 74868, '\P{^Category= _Punct}', "");
Error('\p{Is_General_Category=	/a/PUNCTUATION}');
Error('\P{Is_General_Category=	/a/PUNCTUATION}');
Expect(1, 74867, '\p{Is_General_Category=punctuation}', "");
Expect(0, 74867, '\p{^Is_General_Category=punctuation}', "");
Expect(0, 74867, '\P{Is_General_Category=punctuation}', "");
Expect(1, 74867, '\P{^Is_General_Category=punctuation}', "");
Expect(0, 74868, '\p{Is_General_Category=punctuation}', "");
Expect(1, 74868, '\p{^Is_General_Category=punctuation}', "");
Expect(1, 74868, '\P{Is_General_Category=punctuation}', "");
Expect(0, 74868, '\P{^Is_General_Category=punctuation}', "");
Expect(1, 74867, '\p{Is_General_Category= Punctuation}', "");
Expect(0, 74867, '\p{^Is_General_Category= Punctuation}', "");
Expect(0, 74867, '\P{Is_General_Category= Punctuation}', "");
Expect(1, 74867, '\P{^Is_General_Category= Punctuation}', "");
Expect(0, 74868, '\p{Is_General_Category= Punctuation}', "");
Expect(1, 74868, '\p{^Is_General_Category= Punctuation}', "");
Expect(1, 74868, '\P{Is_General_Category= Punctuation}', "");
Expect(0, 74868, '\P{^Is_General_Category= Punctuation}', "");
Error('\p{Is_Gc=-:=P}');
Error('\P{Is_Gc=-:=P}');
Expect(1, 74867, '\p{Is_Gc=p}', "");
Expect(0, 74867, '\p{^Is_Gc=p}', "");
Expect(0, 74867, '\P{Is_Gc=p}', "");
Expect(1, 74867, '\P{^Is_Gc=p}', "");
Expect(0, 74868, '\p{Is_Gc=p}', "");
Expect(1, 74868, '\p{^Is_Gc=p}', "");
Expect(1, 74868, '\P{Is_Gc=p}', "");
Expect(0, 74868, '\P{^Is_Gc=p}', "");
Expect(1, 74867, '\p{Is_Gc= 	p}', "");
Expect(0, 74867, '\p{^Is_Gc= 	p}', "");
Expect(0, 74867, '\P{Is_Gc= 	p}', "");
Expect(1, 74867, '\P{^Is_Gc= 	p}', "");
Expect(0, 74868, '\p{Is_Gc= 	p}', "");
Expect(1, 74868, '\p{^Is_Gc= 	p}', "");
Expect(1, 74868, '\P{Is_Gc= 	p}', "");
Expect(0, 74868, '\P{^Is_Gc= 	p}', "");
Error('\p{Is_Category=_Punct:=}');
Error('\P{Is_Category=_Punct:=}');
Expect(1, 74867, '\p{Is_Category=punct}', "");
Expect(0, 74867, '\p{^Is_Category=punct}', "");
Expect(0, 74867, '\P{Is_Category=punct}', "");
Expect(1, 74867, '\P{^Is_Category=punct}', "");
Expect(0, 74868, '\p{Is_Category=punct}', "");
Expect(1, 74868, '\p{^Is_Category=punct}', "");
Expect(1, 74868, '\P{Is_Category=punct}', "");
Expect(0, 74868, '\P{^Is_Category=punct}', "");
Expect(1, 74867, '\p{Is_Category=_	Punct}', "");
Expect(0, 74867, '\p{^Is_Category=_	Punct}', "");
Expect(0, 74867, '\P{Is_Category=_	Punct}', "");
Expect(1, 74867, '\P{^Is_Category=_	Punct}', "");
Expect(0, 74868, '\p{Is_Category=_	Punct}', "");
Expect(1, 74868, '\p{^Is_Category=_	Punct}', "");
Expect(1, 74868, '\P{Is_Category=_	Punct}', "");
Expect(0, 74868, '\P{^Is_Category=_	Punct}', "");
Error('\p{General_Category=:=	_Uppercase_LETTER}');
Error('\P{General_Category=:=	_Uppercase_LETTER}');
Expect(1, 120778, '\p{General_Category=uppercaseletter}', "");
Expect(0, 120778, '\p{^General_Category=uppercaseletter}', "");
Expect(0, 120778, '\P{General_Category=uppercaseletter}', "");
Expect(1, 120778, '\P{^General_Category=uppercaseletter}', "");
Expect(0, 120779, '\p{General_Category=uppercaseletter}', "");
Expect(1, 120779, '\p{^General_Category=uppercaseletter}', "");
Expect(1, 120779, '\P{General_Category=uppercaseletter}', "");
Expect(0, 120779, '\P{^General_Category=uppercaseletter}', "");
Expect(1, 120778, '\p{General_Category=_ uppercase_Letter}', "");
Expect(0, 120778, '\p{^General_Category=_ uppercase_Letter}', "");
Expect(0, 120778, '\P{General_Category=_ uppercase_Letter}', "");
Expect(1, 120778, '\P{^General_Category=_ uppercase_Letter}', "");
Expect(0, 120779, '\p{General_Category=_ uppercase_Letter}', "");
Expect(1, 120779, '\p{^General_Category=_ uppercase_Letter}', "");
Expect(1, 120779, '\P{General_Category=_ uppercase_Letter}', "");
Expect(0, 120779, '\P{^General_Category=_ uppercase_Letter}', "");
Error('\p{Gc=:= 	LU}');
Error('\P{Gc=:= 	LU}');
Expect(1, 120778, '\p{Gc=lu}', "");
Expect(0, 120778, '\p{^Gc=lu}', "");
Expect(0, 120778, '\P{Gc=lu}', "");
Expect(1, 120778, '\P{^Gc=lu}', "");
Expect(0, 120779, '\p{Gc=lu}', "");
Expect(1, 120779, '\p{^Gc=lu}', "");
Expect(1, 120779, '\P{Gc=lu}', "");
Expect(0, 120779, '\P{^Gc=lu}', "");
Expect(1, 120778, '\p{Gc=_lu}', "");
Expect(0, 120778, '\p{^Gc=_lu}', "");
Expect(0, 120778, '\P{Gc=_lu}', "");
Expect(1, 120778, '\P{^Gc=_lu}', "");
Expect(0, 120779, '\p{Gc=_lu}', "");
Expect(1, 120779, '\p{^Gc=_lu}', "");
Expect(1, 120779, '\P{Gc=_lu}', "");
Expect(0, 120779, '\P{^Gc=_lu}', "");
Error('\p{Category=-/a/Uppercase_Letter}');
Error('\P{Category=-/a/Uppercase_Letter}');
Expect(1, 120778, '\p{Category=uppercaseletter}', "");
Expect(0, 120778, '\p{^Category=uppercaseletter}', "");
Expect(0, 120778, '\P{Category=uppercaseletter}', "");
Expect(1, 120778, '\P{^Category=uppercaseletter}', "");
Expect(0, 120779, '\p{Category=uppercaseletter}', "");
Expect(1, 120779, '\p{^Category=uppercaseletter}', "");
Expect(1, 120779, '\P{Category=uppercaseletter}', "");
Expect(0, 120779, '\P{^Category=uppercaseletter}', "");
Expect(1, 120778, '\p{Category:     Uppercase_letter}', "");
Expect(0, 120778, '\p{^Category:     Uppercase_letter}', "");
Expect(0, 120778, '\P{Category:     Uppercase_letter}', "");
Expect(1, 120778, '\P{^Category:     Uppercase_letter}', "");
Expect(0, 120779, '\p{Category:     Uppercase_letter}', "");
Expect(1, 120779, '\p{^Category:     Uppercase_letter}', "");
Expect(1, 120779, '\P{Category:     Uppercase_letter}', "");
Expect(0, 120779, '\P{^Category:     Uppercase_letter}', "");
Error('\p{Is_General_Category= :=Lu}');
Error('\P{Is_General_Category= :=Lu}');
Expect(1, 120778, '\p{Is_General_Category: lu}', "");
Expect(0, 120778, '\p{^Is_General_Category: lu}', "");
Expect(0, 120778, '\P{Is_General_Category: lu}', "");
Expect(1, 120778, '\P{^Is_General_Category: lu}', "");
Expect(0, 120779, '\p{Is_General_Category: lu}', "");
Expect(1, 120779, '\p{^Is_General_Category: lu}', "");
Expect(1, 120779, '\P{Is_General_Category: lu}', "");
Expect(0, 120779, '\P{^Is_General_Category: lu}', "");
Expect(1, 120778, '\p{Is_General_Category=-_lu}', "");
Expect(0, 120778, '\p{^Is_General_Category=-_lu}', "");
Expect(0, 120778, '\P{Is_General_Category=-_lu}', "");
Expect(1, 120778, '\P{^Is_General_Category=-_lu}', "");
Expect(0, 120779, '\p{Is_General_Category=-_lu}', "");
Expect(1, 120779, '\p{^Is_General_Category=-_lu}', "");
Expect(1, 120779, '\P{Is_General_Category=-_lu}', "");
Expect(0, 120779, '\P{^Is_General_Category=-_lu}', "");
Error('\p{Is_Gc= :=Uppercase_Letter}');
Error('\P{Is_Gc= :=Uppercase_Letter}');
Expect(1, 120778, '\p{Is_Gc=uppercaseletter}', "");
Expect(0, 120778, '\p{^Is_Gc=uppercaseletter}', "");
Expect(0, 120778, '\P{Is_Gc=uppercaseletter}', "");
Expect(1, 120778, '\P{^Is_Gc=uppercaseletter}', "");
Expect(0, 120779, '\p{Is_Gc=uppercaseletter}', "");
Expect(1, 120779, '\p{^Is_Gc=uppercaseletter}', "");
Expect(1, 120779, '\P{Is_Gc=uppercaseletter}', "");
Expect(0, 120779, '\P{^Is_Gc=uppercaseletter}', "");
Expect(1, 120778, '\p{Is_Gc=	-UPPERCASE_Letter}', "");
Expect(0, 120778, '\p{^Is_Gc=	-UPPERCASE_Letter}', "");
Expect(0, 120778, '\P{Is_Gc=	-UPPERCASE_Letter}', "");
Expect(1, 120778, '\P{^Is_Gc=	-UPPERCASE_Letter}', "");
Expect(0, 120779, '\p{Is_Gc=	-UPPERCASE_Letter}', "");
Expect(1, 120779, '\p{^Is_Gc=	-UPPERCASE_Letter}', "");
Expect(1, 120779, '\P{Is_Gc=	-UPPERCASE_Letter}', "");
Expect(0, 120779, '\P{^Is_Gc=	-UPPERCASE_Letter}', "");
Error('\p{Is_Category=/a/_lu}');
Error('\P{Is_Category=/a/_lu}');
Expect(1, 120778, '\p{Is_Category=lu}', "");
Expect(0, 120778, '\p{^Is_Category=lu}', "");
Expect(0, 120778, '\P{Is_Category=lu}', "");
Expect(1, 120778, '\P{^Is_Category=lu}', "");
Expect(0, 120779, '\p{Is_Category=lu}', "");
Expect(1, 120779, '\p{^Is_Category=lu}', "");
Expect(1, 120779, '\P{Is_Category=lu}', "");
Expect(0, 120779, '\P{^Is_Category=lu}', "");
Expect(1, 120778, '\p{Is_Category=_-LU}', "");
Expect(0, 120778, '\p{^Is_Category=_-LU}', "");
Expect(0, 120778, '\P{Is_Category=_-LU}', "");
Expect(1, 120778, '\P{^Is_Category=_-LU}', "");
Expect(0, 120779, '\p{Is_Category=_-LU}', "");
Expect(1, 120779, '\p{^Is_Category=_-LU}', "");
Expect(1, 120779, '\P{Is_Category=_-LU}', "");
Expect(0, 120779, '\P{^Is_Category=_-LU}', "");
Error('\p{General_Category=:=  Number}');
Error('\P{General_Category=:=  Number}');
Expect(1, 127242, '\p{General_Category=number}', "");
Expect(0, 127242, '\p{^General_Category=number}', "");
Expect(0, 127242, '\P{General_Category=number}', "");
Expect(1, 127242, '\P{^General_Category=number}', "");
Expect(0, 127243, '\p{General_Category=number}', "");
Expect(1, 127243, '\p{^General_Category=number}', "");
Expect(1, 127243, '\P{General_Category=number}', "");
Expect(0, 127243, '\P{^General_Category=number}', "");
Expect(1, 127242, '\p{General_Category=_	NUMBER}', "");
Expect(0, 127242, '\p{^General_Category=_	NUMBER}', "");
Expect(0, 127242, '\P{General_Category=_	NUMBER}', "");
Expect(1, 127242, '\P{^General_Category=_	NUMBER}', "");
Expect(0, 127243, '\p{General_Category=_	NUMBER}', "");
Expect(1, 127243, '\p{^General_Category=_	NUMBER}', "");
Expect(1, 127243, '\P{General_Category=_	NUMBER}', "");
Expect(0, 127243, '\P{^General_Category=_	NUMBER}', "");
Error('\p{Gc: 	-N:=}');
Error('\P{Gc: 	-N:=}');
Expect(1, 127242, '\p{Gc:n}', "");
Expect(0, 127242, '\p{^Gc:n}', "");
Expect(0, 127242, '\P{Gc:n}', "");
Expect(1, 127242, '\P{^Gc:n}', "");
Expect(0, 127243, '\p{Gc:n}', "");
Expect(1, 127243, '\p{^Gc:n}', "");
Expect(1, 127243, '\P{Gc:n}', "");
Expect(0, 127243, '\P{^Gc:n}', "");
Expect(1, 127242, '\p{Gc=_ N}', "");
Expect(0, 127242, '\p{^Gc=_ N}', "");
Expect(0, 127242, '\P{Gc=_ N}', "");
Expect(1, 127242, '\P{^Gc=_ N}', "");
Expect(0, 127243, '\p{Gc=_ N}', "");
Expect(1, 127243, '\p{^Gc=_ N}', "");
Expect(1, 127243, '\P{Gc=_ N}', "");
Expect(0, 127243, '\P{^Gc=_ N}', "");
Error('\p{Category=:= -number}');
Error('\P{Category=:= -number}');
Expect(1, 127242, '\p{Category=number}', "");
Expect(0, 127242, '\p{^Category=number}', "");
Expect(0, 127242, '\P{Category=number}', "");
Expect(1, 127242, '\P{^Category=number}', "");
Expect(0, 127243, '\p{Category=number}', "");
Expect(1, 127243, '\p{^Category=number}', "");
Expect(1, 127243, '\P{Category=number}', "");
Expect(0, 127243, '\P{^Category=number}', "");
Expect(1, 127242, '\p{Category= Number}', "");
Expect(0, 127242, '\p{^Category= Number}', "");
Expect(0, 127242, '\P{Category= Number}', "");
Expect(1, 127242, '\P{^Category= Number}', "");
Expect(0, 127243, '\p{Category= Number}', "");
Expect(1, 127243, '\p{^Category= Number}', "");
Expect(1, 127243, '\P{Category= Number}', "");
Expect(0, 127243, '\P{^Category= Number}', "");
Error('\p{Is_General_Category:/a/ -N}');
Error('\P{Is_General_Category:/a/ -N}');
Expect(1, 127242, '\p{Is_General_Category=n}', "");
Expect(0, 127242, '\p{^Is_General_Category=n}', "");
Expect(0, 127242, '\P{Is_General_Category=n}', "");
Expect(1, 127242, '\P{^Is_General_Category=n}', "");
Expect(0, 127243, '\p{Is_General_Category=n}', "");
Expect(1, 127243, '\p{^Is_General_Category=n}', "");
Expect(1, 127243, '\P{Is_General_Category=n}', "");
Expect(0, 127243, '\P{^Is_General_Category=n}', "");
Expect(1, 127242, '\p{Is_General_Category=-N}', "");
Expect(0, 127242, '\p{^Is_General_Category=-N}', "");
Expect(0, 127242, '\P{Is_General_Category=-N}', "");
Expect(1, 127242, '\P{^Is_General_Category=-N}', "");
Expect(0, 127243, '\p{Is_General_Category=-N}', "");
Expect(1, 127243, '\p{^Is_General_Category=-N}', "");
Expect(1, 127243, '\P{Is_General_Category=-N}', "");
Expect(0, 127243, '\P{^Is_General_Category=-N}', "");
Error('\p{Is_Gc=:= _number}');
Error('\P{Is_Gc=:= _number}');
Expect(1, 127242, '\p{Is_Gc=number}', "");
Expect(0, 127242, '\p{^Is_Gc=number}', "");
Expect(0, 127242, '\P{Is_Gc=number}', "");
Expect(1, 127242, '\P{^Is_Gc=number}', "");
Expect(0, 127243, '\p{Is_Gc=number}', "");
Expect(1, 127243, '\p{^Is_Gc=number}', "");
Expect(1, 127243, '\P{Is_Gc=number}', "");
Expect(0, 127243, '\P{^Is_Gc=number}', "");
Expect(1, 127242, '\p{Is_Gc=-NUMBER}', "");
Expect(0, 127242, '\p{^Is_Gc=-NUMBER}', "");
Expect(0, 127242, '\P{Is_Gc=-NUMBER}', "");
Expect(1, 127242, '\P{^Is_Gc=-NUMBER}', "");
Expect(0, 127243, '\p{Is_Gc=-NUMBER}', "");
Expect(1, 127243, '\p{^Is_Gc=-NUMBER}', "");
Expect(1, 127243, '\P{Is_Gc=-NUMBER}', "");
Expect(0, 127243, '\P{^Is_Gc=-NUMBER}', "");
Error('\p{Is_Category=:=_N}');
Error('\P{Is_Category=:=_N}');
Expect(1, 127242, '\p{Is_Category=n}', "");
Expect(0, 127242, '\p{^Is_Category=n}', "");
Expect(0, 127242, '\P{Is_Category=n}', "");
Expect(1, 127242, '\P{^Is_Category=n}', "");
Expect(0, 127243, '\p{Is_Category=n}', "");
Expect(1, 127243, '\p{^Is_Category=n}', "");
Expect(1, 127243, '\P{Is_Category=n}', "");
Expect(0, 127243, '\P{^Is_Category=n}', "");
Expect(1, 127242, '\p{Is_Category=	-n}', "");
Expect(0, 127242, '\p{^Is_Category=	-n}', "");
Expect(0, 127242, '\P{Is_Category=	-n}', "");
Expect(1, 127242, '\P{^Is_Category=	-n}', "");
Expect(0, 127243, '\p{Is_Category=	-n}', "");
Expect(1, 127243, '\p{^Is_Category=	-n}', "");
Expect(1, 127243, '\P{Is_Category=	-n}', "");
Expect(0, 127243, '\P{^Is_Category=	-n}', "");
Error('\p{General_Category=_/a/FINAL_Punctuation}');
Error('\P{General_Category=_/a/FINAL_Punctuation}');
Expect(1, 11809, '\p{General_Category=finalpunctuation}', "");
Expect(0, 11809, '\p{^General_Category=finalpunctuation}', "");
Expect(0, 11809, '\P{General_Category=finalpunctuation}', "");
Expect(1, 11809, '\P{^General_Category=finalpunctuation}', "");
Expect(0, 11810, '\p{General_Category=finalpunctuation}', "");
Expect(1, 11810, '\p{^General_Category=finalpunctuation}', "");
Expect(1, 11810, '\P{General_Category=finalpunctuation}', "");
Expect(0, 11810, '\P{^General_Category=finalpunctuation}', "");
Expect(1, 11809, '\p{General_Category=  Final_Punctuation}', "");
Expect(0, 11809, '\p{^General_Category=  Final_Punctuation}', "");
Expect(0, 11809, '\P{General_Category=  Final_Punctuation}', "");
Expect(1, 11809, '\P{^General_Category=  Final_Punctuation}', "");
Expect(0, 11810, '\p{General_Category=  Final_Punctuation}', "");
Expect(1, 11810, '\p{^General_Category=  Final_Punctuation}', "");
Expect(1, 11810, '\P{General_Category=  Final_Punctuation}', "");
Expect(0, 11810, '\P{^General_Category=  Final_Punctuation}', "");
Error('\p{Gc=_/a/Pf}');
Error('\P{Gc=_/a/Pf}');
Expect(1, 11809, '\p{Gc=pf}', "");
Expect(0, 11809, '\p{^Gc=pf}', "");
Expect(0, 11809, '\P{Gc=pf}', "");
Expect(1, 11809, '\P{^Gc=pf}', "");
Expect(0, 11810, '\p{Gc=pf}', "");
Expect(1, 11810, '\p{^Gc=pf}', "");
Expect(1, 11810, '\P{Gc=pf}', "");
Expect(0, 11810, '\P{^Gc=pf}', "");
Expect(1, 11809, '\p{Gc=		PF}', "");
Expect(0, 11809, '\p{^Gc=		PF}', "");
Expect(0, 11809, '\P{Gc=		PF}', "");
Expect(1, 11809, '\P{^Gc=		PF}', "");
Expect(0, 11810, '\p{Gc=		PF}', "");
Expect(1, 11810, '\p{^Gc=		PF}', "");
Expect(1, 11810, '\P{Gc=		PF}', "");
Expect(0, 11810, '\P{^Gc=		PF}', "");
Error('\p{Category=	:=FINAL_PUNCTUATION}');
Error('\P{Category=	:=FINAL_PUNCTUATION}');
Expect(1, 11809, '\p{Category:	finalpunctuation}', "");
Expect(0, 11809, '\p{^Category:	finalpunctuation}', "");
Expect(0, 11809, '\P{Category:	finalpunctuation}', "");
Expect(1, 11809, '\P{^Category:	finalpunctuation}', "");
Expect(0, 11810, '\p{Category:	finalpunctuation}', "");
Expect(1, 11810, '\p{^Category:	finalpunctuation}', "");
Expect(1, 11810, '\P{Category:	finalpunctuation}', "");
Expect(0, 11810, '\P{^Category:	finalpunctuation}', "");
Expect(1, 11809, '\p{Category=_-FINAL_Punctuation}', "");
Expect(0, 11809, '\p{^Category=_-FINAL_Punctuation}', "");
Expect(0, 11809, '\P{Category=_-FINAL_Punctuation}', "");
Expect(1, 11809, '\P{^Category=_-FINAL_Punctuation}', "");
Expect(0, 11810, '\p{Category=_-FINAL_Punctuation}', "");
Expect(1, 11810, '\p{^Category=_-FINAL_Punctuation}', "");
Expect(1, 11810, '\P{Category=_-FINAL_Punctuation}', "");
Expect(0, 11810, '\P{^Category=_-FINAL_Punctuation}', "");
Error('\p{Is_General_Category=pf/a/}');
Error('\P{Is_General_Category=pf/a/}');
Expect(1, 11809, '\p{Is_General_Category=pf}', "");
Expect(0, 11809, '\p{^Is_General_Category=pf}', "");
Expect(0, 11809, '\P{Is_General_Category=pf}', "");
Expect(1, 11809, '\P{^Is_General_Category=pf}', "");
Expect(0, 11810, '\p{Is_General_Category=pf}', "");
Expect(1, 11810, '\p{^Is_General_Category=pf}', "");
Expect(1, 11810, '\P{Is_General_Category=pf}', "");
Expect(0, 11810, '\P{^Is_General_Category=pf}', "");
Expect(1, 11809, '\p{Is_General_Category: PF}', "");
Expect(0, 11809, '\p{^Is_General_Category: PF}', "");
Expect(0, 11809, '\P{Is_General_Category: PF}', "");
Expect(1, 11809, '\P{^Is_General_Category: PF}', "");
Expect(0, 11810, '\p{Is_General_Category: PF}', "");
Expect(1, 11810, '\p{^Is_General_Category: PF}', "");
Expect(1, 11810, '\P{Is_General_Category: PF}', "");
Expect(0, 11810, '\P{^Is_General_Category: PF}', "");
Error('\p{Is_Gc=	 Final_Punctuation/a/}');
Error('\P{Is_Gc=	 Final_Punctuation/a/}');
Expect(1, 11809, '\p{Is_Gc=finalpunctuation}', "");
Expect(0, 11809, '\p{^Is_Gc=finalpunctuation}', "");
Expect(0, 11809, '\P{Is_Gc=finalpunctuation}', "");
Expect(1, 11809, '\P{^Is_Gc=finalpunctuation}', "");
Expect(0, 11810, '\p{Is_Gc=finalpunctuation}', "");
Expect(1, 11810, '\p{^Is_Gc=finalpunctuation}', "");
Expect(1, 11810, '\P{Is_Gc=finalpunctuation}', "");
Expect(0, 11810, '\P{^Is_Gc=finalpunctuation}', "");
Expect(1, 11809, '\p{Is_Gc=_Final_Punctuation}', "");
Expect(0, 11809, '\p{^Is_Gc=_Final_Punctuation}', "");
Expect(0, 11809, '\P{Is_Gc=_Final_Punctuation}', "");
Expect(1, 11809, '\P{^Is_Gc=_Final_Punctuation}', "");
Expect(0, 11810, '\p{Is_Gc=_Final_Punctuation}', "");
Expect(1, 11810, '\p{^Is_Gc=_Final_Punctuation}', "");
Expect(1, 11810, '\P{Is_Gc=_Final_Punctuation}', "");
Expect(0, 11810, '\P{^Is_Gc=_Final_Punctuation}', "");
Error('\p{Is_Category: /a/__PF}');
Error('\P{Is_Category: /a/__PF}');
Expect(1, 11809, '\p{Is_Category=pf}', "");
Expect(0, 11809, '\p{^Is_Category=pf}', "");
Expect(0, 11809, '\P{Is_Category=pf}', "");
Expect(1, 11809, '\P{^Is_Category=pf}', "");
Expect(0, 11810, '\p{Is_Category=pf}', "");
Expect(1, 11810, '\p{^Is_Category=pf}', "");
Expect(1, 11810, '\P{Is_Category=pf}', "");
Expect(0, 11810, '\P{^Is_Category=pf}', "");
Expect(1, 11809, '\p{Is_Category=-	Pf}', "");
Expect(0, 11809, '\p{^Is_Category=-	Pf}', "");
Expect(0, 11809, '\P{Is_Category=-	Pf}', "");
Expect(1, 11809, '\P{^Is_Category=-	Pf}', "");
Expect(0, 11810, '\p{Is_Category=-	Pf}', "");
Expect(1, 11810, '\p{^Is_Category=-	Pf}', "");
Expect(1, 11810, '\P{Is_Category=-	Pf}', "");
Expect(0, 11810, '\P{^Is_Category=-	Pf}', "");
Error('\p{General_Category=_/a/Unassigned}');
Error('\P{General_Category=_/a/Unassigned}');
Expect(1, 918000, '\p{General_Category=unassigned}', "");
Expect(0, 918000, '\p{^General_Category=unassigned}', "");
Expect(0, 918000, '\P{General_Category=unassigned}', "");
Expect(1, 918000, '\P{^General_Category=unassigned}', "");
Expect(0, 1114109, '\p{General_Category=unassigned}', "");
Expect(1, 1114109, '\p{^General_Category=unassigned}', "");
Expect(1, 1114109, '\P{General_Category=unassigned}', "");
Expect(0, 1114109, '\P{^General_Category=unassigned}', "");
Expect(1, 918000, '\p{General_Category:   Unassigned}', "");
Expect(0, 918000, '\p{^General_Category:   Unassigned}', "");
Expect(0, 918000, '\P{General_Category:   Unassigned}', "");
Expect(1, 918000, '\P{^General_Category:   Unassigned}', "");
Expect(0, 1114109, '\p{General_Category:   Unassigned}', "");
Expect(1, 1114109, '\p{^General_Category:   Unassigned}', "");
Expect(1, 1114109, '\P{General_Category:   Unassigned}', "");
Expect(0, 1114109, '\P{^General_Category:   Unassigned}', "");
Error('\p{Gc:    	cn/a/}');
Error('\P{Gc:    	cn/a/}');
Expect(1, 918000, '\p{Gc=cn}', "");
Expect(0, 918000, '\p{^Gc=cn}', "");
Expect(0, 918000, '\P{Gc=cn}', "");
Expect(1, 918000, '\P{^Gc=cn}', "");
Expect(0, 1114109, '\p{Gc=cn}', "");
Expect(1, 1114109, '\p{^Gc=cn}', "");
Expect(1, 1114109, '\P{Gc=cn}', "");
Expect(0, 1114109, '\P{^Gc=cn}', "");
Expect(1, 918000, '\p{Gc= Cn}', "");
Expect(0, 918000, '\p{^Gc= Cn}', "");
Expect(0, 918000, '\P{Gc= Cn}', "");
Expect(1, 918000, '\P{^Gc= Cn}', "");
Expect(0, 1114109, '\p{Gc= Cn}', "");
Expect(1, 1114109, '\p{^Gc= Cn}', "");
Expect(1, 1114109, '\P{Gc= Cn}', "");
Expect(0, 1114109, '\P{^Gc= Cn}', "");
Error('\p{Category=-:=Unassigned}');
Error('\P{Category=-:=Unassigned}');
Expect(1, 918000, '\p{Category=unassigned}', "");
Expect(0, 918000, '\p{^Category=unassigned}', "");
Expect(0, 918000, '\P{Category=unassigned}', "");
Expect(1, 918000, '\P{^Category=unassigned}', "");
Expect(0, 1114109, '\p{Category=unassigned}', "");
Expect(1, 1114109, '\p{^Category=unassigned}', "");
Expect(1, 1114109, '\P{Category=unassigned}', "");
Expect(0, 1114109, '\P{^Category=unassigned}', "");
Expect(1, 918000, '\p{Category= -UNASSIGNED}', "");
Expect(0, 918000, '\p{^Category= -UNASSIGNED}', "");
Expect(0, 918000, '\P{Category= -UNASSIGNED}', "");
Expect(1, 918000, '\P{^Category= -UNASSIGNED}', "");
Expect(0, 1114109, '\p{Category= -UNASSIGNED}', "");
Expect(1, 1114109, '\p{^Category= -UNASSIGNED}', "");
Expect(1, 1114109, '\P{Category= -UNASSIGNED}', "");
Expect(0, 1114109, '\P{^Category= -UNASSIGNED}', "");
Error('\p{Is_General_Category= Cn:=}');
Error('\P{Is_General_Category= Cn:=}');
Expect(1, 918000, '\p{Is_General_Category=cn}', "");
Expect(0, 918000, '\p{^Is_General_Category=cn}', "");
Expect(0, 918000, '\P{Is_General_Category=cn}', "");
Expect(1, 918000, '\P{^Is_General_Category=cn}', "");
Expect(0, 1114109, '\p{Is_General_Category=cn}', "");
Expect(1, 1114109, '\p{^Is_General_Category=cn}', "");
Expect(1, 1114109, '\P{Is_General_Category=cn}', "");
Expect(0, 1114109, '\P{^Is_General_Category=cn}', "");
Expect(1, 918000, '\p{Is_General_Category=-	CN}', "");
Expect(0, 918000, '\p{^Is_General_Category=-	CN}', "");
Expect(0, 918000, '\P{Is_General_Category=-	CN}', "");
Expect(1, 918000, '\P{^Is_General_Category=-	CN}', "");
Expect(0, 1114109, '\p{Is_General_Category=-	CN}', "");
Expect(1, 1114109, '\p{^Is_General_Category=-	CN}', "");
Expect(1, 1114109, '\P{Is_General_Category=-	CN}', "");
Expect(0, 1114109, '\P{^Is_General_Category=-	CN}', "");
Error('\p{Is_Gc=_:=unassigned}');
Error('\P{Is_Gc=_:=unassigned}');
Expect(1, 918000, '\p{Is_Gc: unassigned}', "");
Expect(0, 918000, '\p{^Is_Gc: unassigned}', "");
Expect(0, 918000, '\P{Is_Gc: unassigned}', "");
Expect(1, 918000, '\P{^Is_Gc: unassigned}', "");
Expect(0, 1114109, '\p{Is_Gc: unassigned}', "");
Expect(1, 1114109, '\p{^Is_Gc: unassigned}', "");
Expect(1, 1114109, '\P{Is_Gc: unassigned}', "");
Expect(0, 1114109, '\P{^Is_Gc: unassigned}', "");
Expect(1, 918000, '\p{Is_Gc:		Unassigned}', "");
Expect(0, 918000, '\p{^Is_Gc:		Unassigned}', "");
Expect(0, 918000, '\P{Is_Gc:		Unassigned}', "");
Expect(1, 918000, '\P{^Is_Gc:		Unassigned}', "");
Expect(0, 1114109, '\p{Is_Gc:		Unassigned}', "");
Expect(1, 1114109, '\p{^Is_Gc:		Unassigned}', "");
Expect(1, 1114109, '\P{Is_Gc:		Unassigned}', "");
Expect(0, 1114109, '\P{^Is_Gc:		Unassigned}', "");
Error('\p{Is_Category=-/a/Cn}');
Error('\P{Is_Category=-/a/Cn}');
Expect(1, 918000, '\p{Is_Category=cn}', "");
Expect(0, 918000, '\p{^Is_Category=cn}', "");
Expect(0, 918000, '\P{Is_Category=cn}', "");
Expect(1, 918000, '\P{^Is_Category=cn}', "");
Expect(0, 1114109, '\p{Is_Category=cn}', "");
Expect(1, 1114109, '\p{^Is_Category=cn}', "");
Expect(1, 1114109, '\P{Is_Category=cn}', "");
Expect(0, 1114109, '\P{^Is_Category=cn}', "");
Expect(1, 918000, '\p{Is_Category=_	Cn}', "");
Expect(0, 918000, '\p{^Is_Category=_	Cn}', "");
Expect(0, 918000, '\P{Is_Category=_	Cn}', "");
Expect(1, 918000, '\P{^Is_Category=_	Cn}', "");
Expect(0, 1114109, '\p{Is_Category=_	Cn}', "");
Expect(1, 1114109, '\p{^Is_Category=_	Cn}', "");
Expect(1, 1114109, '\P{Is_Category=_	Cn}', "");
Expect(0, 1114109, '\P{^Is_Category=_	Cn}', "");
Error('\p{General_Category=_:=Decimal_Number}');
Error('\P{General_Category=_:=Decimal_Number}');
Expect(1, 120831, '\p{General_Category=decimalnumber}', "");
Expect(0, 120831, '\p{^General_Category=decimalnumber}', "");
Expect(0, 120831, '\P{General_Category=decimalnumber}', "");
Expect(1, 120831, '\P{^General_Category=decimalnumber}', "");
Expect(0, 120832, '\p{General_Category=decimalnumber}', "");
Expect(1, 120832, '\p{^General_Category=decimalnumber}', "");
Expect(1, 120832, '\P{General_Category=decimalnumber}', "");
Expect(0, 120832, '\P{^General_Category=decimalnumber}', "");
Expect(1, 120831, '\p{General_Category= DECIMAL_number}', "");
Expect(0, 120831, '\p{^General_Category= DECIMAL_number}', "");
Expect(0, 120831, '\P{General_Category= DECIMAL_number}', "");
Expect(1, 120831, '\P{^General_Category= DECIMAL_number}', "");
Expect(0, 120832, '\p{General_Category= DECIMAL_number}', "");
Expect(1, 120832, '\p{^General_Category= DECIMAL_number}', "");
Expect(1, 120832, '\P{General_Category= DECIMAL_number}', "");
Expect(0, 120832, '\P{^General_Category= DECIMAL_number}', "");
Error('\p{Gc=:=ND}');
Error('\P{Gc=:=ND}');
Expect(1, 120831, '\p{Gc=nd}', "");
Expect(0, 120831, '\p{^Gc=nd}', "");
Expect(0, 120831, '\P{Gc=nd}', "");
Expect(1, 120831, '\P{^Gc=nd}', "");
Expect(0, 120832, '\p{Gc=nd}', "");
Expect(1, 120832, '\p{^Gc=nd}', "");
Expect(1, 120832, '\P{Gc=nd}', "");
Expect(0, 120832, '\P{^Gc=nd}', "");
Error('\p{Category:	 _Digit:=}');
Error('\P{Category:	 _Digit:=}');
Expect(1, 120831, '\p{Category=digit}', "");
Expect(0, 120831, '\p{^Category=digit}', "");
Expect(0, 120831, '\P{Category=digit}', "");
Expect(1, 120831, '\P{^Category=digit}', "");
Expect(0, 120832, '\p{Category=digit}', "");
Expect(1, 120832, '\p{^Category=digit}', "");
Expect(1, 120832, '\P{Category=digit}', "");
Expect(0, 120832, '\P{^Category=digit}', "");
Expect(1, 120831, '\p{Category=  Digit}', "");
Expect(0, 120831, '\p{^Category=  Digit}', "");
Expect(0, 120831, '\P{Category=  Digit}', "");
Expect(1, 120831, '\P{^Category=  Digit}', "");
Expect(0, 120832, '\p{Category=  Digit}', "");
Expect(1, 120832, '\p{^Category=  Digit}', "");
Expect(1, 120832, '\P{Category=  Digit}', "");
Expect(0, 120832, '\P{^Category=  Digit}', "");
Error('\p{Is_General_Category=/a/- Decimal_NUMBER}');
Error('\P{Is_General_Category=/a/- Decimal_NUMBER}');
Expect(1, 120831, '\p{Is_General_Category=decimalnumber}', "");
Expect(0, 120831, '\p{^Is_General_Category=decimalnumber}', "");
Expect(0, 120831, '\P{Is_General_Category=decimalnumber}', "");
Expect(1, 120831, '\P{^Is_General_Category=decimalnumber}', "");
Expect(0, 120832, '\p{Is_General_Category=decimalnumber}', "");
Expect(1, 120832, '\p{^Is_General_Category=decimalnumber}', "");
Expect(1, 120832, '\P{Is_General_Category=decimalnumber}', "");
Expect(0, 120832, '\P{^Is_General_Category=decimalnumber}', "");
Expect(1, 120831, '\p{Is_General_Category:   __Decimal_NUMBER}', "");
Expect(0, 120831, '\p{^Is_General_Category:   __Decimal_NUMBER}', "");
Expect(0, 120831, '\P{Is_General_Category:   __Decimal_NUMBER}', "");
Expect(1, 120831, '\P{^Is_General_Category:   __Decimal_NUMBER}', "");
Expect(0, 120832, '\p{Is_General_Category:   __Decimal_NUMBER}', "");
Expect(1, 120832, '\p{^Is_General_Category:   __Decimal_NUMBER}', "");
Expect(1, 120832, '\P{Is_General_Category:   __Decimal_NUMBER}', "");
Expect(0, 120832, '\P{^Is_General_Category:   __Decimal_NUMBER}', "");
Error('\p{Is_Gc= ND/a/}');
Error('\P{Is_Gc= ND/a/}');
Expect(1, 120831, '\p{Is_Gc=nd}', "");
Expect(0, 120831, '\p{^Is_Gc=nd}', "");
Expect(0, 120831, '\P{Is_Gc=nd}', "");
Expect(1, 120831, '\P{^Is_Gc=nd}', "");
Expect(0, 120832, '\p{Is_Gc=nd}', "");
Expect(1, 120832, '\p{^Is_Gc=nd}', "");
Expect(1, 120832, '\P{Is_Gc=nd}', "");
Expect(0, 120832, '\P{^Is_Gc=nd}', "");
Expect(1, 120831, '\p{Is_Gc=ND}', "");
Expect(0, 120831, '\p{^Is_Gc=ND}', "");
Expect(0, 120831, '\P{Is_Gc=ND}', "");
Expect(1, 120831, '\P{^Is_Gc=ND}', "");
Expect(0, 120832, '\p{Is_Gc=ND}', "");
Expect(1, 120832, '\p{^Is_Gc=ND}', "");
Expect(1, 120832, '\P{Is_Gc=ND}', "");
Expect(0, 120832, '\P{^Is_Gc=ND}', "");
Error('\p{Is_Category=:=	_Digit}');
Error('\P{Is_Category=:=	_Digit}');
Expect(1, 120831, '\p{Is_Category=digit}', "");
Expect(0, 120831, '\p{^Is_Category=digit}', "");
Expect(0, 120831, '\P{Is_Category=digit}', "");
Expect(1, 120831, '\P{^Is_Category=digit}', "");
Expect(0, 120832, '\p{Is_Category=digit}', "");
Expect(1, 120832, '\p{^Is_Category=digit}', "");
Expect(1, 120832, '\P{Is_Category=digit}', "");
Expect(0, 120832, '\P{^Is_Category=digit}', "");
Expect(1, 120831, '\p{Is_Category=-_DIGIT}', "");
Expect(0, 120831, '\p{^Is_Category=-_DIGIT}', "");
Expect(0, 120831, '\P{Is_Category=-_DIGIT}', "");
Expect(1, 120831, '\P{^Is_Category=-_DIGIT}', "");
Expect(0, 120832, '\p{Is_Category=-_DIGIT}', "");
Expect(1, 120832, '\p{^Is_Category=-_DIGIT}', "");
Expect(1, 120832, '\P{Is_Category=-_DIGIT}', "");
Expect(0, 120832, '\P{^Is_Category=-_DIGIT}', "");
Error('\p{General_Category=:=_paragraph_separator}');
Error('\P{General_Category=:=_paragraph_separator}');
Expect(1, 8233, '\p{General_Category=paragraphseparator}', "");
Expect(0, 8233, '\p{^General_Category=paragraphseparator}', "");
Expect(0, 8233, '\P{General_Category=paragraphseparator}', "");
Expect(1, 8233, '\P{^General_Category=paragraphseparator}', "");
Expect(0, 8234, '\p{General_Category=paragraphseparator}', "");
Expect(1, 8234, '\p{^General_Category=paragraphseparator}', "");
Expect(1, 8234, '\P{General_Category=paragraphseparator}', "");
Expect(0, 8234, '\P{^General_Category=paragraphseparator}', "");
Expect(1, 8233, '\p{General_Category= 	Paragraph_Separator}', "");
Expect(0, 8233, '\p{^General_Category= 	Paragraph_Separator}', "");
Expect(0, 8233, '\P{General_Category= 	Paragraph_Separator}', "");
Expect(1, 8233, '\P{^General_Category= 	Paragraph_Separator}', "");
Expect(0, 8234, '\p{General_Category= 	Paragraph_Separator}', "");
Expect(1, 8234, '\p{^General_Category= 	Paragraph_Separator}', "");
Expect(1, 8234, '\P{General_Category= 	Paragraph_Separator}', "");
Expect(0, 8234, '\P{^General_Category= 	Paragraph_Separator}', "");
Error('\p{Gc=	:=Zp}');
Error('\P{Gc=	:=Zp}');
Expect(1, 8233, '\p{Gc=zp}', "");
Expect(0, 8233, '\p{^Gc=zp}', "");
Expect(0, 8233, '\P{Gc=zp}', "");
Expect(1, 8233, '\P{^Gc=zp}', "");
Expect(0, 8234, '\p{Gc=zp}', "");
Expect(1, 8234, '\p{^Gc=zp}', "");
Expect(1, 8234, '\P{Gc=zp}', "");
Expect(0, 8234, '\P{^Gc=zp}', "");
Expect(1, 8233, '\p{Gc=--Zp}', "");
Expect(0, 8233, '\p{^Gc=--Zp}', "");
Expect(0, 8233, '\P{Gc=--Zp}', "");
Expect(1, 8233, '\P{^Gc=--Zp}', "");
Expect(0, 8234, '\p{Gc=--Zp}', "");
Expect(1, 8234, '\p{^Gc=--Zp}', "");
Expect(1, 8234, '\P{Gc=--Zp}', "");
Expect(0, 8234, '\P{^Gc=--Zp}', "");
Error('\p{Category=:=		Paragraph_Separator}');
Error('\P{Category=:=		Paragraph_Separator}');
Expect(1, 8233, '\p{Category=paragraphseparator}', "");
Expect(0, 8233, '\p{^Category=paragraphseparator}', "");
Expect(0, 8233, '\P{Category=paragraphseparator}', "");
Expect(1, 8233, '\P{^Category=paragraphseparator}', "");
Expect(0, 8234, '\p{Category=paragraphseparator}', "");
Expect(1, 8234, '\p{^Category=paragraphseparator}', "");
Expect(1, 8234, '\P{Category=paragraphseparator}', "");
Expect(0, 8234, '\P{^Category=paragraphseparator}', "");
Expect(1, 8233, '\p{Category=_	PARAGRAPH_Separator}', "");
Expect(0, 8233, '\p{^Category=_	PARAGRAPH_Separator}', "");
Expect(0, 8233, '\P{Category=_	PARAGRAPH_Separator}', "");
Expect(1, 8233, '\P{^Category=_	PARAGRAPH_Separator}', "");
Expect(0, 8234, '\p{Category=_	PARAGRAPH_Separator}', "");
Expect(1, 8234, '\p{^Category=_	PARAGRAPH_Separator}', "");
Expect(1, 8234, '\P{Category=_	PARAGRAPH_Separator}', "");
Expect(0, 8234, '\P{^Category=_	PARAGRAPH_Separator}', "");
Error('\p{Is_General_Category:-_zp/a/}');
Error('\P{Is_General_Category:-_zp/a/}');
Expect(1, 8233, '\p{Is_General_Category:   zp}', "");
Expect(0, 8233, '\p{^Is_General_Category:   zp}', "");
Expect(0, 8233, '\P{Is_General_Category:   zp}', "");
Expect(1, 8233, '\P{^Is_General_Category:   zp}', "");
Expect(0, 8234, '\p{Is_General_Category:   zp}', "");
Expect(1, 8234, '\p{^Is_General_Category:   zp}', "");
Expect(1, 8234, '\P{Is_General_Category:   zp}', "");
Expect(0, 8234, '\P{^Is_General_Category:   zp}', "");
Expect(1, 8233, '\p{Is_General_Category= Zp}', "");
Expect(0, 8233, '\p{^Is_General_Category= Zp}', "");
Expect(0, 8233, '\P{Is_General_Category= Zp}', "");
Expect(1, 8233, '\P{^Is_General_Category= Zp}', "");
Expect(0, 8234, '\p{Is_General_Category= Zp}', "");
Expect(1, 8234, '\p{^Is_General_Category= Zp}', "");
Expect(1, 8234, '\P{Is_General_Category= Zp}', "");
Expect(0, 8234, '\P{^Is_General_Category= Zp}', "");
Error('\p{Is_Gc=/a/PARAGRAPH_separator}');
Error('\P{Is_Gc=/a/PARAGRAPH_separator}');
Expect(1, 8233, '\p{Is_Gc=paragraphseparator}', "");
Expect(0, 8233, '\p{^Is_Gc=paragraphseparator}', "");
Expect(0, 8233, '\P{Is_Gc=paragraphseparator}', "");
Expect(1, 8233, '\P{^Is_Gc=paragraphseparator}', "");
Expect(0, 8234, '\p{Is_Gc=paragraphseparator}', "");
Expect(1, 8234, '\p{^Is_Gc=paragraphseparator}', "");
Expect(1, 8234, '\P{Is_Gc=paragraphseparator}', "");
Expect(0, 8234, '\P{^Is_Gc=paragraphseparator}', "");
Expect(1, 8233, '\p{Is_Gc=-	Paragraph_separator}', "");
Expect(0, 8233, '\p{^Is_Gc=-	Paragraph_separator}', "");
Expect(0, 8233, '\P{Is_Gc=-	Paragraph_separator}', "");
Expect(1, 8233, '\P{^Is_Gc=-	Paragraph_separator}', "");
Expect(0, 8234, '\p{Is_Gc=-	Paragraph_separator}', "");
Expect(1, 8234, '\p{^Is_Gc=-	Paragraph_separator}', "");
Expect(1, 8234, '\P{Is_Gc=-	Paragraph_separator}', "");
Expect(0, 8234, '\P{^Is_Gc=-	Paragraph_separator}', "");
Error('\p{Is_Category=_Zp:=}');
Error('\P{Is_Category=_Zp:=}');
Expect(1, 8233, '\p{Is_Category: zp}', "");
Expect(0, 8233, '\p{^Is_Category: zp}', "");
Expect(0, 8233, '\P{Is_Category: zp}', "");
Expect(1, 8233, '\P{^Is_Category: zp}', "");
Expect(0, 8234, '\p{Is_Category: zp}', "");
Expect(1, 8234, '\p{^Is_Category: zp}', "");
Expect(1, 8234, '\P{Is_Category: zp}', "");
Expect(0, 8234, '\P{^Is_Category: zp}', "");
Expect(1, 8233, '\p{Is_Category= Zp}', "");
Expect(0, 8233, '\p{^Is_Category= Zp}', "");
Expect(0, 8233, '\P{Is_Category= Zp}', "");
Expect(1, 8233, '\P{^Is_Category= Zp}', "");
Expect(0, 8234, '\p{Is_Category= Zp}', "");
Expect(1, 8234, '\p{^Is_Category= Zp}', "");
Expect(1, 8234, '\P{Is_Category= Zp}', "");
Expect(0, 8234, '\P{^Is_Category= Zp}', "");
Error('\p{General_Category=	 MODIFIER_Letter:=}');
Error('\P{General_Category=	 MODIFIER_Letter:=}');
Expect(1, 94111, '\p{General_Category=modifierletter}', "");
Expect(0, 94111, '\p{^General_Category=modifierletter}', "");
Expect(0, 94111, '\P{General_Category=modifierletter}', "");
Expect(1, 94111, '\P{^General_Category=modifierletter}', "");
Expect(0, 94112, '\p{General_Category=modifierletter}', "");
Expect(1, 94112, '\p{^General_Category=modifierletter}', "");
Expect(1, 94112, '\P{General_Category=modifierletter}', "");
Expect(0, 94112, '\P{^General_Category=modifierletter}', "");
Expect(1, 94111, '\p{General_Category=- modifier_Letter}', "");
Expect(0, 94111, '\p{^General_Category=- modifier_Letter}', "");
Expect(0, 94111, '\P{General_Category=- modifier_Letter}', "");
Expect(1, 94111, '\P{^General_Category=- modifier_Letter}', "");
Expect(0, 94112, '\p{General_Category=- modifier_Letter}', "");
Expect(1, 94112, '\p{^General_Category=- modifier_Letter}', "");
Expect(1, 94112, '\P{General_Category=- modifier_Letter}', "");
Expect(0, 94112, '\P{^General_Category=- modifier_Letter}', "");
Error('\p{Gc=__Lm:=}');
Error('\P{Gc=__Lm:=}');
Expect(1, 94111, '\p{Gc=lm}', "");
Expect(0, 94111, '\p{^Gc=lm}', "");
Expect(0, 94111, '\P{Gc=lm}', "");
Expect(1, 94111, '\P{^Gc=lm}', "");
Expect(0, 94112, '\p{Gc=lm}', "");
Expect(1, 94112, '\p{^Gc=lm}', "");
Expect(1, 94112, '\P{Gc=lm}', "");
Expect(0, 94112, '\P{^Gc=lm}', "");
Expect(1, 94111, '\p{Gc:   _	lm}', "");
Expect(0, 94111, '\p{^Gc:   _	lm}', "");
Expect(0, 94111, '\P{Gc:   _	lm}', "");
Expect(1, 94111, '\P{^Gc:   _	lm}', "");
Expect(0, 94112, '\p{Gc:   _	lm}', "");
Expect(1, 94112, '\p{^Gc:   _	lm}', "");
Expect(1, 94112, '\P{Gc:   _	lm}', "");
Expect(0, 94112, '\P{^Gc:   _	lm}', "");
Error('\p{Category= modifier_Letter:=}');
Error('\P{Category= modifier_Letter:=}');
Expect(1, 94111, '\p{Category=modifierletter}', "");
Expect(0, 94111, '\p{^Category=modifierletter}', "");
Expect(0, 94111, '\P{Category=modifierletter}', "");
Expect(1, 94111, '\P{^Category=modifierletter}', "");
Expect(0, 94112, '\p{Category=modifierletter}', "");
Expect(1, 94112, '\p{^Category=modifierletter}', "");
Expect(1, 94112, '\P{Category=modifierletter}', "");
Expect(0, 94112, '\P{^Category=modifierletter}', "");
Expect(1, 94111, '\p{Category= Modifier_LETTER}', "");
Expect(0, 94111, '\p{^Category= Modifier_LETTER}', "");
Expect(0, 94111, '\P{Category= Modifier_LETTER}', "");
Expect(1, 94111, '\P{^Category= Modifier_LETTER}', "");
Expect(0, 94112, '\p{Category= Modifier_LETTER}', "");
Expect(1, 94112, '\p{^Category= Modifier_LETTER}', "");
Expect(1, 94112, '\P{Category= Modifier_LETTER}', "");
Expect(0, 94112, '\P{^Category= Modifier_LETTER}', "");
Error('\p{Is_General_Category=/a/- LM}');
Error('\P{Is_General_Category=/a/- LM}');
Expect(1, 94111, '\p{Is_General_Category=lm}', "");
Expect(0, 94111, '\p{^Is_General_Category=lm}', "");
Expect(0, 94111, '\P{Is_General_Category=lm}', "");
Expect(1, 94111, '\P{^Is_General_Category=lm}', "");
Expect(0, 94112, '\p{Is_General_Category=lm}', "");
Expect(1, 94112, '\p{^Is_General_Category=lm}', "");
Expect(1, 94112, '\P{Is_General_Category=lm}', "");
Expect(0, 94112, '\P{^Is_General_Category=lm}', "");
Expect(1, 94111, '\p{Is_General_Category=_ LM}', "");
Expect(0, 94111, '\p{^Is_General_Category=_ LM}', "");
Expect(0, 94111, '\P{Is_General_Category=_ LM}', "");
Expect(1, 94111, '\P{^Is_General_Category=_ LM}', "");
Expect(0, 94112, '\p{Is_General_Category=_ LM}', "");
Expect(1, 94112, '\p{^Is_General_Category=_ LM}', "");
Expect(1, 94112, '\P{Is_General_Category=_ LM}', "");
Expect(0, 94112, '\P{^Is_General_Category=_ LM}', "");
Error('\p{Is_Gc= :=Modifier_letter}');
Error('\P{Is_Gc= :=Modifier_letter}');
Expect(1, 94111, '\p{Is_Gc=modifierletter}', "");
Expect(0, 94111, '\p{^Is_Gc=modifierletter}', "");
Expect(0, 94111, '\P{Is_Gc=modifierletter}', "");
Expect(1, 94111, '\P{^Is_Gc=modifierletter}', "");
Expect(0, 94112, '\p{Is_Gc=modifierletter}', "");
Expect(1, 94112, '\p{^Is_Gc=modifierletter}', "");
Expect(1, 94112, '\P{Is_Gc=modifierletter}', "");
Expect(0, 94112, '\P{^Is_Gc=modifierletter}', "");
Expect(1, 94111, '\p{Is_Gc=-_modifier_Letter}', "");
Expect(0, 94111, '\p{^Is_Gc=-_modifier_Letter}', "");
Expect(0, 94111, '\P{Is_Gc=-_modifier_Letter}', "");
Expect(1, 94111, '\P{^Is_Gc=-_modifier_Letter}', "");
Expect(0, 94112, '\p{Is_Gc=-_modifier_Letter}', "");
Expect(1, 94112, '\p{^Is_Gc=-_modifier_Letter}', "");
Expect(1, 94112, '\P{Is_Gc=-_modifier_Letter}', "");
Expect(0, 94112, '\P{^Is_Gc=-_modifier_Letter}', "");
Error('\p{Is_Category=-/a/Lm}');
Error('\P{Is_Category=-/a/Lm}');
Expect(1, 94111, '\p{Is_Category=lm}', "");
Expect(0, 94111, '\p{^Is_Category=lm}', "");
Expect(0, 94111, '\P{Is_Category=lm}', "");
Expect(1, 94111, '\P{^Is_Category=lm}', "");
Expect(0, 94112, '\p{Is_Category=lm}', "");
Expect(1, 94112, '\p{^Is_Category=lm}', "");
Expect(1, 94112, '\P{Is_Category=lm}', "");
Expect(0, 94112, '\P{^Is_Category=lm}', "");
Expect(1, 94111, '\p{Is_Category=__Lm}', "");
Expect(0, 94111, '\p{^Is_Category=__Lm}', "");
Expect(0, 94111, '\P{Is_Category=__Lm}', "");
Expect(1, 94111, '\P{^Is_Category=__Lm}', "");
Expect(0, 94112, '\p{Is_Category=__Lm}', "");
Expect(1, 94112, '\p{^Is_Category=__Lm}', "");
Expect(1, 94112, '\P{Is_Category=__Lm}', "");
Expect(0, 94112, '\P{^Is_Category=__Lm}', "");
Error('\p{General_Category=/a/	Other_SYMBOL}');
Error('\P{General_Category=/a/	Other_SYMBOL}');
Expect(1, 128883, '\p{General_Category=othersymbol}', "");
Expect(0, 128883, '\p{^General_Category=othersymbol}', "");
Expect(0, 128883, '\P{General_Category=othersymbol}', "");
Expect(1, 128883, '\P{^General_Category=othersymbol}', "");
Expect(0, 128884, '\p{General_Category=othersymbol}', "");
Expect(1, 128884, '\p{^General_Category=othersymbol}', "");
Expect(1, 128884, '\P{General_Category=othersymbol}', "");
Expect(0, 128884, '\P{^General_Category=othersymbol}', "");
Expect(1, 128883, '\p{General_Category=	Other_SYMBOL}', "");
Expect(0, 128883, '\p{^General_Category=	Other_SYMBOL}', "");
Expect(0, 128883, '\P{General_Category=	Other_SYMBOL}', "");
Expect(1, 128883, '\P{^General_Category=	Other_SYMBOL}', "");
Expect(0, 128884, '\p{General_Category=	Other_SYMBOL}', "");
Expect(1, 128884, '\p{^General_Category=	Other_SYMBOL}', "");
Expect(1, 128884, '\P{General_Category=	Other_SYMBOL}', "");
Expect(0, 128884, '\P{^General_Category=	Other_SYMBOL}', "");
Error('\p{Gc=--So:=}');
Error('\P{Gc=--So:=}');
Expect(1, 128883, '\p{Gc=so}', "");
Expect(0, 128883, '\p{^Gc=so}', "");
Expect(0, 128883, '\P{Gc=so}', "");
Expect(1, 128883, '\P{^Gc=so}', "");
Expect(0, 128884, '\p{Gc=so}', "");
Expect(1, 128884, '\p{^Gc=so}', "");
Expect(1, 128884, '\P{Gc=so}', "");
Expect(0, 128884, '\P{^Gc=so}', "");
Expect(1, 128883, '\p{Gc=_SO}', "");
Expect(0, 128883, '\p{^Gc=_SO}', "");
Expect(0, 128883, '\P{Gc=_SO}', "");
Expect(1, 128883, '\P{^Gc=_SO}', "");
Expect(0, 128884, '\p{Gc=_SO}', "");
Expect(1, 128884, '\p{^Gc=_SO}', "");
Expect(1, 128884, '\P{Gc=_SO}', "");
Expect(0, 128884, '\P{^Gc=_SO}', "");
Error('\p{Category=-	Other_Symbol:=}');
Error('\P{Category=-	Other_Symbol:=}');
Expect(1, 128883, '\p{Category=othersymbol}', "");
Expect(0, 128883, '\p{^Category=othersymbol}', "");
Expect(0, 128883, '\P{Category=othersymbol}', "");
Expect(1, 128883, '\P{^Category=othersymbol}', "");
Expect(0, 128884, '\p{Category=othersymbol}', "");
Expect(1, 128884, '\p{^Category=othersymbol}', "");
Expect(1, 128884, '\P{Category=othersymbol}', "");
Expect(0, 128884, '\P{^Category=othersymbol}', "");
Expect(1, 128883, '\p{Category=--OTHER_SYMBOL}', "");
Expect(0, 128883, '\p{^Category=--OTHER_SYMBOL}', "");
Expect(0, 128883, '\P{Category=--OTHER_SYMBOL}', "");
Expect(1, 128883, '\P{^Category=--OTHER_SYMBOL}', "");
Expect(0, 128884, '\p{Category=--OTHER_SYMBOL}', "");
Expect(1, 128884, '\p{^Category=--OTHER_SYMBOL}', "");
Expect(1, 128884, '\P{Category=--OTHER_SYMBOL}', "");
Expect(0, 128884, '\P{^Category=--OTHER_SYMBOL}', "");
Error('\p{Is_General_Category=:=-SO}');
Error('\P{Is_General_Category=:=-SO}');
Expect(1, 128883, '\p{Is_General_Category=so}', "");
Expect(0, 128883, '\p{^Is_General_Category=so}', "");
Expect(0, 128883, '\P{Is_General_Category=so}', "");
Expect(1, 128883, '\P{^Is_General_Category=so}', "");
Expect(0, 128884, '\p{Is_General_Category=so}', "");
Expect(1, 128884, '\p{^Is_General_Category=so}', "");
Expect(1, 128884, '\P{Is_General_Category=so}', "");
Expect(0, 128884, '\P{^Is_General_Category=so}', "");
Expect(1, 128883, '\p{Is_General_Category=_ So}', "");
Expect(0, 128883, '\p{^Is_General_Category=_ So}', "");
Expect(0, 128883, '\P{Is_General_Category=_ So}', "");
Expect(1, 128883, '\P{^Is_General_Category=_ So}', "");
Expect(0, 128884, '\p{Is_General_Category=_ So}', "");
Expect(1, 128884, '\p{^Is_General_Category=_ So}', "");
Expect(1, 128884, '\P{Is_General_Category=_ So}', "");
Expect(0, 128884, '\P{^Is_General_Category=_ So}', "");
Error('\p{Is_Gc=_/a/Other_symbol}');
Error('\P{Is_Gc=_/a/Other_symbol}');
Expect(1, 128883, '\p{Is_Gc:   othersymbol}', "");
Expect(0, 128883, '\p{^Is_Gc:   othersymbol}', "");
Expect(0, 128883, '\P{Is_Gc:   othersymbol}', "");
Expect(1, 128883, '\P{^Is_Gc:   othersymbol}', "");
Expect(0, 128884, '\p{Is_Gc:   othersymbol}', "");
Expect(1, 128884, '\p{^Is_Gc:   othersymbol}', "");
Expect(1, 128884, '\P{Is_Gc:   othersymbol}', "");
Expect(0, 128884, '\P{^Is_Gc:   othersymbol}', "");
Expect(1, 128883, '\p{Is_Gc:	_	OTHER_symbol}', "");
Expect(0, 128883, '\p{^Is_Gc:	_	OTHER_symbol}', "");
Expect(0, 128883, '\P{Is_Gc:	_	OTHER_symbol}', "");
Expect(1, 128883, '\P{^Is_Gc:	_	OTHER_symbol}', "");
Expect(0, 128884, '\p{Is_Gc:	_	OTHER_symbol}', "");
Expect(1, 128884, '\p{^Is_Gc:	_	OTHER_symbol}', "");
Expect(1, 128884, '\P{Is_Gc:	_	OTHER_symbol}', "");
Expect(0, 128884, '\P{^Is_Gc:	_	OTHER_symbol}', "");
Error('\p{Is_Category=/a/SO}');
Error('\P{Is_Category=/a/SO}');
Expect(1, 128883, '\p{Is_Category=so}', "");
Expect(0, 128883, '\p{^Is_Category=so}', "");
Expect(0, 128883, '\P{Is_Category=so}', "");
Expect(1, 128883, '\P{^Is_Category=so}', "");
Expect(0, 128884, '\p{Is_Category=so}', "");
Expect(1, 128884, '\p{^Is_Category=so}', "");
Expect(1, 128884, '\P{Is_Category=so}', "");
Expect(0, 128884, '\P{^Is_Category=so}', "");
Expect(1, 128883, '\p{Is_Category= So}', "");
Expect(0, 128883, '\p{^Is_Category= So}', "");
Expect(0, 128883, '\P{Is_Category= So}', "");
Expect(1, 128883, '\P{^Is_Category= So}', "");
Expect(0, 128884, '\p{Is_Category= So}', "");
Expect(1, 128884, '\p{^Is_Category= So}', "");
Expect(1, 128884, '\P{Is_Category= So}', "");
Expect(0, 128884, '\P{^Is_Category= So}', "");
Error('\p{General_Category=/a/_-nonspacing_MARK}');
Error('\P{General_Category=/a/_-nonspacing_MARK}');
Expect(1, 917999, '\p{General_Category=nonspacingmark}', "");
Expect(0, 917999, '\p{^General_Category=nonspacingmark}', "");
Expect(0, 917999, '\P{General_Category=nonspacingmark}', "");
Expect(1, 917999, '\P{^General_Category=nonspacingmark}', "");
Expect(0, 918000, '\p{General_Category=nonspacingmark}', "");
Expect(1, 918000, '\p{^General_Category=nonspacingmark}', "");
Expect(1, 918000, '\P{General_Category=nonspacingmark}', "");
Expect(0, 918000, '\P{^General_Category=nonspacingmark}', "");
Expect(1, 917999, '\p{General_Category=	 Nonspacing_MARK}', "");
Expect(0, 917999, '\p{^General_Category=	 Nonspacing_MARK}', "");
Expect(0, 917999, '\P{General_Category=	 Nonspacing_MARK}', "");
Expect(1, 917999, '\P{^General_Category=	 Nonspacing_MARK}', "");
Expect(0, 918000, '\p{General_Category=	 Nonspacing_MARK}', "");
Expect(1, 918000, '\p{^General_Category=	 Nonspacing_MARK}', "");
Expect(1, 918000, '\P{General_Category=	 Nonspacing_MARK}', "");
Expect(0, 918000, '\P{^General_Category=	 Nonspacing_MARK}', "");
Error('\p{Gc= _MN:=}');
Error('\P{Gc= _MN:=}');
Expect(1, 917999, '\p{Gc=mn}', "");
Expect(0, 917999, '\p{^Gc=mn}', "");
Expect(0, 917999, '\P{Gc=mn}', "");
Expect(1, 917999, '\P{^Gc=mn}', "");
Expect(0, 918000, '\p{Gc=mn}', "");
Expect(1, 918000, '\p{^Gc=mn}', "");
Expect(1, 918000, '\P{Gc=mn}', "");
Expect(0, 918000, '\P{^Gc=mn}', "");
Expect(1, 917999, '\p{Gc=_-Mn}', "");
Expect(0, 917999, '\p{^Gc=_-Mn}', "");
Expect(0, 917999, '\P{Gc=_-Mn}', "");
Expect(1, 917999, '\P{^Gc=_-Mn}', "");
Expect(0, 918000, '\p{Gc=_-Mn}', "");
Expect(1, 918000, '\p{^Gc=_-Mn}', "");
Expect(1, 918000, '\P{Gc=_-Mn}', "");
Expect(0, 918000, '\P{^Gc=_-Mn}', "");
Error('\p{Category=:= 	Nonspacing_mark}');
Error('\P{Category=:= 	Nonspacing_mark}');
Expect(1, 917999, '\p{Category=nonspacingmark}', "");
Expect(0, 917999, '\p{^Category=nonspacingmark}', "");
Expect(0, 917999, '\P{Category=nonspacingmark}', "");
Expect(1, 917999, '\P{^Category=nonspacingmark}', "");
Expect(0, 918000, '\p{Category=nonspacingmark}', "");
Expect(1, 918000, '\p{^Category=nonspacingmark}', "");
Expect(1, 918000, '\P{Category=nonspacingmark}', "");
Expect(0, 918000, '\P{^Category=nonspacingmark}', "");
Expect(1, 917999, '\p{Category:  	NONSPACING_Mark}', "");
Expect(0, 917999, '\p{^Category:  	NONSPACING_Mark}', "");
Expect(0, 917999, '\P{Category:  	NONSPACING_Mark}', "");
Expect(1, 917999, '\P{^Category:  	NONSPACING_Mark}', "");
Expect(0, 918000, '\p{Category:  	NONSPACING_Mark}', "");
Expect(1, 918000, '\p{^Category:  	NONSPACING_Mark}', "");
Expect(1, 918000, '\P{Category:  	NONSPACING_Mark}', "");
Expect(0, 918000, '\P{^Category:  	NONSPACING_Mark}', "");
Error('\p{Is_General_Category=/a/	MN}');
Error('\P{Is_General_Category=/a/	MN}');
Expect(1, 917999, '\p{Is_General_Category=mn}', "");
Expect(0, 917999, '\p{^Is_General_Category=mn}', "");
Expect(0, 917999, '\P{Is_General_Category=mn}', "");
Expect(1, 917999, '\P{^Is_General_Category=mn}', "");
Expect(0, 918000, '\p{Is_General_Category=mn}', "");
Expect(1, 918000, '\p{^Is_General_Category=mn}', "");
Expect(1, 918000, '\P{Is_General_Category=mn}', "");
Expect(0, 918000, '\P{^Is_General_Category=mn}', "");
Expect(1, 917999, '\p{Is_General_Category:	_Mn}', "");
Expect(0, 917999, '\p{^Is_General_Category:	_Mn}', "");
Expect(0, 917999, '\P{Is_General_Category:	_Mn}', "");
Expect(1, 917999, '\P{^Is_General_Category:	_Mn}', "");
Expect(0, 918000, '\p{Is_General_Category:	_Mn}', "");
Expect(1, 918000, '\p{^Is_General_Category:	_Mn}', "");
Expect(1, 918000, '\P{Is_General_Category:	_Mn}', "");
Expect(0, 918000, '\P{^Is_General_Category:	_Mn}', "");
Error('\p{Is_Gc=-:=Nonspacing_MARK}');
Error('\P{Is_Gc=-:=Nonspacing_MARK}');
Expect(1, 917999, '\p{Is_Gc=nonspacingmark}', "");
Expect(0, 917999, '\p{^Is_Gc=nonspacingmark}', "");
Expect(0, 917999, '\P{Is_Gc=nonspacingmark}', "");
Expect(1, 917999, '\P{^Is_Gc=nonspacingmark}', "");
Expect(0, 918000, '\p{Is_Gc=nonspacingmark}', "");
Expect(1, 918000, '\p{^Is_Gc=nonspacingmark}', "");
Expect(1, 918000, '\P{Is_Gc=nonspacingmark}', "");
Expect(0, 918000, '\P{^Is_Gc=nonspacingmark}', "");
Expect(1, 917999, '\p{Is_Gc=-_NONSPACING_Mark}', "");
Expect(0, 917999, '\p{^Is_Gc=-_NONSPACING_Mark}', "");
Expect(0, 917999, '\P{Is_Gc=-_NONSPACING_Mark}', "");
Expect(1, 917999, '\P{^Is_Gc=-_NONSPACING_Mark}', "");
Expect(0, 918000, '\p{Is_Gc=-_NONSPACING_Mark}', "");
Expect(1, 918000, '\p{^Is_Gc=-_NONSPACING_Mark}', "");
Expect(1, 918000, '\P{Is_Gc=-_NONSPACING_Mark}', "");
Expect(0, 918000, '\P{^Is_Gc=-_NONSPACING_Mark}', "");
Error('\p{Is_Category=	 Mn/a/}');
Error('\P{Is_Category=	 Mn/a/}');
Expect(1, 917999, '\p{Is_Category=mn}', "");
Expect(0, 917999, '\p{^Is_Category=mn}', "");
Expect(0, 917999, '\P{Is_Category=mn}', "");
Expect(1, 917999, '\P{^Is_Category=mn}', "");
Expect(0, 918000, '\p{Is_Category=mn}', "");
Expect(1, 918000, '\p{^Is_Category=mn}', "");
Expect(1, 918000, '\P{Is_Category=mn}', "");
Expect(0, 918000, '\P{^Is_Category=mn}', "");
Expect(1, 917999, '\p{Is_Category= -Mn}', "");
Expect(0, 917999, '\p{^Is_Category= -Mn}', "");
Expect(0, 917999, '\P{Is_Category= -Mn}', "");
Expect(1, 917999, '\P{^Is_Category= -Mn}', "");
Expect(0, 918000, '\p{Is_Category= -Mn}', "");
Expect(1, 918000, '\p{^Is_Category= -Mn}', "");
Expect(1, 918000, '\P{Is_Category= -Mn}', "");
Expect(0, 918000, '\P{^Is_Category= -Mn}', "");
Error('\p{General_Category=-/a/Dash_Punctuation}');
Error('\P{General_Category=-/a/Dash_Punctuation}');
Expect(1, 65293, '\p{General_Category:dashpunctuation}', "");
Expect(0, 65293, '\p{^General_Category:dashpunctuation}', "");
Expect(0, 65293, '\P{General_Category:dashpunctuation}', "");
Expect(1, 65293, '\P{^General_Category:dashpunctuation}', "");
Expect(0, 65294, '\p{General_Category:dashpunctuation}', "");
Expect(1, 65294, '\p{^General_Category:dashpunctuation}', "");
Expect(1, 65294, '\P{General_Category:dashpunctuation}', "");
Expect(0, 65294, '\P{^General_Category:dashpunctuation}', "");
Expect(1, 65293, '\p{General_Category=--Dash_PUNCTUATION}', "");
Expect(0, 65293, '\p{^General_Category=--Dash_PUNCTUATION}', "");
Expect(0, 65293, '\P{General_Category=--Dash_PUNCTUATION}', "");
Expect(1, 65293, '\P{^General_Category=--Dash_PUNCTUATION}', "");
Expect(0, 65294, '\p{General_Category=--Dash_PUNCTUATION}', "");
Expect(1, 65294, '\p{^General_Category=--Dash_PUNCTUATION}', "");
Expect(1, 65294, '\P{General_Category=--Dash_PUNCTUATION}', "");
Expect(0, 65294, '\P{^General_Category=--Dash_PUNCTUATION}', "");
Error('\p{Gc: :=_pd}');
Error('\P{Gc: :=_pd}');
Expect(1, 65293, '\p{Gc=pd}', "");
Expect(0, 65293, '\p{^Gc=pd}', "");
Expect(0, 65293, '\P{Gc=pd}', "");
Expect(1, 65293, '\P{^Gc=pd}', "");
Expect(0, 65294, '\p{Gc=pd}', "");
Expect(1, 65294, '\p{^Gc=pd}', "");
Expect(1, 65294, '\P{Gc=pd}', "");
Expect(0, 65294, '\P{^Gc=pd}', "");
Expect(1, 65293, '\p{Gc=-	Pd}', "");
Expect(0, 65293, '\p{^Gc=-	Pd}', "");
Expect(0, 65293, '\P{Gc=-	Pd}', "");
Expect(1, 65293, '\P{^Gc=-	Pd}', "");
Expect(0, 65294, '\p{Gc=-	Pd}', "");
Expect(1, 65294, '\p{^Gc=-	Pd}', "");
Expect(1, 65294, '\P{Gc=-	Pd}', "");
Expect(0, 65294, '\P{^Gc=-	Pd}', "");
Error('\p{Category=-	Dash_PUNCTUATION:=}');
Error('\P{Category=-	Dash_PUNCTUATION:=}');
Expect(1, 65293, '\p{Category=dashpunctuation}', "");
Expect(0, 65293, '\p{^Category=dashpunctuation}', "");
Expect(0, 65293, '\P{Category=dashpunctuation}', "");
Expect(1, 65293, '\P{^Category=dashpunctuation}', "");
Expect(0, 65294, '\p{Category=dashpunctuation}', "");
Expect(1, 65294, '\p{^Category=dashpunctuation}', "");
Expect(1, 65294, '\P{Category=dashpunctuation}', "");
Expect(0, 65294, '\P{^Category=dashpunctuation}', "");
Expect(1, 65293, '\p{Category: _Dash_Punctuation}', "");
Expect(0, 65293, '\p{^Category: _Dash_Punctuation}', "");
Expect(0, 65293, '\P{Category: _Dash_Punctuation}', "");
Expect(1, 65293, '\P{^Category: _Dash_Punctuation}', "");
Expect(0, 65294, '\p{Category: _Dash_Punctuation}', "");
Expect(1, 65294, '\p{^Category: _Dash_Punctuation}', "");
Expect(1, 65294, '\P{Category: _Dash_Punctuation}', "");
Expect(0, 65294, '\P{^Category: _Dash_Punctuation}', "");
Error('\p{Is_General_Category= _Pd:=}');
Error('\P{Is_General_Category= _Pd:=}');
Expect(1, 65293, '\p{Is_General_Category=pd}', "");
Expect(0, 65293, '\p{^Is_General_Category=pd}', "");
Expect(0, 65293, '\P{Is_General_Category=pd}', "");
Expect(1, 65293, '\P{^Is_General_Category=pd}', "");
Expect(0, 65294, '\p{Is_General_Category=pd}', "");
Expect(1, 65294, '\p{^Is_General_Category=pd}', "");
Expect(1, 65294, '\P{Is_General_Category=pd}', "");
Expect(0, 65294, '\P{^Is_General_Category=pd}', "");
Expect(1, 65293, '\p{Is_General_Category=	 PD}', "");
Expect(0, 65293, '\p{^Is_General_Category=	 PD}', "");
Expect(0, 65293, '\P{Is_General_Category=	 PD}', "");
Expect(1, 65293, '\P{^Is_General_Category=	 PD}', "");
Expect(0, 65294, '\p{Is_General_Category=	 PD}', "");
Expect(1, 65294, '\p{^Is_General_Category=	 PD}', "");
Expect(1, 65294, '\P{Is_General_Category=	 PD}', "");
Expect(0, 65294, '\P{^Is_General_Category=	 PD}', "");
Error('\p{Is_Gc=dash_PUNCTUATION:=}');
Error('\P{Is_Gc=dash_PUNCTUATION:=}');
Expect(1, 65293, '\p{Is_Gc=dashpunctuation}', "");
Expect(0, 65293, '\p{^Is_Gc=dashpunctuation}', "");
Expect(0, 65293, '\P{Is_Gc=dashpunctuation}', "");
Expect(1, 65293, '\P{^Is_Gc=dashpunctuation}', "");
Expect(0, 65294, '\p{Is_Gc=dashpunctuation}', "");
Expect(1, 65294, '\p{^Is_Gc=dashpunctuation}', "");
Expect(1, 65294, '\P{Is_Gc=dashpunctuation}', "");
Expect(0, 65294, '\P{^Is_Gc=dashpunctuation}', "");
Expect(1, 65293, '\p{Is_Gc=-	dash_PUNCTUATION}', "");
Expect(0, 65293, '\p{^Is_Gc=-	dash_PUNCTUATION}', "");
Expect(0, 65293, '\P{Is_Gc=-	dash_PUNCTUATION}', "");
Expect(1, 65293, '\P{^Is_Gc=-	dash_PUNCTUATION}', "");
Expect(0, 65294, '\p{Is_Gc=-	dash_PUNCTUATION}', "");
Expect(1, 65294, '\p{^Is_Gc=-	dash_PUNCTUATION}', "");
Expect(1, 65294, '\P{Is_Gc=-	dash_PUNCTUATION}', "");
Expect(0, 65294, '\P{^Is_Gc=-	dash_PUNCTUATION}', "");
Error('\p{Is_Category=/a/_Pd}');
Error('\P{Is_Category=/a/_Pd}');
Expect(1, 65293, '\p{Is_Category=pd}', "");
Expect(0, 65293, '\p{^Is_Category=pd}', "");
Expect(0, 65293, '\P{Is_Category=pd}', "");
Expect(1, 65293, '\P{^Is_Category=pd}', "");
Expect(0, 65294, '\p{Is_Category=pd}', "");
Expect(1, 65294, '\p{^Is_Category=pd}', "");
Expect(1, 65294, '\P{Is_Category=pd}', "");
Expect(0, 65294, '\P{^Is_Category=pd}', "");
Expect(1, 65293, '\p{Is_Category=	Pd}', "");
Expect(0, 65293, '\p{^Is_Category=	Pd}', "");
Expect(0, 65293, '\P{Is_Category=	Pd}', "");
Expect(1, 65293, '\P{^Is_Category=	Pd}', "");
Expect(0, 65294, '\p{Is_Category=	Pd}', "");
Expect(1, 65294, '\p{^Is_Category=	Pd}', "");
Expect(1, 65294, '\P{Is_Category=	Pd}', "");
Expect(0, 65294, '\P{^Is_Category=	Pd}', "");
Error('\p{General_Category=/a/	-Letter_Number}');
Error('\P{General_Category=/a/	-Letter_Number}');
Expect(1, 74850, '\p{General_Category=letternumber}', "");
Expect(0, 74850, '\p{^General_Category=letternumber}', "");
Expect(0, 74850, '\P{General_Category=letternumber}', "");
Expect(1, 74850, '\P{^General_Category=letternumber}', "");
Expect(0, 74851, '\p{General_Category=letternumber}', "");
Expect(1, 74851, '\p{^General_Category=letternumber}', "");
Expect(1, 74851, '\P{General_Category=letternumber}', "");
Expect(0, 74851, '\P{^General_Category=letternumber}', "");
Expect(1, 74850, '\p{General_Category=__Letter_NUMBER}', "");
Expect(0, 74850, '\p{^General_Category=__Letter_NUMBER}', "");
Expect(0, 74850, '\P{General_Category=__Letter_NUMBER}', "");
Expect(1, 74850, '\P{^General_Category=__Letter_NUMBER}', "");
Expect(0, 74851, '\p{General_Category=__Letter_NUMBER}', "");
Expect(1, 74851, '\p{^General_Category=__Letter_NUMBER}', "");
Expect(1, 74851, '\P{General_Category=__Letter_NUMBER}', "");
Expect(0, 74851, '\P{^General_Category=__Letter_NUMBER}', "");
Error('\p{Gc:   /a/ Nl}');
Error('\P{Gc:   /a/ Nl}');
Expect(1, 74850, '\p{Gc:nl}', "");
Expect(0, 74850, '\p{^Gc:nl}', "");
Expect(0, 74850, '\P{Gc:nl}', "");
Expect(1, 74850, '\P{^Gc:nl}', "");
Expect(0, 74851, '\p{Gc:nl}', "");
Expect(1, 74851, '\p{^Gc:nl}', "");
Expect(1, 74851, '\P{Gc:nl}', "");
Expect(0, 74851, '\P{^Gc:nl}', "");
Expect(1, 74850, '\p{Gc:_-Nl}', "");
Expect(0, 74850, '\p{^Gc:_-Nl}', "");
Expect(0, 74850, '\P{Gc:_-Nl}', "");
Expect(1, 74850, '\P{^Gc:_-Nl}', "");
Expect(0, 74851, '\p{Gc:_-Nl}', "");
Expect(1, 74851, '\p{^Gc:_-Nl}', "");
Expect(1, 74851, '\P{Gc:_-Nl}', "");
Expect(0, 74851, '\P{^Gc:_-Nl}', "");
Error('\p{Category=letter_Number/a/}');
Error('\P{Category=letter_Number/a/}');
Expect(1, 74850, '\p{Category=letternumber}', "");
Expect(0, 74850, '\p{^Category=letternumber}', "");
Expect(0, 74850, '\P{Category=letternumber}', "");
Expect(1, 74850, '\P{^Category=letternumber}', "");
Expect(0, 74851, '\p{Category=letternumber}', "");
Expect(1, 74851, '\p{^Category=letternumber}', "");
Expect(1, 74851, '\P{Category=letternumber}', "");
Expect(0, 74851, '\P{^Category=letternumber}', "");
Expect(1, 74850, '\p{Category=	_letter_Number}', "");
Expect(0, 74850, '\p{^Category=	_letter_Number}', "");
Expect(0, 74850, '\P{Category=	_letter_Number}', "");
Expect(1, 74850, '\P{^Category=	_letter_Number}', "");
Expect(0, 74851, '\p{Category=	_letter_Number}', "");
Expect(1, 74851, '\p{^Category=	_letter_Number}', "");
Expect(1, 74851, '\P{Category=	_letter_Number}', "");
Expect(0, 74851, '\P{^Category=	_letter_Number}', "");
Error('\p{Is_General_Category=--nl/a/}');
Error('\P{Is_General_Category=--nl/a/}');
Expect(1, 74850, '\p{Is_General_Category=nl}', "");
Expect(0, 74850, '\p{^Is_General_Category=nl}', "");
Expect(0, 74850, '\P{Is_General_Category=nl}', "");
Expect(1, 74850, '\P{^Is_General_Category=nl}', "");
Expect(0, 74851, '\p{Is_General_Category=nl}', "");
Expect(1, 74851, '\p{^Is_General_Category=nl}', "");
Expect(1, 74851, '\P{Is_General_Category=nl}', "");
Expect(0, 74851, '\P{^Is_General_Category=nl}', "");
Expect(1, 74850, '\p{Is_General_Category= NL}', "");
Expect(0, 74850, '\p{^Is_General_Category= NL}', "");
Expect(0, 74850, '\P{Is_General_Category= NL}', "");
Expect(1, 74850, '\P{^Is_General_Category= NL}', "");
Expect(0, 74851, '\p{Is_General_Category= NL}', "");
Expect(1, 74851, '\p{^Is_General_Category= NL}', "");
Expect(1, 74851, '\P{Is_General_Category= NL}', "");
Expect(0, 74851, '\P{^Is_General_Category= NL}', "");
Error('\p{Is_Gc=-:=LETTER_number}');
Error('\P{Is_Gc=-:=LETTER_number}');
Expect(1, 74850, '\p{Is_Gc=letternumber}', "");
Expect(0, 74850, '\p{^Is_Gc=letternumber}', "");
Expect(0, 74850, '\P{Is_Gc=letternumber}', "");
Expect(1, 74850, '\P{^Is_Gc=letternumber}', "");
Expect(0, 74851, '\p{Is_Gc=letternumber}', "");
Expect(1, 74851, '\p{^Is_Gc=letternumber}', "");
Expect(1, 74851, '\P{Is_Gc=letternumber}', "");
Expect(0, 74851, '\P{^Is_Gc=letternumber}', "");
Expect(1, 74850, '\p{Is_Gc=-letter_number}', "");
Expect(0, 74850, '\p{^Is_Gc=-letter_number}', "");
Expect(0, 74850, '\P{Is_Gc=-letter_number}', "");
Expect(1, 74850, '\P{^Is_Gc=-letter_number}', "");
Expect(0, 74851, '\p{Is_Gc=-letter_number}', "");
Expect(1, 74851, '\p{^Is_Gc=-letter_number}', "");
Expect(1, 74851, '\P{Is_Gc=-letter_number}', "");
Expect(0, 74851, '\P{^Is_Gc=-letter_number}', "");
Error('\p{Is_Category= /a/NL}');
Error('\P{Is_Category= /a/NL}');
Expect(1, 74850, '\p{Is_Category=nl}', "");
Expect(0, 74850, '\p{^Is_Category=nl}', "");
Expect(0, 74850, '\P{Is_Category=nl}', "");
Expect(1, 74850, '\P{^Is_Category=nl}', "");
Expect(0, 74851, '\p{Is_Category=nl}', "");
Expect(1, 74851, '\p{^Is_Category=nl}', "");
Expect(1, 74851, '\P{Is_Category=nl}', "");
Expect(0, 74851, '\P{^Is_Category=nl}', "");
Expect(1, 74850, '\p{Is_Category=_	Nl}', "");
Expect(0, 74850, '\p{^Is_Category=_	Nl}', "");
Expect(0, 74850, '\P{Is_Category=_	Nl}', "");
Expect(1, 74850, '\P{^Is_Category=_	Nl}', "");
Expect(0, 74851, '\p{Is_Category=_	Nl}', "");
Expect(1, 74851, '\p{^Is_Category=_	Nl}', "");
Expect(1, 74851, '\P{Is_Category=_	Nl}', "");
Expect(0, 74851, '\P{^Is_Category=_	Nl}', "");
Error('\p{General_Category=	initial_punctuation/a/}');
Error('\P{General_Category=	initial_punctuation/a/}');
Expect(1, 11808, '\p{General_Category=initialpunctuation}', "");
Expect(0, 11808, '\p{^General_Category=initialpunctuation}', "");
Expect(0, 11808, '\P{General_Category=initialpunctuation}', "");
Expect(1, 11808, '\P{^General_Category=initialpunctuation}', "");
Expect(0, 11809, '\p{General_Category=initialpunctuation}', "");
Expect(1, 11809, '\p{^General_Category=initialpunctuation}', "");
Expect(1, 11809, '\P{General_Category=initialpunctuation}', "");
Expect(0, 11809, '\P{^General_Category=initialpunctuation}', "");
Expect(1, 11808, '\p{General_Category= _Initial_punctuation}', "");
Expect(0, 11808, '\p{^General_Category= _Initial_punctuation}', "");
Expect(0, 11808, '\P{General_Category= _Initial_punctuation}', "");
Expect(1, 11808, '\P{^General_Category= _Initial_punctuation}', "");
Expect(0, 11809, '\p{General_Category= _Initial_punctuation}', "");
Expect(1, 11809, '\p{^General_Category= _Initial_punctuation}', "");
Expect(1, 11809, '\P{General_Category= _Initial_punctuation}', "");
Expect(0, 11809, '\P{^General_Category= _Initial_punctuation}', "");
Error('\p{Gc=		PI/a/}');
Error('\P{Gc=		PI/a/}');
Expect(1, 11808, '\p{Gc=pi}', "");
Expect(0, 11808, '\p{^Gc=pi}', "");
Expect(0, 11808, '\P{Gc=pi}', "");
Expect(1, 11808, '\P{^Gc=pi}', "");
Expect(0, 11809, '\p{Gc=pi}', "");
Expect(1, 11809, '\p{^Gc=pi}', "");
Expect(1, 11809, '\P{Gc=pi}', "");
Expect(0, 11809, '\P{^Gc=pi}', "");
Expect(1, 11808, '\p{Gc=	Pi}', "");
Expect(0, 11808, '\p{^Gc=	Pi}', "");
Expect(0, 11808, '\P{Gc=	Pi}', "");
Expect(1, 11808, '\P{^Gc=	Pi}', "");
Expect(0, 11809, '\p{Gc=	Pi}', "");
Expect(1, 11809, '\p{^Gc=	Pi}', "");
Expect(1, 11809, '\P{Gc=	Pi}', "");
Expect(0, 11809, '\P{^Gc=	Pi}', "");
Error('\p{Category=_/a/Initial_Punctuation}');
Error('\P{Category=_/a/Initial_Punctuation}');
Expect(1, 11808, '\p{Category=initialpunctuation}', "");
Expect(0, 11808, '\p{^Category=initialpunctuation}', "");
Expect(0, 11808, '\P{Category=initialpunctuation}', "");
Expect(1, 11808, '\P{^Category=initialpunctuation}', "");
Expect(0, 11809, '\p{Category=initialpunctuation}', "");
Expect(1, 11809, '\p{^Category=initialpunctuation}', "");
Expect(1, 11809, '\P{Category=initialpunctuation}', "");
Expect(0, 11809, '\P{^Category=initialpunctuation}', "");
Expect(1, 11808, '\p{Category=	initial_PUNCTUATION}', "");
Expect(0, 11808, '\p{^Category=	initial_PUNCTUATION}', "");
Expect(0, 11808, '\P{Category=	initial_PUNCTUATION}', "");
Expect(1, 11808, '\P{^Category=	initial_PUNCTUATION}', "");
Expect(0, 11809, '\p{Category=	initial_PUNCTUATION}', "");
Expect(1, 11809, '\p{^Category=	initial_PUNCTUATION}', "");
Expect(1, 11809, '\P{Category=	initial_PUNCTUATION}', "");
Expect(0, 11809, '\P{^Category=	initial_PUNCTUATION}', "");
Error('\p{Is_General_Category=-_pi:=}');
Error('\P{Is_General_Category=-_pi:=}');
Expect(1, 11808, '\p{Is_General_Category=pi}', "");
Expect(0, 11808, '\p{^Is_General_Category=pi}', "");
Expect(0, 11808, '\P{Is_General_Category=pi}', "");
Expect(1, 11808, '\P{^Is_General_Category=pi}', "");
Expect(0, 11809, '\p{Is_General_Category=pi}', "");
Expect(1, 11809, '\p{^Is_General_Category=pi}', "");
Expect(1, 11809, '\P{Is_General_Category=pi}', "");
Expect(0, 11809, '\P{^Is_General_Category=pi}', "");
Expect(1, 11808, '\p{Is_General_Category=__Pi}', "");
Expect(0, 11808, '\p{^Is_General_Category=__Pi}', "");
Expect(0, 11808, '\P{Is_General_Category=__Pi}', "");
Expect(1, 11808, '\P{^Is_General_Category=__Pi}', "");
Expect(0, 11809, '\p{Is_General_Category=__Pi}', "");
Expect(1, 11809, '\p{^Is_General_Category=__Pi}', "");
Expect(1, 11809, '\P{Is_General_Category=__Pi}', "");
Expect(0, 11809, '\P{^Is_General_Category=__Pi}', "");
Error('\p{Is_Gc=/a/initial_PUNCTUATION}');
Error('\P{Is_Gc=/a/initial_PUNCTUATION}');
Expect(1, 11808, '\p{Is_Gc=initialpunctuation}', "");
Expect(0, 11808, '\p{^Is_Gc=initialpunctuation}', "");
Expect(0, 11808, '\P{Is_Gc=initialpunctuation}', "");
Expect(1, 11808, '\P{^Is_Gc=initialpunctuation}', "");
Expect(0, 11809, '\p{Is_Gc=initialpunctuation}', "");
Expect(1, 11809, '\p{^Is_Gc=initialpunctuation}', "");
Expect(1, 11809, '\P{Is_Gc=initialpunctuation}', "");
Expect(0, 11809, '\P{^Is_Gc=initialpunctuation}', "");
Expect(1, 11808, '\p{Is_Gc= Initial_PUNCTUATION}', "");
Expect(0, 11808, '\p{^Is_Gc= Initial_PUNCTUATION}', "");
Expect(0, 11808, '\P{Is_Gc= Initial_PUNCTUATION}', "");
Expect(1, 11808, '\P{^Is_Gc= Initial_PUNCTUATION}', "");
Expect(0, 11809, '\p{Is_Gc= Initial_PUNCTUATION}', "");
Expect(1, 11809, '\p{^Is_Gc= Initial_PUNCTUATION}', "");
Expect(1, 11809, '\P{Is_Gc= Initial_PUNCTUATION}', "");
Expect(0, 11809, '\P{^Is_Gc= Initial_PUNCTUATION}', "");
Error('\p{Is_Category=_:=pi}');
Error('\P{Is_Category=_:=pi}');
Expect(1, 11808, '\p{Is_Category=pi}', "");
Expect(0, 11808, '\p{^Is_Category=pi}', "");
Expect(0, 11808, '\P{Is_Category=pi}', "");
Expect(1, 11808, '\P{^Is_Category=pi}', "");
Expect(0, 11809, '\p{Is_Category=pi}', "");
Expect(1, 11809, '\p{^Is_Category=pi}', "");
Expect(1, 11809, '\P{Is_Category=pi}', "");
Expect(0, 11809, '\P{^Is_Category=pi}', "");
Expect(1, 11808, '\p{Is_Category=		pi}', "");
Expect(0, 11808, '\p{^Is_Category=		pi}', "");
Expect(0, 11808, '\P{Is_Category=		pi}', "");
Expect(1, 11808, '\P{^Is_Category=		pi}', "");
Expect(0, 11809, '\p{Is_Category=		pi}', "");
Expect(1, 11809, '\p{^Is_Category=		pi}', "");
Expect(1, 11809, '\P{Is_Category=		pi}', "");
Expect(0, 11809, '\P{^Is_Category=		pi}', "");
Error('\p{General_Category= Currency_SYMBOL:=}');
Error('\P{General_Category= Currency_SYMBOL:=}');
Expect(1, 65510, '\p{General_Category: currencysymbol}', "");
Expect(0, 65510, '\p{^General_Category: currencysymbol}', "");
Expect(0, 65510, '\P{General_Category: currencysymbol}', "");
Expect(1, 65510, '\P{^General_Category: currencysymbol}', "");
Expect(0, 65511, '\p{General_Category: currencysymbol}', "");
Expect(1, 65511, '\p{^General_Category: currencysymbol}', "");
Expect(1, 65511, '\P{General_Category: currencysymbol}', "");
Expect(0, 65511, '\P{^General_Category: currencysymbol}', "");
Expect(1, 65510, '\p{General_Category= 	Currency_Symbol}', "");
Expect(0, 65510, '\p{^General_Category= 	Currency_Symbol}', "");
Expect(0, 65510, '\P{General_Category= 	Currency_Symbol}', "");
Expect(1, 65510, '\P{^General_Category= 	Currency_Symbol}', "");
Expect(0, 65511, '\p{General_Category= 	Currency_Symbol}', "");
Expect(1, 65511, '\p{^General_Category= 	Currency_Symbol}', "");
Expect(1, 65511, '\P{General_Category= 	Currency_Symbol}', "");
Expect(0, 65511, '\P{^General_Category= 	Currency_Symbol}', "");
Error('\p{Gc=-Sc/a/}');
Error('\P{Gc=-Sc/a/}');
Expect(1, 65510, '\p{Gc=sc}', "");
Expect(0, 65510, '\p{^Gc=sc}', "");
Expect(0, 65510, '\P{Gc=sc}', "");
Expect(1, 65510, '\P{^Gc=sc}', "");
Expect(0, 65511, '\p{Gc=sc}', "");
Expect(1, 65511, '\p{^Gc=sc}', "");
Expect(1, 65511, '\P{Gc=sc}', "");
Expect(0, 65511, '\P{^Gc=sc}', "");
Expect(1, 65510, '\p{Gc=_ sc}', "");
Expect(0, 65510, '\p{^Gc=_ sc}', "");
Expect(0, 65510, '\P{Gc=_ sc}', "");
Expect(1, 65510, '\P{^Gc=_ sc}', "");
Expect(0, 65511, '\p{Gc=_ sc}', "");
Expect(1, 65511, '\p{^Gc=_ sc}', "");
Expect(1, 65511, '\P{Gc=_ sc}', "");
Expect(0, 65511, '\P{^Gc=_ sc}', "");
Error('\p{Category=/a/Currency_symbol}');
Error('\P{Category=/a/Currency_symbol}');
Expect(1, 65510, '\p{Category=currencysymbol}', "");
Expect(0, 65510, '\p{^Category=currencysymbol}', "");
Expect(0, 65510, '\P{Category=currencysymbol}', "");
Expect(1, 65510, '\P{^Category=currencysymbol}', "");
Expect(0, 65511, '\p{Category=currencysymbol}', "");
Expect(1, 65511, '\p{^Category=currencysymbol}', "");
Expect(1, 65511, '\P{Category=currencysymbol}', "");
Expect(0, 65511, '\P{^Category=currencysymbol}', "");
Expect(1, 65510, '\p{Category: -currency_Symbol}', "");
Expect(0, 65510, '\p{^Category: -currency_Symbol}', "");
Expect(0, 65510, '\P{Category: -currency_Symbol}', "");
Expect(1, 65510, '\P{^Category: -currency_Symbol}', "");
Expect(0, 65511, '\p{Category: -currency_Symbol}', "");
Expect(1, 65511, '\p{^Category: -currency_Symbol}', "");
Expect(1, 65511, '\P{Category: -currency_Symbol}', "");
Expect(0, 65511, '\P{^Category: -currency_Symbol}', "");
Error('\p{Is_General_Category=	Sc/a/}');
Error('\P{Is_General_Category=	Sc/a/}');
Expect(1, 65510, '\p{Is_General_Category=sc}', "");
Expect(0, 65510, '\p{^Is_General_Category=sc}', "");
Expect(0, 65510, '\P{Is_General_Category=sc}', "");
Expect(1, 65510, '\P{^Is_General_Category=sc}', "");
Expect(0, 65511, '\p{Is_General_Category=sc}', "");
Expect(1, 65511, '\p{^Is_General_Category=sc}', "");
Expect(1, 65511, '\P{Is_General_Category=sc}', "");
Expect(0, 65511, '\P{^Is_General_Category=sc}', "");
Expect(1, 65510, '\p{Is_General_Category:	_SC}', "");
Expect(0, 65510, '\p{^Is_General_Category:	_SC}', "");
Expect(0, 65510, '\P{Is_General_Category:	_SC}', "");
Expect(1, 65510, '\P{^Is_General_Category:	_SC}', "");
Expect(0, 65511, '\p{Is_General_Category:	_SC}', "");
Expect(1, 65511, '\p{^Is_General_Category:	_SC}', "");
Expect(1, 65511, '\P{Is_General_Category:	_SC}', "");
Expect(0, 65511, '\P{^Is_General_Category:	_SC}', "");
Error('\p{Is_Gc: _:=currency_Symbol}');
Error('\P{Is_Gc: _:=currency_Symbol}');
Expect(1, 65510, '\p{Is_Gc:currencysymbol}', "");
Expect(0, 65510, '\p{^Is_Gc:currencysymbol}', "");
Expect(0, 65510, '\P{Is_Gc:currencysymbol}', "");
Expect(1, 65510, '\P{^Is_Gc:currencysymbol}', "");
Expect(0, 65511, '\p{Is_Gc:currencysymbol}', "");
Expect(1, 65511, '\p{^Is_Gc:currencysymbol}', "");
Expect(1, 65511, '\P{Is_Gc:currencysymbol}', "");
Expect(0, 65511, '\P{^Is_Gc:currencysymbol}', "");
Expect(1, 65510, '\p{Is_Gc= CURRENCY_Symbol}', "");
Expect(0, 65510, '\p{^Is_Gc= CURRENCY_Symbol}', "");
Expect(0, 65510, '\P{Is_Gc= CURRENCY_Symbol}', "");
Expect(1, 65510, '\P{^Is_Gc= CURRENCY_Symbol}', "");
Expect(0, 65511, '\p{Is_Gc= CURRENCY_Symbol}', "");
Expect(1, 65511, '\p{^Is_Gc= CURRENCY_Symbol}', "");
Expect(1, 65511, '\P{Is_Gc= CURRENCY_Symbol}', "");
Expect(0, 65511, '\P{^Is_Gc= CURRENCY_Symbol}', "");
Error('\p{Is_Category= -SC/a/}');
Error('\P{Is_Category= -SC/a/}');
Expect(1, 65510, '\p{Is_Category=sc}', "");
Expect(0, 65510, '\p{^Is_Category=sc}', "");
Expect(0, 65510, '\P{Is_Category=sc}', "");
Expect(1, 65510, '\P{^Is_Category=sc}', "");
Expect(0, 65511, '\p{Is_Category=sc}', "");
Expect(1, 65511, '\p{^Is_Category=sc}', "");
Expect(1, 65511, '\P{Is_Category=sc}', "");
Expect(0, 65511, '\P{^Is_Category=sc}', "");
Expect(1, 65510, '\p{Is_Category=_	sc}', "");
Expect(0, 65510, '\p{^Is_Category=_	sc}', "");
Expect(0, 65510, '\P{Is_Category=_	sc}', "");
Expect(1, 65510, '\P{^Is_Category=_	sc}', "");
Expect(0, 65511, '\p{Is_Category=_	sc}', "");
Expect(1, 65511, '\p{^Is_Category=_	sc}', "");
Expect(1, 65511, '\P{Is_Category=_	sc}', "");
Expect(0, 65511, '\P{^Is_Category=_	sc}', "");
Error('\p{General_Category=:=		Format}');
Error('\P{General_Category=:=		Format}');
Expect(1, 917631, '\p{General_Category:format}', "");
Expect(0, 917631, '\p{^General_Category:format}', "");
Expect(0, 917631, '\P{General_Category:format}', "");
Expect(1, 917631, '\P{^General_Category:format}', "");
Expect(0, 917632, '\p{General_Category:format}', "");
Expect(1, 917632, '\p{^General_Category:format}', "");
Expect(1, 917632, '\P{General_Category:format}', "");
Expect(0, 917632, '\P{^General_Category:format}', "");
Expect(1, 917631, '\p{General_Category=- format}', "");
Expect(0, 917631, '\p{^General_Category=- format}', "");
Expect(0, 917631, '\P{General_Category=- format}', "");
Expect(1, 917631, '\P{^General_Category=- format}', "");
Expect(0, 917632, '\p{General_Category=- format}', "");
Expect(1, 917632, '\p{^General_Category=- format}', "");
Expect(1, 917632, '\P{General_Category=- format}', "");
Expect(0, 917632, '\P{^General_Category=- format}', "");
Error('\p{Gc=/a/Cf}');
Error('\P{Gc=/a/Cf}');
Expect(1, 917631, '\p{Gc=cf}', "");
Expect(0, 917631, '\p{^Gc=cf}', "");
Expect(0, 917631, '\P{Gc=cf}', "");
Expect(1, 917631, '\P{^Gc=cf}', "");
Expect(0, 917632, '\p{Gc=cf}', "");
Expect(1, 917632, '\p{^Gc=cf}', "");
Expect(1, 917632, '\P{Gc=cf}', "");
Expect(0, 917632, '\P{^Gc=cf}', "");
Expect(1, 917631, '\p{Gc=_	cf}', "");
Expect(0, 917631, '\p{^Gc=_	cf}', "");
Expect(0, 917631, '\P{Gc=_	cf}', "");
Expect(1, 917631, '\P{^Gc=_	cf}', "");
Expect(0, 917632, '\p{Gc=_	cf}', "");
Expect(1, 917632, '\p{^Gc=_	cf}', "");
Expect(1, 917632, '\P{Gc=_	cf}', "");
Expect(0, 917632, '\P{^Gc=_	cf}', "");
Error('\p{Category::=-FORMAT}');
Error('\P{Category::=-FORMAT}');
Expect(1, 917631, '\p{Category=format}', "");
Expect(0, 917631, '\p{^Category=format}', "");
Expect(0, 917631, '\P{Category=format}', "");
Expect(1, 917631, '\P{^Category=format}', "");
Expect(0, 917632, '\p{Category=format}', "");
Expect(1, 917632, '\p{^Category=format}', "");
Expect(1, 917632, '\P{Category=format}', "");
Expect(0, 917632, '\P{^Category=format}', "");
Expect(1, 917631, '\p{Category= -FORMAT}', "");
Expect(0, 917631, '\p{^Category= -FORMAT}', "");
Expect(0, 917631, '\P{Category= -FORMAT}', "");
Expect(1, 917631, '\P{^Category= -FORMAT}', "");
Expect(0, 917632, '\p{Category= -FORMAT}', "");
Expect(1, 917632, '\p{^Category= -FORMAT}', "");
Expect(1, 917632, '\P{Category= -FORMAT}', "");
Expect(0, 917632, '\P{^Category= -FORMAT}', "");
Error('\p{Is_General_Category=-Cf/a/}');
Error('\P{Is_General_Category=-Cf/a/}');
Expect(1, 917631, '\p{Is_General_Category=cf}', "");
Expect(0, 917631, '\p{^Is_General_Category=cf}', "");
Expect(0, 917631, '\P{Is_General_Category=cf}', "");
Expect(1, 917631, '\P{^Is_General_Category=cf}', "");
Expect(0, 917632, '\p{Is_General_Category=cf}', "");
Expect(1, 917632, '\p{^Is_General_Category=cf}', "");
Expect(1, 917632, '\P{Is_General_Category=cf}', "");
Expect(0, 917632, '\P{^Is_General_Category=cf}', "");
Expect(1, 917631, '\p{Is_General_Category:	_Cf}', "");
Expect(0, 917631, '\p{^Is_General_Category:	_Cf}', "");
Expect(0, 917631, '\P{Is_General_Category:	_Cf}', "");
Expect(1, 917631, '\P{^Is_General_Category:	_Cf}', "");
Expect(0, 917632, '\p{Is_General_Category:	_Cf}', "");
Expect(1, 917632, '\p{^Is_General_Category:	_Cf}', "");
Expect(1, 917632, '\P{Is_General_Category:	_Cf}', "");
Expect(0, 917632, '\P{^Is_General_Category:	_Cf}', "");
Error('\p{Is_Gc=/a/-FORMAT}');
Error('\P{Is_Gc=/a/-FORMAT}');
Expect(1, 917631, '\p{Is_Gc=format}', "");
Expect(0, 917631, '\p{^Is_Gc=format}', "");
Expect(0, 917631, '\P{Is_Gc=format}', "");
Expect(1, 917631, '\P{^Is_Gc=format}', "");
Expect(0, 917632, '\p{Is_Gc=format}', "");
Expect(1, 917632, '\p{^Is_Gc=format}', "");
Expect(1, 917632, '\P{Is_Gc=format}', "");
Expect(0, 917632, '\P{^Is_Gc=format}', "");
Expect(1, 917631, '\p{Is_Gc= -Format}', "");
Expect(0, 917631, '\p{^Is_Gc= -Format}', "");
Expect(0, 917631, '\P{Is_Gc= -Format}', "");
Expect(1, 917631, '\P{^Is_Gc= -Format}', "");
Expect(0, 917632, '\p{Is_Gc= -Format}', "");
Expect(1, 917632, '\p{^Is_Gc= -Format}', "");
Expect(1, 917632, '\P{Is_Gc= -Format}', "");
Expect(0, 917632, '\P{^Is_Gc= -Format}', "");
Error('\p{Is_Category=-_Cf:=}');
Error('\P{Is_Category=-_Cf:=}');
Expect(1, 917631, '\p{Is_Category=cf}', "");
Expect(0, 917631, '\p{^Is_Category=cf}', "");
Expect(0, 917631, '\P{Is_Category=cf}', "");
Expect(1, 917631, '\P{^Is_Category=cf}', "");
Expect(0, 917632, '\p{Is_Category=cf}', "");
Expect(1, 917632, '\p{^Is_Category=cf}', "");
Expect(1, 917632, '\P{Is_Category=cf}', "");
Expect(0, 917632, '\P{^Is_Category=cf}', "");
Expect(1, 917631, '\p{Is_Category= -cf}', "");
Expect(0, 917631, '\p{^Is_Category= -cf}', "");
Expect(0, 917631, '\P{Is_Category= -cf}', "");
Expect(1, 917631, '\P{^Is_Category= -cf}', "");
Expect(0, 917632, '\p{Is_Category= -cf}', "");
Expect(1, 917632, '\p{^Is_Category= -cf}', "");
Expect(1, 917632, '\P{Is_Category= -cf}', "");
Expect(0, 917632, '\P{^Is_Category= -cf}', "");
Error('\p{IDS_Trinary_Operator=/a/Yes}');
Error('\P{IDS_Trinary_Operator=/a/Yes}');
Expect(1, 12275, '\p{IDS_Trinary_Operator:	yes}', "");
Expect(0, 12275, '\p{^IDS_Trinary_Operator:	yes}', "");
Expect(0, 12275, '\P{IDS_Trinary_Operator:	yes}', "");
Expect(1, 12275, '\P{^IDS_Trinary_Operator:	yes}', "");
Expect(0, 12276, '\p{IDS_Trinary_Operator:	yes}', "");
Expect(1, 12276, '\p{^IDS_Trinary_Operator:	yes}', "");
Expect(1, 12276, '\P{IDS_Trinary_Operator:	yes}', "");
Expect(0, 12276, '\P{^IDS_Trinary_Operator:	yes}', "");
Expect(1, 12275, '\p{IDS_Trinary_Operator=	YES}', "");
Expect(0, 12275, '\p{^IDS_Trinary_Operator=	YES}', "");
Expect(0, 12275, '\P{IDS_Trinary_Operator=	YES}', "");
Expect(1, 12275, '\P{^IDS_Trinary_Operator=	YES}', "");
Expect(0, 12276, '\p{IDS_Trinary_Operator=	YES}', "");
Expect(1, 12276, '\p{^IDS_Trinary_Operator=	YES}', "");
Expect(1, 12276, '\P{IDS_Trinary_Operator=	YES}', "");
Expect(0, 12276, '\P{^IDS_Trinary_Operator=	YES}', "");
Error('\p{IDST=/a/_	y}');
Error('\P{IDST=/a/_	y}');
Expect(1, 12275, '\p{IDST=y}', "");
Expect(0, 12275, '\p{^IDST=y}', "");
Expect(0, 12275, '\P{IDST=y}', "");
Expect(1, 12275, '\P{^IDST=y}', "");
Expect(0, 12276, '\p{IDST=y}', "");
Expect(1, 12276, '\p{^IDST=y}', "");
Expect(1, 12276, '\P{IDST=y}', "");
Expect(0, 12276, '\P{^IDST=y}', "");
Expect(1, 12275, '\p{IDST=	Y}', "");
Expect(0, 12275, '\p{^IDST=	Y}', "");
Expect(0, 12275, '\P{IDST=	Y}', "");
Expect(1, 12275, '\P{^IDST=	Y}', "");
Expect(0, 12276, '\p{IDST=	Y}', "");
Expect(1, 12276, '\p{^IDST=	Y}', "");
Expect(1, 12276, '\P{IDST=	Y}', "");
Expect(0, 12276, '\P{^IDST=	Y}', "");
Error('\p{Is_IDS_Trinary_Operator=	:=T}');
Error('\P{Is_IDS_Trinary_Operator=	:=T}');
Expect(1, 12275, '\p{Is_IDS_Trinary_Operator=t}', "");
Expect(0, 12275, '\p{^Is_IDS_Trinary_Operator=t}', "");
Expect(0, 12275, '\P{Is_IDS_Trinary_Operator=t}', "");
Expect(1, 12275, '\P{^Is_IDS_Trinary_Operator=t}', "");
Expect(0, 12276, '\p{Is_IDS_Trinary_Operator=t}', "");
Expect(1, 12276, '\p{^Is_IDS_Trinary_Operator=t}', "");
Expect(1, 12276, '\P{Is_IDS_Trinary_Operator=t}', "");
Expect(0, 12276, '\P{^Is_IDS_Trinary_Operator=t}', "");
Expect(1, 12275, '\p{Is_IDS_Trinary_Operator= T}', "");
Expect(0, 12275, '\p{^Is_IDS_Trinary_Operator= T}', "");
Expect(0, 12275, '\P{Is_IDS_Trinary_Operator= T}', "");
Expect(1, 12275, '\P{^Is_IDS_Trinary_Operator= T}', "");
Expect(0, 12276, '\p{Is_IDS_Trinary_Operator= T}', "");
Expect(1, 12276, '\p{^Is_IDS_Trinary_Operator= T}', "");
Expect(1, 12276, '\P{Is_IDS_Trinary_Operator= T}', "");
Expect(0, 12276, '\P{^Is_IDS_Trinary_Operator= T}', "");
Error('\p{Is_IDST=/a/ 	TRUE}');
Error('\P{Is_IDST=/a/ 	TRUE}');
Expect(1, 12275, '\p{Is_IDST=true}', "");
Expect(0, 12275, '\p{^Is_IDST=true}', "");
Expect(0, 12275, '\P{Is_IDST=true}', "");
Expect(1, 12275, '\P{^Is_IDST=true}', "");
Expect(0, 12276, '\p{Is_IDST=true}', "");
Expect(1, 12276, '\p{^Is_IDST=true}', "");
Expect(1, 12276, '\P{Is_IDST=true}', "");
Expect(0, 12276, '\P{^Is_IDST=true}', "");
Expect(1, 12275, '\p{Is_IDST= -True}', "");
Expect(0, 12275, '\p{^Is_IDST= -True}', "");
Expect(0, 12275, '\P{Is_IDST= -True}', "");
Expect(1, 12275, '\P{^Is_IDST= -True}', "");
Expect(0, 12276, '\p{Is_IDST= -True}', "");
Expect(1, 12276, '\p{^Is_IDST= -True}', "");
Expect(1, 12276, '\P{Is_IDST= -True}', "");
Expect(0, 12276, '\P{^Is_IDST= -True}', "");
Error('\p{IDS_Trinary_Operator=:=-NO}');
Error('\P{IDS_Trinary_Operator=:=-NO}');
Expect(1, 12276, '\p{IDS_Trinary_Operator=no}', "");
Expect(0, 12276, '\p{^IDS_Trinary_Operator=no}', "");
Expect(0, 12276, '\P{IDS_Trinary_Operator=no}', "");
Expect(1, 12276, '\P{^IDS_Trinary_Operator=no}', "");
Expect(0, 12275, '\p{IDS_Trinary_Operator=no}', "");
Expect(1, 12275, '\p{^IDS_Trinary_Operator=no}', "");
Expect(1, 12275, '\P{IDS_Trinary_Operator=no}', "");
Expect(0, 12275, '\P{^IDS_Trinary_Operator=no}', "");
Expect(1, 12276, '\p{IDS_Trinary_Operator=		NO}', "");
Expect(0, 12276, '\p{^IDS_Trinary_Operator=		NO}', "");
Expect(0, 12276, '\P{IDS_Trinary_Operator=		NO}', "");
Expect(1, 12276, '\P{^IDS_Trinary_Operator=		NO}', "");
Expect(0, 12275, '\p{IDS_Trinary_Operator=		NO}', "");
Expect(1, 12275, '\p{^IDS_Trinary_Operator=		NO}', "");
Expect(1, 12275, '\P{IDS_Trinary_Operator=		NO}', "");
Expect(0, 12275, '\P{^IDS_Trinary_Operator=		NO}', "");
Error('\p{IDST=/a/  N}');
Error('\P{IDST=/a/  N}');
Expect(1, 12276, '\p{IDST=n}', "");
Expect(0, 12276, '\p{^IDST=n}', "");
Expect(0, 12276, '\P{IDST=n}', "");
Expect(1, 12276, '\P{^IDST=n}', "");
Expect(0, 12275, '\p{IDST=n}', "");
Expect(1, 12275, '\p{^IDST=n}', "");
Expect(1, 12275, '\P{IDST=n}', "");
Expect(0, 12275, '\P{^IDST=n}', "");
Expect(1, 12276, '\p{IDST=-	N}', "");
Expect(0, 12276, '\p{^IDST=-	N}', "");
Expect(0, 12276, '\P{IDST=-	N}', "");
Expect(1, 12276, '\P{^IDST=-	N}', "");
Expect(0, 12275, '\p{IDST=-	N}', "");
Expect(1, 12275, '\p{^IDST=-	N}', "");
Expect(1, 12275, '\P{IDST=-	N}', "");
Expect(0, 12275, '\P{^IDST=-	N}', "");
Error('\p{Is_IDS_Trinary_Operator= _f/a/}');
Error('\P{Is_IDS_Trinary_Operator= _f/a/}');
Expect(1, 12276, '\p{Is_IDS_Trinary_Operator: f}', "");
Expect(0, 12276, '\p{^Is_IDS_Trinary_Operator: f}', "");
Expect(0, 12276, '\P{Is_IDS_Trinary_Operator: f}', "");
Expect(1, 12276, '\P{^Is_IDS_Trinary_Operator: f}', "");
Expect(0, 12275, '\p{Is_IDS_Trinary_Operator: f}', "");
Expect(1, 12275, '\p{^Is_IDS_Trinary_Operator: f}', "");
Expect(1, 12275, '\P{Is_IDS_Trinary_Operator: f}', "");
Expect(0, 12275, '\P{^Is_IDS_Trinary_Operator: f}', "");
Expect(1, 12276, '\p{Is_IDS_Trinary_Operator=	_F}', "");
Expect(0, 12276, '\p{^Is_IDS_Trinary_Operator=	_F}', "");
Expect(0, 12276, '\P{Is_IDS_Trinary_Operator=	_F}', "");
Expect(1, 12276, '\P{^Is_IDS_Trinary_Operator=	_F}', "");
Expect(0, 12275, '\p{Is_IDS_Trinary_Operator=	_F}', "");
Expect(1, 12275, '\p{^Is_IDS_Trinary_Operator=	_F}', "");
Expect(1, 12275, '\P{Is_IDS_Trinary_Operator=	_F}', "");
Expect(0, 12275, '\P{^Is_IDS_Trinary_Operator=	_F}', "");
Error('\p{Is_IDST=  FALSE:=}');
Error('\P{Is_IDST=  FALSE:=}');
Expect(1, 12276, '\p{Is_IDST=false}', "");
Expect(0, 12276, '\p{^Is_IDST=false}', "");
Expect(0, 12276, '\P{Is_IDST=false}', "");
Expect(1, 12276, '\P{^Is_IDST=false}', "");
Expect(0, 12275, '\p{Is_IDST=false}', "");
Expect(1, 12275, '\p{^Is_IDST=false}', "");
Expect(1, 12275, '\P{Is_IDST=false}', "");
Expect(0, 12275, '\P{^Is_IDST=false}', "");
Expect(1, 12276, '\p{Is_IDST=_False}', "");
Expect(0, 12276, '\p{^Is_IDST=_False}', "");
Expect(0, 12276, '\P{Is_IDST=_False}', "");
Expect(1, 12276, '\P{^Is_IDST=_False}', "");
Expect(0, 12275, '\p{Is_IDST=_False}', "");
Expect(1, 12275, '\p{^Is_IDST=_False}', "");
Expect(1, 12275, '\P{Is_IDST=_False}', "");
Expect(0, 12275, '\P{^Is_IDST=_False}', "");
Error('\p{Grapheme_Cluster_Break:	__spacingmark/a/}');
Error('\P{Grapheme_Cluster_Break:	__spacingmark/a/}');
Expect(1, 119149, '\p{Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 119149, '\p{^Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 119149, '\P{Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 119149, '\P{^Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 119150, '\p{Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 119150, '\p{^Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 119150, '\P{Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 119150, '\P{^Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 119149, '\p{Grapheme_Cluster_Break=-	SPACINGMARK}', "");
Expect(0, 119149, '\p{^Grapheme_Cluster_Break=-	SPACINGMARK}', "");
Expect(0, 119149, '\P{Grapheme_Cluster_Break=-	SPACINGMARK}', "");
Expect(1, 119149, '\P{^Grapheme_Cluster_Break=-	SPACINGMARK}', "");
Expect(0, 119150, '\p{Grapheme_Cluster_Break=-	SPACINGMARK}', "");
Expect(1, 119150, '\p{^Grapheme_Cluster_Break=-	SPACINGMARK}', "");
Expect(1, 119150, '\P{Grapheme_Cluster_Break=-	SPACINGMARK}', "");
Expect(0, 119150, '\P{^Grapheme_Cluster_Break=-	SPACINGMARK}', "");
Error('\p{GCB= :=sm}');
Error('\P{GCB= :=sm}');
Expect(1, 119149, '\p{GCB=sm}', "");
Expect(0, 119149, '\p{^GCB=sm}', "");
Expect(0, 119149, '\P{GCB=sm}', "");
Expect(1, 119149, '\P{^GCB=sm}', "");
Expect(0, 119150, '\p{GCB=sm}', "");
Expect(1, 119150, '\p{^GCB=sm}', "");
Expect(1, 119150, '\P{GCB=sm}', "");
Expect(0, 119150, '\P{^GCB=sm}', "");
Expect(1, 119149, '\p{GCB= 	SM}', "");
Expect(0, 119149, '\p{^GCB= 	SM}', "");
Expect(0, 119149, '\P{GCB= 	SM}', "");
Expect(1, 119149, '\P{^GCB= 	SM}', "");
Expect(0, 119150, '\p{GCB= 	SM}', "");
Expect(1, 119150, '\p{^GCB= 	SM}', "");
Expect(1, 119150, '\P{GCB= 	SM}', "");
Expect(0, 119150, '\P{^GCB= 	SM}', "");
Error('\p{Is_Grapheme_Cluster_Break=/a/ spacingmark}');
Error('\P{Is_Grapheme_Cluster_Break=/a/ spacingmark}');
Expect(1, 119149, '\p{Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 119149, '\p{^Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 119149, '\P{Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 119149, '\P{^Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 119150, '\p{Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 119150, '\p{^Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 119150, '\P{Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 119150, '\P{^Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 119149, '\p{Is_Grapheme_Cluster_Break=  SPACINGMARK}', "");
Expect(0, 119149, '\p{^Is_Grapheme_Cluster_Break=  SPACINGMARK}', "");
Expect(0, 119149, '\P{Is_Grapheme_Cluster_Break=  SPACINGMARK}', "");
Expect(1, 119149, '\P{^Is_Grapheme_Cluster_Break=  SPACINGMARK}', "");
Expect(0, 119150, '\p{Is_Grapheme_Cluster_Break=  SPACINGMARK}', "");
Expect(1, 119150, '\p{^Is_Grapheme_Cluster_Break=  SPACINGMARK}', "");
Expect(1, 119150, '\P{Is_Grapheme_Cluster_Break=  SPACINGMARK}', "");
Expect(0, 119150, '\P{^Is_Grapheme_Cluster_Break=  SPACINGMARK}', "");
Error('\p{Is_GCB:	 :=SM}');
Error('\P{Is_GCB:	 :=SM}');
Expect(1, 119149, '\p{Is_GCB=sm}', "");
Expect(0, 119149, '\p{^Is_GCB=sm}', "");
Expect(0, 119149, '\P{Is_GCB=sm}', "");
Expect(1, 119149, '\P{^Is_GCB=sm}', "");
Expect(0, 119150, '\p{Is_GCB=sm}', "");
Expect(1, 119150, '\p{^Is_GCB=sm}', "");
Expect(1, 119150, '\P{Is_GCB=sm}', "");
Expect(0, 119150, '\P{^Is_GCB=sm}', "");
Expect(1, 119149, '\p{Is_GCB=--SM}', "");
Expect(0, 119149, '\p{^Is_GCB=--SM}', "");
Expect(0, 119149, '\P{Is_GCB=--SM}', "");
Expect(1, 119149, '\P{^Is_GCB=--SM}', "");
Expect(0, 119150, '\p{Is_GCB=--SM}', "");
Expect(1, 119150, '\p{^Is_GCB=--SM}', "");
Expect(1, 119150, '\P{Is_GCB=--SM}', "");
Expect(0, 119150, '\P{^Is_GCB=--SM}', "");
Error('\p{Grapheme_Cluster_Break=_/a/T}');
Error('\P{Grapheme_Cluster_Break=_/a/T}');
Expect(1, 55291, '\p{Grapheme_Cluster_Break=t}', "");
Expect(0, 55291, '\p{^Grapheme_Cluster_Break=t}', "");
Expect(0, 55291, '\P{Grapheme_Cluster_Break=t}', "");
Expect(1, 55291, '\P{^Grapheme_Cluster_Break=t}', "");
Expect(0, 55292, '\p{Grapheme_Cluster_Break=t}', "");
Expect(1, 55292, '\p{^Grapheme_Cluster_Break=t}', "");
Expect(1, 55292, '\P{Grapheme_Cluster_Break=t}', "");
Expect(0, 55292, '\P{^Grapheme_Cluster_Break=t}', "");
Expect(1, 55291, '\p{Grapheme_Cluster_Break:	-_T}', "");
Expect(0, 55291, '\p{^Grapheme_Cluster_Break:	-_T}', "");
Expect(0, 55291, '\P{Grapheme_Cluster_Break:	-_T}', "");
Expect(1, 55291, '\P{^Grapheme_Cluster_Break:	-_T}', "");
Expect(0, 55292, '\p{Grapheme_Cluster_Break:	-_T}', "");
Expect(1, 55292, '\p{^Grapheme_Cluster_Break:	-_T}', "");
Expect(1, 55292, '\P{Grapheme_Cluster_Break:	-_T}', "");
Expect(0, 55292, '\P{^Grapheme_Cluster_Break:	-_T}', "");
Error('\p{GCB= -T:=}');
Error('\P{GCB= -T:=}');
Expect(1, 55291, '\p{GCB=t}', "");
Expect(0, 55291, '\p{^GCB=t}', "");
Expect(0, 55291, '\P{GCB=t}', "");
Expect(1, 55291, '\P{^GCB=t}', "");
Expect(0, 55292, '\p{GCB=t}', "");
Expect(1, 55292, '\p{^GCB=t}', "");
Expect(1, 55292, '\P{GCB=t}', "");
Expect(0, 55292, '\P{^GCB=t}', "");
Expect(1, 55291, '\p{GCB=__T}', "");
Expect(0, 55291, '\p{^GCB=__T}', "");
Expect(0, 55291, '\P{GCB=__T}', "");
Expect(1, 55291, '\P{^GCB=__T}', "");
Expect(0, 55292, '\p{GCB=__T}', "");
Expect(1, 55292, '\p{^GCB=__T}', "");
Expect(1, 55292, '\P{GCB=__T}', "");
Expect(0, 55292, '\P{^GCB=__T}', "");
Error('\p{Is_Grapheme_Cluster_Break= 	T:=}');
Error('\P{Is_Grapheme_Cluster_Break= 	T:=}');
Expect(1, 55291, '\p{Is_Grapheme_Cluster_Break=t}', "");
Expect(0, 55291, '\p{^Is_Grapheme_Cluster_Break=t}', "");
Expect(0, 55291, '\P{Is_Grapheme_Cluster_Break=t}', "");
Expect(1, 55291, '\P{^Is_Grapheme_Cluster_Break=t}', "");
Expect(0, 55292, '\p{Is_Grapheme_Cluster_Break=t}', "");
Expect(1, 55292, '\p{^Is_Grapheme_Cluster_Break=t}', "");
Expect(1, 55292, '\P{Is_Grapheme_Cluster_Break=t}', "");
Expect(0, 55292, '\P{^Is_Grapheme_Cluster_Break=t}', "");
Expect(1, 55291, '\p{Is_Grapheme_Cluster_Break=	T}', "");
Expect(0, 55291, '\p{^Is_Grapheme_Cluster_Break=	T}', "");
Expect(0, 55291, '\P{Is_Grapheme_Cluster_Break=	T}', "");
Expect(1, 55291, '\P{^Is_Grapheme_Cluster_Break=	T}', "");
Expect(0, 55292, '\p{Is_Grapheme_Cluster_Break=	T}', "");
Expect(1, 55292, '\p{^Is_Grapheme_Cluster_Break=	T}', "");
Expect(1, 55292, '\P{Is_Grapheme_Cluster_Break=	T}', "");
Expect(0, 55292, '\P{^Is_Grapheme_Cluster_Break=	T}', "");
Error('\p{Is_GCB:-/a/T}');
Error('\P{Is_GCB:-/a/T}');
Expect(1, 55291, '\p{Is_GCB=t}', "");
Expect(0, 55291, '\p{^Is_GCB=t}', "");
Expect(0, 55291, '\P{Is_GCB=t}', "");
Expect(1, 55291, '\P{^Is_GCB=t}', "");
Expect(0, 55292, '\p{Is_GCB=t}', "");
Expect(1, 55292, '\p{^Is_GCB=t}', "");
Expect(1, 55292, '\P{Is_GCB=t}', "");
Expect(0, 55292, '\P{^Is_GCB=t}', "");
Expect(1, 55291, '\p{Is_GCB= T}', "");
Expect(0, 55291, '\p{^Is_GCB= T}', "");
Expect(0, 55291, '\P{Is_GCB= T}', "");
Expect(1, 55291, '\P{^Is_GCB= T}', "");
Expect(0, 55292, '\p{Is_GCB= T}', "");
Expect(1, 55292, '\p{^Is_GCB= T}', "");
Expect(1, 55292, '\P{Is_GCB= T}', "");
Expect(0, 55292, '\P{^Is_GCB= T}', "");
Error('\p{Grapheme_Cluster_Break= _LVT/a/}');
Error('\P{Grapheme_Cluster_Break= _LVT/a/}');
Expect(1, 55203, '\p{Grapheme_Cluster_Break=lvt}', "");
Expect(0, 55203, '\p{^Grapheme_Cluster_Break=lvt}', "");
Expect(0, 55203, '\P{Grapheme_Cluster_Break=lvt}', "");
Expect(1, 55203, '\P{^Grapheme_Cluster_Break=lvt}', "");
Expect(0, 55204, '\p{Grapheme_Cluster_Break=lvt}', "");
Expect(1, 55204, '\p{^Grapheme_Cluster_Break=lvt}', "");
Expect(1, 55204, '\P{Grapheme_Cluster_Break=lvt}', "");
Expect(0, 55204, '\P{^Grapheme_Cluster_Break=lvt}', "");
Expect(1, 55203, '\p{Grapheme_Cluster_Break: _ LVT}', "");
Expect(0, 55203, '\p{^Grapheme_Cluster_Break: _ LVT}', "");
Expect(0, 55203, '\P{Grapheme_Cluster_Break: _ LVT}', "");
Expect(1, 55203, '\P{^Grapheme_Cluster_Break: _ LVT}', "");
Expect(0, 55204, '\p{Grapheme_Cluster_Break: _ LVT}', "");
Expect(1, 55204, '\p{^Grapheme_Cluster_Break: _ LVT}', "");
Expect(1, 55204, '\P{Grapheme_Cluster_Break: _ LVT}', "");
Expect(0, 55204, '\P{^Grapheme_Cluster_Break: _ LVT}', "");
Error('\p{GCB=:=LVT}');
Error('\P{GCB=:=LVT}');
Expect(1, 55203, '\p{GCB=lvt}', "");
Expect(0, 55203, '\p{^GCB=lvt}', "");
Expect(0, 55203, '\P{GCB=lvt}', "");
Expect(1, 55203, '\P{^GCB=lvt}', "");
Expect(0, 55204, '\p{GCB=lvt}', "");
Expect(1, 55204, '\p{^GCB=lvt}', "");
Expect(1, 55204, '\P{GCB=lvt}', "");
Expect(0, 55204, '\P{^GCB=lvt}', "");
Expect(1, 55203, '\p{GCB=- LVT}', "");
Expect(0, 55203, '\p{^GCB=- LVT}', "");
Expect(0, 55203, '\P{GCB=- LVT}', "");
Expect(1, 55203, '\P{^GCB=- LVT}', "");
Expect(0, 55204, '\p{GCB=- LVT}', "");
Expect(1, 55204, '\p{^GCB=- LVT}', "");
Expect(1, 55204, '\P{GCB=- LVT}', "");
Expect(0, 55204, '\P{^GCB=- LVT}', "");
Error('\p{Is_Grapheme_Cluster_Break=:=_	LVT}');
Error('\P{Is_Grapheme_Cluster_Break=:=_	LVT}');
Expect(1, 55203, '\p{Is_Grapheme_Cluster_Break=lvt}', "");
Expect(0, 55203, '\p{^Is_Grapheme_Cluster_Break=lvt}', "");
Expect(0, 55203, '\P{Is_Grapheme_Cluster_Break=lvt}', "");
Expect(1, 55203, '\P{^Is_Grapheme_Cluster_Break=lvt}', "");
Expect(0, 55204, '\p{Is_Grapheme_Cluster_Break=lvt}', "");
Expect(1, 55204, '\p{^Is_Grapheme_Cluster_Break=lvt}', "");
Expect(1, 55204, '\P{Is_Grapheme_Cluster_Break=lvt}', "");
Expect(0, 55204, '\P{^Is_Grapheme_Cluster_Break=lvt}', "");
Expect(1, 55203, '\p{Is_Grapheme_Cluster_Break=_lvt}', "");
Expect(0, 55203, '\p{^Is_Grapheme_Cluster_Break=_lvt}', "");
Expect(0, 55203, '\P{Is_Grapheme_Cluster_Break=_lvt}', "");
Expect(1, 55203, '\P{^Is_Grapheme_Cluster_Break=_lvt}', "");
Expect(0, 55204, '\p{Is_Grapheme_Cluster_Break=_lvt}', "");
Expect(1, 55204, '\p{^Is_Grapheme_Cluster_Break=_lvt}', "");
Expect(1, 55204, '\P{Is_Grapheme_Cluster_Break=_lvt}', "");
Expect(0, 55204, '\P{^Is_Grapheme_Cluster_Break=_lvt}', "");
Error('\p{Is_GCB=_	LVT:=}');
Error('\P{Is_GCB=_	LVT:=}');
Expect(1, 55203, '\p{Is_GCB=lvt}', "");
Expect(0, 55203, '\p{^Is_GCB=lvt}', "");
Expect(0, 55203, '\P{Is_GCB=lvt}', "");
Expect(1, 55203, '\P{^Is_GCB=lvt}', "");
Expect(0, 55204, '\p{Is_GCB=lvt}', "");
Expect(1, 55204, '\p{^Is_GCB=lvt}', "");
Expect(1, 55204, '\P{Is_GCB=lvt}', "");
Expect(0, 55204, '\P{^Is_GCB=lvt}', "");
Expect(1, 55203, '\p{Is_GCB=		LVT}', "");
Expect(0, 55203, '\p{^Is_GCB=		LVT}', "");
Expect(0, 55203, '\P{Is_GCB=		LVT}', "");
Expect(1, 55203, '\P{^Is_GCB=		LVT}', "");
Expect(0, 55204, '\p{Is_GCB=		LVT}', "");
Expect(1, 55204, '\p{^Is_GCB=		LVT}', "");
Expect(1, 55204, '\P{Is_GCB=		LVT}', "");
Expect(0, 55204, '\P{^Is_GCB=		LVT}', "");
Error('\p{Grapheme_Cluster_Break=/a/	 control}');
Error('\P{Grapheme_Cluster_Break=/a/	 control}');
Expect(1, 921599, '\p{Grapheme_Cluster_Break=control}', "");
Expect(0, 921599, '\p{^Grapheme_Cluster_Break=control}', "");
Expect(0, 921599, '\P{Grapheme_Cluster_Break=control}', "");
Expect(1, 921599, '\P{^Grapheme_Cluster_Break=control}', "");
Expect(0, 921600, '\p{Grapheme_Cluster_Break=control}', "");
Expect(1, 921600, '\p{^Grapheme_Cluster_Break=control}', "");
Expect(1, 921600, '\P{Grapheme_Cluster_Break=control}', "");
Expect(0, 921600, '\P{^Grapheme_Cluster_Break=control}', "");
Expect(1, 921599, '\p{Grapheme_Cluster_Break=_control}', "");
Expect(0, 921599, '\p{^Grapheme_Cluster_Break=_control}', "");
Expect(0, 921599, '\P{Grapheme_Cluster_Break=_control}', "");
Expect(1, 921599, '\P{^Grapheme_Cluster_Break=_control}', "");
Expect(0, 921600, '\p{Grapheme_Cluster_Break=_control}', "");
Expect(1, 921600, '\p{^Grapheme_Cluster_Break=_control}', "");
Expect(1, 921600, '\P{Grapheme_Cluster_Break=_control}', "");
Expect(0, 921600, '\P{^Grapheme_Cluster_Break=_control}', "");
Error('\p{GCB=-/a/CN}');
Error('\P{GCB=-/a/CN}');
Expect(1, 921599, '\p{GCB:cn}', "");
Expect(0, 921599, '\p{^GCB:cn}', "");
Expect(0, 921599, '\P{GCB:cn}', "");
Expect(1, 921599, '\P{^GCB:cn}', "");
Expect(0, 921600, '\p{GCB:cn}', "");
Expect(1, 921600, '\p{^GCB:cn}', "");
Expect(1, 921600, '\P{GCB:cn}', "");
Expect(0, 921600, '\P{^GCB:cn}', "");
Expect(1, 921599, '\p{GCB:  _CN}', "");
Expect(0, 921599, '\p{^GCB:  _CN}', "");
Expect(0, 921599, '\P{GCB:  _CN}', "");
Expect(1, 921599, '\P{^GCB:  _CN}', "");
Expect(0, 921600, '\p{GCB:  _CN}', "");
Expect(1, 921600, '\p{^GCB:  _CN}', "");
Expect(1, 921600, '\P{GCB:  _CN}', "");
Expect(0, 921600, '\P{^GCB:  _CN}', "");
Error('\p{Is_Grapheme_Cluster_Break=	:=Control}');
Error('\P{Is_Grapheme_Cluster_Break=	:=Control}');
Expect(1, 921599, '\p{Is_Grapheme_Cluster_Break=control}', "");
Expect(0, 921599, '\p{^Is_Grapheme_Cluster_Break=control}', "");
Expect(0, 921599, '\P{Is_Grapheme_Cluster_Break=control}', "");
Expect(1, 921599, '\P{^Is_Grapheme_Cluster_Break=control}', "");
Expect(0, 921600, '\p{Is_Grapheme_Cluster_Break=control}', "");
Expect(1, 921600, '\p{^Is_Grapheme_Cluster_Break=control}', "");
Expect(1, 921600, '\P{Is_Grapheme_Cluster_Break=control}', "");
Expect(0, 921600, '\P{^Is_Grapheme_Cluster_Break=control}', "");
Expect(1, 921599, '\p{Is_Grapheme_Cluster_Break= Control}', "");
Expect(0, 921599, '\p{^Is_Grapheme_Cluster_Break= Control}', "");
Expect(0, 921599, '\P{Is_Grapheme_Cluster_Break= Control}', "");
Expect(1, 921599, '\P{^Is_Grapheme_Cluster_Break= Control}', "");
Expect(0, 921600, '\p{Is_Grapheme_Cluster_Break= Control}', "");
Expect(1, 921600, '\p{^Is_Grapheme_Cluster_Break= Control}', "");
Expect(1, 921600, '\P{Is_Grapheme_Cluster_Break= Control}', "");
Expect(0, 921600, '\P{^Is_Grapheme_Cluster_Break= Control}', "");
Error('\p{Is_GCB=_CN/a/}');
Error('\P{Is_GCB=_CN/a/}');
Expect(1, 921599, '\p{Is_GCB=cn}', "");
Expect(0, 921599, '\p{^Is_GCB=cn}', "");
Expect(0, 921599, '\P{Is_GCB=cn}', "");
Expect(1, 921599, '\P{^Is_GCB=cn}', "");
Expect(0, 921600, '\p{Is_GCB=cn}', "");
Expect(1, 921600, '\p{^Is_GCB=cn}', "");
Expect(1, 921600, '\P{Is_GCB=cn}', "");
Expect(0, 921600, '\P{^Is_GCB=cn}', "");
Expect(1, 921599, '\p{Is_GCB=	CN}', "");
Expect(0, 921599, '\p{^Is_GCB=	CN}', "");
Expect(0, 921599, '\P{Is_GCB=	CN}', "");
Expect(1, 921599, '\P{^Is_GCB=	CN}', "");
Expect(0, 921600, '\p{Is_GCB=	CN}', "");
Expect(1, 921600, '\p{^Is_GCB=	CN}', "");
Expect(1, 921600, '\P{Is_GCB=	CN}', "");
Expect(0, 921600, '\P{^Is_GCB=	CN}', "");
Error('\p{Grapheme_Cluster_Break=-:=Prepend}');
Error('\P{Grapheme_Cluster_Break=-:=Prepend}');
Expect(0, 1, '\p{Grapheme_Cluster_Break=prepend}', "");
Expect(1, 1, '\p{^Grapheme_Cluster_Break=prepend}', "");
Expect(1, 1, '\P{Grapheme_Cluster_Break=prepend}', "");
Expect(0, 1, '\P{^Grapheme_Cluster_Break=prepend}', "");
Expect(0, 1, '\p{Grapheme_Cluster_Break=-PREPEND}', "");
Expect(1, 1, '\p{^Grapheme_Cluster_Break=-PREPEND}', "");
Expect(1, 1, '\P{Grapheme_Cluster_Break=-PREPEND}', "");
Expect(0, 1, '\P{^Grapheme_Cluster_Break=-PREPEND}', "");
Error('\p{GCB=_-PP:=}');
Error('\P{GCB=_-PP:=}');
Expect(0, 1, '\p{GCB=pp}', "");
Expect(1, 1, '\p{^GCB=pp}', "");
Expect(1, 1, '\P{GCB=pp}', "");
Expect(0, 1, '\P{^GCB=pp}', "");
Expect(0, 1, '\p{GCB=	 pp}', "");
Expect(1, 1, '\p{^GCB=	 pp}', "");
Expect(1, 1, '\P{GCB=	 pp}', "");
Expect(0, 1, '\P{^GCB=	 pp}', "");
Error('\p{Is_Grapheme_Cluster_Break=/a/_	Prepend}');
Error('\P{Is_Grapheme_Cluster_Break=/a/_	Prepend}');
Expect(0, 1, '\p{Is_Grapheme_Cluster_Break=prepend}', "");
Expect(1, 1, '\p{^Is_Grapheme_Cluster_Break=prepend}', "");
Expect(1, 1, '\P{Is_Grapheme_Cluster_Break=prepend}', "");
Expect(0, 1, '\P{^Is_Grapheme_Cluster_Break=prepend}', "");
Expect(0, 1, '\p{Is_Grapheme_Cluster_Break=		Prepend}', "");
Expect(1, 1, '\p{^Is_Grapheme_Cluster_Break=		Prepend}', "");
Expect(1, 1, '\P{Is_Grapheme_Cluster_Break=		Prepend}', "");
Expect(0, 1, '\P{^Is_Grapheme_Cluster_Break=		Prepend}', "");
Error('\p{Is_GCB=_:=PP}');
Error('\P{Is_GCB=_:=PP}');
Expect(0, 1, '\p{Is_GCB:	pp}', "");
Expect(1, 1, '\p{^Is_GCB:	pp}', "");
Expect(1, 1, '\P{Is_GCB:	pp}', "");
Expect(0, 1, '\P{^Is_GCB:	pp}', "");
Expect(0, 1, '\p{Is_GCB=_ PP}', "");
Expect(1, 1, '\p{^Is_GCB=_ PP}', "");
Expect(1, 1, '\P{Is_GCB=_ PP}', "");
Expect(0, 1, '\P{^Is_GCB=_ PP}', "");
Error('\p{Grapheme_Cluster_Break=-:=LV}');
Error('\P{Grapheme_Cluster_Break=-:=LV}');
Expect(1, 55176, '\p{Grapheme_Cluster_Break=lv}', "");
Expect(0, 55176, '\p{^Grapheme_Cluster_Break=lv}', "");
Expect(0, 55176, '\P{Grapheme_Cluster_Break=lv}', "");
Expect(1, 55176, '\P{^Grapheme_Cluster_Break=lv}', "");
Expect(0, 55177, '\p{Grapheme_Cluster_Break=lv}', "");
Expect(1, 55177, '\p{^Grapheme_Cluster_Break=lv}', "");
Expect(1, 55177, '\P{Grapheme_Cluster_Break=lv}', "");
Expect(0, 55177, '\P{^Grapheme_Cluster_Break=lv}', "");
Expect(1, 55176, '\p{Grapheme_Cluster_Break=LV}', "");
Expect(0, 55176, '\p{^Grapheme_Cluster_Break=LV}', "");
Expect(0, 55176, '\P{Grapheme_Cluster_Break=LV}', "");
Expect(1, 55176, '\P{^Grapheme_Cluster_Break=LV}', "");
Expect(0, 55177, '\p{Grapheme_Cluster_Break=LV}', "");
Expect(1, 55177, '\p{^Grapheme_Cluster_Break=LV}', "");
Expect(1, 55177, '\P{Grapheme_Cluster_Break=LV}', "");
Expect(0, 55177, '\P{^Grapheme_Cluster_Break=LV}', "");
Error('\p{GCB=	/a/LV}');
Error('\P{GCB=	/a/LV}');
Expect(1, 55176, '\p{GCB=lv}', "");
Expect(0, 55176, '\p{^GCB=lv}', "");
Expect(0, 55176, '\P{GCB=lv}', "");
Expect(1, 55176, '\P{^GCB=lv}', "");
Expect(0, 55177, '\p{GCB=lv}', "");
Expect(1, 55177, '\p{^GCB=lv}', "");
Expect(1, 55177, '\P{GCB=lv}', "");
Expect(0, 55177, '\P{^GCB=lv}', "");
Expect(1, 55176, '\p{GCB=_LV}', "");
Expect(0, 55176, '\p{^GCB=_LV}', "");
Expect(0, 55176, '\P{GCB=_LV}', "");
Expect(1, 55176, '\P{^GCB=_LV}', "");
Expect(0, 55177, '\p{GCB=_LV}', "");
Expect(1, 55177, '\p{^GCB=_LV}', "");
Expect(1, 55177, '\P{GCB=_LV}', "");
Expect(0, 55177, '\P{^GCB=_LV}', "");
Error('\p{Is_Grapheme_Cluster_Break=:=_LV}');
Error('\P{Is_Grapheme_Cluster_Break=:=_LV}');
Expect(1, 55176, '\p{Is_Grapheme_Cluster_Break: lv}', "");
Expect(0, 55176, '\p{^Is_Grapheme_Cluster_Break: lv}', "");
Expect(0, 55176, '\P{Is_Grapheme_Cluster_Break: lv}', "");
Expect(1, 55176, '\P{^Is_Grapheme_Cluster_Break: lv}', "");
Expect(0, 55177, '\p{Is_Grapheme_Cluster_Break: lv}', "");
Expect(1, 55177, '\p{^Is_Grapheme_Cluster_Break: lv}', "");
Expect(1, 55177, '\P{Is_Grapheme_Cluster_Break: lv}', "");
Expect(0, 55177, '\P{^Is_Grapheme_Cluster_Break: lv}', "");
Expect(1, 55176, '\p{Is_Grapheme_Cluster_Break=	LV}', "");
Expect(0, 55176, '\p{^Is_Grapheme_Cluster_Break=	LV}', "");
Expect(0, 55176, '\P{Is_Grapheme_Cluster_Break=	LV}', "");
Expect(1, 55176, '\P{^Is_Grapheme_Cluster_Break=	LV}', "");
Expect(0, 55177, '\p{Is_Grapheme_Cluster_Break=	LV}', "");
Expect(1, 55177, '\p{^Is_Grapheme_Cluster_Break=	LV}', "");
Expect(1, 55177, '\P{Is_Grapheme_Cluster_Break=	LV}', "");
Expect(0, 55177, '\P{^Is_Grapheme_Cluster_Break=	LV}', "");
Error('\p{Is_GCB= LV:=}');
Error('\P{Is_GCB= LV:=}');
Expect(1, 55176, '\p{Is_GCB=lv}', "");
Expect(0, 55176, '\p{^Is_GCB=lv}', "");
Expect(0, 55176, '\P{Is_GCB=lv}', "");
Expect(1, 55176, '\P{^Is_GCB=lv}', "");
Expect(0, 55177, '\p{Is_GCB=lv}', "");
Expect(1, 55177, '\p{^Is_GCB=lv}', "");
Expect(1, 55177, '\P{Is_GCB=lv}', "");
Expect(0, 55177, '\P{^Is_GCB=lv}', "");
Expect(1, 55176, '\p{Is_GCB=_LV}', "");
Expect(0, 55176, '\p{^Is_GCB=_LV}', "");
Expect(0, 55176, '\P{Is_GCB=_LV}', "");
Expect(1, 55176, '\P{^Is_GCB=_LV}', "");
Expect(0, 55177, '\p{Is_GCB=_LV}', "");
Expect(1, 55177, '\p{^Is_GCB=_LV}', "");
Expect(1, 55177, '\P{Is_GCB=_LV}', "");
Expect(0, 55177, '\P{^Is_GCB=_LV}', "");
Error('\p{Grapheme_Cluster_Break=-:=CR}');
Error('\P{Grapheme_Cluster_Break=-:=CR}');
Expect(1, 13, '\p{Grapheme_Cluster_Break=cr}', "");
Expect(0, 13, '\p{^Grapheme_Cluster_Break=cr}', "");
Expect(0, 13, '\P{Grapheme_Cluster_Break=cr}', "");
Expect(1, 13, '\P{^Grapheme_Cluster_Break=cr}', "");
Expect(0, 14, '\p{Grapheme_Cluster_Break=cr}', "");
Expect(1, 14, '\p{^Grapheme_Cluster_Break=cr}', "");
Expect(1, 14, '\P{Grapheme_Cluster_Break=cr}', "");
Expect(0, 14, '\P{^Grapheme_Cluster_Break=cr}', "");
Expect(1, 13, '\p{Grapheme_Cluster_Break=- CR}', "");
Expect(0, 13, '\p{^Grapheme_Cluster_Break=- CR}', "");
Expect(0, 13, '\P{Grapheme_Cluster_Break=- CR}', "");
Expect(1, 13, '\P{^Grapheme_Cluster_Break=- CR}', "");
Expect(0, 14, '\p{Grapheme_Cluster_Break=- CR}', "");
Expect(1, 14, '\p{^Grapheme_Cluster_Break=- CR}', "");
Expect(1, 14, '\P{Grapheme_Cluster_Break=- CR}', "");
Expect(0, 14, '\P{^Grapheme_Cluster_Break=- CR}', "");
Error('\p{GCB=	cr:=}');
Error('\P{GCB=	cr:=}');
Expect(1, 13, '\p{GCB=cr}', "");
Expect(0, 13, '\p{^GCB=cr}', "");
Expect(0, 13, '\P{GCB=cr}', "");
Expect(1, 13, '\P{^GCB=cr}', "");
Expect(0, 14, '\p{GCB=cr}', "");
Expect(1, 14, '\p{^GCB=cr}', "");
Expect(1, 14, '\P{GCB=cr}', "");
Expect(0, 14, '\P{^GCB=cr}', "");
Expect(1, 13, '\p{GCB=CR}', "");
Expect(0, 13, '\p{^GCB=CR}', "");
Expect(0, 13, '\P{GCB=CR}', "");
Expect(1, 13, '\P{^GCB=CR}', "");
Expect(0, 14, '\p{GCB=CR}', "");
Expect(1, 14, '\p{^GCB=CR}', "");
Expect(1, 14, '\P{GCB=CR}', "");
Expect(0, 14, '\P{^GCB=CR}', "");
Error('\p{Is_Grapheme_Cluster_Break=-/a/CR}');
Error('\P{Is_Grapheme_Cluster_Break=-/a/CR}');
Expect(1, 13, '\p{Is_Grapheme_Cluster_Break:	cr}', "");
Expect(0, 13, '\p{^Is_Grapheme_Cluster_Break:	cr}', "");
Expect(0, 13, '\P{Is_Grapheme_Cluster_Break:	cr}', "");
Expect(1, 13, '\P{^Is_Grapheme_Cluster_Break:	cr}', "");
Expect(0, 14, '\p{Is_Grapheme_Cluster_Break:	cr}', "");
Expect(1, 14, '\p{^Is_Grapheme_Cluster_Break:	cr}', "");
Expect(1, 14, '\P{Is_Grapheme_Cluster_Break:	cr}', "");
Expect(0, 14, '\P{^Is_Grapheme_Cluster_Break:	cr}', "");
Expect(1, 13, '\p{Is_Grapheme_Cluster_Break= 	cr}', "");
Expect(0, 13, '\p{^Is_Grapheme_Cluster_Break= 	cr}', "");
Expect(0, 13, '\P{Is_Grapheme_Cluster_Break= 	cr}', "");
Expect(1, 13, '\P{^Is_Grapheme_Cluster_Break= 	cr}', "");
Expect(0, 14, '\p{Is_Grapheme_Cluster_Break= 	cr}', "");
Expect(1, 14, '\p{^Is_Grapheme_Cluster_Break= 	cr}', "");
Expect(1, 14, '\P{Is_Grapheme_Cluster_Break= 	cr}', "");
Expect(0, 14, '\P{^Is_Grapheme_Cluster_Break= 	cr}', "");
Error('\p{Is_GCB= 	CR:=}');
Error('\P{Is_GCB= 	CR:=}');
Expect(1, 13, '\p{Is_GCB=cr}', "");
Expect(0, 13, '\p{^Is_GCB=cr}', "");
Expect(0, 13, '\P{Is_GCB=cr}', "");
Expect(1, 13, '\P{^Is_GCB=cr}', "");
Expect(0, 14, '\p{Is_GCB=cr}', "");
Expect(1, 14, '\p{^Is_GCB=cr}', "");
Expect(1, 14, '\P{Is_GCB=cr}', "");
Expect(0, 14, '\P{^Is_GCB=cr}', "");
Expect(1, 13, '\p{Is_GCB=_cr}', "");
Expect(0, 13, '\p{^Is_GCB=_cr}', "");
Expect(0, 13, '\P{Is_GCB=_cr}', "");
Expect(1, 13, '\P{^Is_GCB=_cr}', "");
Expect(0, 14, '\p{Is_GCB=_cr}', "");
Expect(1, 14, '\p{^Is_GCB=_cr}', "");
Expect(1, 14, '\P{Is_GCB=_cr}', "");
Expect(0, 14, '\P{^Is_GCB=_cr}', "");
Error('\p{Grapheme_Cluster_Break=  Other:=}');
Error('\P{Grapheme_Cluster_Break=  Other:=}');
Expect(1, 921600, '\p{Grapheme_Cluster_Break:	other}', "");
Expect(0, 921600, '\p{^Grapheme_Cluster_Break:	other}', "");
Expect(0, 921600, '\P{Grapheme_Cluster_Break:	other}', "");
Expect(1, 921600, '\P{^Grapheme_Cluster_Break:	other}', "");
Expect(0, 921599, '\p{Grapheme_Cluster_Break:	other}', "");
Expect(1, 921599, '\p{^Grapheme_Cluster_Break:	other}', "");
Expect(1, 921599, '\P{Grapheme_Cluster_Break:	other}', "");
Expect(0, 921599, '\P{^Grapheme_Cluster_Break:	other}', "");
Expect(1, 921600, '\p{Grapheme_Cluster_Break= Other}', "");
Expect(0, 921600, '\p{^Grapheme_Cluster_Break= Other}', "");
Expect(0, 921600, '\P{Grapheme_Cluster_Break= Other}', "");
Expect(1, 921600, '\P{^Grapheme_Cluster_Break= Other}', "");
Expect(0, 921599, '\p{Grapheme_Cluster_Break= Other}', "");
Expect(1, 921599, '\p{^Grapheme_Cluster_Break= Other}', "");
Expect(1, 921599, '\P{Grapheme_Cluster_Break= Other}', "");
Expect(0, 921599, '\P{^Grapheme_Cluster_Break= Other}', "");
Error('\p{GCB= :=XX}');
Error('\P{GCB= :=XX}');
Expect(1, 921600, '\p{GCB=xx}', "");
Expect(0, 921600, '\p{^GCB=xx}', "");
Expect(0, 921600, '\P{GCB=xx}', "");
Expect(1, 921600, '\P{^GCB=xx}', "");
Expect(0, 921599, '\p{GCB=xx}', "");
Expect(1, 921599, '\p{^GCB=xx}', "");
Expect(1, 921599, '\P{GCB=xx}', "");
Expect(0, 921599, '\P{^GCB=xx}', "");
Expect(1, 921600, '\p{GCB= xx}', "");
Expect(0, 921600, '\p{^GCB= xx}', "");
Expect(0, 921600, '\P{GCB= xx}', "");
Expect(1, 921600, '\P{^GCB= xx}', "");
Expect(0, 921599, '\p{GCB= xx}', "");
Expect(1, 921599, '\p{^GCB= xx}', "");
Expect(1, 921599, '\P{GCB= xx}', "");
Expect(0, 921599, '\P{^GCB= xx}', "");
Error('\p{Is_Grapheme_Cluster_Break=_Other:=}');
Error('\P{Is_Grapheme_Cluster_Break=_Other:=}');
Expect(1, 921600, '\p{Is_Grapheme_Cluster_Break=other}', "");
Expect(0, 921600, '\p{^Is_Grapheme_Cluster_Break=other}', "");
Expect(0, 921600, '\P{Is_Grapheme_Cluster_Break=other}', "");
Expect(1, 921600, '\P{^Is_Grapheme_Cluster_Break=other}', "");
Expect(0, 921599, '\p{Is_Grapheme_Cluster_Break=other}', "");
Expect(1, 921599, '\p{^Is_Grapheme_Cluster_Break=other}', "");
Expect(1, 921599, '\P{Is_Grapheme_Cluster_Break=other}', "");
Expect(0, 921599, '\P{^Is_Grapheme_Cluster_Break=other}', "");
Expect(1, 921600, '\p{Is_Grapheme_Cluster_Break=-other}', "");
Expect(0, 921600, '\p{^Is_Grapheme_Cluster_Break=-other}', "");
Expect(0, 921600, '\P{Is_Grapheme_Cluster_Break=-other}', "");
Expect(1, 921600, '\P{^Is_Grapheme_Cluster_Break=-other}', "");
Expect(0, 921599, '\p{Is_Grapheme_Cluster_Break=-other}', "");
Expect(1, 921599, '\p{^Is_Grapheme_Cluster_Break=-other}', "");
Expect(1, 921599, '\P{Is_Grapheme_Cluster_Break=-other}', "");
Expect(0, 921599, '\P{^Is_Grapheme_Cluster_Break=-other}', "");
Error('\p{Is_GCB=	:=XX}');
Error('\P{Is_GCB=	:=XX}');
Expect(1, 921600, '\p{Is_GCB=xx}', "");
Expect(0, 921600, '\p{^Is_GCB=xx}', "");
Expect(0, 921600, '\P{Is_GCB=xx}', "");
Expect(1, 921600, '\P{^Is_GCB=xx}', "");
Expect(0, 921599, '\p{Is_GCB=xx}', "");
Expect(1, 921599, '\p{^Is_GCB=xx}', "");
Expect(1, 921599, '\P{Is_GCB=xx}', "");
Expect(0, 921599, '\P{^Is_GCB=xx}', "");
Expect(1, 921600, '\p{Is_GCB= XX}', "");
Expect(0, 921600, '\p{^Is_GCB= XX}', "");
Expect(0, 921600, '\P{Is_GCB= XX}', "");
Expect(1, 921600, '\P{^Is_GCB= XX}', "");
Expect(0, 921599, '\p{Is_GCB= XX}', "");
Expect(1, 921599, '\p{^Is_GCB= XX}', "");
Expect(1, 921599, '\P{Is_GCB= XX}', "");
Expect(0, 921599, '\P{^Is_GCB= XX}', "");
Error('\p{Grapheme_Cluster_Break=/a/EXTEND}');
Error('\P{Grapheme_Cluster_Break=/a/EXTEND}');
Expect(1, 917999, '\p{Grapheme_Cluster_Break=extend}', "");
Expect(0, 917999, '\p{^Grapheme_Cluster_Break=extend}', "");
Expect(0, 917999, '\P{Grapheme_Cluster_Break=extend}', "");
Expect(1, 917999, '\P{^Grapheme_Cluster_Break=extend}', "");
Expect(0, 918000, '\p{Grapheme_Cluster_Break=extend}', "");
Expect(1, 918000, '\p{^Grapheme_Cluster_Break=extend}', "");
Expect(1, 918000, '\P{Grapheme_Cluster_Break=extend}', "");
Expect(0, 918000, '\P{^Grapheme_Cluster_Break=extend}', "");
Expect(1, 917999, '\p{Grapheme_Cluster_Break= Extend}', "");
Expect(0, 917999, '\p{^Grapheme_Cluster_Break= Extend}', "");
Expect(0, 917999, '\P{Grapheme_Cluster_Break= Extend}', "");
Expect(1, 917999, '\P{^Grapheme_Cluster_Break= Extend}', "");
Expect(0, 918000, '\p{Grapheme_Cluster_Break= Extend}', "");
Expect(1, 918000, '\p{^Grapheme_Cluster_Break= Extend}', "");
Expect(1, 918000, '\P{Grapheme_Cluster_Break= Extend}', "");
Expect(0, 918000, '\P{^Grapheme_Cluster_Break= Extend}', "");
Error('\p{GCB=:=-EX}');
Error('\P{GCB=:=-EX}');
Expect(1, 917999, '\p{GCB=ex}', "");
Expect(0, 917999, '\p{^GCB=ex}', "");
Expect(0, 917999, '\P{GCB=ex}', "");
Expect(1, 917999, '\P{^GCB=ex}', "");
Expect(0, 918000, '\p{GCB=ex}', "");
Expect(1, 918000, '\p{^GCB=ex}', "");
Expect(1, 918000, '\P{GCB=ex}', "");
Expect(0, 918000, '\P{^GCB=ex}', "");
Expect(1, 917999, '\p{GCB=	_EX}', "");
Expect(0, 917999, '\p{^GCB=	_EX}', "");
Expect(0, 917999, '\P{GCB=	_EX}', "");
Expect(1, 917999, '\P{^GCB=	_EX}', "");
Expect(0, 918000, '\p{GCB=	_EX}', "");
Expect(1, 918000, '\p{^GCB=	_EX}', "");
Expect(1, 918000, '\P{GCB=	_EX}', "");
Expect(0, 918000, '\P{^GCB=	_EX}', "");
Error('\p{Is_Grapheme_Cluster_Break=_ extend/a/}');
Error('\P{Is_Grapheme_Cluster_Break=_ extend/a/}');
Expect(1, 917999, '\p{Is_Grapheme_Cluster_Break:   extend}', "");
Expect(0, 917999, '\p{^Is_Grapheme_Cluster_Break:   extend}', "");
Expect(0, 917999, '\P{Is_Grapheme_Cluster_Break:   extend}', "");
Expect(1, 917999, '\P{^Is_Grapheme_Cluster_Break:   extend}', "");
Expect(0, 918000, '\p{Is_Grapheme_Cluster_Break:   extend}', "");
Expect(1, 918000, '\p{^Is_Grapheme_Cluster_Break:   extend}', "");
Expect(1, 918000, '\P{Is_Grapheme_Cluster_Break:   extend}', "");
Expect(0, 918000, '\P{^Is_Grapheme_Cluster_Break:   extend}', "");
Expect(1, 917999, '\p{Is_Grapheme_Cluster_Break=	_extend}', "");
Expect(0, 917999, '\p{^Is_Grapheme_Cluster_Break=	_extend}', "");
Expect(0, 917999, '\P{Is_Grapheme_Cluster_Break=	_extend}', "");
Expect(1, 917999, '\P{^Is_Grapheme_Cluster_Break=	_extend}', "");
Expect(0, 918000, '\p{Is_Grapheme_Cluster_Break=	_extend}', "");
Expect(1, 918000, '\p{^Is_Grapheme_Cluster_Break=	_extend}', "");
Expect(1, 918000, '\P{Is_Grapheme_Cluster_Break=	_extend}', "");
Expect(0, 918000, '\P{^Is_Grapheme_Cluster_Break=	_extend}', "");
Error('\p{Is_GCB=_:=ex}');
Error('\P{Is_GCB=_:=ex}');
Expect(1, 917999, '\p{Is_GCB=ex}', "");
Expect(0, 917999, '\p{^Is_GCB=ex}', "");
Expect(0, 917999, '\P{Is_GCB=ex}', "");
Expect(1, 917999, '\P{^Is_GCB=ex}', "");
Expect(0, 918000, '\p{Is_GCB=ex}', "");
Expect(1, 918000, '\p{^Is_GCB=ex}', "");
Expect(1, 918000, '\P{Is_GCB=ex}', "");
Expect(0, 918000, '\P{^Is_GCB=ex}', "");
Expect(1, 917999, '\p{Is_GCB=- EX}', "");
Expect(0, 917999, '\p{^Is_GCB=- EX}', "");
Expect(0, 917999, '\P{Is_GCB=- EX}', "");
Expect(1, 917999, '\P{^Is_GCB=- EX}', "");
Expect(0, 918000, '\p{Is_GCB=- EX}', "");
Expect(1, 918000, '\p{^Is_GCB=- EX}', "");
Expect(1, 918000, '\P{Is_GCB=- EX}', "");
Expect(0, 918000, '\P{^Is_GCB=- EX}', "");
Error('\p{Grapheme_Cluster_Break= /a/v}');
Error('\P{Grapheme_Cluster_Break= /a/v}');
Expect(1, 55238, '\p{Grapheme_Cluster_Break=v}', "");
Expect(0, 55238, '\p{^Grapheme_Cluster_Break=v}', "");
Expect(0, 55238, '\P{Grapheme_Cluster_Break=v}', "");
Expect(1, 55238, '\P{^Grapheme_Cluster_Break=v}', "");
Expect(0, 55239, '\p{Grapheme_Cluster_Break=v}', "");
Expect(1, 55239, '\p{^Grapheme_Cluster_Break=v}', "");
Expect(1, 55239, '\P{Grapheme_Cluster_Break=v}', "");
Expect(0, 55239, '\P{^Grapheme_Cluster_Break=v}', "");
Expect(1, 55238, '\p{Grapheme_Cluster_Break=-v}', "");
Expect(0, 55238, '\p{^Grapheme_Cluster_Break=-v}', "");
Expect(0, 55238, '\P{Grapheme_Cluster_Break=-v}', "");
Expect(1, 55238, '\P{^Grapheme_Cluster_Break=-v}', "");
Expect(0, 55239, '\p{Grapheme_Cluster_Break=-v}', "");
Expect(1, 55239, '\p{^Grapheme_Cluster_Break=-v}', "");
Expect(1, 55239, '\P{Grapheme_Cluster_Break=-v}', "");
Expect(0, 55239, '\P{^Grapheme_Cluster_Break=-v}', "");
Error('\p{GCB=	/a/v}');
Error('\P{GCB=	/a/v}');
Expect(1, 55238, '\p{GCB=v}', "");
Expect(0, 55238, '\p{^GCB=v}', "");
Expect(0, 55238, '\P{GCB=v}', "");
Expect(1, 55238, '\P{^GCB=v}', "");
Expect(0, 55239, '\p{GCB=v}', "");
Expect(1, 55239, '\p{^GCB=v}', "");
Expect(1, 55239, '\P{GCB=v}', "");
Expect(0, 55239, '\P{^GCB=v}', "");
Expect(1, 55238, '\p{GCB=--V}', "");
Expect(0, 55238, '\p{^GCB=--V}', "");
Expect(0, 55238, '\P{GCB=--V}', "");
Expect(1, 55238, '\P{^GCB=--V}', "");
Expect(0, 55239, '\p{GCB=--V}', "");
Expect(1, 55239, '\p{^GCB=--V}', "");
Expect(1, 55239, '\P{GCB=--V}', "");
Expect(0, 55239, '\P{^GCB=--V}', "");
Error('\p{Is_Grapheme_Cluster_Break=--V/a/}');
Error('\P{Is_Grapheme_Cluster_Break=--V/a/}');
Expect(1, 55238, '\p{Is_Grapheme_Cluster_Break=v}', "");
Expect(0, 55238, '\p{^Is_Grapheme_Cluster_Break=v}', "");
Expect(0, 55238, '\P{Is_Grapheme_Cluster_Break=v}', "");
Expect(1, 55238, '\P{^Is_Grapheme_Cluster_Break=v}', "");
Expect(0, 55239, '\p{Is_Grapheme_Cluster_Break=v}', "");
Expect(1, 55239, '\p{^Is_Grapheme_Cluster_Break=v}', "");
Expect(1, 55239, '\P{Is_Grapheme_Cluster_Break=v}', "");
Expect(0, 55239, '\P{^Is_Grapheme_Cluster_Break=v}', "");
Expect(1, 55238, '\p{Is_Grapheme_Cluster_Break=-V}', "");
Expect(0, 55238, '\p{^Is_Grapheme_Cluster_Break=-V}', "");
Expect(0, 55238, '\P{Is_Grapheme_Cluster_Break=-V}', "");
Expect(1, 55238, '\P{^Is_Grapheme_Cluster_Break=-V}', "");
Expect(0, 55239, '\p{Is_Grapheme_Cluster_Break=-V}', "");
Expect(1, 55239, '\p{^Is_Grapheme_Cluster_Break=-V}', "");
Expect(1, 55239, '\P{Is_Grapheme_Cluster_Break=-V}', "");
Expect(0, 55239, '\P{^Is_Grapheme_Cluster_Break=-V}', "");
Error('\p{Is_GCB=:=__V}');
Error('\P{Is_GCB=:=__V}');
Expect(1, 55238, '\p{Is_GCB=v}', "");
Expect(0, 55238, '\p{^Is_GCB=v}', "");
Expect(0, 55238, '\P{Is_GCB=v}', "");
Expect(1, 55238, '\P{^Is_GCB=v}', "");
Expect(0, 55239, '\p{Is_GCB=v}', "");
Expect(1, 55239, '\p{^Is_GCB=v}', "");
Expect(1, 55239, '\P{Is_GCB=v}', "");
Expect(0, 55239, '\P{^Is_GCB=v}', "");
Expect(1, 55238, '\p{Is_GCB=- V}', "");
Expect(0, 55238, '\p{^Is_GCB=- V}', "");
Expect(0, 55238, '\P{Is_GCB=- V}', "");
Expect(1, 55238, '\P{^Is_GCB=- V}', "");
Expect(0, 55239, '\p{Is_GCB=- V}', "");
Expect(1, 55239, '\p{^Is_GCB=- V}', "");
Expect(1, 55239, '\P{Is_GCB=- V}', "");
Expect(0, 55239, '\P{^Is_GCB=- V}', "");
Error('\p{Grapheme_Cluster_Break=-:=LF}');
Error('\P{Grapheme_Cluster_Break=-:=LF}');
Expect(1, 10, '\p{Grapheme_Cluster_Break:   lf}', "");
Expect(0, 10, '\p{^Grapheme_Cluster_Break:   lf}', "");
Expect(0, 10, '\P{Grapheme_Cluster_Break:   lf}', "");
Expect(1, 10, '\P{^Grapheme_Cluster_Break:   lf}', "");
Expect(0, 11, '\p{Grapheme_Cluster_Break:   lf}', "");
Expect(1, 11, '\p{^Grapheme_Cluster_Break:   lf}', "");
Expect(1, 11, '\P{Grapheme_Cluster_Break:   lf}', "");
Expect(0, 11, '\P{^Grapheme_Cluster_Break:   lf}', "");
Expect(1, 10, '\p{Grapheme_Cluster_Break=-LF}', "");
Expect(0, 10, '\p{^Grapheme_Cluster_Break=-LF}', "");
Expect(0, 10, '\P{Grapheme_Cluster_Break=-LF}', "");
Expect(1, 10, '\P{^Grapheme_Cluster_Break=-LF}', "");
Expect(0, 11, '\p{Grapheme_Cluster_Break=-LF}', "");
Expect(1, 11, '\p{^Grapheme_Cluster_Break=-LF}', "");
Expect(1, 11, '\P{Grapheme_Cluster_Break=-LF}', "");
Expect(0, 11, '\P{^Grapheme_Cluster_Break=-LF}', "");
Error('\p{GCB=/a/LF}');
Error('\P{GCB=/a/LF}');
Expect(1, 10, '\p{GCB=lf}', "");
Expect(0, 10, '\p{^GCB=lf}', "");
Expect(0, 10, '\P{GCB=lf}', "");
Expect(1, 10, '\P{^GCB=lf}', "");
Expect(0, 11, '\p{GCB=lf}', "");
Expect(1, 11, '\p{^GCB=lf}', "");
Expect(1, 11, '\P{GCB=lf}', "");
Expect(0, 11, '\P{^GCB=lf}', "");
Expect(1, 10, '\p{GCB=	lf}', "");
Expect(0, 10, '\p{^GCB=	lf}', "");
Expect(0, 10, '\P{GCB=	lf}', "");
Expect(1, 10, '\P{^GCB=	lf}', "");
Expect(0, 11, '\p{GCB=	lf}', "");
Expect(1, 11, '\p{^GCB=	lf}', "");
Expect(1, 11, '\P{GCB=	lf}', "");
Expect(0, 11, '\P{^GCB=	lf}', "");
Error('\p{Is_Grapheme_Cluster_Break=--LF:=}');
Error('\P{Is_Grapheme_Cluster_Break=--LF:=}');
Expect(1, 10, '\p{Is_Grapheme_Cluster_Break=lf}', "");
Expect(0, 10, '\p{^Is_Grapheme_Cluster_Break=lf}', "");
Expect(0, 10, '\P{Is_Grapheme_Cluster_Break=lf}', "");
Expect(1, 10, '\P{^Is_Grapheme_Cluster_Break=lf}', "");
Expect(0, 11, '\p{Is_Grapheme_Cluster_Break=lf}', "");
Expect(1, 11, '\p{^Is_Grapheme_Cluster_Break=lf}', "");
Expect(1, 11, '\P{Is_Grapheme_Cluster_Break=lf}', "");
Expect(0, 11, '\P{^Is_Grapheme_Cluster_Break=lf}', "");
Expect(1, 10, '\p{Is_Grapheme_Cluster_Break=__LF}', "");
Expect(0, 10, '\p{^Is_Grapheme_Cluster_Break=__LF}', "");
Expect(0, 10, '\P{Is_Grapheme_Cluster_Break=__LF}', "");
Expect(1, 10, '\P{^Is_Grapheme_Cluster_Break=__LF}', "");
Expect(0, 11, '\p{Is_Grapheme_Cluster_Break=__LF}', "");
Expect(1, 11, '\p{^Is_Grapheme_Cluster_Break=__LF}', "");
Expect(1, 11, '\P{Is_Grapheme_Cluster_Break=__LF}', "");
Expect(0, 11, '\P{^Is_Grapheme_Cluster_Break=__LF}', "");
Error('\p{Is_GCB=	/a/lf}');
Error('\P{Is_GCB=	/a/lf}');
Expect(1, 10, '\p{Is_GCB=lf}', "");
Expect(0, 10, '\p{^Is_GCB=lf}', "");
Expect(0, 10, '\P{Is_GCB=lf}', "");
Expect(1, 10, '\P{^Is_GCB=lf}', "");
Expect(0, 11, '\p{Is_GCB=lf}', "");
Expect(1, 11, '\p{^Is_GCB=lf}', "");
Expect(1, 11, '\P{Is_GCB=lf}', "");
Expect(0, 11, '\P{^Is_GCB=lf}', "");
Expect(1, 10, '\p{Is_GCB:	 LF}', "");
Expect(0, 10, '\p{^Is_GCB:	 LF}', "");
Expect(0, 10, '\P{Is_GCB:	 LF}', "");
Expect(1, 10, '\P{^Is_GCB:	 LF}', "");
Expect(0, 11, '\p{Is_GCB:	 LF}', "");
Expect(1, 11, '\p{^Is_GCB:	 LF}', "");
Expect(1, 11, '\P{Is_GCB:	 LF}', "");
Expect(0, 11, '\P{^Is_GCB:	 LF}', "");
Error('\p{Grapheme_Cluster_Break=-_l:=}');
Error('\P{Grapheme_Cluster_Break=-_l:=}');
Expect(1, 43388, '\p{Grapheme_Cluster_Break=l}', "");
Expect(0, 43388, '\p{^Grapheme_Cluster_Break=l}', "");
Expect(0, 43388, '\P{Grapheme_Cluster_Break=l}', "");
Expect(1, 43388, '\P{^Grapheme_Cluster_Break=l}', "");
Expect(0, 43389, '\p{Grapheme_Cluster_Break=l}', "");
Expect(1, 43389, '\p{^Grapheme_Cluster_Break=l}', "");
Expect(1, 43389, '\P{Grapheme_Cluster_Break=l}', "");
Expect(0, 43389, '\P{^Grapheme_Cluster_Break=l}', "");
Expect(1, 43388, '\p{Grapheme_Cluster_Break=- L}', "");
Expect(0, 43388, '\p{^Grapheme_Cluster_Break=- L}', "");
Expect(0, 43388, '\P{Grapheme_Cluster_Break=- L}', "");
Expect(1, 43388, '\P{^Grapheme_Cluster_Break=- L}', "");
Expect(0, 43389, '\p{Grapheme_Cluster_Break=- L}', "");
Expect(1, 43389, '\p{^Grapheme_Cluster_Break=- L}', "");
Expect(1, 43389, '\P{Grapheme_Cluster_Break=- L}', "");
Expect(0, 43389, '\P{^Grapheme_Cluster_Break=- L}', "");
Error('\p{GCB=	 l:=}');
Error('\P{GCB=	 l:=}');
Expect(1, 43388, '\p{GCB=l}', "");
Expect(0, 43388, '\p{^GCB=l}', "");
Expect(0, 43388, '\P{GCB=l}', "");
Expect(1, 43388, '\P{^GCB=l}', "");
Expect(0, 43389, '\p{GCB=l}', "");
Expect(1, 43389, '\p{^GCB=l}', "");
Expect(1, 43389, '\P{GCB=l}', "");
Expect(0, 43389, '\P{^GCB=l}', "");
Expect(1, 43388, '\p{GCB=-L}', "");
Expect(0, 43388, '\p{^GCB=-L}', "");
Expect(0, 43388, '\P{GCB=-L}', "");
Expect(1, 43388, '\P{^GCB=-L}', "");
Expect(0, 43389, '\p{GCB=-L}', "");
Expect(1, 43389, '\p{^GCB=-L}', "");
Expect(1, 43389, '\P{GCB=-L}', "");
Expect(0, 43389, '\P{^GCB=-L}', "");
Error('\p{Is_Grapheme_Cluster_Break=:=- L}');
Error('\P{Is_Grapheme_Cluster_Break=:=- L}');
Expect(1, 43388, '\p{Is_Grapheme_Cluster_Break=l}', "");
Expect(0, 43388, '\p{^Is_Grapheme_Cluster_Break=l}', "");
Expect(0, 43388, '\P{Is_Grapheme_Cluster_Break=l}', "");
Expect(1, 43388, '\P{^Is_Grapheme_Cluster_Break=l}', "");
Expect(0, 43389, '\p{Is_Grapheme_Cluster_Break=l}', "");
Expect(1, 43389, '\p{^Is_Grapheme_Cluster_Break=l}', "");
Expect(1, 43389, '\P{Is_Grapheme_Cluster_Break=l}', "");
Expect(0, 43389, '\P{^Is_Grapheme_Cluster_Break=l}', "");
Expect(1, 43388, '\p{Is_Grapheme_Cluster_Break=  L}', "");
Expect(0, 43388, '\p{^Is_Grapheme_Cluster_Break=  L}', "");
Expect(0, 43388, '\P{Is_Grapheme_Cluster_Break=  L}', "");
Expect(1, 43388, '\P{^Is_Grapheme_Cluster_Break=  L}', "");
Expect(0, 43389, '\p{Is_Grapheme_Cluster_Break=  L}', "");
Expect(1, 43389, '\p{^Is_Grapheme_Cluster_Break=  L}', "");
Expect(1, 43389, '\P{Is_Grapheme_Cluster_Break=  L}', "");
Expect(0, 43389, '\P{^Is_Grapheme_Cluster_Break=  L}', "");
Error('\p{Is_GCB:	-_L:=}');
Error('\P{Is_GCB:	-_L:=}');
Expect(1, 43388, '\p{Is_GCB=l}', "");
Expect(0, 43388, '\p{^Is_GCB=l}', "");
Expect(0, 43388, '\P{Is_GCB=l}', "");
Expect(1, 43388, '\P{^Is_GCB=l}', "");
Expect(0, 43389, '\p{Is_GCB=l}', "");
Expect(1, 43389, '\p{^Is_GCB=l}', "");
Expect(1, 43389, '\P{Is_GCB=l}', "");
Expect(0, 43389, '\P{^Is_GCB=l}', "");
Expect(1, 43388, '\p{Is_GCB=	L}', "");
Expect(0, 43388, '\p{^Is_GCB=	L}', "");
Expect(0, 43388, '\P{Is_GCB=	L}', "");
Expect(1, 43388, '\P{^Is_GCB=	L}', "");
Expect(0, 43389, '\p{Is_GCB=	L}', "");
Expect(1, 43389, '\p{^Is_GCB=	L}', "");
Expect(1, 43389, '\P{Is_GCB=	L}', "");
Expect(0, 43389, '\P{^Is_GCB=	L}', "");
Error('\p{Indic_Syllabic_Category=Consonant_Dead}');
Error('\P{Indic_Syllabic_Category=Consonant_Dead}');
Error('\p{InSC=Consonant_Dead}');
Error('\P{InSC=Consonant_Dead}');
Error('\p{Is_Indic_Syllabic_Category=Consonant_Dead}');
Error('\P{Is_Indic_Syllabic_Category=Consonant_Dead}');
Error('\p{Is_InSC: Consonant_Dead}');
Error('\P{Is_InSC: Consonant_Dead}');
Error('\p{Indic_Syllabic_Category=Avagraha}');
Error('\P{Indic_Syllabic_Category=Avagraha}');
Error('\p{InSC=Avagraha}');
Error('\P{InSC=Avagraha}');
Error('\p{Is_Indic_Syllabic_Category:   Avagraha}');
Error('\P{Is_Indic_Syllabic_Category:   Avagraha}');
Error('\p{Is_InSC=Avagraha}');
Error('\P{Is_InSC=Avagraha}');
Error('\p{Indic_Syllabic_Category=Vowel_Independent}');
Error('\P{Indic_Syllabic_Category=Vowel_Independent}');
Error('\p{InSC=Vowel_Independent}');
Error('\P{InSC=Vowel_Independent}');
Error('\p{Is_Indic_Syllabic_Category=Vowel_Independent}');
Error('\P{Is_Indic_Syllabic_Category=Vowel_Independent}');
Error('\p{Is_InSC=Vowel_Independent}');
Error('\P{Is_InSC=Vowel_Independent}');
Error('\p{Indic_Syllabic_Category=Bindu}');
Error('\P{Indic_Syllabic_Category=Bindu}');
Error('\p{InSC=Bindu}');
Error('\P{InSC=Bindu}');
Error('\p{Is_Indic_Syllabic_Category=Bindu}');
Error('\P{Is_Indic_Syllabic_Category=Bindu}');
Error('\p{Is_InSC=Bindu}');
Error('\P{Is_InSC=Bindu}');
Error('\p{Indic_Syllabic_Category:   Tone_Letter}');
Error('\P{Indic_Syllabic_Category:   Tone_Letter}');
Error('\p{InSC=Tone_Letter}');
Error('\P{InSC=Tone_Letter}');
Error('\p{Is_Indic_Syllabic_Category=Tone_Letter}');
Error('\P{Is_Indic_Syllabic_Category=Tone_Letter}');
Error('\p{Is_InSC:   Tone_Letter}');
Error('\P{Is_InSC:   Tone_Letter}');
Error('\p{Indic_Syllabic_Category=Consonant_Final}');
Error('\P{Indic_Syllabic_Category=Consonant_Final}');
Error('\p{InSC=Consonant_Final}');
Error('\P{InSC=Consonant_Final}');
Error('\p{Is_Indic_Syllabic_Category=Consonant_Final}');
Error('\P{Is_Indic_Syllabic_Category=Consonant_Final}');
Error('\p{Is_InSC=Consonant_Final}');
Error('\P{Is_InSC=Consonant_Final}');
Error('\p{Indic_Syllabic_Category=Consonant_Medial}');
Error('\P{Indic_Syllabic_Category=Consonant_Medial}');
Error('\p{InSC=Consonant_Medial}');
Error('\P{InSC=Consonant_Medial}');
Error('\p{Is_Indic_Syllabic_Category=Consonant_Medial}');
Error('\P{Is_Indic_Syllabic_Category=Consonant_Medial}');
Error('\p{Is_InSC=Consonant_Medial}');
Error('\P{Is_InSC=Consonant_Medial}');
Error('\p{Indic_Syllabic_Category=Register_Shifter}');
Error('\P{Indic_Syllabic_Category=Register_Shifter}');
Error('\p{InSC=Register_Shifter}');
Error('\P{InSC=Register_Shifter}');
Error('\p{Is_Indic_Syllabic_Category=Register_Shifter}');
Error('\P{Is_Indic_Syllabic_Category=Register_Shifter}');
Error('\p{Is_InSC=Register_Shifter}');
Error('\P{Is_InSC=Register_Shifter}');
Error('\p{Indic_Syllabic_Category=Vowel_Dependent}');
Error('\P{Indic_Syllabic_Category=Vowel_Dependent}');
Error('\p{InSC=Vowel_Dependent}');
Error('\P{InSC=Vowel_Dependent}');
Error('\p{Is_Indic_Syllabic_Category=Vowel_Dependent}');
Error('\P{Is_Indic_Syllabic_Category=Vowel_Dependent}');
Error('\p{Is_InSC=Vowel_Dependent}');
Error('\P{Is_InSC=Vowel_Dependent}');
Error('\p{Indic_Syllabic_Category=Tone_Mark}');
Error('\P{Indic_Syllabic_Category=Tone_Mark}');
Error('\p{InSC=Tone_Mark}');
Error('\P{InSC=Tone_Mark}');
Error('\p{Is_Indic_Syllabic_Category=Tone_Mark}');
Error('\P{Is_Indic_Syllabic_Category=Tone_Mark}');
Error('\p{Is_InSC=Tone_Mark}');
Error('\P{Is_InSC=Tone_Mark}');
Error('\p{Indic_Syllabic_Category=Nukta}');
Error('\P{Indic_Syllabic_Category=Nukta}');
Error('\p{InSC: Nukta}');
Error('\P{InSC: Nukta}');
Error('\p{Is_Indic_Syllabic_Category=Nukta}');
Error('\P{Is_Indic_Syllabic_Category=Nukta}');
Error('\p{Is_InSC=Nukta}');
Error('\P{Is_InSC=Nukta}');
Error('\p{Indic_Syllabic_Category=Consonant_Repha}');
Error('\P{Indic_Syllabic_Category=Consonant_Repha}');
Error('\p{InSC=Consonant_Repha}');
Error('\P{InSC=Consonant_Repha}');
Error('\p{Is_Indic_Syllabic_Category=Consonant_Repha}');
Error('\P{Is_Indic_Syllabic_Category=Consonant_Repha}');
Error('\p{Is_InSC=Consonant_Repha}');
Error('\P{Is_InSC=Consonant_Repha}');
Error('\p{Indic_Syllabic_Category: Consonant_Subjoined}');
Error('\P{Indic_Syllabic_Category: Consonant_Subjoined}');
Error('\p{InSC=Consonant_Subjoined}');
Error('\P{InSC=Consonant_Subjoined}');
Error('\p{Is_Indic_Syllabic_Category=Consonant_Subjoined}');
Error('\P{Is_Indic_Syllabic_Category=Consonant_Subjoined}');
Error('\p{Is_InSC=Consonant_Subjoined}');
Error('\P{Is_InSC=Consonant_Subjoined}');
Error('\p{Indic_Syllabic_Category:Virama}');
Error('\P{Indic_Syllabic_Category:Virama}');
Error('\p{InSC=Virama}');
Error('\P{InSC=Virama}');
Error('\p{Is_Indic_Syllabic_Category:Virama}');
Error('\P{Is_Indic_Syllabic_Category:Virama}');
Error('\p{Is_InSC=Virama}');
Error('\P{Is_InSC=Virama}');
Error('\p{Indic_Syllabic_Category=Consonant}');
Error('\P{Indic_Syllabic_Category=Consonant}');
Error('\p{InSC=Consonant}');
Error('\P{InSC=Consonant}');
Error('\p{Is_Indic_Syllabic_Category=Consonant}');
Error('\P{Is_Indic_Syllabic_Category=Consonant}');
Error('\p{Is_InSC=Consonant}');
Error('\P{Is_InSC=Consonant}');
Error('\p{Indic_Syllabic_Category=Consonant_Head_Letter}');
Error('\P{Indic_Syllabic_Category=Consonant_Head_Letter}');
Error('\p{InSC=Consonant_Head_Letter}');
Error('\P{InSC=Consonant_Head_Letter}');
Error('\p{Is_Indic_Syllabic_Category=Consonant_Head_Letter}');
Error('\P{Is_Indic_Syllabic_Category=Consonant_Head_Letter}');
Error('\p{Is_InSC=Consonant_Head_Letter}');
Error('\P{Is_InSC=Consonant_Head_Letter}');
Error('\p{Indic_Syllabic_Category=Visarga}');
Error('\P{Indic_Syllabic_Category=Visarga}');
Error('\p{InSC=Visarga}');
Error('\P{InSC=Visarga}');
Error('\p{Is_Indic_Syllabic_Category=Visarga}');
Error('\P{Is_Indic_Syllabic_Category=Visarga}');
Error('\p{Is_InSC=Visarga}');
Error('\P{Is_InSC=Visarga}');
Error('\p{Indic_Syllabic_Category=Modifying_Letter}');
Error('\P{Indic_Syllabic_Category=Modifying_Letter}');
Error('\p{InSC=Modifying_Letter}');
Error('\P{InSC=Modifying_Letter}');
Error('\p{Is_Indic_Syllabic_Category=Modifying_Letter}');
Error('\P{Is_Indic_Syllabic_Category=Modifying_Letter}');
Error('\p{Is_InSC=Modifying_Letter}');
Error('\P{Is_InSC=Modifying_Letter}');
Error('\p{Indic_Syllabic_Category=Other}');
Error('\P{Indic_Syllabic_Category=Other}');
Error('\p{InSC=Other}');
Error('\P{InSC=Other}');
Error('\p{Is_Indic_Syllabic_Category=Other}');
Error('\P{Is_Indic_Syllabic_Category=Other}');
Error('\p{Is_InSC=Other}');
Error('\P{Is_InSC=Other}');
Error('\p{Indic_Syllabic_Category=Consonant_Placeholder}');
Error('\P{Indic_Syllabic_Category=Consonant_Placeholder}');
Error('\p{InSC=Consonant_Placeholder}');
Error('\P{InSC=Consonant_Placeholder}');
Error('\p{Is_Indic_Syllabic_Category=Consonant_Placeholder}');
Error('\P{Is_Indic_Syllabic_Category=Consonant_Placeholder}');
Error('\p{Is_InSC=Consonant_Placeholder}');
Error('\P{Is_InSC=Consonant_Placeholder}');
Error('\p{Indic_Syllabic_Category=Vowel}');
Error('\P{Indic_Syllabic_Category=Vowel}');
Error('\p{InSC=Vowel}');
Error('\P{InSC=Vowel}');
Error('\p{Is_Indic_Syllabic_Category=Vowel}');
Error('\P{Is_Indic_Syllabic_Category=Vowel}');
Error('\p{Is_InSC:	Vowel}');
Error('\P{Is_InSC:	Vowel}');
Error('\p{Default_Ignorable_Code_Point=/a/Yes}');
Error('\P{Default_Ignorable_Code_Point=/a/Yes}');
Expect(1, 921599, '\p{Default_Ignorable_Code_Point=yes}', "");
Expect(0, 921599, '\p{^Default_Ignorable_Code_Point=yes}', "");
Expect(0, 921599, '\P{Default_Ignorable_Code_Point=yes}', "");
Expect(1, 921599, '\P{^Default_Ignorable_Code_Point=yes}', "");
Expect(0, 921600, '\p{Default_Ignorable_Code_Point=yes}', "");
Expect(1, 921600, '\p{^Default_Ignorable_Code_Point=yes}', "");
Expect(1, 921600, '\P{Default_Ignorable_Code_Point=yes}', "");
Expect(0, 921600, '\P{^Default_Ignorable_Code_Point=yes}', "");
Expect(1, 921599, '\p{Default_Ignorable_Code_Point= -Yes}', "");
Expect(0, 921599, '\p{^Default_Ignorable_Code_Point= -Yes}', "");
Expect(0, 921599, '\P{Default_Ignorable_Code_Point= -Yes}', "");
Expect(1, 921599, '\P{^Default_Ignorable_Code_Point= -Yes}', "");
Expect(0, 921600, '\p{Default_Ignorable_Code_Point= -Yes}', "");
Expect(1, 921600, '\p{^Default_Ignorable_Code_Point= -Yes}', "");
Expect(1, 921600, '\P{Default_Ignorable_Code_Point= -Yes}', "");
Expect(0, 921600, '\P{^Default_Ignorable_Code_Point= -Yes}', "");
Error('\p{DI=/a/_	Y}');
Error('\P{DI=/a/_	Y}');
Expect(1, 921599, '\p{DI=y}', "");
Expect(0, 921599, '\p{^DI=y}', "");
Expect(0, 921599, '\P{DI=y}', "");
Expect(1, 921599, '\P{^DI=y}', "");
Expect(0, 921600, '\p{DI=y}', "");
Expect(1, 921600, '\p{^DI=y}', "");
Expect(1, 921600, '\P{DI=y}', "");
Expect(0, 921600, '\P{^DI=y}', "");
Expect(1, 921599, '\p{DI=- Y}', "");
Expect(0, 921599, '\p{^DI=- Y}', "");
Expect(0, 921599, '\P{DI=- Y}', "");
Expect(1, 921599, '\P{^DI=- Y}', "");
Expect(0, 921600, '\p{DI=- Y}', "");
Expect(1, 921600, '\p{^DI=- Y}', "");
Expect(1, 921600, '\P{DI=- Y}', "");
Expect(0, 921600, '\P{^DI=- Y}', "");
Error('\p{Is_Default_Ignorable_Code_Point=:=-T}');
Error('\P{Is_Default_Ignorable_Code_Point=:=-T}');
Expect(1, 921599, '\p{Is_Default_Ignorable_Code_Point=t}', "");
Expect(0, 921599, '\p{^Is_Default_Ignorable_Code_Point=t}', "");
Expect(0, 921599, '\P{Is_Default_Ignorable_Code_Point=t}', "");
Expect(1, 921599, '\P{^Is_Default_Ignorable_Code_Point=t}', "");
Expect(0, 921600, '\p{Is_Default_Ignorable_Code_Point=t}', "");
Expect(1, 921600, '\p{^Is_Default_Ignorable_Code_Point=t}', "");
Expect(1, 921600, '\P{Is_Default_Ignorable_Code_Point=t}', "");
Expect(0, 921600, '\P{^Is_Default_Ignorable_Code_Point=t}', "");
Expect(1, 921599, '\p{Is_Default_Ignorable_Code_Point= _T}', "");
Expect(0, 921599, '\p{^Is_Default_Ignorable_Code_Point= _T}', "");
Expect(0, 921599, '\P{Is_Default_Ignorable_Code_Point= _T}', "");
Expect(1, 921599, '\P{^Is_Default_Ignorable_Code_Point= _T}', "");
Expect(0, 921600, '\p{Is_Default_Ignorable_Code_Point= _T}', "");
Expect(1, 921600, '\p{^Is_Default_Ignorable_Code_Point= _T}', "");
Expect(1, 921600, '\P{Is_Default_Ignorable_Code_Point= _T}', "");
Expect(0, 921600, '\P{^Is_Default_Ignorable_Code_Point= _T}', "");
Error('\p{Is_DI=_	True/a/}');
Error('\P{Is_DI=_	True/a/}');
Expect(1, 921599, '\p{Is_DI=true}', "");
Expect(0, 921599, '\p{^Is_DI=true}', "");
Expect(0, 921599, '\P{Is_DI=true}', "");
Expect(1, 921599, '\P{^Is_DI=true}', "");
Expect(0, 921600, '\p{Is_DI=true}', "");
Expect(1, 921600, '\p{^Is_DI=true}', "");
Expect(1, 921600, '\P{Is_DI=true}', "");
Expect(0, 921600, '\P{^Is_DI=true}', "");
Expect(1, 921599, '\p{Is_DI=--True}', "");
Expect(0, 921599, '\p{^Is_DI=--True}', "");
Expect(0, 921599, '\P{Is_DI=--True}', "");
Expect(1, 921599, '\P{^Is_DI=--True}', "");
Expect(0, 921600, '\p{Is_DI=--True}', "");
Expect(1, 921600, '\p{^Is_DI=--True}', "");
Expect(1, 921600, '\P{Is_DI=--True}', "");
Expect(0, 921600, '\P{^Is_DI=--True}', "");
Error('\p{Default_Ignorable_Code_Point=-_No:=}');
Error('\P{Default_Ignorable_Code_Point=-_No:=}');
Expect(1, 921600, '\p{Default_Ignorable_Code_Point=no}', "");
Expect(0, 921600, '\p{^Default_Ignorable_Code_Point=no}', "");
Expect(0, 921600, '\P{Default_Ignorable_Code_Point=no}', "");
Expect(1, 921600, '\P{^Default_Ignorable_Code_Point=no}', "");
Expect(0, 921599, '\p{Default_Ignorable_Code_Point=no}', "");
Expect(1, 921599, '\p{^Default_Ignorable_Code_Point=no}', "");
Expect(1, 921599, '\P{Default_Ignorable_Code_Point=no}', "");
Expect(0, 921599, '\P{^Default_Ignorable_Code_Point=no}', "");
Expect(1, 921600, '\p{Default_Ignorable_Code_Point=-_No}', "");
Expect(0, 921600, '\p{^Default_Ignorable_Code_Point=-_No}', "");
Expect(0, 921600, '\P{Default_Ignorable_Code_Point=-_No}', "");
Expect(1, 921600, '\P{^Default_Ignorable_Code_Point=-_No}', "");
Expect(0, 921599, '\p{Default_Ignorable_Code_Point=-_No}', "");
Expect(1, 921599, '\p{^Default_Ignorable_Code_Point=-_No}', "");
Expect(1, 921599, '\P{Default_Ignorable_Code_Point=-_No}', "");
Expect(0, 921599, '\P{^Default_Ignorable_Code_Point=-_No}', "");
Error('\p{DI=/a/-	N}');
Error('\P{DI=/a/-	N}');
Expect(1, 921600, '\p{DI=n}', "");
Expect(0, 921600, '\p{^DI=n}', "");
Expect(0, 921600, '\P{DI=n}', "");
Expect(1, 921600, '\P{^DI=n}', "");
Expect(0, 921599, '\p{DI=n}', "");
Expect(1, 921599, '\p{^DI=n}', "");
Expect(1, 921599, '\P{DI=n}', "");
Expect(0, 921599, '\P{^DI=n}', "");
Expect(1, 921600, '\p{DI:	 n}', "");
Expect(0, 921600, '\p{^DI:	 n}', "");
Expect(0, 921600, '\P{DI:	 n}', "");
Expect(1, 921600, '\P{^DI:	 n}', "");
Expect(0, 921599, '\p{DI:	 n}', "");
Expect(1, 921599, '\p{^DI:	 n}', "");
Expect(1, 921599, '\P{DI:	 n}', "");
Expect(0, 921599, '\P{^DI:	 n}', "");
Error('\p{Is_Default_Ignorable_Code_Point= :=F}');
Error('\P{Is_Default_Ignorable_Code_Point= :=F}');
Expect(1, 921600, '\p{Is_Default_Ignorable_Code_Point=f}', "");
Expect(0, 921600, '\p{^Is_Default_Ignorable_Code_Point=f}', "");
Expect(0, 921600, '\P{Is_Default_Ignorable_Code_Point=f}', "");
Expect(1, 921600, '\P{^Is_Default_Ignorable_Code_Point=f}', "");
Expect(0, 921599, '\p{Is_Default_Ignorable_Code_Point=f}', "");
Expect(1, 921599, '\p{^Is_Default_Ignorable_Code_Point=f}', "");
Expect(1, 921599, '\P{Is_Default_Ignorable_Code_Point=f}', "");
Expect(0, 921599, '\P{^Is_Default_Ignorable_Code_Point=f}', "");
Expect(1, 921600, '\p{Is_Default_Ignorable_Code_Point=--f}', "");
Expect(0, 921600, '\p{^Is_Default_Ignorable_Code_Point=--f}', "");
Expect(0, 921600, '\P{Is_Default_Ignorable_Code_Point=--f}', "");
Expect(1, 921600, '\P{^Is_Default_Ignorable_Code_Point=--f}', "");
Expect(0, 921599, '\p{Is_Default_Ignorable_Code_Point=--f}', "");
Expect(1, 921599, '\p{^Is_Default_Ignorable_Code_Point=--f}', "");
Expect(1, 921599, '\P{Is_Default_Ignorable_Code_Point=--f}', "");
Expect(0, 921599, '\P{^Is_Default_Ignorable_Code_Point=--f}', "");
Error('\p{Is_DI=:=	False}');
Error('\P{Is_DI=:=	False}');
Expect(1, 921600, '\p{Is_DI=false}', "");
Expect(0, 921600, '\p{^Is_DI=false}', "");
Expect(0, 921600, '\P{Is_DI=false}', "");
Expect(1, 921600, '\P{^Is_DI=false}', "");
Expect(0, 921599, '\p{Is_DI=false}', "");
Expect(1, 921599, '\p{^Is_DI=false}', "");
Expect(1, 921599, '\P{Is_DI=false}', "");
Expect(0, 921599, '\P{^Is_DI=false}', "");
Expect(1, 921600, '\p{Is_DI= 	FALSE}', "");
Expect(0, 921600, '\p{^Is_DI= 	FALSE}', "");
Expect(0, 921600, '\P{Is_DI= 	FALSE}', "");
Expect(1, 921600, '\P{^Is_DI= 	FALSE}', "");
Expect(0, 921599, '\p{Is_DI= 	FALSE}', "");
Expect(1, 921599, '\p{^Is_DI= 	FALSE}', "");
Expect(1, 921599, '\P{Is_DI= 	FALSE}', "");
Expect(0, 921599, '\P{^Is_DI= 	FALSE}', "");
Error('\p{ASCII_Hex_Digit:_:=yes}');
Error('\P{ASCII_Hex_Digit:_:=yes}');
Expect(1, 102, '\p{ASCII_Hex_Digit=yes}', "");
Expect(0, 102, '\p{^ASCII_Hex_Digit=yes}', "");
Expect(0, 102, '\P{ASCII_Hex_Digit=yes}', "");
Expect(1, 102, '\P{^ASCII_Hex_Digit=yes}', "");
Expect(0, 103, '\p{ASCII_Hex_Digit=yes}', "");
Expect(1, 103, '\p{^ASCII_Hex_Digit=yes}', "");
Expect(1, 103, '\P{ASCII_Hex_Digit=yes}', "");
Expect(0, 103, '\P{^ASCII_Hex_Digit=yes}', "");
Expect(1, 102, '\p{ASCII_Hex_Digit= YES}', "");
Expect(0, 102, '\p{^ASCII_Hex_Digit= YES}', "");
Expect(0, 102, '\P{ASCII_Hex_Digit= YES}', "");
Expect(1, 102, '\P{^ASCII_Hex_Digit= YES}', "");
Expect(0, 103, '\p{ASCII_Hex_Digit= YES}', "");
Expect(1, 103, '\p{^ASCII_Hex_Digit= YES}', "");
Expect(1, 103, '\P{ASCII_Hex_Digit= YES}', "");
Expect(0, 103, '\P{^ASCII_Hex_Digit= YES}', "");
Error('\p{AHex: /a/ _Y}');
Error('\P{AHex: /a/ _Y}');
Expect(1, 102, '\p{AHex=y}', "");
Expect(0, 102, '\p{^AHex=y}', "");
Expect(0, 102, '\P{AHex=y}', "");
Expect(1, 102, '\P{^AHex=y}', "");
Expect(0, 103, '\p{AHex=y}', "");
Expect(1, 103, '\p{^AHex=y}', "");
Expect(1, 103, '\P{AHex=y}', "");
Expect(0, 103, '\P{^AHex=y}', "");
Expect(1, 102, '\p{AHex= -y}', "");
Expect(0, 102, '\p{^AHex= -y}', "");
Expect(0, 102, '\P{AHex= -y}', "");
Expect(1, 102, '\P{^AHex= -y}', "");
Expect(0, 103, '\p{AHex= -y}', "");
Expect(1, 103, '\p{^AHex= -y}', "");
Expect(1, 103, '\P{AHex= -y}', "");
Expect(0, 103, '\P{^AHex= -y}', "");
Error('\p{Is_ASCII_Hex_Digit= /a/T}');
Error('\P{Is_ASCII_Hex_Digit= /a/T}');
Expect(1, 102, '\p{Is_ASCII_Hex_Digit=t}', "");
Expect(0, 102, '\p{^Is_ASCII_Hex_Digit=t}', "");
Expect(0, 102, '\P{Is_ASCII_Hex_Digit=t}', "");
Expect(1, 102, '\P{^Is_ASCII_Hex_Digit=t}', "");
Expect(0, 103, '\p{Is_ASCII_Hex_Digit=t}', "");
Expect(1, 103, '\p{^Is_ASCII_Hex_Digit=t}', "");
Expect(1, 103, '\P{Is_ASCII_Hex_Digit=t}', "");
Expect(0, 103, '\P{^Is_ASCII_Hex_Digit=t}', "");
Expect(1, 102, '\p{Is_ASCII_Hex_Digit=	T}', "");
Expect(0, 102, '\p{^Is_ASCII_Hex_Digit=	T}', "");
Expect(0, 102, '\P{Is_ASCII_Hex_Digit=	T}', "");
Expect(1, 102, '\P{^Is_ASCII_Hex_Digit=	T}', "");
Expect(0, 103, '\p{Is_ASCII_Hex_Digit=	T}', "");
Expect(1, 103, '\p{^Is_ASCII_Hex_Digit=	T}', "");
Expect(1, 103, '\P{Is_ASCII_Hex_Digit=	T}', "");
Expect(0, 103, '\P{^Is_ASCII_Hex_Digit=	T}', "");
Error('\p{Is_AHex=/a/_true}');
Error('\P{Is_AHex=/a/_true}');
Expect(1, 102, '\p{Is_AHex: true}', "");
Expect(0, 102, '\p{^Is_AHex: true}', "");
Expect(0, 102, '\P{Is_AHex: true}', "");
Expect(1, 102, '\P{^Is_AHex: true}', "");
Expect(0, 103, '\p{Is_AHex: true}', "");
Expect(1, 103, '\p{^Is_AHex: true}', "");
Expect(1, 103, '\P{Is_AHex: true}', "");
Expect(0, 103, '\P{^Is_AHex: true}', "");
Expect(1, 102, '\p{Is_AHex=_	True}', "");
Expect(0, 102, '\p{^Is_AHex=_	True}', "");
Expect(0, 102, '\P{Is_AHex=_	True}', "");
Expect(1, 102, '\P{^Is_AHex=_	True}', "");
Expect(0, 103, '\p{Is_AHex=_	True}', "");
Expect(1, 103, '\p{^Is_AHex=_	True}', "");
Expect(1, 103, '\P{Is_AHex=_	True}', "");
Expect(0, 103, '\P{^Is_AHex=_	True}', "");
Error('\p{ASCII_Hex_Digit=/a/	-NO}');
Error('\P{ASCII_Hex_Digit=/a/	-NO}');
Expect(1, 103, '\p{ASCII_Hex_Digit=no}', "");
Expect(0, 103, '\p{^ASCII_Hex_Digit=no}', "");
Expect(0, 103, '\P{ASCII_Hex_Digit=no}', "");
Expect(1, 103, '\P{^ASCII_Hex_Digit=no}', "");
Expect(0, 102, '\p{ASCII_Hex_Digit=no}', "");
Expect(1, 102, '\p{^ASCII_Hex_Digit=no}', "");
Expect(1, 102, '\P{ASCII_Hex_Digit=no}', "");
Expect(0, 102, '\P{^ASCII_Hex_Digit=no}', "");
Expect(1, 103, '\p{ASCII_Hex_Digit=__no}', "");
Expect(0, 103, '\p{^ASCII_Hex_Digit=__no}', "");
Expect(0, 103, '\P{ASCII_Hex_Digit=__no}', "");
Expect(1, 103, '\P{^ASCII_Hex_Digit=__no}', "");
Expect(0, 102, '\p{ASCII_Hex_Digit=__no}', "");
Expect(1, 102, '\p{^ASCII_Hex_Digit=__no}', "");
Expect(1, 102, '\P{ASCII_Hex_Digit=__no}', "");
Expect(0, 102, '\P{^ASCII_Hex_Digit=__no}', "");
Error('\p{AHex= _N:=}');
Error('\P{AHex= _N:=}');
Expect(1, 103, '\p{AHex=n}', "");
Expect(0, 103, '\p{^AHex=n}', "");
Expect(0, 103, '\P{AHex=n}', "");
Expect(1, 103, '\P{^AHex=n}', "");
Expect(0, 102, '\p{AHex=n}', "");
Expect(1, 102, '\p{^AHex=n}', "");
Expect(1, 102, '\P{AHex=n}', "");
Expect(0, 102, '\P{^AHex=n}', "");
Expect(1, 103, '\p{AHex=_	N}', "");
Expect(0, 103, '\p{^AHex=_	N}', "");
Expect(0, 103, '\P{AHex=_	N}', "");
Expect(1, 103, '\P{^AHex=_	N}', "");
Expect(0, 102, '\p{AHex=_	N}', "");
Expect(1, 102, '\p{^AHex=_	N}', "");
Expect(1, 102, '\P{AHex=_	N}', "");
Expect(0, 102, '\P{^AHex=_	N}', "");
Error('\p{Is_ASCII_Hex_Digit:	-f/a/}');
Error('\P{Is_ASCII_Hex_Digit:	-f/a/}');
Expect(1, 103, '\p{Is_ASCII_Hex_Digit=f}', "");
Expect(0, 103, '\p{^Is_ASCII_Hex_Digit=f}', "");
Expect(0, 103, '\P{Is_ASCII_Hex_Digit=f}', "");
Expect(1, 103, '\P{^Is_ASCII_Hex_Digit=f}', "");
Expect(0, 102, '\p{Is_ASCII_Hex_Digit=f}', "");
Expect(1, 102, '\p{^Is_ASCII_Hex_Digit=f}', "");
Expect(1, 102, '\P{Is_ASCII_Hex_Digit=f}', "");
Expect(0, 102, '\P{^Is_ASCII_Hex_Digit=f}', "");
Expect(1, 103, '\p{Is_ASCII_Hex_Digit=-_F}', "");
Expect(0, 103, '\p{^Is_ASCII_Hex_Digit=-_F}', "");
Expect(0, 103, '\P{Is_ASCII_Hex_Digit=-_F}', "");
Expect(1, 103, '\P{^Is_ASCII_Hex_Digit=-_F}', "");
Expect(0, 102, '\p{Is_ASCII_Hex_Digit=-_F}', "");
Expect(1, 102, '\p{^Is_ASCII_Hex_Digit=-_F}', "");
Expect(1, 102, '\P{Is_ASCII_Hex_Digit=-_F}', "");
Expect(0, 102, '\P{^Is_ASCII_Hex_Digit=-_F}', "");
Error('\p{Is_AHex:	_-false:=}');
Error('\P{Is_AHex:	_-false:=}');
Expect(1, 103, '\p{Is_AHex:   false}', "");
Expect(0, 103, '\p{^Is_AHex:   false}', "");
Expect(0, 103, '\P{Is_AHex:   false}', "");
Expect(1, 103, '\P{^Is_AHex:   false}', "");
Expect(0, 102, '\p{Is_AHex:   false}', "");
Expect(1, 102, '\p{^Is_AHex:   false}', "");
Expect(1, 102, '\P{Is_AHex:   false}', "");
Expect(0, 102, '\P{^Is_AHex:   false}', "");
Expect(1, 103, '\p{Is_AHex=__FALSE}', "");
Expect(0, 103, '\p{^Is_AHex=__FALSE}', "");
Expect(0, 103, '\P{Is_AHex=__FALSE}', "");
Expect(1, 103, '\P{^Is_AHex=__FALSE}', "");
Expect(0, 102, '\p{Is_AHex=__FALSE}', "");
Expect(1, 102, '\p{^Is_AHex=__FALSE}', "");
Expect(1, 102, '\P{Is_AHex=__FALSE}', "");
Expect(0, 102, '\P{^Is_AHex=__FALSE}', "");
Error('\p{NFD_Quick_Check=-:=Yes}');
Error('\P{NFD_Quick_Check=-:=Yes}');
Expect(1, 195102, '\p{NFD_Quick_Check=yes}', "");
Expect(0, 195102, '\p{^NFD_Quick_Check=yes}', "");
Expect(0, 195102, '\P{NFD_Quick_Check=yes}', "");
Expect(1, 195102, '\P{^NFD_Quick_Check=yes}', "");
Expect(0, 195101, '\p{NFD_Quick_Check=yes}', "");
Expect(1, 195101, '\p{^NFD_Quick_Check=yes}', "");
Expect(1, 195101, '\P{NFD_Quick_Check=yes}', "");
Expect(0, 195101, '\P{^NFD_Quick_Check=yes}', "");
Expect(1, 195102, '\p{NFD_Quick_Check=--yes}', "");
Expect(0, 195102, '\p{^NFD_Quick_Check=--yes}', "");
Expect(0, 195102, '\P{NFD_Quick_Check=--yes}', "");
Expect(1, 195102, '\P{^NFD_Quick_Check=--yes}', "");
Expect(0, 195101, '\p{NFD_Quick_Check=--yes}', "");
Expect(1, 195101, '\p{^NFD_Quick_Check=--yes}', "");
Expect(1, 195101, '\P{NFD_Quick_Check=--yes}', "");
Expect(0, 195101, '\P{^NFD_Quick_Check=--yes}', "");
Error('\p{NFD_QC:/a/ Y}');
Error('\P{NFD_QC:/a/ Y}');
Expect(1, 195102, '\p{NFD_QC=y}', "");
Expect(0, 195102, '\p{^NFD_QC=y}', "");
Expect(0, 195102, '\P{NFD_QC=y}', "");
Expect(1, 195102, '\P{^NFD_QC=y}', "");
Expect(0, 195101, '\p{NFD_QC=y}', "");
Expect(1, 195101, '\p{^NFD_QC=y}', "");
Expect(1, 195101, '\P{NFD_QC=y}', "");
Expect(0, 195101, '\P{^NFD_QC=y}', "");
Expect(1, 195102, '\p{NFD_QC=__y}', "");
Expect(0, 195102, '\p{^NFD_QC=__y}', "");
Expect(0, 195102, '\P{NFD_QC=__y}', "");
Expect(1, 195102, '\P{^NFD_QC=__y}', "");
Expect(0, 195101, '\p{NFD_QC=__y}', "");
Expect(1, 195101, '\p{^NFD_QC=__y}', "");
Expect(1, 195101, '\P{NFD_QC=__y}', "");
Expect(0, 195101, '\P{^NFD_QC=__y}', "");
Error('\p{Is_NFD_Quick_Check=_ Yes:=}');
Error('\P{Is_NFD_Quick_Check=_ Yes:=}');
Expect(1, 195102, '\p{Is_NFD_Quick_Check:   yes}', "");
Expect(0, 195102, '\p{^Is_NFD_Quick_Check:   yes}', "");
Expect(0, 195102, '\P{Is_NFD_Quick_Check:   yes}', "");
Expect(1, 195102, '\P{^Is_NFD_Quick_Check:   yes}', "");
Expect(0, 195101, '\p{Is_NFD_Quick_Check:   yes}', "");
Expect(1, 195101, '\p{^Is_NFD_Quick_Check:   yes}', "");
Expect(1, 195101, '\P{Is_NFD_Quick_Check:   yes}', "");
Expect(0, 195101, '\P{^Is_NFD_Quick_Check:   yes}', "");
Expect(1, 195102, '\p{Is_NFD_Quick_Check=--Yes}', "");
Expect(0, 195102, '\p{^Is_NFD_Quick_Check=--Yes}', "");
Expect(0, 195102, '\P{Is_NFD_Quick_Check=--Yes}', "");
Expect(1, 195102, '\P{^Is_NFD_Quick_Check=--Yes}', "");
Expect(0, 195101, '\p{Is_NFD_Quick_Check=--Yes}', "");
Expect(1, 195101, '\p{^Is_NFD_Quick_Check=--Yes}', "");
Expect(1, 195101, '\P{Is_NFD_Quick_Check=--Yes}', "");
Expect(0, 195101, '\P{^Is_NFD_Quick_Check=--Yes}', "");
Error('\p{Is_NFD_QC= _Y/a/}');
Error('\P{Is_NFD_QC= _Y/a/}');
Expect(1, 195102, '\p{Is_NFD_QC=y}', "");
Expect(0, 195102, '\p{^Is_NFD_QC=y}', "");
Expect(0, 195102, '\P{Is_NFD_QC=y}', "");
Expect(1, 195102, '\P{^Is_NFD_QC=y}', "");
Expect(0, 195101, '\p{Is_NFD_QC=y}', "");
Expect(1, 195101, '\p{^Is_NFD_QC=y}', "");
Expect(1, 195101, '\P{Is_NFD_QC=y}', "");
Expect(0, 195101, '\P{^Is_NFD_QC=y}', "");
Expect(1, 195102, '\p{Is_NFD_QC= _Y}', "");
Expect(0, 195102, '\p{^Is_NFD_QC= _Y}', "");
Expect(0, 195102, '\P{Is_NFD_QC= _Y}', "");
Expect(1, 195102, '\P{^Is_NFD_QC= _Y}', "");
Expect(0, 195101, '\p{Is_NFD_QC= _Y}', "");
Expect(1, 195101, '\p{^Is_NFD_QC= _Y}', "");
Expect(1, 195101, '\P{Is_NFD_QC= _Y}', "");
Expect(0, 195101, '\P{^Is_NFD_QC= _Y}', "");
Error('\p{NFD_Quick_Check=-no/a/}');
Error('\P{NFD_Quick_Check=-no/a/}');
Expect(1, 195101, '\p{NFD_Quick_Check=no}', "");
Expect(0, 195101, '\p{^NFD_Quick_Check=no}', "");
Expect(0, 195101, '\P{NFD_Quick_Check=no}', "");
Expect(1, 195101, '\P{^NFD_Quick_Check=no}', "");
Expect(0, 195102, '\p{NFD_Quick_Check=no}', "");
Expect(1, 195102, '\p{^NFD_Quick_Check=no}', "");
Expect(1, 195102, '\P{NFD_Quick_Check=no}', "");
Expect(0, 195102, '\P{^NFD_Quick_Check=no}', "");
Expect(1, 195101, '\p{NFD_Quick_Check=	-No}', "");
Expect(0, 195101, '\p{^NFD_Quick_Check=	-No}', "");
Expect(0, 195101, '\P{NFD_Quick_Check=	-No}', "");
Expect(1, 195101, '\P{^NFD_Quick_Check=	-No}', "");
Expect(0, 195102, '\p{NFD_Quick_Check=	-No}', "");
Expect(1, 195102, '\p{^NFD_Quick_Check=	-No}', "");
Expect(1, 195102, '\P{NFD_Quick_Check=	-No}', "");
Expect(0, 195102, '\P{^NFD_Quick_Check=	-No}', "");
Error('\p{NFD_QC=	_n/a/}');
Error('\P{NFD_QC=	_n/a/}');
Expect(1, 195101, '\p{NFD_QC=n}', "");
Expect(0, 195101, '\p{^NFD_QC=n}', "");
Expect(0, 195101, '\P{NFD_QC=n}', "");
Expect(1, 195101, '\P{^NFD_QC=n}', "");
Expect(0, 195102, '\p{NFD_QC=n}', "");
Expect(1, 195102, '\p{^NFD_QC=n}', "");
Expect(1, 195102, '\P{NFD_QC=n}', "");
Expect(0, 195102, '\P{^NFD_QC=n}', "");
Expect(1, 195101, '\p{NFD_QC=	N}', "");
Expect(0, 195101, '\p{^NFD_QC=	N}', "");
Expect(0, 195101, '\P{NFD_QC=	N}', "");
Expect(1, 195101, '\P{^NFD_QC=	N}', "");
Expect(0, 195102, '\p{NFD_QC=	N}', "");
Expect(1, 195102, '\p{^NFD_QC=	N}', "");
Expect(1, 195102, '\P{NFD_QC=	N}', "");
Expect(0, 195102, '\P{^NFD_QC=	N}', "");
Error('\p{Is_NFD_Quick_Check:   		No/a/}');
Error('\P{Is_NFD_Quick_Check:   		No/a/}');
Expect(1, 195101, '\p{Is_NFD_Quick_Check=no}', "");
Expect(0, 195101, '\p{^Is_NFD_Quick_Check=no}', "");
Expect(0, 195101, '\P{Is_NFD_Quick_Check=no}', "");
Expect(1, 195101, '\P{^Is_NFD_Quick_Check=no}', "");
Expect(0, 195102, '\p{Is_NFD_Quick_Check=no}', "");
Expect(1, 195102, '\p{^Is_NFD_Quick_Check=no}', "");
Expect(1, 195102, '\P{Is_NFD_Quick_Check=no}', "");
Expect(0, 195102, '\P{^Is_NFD_Quick_Check=no}', "");
Expect(1, 195101, '\p{Is_NFD_Quick_Check=-	no}', "");
Expect(0, 195101, '\p{^Is_NFD_Quick_Check=-	no}', "");
Expect(0, 195101, '\P{Is_NFD_Quick_Check=-	no}', "");
Expect(1, 195101, '\P{^Is_NFD_Quick_Check=-	no}', "");
Expect(0, 195102, '\p{Is_NFD_Quick_Check=-	no}', "");
Expect(1, 195102, '\p{^Is_NFD_Quick_Check=-	no}', "");
Expect(1, 195102, '\P{Is_NFD_Quick_Check=-	no}', "");
Expect(0, 195102, '\P{^Is_NFD_Quick_Check=-	no}', "");
Error('\p{Is_NFD_QC=:=  n}');
Error('\P{Is_NFD_QC=:=  n}');
Expect(1, 195101, '\p{Is_NFD_QC=n}', "");
Expect(0, 195101, '\p{^Is_NFD_QC=n}', "");
Expect(0, 195101, '\P{Is_NFD_QC=n}', "");
Expect(1, 195101, '\P{^Is_NFD_QC=n}', "");
Expect(0, 195102, '\p{Is_NFD_QC=n}', "");
Expect(1, 195102, '\p{^Is_NFD_QC=n}', "");
Expect(1, 195102, '\P{Is_NFD_QC=n}', "");
Expect(0, 195102, '\P{^Is_NFD_QC=n}', "");
Expect(1, 195101, '\p{Is_NFD_QC=_ n}', "");
Expect(0, 195101, '\p{^Is_NFD_QC=_ n}', "");
Expect(0, 195101, '\P{Is_NFD_QC=_ n}', "");
Expect(1, 195101, '\P{^Is_NFD_QC=_ n}', "");
Expect(0, 195102, '\p{Is_NFD_QC=_ n}', "");
Expect(1, 195102, '\p{^Is_NFD_QC=_ n}', "");
Expect(1, 195102, '\P{Is_NFD_QC=_ n}', "");
Expect(0, 195102, '\P{^Is_NFD_QC=_ n}', "");
Error('\p{Expands_On_NFKC=Yes}');
Error('\P{Expands_On_NFKC=Yes}');
Error('\p{XO_NFKC=Y}');
Error('\P{XO_NFKC=Y}');
Error('\p{Is_Expands_On_NFKC=T}');
Error('\P{Is_Expands_On_NFKC=T}');
Error('\p{Is_XO_NFKC=True}');
Error('\P{Is_XO_NFKC=True}');
Error('\p{Expands_On_NFKC=No}');
Error('\P{Expands_On_NFKC=No}');
Error('\p{XO_NFKC=N}');
Error('\P{XO_NFKC=N}');
Error('\p{Is_Expands_On_NFKC=F}');
Error('\P{Is_Expands_On_NFKC=F}');
Error('\p{Is_XO_NFKC=False}');
Error('\P{Is_XO_NFKC=False}');
Error('\p{Other_Lowercase=Yes}');
Error('\P{Other_Lowercase=Yes}');
Error('\p{OLower=Y}');
Error('\P{OLower=Y}');
Error('\p{Is_Other_Lowercase=T}');
Error('\P{Is_Other_Lowercase=T}');
Error('\p{Is_OLower:   True}');
Error('\P{Is_OLower:   True}');
Error('\p{Other_Lowercase=No}');
Error('\P{Other_Lowercase=No}');
Error('\p{OLower=N}');
Error('\P{OLower=N}');
Error('\p{Is_Other_Lowercase:	F}');
Error('\P{Is_Other_Lowercase:	F}');
Error('\p{Is_OLower=False}');
Error('\P{Is_OLower=False}');
Error('\p{Pattern_White_Space=	/a/Yes}');
Error('\P{Pattern_White_Space=	/a/Yes}');
Expect(1, 8233, '\p{Pattern_White_Space=yes}', "");
Expect(0, 8233, '\p{^Pattern_White_Space=yes}', "");
Expect(0, 8233, '\P{Pattern_White_Space=yes}', "");
Expect(1, 8233, '\P{^Pattern_White_Space=yes}', "");
Expect(0, 8234, '\p{Pattern_White_Space=yes}', "");
Expect(1, 8234, '\p{^Pattern_White_Space=yes}', "");
Expect(1, 8234, '\P{Pattern_White_Space=yes}', "");
Expect(0, 8234, '\P{^Pattern_White_Space=yes}', "");
Expect(1, 8233, '\p{Pattern_White_Space=	yes}', "");
Expect(0, 8233, '\p{^Pattern_White_Space=	yes}', "");
Expect(0, 8233, '\P{Pattern_White_Space=	yes}', "");
Expect(1, 8233, '\P{^Pattern_White_Space=	yes}', "");
Expect(0, 8234, '\p{Pattern_White_Space=	yes}', "");
Expect(1, 8234, '\p{^Pattern_White_Space=	yes}', "");
Expect(1, 8234, '\P{Pattern_White_Space=	yes}', "");
Expect(0, 8234, '\P{^Pattern_White_Space=	yes}', "");
Error('\p{Pat_WS= /a/Y}');
Error('\P{Pat_WS= /a/Y}');
Expect(1, 8233, '\p{Pat_WS:   y}', "");
Expect(0, 8233, '\p{^Pat_WS:   y}', "");
Expect(0, 8233, '\P{Pat_WS:   y}', "");
Expect(1, 8233, '\P{^Pat_WS:   y}', "");
Expect(0, 8234, '\p{Pat_WS:   y}', "");
Expect(1, 8234, '\p{^Pat_WS:   y}', "");
Expect(1, 8234, '\P{Pat_WS:   y}', "");
Expect(0, 8234, '\P{^Pat_WS:   y}', "");
Expect(1, 8233, '\p{Pat_WS:		Y}', "");
Expect(0, 8233, '\p{^Pat_WS:		Y}', "");
Expect(0, 8233, '\P{Pat_WS:		Y}', "");
Expect(1, 8233, '\P{^Pat_WS:		Y}', "");
Expect(0, 8234, '\p{Pat_WS:		Y}', "");
Expect(1, 8234, '\p{^Pat_WS:		Y}', "");
Expect(1, 8234, '\P{Pat_WS:		Y}', "");
Expect(0, 8234, '\P{^Pat_WS:		Y}', "");
Error('\p{Is_Pattern_White_Space=_/a/T}');
Error('\P{Is_Pattern_White_Space=_/a/T}');
Expect(1, 8233, '\p{Is_Pattern_White_Space=t}', "");
Expect(0, 8233, '\p{^Is_Pattern_White_Space=t}', "");
Expect(0, 8233, '\P{Is_Pattern_White_Space=t}', "");
Expect(1, 8233, '\P{^Is_Pattern_White_Space=t}', "");
Expect(0, 8234, '\p{Is_Pattern_White_Space=t}', "");
Expect(1, 8234, '\p{^Is_Pattern_White_Space=t}', "");
Expect(1, 8234, '\P{Is_Pattern_White_Space=t}', "");
Expect(0, 8234, '\P{^Is_Pattern_White_Space=t}', "");
Expect(1, 8233, '\p{Is_Pattern_White_Space= -T}', "");
Expect(0, 8233, '\p{^Is_Pattern_White_Space= -T}', "");
Expect(0, 8233, '\P{Is_Pattern_White_Space= -T}', "");
Expect(1, 8233, '\P{^Is_Pattern_White_Space= -T}', "");
Expect(0, 8234, '\p{Is_Pattern_White_Space= -T}', "");
Expect(1, 8234, '\p{^Is_Pattern_White_Space= -T}', "");
Expect(1, 8234, '\P{Is_Pattern_White_Space= -T}', "");
Expect(0, 8234, '\P{^Is_Pattern_White_Space= -T}', "");
Error('\p{Is_Pat_WS=/a/_true}');
Error('\P{Is_Pat_WS=/a/_true}');
Expect(1, 8233, '\p{Is_Pat_WS=true}', "");
Expect(0, 8233, '\p{^Is_Pat_WS=true}', "");
Expect(0, 8233, '\P{Is_Pat_WS=true}', "");
Expect(1, 8233, '\P{^Is_Pat_WS=true}', "");
Expect(0, 8234, '\p{Is_Pat_WS=true}', "");
Expect(1, 8234, '\p{^Is_Pat_WS=true}', "");
Expect(1, 8234, '\P{Is_Pat_WS=true}', "");
Expect(0, 8234, '\P{^Is_Pat_WS=true}', "");
Expect(1, 8233, '\p{Is_Pat_WS=	-TRUE}', "");
Expect(0, 8233, '\p{^Is_Pat_WS=	-TRUE}', "");
Expect(0, 8233, '\P{Is_Pat_WS=	-TRUE}', "");
Expect(1, 8233, '\P{^Is_Pat_WS=	-TRUE}', "");
Expect(0, 8234, '\p{Is_Pat_WS=	-TRUE}', "");
Expect(1, 8234, '\p{^Is_Pat_WS=	-TRUE}', "");
Expect(1, 8234, '\P{Is_Pat_WS=	-TRUE}', "");
Expect(0, 8234, '\P{^Is_Pat_WS=	-TRUE}', "");
Error('\p{Pattern_White_Space=:=  No}');
Error('\P{Pattern_White_Space=:=  No}');
Expect(1, 8234, '\p{Pattern_White_Space=no}', "");
Expect(0, 8234, '\p{^Pattern_White_Space=no}', "");
Expect(0, 8234, '\P{Pattern_White_Space=no}', "");
Expect(1, 8234, '\P{^Pattern_White_Space=no}', "");
Expect(0, 8233, '\p{Pattern_White_Space=no}', "");
Expect(1, 8233, '\p{^Pattern_White_Space=no}', "");
Expect(1, 8233, '\P{Pattern_White_Space=no}', "");
Expect(0, 8233, '\P{^Pattern_White_Space=no}', "");
Expect(1, 8234, '\p{Pattern_White_Space:	no}', "");
Expect(0, 8234, '\p{^Pattern_White_Space:	no}', "");
Expect(0, 8234, '\P{Pattern_White_Space:	no}', "");
Expect(1, 8234, '\P{^Pattern_White_Space:	no}', "");
Expect(0, 8233, '\p{Pattern_White_Space:	no}', "");
Expect(1, 8233, '\p{^Pattern_White_Space:	no}', "");
Expect(1, 8233, '\P{Pattern_White_Space:	no}', "");
Expect(0, 8233, '\P{^Pattern_White_Space:	no}', "");
Error('\p{Pat_WS=		n/a/}');
Error('\P{Pat_WS=		n/a/}');
Expect(1, 8234, '\p{Pat_WS=n}', "");
Expect(0, 8234, '\p{^Pat_WS=n}', "");
Expect(0, 8234, '\P{Pat_WS=n}', "");
Expect(1, 8234, '\P{^Pat_WS=n}', "");
Expect(0, 8233, '\p{Pat_WS=n}', "");
Expect(1, 8233, '\p{^Pat_WS=n}', "");
Expect(1, 8233, '\P{Pat_WS=n}', "");
Expect(0, 8233, '\P{^Pat_WS=n}', "");
Expect(1, 8234, '\p{Pat_WS=_N}', "");
Expect(0, 8234, '\p{^Pat_WS=_N}', "");
Expect(0, 8234, '\P{Pat_WS=_N}', "");
Expect(1, 8234, '\P{^Pat_WS=_N}', "");
Expect(0, 8233, '\p{Pat_WS=_N}', "");
Expect(1, 8233, '\p{^Pat_WS=_N}', "");
Expect(1, 8233, '\P{Pat_WS=_N}', "");
Expect(0, 8233, '\P{^Pat_WS=_N}', "");
Error('\p{Is_Pattern_White_Space=-/a/f}');
Error('\P{Is_Pattern_White_Space=-/a/f}');
Expect(1, 8234, '\p{Is_Pattern_White_Space=f}', "");
Expect(0, 8234, '\p{^Is_Pattern_White_Space=f}', "");
Expect(0, 8234, '\P{Is_Pattern_White_Space=f}', "");
Expect(1, 8234, '\P{^Is_Pattern_White_Space=f}', "");
Expect(0, 8233, '\p{Is_Pattern_White_Space=f}', "");
Expect(1, 8233, '\p{^Is_Pattern_White_Space=f}', "");
Expect(1, 8233, '\P{Is_Pattern_White_Space=f}', "");
Expect(0, 8233, '\P{^Is_Pattern_White_Space=f}', "");
Expect(1, 8234, '\p{Is_Pattern_White_Space:- F}', "");
Expect(0, 8234, '\p{^Is_Pattern_White_Space:- F}', "");
Expect(0, 8234, '\P{Is_Pattern_White_Space:- F}', "");
Expect(1, 8234, '\P{^Is_Pattern_White_Space:- F}', "");
Expect(0, 8233, '\p{Is_Pattern_White_Space:- F}', "");
Expect(1, 8233, '\p{^Is_Pattern_White_Space:- F}', "");
Expect(1, 8233, '\P{Is_Pattern_White_Space:- F}', "");
Expect(0, 8233, '\P{^Is_Pattern_White_Space:- F}', "");
Error('\p{Is_Pat_WS=	/a/False}');
Error('\P{Is_Pat_WS=	/a/False}');
Expect(1, 8234, '\p{Is_Pat_WS=false}', "");
Expect(0, 8234, '\p{^Is_Pat_WS=false}', "");
Expect(0, 8234, '\P{Is_Pat_WS=false}', "");
Expect(1, 8234, '\P{^Is_Pat_WS=false}', "");
Expect(0, 8233, '\p{Is_Pat_WS=false}', "");
Expect(1, 8233, '\p{^Is_Pat_WS=false}', "");
Expect(1, 8233, '\P{Is_Pat_WS=false}', "");
Expect(0, 8233, '\P{^Is_Pat_WS=false}', "");
Expect(1, 8234, '\p{Is_Pat_WS= -false}', "");
Expect(0, 8234, '\p{^Is_Pat_WS= -false}', "");
Expect(0, 8234, '\P{Is_Pat_WS= -false}', "");
Expect(1, 8234, '\P{^Is_Pat_WS= -false}', "");
Expect(0, 8233, '\p{Is_Pat_WS= -false}', "");
Expect(1, 8233, '\p{^Is_Pat_WS= -false}', "");
Expect(1, 8233, '\P{Is_Pat_WS= -false}', "");
Expect(0, 8233, '\P{^Is_Pat_WS= -false}', "");
Error('\p{Joining_Group:	/a/rohingya_YEH}');
Error('\P{Joining_Group:	/a/rohingya_YEH}');
Expect(1, 2220, '\p{Joining_Group=rohingyayeh}', "");
Expect(0, 2220, '\p{^Joining_Group=rohingyayeh}', "");
Expect(0, 2220, '\P{Joining_Group=rohingyayeh}', "");
Expect(1, 2220, '\P{^Joining_Group=rohingyayeh}', "");
Expect(0, 2221, '\p{Joining_Group=rohingyayeh}', "");
Expect(1, 2221, '\p{^Joining_Group=rohingyayeh}', "");
Expect(1, 2221, '\P{Joining_Group=rohingyayeh}', "");
Expect(0, 2221, '\P{^Joining_Group=rohingyayeh}', "");
Expect(1, 2220, '\p{Joining_Group=_ Rohingya_Yeh}', "");
Expect(0, 2220, '\p{^Joining_Group=_ Rohingya_Yeh}', "");
Expect(0, 2220, '\P{Joining_Group=_ Rohingya_Yeh}', "");
Expect(1, 2220, '\P{^Joining_Group=_ Rohingya_Yeh}', "");
Expect(0, 2221, '\p{Joining_Group=_ Rohingya_Yeh}', "");
Expect(1, 2221, '\p{^Joining_Group=_ Rohingya_Yeh}', "");
Expect(1, 2221, '\P{Joining_Group=_ Rohingya_Yeh}', "");
Expect(0, 2221, '\P{^Joining_Group=_ Rohingya_Yeh}', "");
Error('\p{Jg=_:=Rohingya_Yeh}');
Error('\P{Jg=_:=Rohingya_Yeh}');
Expect(1, 2220, '\p{Jg=rohingyayeh}', "");
Expect(0, 2220, '\p{^Jg=rohingyayeh}', "");
Expect(0, 2220, '\P{Jg=rohingyayeh}', "");
Expect(1, 2220, '\P{^Jg=rohingyayeh}', "");
Expect(0, 2221, '\p{Jg=rohingyayeh}', "");
Expect(1, 2221, '\p{^Jg=rohingyayeh}', "");
Expect(1, 2221, '\P{Jg=rohingyayeh}', "");
Expect(0, 2221, '\P{^Jg=rohingyayeh}', "");
Expect(1, 2220, '\p{Jg= 	Rohingya_Yeh}', "");
Expect(0, 2220, '\p{^Jg= 	Rohingya_Yeh}', "");
Expect(0, 2220, '\P{Jg= 	Rohingya_Yeh}', "");
Expect(1, 2220, '\P{^Jg= 	Rohingya_Yeh}', "");
Expect(0, 2221, '\p{Jg= 	Rohingya_Yeh}', "");
Expect(1, 2221, '\p{^Jg= 	Rohingya_Yeh}', "");
Expect(1, 2221, '\P{Jg= 	Rohingya_Yeh}', "");
Expect(0, 2221, '\P{^Jg= 	Rohingya_Yeh}', "");
Error('\p{Is_Joining_Group: _-ROHINGYA_Yeh/a/}');
Error('\P{Is_Joining_Group: _-ROHINGYA_Yeh/a/}');
Expect(1, 2220, '\p{Is_Joining_Group=rohingyayeh}', "");
Expect(0, 2220, '\p{^Is_Joining_Group=rohingyayeh}', "");
Expect(0, 2220, '\P{Is_Joining_Group=rohingyayeh}', "");
Expect(1, 2220, '\P{^Is_Joining_Group=rohingyayeh}', "");
Expect(0, 2221, '\p{Is_Joining_Group=rohingyayeh}', "");
Expect(1, 2221, '\p{^Is_Joining_Group=rohingyayeh}', "");
Expect(1, 2221, '\P{Is_Joining_Group=rohingyayeh}', "");
Expect(0, 2221, '\P{^Is_Joining_Group=rohingyayeh}', "");
Expect(1, 2220, '\p{Is_Joining_Group=-Rohingya_Yeh}', "");
Expect(0, 2220, '\p{^Is_Joining_Group=-Rohingya_Yeh}', "");
Expect(0, 2220, '\P{Is_Joining_Group=-Rohingya_Yeh}', "");
Expect(1, 2220, '\P{^Is_Joining_Group=-Rohingya_Yeh}', "");
Expect(0, 2221, '\p{Is_Joining_Group=-Rohingya_Yeh}', "");
Expect(1, 2221, '\p{^Is_Joining_Group=-Rohingya_Yeh}', "");
Expect(1, 2221, '\P{Is_Joining_Group=-Rohingya_Yeh}', "");
Expect(0, 2221, '\P{^Is_Joining_Group=-Rohingya_Yeh}', "");
Error('\p{Is_Jg=_-rohingya_Yeh/a/}');
Error('\P{Is_Jg=_-rohingya_Yeh/a/}');
Expect(1, 2220, '\p{Is_Jg=rohingyayeh}', "");
Expect(0, 2220, '\p{^Is_Jg=rohingyayeh}', "");
Expect(0, 2220, '\P{Is_Jg=rohingyayeh}', "");
Expect(1, 2220, '\P{^Is_Jg=rohingyayeh}', "");
Expect(0, 2221, '\p{Is_Jg=rohingyayeh}', "");
Expect(1, 2221, '\p{^Is_Jg=rohingyayeh}', "");
Expect(1, 2221, '\P{Is_Jg=rohingyayeh}', "");
Expect(0, 2221, '\P{^Is_Jg=rohingyayeh}', "");
Expect(1, 2220, '\p{Is_Jg:   Rohingya_Yeh}', "");
Expect(0, 2220, '\p{^Is_Jg:   Rohingya_Yeh}', "");
Expect(0, 2220, '\P{Is_Jg:   Rohingya_Yeh}', "");
Expect(1, 2220, '\P{^Is_Jg:   Rohingya_Yeh}', "");
Expect(0, 2221, '\p{Is_Jg:   Rohingya_Yeh}', "");
Expect(1, 2221, '\p{^Is_Jg:   Rohingya_Yeh}', "");
Expect(1, 2221, '\P{Is_Jg:   Rohingya_Yeh}', "");
Expect(0, 2221, '\P{^Is_Jg:   Rohingya_Yeh}', "");
Error('\p{Joining_Group:_Swash_kaf/a/}');
Error('\P{Joining_Group:_Swash_kaf/a/}');
Expect(1, 1706, '\p{Joining_Group=swashkaf}', "");
Expect(0, 1706, '\p{^Joining_Group=swashkaf}', "");
Expect(0, 1706, '\P{Joining_Group=swashkaf}', "");
Expect(1, 1706, '\P{^Joining_Group=swashkaf}', "");
Expect(0, 1707, '\p{Joining_Group=swashkaf}', "");
Expect(1, 1707, '\p{^Joining_Group=swashkaf}', "");
Expect(1, 1707, '\P{Joining_Group=swashkaf}', "");
Expect(0, 1707, '\P{^Joining_Group=swashkaf}', "");
Expect(1, 1706, '\p{Joining_Group=	-Swash_KAF}', "");
Expect(0, 1706, '\p{^Joining_Group=	-Swash_KAF}', "");
Expect(0, 1706, '\P{Joining_Group=	-Swash_KAF}', "");
Expect(1, 1706, '\P{^Joining_Group=	-Swash_KAF}', "");
Expect(0, 1707, '\p{Joining_Group=	-Swash_KAF}', "");
Expect(1, 1707, '\p{^Joining_Group=	-Swash_KAF}', "");
Expect(1, 1707, '\P{Joining_Group=	-Swash_KAF}', "");
Expect(0, 1707, '\P{^Joining_Group=	-Swash_KAF}', "");
Error('\p{Jg= SWASH_KAF:=}');
Error('\P{Jg= SWASH_KAF:=}');
Expect(1, 1706, '\p{Jg=swashkaf}', "");
Expect(0, 1706, '\p{^Jg=swashkaf}', "");
Expect(0, 1706, '\P{Jg=swashkaf}', "");
Expect(1, 1706, '\P{^Jg=swashkaf}', "");
Expect(0, 1707, '\p{Jg=swashkaf}', "");
Expect(1, 1707, '\p{^Jg=swashkaf}', "");
Expect(1, 1707, '\P{Jg=swashkaf}', "");
Expect(0, 1707, '\P{^Jg=swashkaf}', "");
Expect(1, 1706, '\p{Jg:_-Swash_KAF}', "");
Expect(0, 1706, '\p{^Jg:_-Swash_KAF}', "");
Expect(0, 1706, '\P{Jg:_-Swash_KAF}', "");
Expect(1, 1706, '\P{^Jg:_-Swash_KAF}', "");
Expect(0, 1707, '\p{Jg:_-Swash_KAF}', "");
Expect(1, 1707, '\p{^Jg:_-Swash_KAF}', "");
Expect(1, 1707, '\P{Jg:_-Swash_KAF}', "");
Expect(0, 1707, '\P{^Jg:_-Swash_KAF}', "");
Error('\p{Is_Joining_Group= _swash_Kaf:=}');
Error('\P{Is_Joining_Group= _swash_Kaf:=}');
Expect(1, 1706, '\p{Is_Joining_Group:swashkaf}', "");
Expect(0, 1706, '\p{^Is_Joining_Group:swashkaf}', "");
Expect(0, 1706, '\P{Is_Joining_Group:swashkaf}', "");
Expect(1, 1706, '\P{^Is_Joining_Group:swashkaf}', "");
Expect(0, 1707, '\p{Is_Joining_Group:swashkaf}', "");
Expect(1, 1707, '\p{^Is_Joining_Group:swashkaf}', "");
Expect(1, 1707, '\P{Is_Joining_Group:swashkaf}', "");
Expect(0, 1707, '\P{^Is_Joining_Group:swashkaf}', "");
Expect(1, 1706, '\p{Is_Joining_Group=		swash_kaf}', "");
Expect(0, 1706, '\p{^Is_Joining_Group=		swash_kaf}', "");
Expect(0, 1706, '\P{Is_Joining_Group=		swash_kaf}', "");
Expect(1, 1706, '\P{^Is_Joining_Group=		swash_kaf}', "");
Expect(0, 1707, '\p{Is_Joining_Group=		swash_kaf}', "");
Expect(1, 1707, '\p{^Is_Joining_Group=		swash_kaf}', "");
Expect(1, 1707, '\P{Is_Joining_Group=		swash_kaf}', "");
Expect(0, 1707, '\P{^Is_Joining_Group=		swash_kaf}', "");
Error('\p{Is_Jg=_	Swash_Kaf:=}');
Error('\P{Is_Jg=_	Swash_Kaf:=}');
Expect(1, 1706, '\p{Is_Jg=swashkaf}', "");
Expect(0, 1706, '\p{^Is_Jg=swashkaf}', "");
Expect(0, 1706, '\P{Is_Jg=swashkaf}', "");
Expect(1, 1706, '\P{^Is_Jg=swashkaf}', "");
Expect(0, 1707, '\p{Is_Jg=swashkaf}', "");
Expect(1, 1707, '\p{^Is_Jg=swashkaf}', "");
Expect(1, 1707, '\P{Is_Jg=swashkaf}', "");
Expect(0, 1707, '\P{^Is_Jg=swashkaf}', "");
Expect(1, 1706, '\p{Is_Jg=- SWASH_Kaf}', "");
Expect(0, 1706, '\p{^Is_Jg=- SWASH_Kaf}', "");
Expect(0, 1706, '\P{Is_Jg=- SWASH_Kaf}', "");
Expect(1, 1706, '\P{^Is_Jg=- SWASH_Kaf}', "");
Expect(0, 1707, '\p{Is_Jg=- SWASH_Kaf}', "");
Expect(1, 1707, '\p{^Is_Jg=- SWASH_Kaf}', "");
Expect(1, 1707, '\P{Is_Jg=- SWASH_Kaf}', "");
Expect(0, 1707, '\P{^Is_Jg=- SWASH_Kaf}', "");
Error('\p{Joining_Group=_	seen:=}');
Error('\P{Joining_Group=_	seen:=}');
Expect(1, 1918, '\p{Joining_Group=seen}', "");
Expect(0, 1918, '\p{^Joining_Group=seen}', "");
Expect(0, 1918, '\P{Joining_Group=seen}', "");
Expect(1, 1918, '\P{^Joining_Group=seen}', "");
Expect(0, 1919, '\p{Joining_Group=seen}', "");
Expect(1, 1919, '\p{^Joining_Group=seen}', "");
Expect(1, 1919, '\P{Joining_Group=seen}', "");
Expect(0, 1919, '\P{^Joining_Group=seen}', "");
Expect(1, 1918, '\p{Joining_Group=	seen}', "");
Expect(0, 1918, '\p{^Joining_Group=	seen}', "");
Expect(0, 1918, '\P{Joining_Group=	seen}', "");
Expect(1, 1918, '\P{^Joining_Group=	seen}', "");
Expect(0, 1919, '\p{Joining_Group=	seen}', "");
Expect(1, 1919, '\p{^Joining_Group=	seen}', "");
Expect(1, 1919, '\P{Joining_Group=	seen}', "");
Expect(0, 1919, '\P{^Joining_Group=	seen}', "");
Error('\p{Jg=seen:=}');
Error('\P{Jg=seen:=}');
Expect(1, 1918, '\p{Jg=seen}', "");
Expect(0, 1918, '\p{^Jg=seen}', "");
Expect(0, 1918, '\P{Jg=seen}', "");
Expect(1, 1918, '\P{^Jg=seen}', "");
Expect(0, 1919, '\p{Jg=seen}', "");
Expect(1, 1919, '\p{^Jg=seen}', "");
Expect(1, 1919, '\P{Jg=seen}', "");
Expect(0, 1919, '\P{^Jg=seen}', "");
Expect(1, 1918, '\p{Jg=_	SEEN}', "");
Expect(0, 1918, '\p{^Jg=_	SEEN}', "");
Expect(0, 1918, '\P{Jg=_	SEEN}', "");
Expect(1, 1918, '\P{^Jg=_	SEEN}', "");
Expect(0, 1919, '\p{Jg=_	SEEN}', "");
Expect(1, 1919, '\p{^Jg=_	SEEN}', "");
Expect(1, 1919, '\P{Jg=_	SEEN}', "");
Expect(0, 1919, '\P{^Jg=_	SEEN}', "");
Error('\p{Is_Joining_Group:   _/a/SEEN}');
Error('\P{Is_Joining_Group:   _/a/SEEN}');
Expect(1, 1918, '\p{Is_Joining_Group=seen}', "");
Expect(0, 1918, '\p{^Is_Joining_Group=seen}', "");
Expect(0, 1918, '\P{Is_Joining_Group=seen}', "");
Expect(1, 1918, '\P{^Is_Joining_Group=seen}', "");
Expect(0, 1919, '\p{Is_Joining_Group=seen}', "");
Expect(1, 1919, '\p{^Is_Joining_Group=seen}', "");
Expect(1, 1919, '\P{Is_Joining_Group=seen}', "");
Expect(0, 1919, '\P{^Is_Joining_Group=seen}', "");
Expect(1, 1918, '\p{Is_Joining_Group:   -	SEEN}', "");
Expect(0, 1918, '\p{^Is_Joining_Group:   -	SEEN}', "");
Expect(0, 1918, '\P{Is_Joining_Group:   -	SEEN}', "");
Expect(1, 1918, '\P{^Is_Joining_Group:   -	SEEN}', "");
Expect(0, 1919, '\p{Is_Joining_Group:   -	SEEN}', "");
Expect(1, 1919, '\p{^Is_Joining_Group:   -	SEEN}', "");
Expect(1, 1919, '\P{Is_Joining_Group:   -	SEEN}', "");
Expect(0, 1919, '\P{^Is_Joining_Group:   -	SEEN}', "");
Error('\p{Is_Jg=_-Seen:=}');
Error('\P{Is_Jg=_-Seen:=}');
Expect(1, 1918, '\p{Is_Jg:seen}', "");
Expect(0, 1918, '\p{^Is_Jg:seen}', "");
Expect(0, 1918, '\P{Is_Jg:seen}', "");
Expect(1, 1918, '\P{^Is_Jg:seen}', "");
Expect(0, 1919, '\p{Is_Jg:seen}', "");
Expect(1, 1919, '\p{^Is_Jg:seen}', "");
Expect(1, 1919, '\P{Is_Jg:seen}', "");
Expect(0, 1919, '\P{^Is_Jg:seen}', "");
Expect(1, 1918, '\p{Is_Jg=Seen}', "");
Expect(0, 1918, '\p{^Is_Jg=Seen}', "");
Expect(0, 1918, '\P{Is_Jg=Seen}', "");
Expect(1, 1918, '\P{^Is_Jg=Seen}', "");
Expect(0, 1919, '\p{Is_Jg=Seen}', "");
Expect(1, 1919, '\p{^Is_Jg=Seen}', "");
Expect(1, 1919, '\P{Is_Jg=Seen}', "");
Expect(0, 1919, '\P{^Is_Jg=Seen}', "");
Error('\p{Joining_Group:	/a/		Beth}');
Error('\P{Joining_Group:	/a/		Beth}');
Expect(1, 1837, '\p{Joining_Group=beth}', "");
Expect(0, 1837, '\p{^Joining_Group=beth}', "");
Expect(0, 1837, '\P{Joining_Group=beth}', "");
Expect(1, 1837, '\P{^Joining_Group=beth}', "");
Expect(0, 1838, '\p{Joining_Group=beth}', "");
Expect(1, 1838, '\p{^Joining_Group=beth}', "");
Expect(1, 1838, '\P{Joining_Group=beth}', "");
Expect(0, 1838, '\P{^Joining_Group=beth}', "");
Expect(1, 1837, '\p{Joining_Group=__BETH}', "");
Expect(0, 1837, '\p{^Joining_Group=__BETH}', "");
Expect(0, 1837, '\P{Joining_Group=__BETH}', "");
Expect(1, 1837, '\P{^Joining_Group=__BETH}', "");
Expect(0, 1838, '\p{Joining_Group=__BETH}', "");
Expect(1, 1838, '\p{^Joining_Group=__BETH}', "");
Expect(1, 1838, '\P{Joining_Group=__BETH}', "");
Expect(0, 1838, '\P{^Joining_Group=__BETH}', "");
Error('\p{Jg=-BETH/a/}');
Error('\P{Jg=-BETH/a/}');
Expect(1, 1837, '\p{Jg=beth}', "");
Expect(0, 1837, '\p{^Jg=beth}', "");
Expect(0, 1837, '\P{Jg=beth}', "");
Expect(1, 1837, '\P{^Jg=beth}', "");
Expect(0, 1838, '\p{Jg=beth}', "");
Expect(1, 1838, '\p{^Jg=beth}', "");
Expect(1, 1838, '\P{Jg=beth}', "");
Expect(0, 1838, '\P{^Jg=beth}', "");
Expect(1, 1837, '\p{Jg=-BETH}', "");
Expect(0, 1837, '\p{^Jg=-BETH}', "");
Expect(0, 1837, '\P{Jg=-BETH}', "");
Expect(1, 1837, '\P{^Jg=-BETH}', "");
Expect(0, 1838, '\p{Jg=-BETH}', "");
Expect(1, 1838, '\p{^Jg=-BETH}', "");
Expect(1, 1838, '\P{Jg=-BETH}', "");
Expect(0, 1838, '\P{^Jg=-BETH}', "");
Error('\p{Is_Joining_Group:	_beth/a/}');
Error('\P{Is_Joining_Group:	_beth/a/}');
Expect(1, 1837, '\p{Is_Joining_Group=beth}', "");
Expect(0, 1837, '\p{^Is_Joining_Group=beth}', "");
Expect(0, 1837, '\P{Is_Joining_Group=beth}', "");
Expect(1, 1837, '\P{^Is_Joining_Group=beth}', "");
Expect(0, 1838, '\p{Is_Joining_Group=beth}', "");
Expect(1, 1838, '\p{^Is_Joining_Group=beth}', "");
Expect(1, 1838, '\P{Is_Joining_Group=beth}', "");
Expect(0, 1838, '\P{^Is_Joining_Group=beth}', "");
Expect(1, 1837, '\p{Is_Joining_Group=-_Beth}', "");
Expect(0, 1837, '\p{^Is_Joining_Group=-_Beth}', "");
Expect(0, 1837, '\P{Is_Joining_Group=-_Beth}', "");
Expect(1, 1837, '\P{^Is_Joining_Group=-_Beth}', "");
Expect(0, 1838, '\p{Is_Joining_Group=-_Beth}', "");
Expect(1, 1838, '\p{^Is_Joining_Group=-_Beth}', "");
Expect(1, 1838, '\P{Is_Joining_Group=-_Beth}', "");
Expect(0, 1838, '\P{^Is_Joining_Group=-_Beth}', "");
Error('\p{Is_Jg=:=	_Beth}');
Error('\P{Is_Jg=:=	_Beth}');
Expect(1, 1837, '\p{Is_Jg=beth}', "");
Expect(0, 1837, '\p{^Is_Jg=beth}', "");
Expect(0, 1837, '\P{Is_Jg=beth}', "");
Expect(1, 1837, '\P{^Is_Jg=beth}', "");
Expect(0, 1838, '\p{Is_Jg=beth}', "");
Expect(1, 1838, '\p{^Is_Jg=beth}', "");
Expect(1, 1838, '\P{Is_Jg=beth}', "");
Expect(0, 1838, '\P{^Is_Jg=beth}', "");
Expect(1, 1837, '\p{Is_Jg=-	BETH}', "");
Expect(0, 1837, '\p{^Is_Jg=-	BETH}', "");
Expect(0, 1837, '\P{Is_Jg=-	BETH}', "");
Expect(1, 1837, '\P{^Is_Jg=-	BETH}', "");
Expect(0, 1838, '\p{Is_Jg=-	BETH}', "");
Expect(1, 1838, '\p{^Is_Jg=-	BETH}', "");
Expect(1, 1838, '\P{Is_Jg=-	BETH}', "");
Expect(0, 1838, '\P{^Is_Jg=-	BETH}', "");
Error('\p{Joining_Group= -Semkath:=}');
Error('\P{Joining_Group= -Semkath:=}');
Expect(1, 1827, '\p{Joining_Group=semkath}', "");
Expect(0, 1827, '\p{^Joining_Group=semkath}', "");
Expect(0, 1827, '\P{Joining_Group=semkath}', "");
Expect(1, 1827, '\P{^Joining_Group=semkath}', "");
Expect(0, 1828, '\p{Joining_Group=semkath}', "");
Expect(1, 1828, '\p{^Joining_Group=semkath}', "");
Expect(1, 1828, '\P{Joining_Group=semkath}', "");
Expect(0, 1828, '\P{^Joining_Group=semkath}', "");
Expect(1, 1827, '\p{Joining_Group:   _	Semkath}', "");
Expect(0, 1827, '\p{^Joining_Group:   _	Semkath}', "");
Expect(0, 1827, '\P{Joining_Group:   _	Semkath}', "");
Expect(1, 1827, '\P{^Joining_Group:   _	Semkath}', "");
Expect(0, 1828, '\p{Joining_Group:   _	Semkath}', "");
Expect(1, 1828, '\p{^Joining_Group:   _	Semkath}', "");
Expect(1, 1828, '\P{Joining_Group:   _	Semkath}', "");
Expect(0, 1828, '\P{^Joining_Group:   _	Semkath}', "");
Error('\p{Jg:/a/-_semkath}');
Error('\P{Jg:/a/-_semkath}');
Expect(1, 1827, '\p{Jg=semkath}', "");
Expect(0, 1827, '\p{^Jg=semkath}', "");
Expect(0, 1827, '\P{Jg=semkath}', "");
Expect(1, 1827, '\P{^Jg=semkath}', "");
Expect(0, 1828, '\p{Jg=semkath}', "");
Expect(1, 1828, '\p{^Jg=semkath}', "");
Expect(1, 1828, '\P{Jg=semkath}', "");
Expect(0, 1828, '\P{^Jg=semkath}', "");
Expect(1, 1827, '\p{Jg= -Semkath}', "");
Expect(0, 1827, '\p{^Jg= -Semkath}', "");
Expect(0, 1827, '\P{Jg= -Semkath}', "");
Expect(1, 1827, '\P{^Jg= -Semkath}', "");
Expect(0, 1828, '\p{Jg= -Semkath}', "");
Expect(1, 1828, '\p{^Jg= -Semkath}', "");
Expect(1, 1828, '\P{Jg= -Semkath}', "");
Expect(0, 1828, '\P{^Jg= -Semkath}', "");
Error('\p{Is_Joining_Group: :=Semkath}');
Error('\P{Is_Joining_Group: :=Semkath}');
Expect(1, 1827, '\p{Is_Joining_Group=semkath}', "");
Expect(0, 1827, '\p{^Is_Joining_Group=semkath}', "");
Expect(0, 1827, '\P{Is_Joining_Group=semkath}', "");
Expect(1, 1827, '\P{^Is_Joining_Group=semkath}', "");
Expect(0, 1828, '\p{Is_Joining_Group=semkath}', "");
Expect(1, 1828, '\p{^Is_Joining_Group=semkath}', "");
Expect(1, 1828, '\P{Is_Joining_Group=semkath}', "");
Expect(0, 1828, '\P{^Is_Joining_Group=semkath}', "");
Expect(1, 1827, '\p{Is_Joining_Group= Semkath}', "");
Expect(0, 1827, '\p{^Is_Joining_Group= Semkath}', "");
Expect(0, 1827, '\P{Is_Joining_Group= Semkath}', "");
Expect(1, 1827, '\P{^Is_Joining_Group= Semkath}', "");
Expect(0, 1828, '\p{Is_Joining_Group= Semkath}', "");
Expect(1, 1828, '\p{^Is_Joining_Group= Semkath}', "");
Expect(1, 1828, '\P{Is_Joining_Group= Semkath}', "");
Expect(0, 1828, '\P{^Is_Joining_Group= Semkath}', "");
Error('\p{Is_Jg=	_Semkath/a/}');
Error('\P{Is_Jg=	_Semkath/a/}');
Expect(1, 1827, '\p{Is_Jg:   semkath}', "");
Expect(0, 1827, '\p{^Is_Jg:   semkath}', "");
Expect(0, 1827, '\P{Is_Jg:   semkath}', "");
Expect(1, 1827, '\P{^Is_Jg:   semkath}', "");
Expect(0, 1828, '\p{Is_Jg:   semkath}', "");
Expect(1, 1828, '\p{^Is_Jg:   semkath}', "");
Expect(1, 1828, '\P{Is_Jg:   semkath}', "");
Expect(0, 1828, '\P{^Is_Jg:   semkath}', "");
Expect(1, 1827, '\p{Is_Jg=_-Semkath}', "");
Expect(0, 1827, '\p{^Is_Jg=_-Semkath}', "");
Expect(0, 1827, '\P{Is_Jg=_-Semkath}', "");
Expect(1, 1827, '\P{^Is_Jg=_-Semkath}', "");
Expect(0, 1828, '\p{Is_Jg=_-Semkath}', "");
Expect(1, 1828, '\p{^Is_Jg=_-Semkath}', "");
Expect(1, 1828, '\P{Is_Jg=_-Semkath}', "");
Expect(0, 1828, '\P{^Is_Jg=_-Semkath}', "");
Error('\p{Joining_Group=/a/	hah}');
Error('\P{Joining_Group=/a/	hah}');
Expect(1, 2210, '\p{Joining_Group=hah}', "");
Expect(0, 2210, '\p{^Joining_Group=hah}', "");
Expect(0, 2210, '\P{Joining_Group=hah}', "");
Expect(1, 2210, '\P{^Joining_Group=hah}', "");
Expect(0, 2211, '\p{Joining_Group=hah}', "");
Expect(1, 2211, '\p{^Joining_Group=hah}', "");
Expect(1, 2211, '\P{Joining_Group=hah}', "");
Expect(0, 2211, '\P{^Joining_Group=hah}', "");
Expect(1, 2210, '\p{Joining_Group=-Hah}', "");
Expect(0, 2210, '\p{^Joining_Group=-Hah}', "");
Expect(0, 2210, '\P{Joining_Group=-Hah}', "");
Expect(1, 2210, '\P{^Joining_Group=-Hah}', "");
Expect(0, 2211, '\p{Joining_Group=-Hah}', "");
Expect(1, 2211, '\p{^Joining_Group=-Hah}', "");
Expect(1, 2211, '\P{Joining_Group=-Hah}', "");
Expect(0, 2211, '\P{^Joining_Group=-Hah}', "");
Error('\p{Jg=	HAH/a/}');
Error('\P{Jg=	HAH/a/}');
Expect(1, 2210, '\p{Jg=hah}', "");
Expect(0, 2210, '\p{^Jg=hah}', "");
Expect(0, 2210, '\P{Jg=hah}', "");
Expect(1, 2210, '\P{^Jg=hah}', "");
Expect(0, 2211, '\p{Jg=hah}', "");
Expect(1, 2211, '\p{^Jg=hah}', "");
Expect(1, 2211, '\P{Jg=hah}', "");
Expect(0, 2211, '\P{^Jg=hah}', "");
Expect(1, 2210, '\p{Jg=	 Hah}', "");
Expect(0, 2210, '\p{^Jg=	 Hah}', "");
Expect(0, 2210, '\P{Jg=	 Hah}', "");
Expect(1, 2210, '\P{^Jg=	 Hah}', "");
Expect(0, 2211, '\p{Jg=	 Hah}', "");
Expect(1, 2211, '\p{^Jg=	 Hah}', "");
Expect(1, 2211, '\P{Jg=	 Hah}', "");
Expect(0, 2211, '\P{^Jg=	 Hah}', "");
Error('\p{Is_Joining_Group=:=--Hah}');
Error('\P{Is_Joining_Group=:=--Hah}');
Expect(1, 2210, '\p{Is_Joining_Group=hah}', "");
Expect(0, 2210, '\p{^Is_Joining_Group=hah}', "");
Expect(0, 2210, '\P{Is_Joining_Group=hah}', "");
Expect(1, 2210, '\P{^Is_Joining_Group=hah}', "");
Expect(0, 2211, '\p{Is_Joining_Group=hah}', "");
Expect(1, 2211, '\p{^Is_Joining_Group=hah}', "");
Expect(1, 2211, '\P{Is_Joining_Group=hah}', "");
Expect(0, 2211, '\P{^Is_Joining_Group=hah}', "");
Expect(1, 2210, '\p{Is_Joining_Group= hah}', "");
Expect(0, 2210, '\p{^Is_Joining_Group= hah}', "");
Expect(0, 2210, '\P{Is_Joining_Group= hah}', "");
Expect(1, 2210, '\P{^Is_Joining_Group= hah}', "");
Expect(0, 2211, '\p{Is_Joining_Group= hah}', "");
Expect(1, 2211, '\p{^Is_Joining_Group= hah}', "");
Expect(1, 2211, '\P{Is_Joining_Group= hah}', "");
Expect(0, 2211, '\P{^Is_Joining_Group= hah}', "");
Error('\p{Is_Jg=/a/	hah}');
Error('\P{Is_Jg=/a/	hah}');
Expect(1, 2210, '\p{Is_Jg=hah}', "");
Expect(0, 2210, '\p{^Is_Jg=hah}', "");
Expect(0, 2210, '\P{Is_Jg=hah}', "");
Expect(1, 2210, '\P{^Is_Jg=hah}', "");
Expect(0, 2211, '\p{Is_Jg=hah}', "");
Expect(1, 2211, '\p{^Is_Jg=hah}', "");
Expect(1, 2211, '\P{Is_Jg=hah}', "");
Expect(0, 2211, '\P{^Is_Jg=hah}', "");
Expect(1, 2210, '\p{Is_Jg=--Hah}', "");
Expect(0, 2210, '\p{^Is_Jg=--Hah}', "");
Expect(0, 2210, '\P{Is_Jg=--Hah}', "");
Expect(1, 2210, '\P{^Is_Jg=--Hah}', "");
Expect(0, 2211, '\p{Is_Jg=--Hah}', "");
Expect(1, 2211, '\p{^Is_Jg=--Hah}', "");
Expect(1, 2211, '\P{Is_Jg=--Hah}', "");
Expect(0, 2211, '\P{^Is_Jg=--Hah}', "");
Error('\p{Joining_Group= Syriac_Waw:=}');
Error('\P{Joining_Group= Syriac_Waw:=}');
Expect(1, 1816, '\p{Joining_Group=syriacwaw}', "");
Expect(0, 1816, '\p{^Joining_Group=syriacwaw}', "");
Expect(0, 1816, '\P{Joining_Group=syriacwaw}', "");
Expect(1, 1816, '\P{^Joining_Group=syriacwaw}', "");
Expect(0, 1817, '\p{Joining_Group=syriacwaw}', "");
Expect(1, 1817, '\p{^Joining_Group=syriacwaw}', "");
Expect(1, 1817, '\P{Joining_Group=syriacwaw}', "");
Expect(0, 1817, '\P{^Joining_Group=syriacwaw}', "");
Expect(1, 1816, '\p{Joining_Group=	Syriac_Waw}', "");
Expect(0, 1816, '\p{^Joining_Group=	Syriac_Waw}', "");
Expect(0, 1816, '\P{Joining_Group=	Syriac_Waw}', "");
Expect(1, 1816, '\P{^Joining_Group=	Syriac_Waw}', "");
Expect(0, 1817, '\p{Joining_Group=	Syriac_Waw}', "");
Expect(1, 1817, '\p{^Joining_Group=	Syriac_Waw}', "");
Expect(1, 1817, '\P{Joining_Group=	Syriac_Waw}', "");
Expect(0, 1817, '\P{^Joining_Group=	Syriac_Waw}', "");
Error('\p{Jg=/a/-	syriac_Waw}');
Error('\P{Jg=/a/-	syriac_Waw}');
Expect(1, 1816, '\p{Jg=syriacwaw}', "");
Expect(0, 1816, '\p{^Jg=syriacwaw}', "");
Expect(0, 1816, '\P{Jg=syriacwaw}', "");
Expect(1, 1816, '\P{^Jg=syriacwaw}', "");
Expect(0, 1817, '\p{Jg=syriacwaw}', "");
Expect(1, 1817, '\p{^Jg=syriacwaw}', "");
Expect(1, 1817, '\P{Jg=syriacwaw}', "");
Expect(0, 1817, '\P{^Jg=syriacwaw}', "");
Expect(1, 1816, '\p{Jg=__Syriac_Waw}', "");
Expect(0, 1816, '\p{^Jg=__Syriac_Waw}', "");
Expect(0, 1816, '\P{Jg=__Syriac_Waw}', "");
Expect(1, 1816, '\P{^Jg=__Syriac_Waw}', "");
Expect(0, 1817, '\p{Jg=__Syriac_Waw}', "");
Expect(1, 1817, '\p{^Jg=__Syriac_Waw}', "");
Expect(1, 1817, '\P{Jg=__Syriac_Waw}', "");
Expect(0, 1817, '\P{^Jg=__Syriac_Waw}', "");
Error('\p{Is_Joining_Group=:=_Syriac_waw}');
Error('\P{Is_Joining_Group=:=_Syriac_waw}');
Expect(1, 1816, '\p{Is_Joining_Group=syriacwaw}', "");
Expect(0, 1816, '\p{^Is_Joining_Group=syriacwaw}', "");
Expect(0, 1816, '\P{Is_Joining_Group=syriacwaw}', "");
Expect(1, 1816, '\P{^Is_Joining_Group=syriacwaw}', "");
Expect(0, 1817, '\p{Is_Joining_Group=syriacwaw}', "");
Expect(1, 1817, '\p{^Is_Joining_Group=syriacwaw}', "");
Expect(1, 1817, '\P{Is_Joining_Group=syriacwaw}', "");
Expect(0, 1817, '\P{^Is_Joining_Group=syriacwaw}', "");
Expect(1, 1816, '\p{Is_Joining_Group= Syriac_WAW}', "");
Expect(0, 1816, '\p{^Is_Joining_Group= Syriac_WAW}', "");
Expect(0, 1816, '\P{Is_Joining_Group= Syriac_WAW}', "");
Expect(1, 1816, '\P{^Is_Joining_Group= Syriac_WAW}', "");
Expect(0, 1817, '\p{Is_Joining_Group= Syriac_WAW}', "");
Expect(1, 1817, '\p{^Is_Joining_Group= Syriac_WAW}', "");
Expect(1, 1817, '\P{Is_Joining_Group= Syriac_WAW}', "");
Expect(0, 1817, '\P{^Is_Joining_Group= Syriac_WAW}', "");
Error('\p{Is_Jg=/a/SYRIAC_WAW}');
Error('\P{Is_Jg=/a/SYRIAC_WAW}');
Expect(1, 1816, '\p{Is_Jg=syriacwaw}', "");
Expect(0, 1816, '\p{^Is_Jg=syriacwaw}', "");
Expect(0, 1816, '\P{Is_Jg=syriacwaw}', "");
Expect(1, 1816, '\P{^Is_Jg=syriacwaw}', "");
Expect(0, 1817, '\p{Is_Jg=syriacwaw}', "");
Expect(1, 1817, '\p{^Is_Jg=syriacwaw}', "");
Expect(1, 1817, '\P{Is_Jg=syriacwaw}', "");
Expect(0, 1817, '\P{^Is_Jg=syriacwaw}', "");
Expect(1, 1816, '\p{Is_Jg= _syriac_Waw}', "");
Expect(0, 1816, '\p{^Is_Jg= _syriac_Waw}', "");
Expect(0, 1816, '\P{Is_Jg= _syriac_Waw}', "");
Expect(1, 1816, '\P{^Is_Jg= _syriac_Waw}', "");
Expect(0, 1817, '\p{Is_Jg= _syriac_Waw}', "");
Expect(1, 1817, '\p{^Is_Jg= _syriac_Waw}', "");
Expect(1, 1817, '\P{Is_Jg= _syriac_Waw}', "");
Expect(0, 1817, '\P{^Is_Jg= _syriac_Waw}', "");
Error('\p{Joining_Group=/a/	_Taw}');
Error('\P{Joining_Group=/a/	_Taw}');
Expect(1, 1836, '\p{Joining_Group=taw}', "");
Expect(0, 1836, '\p{^Joining_Group=taw}', "");
Expect(0, 1836, '\P{Joining_Group=taw}', "");
Expect(1, 1836, '\P{^Joining_Group=taw}', "");
Expect(0, 1837, '\p{Joining_Group=taw}', "");
Expect(1, 1837, '\p{^Joining_Group=taw}', "");
Expect(1, 1837, '\P{Joining_Group=taw}', "");
Expect(0, 1837, '\P{^Joining_Group=taw}', "");
Expect(1, 1836, '\p{Joining_Group=		taw}', "");
Expect(0, 1836, '\p{^Joining_Group=		taw}', "");
Expect(0, 1836, '\P{Joining_Group=		taw}', "");
Expect(1, 1836, '\P{^Joining_Group=		taw}', "");
Expect(0, 1837, '\p{Joining_Group=		taw}', "");
Expect(1, 1837, '\p{^Joining_Group=		taw}', "");
Expect(1, 1837, '\P{Joining_Group=		taw}', "");
Expect(0, 1837, '\P{^Joining_Group=		taw}', "");
Error('\p{Jg= 	Taw:=}');
Error('\P{Jg= 	Taw:=}');
Expect(1, 1836, '\p{Jg=taw}', "");
Expect(0, 1836, '\p{^Jg=taw}', "");
Expect(0, 1836, '\P{Jg=taw}', "");
Expect(1, 1836, '\P{^Jg=taw}', "");
Expect(0, 1837, '\p{Jg=taw}', "");
Expect(1, 1837, '\p{^Jg=taw}', "");
Expect(1, 1837, '\P{Jg=taw}', "");
Expect(0, 1837, '\P{^Jg=taw}', "");
Expect(1, 1836, '\p{Jg=-_taw}', "");
Expect(0, 1836, '\p{^Jg=-_taw}', "");
Expect(0, 1836, '\P{Jg=-_taw}', "");
Expect(1, 1836, '\P{^Jg=-_taw}', "");
Expect(0, 1837, '\p{Jg=-_taw}', "");
Expect(1, 1837, '\p{^Jg=-_taw}', "");
Expect(1, 1837, '\P{Jg=-_taw}', "");
Expect(0, 1837, '\P{^Jg=-_taw}', "");
Error('\p{Is_Joining_Group=:=	_TAW}');
Error('\P{Is_Joining_Group=:=	_TAW}');
Expect(1, 1836, '\p{Is_Joining_Group=taw}', "");
Expect(0, 1836, '\p{^Is_Joining_Group=taw}', "");
Expect(0, 1836, '\P{Is_Joining_Group=taw}', "");
Expect(1, 1836, '\P{^Is_Joining_Group=taw}', "");
Expect(0, 1837, '\p{Is_Joining_Group=taw}', "");
Expect(1, 1837, '\p{^Is_Joining_Group=taw}', "");
Expect(1, 1837, '\P{Is_Joining_Group=taw}', "");
Expect(0, 1837, '\P{^Is_Joining_Group=taw}', "");
Expect(1, 1836, '\p{Is_Joining_Group=	_TAW}', "");
Expect(0, 1836, '\p{^Is_Joining_Group=	_TAW}', "");
Expect(0, 1836, '\P{Is_Joining_Group=	_TAW}', "");
Expect(1, 1836, '\P{^Is_Joining_Group=	_TAW}', "");
Expect(0, 1837, '\p{Is_Joining_Group=	_TAW}', "");
Expect(1, 1837, '\p{^Is_Joining_Group=	_TAW}', "");
Expect(1, 1837, '\P{Is_Joining_Group=	_TAW}', "");
Expect(0, 1837, '\P{^Is_Joining_Group=	_TAW}', "");
Error('\p{Is_Jg= -taw/a/}');
Error('\P{Is_Jg= -taw/a/}');
Expect(1, 1836, '\p{Is_Jg:	taw}', "");
Expect(0, 1836, '\p{^Is_Jg:	taw}', "");
Expect(0, 1836, '\P{Is_Jg:	taw}', "");
Expect(1, 1836, '\P{^Is_Jg:	taw}', "");
Expect(0, 1837, '\p{Is_Jg:	taw}', "");
Expect(1, 1837, '\p{^Is_Jg:	taw}', "");
Expect(1, 1837, '\P{Is_Jg:	taw}', "");
Expect(0, 1837, '\P{^Is_Jg:	taw}', "");
Expect(1, 1836, '\p{Is_Jg=	 Taw}', "");
Expect(0, 1836, '\p{^Is_Jg=	 Taw}', "");
Expect(0, 1836, '\P{Is_Jg=	 Taw}', "");
Expect(1, 1836, '\P{^Is_Jg=	 Taw}', "");
Expect(0, 1837, '\p{Is_Jg=	 Taw}', "");
Expect(1, 1837, '\p{^Is_Jg=	 Taw}', "");
Expect(1, 1837, '\P{Is_Jg=	 Taw}', "");
Expect(0, 1837, '\P{^Is_Jg=	 Taw}', "");
Error('\p{Joining_Group=	_Final_Semkath/a/}');
Error('\P{Joining_Group=	_Final_Semkath/a/}');
Expect(1, 1828, '\p{Joining_Group=finalsemkath}', "");
Expect(0, 1828, '\p{^Joining_Group=finalsemkath}', "");
Expect(0, 1828, '\P{Joining_Group=finalsemkath}', "");
Expect(1, 1828, '\P{^Joining_Group=finalsemkath}', "");
Expect(0, 1829, '\p{Joining_Group=finalsemkath}', "");
Expect(1, 1829, '\p{^Joining_Group=finalsemkath}', "");
Expect(1, 1829, '\P{Joining_Group=finalsemkath}', "");
Expect(0, 1829, '\P{^Joining_Group=finalsemkath}', "");
Expect(1, 1828, '\p{Joining_Group=_-final_semkath}', "");
Expect(0, 1828, '\p{^Joining_Group=_-final_semkath}', "");
Expect(0, 1828, '\P{Joining_Group=_-final_semkath}', "");
Expect(1, 1828, '\P{^Joining_Group=_-final_semkath}', "");
Expect(0, 1829, '\p{Joining_Group=_-final_semkath}', "");
Expect(1, 1829, '\p{^Joining_Group=_-final_semkath}', "");
Expect(1, 1829, '\P{Joining_Group=_-final_semkath}', "");
Expect(0, 1829, '\P{^Joining_Group=_-final_semkath}', "");
Error('\p{Jg:   -/a/Final_semkath}');
Error('\P{Jg:   -/a/Final_semkath}');
Expect(1, 1828, '\p{Jg=finalsemkath}', "");
Expect(0, 1828, '\p{^Jg=finalsemkath}', "");
Expect(0, 1828, '\P{Jg=finalsemkath}', "");
Expect(1, 1828, '\P{^Jg=finalsemkath}', "");
Expect(0, 1829, '\p{Jg=finalsemkath}', "");
Expect(1, 1829, '\p{^Jg=finalsemkath}', "");
Expect(1, 1829, '\P{Jg=finalsemkath}', "");
Expect(0, 1829, '\P{^Jg=finalsemkath}', "");
Expect(1, 1828, '\p{Jg=  Final_Semkath}', "");
Expect(0, 1828, '\p{^Jg=  Final_Semkath}', "");
Expect(0, 1828, '\P{Jg=  Final_Semkath}', "");
Expect(1, 1828, '\P{^Jg=  Final_Semkath}', "");
Expect(0, 1829, '\p{Jg=  Final_Semkath}', "");
Expect(1, 1829, '\p{^Jg=  Final_Semkath}', "");
Expect(1, 1829, '\P{Jg=  Final_Semkath}', "");
Expect(0, 1829, '\P{^Jg=  Final_Semkath}', "");
Error('\p{Is_Joining_Group=/a/final_SEMKATH}');
Error('\P{Is_Joining_Group=/a/final_SEMKATH}');
Expect(1, 1828, '\p{Is_Joining_Group=finalsemkath}', "");
Expect(0, 1828, '\p{^Is_Joining_Group=finalsemkath}', "");
Expect(0, 1828, '\P{Is_Joining_Group=finalsemkath}', "");
Expect(1, 1828, '\P{^Is_Joining_Group=finalsemkath}', "");
Expect(0, 1829, '\p{Is_Joining_Group=finalsemkath}', "");
Expect(1, 1829, '\p{^Is_Joining_Group=finalsemkath}', "");
Expect(1, 1829, '\P{Is_Joining_Group=finalsemkath}', "");
Expect(0, 1829, '\P{^Is_Joining_Group=finalsemkath}', "");
Expect(1, 1828, '\p{Is_Joining_Group=FINAL_Semkath}', "");
Expect(0, 1828, '\p{^Is_Joining_Group=FINAL_Semkath}', "");
Expect(0, 1828, '\P{Is_Joining_Group=FINAL_Semkath}', "");
Expect(1, 1828, '\P{^Is_Joining_Group=FINAL_Semkath}', "");
Expect(0, 1829, '\p{Is_Joining_Group=FINAL_Semkath}', "");
Expect(1, 1829, '\p{^Is_Joining_Group=FINAL_Semkath}', "");
Expect(1, 1829, '\P{Is_Joining_Group=FINAL_Semkath}', "");
Expect(0, 1829, '\P{^Is_Joining_Group=FINAL_Semkath}', "");
Error('\p{Is_Jg:- Final_Semkath:=}');
Error('\P{Is_Jg:- Final_Semkath:=}');
Expect(1, 1828, '\p{Is_Jg=finalsemkath}', "");
Expect(0, 1828, '\p{^Is_Jg=finalsemkath}', "");
Expect(0, 1828, '\P{Is_Jg=finalsemkath}', "");
Expect(1, 1828, '\P{^Is_Jg=finalsemkath}', "");
Expect(0, 1829, '\p{Is_Jg=finalsemkath}', "");
Expect(1, 1829, '\p{^Is_Jg=finalsemkath}', "");
Expect(1, 1829, '\P{Is_Jg=finalsemkath}', "");
Expect(0, 1829, '\P{^Is_Jg=finalsemkath}', "");
Expect(1, 1828, '\p{Is_Jg=-Final_Semkath}', "");
Expect(0, 1828, '\p{^Is_Jg=-Final_Semkath}', "");
Expect(0, 1828, '\P{Is_Jg=-Final_Semkath}', "");
Expect(1, 1828, '\P{^Is_Jg=-Final_Semkath}', "");
Expect(0, 1829, '\p{Is_Jg=-Final_Semkath}', "");
Expect(1, 1829, '\p{^Is_Jg=-Final_Semkath}', "");
Expect(1, 1829, '\P{Is_Jg=-Final_Semkath}', "");
Expect(0, 1829, '\P{^Is_Jg=-Final_Semkath}', "");
Error('\p{Joining_Group=_ Knotted_Heh:=}');
Error('\P{Joining_Group=_ Knotted_Heh:=}');
Expect(1, 1791, '\p{Joining_Group=knottedheh}', "");
Expect(0, 1791, '\p{^Joining_Group=knottedheh}', "");
Expect(0, 1791, '\P{Joining_Group=knottedheh}', "");
Expect(1, 1791, '\P{^Joining_Group=knottedheh}', "");
Expect(0, 1792, '\p{Joining_Group=knottedheh}', "");
Expect(1, 1792, '\p{^Joining_Group=knottedheh}', "");
Expect(1, 1792, '\P{Joining_Group=knottedheh}', "");
Expect(0, 1792, '\P{^Joining_Group=knottedheh}', "");
Expect(1, 1791, '\p{Joining_Group= _Knotted_Heh}', "");
Expect(0, 1791, '\p{^Joining_Group= _Knotted_Heh}', "");
Expect(0, 1791, '\P{Joining_Group= _Knotted_Heh}', "");
Expect(1, 1791, '\P{^Joining_Group= _Knotted_Heh}', "");
Expect(0, 1792, '\p{Joining_Group= _Knotted_Heh}', "");
Expect(1, 1792, '\p{^Joining_Group= _Knotted_Heh}', "");
Expect(1, 1792, '\P{Joining_Group= _Knotted_Heh}', "");
Expect(0, 1792, '\P{^Joining_Group= _Knotted_Heh}', "");
Error('\p{Jg=--Knotted_Heh/a/}');
Error('\P{Jg=--Knotted_Heh/a/}');
Expect(1, 1791, '\p{Jg=knottedheh}', "");
Expect(0, 1791, '\p{^Jg=knottedheh}', "");
Expect(0, 1791, '\P{Jg=knottedheh}', "");
Expect(1, 1791, '\P{^Jg=knottedheh}', "");
Expect(0, 1792, '\p{Jg=knottedheh}', "");
Expect(1, 1792, '\p{^Jg=knottedheh}', "");
Expect(1, 1792, '\P{Jg=knottedheh}', "");
Expect(0, 1792, '\P{^Jg=knottedheh}', "");
Expect(1, 1791, '\p{Jg=--knotted_Heh}', "");
Expect(0, 1791, '\p{^Jg=--knotted_Heh}', "");
Expect(0, 1791, '\P{Jg=--knotted_Heh}', "");
Expect(1, 1791, '\P{^Jg=--knotted_Heh}', "");
Expect(0, 1792, '\p{Jg=--knotted_Heh}', "");
Expect(1, 1792, '\p{^Jg=--knotted_Heh}', "");
Expect(1, 1792, '\P{Jg=--knotted_Heh}', "");
Expect(0, 1792, '\P{^Jg=--knotted_Heh}', "");
Error('\p{Is_Joining_Group=/a/	_Knotted_Heh}');
Error('\P{Is_Joining_Group=/a/	_Knotted_Heh}');
Expect(1, 1791, '\p{Is_Joining_Group:   knottedheh}', "");
Expect(0, 1791, '\p{^Is_Joining_Group:   knottedheh}', "");
Expect(0, 1791, '\P{Is_Joining_Group:   knottedheh}', "");
Expect(1, 1791, '\P{^Is_Joining_Group:   knottedheh}', "");
Expect(0, 1792, '\p{Is_Joining_Group:   knottedheh}', "");
Expect(1, 1792, '\p{^Is_Joining_Group:   knottedheh}', "");
Expect(1, 1792, '\P{Is_Joining_Group:   knottedheh}', "");
Expect(0, 1792, '\P{^Is_Joining_Group:   knottedheh}', "");
Expect(1, 1791, '\p{Is_Joining_Group:   	_Knotted_heh}', "");
Expect(0, 1791, '\p{^Is_Joining_Group:   	_Knotted_heh}', "");
Expect(0, 1791, '\P{Is_Joining_Group:   	_Knotted_heh}', "");
Expect(1, 1791, '\P{^Is_Joining_Group:   	_Knotted_heh}', "");
Expect(0, 1792, '\p{Is_Joining_Group:   	_Knotted_heh}', "");
Expect(1, 1792, '\p{^Is_Joining_Group:   	_Knotted_heh}', "");
Expect(1, 1792, '\P{Is_Joining_Group:   	_Knotted_heh}', "");
Expect(0, 1792, '\P{^Is_Joining_Group:   	_Knotted_heh}', "");
Error('\p{Is_Jg=/a/knotted_Heh}');
Error('\P{Is_Jg=/a/knotted_Heh}');
Expect(1, 1791, '\p{Is_Jg=knottedheh}', "");
Expect(0, 1791, '\p{^Is_Jg=knottedheh}', "");
Expect(0, 1791, '\P{Is_Jg=knottedheh}', "");
Expect(1, 1791, '\P{^Is_Jg=knottedheh}', "");
Expect(0, 1792, '\p{Is_Jg=knottedheh}', "");
Expect(1, 1792, '\p{^Is_Jg=knottedheh}', "");
Expect(1, 1792, '\P{Is_Jg=knottedheh}', "");
Expect(0, 1792, '\P{^Is_Jg=knottedheh}', "");
Expect(1, 1791, '\p{Is_Jg: 	Knotted_Heh}', "");
Expect(0, 1791, '\p{^Is_Jg: 	Knotted_Heh}', "");
Expect(0, 1791, '\P{Is_Jg: 	Knotted_Heh}', "");
Expect(1, 1791, '\P{^Is_Jg: 	Knotted_Heh}', "");
Expect(0, 1792, '\p{Is_Jg: 	Knotted_Heh}', "");
Expect(1, 1792, '\p{^Is_Jg: 	Knotted_Heh}', "");
Expect(1, 1792, '\P{Is_Jg: 	Knotted_Heh}', "");
Expect(0, 1792, '\P{^Is_Jg: 	Knotted_Heh}', "");
Error('\p{Joining_Group=/a/ kaf}');
Error('\P{Joining_Group=/a/ kaf}');
Expect(1, 1919, '\p{Joining_Group:kaf}', "");
Expect(0, 1919, '\p{^Joining_Group:kaf}', "");
Expect(0, 1919, '\P{Joining_Group:kaf}', "");
Expect(1, 1919, '\P{^Joining_Group:kaf}', "");
Expect(0, 1920, '\p{Joining_Group:kaf}', "");
Expect(1, 1920, '\p{^Joining_Group:kaf}', "");
Expect(1, 1920, '\P{Joining_Group:kaf}', "");
Expect(0, 1920, '\P{^Joining_Group:kaf}', "");
Expect(1, 1919, '\p{Joining_Group=_	KAF}', "");
Expect(0, 1919, '\p{^Joining_Group=_	KAF}', "");
Expect(0, 1919, '\P{Joining_Group=_	KAF}', "");
Expect(1, 1919, '\P{^Joining_Group=_	KAF}', "");
Expect(0, 1920, '\p{Joining_Group=_	KAF}', "");
Expect(1, 1920, '\p{^Joining_Group=_	KAF}', "");
Expect(1, 1920, '\P{Joining_Group=_	KAF}', "");
Expect(0, 1920, '\P{^Joining_Group=_	KAF}', "");
Error('\p{Jg=/a/Kaf}');
Error('\P{Jg=/a/Kaf}');
Expect(1, 1919, '\p{Jg=kaf}', "");
Expect(0, 1919, '\p{^Jg=kaf}', "");
Expect(0, 1919, '\P{Jg=kaf}', "");
Expect(1, 1919, '\P{^Jg=kaf}', "");
Expect(0, 1920, '\p{Jg=kaf}', "");
Expect(1, 1920, '\p{^Jg=kaf}', "");
Expect(1, 1920, '\P{Jg=kaf}', "");
Expect(0, 1920, '\P{^Jg=kaf}', "");
Expect(1, 1919, '\p{Jg=__Kaf}', "");
Expect(0, 1919, '\p{^Jg=__Kaf}', "");
Expect(0, 1919, '\P{Jg=__Kaf}', "");
Expect(1, 1919, '\P{^Jg=__Kaf}', "");
Expect(0, 1920, '\p{Jg=__Kaf}', "");
Expect(1, 1920, '\p{^Jg=__Kaf}', "");
Expect(1, 1920, '\P{Jg=__Kaf}', "");
Expect(0, 1920, '\P{^Jg=__Kaf}', "");
Error('\p{Is_Joining_Group=_Kaf/a/}');
Error('\P{Is_Joining_Group=_Kaf/a/}');
Expect(1, 1919, '\p{Is_Joining_Group=kaf}', "");
Expect(0, 1919, '\p{^Is_Joining_Group=kaf}', "");
Expect(0, 1919, '\P{Is_Joining_Group=kaf}', "");
Expect(1, 1919, '\P{^Is_Joining_Group=kaf}', "");
Expect(0, 1920, '\p{Is_Joining_Group=kaf}', "");
Expect(1, 1920, '\p{^Is_Joining_Group=kaf}', "");
Expect(1, 1920, '\P{Is_Joining_Group=kaf}', "");
Expect(0, 1920, '\P{^Is_Joining_Group=kaf}', "");
Expect(1, 1919, '\p{Is_Joining_Group=- KAF}', "");
Expect(0, 1919, '\p{^Is_Joining_Group=- KAF}', "");
Expect(0, 1919, '\P{Is_Joining_Group=- KAF}', "");
Expect(1, 1919, '\P{^Is_Joining_Group=- KAF}', "");
Expect(0, 1920, '\p{Is_Joining_Group=- KAF}', "");
Expect(1, 1920, '\p{^Is_Joining_Group=- KAF}', "");
Expect(1, 1920, '\P{Is_Joining_Group=- KAF}', "");
Expect(0, 1920, '\P{^Is_Joining_Group=- KAF}', "");
Error('\p{Is_Jg= _Kaf/a/}');
Error('\P{Is_Jg= _Kaf/a/}');
Expect(1, 1919, '\p{Is_Jg=kaf}', "");
Expect(0, 1919, '\p{^Is_Jg=kaf}', "");
Expect(0, 1919, '\P{Is_Jg=kaf}', "");
Expect(1, 1919, '\P{^Is_Jg=kaf}', "");
Expect(0, 1920, '\p{Is_Jg=kaf}', "");
Expect(1, 1920, '\p{^Is_Jg=kaf}', "");
Expect(1, 1920, '\P{Is_Jg=kaf}', "");
Expect(0, 1920, '\P{^Is_Jg=kaf}', "");
Expect(1, 1919, '\p{Is_Jg=-Kaf}', "");
Expect(0, 1919, '\p{^Is_Jg=-Kaf}', "");
Expect(0, 1919, '\P{Is_Jg=-Kaf}', "");
Expect(1, 1919, '\P{^Is_Jg=-Kaf}', "");
Expect(0, 1920, '\p{Is_Jg=-Kaf}', "");
Expect(1, 1920, '\p{^Is_Jg=-Kaf}', "");
Expect(1, 1920, '\P{Is_Jg=-Kaf}', "");
Expect(0, 1920, '\P{^Is_Jg=-Kaf}', "");
Error('\p{Joining_Group:		FE:=}');
Error('\P{Joining_Group:		FE:=}');
Expect(1, 1871, '\p{Joining_Group=fe}', "");
Expect(0, 1871, '\p{^Joining_Group=fe}', "");
Expect(0, 1871, '\P{Joining_Group=fe}', "");
Expect(1, 1871, '\P{^Joining_Group=fe}', "");
Expect(0, 1872, '\p{Joining_Group=fe}', "");
Expect(1, 1872, '\p{^Joining_Group=fe}', "");
Expect(1, 1872, '\P{Joining_Group=fe}', "");
Expect(0, 1872, '\P{^Joining_Group=fe}', "");
Expect(1, 1871, '\p{Joining_Group=	 Fe}', "");
Expect(0, 1871, '\p{^Joining_Group=	 Fe}', "");
Expect(0, 1871, '\P{Joining_Group=	 Fe}', "");
Expect(1, 1871, '\P{^Joining_Group=	 Fe}', "");
Expect(0, 1872, '\p{Joining_Group=	 Fe}', "");
Expect(1, 1872, '\p{^Joining_Group=	 Fe}', "");
Expect(1, 1872, '\P{Joining_Group=	 Fe}', "");
Expect(0, 1872, '\P{^Joining_Group=	 Fe}', "");
Error('\p{Jg:/a/	_fe}');
Error('\P{Jg:/a/	_fe}');
Expect(1, 1871, '\p{Jg=fe}', "");
Expect(0, 1871, '\p{^Jg=fe}', "");
Expect(0, 1871, '\P{Jg=fe}', "");
Expect(1, 1871, '\P{^Jg=fe}', "");
Expect(0, 1872, '\p{Jg=fe}', "");
Expect(1, 1872, '\p{^Jg=fe}', "");
Expect(1, 1872, '\P{Jg=fe}', "");
Expect(0, 1872, '\P{^Jg=fe}', "");
Expect(1, 1871, '\p{Jg=-	fe}', "");
Expect(0, 1871, '\p{^Jg=-	fe}', "");
Expect(0, 1871, '\P{Jg=-	fe}', "");
Expect(1, 1871, '\P{^Jg=-	fe}', "");
Expect(0, 1872, '\p{Jg=-	fe}', "");
Expect(1, 1872, '\p{^Jg=-	fe}', "");
Expect(1, 1872, '\P{Jg=-	fe}', "");
Expect(0, 1872, '\P{^Jg=-	fe}', "");
Error('\p{Is_Joining_Group:   :=-_Fe}');
Error('\P{Is_Joining_Group:   :=-_Fe}');
Expect(1, 1871, '\p{Is_Joining_Group: fe}', "");
Expect(0, 1871, '\p{^Is_Joining_Group: fe}', "");
Expect(0, 1871, '\P{Is_Joining_Group: fe}', "");
Expect(1, 1871, '\P{^Is_Joining_Group: fe}', "");
Expect(0, 1872, '\p{Is_Joining_Group: fe}', "");
Expect(1, 1872, '\p{^Is_Joining_Group: fe}', "");
Expect(1, 1872, '\P{Is_Joining_Group: fe}', "");
Expect(0, 1872, '\P{^Is_Joining_Group: fe}', "");
Expect(1, 1871, '\p{Is_Joining_Group=- fe}', "");
Expect(0, 1871, '\p{^Is_Joining_Group=- fe}', "");
Expect(0, 1871, '\P{Is_Joining_Group=- fe}', "");
Expect(1, 1871, '\P{^Is_Joining_Group=- fe}', "");
Expect(0, 1872, '\p{Is_Joining_Group=- fe}', "");
Expect(1, 1872, '\p{^Is_Joining_Group=- fe}', "");
Expect(1, 1872, '\P{Is_Joining_Group=- fe}', "");
Expect(0, 1872, '\P{^Is_Joining_Group=- fe}', "");
Error('\p{Is_Jg=:=	Fe}');
Error('\P{Is_Jg=:=	Fe}');
Expect(1, 1871, '\p{Is_Jg=fe}', "");
Expect(0, 1871, '\p{^Is_Jg=fe}', "");
Expect(0, 1871, '\P{Is_Jg=fe}', "");
Expect(1, 1871, '\P{^Is_Jg=fe}', "");
Expect(0, 1872, '\p{Is_Jg=fe}', "");
Expect(1, 1872, '\p{^Is_Jg=fe}', "");
Expect(1, 1872, '\P{Is_Jg=fe}', "");
Expect(0, 1872, '\P{^Is_Jg=fe}', "");
Expect(1, 1871, '\p{Is_Jg= _FE}', "");
Expect(0, 1871, '\p{^Is_Jg= _FE}', "");
Expect(0, 1871, '\P{Is_Jg= _FE}', "");
Expect(1, 1871, '\P{^Is_Jg= _FE}', "");
Expect(0, 1872, '\p{Is_Jg= _FE}', "");
Expect(1, 1872, '\p{^Is_Jg= _FE}', "");
Expect(1, 1872, '\P{Is_Jg= _FE}', "");
Expect(0, 1872, '\P{^Is_Jg= _FE}', "");
Error('\p{Joining_Group=-Hamza_On_Heh_Goal:=}');
Error('\P{Joining_Group=-Hamza_On_Heh_Goal:=}');
Expect(1, 1731, '\p{Joining_Group=hamzaonhehgoal}', "");
Expect(0, 1731, '\p{^Joining_Group=hamzaonhehgoal}', "");
Expect(0, 1731, '\P{Joining_Group=hamzaonhehgoal}', "");
Expect(1, 1731, '\P{^Joining_Group=hamzaonhehgoal}', "");
Expect(0, 1732, '\p{Joining_Group=hamzaonhehgoal}', "");
Expect(1, 1732, '\p{^Joining_Group=hamzaonhehgoal}', "");
Expect(1, 1732, '\P{Joining_Group=hamzaonhehgoal}', "");
Expect(0, 1732, '\P{^Joining_Group=hamzaonhehgoal}', "");
Expect(1, 1731, '\p{Joining_Group= Hamza_On_Heh_Goal}', "");
Expect(0, 1731, '\p{^Joining_Group= Hamza_On_Heh_Goal}', "");
Expect(0, 1731, '\P{Joining_Group= Hamza_On_Heh_Goal}', "");
Expect(1, 1731, '\P{^Joining_Group= Hamza_On_Heh_Goal}', "");
Expect(0, 1732, '\p{Joining_Group= Hamza_On_Heh_Goal}', "");
Expect(1, 1732, '\p{^Joining_Group= Hamza_On_Heh_Goal}', "");
Expect(1, 1732, '\P{Joining_Group= Hamza_On_Heh_Goal}', "");
Expect(0, 1732, '\P{^Joining_Group= Hamza_On_Heh_Goal}', "");
Error('\p{Jg:   	Teh_Marbuta_goal:=}');
Error('\P{Jg:   	Teh_Marbuta_goal:=}');
Expect(1, 1731, '\p{Jg=tehmarbutagoal}', "");
Expect(0, 1731, '\p{^Jg=tehmarbutagoal}', "");
Expect(0, 1731, '\P{Jg=tehmarbutagoal}', "");
Expect(1, 1731, '\P{^Jg=tehmarbutagoal}', "");
Expect(0, 1732, '\p{Jg=tehmarbutagoal}', "");
Expect(1, 1732, '\p{^Jg=tehmarbutagoal}', "");
Expect(1, 1732, '\P{Jg=tehmarbutagoal}', "");
Expect(0, 1732, '\P{^Jg=tehmarbutagoal}', "");
Expect(1, 1731, '\p{Jg=__teh_Marbuta_goal}', "");
Expect(0, 1731, '\p{^Jg=__teh_Marbuta_goal}', "");
Expect(0, 1731, '\P{Jg=__teh_Marbuta_goal}', "");
Expect(1, 1731, '\P{^Jg=__teh_Marbuta_goal}', "");
Expect(0, 1732, '\p{Jg=__teh_Marbuta_goal}', "");
Expect(1, 1732, '\p{^Jg=__teh_Marbuta_goal}', "");
Expect(1, 1732, '\P{Jg=__teh_Marbuta_goal}', "");
Expect(0, 1732, '\P{^Jg=__teh_Marbuta_goal}', "");
Error('\p{Is_Joining_Group=  HAMZA_on_HEH_Goal:=}');
Error('\P{Is_Joining_Group=  HAMZA_on_HEH_Goal:=}');
Expect(1, 1731, '\p{Is_Joining_Group=hamzaonhehgoal}', "");
Expect(0, 1731, '\p{^Is_Joining_Group=hamzaonhehgoal}', "");
Expect(0, 1731, '\P{Is_Joining_Group=hamzaonhehgoal}', "");
Expect(1, 1731, '\P{^Is_Joining_Group=hamzaonhehgoal}', "");
Expect(0, 1732, '\p{Is_Joining_Group=hamzaonhehgoal}', "");
Expect(1, 1732, '\p{^Is_Joining_Group=hamzaonhehgoal}', "");
Expect(1, 1732, '\P{Is_Joining_Group=hamzaonhehgoal}', "");
Expect(0, 1732, '\P{^Is_Joining_Group=hamzaonhehgoal}', "");
Expect(1, 1731, '\p{Is_Joining_Group:HAMZA_On_HEH_GOAL}', "");
Expect(0, 1731, '\p{^Is_Joining_Group:HAMZA_On_HEH_GOAL}', "");
Expect(0, 1731, '\P{Is_Joining_Group:HAMZA_On_HEH_GOAL}', "");
Expect(1, 1731, '\P{^Is_Joining_Group:HAMZA_On_HEH_GOAL}', "");
Expect(0, 1732, '\p{Is_Joining_Group:HAMZA_On_HEH_GOAL}', "");
Expect(1, 1732, '\p{^Is_Joining_Group:HAMZA_On_HEH_GOAL}', "");
Expect(1, 1732, '\P{Is_Joining_Group:HAMZA_On_HEH_GOAL}', "");
Expect(0, 1732, '\P{^Is_Joining_Group:HAMZA_On_HEH_GOAL}', "");
Error('\p{Is_Jg=/a/Teh_Marbuta_Goal}');
Error('\P{Is_Jg=/a/Teh_Marbuta_Goal}');
Expect(1, 1731, '\p{Is_Jg=tehmarbutagoal}', "");
Expect(0, 1731, '\p{^Is_Jg=tehmarbutagoal}', "");
Expect(0, 1731, '\P{Is_Jg=tehmarbutagoal}', "");
Expect(1, 1731, '\P{^Is_Jg=tehmarbutagoal}', "");
Expect(0, 1732, '\p{Is_Jg=tehmarbutagoal}', "");
Expect(1, 1732, '\p{^Is_Jg=tehmarbutagoal}', "");
Expect(1, 1732, '\P{Is_Jg=tehmarbutagoal}', "");
Expect(0, 1732, '\P{^Is_Jg=tehmarbutagoal}', "");
Expect(1, 1731, '\p{Is_Jg: _-Teh_marbuta_Goal}', "");
Expect(0, 1731, '\p{^Is_Jg: _-Teh_marbuta_Goal}', "");
Expect(0, 1731, '\P{Is_Jg: _-Teh_marbuta_Goal}', "");
Expect(1, 1731, '\P{^Is_Jg: _-Teh_marbuta_Goal}', "");
Expect(0, 1732, '\p{Is_Jg: _-Teh_marbuta_Goal}', "");
Expect(1, 1732, '\p{^Is_Jg: _-Teh_marbuta_Goal}', "");
Expect(1, 1732, '\P{Is_Jg: _-Teh_marbuta_Goal}', "");
Expect(0, 1732, '\P{^Is_Jg: _-Teh_marbuta_Goal}', "");
Error('\p{Joining_Group= 	ALAPH:=}');
Error('\P{Joining_Group= 	ALAPH:=}');
Expect(1, 1808, '\p{Joining_Group=alaph}', "");
Expect(0, 1808, '\p{^Joining_Group=alaph}', "");
Expect(0, 1808, '\P{Joining_Group=alaph}', "");
Expect(1, 1808, '\P{^Joining_Group=alaph}', "");
Expect(0, 1809, '\p{Joining_Group=alaph}', "");
Expect(1, 1809, '\p{^Joining_Group=alaph}', "");
Expect(1, 1809, '\P{Joining_Group=alaph}', "");
Expect(0, 1809, '\P{^Joining_Group=alaph}', "");
Expect(1, 1808, '\p{Joining_Group=_ Alaph}', "");
Expect(0, 1808, '\p{^Joining_Group=_ Alaph}', "");
Expect(0, 1808, '\P{Joining_Group=_ Alaph}', "");
Expect(1, 1808, '\P{^Joining_Group=_ Alaph}', "");
Expect(0, 1809, '\p{Joining_Group=_ Alaph}', "");
Expect(1, 1809, '\p{^Joining_Group=_ Alaph}', "");
Expect(1, 1809, '\P{Joining_Group=_ Alaph}', "");
Expect(0, 1809, '\P{^Joining_Group=_ Alaph}', "");
Error('\p{Jg=_:=Alaph}');
Error('\P{Jg=_:=Alaph}');
Expect(1, 1808, '\p{Jg=alaph}', "");
Expect(0, 1808, '\p{^Jg=alaph}', "");
Expect(0, 1808, '\P{Jg=alaph}', "");
Expect(1, 1808, '\P{^Jg=alaph}', "");
Expect(0, 1809, '\p{Jg=alaph}', "");
Expect(1, 1809, '\p{^Jg=alaph}', "");
Expect(1, 1809, '\P{Jg=alaph}', "");
Expect(0, 1809, '\P{^Jg=alaph}', "");
Expect(1, 1808, '\p{Jg=	 alaph}', "");
Expect(0, 1808, '\p{^Jg=	 alaph}', "");
Expect(0, 1808, '\P{Jg=	 alaph}', "");
Expect(1, 1808, '\P{^Jg=	 alaph}', "");
Expect(0, 1809, '\p{Jg=	 alaph}', "");
Expect(1, 1809, '\p{^Jg=	 alaph}', "");
Expect(1, 1809, '\P{Jg=	 alaph}', "");
Expect(0, 1809, '\P{^Jg=	 alaph}', "");
Error('\p{Is_Joining_Group=:= _alaph}');
Error('\P{Is_Joining_Group=:= _alaph}');
Expect(1, 1808, '\p{Is_Joining_Group=alaph}', "");
Expect(0, 1808, '\p{^Is_Joining_Group=alaph}', "");
Expect(0, 1808, '\P{Is_Joining_Group=alaph}', "");
Expect(1, 1808, '\P{^Is_Joining_Group=alaph}', "");
Expect(0, 1809, '\p{Is_Joining_Group=alaph}', "");
Expect(1, 1809, '\p{^Is_Joining_Group=alaph}', "");
Expect(1, 1809, '\P{Is_Joining_Group=alaph}', "");
Expect(0, 1809, '\P{^Is_Joining_Group=alaph}', "");
Expect(1, 1808, '\p{Is_Joining_Group:_Alaph}', "");
Expect(0, 1808, '\p{^Is_Joining_Group:_Alaph}', "");
Expect(0, 1808, '\P{Is_Joining_Group:_Alaph}', "");
Expect(1, 1808, '\P{^Is_Joining_Group:_Alaph}', "");
Expect(0, 1809, '\p{Is_Joining_Group:_Alaph}', "");
Expect(1, 1809, '\p{^Is_Joining_Group:_Alaph}', "");
Expect(1, 1809, '\P{Is_Joining_Group:_Alaph}', "");
Expect(0, 1809, '\P{^Is_Joining_Group:_Alaph}', "");
Error('\p{Is_Jg=/a/ALAPH}');
Error('\P{Is_Jg=/a/ALAPH}');
Expect(1, 1808, '\p{Is_Jg=alaph}', "");
Expect(0, 1808, '\p{^Is_Jg=alaph}', "");
Expect(0, 1808, '\P{Is_Jg=alaph}', "");
Expect(1, 1808, '\P{^Is_Jg=alaph}', "");
Expect(0, 1809, '\p{Is_Jg=alaph}', "");
Expect(1, 1809, '\p{^Is_Jg=alaph}', "");
Expect(1, 1809, '\P{Is_Jg=alaph}', "");
Expect(0, 1809, '\P{^Is_Jg=alaph}', "");
Expect(1, 1808, '\p{Is_Jg:   _-ALAPH}', "");
Expect(0, 1808, '\p{^Is_Jg:   _-ALAPH}', "");
Expect(0, 1808, '\P{Is_Jg:   _-ALAPH}', "");
Expect(1, 1808, '\P{^Is_Jg:   _-ALAPH}', "");
Expect(0, 1809, '\p{Is_Jg:   _-ALAPH}', "");
Expect(1, 1809, '\p{^Is_Jg:   _-ALAPH}', "");
Expect(1, 1809, '\P{Is_Jg:   _-ALAPH}', "");
Expect(0, 1809, '\P{^Is_Jg:   _-ALAPH}', "");
Error('\p{Joining_Group=/a/ 	lamadh}');
Error('\P{Joining_Group=/a/ 	lamadh}');
Expect(1, 1824, '\p{Joining_Group=lamadh}', "");
Expect(0, 1824, '\p{^Joining_Group=lamadh}', "");
Expect(0, 1824, '\P{Joining_Group=lamadh}', "");
Expect(1, 1824, '\P{^Joining_Group=lamadh}', "");
Expect(0, 1825, '\p{Joining_Group=lamadh}', "");
Expect(1, 1825, '\p{^Joining_Group=lamadh}', "");
Expect(1, 1825, '\P{Joining_Group=lamadh}', "");
Expect(0, 1825, '\P{^Joining_Group=lamadh}', "");
Expect(1, 1824, '\p{Joining_Group=  lamadh}', "");
Expect(0, 1824, '\p{^Joining_Group=  lamadh}', "");
Expect(0, 1824, '\P{Joining_Group=  lamadh}', "");
Expect(1, 1824, '\P{^Joining_Group=  lamadh}', "");
Expect(0, 1825, '\p{Joining_Group=  lamadh}', "");
Expect(1, 1825, '\p{^Joining_Group=  lamadh}', "");
Expect(1, 1825, '\P{Joining_Group=  lamadh}', "");
Expect(0, 1825, '\P{^Joining_Group=  lamadh}', "");
Error('\p{Jg= 	Lamadh:=}');
Error('\P{Jg= 	Lamadh:=}');
Expect(1, 1824, '\p{Jg=lamadh}', "");
Expect(0, 1824, '\p{^Jg=lamadh}', "");
Expect(0, 1824, '\P{Jg=lamadh}', "");
Expect(1, 1824, '\P{^Jg=lamadh}', "");
Expect(0, 1825, '\p{Jg=lamadh}', "");
Expect(1, 1825, '\p{^Jg=lamadh}', "");
Expect(1, 1825, '\P{Jg=lamadh}', "");
Expect(0, 1825, '\P{^Jg=lamadh}', "");
Expect(1, 1824, '\p{Jg=_Lamadh}', "");
Expect(0, 1824, '\p{^Jg=_Lamadh}', "");
Expect(0, 1824, '\P{Jg=_Lamadh}', "");
Expect(1, 1824, '\P{^Jg=_Lamadh}', "");
Expect(0, 1825, '\p{Jg=_Lamadh}', "");
Expect(1, 1825, '\p{^Jg=_Lamadh}', "");
Expect(1, 1825, '\P{Jg=_Lamadh}', "");
Expect(0, 1825, '\P{^Jg=_Lamadh}', "");
Error('\p{Is_Joining_Group=/a/	Lamadh}');
Error('\P{Is_Joining_Group=/a/	Lamadh}');
Expect(1, 1824, '\p{Is_Joining_Group=lamadh}', "");
Expect(0, 1824, '\p{^Is_Joining_Group=lamadh}', "");
Expect(0, 1824, '\P{Is_Joining_Group=lamadh}', "");
Expect(1, 1824, '\P{^Is_Joining_Group=lamadh}', "");
Expect(0, 1825, '\p{Is_Joining_Group=lamadh}', "");
Expect(1, 1825, '\p{^Is_Joining_Group=lamadh}', "");
Expect(1, 1825, '\P{Is_Joining_Group=lamadh}', "");
Expect(0, 1825, '\P{^Is_Joining_Group=lamadh}', "");
Expect(1, 1824, '\p{Is_Joining_Group:   	-Lamadh}', "");
Expect(0, 1824, '\p{^Is_Joining_Group:   	-Lamadh}', "");
Expect(0, 1824, '\P{Is_Joining_Group:   	-Lamadh}', "");
Expect(1, 1824, '\P{^Is_Joining_Group:   	-Lamadh}', "");
Expect(0, 1825, '\p{Is_Joining_Group:   	-Lamadh}', "");
Expect(1, 1825, '\p{^Is_Joining_Group:   	-Lamadh}', "");
Expect(1, 1825, '\P{Is_Joining_Group:   	-Lamadh}', "");
Expect(0, 1825, '\P{^Is_Joining_Group:   	-Lamadh}', "");
Error('\p{Is_Jg=	/a/Lamadh}');
Error('\P{Is_Jg=	/a/Lamadh}');
Expect(1, 1824, '\p{Is_Jg=lamadh}', "");
Expect(0, 1824, '\p{^Is_Jg=lamadh}', "");
Expect(0, 1824, '\P{Is_Jg=lamadh}', "");
Expect(1, 1824, '\P{^Is_Jg=lamadh}', "");
Expect(0, 1825, '\p{Is_Jg=lamadh}', "");
Expect(1, 1825, '\p{^Is_Jg=lamadh}', "");
Expect(1, 1825, '\P{Is_Jg=lamadh}', "");
Expect(0, 1825, '\P{^Is_Jg=lamadh}', "");
Expect(1, 1824, '\p{Is_Jg:   --lamadh}', "");
Expect(0, 1824, '\p{^Is_Jg:   --lamadh}', "");
Expect(0, 1824, '\P{Is_Jg:   --lamadh}', "");
Expect(1, 1824, '\P{^Is_Jg:   --lamadh}', "");
Expect(0, 1825, '\p{Is_Jg:   --lamadh}', "");
Expect(1, 1825, '\p{^Is_Jg:   --lamadh}', "");
Expect(1, 1825, '\P{Is_Jg:   --lamadh}', "");
Expect(0, 1825, '\P{^Is_Jg:   --lamadh}', "");
Error('\p{Joining_Group=_/a/e}');
Error('\P{Joining_Group=_/a/e}');
Expect(1, 1829, '\p{Joining_Group=e}', "");
Expect(0, 1829, '\p{^Joining_Group=e}', "");
Expect(0, 1829, '\P{Joining_Group=e}', "");
Expect(1, 1829, '\P{^Joining_Group=e}', "");
Expect(0, 1830, '\p{Joining_Group=e}', "");
Expect(1, 1830, '\p{^Joining_Group=e}', "");
Expect(1, 1830, '\P{Joining_Group=e}', "");
Expect(0, 1830, '\P{^Joining_Group=e}', "");
Expect(1, 1829, '\p{Joining_Group=	-E}', "");
Expect(0, 1829, '\p{^Joining_Group=	-E}', "");
Expect(0, 1829, '\P{Joining_Group=	-E}', "");
Expect(1, 1829, '\P{^Joining_Group=	-E}', "");
Expect(0, 1830, '\p{Joining_Group=	-E}', "");
Expect(1, 1830, '\p{^Joining_Group=	-E}', "");
Expect(1, 1830, '\P{Joining_Group=	-E}', "");
Expect(0, 1830, '\P{^Joining_Group=	-E}', "");
Error('\p{Jg=/a/_ E}');
Error('\P{Jg=/a/_ E}');
Expect(1, 1829, '\p{Jg=e}', "");
Expect(0, 1829, '\p{^Jg=e}', "");
Expect(0, 1829, '\P{Jg=e}', "");
Expect(1, 1829, '\P{^Jg=e}', "");
Expect(0, 1830, '\p{Jg=e}', "");
Expect(1, 1830, '\p{^Jg=e}', "");
Expect(1, 1830, '\P{Jg=e}', "");
Expect(0, 1830, '\P{^Jg=e}', "");
Expect(1, 1829, '\p{Jg=	_E}', "");
Expect(0, 1829, '\p{^Jg=	_E}', "");
Expect(0, 1829, '\P{Jg=	_E}', "");
Expect(1, 1829, '\P{^Jg=	_E}', "");
Expect(0, 1830, '\p{Jg=	_E}', "");
Expect(1, 1830, '\p{^Jg=	_E}', "");
Expect(1, 1830, '\P{Jg=	_E}', "");
Expect(0, 1830, '\P{^Jg=	_E}', "");
Error('\p{Is_Joining_Group=	e/a/}');
Error('\P{Is_Joining_Group=	e/a/}');
Expect(1, 1829, '\p{Is_Joining_Group=e}', "");
Expect(0, 1829, '\p{^Is_Joining_Group=e}', "");
Expect(0, 1829, '\P{Is_Joining_Group=e}', "");
Expect(1, 1829, '\P{^Is_Joining_Group=e}', "");
Expect(0, 1830, '\p{Is_Joining_Group=e}', "");
Expect(1, 1830, '\p{^Is_Joining_Group=e}', "");
Expect(1, 1830, '\P{Is_Joining_Group=e}', "");
Expect(0, 1830, '\P{^Is_Joining_Group=e}', "");
Expect(1, 1829, '\p{Is_Joining_Group= E}', "");
Expect(0, 1829, '\p{^Is_Joining_Group= E}', "");
Expect(0, 1829, '\P{Is_Joining_Group= E}', "");
Expect(1, 1829, '\P{^Is_Joining_Group= E}', "");
Expect(0, 1830, '\p{Is_Joining_Group= E}', "");
Expect(1, 1830, '\p{^Is_Joining_Group= E}', "");
Expect(1, 1830, '\P{Is_Joining_Group= E}', "");
Expect(0, 1830, '\P{^Is_Joining_Group= E}', "");
Error('\p{Is_Jg=:=__E}');
Error('\P{Is_Jg=:=__E}');
Expect(1, 1829, '\p{Is_Jg=e}', "");
Expect(0, 1829, '\p{^Is_Jg=e}', "");
Expect(0, 1829, '\P{Is_Jg=e}', "");
Expect(1, 1829, '\P{^Is_Jg=e}', "");
Expect(0, 1830, '\p{Is_Jg=e}', "");
Expect(1, 1830, '\p{^Is_Jg=e}', "");
Expect(1, 1830, '\P{Is_Jg=e}', "");
Expect(0, 1830, '\P{^Is_Jg=e}', "");
Expect(1, 1829, '\p{Is_Jg=-e}', "");
Expect(0, 1829, '\p{^Is_Jg=-e}', "");
Expect(0, 1829, '\P{Is_Jg=-e}', "");
Expect(1, 1829, '\P{^Is_Jg=-e}', "");
Expect(0, 1830, '\p{Is_Jg=-e}', "");
Expect(1, 1830, '\p{^Is_Jg=-e}', "");
Expect(1, 1830, '\P{Is_Jg=-e}', "");
Expect(0, 1830, '\P{^Is_Jg=-e}', "");
Error('\p{Joining_Group=_/a/Yeh_BARREE}');
Error('\P{Joining_Group=_/a/Yeh_BARREE}');
Expect(1, 1747, '\p{Joining_Group=yehbarree}', "");
Expect(0, 1747, '\p{^Joining_Group=yehbarree}', "");
Expect(0, 1747, '\P{Joining_Group=yehbarree}', "");
Expect(1, 1747, '\P{^Joining_Group=yehbarree}', "");
Expect(0, 1748, '\p{Joining_Group=yehbarree}', "");
Expect(1, 1748, '\p{^Joining_Group=yehbarree}', "");
Expect(1, 1748, '\P{Joining_Group=yehbarree}', "");
Expect(0, 1748, '\P{^Joining_Group=yehbarree}', "");
Expect(1, 1747, '\p{Joining_Group=-	YEH_barree}', "");
Expect(0, 1747, '\p{^Joining_Group=-	YEH_barree}', "");
Expect(0, 1747, '\P{Joining_Group=-	YEH_barree}', "");
Expect(1, 1747, '\P{^Joining_Group=-	YEH_barree}', "");
Expect(0, 1748, '\p{Joining_Group=-	YEH_barree}', "");
Expect(1, 1748, '\p{^Joining_Group=-	YEH_barree}', "");
Expect(1, 1748, '\P{Joining_Group=-	YEH_barree}', "");
Expect(0, 1748, '\P{^Joining_Group=-	YEH_barree}', "");
Error('\p{Jg=/a/YEH_Barree}');
Error('\P{Jg=/a/YEH_Barree}');
Expect(1, 1747, '\p{Jg=yehbarree}', "");
Expect(0, 1747, '\p{^Jg=yehbarree}', "");
Expect(0, 1747, '\P{Jg=yehbarree}', "");
Expect(1, 1747, '\P{^Jg=yehbarree}', "");
Expect(0, 1748, '\p{Jg=yehbarree}', "");
Expect(1, 1748, '\p{^Jg=yehbarree}', "");
Expect(1, 1748, '\P{Jg=yehbarree}', "");
Expect(0, 1748, '\P{^Jg=yehbarree}', "");
Expect(1, 1747, '\p{Jg=-YEH_Barree}', "");
Expect(0, 1747, '\p{^Jg=-YEH_Barree}', "");
Expect(0, 1747, '\P{Jg=-YEH_Barree}', "");
Expect(1, 1747, '\P{^Jg=-YEH_Barree}', "");
Expect(0, 1748, '\p{Jg=-YEH_Barree}', "");
Expect(1, 1748, '\p{^Jg=-YEH_Barree}', "");
Expect(1, 1748, '\P{Jg=-YEH_Barree}', "");
Expect(0, 1748, '\P{^Jg=-YEH_Barree}', "");
Error('\p{Is_Joining_Group=		yeh_BARREE:=}');
Error('\P{Is_Joining_Group=		yeh_BARREE:=}');
Expect(1, 1747, '\p{Is_Joining_Group=yehbarree}', "");
Expect(0, 1747, '\p{^Is_Joining_Group=yehbarree}', "");
Expect(0, 1747, '\P{Is_Joining_Group=yehbarree}', "");
Expect(1, 1747, '\P{^Is_Joining_Group=yehbarree}', "");
Expect(0, 1748, '\p{Is_Joining_Group=yehbarree}', "");
Expect(1, 1748, '\p{^Is_Joining_Group=yehbarree}', "");
Expect(1, 1748, '\P{Is_Joining_Group=yehbarree}', "");
Expect(0, 1748, '\P{^Is_Joining_Group=yehbarree}', "");
Expect(1, 1747, '\p{Is_Joining_Group=Yeh_BARREE}', "");
Expect(0, 1747, '\p{^Is_Joining_Group=Yeh_BARREE}', "");
Expect(0, 1747, '\P{Is_Joining_Group=Yeh_BARREE}', "");
Expect(1, 1747, '\P{^Is_Joining_Group=Yeh_BARREE}', "");
Expect(0, 1748, '\p{Is_Joining_Group=Yeh_BARREE}', "");
Expect(1, 1748, '\p{^Is_Joining_Group=Yeh_BARREE}', "");
Expect(1, 1748, '\P{Is_Joining_Group=Yeh_BARREE}', "");
Expect(0, 1748, '\P{^Is_Joining_Group=Yeh_BARREE}', "");
Error('\p{Is_Jg=/a/_ Yeh_BARREE}');
Error('\P{Is_Jg=/a/_ Yeh_BARREE}');
Expect(1, 1747, '\p{Is_Jg=yehbarree}', "");
Expect(0, 1747, '\p{^Is_Jg=yehbarree}', "");
Expect(0, 1747, '\P{Is_Jg=yehbarree}', "");
Expect(1, 1747, '\P{^Is_Jg=yehbarree}', "");
Expect(0, 1748, '\p{Is_Jg=yehbarree}', "");
Expect(1, 1748, '\p{^Is_Jg=yehbarree}', "");
Expect(1, 1748, '\P{Is_Jg=yehbarree}', "");
Expect(0, 1748, '\P{^Is_Jg=yehbarree}', "");
Expect(1, 1747, '\p{Is_Jg=_-Yeh_Barree}', "");
Expect(0, 1747, '\p{^Is_Jg=_-Yeh_Barree}', "");
Expect(0, 1747, '\P{Is_Jg=_-Yeh_Barree}', "");
Expect(1, 1747, '\P{^Is_Jg=_-Yeh_Barree}', "");
Expect(0, 1748, '\p{Is_Jg=_-Yeh_Barree}', "");
Expect(1, 1748, '\p{^Is_Jg=_-Yeh_Barree}', "");
Expect(1, 1748, '\P{Is_Jg=_-Yeh_Barree}', "");
Expect(0, 1748, '\P{^Is_Jg=_-Yeh_Barree}', "");
Error('\p{Joining_Group=/a/	nya}');
Error('\P{Joining_Group=/a/	nya}');
Expect(1, 1725, '\p{Joining_Group=nya}', "");
Expect(0, 1725, '\p{^Joining_Group=nya}', "");
Expect(0, 1725, '\P{Joining_Group=nya}', "");
Expect(1, 1725, '\P{^Joining_Group=nya}', "");
Expect(0, 1726, '\p{Joining_Group=nya}', "");
Expect(1, 1726, '\p{^Joining_Group=nya}', "");
Expect(1, 1726, '\P{Joining_Group=nya}', "");
Expect(0, 1726, '\P{^Joining_Group=nya}', "");
Expect(1, 1725, '\p{Joining_Group:     NYA}', "");
Expect(0, 1725, '\p{^Joining_Group:     NYA}', "");
Expect(0, 1725, '\P{Joining_Group:     NYA}', "");
Expect(1, 1725, '\P{^Joining_Group:     NYA}', "");
Expect(0, 1726, '\p{Joining_Group:     NYA}', "");
Expect(1, 1726, '\p{^Joining_Group:     NYA}', "");
Expect(1, 1726, '\P{Joining_Group:     NYA}', "");
Expect(0, 1726, '\P{^Joining_Group:     NYA}', "");
Error('\p{Jg=__nya/a/}');
Error('\P{Jg=__nya/a/}');
Expect(1, 1725, '\p{Jg=nya}', "");
Expect(0, 1725, '\p{^Jg=nya}', "");
Expect(0, 1725, '\P{Jg=nya}', "");
Expect(1, 1725, '\P{^Jg=nya}', "");
Expect(0, 1726, '\p{Jg=nya}', "");
Expect(1, 1726, '\p{^Jg=nya}', "");
Expect(1, 1726, '\P{Jg=nya}', "");
Expect(0, 1726, '\P{^Jg=nya}', "");
Expect(1, 1725, '\p{Jg=	-Nya}', "");
Expect(0, 1725, '\p{^Jg=	-Nya}', "");
Expect(0, 1725, '\P{Jg=	-Nya}', "");
Expect(1, 1725, '\P{^Jg=	-Nya}', "");
Expect(0, 1726, '\p{Jg=	-Nya}', "");
Expect(1, 1726, '\p{^Jg=	-Nya}', "");
Expect(1, 1726, '\P{Jg=	-Nya}', "");
Expect(0, 1726, '\P{^Jg=	-Nya}', "");
Error('\p{Is_Joining_Group=-:=Nya}');
Error('\P{Is_Joining_Group=-:=Nya}');
Expect(1, 1725, '\p{Is_Joining_Group=nya}', "");
Expect(0, 1725, '\p{^Is_Joining_Group=nya}', "");
Expect(0, 1725, '\P{Is_Joining_Group=nya}', "");
Expect(1, 1725, '\P{^Is_Joining_Group=nya}', "");
Expect(0, 1726, '\p{Is_Joining_Group=nya}', "");
Expect(1, 1726, '\p{^Is_Joining_Group=nya}', "");
Expect(1, 1726, '\P{Is_Joining_Group=nya}', "");
Expect(0, 1726, '\P{^Is_Joining_Group=nya}', "");
Expect(1, 1725, '\p{Is_Joining_Group=_NYA}', "");
Expect(0, 1725, '\p{^Is_Joining_Group=_NYA}', "");
Expect(0, 1725, '\P{Is_Joining_Group=_NYA}', "");
Expect(1, 1725, '\P{^Is_Joining_Group=_NYA}', "");
Expect(0, 1726, '\p{Is_Joining_Group=_NYA}', "");
Expect(1, 1726, '\p{^Is_Joining_Group=_NYA}', "");
Expect(1, 1726, '\P{Is_Joining_Group=_NYA}', "");
Expect(0, 1726, '\P{^Is_Joining_Group=_NYA}', "");
Error('\p{Is_Jg=/a/_NYA}');
Error('\P{Is_Jg=/a/_NYA}');
Expect(1, 1725, '\p{Is_Jg=nya}', "");
Expect(0, 1725, '\p{^Is_Jg=nya}', "");
Expect(0, 1725, '\P{Is_Jg=nya}', "");
Expect(1, 1725, '\P{^Is_Jg=nya}', "");
Expect(0, 1726, '\p{Is_Jg=nya}', "");
Expect(1, 1726, '\p{^Is_Jg=nya}', "");
Expect(1, 1726, '\P{Is_Jg=nya}', "");
Expect(0, 1726, '\P{^Is_Jg=nya}', "");
Expect(1, 1725, '\p{Is_Jg=	Nya}', "");
Expect(0, 1725, '\p{^Is_Jg=	Nya}', "");
Expect(0, 1725, '\P{Is_Jg=	Nya}', "");
Expect(1, 1725, '\P{^Is_Jg=	Nya}', "");
Expect(0, 1726, '\p{Is_Jg=	Nya}', "");
Expect(1, 1726, '\p{^Is_Jg=	Nya}', "");
Expect(1, 1726, '\P{Is_Jg=	Nya}', "");
Expect(0, 1726, '\P{^Is_Jg=	Nya}', "");
Error('\p{Joining_Group=/a/ NUN}');
Error('\P{Joining_Group=/a/ NUN}');
Expect(1, 1826, '\p{Joining_Group=nun}', "");
Expect(0, 1826, '\p{^Joining_Group=nun}', "");
Expect(0, 1826, '\P{Joining_Group=nun}', "");
Expect(1, 1826, '\P{^Joining_Group=nun}', "");
Expect(0, 1827, '\p{Joining_Group=nun}', "");
Expect(1, 1827, '\p{^Joining_Group=nun}', "");
Expect(1, 1827, '\P{Joining_Group=nun}', "");
Expect(0, 1827, '\P{^Joining_Group=nun}', "");
Expect(1, 1826, '\p{Joining_Group:_Nun}', "");
Expect(0, 1826, '\p{^Joining_Group:_Nun}', "");
Expect(0, 1826, '\P{Joining_Group:_Nun}', "");
Expect(1, 1826, '\P{^Joining_Group:_Nun}', "");
Expect(0, 1827, '\p{Joining_Group:_Nun}', "");
Expect(1, 1827, '\p{^Joining_Group:_Nun}', "");
Expect(1, 1827, '\P{Joining_Group:_Nun}', "");
Expect(0, 1827, '\P{^Joining_Group:_Nun}', "");
Error('\p{Jg=/a/	_Nun}');
Error('\P{Jg=/a/	_Nun}');
Expect(1, 1826, '\p{Jg=nun}', "");
Expect(0, 1826, '\p{^Jg=nun}', "");
Expect(0, 1826, '\P{Jg=nun}', "");
Expect(1, 1826, '\P{^Jg=nun}', "");
Expect(0, 1827, '\p{Jg=nun}', "");
Expect(1, 1827, '\p{^Jg=nun}', "");
Expect(1, 1827, '\P{Jg=nun}', "");
Expect(0, 1827, '\P{^Jg=nun}', "");
Expect(1, 1826, '\p{Jg=-Nun}', "");
Expect(0, 1826, '\p{^Jg=-Nun}', "");
Expect(0, 1826, '\P{Jg=-Nun}', "");
Expect(1, 1826, '\P{^Jg=-Nun}', "");
Expect(0, 1827, '\p{Jg=-Nun}', "");
Expect(1, 1827, '\p{^Jg=-Nun}', "");
Expect(1, 1827, '\P{Jg=-Nun}', "");
Expect(0, 1827, '\P{^Jg=-Nun}', "");
Error('\p{Is_Joining_Group=-:=Nun}');
Error('\P{Is_Joining_Group=-:=Nun}');
Expect(1, 1826, '\p{Is_Joining_Group=nun}', "");
Expect(0, 1826, '\p{^Is_Joining_Group=nun}', "");
Expect(0, 1826, '\P{Is_Joining_Group=nun}', "");
Expect(1, 1826, '\P{^Is_Joining_Group=nun}', "");
Expect(0, 1827, '\p{Is_Joining_Group=nun}', "");
Expect(1, 1827, '\p{^Is_Joining_Group=nun}', "");
Expect(1, 1827, '\P{Is_Joining_Group=nun}', "");
Expect(0, 1827, '\P{^Is_Joining_Group=nun}', "");
Expect(1, 1826, '\p{Is_Joining_Group=_	Nun}', "");
Expect(0, 1826, '\p{^Is_Joining_Group=_	Nun}', "");
Expect(0, 1826, '\P{Is_Joining_Group=_	Nun}', "");
Expect(1, 1826, '\P{^Is_Joining_Group=_	Nun}', "");
Expect(0, 1827, '\p{Is_Joining_Group=_	Nun}', "");
Expect(1, 1827, '\p{^Is_Joining_Group=_	Nun}', "");
Expect(1, 1827, '\P{Is_Joining_Group=_	Nun}', "");
Expect(0, 1827, '\P{^Is_Joining_Group=_	Nun}', "");
Error('\p{Is_Jg=/a/	-NUN}');
Error('\P{Is_Jg=/a/	-NUN}');
Expect(1, 1826, '\p{Is_Jg=nun}', "");
Expect(0, 1826, '\p{^Is_Jg=nun}', "");
Expect(0, 1826, '\P{Is_Jg=nun}', "");
Expect(1, 1826, '\P{^Is_Jg=nun}', "");
Expect(0, 1827, '\p{Is_Jg=nun}', "");
Expect(1, 1827, '\p{^Is_Jg=nun}', "");
Expect(1, 1827, '\P{Is_Jg=nun}', "");
Expect(0, 1827, '\P{^Is_Jg=nun}', "");
Expect(1, 1826, '\p{Is_Jg:	--nun}', "");
Expect(0, 1826, '\p{^Is_Jg:	--nun}', "");
Expect(0, 1826, '\P{Is_Jg:	--nun}', "");
Expect(1, 1826, '\P{^Is_Jg:	--nun}', "");
Expect(0, 1827, '\p{Is_Jg:	--nun}', "");
Expect(1, 1827, '\p{^Is_Jg:	--nun}', "");
Expect(1, 1827, '\P{Is_Jg:	--nun}', "");
Expect(0, 1827, '\P{^Is_Jg:	--nun}', "");
Error('\p{Joining_Group=  Lam/a/}');
Error('\P{Joining_Group=  Lam/a/}');
Expect(1, 2214, '\p{Joining_Group=lam}', "");
Expect(0, 2214, '\p{^Joining_Group=lam}', "");
Expect(0, 2214, '\P{Joining_Group=lam}', "");
Expect(1, 2214, '\P{^Joining_Group=lam}', "");
Expect(0, 2215, '\p{Joining_Group=lam}', "");
Expect(1, 2215, '\p{^Joining_Group=lam}', "");
Expect(1, 2215, '\P{Joining_Group=lam}', "");
Expect(0, 2215, '\P{^Joining_Group=lam}', "");
Expect(1, 2214, '\p{Joining_Group=_Lam}', "");
Expect(0, 2214, '\p{^Joining_Group=_Lam}', "");
Expect(0, 2214, '\P{Joining_Group=_Lam}', "");
Expect(1, 2214, '\P{^Joining_Group=_Lam}', "");
Expect(0, 2215, '\p{Joining_Group=_Lam}', "");
Expect(1, 2215, '\p{^Joining_Group=_Lam}', "");
Expect(1, 2215, '\P{Joining_Group=_Lam}', "");
Expect(0, 2215, '\P{^Joining_Group=_Lam}', "");
Error('\p{Jg=-:=LAM}');
Error('\P{Jg=-:=LAM}');
Expect(1, 2214, '\p{Jg=lam}', "");
Expect(0, 2214, '\p{^Jg=lam}', "");
Expect(0, 2214, '\P{Jg=lam}', "");
Expect(1, 2214, '\P{^Jg=lam}', "");
Expect(0, 2215, '\p{Jg=lam}', "");
Expect(1, 2215, '\p{^Jg=lam}', "");
Expect(1, 2215, '\P{Jg=lam}', "");
Expect(0, 2215, '\P{^Jg=lam}', "");
Expect(1, 2214, '\p{Jg=	Lam}', "");
Expect(0, 2214, '\p{^Jg=	Lam}', "");
Expect(0, 2214, '\P{Jg=	Lam}', "");
Expect(1, 2214, '\P{^Jg=	Lam}', "");
Expect(0, 2215, '\p{Jg=	Lam}', "");
Expect(1, 2215, '\p{^Jg=	Lam}', "");
Expect(1, 2215, '\P{Jg=	Lam}', "");
Expect(0, 2215, '\P{^Jg=	Lam}', "");
Error('\p{Is_Joining_Group=/a/Lam}');
Error('\P{Is_Joining_Group=/a/Lam}');
Expect(1, 2214, '\p{Is_Joining_Group=lam}', "");
Expect(0, 2214, '\p{^Is_Joining_Group=lam}', "");
Expect(0, 2214, '\P{Is_Joining_Group=lam}', "");
Expect(1, 2214, '\P{^Is_Joining_Group=lam}', "");
Expect(0, 2215, '\p{Is_Joining_Group=lam}', "");
Expect(1, 2215, '\p{^Is_Joining_Group=lam}', "");
Expect(1, 2215, '\P{Is_Joining_Group=lam}', "");
Expect(0, 2215, '\P{^Is_Joining_Group=lam}', "");
Expect(1, 2214, '\p{Is_Joining_Group=_ lam}', "");
Expect(0, 2214, '\p{^Is_Joining_Group=_ lam}', "");
Expect(0, 2214, '\P{Is_Joining_Group=_ lam}', "");
Expect(1, 2214, '\P{^Is_Joining_Group=_ lam}', "");
Expect(0, 2215, '\p{Is_Joining_Group=_ lam}', "");
Expect(1, 2215, '\p{^Is_Joining_Group=_ lam}', "");
Expect(1, 2215, '\P{Is_Joining_Group=_ lam}', "");
Expect(0, 2215, '\P{^Is_Joining_Group=_ lam}', "");
Error('\p{Is_Jg=	:=Lam}');
Error('\P{Is_Jg=	:=Lam}');
Expect(1, 2214, '\p{Is_Jg=lam}', "");
Expect(0, 2214, '\p{^Is_Jg=lam}', "");
Expect(0, 2214, '\P{Is_Jg=lam}', "");
Expect(1, 2214, '\P{^Is_Jg=lam}', "");
Expect(0, 2215, '\p{Is_Jg=lam}', "");
Expect(1, 2215, '\p{^Is_Jg=lam}', "");
Expect(1, 2215, '\P{Is_Jg=lam}', "");
Expect(0, 2215, '\P{^Is_Jg=lam}', "");
Expect(1, 2214, '\p{Is_Jg:   --Lam}', "");
Expect(0, 2214, '\p{^Is_Jg:   --Lam}', "");
Expect(0, 2214, '\P{Is_Jg:   --Lam}', "");
Expect(1, 2214, '\P{^Is_Jg:   --Lam}', "");
Expect(0, 2215, '\p{Is_Jg:   --Lam}', "");
Expect(1, 2215, '\p{^Is_Jg:   --Lam}', "");
Expect(1, 2215, '\P{Is_Jg:   --Lam}', "");
Expect(0, 2215, '\P{^Is_Jg:   --Lam}', "");
Error('\p{Joining_Group=:=- ain}');
Error('\P{Joining_Group=:=- ain}');
Expect(1, 1887, '\p{Joining_Group:   ain}', "");
Expect(0, 1887, '\p{^Joining_Group:   ain}', "");
Expect(0, 1887, '\P{Joining_Group:   ain}', "");
Expect(1, 1887, '\P{^Joining_Group:   ain}', "");
Expect(0, 1888, '\p{Joining_Group:   ain}', "");
Expect(1, 1888, '\p{^Joining_Group:   ain}', "");
Expect(1, 1888, '\P{Joining_Group:   ain}', "");
Expect(0, 1888, '\P{^Joining_Group:   ain}', "");
Expect(1, 1887, '\p{Joining_Group:		-ain}', "");
Expect(0, 1887, '\p{^Joining_Group:		-ain}', "");
Expect(0, 1887, '\P{Joining_Group:		-ain}', "");
Expect(1, 1887, '\P{^Joining_Group:		-ain}', "");
Expect(0, 1888, '\p{Joining_Group:		-ain}', "");
Expect(1, 1888, '\p{^Joining_Group:		-ain}', "");
Expect(1, 1888, '\P{Joining_Group:		-ain}', "");
Expect(0, 1888, '\P{^Joining_Group:		-ain}', "");
Error('\p{Jg=_:=Ain}');
Error('\P{Jg=_:=Ain}');
Expect(1, 1887, '\p{Jg=ain}', "");
Expect(0, 1887, '\p{^Jg=ain}', "");
Expect(0, 1887, '\P{Jg=ain}', "");
Expect(1, 1887, '\P{^Jg=ain}', "");
Expect(0, 1888, '\p{Jg=ain}', "");
Expect(1, 1888, '\p{^Jg=ain}', "");
Expect(1, 1888, '\P{Jg=ain}', "");
Expect(0, 1888, '\P{^Jg=ain}', "");
Expect(1, 1887, '\p{Jg=_-Ain}', "");
Expect(0, 1887, '\p{^Jg=_-Ain}', "");
Expect(0, 1887, '\P{Jg=_-Ain}', "");
Expect(1, 1887, '\P{^Jg=_-Ain}', "");
Expect(0, 1888, '\p{Jg=_-Ain}', "");
Expect(1, 1888, '\p{^Jg=_-Ain}', "");
Expect(1, 1888, '\P{Jg=_-Ain}', "");
Expect(0, 1888, '\P{^Jg=_-Ain}', "");
Error('\p{Is_Joining_Group= _ain:=}');
Error('\P{Is_Joining_Group= _ain:=}');
Expect(1, 1887, '\p{Is_Joining_Group=ain}', "");
Expect(0, 1887, '\p{^Is_Joining_Group=ain}', "");
Expect(0, 1887, '\P{Is_Joining_Group=ain}', "");
Expect(1, 1887, '\P{^Is_Joining_Group=ain}', "");
Expect(0, 1888, '\p{Is_Joining_Group=ain}', "");
Expect(1, 1888, '\p{^Is_Joining_Group=ain}', "");
Expect(1, 1888, '\P{Is_Joining_Group=ain}', "");
Expect(0, 1888, '\P{^Is_Joining_Group=ain}', "");
Expect(1, 1887, '\p{Is_Joining_Group=	-AIN}', "");
Expect(0, 1887, '\p{^Is_Joining_Group=	-AIN}', "");
Expect(0, 1887, '\P{Is_Joining_Group=	-AIN}', "");
Expect(1, 1887, '\P{^Is_Joining_Group=	-AIN}', "");
Expect(0, 1888, '\p{Is_Joining_Group=	-AIN}', "");
Expect(1, 1888, '\p{^Is_Joining_Group=	-AIN}', "");
Expect(1, 1888, '\P{Is_Joining_Group=	-AIN}', "");
Expect(0, 1888, '\P{^Is_Joining_Group=	-AIN}', "");
Error('\p{Is_Jg=/a/-ain}');
Error('\P{Is_Jg=/a/-ain}');
Expect(1, 1887, '\p{Is_Jg:ain}', "");
Expect(0, 1887, '\p{^Is_Jg:ain}', "");
Expect(0, 1887, '\P{Is_Jg:ain}', "");
Expect(1, 1887, '\P{^Is_Jg:ain}', "");
Expect(0, 1888, '\p{Is_Jg:ain}', "");
Expect(1, 1888, '\p{^Is_Jg:ain}', "");
Expect(1, 1888, '\P{Is_Jg:ain}', "");
Expect(0, 1888, '\P{^Is_Jg:ain}', "");
Expect(1, 1887, '\p{Is_Jg=	 AIN}', "");
Expect(0, 1887, '\p{^Is_Jg=	 AIN}', "");
Expect(0, 1887, '\P{Is_Jg=	 AIN}', "");
Expect(1, 1887, '\P{^Is_Jg=	 AIN}', "");
Expect(0, 1888, '\p{Is_Jg=	 AIN}', "");
Expect(1, 1888, '\p{^Is_Jg=	 AIN}', "");
Expect(1, 1888, '\P{Is_Jg=	 AIN}', "");
Expect(0, 1888, '\P{^Is_Jg=	 AIN}', "");
Error('\p{Joining_Group=_/a/Heh}');
Error('\P{Joining_Group=_/a/Heh}');
Expect(1, 1607, '\p{Joining_Group: heh}', "");
Expect(0, 1607, '\p{^Joining_Group: heh}', "");
Expect(0, 1607, '\P{Joining_Group: heh}', "");
Expect(1, 1607, '\P{^Joining_Group: heh}', "");
Expect(0, 1608, '\p{Joining_Group: heh}', "");
Expect(1, 1608, '\p{^Joining_Group: heh}', "");
Expect(1, 1608, '\P{Joining_Group: heh}', "");
Expect(0, 1608, '\P{^Joining_Group: heh}', "");
Expect(1, 1607, '\p{Joining_Group=_heh}', "");
Expect(0, 1607, '\p{^Joining_Group=_heh}', "");
Expect(0, 1607, '\P{Joining_Group=_heh}', "");
Expect(1, 1607, '\P{^Joining_Group=_heh}', "");
Expect(0, 1608, '\p{Joining_Group=_heh}', "");
Expect(1, 1608, '\p{^Joining_Group=_heh}', "");
Expect(1, 1608, '\P{Joining_Group=_heh}', "");
Expect(0, 1608, '\P{^Joining_Group=_heh}', "");
Error('\p{Jg:    _HEH:=}');
Error('\P{Jg:    _HEH:=}');
Expect(1, 1607, '\p{Jg=heh}', "");
Expect(0, 1607, '\p{^Jg=heh}', "");
Expect(0, 1607, '\P{Jg=heh}', "");
Expect(1, 1607, '\P{^Jg=heh}', "");
Expect(0, 1608, '\p{Jg=heh}', "");
Expect(1, 1608, '\p{^Jg=heh}', "");
Expect(1, 1608, '\P{Jg=heh}', "");
Expect(0, 1608, '\P{^Jg=heh}', "");
Expect(1, 1607, '\p{Jg=		Heh}', "");
Expect(0, 1607, '\p{^Jg=		Heh}', "");
Expect(0, 1607, '\P{Jg=		Heh}', "");
Expect(1, 1607, '\P{^Jg=		Heh}', "");
Expect(0, 1608, '\p{Jg=		Heh}', "");
Expect(1, 1608, '\p{^Jg=		Heh}', "");
Expect(1, 1608, '\P{Jg=		Heh}', "");
Expect(0, 1608, '\P{^Jg=		Heh}', "");
Error('\p{Is_Joining_Group::=-_Heh}');
Error('\P{Is_Joining_Group::=-_Heh}');
Expect(1, 1607, '\p{Is_Joining_Group=heh}', "");
Expect(0, 1607, '\p{^Is_Joining_Group=heh}', "");
Expect(0, 1607, '\P{Is_Joining_Group=heh}', "");
Expect(1, 1607, '\P{^Is_Joining_Group=heh}', "");
Expect(0, 1608, '\p{Is_Joining_Group=heh}', "");
Expect(1, 1608, '\p{^Is_Joining_Group=heh}', "");
Expect(1, 1608, '\P{Is_Joining_Group=heh}', "");
Expect(0, 1608, '\P{^Is_Joining_Group=heh}', "");
Expect(1, 1607, '\p{Is_Joining_Group= _Heh}', "");
Expect(0, 1607, '\p{^Is_Joining_Group= _Heh}', "");
Expect(0, 1607, '\P{Is_Joining_Group= _Heh}', "");
Expect(1, 1607, '\P{^Is_Joining_Group= _Heh}', "");
Expect(0, 1608, '\p{Is_Joining_Group= _Heh}', "");
Expect(1, 1608, '\p{^Is_Joining_Group= _Heh}', "");
Expect(1, 1608, '\P{Is_Joining_Group= _Heh}', "");
Expect(0, 1608, '\P{^Is_Joining_Group= _Heh}', "");
Error('\p{Is_Jg=_Heh/a/}');
Error('\P{Is_Jg=_Heh/a/}');
Expect(1, 1607, '\p{Is_Jg=heh}', "");
Expect(0, 1607, '\p{^Is_Jg=heh}', "");
Expect(0, 1607, '\P{Is_Jg=heh}', "");
Expect(1, 1607, '\P{^Is_Jg=heh}', "");
Expect(0, 1608, '\p{Is_Jg=heh}', "");
Expect(1, 1608, '\p{^Is_Jg=heh}', "");
Expect(1, 1608, '\P{Is_Jg=heh}', "");
Expect(0, 1608, '\P{^Is_Jg=heh}', "");
Expect(1, 1607, '\p{Is_Jg:_ heh}', "");
Expect(0, 1607, '\p{^Is_Jg:_ heh}', "");
Expect(0, 1607, '\P{Is_Jg:_ heh}', "");
Expect(1, 1607, '\P{^Is_Jg:_ heh}', "");
Expect(0, 1608, '\p{Is_Jg:_ heh}', "");
Expect(1, 1608, '\p{^Is_Jg:_ heh}', "");
Expect(1, 1608, '\P{Is_Jg:_ heh}', "");
Expect(0, 1608, '\P{^Is_Jg:_ heh}', "");
Error('\p{Joining_Group:   _/a/Qaph}');
Error('\P{Joining_Group:   _/a/Qaph}');
Expect(1, 1833, '\p{Joining_Group=qaph}', "");
Expect(0, 1833, '\p{^Joining_Group=qaph}', "");
Expect(0, 1833, '\P{Joining_Group=qaph}', "");
Expect(1, 1833, '\P{^Joining_Group=qaph}', "");
Expect(0, 1834, '\p{Joining_Group=qaph}', "");
Expect(1, 1834, '\p{^Joining_Group=qaph}', "");
Expect(1, 1834, '\P{Joining_Group=qaph}', "");
Expect(0, 1834, '\P{^Joining_Group=qaph}', "");
Expect(1, 1833, '\p{Joining_Group=	_qaph}', "");
Expect(0, 1833, '\p{^Joining_Group=	_qaph}', "");
Expect(0, 1833, '\P{Joining_Group=	_qaph}', "");
Expect(1, 1833, '\P{^Joining_Group=	_qaph}', "");
Expect(0, 1834, '\p{Joining_Group=	_qaph}', "");
Expect(1, 1834, '\p{^Joining_Group=	_qaph}', "");
Expect(1, 1834, '\P{Joining_Group=	_qaph}', "");
Expect(0, 1834, '\P{^Joining_Group=	_qaph}', "");
Error('\p{Jg=:=-QAPH}');
Error('\P{Jg=:=-QAPH}');
Expect(1, 1833, '\p{Jg=qaph}', "");
Expect(0, 1833, '\p{^Jg=qaph}', "");
Expect(0, 1833, '\P{Jg=qaph}', "");
Expect(1, 1833, '\P{^Jg=qaph}', "");
Expect(0, 1834, '\p{Jg=qaph}', "");
Expect(1, 1834, '\p{^Jg=qaph}', "");
Expect(1, 1834, '\P{Jg=qaph}', "");
Expect(0, 1834, '\P{^Jg=qaph}', "");
Expect(1, 1833, '\p{Jg=_Qaph}', "");
Expect(0, 1833, '\p{^Jg=_Qaph}', "");
Expect(0, 1833, '\P{Jg=_Qaph}', "");
Expect(1, 1833, '\P{^Jg=_Qaph}', "");
Expect(0, 1834, '\p{Jg=_Qaph}', "");
Expect(1, 1834, '\p{^Jg=_Qaph}', "");
Expect(1, 1834, '\P{Jg=_Qaph}', "");
Expect(0, 1834, '\P{^Jg=_Qaph}', "");
Error('\p{Is_Joining_Group=__Qaph:=}');
Error('\P{Is_Joining_Group=__Qaph:=}');
Expect(1, 1833, '\p{Is_Joining_Group=qaph}', "");
Expect(0, 1833, '\p{^Is_Joining_Group=qaph}', "");
Expect(0, 1833, '\P{Is_Joining_Group=qaph}', "");
Expect(1, 1833, '\P{^Is_Joining_Group=qaph}', "");
Expect(0, 1834, '\p{Is_Joining_Group=qaph}', "");
Expect(1, 1834, '\p{^Is_Joining_Group=qaph}', "");
Expect(1, 1834, '\P{Is_Joining_Group=qaph}', "");
Expect(0, 1834, '\P{^Is_Joining_Group=qaph}', "");
Expect(1, 1833, '\p{Is_Joining_Group= QAPH}', "");
Expect(0, 1833, '\p{^Is_Joining_Group= QAPH}', "");
Expect(0, 1833, '\P{Is_Joining_Group= QAPH}', "");
Expect(1, 1833, '\P{^Is_Joining_Group= QAPH}', "");
Expect(0, 1834, '\p{Is_Joining_Group= QAPH}', "");
Expect(1, 1834, '\p{^Is_Joining_Group= QAPH}', "");
Expect(1, 1834, '\P{Is_Joining_Group= QAPH}', "");
Expect(0, 1834, '\P{^Is_Joining_Group= QAPH}', "");
Error('\p{Is_Jg=	-Qaph:=}');
Error('\P{Is_Jg=	-Qaph:=}');
Expect(1, 1833, '\p{Is_Jg=qaph}', "");
Expect(0, 1833, '\p{^Is_Jg=qaph}', "");
Expect(0, 1833, '\P{Is_Jg=qaph}', "");
Expect(1, 1833, '\P{^Is_Jg=qaph}', "");
Expect(0, 1834, '\p{Is_Jg=qaph}', "");
Expect(1, 1834, '\p{^Is_Jg=qaph}', "");
Expect(1, 1834, '\P{Is_Jg=qaph}', "");
Expect(0, 1834, '\P{^Is_Jg=qaph}', "");
Expect(1, 1833, '\p{Is_Jg=QAPH}', "");
Expect(0, 1833, '\p{^Is_Jg=QAPH}', "");
Expect(0, 1833, '\P{Is_Jg=QAPH}', "");
Expect(1, 1833, '\P{^Is_Jg=QAPH}', "");
Expect(0, 1834, '\p{Is_Jg=QAPH}', "");
Expect(1, 1834, '\p{^Is_Jg=QAPH}', "");
Expect(1, 1834, '\P{Is_Jg=QAPH}', "");
Expect(0, 1834, '\P{^Is_Jg=QAPH}', "");
Error('\p{Joining_Group=:=_-HETH}');
Error('\P{Joining_Group=:=_-HETH}');
Expect(1, 1818, '\p{Joining_Group:	heth}', "");
Expect(0, 1818, '\p{^Joining_Group:	heth}', "");
Expect(0, 1818, '\P{Joining_Group:	heth}', "");
Expect(1, 1818, '\P{^Joining_Group:	heth}', "");
Expect(0, 1819, '\p{Joining_Group:	heth}', "");
Expect(1, 1819, '\p{^Joining_Group:	heth}', "");
Expect(1, 1819, '\P{Joining_Group:	heth}', "");
Expect(0, 1819, '\P{^Joining_Group:	heth}', "");
Error('\p{Jg=/a/	Heth}');
Error('\P{Jg=/a/	Heth}');
Expect(1, 1818, '\p{Jg=heth}', "");
Expect(0, 1818, '\p{^Jg=heth}', "");
Expect(0, 1818, '\P{Jg=heth}', "");
Expect(1, 1818, '\P{^Jg=heth}', "");
Expect(0, 1819, '\p{Jg=heth}', "");
Expect(1, 1819, '\p{^Jg=heth}', "");
Expect(1, 1819, '\P{Jg=heth}', "");
Expect(0, 1819, '\P{^Jg=heth}', "");
Expect(1, 1818, '\p{Jg=-HETH}', "");
Expect(0, 1818, '\p{^Jg=-HETH}', "");
Expect(0, 1818, '\P{Jg=-HETH}', "");
Expect(1, 1818, '\P{^Jg=-HETH}', "");
Expect(0, 1819, '\p{Jg=-HETH}', "");
Expect(1, 1819, '\p{^Jg=-HETH}', "");
Expect(1, 1819, '\P{Jg=-HETH}', "");
Expect(0, 1819, '\P{^Jg=-HETH}', "");
Error('\p{Is_Joining_Group=:=Heth}');
Error('\P{Is_Joining_Group=:=Heth}');
Expect(1, 1818, '\p{Is_Joining_Group:	heth}', "");
Expect(0, 1818, '\p{^Is_Joining_Group:	heth}', "");
Expect(0, 1818, '\P{Is_Joining_Group:	heth}', "");
Expect(1, 1818, '\P{^Is_Joining_Group:	heth}', "");
Expect(0, 1819, '\p{Is_Joining_Group:	heth}', "");
Expect(1, 1819, '\p{^Is_Joining_Group:	heth}', "");
Expect(1, 1819, '\P{Is_Joining_Group:	heth}', "");
Expect(0, 1819, '\P{^Is_Joining_Group:	heth}', "");
Expect(1, 1818, '\p{Is_Joining_Group=Heth}', "");
Expect(0, 1818, '\p{^Is_Joining_Group=Heth}', "");
Expect(0, 1818, '\P{Is_Joining_Group=Heth}', "");
Expect(1, 1818, '\P{^Is_Joining_Group=Heth}', "");
Expect(0, 1819, '\p{Is_Joining_Group=Heth}', "");
Expect(1, 1819, '\p{^Is_Joining_Group=Heth}', "");
Expect(1, 1819, '\P{Is_Joining_Group=Heth}', "");
Expect(0, 1819, '\P{^Is_Joining_Group=Heth}', "");
Error('\p{Is_Jg=_:=Heth}');
Error('\P{Is_Jg=_:=Heth}');
Expect(1, 1818, '\p{Is_Jg=heth}', "");
Expect(0, 1818, '\p{^Is_Jg=heth}', "");
Expect(0, 1818, '\P{Is_Jg=heth}', "");
Expect(1, 1818, '\P{^Is_Jg=heth}', "");
Expect(0, 1819, '\p{Is_Jg=heth}', "");
Expect(1, 1819, '\p{^Is_Jg=heth}', "");
Expect(1, 1819, '\P{Is_Jg=heth}', "");
Expect(0, 1819, '\P{^Is_Jg=heth}', "");
Expect(1, 1818, '\p{Is_Jg=-HETH}', "");
Expect(0, 1818, '\p{^Is_Jg=-HETH}', "");
Expect(0, 1818, '\P{Is_Jg=-HETH}', "");
Expect(1, 1818, '\P{^Is_Jg=-HETH}', "");
Expect(0, 1819, '\p{Is_Jg=-HETH}', "");
Expect(1, 1819, '\p{^Is_Jg=-HETH}', "");
Expect(1, 1819, '\P{Is_Jg=-HETH}', "");
Expect(0, 1819, '\P{^Is_Jg=-HETH}', "");
Error('\p{Joining_Group=/a/Teth}');
Error('\P{Joining_Group=/a/Teth}');
Expect(1, 1820, '\p{Joining_Group=teth}', "");
Expect(0, 1820, '\p{^Joining_Group=teth}', "");
Expect(0, 1820, '\P{Joining_Group=teth}', "");
Expect(1, 1820, '\P{^Joining_Group=teth}', "");
Expect(0, 1821, '\p{Joining_Group=teth}', "");
Expect(1, 1821, '\p{^Joining_Group=teth}', "");
Expect(1, 1821, '\P{Joining_Group=teth}', "");
Expect(0, 1821, '\P{^Joining_Group=teth}', "");
Expect(1, 1820, '\p{Joining_Group: __Teth}', "");
Expect(0, 1820, '\p{^Joining_Group: __Teth}', "");
Expect(0, 1820, '\P{Joining_Group: __Teth}', "");
Expect(1, 1820, '\P{^Joining_Group: __Teth}', "");
Expect(0, 1821, '\p{Joining_Group: __Teth}', "");
Expect(1, 1821, '\p{^Joining_Group: __Teth}', "");
Expect(1, 1821, '\P{Joining_Group: __Teth}', "");
Expect(0, 1821, '\P{^Joining_Group: __Teth}', "");
Error('\p{Jg=	/a/teth}');
Error('\P{Jg=	/a/teth}');
Expect(1, 1820, '\p{Jg=teth}', "");
Expect(0, 1820, '\p{^Jg=teth}', "");
Expect(0, 1820, '\P{Jg=teth}', "");
Expect(1, 1820, '\P{^Jg=teth}', "");
Expect(0, 1821, '\p{Jg=teth}', "");
Expect(1, 1821, '\p{^Jg=teth}', "");
Expect(1, 1821, '\P{Jg=teth}', "");
Expect(0, 1821, '\P{^Jg=teth}', "");
Expect(1, 1820, '\p{Jg=__teth}', "");
Expect(0, 1820, '\p{^Jg=__teth}', "");
Expect(0, 1820, '\P{Jg=__teth}', "");
Expect(1, 1820, '\P{^Jg=__teth}', "");
Expect(0, 1821, '\p{Jg=__teth}', "");
Expect(1, 1821, '\p{^Jg=__teth}', "");
Expect(1, 1821, '\P{Jg=__teth}', "");
Expect(0, 1821, '\P{^Jg=__teth}', "");
Error('\p{Is_Joining_Group=/a/	teth}');
Error('\P{Is_Joining_Group=/a/	teth}');
Expect(1, 1820, '\p{Is_Joining_Group=teth}', "");
Expect(0, 1820, '\p{^Is_Joining_Group=teth}', "");
Expect(0, 1820, '\P{Is_Joining_Group=teth}', "");
Expect(1, 1820, '\P{^Is_Joining_Group=teth}', "");
Expect(0, 1821, '\p{Is_Joining_Group=teth}', "");
Expect(1, 1821, '\p{^Is_Joining_Group=teth}', "");
Expect(1, 1821, '\P{Is_Joining_Group=teth}', "");
Expect(0, 1821, '\P{^Is_Joining_Group=teth}', "");
Expect(1, 1820, '\p{Is_Joining_Group=	_Teth}', "");
Expect(0, 1820, '\p{^Is_Joining_Group=	_Teth}', "");
Expect(0, 1820, '\P{Is_Joining_Group=	_Teth}', "");
Expect(1, 1820, '\P{^Is_Joining_Group=	_Teth}', "");
Expect(0, 1821, '\p{Is_Joining_Group=	_Teth}', "");
Expect(1, 1821, '\p{^Is_Joining_Group=	_Teth}', "");
Expect(1, 1821, '\P{Is_Joining_Group=	_Teth}', "");
Expect(0, 1821, '\P{^Is_Joining_Group=	_Teth}', "");
Error('\p{Is_Jg=/a/-TETH}');
Error('\P{Is_Jg=/a/-TETH}');
Expect(1, 1820, '\p{Is_Jg=teth}', "");
Expect(0, 1820, '\p{^Is_Jg=teth}', "");
Expect(0, 1820, '\P{Is_Jg=teth}', "");
Expect(1, 1820, '\P{^Is_Jg=teth}', "");
Expect(0, 1821, '\p{Is_Jg=teth}', "");
Expect(1, 1821, '\p{^Is_Jg=teth}', "");
Expect(1, 1821, '\P{Is_Jg=teth}', "");
Expect(0, 1821, '\P{^Is_Jg=teth}', "");
Expect(1, 1820, '\p{Is_Jg=_Teth}', "");
Expect(0, 1820, '\p{^Is_Jg=_Teth}', "");
Expect(0, 1820, '\P{Is_Jg=_Teth}', "");
Expect(1, 1820, '\P{^Is_Jg=_Teth}', "");
Expect(0, 1821, '\p{Is_Jg=_Teth}', "");
Expect(1, 1821, '\p{^Is_Jg=_Teth}', "");
Expect(1, 1821, '\P{Is_Jg=_Teth}', "");
Expect(0, 1821, '\P{^Is_Jg=_Teth}', "");
Error('\p{Joining_Group=_:=dalath_Rish}');
Error('\P{Joining_Group=_:=dalath_Rish}');
Expect(1, 1839, '\p{Joining_Group=dalathrish}', "");
Expect(0, 1839, '\p{^Joining_Group=dalathrish}', "");
Expect(0, 1839, '\P{Joining_Group=dalathrish}', "");
Expect(1, 1839, '\P{^Joining_Group=dalathrish}', "");
Expect(0, 1840, '\p{Joining_Group=dalathrish}', "");
Expect(1, 1840, '\p{^Joining_Group=dalathrish}', "");
Expect(1, 1840, '\P{Joining_Group=dalathrish}', "");
Expect(0, 1840, '\P{^Joining_Group=dalathrish}', "");
Expect(1, 1839, '\p{Joining_Group=_dalath_RISH}', "");
Expect(0, 1839, '\p{^Joining_Group=_dalath_RISH}', "");
Expect(0, 1839, '\P{Joining_Group=_dalath_RISH}', "");
Expect(1, 1839, '\P{^Joining_Group=_dalath_RISH}', "");
Expect(0, 1840, '\p{Joining_Group=_dalath_RISH}', "");
Expect(1, 1840, '\p{^Joining_Group=_dalath_RISH}', "");
Expect(1, 1840, '\P{Joining_Group=_dalath_RISH}', "");
Expect(0, 1840, '\P{^Joining_Group=_dalath_RISH}', "");
Error('\p{Jg=	/a/dalath_Rish}');
Error('\P{Jg=	/a/dalath_Rish}');
Expect(1, 1839, '\p{Jg: dalathrish}', "");
Expect(0, 1839, '\p{^Jg: dalathrish}', "");
Expect(0, 1839, '\P{Jg: dalathrish}', "");
Expect(1, 1839, '\P{^Jg: dalathrish}', "");
Expect(0, 1840, '\p{Jg: dalathrish}', "");
Expect(1, 1840, '\p{^Jg: dalathrish}', "");
Expect(1, 1840, '\P{Jg: dalathrish}', "");
Expect(0, 1840, '\P{^Jg: dalathrish}', "");
Expect(1, 1839, '\p{Jg=	Dalath_RISH}', "");
Expect(0, 1839, '\p{^Jg=	Dalath_RISH}', "");
Expect(0, 1839, '\P{Jg=	Dalath_RISH}', "");
Expect(1, 1839, '\P{^Jg=	Dalath_RISH}', "");
Expect(0, 1840, '\p{Jg=	Dalath_RISH}', "");
Expect(1, 1840, '\p{^Jg=	Dalath_RISH}', "");
Expect(1, 1840, '\P{Jg=	Dalath_RISH}', "");
Expect(0, 1840, '\P{^Jg=	Dalath_RISH}', "");
Error('\p{Is_Joining_Group=-:=dalath_Rish}');
Error('\P{Is_Joining_Group=-:=dalath_Rish}');
Expect(1, 1839, '\p{Is_Joining_Group=dalathrish}', "");
Expect(0, 1839, '\p{^Is_Joining_Group=dalathrish}', "");
Expect(0, 1839, '\P{Is_Joining_Group=dalathrish}', "");
Expect(1, 1839, '\P{^Is_Joining_Group=dalathrish}', "");
Expect(0, 1840, '\p{Is_Joining_Group=dalathrish}', "");
Expect(1, 1840, '\p{^Is_Joining_Group=dalathrish}', "");
Expect(1, 1840, '\P{Is_Joining_Group=dalathrish}', "");
Expect(0, 1840, '\P{^Is_Joining_Group=dalathrish}', "");
Expect(1, 1839, '\p{Is_Joining_Group=-dalath_Rish}', "");
Expect(0, 1839, '\p{^Is_Joining_Group=-dalath_Rish}', "");
Expect(0, 1839, '\P{Is_Joining_Group=-dalath_Rish}', "");
Expect(1, 1839, '\P{^Is_Joining_Group=-dalath_Rish}', "");
Expect(0, 1840, '\p{Is_Joining_Group=-dalath_Rish}', "");
Expect(1, 1840, '\p{^Is_Joining_Group=-dalath_Rish}', "");
Expect(1, 1840, '\P{Is_Joining_Group=-dalath_Rish}', "");
Expect(0, 1840, '\P{^Is_Joining_Group=-dalath_Rish}', "");
Error('\p{Is_Jg:	:=Dalath_rish}');
Error('\P{Is_Jg:	:=Dalath_rish}');
Expect(1, 1839, '\p{Is_Jg:	dalathrish}', "");
Expect(0, 1839, '\p{^Is_Jg:	dalathrish}', "");
Expect(0, 1839, '\P{Is_Jg:	dalathrish}', "");
Expect(1, 1839, '\P{^Is_Jg:	dalathrish}', "");
Expect(0, 1840, '\p{Is_Jg:	dalathrish}', "");
Expect(1, 1840, '\p{^Is_Jg:	dalathrish}', "");
Expect(1, 1840, '\P{Is_Jg:	dalathrish}', "");
Expect(0, 1840, '\P{^Is_Jg:	dalathrish}', "");
Expect(1, 1839, '\p{Is_Jg=__DALATH_rish}', "");
Expect(0, 1839, '\p{^Is_Jg=__DALATH_rish}', "");
Expect(0, 1839, '\P{Is_Jg=__DALATH_rish}', "");
Expect(1, 1839, '\P{^Is_Jg=__DALATH_rish}', "");
Expect(0, 1840, '\p{Is_Jg=__DALATH_rish}', "");
Expect(1, 1840, '\p{^Is_Jg=__DALATH_rish}', "");
Expect(1, 1840, '\P{Is_Jg=__DALATH_rish}', "");
Expect(0, 1840, '\P{^Is_Jg=__DALATH_rish}', "");
Error('\p{Joining_Group=_:=Yudh_He}');
Error('\P{Joining_Group=_:=Yudh_He}');
Expect(1, 1822, '\p{Joining_Group:	yudhhe}', "");
Expect(0, 1822, '\p{^Joining_Group:	yudhhe}', "");
Expect(0, 1822, '\P{Joining_Group:	yudhhe}', "");
Expect(1, 1822, '\P{^Joining_Group:	yudhhe}', "");
Expect(0, 1823, '\p{Joining_Group:	yudhhe}', "");
Expect(1, 1823, '\p{^Joining_Group:	yudhhe}', "");
Expect(1, 1823, '\P{Joining_Group:	yudhhe}', "");
Expect(0, 1823, '\P{^Joining_Group:	yudhhe}', "");
Expect(1, 1822, '\p{Joining_Group=_Yudh_He}', "");
Expect(0, 1822, '\p{^Joining_Group=_Yudh_He}', "");
Expect(0, 1822, '\P{Joining_Group=_Yudh_He}', "");
Expect(1, 1822, '\P{^Joining_Group=_Yudh_He}', "");
Expect(0, 1823, '\p{Joining_Group=_Yudh_He}', "");
Expect(1, 1823, '\p{^Joining_Group=_Yudh_He}', "");
Expect(1, 1823, '\P{Joining_Group=_Yudh_He}', "");
Expect(0, 1823, '\P{^Joining_Group=_Yudh_He}', "");
Error('\p{Jg=/a/Yudh_he}');
Error('\P{Jg=/a/Yudh_he}');
Expect(1, 1822, '\p{Jg=yudhhe}', "");
Expect(0, 1822, '\p{^Jg=yudhhe}', "");
Expect(0, 1822, '\P{Jg=yudhhe}', "");
Expect(1, 1822, '\P{^Jg=yudhhe}', "");
Expect(0, 1823, '\p{Jg=yudhhe}', "");
Expect(1, 1823, '\p{^Jg=yudhhe}', "");
Expect(1, 1823, '\P{Jg=yudhhe}', "");
Expect(0, 1823, '\P{^Jg=yudhhe}', "");
Expect(1, 1822, '\p{Jg=  Yudh_He}', "");
Expect(0, 1822, '\p{^Jg=  Yudh_He}', "");
Expect(0, 1822, '\P{Jg=  Yudh_He}', "");
Expect(1, 1822, '\P{^Jg=  Yudh_He}', "");
Expect(0, 1823, '\p{Jg=  Yudh_He}', "");
Expect(1, 1823, '\p{^Jg=  Yudh_He}', "");
Expect(1, 1823, '\P{Jg=  Yudh_He}', "");
Expect(0, 1823, '\P{^Jg=  Yudh_He}', "");
Error('\p{Is_Joining_Group= -YUDH_he/a/}');
Error('\P{Is_Joining_Group= -YUDH_he/a/}');
Expect(1, 1822, '\p{Is_Joining_Group: yudhhe}', "");
Expect(0, 1822, '\p{^Is_Joining_Group: yudhhe}', "");
Expect(0, 1822, '\P{Is_Joining_Group: yudhhe}', "");
Expect(1, 1822, '\P{^Is_Joining_Group: yudhhe}', "");
Expect(0, 1823, '\p{Is_Joining_Group: yudhhe}', "");
Expect(1, 1823, '\p{^Is_Joining_Group: yudhhe}', "");
Expect(1, 1823, '\P{Is_Joining_Group: yudhhe}', "");
Expect(0, 1823, '\P{^Is_Joining_Group: yudhhe}', "");
Expect(1, 1822, '\p{Is_Joining_Group=  yudh_He}', "");
Expect(0, 1822, '\p{^Is_Joining_Group=  yudh_He}', "");
Expect(0, 1822, '\P{Is_Joining_Group=  yudh_He}', "");
Expect(1, 1822, '\P{^Is_Joining_Group=  yudh_He}', "");
Expect(0, 1823, '\p{Is_Joining_Group=  yudh_He}', "");
Expect(1, 1823, '\p{^Is_Joining_Group=  yudh_He}', "");
Expect(1, 1823, '\P{Is_Joining_Group=  yudh_He}', "");
Expect(0, 1823, '\P{^Is_Joining_Group=  yudh_He}', "");
Error('\p{Is_Jg=:=--Yudh_HE}');
Error('\P{Is_Jg=:=--Yudh_HE}');
Expect(1, 1822, '\p{Is_Jg=yudhhe}', "");
Expect(0, 1822, '\p{^Is_Jg=yudhhe}', "");
Expect(0, 1822, '\P{Is_Jg=yudhhe}', "");
Expect(1, 1822, '\P{^Is_Jg=yudhhe}', "");
Expect(0, 1823, '\p{Is_Jg=yudhhe}', "");
Expect(1, 1823, '\p{^Is_Jg=yudhhe}', "");
Expect(1, 1823, '\P{Is_Jg=yudhhe}', "");
Expect(0, 1823, '\P{^Is_Jg=yudhhe}', "");
Expect(1, 1822, '\p{Is_Jg=- YUDH_He}', "");
Expect(0, 1822, '\p{^Is_Jg=- YUDH_He}', "");
Expect(0, 1822, '\P{Is_Jg=- YUDH_He}', "");
Expect(1, 1822, '\P{^Is_Jg=- YUDH_He}', "");
Expect(0, 1823, '\p{Is_Jg=- YUDH_He}', "");
Expect(1, 1823, '\p{^Is_Jg=- YUDH_He}', "");
Expect(1, 1823, '\P{Is_Jg=- YUDH_He}', "");
Expect(0, 1823, '\P{^Is_Jg=- YUDH_He}', "");
Error('\p{Joining_Group=/a/Noon}');
Error('\P{Joining_Group=/a/Noon}');
Expect(1, 1897, '\p{Joining_Group=noon}', "");
Expect(0, 1897, '\p{^Joining_Group=noon}', "");
Expect(0, 1897, '\P{Joining_Group=noon}', "");
Expect(1, 1897, '\P{^Joining_Group=noon}', "");
Expect(0, 1898, '\p{Joining_Group=noon}', "");
Expect(1, 1898, '\p{^Joining_Group=noon}', "");
Expect(1, 1898, '\P{Joining_Group=noon}', "");
Expect(0, 1898, '\P{^Joining_Group=noon}', "");
Expect(1, 1897, '\p{Joining_Group=--NOON}', "");
Expect(0, 1897, '\p{^Joining_Group=--NOON}', "");
Expect(0, 1897, '\P{Joining_Group=--NOON}', "");
Expect(1, 1897, '\P{^Joining_Group=--NOON}', "");
Expect(0, 1898, '\p{Joining_Group=--NOON}', "");
Expect(1, 1898, '\p{^Joining_Group=--NOON}', "");
Expect(1, 1898, '\P{Joining_Group=--NOON}', "");
Expect(0, 1898, '\P{^Joining_Group=--NOON}', "");
Error('\p{Jg=:=-Noon}');
Error('\P{Jg=:=-Noon}');
Expect(1, 1897, '\p{Jg:   noon}', "");
Expect(0, 1897, '\p{^Jg:   noon}', "");
Expect(0, 1897, '\P{Jg:   noon}', "");
Expect(1, 1897, '\P{^Jg:   noon}', "");
Expect(0, 1898, '\p{Jg:   noon}', "");
Expect(1, 1898, '\p{^Jg:   noon}', "");
Expect(1, 1898, '\P{Jg:   noon}', "");
Expect(0, 1898, '\P{^Jg:   noon}', "");
Expect(1, 1897, '\p{Jg=-_Noon}', "");
Expect(0, 1897, '\p{^Jg=-_Noon}', "");
Expect(0, 1897, '\P{Jg=-_Noon}', "");
Expect(1, 1897, '\P{^Jg=-_Noon}', "");
Expect(0, 1898, '\p{Jg=-_Noon}', "");
Expect(1, 1898, '\p{^Jg=-_Noon}', "");
Expect(1, 1898, '\P{Jg=-_Noon}', "");
Expect(0, 1898, '\P{^Jg=-_Noon}', "");
Error('\p{Is_Joining_Group:   	NOON/a/}');
Error('\P{Is_Joining_Group:   	NOON/a/}');
Expect(1, 1897, '\p{Is_Joining_Group=noon}', "");
Expect(0, 1897, '\p{^Is_Joining_Group=noon}', "");
Expect(0, 1897, '\P{Is_Joining_Group=noon}', "");
Expect(1, 1897, '\P{^Is_Joining_Group=noon}', "");
Expect(0, 1898, '\p{Is_Joining_Group=noon}', "");
Expect(1, 1898, '\p{^Is_Joining_Group=noon}', "");
Expect(1, 1898, '\P{Is_Joining_Group=noon}', "");
Expect(0, 1898, '\P{^Is_Joining_Group=noon}', "");
Expect(1, 1897, '\p{Is_Joining_Group=	-noon}', "");
Expect(0, 1897, '\p{^Is_Joining_Group=	-noon}', "");
Expect(0, 1897, '\P{Is_Joining_Group=	-noon}', "");
Expect(1, 1897, '\P{^Is_Joining_Group=	-noon}', "");
Expect(0, 1898, '\p{Is_Joining_Group=	-noon}', "");
Expect(1, 1898, '\p{^Is_Joining_Group=	-noon}', "");
Expect(1, 1898, '\P{Is_Joining_Group=	-noon}', "");
Expect(0, 1898, '\P{^Is_Joining_Group=	-noon}', "");
Error('\p{Is_Jg=	/a/Noon}');
Error('\P{Is_Jg=	/a/Noon}');
Expect(1, 1897, '\p{Is_Jg=noon}', "");
Expect(0, 1897, '\p{^Is_Jg=noon}', "");
Expect(0, 1897, '\P{Is_Jg=noon}', "");
Expect(1, 1897, '\P{^Is_Jg=noon}', "");
Expect(0, 1898, '\p{Is_Jg=noon}', "");
Expect(1, 1898, '\p{^Is_Jg=noon}', "");
Expect(1, 1898, '\P{Is_Jg=noon}', "");
Expect(0, 1898, '\P{^Is_Jg=noon}', "");
Expect(1, 1897, '\p{Is_Jg=Noon}', "");
Expect(0, 1897, '\p{^Is_Jg=Noon}', "");
Expect(0, 1897, '\P{Is_Jg=Noon}', "");
Expect(1, 1897, '\P{^Is_Jg=Noon}', "");
Expect(0, 1898, '\p{Is_Jg=Noon}', "");
Expect(1, 1898, '\p{^Is_Jg=Noon}', "");
Expect(1, 1898, '\P{Is_Jg=Noon}', "");
Expect(0, 1898, '\P{^Is_Jg=Noon}', "");
Error('\p{Joining_Group=	:=Mim}');
Error('\P{Joining_Group=	:=Mim}');
Expect(1, 1825, '\p{Joining_Group=mim}', "");
Expect(0, 1825, '\p{^Joining_Group=mim}', "");
Expect(0, 1825, '\P{Joining_Group=mim}', "");
Expect(1, 1825, '\P{^Joining_Group=mim}', "");
Expect(0, 1826, '\p{Joining_Group=mim}', "");
Expect(1, 1826, '\p{^Joining_Group=mim}', "");
Expect(1, 1826, '\P{Joining_Group=mim}', "");
Expect(0, 1826, '\P{^Joining_Group=mim}', "");
Expect(1, 1825, '\p{Joining_Group= mim}', "");
Expect(0, 1825, '\p{^Joining_Group= mim}', "");
Expect(0, 1825, '\P{Joining_Group= mim}', "");
Expect(1, 1825, '\P{^Joining_Group= mim}', "");
Expect(0, 1826, '\p{Joining_Group= mim}', "");
Expect(1, 1826, '\p{^Joining_Group= mim}', "");
Expect(1, 1826, '\P{Joining_Group= mim}', "");
Expect(0, 1826, '\P{^Joining_Group= mim}', "");
Error('\p{Jg=:=MIM}');
Error('\P{Jg=:=MIM}');
Expect(1, 1825, '\p{Jg=mim}', "");
Expect(0, 1825, '\p{^Jg=mim}', "");
Expect(0, 1825, '\P{Jg=mim}', "");
Expect(1, 1825, '\P{^Jg=mim}', "");
Expect(0, 1826, '\p{Jg=mim}', "");
Expect(1, 1826, '\p{^Jg=mim}', "");
Expect(1, 1826, '\P{Jg=mim}', "");
Expect(0, 1826, '\P{^Jg=mim}', "");
Expect(1, 1825, '\p{Jg=_-MIM}', "");
Expect(0, 1825, '\p{^Jg=_-MIM}', "");
Expect(0, 1825, '\P{Jg=_-MIM}', "");
Expect(1, 1825, '\P{^Jg=_-MIM}', "");
Expect(0, 1826, '\p{Jg=_-MIM}', "");
Expect(1, 1826, '\p{^Jg=_-MIM}', "");
Expect(1, 1826, '\P{Jg=_-MIM}', "");
Expect(0, 1826, '\P{^Jg=_-MIM}', "");
Error('\p{Is_Joining_Group=- Mim:=}');
Error('\P{Is_Joining_Group=- Mim:=}');
Expect(1, 1825, '\p{Is_Joining_Group=mim}', "");
Expect(0, 1825, '\p{^Is_Joining_Group=mim}', "");
Expect(0, 1825, '\P{Is_Joining_Group=mim}', "");
Expect(1, 1825, '\P{^Is_Joining_Group=mim}', "");
Expect(0, 1826, '\p{Is_Joining_Group=mim}', "");
Expect(1, 1826, '\p{^Is_Joining_Group=mim}', "");
Expect(1, 1826, '\P{Is_Joining_Group=mim}', "");
Expect(0, 1826, '\P{^Is_Joining_Group=mim}', "");
Expect(1, 1825, '\p{Is_Joining_Group=_ MIM}', "");
Expect(0, 1825, '\p{^Is_Joining_Group=_ MIM}', "");
Expect(0, 1825, '\P{Is_Joining_Group=_ MIM}', "");
Expect(1, 1825, '\P{^Is_Joining_Group=_ MIM}', "");
Expect(0, 1826, '\p{Is_Joining_Group=_ MIM}', "");
Expect(1, 1826, '\p{^Is_Joining_Group=_ MIM}', "");
Expect(1, 1826, '\P{Is_Joining_Group=_ MIM}', "");
Expect(0, 1826, '\P{^Is_Joining_Group=_ MIM}', "");
Error('\p{Is_Jg=__Mim:=}');
Error('\P{Is_Jg=__Mim:=}');
Expect(1, 1825, '\p{Is_Jg=mim}', "");
Expect(0, 1825, '\p{^Is_Jg=mim}', "");
Expect(0, 1825, '\P{Is_Jg=mim}', "");
Expect(1, 1825, '\P{^Is_Jg=mim}', "");
Expect(0, 1826, '\p{Is_Jg=mim}', "");
Expect(1, 1826, '\p{^Is_Jg=mim}', "");
Expect(1, 1826, '\P{Is_Jg=mim}', "");
Expect(0, 1826, '\P{^Is_Jg=mim}', "");
Expect(1, 1825, '\p{Is_Jg=-_MIM}', "");
Expect(0, 1825, '\p{^Is_Jg=-_MIM}', "");
Expect(0, 1825, '\P{Is_Jg=-_MIM}', "");
Expect(1, 1825, '\P{^Is_Jg=-_MIM}', "");
Expect(0, 1826, '\p{Is_Jg=-_MIM}', "");
Expect(1, 1826, '\p{^Is_Jg=-_MIM}', "");
Expect(1, 1826, '\P{Is_Jg=-_MIM}', "");
Expect(0, 1826, '\P{^Is_Jg=-_MIM}', "");
Error('\p{Joining_Group=_ HEH_Goal/a/}');
Error('\P{Joining_Group=_ HEH_Goal/a/}');
Expect(1, 1730, '\p{Joining_Group=hehgoal}', "");
Expect(0, 1730, '\p{^Joining_Group=hehgoal}', "");
Expect(0, 1730, '\P{Joining_Group=hehgoal}', "");
Expect(1, 1730, '\P{^Joining_Group=hehgoal}', "");
Expect(0, 1731, '\p{Joining_Group=hehgoal}', "");
Expect(1, 1731, '\p{^Joining_Group=hehgoal}', "");
Expect(1, 1731, '\P{Joining_Group=hehgoal}', "");
Expect(0, 1731, '\P{^Joining_Group=hehgoal}', "");
Expect(1, 1730, '\p{Joining_Group=_	Heh_goal}', "");
Expect(0, 1730, '\p{^Joining_Group=_	Heh_goal}', "");
Expect(0, 1730, '\P{Joining_Group=_	Heh_goal}', "");
Expect(1, 1730, '\P{^Joining_Group=_	Heh_goal}', "");
Expect(0, 1731, '\p{Joining_Group=_	Heh_goal}', "");
Expect(1, 1731, '\p{^Joining_Group=_	Heh_goal}', "");
Expect(1, 1731, '\P{Joining_Group=_	Heh_goal}', "");
Expect(0, 1731, '\P{^Joining_Group=_	Heh_goal}', "");
Error('\p{Jg:    _Heh_Goal:=}');
Error('\P{Jg:    _Heh_Goal:=}');
Expect(1, 1730, '\p{Jg=hehgoal}', "");
Expect(0, 1730, '\p{^Jg=hehgoal}', "");
Expect(0, 1730, '\P{Jg=hehgoal}', "");
Expect(1, 1730, '\P{^Jg=hehgoal}', "");
Expect(0, 1731, '\p{Jg=hehgoal}', "");
Expect(1, 1731, '\p{^Jg=hehgoal}', "");
Expect(1, 1731, '\P{Jg=hehgoal}', "");
Expect(0, 1731, '\P{^Jg=hehgoal}', "");
Expect(1, 1730, '\p{Jg=_-Heh_GOAL}', "");
Expect(0, 1730, '\p{^Jg=_-Heh_GOAL}', "");
Expect(0, 1730, '\P{Jg=_-Heh_GOAL}', "");
Expect(1, 1730, '\P{^Jg=_-Heh_GOAL}', "");
Expect(0, 1731, '\p{Jg=_-Heh_GOAL}', "");
Expect(1, 1731, '\p{^Jg=_-Heh_GOAL}', "");
Expect(1, 1731, '\P{Jg=_-Heh_GOAL}', "");
Expect(0, 1731, '\P{^Jg=_-Heh_GOAL}', "");
Error('\p{Is_Joining_Group=/a/_	Heh_Goal}');
Error('\P{Is_Joining_Group=/a/_	Heh_Goal}');
Expect(1, 1730, '\p{Is_Joining_Group=hehgoal}', "");
Expect(0, 1730, '\p{^Is_Joining_Group=hehgoal}', "");
Expect(0, 1730, '\P{Is_Joining_Group=hehgoal}', "");
Expect(1, 1730, '\P{^Is_Joining_Group=hehgoal}', "");
Expect(0, 1731, '\p{Is_Joining_Group=hehgoal}', "");
Expect(1, 1731, '\p{^Is_Joining_Group=hehgoal}', "");
Expect(1, 1731, '\P{Is_Joining_Group=hehgoal}', "");
Expect(0, 1731, '\P{^Is_Joining_Group=hehgoal}', "");
Expect(1, 1730, '\p{Is_Joining_Group=--heh_GOAL}', "");
Expect(0, 1730, '\p{^Is_Joining_Group=--heh_GOAL}', "");
Expect(0, 1730, '\P{Is_Joining_Group=--heh_GOAL}', "");
Expect(1, 1730, '\P{^Is_Joining_Group=--heh_GOAL}', "");
Expect(0, 1731, '\p{Is_Joining_Group=--heh_GOAL}', "");
Expect(1, 1731, '\p{^Is_Joining_Group=--heh_GOAL}', "");
Expect(1, 1731, '\P{Is_Joining_Group=--heh_GOAL}', "");
Expect(0, 1731, '\P{^Is_Joining_Group=--heh_GOAL}', "");
Error('\p{Is_Jg:/a/HEH_goal}');
Error('\P{Is_Jg:/a/HEH_goal}');
Expect(1, 1730, '\p{Is_Jg=hehgoal}', "");
Expect(0, 1730, '\p{^Is_Jg=hehgoal}', "");
Expect(0, 1730, '\P{Is_Jg=hehgoal}', "");
Expect(1, 1730, '\P{^Is_Jg=hehgoal}', "");
Expect(0, 1731, '\p{Is_Jg=hehgoal}', "");
Expect(1, 1731, '\p{^Is_Jg=hehgoal}', "");
Expect(1, 1731, '\P{Is_Jg=hehgoal}', "");
Expect(0, 1731, '\P{^Is_Jg=hehgoal}', "");
Expect(1, 1730, '\p{Is_Jg=_HEH_Goal}', "");
Expect(0, 1730, '\p{^Is_Jg=_HEH_Goal}', "");
Expect(0, 1730, '\P{Is_Jg=_HEH_Goal}', "");
Expect(1, 1730, '\P{^Is_Jg=_HEH_Goal}', "");
Expect(0, 1731, '\p{Is_Jg=_HEH_Goal}', "");
Expect(1, 1731, '\p{^Is_Jg=_HEH_Goal}', "");
Expect(1, 1731, '\P{Is_Jg=_HEH_Goal}', "");
Expect(0, 1731, '\P{^Is_Jg=_HEH_Goal}', "");
Error('\p{Joining_Group= 	khaph/a/}');
Error('\P{Joining_Group= 	khaph/a/}');
Expect(1, 1870, '\p{Joining_Group=khaph}', "");
Expect(0, 1870, '\p{^Joining_Group=khaph}', "");
Expect(0, 1870, '\P{Joining_Group=khaph}', "");
Expect(1, 1870, '\P{^Joining_Group=khaph}', "");
Expect(0, 1871, '\p{Joining_Group=khaph}', "");
Expect(1, 1871, '\p{^Joining_Group=khaph}', "");
Expect(1, 1871, '\P{Joining_Group=khaph}', "");
Expect(0, 1871, '\P{^Joining_Group=khaph}', "");
Expect(1, 1870, '\p{Joining_Group: -KHAPH}', "");
Expect(0, 1870, '\p{^Joining_Group: -KHAPH}', "");
Expect(0, 1870, '\P{Joining_Group: -KHAPH}', "");
Expect(1, 1870, '\P{^Joining_Group: -KHAPH}', "");
Expect(0, 1871, '\p{Joining_Group: -KHAPH}', "");
Expect(1, 1871, '\p{^Joining_Group: -KHAPH}', "");
Expect(1, 1871, '\P{Joining_Group: -KHAPH}', "");
Expect(0, 1871, '\P{^Joining_Group: -KHAPH}', "");
Error('\p{Jg=:=		Khaph}');
Error('\P{Jg=:=		Khaph}');
Expect(1, 1870, '\p{Jg=khaph}', "");
Expect(0, 1870, '\p{^Jg=khaph}', "");
Expect(0, 1870, '\P{Jg=khaph}', "");
Expect(1, 1870, '\P{^Jg=khaph}', "");
Expect(0, 1871, '\p{Jg=khaph}', "");
Expect(1, 1871, '\p{^Jg=khaph}', "");
Expect(1, 1871, '\P{Jg=khaph}', "");
Expect(0, 1871, '\P{^Jg=khaph}', "");
Expect(1, 1870, '\p{Jg=KHAPH}', "");
Expect(0, 1870, '\p{^Jg=KHAPH}', "");
Expect(0, 1870, '\P{Jg=KHAPH}', "");
Expect(1, 1870, '\P{^Jg=KHAPH}', "");
Expect(0, 1871, '\p{Jg=KHAPH}', "");
Expect(1, 1871, '\p{^Jg=KHAPH}', "");
Expect(1, 1871, '\P{Jg=KHAPH}', "");
Expect(0, 1871, '\P{^Jg=KHAPH}', "");
Error('\p{Is_Joining_Group= :=Khaph}');
Error('\P{Is_Joining_Group= :=Khaph}');
Expect(1, 1870, '\p{Is_Joining_Group=khaph}', "");
Expect(0, 1870, '\p{^Is_Joining_Group=khaph}', "");
Expect(0, 1870, '\P{Is_Joining_Group=khaph}', "");
Expect(1, 1870, '\P{^Is_Joining_Group=khaph}', "");
Expect(0, 1871, '\p{Is_Joining_Group=khaph}', "");
Expect(1, 1871, '\p{^Is_Joining_Group=khaph}', "");
Expect(1, 1871, '\P{Is_Joining_Group=khaph}', "");
Expect(0, 1871, '\P{^Is_Joining_Group=khaph}', "");
Expect(1, 1870, '\p{Is_Joining_Group=_-Khaph}', "");
Expect(0, 1870, '\p{^Is_Joining_Group=_-Khaph}', "");
Expect(0, 1870, '\P{Is_Joining_Group=_-Khaph}', "");
Expect(1, 1870, '\P{^Is_Joining_Group=_-Khaph}', "");
Expect(0, 1871, '\p{Is_Joining_Group=_-Khaph}', "");
Expect(1, 1871, '\p{^Is_Joining_Group=_-Khaph}', "");
Expect(1, 1871, '\P{Is_Joining_Group=_-Khaph}', "");
Expect(0, 1871, '\P{^Is_Joining_Group=_-Khaph}', "");
Error('\p{Is_Jg=:=	KHAPH}');
Error('\P{Is_Jg=:=	KHAPH}');
Expect(1, 1870, '\p{Is_Jg=khaph}', "");
Expect(0, 1870, '\p{^Is_Jg=khaph}', "");
Expect(0, 1870, '\P{Is_Jg=khaph}', "");
Expect(1, 1870, '\P{^Is_Jg=khaph}', "");
Expect(0, 1871, '\p{Is_Jg=khaph}', "");
Expect(1, 1871, '\p{^Is_Jg=khaph}', "");
Expect(1, 1871, '\P{Is_Jg=khaph}', "");
Expect(0, 1871, '\P{^Is_Jg=khaph}', "");
Expect(1, 1870, '\p{Is_Jg=	_KHAPH}', "");
Expect(0, 1870, '\p{^Is_Jg=	_KHAPH}', "");
Expect(0, 1870, '\P{Is_Jg=	_KHAPH}', "");
Expect(1, 1870, '\P{^Is_Jg=	_KHAPH}', "");
Expect(0, 1871, '\p{Is_Jg=	_KHAPH}', "");
Expect(1, 1871, '\p{^Is_Jg=	_KHAPH}', "");
Expect(1, 1871, '\P{Is_Jg=	_KHAPH}', "");
Expect(0, 1871, '\P{^Is_Jg=	_KHAPH}', "");
Error('\p{Joining_Group=/a/_SAD}');
Error('\P{Joining_Group=/a/_SAD}');
Expect(1, 1787, '\p{Joining_Group=sad}', "");
Expect(0, 1787, '\p{^Joining_Group=sad}', "");
Expect(0, 1787, '\P{Joining_Group=sad}', "");
Expect(1, 1787, '\P{^Joining_Group=sad}', "");
Expect(0, 1788, '\p{Joining_Group=sad}', "");
Expect(1, 1788, '\p{^Joining_Group=sad}', "");
Expect(1, 1788, '\P{Joining_Group=sad}', "");
Expect(0, 1788, '\P{^Joining_Group=sad}', "");
Expect(1, 1787, '\p{Joining_Group=_Sad}', "");
Expect(0, 1787, '\p{^Joining_Group=_Sad}', "");
Expect(0, 1787, '\P{Joining_Group=_Sad}', "");
Expect(1, 1787, '\P{^Joining_Group=_Sad}', "");
Expect(0, 1788, '\p{Joining_Group=_Sad}', "");
Expect(1, 1788, '\p{^Joining_Group=_Sad}', "");
Expect(1, 1788, '\P{Joining_Group=_Sad}', "");
Expect(0, 1788, '\P{^Joining_Group=_Sad}', "");
Error('\p{Jg=_sad/a/}');
Error('\P{Jg=_sad/a/}');
Expect(1, 1787, '\p{Jg=sad}', "");
Expect(0, 1787, '\p{^Jg=sad}', "");
Expect(0, 1787, '\P{Jg=sad}', "");
Expect(1, 1787, '\P{^Jg=sad}', "");
Expect(0, 1788, '\p{Jg=sad}', "");
Expect(1, 1788, '\p{^Jg=sad}', "");
Expect(1, 1788, '\P{Jg=sad}', "");
Expect(0, 1788, '\P{^Jg=sad}', "");
Expect(1, 1787, '\p{Jg=_ Sad}', "");
Expect(0, 1787, '\p{^Jg=_ Sad}', "");
Expect(0, 1787, '\P{Jg=_ Sad}', "");
Expect(1, 1787, '\P{^Jg=_ Sad}', "");
Expect(0, 1788, '\p{Jg=_ Sad}', "");
Expect(1, 1788, '\p{^Jg=_ Sad}', "");
Expect(1, 1788, '\P{Jg=_ Sad}', "");
Expect(0, 1788, '\P{^Jg=_ Sad}', "");
Error('\p{Is_Joining_Group=  SAD/a/}');
Error('\P{Is_Joining_Group=  SAD/a/}');
Expect(1, 1787, '\p{Is_Joining_Group=sad}', "");
Expect(0, 1787, '\p{^Is_Joining_Group=sad}', "");
Expect(0, 1787, '\P{Is_Joining_Group=sad}', "");
Expect(1, 1787, '\P{^Is_Joining_Group=sad}', "");
Expect(0, 1788, '\p{Is_Joining_Group=sad}', "");
Expect(1, 1788, '\p{^Is_Joining_Group=sad}', "");
Expect(1, 1788, '\P{Is_Joining_Group=sad}', "");
Expect(0, 1788, '\P{^Is_Joining_Group=sad}', "");
Expect(1, 1787, '\p{Is_Joining_Group: - Sad}', "");
Expect(0, 1787, '\p{^Is_Joining_Group: - Sad}', "");
Expect(0, 1787, '\P{Is_Joining_Group: - Sad}', "");
Expect(1, 1787, '\P{^Is_Joining_Group: - Sad}', "");
Expect(0, 1788, '\p{Is_Joining_Group: - Sad}', "");
Expect(1, 1788, '\p{^Is_Joining_Group: - Sad}', "");
Expect(1, 1788, '\P{Is_Joining_Group: - Sad}', "");
Expect(0, 1788, '\P{^Is_Joining_Group: - Sad}', "");
Error('\p{Is_Jg=:=SAD}');
Error('\P{Is_Jg=:=SAD}');
Expect(1, 1787, '\p{Is_Jg=sad}', "");
Expect(0, 1787, '\p{^Is_Jg=sad}', "");
Expect(0, 1787, '\P{Is_Jg=sad}', "");
Expect(1, 1787, '\P{^Is_Jg=sad}', "");
Expect(0, 1788, '\p{Is_Jg=sad}', "");
Expect(1, 1788, '\p{^Is_Jg=sad}', "");
Expect(1, 1788, '\P{Is_Jg=sad}', "");
Expect(0, 1788, '\P{^Is_Jg=sad}', "");
Expect(1, 1787, '\p{Is_Jg=_ Sad}', "");
Expect(0, 1787, '\p{^Is_Jg=_ Sad}', "");
Expect(0, 1787, '\P{Is_Jg=_ Sad}', "");
Expect(1, 1787, '\P{^Is_Jg=_ Sad}', "");
Expect(0, 1788, '\p{Is_Jg=_ Sad}', "");
Expect(1, 1788, '\p{^Is_Jg=_ Sad}', "");
Expect(1, 1788, '\P{Is_Jg=_ Sad}', "");
Expect(0, 1788, '\P{^Is_Jg=_ Sad}', "");
Error('\p{Joining_Group: /a/ -QAF}');
Error('\P{Joining_Group: /a/ -QAF}');
Expect(1, 2213, '\p{Joining_Group:qaf}', "");
Expect(0, 2213, '\p{^Joining_Group:qaf}', "");
Expect(0, 2213, '\P{Joining_Group:qaf}', "");
Expect(1, 2213, '\P{^Joining_Group:qaf}', "");
Expect(0, 2214, '\p{Joining_Group:qaf}', "");
Expect(1, 2214, '\p{^Joining_Group:qaf}', "");
Expect(1, 2214, '\P{Joining_Group:qaf}', "");
Expect(0, 2214, '\P{^Joining_Group:qaf}', "");
Expect(1, 2213, '\p{Joining_Group=-Qaf}', "");
Expect(0, 2213, '\p{^Joining_Group=-Qaf}', "");
Expect(0, 2213, '\P{Joining_Group=-Qaf}', "");
Expect(1, 2213, '\P{^Joining_Group=-Qaf}', "");
Expect(0, 2214, '\p{Joining_Group=-Qaf}', "");
Expect(1, 2214, '\p{^Joining_Group=-Qaf}', "");
Expect(1, 2214, '\P{Joining_Group=-Qaf}', "");
Expect(0, 2214, '\P{^Joining_Group=-Qaf}', "");
Error('\p{Jg=-QAF:=}');
Error('\P{Jg=-QAF:=}');
Expect(1, 2213, '\p{Jg=qaf}', "");
Expect(0, 2213, '\p{^Jg=qaf}', "");
Expect(0, 2213, '\P{Jg=qaf}', "");
Expect(1, 2213, '\P{^Jg=qaf}', "");
Expect(0, 2214, '\p{Jg=qaf}', "");
Expect(1, 2214, '\p{^Jg=qaf}', "");
Expect(1, 2214, '\P{Jg=qaf}', "");
Expect(0, 2214, '\P{^Jg=qaf}', "");
Expect(1, 2213, '\p{Jg=	QAF}', "");
Expect(0, 2213, '\p{^Jg=	QAF}', "");
Expect(0, 2213, '\P{Jg=	QAF}', "");
Expect(1, 2213, '\P{^Jg=	QAF}', "");
Expect(0, 2214, '\p{Jg=	QAF}', "");
Expect(1, 2214, '\p{^Jg=	QAF}', "");
Expect(1, 2214, '\P{Jg=	QAF}', "");
Expect(0, 2214, '\P{^Jg=	QAF}', "");
Error('\p{Is_Joining_Group=__QAF:=}');
Error('\P{Is_Joining_Group=__QAF:=}');
Expect(1, 2213, '\p{Is_Joining_Group:   qaf}', "");
Expect(0, 2213, '\p{^Is_Joining_Group:   qaf}', "");
Expect(0, 2213, '\P{Is_Joining_Group:   qaf}', "");
Expect(1, 2213, '\P{^Is_Joining_Group:   qaf}', "");
Expect(0, 2214, '\p{Is_Joining_Group:   qaf}', "");
Expect(1, 2214, '\p{^Is_Joining_Group:   qaf}', "");
Expect(1, 2214, '\P{Is_Joining_Group:   qaf}', "");
Expect(0, 2214, '\P{^Is_Joining_Group:   qaf}', "");
Expect(1, 2213, '\p{Is_Joining_Group:	__Qaf}', "");
Expect(0, 2213, '\p{^Is_Joining_Group:	__Qaf}', "");
Expect(0, 2213, '\P{Is_Joining_Group:	__Qaf}', "");
Expect(1, 2213, '\P{^Is_Joining_Group:	__Qaf}', "");
Expect(0, 2214, '\p{Is_Joining_Group:	__Qaf}', "");
Expect(1, 2214, '\p{^Is_Joining_Group:	__Qaf}', "");
Expect(1, 2214, '\P{Is_Joining_Group:	__Qaf}', "");
Expect(0, 2214, '\P{^Is_Joining_Group:	__Qaf}', "");
Error('\p{Is_Jg=/a/qaf}');
Error('\P{Is_Jg=/a/qaf}');
Expect(1, 2213, '\p{Is_Jg=qaf}', "");
Expect(0, 2213, '\p{^Is_Jg=qaf}', "");
Expect(0, 2213, '\P{Is_Jg=qaf}', "");
Expect(1, 2213, '\P{^Is_Jg=qaf}', "");
Expect(0, 2214, '\p{Is_Jg=qaf}', "");
Expect(1, 2214, '\p{^Is_Jg=qaf}', "");
Expect(1, 2214, '\P{Is_Jg=qaf}', "");
Expect(0, 2214, '\P{^Is_Jg=qaf}', "");
Expect(1, 2213, '\p{Is_Jg=  Qaf}', "");
Expect(0, 2213, '\p{^Is_Jg=  Qaf}', "");
Expect(0, 2213, '\P{Is_Jg=  Qaf}', "");
Expect(1, 2213, '\P{^Is_Jg=  Qaf}', "");
Expect(0, 2214, '\p{Is_Jg=  Qaf}', "");
Expect(1, 2214, '\p{^Is_Jg=  Qaf}', "");
Expect(1, 2214, '\P{Is_Jg=  Qaf}', "");
Expect(0, 2214, '\P{^Is_Jg=  Qaf}', "");
Error('\p{Joining_Group= _kaph:=}');
Error('\P{Joining_Group= _kaph:=}');
Expect(1, 1823, '\p{Joining_Group=kaph}', "");
Expect(0, 1823, '\p{^Joining_Group=kaph}', "");
Expect(0, 1823, '\P{Joining_Group=kaph}', "");
Expect(1, 1823, '\P{^Joining_Group=kaph}', "");
Expect(0, 1824, '\p{Joining_Group=kaph}', "");
Expect(1, 1824, '\p{^Joining_Group=kaph}', "");
Expect(1, 1824, '\P{Joining_Group=kaph}', "");
Expect(0, 1824, '\P{^Joining_Group=kaph}', "");
Expect(1, 1823, '\p{Joining_Group=  Kaph}', "");
Expect(0, 1823, '\p{^Joining_Group=  Kaph}', "");
Expect(0, 1823, '\P{Joining_Group=  Kaph}', "");
Expect(1, 1823, '\P{^Joining_Group=  Kaph}', "");
Expect(0, 1824, '\p{Joining_Group=  Kaph}', "");
Expect(1, 1824, '\p{^Joining_Group=  Kaph}', "");
Expect(1, 1824, '\P{Joining_Group=  Kaph}', "");
Expect(0, 1824, '\P{^Joining_Group=  Kaph}', "");
Error('\p{Jg=KAPH/a/}');
Error('\P{Jg=KAPH/a/}');
Expect(1, 1823, '\p{Jg=kaph}', "");
Expect(0, 1823, '\p{^Jg=kaph}', "");
Expect(0, 1823, '\P{Jg=kaph}', "");
Expect(1, 1823, '\P{^Jg=kaph}', "");
Expect(0, 1824, '\p{Jg=kaph}', "");
Expect(1, 1824, '\p{^Jg=kaph}', "");
Expect(1, 1824, '\P{Jg=kaph}', "");
Expect(0, 1824, '\P{^Jg=kaph}', "");
Expect(1, 1823, '\p{Jg=-KAPH}', "");
Expect(0, 1823, '\p{^Jg=-KAPH}', "");
Expect(0, 1823, '\P{Jg=-KAPH}', "");
Expect(1, 1823, '\P{^Jg=-KAPH}', "");
Expect(0, 1824, '\p{Jg=-KAPH}', "");
Expect(1, 1824, '\p{^Jg=-KAPH}', "");
Expect(1, 1824, '\P{Jg=-KAPH}', "");
Expect(0, 1824, '\P{^Jg=-KAPH}', "");
Error('\p{Is_Joining_Group= :=Kaph}');
Error('\P{Is_Joining_Group= :=Kaph}');
Expect(1, 1823, '\p{Is_Joining_Group=kaph}', "");
Expect(0, 1823, '\p{^Is_Joining_Group=kaph}', "");
Expect(0, 1823, '\P{Is_Joining_Group=kaph}', "");
Expect(1, 1823, '\P{^Is_Joining_Group=kaph}', "");
Expect(0, 1824, '\p{Is_Joining_Group=kaph}', "");
Expect(1, 1824, '\p{^Is_Joining_Group=kaph}', "");
Expect(1, 1824, '\P{Is_Joining_Group=kaph}', "");
Expect(0, 1824, '\P{^Is_Joining_Group=kaph}', "");
Expect(1, 1823, '\p{Is_Joining_Group=	-KAPH}', "");
Expect(0, 1823, '\p{^Is_Joining_Group=	-KAPH}', "");
Expect(0, 1823, '\P{Is_Joining_Group=	-KAPH}', "");
Expect(1, 1823, '\P{^Is_Joining_Group=	-KAPH}', "");
Expect(0, 1824, '\p{Is_Joining_Group=	-KAPH}', "");
Expect(1, 1824, '\p{^Is_Joining_Group=	-KAPH}', "");
Expect(1, 1824, '\P{Is_Joining_Group=	-KAPH}', "");
Expect(0, 1824, '\P{^Is_Joining_Group=	-KAPH}', "");
Error('\p{Is_Jg=	_KAPH:=}');
Error('\P{Is_Jg=	_KAPH:=}');
Expect(1, 1823, '\p{Is_Jg=kaph}', "");
Expect(0, 1823, '\p{^Is_Jg=kaph}', "");
Expect(0, 1823, '\P{Is_Jg=kaph}', "");
Expect(1, 1823, '\P{^Is_Jg=kaph}', "");
Expect(0, 1824, '\p{Is_Jg=kaph}', "");
Expect(1, 1824, '\p{^Is_Jg=kaph}', "");
Expect(1, 1824, '\P{Is_Jg=kaph}', "");
Expect(0, 1824, '\P{^Is_Jg=kaph}', "");
Expect(1, 1823, '\p{Is_Jg=	_KAPH}', "");
Expect(0, 1823, '\p{^Is_Jg=	_KAPH}', "");
Expect(0, 1823, '\P{Is_Jg=	_KAPH}', "");
Expect(1, 1823, '\P{^Is_Jg=	_KAPH}', "");
Expect(0, 1824, '\p{Is_Jg=	_KAPH}', "");
Expect(1, 1824, '\p{^Is_Jg=	_KAPH}', "");
Expect(1, 1824, '\P{Is_Jg=	_KAPH}', "");
Expect(0, 1824, '\P{^Is_Jg=	_KAPH}', "");
Error('\p{Joining_Group= -Feh:=}');
Error('\P{Joining_Group= -Feh:=}');
Expect(1, 2212, '\p{Joining_Group=feh}', "");
Expect(0, 2212, '\p{^Joining_Group=feh}', "");
Expect(0, 2212, '\P{Joining_Group=feh}', "");
Expect(1, 2212, '\P{^Joining_Group=feh}', "");
Expect(0, 2213, '\p{Joining_Group=feh}', "");
Expect(1, 2213, '\p{^Joining_Group=feh}', "");
Expect(1, 2213, '\P{Joining_Group=feh}', "");
Expect(0, 2213, '\P{^Joining_Group=feh}', "");
Expect(1, 2212, '\p{Joining_Group= FEH}', "");
Expect(0, 2212, '\p{^Joining_Group= FEH}', "");
Expect(0, 2212, '\P{Joining_Group= FEH}', "");
Expect(1, 2212, '\P{^Joining_Group= FEH}', "");
Expect(0, 2213, '\p{Joining_Group= FEH}', "");
Expect(1, 2213, '\p{^Joining_Group= FEH}', "");
Expect(1, 2213, '\P{Joining_Group= FEH}', "");
Expect(0, 2213, '\P{^Joining_Group= FEH}', "");
Error('\p{Jg=_ FEH/a/}');
Error('\P{Jg=_ FEH/a/}');
Expect(1, 2212, '\p{Jg=feh}', "");
Expect(0, 2212, '\p{^Jg=feh}', "");
Expect(0, 2212, '\P{Jg=feh}', "");
Expect(1, 2212, '\P{^Jg=feh}', "");
Expect(0, 2213, '\p{Jg=feh}', "");
Expect(1, 2213, '\p{^Jg=feh}', "");
Expect(1, 2213, '\P{Jg=feh}', "");
Expect(0, 2213, '\P{^Jg=feh}', "");
Expect(1, 2212, '\p{Jg=-_Feh}', "");
Expect(0, 2212, '\p{^Jg=-_Feh}', "");
Expect(0, 2212, '\P{Jg=-_Feh}', "");
Expect(1, 2212, '\P{^Jg=-_Feh}', "");
Expect(0, 2213, '\p{Jg=-_Feh}', "");
Expect(1, 2213, '\p{^Jg=-_Feh}', "");
Expect(1, 2213, '\P{Jg=-_Feh}', "");
Expect(0, 2213, '\P{^Jg=-_Feh}', "");
Error('\p{Is_Joining_Group=/a/	feh}');
Error('\P{Is_Joining_Group=/a/	feh}');
Expect(1, 2212, '\p{Is_Joining_Group=feh}', "");
Expect(0, 2212, '\p{^Is_Joining_Group=feh}', "");
Expect(0, 2212, '\P{Is_Joining_Group=feh}', "");
Expect(1, 2212, '\P{^Is_Joining_Group=feh}', "");
Expect(0, 2213, '\p{Is_Joining_Group=feh}', "");
Expect(1, 2213, '\p{^Is_Joining_Group=feh}', "");
Expect(1, 2213, '\P{Is_Joining_Group=feh}', "");
Expect(0, 2213, '\P{^Is_Joining_Group=feh}', "");
Expect(1, 2212, '\p{Is_Joining_Group=_ FEH}', "");
Expect(0, 2212, '\p{^Is_Joining_Group=_ FEH}', "");
Expect(0, 2212, '\P{Is_Joining_Group=_ FEH}', "");
Expect(1, 2212, '\P{^Is_Joining_Group=_ FEH}', "");
Expect(0, 2213, '\p{Is_Joining_Group=_ FEH}', "");
Expect(1, 2213, '\p{^Is_Joining_Group=_ FEH}', "");
Expect(1, 2213, '\P{Is_Joining_Group=_ FEH}', "");
Expect(0, 2213, '\P{^Is_Joining_Group=_ FEH}', "");
Error('\p{Is_Jg=	/a/FEH}');
Error('\P{Is_Jg=	/a/FEH}');
Expect(1, 2212, '\p{Is_Jg=feh}', "");
Expect(0, 2212, '\p{^Is_Jg=feh}', "");
Expect(0, 2212, '\P{Is_Jg=feh}', "");
Expect(1, 2212, '\P{^Is_Jg=feh}', "");
Expect(0, 2213, '\p{Is_Jg=feh}', "");
Expect(1, 2213, '\p{^Is_Jg=feh}', "");
Expect(1, 2213, '\P{Is_Jg=feh}', "");
Expect(0, 2213, '\P{^Is_Jg=feh}', "");
Error('\p{Joining_Group=/a/-FARSI_Yeh}');
Error('\P{Joining_Group=/a/-FARSI_Yeh}');
Expect(1, 1910, '\p{Joining_Group=farsiyeh}', "");
Expect(0, 1910, '\p{^Joining_Group=farsiyeh}', "");
Expect(0, 1910, '\P{Joining_Group=farsiyeh}', "");
Expect(1, 1910, '\P{^Joining_Group=farsiyeh}', "");
Expect(0, 1911, '\p{Joining_Group=farsiyeh}', "");
Expect(1, 1911, '\p{^Joining_Group=farsiyeh}', "");
Expect(1, 1911, '\P{Joining_Group=farsiyeh}', "");
Expect(0, 1911, '\P{^Joining_Group=farsiyeh}', "");
Expect(1, 1910, '\p{Joining_Group=--FARSI_YEH}', "");
Expect(0, 1910, '\p{^Joining_Group=--FARSI_YEH}', "");
Expect(0, 1910, '\P{Joining_Group=--FARSI_YEH}', "");
Expect(1, 1910, '\P{^Joining_Group=--FARSI_YEH}', "");
Expect(0, 1911, '\p{Joining_Group=--FARSI_YEH}', "");
Expect(1, 1911, '\p{^Joining_Group=--FARSI_YEH}', "");
Expect(1, 1911, '\P{Joining_Group=--FARSI_YEH}', "");
Expect(0, 1911, '\P{^Joining_Group=--FARSI_YEH}', "");
Error('\p{Jg:   	/a/Farsi_yeh}');
Error('\P{Jg:   	/a/Farsi_yeh}');
Expect(1, 1910, '\p{Jg=farsiyeh}', "");
Expect(0, 1910, '\p{^Jg=farsiyeh}', "");
Expect(0, 1910, '\P{Jg=farsiyeh}', "");
Expect(1, 1910, '\P{^Jg=farsiyeh}', "");
Expect(0, 1911, '\p{Jg=farsiyeh}', "");
Expect(1, 1911, '\p{^Jg=farsiyeh}', "");
Expect(1, 1911, '\P{Jg=farsiyeh}', "");
Expect(0, 1911, '\P{^Jg=farsiyeh}', "");
Expect(1, 1910, '\p{Jg=--Farsi_yeh}', "");
Expect(0, 1910, '\p{^Jg=--Farsi_yeh}', "");
Expect(0, 1910, '\P{Jg=--Farsi_yeh}', "");
Expect(1, 1910, '\P{^Jg=--Farsi_yeh}', "");
Expect(0, 1911, '\p{Jg=--Farsi_yeh}', "");
Expect(1, 1911, '\p{^Jg=--Farsi_yeh}', "");
Expect(1, 1911, '\P{Jg=--Farsi_yeh}', "");
Expect(0, 1911, '\P{^Jg=--Farsi_yeh}', "");
Error('\p{Is_Joining_Group= -Farsi_Yeh/a/}');
Error('\P{Is_Joining_Group= -Farsi_Yeh/a/}');
Expect(1, 1910, '\p{Is_Joining_Group=farsiyeh}', "");
Expect(0, 1910, '\p{^Is_Joining_Group=farsiyeh}', "");
Expect(0, 1910, '\P{Is_Joining_Group=farsiyeh}', "");
Expect(1, 1910, '\P{^Is_Joining_Group=farsiyeh}', "");
Expect(0, 1911, '\p{Is_Joining_Group=farsiyeh}', "");
Expect(1, 1911, '\p{^Is_Joining_Group=farsiyeh}', "");
Expect(1, 1911, '\P{Is_Joining_Group=farsiyeh}', "");
Expect(0, 1911, '\P{^Is_Joining_Group=farsiyeh}', "");
Expect(1, 1910, '\p{Is_Joining_Group=		farsi_Yeh}', "");
Expect(0, 1910, '\p{^Is_Joining_Group=		farsi_Yeh}', "");
Expect(0, 1910, '\P{Is_Joining_Group=		farsi_Yeh}', "");
Expect(1, 1910, '\P{^Is_Joining_Group=		farsi_Yeh}', "");
Expect(0, 1911, '\p{Is_Joining_Group=		farsi_Yeh}', "");
Expect(1, 1911, '\p{^Is_Joining_Group=		farsi_Yeh}', "");
Expect(1, 1911, '\P{Is_Joining_Group=		farsi_Yeh}', "");
Expect(0, 1911, '\P{^Is_Joining_Group=		farsi_Yeh}', "");
Error('\p{Is_Jg=_:=FARSI_YEH}');
Error('\P{Is_Jg=_:=FARSI_YEH}');
Expect(1, 1910, '\p{Is_Jg=farsiyeh}', "");
Expect(0, 1910, '\p{^Is_Jg=farsiyeh}', "");
Expect(0, 1910, '\P{Is_Jg=farsiyeh}', "");
Expect(1, 1910, '\P{^Is_Jg=farsiyeh}', "");
Expect(0, 1911, '\p{Is_Jg=farsiyeh}', "");
Expect(1, 1911, '\p{^Is_Jg=farsiyeh}', "");
Expect(1, 1911, '\P{Is_Jg=farsiyeh}', "");
Expect(0, 1911, '\P{^Is_Jg=farsiyeh}', "");
Expect(1, 1910, '\p{Is_Jg=	farsi_Yeh}', "");
Expect(0, 1910, '\p{^Is_Jg=	farsi_Yeh}', "");
Expect(0, 1910, '\P{Is_Jg=	farsi_Yeh}', "");
Expect(1, 1910, '\P{^Is_Jg=	farsi_Yeh}', "");
Expect(0, 1911, '\p{Is_Jg=	farsi_Yeh}', "");
Expect(1, 1911, '\p{^Is_Jg=	farsi_Yeh}', "");
Expect(1, 1911, '\P{Is_Jg=	farsi_Yeh}', "");
Expect(0, 1911, '\P{^Is_Jg=	farsi_Yeh}', "");
Error('\p{Joining_Group=-/a/Reversed_Pe}');
Error('\P{Joining_Group=-/a/Reversed_Pe}');
Expect(1, 1831, '\p{Joining_Group=reversedpe}', "");
Expect(0, 1831, '\p{^Joining_Group=reversedpe}', "");
Expect(0, 1831, '\P{Joining_Group=reversedpe}', "");
Expect(1, 1831, '\P{^Joining_Group=reversedpe}', "");
Expect(0, 1832, '\p{Joining_Group=reversedpe}', "");
Expect(1, 1832, '\p{^Joining_Group=reversedpe}', "");
Expect(1, 1832, '\P{Joining_Group=reversedpe}', "");
Expect(0, 1832, '\P{^Joining_Group=reversedpe}', "");
Expect(1, 1831, '\p{Joining_Group:	--reversed_pe}', "");
Expect(0, 1831, '\p{^Joining_Group:	--reversed_pe}', "");
Expect(0, 1831, '\P{Joining_Group:	--reversed_pe}', "");
Expect(1, 1831, '\P{^Joining_Group:	--reversed_pe}', "");
Expect(0, 1832, '\p{Joining_Group:	--reversed_pe}', "");
Expect(1, 1832, '\p{^Joining_Group:	--reversed_pe}', "");
Expect(1, 1832, '\P{Joining_Group:	--reversed_pe}', "");
Expect(0, 1832, '\P{^Joining_Group:	--reversed_pe}', "");
Error('\p{Jg=		reversed_Pe/a/}');
Error('\P{Jg=		reversed_Pe/a/}');
Expect(1, 1831, '\p{Jg:   reversedpe}', "");
Expect(0, 1831, '\p{^Jg:   reversedpe}', "");
Expect(0, 1831, '\P{Jg:   reversedpe}', "");
Expect(1, 1831, '\P{^Jg:   reversedpe}', "");
Expect(0, 1832, '\p{Jg:   reversedpe}', "");
Expect(1, 1832, '\p{^Jg:   reversedpe}', "");
Expect(1, 1832, '\P{Jg:   reversedpe}', "");
Expect(0, 1832, '\P{^Jg:   reversedpe}', "");
Expect(1, 1831, '\p{Jg=-Reversed_Pe}', "");
Expect(0, 1831, '\p{^Jg=-Reversed_Pe}', "");
Expect(0, 1831, '\P{Jg=-Reversed_Pe}', "");
Expect(1, 1831, '\P{^Jg=-Reversed_Pe}', "");
Expect(0, 1832, '\p{Jg=-Reversed_Pe}', "");
Expect(1, 1832, '\p{^Jg=-Reversed_Pe}', "");
Expect(1, 1832, '\P{Jg=-Reversed_Pe}', "");
Expect(0, 1832, '\P{^Jg=-Reversed_Pe}', "");
Error('\p{Is_Joining_Group=	:=Reversed_PE}');
Error('\P{Is_Joining_Group=	:=Reversed_PE}');
Expect(1, 1831, '\p{Is_Joining_Group=reversedpe}', "");
Expect(0, 1831, '\p{^Is_Joining_Group=reversedpe}', "");
Expect(0, 1831, '\P{Is_Joining_Group=reversedpe}', "");
Expect(1, 1831, '\P{^Is_Joining_Group=reversedpe}', "");
Expect(0, 1832, '\p{Is_Joining_Group=reversedpe}', "");
Expect(1, 1832, '\p{^Is_Joining_Group=reversedpe}', "");
Expect(1, 1832, '\P{Is_Joining_Group=reversedpe}', "");
Expect(0, 1832, '\P{^Is_Joining_Group=reversedpe}', "");
Expect(1, 1831, '\p{Is_Joining_Group=-	REVERSED_Pe}', "");
Expect(0, 1831, '\p{^Is_Joining_Group=-	REVERSED_Pe}', "");
Expect(0, 1831, '\P{Is_Joining_Group=-	REVERSED_Pe}', "");
Expect(1, 1831, '\P{^Is_Joining_Group=-	REVERSED_Pe}', "");
Expect(0, 1832, '\p{Is_Joining_Group=-	REVERSED_Pe}', "");
Expect(1, 1832, '\p{^Is_Joining_Group=-	REVERSED_Pe}', "");
Expect(1, 1832, '\P{Is_Joining_Group=-	REVERSED_Pe}', "");
Expect(0, 1832, '\P{^Is_Joining_Group=-	REVERSED_Pe}', "");
Error('\p{Is_Jg=-	reversed_pe/a/}');
Error('\P{Is_Jg=-	reversed_pe/a/}');
Expect(1, 1831, '\p{Is_Jg=reversedpe}', "");
Expect(0, 1831, '\p{^Is_Jg=reversedpe}', "");
Expect(0, 1831, '\P{Is_Jg=reversedpe}', "");
Expect(1, 1831, '\P{^Is_Jg=reversedpe}', "");
Expect(0, 1832, '\p{Is_Jg=reversedpe}', "");
Expect(1, 1832, '\p{^Is_Jg=reversedpe}', "");
Expect(1, 1832, '\P{Is_Jg=reversedpe}', "");
Expect(0, 1832, '\P{^Is_Jg=reversedpe}', "");
Expect(1, 1831, '\p{Is_Jg=Reversed_PE}', "");
Expect(0, 1831, '\p{^Is_Jg=Reversed_PE}', "");
Expect(0, 1831, '\P{Is_Jg=Reversed_PE}', "");
Expect(1, 1831, '\P{^Is_Jg=Reversed_PE}', "");
Expect(0, 1832, '\p{Is_Jg=Reversed_PE}', "");
Expect(1, 1832, '\p{^Is_Jg=Reversed_PE}', "");
Expect(1, 1832, '\P{Is_Jg=Reversed_PE}', "");
Expect(0, 1832, '\P{^Is_Jg=Reversed_PE}', "");
Error('\p{Joining_Group=:=Shin}');
Error('\P{Joining_Group=:=Shin}');
Expect(1, 1835, '\p{Joining_Group=shin}', "");
Expect(0, 1835, '\p{^Joining_Group=shin}', "");
Expect(0, 1835, '\P{Joining_Group=shin}', "");
Expect(1, 1835, '\P{^Joining_Group=shin}', "");
Expect(0, 1836, '\p{Joining_Group=shin}', "");
Expect(1, 1836, '\p{^Joining_Group=shin}', "");
Expect(1, 1836, '\P{Joining_Group=shin}', "");
Expect(0, 1836, '\P{^Joining_Group=shin}', "");
Expect(1, 1835, '\p{Joining_Group=-	shin}', "");
Expect(0, 1835, '\p{^Joining_Group=-	shin}', "");
Expect(0, 1835, '\P{Joining_Group=-	shin}', "");
Expect(1, 1835, '\P{^Joining_Group=-	shin}', "");
Expect(0, 1836, '\p{Joining_Group=-	shin}', "");
Expect(1, 1836, '\p{^Joining_Group=-	shin}', "");
Expect(1, 1836, '\P{Joining_Group=-	shin}', "");
Expect(0, 1836, '\P{^Joining_Group=-	shin}', "");
Error('\p{Jg=/a/	Shin}');
Error('\P{Jg=/a/	Shin}');
Expect(1, 1835, '\p{Jg=shin}', "");
Expect(0, 1835, '\p{^Jg=shin}', "");
Expect(0, 1835, '\P{Jg=shin}', "");
Expect(1, 1835, '\P{^Jg=shin}', "");
Expect(0, 1836, '\p{Jg=shin}', "");
Expect(1, 1836, '\p{^Jg=shin}', "");
Expect(1, 1836, '\P{Jg=shin}', "");
Expect(0, 1836, '\P{^Jg=shin}', "");
Expect(1, 1835, '\p{Jg= Shin}', "");
Expect(0, 1835, '\p{^Jg= Shin}', "");
Expect(0, 1835, '\P{Jg= Shin}', "");
Expect(1, 1835, '\P{^Jg= Shin}', "");
Expect(0, 1836, '\p{Jg= Shin}', "");
Expect(1, 1836, '\p{^Jg= Shin}', "");
Expect(1, 1836, '\P{Jg= Shin}', "");
Expect(0, 1836, '\P{^Jg= Shin}', "");
Error('\p{Is_Joining_Group:   :=-SHIN}');
Error('\P{Is_Joining_Group:   :=-SHIN}');
Expect(1, 1835, '\p{Is_Joining_Group=shin}', "");
Expect(0, 1835, '\p{^Is_Joining_Group=shin}', "");
Expect(0, 1835, '\P{Is_Joining_Group=shin}', "");
Expect(1, 1835, '\P{^Is_Joining_Group=shin}', "");
Expect(0, 1836, '\p{Is_Joining_Group=shin}', "");
Expect(1, 1836, '\p{^Is_Joining_Group=shin}', "");
Expect(1, 1836, '\P{Is_Joining_Group=shin}', "");
Expect(0, 1836, '\P{^Is_Joining_Group=shin}', "");
Expect(1, 1835, '\p{Is_Joining_Group= _Shin}', "");
Expect(0, 1835, '\p{^Is_Joining_Group= _Shin}', "");
Expect(0, 1835, '\P{Is_Joining_Group= _Shin}', "");
Expect(1, 1835, '\P{^Is_Joining_Group= _Shin}', "");
Expect(0, 1836, '\p{Is_Joining_Group= _Shin}', "");
Expect(1, 1836, '\p{^Is_Joining_Group= _Shin}', "");
Expect(1, 1836, '\P{Is_Joining_Group= _Shin}', "");
Expect(0, 1836, '\P{^Is_Joining_Group= _Shin}', "");
Error('\p{Is_Jg:   /a/Shin}');
Error('\P{Is_Jg:   /a/Shin}');
Expect(1, 1835, '\p{Is_Jg=shin}', "");
Expect(0, 1835, '\p{^Is_Jg=shin}', "");
Expect(0, 1835, '\P{Is_Jg=shin}', "");
Expect(1, 1835, '\P{^Is_Jg=shin}', "");
Expect(0, 1836, '\p{Is_Jg=shin}', "");
Expect(1, 1836, '\p{^Is_Jg=shin}', "");
Expect(1, 1836, '\P{Is_Jg=shin}', "");
Expect(0, 1836, '\P{^Is_Jg=shin}', "");
Expect(1, 1835, '\p{Is_Jg= Shin}', "");
Expect(0, 1835, '\p{^Is_Jg= Shin}', "");
Expect(0, 1835, '\P{Is_Jg= Shin}', "");
Expect(1, 1835, '\P{^Is_Jg= Shin}', "");
Expect(0, 1836, '\p{Is_Jg= Shin}', "");
Expect(1, 1836, '\p{^Is_Jg= Shin}', "");
Expect(1, 1836, '\P{Is_Jg= Shin}', "");
Expect(0, 1836, '\P{^Is_Jg= Shin}', "");
Error('\p{Joining_Group:/a/YEH}');
Error('\P{Joining_Group:/a/YEH}');
Expect(1, 2217, '\p{Joining_Group=yeh}', "");
Expect(0, 2217, '\p{^Joining_Group=yeh}', "");
Expect(0, 2217, '\P{Joining_Group=yeh}', "");
Expect(1, 2217, '\P{^Joining_Group=yeh}', "");
Expect(0, 2218, '\p{Joining_Group=yeh}', "");
Expect(1, 2218, '\p{^Joining_Group=yeh}', "");
Expect(1, 2218, '\P{Joining_Group=yeh}', "");
Expect(0, 2218, '\P{^Joining_Group=yeh}', "");
Expect(1, 2217, '\p{Joining_Group= YEH}', "");
Expect(0, 2217, '\p{^Joining_Group= YEH}', "");
Expect(0, 2217, '\P{Joining_Group= YEH}', "");
Expect(1, 2217, '\P{^Joining_Group= YEH}', "");
Expect(0, 2218, '\p{Joining_Group= YEH}', "");
Expect(1, 2218, '\p{^Joining_Group= YEH}', "");
Expect(1, 2218, '\P{Joining_Group= YEH}', "");
Expect(0, 2218, '\P{^Joining_Group= YEH}', "");
Error('\p{Jg=		yeh/a/}');
Error('\P{Jg=		yeh/a/}');
Expect(1, 2217, '\p{Jg=yeh}', "");
Expect(0, 2217, '\p{^Jg=yeh}', "");
Expect(0, 2217, '\P{Jg=yeh}', "");
Expect(1, 2217, '\P{^Jg=yeh}', "");
Expect(0, 2218, '\p{Jg=yeh}', "");
Expect(1, 2218, '\p{^Jg=yeh}', "");
Expect(1, 2218, '\P{Jg=yeh}', "");
Expect(0, 2218, '\P{^Jg=yeh}', "");
Expect(1, 2217, '\p{Jg=  Yeh}', "");
Expect(0, 2217, '\p{^Jg=  Yeh}', "");
Expect(0, 2217, '\P{Jg=  Yeh}', "");
Expect(1, 2217, '\P{^Jg=  Yeh}', "");
Expect(0, 2218, '\p{Jg=  Yeh}', "");
Expect(1, 2218, '\p{^Jg=  Yeh}', "");
Expect(1, 2218, '\P{Jg=  Yeh}', "");
Expect(0, 2218, '\P{^Jg=  Yeh}', "");
Error('\p{Is_Joining_Group=:=	 Yeh}');
Error('\P{Is_Joining_Group=:=	 Yeh}');
Expect(1, 2217, '\p{Is_Joining_Group=yeh}', "");
Expect(0, 2217, '\p{^Is_Joining_Group=yeh}', "");
Expect(0, 2217, '\P{Is_Joining_Group=yeh}', "");
Expect(1, 2217, '\P{^Is_Joining_Group=yeh}', "");
Expect(0, 2218, '\p{Is_Joining_Group=yeh}', "");
Expect(1, 2218, '\p{^Is_Joining_Group=yeh}', "");
Expect(1, 2218, '\P{Is_Joining_Group=yeh}', "");
Expect(0, 2218, '\P{^Is_Joining_Group=yeh}', "");
Expect(1, 2217, '\p{Is_Joining_Group:    	Yeh}', "");
Expect(0, 2217, '\p{^Is_Joining_Group:    	Yeh}', "");
Expect(0, 2217, '\P{Is_Joining_Group:    	Yeh}', "");
Expect(1, 2217, '\P{^Is_Joining_Group:    	Yeh}', "");
Expect(0, 2218, '\p{Is_Joining_Group:    	Yeh}', "");
Expect(1, 2218, '\p{^Is_Joining_Group:    	Yeh}', "");
Expect(1, 2218, '\P{Is_Joining_Group:    	Yeh}', "");
Expect(0, 2218, '\P{^Is_Joining_Group:    	Yeh}', "");
Error('\p{Is_Jg=_/a/Yeh}');
Error('\P{Is_Jg=_/a/Yeh}');
Expect(1, 2217, '\p{Is_Jg=yeh}', "");
Expect(0, 2217, '\p{^Is_Jg=yeh}', "");
Expect(0, 2217, '\P{Is_Jg=yeh}', "");
Expect(1, 2217, '\P{^Is_Jg=yeh}', "");
Expect(0, 2218, '\p{Is_Jg=yeh}', "");
Expect(1, 2218, '\p{^Is_Jg=yeh}', "");
Expect(1, 2218, '\P{Is_Jg=yeh}', "");
Expect(0, 2218, '\P{^Is_Jg=yeh}', "");
Expect(1, 2217, '\p{Is_Jg=__yeh}', "");
Expect(0, 2217, '\p{^Is_Jg=__yeh}', "");
Expect(0, 2217, '\P{Is_Jg=__yeh}', "");
Expect(1, 2217, '\P{^Is_Jg=__yeh}', "");
Expect(0, 2218, '\p{Is_Jg=__yeh}', "");
Expect(1, 2218, '\p{^Is_Jg=__yeh}', "");
Expect(1, 2218, '\P{Is_Jg=__yeh}', "");
Expect(0, 2218, '\P{^Is_Jg=__yeh}', "");
Error('\p{Joining_Group=-_Dal:=}');
Error('\P{Joining_Group=-_Dal:=}');
Expect(1, 1882, '\p{Joining_Group=dal}', "");
Expect(0, 1882, '\p{^Joining_Group=dal}', "");
Expect(0, 1882, '\P{Joining_Group=dal}', "");
Expect(1, 1882, '\P{^Joining_Group=dal}', "");
Expect(0, 1883, '\p{Joining_Group=dal}', "");
Expect(1, 1883, '\p{^Joining_Group=dal}', "");
Expect(1, 1883, '\P{Joining_Group=dal}', "");
Expect(0, 1883, '\P{^Joining_Group=dal}', "");
Expect(1, 1882, '\p{Joining_Group:	-DAL}', "");
Expect(0, 1882, '\p{^Joining_Group:	-DAL}', "");
Expect(0, 1882, '\P{Joining_Group:	-DAL}', "");
Expect(1, 1882, '\P{^Joining_Group:	-DAL}', "");
Expect(0, 1883, '\p{Joining_Group:	-DAL}', "");
Expect(1, 1883, '\p{^Joining_Group:	-DAL}', "");
Expect(1, 1883, '\P{Joining_Group:	-DAL}', "");
Expect(0, 1883, '\P{^Joining_Group:	-DAL}', "");
Error('\p{Jg:   :=	-DAL}');
Error('\P{Jg:   :=	-DAL}');
Expect(1, 1882, '\p{Jg=dal}', "");
Expect(0, 1882, '\p{^Jg=dal}', "");
Expect(0, 1882, '\P{Jg=dal}', "");
Expect(1, 1882, '\P{^Jg=dal}', "");
Expect(0, 1883, '\p{Jg=dal}', "");
Expect(1, 1883, '\p{^Jg=dal}', "");
Expect(1, 1883, '\P{Jg=dal}', "");
Expect(0, 1883, '\P{^Jg=dal}', "");
Expect(1, 1882, '\p{Jg=	Dal}', "");
Expect(0, 1882, '\p{^Jg=	Dal}', "");
Expect(0, 1882, '\P{Jg=	Dal}', "");
Expect(1, 1882, '\P{^Jg=	Dal}', "");
Expect(0, 1883, '\p{Jg=	Dal}', "");
Expect(1, 1883, '\p{^Jg=	Dal}', "");
Expect(1, 1883, '\P{Jg=	Dal}', "");
Expect(0, 1883, '\P{^Jg=	Dal}', "");
Error('\p{Is_Joining_Group=_/a/Dal}');
Error('\P{Is_Joining_Group=_/a/Dal}');
Expect(1, 1882, '\p{Is_Joining_Group=dal}', "");
Expect(0, 1882, '\p{^Is_Joining_Group=dal}', "");
Expect(0, 1882, '\P{Is_Joining_Group=dal}', "");
Expect(1, 1882, '\P{^Is_Joining_Group=dal}', "");
Expect(0, 1883, '\p{Is_Joining_Group=dal}', "");
Expect(1, 1883, '\p{^Is_Joining_Group=dal}', "");
Expect(1, 1883, '\P{Is_Joining_Group=dal}', "");
Expect(0, 1883, '\P{^Is_Joining_Group=dal}', "");
Error('\p{Is_Jg= :=DAL}');
Error('\P{Is_Jg= :=DAL}');
Expect(1, 1882, '\p{Is_Jg=dal}', "");
Expect(0, 1882, '\p{^Is_Jg=dal}', "");
Expect(0, 1882, '\P{Is_Jg=dal}', "");
Expect(1, 1882, '\P{^Is_Jg=dal}', "");
Expect(0, 1883, '\p{Is_Jg=dal}', "");
Expect(1, 1883, '\p{^Is_Jg=dal}', "");
Expect(1, 1883, '\P{Is_Jg=dal}', "");
Expect(0, 1883, '\P{^Is_Jg=dal}', "");
Expect(1, 1882, '\p{Is_Jg=	-Dal}', "");
Expect(0, 1882, '\p{^Is_Jg=	-Dal}', "");
Expect(0, 1882, '\P{Is_Jg=	-Dal}', "");
Expect(1, 1882, '\P{^Is_Jg=	-Dal}', "");
Expect(0, 1883, '\p{Is_Jg=	-Dal}', "");
Expect(1, 1883, '\p{^Is_Jg=	-Dal}', "");
Expect(1, 1883, '\P{Is_Jg=	-Dal}', "");
Expect(0, 1883, '\P{^Is_Jg=	-Dal}', "");
Error('\p{Joining_Group=He:=}');
Error('\P{Joining_Group=He:=}');
Expect(1, 1815, '\p{Joining_Group:he}', "");
Expect(0, 1815, '\p{^Joining_Group:he}', "");
Expect(0, 1815, '\P{Joining_Group:he}', "");
Expect(1, 1815, '\P{^Joining_Group:he}', "");
Expect(0, 1816, '\p{Joining_Group:he}', "");
Expect(1, 1816, '\p{^Joining_Group:he}', "");
Expect(1, 1816, '\P{Joining_Group:he}', "");
Expect(0, 1816, '\P{^Joining_Group:he}', "");
Expect(1, 1815, '\p{Joining_Group= 	He}', "");
Expect(0, 1815, '\p{^Joining_Group= 	He}', "");
Expect(0, 1815, '\P{Joining_Group= 	He}', "");
Expect(1, 1815, '\P{^Joining_Group= 	He}', "");
Expect(0, 1816, '\p{Joining_Group= 	He}', "");
Expect(1, 1816, '\p{^Joining_Group= 	He}', "");
Expect(1, 1816, '\P{Joining_Group= 	He}', "");
Expect(0, 1816, '\P{^Joining_Group= 	He}', "");
Error('\p{Jg= :=he}');
Error('\P{Jg= :=he}');
Expect(1, 1815, '\p{Jg=he}', "");
Expect(0, 1815, '\p{^Jg=he}', "");
Expect(0, 1815, '\P{Jg=he}', "");
Expect(1, 1815, '\P{^Jg=he}', "");
Expect(0, 1816, '\p{Jg=he}', "");
Expect(1, 1816, '\p{^Jg=he}', "");
Expect(1, 1816, '\P{Jg=he}', "");
Expect(0, 1816, '\P{^Jg=he}', "");
Expect(1, 1815, '\p{Jg=	He}', "");
Expect(0, 1815, '\p{^Jg=	He}', "");
Expect(0, 1815, '\P{Jg=	He}', "");
Expect(1, 1815, '\P{^Jg=	He}', "");
Expect(0, 1816, '\p{Jg=	He}', "");
Expect(1, 1816, '\p{^Jg=	He}', "");
Expect(1, 1816, '\P{Jg=	He}', "");
Expect(0, 1816, '\P{^Jg=	He}', "");
Error('\p{Is_Joining_Group=	/a/he}');
Error('\P{Is_Joining_Group=	/a/he}');
Expect(1, 1815, '\p{Is_Joining_Group=he}', "");
Expect(0, 1815, '\p{^Is_Joining_Group=he}', "");
Expect(0, 1815, '\P{Is_Joining_Group=he}', "");
Expect(1, 1815, '\P{^Is_Joining_Group=he}', "");
Expect(0, 1816, '\p{Is_Joining_Group=he}', "");
Expect(1, 1816, '\p{^Is_Joining_Group=he}', "");
Expect(1, 1816, '\P{Is_Joining_Group=he}', "");
Expect(0, 1816, '\P{^Is_Joining_Group=he}', "");
Expect(1, 1815, '\p{Is_Joining_Group:		_he}', "");
Expect(0, 1815, '\p{^Is_Joining_Group:		_he}', "");
Expect(0, 1815, '\P{Is_Joining_Group:		_he}', "");
Expect(1, 1815, '\P{^Is_Joining_Group:		_he}', "");
Expect(0, 1816, '\p{Is_Joining_Group:		_he}', "");
Expect(1, 1816, '\p{^Is_Joining_Group:		_he}', "");
Expect(1, 1816, '\P{Is_Joining_Group:		_he}', "");
Expect(0, 1816, '\P{^Is_Joining_Group:		_he}', "");
Error('\p{Is_Jg=-	HE:=}');
Error('\P{Is_Jg=-	HE:=}');
Expect(1, 1815, '\p{Is_Jg=he}', "");
Expect(0, 1815, '\p{^Is_Jg=he}', "");
Expect(0, 1815, '\P{Is_Jg=he}', "");
Expect(1, 1815, '\P{^Is_Jg=he}', "");
Expect(0, 1816, '\p{Is_Jg=he}', "");
Expect(1, 1816, '\p{^Is_Jg=he}', "");
Expect(1, 1816, '\P{Is_Jg=he}', "");
Expect(0, 1816, '\P{^Is_Jg=he}', "");
Expect(1, 1815, '\p{Is_Jg=_He}', "");
Expect(0, 1815, '\p{^Is_Jg=_He}', "");
Expect(0, 1815, '\P{Is_Jg=_He}', "");
Expect(1, 1815, '\P{^Is_Jg=_He}', "");
Expect(0, 1816, '\p{Is_Jg=_He}', "");
Expect(1, 1816, '\p{^Is_Jg=_He}', "");
Expect(1, 1816, '\P{Is_Jg=_He}', "");
Expect(0, 1816, '\P{^Is_Jg=_He}', "");
Error('\p{Joining_Group=-/a/no_Joining_Group}');
Error('\P{Joining_Group=-/a/no_Joining_Group}');
Expect(1, 2221, '\p{Joining_Group=nojoininggroup}', "");
Expect(0, 2221, '\p{^Joining_Group=nojoininggroup}', "");
Expect(0, 2221, '\P{Joining_Group=nojoininggroup}', "");
Expect(1, 2221, '\P{^Joining_Group=nojoininggroup}', "");
Expect(0, 2220, '\p{Joining_Group=nojoininggroup}', "");
Expect(1, 2220, '\p{^Joining_Group=nojoininggroup}', "");
Expect(1, 2220, '\P{Joining_Group=nojoininggroup}', "");
Expect(0, 2220, '\P{^Joining_Group=nojoininggroup}', "");
Expect(1, 2221, '\p{Joining_Group: -_No_joining_Group}', "");
Expect(0, 2221, '\p{^Joining_Group: -_No_joining_Group}', "");
Expect(0, 2221, '\P{Joining_Group: -_No_joining_Group}', "");
Expect(1, 2221, '\P{^Joining_Group: -_No_joining_Group}', "");
Expect(0, 2220, '\p{Joining_Group: -_No_joining_Group}', "");
Expect(1, 2220, '\p{^Joining_Group: -_No_joining_Group}', "");
Expect(1, 2220, '\P{Joining_Group: -_No_joining_Group}', "");
Expect(0, 2220, '\P{^Joining_Group: -_No_joining_Group}', "");
Error('\p{Jg= No_Joining_group:=}');
Error('\P{Jg= No_Joining_group:=}');
Expect(1, 2221, '\p{Jg=nojoininggroup}', "");
Expect(0, 2221, '\p{^Jg=nojoininggroup}', "");
Expect(0, 2221, '\P{Jg=nojoininggroup}', "");
Expect(1, 2221, '\P{^Jg=nojoininggroup}', "");
Expect(0, 2220, '\p{Jg=nojoininggroup}', "");
Expect(1, 2220, '\p{^Jg=nojoininggroup}', "");
Expect(1, 2220, '\P{Jg=nojoininggroup}', "");
Expect(0, 2220, '\P{^Jg=nojoininggroup}', "");
Expect(1, 2221, '\p{Jg:  _No_JOINING_Group}', "");
Expect(0, 2221, '\p{^Jg:  _No_JOINING_Group}', "");
Expect(0, 2221, '\P{Jg:  _No_JOINING_Group}', "");
Expect(1, 2221, '\P{^Jg:  _No_JOINING_Group}', "");
Expect(0, 2220, '\p{Jg:  _No_JOINING_Group}', "");
Expect(1, 2220, '\p{^Jg:  _No_JOINING_Group}', "");
Expect(1, 2220, '\P{Jg:  _No_JOINING_Group}', "");
Expect(0, 2220, '\P{^Jg:  _No_JOINING_Group}', "");
Error('\p{Is_Joining_Group=	:=NO_joining_GROUP}');
Error('\P{Is_Joining_Group=	:=NO_joining_GROUP}');
Expect(1, 2221, '\p{Is_Joining_Group=nojoininggroup}', "");
Expect(0, 2221, '\p{^Is_Joining_Group=nojoininggroup}', "");
Expect(0, 2221, '\P{Is_Joining_Group=nojoininggroup}', "");
Expect(1, 2221, '\P{^Is_Joining_Group=nojoininggroup}', "");
Expect(0, 2220, '\p{Is_Joining_Group=nojoininggroup}', "");
Expect(1, 2220, '\p{^Is_Joining_Group=nojoininggroup}', "");
Expect(1, 2220, '\P{Is_Joining_Group=nojoininggroup}', "");
Expect(0, 2220, '\P{^Is_Joining_Group=nojoininggroup}', "");
Expect(1, 2221, '\p{Is_Joining_Group=_NO_Joining_Group}', "");
Expect(0, 2221, '\p{^Is_Joining_Group=_NO_Joining_Group}', "");
Expect(0, 2221, '\P{Is_Joining_Group=_NO_Joining_Group}', "");
Expect(1, 2221, '\P{^Is_Joining_Group=_NO_Joining_Group}', "");
Expect(0, 2220, '\p{Is_Joining_Group=_NO_Joining_Group}', "");
Expect(1, 2220, '\p{^Is_Joining_Group=_NO_Joining_Group}', "");
Expect(1, 2220, '\P{Is_Joining_Group=_NO_Joining_Group}', "");
Expect(0, 2220, '\P{^Is_Joining_Group=_NO_Joining_Group}', "");
Error('\p{Is_Jg= /a/No_JOINING_Group}');
Error('\P{Is_Jg= /a/No_JOINING_Group}');
Expect(1, 2221, '\p{Is_Jg=nojoininggroup}', "");
Expect(0, 2221, '\p{^Is_Jg=nojoininggroup}', "");
Expect(0, 2221, '\P{Is_Jg=nojoininggroup}', "");
Expect(1, 2221, '\P{^Is_Jg=nojoininggroup}', "");
Expect(0, 2220, '\p{Is_Jg=nojoininggroup}', "");
Expect(1, 2220, '\p{^Is_Jg=nojoininggroup}', "");
Expect(1, 2220, '\P{Is_Jg=nojoininggroup}', "");
Expect(0, 2220, '\P{^Is_Jg=nojoininggroup}', "");
Expect(1, 2221, '\p{Is_Jg=	NO_joining_Group}', "");
Expect(0, 2221, '\p{^Is_Jg=	NO_joining_Group}', "");
Expect(0, 2221, '\P{Is_Jg=	NO_joining_Group}', "");
Expect(1, 2221, '\P{^Is_Jg=	NO_joining_Group}', "");
Expect(0, 2220, '\p{Is_Jg=	NO_joining_Group}', "");
Expect(1, 2220, '\p{^Is_Jg=	NO_joining_Group}', "");
Expect(1, 2220, '\P{Is_Jg=	NO_joining_Group}', "");
Expect(0, 2220, '\P{^Is_Jg=	NO_joining_Group}', "");
Error('\p{Joining_Group=/a/ Burushaski_Yeh_Barree}');
Error('\P{Joining_Group=/a/ Burushaski_Yeh_Barree}');
Expect(1, 1915, '\p{Joining_Group=burushaskiyehbarree}', "");
Expect(0, 1915, '\p{^Joining_Group=burushaskiyehbarree}', "");
Expect(0, 1915, '\P{Joining_Group=burushaskiyehbarree}', "");
Expect(1, 1915, '\P{^Joining_Group=burushaskiyehbarree}', "");
Expect(0, 1916, '\p{Joining_Group=burushaskiyehbarree}', "");
Expect(1, 1916, '\p{^Joining_Group=burushaskiyehbarree}', "");
Expect(1, 1916, '\P{Joining_Group=burushaskiyehbarree}', "");
Expect(0, 1916, '\P{^Joining_Group=burushaskiyehbarree}', "");
Expect(1, 1915, '\p{Joining_Group=		burushaski_Yeh_BARREE}', "");
Expect(0, 1915, '\p{^Joining_Group=		burushaski_Yeh_BARREE}', "");
Expect(0, 1915, '\P{Joining_Group=		burushaski_Yeh_BARREE}', "");
Expect(1, 1915, '\P{^Joining_Group=		burushaski_Yeh_BARREE}', "");
Expect(0, 1916, '\p{Joining_Group=		burushaski_Yeh_BARREE}', "");
Expect(1, 1916, '\p{^Joining_Group=		burushaski_Yeh_BARREE}', "");
Expect(1, 1916, '\P{Joining_Group=		burushaski_Yeh_BARREE}', "");
Expect(0, 1916, '\P{^Joining_Group=		burushaski_Yeh_BARREE}', "");
Error('\p{Jg=-_Burushaski_yeh_Barree:=}');
Error('\P{Jg=-_Burushaski_yeh_Barree:=}');
Expect(1, 1915, '\p{Jg=burushaskiyehbarree}', "");
Expect(0, 1915, '\p{^Jg=burushaskiyehbarree}', "");
Expect(0, 1915, '\P{Jg=burushaskiyehbarree}', "");
Expect(1, 1915, '\P{^Jg=burushaskiyehbarree}', "");
Expect(0, 1916, '\p{Jg=burushaskiyehbarree}', "");
Expect(1, 1916, '\p{^Jg=burushaskiyehbarree}', "");
Expect(1, 1916, '\P{Jg=burushaskiyehbarree}', "");
Expect(0, 1916, '\P{^Jg=burushaskiyehbarree}', "");
Expect(1, 1915, '\p{Jg= _burushaski_Yeh_Barree}', "");
Expect(0, 1915, '\p{^Jg= _burushaski_Yeh_Barree}', "");
Expect(0, 1915, '\P{Jg= _burushaski_Yeh_Barree}', "");
Expect(1, 1915, '\P{^Jg= _burushaski_Yeh_Barree}', "");
Expect(0, 1916, '\p{Jg= _burushaski_Yeh_Barree}', "");
Expect(1, 1916, '\p{^Jg= _burushaski_Yeh_Barree}', "");
Expect(1, 1916, '\P{Jg= _burushaski_Yeh_Barree}', "");
Expect(0, 1916, '\P{^Jg= _burushaski_Yeh_Barree}', "");
Error('\p{Is_Joining_Group=/a/	burushaski_yeh_BARREE}');
Error('\P{Is_Joining_Group=/a/	burushaski_yeh_BARREE}');
Expect(1, 1915, '\p{Is_Joining_Group=burushaskiyehbarree}', "");
Expect(0, 1915, '\p{^Is_Joining_Group=burushaskiyehbarree}', "");
Expect(0, 1915, '\P{Is_Joining_Group=burushaskiyehbarree}', "");
Expect(1, 1915, '\P{^Is_Joining_Group=burushaskiyehbarree}', "");
Expect(0, 1916, '\p{Is_Joining_Group=burushaskiyehbarree}', "");
Expect(1, 1916, '\p{^Is_Joining_Group=burushaskiyehbarree}', "");
Expect(1, 1916, '\P{Is_Joining_Group=burushaskiyehbarree}', "");
Expect(0, 1916, '\P{^Is_Joining_Group=burushaskiyehbarree}', "");
Expect(1, 1915, '\p{Is_Joining_Group=-_Burushaski_YEH_BARREE}', "");
Expect(0, 1915, '\p{^Is_Joining_Group=-_Burushaski_YEH_BARREE}', "");
Expect(0, 1915, '\P{Is_Joining_Group=-_Burushaski_YEH_BARREE}', "");
Expect(1, 1915, '\P{^Is_Joining_Group=-_Burushaski_YEH_BARREE}', "");
Expect(0, 1916, '\p{Is_Joining_Group=-_Burushaski_YEH_BARREE}', "");
Expect(1, 1916, '\p{^Is_Joining_Group=-_Burushaski_YEH_BARREE}', "");
Expect(1, 1916, '\P{Is_Joining_Group=-_Burushaski_YEH_BARREE}', "");
Expect(0, 1916, '\P{^Is_Joining_Group=-_Burushaski_YEH_BARREE}', "");
Error('\p{Is_Jg: /a/-BURUSHASKI_Yeh_BARREE}');
Error('\P{Is_Jg: /a/-BURUSHASKI_Yeh_BARREE}');
Expect(1, 1915, '\p{Is_Jg:burushaskiyehbarree}', "");
Expect(0, 1915, '\p{^Is_Jg:burushaskiyehbarree}', "");
Expect(0, 1915, '\P{Is_Jg:burushaskiyehbarree}', "");
Expect(1, 1915, '\P{^Is_Jg:burushaskiyehbarree}', "");
Expect(0, 1916, '\p{Is_Jg:burushaskiyehbarree}', "");
Expect(1, 1916, '\p{^Is_Jg:burushaskiyehbarree}', "");
Expect(1, 1916, '\P{Is_Jg:burushaskiyehbarree}', "");
Expect(0, 1916, '\P{^Is_Jg:burushaskiyehbarree}', "");
Expect(1, 1915, '\p{Is_Jg=- BURUSHASKI_Yeh_Barree}', "");
Expect(0, 1915, '\p{^Is_Jg=- BURUSHASKI_Yeh_Barree}', "");
Expect(0, 1915, '\P{Is_Jg=- BURUSHASKI_Yeh_Barree}', "");
Expect(1, 1915, '\P{^Is_Jg=- BURUSHASKI_Yeh_Barree}', "");
Expect(0, 1916, '\p{Is_Jg=- BURUSHASKI_Yeh_Barree}', "");
Expect(1, 1916, '\p{^Is_Jg=- BURUSHASKI_Yeh_Barree}', "");
Expect(1, 1916, '\P{Is_Jg=- BURUSHASKI_Yeh_Barree}', "");
Expect(0, 1916, '\P{^Is_Jg=- BURUSHASKI_Yeh_Barree}', "");
Error('\p{Joining_Group=/a/ 	TEH_Marbuta}');
Error('\P{Joining_Group=/a/ 	TEH_Marbuta}');
Expect(1, 1749, '\p{Joining_Group:   tehmarbuta}', "");
Expect(0, 1749, '\p{^Joining_Group:   tehmarbuta}', "");
Expect(0, 1749, '\P{Joining_Group:   tehmarbuta}', "");
Expect(1, 1749, '\P{^Joining_Group:   tehmarbuta}', "");
Expect(0, 1750, '\p{Joining_Group:   tehmarbuta}', "");
Expect(1, 1750, '\p{^Joining_Group:   tehmarbuta}', "");
Expect(1, 1750, '\P{Joining_Group:   tehmarbuta}', "");
Expect(0, 1750, '\P{^Joining_Group:   tehmarbuta}', "");
Expect(1, 1749, '\p{Joining_Group=	 teh_Marbuta}', "");
Expect(0, 1749, '\p{^Joining_Group=	 teh_Marbuta}', "");
Expect(0, 1749, '\P{Joining_Group=	 teh_Marbuta}', "");
Expect(1, 1749, '\P{^Joining_Group=	 teh_Marbuta}', "");
Expect(0, 1750, '\p{Joining_Group=	 teh_Marbuta}', "");
Expect(1, 1750, '\p{^Joining_Group=	 teh_Marbuta}', "");
Expect(1, 1750, '\P{Joining_Group=	 teh_Marbuta}', "");
Expect(0, 1750, '\P{^Joining_Group=	 teh_Marbuta}', "");
Error('\p{Jg=:=-Teh_marbuta}');
Error('\P{Jg=:=-Teh_marbuta}');
Expect(1, 1749, '\p{Jg=tehmarbuta}', "");
Expect(0, 1749, '\p{^Jg=tehmarbuta}', "");
Expect(0, 1749, '\P{Jg=tehmarbuta}', "");
Expect(1, 1749, '\P{^Jg=tehmarbuta}', "");
Expect(0, 1750, '\p{Jg=tehmarbuta}', "");
Expect(1, 1750, '\p{^Jg=tehmarbuta}', "");
Expect(1, 1750, '\P{Jg=tehmarbuta}', "");
Expect(0, 1750, '\P{^Jg=tehmarbuta}', "");
Expect(1, 1749, '\p{Jg= 	Teh_Marbuta}', "");
Expect(0, 1749, '\p{^Jg= 	Teh_Marbuta}', "");
Expect(0, 1749, '\P{Jg= 	Teh_Marbuta}', "");
Expect(1, 1749, '\P{^Jg= 	Teh_Marbuta}', "");
Expect(0, 1750, '\p{Jg= 	Teh_Marbuta}', "");
Expect(1, 1750, '\p{^Jg= 	Teh_Marbuta}', "");
Expect(1, 1750, '\P{Jg= 	Teh_Marbuta}', "");
Expect(0, 1750, '\P{^Jg= 	Teh_Marbuta}', "");
Error('\p{Is_Joining_Group:  -Teh_marbuta/a/}');
Error('\P{Is_Joining_Group:  -Teh_marbuta/a/}');
Expect(1, 1749, '\p{Is_Joining_Group=tehmarbuta}', "");
Expect(0, 1749, '\p{^Is_Joining_Group=tehmarbuta}', "");
Expect(0, 1749, '\P{Is_Joining_Group=tehmarbuta}', "");
Expect(1, 1749, '\P{^Is_Joining_Group=tehmarbuta}', "");
Expect(0, 1750, '\p{Is_Joining_Group=tehmarbuta}', "");
Expect(1, 1750, '\p{^Is_Joining_Group=tehmarbuta}', "");
Expect(1, 1750, '\P{Is_Joining_Group=tehmarbuta}', "");
Expect(0, 1750, '\P{^Is_Joining_Group=tehmarbuta}', "");
Expect(1, 1749, '\p{Is_Joining_Group=_-teh_Marbuta}', "");
Expect(0, 1749, '\p{^Is_Joining_Group=_-teh_Marbuta}', "");
Expect(0, 1749, '\P{Is_Joining_Group=_-teh_Marbuta}', "");
Expect(1, 1749, '\P{^Is_Joining_Group=_-teh_Marbuta}', "");
Expect(0, 1750, '\p{Is_Joining_Group=_-teh_Marbuta}', "");
Expect(1, 1750, '\p{^Is_Joining_Group=_-teh_Marbuta}', "");
Expect(1, 1750, '\P{Is_Joining_Group=_-teh_Marbuta}', "");
Expect(0, 1750, '\P{^Is_Joining_Group=_-teh_Marbuta}', "");
Error('\p{Is_Jg=:=Teh_Marbuta}');
Error('\P{Is_Jg=:=Teh_Marbuta}');
Expect(1, 1749, '\p{Is_Jg:	tehmarbuta}', "");
Expect(0, 1749, '\p{^Is_Jg:	tehmarbuta}', "");
Expect(0, 1749, '\P{Is_Jg:	tehmarbuta}', "");
Expect(1, 1749, '\P{^Is_Jg:	tehmarbuta}', "");
Expect(0, 1750, '\p{Is_Jg:	tehmarbuta}', "");
Expect(1, 1750, '\p{^Is_Jg:	tehmarbuta}', "");
Expect(1, 1750, '\P{Is_Jg:	tehmarbuta}', "");
Expect(0, 1750, '\P{^Is_Jg:	tehmarbuta}', "");
Expect(1, 1749, '\p{Is_Jg=		TEH_Marbuta}', "");
Expect(0, 1749, '\p{^Is_Jg=		TEH_Marbuta}', "");
Expect(0, 1749, '\P{Is_Jg=		TEH_Marbuta}', "");
Expect(1, 1749, '\P{^Is_Jg=		TEH_Marbuta}', "");
Expect(0, 1750, '\p{Is_Jg=		TEH_Marbuta}', "");
Expect(1, 1750, '\p{^Is_Jg=		TEH_Marbuta}', "");
Expect(1, 1750, '\P{Is_Jg=		TEH_Marbuta}', "");
Expect(0, 1750, '\P{^Is_Jg=		TEH_Marbuta}', "");
Error('\p{Joining_Group=	:=reh}');
Error('\P{Joining_Group=	:=reh}');
Expect(1, 2218, '\p{Joining_Group=reh}', "");
Expect(0, 2218, '\p{^Joining_Group=reh}', "");
Expect(0, 2218, '\P{Joining_Group=reh}', "");
Expect(1, 2218, '\P{^Joining_Group=reh}', "");
Expect(0, 2219, '\p{Joining_Group=reh}', "");
Expect(1, 2219, '\p{^Joining_Group=reh}', "");
Expect(1, 2219, '\P{Joining_Group=reh}', "");
Expect(0, 2219, '\P{^Joining_Group=reh}', "");
Expect(1, 2218, '\p{Joining_Group=  Reh}', "");
Expect(0, 2218, '\p{^Joining_Group=  Reh}', "");
Expect(0, 2218, '\P{Joining_Group=  Reh}', "");
Expect(1, 2218, '\P{^Joining_Group=  Reh}', "");
Expect(0, 2219, '\p{Joining_Group=  Reh}', "");
Expect(1, 2219, '\p{^Joining_Group=  Reh}', "");
Expect(1, 2219, '\P{Joining_Group=  Reh}', "");
Expect(0, 2219, '\P{^Joining_Group=  Reh}', "");
Error('\p{Jg= Reh:=}');
Error('\P{Jg= Reh:=}');
Expect(1, 2218, '\p{Jg=reh}', "");
Expect(0, 2218, '\p{^Jg=reh}', "");
Expect(0, 2218, '\P{Jg=reh}', "");
Expect(1, 2218, '\P{^Jg=reh}', "");
Expect(0, 2219, '\p{Jg=reh}', "");
Expect(1, 2219, '\p{^Jg=reh}', "");
Expect(1, 2219, '\P{Jg=reh}', "");
Expect(0, 2219, '\P{^Jg=reh}', "");
Expect(1, 2218, '\p{Jg= REH}', "");
Expect(0, 2218, '\p{^Jg= REH}', "");
Expect(0, 2218, '\P{Jg= REH}', "");
Expect(1, 2218, '\P{^Jg= REH}', "");
Expect(0, 2219, '\p{Jg= REH}', "");
Expect(1, 2219, '\p{^Jg= REH}', "");
Expect(1, 2219, '\P{Jg= REH}', "");
Expect(0, 2219, '\P{^Jg= REH}', "");
Error('\p{Is_Joining_Group=_ Reh/a/}');
Error('\P{Is_Joining_Group=_ Reh/a/}');
Expect(1, 2218, '\p{Is_Joining_Group=reh}', "");
Expect(0, 2218, '\p{^Is_Joining_Group=reh}', "");
Expect(0, 2218, '\P{Is_Joining_Group=reh}', "");
Expect(1, 2218, '\P{^Is_Joining_Group=reh}', "");
Expect(0, 2219, '\p{Is_Joining_Group=reh}', "");
Expect(1, 2219, '\p{^Is_Joining_Group=reh}', "");
Expect(1, 2219, '\P{Is_Joining_Group=reh}', "");
Expect(0, 2219, '\P{^Is_Joining_Group=reh}', "");
Expect(1, 2218, '\p{Is_Joining_Group:   -reh}', "");
Expect(0, 2218, '\p{^Is_Joining_Group:   -reh}', "");
Expect(0, 2218, '\P{Is_Joining_Group:   -reh}', "");
Expect(1, 2218, '\P{^Is_Joining_Group:   -reh}', "");
Expect(0, 2219, '\p{Is_Joining_Group:   -reh}', "");
Expect(1, 2219, '\p{^Is_Joining_Group:   -reh}', "");
Expect(1, 2219, '\P{Is_Joining_Group:   -reh}', "");
Expect(0, 2219, '\P{^Is_Joining_Group:   -reh}', "");
Error('\p{Is_Jg= /a/reh}');
Error('\P{Is_Jg= /a/reh}');
Expect(1, 2218, '\p{Is_Jg=reh}', "");
Expect(0, 2218, '\p{^Is_Jg=reh}', "");
Expect(0, 2218, '\P{Is_Jg=reh}', "");
Expect(1, 2218, '\P{^Is_Jg=reh}', "");
Expect(0, 2219, '\p{Is_Jg=reh}', "");
Expect(1, 2219, '\p{^Is_Jg=reh}', "");
Expect(1, 2219, '\P{Is_Jg=reh}', "");
Expect(0, 2219, '\P{^Is_Jg=reh}', "");
Expect(1, 2218, '\p{Is_Jg=	_Reh}', "");
Expect(0, 2218, '\p{^Is_Jg=	_Reh}', "");
Expect(0, 2218, '\P{Is_Jg=	_Reh}', "");
Expect(1, 2218, '\P{^Is_Jg=	_Reh}', "");
Expect(0, 2219, '\p{Is_Jg=	_Reh}', "");
Expect(1, 2219, '\p{^Is_Jg=	_Reh}', "");
Expect(1, 2219, '\P{Is_Jg=	_Reh}', "");
Expect(0, 2219, '\P{^Is_Jg=	_Reh}', "");
Error('\p{Joining_Group=:=__Zain}');
Error('\P{Joining_Group=:=__Zain}');
Expect(1, 1817, '\p{Joining_Group=zain}', "");
Expect(0, 1817, '\p{^Joining_Group=zain}', "");
Expect(0, 1817, '\P{Joining_Group=zain}', "");
Expect(1, 1817, '\P{^Joining_Group=zain}', "");
Expect(0, 1818, '\p{Joining_Group=zain}', "");
Expect(1, 1818, '\p{^Joining_Group=zain}', "");
Expect(1, 1818, '\P{Joining_Group=zain}', "");
Expect(0, 1818, '\P{^Joining_Group=zain}', "");
Expect(1, 1817, '\p{Joining_Group=	Zain}', "");
Expect(0, 1817, '\p{^Joining_Group=	Zain}', "");
Expect(0, 1817, '\P{Joining_Group=	Zain}', "")