/*      $OpenBSD: aes_xts.c,v 1.1 2008/06/09 16:15:05 djm Exp $  */

/*
 * Copyright (c) 2002 Markus Friedl.  All rights reserved.
 * Copyright (c) 2008 Damien Miller.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/sysctl.h>
#include <crypto/cryptodev.h>
#include <err.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

static int
syscrypt(const unsigned char *key, size_t klen, u_int64_t unit_number,
    const unsigned char *in, unsigned char *out, size_t len, int do_encrypt)
{
	struct session_op session;
	struct crypt_op cryp;
	int cryptodev_fd = -1, fd = -1;

	if ((cryptodev_fd = open("/dev/crypto", O_RDWR, 0)) < 0) {
		warn("/dev/crypto");
		goto err;
	}
	if (ioctl(cryptodev_fd, CRIOGET, &fd) == -1) {
		warn("CRIOGET failed");
		goto err;
	}
	memset(&session, 0, sizeof(session));
	session.cipher = CRYPTO_AES_XTS;
	session.key = (caddr_t) key;
	session.keylen = klen;
	if (ioctl(fd, CIOCGSESSION, &session) == -1) {
		warn("CIOCGSESSION");
		goto err;
	}
	memset(&cryp, 0, sizeof(cryp));
	cryp.ses = session.ses;
	cryp.op = do_encrypt ? COP_ENCRYPT : COP_DECRYPT;
	cryp.flags = 0;
	cryp.len = len;
	cryp.src = (caddr_t) in;
	cryp.dst = (caddr_t) out;
	cryp.iv = (caddr_t) &unit_number;
	cryp.mac = 0;
	if (ioctl(fd, CIOCCRYPT, &cryp) == -1) {
		warn("CIOCCRYPT");
		goto err;
	}
	if (ioctl(fd, CIOCFSESSION, &session.ses) == -1) {
		warn("CIOCFSESSION");
		goto err;
	}
	close(fd);
	close(cryptodev_fd);
	return (0);

err:
	if (fd != -1)
		close(fd);
	if (cryptodev_fd != -1)
		close(cryptodev_fd);
	return (-1);
}

static int
getallowsoft(void)
{
	int mib[2], old;
	size_t olen;

	olen = sizeof(old);

	mib[0] = CTL_KERN;
	mib[1] = KERN_CRYPTODEVALLOWSOFT;
	if (sysctl(mib, 2, &old, &olen, NULL, 0) < 0)
		err(1, "sysctl failed");

	return old;
}

static void
setallowsoft(int new)
{
	int mib[2], old;
	size_t olen, nlen;

	olen = nlen = sizeof(new);

	mib[0] = CTL_KERN;
	mib[1] = KERN_CRYPTODEVALLOWSOFT;

	if (sysctl(mib, 2, &old, &olen, &new, nlen) < 0)
		err(1, "sysctl failed");
}

static int
match(unsigned char *a, unsigned char *b, size_t len)
{
	size_t i;

	if (memcmp(a, b, len) == 0)
		return (1);

	warnx("decrypt/plaintext mismatch");

	for (i = 0; i < len; i++)
		printf("%2.2x", a[i]);
	printf("\n");
	for (i = 0; i < len; i++)
		printf("%2.2x", b[i]);
	printf("\n");

	return (0);
}

struct aes_xts_tv {
	u_int64_t seqno;
	u_int key_len;
	u_int8_t key[64];
	u_int text_len;
	u_int8_t plaintext[512];
	u_int8_t ciphertext[512];
};

struct aes_xts_tv aes_xts_test_vectors[] = {
	{
		0x00000000ULL,
		32,
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
		},
		32,
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
		}, {
			0x91, 0x7c, 0xf6, 0x9e, 0xbd, 0x68, 0xb2, 0xec, 
			0x9b, 0x9f, 0xe9, 0xa3, 0xea, 0xdd, 0xa6, 0x92, 
			0xcd, 0x43, 0xd2, 0xf5, 0x95, 0x98, 0xed, 0x85, 
			0x8c, 0x02, 0xc2, 0x65, 0x2f, 0xbf, 0x92, 0x2e, 
		},
	},
	{
		0x3333333333ULL,
		32,
		{
			0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 
			0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 
			0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 
			0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 
		},
		32,
		{
			0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 
			0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 
			0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 
			0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 
		}, {
			0xc4, 0x54, 0x18, 0x5e, 0x6a, 0x16, 0x93, 0x6e, 
			0x39, 0x33, 0x40, 0x38, 0xac, 0xef, 0x83, 0x8b, 
			0xfb, 0x18, 0x6f, 0xff, 0x74, 0x80, 0xad, 0xc4, 
			0x28, 0x93, 0x82, 0xec, 0xd6, 0xd3, 0x94, 0xf0, 
		},
	},
	{
		0x3333333333ULL,
		32,
		{
			0xff, 0xfe, 0xfd, 0xfc, 0xfb, 0xfa, 0xf9, 0xf8, 
			0xf7, 0xf6, 0xf5, 0xf4, 0xf3, 0xf2, 0xf1, 0xf0, 
			0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 
			0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 
		},
		32,
		{
			0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 
			0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 
			0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 
			0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 
		}, {
			0xaf, 0x85, 0x33, 0x6b, 0x59, 0x7a, 0xfc, 0x1a, 
			0x90, 0x0b, 0x2e, 0xb2, 0x1e, 0xc9, 0x49, 0xd2, 
			0x92, 0xdf, 0x4c, 0x04, 0x7e, 0x0b, 0x21, 0x53, 
			0x21, 0x86, 0xa5, 0x97, 0x1a, 0x22, 0x7a, 0x89, 
		},
	},
	{
		0x00000000ULL,
		32,
		{
			0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45, 
			0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26, 
			0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93, 
			0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95, 
		},
		512,
		{
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 
			0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 
			0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 
			0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 
			0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 
			0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 
			0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 
			0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 
			0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 
			0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 
			0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 
			0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 
			0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 
			0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 
			0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 
			0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
			0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 
			0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
			0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 
			0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
			0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 
			0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
			0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 
			0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 
			0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
			0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff, 
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 
			0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 
			0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 
			0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 
			0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 
			0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 
			0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 
			0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 
			0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 
			0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 
			0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 
			0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 
			0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 
			0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 
			0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 
			0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
			0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 
			0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
			0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 
			0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
			0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 
			0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
			0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 
			0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 
			0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
			0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff, 
		}, {
			0x27, 0xa7, 0x47, 0x9b, 0xef, 0xa1, 0xd4, 0x76, 
			0x48, 0x9f, 0x30, 0x8c, 0xd4, 0xcf, 0xa6, 0xe2, 
			0xa9, 0x6e, 0x4b, 0xbe, 0x32, 0x08, 0xff, 0x25, 
			0x28, 0x7d, 0xd3, 0x81, 0x96, 0x16, 0xe8, 0x9c, 
			0xc7, 0x8c, 0xf7, 0xf5, 0xe5, 0x43, 0x44, 0x5f, 
			0x83, 0x33, 0xd8, 0xfa, 0x7f, 0x56, 0x00, 0x00, 
			0x05, 0x27, 0x9f, 0xa5, 0xd8, 0xb5, 0xe4, 0xad, 
			0x40, 0xe7, 0x36, 0xdd, 0xb4, 0xd3, 0x54, 0x12, 
			0x32, 0x80, 0x63, 0xfd, 0x2a, 0xab, 0x53, 0xe5, 
			0xea, 0x1e, 0x0a, 0x9f, 0x33, 0x25, 0x00, 0xa5, 
			0xdf, 0x94, 0x87, 0xd0, 0x7a, 0x5c, 0x92, 0xcc, 
			0x51, 0x2c, 0x88, 0x66, 0xc7, 0xe8, 0x60, 0xce, 
			0x93, 0xfd, 0xf1, 0x66, 0xa2, 0x49, 0x12, 0xb4, 
			0x22, 0x97, 0x61, 0x46, 0xae, 0x20, 0xce, 0x84, 
			0x6b, 0xb7, 0xdc, 0x9b, 0xa9, 0x4a, 0x76, 0x7a, 
			0xae, 0xf2, 0x0c, 0x0d, 0x61, 0xad, 0x02, 0x65, 
			0x5e, 0xa9, 0x2d, 0xc4, 0xc4, 0xe4, 0x1a, 0x89, 
			0x52, 0xc6, 0x51, 0xd3, 0x31, 0x74, 0xbe, 0x51, 
			0xa1, 0x0c, 0x42, 0x11, 0x10, 0xe6, 0xd8, 0x15, 
			0x88, 0xed, 0xe8, 0x21, 0x03, 0xa2, 0x52, 0xd8, 
			0xa7, 0x50, 0xe8, 0x76, 0x8d, 0xef, 0xff, 0xed, 
			0x91, 0x22, 0x81, 0x0a, 0xae, 0xb9, 0x9f, 0x91, 
			0x72, 0xaf, 0x82, 0xb6, 0x04, 0xdc, 0x4b, 0x8e, 
			0x51, 0xbc, 0xb0, 0x82, 0x35, 0xa6, 0xf4, 0x34, 
			0x13, 0x32, 0xe4, 0xca, 0x60, 0x48, 0x2a, 0x4b, 
			0xa1, 0xa0, 0x3b, 0x3e, 0x65, 0x00, 0x8f, 0xc5, 
			0xda, 0x76, 0xb7, 0x0b, 0xf1, 0x69, 0x0d, 0xb4, 
			0xea, 0xe2, 0x9c, 0x5f, 0x1b, 0xad, 0xd0, 0x3c, 
			0x5c, 0xcf, 0x2a, 0x55, 0xd7, 0x05, 0xdd, 0xcd, 
			0x86, 0xd4, 0x49, 0x51, 0x1c, 0xeb, 0x7e, 0xc3, 
			0x0b, 0xf1, 0x2b, 0x1f, 0xa3, 0x5b, 0x91, 0x3f, 
			0x9f, 0x74, 0x7a, 0x8a, 0xfd, 0x1b, 0x13, 0x0e, 
			0x94, 0xbf, 0xf9, 0x4e, 0xff, 0xd0, 0x1a, 0x91, 
			0x73, 0x5c, 0xa1, 0x72, 0x6a, 0xcd, 0x0b, 0x19, 
			0x7c, 0x4e, 0x5b, 0x03, 0x39, 0x36, 0x97, 0xe1, 
			0x26, 0x82, 0x6f, 0xb6, 0xbb, 0xde, 0x8e, 0xcc, 
			0x1e, 0x08, 0x29, 0x85, 0x16, 0xe2, 0xc9, 0xed, 
			0x03, 0xff, 0x3c, 0x1b, 0x78, 0x60, 0xf6, 0xde, 
			0x76, 0xd4, 0xce, 0xcd, 0x94, 0xc8, 0x11, 0x98, 
			0x55, 0xef, 0x52, 0x97, 0xca, 0x67, 0xe9, 0xf3, 
			0xe7, 0xff, 0x72, 0xb1, 0xe9, 0x97, 0x85, 0xca, 
			0x0a, 0x7e, 0x77, 0x20, 0xc5, 0xb3, 0x6d, 0xc6, 
			0xd7, 0x2c, 0xac, 0x95, 0x74, 0xc8, 0xcb, 0xbc, 
			0x2f, 0x80, 0x1e, 0x23, 0xe5, 0x6f, 0xd3, 0x44, 
			0xb0, 0x7f, 0x22, 0x15, 0x4b, 0xeb, 0xa0, 0xf0, 
			0x8c, 0xe8, 0x89, 0x1e, 0x64, 0x3e, 0xd9, 0x95, 
			0xc9, 0x4d, 0x9a, 0x69, 0xc9, 0xf1, 0xb5, 0xf4, 
			0x99, 0x02, 0x7a, 0x78, 0x57, 0x2a, 0xee, 0xbd, 
			0x74, 0xd2, 0x0c, 0xc3, 0x98, 0x81, 0xc2, 0x13, 
			0xee, 0x77, 0x0b, 0x10, 0x10, 0xe4, 0xbe, 0xa7, 
			0x18, 0x84, 0x69, 0x77, 0xae, 0x11, 0x9f, 0x7a, 
			0x02, 0x3a, 0xb5, 0x8c, 0xca, 0x0a, 0xd7, 0x52, 
			0xaf, 0xe6, 0x56, 0xbb, 0x3c, 0x17, 0x25, 0x6a, 
			0x9f, 0x6e, 0x9b, 0xf1, 0x9f, 0xdd, 0x5a, 0x38, 
			0xfc, 0x82, 0xbb, 0xe8, 0x72, 0xc5, 0x53, 0x9e, 
			0xdb, 0x60, 0x9e, 0xf4, 0xf7, 0x9c, 0x20, 0x3e, 
			0xbb, 0x14, 0x0f, 0x2e, 0x58, 0x3c, 0xb2, 0xad, 
			0x15, 0xb4, 0xaa, 0x5b, 0x65, 0x50, 0x16, 0xa8, 
			0x44, 0x92, 0x77, 0xdb, 0xd4, 0x77, 0xef, 0x2c, 
			0x8d, 0x6c, 0x01, 0x7d, 0xb7, 0x38, 0xb1, 0x8d, 
			0xeb, 0x4a, 0x42, 0x7d, 0x19, 0x23, 0xce, 0x3f, 
			0xf2, 0x62, 0x73, 0x57, 0x79, 0xa4, 0x18, 0xf2, 
			0x0a, 0x28, 0x2d, 0xf9, 0x20, 0x14, 0x7b, 0xea, 
			0xbe, 0x42, 0x1e, 0xe5, 0x31, 0x9d, 0x05, 0x68, 
		},
	},
	{
		0x00000001ULL,
		32,
		{
			0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45, 
			0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26, 
			0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93, 
			0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95, 
		},
		512,
		{
			0x27, 0xa7, 0x47, 0x9b, 0xef, 0xa1, 0xd4, 0x76, 
			0x48, 0x9f, 0x30, 0x8c, 0xd4, 0xcf, 0xa6, 0xe2, 
			0xa9, 0x6e, 0x4b, 0xbe, 0x32, 0x08, 0xff, 0x25, 
			0x28, 0x7d, 0xd3, 0x81, 0x96, 0x16, 0xe8, 0x9c, 
			0xc7, 0x8c, 0xf7, 0xf5, 0xe5, 0x43, 0x44, 0x5f, 
			0x83, 0x33, 0xd8, 0xfa, 0x7f, 0x56, 0x00, 0x00, 
			0x05, 0x27, 0x9f, 0xa5, 0xd8, 0xb5, 0xe4, 0xad, 
			0x40, 0xe7, 0x36, 0xdd, 0xb4, 0xd3, 0x54, 0x12, 
			0x32, 0x80, 0x63, 0xfd, 0x2a, 0xab, 0x53, 0xe5, 
			0xea, 0x1e, 0x0a, 0x9f, 0x33, 0x25, 0x00, 0xa5, 
			0xdf, 0x94, 0x87, 0xd0, 0x7a, 0x5c, 0x92, 0xcc, 
			0x51, 0x2c, 0x88, 0x66, 0xc7, 0xe8, 0x60, 0xce, 
			0x93, 0xfd, 0xf1, 0x66, 0xa2, 0x49, 0x12, 0xb4, 
			0x22, 0x97, 0x61, 0x46, 0xae, 0x20, 0xce, 0x84, 
			0x6b, 0xb7, 0xdc, 0x9b, 0xa9, 0x4a, 0x76, 0x7a, 
			0xae, 0xf2, 0x0c, 0x0d, 0x61, 0xad, 0x02, 0x65, 
			0x5e, 0xa9, 0x2d, 0xc4, 0xc4, 0xe4, 0x1a, 0x89, 
			0x52, 0xc6, 0x51, 0xd3, 0x31, 0x74, 0xbe, 0x51, 
			0xa1, 0x0c, 0x42, 0x11, 0x10, 0xe6, 0xd8, 0x15, 
			0x88, 0xed, 0xe8, 0x21, 0x03, 0xa2, 0x52, 0xd8, 
			0xa7, 0x50, 0xe8, 0x76, 0x8d, 0xef, 0xff, 0xed, 
			0x91, 0x22, 0x81, 0x0a, 0xae, 0xb9, 0x9f, 0x91, 
			0x72, 0xaf, 0x82, 0xb6, 0x04, 0xdc, 0x4b, 0x8e, 
			0x51, 0xbc, 0xb0, 0x82, 0x35, 0xa6, 0xf4, 0x34, 
			0x13, 0x32, 0xe4, 0xca, 0x60, 0x48, 0x2a, 0x4b, 
			0xa1, 0xa0, 0x3b, 0x3e, 0x65, 0x00, 0x8f, 0xc5, 
			0xda, 0x76, 0xb7, 0x0b, 0xf1, 0x69, 0x0d, 0xb4, 
			0xea, 0xe2, 0x9c, 0x5f, 0x1b, 0xad, 0xd0, 0x3c, 
			0x5c, 0xcf, 0x2a, 0x55, 0xd7, 0x05, 0xdd, 0xcd, 
			0x86, 0xd4, 0x49, 0x51, 0x1c, 0xeb, 0x7e, 0xc3, 
			0x0b, 0xf1, 0x2b, 0x1f, 0xa3, 0x5b, 0x91, 0x3f, 
			0x9f, 0x74, 0x7a, 0x8a, 0xfd, 0x1b, 0x13, 0x0e, 
			0x94, 0xbf, 0xf9, 0x4e, 0xff, 0xd0, 0x1a, 0x91, 
			0x73, 0x5c, 0xa1, 0x72, 0x6a, 0xcd, 0x0b, 0x19, 
			0x7c, 0x4e, 0x5b, 0x03, 0x39, 0x36, 0x97, 0xe1, 
			0x26, 0x82, 0x6f, 0xb6, 0xbb, 0xde, 0x8e, 0xcc, 
			0x1e, 0x08, 0x29, 0x85, 0x16, 0xe2, 0xc9, 0xed, 
			0x03, 0xff, 0x3c, 0x1b, 0x78, 0x60, 0xf6, 0xde, 
			0x76, 0xd4, 0xce, 0xcd, 0x94, 0xc8, 0x11, 0x98, 
			0x55, 0xef, 0x52, 0x97, 0xca, 0x67, 0xe9, 0xf3, 
			0xe7, 0xff, 0x72, 0xb1, 0xe9, 0x97, 0x85, 0xca, 
			0x0a, 0x7e, 0x77, 0x20, 0xc5, 0xb3, 0x6d, 0xc6, 
			0xd7, 0x2c, 0xac, 0x95, 0x74, 0xc8, 0xcb, 0xbc, 
			0x2f, 0x80, 0x1e, 0x23, 0xe5, 0x6f, 0xd3, 0x44, 
			0xb0, 0x7f, 0x22, 0x15, 0x4b, 0xeb, 0xa0, 0xf0, 
			0x8c, 0xe8, 0x89, 0x1e, 0x64, 0x3e, 0xd9, 0x95, 
			0xc9, 0x4d, 0x9a, 0x69, 0xc9, 0xf1, 0xb5, 0xf4, 
			0x99, 0x02, 0x7a, 0x78, 0x57, 0x2a, 0xee, 0xbd, 
			0x74, 0xd2, 0x0c, 0xc3, 0x98, 0x81, 0xc2, 0x13, 
			0xee, 0x77, 0x0b, 0x10, 0x10, 0xe4, 0xbe, 0xa7, 
			0x18, 0x84, 0x69, 0x77, 0xae, 0x11, 0x9f, 0x7a, 
			0x02, 0x3a, 0xb5, 0x8c, 0xca, 0x0a, 0xd7, 0x52, 
			0xaf, 0xe6, 0x56, 0xbb, 0x3c, 0x17, 0x25, 0x6a, 
			0x9f, 0x6e, 0x9b, 0xf1, 0x9f, 0xdd, 0x5a, 0x38, 
			0xfc, 0x82, 0xbb, 0xe8, 0x72, 0xc5, 0x53, 0x9e, 
			0xdb, 0x60, 0x9e, 0xf4, 0xf7, 0x9c, 0x20, 0x3e, 
			0xbb, 0x14, 0x0f, 0x2e, 0x58, 0x3c, 0xb2, 0xad, 
			0x15, 0xb4, 0xaa, 0x5b, 0x65, 0x50, 0x16, 0xa8, 
			0x44, 0x92, 0x77, 0xdb, 0xd4, 0x77, 0xef, 0x2c, 
			0x8d, 0x6c, 0x01, 0x7d, 0xb7, 0x38, 0xb1, 0x8d, 
			0xeb, 0x4a, 0x42, 0x7d, 0x19, 0x23, 0xce, 0x3f, 
			0xf2, 0x62, 0x73, 0x57, 0x79, 0xa4, 0x18, 0xf2, 
			0x0a, 0x28, 0x2d, 0xf9, 0x20, 0x14, 0x7b, 0xea, 
			0xbe, 0x42, 0x1e, 0xe5, 0x31, 0x9d, 0x05, 0x68, 
		}, {
			0x26, 0x4d, 0x3c, 0xa8, 0x51, 0x21, 0x94, 0xfe, 
			0xc3, 0x12, 0xc8, 0xc9, 0x89, 0x1f, 0x27, 0x9f, 
			0xef, 0xdd, 0x60, 0x8d, 0x0c, 0x02, 0x7b, 0x60, 
			0x48, 0x3a, 0x3f, 0xa8, 0x11, 0xd6, 0x5e, 0xe5, 
			0x9d, 0x52, 0xd9, 0xe4, 0x0e, 0xc5, 0x67, 0x2d, 
			0x81, 0x53, 0x2b, 0x38, 0xb6, 0xb0, 0x89, 0xce, 
			0x95, 0x1f, 0x0f, 0x9c, 0x35, 0x59, 0x0b, 0x8b, 
			0x97, 0x8d, 0x17, 0x52, 0x13, 0xf3, 0x29, 0xbb, 
			0x1c, 0x2f, 0xd3, 0x0f, 0x2f, 0x7f, 0x30, 0x49, 
			0x2a, 0x61, 0xa5, 0x32, 0xa7, 0x9f, 0x51, 0xd3, 
			0x6f, 0x5e, 0x31, 0xa7, 0xc9, 0xa1, 0x2c, 0x28, 
			0x60, 0x82, 0xff, 0x7d, 0x23, 0x94, 0xd1, 0x8f, 
			0x78, 0x3e, 0x1a, 0x8e, 0x72, 0xc7, 0x22, 0xca, 
			0xaa, 0xa5, 0x2d, 0x8f, 0x06, 0x56, 0x57, 0xd2, 
			0x63, 0x1f, 0xd2, 0x5b, 0xfd, 0x8e, 0x5b, 0xaa, 
			0xd6, 0xe5, 0x27, 0xd7, 0x63, 0x51, 0x75, 0x01, 
			0xc6, 0x8c, 0x5e, 0xdc, 0x3c, 0xdd, 0x55, 0x43, 
			0x5c, 0x53, 0x2d, 0x71, 0x25, 0xc8, 0x61, 0x4d, 
			0xee, 0xd9, 0xad, 0xaa, 0x3a, 0xca, 0xde, 0x58, 
			0x88, 0xb8, 0x7b, 0xef, 0x64, 0x1c, 0x4c, 0x99, 
			0x4c, 0x80, 0x91, 0xb5, 0xbc, 0xd3, 0x87, 0xf3, 
			0x96, 0x3f, 0xb5, 0xbc, 0x37, 0xaa, 0x92, 0x2f, 
			0xbf, 0xe3, 0xdf, 0x4e, 0x5b, 0x91, 0x5e, 0x6e, 
			0xb5, 0x14, 0x71, 0x7b, 0xdd, 0x2a, 0x74, 0x07, 
			0x9a, 0x50, 0x73, 0xf5, 0xc4, 0xbf, 0xd4, 0x6a, 
			0xdf, 0x7d, 0x28, 0x2e, 0x7a, 0x39, 0x3a, 0x52, 
			0x57, 0x9d, 0x11, 0xa0, 0x28, 0xda, 0x4d, 0x9c, 
			0xd9, 0xc7, 0x71, 0x24, 0xf9, 0x64, 0x8e, 0xe3, 
			0x83, 0xb1, 0xac, 0x76, 0x39, 0x30, 0xe7, 0x16, 
			0x2a, 0x8d, 0x37, 0xf3, 0x50, 0xb2, 0xf7, 0x4b, 
			0x84, 0x72, 0xcf, 0x09, 0x90, 0x20, 0x63, 0xc6, 
			0xb3, 0x2e, 0x8c, 0x2d, 0x92, 0x90, 0xce, 0xfb, 
			0xd7, 0x34, 0x6d, 0x1c, 0x77, 0x9a, 0x0d, 0xf5, 
			0x0e, 0xdc, 0xde, 0x45, 0x31, 0xda, 0x07, 0xb0, 
			0x99, 0xc6, 0x38, 0xe8, 0x3a, 0x75, 0x59, 0x44, 
			0xdf, 0x2a, 0xef, 0x1a, 0xa3, 0x17, 0x52, 0xfd, 
			0x32, 0x3d, 0xcb, 0x71, 0x0f, 0xb4, 0xbf, 0xbb, 
			0x9d, 0x22, 0xb9, 0x25, 0xbc, 0x35, 0x77, 0xe1, 
			0xb8, 0x94, 0x9e, 0x72, 0x9a, 0x90, 0xbb, 0xaf, 
			0xea, 0xcf, 0x7f, 0x78, 0x79, 0xe7, 0xb1, 0x14, 
			0x7e, 0x28, 0xba, 0x0b, 0xae, 0x94, 0x0d, 0xb7, 
			0x95, 0xa6, 0x1b, 0x15, 0xec, 0xf4, 0xdf, 0x8d, 
			0xb0, 0x7b, 0x82, 0x4b, 0xb0, 0x62, 0x80, 0x2c, 
			0xc9, 0x8a, 0x95, 0x45, 0xbb, 0x2a, 0xae, 0xed, 
			0x77, 0xcb, 0x3f, 0xc6, 0xdb, 0x15, 0xdc, 0xd7, 
			0xd8, 0x0d, 0x7d, 0x5b, 0xc4, 0x06, 0xc4, 0x97, 
			0x0a, 0x34, 0x78, 0xad, 0xa8, 0x89, 0x9b, 0x32, 
			0x91, 0x98, 0xeb, 0x61, 0xc1, 0x93, 0xfb, 0x62, 
			0x75, 0xaa, 0x8c, 0xa3, 0x40, 0x34, 0x4a, 0x75, 
			0xa8, 0x62, 0xae, 0xbe, 0x92, 0xee, 0xe1, 0xce, 
			0x03, 0x2f, 0xd9, 0x50, 0xb4, 0x7d, 0x77, 0x04, 
			0xa3, 0x87, 0x69, 0x23, 0xb4, 0xad, 0x62, 0x84, 
			0x4b, 0xf4, 0xa0, 0x9c, 0x4d, 0xbe, 0x8b, 0x43, 
			0x97, 0x18, 0x4b, 0x74, 0x71, 0x36, 0x0c, 0x95, 
			0x64, 0x88, 0x0a, 0xed, 0xdd, 0xb9, 0xba, 0xa4, 
			0xaf, 0x2e, 0x75, 0x39, 0x4b, 0x08, 0xcd, 0x32, 
			0xff, 0x47, 0x9c, 0x57, 0xa0, 0x7d, 0x3e, 0xab, 
			0x5d, 0x54, 0xde, 0x5f, 0x97, 0x38, 0xb8, 0xd2, 
			0x7f, 0x27, 0xa9, 0xf0, 0xab, 0x11, 0x79, 0x9d, 
			0x7b, 0x7f, 0xfe, 0xfb, 0x27, 0x04, 0xc9, 0x5c, 
			0x6a, 0xd1, 0x2c, 0x39, 0xf1, 0xe8, 0x67, 0xa4, 
			0xb7, 0xb1, 0xd7, 0x81, 0x8a, 0x4b, 0x75, 0x3d, 
			0xfd, 0x2a, 0x89, 0xcc, 0xb4, 0x5e, 0x00, 0x1a, 
			0x03, 0xa8, 0x67, 0xb1, 0x87, 0xf2, 0x25, 0xdd, 
		},
	},
	{
		0x00000002ULL,
		32,
		{
			0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45, 
			0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26, 
			0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93, 
			0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95, 
		},
		512,
		{
			0x26, 0x4d, 0x3c, 0xa8, 0x51, 0x21, 0x94, 0xfe, 
			0xc3, 0x12, 0xc8, 0xc9, 0x89, 0x1f, 0x27, 0x9f, 
			0xef, 0xdd, 0x60, 0x8d, 0x0c, 0x02, 0x7b, 0x60, 
			0x48, 0x3a, 0x3f, 0xa8, 0x11, 0xd6, 0x5e, 0xe5, 
			0x9d, 0x52, 0xd9, 0xe4, 0x0e, 0xc5, 0x67, 0x2d, 
			0x81, 0x53, 0x2b, 0x38, 0xb6, 0xb0, 0x89, 0xce, 
			0x95, 0x1f, 0x0f, 0x9c, 0x35, 0x59, 0x0b, 0x8b, 
			0x97, 0x8d, 0x17, 0x52, 0x13, 0xf3, 0x29, 0xbb, 
			0x1c, 0x2f, 0xd3, 0x0f, 0x2f, 0x7f, 0x30, 0x49, 
			0x2a, 0x61, 0xa5, 0x32, 0xa7, 0x9f, 0x51, 0xd3, 
			0x6f, 0x5e, 0x31, 0xa7, 0xc9, 0xa1, 0x2c, 0x28, 
			0x60, 0x82, 0xff, 0x7d, 0x23, 0x94, 0xd1, 0x8f, 
			0x78, 0x3e, 0x1a, 0x8e, 0x72, 0xc7, 0x22, 0xca, 
			0xaa, 0xa5, 0x2d, 0x8f, 0x06, 0x56, 0x57, 0xd2, 
			0x63, 0x1f, 0xd2, 0x5b, 0xfd, 0x8e, 0x5b, 0xaa, 
			0xd6, 0xe5, 0x27, 0xd7, 0x63, 0x51, 0x75, 0x01, 
			0xc6, 0x8c, 0x5e, 0xdc, 0x3c, 0xdd, 0x55, 0x43, 
			0x5c, 0x53, 0x2d, 0x71, 0x25, 0xc8, 0x61, 0x4d, 
			0xee, 0xd9, 0xad, 0xaa, 0x3a, 0xca, 0xde, 0x58, 
			0x88, 0xb8, 0x7b, 0xef, 0x64, 0x1c, 0x4c, 0x99, 
			0x4c, 0x80, 0x91, 0xb5, 0xbc, 0xd3, 0x87, 0xf3, 
			0x96, 0x3f, 0xb5, 0xbc, 0x37, 0xaa, 0x92, 0x2f, 
			0xbf, 0xe3, 0xdf, 0x4e, 0x5b, 0x91, 0x5e, 0x6e, 
			0xb5, 0x14, 0x71, 0x7b, 0xdd, 0x2a, 0x74, 0x07, 
			0x9a, 0x50, 0x73, 0xf5, 0xc4, 0xbf, 0xd4, 0x6a, 
			0xdf, 0x7d, 0x28, 0x2e, 0x7a, 0x39, 0x3a, 0x52, 
			0x57, 0x9d, 0x11, 0xa0, 0x28, 0xda, 0x4d, 0x9c, 
			0xd9, 0xc7, 0x71, 0x24, 0xf9, 0x64, 0x8e, 0xe3, 
			0x83, 0xb1, 0xac, 0x76, 0x39, 0x30, 0xe7, 0x16, 
			0x2a, 0x8d, 0x37, 0xf3, 0x50, 0xb2, 0xf7, 0x4b, 
			0x84, 0x72, 0xcf, 0x09, 0x90, 0x20, 0x63, 0xc6, 
			0xb3, 0x2e, 0x8c, 0x2d, 0x92, 0x90, 0xce, 0xfb, 
			0xd7, 0x34, 0x6d, 0x1c, 0x77, 0x9a, 0x0d, 0xf5, 
			0x0e, 0xdc, 0xde, 0x45, 0x31, 0xda, 0x07, 0xb0, 
			0x99, 0xc6, 0x38, 0xe8, 0x3a, 0x75, 0x59, 0x44, 
			0xdf, 0x2a, 0xef, 0x1a, 0xa3, 0x17, 0x52, 0xfd, 
			0x32, 0x3d, 0xcb, 0x71, 0x0f, 0xb4, 0xbf, 0xbb, 
			0x9d, 0x22, 0xb9, 0x25, 0xbc, 0x35, 0x77, 0xe1, 
			0xb8, 0x94, 0x9e, 0x72, 0x9a, 0x90, 0xbb, 0xaf, 
			0xea, 0xcf, 0x7f, 0x78, 0x79, 0xe7, 0xb1, 0x14, 
			0x7e, 0x28, 0xba, 0x0b, 0xae, 0x94, 0x0d, 0xb7, 
			0x95, 0xa6, 0x1b, 0x15, 0xec, 0xf4, 0xdf, 0x8d, 
			0xb0, 0x7b, 0x82, 0x4b, 0xb0, 0x62, 0x80, 0x2c, 
			0xc9, 0x8a, 0x95, 0x45, 0xbb, 0x2a, 0xae, 0xed, 
			0x77, 0xcb, 0x3f, 0xc6, 0xdb, 0x15, 0xdc, 0xd7, 
			0xd8, 0x0d, 0x7d, 0x5b, 0xc4, 0x06, 0xc4, 0x97, 
			0x0a, 0x34, 0x78, 0xad, 0xa8, 0x89, 0x9b, 0x32, 
			0x91, 0x98, 0xeb, 0x61, 0xc1, 0x93, 0xfb, 0x62, 
			0x75, 0xaa, 0x8c, 0xa3, 0x40, 0x34, 0x4a, 0x75, 
			0xa8, 0x62, 0xae, 0xbe, 0x92, 0xee, 0xe1, 0xce, 
			0x03, 0x2f, 0xd9, 0x50, 0xb4, 0x7d, 0x77, 0x04, 
			0xa3, 0x87, 0x69, 0x23, 0xb4, 0xad, 0x62, 0x84, 
			0x4b, 0xf4, 0xa0, 0x9c, 0x4d, 0xbe, 0x8b, 0x43, 
			0x97, 0x18, 0x4b, 0x74, 0x71, 0x36, 0x0c, 0x95, 
			0x64, 0x88, 0x0a, 0xed, 0xdd, 0xb9, 0xba, 0xa4, 
			0xaf, 0x2e, 0x75, 0x39, 0x4b, 0x08, 0xcd, 0x32, 
			0xff, 0x47, 0x9c, 0x57, 0xa0, 0x7d, 0x3e, 0xab, 
			0x5d, 0x54, 0xde, 0x5f, 0x97, 0x38, 0xb8, 0xd2, 
			0x7f, 0x27, 0xa9, 0xf0, 0xab, 0x11, 0x79, 0x9d, 
			0x7b, 0x7f, 0xfe, 0xfb, 0x27, 0x04, 0xc9, 0x5c, 
			0x6a, 0xd1, 0x2c, 0x39, 0xf1, 0xe8, 0x67, 0xa4, 
			0xb7, 0xb1, 0xd7, 0x81, 0x8a, 0x4b, 0x75, 0x3d, 
			0xfd, 0x2a, 0x89, 0xcc, 0xb4, 0x5e, 0x00, 0x1a, 
			0x03, 0xa8, 0x67, 0xb1, 0x87, 0xf2, 0x25, 0xdd, 
		}, {
			0xfa, 0x76, 0x2a, 0x36, 0x80, 0xb7, 0x60, 0x07, 
			0x92, 0x8e, 0xd4, 0xa4, 0xf4, 0x9a, 0x94, 0x56, 
			0x03, 0x1b, 0x70, 0x47, 0x82, 0xe6, 0x5e, 0x16, 
			0xce, 0xcb, 0x54, 0xed, 0x7d, 0x01, 0x7b, 0x5e, 
			0x18, 0xab, 0xd6, 0x7b, 0x33, 0x8e, 0x81, 0x07, 
			0x8f, 0x21, 0xed, 0xb7, 0x86, 0x8d, 0x90, 0x1e, 
			0xbe, 0x9c, 0x73, 0x1a, 0x7c, 0x18, 0xb5, 0xe6, 
			0xde, 0xc1, 0xd6, 0xa7, 0x2e, 0x07, 0x8a, 0xc9, 
			0xa4, 0x26, 0x2f, 0x86, 0x0b, 0xee, 0xfa, 0x14, 
			0xf4, 0xe8, 0x21, 0x01, 0x82, 0x72, 0xe4, 0x11, 
			0xa9, 0x51, 0x50, 0x2b, 0x6e, 0x79, 0x06, 0x6e, 
			0x84, 0x25, 0x2c, 0x33, 0x46, 0xf3, 0xaa, 0x62, 
			0x34, 0x43, 0x51, 0xa2, 0x91, 0xd4, 0xbe, 0xdc, 
			0x7a, 0x07, 0x61, 0x8b, 0xde, 0xa2, 0xaf, 0x63, 
			0x14, 0x5c, 0xc7, 0xa4, 0xb8, 0xd4, 0x07, 0x06, 
			0x91, 0xae, 0x89, 0x0c, 0xd6, 0x57, 0x33, 0xe7, 
			0x94, 0x6e, 0x90, 0x21, 0xa1, 0xdf, 0xfc, 0x4c, 
			0x59, 0xf1, 0x59, 0x42, 0x5e, 0xe6, 0xd5, 0x0c, 
			0xa9, 0xb1, 0x35, 0xfa, 0x61, 0x62, 0xce, 0xa1, 
			0x8a, 0x93, 0x98, 0x38, 0xdc, 0x00, 0x0f, 0xb3, 
			0x86, 0xfa, 0xd0, 0x86, 0xac, 0xce, 0x5a, 0xc0, 
			0x7c, 0xb2, 0xec, 0xe7, 0xfd, 0x58, 0x0b, 0x00, 
			0xcf, 0xa5, 0xe9, 0x85, 0x89, 0x63, 0x1d, 0xc2, 
			0x5e, 0x8e, 0x2a, 0x3d, 0xaf, 0x2f, 0xfd, 0xec, 
			0x26, 0x53, 0x16, 0x59, 0x91, 0x2c, 0x9d, 0x8f, 
			0x7a, 0x15, 0xe5, 0x86, 0x5e, 0xa8, 0xfb, 0x58, 
			0x16, 0xd6, 0x20, 0x70, 0x52, 0xbd, 0x71, 0x28, 
			0xcd, 0x74, 0x3c, 0x12, 0xc8, 0x11, 0x87, 0x91, 
			0xa4, 0x73, 0x68, 0x11, 0x93, 0x5e, 0xb9, 0x82, 
			0xa5, 0x32, 0x34, 0x9e, 0x31, 0xdd, 0x40, 0x1e, 
			0x0b, 0x66, 0x0a, 0x56, 0x8c, 0xb1, 0xa4, 0x71, 
			0x1f, 0x55, 0x2f, 0x55, 0xde, 0xd5, 0x9f, 0x1f, 
			0x15, 0xbf, 0x71, 0x96, 0xb3, 0xca, 0x12, 0xa9, 
			0x1e, 0x48, 0x8e, 0xf5, 0x9d, 0x64, 0xf3, 0xa0, 
			0x2b, 0xf4, 0x52, 0x39, 0x49, 0x9a, 0xc6, 0x17, 
			0x6a, 0xe3, 0x21, 0xc4, 0xa2, 0x11, 0xec, 0x54, 
			0x53, 0x65, 0x97, 0x1c, 0x5d, 0x3f, 0x4f, 0x09, 
			0xd4, 0xeb, 0x13, 0x9b, 0xfd, 0xf2, 0x07, 0x3d, 
			0x33, 0x18, 0x0b, 0x21, 0x00, 0x2b, 0x65, 0xcc, 
			0x98, 0x65, 0xe7, 0x6c, 0xb2, 0x4c, 0xd9, 0x2c, 
			0x87, 0x4c, 0x24, 0xc1, 0x83, 0x50, 0x39, 0x9a, 
			0x93, 0x6a, 0xb3, 0x63, 0x70, 0x79, 0x29, 0x5d, 
			0x76, 0xc4, 0x17, 0x77, 0x6b, 0x94, 0xef, 0xce, 
			0x3a, 0x0e, 0xf7, 0x20, 0x6b, 0x15, 0x11, 0x05, 
			0x19, 0x65, 0x5c, 0x95, 0x6c, 0xbd, 0x8b, 0x24, 
			0x89, 0x40, 0x5e, 0xe2, 0xb0, 0x9a, 0x6b, 0x6e, 
			0xeb, 0xe0, 0xc5, 0x37, 0x90, 0xa1, 0x2a, 0x89, 
			0x98, 0x37, 0x8b, 0x33, 0xa5, 0xb7, 0x11, 0x59, 
			0x62, 0x5f, 0x4b, 0xa4, 0x9d, 0x2a, 0x2f, 0xdb, 
			0xa5, 0x9f, 0xbf, 0x08, 0x97, 0xbc, 0x7a, 0xab, 
			0xd8, 0xd7, 0x07, 0xdc, 0x14, 0x0a, 0x80, 0xf0, 
			0xf3, 0x09, 0xf8, 0x35, 0xd3, 0xda, 0x54, 0xab, 
			0x58, 0x4e, 0x50, 0x1d, 0xfa, 0x0e, 0xe9, 0x77, 
			0xfe, 0xc5, 0x43, 0xf7, 0x41, 0x86, 0xa8, 0x02, 
			0xb9, 0xa3, 0x7a, 0xdb, 0x3e, 0x82, 0x91, 0xec, 
			0xa0, 0x4d, 0x66, 0x52, 0x0d, 0x22, 0x9e, 0x60, 
			0x40, 0x1e, 0x72, 0x82, 0xbe, 0xf4, 0x86, 0xae, 
			0x05, 0x9a, 0xa7, 0x06, 0x96, 0xe0, 0xe3, 0x05, 
			0xd7, 0x77, 0x14, 0x0a, 0x7a, 0x88, 0x3e, 0xcd, 
			0xcb, 0x69, 0xb9, 0xff, 0x93, 0x8e, 0x8a, 0x42, 
			0x31, 0x86, 0x4c, 0x69, 0xca, 0x2c, 0x20, 0x43, 
			0xbe, 0xd0, 0x07, 0xff, 0x3e, 0x60, 0x5e, 0x01, 
			0x4b, 0xcf, 0x51, 0x81, 0x38, 0xdc, 0x3a, 0x25, 
			0xc5, 0xe2, 0x36, 0x17, 0x1a, 0x2d, 0x01, 0xd6, 
		},
	},
	{
		0x000000fdULL,
		32,
		{
			0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45, 
			0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26, 
			0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93, 
			0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95, 
		},
		512,
		{
			0x8e, 0x41, 0xb7, 0x8c, 0x39, 0x0b, 0x5a, 0xf9, 
			0xd7, 0x58, 0xbb, 0x21, 0x4a, 0x67, 0xe9, 0xf6, 
			0xbf, 0x77, 0x27, 0xb0, 0x9a, 0xc6, 0x12, 0x40, 
			0x84, 0xc3, 0x76, 0x11, 0x39, 0x8f, 0xa4, 0x5d, 
			0xaa, 0xd9, 0x48, 0x68, 0x60, 0x0e, 0xd3, 0x91, 
			0xfb, 0x1a, 0xcd, 0x48, 0x57, 0xa9, 0x5b, 0x46, 
			0x6e, 0x62, 0xef, 0x9f, 0x4b, 0x37, 0x72, 0x44, 
			0xd1, 0xc1, 0x52, 0xe7, 0xb3, 0x0d, 0x73, 0x1a, 
			0xad, 0x30, 0xc7, 0x16, 0xd2, 0x14, 0xb7, 0x07, 
			0xae, 0xd9, 0x9e, 0xb5, 0xb5, 0xe5, 0x80, 0xb3, 
			0xe8, 0x87, 0xcf, 0x74, 0x97, 0x46, 0x56, 0x51, 
			0xd4, 0xb6, 0x0e, 0x60, 0x42, 0x05, 0x1d, 0xa3, 
			0x69, 0x3c, 0x3b, 0x78, 0xc1, 0x44, 0x89, 0x54, 
			0x3b, 0xe8, 0xb6, 0xad, 0x0b, 0xa6, 0x29, 0x56, 
			0x5b, 0xba, 0x20, 0x23, 0x13, 0xba, 0x7b, 0x0d, 
			0x0c, 0x94, 0xa3, 0x25, 0x2b, 0x67, 0x6f, 0x46, 
			0xcc, 0x02, 0xce, 0x0f, 0x8a, 0x7d, 0x34, 0xc0, 
			0xed, 0x22, 0x91, 0x29, 0x67, 0x3c, 0x1f, 0x61, 
			0xae, 0xd5, 0x79, 0xd0, 0x8a, 0x92, 0x03, 0xa2, 
			0x5a, 0xac, 0x3a, 0x77, 0xe9, 0xdb, 0x60, 0x26, 
			0x79, 0x96, 0xdb, 0x38, 0xdf, 0x63, 0x73, 0x56, 
			0xd9, 0xdc, 0xd1, 0x63, 0x2e, 0x36, 0x99, 0x39, 
			0xf2, 0xa2, 0x9d, 0x89, 0x34, 0x5c, 0x66, 0xe0, 
			0x50, 0x66, 0xf1, 0xa3, 0x67, 0x7a, 0xef, 0x18, 
			0xde, 0xa4, 0x11, 0x3f, 0xae, 0xb6, 0x29, 0xe4, 
			0x67, 0x21, 0xa6, 0x6d, 0x0a, 0x7e, 0x78, 0x5d, 
			0x3e, 0x29, 0xaf, 0x25, 0x94, 0xeb, 0x67, 0xdf, 
			0xa9, 0x82, 0xaf, 0xfe, 0x0a, 0xac, 0x05, 0x8f, 
			0x6e, 0x15, 0x86, 0x42, 0x69, 0xb1, 0x35, 0x41, 
			0x82, 0x61, 0xfc, 0x3a, 0xfb, 0x08, 0x94, 0x72, 
			0xcf, 0x68, 0xc4, 0x5d, 0xd7, 0xf2, 0x31, 0xc6, 
			0x24, 0x9b, 0xa0, 0x25, 0x5e, 0x1e, 0x03, 0x38, 
			0x33, 0xfc, 0x4d, 0x00, 0xa3, 0xfe, 0x02, 0x13, 
			0x2d, 0x7b, 0xc3, 0x87, 0x36, 0x14, 0xb8, 0xae, 
			0xe3, 0x42, 0x73, 0x58, 0x1e, 0xa0, 0x32, 0x5c, 
			0x81, 0xf0, 0x27, 0x0a, 0xff, 0xa1, 0x36, 0x41, 
			0xd0, 0x52, 0xd3, 0x6f, 0x07, 0x57, 0xd4, 0x84, 
			0x01, 0x43, 0x54, 0xd0, 0x2d, 0x68, 0x83, 0xca, 
			0x15, 0xc2, 0x4d, 0x8c, 0x39, 0x56, 0xb1, 0xbd, 
			0x02, 0x7b, 0xcf, 0x41, 0xf1, 0x51, 0xfd, 0x80, 
			0x23, 0xc5, 0x34, 0x0e, 0x56, 0x06, 0xf3, 0x7e, 
			0x90, 0xfd, 0xb8, 0x7c, 0x86, 0xfb, 0x4f, 0xa6, 
			0x34, 0xb3, 0x71, 0x8a, 0x30, 0xba, 0xce, 0x06, 
			0xa6, 0x6e, 0xaf, 0x8f, 0x63, 0xc4, 0xaa, 0x3b, 
			0x63, 0x78, 0x26, 0xa8, 0x7f, 0xe8, 0xcf, 0xa4, 
			0x42, 0x82, 0xe9, 0x2c, 0xb1, 0x61, 0x5a, 0xf3, 
			0xa2, 0x8e, 0x53, 0xbc, 0x74, 0xc7, 0xcb, 0xa1, 
			0xa0, 0x97, 0x7b, 0xe9, 0x06, 0x5d, 0x0c, 0x1a, 
			0x5d, 0xec, 0x6c, 0x54, 0xae, 0x38, 0xd3, 0x7f, 
			0x37, 0xaa, 0x35, 0x28, 0x3e, 0x04, 0x8e, 0x55, 
			0x30, 0xa8, 0x5c, 0x4e, 0x7a, 0x29, 0xd7, 0xb9, 
			0x2e, 0xc0, 0xc3, 0x16, 0x9c, 0xdf, 0x2a, 0x80, 
			0x5c, 0x76, 0x04, 0xbc, 0xe6, 0x00, 0x49, 0xb9, 
			0xfb, 0x7b, 0x8e, 0xaa, 0xc1, 0x0f, 0x51, 0xae, 
			0x23, 0x79, 0x4c, 0xeb, 0xa6, 0x8b, 0xb5, 0x81, 
			0x12, 0xe2, 0x93, 0xb9, 0xb6, 0x92, 0xca, 0x72, 
			0x1b, 0x37, 0xc6, 0x62, 0xf8, 0x57, 0x4e, 0xd4, 
			0xdb, 0xa6, 0xf8, 0x8e, 0x17, 0x08, 0x81, 0xc8, 
			0x2c, 0xdd, 0xc1, 0x03, 0x4a, 0x0c, 0xa7, 0xe2, 
			0x84, 0xbf, 0x09, 0x62, 0xb6, 0xb2, 0x62, 0x92, 
			0xd8, 0x36, 0xfa, 0x9f, 0x73, 0xc1, 0xac, 0x77, 
			0x0e, 0xef, 0x0f, 0x2d, 0x3a, 0x1e, 0xaf, 0x61, 
			0xd3, 0xe0, 0x35, 0x55, 0xfd, 0x42, 0x4e, 0xed, 
			0xd6, 0x7e, 0x18, 0xa1, 0x80, 0x94, 0xf8, 0x88, 
		}, {
			0xd5, 0x5f, 0x68, 0x4f, 0x81, 0xf4, 0x42, 0x6e, 
			0x9f, 0xde, 0x92, 0xa5, 0xff, 0x02, 0xdf, 0x2a, 
			0xc8, 0x96, 0xaf, 0x63, 0x96, 0x28, 0x88, 0xa9, 
			0x79, 0x10, 0xc1, 0x37, 0x9e, 0x20, 0xb0, 0xa3, 
			0xb1, 0xdb, 0x61, 0x3f, 0xb7, 0xfe, 0x2e, 0x07, 
			0x00, 0x43, 0x29, 0xea, 0x5c, 0x22, 0xbf, 0xd3, 
			0x3e, 0x3d, 0xbe, 0x4c, 0xf5, 0x8c, 0xc6, 0x08, 
			0xc2, 0xc2, 0x6c, 0x19, 0xa2, 0xe2, 0xfe, 0x22, 
			0xf9, 0x87, 0x32, 0xc2, 0xb5, 0xcb, 0x84, 0x4c, 
			0xc6, 0xc0, 0x70, 0x2d, 0x91, 0xe1, 0xd5, 0x0f, 
			0xc4, 0x38, 0x2a, 0x7e, 0xba, 0x56, 0x35, 0xcd, 
			0x60, 0x24, 0x32, 0xa2, 0x30, 0x6a, 0xc4, 0xce, 
			0x82, 0xf8, 0xd7, 0x0c, 0x8d, 0x9b, 0xc1, 0x5f, 
			0x91, 0x8f, 0xe7, 0x1e, 0x74, 0xc6, 0x22, 0xd5, 
			0xcf, 0x71, 0x17, 0x8b, 0xf6, 0xe0, 0xb9, 0xcc, 
			0x9f, 0x2b, 0x41, 0xdd, 0x8d, 0xbe, 0x44, 0x1c, 
			0x41, 0xcd, 0x0c, 0x73, 0xa6, 0xdc, 0x47, 0xa3, 
			0x48, 0xf6, 0x70, 0x2f, 0x9d, 0x0e, 0x9b, 0x1b, 
			0x14, 0x31, 0xe9, 0x48, 0xe2, 0x99, 0xb9, 0xec, 
			0x22, 0x72, 0xab, 0x2c, 0x5f, 0x0c, 0x7b, 0xe8, 
			0x6a, 0xff, 0xa5, 0xde, 0xc8, 0x7a, 0x0b, 0xee, 
			0x81, 0xd3, 0xd5, 0x00, 0x07, 0xed, 0xaa, 0x2b, 
			0xcf, 0xcc, 0xb3, 0x56, 0x05, 0x15, 0x5f, 0xf3, 
			0x6e, 0xd8, 0xed, 0xd4, 0xa4, 0x0d, 0xcd, 0x4b, 
			0x24, 0x3a, 0xcd, 0x11, 0xb2, 0xb9, 0x87, 0xbd, 
			0xbf, 0xaf, 0x91, 0xa7, 0xca, 0xc2, 0x7e, 0x9c, 
			0x5a, 0xea, 0x52, 0x5e, 0xe5, 0x3d, 0xe7, 0xb2, 
			0xd3, 0x33, 0x2c, 0x86, 0x44, 0x40, 0x2b, 0x82, 
			0x3e, 0x94, 0xa7, 0xdb, 0x26, 0x27, 0x6d, 0x2d, 
			0x23, 0xaa, 0x07, 0x18, 0x0f, 0x76, 0xb4, 0xfd, 
			0x29, 0xb9, 0xc0, 0x82, 0x30, 0x99, 0xc9, 0xd6, 
			0x2c, 0x51, 0x98, 0x80, 0xae, 0xe7, 0xe9, 0x69, 
			0x76, 0x17, 0xc1, 0x49, 0x7d, 0x47, 0xbf, 0x3e, 
			0x57, 0x19, 0x50, 0x31, 0x14, 0x21, 0xb6, 0xb7, 
			0x34, 0xd3, 0x8b, 0x0d, 0xb9, 0x1e, 0xb8, 0x53, 
			0x31, 0xb9, 0x1e, 0xa9, 0xf6, 0x15, 0x30, 0xf5, 
			0x45, 0x12, 0xa5, 0xa5, 0x2a, 0x4b, 0xad, 0x58, 
			0x9e, 0xb6, 0x97, 0x81, 0xd5, 0x37, 0xf2, 0x32, 
			0x97, 0xbb, 0x45, 0x9b, 0xda, 0xd2, 0x94, 0x8a, 
			0x29, 0xe1, 0x55, 0x0b, 0xf4, 0x78, 0x7e, 0x0b, 
			0xe9, 0x5b, 0xb1, 0x73, 0xcf, 0x5f, 0xab, 0x17, 
			0xda, 0xb7, 0xa1, 0x3a, 0x05, 0x2a, 0x63, 0x45, 
			0x3d, 0x97, 0xcc, 0xec, 0x1a, 0x32, 0x19, 0x54, 
			0x88, 0x6b, 0x7a, 0x12, 0x99, 0xfa, 0xae, 0xec, 
			0xae, 0x35, 0xc6, 0xea, 0xac, 0xa7, 0x53, 0xb0, 
			0x41, 0xb5, 0xe5, 0xf0, 0x93, 0xbf, 0x83, 0x39, 
			0x7f, 0xd2, 0x1d, 0xd6, 0xb3, 0x01, 0x20, 0x66, 
			0xfc, 0xc0, 0x58, 0xcc, 0x32, 0xc3, 0xb0, 0x9d, 
			0x75, 0x62, 0xde, 0xe2, 0x95, 0x09, 0xb5, 0x83, 
			0x93, 0x92, 0xc9, 0xff, 0x05, 0xf5, 0x1f, 0x31, 
			0x66, 0xaa, 0xac, 0x4a, 0xc5, 0xf2, 0x38, 0x03, 
			0x8a, 0x30, 0x45, 0xe6, 0xf7, 0x2e, 0x48, 0xef, 
			0x0f, 0xe8, 0xbc, 0x67, 0x5e, 0x82, 0xc3, 0x18, 
			0xa2, 0x68, 0xe4, 0x39, 0x70, 0x27, 0x1b, 0xf1, 
			0x19, 0xb8, 0x1b, 0xf6, 0xa9, 0x82, 0x74, 0x65, 
			0x54, 0xf8, 0x4e, 0x72, 0xb9, 0xf0, 0x02, 0x80, 
			0xa3, 0x20, 0xa0, 0x81, 0x42, 0x92, 0x3c, 0x23, 
			0xc8, 0x83, 0x42, 0x3f, 0xf9, 0x49, 0x82, 0x7f, 
			0x29, 0xbb, 0xac, 0xdc, 0x1c, 0xcd, 0xb0, 0x49, 
			0x38, 0xce, 0x60, 0x98, 0xc9, 0x5b, 0xa6, 0xb3, 
			0x25, 0x28, 0xf4, 0xef, 0x78, 0xee, 0xd7, 0x78, 
			0xb2, 0xe1, 0x22, 0xdd, 0xfd, 0x1c, 0xbd, 0xd1, 
			0x1d, 0x1c, 0x0a, 0x67, 0x83, 0xe0, 0x11, 0xfc, 
			0x53, 0x6d, 0x63, 0xd0, 0x53, 0x26, 0x06, 0x37, 
		},
	},
	{
		0x000000feULL,
		32,
		{
			0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45, 
			0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26, 
			0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93, 
			0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95, 
		},
		512,
		{
			0xd5, 0x5f, 0x68, 0x4f, 0x81, 0xf4, 0x42, 0x6e, 
			0x9f, 0xde, 0x92, 0xa5, 0xff, 0x02, 0xdf, 0x2a, 
			0xc8, 0x96, 0xaf, 0x63, 0x96, 0x28, 0x88, 0xa9, 
			0x79, 0x10, 0xc1, 0x37, 0x9e, 0x20, 0xb0, 0xa3, 
			0xb1, 0xdb, 0x61, 0x3f, 0xb7, 0xfe, 0x2e, 0x07, 
			0x00, 0x43, 0x29, 0xea, 0x5c, 0x22, 0xbf, 0xd3, 
			0x3e, 0x3d, 0xbe, 0x4c, 0xf5, 0x8c, 0xc6, 0x08, 
			0xc2, 0xc2, 0x6c, 0x19, 0xa2, 0xe2, 0xfe, 0x22, 
			0xf9, 0x87, 0x32, 0xc2, 0xb5, 0xcb, 0x84, 0x4c, 
			0xc6, 0xc0, 0x70, 0x2d, 0x91, 0xe1, 0xd5, 0x0f, 
			0xc4, 0x38, 0x2a, 0x7e, 0xba, 0x56, 0x35, 0xcd, 
			0x60, 0x24, 0x32, 0xa2, 0x30, 0x6a, 0xc4, 0xce, 
			0x82, 0xf8, 0xd7, 0x0c, 0x8d, 0x9b, 0xc1, 0x5f, 
			0x91, 0x8f, 0xe7, 0x1e, 0x74, 0xc6, 0x22, 0xd5, 
			0xcf, 0x71, 0x17, 0x8b, 0xf6, 0xe0, 0xb9, 0xcc, 
			0x9f, 0x2b, 0x41, 0xdd, 0x8d, 0xbe, 0x44, 0x1c, 
			0x41, 0xcd, 0x0c, 0x73, 0xa6, 0xdc, 0x47, 0xa3, 
			0x48, 0xf6, 0x70, 0x2f, 0x9d, 0x0e, 0x9b, 0x1b, 
			0x14, 0x31, 0xe9, 0x48, 0xe2, 0x99, 0xb9, 0xec, 
			0x22, 0x72, 0xab, 0x2c, 0x5f, 0x0c, 0x7b, 0xe8, 
			0x6a, 0xff, 0xa5, 0xde, 0xc8, 0x7a, 0x0b, 0xee, 
			0x81, 0xd3, 0xd5, 0x00, 0x07, 0xed, 0xaa, 0x2b, 
			0xcf, 0xcc, 0xb3, 0x56, 0x05, 0x15, 0x5f, 0xf3, 
			0x6e, 0xd8, 0xed, 0xd4, 0xa4, 0x0d, 0xcd, 0x4b, 
			0x24, 0x3a, 0xcd, 0x11, 0xb2, 0xb9, 0x87, 0xbd, 
			0xbf, 0xaf, 0x91, 0xa7, 0xca, 0xc2, 0x7e, 0x9c, 
			0x5a, 0xea, 0x52, 0x5e, 0xe5, 0x3d, 0xe7, 0xb2, 
			0xd3, 0x33, 0x2c, 0x86, 0x44, 0x40, 0x2b, 0x82, 
			0x3e, 0x94, 0xa7, 0xdb, 0x26, 0x27, 0x6d, 0x2d, 
			0x23, 0xaa, 0x07, 0x18, 0x0f, 0x76, 0xb4, 0xfd, 
			0x29, 0xb9, 0xc0, 0x82, 0x30, 0x99, 0xc9, 0xd6, 
			0x2c, 0x51, 0x98, 0x80, 0xae, 0xe7, 0xe9, 0x69, 
			0x76, 0x17, 0xc1, 0x49, 0x7d, 0x47, 0xbf, 0x3e, 
			0x57, 0x19, 0x50, 0x31, 0x14, 0x21, 0xb6, 0xb7, 
			0x34, 0xd3, 0x8b, 0x0d, 0xb9, 0x1e, 0xb8, 0x53, 
			0x31, 0xb9, 0x1e, 0xa9, 0xf6, 0x15, 0x30, 0xf5, 
			0x45, 0x12, 0xa5, 0xa5, 0x2a, 0x4b, 0xad, 0x58, 
			0x9e, 0xb6, 0x97, 0x81, 0xd5, 0x37, 0xf2, 0x32, 
			0x97, 0xbb, 0x45, 0x9b, 0xda, 0xd2, 0x94, 0x8a, 
			0x29, 0xe1, 0x55, 0x0b, 0xf4, 0x78, 0x7e, 0x0b, 
			0xe9, 0x5b, 0xb1, 0x73, 0xcf, 0x5f, 0xab, 0x17, 
			0xda, 0xb7, 0xa1, 0x3a, 0x05, 0x2a, 0x63, 0x45, 
			0x3d, 0x97, 0xcc, 0xec, 0x1a, 0x32, 0x19, 0x54, 
			0x88, 0x6b, 0x7a, 0x12, 0x99, 0xfa, 0xae, 0xec, 
			0xae, 0x35, 0xc6, 0xea, 0xac, 0xa7, 0x53, 0xb0, 
			0x41, 0xb5, 0xe5, 0xf0, 0x93, 0xbf, 0x83, 0x39, 
			0x7f, 0xd2, 0x1d, 0xd6, 0xb3, 0x01, 0x20, 0x66, 
			0xfc, 0xc0, 0x58, 0xcc, 0x32, 0xc3, 0xb0, 0x9d, 
			0x75, 0x62, 0xde, 0xe2, 0x95, 0x09, 0xb5, 0x83, 
			0x93, 0x92, 0xc9, 0xff, 0x05, 0xf5, 0x1f, 0x31, 
			0x66, 0xaa, 0xac, 0x4a, 0xc5, 0xf2, 0x38, 0x03, 
			0x8a, 0x30, 0x45, 0xe6, 0xf7, 0x2e, 0x48, 0xef, 
			0x0f, 0xe8, 0xbc, 0x67, 0x5e, 0x82, 0xc3, 0x18, 
			0xa2, 0x68, 0xe4, 0x39, 0x70, 0x27, 0x1b, 0xf1, 
			0x19, 0xb8, 0x1b, 0xf6, 0xa9, 0x82, 0x74, 0x65, 
			0x54, 0xf8, 0x4e, 0x72, 0xb9, 0xf0, 0x02, 0x80, 
			0xa3, 0x20, 0xa0, 0x81, 0x42, 0x92, 0x3c, 0x23, 
			0xc8, 0x83, 0x42, 0x3f, 0xf9, 0x49, 0x82, 0x7f, 
			0x29, 0xbb, 0xac, 0xdc, 0x1c, 0xcd, 0xb0, 0x49, 
			0x38, 0xce, 0x60, 0x98, 0xc9, 0x5b, 0xa6, 0xb3, 
			0x25, 0x28, 0xf4, 0xef, 0x78, 0xee, 0xd7, 0x78, 
			0xb2, 0xe1, 0x22, 0xdd, 0xfd, 0x1c, 0xbd, 0xd1, 
			0x1d, 0x1c, 0x0a, 0x67, 0x83, 0xe0, 0x11, 0xfc, 
			0x53, 0x6d, 0x63, 0xd0, 0x53, 0x26, 0x06, 0x37, 
		}, {
			0x72, 0xef, 0xc1, 0xeb, 0xfe, 0x1e, 0xe2, 0x59, 
			0x75, 0xa6, 0xeb, 0x3a, 0xa8, 0x58, 0x9d, 0xda, 
			0x2b, 0x26, 0x1f, 0x1c, 0x85, 0xbd, 0xab, 0x44, 
			0x2a, 0x9e, 0x5b, 0x2d, 0xd1, 0xd7, 0xc3, 0x95, 
			0x7a, 0x16, 0xfc, 0x08, 0xe5, 0x26, 0xd4, 0xb1, 
			0x22, 0x3f, 0x1b, 0x12, 0x32, 0xa1, 0x1a, 0xf2, 
			0x74, 0xc3, 0xd7, 0x0d, 0xac, 0x57, 0xf8, 0x3e, 
			0x09, 0x83, 0xc4, 0x98, 0xf1, 0xa6, 0xf1, 0xae, 
			0xcb, 0x02, 0x1c, 0x3e, 0x70, 0x08, 0x5a, 0x1e, 
			0x52, 0x7f, 0x1c, 0xe4, 0x1e, 0xe5, 0x91, 0x1a, 
			0x82, 0x02, 0x01, 0x61, 0x52, 0x9c, 0xd8, 0x27, 
			0x73, 0x76, 0x2d, 0xaf, 0x54, 0x59, 0xde, 0x94, 
			0xa0, 0xa8, 0x2a, 0xda, 0xe7, 0xe1, 0x70, 0x3c, 
			0x80, 0x85, 0x43, 0xc2, 0x9e, 0xd6, 0xfb, 0x32, 
			0xd9, 0xe0, 0x04, 0x32, 0x7c, 0x13, 0x55, 0x18, 
			0x0c, 0x99, 0x5a, 0x07, 0x74, 0x14, 0x93, 0xa0, 
			0x9c, 0x21, 0xba, 0x01, 0xa3, 0x87, 0x88, 0x2d, 
			0xa4, 0xf6, 0x25, 0x34, 0xb8, 0x7b, 0xb1, 0x5d, 
			0x60, 0xd1, 0x97, 0x20, 0x1c, 0x0f, 0xd3, 0xbf, 
			0x30, 0xc1, 0x50, 0x0a, 0x3e, 0xcf, 0xec, 0xdd, 
			0x66, 0xd8, 0x72, 0x1f, 0x90, 0xbc, 0xc4, 0xc1, 
			0x7e, 0xe9, 0x25, 0xc6, 0x1b, 0x0a, 0x03, 0x72, 
			0x7a, 0x9c, 0x0d, 0x5f, 0x5c, 0xa4, 0x62, 0xfb, 
			0xfa, 0x0a, 0xf1, 0xc2, 0x51, 0x3a, 0x9d, 0x9d, 
			0x4b, 0x53, 0x45, 0xbd, 0x27, 0xa5, 0xf6, 0xe6, 
			0x53, 0xf7, 0x51, 0x69, 0x3e, 0x6b, 0x6a, 0x2b, 
			0x8e, 0xad, 0x57, 0xd5, 0x11, 0xe0, 0x0e, 0x58, 
			0xc4, 0x5b, 0x7b, 0x8d, 0x00, 0x5a, 0xf7, 0x92, 
			0x88, 0xf5, 0xc7, 0xc2, 0x2f, 0xd4, 0xf1, 0xbf, 
			0x7a, 0x89, 0x8b, 0x03, 0xa5, 0x63, 0x4c, 0x6a, 
			0x1a, 0xe3, 0xf9, 0xfa, 0xe5, 0xde, 0x4f, 0x29, 
			0x6a, 0x28, 0x96, 0xb2, 0x3e, 0x7e, 0xd4, 0x3e, 
			0xd1, 0x4f, 0xa5, 0xa2, 0x80, 0x3f, 0x4d, 0x28, 
			0xf0, 0xd3, 0xff, 0xcf, 0x24, 0x75, 0x76, 0x77, 
			0xae, 0xbd, 0xb4, 0x7b, 0xb3, 0x88, 0x37, 0x87, 
			0x08, 0x94, 0x8a, 0x8d, 0x41, 0x26, 0xed, 0x18, 
			0x39, 0xe0, 0xda, 0x29, 0xa5, 0x37, 0xa8, 0xc1, 
			0x98, 0xb3, 0xc6, 0x6a, 0xb0, 0x07, 0x12, 0xdd, 
			0x26, 0x16, 0x74, 0xbf, 0x45, 0xa7, 0x3d, 0x67, 
			0xf7, 0x69, 0x14, 0xf8, 0x30, 0xca, 0x01, 0x4b, 
			0x65, 0x59, 0x6f, 0x27, 0xe4, 0xcf, 0x62, 0xde, 
			0x66, 0x12, 0x5a, 0x55, 0x66, 0xdf, 0x99, 0x75, 
			0x15, 0x56, 0x28, 0xb4, 0x00, 0xfb, 0xfb, 0x3a, 
			0x29, 0x04, 0x0e, 0xd5, 0x0f, 0xaf, 0xfd, 0xbb, 
			0x18, 0xae, 0xce, 0x7c, 0x5c, 0x44, 0x69, 0x32, 
			0x60, 0xaa, 0xb3, 0x86, 0xc0, 0xa3, 0x7b, 0x11, 
			0xb1, 0x14, 0xf1, 0xc4, 0x15, 0xae, 0xbb, 0x65, 
			0x3b, 0xe4, 0x68, 0x17, 0x94, 0x28, 0xd4, 0x3a, 
			0x4d, 0x8b, 0xc3, 0xec, 0x38, 0x81, 0x3e, 0xca, 
			0x30, 0xa1, 0x3c, 0xf1, 0xbb, 0x18, 0xd5, 0x24, 
			0xf1, 0x99, 0x2d, 0x44, 0xd8, 0xb1, 0xa4, 0x2e, 
			0xa3, 0x0b, 0x22, 0xe6, 0xc9, 0x5b, 0x19, 0x9d, 
			0x8d, 0x18, 0x2f, 0x88, 0x40, 0xb0, 0x9d, 0x05, 
			0x95, 0x85, 0xc3, 0x1a, 0xd6, 0x91, 0xfa, 0x06, 
			0x19, 0xff, 0x03, 0x8a, 0xca, 0x2c, 0x39, 0xa9, 
			0x43, 0x42, 0x11, 0x57, 0x36, 0x17, 0x17, 0xc4, 
			0x9d, 0x32, 0x20, 0x28, 0xa7, 0x46, 0x48, 0x11, 
			0x3b, 0xd8, 0xc9, 0xd7, 0xec, 0x77, 0xcf, 0x3c, 
			0x89, 0xc1, 0xec, 0x87, 0x18, 0xce, 0xff, 0x85, 
			0x16, 0xd9, 0x6b, 0x34, 0xc3, 0xc6, 0x14, 0xf1, 
			0x06, 0x99, 0xc9, 0xab, 0xc4, 0xed, 0x04, 0x11, 
			0x50, 0x62, 0x23, 0xbe, 0xa1, 0x6a, 0xf3, 0x5c, 
			0x88, 0x3a, 0xcc, 0xdb, 0xe1, 0x10, 0x4e, 0xef, 
			0x0c, 0xfd, 0xb5, 0x4e, 0x12, 0xfb, 0x23, 0x0a, 
		},
	},
	{
		0x000000ffULL,
		32,
		{
			0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45, 
			0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26, 
			0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93, 
			0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95, 
		},
		512,
		{
			0x72, 0xef, 0xc1, 0xeb, 0xfe, 0x1e, 0xe2, 0x59, 
			0x75, 0xa6, 0xeb, 0x3a, 0xa8, 0x58, 0x9d, 0xda, 
			0x2b, 0x26, 0x1f, 0x1c, 0x85, 0xbd, 0xab, 0x44, 
			0x2a, 0x9e, 0x5b, 0x2d, 0xd1, 0xd7, 0xc3, 0x95, 
			0x7a, 0x16, 0xfc, 0x08, 0xe5, 0x26, 0xd4, 0xb1, 
			0x22, 0x3f, 0x1b, 0x12, 0x32, 0xa1, 0x1a, 0xf2, 
			0x74, 0xc3, 0xd7, 0x0d, 0xac, 0x57, 0xf8, 0x3e, 
			0x09, 0x83, 0xc4, 0x98, 0xf1, 0xa6, 0xf1, 0xae, 
			0xcb, 0x02, 0x1c, 0x3e, 0x70, 0x08, 0x5a, 0x1e, 
			0x52, 0x7f, 0x1c, 0xe4, 0x1e, 0xe5, 0x91, 0x1a, 
			0x82, 0x02, 0x01, 0x61, 0x52, 0x9c, 0xd8, 0x27, 
			0x73, 0x76, 0x2d, 0xaf, 0x54, 0x59, 0xde, 0x94, 
			0xa0, 0xa8, 0x2a, 0xda, 0xe7, 0xe1, 0x70, 0x3c, 
			0x80, 0x85, 0x43, 0xc2, 0x9e, 0xd6, 0xfb, 0x32, 
			0xd9, 0xe0, 0x04, 0x32, 0x7c, 0x13, 0x55, 0x18, 
			0x0c, 0x99, 0x5a, 0x07, 0x74, 0x14, 0x93, 0xa0, 
			0x9c, 0x21, 0xba, 0x01, 0xa3, 0x87, 0x88, 0x2d, 
			0xa4, 0xf6, 0x25, 0x34, 0xb8, 0x7b, 0xb1, 0x5d, 
			0x60, 0xd1, 0x97, 0x20, 0x1c, 0x0f, 0xd3, 0xbf, 
			0x30, 0xc1, 0x50, 0x0a, 0x3e, 0xcf, 0xec, 0xdd, 
			0x66, 0xd8, 0x72, 0x1f, 0x90, 0xbc, 0xc4, 0xc1, 
			0x7e, 0xe9, 0x25, 0xc6, 0x1b, 0x0a, 0x03, 0x72, 
			0x7a, 0x9c, 0x0d, 0x5f, 0x5c, 0xa4, 0x62, 0xfb, 
			0xfa, 0x0a, 0xf1, 0xc2, 0x51, 0x3a, 0x9d, 0x9d, 
			0x4b, 0x53, 0x45, 0xbd, 0x27, 0xa5, 0xf6, 0xe6, 
			0x53, 0xf7, 0x51, 0x69, 0x3e, 0x6b, 0x6a, 0x2b, 
			0x8e, 0xad, 0x57, 0xd5, 0x11, 0xe0, 0x0e, 0x58, 
			0xc4, 0x5b, 0x7b, 0x8d, 0x00, 0x5a, 0xf7, 0x92, 
			0x88, 0xf5, 0xc7, 0xc2, 0x2f, 0xd4, 0xf1, 0xbf, 
			0x7a, 0x89, 0x8b, 0x03, 0xa5, 0x63, 0x4c, 0x6a, 
			0x1a, 0xe3, 0xf9, 0xfa, 0xe5, 0xde, 0x4f, 0x29, 
			0x6a, 0x28, 0x96, 0xb2, 0x3e, 0x7e, 0xd4, 0x3e, 
			0xd1, 0x4f, 0xa5, 0xa2, 0x80, 0x3f, 0x4d, 0x28, 
			0xf0, 0xd3, 0xff, 0xcf, 0x24, 0x75, 0x76, 0x77, 
			0xae, 0xbd, 0xb4, 0x7b, 0xb3, 0x88, 0x37, 0x87, 
			0x08, 0x94, 0x8a, 0x8d, 0x41, 0x26, 0xed, 0x18, 
			0x39, 0xe0, 0xda, 0x29, 0xa5, 0x37, 0xa8, 0xc1, 
			0x98, 0xb3, 0xc6, 0x6a, 0xb0, 0x07, 0x12, 0xdd, 
			0x26, 0x16, 0x74, 0xbf, 0x45, 0xa7, 0x3d, 0x67, 
			0xf7, 0x69, 0x14, 0xf8, 0x30, 0xca, 0x01, 0x4b, 
			0x65, 0x59, 0x6f, 0x27, 0xe4, 0xcf, 0x62, 0xde, 
			0x66, 0x12, 0x5a, 0x55, 0x66, 0xdf, 0x99, 0x75, 
			0x15, 0x56, 0x28, 0xb4, 0x00, 0xfb, 0xfb, 0x3a, 
			0x29, 0x04, 0x0e, 0xd5, 0x0f, 0xaf, 0xfd, 0xbb, 
			0x18, 0xae, 0xce, 0x7c, 0x5c, 0x44, 0x69, 0x32, 
			0x60, 0xaa, 0xb3, 0x86, 0xc0, 0xa3, 0x7b, 0x11, 
			0xb1, 0x14, 0xf1, 0xc4, 0x15, 0xae, 0xbb, 0x65, 
			0x3b, 0xe4, 0x68, 0x17, 0x94, 0x28, 0xd4, 0x3a, 
			0x4d, 0x8b, 0xc3, 0xec, 0x38, 0x81, 0x3e, 0xca, 
			0x30, 0xa1, 0x3c, 0xf1, 0xbb, 0x18, 0xd5, 0x24, 
			0xf1, 0x99, 0x2d, 0x44, 0xd8, 0xb1, 0xa4, 0x2e, 
			0xa3, 0x0b, 0x22, 0xe6, 0xc9, 0x5b, 0x19, 0x9d, 
			0x8d, 0x18, 0x2f, 0x88, 0x40, 0xb0, 0x9d, 0x05, 
			0x95, 0x85, 0xc3, 0x1a, 0xd6, 0x91, 0xfa, 0x06, 
			0x19, 0xff, 0x03, 0x8a, 0xca, 0x2c, 0x39, 0xa9, 
			0x43, 0x42, 0x11, 0x57, 0x36, 0x17, 0x17, 0xc4, 
			0x9d, 0x32, 0x20, 0x28, 0xa7, 0x46, 0x48, 0x11, 
			0x3b, 0xd8, 0xc9, 0xd7, 0xec, 0x77, 0xcf, 0x3c, 
			0x89, 0xc1, 0xec, 0x87, 0x18, 0xce, 0xff, 0x85, 
			0x16, 0xd9, 0x6b, 0x34, 0xc3, 0xc6, 0x14, 0xf1, 
			0x06, 0x99, 0xc9, 0xab, 0xc4, 0xed, 0x04, 0x11, 
			0x50, 0x62, 0x23, 0xbe, 0xa1, 0x6a, 0xf3, 0x5c, 
			0x88, 0x3a, 0xcc, 0xdb, 0xe1, 0x10, 0x4e, 0xef, 
			0x0c, 0xfd, 0xb5, 0x4e, 0x12, 0xfb, 0x23, 0x0a, 
		}, {
			0x32, 0x60, 0xae, 0x8d, 0xad, 0x1f, 0x4a, 0x32, 
			0xc5, 0xca, 0xfe, 0x3a, 0xb0, 0xeb, 0x95, 0x54, 
			0x9d, 0x46, 0x1a, 0x67, 0xce, 0xb9, 0xe5, 0xaa, 
			0x2d, 0x3a, 0xfb, 0x62, 0xde, 0xce, 0x05, 0x53, 
			0x19, 0x3b, 0xa5, 0x0c, 0x75, 0xbe, 0x25, 0x1e, 
			0x08, 0xd1, 0xd0, 0x8f, 0x10, 0x88, 0x57, 0x6c, 
			0x7e, 0xfd, 0xfa, 0xaf, 0x3f, 0x45, 0x95, 0x59, 
			0x57, 0x1e, 0x12, 0x51, 0x17, 0x53, 0xb0, 0x7a, 
			0xf0, 0x73, 0xf3, 0x5d, 0xa0, 0x6a, 0xf0, 0xce, 
			0x0b, 0xbf, 0x6b, 0x8f, 0x5c, 0xcc, 0x5c, 0xea, 
			0x50, 0x0e, 0xc1, 0xb2, 0x11, 0xbd, 0x51, 0xf6, 
			0x3b, 0x60, 0x6b, 0xf6, 0x52, 0x87, 0x96, 0xca, 
			0x12, 0x17, 0x3b, 0xa3, 0x9b, 0x89, 0x35, 0xee, 
			0x44, 0xcc, 0xce, 0x64, 0x6f, 0x90, 0xa4, 0x5b, 
			0xf9, 0xcc, 0xc5, 0x67, 0xf0, 0xac, 0xe1, 0x3d, 
			0xc2, 0xd5, 0x3e, 0xbe, 0xed, 0xc8, 0x1f, 0x58, 
			0xb2, 0xe4, 0x11, 0x79, 0xdd, 0xdf, 0x0d, 0x5a, 
			0x5c, 0x42, 0xf5, 0xd8, 0x50, 0x6c, 0x1a, 0x5d, 
			0x2f, 0x8f, 0x59, 0xf3, 0xea, 0x87, 0x3c, 0xbc, 
			0xd0, 0xee, 0xc1, 0x9a, 0xcb, 0xf3, 0x25, 0x42, 
			0x3b, 0xd3, 0xdc, 0xb8, 0xc2, 0xb1, 0xbf, 0x1d, 
			0x1e, 0xae, 0xd0, 0xeb, 0xa7, 0xf0, 0x69, 0x8e, 
			0x43, 0x14, 0xfb, 0xeb, 0x2f, 0x15, 0x66, 0xd1, 
			0xb9, 0x25, 0x30, 0x08, 0xcb, 0xcc, 0xf4, 0x5a, 
			0x2b, 0x0d, 0x9c, 0x5c, 0x9c, 0x21, 0x47, 0x4f, 
			0x40, 0x76, 0xe0, 0x2b, 0xe2, 0x60, 0x50, 0xb9, 
			0x9d, 0xee, 0x4f, 0xd6, 0x8a, 0x4c, 0xf8, 0x90, 
			0xe4, 0x96, 0xe4, 0xfc, 0xae, 0x7b, 0x70, 0xf9, 
			0x4e, 0xa5, 0xa9, 0x06, 0x2d, 0xa0, 0xda, 0xeb, 
			0xa1, 0x99, 0x3d, 0x2c, 0xcd, 0x1d, 0xd3, 0xc2, 
			0x44, 0xb8, 0x42, 0x88, 0x01, 0x49, 0x5a, 0x58, 
			0xb2, 0x16, 0x54, 0x7e, 0x7e, 0x84, 0x7c, 0x46, 
			0xd1, 0xd7, 0x56, 0x37, 0x7b, 0x62, 0x42, 0xd2, 
			0xe5, 0xfb, 0x83, 0xbf, 0x75, 0x2b, 0x54, 0xe0, 
			0xdf, 0x71, 0xe8, 0x89, 0xf3, 0xa2, 0xbb, 0x0f, 
			0x4c, 0x10, 0x80, 0x5b, 0xf3, 0xc5, 0x90, 0x37, 
			0x6e, 0x3c, 0x24, 0xe2, 0x2f, 0xf5, 0x7f, 0x7f, 
			0xa9, 0x65, 0x57, 0x73, 0x75, 0x32, 0x5c, 0xea, 
			0x5d, 0x92, 0x0d, 0xb9, 0x4b, 0x9c, 0x33, 0x6b, 
			0x45, 0x5f, 0x6e, 0x89, 0x4c, 0x01, 0x86, 0x6f, 
			0xe9, 0xfb, 0xb8, 0xc8, 0xd3, 0xf7, 0x0a, 0x29, 
			0x57, 0x28, 0x5f, 0x6d, 0xfb, 0x5d, 0xcd, 0x8c, 
			0xbf, 0x54, 0x78, 0x2f, 0x8f, 0xe7, 0x76, 0x6d, 
			0x47, 0x23, 0x81, 0x99, 0x13, 0xac, 0x77, 0x34, 
			0x21, 0xe3, 0xa3, 0x10, 0x95, 0x86, 0x6b, 0xad, 
			0x22, 0xc8, 0x6a, 0x60, 0x36, 0xb2, 0x51, 0x8b, 
			0x20, 0x59, 0xb4, 0x22, 0x9d, 0x18, 0xc8, 0xc2, 
			0xcc, 0xbd, 0xf9, 0x06, 0xc6, 0xcc, 0x6e, 0x82, 
			0x46, 0x4e, 0xe5, 0x7b, 0xdd, 0xb0, 0xbe, 0xbc, 
			0xb1, 0xdc, 0x64, 0x53, 0x25, 0xbf, 0xb3, 0xe6, 
			0x65, 0xef, 0x72, 0x51, 0x08, 0x2c, 0x88, 0xeb, 
			0xb1, 0xcf, 0x20, 0x3b, 0xd7, 0x79, 0xfd, 0xd3, 
			0x86, 0x75, 0x71, 0x3c, 0x8d, 0xaa, 0xdd, 0x17, 
			0xe1, 0xca, 0xbe, 0xe4, 0x32, 0xb0, 0x97, 0x87, 
			0xb6, 0xdd, 0xf3, 0x30, 0x4e, 0x38, 0xb7, 0x31, 
			0xb4, 0x5d, 0xf5, 0xdf, 0x51, 0xb7, 0x8f, 0xcf, 
			0xb3, 0xd3, 0x24, 0x66, 0x02, 0x8d, 0x0b, 0xa3, 
			0x65, 0x55, 0xe7, 0xe1, 0x1a, 0xb0, 0xee, 0x06, 
			0x66, 0x06, 0x1d, 0x16, 0x45, 0xd9, 0x62, 0x44, 
			0x4b, 0xc4, 0x7a, 0x38, 0x18, 0x89, 0x30, 0xa8, 
			0x4b, 0x4d, 0x56, 0x13, 0x95, 0xc7, 0x3c, 0x08, 
			0x70, 0x21, 0x92, 0x7c, 0xa6, 0x38, 0xb7, 0xaf, 
			0xc8, 0xa8, 0x67, 0x9c, 0xcb, 0x84, 0xc2, 0x65, 
			0x55, 0x44, 0x0e, 0xc7, 0xf1, 0x04, 0x45, 0xcd, 
		},
	},
	{
		0x000000ffULL,
		64,
		{
			0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45, 
			0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26, 
			0x62, 0x49, 0x77, 0x57, 0x24, 0x70, 0x93, 0x69, 
			0x99, 0x59, 0x57, 0x49, 0x66, 0x96, 0x76, 0x27, 
			0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93, 
			0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95, 
			0x02, 0x88, 0x41, 0x97, 0x16, 0x93, 0x99, 0x37, 
			0x51, 0x05, 0x82, 0x09, 0x74, 0x94, 0x45, 0x92, 
		},
		512,
		{
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 
			0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 
			0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 
			0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 
			0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 
			0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 
			0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 
			0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 
			0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 
			0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 
			0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 
			0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 
			0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 
			0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 
			0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 
			0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
			0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 
			0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
			0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 
			0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
			0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 
			0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
			0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 
			0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 
			0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
			0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff, 
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 
			0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 
			0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 
			0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 
			0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 
			0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 
			0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 
			0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 
			0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 
			0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 
			0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 
			0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 
			0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 
			0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 
			0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 
			0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
			0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 
			0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
			0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 
			0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
			0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 
			0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
			0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 
			0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 
			0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
			0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff, 
		}, {
			0x1c, 0x3b, 0x3a, 0x10, 0x2f, 0x77, 0x03, 0x86, 
			0xe4, 0x83, 0x6c, 0x99, 0xe3, 0x70, 0xcf, 0x9b, 
			0xea, 0x00, 0x80, 0x3f, 0x5e, 0x48, 0x23, 0x57, 
			0xa4, 0xae, 0x12, 0xd4, 0x14, 0xa3, 0xe6, 0x3b, 
			0x5d, 0x31, 0xe2, 0x76, 0xf8, 0xfe, 0x4a, 0x8d, 
			0x66, 0xb3, 0x17, 0xf9, 0xac, 0x68, 0x3f, 0x44, 
			0x68, 0x0a, 0x86, 0xac, 0x35, 0xad, 0xfc, 0x33, 
			0x45, 0xbe, 0xfe, 0xcb, 0x4b, 0xb1, 0x88, 0xfd, 
			0x57, 0x76, 0x92, 0x6c, 0x49, 0xa3, 0x09, 0x5e, 
			0xb1, 0x08, 0xfd, 0x10, 0x98, 0xba, 0xec, 0x70, 
			0xaa, 0xa6, 0x69, 0x99, 0xa7, 0x2a, 0x82, 0xf2, 
			0x7d, 0x84, 0x8b, 0x21, 0xd4, 0xa7, 0x41, 0xb0, 
			0xc5, 0xcd, 0x4d, 0x5f, 0xff, 0x9d, 0xac, 0x89, 
			0xae, 0xba, 0x12, 0x29, 0x61, 0xd0, 0x3a, 0x75, 
			0x71, 0x23, 0xe9, 0x87, 0x0f, 0x8a, 0xcf, 0x10, 
			0x00, 0x02, 0x08, 0x87, 0x89, 0x14, 0x29, 0xca, 
			0x2a, 0x3e, 0x7a, 0x7d, 0x7d, 0xf7, 0xb1, 0x03, 
			0x55, 0x16, 0x5c, 0x8b, 0x9a, 0x6d, 0x0a, 0x7d, 
			0xe8, 0xb0, 0x62, 0xc4, 0x50, 0x0d, 0xc4, 0xcd, 
			0x12, 0x0c, 0x0f, 0x74, 0x18, 0xda, 0xe3, 0xd0, 
			0xb5, 0x78, 0x1c, 0x34, 0x80, 0x3f, 0xa7, 0x54, 
			0x21, 0xc7, 0x90, 0xdf, 0xe1, 0xde, 0x18, 0x34, 
			0xf2, 0x80, 0xd7, 0x66, 0x7b, 0x32, 0x7f, 0x6c, 
			0x8c, 0xd7, 0x55, 0x7e, 0x12, 0xac, 0x3a, 0x0f, 
			0x93, 0xec, 0x05, 0xc5, 0x2e, 0x04, 0x93, 0xef, 
			0x31, 0xa1, 0x2d, 0x3d, 0x92, 0x60, 0xf7, 0x9a, 
			0x28, 0x9d, 0x6a, 0x37, 0x9b, 0xc7, 0x0c, 0x50, 
			0x84, 0x14, 0x73, 0xd1, 0xa8, 0xcc, 0x81, 0xec, 
			0x58, 0x3e, 0x96, 0x45, 0xe0, 0x7b, 0x8d, 0x96, 
			0x70, 0x65, 0x5b, 0xa5, 0xbb, 0xcf, 0xec, 0xc6, 
			0xdc, 0x39, 0x66, 0x38, 0x0a, 0xd8, 0xfe, 0xcb, 
			0x17, 0xb6, 0xba, 0x02, 0x46, 0x9a, 0x02, 0x0a, 
			0x84, 0xe1, 0x8e, 0x8f, 0x84, 0x25, 0x20, 0x70, 
			0xc1, 0x3e, 0x9f, 0x1f, 0x28, 0x9b, 0xe5, 0x4f, 
			0xbc, 0x48, 0x14, 0x57, 0x77, 0x8f, 0x61, 0x60, 
			0x15, 0xe1, 0x32, 0x7a, 0x02, 0xb1, 0x40, 0xf1, 
			0x50, 0x5e, 0xb3, 0x09, 0x32, 0x6d, 0x68, 0x37, 
			0x8f, 0x83, 0x74, 0x59, 0x5c, 0x84, 0x9d, 0x84, 
			0xf4, 0xc3, 0x33, 0xec, 0x44, 0x23, 0x88, 0x51, 
			0x43, 0xcb, 0x47, 0xbd, 0x71, 0xc5, 0xed, 0xae, 
			0x9b, 0xe6, 0x9a, 0x2f, 0xfe, 0xce, 0xb1, 0xbe, 
			0xc9, 0xde, 0x24, 0x4f, 0xbe, 0x15, 0x99, 0x2b, 
			0x11, 0xb7, 0x7c, 0x04, 0x0f, 0x12, 0xbd, 0x8f, 
			0x6a, 0x97, 0x5a, 0x44, 0xa0, 0xf9, 0x0c, 0x29, 
			0xa9, 0xab, 0xc3, 0xd4, 0xd8, 0x93, 0x92, 0x72, 
			0x84, 0xc5, 0x87, 0x54, 0xcc, 0xe2, 0x94, 0x52, 
			0x9f, 0x86, 0x14, 0xdc, 0xd2, 0xab, 0xa9, 0x91, 
			0x92, 0x5f, 0xed, 0xc4, 0xae, 0x74, 0xff, 0xac, 
			0x6e, 0x33, 0x3b, 0x93, 0xeb, 0x4a, 0xff, 0x04, 
			0x79, 0xda, 0x9a, 0x41, 0x0e, 0x44, 0x50, 0xe0, 
			0xdd, 0x7a, 0xe4, 0xc6, 0xe2, 0x91, 0x09, 0x00, 
			0x57, 0x5d, 0xa4, 0x01, 0xfc, 0x07, 0x05, 0x9f, 
			0x64, 0x5e, 0x8b, 0x7e, 0x9b, 0xfd, 0xef, 0x33, 
			0x94, 0x30, 0x54, 0xff, 0x84, 0x01, 0x14, 0x93, 
			0xc2, 0x7b, 0x34, 0x29, 0xea, 0xed, 0xb4, 0xed, 
			0x53, 0x76, 0x44, 0x1a, 0x77, 0xed, 0x43, 0x85, 
			0x1a, 0xd7, 0x7f, 0x16, 0xf5, 0x41, 0xdf, 0xd2, 
			0x69, 0xd5, 0x0d, 0x6a, 0x5f, 0x14, 0xfb, 0x0a, 
			0xab, 0x1c, 0xbb, 0x4c, 0x15, 0x50, 0xbe, 0x97, 
			0xf7, 0xab, 0x40, 0x66, 0x19, 0x3c, 0x4c, 0xaa, 
			0x77, 0x3d, 0xad, 0x38, 0x01, 0x4b, 0xd2, 0x09, 
			0x2f, 0xa7, 0x55, 0xc8, 0x24, 0xbb, 0x5e, 0x54, 
			0xc4, 0xf3, 0x6f, 0xfd, 0xa9, 0xfc, 0xea, 0x70, 
			0xb9, 0xc6, 0xe6, 0x93, 0xe1, 0x48, 0xc1, 0x51, 
		},
	},
	{
		0x0000ffffULL,
		64,
		{
			0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45, 
			0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26, 
			0x62, 0x49, 0x77, 0x57, 0x24, 0x70, 0x93, 0x69, 
			0x99, 0x59, 0x57, 0x49, 0x66, 0x96, 0x76, 0x27, 
			0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93, 
			0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95, 
			0x02, 0x88, 0x41, 0x97, 0x16, 0x93, 0x99, 0x37, 
			0x51, 0x05, 0x82, 0x09, 0x74, 0x94, 0x45, 0x92, 
		},
		512,
		{
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 
			0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 
			0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 
			0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 
			0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 
			0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 
			0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 
			0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 
			0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 
			0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 
			0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 
			0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 
			0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 
			0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 
			0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 
			0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
			0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 
			0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
			0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 
			0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
			0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 
			0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
			0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 
			0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 
			0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
			0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff, 
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 
			0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 
			0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 
			0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 
			0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 
			0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 
			0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 
			0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 
			0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 
			0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 
			0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 
			0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 
			0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 
			0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 
			0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 
			0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
			0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 
			0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
			0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 
			0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
			0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 
			0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
			0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 
			0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 
			0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
			0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff, 
		}, {
			0x77, 0xa3, 0x12, 0x51, 0x61, 0x8a, 0x15, 0xe6, 
			0xb9, 0x2d, 0x1d, 0x66, 0xdf, 0xfe, 0x7b, 0x50, 
			0xb5, 0x0b, 0xad, 0x55, 0x23, 0x05, 0xba, 0x02, 
			0x17, 0xa6, 0x10, 0x68, 0x8e, 0xff, 0x7e, 0x11, 
			0xe1, 0xd0, 0x22, 0x54, 0x38, 0xe0, 0x93, 0x24, 
			0x2d, 0x6d, 0xb2, 0x74, 0xfd, 0xe8, 0x01, 0xd4, 
			0xca, 0xe0, 0x6f, 0x20, 0x92, 0xc7, 0x28, 0xb2, 
			0x47, 0x85, 0x59, 0xdf, 0x58, 0xe8, 0x37, 0xc2, 
			0x46, 0x9e, 0xe4, 0xa4, 0xfa, 0x79, 0x4e, 0x4b, 
			0xbc, 0x7f, 0x39, 0xbc, 0x02, 0x6e, 0x3c, 0xb7, 
			0x2c, 0x33, 0xb0, 0x88, 0x8f, 0x25, 0xb4, 0xac, 
			0xf5, 0x6a, 0x2a, 0x98, 0x04, 0xf1, 0xce, 0x6d, 
			0x3d, 0x6e, 0x1d, 0xc6, 0xca, 0x18, 0x1d, 0x4b, 
			0x54, 0x61, 0x79, 0xd5, 0x55, 0x44, 0xaa, 0x77, 
			0x60, 0xc4, 0x0d, 0x06, 0x74, 0x15, 0x39, 0xc7, 
			0xe3, 0xcd, 0x9d, 0x2f, 0x66, 0x50, 0xb2, 0x01, 
			0x3f, 0xd0, 0xee, 0xb8, 0xc2, 0xb8, 0xe3, 0xd8, 
			0xd2, 0x40, 0xcc, 0xae, 0x2d, 0x4c, 0x98, 0x32, 
			0x0a, 0x74, 0x42, 0xe1, 0xc8, 0xd7, 0x5a, 0x42, 
			0xd6, 0xe6, 0xcf, 0xa4, 0xc2, 0xec, 0xa1, 0x79, 
			0x8d, 0x15, 0x8c, 0x7a, 0xec, 0xdf, 0x82, 0x49, 
			0x0f, 0x24, 0xbb, 0x9b, 0x38, 0xe1, 0x08, 0xbc, 
			0xda, 0x12, 0xc3, 0xfa, 0xf9, 0xa2, 0x11, 0x41, 
			0xc3, 0x61, 0x3b, 0x58, 0x36, 0x7f, 0x92, 0x2a, 
			0xaa, 0x26, 0xcd, 0x22, 0xf2, 0x3d, 0x70, 0x8d, 
			0xae, 0x69, 0x9a, 0xd7, 0xcb, 0x40, 0xa8, 0xad, 
			0x0b, 0x6e, 0x27, 0x84, 0x97, 0x3d, 0xcb, 0x60, 
			0x56, 0x84, 0xc0, 0x8b, 0x8d, 0x69, 0x98, 0xc6, 
			0x9a, 0xac, 0x04, 0x99, 0x21, 0x87, 0x1e, 0xbb, 
			0x65, 0x30, 0x1a, 0x46, 0x19, 0xca, 0x80, 0xec, 
			0xb4, 0x85, 0xa3, 0x1d, 0x74, 0x42, 0x23, 0xce, 
			0x8d, 0xdc, 0x23, 0x94, 0x82, 0x8d, 0x6a, 0x80, 
			0x47, 0x0c, 0x09, 0x2f, 0x5b, 0xa4, 0x13, 0xc3, 
			0x37, 0x8f, 0xa6, 0x05, 0x42, 0x55, 0xc6, 0xf9, 
			0xdf, 0x44, 0x95, 0x86, 0x2b, 0xbb, 0x32, 0x87, 
			0x68, 0x1f, 0x93, 0x1b, 0x68, 0x7c, 0x88, 0x8a, 
			0xbf, 0x84, 0x4d, 0xfc, 0x8f, 0xc2, 0x83, 0x31, 
			0xe5, 0x79, 0x92, 0x8c, 0xd1, 0x2b, 0xd2, 0x39, 
			0x0a, 0xe1, 0x23, 0xcf, 0x03, 0x81, 0x8d, 0x14, 
			0xde, 0xdd, 0xe5, 0xc0, 0xc2, 0x4c, 0x8a, 0xb0, 
			0x18, 0xbf, 0xca, 0x75, 0xca, 0x09, 0x6f, 0x2d, 
			0x53, 0x1f, 0x3d, 0x16, 0x19, 0xe7, 0x85, 0xf1, 
			0xad, 0xa4, 0x37, 0xca, 0xb9, 0x2e, 0x98, 0x05, 
			0x58, 0xb3, 0xdc, 0xe1, 0x47, 0x4a, 0xfb, 0x75, 
			0xbf, 0xed, 0xbf, 0x8f, 0xf5, 0x4c, 0xb2, 0x61, 
			0x8e, 0x02, 0x44, 0xc9, 0xac, 0x0d, 0x3c, 0x66, 
			0xfb, 0x51, 0x59, 0x8c, 0xd2, 0xdb, 0x11, 0xf9, 
			0xbe, 0x39, 0x79, 0x1a, 0xbe, 0x44, 0x7c, 0x63, 
			0x09, 0x4f, 0x7c, 0x45, 0x3b, 0x7f, 0xf8, 0x7c, 
			0xb5, 0xbb, 0x36, 0xb7, 0xc7, 0x9e, 0xfb, 0x08, 
			0x72, 0xd1, 0x70, 0x58, 0xb8, 0x3b, 0x15, 0xab, 
			0x08, 0x66, 0xad, 0x8a, 0x58, 0x65, 0x6c, 0x5a, 
			0x7e, 0x20, 0xdb, 0xdf, 0x30, 0x8b, 0x24, 0x61, 
			0xd9, 0x7c, 0x0e, 0xc0, 0x02, 0x4a, 0x27, 0x15, 
			0x05, 0x52, 0x49, 0xcf, 0x3b, 0x47, 0x8d, 0xdd, 
			0x47, 0x40, 0xde, 0x65, 0x4f, 0x75, 0xca, 0x68, 
			0x6e, 0x0d, 0x73, 0x45, 0xc6, 0x9e, 0xd5, 0x0c, 
			0xdc, 0x2a, 0x8b, 0x33, 0x2b, 0x1f, 0x88, 0x24, 
			0x10, 0x8a, 0xc9, 0x37, 0xeb, 0x05, 0x05, 0x85, 
			0x60, 0x8e, 0xe7, 0x34, 0x09, 0x7f, 0xc0, 0x90, 
			0x54, 0xfb, 0xff, 0x89, 0xee, 0xae, 0xea, 0x79, 
			0x1f, 0x4a, 0x7a, 0xb1, 0xf9, 0x86, 0x82, 0x94, 
			0xa4, 0xf9, 0xe2, 0x7b, 0x42, 0xaf, 0x81, 0x00, 
			0xcb, 0x9d, 0x59, 0xce, 0xf9, 0x64, 0x58, 0x03, 
		},
	},
	{
		0x00ffffffULL,
		64,
		{
			0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45, 
			0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26, 
			0x62, 0x49, 0x77, 0x57, 0x24, 0x70, 0x93, 0x69, 
			0x99, 0x59, 0x57, 0x49, 0x66, 0x96, 0x76, 0x27, 
			0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93, 
			0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95, 
			0x02, 0x88, 0x41, 0x97, 0x16, 0x93, 0x99, 0x37, 
			0x51, 0x05, 0x82, 0x09, 0x74, 0x94, 0x45, 0x92, 
		},
		512,
		{
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 
			0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 
			0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 
			0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 
			0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 
			0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 
			0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 
			0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 
			0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 
			0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 
			0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 
			0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 
			0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 
			0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 
			0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 
			0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
			0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 
			0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
			0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 
			0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
			0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 
			0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
			0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 
			0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 
			0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
			0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff, 
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 
			0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 
			0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 
			0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 
			0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 
			0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 
			0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 
			0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 
			0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 
			0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 
			0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 
			0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 
			0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 
			0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 
			0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 
			0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
			0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 
			0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
			0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 
			0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
			0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 
			0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
			0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 
			0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 
			0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
			0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff, 
		}, {
			0xe3, 0x87, 0xaa, 0xa5, 0x8b, 0xa4, 0x83, 0xaf, 
			0xa7, 0xe8, 0xeb, 0x46, 0x97, 0x78, 0x31, 0x7e, 
			0xcf, 0x4c, 0xf5, 0x73, 0xaa, 0x9d, 0x4e, 0xac, 
			0x23, 0xf2, 0xcd, 0xf9, 0x14, 0xe4, 0xe2, 0x00, 
			0xa8, 0xb4, 0x90, 0xe4, 0x2e, 0xe6, 0x46, 0x80, 
			0x2d, 0xc6, 0xee, 0x2b, 0x47, 0x1b, 0x27, 0x81, 
			0x95, 0xd6, 0x09, 0x18, 0xec, 0xec, 0xb4, 0x4b, 
			0xf7, 0x99, 0x66, 0xf8, 0x3f, 0xab, 0xa0, 0x49, 
			0x92, 0x98, 0xeb, 0xc6, 0x99, 0xc0, 0xc8, 0x63, 
			0x47, 0x15, 0xa3, 0x20, 0xbb, 0x4f, 0x07, 0x5d, 
			0x62, 0x2e, 0x74, 0xc8, 0xc9, 0x32, 0x00, 0x4f, 
			0x25, 0xb4, 0x1e, 0x36, 0x10, 0x25, 0xb5, 0xa8, 
			0x78, 0x15, 0x39, 0x1f, 0x61, 0x08, 0xfc, 0x4a, 
			0xfa, 0x6a, 0x05, 0xd9, 0x30, 0x3c, 0x6b, 0xa6, 
			0x8a, 0x12, 0x8a, 0x55, 0x70, 0x5d, 0x41, 0x59, 
			0x85, 0x83, 0x2f, 0xde, 0xaa, 0xe6, 0xc8, 0xe1, 
			0x91, 0x10, 0xe8, 0x4d, 0x1b, 0x1f, 0x19, 0x9a, 
			0x26, 0x92, 0x11, 0x9e, 0xdc, 0x96, 0x13, 0x26, 
			0x58, 0xf0, 0x9d, 0xa7, 0xc6, 0x23, 0xef, 0xce, 
			0xc7, 0x12, 0x53, 0x7a, 0x3d, 0x94, 0xc0, 0xbf, 
			0x5d, 0x7e, 0x35, 0x2e, 0xc9, 0x4a, 0xe5, 0x79, 
			0x7f, 0xdb, 0x37, 0x7d, 0xc1, 0x55, 0x11, 0x50, 
			0x72, 0x1a, 0xdf, 0x15, 0xbd, 0x26, 0xa8, 0xef, 
			0xc2, 0xfc, 0xaa, 0xd5, 0x68, 0x81, 0xfa, 0x9e, 
			0x62, 0x46, 0x2c, 0x28, 0xf3, 0x0a, 0xe1, 0xce, 
			0xac, 0xa9, 0x3c, 0x34, 0x5c, 0xf2, 0x43, 0xb7, 
			0x3f, 0x54, 0x2e, 0x20, 0x74, 0xa7, 0x05, 0xbd, 
			0x26, 0x43, 0xbb, 0x9f, 0x7c, 0xc7, 0x9b, 0xb6, 
			0xe7, 0x09, 0x1e, 0xa6, 0xe2, 0x32, 0xdf, 0x0f, 
			0x9a, 0xd0, 0xd6, 0xcf, 0x50, 0x23, 0x27, 0x87, 
			0x6d, 0x82, 0x20, 0x7a, 0xbf, 0x21, 0x15, 0xcd, 
			0xac, 0xf6, 0xd5, 0xa4, 0x8f, 0x6c, 0x18, 0x79, 
			0xa6, 0x5b, 0x11, 0x5f, 0x0f, 0x8b, 0x3c, 0xb3, 
			0xc5, 0x9d, 0x15, 0xdd, 0x8c, 0x76, 0x9b, 0xc0, 
			0x14, 0x79, 0x5a, 0x18, 0x37, 0xf3, 0x90, 0x1b, 
			0x58, 0x45, 0xeb, 0x49, 0x1a, 0xdf, 0xef, 0xe0, 
			0x97, 0xb1, 0xfa, 0x30, 0xa1, 0x2f, 0xc1, 0xf6, 
			0x5b, 0xa2, 0x29, 0x05, 0x03, 0x15, 0x39, 0x97, 
			0x1a, 0x10, 0xf2, 0xf3, 0x6c, 0x32, 0x1b, 0xb5, 
			0x13, 0x31, 0xcd, 0xef, 0xb3, 0x9e, 0x39, 0x64, 
			0xc7, 0xef, 0x07, 0x99, 0x94, 0xf5, 0xb6, 0x9b, 
			0x2e, 0xdd, 0x83, 0xa7, 0x1e, 0xf5, 0x49, 0x97, 
			0x1e, 0xe9, 0x3f, 0x44, 0xea, 0xc3, 0x93, 0x8f, 
			0xcd, 0xd6, 0x1d, 0x01, 0xfa, 0x71, 0x79, 0x9d, 
			0xa3, 0xa8, 0x09, 0x1c, 0x4c, 0x48, 0xaa, 0x9e, 
			0xd2, 0x63, 0xff, 0x07, 0x49, 0xdf, 0x95, 0xd4, 
			0x4f, 0xef, 0x6a, 0x0b, 0xb5, 0x78, 0xec, 0x69, 
			0x45, 0x6a, 0xa5, 0x40, 0x8a, 0xe3, 0x2c, 0x7a, 
			0xf0, 0x8a, 0xd7, 0xba, 0x89, 0x21, 0x28, 0x7e, 
			0x3b, 0xbe, 0xe3, 0x1b, 0x76, 0x7b, 0xe0, 0x6a, 
			0x0e, 0x70, 0x5c, 0x86, 0x4a, 0x76, 0x91, 0x37, 
			0xdf, 0x28, 0x29, 0x22, 0x83, 0xea, 0x81, 0xa2, 
			0x48, 0x02, 0x41, 0xb4, 0x4d, 0x99, 0x21, 0xcd, 
			0xbe, 0xc1, 0xbc, 0x28, 0xdc, 0x1f, 0xda, 0x11, 
			0x4b, 0xd8, 0xe5, 0x21, 0x7a, 0xc9, 0xd8, 0xeb, 
			0xaf, 0xa7, 0x20, 0xe9, 0xda, 0x4f, 0x9a, 0xce, 
			0x23, 0x1c, 0xc9, 0x49, 0xe5, 0xb9, 0x6f, 0xe7, 
			0x6f, 0xfc, 0x21, 0x06, 0x3f, 0xdd, 0xc8, 0x3a, 
			0x6b, 0x86, 0x79, 0xc0, 0x0d, 0x35, 0xe0, 0x95, 
			0x76, 0xa8, 0x75, 0x30, 0x5b, 0xed, 0x5f, 0x36, 
			0xed, 0x24, 0x2c, 0x89, 0x00, 0xdd, 0x1f, 0xa9, 
			0x65, 0xbc, 0x95, 0x0d, 0xfc, 0xe0, 0x9b, 0x13, 
			0x22, 0x63, 0xa1, 0xee, 0xf5, 0x2d, 0xd6, 0x88, 
			0x8c, 0x30, 0x9f, 0x5a, 0x7d, 0x71, 0x28, 0x26, 
		},
	},
	{
		0xffffffffULL,
		64,
		{
			0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45, 
			0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26, 
			0x62, 0x49, 0x77, 0x57, 0x24, 0x70, 0x93, 0x69, 
			0x99, 0x59, 0x57, 0x49, 0x66, 0x96, 0x76, 0x27, 
			0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93, 
			0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95, 
			0x02, 0x88, 0x41, 0x97, 0x16, 0x93, 0x99, 0x37, 
			0x51, 0x05, 0x82, 0x09, 0x74, 0x94, 0x45, 0x92, 
		},
		512,
		{
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 
			0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 
			0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 
			0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 
			0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 
			0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 
			0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 
			0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 
			0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 
			0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 
			0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 
			0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 
			0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 
			0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 
			0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 
			0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
			0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 
			0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
			0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 
			0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
			0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 
			0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
			0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 
			0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 
			0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
			0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff, 
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 
			0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 
			0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 
			0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 
			0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 
			0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 
			0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 
			0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 
			0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 
			0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 
			0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 
			0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 
			0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 
			0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 
			0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 
			0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
			0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 
			0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
			0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 
			0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
			0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 
			0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
			0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 
			0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 
			0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
			0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff, 
		}, {
			0xbf, 0x53, 0xd2, 0xda, 0xde, 0x78, 0xe8, 0x22, 
			0xa4, 0xd9, 0x49, 0xa9, 0xbc, 0x67, 0x66, 0xb0, 
			0x1b, 0x06, 0xa8, 0xef, 0x70, 0xd2, 0x67, 0x48, 
			0xc6, 0xa7, 0xfc, 0x36, 0xd8, 0x0a, 0xe4, 0xc5, 
			0x52, 0x0f, 0x7c, 0x4a, 0xb0, 0xac, 0x85, 0x44, 
			0x42, 0x4f, 0xa4, 0x05, 0x16, 0x2f, 0xef, 0x5a, 
			0x6b, 0x7f, 0x22, 0x94, 0x98, 0x06, 0x36, 0x18, 
			0xd3, 0x9f, 0x00, 0x03, 0xcb, 0x5f, 0xb8, 0xd1, 
			0xc8, 0x6b, 0x64, 0x34, 0x97, 0xda, 0x1f, 0xf9, 
			0x45, 0xc8, 0xd3, 0xbe, 0xde, 0xca, 0x4f, 0x47, 
			0x97, 0x02, 0xa7, 0xa7, 0x35, 0xf0, 0x43, 0xdd, 
			0xb1, 0xd6, 0xaa, 0xad, 0xe3, 0xc4, 0xa0, 0xac, 
			0x7c, 0xa7, 0xf3, 0xfa, 0x52, 0x79, 0xbe, 0xf5, 
			0x6f, 0x82, 0xcd, 0x7a, 0x2f, 0x38, 0x67, 0x2e, 
			0x82, 0x48, 0x14, 0xe1, 0x07, 0x00, 0x30, 0x0a, 
			0x05, 0x5e, 0x16, 0x30, 0xb8, 0xf1, 0xcb, 0x0e, 
			0x91, 0x9f, 0x5e, 0x94, 0x20, 0x10, 0xa4, 0x16, 
			0xe2, 0xbf, 0x48, 0xcb, 0x46, 0x99, 0x3d, 0x3c, 
			0xb6, 0xa5, 0x1c, 0x19, 0xba, 0xcf, 0x86, 0x47, 
			0x85, 0xa0, 0x0b, 0xc2, 0xec, 0xff, 0x15, 0xd3, 
			0x50, 0x87, 0x5b, 0x24, 0x6e, 0xd5, 0x3e, 0x68, 
			0xbe, 0x6f, 0x55, 0xbd, 0x7e, 0x05, 0xcf, 0xc2, 
			0xb2, 0xed, 0x64, 0x32, 0x19, 0x8a, 0x64, 0x44, 
			0xb6, 0xd8, 0xc2, 0x47, 0xfa, 0xb9, 0x41, 0xf5, 
			0x69, 0x76, 0x8b, 0x5c, 0x42, 0x93, 0x66, 0xf1, 
			0xd3, 0xf0, 0x0f, 0x03, 0x45, 0xb9, 0x61, 0x23, 
			0xd5, 0x62, 0x04, 0xc0, 0x1c, 0x63, 0xb2, 0x2c, 
			0xe7, 0x8b, 0xaf, 0x11, 0x6e, 0x52, 0x5e, 0xd9, 
			0x0f, 0xde, 0xa3, 0x9f, 0xa4, 0x69, 0x49, 0x4d, 
			0x38, 0x66, 0xc3, 0x1e, 0x05, 0xf2, 0x95, 0xff, 
			0x21, 0xfe, 0xa8, 0xd4, 0xe6, 0xe1, 0x3d, 0x67, 
			0xe4, 0x7c, 0xe7, 0x22, 0xe9, 0x69, 0x8a, 0x1c, 
			0x10, 0x48, 0xd6, 0x8e, 0xbc, 0xde, 0x76, 0xb8, 
			0x6f, 0xcf, 0x97, 0x6e, 0xab, 0x8a, 0xa9, 0x79, 
			0x02, 0x68, 0xb7, 0x06, 0x8e, 0x01, 0x7a, 0x8b, 
			0x9b, 0x74, 0x94, 0x09, 0x51, 0x4f, 0x10, 0x53, 
			0x02, 0x7f, 0xd1, 0x6c, 0x37, 0x86, 0xea, 0x1b, 
			0xac, 0x5f, 0x15, 0xcb, 0x79, 0x71, 0x1e, 0xe2, 
			0xab, 0xe8, 0x2f, 0x5c, 0xf8, 0xb1, 0x3a, 0xe7, 
			0x30, 0x30, 0xef, 0x5b, 0x9e, 0x44, 0x57, 0xe7, 
			0x5d, 0x13, 0x04, 0xf9, 0x88, 0xd6, 0x2d, 0xd6, 
			0xfc, 0x4b, 0x94, 0xed, 0x38, 0xba, 0x83, 0x1d, 
			0xa4, 0xb7, 0x63, 0x49, 0x71, 0xb6, 0xcd, 0x8e, 
			0xc3, 0x25, 0xd9, 0xc6, 0x1c, 0x00, 0xf1, 0xdf, 
			0x73, 0x62, 0x7e, 0xd3, 0x74, 0x5a, 0x5e, 0x84, 
			0x89, 0xf3, 0xa9, 0x5c, 0x69, 0x63, 0x9c, 0x32, 
			0xcd, 0x6e, 0x1d, 0x53, 0x7a, 0x85, 0xf7, 0x5c, 
			0xc8, 0x44, 0x72, 0x6e, 0x8a, 0x72, 0xfc, 0x00, 
			0x77, 0xad, 0x22, 0x00, 0x0f, 0x1d, 0x50, 0x78, 
			0xf6, 0xb8, 0x66, 0x31, 0x8c, 0x66, 0x8f, 0x1a, 
			0xd0, 0x3d, 0x5a, 0x5f, 0xce, 0xd5, 0x21, 0x9f, 
			0x2e, 0xab, 0xbd, 0x0a, 0xa5, 0xc0, 0xf4, 0x60, 
			0xd1, 0x83, 0xf0, 0x44, 0x04, 0xa0, 0xd6, 0xf4, 
			0x69, 0x55, 0x8e, 0x81, 0xfa, 0xb2, 0x4a, 0x16, 
			0x79, 0x05, 0xab, 0x4c, 0x78, 0x78, 0x50, 0x2a, 
			0xd3, 0xe3, 0x8f, 0xdb, 0xe6, 0x2a, 0x41, 0x55, 
			0x6c, 0xec, 0x37, 0x32, 0x57, 0x59, 0x53, 0x3c, 
			0xe8, 0xf2, 0x5f, 0x36, 0x7c, 0x87, 0xbb, 0x55, 
			0x78, 0xd6, 0x67, 0xae, 0x93, 0xf9, 0xe2, 0xfd, 
			0x99, 0xbc, 0xbc, 0x5f, 0x2f, 0xbb, 0xa8, 0x8c, 
			0xf6, 0x51, 0x61, 0x39, 0x42, 0x0f, 0xcf, 0xf3, 
			0xb7, 0x36, 0x1d, 0x86, 0x32, 0x2c, 0x4b, 0xd8, 
			0x4c, 0x82, 0xf3, 0x35, 0xab, 0xb1, 0x52, 0xc4, 
			0xa9, 0x34, 0x11, 0x37, 0x3a, 0xaa, 0x82, 0x20, 
		},
	},
	{
		0xffffffffffULL,
		64,
		{
			0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45, 
			0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26, 
			0x62, 0x49, 0x77, 0x57, 0x24, 0x70, 0x93, 0x69, 
			0x99, 0x59, 0x57, 0x49, 0x66, 0x96, 0x76, 0x27, 
			0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93, 
			0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95, 
			0x02, 0x88, 0x41, 0x97, 0x16, 0x93, 0x99, 0x37, 
			0x51, 0x05, 0x82, 0x09, 0x74, 0x94, 0x45, 0x92, 
		},
		512,
		{
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 
			0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 
			0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 
			0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 
			0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 
			0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 
			0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 
			0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 
			0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 
			0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 
			0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 
			0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 
			0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 
			0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 
			0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 
			0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
			0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 
			0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
			0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 
			0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
			0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 
			0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
			0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 
			0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 
			0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
			0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff, 
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 
			0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 
			0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 
			0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 
			0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 
			0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 
			0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 
			0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 
			0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 
			0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 
			0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 
			0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 
			0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 
			0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 
			0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 
			0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 
			0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 
			0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 
			0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 
			0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 
			0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 
			0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 
			0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 
			0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 
			0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 
			0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff, 
		}, {
			0x64, 0x49, 0x7e, 0x5a, 0x83, 0x1e, 0x4a, 0x93, 
			0x2c, 0x09, 0xbe, 0x3e, 0x53, 0x93, 0x37, 0x6d, 
			0xaa, 0x59, 0x95, 0x48, 0xb8, 0x16, 0x03, 0x1d, 
			0x22, 0x4b, 0xbf, 0x50, 0xa8, 0x18, 0xed, 0x23, 
			0x50, 0xea, 0xe7, 0xe9, 0x60, 0x87, 0xc8, 0xa0, 
			0xdb, 0x51, 0xad, 0x29, 0x0b, 0xd0, 0x0c, 0x1a, 
			0xc1, 0x62, 0x08, 0x57, 0x63, 0x5b, 0xf2, 0x46, 
			0xc1, 0x76, 0xab, 0x46, 0x3b, 0xe3, 0x0b, 0x80, 
			0x8d, 0xa5, 0x48, 0x08, 0x1a, 0xc8, 0x47, 0xb1, 
			0x58, 0xe1, 0x26, 0x4b, 0xe2, 0x5b, 0xb0, 0x91, 
			0x0b, 0xbc, 0x92, 0x64, 0x71, 0x08, 0x08, 0x94, 
			0x15, 0xd4, 0x5f, 0xab, 0x1b, 0x3d, 0x26, 0x04, 
			0xe8, 0xa8, 0xef, 0xf1, 0xae, 0x40, 0x20, 0xcf, 
			0xa3, 0x99, 0x36, 0xb6, 0x68, 0x27, 0xb2, 0x3f, 
			0x37, 0x1b, 0x92, 0x20, 0x0b, 0xe9, 0x02, 0x51, 
			0xe6, 0xd7, 0x3c, 0x5f, 0x86, 0xde, 0x5f, 0xd4, 
			0xa9, 0x50, 0x78, 0x19, 0x33, 0xd7, 0x9a, 0x28, 
			0x27, 0x2b, 0x78, 0x2a, 0x2e, 0xc3, 0x13, 0xef, 
			0xdf, 0xcc, 0x06, 0x28, 0xf4, 0x3d, 0x74, 0x4c, 
			0x2d, 0xc2, 0xff, 0x3d, 0xcb, 0x66, 0x99, 0x9b, 
			0x50, 0xc7, 0xca, 0x89, 0x5b, 0x0c, 0x64, 0x79, 
			0x1e, 0xea, 0xa5, 0xf2, 0x94, 0x99, 0xfb, 0x1c, 
			0x02, 0x6f, 0x84, 0xce, 0x5b, 0x5c, 0x72, 0xba, 
			0x10, 0x83, 0xcd, 0xdb, 0x5c, 0xe4, 0x54, 0x34, 
			0x63, 0x16, 0x65, 0xc3, 0x33, 0xb6, 0x0b, 0x11, 
			0x59, 0x3f, 0xb2, 0x53, 0xc5, 0x17, 0x9a, 0x2c, 
			0x8d, 0xb8, 0x13, 0x78, 0x2a, 0x00, 0x48, 0x56, 
			0xa1, 0x65, 0x30, 0x11, 0xe9, 0x3f, 0xb6, 0xd8, 
			0x76, 0xc1, 0x83, 0x66, 0xdd, 0x86, 0x83, 0xf5, 
			0x34, 0x12, 0xc0, 0xc1, 0x80, 0xf9, 0xc8, 0x48, 
			0x59, 0x2d, 0x59, 0x3f, 0x86, 0x09, 0xca, 0x73, 
			0x63, 0x17, 0xd3, 0x56, 0xe1, 0x3e, 0x2b, 0xff, 
			0x3a, 0x9f, 0x59, 0xcd, 0x9a, 0xeb, 0x19, 0xcd, 
			0x48, 0x25, 0x93, 0xd8, 0xc4, 0x61, 0x28, 0xbb, 
			0x32, 0x42, 0x3b, 0x37, 0xa9, 0xad, 0xfb, 0x48, 
			0x2b, 0x99, 0x45, 0x3f, 0xbe, 0x25, 0xa4, 0x1b, 
			0xf6, 0xfe, 0xb4, 0xaa, 0x0b, 0xef, 0x5e, 0xd2, 
			0x4b, 0xf7, 0x3c, 0x76, 0x29, 0x78, 0x02, 0x54, 
			0x82, 0xc1, 0x31, 0x15, 0xe4, 0x01, 0x5a, 0xac, 
			0x99, 0x2e, 0x56, 0x13, 0xa3, 0xb5, 0xc2, 0xf6, 
			0x85, 0xb8, 0x47, 0x95, 0xcb, 0x6e, 0x9b, 0x26, 
			0x56, 0xd8, 0xc8, 0x81, 0x57, 0xe5, 0x2c, 0x42, 
			0xf9, 0x78, 0xd8, 0x63, 0x4c, 0x43, 0xd0, 0x6f, 
			0xea, 0x92, 0x8f, 0x28, 0x22, 0xe4, 0x65, 0xaa, 
			0x65, 0x76, 0xe9, 0xbf, 0x41, 0x93, 0x84, 0x50, 
			0x6c, 0xc3, 0xce, 0x3c, 0x54, 0xac, 0x1a, 0x6f, 
			0x67, 0xdc, 0x66, 0xf3, 0xb3, 0x01, 0x91, 0xe6, 
			0x98, 0x38, 0x0b, 0xc9, 0x99, 0xb0, 0x5a, 0xbc, 
			0xe1, 0x9d, 0xc0, 0xc6, 0xdc, 0xc2, 0xdd, 0x00, 
			0x1e, 0xc5, 0x35, 0xba, 0x18, 0xde, 0xb2, 0xdf, 
			0x1a, 0x10, 0x10, 0x23, 0x10, 0x83, 0x18, 0xc7, 
			0x5d, 0xc9, 0x86, 0x11, 0xa0, 0x9d, 0xc4, 0x8a, 
			0x0a, 0xcd, 0xec, 0x67, 0x6f, 0xab, 0xdf, 0x22, 
			0x2f, 0x07, 0xe0, 0x26, 0xf0, 0x59, 0xb6, 0x72, 
			0xb5, 0x6e, 0x5c, 0xbc, 0x8e, 0x1d, 0x21, 0xbb, 
			0xd8, 0x67, 0xdd, 0x92, 0x72, 0x12, 0x05, 0x46, 
			0x81, 0xd7, 0x0e, 0xa7, 0x37, 0x13, 0x4c, 0xdf, 
			0xce, 0x93, 0xb6, 0xf8, 0x2a, 0xe2, 0x24, 0x23, 
			0x27, 0x4e, 0x58, 0xa0, 0x82, 0x1c, 0xc5, 0x50, 
			0x2e, 0x2d, 0x0a, 0xb4, 0x58, 0x5e, 0x94, 0xde, 
			0x69, 0x75, 0xbe, 0x5e, 0x0b, 0x4e, 0xfc, 0xe5, 
			0x1c, 0xd3, 0xe7, 0x0c, 0x25, 0xa1, 0xfb, 0xbb, 
			0xd6, 0x09, 0xd2, 0x73, 0xad, 0x5b, 0x0d, 0x59, 
			0x63, 0x1c, 0x53, 0x1f, 0x6a, 0x0a, 0x57, 0xb9, 
		},
	},
};
#define N_VECTORS (sizeof(aes_xts_test_vectors) / sizeof(*aes_xts_test_vectors))

int
main(int argc, char **argv)
{
	int allowed = 0, fail = 0;
	size_t i;
	struct aes_xts_tv *tv;
	u_int8_t result[512];

	if (geteuid() == 0) {
		allowed = getallowsoft();
		if (allowed == 0)
			setallowsoft(1);
	}

	for (i = 0; i < N_VECTORS; i++) {
		tv = &aes_xts_test_vectors[i];

		/* Encrypt test */
		if (syscrypt(tv->key, tv->key_len, tv->seqno, tv->plaintext,
		    result, tv->text_len, 1) < 0) {
			warnx("encrypt with /dev/crypto failed");
			fail++;
			break;
		}
		if (!match(result, tv->ciphertext, tv->text_len)) {
			fail++;
			break;
		} else
			printf("OK encrypt test vector %zu\n", i);

		/* Decrypt test */
		if (syscrypt(tv->key, tv->key_len, tv->seqno, tv->ciphertext,
		    result, tv->text_len, 0) < 0) {
			warnx("decrypt with /dev/crypto failed");
			fail++;
			break;
		}
		if (!match(result, tv->plaintext, tv->text_len)) {
			fail++;
			break;
		} else
			printf("OK decrypt test vector %zu\n", i);
	}

	if (geteuid() == 0 && allowed == 0)
		setallowsoft(0);
	exit((fail > 0) ? 1 : 0);
}

