#!/usr/bin/python
# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2007-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.
#
# Author: Alessandro Decina <alessandro@fluendo.com>

from twisted.trial.unittest import TestCase
from elisa.core.launcher import Launcher
import os

class LauncherMock(Launcher):
    platform = 'Linux'

    def __init__(self):
        Launcher.__init__(self)
        self.infos = []
        self.errors = []
        self.environment = {}
        self.running = False
        self.already_running = False
        self.voodoo_done = False
        self.exit_code = -1

    def info(self, message):
        self.infos.append(message)

    def error(self, message):
        self.errors.append(message)

    def get_platform(self):
        return self.platform

    def get_environment(self):
        return self.environment

    def get_installation_path(self):
        # this is called only when platform == 'Windows'
        return 'c:\\Program Files\\Elisa-0.9\\'

    def main_before_voodoo(self):
        # we want to test this differently, we don't want to mess the whole
        # test run by swapping cores and whatnot
        self.voodoo_done = True

    def run_application(self):
        self.running = True

    def poke_running_instance(self):
        return self.already_running

    def exit(self, code):
        self.exit_code = code

class TestLauncherBase(object):
    platform = None

    def setUp(self):
        self.launcher = LauncherMock()
        self.launcher.platform = self.platform

    def test_base(self):
        argv = ['python.exe', 'elisa.py']
        self.launcher.main(argv)

        self.failUnless(self.launcher.voodoo_done)
        # exit() must not have been called
        self.failUnlessEqual(self.launcher.exit_code, -1)
        self.failUnless(self.launcher.running)

    def test_already_running(self):
        self.launcher.already_running = True

        argv = ['python.exe', 'elisa.py']
        self.launcher.main(argv)

        self.failUnless(self.launcher.voodoo_done)
        # exit(2) means elisa was running
        self.failUnlessEqual(self.launcher.exit_code, 2)
    
class TestLauncherUnix(TestLauncherBase, TestCase):
    platform = 'Linux'

class TestLauncherWindows(TestLauncherBase, TestCase):
    platform = 'Windows'

    def test_environment(self):
        env = self.launcher.environment
        env['PGM_PLUGIN_PATH'] = '12'
        env['PATH'] = '42'
        # keep a copy, needed later
        original_env = dict(env)

        argv = ['python.exe', 'elisa.py']
        self.launcher.main(argv)

        env = self.launcher.environment
        self.failIfEqual(env, original_env)

        value = env['PATH']
        path = value.split(os.path.pathsep)
        self.failUnlessEqual(path[len(path)-1],
                original_env['PATH'])
        
        value = env['PGM_PLUGIN_PATH']
        self.failUnlessEqual(value.split(os.path.pathsep)[0],
                original_env['PGM_PLUGIN_PATH'])
