/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 1999-2004 Ryan Bobko
                       ryan@ostrich-emulators.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************** * * * * * * * * * * * * **************************/

#ifndef _QHACCMAINTENANCE_H
#define _QHACCMAINTENANCE_H

#include "tableops.h"

#include <memory>

#include <qfont.h>
#include <qstring.h>
#include <qdatetime.h>

class QHacc;
class QHaccTable;
class QHaccIOPlugin;
class QHaccResultSet;


/****************************************************************/
/** QHACC engine extensions                                    **/
/**                                                            **/
/**                                                            **/
/**                                                            **/
/**                                                            **/
/****************************************************************/

class QHaccExt{
 public:
	QHaccExt( QHacc * qhacc );
  ~QHaccExt();

	/**
	 * @return an array of QHaccResultSets used for most import/export
	 * operations. This array must be deleted by the caller
	 */
	static QHaccResultSet * getRSSet();

	/**
	 * Archive transactions before the given date. Archives remove the
	 * affected transactions and their splits from the engine and reset
	 * account balances to a pre-archival state.
	 */
	void archive( const QDate& );

	/**
	 * Archive the the given account. Archives remove the affected
	 * transactions and their splits from the engine and reset other account
	 * balances to a pre-archival state.
	 */
	void archive( const Account& );

	/**
	 * Restore transactions from the given importer
	 * @param imp a plugin that contains the data to be restored
	 */
	void restore( QHaccIOPlugin * imp );

	/**
	 * Export the data currently in the engine to the give QHACC_HOME. This
	 * function does not perform a alter the engine's data in any way.
	 * 
	 * @param home The QHACC_HOME to export to.
	 * @return True, if there are no errors accessing the QHACC_HOME
	 */
	bool exp( const char * home );

	/**
	 * Import the data from the give QHACC_HOME into the engine. This
	 * function does not perform a save.
	 * 
	 * @param home The QHACC_HOME on which to operate.
	 * @return True, if there are no errors accessing the QHACC_HOME
	 */
	bool imp( const char * home );

	/**
	 * Reconcile the transactions in the given file home location. This
	 * checks for duplicate transactions and if it finds any, marks them
	 * as reconciled in the engine.
	 * 
	 * @param home The QHACC_HOME on which to operate.
	 * @return True, if there are no errors accessing the QHACC_HOME
	 */
	bool reco( const char * home );

	/**
	 * Verify that the database is in a consistent state. This is done
	 * by checking that all transactions reference an existing account.
	 *
	 * @param andFix If orphaned transactions are found, delete them.
	 * @return True, if the database is in a consistent state at the end
	 * of this function.
	 */
	bool verify( bool andFix );

	/**
	 * Make a creation script for creating the given QHACC_HOME
	 * @param str The QHACC_HOME to create
	 * @return A string containing all the commands necessary for creating 
	 * a QHACC_HOME
	 */
	QString create( const QString& str );

 private:
	QHacc * engine;
	static const int OLDID, NEWID, DUPE, NCOLS;

	QHaccResultSet * getMergedImpOf( QHaccIOPlugin * db, bool reco ) const;

	static TableGet getGetter( Table t, bool forReco );
	static void dupeError( Table t, const TableRow& row,
												 const QString& printer );
  auto_ptr<QHaccResultSet> getMerged( Table model, QHaccTable& import,
																			int modelidcol, int importidcol,
																			const TableGet& modelselection,
																			const TableGet& importselection ) const;
	void conserveIDs( QHaccResultSet * ) const;
	void replaceEngineData( QHaccResultSet * );

	// return a resultset of orphans
	auto_ptr<QHaccResultSet> iverify( QHaccTable& child, int ccol,
																		QHaccTable& parent, int pcol ) const;

	void iarchive( auto_ptr<QHaccResultSet> );
};

#endif
