# Orca
#
# Copyright 2010 Consorcio Fernando de los Rios.
# Author: Javier Hernandez Antunez <jhernandez@emergya.es>
# Author: Alejandro Leiva <aleiva@emergya.es>
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., Franklin Street, Fifth Floor,
# Boston MA  02110-1301 USA.

"""Displays the Save Profile As dialog."""

__id__        = "$Id$"
__version__   = "$Revision$"
__date__      = "$Date$"
__copyright__ = "Copyright (c) 2010 Consorcio Fernando de los Rios."
__license__   = "LGPL"

import locale
import sys
from gi.repository import Gtk

from . import orca_state
from .orca_i18n import _

OS = None
newProfile = None

class OrcaProfileGUI(Gtk.Dialog):

    def __init__(self):
        """Initialize the Orca profile configuration GUI."""

        Gtk.Dialog.__init__(self)

        # Translators: Profiles in Orca make it possible for users to
        # quickly switch amongst a group of pre-defined settings (e.g.
        # an 'English' profile for reading text written in English using
        # an English-language speech synthesizer and braille rules, and
        # a similar 'Spanish' profile for reading Spanish text. The
        # following string is the title of a dialog in which users can
        # save a newly-defined profile.
        #
        self.set_title(_('Save Profile As'))
        self.set_has_resize_grip(False)

        self.add_button('gtk-cancel', Gtk.ResponseType.CANCEL)
        self.add_button('gtk-save', Gtk.ResponseType.ACCEPT)

        grid = Gtk.Grid()
        grid.set_property('margin', 12)
        grid.set_row_spacing(10)
        grid.set_column_spacing(10)

        # Right now the content area is a GtkBox. We'll need to update
        # this once GtkBox is fully deprecated.
        contentArea = self.get_content_area()
        contentArea.pack_start(grid, True, True, 0)

        self.profileEntry = Gtk.Entry()
        self.profileEntry.set_property('hexpand', True)
        self.profileEntry.set_activates_default(True)
        grid.attach(self.profileEntry, 1, 0, 1, 1)

        # Translators: Profiles in Orca make it possible for users to
        # quickly switch amongst a group of pre-defined settings (e.g.
        # an 'English' profile for reading text written in English using
        # an English-language speech synthesizer and braille rules, and
        # a similar 'Spanish' profile for reading Spanish text. The
        # following string is the label for a text entry in which the user
        # enters the name of a new settings profile being saved via the
        # 'Save Profile As' dialog.
        #
        label = Gtk.Label(_('_Profile Name:'))
        label.set_use_underline(True)
        label.set_mnemonic_widget(self.profileEntry)
        grid.attach(label, 0, 0, 1, 1)

        defaultButton = self.get_widget_for_response(Gtk.ResponseType.ACCEPT)
        defaultButton.set_property('can-default', True)
        defaultButton.set_property('has-default', True)

        self.connect('response', self.onResponse)
        self.connect('destroy', self.onDestroy)

        self.searchString = None
        self.profileString = None
        self.prefsDialog = None

    def init(self):
        self.profileString = ''

    def showGUI(self, prefsDialog):
        """Show the Save Profile As dialog."""

        self.show_all()
        self.prefsDialog = prefsDialog
        self.profileEntry.set_text(self.profileString)

        ts = orca_state.lastInputEventTimestamp
        if ts == 0:
            ts = Gtk.get_current_event_time()
        self.present_with_time(ts)

    def onResponse(self, widget, response):
        """Signal handler for the responses emitted by the dialog."""

        if response in [Gtk.ResponseType.CANCEL, Gtk.ResponseType.DELETE_EVENT]:
            self.hide()
            return

        if response == Gtk.ResponseType.ACCEPT:
            global newProfile

            newProfile = self.profileEntry.get_text()
            if newProfile:
                self.destroy()
            if self.prefsDialog:
                self.prefsDialog.saveProfile(newProfile)

    def onDestroy(self, widget):
        """Signal handler for the 'destroy' signal of the dialog."""

        global OS

        OS = None

def showProfileUI(prefsDialog=None):
    global OS
    global newProfile

    newProfile = None

    if not OS:
        OS = OrcaProfileGUI()
        OS.init()

    OS.showGUI(prefsDialog)

def main():
    locale.setlocale(locale.LC_ALL, '')

    showProfileUI()

    Gtk.main()
    sys.exit(0)

if __name__ == "__main__":
    main()
