# -*- coding: utf-8 -*-

# KCacheGrind object taken from lsprofcalltree.py written by:
# David Allouche
# Jp Calderone & Itamar Shtull-Trauring
# Johan Dahlin

"""
Profiling helpers
"""

import cProfile


def label(code):
    if isinstance(code, str):
        return ('~', 0, code)    # built-in functions ('~' sorts at the end)
    else:
        return '%s %s:%d' % (code.co_name, code.co_filename, code.co_firstlineno)

class KCacheGrind(object):
    """ Convert a L{profile.Stats} object into data suitable for visualisation
        in KCacheGrind.
    """
    def __init__(self, profiler):
        self.data = profiler.getstats()
        self.out_file = None

    def output(self, out_file):
        self.out_file = out_file
        print >> out_file, 'events: Ticks'
        self._print_summary()
        for entry in self.data:
            self._entry(entry)

    def _print_summary(self):
        max_cost = 0
        for entry in self.data:
            totaltime = int(entry.totaltime * 1000)
            max_cost = max(max_cost, totaltime)
        print >> self.out_file, 'summary: %d' % (max_cost,)

    def _entry(self, entry):
        out_file = self.out_file
        code = entry.code
        inlinetime = int(entry.inlinetime * 1000)
        #print >> out_file, 'ob=%s' % (code.co_filename,)
        if isinstance(code, str):
            print >> out_file, 'fi=~'
        else:
            print >> out_file, 'fi=%s' % (code.co_filename,)
        print >> out_file, 'fn=%s' % (label(code),)
        if isinstance(code, str):
            print >> out_file, '0 ', inlinetime
        else:
            print >> out_file, '%d %d' % (code.co_firstlineno, inlinetime)
        # recursive calls are counted in entry.calls
        if entry.calls:
            calls = entry.calls
        else:
            calls = []
        if isinstance(code, str):
            lineno = 0
        else:
            lineno = code.co_firstlineno
        for subentry in calls:
            self._subentry(lineno, subentry)
        print >> out_file

    def _subentry(self, lineno, subentry):
        out_file = self.out_file
        code = subentry.code
        totaltime = int(subentry.totaltime * 1000)
        #print >> out_file, 'cob=%s' % (code.co_filename,)
        print >> out_file, 'cfn=%s' % (label(code),)
        if isinstance(code, str):
            print >> out_file, 'cfi=~'
            print >> out_file, 'calls=%d 0' % (subentry.callcount,)
        else:
            print >> out_file, 'cfi=%s' % (code.co_filename,)
            print >> out_file, 'calls=%d %d' % (
                subentry.callcount, code.co_firstlineno)
        print >> out_file, '%d %d' % (lineno, totaltime)


def profileit(func):
    """ Decorator profiling the decorated callable using cProfile. It outputs
        the result both in the console and in a file suitable for visualisation
        in KCacheGrind.

        Usage example:

            @profileit
            def my_function(arg1, ...):
                ...
    """
    def _profiled_func(*args, **kwargs):
        local_func = func
        prof = cProfile.Profile()
        prof = prof.runctx('result = local_func(*args, **kwargs)',
                           globals(), locals())
        prof.print_stats(sort=2)
        k = KCacheGrind(prof)
        data = open('prof.kgrind', 'w+')
        k.output(data)
        data.close()
        return locals()['result']
    return _profiled_func
