class Struct

  include Enumerable

  class << self
    alias_method :subclass_new, :new
  end

  ##
  # call-seq:
  #   Struct.new( [aString] [, aSym]+> )    => StructClass
  #   StructClass.new(arg, ...)             => obj
  #   StructClass[arg, ...]                 => obj
  #
  # Creates a new class, named by <em>aString</em>, containing accessor
  # methods for the given symbols. If the name <em>aString</em> is omitted,
  # an anonymous structure class will be created. Otherwise, the name of
  # this struct will appear as a constant in class <tt>Struct</tt>, so it
  # must be unique for all <tt>Struct</tt>s in the system and should start
  # with a capital letter. Assigning a structure class to a constant
  # effectively gives the class the name of the constant.
  #
  # <tt>Struct::new</tt> returns a new <tt>Class</tt> object, which can then
  # be used to create specific instances of the new structure. The number of
  # actual parameters must be less than or equal to the number of attributes
  # defined for this class; unset parameters default to \nil{}. Passing too
  # many parameters will raise an \E{ArgumentError}.
  #
  # The remaining methods listed in this section (class and instance) are
  # defined for this generated class.
  #
  #    # Create a structure with a name in Struct
  #    Struct.new("Customer", :name, :address)    #=> Struct::Customer
  #    Struct::Customer.new("Dave", "123 Main")   #=> #<Struct::Customer
  # name="Dave", address="123 Main">
  #    # Create a structure named by its constant
  #    Customer = Struct.new(:name, :address)     #=> Customer
  #    Customer.new("Dave", "123 Main")           #=> #<Customer
  # name="Dave", address="123 Main">

  def self.new(klass_name, *attrs, &block)
    if klass_name
      begin
        klass_name = StringValue klass_name
      rescue TypeError
        attrs.unshift klass_name
        klass_name = nil
      end
    end

    attrs = attrs.map do |a|
      case a
      when Symbol
        a
      when String
        sym = a.to_sym
        unless sym.kind_of? Symbol
          raise TypeError, "#to_sym didn't return a symbol"
        end
        sym
      else
        raise TypeError, "#{a.inspect} is not a symbol"
      end
    end

    klass = Class.new self do

      attr_accessor(*attrs)

      def self.new(*args, &block)
        return subclass_new(*args, &block)
      end

      def self.[](*args)
        return new(*args)
      end

      const_set :STRUCT_ATTRS, attrs
      specialize_initialize
    end

    Struct.const_set klass_name, klass if klass_name

    klass.module_eval(&block) if block

    return klass
  end

  # Don't specialize any thing created in the kernel. We hook up
  # better form of this in delta.
  def self.specialize_initialize
  end

  def self.make_struct(name, attrs)
    new name, *attrs
  end

  def _attrs # :nodoc:
    return self.class::STRUCT_ATTRS
  end

  def instance_variables
    # Hide the ivars used to store the struct fields
    super() - _attrs.map { |a| "@#{a}" }
  end

  def initialize(*args)
    attrs = self.class::STRUCT_ATTRS

    unless args.length <= attrs.length
      raise ArgumentError, "Expected #{attrs.size}, got #{args.size}"
    end

    attrs.each_with_index do |attr, i|
      instance_variable_set :"@#{attr}", args[i]
    end
  end

  private :initialize

  ##
  # call-seq:
  #   struct == other_struct     => true or false
  #
  # Equality---Returns <tt>true</tt> if <em>other_struct</em> is equal to
  # this one: they must be of the same class as generated by
  # <tt>Struct::new</tt>, and the values of all instance variables must be
  # equal (according to <tt>Object#==</tt>).
  #
  #    Customer = Struct.new(:name, :address, :zip)
  #    joe   = Customer.new("Joe Smith", "123 Maple, Anytown NC", 12345)
  #    joejr = Customer.new("Joe Smith", "123 Maple, Anytown NC", 12345)
  #    jane  = Customer.new("Jane Doe", "456 Elm, Anytown NC", 12345)
  #    joe == joejr   #=> true
  #    joe == jane    #=> false

  def ==(other)
    return false if self.class != other.class

    Thread.detect_recursion self, other do
      return self.values == other.values
    end

    # Subtle: if we are here, we are recursing and haven't found any difference, so:
    true
  end

  ##
  # call-seq:
  #   struct[symbol]    => anObject
  #   struct[fixnum]    => anObject 
  #
  # Attribute Reference---Returns the value of the instance variable named
  # by <em>symbol</em>, or indexed (0..length-1) by <em>fixnum</em>. Will
  # raise <tt>NameError</tt> if the named variable does not exist, or
  # <tt>IndexError</tt> if the index is out of range.
  #
  #    Customer = Struct.new(:name, :address, :zip)
  #    joe = Customer.new("Joe Smith", "123 Maple, Anytown NC", 12345)
  #    joe["name"]   #=> "Joe Smith"
  #    joe[:name]    #=> "Joe Smith"
  #    joe[0]        #=> "Joe Smith"

  def [](var)
    case var
    when Symbol, String
      # ok
    else
      var = Integer(var)
      a_len = _attrs.length
      if var > a_len - 1
        raise IndexError, "offset #{var} too large for struct(size:#{a_len})"
      end
      if var < -a_len
        raise IndexError, "offset #{var + a_len} too small for struct(size:#{a_len})"
      end
      var = _attrs[var]
    end

    unless _attrs.include? var.to_sym
      raise NameError, "no member '#{var}' in struct"
    end

    return instance_variable_get(:"@#{var}")
  end

  ##
  # call-seq:
  #   struct[symbol] = obj    => obj
  #   struct[fixnum] = obj    => obj
  #
  # Attribute Assignment---Assigns to the instance variable named by
  # <em>symbol</em> or <em>fixnum</em> the value <em>obj</em> and returns
  # it. Will raise a <tt>NameError</tt> if the named variable does not
  # exist, or an <tt>IndexError</tt> if the index is out of range.
  #
  #    Customer = Struct.new(:name, :address, :zip)
  #    joe = Customer.new("Joe Smith", "123 Maple, Anytown NC", 12345)
  #    joe["name"] = "Luke"
  #    joe[:zip]   = "90210"
  #    joe.name   #=> "Luke"
  #    joe.zip    #=> "90210"

  def []=(var, obj)
    case var
    when Symbol
      unless _attrs.include? var
        raise NameError, "no member '#{var}' in struct"
      end
    when String
      var = var.to_sym
      unless _attrs.include? var
        raise NameError, "no member '#{var}' in struct"
      end
    else
      var = Integer(var)
      a_len = _attrs.length
      if var > a_len - 1
        raise IndexError, "offset #{var} too large for struct(size:#{a_len})"
      end
      if var < -a_len
        raise IndexError, "offset #{var + a_len} too small for struct(size:#{a_len})"
      end

      var = _attrs[var]
    end

    return instance_variable_set(:"@#{var}", obj)
  end

  ##
  # call-seq:
  #   struct.eql?(other)   => true or false
  #
  # Two structures are equal if they are the same object, or if all their
  # fields are equal (using <tt>eql?</tt>).

  def eql?(other)
    return true if equal? other
    return false if self.class != other.class

    Thread.detect_recursion self, other do
      _attrs.each do |var|
        mine =   instance_variable_get(:"@#{var}")
        theirs = other.instance_variable_get(:"@#{var}")

        return false unless mine.eql? theirs
      end
    end

    # Subtle: if we are here, then no difference was found, or we are recursing
    # In either case, return
    true
  end

  def each(&block)
    return to_enum :each unless block_given?
    values.each(&block)
  end

  def each_pair
    return to_enum :each_pair unless block_given?
    _attrs.map { |var| yield var, instance_variable_get(:"@#{var}") }
  end

  ##
  # call-seq:
  #   struct.hash   => fixnum
  #
  # Return a hash value based on this struct's contents.

  def hash
    hash_val = size
    return _attrs.size if Thread.detect_outermost_recursion self do
      _attrs.each { |var| hash_val ^= instance_variable_get(:"@#{var}").hash }
    end
    return hash_val
  end

  ##
  # call-seq:
  #   struct.length    => fixnum
  #   struct.size      => fixnum
  #
  # Returns the number of instance variables.
  #
  #    Customer = Struct.new(:name, :address, :zip)
  #    joe = Customer.new("Joe Smith", "123 Maple, Anytown NC", 12345)
  #    joe.length   #=> 3

  def length
    return _attrs.length
  end

  alias_method :size, :length

  def self.length
    return self::STRUCT_ATTRS.size
  end

  ##
  # call-seq:
  #   struct.members    => array
  #
  # Returns an array of strings representing the names of the instance
  # variables.
  #
  #    Customer = Struct.new(:name, :address, :zip)
  #    joe = Customer.new("Joe Smith", "123 Maple, Anytown NC", 12345)
  #    joe.members   #=> ["name", "address", "zip"]

  def self.members
    return self::STRUCT_ATTRS.map { |member| member.to_s }
  end

  def members
    return self.class.members
  end

  ##
  # call-seq:
  #   struct.select(fixnum, ... )   => array
  #   struct.select {|i| block }    => array
  #
  # The first form returns an array containing the elements in
  # <em>struct</em> corresponding to the given indices. The second form
  # invokes the block passing in successive elements from <em>struct</em>,
  # returning an array containing those elements for which the block returns
  # a true value (equivalent to <tt>Enumerable#select</tt>).
  #
  #    Lots = Struct.new(:a, :b, :c, :d, :e, :f)
  #    l = Lots.new(11, 22, 33, 44, 55, 66)
  #    l.select(1, 3, 5)               #=> [22, 44, 66]
  #    l.select(0, 2, 4)               #=> [11, 33, 55]
  #    l.select(-1, -3, -5)            #=> [66, 44, 22]
  #    l.select {|v| (v % 2).zero? }   #=> [22, 44, 66]

  def select(&block)
    to_a.select(&block)
  end

  ##
  # call-seq:
  #   struct.to_a     => array
  #   struct.values   => array
  #
  # Returns the values for this instance as an array.
  #
  #    Customer = Struct.new(:name, :address, :zip)
  #    joe = Customer.new("Joe Smith", "123 Maple, Anytown NC", 12345)
  #    joe.to_a[1]   #=> "123 Maple, Anytown NC"

  def to_a
    return _attrs.map { |var| instance_variable_get :"@#{var}" }
  end

  ##
  # call-seq:
  #   struct.to_s      => string
  #   struct.inspect   => string
  #
  # Describe the contents of this struct in a string.

  def to_s
    return "[...]" if Thread.guarding? self

    Thread.recursion_guard self do
      values = []

      _attrs.each do |var|
        val = instance_variable_get :"@#{var}"
        values << "#{var}=#{val.inspect}"
      end

      "#<struct #{self.class.inspect} #{values.join(', ')}>"
    end
  end

  alias_method :inspect, :to_s

  ##
  # call-seq:
  #   struct.to_a     => array
  #   struct.values   => array
  #
  # Returns the values for this instance as an array.
  #
  #    Customer = Struct.new(:name, :address, :zip)
  #    joe = Customer.new("Joe Smith", "123 Maple, Anytown NC", 12345)
  #    joe.to_a[1]   #=> "123 Maple, Anytown NC"

  alias_method :values, :to_a

  ##
  # call-seq:
  #   struct.values_at(selector,... )  => an_array
  #
  # Returns an array containing the elements in <em>self</em> corresponding
  # to the given selector(s). The selectors may be either integer indices or
  # ranges. See also </code>.select<code>.
  #
  #    a = %w{ a b c d e f }
  #    a.values_at(1, 3, 5)
  #    a.values_at(1, 3, 5, 7)
  #    a.values_at(-1, -3, -5, -7)
  #    a.values_at(1..3, 2...5)

  def values_at(*args)
    to_a.values_at(*args)
  end
end

