/*******************************************************************************
 * Copyright (c) 2006 BEA Systems, Inc. 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    rfrost@bea.com - initial API and implementation
 *******************************************************************************/
package org.eclipse.jst.server.generic.weblogic;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jst.server.generic.core.internal.CorePlugin;
import org.eclipse.jst.server.generic.core.internal.GenericServer;
import org.eclipse.jst.server.generic.core.internal.Trace;
import org.eclipse.jst.server.generic.core.internal.publishers.AbstractModuleAssembler;
import org.eclipse.jst.server.generic.core.internal.publishers.AntPublisher;
import org.eclipse.wst.server.core.IModule;
import org.eclipse.wst.server.core.IModuleArtifact;
import org.eclipse.wst.server.core.IModuleType;
import org.eclipse.wst.server.core.IServer;
import org.eclipse.wst.server.core.internal.Server;
import org.eclipse.wst.server.core.model.IModuleResourceDelta;
import org.eclipse.wst.server.core.util.ProjectModule;

/**
 * <p>Subclass of <code>AntPublisher</code> used by the WebLogic generic adapters. Currently adds the following 
 * features:</p>
 * <ul>
 * <li>EARs are assembled in fully exploded form (i.e. child modules are exploded rather than included as archives).
 * <li>Deployment is only performed if there have been changes in the underlying module projects.
 * </ul>
 * <p>Other WLS-specific enhancements may be added later.</p>
 */
public class GenericWebLogicPublisher extends AntPublisher {
	
	 /* (non-Javadoc)
	 * @see org.eclipse.wtp.server.core.model.IPublisher#publish(org.eclipse.wtp.server.core.resources.IModuleResource[], org.eclipse.core.runtime.IProgressMonitor)
	 */
	public IStatus[] publish(IModuleArtifact[] resource, IProgressMonitor monitor){
		final IModule[] module = getModule();
		final GenericServer server = getServer();
		if(module.length>1)// only respond to root module calls. 
			return null;
		if(monitor.isCanceled())
			return null;

		// check if there are any changes in the root module or child modules (don't want to 
		// publish unless we need to publish)
		final List modulePath = new ArrayList();
		modulePath.add(module[0]);
		if (!needToPublish(modulePath, module[0], server)) {
			Trace.trace(Trace.FINER, "GenericWebLogicAntPublisher: no module changes, publish not necessary");
			return null;
		}
		
		// Delegate to standard generic AntPublisher
		IStatus[] status = super.publish(resource, monitor);
		                
		// Update the publish status of the child modules as appropriate
		if (status == null) {
        	updateChildModulePublishState(modulePath, module[0], IServer.PUBLISH_STATE_NONE, server);            
		} else {
        	updateChildModulePublishState(modulePath, module[0], IServer.PUBLISH_STATE_UNKNOWN, server);
		}
		
		return status;
	}

	/**
	 * Override assembleModule() to ensure that EAR assembly generates a fully exploded EAR.
	 */
	protected void assembleModule(final IProgressMonitor monitor) throws CoreException {
		final IModule[] module = getModule();
		final GenericServer server = getServer();
		if (isModuleType(module[0], "jst.ear")) { //$NON-NLS-1$
			final GenericWebLogicEarAssembler assembler = new GenericWebLogicEarAssembler(module[0], server, AbstractModuleAssembler.Factory.getDefaultAssembleRoot(module[0], server));
			assembler.assemble(monitor);
		} else {
			super.assembleModule(monitor);
		}
	}
	
	private static boolean isModuleType(IModule module, String moduleTypeId){	
		if(module.getModuleType()!=null && moduleTypeId.equals(module.getModuleType().getId()))
			return true;
		return false;
	}
	
	/*
	 * Determines if a publish is required based on the publish resource delta.
	 */
	private boolean needToPublish(List modulePath, IModule module, GenericServer genericServer) {
		final Server server = (Server) genericServer.getServer();
		final IModule[] modules = (IModule[]) modulePath.toArray(new IModule[]{});
		final IModuleResourceDelta[] delta = server.getPublishedResourceDelta(modules);

		if (delta.length > 0) { 
			if (CorePlugin.getDefault().isDebugging()) {
				final ProjectModule pm =(ProjectModule)module.loadAdapter(ProjectModule.class, new NullProgressMonitor());
				Trace.trace(Trace.FINER, "GenericWebLogicPublisher: Module " + pm.getId() + " has changes, path length: " + modulePath.size()); //$NON-NLS-1$ //$NON-NLS-2$
			}
			// current module has changes
			return true;
		}
		
		// only check children for EARs
		// XXX this is due to a limitation of the WTP server framework - it does not track 
		// publish resource deltas for certain modules (like web-inf/lib associations for web modules)
		// user will need to "touch" one of the modules associated with the server to ensure a 
		// deploy in that case (also does not support FULL_PUBLISH)
		if (isModuleType(module, "jst.ear")) {
			final IModule[] childModules = genericServer.getChildModules(new IModule[]{module});
			for (int i = 0; i < childModules.length; i++) {
				modulePath.add(childModules[i]);
				final boolean publish = needToPublish(modulePath, childModules[i], genericServer);
				modulePath.remove(modulePath.size()-1);
				if (publish) {
					return true;
				} 
			}
		}
		if (CorePlugin.getDefault().isDebugging()) {
			final ProjectModule pm =(ProjectModule)module.loadAdapter(ProjectModule.class, new NullProgressMonitor());
			Trace.trace(Trace.FINER, "GenericWebLogicPublisher: Module " + pm.getId() + " does not have changes"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		return false;
	}

	/*
	 * Updates the child module status (GenericServerBehavior only updates the root)
	 */
	private void updateChildModulePublishState(List modulePath, IModule module, int publishState, GenericServer genericServer) {
		final Server server = (Server) genericServer.getServer();
		// only update children of EARs
		final IModuleType moduleType = module.getModuleType();
		if (moduleType == null || !"jst.ear".equals(moduleType.getId())) {
			return;
		}
		final IModule[] childModules = genericServer.getChildModules(new IModule[]{module});
		for (int i = 0; i < childModules.length; i++) {
			modulePath.add(childModules[i]);
			// update this child
			if (CorePlugin.getDefault().isDebugging()) {
				final ProjectModule pm =(ProjectModule)childModules[i].loadAdapter(ProjectModule.class, new NullProgressMonitor());
				Trace.trace(Trace.FINER, "GenericWebLogicPublisher: Updating module publish state for child module" + pm.getId() + " to " + publishState); //$NON-NLS-1$ //$NON-NLS-2$
			}
			server.setModulePublishState((IModule[]) modulePath.toArray(new IModule[]{}), publishState);
			// recursively update sub-children
			updateChildModulePublishState(modulePath, childModules[i], publishState, genericServer);
			modulePath.remove(modulePath.size()-1);
		}
	}
}
