/*
 * Decompiled with CFR 0.152.
 */
package com.intellij.util.io;

import com.intellij.openapi.Forceable;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.util.io.Bits;
import com.intellij.util.io.MappedBufferWrapper;
import com.intellij.util.io.ReadWriteMappedBufferWrapper;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilterInputStream;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import org.jetbrains.annotations.NonNls;

public class MappedFile
implements Forceable {
    private static final Logger LOG = Logger.getInstance("#com.intellij.util.io.MappedFile");
    private MappedBufferWrapper myHolder;
    private final File myFile;
    private long myRealSize;
    private long mySize;
    private long myPosition;
    private boolean myIsDirty = false;
    @NonNls
    private static final String UTF_8_CHARSET_NAME = "UTF-8";
    @NonNls
    private static final String RW = "rw";
    private final byte[] buffer = new byte[8];

    public MappedFile(File file, int initialSize) throws IOException {
        this.myFile = file;
        if (!file.exists() || file.length() == 0L) {
            this.writeLength(0L);
        }
        this.myPosition = 0L;
        this.map();
        this.mySize = this.readLength();
        if (this.mySize == 0L) {
            this.resize(initialSize);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long readLength() {
        File lengthFile = this.getLengthFile();
        FilterInputStream stream = null;
        try {
            stream = new DataInputStream(new FileInputStream(lengthFile));
            long l = ((DataInputStream)stream).readLong();
            return l;
        }
        catch (IOException e) {
            this.writeLength(this.myRealSize);
            long l = this.myRealSize;
            return l;
        }
        finally {
            if (stream != null) {
                try {
                    stream.close();
                }
                catch (IOException e) {
                    LOG.error(e);
                }
            }
        }
    }

    private File getLengthFile() {
        return new File(this.myFile.getPath() + ".len");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeLength(long len) {
        File lengthFile = this.getLengthFile();
        FilterOutputStream stream = null;
        try {
            stream = new DataOutputStream(new FileOutputStream(lengthFile));
            ((DataOutputStream)stream).writeLong(len);
        }
        catch (FileNotFoundException e) {
            LOG.error(e);
        }
        catch (IOException e) {
            LOG.error(e);
        }
        finally {
            if (stream != null) {
                try {
                    stream.close();
                }
                catch (IOException e) {
                    LOG.error(e);
                }
            }
        }
    }

    private void map() {
        this.myHolder = new ReadWriteMappedBufferWrapper(this.myFile);
        this.myRealSize = this.myFile.length();
        if (LOG.isDebugEnabled()) {
            LOG.assertTrue(this.myPosition > 0L && this.myPosition < this.myRealSize || this.myPosition == 0L && this.myRealSize == 0L, "myPosition=" + this.myPosition + ", myRealSize=" + this.myRealSize);
        }
        this.myHolder.buf().position((int)this.myPosition);
    }

    public short getShort(int index) throws IOException {
        this.seek(index);
        return this.readShort();
    }

    public short readShort() throws IOException {
        this.get(this.buffer, 0, 2);
        return Bits.getShort(this.buffer, 0);
    }

    public void putShort(int index, short value) throws IOException {
        this.seek(index);
        this.writeShort(value);
    }

    public void writeShort(int value) throws IOException {
        Bits.putShort(this.buffer, 0, (short)value);
        this.put(this.buffer, 0, 2);
    }

    public int getInt(int index) throws IOException {
        this.seek(index);
        return this.readInt();
    }

    public long getLong(int index) throws IOException {
        this.seek(index);
        return this.readLong();
    }

    public void putInt(int index, int value) throws IOException {
        this.seek(index);
        this.writeInt(value);
    }

    public void putLong(int index, long value) throws IOException {
        this.seek(index);
        this.writeLong(value);
    }

    public byte get(int index) throws IOException {
        this.seek(index);
        return this.readByte();
    }

    public void put(int index, byte value) throws IOException {
        this.seek(index);
        this.writeByte(value);
    }

    public void get(int index, byte[] dst, int offset, int length) throws IOException {
        this.seek(index);
        this.get(dst, offset, length);
    }

    public void get(byte[] dst, int offset, int length) throws IOException {
        if (this.myPosition + (long)length > this.mySize) {
            throw new EOFException();
        }
        this.buf().get(dst, offset, length);
        this.myPosition += (long)length;
    }

    public void put(int index, byte[] src, int offset, int length) throws IOException {
        this.seek(index);
        this.put(src, offset, length);
    }

    public void seek(long pos) throws IOException {
        this.ensureSize(pos);
        this.buf().position((int)pos);
        this.myPosition = pos;
        if (pos > this.mySize) {
            this.mySize = pos;
        }
    }

    private ByteBuffer buf() {
        if (!this.isMapped()) {
            this.map();
        }
        return this.myHolder.buf();
    }

    private void ensureSize(long pos) throws IOException {
        while (pos >= this.myRealSize) {
            this.expand();
        }
    }

    private void expand() throws IOException {
        this.resize((int)((this.myRealSize + 1L) * 13L) >> 3);
    }

    public void put(byte[] src, int offset, int length) throws IOException {
        this.ensureSize(this.myPosition + (long)length);
        this.myIsDirty = true;
        this.buf().put(src, offset, length);
        this.myPosition += (long)length;
        if (this.myPosition > this.mySize) {
            this.mySize = this.myPosition;
        }
    }

    public void flush() {
        if (this.myIsDirty) {
            this.writeLength(this.mySize);
            ByteBuffer buf = this.buf();
            if (buf instanceof MappedByteBuffer) {
                MappedBufferWrapper.tryForce(buf);
            }
            this.myIsDirty = false;
        }
    }

    @Override
    public void force() {
        this.flush();
    }

    @Override
    public boolean isDirty() {
        return this.myIsDirty;
    }

    public void close() {
        if (this.myIsDirty) {
            this.writeLength(this.mySize);
        }
        this.unmap();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void resize(int size) throws IOException {
        int current = (int)this.myRealSize;
        if (current == size) {
            return;
        }
        this.unmap();
        RandomAccessFile raf = new RandomAccessFile(this.myFile, RW);
        try {
            raf.setLength(size);
        }
        finally {
            raf.close();
        }
        this.map();
    }

    public final long length() {
        return this.mySize;
    }

    public long getFilePointer() {
        return this.myPosition;
    }

    public int readInt() throws IOException {
        this.get(this.buffer, 0, 4);
        return Bits.getInt(this.buffer, 0);
    }

    public long readLong() throws IOException {
        this.get(this.buffer, 0, 8);
        return Bits.getLong(this.buffer, 0);
    }

    public void writeInt(int value) throws IOException {
        Bits.putInt(this.buffer, 0, value);
        this.put(this.buffer, 0, 4);
    }

    public void writeLong(long value) throws IOException {
        Bits.putLong(this.buffer, 0, value);
        this.put(this.buffer, 0, 8);
    }

    public String readUTF() throws IOException {
        try {
            int len = this.readInt();
            byte[] bytes = new byte[len];
            this.get(bytes, 0, len);
            return new String(bytes, UTF_8_CHARSET_NAME);
        }
        catch (UnsupportedEncodingException e) {
            return "";
        }
    }

    public void writeUTF(String value) throws IOException {
        try {
            byte[] bytes = value.getBytes(UTF_8_CHARSET_NAME);
            this.writeInt(bytes.length);
            this.put(bytes, 0, bytes.length);
        }
        catch (UnsupportedEncodingException unsupportedEncodingException) {
            // empty catch block
        }
    }

    public int readUnsignedShort() throws IOException {
        this.get(this.buffer, 0, 2);
        int ch1 = this.buffer[0] & 0xFF;
        int ch2 = this.buffer[1] & 0xFF;
        return (ch1 << 8) + ch2;
    }

    public char readChar() throws IOException {
        return (char)this.readUnsignedShort();
    }

    public void writeChar(char value) throws IOException {
        this.writeShort(value);
    }

    public byte readByte() throws IOException {
        this.get(this.buffer, 0, 1);
        return this.buffer[0];
    }

    public void writeByte(byte value) throws IOException {
        this.buffer[0] = value;
        this.put(this.buffer, 0, 1);
    }

    private void unmap() {
        if (this.myHolder != null) {
            this.myHolder.unmap();
        }
    }

    public boolean isMapped() {
        return this.myHolder.isMapped();
    }
}

