
/*
 * Copyright (c) 2002 by The XFree86 Project, Inc.
 * Author: Ivan Pascal.
 *
 * Based on the code from bsd_io.c which is
 * Copyright 1992 by Rich Murphey <Rich@Rice.edu>
 * Copyright 1993 by David Dawes <dawes@xfree86.org>
 */

#define NEED_EVENTS

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <X11/X.h>
#include <termios.h>

#include "compiler.h"

#include "xf86.h"
#include "xf86Priv.h"
#include "xf86_OSlib.h"
#include "xf86Parser.h"

#include "xf86Xinput.h"
#include "xf86OSKbd.h"
#include "atKeynames.h"

#ifdef WSCONS_SUPPORT
#include <dev/wscons/wsconsio.h>
#include <dev/wscons/wsksymdef.h>

#define KB_OVRENC \
	{ KB_UK,	"gb" }, \
	{ KB_SV,	"se" }, \
	{ KB_SG,	"ch(de)" }, \
	{ KB_SF,	"ch(fr)" }, \
	{ KB_LA,	"latam" }, \
	{ KB_CF,	"ca(fr)" }

struct nameint {
  int val;
  char *name;
} kbdenc[] = { KB_OVRENC, KB_ENCTAB, { 0 } };

struct nameint kbdvar[] = {
	{ KB_NODEAD, "nodeadkeys" },
	{ KB_DVORAK, "dvorak" },
	{ 0 }
};

struct nameint kbdopt[] = {
	{ KB_SWAPCTRLCAPS, "ctrl:swapcaps" },
	{ 0 }
};
#endif

extern void KbdGetMapping(InputInfoPtr pInfo, KeySymsPtr pKeySyms,
                          CARD8 *pModMap);

extern int priv_open_device(const char *dev);

extern Bool VTSwitchEnabled;

static KbdProtocolRec protocols[] = {
   {"standard", PROT_STD },
#ifdef WSCONS_SUPPORT
   {"wskbd", PROT_WSCONS },
#endif
   { NULL, PROT_UNKNOWN_KBD }
};

typedef struct {
   struct termios kbdtty;
} BsdKbdPrivRec, *BsdKbdPrivPtr;

static
int KbdInit(InputInfoPtr pInfo, int what)
{
    KbdDevPtr pKbd = (KbdDevPtr) pInfo->private;
    BsdKbdPrivPtr priv = (BsdKbdPrivPtr) pKbd->private;

    if (pKbd->isConsole) {
        switch (pKbd->consType) {
#if defined(PCCONS_SUPPORT) || defined(SYSCONS_SUPPORT) || defined (PCVT_SUPPORT)  || defined (WSCONS_SUPPORT)
	    case PCCONS:
	    case SYSCONS:
	    case PCVT:
#if defined WSCONS_SUPPORT
            case WSCONS:
#endif
 	         tcgetattr(pInfo->fd, &(priv->kbdtty));
#endif
	         break;
        }
    }

    return Success;
}

static void
SetKbdLeds(InputInfoPtr pInfo, int leds)
{
    KbdDevPtr pKbd = (KbdDevPtr) pInfo->private;
    int real_leds = 0;

#ifdef LED_CAP
    if (leds & XLED1)  real_leds |= LED_CAP;
#endif
#ifdef LED_NUM
    if (leds & XLED2)  real_leds |= LED_NUM;
#endif
#ifdef LED_SCR
    if (leds & XLED3)  real_leds |= LED_SCR;
    if (leds & XLED4)  real_leds |= LED_SCR;
#endif

    switch (pKbd->consType) {

	case PCCONS:
		break;
#if defined (SYSCONS_SUPPORT) || defined (PCVT_SUPPORT)
	case SYSCONS:
	case PCVT:
	     ioctl(pInfo->fd, KDSETLED, real_leds);
	     break;
#endif
#if defined(WSCONS_SUPPORT)
        case WSCONS:
             ioctl(pInfo->fd, WSKBDIO_SETLEDS, &real_leds);
             break;
#endif
    }
}

static int
GetKbdLeds(InputInfoPtr pInfo)
{
    KbdDevPtr pKbd = (KbdDevPtr) pInfo->private;
    int leds = 0, real_leds = 0;

    switch (pKbd->consType) {
	case PCCONS:
	     break;
#if defined (SYSCONS_SUPPORT) || defined (PCVT_SUPPORT)
	case SYSCONS:
	case PCVT:
	     ioctl(pInfo->fd, KDGETLED, &real_leds);
	     break;
#endif
#if defined(WSCONS_SUPPORT)
        case WSCONS:
             ioctl(pInfo->fd, WSKBDIO_GETLEDS, &real_leds);
             break;
#endif
    }

#ifdef LED_CAP
    if (real_leds & LED_CAP) leds |= XLED1;
#endif
#ifdef LED_NUM
    if (real_leds & LED_NUM) leds |= XLED2;
#endif
#ifdef LED_SCR
    if (real_leds & LED_SCR) leds |= XLED3;
#endif

    return(leds);
}

static void
SetKbdRepeat(InputInfoPtr pInfo, char rad)
{
    KbdDevPtr pKbd = (KbdDevPtr) pInfo->private;
    switch (pKbd->consType) {

	case PCCONS:
		break;
#if defined (SYSCONS_SUPPORT) || defined (PCVT_SUPPORT)
	case SYSCONS:
	case PCVT:
		ioctl(pInfo->fd, KDSETRAD, rad);
		break;
#endif
    }
}

static int
KbdOn(InputInfoPtr pInfo, int what)
{
    KbdDevPtr pKbd = (KbdDevPtr) pInfo->private;
#if defined(SYSCONS_SUPPORT) || defined(PCCONS_SUPPORT) || defined(PCVT_SUPPORT) || defined(WSCONS_SUPPORT)
    BsdKbdPrivPtr priv = (BsdKbdPrivPtr) pKbd->private;
    struct termios nTty;
#endif
#ifdef WSCONS_SUPPORT
    int option;
#endif

    if (pKbd->isConsole) {
        switch (pKbd->consType) {

#if defined(SYSCONS_SUPPORT) || defined(PCCONS_SUPPORT) || defined(PCVT_SUPPORT) || defined(WSCONS_SUPPORT)
	    case SYSCONS:
	    case PCCONS:
	    case PCVT:
#ifdef WSCONS_SUPPORT
            case WSCONS:
#endif
		 nTty = priv->kbdtty;
		 nTty.c_iflag = IGNPAR | IGNBRK;
		 nTty.c_oflag = 0;
		 nTty.c_cflag = CREAD | CS8;
		 nTty.c_lflag = 0;
		 nTty.c_cc[VTIME] = 0;
		 nTty.c_cc[VMIN] = 1;
		 cfsetispeed(&nTty, 9600);
		 cfsetospeed(&nTty, 9600);
		 if (tcsetattr(pInfo->fd, TCSANOW, &nTty) < 0) {
			 xf86Msg(X_ERROR, "KbdOn: tcsetattr: %s\n",
			     strerror(errno));
		 }
                 break; 
#endif 
        }
#if defined (SYSCONS_SUPPORT) || defined (PCVT_SUPPORT) || defined (WSCONS_SUPPORT)
        switch (pKbd->consType) {
	    case SYSCONS:
	    case PCVT:
#ifdef K_CODE
                 if (pKbd->CustomKeycodes)
		     ioctl(pInfo->fd, KDSKBMODE, K_CODE);
	         else
	             ioctl(pInfo->fd, KDSKBMODE, K_RAW);
#else
		 ioctl(pInfo->fd, KDSKBMODE, K_RAW);
#endif
	         break;
#endif
#ifdef WSCONS_SUPPORT
            case WSCONS:
                 option = WSKBD_RAW;
                 if (ioctl(pInfo->fd, WSKBDIO_SETMODE, &option) == -1) {
			 FatalError("can't switch keyboard to raw mode. "
				    "Enable support for it in the kernel\n"
				    "or use for example:\n\n"
				    "Option \"Protocol\" \"wskbd\"\n"
				    "Option \"Device\" \"/dev/wskbd0\"\n"
				    "\nin your xorg.conf(5) file\n");
		 }
		 break;
#endif
        }
    }
    return Success;
}

static int
KbdOff(InputInfoPtr pInfo, int what)
{
    KbdDevPtr pKbd = (KbdDevPtr) pInfo->private;
    BsdKbdPrivPtr priv = (BsdKbdPrivPtr) pKbd->private;
#ifdef WSCONS_SUPPORT
    int option;
#endif

    if (pKbd->isConsole) {
        switch (pKbd->consType) {
#if defined (SYSCONS_SUPPORT) || defined (PCVT_SUPPORT)
	    case SYSCONS:
	    case PCVT:
	         ioctl(pInfo->fd, KDSKBMODE, K_XLATE);
	         /* FALL THROUGH */
#endif
#if defined(SYSCONS_SUPPORT) || defined(PCCONS_SUPPORT) || defined(PCVT_SUPPORT)
	    case PCCONS:
	         tcsetattr(pInfo->fd, TCSANOW, &(priv->kbdtty));
	         break;
#endif
#ifdef WSCONS_SUPPORT
            case WSCONS:
                 option = WSKBD_TRANSLATED;
                 ioctl(xf86Info.consoleFd, WSKBDIO_SETMODE, &option);
                 tcsetattr(pInfo->fd, TCSANOW, &(priv->kbdtty));
	         break;
#endif
        }
    }
    return Success;
}

static void
SoundBell(InputInfoPtr pInfo, int loudness, int pitch, int duration)
{
    KbdDevPtr pKbd = (KbdDevPtr) pInfo->private;
#ifdef WSCONS_SUPPORT
    struct wskbd_bell_data wsb;
#endif

    if (loudness && pitch) {
    	switch (pKbd->consType) {
#ifdef PCCONS_SUPPORT
	    case PCCONS:
	         { int data[2];
		   data[0] = pitch;
		   data[1] = (duration * loudness) / 50;
		   ioctl(pInfo->fd, CONSOLE_X_BELL, data);
		   break;
		 }
#endif
#if defined (SYSCONS_SUPPORT) || defined (PCVT_SUPPORT)
	    case SYSCONS:
	    case PCVT:
		 ioctl(pInfo->fd, KDMKTONE,
		 ((1193190 / pitch) & 0xffff) |
		 (((unsigned long)duration*loudness/50)<<16));
		 break;
#endif
#if defined (WSCONS_SUPPORT)
            case WSCONS:
                 wsb.which = WSKBD_BELL_DOALL;
                 wsb.pitch = pitch;
                 wsb.period = duration;
                 wsb.volume = loudness;
                 ioctl(pInfo->fd, WSKBDIO_COMPLEXBELL, &wsb);
                 break;
#endif
	}
    }
}

static void
stdReadInput(InputInfoPtr pInfo)
{
    KbdDevPtr pKbd = (KbdDevPtr) pInfo->private;
    unsigned char rBuf[64];
    int nBytes, i;
    if ((nBytes = read( pInfo->fd, (char *)rBuf, sizeof(rBuf))) > 0) {
       for (i = 0; i < nBytes; i++)
	   pKbd->PostEvent(pInfo, rBuf[i] & 0x7f,
                           rBuf[i] & 0x80 ? FALSE : TRUE);
       }
}

#ifdef WSCONS_SUPPORT

static void
WSReadInput(InputInfoPtr pInfo)
{
    KbdDevPtr pKbd = (KbdDevPtr) pInfo->private;
    struct wscons_event events[64];
    int type;
    int blocked, n, i;

    if ((n = read( pInfo->fd, events, sizeof(events))) > 0) {
        n /=  sizeof(struct wscons_event);
        for (i = 0; i < n; i++) {
	    type = events[i].type;
	    if (type == WSCONS_EVENT_KEY_UP || type == WSCONS_EVENT_KEY_DOWN) {
		/* It seems better to block SIGIO there */
		blocked = xf86BlockSIGIO();
		pKbd->PostEvent(pInfo, (unsigned int)(events[i].value),
				type == WSCONS_EVENT_KEY_DOWN ? TRUE : FALSE);
		xf86UnblockSIGIO(blocked);
	    }
	} /* for */
    }
}

static void
printWsType(char *type, char *devname)
{
    xf86Msg(X_PROBED, "%s: Keyboard type: %s\n", devname, type); 
}
#endif

static Bool
OpenKeyboard(InputInfoPtr pInfo)
{
    KbdDevPtr pKbd = (KbdDevPtr) pInfo->private;
    int i;
    KbdProtocolId prot = PROT_UNKNOWN_KBD;
    char *s;
#ifdef WSCONS_SUPPORT
    kbd_t wsenc = 0;
#endif

    s = xf86SetStrOption(pInfo->options, "Protocol", NULL);
    for (i = 0; protocols[i].name; i++) {
        if (xf86NameCmp(s, protocols[i].name) == 0) {
           prot = protocols[i].id;
           break;
        }
    }

    switch (prot) {
    	case PROT_STD:
           pInfo->read_input = stdReadInput;
           break;
#ifdef WSCONS_SUPPORT
        case PROT_WSCONS:
           pInfo->read_input = WSReadInput;
           break;
#endif
        default:
           xf86Msg(X_ERROR,"\"%s\" is not a valid keyboard protocol name\n", s);
           xfree(s);
           return FALSE;
    }
    xf86Msg(X_CONFIG, "%s: Protocol: %s\n", pInfo->name, s);
    xfree(s);

    s = xf86SetStrOption(pInfo->options, "Device", NULL);
    if (s == NULL) {
       if (prot == PROT_WSCONS) {
           xf86Msg(X_ERROR,"A \"device\" option is required with"
                                  " the \"wskbd\" keyboard protocol\n");
           return FALSE;
       } else {
           pInfo->fd = xf86Info.consoleFd;
           pKbd->isConsole = TRUE;
           pKbd->consType = xf86Info.consType;
       }
    } else {
#ifndef X_PRIVSEP
       pInfo->fd = open(s, O_RDONLY | O_NONBLOCK | O_EXCL);
#else
       pInfo->fd = priv_open_device(s);
#endif
       if (pInfo->fd == -1) {
           xf86Msg(X_ERROR, "%s: cannot open \"%s\"\n", pInfo->name, s);
           xfree(s);
           return FALSE;
       }
       pKbd->isConsole = FALSE;
       pKbd->consType = xf86Info.consType;
       xfree(s);
    }

#if defined (SYSCONS_SUPPORT) || defined (PCVT_SUPPORT)
    if (pKbd->isConsole &&
        ((pKbd->consType == SYSCONS) || (pKbd->consType == PCVT)))
        pKbd->vtSwitchSupported = TRUE;
#endif

#ifdef WSCONS_SUPPORT
    if (prot == PROT_WSCONS) {
       pKbd->consType = WSCONS;
       /* Find out keyboard type */
       if (ioctl(pInfo->fd, WSKBDIO_GTYPE, &(pKbd->wsKbdType)) == -1) {
           xf86Msg(X_ERROR, "%s: cannot get keyboard type", pInfo->name);
           close(pInfo->fd);
           return FALSE;
       }
       switch (pKbd->wsKbdType) {
           case WSKBD_TYPE_PC_XT:
               printWsType("XT", pInfo->name);
               break;
           case WSKBD_TYPE_PC_AT:
               printWsType("AT", pInfo->name);
               break;
           case WSKBD_TYPE_USB:
               printWsType("USB", pInfo->name);
               break;
#ifdef WSKBD_TYPE_ADB
	   case WSKBD_TYPE_ADB:
               printWsType("ADB", pInfo->name);
               break;
#endif
#ifdef WSKBD_TYPE_SUN
           case WSKBD_TYPE_SUN:
               printWsType("Sun", pInfo->name);
               break;
#endif
#ifdef WSKBD_TYPE_SUN5
	   case WSKBD_TYPE_SUN5:
	       printWsType("Sun5", pInfo->name);
	       break;
#endif
           case WSKBD_TYPE_LK201:
               printWsType("LK-201", pInfo->name);
               break;
           case WSKBD_TYPE_LK401:
               printWsType("LK-401", pInfo->name);
               break;
           default:
               xf86Msg(X_ERROR, "%s: Unsupported wskbd type \"%d\"",
                                pInfo->name, pKbd->wsKbdType);
               close(pInfo->fd);
               return FALSE;
       }
    }

    /*
     * Try to get the configured keyboard translation from the wscons
     * keyboard driver (see kbd(8) for more information) if no
     * XkbLayout has been specified.  Do this even if the protocol is
     * not wskbd.
     */
    if (xf86findOption(pInfo->options, "XkbLayout") != NULL)
        return TRUE;

    if (ioctl(pInfo->fd, WSKBDIO_GETENCODING, &wsenc) == -1) {
	/* Ignore the error, we just use the defaults */
	xf86Msg(X_ERROR, "%s: error getting wscons layout name: %s\n",
		pInfo->name, strerror(errno));
	return TRUE;
    }
    if (KB_ENCODING(wsenc) == KB_USER) {
	/* Ignore wscons "user" layout */
	xf86Msg(X_INFO, "%s: ignoring \"user\" wscons layout\n", pInfo->name);
	xf86addNewOption(pInfo->options, "XkbLayout", "us");
	return TRUE;
    }

    for (i = 0; kbdenc[i].val; i++)
	if(KB_ENCODING(wsenc) == kbdenc[i].val) {
	    xf86Msg(X_PROBED, "%s: using wscons layout %s\n",
		    pInfo->name, kbdenc[i].name);
            xf86addNewOption(pInfo->options, "XkbLayout", kbdenc[i].name);
            break;
        }
    if (xf86findOption(pInfo->options, "XkbVariant") == NULL)
        for (i = 0; kbdvar[i].val; i++)
            if (KB_VARIANT(wsenc) == kbdvar[i].val) {
		xf86Msg(X_PROBED, "%s: using wscons variant %s\n",
			pInfo->name, kbdvar[i].name);
                xf86addNewOption(pInfo->options, "XkbVariant", kbdvar[i].name);
                break;
            }
    if (xf86findOption(pInfo->options, "XkbOptions") == NULL)
        for (i = 0; kbdopt[i].val; i++)
            if (KB_VARIANT(wsenc) == kbdopt[i].val) {
		xf86Msg(X_PROBED, "%s: using wscons option %s\n",
			pInfo->name, kbdopt[i].name);
                xf86addNewOption(pInfo->options, "XkbOptions", kbdopt[i].name);
                break;
            }
#endif
    return TRUE;
}

_X_EXPORT Bool
xf86OSKbdPreInit(InputInfoPtr pInfo)
{
    KbdDevPtr pKbd = pInfo->private;

    pKbd->KbdInit	= KbdInit;
    pKbd->KbdOn		= KbdOn;
    pKbd->KbdOff	= KbdOff;
    pKbd->Bell		= SoundBell;
    pKbd->SetLeds	= SetKbdLeds;
    pKbd->GetLeds	= GetKbdLeds;
    pKbd->SetKbdRepeat	= SetKbdRepeat;
    pKbd->KbdGetMapping	= KbdGetMapping;

    pKbd->RemapScanCode = NULL;

    pKbd->OpenKeyboard = OpenKeyboard;
    pKbd->vtSwitchSupported = FALSE;
    pKbd->CustomKeycodes = FALSE;
    
    pKbd->private = xcalloc(sizeof(BsdKbdPrivRec), 1);
    if (pKbd->private == NULL) {
       xf86Msg(X_ERROR,"can't allocate keyboard OS private data\n");
       return FALSE;
    }
    return TRUE;
}
