/*
 * Decompiled with CFR 0.152.
 */
package com.intellij.util.io.storage;

import com.intellij.openapi.Disposable;
import com.intellij.openapi.Forceable;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.util.io.FileUtil;
import com.intellij.util.ArrayUtil;
import com.intellij.util.io.PagePool;
import com.intellij.util.io.RecordDataOutput;
import com.intellij.util.io.storage.DataTable;
import com.intellij.util.io.storage.RecordsTable;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.IOException;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;

public class Storage
implements Disposable,
Forceable {
    private static final Logger LOG = Logger.getInstance("#com.intellij.util.io.storage.Storage");
    private final Object lock = new Object();
    private final RecordsTable myRecordsTable;
    private DataTable myDataTable;
    private final PagePool myPool;
    @NonNls
    public static final String INDEX_EXTENSION = ".storageRecordIndex";
    @NonNls
    public static final String DATA_EXTENSION = ".storageData";
    private static final int MAX_PAGES_TO_FLUSH_AT_A_TIME = 50;

    public static boolean deleteFiles(String storageFilePath) {
        File recordsFile = new File(storageFilePath + INDEX_EXTENSION);
        File dataFile = new File(storageFilePath + DATA_EXTENSION);
        boolean deletedRecordsFile = FileUtil.delete(recordsFile);
        boolean deletedDataFile = FileUtil.delete(dataFile);
        return deletedRecordsFile && deletedDataFile;
    }

    public static void convertFromOldExtensions(String storageFilePath) {
        FileUtil.delete(new File(storageFilePath + ".rindex"));
        FileUtil.delete(new File(storageFilePath + ".data"));
    }

    @NotNull
    public static Storage create(String storageFilePath) throws IOException {
        Storage storage = Storage.create(storageFilePath, PagePool.SHARED);
        if (storage == null) {
            throw new IllegalStateException("@NotNull method com/intellij/util/io/storage/Storage.create must not return null");
        }
        return storage;
    }

    @NotNull
    public static Storage create(String storageFilePath, PagePool pool) throws IOException {
        Storage storage = Storage.create(storageFilePath, pool, 0);
        if (storage == null) {
            throw new IllegalStateException("@NotNull method com/intellij/util/io/storage/Storage.create must not return null");
        }
        return storage;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Lifted jumps to return sites
     */
    @NotNull
    private static Storage create(String storageFilePath, PagePool pool, int retryCount) throws IOException {
        Storage storage;
        DataTable dataTable;
        Storage.convertFromOldExtensions(storageFilePath);
        File recordsFile = new File(storageFilePath + INDEX_EXTENSION);
        File dataFile = new File(storageFilePath + DATA_EXTENSION);
        if (recordsFile.exists() != dataFile.exists()) {
            Storage.deleteFiles(storageFilePath);
        }
        FileUtil.createIfDoesntExist(recordsFile);
        FileUtil.createIfDoesntExist(dataFile);
        RecordsTable recordsTable = null;
        try {
            recordsTable = new RecordsTable(recordsFile, pool);
            dataTable = new DataTable(dataFile, pool);
        }
        catch (IOException e) {
            boolean deleted;
            LOG.info(e.getMessage());
            if (recordsTable != null) {
                recordsTable.dispose();
            }
            if (!(deleted = Storage.deleteFiles(storageFilePath))) {
                throw new IOException("Can't delete caches at: " + storageFilePath);
            }
            if (retryCount >= 5) {
                throw new IOException("Can't create storage at: " + storageFilePath);
            }
            Storage storage2 = Storage.create(storageFilePath, pool, retryCount + 1);
            storage = storage2;
            if (storage2 != null) return storage;
            throw new IllegalStateException("@NotNull method com/intellij/util/io/storage/Storage.create must not return null");
        }
        Storage storage3 = new Storage(storageFilePath, recordsTable, dataTable, pool);
        storage = storage3;
        if (storage3 == null) throw new IllegalStateException("@NotNull method com/intellij/util/io/storage/Storage.create must not return null");
        return storage;
    }

    private Storage(String path, RecordsTable recordsTable, DataTable dataTable, PagePool pool) {
        this.myRecordsTable = recordsTable;
        this.myDataTable = dataTable;
        this.myPool = pool;
        if (this.myDataTable.isCompactNecessary()) {
            this.compact(path);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void compact(String path) {
        Object object = this.lock;
        synchronized (object) {
            LOG.info("Space waste in " + path + " is " + this.myDataTable.getWaste() + " bytes. Compacting now.");
            long start = System.currentTimeMillis();
            try {
                File newDataFile = new File(path + ".storageData.backup");
                FileUtil.delete(newDataFile);
                FileUtil.createIfDoesntExist(newDataFile);
                File oldDataFile = new File(path + DATA_EXTENSION);
                DataTable newDataTable = new DataTable(newDataFile, this.myPool);
                int count = this.myRecordsTable.getRecordsCount();
                for (int i = 0; i < count; ++i) {
                    long addr = this.myRecordsTable.getAddress(i);
                    int size = this.myRecordsTable.getSize(i);
                    if (size <= 0) continue;
                    assert (addr > 0L);
                    int capacity = Storage.calcCapacity(size);
                    long newaddr = newDataTable.allocateSpace(capacity);
                    byte[] bytes = new byte[size];
                    this.myDataTable.readBytes(addr, bytes);
                    newDataTable.writeBytes(newaddr, bytes);
                    this.myRecordsTable.setAddress(i, newaddr);
                    this.myRecordsTable.setCapacity(i, capacity);
                }
                this.myDataTable.dispose();
                newDataTable.dispose();
                if (!FileUtil.delete(oldDataFile)) {
                    throw new IOException("Can't delete file: " + oldDataFile);
                }
                newDataFile.renameTo(oldDataFile);
                this.myDataTable = new DataTable(oldDataFile, this.myPool);
            }
            catch (IOException e) {
                LOG.info("Compact failed: " + e.getMessage());
            }
            long timedelta = System.currentTimeMillis() - start;
            LOG.info("Done compacting in " + timedelta + "msec.");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int getVersion() {
        Object object = this.lock;
        synchronized (object) {
            return this.myRecordsTable.getVersion();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setVersion(int expectedVersion) {
        Object object = this.lock;
        synchronized (object) {
            this.myRecordsTable.setVersion(expectedVersion);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void force() {
        Object object = this.lock;
        synchronized (object) {
            this.myDataTable.force();
            this.myRecordsTable.force();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean flushSome() {
        Object object = this.lock;
        synchronized (object) {
            boolean okRecords = this.myRecordsTable.flushSome(50);
            boolean okData = this.myDataTable.flushSome(50);
            return okRecords && okData;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean isDirty() {
        Object object = this.lock;
        synchronized (object) {
            return this.myDataTable.isDirty() || this.myRecordsTable.isDirty();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int createNewRecord() throws IOException {
        Object object = this.lock;
        synchronized (object) {
            return this.myRecordsTable.createNewRecord();
        }
    }

    public StorageDataOutput createStream() throws IOException {
        return this.writeStream(this.createNewRecord());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void appendBytes(int record, byte[] bytes) {
        assert (record > 0);
        int delta = bytes.length;
        if (delta == 0) {
            return;
        }
        Object object = this.lock;
        synchronized (object) {
            int capacity = this.myRecordsTable.getCapacity(record);
            int oldSize = this.myRecordsTable.getSize(record);
            int newSize = oldSize + delta;
            if (newSize > capacity) {
                if (oldSize > 0) {
                    byte[] newbytes = new byte[newSize];
                    System.arraycopy(this.readBytes(record), 0, newbytes, 0, oldSize);
                    System.arraycopy(bytes, 0, newbytes, oldSize, delta);
                    this.writeBytes(record, newbytes);
                } else {
                    this.writeBytes(record, bytes);
                }
            } else {
                long address = this.myRecordsTable.getAddress(record) + (long)oldSize;
                this.myDataTable.writeBytes(address, bytes);
                this.myRecordsTable.setSize(record, newSize);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void writeBytes(int record, byte[] bytes) {
        assert (record > 0);
        Object object = this.lock;
        synchronized (object) {
            long address;
            int requiredLength = bytes.length;
            int currentCapacity = this.myRecordsTable.getCapacity(record);
            int currentSize = this.myRecordsTable.getSize(record);
            assert (currentSize >= 0);
            if (requiredLength == 0 && currentSize == 0) {
                return;
            }
            if (currentCapacity >= requiredLength) {
                address = this.myRecordsTable.getAddress(record);
            } else {
                if (currentCapacity > 0) {
                    this.myDataTable.reclaimSpace(currentCapacity);
                }
                int newCapacity = Storage.calcCapacity(requiredLength);
                address = this.myDataTable.allocateSpace(newCapacity);
                this.myRecordsTable.setAddress(record, address);
                this.myRecordsTable.setCapacity(record, newCapacity);
            }
            this.myDataTable.writeBytes(address, bytes);
            this.myRecordsTable.setSize(record, requiredLength);
        }
    }

    private static int calcCapacity(int requiredLength) {
        return Math.max(64, Storage.nearestPowerOfTwo(requiredLength * 3 / 2));
    }

    private static int nearestPowerOfTwo(int n) {
        int power = 1;
        while (n != 0) {
            power *= 2;
            n /= 2;
        }
        return power;
    }

    public StorageDataOutput writeStream(int record) {
        return new StorageDataOutput(this, record);
    }

    public AppenderStream appendStream(int record) {
        return new AppenderStream(record);
    }

    public DataInputStream readStream(int record) {
        byte[] bytes = this.readBytes(record);
        return new DataInputStream(new ByteArrayInputStream(bytes));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public byte[] readBytes(int record) {
        assert (record > 0);
        Object object = this.lock;
        synchronized (object) {
            int length = this.myRecordsTable.getSize(record);
            if (length == 0) {
                return ArrayUtil.EMPTY_BYTE_ARRAY;
            }
            assert (length > 0);
            long address = this.myRecordsTable.getAddress(record);
            byte[] result = new byte[length];
            this.myDataTable.readBytes(address, result);
            return result;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteRecord(int record) {
        assert (record > 0);
        Object object = this.lock;
        synchronized (object) {
            this.myRecordsTable.deleteRecord(record);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void dispose() {
        Object object = this.lock;
        synchronized (object) {
            this.force();
            this.myRecordsTable.dispose();
            this.myDataTable.dispose();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void checkSanity(int record) {
        Object object = this.lock;
        synchronized (object) {
            int size = this.myRecordsTable.getSize(record);
            assert (size >= 0);
            long address = this.myRecordsTable.getAddress(record);
            assert (address >= 0L);
            assert (address + (long)size < this.myDataTable.getFileSize());
        }
    }

    public class AppenderStream
    extends DataOutputStream {
        private final int myRecordId;

        public AppenderStream(int recordId) {
            super(new ByteArrayOutputStream());
            this.myRecordId = recordId;
        }

        @Override
        public void close() throws IOException {
            super.close();
            Storage.this.appendBytes(this.myRecordId, ((ByteArrayOutputStream)this.out).toByteArray());
        }
    }

    public static class StorageDataOutput
    extends DataOutputStream
    implements RecordDataOutput {
        private final Storage myStorage;
        private final int myRecordId;

        public StorageDataOutput(Storage storage, int recordId) {
            super(new ByteArrayOutputStream());
            this.myStorage = storage;
            this.myRecordId = recordId;
        }

        @Override
        public void close() throws IOException {
            super.close();
            this.myStorage.writeBytes(this.myRecordId, ((ByteArrayOutputStream)this.out).toByteArray());
        }

        @Override
        public int getRecordId() {
            return this.myRecordId;
        }
    }
}

