/*******************************************************************************
* Copyright (c) 2000, 2004 IBM Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
*
* Contributors:
*     IBM Corporation - initial API and implementation
*******************************************************************************/
package org.eclipse.wst.rdb.internal.outputview;

import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.FileOutputStream;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.MenuEvent;
import org.eclipse.swt.events.MenuListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.printing.PrintDialog;
import org.eclipse.swt.printing.Printer;
import org.eclipse.swt.printing.PrinterData;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;

/**
 * This class provides a menu and handles the actions that are specific to the
 * messages area of the output view.
 */
public class OVMessageAreaMenu implements MenuListener, SelectionListener
{
   /** Messages are displayed by this component. */
   protected StyledText myTextArea;
   /** Our menu. */
   protected Menu myMenu;
   /** Our copy menu item. */
   protected MenuItem copyItem;
   /** Our select all menu item. */
   protected MenuItem selectAllItem;
   /** Our save menu item. */
   protected MenuItem saveItem;
   /** Our print menu item. */
   protected MenuItem printItem;
   /** Our first separator. */
   protected MenuItem menuSeparator1;
   /** Our second separator. */
   protected MenuItem menuSeparator2;
   /** Our third separator. */
   protected MenuItem menuSeparator3;
   
   /**
    * Constructs an instance of this class.
    * @param aMessageArea The Message area of output view
    */
   public OVMessageAreaMenu(StyledText aMessageArea)
   {
      myTextArea = aMessageArea;
      myMenu = new Menu(myTextArea);
      copyItem = new MenuItem(myMenu, SWT.NONE); // Copy
      copyItem.setText(OutputViewPlugin.getString("OV_MESSAGE_COPY"));
      //copyItem.setAccelerator(SWT.CTRL + 'C');
      menuSeparator1 = new MenuItem(myMenu, SWT.SEPARATOR);// separator
      selectAllItem = new MenuItem(myMenu, SWT.NONE); // Select All
      selectAllItem.setText(OutputViewPlugin.getString("OV_MESSAGE_SELECTALL"));
      menuSeparator2 = new MenuItem(myMenu, SWT.SEPARATOR);// separator
      saveItem = new MenuItem(myMenu, SWT.NONE); // Save Output
      saveItem.setText(OutputViewPlugin.getString("OV_MESSAGE_SAVE_AS"));
      menuSeparator3 = new MenuItem(myMenu, SWT.SEPARATOR);// separator
      printItem = new MenuItem(myMenu, SWT.NONE); // Print
      printItem.setText(OutputViewPlugin.getString("OV_MESSAGE_PRINT"));
      listenAll();
   }
   
   /**
    * Adds listeners for the menu, and its items.
    */
   private void listenAll()
   {
      myMenu.addMenuListener(this);
      copyItem.addSelectionListener(this);
      selectAllItem.addSelectionListener(this);
      saveItem.addSelectionListener(this);
      printItem.addSelectionListener(this);
   }
   
   /**
    * Gets the menu.
    * User of this object must call this method to assign this menu to the Control.
    * @return The Menu object.
    */
   public Menu getMenu()
   {
      return myMenu;
   }
   
   /**
    * Launches file chooser to let user save message to a file.
    */
   private void saveOutput()
   {
      FileDialog fileDialog = new FileDialog(myTextArea.getShell(), SWT.SAVE);
      String fileName = fileDialog.open();
      if (fileName != null)
      {
         try
         {
            File file = new File(fileName);
            if (file.isDirectory())
            {
               displayDirectoryError(myTextArea.getShell(), fileName);
               // return is needed because displayDirectoryError(...) makes
               // a recursive call to this method.
               return;
            }
            if (file.exists())
            {
               // launch question dialog
               Object[] obj = {fileName};
               String message = OutputViewPlugin.getString(
                        "OV_MESSAGE_FILE_EXISTS_DESC", obj);
               MessageBox box = new MessageBox(myTextArea.getShell(),
                        SWT.YES | SWT.NO);
               box.setText(OutputViewPlugin.getString("OV_MESSAGE_FILE_EXISTS_TITLE"));
               box.setMessage(message);
               if (box.open() != SWT.YES)
               {
                  return;
               }
            }
            // Use the encoding specified by the user in Window->Preferences
            // to save the file   
            FileOutputStream fos = new FileOutputStream(file);
            OutputStreamWriter outstream = null;
            String encoding = OutputUtil.getCharacterEncoding();
            if (encoding != null && !encoding.equals(""))
            {
            	outstream = new OutputStreamWriter(fos, encoding);
            }
            else
            {
            	outstream = new OutputStreamWriter(fos);
            }
            BufferedWriter bw = new BufferedWriter(outstream);
            
            //BufferedWriter bw = new BufferedWriter(new FileWriter(file));
            bw.write(myTextArea.getText());
            bw.flush();
            bw.close();
         }
         catch (IOException ex)
         {
            MessageBox box = new MessageBox(myTextArea.getShell(),
                    SWT.ICON_ERROR);
            box.setText(OutputViewPlugin.getString("OV_STATUS_ERROR"));
            box.setMessage(ex.getMessage());
            box.open();
         }        	
      }
   }
   
   /**
    * Displays error message box to let user know that the file that selected
    * is a directory.  This is needed for Linux because the fileChooser allows
    * the directory name as a file.
    * @param aShell the parent shell
    * @param aFile the file name
    */
   private void displayDirectoryError(Shell aShell, String aFile)
   {
      // display error message because a directory was chosen
      Object[] obj = {aFile};
      String message = OutputViewPlugin.getString(
               "OV_MESSAGE_FILE_ISDIRECTORY_DESC", obj);
      MessageBox box = new MessageBox(aShell, SWT.ICON_ERROR | SWT.OK);
      box.setText(OutputViewPlugin.getString("OV_MESSAGE_FILE_ISDIRECTORY_TITLE"));
      box.setMessage(message);
      box.open();
      // reopen save dialog again
      saveOutput();
   }
   
   /**
    * Launches print dialog to print message area of output view
    */
   private void printOutput()
   {
      PrintDialog dialog = new PrintDialog(myTextArea.getShell(), SWT.NONE);
      if (myTextArea.getSelectionCount() > 0)
      {
         dialog.setScope(PrinterData.SELECTION);
      }
      PrinterData printerData = dialog.open();
      if (printerData != null)
      {
         Printer printer = new Printer(printerData);
         myTextArea.print(printer).run();
         printer.dispose();
      }
   }
   
   /**
    * Handes the event when the menu is hidden. Does nothing.
    * @param anEvent The MenuEvent.
    */
   public void menuHidden(MenuEvent anEvent)
   {
      // ignore
   }
   
   /**
    * Handes the event when the menu is shown.
    * @param anEvent The MenuEvent
    */
   public void menuShown(MenuEvent anEvent)
   {
      // enable Copy menuItem only if text is selected
      if (myTextArea.getSelectionCount() > 0)
      {
         copyItem.setEnabled(true);
      }
      else
      {
         copyItem.setEnabled(false);
      }
      
      // disable Select All if all characters are selected
      selectAllItem.setEnabled(!(myTextArea.getSelectionCount() ==
         myTextArea.getCharCount()));
      // disable Save Output if no text in area
      saveItem.setEnabled(myTextArea.getCharCount() > 0);
      
      // disable Print if no text in area
      printItem.setEnabled(myTextArea.getCharCount() > 0);
   }
   
   /**
    * Handes the event when the default selection occurs. Does nothing.
    * @param anEvent The SelectionEvent
    */
   public void widgetDefaultSelected(SelectionEvent anEvent)
   {
      // ignore
   }
   
   /**
    * Handes the event when the the selection of a menuItem occurs.
    * @param anEvent The SelectionEvent
    */
   public void widgetSelected(SelectionEvent anEvent)
   {
      Object source = anEvent.getSource();
      // copy text to clip board
      if (source == copyItem)
      {
         myTextArea.copy();
      }
      // select all text
      else if (source == selectAllItem)
      {
         myTextArea.selectAll();
      }
      // launch file chooser
      else if (source == saveItem)
      {
         saveOutput();
      }
      // launch print dialog
      else if (source == printItem)
      {
         printOutput();
      }
   }
}