/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.internal.core.connection;

import java.io.FileNotFoundException;
import java.sql.Connection;
import java.sql.DriverPropertyInfo;
import java.sql.SQLException;

import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinition;
import org.eclipse.wst.rdb.internal.models.sql.schema.SQLObject;


/**
 * Connection Manager is the central place to handle all JDBC connection information.
 * The singlton instance can be got from DataToolsPlugin. 
 */
public interface ConnectionManager {

    /**
     * This should be used as follow:
     *  <code>
     *   <li>ConnectionManager manager = corePlugin.getConnectionManager();
     *   <li>EAnnotation eAnnotation = target.getEAnnotation(manager.getConnectionEAnnotationUri());
     *   <li>String connectionKey = (String) eAnnotation.getDetails().get(manager.getConnectionEAnnotationKey());
     *   <li>manager.getConnectionInfo(connectionKey);
     *   </code>
     * 
     * @return the Connection URI
     */
    public String getConnectionEAnnotationUri ();
    
    /**
     * @return The Connection URI key
     */
    public String getConnectionEAnnotationKey ();
    
    /**
     * Helper function to add an annotation to keep connection info name
     * @param obj any SQLObject. Typically, it is a database object
     * @param info the connection info associated with
     */
    public void setConnectionInfo(SQLObject obj, ConnectionInfo info);
    
    /**
     * Helper function to add an annotation to keep connection info name
     * @param obj any SQLObject. Typically, it is a database object
     * @param infoName the connection info name associated with
     */
    public void setConnectionInfo(SQLObject obj, String infoName);
    
    /**
     * Helper function to retrieve the connection info name
     * @param obj any SQLObject. Typically, it is a database object
     * @return the connection info name. Null if no such annotation defined
     */
    public String getConnectionInfoName(SQLObject obj);
    
    /**
     * Helper function to retrieve the connection info
     * @param obj any SQLObject. Typically, it is a database object
     * @return the connection info. Null if no such annotation defined
     */
    public ConnectionInfo getConnectionInfo(SQLObject obj);
    
    /**
	 * Retrieve the default connection info based on previous user input.
	 * @param def the target database definition
	 * @return the default connection info
	 */
	public ConnectionInfo getConnectionInfo(DatabaseDefinition def);
	
	/**
	 * Create a named connection info.
	 * @param name connection info name, not null
	 * @return the named connection info, not null
	 */
	public ConnectionInfo createConnectionInfo(DatabaseDefinition def, String name);
	
	/**
	 * Retrieve a named connection info. If it doesn't exist, a null will be returned.
	 * @param name connection info name, not null
	 * @return the named connection info
	 */
	public ConnectionInfo getConnectionInfo(String name);

	/**
	 * 
	 * @return all named connection info, saved and unsaved
	 */
	public ConnectionInfo[] getAllNamedConnectionInfo();
	
	/**
	 * Remove the named connection info from current session and workspace
	 * @param name
	 */
	public void removeConnectionInfo(String name);
	
	/**
	 * Helper functions to retrieve all schema names
	 * @param connection a JDBC connection
	 * @return an array of schema names
	 * @throws SQLException
	 */
	public String[] getSchemaNames(Connection connection) throws SQLException;
	
	/**
	 * Gets information about the possible properties
	 * @param info
	 * @return
	 * @throws SQLException
	 * @throws ClassNotFoundException
	 * @throws IllegalAccessException
	 * @throws InstantiationException
	 */
	public DriverPropertyInfo[] getDriverPropertyInfo(ConnectionInfo info) throws FileNotFoundException, SQLException, ClassNotFoundException, IllegalAccessException, InstantiationException;
	
	public boolean addListener(ConnectionManagerListener listener);
	public boolean removeListener(ConnectionManagerListener listener);	
}
