<?php
/**
 * StatusNet, the distributed open-source microblogging tool
 *
 * Plugin to check submitted notices with blogspam.net
 *
 * PHP version 5
 *
 * LICENCE: This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @category  Plugin
 * @package   StatusNet
 * @author    Evan Prodromou <evan@status.net>
 * @copyright 2009 StatusNet, Inc.
 * @license   http://www.fsf.org/licensing/licenses/agpl-3.0.html GNU Affero General Public License version 3.0
 * @link      http://status.net/
 */

if (!defined('STATUSNET')) {
    exit(1);
}

define('BLOGSPAMNETPLUGIN_VERSION', '0.1');

/**
 * Plugin to check submitted notices with blogspam.net
 *
 * When new notices are saved, we check their text with blogspam.net (or
 * a compatible service).
 *
 * Blogspam.net is supposed to catch blog comment spam, and I found that
 * some of its tests (min/max size, bayesian match) gave a lot of false positives.
 * So, I've turned those tests off by default. This may not get as many
 * hits, but it's better than nothing.
 *
 * @category Plugin
 * @package  StatusNet
 * @author   Evan Prodromou <evan@status.net>
 * @license  http://www.fsf.org/licensing/licenses/agpl-3.0.html GNU Affero General Public License version 3.0
 * @link     http://status.net/
 *
 * @see      Event
 */

class BlogspamNetPlugin extends Plugin
{
    var $baseUrl = 'http://test.blogspam.net:8888/';

    function __construct($url=null)
    {
        parent::__construct();
        if ($url) {
            $this->baseUrl = $url;
        }
    }

    function onStartNoticeSave($notice)
    {
        $args = $this->testArgs($notice);
        common_debug("Blogspamnet args = " . print_r($args, TRUE));
        $request = xmlrpc_encode_request('testComment', array($args));
        $context = stream_context_create(array('http' => array('method' => "POST",
                                                               'header' =>
                                                               "Content-Type: text/xml\r\n".
                                                               "User-Agent: " . $this->userAgent(),
                                                               'content' => $request)));
        $file = file_get_contents($this->baseUrl, false, $context);
        $response = xmlrpc_decode($file);
        if (xmlrpc_is_fault($response)) {
            throw new ServerException("$response[faultString] ($response[faultCode])", 500);
        } else {
            common_debug("Blogspamnet results = " . $response);
            if (preg_match('/^ERROR(:(.*))?$/', $response, $match)) {
                throw new ServerException(sprintf(_("Error from %s: %s"), $this->baseUrl, $match[2]), 500);
            } else if (preg_match('/^SPAM(:(.*))?$/', $response, $match)) {
                throw new ClientException(sprintf(_("Spam checker results: %s"), $match[2]), 400);
            } else if (preg_match('/^OK$/', $response)) {
                // don't do anything
            } else {
                throw new ServerException(sprintf(_("Unexpected response from %s: %s"), $this->baseUrl, $response), 500);
            }
        }
        return true;
    }

    function testArgs($notice)
    {
        $args = array();
        $args['comment'] = $notice->content;
        $args['ip'] = $this->getClientIP();

        if (isset($_SERVER) && array_key_exists('HTTP_USER_AGENT', $_SERVER)) {
            $args['agent'] = $_SERVER['HTTP_USER_AGENT'];
        }

        $profile = $notice->getProfile();

        if ($profile && $profile->homepage) {
            $args['link'] = $profile->homepage;
        }

        if ($profile && $profile->fullname) {
            $args['name'] = $profile->fullname;
        } else {
            $args['name'] = $profile->nickname;
        }

        $args['site'] = common_root_url();
        $args['version'] = $this->userAgent();

        $args['options'] = "max-size=140,min-size=0,min-words=0,exclude=bayasian";

        return $args;
    }

    function getClientIP()
    {
        if (isset($_SERVER) && array_key_exists('REQUEST_METHOD', $_SERVER)) {
            // Note: order matters here; use proxy-forwarded stuff first
            foreach (array('HTTP_X_FORWARDED_FOR', 'CLIENT-IP', 'REMOTE_ADDR') as $k) {
                if (isset($_SERVER[$k])) {
                    return $_SERVER[$k];
                }
            }
        }
        return '127.0.0.1';
    }

    function userAgent()
    {
        return 'BlogspamNetPlugin/'.BLOGSPAMNETPLUGIN_VERSION . ' StatusNet/' . STATUSNET_VERSION;
    }
}
