# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.
#
# Authors: Olivier Tilloy <olivier@fluendo.com>
#          Benjamin Kampmann <benjamin@fluendo.com>

"""
Metadata capabilities are a specific type of capability that expose metadata
retrieval facilities.
"""

from elisa.core.components.capability import Capability

from twisted.internet import defer


class MetadataCapability(Capability):

    """
    A metadata capability is a special type of capability that provides
    metadata retrieval services on top of a given resource provider.

    For example, this could be a music album cover retriever built on top of
    the Amazon resource provider.

    A metadata capability has a rank that determines the level of confidence
    the metadata manager should give it when retrieving metadata. It allows to
    give a capability a higher priority over another one.

    @cvar rank: the rank of the capability for the metadata manager
    @type rank: positive C{int}
    """

    # Defaults to 0 (lowest priority)
    # Set me when implementing a metadata capability
    rank = 0

    def able_to_handle(self, model):
        """
        Test whether the capability is able to handle a metadata request on the
        given model.

        @param model: the model for which metadata is requested
        @type model:  a subclass of L{elisa.core.components.model.Model}

        @return: True if the capability is able to handle the given model,
                 False otherwise
        @rtype:  C{bool}
        """
        return False

    def get_metadata(self, model):
        """
        Try to retrieve metadata and populate the model with it.

        @param model: the model that should be populated with the metadata
        @type model:  a subclass of L{elisa.core.components.model.Model}

        @return:      a deferred fired when the metadata retrieval is complete
        @rtype:       L{twisted.internet.defer.Deferred}
        """
        return defer.fail(NotImplementedError())
