/*
 * Copyright 2005 - 2009  Zarafa B.V.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License, version 3, 
 * as published by the Free Software Foundation with the following additional 
 * term according to sec. 7:
 *  
 * According to sec. 7 of the GNU Affero General Public License, version
 * 3, the terms of the AGPL are supplemented with the following terms:
 * 
 * "Zarafa" is a registered trademark of Zarafa B.V. The licensing of
 * the Program under the AGPL does not imply a trademark license.
 * Therefore any rights, title and interest in our trademarks remain
 * entirely with us.
 * 
 * However, if you propagate an unmodified version of the Program you are
 * allowed to use the term "Zarafa" to indicate that you distribute the
 * Program. Furthermore you may use our trademarks where it is necessary
 * to indicate the intended purpose of a product or service provided you
 * use it in accordance with honest practices in industrial or commercial
 * matters.  If you want to propagate modified versions of the Program
 * under the name "Zarafa" or "Zarafa Server", you may only do so if you
 * have a written permission by Zarafa B.V. (to acquire a permission
 * please contact Zarafa at trademark@zarafa.com).
 * 
 * The interactive user interface of the software displays an attribution
 * notice containing the term "Zarafa" and/or the logo of Zarafa.
 * Interactive user interfaces of unmodified and modified versions must
 * display Appropriate Legal Notices according to sec. 5 of the GNU
 * Affero General Public License, version 3, when you propagate
 * unmodified or modified versions of the Program. In accordance with
 * sec. 7 b) of the GNU Affero General Public License, version 3, these
 * Appropriate Legal Notices must retain the logo of Zarafa or display
 * the words "Initial Development by Zarafa" if the display of the logo
 * is not reasonably feasible for technical reasons."
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

/**
* This class is managing keyboard- and mouse input from the user and sending these
* events to the correct module by keeping track of the "focus"
*/
function InputManager()
{
	this.objects = new Object();
	this.objectCount = 0;
	this.focusId = false;
	this.oldFocusId = false;

	this.bindings = {
					"mouseup": new Array(), 
					"mousedown": new Array(), 
					"mousemove": new Array(), 
					"contextmenu": new Array(), 
					"keydown": new Array(), 
					"keyup": new Array(),
					"focus": new Array(), // focus & blur events are generated by the inputmanager!
					"blur" : new Array()
					};
}


InputManager.prototype.init = function()
{
	// register several events we want to handle
	dhtml.addEvent(this, document.body, "mouseup", eventInputManager);
	dhtml.addEvent(this, document.body, "mousedown", eventInputManager);
	dhtml.addEvent(this, document.body, "mousemove", eventInputManager);
	dhtml.addEvent(this, document.body, "contextmenu", eventInputManager);
	dhtml.addEvent(this, document, "keydown", eventInputManager);
	dhtml.addEvent(this, document, "keyup", eventInputManager);

	// register the webclient object, which can be used by anyone who want to bind an event but didn't have a object.
	this.addObject(webclient, dhtml.getElementById("main"));
	this.defaultId = this.getObjectId(webclient);

	// add focus handling
	dhtml.addEvent(this, document, "focus", eventInputManagerFocusControl);
	dhtml.addEvent(this, document, "blur", eventInputManagerFocusControl);
	dhtml.addEvent(this, document.body, "mousedown", eventInputManagerFocusControl)

	this.resetTimeout();
}

InputManager.prototype.resetTimeout = function()
{
	if (this.timeoutTimer){
		window.clearTimeout(this.timeoutTimer);
		this.timeoutTimer = false;
	}
	
	if (typeof webclient != "undefined"){
		var timer = webclient.settings.get("global/auto_logout", 0);
		if (timer>0){
			this.timeoutTimer = window.setTimeout(eventInputTimerLogout, timer);
		}
	}
}

/**
* Use this function to register your module, the focusElement is the main container of the module
*/
InputManager.prototype.addObject = function(newObject, focusElement)
{
	if (!focusElement) {
		focusElement = false;
	}
	this.objects[this.objectCount++] = {"object":newObject,"element":focusElement};
}

InputManager.prototype.removeObject = function(object)
{
	var objectId = this.getObjectId(object);
	
	// first remove all events for this object
	for(var eventType in this.bindings){
		for(var i=0; i<this.bindings[eventType].length; i++){
			var binding = this.bindings[eventType][i];
			if (binding && binding.objectId == objectId){
				this.bindings[eventType].splice(i,1);
				i--;
			}
		}
	}

	// next delete the object itself
	delete this.objects[objectId];
}

/*
* Checks if the given module is selected
*/
InputManager.prototype.hasFocus = function(object)
{
	var result = false;
	if (this.focusId!=false)
		result = (object==this.objects[this.focusId]["object"]);
	return result;
}

/*
* Internal function to keep track of focus
*/
InputManager.prototype.handleFocus = function(event)
{
	this.resetTimeout();

	switch(event.type)
	{
		case "blur": // other window/iframe
			this.oldFocusId = this.focusId;
			this.focusId = false;
			break;
		case "focus":
			this.focusId = this.oldFocusId;
			break;
		case "mousedown": // manual focus
			
			var objectId = false;
			var element = event.target;
			
			while(element!=null){
				objectId = this.getObjectIdByElement(element);
				if (objectId===false){
					element = element.parentNode;
				}else{
					break;
				}
			}

			if (this.focusId === false){
				window.focus();
			}
			
			// send blur event to focusId
			this.handleEvent(false, {"type":"blur"});

			this.focusId = objectId;

			// send focus event to focusId
			this.handleEvent(false, {"type":"focus"});

			break;
	}
	return true;
}


InputManager.prototype.getObjectIdByElement = function(element)
{
	for(var i in this.objects){
		if (this.objects[i]["element"] == element){
			return i;
		}
	}
	return false;
}

InputManager.prototype.getObjectId = function(object)
{
	for(var i in this.objects){
		if (this.objects[i]["object"] == object){
			return i;
		}
	}
	return false;
}

/**
* After you registered your module, you can use this function to bind some events to your module
*/
InputManager.prototype.bindEvent = function(object, eventType, handler)
{
	if (typeof this.bindings[eventType] == "undefined"){
		return false;
	}

	var binding = new Object();
	binding.objectId = this.getObjectId(object);
	binding.handler = handler;

	this.bindings[eventType].push(binding);
	return true;
}

InputManager.prototype.unbindEvent = function(object, eventType, handler)
{
	var objectId = this.getObjectId(object);
	for(var i in this.bindings[eventType]){
		var binding = this.bindings[eventType][i];
		if (binding && binding.objectId && binding.objectId == objectId && binding.handler == handler){
			this.bindings[eventType].splice(i,1);
			return true;
		}
	}
	return false;
}

InputManager.prototype.handleEvent = function(element, event)
{
	this.resetTimeout();
	var returnValue = true
	for(var i in this.bindings[event.type]) {
		var binding = this.bindings[event.type][i];
		if (binding && binding.objectId && (binding.objectId == this.defaultId || binding.objectId == this.focusId)){
			if (binding.handler(this.objects[binding.objectId]["object"], element, event) === false)
				returnValue = false;
		}
	}
	return returnValue;
}

/**
 * Function which tells whether event has 
 * already been binded or not.
 * @param object object 	module object
 * @param string eventType 	type of event
 * @param object handler 	event handler
 * @return boolean 			true if event is already binded, false if not binded
 */
InputManager.prototype.hasEvent = function(object, eventType, handler)
{
	var objectId = this.getObjectId(object);
	for(var i in this.bindings[eventType]){
		var binding = this.bindings[eventType][i];
		if (binding && binding.objectId && binding.objectId == objectId && binding.handler == handler){
			return true;
		}
	}
	return false;
}

function eventInputManagerFocusControl(inputmanager, element, event)
{
	return inputmanager.handleFocus(event);
}

function eventInputManager(inputmanager, element, event)
{
	return inputmanager.handleEvent(element, event);
}

function eventInputTimerLogout()
{
	document.location = "index.php?logout=auto";
}

