# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.
#
# Author: Gernot Klimscha <gernot@fluendo.com>

from elisa.core.media_uri import MediaUri
import os, platform
from twisted.trial.unittest import TestCase, SkipTest

# import the resource to be tested
from elisa.plugins.ipod.ipod_resource import IpodResource
# import the models
from elisa.plugins.base.models.audio import AlbumModel, TrackModel, ArtistModel
from elisa.plugins.ipod.models import IpodTrackModel, IpodSongModel, \
                                      IpodTrackListModel, IpodSongListModel, \
                                      IpodAlbumListModel, IpodArtistListModel

class TestIpodResource(TestCase):
    tests_dir = os.path.dirname(__file__)

    def setUp(self):
        def created(resource):
            self.resource = resource

            self.itunesdb = os.path.join(self.tests_dir, "data", "iTunesDB")
            self._ipod = MediaUri("ipod:///%s?action=tracks" % self.itunesdb)

        d = IpodResource.create({})
        d.addCallback(created)
        return d

    def tearDown(self):
        return self.resource.clean()

    def test_non_implemented(self):
        """
        test what happens if a non implemented feature is requested
        """
        model = None
        self._ipod.set_param('action', 'photos')
        model, dfr = self.resource.get(self._ipod, model)
        self.assertFailure(dfr, NotImplementedError)
        return dfr

    def test_artist_requests(self):
        """
        test if artist request works
        """
        def response(model):
            self.assertEquals(len(model), 43)
            self.assertEquals(model.artists[1].name, 'Archive')

        model = None
        model, dfr = self.resource.get(self._ipod, model)
        return dfr.addCallback(response)

    def test_album_requests(self):
        """
        test if album request works
        """
        def response(model):
            self.assertEquals(len(model), 4)
            self.assertEquals(model.albums[1].album, u'Take My Head')

        model = None
        self._ipod.set_param('search', 'albums')
        self._ipod.set_param('artist', 'Archive')
        model, dfr = self.resource.get(self._ipod, model)
        return dfr.addCallback(response)

    def test_song_requests(self):
        """
        test if song request works
        """
        def response(model):
            self.assertEquals(len(model), 44)
            self.assertEquals(model.songs[1].title, u'sit_back_down')

        model = None
        self._ipod.set_param('search', 'songs')
        self._ipod.set_param('artist', 'Archive')
        model, dfr = self.resource.get(self._ipod, model)
        return dfr.addCallback(response)


    def test_all_artists_requests(self):
        """
        test if all_artits filter request works
        """
        def response(model):
            self.failIf(not isinstance(model, IpodArtistListModel))

        model = None
        self._ipod.set_param('search', 'artists')
        model, dfr = self.resource.get(self._ipod, model)
        return dfr.addCallback(response)


    def test_all_albums_requests(self):
        """
        test if all_albums filter request works
        """
        def response(model):
            self.assertEquals(len(model), 59)
            self.assertEquals(model.albums[5].album, u'Dies Irae [CD1]')

        model = None
        self._ipod.set_param('search', 'albums')
        model, dfr = self.resource.get(self._ipod, model)
        return dfr.addCallback(response)

    def test_all_song_requests(self):
        """
        test if all_songs filter request works
        """
        def response(model):
            self.assertEquals(len(model), 663)
            self.assertEquals(model.songs[25].title, u'Your Legs Grow')

        model = None
        self._ipod.set_param('search', 'songs')
        model, dfr = self.resource.get(self._ipod, model)
        return dfr.addCallback(response)

    def test_filter_songs_for_album_requests(self):
        """
        test if songs_for_album filter request works
        """
        def response(model):
            self.assertEquals(len(model), 10)
            self.assertEquals(model.songs[9].title, u'Rest My Head on You')

        model = None
        self._ipod.set_param('search', 'songs')
        self._ipod.set_param('album', 'Take My Head')
        model, dfr = self.resource.get(self._ipod, model)
        return dfr.addCallback(response)

    def test_return_data(self):
        """
        test if returning of already parsed data works
        """
        def p(model):
            self.assertEquals(len(model), 1)
            self.assertEquals(model.artists[0].name, u'Nightwish')

        model = None
        db = None
        track_list = IpodTrackListModel()
        track_list.tracks.append(IpodTrackModel(u'Nightwish', 'Album'))
        if platform.system() == 'Windows':
            fake_path = "fake_path"
        else:
            fake_path = "/fake_path"
        self.resource.dbs = {fake_path : (db, track_list)}
        fake_uri = MediaUri("ipod:///fake_path?action=tracks")
        model, dfr = self.resource.get(fake_uri, model)
        return dfr.addCallback(p)

