/**
* Prototyping the default Date object
*
*@author Michael Erkens <m.erkens@connectux.com>
*/


/**
 * Function which returns the number of days in a month.
 * If month is given, use this month instead of this.getMonth()
 * @return integer number of days in a month 
 */ 
Date.prototype.getDaysInMonth = function(month, year)
{
	if (typeof month == "undefined"){
		month = this.getMonth();
	}
	if (typeof year == "undefined"){
		year = this.getFullYear();
	}

	var DAYS_IN_MONTH = new Array(31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31);

	var days = 0;
	
	if(DAYS_IN_MONTH[month]) {
		days = DAYS_IN_MONTH[month];
		
		// Month is 'February', check is this year is a leap year.
		if(month == 1) {
			if(this.isLeapYear(year)) {
				days++;
			}
		}
	}
	
	return days;
}


/**
 * Function will add one or more days to current date.
 */
Date.prototype.addDays = function(dayCount)
{
	if (!dayCount) {
		dayCount = 1;
	}
	
	var oldOffset = this.getTimezoneOffset();
	this.setTime(this.getTime()+(ONE_DAY*dayCount));
	var newOffset = this.getTimezoneOffset();
	
	this.setTime(this.getTime()+((newOffset-oldOffset)*60000));
} 

/**
 * Function which verifies if the given year is a leap year.
 * @return boolean true - year is leap year, false - year is no leap year 
 */ 
Date.prototype.isLeapYear = function(year)
{
	if (typeof year == "undefined"){
		year = this.getFullYear();
	}

	isLeapYear = false;
	
	if((year % 4) == 0 || (year % 400) == 0) {
		isLeapYear = true;
	}
	
	return isLeapYear;
}

/**
 * Returns the number of the week in year, as defined in ISO 8601. 
 */
Date.prototype.getWeekNumber = function(){
	var d = new Date(this.getFullYear(), this.getMonth(), this.getDate(), 0, 0, 0);
	var DoW = d.getDay();
	d.setDate(d.getDate() - (DoW + 6) % 7 + 3); // Nearest Thu
	var ms = d.valueOf(); // GMT
	d.setMonth(0);
	d.setDate(4); // Thu in Week 1
	return Math.round((ms - d.valueOf()) / (7 * 864e5)) + 1;
}

/**
 * Get the week number of the month (1 to 5)
 */
 
Date.prototype.getWeekNrOfMonth = function()
{
	var week = this.getWeekNumber();
	var monthstart = this.getTime() - (this.getDate()-1) * 24 * 60 * 60 * 1000;
	var monthDate = new Date(monthstart);
	var monthstartweek = monthDate.getWeekNumber();
	return week - monthstartweek + 1; 
}
 

/**
 * Helper function to convert a year to a full year
 */
Date.prototype.y2k = function(year) 
{ 
	return (year < 1000) ? year + 1900 : year;
}

/**
 * Function which sets the day, month and year. It sets the hours, minutes,
 * seconds and miliseconds to 0 (zero). 
 * @param integer day the day
 * @param integer month the month
 * @param integer year the full year (2006)
 * @return integer time stamp  
 */  
Date.prototype.setTimeStamp = function(day, month, year)
{
	this.setTime(new Date(year, month-1, day).getTime());

	return this.getTime();
}

/**
 * Function will return the time as "07:05"
 * @return string 
 */
Date.prototype.toTime = function()
{
	var hours = this.getHours()
	var minutes = this.getMinutes();
	return (hours<10?"0":"")+hours+":"+(minutes<10?"0":"")+minutes; 
}
/**
 * Function will return the date as "28-02-2006"
 * @return string 
 */ 
Date.prototype.toDate = function()
{
	var date = this.getDate();
	var month = this.getMonth()+1;//month start at 0
	var year = this.getFullYear();
	return (date<10?"0":"")+date+"-"+(month<10?"0":"")+month+"-"+year;
}

Date.prototype.toDateTime = function()
{
	return this.toDate()+" "+this.toTime();
}

/**
 * Function will return the first day of a week
 * @return Date
 * 
 * TODO: week can also start on sunday, check settings
 * TODO: check DST
 */	 
Date.prototype.getStartDateOfWeek = function()
{
	var day = this.getDay();
	if (day == 0){
		day = 7;
	}
	return new Date (this-((day-1)*ONE_DAY));//Monday
	//return new Date (this-(day*ONE_DAY));//Sunday
}

/**
 * Function will check "day" == same day as this object
 * @param date
 * @return Bool   
 */ 
Date.prototype.isSameDay = function(day)
{
	return 	this.getDate() == day.getDate() && 
			this.getMonth() == day.getMonth() && 
			this.getYear() == day.getYear();
}

/**
 * Function will convert "14:13" -> "14:30" or "14:32 " -> "15:00"
 * @return int: time in milliseconds 
 */
Date.prototype.ceilHalfhour = function()
{
	var minutes = parseInt(this.getMinutes(),10);
	var seconds = parseInt(this.getSeconds(),10);
	var oldValue = this.getTime();
	var result = this.getTime();
	if(minutes != 0 && minutes < 30){
		result = oldValue+((30-minutes)*60000);
	}
	if(minutes > 30 && minutes < 60){
		result = oldValue+((60-minutes)*60000);
	}
	return result-(seconds*1000);
}
/**
 * Function will convert "14:13" -> "14:00" or "14:32 " -> "14:30"
 * @return int: time in milliseconds 
 */
Date.prototype.floorHalfhour = function()
{
	var minutes = parseInt(this.getMinutes(),10);
	var seconds = parseInt(this.getSeconds(),10);
	var oldValue = this.getTime();
	var result = this.getTime();
	if(minutes < 30){
		result = oldValue-(minutes*60000);
	}
	if(minutes > 30 && minutes < 60){
		result = oldValue-((minutes-30)*60000);
	}
	return result-(seconds*1000);
}

Date.prototype.getMonthTranslation = function()
{
	return MONTHS[this.getMonth()];
}
Date.prototype.getMonthShortTranslation = function()
{
	return MONTHS_SHORT[this.getMonth()];
}

Date.prototype.getDayTranslation = function()
{
	return DAYS[this.getDay()];
}
Date.prototype.getDayShortTranslation = function()
{
	return DAYS_SHORT[this.getDay()];
}

/**
 * Function will set the time of timestamp on zero "12-jan-06 14:12" -> "12-jan-06 0:0"
 * @return int: unixtimestamp   
 */
function timeToZero(unixtimestamp)
{
	var tempDate = new Date(unixtimestamp*1000);
	tempDate.setHours(0);
	tempDate.setMinutes(0);
	tempDate.setSeconds(0);
	return parseInt(Math.floor(tempDate.getTime()/1000),10);
}

/**
 * Function will convert time "1:40" to seconds "6000" 
 */
function timeToSeconds(value)
{
	value = value.split(":");
	var hour = parseInt(value[0], 10);
	var min = parseInt(value[1], 10);
	return ((hour*60)+min)*60;
}

/**
 * Function will convert seconds "6000" to time "1:40"
 * @param value = int time in seconds
 * @param fourDigit = boolean true for "01:40" or false for "1:40"  
 */
function secondsToTime(value,fourDigit)
{
	var minutes = parseInt(value/60);
	var hours = parseInt(minutes/60);
	var result;
	
	minutes = minutes - (hours*60);	
	
	if(fourDigit == false){
		result = hours+":"+minutes;
	}
	else{
		result = (hours<10?"0":"")+hours+":"+(minutes<10?"0":"")+minutes;
	}
	
	return result
}

/**
 * Date.strftime() 
 * string format ( string format )
 * Formatting rules according to http://php.net/strftime
 *
 * Copyright (C) 2006  Dao Gottwald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * Contact information:
 *   Dao Gottwald  <dao at design-noir.de>
 *
 * Changed by M. Erkens for use with Zarafa WebAccess
 *  - use translation from Zarafa WebAccess
 *
 * @version  0.7 (changed)
 * @url      http://design-noir.de/webdev/JS/Date.format/
 */
Date._pad = function(num, len) {
	for (var i = 1; i <= len; i++)
		if (num < Math.pow(10, i))
			return new Array(len-i+1).join(0) + num;
	return num;
}

Date.prototype.strftime = function(format) {
	if (format.indexOf('%%') > -1) { // a literal `%' character
		format = format.split('%%');
		for (var i = 0; i < format.length; i++)
			format[i] = this.format(format[i]);
		return format.join('%');
	}
	format = format.replace(/%D/g, '%m/%d/%y'); // same as %m/%d/%y
	format = format.replace(/%r/g, '%I:%M:%S %p'); // time in a.m. and p.m. notation
	format = format.replace(/%R/g, '%H:%M:%S'); // time in 24 hour notation
	format = format.replace(/%T/g, '%H:%M:%S'); // current time, equal to %H:%M:%S
	format = format.replace(/%x/g, _("%d-%m-%Y")); // preferred date representation for the current locale without the time
	format = format.replace(/%X/g, _("%H:%M")); // preferred time representation for the current locale without the date
	var dateObj = this;
	return format.replace(/%([aAbhBcCdegGHIjmMnpStuUVWwyYzZ])/g, function(match0, match1) {
		return dateObj.format_callback(match0, match1);
	});
}

Date.prototype.format_callback = function(match0, match1) {
	switch (match1) {
		case 'a': // abbreviated weekday name according to the current locale
			return DAYS_SHORT[this.getDay()];
		case 'A': // full weekday name according to the current locale
			return DAYS[this.getDay()];
		case 'b':
		case 'h': // abbreviated month name according to the current locale
			return MONTHS_SHORT[this.getMonth()];
		case 'B': // full month name according to the current locale
			return MONTHS[this.getMonth()];
		case 'c': // preferred date and time representation for the current locale
			return this.toLocaleString();
		case 'C': // century number (the year divided by 100 and truncated to an integer, range 00 to 99)
			return Math.floor(this.getFullYear() / 100);
		case 'd': // day of the month as a decimal number (range 01 to 31)
			return Date._pad(this.getDate(), 2);
		case 'e': // day of the month as a decimal number, a single digit is preceded by a space (range ' 1' to '31')
			return Date._pad(this.getDate(), 2);
		/*case 'g': // like %G, but without the century
			return ;
		case 'G': // The 4-digit year corresponding to the ISO week number (see %V). This has the same format and value as %Y, except that if the ISO week number belongs to the previous or next year, that year is used instead
			return ;*/
		case 'H': // hour as a decimal number using a 24-hour clock (range 00 to 23)
			return Date._pad(this.getHours(), 2);
		case 'I': // hour as a decimal number using a 12-hour clock (range 01 to 12)
			var hour = Date._pad(this.getHours() % 12, 2);
			return (hour != "00")?hour:"12";
		case 'j': // day of the year as a decimal number (range 001 to 366)
			return Date._pad(this.getMonth() * 30 + Math.ceil(this.getMonth() / 2) + this.getDay() - 2 * (this.getMonth() > 1) + (!(this.getFullYear() % 400) || (!(this.getFullYear() % 4) && this.getFullYear() % 100)), 3);
		case 'm': // month as a decimal number (range 01 to 12)
			return Date._pad(this.getMonth() + 1, 2);
		case 'M': // minute as a decimal number
			return Date._pad(this.getMinutes(), 2);
		case 'n': // newline character
			return '\n';
		case 'p': // either `am' or `pm' according to the given time value, or the corresponding strings for the current locale
			return this.getHours() < 12 ? 'am' : 'pm';
		case 'S': // second as a decimal number
			return Date._pad(this.getSeconds(), 2);
		case 't': // tab character
			return '\t';
		case 'u': // weekday as a decimal number [1,7], with 1 representing Monday
			return this.getDay() || 7;
		/*case 'U': // week number of the current year as a decimal number, starting with the first Sunday as the first day of the first week
			return ;
		case 'V': // The ISO 8601:1988 week number of the current year as a decimal number, range 01 to 53, where week 1 is the first week that has at least 4 days in the current year, and with Monday as the first day of the week. (Use %G or %g for the year component that corresponds to the week number for the specified timestamp.)
			return ;
		case 'W': // week number of the current year as a decimal number, starting with the first Monday as the first day of the first week
			return ;*/
		case 'w': // day of the week as a decimal, Sunday being 0
			return this.getDay();
		case 'y': // year as a decimal number without a century (range 00 to 99)
			return this.getFullYear().toString().substr(2);
		case 'Y': // year as a decimal number including the century
			return this.getFullYear();
		/*case 'z':
		case 'Z': // time zone or name or abbreviation
			return ;*/
		default:
			return match0;
	}
}

Date.prototype.strftime_gmt = function(format)
{
	var timestamp = this.getTime();
	timestamp += this.getTimezoneOffset()*60000;
	
	var gmtDate = new Date(timestamp);
	return gmtDate.strftime(format);
}

/**
* returns a string with the difference between two dates in a simple form
*
* otherDate: the second date object, to which 'this' is compared
* futureString: string to be used when otherDate is in the future compared to this
* pastString: string to be used when pastString is in the past compared to this
*
* Note for futureString and pastString: When these are set and not an empty string they are used as described.
*     when they contain '%s' this is replaced by the difference in time.
*
* Example:
*   alert(date1.simpleDiffString(date2, "", "%s overdue"));
*/
Date.prototype.simpleDiffString = function(otherDate, futureString, pastString)
{
	var result = false;

	var time1 = this.getTime();
	var time2 = otherDate.getTime();

	var diff = time1 - time2;
	
	var shellString = "%s";
	if (diff>=0 && typeof(futureString)=="string" && futureString!=""){
		shellString = futureString;
	}else if (diff<0 && typeof(pastString)=="string" && pastString!=""){
		shellString = pastString;
	}
	
	// get diff in minutes
	diff = Math.round(Math.abs(diff)/60000);
	if (diff<=1)
		result = _("now");
	if (!result && diff<59)
		result = _("%d minutes").sprintf(diff);

	// now convert diff in hours
	diff = Math.round(diff/60);
	if (!result && diff<=1)
		result = _("1 hour");
	if (!result && diff<=23)
		result = _("%d hours").sprintf(diff);

	// diff in days
	diff = Math.round(diff/24);
	if (!result && diff<=1)
		result = _("1 day");
	if (!result && diff<=13)
		result = _("%d days").sprintf(diff);
	
	// diff in weeks
	diff = Math.floor(diff/7);
	if (!result && diff<=1)
		result = _("1 week");
	if (!result)
		result = _("%d weeks").sprintf(diff);
	
	return shellString.sprintf(result);
}

function strftime(format, timestamp) 
{
	var t = new Date();
	if (typeof(timestamp) != "undefined")
		t.setTime(parseInt(timestamp)*1000);

	return t.strftime(format);
}

function strftime_gmt(format,timestamp) {
	var t = new Date();
	if (typeof(timestamp) != "undefined")
		t.setTime(parseInt(timestamp)*1000);

	return t.strftime_gmt(format);
}
