<?php
/*
 * Copyright (C) 2005 - 2008  Zarafa B.V.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License, version 3, 
 * as published by the Free Software Foundation with the following additional 
 * term according to sec. 7:
 * 
 * "Zarafa" is a registered trademark of Zarafa B.V. The licensing of the Program
 * under the AGPL does not imply a trademark license. Therefore any rights,
 * title and interest in our trademarks remain entirely with us.
 * However, if you propagate an unmodified version of the Program you are 
 * required to use the term "Zarafa" to indicate that you distribute the 
 * Program. Furthermore you may use our trademarks where it is necessary to 
 * indicate the intended purpose of a product or service provided you use it in 
 * accordance with honest practices in industrial or commercial matters.
 * If you want to propagate modified versions of the Program under the name
 * "Zarafa" or "Zarafa Server", you may only do so if you have a written
 * permission by Zarafa B.V. (to acquire a permission please contact Zarafa at
 * trademark@zarafa.com).
 * The user interface of the software displays a attribution notice containing
 * the term "Zarafa" and/or the Logo of Zarafa. You have to preserve these
 * attribution notices when you propagate unmodified or modified versions of
 * the Program.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

?>
<?

class Meetingrequest {
	/*
	 * NOTE
	 *
	 * This class is designed to modify and update meeting request properties
	 * and to search for linked appointments in the calendar. It does not
	 * - set standard properties like subject or location
	 * - commit property changes through savechanges() (except in accept() and decline())
	 *
	 * To set all the other properties, just handle the item as any other appointment
	 * item. You aren't even required to set those properties before or after using
	 * this class. If you update properties before REsending a meeting request (ie with
	 * a time change) you MUST first call updateMeetingRequest() so the internal counters
	 * can be updated. You can then submit the message any way you like.
	 *
	 */
	 
	/*
	 * How to use
	 * ----------
	 *
	 * Sending a meeting request:
	 * - Create appointment item as normal, but as 'tentative'
	 *   (this is the state of the item when the receiving user has received but
	 *    not accepted the item)
	 * - Set recipients as normally in e-mails
	 * - Create Meetingrequest class instance
	 * - Call setMeetingRequest(), this turns on all the meeting request properties in the
	 *   calendar item
	 * - Call sendMeetingRequest(), this sends a copy of the item with some extra properties
	 *
	 * Updating a meeting request:
	 * - Create Meetingrequest class instance
	 * - Call updateMeetingRequest(), this updates the counters
	 * - Call sendMeetingRequest()
	 *
	 * Clicking on a an e-mail:
	 * - Create Meetingrequest class instance
	 * - Check isMeetingRequest(), if true:
	 *   - Check isLocalOrganiser(), if true then ignore the message
	 *   - Check isInCalendar(), if not call doAccept(true, false, false). This adds the item in your 
	 *     calendar as tentative without sending a response
	 *   - Show Accept, Tentative, Decline buttons
	 *   - When the user presses Accept, Tentative or Decline, call doAccept(false, true, true), 
	 *     doAccept(true, true, true) or doDecline(true) respectively to really accept or decline and
	 *     send the response. This will remove the request from your inbox.
	 * - Check isMeetingRequestResponse, if true:
	 *   - Check isLocalOrganiser(), if not true then ignore the message
	 *   - Call processMeetingRequestResponse()
	 *     This will update the trackstatus of all recipients, and set the item to 'busy'
	 *     when all the recipients have accepted.
	 * - Check isMeetingCancellation(), if true:
	 *   - Check isLocalOrganiser(), if true then ignore the message
	 *   - Check isInCalendar(), if not, then ignore
	 *     Call processMeetingCancellation()
	 *   - Show 'Remove item' button to user
	 *   - When userpresses button, call doCancel(), which removes the item from your 
	 *     calendar and deletes the message
	 */
	 
	// All properties for a recipient that are interesting
	var $recipprops = Array(PR_ENTRYID, PR_DISPLAY_NAME, PR_EMAIL_ADDRESS, PR_RECIPIENT_ENTRYID, PR_RECIPIENT_TYPE, PR_SEND_INTERNET_ENCODING, PR_SEND_RICH_INFO, PR_RECIPIENT_DISPLAY_NAME, PR_ADDRTYPE, PR_DISPLAY_TYPE, PR_RECIPIENT_TRACKSTATUS, PR_RECIPIENT_FLAGS, PR_ROWID);
	 
	/**
	 * Constructor
	 *
	 * Takes a store and a message. The message is an appointment item
	 * that should be converted into a meeting request or an incoming
	 * e-mail message that is a meeting request.
	 *
	 * The $session variable is optional, but required if the following features
	 * are to be used:
	 *
	 * - Sending meeting requests for meetings that are not in your own store
	 * - Sending meeting requests to resources, resource availability checking and resource freebusy updates
	 */
	 
	function Meetingrequest($store, $message, $session = false) 
	{
		$this->store = $store;
		$this->message = $message;
		$this->session = $session;

		$properties["goid"] = "PT_BINARY:PSETID_Meeting:0x3";
		$properties["goid2"] = "PT_BINARY:PSETID_Meeting:0x23";
		$properties["type"] = "PT_STRING8:PSETID_Meeting:0x24";
		$properties["meetingrecurring"] = "PT_BOOLEAN:PSETID_Meeting:0x5";
		$properties["unknown2"] = "PT_BOOLEAN:PSETID_Meeting:0xa";
		$properties["unknown3"] = "PT_SYSTIME:PSETID_Meeting:0x1";
		$properties["unknown4"] = "PT_SYSTIME:PSETID_Meeting:0x1a";
		$properties["meetingstatus"] = "PT_LONG:PSETID_Appointment:0x8217";
		$properties["responsestatus"] = "PT_LONG:PSETID_Appointment:0x8218";
		$properties["unknown6"] = "PT_LONG:PSETID_Meeting:0x4";
		$properties["replytime"] = "PT_LONG:PSETID_Appointment:0x8220";
		$properties["usetnef"] = "PT_BOOLEAN:PSETID_Common:0x8582";
		$properties["recurrence_data"] = "PT_BINARY:PSETID_Appointment:0x8216";
		$properties["reminderminutes"] = "PT_LONG:PSETID_Common:0x8501";
		$properties["reminderset"] = "PT_BOOLEAN:PSETID_Common:0x8503";
		$properties["sendasical"] = "PT_BOOLEAN:PSETID_Appointment:0x8200";
		$properties["updatecounter"] = "PT_LONG:PSETID_Appointment:0x8201";
		$properties["unknown7"] = "PT_LONG:PSETID_Appointment:0x8202";
		$properties["busystatus"] = "PT_LONG:PSETID_Appointment:0x8205";
		$properties["start"] = "PT_SYSTIME:PSETID_Appointment:0x820d";
		$properties["responselocation"] = "PT_STRING8:PSETID_Meeting:0x2";
		$properties["location"] = "PT_STRING8:PSETID_Appointment:0x8208";
		$properties["requestsent"] = "PT_BOOLEAN:PSETID_Appointment:0x8229";
		$properties["startdate"] = "PT_SYSTIME:PSETID_Appointment:0x820d";
		$properties["duedate"] = "PT_SYSTIME:PSETID_Appointment:0x820e";
		$properties["recurring"] = "PT_BOOLEAN:PSETID_Appointment:0x8223";
		// Propose new time properties
		$properties["proposed_start_whole"] = "PT_SYSTIME:PSETID_Appointment:0x8250";
		$properties["proposed_end_whole"] = "PT_SYSTIME:PSETID_Appointment:0x8251";
		$properties["proposed_duration"] = "PT_LONG:PSETID_Appointment:0x8256";
		$properties["counter_proposal"] = "PT_BOOLEAN:PSETID_Appointment:0x8257";
		$properties["recurring_pattern"] = "PT_STRING8:PSETID_Appointment:0x8232";

		$this->proptags = getPropIdsFromStrings($store, $properties);
	}

	/**
	 * Returns TRUE if the message pointed to is an incoming meeting request and should
	 * therefore be replied to with doAccept or doDecline()
	 */
	function isMeetingRequest()
	{
		$props = mapi_getprops($this->message, Array(PR_MESSAGE_CLASS));
		
		if(isset($props[PR_MESSAGE_CLASS]) && $props[PR_MESSAGE_CLASS] == "IPM.Schedule.Meeting.Request")
			return true;
	}
	
	/**
	 * Returns TRUE if the message pointed to is a returning meeting request response
	 */
	function isMeetingRequestResponse()
	{
		$props = mapi_getprops($this->message, Array(PR_MESSAGE_CLASS));
		
		if(isset($props[PR_MESSAGE_CLASS]) && strpos($props[PR_MESSAGE_CLASS], "IPM.Schedule.Meeting.Resp") === 0)
			return true;
	}
	
	/**
	 * Returns TRUE if the message pointed to is a cancellation request
	 */
	function isMeetingCancellation()
	{
		$props = mapi_getprops($this->message, Array(PR_MESSAGE_CLASS));
		
		if(isset($props[PR_MESSAGE_CLASS]) && $props[PR_MESSAGE_CLASS] == "IPM.Schedule.Meeting.Canceled")
			return true;
	}
	
	/**
	 * Process an incoming meeting request response. This updates the appointment
	 * in your calendar to show whether the user has accepted or declined.
	 */
	function processMeetingRequestResponse()
	{
		if(!$this->isLocalOrganiser())
			return;
			
		if(!$this->isMeetingRequestResponse())
			return;
		
		// All properties for a recipient that are interesting
		$recipprops = Array(PR_ENTRYID, PR_DISPLAY_NAME, PR_EMAIL_ADDRESS, PR_RECIPIENT_ENTRYID, PR_RECIPIENT_TYPE, PR_SEND_INTERNET_ENCODING, PR_SEND_RICH_INFO, PR_RECIPIENT_DISPLAY_NAME, PR_ADDRTYPE, PR_DISPLAY_TYPE, PR_RECIPIENT_TRACKSTATUS, PR_RECIPIENT_FLAGS, PR_ROWID);
		
		// Get information we need rfom the response message
		$messageprops = mapi_getprops($this->message, Array(
													$this->proptags['goid'], 
													PR_OWNER_APPT_ID, 
													PR_SENT_REPRESENTING_EMAIL_ADDRESS, 
													PR_SENT_REPRESENTING_NAME,
													PR_SENT_REPRESENTING_ADDRTYPE,
													PR_SENT_REPRESENTING_ENTRYID,
													PR_MESSAGE_DELIVERY_TIME,
													PR_MESSAGE_CLASS,
													$this->proptags['proposed_start_whole'],
													$this->proptags['proposed_end_whole'],
													$this->proptags['proposed_duration'],
													$this->proptags['counter_proposal']));
		
		$goid = $messageprops[$this->proptags['goid']];
		$apptid = $messageprops[PR_OWNER_APPT_ID];
		$email = $messageprops[PR_SENT_REPRESENTING_EMAIL_ADDRESS];
		$senderentryid = $messageprops[PR_SENT_REPRESENTING_ENTRYID];
		$messageclass = $messageprops[PR_MESSAGE_CLASS];
		$deliverytime = $messageprops[PR_MESSAGE_DELIVERY_TIME];
		
		if(!isset($goid) || !isset($apptid) || !isset($email))
			return;
			
		// First, find the items in the calendar by OID
		$calendaritems = $this->findCalendarItems($goid, $apptid);
		
		// $calendaritems now contains the ENTRYID's of all the calendar items to which
		// this meeting request points.
		
		// Open the calendar items, and update all the recipients of the calendar item that match
		// the email address of the response.
		
		foreach($calendaritems as $entryid) {
			// Open each calendar item, find the sender in the recipient table, and update their row
			$calendaritem = mapi_msgstore_openentry($this->store, $entryid);
			
			if($calendaritem == false)
				continue;
				
			$reciptable = mapi_message_getrecipienttable($calendaritem);
			$recipients = mapi_table_queryallrows($reciptable, $recipprops);

			// FIXME we should look at the updatecounter property and compare it
			// to the counter in the recipient to see if this update is actually
			// newer than the status in the calendar item			
			$found = false;
			
			$totalrecips = 0;
			$acceptedrecips = 0;
			
			foreach($recipients as $recipient) {
				$totalrecips++;
				if($this->compareABEntryIDs($recipient[PR_ENTRYID],$senderentryid)) {
					// The email address matches, update the row
					$recipient[PR_RECIPIENT_TRACKSTATUS] = $this->getTrackStatus($messageclass);
					$recipient[PR_RECIPIENT_TRACKSTATUS_TIME] = $deliverytime;

					// If this is a counter proposal, set the proposal properties in the recipient row
					if(isset($messageprops[$this->proptags['counter_proposal']]) && $messageprops[$this->proptags['counter_proposal']]){
						$recipient[PR_PROPOSENEWTIME_START] = $messageprops[$this->proptags['proposed_start_whole']];
						$recipient[PR_PROPOSENEWTIME_END] = $messageprops[$this->proptags['proposed_end_whole']];
						$recipient[PR_PROPOSEDNEWTIME] = $messageprops[$this->proptags['counter_proposal']];
					}

					mapi_message_modifyrecipients($calendaritem, MODRECIP_MODIFY, Array($recipient));
					$found = true;
				}
				if($recipient[PR_RECIPIENT_TRACKSTATUS] == olResponseAccepted)
					$acceptedrecips++;
			}
			
			// If the recipient was not found in the original calendar item, 
			// then add the recpient as a new optional recipient
			if(!$found) {
				$recipient = Array();
				$recipient[PR_ENTRYID] = $messageprops[PR_SENT_REPRESENTING_ENTRYID];
				$recipient[PR_EMAIL_ADDRESS] = $messageprops[PR_SENT_REPRESENTING_EMAIL_ADDRESS];
				$recipient[PR_DISPLAY_NAME] = $messageprops[PR_SENT_REPRESENTING_NAME];
				$recipient[PR_ADDRTYPE] = $messageprops[PR_SENT_REPRESENTING_ADDRTYPE];
				$recipient[PR_RECIPIENT_TYPE] = MAPI_CC;
				$recipient[PR_RECIPIENT_TRACKSTATUS] = $this->getTrackStatus($messageclass);
				$recipient[PR_RECIPIENT_TRACKSTATUS_TIME] = $deliverytime;

				// If this is a counter proposal, set the proposal properties in the recipient row
				if($messageprops[$this->proptags['counter_proposal']]){
					$recipient[PR_PROPOSENEWTIME_START] = $messageprops[$this->proptags['proposed_start_whole']];
					$recipient[PR_PROPOSENEWTIME_END] = $messageprops[$this->proptags['proposed_end_whole']];
					$recipient[PR_PROPOSEDNEWTIME] = $messageprops[$this->proptags['counter_proposal']];
				}

				mapi_message_modifyrecipients($calendaritem, MODRECIP_ADD, Array($recipient));
				$totalrecips++;
				if($recipient[PR_RECIPIENT_TRACKSTATUS] == olResponseAccepted)
					$acceptedrecips++;
			}
			
			// If all the people in the meeting have accepted, change the status into 'busy'
			if($totalrecips == $acceptedrecips) {
				$props = Array();
				$props[$this->proptags['busystatus']] = 2; // olBusy
				
				mapi_message_setprops($calendaritem, $props);
			}

//TODO: Upate counter proposal number property on message
/*
If it is the first time this attendee has proposed a new date/time, increment the value of the PidLidAppointmentProposalNumber property on the organizers meeting object, by 0x00000001. If this property did not previously exist on the organizers meeting object, it MUST be set with a value of 0x00000001.
*/
			// If this is a counter proposal, set the counter proposal indicator boolean
			if(isset($messageprops[$this->proptags['counter_proposal']])){
				$props = Array();
				if($messageprops[$this->proptags['counter_proposal']]){
					$props[$this->proptags['counter_proposal']] = true;
				}else{
					$props[$this->proptags['counter_proposal']] = false;
				}
				
				mapi_message_setprops($calendaritem, $props);
			}

			
			mapi_message_savechanges($calendaritem);
		}
	}

	/**
	 * Process an incoming meeting request cancellation. This updates the 
	 * appointment in your calendar to show that the meeting has been cancelled.
	 */
	function processMeetingCancellation()
	{
		if($this->isLocalOrganiser())
			return;
			
		if(!$this->isMeetingCancellation())
			return;

		if(!$this->isInCalendar())
			return;

		$listProperties = $this->proptags;
		$listProperties['owner_appt_id'] = PR_OWNER_APPT_ID;
		$listProperties['subject'] = PR_SUBJECT;
		$messageprops = mapi_getprops($this->message, $listProperties);

		$goid = $messageprops[$this->proptags['goid']];
		$apptid = $messageprops[PR_OWNER_APPT_ID];
		if(!isset($goid) || !isset($apptid))
			return;

		// First, find the items in the calendar by GOID
		$calendaritems = $this->findCalendarItems($goid, $apptid);

		foreach($calendaritems as $entryid) {
			// Open each calendar item and set the properties of the cancellation object
			$calendaritem = mapi_msgstore_openentry($this->store, $entryid);
			mapi_message_setprops($calendaritem, $messageprops);
			mapi_savechanges($calendaritem);
		}
	}

	/**
	 * Returns true if the item is already in the calendar
	 */
	function isInCalendar() {
		$messageprops = mapi_getprops($this->message, Array($this->proptags['goid'], PR_OWNER_APPT_ID));
		$goid = $messageprops[$this->proptags['goid']];
		$apptid = $messageprops[PR_OWNER_APPT_ID];

		$entryid = $this->findCalendarItems($goid, $apptid);
		
		return is_array($entryid);
	}
		
	/**
	 * Accepts the meeting request by moving the item to the calendar
	 * and sending a confirmation message back to the sender. If $tentative
	 * is TRUE, then the item is accepted tentatively. After accepting, you
	 * can't use this class instance any more. The message is closed. If you
	 * specify TRUE for 'move', then the item is actually moved (from your
	 * inbox probably) to the calendar. If you don't, it is copied into
     * your calendar.
	 */
	function doAccept($tentative, $sendresponse, $move, $newProposedStartTime=false, $newProposedEndTime=false, $body=false, $userAction = false, $store=false)
	{
		if($this->isLocalOrganiser())
			return false;
			
		if($sendresponse) {
			$this->createResponse($tentative ? olResponseTentative : olResponseAccepted, $newProposedStartTime, $newProposedEndTime, $body, $store);
		}

		// Remove any previous calendar items with this goid and appt id
		$messageprops = mapi_getprops($this->message, Array(PR_MESSAGE_CLASS, $this->proptags['goid'], PR_OWNER_APPT_ID, $this->proptags['updatecounter'], PR_PROCESSED));
		$goid = $messageprops[$this->proptags['goid']];
		$apptid = $messageprops[PR_OWNER_APPT_ID];

		/**
		 *	if this function is called automatically with meeting request object then there will be
		 *	two possibilitites 
		 *	1) meeting request is opened first time, in this case make a tentative appointment in 
				recipient's calendar
		 *	2) after this every subsequest request to open meeting request will not do any processing
		 */
		if($messageprops[PR_MESSAGE_CLASS] == "IPM.Schedule.Meeting.Request" && $userAction == false) {
			if(isset($messageprops[PR_PROCESSED]) && $messageprops[PR_PROCESSED] == true) {
				// if meeting request is already processed then don't do anything
				return false;
			} else {
				mapi_setprops($this->message, Array(PR_PROCESSED => true));
				mapi_message_savechanges($this->message);
			}
		}

		$calendar = $this->openDefaultCalendar();
		
		$entryids = $this->findCalendarItems($goid, $apptid);
		
		if(is_array($entryids)) {
		    // Only check the first, there should only be one anyway...
		    $previtem = mapi_msgstore_openentry($this->store, $entryids[0]);
		    
		    $prevcounterprops = mapi_getprops($previtem, array($this->proptags['updatecounter']));
		    
		    // Check if the existing item has an updatecounter that is lower than the request we are processing. If not, then we ignore this call, since the
		    // meeting request is out of date.
		    /*
		     	if(message_counter < appointment_counter) do_nothing
				if(message_counter == appointment_counter) do_something_if_the_user_tells_us (userAction == true)
				if(message_counter > appointment_counter) do_something_even_automatically
			*/
		    if(isset($prevcounterprops[$this->proptags['updatecounter']]) && $messageprops[$this->proptags['updatecounter']] < $prevcounterprops[$this->proptags['updatecounter']]) {
		    	return false;
		    } else if(isset($prevcounterprops[$this->proptags['updatecounter']]) && $messageprops[$this->proptags['updatecounter']] == $prevcounterprops[$this->proptags['updatecounter']]) {
		    	if($userAction == false) {
		    		return false;
		    	}
		    }
		}
		
		if(is_array($entryids)) 
			mapi_folder_deletemessages($calendar, $entryids);
		
		if($move) {
			// All we have to do is open the default calendar,
			// set the mesage class correctly to be an appointment item
			// and move it to the calendar folder
		
			$sourcefolder = $this->openParentFolder();
			$status = $tentative ? 1 : 2;
			$response = $tentative ? olResponseTentative : olResponseAccepted;
		
			mapi_setprops($this->message, Array(PR_MESSAGE_CLASS => "IPM.Appointment", $this->proptags['busystatus'] => $status, $this->proptags['responsestatus'] => $response ));

			mapi_message_savechanges($this->message);
			$messageprops = mapi_getprops($this->message, Array(PR_ENTRYID));
		
			// Release the message
			$this->message = null;
		
			// Move the message to the calendar
		    mapi_folder_copymessages($sourcefolder, array($messageprops[PR_ENTRYID]), $calendar, MESSAGE_MOVE);
		    
		    $entryid = $messageprops[PR_ENTRYID];
		} else {
			// Create a new appointment with duplicate properties and recipient, but as an IPM.Appointment
			$new = mapi_folder_createmessage($calendar);

			$status = $tentative ? 1 : 2;
			$response = $tentative ? olResponseTentative : olResponseAccepted;
		
			$props = mapi_getprops($this->message);
			mapi_setprops($new, $props);
			mapi_setprops($new, Array(PR_MESSAGE_CLASS => "IPM.Appointment", $this->proptags['busystatus'] => $status, $this->proptags['responsestatus'] => $response ));
			
			$reciptable = mapi_message_getrecipienttable($this->message);
			$recips = mapi_table_queryallrows($reciptable, $this->recipprops);
			
			$this->addOrganizer($props, $recips);
			
			mapi_message_modifyrecipients($new, MODRECIP_ADD, $recips);
			
			mapi_message_savechanges($new);
			
			$props = mapi_getprops($new, array(PR_ENTRYID));
			
			$entryid = $props[PR_ENTRYID];
		} 
		
		return $entryid;
	}
	
	/**
	 * Declines the meeting request by moving the item to the deleted
	 * items folder and sending a decline message. After declining, you
	 * can't use this class instance any more. The message is closed.
	 */
	function doDecline($sendresponse, $store=false)
	{
		if($this->isLocalOrganiser())
			return;

		if($sendresponse) {
			$this->createResponse(olResponseDeclined, false, false, false, $store);
		}
		
		// Remove any previous calendar items with this goid and appt id
		$messageprops = mapi_getprops($this->message, Array($this->proptags['goid'], PR_OWNER_APPT_ID));
		$goid = $messageprops[$this->proptags['goid']];
		$apptid = $messageprops[PR_OWNER_APPT_ID];
		
		$entryids = $this->findCalendarItems($goid, $apptid);
		$calendar = $this->openDefaultCalendar();
		
		mapi_folder_deletemessages($calendar, $entryids);
		
		// All we have to do to decline, is to move the item to the waste basket
		
		$wastebasket = $this->openDefaultWastebasket();
		$sourcefolder = $this->openParentFolder();
		
		$messageprops = mapi_getprops($this->message, Array(PR_ENTRYID));
		
		// Release the message
		$this->message = null;
		
		// Move the message to the waste basket
		mapi_folder_copymessages($sourcefolder, Array($messageprops[PR_ENTRYID]), $wastebasket, MESSAGE_MOVE);
	}

    /**
     * Removes a meeting request from the calendar when the user presses the
     * 'remove from calendar' button in response to a meeting cancellation.
     */	
	function doRemoveFromCalendar() 
	{
		if($this->isLocalOrganiser())
			return false;
			
		$messageprops = mapi_getprops($this->message, Array(PR_ENTRYID, $this->proptags['goid'], PR_OWNER_APPT_ID));
		$goid = $messageprops[$this->proptags['goid']];
		$apptid = $messageprops[PR_OWNER_APPT_ID];
		
		$calendar = $this->openDefaultCalendar();
		
		$entryids = $this->findCalendarItems($goid, $apptid);
		
		if(is_array($entryids)) 
			mapi_folder_deletemessages($calendar, $entryids);

		// Now remove message to the wastebasket
		$wastebasket = $this->openDefaultWastebasket();
		$sourcefolder = $this->openParentFolder();

		// Release the message
		$this->message = null;
		
		// Move the message to the waste basket
		mapi_folder_copymessages($sourcefolder, Array($messageprops[PR_ENTRYID]), $wastebasket, MESSAGE_MOVE);
	}
	
	/**
	 * Removes the meeting request by moving the item to the deleted
	 * items folder. After canceling, youcan't use this class instance 
	 * any more. The message is closed.
	 */
	function doCancel()
	{
		if($this->isLocalOrganiser())
			return;
		if(!$this->isMeetingCancellation())
			return;
			
		// Remove any previous calendar items with this goid and appt id
		$messageprops = mapi_getprops($this->message, Array($this->proptags['goid'], PR_OWNER_APPT_ID));
		$goid = $messageprops[$this->proptags['goid']];
		$apptid = $messageprops[PR_OWNER_APPT_ID];
		
		$entryids = $this->findCalendarItems($goid, $apptid);
		$calendar = $this->openDefaultCalendar();
		
		mapi_folder_deletemessages($calendar, $entryids);
		
		// All we have to do to decline, is to move the item to the waste basket
		
		$wastebasket = $this->openDefaultWastebasket();
		$sourcefolder = $this->openParentFolder();
		
		$messageprops = mapi_getprops($this->message, Array(PR_ENTRYID));
		
		// Release the message
		$this->message = null;
		
		// Move the message to the waste basket
		mapi_folder_copymessages($sourcefolder, Array($messageprops[PR_ENTRYID]), $wastebasket, MESSAGE_MOVE);
	}


	/**
	 * Sets the properties in the message so that is can be sent
	 * as a meeting request. The caller has to submit the message. This
	 * is only used for new MeetingRequests. Pass the appointment item as $message
	 * in the constructor to do this.
	 */
	function setMeetingRequest()
	{
		$props = Array();

		// Create a new global id for this item
	    $goid = pack("H*", "040000008200E00074C5B7101A82E00800000000");
	    for ($i=0; $i<36; $i++)
		    $goid .= chr(rand(0, 255));
       
        // Create a new appointment id for this item
   		$apptid = rand();
       
		$props[PR_OWNER_APPT_ID] = $apptid;
		$props[PR_ICON_INDEX] = 1026; 
		$props[$this->proptags['goid']] = $goid;
		$props[$this->proptags['goid2']] = $goid;
		$props[$this->proptags['updatecounter']] = 1;
		$props[$this->proptags['meetingstatus']] = olMeeting;
		$props[$this->proptags['responsestatus']] = olResponseOrganized; 
		
		mapi_setprops($this->message, $props);
	}

	/**
	 * Sends a meeting request by copying it to the outbox, converting
	 * the message class, adding some properties that are required only
	 * for sending the message and submitting the message. Set cancel to 
	 * true if you wish to completely cancel the meeting request. You can
	 * specify an optional 'prefix' to prefix the sent message, which is normally
	 * 'Canceled: '
	 */
	function sendMeetingRequest($cancel, $prefix = false)
	{
		$submitMsg = true;
		$includesResources = false;
		$nonAcceptingResources = Array();

		// Get the properties of the message
		$messageprops = mapi_getprops($this->message);

		// PR_START_DATE and PR_END_DATE will be used by outlook to show the position in the calendar
		if (!isset($messageprops[PR_START_DATE])) {
			$addtimeprops[PR_START_DATE] = $messageprops[$this->proptags['startdate']];
			$addtimeprops[PR_END_DATE] = $messageprops[$this->proptags['duedate']];
			mapi_setprops($this->message, $addtimeprops);
		}

		/**
		 * Before the message is sent to the recipients, it has to be placed in 
		 * the store of the BCC-recipients (resources).
		 */
		// Get resource recipients
		$getResourcesRestriction = Array(RES_AND,
			Array(Array(RES_PROPERTY,
				Array(RELOP => RELOP_EQ,	// Equals recipient type 3: Resource
					ULPROPTAG => PR_RECIPIENT_TYPE,
					VALUE => array(PR_RECIPIENT_TYPE =>3) 
				)
			))
		);
		$recipienttable = mapi_message_getrecipienttable($this->message);
		$resourceRecipients = mapi_table_queryallrows($recipienttable, $this->recipprops, $getResourcesRestriction);

		$errorSetResource = false;
		$resourceRecipData = Array();

		// Put appointment into store resource users
		$i = 0;
		while(!$errorSetResource && $i < count($resourceRecipients)){
			$request = array(array(PR_DISPLAY_NAME => $resourceRecipients[$i][PR_DISPLAY_NAME]));
			$ab = mapi_openaddressbook($this->session);
			$ret = mapi_ab_resolvename($ab, $request, EMS_AB_ADDRESS_LOOKUP);
			$result = mapi_last_hresult();
			if ($result == NOERROR){
				$result = $ret[0][PR_ENTRYID];
			}
			$resourceUsername = $ret[0][PR_EMAIL_ADDRESS];
			$resourceABEntryID = $ret[0][PR_ENTRYID];

			// Get StoreEntryID by username
			$user_entryid = mapi_msgstore_createentryid($this->store, $resourceUsername);

			// Open store of the user
			$userStore = mapi_openmsgstore($this->session, $user_entryid);
			// Open root folder
			$userRoot = mapi_msgstore_openentry($userStore, null);
			// Get calendar entryID
			$userRootProps = mapi_getprops($userRoot, array(PR_STORE_ENTRYID, PR_IPM_APPOINTMENT_ENTRYID, PR_FREEBUSY_ENTRYIDS));

			/**
			 * Get the LocalFreebusy message that contains the properties that 
			 * are set to accept or decline resource meeting requests
			 */
			// Use PR_FREEBUSY_ENTRYIDS[1] to open folder the LocalFreeBusy msg
			$localFreebusyMsg = mapi_msgstore_openentry($userStore, $userRootProps[PR_FREEBUSY_ENTRYIDS][1]);
			if($localFreebusyMsg){
				$props = mapi_getprops($localFreebusyMsg, array(PR_PROCESS_MEETING_REQUESTS, PR_DECLINE_RECURRING_MEETING_REQUESTS, PR_DECLINE_CONFLICTING_MEETING_REQUESTS));

				$acceptMeetingRequests = ($props[PR_PROCESS_MEETING_REQUESTS])?1:0;
				$declineRecurringMeetingRequests = ($props[PR_DECLINE_RECURRING_MEETING_REQUESTS])?1:0;
				$declineConflictingMeetingRequests = ($props[PR_DECLINE_CONFLICTING_MEETING_REQUESTS])?1:0;
				if(!$acceptMeetingRequests){
					/**
					 * When a resource has not been set to automatically accept meeting requests, 
					 * the meeting request has to be sent to him rather than being put directly into
					 * his calendar. No error should be returned.
					 */
					//$errorSetResource = 2;
					$nonAcceptingResources[] = $resourceRecipients[$i];
				}else{
					if($declineRecurringMeetingRequests && !$cancel){
						// Check if appointment is recurring
						if($messageprops[ $this->proptags['recurring'] ]){
							$errorSetResource = 3;
						}
					}
					if($declineConflictingMeetingRequests && !$cancel){
						// Check for conflicting items
						$conflicting = false;
						
						// Open the calendar
						$calFolder = mapi_msgstore_openentry($userStore, $userRootProps[PR_IPM_APPOINTMENT_ENTRYID]);
						
						if($calFolder) {
    						// Get all items in the timeframe that we want to book, and get the goid, apptid and busystatus for each
    						$items = Recurrence::getCalendarItems($userStore, $calFolder, $messageprops[$this->proptags['startdate']], $messageprops[$this->proptags['duedate']], array($this->proptags['goid'], $this->proptags['busystatus'], PR_OWNER_APPT_ID));
    						
    						// See if any of the items are both busy and NOT the item that we are trying to update (we don't want to conflict with ourself). 
    						// If there are any, then there is a conflict.
    						foreach($items as $item) {
    						    if($item[$this->proptags['busystatus']] > 0 && 
    						       $item[$this->proptags['goid']] != $messageprops[$this->proptags['goid']] &&
    						       $item[PR_OWNER_APPT_ID] != $messageprops[$this->proptags[PR_OWNER_APPT_ID]]) 
                                {
    						           $conflicting = true;
    						           break;
                                }
    						}
                        } else {
                            $errorSetResource = 1; // No access
                        }
                        
						if($conflicting){
							$errorSetResource = 4; // Conflict
						}
					}
				}

			}

			if(!$errorSetResource){
				// Open Calendar folder   [check hresult==0]
				$calFolder = mapi_msgstore_openentry($userStore, $userRootProps[PR_IPM_APPOINTMENT_ENTRYID]);
				if($calFolder){
					$calFolderProps = mapi_getProps($calFolder, Array(PR_ACCESS));
					$calFolderProps[PR_ACCESS];
					if(($calFolderProps[PR_ACCESS] & MAPI_ACCESS_CREATE_CONTENTS) == 0){
						$accessToFolder = false;
					}else{
						$accessToFolder = true;
					}
				}else{
					$accessToFolder = false;
				}
			}

			if(!$errorSetResource && $accessToFolder){

				/**
				 * Check if the meeting request that has to be set in the 
				 * calendar of the resource has not been set already by checking
				 * whether a message with the same GOID and PR_OWNER_APPT_ID is 
				 * already present in their calendar. If so, update that 
				 * message. If not create a new message.
				 */
				// Find the item by restricting all items to the correct ID
				$findMatchingOWNERAPPTID = Array(RES_AND,
					Array( 
						Array(RES_PROPERTY,
							Array(RELOP => RELOP_EQ,
								  ULPROPTAG => $this->proptags['goid'],
								  VALUE => $messageprops[$this->proptags['goid']]
							)
						),
						Array(RES_PROPERTY,
							Array(RELOP => RELOP_EQ,
								  ULPROPTAG => PR_OWNER_APPT_ID,
								  VALUE => $messageprops[PR_OWNER_APPT_ID]
							)
						)
					)
				);
				$calendarcontents = mapi_folder_getcontentstable($calFolder);
				$rows = mapi_table_queryallrows($calendarcontents, Array(PR_ENTRYID), $findMatchingOWNERAPPTID);
				if(count($rows) == 0){
					$newResourceMsg = mapi_folder_createmessage($calFolder);
					mapi_message_savechanges($newResourceMsg);
				}else{
					$calendaritems = Array();
					// In principle, there should only be one row, but we'll handle them all just in case	
					foreach($rows as $row) {
						$calendaritems[] = $row[PR_ENTRYID];
					}
					$newResourceMsg = mapi_msgstore_openentry($userStore, $calendaritems[0]);
				}

				// Copy the entire message into the newResourceMsg meeting request message
				$messageprops = mapi_getprops($this->message);
				
				// Prefix the subject if needed
				if($prefix && isset($messageprops[PR_SUBJECT])) {
				    $messageprops[PR_SUBJECT] = $prefix . $messageprops[PR_SUBJECT];
				}
				
				// Set status to cancelled if needed
				if($cancel) {
				    $messageprops[$this->proptags['meetingstatus']] = olMeetingCanceled;
				    $messageprops[$this->proptags['busystatus']] = 0; // Free
                }
				
				// Remove the PR_ICON_INDEX as it is not needed in the sent message and it also
				// confuses the Zarafa webaccess
				$messageprops[PR_ICON_INDEX] = null;
				$messageprops[PR_RESPONSE_REQUESTED] = true;

				$addrinfo = $this->getOwnerAddress($this->store);

				if($addrinfo) {
					list($ownername, $owneremailaddr, $owneraddrtype, $ownerentryid, $ownersearchkey) = $addrinfo;
					
					$messageprops[PR_SENT_REPRESENTING_EMAIL_ADDRESS] = $owneremailaddr;
					$messageprops[PR_SENT_REPRESENTING_NAME] = $ownername;
					$messageprops[PR_SENT_REPRESENTING_ADDRTYPE] = $owneraddrtype;
					$messageprops[PR_SENT_REPRESENTING_ENTRYID] = $ownerentryid;
					$messageprops[PR_SENT_REPRESENTING_SEARCH_KEY] = $ownersearchkey;
				}
				
				mapi_setprops($newResourceMsg, $messageprops);
				
				// Copy attachments
				$attachmentTable = mapi_message_getattachmenttable($this->message);
				if($attachmentTable) {
					$attachments = mapi_table_queryallrows($attachmentTable, array(PR_ATTACH_NUM, PR_ATTACH_SIZE, PR_ATTACH_LONG_FILENAME, PR_ATTACHMENT_HIDDEN, PR_DISPLAY_NAME, PR_ATTACH_METHOD));

					foreach($attachments as $attach_props){
						// create a newResourceMsg attachment
						$attach_newResourceMsg = mapi_message_createattach($newResourceMsg);
						mapi_setprops($attach_newResourceMsg, $attach_props);
						$stream_newResourceMsg = mapi_openpropertytostream($attach_newResourceMsg, PR_ATTACH_DATA_BIN, MAPI_CREATE | MAPI_MODIFY);

						// open the attachment
						$attach_old = mapi_message_openattach($this->message, (int) $attach_props[PR_ATTACH_NUM]);
						$stream_old = mapi_openpropertytostream($attach_old, PR_ATTACH_DATA_BIN);

						// copy stream
						$stat = mapi_stream_stat($stream_old);
						mapi_stream_setsize($stream_newResourceMsg, $stat["cb"]);
						$block_size = (defined("BLOCK_SIZE")?BLOCK_SIZE:1048576);
						for($i = 0; $i < $stat["cb"]; $i += $block_size) {
							mapi_stream_write($stream_newResourceMsg, mapi_stream_read($stream_old, $block_size));
						}

						mapi_stream_commit($stream_newResourceMsg);
						mapi_savechanges($attach_newResourceMsg);
					}
				}

				// Copy all recipients too
				$recipienttable = mapi_message_getrecipienttable($this->message);
				$recipients = mapi_table_queryallrows($recipienttable, $this->recipprops);
				mapi_message_modifyrecipients($newResourceMsg, MODRECIP_ADD, $recipients);

                // FIXME: this following code looks unused, since $props is not used any further ?
                
				$props[$this->proptags['busystatus']] = 2; // The default status (Busy)
				if(!$cancel) {
    				$props[$this->proptags['meetingstatus']] = olMeetingReceived; // The recipient is receiving the request
                } else {
    				$props[$this->proptags['meetingstatus']] = olMeetingCanceled; // The meeting has been canceled
                }
				$props[$this->proptags['responsestatus']] = olResponseAccepted; // The resource autmatically accepts the appointment

				$props[PR_MESSAGE_CLASS] = "IPM.Appointment";

				mapi_message_savechanges($newResourceMsg);

				$resourceRecipData[] = Array(
					'store' => $userStore,
					'folder' => $calFolder,
					'msg' => $newResourceMsg,
				);
				$includesResources = true;
			}else{
				/**
				 * If no other errors occured and you have no access to the 
				 * folder of the resource, throw an error=1.
				 */
				if(!$errorSetResource){
					$errorSetResource = 1;
				}

				for($j=0;$j<count($resourceRecipData);$j++){
					// Get the EntryID
					$props = mapi_message_getprops($resourceRecipData[$j]['msg']);

					mapi_folder_deletemessages($resourceRecipData[$j]['folder'], Array($props[PR_ENTRYID]), DELETE_HARD_DELETE);
				}
				$recipientDisplayname = $resourceRecipients[$i][PR_DISPLAY_NAME];
			}
			$i++;
		}



		if($submitMsg && !$errorSetResource){
			// Publish updated free/busy information
			for($i=0;$i<count($resourceRecipData);$i++){
				$storeProps = mapi_msgstore_getprops($resourceRecipData[$i]['store'], array(PR_MAILBOX_OWNER_ENTRYID));
				if (isset($storeProps[PR_MAILBOX_OWNER_ENTRYID])){
					$pub = new FreeBusyPublish($this->session, $resourceRecipData[$i]['store'], $resourceRecipData[$i]['folder'], $storeProps[PR_MAILBOX_OWNER_ENTRYID]);
					$pub->publishFB(time() - (7 * 24 * 60 * 60), 6 * 30 * 24 * 60 * 60); // publish from one week ago, 6 months ahead
				}
			}


			/*********************************************
			 * Submit message to non-resource recipients
			 */

			// Set BusyStatus to olTentative (1)
			// Set MeetingStatus to olMeetingReceived
			// Set ResponseStatus to olResponseNotResponded
			
			$outbox = $this->openDefaultOutbox($this->openDefaultStore());
			
			$new = mapi_folder_createmessage($outbox);
			
			// Copy the entire message into the new meeting request message
			$messageprops = mapi_getprops($this->message);
			
			// Remove the PR_ICON_INDEX as it is not needed in the sent message and it also
			// confuses the Zarafa webaccess
			$messageprops[PR_ICON_INDEX] = null;
			$messageprops[PR_RESPONSE_REQUESTED] = true;
			
			// Prefix the subject if needed
			if($prefix && isset($messageprops[PR_SUBJECT])) {
			    $messageprops[PR_SUBJECT] = $prefix . $messageprops[PR_SUBJECT];
			}

			$addrinfo = $this->getOwnerAddress($this->store);
			
			if($addrinfo) {
				list($ownername, $owneremailaddr, $owneraddrtype, $ownerentryid, $ownersearchkey) = $addrinfo;
				$messageprops[PR_SENT_REPRESENTING_EMAIL_ADDRESS] = $owneremailaddr;
				$messageprops[PR_SENT_REPRESENTING_NAME] = $ownername;
				$messageprops[PR_SENT_REPRESENTING_ADDRTYPE] = $owneraddrtype;
				$messageprops[PR_SENT_REPRESENTING_ENTRYID] = $ownerentryid;
				$messageprops[PR_SENT_REPRESENTING_SEARCH_KEY] = $ownersearchkey;
			}
			
			mapi_setprops($new, $messageprops);
			
			// Copy attachments
			$attachmentTable = mapi_message_getattachmenttable($this->message);
			if($attachmentTable) {
				$attachments = mapi_table_queryallrows($attachmentTable, array(PR_ATTACH_NUM, PR_ATTACH_SIZE, PR_ATTACH_LONG_FILENAME, PR_ATTACHMENT_HIDDEN, PR_DISPLAY_NAME, PR_ATTACH_METHOD));

				foreach($attachments as $attach_props){
					// create a new attachment
					$attach_new = mapi_message_createattach($new);
					mapi_setprops($attach_new, $attach_props);
					$stream_new = mapi_openpropertytostream($attach_new, PR_ATTACH_DATA_BIN, MAPI_CREATE | MAPI_MODIFY);

					// open the attachment
					$attach_old = mapi_message_openattach($this->message, (int) $attach_props[PR_ATTACH_NUM]);
					$stream_old = mapi_openpropertytostream($attach_old, PR_ATTACH_DATA_BIN);

					// copy stream
					$stat = mapi_stream_stat($stream_old);
					mapi_stream_setsize($stream_new, $stat["cb"]);
					$block_size = (defined("BLOCK_SIZE")?BLOCK_SIZE:1048576);
					for($i = 0; $i < $stat["cb"]; $i += $block_size) {
						mapi_stream_write($stream_new, mapi_stream_read($stream_old, $block_size));
					}

					mapi_stream_commit($stream_new);
					mapi_savechanges($attach_new);
				}
			}

			// Copy recipients too
			$recipienttable = mapi_message_getrecipienttable($this->message);
			$stripResourcesRestriction = Array(RES_AND,
				Array(Array(RES_PROPERTY,
					Array(RELOP => RELOP_NE,	// Does not equal recipient type: 3 (Resource)
						ULPROPTAG => PR_RECIPIENT_TYPE,
						VALUE => array(PR_RECIPIENT_TYPE =>3) 
					)
				))
			);
			$recipients = mapi_table_queryallrows($recipienttable, $this->recipprops, $stripResourcesRestriction);
			/**
			 * Add resource recipients that did not automatically accept the meeting request.
			 * (note: meaning that they did not decline the meeting request)
			 */
			for($i=0;$i<count($nonAcceptingResources);$i++){
				$recipients[] = $nonAcceptingResources[$i];
			}

			if(count($recipients) > 0){
				// Strip out the sender/"owner" recipient
				mapi_message_modifyrecipients($new, MODRECIP_ADD, $recipients);

				// Set some properties that are different in the sent request than
				// in the item in our calendar		
				$props[$this->proptags['busystatus']] = 1; // The default status when not accepted
				if(!$cancel) 
    				$props[$this->proptags['meetingstatus']] = olMeetingReceived; // The recipient is receiving the request
                else 
    				$props[$this->proptags['meetingstatus']] = olMeetingCanceled; // It's a cancel request
    				
				$props[$this->proptags['responsestatus']] = olResponseNotResponded; // The recipient has not responded yet
				
				if($cancel)
					$props[PR_MESSAGE_CLASS] = "IPM.Schedule.Meeting.Canceled";
				else
					$props[PR_MESSAGE_CLASS] = "IPM.Schedule.Meeting.Request";
				
				$sentmail = $this->getDefaultSentmailEntryID($this->openDefaultStore());
				$props[PR_SENTMAIL_ENTRYID] = $sentmail;

				//Set recurrence pattern in meeting request mail
				if (isset($messageprops[$this->proptags['recurring_pattern']])){
					$props[$this->proptags['recurring_pattern']] = $messageprops[$this->proptags['recurring_pattern']];
				}
				
				mapi_setprops($new, $props);

				mapi_message_savechanges($new);

				// Submit message to non-resource recipients
				mapi_message_submitmessage($new);

			}

			/**************************************************************
			 * Set the BCC-recipients (resources) tackstatus to accepted.
			 */
			// Get resource recipients
			$getResourcesRestriction = Array(RES_AND,
				Array(Array(RES_PROPERTY,
					Array(RELOP => RELOP_EQ,	// Equals recipient type 3: Resource
						ULPROPTAG => PR_RECIPIENT_TYPE,
						VALUE => array(PR_RECIPIENT_TYPE =>3) 
					)
				))
			);
			$recipienttable = mapi_message_getrecipienttable($this->message);
			$resourceRecipients = mapi_table_queryallrows($recipienttable, $this->recipprops, $getResourcesRestriction);
			if(count($resourceRecipients) > 0){
				// Set Tracking status of resource recipients to olResponseAccepted (3)
				for($i=0;$i<count($resourceRecipients);$i++){
					$resourceRecipients[$i][PR_RECIPIENT_TRACKSTATUS] = olResponseAccepted;
					$resourceRecipients[$i][PR_RECIPIENT_TRACKSTATUS_TIME] = mktime();
				}
				mapi_message_modifyrecipients($this->message, MODRECIP_MODIFY, $resourceRecipients);
			}

			// Set requestsent to 'true' (turns on 'tracking', etc)
			mapi_setprops($this->message, array($this->proptags['requestsent'] => true));
			mapi_savechanges($this->message);

			return true;
		}elseif($errorSetResource){
			return Array(
				'error' => $errorSetResource,
				'displayname' => $recipientDisplayname
			);
		}else{
			return true;
		}
	}


	function getFreeBusyInfo($entryID,$start,$end)
	{
		$result = array();
		$fbsupport = mapi_freebusysupport_open($this->session);
		$fbDataArray = mapi_freebusysupport_loaddata($fbsupport, array($entryID));

		if($fbDataArray[0] != NULL){
			foreach($fbDataArray as $fbDataUser){
				$rangeuser1 = mapi_freebusydata_getpublishrange($fbDataUser);
				if($rangeuser1 == NULL){
					return $result;
				}

				$enumblock = mapi_freebusydata_enumblocks($fbDataUser, $start, $end);
				mapi_freebusyenumblock_reset($enumblock);

				while(true){
					$blocks = mapi_freebusyenumblock_next($enumblock, 100);
					if(!$blocks){
						break;
					}
					foreach($blocks as $blockItem){
						$result[] = $blockItem;
					}
				}
			}
		}

		mapi_freebusysupport_close($fbsupport);
		return $result;
	}

	/**
	 * Updates the message after an update has been performed (for example,
	 * changing the time of the meeting). This must be called before re-sending
	 * the meeting request. You can also call this function instead of 'setMeetingRequest()'
	 * as it will automatically call setMeetingRequest on this object if it is the first
	 * call to this function.
	 */
	function updateMeetingRequest()
	{
		$messageprops = mapi_getprops($this->message, Array($this->proptags['updatecounter']));
		
		if(!isset($messageprops[$this->proptags['updatecounter']]))
			$this->setMeetingRequest();
		else {
			$counter = $messageprops[$this->proptags['updatecounter']]+1;
			
			mapi_setprops($this->message, Array($this->proptags['updatecounter'] => $counter));
		}
	}
	
	/**
	 * Returns TRUE if we are the organiser of the meeting. 
	 */
	function isLocalOrganiser()
	{
		if($this->isMeetingRequest() || $this->isMeetingRequestResponse()) {
			$messageid = $this->getAppointmentEntryID();
			
			if(!isset($messageid))
				return false;
				
			$message = mapi_msgstore_openentry($this->store, $messageid);
		} else {
			$message = $this->message;
		}
		
		$messageprops = mapi_getprops($message, Array($this->proptags['responsestatus']));
		
		return $messageprops[$this->proptags['responsestatus']] == olResponseOrganized;
	}
	
	/**
	 * Returns the entryid of the appointment that this message points at. This is
	 * only used on messages that are not in the calendar.
	 */
	function getAppointmentEntryID()
	{
		$messageprops = mapi_getprops($this->message, Array($this->proptags['goid'], PR_OWNER_APPT_ID));
		
		$goid = $messageprops[$this->proptags['goid']];
		$apptid = $messageprops[PR_OWNER_APPT_ID];
		
		$items = $this->findCalendarItems($goid, $apptid);
		
		if(count($items) == 0)
			return;
		
		// There should be just one item. If there are more, we just take the first one
		return $items[0];
	}
	
	/***************************************************************************************************
	 * Support functions - INTERNAL ONLY
	 ***************************************************************************************************
	 */
	 
	/**
	 * Return the tracking status of a recipient based on the IPM class (passed)
	 */
	function getTrackStatus($class) {
		$status = olResponseNone;
		switch($class)
		{
			case "IPM.Schedule.Meeting.Resp.Pos":
				$status = olResponseAccepted;
				break;
				
			case "IPM.Schedule.Meeting.Resp.Tent":
				$status = olResponseTentative;
				break;	
				
			case "IPM.Schedule.Meeting.Resp.Neg":
				$status = olResponseDeclined;
				break;
		}
		return $status;
	}
	
	function openParentFolder() {
		$messageprops = mapi_getprops($this->message, Array(PR_PARENT_ENTRYID));
		
		$parentfolder = mapi_msgstore_openentry($this->store, $messageprops[PR_PARENT_ENTRYID]);
		return $parentfolder;
	}
	
	function openDefaultCalendar() {
		return $this->openDefaultFolder(PR_IPM_APPOINTMENT_ENTRYID);
	}	

	function openDefaultOutbox($store=false) {
		return $this->openBaseFolder(PR_IPM_OUTBOX_ENTRYID, $store);
	}	
	
	function openDefaultWastebasket() {
		return $this->openBaseFolder(PR_IPM_WASTEBASKET_ENTRYID);
	}
	
	function getDefaultWastebasketEntryID() {
		return $this->getBaseEntryID(PR_IPM_WASTEBASKET_ENTRYID);
	}
	
	function getDefaultSentmailEntryID($store=false) {
		return $this->getBaseEntryID(PR_IPM_SENTMAIL_ENTRYID, $store);
	}
	
	function getDefaultFolderEntryID($prop) {
		$inbox = mapi_msgstore_getreceivefolder($this->store);
		$inboxprops = mapi_getprops($inbox, Array($prop));
		if(!isset($inboxprops[$prop]))
			return;
			
		return $inboxprops[$prop];
	}
			
	function openDefaultFolder($prop) {
		$entryid = $this->getDefaultFolderEntryID($prop);
		$folder = mapi_msgstore_openentry($this->store, $entryid);
		
		return $folder;
	}	

	function getBaseEntryID($prop, $store=false) {
		$storeprops = mapi_getprops( (($store)?$store:$this->store) , Array($prop));
		if(!isset($storeprops[$prop]))
			return;
			
		return $storeprops[$prop];
	}
			
	function openBaseFolder($prop, $store=false) {
		$entryid = $this->getBaseEntryID($prop, $store);
		$folder = mapi_msgstore_openentry( (($store)?$store:$this->store) , $entryid);
		
		return $folder;
	}	

	function createResponse($status, $proposalStartTime=false, $proposalEndTime=false, $body=false, $store=false) {
		// Create a response message
		$messageprops = mapi_getprops($this->message, Array(PR_SENT_REPRESENTING_ENTRYID, 
															PR_SENT_REPRESENTING_EMAIL_ADDRESS,
															PR_SENT_REPRESENTING_ADDRTYPE,
															PR_SENT_REPRESENTING_NAME,
															$this->proptags['goid'],
															$this->proptags['start'],
															$this->proptags['location'],
															$this->proptags['startdate'],
															$this->proptags['duedate'],
															PR_SUBJECT,
															PR_MESSAGE_CLASS,
															PR_OWNER_APPT_ID
													));
			
		$recip = Array();
		$recip[PR_ENTRYID] = $messageprops[PR_SENT_REPRESENTING_ENTRYID];
		$recip[PR_EMAIL_ADDRESS] = $messageprops[PR_SENT_REPRESENTING_EMAIL_ADDRESS];
		$recip[PR_ADDRTYPE] = $messageprops[PR_SENT_REPRESENTING_ADDRTYPE];
		$recip[PR_DISPLAY_NAME] = $messageprops[PR_SENT_REPRESENTING_NAME];
		$recip[PR_RECIPIENT_TYPE] = MAPI_TO;

		switch($status) {
			case olResponseAccepted:
				$classpostfix = "Pos";
				$subjectprefix = _("Accepted");
				break;
			case olResponseDeclined:
				$classpostfix = "Neg";
				$subjectprefix = _("Declined");
				break;
			case olResponseTentative:
				$classpostfix = "Tent";
				$subjectprefix = _("Tentatively accepted");
				break;
		}
			
		$props[PR_SUBJECT] = $subjectprefix . ": " . $messageprops[PR_SUBJECT];													
		$props[PR_MESSAGE_CLASS] = "IPM.Schedule.Meeting.Resp." . $classpostfix;
		$props[PR_OWNER_APPT_ID] = $messageprops[PR_OWNER_APPT_ID];
		$props[$this->proptags['goid']] = $messageprops[$this->proptags['goid']];
		$props[$this->proptags['goid2']] = $messageprops[$this->proptags['goid']];
		$props[$this->proptags['updatecounter']] = 1;

		// get the default store, in which we have to store the accepted email by delegate or normal user.
		$defaultStore = $this->openDefaultStore();
		$props[PR_SENTMAIL_ENTRYID] = $this->getDefaultSentmailEntryID($defaultStore);

		if($proposalStartTime && $proposalEndTime){
			$props[$this->proptags['proposed_start_whole']] = $proposalStartTime;
			$props[$this->proptags['proposed_end_whole']] = $proposalEndTime;
			$props[$this->proptags['proposed_duration']] = round($proposalEndTime - $proposalStartTime)/60;
			$props[$this->proptags['counter_proposal']] = true;
			if($body){
				$props[PR_BODY] = $body;
			}
		}
		
		// PR_START_DATE is used in the UI in Outlook on the response message
		$props[PR_START_DATE] = $messageprops[$this->proptags['start']];
		// responselocation is used in the UI in Outlook on the response message
		$props[$this->proptags['responselocation']] = $messageprops[$this->proptags['location']];
		$props[$this->proptags['location']] = $messageprops[$this->proptags['location']];
		
		//Set startdate and duedate of Appointment.
		$props[$this->proptags['startdate']] = $messageprops[$this->proptags['startdate']];
		$props[$this->proptags['duedate']] = $messageprops[$this->proptags['duedate']];

		//check if $store is set and it is not equal to $defaultStore (means its the delegation case) 
		if(isset($store) && $store !== $defaultStore){
			// get the properties of the other user (for which the logged in user is a delegate).
			$storeProps = mapi_getprops($store, array(PR_MAILBOX_OWNER_ENTRYID));
			$addrbook = mapi_openaddressbook($this->session);
			$addrbookitem = mapi_ab_openentry($addrbook, $storeProps[PR_MAILBOX_OWNER_ENTRYID]); 
			$addrbookitemprops = mapi_getprops($addrbookitem, array(PR_DISPLAY_NAME, PR_EMAIL_ADDRESS));

			// setting the following properties will ensure that the delegation part of message.
			$props[PR_SENT_REPRESENTING_ENTRYID] = $storeProps[PR_MAILBOX_OWNER_ENTRYID];		
			$props[PR_SENT_REPRESENTING_NAME] = $addrbookitemprops[PR_DISPLAY_NAME];
			$props[PR_SENT_REPRESENTING_EMAIL_ADDRESS] = $addrbookitemprops[PR_EMAIL_ADDRESS];
			$props[PR_SENT_REPRESENTING_ADDRTYPE] = "ZARAFA";

			// get the properties of default store and set it accordingly
			$defaultStoreProps = mapi_getprops($defaultStore, array(PR_MAILBOX_OWNER_ENTRYID));
			$addrbookitem = mapi_ab_openentry($addrbook, $defaultStore[PR_MAILBOX_OWNER_ENTRYID]); 
			$addrbookitemprops = mapi_getprops($addrbookitem, array(PR_DISPLAY_NAME, PR_EMAIL_ADDRESS));

			// set the following properties will ensure the sender's details, which will be the default user in this case.
			//the function returns array($name, $emailaddr, $addrtype, $entryid, $searchkey);
			$defaultUserDetails = $this->getOwnerAddress($defaultStore);
			$props[PR_SENDER_ENTRYID] = $defaultUserDetails[3];
			$props[PR_SENDER_EMAIL_ADDRESS] = $defaultUserDetails[1];
			$props[PR_SENDER_NAME] = $defaultUserDetails[0];
			$props[PR_SENDER_ADDRTYPE] = $defaultUserDetails[2];;
		}

		// pass the default store to get the required store.
		$outbox = $this->openDefaultOutbox($defaultStore);
				
		$message = mapi_folder_createmessage($outbox);
		mapi_setprops($message, $props);
		mapi_message_modifyrecipients($message, MODRECIP_ADD, Array($recip));
		mapi_message_savechanges($message);
		mapi_message_submitmessage($message);
	}

	function findCalendarItems($goid, $apptid) {
		// Open the Calendar
		$calendar = $this->openDefaultCalendar();
		
		// Find the item by restricting all items to the correct ID
		$restrict = Array(RES_AND,
							Array( 
									Array(RES_PROPERTY,
													Array(RELOP => RELOP_EQ,
														  ULPROPTAG => $this->proptags['goid'],
														  VALUE => $goid
													)
									),
									Array(RES_PROPERTY,
													Array(RELOP => RELOP_EQ,
														  ULPROPTAG => PR_OWNER_APPT_ID,
														  VALUE => $apptid
													)
									)
								)
							);
								
						
		$calendarcontents = mapi_folder_getcontentstable($calendar);
		
		$rows = mapi_table_queryallrows($calendarcontents, Array(PR_ENTRYID), $restrict);
				
		if(count($rows) == 0)
			return;
		
		$calendaritems = Array();

		// In principle, there should only be one row, but we'll handle them all just in case	
		foreach($rows as $row) {
			$calendaritems[] = $row[PR_ENTRYID];
		}									
			
		return $calendaritems;
	}
	
	// Returns TRUE if both entryid's are equal. Equality is defined by both entryid's pointing at the
	// same SMTP address when converted to SMTP
	function compareABEntryIDs($entryid1, $entryid2) {
		// If the session was not passed, just do a 'normal' compare.
		if(!$this->session)
			return $entryid1 == $entryid2;
			
	    $smtp1 = $this->getSMTPAddress($entryid1);
	    $smtp2 = $this->getSMTPAddress($entryid2);
	    
	    if($smtp1 == $smtp2)
	        return true;
        else
            return false;
	}

	// Gets the SMTP address of the passed addressbook entryid	
	function getSMTPAddress($entryid) {
		if(!$this->session)
			return false;
			
		$ab = mapi_openaddressbook($this->session);
	    
	    $abitem = mapi_ab_openentry($ab, $entryid);
	    
	    if(!$abitem)
	        return "";
	        
        $props = mapi_getprops($abitem, array(PR_ADDRTYPE, PR_EMAIL_ADDRESS, PR_SMTP_ADDRESS));
        
        if($props[PR_ADDRTYPE] == "SMTP") {
            return $props[PR_EMAIL_ADDRESS];
        }
        else return $props[PR_SMTP_ADDRESS];
	}
	
	// Gets the properties associated with the owner of the passed store: PR_DISPLAY_NAME, PR_EMAIL_ADDR, PR_ADDRTYPE, PR_ENTRYID, PR_SEARCH_KEY
	function getOwnerAddress($store)
	{
		if(!$this->session)
			return false;
			
		$props = mapi_getprops($store, array(PR_MAILBOX_OWNER_ENTRYID));
		
		if($props[PR_MAILBOX_OWNER_ENTRYID]) {
			$ab = mapi_openaddressbook($this->session);
			
			$mailuser = mapi_ab_openentry($ab, $props[PR_MAILBOX_OWNER_ENTRYID]);
			if(!$mailuser)
				return false;
				
			$userprops = mapi_getprops($mailuser, array(PR_ADDRTYPE, PR_DISPLAY_NAME, PR_EMAIL_ADDRESS));
			
			$addrtype = $userprops[PR_ADDRTYPE];
			$name = $userprops[PR_DISPLAY_NAME];
			$emailaddr = $userprops[PR_EMAIL_ADDRESS];
			$searchkey = strtoupper($addrtype) . ":" . strtoupper($emailaddr);
			$entryid = $props[PR_MAILBOX_OWNER_ENTRYID];
			
			return array($name, $emailaddr, $addrtype, $entryid, $searchkey);
		}
		return false;
	}
	
	// Opens this session's default message store
	function openDefaultStore()
	{
		$storestable = mapi_getmsgstorestable($this->session);
		$rows = mapi_table_queryallrows($storestable, array(PR_ENTRYID, PR_DEFAULT_STORE));
		$entry = false;
		
		foreach($rows as $row) {
			if(isset($row[PR_DEFAULT_STORE]) && $row[PR_DEFAULT_STORE]) {
				$entryid = $row[PR_ENTRYID];
				break;
			}
		}
		
		if(!$entryid)
			return false;
			
		return mapi_openmsgstore($this->session, $entryid);
	}
	/**
	 *  Function which adds organizer to recipient list which is passed.
	 *  This function also checks if it has organizer.
	 *
	 * @param array $messageProps message properties
	 * @param array $recipients	recipients list of message.
	 */
	function addOrganizer($messageProps, &$recipients){

		$hasOrganizer = false;
		// Check if meeting already has an organizer.
		foreach ($recipients as $recipient){
			if ($recipient[PR_RECIPIENT_FLAGS] == 3)	$hasOrganizer = true;
		}

		if (!$hasOrganizer){
			// Create organizer.
			$organizer = array();
			$organizer[PR_ENTRYID] = $messageProps[PR_SENT_REPRESENTING_ENTRYID];
			$organizer[PR_DISPLAY_NAME] = $messageProps[PR_SENT_REPRESENTING_NAME];
			$organizer[PR_EMAIL_ADDRESS] = $messageProps[PR_SENT_REPRESENTING_EMAIL_ADDRESS];
			$organizer[PR_RECIPIENT_TYPE] = MAPI_TO;
			$organizer[PR_RECIPIENT_DISPLAY_NAME] = $messageProps[PR_SENT_REPRESENTING_NAME];
			$organizer[PR_ADDRTYPE] = empty($messageProps[PR_SENT_REPRESENTING_ADDRTYPE])?'SMTP':$messageProps[PR_SENT_REPRESENTING_ADDRTYPE];
			$organizer[PR_RECIPIENT_TRACKSTATUS] = 0;
			$organizer[PR_RECIPIENT_FLAGS] = 3;

			// Add organizer to recipients list.
			array_push($recipients, $organizer);
		}
	}
}
?>
