package Finance::Currency::Convert::XE;

use 5.006;
use strict;
use warnings;

use vars qw($VERSION);
$VERSION = '0.15';

#--------------------------------------------------------------------------

=head1 NAME

Finance::Currency::Convert::XE - Currency conversion module.

=head1 SYNOPSIS

  use Finance::Currency::Convert::XE;
  my $obj = Finance::Currency::Convert::XE->new()
                || die "Failed to create object\n" ;

  my $value = $obj->convert(
                    'source' => 'GBP',
                    'target' => 'EUR',
                    'value' => '123.45',
                    'format' => 'text'
            )   || die "Could not convert: " . $obj->error . "\n";

  my @currencies = $obj->currencies;

or

  use Finance::Currency::Convert::XE;
  my $obj = Finance::Currency::Convert::XE->new(
                    'source' => 'GBP',
                    'target' => 'EUR',
                    'format' => 'text'
            )   || die "Failed to create object\n" ;

  my $value = $obj->convert(
                    'value' => '123.45',
                    'format' => 'abbv'
           )   || die "Could not convert: " . $obj->error . "\n";

  $value = $obj->convert('123.45')
                || die "Could not convert: " . $obj->error . "\n";

  my @currencies = $obj->currencies;

=head1 DESCRIPTION

Currency conversion module using XE.com's Universal Currency Converter (tm)
site.

WARNING: Do not use this module for any commercial purposes, unless you have
obtain an explicit license to use the service provided by XE.com. For further
details please read the Terms and Conditions available at L<http://www.xe.com>.

=over

=item * http://www.xe.com/errors/noautoextract.htm

=back

=cut

#--------------------------------------------------------------------------

###########################################################################
#Library Modules                                                          #
###########################################################################

use WWW::Mechanize;
use HTML::TokeParser;

###########################################################################
#Constants                                                                #
###########################################################################

use constant    UCC => 'http://www.xe.com/ucc';

###########################################################################
#Variables                                                                #
###########################################################################

my %currencies; # only need to load once!
my @defaults = ('source', 'target', 'format');

my $web = WWW::Mechanize->new();
$web->agent_alias( 'Windows Mozilla' );

#--------------------------------------------------------------------------

###########################################################################
#Interface Functions                                                      #
###########################################################################

=head1 METHODS

=over 4

=item new

Creates a new Finance::Currency::Convert::XE object. Can be supplied with
default values for source and target currency, and the format required of the
output. These can be overridden in the convert() method.

=cut

sub new {
    my ($this, @args) = @_;
    my $class = ref($this) || $this;
    my $self = {};
    bless $self, $class;
    $self->_initialize(@args);
    return $self;
}

=item currencies

Returns a plain array of the currencies available for conversion.

=cut

sub currencies {
    my $self = shift;
    return sort keys %currencies;
}

=item add_currencies

Allows the user to add currencies to the internal hash. Currencies can be added
as per the code below:

    $self->add_currencies(
                    ZZZ => {text => 'An Example', symbol => '$'},
                    ZZY => {text => 'Testing'} );

Note that unless otherwise stated, the symbol will be set to '&#164;'. The code
used must be 3 characters in length, and a text part must be included.

=cut

sub add_currencies {
    my ($self,%hash) = @_;
    for my $code (keys %hash) {
        if($code =~ /[A-Z]{3}/i) {
            $code = uc $code;
            if($hash{$code}->{text}) {
                $currencies{$code}->{name}   = $hash{$code}->{text}   || die;
                $currencies{$code}->{symbol} = $hash{$code}->{symbol} || '&#164;';
            } else {
                $self->{error} = "User currency '$code' has no text part";
            }
        } else {
            $self->{error} = "User currency '$code' is invalid";
        }
    }
}

=item convert

Converts some currency value into another using XE.com's UCC.

An anonymous hash is used to pass parameters. Legal hash keys and values
are as follows:

  convert(
    source => $currency_from,
    target => $currency_to,
    value  => $currency_from_value,
    format => $print_format
  );

The format key is optional, and takes one of the following strings:

  'number' (returns '12.34')
  'symbol' (returns '&#163;12.34')
  'text'   (returns '12.34 Great Britain, Pound')
  'abbv'   (returns '12.34 GBP')

If format key is omitted, 'number' is assumed and the converted value
is returned.

If only a value is passed, it is assumed that this is the value to be
converted and the remaining parameters will be defined by the defaults set
in the constructor. Note that no internal defaults are assumed.

Note that not all countries have symbols in the standard character set.
Where known the appropriate currency symbol is used, otherwise the
generic currency symbol is used.

It should also be noted that there is a recommendation to use only the
standardised three letter abbreviation ('abbv' above). However, for
further reading please see:

  http://www.jhall.demon.co.uk/currency/
  http://www.jhall.demon.co.uk/currency/by_symbol.html

=cut

sub convert {
    my $self = shift;
    my %params = @_ > 1 ? @_ : (value => $_[0]);
    $params{$_} ||= $self->{$_} for(@defaults);

    undef $self->{error};
    unless( $params{source} ){
        $self->{error} = 'Source currency is blank. This parameter is required';
        return;
    }

    unless( exists($currencies{$params{source}}) ){
        $self->{error} = 'Source currency "' . $params{source} . '" is not available';
        return;
    }

    unless( $params{target} ){
        $self->{error} = 'Target currency is blank. This parameter is required';
        return;
    }

    unless( exists($currencies{$params{target}}) ){
        $self->{error} = 'Target currency "' . $params{target} . '" is not available';
        return;
    }

    # store later use
    $self->{code} = $params{target};
    $self->{name} = $currencies{$params{target}}->{name};
    $self->{symbol} = $currencies{$params{target}}->{symbol};
    $self->{string} = $self->_format($params{format});

    # This "feature" is actually useful as a pass-thru filter.
    if( $params{source} eq $params{target} ) {
        return sprintf $self->{string}, $params{value}
    }

    # get the base site
    $web->get( UCC );
    unless($web->success()) {
        $self->{error} = 'Unable to retrieve webpage';
        return;
    }

    # complete and submit the form
    $web->submit_form(
            form_name => 'ucc',
            fields    => { 'From'   => $params{source},
                           'To'     => $params{target},
                           'Amount' => $params{value} } );
    unless($web->success()) {
        $self->{error} = 'Unable to retrieve webform';
        return;
    }

    # return the converted value
    return $self->_extract_text($web->content());
}

=item error

Returns a (hopefully) meaningful error string.

=cut

sub error {
    my $self = shift;
    return $self->{error};
}

###########################################################################
#Internal Functions                                                       #
###########################################################################

sub _initialize {
    my($self, %params) = @_;
    # set defaults
    $self->{$_} = $params{$_}   for(@defaults);

    return  if(keys %currencies);
    local($_);

    # Extract the mapping of currencies and their atrributes
    while(<Finance::Currency::Convert::XE::DATA>){
        chomp;
        my ($code,$text,$symbol) = split "";
        $currencies{$code}->{name} = $text;
        $currencies{$code}->{symbol} = $symbol;
    }

    return;
}

# Formats the return string to the requirements of the caller
sub _format {
    my($self, $form) = @_;

    my %formats = (
        'symbol' => $self->{symbol} . '%.02f',
        'abbv'   => '%.02f ' . $self->{code},
        'text'   => '%.02f ' . $self->{name},
        'number' => '%.02f',
    );

    return $formats{$form}              if(defined $form && $formats{$form});
    return '%.02f';
}

# Extract the text from the html we get back from UCC and return
# it (keying on the fact that what we want is in the table after
# the faq link).
sub _extract_text {
    my($self, $html) = @_;
    my $tag;
    my $p = HTML::TokeParser->new(\$html);

    # look for the faq link
    while(1) {
        return  unless($tag = $p->get_tag('a'));
        last    if(defined $tag->[1]{href} && $tag->[1]{href} =~ /faq/);
    }

    # jump to the next table
    $tag = $p->get_tag('table');

    # from there look for the target value
    while($p->get_token) {
        my $text = $p->get_trimmed_text;

        if(my ($value) = $text =~ /([\d\.\,]+) $self->{code}/) {
            $value =~ s/,//g;
            return sprintf $self->{string}, $value;
        }
    }

    # didn't find anything
    return;
}

1;

#--------------------------------------------------------------------------

=back

=head1 TERMS OF USE

XE.com have a Terms of Use policy that states:

  This website is for informational purposes only and is not intended to
  provide specific commercial, financial, investment, accounting, tax, or
  legal advice. It is provided to you solely for your own personal,
  non-commercial use and not for purposes of resale, distribution, public
  display or performance, or any other uses by you in any form or manner
  whatsoever. Unless otherwise indicated on this website, you may display,
  download, archive, and print a single copy of any information on this
  website, or otherwise distributed from XE.com, for such personal,
  non-commercial use, provided it is done pursuant to the User Conduct and
  Obligations set forth herein.

As such this software is for personal use ONLY. No liability is accepted by
the author for abuse or miuse of the software herein. Use of this software
is only permitted under the terms stipulated by XE.com.

The full legal document is available at L<http://www.xe.com/legal/>

=head1 TODO

Currency symbols are currently specified with a generic symbol, if the
currency symbol is unknown. Are there any other symbols available in
Unicode? Let me know if there are.

=head1 SEE ALSO

  WWW::Mechanize
  HTML::TokeParser

=head1 SUPPORT

There are no known bugs at the time of this release. However, if you spot a
bug or are experiencing difficulties that are not explained within the POD
documentation, please submit a bug to the RT system (see link below). However,
it would help greatly if you are able to pinpoint problems or even supply a
patch.

Fixes are dependant upon their severity and my availablity. Should a fix not
be forthcoming, please feel free to (politely) remind me by sending an email
to barbie@cpan.org .

RT: L<http://rt.cpan.org/Public/Dist/Display.html?Name=Finance-Currency-Convert-XE>

=head1 AUTHOR

  Barbie, <barbie@cpan.org>
  for Miss Barbell Productions <http://www.missbarbell.co.uk>.

=head1 COPYRIGHT

  Copyright  2002-2008 Barbie for Miss Barbell Productions.

  This library is free software; you can redistribute it and/or modify it under
  the same terms as Perl itself, using the Artistic License.

The full text of the licenses can be found in the Artistic file included with
this distribution, or in perlartistic file as part of Perl installation, in
the 5.8.1 release or later.

=cut

#--------------------------------------------------------------------------

__DATA__
AEDUnited Arab Emirates, Dirham&#164;
AFNAfghanistan, Afghani&#164;
ALLAlbania, Lek&#164;
AMDArmenia, Dram&#164;
ANGNetherlands Antilles Guilder&#164;
AOAAngola, Kwanza&#164;
ARSArgentina, Peso&#164;
AUDAustralia, Dollar$
AWGAruba, Guilder&#164;
AZNAzerbaijan, New Manat&#164;
BAMBosnia and Herzegovina, Convertible Marka&#164;
BBDBarbados, Dollar&#164;
BDTBangladesh, Taka&#164;
BGNBulgaria, Lev&#164;
BHDBahrain, Dinar&#164;
BIFBurundi, Franc&#164;
BMDBermuda, Dollar&#164;
BNDBrunei, Dollar&#164;
BOBBolivia, Boliviano&#164;
BRLBrazil, Real&#164;
BSDBahamas, Dollar&#164;
BTNBhutan, Ngultrum&#164;
BWPBotswana, Pula&#164;
BYRBelarus, Ruble&#164;
BZDBelize, Dollar&#164;
CADCanada, Dollar$
CDFCongo/Kinshasa, Franc&#164;
CHFSwitzerland, Franc&#164;
CLPChile, Peso&#164;
CNYChina, Yuan Renminbi&#164;
COPColombia, Peso&#164;
CRCCosta Rica, Colon&#164;
CUCCuba, Convertible Peso&#164;
CUPCuba, Peso&#164;
CVECape Verde, Escudo&#164;
CZKCzech Republic, Koruna&#164;
DJFDjibouti, Franc&#164;
DKKDenmark, Krone&#164;
DOPDominican Republic, Peso&#164;
DZDAlgeria, Dinar&#164;
EEKEstonia, Kroon&#164;
EGPEgypt, Pound&#164;
ERNEritrea, Nakfa&#164;
ETBEthiopia, Birr&#164;
EUREuro&#8364;
FJDFiji, Dollar&#164;
FKPFalkland Islands, Pound&#164;
GBPGreat Britain, Pound&#163;
GELGeorgia, Lari&#164;
GGPGuernsey, Pound&#164;
GHSGhana, Cedi&#164;
GIPGibraltar, Pound&#164;
GMDGambia, Dalasi&#164;
GNFGuinea, Franc&#164;
GTQGuatemala, Quetzal&#164;
GYDGuyana, Dollar&#164;
HKDHong Kong, Dollar&#164;
HNLHonduras, Lempira&#164;
HRKCroatia, Kuna&#164;
HTGHaiti, Gourde&#164;
HUFHungary, Forint&#164;
IDRIndonesia, Rupiah&#164;
ILSIsrael, New Shekel&#8362;
IMPIsle of Man, Pound&#164;
INRIndia, Rupee&#8360;
IQDIraq, Dinar&#164;
IRRIran, Rial&#164;
ISKIceland, Krona&#164;
JEPJersey, Pound&#164;
JMDJamaica, Dollar&#164;
JODJordan, Dinar&#164;
JPYJapan, Yen&#165;
KESKenya, Shilling&#164;
KGSKyrgyzstan, Som&#164;
KHRCambodia, Riel&#164;
KMFComoros, Franc&#164;
KPWNorth Korea, Won&#164;
KRWSouth Korea, Won&#8361;
KWDKuwait, Dinar&#164;
KYDCayman Islands, Dollar&#164;
KZTKazakhstan, Tenge&#164;
LAKLaos, Kip&#164;
LBPLebanon, Pound&#164;
LKRSri Lanka, Rupee&#164;
LRDLiberia, Dollar&#164;
LSLLesotho, Loti&#164;
LTLLithuania, Litas&#164;
LVLLatvia, Lat&#164;
LYDLibya, Dinar&#164;
MADMorocco, Dirham&#164;
MDLMoldova, Leu&#164;
MGAMadagascar, Ariary&#164;
MKDMacedonia, Denar&#164;
MMKBurma, Kyat&#164;
MMKMyanmar, Kyat&#164;
MNTMongolia, Tughrik&#164;
MOPMacau, Pataca&#164;
MROMauritania, Ouguiya&#164;
MURMauritius, Rupee&#164;
MVRMaldives, Rufiyaa&#164;
MWKMalawi, Kwacha&#164;
MXNMexico, Peso&#164;
MYRMalaysia, Ringgit&#164;
MZNMozambique, Metical&#164;
NADNamibia, Dollar&#164;
NGNNigeria, Naira&#164;
NIONicaragua, Cordoba&#164;
NOKNorway, Krone&#164;
NPRNepal, Rupee&#164;
NZDNew Zealand, Dollar&#164;
OMROman, Rial&#164;
PABPanama, Balboa&#164;
PENPeru, Nuevo Sol&#164;
PGKPapua New Guinea, Kina&#164;
PHPPhilippines, Peso&#164;
PKRPakistan, Rupee&#8360;
PLNPoland, Zloty&#164;
PYGParaguay, Guarani&#164;
QARQatar, Riyal&#164;
RONRomania, New Leu&#164;
RSDSerbia, Dinar&#164;
RUBRussia, Ruble&#164;
RWFRwanda, Franc&#164;
SARSaudi Arabia, Riyal&#164;
SBDSolomon Islands, Dollar&#164;
SCRSeychelles, Rupee&#164;
SDGSudan, Pound&#164;
SEKSweden, Krona&#164;
SGDSingapore, Dollar&#164;
SHPSaint Helena, Pound&#164;
SKKSlovakia, Koruna&#164;
SLLSierra Leone, Leone&#164;
SOSSomalia, Shilling&#164;
SPLSeborga, Luigino&#164;
SRDSuriname, Dollar&#164;
STDSo Tome and Principe, Dobra&#164;
SVCEl Salvador, Colon&#164;
SYPSyria, Pound&#164;
SZLSwaziland, Lilangeni&#164;
THBThailand, Baht&#3647;
TJSTajikistan, Somoni&#164;
TMMTurkmenistan, Manat&#164;
TNDTunisia, Dinar&#164;
TOPTonga, Pa'anga&#164;
TRYTurkey, New Lira&#164;
TTDTrinidad and Tobago, Dollar&#164;
TVDTuvalu, Dollar&#164;
TWDTaiwan, New Dollar&#164;
TZSTanzania, Shilling&#164;
UAHUkraine, Hryvna&#164;
UGXUganda, Shilling&#164;
USDUnited States, Dollar$
UYUUruguay, Peso&#164;
UZSUzbekistan, Som&#164;
VEFVenezuela, Bolivar Fuerte&#164;
VNDVietnam, Dong&#164;
VUVVanuatu, Vatu&#164;
WSTSamoa, Tala&#164;
WSTWestern Samoa, Tala&#164;
XAFCFA Communaut Financire Africaine BEAC Franc&#164;
XAGSilver Ounce&#164;
XAUGold Ounce&#164;
XCDEast Caribbean Dollar&#164;
XDRInternational Monetary Fund Special Drawing Right&#164;
XOFCFA Communaut Financire Africaine BCEAO Franc&#164;
XPDPalladium Ounce&#164;
XPFComptoirs Franais du Pacifique Franc&#164;
XPTPlatinum Ounce&#164;
YERYemen, Rial&#164;
ZARSouth Africa, Rand&#164;
ZMKZambia, Kwacha&#164;
ZWDZimbabwe, Dollar&#164;
