/*******************************************************************************
* Copyright (c) 2000, 2005 IBM Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
*
* Contributors:
*     IBM Corporation - initial API and implementation
*******************************************************************************/

package org.eclipse.wst.rdb.internal.outputview;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;

import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.core.runtime.IStatus;

/**
 * Dialog to show a Browser Composite with rendered XML output
 * @author quyon
 */
public class DialogShowXMLField extends Dialog
{
    protected String xmlText;
    protected String dialogTitle;
    protected Button btnClose;
    protected Button btnSaveAs;    
    
    private static final String XML_TABS_EXT_POINT = 
        "org.eclipse.wst.rdb.outputview.xmlTabs";
    private static final String XML_TAB_ATTR_CLASS = 
        "class";
    
    /**
     * Creates an instance of DialogShowXMLField
     * @param shell the parent Shell
     * @param title the dialog title
     * @param text the XML text
     */
    public DialogShowXMLField(Shell shell, String title, String text)
    {
        super(shell);
        xmlText = text;
        dialogTitle = title;        
    }
    
    /**
     * Sets the window title when the shell is configured.
     * @param newShell The shell.
     */
    protected void configureShell(Shell newShell)
    {
       super.configureShell(newShell);
       newShell.setText(dialogTitle);
    }
    
    /** 
     * Sets the shell style when the dialog is opened.
     */
    public int open()
    {
       setShellStyle(getShellStyle() | SWT.RESIZE);
       return super.open();
    }
    
    /**
     * Creates the elements of the dialog area.
     * @param parent The parent composite.
     * @return The dialog area.
     */
    protected Control createDialogArea(Composite parent)
    {
        Composite composite = (Composite) super.createDialogArea(parent);
        GridData gridData;
        
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 1;
        composite.setLayout(gridLayout);
        
        gridData = new GridData();
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessVerticalSpace = true;
        gridData.verticalAlignment = GridData.FILL;
        
        gridData.widthHint = 480;
        gridData.heightHint = 300;         
        
        TabFolder tabFolder = new TabFolder(composite, SWT.BOTTOM);
        tabFolder.setLayoutData(gridData);
        
        // get any additional tabs from extensions
        addTabItemsFromExtensions(tabFolder);
        
        createSourceTabItem(tabFolder);
        
        return composite;
    }
    
    /**
     * Gets the tabs to the folder provided by extensions.
     * <p>
     * Each TabItem provided by the extensions will be added to the parent folder
     * @param parent the parent TabFolder
     */
    private void addTabItemsFromExtensions(TabFolder parent)
    {
    	IExtensionRegistry extensionRegistry = Platform.getExtensionRegistry();
        IExtensionPoint extensionPoint = 
            extensionRegistry.getExtensionPoint(XML_TABS_EXT_POINT);
        IExtension [] extensions = extensionPoint.getExtensions();        
        for (int i=0;i<extensions.length;i++)
        {
        	IExtension ext = extensions[i];
            IConfigurationElement [] configElements = ext.getConfigurationElements();
            for (int j=0;j<configElements.length;j++)
            {
            	IConfigurationElement currentConfigElement = configElements[j];
                try
                {                    
                    IXMLTab myXMLTabs = 
                		(IXMLTab)currentConfigElement.createExecutableExtension(XML_TAB_ATTR_CLASS);
                    myXMLTabs.init(xmlText);                    
                    myXMLTabs.parse();
                    myXMLTabs.getXMLTab(parent);
                }
                catch (CoreException ex)
                {
                    OutputViewPlugin.getPlugin().writeLog(IStatus.ERROR, 0, 
                            "DialogShowXMLField:addTabItemsFromExtensions()", ex); //$NON-NLS-1$
                }
            }
        }
    }
    
    /**
     * Creates the Source TabItem
     * @param parent the parent TabFolder
     */
    private void createSourceTabItem(TabFolder parent)
    {
        Text sourceText = new Text(parent, SWT.MULTI |
                SWT.READ_ONLY | SWT.WRAP | SWT.V_SCROLL);
        sourceText.setText(xmlText);
        TabItem item = new TabItem(parent, SWT.NONE);
        item.setText(OutputViewPlugin.getString("XML_DIALOG_SOURCE"));
        item.setControl(sourceText);
    }    
    
    /**
     * Creates Save As and Close buttons to replace the Ok and Cancel buttons.
     * Save As will save the XML source to a file and Close will use the
     * Cancel-Button-behavior to reuse cancelPressed().
     * @param parent Our parent composite.
     */
    protected void createButtonsForButtonBar(Composite parent)
    {
        //  Overrides Ok button to save XML source when clicked
        btnSaveAs = createButton(parent, IDialogConstants.OK_ID,
             OutputViewPlugin.getString("OV_MESSAGE_SAVE_AS"), false);
        
        btnClose = createButton(parent, IDialogConstants.CANCEL_ID, 
            IDialogConstants.CLOSE_LABEL, false);       
    }
    
    /**
     * Called when Ok(Save As...) button is pressed
     */
    protected void okPressed()
    {
        saveOutput();               
    }    
    
    /**
     * Displays an error message box to let a user know that the file that is selected
     * is a directory.  This is needed for Linux because the fileChooser allows
     * the directory name as a file.
     * @param aShell the parent shell
     * @param aFile the file name
     */
    protected void displayDirectoryError(Shell aShell, String aFile)
    {
       // display error message because a directory was chosen
       Object[] obj = {aFile};
       String message = OutputViewPlugin.getString(
                "OV_MESSAGE_FILE_ISDIRECTORY_DESC", obj);
       MessageBox box = new MessageBox(aShell, SWT.ICON_ERROR | SWT.OK);
       box.setText(OutputViewPlugin.getString("OV_MESSAGE_FILE_ISDIRECTORY_TITLE"));
       box.setMessage(message);
       box.open();
       // reopen save dialog again
       saveOutput();
    }
    
    /**
     * Launches file chooser to let a user save results to a file.
     */
    private void saveOutput()
    {
        FileDialog fileDialog = new FileDialog(this.getShell(), SWT.SAVE);
        String fileName = fileDialog.open();
        if (fileName != null)
        {
            try
            {
                File file = new File(fileName);
                if (file.isDirectory())
                {
                    displayDirectoryError(this.getShell(), fileName);
                    // return is needed because displayDirectoryError(...) makes
                    // a recursive call to this method.
                    return;
                }
                if (file.exists())
                {
                    // launch question dialog
                    Object[] obj = {fileName};
                    String message = OutputViewPlugin.getString(
                         "OV_MESSAGE_FILE_EXISTS_DESC", obj);
                    MessageBox box = new MessageBox(this.getShell(),
                         SWT.YES | SWT.NO);
                    box.setText(OutputViewPlugin.getString("OV_MESSAGE_FILE_EXISTS_TITLE"));
                    box.setMessage(message);
                    if (box.open() != SWT.YES)
                    {
                        return;
                    }
                }
                BufferedWriter bw = new BufferedWriter(new FileWriter(file));
                bw.write(xmlText);             
                bw.flush();
                bw.close();
            }
            catch (IOException ex)
            {
                MessageBox box = new MessageBox(this.getShell(),
                    SWT.ICON_ERROR);
                box.setText(OutputViewPlugin.getString("OV_STATUS_ERROR"));
                box.setMessage(ex.getMessage());
                box.open();
            }
        }
    }       
}