/*

Copyright (C) 2004 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __libnd_traceset_h
#define __libnd_traceset_h

#include <libnd.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

typedef struct lnd_traceset LND_TraceSet;

typedef enum {
  LND_TRACESET_ERROR_SKIP,
  LND_TRACESET_ERROR_ABORT,
  LND_TRACESET_ERROR
} LND_TraceSetMode;

/**
 * LND_TraceSetCB - signature of the standard callback.
 * @set: pointer to the traceset used for the iteration.
 * @trace: a pointer to the trace currently iterated.
 * @user_data: arbitrary data passed to libnd_traceset_foreach().
 *
 * This is the normal iterator's callback signature. It provides
 * a full-blown LND_Trace handle to the user. This handle is
 * ephemeral in the sense that it can only be used during the
 * callback. If you want to keep the trace around, keep the filename
 * around or find another way to recreate it later.
 *
 * Returns: %TRUE if iteration is to continue, %FALSE if you want it to abort.
 */
typedef gboolean (*LND_TraceSetCB)(LND_TraceSet *set, LND_Trace *trace, void *user_data);

/**
 * LND_TraceSetPcapCB - signature of the more simple pcap-based callback.
 * @set: pointer to the traceset used for the iteration.
 * @pcap: pcap handle for the currently iterated trace.
 * @user_data: arbitrary data passed to libnd_traceset_foreach().
 *
 * This is the more simplistic (but more performant) callback in which
 * a pcap handle is provided instead of at full-blown LDN_Trace.
 * Otherwise it behaves just like a LND_TraceSetCB callback.
 *
 * Returns: %TRUE if iteration is to continue, %FALSE if you want it to abort.
 */
typedef gboolean (*LND_TraceSetPcapCB)(LND_TraceSet *set, pcap_t *pcap, void *user_data);

/**
 * LND_TraceSetNameCB - signature of the name-based callback.
 * @set: pointer to the traceset used for the iteration.
 * @name: name of trace file currently iterated.
 * @user_data: arbitrary data passed to libnd_traceset_foreach().
 *
 * This version is used with libnd_traceset_foreach_name() and provides
 * the filename of the trace currently iterated.
 *
 * Returns: %TRUE if iteration is to continue, %FALSE if you want it to abort.
 */
typedef gboolean (*LND_TraceSetNameCB)(LND_TraceSet *set, const char *name, void *user_data);


LND_TraceSet    *libnd_traceset_new(void);

/**
 * libnd_traceset_keep_open - keep flows open until set is destroyed.
 * @set: trace set.
 *
 * The function ensures that when a set contains multiple traces, the
 * TraceSet plugin does not release traces that have been iterated over.
 * It does not guarantee that the traces are being kept alive by all
 * means. You basically want to use this if you iterate over a number
 * of traces that you specify by file name where you do keep pointers
 * to former traces around.
 */
void             libnd_traceset_keep_open(LND_TraceSet *set);

/**
 * libnd_traceset_check_files - whether to check for valid pcap files when adding
 * @set: set to set flag for.
 * @do_check: whether to check or not.
 *
 * If @do_check is %TRUE, then filnames added to a set are checked
 * for whether they are valid pcap files, if %FALSE then they aren't.
 * Default is %FALSE.
 */
void libnd_traceset_check_files(LND_TraceSet *set, gboolean do_check);

LND_TraceSet    *libnd_traceset_copy(const LND_TraceSet *set);
void             libnd_traceset_free(LND_TraceSet *set);

gboolean         libnd_traceset_add_trace(LND_TraceSet *set, LND_Trace *trace);
gboolean         libnd_traceset_add_trace_name(LND_TraceSet *set, const char *file_name);
gboolean         libnd_traceset_add_trace_name_list(LND_TraceSet *set, const char *file_name);

void             libnd_traceset_set_mode(LND_TraceSet *set,
					 LND_TraceSetMode mode);

LND_TraceSetMode libnd_traceset_get_mode(LND_TraceSet *set);

int              libnd_traceset_get_size(LND_TraceSet *set);

/**
 * libnd_traceset_partition - partitions a traceset into a number of smaller sets.
 * @set: set to partition.
 * @num: number of smaller sets to create.
 *
 * The function creates up to @num smaller trace sets out of @set, with disjoint sets
 * of trace files. @set is not modified.
 *
 * Returns: list of new trace sets, %NULL on error.
 */
GList           *libnd_traceset_partition(LND_TraceSet *set, int num);

void             libnd_traceset_foreach(LND_TraceSet *set,
					LND_TraceSetCB callback,
					void *user_data);

void             libnd_traceset_foreach_pcap(LND_TraceSet *set,
					     LND_TraceSetPcapCB callback,
					     void *user_data);

void             libnd_traceset_foreach_name(LND_TraceSet *set,
					     LND_TraceSetNameCB callback,
					     void *user_data);


#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif
